#!/usr/bin/env perl
# pmltq_nobtred.pl     pajas@ufal.mff.cuni.cz     2009/06/22 13:09:08

our $VERSION="0.1";

use warnings;
use strict;
$|=1;

use Getopt::Long;
use Pod::Usage;
Getopt::Long::Configure ("bundling");
my %opts;
GetOptions(\%opts,
	   "query|Q=s",
	   "query-pml-file|p=s",
	   "query-file|f=s",
	   "apply-filters|a",
	   "query-id|i=s",
	   "netgraph-query=s",

	   'help|h',
	   'usage|u',
	   'version|V',
	   'man',
	   'debug|D=i',
	   #	'quiet|q',
       ) or $opts{usage}=1;

if ($opts{usage}) {
  pod2usage(-msg => 'pmltq_nobtred.pl');
}
if ($opts{help}) {
  pod2usage(-exitstatus => 0, -verbose => 1);
}
if ($opts{man}) {
  pod2usage(-exitstatus => 0, -verbose => 2);
}
if ($opts{version}) {
  print "$VERSION\n";
  exit;
}

#####################
# Code to provide stuff required from btred
#####################
use FindBin qw($RealBin);
use lib ($RealBin.'/../libs/fslib',
	 $RealBin.'/../libs/pml-base',
	 $RealBin.'/../libs/pmltq',
	 ((do { chomp($ENV{TREDLIB}||=`btred -q --lib`); 1 } && $ENV{TREDLIB} && -d $ENV{TREDLIB}) ? $ENV{TREDLIB} :
	   die "Please set the TREDLIB environment variable to point to tred/tredlib!\n")
	);

use Treex::PML;
Treex::PML::AddResourcePath($RealBin.'/../resources');
Treex::PML::UseBackends(qw(Storable PMLBackend PMLTransformBackend));

{
  package TredMacro;
  use TrEd::Basics;
  BEGIN { require 'tred.def'; }
  no warnings qw(redefine);
  sub DetermineNodeType {
    my ($node)=@_;
    Treex::PML::Document->determine_node_type($node);
  }
}

{
  package PML;
  sub Schema {
    &TrEd::Basics::fileSchema;
  }
  sub GetNodeByID {
    my ($id,$fsfile)=@_;
    my $h = $fsfile->appData('id-hash');
    return $h && $id && $h->{$id};
  }
}

use Tree_Query::TypeMapper;
use Tree_Query::BtredEvaluator;

#####################

package main;
use utf8;

unless (@ARGV) {
  die "Usage: $0 [options] <filename>(s)!"
}

#
# Load query from the STDIN, a file or a string
#

$Tree_Query::BtredEvaluator::DEBUG=$opts{debug};

my ($query,$query_file);
if ($opts{stdin}) {
  local $/;
  $opts{query} = <>;
} elsif ($opts{'query-pml-file'}) {
  $query_file = openFile($opts{'query-pml-file'})
    or die "Failed to open PML query file: $opts{'query-pml-file'}!";
  $query = first {
    !$opts{'query-id'} or $_->{id} eq $opts{'query-id'}
  } $query_file->trees;
  die "Didn't find query $opts{'query-id'} in query file $opts{'query-pml-file'}!"
    unless $query;
} elsif ($opts{'query-file'}) {
  open my $fh, '<:utf8', $opts{'query-file'} || die "Cannot open query file ".$opts{'query-file'}.": $!\n";
  local $/;
  $query = <$fh>;
  if ($opts{'query-id'}) {
    $query=~s/#\s*==\s*query:\s*\Q$opts{'query-id'}\E\s* ==(.*?)(?:#\s*==\s*query:\s*\w+\s*==.*|$)/$1/s;
  }
} else {
  $query = $opts{query};
}
if (!ref($query) and $opts{'netgraph-query'}) {
  require Tree_Query::NG2PMLTQ;
  $query = Tree_Query::NG2PMLTQ::ng2pmltq($query,{type=>$opts{'netgraph-query'}});
}
die "Query is empty!" unless $query;

#####################################################
# open a data file and related files on lower layers

sub openFile {
  my ($filename)=@_;
  my $fsfile = Treex::PML::Factory->createDocumentFromFile(shift,{backends => \@backends});
  if ($Treex::PML::FSError) {
    die "Error loading file $filename: $Treex::PML::FSError ($!)\n";
  }
  my $requires = $fsfile->metaData('fs-require');
  if ($requires) {
    for my $req (@$requires) {
      my $req_filename = $req->[1]->abs($fsfile->URL);
      my $secondary = $fsfile->appData('ref');
      unless ($secondary) {
	$secondary = {};
	$fsfile->changeAppData('ref',$secondary);
      }
      my $sf = openFile($req_filename,$fsfile);
      $secondary->{$req->[0]}=$sf;
    }
  }
  return $fsfile;
}

my $fsfile = openFile(shift);

#################################################
#
# Compile query and initialize the query enginge

my $evaluator = Tree_Query::BtredEvaluator->new($query, {
  fsfile => $fsfile,
  # tree => $fsfile->tree(0), # query only a specific tree
  # no_plan => 1, # do not let the planner rewrite my query
                  # in this case, the query must not be a forest!
});

binmode STDOUT, ':utf8';

# iterate over several files (or maybe several scattered trees)
sub next_file {
  return unless @ARGV;
  $fsfile = openFile(shift @ARGV);
  # reusing the evaluator for next file
  my $iter = $evaluator->get_first_iterator;
  $iter->set_file($fsfile);
  # $iter->set_tree($fsfile->tree(0)); # if tree searching a specific tree
  $evaluator->reset(); # prepare for next file
  return 1
}

# running the query and print results
if ($evaluator->get_filters()) {
  # query with filters (produces text output)

  ## customize output from the final filter
  $evaluator->init_filters({
    init => sub { print("-" x 60, "\n") },
    process_row => sub { my ($self,$row)=@_; print(join("\t",@$row)."\n"); },
    finish => sub { print("-" x 60, "\n"); }
   });
  do {{
    $evaluator->run_filters while $evaluator->find_next_match(); # feed the filter pipe
  }} while (next_file());
  $evaluator->flush_filters; # flush the pipe

} else {
    # query without a fitlter (just selects nodes)
  do {{
    while ($evaluator->find_next_match()) {
      # get whatever data

      ## named query node:
      # print $evaluator->get_result_node('n')->attr('id')."\n";

      ## the order of columns may be differnt than the order of query nodes
      ## since the query can be rewritten by the planner
      print join("\t", map $_->attr('id'), @{$evaluator->get_results})."\n";
    }
  }} while (next_file());
}
#################################

__END__

=head1 NAME

pmltq_nobtred.pl - run PML-TQ query on PML files without database or Btred

=head1 SYNOPSIS

pmltq_nobtred.pl [--query query| --query-file filename | -query-pml-file filename ] file(s)

or

  pmltq_nobtred.pl -u          for usage

  pmltq_nobtred.pl -h          for help

  pmltq_nobtred.pl --man       for the manual page

  pmltq_nobtred.pl --version   for version

=head1 DESCRIPTION

This script is a demo that shows how to use PML-TQ API to query PML files without Btred.

Currently the Treex::PML and the PML-TQ libraries are required as well as
the following files from TrEd:

  tred.def
  TrEd::Basics

In order to determine path to these files, the TREDLIB environment
variable is used; if not set, btred --lib is called.

=over 5

=item	B<--stdin>

Read query from the standard input.

=item	B<--query|-Q> string

Specify PML-TQ query on the command-line.

=item	B<--query-file> filename

Read PML-TQ query from a given (utf-8 encoded text) file

=item	B<--query-pml-file> filename

Read PML-TQ query from a given PML file

=item	B<--query-id> ID

Use query with a given ID. If the input is a text file, it can contain more than one
query. In that case, each query must start with a line of the following form:

  # == query: ID ==

where ID is a unique identifier of the query. This option can be used
to select a single query from the input.

If the input is a PML file, then the ID is just the id of the query tree.

=item B<--usage|-u>

Print a brief help message on usage and exits.

=item B<--help|-h>

Prints the help page and exits.

=item B<--man>

Displays the help as manual page.

=item B<--version>

Print program version.

=back

=head1 AUTHOR

Petr Pajas, E<lt>pajas@sup.ms.mff.cuni.czE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009 by Petr Pajas

This program is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.2 or,
at your option, any later version of Perl 5 you may have available.

=head1 BUGS

None reported... yet.

=cut
