package com.vizdom.dbd.jdbc;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import junit.framework.*;


/*
 * Tests the CgiProcess class.
 */
public class TestCgiProcess extends TestCase
{
    /*
    // Use this to run one or more tests alone.
    public static Test suite()
    {
        return new TestCgiProcess("testDbiReturnData");
    }
    */

    // Directory name; must be on classpath.
    private String mResourceLocation = "DbiTestData";
    private String mNoDbiScript = "nodbi.pl";
    private String mNoDbiCommandLine = "perl " + mNoDbiScript;
    private String mDbiScript = "dbi.pl";
    private String mDbiCommandLine = "perl " + mDbiScript;
    private String mDbiScriptTraceFile = "dbitrace";

    private java.sql.Connection mConnection = null;
    private String mJdbcDriver = "com.opentext.basis.jdbc.BasisDriver";
    private String mJdbcUrl = "jdbc:opentext:basis:";
    private String mJdbcConnectionPropertiesFile = "jdbc.properties";
    private Properties mArgs;
    private Properties mDbiExpected;

    public TestCgiProcess(String aTestName)
    {
        super(aTestName);
    }

    // Create data for use as parameters and as tests against script output.
    // Note that there's also test-specific setup.
    protected void setUp()
    {
        mArgs = new Properties();
        for (int i = 0; i < 5; i++)
            mArgs.put("name" + i, "value" + i);
        mDbiExpected = (Properties) mArgs.clone();
        mDbiExpected.put("ENO", "1001");
    }

    // Try to get rid of resources. Ignore errors.
    protected void tearDown()
    {
        new File(mNoDbiScript).delete(); 
        new File(mDbiScript).delete();
        if (mConnection != null)
        {
            try { mConnection.close(); } catch (Exception e) {}
        }
    }
    
    // Set up data for tests which don't use a JDBC connection.
    private void mNoDbiSetup()
    {
        try
        {
            if (new File(mNoDbiScript).exists())
                fail(mNoDbiScript + " already exists");
            mCopyStream(ClassLoader.getSystemResourceAsStream(
                mResourceLocation + "/" + mNoDbiScript), mNoDbiScript);
        }
        catch (IOException e)
        {
            fail("Unable to set up " + mNoDbiScript);
        }
    }

    // Set up data for tests which use a JDBC connection.
    private void mDbiSetup()
    {
        try
        {
            if (new File(mDbiScript).exists())
                fail(mDbiScript + " already exists");
            mCopyStream(ClassLoader.getSystemResourceAsStream(
                mResourceLocation + "/" + mDbiScript), mDbiScript);
        }
        catch (IOException e)
        {
            fail("Unable to set up " + mDbiScript);
        }

        try
        {
            Class.forName(mJdbcDriver);
            InputStream in = ClassLoader.getSystemResourceAsStream(
                mResourceLocation + "/" + mJdbcConnectionPropertiesFile);
            Properties properties = new Properties();
            properties.load(in);
            in.close();

            mConnection = java.sql.DriverManager.getConnection(
                mJdbcUrl, properties);
        }
        catch (Exception e)
        {
            fail("Unable to establish JDBC connection: " + e.toString());
        }

        new File(mDbiScriptTraceFile).delete();
    }


    // Run tests which don't use a JDBC connection.
    public void testNoDbiReturnData()
    {
        mNoDbiSetup();
        String error;
        Properties properties;
        Hashtable results;
        try
        {
            // (command, Hashtable) test.
            CgiProcess cgi = new CgiProcess(mNoDbiCommandLine, mArgs);
            // Calling cgi.getOutputStream here worked, even though
            // the stream was used and closed in CgiProcess();
            // Writing to it had no effect and threw no exceptions. 
            // In this case, we've sent the expected content length 
            // to the script, so a CGI library would probably ignore
            // any extra information anyway. 
            results = mGetOutput(cgi);
            error = (String) results.get("stderr");
            properties = (Properties) results.get("properties");

            assertTrue("(Hashtable) Errors returned from script: " + error, 
                error == null || error.equals(""));
            assertTrue("(Hashtable) Data doesn't match parameters", 
                mCompare(mArgs, properties));


            // (command, content-type, length) test. 
            String params = mGetCgiParamString(mArgs);
            byte[] data = params.getBytes();
            cgi = new CgiProcess(mNoDbiCommandLine, 
                "application/x-www-form-urlencoded", data.length);
            OutputStream out = cgi.getOutputStream();
            out.write(data);
            out.close();

            results = mGetOutput(cgi);
            error = (String) results.get("stderr");
            properties = (Properties) results.get("properties");

            assertTrue("(type,length) Errors returned from script: " + error, 
                error == null || error.equals(""));
            assertTrue("(type,length) Data doesn't match parameters", 
                mCompare(mArgs, properties));
        }
        catch (Exception e)
        {
            fail("Unexpected exception: " + e.toString());
        }
    }

    // See what happens when the arguments are bad.
    public void testNoDbiArgs()
    {
        mNoDbiSetup();
        CgiProcess cgi;
        try
        {
            // Null command line will throw NullPointerException.
            try
            {
                cgi = new CgiProcess(null, mArgs);
                fail("(hashtable) Null command line didn't fail");
                cgi.destroy();
            }
            catch (NullPointerException e) { /* expected */ }

            // Null arguments will send no parameters to subprocess.
            try
            {
                cgi = new CgiProcess(mNoDbiCommandLine, null);
                cgi.destroy();
            }
            catch (NullPointerException e) 
            {
                fail("(hashtable) Null arguments: " + e);
            }

            // Null command line will throw NullPointerException.
            try
            {
                cgi = new CgiProcess(null, "content-type", 3);
                fail("(type,length) Null command line didn't fail");
                cgi.destroy();
            }
            catch (NullPointerException e) { /* expected */ }

            // Null content-type is the responsibility of the
            // new process's CGI handling to interpret; incorrect
            // content-length likewise.
            try
            {
                cgi = new CgiProcess(mNoDbiCommandLine, null, 3);
                cgi.destroy();
            }
            catch (Exception f) { fail("(type,length) content-type " + f); }
                
        }
        catch (Exception e)
        {
            fail("Unexpected exception: " + e.toString());
        }
    }

    // Test bad command lines.
    public void testNoDbiInvalidCommandLine()
    {
        mNoDbiSetup();
        try
        {
            // This test assumes our command line is "perl foo"; hence
            // the different behavior in the two cases.
            CgiProcess cgi = new CgiProcess(mNoDbiCommandLine + "x", mArgs);
            int ret = cgi.waitFor();
            assertTrue("Process with invalid command line returned 0", ret != 0);

            try
            {
                cgi = new CgiProcess("x" + mNoDbiCommandLine, mArgs);
                fail("Invalid command line didn't fail");
            }
            catch (IOException ie) { }
        }
        catch (Exception e)
        {
            fail(e.toString());
        }
    }

    // Test use of a JDBC connection from DBI.
    public void testDbiReturnData()
    {
        mDbiSetup();
        CgiProcess cgi;
        Hashtable results;
        Properties properties;
        String error;
        try
        {

            // (command, Hashtable) test.
            cgi = new CgiProcess(mDbiCommandLine, mArgs,
                mConnection);

            results = mGetOutput(cgi);
            error = (String) results.get("stderr");
            properties = (Properties) results.get("properties");

            assertTrue("(Hashtable) Errors returned from script: " + error, 
                error == null || error.equals(""));
            assertTrue("(Hashtable) Data returned doesn't match parameters", 
                mCompare(mDbiExpected, properties));
            


            // (command, content-type, length) test. 
            // This is a bit tedious. We need to include the port in 
            // the data sent to the CGI process, therefore the port
            // name=value pair is part of the content-length. However, 
            // we don't know the port number until after we've started
            // the process and sent it the content-length. So, assume
            // port numbers have no more than 6 digits and fake it. Note
            // that this test also assumes that all digit characters have
            // the same number of bytes, so that 0 is a valid placeholder
            // for the actual port digits.

            String params = mGetCgiParamString(mArgs);
            // mGetCgiParamString includes a trailing '&'
            params += CgiProcess.PORT_KEY + "=000000"; 

            cgi = new CgiProcess(mDbiCommandLine, 
                "application/x-www-form-urlencoded", 
                params.getBytes().length, mConnection);

            // Replace the placeholder 000000 with the actual port number.
            StringBuffer paramBuf = new StringBuffer(params);
            paramBuf.setLength(paramBuf.length() - 6);
            paramBuf.append(
                mLeftZeroPad(String.valueOf(cgi.getServerPort()), 6));

            OutputStream out = cgi.getOutputStream();
            out.write(paramBuf.toString().getBytes());
            out.close();

            results = mGetOutput(cgi);
            error = (String) results.get("stderr");
            properties = (Properties) results.get("properties");

            assertTrue("Errors returned from script: " + error, 
                error == null || error.equals(""));
            assertTrue("(type,length) Data returned doesn't match parameters", 
                mCompare(mDbiExpected, properties));
        }
        catch (Exception e)
        {
            fail("Unexpected exception: " + e.toString());
        }
    }


    // The test Perl scripts return something that looks like 
    // a Properties file to stdout and write errors to stderr.
    private Hashtable mGetOutput(Process aProcess) throws IOException
    {
        Hashtable results = new Hashtable();
        // stderr
        Reader reader = new InputStreamReader(aProcess.getErrorStream());
        char[] chars = new char[1024];
        java.io.CharArrayWriter buffer = 
            new java.io.CharArrayWriter(5096);
        int count;
        while ((count = reader.read(chars)) != -1)
            buffer.write(chars, 0, count);
        buffer.close();

        results.put("stderr", buffer.toString());
        
        Properties properties = new Properties();
        properties.load(aProcess.getInputStream());
        results.put("properties", properties);

        return results;
    }

    // Compare two Properties objects.
    private boolean mCompare(Properties aProperties1, Properties aProperties2)
    {
        if (aProperties1 == null && aProperties2 == null)
            return true;
        if (aProperties1 == null || aProperties2 == null)
            return false;
        if (aProperties1.size() != aProperties2.size())
            return false;

        Enumeration keys1 = aProperties1.keys();
        while (keys1.hasMoreElements())
        {
            String key1 = (String) keys1.nextElement();
            if (! aProperties1.getProperty(key1).equals(
                aProperties2.getProperty(key1)))
                return false;
        }
        return true;       
    }

    private void mCopyStream(InputStream anOriginal, String aCopyFileName)
        throws IOException
    {
        OutputStream copy = new BufferedOutputStream(
            new FileOutputStream(aCopyFileName));
        byte[] buffer = new byte[4096];
        int count;
        while ((count = anOriginal.read(buffer, 0, buffer.length)) != -1)
            copy.write(buffer, 0, count);
        anOriginal.close();
        copy.close();
    }

    private String mGetCgiParamString(java.util.Hashtable aHashtable)
    {
        if (aHashtable == null)
            return "";
        StringBuffer params = new StringBuffer(1024); // size???
        Enumeration keys = aHashtable.keys();
        while (keys.hasMoreElements())
        {
            String key = (String) keys.nextElement();
            String value = aHashtable.get(key).toString();
            if (value == null)  // Somebody's toString could do this. ???
                value = ""; // URLEncoder doesn't like null arguments.
            
            params.append(java.net.URLEncoder.encode(key));
            params.append("=");
            params.append(java.net.URLEncoder.encode(value));
            params.append("&");
        }
        return params.toString();      
    }


    private String mLeftZeroPad(String anOriginal, int aDesiredLength)
    {
        int diff;
        if ((diff = aDesiredLength - anOriginal.length()) > 0)
        {
            StringBuffer buffer = 
                new StringBuffer(anOriginal.length() + diff);
            buffer.append(anOriginal);
            while (diff-- > 0)
                buffer.insert(0, '0');
            return buffer.toString();
        }
        return anOriginal;
    }

}
