#!/usr/bin/perl -w

# Copyright (c) 2015 Paulo Custodio. All Rights Reserved.
# May be used/distributed under the GPL.

use strict;
use warnings;
use Test::More;
use Capture::Tiny 'capture';
use Path::Tiny;
use POSIX 'strftime';

my $ms;
my $test1 = "test~";
my($out,$err,@res);

sub void(&) { $_[0]->(); () }

use_ok 'Text::MacroScript';

path($test1)->spew(norm_nl(<<'END'));
Test text with hello
%DEFINE hello [world]
Test text with hello
END

#------------------------------------------------------------------------------
# -file
$ms = new_ok('Text::MacroScript' => [ -file => [ $test1 ] ] );
is $ms->expand("hello"), "world";

#------------------------------------------------------------------------------
# %LOAD
$ms = new_ok('Text::MacroScript');
is $ms->expand("hello"), "hello";
is $ms->expand("%LOAD[$test1]\n"), "";
is $ms->expand("hello"), "world";

#------------------------------------------------------------------------------
# load_file
$ms = new_ok('Text::MacroScript');
is $ms->expand("hello"), "hello";
$ms->load_file($test1);
is $ms->expand("hello"), "world";

#------------------------------------------------------------------------------
# expand_file
$ms = new_ok('Text::MacroScript');
@res = $ms->expand_file($test1);
is_deeply \@res, [
	"Test text with hello\n",
	"Test text with world\n",
];

$ms = new_ok('Text::MacroScript');
($out,$err,@res) = capture { void { $ms->expand_file($test1); } };
is $out, 
	"Test text with hello\n".
	"Test text with world\n";
is $err, "";
is_deeply \@res, [];

#------------------------------------------------------------------------------
# %INCLUDE
$ms = new_ok('Text::MacroScript');
is $ms->expand("%INCLUDE[$test1]\n"), 
	"Test text with hello\n".
	"Test text with world\n";

#------------------------------------------------------------------------------
# %REQUIRE
$ms = new_ok('Text::MacroScript');
is $ms->expand("%REQUIRE[macroutil.pl]\n"), "";
is $ms->expand("%DEFINE_SCRIPT copyright [copyright(#0,#1)]"), "";
is $ms->expand("copyright['Paulo Custodio'|2015]"), 
	"<hr />\n".
	"Copyright &copy; 2015 Paulo Custodio. All&nbsp;Rights&nbsp;Reserved. ".
	"Updated&nbsp;".strftime("%Y/%m/%d", localtime).".\n".
	"<!-- Generated by Text::MacroScript -->\n";

ok unlink($test1);

done_testing;


sub norm_nl {
	local($_) = @_;
	s/\r\n/\n/g;
	return $_;
}
