
package Term::Size::ReadKey;

use strict;

require Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(chars pixels);

our $VERSION = 0.01;

=head1 NAME

Term::Size::ReadKey - Perl extension for retrieving terminal size (using Term::ReadKey)

=head1 SYNOPSIS

    use Term::Size::ReadKey;

    ($columns, $rows) = Term::Size::ReadKey::chars *STDOUT{IO};
    ($x, $y) = Term::Size::ReadKey::pixels;

=head1 DESCRIPTION

Yet another implementation of C<Term::Size>. Now using 
C<Term::ReadKey> to do the hard work.

=head2 FUNCTIONS

=over 4

=item B<chars>

    ($columns, $rows) = chars($h);
    $columns = chars($h);

C<chars> returns the terminal size in units of characters
corresponding to the given filehandle C<$h>.
If the argument is ommitted, C<*STDIN{IO}> is used.
In scalar context, it returns the terminal width.

=item B<pixels>

    ($x, $y) = pixels($h);
    $x = pixels($h);

C<pixels> returns the terminal size in units of pixels
corresponding to the given filehandle C<$h>.
If the argument is ommitted, C<*STDIN{IO}> is used.
In scalar context, it returns the terminal width.

Many systems with character-only terminals will return C<(0, 0)>.

=back

=head1 SEE ALSO

It all began with L<Term::Size> by Tim Goodwin. You may want to
have a look at:

    Term::Size
    Term::Size::Unix
    Term::Size::Win32
    Term::Size::Perl

You may as well be interested in what more C<Term::ReadKey> does.

    Term::ReadKey

It would be helpful if you send me the F<Params.pm> generated by
the probe at build time.
Please reports bugs via CPAN RT, 
http://rt.cpan.org/NoAuth/Bugs.html?Dist=Term-Size-ReadKey

=head1 AUTHOR

A. R. Ferreira, E<lt>ferreira@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by A. R. Ferreira

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

require Term::ReadKey;


# ( row, col, x, y )
sub _winsize {
    my $h = shift || *STDIN;
    return Term::ReadKey::GetTerminalSize($h);
}

sub chars {
    my @sz = _winsize(shift);
    return @sz[0, 1] if wantarray;
    return $sz[0];
}

sub pixels {
    my @sz = _winsize(shift);
    return @sz[2, 3] if wantarray;
    return $sz[2];
}

1;

