#!/usr/bin/env perl

use strict;
use warnings;

use Test::More tests => 2692;

BEGIN {use_ok('Finance::AMEX::Transaction')}

use lib '.';
use t::lib::CompareFile;

my $file = 't/data/AMEX/SAMPLE.GRRCN Delimited (US) v1.2.txt';

my $counts = {
  HEADER      => {want => 1,  have => 0},
  TRAILER     => {want => 1,  have => 0},
  SUMMARY     => {want => 1,  have => 0},
  TAXRECORD   => {want => 0,  have => 0},
  SUBMISSION  => {want => 2,  have => 0},
  TRANSACTION => {want => 40, have => 0},
  TXNPRICING  => {want => 40, have => 0},
  CHARGEBACK  => {want => 6,  have => 0},
  ADJUSTMENT  => {want => 6,  have => 0},
  FEEREVENUE  => {want => 1,  have => 0},
};

my $data = do {local $/ = undef; <DATA>};

t::lib::CompareFile::compare('GRRCN', $file, $data, $counts);

done_testing();

__DATA__
{
  "HEADER": [
    {"FILE_CREATION_DATE": "20170425", "FILE_CREATION_TIME": "044103", "FILE_ID": "GRRCN", "FILE_NAME": "SAMPLES.COM", "FILE_VERSION_NUMBER": "1.01", "RECORD_TYPE": "HEADER", "SEQUENTIAL_NUMBER": "0000000001"}
  ],
  "TRAILER": [
    {"RECORD_TYPE": "TRAILER", "SEQUENTIAL_NUMBER": "0000000001", "TOTAL_RECORD_COUNT": "0000098"}
  ],
  "SUMMARY": [
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "BANK_ACCOUNT_NUMBER": "012345678", "BANK_IDENTIFIER_CODE": "", "INTERNATIONAL_BANK_ACCOUNT_NUMBER": "", "OPENING_DEBIT_BALANCE_AMOUNT": " 000000000000000", "PAYEE_DIRECT_DEPOSIT_NUMBER": "0000000000000", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_ADJUSTMENT_AMOUNT": "-000000000000800", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "PAYMENT_DISCOUNT_AMOUNT": " 000000000000040", "PAYMENT_GROSS_AMOUNT": " 000000000004000", "PAYMENT_NET_AMOUNT": " 000000000002760", "PAYMENT_SERVICE_FEE_AMOUNT": " 000000000000400", "PAYMENT_TAX_AMOUNT": " 000000000000000", "RECORD_TYPE": "SUMMARY", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "UNIQUE_PAYMENT_REFERENCE_NUMBER": "065A1234"}
  ],
  "TAXRECORD": [
  ],
  "SUBMISSION": [
    {"ACCELERATION_DATE": "", "ACCELERATION_NUMBER": "", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "BUSINESS_SUBMISSION_DATE": "20170305", "INSTALLMENT_NUMBER": "", "NUMBER_OF_DAYS_IN_ADVANCE": "", "ORIGINAL_SETTLEMENT_DATE": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "SUBMISSION", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_ACCELERATION_FEE_AMOUNT": "", "SUBMISSION_ACCELERATION_FEE_NET_AMOUNT": "", "SUBMISSION_CREDIT_GROSS_AMOUNT": " 000000000000000", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_DEBIT_GROSS_AMOUNT": " 000000000002000", "SUBMISSION_DISCOUNT_AMOUNT": " 000000000000020", "SUBMISSION_DISCOUNT_RATE": "000100", "SUBMISSION_EXCHANGE_RATE": "", "SUBMISSION_GROSS_AMOUNT_IN_PAYMENT_CURRENCY": " 000000000002000", "SUBMISSION_GROSS_AMOUNT_IN_SUBMISSION_CURRENCY": " 000000000002000", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBMISSION_NET_AMOUNT": " 000000000001780", "SUBMISSION_SERVICE_FEE_AMOUNT": " 000000000000200", "SUBMISSION_TAX_AMOUNT": " 000000000000000", "SUBMISSION_TAX_RATE": "", "TRACKING_ID": "", "TRANSACTION_COUNT": "0000200"},
    {"ACCELERATION_DATE": "", "ACCELERATION_NUMBER": "", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "BUSINESS_SUBMISSION_DATE": "20170305", "INSTALLMENT_NUMBER": "", "NUMBER_OF_DAYS_IN_ADVANCE": "", "ORIGINAL_SETTLEMENT_DATE": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "SUBMISSION", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_ACCELERATION_FEE_AMOUNT": "", "SUBMISSION_ACCELERATION_FEE_NET_AMOUNT": "", "SUBMISSION_CREDIT_GROSS_AMOUNT": " 000000000000000", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_DEBIT_GROSS_AMOUNT": " 000000000002000", "SUBMISSION_DISCOUNT_AMOUNT": " 000000000000020", "SUBMISSION_DISCOUNT_RATE": "000100", "SUBMISSION_EXCHANGE_RATE": "", "SUBMISSION_GROSS_AMOUNT_IN_PAYMENT_CURRENCY": " 000000000002000", "SUBMISSION_GROSS_AMOUNT_IN_SUBMISSION_CURRENCY": " 000000000002000", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBMISSION_NET_AMOUNT": " 000000000001780", "SUBMISSION_SERVICE_FEE_AMOUNT": " 000000000000200", "SUBMISSION_TAX_AMOUNT": " 000000000000000", "SUBMISSION_TAX_RATE": "", "TRACKING_ID": "", "TRANSACTION_COUNT": "0000020"}
  ],
  "TRANSACTION": [
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EL", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "197292", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WA3U3JON14GS",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "143047"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EM", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "983005", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "1112005998XX4", "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "130648"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EN", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "146255", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "1112006031XXN", "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "130743"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EO", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "615187", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W9YVQOEI0FGE",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "132305"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EP", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "232301", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6DVRORZ08ES",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "123022"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EQ", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "178972", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WA8SX6S80ATP",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "170055"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217ER", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "249606", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WBXZULJZ0EI3",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "132118"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217ES", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "282681", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W60NG4RX07OA",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "022427"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217ET", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "244622", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAQZRKDK016V",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170303", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "092335"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EU", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "247540", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WBT15H7Q0715",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170303", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "212646"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EV", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "222843", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6W2MSYU02UE",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170303", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "142732"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EW", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "125967", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WANOQYQO08Z1",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "123110"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EX", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "251176", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6AKPI0P0ARE",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "190408"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EY", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "264411", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6PGJC3V04WX",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "152026"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217EZ", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "140208", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W79B192O0F20",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "152026"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217F0", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "177830", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6R44AAB0GDC",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "120625"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217F1", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "100495", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WBCHV2HP0GML",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170304", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "130724"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217F2", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "280638", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W679O5AV07FW",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "122646"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217F3", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "203274", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAUAY8NG0H6S",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "125835"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC5K331958217F4", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "127408", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W5YZY4BJ05PU",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "194206"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9F", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "267643", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6FJAZUA04I9",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "115929"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9G", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "101040", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WBPQ47ET0A5H",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "155220"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9H", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "140673", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAVYFPUR09AQ",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "124505"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9I", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "275173", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WA5HS5CZ01V4",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "103706"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9J", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "122221", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAXLYXAV08P3",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "105606"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9K", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "285962", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6C89F8V0CPL",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "023158"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9L", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "148660", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAFEYEQQ022X",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "110033"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9M", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "117086", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W711CXH40HAR",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "192319"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9N", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "145237", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WBT18XPH0JBP",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "181548"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9O", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "280357", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WADRJLPM0CJR",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "215121"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9P", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "100601", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAKDPWSP075O",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "094632"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9Q", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "222112", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAVYBWVF02K9",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "112619"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9R", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "205540", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W5F5FKW102B1",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "125512"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9S", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "297576", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W5F5DM7Y028Z",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "112846"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9T", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "102221", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W5SDUIE70IPK",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "184533"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9U", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "725863", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "WAZ9KKO20FH0",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "164609"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9V", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "908725", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W5VOX9UN0HJY",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "193854"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9W", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "163978", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W74CFMB70GE9",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "192002"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M06HC683818CMM0R9X", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "258731", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6AKP5GM098J",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "143253"},
    {"ACCELERATION_AMOUNT": "", "ACQUIRER_REFERENCE_NUMBER": "M12HC6B15092YH00LJ", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMERICAN_EXPRESS_PROCESSING_DATE": "20170305", "APPROVAL_CODE": "247103", "BUSINESS_SUBMISSION_DATE": "20170305", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CARDMEMBER_REFERENCE_NUMBER": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_1": "N", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_2": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_3": "", "DATA_QUALITY_NON_COMPLIANT_ERROR_CODE_4": "", "DATA_QUALITY_NON_COMPLIANT_INDICATOR": "", "FIRST_INSTALLMENT_AMOUNT": "", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INSTALLMENT_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "W6PGJ44B048R",  "MERCHANT_CATEGORY_CODE": "5310", "MERCHANT_LOCATION_ID": "", "NON_SWIPED_INDICATOR": "", "NUMBER_OF_INSTALLMENTS": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TRANSACTN", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_GROSS_AMOUNT": " 000000000000100", "SUBMISSION_INVOICE_NUMBER": "000054", "SUBMISSION_MERCHANT_ID": "1234567891", "SUBSEQUENT_INSTALLMENT_AMOUNT": "", "TERMINAL_ID": "", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170228", "TRANSACTION_ID": "", "TRANSACTION_REJECTED_INDICATOR": "Y", "TRANSACTION_TIME": "000000"}
  ],
  "TXNPRICING": [
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WA3U3JON14GS",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "1112005998XX4", "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "1112006031XXN", "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W9YVQOEI0FGE",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6DVRORZ08ES",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WA8SX6S80ATP",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WBXZULJZ0EI3",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W60NG4RX07OA",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAQZRKDK016V",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WBT15H7Q0715",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6W2MSYU02UE",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WANOQYQO08Z1",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6AKPI0P0ARE",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6PGJC3V04WX",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W79B192O0F20",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6R44AAB0GDC",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WBCHV2HP0GML",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W679O5AV07FW",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAUAY8NG0H6S",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W5YZY4BJ05PU",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6FJAZUA04I9",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WBPQ47ET0A5H",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAVYFPUR09AQ",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WA5HS5CZ01V4",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAXLYXAV08P3",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6C89F8V0CPL",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAFEYEQQ022X",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W711CXH40HAR",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WBT18XPH0JBP",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WADRJLPM0CJR",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAKDPWSP075O",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAVYBWVF02K9",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W5F5FKW102B1",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W5F5DM7Y028Z",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W5SDUIE70IPK",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "WAZ9KKO20FH0",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W5VOX9UN0HJY",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W74CFMB70GE9",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6AKP5GM098J",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000010000", "DISCOUNT_RATE": " 001000", "FEE_AMOUNT": " 000000000000000100000", "FEE_CODE": "", "INVOICE_REFERENCE_NUMBER": "W6PGJ44B048R",  "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "TXNPRICING", "SELLER_ID": "", "SETTLEMENT_ACCOUNT_TYPE_CODE": "002", "SUBMISSION_MERCHANT_ID": "1234567891", "TRANSACTION_AMOUNT": " 000000000000100", "TRANSACTION_DATE": "20170305"}
  ],
  "CHARGEBACK": [
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "323350", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CREDIT ADJUSTMENT-REVERSAL OF PREVIOUS DEBIT", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": " 000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK01", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": " 000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "323350", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "324871", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CHARGEBACK - FRAUD RELATED",                   "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK02", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "324871", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "325364", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CHARGEBACK - CARDMEMBER DISPUTE",              "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK03", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325364", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "325558", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CHARGEBACK - FRAUD RELATED",                   "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK04", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325558", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "325682", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CHARGEBACK - FRAUD RELATED",                   "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK05", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325682", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "CHARGEBACK_NUMBER": "326296", "CHARGEBACK_REASON_CODE": "", "CHARGEBACK_REASON_DESCRIPTION": "CHARGEBACK - FRAUD RELATED",                   "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "CHARGEBACK06", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "CHARGEBACK", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "326296", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"}
  ],
  "ADJUSTMENT": [
    {"ADJUSTMENT_NUMBER": "323350", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "CREDIT ADJUSTMENT-REVERSAL OF PREVIOUS DEBIT", "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": " 000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT01", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": " 000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "323350", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"ADJUSTMENT_NUMBER": "324871", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "ADJUSTMENT - FRAUD ADJUSTMENT",                "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT02", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "324871", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"ADJUSTMENT_NUMBER": "325364", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "ADJUSTMENT - ACCOUNT ADJUSTMENT",              "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT03", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325364", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"ADJUSTMENT_NUMBER": "325558", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "ADJUSTMENT - FRAUD ADJUSTMENT",                "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT04", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325558", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"ADJUSTMENT_NUMBER": "325682", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "ADJUSTMENT - FRAUD ADJUSTMENT",                "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT05", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "325682", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"},
    {"ADJUSTMENT_NUMBER": "326296", "ADJUSTMENT_REASON_CODE": "", "ADJUSTMENT_REASON_DESCRIPTION": "ADJUSTMENT - FRAUD ADJSUTMENT",                "AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "AMEX_PROCESSING_DATE": "20170307", "BATCH_CODE": "566", "BILL_CODE": "652", "BUSINESS_SUBMISSION_DATE": "20170306", "CARDMEMBER_ACCOUNT_NUMBER": "123456XXXXX1234", "DISCOUNT_AMOUNT": " 000000000000000", "DISCOUNT_RATE": " 000100", "GROSS_AMOUNT": "-000000000000100", "INDUSTRY_SPECIFIC_REFERENCE_NUMBER": "", "INVOICE_REFERENCE_NUMBER": "ADJUSTMENT06", "MERCHANT_LOCATION_ID": "", "NET_AMOUNT": "-000000000000100", "PAYEE_MERCHANT_ID": "1234567891", "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "ADJUSTMENT", "SELLER_ID": "", "SERVICE_FEE_AMOUNT": " 000000000000000", "SERVICE_FEE_RATE": " 000000", "SETTLEMENT_ACCOUNT_TYPE_CODE": "", "SUBMISSION_CURRENCY": "USD", "SUBMISSION_INVOICE_NUMBER": "326296", "SUBMISSION_MERCHANT_ID": "1234567891", "TAX_AMOUNT": " 000000000000000"}
  ],
  "FEEREVENUE": [
    {"AMERICAN_EXPRESS_PAYMENT_NUMBER": "065A6808", "ASSET_BILLING_AMOUNT": " 000000000000000", "ASSET_BILLING_DESCRIPTION": "", "ASSET_BILLING_TAX": "", "BATCH_CODE": "", "BILL_CODE": "", "FEE_OR_REVENUE_AMOUNT": " 000000000000100", "FEE_OR_REVENUE_DESCRIPTION": "NON-SWIPED ADJUSTMENT", "MERCHANT_LOCATION_ID": "", "PAYEE_MERCHANT_ID": "1234567891", "PAY_IN_GROSS_INDICATOR": 0, "PAYMENT_CURRENCY": "USD", "PAYMENT_DATE": "20170307", "RECORD_TYPE": "FEEREVENUE", "SUBMISSION_MERCHANT_ID": "1234567891"}
  ]
}
