package Pluto::Engine;

use 5.014;

use strict;
use warnings;

use registry;
use routines;

use Data::Object::Class;
use Data::Object::ClassHas;
use Data::Object::Space;
use Rewire;

our $VERSION = '0.05'; # VERSION

# ATTRIBUTES

has 'package' => (
  is => 'ro',
  isa => 'Str',
  req => 1,
);

has 'config' => (
  is => 'ro',
  isa => 'HashRef',
  opt => 1,
);

has 'library' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_library($self) {
  $self->space->package->meta;
}

has 'services' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_services($self) {
  Rewire->new(services => $self->config);
}

has 'space' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_space($self) {
  Data::Object::Space->new($self->package);
}

# METHODS

method call(Str | Object $invocant, Str $routine, Any @arguments) {
  if (Scalar::Util::blessed($invocant)) {
    return $invocant->$routine(@arguments);
  }
  else {
    my $space = Data::Object::Space->new($invocant);
    my $package = $space->load;

    if (my $routine = $package->can($routine)) {
      return $routine->(@arguments);
    }

    if ($package->can('AUTOLOAD')) {
      no strict 'refs';
      return &{"${package}::${routine}"}(@arguments);
    }
  }
}

method chain(Str | Object $invocant, Str | ArrayRef @routines) {
  for my $next (map +(ref($_) eq 'ARRAY' ? $_ : [$_]), @routines) {
    $invocant = $self->call($invocant, @$next);
  }

  return $invocant;
}

1;

=encoding utf8

=head1 NAME

Pluto::Engine

=cut

=head1 ABSTRACT

Functional Scripting for Perl 5

=cut

=head1 SYNOPSIS

  package Example;

  use Pluto::Engine;

  my $engine = Pluto::Engine->new(
    package => 'example',
    config => { tempdir => { package => 'File/Temp' } },
  );

=cut

=head1 DESCRIPTION

This package provides an engine for resolving dependencies and dispatching
function calls, enabling a functional programming environment for Perl 5. See
L<Rewire> for more information on dependency injection.

=cut

=head1 LIBRARIES

This package uses type constraints from:

L<Types::Standard>

=cut

=head1 ATTRIBUTES

This package has the following attributes:

=cut

=head2 config

  config(HashRef)

This attribute is read-only, accepts C<(HashRef)> values, and is optional.

=cut

=head2 library

  library(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head2 package

  package(Str)

This attribute is read-only, accepts C<(Str)> values, and is required.

=cut

=head2 services

  services(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head2 space

  space(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head1 METHODS

This package implements the following methods:

=cut

=head2 call

  call(Str | Object $invocant, Str $routine, Any @arguments) : Any

The call method dispatches a routine call.

=over 4

=item call example #1

  # given: synopsis

  $engine->call('File::Temp', 'new', 'File::Temp');

=back

=cut

=head2 chain

  chain(Str | Object $invocant, Str | ArrayRef @routines) : Any

The chain method chains a routine calls on the invocant and routine names
provided.

=over 4

=item chain example #1

  # given: synopsis

  $engine->chain(path('/var'), ['child', 'logs'], 'absolute');

=back

=cut

=head1 AUTHOR

Al Newkirk, C<awncorp@cpan.org>

=head1 LICENSE

Copyright (C) 2011-2019, Al Newkirk, et al.

This is free software; you can redistribute it and/or modify it under the terms
of the The Apache License, Version 2.0, as elucidated in the L<"license
file"|https://github.com/iamalnewkirk/pluto/blob/master/LICENSE>.

=head1 PROJECT

L<Wiki|https://github.com/iamalnewkirk/pluto/wiki>

L<Project|https://github.com/iamalnewkirk/pluto>

L<Initiatives|https://github.com/iamalnewkirk/pluto/projects>

L<Milestones|https://github.com/iamalnewkirk/pluto/milestones>

L<Contributing|https://github.com/iamalnewkirk/pluto/blob/master/CONTRIBUTE.md>

L<Issues|https://github.com/iamalnewkirk/pluto/issues>

=cut