package Lingua::ZH::MMSEG;
use strict;
use warnings;
use utf8;
use Encode qw (is_utf8);
use encoding 'utf8';
use List::Util qw(sum);

our $VERSION=0.4000;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(mmseg fmm);

=encoding utf8

=head1 NAME

Lingua::ZH::MMSEG Mandarin Chinese segmentation

=head1 SYNOPSIS

    #!/usr/bin/perl
    use utf8;
    use Lingua::ZH::MMSEG;

    my $zh_string="現代漢語的複合動詞可分三個結構語意關係來探討";

    my @phrases = mmseg($zh_string);
    # use MMSEG algorithm

    my @phrases = fmm($zh_string);
    # use Forward Maximum Matching algorithm

=head1 DESCRIPTION

A problem in computational analysis of Chinese text is that there are no word
boundaries in conventionally printed text. Since the word is such a fundamental
linguistic unit, it is necessary to identify words in Chinese text so that
higher-level analyses can be performed. 

Lingua::ZH::MMSEG implements L<MMSEG|http://technology.chtsai.org/mmseg/>
original developed by L<Chih-Hao-Tsai|http://chtsai.org/>. The whole module is
rewritten in pure Perl, and the phrase library is 新酷音 forked from
L<OpenFoundry|http://www.openfoundry.org/of/projects/436>.

=head1 INSTALL

To install this module, just type

    cpanm Lingua::ZH::MMSEG

If you don't have cpanm,

    curl -LO http://bit.ly/cpanm
    chmod +x cpanm
    sudo cp cpanm /usr/local/bin

=head1 USAGE

Since this module has no dependency at all, you just simply create a new perl
script as shown in SYNOPSIS.

=head1 FUNCTIONS

=head2 C<mmseg>

    my @phrases = mmseg($zh_string);

Use L<MMSEG|http://technology.chtsai.org/mmseg/> algorithm to generate segmented
chinese phrases.

=head2 C<fmm>

    my @phrases = fmm($zh_string);

Use forward maximum matching algorithm to generate segmented chinese phrases.
It has lower complexity compare to mmseg, but it cannot solve phrase ambiguities.

=head1 AUTHOR

Felix Ren-Chyan Chern (dryman) C<< <idryman@gmail.com> >>

=head1 LICENSE AND COPYRIGHT

L<GNU Lesser General Public License 2.1 |http://www.opensource.org/licenses/lgpl-2.1.php>

=cut

our %dict;

while (<DATA>){
  chomp;
  my ($phrase,$freq) = split;
  $dict{$phrase}=$freq;
}

sub mmseg {
  my $string = shift;
  my @phrases;
  die unless is_utf8($string);
  chomp ($string);
  for my $str (split (/([ -~]+)/, $string)) {
    if ($str =~ /^[ -~]/) {
      push @phrases, $str;
      next;
    }
    while($str){
      my $word1 = &_mmseg($str);
      push @phrases, $word1;
      $str = substr $str, length $word1;
    }
  }
  return @phrases;
}

sub fmm {
  my $string = shift;
  my @phrases;
  die unless is_utf8($string);
  chomp ($string);
  for my $str (split (/([ -~])+/, $string)) {
    if ($str =~ /^[ -~]/) {
      push @phrases, $str,
      next;
    }
    while($str){
      for (reverse (1..(length $str))) {
        my $match = substr $str, 0, $_;
        if (defined $dict{$match} or $_==1){
          push @phrases, $match;
          $str = substr $str, $_;
          last;
        }
      }
    }
  }
  return @phrases;
}

 
sub _mmseg {
  my $str = shift;
  my @chunk = &_findChunk($str);

  return $chunk[0]->{w1} if $#chunk == 0;

  # rule 1, find max length chunks
  my @mlc_tmp = sort {$b->{len} <=> $a->{len}} @chunk;
  my @max_len_chunk = grep {$_->{len} == $mlc_tmp[0]->{len}} @mlc_tmp;
  return $max_len_chunk[0]->{w1} if $#max_len_chunk == 0;

  # rule 2, find max avg length chunks
  my @malc_tmp = sort {$b->{avglen} <=> $a->{avglen}} @max_len_chunk;
  my @max_avglen_chunk = grep {$_->{avglen} == $malc_tmp[0]->{avglen}} @malc_tmp;
  return $max_avglen_chunk[0]->{w1} if $#max_avglen_chunk == 0;

  # rule 3, smallest varience
  for (@max_len_chunk) {
    my $avg = $_->{avglen};
    my @word = ($_->{w1},$_->{w2},$_->{w3});
    pop @word unless $word[$#word];
    my @len = map {length $_} @word;
    my $varience = sqrt ((sum (map {$_**2-$avg**2} @len))/(scalar @len));
    $_->{varience} = $varience;
  }
  my @mvc_tmp = sort {$a->{varience} <=> $b->{varience}} @max_len_chunk;
  my @min_varience_chunk = 
    grep {abs($_->{varience} - $mvc_tmp[0]->{varience})<0.01} @mvc_tmp;
  return $min_varience_chunk[0]->{w1} if $#min_varience_chunk == 0;
  # rule 4, check length one word and choose max freq of it
  for (@min_varience_chunk) {
    my $freq = 0;
    $freq += $dict{$_->{w1}} if length $_->{w1} == 1;
    $freq += $dict{$_->{w2}} if length $_->{w2} == 1;
    $freq += $dict{$_->{w3}} if length $_->{w3} == 1;
    $_->{freq} = $freq;
  }
  my @last = sort {$b->{freq} <=> $a->{freq}} @min_varience_chunk;
  return $last[0]->{w1};
}

    


sub _findChunk{
  my $str = shift;
  my $index = 0;
  my @chunk;

  my @word1;
  # will fail if $str=""
  for (1..(length($str) - $index)){
    my $substr = substr $str, $index, $_;
    push @word1, $substr if defined $dict{$substr} or $_==1;
  }
  foreach my $w1 (@word1){
    my $l1 = length $w1;
    my $index = $index + $l1;
    my @word2;
    
    if (length($str) - $index == 0){
      push @chunk, {
        w1 => $w1,
        w2 => undef,
        w3 => undef,
        len => $l1,
        avglen => $l1,
      };
      next;
    }

    for (1..(length($str) - $index)){
      my $substr = substr $str, $index, $_;
      push @word2, $substr if defined $dict{$substr} or $_==1;
    }

    foreach my $w2 (@word2){
      my $l2 = length $w2;
      my $index = $index + $l2;

      if (length($str) - $index == 0){
        push @chunk, {
          w1 => $w1,
          w2 => $w2,
          w3 => undef,
          len => $l1+$l2,
          avglen => ($l1+$l2)/2,
        };
        next;
      }

      for (1..(length($str) - $index)){
        my $substr = substr $str, $index, $_;
        if (defined $dict{$substr} or $_==1){
          my $w3 = $substr;
          my $l3 = length $w3;
          push @chunk, {
            w1 => $w1,
            w2 => $w2,
            w3 => $w3,
            len => $l1+$l2+$l3,
            avglen => ($l1+$l2+$l3)/3,
          };
        }
      }
    }
  }
  return @chunk;
}

1;

__DATA__
ˇ 1
ˊ 1
ˋ 1
˙ 1
○ 1
ㄅ 0
ㄆ 0
ㄇ 0
ㄈ 0
ㄉ 0
ㄊ 0
ㄋ 0
ㄌ 0
ㄍ 0
ㄎ 0
ㄏ 0
ㄐ 0
ㄑ 0
ㄒ 0
ㄓ 1
ㄔ 0
ㄕ 1
ㄖ 1
ㄗ 1
ㄘ 1
ㄙ 1
ㄚ 1
ㄛ 0
ㄜ 1
ㄝ 0
ㄞ 1
ㄟ 1
ㄠ 1
ㄡ 1
ㄢ 1
ㄣ 1
ㄤ 1
ㄥ 1
ㄦ 1
ㄧ 1
ㄨ 1
ㄩ 1
㐀 0
㐁 0
㐄 0
㐅 0
㐌 0
㐖 0
㐜 0
㐡 0
㐡 0
㐤 0
㐨 0
㐩 0
㐫 0
㐬 0
㐭 0
㐮 0
㐰 0
㐰 0
㐱 0
㐲 0
㐳 0
㐴 0
㐸 0
㐹 0
㐹 0
㐺 0
㐻 0
㐼 0
㑁 0
㑂 0
㑃 0
㑄 0
㑅 0
㑉 0
㑊 0
㑋 0
㑌 0
㑍 0
㑎 0
㑏 0
㑐 0
㑑 0
㑑 0
㑗 0
㑘 0
㑙 0
㑚 0
㑛 0
㑜 0
㑜 0
㑜 0
㑝 0
㑞 0
㑟 0
㑣 0
㑤 0
㑥 0
㑦 0
㑧 0
㑮 0
㑯 0
㑰 0
㑱 0
㑳 0
㑴 0
㑴 0
㑵 0
㑶 0
㑷 0
㑹 0
㑹 0
㑹 0
㑺 0
㑺 0
㑻 0
㑼 0
㑽 0
㑾 0
㑿 0
㒀 0
㒀 0
㒁 0
㒂 0
㒃 0
㒄 0
㒅 0
㒆 0
㒆 0
㒇 0
㒈 0
㒉 0
㒉 0
㒊 0
㒌 0
㒍 0
㒎 0
㒑 0
㒒 0
㒓 0
㒔 0
㒕 0
㒖 0
㒗 0
㒙 0
㒚 0
㒛 0
㒜 0
㒝 0
㒝 0
㒞 0
㒞 0
㒟 0
㒠 0
㒡 0
㒤 0
㒤 0
㒥 0
㒦 0
㒦 0
㒧 0
㒨 0
㒩 0
㒪 0
㒫 0
㒭 0
㒯 0
㒯 0
㒰 0
㒲 0
㒳 0
㒳 0
㒴 0
㒵 0
㒶 0
㒷 0
㒷 0
㒸 0
㒹 0
㒺 0
㒻 0
㒼 0
㒽 0
㒾 0
㒿 0
㓂 0
㓃 0
㓄 0
㓅 0
㓇 0
㓇 0
㓈 0
㓉 0
㓊 0
㓋 0
㓌 0
㓎 0
㓎 0
㓏 0
㓐 0
㓑 0
㓓 0
㓔 0
㓕 0
㓖 0
㓗 0
㓘 0
㓚 0
㓛 0
㓜 0
㓝 0
㓞 0
㓞 0
㓟 0
㓠 0
㓠 0
㓡 0
㓢 0
㓣 0
㓣 0
㓤 0
㓧 0
㓨 0
㓩 0
㓬 0
㓭 0
㓮 0
㓯 0
㓯 0
㓱 0
㓲 0
㓳 0
㓴 0
㓵 0
㓶 0
㓷 0
㓸 0
㓹 0
㓹 0
㓺 0
㓺 0
㓻 0
㓼 0
㓽 0
㓾 0
㓾 0
㔀 0
㔁 0
㔂 0
㔃 0
㔃 0
㔄 0
㔅 0
㔆 0
㔇 0
㔊 0
㔊 0
㔋 0
㔋 0
㔌 0
㔍 0
㔎 0
㔏 0
㔐 0
㔑 0
㔒 0
㔓 0
㔕 0
㔗 0
㔗 0
㔘 0
㔘 0
㔙 0
㔚 0
㔜 0
㔝 0
㔝 0
㔞 0
㔠 0
㔠 0
㔡 0
㔢 0
㔢 0
㔣 0
㔤 0
㔥 0
㔦 0
㔧 0
㔨 0
㔩 0
㔪 0
㔬 0
㔭 0
㔭 0
㔮 0
㔯 0
㔯 0
㔰 0
㔱 0
㔲 0
㔳 0
㔴 0
㔵 0
㔶 0
㔷 0
㔸 0
㔹 0
㔺 0
㔻 0
㔻 0
㔼 0
㔽 0
㔾 0
㔿 0
㕀 0
㕁 0
㕂 0
㕃 0
㕄 0
㕅 0
㕆 0
㕇 0
㕈 0
㕉 0
㕊 0
㕋 0
㕌 0
㕍 0
㕍 0
㕎 0
㕏 0
㕐 0
㕐 0
㕑 0
㕒 0
㕔 0
㕕 0
㕖 0
㕗 0
㕘 0
㕘 0
㕘 0
㕘 0
㕙 0
㕚 0
㕛 0
㕜 0
㕞 0
㕟 0
㕠 0
㕠 0
㕡 0
㕡 0
㕢 0
㕣 0
㕤 0
㕤 0
㕥 0
㕦 0
㕧 0
㕩 0
㕪 0
㕫 0
㕬 0
㕬 0
㕮 0
㕯 0
㕰 0
㕰 0
㕱 0
㕲 0
㕲 0
㕳 0
㕶 0
㕷 0
㕸 0
㕹 0
㕺 0
㕺 0
㕻 0
㕼 0
㖀 0
㖁 0
㖂 0
㖃 0
㖄 0
㖅 0
㖆 0
㖇 0
㖏 0
㖐 0
㖑 0
㖒 0
㖓 0
㖔 0
㖕 0
㖖 0
㖗 0
㖘 0
㖟 0
㖠 0
㖠 0
㖠 0
㖠 0
㖡 0
㖢 0
㖣 0
㖤 0
㖥 0
㖦 0
㖧 0
㖧 0
㖨 0
㖩 0
㖩 0
㖪 0
㖪 0
㖪 0
㖴 0
㖵 0
㖶 0
㖷 0
㖸 0
㖸 0
㖸 0
㖹 0
㖺 0
㖻 0
㖼 0
㖽 0
㖾 0
㖿 0
㗀 0
㗁 0
㗂 0
㗃 0
㗄 0
㗅 0
㗆 0
㗇 0
㗈 0
㗉 0
㗊 0
㗋 0
㗌 0
㗍 0
㗐 0
㗐 0
㗐 0
㗓 0
㗔 0
㗕 0
㗖 0
㗗 0
㗘 0
㗙 0
㗚 0
㗛 0
㗜 0
㗢 0
㗢 0
㗢 0
㗣 0
㗤 0
㗤 0
㗥 0
㗥 0
㗦 0
㗧 0
㗧 0
㗨 0
㗨 0
㗩 0
㗪 0
㗫 0
㗬 0
㗭 0
㗰 0
㗱 0
㗲 0
㗳 0
㗴 0
㗵 0
㗵 0
㗶 0
㗻 0
㗼 0
㗽 0
㗾 0
㗿 0
㘀 0
㘁 0
㘂 0
㘅 0
㘆 0
㘇 0
㘈 0
㘈 0
㘉 0
㘊 0
㘋 0
㘌 0
㘍 0
㘐 0
㘑 0
㘓 0
㘔 0
㘕 0
㘖 0
㘙 0
㘚 0
㘛 0
㘜 0
㘝 0
㘞 0
㘞 0
㘟 0
㘠 0
㘡 0
㘡 0
㘢 0
㘣 0
㘤 0
㘥 0
㘥 0
㘦 0
㘧 0
㘨 0
㘨 0
㘩 0
㘪 0
㘫 0
㘬 0
㘬 0
㘬 0
㘯 0
㘯 0
㘰 0
㘱 0
㘲 0
㘳 0
㘴 0
㘶 0
㘷 0
㘸 0
㘹 0
㘺 0
㘻 0
㘻 0
㘿 0
㙀 0
㙁 0
㙂 0
㙃 0
㙄 0
㙅 0
㙇 0
㙈 0
㙉 0
㙉 0
㙊 0
㙍 0
㙎 0
㙏 0
㙐 0
㙑 0
㙒 0
㙓 0
㙕 0
㙙 0
㙚 0
㙛 0
㙜 0
㙝 0
㙞 0
㙡 0
㙢 0
㙣 0
㙤 0
㙤 0
㙥 0
㙧 0
㙨 0
㙨 0
㙩 0
㙪 0
㙫 0
㙬 0
㙯 0
㙰 0
㙱 0
㙲 0
㙲 0
㙳 0
㙳 0
㙴 0
㙵 0
㙶 0
㙷 0
㙸 0
㙹 0
㙺 0
㙻 0
㙼 0
㙼 0
㙾 0
㚀 0
㚁 0
㚂 0
㚃 0
㚄 0
㚄 0
㚅 0
㚇 0
㚉 0
㚊 0
㚋 0
㚌 0
㚍 0
㚎 0
㚏 0
㚐 0
㚒 0
㚓 0
㚔 0
㚕 0
㚖 0
㚗 0
㚘 0
㚚 0
㚜 0
㚜 0
㚝 0
㚝 0
㚞 0
㚟 0
㚠 0
㚡 0
㚢 0
㚣 0
㚣 0
㚤 0
㚥 0
㚦 0
㚧 0
㚨 0
㚩 0
㚪 0
㚫 0
㚬 0
㚭 0
㚮 0
㚰 0
㚱 0
㚲 0
㚲 0
㚴 0
㚵 0
㚶 0
㚷 0
㚹 0
㚺 0
㚻 0
㚼 0
㚽 0
㚾 0
㚿 0
㛁 0
㛂 0
㛃 0
㛅 0
㛊 0
㛍 0
㛎 0
㛏 0
㛐 0
㛑 0
㛒 0
㛓 0
㛔 0
㛕 0
㛖 0
㛗 0
㛘 0
㛙 0
㛚 0
㛚 0
㛛 0
㛛 0
㛛 0
㛜 0
㛝 0
㛞 0
㛥 0
㛦 0
㛧 0
㛨 0
㛩 0
㛩 0
㛪 0
㛪 0
㛫 0
㛬 0
㛭 0
㛮 0
㛯 0
㛰 0
㛱 0
㛱 0
㛲 0
㛳 0
㛴 0
㛵 0
㛶 0
㛷 0
㛸 0
㛹 0
㛺 0
㛻 0
㛼 0
㛼 0
㛽 0
㛽 0
㛾 0
㜂 0
㜃 0
㜄 0
㜅 0
㜆 0
㜇 0
㜈 0
㜉 0
㜊 0
㜋 0
㜌 0
㜍 0
㜎 0
㜏 0
㜐 0
㜑 0
㜒 0
㜓 0
㜕 0
㜖 0
㜗 0
㜘 0
㜘 0
㜘 0
㜙 0
㜚 0
㜛 0
㜜 0
㜝 0
㜞 0
㜞 0
㜠 0
㜡 0
㜢 0
㜣 0
㜤 0
㜤 0
㜤 0
㜤 0
㜥 0
㜦 0
㜧 0
㜨 0
㜩 0
㜪 0
㜫 0
㜮 0
㜮 0
㜯 0
㜰 0
㜱 0
㜲 0
㜳 0
㜴 0
㜵 0
㜶 0
㜷 0
㜷 0
㜸 0
㜹 0
㜺 0
㜻 0
㜻 0
㜼 0
㜽 0
㜽 0
㜾 0
㜿 0
㝀 0
㝀 0
㝁 0
㝂 0
㝂 0
㝃 0
㝃 0
㝄 0
㝄 0
㝅 0
㝇 0
㝈 0
㝊 0
㝊 0
㝌 0
㝍 0
㝎 0
㝏 0
㝐 0
㝑 0
㝒 0
㝒 0
㝓 0
㝔 0
㝔 0
㝔 0
㝕 0
㝕 0
㝖 0
㝗 0
㝗 0
㝘 0
㝙 0
㝛 0
㝛 0
㝛 0
㝜 0
㝝 0
㝞 0
㝟 0
㝠 0
㝢 0
㝣 0
㝤 0
㝥 0
㝦 0
㝧 0
㝩 0
㝪 0
㝬 0
㝭 0
㝮 0
㝯 0
㝰 0
㝱 0
㝲 0
㝴 0
㝵 0
㝵 0
㝶 0
㝷 0
㝷 0
㝷 0
㝸 0
㝹 0
㝺 0
㝻 0
㝽 0
㝾 0
㝿 0
㞀 0
㞁 0
㞂 0
㞃 0
㞅 0
㞆 0
㞆 0
㞇 0
㞈 0
㞊 0
㞋 0
㞌 0
㞏 0
㞏 0
㞐 0
㞐 0
㞑 0
㞑 0
㞒 0
㞒 0
㞓 0
㞓 0
㞔 0
㞕 0
㞕 0
㞕 0
㞖 0
㞗 0
㞘 0
㞚 0
㞛 0
㞜 0
㞟 0
㞡 0
㞣 0
㞤 0
㞤 0
㞥 0
㞦 0
㞧 0
㞧 0
㞧 0
㞧 0
㞨 0
㞩 0
㞩 0
㞪 0
㞫 0
㞬 0
㞭 0
㞯 0
㞰 0
㞱 0
㞲 0
㞳 0
㞴 0
㞵 0
㞶 0
㞷 0
㞷 0
㞹 0
㞺 0
㞻 0
㞼 0
㞽 0
㞾 0
㞿 0
㟀 0
㟁 0
㟄 0
㟅 0
㟇 0
㟈 0
㟉 0
㟊 0
㟋 0
㟌 0
㟍 0
㟎 0
㟎 0
㟏 0
㟐 0
㟑 0
㟒 0
㟓 0
㟔 0
㟖 0
㟗 0
㟗 0
㟘 0
㟙 0
㟚 0
㟛 0
㟛 0
㟛 0
㟛 0
㟜 0
㟝 0
㟝 0
㟞 0
㟟 0
㟠 0
㟠 0
㟥 0
㟦 0
㟧 0
㟨 0
㟩 0
㟪 0
㟫 0
㟬 0
㟭 0
㟯 0
㟰 0
㟱 0
㟲 0
㟳 0
㟴 0
㟵 0
㟶 0
㟸 0
㟹 0
㟺 0
㟻 0
㟼 0
㟽 0
㟽 0
㟾 0
㟿 0
㟿 0
㠀 0
㠁 0
㠂 0
㠄 0
㠅 0
㠆 0
㠇 0
㠈 0
㠉 0
㠊 0
㠋 0
㠌 0
㠍 0
㠎 0
㠏 0
㠐 0
㠐 0
㠑 0
㠒 0
㠒 0
㠓 0
㠔 0
㠕 0
㠖 0
㠗 0
㠘 0
㠙 0
㠚 0
㠛 0
㠜 0
㠜 0
㠝 0
㠠 0
㠡 0
㠢 0
㠥 0
㠦 0
㠧 0
㠨 0
㠩 0
㠪 0
㠭 0
㠮 0
㠯 0
㠱 0
㠱 0
㠲 0
㠴 0
㠵 0
㠶 0
㠶 0
㠷 0
㠸 0
㠹 0
㠺 0
㠻 0
㠽 0
㠾 0
㡀 0
㡁 0
㡂 0
㡃 0
㡃 0
㡄 0
㡄 0
㡄 0
㡅 0
㡆 0
㡇 0
㡇 0
㡈 0
㡈 0
㡈 0
㡉 0
㡊 0
㡋 0
㡌 0
㡎 0
㡏 0
㡏 0
㡏 0
㡏 0
㡏 0
㡐 0
㡑 0
㡒 0
㡓 0
㡓 0
㡔 0
㡔 0
㡔 0
㡔 0
㡔 0
㡕 0
㡖 0
㡗 0
㡘 0
㡙 0
㡚 0
㡛 0
㡜 0
㡝 0
㡞 0
㡟 0
㡠 0
㡠 0
㡢 0
㡣 0
㡣 0
㡥 0
㡦 0
㡧 0
㡧 0
㡨 0
㡩 0
㡪 0
㡫 0
㡫 0
㡭 0
㡮 0
㡯 0
㡯 0
㡰 0
㡱 0
㡲 0
㡳 0
㡳 0
㡴 0
㡵 0
㡶 0
㡷 0
㡸 0
㡸 0
㡹 0
㡺 0
㡻 0
㡽 0
㡾 0
㡿 0
㢀 0
㢁 0
㢂 0
㢂 0
㢃 0
㢄 0
㢅 0
㢆 0
㢈 0
㢉 0
㢊 0
㢋 0
㢍 0
㢎 0
㢏 0
㢏 0
㢑 0
㢒 0
㢓 0
㢔 0
㢕 0
㢖 0
㢖 0
㢗 0
㢘 0
㢙 0
㢙 0
㢚 0
㢛 0
㢜 0
㢝 0
㢞 0
㢟 0
㢠 0
㢡 0
㢣 0
㢥 0
㢦 0
㢧 0
㢨 0
㢩 0
㢬 0
㢮 0
㢮 0
㢯 0
㢯 0
㢰 0
㢲 0
㢲 0
㢳 0
㢴 0
㢵 0
㢶 0
㢸 0
㢺 0
㢻 0
㢼 0
㢼 0
㢽 0
㢾 0
㣀 0
㣁 0
㣂 0
㣃 0
㣄 0
㣅 0
㣆 0
㣇 0
㣇 0
㣈 0
㣉 0
㣋 0
㣌 0
㣍 0
㣎 0
㣏 0
㣐 0
㣑 0
㣒 0
㣓 0
㣔 0
㣔 0
㣘 0
㣙 0
㣚 0
㣛 0
㣜 0
㣝 0
㣞 0
㣟 0
㣠 0
㣠 0
㣢 0
㣤 0
㣥 0
㣦 0
㣦 0
㣫 0
㣫 0
㣬 0
㣭 0
㣯 0
㣰 0
㣱 0
㣲 0
㣵 0
㣶 0
㣷 0
㣺 0
㣻 0
㣼 0
㣽 0
㣾 0
㣿 0
㣿 0
㤁 0
㤂 0
㤃 0
㤃 0
㤃 0
㤄 0
㤄 0
㤅 0
㤆 0
㤇 0
㤈 0
㤉 0
㤊 0
㤊 0
㤋 0
㤌 0
㤍 0
㤎 0
㤏 0
㤐 0
㤐 0
㤑 0
㤒 0
㤓 0
㤔 0
㤕 0
㤖 0
㤚 0
㤛 0
㤛 0
㤜 0
㤝 0
㤞 0
㤞 0
㤞 0
㤟 0
㤠 0
㤡 0
㤡 0
㤢 0
㤣 0
㤤 0
㤥 0
㤦 0
㤧 0
㤨 0
㤩 0
㤪 0
㤪 0
㤬 0
㤮 0
㤯 0
㤰 0
㤱 0
㤲 0
㤳 0
㤴 0
㤴 0
㤵 0
㤶 0
㤷 0
㤸 0
㤹 0
㤺 0
㤺 0
㤺 0
㤻 0
㤻 0
㤾 0
㤾 0
㤿 0
㥀 0
㥀 0
㥁 0
㥂 0
㥃 0
㥃 0
㥄 0
㥄 0
㥅 0
㥆 0
㥇 0
㥈 0
㥉 0
㥊 0
㥊 0
㥌 0
㥍 0
㥍 0
㥍 0
㥎 0
㥏 0
㥐 0
㥐 0
㥑 0
㥒 0
㥒 0
㥓 0
㥔 0
㥕 0
㥖 0
㥚 0
㥛 0
㥜 0
㥝 0
㥞 0
㥟 0
㥠 0
㥡 0
㥢 0
㥣 0
㥤 0
㥥 0
㥦 0
㥦 0
㥧 0
㥨 0
㥩 0
㥫 0
㥬 0
㥭 0
㥭 0
㥮 0
㥯 0
㥰 0
㥱 0
㥲 0
㥳 0
㥴 0
㥴 0
㥵 0
㥶 0
㥶 0
㥷 0
㥸 0
㥹 0
㥺 0
㥺 0
㥼 0
㥼 0
㥽 0
㥽 0
㥾 0
㥿 0
㦀 0
㦁 0
㦁 0
㦂 0
㦃 0
㦄 0
㦅 0
㦅 0
㦆 0
㦇 0
㦉 0
㦉 0
㦊 0
㦋 0
㦌 0
㦍 0
㦎 0
㦏 0
㦏 0
㦐 0
㦑 0
㦒 0
㦓 0
㦔 0
㦔 0
㦕 0
㦖 0
㦖 0
㦗 0
㦘 0
㦚 0
㦛 0
㦛 0
㦜 0
㦝 0
㦞 0
㦟 0
㦡 0
㦢 0
㦣 0
㦤 0
㦤 0
㦥 0
㦥 0
㦦 0
㦧 0
㦨 0
㦩 0
㦪 0
㦫 0
㦬 0
㦭 0
㦮 0
㦮 0
㦯 0
㦰 0
㦲 0
㦴 0
㦵 0
㦶 0
㦷 0
㦸 0
㦸 0
㦹 0
㦺 0
㦻 0
㦼 0
㦽 0
㦾 0
㦿 0
㦿 0
㧀 0
㧁 0
㧁 0
㧂 0
㧃 0
㧃 0
㧄 0
㧅 0
㧆 0
㧆 0
㧇 0
㧈 0
㧉 0
㧊 0
㧊 0
㧋 0
㧌 0
㧍 0
㧍 0
㧎 0
㧒 0
㧒 0
㧓 0
㧔 0
㧕 0
㧖 0
㧗 0
㧗 0
㧗 0
㧘 0
㧙 0
㧙 0
㧚 0
㧡 0
㧢 0
㧣 0
㧤 0
㧥 0
㧦 0
㧧 0
㧨 0
㧩 0
㧪 0
㧫 0
㧫 0
㧬 0
㧭 0
㧮 0
㧯 0
㧱 0
㧱 0
㧱 0
㧲 0
㧳 0
㧳 0
㧴 0
㧵 0
㧵 0
㧶 0
㧷 0
㧸 0
㧹 0
㧺 0
㧻 0
㧼 0
㧽 0
㧾 0
㧿 0
㧿 0
㨀 0
㨀 0
㨁 0
㨁 0
㨂 0
㨃 0
㨄 0
㨄 0
㨅 0
㨆 0
㨇 0
㨈 0
㨉 0
㨊 0
㨊 0
㨋 0
㨌 0
㨍 0
㨎 0
㨏 0
㨐 0
㨑 0
㨒 0
㨓 0
㨔 0
㨕 0
㨖 0
㨗 0
㨙 0
㨙 0
㨚 0
㨚 0
㨛 0
㨛 0
㨜 0
㨝 0
㨞 0
㨟 0
㨟 0
㨠 0
㨡 0
㨢 0
㨣 0
㨣 0
㨤 0
㨥 0
㨦 0
㨧 0
㨨 0
㨩 0
㨪 0
㨭 0
㨮 0
㨯 0
㨰 0
㨱 0
㨲 0
㨳 0
㨳 0
㨴 0
㨵 0
㨵 0
㨶 0
㨷 0
㨷 0
㨸 0
㨹 0
㨹 0
㨺 0
㨻 0
㨼 0
㨽 0
㨽 0
㨾 0
㨿 0
㩀 0
㩃 0
㩄 0
㩅 0
㩆 0
㩇 0
㩈 0
㩉 0
㩊 0
㩋 0
㩋 0
㩌 0
㩍 0
㩎 0
㩐 0
㩒 0
㩓 0
㩔 0
㩕 0
㩖 0
㩗 0
㩗 0
㩘 0
㩙 0
㩙 0
㩙 0
㩙 0
㩚 0
㩛 0
㩛 0
㩜 0
㩝 0
㩞 0
㩠 0
㩡 0
㩢 0
㩣 0
㩤 0
㩥 0
㩦 0
㩦 0
㩧 0
㩨 0
㩩 0
㩪 0
㩪 0
㩪 0
㩬 0
㩭 0
㩭 0
㩯 0
㩰 0
㩰 0
㩰 0
㩱 0
㩲 0
㩳 0
㩴 0
㩴 0
㩵 0
㩶 0
㩶 0
㩷 0
㩹 0
㩺 0
㩻 0
㩻 0
㩻 0
㩽 0
㩾 0
㩾 0
㩿 0
㪀 0
㪁 0
㪁 0
㪁 0
㪁 0
㪂 0
㪄 0
㪅 0
㪅 0
㪅 0
㪇 0
㪈 0
㪉 0
㪋 0
㪋 0
㪌 0
㪍 0
㪎 0
㪎 0
㪏 0
㪑 0
㪒 0
㪓 0
㪕 0
㪖 0
㪚 0
㪚 0
㪜 0
㪜 0
㪝 0
㪞 0
㪟 0
㪟 0
㪟 0
㪟 0
㪠 0
㪠 0
㪠 0
㪡 0
㪢 0
㪣 0
㪤 0
㪥 0
㪦 0
㪨 0
㪩 0
㪫 0
㪬 0
㪭 0
㪮 0
㪯 0
㪰 0
㪱 0
㪴 0
㪵 0
㪶 0
㪶 0
㪷 0
㪸 0
㪹 0
㪺 0
㪻 0
㪼 0
㪾 0
㪿 0
㪿 0
㪿 0
㫀 0
㫁 0
㫂 0
㫃 0
㫄 0
㫅 0
㫊 0
㫊 0
㫋 0
㫋 0
㫍 0
㫎 0
㫎 0
㫏 0
㫐 0
㫑 0
㫒 0
㫓 0
㫔 0
㫔 0
㫕 0
㫗 0
㫘 0
㫙 0
㫚 0
㫚 0
㫚 0
㫛 0
㫜 0
㫝 0
㫞 0
㫟 0
㫠 0
㫡 0
㫤 0
㫥 0
㫦 0
㫧 0
㫧 0
㫩 0
㫪 0
㫫 0
㫮 0
㫯 0
㫯 0
㫰 0
㫱 0
㫲 0
㫲 0
㫳 0
㫴 0
㫴 0
㫵 0
㫶 0
㫷 0
㫸 0
㫹 0
㫺 0
㫺 0
㫻 0
㫼 0
㫽 0
㬁 0
㬂 0
㬃 0
㬄 0
㬅 0
㬅 0
㬆 0
㬇 0
㬈 0
㬉 0
㬉 0
㬊 0
㬋 0
㬌 0
㬌 0
㬍 0
㬎 0
㬏 0
㬐 0
㬒 0
㬓 0
㬔 0
㬕 0
㬗 0
㬘 0
㬙 0
㬚 0
㬛 0
㬜 0
㬝 0
㬞 0
㬠 0
㬠 0
㬡 0
㬢 0
㬣 0
㬥 0
㬥 0
㬦 0
㬧 0
㬧 0
㬩 0
㬪 0
㬪 0
㬫 0
㬬 0
㬭 0
㬭 0
㬮 0
㬯 0
㬰 0
㬰 0
㬱 0
㬲 0
㬳 0
㬴 0
㬵 0
㬶 0
㬸 0
㬹 0
㬻 0
㬻 0
㬻 0
㬼 0
㬿 0
㭁 0
㭂 0
㭃 0
㭄 0
㭇 0
㭈 0
㭊 0
㭋 0
㭌 0
㭑 0
㭒 0
㭓 0
㭔 0
㭕 0
㭘 0
㭙 0
㭚 0
㭛 0
㭛 0
㭜 0
㭝 0
㭞 0
㭟 0
㭠 0
㭡 0
㭡 0
㭢 0
㭨 0
㭩 0
㭩 0
㭪 0
㭫 0
㭬 0
㭭 0
㭭 0
㭭 0
㭮 0
㭮 0
㭯 0
㭯 0
㭰 0
㭱 0
㭱 0
㭲 0
㭸 0
㭹 0
㭺 0
㭻 0
㭼 0
㭽 0
㭾 0
㭿 0
㮀 0
㮁 0
㮂 0
㮃 0
㮄 0
㮄 0
㮅 0
㮆 0
㮇 0
㮈 0
㮋 0
㮋 0
㮌 0
㮍 0
㮎 0
㮏 0
㮏 0
㮏 0
㮐 0
㮑 0
㮑 0
㮒 0
㮓 0
㮓 0
㮓 0
㮔 0
㮔 0
㮕 0
㮖 0
㮗 0
㮘 0
㮙 0
㮚 0
㮛 0
㮛 0
㮜 0
㮝 0
㮞 0
㮟 0
㮡 0
㮣 0
㮣 0
㮥 0
㮦 0
㮨 0
㮩 0
㮪 0
㮫 0
㮫 0
㮫 0
㮬 0
㮭 0
㮮 0
㮮 0
㮯 0
㮯 0
㮰 0
㮰 0
㮱 0
㮲 0
㮳 0
㮴 0
㮵 0
㮵 0
㮶 0
㮷 0
㮸 0
㮹 0
㮺 0
㮿 0
㯀 0
㯁 0
㯂 0
㯃 0
㯅 0
㯅 0
㯆 0
㯆 0
㯇 0
㯈 0
㯉 0
㯊 0
㯋 0
㯌 0
㯌 0
㯍 0
㯎 0
㯏 0
㯏 0
㯐 0
㯓 0
㯔 0
㯕 0
㯖 0
㯗 0
㯘 0
㯙 0
㯚 0
㯛 0
㯜 0
㯜 0
㯝 0
㯞 0
㯟 0
㯠 0
㯡 0
㯡 0
㯢 0
㯣 0
㯤 0
㯥 0
㯦 0
㯦 0
㯪 0
㯪 0
㯫 0
㯬 0
㯭 0
㯮 0
㯯 0
㯯 0
㯰 0
㯱 0
㯲 0
㯳 0
㯳 0
㯵 0
㯶 0
㯷 0
㯸 0
㯹 0
㯺 0
㯻 0
㯾 0
㯿 0
㰀 0
㰁 0
㰂 0
㰃 0
㰄 0
㰅 0
㰅 0
㰆 0
㰈 0
㰉 0
㰊 0
㰋 0
㰌 0
㰍 0
㰎 0
㰏 0
㰐 0
㰑 0
㰔 0
㰕 0
㰖 0
㰖 0
㰗 0
㰘 0
㰙 0
㰚 0
㰛 0
㰝 0
㰟 0
㰠 0
㰡 0
㰢 0
㰤 0
㰤 0
㰥 0
㰥 0
㰦 0
㰧 0
㰨 0
㰩 0
㰪 0
㰫 0
㰬 0
㰭 0
㰮 0
㰯 0
㰰 0
㰰 0
㰰 0
㰱 0
㰲 0
㰳 0
㰳 0
㰴 0
㰵 0
㰶 0
㰷 0
㰹 0
㰹 0
㰺 0
㰻 0
㰼 0
㰼 0
㰽 0
㰾 0
㰿 0
㱀 0
㱁 0
㱁 0
㱂 0
㱃 0
㱃 0
㱄 0
㱄 0
㱅 0
㱅 0
㱆 0
㱇 0
㱈 0
㱉 0
㱊 0
㱋 0
㱌 0
㱍 0
㱎 0
㱏 0
㱏 0
㱒 0
㱔 0
㱔 0
㱕 0
㱕 0
㱖 0
㱗 0
㱘 0
㱘 0
㱙 0
㱙 0
㱚 0
㱛 0
㱜 0
㱞 0
㱞 0
㱟 0
㱟 0
㱠 0
㱡 0
㱢 0
㱣 0
㱤 0
㱥 0
㱥 0
㱦 0
㱧 0
㱨 0
㱨 0
㱪 0
㱫 0
㱬 0
㱭 0
㱯 0
㱰 0
㱱 0
㱲 0
㱳 0
㱴 0
㱵 0
㱶 0
㱶 0
㱸 0
㱹 0
㱺 0
㱻 0
㱼 0
㱽 0
㱽 0
㱾 0
㱾 0
㱿 0
㱿 0
㱿 0
㲀 0
㲀 0
㲁 0
㲁 0
㲂 0
㲃 0
㲄 0
㲄 0
㲅 0
㲆 0
㲇 0
㲈 0
㲈 0
㲉 0
㲊 0
㲋 0
㲌 0
㲎 0
㲏 0
㲒 0
㲓 0
㲔 0
㲕 0
㲖 0
㲗 0
㲘 0
㲘 0
㲙 0
㲚 0
㲛 0
㲜 0
㲝 0
㲝 0
㲞 0
㲞 0
㲟 0
㲠 0
㲠 0
㲡 0
㲢 0
㲣 0
㲤 0
㲥 0
㲦 0
㲧 0
㲨 0
㲪 0
㲫 0
㲬 0
㲭 0
㲯 0
㲰 0
㲱 0
㲲 0
㲳 0
㲳 0
㲴 0
㲵 0
㲶 0
㲶 0
㲷 0
㲸 0
㲹 0
㲹 0
㲺 0
㲻 0
㲻 0
㲼 0
㲽 0
㲽 0
㲾 0
㲾 0
㳀 0
㳁 0
㳂 0
㳂 0
㳃 0
㳄 0
㳄 0
㳅 0
㳆 0
㳇 0
㳈 0
㳊 0
㳊 0
㳋 0
㳌 0
㳍 0
㳎 0
㳏 0
㳐 0
㳑 0
㳒 0
㳒 0
㳒 0
㳒 0
㳗 0
㳘 0
㳙 0
㳚 0
㳛 0
㳜 0
㳝 0
㳞 0
㳤 0
㳥 0
㳦 0
㳧 0
㳧 0
㳨 0
㳩 0
㳪 0
㳫 0
㳫 0
㳬 0
㳬 0
㳭 0
㳭 0
㳮 0
㳮 0
㳳 0
㳴 0
㳵 0
㳶 0
㳷 0
㳸 0
㳹 0
㳺 0
㳻 0
㳼 0
㳽 0
㳽 0
㳾 0
㳿 0
㳿 0
㴀 0
㴁 0
㴂 0
㴃 0
㴄 0
㴅 0
㴆 0
㴆 0
㴇 0
㴈 0
㴉 0
㴉 0
㴊 0
㴏 0
㴐 0
㴐 0
㴑 0
㴑 0
㴒 0
㴓 0
㴔 0
㴕 0
㴖 0
㴖 0
㴖 0
㴗 0
㴘 0
㴙 0
㴚 0
㴛 0
㴜 0
㴜 0
㴝 0
㴞 0
㴥 0
㴦 0
㴧 0
㴨 0
㴩 0
㴪 0
㴫 0
㴬 0
㴭 0
㴮 0
㴰 0
㴱 0
㴳 0
㴴 0
㴴 0
㴵 0
㴶 0
㴶 0
㴷 0
㴸 0
㴻 0
㴼 0
㴽 0
㴾 0
㴿 0
㵀 0
㵁 0
㵂 0
㵃 0
㵄 0
㵅 0
㵆 0
㵊 0
㵋 0
㵌 0
㵍 0
㵎 0
㵏 0
㵐 0
㵑 0
㵒 0
㵒 0
㵓 0
㵔 0
㵕 0
㵖 0
㵗 0
㵘 0
㵙 0
㵝 0
㵞 0
㵟 0
㵠 0
㵡 0
㵢 0
㵣 0
㵤 0
㵥 0
㵦 0
㵧 0
㵨 0
㵨 0
㵩 0
㵪 0
㵪 0
㵫 0
㵬 0
㵭 0
㵮 0
㵯 0
㵰 0
㵱 0
㵱 0
㵱 0
㵲 0
㵳 0
㵴 0
㵵 0
㵵 0
㵷 0
㵸 0
㵸 0
㵹 0
㵺 0
㵻 0
㵽 0
㵾 0
㵿 0
㶀 0
㶁 0
㶄 0
㶅 0
㶆 0
㶇 0
㶈 0
㶌 0
㶍 0
㶎 0
㶏 0
㶑 0
㶒 0
㶓 0
㶔 0
㶕 0
㶖 0
㶗 0
㶘 0
㶚 0
㶛 0
㶜 0
㶞 0
㶟 0
㶠 0
㶡 0
㶣 0
㶤 0
㶦 0
㶧 0
㶧 0
㶨 0
㶪 0
㶫 0
㶫 0
㶫 0
㶬 0
㶭 0
㶮 0
㶮 0
㶯 0
㶰 0
㶲 0
㶳 0
㶴 0
㶵 0
㶹 0
㶺 0
㶻 0
㶼 0
㶾 0
㶾 0
㶿 0
㷀 0
㷂 0
㷃 0
㷄 0
㷅 0
㷆 0
㷇 0
㷈 0
㷉 0
㷉 0
㷊 0
㷊 0
㷋 0
㷍 0
㷎 0
㷎 0
㷏 0
㷐 0
㷑 0
㷒 0
㷓 0
㷔 0
㷔 0
㷕 0
㷖 0
㷗 0
㷘 0
㷙 0
㷟 0
㷠 0
㷡 0
㷢 0
㷣 0
㷤 0
㷥 0
㷦 0
㷫 0
㷬 0
㷭 0
㷮 0
㷯 0
㷰 0
㷳 0
㷴 0
㷵 0
㷶 0
㷷 0
㷷 0
㷸 0
㷸 0
㷹 0
㷻 0
㷻 0
㷼 0
㷼 0
㷾 0
㷿 0
㸀 0
㸀 0
㸁 0
㸂 0
㸃 0
㸄 0
㸄 0
㸄 0
㸅 0
㸈 0
㸉 0
㸉 0
㸊 0
㸊 0
㸋 0
㸌 0
㸍 0
㸎 0
㸏 0
㸐 0
㸑 0
㸒 0
㸒 0
㸓 0
㸔 0
㸔 0
㸕 0
㸖 0
㸗 0
㸘 0
㸙 0
㸚 0
㸛 0
㸜 0
㸝 0
㸝 0
㸞 0
㸠 0
㸡 0
㸢 0
㸣 0
㸤 0
㸥 0
㸦 0
㸧 0
㸧 0
㸨 0
㸩 0
㸪 0
㸪 0
㸫 0
㸫 0
㸬 0
㸭 0
㸮 0
㸯 0
㸰 0
㸱 0
㸱 0
㸲 0
㸳 0
㸵 0
㸶 0
㸷 0
㸸 0
㸸 0
㸹 0
㸹 0
㸺 0
㸻 0
㸼 0
㸽 0
㸾 0
㹀 0
㹁 0
㹂 0
㹃 0
㹄 0
㹅 0
㹆 0
㹇 0
㹇 0
㹉 0
㹉 0
㹊 0
㹋 0
㹌 0
㹍 0
㹎 0
㹏 0
㹐 0
㹑 0
㹒 0
㹓 0
㹔 0
㹕 0
㹖 0
㹖 0
㹗 0
㹘 0
㹙 0
㹚 0
㹛 0
㹜 0
㹝 0
㹞 0
㹞 0
㹟 0
㹟 0
㹠 0
㹡 0
㹡 0
㹢 0
㹣 0
㹤 0
㹤 0
㹥 0
㹦 0
㹧 0
㹫 0
㹬 0
㹬 0
㹭 0
㹮 0
㹮 0
㹯 0
㹱 0
㹲 0
㹳 0
㹴 0
㹵 0
㹶 0
㹷 0
㹸 0
㹹 0
㹺 0
㹻 0
㹻 0
㹼 0
㹽 0
㹾 0
㹾 0
㹿 0
㹿 0
㺀 0
㺁 0
㺂 0
㺃 0
㺄 0
㺅 0
㺉 0
㺊 0
㺋 0
㺌 0
㺎 0
㺏 0
㺐 0
㺑 0
㺑 0
㺑 0
㺑 0
㺒 0
㺒 0
㺒 0
㺓 0
㺔 0
㺔 0
㺕 0
㺕 0
㺖 0
㺗 0
㺘 0
㺚 0
㺛 0
㺜 0
㺝 0
㺞 0
㺟 0
㺠 0
㺡 0
㺣 0
㺤 0
㺥 0
㺦 0
㺧 0
㺩 0
㺪 0
㺫 0
㺬 0
㺬 0
㺭 0
㺮 0
㺱 0
㺲 0
㺳 0
㺴 0
㺵 0
㺷 0
㺸 0
㺹 0
㺺 0
㺺 0
㺿 0
㻀 0
㻁 0
㻁 0
㻂 0
㻃 0
㻄 0
㻈 0
㻉 0
㻊 0
㻋 0
㻌 0
㻍 0
㻎 0
㻑 0
㻒 0
㻓 0
㻔 0
㻕 0
㻖 0
㻗 0
㻗 0
㻗 0
㻝 0
㻞 0
㻞 0
㻟 0
㻠 0
㻡 0
㻡 0
㻧 0
㻨 0
㻫 0
㻬 0
㻭 0
㻮 0
㻯 0
㻰 0
㻱 0
㻱 0
㻲 0
㻵 0
㻶 0
㻷 0
㻹 0
㻺 0
㻽 0
㻽 0
㻾 0
㻿 0
㼂 0
㼂 0
㼄 0
㼇 0
㼇 0
㼉 0
㼊 0
㼋 0
㼌 0
㼍 0
㼎 0
㼏 0
㼐 0
㼑 0
㼒 0
㼓 0
㼔 0
㼕 0
㼖 0
㼗 0
㼘 0
㼙 0
㼚 0
㼛 0
㼛 0
㼛 0
㼜 0
㼝 0
㼞 0
㼟 0
㼠 0
㼡 0
㼢 0
㼣 0
㼤 0
㼥 0
㼥 0
㼦 0
㼧 0
㼨 0
㼩 0
㼩 0
㼪 0
㼫 0
㼫 0
㼬 0
㼭 0
㼮 0
㼮 0
㼯 0
㼰 0
㼱 0
㼱 0
㼲 0
㼳 0
㼴 0
㼵 0
㼶 0
㼷 0
㼷 0
㼸 0
㼹 0
㼹 0
㼺 0
㼻 0
㼼 0
㼽 0
㼾 0
㼿 0
㽀 0
㽁 0
㽁 0
㽂 0
㽃 0
㽆 0
㽇 0
㽈 0
㽉 0
㽊 0
㽋 0
㽌 0
㽍 0
㽎 0
㽐 0
㽐 0
㽑 0
㽒 0
㽕 0
㽖 0
㽗 0
㽗 0
㽘 0
㽙 0
㽚 0
㽛 0
㽛 0
㽜 0
㽝 0
㽞 0
㽟 0
㽠 0
㽡 0
㽢 0
㽣 0
㽤 0
㽥 0
㽦 0
㽨 0
㽩 0
㽪 0
㽫 0
㽬 0
㽭 0
㽮 0
㽯 0
㽯 0
㽯 0
㽰 0
㽱 0
㽱 0
㽲 0
㽲 0
㽴 0
㽵 0
㽷 0
㽸 0
㽹 0
㽹 0
㽺 0
㽽 0
㽾 0
㽿 0
㾀 0
㾀 0
㾁 0
㾂 0
㾃 0
㾄 0
㾅 0
㾆 0
㾇 0
㾈 0
㾉 0
㾊 0
㾋 0
㾋 0
㾌 0
㾌 0
㾍 0
㾎 0
㾏 0
㾏 0
㾐 0
㾑 0
㾑 0
㾒 0
㾒 0
㾓 0
㾔 0
㾕 0
㾖 0
㾗 0
㾘 0
㾙 0
㾙 0
㾚 0
㾛 0
㾜 0
㾝 0
㾞 0
㾟 0
㾠 0
㾡 0
㾢 0
㾣 0
㾤 0
㾥 0
㾦 0
㾧 0
㾨 0
㾩 0
㾪 0
㾫 0
㾬 0
㾭 0
㾮 0
㾯 0
㾯 0
㾰 0
㾶 0
㾷 0
㾸 0
㾹 0
㾹 0
㾺 0
㾻 0
㾼 0
㾽 0
㾽 0
㾾 0
㾾 0
㾿 0
㿁 0
㿂 0
㿃 0
㿃 0
㿄 0
㿅 0
㿅 0
㿆 0
㿇 0
㿈 0
㿉 0
㿉 0
㿊 0
㿋 0
㿋 0
㿌 0
㿍 0
㿎 0
㿎 0
㿏 0
㿑 0
㿒 0
㿒 0
㿒 0
㿓 0
㿕 0
㿖 0
㿖 0
㿗 0
㿗 0
㿘 0
㿙 0
㿚 0
㿛 0
㿜 0
㿜 0
㿝 0
㿞 0
㿞 0
㿟 0
㿡 0
㿢 0
㿢 0
㿣 0
㿤 0
㿥 0
㿦 0
㿧 0
㿧 0
㿨 0
㿩 0
㿪 0
㿫 0
㿬 0
㿭 0
㿭 0
㿮 0
㿯 0
㿰 0
㿰 0
㿱 0
㿲 0
㿴 0
㿵 0
㿵 0
㿶 0
㿷 0
㿸 0
㿸 0
㿹 0
㿺 0
㿻 0
㿼 0
㿽 0
㿾 0
㿿 0
䀀 0
䀁 0
䀃 0
䀄 0
䀅 0
䀆 0
䀇 0
䀈 0
䀉 0
䀊 0
䀋 0
䀋 0
䀍 0
䀎 0
䀏 0
䀐 0
䀑 0
䀑 0
䀒 0
䀓 0
䀔 0
䀕 0
䀖 0
䀗 0
䀘 0
䀙 0
䀚 0
䀛 0
䀛 0
䀜 0
䀞 0
䀞 0
䀞 0
䀟 0
䀠 0
䀡 0
䀢 0
䀣 0
䀣 0
䀤 0
䀧 0
䀨 0
䀨 0
䀩 0
䀪 0
䀫 0
䀫 0
䀬 0
䀭 0
䀮 0
䀯 0
䀰 0
䀱 0
䀲 0
䀳 0
䀴 0
䀴 0
䀶 0
䀷 0
䀸 0
䀸 0
䀹 0
䀺 0
䀺 0
䀻 0
䀼 0
䀽 0
䀾 0
䀿 0
䁁 0
䁂 0
䁃 0
䁄 0
䁅 0
䁆 0
䁇 0
䁈 0
䁈 0
䁉 0
䁊 0
䁋 0
䁌 0
䁍 0
䁍 0
䁎 0
䁎 0
䁎 0
䁏 0
䁐 0
䁑 0
䁒 0
䁓 0
䁔 0
䁔 0
䁕 0
䁗 0
䁘 0
䁙 0
䁚 0
䁛 0
䁜 0
䁝 0
䁞 0
䁞 0
䁟 0
䁠 0
䁢 0
䁣 0
䁤 0
䁤 0
䁥 0
䁦 0
䁦 0
䁧 0
䁨 0
䁩 0
䁫 0
䁬 0
䁭 0
䁮 0
䁯 0
䁰 0
䁱 0
䁲 0
䁳 0
䁴 0
䁵 0
䁶 0
䁷 0
䁸 0
䁹 0
䁺 0
䁻 0
䁼 0
䁽 0
䁾 0
䁿 0
䂀 0
䂁 0
䂂 0
䂃 0
䂄 0
䂅 0
䂆 0
䂇 0
䂈 0
䂉 0
䂊 0
䂌 0
䂍 0
䂎 0
䂏 0
䂐 0
䂑 0
䂒 0
䂓 0
䂔 0
䂕 0
䂖 0
䂖 0
䂗 0
䂘 0
䂚 0
䂛 0
䂜 0
䂝 0
䂞 0
䂞 0
䂟 0
䂠 0
䂡 0
䂡 0
䂢 0
䂤 0
䂥 0
䂥 0
䂦 0
䂧 0
䂩 0
䂩 0
䂪 0
䂫 0
䂬 0
䂭 0
䂮 0
䂯 0
䂰 0
䂱 0
䂲 0
䂳 0
䂳 0
䂴 0
䂶 0
䂷 0
䂸 0
䂹 0
䂺 0
䂺 0
䂻 0
䂼 0
䂽 0
䂾 0
䂿 0
䃀 0
䃁 0
䃂 0
䃇 0
䃈 0
䃉 0
䃉 0
䃊 0
䃋 0
䃌 0
䃍 0
䃎 0
䃏 0
䃑 0
䃒 0
䃓 0
䃔 0
䃕 0
䃖 0
䃗 0
䃘 0
䃘 0
䃘 0
䃙 0
䃚 0
䃚 0
䃛 0
䃝 0
䃞 0
䃠 0
䃡 0
䃢 0
䃣 0
䃤 0
䃥 0
䃦 0
䃧 0
䃩 0
䃪 0
䃫 0
䃬 0
䃭 0
䃮 0
䃰 0
䃱 0
䃱 0
䃲 0
䃳 0
䃴 0
䃵 0
䃶 0
䃶 0
䃶 0
䃷 0
䃸 0
䃹 0
䃺 0
䃺 0
䃻 0
䃽 0
䃾 0
䃿 0
䄀 0
䄁 0
䄂 0
䄃 0
䄄 0
䄅 0
䄆 0
䄆 0
䄆 0
䄇 0
䄈 0
䄉 0
䄋 0
䄌 0
䄍 0
䄎 0
䄏 0
䄐 0
䄑 0
䄒 0
䄒 0
䄓 0
䄔 0
䄕 0
䄘 0
䄙 0
䄚 0
䄛 0
䄜 0
䄝 0
䄟 0
䄠 0
䄠 0
䄡 0
䄡 0
䄢 0
䄢 0
䄢 0
䄤 0
䄥 0
䄦 0
䄧 0
䄨 0
䄩 0
䄪 0
䄫 0
䄫 0
䄬 0
䄭 0
䄮 0
䄯 0
䄰 0
䄱 0
䄲 0
䄶 0
䄷 0
䄸 0
䄺 0
䄺 0
䄻 0
䄻 0
䄻 0
䄼 0
䄽 0
䄾 0
䄿 0
䅀 0
䅁 0
䅂 0
䅃 0
䅃 0
䅄 0
䅅 0
䅆 0
䅇 0
䅊 0
䅋 0
䅌 0
䅍 0
䅎 0
䅏 0
䅏 0
䅐 0
䅑 0
䅒 0
䅓 0
䅕 0
䅗 0
䅘 0
䅙 0
䅚 0
䅛 0
䅜 0
䅜 0
䅜 0
䅝 0
䅞 0
䅠 0
䅡 0
䅢 0
䅣 0
䅤 0
䅥 0
䅦 0
䅧 0
䅨 0
䅩 0
䅬 0
䅭 0
䅮 0
䅯 0
䅰 0
䅱 0
䅲 0
䅳 0
䅴 0
䅵 0
䅶 0
䅶 0
䅷 0
䅸 0
䅹 0
䅺 0
䅻 0
䅼 0
䅽 0
䅽 0
䅾 0
䅿 0
䆀 0
䆁 0
䆂 0
䆂 0
䆂 0
䆃 0
䆄 0
䆈 0
䆉 0
䆊 0
䆋 0
䆍 0
䆎 0
䆏 0
䆏 0
䆐 0
䆑 0
䆒 0
䆓 0
䆔 0
䆕 0
䆖 0
䆗 0
䆘 0
䆘 0
䆙 0
䆚 0
䆚 0
䆛 0
䆜 0
䆝 0
䆝 0
䆞 0
䆞 0
䆞 0
䆟 0
䆠 0
䆡 0
䆢 0
䆣 0
䆤 0
䆥 0
䆦 0
䆧 0
䆨 0
䆩 0
䆪 0
䆫 0
䆬 0
䆭 0
䆯 0
䆱 0
䆲 0
䆳 0
䆵 0
䆶 0
䆷 0
䆸 0
䆹 0
䆹 0
䆺 0
䆻 0
䆼 0
䆽 0
䆾 0
䆿 0
䇀 0
䇂 0
䇂 0
䇃 0
䇃 0
䇅 0
䇇 0
䇇 0
䇈 0
䇋 0
䇌 0
䇍 0
䇎 0
䇏 0
䇐 0
䇑 0
䇒 0
䇓 0
䇔 0
䇕 0
䇖 0
䇗 0
䇘 0
䇙 0
䇚 0
䇛 0
䇜 0
䇜 0
䇝 0
䇞 0
䇞 0
䇟 0
䇠 0
䇡 0
䇢 0
䇢 0
䇣 0
䇤 0
䇥 0
䇦 0
䇦 0
䇧 0
䇧 0
䇨 0
䇨 0
䇩 0
䇫 0
䇬 0
䇬 0
䇬 0
䇭 0
䇮 0
䇯 0
䇰 0
䇱 0
䇲 0
䇲 0
䇵 0
䇶 0
䇷 0
䇸 0
䇹 0
䇺 0
䇻 0
䇼 0
䇽 0
䇾 0
䈀 0
䈁 0
䈂 0
䈃 0
䈄 0
䈅 0
䈆 0
䈇 0
䈇 0
䈈 0
䈈 0
䈉 0
䈊 0
䈋 0
䈌 0
䈍 0
䈎 0
䈏 0
䈐 0
䈐 0
䈑 0
䈒 0
䈓 0
䈔 0
䈕 0
䈖 0
䈗 0
䈘 0
䈙 0
䈚 0
䈛 0
䈜 0
䈝 0
䈞 0
䈟 0
䈠 0
䈠 0
䈡 0
䈡 0
䈢 0
䈢 0
䈣 0
䈤 0
䈥 0
䈦 0
䈧 0
䈪 0
䈫 0
䈬 0
䈭 0
䈭 0
䈮 0
䈯 0
䈰 0
䈱 0
䈲 0
䈳 0
䈴 0
䈵 0
䈶 0
䈷 0
䈸 0
䈹 0
䈹 0
䈺 0
䈻 0
䈼 0
䈽 0
䈾 0
䈿 0
䉀 0
䉁 0
䉂 0
䉃 0
䉄 0
䉅 0
䉆 0
䉈 0
䉉 0
䉊 0
䉋 0
䉌 0
䉍 0
䉎 0
䉏 0
䉐 0
䉑 0
䉒 0
䉓 0
䉔 0
䉕 0
䉖 0
䉘 0
䉙 0
䉚 0
䉛 0
䉜 0
䉝 0
䉞 0
䉟 0
䉠 0
䉠 0
䉡 0
䉢 0
䉣 0
䉥 0
䉦 0
䉧 0
䉨 0
䉪 0
䉫 0
䉬 0
䉭 0
䉮 0
䉯 0
䉰 0
䉰 0
䉲 0
䉳 0
䉴 0
䉵 0
䉶 0
䉷 0
䉸 0
䉹 0
䉺 0
䉻 0
䉼 0
䉽 0
䉾 0
䉿 0
䊀 0
䊁 0
䊂 0
䊂 0
䊃 0
䊄 0
䊅 0
䊆 0
䊇 0
䊇 0
䊈 0
䊉 0
䊊 0
䊋 0
䊍 0
䊎 0
䊏 0
䊐 0
䊑 0
䊒 0
䊓 0
䊔 0
䊕 0
䊖 0
䊗 0
䊘 0
䊙 0
䊚 0
䊛 0
䊜 0
䊝 0
䊞 0
䊟 0
䊠 0
䊠 0
䊡 0
䊡 0
䊣 0
䊤 0
䊤 0
䊥 0
䊦 0
䊧 0
䊨 0
䊩 0
䊪 0
䊫 0
䊬 0
䊭 0
䊮 0
䊯 0
䊰 0
䊱 0
䊲 0
䊲 0
䊳 0
䊴 0
䊵 0
䊶 0
䊹 0
䊺 0
䊻 0
䊼 0
䊽 0
䊽 0
䊾 0
䊿 0
䋀 0
䋁 0
䋁 0
䋂 0
䋃 0
䋄 0
䋈 0
䋈 0
䋉 0
䋊 0
䋋 0
䋌 0
䋌 0
䋎 0
䋎 0
䋏 0
䋏 0
䋐 0
䋑 0
䋒 0
䋓 0
䋔 0
䋕 0
䋕 0
䋖 0
䋗 0
䋘 0
䋙 0
䋚 0
䋛 0
䋜 0
䋜 0
䋞 0
䋟 0
䋠 0
䋡 0
䋢 0
䋣 0
䋣 0
䋤 0
䋥 0
䋦 0
䋧 0
䋨 0
䋩 0
䋫 0
䋬 0
䋭 0
䋮 0
䋯 0
䋯 0
䋰 0
䋱 0
䋲 0
䋳 0
䋴 0
䋵 0
䋶 0
䋶 0
䋷 0
䋸 0
䋹 0
䋺 0
䋺 0
䋻 0
䋼 0
䋽 0
䋾 0
䋾 0
䋿 0
䌀 0
䌁 0
䌂 0
䌃 0
䌄 0
䌄 0
䌅 0
䌆 0
䌇 0
䌇 0
䌈 0
䌉 0
䌊 0
䌊 0
䌊 0
䌌 0
䌌 0
䌍 0
䌎 0
䌏 0
䌐 0
䌑 0
䌒 0
䌓 0
䌓 0
䌕 0
䌖 0
䌗 0
䌘 0
䌙 0
䌚 0
䌛 0
䌛 0
䌛 0
䌜 0
䌝 0
䌞 0
䌟 0
䌠 0
䌡 0
䌢 0
䌣 0
䌤 0
䌥 0
䌦 0
䌧 0
䌧 0
䌨 0
䌨 0
䌩 0
䌪 0
䌫 0
䌫 0
䌬 0
䌭 0
䌭 0
䌮 0
䌯 0
䌯 0
䌯 0
䌰 0
䌱 0
䌲 0
䌴 0
䌵 0
䍂 0
䍃 0
䍄 0
䍅 0
䍆 0
䍇 0
䍈 0
䍉 0
䍊 0
䍋 0
䍋 0
䍌 0
䍍 0
䍎 0
䍐 0
䍑 0
䍒 0
䍒 0
䍓 0
䍔 0
䍕 0
䍕 0
䍖 0
䍗 0
䍘 0
䍙 0
䍚 0
䍛 0
䍜 0
䍝 0
䍞 0
䍟 0
䍡 0
䍢 0
䍣 0
䍤 0
䍥 0
䍦 0
䍧 0
䍨 0
䍨 0
䍩 0
䍩 0
䍪 0
䍫 0
䍬 0
䍭 0
䍮 0
䍯 0
䍯 0
䍰 0
䍱 0
䍲 0
䍳 0
䍳 0
䍳 0
䍴 0
䍵 0
䍶 0
䍷 0
䍸 0
䍹 0
䍺 0
䍻 0
䍼 0
䍽 0
䍾 0
䍿 0
䎀 0
䎁 0
䎂 0
䎃 0
䎄 0
䎅 0
䎆 0
䎇 0
䎈 0
䎉 0
䎊 0
䎊 0
䎋 0
䎌 0
䎍 0
䎎 0
䎏 0
䎐 0
䎑 0
䎒 0
䎓 0
䎔 0
䎕 0
䎖 0
䎗 0
䎘 0
䎙 0
䎚 0
䎜 0
䎝 0
䎟 0
䎠 0
䎡 0
䎢 0
䎣 0
䎤 0
䎦 0
䎧 0
䎨 0
䎨 0
䎪 0
䎫 0
䎭 0
䎮 0
䎯 0
䎰 0
䎱 0
䎱 0
䎲 0
䎳 0
䎴 0
䎵 0
䎶 0
䎷 0
䎸 0
䎹 0
䎹 0
䎺 0
䎻 0
䎼 0
䎽 0
䎾 0
䎿 0
䏀 0
䏁 0
䏂 0
䏃 0
䏄 0
䏅 0
䏆 0
䏇 0
䏈 0
䏊 0
䏋 0
䏌 0
䏍 0
䏍 0
䏎 0
䏏 0
䏐 0
䏑 0
䏒 0
䏓 0
䏔 0
䏖 0
䏖 0
䏗 0
䏘 0
䏙 0
䏚 0
䏛 0
䏜 0
䏞 0
䏟 0
䏠 0
䏠 0
䏡 0
䏢 0
䏣 0
䏤 0
䏥 0
䏦 0
䏧 0
䏧 0
䏨 0
䏩 0
䏩 0
䏪 0
䏫 0
䏬 0
䏯 0
䏰 0
䏱 0
䏲 0
䏳 0
䏴 0
䏴 0
䏵 0
䏶 0
䏷 0
䏷 0
䏸 0
䏹 0
䏺 0
䏻 0
䏼 0
䏽 0
䏽 0
䏾 0
䏿 0
䐀 0
䐁 0
䐂 0
䐃 0
䐃 0
䐄 0
䐄 0
䐅 0
䐇 0
䐈 0
䐉 0
䐊 0
䐋 0
䐌 0
䐍 0
䐎 0
䐏 0
䐏 0
䐐 0
䐑 0
䐒 0
䐓 0
䐔 0
䐔 0
䐕 0
䐖 0
䐗 0
䐗 0
䐘 0
䐙 0
䐚 0
䐛 0
䐜 0
䐝 0
䐞 0
䐟 0
䐟 0
䐠 0
䐡 0
䐢 0
䐢 0
䐢 0
䐣 0
䐤 0
䐥 0
䐦 0
䐧 0
䐨 0
䐩 0
䐩 0
䐩 0
䐫 0
䐬 0
䐭 0
䐮 0
䐯 0
䐰 0
䐱 0
䐲 0
䐳 0
䐴 0
䐵 0
䐶 0
䐶 0
䐶 0
䐷 0
䐸 0
䐹 0
䐺 0
䐻 0
䐼 0
䐾 0
䐿 0
䑀 0
䑁 0
䑂 0
䑃 0
䑄 0
䑅 0
䑆 0
䑇 0
䑈 0
䑉 0
䑊 0
䑋 0
䑌 0
䑍 0
䑎 0
䑏 0
䑐 0
䑑 0
䑒 0
䑓 0
䑔 0
䑕 0
䑗 0
䑘 0
䑙 0
䑚 0
䑛 0
䑜 0
䑝 0
䑞 0
䑟 0
䑠 0
䑡 0
䑢 0
䑣 0
䑤 0
䑥 0
䑥 0
䑦 0
䑧 0
䑨 0
䑪 0
䑪 0
䑫 0
䑬 0
䑭 0
䑯 0
䑰 0
䑱 0
䑲 0
䑲 0
䑳 0
䑴 0
䑵 0
䑵 0
䑶 0
䑹 0
䑹 0
䑺 0
䑺 0
䑻 0
䑼 0
䑽 0
䑿 0
䒀 0
䒁 0
䒂 0
䒃 0
䒃 0
䒄 0
䒅 0
䒆 0
䒈 0
䒉 0
䒊 0
䒋 0
䒌 0
䒍 0
䒎 0
䒏 0
䒐 0
䒐 0
䒒 0
䒓 0
䒔 0
䒕 0
䒖 0
䒗 0
䒘 0
䒚 0
䒛 0
䒜 0
䒝 0
䒞 0
䒟 0
䒠 0
䒡 0
䒢 0
䒣 0
䒤 0
䒥 0
䒦 0
䒧 0
䒧 0
䒨 0
䒩 0
䒪 0
䒫 0
䒬 0
䒬 0
䒭 0
䒰 0
䒱 0
䒲 0
䒳 0
䒴 0
䒵 0
䒶 0
䒷 0
䒸 0
䒹 0
䒺 0
䒻 0
䒼 0
䒽 0
䓁 0
䓂 0
䓃 0
䓄 0
䓅 0
䓆 0
䓇 0
䓈 0
䓉 0
䓊 0
䓌 0
䓍 0
䓎 0
䓏 0
䓐 0
䓑 0
䓒 0
䓓 0
䓗 0
䓘 0
䓙 0
䓙 0
䓚 0
䓛 0
䓝 0
䓞 0
䓟 0
䓠 0
䓢 0
䓣 0
䓤 0
䓥 0
䓦 0
䓧 0
䓭 0
䓮 0
䓯 0
䓰 0
䓱 0
䓱 0
䓲 0
䓳 0
䓴 0
䓵 0
䓶 0
䓶 0
䓷 0
䓸 0
䓹 0
䓺 0
䓺 0
䓼 0
䓽 0
䓾 0
䓿 0
䔀 0
䔁 0
䔂 0
䔃 0
䔄 0
䔅 0
䔆 0
䔇 0
䔈 0
䔉 0
䔊 0
䔋 0
䔌 0
䔍 0
䔎 0
䔏 0
䔐 0
䔑 0
䔑 0
䔑 0
䔒 0
䔒 0
䔓 0
䔔 0
䔕 0
䔖 0
䔖 0
䔗 0
䔘 0
䔙 0
䔚 0
䔛 0
䔛 0
䔜 0
䔝 0
䔞 0
䔟 0
䔠 0
䔠 0
䔡 0
䔢 0
䔢 0
䔢 0
䔣 0
䔤 0
䔥 0
䔩 0
䔪 0
䔫 0
䔫 0
䔬 0
䔭 0
䔯 0
䔰 0
䔲 0
䔳 0
䔴 0
䔵 0
䔶 0
䔶 0
䔷 0
䔸 0
䔹 0
䔺 0
䔻 0
䔼 0
䔽 0
䔾 0
䔿 0
䕀 0
䕁 0
䕃 0
䕄 0
䕅 0
䕆 0
䕇 0
䕈 0
䕉 0
䕊 0
䕋 0
䕌 0
䕍 0
䕎 0
䕏 0
䕑 0
䕑 0
䕒 0
䕓 0
䕔 0
䕕 0
䕕 0
䕕 0
䕖 0
䕗 0
䕘 0
䕙 0
䕚 0
䕛 0
䕝 0
䕞 0
䕟 0
䕠 0
䕡 0
䕢 0
䕤 0
䕥 0
䕦 0
䕧 0
䕨 0
䕩 0
䕪 0
䕫 0
䕬 0
䕭 0
䕮 0
䕯 0
䕰 0
䕱 0
䕲 0
䕳 0
䕴 0
䕵 0
䕸 0
䕸 0
䕹 0
䕺 0
䕻 0
䕼 0
䕽 0
䕾 0
䕿 0
䖀 0
䖁 0
䖂 0
䖃 0
䖅 0
䖆 0
䖇 0
䖈 0
䖉 0
䖊 0
䖋 0
䖋 0
䖌 0
䖍 0
䖍 0
䖎 0
䖏 0
䖏 0
䖐 0
䖑 0
䖒 0
䖓 0
䖔 0
䖔 0
䖕 0
䖖 0
䖗 0
䖘 0
䖙 0
䖛 0
䖜 0
䖝 0
䖝 0
䖞 0
䖟 0
䖠 0
䖡 0
䖢 0
䖣 0
䖤 0
䖤 0
䖥 0
䖦 0
䖦 0
䖧 0
䖩 0
䖫 0
䖭 0
䖮 0
䖯 0
䖯 0
䖰 0
䖱 0
䖲 0
䖲 0
䖳 0
䖴 0
䖵 0
䖶 0
䖷 0
䖸 0
䖸 0
䖹 0
䖹 0
䖺 0
䖻 0
䖻 0
䖼 0
䖽 0
䖽 0
䖿 0
䗀 0
䗁 0
䗂 0
䗃 0
䗄 0
䗅 0
䗆 0
䗆 0
䗇 0
䗇 0
䗈 0
䗉 0
䗊 0
䗋 0
䗌 0
䗍 0
䗍 0
䗎 0
䗏 0
䗑 0
䗒 0
䗓 0
䗔 0
䗕 0
䗕 0
䗗 0
䗘 0
䗙 0
䗚 0
䗛 0
䗜 0
䗝 0
䗞 0
䗟 0
䗠 0
䗡 0
䗢 0
䗣 0
䗤 0
䗥 0
䗦 0
䗧 0
䗨 0
䗩 0
䗪 0
䗫 0
䗫 0
䗫 0
䗬 0
䗮 0
䗯 0
䗰 0
䗱 0
䗱 0
䗱 0
䗲 0
䗴 0
䗵 0
䗶 0
䗷 0
䗸 0
䗺 0
䗼 0
䗽 0
䗾 0
䗿 0
䘀 0
䘁 0
䘂 0
䘃 0
䘄 0
䘅 0
䘆 0
䘇 0
䘈 0
䘉 0
䘊 0
䘋 0
䘌 0
䘍 0
䘎 0
䘏 0
䘐 0
䘐 0
䘑 0
䘑 0
䘒 0
䘒 0
䘒 0
䘓 0
䘔 0
䘔 0
䘕 0
䘖 0
䘗 0
䘘 0
䘘 0
䘙 0
䘚 0
䘚 0
䘜 0
䘝 0
䘟 0
䘠 0
䘡 0
䘢 0
䘣 0
䘤 0
䘥 0
䘦 0
䘨 0
䘩 0
䘩 0
䘩 0
䘪 0
䘫 0
䘬 0
䘭 0
䘮 0
䘮 0
䘯 0
䘰 0
䘱 0
䘳 0
䘴 0
䘴 0
䘵 0
䘶 0
䘶 0
䘷 0
䘹 0
䘺 0
䘺 0
䘻 0
䘼 0
䘽 0
䘾 0
䘿 0
䙀 0
䙀 0
䙀 0
䙁 0
䙂 0
䙂 0
䙃 0
䙄 0
䙄 0
䙅 0
䙆 0
䙇 0
䙈 0
䙉 0
䙎 0
䙎 0
䙏 0
䙏 0
䙏 0
䙐 0
䙑 0
䙑 0
䙒 0
䙕 0
䙖 0
䙖 0
䙖 0
䙗 0
䙘 0
䙙 0
䙚 0
䙛 0
䙜 0
䙝 0
䙝 0
䙞 0
䙟 0
䙠 0
䙡 0
䙡 0
䙢 0
䙣 0
䙤 0
䙥 0
䙦 0
䙧 0
䙨 0
䙩 0
䙪 0
䙬 0
䙭 0
䙮 0
䙮 0
䙯 0
䙰 0
䙱 0
䙲 0
䙳 0
䙳 0
䙴 0
䙵 0
䙷 0
䙷 0
䙷 0
䙷 0
䙸 0
䙸 0
䙹 0
䙹 0
䙺 0
䙼 0
䙼 0
䙽 0
䙿 0
䚀 0
䚁 0
䚂 0
䚃 0
䚄 0
䚅 0
䚆 0
䚇 0
䚈 0
䚉 0
䚊 0
䚋 0
䚌 0
䚍 0
䚎 0
䚏 0
䚐 0
䚐 0
䚑 0
䚒 0
䚓 0
䚓 0
䚔 0
䚔 0
䚖 0
䚗 0
䚗 0
䚘 0
䚙 0
䚙 0
䚚 0
䚛 0
䚜 0
䚜 0
䚝 0
䚞 0
䚟 0
䚠 0
䚡 0
䚢 0
䚣 0
䚥 0
䚦 0
䚧 0
䚨 0
䚩 0
䚩 0
䚩 0
䚪 0
䚫 0
䚭 0
䚮 0
䚯 0
䚰 0
䚱 0
䚱 0
䚲 0
䚳 0
䚴 0
䚵 0
䚶 0
䚷 0
䚷 0
䚸 0
䚹 0
䚺 0
䚻 0
䚼 0
䚽 0
䚾 0
䚿 0
䛀 0
䛀 0
䛀 0
䛁 0
䛂 0
䛃 0
䛄 0
䛄 0
䛅 0
䛆 0
䛇 0
䛈 0
䛉 0
䛊 0
䛋 0
䛌 0
䛍 0
䛎 0
䛏 0
䛏 0
䛐 0
䛑 0
䛒 0
䛔 0
䛕 0
䛖 0
䛗 0
䛘 0
䛙 0
䛚 0
䛛 0
䛜 0
䛝 0
䛞 0
䛟 0
䛠 0
䛡 0
䛢 0
䛣 0
䛤 0
䛥 0
䛥 0
䛥 0
䛦 0
䛧 0
䛧 0
䛧 0
䛨 0
䛩 0
䛩 0
䛪 0
䛫 0
䛬 0
䛭 0
䛮 0
䛯 0
䛰 0
䛱 0
䛲 0
䛳 0
䛳 0
䛴 0
䛵 0
䛶 0
䛷 0
䛸 0
䛹 0
䛺 0
䛻 0
䛼 0
䛽 0
䛾 0
䛿 0
䜀 0
䜁 0
䜂 0
䜃 0
䜄 0
䜅 0
䜅 0
䜆 0
䜇 0
䜈 0
䜈 0
䜉 0
䜊 0
䜋 0
䜌 0
䜍 0
䜎 0
䜏 0
䜐 0
䜑 0
䜒 0
䜓 0
䜔 0
䜔 0
䜕 0
䜖 0
䜗 0
䜘 0
䜙 0
䜙 0
䜚 0
䜛 0
䜜 0
䜝 0
䜞 0
䜟 0
䜠 0
䜡 0
䜢 0
䜪 0
䜫 0
䜫 0
䜬 0
䜭 0
䜮 0
䜯 0
䜰 0
䜱 0
䜲 0
䜴 0
䜴 0
䜵 0
䜵 0
䜶 0
䜷 0
䜸 0
䜸 0
䜹 0
䜺 0
䜻 0
䜻 0
䜼 0
䜼 0
䜼 0
䜼 0
䜽 0
䜾 0
䜿 0
䝀 0
䝁 0
䝂 0
䝃 0
䝄 0
䝅 0
䝆 0
䝇 0
䝈 0
䝊 0
䝋 0
䝌 0
䝌 0
䝍 0
䝎 0
䝏 0
䝐 0
䝑 0
䝒 0
䝓 0
䝔 0
䝕 0
䝖 0
䝗 0
䝘 0
䝚 0
䝜 0
䝝 0
䝝 0
䝞 0
䝞 0
䝟 0
䝟 0
䝠 0
䝡 0
䝡 0
䝢 0
䝢 0
䝣 0
䝤 0
䝥 0
䝥 0
䝦 0
䝧 0
䝨 0
䝨 0
䝩 0
䝪 0
䝫 0
䝬 0
䝭 0
䝮 0
䝯 0
䝰 0
䝱 0
䝲 0
䝳 0
䝳 0
䝴 0
䝵 0
䝶 0
䝸 0
䝹 0
䝺 0
䝻 0
䝽 0
䝾 0
䝿 0
䞀 0
䞁 0
䞂 0
䞃 0
䞄 0
䞅 0
䞆 0
䞈 0
䞉 0
䞊 0
䞋 0
䞋 0
䞑 0
䞒 0
䞓 0
䞓 0
䞔 0
䞕 0
䞖 0
䞗 0
䞗 0
䞘 0
䞙 0
䞚 0
䞚 0
䞜 0
䞜 0
䞝 0
䞞 0
䞞 0
䞟 0
䞡 0
䞢 0
䞣 0
䞤 0
䞤 0
䞤 0
䞥 0
䞦 0
䞧 0
䞨 0
䞩 0
䞪 0
䞫 0
䞬 0
䞭 0
䞭 0
䞮 0
䞯 0
䞯 0
䞰 0
䞰 0
䞰 0
䞱 0
䞳 0
䞳 0
䞴 0
䞵 0
䞶 0
䞷 0
䞷 0
䞸 0
䞹 0
䞺 0
䞻 0
䞼 0
䞽 0
䞾 0
䞾 0
䞿 0
䟀 0
䟁 0
䟂 0
䟃 0
䟄 0
䟄 0
䟄 0
䟅 0
䟆 0
䟇 0
䟈 0
䟉 0
䟊 0
䟋 0
䟌 0
䟌 0
䟍 0
䟎 0
䟏 0
䟐 0
䟑 0
䟒 0
䟓 0
䟓 0
䟔 0
䟔 0
䟕 0
䟖 0
䟗 0
䟘 0
䟘 0
䟙 0
䟚 0
䟛 0
䟜 0
䟝 0
䟞 0
䟟 0
䟠 0
䟡 0
䟢 0
䟣 0
䟤 0
䟥 0
䟦 0
䟧 0
䟨 0
䟩 0
䟩 0
䟪 0
䟪 0
䟫 0
䟬 0
䟭 0
䟮 0
䟮 0
䟯 0
䟰 0
䟱 0
䟴 0
䟵 0
䟶 0
䟷 0
䟷 0
䟸 0
䟸 0
䟹 0
䟺 0
䟻 0
䟼 0
䟽 0
䟽 0
䟽 0
䟾 0
䟾 0
䟿 0
䠀 0
䠀 0
䠁 0
䠂 0
䠃 0
䠄 0
䠅 0
䠆 0
䠇 0
䠈 0
䠉 0
䠊 0
䠋 0
䠍 0
䠎 0
䠎 0
䠏 0
䠏 0
䠐 0
䠑 0
䠑 0
䠑 0
䠒 0
䠓 0
䠓 0
䠓 0
䠔 0
䠕 0
䠗 0
䠘 0
䠙 0
䠛 0
䠜 0
䠝 0
䠞 0
䠟 0
䠠 0
䠡 0
䠢 0
䠣 0
䠤 0
䠥 0
䠥 0
䠦 0
䠧 0
䠨 0
䠩 0
䠪 0
䠫 0
䠬 0
䠭 0
䠮 0
䠯 0
䠰 0
䠰 0
䠱 0
䠲 0
䠳 0
䠴 0
䠵 0
䠶 0
䠶 0
䠶 0
䠶 0
䠷 0
䠸 0
䠹 0
䠻 0
䠼 0
䠽 0
䠾 0
䠿 0
䠿 0
䡀 0
䡀 0
䡀 0
䡁 0
䡂 0
䡂 0
䡃 0
䡅 0
䡅 0
䡆 0
䡇 0
䡈 0
䡈 0
䡈 0
䡉 0
䡊 0
䡋 0
䡋 0
䡌 0
䡍 0
䡎 0
䡏 0
䡐 0
䡑 0
䡒 0
䡓 0
䡔 0
䡕 0
䡖 0
䡗 0
䡘 0
䡙 0
䡚 0
䡛 0
䡜 0
䡝 0
䡞 0
䡟 0
䡠 0
䡡 0
䡢 0
䡣 0
䡣 0
䡤 0
䡥 0
䡦 0
䡨 0
䡩 0
䡩 0
䡩 0
䡪 0
䡫 0
䡬 0
䡭 0
䡮 0
䡯 0
䡰 0
䡰 0
䡰 0
䡲 0
䡴 0
䡵 0
䡶 0
䡷 0
䡹 0
䡺 0
䡼 0
䡽 0
䡾 0
䡿 0
䢃 0
䢄 0
䢅 0
䢇 0
䢈 0
䢉 0
䢊 0
䢋 0
䢍 0
䢎 0
䢐 0
䢑 0
䢑 0
䢒 0
䢓 0
䢔 0
䢕 0
䢖 0
䢖 0
䢙 0
䢚 0
䢚 0
䢛 0
䢜 0
䢜 0
䢝 0
䢞 0
䢟 0
䢠 0
䢡 0
䢡 0
䢡 0
䢢 0
䢤 0
䢥 0
䢦 0
䢧 0
䢨 0
䢨 0
䢨 0
䢨 0
䢩 0
䢩 0
䢪 0
䢪 0
䢪 0
䢫 0
䢬 0
䢭 0
䢮 0
䢯 0
䢯 0
䢰 0
䢱 0
䢲 0
䢳 0
䢴 0
䢵 0
䢶 0
䢷 0
䢷 0
䢷 0
䢷 0
䢷 0
䢷 0
䢷 0
䢷 0
䢸 0
䢹 0
䢹 0
䢻 0
䢼 0
䢽 0
䢾 0
䣀 0
䣁 0
䣂 0
䣃 0
䣄 0
䣅 0
䣆 0
䣇 0
䣈 0
䣊 0
䣋 0
䣌 0
䣍 0
䣍 0
䣏 0
䣐 0
䣑 0
䣓 0
䣔 0
䣕 0
䣕 0
䣖 0
䣖 0
䣗 0
䣗 0
䣘 0
䣙 0
䣙 0
䣚 0
䣛 0
䣜 0
䣝 0
䣞 0
䣟 0
䣠 0
䣡 0
䣢 0
䣣 0
䣤 0
䣤 0
䣥 0
䣧 0
䣨 0
䣩 0
䣪 0
䣫 0
䣬 0
䣬 0
䣭 0
䣮 0
䣯 0
䣰 0
䣱 0
䣱 0
䣲 0
䣴 0
䣵 0
䣵 0
䣶 0
䣶 0
䣷 0
䣸 0
䣸 0
䣸 0
䣹 0
䣺 0
䣻 0
䣼 0
䣽 0
䣾 0
䤁 0
䤁 0
䤁 0
䤂 0
䤄 0
䤅 0
䤆 0
䤇 0
䤈 0
䤈 0
䤉 0
䤊 0
䤊 0
䤋 0
䤌 0
䤍 0
䤎 0
䤏 0
䤐 0
䤑 0
䤒 0
䤓 0
䤔 0
䤕 0
䤖 0
䤗 0
䤘 0
䤙 0
䤚 0
䤛 0
䤛 0
䤜 0
䤝 0
䤞 0
䤟 0
䤠 0
䤡 0
䤢 0
䤤 0
䤥 0
䤦 0
䤧 0
䤨 0
䤪 0
䤫 0
䤬 0
䤭 0
䤮 0
䤯 0
䤰 0
䤱 0
䤳 0
䤴 0
䤵 0
䤵 0
䤶 0
䤶 0
䤷 0
䤸 0
䤹 0
䤺 0
䤻 0
䤽 0
䤾 0
䤿 0
䥀 0
䥁 0
䥂 0
䥃 0
䥄 0
䥅 0
䥆 0
䥈 0
䥈 0
䥉 0
䥊 0
䥊 0
䥊 0
䥋 0
䥌 0
䥍 0
䥎 0
䥏 0
䥐 0
䥑 0
䥒 0
䥔 0
䥕 0
䥖 0
䥖 0
䥗 0
䥘 0
䥘 0
䥙 0
䥚 0
䥛 0
䥝 0
䥞 0
䥟 0
䥠 0
䥡 0
䥢 0
䥣 0
䥣 0
䥤 0
䥥 0
䥦 0
䥧 0
䥧 0
䥧 0
䥨 0
䥨 0
䥩 0
䥪 0
䥫 0
䥬 0
䥭 0
䥭 0
䥮 0
䥯 0
䥯 0
䥰 0
䥱 0
䥳 0
䥴 0
䥴 0
䥴 0
䥵 0
䥶 0
䥶 0
䥷 0
䥷 0
䥷 0
䥸 0
䦇 0
䦈 0
䦈 0
䦊 0
䦋 0
䦌 0
䦍 0
䦎 0
䦏 0
䦐 0
䦑 0
䦒 0
䦓 0
䦔 0
䦔 0
䦕 0
䦖 0
䦖 0
䦗 0
䦘 0
䦙 0
䦙 0
䦚 0
䦜 0
䦝 0
䦞 0
䦟 0
䦠 0
䦠 0
䦠 0
䦡 0
䦢 0
䦣 0
䦤 0
䦥 0
䦦 0
䦧 0
䦧 0
䦧 0
䦧 0
䦨 0
䦪 0
䦫 0
䦬 0
䦭 0
䦮 0
䦯 0
䦱 0
䦲 0
䦳 0
䦴 0
䦵 0
䦹 0
䦺 0
䦻 0
䦼 0
䦼 0
䦽 0
䧁 0
䧂 0
䧃 0
䧄 0
䧅 0
䧆 0
䧉 0
䧊 0
䧋 0
䧋 0
䧌 0
䧍 0
䧍 0
䧎 0
䧐 0
䧐 0
䧑 0
䧒 0
䧓 0
䧔 0
䧕 0
䧖 0
䧗 0
䧘 0
䧙 0
䧚 0
䧚 0
䧚 0
䧛 0
䧛 0
䧛 0
䧜 0
䧝 0
䧞 0
䧞 0
䧟 0
䧟 0
䧠 0
䧡 0
䧡 0
䧢 0
䧣 0
䧣 0
䧤 0
䧥 0
䧦 0
䧧 0
䧨 0
䧪 0
䧬 0
䧮 0
䧮 0
䧯 0
䧰 0
䧱 0
䧲 0
䧵 0
䧶 0
䧷 0
䧸 0
䧹 0
䧺 0
䧻 0
䧼 0
䧼 0
䧼 0
䧽 0
䧾 0
䧿 0
䧿 0
䨀 0
䨁 0
䨂 0
䨃 0
䨅 0
䨆 0
䨆 0
䨇 0
䨇 0
䨈 0
䨉 0
䨊 0
䨋 0
䨋 0
䨌 0
䨌 0
䨌 0
䨍 0
䨎 0
䨎 0
䨐 0
䨑 0
䨒 0
䨓 0
䨔 0
䨔 0
䨔 0
䨖 0
䨗 0
䨘 0
䨘 0
䨙 0
䨛 0
䨜 0
䨝 0
䨞 0
䨟 0
䨠 0
䨡 0
䨢 0
䨣 0
䨤 0
䨥 0
䨥 0
䨦 0
䨧 0
䨨 0
䨩 0
䨪 0
䨫 0
䨬 0
䨭 0
䨮 0
䨮 0
䨯 0
䨰 0
䨱 0
䨲 0
䨳 0
䨴 0
䨵 0
䨶 0
䨷 0
䨷 0
䨸 0
䨹 0
䨺 0
䨻 0
䨼 0
䨽 0
䨾 0
䨿 0
䩀 0
䩁 0
䩂 0
䩃 0
䩅 0
䩆 0
䩇 0
䩇 0
䩈 0
䩉 0
䩊 0
䩋 0
䩌 0
䩍 0
䩏 0
䩐 0
䩑 0
䩒 0
䩓 0
䩔 0
䩔 0
䩔 0
䩕 0
䩖 0
䩖 0
䩘 0
䩙 0
䩚 0
䩚 0
䩛 0
䩜 0
䩝 0
䩝 0
䩞 0
䩞 0
䩟 0
䩟 0
䩠 0
䩡 0
䩡 0
䩢 0
䩣 0
䩤 0
䩥 0
䩦 0
䩧 0
䩨 0
䩩 0
䩪 0
䩫 0
䩬 0
䩭 0
䩮 0
䩯 0
䩯 0
䩰 0
䩱 0
䩲 0
䩳 0
䩴 0
䩵 0
䩶 0
䩷 0
䩸 0
䩸 0
䩹 0
䩺 0
䩻 0
䩼 0
䩾 0
䩿 0
䪀 0
䪁 0
䪂 0
䪃 0
䪄 0
䪅 0
䪆 0
䪇 0
䪇 0
䪈 0
䪉 0
䪊 0
䪋 0
䪌 0
䪍 0
䪎 0
䪏 0
䪐 0
䪑 0
䪒 0
䪓 0
䪔 0
䪕 0
䪖 0
䪗 0
䪘 0
䪙 0
䪚 0
䪛 0
䪜 0
䪝 0
䪝 0
䪝 0
䪞 0
䪠 0
䪠 0
䪡 0
䪡 0
䪢 0
䪢 0
䪣 0
䪣 0
䪤 0
䪥 0
䪥 0
䪦 0
䪧 0
䪨 0
䪩 0
䪫 0
䪬 0
䪭 0
䪮 0
䪯 0
䪱 0
䪱 0
䪳 0
䪴 0
䪴 0
䪴 0
䪵 0
䪶 0
䪷 0
䪷 0
䪸 0
䪹 0
䪹 0
䪺 0
䪻 0
䪼 0
䪽 0
䪾 0
䪿 0
䪿 0
䫀 0
䫁 0
䫂 0
䫃 0
䫃 0
䫄 0
䫅 0
䫆 0
䫇 0
䫈 0
䫈 0
䫉 0
䫊 0
䫋 0
䫋 0
䫌 0
䫍 0
䫍 0
䫎 0
䫐 0
䫒 0
䫔 0
䫔 0
䫔 0
䫔 0
䫕 0
䫖 0
䫖 0
䫗 0
䫘 0
䫘 0
䫙 0
䫚 0
䫛 0
䫝 0
䫞 0
䫟 0
䫠 0
䫡 0
䫣 0
䫤 0
䫥 0
䫦 0
䫦 0
䫧 0
䫨 0
䫩 0
䫪 0
䫫 0
䫬 0
䫬 0
䫭 0
䫮 0
䫰 0
䫱 0
䫲 0
䫳 0
䫴 0
䫵 0
䫶 0
䫷 0
䫸 0
䫸 0
䫹 0
䫺 0
䫻 0
䫼 0
䫽 0
䫾 0
䫿 0
䬀 0
䬁 0
䬂 0
䬃 0
䬄 0
䬅 0
䬆 0
䬇 0
䬈 0
䬉 0
䬊 0
䬋 0
䬌 0
䬍 0
䬎 0
䬏 0
䬐 0
䬑 0
䬒 0
䬓 0
䬔 0
䬕 0
䬕 0
䬖 0
䬗 0
䬘 0
䬙 0
䬚 0
䬛 0
䬜 0
䬝 0
䬞 0
䬟 0
䬠 0
䬡 0
䬢 0
䬢 0
䬣 0
䬤 0
䬤 0
䬤 0
䬥 0
䬦 0
䬦 0
䬧 0
䬧 0
䬨 0
䬩 0
䬪 0
䬫 0
䬬 0
䬭 0
䬭 0
䬮 0
䬯 0
䬯 0
䬰 0
䬱 0
䬲 0
䬳 0
䬴 0
䬴 0
䬵 0
䬵 0
䬶 0
䬷 0
䬸 0
䬹 0
䬺 0
䬻 0
䬼 0
䬽 0
䬽 0
䬾 0
䬿 0
䭀 0
䭁 0
䭂 0
䭃 0
䭃 0
䭄 0
䭅 0
䭅 0
䭆 0
䭇 0
䭇 0
䭈 0
䭉 0
䭊 0
䭋 0
䭌 0
䭍 0
䭎 0
䭐 0
䭑 0
䭑 0
䭑 0
䭒 0
䭓 0
䭔 0
䭕 0
䭖 0
䭗 0
䭘 0
䭙 0
䭚 0
䭛 0
䭝 0
䭞 0
䭟 0
䭠 0
䭡 0
䭢 0
䭣 0
䭤 0
䭥 0
䭦 0
䭧 0
䭨 0
䭩 0
䭫 0
䭫 0
䭬 0
䭬 0
䭭 0
䭮 0
䭯 0
䭰 0
䭱 0
䭲 0
䭳 0
䭴 0
䭵 0
䭶 0
䭷 0
䭸 0
䭹 0
䭺 0
䭻 0
䭼 0
䭽 0
䭾 0
䭾 0
䭿 0
䮀 0
䮁 0
䮂 0
䮃 0
䮄 0
䮅 0
䮆 0
䮇 0
䮈 0
䮉 0
䮊 0
䮋 0
䮌 0
䮍 0
䮎 0
䮏 0
䮐 0
䮐 0
䮑 0
䮒 0
䮓 0
䮓 0
䮔 0
䮕 0
䮖 0
䮗 0
䮘 0
䮘 0
䮙 0
䮚 0
䮛 0
䮜 0
䮝 0
䮞 0
䮟 0
䮠 0
䮡 0
䮢 0
䮣 0
䮤 0
䮥 0
䮦 0
䮧 0
䮨 0
䮩 0
䮪 0
䮫 0
䮬 0
䮭 0
䮮 0
䮯 0
䮯 0
䮰 0
䮱 0
䮲 0
䮳 0
䮴 0
䮵 0
䮶 0
䮷 0
䮸 0
䮸 0
䮹 0
䮹 0
䮺 0
䮻 0
䮼 0
䮽 0
䮾 0
䮿 0
䮿 0
䯀 0
䯁 0
䯂 0
䯂 0
䯆 0
䯇 0
䯈 0
䯉 0
䯊 0
䯊 0
䯋 0
䯌 0
䯍 0
䯎 0
䯎 0
䯏 0
䯐 0
䯑 0
䯒 0
䯓 0
䯔 0
䯕 0
䯖 0
䯗 0
䯗 0
䯘 0
䯙 0
䯚 0
䯛 0
䯜 0
䯝 0
䯞 0
䯟 0
䯠 0
䯡 0
䯢 0
䯣 0
䯤 0
䯥 0
䯦 0
䯧 0
䯨 0
䯪 0
䯫 0
䯬 0
䯭 0
䯮 0
䯯 0
䯰 0
䯱 0
䯲 0
䯳 0
䯴 0
䯵 0
䯶 0
䯷 0
䯸 0
䯹 0
䯺 0
䯼 0
䯽 0
䯽 0
䯽 0
䯾 0
䯿 0
䯿 0
䯿 0
䰀 0
䰁 0
䰂 0
䰃 0
䰅 0
䰆 0
䰈 0
䰉 0
䰊 0
䰋 0
䰌 0
䰌 0
䰍 0
䰎 0
䰏 0
䰏 0
䰐 0
䰑 0
䰒 0
䰓 0
䰔 0
䰔 0
䰕 0
䰖 0
䰗 0
䰘 0
䰘 0
䰙 0
䰚 0
䰛 0
䰛 0
䰜 0
䰜 0
䰝 0
䰞 0
䰟 0
䰠 0
䰡 0
䰢 0
䰣 0
䰤 0
䰥 0
䰦 0
䰧 0
䰨 0
䰩 0
䰪 0
䰫 0
䰬 0
䰭 0
䰯 0
䰰 0
䰱 0
䰲 0
䰴 0
䰷 0
䰸 0
䰹 0
䰹 0
䰺 0
䰻 0
䰼 0
䰼 0
䰼 0
䰽 0
䰾 0
䰿 0
䱀 0
䱁 0
䱂 0
䱃 0
䱄 0
䱅 0
䱆 0
䱇 0
䱈 0
䱈 0
䱉 0
䱊 0
䱋 0
䱌 0
䱍 0
䱎 0
䱏 0
䱏 0
䱐 0
䱑 0
䱒 0
䱓 0
䱔 0
䱔 0
䱔 0
䱕 0
䱖 0
䱗 0
䱘 0
䱙 0
䱚 0
䱛 0
䱜 0
䱝 0
䱝 0
䱞 0
䱟 0
䱠 0
䱡 0
䱢 0
䱣 0
䱤 0
䱥 0
䱦 0
䱧 0
䱨 0
䱫 0
䱬 0
䱭 0
䱮 0
䱯 0
䱰 0
䱱 0
䱲 0
䱳 0
䱳 0
䱵 0
䱶 0
䱷 0
䱸 0
䱹 0
䱺 0
䱻 0
䱼 0
䱽 0
䱾 0
䱿 0
䲀 0
䲁 0
䲂 0
䲃 0
䲃 0
䲃 0
䲄 0
䲅 0
䲆 0
䲊 0
䲋 0
䲌 0
䲍 0
䲎 0
䲏 0
䲐 0
䲑 0
䲒 0
䲓 0
䲔 0
䲔 0
䲕 0
䲖 0
䲗 0
䲙 0
䲚 0
䲛 0
䲜 0
䲥 0
䲦 0
䲧 0
䲨 0
䲪 0
䲫 0
䲬 0
䲮 0
䲯 0
䲯 0
䲰 0
䲱 0
䲲 0
䲳 0
䲴 0
䲵 0
䲵 0
䲵 0
䲷 0
䲸 0
䲹 0
䲺 0
䲻 0
䲼 0
䲽 0
䲽 0
䲾 0
䲿 0
䳀 0
䳀 0
䳁 0
䳂 0
䳂 0
䳂 0
䳃 0
䳃 0
䳄 0
䳅 0
䳆 0
䳇 0
䳈 0
䳉 0
䳊 0
䳋 0
䳌 0
䳍 0
䳎 0
䳏 0
䳏 0
䳐 0
䳑 0
䳒 0
䳓 0
䳔 0
䳕 0
䳖 0
䳗 0
䳘 0
䳙 0
䳚 0
䳛 0
䳜 0
䳝 0
䳞 0
䳟 0
䳠 0
䳡 0
䳢 0
䳣 0
䳤 0
䳦 0
䳧 0
䳨 0
䳩 0
䳪 0
䳫 0
䳬 0
䳭 0
䳮 0
䳯 0
䳯 0
䳰 0
䳱 0
䳱 0
䳲 0
䳳 0
䳴 0
䳵 0
䳶 0
䳶 0
䳷 0
䳸 0
䳻 0
䳻 0
䳽 0
䳽 0
䳾 0
䳿 0
䴀 0
䴁 0
䴂 0
䴃 0
䴄 0
䴅 0
䴆 0
䴆 0
䴇 0
䴈 0
䴉 0
䴊 0
䴋 0
䴌 0
䴍 0
䴎 0
䴏 0
䴏 0
䴐 0
䴑 0
䴒 0
䴚 0
䴛 0
䴜 0
䴝 0
䴞 0
䴟 0
䴠 0
䴡 0
䴡 0
䴢 0
䴢 0
䴣 0
䴤 0
䴥 0
䴦 0
䴧 0
䴩 0
䴪 0
䴫 0
䴬 0
䴭 0
䴮 0
䴯 0
䴰 0
䴱 0
䴲 0
䴳 0
䴴 0
䴵 0
䴶 0
䴷 0
䴸 0
䴹 0
䴹 0
䴹 0
䴺 0
䴻 0
䴼 0
䴽 0
䴾 0
䴿 0
䵀 0
䵁 0
䵂 0
䵃 0
䵃 0
䵄 0
䵅 0
䵆 0
䵇 0
䵈 0
䵉 0
䵊 0
䵋 0
䵌 0
䵍 0
䵎 0
䵏 0
䵐 0
䵑 0
䵒 0
䵓 0
䵔 0
䵕 0
䵖 0
䵗 0
䵘 0
䵙 0
䵚 0
䵛 0
䵜 0
䵝 0
䵞 0
䵞 0
䵟 0
䵠 0
䵡 0
䵢 0
䵣 0
䵤 0
䵥 0
䵦 0
䵧 0
䵨 0
䵩 0
䵪 0
䵫 0
䵬 0
䵭 0
䵮 0
䵯 0
䵰 0
䵱 0
䵱 0
䵲 0
䵳 0
䵴 0
䵴 0
䵴 0
䵵 0
䵵 0
䵶 0
䵷 0
䵸 0
䵺 0
䵻 0
䵼 0
䵽 0
䵾 0
䵿 0
䶀 0
䶀 0
䶁 0
䶂 0
䶂 0
䶂 0
䶂 0
䶃 0
䶄 0
䶅 0
䶆 0
䶇 0
䶈 0
䶉 0
䶊 0
䶋 0
䶌 0
䶌 0
䶎 0
䶏 0
䶏 0
䶐 0
䶑 0
䶒 0
䶓 0
䶔 0
䶕 0
䶖 0
䶗 0
䶘 0
䶙 0
䶚 0
䶛 0
䶜 0
䶝 0
䶞 0
䶟 0
䶠 0
䶡 0
䶢 0
䶣 0
䶤 0
䶥 0
䶦 0
䶧 0
䶧 0
䶨 0
䶩 0
䶪 0
䶫 0
䶬 0
䶯 0
䶰 0
䶱 0
䶳 0
䶴 0
䶵 0
䶵 0
䶵 0
一 34112
一 181207
一 14785
一一 2095
一丁不識 0
一丁不識 0
一丁點 17
一丁點 17
一丁點兒 0
一丁點兒 0
一丁點兒 0
一丁點兒 0
一丈 20
一丈 20
一上來 22
一上來 22
一上去 2
一上去 2
一上台 6
一上台 6
一上場 11
一上場 11
一上場 11
一上場 11
一下 10576
一下 10576
一下下 73
一下下 73
一下來 11
一下來 11
一下去 6
一下去 6
一下台 8
一下台 8
一下場 1
一下場 1
一下場 1
一下場 1
一下子 492
一下子 492
一不 2847
一不 2847
一不做二不休 0
一不做二不休 0
一不小心 130
一不小心 130
一不注意 2
一不注意 2
一世 1499
一世 753
一世紀 1585
一世紀 1585
一世英名 16
一世英名 16
一丘之貉 5
一丘之貉 5
一丙 31
一中 409
一中 409
一串 180
一串 180
一串串 26
一串串 26
一九九一 159
一九九七 590
一九九三 171
一九九九 200
一九九二 158
一九九五 357
一九九八 912
一九九六 464
一九九四 244
一乾二淨 24
一乾二淨 24
一了百了 6
一了百了 6
一了百當 0
一了百當 0
一事 1500
一事 753
一事無成 25
一事無成 25
一二八事變 0
一五 709
一五一十 20
一些 15925
一些 15925
一些些 50
一些些 50
一人 68
一人 68
一人份 12
一人份 12
一人傳虛 0
一人傳虛 0
一介 43
一介 43
一介不取 1
一介不取 1
一仗 22
一仗 22
一付 411
一付 158
一代 963
一代 963
一代不如一代 4
一代不如一代 4
一代不如一代 4
一代不如一代 4
一代代 12
一代代 12
一代巨擘 0
一代巨擘 0
一代新人換舊人 0
一代新人換舊人 0
一代英豪 1
一代英豪 1
一代豪傑 1
一代豪傑 1
一以貫之 0
一以貫之 0
一件 2033
一件 2033
一件事 593
一件事 593
一任 234
一任 234
一份 3871
一份 3871
一伍 1
一伍 1
一伙 11
一伙 11
一伙人 4
一伙人 4
一似 9
一似 9
一位 6686
一位 6686
一位數 17
一位數 17
一住 7
一住 7
一佛出世 0
一佛出世 0
一併 773
一併 773
一併處理 12
一併處理 12
一併處理 12
一併處理 12
一來 314
一來 314
一來一往 15
一來一往 15
一來一往 15
一來一往 15
一來到 4
一來到 4
一例 216
一例 216
一信 107
一信 107
一個 1
一個 65753
一個 65753
一個人 1
一個人 3284
一個人 3284
一個個 281
一個個 281
一個個 281
一個個 281
一個個 281
一個個 281
一個個 281
一個個 281
一個勁兒 1
一個勁兒 3
一個勁兒 3
一個勁兒 3
一個勁兒 3
一個巴掌拍不響 1
一個巴掌拍不響 0
一個巴掌拍不響 0
一個接一個 1
一個接一個 47
一個接一個 47
一個接一個 47
一個接一個 47
一個月 1
一個月 11998
一個月 11998
一個樣 1
一個樣 12
一個樣 12
一個蘿蔔一個坑 0
一個蘿蔔一個坑 2
一個蘿蔔一個坑 2
一個蘿蔔一個坑 2
一個蘿蔔一個坑 2
一個頭兩個大 0
一個頭兩個大 0
一個頭兩個大 1
一個鼻孔 1
一個鼻孔 1
一個鼻孔 1
一倍 317
一倍 317
一倡三歎 0
一倡三歎 0
一倡百和 0
一倡百和 0
一偏 6
一偏 6
一做 20
一做 20
一做到 0
一做到 0
一停 17
一停 17
一側 99
一側 99
一傳十 13
一億 782
一億 782
一億人 9
一億人 9
一億元 208
一億元 208
一元 435
一元 435
一元化 88
一元化 88
一元復始 14
一元復始 14
一元方程 0
一元方程 0
一元方程式 0
一元方程式 0
一元說 2
一元說 2
一元論 6
一元論 6
一兆 53
一兆 53
一兆元 14
一兆元 14
一克 13
一克 13
一內一外 0
一內一外 0
一內一外 0
一內一外 0
一兩 157
一兩 157
一共 467
一共 467
一共是 9
一共是 9
一兵 6
一兵 6
一兵一卒 5
一兵一卒 5
一兵一卒 5
一兵一卒 5
一具 226
一具 226
一冊 429
一冊 429
一再 1026
一再 1026
一再叮嚀 0
一再叮嚀 0
一再的 91
一再的 91
一准 0
一准 0
一出 127
一出 127
一出來 14
一出來 14
一出去 4
一出去 4
一出場 19
一出場 19
一出場 19
一出場 19
一刀 259
一刀 259
一刀兩斷 6
一刀兩斷 6
一分 552
一分 552
一分 552
一分一秒 39
一分一秒 39
一分一秒 39
一分一秒 39
一分子 21
一分子 21
一分子 21
一分收獲 4
一分收穫 4
一分收穫 4
一分為二 25
一分為二 25
一分耕耘 28
一分耕耘 28
一分錢 57
一分錢 57
一分錢一分貨 0
一分錢一分貨 0
一分錢一分貨 0
一分錢一分貨 0
一分鐘 369
一分鐘 369
一切 5997
一切 5997
一切從簡 0
一切從簡 0
一切都 447
一切都 447
一切都在 49
一切都在 49
一切都是 220
一切都是 220
一划 0
一划 0
一別 36
一別 36
一別多年 1
一別多年 1
一到 400
一到 792
一券 3
一券 3
一刻 424
一刻 424
一刻千金 4
一刻千金 4
一則 15890
一則 15890
一則以喜 4
一則以喜 4
一則以憂 5
一則以憂 5
一前 13
一前 13
一前一後 23
一前一後 23
一前一後 23
一前一後 23
一剎那 38
一剎那 38
一剎那 38
一剎那 38
一副 412
一副 412
一割 0
一割 0
一劃 61
一劃 61
一劑 73
一劑 73
一加侖 2
一加侖 2
一動 186
一動 186
一動不動 7
一動不動 7
一動不如一靜 2
一動不如一靜 2
一動不如一靜 2
一動不如一靜 2
一勞永逸 48
一勞永逸 48
一包 131
一包 131
一匙 17
一匙 17
一匙之量 0
一匙之量 0
一匹 99
一匹 99
一區 237
一區 237
一千 2751
一千 2751
一千人 38
一千人 38
一千個 84
一千個 84
一千元 286
一千元 286
一千塊 10
一千塊 10
一千萬 162
一千萬 162
一千零一 115
一千零一 115
一千零一夜 0
一千零一夜 0
一半 1902
一半 1902
一去 68
一去 68
一去不回 12
一去不回 12
一去不回頭 0
一去不回頭 0
一去不回頭 0
一去不回頭 0
一去不復 0
一去不復 0
一去不復返 10
一去不復返 10
一去不返 13
一去不返 13
一反常態 15
一反常態 15
一口 577
一口 577
一口咬定 19
一口咬定 19
一口氣 461
一口氣 461
一口飯 0
一口飯 0
一古腦兒 4
一古腦兒 4
一古腦兒 4
一古腦兒 4
一句 1482
一句 1482
一句話 737
一句話 737
一只 297
一只 297
一台 1640
一台 1640
一吋 160
一吋 160
一同 1145
一同 1145
一名 2286
一名 2286
一吐 28
一吐 28
一吐為快 8
一吐為快 8
一向 1355
一向 1355
一向是 151
一向是 151
一告 2
一告 2
一呎 2
一呎 2
一周 440
一周 440
一周年 24
一周年 24
一味 306
一味 306
一呼百應 5
一呼百應 5
一呼百諾 3
一呼百諾 3
一命 73
一命 73
一命嗚呼 10
一命嗚呼 10
一和 77
一和 77
一咬 8
一咬 8
一品 118
一品 118
一品夫人 8
一品夫人 8
一品鍋 0
一品鍋 0
一哄而上 0
一哄而上 0
一哄而散 4
一哄而散 4
一員 289
一員 289
一員大將 11
一員大將 11
一唱一和 2
一唱一和 2
一唱一和 2
一唱一和 2
一問 231
一問 231
一問一答 0
一問一答 0
一問三不知 7
一問三不知 7
一問之下 13
一問之下 13
一喊 8
一喊 8
一喊出 0
一喊出 0
一喊到 0
一喊到 0
一喝 9
一喝 9
一喝 9
一喝 9
一喝到 0
一喝到 0
一噸 19
一噸 19
一噸半 0
一噸半 0
一回 372
一回 372
一回事 221
一回事 221
一回來 19
一回來 19
一回到 11
一回到 11
一回去 1
一回去 1
一回生二回熟 0
一回生二回熟 0
一回生兩回熟 0
一回生兩回熟 0
一圈 213
一圈 213
一國 524
一國 524
一國之主 5
一國之主 5
一國之君 5
一國之君 5
一國兩制 0
一國兩制 0
一圓 69
一圓 69
一團 231
一團 231
一團和氣 8
一團和氣 8
一團糟 11
一團糟 11
一在 75
一在 75
一地 180
一地 180
一地裡 0
一地裡 0
一堂 148
一堂 148
一堂課 52
一堂課 52
一堆 1198
一堆 1198
一堆堆 11
一堆堆 11
一報 22
一報 22
一報到 0
一報到 0
一報還一報 0
一報還一報 0
一報還一報 0
一報還一報 0
一場 29
一場 29
一場 2913
一場 2913
一場空 21
一場空 21
一場空 21
一場空 21
一塊 1032
一塊 1032
一塊兒 42
一塊兒 42
一塊兒 42
一塊兒 42
一塊塊 35
一塊塊 35
一塊肉 0
一塊肉 0
一塊錢 2153
一塊錢 2153
一塌糊塗 12
一塌糊塗 12
一塌糊塗 12
一塌糊塗 12
一塗 0
一塗 0
一填 14
一填 14
一塵不染 12
一塵不染 12
一壁 2
一壁 2
一壓 8
一壓 8
一壓下 0
一壓下 0
一壓到 0
一壓到 0
一壓在 0
一壓在 0
一壘 44
一壘 44
一壘手 31
一壘手 31
一壺 64
一壺 64
一壺水 5
一壺水 5
一壺酒 7
一壺酒 7
一夔已足 0
一夔已足 0
一夕 84
一夕 84
一夕之間 38
一夕之間 38
一多 47
一多 47
一夜 953
一夜 953
一夜情 1
一夜情 1
一夜無夢 0
一夜無夢 0
一夜致富 0
一夜致富 0
一夜風流 1
一夜風流 1
一夢漫言 1
一夥 22
一夥 22
一夥人 17
一夥人 17
一夥兒 0
一夥兒 0
一夥兒 0
一夥兒 0
一大堆 464
一大堆 464
一大塊 32
一大塊 32
一大早 124
一大早 124
一大會兒 0
一大會兒 0
一大會兒 0
一大會兒 0
一大片 100
一大片 100
一天 5007
一天 5007
一天到晚 122
一天到晚 122
一天天 61
一天天 61
一夫 64
一夫 64
一夫一妻 11
一夫一妻 11
一夫一妻 11
一夫一妻 11
一夫一妻制 7
一夫一妻制 7
一夫一妻制 7
一夫一妻制 7
一夫多妻 8
一夫多妻 8
一夫當關 4
一夫當關 4
一失足成千古恨 1
一失足成千古恨 1
一套 3390
一套 3390
一套套 3
一套套 3
一女一男 2
一女一男 2
一女一男 2
一女一男 2
一好 16
一好 16
一如 420
一如 420
一如往昔 16
一如往昔 16
一如既往 4
一如既往 4
一妻 6
一妻 6
一妻制 1
一妻制 1
一妻多夫 0
一妻多夫 0
一妻多夫制 0
一妻多夫制 0
一子一女 9
一子一女 9
一子一女 9
一子一女 9
一孔之見 0
一孔之見 0
一字 549
一字 549
一字不差 0
一字不差 0
一字不漏 0
一字不漏 0
一字不識 0
一字不識 0
一字之差 6
一字之差 6
一字之師 0
一字之師 0
一字千金 5
一字千金 5
一字師 0
一字師 0
一存 2
一存 2
一孝 8
一孝 8
一季 884
一季 884
一學 270
一學 270
一學到 0
一學到 0
一學期 242
一學期 242
一學起 0
一學起 0
一官半職 3
一官半職 3
一定 6638
一定 6638
一定在 37
一定在 37
一定是 653
一定是 653
一定會 1546
一定會 1546
一定有 326
一定有 326
一定能 179
一定能 179
一客 53
一客 53
一家 2968
一家 2968
一家之主 0
一家之主 0
一家人 209
一家人 209
一家大小 0
一家大小 0
一家子 121
一家子 121
一宿 0
一宿 0
一寫 17
一寫 17
一寫出 0
一寫出 0
一寫到 0
一寫到 0
一寶 27
一寶 27
一寸 71
一寸 71
一寸光陰一寸金 1
一寸光陰一寸金 1
一寸光陰一寸金 1
一寸光陰一寸金 1
一封 5906
一封 5906
一封信 484
一封信 484
一封封 8
一封封 8
一將 50
一將 50
一將功成萬骨枯 1
一將功成萬骨枯 1
一將難求 0
一將難求 0
一尊 103
一尊 103
一對 1157
一對 1157
一對一 230
一對多 0
一對多 0
一對對 15
一對對 15
一小 305
一小 305
一小堆 3
一小堆 3
一小塊 46
一小塊 46
一小撮 17
一小撮 17
一小時 490
一小時 490
一小片 24
一小片 24
一尺 67
一尺 67
一局 122
一局 122
一局上半 0
一局上半 0
一局下半 0
一局下半 0
一屁股 26
一屁股 26
一屁股債 3
一屁股債 3
一屆 2341
一屆 2341
一展身手 0
一展身手 0
一展長才 18
一展長才 18
一層 7021
一層 7021
一層層 43
一層層 43
一層樓 44
一層樓 44
一山 77
一山 77
一山還比一山高 1
一山還比一山高 1
一山還比一山高 1
一山還比一山高 1
一左一右 4
一左一右 4
一己 124
一己 124
一己之私 10
一己之私 10
一己之見 0
一己之見 0
一巴掌 25
一巴掌 25
一巷 131
一帆風順 35
一帆風順 35
一席 141
一席 141
一席之地 153
一席之地 153
一席話 47
一席話 47
一帶 622
一帶 622
一幀 16
一幀 16
一幅 583
一幅 583
一幕 496
一幕 496
一幫 20
一幫 20
一干 58
一干 58
一年 8581
一年半 100
一年 8581
一年半 100
一年之計 8
一年之計 8
一年到頭 17
一年到頭 17
一年半載 17
一年半載 17
一年四季 73
一年四季 73
一年生 49
一年生 49
一年生植 0
一年生植 0
一年級 1328
一年級 1328
一度 991
一度 991
一座 1439
一座 1439
一廂 3
一廂 3
一廂情願 67
一廂情願 67
一廳 67
一廳 67
一式 1499
一式 753
一張 29114
一張 29114
一彈 15
一彈 15
一彈指 1
一彈指 1
一往 23
一往 23
一往情深 15
一往情深 15
一往直前 1
一往直前 1
一律 1280
一律 1280
一得 26
一得 26
一得到 3
一得到 3
一德一心 0
一德一心 0
一德一心 0
一德一心 0
一心 476
一心 476
一心一意 71
一心一意 71
一心一意 71
一心一意 71
一心二用 0
一心二用 0
一心向上 0
一心向上 0
一忍再忍 0
一忍再忍 0
一忠 408
一忠 33
一念 115
一念 115
一念三千 1
一念之差 21
一念之差 21
一念之間 43
一念之間 43
一念無明 1
一忽兒 3
一忽兒 3
一忽兒 3
一忽兒 3
一忽見 0
一忽見 0
一怒 11
一怒 11
一怒之下 17
一怒之下 17
一息尚存 6
一息尚存 6
一想 373
一想 373
一想之下 0
一想之下 0
一意 36
一意 36
一意孤行 20
一意孤行 20
一愛 12
一愛 12
一愛上 0
一愛上 0
一愣 27
一愣 27
一應 49
一應 49
一應俱全 124
一應俱全 124
一成 442
一成 442
一成不變 92
一成不變 92
一截 66
一截 66
一戰 129
一戰 129
一戰之下 0
一戰之下 0
一戶 147
一戶 147
一戶戶 2
一戶戶 2
一房 51
一房 51
一房一廳 25
一房一廳 25
一房一廳 25
一房一廳 25
一所 569
一所 569
一所所 0
一所所 0
一扇 149
一扇 149
一扇門 39
一扇門 39
一手 922
一手 582
一手交貨 7
一手交貨 7
一手交錢 8
一手交錢 8
一手包辦 27
一手包辦 27
一手提拔 0
一手提拔 0
一手遮天 4
一手遮天 4
一打 53
一打 53
一打 53
一打 53
一扯 11
一扯 11
一批 656
一批 656
一技 26
一技 26
一技之長 70
一技之長 70
一把 895
一把 895
一把手 2
一把手 2
一把抓 19
一把抓 19
一把辛酸淚 0
一把辛酸淚 0
一披 1
一披 1
一抬 15
一抬 15
一抬到 0
一抬到 0
一抬頭 11
一抬頭 11
一抱 20
一抱 20
一抹 76
一抹 76
一拉 29
一拉 29
一拉之下 0
一拉之下 0
一拍 72
一拍 72
一拍即合 21
一拍即合 21
一招 246
一招 246
一招半式 14
一招半式 14
一拜 23
一拜 23
一拜天地 1
一拜天地 1
一拳 54
一拳 54
一指神功 0
一指神功 0
一按 59
一按 59
一按下 1
一按下 1
一按到 0
一按到 0
一按在 0
一按在 0
一掃 81
一掃 81
一掃而光 4
一掃而光 4
一掃而空 26
一掃而空 26
一掉 0
一掉 0
一掉頭 0
一掉頭 0
一掌 134
一掌 134
一排 203
一排 203
一掛 12
一掛 12
一掛之下 0
一掛之下 0
一推 30
一推 30
一推之下 0
一推之下 0
一提 104
一提 104
一提之下 0
一提之下 0
一揮 60
一揮 60
一揮而就 2
一揮而就 2
一搖 23
一搖 23
一搞 60
一搞 60
一摸 25
一摸 25
一摸之下 0
一摸之下 0
一撥 19
一撥 19
一播 2
一播 2
一播 2
一播 2
一播出 1
一播出 1
一播出 1
一播出 1
一撮 22
一撮 22
一撮 22
一撮 22
一擁而上 0
一擁而上 0
一擲千金 2
一擲千金 2
一支 981
一支 981
一改 74
一改 74
一放 26
一放 26
一敗塗地 31
一敗塗地 31
一敷 1
一敷 1
一數 47
一數 47
一文 383
一文 383
一文不值 29
一文不值 29
一文不名 2
一文不名 2
一文錢 7
一文錢 7
一斤 74
一斤 74
一新 286
一新 286
一新耳目 0
一新耳目 0
一斷 8
一斷 8
一方 837
一方 837
一方面 2096
一方面 2096
一旁 486
一旁 486
一旁涼快 0
一旁涼快 0
一旅 13
一旅 13
一日 6486
一日 6486
一日三市 0
一日三市 0
一日三秋 0
一日三秋 0
一日不見 0
一日不見 0
一日之計 1
一日之計 1
一日之計在於晨 10
一日之計在於晨 10
一日之雅 0
一日之雅 0
一日九遷 0
一日九遷 0
一日份 1
一日份 1
一日千里 65
一日千里 65
一日夫妻百日恩 1
一日夫妻百日恩 1
一日數驚 0
一日數驚 0
一日為師 4
一日為師 4
一旦 2175
一旦 2175
一早 219
一早 219
一早就 28
一早就 28
一旬 2
一旬 2
一星 119
一星 119
一星期 4741
一星期 4741
一昧 0
一是 1499
一是 753
一時 1281
一時 1281
一時一刻 2
一時一刻 2
一時一刻 2
一時一刻 2
一時三刻 0
一時三刻 0
一時之選 23
一時之選 23
一時之間 146
一時之間 146
一時半刻 0
一時半刻 0
一時糊塗 0
一時糊塗 0
一時間 115
一時間 115
一晃 35
一晃 35
一晃 35
一晃 35
一晃眼 14
一晃眼 14
一晃眼 14
一晃眼 14
一晃而過 0
一晃而過 0
一晃而過 0
一晃而過 0
一晚 201
一晚 201
一晝 1
一晝 1
一智 41
一智 41
一暴十寒 0
一暴十寒 0
一曝十寒 3
一曝十寒 3
一曲 184
一曲 184
一會 337
一會 337
一會兒 346
一會兒 346
一會兒 346
一會兒 346
一月 4204
一月份 263
一有 216
一有 216
一服 3
一服 3
一望無垠 8
一望無垠 8
一望無邊 1
一望無邊 1
一望無際 56
一望無際 56
一望而知 4
一望而知 4
一朝 51
一朝 51
一朝一夕 20
一朝一夕 20
一朝一夕 20
一朝一夕 20
一朝之忿 0
一朝之忿 0
一朝天子 2
一朝天子 2
一朝天子一朝臣 1
一朝天子一朝臣 1
一朝天子一朝臣 1
一朝天子一朝臣 1
一朝被蛇咬 0
一朝被蛇咬 0
一期 2502
一期 2502
一期稻作 5
一期稻作 5
一木難扶 0
一木難扶 0
一木難支 0
一木難支 0
一本 1838
一本 1838
一本初衷 13
一本初衷 13
一本書 2359
一本書 2359
一本正經 24
一本正經 24
一本萬利 4
一本萬利 4
一朵 210
一朵 210
一朵花 25
一朵花 25
一村 69
一村 69
一束 136
一束 136
一束花 19
一束花 19
一杯 606
一杯 606
一板 13
一板 13
一板一眼 21
一板一眼 21
一枚 204
一枚 204
一枝 122
一枝 122
一枝之棲 0
一枝之棲 0
一枝春 2
一枝春 2
一枝獨秀 47
一枝獨秀 47
一枝筆 5
一枝筆 5
一枝花 0
一枝花 0
一枝草 0
一枝草 0
一枝香 3
一枝香 3
一架 301
一架 301
一柱擎天 7
一柱擎天 7
一株 130
一株 130
一根 433
一根 433
一根根 19
一根根 19
一根煙 13
一根煙 13
一格 137
一格 137
一桌 69
一桌 69
一桶 49
一桶 49
一桶桶 1
一桶桶 1
一桿 73
一桿 73
一條 4618
一條 4618
一條條 40
一條條 40
一條鞭 0
一條鞭 0
一條鞭法 0
一條鞭法 0
一條龍 27
一條龍 27
一梯 20
一梯 20
一梯次 25
一梯次 25
一棒 44
一棒 44
一棟 413
一棟 413
一棟棟 18
一棟棟 18
一棵 1761
一棵 290
一概 142
一概 142
一概不理 0
一概不理 0
一概不管 0
一概不管 0
一概而論 19
一概而論 19
一槍 61
一槍 61
一槓 0
一槓 0
一樁 131
一樁 131
一樓 1619
一樓 1619
一模一樣 243
一模一樣 243
一模一樣 243
一模一樣 243
一樣 9025
一樣 9025
一樣可以 132
一樣可以 132
一樣在 110
一樣在 110
一樣是 178
一樣是 178
一樣會 63
一樣會 63
一樣有 102
一樣有 102
一樣能 36
一樣能 36
一樹百穫 0
一樹百穫 0
一檔 45
一檔 45
一櫃 4
一櫃 4
一次 13949
一次 13949
一次方程 0
一次方程 0
一次方程式 0
一次方程式 0
一次次 63
一次次 63
一正 33
一正 33
一正一負 1
一正一負 1
一正一負 1
一正一負 1
一步 2846
一步 2846
一步一腳印 0
一步一腳印 0
一步一腳印 0
一步一腳印 0
一步步 216
一步步 216
一步法 1
一步法 1
一步登天 13
一步登天 13
一歲 467
一歲 467
一死 144
一死 144
一死一傷 24
一死一傷 24
一死一傷 24
一死一傷 24
一死了之 2
一死了之 2
一段 8654
一段 8654
一段票 11
一段票 11
一段路 57
一段路 57
一比 129
一比 129
一毛 23
一毛 23
一毛不拔 14
一毛不拔 14
一毛錢 41
一毛錢 41
一毫 16
一毫 16
一民 46
一民 46
一氣 70
一氣 70
一氣之下 22
一氣之下 22
一氣呵成 49
一氣呵成 49
一氧 1
一氧 1
一氧化物 1
一氧化物 1
一氧化碳 158
一氧化碳 158
一氧化鉛 1
一氧化鉛 1
一池 23
一池 23
一池春水 7
一池春水 7
一池秋水 0
一池秋水 0
一決勝負 7
一決勝負 7
一決雌雄 3
一決雌雄 3
一決雌雄 3
一沐三握髮 0
一沐三握髮 0
一沖 7
一沖 7
一泡 16
一泡 16
一波 1078
一波 1078
一波 1078
一波 1078
一波三折 26
一波三折 26
一波三折 26
一波三折 26
一波又起 7
一波又起 7
一波又起 7
一波又起 7
一波未平 7
一波未平 7
一波未平 7
一波未平 7
一洗 24
一洗 24
一洗塵俗 0
一洗塵俗 0
一洞 21
一洞 21
一派 112
一派 112
一派胡言 13
一派胡言 13
一流 663
一流 663
一流人才 21
一流人才 21
一流學府 0
一流學府 0
一涼 5
一涼 5
一添 2
一添 2
一清 96
一清 96
一清二楚 45
一清二楚 45
一清二白 0
一清二白 0
一清如水 0
一清如水 0
一清專案 0
一清專案 0
一清早 1
一清早 1
一測 4
一測 4
一準 1
一準 1
一溜煙 25
一溜煙 25
一溜煙 25
一溜煙 25
一滑 3
一滑 3
一滴 135
一滴 135
一滴水 8
一滴水 8
一潑 2
一潑 2
一瀉千里 2
一瀉千里 2
一火 3
一火 3
一炮 18
一炮 18
一炮而紅 18
一炮而紅 18
一炷香 3
一炷香 3
一無 84
一無 84
一無可取 4
一無可取 4
一無所得 4
一無所得 4
一無所恃 0
一無所恃 0
一無所有 57
一無所有 57
一無所獲 7
一無所獲 7
一無所知 60
一無所知 60
一無所長 1
一無所長 1
一無是處 24
一無是處 24
一無長物 0
一無長物 0
一無長物 0
一無長物 0
一無長物 0
一無長物 0
一煮 8
一煮 8
一營 19
一營 19
一爭長短 0
一爭長短 0
一片 2881
一片 2881
一片丹心 0
一片丹心 0
一片冰心 0
一片冰心 0
一片成名 1
一片成名 1
一片汪洋 0
一片汪洋 0
一片漆黑 0
一片漆黑 0
一片漆黑 0
一片漆黑 0
一片片 37
一片片 37
一片狼籍 0
一片狼籍 0
一片狼藉 0
一片狼藉 0
一片痴心 0
一片痴心 0
一片苦心 0
一片苦心 0
一版 305
一版 305
一物 190
一物 190
一物剋一物 5
一物剋一物 5
一物剋一物 5
一物剋一物 5
一物降一物 0
一物降一物 0
一物降一物 0
一物降一物 0
一牽 4
一牽 4
一狀 15
一狀 15
一狐之腋 0
一狐之腋 0
一獎 21
一獎 21
一班 1013
一班 1013
一班人 15
一班人 15
一球 122
一球 122
一環 394
一環 394
一瓣 13
一瓣 13
一生 2171
一生 2171
一生一世 66
一生一世 66
一生一世 66
一生一世 66
一生受用不盡 0
一生受用不盡 0
一產 9
一產 9
一用 36
一用 36
一用到 0
一用到 0
一甲 229
一甲 229
一甲子 31
一甲子 31
一男 30
一男 30
一男一女 41
一男一女 41
一男一女 41
一男一女 41
一男二女 1
一男二女 1
一男半女 1
一男半女 1
一番 1106
一番 1106
一番苦心 3
一番苦心 3
一番話 0
一番話 0
一畫 23
一畫 23
一當 7
一當 7
一疊 53
一疊 53
一病 11
一病 11
一病不起 4
一病不起 4
一瘸一拐 4
一瘸一拐 4
一登龍門 0
一登龍門 0
一發 64
一發 64
一發不可 3
一發不可 3
一發不可收拾 53
一發不可收拾 53
一百 5367
一百 5367
一百人 54
一百人 54
一百個 103
一百個 103
一百元 128
一百元 128
一百塊 10
一百塊 10
一百萬 276
一百萬 276
一盒 90
一盒 90
一盒盒 1
一盒盒 1
一盞 111
一盞 111
一盞燈 34
一盞燈 34
一盞盞 9
一盞盞 9
一盤 98
一盤 98
一盤散沙 11
一盤散沙 11
一盤棋 9
一盤棋 9
一盤菜 5
一盤菜 5
一目了然 83
一目了然 83
一目十行 2
一目十行 2
一目瞭然 34
一目瞭然 34
一直 6917
一直 6917
一直在 597
一直在 597
一直是 794
一直是 794
一直都 231
一直都 231
一相情願 4
一相情願 4
一看 470
一看 470
一看之下 4
一看之下 4
一看再看 12
一看再看 12
一看出 0
一看出 0
一看到 125
一看到 125
一眨 7
一眨 7
一眨眼 21
一眨眼 21
一眼 565
一眼 565
一眼看穿 4
一眼看穿 4
一睹 115
一睹 115
一睹之下 0
一睹之下 0
一瞄 1
一瞄 1
一瞥 65
一瞥 65
一瞧 32
一瞧 32
一瞧之下 0
一瞧之下 0
一瞬 31
一瞬 31
一瞬間 177
一瞬間 177
一瞻丰采 0
一瞻丰采 0
一知半解 40
一知半解 40
一石二鳥 8
一石二鳥 8
一石兩鳥 4
一石兩鳥 4
一砍 4
一砍 4
一砲而紅 0
一砲而紅 0
一破 4
一破 4
一碗 161
一碗 161
一碟 53
一碟 53
一碼 90
一碼 90
一磅 19
一磅 19
一神教 2
一神教 2
一神論 12
一神論 12
一票 470
一票 470
一票人 28
一票人 28
一票人馬 0
一票人馬 0
一禁 1
一禁 1
一禮 10
一禮 10
一科 1761
一科 316
一秒 140
一秒 140
一秒鐘 77
一秒鐘 77
一程 30
一程 30
一種 11107
一種 11107
一窩 40
一窩 40
一窩蜂 73
一窩蜂 73
一窩風 1
一窩風 1
一窮二白 5
一窮二白 5
一窺究竟 0
一窺究竟 0
一竅 6
一竅 6
一竅不通 32
一竅不通 32
一站 235
一站 235
一站站 2
一站站 2
一章 618
一章 618
一端 266
一端 266
一竿 13
一竿 13
一竿子 18
一竿子 18
一笑 600
一笑 600
一笑置之 20
一笑置之 20
一筆 41749
一筆 41749
一筆劃 2
一筆劃 2
一筆勾銷 5
一筆勾銷 5
一筆抹煞 0
一筆抹煞 0
一等 342
一等 342
一等一 18
一等一 18
一等兵 0
一等兵 0
一等艙 0
一等艙 0
一等親 0
一等親 0
一筒 7
一筒 7
一算 57
一算 57
一算之下 0
一算之下 0
一箭之仇 0
一箭之仇 0
一箭雙雕 0
一箭雙雕 0
一箭雙鵰 9
一箭雙鵰 9
一箱 48
一箱 48
一節 504
一節 504
一節課 41
一節課 41
一篇 6525
一篇 6525
一簍 5
一簍 5
一簍簍 1
一簍簍 1
一簞食 2
一簞食 2
一簧兩舌 1
一簧兩舌 1
一簽 7
一簽 7
一籃 15
一籃 15
一籌 131
一籌 131
一籌莫展 23
一籌莫展 23
一籮筐 374
一籮筐 374
一粒 187
一粒 187
一粗一細 0
一粗一細 0
一粗一細 0
一粗一細 0
一系 47
一系 47
一系列 940
一系列 940
一約 7
一約 7
一紅 32
一紅 32
一紙 75
一紙 75
一紙空文 0
一紙空文 0
一級 1053
一級 1053
一組 1364
一組 1364
一絕 20
一絕 20
一統 86
一統 86
一統天下 15
一統天下 15
一絲 317
一絲 317
一絲一毫 12
一絲一毫 12
一絲一毫 12
一絲一毫 12
一絲不掛 25
一絲不掛 25
一絲不苟 18
一絲不苟 18
一經 224
一經 224
一網打盡 46
一網打盡 46
一線 451
一線 451
一線之隔 0
一線之隔 0
一線天 0
一線天 0
一線希望 0
一線希望 0
一線生機 28
一線生機 28
一縮 14
一縮 14
一縷 63
一縷 63
一總 41
一總 41
一缸 23
一缸 23
一缸水 1
一缸水 1
一罐 67
一罐 67
一群 1904
一群 1904
一群人 218
一群人 218
一群群 18
一群群 18
一義 15
一義 15
一翻 48
一翻 48
一翻兩瞪眼 1
一翻兩瞪眼 1
一翻出 0
一翻出 0
一翻到 0
一翻到 0
一老 20
一老 20
一者 620
一者 620
一而再 36
一而再 36
一聯 45
一聯 45
一聲 1384
一聲 1384
一聲不吭 3
一聲不吭 3
一聲不響 17
一聲不響 17
一聲令下 43
一聲令下 43
一聲聲 20
一聲聲 20
一聽 174
一聽 174
一聽 174
一聽 174
一聽之下 5
一聽之下 5
一肚子 19
一肚子 19
一肚子氣 7
一肚子氣 7
一肚子火 5
一肚子火 5
一股 1279
一股 1279
一股勁兒 4
一股勁兒 4
一股勁兒 4
一股勁兒 4
一股腦兒 18
一股腦兒 18
一股腦兒 18
一股腦兒 18
一胎 62
一胎 62
一胎化 11
一胎化 11
一脈相傳 5
一脈相傳 5
一脈相傳 5
一脈相傳 5
一脈相承 10
一脈相承 10
一脈相承 10
一脈相承 10
一腔 9
一腔 9
一腔熱血 2
一腔熱血 2
一腔熱血 2
一腔熱血 2
一腔熱誠 0
一腔熱誠 0
一腳 303
一腳 303
一腳踢開 0
一腳踢開 0
一腿 26
一腿 26
一臂 13
一臂 13
一臂 13
一臂 13
一臂之力 68
一臂之力 68
一臂之力 68
一臂之力 68
一臉 206
一臉 206
一臉茫然 19
一臉茫然 19
一臉茫然 19
一臉茫然 19
一致 1505
一致 1505
一致性 252
一致性 252
一致百慮 0
一致百慮 0
一臺 46
一臺 46
一舉 331
一舉 331
一舉一動 98
一舉一動 98
一舉一動 98
一舉兩得 35
一舉兩得 35
一舉成名 13
一舉成名 13
一舉成擒 0
一舉成擒 0
一舉數得 0
一舉數得 0
一般 23574
一般 23574
一般人 1411
一般人 1411
一般人員 0
一般人員 0
一般來說 338
一般來說 338
一般化 34
一般化 34
一般官員 0
一般官員 0
一般常識 0
一般常識 0
一般性 462
一般性 462
一般情形 0
一般情形 0
一般狀況 0
一般狀況 0
一般等價 0
一般等價 0
一般等價物 0
一般等價物 0
一般而言 651
一般而言 651
一般見識 6
一般見識 6
一般說來 101
一般說來 101
一船 19
一船 19
一艘 302
一艘 302
一艘船 24
一艘船 24
一色 63
一色 63
一草 11
一草 11
一草一木 27
一草一木 27
一草一木 27
一草一木 27
一萬 1161
一萬 1161
一萬人 61
一萬人 61
一萬個 158
一萬個 158
一萬元 508
一萬元 508
一萬塊 6
一萬塊 6
一落千丈 11
一落千丈 11
一葉 37
一葉 37
一葉扁舟 0
一葉扁舟 0
一葉知秋 11
一葉知秋 11
一葦渡江 0
一葦渡江 0
一蓋 9
一蓋 9
一處 817
一處 817
一號 2171
一號 2171
一行 836
一行 836
一行三昧 1
一行人 180
一行人 180
一行行 14
一行行 14
一街 635
一街 635
一衝 4
一衝 4
一表 24
一表 24
一表人才 0
一表人才 0
一表人材 0
一表人材 0
一袋 42
一袋 42
一被 58
一被 58
一裝 10
一裝 10
一要 33
一要 33
一要到 0
一要到 0
一見 316
一見 316
一見傾心 2
一見傾心 2
一見如故 14
一見如故 14
一見鍾情 58
一見鍾情 58
一見面 18
一見面 18
一視同仁 60
一視同仁 60
一親芳澤 15
一親芳澤 15
一覺 93
一覺 93
一覺睡到天亮 0
一覺睡到天亮 0
一覽 925
一覽 925
一覽無遺 56
一覽無遺 56
一覽無餘 0
一覽無餘 0
一覽表 3771
一覽表 3771
一角 268
一角 268
一角錢 1
一角錢 1
一觸即爆 0
一觸即爆 0
一觸即發 78
一觸即發 78
一言 108
一言 108
一言一行 25
一言一行 25
一言一行 25
一言一行 25
一言不合 6
一言不合 6
一言不和 2
一言不和 2
一言不發 0
一言不發 0
一言中的 0
一言中的 0
一言九鼎 7
一言九鼎 7
一言以蔽之 18
一言半語 0
一言半語 0
一言喪邦 0
一言喪邦 0
一言堂 37
一言堂 37
一言既出 0
一言既出 0
一言為定 15
一言為定 15
一言興邦 0
一言興邦 0
一言難盡 32
一言難盡 32
一訂 4
一訂 4
一記 103
一記 103
一試 1499
一試 753
一試之下 5
一試之下 5
一語 101
一語 101
一語不發 17
一語不發 17
一語中的 0
一語中的 0
一語成讖 6
一語成讖 6
一語破的 0
一語破的 0
一語道破 13
一語道破 13
一語雙關 4
一語雙關 4
一語驚醒 3
一語驚醒 3
一誤再誤 0
一誤再誤 0
一說出 4
一說出 4
一說到 9
一說到 9
一說起 3
一說起 3
一課 120
一課 120
一調 12
一調 12
一調之下 0
一調之下 0
一諾值千金 0
一諾值千金 0
一諾千金 13
一諾千金 13
一講 51
一講 51
一講出 0
一講出 0
一講到 3
一講到 3
一讀 147
一讀 147
一讀再讀 7
一讀再讀 7
一變 57
一變 57
一貧如洗 2
一貧如洗 2
一貫 787
一貫 787
一貫下來 0
一貫下來 0
一貫作業 120
一貫作業 120
一貫性 42
一貫性 42
一貫道 101
一貫道 101
一買 15
一買 15
一買一賣 1
一買一賣 1
一買一賣 1
一買一賣 1
一賣 5
一賣 5
一贏 6
一贏 6
一走 61
一走 61
一走了之 20
一走了之 20
一起 6289
一起 6289
一起來 963
一起來 963
一起兒 0
一起兒 0
一起兒 0
一起兒 0
一起到 76
一起到 76
一起去 476
一起去 476
一趟 476
一趟 476
一足 15
一足 15
一族 30
一族 30
一跌再跌 0
一跌再跌 0
一跟 23
一跟 23
一路 4301
一路 4301
一路上 204
一路上 204
一路平安 0
一路平安 0
一路福星 0
一路福星 0
一路重挫 0
一路重挫 0
一路順風 0
一路順風 25
一跳 81
一跳 81
一踢 15
一踢 15
一蹴可及 10
一蹴可及 10
一蹴可幾 0
一蹴可幾 0
一蹴可成 4
一蹴可成 4
一蹴而及 0
一蹴而及 0
一蹶不振 35
一蹶不振 35
一躍 83
一躍 83
一躍龍門 1
一躍龍門 1
一身 576
一身 576
一身兩役 0
一身兩役 0
一身是汗 0
一身是汗 0
一身是膽 1
一身是膽 1
一躲 1
一躲 1
一車 62
一車 62
一車子 4
一車子 4
一軍 51
一軍 51
一較高下 18
一較高下 18
一較高下 18
一較高下 18
一輛 417
一輛 417
一輛車 31
一輛車 31
一輩 84
一輩 84
一輩兒 0
一輩兒 0
一輩兒 0
一輩兒 0
一輩子 595
一輩子 595
一輪 151
一輪 151
一轉 123
一轉 123
一轉眼 29
一轉眼 29
一辦 13
一辦 13
一近 10
一近 10
一途 89
一途 89
一逕 21
一逕 21
一通 311
一通 311
一通通 0
一通通 0
一逞獸慾 0
一逞獸慾 0
一連 201
一連 201
一連串 816
一連串 816
一週 1795
一週 1795
一週年 63
一週年 63
一進 146
一進 146
一進來 19
一進來 19
一進到 16
一進到 16
一進去 14
一進去 14
一遍 373
一遍 373
一過 61
一過 61
一過來 1
一過來 1
一過去 5
一過去 5
一道 399
一道 791
一道菜 0
一道菜 0
一遭 102
一遭 102
一遮 3
一遮 3
一遮之下 0
一遮之下 0
一邊 1401
一邊 1401
一邊倒 2
一邊倒 2
一部 2846
一部 2846
一部分 470
一部分 470
一鄰 46
一鄰 46
一醉解千愁 0
一醉解千愁 0
一里 14
一里 14
一重 67
一重 67
一量 12
一量 12
一針 41
一針 41
一針見血 43
一針見血 43
一錢 37
一錢 37
一錢不值 0
一錢不值 0
一錯 1
一錯 1
一錯再錯 7
一錯再錯 7
一鍋 85
一鍋 85
一鎊 3
一鎊 3
一鎖 1
一鎖 1
一鏟 1
一鏟 1
一長 73
一長 73
一長一短 3
一長一短 3
一長一短 3
一長一短 3
一門 422
一門 422
一門忠烈 1
一門忠烈 1
一閃 175
一閃 175
一閃即逝 10
一閃即逝 10
一閃而過 7
一閃而過 7
一開 87
一開 87
一開出 1
一開出 1
一開出來 0
一開出來 0
一開到 0
一開到 0
一間 958
一間 958
一關 135
一關 135
一闡提 0
一闡提 0
一防 0
一防 0
一陣 1103
一陣 1103
一陣子 716
一陣子 716
一陣陣 75
一陣陣 75
一陣雨 5
一陣雨 5
一陣風 38
一陣風 38
一隅 110
一隅 110
一隊 172
一隊 172
一階 66
一階 66
一隔 5
一隔 5
一隻 2874
一隻 2874
一集 422
一集 422
一雙 445
一雙 445
一雙雙 6
一雙雙 6
一雨成災 0
一雨成災 0
一雨成秋 0
一雨成秋 0
一霎 4
一霎 4
一霎時 5
一霎時 5
一霎眼 1
一霎眼 1
一青 7
一青 7
一面 1547
一面 1547
一面之交 1
一面之交 1
一面之緣 9
一面之緣 9
一面之詞 4
一面之詞 4
一面之辭 0
一面之辭 0
一面之雅 0
一面之雅 0
一面倒 17
一面倒 17
一鞭 7
一鞭 7
一頁 44283
一頁 44283
一頂 78
一頂 78
一項 23158
一項 23158
一頓 295
一頓 295
一頓飯 0
一頓飯 0
一頭 315
一頭 315
一頭撞死 0
一頭撞死 0
一頭熱 9
一頭熱 9
一頭霧水 46
一頭霧水 46
一顆 1761
一顆 1761
一題 193
一題 193
一類 1398
一類 1398
一顰一笑 18
一顰一笑 18
一顰一笑 18
一飛沖天 0
一飛沖天 0
一飯一啄 0
一飯一啄 0
一飯一啄 0
一飯一啄 0
一飯千金 0
一飯千金 0
一飲一啄 0
一飲一啄 0
一飲一啄 0
一飲一啄 0
一飲而盡 22
一飲而盡 22
一飽 20
一飽 20
一飽口福 0
一飽口福 0
一飽眼福 0
一飽眼福 0
一飽耳福 0
一飽耳福 0
一養 4
一養 4
一餐 105
一餐 105
一餐飯 7
一餐飯 7
一首 921
一首 921
一首歌 205
一首歌 205
一馬 21
一馬 21
一馬當先 32
一馬當先 32
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一骨碌 1
一體 545
一體 545
一體成形 0
一體成形 0
一髮千鈞 0
一髮千鈞 0
一鬨而散 0
一鬨而散 0
一鱗半爪 1
一鱗半爪 1
一鳴驚人 25
一鳴驚人 25
一麾出守 0
一點 6680
一點 6680
一點一滴 120
一點一滴 120
一點一滴 120
一點也 274
一點也 274
一點也不 205
一點也不 205
一點也沒 16
一點也沒 16
一點兒 67
一點兒 67
一點兒 67
一點兒 67
一點鐘 11
一點鐘 11
一黨 65
一黨 65
一黨制 1
一黨制 1
一黨獨大 51
一黨獨大 51
一鼓作氣 13
一鼓作氣 13
一鼻子 0
一鼻子 0
一鼻子 0
一鼻子 0
一鼻子灰 2
一鼻子灰 2
一鼻子灰 2
一鼻子灰 2
一鼻孔 0
一鼻孔 0
一鼻孔出氣 2
一鼻孔出氣 2
一齊 316
一齊 316
一齣 230
一齣 230
一齣戲 42
一齣戲 42
丁 3
丁 3538
丁丁 86
丁丁 86
丁丁冬冬 0
丁丁當當 0
丁上 2
丁下 0
丁二烯 52
丁基橡膠 6
丁壯 0
丁字 22
丁字型 0
丁字尺 2
丁字形 5
丁字褲 27
丁字鎬 0
丁憂 11
丁是丁卯是卯 0
丁母憂 0
丁烯 28
丁烷 30
丁班 111
丁男 7
丁當 1
丁種 17
丁種維生素 0
丁等 78
丁級 5
丁肇中 0
丁艱 0
丁苯橡膠 1
丁酸 5
丁醇 36
丁鈴噹啷 0
丁項 0
丁香 48
丁香油 1
丁香花 1
丂 0
七 20690
七一 436
七七 447
七七事變 12
七上八下 12
七下 6
七世 16
七世夫妻 0
七人 326
七仙女 1
七代 101
七位 263
七位數 95
七俠五義 15
七個 540
七個人 22
七倍 41
七億 293
七億人 2
七元 89
七公 96
七出 1
七分 156
七分鐘 25
七劃 26
七十 7501
七十二變 11
七千 777
七千人 16
七千元 34
七古 0
七名 142
七員 9
七哥 0
七喜 41
七嘴八舌 191
七回 60
七圈 0
七國 56
七團 21
七堵 117
七夕 191
七天 464
七套 3
七孔流血 0
七孔生煙 0
七寶 1
七尺 22
七尺之驅 0
七層 48
七層樓 11
七巧板 14
七巷 77
七平八穩 0
七年 11800
七年之癢 5
七年級 38
七度 52
七座 32
七弟 4
七弦琴 2
七彩 335
七彩繽紛 20
七律 6
七情 15
七情六慾 20
七情六欲 0
七成 447
七手八腳 11
七扭八歪 0
七折 73
七折八扣 3
七拐八拐 0
七拳 0
七拼八湊 7
七排 0
七擒七縱 0
七支 21
七日 2341
七早八早 7
七星 297
七星壇 0
七星山 53
七星岩 14
七星巖 1
七星板 0
七星燈 0
七星瓢蟲 0
七景 0
七月 6202
七月份 545
七月革命 0
七杯 0
七枝 1
七格 1
七桶 0
七梯次 1
七樓 317
七橫八豎 0
七次 174
七步 13
七步之才 0
七步奇才 0
七步成詩 0
七步詩 0
七派 1
七滅諍法 1
七滴 0
七爺 8
七爺八爺 12
七片 10
七百 686
七百人 12
七百元 19
七盒 0
七盤 1
七盤菜 0
七票 15
七秒 10
七秒鐘 0
七種 135
七竅 9
七竅冒煙 0
七竅生煙 4
七站 2
七筆 8
七節 54
七節課 0
七篇 61
七級 75
七級風 3
七絃琴 0
七絕 7
七線 16
七美 234
七老八十 6
七聲 2
七色 42
七色板 0
七萬 329
七葉膽 1
七葷八素 0
七藝 0
七號 682
七行 12
七行詩 0
七袋 2
七覺支 1
七言 14
七言律詩 4
七言絕句 4
七言詩 3
七路 116
七輛 9
七輛車 0
七通 2
七部 46
七鄰 24
七里香 13
七重 65
七重唱 0
七重奏 0
七除八扣 0
七隻 26
七雄 6
七零八落 14
七音 3
七題 5
七顛八倒 0
七點 475
七點時 1
七點鐘 10
丄 0
丄 0
丅 0
万 423
万 423
丈 489
丈二和尚 1
丈二金剛 7
丈人 21
丈六金身 1
丈夫 788
丈母 0
丈母娘 10
丈量 40
丈量法 0
三 1589
三 71
三 71494
三一學院 3
三七 388
三七五減租 35
三七仔 4
三三兩兩 9
三上 96
三下 111
三不五時 0
三不朽 1
三不管 9
三世 200
三世紀 22
三丙 41
三乙 448
三九 341
三五好友 0
三五成群 14
三人 1716
三人份 2
三人成虎 0
三人行 94
三仁 32
三代 422
三代同堂 38
三令五申 9
三任 80
三份 166
三伍 1
三伏 5
三伯 4
三伯母 0
三伯父 0
三位 1155
三位一體 38
三位一體 38
三位數 114
三保太監 0
三俠五義 5
三信 136
三個 3765
三個世界 1
三個人 238
三個和尚沒水喝 1
三個月 1941
三個臭皮匠 4
三倍 226
三催四請 0
三億 375
三億人 1
三億元 164
三元 343
三元裡 0
三元里 2
三光 306
三克 3
三八 418
三八婦女節 6
三公 45
三具 13
三分 1
三分之一 579
三分之二 455
三分法 1
三分球 1
三分線 1
三分鐘 224
三分頭 1
三劍客 0
三十 7295
三十七道品 1
三十二相 1
三十六計 42
三十年代 42
三十而立 19
三千 1226
三千人 48
三千個 17
三千元 290
三千塊 4
三原 18
三原色 0
三叉戟 1
三叉骨 0
三句 39
三句不離 2
三句話 22
三句話不離本行 1
三台 218
三台聯播 0
三合 135
三合一 1259
三合土 7
三合會 3
三合板 6
三吋 14
三吋半 5
三同 7
三向 18
三吳 8
三周 38
三周年 20
三和弦 6
三和路 189
三品 13
三員 5
三哥 52
三商 258
三商虎 95
三商行 54
三商銀 301
三善根 0
三國 637
三國志 184
三國時代 31
三國演義 151
三國誌 30
三國鼎立 18
三圓 6
三堆 0
三場 25
三場 250
三塊 52
三塊錢 6
三壇大戒 1
三壘手 27
三多 369
三夜 185
三大 1600
三大塊 4
三大差別 0
三大政策 0
三大洋 4
三大片 1
三大革命 0
三大革命運動 0
三大類 115
三天 1617
三天兩頭 31
三天打魚 2
三夾板 16
三好 52
三妻四妾 0
三姑 11
三姑六婆 23
三姓 10
三字 269
三字經 70
三孝 11
三季 1288
三學 1
三定 4
三客 0
三害 3
三寶 290
三寶弟子 11
三寸 13
三寸丁 2
三寸不爛 0
三寸不爛之舌 17
三寸舌 1
三寸金蓮 9
三專 55
三專制 1
三專生 0
三對 72
三小 36
三小塊 0
三小時 187
三小片 0
三尺 73
三局 137
三屆 1574
三層 307
三層樓 33
三山五嶽 2
三岔口 0
三岔路 0
三岔路口 2
三峽 1127
三巷 97
三師 6
三年 7786
三年五載 3
三年制 71
三年有成 10
三年生 21
三年級 1226
三度 261
三度空間 201
三座大山 0
三廢 4
三式 13
三弦 14
三弦樂器 0
三弦琴 2
三張 296
三張犁 3
三從 4
三從四德 3
三復 4
三心二意 8
三心兩意 8
三思 94
三思 94
三思而後 3
三思而後 3
三思而後行 43
三思而後行 43
三思而行 7
三思而行 7
三態 23
三態變化 4
三成 642
三成新 0
三截 9
三戰 45
三戰兩勝 3
三戶 25
三戶亡秦 0
三房 95
三房一廳 13
三房一廳 13
三房二廳 32
三才 34
三打 22
三打 22
三拍 18
三振 160
三振出局 11
三掌 14
三排 13
三摩地 1
三支 104
三教九流 7
三斑家蚊 6
三方 144
三方面 176
三族 17
三日 2131
三日份 1
三日打魚 0
三明治 153
三星 828
三星期 48
三星級 15
三春柳 1
三昧 1
三昧真火 1
三昧耶 1
三昧耶戒 1
三昧耶曼陀羅 1
三時 247
三晉 6
三晚 37
三智 17
三更 57
三更 57
三更半夜 39
三更半夜 39
三月 5013
三月份 352
三朝 16
三朝 16
三朝元老 5
三杯 58
三杯雞 1
三板 9
三枝 12
三格 17
三棱 10
三棱鏡 0
三棵 19
三極 17
三極真空 0
三極管 1
三槓 1
三樁 4
三樓 982
三樓住戶 0
三樣 45
三權分立 44
三次 1057
三正 14
三步 109
三歲 425
三歲時 22
三段 2928
三段制 1
三段式 72
三段票 1
三段論法 2
三民 2035
三民主義 422
三民路 838
三氯化鉍 0
三氯化鉍 0
三氯甲烷 39
三氯甲烷 39
三水 29
三法 71
三法印 1
三流 24
三淨肉 1
三溫暖 257
三焦 0
三爺 18
三版 198
三牲 8
三班 279
三班制 12
三瓣 0
三生 37
三生有幸 10
三申五令 0
三番五次 3
三番兩次 11
三百 1827
三百人 47
三百個 25
三百元 64
三百六十 146
三百六十度 17
三百六十行 5
三百塊 5
三百篇 1
三皇 31
三皇五帝 0
三盒 2
三盞 1
三相 149
三眼 34
三碗 12
三禮 6
三秋 5
三科 122
三秒 94
三秒鐘 19
三稜錐 0
三稜鏡 12
三站 25
三章 277
三等 331
三等艙 4
三等親 15
三箱 7
三節 257
三節課 4
三級 920
三級跳 21
三級跳遠 3
三結合 1
三統 16
三維縣 0
三綱 3
三綱五常 1
三緘其口 29
三總 104
三缸 3
三缺一 12
三義 587
三老 15
三老四嚴 0
三聯 67
三聯劇 49
三聯單 14
三聯式 49
三聲 96
三聲無奈 26
三股 90
三胎 1
三胞胎 16
三腳 68
三腳架 25
三腳貓 12
三自性 1
三色 167
三色筆 0
三芝 445
三苦 1
三菱 638
三萬 510
三萬人 26
三萬個 7
三萬元 235
三萬塊 1
三葉期 0
三葉蟲 0
三藏 56
三藏 56
三藐三菩提 1
三藩市 9
三號 1011
三行 36
三行 36
三街 311
三袋 3
三複 0
三角 833
三角函數 30
三角學 2
三角帆 5
三角形 267
三角戀愛 1
三角方程 0
三角旗 1
三角板 9
三角架 17
三角柱 0
三角洲 67
三角測量 17
三角瓶 1
三角皮帶 3
三角肌 10
三角褲 17
三角錐 0
三角錢 1
三角鐵 12
三言兩語 30
三課 34
三請四喚 0
三諦 1
三讀 88
三讀會 7
三讀通過 144
三足鼎立 17
三跪九叩 0
三軍 157
三軍總醫院 241
三輕 19
三輪 47
三輪兒 0
三輪兒 0
三輪式 2
三輪車 142
三通 378
三通閥 1
三連 102
三週 336
三週年 530
三達德 1
三部 245
三部合唱 1
三部曲 135
三重 1045
三重人 1
三重人格 2
三重唱 7
三重奏 129
三重市 2580
三重秘傳 1
三針 5
三錢 21
三鍋 3
三長兩短 10
三長制 0
三門 171
三門峽 4
三閭大夫 0
三關 19
三陣 1
三陽 407
三陽開泰 2
三隊 42
三階 21
三障 1
三隻 213
三隻手 13
三集 212
三青團 1
三面 145
三面亞當 0
三面夏娃 0
三面紅旗 0
三面體 0
三頁 381
三項 1388
三項式 3
三頭 46
三頭六臂 6
三頭六臂 6
三顆 149
三顧茅廬 11
三顧茅蘆 1
三風十愆 0
三餐 179
三餐不繼 4
三魂七魄 0
三鮮 9
三麥 0
三點 690
三點式 28
三點水 2
三點鐘 18
三圍 56
上 146
上 168872
上一代 78
上一代 78
上上 86
上上下下 57
上上之策 3
上上個月 100
上上星期 7
上下 1391
上下其手 7
上下文 17
上下級 20
上下顛倒 5
上不來 7
上不去 19
上不起 1
上世 3
上中下游 0
上中農 0
上乘 80
上乘 80
上了 672
上交 16
上京 19
上人 504
上代 10
上任 335
上位 75
上來 967
上來到 0
上供 59
上傳 98
上元 23
上元節 1
上光 49
上兵 10
上冊 142
上凍 1
上刑 5
上列 692
上到 110
上前 137
上前來 1
上前去 10
上匾 0
上區 6
上千 182
上千人 14
上升 1186
上午 7856
上半 199
上半場 10
上半場 109
上半夜 1
上半天 4
上半年 1173
上半身 53
上半部 34
上原 47
上去 513
上口 13
上古 76
上古史 33
上古時代 10
上台 316
上台演唱 2
上台演奏 0
上台演講 2
上司 153
上吊 1000
上吊自殺 9
上吐下瀉 20
上告 5
上周 167
上品 89
上回 195
上圈 6
上國 119
上坡 66
上坡路 5
上坡道 0
上堂 10
上報 41
上場 31
上場 318
上場比賽 14
上場比賽 14
上墳 1
上壘 49
上士 7
上夜 12
上大學 66
上天 242
上天堂 38
上天無路 0
上奏 9
上套兒 0
上套兒 0
上好 105
上好的 16
上學 434
上學時 7
上學校 34
上官 186
上官婉兒 0
上客 8
上將 299
上將軍官 0
上尉 63
上尉軍官 0
上屆 111
上屈 6
上層 1360
上層建築 4
上層社會 10
上層組織 1
上山 339
上山下海 44
上山下鄉 6
上岸 153
上岸時 3
上工 25
上市 7496
上帝 3442
上帝教 4
上師 0
上年 880
上年紀 3
上床 93
上床睡覺 25
上座 26
上弔 2
上弦 11
上弦月 18
上得 101
上得 101
上得起 0
上得起 0
上心 7
上思 5
上情下達 0
上意識 0
上戰 5
上戰場 16
上戰場 16
上房 14
上手 198
上排 28
上接 26
上揚 498
上操 2
上文 68
上方 738
上旋 10
上日 26
上旬 282
上昇 235
上明 26
上映 652
上映期間 0
上智 75
上書 39
上書房 0
上有 2331
上有天堂 3
上朝 19
上杭 0
上林 65
上架 239
上校 179
上校軍官 0
上桌 67
上梁不正 0
上榜 130
上榜名單 3
上樑 64
上樑不正 0
上樑不正下樑歪 1
上樓 85
上樓來 3
上樓去 15
上樹 23
上機 458
上機時間 6
上檔 122
上櫃 0
上次 1518
上款 6
上段 25
上氣不接下氣 14
上水 24
上水船 1
上沃爾特 0
上油 22
上流 58
上流人士 2
上流社會 21
上海 2521
上海人 22
上海市 272
上海港 1
上海灘 10
上海話 8
上清 20
上游 645
上游工業 1
上溯 24
上漆 30
上演 389
上漲 1038
上漿 61
上潮 1
上火 18
上燈 7
上片 20
上片時間 0
上牌 2
上班 1521
上班族 879
上班時間 421
上班服 5
上略 18
上當 151
上疏 0
上癮 75
上發條 11
上百 299
上百人 17
上皮 124
上皮細胞 90
上皮組織 11
上盤 8
上相 42
上真 34
上眼 10
上眼皮 11
上知 17
上知 17
上石 9
上禮 23
上移 0
上空 246
上空裝 0
上站 533
上章 1
上端 79
上第 153
上等 137
上等兵 0
上等貨 0
上策 57
上算 18
上簇 1
上籃 12
上籃得分 0
上籤 3
上級 658
上網 0
上緊 24
上線 1985
上緣 47
上繳 16
上繳利潤 0
上聯 16
上聲 20
上肉 11
上肢 69
上肢骨 0
上肥 0
上脘 0
上膛 7
上臂 29
上臂 29
上臂骨 0
上臂骨 0
上自 47
上至 88
上臺 22
上舉 18
上舖 7
上船 92
上色 89
上菜 58
上萬 109
上葉 7
上著 22
上蒼 51
上藥 14
上藥膏 0
上蠟 14
上行 111
上行下效 8
上行列車 0
上街 90
上衣 172
上裝 38
上言 15
上計 8
上訪 4
上訴 355
上訴人 651
上課 3421
上課時 155
上課時數 125
上課證 26
上諭 0
上議院 15
上變 6
上賓 24
上起 22
上起來 3
上路 546
上身 173
上車 172
上車時 5
上軌道 47
上輩子 25
上轎 8
上農 4
上述 5259
上週 605
上進 91
上進心 26
上過 127
上道 20
上達 34
上達天聽 0
上達天聽 0
上選 171
上邊 24
上邦 1
上部 108
上酒家 12
上釉 15
上金 20
上鉤 24
上銀幕 11
上鋪 18
上錯 41
上錯天堂投錯胎 1
上鎖 50
上鏡頭 11
上門 122
上門來 12
上門去 3
上關 0
上限 538
上陣 81
上集 67
上面 2274
上頁 9770
上頜 12
上頭 199
上頭 199
上額 3
上顎 54
上顎骨 2
上風 63
上首 14
上香 39
上馬 53
上駟 3
上體 10
上點 94
上黨 2
下 111802
下一代 631
下一代 631
下一個 2217
下一個 2217
下一步 301
下一步 301
下下 34
下下之策 2
下下個月 3
下下星期 4
下不了台 7
下不了臺 0
下不來 5
下不去 2
下不為例 5
下丘腦 0
下中農 0
下乘 6
下乘 6
下了 434
下人 33
下代 9
下令 349
下任 54
下位 25
下作 42
下來 1940
下來到 6
下傳 10
下傾 4
下元 2
下光 5
下冊 139
下凡 32
下出 11
下列 8483
下到 95
下功夫 41
下區 1
下午 10325
下午茶 226
下半 181
下半場 10
下半場 103
下半夜 2
下半年 603
下半旗 0
下半月 12
下半身 103
下去 1172
下口 1
下句 0
下句話 0
下召 3
下台 441
下台一鞠躬 14
下台一鞠躬 14
下同 57
下周 135
下品 10
下唇 28
下問 7
下喙 0
下嚥 22
下回 168
下圖 438
下土 6
下在 74
下地 24
下地無門 0
下坡 113
下坡路 2
下垂 129
下堂 12
下場 20
下場 209
下場比賽 1
下場比賽 1
下墜 22
下墜球 0
下壓 46
下士 200
下女 8
下好 24
下嫁 7
下子 19
下子 19
下學 242
下學時 3
下完 20
下官 8
下定 20
下定決心 128
下定義 17
下屆 248
下層 226
下層社會 9
下層組織 0
下屬 80
下山 218
下崗 21
下工 4
下工夫 8
下巴 173
下巴頰 1
下巴顆 0
下市 49
下帖 1
下帖子 0
下年 10
下年度 97
下床 39
下座 7
下廚 38
下廚房 1
下廠 2
下弦 9
下弦月 4
下得 50
下情 6
下情上達 2
下意識 46
下愚 2
下懷 2
下成 23
下房 1
下手 219
下拜 10
下挫 235
下排 22
下接 15
下擺 11
下放 57
下文 119
下斷語 5
下方 779
下旬 509
下有 243
下有蘇杭 3
下朝 3
下朝 3
下木 1
下柬 0
下條子 0
下棋 108
下榻 39
下樓 144
下樓來 4
下樓去 13
下標 350
下樹 6
下檔 127
下次 1941
下款 3
下段 29
下殺 0
下毒 44
下毒手 60
下氣 12
下水 269
下水典禮 16
下水禮 1
下水船 0
下水道 482
下江 3
下沉 56
下河 6
下法 18
下注 43
下洩 0
下流 97
下流人士 0
下流社會 2
下海 111
下港 8
下游 893
下游工業 2
下滑 506
下潮 0
下瀉 7
下為 519
下為 519
下片 60
下片時 6
下牙 1
下獄 13
下班 477
下班時間 64
下生 19
下田 30
下界 9
下略 7
下疳 7
下痢 148
下發 37
下的 2503
下盤 19
下直 2
下相 5
下眼皮 2
下禮 0
下移 0
下種 7
下站 12
下章 1
下端 95
下第 79
下筆 52
下筆成章 0
下等 26
下等貨 0
下策 19
下箸 2
下籤 0
下級 107
下級機關 10
下級組織 2
下結論 7
下網 114
下線 187
下緣 37
下聘 5
下聯 32
下肉 5
下肚 26
下肢 176
下肢骨 0
下脣 0
下腳 27
下腳料 6
下腹 39
下膊 0
下臂 0
下臂 0
下自 20
下至 106
下臺 17
下舖 6
下船 64
下菜 0
下落 134
下落不明 44
下葉 5
下著 71
下著 71
下著雨 25
下著雪 2
下葬 10
下藥 27
下處 16
下處 16
下蛋 26
下蠱 0
下行 84
下行列車 0
下表 436
下襬 8
下視丘 0
下詔 7
下課 190
下課後 48
下課時 20
下課時間 27
下議院 14
下賤 26
下走 28
下起 71
下起來 1
下跌 1111
下跪 97
下身 27
下車 506
下車時 11
下載 100
下載 73
下輩 4
下輩子 34
下轎 3
下述 246
下逐客令 2
下週 461
下過 48
下達 158
下邊 24
下部 94
下鄉 123
下酒 23
下野 10
下鍋 28
下門 1
下關 8
下降 1494
下限 308
下院 15
下陷 99
下集 56
下雨 418
下雨天 86
下雪 91
下霜 0
下面 2362
下頁 841
下頜 15
下頭 22
下頷 12
下額 1
下顎 100
下顎骨 13
下風 31
下飯 14
下餘 1
下首 7
下馬 71
下馬威 7
下駟 0
下體 50
下麵 2361
丌 15
不 8
不 8
不 8
不 8
不 82901
不 82901
不瞞 100
不為人知 100
不一 913
不一定 520
不一定 520
不一會兒 37
不一會兒 37
不一而足 31
不三不四 6
不上 457
不上不下 2
不上來 23
不上去 38
不上道 2
不下 288
不下來 44
不下去 80
不下於 41
不世之材 0
不世出 5
不中 37
不中不西 0
不中意 1
不中用 10
不丹 23
不久 1358
不久就 79
不乏 245
不乏其人 9
不也 128
不也是 103
不了 939
不了 939
不了了之 27
不了情 43
不予 874
不予置評 23
不二 139
不二價 18
不二法門 120
不亞於 89
不亢不卑 1
不亦 31
不亦樂乎 59
不亮也光 4
不人道 16
不仁 12
不介意 65
不以 352
不以為忤 0
不以為然 126
不以為苦 0
不休 171
不似 132
不但 4155
不但是 272
不低 77
不住 402
不佔 131
不作 207
不作第二 1
不作第二人想 2
不作美 4
不作聲 4
不佳 1284
不來 339
不來梅 6
不來電 8
不依 161
不便 950
不俗 45
不信 501
不信任 109
不信任投票 9
不信邪 25
不修 45
不修邊幅 9
不倒 48
不倒翁 21
不借 10
不倫不類 14
不值 75
不值一提 5
不值一提 5
不值一文 1
不值一文 1
不值一看 1
不值一看 1
不值一笑 0
不值一笑 0
不值一顧 2
不值一顧 2
不值得 143
不值得 143
不假外出 4
不假外求 0
不假思索 19
不偏 23
不偏不倚 21
不做 380
不做聲 2
不停 386
不停地 179
不停的 273
不備 10
不傷 79
不傷和氣 2
不僅 3683
不僅僅 50
不僅僅是 59
不僅是 577
不像 1017
不像樣 1
不像話 12
不先 63
不先來 0
不先去 0
不光 19
不光在 2
不光是 24
不光會 2
不克 82
不克 82
不免 386
不免有些 9
不入 65
不入流 12
不入虎穴 8
不全 387
不全是 8
不全會 1
不兩立 7
不公 239
不公不義 88
不共 1
不共 1
不共天地 1
不共天地 1
不共戴天 7
不共戴天之仇 2
不共法 1
不共法 1
不具 477
不再 3462
不再出版 0
不再是 596
不再需要 0
不冒 1
不冷 26
不冷不熱 9
不准 445
不凍港 1
不凡 168
不出 829
不出來 304
不出去 48
不出所料 9
不分 753
不分彼此 11
不分晝夜 11
不分男女老幼 7
不分皂白 0
不分軒輊 4
不分輕重緩急 0
不分青紅 0
不分青紅皂白 25
不分首從 1
不切 7
不切實際 108
不切題 0
不列顛 21
不列顛 22
不利 802
不利於 136
不到 2905
不前 26
不前不後 1
不力 108
不加 146
不加思索 7
不加考慮 0
不動 584
不動地 1
不動地 1
不動尊 1
不動尊 1
不動尊金剛明王 1
不動尊金剛明王 1
不動明王 0
不動產 2090
不動聲色 27
不務正業 20
不勝 91
不勝其擾 2
不勝其煩 10
不勝唏噓 13
不勝感激 104
不勝枚舉 92
不勝負荷 14
不勝酒力 2
不勞而獲 24
不升 8
不升反降 28
不卑不亢 14
不即不離 4
不厭 48
不厭其煩 74
不厭其詳 4
不去 751
不及 853
不及格 273
不及物動詞 2
不受 1360
不受歡迎 42
不另 141
不只 739
不只是 900
不可 4901
不可一世 30
不可不 93
不可以 703
不可以道 0
不可以道里計 3
不可偏廢 14
不可勝數 8
不可及 0
不可同日 3
不可同日而語 22
不可名狀 0
不可告人 36
不可多得 94
不可思議 424
不可想像 9
不可或缺 524
不可戰勝 0
不可抗力 80
不可捉摸 8
不可收拾 32
不可救藥 1
不可數 0
不可旁貸 0
不可測知 2
不可理喻 11
不可知論 6
不可磨滅 31
不可終日 2
不可能 1428
不可言喻 0
不可諱言 39
不可逾越 0
不可避免 157
不可開交 13
不可限量 23
不吃 302
不吃白不吃 1
不合 551
不合作 36
不合時宜 87
不合標準 3
不合規定 53
不吊 1
不同 17884
不同以往 0
不同凡響 68
不同印本 0
不同於 542
不同點 86
不名數 0
不名譽 10
不吐 49
不吐不快 0
不向 72
不吝 556
不吝指正 57
不含 1105
不含稅 23
不含糊 10
不吭一聲 0
不吭一聲 0
不吭聲 9
不吵 5
不吵不鬧 0
不告而別 26
不周 51
不和 144
不問 144
不問前因後果 0
不問是非曲直 0
不問青紅皂白 1
不啻 99
不善 110
不善於 25
不喜歡 689
不單 113
不單是 84
不器 0
不圖 3
不在 1381
不在乎 189
不在了 23
不在場 10
不在場 0
不在家 56
不在少數 0
不在意 43
不在此限 393
不在話下 75
不均 161
不堪 352
不堪一擊 26
不堪一擊 26
不堪入目 12
不堪入耳 9
不堪回首 25
不堪設想 15
不塗 3
不墜 64
不壞 70
不壹 1
不外 74
不外乎 177
不外借 59
不外是 66
不多 1012
不多久 14
不多於 16
不多會兒 0
不多會兒 0
不多見 72
不夜城 167
不夠 1906
不夠看 19
不大 1059
不大不小 9
不大好 27
不大想 3
不太 1814
不太好 136
不太想 19
不太會 47
不太能 50
不失 211
不失時機 1
不失為 110
不失眾望 0
不失良機 0
不夷不惠 0
不奈 3
不奈煩 4
不奏 0
不好 2486
不好受 0
不好惹 5
不好意思 702
不好過 11
不如 1894
不如歸去 26
不妙 73
不妥 128
不妨 1279
不孕 137
不孕症 280
不存在 240
不孚眾望 0
不孝 28
不孝子 1
不孝有三 3
不學 39
不學無術 11
不守 48
不守信用 1
不安 690
不安於位 1
不安於室 4
不安本分 0
不完 409
不定 394
不定冠詞 0
不定形 1
不定期 749
不定根 2
不定詞 1
不定量 2
不宜 944
不客氣 96
不宣而戰 2
不害 1
不害 1
不容 615
不容分說 0
不容置喙 2
不容置疑 34
不容置辯 1
不寒而慄 20
不察 33
不實 701
不寧 10
不封 2
不將 137
不尋常 102
不對 779
不對勁 44
不對味 2
不對頭 6
不小 300
不小於 17
不少 4272
不少於 38
不就 378
不就是 187
不屈 29
不屈不撓 33
不屑 177
不屑一顧 16
不屑一顧 16
不屑於 6
不屬於 241
不巧 58
不巧的是 11
不差 151
不已 427
不帶 115
不常 104
不干涉 14
不干涉主義 1
不平 281
不平之鳴 0
不平則鳴 10
不平等條約 15
不平而鳴 0
不平衡 177
不幸 1159
不幸中之 1
不幸中之大幸 2
不幸中的大幸 12
不弱 30
不強不弱 1
不彰 8
不往 7
不待 49
不待說 0
不得 8161
不得不 828
不得了 135
不得人心 3
不得其法 1
不得其解 6
不得好死 0
不得已 209
不得而知 83
不得要領 10
不從 82
不復 176
不循常軌 0
不必 3550
不必要 138
不忍 227
不忍卒睹 0
不忍心 58
不忘 323
不忘本 3
不忙 17
不快 177
不快不慢 3
不忮不求 0
不念 9
不念舊情 1
不念舊惡 1
不怎麼 124
不怎麼 124
不怎麼 124
不怎麼 124
不怕 715
不怕死 14
不怕沒柴燒 5
不怕苦 6
不怕難 4
不急 1
不急不徐 4
不急之務 0
不恥 15
不恥下問 5
不息 48
不悅 107
不惑 28
不惑之年 6
不惜 319
不惜一切 14
不惜一切 14
不惜一戰 2
不惜一戰 2
不惜工本 5
不惜成本 8
不惜犧牲 18
不惟 16
不想 1932
不愁 37
不愁吃 1
不愁穿 0
不意 46
不感 20
不感興趣 25
不愧 33
不愧 33
不愧不怍 0
不愧是 92
不愧為 20
不慌 9
不慌不忙 26
不慍不火 0
不慎 244
不慢 10
不慣 10
不懂 1572
不懂事 24
不懈 127
不應 1000
不應 1000
不應該 369
不懷 8
不懷好意 27
不懼 33
不成 459
不成功 64
不成問題 47
不成器 9
不成敬意 2
不成文法 8
不成文規定 4
不成材 7
不成氣候 0
不成體統 1
不戰 31
不戰而敗 1
不戰而退 2
不才 88
不打 126
不打不相識 9
不打緊 14
不打自招 37
不托 0
不把 263
不折 13
不折不扣 67
不折不扣 67
不抽 38
不拔 15
不拖 13
不拘 803
不拘一格 2
不拘一格 2
不拘小節 27
不拘形式 8
不拘禮節 0
不拿 34
不拿白不拿 0
不按牌理 4
不按牌理出牌 21
不挑 34
不振 204
不掉 133
不掛 7
不揍 0
不提 94
不提也罷 3
不換 86
不擅 27
不擇 10
不擇手段 70
不擋 6
不擋人財路 0
不支 67
不支倒地 12
不改 107
不改其 14
不攻自破 10
不放 230
不放過 18
不敏 4
不敗 178
不敗之地 4
不教而誅 0
不敢 2063
不敢當 35
不散 42
不敬 46
不敵 112
不敷 207
不文 4
不料 340
不新 6
不新不舊 0
不斷 3989
不斷地 1088
不斷的 1087
不斷革命 0
不斷革命論 2
不旋腫 0
不日 13
不早 41
不早了 9
不明 956
不明不白 20
不明事理 2
不明究理 0
不明飛行物 61
不易 1948
不是 14114
不是之處 1
不時 401
不時之需 12
不晚 12
不景氣 737
不智 48
不智之舉 0
不暇 15
不暇顧及 0
不曉得 565
不曉得 565
不更事 0
不更事 0
不曾 639
不曾在 57
不曾有 22
不曾有過 6
不替 15
不會 533
不會 8286
不會不 22
不會不 22
不會是 207
不會有 608
不服 333
不服氣 0
不服老 0
不服輸 14
不期 13
不期然 15
不期而遇 20
不朽 166
不朽名作 2
不材 6
不枉 48
不果 7
不查 21
不根 2
不樂 31
不樂於 2
不欲 63
不欺暗室 0
不歡而散 19
不止 333
不止是 56
不正 243
不正之風 2
不正常 483
不正當 174
不死 294
不死心 30
不比 137
不毛 6
不毛之地 9
不氣 7
不求 174
不求上進 2
不求利 0
不求名 1
不求名利 1
不求有功 1
不求甚解 8
不求聞達 3
不治 100
不治之症 17
不治而死 1
不法 486
不法之徒 14
不法行為 47
不消 37
不消說 4
不淑 0
不淨觀 1
不淺 31
不清 315
不清不楚 12
不清不白 2
不減當年 5
不渝 39
不測 32
不準 445
不準 123
不溯既往 0
不滅 105
不滅定律 0
不滿 1103
不漲 33
不漲反跌 23
不澀 5
不澀不苦 0
不濟 22
不濟事 0
不為 486
不為 486
不為所動 31
不為牛後 3
不為瓦全 2
不無 105
不無可疑 0
不無可疑之處 0
不無小補 12
不然 1819
不然就 124
不煩 18
不爭 116
不爭氣 20
不爽 275
不特 15
不犯 21
不獨 29
不獨有偶 0
不玩 51
不理 316
不理不睬 20
不甘 126
不甘寂寞 31
不甘心 96
不甘示弱 49
不甚 272
不生不死 0
不生不滅 21
不用 3658
不用說 124
不由 166
不由分說 16
不由得 150
不由自主 148
不男不女 6
不畏 134
不畏強敵 0
不畏強權 5
不畏艱險 1
不畏艱難 19
不留 130
不留情 12
不留情面 8
不留餘地 2
不當 1415
不疑有他 0
不疾不徐 0
不痛 42
不痛不癢 19
不白之冤 5
不盡 463
不盡然 18
不相 122
不相上下 29
不相容 102
不相干 64
不相往來 2
不相聞問 0
不省人事 19
不看 537
不真 16
不真不假 0
不眠不休 47
不睬 10
不知 60
不知不覺 406
不知凡幾 0
不知分寸 0
不知去向 46
不知好人 0
不知好人心 0
不知好歹 13
不知就裡 0
不知廉恥 5
不知情 105
不知所云 50
不知所措 188
不知死活 21
不知為不知 0
不知甘苦 0
不知究理 0
不知羞恥 13
不知者不罪 0
不知者無罪 3
不知自愛 0
不知覺 12
不知輕重 3
不知道 4886
不短 23
不破不立 1
不礙 6
不礙事 0
不祥 65
不祥之兆 3
不禁 999
不移 47
不稂不秀 0
不稂不秀 0
不稂不莠 0
不穩 309
不空如來藏 1
不符 685
不第 10
不等 353
不等價交換 0
不等式 39
不等號 0
不等邊 5
不等量 0
不算 639
不算什麼 31
不算數 7
不管 2420
不管三七二十一 10
不管事 878
不管什麼 21
不管什麼 21
不管怎樣 53
不管怎樣 53
不管怎麼 21
不管怎麼 21
不管是 879
不管是誰 11
不簡單 66
不約 7
不約而同 144
不純 112
不結盟 3
不結盟國 0
不結盟國家 1
不結盟運動 1
不絕 115
不絕如縷 4
不絕於耳 42
不經 158
不經一事 7
不經一事 7
不經一事不長一智 3
不經事 1
不經意 140
不經濟 38
不縫 0
不繁 0
不缺 52
不置 16
不置可否 13
不羈 54
不義 88
不義之財 6
不習慣 103
不翼而飛 19
不老 89
不老草 0
不耐 148
不耐 148
不耐煩 75
不聞不問 22
不聲不響 7
不聽 200
不肖 271
不肖子 271
不肖子孫 2
不肯 603
不肯來 0
不肯到 5
不肯去 4
不能 10231
不能不 197
不能夠 102
不能用 301
不能自己 30
不能自已 30
不能自拔 15
不能言傳 3
不脛而走 6
不自 71
不自主 53
不自在 32
不自由 37
不自禁 16
不自禁 16
不自覺 176
不自量力 12
不臭 7
不至於 244
不致 744
不致於 158
不與 176
不興 7
不舉 0
不良 1710
不良份子 16
不良份子 16
不良傾向 1
不良導體 1
不良少年 36
不良影響 106
不良於行 0
不良習慣 8
不良青少年 3
不苟言笑 1
不若 181
不落 52
不落人後 75
不落俗套 0
不落窠臼 0
不著 131
不著痕跡 17
不著邊際 5
不著陸 0
不著陸飛行 0
不薄 7
不藥而癒 15
不虛 4
不虛此生 0
不虛此行 49
不虞 25
不虞匱乏 31
不行 764
不行了 79
不被 445
不要 12131
不要緊 49
不要臉 64
不見 1411
不見不散 12
不見了 639
不見天日 13
不見得 362
不見得 362
不見棺材不掉淚 2
不見經傳 3
不規則 202
不覺 263
不覺得 191
不覺得 191
不解 319
不解之仇 1
不解之謎 0
不解事 0
不解風情 3
不言不語 12
不言而喻 25
不計 277
不計其數 64
不計成本 10
不託 0
不記名投票 3
不許 131
不許動 2
不許失敗 1
不該 754
不詳 330
不認 81
不認帳 7
不說 377
不說出 5
不調 73
不談 267
不談別的 0
不請 15
不請自來 17
不論 2787
不論什麼 15
不論什麼 15
不論什麼 15
不論如何 69
不論怎樣 3
不論怎樣 3
不論怎麼 9
不論怎麼 9
不論怎麼 9
不論怎麼 9
不論是 1096
不論是誰 8
不諱 11
不諱言 0
不謀其政 0
不謀其職 0
不謀而合 36
不謂 5
不講 80
不講情面 1
不講理 29
不講道理 2
不識一丁 0
不識一丁 0
不識之無 0
不識大體 2
不識抬舉 4
不識時務 7
不識泰山 1
不識相 0
不識貨 5
不讀 42
不變 944
不變資本 2
不讓 781
不讓人 20
不讓鬚眉 12
不貞 8
不負 501
不負所托 1
不負眾望 20
不貲 0
不貳 5
不貳過 0
不貴 57
不費 40
不費吹灰之力 18
不賴 62
不走 176
不起 352
不起來 119
不起眼 61
不足 3487
不足取 8
不足掛齒 7
不足為外人道也 1
不足為奇 48
不足為憑 0
不足為訓 1
不足輕重 1
不足道 10
不足道也 1
不跌 17
不跌反漲 1
不跟 113
不跟人 2
不跳 10
不軌 21
不輕 64
不輕不重 1
不輸 104
不輸給 19
不辨是非 1
不辭 32
不辭勞苦 9
不辭而別 6
不辭辛勞 44
不辭辛苦 10
不辱使命 3
不近 14
不近人情 5
不迭 9
不追 29
不退 145
不退反進 0
不送 40
不透 83
不透光 30
不透明 42
不透明性 2
不透水 50
不通 214
不通人情 0
不通過 27
不速之客 32
不週 34
不進 77
不進不退 1
不進來 11
不進則退 5
不進去 54
不進反退 5
不遂 6
不過 18703
不過來 40
不過去 21
不過問 8
不過如此 44
不過是 599
不過爾爾 6
不遑多讓 0
不道德 57
不遜 12
不遜於 62
不遠千里 13
不遠千里而來 6
不適 719
不適合 416
不遲 29
不遺餘力 238
不避 23
不避艱險 0
不配 102
不酸 4
不酸不甜 0
不酸不苦 0
不酸不辣 0
不醜 12
不量力 0
不釣 1
不錯 4858
不錯的 480
不鏽鋼 398
不長 88
不長 88
不長一智 6
不長一智 6
不長不短 2
不關 62
不關痛癢 1
不降 31
不降反升 9
不限 655
不限在 3
不限於 66
不隨 28
不隨意 46
不隨意肌 3
不雅 121
不雅觀 3
不難 496
不需 1415
不需要 1072
不靈 28
不須 638
不領 6
不領情 0
不願 1897
不願 333
不願意 364
不顧 305
不顧一切 79
不顧一切 79
不顧大局 0
不顧後果 2
不食人間煙火 20
不飽和脂肪酸 1
不騎 4
不驕不躁 0
不驕不餒 0
不高 423
不高不低 1
不高不矮 0
不鳴則已 1
不黏 9
不點 12
不齊 54
不齒 30
与 852
丏 8
丐 360
丐幫 74
丑 1636
丑劇 3
丑旦 0
丑時 0
丑角 50
丑角 50
丒 0
且 1
且 19
且 19961
且慢 31
且戰且走 0
且歌且舞 0
且看 118
且看 118
且聽下回分解 0
且說 93
丕 198
丕業 0
丕變 55
丕顯 0
世 11044
世上 670
世上無難事 1
世上無難事只怕有心人 1
世世 6
世世代代 38
世主 1
世亂識忠臣 1
世事 873
世事通考 1
世交 6
世人 662
世仇 16
世代 1521
世代交替 0
世代相傳 18
世伯 3
世俗 215
世俗主義 3
世俗之見 0
世俗諦 1
世傳 14
世儒 1
世兄 6
世典 1
世出世 1
世務 4
世及 1
世叔 0
世味 1
世嗣宦族 1
世外 54
世外桃源 94
世好 1
世好 1
世婦 1
世子 10
世宗 47
世家 257
世家子弟 1
世尊 167
世局 39
世局多變 0
世居 37
世弟 0
世德 10
世恩錄 1
世情 1
世情看冷暖 1
世情看冷暖人面逐高低 1
世態 1
世態人情 1
世態炎涼 7
世所周知 0
世故 29
世教 1
世敵 1
世新 594
世新大學 1
世族 1
世業 1
世母 1
世法 1
世海他人 1
世父 1
世牒 1
世界 30817
世界上 1213
世界主義 10
世界人 7
世界人權宣言 1
世界人權節 1
世界協定時 1
世界反共聯盟 1
世界史 87
世界地圖 65
世界地球日 1
世界報 6
世界大同 34
世界大戰 146
世界屋脊 0
世界展望會 1
世界市場 77
世界市場 77
世界性 208
世界文化 43
世界新專 8
世界新聞傳播學院 1
世界新聞專科學校 11
世界新聞職業學校 1
世界末日 262
世界潮流 104
世界紀錄 1
世界級 275
世界經濟 201
世界經濟危機 0
世界華文文學獎 1
世界觀 151
世界語 7
世界貨幣 4
世界貿易中心 86
世界貿易大樓 1
世界貿易組織 1
世界運動會 1
世界銀行 0
世盟 16
世祚 1
世系 11
世紀 6603
世紀末 567
世綵堂 1
世綵堂帖 1
世胄 1
世胙 1
世臣 2
世臺 1
世表 1
世衰道微 0
世襲 30
世親 1
世說新語 19
世誼 0
世誼 0
世講 1
世譜 1
世讎 1
世貿 615
世貿中心 245
世貿大樓 33
世路 1
世運 16
世運會 1
世道 25
世道人心 6
世道人情 1
世銀 0
世間 762
世間法 1
世阿彌 30
世面 28
世風 1
世風不古 0
世風日下 17
世風澆薄 0
丗 0
丘 1085
丘中有麻 1
丘井 1
丘八 1
丘園 1
丘墓 1
丘墟 1
丘墳 1
丘壑 3
丘壟 1
丘嫂 1
丘山 1
丘木 1
丘疹 1
丘腦 1
丘處機 0
丘逢甲 1
丘遲 1
丘陵 114
丘陵地 17
丘陵地區 5
丘陵地帶 8
丘陵地形 4
丘隴 1
丙 2275
丙丁 53
丙國 0
丙校 0
丙烯酸 149
丙烯醛 17
丙烷 61
丙烷氣 1
丙種 95
丙等 58
丙等體位 27
丙級 328
丙綸 1
丙線 4
丙酮 103
丙醇 37
丙項 0
丞 472
丞相 93
丟 1017
丟三落四 8
丟上 1
丟上來 6
丟上去 1
丟下 80
丟下不管 1
丟下來 1
丟下去 4
丟了 177
丟人 26
丟人現眼 17
丟到 109
丟卒保車 0
丟在 65
丟在腦後 0
丟失 35
丟得 3
丟手 1
丟手不管 0
丟掉 136
丟擲 16
丟棄 233
丟的 44
丟盔棄甲 2
丟眼色 0
丟給 71
丟置 2
丟置不管 0
丟臉 125
丟起 0
丟起來 0
丟開 9
丠 0
両 0
両 0
丢 0
丣 0
两 0
两 0
並 8190
並上 22
並上來 0
並上去 0
並不 6393
並不在 63
並不是 2331
並不會 519
並不能 447
並且 6791
並且在 382
並且是 79
並且會 52
並且有 156
並且能 79
並先 82
並入 16
並再 91
並出 92
並出來 0
並出去 1
並列 366
並列冠軍 0
並列埠 0
並列第一 3
並列譯文 0
並到 39
並力 8
並可 1495
並可以 110
並合 39
並同 275
並回 44
並回來 0
並回去 0
並在 1801
並存 172
並存在 2
並存著 5
並將 2591
並往 18
並得 734
並從 259
並排 80
並排而坐 2
並於 2321
並是 12
並會 156
並有 1590
並未 2470
並條 1
並沒 307
並沒有 2717
並派 40
並為 542
並無 1879
並無不可 2
並無不當 8
並用 325
並由 1193
並當 45
並立 35
並翼齊飛 0
並聯 63
並肩 73
並肩作戰 19
並肩而行 6
並能 536
並能夠 10
並與 904
並舉 128
並舉手 0
並蒂 8
並行 142
並行不悖 11
並行不悖 11
並要 124
並請 3101
並論 21
並躺 2
並躺在 0
並進 701
並進來 0
並進去 0
並過 5
並過來 0
並過去 0
並邀 236
並重 449
並非 2548
並非在 35
並非是 125
並駕齊驅 34
丨 0
丩 0
个 0
个 0
丫 691
丫丫 120
丫嬛 1
丫子 0
丫環 11
丫頭 446
丫頭 446
丫鬟 16
中 14865
中 148655
中上 66
中上層 11
中上游 27
中上程度 7
中上等 1
中上級 0
中上階層 7
中上階級 0
中下 71
中下層 8
中下游 83
中下程度 0
中下等 0
中下級 2
中下階層 14
中下階級 2
中世 17
中世紀 138
中了暑 0
中了標 0
中了毒 0
中亞 140
中亞國家 7
中亞地區 4
中人 390
中介 152
中介子 0
中代 7
中位數 35
中低 267
中低層 4
中使 36
中俄 53
中俄友好條約 0
中保 32
中傷 40
中元 114
中元普渡 1
中元節 28
中共 9449
中共中央 125
中共政府 23
中共軍隊 12
中冊 9
中分 55
中副 719
中北歐 0
中北美 0
中北部 24
中北部地區 0
中北非 0
中區 1450
中午 1375
中南 287
中南亞 0
中南半島 41
中南歐 0
中南海 87
中南美 177
中南美洲 233
中南部 270
中南部地區 35
中南非 5
中印 33
中印國界 0
中原 1002
中原地區 5
中原大學 1740
中原板蕩 1
中原標準時間 31
中原逐鹿 1
中古 623
中古世紀 41
中古史 10
中古時代 9
中古車 624
中台灣 121
中台灣地區 2
中台禪寺 1
中台醫專 42
中和 1603
中和劑 18
中和力 2
中和市 2605
中唐 16
中唐時期 1
中固 5
中圈 6
中圈 6
中圈套 2
中圈套 2
中國 29568
中國人 1612
中國人民 204
中國史 105
中國史綱 1
中國同盟會 0
中國商業銀行 3
中國商銀 227
中國國民黨 663
中國城 109
中國大陸 0
中國字 21
中國工人 2
中國工商專科學校 146
中國年 17
中國廣播公司 135
中國廣播公司 135
中國式 67
中國文化 326
中國文化大學 1461
中國文化學院 14
中國文學 595
中國文藝 20
中國文藝學會 1
中國時報 5877
中國海 181
中國海事專科學校 61
中國海專 34
中國畫 48
中國石油 1038
中國石油公司 365
中國社會 134
中國社會科學院 53
中國菜 67
中國話 15
中國農民 17
中國農民銀行 81
中國醫藥 196
中國醫藥大學 1
中國醫藥學院 1057
中國銀行 21
中國電視公司 60
中國青年 52
中國青年反共救國團 16
中土 30
中垂線 1
中型 464
中型公車 3
中堅 95
中堅份子 42
中場 10
中場 106
中壢 1389
中壢人 2
中壢地區 19
中壢市 1934
中士 12
中外 606
中大 1196
中大型 108
中天 3227
中央 13130
中央信託局 166
中央圖書館 179
中央大學 2656
中央委員 71
中央委員會 71
中央山脈 252
中央山脈 252
中央情報局 76
中央政府 2540
中央新村 10
中央日報 4418
中央標準局 510
中央研究院 64775
中央社 20208
中央處 7
中央處理器 237
中央處理器 237
中央警官學校 41
中央軍 3
中央通訊社 309
中央酒店 0
中央銀行 648
中央集權 47
中央面 0
中央黨部 258
中子 166
中子彈 1
中子星 17
中子流 0
中學 3788
中學生 117
中學西用 0
中將 487
中將湯 4
中專 16
中尉 119
中尉軍官 1
中小 771
中小企業 2680
中小企業銀行 396
中小學 2415
中小學校 0
中小學生 0
中局 12
中層 54
中層階級 2
中山 4133
中山北路 1302
中山區 1119
中山南路 204
中山堂 133
中山大學 2912
中山服 0
中山樓 23
中山裝 5
中山路 6483
中山醫學大學 1
中山醫學院 491
中山陵 7
中崙 128
中崙站 4
中嶽 9
中州 203
中州工商專校 1
中州工專 4
中州工業專科學校 1
中州技術學院 1
中巴 40
中常 211
中常委 0
中常會 370
中平 159
中年 380
中年人 80
中年期 8
中年級 257
中度 241
中度颱風 20
中庭 371
中庸 134
中庸之道 11
中庸思想 0
中廣 395
中廣公司 13
中式 536
中彈 34
中彩 2
中影 153
中微子 0
中德 103
中心 50424
中心人物 4
中心任務 11
中心內容 2
中心區 33
中心地帶 10
中心地段 0
中心對稱 3
中心德目 0
中心思想 25
中心線 58
中心角 2
中心診所 31
中心體 5
中心點 88
中性 336
中性人 2
中性名詞 0
中性染料 0
中情 116
中意 171
中括號 1
中拳 0
中拳 0
中指 91
中提琴 61
中支 9
中文 25477
中文化 0
中文字 246
中文書 147
中文版 1
中文系 435
中文電腦 107
中斷 760
中斷點 11
中日 787
中日戰爭 21
中日文化交流 11
中日甲午戰爭 11
中日詞典 0
中日韓 1
中旬 835
中時 29
中時晚報 4376
中晚期 20
中晚期作品 1
中景 17
中暑 56
中書令 1
中書省 3
中朝 4
中期 548
中材 51
中東 561
中東國家 9
中東地區 103
中板 21
中校 124
中極 107
中槍 15
中樓 12
中標 0
中樞 166
中樞神經 154
中樞神經系統 102
中樞系統 2
中橫 364
中欄 0
中歐 93
中歐國家 3
中歐地區 0
中止 249
中正 6036
中正國中 74
中正國小 145
中正國民中學 1
中正國民小學 1
中正大學 1
中正廳 4
中正機場 41
中正機場 410
中正理工學院 336
中正紀念堂 280
中正路 5807
中段 94
中毒 1196
中毒已深 1
中毒者 23
中比 19
中比關係 0
中氣 30
中氣不足 2
中沙群島 3
中油 1594
中法 101
中法戰爭 17
中法詞典 0
中波 68
中波 68
中波段 0
中波段 0
中波長 0
中波長 0
中泰 109
中洲 109
中派 12
中流 45
中流砥柱 20
中港溪 45
中游 127
中營 35
中獎 1381
中班 92
中生 75
中生代 151
中產 75
中產階級 139
中用 91
中略 0
中盤 28
中盤商 9
中看 127
中看不中吃 0
中看不中用 6
中看不中聽 0
中知 5
中短 26
中短期 13
中短波 0
中短波 0
中研院 1524
中碳鋼 5
中秋 395
中秋佳節 34
中秋月餅 23
中秋節 312
中科院 379
中程 295
中程計劃 40
中程飛彈 4
中稻 1
中空 230
中立 397
中立不倚 0
中立主義 1
中立區 1
中立國 4
中立國家 3
中立地區 0
中立地帶 0
中立法 19
中立派 0
中站 36
中等 2059
中等專業學校 4
中等教育 175
中等生 0
中等艙 0
中筋麵粉 1
中策 3
中篇 27
中篇小說 58
中簽 0
中紐 25
中級 773
中級人民法院 7
中級品 2
中統 58
中經 95
中線 204
中緯度 33
中縣 438
中繼 255
中繼站 55
中美 1060
中美洲 592
中義 43
中耕 19
中耳 116
中耳炎 133
中聽 20
中肯 48
中胚層 2
中腦 11
中臺科技大學 1
中臺醫事技術專科學校 1
中臺醫專 4
中臺醫護技術學院 1
中興 3516
中興大學 2761
中興大業 8
中興小學 0
中興新村 292
中興法商學院 49
中英 379
中英夾雜 3
中英詞典 0
中草藥 43
中菜 36
中華 15180
中華人民共和國 595
中華商場 14
中華商場 14
中華大學 1
中華工商專科學校 1
中華工專 12
中華工業專科學校 0
中華技術學院 1
中華文化 327
中華日報 311
中華書局 35
中華民國 25670
中華民族 234
中華經濟研究院 164
中華醫專 56
中華開放醫院 0
中華隊 618
中華電信 0
中華電視 207
中華電視台 26
中華電視臺 0
中華革命黨 5
中落 8
中葉 111
中葡 7
中葡詞典 0
中藥 868
中藥學 0
中藥店 14
中藥房 49
中藥舖 7
中號 9
中行 38
中衛 202
中表 19
中裝 15
中西 469
中西合併 4
中西合璧 18
中西思想 12
中西文化 25
中西詞典 0
中西部 54
中規中矩 22
中規中矩 22
中視 546
中視公司 6
中覺 2
中計 40
中論 9
中諦 1
中譯 127
中譯本 77
中越 34
中越國界 0
中距離 20
中距離賽跑 0
中路 533
中軍 31
中輟 105
中轉 18
中農 16
中近東 0
中途 519
中途島 10
中途站 19
中道 86
中選 545
中選會 0
中邪 0
中郎 6
中郎將 9
中部 1585
中部地區 415
中部地方 8
中醫 2199
中醫學院 27
中醫師 411
中醫界 8
中量 16
中量級 20
中鋒 68
中鋼 623
中鏢 0
中鐵 32
中長期 623
中長波 1
中長波 1
中長鐵路 4
中間 1723
中間人 21
中間商 16
中間性 1
中間派 3
中間色 2
中間體 45
中陰 1
中陰身 1
中隊 160
中隊長 13
中集 23
中青年 15
中非 79
中非共和國 5
中非國家 0
中非地區 0
中靶 0
中靶 0
中韓 112
中韓詞典 0
中音 27
中音部 0
中頻 37
中風 446
中飯 18
中飯時 1
中飽 1
中飽私囊 10
中餐 151
中餐廳 48
中高 175
中高層 4
中高級 20
中高音 4
中黃 28
中點 62
中點站 3
中齡林 0
丮 1
丯 0
丰 691
丰南 0
丰台 0
丰姿 20
丰采 35
丰韻 1
丱 45
串 1272
串上 1
串上來 0
串上去 0
串供 8
串供 8
串出來 0
串列 633
串成 19
串演 3
串繞電動機 0
串聯 168
串聯式 8
串街 0
串通 32
串連 295
串鈴 2
串門 5
串門子 50
串騙 0
丳 122
丵 0
丶 0
丸 973
丸子 447
丸藥 2
丸藥盒 0
丸髻 0
丹 16995
丹丸 3
丹尼爾 104
丹徒 0
丹心 11
丹忱 2
丹書鐵券 1
丹桂 7
丹楓 9
丹毒 43
丹田 64
丹田功 0
丹砂 3
丹粒 0
丹紅 5
丹紅色 0
丹藥 14
丹詔 1
丹陽 48
丹青 359
丹青圖 0
丹頂鶴 1
丹頓 9
丹鳳 70
丹鳳眼 2
丹麥 545
丹麥人 10
主 51412
主事 29
主事者 58
主人 1120
主人公 32
主人家 11
主人翁 135
主任 8494
主任委員 943
主伐 3
主位 2
主使 32
主使人 3
主使者 11
主供 6
主供 6
主修 529
主修科目 13
主值 0
主僕 22
主公 51
主刑 1
主力 1445
主力艦 5
主力軍 3
主動 2716
主動性 59
主動權 9
主動脈 88
主動脈 88
主動脈瘤 11
主動脈瘤 11
主動軸 8
主句 1
主名 56
主和 51
主和派 1
主唱 180
主因 505
主基督 22
主場 0
主場 0
主外 22
主委 1135
主婚 4
主婚人 8
主婦 276
主嫌 0
主子 35
主客 21
主客觀 47
主宰 275
主宰權 3
主審 55
主審官 1
主將 108
主導 1109
主導作用 0
主導權 70
主峰 120
主帆 2
主帥 116
主席 3315
主席台 2
主席團 74
主幹 142
主廚 116
主張 2675
主從 370
主從 370
主意 175
主應力 8
主戰 14
主戰派 2
主戲 6
主持 2172
主持人 2899
主持正義 13
主控台 9
主控權 0
主播 0
主攻 73
主政 109
主政者 51
主教 121
主文 535
主旋律 34
主日學 159
主旨 1334
主枝 5
主根 3
主格 0
主業 40
主機 4271
主機板 2885
主檔 221
主權 1031
主權國 10
主權國家 101
主權在民 50
主次 27
主母 10
主治 211
主治大夫 19
主治大夫 19
主治醫師 1425
主治醫生 7
主流 1372
主流派 41
主演 205
主潮 0
主牆 8
主犯 3
主產 17
主監 1
主祭 16
主禱文 6
主稿 0
主筆 90
主管 6149
主管人員 176
主管機構 29
主管機關 6369
主管部門 56
主線 84
主編 935
主義 2349
主考 11
主考官 35
主脈 7
主脈 7
主脈力 0
主腦 7
主腦人 0
主菜 60
主要 27694
主要人物 41
主要原因 613
主要問題 38
主要因素 177
主要性 2
主要是 2308
主要樹種 7
主要矛盾 2
主見 76
主視圖 0
主觀 464
主觀主義 3
主觀唯心主義 0
主觀性 18
主觀能動 0
主觀能動性 0
主角 1243
主角 1243
主計 373
主計室 107
主計處 1175
主計長 52
主記 1
主記憶體 183
主詞 30
主試 37
主試官 3
主語 3
主調 14
主謀 19
主謀者 8
主講 973
主賓 3
主路線 0
主軸 604
主辦 2158
主辦人 89
主辦單位 1575
主辦國 78
主辦權 39
主辦者 27
主長 3
主開關 4
主音 11
主項 0
主題 41631
主題曲 461
主題歌 11
主顧 8
主食 93
主體 858
主體性 105
主體際性 0
丼 51
丼 51
丽 0
丽 0
丿 0
乀 0
乁 0
乂 24
乂 24
乃 6704
乃兄 4
乃公 22
乃就 19
乃弟 0
乃是 1896
乃父 7
乃祖 0
乃翁 2
乃能 20
乃至 510
乃至於 242
乃蠻 0
久 17836
久久 731
久仰 11
久仰大名 2
久候 13
久別 7
久別重逢 11
久存 3
久已 20
久旱不雨 1
久旱逢甘雨 1
久旱逢甘霖 5
久留 24
久病 20
久病在床 0
久病成良醫 22
久經 17
久經考驗 1
久而久之 96
久負盛名 0
久違 75
久遠 129
久長 12
久雨 1
久雨成災 0
乆 0
乇 1
么 232
么么唱唱 0
么兒 2
么喝 6
么妹 4
么弟 2
么爺 0
义 0
之 327781
之慨 100
之一 7259
之上 778
之下 2711
之中 2848
之主 750
之久 286
之乎者也 1
之二 1429
之交 106
之內 1228
之分 231
之初 537
之前 5400
之友 1296
之右 23
之地 0
之外 4158
之多 475
之夜 774
之子 346
之字路 1
之害 38
之家 2190
之屋 77
之島 211
之左 34
之後 13336
之恩 53
之意 525
之愛 240
之戰 255
之所以 990
之於 277
之時 819
之樂 181
之流 92
之無 102
之秋 20
之罪 169
之至 50
之舉 98
之舞 83
之處 1893
之見 71
之計 118
之身 71
之輩 54
之道 1289
之間 7387
之際 1099
之類 707
之餘 593
乍 116
乍冷 1
乍冷乍熱 0
乍富 0
乍寒乍熱 0
乍得 0
乍暖 0
乍暖乍寒 0
乍暖還寒 6
乍然 10
乍熱 0
乍熱乍冷 0
乍現 73
乍看 1
乍看之下 54
乍看來 1
乍聽 11
乍聽之下 20
乍見 24
乍讀 0
乍讀之下 0
乍雨乍晴 0
乎 1555
乎呼 0
乎嚕 0
乏 312
乏人問津 55
乏力 33
乏力感 1
乏味 94
乏善 1
乏善可陳 29
乏汽 0
乑 0
乒 58
乒乒乓乓 13
乒乓 38
乒乓外交 0
乒乓球 53
乒乓球拍 0
乒乓球桌 5
乒乓球賽 0
乓 27
乔 0
乔 0
乖 535
乖乖 441
乖乖牌 1
乖乖虎 2
乖僻 2
乖到 1
乖孩子 11
乖寶寶 26
乖巧 89
乖張 13
乖得 1
乖戾 8
乖異 0
乖舛 5
乖覺 0
乖謬 2
乖違 4
乖隔 0
乗 0
乗 0
乘 241
乘 14757
乘上 43
乘了 1
乘人之危 4
乘便 3
乘偽行詐 0
乘其不備 1
乘具 5
乘冪 1
乘到 0
乘勝 7
乘勝追擊 27
乘勢 29
乘在 3
乘坐 215
乘客 460
乘得 0
乘敵不備 0
乘數 48
乘方 3
乘機 54
乘法 100
乘法器 11
乘涼 27
乘積 22
乘空 2
乘興 7
乘興而來 3
乘興而去 1
乘興而往 0
乘興而往敗興而歸 0
乘船 62
乘著 94
乘虛 7
乘虛而入 7
乘號 3
乘起 0
乘起來 1
乘車 161
乘過 2
乘過來 0
乘過去 0
乘間 1
乘除 13
乘隙 3
乘風 19
乘風破浪 19
乘馬 7
乘龍快婿 0
乙 6223
乙丙橡膠 1
乙二醇 98
乙卯 6
乙太 321
乙太網路 0
乙存 3
乙炔 51
乙烯 378
乙烷 110
乙班 331
乙稀 0
乙種 142
乙種維生素 0
乙等 114
乙苯 16
乙酸 50
乙酸乙酯 16
乙醇 129
乙醇溶液 2
乙醇溶液 2
乙醚 63
乙醛 35
乚 0
乚 0
乜 35
乜 35
乜乜斜斜 0
乜斜 1
九 17837
九一一 730
九一八事變 12
九三軍人節 4
九世 19
九世之仇 0
九世之讎 0
九世紀 13
九丘 1
九九 730
九九乘法 12
九九乘法表 10
九九八十一 3
九九登高 0
九九表 1
九九重陽 8
九二一 730
九五 954
九份 1
九位 128
九位數 1
九個 337
九個人 7
九個月 220
九倍 14
九億 179
九元 91
九分 124
九分新 0
九分熟 0
九分鐘 5
九劃 16
九區 24
九十 2167
九千 359
九千人 7
九千個 2
九千元 27
九千塊 1
九卿 1
九命奇冤 0
九命怪貓 1
九品 23
九品中正 3
九員 4
九圈 0
九圓 0
九坪 7
九塊 8
九塊錢 0
九大 175
九大行星 54
九天 105
九天之外 1
九天仙女 0
九天玄女 0
九如 536
九孔 89
九宮 32
九宮格 7
九宮鍵 0
九宵 6
九宵雲 0
九宵雲外 0
九局 27
九局上半 1
九局下半 4
九屆 560
九層塔 1
九州 234
九師 4
九幕 38
九年 2463
九年級 4
九式 10
九彎十八拐 4
九成 400
九成新 69
九打 2
九折 194
九折優待 97
九族 74
九族文化村 180
九日 1723
九時 560
九晚 47
九曲橋 7
九曲洞 0
九月 6097
九月份 768
九杯 1
九桶 0
九棵 3
九樓 137
九樓住戶 0
九歌 880
九步 5
九歲 180
九死 0
九死一生 15
九死一生 15
九死不悔 0
九江 44
九泉 4
九泉之下 5
九洲 21
九流 2
九牛一毛 2
九牛一毛 2
九牛二虎 1
九牛二虎之力 9
九班 93
九瓶 1
九百 472
九百人 6
九百個 0
九百元 23
九百塊 1
九皋 0
九碗 0
九科 6
九秒 17
九秒鐘 1
九竅 4
九站 4
九章 137
九章算術 4
九節 49
九節課 1
九級 27
九級風 0
九組 50
九經 4
九缸 0
九萬 161
九萬人 9
九萬個 1
九萬元 30
九萬塊 0
九號 645
九街 60
九角 3
九角形 0
九課 6
九轉金丹 0
九通 13
九部 28
九重 0
九重天 0
九重葛 27
九集 59
九霄 21
九霄雲外 10
九頭鳥 0
九顆 3
九點 575
九點鐘 13
九龍 274
九龍區 0
九龍半島 1
九龍灣 3
乞 105
乞丐 135
乞乞縮縮 0
乞兒 11
乞命 1
乞和 1
乞巧 0
乞憐 0
乞援 1
乞求 18
乞糧 0
乞討 22
乞貸 0
乞降 0
乞靈 1
乞食 24
也 84348
也不 4648
也不到 13
也不在 48
也不是 943
也不會 933
也不能 693
也似 40
也似地 1
也似地 1
也來 442
也像 147
也先 12
也再 37
也到 118
也去 114
也可以 4965
也向 136
也咬 1
也在 1725
也好 592
也對 499
也對於 21
也就 1859
也就是 3409
也就是說 991
也想 697
也把 196
也拔 4
也搖 2
也是 14303
也會 3589
也會在 98
也會有 292
也有 6980
也沒 1283
也沒有 1730
也罷 129
也能 1888
也能夠 161
也要 3519
也許 3771
也許在 89
也許是 618
也許會 226
也許有 94
也許能 39
也過 28
也過來 5
也過去 4
也門 0
乢 0
乢 0
乢 0
乣 0
乨 0
乨 0
乩 25
乩童 45
乪 0
乱 0
乱 0
乳 1729
乳光 0
乳兒 2
乳劑 63
乳化 218
乳化劑 56
乳化液 8
乳化液 8
乳名 8
乳品 107
乳娘 0
乳房 759
乳暈 26
乳母 18
乳汁 85
乳汁狀 1
乳液 123
乳液 123
乳液聚合 0
乳液聚合 0
乳溝 5
乳濁液 4
乳濁液 4
乳燕 3
乳牛 187
乳狀 11
乳癌 422
乳白 22
乳白色 34
乳粉 6
乳粥 0
乳糖 74
乳糜 9
乳糜管 0
乳紫色 0
乳缽 1
乳脂 3
乳脂狀 1
乳脂糖 0
乳脂肪 2
乳脂酸 0
乳腺 91
乳膠 81
乳臭 3
乳臭未乾 5
乳草 1
乳製品 66
乳豬 40
乳質 0
乳酒 0
乳酪 140
乳酪 140
乳酸 121
乳酸菌 54
乳酸鈣 2
乳霜 1
乳頭 215
乳頭狀 17
乳類 14
乳餅 0
乳香 22
乳香樹 0
乳香脂 0
乳鴿 2
乳齒 18
乴 0
乵 0
乹 0
乹 0
乾 272
乾 2727
乾上 4
乾乾淨淨 49
乾了 77
乾井 0
乾個 4
乾個夠 0
乾兒子 8
乾冰 52
乾冷 13
乾咳 25
乾哥 6
乾哥哥 1
乾唱 0
乾嘔 3
乾嚎 0
乾地 5
乾坤 296
乾女兒 7
乾妹 16
乾妹妹 7
乾姊 3
乾姊姊 2
乾姊姊 2
乾娘 3
乾媽 20
乾季 18
乾屍 3
乾巴巴 3
乾巴巴的 1
乾弟 3
乾弟弟 2
乾得 4
乾性 66
乾性油 0
乾性皮膚 31
乾打雷 0
乾掉 19
乾旱 134
乾旱地區 4
乾材 3
乾杯 49
乾果 10
乾枯 32
乾柴 6
乾柴烈火 10
乾池 3
乾洗 112
乾洗店 38
乾涸 52
乾涼 0
乾淨 1081
乾淨俐落 19
乾渠 0
乾渴 19
乾溝 5
乾澀 0
乾熱 18
乾燥 1012
乾燥劑 23
乾燥器 25
乾燥機 188
乾燥箱 7
乾爸 6
乾爸爸 2
乾爹 20
乾爽 38
乾片 1
乾田 0
乾瘦 2
乾癟 10
乾癟癟 1
乾癬 68
乾的 87
乾眼病 0
乾瞪眼 6
乾禮 0
乾笑 14
乾等 6
乾粉 69
乾糧 36
乾絲 1
乾肉片 0
乾股 15
乾脆 573
乾草 21
乾草叉 0
乾草堆 1
乾草機 0
乾草粉 0
乾菜 1
乾著急 3
乾薪 0
乾裂 29
乾親 0
乾貝 1
乾躁 5
乾酪 26
乾酵母 5
乾闥婆 1
乾隆 247
乾隆年間 23
乾隆皇帝 18
乾雷 2
乾電池 160
乾飯 4
乾餾 4
乾餾法 1
乾鮮 2
乾麵 1
乿 3
亀 0
亀 0
亀 0
亁 0
亁 0
亂 2633
亂七八糟 178
亂世 240
亂世佳人 28
亂世兒女 7
亂丟 43
亂丟垃圾 15
亂丟紙屑 3
亂亂 31
亂亂的 12
亂了 86
亂了手腳 5
亂事 7
亂交 1
亂作一團 0
亂來 43
亂來亂去 0
亂倒 11
亂倒垃圾 4
亂倫 57
亂做 7
亂兵 2
亂到 5
亂刺 0
亂動 26
亂叫 28
亂吃 11
亂吵 0
亂吼 0
亂吼亂叫 2
亂哄哄 5
亂喊 4
亂喊亂叫 1
亂嘈嘈 0
亂國 1
亂塗 10
亂子 16
亂寫 14
亂射 5
亂局 15
亂弄 3
亂彈琴 0
亂得 5
亂性 15
亂成 6
亂成一團 18
亂打 25
亂扯 3
亂抄 0
亂掉 32
亂推 0
亂搞 56
亂撲撲 0
亂擠 3
亂數 120
亂數 120
亂槍打鳥 1
亂民 3
亂流 54
亂源 18
亂烘烘 4
亂生 3
亂用 32
亂畫 14
亂的 92
亂相 7
亂真 21
亂碼 514
亂答 2
亂糟糟 6
亂罵 52
亂臣 1
亂臣賊子 3
亂花 4
亂花錢 15
亂葬崗 6
亂蓬蓬 0
亂行 2
亂視 12
亂設 4
亂試 4
亂說 61
亂說一通 3
亂說話 27
亂調 0
亂講 47
亂象 159
亂賊 3
亂走 23
亂起來 3
亂跑 49
亂跑亂跳 0
亂蹦亂跳 1
亂軍 6
亂邦 1
亂鋸 0
亂闖 7
亂離 4
亂飛 21
亂騎 0
亂髮 8
亂點鴛鴦 0
亂點鴛鴦譜 3
亂黨 7
亃 69
亄 110
亅 0
了 1259
了 125982
了不得 13
了不起 206
了不起 206
了了 75
了事 87
了債 2
了卻 0
了得 59
了得 59
了悟 46
了斷 41
了望 22
了案 4
了此殘生 0
了無 0
了無掛礙 0
了無新意 0
了無生機 0
了無生趣 0
了無痕跡 2
了無罣礙 0
了無長進 0
了然 56
了當 65
了結 145
了解 9267
了解到 219
了解情況 10
了願 7
亇 0
亇 0
予 4167
予 4167
予人 93
予以 2368
予取予求 40
争 0
事 52404
事不宜遲 16
事不關己 16
事主 23
事事 874
事事如意 32
事事物物 4
事件 7320
事例 82
事倍功半 31
事假 35
事先 1307
事兒 30
事兒 30
事出有因 16
事到如今 18
事到臨頭 5
事前 653
事前事後 7
事功 18
事務 1200
事務員 20
事務官 15
事務局 141
事務性 129
事務所 3059
事務股 29
事務長 41
事勢 1
事半功倍 128
事在人為 15
事宜 2188
事實 2573
事實上 257
事實勝於雄辯 6
事實審 1
事實證明 56
事已如此 0
事後 805
事必躬親 8
事情 3075
事態 77
事成之後 9
事故 1305
事業 13085
事業單位 549
事業家 8
事業心 14
事業部 521
事機 3
事權 142
事物 1210
事理 61
事由 254
事略 25
事發 0
事端 27
事與願違 29
事親至孝 5
事證 0
事變 82
事跡 47
事蹟 415
事過境遷 12
事關 60
事關重大 1
事隔多年 13
事非得已 0
事項 8900
事體 4
二 64877
二一添作五 0
二丁 38
二丁掛 9
二三 912
二上 85
二下 153
二世 114
二乘冪 0
二乙胺基 0
二二 516
二二八 625
二二八事件 0
二二八事變 16
二二八和平公園 0
二二得四 0
二二拍子 0
二五一十 0
二人 1393
二人份 7
二人同心 0
二付 4
二代 836
二任 127
二份 275
二伍 0
二伯 13
二伯母 0
二伯父 0
二位 512
二位數 59
二來 153
二信 63
二個 1251
二個人 52
二個月 628
二倍 155
二倍體 10
二價 28
二億 398
二億人 3
二億元 198
二元 278
二元論 13
二八 560
二八佳人 3
二八十六 0
二八年華 3
二具 21
二分 640
二分之一 0
二分法 31
二分鐘 34
二分音符 5
二副 18
二劃 44
二動 21
二十 16614
二十一條 430
二十一點 26
二十世紀 511
二十九 0
二十五史 26
二十四 635
二十四史 2
二十四孝 11
二十四小時 17
二十四節氣 27
二十天 41
二千 1700
二千人 36
二千個 11
二千元 158
二千塊 5
二半 5
二卵生 0
二叔 22
二叔叔 0
二句 32
二句話 6
二名 432
二員 11
二哥 107
二回 85
二回熟 2
二圓 1
二堂哥 0
二堂妹 0
二堂姊 0
二堂姊夫 0
二堂嫂 0
二堂弟 0
二塊 17
二塊錢 7
二壘 44
二壘手 23
二夜 111
二天 395
二女 24
二女一男 1
二女一男 1
二女二男 0
二女兒 16
二妹 19
二妹夫 0
二姊 18
二姊夫 0
二姐 51
二姑 1
二姑丈 0
二姑媽 2
二姑母 0
二姨 2
二姨媽 0
二嫂 23
二嬸 2
二嬸婆 0
二子 113
二子一女 0
二子一女 0
二子二女 0
二季 457
二專 1199
二專制 0
二尖瓣 0
二局 50
二局上半 1
二局下半 0
二屆 1732
二屆國代 5
二屆國代選舉 9
二帝三王 0
二幅 8
二年 4500
二年制 781
二年生 27
二年生植物 0
二年級 1196
二度 520
二度春 0
二度梅 0
二式 18
二弟 20
二弦 3
二形人 0
二心 15
二房東 11
二手 1609
二手煙 22
二手菸 0
二手貨 121
二技 0
二把 14
二把手 1
二指 15
二日 2826
二日份 1
二晚 27
二月 3959
二月份 310
二月革命 0
二期 1862
二期稻作 12
二杯 20
二板 16
二林 467
二枚 16
二桶 2
二棟 44
二極 14
二極管 1
二極體 401
二樓 1533
二樓住戶 0
二樣 11
二次 1982
二次函數 0
二次大戰 295
二次感染 0
二次方程 0
二次方程式 0
二次曲線 0
二次汙染 0
二次污染 0
二次破碎 0
二次電子 9
二次革命 2
二步 89
二段 5537
二段制 1
二段式 212
二段票 0
二氧 22
二氧化 88
二氧化氮 23
二氧化矽 24
二氧化硅 1
二氧化硫 140
二氧化碳 945
二氧化鈦 1
二氧化鉛 1
二氧化錳 11
二氯乙烷 33
二水 103
二流 21
二流人才 1
二流角色 1
二流角色 1
二滴 3
二爺 30
二版 360
二獎 56
二班 282
二班制 3
二甲 350
二甲胺基 0
二甲苯 45
二男 12
二男一女 1
二男一女 1
二男二女 0
二百 1863
二百二 72
二百五 122
二百人 31
二百個 9
二百元 173
二百塊 0
二硫化物 0
二硫化碳 27
二硫化鉬 1
二硫基丙 0
二碗 5
二碳化鈣 0
二科 171
二秒 16
二秒鐘 2
二種 613
二端 8
二等 213
二等兵 5
二等艙 33
二等親 1
二筒 0
二簧 1
二籠 0
二粒 8
二級 575
二級風 0
二綠 0
二老 22
二老板 0
二者 539
二者不可兼得 2
二者之間 32
二聯 51
二聯單 3
二聯式 56
二聲 23
二股 60
二胇基丙醇 0
二胡 127
二舅 2
二舅媽 0
二舅母 0
二艘 25
二萬 473
二萬五千 58
二萬人 21
二萬個 3
二萬元 140
二萬塊 1
二葉松 1
二虎相鬥 0
二號 1271
二號分機 0
二號機 78
二行程 52
二街 536
二表哥 2
二表妹 0
二表姊 0
二表姊夫 0
二表嫂 1
二表弟 0
二袋 0
二話 13
二話不說 68
二課 51
二諦 1
二讀 0
二象性 1
二路 2341
二路縱隊 0
二輪 143
二輪電影 8
二輪馬車 0
二通 9
二連 40
二進位 90
二進位制 0
二進制 15
二進制碼 0
二遍 6
二邊 8
二郎山 0
二郎腿 2
二部 407
二部制 4
二部合唱 5
二部曲 17
二里 7
二重 156
二重人格 0
二重唱 85
二重奏 26
二重性 2
二錢 19
二鍋頭 0
二鍋頭 0
二門 73
二集 220
二面角 1
二頁 525
二頂 0
二項式 2
二頭肌 17
二首 113
二首歌 2
二黃 27
二點 629
二點鐘 3
二黨制 0
二龍 16
二維 18
亍 8
于 2291
于七 5
于于 0
于今 6
于右任 19
于思 2
于思 2
于是 211
于是乎 3
于歸 3
于田 0
于飛 6
于餘曲折 0
亏 0
亏 0
云 1742
云云 93
云爾 5
互 5071
互不 126
互不 126
互不侵犯 8
互不侵犯條約 1
互不干涉 3
互不干涉內政 5
互不相讓 16
互交 6
互以 5
互保 10
互信 185
互信互諒 2
互信互重 3
互信無猜 0
互別苗頭 0
互利 126
互利性 0
互制 15
互加 1
互助 566
互助合作 57
互助會 50
互助組 2
互動 3442
互動式 1
互向 0
互唱 2
互問 1
互喊 2
互填 0
互市 3
互得 1
互惠 253
互惠協定 6
互惠待遇 0
互惠條約 0
互愛 20
互感 2
互成 4
互打 6
互指 47
互推 67
互換 142
互換性 5
互教互學 0
互教互學 0
互斥 26
互於 0
互有 60
互有勝負 1
互有權 0
互有輸贏 0
互毆 24
互派 7
互派大使 3
互添 0
互溶 11
互濟會 0
互為 107
互為因果 10
互為奧援 0
互為條件 0
互爭 12
互爭雄長 0
互生 27
互生葉 0
互用 12
互異 34
互發 1
互相 1942
互相切磋 33
互相幫助 18
互相標榜 0
互答 1
互罰 0
互致問候 0
互與 1
互補 209
互補式 16
互補色 4
互見 34
互訪 150
互調 58
互調 58
互諒 10
互謀 0
互謀其利 1
互證 1
互質 0
互轉 32
互通 141
互通性 33
互通有無 42
互通聲氣 2
互道 27
互道珍重 21
互選 108
互鬧 0
互點 0
亓 36
五 31344
五一 331
五一勞動節 17
五丁 11
五上 31
五下 18
五不救罪 1
五世 26
五世同堂 0
五丙 14
五中 20
五串 0
五乙 39
五五 409
五五波 20
五五波 20
五五草案 0
五人 746
五人份 0
五代 336
五代十國 6
五代同堂 30
五件 83
五件事 3
五份 44
五位 412
五位數 74
五保 0
五個 1244
五個人 341
五個月 188
五倍子 2
五倍子蟲 0
五倫 9
五停心觀 1
五億 370
五元 295
五兆 8
五光十色 64
五克 7
五內 5
五內俱焚 1
五內如焚 1
五兩 23
五公里 61
五具 7
五冊 57
五刀 5
五分 549
五分明兒 1
五分法身 1
五分錢 0
五分鐘 398
五刑 2
五劃 26
五力 1
五加皮 1
五匙 0
五十 5857
五十元 154
五十塊 6
五十步笑百步 7
五十肩 0
五千 1153
五千人 59
五千個 17
五千元 322
五千塊 12
五原 9
五反運動 1
五口 36
五口之家 3
五句 11
五只 9
五台 1664
五台山 22
五名 412
五呎 12
五周 3
五周年 32
五味 28
五味子 13
五味雜陳 16
五員 9
五哥 28
五四 683
五四時代 5
五四運動 33
五圈 4
五國 99
五圓 6
五坪 30
五場 24
五場 243
五堵 46
五塊 26
五塊錢 16
五壺 0
五夜 36
五大 765
五大明王 1
五大洲 45
五天 625
五姊 0
五姊夫 0
五姑 0
五子 44
五子棋 165
五子登科 3
五孔 2
五字 64
五季 8
五官 96
五官端正 4
五客 0
五家 267
五封 17
五專 762
五專生 23
五小 7
五小時 44
五尺 42
五局 52
五局上半 4
五局下半 0
五屆 805
五層 91
五層樓 33
五峰 308
五嶺 1
五嶽 16
五巷 106
五帝 21
五師 2
五席 57
五常 134
五幢 1
五年 9243
五年級 427
五年計劃 24
五床 6
五度 129
五度五關 5
五式 11
五弟 10
五弦琴 4
五彩 80
五彩奪目 0
五彩繽紛 129
五律 12
五成 357
五成新 6
五戒 34
五截 0
五戶 18
五打 7
五折 138
五招 7
五拳 1
五指 48
五指山 66
五支 45
五文 3
五文錢 2
五斗櫃 7
五斗櫥 0
五方 61
五方雜處 1
五日 2471
五日份 0
五星 182
五星上將 2
五星旗 13
五星紅旗 1
五星級 257
五時 278
五晚 41
五智 1
五更 30
五更 30
五更天 0
五月 4968
五月份 378
五月節 2
五月花 14
五服 1
五朔節 0
五期 782
五板 5
五枝 1
五架 13
五柳先生 1
五根 1
五格 3
五桶 1
五梏 0
五條 1994
五棓子 0
五棟 13
五棵 4
五樓 502
五樓住戶 0
五檔 10
五權 940
五權分立 9
五權憲法 30
五次 340
五次犯規 0
五步 31
五歲 676
五歲時 31
五段 884
五毒 23
五比丘 1
五毛 2
五毛錢 7
五洞 5
五洲 239
五湖四海 17
五無間業 1
五燈獎 0
五營 16
五爺 0
五版 63
五班 176
五瓶 1
五甲 354
五發 0
五百 1687
五百人 52
五百個 33
五百個 33
五百元 217
五百塊 9
五盒 1
五盒裝 0
五眼 1
五短 3
五短身材 3
五碗 2
五碼 50
五祖 7
五福 828
五福臨門 28
五科 45
五秒 94
五秒鐘 25
五種 388
五穀 98
五穀豐收 2
五穀雜糧 0
五筆 21
五筆劃 0
五筒 0
五箱 3
五節 85
五節課 3
五米 33
五粒 10
五粒裝 0
五級 217
五經 48
五線 54
五線譜 52
五罐 0
五聲 12
五股 514
五股鄉 924
五育並重 27
五胎 0
五胡 2
五胡之亂 0
五胡亂華 0
五胡十六國 5
五臟 60
五臟俱全 26
五臟六腑 36
五臟廟 0
五船 2
五色 102
五色旗 2
五色鳥 1
五花八門 222
五花大綁 6
五花爨弄 1
五花肉 10
五華 28
五華山 0
五萬 586
五萬人 47
五萬個 4
五萬元 303
五萬塊 2
五葉 28
五蘊 1
五蘊熾盛苦 1
五處 54
五號 855
五行 293
五行字 0
五行生剋 7
五街 136
五袋 3
五見 1
五角 27
五角大廈 67
五角大廈 67
五角形 10
五角星 3
五角錐 0
五角錢 1
五言 134
五言律詩 1
五言絕句 2
五言詩 4
五課 17
五識 1
五谷 55
五起 5
五路 418
五路縱隊 0
五輕 20
五輛 15
五輛車 3
五逆 1
五逆罪 1
五通 30
五週 136
五週年 104
五遍 3
五道 33
五邊 21
五邊形 0
五部 97
五里 35
五里霧 0
五里霧中 4
五金 2343
五金店 9
五金行 1
五銖 2
五銖錢 0
五錢 25
五門 1
五間 71
五關 8
五院 40
五陰 1
五陰世間 1
五隻 53
五集 108
五雙 3
五雷轟頂 4
五霸 4
五面 20
五面體 0
五音 89
五音不全 4
五頁 174
五頂 2
五項 375
五項原則 11
五頭 12
五顆 100
五題 25
五顏六色 35
五餐 4
五餐飯 0
五香 14
五香十味 0
五馬 16
五馬分屍 4
五體投地 22
五點 515
五點鐘 11
五黨 0
五齣 10
五齣戲 0
五龍 38
井 2911
井中求火 0
井中視星 0
井井有條 7
井口 19
井台 1
井噴 0
井壁 5
井岡山 1
井底 23
井底之蛙 18
井底蛙 2
井底銀瓶 0
井架 2
井水 75
井水不犯河水 3
井渫不食 0
井灌 0
井然 10
井然有序 45
井然有條 0
井田 4
井田制 0
井田制度 2
井筒 1
井臼 2
井蛙 4
井蛙語海 0
井陘 0
井鹽 5
亖 0
亗 0
亘 0
亘 0
亘 0
亙 301
亙 301
亙古 53
亙古 53
些 6168
些些 103
些兒 9
些兒 9
些小 247
些少 6
些微 617
些許 319
亜 0
亜 0
亝 0
亝 0
亝 0
亞 1395
亞 13956
亞丁 22
亞伯 128
亞伯拉罕 117
亞利安人 0
亞利桑納州 7
亞利桑那 40
亞利桑那州 19
亞原子性 0
亞喬木 2
亞太 4451
亞太地區 873
亞太營運中心 1
亞太營運中心 1
亞太經濟合作會議 1
亞太經濟合作會議 1
亞太金融中心 1
亞太金融中心 1
亞得里亞海 2
亞斯 100
亞東 295
亞東工專 117
亞東工業 7
亞東工業專科學校 85
亞東關係協會 19
亞歷山大 119
亞歷山大帝國 0
亞洲 7767
亞洲人 46
亞洲信託 26
亞洲各國 178
亞洲四小龍 1
亞洲四小龍 1
亞洲國家 259
亞洲地區 373
亞洲大學 1
亞洲大學 1
亞洲杯 13
亞洲運動會 1
亞熱帶 150
亞熱帶地區 25
亞熱帶氣候 16
亞特蘭大 231
亞犬 0
亞理斯多德 7
亞瑟 597
亞瑟王 12
亞當 243
亞當史密斯 2
亞當斯 43
亞盟 18
亞硝酸 28
亞硫酸 39
亞硫酸鹽 17
亞磷酸鹽 0
亞細亞 182
亞細亞洲 0
亞綱 9
亞美利加 4
亞美利加洲 0
亞美尼亞 28
亞美尼亞人 1
亞聖 95
亞聖 95
亞肩疊背 1
亞胺 42
亞裔 243
亞軍 227
亞軍 227
亞軍獎 2
亞軍獎牌 0
亞運 1971
亞運會 146
亞運選手 35
亞里士多德 9
亞里斯多德 10
亞鉛 22
亞鐵 25
亞非會議 0
亞馬孫河 1
亞馬遜 56
亞馬遜族 2
亞馬遜河 24
亞馬遜盆地 1
亞麻 45
亞麻布 9
亟 144
亟 144
亟待 247
亟欲 40
亟需 154
亟須 52
亡 11
亡 28962
亡佚 100
亡了 7
亡佚 107
亡兄 0
亡友 3
亡命 121
亡命之徒 7
亡命徒 0
亡國 1480
亡國之音 0
亡國奴 4
亡國滅種 1
亡國論 4
亡國論者 0
亡夫 13
亡失 1
亡妹 0
亡故 44
亡歿 0
亡母 3
亡父 0
亡羊補牢 40
亡者 48
亡逝 0
亡靈 27
亡魂 107
亢 176
亢奮 65
亢旱 3
亢直 0
亢進 161
亢龍有悔 0
亣 0
亣 0
亣 0
交 17420
交女友 100
交上 43
交上來 0
交上去 0
交下 5
交下來 0
交下去 0
交了 99
交互 265
交互作用 288
交互進行 0
交些 5
交交 14
交付 459
交付給 26
交代 316
交件 46
交保 111
交個 363
交出 209
交出來 17
交出去 1
交到 228
交割 854
交割股 5
交加 18
交卷 11
交卸 5
交叉 661
交叉口 111
交叉點 53
交友 4844
交友不慎 15
交合 25
交售 0
交回 44
交回來 0
交回去 0
交城 0
交大 2165
交好 23
交媾 10
交完 3
交對 2
交尾 11
交尾季節 0
交工 16
交差 17
交差了事 1
交帳 1
交往 662
交往甚密 0
交待 163
交情 93
交惡 13
交惡 13
交感 39
交感神經 129
交戰 207
交戰國 1
交戰圖 0
交戰狀態 1
交手 99
交拜 2
交接 396
交接點 2
交換 3927
交換價值 3
交換律 1
交換意見 377
交換所 17
交換條件 25
交換機 241
交換法 1
交換碼 0
交於 19
交易 9658
交易所 529
交易法 770
交易稅 240
交替 285
交替法 0
交會 217
交會法 11
交杯酒 0
交棒 0
交椅 35
交歡 21
交流 10958
交流道 1060
交流電 36
交流電力 0
交流電力機車 0
交涉 124
交淺言深 2
交清 20
交班 9
交界 119
交界處 62
交疊 24
交白卷 4
交相 37
交移 1
交納 9
交結 7
交給 807
交織 127
交織成 42
交織著 16
交與 34
交融 65
交角 13
交託 40
交誼 126
交誼 126
交誼廳 238
交誼廳 238
交誼會 1
交誼會 1
交談 829
交談起來 2
交警 14
交貨 506
交貨單 0
交貨期 30
交貨港 0
交賬 0
交趾 107
交通 8992
交通事故 239
交通信號 1
交通史 3
交通員 0
交通大學 66412
交通安全 537
交通局 179
交通工具 640
交通法庭 0
交通網 78
交通線 3
交通處 611
交通規則 43
交通警察 50
交通車 120
交通部 2568
交通部長 76
交通銀行 157
交通阻塞 16
交通阻塞 16
交遊廣闊 5
交運 23
交過 32
交過來 1
交過去 1
交道 1
交還 45
交還給 12
交配 127
交配季節 1
交鋒 99
交錯 384
交錯法 0
交際 100
交際應酬 29
交際舞 17
交際花 2
交際處長 0
交際費 7
交際賽 0
交集 1118
交響 77
交響曲 412
交響樂 59
交響樂團 810
交響樂章 1
交響樂隊 1
交響詩 33
交頭接耳 18
交點 44
亥 155
亦 14787
亦是 3000
亦且 7
亦即 1275
亦可 2670
亦同 664
亦因 129
亦將 622
亦對 60
亦對於 4
亦就 24
亦得 230
亦必 78
亦應 511
亦或 178
亦有 1305
亦步亦趨 23
亦比 22
亦然 82
亦當 23
亦經 51
亦需 97
亦需要 15
亦須 133
亨 10
亨 10
亨 1085
亨利 193
亨利四世 6
亨通 38
享 1998
享受 3420
享受到 196
享國 6
享壽 3
享年 65
享有 1745
享樂 87
享樂主義 15
享用 516
享福 21
享譽 96
京 2926
京兆尹 95
京劇 149
京口 3
京城 95
京山 16
京師 50
京師人 0
京府 0
京戲 9
京房 0
京滬 31
京漢鐵路 0
京片子 2
京畿 24
京胡 8
京腔 1
京華 490
京西 6
京話 0
京調 15
京都 565
亭 3999
亭亭 11
亭亭玉立 10
亭午 0
亭台 11
亭台樓閣 24
亭子 20
亭長 0
亮 2973
亮了 74
亮光 115
亮到 7
亮堂 1
亮底 0
亮度 544
亮晶晶 32
亮片 33
亮相 226
亮眼 0
亮票 4
亮節 3
亮著 57
亮話 2
亮起 113
亮閃閃 7
亮麗 568
亯 0
亰 0
亰 0
亱 0
亲 0
亲 0
亳 126
亳縣 0
亴 0
亶 93
亷 0
亹 1
亹 60
亹亹 1
亹亹不倦 1
亹亹不卷 1
人 120173
人一己百 0
人丁 24
人丁旺盛 1
人上有人 1
人下 61
人不像人 0
人不可貌相 3
人不為己 2
人不犯我 0
人不知鬼不覺 0
人不聊生 0
人世 5037
人中 279
人中之龍 0
人中豪傑 0
人主 33
人之將死 0
人之常情 19
人事 5038
人事不省 0
人事全非 7
人事室 890
人事異動 0
人事皆非 0
人事管理 203
人事處 175
人事行政 142
人事部 66
人云亦云 24
人亡政息 0
人亡政息 0
人亡物在 0
人亡物在 0
人人 1171
人人得而 3
人人有則 1
人人稱快 0
人人自危 11
人代 69
人代大會 1
人仰馬翻 12
人來人往 39
人們 3110
人倫 37
人傑 56
人傑地靈 1
人像 214
人像畫 14
人像靶 0
人兒 71
人兒 71
人前 71
人前人後 4
人剝削人 0
人力 5690
人力不足 70
人力資源 0
人力車 9
人去 431
人去樓空 15
人參 177
人參果 0
人參精 0
人參酒 0
人叢 11
人口 4156
人口分布 0
人口問題 0
人口增加 47
人口密度 0
人口普查 15
人口爆炸 7
人口結構 0
人口統計 82
人口論 4
人口販子 5
人口過剩 1
人口金字塔 0
人叫 116
人叫聲 0
人吃人 3
人各有志 4
人同此心 3
人名 292
人君 12
人命 217
人命關天 7
人和 772
人品 72
人員 23467
人善被人欺 0
人單勢薄 0
人單勢薄 0
人因 0
人困馬乏 1
人團法 0
人在 2976
人在何方 4
人在何處 4
人在江湖 0
人在福中不知福 0
人地 42
人地生疏 0
人堆 19
人士 5038
人壽 60
人壽保險 664
人外有人 11
人多 271
人多勢眾 0
人多口雜 1
人多嘴雜 0
人多手雜 0
人大 822
人如其名 0
人如其面 0
人妖 33
人子 80
人字架 0
人存政舉 1
人孤勢單 0
人定 16
人定勝天 30
人家 1755
人家說 136
人寰 12
人小鬼大 5
人山人海 27
人工 1997
人工化 11
人工受孕 18
人工受精 0
人工呼吸 86
人工孵化 1
人工島 0
人工工資 2
人工心臟 24
人工成本 0
人工授精 61
人工效率 1
人工智慧 1132
人工林 36
人工氣胸 0
人工流產 73
人工湖 28
人工甘味 9
人工色素 0
人工閱卷 0
人工降雨 0
人工養殖 0
人師 51
人形 109
人影 212
人影兒 0
人影兒 0
人後 58
人微言輕 2
人心 610
人心不古 10
人心不同 1
人心不同如其面 0
人心不足蛇吞象 3
人心大快 0
人心思漢 0
人心惟危 2
人心惶惶 45
人心所向 0
人心是肉做的 0
人心果 0
人心渙散 2
人心皇皇 0
人心莫測 0
人心隔肚皮 0
人心難測 0
人性 846
人性化 438
人性尊嚴 35
人性本善 11
人性本惡 8
人性論 20
人情 233
人情世故 9
人情之常 0
人情債 8
人情冷暖 0
人情味 65
人情味兒 0
人情味兒 0
人慾 1
人慾橫流 4
人我 101
人我不分 0
人所共知 2
人手 237
人手一冊 14
人手一冊 14
人手不足 21
人才 5694
人才出眾 0
人才外流 4
人才濟濟 18
人才輩出 20
人才難得 0
人數 9339
人文 3833
人文主義 41
人文地理 49
人文教育 0
人文景觀 0
人文科學 177
人文薈萃 58
人日 27
人時 229
人望 21
人本 505
人本主義 20
人本說 0
人本論 0
人材 129
人格 1012
人格保證 1
人格分裂 12
人格化 7
人格擔保 0
人格教育 28
人格權 0
人格測驗 0
人棄我取 0
人極 28
人模人樣 2
人模人樣 2
人樣 12
人機 262
人機一體 3
人機一體 3
人權 2912
人權主義 1
人權宣言 136
人權思想 11
人權節 1
人次 943
人欲 69
人死留名 1
人比人氣死人 0
人氏 18
人民 4978
人民之聲 0
人民之聲報 0
人民代表 7
人民代表大會 51
人民內部 0
人民內部矛盾 1
人民公社 21
人民公社化 0
人民出版社 31
人民團體 155
人民報 0
人民大會 2
人民大會堂 18
人民委員 2
人民委員會 1
人民幣 712
人民性 0
人民意志 4
人民戰爭 3
人民政府 102
人民日報 85
人民武裝 6
人民民主國家 0
人民民主專政 15
人民法庭 0
人民法院 120
人民警察 2
人民軍隊 1
人民陣線 2
人民黨 12
人氣 1203
人氣旺盛 5
人治 36
人流 14
人浮於事 4
人浮於食 0
人海 79
人海戰術 14
人海茫茫 0
人渣 24
人溺己溺 16
人滿 17
人滿之患 0
人滿為患 14
人潮 541
人為 1638
人為刀俎 0
人為地 1
人為萬物之靈 2
人為財死 0
人無千日 0
人無千日好 0
人無遠慮 1
人煙 41
人煙稀少 20
人爭一口氣 0
人爭一口氣 0
人牆 27
人物 6094
人物性格 12
人物畫 21
人物誌 66
人犯 114
人猿 24
人猿泰山 3
人瑞 40
人生 4523
人生七十 9
人生價值 12
人生哲學 110
人生在世 0
人生地不熟 1
人生如夢 5
人生如朝露 0
人生如萍 0
人生意義 12
人生朝露 0
人生舞台 0
人生觀 224
人生路不熟 0
人畜興旺 0
人皮 16
人盡可夫 3
人盡皆知 10
人相 1
人相學 1
人神共憤 6
人禍 68
人種 95
人種學 10
人稱 209
人稱代名詞 4
人窮志不窮 0
人窮志短 0
人算不如天算 0
人糞尿 0
人緣 137
人群 407
人群中 92
人老珠黃 0
人聲 107
人聲吵雜 1
人聲鼎沸 18
人肉 32
人肉市場 0
人肉市場 0
人脈 128
人腦 1
人臣 4
人蔘 99
人行道 137
人行陸橋 0
人行地下道 2
人要衣裝 6
人見 68
人見人愛 52
人言 26
人言可畏 2
人言嘖嘖 0
人言籍籍 0
人言鑿鑿 0
人言鑿鑿 0
人証 1
人話 270
人語馬嘶 0
人謀不臧 13
人證 39
人財兩失 0
人財兩得 0
人財兩空 0
人質 135
人質 135
人贓俱獲 5
人贓俱獲 5
人跡 26
人跡杳然 0
人跡杳然 0
人跡罕至 16
人身 454
人身依附 0
人身保護令 0
人身攻擊 172
人身自由 30
人造 386
人造冰 1
人造地球衛星 1
人造板 1
人造棉 8
人造橡膠 8
人造皮 10
人造絲 17
人造纖維 131
人造肥料 0
人造花 69
人造衛星 113
人造雨 1
人造革 1
人逢喜事精神爽 0
人道 314
人道主義 45
人道立場 0
人道立場 0
人道精神 24
人選 1123
人部 8
人間 1729
人間仙境 26
人間冷暖 0
人間喜劇 0
人間地獄 8
人間天堂 22
人間詞話 0
人際 647
人非木石 0
人非聖賢 0
人非草木 1
人面 38
人面桃花 2
人面獅身 9
人面獸心 3
人頭 334
人頭支票 0
人頭稅 4
人頭落地 0
人類 7601
人類史 7
人類史上 20
人類學 298
人類學家 36
人飢己飢 5
人馬 321
人馬宮 1
人馬座 39
人體 2228
人體上 11
人體工學 199
人體素描 5
人體解剖 0
人體解剖 0
人高馬大 4
人鬼 14
人鬼不分 0
人鬼殊途 0
人魚 158
人魚公主 16
人龍 19
亼 0
亾 0
亾 0
什 1888
什 1888
什 1888
什件 0
什物 4
什錦 98
什錦炒麵 9
什錦菜 1
什錦麵 0
什麼 20445
什麼 20445
什麼事 543
什麼事 543
什麼人 167
什麼人 167
什麼的 169
什麼的 169
仁 10171
仁人 41
仁人君子 58
仁人志士 18
仁信 29
仁兄 143
仁化 59
仁厚 24
仁君 7
仁和 390
仁壽 58
仁壽街 1
仁孝 6
仁宗 38
仁弟 2
仁德 1221
仁心 45
仁心仁術 5
仁愛 1040
仁愛國中 40
仁愛國小 76
仁愛為接物之本 2
仁愛路 1189
仁愛醫院 132
仁慈 128
仁慈心 1
仁政 42
仁昌 9
仁武 568
仁民愛物 0
仁波切 1
仁波切 1
仁義 209
仁義禮智信 2
仁義道德 13
仁者 26
仁者樂山 16
仁者無敵 0
仁至義盡 2
仁術 3
仁言利博 0
仁醫 91
仂 15
仃 243
仄 1268
仄聲 6
仄韻 1
仅 0
仆 161
仆倒 13
仇 137
仇 1379
仇人 77
仇家 30
仇怨 7
仇恨 170
仇敵 99
仇殺 14
仇視 16
仇隙 12
仈 14
仉 13
今 33812
今世 59
今人 67
今兒 11
今兒 11
今兒個 15
今兒個 15
今冬 14
今古奇觀 3
今名 14
今夏 106
今夕 13
今夜 314
今天 25469
今天在 504
今天是 657
今天會 39
今天有 123
今天能 28
今宵 61
今年 11343
今後 1487
今愁古恨 0
今文 12
今文學 0
今文經學 0
今日 15515
今日世界 27
今早 0
今明兩天 16
今昔 62
今是昨非 0
今晚 1023
今晨 90
今朝 46
今朝有酒今朝醉 5
今歲 5
今生 356
今生今世 28
今番 15
今非昔比 11
介 69
介 6952
介係詞 1
介入 1349
介冑 0
介在 22
介壽 507
介子 6
介意 168
介於 395
介殼 12
介紹 33686
介紹人 100
介紹信 7
介紹出來 3
介紹所 23
介繫詞 0
介蟲 0
介詞 13
介質 187
介達電位 0
介電常數 17
介面 3323
介面卡 295
仌 0
仌 0
仍 13938
仍來 3
仍依 0
仍到 3
仍去 4
仍在 2027
仍應 583
仍是 1534
仍會 361
仍有 2122
仍然 3067
仍然在 99
仍然是 347
仍然會 59
仍能 313
仍舊 272
仍舊在 16
仍舊是 52
仍舊會 5
仍舊有 8
仍要 174
从 0
从 0
从 0
从 0
仏 0
仏 0
仏 0
仐 0
仔 51
仔 5117
仔 5117
仔密 1
仔細 1508
仔細想 16
仔細看 80
仔細聽 14
仕 1036
仕女 100
仕女圖 9
仕官 0
仕宦 8
仕途 21
仕進 7
他 101399
他了 1
他上 56
他上來 9
他上去 1
他下 50
他下來 3
他下去 3
他不 891
他不 891
他不是 131
他不會 99
他人 4163
他來 249
他倆 86
他們 23296
他們倆 71
他們倆 71
他們的 4199
他儂 0
他先 85
他出 44
他出來 32
他出去 31
他到 184
他力 12
他動詞 0
他化自在天 1
他國 198
他在 1997
他媽的 87
他山之石 0
他律性 1
他心通 1
他想 307
他放 34
他方 107
他日 55
他是 2094
他有 734
他殺 59
他物 24
他的 11566
他種 48
他自己 670
他處 71
他要 495
他認為 670
他說 1944
他請 45
他過 35
他過來 3
他過去 42
他鄉 73
他鄉遇故知 25
他院 17
他項 169
仗 208
仗勢 2
仗勢欺人 4
仗恃 5
仗持 0
仗持著 0
仗義 37
仗義執言 26
仗義疏財 1
付 1869
付上 23
付下 3
付下去 1
付之一炬 33
付之一炬 33
付之一笑 0
付之一笑 0
付之東流 2
付之流水 1
付之闕如 0
付了 43
付予 50
付交 1
付來 0
付出 1493
付刊 0
付到 0
付印 29
付去 0
付在 4
付好 1
付帳 45
付得 11
付息 243
付托 2
付掉 1
付梓 27
付款 1273
付款人 17
付款單 9
付款日 6
付清 56
付現 35
付給 67
付與 30
付著 7
付訖 7
付託 21
付託重任 0
付諸 0
付諸實行 22
付諸東流 5
付諸流水 16
付諸行動 48
付費 963
付費者 11
付賬 4
付起 2
付足 2
付過 5
付郵 2206
付錢 138
付點 43
仙 5248
仙丹 87
仙人 74
仙人掌 143
仙人跳 3
仙侶 1
仙后座 14
仙境 381
仙女 85
仙女座 31
仙姑 9
仙子 187
仙家 8
仙岩 7
仙師 9
仙府 0
仙桃 14
仙樂 41
仙王座 6
仙界 13
仙童 5
仙翁 17
仙草 128
仙草蜜 0
仙藥 3
仙術 0
仙貝 0
仙逝 254
仙鄉 122
仙露明珠 0
仙風道骨 4
仙鶴 11
仚 10
仛 0
仜 8
仝 45
仞 45
仟 2626
仟克 5
仟卡 11
仠 0
仡 7
仢 0
代 7495
代之以 27
代人 55
代付款 0
代代 42
代代相傳 30
代代相承 1
代價 580
代償 57
代出 19
代勞 57
代取 3
代名 22
代名詞 89
代售 135
代售處 1
代唱 2
代唱者 0
代墊 37
代天巡狩 0
代夫 11
代子 25
代寫 11
代工 983
代序 68
代庖 2
代打 64
代打者 0
代換 38
代換品 0
代收 283
代收款 12
代數 169
代數和 3
代數學 13
代數式 3
代書 645
代書事務所 551
代替 790
代替品 12
代查 5
代步 112
代母 1
代溝 70
代演 1
代為 798
代為說項 0
代父 12
代班 0
代理 6433
代理人 1022
代理商 906
代理官 0
代理店 29
代理行 24
代用 58
代用品 46
代碼 3581
代筆 19
代管 116
代管人 0
代簽 4
代簽人 1
代糖 0
代罪羔羊 38
代耕 14
代與 112
代號 4538
代行 121
代表 11702
代表人 361
代表人物 38
代表作 211
代表團 363
代表大會 391
代表性 579
代表會 136
代表權 50
代表隊 429
代言 130
代言人 309
代訂 86
代詞 7
代課 365
代課老師 62
代謝 764
代謝作用 23
代謝率 14
代謝能 11
代議 27
代議制 5
代議政治 8
代購 882
代跑 5
代跑員 0
代跑者 1
代辦 597
代辦所 18
代辦處 17
代金 96
代銷 186
代銷商 5
代錄 3
代電 12
代領 26
代點 4
令 10125
令人 5312
令人噴飯 17
令人驚駭 2
令人髮指 35
令兄 9
令出如山 1
令出惟行 0
令友 0
令叔 0
令名 0
令坦 0
令堂 12
令妹 10
令姊 1
令嬡 3
令孫 4
令尊 32
令尊大人 0
令岳 0
令師 13
令弟 7
令徒 1
令愛 7
令旗 0
令望 0
令正 5
令牌 8
令瑗 0
令甥 1
令祖 1
令箭 3
令節 1
令行 8
令行禁止 0
令親 2
令譽 2
令郎 4
令酒 1
以 133539
以你 20000
以上 26530
以下 16458
以不變應萬變 14
以來 3881
以供 869
以便 2341
以候 3
以候通知 0
以假亂真 0
以假謬真 0
以偏概全 16
以備 174
以備不時之需 25
以價制量 0
以儆效尤 0
以免 2472
以內 860
以其 1098
以前 6347
以力服人 0
以北 205
以南 192
以卵投石 0
以卵擊石 3
以原就被 1
以原就被原則 1
以及 21485
以及人之老 3
以古況今 0
以古非今 0
以右 22
以和為貴 0
以外 2033
以多勝少 0
以太 33
以夷制夷 5
以寡擊眾 2
以寡敵眾 1
以小吃大 0
以左 49
以往 2156
以後 6448
以德報怨 23
以德服人 4
以怨報德 2
以攻為守 3
以故 20
以文會友 14
以是 54
以智取勝 4
以暴制暴 14
以暴易暴 1
以有限 39
以期 932
以東 121
以柔克剛 12
以次 20
以此 875
以此為戒 2
以此為限 3
以此類推 61
以毒攻毒 2
以求 382
以淚洗面 1
以為 3590
以為在 38
以為是 258
以為會 13
以牙還牙 11
以理服人 1
以直報怨 0
以眼還眼 4
以示 207
以示公平 1
以策 10
以策安全 18
以管窺天 0
以簡馭繁 0
以至 263
以至於 258
以致 1437
以色列 988
以色列人 215
以西 147
以訛傳訛 20
以詞害意 1
以貌取人 10
以貨易貨 3
以資 175
以身 54
以身作則 83
以身相許 12
以身試法 13
以農立國 7
以近 54
以退為進 11
以進為退 0
以逸待勞 8
以遠 20
以鄰為壑 0
以防 466
以防萬一 25
以降 120
以靜制動 5
以點帶面 0
仦 0
仦 0
仧 0
仧 0
仧 0
仨 9
仩 12
仮 0
仮 0
仮 0
仯 0
仯 0
仰 705
仰事俯畜 0
仰人鼻息 0
仰仗 26
仰仗著 1
仰光 27
仰天 59
仰天一望 0
仰天一望 0
仰天長嘯 5
仰式 21
仰德大道 58
仰慕 68
仰慕者 23
仰望 201
仰止 11
仰看 6
仰看著 0
仰臥 72
仰臥起坐 26
仰角 73
仰賴 322
仰賴著 1
仰起 35
仰靠 2
仰靠著 0
仰面 13
仰韶 3
仰韶文化 7
仰頭 23
仰頭一看 2
仰頭一看 2
仰高 2
仱 78
仱 78
仲 5128
仲介 1910
仲介公司 214
仲兄 0
仲冬 2
仲呂 0
仲夏 49
仲夏夜 24
仲夏夜之夢 32
仲尼 20
仲春 2
仲秋 60
仲裁 347
仲裁人 17
仳 9
仳離 2
仴 61
仵 194
仵作 0
件 25456
件數 575
价 194
价 194
仸 0
仹 0
仺 0
任 1455
任 14100
任事 35
任人 53
任人宰割 11
任人擺佈 10
任何 20497
任何事 168
任何人 864
任便 3
任俠 3
任免 84
任內 359
任其 86
任務 2892
任務在身 0
任勞任怨 25
任命 445
任命狀 0
任性 121
任意 1613
任憑 111
任教 667
任教於 192
任教者 9
任期 1158
任為 32
任用 698
任由 195
任由擺佈 0
任職 1614
任課 430
任課老師 222
任選 0
任重道遠 25
仼 0
仼 0
份 14142
份內 40
份地 40
份外 31
份子 459
份子 459
份量 278
份額 10
仾 0
仿 651
仿佛 202
仿傚 10
仿冒 339
仿冒品 50
仿制 1
仿古 78
仿單 47
仿宋 14
仿宋體 4
仿效 102
仿照 81
仿生 8
仿生學 2
仿製 25
仿製品 4
仿造 18
仿造品 0
伀 219
企 5844
企仰 1
企劃 2683
企劃人 9
企劃廳 17
企劃書 2683
企劃案 2683
企圖 1135
企圖心 190
企慕 0
企望 14
企業 59871
企業化 154
企業基金 15
企業家 374
企業界 822
企業管理 2276
企業管理系 67
企業管理費 1
企業財務 192
企機 3
企求 45
企盼 91
企禱 1
企管 1253
企管科 178
企管系 662
企鵝 768
企鵝牌 0
伂 199
伃 0
伄 93
伅 197
伆 0
伇 0
伈 216
伉 248
伉儷 138
伊 4643
伊人 67
伊利亞特 1
伊利運河 0
伊始 32
伊尹 42
伊底帕斯 12
伊底帕斯情結 1
伊拉克 944
伊拉克人 5
伊斯坦堡 113
伊斯蘭 68
伊斯蘭堡 0
伊斯蘭教 58
伊朗 249
伊朗人 1
伊朗高原 4
伊東 35
伊比利半島 17
伊犁河 1
伊甸園 152
伊甸樂園 2
伊紅 20
伊索 42
伊莉莎白 51
伋 20
伌 0
伍 4097
伍佰 97
伍子胥 24
伍廷芳 0
伍德合金 0
伎 352
伎倆 33
伏 552
伏下 8
伏下來 0
伏下去 1
伏倒 2
伏兵 24
伏到 0
伏在 63
伏地 14
伏地挺身 26
伏天 0
伏妖 1
伏安 12
伏屍 6
伏屍遍野 0
伏屍遍野 0
伏擊 14
伏案 7
伏法 11
伏流 15
伏爾加河 1
伏特 236
伏特加 64
伏特計 6
伏生 2
伏筆 57
伏罪 1
伏羅希洛夫 0
伏羲 31
伏羲氏 4
伏臥 3
伏著 13
伏藏 1
伏虎 6
伏襲 0
伏誅 1
伏貼 2
伏身 7
伏首 1
伏首稱臣 1
伐 418
伐 418
伐區 3
伐區 3
伐善 0
伐善 0
伐木 119
伐木 119
伐木區 1
伐木區 1
伐木業 1
伐木業 1
休 3352
休假 560
休假日 37
休克 170
休兵 33
休刊 1
休到 1
休士頓 236
休學 228
休學在家 0
休市 57
休怪 16
休息 2193
休息室 94
休息時間 105
休息站 1
休想 54
休憩 281
休戚 1
休戚相關 6
休戚與共 10
休戰 7
休整 1
休旅車 0
休書 3
休會 38
休會期 1
休會期間 18
休業 11
休業式 13
休止 21
休止符 41
休眠 147
休眠期 13
休耕 46
休耕地 1
休課 4
休謨 8
休閑 56
休閒 15595
休閒活動 550
休養 67
休養生息 11
伒 60
伓 56
伔 20
伕 163
伕子 1
伖 0
伖 0
众 0
优 253
伙 434
伙伕 0
伙伴 1230
伙同 0
伙夫 2
伙房 5
伙計 37
伙食 88
伙食費 19
会 0
会 0
会 0
会 0
伛 0
伛 0
伜 0
伝 140
伢 156
伢子 0
伨 0
伨 0
伬 99
伭 2
伮 0
伯 15
伯 44
伯 18230
伯仲 17
伯仲之間 22
伯伯 290
伯公 75
伯利 44
伯利恆 45
伯利茲 2
伯勞 110
伯勞鳥 7
伯婆 4
伯德 3
伯恩 94
伯明罕 104
伯根地 0
伯樂 140
伯母 38
伯爵 164
伯爵夫人 9
伯父 31
伯父母 0
伯爾 17
伯爾尼 0
伯祖 0
伯祖母 0
估 122
估 1228
估價 875
估價單 61
估到 10
估單 7
估成 11
估摸 0
估測 54
估稅員 0
估算 461
估衣 0
估衣 0
估計 1674
估計值 30
估計量 10
估量 23
伱 0
伲 0
伳 6
伴 1138
伴侶 386
伴偶 0
伴兒 4
伴兒 4
伴唱 93
伴唱機 14
伴奏 235
伴娘 10
伴手 1
伴舞 14
伴讀 28
伴送 0
伴郎 7
伴隨 346
伴隨著 223
伴音 0
伴食 0
伵 0
伶 713
伶人 6
伶仃 2
伶俐 38
伶官 6
伶牙俐齒 7
伷 0
伸 1720
伸上 1
伸上來 0
伸上去 0
伸下 3
伸下來 1
伸下去 1
伸伸 1
伸伸手 0
伸伸腳 0
伸來 2
伸來伸去 0
伸入 71
伸冤 14
伸出 412
伸出來 24
伸出去 6
伸到 33
伸去 1
伸及 1
伸向 48
伸回 0
伸回來 0
伸回去 0
伸在 8
伸展 314
伸展出 0
伸展出去 0
伸展到 10
伸平 2
伸延 6
伸張 63
伸張正義 25
伸懶腰 8
伸手 377
伸手不見五指 15
伸港 254
伸直 79
伸縮 257
伸縮尺 0
伸縮性 22
伸縮自如 7
伸腰 2
伸腳 2
伸腿 4
伸舌頭 1
伸舌頭 1
伸著 7
伸訴 3
伸起 1
伸起來 0
伸進 30
伸進來 2
伸進去 9
伸過 8
伸過來 5
伸過去 6
伸長 145
伸開 19
伹 0
伺 185
伺 185
伺候 75
伺候 75
伺察 0
伺服 5158
伺服器 0
伺機 128
伺機而動 5
伺隙 0
伻 11
似 2466
似乎 4876
似乎是 459
似地 142
似懂非懂 23
似星體 0
似是 145
似是而非 57
似曾 10
似曾相識 67
似有若無 17
似的 666
似非而是 0
伽 276
伽 276
伽利略 1
伽利略 65
伽瑪射線 1
伽藍 18
伾 5
伿 1
佀 0
佁 10
佂 0
佂 0
佃 134
佃婦 0
佃客 0
佃戶 8
佃權 32
佃租 3
佃農 33
佄 0
佄 0
佅 0
但 63849
但丁 17
但凡 11
但在 2602
但從 275
但憑 17
但是 21957
但是在 1025
但是有 215
但書 130
但有 918
但求 67
但求無過 0
但無 267
但說無妨 0
但須 173
但願 348
但願如此 4
但馬 16
佇 99
佇候 2
佇列 39
佇立 91
佇足 19
佈 1108
佈下 30
佈告 334
佈告欄 3765
佈哨 0
佈局 364
佈崗 4
佈施 22
佈景 466
佈滿 108
佈線 73
佈置 687
佈署 67
佈道 286
佈道大會 2
佈達 37
佈達式 0
佈防 5
佈陣 9
佈雷 22
佈雷艇 1
佈雷隊 0
佉 0
佊 0
佋 0
佌 0
位 2481
位似中心 0
位似變換 0
位元 1618
位在 516
位址 13004
位址欄 7
位子 196
位尊 5
位尊望重 0
位居 224
位居要津 1
位差 9
位數 240
位於 5234
位極人臣 0
位標 3
位次 7
位相 59
位移 203
位移電流 4
位置 7841
位能 62
位處 129
低 7555
低三下四 1
低下 163
低下來 1
低下去 2
低下頭來 11
低下頭去 11
低了 50
低人 16
低人一等 4
低人一等 4
低估 122
低低 35
低低的 11
低俗 32
低值易耗 0
低值易耗品 0
低價 1222
低價買進 2
低價賣出 3
低價高報 0
低凹 2
低分 46
低到 31
低劣 36
低劣性 0
低合金鋼 12
低周波 0
低周波 0
低唱 2
低回 7
低地 29
低垂 30
低壓 388
低姿勢 6
低姿態 24
低平 12
低年 11
低年級 314
低廉 337
低得 14
低微 18
低成本 216
低收入 375
低於 1256
低柔 1
低標 0
低檔 194
低欄 2
低欄賽 0
低段 5
低氣壓 82
低沉 58
低洼 0
低消耗 8
低溫 530
低溫氣候 0
低溫特報 7
低潮 194
低潮期 23
低濕 11
低熱 10
低熱量 38
低產 5
低產期 0
低產量 2
低的 240
低盪 4
低眉 2
低矮 31
低碳鋼 15
低穹 0
低空 65
低空飛行 2
低窪 39
低窪地區 29
低端 0
低等 30
低等動物 12
低等植物 3
低等生物 5
低筋麵粉 1
低級 121
低緩 4
低耗 36
低聲 166
低聲下氣 8
低胸 11
低胸裝 2
低能 80
低能兒 6
低脂 1
低腰 8
低落 259
低著 6
低著頭 72
低處 26
低血壓 111
低血糖 165
低語 27
低調 227
低賤 7
低迷 438
低迷不振 33
低速 118
低週波 0
低週波 0
低陷 3
低階 298
低階語言 22
低音 318
低音管 50
低音部 7
低頭 266
低頭不語 5
低頻 125
低飛 12
低飛球 0
低首 3
低首下心 0
低點 152
住 8169
住下 8
住下來 5
住下去 6
住個 2
住到 18
住口 13
住嘴 3
住在 1602
住地 53
住址 2312
住外 14
住宅 3308
住宅區 398
住家 618
住宿 3412
住宿生 36
住寺 0
住居 71
住屋 170
住得 40
住得 40
住慣 4
住戶 983
住房 120
住所 246
住手 29
住持 73
住校 80
住校生 23
住民 224
住的 363
住者 22
住著 61
住處 157
住診 16
住起 1
住起來 3
住院 1353
住院手續 15
住院費 4
住院費用 38
住院醫師 663
佐 1183
佐使 3
佐料 70
佐理 18
佐理員 13
佐籐 5
佐証 2
佐證 96
佐餐 13
佑 1703
佒 3
体 17
体 1745
佔 42574
佔上風 25
佔下 4
佔了 311
佔位 23
佔住 33
佔佔 0
佔便宜 22
佔個 5
佔優勢 45
佔先 2
佔光 6
佔到 68
佔去 26
佔取 1
佔在 2
佔地 337
佔城 1
佔多數 21
佔好 3
佔得 13
佔得 13
佔掉 14
佔據 208
佔有 541
佔有慾 27
佔有權 0
佔有率 833
佔滿 29
佔為 6
佔為己有 5
佔用 346
佔盡 32
佔線 17
佔著 23
佔起 0
佔起來 0
佔過 0
佔過來 0
佔過去 0
佔領 222
佔領區 4
佔領地 0
佔領者 0
佔領軍 0
何 9900
何不 401
何事 144
何人 187
何仙姑 5
何以 494
何以自處 0
何以見得 13
何來 143
何其 138
何去何從 176
何嘗 126
何國 51
何在 383
何地 98
何堪 3
何奇不有 0
何如 68
何妨 125
何干 29
何年 12
何年何月 16
何年何月何日 2
何德何能 7
何必 592
何必如此 8
何必當初 11
何患 5
何患無詞 1
何承天 1
何故 79
何方 195
何日 79
何時 1869
何時何地 28
何晏 3
何曾 27
何月 37
何月何日 1
何樂不為 17
何樂而不為 31
何止 26
何況 887
何況是 91
何為 141
何為 141
何獨 6
何用 84
何種 1581
何等 274
何者 1235
何能 40
何至於 0
何至於此 2
何若 12
何苦 125
何苦來哉 4
何處 2011
何許人 53
何謂 1828
何足 28
何足掛齒 4
何足輕重 0
何隊 1
何難之有 3
何需 58
何須 77
何首烏 27
佖 4
佗 15
佗 15
佗 15
佘 60
佘太君 1
余 2514
余上沅 1
佚 68
佚失 6
佛 659
佛 6597
佛事 37
佛佗 3
佛像 182
佛光 273
佛光寺 5
佛光山 344
佛典 176
佛力 10
佛勞倫斯 0
佛勞倫斯人 0
佛國 38
佛圖澄 1
佛土 11
佛堂 78
佛塔 22
佛學 1552
佛學界 4
佛家 76
佛家弟子 0
佛家語 1
佛寶 17
佛寺 105
佛山 72
佛得角 0
佛性 90
佛手柑 17
佛提樹 0
佛教 3805
佛教徒 157
佛殿 12
佛法 881
佛法徒 0
佛法無邊 3
佛洛伊德 63
佛洛依德 50
佛洛依德派 0
佛牙 0
佛眼 1
佛祖 84
佛經 252
佛羅稜薩 0
佛羅里達 153
佛羅里達州 67
佛老 1
佛號 61
佛要金裝 3
佛誕 7
佛語 1
佛門 110
佛門弟子 8
佛陀 576
佛音 1
佛頭 2
佛骨 14
佛龕 1
作 27
作 27
作 27179
作主 84
作亂 17
作了 468
作事 59
作人 983
作伴 33
作保 15
作假 31
作偽 4
作價 54
作准 8
作出 702
作出來 5
作到 80
作功 16
作勢 23
作古 14
作品 14753
作品展覽 60
作嘔 22
作圖 91
作圖器 0
作坊 1344
作壁上觀 7
作多 192
作夢 140
作夢也沒想到 1
作奸犯科 8
作好 244
作姦犯科 2
作威作福 6
作嫁 3
作孽 8
作官 10
作客 55
作家 3034
作對 67
作工 98
作廢 68
作弄 29
作弊 190
作怒 0
作怪 71
作息 179
作息時間 25
作息時間表 6
作惡 30
作惡多端 8
作愛 49
作態 8
作成 526
作戰 733
作戲 9
作手腳 3
作揖 13
作文 745
作文簿 11
作文課 9
作料 7
作曲 421
作曲家 693
作月子 4
作東 8
作案 126
作梗 8
作業 17854
作業員 77
作業本 7
作業簿 22
作業系統 3028
作樂 25
作法 1867
作法自斃 0
作活 6
作準 4
作為 7992
作物 761
作生意 23
作用 3820
作用力 57
作用點 2
作畫 177
作痛 5
作祟 51
作票 1
作禮拜 4
作秀 82
作筆記 17
作答 266
作繭自縛 5
作罷 46
作者 22976
作聲 14
作莊 2
作記號 5
作証 24
作詞 137
作詞家 6
作詩 37
作證 195
作賤 4
作賤自己 4
作陪 18
作難 6
作響 37
作風 281
作飯 5
作鬼臉 0
佝 12
佝 12
佝僂 10
佝僂病 2
佞 23
佞人 4
佞佛 0
佞口 0
佟 102
你 89941
你作 100
你丟我撿 0
你來 624
你來我往 29
你們 8577
你們 8577
你們的 1350
你在 1769
你好 1227
你好嗎 75
你媽 76
你媽的屄 29
你將 671
你想 1868
你愛 235
你我 971
你我他 71
你找 166
你推我讓 0
你搞 38
你是 3896
你是為了 7
你會 1651
你有 1913
你死 42
你死我活 17
你爸 76
你的 15965
你等 68
你老 85
你自己 758
你藏 7
你行 57
你要 2526
你說 1457
你講 76
你轉 9
你追我趕 0
佡 13
佢 106
佣 89
佣 89
佣人 12
佣工 0
佣工 0
佣金 166
佣金 166
佤 58
佧 7
佩 2146
佩刀 16
佩劍 15
佩帶 30
佩帶在 1
佩帶著 2
佩戴 1
佩掛 10
佩服 489
佩槍 1
佩玉 13
佩著 5
佪 300
佫 361
佬 291
佭 0
佮 127
佯 117
佯死 0
佯狂 0
佯稱 0
佯笑 0
佯裝 29
佯言 0
佰 264
佰 2645
佱 0
佱 0
佱 0
佱 0
佲 0
佲 0
佳 37000
佳人 311
佳作 791
佳作獎 54
佳偶 55
佳偶天成 2
佳句 25
佳城 4
佳境 19
佳客 2
佳日 3
佳景 8
佳期 10
佳期已近 0
佳木斯 30
佳木斯港 0
佳構 9
佳樂水 27
佳節 163
佳篇 1
佳績 251
佳茗 0
佳評 31
佳評如潮 15
佳話 15
佳賓 22
佳釀 24
佳里 696
佳音 170
佳餚 170
佳餚美酒 3
佳麗 116
佴 259
併 2258
併入 323
併列 23
併到 2
併合 13
併吞 67
併吞下 0
併在 8
併成 18
併排 45
併攏 18
併案 107
併案處理 1
併案處理 1
併為 20
併發 196
併發症 493
併科 212
併購 0
併進 5
佶 326
佷 824
佸 342
佹 202
佺 23
佻 15
佻 15
佻巧 0
佻薄 0
佻達 0
佼 496
佼佼者 59
佽 1811
佾 39
佾生 1
佾舞 14
佾舞生 0
使 352
使 35219
使上 49
使不得 3
使不得 3
使令 3
使出 205
使出來 10
使出渾身解數 49
使到 16
使勁 39
使勁兒 0
使勁兒 0
使君 32
使命 571
使命感 131
使喚 30
使團 79
使壞 18
使女 33
使役 0
使徒 625
使徒行傳 10
使得 6573
使性子 0
使氣 13
使然 97
使用 63028
使用人 219
使用價值 36
使用權 371
使用法 53
使用率 382
使用者 15989
使用費 169
使相 28
使眼色 3
使節 59
使節團 7
使者 259
使脾氣 1
使臣 9
使館 93
使館區 4
侀 327
侁 206
侂 247
侃 161
侃侃而談 29
侃直 0
侄 211
侄女 3
侄子 4
侅 201
侅 201
來 0
來 69889
來上 74
來下 43
來不及 371
來不得 0
來世 77
來之 154
來之不易 5
來了 2342
來人 115
來件 10
來來 666
來來回回 25
來來往往 41
來信 4600
來個 228
來函 848
來到 3511
來勁 2
來勢 18
來勢洶洶 80
來去 222
來去分明 0
來去自如 15
來台 1684
來台訪問 81
來回 604
來回票 10
來安 10
來客 113
來年 65
來往 272
來得 565
來得及 114
來復 11
來意 29
來意不明 0
來攻 16
來文 103
來日 108
來日方長 19
來日無多 2
來書 14
來此 343
來歷 84
來歷不明 16
來歸 19
來源 5914
來照 16
來犯 22
來生 97
來由 82
來的 5283
來看 3226
來福 52
來福槍 10
來福線 6
來稿 340
來給 305
來者 76
來者不善 7
來者不善 7
來者不善善者不來 1
來者不拒 22
來者不拒 22
來者可追 4
來聽 120
來臨 1035
來自 8668
來自於 1077
來華 247
來華訪問 31
來著 26
來處 40
來複 0
來襲 0
來見 85
來訪 2034
來試 42
來話顯示 0
來說 3387
來賓 3455
來賓席 2
來賓致詞 15
來路 22
來路不明 48
來車 33
來過 187
來電 1095
來電顯示 0
來頭 22
來頭不小 2
來體 18
來點 120
來龍去脈 957
來龍去脈 957
侇 158
侈 99
侈言 5
侈談 3
侈談和平 0
侈靡 2
侉 166
侉 166
侊 0
侊 0
例 5183
例假 27
例假日 255
例句 53
例外 510
例如 7132
例如說 46
例如說是 0
例子 1443
例文 0
例會 175
例行 515
例行公事 13
例行工作 21
例行決策 0
例言 3
例証 7
例語 0
例證 91
例題 80
侌 0
侌 0
侍 297
侍中 3
侍候 29
侍僕 0
侍奉 22
侍女 36
侍從 28
侍從官 0
侍從長 0
侍童 0
侍者 88
侍衛 33
侍衛長 4
侍郎 19
侍養 0
侎 0
侏 154
侏儒 70
侏儒症 4
侏儸紀 127
侐 796
侑 108
侒 314
侓 0
侔 241
侕 345
侖 282
侗 546
侘 242
侙 0
侚 531
供 1006
供 10069
供不應求 132
供住 7
供佛 28
供作 0
供作 0
供出 42
供品 13
供售 17
供在 6
供堂 0
供奉 209
供應 4414
供應國 38
供應地 6
供應站 83
供暖 1
供暖系統 0
供桌 19
供水 227
供水系統 47
供求 25
供求失調 0
供求平衡 2
供求律 0
供狀 0
供神 1
供稱 30
供糧 0
供給 1042
供給制 4
供職 7
供膳宿 8
供製 21
供詞 16
供認 5
供說 0
供述 18
供過於求 156
供銷 87
供銷中心 1
供銷合作 0
供銷合作社 0
供電 245
供電系統 28
供需 550
供需失調 32
供需平衡 16
供養 257
供養 257
供養 257
供養父母 2
侜 96
依 34000
依人籬下 0
依仗 4
依侍 1389
依依 120
依依不捨 69
依偎 58
依偎 58
依傍 6
依傍在 1
依允 4
依利諾 8
依存 67
依存度 44
依存性 9
依山傍水 24
依序 781
依序是 33
依從 25
依從性 0
依循 266
依循著 18
依戀 73
依據 6776
依杖 0
依樣 40
依樣畫葫蘆 24
依次 194
依次是 6
依此類推 51
依歸 81
依法 2077
依法嚴懲 2
依法嚴辦 4
依法行事 26
依法辦理 79
依然 1505
依然如故 0
依然故我 9
依然是 134
依照 3048
依照著 3
依物 15
依稀 91
依稀記得 13
依約 112
依老賣老 5
依舊 657
依舊在 33
依舊是 82
依舊會 4
依託 6
依賴 957
依賴度 17
依賴性 61
依賴成性 0
依附 410
依附在 27
依附於 13
依靠 376
依靠在 4
依靠著 4
依順 3
依順著 1
依麗莎白 4
侞 106
侟 0
侠 0
侠 0
価 0
価 0
侢 0
侣 0
侮 46
侮弄 0
侮慢 1
侮罵 0
侮蔑 8
侮辱 101
侮辱性 1
侯 2447
侯爵 20
侯爵夫人 0
侯爺 4
侯門如海 0
侯門深似海 1
侰 0
侱 0
侱 0
侲 19
侳 39
侴 0
侵 840
侵佔 85
侵佔到 2
侵佔罪 1
侵併 0
侵入 451
侵入體 0
侵凌 3
侵占 81
侵占到 0
侵占罪 7
侵吞 15
侵吞公款 1
侵奪 6
侵害 1202
侵害到 15
侵擾 39
侵擾到 0
侵權 0
侵犯 652
侵犯到 50
侵略 203
侵略國 1
侵略性 25
侵略成性 0
侵略戰爭 6
侵略者 25
侵蝕 331
侵蝕作用 13
侵蝕到 8
侵蝕性 5
侵襲 287
侵襲到 0
侵越 1
侶 1860
侷 29
侷促 25
侷限 298
侸 0
侸 0
侹 0
侺 0
侻 0
侼 0
侽 0
侾 0
便 1047
便 6146
便不能 100
便中 12
便利 1798
便利商店 346
便去 37
便可 3265
便名 0
便在 245
便士 29
便壺 0
便宜 2025
便宜行事 11
便宜貨 22
便尿 0
便帽 13
便應 34
便所 21
便捷 428
便於 460
便是 1764
便會 787
便有 311
便服 37
便條 32
便條兒 0
便條兒 0
便條紙 67
便橋 6
便步 70
便步走 0
便民 953
便民措施 46
便池 4
便溺 4
便溺 4
便當 434
便當盒 30
便祕 83
便祕 83
便秘 254
便箋 57
便給 21
便能 407
便菜 0
便藥 2
便衣 20
便衣人員 1
便衣偵探 0
便衣警察 2
便衣隊 0
便裝 10
便要 160
便車 60
便辟 0
便道 67
便門 2
便鞋 11
便飯 4
便餐 8
俀 84
俁 13
係 11423
係數 478
係為 315
促 1671
促使 1446
促性腺激素 5
促成 775
促狹 5
促膝 14
促膝談心 0
促請 126
促進 4911
促進會 408
促進派 0
促銷 1906
俄 698
俄 698
俄亥俄 173
俄亥俄州 65
俄人 0
俄人 0
俄克拉荷馬州 0
俄共 18
俄共 18
俄勒崗 0
俄喀河 0
俄國 540
俄國 540
俄國人 24
俄國人 24
俄國共產黨 0
俄國共產黨 0
俄國話 0
俄國話 0
俄帝 0
俄帝 0
俄式 4
俄式 4
俄文 63
俄文 63
俄文系 16
俄文系 16
俄文組 6
俄文組 6
俄皇 2
俄皇 2
俄羅斯 176
俄羅斯 1765
俄羅斯人 11
俄羅斯人 11
俄羅斯平原 1
俄羅斯平原 1
俄而 5
俄語 77
俄語 77
俄語系 2
俄語系 2
俄語組 2
俄語組 2
俄軍 23
俄軍 23
俄頃 4
俅 46
俆 0
俆 0
俇 166
俈 0
俉 31
俊 59
俊 5943
俊俏 25
俊偉 90
俊傑 255
俊國 22
俊國熊 2
俊材 1
俊爽 0
俊男 0
俊秀 137
俊美 47
俊逸 48
俊雅 4
俊馬 0
俋 486
俌 0
俍 9
俎 55
俎上肉 3
俏 403
俏皮 56
俏皮話 14
俏麗 60
俏麗短髮 1
俐 364
俐落 119
俑 42
俒 0
俒 0
俓 51
俔 22
俕 0
俖 17
俗 1202
俗不可耐 10
俗事 9
俗人 13
俗例 1
俗務 12
俗務纏身 0
俗化 8
俗名 366
俗士 1
俗套 14
俗子 5
俗字 4
俗學 0
俗家 16
俗家子弟 0
俗尚 0
俗念 3
俗文學 14
俗業 1
俗樂 5
俗氣 35
俗物 2
俗稱 403
俗緣 0
俗艷 2
俗話 36
俗話說 45
俗語 101
俗語說 28
俗論 0
俗諺 33
俗體 1
俗體字 0
俘 35
俘獲 9
俘虜 87
俘虜營 0
俙 255
俚 23
俚俗 10
俚曲 1
俚歌 1
俚言 0
俚語 490
俚諺 6
俚醫 0
俛 14
俛 14
俜 205
保 11855
保不住 9
保久 9
保久乳 7
保人 187
保住 163
保佑 139
保值 60
保健 5119
保健室 128
保全 1187
保全人員 28
保全公司 34
保力達 0
保加利亞 134
保單 603
保固 547
保固期 83
保國衛民 10
保土 2
保墑 0
保姆 54
保存 2260
保存期限 123
保存著 10
保存起來 11
保守 1096
保守主義 23
保守政府 0
保守派 51
保守黨 21
保安 527
保安大隊 18
保安林 47
保安警察 193
保安隊 10
保定 20
保定市 1
保家 5
保家衛國 6
保密 521
保密防諜 8
保山 4
保持 3378
保持在 132
保持有 10
保持著 86
保持警惕 0
保持身材 7
保時捷 1
保暖 102
保有 634
保正 9
保母 110
保民 15
保民衛國 1
保溫 296
保溫杯 6
保溫瓶 9
保溫箱 22
保溼 15
保濕 1
保用 7
保用期間 0
保用期限 1
保甲 7
保甲法 0
保留 2553
保留住 6
保留圖書 0
保留在 40
保留地 228
保留盈餘 59
保留著 16
保留餘地 0
保皇 3
保皇派 5
保皇黨 2
保祐 5
保稅 192
保稅倉庫 47
保管 1065
保管箱 85
保管費 20
保結 3
保羅 710
保羅三世 0
保育 4746
保育員 17
保育院 6
保舉 8
保苗 0
保荐 4
保薦 24
保藏 36
保藏品 0
保衛 272
保衛和平 2
保衛國家 13
保衛祖國 2
保衛組 0
保衛者 1
保証 306
保証人 7
保証供應 0
保証金 28
保證 3879
保證人 194
保證供應 0
保證書 200
保證金 742
保警 50
保護 10266
保護主義 40
保護人 21
保護區 1
保護國 3
保護性 24
保護者 20
保護色 13
保護關稅 2
保護鳥 0
保費 742
保質 0
保赦 0
保身 19
保送 497
保送入學 62
保釋 18
保釋金 1
保重 178
保金 4
保釣 39
保釣運動 18
保鏢 52
保鑣 75
保長 67
保防 60
保防工作 5
保防教育 2
保障 3767
保險 11187
保險人 417
保險信封 0
保險公司 746
保險單 68
保險套 368
保險帶 0
保險庫 5
保險桿 74
保險槓 1
保險櫃 39
保險法 315
保險箱 74
保險絲 106
保險費 549
保險金 416
保險金額 173
保險門 0
保養 2059
保養品 349
保養站 0
保養費 4
保養部 1
保馬法 0
保駕 1
保鮮膜 1
保麗龍 93
保齡球 706
保齡球賽 24
保齡球館 224
俞 1
俞 1
俞 573
俞允 28
俞振飛 1
俟 608
俟 608
俠 3103
俠士 0
俠女 32
俠客 100
俠氣 3
俠盜 40
俠義 50
俠義精神 2
俠行 0
俠骨 18
俠骨 18
信 2
信 26305
信不過 3
信以為真 15
信仰 1750
信仰主義 1
信仰自由 10
信件 3314
信任 530
信任感 25
信任案 72
信佛 37
信佛教 23
信使 13
信個 8
信內 24
信其言 0
信函 394
信到 51
信口 8
信口胡說 0
信口開河 6
信口雌黃 4
信口雌黃 4
信史 10
信合社 0
信天翁 27
信奉 93
信女 1
信守 282
信守不棄 1
信守不棄 1
信守不渝 1
信實 91
信封 898
信差 31
信徒 969
信得過 9
信從 17
信心 2321
信心十足 53
信念 478
信息 975
信息論 0
信手 48
信手拈來 13
信托 21
信托貿易 0
信教 31
信文 32
信服 82
信札 9
信條 84
信步 29
信步閒遊 0
信物 22
信用 2047
信用卡 4181
信用可靠 9
信用合作 15
信用合作社 597
信用放款 7
信用狀 252
信用証 0
信用調查 9
信用證 1
信用貸款 111
信皮 1
信眾 0
信筆 5
信筆拈來 0
信筆直書 0
信筒 0
信箋 19
信管 5
信箱 18480
信箱號碼 59
信簡 3
信紙 96
信義 2328
信義國中 15
信義國小 59
信義為立業之本 2
信義計劃 4
信義計劃區 30
信義路 1246
信號 1309
信號台 13
信號彈 5
信號旗 1
信號槍 0
信號機 0
信號燈 9
信號系統 7
信號臺 0
信託 1863
信託公司 99
信託局 3
信誓旦旦 54
信譽 201
信譽卓著 12
信貸 136
信貸資金 1
信賞 1
信賞必罰 0
信賴 579
信陽 34
信風 14
信風帶 2
信鳥 0
信鴿 19
俢 0
俣 0
俥 0
俥 0
俩 0
俩 0
俬 10
修 12208
修修 22
修修補補 2
修到 19
修剪 119
修士 36
修女 277
修好 147
修定 118
修己 5
修建 233
修得 62
修復 570
修憲 0
修成 44
修持 169
修指甲 3
修撰 2
修改 6989
修改後 80
修整 93
修明 24
修書 17
修業 607
修業年限 379
修業期滿 67
修橋舖路 3
修橋舖路 3
修橋補路 0
修正 7925
修正主義 6
修正案 418
修正檔 0
修水 10
修水利 1
修治 4
修煉 63
修理 789
修理匠 0
修理店 7
修理站 3
修理費 30
修理起來 0
修理部 1
修的 179
修築 60
修築公路 3
修築工事 0
修築工程 0
修築道路 2
修繕 282
修繕費 45
修羅 1
修習 1044
修耕 0
修葺 7
修行 1036
修補 315
修複 1
修訂 2248
修訂本 36
修訂版 134
修詞學 0
修課 544
修護 815
修護站 6
修護隊 0
修起 5
修起來 1
修路 39
修蹄 0
修身 139
修身養性 22
修辭 102
修辭學 29
修辭格 0
修造 137
修造廠 11
修道 124
修道人 25
修道士 5
修道院 61
修配 186
修配廠 68
修長 76
修面 28
修飾 253
修飾句 0
修飾字 3
修飾詞 2
修飾語 11
修養 207
修養成 0
修鬍刀 0
修齊 17
俯 160
俯下 4
俯下去 0
俯仰 22
俯仰之間 0
俯仰無愧 3
俯仰由人 0
俯伏 17
俯允 2
俯姿 0
俯射 0
俯射角 0
俯就 3
俯念 0
俯拾 7
俯拾即是 4
俯拾皆是 13
俯曲 0
俯沖 0
俯泳 0
俯看 11
俯瞰 107
俯臥 40
俯臥撐 4
俯衝 26
俯衝角 0
俯衝點 0
俯視 63
俯視圖 15
俯角 12
俯身 40
俯頭 3
俯首 27
俯首就範 0
俯首帖耳 0
俯首稱臣 7
俯首認罪 3
俯首認錯 0
俰 0
俱 1
俱 848
俱佳 50
俱備 47
俱全 63
俱往 2
俱樂部 6846
俱發 4
俱碎 0
俲 0
俳 73
俳優 1
俳句 73
俴 82
俵 161
俶 498
俶 498
俷 181
俸 191
俸恤 0
俸祿 5
俸給 76
俸褖 0
俸錢 0
俹 0
俹 0
俺 229
俺們 4
俻 0
俼 0
俼 0
俼 0
俽 0
俾 96
俾 967
俾 967
俾使 228
俾使 228
俾倪 0
俾倪 0
俾斯麥 17
俾斯麥 17
俾晝作夜 0
俾晝作夜 0
俾能 120
俾能 120
俾資挹注 0
俾資挹注 0
俿 0
倀 63
倂 0
倃 0
倄 0
倅 126
倆 573
倆 573
倇 268
倈 0
倈 0
倉 1337
倉促 58
倉促間 5
倉儲 1152
倉卒 25
倉州 0
倉庫 612
倉庫業 2
倉庫管理員 9
倉廒 1
倉廩 2
倉房 6
倉海桑田 0
倉猝 1
倉皇 31
倉頡 159
倉頡輸入 9
倉頡輸入法 87
倊 0
個 36766
個 36766
個中 292
個中三昧 1
個人 25179
個人主義 61
個人利益 15
個人化 150
個人得失 0
個人性 20
個人所得 33
個人所得稅 32
個人消費 38
個位 187
個例 39
個個 336
個兒 15
個兒 15
個別 1956
個別差異 145
個別性 27
個別情況 6
個別指導 34
個別教學 19
個別測驗 5
個別現象 8
個別談話 12
個別輔導 67
個子 508
個性 2088
個性化 123
個把 37
個把月 6
個數 176
個案 3047
個案紀錄 6
個舊 29
個裡 16
個頭 131
個體 764
個體戶 85
個體所有 0
個體所有制 0
個體手工 0
個體手工業 0
個體發育 1
個體經濟 37
倌 207
倍 3077
倍加 24
倍受 69
倍增 123
倍感 85
倍數 311
倍率 107
倍速 0
倎 41
倏 206
倏地 63
倏地 63
倏忽 16
倏然 25
倐 0
們 9385
們 19397
倒 22352
倒 65358
倒吐 10
倒吐 10
倒上 4
倒上來 0
倒上去 0
倒下 113
倒下來 10
倒下去 8
倒不如 161
倒不是 73
倒不會 8
倒不開 0
倒不開 0
倒也 166
倒也不 11
倒也不是 8
倒也不會 0
倒了 121
倒伏 21
倒來 3
倒來倒去 0
倒倒 3
倒出 28
倒出來 6
倒出去 0
倒到 12
倒包 0
倒去 6
倒反 12
倒反 12
倒句 0
倒台 8
倒吃 2
倒吃甘蔗 21
倒向 22
倒味口 1
倒嗓 1
倒嗓子 0
倒嚼 0
倒嚼 0
倒嚼 0
倒回 19
倒回來 2
倒回去 4
倒在 173
倒地 92
倒地不起 12
倒坍 0
倒塌 145
倒好 11
倒寫 0
倒屣相迎 0
倒帳 64
倒帶 13
倒帶機 0
倒彩 0
倒影 84
倒得 11
倒念 0
倒懸 7
倒戈 108
倒戈卸甲 0
倒扣 32
倒把 9
倒拉 3
倒拉一把 0
倒掉 41
倒掛 12
倒掛金鉤 0
倒推 0
倒換 3
倒換 3
倒放 2
倒敘 9
倒敘法 1
倒數 568
倒斃 5
倒映 9
倒映在 9
倒是 997
倒是 997
倒會 20
倒杯 8
倒杯水 1
倒杯茶 1
倒栽蔥 1
倒楣 145
倒楣鬼 7
倒毀 1
倒水 14
倒流 42
倒滿 12
倒灌 38
倒為 2
倒的 59
倒立 56
倒給 8
倒置 23
倒胃 4
倒胃口 11
倒背 1
倒背如流 7
倒臥 16
倒茶 22
倒行 0
倒行逆施 2
倒裝 4
倒裝 4
倒裝句 8
倒裝句 8
倒裝法 0
倒貼 14
倒賠 0
倒賣 3
倒賬 0
倒走 3
倒起 0
倒起來 0
倒車 48
倒轉 59
倒退 123
倒進 18
倒進來 0
倒進去 1
倒運 2
倒過 8
倒過來 20
倒過去 0
倒錯 10
倒閉 221
倒閣 40
倒霉 51
倒霉鬼 3
倒頭 19
倓 276
倔 43
倔 43
倔強 39
倕 6
倖 83
倖免 71
倖免於難 12
倖存 29
倖存者 19
倖幸 0
倗 284
倘 213
倘佯 22
倘使 17
倘或 4
倘有 42
倘然 4
倘若 455
倘若在 7
倘若是 12
倘若會 0
倘若有 6
倘若能 9
候 1
候 1475
候光 3
候命 1
候客室 1
候審 4
候教 13
候機 8
候機室 23
候船 2
候船室 4
候補 188
候補人 0
候補者 5
候診 24
候診室 33
候車 18
候車亭 9
候車室 1
候車站 4
候選 483
候選人 5845
候選者 4
候銷 0
候駕 1
候魚 2
候鳥 262
倚 374
倚仗 10
倚偎 1
倚多為勝 2
倚天 592
倚天劍 1
倚山傍水 0
倚恃 1
倚望 0
倚牆而立 0
倚立 0
倚老賣老 5
倚賴 87
倚賴 87
倚賴心 0
倚賴性 1
倚賴性 1
倚賴成性 0
倚重 34
倚門 2
倚門賣俏 0
倚靠 103
倚靠在 4
倚靠著 0
倚音 0
倛 356
倜 158
倜儻 4
倝 0
倞 625
借 2389
借了 49
借人 20
借以 19
借住 32
借作 3
借來 52
借來借去 0
借個 27
借個火 1
借借 3
借借看 0
借債 1
借債度日 0
借光 9
借入款 8
借出 164
借出來 3
借出去 1
借刀殺人 14
借到 27
借助 113
借助於 18
借去 5
借口 10
借古諷今 2
借唱 0
借問 68
借回 5
借回來 0
借回去 1
借地 0
借坐 1
借好 1
借完 1
借宿 7
借宿一夜 1
借宿一晚 0
借屍還魂 16
借彈 1
借得 12
借托 0
借抄 3
借抽 0
借招 0
借據 19
借支 21
借放 1
借故 5
借方 44
借書 780
借書證 292
借有 9
借條 0
借款 311
借款人 59
借此 42
借此機會 17
借滿 1
借火 0
借球 0
借用 913
借的 53
借看 2
借看一下 0
借穿 0
借端 0
借端生事 0
借箸代籌 0
借箸代謀 0
借給 81
借花獻佛 3
借著 35
借託 0
借詞 4
借調 92
借讀 2
借讓 0
借貸 261
借貸資本 0
借走 7
借過 10
借過一下 4
借道 2
借酒澆愁 3
借酒裝瘋 5
借重 128
借錢 93
借鏡 154
借鑑 4
借鑒 16
借閱 1209
借題 3
借題發揮 14
借點 2
倠 41
倡 293
倡 293
倡亂 2
倡優 0
倡優 1
倡導 333
倡導者 11
倡狂 1
倡言 23
倡議 165
倢 129
倣 146
倣古 0
倣效 8
倣真 0
値 0
値 0
倥 55
倥 55
倦 252
倦勤 7
倦容 8
倦怠 69
倦意 7
倦意頓消 0
倦感 0
倦態 1
倦游 0
倦遊 1
倦鳥歸巢 0
倦鳥知歸 0
倦鳥知返 0
倧 12
倨 40
倨傲 4
倩 692
倩倩 12
倩女 22
倩女幽魂 16
倩影 310
倪 5863
倪嗣沖 0
倪瓚 6
倪雲林 6
倫 3668
倫常 27
倫敦 1192
倫次 1
倫理 1146
倫理學 194
倫琴 0
倫琴射線 0
倫紀 0
倬 251
倭 49
倭奴 0
倭寇 16
倭寇船 0
倮 0
倯 0
倰 0
倱 0
倲 0
倳 121
倴 0
倵 1
倶 0
倶 0
倷 1
倸 0
倹 0
倹 0
值 9136
值勤 80
值勤者 1
值回票價 1
值域 0
值夜 14
值夜班 5
值完 3
值得 44
值得 4480
值得一提 323
值得一提 323
值得一提 323
值得一提 323
值得一看 134
值得一看 134
值得一看 134
值得一看 134
值日 57
值日官 2
值日生 33
值日表 2
值早班 0
值星 13
值星官 2
值星班長 1
值晚班 1
值此 84
值班 177
值班人 0
值班員 1
值班室 6
值錢 53
偀 9
偁 6
偁 6
偂 0
偃 346
偃旗息鼓 4
偃月 8
偃武修文 0
偃臥 1
偄 0
偄 0
偅 6
偆 8
假 6107
假 4390
假人 20
假仁假義 1
假令 4
假以時日 33
假作 12
假使 300
假使在 7
假使會 0
假使有 6
假使能 1
假使說 0
假借 82
假借法 0
假傳 6
假傳聖旨 0
假充 1
假公濟私 7
假冒 70
假冒著 0
假分數 3
假力於人 0
假動作 6
假名 37
假名假姓 0
假名字 6
假單 14
假圖 4
假執行 27
假報 0
假如 1524
假如是 45
假如會 1
假如有 65
假如能 8
假如說 7
假定 142
假定性 1
假定是 6
假定有 2
假定說 0
假寐 7
假山 42
假得 3
假性 109
假性近視 16
假情假意 3
假想 61
假想敵 38
假惺惺 7
假意 20
假慈悲 2
假戲真做 4
假手 12
假手他人 8
假托 0
假扣押 56
假扮 36
假扮作 0
假撇清 0
假日 1534
假期 1347
假果 1
假根 0
假條 2
假植 3
假正經 3
假死 9
假牌 2
假牙 100
假牙齒 0
假用 2
假畫 2
假的 259
假皮 0
假相 16
假睡 1
假若 242
假若在 9
假若是 2
假若會 0
假若有 5
假若能 5
假藥 1
假裝 187
假裝在 3
假裝是 12
假裝會 0
假裝有 0
假裝能 0
假言判斷 0
假託 6
假設 1496
假設在 42
假設是 31
假設會 3
假設有 51
假設能 4
假設說 7
假話 15
假說 86
假諦 1
假象 102
假貸 0
假退役 1
假送 0
假造 32
假造文件 0
假道 14
假道學 3
假酒 8
假釋 72
假釋官 1
假釋期間 0
假錢 1
假鎖 0
假門 1
假面 47
假面具 12
假髮 64
假髻 0
假魚 5
偈 132
偈 132
偉 6731
偉人 146
偉大 1286
偉業 20
偉績 3
偊 2
偋 2
偌 393
偌大 53
偍 9
偎 329
偎乾就濕 0
偎依 3
偎倚 2
偎近 1
偏 1298
偏上 3
偏上來 0
偏上去 0
偏下 2
偏不 25
偏不湊巧 0
偏了 14
偏低 341
偏來 1
偏來偏去 0
偏倒 0
偏倚 5
偏偏 365
偏僻 81
偏僻地區 3
偏光 77
偏光鏡 1
偏到 0
偏勞 2
偏北 19
偏南 14
偏去 0
偏右 38
偏向 270
偏回 0
偏回來 0
偏回去 0
偏在 8
偏執 36
偏執狂 8
偏壓 23
偏外 4
偏失 8
偏好 300
偏守 0
偏安 8
偏室 1
偏射 0
偏左 28
偏巧 6
偏差 385
偏巷 0
偏廢 11
偏後 1
偏得 0
偏微分 84
偏德 0
偏心 81
偏心圓 0
偏心輪 1
偏愛 116
偏房 1
偏振 33
偏振光 8
偏振片 6
偏掉 3
偏斜 10
偏方 141
偏於 20
偏旁 8
偏是 11
偏暗 3
偏會 0
偏有 6
偏東 19
偏東北 1
偏東南 1
偏析 5
偏枯 5
偏歪 5
偏流 3
偏激 46
偏狂 0
偏的 5
偏短 3
偏私 6
偏移 68
偏窄 0
偏紫 2
偏累 0
偏細 0
偏聽 0
偏至 0
偏航 6
偏著 5
偏處 2
偏處 2
偏處一隅 1
偏蝕 1
偏袒 35
偏西 12
偏西北 0
偏西南 1
偏要 30
偏見 156
偏角 8
偏護 2
偏護著 0
偏轉 17
偏轉線圈 0
偏辭 0
偏遠 280
偏遠地區 329
偏重 214
偏重在 37
偏重於 102
偏鋒 6
偏長 0
偏門 3
偏離 147
偏離正軌 2
偏靠 0
偏靠在 0
偏頗 71
偏頭 22
偏頭痛 69
偏顧 0
偏顧著 0
偏食 48
偏高 324
偐 0
偐 0
偑 16
偒 0
偒 0
偓 6
偔 0
偕 462
偕 462
偕同 84
偕老 13
偕行 12
偖 0
偖 0
偗 7
偘 0
偙 0
做 24148
做上 15
做下來 0
做下去 23
做主 58
做了 1338
做事 649
做事情 23
做人 288
做人情 3
做什麼 674
做什麼 674
做伴 12
做作 69
做保 5
做假 17
做做 201
做做事 2
做出 1223
做出來 50
做到 1177
做功 9
做嘔 2
做圈套 0
做在 28
做壽 2
做夢 148
做大 72
做好 1432
做好做歹 0
做媒 9
做完 201
做官 28
做客 34
做對 52
做工 66
做得 575
做得 575
做得到 57
做得到 57
做愛 294
做成 469
做戲 16
做手腳 3
做文章 37
做東 5
做案 19
做法 1039
做活 18
做為 2702
做牛做馬 12
做生意 135
做生日 5
做的 2512
做禮拜 19
做答 16
做聲 8
做股票 13
做莊 1
做菜 43
做著 66
做詩 6
做買賣 13
做賊心虛 3
做起 282
做起來 31
做錯 147
做飯 23
做鬼 13
做鬼臉 1
偛 6
停 4022
停上 2
停上來 0
停上去 0
停下 142
停下來 173
停下去 0
停不下 6
停不下來 18
停不住 4
停了 163
停住 68
停來停去 0
停停 25
停停看 0
停刊 91
停刊處分 0
停利 1
停到 11
停勻 1
停售 14
停在 259
停好 10
停妥 5
停學 5
停完 0
停屍 3
停屍間 10
停工 218
停工處分 10
停得 5
停徵 10
停息 20
停戰 11
停戰協定 6
停戰和約 0
停手 16
停損 1
停損點 100
停播 10
停播 10
停擺 167
停放 127
停放在 25
停會 13
停有 4
停板 19
停業 197
停業處分 0
停機 126
停機坪 40
停權 0
停歇 59
停止 3329
停止下來 3
停步 13
停水 32
停水通知 1
停泊 426
停泊費 0
停滯 130
停滯不前 44
停火 34
停火協定 1
停產 101
停用 217
停留 591
停留在 332
停留時間 87
停當 6
停筆 8
停約 0
停經 179
停經期 5
停經片 1
停置 2
停職 121
停航 7
停船 6
停船靠岸 0
停著 9
停薪 15
停製 0
停訓 0
停診 4
停話 19
停課 135
停賽 8
停蹄 2
停車 1011
停車位 301
停車場 105
停車場 1054
停車燈 0
停車費 1
停載 1
停轉 2
停辦 160
停進 0
停進來 0
停進去 1
停過 16
停過來 0
停過去 0
停錯 0
停閉 0
停開 31
停電 217
停電通知 1
停靈 3
停靠 120
停靠在 9
停頓 180
停頓下來 6
停顯 0
停飛 58
停食 0
停駐 26
停駛 60
偝 7
偞 3
偟 4
偠 1
偡 5
偢 3
偢 3
偣 10
偤 12
健 5722
健保 1750
健保卡 0
健保局 429
健健康康 36
健兒 71
健全 1641
健全性 5
健在 48
健壯 71
健壯如牛 1
健將 25
健康 18970
健康器 3
健康檢查 858
健康食品 1
健忘 79
健忘症 7
健教 40
健旺 3
健朗 6
健步 18
健步如飛 8
健美 130
健美操 0
健行 675
健行工專 196
健行隊 3
健談 29
健走 1
健身 667
健身房 178
健身操 23
健身運動 37
偦 0
偦 0
偧 0
偨 1
偩 6
偪 4
偫 1
偬 0
偭 233
偮 3
偯 266
偰 3
偱 0
偲 8
偳 4
側 28
側 2876
側向 28
側壁 27
側室 1
側旁 1
側枝 15
側根 1
側目 39
側目而視 0
側睡 3
側筆 0
側線 21
側翼 6
側耳 29
側聞 1
側臥 19
側臥法 0
側舷炮 0
側芽 10
側視 19
側視圈 0
側身 44
側身而臥 0
側近 1
側邊 39
側部 3
側重 85
側門 59
側面 244
側投 12
側面圖 20
側面性 0
偵 1030
偵候 0
偵問 0
偵察 98
偵察兵 5
偵察地形 0
偵察機 6
偵察連 0
偵探 599
偵探小說 21
偵探片 1
偵查 326
偵測 2220
偵測碼 1
偵破 130
偵緝 15
偵聽 1
偵聽台 0
偵聽員 0
偵聽器 0
偵訊 171
偵辦 514
偵錯 92
偵防 110
偵騎 1
偶 16151
偶一為之 5
偶像 1788
偶像人物 3
偶像劇 1
偶像化 2
偶像崇拜 10
偶合 37
偶性 1
偶感 10
偶數 40
偶數排 0
偶數班 0
偶數號 1
偶有 244
偶然 454
偶然之間 1
偶然性 2
偶然間 19
偶爾 660
偶發 55
偶發事件 63
偶發性 18
偶筆 0
偶而 458
偶而為之 1
偶記 26
偶語 2
偶起 2
偶遇 29
偷 1078
偷上 1
偷上來 0
偷上去 0
偷下 0
偷下來 0
偷下去 0
偷了 65
偷人 6
偷作 0
偷來 40
偷來偷去 0
偷做 1
偷停 0
偷偷 705
偷偷地 71
偷偷地 71
偷偷插入 0
偷偷摸摸 31
偷光 3
偷出 3
偷出來 2
偷出去 0
偷到 21
偷功夫 0
偷加 0
偷包 0
偷去 7
偷取 22
偷叫 0
偷吸 1
偷喝 5
偷回 3
偷回來 1
偷回去 0
偷報 0
偷填 0
偷壘 0
偷天換日 10
偷夾 0
偷存 0
偷安 3
偷宰 0
偷寄 0
偷工 3
偷工減料 40
偷得 7
偷得浮生半日閒 10
偷情 42
偷懶 465
偷打 2
偷抄 1
偷抱 5
偷拍 0
偷拿 8
偷換 2
偷搞 30
偷搬 2
偷收 0
偷教 0
偷數 0
偷數 0
偷書 12
偷架 0
偷水 1
偷油 1
偷法 0
偷泡 0
偷洗 0
偷涼 0
偷渡 123
偷渡客 44
偷渡過境 0
偷溜 10
偷溜走 0
偷炒 0
偷營 1
偷獵 1
偷球 0
偷生 9
偷發 0
偷的 20
偷盜 28
偷看 196
偷睡 0
偷砍 0
偷稅 3
偷種 0
偷空 3
偷空 3
偷窺 141
偷窺狂 6
偷竊 125
偷竊狂 2
偷竊罪 2
偷笑 51
偷聽 39
偷腥 9
偷花 0
偷花賊 0
偷著 4
偷襲 64
偷調 0
偷調 0
偷請 0
偷講 0
偷讀 0
偷走 62
偷起 0
偷起來 0
偷跑 43
偷跳 0
偷車 19
偷車賊 7
偷載 0
偷辦 0
偷過 7
偷過來 2
偷過去 0
偷錢 8
偷開 0
偷開來 0
偷開到 0
偷開去 0
偷閒 13
偷雞不著反折一把米 1
偷雞不著蝕把米 5
偷雞摸狗 13
偷領 0
偷食禁果 0
偷香竊玉 2
偷騎 0
偸 0
偹 0
偺 213
偺 213
偺們 0
偻 0
偼 0
偽 598
偽名 0
偽名 0
偽君子 48
偽命 0
偽善 22
偽善 22
偽善者 0
偽學 4
偽學者 0
偽幣 17
偽政府 2
偽政權 4
偽書 3
偽本 1
偽滿 9
偽滿州國 7
偽滿政府 0
偽科學 0
偽基 600
偽基百科 600
偽基百科人 500
偽基人 600
偽經 5
偽經 5
偽藥 6
偽裝 138
偽裝做 0
偽裝品 0
偽裝物 0
偽言 0
偽証 1
偽誓 0
偽誓 0
偽謬 0
偽證 52
偽證罪 11
偽足 1
偽軍 2
偽造 246
偽造 246
偽造品 1
偽造文書 54
偽鈔 52
偽飾 1
傀 48
傀 48
傀儡 86
傀儡戲 32
傀儡政府 2
傀儡政權 5
傁 0
傁 0
傂 3
傃 2
傄 0
傅 1986
傅利葉 12
傅園 2
傅斯年 66
傅會 0
傅科 26
傅科擺 26
傅立葉 26
傆 0
傇 4
傈 0
傉 0
傊 0
傋 5
傌 27
傍 215
傍 215
傍 215
傍偟 1
傍午 0
傍徨 3
傍晚 476
傍晚時候 2
傍系 0
傍花隨柳 0
傍若無人 0
傍邊 3
傎 5
傏 0
傐 0
傑 4011
傑作 204
傑克 385
傑克森 48
傑克遜 48
傑出 1741
傒 10
傒 10
傓 0
傓 0
傔 0
傕 10
傖 752
傖俗 0
傖父 0
傗 0
傘 1204
傘兵 23
傘具 7
傘包 6
傘形 6
傘索 0
傘齒輪 13
備 7598
備下 8
備件 4
備份 954
備取 244
備取生 91
備受 407
備品 50
備嘗 2
備嘗艱辛 0
備忘 65
備忘錄 353
備感 34
備戰 75
備戰狀態 3
備抵 0
備抵折舊 0
備援 0
備料 24
備有 746
備查 617
備案 139
備注 25
備用 273
備用品 25
備索 75
備考 130
備而不用 10
備耕 2
備胎 33
備至 39
備荒 0
備註 3581
備註欄 462
備詢 0
備課 9
備車 7
備車前來 0
備車前往 0
備載 43
傚 107
傚法 3
傛 5
傜 20
傝 7
傞 3
傟 0
傠 0
傡 0
傢 450
傢伙 412
傢伙 412
傢俱 14
傢俱 1481
傢具 263
傣 6
傣族 10
傪 0
傫 0
催 879
催收 100
催促 107
催化 187
催化作用 11
催化劑 89
催化裂化 0
催告 74
催情 0
催毀 6
催毀性 0
催淚 8
催淚彈 8
催生 214
催生作用 0
催產 9
催眠 243
催眠曲 13
催眠療法 0
催眠術 21
催租 0
催稿 107
催肥 0
催芽 13
催討 11
催賬單 0
催趕 1
催趕著 1
催逼 10
傭 26
傭 265
傭人 13
傭兵 144
傭工 11
傭懶 2
傭書 0
傭金 6
傮 55
傯 50
傰 126
傱 91
傲 901
傲世 11
傲人 104
傲性 1
傲慢 61
傲慢與偏見 23
傲氣 47
傲氣凌人 3
傲然 22
傲然自得 0
傲物 1
傲視 71
傲視一切 0
傲視群倫 11
傲視群芳 0
傲起來 0
傲骨 7
傲骨凌人 0
傳 2335
傳 23358
傳上 19
傳上來 9
傳上去 1
傳下 41
傳下來 8
傳下去 10
傳不到 6
傳世 61
傳人 0
傳代 1
傳令 33
傳令兵 8
傳佈 40
傳位 2
傳來 702
傳來傳去 8
傳信 46
傳信鴿 0
傳傳 8
傳入 267
傳入神經 0
傳出 891
傳出來 11
傳出去 0
傳到 263
傳制權 0
傳動 597
傳動力 1
傳動帶 7
傳動比 1
傳動裝置 5
傳動軸 38
傳動鏈 0
傳去 4
傳問 8
傳喚 81
傳單 187
傳回 601
傳國 7
傳報 18
傳奇 1500
傳奇性 34
傳媒 0
傳子 6
傳宗 17
傳宗接代 21
傳家 50
傳家之寶 9
傳家寶 4
傳審 0
傳對 1
傳導 212
傳導性 9
傳導電流 0
傳導體 1
傳布 10
傳得 25
傳情 201
傳感器 18
傳承 785
傳授 506
傳授法 1
傳授給 36
傳揚 113
傳播 3683
傳播媒介 25
傳播媒體 335
傳播學 30
傳播學系 145
傳播界 54
傳播給 8
傳播者 15
傳教 145
傳教士 80
傳旨 2
傳書 16
傳書鴿 0
傳染 522
傳染性 257
傳染病 693
傳染給 57
傳為 20
傳為佳話 6
傳為美談 6
傳熱 29
傳熱係數 3
傳熱性 0
傳球 79
傳球法 0
傳略 19
傳的 249
傳真 21690
傳真機 903
傳真電報 0
傳神 94
傳票 298
傳給 334
傳統 9351
傳統化 2
傳統名稱 0
傳統市場 1
傳統性 36
傳統教育 26
傳統觀念 30
傳習 101
傳聞 406
傳聞中 37
傳聲 18
傳聲筒 20
傳自 20
傳著 5
傳衣缽 0
傳言 332
傳訊 562
傳記 597
傳記文學 33
傳話 48
傳誦 12
傳誦一時 0
傳說 1611
傳說中 217
傳說人物 1
傳諭 4
傳譯出 0
傳質 0
傳起 2
傳起來 0
傳輸 4390
傳送 2626
傳送帶 1
傳送機 9
傳送率 2
傳遍 59
傳遍 59
傳過 15
傳過來 13
傳過去 7
傳道 699
傳達 1000
傳達出來 9
傳達室 12
傳達物 1
傳達處 0
傳遞 1552
傳遞性 1
傳銷 0
傳開 35
傳閱 43
傳電 110
傳頌 19
傳頌一時 3
傳頌千古 2
傴 51
傴僂 1
債 1579
債主 17
債券 1968
債務 1068
債務人 161
債務國 3
債台高築 12
債多不愁 0
債戶 2
債權 511
債權人 215
債權國 6
債權團 0
債款 6
債臺高築 0
傶 266
傷 19600
傷了 160
傷亡 355
傷亡人數 11
傷亡慘重 10
傷人 149
傷來傷去 0
傷兵 29
傷到 61
傷勢 104
傷口 629
傷口發炎 8
傷員 2
傷在 14
傷天害理 12
傷害 2827
傷害到 84
傷害力 8
傷害性 17
傷害罪 10
傷寒 113
傷寒桿菌 8
傷寒病 1
傷得 15
傷心 999
傷心欲絕 25
傷患 118
傷患人數 0
傷悲 46
傷感 68
傷感情 18
傷懷 9
傷殘 282
傷疤 9
傷病員 1
傷痕 185
傷痕累累 26
傷痛 200
傷神 20
傷科 140
傷者 110
傷肺 3
傷胃 13
傷腦筋 266
傷處 20
傷財 7
傷身 61
傷過 7
傷錯 0
傷風 89
傷風克 0
傷風感冒 9
傷風敗俗 7
傸 98
傹 0
傺 108
傻 693
傻事 29
傻人 10
傻人自有傻福 0
傻勁 24
傻呵呵 0
傻子 67
傻小子 4
傻氣 6
傻瓜 421
傻瓜蛋 0
傻眼 0
傻笑 49
傻蛋 15
傻裡傻氣 1
傻話 11
傻頭傻腦 2
傼 0
傽 66
傾 603
傾 603
傾下 3
傾下來 0
傾下去 0
傾倒 174
傾倒 174
傾倒垃圾 13
傾側 1
傾入 8
傾出 9
傾刻 7
傾力 31
傾動 2
傾化 0
傾危 1
傾吐 31
傾吐衷情 0
傾向 1001
傾向性 1
傾向於 164
傾囊相助 0
傾囊相授 16
傾國傾城 15
傾城傾國 1
傾家 2
傾家蕩產 14
傾巢來犯 0
傾巢而出 22
傾度 1
傾復 0
傾心 60
傾心吐意 6
傾慕 25
傾斜 305
傾斜度 17
傾斜角 21
傾斜面 4
傾注 19
傾瀉 19
傾盆 3
傾盆大雨 18
傾箱倒櫃 0
傾箱倒篋 0
傾耳 6
傾聽 323
傾船 0
傾船 0
傾蓋 2
傾複 0
傾覆 31
傾角 32
傾訴 101
傾訴衷情 0
傾訴衷曲 0
傾談 6
傾軋 19
傾銷 477
傾銷法 5
傾銷稅 87
傾陷 0
傾頹 8
傿 183
僀 0
僁 113
僂 66
僃 0
僄 85
僅 77
僅 7776
僅以身免 1
僅供 275
僅供參考 866
僅僅 293
僅僅是 77
僅及 54
僅只 84
僅可 210
僅存 147
僅是 428
僅有 1043
僅次於 217
僅此 104
僅能 580
僅見 62
僆 0
僇 300
僇辱 0
僈 0
僉 1
僊 0
僋 0
僌 0
働 0
僎 0
像 14001
像似 20
像差 25
像是 2006
像模像樣 0
像樣 46
像樣兒 0
像樣兒 0
像煞有介事 0
像片 82
像素 1
像話 4
像距 0
僐 0
僑 2888
僑光商專 58
僑務 400
僑務委員 71
僑務委員會 456
僑匯 2
僑團 58
僑委會 323
僑居 65
僑居國 16
僑居地 78
僑校 37
僑民 272
僑生 1040
僑界 165
僑眷 1
僑聯 40
僑胞 260
僑資 16
僑陵 0
僒 0
僓 31
僔 29
僕 424
僕人 188
僕僕風塵 6
僕婦 2
僕役 21
僕從 7
僖 89
僗 41
僘 0
僙 0
僚 200
僚友 2
僚屬 3
僛 6
僜 0
僜 0
僝 27
僞 0
僞 0
僟 0
僠 20
僡 0
僢 0
僢 0
僣 4
僤 8
僥 127
僥 127
僥倖 81
僥倖取勝 0
僥倖獲勝 0
僥幸 13
僦 20
僧 985
僧人 81
僧侶 74
僧俗 10
僧堂 3
僧多粥少 12
僧孺 0
僧寺 1
僧尼 23
僧徒 0
僧服 6
僧格林沁 0
僧袍 4
僧院 4
僧鞋 1
僨 12
僩 175
僪 23
僫 0
僫 0
僫 0
僬 4
僭 204
僭號 0
僭越 8
僮 147
僮僕 2
僯 7
僰 24
僱 824
僱主 87
僱人 209
僱佣 1
僱佣 1
僱到 1
僱員 40
僱工 26
僱用 456
僱農 0
僳 19
僴 0
僴 0
僵 84
僵住 9
僵化 120
僵固 12
僵尸 3
僵局 117
僵屍 13
僵持 38
僵持不下 23
僵掉 0
僵死 2
僵直 166
僵直性脊椎炎 1
僵硬 209
僵立 5
僵臥 0
僶 17
僷 0
僸 25
價 10
價 10676
價位 888
價值 4713
價值尺度 4
價值形式 0
價值規律 0
價值觀 554
價值論 14
價值連城 15
價值量 6
價單 139
價差 0
價廉物美 60
價格 11121
價格表 233
價款 182
價目 171
價目單 7
價目表 397
價碼 67
價碼兒 0
價碼兒 0
價錢 901
價電子 10
僺 0
僻 133
僻 133
僻壤 2
僻巷 1
僻處一隅 0
僻處一隅 0
僻道 0
僻遠 4
僻陋 0
僻靜 20
僼 0
僽 18
僾 18
僿 5
儀 4989
儀仗 2
儀仗隊 3
儀制 5
儀器 4665
儀容 86
儀容檢查 8
儀式 754
儀態 38
儀態萬千 1
儀態萬方 1
儀禮 15
儀節 6
儀表 575
儀表出眾 0
儀表堂堂 1
儀表板 1
儀軌 1
儀錶 122
儀隊 72
儁 0
儁 0
儂 726
儃 3
億 8957
億元 4452
億測 0
億萬 128
億萬富翁 9
億載金城 52
儅 93
儆 60
儇 22
儈 12
儉 138
儉以養廉 0
儉樸 22
儉省 0
儉約 3
儊 7
儋 63
儋 63
儌 32
儌 32
儌 32
儍 0
儎 0
儎 0
儐 153
儐 153
儐相 4
儑 11
儒 1371
儒墨 4
儒學 118
儒家 293
儒將 2
儒教 66
儒林 118
儒林外史 14
儒生 16
儒者 29
儒艮 5
儒術 4
儒道 14
儒雅 7
儓 35
儔 171
儕 205
儕輩 2
儖 0
儗 22
儘 1016
儘 1016
儘儘 0
儘先 9
儘其所有 0
儘力 21
儘快 745
儘快 745
儘早 280
儘是 19
儘是 19
儘管 1600
儘速 1490
儘速 1490
儘量 1355
儙 0
儚 14
儛 0
儜 6
儝 0
儞 0
償 549
償付 39
償債 67
償債基金 8
償命 17
償回 2
償回來 0
償回去 0
償清 2
償還 461
償願 0
儠 162
儡 19
儢 71
儣 0
儤 105
儥 80
儦 87
儧 0
儨 0
儩 132
優 8540
優伶 6
優先 2218
優先權 244
優先發展 23
優先股 3
優劣 214
優勝 384
優勝劣敗 18
優勝獎 14
優勝者 50
優勝隊 2
優勢 1531
優厚 86
優哉游哉 3
優待 1196
優待券 14
優待票 39
優待證 0
優惠 5124
優惠價 10336
優惠價格 142
優惠待遇 17
優惠關稅 13
優撫 1
優於 294
優柔 2
優柔寡斷 36
優渥 57
優生 305
優生學 12
優異 2012
優異性 2
優秀 1754
優秀學生 107
優等 292
優等獎 112
優等生 6
優缺點 230
優美 811
優良 3489
優良單株 1
優良性 0
優裕 7
優質 525
優質鋼 0
優越 235
優越性 40
優越感 20
優遇 68
優遊 14
優遊自在 1
優遊自得 1
優選法 1
優酪乳 52
優閒 28
優雅 508
優養化 37
優點 1313
儫 0
儬 0
儬 0
儭 46
儮 42
儯 0
儰 33
儱 102
儲 1563
儲 1563
儲位 20
儲值卡 0
儲值票 0
儲備 426
儲備品 0
儲備糧 1
儲備量 1
儲君 40
儲存 4157
儲存 4157
儲存區 9
儲存區 9
儲存器 4
儲存器 4
儲存室 4
儲存庫 28
儲存庫 28
儲存起來 26
儲存起來 26
儲存量 37
儲存量 37
儲戶 14
儲放槽 0
儲水 38
儲水 38
儲水量 2
儲水量 2
儲油 16
儲油 16
儲油量 0
儲油量 0
儲積 4
儲蓄 634
儲蓄存款 141
儲蓄機構 1
儲蓄金 3
儲蓄銀行 86
儲藏 135
儲藏室 80
儲藏室 80
儲藏所 1
儲藏所 1
儲藏量 1
儲藏量 1
儲訓 137
儲訓人才 0
儲訓幹部 7
儲起來 0
儲起來 0
儲量 35
儲金 186
儲電量 0
儳 4
儴 29
儵 37
儶 0
儶 0
儷 628
儷人 42
儷影 12
儷辭 0
儸 167
儹 75
儹積 1
儹錢 1
儺 70
儻 12
儼 36
儼如 6
儼然 242
儼然是 29
儽 44
儽 44
儾 0
儿 920
兀 112
兀自 56
兀鷹 10
允 1798
允准 6
允可 4
允執厥中 0
允文允武 12
允當 28
允許 1513
允諾 182
兂 0
元 59466
元世祖 11
元代 100
元代人 0
元件 2968
元任 17
元元旦 0
元兇 41
元凶 0
元初 13
元勳 11
元史 19
元嘉 8
元培 38
元培醫事技術學校 0
元培醫專 51
元太祖 0
元始 16
元子 5
元宵 67
元宵夜 5
元宵節 70
元寶 116
元山 51
元帝 12
元帥 134
元年 299
元惡 3
元戎 3
元日 24
元旦 323
元旦假期 12
元明 76
元明兩代 0
元智大學 1
元曲 13
元月 1090
元月份 109
元朔 0
元朝 66
元朝人 1
元本 46
元氣 89
元氣大衰 0
元狩 0
元祖 38
元神 31
元素 1084
元素符號 5
元素週期律 0
元素週期表 1
元老 131
元老級 10
元老院 4
元色 6
元配 13
元音 5
元首 232
元首級 2
元龍高臥 0
兄 2406
兄友弟恭 6
兄台 0
兄妹 112
兄嫂 10
兄弟 1776
兄弟二人 11
兄弟們 45
兄弟兩人 9
兄弟會 0
兄弟象 97
兄弟鬩牆 9
兄弟黨 1
兄終弟及 1
兄臺 5
兄長 82
充 2314
充任 50
充份 0
充仿 0
充作 39
充充 22
充充氣 0
充公 4
充其量 110
充分 2318
充到 1
充塞 22
充塞著 5
充填 197
充實 1770
充數 12
充斥 298
充斥市面 3
充氣 116
充氣式 15
充沛 165
充滿 2524
充滿了 610
充滿著 201
充當 174
充盈 31
充盛 0
充耳不聞 13
充耳無聞 0
充腹 2
充血 84
充血性 25
充裕 246
充要條件 5
充足 338
充軍 6
充闊 0
充闊氣 0
充電 693
充電器 155
充電式 39
充類至盡 0
充飢 33
充饑 7
兆 6435
兆民 0
兆赫 79
兆頭 21
兇 2407
兇兇 3
兇兇惡惡 0
兇兇狠狠 0
兇器 22
兇嫌 27
兇宅 1
兇得 1
兇得 1
兇得狠 0
兇得狠 0
兇悍 15
兇惡 22
兇手 179
兇暴 10
兇案 12
兇殘 17
兇殺 130
兇殺案 17
兇犯 3
兇狠 15
兇猛 45
兇的 10
兇神惡煞 3
兇起來 0
兇險 0
先 13930
先上 68
先上來 4
先上去 4
先下 74
先下來 0
先下去 16
先下手 1
先下手為強 12
先世 6
先了 110
先了一步 1
先了一步 1
先人 101
先付 37
先令 24
先住 46
先佔 13
先來 205
先來先上 0
先來先下 0
先來後上 0
先來後下 0
先例 112
先倒 80
先借 130
先兄 1
先兆 15
先入之見 0
先入為主 30
先公 6
先出 43
先出來 7
先出去 3
先刊 2
先列 26
先到 331
先前 1303
先剪 10
先包 6
先去 237
先取 82
先史時代 1
先吃 25
先向 278
先君 5
先吸 2
先哭 0
先哲 11
先唱 3
先喝 15
先嘗 2
先嚐 1
先嚴 3
先回 59
先回來 4
先回到 6
先回去 15
先在 307
先報 79
先天 465
先天不良 12
先天不足 12
先天性 600
先夫 11
先套 0
先妣 0
先存 20
先守 4
先守後攻 0
先宰 3
先寄 30
先寫 179
先將 652
先導 194
先小人後君子 0
先就 78
先帝 20
先師 16
先帶 22
先府君 0
先延 0
先往 19
先後 937
先後順序 97
先得 50
先從 248
先想 85
先慈 1
先憂後樂 0
先應 11
先戰 0
先戴 2
先手 15
先打 99
先找 159
先把 362
先抱 2
先抽 11
先拍 60
先挑 90
先排 60
先採 17
先接 11
先撿 0
先攻後守 0
先救 16
先敷 0
先斬後奏 6
先斷 2
先於 80
先是 255
先會 7
先有 213
先服 9
先期 330
先機 158
先歉 0
先母 2
先民 247
先決 63
先決問題 0
先決條件 116
先河 34
先泡 9
先洗 9
先派 15
先減 14
先測 80
先演 2
先為 54
先烈 18
先照 20
先父 25
先猜 5
先王 20
先生 12761
先用 229
先發 250
先發制人 27
先的 44
先皇 5
先盛後衰 8
先看 643
先睹為快 153
先知 816
先知先覺 14
先砍 40
先祖 28
先禮後兵 6
先秦 175
先站 10
先端 49
先簽 17
先縫 0
先義後利 0
先考 9
先聊 10
先聲 43
先聲奪人 2
先能 21
先苦後甘 0
先行 978
先行者 10
先裁 0
先要 144
先見 22
先見之明 17
先覺 34
先討 12
先記 6
先調 9
先談 38
先請 73
先講 28
先讀 1217
先讓 150
先讓人 2
先賢 78
先贏後輸 0
先起來 5
先跑 16
先跳 10
先踩 0
先載 1
先輩 37
先輩遺訓 0
先逛 3
先進 3069
先進來 4
先進先出 19
先進去 6
先進國家 700
先進後出 3
先進集體 0
先過 11
先過來 1
先過去 2
先達 11
先遣 36
先遣部隊 9
先遣隊 1
先量 5
先釣 1
先鋒 396
先鋒隊 6
先開 50
先降 6
先頭 7
先頭部隊 5
先馳得點 10
先驅 212
先驗 18
先驗論 1
光 20415
光上 11
光下 19
光了 77
光亮 158
光亮亮 1
光來 26
光光 153
光光亮亮 0
光光明明 0
光光榮榮 0
光光滑滑 0
光化 39
光化作用 5
光化學 141
光去 2
光受 1
光吃 7
光合 43
光合作用 111
光向 3
光和 100
光唱 0
光唸 1
光喝 2
光圈 138
光在 96
光坦 0
光大 68
光大起來 0
光天化日 10
光天化日之下 10
光子 140
光子 140
光學 2372
光學玻璃 36
光守 0
光宗 10
光宗耀祖 5
光山 12
光州 21
光帶 20
光年 104
光度 136
光度表 0
光度計 74
光彩 185
光彩奪目 16
光彩耀眼 2
光彩起來 0
光影 166
光得 1
光復 3263
光復國土 0
光復國小 63
光復會 0
光復節 58
光念 0
光怪 3
光怪陸離 25
光想 15
光手 2
光採 2
光收 11
光敏 46
光數 10
光明 1377
光明日報 4
光明正大 45
光明燈 1
光明磊落 11
光明面 20
光是 435
光是在 21
光景 507
光會 46
光有 46
光束 179
光桿 3
光桿兒 0
光桿兒 0
光棍 10
光棍兒 0
光棍兒 0
光榮 527
光榮榜 0
光榮革命 3
光武 291
光武帝 6
光氣 44
光油油 0
光泉 58
光波 64
光波 64
光源 1034
光溜 4
光溜溜 9
光滑 199
光滑性 0
光滑滑 0
光潔 16
光潔度 1
光潤 10
光澤 299
光火 10
光焰 3
光照 183
光照階段 0
光燦 19
光物理 1
光猜 0
光環 151
光用 47
光的 410
光碟 11812
光碟機 2005
光碟片 1
光禿 10
光禿禿 9
光筆 28
光緒 256
光緒帝 6
光緒年間 21
光緒皇帝 4
光線 670
光纖 1261
光纖通信 87
光纜 72
光罩 100
光耀 46
光考 1
光聽 22
光能 37
光脆性 0
光腳 4
光膀子 0
光臨 4969
光色 31
光艷 0
光艷動人 0
光芒 392
光芒萬丈 11
光華 1854
光華商場 38
光華商場 383
光著 46
光著頭 1
光表 9
光要 17
光說 16
光說不做 0
光說不練 16
光譜 485
光譜儀 164
光譜分析 35
光譜線 4
光讀 10
光軸 19
光輝 377
光輝奪目 2
光輝燦爛 11
光輝耀眼 0
光輪 4
光輸 105
光輻射 2
光送 3
光送不賣 0
光速 153
光過 1
光采 67
光閃閃 9
光陰 130
光陰似箭 12
光陰荏苒 2
光電 3926
光電二極 0
光電二極管 1
光電子 168
光電效應 13
光電池 7
光電流 0
光電管 5
光靠 99
光面 64
光頭 206
光顧 88
光風霽月 2
光餅 4
光體 18
光鮮 51
光鮮亮麗 19
兊 0
兊 0
克 20000
克什米爾 4
克倫威爾 2
克儉 4
克分子 5
克分子濃度 1
克利 154
克利福洛 0
克制 121
克勤 70
克勤克儉 1
克原子 0
克原子 0
克己 16
克己奉公 0
克己復禮 6
克希荷夫 1
克希荷夫定律 1
克強 37
克復 5
克扣 0
克拉 332
克拉克 81
克敵 16
克敵制勝 5
克服 891
克服困難 36
克爾 64
克當量 7
克盡 41
克盡夫道 0
克盡婦道 0
克紹箕裘 2
克羅埃西亞 0
克羅齊 0
克苦耐勞 4
克萊 596
克裡 0
克裡姆林宮 0
克複 0
克郎 16
克里 272
克里夫蘭 52
克里姆林宮 12
克里島 0
克里特 7
克里特克 0
克里特克里島 0
克里特島 6
克里米亞 2
克里米亞半島 1
克里米亞戰爭 2
克難 27
克難 27
克難街 4
克食 1
克魯克斯 0
兌 681
兌付 22
兌坊 0
兌坊 0
兌換 362
兌換率 15
兌款 6
兌獎 30
兌現 158
免 78
免 7825
免不了 139
免不得 3
免不得 3
免俗 26
免刑 5
免去 105
免定 0
免強 10
免強 10
免役 64
免得 424
免戰牌 7
免掉 4
免於 379
免死狀 0
免洗 29
免洗筷 1
免洗餐具 91
免用 59
免疫 1437
免疫力 171
免疫學 234
免疫性 42
免疫系統 227
免疫血清 12
免票 12
免票入場 4
免票入場 4
免禮 0
免稅 640
免稅品 30
免稅商店 45
免罪 1
免罰 10
免職 91
免與 5
免訂 1
免訴權 0
免試 294
免調 2
免調 2
免談 67
免責權 27
免費 16750
免除 553
免驗 18
兎 0
兏 0
兏 0
兏 0
児 0
児 0
兑 0
兑 0
兒 7
兒 71
兒 7116
兒化 1
兒化韻 0
兒女 513
兒女之命 5
兒女債 0
兒女情長 1
兒女成群 1
兒媳 3
兒媳婦 7
兒媳婦兒 0
兒媳婦兒 0
兒子 1673
兒子 1673
兒孫 26
兒孫滿堂 2
兒孫自有兒孫福 20
兒戲 29
兒時 132
兒歌 144
兒科 324
兒科醫生 0
兒童 13985
兒童圖書 21
兒童圖書館 58
兒童團 2
兒童心理 111
兒童心理學 7
兒童教育 254
兒童文學 525
兒童期 34
兒童樂園 269
兒童節 50
兒童節目 94
兒童讀物 136
兒茶 26
兒茶素 1
兒茶酚 4
兒馬 3
兓 0
兔 1061
兔兒 18
兔兒 18
兔唇 11
兔子 340
兔子不吃窩邊草 0
兔崽子 13
兔年 27
兔死狐悲 3
兔死狗烹 1
兔毛 2
兔毫 0
兔肉 4
兔脣 0
兔脫 2
兔腳 1
兕 11
兖 0
兗 66
兗州 7
兘 0
兘 0
党 506
兜 362
兜住 2
兜兒 0
兜兒 0
兜兜 2
兜兜風 3
兜剿 0
兜售 57
兜圈 2
兜圈子 5
兜子 1
兜帽 1
兜攬 1
兜攬生意 0
兜率天 1
兜率天宮 1
兜生意 0
兜著 5
兜著走 2
兜風 45
兟 2
兠 0
兢 217
兢兢業業 47
兤 0
兤 0
入 15873
入不敷出 18
入世 45
入主 264
入主中原 1
入五 21
入伍 135
入伍通知 0
入伍須知 0
入伙 0
入侵 697
入倉 4
入冬 49
入出境管理局 0
入列 13
入口 1068
入口即化 0
入口稅 0
入味 21
入國問俗 0
入圍 311
入團 17
入土 15
入土為安 7
入地無門 0
入場 31
入場 317
入場券 14
入場券 149
入場式 0
入場式 0
入場證 21
入場證 219
入境 829
入境問俗 0
入境手續 15
入境簽證 16
入境證 30
入境隨俗 15
入夜 59
入夜後 28
入夢 52
入夥 10
入學 4094
入學年齡 32
入學校 14
入學考試 740
入定 18
入室 29
入室弟子 0
入寇 7
入射 44
入射光 3
入射波 5
入射波 5
入射線 1
入射角 10
入射點 0
入山證 1
入席 9
入帳 1
入店 13
入座 40
入庫 52
入役 0
入微 18
入手 52
入托 0
入教 4
入時 77
入會 3446
入木三分 2
入格 0
入梅 1
入款 6
入殮 51
入氣室 0
入水管 0
入流 16
入海 73
入海口 9
入港 48
入滲 17
入營 98
入獄 71
入畫 21
入直 3
入眼 14
入睡 105
入社 68
入神 28
入籍 67
入耳 28
入聲 44
入股 114
入股分紅 0
入貢 1
入賬 1
入贅 10
入超 46
入超國 2
入迷 23
入道 16
入選 529
入選作品 0
入選為 9
入鄉隨俗 1
入門 2594
入門書 0
入閣 42
入闈 0
入院 70
入骨 13
入魔 18
入黨 43
兦 0
兦 0
內 42895
內中 117
內主 64
內亂 54
內亂外患 5
內亂罪 6
內人 94
內伸 1
內侍 4
內侵 2
內偏 2
內側 206
內債 4
內傷 55
內兄 2
內公切線 0
內出 42
內出血 66
內分 170
內分泌 590
內分泌腺 22
內切圓 5
內制 2
內力 116
內功 78
內助 2
內務 34
內務府 4
內務櫃 3
內務部 1
內勤 44
內勤人員 11
內包 18
內史 10
內向 303
內含 708
內因 66
內圈 25
內園 3
內圓 8
內圓外方 1
內在 606
內在的 199
內在美 46
內地 198
內地人 3
內埔 556
內埔 556
內場 26
內場 26
內壢 89
內外 1085
內外並重 0
內外交困 1
內外夾擊 0
內外夾攻 5
內奸 14
內姪 0
內子 17
內存 33
內孫 0
內學 73
內宅 2
內定 312
內室 18
內容 64748
內容和形式 2
內容和形式 2
內層 99
內幕 456
內幕消息 26
內幕重重 0
內平 5
內府 1
內庭 8
內廳 19
內廷 4
內建 1498
內建式 41
內弟 0
內心 1095
內心世界 97
內心深處 112
內心裡 17
內心話 11
內思 25
內急 9
內患 1
內情 81
內情不詳 0
內憂 15
內憂外患 32
內應 166
內應力 4
內戮 0
內戰 138
內接 100
內接式 10
內插 68
內插法 4
內摩擦 3
內放 25
內政 351
內政府 4
內政部 3749
內政部長 119
內文 0
內斜 31
內斜視 10
內景 15
內有 995
內服 30
內服藥 2
內柔 0
內柔外剛 0
內江 19
內江街 18
內河 36
內流區域 1
內流河 3
內海 57
內海地區 2
內涵 1008
內港 11
內湖 1145
內湖區 959
內湖國中 25
內湖國小 19
內澇 1
內焰 1
內熱 24
內燃 14
內燃器 0
內燃機 67
內燃機車 2
內燄 0
內營力 2
內犯 0
內環 51
內環境 25
內用 0
內疚 31
內痔 3
內皮 97
內省 43
內省法 0
內眷 0
內碼 439
內神通外鬼 1
內科 2419
內科手術 0
內科醫生 13
內積 21
內經 93
內線 164
內線交易 72
內羅畢 0
內耗 5
內耳 55
內胎 40
內能 48
內膜 373
內臟 195
內至 84
內舉不避親 5
內華達州 20
內蒙 25
內蒙古 45
內蒙古自治區 23
內行 82
內行人 37
內行話 1
內衣 744
內衣褲 85
內裙 0
內裡 0
內製 20
內褲 296
內視鏡 1
內親 8
內觀 10
內角 140
內角球 14
內訌 46
內調 13
內變 12
內賊 2
內轉 31
內邊 2
內部 4962
內部矛盾 3
內部聯系 0
內部聯繫 0
內野 61
內野區 0
內野手 25
內銷 3756
內銷品 10
內銷貨 0
內鋪 1
內錯角 0
內門 222
內閣 1180
內閣制 103
內閣大臣 0
內閣會議 13
內閣總理 22
內防 7
內阻 3
內降 0
內降 0
內院 18
內陸 239
內陸國 3
內陸地區 19
內陸河 1
內陸海 1
內隔 4
內電阻 5
內需 0
內面 33
內頁 556
內項 4
內顧之憂 2
內食 2
內鬥 46
內鬨 51
全 35212
全下 2
全不 75
全不是 3
全不會 0
全不能 0
全世界 2372
全乾 8
全人 212
全倒 44
全像 199
全出 8
全出來 0
全出去 0
全到 2
全副 47
全副武裝 16
全副精力 0
全力 5261
全力以赴 232
全力支持 138
全勝 67
全勝記錄 0
全勤 24
全勤獎 6
全勤獎金 3
全區 0
全去 2
全反射 6
全台 574
全台灣 435
全名 257
全名是 33
全向 10
全因 11
全國 10309
全國上下 18
全國人民 174
全國各地 142
全國同胞 30
全國性 589
全國紀錄 32
全國運動 10
全國運動會 8
全城 52
全域 0
全域變數 0
全場 28
全場 287
全場歡呼 0
全場歡呼 0
全壘打 448
全天 256
全天候 255
全天性 2
全套 433
全套教材 0
全套設備 3
全始全終 2
全家 824
全家樂 11
全家福 297
全對 13
全局 64
全市 400
全師 2
全師官兵 0
全年 1205
全年收入 8
全年雨量 2
全形 46
全彩 100
全心 160
全心全意 78
全心投入 55
全息 8
全意 9
全憑 57
全所 13
全所同仁 2
全才 14
全打 6
全數 481
全文 7716
全文如下 9
全新 2368
全日 364
全日制 13
全是 433
全景 223
全書 528
全會 141
全有 49
全材 0
全村 54
全校 1019
全校同學 26
全校學生 81
全校師生 365
全權 71
全權代表 3
全權處理 12
全權處理 12
全毛 0
全毛料 0
全民 5147
全民健保 743
全民所有 4
全民所有制 11
全民投票 2
全民政治 11
全民皆兵 3
全民運動 148
全為 65
全無 135
全然 325
全然不同 23
全然不知 4
全然不解 1
全片 70
全班 325
全班同學 45
全班學生 5
全球 101898
全球化 397
全球性 398
全生育期 0
全當 7
全盛 29
全盛時期 32
全盤 307
全盤否定 8
全盤托出 0
全盤接受 7
全盤皆輸 6
全盤考慮 12
全省 1
全看 37
全真 18
全真教 18
全瞎 3
全神 18
全神貫注 29
全票 111
全票價 0
全程 679
全程參加 12
全稱 96
全等 35
全等號 0
全篇 54
全篇文章 2
全組 49
全線 116
全線作戰 0
全線出擊 0
全線貫通 1
全線通車 25
全縣 425
全美 838
全美國 55
全考 0
全職 433
全能 386
全能運動 3
全能運動員 2
全脂 10
全脂奶粉 9
全蝕 28
全貌 154
全責 29
全距離 0
全身 1673
全身上下 50
全身像 2
全身檢查 10
全身而退 29
全軍 38
全軍覆沒 34
全速 37
全速前進 2
全速進行 1
全過 2
全選 158
全部 9228
全部是 43
全部會 40
全部有 50
全部都 135
全部都是 75
全都 390
全都是 115
全都會 40
全都有 40
全長 290
全開 86
全陪 4
全隊 90
全集 566
全靠 85
全面 3506
全面實施 142
全面性 344
全面戰爭 3
全面發展 27
全面規劃 9
全面進行 15
全音 16
全音符 4
全項 3
全額 266
全額交割 0
全額交割股 1
全額獎學金 13
全額獎金 0
全體 2342
全體利益 1
全體同仁 118
全體官兵 7
全體師生 106
全黨 52
全黨同志 3
兩 18003
兩下 215
兩下子 10
兩人 3436
兩人三腳 1
兩人份 16
兩人同心 0
兩代 59
兩件 165
兩件事 76
兩任 26
兩份 128
兩伊 8
兩位 1626
兩位數 36
兩便 3
兩個 17146
兩個人 711
兩個月 601
兩倍 207
兩側 529
兩儀 93
兩億 63
兩億人 5
兩億元 22
兩元 29
兩克 0
兩全 18
兩全其美 31
兩兩 22
兩具 23
兩冊 35
兩刀 18
兩分 87
兩分鐘 65
兩利 4
兩劑 5
兩千 1170
兩千人 15
兩千個 6
兩千元 58
兩千塊 20
兩半 48
兩口 36
兩口子 5
兩口氣 3
兩句 174
兩句話 66
兩可 20
兩可之間 0
兩台 257
兩周 27
兩周年 4
兩員 5
兩回 14
兩回事 38
兩圈 6
兩國 935
兩國人民 20
兩圓 3
兩地 328
兩地相思 20
兩堆 30
兩塊 97
兩塊錢 30
兩壺 40
兩夜 98
兩大 952
兩大塊 20
兩大片 20
兩大類 116
兩天 10550
兩天晒網 0
兩女 42
兩女一男 3
兩女一男 3
兩女兩男 0
兩季 51
兩客 5
兩害相權 3
兩害相權取其輕 6
兩家 653
兩家人 1
兩審終審 0
兩審終審制 0
兩封 35
兩封信 6
兩對 60
兩小時 125
兩小無猜 14
兩尾 7
兩局 14
兩屆 82
兩層 103
兩層樓 32
兩岸 10176
兩岸通航 25
兩岸關係 1314
兩市 164
兩師 0
兩席 21
兩幕間 0
兩年 2211
兩年生 0
兩度 166
兩廂情願 1
兩廣 8
兩廣總督 4
兩廳 69
兩式 4
兩性 1270
兩性之間 32
兩性人 0
兩性花 2
兩性間 11
兩情相悅 26
兩成 131
兩截 10
兩截式 3
兩戶 17
兩房 25
兩手 228
兩手俱利 0
兩手空空 15
兩打 3
兩把 61
兩把刷子 6
兩拍 4
兩拳 4
兩排 33
兩支 147
兩敗俱傷 23
兩方 133
兩方面 209
兩旁 265
兩日 163
兩日份 0
兩晉 8
兩晚 16
兩期 72
兩本 136
兩本書 65
兩朵 17
兩朵花 5
兩杯 55
兩板 3
兩枝 22
兩架 41
兩格 16
兩桶 4
兩桿 8
兩條腿走路 3
兩條路 29
兩條道路 0
兩梯 1
兩梯次 21
兩棲 78
兩棲作戰 4
兩棲動物 12
兩棲登陸 8
兩棲部隊 7
兩棲類 62
兩棵 25
兩極 140
兩極分化 0
兩極化 97
兩極管 0
兩槍 10
兩槓 0
兩樣 192
兩檔 25
兩權分立 0
兩次 1052
兩次三番 0
兩次運球 0
兩步 392
兩歲 177
兩段 99
兩段制 0
兩段式 34
兩段票 0
兩江 13
兩河 14
兩派 64
兩派人馬 9
兩浙 10
兩淮 5
兩湖 8
兩湖盆地 3
兩湖總督 0
兩滴 6
兩漢 62
兩潘 0
兩版 17
兩班 84
兩班制 2
兩瓣 0
兩用 204
兩男 124
兩男一女 4
兩男一女 4
兩男兩女 2
兩百 686
兩百人 18
兩百個 16
兩百元 49
兩百塊 9
兩相 69
兩相情願 1
兩眼 194
兩眼發直 3
兩碗 10
兩碗飯 1
兩碼 33
兩碼事 0
兩碼子 4
兩票 35
兩科 70
兩秒 14
兩秒鐘 10
兩種 2656
兩種人 28
兩立 7
兩端 248
兩筒 0
兩箱 4
兩節 38
兩節課 3
兩粒 34
兩粒裝 0
兩粵 1
兩粵總督 0
兩級 53
兩缸 1
兩翼 26
兩者 1465
兩者缺一不可 2
兩耳 29
兩聯 5
兩聯式 2
兩肋 11
兩胎 1
兩腳 51
兩腳書櫥 0
兩腳規 0
兩舌 1
兩萬 216
兩萬人 21
兩萬個 6
兩萬元 43
兩萬塊 4
兩葉 7
兩虎相鬥 0
兩虎相鬥必有一傷 0
兩號 0
兩袋 1
兩袖 3
兩袖清風 4
兩角 13
兩訖 0
兩記 0
兩課 3
兩起 45
兩路 55
兩路橫隊 0
兩路縱隊 0
兩軍 67
兩軍人馬 0
兩輛 29
兩輛車 5
兩通 2
兩造 150
兩連 0
兩週 335
兩週年 20
兩遍 6
兩道 75
兩道菜 4
兩邊 338
兩邊倒 0
兩邊討好 0
兩部 392
兩重 20
兩重性 2
兩重意義 1
兩錢 3
兩鍋 19
兩間 77
兩院 76
兩院制 67
兩隊 145
兩階 4
兩隻 453
兩集 33
兩雙 28
兩難 125
兩霸 0
兩面 228
兩面作戰 0
兩面倒 0
兩面光 0
兩面夾攻 0
兩面性 1
兩面手法 16
兩面派 0
兩面討好 0
兩頁 41
兩項 661
兩頭 81
兩頭尖 2
兩頭為難 0
兩頭空 2
兩頭落空 4
兩頭蛇 2
兩頭跑 9
兩頰 34
兩顆 254
兩題 8
兩首 116
兩首歌 20
兩點 360
兩點鐘 7
兩黨 619
兩黨制 2
兪 0
兪 0
兪 0
八 27729
八七水災 7
八下 2
八世 19
八世紀 16
八九 668
八九不離十 25
八仙 196
八仙桌 6
八仙洞 55
八仙過海 9
八份 28
八位 159
八位數 171
八佾 0
八佾舞 0
八個 663
八個 663
八克 3
八具 3
八分 105
八分熟 1
八分鐘 31
八分音符 0
八包 0
八區 22
八十 40415
八十種好 1
八千 613
八卦 2136
八卦丹 0
八卦山 133
八卦陣 12
八句 5
八台 29
八呎 5
八周 3
八周年 7
八哥 94
八哥兒 0
八哥兒 0
八喜 1
八圈 2
八國 41
八國聯軍 27
八場 41
八場 41
八堵 61
八塊 16
八塊錢 2
八夜 2
八大 336
八大山人 5
八大明王 1
八大胡同 0
八大金剛 1
八天 152
八字 374
八字憲法 0
八字腳 1
八字鬍 6
八季 2
八客 0
八家 133
八寶 67
八寶粥 21
八寶飯 2
八封 4
八小 4
八小時 125
八尺 22
八局 34
八局上半 4
八局下半 2
八屆 706
八層 33
八層樓 5
八席 14
八年 5059
八年級 15
八度 145
八廠 3
八張 86
八德 1930
八成 364
八成新 18
八戒 12
八截 0
八戶 38
八扇 0
八打 2
八折 196
八拜 1
八拜之交 2
八拳 0
八掌溪 45
八支 28
八斗子 90
八斤 1
八斤重 0
八方 186
八旗 4
八日 1803
八日上午 29
八日下午 41
八時 416
八晚 0
八景 87
八月 5546
八月份 604
八期 597
八朵 0
八枝 2
八格 11
八桶 0
八棵 2
八樓 201
八次 177
八歌 0
八正道 1
八步 9
八歲 205
八歲時 13
八段錦 4
八營 0
八爺 4
八版 23
八珍 7
八班 86
八百 858
八百壯士 1
八盒 0
八目鰻 3
八碗 1
八科 12
八秒 14
八秒鐘 0
八站 3
八章 141
八筒 0
八節 32
八粒 5
八級 30
八線 24
八罐 0
八股 42
八股文 2
八腳 4
八色鳥 1
八苦 1
八荒 7
八萬 311
八萬四千法門 1
八號 720
八號分機 0
八行書 1
八街 68
八角 85
八角形 15
八角茴香 3
八角錢 0
八課 5
八識 1
八路 38
八路軍 8
八通 15
八通關 200
八週 93
八週年 35
八進位 3
八遍 1
八邊 2
八邊形 1
八邪 1
八部 144
八里 327
八里鄉 182
八錢 1
八門 16
八開 16
八開本 1
八關 46
八陣圖 6
八隻 34
八集 82
八面 49
八面威風 0
八面玲瓏 25
八音 131
八音盒 0
八音節 0
八頁 107
八顆 23
八類 19
八風 1
八風吹不倒 1
八風吹不動 1
八首 31
八首歌 1
八點 517
八點檔 0
八點鐘 13
公 18783
公丈 0
公主 750
公事 164
公事公辦 0
公事包 163
公事房 0
公人 23
公仔 100
公休 33
公休日 3
公佈 5077
公佈出來 14
公佈欄 2992
公使 29
公使團 0
公使館 3
公侯 2
公侯伯子男 1
公保 106
公保大樓 10
公信力 206
公俸 0
公倍數 8
公假 80
公債 548
公債券 1
公僕 36
公價 1
公允 29
公元 1952
公元前 51
公克 680
公兔 2
公兩 5
公公 107
公公 107
公公平平 0
公公正正 0
公共 5838
公共事業 47
公共場所 30
公共場所 305
公共安全 479
公共廁所 20
公共政策 263
公共汽車 92
公共汽車站 2
公共租界 0
公共秩序 44
公共積累 0
公共行政 631
公共衛生 990
公共設施 923
公共財產 2
公共責任 1
公共道德 2
公共關係 277
公共電視 136
公共電話 50
公出 12
公函 53
公分 2701
公分子 0
公分母 0
公切線 1
公制 47
公劉 0
公務 1018
公務人員 2529
公務員 998
公勺 0
公升 464
公卿 33
公厘 123
公司 91055
公司債 520
公司法 261
公告 21713
公告價格 3
公告地價 47
公告欄 295
公器 0
公噸 1228
公因式 6
公因數 16
公國 58
公園 4548
公地 101
公垂線 1
公堂 15
公報 1622
公報私仇 6
公墓 158
公奴 0
公娼 254
公婆 60
公子 671
公子哥兒 1
公子哥兒 1
公子王孫 0
公孫龍 5
公安 259
公安人員 14
公安局 28
公安法 0
公安派 1
公安部 10
公安部隊 0
公定 21
公定價 1
公定價格 2
公害 822
公家 114
公家機構 22
公家機關 121
公寓 1339
公審 12
公寸 3
公尺 3381
公尺見方 5
公差 108
公差費 0
公布 3722
公布欄 203
公帑 55
公帳 2
公平 3507
公平合理 52
公平性 116
公平會 0
公平無私 2
公幹 8
公庫 103
公廁 119
公式 604
公式化 36
公引 0
公役 0
公德 14
公德心 63
公忠體國 5
公意 3
公憤 15
公懲會 0
公所 1172
公才公望 0
公投 548
公推 32
公撮 8
公教 258
公教人員 283
公敵 19
公文 1310
公文信 0
公文書 25
公文袋 3
公斗 0
公斤 1948
公斷 17
公方 2
公方代表 0
公族 1
公曆 2
公會 3375
公有 358
公有制 11
公有土地 107
公案 53
公檢法 0
公權 19
公權力 245
公款 57
公正 644
公正不阿 0
公正性 43
公正無私 7
公歷 0
公比 11
公毫 0
公民 970
公民投票 248
公民權 47
公民營 534
公民科 16
公民課 4
公決 10
公法 153
公海 51
公海自由 2
公演 421
公然 209
公然侮辱 11
公然侮辱 11
公然反對 2
公然反抗 0
公營 361
公營事業 620
公營企業 133
公爵 134
公爵夫人 6
公爾忘私 0
公牘 0
公牛 296
公物 78
公狗 59
公狼 1
公獅 8
公理 72
公理化 3
公產 23
公用 1245
公用事業 451
公用程式 147
公用電話 93
公畜 3
公畝 34
公益 2402
公益事業 45
公益彩券 1
公益金 25
公眾 745
公石 3
公示 32
公社 65
公社化 0
公祭 70
公私 485
公私不分 5
公私交迫 0
公私兩便 0
公私兩利 0
公私兼顧 2
公私分明 5
公私合營 6
公秉 62
公稱 8
公積金 20
公立 594
公立學校 372
公立醫院 132
公章 1
公筷 1
公筷母匙 6
公糧 16
公約 1334
公約國 4
公約數 6
公署 108
公羊 55
公羊傳 1
公義 304
公老虎 0
公而忘私 1
公職 491
公職人員 322
公聽並觀 0
公聽會 382
公舉 0
公虎 0
公蟻 0
公視 0
公設 135
公設辯護 0
公設辯護人 12
公訴 34
公訴人 3
公証 46
公評 0
公認 272
公認是 17
公認為 61
公說公有理 6
公論 25
公諸同好 2
公諸於世 39
公證 337
公證人 56
公證會 0
公證結婚 15
公證處 25
公議 6
公象 2
公豬 101
公費 624
公費生 145
公費留學 137
公賣 39
公賣局 189
公路 2325
公路局 749
公車 1336
公車票 6
公車站 86
公車管理 0
公車管理處 4
公車處 26
公輸 4
公轉 72
公道 274
公道話 23
公里 2028
公釐 174
公錢 1
公門 4
公門桃李 0
公開 5385
公開信 98
公開出來 1
公開性 9
公開競選 0
公開賽 275
公關 823
公雞 41
公頃 1547
公項 1
公餘 35
公餘之暇 5
公館 650
公馬 2
公鵝 0
公鹿 6
公鼠 0
六 23418
六下 28
六世 27
六世紀 21
六串 0
六人 370
六人份 2
六任 37
六份 33
六位 331
六位數 90
六信 21
六個 780
六個人 50
六個月 1135
六億 227
六億人 1
六億元 115
六入 1
六兩 17
六六六 14
六六大順 0
六具 3
六凡四聖 1
六刀 2
六分 166
六分之一 30
六分儀 20
六分鐘 44
六劃 39
六區 43
六十 4896
六十年代 116
六千 736
六千人 27
六千個 9
六千元 121
六千塊 4
六口 23
六口之家 1
六句 12
六句話 1
六合 286
六合彩 40
六名 220
六周 8
六周年 2
六和 163
六哥 15
六回 78
六國 121
六場 95
六場 95
六堵 24
六塊 37
六塊錢 0
六塵 1
六境 1
六壺 0
六壺酒 0
六大 2579
六大塊 0
六大片 0
六大類 39
六妹 25
六季 2
六安 20
六家 297
六小 9
六小時 88
六尺之驅 0
六尾 2
六局 37
六局上半 2
六局下半 0
六屆 865
六巷 72
六師 8
六年 10740
六年級 575
六度 68
六度萬行 1
六廠 10
六弄 11
六式 18
六弟 2
六弦琴 8
六張 112
六張犁 20
六成 421
六成新 1
六截 0
六戶 25
六房 3
六手 2
六打 1
六折 105
六排 4
六支 49
六斤 1
六斤重 0
六日 1964
六星 1
六星彩 1
六星期 23
六時 165
六晚 17
六曲 2
六書 10
六月 5236
六月份 460
六月雪 0
六朝 84
六期 699
六枚 9
六枝 1
六株 2
六根 50
六根清淨 4
六格 4
六桶 0
六條 1987
六梯 0
六梯次 2
六棟 9
六棵 1
六樓 734
六欄 1
六次 199
六欲 3
六步 7
六歲 352
六歲時 14
六法 77
六法全書 85
六波羅蜜 1
六波羅蜜 1
六洞 3
六派 1
六滴 0
六營 8
六版 57
六班 139
六瓣 1
六甲 455
六畜 5
六發 5
六百 1010
六百人 19
六百個 10
六百元 65
六百塊 0
六碗 2
六神無主 14
六禮 4
六科 35
六種 214
六章 173
六筒 0
六箱 0
六節 65
六節詩 0
六節課 2
六簍 0
六籃 0
六粒 5
六粒裝 0
六級 92
六級風 1
六經 7
六線 31
六聯 12
六聲 3
六腑 5
六腳 249
六船 0
六萬 258
六萬人 26
六萬個 11
六萬元 149
六萬塊 0
六藝 18
六處 71
六號 695
六街 116
六親 14
六親不認 8
六親無靠 0
六角 20
六角形 20
六課 13
六起 5
六路 191
六路橫隊 0
六路縱隊 0
六軍 1
六輕 312
六輛 9
六通 9
六連 17
六週 149
六週年 23
六遍 2
六道 90
六道輪迴 0
六邊 2
六邊形 4
六部 86
六錢 7
六鍋 0
六鎮 5
六門 1
六階 6
六隻 41
六集 97
六面 27
六面體 4
六韜 10
六項 252
六顆 23
六題 15
六類 105
六首 37
六首歌 1
六點 456
六點鐘 16
六齣 6
六龜 439
兮 426
共 113
共 113
共 11347
共上 6
共下 8
共主 7
共乘 0
共事 82
共享 1767
共來 3
共保 6
共價 7
共價鍵 11
共出 12
共切 1
共匪 27
共印 2
共去 0
共取 8
共吃 4
共同 11645
共同市場 75
共同市場 75
共同性 74
共同海損 7
共同綱領 1
共同語 0
共同體 244
共同點 35
共和 297
共和制 9
共和國 654
共和軍 5
共和黨 282
共唱 0
共商 88
共商國事 1
共圖 27
共圖國事 0
共在 12
共基極 2
共奪 3
共存 289
共存亡 60
共學 7
共守 8
共對 238
共屬 2
共得 34
共從 8
共性 5
共患難 10
共抵 0
共拜 0
共持 4
共振 324
共振效應 0
共推 16
共擺 0
共效 1
共有 3926
共析 0
共析鋼 2
共棲 4
共榮互利 0
共樂 2
共歉 0
共治 0
共法 1
共測 1
共濟會 1
共犯 86
共獲 20
共生 302
共生死 2
共產 138
共產主義 158
共產主義者 7
共產主義道德 0
共產制 1
共產國際 10
共產黨 369
共產黨人 6
共產黨員 16
共產黨宣言 4
共用 838
共登 3
共發射極 0
共相 0
共睡 1
共睡一床 0
共睡一床 0
共研 4
共端 0
共管 8
共約 115
共給 5
共聚 50
共聚一堂 32
共聚一堂 32
共聚反應 1
共聚物 24
共背 0
共舞 223
共苦 1
共處 50
共處一室 4
共處一室 4
共虧 0
共襄盛舉 274
共要 34
共計 1105
共設 71
共診 0
共請 0
共謀 70
共謀國事 0
共識 1646
共買 6
共費 1
共賺 1
共赴國難 1
共軍 170
共軛 47
共軛像 0
共軛複數 1
共軛角 0
共軛點 0
共載 1
共輸 2
共轉 3
共通 330
共通性 74
共通點 31
共進 86
共進會 8
共運 3
共邀 11
共鋪 0
共開 22
共降 4
共需 48
共青團 6
共養 7
共騎 3
共體時艱 0
共鳴 265
共鳴器 4
共黨 116
兲 0
关 0
兵 1931
兵丁 12
兵不厭詐 1
兵不厭詐 1
兵不血刃 7
兵不血刃 7
兵事 1
兵備 6
兵兵 6
兵刃 73
兵刃相接 0
兵制 7
兵力 280
兵卒 7
兵員 29
兵器 213
兵團 66
兵士 36
兵多將廣 1
兵學 20
兵家 22
兵家大忌 0
兵家必爭 28
兵家必爭之地 24
兵將 13
兵工 29
兵工廠 26
兵工署 1
兵強馬壯 1
兵役 704
兵役法 30
兵役節 0
兵患 0
兵戎相見 7
兵敗如山倒 1
兵書 14
兵權 7
兵法 147
兵源 22
兵災 0
兵營 42
兵營區 0
兵燹 3
兵甲 8
兵略 4
兵禍 1
兵科 11
兵種 71
兵站 1
兵符 25
兵籍 18
兵籍卡 0
兵籍號碼 4
兵籍表 1
兵臨城下 8
兵船 2
兵艦 1
兵荒馬亂 22
兵變 25
兵貴神速 3
兵連禍結 1
兵部 12
兵餉 0
兵馬 50
兵馬俑 21
其 1
其 66108
其一 595
其中 13312
其事 109
其二 221
其他 34169
其他人 649
其他應計負債 0
其他支出 152
其他的 2137
其他費用 36
其內 218
其勢兇兇 0
其外 50
其它 12258
其實 8902
其實不然 55
其實在 210
其實是 831
其實有 130
其後 552
其情可憫 0
其時 106
其樂無窮 4
其次 1125
其次是 295
其為 378
其為 378
其貌不揚 13
其間 712
其餘 2197
其餘的 260
具 13920
具備 3328
具名 133
具報 13
具文 33
具有 11016
具現 33
具結 28
具結書 14
具象 33
具足戒 1
具體 2606
具體勞動 0
具體化 46
具體化起來 0
具體地說 7
具體性 9
具體物 6
具體而微 23
典 2594
典型 893
典型人物 0
典型化 0
典型性 3
典型調查 0
典押 1
典故 132
典業 3
典獄官 0
典獄長 28
典當 92
典禮 662
典章 35
典範 450
典籍 111
典藏 1202
典試 31
典試委員 16
典試委員會 214
典賣 5
典雅 258
兼 3526
兼之 50
兼了 3
兼任 1850
兼任教師 251
兼併 30
兼修 46
兼備 89
兼具 881
兼取 1
兼善 5
兼善天下 6
兼容 36
兼容並包 3
兼容並蓄 32
兼差 118
兼并 10
兼得 25
兼愛 16
兼收 340
兼收並蓄 1
兼旬 0
兼營 194
兼祧 0
兼程 11
兼管 22
兼籌並顧 17
兼而有之 7
兼職 579
兼行 72
兼課 64
兼辦 88
兼通 3
兼顧 872
兾 0
冀 291
冀望 98
冂 0
冂 0
冃 0
冄 0
内 0
内 0
円 0
冇 103
冈 0
冉 84
冉冉 24
冉冉上升 6
冊 2822
冊子 30
冊封 15
冊文 2
冊立 3
冊葉 0
冊頁 20
冋 0
册 0
册 0
再 36484
再三 203
再三再四 0
再三的 14
再上 140
再上來 32
再上去 8
再下 144
再下來 14
再下去 8
再不 325
再不來 6
再不到 0
再不去 2
再也 513
再也不 99
再也不是 23
再也不會 42
再也不肯 2
再也不能 41
再也沒有 133
再來 1459
再來一個 10
再來一個 10
再來一個 10
再依 268
再保險 40
再倒 25
再借 13
再借不難 2
再做 351
再入 25
再再 51
再出 163
再出來 17
再出去 2
再出現 83
再到 265
再刻 1
再則 60
再剪 7
再割 4
再去 560
再取 32
再向 254
再咬 9
再唱 27
再問 351
再回 166
再回來 92
再回到 148
再回去 21
再在 143
再壓 13
再大 70
再好 120
再好不過 12
再好也不過 2
再娶 8
再婚 38
再婚者 2
再嫁 9
再嫁夫人 0
再學 18
再寄 97
再審 93
再就是 7
再度 2959
再延 23
再往 169
再想 134
再慢 14
再打 182
再把 0
再押 1
再拜 15
再按 721
再接 109
再接再厲 79
再搖 0
再搬 7
再撿 7
再擠 2
再教育 62
再於 105
再會 118
再有 493
再次 2334
再殺 14
再沒 56
再沒有 80
再混 5
再滿 1
再演 24
再為 197
再煎 1
再燃 4
再版 99
再版書 0
再現 419
再生 1894
再生制動 0
再生父母 3
再生產 22
再生緣 4
再看 483
再移 30
再種 3
再笑 7
再答 9
再結晶 9
再者 519
再與 222
再補 78
再製品 3
再製鹽 0
再要 31
再見 645
再見面 27
再討 3
再記 8
再試 74
再試一下 17
再試一下 17
再試一次 30
再試一次 30
再認 20
再說 819
再說下去 11
再請 197
再議 0
再讀 26
再賠 2
再賭 5
再贏 4
再起 188
再起來 2
再跟 181
再辦 60
再逃 4
再造 890
再進 379
再進來 8
再進去 11
再運 22
再過 208
再過來 6
再過去 11
再配 80
再醮 0
再陪 12
再騎 1
冎 0
冎 0
冏 9
冐 0
冐 0
冑 38
冒 406
冒 406
冒上 4
冒上來 0
冒上去 0
冒了 18
冒充 54
冒冒 0
冒冒失失 4
冒冒險 0
冒出 214
冒出來 35
冒升 1
冒口 2
冒名 57
冒名頂替 14
冒失 8
冒失鬼 0
冒往 0
冒昧 38
冒死 10
冒氣 5
冒汗 21
冒泡 16
冒瀆 3
冒火 7
冒然 39
冒煙 46
冒牌 18
冒牌貨 8
冒犯 66
冒犯之處 6
冒用 140
冒稱 4
冒著 154
冒著危險 5
冒號 16
冒血 0
冒貸 3
冒貸案 0
冒起 8
冒起來 0
冒進 10
冒險 1079
冒險主義 0
冒險家 32
冒險性 12
冒險犯難 30
冒雨 46
冒雪 1
冒頂 0
冒領 19
冓 21
冔 14
冕 333
冖 0
冖 0
冗 134
冗位 0
冗員 7
冗數 0
冗筆 0
冗職 0
冗言 0
冗詞 0
冗詞贅句 0
冗語 0
冗費 0
冗贅 2
冗長 79
冗雜 0
冘 3
写 0
写 0
冝 0
冞 237
冟 0
冠 369
冠 1191
冠亞軍 22
冠亞軍 22
冠冕 37
冠冕堂皇 20
冠心病 69
冠毛 3
冠狀 57
冠狀動脈 209
冠狀動脈 209
冠玉 6
冠王 43
冠禮 3
冠絕 7
冠絕群芳 0
冠群 96
冠蓋 3
冠蓋雲集 10
冠詞 20
冠軍 1558
冠軍賽 100
冡 0
冢 34
冣 0
冤 115
冤仇 18
冤冤相報 9
冤大頭 28
冤孽 4
冤家 35
冤家路窄 7
冤屈 18
冤情 12
冤有頭債有主 2
冤枉 157
冤案 13
冤獄 40
冤讎 0
冤鬼 0
冤魂 26
冥 465
冥冥 32
冥冥中 26
冥冥之中 10
冥器 0
冥報 2
冥壽 2
冥婚 1
冥府 5
冥思 10
冥想 239
冥晦 0
冥王 0
冥王星 69
冥紙 24
冥衣 0
冥誕 27
冥錢 0
冥間 0
冥頑 5
冥頑不靈 10
冦 0
冨 0
冪 1325
冬 20016
冬令 164
冬令救濟 12
冬冬 35
冬天 839
冬季 727
冬季世界 0
冬學 1
冬小麥 3
冬日 61
冬暖 6
冬暖夏涼 13
冬烘 20
冬瓜 64
冬瓜湯 0
冬瓜茶 17
冬眠 48
冬眠期 2
冬筍 5
冬節 3
冬粉 16
冬耕 0
冬至 66
冬菇 41
冬藏 1
冬蟲夏草 50
冬衣 24
冬裝 6
冬閑 0
冬閒 0
冬防 3
冬雨 7
冬雪 33
冬霧 0
冬青 14
冬青樹 0
冬風 5
冬麥 5
冭 0
冰 2948
冰上 81
冰上表演 0
冰上運動 0
冰了 3
冰人 4
冰冰 719
冰冰冷冷 2
冰冰涼涼 1
冰冷 266
冰凌 1
冰凍 80
冰凍三尺 10
冰凍食品 0
冰刀 12
冰品 0
冰場 3
冰場 3
冰塊 137
冰壑 0
冰天雪地 31
冰好 1
冰宮 28
冰寒 8
冰封 36
冰屑 1
冰山 127
冰山一角 0
冰山一角 0
冰山美人 1
冰島 122
冰川 73
冰床 2
冰店 17
冰廠 4
冰成 1
冰敷 1
冰斧 1
冰晶 40
冰晶石 2
冰期 35
冰枕 10
冰柱 6
冰棍兒 0
冰棍兒 0
冰棒 56
冰櫃 13
冰水 230
冰沙 1
冰河 195
冰河時期 13
冰河期 29
冰海 10
冰消瓦解 1
冰涼 68
冰淇淋 296
冰清玉潔 10
冰炭 1
冰片 8
冰球 33
冰盞兒 0
冰盞兒 0
冰窖 3
冰箱 731
冰箱門 4
冰糕 2
冰糖 50
冰糖水 3
冰罅 0
冰脊 0
冰脊 0
冰蝕湖 0
冰袋 7
冰起 0
冰起來 5
冰輪 1
冰過 8
冰釋 10
冰鎮 1
冰雕 28
冰雪 93
冰雪聰明 16
冰雪遍地 0
冰雪遍地 0
冰雹 65
冰霜 16
冰鞋 2
冰點 40
冱 96
冲 0
决 0
冴 0
况 0
冶 360
冶工 3
冶煉 29
冶煉廠 4
冶豔 6
冶遊 2
冶金 161
冶金學 10
冶金工業 12
冶金業 6
冶鐵 3
冷 3241
冷下來 1
冷不冷 3
冷不防 24
冷作 20
冷僻 16
冷光 48
冷冰冰 34
冷冷 221
冷冷淡淡 1
冷冷清清 26
冷冷的 136
冷冷靜靜 1
冷凍 2666
冷凍劑 30
冷凍室 7
冷凍庫 1
冷凍機 44
冷凍櫃 54
冷凍箱 2
冷凍食品 1
冷凝 86
冷凝劑 0
冷凝器 54
冷加工 2
冷卻 767
冷卻劑 11
冷卻器 46
冷嘲 3
冷嘲熱諷 17
冷嘲熱諷 17
冷場 15
冷場 15
冷天 12
冷媒 447
冷宮 9
冷寂 3
冷峻 21
冷巷 1
冷庫 1
冷得 19
冷感症 3
冷戰 305
冷敷 17
冷敷法 1
冷暖 125
冷暖人間 4
冷暖自知 29
冷杉 33
冷杉木 0
冷板凳 7
冷槍 0
冷氣 920
冷氣團 99
冷氣機 835
冷氣設備 13
冷水 174
冷水器 0
冷水機 5
冷水浴 1
冷汗 130
冷汗直流 4
冷泉 87
冷洗精 2
冷流 4
冷淡 121
冷清 172
冷清清 7
冷漠 300
冷澀 1
冷熱 194
冷熱病 0
冷療法 1
冷的 161
冷盤 32
冷眼 109
冷眼旁觀 27
冷眼相待 1
冷笑 154
冷笑話 154
冷箭 13
冷色 6
冷若 75
冷若冰霜 7
冷落 55
冷藏 428
冷藏器 0
冷藏室 15
冷藏庫 27
冷藏櫃 6
冷藏法 0
冷藏箱 133
冷藏車 5
冷血 100
冷血動物 11
冷言冷語 4
冷言諷語 0
冷話 0
冷豔 2
冷軋 81
冷軋機 7
冷軋鋼板 1
冷酷 159
冷酷無情 21
冷醃法 0
冷鋒 76
冷鑄 0
冷門 84
冷門課 0
冷霜 16
冷靜 387
冷靜下來 25
冷面 22
冷風 92
冷颼颼 24
冷食 4
冷食部 0
冷飲 69
冷飲店 23
冷飲部 3
冸 0
冹 3
冺 0
冺 0
冼 368
冼星海 0
冽 41
冽泉 0
冾 186
冿 0
净 0
凁 0
凂 0
凂 0
凂 0
凃 0
凄 61
凅 90
准 1465
准予 424
准假 20
准將 21
准考證 243
准許 360
凇 0
凈 49
凉 0
凉 0
凊 35
凊 35
凋 133
凋敝 4
凋殘 2
凋萎 13
凋落 14
凋謝 39
凋逝 1
凋零 79
凌 1967
凌亂 64
凌亂不堪 10
凌人 5
凌厲 70
凌厲攻勢 1
凌晨 913
凌晨時分 4
凌氣逼人 0
凌波 88
凌波 88
凌空 57
凌空飛行 0
凌空高飛 0
凌虐 76
凌辱 26
凌遲 11
凌雲 134
凌雲壯志 1
凌霄 40
凌霄飛車 1
凌駕 75
凌駕在 6
凍 668
凍住 1
凍傷 38
凍僵 11
凍兒 0
凍兒 0
凍到 2
凍原 5
凍原地帶 0
凍土 4
凍害 0
凍得 18
凍死 39
凍港 1
凍瘡 2
凍硬 0
凍糕 0
凍結 455
凍結成 0
凍脹 0
凍裂 3
凍過 0
凍頂 27
凍頂烏龍 0
凍頂烏龍茶 20
凍餒 1
凎 28
减 0
凐 8
凐 8
凑 0
凒 0
凓 0
凔 7
凕 0
凖 0
凗 36
凘 9
凙 0
凚 0
凛 0
凜 78
凜冽 14
凜凜 12
凜然 15
凝 865
凝乳 5
凝合 1
凝合力 0
凝固 135
凝固點 7
凝寒 1
凝思 4
凝想 2
凝成 13
凝望 56
凝止 3
凝汽器 0
凝滯 7
凝為 1
凝眸 5
凝神 49
凝神靜聽 2
凝結 171
凝結劑 0
凝結力 0
凝結器 4
凝結水 4
凝結為 1
凝結點 0
凝聚 684
凝聚力 48
凝脂 3
凝膠 1
凝血 122
凝血劑 47
凝視 137
凝視著 104
凝重 69
凝集 104
凝集力 0
凝集反應 5
凞 6
凟 0
几 655
几 655
几几 0
几案 3
几案之才 0
几案之才 0
凡 5444
凡事 499
凡事起頭難 5
凡人 177
凡例 20
凡俗 15
凡因 27
凡在 181
凡士林 14
凡夫 112
凡夫俗子 36
凡心 17
凡應 5
凡指 4
凡於 89
凡是 684
凡有 167
凡此 40
凡此種種 45
凡民 14
凡為 23
凡為 23
凡無 7
凡爾 12
凡爾賽 106
凡百 6
凡立司 0
凡能 18
凡與 46
凡需 8
凡須 2
凢 0
凣 0
凤 0
凥 0
凥 0
凨 0
凨 0
凬 0
凬 0
凭 0
凮 0
凮 0
凰 897
凰鳳 1
凱 11870
凱塞琳 6
凱子 34
凱恩斯 61
凱悅 198
凱撒 200
凱旋 229
凱旋歸來 2
凱旋門 30
凱歌 29
凱莉 205
凱達格蘭 0
凲 0
凲 0
凳 1000
凳子 13
凴 0
凵 3
凵 3
凶 779
凶事 0
凶兆 12
凶具 0
凶器 8
凶多吉少 15
凶嫌 0
凶宅 2
凶年 0
凶徒 3
凶悍 17
凶惡 12
凶手 123
凶暴 5
凶橫 0
凶歲 0
凶殘 15
凶殺 0
凶殺案 4
凶氣 0
凶犯 4
凶狠 23
凶猛 29
凶相 0
凶相畢露 0
凶神惡煞 7
凶神附體 0
凶終隙末 0
凶耗 0
凶險 21
凷 0
凸 1
凸 1
凸 1
凸 363
凸凹 2
凸出 56
凸出來 0
凸出去 0
凸出成 1
凸嘴 0
凸多邊形 4
凸板印刷 0
凸版 41
凸版印刷 12
凸狀 2
凸狀物 0
凸緣 40
凸著 0
凸角 1
凸起 41
凸起來 0
凸輪 123
凸輪軸 47
凸透鏡 1
凸透鏡 12
凸邊 0
凸鏡 8
凸面 7
凸面鏡 2
凸顯 446
凸顯出 50
凹 251
凹下 6
凹下來 1
凹下去 1
凹凸 1205
凹凸不平 25
凹凸鏡 4
凹凹 6
凹凹凸凸 3
凹地 7
凹形 6
凹槽 32
凹槽鐵軌 0
凹洞 40
凹洞裡 0
凹溝 3
凹版 28
凹版印刷 13
凹痕 6
凹窪 0
凹處 14
凹角 0
凹透鏡 11
凹鏡 1
凹陷 96
凹陷下去 1
凹面 4
凹面鏡 7
出 1
出 28091
出上 12
出下 20
出不起 2
出世 108
出世法 1
出主意 11
出乎 73
出乎意外 16
出乎意料 52
出乎意料之外 11
出乖露醜 0
出亂子 3
出了 1056
出事 160
出亡 6
出人 32
出人意外 15
出人意料 34
出人意表 31
出人頭地 33
出仕 7
出任 652
出使 29
出來 4273
出來時 44
出個 61
出個價 8
出借 81
出借書 0
出價 240
出入 506
出入口 185
出入境 117
出入証 0
出入證 6
出公差 1
出兵 74
出其不備 0
出其不意 42
出典 14
出出 9
出出氣 0
出出進進 0
出刊 800
出列 30
出到 44
出力 171
出動 195
出勤 112
出勤率 0
出去 1753
出去時 3
出口 9490
出口傷人 1
出口入耳 0
出口區 22
出口商 188
出口成章 7
出口稅 4
出口貨 29
出名 197
出品 480
出售 1734
出售一空 0
出售一空 0
出唱片 23
出國 2053
出國前 82
出國時 30
出國者 19
出土 198
出土文物 17
出在 239
出埃及記 44
出場 31
出場 316
出塞 9
出塵 22
出境 760
出境簽證 1
出境許可證 3
出境証 1
出境證 25
出外 182
出外人 13
出外景 18
出奇 154
出奇制勝 16
出奔 1
出好 47
出嫁 55
出家 249
出家人 76
出將入相 2
出局 148
出山 20
出岔 2
出岔子 1
出巡 40
出工 3
出差 229
出差費 3
出師 31
出師不利 6
出師表 14
出席 2339
出席率 46
出庭 43
出庭作證 8
出庭應訊 11
出廠 334
出廠價 15
出廠價格 3
出征 77
出得 12
出得 12
出恭 2
出息 15
出戰 95
出手 342
出招 127
出掌 28
出擊 557
出操 9
出操課 0
出文 4
出於 323
出書 181
出有 115
出校 7
出此下策 0
出殯 22
出殯日 0
出毛病 16
出氣 34
出氣筒 7
出水 99
出水管 1
出水芙蓉 5
出汗 76
出江 26
出污泥而不染 5
出沒 127
出沒不定 1
出沒無常 0
出油 19
出洋 9
出洋相 3
出活 4
出海 172
出海口 1
出海打魚 0
出海捕魚 16
出淤泥而不染 4
出清 143
出港 163
出港證 0
出溜 1
出爐 1039
出爾反爾 9
出版 23407
出版前編目 0
出版商 249
出版地缺 0
出版所 3
出版日 59
出版日期 1224
出版期缺 0
出版業 475
出版業務 16
出版物 131
出版界 88
出版社 9109
出版社不詳 0
出版社叢集 0
出版社書目 1
出版社清樣 0
出版社目錄 0
出版社目錄卡 0
出版社裝禎 0
出版者 544
出版項 222
出牌 14
出獄 82
出獵 19
出現 13010
出球 1
出生 2389
出生入死 20
出生地 226
出生率 24
出生紙 1
出生證明 17
出產 166
出產地 5
出界 13
出發 1576
出發點 163
出的 1224
出眾 87
出示 150
出示證件 1
出社會 42
出神 49
出神入化 21
出科 9
出租 2211
出租人 98
出租店 50
出稿 1
出竅 13
出站 8
出籠 170
出糗 100
出納 372
出納員 5
出納科 3
出納系統 3
出納組 171
出納處 0
出給 10
出線 0
出缺 88
出聲 134
出脫 130
出自 387
出自於 54
出航 52
出色 334
出芽 6
出芽法 0
出苗 0
出苗率 0
出草 34
出落 4
出處 368
出血 820
出血性 117
出血病 5
出行 120
出言 32
出言不遜 14
出言成章 0
出訪 86
出診 20
出調 13
出謀劃策 0
出讓 295
出谷遷喬 0
出貨 1381
出貨單 25
出費 5
出資 270
出資人 23
出賣 381
出賽 119
出走 156
出起 2
出起來 0
出超 88
出超國 1
出超額 2
出路 525
出身 514
出車 52
出車禍 21
出軌 88
出軌行為 3
出軍 4
出逃 4
出遊 218
出過 79
出道 114
出道時 5
出遠門 21
出醜 4
出量 72
出鋒頭 10
出鋒頭 10
出錢 91
出錢出力 40
出錯 120
出鎚 100
出鐵 3
出門 553
出門在外 27
出門時 35
出閣 9
出關 33
出院 349
出陣 28
出險 41
出雙入對 0
出面 512
出頭 168
出頭天 56
出頭露面 0
出題 105
出類拔萃 26
出風頭 18
出風頭 18
出馬 134
出馬上場 0
出馬上場 0
出馬上陣 0
出點 11
函 5038
函令 135
函件 120
函告 29
函商 0
函寄 46
函式 0
函授 112
函授學校 214
函授教育 6
函授班 19
函數 1176
函梏關 0
函洞 1
函索 278
函蘊 0
函購 123
函購量 0
函送 215
函電 7
函養 0
凾 0
刀 3349
刀下 29
刀下留人 1
刀俎 0
刀俎魚肉 0
刀傷 18
刀光 30
刀光劍影 16
刀兵 8
刀具 273
刀具耐用度 0
刀刃 34
刀刻 3
刀刻 3
刀削麵 16
刀割 16
刀劍 152
刀匠 0
刀叉 19
刀口 34
刀子 84
刀尖 25
刀山 21
刀山劍樹 0
刀把 11
刀數 1
刀架 11
刀柄 45
刀械 107
刀棍 2
刀棍齊加 0
刀槍 23
刀槍不入 16
刀槍不入 16
刀法 69
刀片 79
刀疤 58
刀痕 8
刀筆 0
刀箭 3
刀耕火耨 0
刀背 15
刀鋒 73
刀鎗 2
刀面 4
刀鞘 11
刀馬旦 2
刁 1399
刁悍 0
刁斗 2
刁民 1
刁滑 0
刁猾 0
刁蠻 21
刁鑽 20
刁鑽古怪 2
刁難 42
刁頑 0
刂 0
刃 285
刃具 2
刃口 1
刃角 0
刃鋒 2
刄 0
刅 0
分 14141
分 24141
分不出 23
分不清 68
分不清楚 26
分不開 50
分久必合 3
分之 657
分了 76
分享 5347
分付 17
分佈 1883
分佈區 10
分佈圖 393
分佔 18
分作 30
分來 21
分來分去 1
分信 1
分個 9
分做 15
分光 157
分光度 0
分光計 9
分光鏡 19
分內 52
分內之事 1
分公司 4156
分冊 56
分出 117
分出來 6
分出去 2
分分 40
分分合合 0
分列 59
分列式 7
分列為 15
分別 6663
分別出 53
分別部居 0
分利 6
分到 65
分割 1440
分割為 12
分力 9
分化 325
分區 2017
分區賽 3
分去 2
分叉 29
分句 3
分吃 1
分合 26
分唱 0
分國 13
分在 92
分地 33
分場 62
分場 62
分壓力 0
分外 73
分奏 3
分好 18
分娩 188
分子 1642
分子 1642
分子力 1
分子尾 13
分子式 169
分子生物 279
分子生物學 259
分子篩 6
分子結構 45
分子論 6
分子運動 7
分子運動論 1
分子量 177
分子電流 0
分季 2
分守 3
分宜 0
分家 50
分寄 29
分寸 58
分寸已亂 0
分封 54
分對 12
分局 1927
分居 131
分屍 8
分屍案 2
分層 82
分層負責 66
分屬 124
分岐 15
分岔 7
分崩離兮 0
分崩離析 8
分工 764
分工合作 105
分布 1289
分布區 50
分帳 10
分床 2
分床而眠 0
分床而睡 0
分店 260
分度 54
分度器 1
分度規 0
分度頭 3
分庭 6
分庭抗禮 20
分廠 39
分式 31
分式方程 0
分得 76
分得出 3
分得到 1
分得清 2
分心 85
分憂 19
分憂解勞 0
分成 1026
分我杯羹 0
分截 0
分戶帳 0
分手 437
分批 146
分批交貨 11
分批進行 1
分持 5
分指 2
分排 10
分擔 318
分攤 213
分支 463
分支機關 2
分散 1689
分散主義 0
分散介質 0
分散劑 44
分散度 2
分散染料 141
分散相 1
分散開來 0
分數 2041
分文 45
分文不值 0
分文不取 0
分斤掰兩 0
分斤撥兩 0
分日線 0
分明 180
分明是 30
分時 125
分時走勢圖 1
分曉 23
分書 39
分會 1163
分有 56
分期 542
分期付款 251
分析 23807
分析儀 462
分析出來 7
分析器 30
分析家 122
分析師 100
分析語 10
分枝 76
分架 0
分校 871
分格 74
分機 1871
分檔 15
分欄 19
分權 65
分款 6
分歧 263
分段 290
分段收票 0
分段進行 3
分母 24
分毫 9
分毫不差 4
分水 20
分水嶺 54
分水線 1
分油 4
分治 147
分泌 720
分泌物 244
分泌管 1
分泌腺 4
分法 48
分派 126
分流 130
分清 23
分為 3493
分爨 0
分牌 4
分班 771
分班考試 2
分甘共苦 0
分生 150
分生孢子 20
分生組織 3
分用 11
分由 137
分界 79
分界線 17
分當 3
分登 8
分發 972
分的 945
分相 17
分省 7
分社 522
分神 12
分科 249
分秒 46
分秒不差 4
分秒必爭 17
分租 197
分種 3
分立 84
分站 266
分章 18
分等 24
分管 28
分箱 0
分節 29
分籃 0
分米 1
分紅 115
分級 2116
分級淬火 0
分組 1224
分組活動 15
分組討論 89
分給 74
分線 23
分線規 0
分縣 10
分置 33
分署 4
分群 17
分而治之 2
分至 310
分色 50
分色鏡 2
分菜 1
分蔥 0
分蘗 4
分蘗期 0
分蘗節 0
分號 35
分行 2091
分被 26
分裂 542
分裂主義 4
分裂性 2
分裂法 0
分裂為 10
分裝 91
分角器 0
分解 1070
分解成 60
分解掉 3
分解為 29
分設 151
分詞 9
分詞構句 0
分說 13
分課 6
分貝 116
分賬 0
分贓 25
分贓不均 1
分走 2
分赴 36
分起 38
分起來 4
分路 11
分路揚鑣 0
分身 182
分身乏術 7
分車 1
分載 2
分辨 523
分辨善惡 1
分辨是非 6
分辨真假 7
分辯 8
分送 310
分途 4
分進 3
分進合擊 8
分運 5
分過 5
分過來 0
分過去 0
分道 63
分道揚鑣 20
分選 26
分邊 2
分部 333
分配 3017
分配器 29
分配律 2
分配率 1
分配給 67
分配閥 0
分野 62
分量 93
分針 18
分釵斷帶 0
分銷 64
分銷店 0
分錄 35
分錢 22
分鐘 3450
分長 3
分門 8
分門別類 86
分開 664
分院 549
分隊 144
分隔 1400
分隔島 20
分集 7
分離 1612
分離出來 29
分離法 12
分頁 61
分領 1
分頭 53
分頭來 0
分頭去 6
分頭辦理 1
分類 22235
分類學 48
分類帳 28
分類法 315
分類賬 0
分飛 13
分食 37
分飾 8
分飾兩角 5
分飾兩角 5
分館 1318
分餾 9
分餾器 0
分餾塔 2
分餾法 1
分香賣履 0
分駕 0
分點 48
分黨 3
分齡 47
切 2825
切 2825
切上 11
切上來 1
切上去 0
切下 33
切下來 5
切下去 1
切中 17
切中 17
切中要害 8
切中要害 8
切事 34
切個 3
切入 369
切出 10
切出來 1
切出去 0
切分音 1
切切 37
切切私語 0
切到 26
切削 295
切削 295
切削力 11
切削力 11
切削性能 0
切削性能 0
切削機 12
切削機 12
切削用量 0
切削用量 0
切削角 0
切削角 0
切削速度 2
切削速度 2
切割 548
切割成 28
切割機 97
切勿 0
切厚 2
切去 8
切口 52
切合 90
切合實際 18
切合需要 4
切圓 5
切塊 42
切宜 2
切實 273
切實可行 10
切實有效 1
切屑 4
切忌 106
切成 146
切掉 26
切換 833
切斷 258
切時 272
切望 20
切片 267
切片機 18
切片檢查 50
切當 4
切直 2
切碎 47
切磋 158
切磋琢磨 7
切細 10
切結 273
切結書 1
切線 43
切肉 5
切肉刀 1
切腹 20
切腹自殺 9
切膚 22
切膚之痛 1
切莫 59
切菜 16
切菜刀 0
切要 18
切記 241
切起 0
切起來 1
切身 129
切身利益 5
切身經驗 4
切近 5
切進 6
切進來 0
切進去 1
切過 9
切過來 0
切過去 0
切錯 1
切開 132
切除 575
切面 28
切音 6
切題 1
切麵 2
切點 5
切齊 30
切齒 12
刈 301
刈包 4
刈稻 0
刈草 1
刈草機 0
刉 124
刊 6552
刊了 2
刊些 1
刊佈 1
刊出 305
刊出來 3
刊到 2
刊印 34
刊在 32
刊布 3
刊本 23
刊正 1
刊物 1486
刊登 15079
刊著 2
刊行 141
刊誤 2
刊起 2
刊載 579
刊過 1
刊錯 1
刊頭 351
刋 0
刌 2
刎 5
刎頸 3
刏 0
刐 55
刑 740
刑事 773
刑事局 1
刑事案件 98
刑事法 10
刑事法庭 1
刑事犯 6
刑事犯罪 6
刑事處分 20
刑事訴訟 85
刑事訴訟法 213
刑事警察 53
刑事責任 75
刑具 8
刑名 0
刑場 16
刑場 16
刑庭 2
刑律 1
刑房 3
刑書 1
刑期 23
刑期無刑 1
刑案 142
刑求 0
刑法 600
刑法志 0
刑罰 187
刑訊 5
刑警 133
刑警隊 28
刑責 116
刑辱 0
刑部 6
刑部侍郎 1
划 113
划上 0
划下 1
划不來 39
划了 12
划來划去 1
划具 0
划出 4
划到 2
划動 3
划去 0
划子 0
划得來 11
划得來 11
划拳 3
划槳 8
划水 19
划算 210
划船 118
划艇 1
划著 8
划著走 0
划行 2
划走 1
划起 0
划起來 0
划進 2
划進來 0
划進去 0
划過 5
划過來 0
划過去 0
划龍舟 1
刓 352
刓 352
刔 0
刕 0
刖 15
列 12496
列上 206
列上來 0
列上去 2
列了 35
列位 8
列傳 173
列入 1408
列出 1367
列出來 31
列到 7
列印 3908
列印機 34
列國 102
列夫 6
列子 31
列寧 35
列寧主義 1
列寧全集 0
列寧格勒 26
列寧選集 0
列島 30
列嶼 0
列席 298
列席代表 2
列強 74
列成 13
列支敦斯登 8
列星 1
列氏寒暑表 0
列為 1192
列祖列宗 0
列管 559
列管流氓 0
列舉 635
列舉如下 25
列舉如右 0
列表 2215
列表機 148
列車 1031
列車員 0
列車長 11
列隊 26
列隊歡迎 1
刘 0
刜 4
初 5194
初一 88
初三 60
初上 32
初中 207
初中生 4
初乳 12
初二 48
初五 32
初交 5
初八 29
初冬 17
初出茅廬 10
初刊 2
初創 53
初十 21
初吻 19
初唐 8
初四 36
初夏 63
初夜 15
初始 216
初學 202
初學者 767
初審 483
初度 6
初征 1
初戀 123
初戀情人 28
初掌 3
初旬 12
初春 39
初春時節 0
初時 70
初更 10
初更 10
初期 1644
初校樣 0
初次 712
初次見面 50
初步 1631
初步編目 0
初步階段 6
初段 16
初演 14
初版 7943
初版書 0
初犯 7
初獻 0
初生 106
初生之犢 17
初生之犢不畏虎 2
初秋 39
初稿 160
初等 191
初等教育 279
初級 772
初級中學 43
初級班 187
初級社 0
初級線圈 0
初航 0
初衷 53
初診 442
初試 149
初試啼聲 29
初賽 139
初轉法輪 1
初速 5
初速度 1
初選 553
初開 39
初階 133
初露鋒芒 1
初露鋒芒 1
初願 1
刞 6
刟 0
删 0
刡 7
刢 0
刣 0
判 1951
判下 0
判下來 0
判下去 0
判例 197
判刑 176
判別 173
判別式 1
判官 57
判定 575
判成 7
判斷 2125
判斷力 72
判斷句 10
判斷是非 8
判明 7
判案 2
判死罪 1
判決 1150
判決書 55
判罪 15
判若 2
判若兩人 9
判若雲泥 0
判處 113
判讀 100
判起 0
判起來 0
別 23003
別上 18
別上來 0
別上去 4
別下 3
別下來 0
別下去 0
別了 30
別人 4705
別人的 2150
別使 10
別來 86
別來無恙 16
別個 10
別倒 3
別傳 16
別傻 12
別傻了 46
別像 25
別具一格 11
別具一格 11
別具匠心 0
別具慧眼 1
別再 420
別出 11
別出來 6
別出去 2
別出心裁 39
別到 7
別刻 48
別加 13
別動 13
別動隊 3
別區 22
別去 37
別又 12
別受 8
別句 0
別叫 17
別名 461
別向 41
別哭 50
別嚇 19
別嚇人 2
別國 43
別在 138
別墅 798
別套 1
別字 7
別守 0
別室 2
別家 89
別寄 42
別將 29
別對 62
別局 0
別巷 0
別師 1
別彈 0
別後 22
別得 6
別從 55
別忙 8
別急 117
別怪 28
別情依依 0
別想 133
別成 6
別扭 5
別把 128
別拍 2
別拔 0
別拿 14
別排 5
別描頭 0
別提 67
別搬 0
別擠 0
別支 0
別收 3
別教 5
別數 4
別數 4
別族 3
別是 53
別替 8
別有 208
別有天地 3
別有所指 0
別有用心 19
別有用意 3
別本 6
別束 0
別枝 3
別架 3
別校 24
別棟 2
別棵 48
別業 16
別樹一幟 4
別樹一格 2
別樹一格 2
別殺 5
別氣 8
別派 12
別為 144
別為 144
別無 109
別無他法 11
別無分號 3
別班 58
別現 1
別理 35
別生 41
別生氣 66
別用 40
別由 167
別當 30
別的 1494
別的人 39
別省 0
別看 60
別科 49
別種 36
別種的 0
別稱 55
別笑 31
別答 10
別管 46
別箱 0
別組 32
別給 29
別緒 0
別線 4
別緻 40
別縣 1
別罐 0
別致 11
別著 12
別處 109
別號 110
別被 115
別討 4
別記 4
別試 14
別說 245
別說出 0
別說到 0
別說起 0
別課 37
別談 13
別請 7
別講 4
別讓 338
別賞 5
別走 61
別起 1
別起來 0
別跑 13
別踢 0
別輸 4
別轉 11
別辦 3
別逃 3
別過 9
別過來 20
別過去 1
別鄉 0
別針 35
別開 39
別開生面 38
別關 5
別隊 4
別隻 5
別集 28
別離 113
別項 6
別顆 48
別題 1
別館 150
別首 0
別騎 0
別體 1
別黨 0
刦 0
刧 0
刨 79
刨 79
刨冰 35
刨出來 1
刨刀 5
刨子 0
刨床 10
刨根 0
刨根問底 0
刨花 1
刨除 4
利 16686
利上滾利 0
利人 83
利人利己 9
利他 62
利他主義 10
利令智昏 3
利便 4
利刃 31
利劍 19
利口 4
利口捷給 0
利口酒 1
利器 765
利國 44
利國利民 0
利基 0
利多 726
利多弊少 3
利多於弊 10
利子 4
利害 291
利害衝突 4
利害關係 99
利尿劑 59
利己 33
利己主義 9
利己利人 4
利市 20
利弊 50
利弊得失 52
利得 90
利息 884
利息支出 36
利息收入 77
利息費用 15
利慾 5
利慾薰心 6
利於 388
利權 26
利比亞 49
利比裡亞 0
利比里亞 0
利民 51
利源 11
利潤 871
利潤率 25
利潤留成 0
利物浦 46
利物浦港 1
利率 3730
利瑪竇 9
利用 19938
利用係數 0
利用價值 47
利用厚生 1
利用率 121
利益 2651
利益輸送 65
利祿 13
利空 358
利索 1
利落 5
利誘 46
利鈍 3
利錢 1
利雅得 0
利馬 51
利馬竇 3
刪 16716
刪剪 1
刪去 38
刪成 3
刪掉 149
刪改 110
刪汰 0
刪減 100
刪減成 1
刪節 3
刪節本 0
刪節號 0
刪補 1
刪詞 0
刪除 4502
别 0
刮 37
刮 370
刮下 4
刮下來 1
刮下去 0
刮了 6
刮來刮去 0
刮傷 88
刮刀 28
刮刮叫 0
刮刮樂 0
刮刮雜雜 0
刮到 6
刮去 3
刮取 2
刮在 0
刮地皮 0
刮垢 0
刮好 0
刮得 3
刮打 0
刮掉 11
刮搭板兒 0
刮搭板兒 0
刮痕 26
刮痧 23
刮的 9
刮目相待 0
刮目相看 72
刮研 0
刮破 2
刮腸洗胃 0
刮臉 1
刮臉刀 0
刮舌 9
刮著 10
刮走 6
刮起 26
刮起來 0
刮過 1
刮過來 0
刮過去 0
刮風 12
刮骨 5
刮鬍刀 32
刮鬍子 44
刯 0
到 65361
到了 3638
到任 79
到來 518
到到 15
到場 37
到場 377
到場人員 0
到場人員 0
到外 60
到家 247
到底 3023
到底在 129
到底是 829
到底會 60
到底有 389
到得 37
到得 37
到手 84
到時 439
到時候 348
到會 97
到期 624
到校 258
到案 214
到此 1143
到此一遊 1
到此為止 43
到的 1829
到站 48
到職 184
到職人員 0
到臨 30
到處 1125
到處是 35
到處都 82
到訪 0
到過 183
到達 1416
到頂 15
到頭 48
到頭來 95
到點 3
到齊 85
刱 258
刲 117
刳 514
刴 0
刵 219
制 5668
制件 0
制伏 17
制住 9
制作 137
制備 2
制冷 12
制冷機 0
制劑 4
制動 63
制動器 29
制動缸 0
制勝 24
制勝之道 0
制品 42
制圖 47
制定 1769
制定出來 4
制幣 0
制度 8026
制度化 247
制式 157
制憲 98
制憲會議 17
制成品 1
制敵機先 6
制服 303
制止 187
制海權 5
制版 5
制禮作樂 1
制空權 13
制糖 1
制約 129
制衡 149
制衡作用 2
制表 17
制裁 263
制訂 461
制造 42
制酸劑 32
制酸性 0
制醣 0
制錢 1
制面 2
制革 0
制音器 0
制高點 7
刷 22
刷 2243
刷上 15
刷上來 0
刷上去 0
刷下 3
刷下來 0
刷下去 1
刷來 1
刷來刷去 0
刷刷 31
刷刷牙 1
刷刷聲 0
刷卡 0
刷子 93
刷掉 8
刷新 58
刷新紀錄 12
刷洗 33
刷牆 1
刷牙 146
刷牙漱口 1
刷起 0
刷起來 0
券 3836
券商 0
刹 0
刹 0
刺 1075
刺上 10
刺上來 1
刺上去 0
刺下 3
刺下來 0
刺下去 2
刺人 8
刺傷 55
刺出 17
刺出來 0
刺出去 0
刺刀 15
刺到 17
刺刺 19
刺史 18
刺字 4
刺客 96
刺客列傳 3
刺戳 1
刺探 29
刺探出 0
刺有 1
刺桐 30
刺槍 0
刺槐 11
刺殺 63
刺殺出局 2
刺毛輥 0
刺激 2416
刺激作用 35
刺激劑 13
刺激性 161
刺激法 6
刺痛 97
刺痛性 0
刺癢 0
刺目 7
刺眼 53
刺破 18
刺穿 27
刺網 20
刺繡 347
刺耳 61
刺股 1
刺著 9
刺蝟 28
刺身 1
刺配 2
刺針 50
刺開 0
刺青 48
刺骨 27
刺鼠 7
刺鼻 19
刻 18088
刻 1434
刻上 22
刻上 22
刻上來 0
刻上來 0
刻上去 0
刻上去 0
刻下 15
刻下 15
刻下來 1
刻下來 1
刻下去 0
刻下去 0
刻不容緩 147
刻了 9
刻作 3
刻作 3
刻個 2
刻個 2
刻出 9
刻出 9
刻刀 3
刻到 4
刻劃 113
刻劃入微 2
刻劃出 19
刻印 134
刻即 3
刻圖章 0
刻在 57
刻在 57
刻字 76
刻字 76
刻度 208
刻度尺 1
刻度盤 3
刻得 3
刻得 3
刻意 646
刻意求精 0
刻成 8
刻成 8
刻於 11
刻於 11
刻木為吏 1
刻本 17
刻板 197
刻毒 1
刻燭賦詩 1
刻版 20
刻版印刷 0
刻畫 91
刻畫來 0
刻畫無鹽 1
刻痕 24
刻肌刻骨 1
刻舟求劍 0
刻苦 63
刻苦耐勞 37
刻著 26
刻薄 23
刻薄話 0
刻蠟紙 0
刻起 35
刻起 35
刻起來 0
刻起來 0
刻足 0
刻過 0
刻過 0
刻過來 0
刻過來 0
刻過去 0
刻過去 0
刻鐘 12
刻骨 10
刻骨銘心 87
刻骨鏤心 0
刻鵠類鶩 1
刼 0
刽 0
刽 0
剁 49
剁成 49
剁碎 12
剃 87
剃了 2
剃刀 29
剃度 31
剃得 2
剃毛 34
剃頭 34
剃頭髮 0
剃髮 15
剃髮為尼 0
剃鬚 0
剄 19
剅 0
剅 0
剆 194
則 33335
則否 20
則在 886
則是 4867
則會 565
則有 935
則為 1019
則為 1019
則無 259
則要 240
剈 0
剉 16
剉冰 16
削 406
削 406
削了 7
削了 7
削來削去 0
削來削去 0
削價 90
削價 90
削削 0
削削 0
削去 17
削去 17
削尖 3
削尖 3
削平 8
削平 8
削弱 129
削弱 129
削成 6
削成 6
削斷 4
削斷 4
削減 306
削減 306
削減到 0
削減到 0
削減成 0
削減成 0
削皮 24
削皮 24
削皮刀 38
削皮刀 38
削職 1
削職 1
削著 0
削著 0
削薄 60
削薄 6
削蘋果 10
削蘋果 10
削足適履 6
削足適履 6
削過 1
削過 1
削鉛筆 6
削鉛筆 6
削鉛筆機 6
削鉛筆機 6
削除 10
削除 10
削髮 3
削髮 3
削髮為僧 0
削髮為僧 0
削髮為尼 1
削髮為尼 1
剋 132
剋制 18
剋星 149
剋服 2
剋死 2
剋薄 0
剌 180
剌 180
剌 180
前 5207
前陣子 100
前陣子 100
前代 100
前世 223
前事不忘 0
前人 216
前人種樹 12
前仆後繼 32
前仇 2
前仰後合 2
前件 0
前任 283
前作 44
前來 1001
前例 40
前俯後仰 1
前倨後恭 3
前傾 34
前兆 40
前先 217
前列 95
前列腺 130
前前後後 36
前功盡棄 18
前半 122
前半夜 0
前半段 38
前半部 30
前去 234
前句 3
前台 94
前呼後擁 6
前哨 41
前哨戰 33
前哨站 9
前因 85
前因後果 27
前園 1
前場 18
前場 18
前塵 26
前塵往事 13
前夕 459
前夜 13
前天 350
前夫 59
前奏 39
前奏曲 34
前套 5
前妻 49
前嫌 8
前導 65
前山 67
前巷 0
前年 212
前庭 87
前廠 0
前廳 49
前往 4827
前後 1475
前後任 5
前後夾攻 4
前後左右 27
前後相悖 0
前後相隨 0
前後矛盾 6
前思後想 0
前傳 1000
前情 8
前情摘要 0
前愆 1
前所 240
前所未有 272
前所未見 43
前挽後推 0
前排 42
前排觀眾 0
前提 700
前撲後繼 3
前敵 0
前文 65
前方 449
前日 190
前晚 67
前景 1277
前有 230
前期 478
前期損益 4
前桅 0
前棟 14
前樓 4
前橋 2
前次 232
前款 144
前步 1
前段 213
前沿 5
前清 20
前滾 5
前滾翻 12
前漢 5
前為 193
前無古人 9
前生 50
前瞻 452
前瞻性 330
前短後長 0
前科 67
前秦 4
前程 126
前程似錦 10
前程萬里 3
前窗 0
前站 9
前端 302
前線 262
前緣 62
前置詞 3
前者 932
前肢 28
前腦 7
前腳 27
前腿 6
前臂 42
前臂 42
前臂骨 0
前臂骨 0
前臺 9
前艙 2
前茅 28
前菜 1
前葉 19
前藏 36
前藏 36
前行 143
前衛 348
前言 2552
前言不對後語 0
前蹄 11
前身 335
前車 22
前車之覆 0
前車之鑑 26
前輩 1209
前輪 114
前述 1118
前途 736
前途無亮 3
前途無量 11
前進 1039
前邊 12
前部 28
前鋒 253
前鎮 541
前門 39
前院 64
前面 2727
前項 3013
前頭 147
前額 44
前驅 43
剎 171
剎 171
剎住 2
剎時 17
剎車 154
剎那 170
剎那之間 13
剎那間 133
剏 0
剏 0
剒 180
剓 0
剔 67
剔 67
剔了 1
剔出 7
剔去 0
剔得 0
剔得 0
剔牙 2
剔著 0
剔透 13
剔開 0
剔除 96
剕 94
剖 258
剖 258
剖 258
剖成 7
剖析 623
剖白 8
剖白 8
剖示圖 0
剖肝泣血 0
剖腹 97
剖腹生產 56
剖腹產 1
剖腹自殺 0
剖視圖 1
剖解 2
剖開 18
剖面 112
剖面圖 91
剗 0
剘 0
剙 0
剙 0
剚 17
剛 5219
剛乾 1
剛估 0
剛使 6
剛來 48
剛來的 10
剛健 8
剛僱 1
剛出 131
剛出來 7
剛出去 0
剛出道 17
剛到 85
剛剛 1355
剛剛在 46
剛剛是 60
剛加 25
剛勁 4
剛勇 1
剛印 2
剛去 36
剛取 7
剛回 26
剛回來 18
剛回到 5
剛回去 0
剛夾 1
剛好 822
剛好在 58
剛好是 80
剛娶 0
剛嫁 0
剛完 50
剛定 1
剛巧 33
剛度 14
剛強 70
剛從 159
剛性 164
剛愎 5
剛愎自用 8
剛成 18
剛成立 35
剛才 943
剛才在 29
剛才是 12
剛扣 1
剛抬 1
剛搬 21
剛搶 1
剛撕 0
剛會 3
剛板硬正 0
剛果 74
剛果河 0
剛架 21
剛柔 38
剛柔並濟 21
剛正 11
剛正不阿 4
剛死 3
剛毅 23
剛毅不屈 2
剛毅木納 1
剛毅木訥 3
剛毅木訥 3
剛毛 8
剛溼 0
剛灌 5
剛為 6
剛烈 11
剛燒 1
剛玉 3
剛玩 9
剛盛 0
剛直 6
剛硬 20
剛纔 7
剛背 0
剛蒸 0
剛被 37
剛補 3
剛要 38
剛訂 0
剛試 3
剛讀 7
剛貼 3
剛賜 0
剛贏 5
剛趕 1
剛跑 4
剛辦 4
剛過 53
剛過來 0
剛過去 1
剛醒 3
剛鋪 1
剛鋸 0
剛鍍 0
剛關 0
剛雇 0
剛頒 0
剛領 9
剛體 12
剜 17
剜肉補瘡 0
剜肉醫瘡 0
剝 251
剝下 31
剝削 130
剝削階級 0
剝去 9
剝奪 255
剝奪走 0
剝掉 9
剝皮 98
剝落 88
剝蝕 5
剝開 21
剝除 22
剝離 136
剞 116
剞 116
剟 7
剠 0
剡 17
剢 433
剣 0
剤 0
剥 0
剥 0
剧 0
剧 0
剨 0
剩 6400
剩下 707
剩下來 5
剩了 2
剩個 0
剩得 1
剩的 58
剩磁 4
剩肉 0
剩菜 19
剩過 0
剩酒 0
剩錢 0
剩飯 8
剩飯剩菜 3
剩餘 654
剩餘價值 24
剩餘價值率 0
剩餘勞力 0
剩餘勞動 0
剩餘物 1
剩餘產品 0
剪 911
剪下 124
剪下來 80
剪下去 10
剪了 25
剪作 4
剪個 4
剪出 10
剪刀 197
剪刀差 0
剪切 49
剪到 4
剪剪 7
剪力 79
剪去 10
剪反 0
剪取 3
剪報 223
剪子 4
剪字 0
剪對 0
剪彩 12
剪彩儀式 0
剪影 977
剪得 9
剪應力 15
剪應力 15
剪成 15
剪截 1
剪掉 43
剪接 209
剪接室 3
剪接法 0
剪斷 66
剪毛 5
剪毛機 3
剪法 5
剪片 11
剪短 31
剪票 4
剪票員 1
剪紙 134
剪綵 90
剪線 5
剪草 3
剪草除根 0
剪著 1
剪裁 86
剪裁法 3
剪貼 94
剪貼出 0
剪貼簿 167
剪起 0
剪起來 0
剪輯 418
剪輯法 0
剪過 4
剪過來 0
剪過去 0
剪開 18
剪除 24
剪頭髮 10
剪髮 64
剪點 0
剪齊 0
剫 5
剬 4
剭 8
剮 15
副 5601
副主任 4
副主任委員 175
副主席 362
副主教 2
副井 0
副作用 778
副刊 580
副原料 2
副反應 2
副司令 21
副團長 44
副官 18
副官處長 0
副審 15
副將 20
副座 5
副廠長 13
副所長 132
副手 21
副教授 4633
副會長 304
副本 395
副校長 223
副案 0
副業 35
副機 72
副檔名 0
副熱帶 54
副班長 31
副理 195
副產品 59
副產物 53
副甲狀線 0
副相 0
副社長 94
副科 18
副程式 131
副線 11
副線圈 1
副總 451
副總司令 10
副總理 145
副總統 1106
副總經理 613
副總裁 224
副署 92
副翼 28
副職 5
副腎 39
副董 32
副董事長 163
副處長 161
副詞 14
副詞片語 2
副議長 602
副軸 2
副院長 847
副題 14
副食 44
副食品 57
剰 0
割 554
割上 0
割下 18
割下來 4
割下去 1
割了 10
割作 0
割來 1
割來割去 0
割傷 39
割入 1
割出 8
割到 5
割剩 0
割包 7
割包皮 20
割去 4
割取 9
割回 0
割回來 0
割回去 0
割在 0
割地 7
割地賠款 2
割好 0
割完 7
割席絕交 0
割得 2
割愛 253
割成 10
割捨 62
割捨不下 8
割掉 16
割據 15
割斷 13
割法 1
割破 4
割禮 29
割稻 7
割稻機 1
割給 1
割絨 0
割線 11
割腕 9
割腕自殺 8
割草 11
割草機 28
割著 1
割裂 21
割讓 44
割讓土地 1
割走 0
割起 1
割起來 0
割進 1
割進來 0
割進去 10
割過 50
割過來 0
割過去 0
割開 13
割除 24
割離 3
剳 0
剳 0
剴 60
剴切 6
創 2195
創 5661
創下 587
創世紀 300
創作 4591
創作人 62
創作力 15
創作家 4
創作實踐 0
創作技巧 4
創作方法 2
創作者 257
創傷 232
創出 78
創刊 421
創刊號 285
創刊詞 14
創制 69
創制權 7
創口 3
創始 160
創始人 107
創始者 27
創建 278
創建人 5
創建者 4
創意 2486
創投 0
創新 2689
創新作品 2
創校 360
創業 1307
創業基金 13
創業楷模 21
創業者 27
創痍滿目 0
創痕 4
創痛 20
創立 1478
創立出 0
創紀錄 45
創舉 112
創製 14
創見 86
創設 284
創說 0
創辦 351
創辦人 417
創辦者 8
創造 4950
創造出 422
創造力 344
創造性 169
創造成 3
創鉅痛深 0
剶 0
剷 86
剷出 0
剷平 11
剷除 83
剸 79
剸 79
剹 0
剺 47
剻 76
剼 72
剽 53
剽劫 0
剽悍 13
剽掠 0
剽竊 33
剾 0
剿 175
剿 175
剿共 2
剿匪 5
剿匪抗戰 0
剿匪時期 0
剿平 2
剿捕 1
剿撫兼施 0
剿滅 3
剿獲 0
剿襲 0
剿除 1
劀 3
劁 4
劂 249
劃 47
劃 4750
劃一 94
劃一不二 0
劃上 116
劃下 115
劃了 18
劃來劃去 0
劃出 87
劃分 658
劃分為 84
劃到 1
劃劃 5
劃去 5
劃在 12
劃地 66
劃地自限 13
劃定 253
劃得 5
劃得 5
劃成 18
劃掉 3
劃撥 1004
劃撥帳號 519
劃時代 80
劃款 4
劃歸 56
劃法 149
劃清 8
劃清界線 23
劃清界限 11
劃界 6
劃破 58
劃線 56
劃足 2
劃開 16
劃開 16
劄 5
劄記 17
劅 0
劆 0
劇 13919
劇中 321
劇中人 16
劇作 52
劇作家 65
劇力萬鈞 3
劇動 1
劇務 1
劇務員 0
劇名 41
劇團 1076
劇場 195
劇場 1950
劇增 49
劇展 25
劇情 3004
劇挫 1
劇曲 3
劇本 709
劇校 29
劇毒 51
劇烈 362
劇烈性 1
劇照 313
劇痛 56
劇目 86
劇種 75
劇終 16
劇評 8
劇評人 1
劇變 42
劇跌 3
劇院 418
劈 187
劈倒 0
劈刀 1
劈到 1
劈刺 0
劈刺訓練 0
劈削 1
劈劈 2
劈劈啪啪 1
劈劈看 1
劈哩啪啦 4
劈啪 0
劈成 5
劈手 1
劈掌 1
劈材 0
劈柴 4
劈腿 100
劈裂 5
劈開 19
劈面 3
劈頭 31
劉 15360
劉備 246
劉勰 12
劉向 17
劉坤一 1
劉姥姥 28
劉姥姥進大觀園 1
劉安 42
劉宋 8
劉宗敏 0
劉少奇 1
劉師培 1
劉康民 14
劉德華 325
劉徽 8
劉文正 14
劉文聰 14
劉海 18
劉淵 7
劉燈 8
劉玄 17
劉琨 1
劉福通 0
劉秀 137
劉聰 14
劉胡蘭 0
劉表 23
劉裕 8
劉邦 218
劉銘傳 54
劉長瑜 1
劉雪華 14
劉體純 0
劊 10
劊子手 43
劋 3
劌 26
劍 6816
劍下 23
劍俠 16
劍光 39
劍及屨及 2
劍名 8
劍客 266
劍尾 10
劍拔弩張 17
劍柄 27
劍橋 200
劍橋大學 92
劍氣 46
劍法 152
劍潭 155
劍王 4
劍眉 7
劍背 2
劍葉 3
劍蘭 12
劍蘭花 0
劍術 90
劍訣 17
劍道 101
劍鞘 21
劍頭 3
劍魚 6
劍麻 0
劍龍 34
劎 0
劐 0
劐 0
劑 6836
劑量 1459
劒 0
劓 162
劔 0
劕 0
劖 36
劗 172
劗 172
劗 172
劘 81
劙 33
劚 0
力 16686
力不勝任 0
力不從心 62
力主 33
力促 78
力偶 5
力勸 20
力單勢弱 0
力回饋 1
力圖 66
力圖振作 16
力場 70
力場 70
力士 155
力士捉蠅 0
力多邊形 0
力大如牛 0
力大無比 3
力大無窮 11
力學 1008
力學不倦 0
力守 32
力度 47
力役 0
力戰 17
力所不及 1
力所未逮 0
力所能及 5
力拔山兮 1
力拼 17
力拼眾敵 0
力挽狂瀾 34
力排 111
力排眾議 5
力推 75
力擒 0
力有未逮 0
力氣 766
力求 414
力求上進 3
力爭 109
力爭上游 38
力疾 1
力疾從公 0
力盡 12
力盡筋疲 1
力矩 42
力系 11
力能 22
力臂 9
力臂 9
力薄才疏 0
力薦 7
力行 588
力距 1
力退 1
力退眾敵 0
力透紙背 0
力道 225
力道十足 0
力量 3348
力量對比 3
力錢 0
力點 12
劜 0
劝 0
办 0
功 3814
功不可沒 70
功令 0
功利 500
功利主義 300
功到自然成 0
功力 595
功勛 8
功勞 115
功勞簿 0
功勳 13
功勳彪炳 1
功名 57
功大於過 0
功夫 843
功學社 102
功底 6
功德 396
功德會 1
功德無量 46
功成不居 0
功成名就 16
功成名遂 1
功成身退 45
功效 757
功敗垂成 15
功服 0
功業 25
功率 895
功率因數 31
功率放大 5
功率放大器 23
功用 442
功績 151
功耗 100
功能 21257
功能表 475
功能鍵 173
功臣 114
功虧一簣 27
功虧一簣 27
功行圓滿 0
功課 649
功課表 281
功過 58
功過相抵 5
功首罪魁 0
加 37052
加上 7221
加上來 2
加上去 26
加下 7
加下來 0
加下去 1
加之 79
加了 206
加些 38
加人 14
加人一等 0
加付 21
加以 3547
加以解決 23
加來加去 2
加侖 77
加個 60
加倍 227
加倍努力 10
加倫河 1
加值 726
加價 120
加入 14325
加入到 22
加冕 11
加冕典禮 2
加冕禮 3
加出 3
加分 172
加分法 0
加利略 5
加利福尼亞 8
加利福尼亞州 11
加到 153
加剪 0
加劇 137
加力 32
加力燃燒室 0
加加 23
加加減減 9
加勒比 27
加勒比海 334
加勒比海群島 3
加升 2
加印 43
加厚 10
加味 35
加哩粉 0
加回 4
加回來 0
加回去 0
加固 16
加固作用 0
加國 205
加在 194
加報 3
加場 2
加場 2
加場賽 0
加場賽 0
加塗 1
加填 14
加增 16
加壓 211
加壓法 5
加壓艙 0
加多 25
加多少 1
加夠 0
加大 402
加太基 0
加太蘭 0
加套 10
加好 70
加守 0
加完 50
加官 9
加官晉爵 3
加官進祿 0
加定 13
加害 140
加害於 2
加寄 1
加密 315
加富爾 1
加寫 3
加寬 28
加封 27
加對 9
加州 1866
加州人 2
加工 4756
加工出口 33
加工出口區 698
加工區 181
加工品 110
加工工業 6
加工廠 702
加工成本 9
加工業 180
加工硬化 2
加工訂貨 0
加工餘量 0
加布 16
加帶 3
加座 2
加強 9699
加強法 0
加彭 8
加得 19
加德滿都 21
加快 561
加快步伐 5
加快腳步 37
加急 3
加成 97
加成收費 0
加成計費 0
加打 4
加把 7
加把勁 45
加拈 0
加拉加斯 0
加拿大 3245
加拿大人 16
加持 100
加按 0
加捲 0
加捻 0
加排 10
加掛 27
加掛車廂 0
加撥 11
加播 1
加播 1
加收 192
加敷 0
加數 12
加料 29
加料處理 1
加料處理 1
加於 47
加有 10
加杯 0
加杯水 0
加架 0
加框 8
加檔 3
加權 372
加權指數 227
加氫 18
加氫精制 0
加氫裂化 0
加氯 28
加氯處理 2
加氯處理 2
加水 185
加油 3256
加油戰 0
加油添醋 20
加油站 1155
加油車 0
加法 120
加注 29
加洗 20
加洗照片 3
加派 30
加深 350
加深處理 0
加深處理 0
加添 48
加減 131
加減乘除 14
加減法 17
加溫 261
加溫法 0
加滿 34
加演 75
加漲 0
加漲 0
加潑 0
加為 7
加煎 0
加熱 664
加熱器 44
加熱法 1
加熱爐 25
加爾各答 33
加爾各達 0
加班 211
加班津貼 1
加班費 67
加班車 26
加生 2
加產 1
加當 3
加登 4
加發 90
加的 104
加盟 1370
加盟共和國 6
加盟店 298
加硝 1
加硝處理 0
加硝處理 0
加碘 7
加碘處理 0
加碘處理 0
加碼 372
加稅 37
加穿 4
加等 15
加算 10
加粗 22
加納 36
加組 0
加給 169
加緊 184
加緊努力 7
加緊練習 5
加緊腳步 33
加緊訓練 4
加線 2
加縫 1
加總 0
加罰 4
加考 65
加考科目 2
加聚反應 0
加色 5
加色劑 0
加色法 1
加色處理 0
加色處理 0
加菜 6
加菜金 9
加菲 4
加菲貓 4
加蓋 385
加蓋樓頂 0
加蓬 0
加薩走廊 5
加薪 70
加藥 38
加號 17
加補 4
加裝 463
加裝在 2
加裡寧 0
加西亞 3
加訂 30
加計 140
加訓 3
加設 51
加註 209
加詞 0
加說 13
加課 8
加諸 85
加護 444
加賽 14
加賽一場 5
加賽一場 5
加賽一場 5
加賽一場 5
加贈 31
加起來 115
加足 1
加足馬力 7
加車 1
加輪 1
加農砲 4
加退選 104
加送 69
加速 3281
加速卡 100
加速器 221
加速度 120
加速泵 0
加速進行 30
加速運動 5
加造 0
加進 76
加進來 50
加進去 75
加過 14
加過來 0
加過去 0
加選 42
加選法 0
加酒 11
加酸 11
加酸處理 2
加酸處理 2
加里 69
加里寧 0
加重 403
加重比例 1
加重計分 23
加錢 13
加錯 0
加鎖 11
加長 111
加開 95
加開班次 4
加附 77
加隊 1
加飛 5
加餐 12
加高 105
加鹽 37
加鹽加醋 0
加點 96
劣 450
劣勢 129
劣化 1
劣品 10
劣弧 0
劣性 1
劣根 10
劣根性 13
劣種 0
劣等 7
劣等品 1
劣等性 0
劣等貨 0
劣紳 0
劣質 33
劣跡 5
劣酒 2
劣馬 2
劤 0
劥 0
劦 91
劧 0
动 0
助 8168
助人 206
助人為快樂之本 18
助人為樂 2
助力 134
助力器械 0
助動詞 4
助威 11
助學 93
助學貸款 1
助學金 737
助戰 15
助手 141
助攻 84
助教 1164
助桀為虐 1
助消化 16
助燃 24
助理 3041
助理員 37
助產 65
助產士 110
助益 8404
助紂為虐 8
助聽器 116
助興 37
助虐 0
助詞 21
助語 0
助跑 21
助辯 0
助選 462
助選員 44
助選團 112
助長 116
助長聲勢 3
助陣 160
助餉 0
努 583
努力 6374
努力下去 10
努力以赴 31
努嘴 5
努嘴兒 0
努嘴兒 0
努爾哈赤 10
努瓦克肖特 0
努目 0
努著 0
努著嘴 0
努起 0
努起嘴 0
努起嘴來 0
劫 606
劫匪 5
劫去 4
劫奪 7
劫富濟貧 7
劫寨 11
劫後餘生 13
劫持 19
劫持人質 2
劫掠 36
劫數 21
劫數難逃 2
劫標 1
劫機 157
劫營 1
劫牢 0
劫獄 1
劫盜 0
劫簸 0
劫船 0
劫走 9
劫難 27
劬 8
劬勞 3
劭 101
劮 7
劯 0
劰 0
励 0
劳 0
劳 0
労 0
労 0
劵 0
劶 0
劷 0
劸 0
効 0
劺 0
劻 6
劼 233
劽 0
劾 62
势 0
勀 167
勁 1737
勁 1737
勁健 1
勁兒 13
勁兒 13
勁力 29
勁敵 27
勁旅 27
勁旅 27
勁直 3
勁草 7
勁裝 0
勁道 27
勁頭 2
勁頭十足 1
勁風 22
勁風 22
勂 11
勃 554
勃列日涅夫 0
勃勃 18
勃然 7
勃然大怒 12
勃發 29
勃興 9
勃谿 2
勃起 212
勄 0
勅 0
勆 0
勇 2426
勇冠三軍 2
勇勇敢敢 0
勇士 406
勇夫 8
勇將 9
勇往 15
勇往直前 69
勇悍 2
勇敢 596
勇敢善戰 3
勇於 349
勇於認錯 10
勇武 3
勇氣 669
勇氣十足 2
勇為 4
勇猛 65
勇猛過人 0
勇者不懼 2
勇退 1
勇銳 0
勈 0
勉 812
勉力 35
勉勉強強 5
勉勵 242
勉強 535
勉為 5
勉為其難 19
勊 0
勌 0
勍 135
勎 0
勏 0
勐 0
勑 0
勒 215
勒 2154
勒令 142
勒令 142
勒住 18
勒兵 2
勒死 18
勒索 146
勒緊 5
勒緊褲帶 3
勒贖 43
勒那海 0
勒馬 26
勓 7
勔 0
動 14987
動上 15
動不動 96
動亂 170
動亂不安 3
動了 177
動人 461
動人心弦 17
動人心魄 5
動令 10
動作 4834
動來動去 3
動兵 1
動刀 19
動刀動槍 1
動刑 2
動到 30
動力 2299
動力學 402
動力機 2
動力機械 147
動力裝置 0
動動 48
動動手 13
動動看 2
動動腦 98
動口 13
動名詞 11
動向 1986
動向不明 2
動員 942
動員令 66
動員戡亂 5
動員戡亂時期 50
動土 383
動地 130
動壓 10
動如脫兔 4
動容 78
動工 354
動干戈 2
動彈 49
動彈不得 54
動得 30
動心 79
動怒 23
動情 82
動感 471
動態 7248
動態平衡 19
動手 633
動手動腳 8
動手腳 18
動手術 44
動搖 185
動摩擦力 1
動植物 566
動槍 16
動機 1151
動武 144
動氣 7
動滑輪 4
動火 8
動物 6831
動物園 939
動物學 113
動物學家 17
動物性 109
動物性飼料 1
動物油 9
動物界 16
動物醫院 1
動物體 28
動狀詞 0
動產 204
動用 354
動畫 5406
動盪 119
動盪不安 41
動筆 45
動粗 9
動線 100
動聽 86
動能 176
動脈 318
動脈 318
動脈弓 7
動脈弓 7
動脈瘤 24
動脈瘤 24
動脈硬化 153
動脈硬化 153
動脈血 29
動脈血 29
動腦 212
動腦子 0
動腦筋 29
動腳 7
動蕩 11
動蕩不安 2
動覺 0
動詞 124
動詞變化 2
動議 74
動起 26
動起來 63
動身 29
動身前往 10
動輒 145
動輒得咎 12
動過 40
動過手腳 20
動配合 1
動量 153
動量矩 0
動電學 0
動靜 140
勖 8
勗 57
勗勉 8
勘 708
勘 708
勘亂 6
勘察 110
勘探 16
勘探員 0
勘探工程 0
勘探隊 1
勘查 260
勘測 33
勘測員 0
勘測隊 0
勘災 44
勘誤 20
勘誤表 46
勘驗 89
務 12123
務使 69
務實 674
務必 2967
務本 15
務求 85
務農 56
務農人家 0
務須 51
務頭 0
務頭 0
勛 362
勛勉 0
勛章 10
勜 0
勝 6368
勝 6368
勝不驕 7
勝之不武 2
勝了 33
勝仗 12
勝任 200
勝任愉快 27
勝似 3
勝利 1198
勝利在望 13
勝利果實 2
勝利歸來 1
勝利者 31
勝利隊 0
勝券 1
勝券在握 68
勝在 9
勝地 184
勝得 12
勝敗 51
勝敗乃兵家常事 1
勝於 136
勝景 57
勝會 3
勝樂金剛 1
勝率 0
勝的 68
勝算 139
勝訴 68
勝負 213
勝跡 5
勝過 259
勝過一個諸葛亮 1
勝過一個諸葛亮 1
勝過一個諸葛亮 1
勝部 9
勝部冠軍 2
勞 430
勞 4309
勞乏 0
勞人費馬 0
勞作 132
勞作課 1
勞保 394
勞保單 0
勞保局 77
勞倫斯 56
勞力 449
勞力士 23
勞力士錶 2
勞動 1131
勞動人民 0
勞動保護 12
勞動保險 5
勞動力 255
勞動化 0
勞動基準法 154
勞動定額 0
勞動對象 2
勞動工具 0
勞動工分 1
勞動強度 2
勞動改造 1
勞動新聞 0
勞動日 1
勞動時間 3
勞動條件 65
勞動模範 0
勞動法 36
勞動生產力 62
勞動生產率 5
勞動產品 1
勞動競賽 0
勞動節 33
勞動紀律 1
勞動群眾 2
勞動者 126
勞動英雄 0
勞動資料 8
勞動過程 9
勞動階級 5
勞務 344
勞基法 299
勞委會 0
勞工 5488
勞工保險 229
勞工保險局 53
勞工委員會 0
勞工退休準備金 1
勞工階層 2
勞工階級 8
勞師動眾 17
勞形 4
勞役 14
勞役地租 0
勞心 25
勞心者治人 1
勞改 13
勞改營 5
勞方 43
勞模 0
勞民 0
勞民傷財 10
勞煩 24
勞燕分飛 3
勞瘁 4
勞碌 17
勞碌命 53
勞神 8
勞累 85
勞績 2
勞而無功 2
勞苦 66
勞苦功高 20
勞資 311
勞資糾紛 60
勞資關係 157
勞資雙方 78
勞軍 36
勞軍團 2
勞逸不均 1
勞頓 7
勞駕 24
募 279
募兵 8
募兵制 13
募得 48
募捐 28
募款 488
募用 0
募集 487
勠 0
勡 0
勡 0
勢 3385
勢不兩立 18
勢之所趨 3
勢利 4346
勢利眼 5
勢力 4345
勢力範圍 34
勢在必得 16
勢在必行 67
勢均力敵 17
勢如破竹 15
勢將 180
勢必 775
勢必在 15
勢必會 71
勢必能 4
勢所必然 2
勢所難免 5
勢派 0
勢能 7
勢頭 12
勣 140
勤 3321
勤以補拙 1
勤僕 1
勤儉 54
勤儉務實 2
勤儉持家 5
勤儉樸實 7
勤儉為服務之本 1
勤勉 30
勤勉不倦 1
勤務 392
勤務兵 1
勤務員 0
勤勞 109
勤勞努力 1
勤奮 136
勤奮努力 8
勤學 51
勤學苦練 2
勤工儉學 3
勤快 47
勤懇 3
勤有功嬉無益 0
勤樸 12
勤洗勤換 0
勤王 4
勤益工專 27
勤練 45
勤能補拙 0
勤苦 8
勤謹 2
勥 0
勦 178
勦 178
勧 0
勨 0
勩 4
勪 0
勫 3
勬 0
勭 0
勭 0
勮 0
勯 7
勰 12
勱 27
勲 0
勳 1425
勳業 5
勳爵 1
勳章 154
勴 54
勵 1042
勵志 272
勵磁機 0
勵精圖治 14
勵行 28
勵行 28
勶 0
勷 75
勷 75
勸 716
勸世歌 2
勸住 2
勸出 0
勸出來 0
勸出去 0
勸到 0
勸勉 56
勸募 151
勸募者 0
勸化 12
勸告 81
勸和 1
勸善 6
勸回 1
勸回來 0
勸回去 0
勸學篇 0
勸導 100
勸慰 8
勸戒 4
勸捐 0
勸架 12
勸業 54
勸止 5
勸解 9
勸誘 6
勸說 102
勸諫 6
勸起 0
勸起來 0
勸農 1
勸退 91
勸進 79
勸過 4
勸酒 6
勸開 2
勸阻 39
勸阻無效 2
勸降 9
勺 97
勺 97
勺子 2
勻 597
勻出 2
勻出來 0
勻攤 0
勻整 3
勻染 23
勻淨 0
勻稱 29
勻速圓周運動 0
勻速直線運動 0
勻速轉動 0
勻速運動 0
勼 5
勽 0
勽 0
勾 880
勾上 2
勾下 1
勾中 25
勾了 4
勾住 15
勾來勾去 0
勾出 5
勾勒 211
勾動 12
勾勾 33
勾去 3
勾取 3
勾在 6
勾引 91
勾心鬥角 23
勾掉 1
勾搭 10
勾欄 91
勾消 5
勾留 5
勾畫 7
勾畫出 8
勾當 43
勾結 122
勾股定理 0
勾股弦定理 0
勾肩搭背 0
勾芡 40
勾起 84
勾踐 4
勾通 0
勾過 0
勾過來 0
勾過去 0
勾銷 3
勾除 0
勾魂 21
勿 2584
勿言 100
勿說 100
勿聽 100
勿視 19
勿食 673
勿庸贅言 0
勿忘 42
勿忘我 10
勿忘草 0
勿施於人 9
勿枉勿縱 9
匀 0
匃 0
匄 0
包 5823
包上 13
包上來 1
包上去 0
包下 17
包下來 10
包下去 0
包住 47
包傷 0
包公 14
包公案 1
包准 0
包包 115
包去 1
包吃 0
包吃包住 1
包吃包穿 0
包君 1
包君滿意 18
包含 2981
包含著 48
包商 120
包圍 330
包圍住 17
包圍圈 0
包圍著 28
包圍起來 8
包在 49
包場 15
包場 15
包夾 16
包子 259
包容 381
包容性 26
包工 239
包干 2
包庇 72
包廂 100
包心菜 12
包成 9
包扎 2
包打聽 9
包抄 7
包括 18289
包拯 6
包換 14
包攬 15
包攬生意 0
包書皮 0
包書皮紙 0
包梏 0
包機 79
包機來 0
包機去 0
包涵 2981
包產 5
包產到戶 8
包皮 96
包票 3
包租 12
包穀 1
包管 8
包箱 1
包紮 93
包紮法 9
包紮著 2
包給 30
包緊 1
包羅 41
包羅萬象 128
包腳 3
包船 2
包船來 0
包船去 0
包著 34
包藏 15
包藏禍心 4
包袱 227
包裝 4040
包裝品 8
包裝紙 53
包裹 320
包谷 0
包走 1
包起 13
包起來 26
包身 2
包車 11
包車來 0
包車去 0
包辦 97
包退 10
包醫 9
包銷 21
包青天 47
包頭 16
包頭市 0
包飯 1
匆 47
匆促 50
匆促之間 0
匆促間 1
匆匆 252
匆匆忙忙 35
匆卒 0
匆忙 105
匆忙之間 0
匆忙間 2
匆遽 1
匇 0
匈 117
匈奴 37
匈奴人 3
匈牙利 406
匈牙利人 2
匈牙利大平原 0
匈牙利語 4
匉 12
匊 242
匋 213
匌 0
匍 25
匍伏 5
匍匐 26
匍匐前進 4
匍匐運動 0
匎 43
匏 150
匐 482
匐伏 1
匐匍 2
匑 8
匒 3
匓 0
匔 0
匔 0
匕 23
匕首 70
化 1
化 23783
化了 114
化作 142
化做 1
化合 36
化合價 3
化合反應 0
化合性 0
化合態 0
化合物 702
化合量 0
化名 64
化境 10
化外 41
化外之民 2
化妝 554
化妝台 7
化妝品 1149
化妝室 12
化妝師 1
化妝棉 1
化妝水 1
化妝箱 28
化子 9
化學 8960
化學作用 39
化學元素 26
化學公司 87
化學分析 119
化學劑 11
化學反應 163
化學反應速度 0
化學合成 40
化學品 650
化學家 28
化學工廠 170
化學工業 1749
化學工程 793
化學平衡 20
化學式 44
化學性 98
化學性質 52
化學戰 8
化學戰劑 5
化學戰爭 1
化學方程 0
化學方程式 6
化學武器 20
化學當量 0
化學纖維 63
化學肥料 48
化學能 4
化學變化 33
化學鍵 18
化學需氧量 34
化工 3268
化工廠 135
化工業 159
化工系 408
化干戈為玉帛 6
化性 79
化成 513
化整為零 17
化敵為友 12
化武 17
化油器 76
化消 380
化為 380
化為泡影 2
化為烏有 15
化生 45
化痰 20
化痰止咳 2
化療 1
化石 506
化石林 0
化簡 0
化糞池 61
化緣 16
化繁為簡 15
化纖 203
化纖廠 35
化肥 35
化肥廠 0
化育 52
化腐朽為神奇 29
化膿 45
化膿性 58
化處 214
化處 214
化裝 50
化裝舞會 14
化解 545
化費 5
化身 174
化身為 51
化鐵爐 2
化開 26
化除 17
化險為夷 24
化雨 37
化零為整 2
化驗 142
化驗品 0
化驗室 21
化鴟為鳳 0
北 12495
北一 302
北一女 95
北一女中 0
北上 385
北上列車 1
北二高 0
北亞 54
北京 3815
北京人 25
北京城 15
北京大學 234
北京市 185
北京烤鴨 19
北京狗 31
北京猿人 4
北京話 27
北京鴨 2
北伐 48
北伐 48
北伐戰爭 0
北伐戰爭 0
北伐軍 4
北伐軍 4
北側 175
北冰洋 2
北勢溪 13
北區 2598
北半 3
北半球 128
北半部 5
北卡羅來納州 1
北周 9
北商 31
北回歸線 46
北國 153
北國風光 0
北地 192
北城 90
北埔 271
北埔 271
北大 267
北大荒 9
北大西洋 52
北安 60
北安路 82
北宋 76
北宜 83
北宜公路 41
北宜高速公路 31
北寒帶 1
北寧 75
北寧鐵路 0
北山 95
北山酒精 0
北山酒經 0
北岸 100
北市 7579
北市人 0
北市商 2
北師 137
北師專 3
北平 335
北平人 2
北平市 3
北平西路 0
北平話 3
北從 5
北愛爾蘭 46
北愛爾蘭省 0
北戴河 13
北房 2
北投 703
北投區 738
北援 0
北支 18
北斗 511
北斗七星 32
北斗星 17
北新 257
北方 546
北方人 9
北方話 1
北曲 546
北朝 17
北杜味拿河 0
北極 161
北極光 14
北極區 1
北極圈 15
北極星 95
北極海 6
北極熊 54
北橫 72
北橫公路 22
北歐 259
北歐國家 13
北段 32
北江 7
北洋 91
北洋大學 0
北洋武備學堂 0
北洋軍 0
北洋軍閥 4
北海 221
北海岸 134
北海道 282
北海道人 0
北涼 2
北港 510
北港人 0
北港溪 55
北港鎮 587
北溫帶 4
北濱 62
北濱公路 6
北瓜 1
北疆 8
北站 38
北端 101
北竿 41
北竿島 0
北約 62
北約組織 6
北緯 105
北緯度 0
北縣 7958
北縣人 1
北美 1147
北美洲 222
北義 546
北至 68
北藝 566
北藝大 566
北街 205
北赤道洋流 3
北起 30
北越 13
北越軍 0
北路 2435
北軍 3
北迴 44
北迴歸線 32
北迴鐵路 35
北邊 113
北郊 22
北部 1709
北部地區 268
北部灣 2
北醫 308
北門 662
北門一帶 1
北門一帶 1
北非 62
北非國家 0
北面 77
北韓 830
北風 48
北魏 24
北麓 8
北齊 4
匘 0
匙 382
匙 382
匙子 0
匚 8
匚 8
匛 0
匛 0
匜 16
匝 129
匝道 0
匞 0
匟 60
匟床 0
匠 309
匠人 9
匠心 22
匠心獨具 11
匠氣 8
匡 274
匡助 1
匡復 2
匡救 0
匡時濟世 0
匡正 39
匡濟 1
匡複 0
匢 218
匣 746
匣子 16
匣式 10
匣缽 1
匥 0
匧 0
匨 0
匨 0
匩 0
匪 265
匪人 2
匪偽 3
匪共 1
匪兵 0
匪區 2
匪史 0
匪夷所思 39
匪幫 1
匪幹 2
匪徒 34
匪患 0
匪情 5
匪懈 1
匪方 0
匪話 0
匪諜 29
匪貨 0
匪軍 1
匪酋 0
匪類 69
匪首 1
匪黨 0
匫 0
匫 0
匬 0
匭 6
匯 4263
匯兌 274
匯入 233
匯出 261
匯合 43
匯報 103
匯差 0
匯市 0
匯成 25
匯款 400
匯款人 9
匯款單 14
匯水 1
匯流 452
匯流排 0
匯流處 9
匯流點 2
匯為 6
匯率 2645
匯率差額 0
匯票 384
匯編 19
匯編語言 2
匯總 19
匯聚 97
匯豐銀行 113
匯費 2
匯錢 22
匯集 298
匯集點 1
匯電 0
匰 3
匱 160
匱 160
匱乏 97
匱竭 0
匲 0
匳 0
匴 11
匵 0
匶 0
匶 0
匷 4
匸 0
匹 121
匹 1214
匹夫 18
匹夫之勇 1
匹夫匹婦 4
匹夫懷璧 0
匹夫有責 0
匹敵 43
匹練 2
匹茲堡 90
匹薩 2
匹薩餅 0
匹配 88
匹馬單槍 0
区 0
区 0
匼 156
匽 19
匾 266
匾額 64
匿 1193
匿名 365
匿名信 16
匿情 0
匿戶 0
匿料 0
匿藏 0
匿跡 7
匿跡消聲 0
匿蹟 0
區 80
區 80114
區公所 313
區分 1876
區分法 2
區別 778
區別性 4
區別詞 0
區劃 226
區區 170
區區小事 100
區域 9172
區域化 10
區域性 511
區域間 26
區塊 100
區委 22
區有 137
區段 399
區運 210
區運會 23
區長 79
區間 403
區間車 8
區隔 595
十 83239
十一 4059
十一人 152
十一個 123
十一個 123
十一元 6
十一分 10
十一塊 1
十一天 26
十一年 250
十一日 902
十一月 5696
十一樓 96
十一歲 69
十一萬 54
十一號 147
十一點 195
十七 7941
十七人 45
十七個 66
十七個 66
十七元 10
十七分 12
十七塊 1
十七天 6
十七日 1110
十七歲 257
十七號 91
十丈 27
十三 3599
十三人 40
十三個 84
十三個 84
十三元 13
十三分 10
十三塊 3
十三天 11
十三太保 2
十三日 787
十三歲 97
十三經 23
十三號 111
十三陵 9
十三點 76
十下 7
十世 14
十世紀 11
十串 0
十之八九 22
十九 4196
十九人 101
十九個 31
十九個 31
十九元 8
十九分 6
十九塊 0
十九天 12
十九日 1029
十九歲 114
十九號 69
十二 6624
十二人 97
十二個 197
十二個 197
十二元 24
十二分 13
十二因緣 1
十二塊 7
十二天 39
十二平均律 5
十二年 344
十二律 0
十二指腸 151
十二日 954
十二月 7161
十二樓 53
十二歲 269
十二生肖 109
十二碼球 7
十二萬分 49
十二處 1
十二號 100
十二點 242
十五 4246
十五人 100
十五個 117
十五個 117
十五元 38
十五分 98
十五塊 5
十五天 85
十五日 1421
十五歲 271
十五號 140
十人 317
十人份 2
十付 2
十代 10
十令 0
十件 51
十件事 1
十任 12
十份 43
十位 236
十位數 8
十來 66
十來個 16
十來個 16
十來分 1
十來天 2
十信 36
十個 624
十個 624
十個人 20
十個人 20
十個月 110
十個月 110
十倍 233
十傳百 16
十億 578
十元 335
十克 8
十克拉 1
十全 191
十全十美 73
十全老人 0
十兩 40
十八 4120
十八人 83
十八個 101
十八個 101
十八元 24
十八分 10
十八地獄 1
十八塊 1
十八天 12
十八姑娘 0
十八姑娘一朵花 2
十八姑娘一朵花 2
十八層地 20
十八層地獄 1
十八日 1090
十八歲 417
十八界 1
十八相送 4
十八羅漢 51
十八般武藝 25
十八號 102
十八變 1
十公克 4
十公分 55
十公升 4
十公尺 100
十公斤 24
十公里 56
十六 5550
十六人 45
十六個 91
十六個 91
十六元 35
十六分 23
十六分音符 0
十六塊 1
十六天 19
十六日 1068
十六歲 209
十六號 108
十六進位 48
十六進制 6
十具 21
十冊 83
十冬臘月 0
十分 4826
十分寮 17
十分肯定 0
十分鐘 392
十則 53
十副 1
十劃 22
十加侖 0
十包 8
十包裝 0
十匙 0
十區 19
十卷 142
十卷 142
十口 1
十句 22
十句話 8
十只 4
十台 10
十名 481
十呎 25
十周 5
十周年 61
十員 11
十善 1
十四 3957
十四人 37
十四個 74
十四個 74
十四元 23
十四分 19
十四塊 0
十四天 56
十四日 1468
十四歲 165
十四號 74
十四行詩 5
十回 94
十圈 8
十國 42
十圓 7
十團 7
十坪 63
十堂 3
十堂課 1
十場 50
十場 50
十塊 35
十塊錢 25
十多 561
十多人 80
十多個 133
十多個 133
十多元 1
十多分 1
十多分 1
十多塊 3
十多天 33
十多年 163
十多次 7
十多歲 102
十多號 0
十夜 6
十大 1947
十大塊 0
十大建設 27
十大類 24
十天 389
十天份 0
十妹 0
十姊妹 0
十姊妹 0
十字 474
十字形 10
十字架 236
十字繡 0
十字花 24
十字街口 2
十字街頭 2
十字路 34
十字路口 86
十字軍 57
十字軍東征 1
十字鎬 0
十字頭 2
十客 1
十室九空 1
十家 125
十封 8
十封信 6
十小時 42
十尺 16
十局 12
十屆 568
十層 46
十層樓 9
十巷 25
十師 0
十常八九 5
十幀 1
十幅 15
十幢 0
十平方 9
十年 2598
十年前 242
十年後 105
十年樹木 8
十幾 335
十幾人 3
十幾個 68
十幾個 68
十幾元 2
十幾分 2
十幾塊 1
十幾天 16
十幾年 131
十幾次 13
十幾歲 39
十幾號 0
十床 21
十度 95
十座 42
十式 1
十張 151
十強 1
十惡 7
十惡不赦 21
十成 20
十成把握 1
十戒 13
十打 1
十拳 1
十拿九穩 3
十捲 8
十排 0
十支 42
十斤 21
十方 1
十方地面 1
十日 2879
十日份 0
十星期 0
十時 366
十晚 1
十曲 1
十月 6459
十月份 901
十月十日 96
十月革命 4
十有八九 5
十期 391
十朵 1
十朵花 0
十束 1
十杯 5
十杯水 0
十杯酒 1
十枚 12
十枝 5
十架 122
十株 7
十格 10
十框 0
十桶 2
十條 1643
十棟 8
十棵 10
十槍 0
十樓 211
十樣 2
十檔 9
十次 248
十次革命 2
十步 31
十歲 1365
十歲時 56
十段 32
十段變速 0
十殿閻王 0
十法界 1
十泡 0
十洞 3
十派 0
十滴 4
十炮 0
十營 2
十爺 0
十片 35
十片裝 4
十版 26
十獎 72
十班 62
十瓶 4
十瓶裝 0
十瓶酒 1
十甲 31
十男 3
十界 1
十發 4
十盒 6
十盞 2
十砲 0
十碗 1
十科 8
十秒 147
十秒鐘 30
十種 190
十立方 3
十站 6
十章 107
十筐 0
十筒 0
十箱 7
十節 35
十節課 0
十篇 98
十粒 6
十級 24
十組 69
十線 14
十罐 1
十美圖 0
十股 10
十臺 1
十船 5
十色 83
十萬 1269
十萬人 95
十萬個 37
十萬元 303
十萬八千里 15
十萬分 14
十萬分 14
十萬塊 1
十萬火急 41
十葉 1
十處 28
十號 277
十行 23
十行字 0
十行紙 17
十街 56
十袋 0
十角 6
十角形 0
十誡 54
十課 6
十賭九輸 0
十起 26
十趟 3
十足 506
十足把握 7
十足蟲 0
十路 37
十車 2
十車 2
十軍 3
十輛 29
十輛車 2
十輪 2
十通 10
十連 7
十週 89
十週年 186
十進位 31
十進位制 3
十進位碼 0
十進值 0
十進制 10
十進法 3
十遍 21
十道 11
十道菜 0
十邊 0
十邊形 0
十部 61
十鄰 16
十里 99
十里洋場 7
十里洋場 7
十重 5
十針 1
十錢 3
十鍋 0
十間 25
十關 4
十隻 95
十集 69
十面 18
十面埋伏 22
十面體 2
十頁 63
十頂 0
十項 197
十項全能 13
十項建設 5
十項運動 4
十頭 6
十顆 31
十題 32
十首 52
十首歌 2
十點 380
十點鐘 13
十齣 7
十齣戲 1
卂 0
千 22517
千伏 6
千佛洞 6
千億 323
千克 5
千刀萬剮 3
千分 53
千分之 521
千分之一 88
千分表 0
千千 77
千千萬萬 80
千卡 18
千古 137
千古恨 0
千古遺恨 0
千叮萬囑 3
千呼萬喚 40
千夫所指 7
千奇百怪 51
千嬌百媚 14
千字文 11
千家萬戶 1
千家詩 6
千層糕 0
千層麵 1
千山萬水 14
千島群島 4
千巖競秀 0
千差萬別 3
千年 497
千年怪獸 1
千思萬想 1
千恩萬謝 1
千態萬狀 1
千慮一失 0
千慮一失 0
千慮一得 1
千慮一得 1
千戶 59
千斤 31
千斤重擔 3
千斤頂 35
千方百計 47
千步 3
千歲 85
千瓦 35
千瓦小時 3
千畝 1
千瘡百孔 9
千百萬 18
千真萬確 27
千禧 0
千禧 0
千禧年 0
千禧年 0
千禧蟲 0
千禧蟲 0
千秋 201
千秋大業 0
千秋萬世 8
千秋萬代 1
千秋萬歲 0
千篇一律 26
千篇一律 26
千米 9
千粒重 1
千紅萬紫 1
千絲萬縷 8
千萬 2718
千萬富翁 6
千言 5
千言萬語 67
千變萬化 96
千赫 46
千軍萬馬 23
千載 23
千載一時 4
千載一時 4
千載難逢 39
千辛萬苦 55
千迴百折 0
千迴百轉 1
千里 482
千里目 1
千里眼 55
千里迢迢 53
千里送鵝毛 0
千里達 16
千里鏡 1
千里馬 397
千金 200
千金小姐 7
千金敝帚 1
千鈞一髮 23
千鈞一髮 23
千錘百煉 0
千錘百鍊 13
千門萬戶 3
千難萬險 0
千難萬難 1
千面人 4
千頁群島 0
千頭萬緒 39
千鳥 2
卄 0
卅 2445
卅一 8
卅七 8
卅三 8
卅九 8
卅二 8
卅五 8
卅八 8
卅六 8
卅四 8
卆 0
卆 0
升 3242
升上 91
升上來 5
升上去 6
升了 13
升任 147
升來 2
升來升去 0
升值 473
升入 15
升冪 1
升冪級數 0
升出 4
升出來 1
升到 91
升力 18
升勢 8
升勢受阻 0
升升 0
升回 3
升回來 0
升回去 0
升在 3
升堂 6
升堂入室 1
升堂拜母 0
升壓 10
升天 78
升好 2
升學 1468
升學主義 60
升學壓力 63
升學率 1
升學考試 45
升官 45
升官發財 6
升平 0
升得 4
升成 6
升斗小民 12
升旗 88
升旗典禮 25
升格 220
升水 24
升汞 0
升火 24
升為 107
升班 9
升班考試 0
升的 93
升空 146
升等 302
升等考試 97
升級 2586
升至 331
升華 8
升號 0
升記號 0
升調 9
升起 149
升起來 5
升速 3
升過 1
升遷 143
升遷制度 6
升遷管道 28
升降 224
升降梯 21
升降機 58
升降記號 2
升限 1
升階 5
升騰 3
升高 655
升高自下 0
升點 0
午 2418
午休 117
午前 41
午夜 371
午夜驚魂 5
午安 195
午後 366
午時 64
午時茶 0
午睡 39
午砲 0
午覺 20
午門 0
午間 292
午飯 40
午飯時間 3
午餐 1205
午餐時間 29
午餐費 10
卉 272
半 4655
半下 3
半世 8
半仙 24
半信半疑 35
半個 504
半個 504
半倒 7
半價 83
半價優待 4
半億 0
半兩 4
半分 26
半分鐘 14
半制品 0
半包 3
半句 30
半句話 6
半吊子 2
半吐半吞 0
半吐半露 0
半圈 11
半圓 38
半圓形 29
半圓規 0
半坪 2
半場 77
半場 77
半壁 10
半壁江山 4
半夜 382
半夜三更 8
半夜三更 8
半夜裡 11
半大 14
半大不小 10
半天 366
半失業 2
半子 1
半家 2
半封建 0
半封建半殖民地 2
半導體 6960
半小時 281
半局 3
半山 19
半山腰 22
半島 251
半工半讀 30
半年 1736
半年份 9
半年刊 130
半年報 0
半張 22
半形 1
半徑 299
半徑大小 1
半成品 138
半截 33
半截入土 0
半打 10
半推 3
半推半就 4
半掩半開 0
半支 4
半政府 0
半數 355
半數以上 116
半數以下 2
半文不值 0
半斤 33
半斤八兩 7
半新 6
半新不舊 4
半旗 4
半日 186
半日制 1
半晌 107
半月 288
半月刊 46
半月形 10
半本 7
半板 0
半枝 4
半架 1
半條 19
半棟 1
半棵 2
半步 14
半歲 6
半死 41
半死不活 10
半殖民地 1
半波整流 2
半波整流 2
半波整流器 4
半波整流器 4
半滿 6
半熟 13
半球 68
半球形 7
半瓶 19
半瓶醋 2
半生 148
半生不熟 2
半甲 4
半白 5
半百 34
半真 1
半真半假 1
半睡半醒 0
半睡眠狀態 0
半碼 11
半票 64
半票價 0
半私營 0
半秒 6
半秒鐘 2
半空 39
半空中 57
半章 0
半箱 0
半節 5
半節課 0
半粒 8
半精裝 0
半統靴 0
半自動 102
半衰期 80
半袋 1
半製品 8
半課 0
半買半送 4
半路 50
半路上 15
半路出家 0
半蹲 12
半身 208
半身不遂 28
半身像 6
半身照 1
半車 1
半輩子 6
半透明 81
半透明體 0
半途 57
半途上 1
半途而廢 31
半邊 71
半邊天 58
半部 100
半醒半睡 0
半里 14
半門 0
半開 67
半隊 0
半隻 14
半集 0
半面 13
半面之交 0
半面之舊 0
半音 15
半音階 7
半頁 12
半頭 0
半顆 7
半飽 3
半飽半餓 0
半餉 3
半首 20
半點 131
半點鐘 0
卋 0
卌 30
卍 109
协 0
卐 0
卑 291
卑下 10
卑不足道 0
卑俗 0
卑劣 34
卑劣性 0
卑南 612
卑南溪 17
卑微 78
卑怯 0
卑怯 0
卑濕 1
卑職 2
卑親屬 23
卑賤 18
卑躬屈節 0
卑躬屈膝 4
卑辭 1
卑鄙 56
卑鄙無恥 6
卑陋 0
卒 7600
卒 4
卒子 8
卒業 31
卓 15
卓 1558
卓有成效 4
卓溪 142
卓然 26
卓然有成 17
卓立 14
卓絕 12
卓著 104
卓蘭 331
卓見 22
卓識 2
卓越 953
卓越性 5
協 6309
協作 20
協力 388
協力車 23
協助 9545
協助人 4
協助者 20
協合 17
協同 394
協同作戰 5
協同動作 1
協和 363
協和醫院 7
協商 1498
協商會 4
協商會議 83
協奏 32
協奏曲 306
協定 2111
協尋 0
協律 3
協會 17171
協洽 1
協理 237
協約 52
協約國 5
協統 0
協調 3024
協調會 264
協調會議 64
協議 1493
協辦 712
協辦人 24
協辦者 1
協防 61
協防部 0
单 0
单 0
单 0
南 1
南 4586
南一中 48
南丁格爾 27
南下 528
南下列車 2
南中 44
南二中 97
南亞 736
南亞 736
南亞工專 32
南亞工專 32
南京 982
南京人 1
南京市 44
南京東路 2173
南京話 1
南人 13
南來 16
南來北往 11
南側 169
南充 9
南出 10
南到 34
南劇 11
南加羅林納洲 0
南勢溪 14
南勢角 51
南匈奴 4
南北 833
南北向 36
南北戰爭 17
南北方 5
南北朝 79
南北極 15
南北緯 5
南北議和 0
南北貨 174
南匯 2
南區 2335
南十字星 5
南半 5
南半球 117
南半部 8
南南北北 1
南卡羅來納州 18
南去 3
南口 30
南台 2504
南台工專 7
南史 2
南向 200
南呂 0
南和 112
南唐 28
南回歸線 3
南國 179
南國風光 1
南園 53
南城 10
南宋 70
南宋人 0
南宮 348
南寒帶 0
南寧 85
南寧市 8
南山 562
南山人壽 128
南岸 111
南島 127
南嵌溪 0
南嶺 3
南嶽 9
南市 406
南師 369
南平 97
南庄 237
南往 7
南征 15
南征北戰 3
南征北討 8
南戲 23
南投 2097
南投人 8
南投市 1151
南投縣 7384
南斯拉夫 161
南方 1142
南方人 7
南方大港 0
南方澳 81
南方話 0
南昌 177
南昌市 7
南昌街 68
南明 7
南曲 19
南有 18
南朝 36
南朝鮮 0
南柯一夢 8
南柯一夢 8
南極 350
南極光 1
南極圈 1
南極洲 31
南榮 206
南榮工專 10
南橫 132
南橫公路 81
南歐 60
南歐國家 2
南段 39
南沙 111
南沙群島 89
南洋 227
南洋國家 0
南洋群島 5
南流 3
南海 473
南海學園 58
南海岸 2
南海縣人 0
南海路 89
南涼 0
南港 903
南港區 577
南港國中 23
南港國小 28
南港路 230
南湖大山 33
南溫帶 1
南漢 8
南潯 0
南澳 243
南澳溪 3
南無 85
南無 85
南無耶 1
南無阿彌陀佛 45
南無阿彌陀佛 45
南燕 38
南王 56
南瓜 192
南由 2
南疆 10
南皮 1
南社 53
南科達 0
南站 17
南端 154
南管 191
南緯 40
南緯度 0
南縣 404
南羅得西亞 0
南美 239
南美各國 3
南美國家 7
南美洲 165
南胡 26
南腔 0
南腔北調 5
南自 6
南至 56
南臺 158
南臺工專 2
南船 3
南船北馬 1
南華 382
南蠻 6
南蠻鴃舌 0
南行 29
南街 235
南詔 2
南詞 11
南贍部洲 1
南越 23
南路 3703
南軍 7
南轅北轍 32
南迴 71
南迴鐵路 48
南通 95
南遊 1
南達 42
南邊 86
南郡 31
南部 1673
南部地區 307
南部地方 12
南鄭 5
南針 7
南門 325
南門國中 18
南門國小 20
南門市場 14
南門市場 14
南開 253
南開大學 25
南開工專 18
南關 3
南陽 292
南陽盆地 0
南陽街 57
南非 1163
南非共和國 23
南非各國 0
南非國家 14
南面 55
南韓 2015
南風 106
南飛 56
南麓 14
南齊 3
南齊書 1
単 0
単 0
単 0
卙 0
卙 0
博 18230
博愛座 100
博取 23
博古通今 3
博士 8999
博士學位 567
博士後 0
博士班 1902
博大 28
博大精深 26
博奕 38
博學 71
博學多聞 10
博客 1
博弈 7
博得 87
博愛 797
博愛路 333
博斯普魯斯海峽 4
博施濟眾 0
博浪鼓 1
博物 150
博物學家 2
博物院 21
博物館 4272
博聞 40
博聞強記 4
博茨瓦納 0
博覽 238
博覽會 1155
博覽群籍 1
博達 21
博雅 94
卛 0
卛 0
卛 0
卜 1032
卜卜 6
卜卦 60
卜卦家 0
卜居 9
卜筮 5
卜筮官 0
卜算 5
卜辭 10
卜骨 0
卝 0
卞 874
卞梁 0
卟 0
占 247
占 2473
占上風 7
占下 2
占了 61
占了 61
占位 6
占住 3
占便宜 1
占個 0
占優勢 10
占先 0
占光 0
占到 3
占卜 462
占卜 462
占卜術 3
占卜術 3
占占 0
占去 15
占取 2
占在 3
占地 63
占城 3
占多數 11
占好 0
占得 1
占掉 0
占據 36
占星 334
占星學 20
占星家 2
占星術 112
占有 442
占有權 0
占滿 4
占為己有 0
占用 71
占盡 10
占筮 4
占線 1
占著 2
占起 0
占領 54
占領者 0
卡 307
卡 30793
卡介苗 34
卡位 0
卡住 55
卡其 27
卡其 27
卡其布 1
卡其布 1
卡其褲 0
卡其褲 0
卡到 23
卡司 0
卡啦 25
卡在 214
卡塔爾 1
卡夫卡 31
卡尺 2
卡布其諾 4
卡帶 203
卡帶機 1
卡帶櫃 0
卡帶盒 0
卡帶箱 0
卡座 17
卡式 96
卡得 18
卡拉奇 0
卡拉揚 103
卡搭聲 1
卡榫 47
卡洛斯 22
卡爾 259
卡片 2300
卡片櫃 1
卡片盒 0
卡片箱 1
卡特 146
卡特爾 43
卡的 926
卡盤 1
卡紙 0
卡羅素 16
卡翁達 0
卡耐基 25
卡芬雅克 0
卡薩布蘭加 4
卡裡 7
卡諾 69
卡賓槍 9
卡路里 47
卡車 231
卡車司機 12
卡車工 0
卡迪拉克 0
卡通 2023
卡通世界 27
卡通影片 26
卡通片 30
卡通電影 12
卡里 33
卡門 144
卣 17
卤 0
卦 825
卦千紙 0
卦象 63
卧 0
卨 0
卩 0
卪 0
卬 20
卬 20
卭 0
卮 7
卯 218
卯上 61
卯上 61
卯勁 0
卯時 8
卯盡全力 0
卯起 2
卯起來 18
卯足 75
卯足了勁 19
卯金刀 1
印 5277
印上 27
印上來 0
印上去 1
印下 15
印下來 8
印下去 1
印不出 61
印不到 1
印人 7
印作 4
印來印去 0
印信 145
印光 1
印光大師 1
印出 511
印出來 76
印出去 1
印到 19
印刷 5272
印刷人 6
印刷品 182
印刷工 3
印刷廠 351
印刷業 489
印刷業者 23
印刷機 199
印刷術 30
印刷體 11
印加 22
印加帝國 8
印印 26
印售 1
印地安 143
印地安人 98
印地安納 3
印地安納大學 7
印地安納州 3
印地安話 0
印堂 9
印堂發黑 0
印契 1
印好 20
印子 7
印字機 32
印寫機 0
印尼 4642
印尼人 45
印度 2592
印度人 49
印度半島 11
印度大平原 0
印度尼西亞 15
印度支那 17
印度教 56
印度洋 200
印度豹 1
印式 1
印式料理 1
印得 6
印得 6
印成 26
印把子 0
印支 7
印有 76
印本 16
印染 865
印染廠 20
印模 101
印歐 17
印泥 21
印滿 3
印痕 17
印發 72
印的 104
印章 423
印第安 217
印第安人 67
印花 366
印花布 24
印花稅 143
印行 159
印表 167
印表機 3413
印製 603
印記 68
印証 27
印證 290
印象 2032
印象主義 19
印象派 70
印起 0
印起來 0
印過 7
印過來 0
印過去 0
印錯 9
印鑑 326
危 723
危亡 1
危亡之秋 0
危亡之際 0
危及 219
危在旦夕 80
危城 40
危境 2
危壁 0
危如 0
危如累卵 1
危室 0
危害 1162
危害到 36
危害性 74
危局 3
危崖 9
危急 167
危急存亡 7
危急關頭 4
危急關頭 4
危樓 17
危機 6041
危機四伏 41
危機重重 8
危殆 9
危牆 2
危疑 1
危篤 0
危若 1
危若累卵 0
危言 7
危言聳聽 250
危語 0
危險 2925
危險區 40
危險性 530
危險物 43
危難 44
卲 12
即 18300
即今 51
即令 50
即令是 9
即位 54
即使 3794
即使是 682
即便 256
即刻 357
即刻起 6
即可 8140
即告 37
即墨 4
即夜 0
即將 4295
即已 260
即席 26
即席演講 11
即或 31
即日 115
即日起 979
即早 32
即是 1613
即時 10920
即景 15
即期 100
即期支票 48
即為 874
即物窮理 0
即興 134
即興 134
即興曲 8
即興詩 0
即速 10
却 0
卵 561
卵圓形 7
卵塊 8
卵子 97
卵巢 397
卵巢癌 1
卵形 30
卵殼 6
卵泡 15
卵生 12
卵白 5
卵石 30
卵磷脂 48
卵管 2
卵細胞 7
卵翼 0
卵翼之下 0
卵胎生 4
卵裂 1
卵黃 15
卶 0
卷 722
卷 7222
卷 7222
卷入 5
卷取 3
卷土重來 0
卷子 6
卷子本 0
卷宗 48
卷尺 5
卷層雲 2
卷帙 30
卷帙 8
卷揚機 0
卷曲 7
卷染 1
卷煙 2
卷筒 3
卷緯 0
卷繞 1
卷裝 4
卷軸 30
卷軸 30
卷軸式 1
卷雲 3
卷領 0
卷髮 1
卷髮 1
卷鬚 0
卸 336
卸下 141
卸下來 3
卸下去 0
卸了 4
卸任 343
卸入 0
卸到 0
卸去 12
卸在 2
卸好 0
卸妝 68
卸妝油 1
卸完 4
卸得 1
卸成 0
卸法 0
卸的 4
卸裝 7
卸裝法 0
卸貨 63
卸貨機 0
卸責 8
卸車 2
卸載 25
卸載 25
卸進 0
卸進來 0
卸進去 0
卸過 0
卸過來 0
卸過去 0
卸除 10
卹 110
卹金 1
卺 0
卺 0
卻 17926
卻不 1308
卻不 1308
卻不是 187
卻不會 76
卻不能 213
卻之不恭 2
卻又 1162
卻回來 0
卻回去 0
卻因 537
卻在 657
卻是 2558
卻會 152
卻有 790
卻步 64
卻沒 225
卻沒有 535
卻沒能 70
卻由 58
卻能 239
卻要 202
卻說 265
卼 346
卽 0
卽 0
卾 0
卿 1724
卿卿 3
卿卿我我 8
卿士 0
卿大夫 5
卿大夫 5
卿相 2
厀 0
厂 73
厂 73
厃 0
厃 0
厃 0
厄 387
厄瓜多爾 36
厄運 46
历 0
厇 0
厇 0
厈 0
厈 0
厉 0
厉 0
厊 169
压 0
厎 12
厏 4
厒 86
厓 0
厓 0
厓 0
厔 7
厖 33
厗 26
厘 437
厘米 30
厘金 0
厙 61
厚 1571
厚人薄己 0
厚利 7
厚到 1
厚厚 29
厚厚的 76
厚古薄今 0
厚實 118
厚己薄人 0
厚度 677
厚彼薄此 1
厚待 3
厚得 1
厚德 31
厚德國小 5
厚德載福 0
厚恩 9
厚意 2
厚愛 35
厚望 21
厚板 18
厚植國力 6
厚此薄彼 6
厚漆 0
厚澤 20
厚片 15
厚生 207
厚用 0
厚的 115
厚祿 3
厚禮 7
厚粉 1
厚紙 12
厚紙板 19
厚臉 0
厚臉皮 150
厚著 1
厚著臉皮 14
厚葬 5
厚薄 40
厚薪 0
厚誼 0
厚誼 1
厚賜 3
厚賜 3
厚起來 0
厚道 24
厚重 120
厚顏 7
厚顏無恥 5
厚黑學 1
厜 7
厝 1295
厞 6
原 22784
原主 23
原人 19
原以 52
原以為 82
原件 178
原任 80
原位 83
原住民 4913
原作 123
原作者 299
原來 3932
原來如此 113
原來是 492
原價 444
原先 1261
原先是 37
原函數 0
原則 6204
原則上 701
原則性 59
原創 100
原創者 0
原動力 167
原動機 17
原原本本 11
原名 302
原告 501
原告人 2
原味 100
原因 7321
原因無他 14
原在 57
原地 214
原地打轉 4
原地打轉 4
原址 42
原型 210
原基 8
原始 3511
原始人 39
原始佛教 1
原始公社 2
原始公社制度 0
原始動物 0
原始憑証 3
原始時代 9
原始森林 36
原始碼 1000
原始社會 7
原始積累 0
原始舞 0
原始記錄 3
原始部落 21
原委 191
原委會 0
原子 914
原子價 3
原子光譜 9
原子反應 0
原子反應堆 0
原子學 1
原子序 13
原子序數 1
原子彈 70
原子數 6
原子時代 4
原子核 66
原子核反應堆 0
原子爐 8
原子筆 239
原子結構 22
原子能 297
原子能委員會 549
原子能發電 0
原子說 1
原子論 1
原子量 8
原定 173
原宥 10
原審 79
原封 80
原封不動 23
原廠 0
原式 7
原形 48
原形畢露 20
原形質 1
原從 2
原性 36
原意 193
原故 21
原文 4522
原文書 683
原料 2239
原於 6
原是 367
原有 1534
原木 176
原本 3744
原本在 108
原本是 294
原本會 4
原材料 103
原案 60
原棉 28
原樣 180
原汁 1
原油 286
原液 25
原液 25
原為 294
原無 29
原煤 1
原版 597
原版書 1
原物 12
原狀 141
原班 14
原班人馬 18
原理 2779
原生 261
原生動物 14
原生暈 0
原生質 9
原產 122
原產地 145
原用 10
原由 89
原礦 7
原種 111
原稿 168
原籍 34
原糧 0
原素 25
原線圈 0
原罪 61
原聲帶 100
原職 22
原能會 0
原腸胚 1
原自 8
原色 151
原葉體 0
原著 1369
原著者 15
原處 160
原蟲 86
原裝 334
原裝貨 1
原訂 187
原設 41
原詩 12
原諒 936
原貌 114
原質 5
原路 117
原鄉 191
原鄉人 1
原野 124
原野奇俠 1
原野遊俠 0
原隊 2
原隊人馬 0
原電池 2
原需 4
原音 107
原須 19
原體 63
原鹽 1
原點 148
厠 0
厠 0
厠 0
厡 0
厡 0
厤 3
厥 374
厥功甚偉 1
厥為 5
厦 0
厦 0
厧 5
厨 0
厩 0
厩 0
厪 0
厪 0
厫 0
厬 3
厭 153
厭 1531
厭世 15
厭倦 121
厭厭 3
厭惡 175
厭戰 4
厭戰心理 0
厭棄 12
厭氧 67
厭煩 82
厭物病 0
厭膩 3
厭舊喜新 0
厭食 50
厭食症 26
厯 0
厰 0
厱 0
厱 0
厲 243
厲兵秣馬 0
厲害 834
厲聲 55
厲聲叱斥 0
厲色 0
厲行 29
厲行節約 1
厲階 0
厲鬼 8
厳 0
厴 86
厷 0
厸 0
厸 0
厹 3
厹 3
厺 0
去 1
去 28394
去不得 3
去不得 3
去不成 4
去世 304
去了 1438
去事 9
去住 26
去你的 15
去信 12
去借 170
去偽存真 0
去函 82
去到 74
去勢 41
去去 96
去去就來 1
去向 131
去向不明 9
去吸 8
去夏 6
去官 2
去就 111
去尾 3
去年 4694
去年今天 0
去年度 47
去年此時 9
去得 34
去思 15
去惡從善 0
去掉 144
去接 62
去救 54
去暑 1
去歲 5
去死吧 14
去殼 7
去氧核糖核酸 0
去汙粉 0
去污 42
去污力 1
去污粉 5
去洗 41
去火 2
去煮 30
去玩 10
去留 92
去病 2
去的 1196
去粗取精 0
去聲 9
去職 54
去蕪存菁 22
去處 431
去路 35
去逝 22
去過 100
去除 1176
去雄 0
去雜去劣 0
去頭 6
去頭去尾 1
去騎 2
去騎 2
厽 0
厾 0
叀 0
叁 0
叁 0
叁 0
叁 0
参 0
参 0
参 0
参 0
參 1
參 200
參 743
參 743
參事 60
參佐 1
參加 14707
參加人 179
參加國 78
參加者 331
參半 36
參合 5
參商 1
參堂 0
參天 34
參奏 1
參將 3
參展 890
參差 24
參差不齊 48
參差錯落 0
參戰 27
參戰國 0
參拜 45
參政 215
參政權 29
參數 1482
參校 1
參照 704
參照卡 0
參照物 0
參看 110
參禪 40
參考 18504
參考值 462
參考價值 126
參考座標 3
參考性 33
參考手冊 104
參考文獻 552
參考書 709
參考書目 579
參考法 0
參考特藏 0
參與 27761
參與 27761
參與制 0
參與感 44
參與者 267
參與者 267
參見 1186
參見互照 0
參見注 0
參觀 6029
參觀團體 1
參觀者 440
參訪 0
參謀 127
參謀總部 1
參謀總長 77
參謀長 23
參謁 2
參議 42
參議員 160
參議院 98
參賽 708
參賽者 185
參贊 9
參軍 14
參透 23
參道 13
參選 2329
參酌 318
參量 2
參錯 0
參閱 1302
參院 135
參預 3
叄 0
叅 0
叅 0
叅 0
叅 0
又 32981
又愛 100
又沒 100
又再 170
又去 117
又及 2
又可 0
又哭又笑 11
又在 551
又快又好 7
又打又罵 1
又搖又擺 0
又是 1955
又會 374
又會到 0
又會在 8
又會有 24
又有 2114
又稱 1
又紅又專 0
又能 414
又能夠 22
又要 881
又說 396
又走 50
又進 37
又進來 3
又進去 1
又飢又渴 1
又驚又喜 18
叉 6
叉 65
叉 659
叉子 21
叉形 5
叉手 4
叉架 1
叉燒 16
叉燒包 197
叉腰 3
叉號 0
叉路 133
叉車 5
叉齒 1
及 18348
及人 572
及其 10213
及於 159
及早 550
及早準備 19
及時 1064
及時努力 1
及時行樂 11
及時雨 24
及期 69
及格 992
及格賽 0
及格邊緣 17
及物 116
及物動詞 3
及瓜而代 0
及第 1152
及至 105
及鋒而試 0
友 23874
友人 580
友台 27
友善 694
友好 264
友好代表 0
友好代表團 0
友好條約 13
友情 382
友愛 163
友方 5
友朋 24
友民黨 20
友生 11
友誼 546
友誼 546
友誼賽 29
友誼賽 29
友軍 12
友邦 326
友隊 9
友風子雨 0
友黨 0
双 0
双 0
反 6700
反 865
反中子 0
反串 49
反主為客 0
反之 406
反之亦然 20
反了 17
反交 6
反作用 14
反作用力 21
反來 5
反例 8
反侵略 3
反侵略戰 0
反侵略戰爭 0
反修 7
反個 1
反倒 150
反倒是 120
反側 4
反光 231
反光鏡 12
反共 96
反共抗俄 8
反其道 2
反其道而行 26
反函數 4
反切 5
反到 11
反制 177
反剪 0
反動 35
反動份子 1
反動份子 1
反動分子 0
反動分子 0
反動派 1
反反 7
反叛 69
反叛軍 22
反叛黨 0
反右派鬥爭 0
反向 157
反向而行 1
反告 1
反咬 8
反咬一口 1
反咬一口 1
反哺 63
反唇相譏 5
反唱 3
反問 79
反噬 7
反回頭 0
反坐 0
反坦克 5
反守為攻 9
反定理 0
反客為主 16
反寫 1
反射 881
反射光 29
反射動作 20
反射定律 1
反射弧 2
反射波 1
反射波 1
反射爐 3
反射角 2
反將 23
反將 23
反對 2288
反對派 38
反對票 10
反對者 62
反對黨 205
反導向 0
反導向飛彈 0
反差 420
反帝 8
反常 54
反彈 1228
反彈力 6
反得 10
反復 39
反德 1
反悔 50
反情報 7
反意 2
反感 118
反應 6804
反應出 103
反應器 262
反應堆 2
反應式 92
反應方程 1
反應時間 63
反應熱 12
反應爐 80
反應物 27
反成 42
反戰 19
反戰份子 0
反戰份子 0
反手 93
反托拉斯 0
反抗 231
反抗軍 14
反拍 4
反持 0
反掌 10
反排 0
反接 6
反推 13
反推力 3
反提 0
反撞力 0
反撲 135
反擊 316
反攻 210
反攻倒算 0
反攻復國 4
反攻復國大業 0
反攻為守 0
反敗為勝 34
反文兒 0
反文兒 0
反於 3
反映 1670
反映出 224
反映論 0
反是 22
反時鐘 0
反時鐘方向 2
反本 2
反杜林論 0
反核 0
反正 1253
反殖 10
反毒 234
反毒害 1
反比 33
反比例 1
反民主 7
反求諸己 9
反沖 0
反法 1
反法 1
反法西斯 0
反派 45
反浪費 1
反清 15
反清復明 15
反演 11
反潛 51
反潛艇 0
反為 16
反無 3
反煙 6
反煙運動 0
反照 16
反照鏡 0
反物質 27
反犬旁兒 0
反犬旁兒 0
反猶太 3
反璞歸真 12
反生 2
反白 127
反白字 5
反的 16
反目 9
反目成仇 14
反目無情 0
反相 42
反省 714
反省起來 0
反看 1
反碼 0
反穿 6
反粒子 0
反綁 4
反經 3
反美 15
反義 2
反義字 6
反義詞 4
反老還童 0
反而 2305
反而在 39
反而是 322
反而會 197
反胃 19
反能 7
反脣相譏 0
反芻 29
反芻動物 10
反英 3
反衝 3
反衝力 0
反被 34
反複 14
反襯 1
反要 7
反覆 642
反覆思量 5
反覆無常 13
反觀 283
反訴 2
反証 4
反証法 0
反詰 0
反話 40
反論 3
反諷 49
反諷 49
反證 25
反證法 1
反讀 0
反賓為主 0
反質子 1
反踢 1
反躬自問 0
反躬自省 6
反轉 553
反轉片 0
反退 1
反過 25
反過來 95
反過來說 37
反遭毒手 1
反邪歸正 1
反鎖 19
反間 50
反間計 12
反間諜 12
反電動勢 3
反霸 0
反面 121
反面人物 0
反面教員 0
反面教材 3
反面無情 1
反面角色 0
反面角色 0
反革命 7
反革命份子 1
反革命份子 3
反響 9
反顏相向 0
反顧 8
反餽 4
反饋 23
反駁 269
収 0
叏 0
叏 0
叐 0
叒 0
叒 0
叓 0
叔 629
叔伯 9
叔公 5
叔叔 478
叔姪 5
叔婆 1
叔嬸 0
叔孫通 0
叔本華 13
叔母 0
叔父 41
叔祖 5
叔祖母 0
叔舅 0
叕 0
取 6862
取下 151
取之不盡 33
取了 94
取代 1872
取代法 0
取保 3
取信 20
取信於 11
取信於人 6
取出 532
取出來 19
取勝 150
取去 11
取名 185
取向 461
取回 242
取回來 2
取回去 0
取巧 23
取得 8216
取快一時 0
取悅 46
取捨 117
取捨不定 0
取捨之間 3
取捨難定 0
取掉 1
取景 61
取暖 40
取暖法 0
取材 117
取材於 27
取材自 293
取樂 8
取樣 795
取款 21
取決 44
取決於 245
取法 29
取法乎上 5
取法於 4
取消 3452
取火 8
取用 252
取笑 47
取給 3
取經 64
取締 658
取締法 1
取義 6
取而代之 136
取自 319
取舍 2
取道 28
取長補短 2
取開 4
取鬧 1
取齊 1
受 15452
受不了 442
受不得 3
受不得 3
受之有愧 2
受之無愧 1
受了 279
受人 379
受人之託 2
受俘 0
受俸 0
受傷 1177
受傷害 84
受傷者 14
受刑 55
受刑人 136
受刑者 2
受到 5748
受制 125
受制於人 7
受創 0
受力鋼筋 0
受受 1
受命 54
受困 0
受壓迫 19
受夠 3
受夠了 32
受委屈 15
受孕 80
受孕率 7
受完 9
受害 398
受害人 155
受害者 472
受寒 9
受審 40
受寵 12
受寵若驚 15
受屈 0
受得了 13
受得住 4
受惠 266
受惠無窮 2
受戒 18
受托 5
受折磨 4
受持 1
受挫 125
受援 5
受損 527
受損率 0
受教 189
受教育 119
受文者 331
受有 60
受業 21
受檢 123
受權 10
受欺侮 0
受款 139
受氣 11
受氣包 2
受洗 111
受涼 2
受潮 40
受災 272
受災地區 6
受熱 52
受獎 96
受理 2224
受用 74
受用不盡 16
受用無窮 8
受田 2
受病 5
受益 425
受益不淺 1
受益人 224
受益費 85
受盡 53
受盡壓迫 0
受盡折磨 10
受禮 7
受禮人 1
受粉 5
受精 112
受精卵 53
受約束 3
受累 61
受罪 17
受罰 74
受聘 161
受胎 10
受胎率 13
受自 16
受苦 188
受苦受難 9
受著 31
受蘊 1
受訓 408
受訓人 5
受訓者 17
受託 221
受託人 64
受託承銷 0
受託承銷品 0
受訪者 511
受詞 9
受話器 8
受話機 0
受賄 40
受贈人 26
受辱 24
受辱 24
受過 352
受邀 214
受邀者 9
受錯 0
受阻 126
受阻於 5
受降 19
受降儀式 0
受降典禮 1
受限 119
受限於 273
受雇 191
受難 115
受難者 98
受騙 104
受騙上當 7
受驚 112
受點 5
叙 0
叚 0
叚 0
叛 149
叛亂 61
叛亂份子 3
叛亂份子 3
叛亂分子 0
叛亂分子 0
叛匪 0
叛國 24
叛國罪 3
叛將 12
叛徒 40
叛敵 0
叛變 40
叛賊 1
叛賣 0
叛軍 16
叛逆 160
叛逆者 2
叛離 9
叛離正軌 0
叛黨 2
叜 0
叜 0
叝 0
叞 0
叟 46
叠 0
叠 0
叡 180
叢 747
叢刊 685
叢密 1
叢山峻嶺 6
叢書 4308
叢木 1
叢林 574
叢林地 1
叢林戰 4
叢生 95
叢聚 4
叢脞 0
叢談 1
叢集 4
口 7545
口不擇言 10
口中 654
口乾 35
口乾舌燥 14
口交 109
口令 43
口伐 0
口伐 0
口供 36
口信 6
口傳 49
口傳心授 2
口兒 10
口兒 10
口內 63
口出 75
口出不遜 0
口出惡言 1
口出穢言 10
口到 17
口印 1
口口 38
口口聲聲 72
口吃 59
口吐 22
口吐白沫 8
口吻 106
口味 647
口味淡 0
口味重 6
口哨 32
口哨聲 3
口喊 3
口喊著 0
口器 23
口型 9
口外 40
口子 11
口實 6
口岸 42
口形 9
口徑 221
口德 19
口惠 14
口感 177
口才 108
口才好 0
口技 8
口拙 1
口授 18
口授法 0
口是心非 44
口服 723
口服心不服 1
口服心服 2
口服藥 33
口業 100
口氣 297
口水 616
口水直流 6
口沫 4
口沫橫飛 190
口渴 44
口渴難忍 0
口湖 316
口無擇言 0
口燥脣乾 0
口琴 181
口生蓮花 0
口白 100
口碑 222
口碑載道 0
口福 38
口糧 15
口約 96
口紅 145
口罩 118
口耳 4
口耳相傳 360
口脣 0
口腔 1064
口腔期 0
口腔炎 18
口腔癌 0
口腔癌 0
口腔衛生 102
口腔黏膜 0
口腔黏膜 0
口腹 5
口腹之慾 7
口腹之欲 0
口臭 70
口舌 45
口舌之爭 4
口若懸河 9
口蓋 16
口號 335
口蜜腹劍 1
口血未乾 0
口袋 434
口裡 38
口角 68
口角炎 12
口訣 45
口試 844
口誅筆伐 15
口語 282
口語化 22
口語體 0
口誤 1
口誦 6
口說 38
口說無憑 8
口譯 236
口譯員 20
口蹄疫 0
口述 235
口述出來 0
口述法 0
口部 53
口音 450
口頭 405
口頭上 58
口頭禪 34
口頭語 1
口風 18
口風琴 0
口香 30
口香糖 112
口齒 17
口齒不清 15
口齒伶俐 3
口齒清晰 8
古 5991
古井 38
古亭 143
古亭區 11
古亭國中 5
古亭國小 33
古人 312
古今 298
古今中外 115
古代 1086
古代人 11
古代史 14
古來 45
古典 2068
古典主義 17
古典作品 2
古典學派 22
古典文學 103
古典派 12
古典藝術 1
古典音樂 747
古剎 23
古厝 170
古古怪怪 0
古名 16
古國 39
古坑 5
古城 178
古堡 322
古塔 26
古墓 314
古墳 6
古妝 0
古字 5
古宅 40
古寺 0
古屋 128
古屋傳奇 0
古巴 167
古巴人 3
古希臘 145
古廟 25
古往今來 59
古徑 2
古怪 127
古意 52
古意盎然 27
古文 327
古文字 14
古文字學 13
古文學 1
古文觀止 6
古文體 0
古早 50
古早人 5
古時 97
古時候 52
古書 72
古書屋 2
古書店 5
古有明訓 3
古本 17
古板 220
古柯鹼 57
古樓 16
古樸 70
古法 39
古為今用 1
古物 242
古猿 0
古玩 83
古琴 35
古生代 9
古生物 119
古畫 31
古神 8
古神廟 0
古稀 7
古箏 57
古籍 240
古羅馬 29
古老 664
古聖先賢 0
古舊 17
古色古香 53
古董 346
古董商 7
古董店 9
古董鋪 0
古蘭經 3
古號 0
古裝 36
古裝戲 3
古裝片 2
古訓 6
古詩 170
古話 9
古語 8
古諺 7
古跡 23
古蹟 2003
古道 605
古道熱腸 4
古都 478
古銅 15
古銅色 17
古錢 10
古雅 12
古韻 0
古風 54
古體 0
古體文 0
古體詩 1
古龍 57
古龍水 30
句 116
句 13919
句句實話 0
句型 135
句型分析 3
句型練習 10
句子 394
句式 9
句數 1
句法 36
句法分析 1
句號 13
句讀 40
句踐 1
句點 1550
另 10126
另一 6261
另一回 2
另一回 2
另一回事 16
另一回事 16
另一方面 1630
另一方面 1630
另一樣 6
另一樣 6
另一次 78
另一次 78
另一種 1120
另一種 1120
另一邊 162
另一邊 162
另一面 101
另一面 101
另借 1
另再 13
另加 201
另取 15
另唱 0
另在 106
另外 8443
另定 123
另寄 10
另想辦法 2
另成 27
另成一家 0
另成一家 0
另打 0
另打主意 0
另接 4
另搞 1
另搞一套 0
另搞一套 0
另方面 153
另於 113
另日 11
另有 1982
另有企圖 1
另有打算 1
另有高就 1
另為 175
另當別論 100
另眼 72
另眼相待 1
另眼相看 14
另眼看待 3
另聽 0
另與 23
另行 536
另行通知 185
另訂 210
另計 130
另記 4
另請 72
另請高明 4
另謀 14
另謀出路 2
另議 76
另起 18
另起爐灶 17
另通 14
另開 51
另開一張 1
另開一張 1
另闢 72
另闢蹊徑 13
另類 500
叧 0
叧 0
叧 0
叨 39
叨 39
叨光 0
叨叨 8
叨叨絮絮 0
叨擾 19
叩 1678
叩來叩去 0
叩出 0
叩問 20
叩得 1
叩應 0
叩拜 100
叩見 50
叩診 2
叩謝 80
叩門 34
叩關 24
叩頭 25
叩首 70
只 1029
只 10295
只不過 812
只不過是 381
只亮 1
只亮出 0
只亮到 0
只住 5
只佔 106
只佔到 0
只作 45
只來 130
只來到 0
只借 1
只借不還 0
只借到 0
只做 106
只停 25
只傳 160
只傳出 0
只再 1
只准 26
只分 9
只刊 0
只判 1
只剩 415
只剩下 439
只加 12
只加到 0
只動 5
只包 3
只印 14
只去 190
只受 10
只受到 6
只可 135
只可以 36
只可在 18
只吃 58
只合 5
只含 37
只和 30
只唱 12
只唱出 1
只唱到 0
只問 54
只喊 4
只喝 26
只喝到 0
只回 40
只回來 1
只回到 0
只回去 1
只因 366
只因為 177
只圖 10
只在 546
只坐 5
只報 100
只塗 2
只多 80
只多不少 0
只夠 20
只夠到 0
只夠在 0
只套 0
只好 1705
只好來 12
只好去 23
只好在 45
只學 13
只學到 3
只學會 1
只守 0
只守不攻 2
只守到 0
只定 1
只害 0
只寫 34
只寫出 1
只寫到 0
只寫在 0
只封 1
只射 0
只射到 0
只將 45
只對 180
只差 92
只帶 36
只幫 6
只幫忙 1
只延 0
只得 442
只得到 22
只從 35
只怕 263
只怕是 11
只怕會 5
只怕有心人 10
只怪 32
只想 638
只想來 1
只想出 0
只想到 30
只想去 1
只憑 57
只懂 22
只懂得 10
只應 140
只應 14
只成 12
只戴 1
只打 47
只找 120
只找出 100
只找到 32
只把 63
只抱 0
只拜 0
只拼 0
只挑 7
只捉 1
只捉到 1
只排 8
只採 12
只採到 0
只接 110
只接到 10
只提 53
只插 0
只插到 0
只搬 0
只撞 2
只擋 0
只擺 3
只收 48
只收到 9
只放 24
只敢 30
只數 1
只數到 0
只是 11560
只是在 375
只是會 28
只是有 73
只是為了 190
只替 1
只會 1029
只會到 0
只會在 55
只有 23531
只有在 575
只查 6
只查出 0
只查到 3
只比 61
只求 156
只求無過 1
只求自保 0
只派 8
只流 0
只流出 0
只流到 0
只消 23
只漏 0
只漏出 0
只漏到 0
只演 5
只為 350
只為 350
只為了 163
只照 4
只照到 1
只煮 0
只猜 30
只猜出 100
只猜到 100
只玩 130
只產 16
只用 224
只用到 7
只留 56
只留下 89
只留到 0
只畫 8
只當 26
只發 39
只盡 0
只盡到 0
只看 198
只看出 10
只看到 130
只睡 14
只睡到 0
只砍 0
只租 17
只租不買 0
只租不賣 1
只種 3
只稱 7
只穿 25
只站 5
只站到 0
只等 37
只答 4
只管 78
只管來 1
只管出 0
只管到 0
只管去 2
只管在 2
只約 40
只紅 2
只紅到 0
只經 3
只經過 2
只編 2
只縫 0
只罰 5
只考 35
只考出 0
只考到 0
只聘 0
只聞 34
只聽 201
只聽到 44
只肯 15
只肯來 0
只肯出 0
只肯到 0
只肯去 0
只肯在 1
只背 1
只能 4024
只能在 347
只能有 53
只花 57
只衝 0
只被 40
只補 2
只裝 150
只要 13332
只要在 457
只要是 564
只要會 29
只要有 528
只要能 234
只見 1225
只訂 3
只記 11
只記到 1
只記在 10
只記得 58
只設 13
只設到 0
只設在 0
只許 20
只許成功 1
只說 153
只說不做 20
只說出 30
只說到 30
只說是 40
只調 6
只調 6
只講 23
只講出 0
只講到 2
只讀 14
只變 0
只讓 52
只讓出 0
只讓到 0
只買 25
只買到 10
只費 2
只賠 1
只賣 168
只賭 0
只賺 18
只賺到 0
只贏 4
只贏不輸 0
只走 8
只走到 1
只趕 1
只趕來 0
只趕出 0
只趕到 0
只趕去 0
只趕回 0
只跌 2
只跟 23
只跟來 0
只跟到 0
只跟去 0
只跟在 0
只跳 0
只踢 1
只踩 1
只踩到 0
只載 3
只輸 6
只輸到 0
只轉 4
只轉 4
只轉到 0
只轟 0
只辦 9
只追 2
只追到 1
只追回 0
只送 11
只送不賣 8
只通 6
只通到 0
只進 11
只進到 0
只過 5
只過來 0
只過去 0
只達 36
只配 7
只量 2
只量出 0
只量到 0
只銷 0
只鋪 0
只錄 4
只錄到 0
只鍍 0
只開 37
只開來 0
只開出 2
只開到 4
只開去 0
只開回 0
只闖 0
只闖出 0
只闖到 0
只防 1
只降 1
只降到 0
只限 131
只限到 0
只限在 2
只限於 154
只陪 1
只需 875
只需要 440
只露 3
只露 3
只露出 9
只露出 9
只露到 0
只露到 0
只靠 102
只須 274
只領 3
只領到 1
只願 78
只願意 100
只顧 76
只顧到 0
只顧在 1
只飛 2
只養 6
只點 7
叫 20029
叫上 13
叫上來 0
叫上去 1
叫下 1
叫下來 0
叫下去 0
叫了 165
叫些 2
叫住 25
叫作 139
叫個 19
叫做 1011
叫停 29
叫價 0
叫出 211
叫出來 28
叫出去 1
叫到 37
叫動 0
叫化 39
叫化子 9
叫化子雞 1
叫去 17
叫叫 7
叫喊 27
叫喊聲 7
叫喊起來 0
叫喚 28
叫喚聲 2
叫嚷 17
叫嚷聲 2
叫囂 22
叫回 12
叫回來 5
叫回去 0
叫好 75
叫好不叫座 12
叫好又叫座 25
叫完 2
叫對 2
叫屈 16
叫座 22
叫座不叫好 0
叫座又叫好 0
叫得 17
叫急 7
叫慣 2
叫渴 1
叫牌 36
叫的 47
叫給 3
叫罵 16
叫罵聲 3
叫聲 199
叫花子 2
叫苦 25
叫苦連天 34
叫著 103
叫號 14
叫號 14
叫賣 36
叫賣聲 4
叫起 12
叫起來 15
叫車 27
叫進來 2
叫進去 1
叫過 11
叫過來 4
叫過去 2
叫醒 65
叫錯 3
叫門 8
叫門聲 0
叫陣 41
叫鬼 4
叫魂 1
叫點 1
召 190
召 1903
召 1903
召之即來 1
召募 94
召募 94
召喚 190
召回 164
召妓 0
召安 1
召租 0
召租 0
召見 92
召見 92
召請 10
召開 3502
召開大會 8
召開會議 154
召集 865
召集人 1116
召集人 1116
召集令 61
叭 133
叭叭 10
叮 182
叮人 1
叮人戰術 0
叮叮 20
叮叮噹噹 6
叮咚 1
叮噹 469
叮噹聲 0
叮噹響 0
叮嚀 196
叮囑 19
叮當 6
可 1
可 73172
可上 189
可上來 2
可上去 2
可下 211
可下來 1
可下去 0
可不 770
可不可 44
可不可以 50
可不是 264
可不會 21
可不能 79
可不行 8
可乘 29
可乘之機 17
可也 63
可也沒 4
可也沒有 0
可人 55
可以 61525
可以說 710
可以說是 877
可作 253
可佩 15
可侖坡 0
可供 1246
可供參考 96
可信 102
可信度 96
可倫坡 64
可停 34
可傳 23
可先 631
可先來 3
可先去 1
可免 202
可免於 11
可分 293
可判 36
可別 270
可到 288
可加 109
可加可減 0
可動性 2
可厭 6
可去 102
可取 59
可取之處 17
可受 43
可口 178
可口可樂 184
可可 187
可可亞 21
可可樹 3
可可粉 6
可可豆 11
可向 443
可否 2840
可哀 2
可售性 0
可唱 6
可問 26
可喜 95
可喜可賀 37
可嘆 18
可嘉 29
可圈可點 38
可在 1770
可坐 24
可塑 33
可塑性 46
可壓 10
可壓縮 12
可壓縮性 0
可多 127
可多可少 1
可大 84
可大可小 10
可好 38
可好了 9
可學 35
可守可攻 1
可容 55
可對 145
可小可大 1
可少不了 1
可少可多 0
可就 328
可巧 2
可延 13
可往 20
可得 715
可怕 980
可怖 40
可恥 92
可恨 73
可悲 159
可惜 1574
可惜的是 176
可惡 443
可惱 3
可想而知 109
可愛 3185
可慢可快 0
可慮 60
可憎 22
可憐 1224
可憐兮兮 0
可憐可憫 1
可憐相 2
可憐虫 0
可憐蟲 22
可把 172
可投 37
可拉 90
可拔 2
可捐 0
可控硅 0
可搖 20
可擋 32
可收 163
可攻 10
可攻可守 2
可放 106
可救 26
可敬 54
可敬可佩 2
可數 17
可數名詞 7
可於 522
可是 9094
可是會 67
可是有 87
可是要 97
可曾 141
可有 460
可有可無 113
可望 1737
可望在 99
可望而不可及 13
可期 244
可染 25
可樂 349
可樂餅 1
可欺 4
可歌可泣 28
可歎 4
可求 28
可求 28
可汗 60
可沒 65
可沒 65
可沒有 34
可沿 7
可溶 11
可溶性 63
可演 6
可為 423
可為 423
可無 36
可燃 84
可燃性 166
可燃物 12
可爬 2
可用 2749
可用到 3
可用在 40
可用性 36
可畏 17
可當 130
可疑 197
可疑人物 4
可看 634
可看性 60
可真是 96
可真會 5
可真有 5
可真能 0
可真要 2
可知 694
可知性 1
可租 23
可穿 20
可笑 189
可笑的是 21
可算 38
可算是 103
可組 17
可緊可鬆 0
可編 22
可聞 24
可聽 73
可聽性 6
可背 5
可能 19216
可能再 104
可能在 614
可能性 874
可能是 3016
可能會 2101
可能有 624
可能要 363
可與 583
可藏 10
可蘭經 15
可行 833
可行之道 22
可行性 846
可被 192
可要 237
可見 2244
可見一斑 54
可見一斑 54
可見光 97
可見度 100
可見得 320
可親 62
可觀 388
可說 480
可說是 1194
可謂 563
可議之處 19
可讀 60
可讀性 41
可變 255
可變性 40
可變資本 3
可變電容 2
可變電容器 3
可貴 189
可貴性 0
可賠 0
可起來 0
可逆 89
可逆反應 0
可造之材 6
可進 86
可進來 0
可進去 0
可進可退 1
可遇而不可求 15
可運 26
可運到 0
可過 31
可過來 0
可過去 0
可邀 2
可都 140
可都是 35
可鄙 4
可配 10
可鍛鑄鐵 4
可長可短 1
可防 39
可限 3
可陪 4
可隨 91
可隨著 7
可靠 1020
可靠性 136
可願 4
可願意 1
可風 21
可騎 5
可麗餅 92
可點 45
台 32727
台上 401
台下 184
台中 11480
台中人 32
台中市 8520
台中港 571
台中盆地 12
台中縣 13859
台前 89
台前台後 1
台北 33023
台北人 137
台北區 199
台北市 48552
台北縣 23571
台北訊 248
台南 7709
台南人 30
台南大學 1
台南市 7223
台南縣 9627
台商 2113
台啟 0
台地 129
台基 14
台塑 801
台大 6334
台大學生 47
台大醫院 842
台子 18
台安 122
台客 1
台尺 20
台州 7
台布 7
台幣 1478
台式 169
台後 103
台拉維夫 7
台文 0
台斤 51
台東 2664
台東人 11
台東市 1752
台東縣 4053
台柱 16
台步 7
台汽 205
台汽客運 87
台海 446
台海兩岸 132
台澎 35
台澎金馬 2936
台灣 124258
台灣人 1850
台灣區 1871
台灣地區 4862
台灣大學 4310
台灣海峽 196
台灣玉 3
台灣省 12748
台灣省立 1958
台灣銀行 480
台灣鋁業 1
台灣鋁業公司 0
台灣電力 114
台灣電力公司 232
台照 3
台燈 47
台獨 1157
台獨運動 41
台玻 47
台球 0
台甫 1
台省 296
台端 95
台糖 418
台糖公司 146
台聯 12
台聯黨 12
台肥 124
台胞 64
台胞證 39
台航 71
台菜 34
台菜餐廳 1
台藝 480
台裝 6
台裝貨 0
台製 124
台製品 6
台西 324
台視 817
台視公司 6
台詞 57
台語 1168
台資 177
台車 124
台金 26
台銀 243
台銜 0
台鋁 5
台鐵 0
台鑒 1
台階 78
台電 1010
台電公司 407
台音 0
台風 19
台風穩健 1
台駕 3
台鳳 1022
台鳳公司 42
叱 88
叱吒 38
叱吒風雲 22
叱喝 8
叱責 2
史 10799
史上 859
史丹佛大學 95
史冊 5
史前 435
史前時代 21
史博館 0
史學 678
史學家 41
史學界 10
史官 6
史家 63
史家莊 0
史密斯 130
史實 106
史料 478
史書 55
史東 1
史無前例 76
史特勞斯 52
史特林堡 33
史特龍 33
史瓦濟蘭 0
史瓦辛格 280
史略 37
史籍 44
史蒂文斯 2
史蒂文生 6
史記 214
史評 7
史詩 90
史語所 280
史論 89
史跡 13
史蹟 244
史迪威 13
史達林 33
史館 104
右 3153
右上 94
右上方 34
右上角 112
右下 72
右下方 30
右下角 99
右乳 3
右侍郎 0
右側 341
右傾 16
右傾 16
右傾分子 0
右傾分子 0
右傾機會 1
右傾機會 1
右傾機會主義 5
右傾機會主義 5
右券 0
右前 28
右前方 17
右半 20
右半塊 0
右半球 1
右半部 10
右外野 1
右後 19
右後方 100
右後衛 1
右心室 51
右心房 29
右手 566
右手定則 3
右手掌 12
右手邊 33
右拳 20
右方 149
右旋 12
右派 33
右派分子 1
右眉 1
右眼 56
右移 0
右端 6
右翼 47
右耳 21
右肩 36
右肺 9
右腎 3
右腦 82
右腳 85
右腿 34
右腿骨 0
右膝 8
右膝蓋 1
右臂 29
右臂 29
右行 5
右軍 5
右轉 182
右邊 660
右鉤拳 0
右鍵 500
右面 12
右頰 3
叴 0
叴 0
叵 16
叵測 4
叶 0
号 0
号 0
司 7958
司令 198
司令台 36
司令員 5
司令官 33
司令臺 3
司令部 126
司儀 44
司務 0
司南 3
司命 5
司塔斯弗 0
司寇 1
司徒 107
司書 19
司業 11
司機 800
司機員 3
司法 2007
司法制度 36
司法官 181
司法機關 194
司法權 43
司法獨立 22
司法界 13
司法程序 45
司法部 220
司法部長 24
司法院 778
司法院長 44
司空 14
司空見慣 38
司農司 0
司鐸 9
司長 282
司馬 392
司馬光 61
司馬懿 89
司馬昭 9
司馬昭之心 6
司馬遷 38
叹 0
叻 26
叻 26
叼 1399
叼著 14
叼走 4
叽 0
叿 0
吀 0
吀 0
吁 63
吁吁 24
吁氣 0
吁請 0
吂 0
吃 72
吃 7286
吃上 13
吃上官司 6
吃下 87
吃下來 30
吃下去 23
吃不上 0
吃不下 36
吃不了 60
吃不來 0
吃不出 1
吃不出來 10
吃不到 29
吃不完 48
吃不消 48
吃不開 3
吃了 587
吃了飯 2
吃人 54
吃來 4
吃來吃去 2
吃光 27
吃八方 1
吃出 106
吃到 253
吃到飽 100
吃力 153
吃力不討好 32
吃勁 0
吃十方 1
吃去 6
吃吃 84
吃吃 84
吃吃喝喝 57
吃啞巴虧 0
吃喝嫖賭 0
吃喝玩樂 236
吃奶 44
吃奶的力氣 6
吃完 214
吃官司 8
吃宵夜 120
吃得 227
吃得 227
吃得下 3
吃得下 3
吃得了 0
吃得了 0
吃得來 0
吃得來 0
吃得出 2
吃得出 2
吃得出來 0
吃得出來 0
吃得到 5
吃得到 5
吃得完 0
吃得完 0
吃得消 3
吃得消 3
吃得苦中苦 4
吃得苦中苦 4
吃得起 4
吃得起 4
吃得開 6
吃得開 6
吃成 7
吃掉 150
吃敗仗 3
吃本 0
吃水 82
吃水線 2
吃水量 0
吃油 8
吃法 69
吃現成飯 0
吃癟 0
吃盡 28
吃相 0
吃眼前虧 4
吃硬 4
吃硬不吃軟 3
吃糖 32
吃糧 0
吃素 273
吃緊 162
吃肉 100
吃膩 70
吃苦 35
吃苦耐勞 25
吃苦頭 0
吃苦頭 0
吃茶 17
吃草 40
吃著 66
吃藥 299
吃藥前 3
吃藥後 5
吃藥時 50
吃虧 142
吃虧就是佔便宜 2
吃裡扒外 1
吃角子老虎 33
吃豆乾 2
吃豆腐 23
吃貨 0
吃起 15
吃起來 63
吃起飯來 0
吃軟 7
吃軟不吃硬 8
吃軟飯 1
吃醋 74
吃重 38
吃錢 4
吃錯 5
吃錯藥 7
吃閉門羹 12
吃閒飯 3
吃食 43
吃飯 678
吃飯時間 17
吃飽 134
吃飽沒事幹 2
吃香 70
吃驚 90
吃鱉 0
吃鴨蛋 2
吃鹽 5
吃齋 9
吃齋唸佛 3
各 37051
各上 10
各上來 0
各上去 0
各下 2
各不 83
各不相同 39
各不相謀 0
各不相讓 1
各享 1
各人 360
各人自掃門前雪 0
各付 2
各付各的 0
各代 20
各以 45
各件 19
各份 1
各伍 0
各位 11424
各作 17
各使 43
各來 3
各例 2
各供 8
各依 48
各個 1642
各個擊破 5
各借 1
各做 12
各具 107
各兼 0
各出 28
各分 287
各別 201
各到 0
各削 0
各削 0
各剩 0
各剪 0
各創 61
各創一格 0
各創一格 0
各加 14
各勝 5
各包 0
各區 600
各區域 141
各升 0
各半 46
各印 0
各去 1
各取 30
各取所需 33
各吃 1
各各 44
各吊 0
各向 8
各向異性 3
各吸 0
各唱 7
各唸 0
各喝 2
各單位 1628
各回 7
各回合 2
各因 9
各國 4998
各園 2
各團 53
各團體 43
各在 38
各地 3671
各地區 530
各地方 224
各型 284
各型各式 0
各城 16
各城市 62
各執己見 2
各報 235
各場 11
各場 116
各填 2
各大 2343
各大報 68
各奏 1
各奔 9
各奔前程 4
各如 8
各如其人 0
各如其面 1
各季 22
各學 608
各安職守 0
各室 20
各家 1053
各寄 4
各射 2
各對 6
各就 19
各就各位 9
各局 60
各層 220
各層次 18
各屬 14
各島 19
各島嶼 9
各巷 1
各市 18
各市區 1
各帶 12
各幫 0
各幫派 1
各年 29
各年次 1
各年級 193
各床 0
各廂 0
各廠 149
各廳 47
各延 1
各建 3
各式 2976
各式各樣 753
各彈 6
各形 4
各形各色 9
各往 7
各得 368
各得其所 3
各忙 1
各忙各的 2
各念 1
各想 3
各憑 8
各憑本事 9
各懷鬼胎 14
各截 0
各戴 0
各戶 49
各打 13
各抒己見 6
各投 41
各投其好 0
各拍 1
各持己見 8
各指 22
各排 6
各接 6
各揍 0
各提 32
各搖 0
各搞 0
各搞各的 0
各搬 0
各搭 0
各撥 1
各播 1
各播 1
各撿 0
各擠 0
各攜 0
各攤 3
各攻 1
各救 0
各敷 0
各數 8
各數 8
各方 1125
各方矚目 8
各方面 603
各施 12
各族 86
各替 0
各會 27
各有 1238
各有勝負 0
各有千秋 10
各有巧妙不同 2
各有所好 5
各有輸贏 0
各服 2
各期 326
各期刊 16
各架 0
各校 2105
各梳 0
各棒 2
各榜 0
各樓 24
各樓層 111
各樣 150
各次 67
各段 27
各段落 3
各殺 0
各比 3
各河 6
各河流 2
各沿 0
各洗 11
各派 141
各添 5
各減 3
各湖 0
各湖泊 0
各演 5
各潑 0
各為 190
各為 190
各版 71
各版面 4
各牽 0
各玩 5
各班 423
各班次 12
各班級 31
各理 8
各生 24
各用 17
各界 1680
各界人士 176
各異 112
各當 8
各發 31
各的 116
各盒 0
各盟 0
各盟邦 0
各盡 13
各盡所能 6
各省 152
各省市 46
各省縣 0
各睡 1
各社 95
各科 1429
各種 13185
各種事 20
各種人 10
各種各樣 41
各立 14
各立門戶 0
各端 4
各笑 0
各答 0
各箱 0
各節 60
各簽 0
各粒 0
各系 505
各系所 392
各級 3701
各級領導 7
各級黨委 0
各組 733
各給 2
各線 28
各線路 1
各縣 151
各縣市 1961
各罰 0
各聽 1
各背 1
各自 1265
各自為政 37
各舉 31
各色 80
各色各樣 10
各處 567
各號 7
各行 47
各行其是 7
各行各業 333
各行業 158
各街 6
各街道 3
各被 11
各裝 8
各要 7
各記 11
各設 31
各試 17
各說 5
各說各的 2
各說各話 54
各課 18
各調 9
各談 2
各謀 1
各謀出路 0
各謀生路 0
各講 2
各講各的 0
各賞 1
各走 4
各走各的 3
各路 139
各路人馬 36
各踢 0
各車 131
各車廂 2
各車次 1
各車輛 2
各輯 3
各輸 2
各辦 24
各述己見 2
各送 6
各進 12
各過 0
各道 2
各道路 3
各邀 1
各部 339
各部會 214
各部門 324
各鄉 39
各鄉鎮 293
各里 68
各釣 2
各鋪 1
各鋪 1
各鍵 2
各門 26
各門派 2
各開 27
各間 5
各闖 0
各闖各的 0
各附 10
各降 6
各降 6
各院 53
各院系 34
各隊 121
各隊伍 3
各階 10
各階層 161
各集 9
各需 6
各項 8425
各項目 35
各須 5
各頒 14
各領 18
各領風騷 12
各頭 0
各題 26
各類 4798
各顯神通 19
各養 5
各餐 4
各騎 1
各黨 90
各黨派 66
吅 0
吆 17
吆喊 0
吆喝 39
吇 43
合 200
合 9900
合上 29
合不來 6
合久必分 2
合乎 562
合乎邏輯 9
合付 1
合伙 33
合伙人 15
合住 9
合作 12876
合作主義 0
合作化 3
合作商店 1
合作學 1
合作所 12
合作無間 30
合作社 685
合作金庫 315
合併 2419
合併在 19
合借 3
合做 9
合出 3
合到 7
合則兩利 6
合劑 172
合力 222
合十 33
合口 2
合吃 4
合合 28
合同 263
合唱 515
合唱團 1096
合唱隊 2
合圍 6
合在 28
合在一起 22
合在一起 22
合報 22
合夥 173
合夥人 110
合奏 184
合奏曲 7
合好 8
合婚 15
合子 1
合定 5
合宜 134
合家 16
合家歡 73
合并 7
合府 1
合度 9
合式 16
合彈 0
合影 215
合影留念 38
合得 15
合得 15
合得來 12
合心 7
合情 14
合情合理 23
合情理 2
合意 75
合成 1849
合成樹脂 87
合成橡膠 109
合成氨 2
合成洗滌 0
合成洗滌劑 6
合成物 8
合成纖維 156
合成體 0
合抱 15
合拍 21
合掌 29
合採 4
合攏 12
合數 19
合於 449
合時 26
合會 249
合板 201
合校 11
合格 2579
合格者 250
合歡 187
合歡山 207
合江省 2
合法 1721
合法化 107
合法性 108
合法鬥爭 0
合洗 1
合派 0
合流 103
合流處 1
合演 47
合為 47
合照 322
合照留念 3
合營 17
合牽 0
合班 43
合理 2781
合理化 305
合理合情 27
合理密植 0
合理布局 0
合璧 7
合瓣 1
合用 118
合發 57
合眼 9
合眾 53
合眾國 13
合眾社 3
合租 22
合站 0
合算 14
合簽 6
合約 1795
合組 122
合編 50
合縫 2
合縱 85
合縱 85
合群 61
合群性 2
合而為一 82
合聲 17
合股 19
合肥 27
合肥人 0
合肥市 3
合胃口 0
合著 198
合葬 14
合蓋 5
合補 2
合訂本 174
合計 2128
合調 9
合調 9
合請 6
合諧 15
合謀 6
合議制 10
合議庭 27
合讀 2
合資 599
合購 55
合起 18
合起來 50
合踢 0
合踩 0
合身 44
合轍 1
合轍押韻 0
合辦 387
合通 3
合適 744
合金 503
合金元素 1
合金鋼 81
合釣 0
合鋸 0
合閘 0
合音 28
合領 0
合養 24
合騎 4
合體 36
吉 18300
吉人 7
吉人天相 3
吉他 781
吉他手 77
吉他社 235
吉兆 6
吉光片羽 16
吉凶 68
吉凶未卜 1
吉利 221
吉利果 1
吉利話 0
吉卜賽人 3
吉安 163
吉布提 0
吉慶 34
吉房 1
吉日 18
吉星高照 6
吉普 707
吉普賽 59
吉普賽人 13
吉普車 298
吉期 0
吉林 152
吉林人 0
吉林市 11
吉林省 42
吉林路 204
吉爾 216
吉田 399
吉祥 498
吉羊 6
吉蔑族 0
吉貝 97
吉達 65
吉隆坡 308
吊 726
吊上 5
吊上來 1
吊上去 2
吊下 1
吊下來 0
吊下去 0
吊了 6
吊來吊去 0
吊個 0
吊兒郎當 4
吊兒郎當 4
吊到 1
吊去 0
吊取 0
吊吊 0
吊嗓 0
吊嗓子 0
吊嗓子 0
吊好 0
吊子 2
吊帶 51
吊帶褲 8
吊床 10
吊得 0
吊得 0
吊掛 69
吊掛在 2
吊掛著 0
吊杆 0
吊架 32
吊桶 1
吊桿 17
吊橋 186
吊死 18
吊燈 92
吊環 12
吊盤 1
吊索 8
吊著 11
吊裝 47
吊褲 2
吊褲帶 1
吊襪 3
吊襪帶 10
吊走 1
吊起 10
吊起來 9
吊車 250
吊鉤 11
吊銷 56
吊銷執照 3
吊鐘 12
吊飾 1
吋 2833
同 14414
同一 3212
同一性 27
同上 487
同下 28
同中有異 0
同事 726
同人 593
同仁 2249
同仇敵愾 14
同伙 7
同伴 350
同位 17
同位素 160
同位角 0
同位語 0
同來 157
同來同往 1
同做 4
同僚 58
同儕 201
同出 14
同出同進 0
同分 57
同分數 1
同到 29
同化 100
同化作用 7
同區 36
同升 1
同去 67
同右 16
同吃 8
同吃同住 0
同名 180
同名同姓 9
同名數 0
同向 29
同唱 14
同國 36
同在 357
同地 11
同坐 13
同堂 8
同夥 40
同好 767
同姓 16
同姓同名 0
同學 8353
同學會 2559
同學錄 36
同宗 16
同定 5
同室操戈 6
同宿 3
同對 16
同居 182
同居人 22
同屋 0
同工 490
同工不同酬 15
同工同酬 16
同工異曲 1
同工異調 1
同左 78
同年 608
同年代 6
同年同月 9
同年同月同日 6
同年級 15
同床 17
同床異夢 6
同度 20
同張 7
同往 27
同心 310
同心協力 125
同心同德 5
同心圓 44
同志 4877
同性 213
同性戀 717
同性相斥 3
同情 382
同情心 63
同情者 1
同意 5478
同意書 392
同意權 197
同感 205
同房 10
同房兄弟 0
同打 5
同指 2
同排 4
同支 0
同改 1
同文 27
同文同種 16
同旁內角 0
同日 90
同日生 7
同日而語 5
同是 174
同是天涯淪落人 1
同時 16778
同時代 15
同時候 1
同月 277
同月同日 4
同有 21
同期 1655
同枕共眠 0
同校 45
同校 45
同案 29
同梯 14
同業 831
同業公會 9543
同樂 155
同樂會 12
同樣 3386
同樣在 53
同樣是 250
同樣會 28
同機 10
同正 25
同步 2091
同步進行 30
同步電動 0
同步電動機 9
同歸 11
同歸於盡 20
同母 2
同母異父 4
同氣 7
同治 74
同派 2
同流合汙 3
同流合污 6
同渡 3
同源 65
同溫 6
同溫同壓 1
同溫層 10
同為 282
同父 22
同父異母 12
同班 56
同班同學 73
同理 230
同理可證 10
同甘共苦 15
同生 23
同生共死 13
同生死 3
同產 5
同用 10
同當 10
同病相憐 18
同盟 215
同盟國 27
同盟會 33
同盟條約 1
同盟罷工 0
同盟軍 16
同省 9
同知 16
同科 18
同種 61
同窗 79
同站 7
同章 2
同等 319
同等學力 212
同等學歷 82
同篇 3
同系 20
同約 3
同級 470
同素異形 0
同素異形體 2
同素異性 0
同素異性體 0
同線 2
同縣 7
同義 35
同義字 66
同義詞 25
同聲 72
同職 28
同胞 575
同胞兄弟 4
同胞愛 11
同舟共濟 14
同蓋 1
同號 12
同行 412
同行 412
同行同業 0
同行是冤家 0
同袍 12
同被 38
同訂 1
同論 1
同謀 13
同謀者 0
同質 128
同走 9
同跑 0
同路 724
同路人 8
同車 28
同軸 75
同軸電纜 67
同輩 27
同進 5
同進同出 5
同過 5
同道 87
同達 2
同邊 3
同鄉 390
同鄉會 0
同酬 29
同重 8
同量 8
同長 12
同長 12
同門 49
同隊 16
同音 35
同音字 19
同音詞 1
同項 15
同題 6
同類 285
同類相殘 5
同類項 0
同體 45
同高 13
同點 14
同黨 87
同齡 31
同齡林 0
名 73301
名下 90
名不副實 1
名不副實 1
名不符實 5
名不虛傳 19
名不見經傳 25
名主 12
名人 1514
名人錄 175
名份 16
名伶 11
名位 31
名作 279
名儒 7
名兒 10
名兒 10
名冊 1216
名分 31
名列 295
名列前茅 61
名利 169
名利客 0
名利雙收 18
名副其實 40
名劍 26
名動金甌 1
名勝 372
名勝古蹟 120
名古屋 258
名古屋人 0
名句 41
名叫 365
名品 101
名品店 22
名單 5374
名嘴 38
名器 7
名園 21
名地 10
名垂不朽 0
名垂青史 0
名城 92
名堂 30
名士 18
名女人 9
名如其人 0
名媛 21
名子 54
名字 4014
名字 4014
名存實亡 6
名學 165
名家 891
名實 21
名寺 1
名將 106
名山 64
名山大川 12
名帖 3
名帖 3
名師 154
名師出高徒 1
名廚 57
名手 14
名揚 82
名揚四海 8
名教 13
名數 11
名曲 245
名書 48
名望 17
名校 133
名條 32
名次 704
名次表 19
名歌 12
名歌手 10
名歌星 3
名正言順 29
名氏 4
名氣 218
名流 109
名滿 6
名滿天下 12
名為 730
名片 946
名片盒 7
名牌 527
名犬 23
名狗 1
名理 4
名琴 22
名產 125
名畫 188
名畫家 14
名目 130
名目繁多 2
名稱 37168
名符其實 44
名筆 7
名節 9
名簿 32
名系 38
名義 707
名義上 34
名義工資 0
名聞中外 3
名聞於世 2
名聞遐邇 27
名聲 143
名聲遠播 1
名腳兒 0
名腳兒 0
名色 1
名花 9
名花有主 3
名菜 82
名落孫山 9
名著 418
名號 205
名表 86
名角 10
名言 141
名言錄 13
名詞 1346
名諱 2
名譟一時 5
名譟一時 5
名譽 379
名譽博士 18
名譽會長 23
名譽職 2
名貴 62
名車 102
名過其實 0
名酒 17
名醫 157
名錶 50
名門 58
名門世家 0
名門貴族 1
名門閨秀 1
名間 100
名間鄉 224
名額 1742
名馬 16
名駒 5
名鳥 1
后 1475
后冠 16
后土 11
后妃 10
后座 6
后稷 1
后羿 29
后里 710
吏 90
吏治 13
吏部 0
吐 5716
吐 1062
吐下 1
吐下來 0
吐下去 0
吐來吐去 0
吐出 176
吐出來 38
吐到 4
吐司 26
吐司麵包 0
吐吐 19
吐吐氣 1
吐在 5
吐好 0
吐成 1
吐掉 4
吐氣 76
吐沫 0
吐瀉 4
吐番 3
吐痰 12
吐絮 0
吐絲 18
吐舌 15
吐舌頭 7
吐舌頭 7
吐苦水 24
吐著 8
吐蕃 24
吐血 90
吐血 90
吐谷渾 3
吐過 1
吐過來 0
吐過去 0
吐露 58
吐露出 6
吐魯番 24
向 14000
向上 670
向下 536
向人 115
向來 464
向來是 80
向來都是 13
向例 0
向光 9
向光性 2
向內 86
向內走 0
向前 710
向前看 8
向前看齊 0
向前走 75
向前進 6
向北 104
向北看 1
向北走 2
向南 118
向南看 1
向南走 6
向右 163
向右看 0
向右看齊 0
向右轉 13
向右轉走 0
向善 48
向地 97
向地性 2
向培良 1
向外 362
向外看 3
向外走 2
向天 98
向學 206
向導 17
向左 129
向左看 3
向左看齊 0
向左轉 7
向左轉走 0
向度 80
向往 10
向後 186
向後看 2
向後走 11
向後轉 33
向後轉走 0
向心 96
向心力 128
向性 75
向戌 2
向斜 11
向日 110
向日性 0
向日葵 224
向晚 34
向東 120
向東看 0
向東走 5
向海 108
向火 5
向用 17
向秀麗 0
向背 8
向著 161
向西 129
向觸 3
向量 548
向量分析 7
向陽 245
向陽大道 0
向隅 65
向風針 0
吒 68
吓 0
吓 0
吔 1
吕 0
吘 6
吙 5
吚 0
君 7397
君主 76
君主主義 0
君主制 0
君主國 1
君主專制 4
君主專制制 0
君主專政 0
君主政體 2
君主權 0
君主立憲 12
君主立憲制 1
君士坦丁 6
君士坦丁堡 11
君子 355
君子之交 7
君子之交淡如水 2
君子動口不動手 1
君子協定 4
君權 12
君王 153
君臨天下 14
吜 4
吝 78
吝嗇 67
吝嗇鬼 3
吝惜 23
吞 228
吞下 59
吞下來 0
吞下去 12
吞了 18
吞了下去 8
吞併 7
吞到 3
吞口水 10
吞吐 21
吞吐量 21
吞吞 4
吞吞吐吐 13
吞咽 4
吞噬 110
吞嚥 93
吞恨 0
吞掉 8
吞服 12
吞沒 32
吞滅 7
吞聲 6
吞聲忍氣 1
吞蝕 1
吞進 8
吞進去 3
吞雲吐霧 8
吞食 59
吟 1094
吟味 0
吟哦 19
吟唱 71
吟詠 27
吟詩 55
吟詩作對 3
吟誦 21
吟遊 104
吠 241
吠叫 11
吠形吠聲 0
吠影吠聲 0
吠聲 2
吠聲吠影 0
吠陀 241
吡 0
吡 0
吢 0
吣 0
吤 212
吥 7
否 2
否 2590
否則 4793
否定 410
否定一切 1
否定句 4
否定性 1
否定詞 3
否極泰來 32
否決 138
否決權 20
否認 1287
吧 2135
吧 21351
吧女 8
吨 429
吨 429
吩 77
吩咐 284
吪 7
含 12
含 12464
含嗔 10
含上 10
含下 4
含了 23
含入 17
含入口中 1
含入口內 0
含冤 13
含冤而死 1
含冤而終 0
含冤負屈 0
含到 3
含含糊糊 4
含在 57
含在口中 4
含在口內 0
含垢忍辱 0
含垢忍辱 0
含得 1
含怒 1
含怨 6
含怨而死 0
含怨而終 0
含恨 15
含恨在心 0
含恨而死 2
含恨而終 1
含情 8
含情脈脈 7
含意 131
含有 1937
含氧酸 3
含氧量 29
含氫 2
含水 141
含水層 39
含水量 73
含沙 9
含沙射影 7
含沙量 6
含油 28
含油層 0
含油量 5
含淚 73
含混 23
含混 23
含混不清 8
含混不清 8
含片 2
含笑 155
含笑九泉 0
含笑而死 1
含笑而終 0
含糊 51
含糊不清 8
含糊其詞 2
含羞 19
含羞草 26
含義 84
含苞 8
含苞待放 14
含著 92
含蓄 85
含蘊 7
含蠟 0
含血 5
含血噴人 2
含血噴人 2
含起 2
含起來 0
含辛茹苦 5
含過 19
含酸 10
含量 1394
含金量 1
含飴弄孫 10
听 780
吭 34
吭 34
吭氣 2
吭聲 6
吮 139
吮吸 2
吮墨 1
启 0
吰 9
吰 9
吱 202
吱 202
吱吱 33
吱吱 33
吱吱叫 4
吱吱叫 4
吱吱喳喳 17
吱吱喳喳 17
吱吱響 0
吱吱響 0
吱唔 2
吱唔 2
吱喳 6
吱喳 6
吱聲 2
吱聲 2
吲 0
吲 0
吳 22293
吳下阿蒙 5
吳中 87
吳佩孚 2
吳元薰 0
吳國 242
吳家驤 0
吳庭艷 0
吳廣 14
吳敬梓 5
吳梅 32
吳淞 13
吳淞江 4
吳王 59
吳祿貞 0
吳縣 2
吳興 188
吳起 7
吳越 8
吳越同舟 0
吳郡 3
吳郭魚 123
吳鎮 8
吳鳳 408
吳鳳工專 16
吳鳳廟 10
吴 0
吵 1119
吵上 0
吵來吵去 9
吵吵 4
吵吵嚷嚷 0
吵吵鬧鬧 21
吵嘴 6
吵嚷 6
吵架 165
吵著 45
吵起 8
吵起來 5
吵過 5
吵過來 0
吵過去 0
吵雜 85
吵鬧 69
吶 221
吶喊 164
吶喊助威 1
吶喊助陣 2
吷 13
吸 2011
吸上 3
吸上來 2
吸上去 1
吸下 0
吸下來 1
吸下去 2
吸了 61
吸住 13
吸來 1
吸來吸去 0
吸光 52
吸光性 0
吸入 546
吸出 21
吸出來 5
吸出去 0
吸到 14
吸力 33
吸去 3
吸反 0
吸取 220
吸取經驗 9
吸吮 76
吸吸 12
吸咽者 0
吸回 1
吸回來 0
吸回去 0
吸地 0
吸地板 1
吸地毯 0
吸塵 28
吸塵器 96
吸墨 2
吸墨紙 1
吸完 3
吸引 2741
吸引人 300
吸引住 34
吸引力 387
吸得 4
吸得 4
吸成 0
吸掉 7
吸收 2278
吸收光譜 96
吸收劑 39
吸收力 13
吸收塔 8
吸收度 3
吸收性 25
吸斷 1
吸棉 0
吸毒 99
吸毒者 10
吸氣 77
吸水 131
吸水性 67
吸水紙 2
吸汗 34
吸汗性 4
吸油煙機 0
吸法 4
吸濕 36
吸濕性 20
吸煙 183
吸煙客 0
吸煙室 1
吸煙者 23
吸熱 18
吸的 14
吸盤 38
吸管 83
吸管蟲 0
吸綿 1
吸菸 0
吸著 35
吸蟲 26
吸血 51
吸血鬼 122
吸走 13
吸起 2
吸起來 3
吸跑 0
吸透 0
吸進 33
吸進來 1
吸進去 5
吸遍 0
吸遍 0
吸過 7
吸過來 0
吸過去 2
吸金 62
吸鐵 2
吸鐵石 1
吸附 396
吸附劑 40
吸附力 6
吸附性 12
吸附水 0
吸食 82
吸食毒品 11
吸飽 2
吹 108
吹 1084
吹上 4
吹上來 0
吹上去 0
吹下 5
吹下來 0
吹下去 0
吹乾 34
吹了 40
吹來 68
吹來吹去 1
吹倒 13
吹入 25
吹出 44
吹出來 4
吹出去 0
吹到 30
吹動 10
吹去 2
吹口哨 8
吹向 30
吹吹打打 1
吹哨 4
吹哨子 4
吹哨聲 1
吹喇叭 17
吹噓 53
吹噓大王 0
吹奏 65
吹奏樂 0
吹奏樂器 5
吹彈 14
吹得 41
吹得 41
吹得天花亂墜 0
吹得天花亂墜 0
吹成 5
吹打 8
吹拂 45
吹捧 6
吹掉 3
吹散 27
吹樂 1
吹毛求疵 17
吹氣 194
吹法 3
吹法螺 0
吹火 0
吹灰 2
吹灰之力 2
吹煉 2
吹熄 10
吹牛 90
吹牛大王 0
吹的 48
吹皺一池春水 9
吹皺一池春水 9
吹直 9
吹笛 26
吹管 21
吹箭 11
吹簫 11
吹著 65
吹襲 50
吹走 19
吹起 118
吹起來 2
吹進 13
吹進來 1
吹進去 0
吹開 4
吹頭 0
吹頭髮 5
吹風 91
吹風機 105
吹髮 5
吹鬍子瞪眼睛 1
吹鼓手 1
吺 0
吻 952
吻別 40
吻別時 0
吻合 146
吻抱 1
吻頸 0
吼 255
吼到 1
吼叫 29
吼吼 5
吼聲 32
吼著 10
吼起來 2
吽 280
吽 280
吾 1583
吾人 433
吾儕 6
吾兄 15
吾子 7
吾師 34
吾愛 88
吾王 6
吾等 66
吾輩 12
吾黨 6
吿 0
吿 0
呀 5235
呀 5235
呀呀 37
呀呀學語 0
呁 227
呂 4478
呂不韋 1
呂后 10
呂宋 41
呂布 42
呂洞賓 1
呂赫若 1
呃 352
呃喔 1
呃逆 0
呄 0
呅 11
呆 100
呆 1005
呆了 178
呆了 178
呆人 0
呆人 0
呆住 19
呆住 19
呆呆 245
呆呆 245
呆呆 245
呆呆 245
呆呆傻傻 0
呆呆傻傻 0
呆呆傻傻 0
呆呆傻傻 0
呆呆掙掙 0
呆呆掙掙 0
呆呆掙掙 0
呆呆掙掙 0
呆呆笨笨 0
呆呆笨笨 0
呆呆笨笨 0
呆呆笨笨 0
呆在 41
呆在 41
呆子 64
呆子 64
呆帳 190
呆帳 190
呆帳損失 4
呆帳損失 4
呆想 2
呆想 2
呆料 9
呆料 9
呆板 53
呆板 53
呆楞楞 0
呆楞楞 0
呆滯 28
呆滯 28
呆滯無神 0
呆滯無神 0
呆笑 1
呆笑 1
呆笨 0
呆笨 0
呆若 1
呆若 1
呆若木雞 8
呆若木雞 8
呆著 10
呆著 10
呆裡呆氣 0
呆裡呆氣 0
呆裡呆氣 0
呆裡呆氣 0
呆賬 0
呆賬 0
呆頭 9
呆頭 9
呆頭呆腦 2
呆頭呆腦 2
呆頭鵝 9
呆頭鵝 9
呇 467
呈 2757
呈上 26
呈交 12
呈報 182
呈文 5
呈獻 27
呈現 3572
呈現出 351
呈給 8
呈繳本 0
呈請 92
呈送 47
呈遞 20
呈閱 4
呈露 2
呉 0
告 43
告 4365
告一段落 183
告一段落 183
告假 8
告假還鄉 0
告別 333
告別式 36
告吹 12
告密 8
告密者 3
告急 340
告慰 12
告成 14
告捷 28
告擾 0
告朔 0
告朔 0
告狀 28
告病 4
告發 128
告白 634
告知 2696
告示 225
告示板 45
告終 23
告罄 3
告罪 1
告老 3
告老還鄉 13
告訴 10825
告訴乃論 40
告誡 78
告謝 2
告警 18
告貸 2
告貸無門 2
告辭 53
告退 12
告饒 1
呋 0
呋 0
呋 0
呌 0
呍 0
呎 434
呏 50
呐 0
呐 0
呑 0
呒 1
呔 17
呚 0
呝 0
呞 0
呟 0
呠 15
呡 68
呢 1724
呢 17247
呢喃 66
呢喃細語 0
呢子 0
呢絨 126
呣 9
呤 142
呥 144
呦 8541
呧 8
周 9601
周一 95
周三 48
周二 61
周五 148
周人 41
周代 17
周作人 1
周傳英 1
周全 104
周公 75
周六 130
周刊 1137
周到 50
周勃 0
周匝 5
周告 0
周四 47
周圍 1068
周圍環境 70
周報 263
周天 133
周家 42
周密 61
周年 257
周延 0
周恩來 316
周慧敏 134
周折 3
周數 6
周文 162
周文王 5
周旋 44
周旋到底 1
周日 124
周易 50
周書 19
周朝 29
周期 106
周期函數 0
周期律 0
周期性 14
周末 138
周樹人 1
周武王 5
周歲 27
周波 10
周波 10
周游 2
周濟 12
周率 1
周瑜 63
周相 9
周知 59
周禮 21
周章 3
周緣 5
周而復始 17
周而複始 0
周至 34
周薪 0
周角 1
周詳 73
周身 27
周轉 88
周轉不靈 12
周遊 70
周遊列國 13
周道 30
周遭 565
周邊 1680
周長 46
呩 0
呪 0
呫 0
呬 0
呭 0
呮 0
呯 0
呰 5
呱 69
呱 69
呱 69
呱呱 300
呱呱叫 7
呱呱叫 7
呱呱墜地 2
呱呱墜地 2
呲 0
呲 0
味 19031
味兒 35
味兒 35
味全 420
味全龍 94
味同嚼蠟 2
味噌 0
味噌湯 6
味塵 1
味方 792
味精 80
味素 22
味素包 0
味蕾 0
味覺 80
味覺器 0
味道 792
呴 131
呵 2547
呵令 0
呵凍 1
呵叱 1
呵呵 2958
呵呵大笑 2
呵喝 0
呵斥 2
呵欠 13
呵欠連天 0
呵氣 3
呵癢 2
呵護 159
呵責 1
呶 8
呷 94
呷呷 0
呸 166
呹 0
呺 9
呻 20
呻吟 109
呼 2248
呼之 10
呼之即來 1
呼之欲出 52
呼來喚去 0
呼出 34
呼出來 2
呼口號 10
呼叫 1048
呼叫器 907
呼吁 0
呼吸 1918
呼吸作用 39
呼吸器 151
呼吸器官 33
呼吸困難 151
呼吸系統 101
呼吸聲 24
呼吸道 609
呼呼 165
呼呼大睡 8
呼和浩特 12
呼和浩特市 1
呼哧 0
呼哧呼哧 1
呼喊 103
呼喚 285
呼嘯 46
呼嘯而過 20
呼嚕 31
呼圖克圖 1
呼天搶地 8
呼天搶地 8
呼應 236
呼應聲 0
呼拉圈 1
呼救 22
呼救信號 1
呼朋引類 0
呼氣 67
呼籲 2002
呼聲 211
呼聲甚高 1
呼蘭河流 0
呼蘭河流域 1
呼號 158
呼語 1
呼起 2
呼起來 0
呼風喚雨 27
命 5176
命不由人 0
命不該絕 3
命中 147
命中注定 24
命中率 68
命中註定 7
命令 1703
命令句 2
命作 8
命名 1267
命喪 37
命喪九泉 0
命在 23
命在旦夕 6
命大 8
命定 39
命宮 28
命根 4
命根子 8
命案 279
命為 13
命理 0
命相 61
命相學 1
命相家 0
命相館 103
命脈 101
命苦 19
命薄 3
命該 8
命該如此 6
命途 1
命途多舛 0
命運 1293
命運中 8
命題 408
命館 31
呾 157
呿 195
咀 38
咀咒 15
咀嚼 152
咀嚼出 0
咀嚼出 0
咁 91
咂 73
咂嘴 1
咃 0
咄 18
咄咄怪事 1
咄咄逼人 11
咅 0
咆 27
咆哮 52
咆哮山莊 17
咆哮起來 0
咆嘯 5
咇 5
咈 25
咉 0
咊 0
咊 0
咊 0
咊 0
咊 0
咊 0
咋 27
咋 27
咋舌 33
咋辦 0
和 1015
和 1015
和 10000
和 101537
和 2015
和了 12
和事佬 10
和事老 1
和加 76
和協 47
和南 83
和合 105
和和 40
和和氣氣 5
和善 71
和好 90
和好相處 1
和尚 621
和局 9
和州 12
和差化積 0
和帝 9
和平 2983
和平共存 28
和平共處 28
和平口號 1
和平國中 20
和平國小 46
和平島 70
和平攻勢 6
和平東路 719
和平溪 10
和平為處世之本 1
和平街 210
和平西路 245
和平談判 10
和平路 469
和平過渡 1
和平醫院 66
和平鴿 5
和弄 1
和式 70
和弦 88
和悅 9
和政 23
和數 67
和文 82
和易 22
和易近人 0
和春 283
和暖 14
和暢 1
和會 26
和服 44
和村 334
和棋 3
和樂 89
和氏璧 39
和氣 66
和氣生財 5
和氣致祥 1
和洽 1
和煦 26
和牌 7
和田 46
和盤托出 7
和睦 111
和睦相處 19
和穆 5
和約 80
和絃 17
和緩 64
和縣 26
和美 1291
和聲 92
和英字典 0
和英詞典 0
和菜 15
和著 51
和藹 38
和藹可親 39
和衣 1
和衣而睡 0
和衣而臥 0
和衷 0
和衷共濟 10
和親 32
和親政策 0
和解 317
和詩 19
和談 39
和諧 810
和諧一致 4
和諧無間 0
和議 11
和豐 45
和買 6
和靜 7
和順 37
和頭 12
和顏悅色 6
和風 232
和風徐徐 1
和風拂面 0
和風細雨 0
和鳴 8
和麵 9
咍 19
咎 130
咎由自取 11
咏 0
咐 26
咑 23
咒 411
咒文 81
咒符 0
咒罵 62
咒語 98
咓 0
咔 0
咕 82
咕咕 77
咕咚 1
咕咚一聲 0
咕咾肉 0
咕唧 0
咕嘎 82
咕嘟 3
咕噥 12
咕嚕 43
咕嚕一聲 0
咕嚕咕嚕 48
咕狗 1
咖 270
咖哩 107
咖哩 107
咖哩汁 0
咖哩汁 0
咖哩粉 8
咖哩粉 8
咖哩雞 13
咖哩雞 13
咖哩飯 11
咖哩飯 11
咖啡 3878
咖啡因 128
咖啡壺 38
咖啡室 6
咖啡屋 341
咖啡店 155
咖啡廳 185
咖啡杯 30
咖啡機 1
咖啡色 138
咖啡豆 151
咖啡館 470
咘 20
咚 20016
咚咚 90
咚咚聲 0
咞 0
咟 0
咠 0
咡 326
咢 5
咣 0
咣 0
咤 0
咤 0
咥 84
咥 84
咦 468
咧 957
咧 957
咧嘴 12
咧嘴一笑 3
咧嘴大笑 0
咧開 1
咧開嘴笑 0
咨 101
咨文 6
咨詢 243
咨議 3
咩 571
咩咩 67
咩咩叫 0
咪 568
咪咪 173
咪咪叫 1
咫 492
咫尺 41
咫尺天涯 8
咬 11906
咬下 4
咬了 21
咬了一口 10
咬了一口 10
咬人 97
咬住 40
咬住不放 2
咬入 1
咬到 29
咬合 1
咬好 1
咬字 18
咬字清晰 3
咬字清楚 0
咬定 4
咬得 8
咬得 8
咬文嚼字 11
咬文嚼字 11
咬牙 44
咬牙切齒 37
咬牙根 0
咬牙齒 1
咬的 28
咬碎 4
咬緊 15
咬緊牙根 8
咬緊牙關 14
咬耳朵 3
咬耳朵 3
咬菜根 0
咬著 83
咬著不放 0
咬起 2
咬起來 1
咬釘 1
咬釘嚼鐵 0
咬釘嚼鐵 0
咬點 0
咭 192
咭吱咯吱 0
咭咭 11
咭片 1
咮 2
咯 1
咯 297
咯 297
咯 297
咯 297
咯吱 0
咯吱 0
咯咯 26
咯咯 26
咯咯叫 0
咯咯叫 0
咯咯笑 1
咯咯笑 1
咯咯聲 0
咯咯聲 0
咯血 3
咯血 3
咰 4
咰 4
咱 569
咱 569
咱倆 7
咱們 864
咱家 72
咲 0
咳 43
咳 43
咳 439
咳 439
咳嗽 387
咳嗽藥 9
咳得 4
咳得 4
咳痰 11
咳聲歎氣 0
咳血 17
咳血 17
咴 0
咵 0
咶 4
咷 3
咸 283
咸信 16
咸宜 3
咸寧 18
咸認為 29
咸豐 73
咸豐草 1
咸陽 46
咹 0
咺 17
咻 318
咻 318
咼 3
咽 302
咽 302
咽 302
咽住 0
咽哽 0
咽喉 208
咽頭 22
咽頭炎 22
咾 15
咿 109
咿咿呀呀 1
咿咿唔唔 2
咿啞 2
咿啞學語 0
哀 471
哀傷 132
哀傷動人 0
哀傷感人 0
哀兵必勝 0
哀叫 19
哀告 4
哀咽 0
哀哉 11
哀喚 1
哀嘆 7
哀子 0
哀家 2
哀帝 4
哀弔 4
哀思 12
哀怨 55
哀怨動人 3
哀怨感人 0
哀悼 73
哀悼之意 3
哀惜 0
哀愁 105
哀意 0
哀慟 15
哀慟不已 2
哀憐 3
哀懇 1
哀戚 16
哀榮 1
哀樂 13
哀歌 28
哀求 66
哀求不已 0
哀泣 1
哀痛 25
哀痛不已 0
哀的美敦書 1
哀矜 8
哀矜勿喜 0
哀苦 0
哀莫大於心死 4
哀號 32
哀號不已 0
哀訴 1
哀豔 2
哀鳴 14
哀鳴不已 0
哀鴻 4
哀鴻遍野 13
哀鴻遍野 13
品 20237
品位 28
品名 857
品味 980
品嘗 119
品嚐 525
品學 16
品學兼優 54
品德 210
品德好 0
品性 41
品性不好 1
品性好 0
品格 88
品格高尚 2
品流 18
品牌 5895
品目 206
品種 1343
品種改良 38
品管 811
品管圈 24
品系 130
品紅 6
品級 31
品綠 1
品脫 7
品花寶鑒 0
品茗 87
品茶 26
品藍 0
品行 153
品行不良 2
品行優良 5
品行好 0
品評 33
品論 8
品貌 3
品貌出眾 0
品貌雙全 0
品質 12373
品質因數 0
品質管制 431
品鑑 17
品頭論足 10
品類 389
品類繁多 0
哂 29
哂然 0
哂笑 3
哃 4
哄 751
哄 751
哄亂 0
哄了 2
哄人 2
哄來哄去 0
哄傳 0
哄動 5
哄哄 811
哄堂大笑 9
哄得 2
哄抬 15
哄抬物價 1
哄笑 7
哄誘 0
哄走 0
哄騙 9
哅 5
哅 5
哆 27
哆嗦 11
哆囉哆嗦 0
哇 1
哇 3253
哇哇 85
哇哇叫 12
哇哇地 3
哇哇地 3
哇哇大叫 6
哇哇大哭 3
哇哇聲 0
哇啦 41
哇靠 49
哈 100
哈 714
哈 7141
哈佛 286
哈佛大學 326
哈克 72
哈利 164
哈叭狗 0
哈叭狗 0
哈哈 4626
哈哈大笑 92
哈哈笑 33
哈哈鏡 5
哈囉 643
哈士奇 1
哈密 50
哈密瓜 22
哈尼族 3
哈巴狗 33
哈巴狗 33
哈巴狗 33
哈巴狗 33
哈桑 3
哈欠 19
哈氣 0
哈爾濱 169
哈爾濱市 33
哈爾濱港 0
哈特 127
哈瓦那 8
哈維 43
哈腰 7
哈薩克 0
哈薩克人 1
哈達 25
哈里路亞 1
哈雷 79
哈雷彗星 24
哉 503
哊 0
哌 0
哌 0
哌 0
响 0
哎 479
哎呀 178
哎呀一聲 0
哎呀一聲 0
哎唷 16
哎唷一聲 0
哎唷一聲 0
哎喲 26
哎喲 100
哏 97
哐 0
哖 13
哚 0
哞 20
哠 9
員 0
員 10670
員外 125
員山 301
員工 7585
員工福利 105
員林 1680
員林市 3
員警 280
員額 580
哢 24
哣 0
哤 16
哥 4123
哥倫布 61
哥倫比亞 598
哥兒 15
哥兒 15
哥兒倆 1
哥兒倆 1
哥兒倆 1
哥兒倆 1
哥兒們 17
哥兒們 17
哥哥 1097
哥哥 1097
哥德 65
哥德人 2
哥德式 48
哥斯大黎加 0
哥斯達黎加 21
哥本哈根 62
哥白尼 19
哥羅仿 0
哥羅芳 0
哥老會 0
哥達綱領 0
哥達綱領批判 0
哦 414
哦 4147
哦呵 5
哦喲 0
哦喲 100
哧 20
哧 20
哨 142
哨兵 9
哨子 19
哨所 3
哨棒 0
哨音 15
哨音聲 0
哨音響 0
哩 1000
哩 1257
哩 1257
哩嚕 1
哪 25
哪 2556
哪 32190
哪些 5893
哪些人 206
哪件 79
哪件事 51
哪份 229
哪位 1281
哪來 78
哪個 5167
哪個人 335
哪個月 4
哪兒 905
哪兒 905
哪具 21
哪冊 0
哪匹 15
哪區 2
哪句 68
哪句話 60
哪台 105
哪回 14
哪圈 2
哪國 23
哪堪 3
哪壺 5
哪夜 33
哪天 1096
哪層 13
哪層樓 2
哪年 262
哪幾 52
哪幾個 67
哪幾天 58
哪幾次 0
哪怕 46
哪怕是 27
哪戶 3
哪批 19
哪支 39
哪方面 38
哪是 2425
哪時 1519
哪時候 481
哪會 153
哪有 485
哪期 3
哪本 91
哪本書 59
哪枝 7
哪棵 35
哪樣 1193
哪營 1
哪片 90
哪班 23
哪盞 11
哪科 15
哪種 2080
哪種人 45
哪站 5
哪筆 16
哪箱 0
哪篇 108
哪群 92
哪群人 16
哪能 53
哪般 100
哪行 20
哪裡 3578
哪課 0
哪起 10
哪輛 44
哪輛車 11
哪邊 3578
哪部 74
哪鍋 210
哪間 93
哪隻 393
哪集 15
哪頁 6
哪頂 11
哪項 35
哪顆 124
哪首 75
哪首歌 42
哫 73
哬 0
哭 1124
哭上 0
哭上來 0
哭上去 0
哭了 177
哭來 0
哭來哭去 1
哭個 11
哭個夠 1
哭個痛快 2
哭出 19
哭出來 18
哭去 1
哭叫 14
哭哭 26
哭哭啼啼 26
哭哭笑笑 0
哭喪 2
哭喪棒 1
哭喪臉 0
哭喪著臉 6
哭墓 4
哭得 53
哭成 8
哭成一團 3
哭泣 373
哭盡 0
哭笑不得 38
哭聲 68
哭腫 1
哭臉 2
哭著 81
哭號 9
哭訴 46
哭調 7
哭起 2
哭起來 17
哭過 21
哭醒 3
哭靈 1
哭髒 0
哭鬧 63
哭爸 1
哭夭 1
哮 103
哮 103
哮喘 227
哮喘病 1
哮喘病 1
哯 0
哰 0
哰 0
哱 60
哲 3880
哲人 31
哲嗣 2
哲學 3510
哲學家 195
哲學思想 72
哲學系 302
哲理 141
哳 29
哴 0
哵 0
哶 0
哶 0
哷 0
哸 1
哹 0
哺 190
哺乳 187
哺乳動物 158
哺乳類 185
哺育 81
哺育之恩 0
哻 9
哼 447
哼上 3
哼個 0
哼出 8
哼到 0
哼哈二將 1
哼哼 74
哼哼唧唧 0
哼哼歌 0
哼唱 23
哼得 0
哼得 0
哼歌 3
哼的 5
哼聲 17
哼著 22
哼起 6
哼起來 1
哽 22
哽住 4
哽咽 67
哽哽咽咽 2
哽嚥 0
哽在 5
哽塞 8
哽著 0
哾 0
哾 0
哿 596
哿 596
唀 0
唁 5
唁電 0
唂 0
唃 4
唄 44
唄 440
唄唱 1
唄讚 1
唅 38
唆 17
唆使 19
唆使者 1
唇 567
唇亡齒寒 15
唇槍舌劍 9
唇槍舌戰 8
唇瓣 8
唇疤疹 0
唇膏 34
唇舌 15
唇舌之爭 0
唇色 9
唇角 3
唇部 40
唇音 0
唇齒 7
唇齒之邦 0
唇齒相依 13
唈 9
唉 2340
唉 2340
唉唉 45
唉唉 45
唉唉叫 1
唉聲嘆氣 8
唉聲歎氣 0
唊 7
唋 2
唌 158
唍 0
唎 12
唏 17
唏噓 11
唐 4186
唐三藏 20
唐中 5
唐人 54
唐人街 32
唐代 210
唐代人 0
唐伯虎 17
唐僧 6
唐兵 0
唐努山 0
唐努烏梁海 1
唐吉訶德 6
唐太宗 21
唐宋八大家 11
唐尼 9
唐山 153
唐山大兄 3
唐律 7
唐才常 0
唐明皇 16
唐朝 415
唐朝人 0
唐氏症 1
唐玄宗 11
唐王 4
唐甄 0
唐突 35
唐紹儀 0
唐繼堯 4
唐老鴨 24
唐虞 1
唐裝 1
唐詩 172
唐詩三百首 807
唐賽兒 4
唐高祖 1
唑 0
唒 0
唓 0
唔 421
唕 0
唖 0
唖 0
唖 0
唗 0
唗 0
唘 0
唙 0
唚 0
唦 5
唧 30
唧 30
唧咕 100
唧唧 17
唧唧咕咕 0
唧拉喳拉 0
唧筒 17
唩 0
唪 0
唪經 0
唫 0
唬 132
唬人 18
唬住 5
唬來唬去 0
唬弄 1
唬得 6
唬爛 1
唭 0
售 5475
售價 3480
售出 140
售完 117
售後服務 1
售票 1047
售票亭 6
售票口 14
售票員 9
售票處 16
售罄 22
售貨 27
售貨員 27
售賣 17
唯 6890
唯一 3737
唯你 3
唯你是問 1
唯利是圖 8
唯力是視 1
唯命是從 5
唯命是聽 0
唯唯 2
唯唯否否 0
唯唯諾諾 8
唯妙 1
唯妙唯肖 10
唯心 39
唯心主義 6
唯心史觀 0
唯心論 9
唯恐 114
唯恐天下不亂 7
唯我獨尊 22
唯有 1305
唯有讀書高 2
唯物 24
唯物主義 8
唯物史觀 5
唯物論 22
唯物辨証 0
唯物辨証法 0
唯物辯証 0
唯物辯証法 0
唯物辯證法 1
唯獨 175
唯生產力論 0
唯美 111
唯美主義 13
唯美派 1
唯肖 2
唯讀 50
唰 24
唱 2292
唱下 9
唱了 51
唱來唱去 0
唱個 9
唱出 136
唱出來 6
唱到 70
唱功 13
唱友 0
唱反 4
唱反調 23
唱名 20
唱和 28
唱唱 41
唱回 0
唱回來 0
唱回去 0
唱對 0
唱對台戲 1
唱得 65
唱戲 15
唱本 3
唱機 97
唱歌 904
唱片 5456
唱片目錄 2
唱獨角戲 18
唱獨角戲 18
唱的 309
唱盤 103
唱破 0
唱票 3
唱票員 2
唱籌量沙 0
唱腔 178
唱著 131
唱詞 18
唱詩 83
唱詩班 8
唱起 54
唱起來 8
唱遊 132
唱酬 3
唱針 3
唱高 9
唱高調 5
唱點 0
唲 0
唳 0
唴 0
唵 0
唶 0
唷 1934
唸 3724
唸一些 7
唸一些 7
唸一唸 2
唸下去 1
唸了 42
唸佛 15
唸作 6
唸來唸去 0
唸到 15
唸咒 17
唸唸 19
唸唸有詞 18
唸好 6
唸完 50
唸得 11
唸得 11
唸成 6
唸書 371
唸書下去 0
唸法 28
唸給 7
唸經 26
唸著 44
唸起來 21
唸過 24
唸錯 12
唸點 7
唹 0
唺 0
唻 0
唼 0
唼 0
唼 0
唼唼 0
唼氣 0
唼眼 0
唼眼 0
唼血 0
唼血 0
唽 0
唾 80
唾手可得 20
唾棄 116
唾沫 9
唾液 82
唾液 82
唾液腺 17
唾液腺 17
唾罵 8
唾腺 4
唾面自乾 0
唿 0
唿 0
啀 0
啁 0
啁 0
啁啾 9
啂 0
啃 321
啃上 0
啃下 2
啃來啃去 0
啃出 0
啃到 1
啃去 0
啃得 2
啃掉 0
啃書 3
啃書本 1
啃書蟲 0
啃著 7
啃著 7
啃蝕 10
啃起 0
啃起來 0
啃進 0
啃過 2
啃過來 0
啃過去 0
啄 267
啄傷 3
啄木鳥 83
啄食 22
啅 0
商 19598
商丘 12
商丘縣 0
商事法 126
商人 454
商人資本 0
商代 65
商令 0
商務 3258
商務代表 36
商務印書館 58
商務旅行 7
商務會議 8
商務考察 9
商務考察團 0
商務艙 1
商務辦事 1
商務辦事處 57
商務部 129
商務部長 21
商務院 0
商君 1
商君書 4
商周 18
商品 9043
商品價格 48
商品化 249
商品展覽 4
商品展覽會 2
商品標示 22
商品流轉 0
商品流通 13
商品流通費 0
商品生產 5
商品盤損 9
商品盤盈 0
商品目錄 19
商品糧 5
商品經濟 9
商品資本 3
商品資金 0
商品輸出 1
商圈 507
商圖 24
商圖國事 0
商團 8
商埠 5
商場 133
商場 1335
商場如戰場 6
商場如戰場 6
商場如戰場 6
商學系 50
商學院 288
商定 64
商客 0
商家 431
商專 1046
商展 215
商州 7
商工 2
商工職業學校 113
商店 2540
商情 1914
商擬 16
商數 64
商旅 130
商會 486
商朝 20
商朝人 0
商業 18144
商業中心 55
商業公司 16
商業利潤 4
商業化 206
商業區 239
商業專科學校 636
商業機構 99
商業港 0
商業登記 93
商業網 268
商業職業學校 162
商業資本 6
商業部 25
商業銀行 1885
商榷 47
商標 2114
商標權 57
商標法 144
商機 0
商權 3
商法 52
商洛 0
商洽 13
商港 246
商湯 7
商王 14
商用 5681
商用品 0
商界 108
商社 55
商科 336
商約 6
商職 85
商船 141
商船海事職業學校 0
商船隊 0
商號 558
商行 1115
商訂 32
商討 129
商討對策 0
商談 188
商談對策 0
商請 121
商謎 0
商議 539
商譽 56
商販 10
商賈 12
商路 34
商輪 6
商酌 4
商量 186
商量對策 1
商銀 571
商隊 11
商鞅 15
商高定理 0
啇 0
啇 0
啈 0
啉 0
啊 1125
啊 21187
啊呀 115
啊哈 32
啊唷 0
啋 0
啌 0
啌 0
啍 0
啎 0
問 15158
問上 4
問上問下 0
問世 403
問了 221
問人 15
問住 2
問來 5
問來問去 1
問供 1
問俗 1
問個 84
問個問題 16
問倒 10
問候 429
問出 43
問到 224
問卜 10
問卷 1032
問卷調查 1214
問卷調查表 47
問及 107
問問 262
問問看 62
問好 136
問安 50
問安者 0
問得 27
問得 27
問心 3
問心有愧 0
問心無愧 23
問政 314
問政於民 2
問斬 2
問朝 0
問東問西 2
問案 4
問津 8
問的 496
問答 2756
問答式 18
問答法 12
問答題 144
問罪 7
問荊 0
問號 115
問訊 13
問話 61
問起 95
問起來 2
問路 54
問過 167
問道 453
問道於盲 0
問長問短 2
問長道短 0
問難 3
問題 64333
問題人物 2
問題兒童 11
問題學生 12
問題家庭 16
問題少年 3
問題是 831
問題重重 20
問鼎 78
啐 1
啐 33
啑 3
啒 0
啓 0
啔 0
啕 10
啖 212
啗 0
啘 0
啙 0
啚 0
啚 0
啛 0
啜 211
啜泣 23
啜飲 34
啝 0
啞 606
啞 606
啞劇 2
啞口無言 14
啞吧 13
啞啞 7
啞啞 7
啞子 5
啞巴 30
啞巴吃黃蓮 5
啞然 9
啞然失笑 15
啞謎 2
啞鈴 29
啟 6440
啟事 1890
啟人疑竇 1
啟動 2023
啟口 8
啟奏 4
啟始 75
啟封 7
啟德機場 42
啟德機場 42
啟明 228
啟明學校 91
啟智 0
啟智班 158
啟用 1167
啟發 603
啟發式 43
啟發式教育 1
啟示 1164
啟示錄 0
啟程 208
啟稟 5
啟聰 315
啟聰學校 180
啟航 1041
啟蒙 237
啟蒙運動 18
啟迪 86
啟運 13
啟閉 18
啟開 20
啟齒 13
啠 0
啡 298
啢 0
啣 113
啣接 0
啤 95
啤酒 926
啤酒廠 28
啤酒瓶 1
啤酒罐 3
啤酒肚 1
啥 1527
啥事 97
啥子 10
啦 9325
啦 13707
啦啦隊 159
啦啦隊員 0
啦啦隊長 6
啦瘌 0
啨 0
啨 0
啪 338
啪 338
啪啦 36
啪啪 120
啪啪 120
啪噠聲 0
啪噠聲 0
啪達 0
啵 0
啵 62
啶 0
啷 13
啷鐺 0
啺 0
啻 55
啻 55
啼 108
啼叫 11
啼哭 31
啼啼哭哭 0
啼明 0
啼笑 8
啼笑皆非 32
啼聲 11
啽 0
啾 40
啾叫 0
啾唧 0
啾啾 26
啾嗽 0
啿 0
喀 37
喀 37
喀 373
喀什米爾 34
喀吧 0
喀嚓 9
喀嚓一聲 0
喀嚓一聲 0
喀土穆 1
喀布爾 1
喀拉蚩 16
喀斯特 2
喀斯特水 0
喀琅施塔得 0
喀麥隆 57
喁 0
喂 81
喂喂 101
喃 17
喃喃 109
喃喃地 18
喃喃地 18
喃喃自語 53
善 5014
善事 74
善人 43
善加 176
善化 423
善友 1
善哉 38
善坐 1
善堂 28
善報 10
善士 14
善始善終 1
善導寺 24
善待 101
善後 202
善後事宜 24
善心 78
善心人士 203
善忘 0
善思 1
善性 14
善惡 176
善惡不分 1
善意 346
善意回應 37
善戰 8
善才 3
善文 4
善於 268
善於詞令 0
善於辭令 0
善書 10
善有善報 12
善本 51
善本書 30
善根 31
善為 8
善為保管 0
善用 1018
善男信女 22
善知識 1
善終 33
善罷干休 6
善罷甘休 2
善自保重 1
善舉 12
善良 499
善行 51
善言 18
善變 67
善辯 4
善類 9
喅 0
喆 100
喇 224
喇 224
喇 224
喇叭 1591
喇叭口 5
喇叭手 5
喇叭筒 0
喇叭管 0
喇叭花 9
喇叭褲 13
喇嘛 75
喇嘛廟 4
喇嘛教 4
喇嘛教徒 0
喈 0
喈 0
喉 384
喉嚨 222
喉嚨痛 54
喉炎 14
喉片 2
喉癌 1
喉科 18
喉管 10
喉糖 1
喉結 8
喉舌 9
喉部 56
喉音 1
喉頭 66
喉頭炎 21
喉骨 1
喊 641
喊上 1
喊上來 0
喊上去 0
喊下 2
喊下來 0
喊下去 0
喊了 36
喊住 2
喊來 3
喊來喊去 0
喊停 58
喊價 44
喊冤 30
喊出 117
喊出來 1
喊出去 0
喊到 14
喊叫 41
喊叫聲 3
喊嚷 0
喊回 0
喊回來 0
喊回去 0
喊堂威 0
喊好 1
喊得 13
喊成 0
喊救 10
喊破 3
喊窮 1
喊著 94
喊話 220
喊起 5
喊起來 2
喊進 0
喊進來 0
喊進去 0
喊過 2
喊醒 0
喋 15
喋喋 4
喋喋不休 26
喋血 28
喋血記 1
喌 0
喍 0
喎 0
喏 25
喐 0
喑 1
喑啞 30
喒 8
喓 0
喔 1
喔 1
喔 13867
喔唷 0
喔喔 39
喔喔 39
喔喔叫 0
喔喔叫 0
喔嚄 0
喕 0
喖 0
喗 0
喘 378
喘出 0
喘喘 7
喘喘氣 1
喘噓噓 4
喘息 129
喘息未定 0
喘氣 47
喙 75
喚 427
喚作 12
喚來 4
喚來喚去 0
喚出 9
喚名 0
喚回 26
喚起 318
喚醒 240
喛 0
喜 4072
喜上眉梢 3
喜不自勝 5
喜事 103
喜人 3
喜信 10
喜出望外 24
喜劇 362
喜劇片 25
喜功 1
喜報 0
喜好 636
喜娘 2
喜子 180
喜孜孜 18
喜宴 107
喜容 1
喜帖 91
喜帳 4
喜幛 3
喜形於色 3
喜從何來 0
喜從天降 4
喜怒 18
喜怒不形於色 7
喜怒哀樂 85
喜怒無常 6
喜悅 848
喜悅之情 6
喜惡 10
喜愛 1911
喜愛著 1
喜感 16
喜慶 69
喜慶宴會 9
喜憂參半 1
喜新厭舊 18
喜極而泣 17
喜樂 461
喜歡 11487
喜歡著 9
喜氣 29
喜氣洋洋 19
喜洋洋 36
喜瑪拉雅 46
喜笑頻開 0
喜笑顏開 1
喜筵 0
喜聞樂見 1
喜色 12
喜訊 51
喜訊傳來 0
喜躍 2
喜酒 25
喜金剛 1
喜餅 1
喜馬拉雅 44
喜馬拉雅山 28
喜鵲 31
喝 221
喝 2215
喝上 13
喝下 58
喝下去 14
喝了 220
喝令 18
喝住 4
喝來喝去 1
喝個 36
喝倒彩 2
喝入 4
喝出 7
喝出 7
喝到 49
喝口 7
喝口水 5
喝喝 59
喝夠 2
喝完 79
喝彩 9
喝彩聲 1
喝得 30
喝得 30
喝掉 10
喝杯 117
喝水 260
喝湯 26
喝的 136
喝盡 2
喝茶 203
喝著 59
喝著 59
喝西北風 2
喝西北風兒 0
喝西北風兒 0
喝起 9
喝起來 15
喝進 4
喝進 4
喝進去 0
喝進去 0
喝過 43
喝道 128
喝酒 516
喝醉 115
喝采 156
喝風 1
喞 0
喞 1
喟 54
喟然 3
喟然長嘆 0
喠 0
喡 0
喢 0
喣 18
喤 0
喥 0
喦 0
喧 121
喧嘩 107
喧嘩聲 1
喧嚷 18
喧嚷一時 0
喧嚷一時 0
喧嚷聲 0
喧囂 64
喧囂聲 0
喧天 5
喧擾 9
喧擾已久 0
喧譁 15
喧賓奪主 22
喧騰 13
喧騰一時 22
喧騰一時 22
喧騰已久 5
喧騰聲 0
喧騷 0
喧鬧 66
喧鬧不休 1
喨 0
喩 0
喩 0
喪 474
喪 474
喪主 0
喪主 0
喪亂 3
喪事 39
喪假 29
喪偶 44
喪偶之痛 1
喪命 78
喪夫 5
喪夫之痛 0
喪失 870
喪妻 5
喪妻之痛 4
喪家 29
喪家之犬 2
喪家狗 0
喪居 0
喪心病狂 10
喪志 30
喪服 9
喪權辱國 10
喪母 5
喪母之痛 0
喪氣 11
喪氣話 1
喪父 14
喪父之痛 0
喪生 168
喪盡天良 4
喪禮 77
喪膽 14
喪葬 128
喪葬費 11
喪身 4
喪鐘 10
喪鐘 10
喫 13
喫官司 0
喫肉 0
喫苦 0
喫茶 0
喫飯 0
喫飯防噎 0
喫香 0
喫驚 0
喬 2141
喬妝 0
喬木 152
喬治 639
喬治亞 210
喬治亞州 32
喬治史考特 4
喬治敦 0
喬治華盛頓 21
喬裝 24
喬遷 72
喬遷之喜 4
喭 0
單 16
單 16995
單 208
單一 2199
單一化 17
單一性 41
單一體 1
單丁 6
單于 13
單交 10
單交種 1
單人 250
單人床 23
單人房 182
單人舞 0
單位 29444
單位制 1
單位工程 4
單位成本 36
單位根 2
單位面積 61
單位面積產量 5
單作 1
單倍體 3
單傳 30
單價 1158
單元 4500
單元體 8
單兵 29
單兵攻擊 1
單刀 16
單刀直入 16
單刀赴會 5
單利 5
單句 8
單名 7
單名數 0
單向 252
單向行駛 1
單向通車 2
單向閥 4
單單 189
單單在 9
單單是 10
單單會 0
單單有 0
單單能 0
單在 22
單子 55
單子 55
單子葉植物 90
單孔 37
單字 507
單字表 0
單季 43
單寧 40
單寧酸 1
單層 267
單層式 0
單峰駝 0
單幅 3
單座 11
單座式 0
單從 32
單循環 8
單性 19
單性花 1
單憑 45
單手 61
單手操作 5
單打 105
單打獨鬥 49
單指 22
單挑 71
單排 33
單掛號 5
單擊 0
單據 169
單擺 26
單數 27
單數形 1
單斜晶硫 0
單斜晶硫 0
單方 67
單方面 76
單於 14
單日 82
單日份 0
單是 111
單晶體 0
單曲 524
單曲榜 27
單月 260
單期 6
單株選擇 0
單槍 492
單槍匹馬 25
單槓 16
單槽 92
單槽式 2
單機 347
單母音 1
單片 75
單獨 1048
單獨到 3
單獨在 14
單產 2
單相 121
單相思 7
單眼 116
單眼皮 12
單程 66
單程票 6
單簧管 25
單糖 4
單系 27
單純 1442
單純化 48
單細胞 20
單絲 4
單線 59
單線行駛 0
單線通車 11
單翼 5
單職 0
單胞藻 1
單腳 29
單色 184
單色光 4
單薄 47
單號 55
單行 48
單行本 65
單行法 3
單行道 25
單衣 3
單親 0
單詞 20
單說 8
單調 160
單調乏味 6
單調無味 1
單質 17
單身 523
單身在外 0
單身漢 22
單身貴族 67
單車 558
單車比賽 0
單車競賽 1
單軌 22
單軌鐵路 1
單軌鐵道 3
單輛 0
單選 51
單選題 14
單邊 44
單醣 5
單鍵 65
單門 7
單門式 1
單門獨戶 0
單隻 7
單靠 79
單靠著 1
單面 135
單音 32
單音字 0
單音節 9
單音詞 1
單項 218
單項工程 1
單項式 3
單頻 0
單飛 198
單養 1
單體 301
單點 1
喯 0
喯 0
喯 0
喱 10
喲 841
喳 99
喳 99
喳 99
喳喳 10
喳喳叫 1
喴 0
喵 1199
営 0
喻 440
喻為 75
喻詞 0
喿 0
嗀 0
嗁 0
嗂 0
嗃 0
嗄 0
嗄 0
嗅 8600
嗅出 34
嗅到 58
嗅神經 2
嗅覺 129
嗅覺作用 1
嗅鹽 0
嗆 243
嗆 243
嗆了 1
嗆到 19
嗆得 4
嗆的 9
嗆起來 0
嗆鼻 0
嗇 25
嗈 0
嗉 17
嗉囊 6
嗊 0
嗋 0
嗌 0
嗍 2
嗎 0
嗎 2643
嗎 26437
嗎啡 113
嗏 0
嗐 0
嗑 20000
嗑 1
嗑牙 7
嗑藥 21
嗒 0
嗓 37
嗓子 45
嗓聲 0
嗓門 26
嗓門兒 1
嗓門兒 1
嗓音 82
嗔 287
嗔喝 1
嗔忿 1
嗔怒 5
嗔怪 1
嗔拳不打笑面 1
嗔睨 1
嗔色 1
嗔詬 1
嗔責 1
嗔道 1
嗕 0
嗕 0
嗖 19
嗗 0
嗗 0
嗘 0
嗙 0
嗚 1079
嗚兒嗚兒 0
嗚兒嗚兒 0
嗚呼 14
嗚呼哀哉 8
嗚咽 37
嗚唈 0
嗚啦 5
嗚嗚 321
嗚嗚咽咽 1
嗚嗚的 4
嗛 55
嗛 55
嗛 55
嗜 52404
嗜好 796
嗜欲 3
嗜眠症 0
嗜酒如命 1
嗝 70
嗝兒 0
嗝兒 0
嗟 59
嗟 59
嗟 59
嗟來之食 0
嗟來食 2
嗟嘆 5
嗟歎 3
嗠 0
嗡 71
嗡嗡 30
嗡嗡作響 4
嗡嗡叫 2
嗡嗡地 0
嗡嗡地 0
嗡嗡聲 13
嗢 0
嗣 1456
嗣后 0
嗣後 158
嗣承 0
嗣業 1
嗤 99
嗤之以鼻 25
嗤笑 6
嗥 22
嗥叫 1
嗦 24
嗨 1178
嗨 1178
嗨嗨 47
嗩 77
嗩吶 40
嗪 0
嗬 0
嗯 5084
嗯嗯 88
嗲 23
嗲地 0
嗲聲嗲氣 0
嗶 278
嗶哩啪啦 0
嗶嗶扣 2
嗶嗶聲 12
嗶嘰 0
嗷 10
嗷嗷 2
嗷嗷待哺 150
嗸 0
嗹 0
嗺 0
嗻 0
嗻 0
嗻 0
嗼 0
嗽 134
嗽血 0
嗾 1
嗾 1
嗾使 0
嗿 0
嘀 118
嘀咕 24
嘀嘀咕咕 3
嘁 0
嘁 0
嘁嘁喳喳 0
嘁嘁沙沙 0
嘂 0
嘃 0
嘄 0
嘅 0
嘅 0
嘆 1035
嘆息 203
嘆服 6
嘆氣 42
嘆為觀止 86
嘆聲 2
嘆賞 1
嘇 0
嘇 0
嘇 0
嘈 201
嘈嚷 0
嘈雜 34
嘈雜聲 9
嘉 37000
嘉中 21
嘉勉 57
嘉南 873
嘉南大圳 28
嘉南藥專 26
嘉定 14
嘉尚 2
嘉峪關 3
嘉年華 1
嘉年華會 181
嘉惠 231
嘉慶 166
嘉獎 66
嘉獎令 0
嘉祐 9
嘉禮 2
嘉禾 83
嘉納 7
嘉義 5575
嘉義人 58
嘉義市 4824
嘉義師範 4
嘉義師範學院 212
嘉義縣 5911
嘉義農專 57
嘉興 159
嘉興府 1
嘉興街 30
嘉言 33
嘉言錄 8
嘉許 38
嘉評 2
嘉賓 136
嘉釀 0
嘉陵江 11
嘉陽 22
嘉靖 34
嘉香肉 0
嘊 0
嘊 0
嘋 0
嘌 1
嘌 1
嘌呤 1
嘌唱 1
嘍 314
嘍嘍 10
嘍囉 11
嘎 294
嘎叭 0
嘎吱 6
嘎啦 6
嘎嘎 30
嘎嘎聲 0
嘎嘎響 1
嘏 0
嘐 0
嘑 0
嘒 0
嘓 1
嘓嘓 1
嘔 1
嘔 584
嘔 584
嘔 584
嘔吐 371
嘔吐時 2
嘔吐物 18
嘔咐 1
嘔唲 1
嘔啞 2
嘔啞 2
嘔喻 1
嘔嘔 1
嘔嘔啞啞 1
嘔心 45
嘔心吐膽 1
嘔心瀝血 1
嘔心瀝血 30
嘔心瀝血之作 14
嘔心絞腦 1
嘔心血 1
嘔氣 8
嘔笑 1
嘔血 5
嘔血 5
嘔血時 0
嘔逆 1
嘔鴉 1
嘕 0
嘖 288
嘖嘖 13
嘖嘖稱奇 23
嘖嘖稱羨 0
嘖有煩言 0
嘗 573
嘗了 4
嘗來嘗去 0
嘗個 0
嘗出 0
嘗到 50
嘗嘗 39
嘗甜頭 0
嘗膽 1
嘗試 1705
嘗試性 9
嘗點 0
嘘 0
嘙 0
嘛 1
嘛 2759
嘛 26436
嘜 0
嘜 0
嘜頭 15
嘝 0
嘞 0
嘞 0
嘟 461
嘟嘟 310
嘟嘟囔囔 0
嘟嘟聲 2
嘟嘟響 2
嘟嚕 2
嘟嚷 2
嘟囔 1
嘦 0
嘧 0
嘨 0
嘩 58
嘩 58
嘩啦 39
嘩啦一聲 2
嘩啦啦 14
嘩啦聲 0
嘩喇 1
嘩喇喇 0
嘩嘩 12
嘩嘩啦 0
嘩嘩啦啦 2
嘩嘩的 0
嘩然 14
嘩的一聲 1
嘩變 0
嘪 0
嘫 0
嘬 0
嘭 0
嘮 1
嘮叨 39
嘮嘮叨叨 11
嘯 349
嘯傲江湖 5
嘯聲 11
嘯詠 0
嘯鬧 0
嘰 1
嘰咕 10
嘰哩咕嚕 8
嘰喳不斷 0
嘰嘰咕咕 0
嘰嘰喳喳 29
嘰嘰嘎嘎 1
嘱 0
嘲 90
嘲 90
嘲哳 1
嘲弄 48
嘲笑 130
嘲笑戲弄 0
嘲罵 1
嘲訕 0
嘲諷 100
嘳 0
嘴 1356
嘴上 67
嘴上無毛 2
嘴唇 203
嘴巴 621
嘴快 0
嘴渴 0
嘴甜 5
嘴甜心苦 0
嘴硬 8
嘴碎 8
嘴笨 0
嘴脣 7
嘴臉 48
嘴裡 127
嘴角 163
嘴邊 65
嘴部 9
嘴饞 7
嘵 0
嘵嘵 1
嘶 67
嘶叫 1
嘶吼 16
嘶啞 18
嘶喊 16
嘶鳴 9
嘷 0
嘷 0
嘸 180
嘸 187
嘸蝦米 0
嘸蝦米 1
嘹 10
嘹亮 36
嘹喨 1
嘺 4
嘻 551
嘻嘻 420
嘻嘻哈哈 33
嘻皮笑臉 10
嘻笑 22
嘻笑怒罵 7
嘼 0
嘼 0
嘼 0
嘼 0
嘽 0
嘾 0
嘿 0
嘿 1021
嘿嘿 832
噀 0
噁 66
噁 66
噁心 318
噂 0
噃 0
噅 0
噅 0
噆 0
噇 0
噈 0
噉 0
噊 0
噋 0
噌 0
噌 0
噌 0
噍 0
噍 0
噎 30
噎住 5
噎到 3
噎著 0
噏 0
噐 0
噑 0
噑 0
噒 0
噓 126
噓下台 0
噓下台來 0
噓下台去 0
噓唏 0
噓唏不已 1
噓寒問暖 15
噓尿 0
噓氣 0
噓聲 14
噓聲四起 1
噓開 0
噔 0
噕 0
噖 0
噗 1286
噗哧 7
噗哧 7
噗嗤 5
噗嗤一聲 1
噗嗤一聲 1
噗噗 52
噗噗跳 2
噗通 10
噗通一聲 3
噗通一聲 3
噘 10
噘嘴 3
噘嘴兒 0
噘嘴兒 0
噙 137
噚 0
噞 0
噟 0
噟 0
噠 1
噠噠 100
噡 0
噢 0
噢 0
噣 0
噤 56
噤口 2
噤聲 8
噤若寒蟬 12
噥 76
噥噥 7
噦 0
噧 0
器 21702
器件 27
器使 101
器具 878
器宇 2
器宇不凡 1
器宇軒昂 0
器官 971
器度 83
器材 5715
器材店 9
器材行 193
器械 573
器械體操 0
器樂 78
器物 141
器物錄 0
器用 100
器皿 200
器識 8
器重 28
器量 16
器量 87
噩 1
噩兆 1
噩夢 1
噩耗 1
噩耗傳來 0
噩運 1
噩魚 0
噪 163
噪嚷 0
噪聒 0
噪聲 4
噪音 1023
噪音控制 31
噪音管制 155
噪音量 17
噫 190
噬 144
噬箘體 0
噬菌體 18
噬食弧菌 0
噬食菌 0
噭 1
噮 1
噯 1
噯呀 0
噯呀 0
噰 0
噱 20
噱 20
噱頭 54
噲 0
噳 0
噴 1184
噴上 29
噴上來 1
噴上去 0
噴下 1
噴下來 0
噴下去 0
噴入 17
噴出 139
噴出來 5
噴出去 0
噴口 2
噴吐 1
噴嘴 96
噴嘴式 0
噴噴 3
噴噴水 0
噴嚏 20
噴在 18
噴墨 736
噴墨印表機 1
噴墨式 43
噴壺 0
噴射 621
噴射器 15
噴射客機 27
噴射式 57
噴射機 69
噴射泵 5
噴槍 61
噴氣 23
噴氣孔 4
噴氣式 7
噴氣式飛機 0
噴氣機 0
噴氣發動 0
噴氣發動機 0
噴水 112
噴水池 51
噴油器 3
噴泉 85
噴洒 5
噴漆 142
噴濺 9
噴灌 9
噴灑 147
噴灑在 11
噴灑著 0
噴火 31
噴火口 2
噴火器 6
噴火山 0
噴火龍 2
噴發 31
噴管 4
噴粉器 1
噴絲頭 2
噴花 1
噴著 4
噴著氣 0
噴著水 0
噴血 11
噴起 1
噴起來 0
噴鍍 0
噴雲吐霧 0
噴霧 230
噴霧器 75
噴霧式 25
噴頭 85
噴飯 13
噴香 21
噴鼻兒香 0
噴鼻兒香 0
噵 0
噶 1
噷 0
噸 1785
噸位 44
噸公里 28
噹 100
噹 1
噹啷 0
噹噹 61
噻 0
噽 0
噾 0
噿 0
嚀 0
嚁 0
嚂 0
嚃 0
嚄 44
嚄 44
嚅 25
嚆 0
嚇 39
嚇 393
嚇一跳 66
嚇一跳 66
嚇不了 0
嚇了 52
嚇了一跳 98
嚇了一跳 98
嚇人 141
嚇住 10
嚇來嚇去 0
嚇倒 7
嚇出 5
嚇出病來 0
嚇到 106
嚇哭 3
嚇唬 15
嚇壞 69
嚇得 177
嚇怕 0
嚇成 0
嚇死 77
嚇死人 62
嚇的 37
嚇破 2
嚇破膽 4
嚇老 0
嚇走 21
嚇跑 48
嚇退 18
嚇過 1
嚇阻 117
嚇阻力量 2
嚈 0
嚉 0
嚊 0
嚊 0
嚋 0
嚋 0
嚌 0
嚌嚌嘈嘈 0
嚍 0
嚎 76
嚎叫 6
嚎咷 0
嚎哭 3
嚎啕 7
嚎啕大哭 19
嚏 363
嚐 425
嚐了 1
嚐來嚐去 0
嚐個 1
嚐出 1
嚐到 73
嚐嚐 50
嚐試 180
嚐點 1
嚓 1
嚔 0
嚕 351
嚕囌 0
嚖 0
嚗 0
嚘 0
嚘然而止 0
嚙 55
嚙合 14
嚙齒類 1
嚚 0
嚛 0
嚛 0
嚜 0
嚝 0
嚞 0
嚟 0
嚠 0
嚥 1
嚥下 23
嚥下去 0
嚥不下 0
嚥不下這口氣 3
嚥喉 0
嚥氣 1
嚥頭 0
嚦 1
嚧 0
嚨 0
嚩 0
嚪 0
嚫 0
嚬 0
嚭 0
嚮 1
嚮導 172
嚮導公司 0
嚮導員 21
嚮往 307
嚮應 84
嚮日 0
嚮邇 0
嚯 0
嚯 0
嚯 0
嚱 0
嚲 0
嚳 0
嚴 11100
嚴令 8
嚴以律己 6
嚴冬 48
嚴刑 10
嚴刑峻法 7
嚴刑峻罰 4
嚴刑拷打 4
嚴加 61
嚴加戒備 9
嚴加管教 1
嚴加防範 2
嚴厲 269
嚴命 0
嚴嚴 4
嚴嚴密密 0
嚴嚴格格 0
嚴守 88
嚴守中立 2
嚴家淦 13
嚴密 319
嚴密性 2
嚴寒 44
嚴寒季節 0
嚴察 0
嚴實 0
嚴峻 67
嚴嵩 17
嚴師 9
嚴律 0
嚴復 16
嚴慶齡 7
嚴懲 109
嚴整 17
嚴明 36
嚴查 29
嚴格 2777
嚴格主義 1
嚴格來說 32
嚴格說來 45
嚴正 216
嚴法 7
嚴法峻刑 0
嚴父 7
嚴父慈母 0
嚴禁 339
嚴禁吸煙 5
嚴禁煙火 8
嚴絲合縫 0
嚴緊 5
嚴肅 452
嚴苛 152
嚴詞 76
嚴謹 466
嚴謹性 6
嚴責 2
嚴辦 25
嚴酷 50
嚴酷性 0
嚴重 5593
嚴重事故 11
嚴重性 168
嚴防 109
嚴防豪雨 5
嚴陣以待 22
嚴霜 1
嚵 0
嚶 1
嚶嚶 8
嚷 1
嚷 1
嚷出 0
嚷出來 0
嚷出去 0
嚷嚷 1
嚷嚷 26
嚷起來 0
嚷鬧 2
嚻 0
嚻 0
嚼 209
嚼 209
嚼 209
嚼不了 0
嚼勁 1
嚼舌 0
嚼舌根 3
嚼舌自盡 0
嚼舌頭 0
嚼蠟 0
嚽 0
嚾 0
嚾呼 0
嚿 0
囀 0
囁 0
囂 63
囂張 76
囂浮 0
囃 0
囅 0
囆 0
囇 0
囈 1
囈語 14
囉 4188
囉 1
囉哩囉唆 2
囉唆 39
囉嘍 12
囉囉唆唆 0
囉囉嗦嗦 1
囊 596
囊中 21
囊中物 3
囊括 132
囊空如洗 0
囊胚 29
囊腫 161
囋 0
囌 0
囍 395
囏 0
囐 0
囐 0
囐 0
囐 0
囑 159
囑令 2
囑咐 58
囑托 1
囑目 23
囑託 86
囒 0
囓 163
囓合 4
囓齒類 11
囔 1
囗 0
囘 0
囙 0
囚 122
囚人 2
囚居 0
囚徒 9
囚房 7
囚牢 8
囚犯 61
囚禁 57
囚籠 7
囚糧 0
囚衣 1
囚車 4
囚首垢面 0
四 43282
四丁 9
四上 69
四下 122
四不像 24
四世 35
四世紀 10
四串 0
四乙 106
四五運動 0
四人 778
四人份 2
四人幫 16
四仁 4
四代 172
四代同堂 1
四任 63
四份 164
四位 532
四位數 103
四個 2087
四個人 198
四個月 402
四個現代化 5
四倍體 0
四健會 72
四億 284
四億人 0
四億元 75
四元數 1
四克 0
四兩 25
四兩撥千斤 8
四具 15
四出 18
四分 220
四分之一 331
四分之三 165
四分之二 1
四分五裂 26
四分法 3
四分鐘 43
四分音符 4
四列 5
四則 39
四則運算 19
四化 43
四匹 5
四區 96
四十 0
四千 724
四千人 34
四千個 8
四千元 130
四千塊 0
四叔 5
四口 44
四句 75
四史 4
四合 168
四合院 38
四名 324
四向 15
四周 798
四周圍 21
四周年 9
四呼 1
四員 12
四哥 25
四四 284
四四方方 7
四國 180
四國銀行 0
四國銀行團 0
四圍 10
四圓 1
四坪 11
四堆 3
四場 14
四場 147
四境 5
四壁 14
四壺 1
四外 8
四夜 90
四大 1541
四大塊 0
四大家族 4
四大片 0
四大皆空 6
四大菩薩 1
四大金剛 1
四天王天 1
四如意足 1
四孔 10
四字 343
四季 1291
四季如春 16
四季豆 50
四客 1
四射 118
四對 23
四小 16
四小塊 0
四小時 241
四小片 0
四小龍 97
四局 77
四局上半 0
四局下半 0
四屆 1065
四層 118
四層樓 21
四川 463
四川人 3
四川盆地 17
四川省 160
四巷 60
四平 104
四平八穩 9
四平調 0
四年 5897
四年制 236
四年生 16
四年級 792
四度空間 37
四庫 14
四庫全書 138
四式 15
四弘誓 1
四弘誓願 1
四張 220
四德 42
四念住 1
四念處 1
四截 1
四戰 16
四戶 13
四房 49
四房二廳 6
四所 116
四打 5
四批 24
四技 0
四捨五入 44
四支 62
四散 51
四方 260
四方八面 16
四方響應 0
四旁 3
四日 1849
四日份 0
四星 81
四星彩 33
四星期 32
四星級 33
四時 278
四時不絕 0
四時八節 1
四晚 32
四智 5
四書 62
四書五經 9
四月 4463
四月份 455
四月八 10
四期 991
四板 5
四枝 3
四格 66
四桶 1
四棟 19
四棵 2
四極管 0
四樓 788
四樓住戶 1
四樣 17
四檔 18
四櫃 0
四次 356
四正勤 1
四正斷 1
四步 29
四歲 280
四歲時 16
四段 1914
四民 18
四氯 70
四氯 70
四氯化碳 56
四氯化碳 56
四氯甲烷 1
四氯甲烷 1
四法界 1
四洞 6
四洲志 0
四海 607
四海一家 47
四海一家 47
四海工業專科學校 0
四海為家 5
四濺 15
四爺 4
四版 99
四物湯 0
四班 191
四現 0
四球 20
四瓣 1
四瓣花 0
四瓶 7
四男 9
四百 1292
四百人 22
四百個 10
四百元 58
四百塊 3
四盒 3
四盞 3
四相 1
四眼 28
四眼田雞 14
四眼魚 0
四碗 0
四碼 50
四神湯 13
四票 12
四科 123
四秒 16
四秒鐘 3
四站 13
四章 247
四端 23
四筒 2
四箱 3
四節 87
四節課 5
四粒 3
四級 279
四絃琴 0
四維 455
四維八德 7
四維路 228
四線 71
四縣 0
四美 17
四聖諦 1
四聲 56
四肢 294
四肢無力 20
四肢骨 3
四腳 43
四腳朝天 9
四腳蛇 0
四至 265
四船 2
四色 72
四色筆 0
四萬 354
四萬人 17
四萬個 5
四萬元 200
四萬塊 0
四萬萬 2
四萬萬同胞 0
四處 720
四處奔走 7
四處尋找 22
四處張望 9
四號 732
四號台 7
四行 57
四行 57
四街 151
四袋 0
四角 128
四角帽 0
四角形 10
四角號碼 0
四角錐 0
四言詩 0
四課 26
四諦 1
四象 19
四起 98
四路 560
四路縱隊 1
四輛 26
四輪 257
四輪車 6
四通 134
四通八達 73
四週 446
四週年 24
四邊 51
四邊形 62
四邑 0
四郊 3
四部 188
四部合唱 2
四鄉 6
四鄰 49
四里 12
四重 53
四重唱 6
四重奏 168
四野 7
四野茫茫 2
四錢 7
四鍋 1
四鎮 1
四開 10
四開本 1
四隅 1
四隊 33
四階 17
四隻 95
四集 121
四面 168
四面佛 8
四面八方 101
四面受困 0
四面受敵 1
四面楚歌 20
四面體 20
四音 5
四音節 1
四頁 282
四頂 1
四項 606
四預流支 1
四顆 67
四類 258
四顧 20
四首 75
四首歌 7
四體 3
四點 512
四點鐘 14
四黨 9
四齣 16
四齣戲 3
囜 0
囝 61
囝 61
囝仔 61
回 225564
回中 2572
回了 96
回事 20
回交 187
回人 63
回來 2375
回信 6090
回個 78
回光返照 3
回光鏡 0
回函 0
回到 28708
回動 2
回升 437
回去 1014
回台 397
回合 357
回吐 117
回味 155
回味無窮 57
回咬 0
回單 6
回嘴 15
回回 15
回國 749
回執 22
回報 465
回填 113
回天乏術 14
回娘家 109
回家 1834
回家路上 8
回家途中 4
回師 160
回得 2
回得來 2
回得去 2
回復 727
回心 38
回心轉意 37
回想 375
回想出 1
回想到 4
回想起來 51
回憶 1423
回憶說 14
回憶起 42
回憶錄 152
回應 28459
回手 10
回扣 29
回拜 0
回掉 2
回擊 19
回收 3759
回收率 127
回教 597
回教人 0
回教信徒 3
回教帝國 0
回教徒 93
回敬 21
回文 271
回旋 6
回旋加速 0
回旋加速器 0
回族 34
回族人 4
回映 1
回春 121
回暖 31
回本 1359
回條 40
回歸 945
回歸熱 10
回歸線 9
回民 115
回水 21
回波 13
回波 13
回流 301
回游 1
回溯 220
回溯到 9
回潮 3
回潮率 0
回火 100
回爐 1
回環 11
回祿之災 2
回禮 7
回程 214
回答 14561
回答出來 4
回答說 153
回籠 59
回紇 7
回紇兵 0
回紇軍 0
回紋針 0
回絕 36
回絲 0
回罵 0
回老家 14
回聲 58
回航 45
回航途中 0
回著 7
回著 7
回蕩 5
回覆 22555
回見 14
回診 26
回話 2866
回護 47
回贖 4
回起 14
回跌 27
回路 158
回身 35
回轉 104
回轉半徑 0
回返 14
回途 5
回遊 1
回過 47
回過來 1
回部 2
回郵信封 288
回鄉 74
回采 0
回鍋油 6
回鍋肉 1
回門 74
回電 2378
回青 5
回音 176
回響 158
回頭 693
回頭一看 21
回頭一看 21
回頭是岸 19
回頭見 3
回頭路 8
回顧 2310
回顧過去 51
回風 17
回飛棒 0
回饋 2684
回饋性 2
回首 325
回馬槍 11
回駁 1
回魂 13
囟 0
囟門 2
因 30408
因之 172
因人 111
因人成事 0
因人而異 110
因公 213
因公殉職 8
因公殉難 0
因公請假 4
因利乘便 0
因勢利導 19
因噎廢食 9
因地 77
因地制宜 59
因子 922
因小失大 18
因年 5
因年而異 0
因式 6
因式分解 5
因循 20
因循守舊 1
因循舊規 0
因循苟且 5
因應 4620
因應之道 166
因故 439
因故取消 24
因故未來 0
因故辭職 15
因數 59
因數分解 3
因日 42
因日而異 0
因時制宜 7
因材 3
因材施教 107
因果 290
因果報應 22
因果律 21
因果論 4
因果關係 95
因次 20
因此 23278
因此之故 14
因此在 1130
因此是 51
因此會 48
因此有 217
因此而 471
因此能 58
因為 35499
因為是 281
因為會 63
因為有 557
因為能 22
因由 36
因病 140
因病未來 0
因病請假 1
因禍得福 11
因素 4840
因緣 336
因緣觀 1
因緣際會 59
因而 2689
因而在 91
因而是 9
因而會 9
因而有 42
因襲 18
因襲舊規 0
因襲陳規 0
因陋就簡 2
因雨 21
因雨取消 1
因雨延期 0
因雨暫停 1
因革 1
囡 179
囡仔 4
囡囡 4
団 0
囤 37
囤 37
囤積 37
囤積居奇 2
囥 0
囦 0
囧 4440
囩 0
囪 0
囪 77
囫 9
囫圇 7
囫圇吞棗 11
囫圇竹 0
囬 0
园 0
囮 0
囮 0
囯 0
困 2103
困乏 5
困了 4
困住 23
困倦 3
困倦起來 0
困厄 9
困圍 0
困在 84
困境 839
困守 4
困局 21
困心衡慮 0
困惑 248
困擾 1673
困於 42
困獸 7
困獸之鬥 3
困獸猶鬥 3
困窘 46
困窮 2
困竭 0
困苦 70
困阨 3
困難 3233
困難重重 55
困頓 47
囱 0
囱 0
囲 0
図 0
囶 0
囷 0
囸 0
囸 0
囹 1
囹圄 70
固 2155
固化 174
固化劑 6
固執 143
固執己見 12
固如 1
固守 95
固守崗位 1
固定 3878
固定值 1
固定床 9
固定性 22
固定成本 33
固定收入 15
固定物 2
固定班次 3
固定資本 23
固定資產 266
固定資金 2
固定量 2
固定額 0
固實 10
固山 3
固態 328
固態燃料 3
固於 3
固有 234
固有性 0
固有文化 39
固有頻率 0
固氮 8
固氮作用 2
固溶體 3
固為 12
固無 4
固然 543
固然在 12
固然是 116
固然會 4
固然有 31
固結 11
固網 0
固若金湯 13
固醇 456
固陋 0
固體 493
固體化 1
固體燃料 18
囻 0
囼 0
国 0
囿 292
囿於一時 0
囿於一時 0
囿於成見 0
圀 0
圁 0
圂 0
圃 1
圄 0
圅 0
圇 0
圈 0
圈 0
圈 21
圈 2110
圈住 17
圈兒 8
圈兒 8
圈內 66
圈內 66
圈內人 17
圈內人 17
圈圈 149
圈地 4
圈外 16
圈外 16
圈外人 5
圈外人 5
圈套 31
圈子 87
圈子裡 8
圈形 2
圈梁 0
圈樑 0
圈環 5
圈裡 21
圈裡人 0
圈起 6
圈起來 8
圈選 95
圈點 17
圉 149
圉限 0
圊 0
國 40672
國一 186
國一上 2
國一下 1
國一生 1
國丈 2
國三 202
國三上 2
國三下 7
國三生 5
國中 9448
國中生 227
國事 54
國事犯 0
國二 160
國二上 0
國二下 2
國二生 2
國亡 2
國交 6
國人 2310
國代 1246
國代選舉 75
國企 0
國債 64
國光 630
國光號 22
國內 23511
國內外 4373
國內市場 28
國內市場 286
國內戰爭 0
國內法 34
國內生產 275
國內生產淨額 1
國內貿易 10
國公 49
國劇 112
國劇團 5
國劇社 9
國力 184
國務 146
國務卿 90
國務會議 2
國務秘書 1
國務總理 8
國務部長 3
國務院 489
國勢 33
國協 171
國史 178
國史大綱 2
國史館 105
國名 122
國君 35
國命 58
國喪 1
國四 732
國四班 3
國園 10
國土 868
國基 68
國境 105
國境線 1
國士 36
國外 9231
國外市場 50
國外市場 50
國外貿易 7
國大 1814
國大代表 399
國大代表部 0
國姓 292
國姓爺 6
國威 80
國子監 16
國字 169
國字臉 1
國學 128
國學常識 7
國學社 1
國宅 783
國宅會 0
國宅處 77
國安局 80
國安法 18
國定 76
國定假日 238
國定紀念日 8
國宴 35
國家 21593
國家主義 18
國家元首 42
國家公園 3094
國家利益 76
國家制度 9
國家化 19
國家圖書館 0
國家基金 5
國家報 0
國家安全 510
國家安全會議 69
國家形式 2
國家性 10
國家標準 403
國家機器 50
國家機密 55
國家機構 37
國家機關 53
國家權力 25
國家權力機關 4
國家歌劇院 2
國家法 7
國家社會主義 5
國家科學委員會 63253
國家積累 0
國家考試 282
國家至上 1
國家與革命 0
國家興亡 7
國家行政學院 1
國家行政機關 2
國家資本 2
國家資本主義 4
國家銀行 44
國家預算 30
國富 84
國富民安 1
國富民強 3
國富論 5
國寶 422
國寶級 57
國小 14705
國帑 0
國師 27
國幣 14
國度 639
國庫 361
國庫券 74
國建 176
國建學術聯誼會 2
國建計劃 3
國強 144
國強民富 1
國徽 26
國恥 14
國情 910
國情咨文 4
國慶 387
國慶大典 2
國慶日 105
國慶節 2
國戚 0
國手 162
國政 0
國故 5
國教 383
國教司 48
國教派 2
國敵 0
國文 1680
國文科 163
國文課 71
國旗 310
國旗歌 9
國是 384
國是會議 192
國曆 137
國書 61
國會 2659
國會制 2
國會議員 305
國有 682
國有化 10
國有政策 0
國有財產 33
國有財產局 166
國朝 21
國本 46
國校 157
國格 13
國樂 330
國樂團 288
國樂家 2
國樂社 132
國標舞 1
國權 12
國歌 129
國步 3
國步艱難 0
國殤 7
國母 5
國民 5460
國民中學 3819
國民住宅 356
國民兵 73
國民報 0
國民外交 27
國民大會 878
國民大會代表 289
國民學校 266
國民小學 9237
國民平均 13
國民平均所得 5
國民性 8
國民所得 301
國民收入 1
國民政府 194
國民教育 765
國民會議 9
國民生產毛額 132
國民生產淨額 2
國民生產總值 14
國民經濟 124
國民經濟計劃 3
國民警衛隊 0
國民身分 66
國民身分証 5
國民身分證 219
國民軍 2
國民革命 29
國民革命軍 12
國民革命黨 0
國民黨 8545
國民黨員 26
國法 35
國法難容 0
國泰 871
國泰人壽 257
國泰公司 1
國泰民安 35
國泰產物 5
國泰綜合醫院 62
國泰醫院 126
國營 287
國營事業 417
國營企業 105
國營商業 3
國營農場 2
國營農場 2
國父 458
國父思想 95
國父紀念館 352
國父遺囑 0
國片 195
國片商 0
國片展 0
國特 36
國特局 0
國王 687
國王的新衣 1
國璽 85
國產 2163
國產品 73
國用 13
國界 244
國界線 0
國畫 447
國畫家 3
國破家亡 3
國社黨 0
國科會 3748
國稅 208
國稅局 1018
國立 89150
國立中央大學 1021
國立大學 303
國立藝專 72
國策 366
國策顧問 106
國籍 752
國粹 41
國統區 0
國統會 0
國聯 476
國腳 16
國腿 0
國舅 27
國航 65
國航局 0
國色 0
國色天香 5
國花 56
國華 758
國華人壽 99
國葬 1
國號 273
國術 428
國術團 2
國術社 53
國計 10
國計民生 13
國語 1428
國語劇 0
國語文 394
國語日報 243
國語科 165
國語羅馬字 3
國語課 10
國變 3
國讎 0
國貨 55
國貨國運 0
國貿 404
國貿局 268
國貿科 82
國貿系 144
國賊 9
國賓 250
國賓戲院 85
國賓飯店 59
國軍 1189
國軍退除役官兵 262
國輪 25
國輪國造 0
國運 187
國運昌隆 19
國道 358
國都 83
國醫 54
國門 39
國防 1215
國防力量 14
國防大學 1
國防委員 8
國防委員會 27
國防工業 61
國防役 1
國防管理 32
國防管理學院 300
國防論 0
國防費 7
國防軍 27
國防部 1348
國防部長 215
國防醫學 12
國防醫學院 310
國際 44328
國際主義 15
國際公法 29
國際公約 72
國際勞動節 0
國際化 1136
國際博覽 0
國際博覽會 5
國際商專 12
國際單位 5
國際單位制 2
國際學舍 16
國際局勢 63
國際市場 53
國際市場 539
國際帝國主義 0
國際性 575
國際慣例 36
國際排總 3
國際換日線 25
國際收支 83
國際文教 47
國際文教組織 3
國際標準 689
國際機場 35
國際機場 352
國際歌 5
國際法 223
國際法庭 21
國際法院 5
國際牌 283
國際特赦組織 60
國際私法 49
國際經濟 436
國際聯盟 32
國際舞台 77
國際觀 133
國際貿易 1804
國際都市 26
國際鋁卡 0
國際開發總會 0
國際關係 338
國際音標 1
國際飯店 70
國難 12
國難方殷 0
國難日 0
國難財 2
國音 38
國風 103
國體 26
國魂 13
圌 0
圍 6888
圍事 100
圍上 7
圍上來 2
圍上去 3
圍下 1
圍下來 0
圍下去 0
圍了 27
圍住 63
圍兜 8
圍兜兜 0
圍兜起來 0
圍剿 45
圍包 2
圍困 32
圍困在 2
圍坐 10
圍坐在 9
圍坐著 0
圍城 22
圍場 7
圍場 7
圍堵 96
圍墾 0
圍子 1
圍子 1
圍屏 7
圍岩 2
圍巖 0
圍巾 80
圍得 1
圍得 1
圍成 50
圍捕 44
圍擒 0
圍攏 3
圍攏過來 2
圍攻 63
圍棋 275
圍棋賽 18
圍標 54
圍殲 2
圍毆 36
圍爐 42
圍牆 223
圍的 30
圍看 3
圍看著 0
圍砌 0
圍網 39
圍網漁業 9
圍繞 142
圍繞在 68
圍繞著 104
圍著 88
圍裙 27
圍裙帶 0
圍視 1
圍觀 76
圍觀者 8
圍起 11
圍起來 5
圍過來 9
圍過去 2
圍靠 0
圍靠在 0
圎 0
圏 0
圏 0
圏 0
圏 0
圐 0
圑 0
園 59465
園丁 245
園內 480
園區 1806
園圃 52
園地 7960
園外 12
園子 78
園寢 0
園庭 0
園林 139
園田 15
園田化 0
園藝 1703
園藝家 40
園裡 159
園遊會 0
園長 162
園陵 1
圓 59465
圓光 19
圓台 5
圓周 43
圓周率 14
圓周角 2
圓圈 111
圓圈式 2
圓圓 94
圓圓的 62
圓圓胖胖 1
圓場 11
圓場 11
圓夢 80
圓子 0
圓寂 29
圓屋頂 0
圓山 398
圓弧 141
圓弧規 0
圓形 510
圓形槽 0
圓徑 4
圓心 43
圓心角 4
圓成 8
圓房 1
圓整 16
圓明園 8
圓暢 0
圓月 21
圓木 14
圓板 9
圓板狀 0
圓柱 158
圓柱狀 6
圓柱面 2
圓柱體 21
圓桌 79
圓桌會議 114
圓桌武士 76
圓桶 15
圓渦 0
圓渾 3
圓溜溜 1
圓滑 66
圓滿 828
圓滿結束 78
圓潤 57
圓熟 18
圓珠 6
圓珠筆 0
圓球 70
圓球兒 0
圓球兒 0
圓環 292
圓環一帶 0
圓盤 123
圓石 12
圓筆 1
圓筒 95
圓筒狀 10
圓籽 0
圓籽湯 0
圓經 0
圓臉 7
圓舞曲 55
圓融 149
圓融三諦 1
圓規 13
圓覺經 1
圓謊 10
圓通 105
圓通寺 24
圓鋼 0
圓錐 85
圓錐形 22
圓錐曲線 4
圓錐體 3
圓鍬 4
圓面積 4
圓頂 87
圓領 28
圓顱方趾 0
圓餅 8
圔 0
圖 29988
圖上 216
圖下 14
圖下方 1
圖中 100
圖例 244
圖們 4
圖們江 5
圖像 1476
圖像資料 13
圖內 26
圖冊 16
圖利 171
圖利他人 20
圖右 27
圖名 42
圖名圖利 0
圖存 10
圖左 36
圖式 89
圖形 6108
圖拉 5
圖文 1339
圖文並茂 115
圖景 5
圖書 10695
圖書室 756
圖書店 0
圖書目錄 2588
圖書費 4
圖書資料 570
圖書館 30749
圖書館學 383
圖板 24
圖案 2059
圖樣 397
圖檔 0
圖為 142
圖片 7978
圖片說明 61
圖版 29
圖畫 356
圖畫冊 2
圖畫書 188
圖畫紙 17
圖示 1778
圖章 40
圖籍 14
圖紙 26
圖線 14
圖表 713
圖表說明 31
圖裡 8
圖解 374
圖解法 23
圖記 25
圖說 827
圖謀 37
圖謀不軌 4
圖象 70
圖釘 34
圖錐體 0
圖鑑 115
圖騰 167
圖騰崇拜 8
圗 0
團 13222
團保 71
團名 51
團員 597
團圓 94
團圓夜 2
團圓日 0
團圓節 0
團圓飯 5
團團 139
團團圍住 19
團團轉 34
團坐 0
團契 879
團委 37
團扇 8
團拜 67
團旗 33
團服 22
團狀 6
團管區 54
團結 694
團結一致 33
團結就是力量 7
團練 37
團聚 70
團臍 0
團茶 0
團購 1
團部 67
團長 348
團隊 2045
團隊精神 139
團體 9874
團體冠軍 16
團體性 12
團體操 1
團體活動 136
團體票 42
團體賽 79
團魚 0
圚 0
圛 0
圜 95
圜 95
圜鑿方枘 0
圝 0
圞 0
土 5715
土上 10
土人 38
土人族 0
土俗 10
土包 2
土包子 4
土匪 62
土司 57
土司麵包 12
土味十足 0
土土 16
土土氣氣 0
土地 11176
土地公 202
土地國有 1
土地國有化 0
土地增值 60
土地婆 5
土地局 1
土地廟 15
土地所有 68
土地改良 28
土地改革 55
土地改革運動 0
土地權 39
土地稅 81
土地証 0
土地證 0
土地資源 191
土地重劃 113
土地銀行 260
土地革命 0
土坑 3
土坯 2
土城 987
土城鄉 37
土堆 13
土塊 21
土壤 2084
土壤學 14
土壤微生 0
土壤微生物 8
土壤有機 2
土壤有機質 5
土壤水 6
土壤溫度 3
土壤溶液 4
土壤溶液 4
土壤濕度 1
土壤結構 6
土壤肥力 13
土壤膠體 1
土層 54
土崩 10
土崩瓦解 2
土工 17
土布 4
土庫 495
土性 77
土拉河 1
土改 3
土斷 1
土方 138
土族 5
土星 182
土曜日 0
土木 1999
土木工程 980
土木工程學系 313
土木工程系 149
土木系 334
土棍 0
土樣 20
土氣 9
土法 17
土法煉鋼 11
土法生產 0
土洋並舉 0
土洋結合 0
土溫 7
土牆 181
土牢 1
土特產 2
土狗 87
土狼 0
土生 18
土生土長 43
土產 142
土番 8
土皇帝 4
土石 0
土石方 172
土石流 0
土砂 16
土礫 0
土種 6
土穴 3
土糞 0
土耳其 953
土耳其人 5
土耳其浴 0
土肥 8
土腥味兒 0
土腥味兒 0
土菩薩過江 1
土著 123
土葬 4
土虱 22
土蛋 4
土裡 25
土裡土氣 0
土製 31
土製品 11
土話 7
土語 10
土谷祠 0
土豆 73
土豪 3
土豪劣紳 0
土貨 2
土質 92
土路 7
土造 0
土遁 0
土銀 135
土門 2
土雞 85
土雞城 27
土雞場 1
土雞場 1
土霉素 0
土頭土腦 0
土頭土臉 0
土風 3
土風舞 173
土魯蕃窪 0
土黃 5
土黃色 20
土黴素 0
土龍 0
圠 0
圡 0
圢 0
圣 0
圣 0
圤 0
圥 0
圧 0
在 0
在 36484
在一起 1152
在一起 1152
在上 556
在下 1154
在世 131
在乎 555
在位 53
在位者 8
在來米 3
在做 586
在先 80
在內 939
在出 42
在制品 0
在刻 4
在割 2
在劫難逃 4
在即 145
在台 2410
在台協會 173
在咬 5
在唱 51
在商言商 17
在在 342
在地 546
在堂 2
在場 50
在場 506
在外 546
在天之靈 0
在學 627
在學生 110
在家 913
在家人 28
在審 12
在幫 93
在座 126
在心 193
在想 376
在意 397
在所不惜 24
在所不辭 6
在所難免 60
在打 132
在押 16
在押犯 0
在按 43
在握 22
在搖 7
在搬 4
在教 327
在於 4072
在望 42
在朝 27
在朝 27
在校 841
在校生 192
在案 87
在業 11
在此 6262
在此一舉 3
在此一舉 3
在混 30
在演 77
在為 223
在營 144
在營休假 0
在理 28
在生 215
在產品 193
在看 421
在眼裡 9
在種 29
在笑 61
在編 18
在職 1337
在職幹部 0
在職的 0
在職訓練 177
在職進修 455
在行 113
在補 23
在要 56
在記 8
在說 316
在請 29
在讀 93
在賭 9
在跟 154
在辦 53
在逃 63
在逃犯 0
在運 29
在過 163
在野 113
在野黨 326
在陪 15
在騎 8
圩 20
圩 20
圩田 0
圪 53
圫 0
圬 58
圭 302
圭亞那 13
圭臬 16
圭表 4
圮 122
圮毀 0
圯 58
地 147269
地 40
地 58528
地上 1791
地下 2241
地下城 0
地下室 561
地下害蟲 0
地下工作 0
地下工廠 12
地下掩體 0
地下水 797
地下水位 38
地下活動 3
地下組織 7
地下莖 10
地下街 1
地下軍 1
地下道 79
地下錢莊 1
地下鐵 100
地下鐵道 1
地下開採 1
地中 65
地中海 457
地主 319
地主之誼 16
地主之誼 16
地主分子 0
地主國 54
地主隊 18
地主階級 2
地位 2608
地位級 0
地位高 2
地保 16
地價 600
地價稅 245
地利 202
地力 47
地動 64
地動儀 2
地勢 170
地勤 43
地勤人員 13
地區 20483
地區性 259
地名 507
地圖 4516
地圖冊 6
地圖集 53
地地道道 2
地址 15478
地坼天崩 0
地域 423
地域性 50
地域觀念 1
地基 74
地基承載力 0
地堡 2
地塊 7
地壓 5
地大物博 6
地契 10
地委 9
地學 82
地對地 12
地對空 7
地層 412
地層下陷 210
地峽 5
地帶 724
地幔 4
地平 61
地平線 155
地底 100
地底下 21
地府 26
地廣 18
地廣人稀 12
地形 1378
地形圖 182
地心 50
地心吸力 8
地心引力 60
地志 6
地拉那 4
地攤 20
地支 32
地政 1609
地政事務所 754
地政司 0
地政處 0
地方 12402
地方分權 33
地方國營 0
地方官 13
地方工業 1
地方建設 138
地方志 23
地方性 137
地方戲 14
地方政府 926
地方新聞 76
地方武裝 0
地方法院 1072
地方病 0
地方稅 156
地方自治 383
地方軍 2
地方部隊 0
地板 1305
地核 14
地極 17
地標 1
地檢署 0
地檢處 4
地權 48
地步 406
地段 869
地殼 96
地殼運動 6
地毯 441
地毯式轟炸 0
地波 6
地波 6
地洞 63
地溫 4
地滑 59
地滾球 0
地火 25
地熱 84
地熱能 7
地牢 25
地物 51
地獄 547
地獄谷 10
地獄道 1
地球 4470
地球儀 100
地球化學 79
地球日 26
地球村 100
地球科學 615
地球同步軌道 1
地球靜止軌道 1
地理 3856
地理位置 900
地理學 188
地理學家 10
地理環境 183
地理課 2
地瓜 105
地瓜湯 3
地瓜粉 1
地產 3643
地產稅 2
地界 29
地畝 0
地當 31
地痞 1
地痞流氓 5
地皮 11
地盤 218
地磁 36
地磅 51
地磚 167
地租 49
地秤 9
地稅 26
地積 2
地穴 15
地窖 36
地窖子 1
地窨子 0
地籍 1
地籍圖 230
地維 5
地線 38
地緣 128
地緣關係 35
地縫 0
地縫裡 0
地老天荒 11
地老天荒不了情 1
地腳 1
地藏王 1
地藏王菩薩 1
地藏菩薩 1
地處 223
地處 223
地蜜 0
地蠟 0
地衣 9
地表 484
地誌 24
地貌 94
地質 1963
地質圖 76
地質學 186
地質學家 30
地質編錄 0
地軸 10
地道 190
地道口 1
地邊 6
地錢 5
地鐵 251
地院 0
地陪 22
地雷 430
地電 7
地震 1442
地震儀 6
地震學 24
地震預報 0
地面 1157
地面上 116
地面下 9
地面水 29
地面溫度 3
地面站 49
地面部隊 12
地頭 15
地頭蛇 2
地點 10600
地鼠 122
地龍 41
圱 0
圲 0
圳 476
圳 476
圳 476
圳溝 11
圴 215
圴 215
圵 0
圶 0
圻 1220
圼 0
圽 0
圾 275
圾 275
圿 0
址 3502
坁 145
坂 0
坃 0
坃 0
坄 0
坅 130
坆 0
坆 0
均 15067
均一 73
均享 29
均分 54
均勢 20
均勻 530
均可 1406
均夷線 0
均安 20
均富 70
均已 352
均攤 8
均有 1210
均標 0
均權 15
均沾 3
均無 185
均熱 3
均熱爐 0
均田 0
均田制 1
均田制度 1
均相 76
均等 101
均衡 943
均衡論 3
均貧 2
均貧富 0
均霑 13
坈 0
坉 45
坊 3253
坊 3253
坊肆 0
坊間 315
坋 1166
坌 132
坌集 0
坍 65
坍倒 0
坍塌 44
坍壞 0
坍崩 3
坍方 124
坍臺 0
坍落度 0
坎 1089
坎坷 102
坎坷不平 1
坎城 114
坎城影展 47
坎培拉 62
坎帕拉 1
坎肩 1
坏 13
坏 132
坏 132
坐 4099
坐上 113
坐上來 1
坐上去 3
坐下 210
坐下來 132
坐下去 5
坐不住 4
坐不穩 2
坐不起 0
坐了 136
坐井觀天 6
坐享其成 11
坐以待斃 11
坐以待旦 2
坐位 35
坐具 1
坐冷板凳 2
坐到 81
坐化 4
坐吃山空 8
坐在 1445
坐地分贓 1
坐坐 671
坐墊 24
坐大 20
坐失良機 3
坐守 5
坐定 26
坐山 5
坐山觀虎 1
坐山觀虎鬥 3
坐席 11
坐床 5
坐待 3
坐得 16
坐得 16
坐懷不亂 1
坐收 34
坐收漁利 1
坐月子 77
坐標 59
坐法 5
坐滿 22
坐牢 47
坐的 92
坐禪 45
坐禪關 1
坐立 22
坐立不安 25
坐船 18
坐船來 0
坐船到 1
坐船去 1
坐落 789
坐著 254
坐視 55
坐視不管 12
坐觀成敗 0
坐談會 0
坐起 27
坐起來 23
坐車 60
坐車來 0
坐車到 10
坐車去 2
坐鎮 55
坐領乾薪 2
坐騎 26
坐骨 75
坐骨神經 31
坑 1888
坑人 17
坑儒 0
坑坑 2
坑坑洞洞 9
坑坑窪窪 0
坑掉 0
坑木 0
坑殺 8
坑洞 68
坑裡 8
坑起來 0
坑道 30
坑道工事 0
坑道戰 0
坒 207
坓 0
坔 0
坔 0
坕 0
坖 0
块 0
坘 0
坘 0
坟 0
坟 0
坡 1406
坡地 391
坡度 251
坡田 0
坡路 26
坡道 54
坢 37
坣 0
坤 2769
坤伶 2
坤宅 0
坤輿 0
坤造 1
坥 0
坦 686
坦克 128
坦克兵 0
坦克車 13
坦坦 23
坦平 1
坦承 262
坦桑尼亞 11
坦潔 0
坦然 169
坦然自若 0
坦率 53
坦白 125
坦白 125
坦白說 48
坦白講 8
坦盪 1
坦直 1
坦胸露背 1
坦腹 0
坦蕩 15
坦蕩蕩 7
坦言 0
坦誠 87
坦誠以待 0
坦誠相待 4
坦途 25
坧 0
坧 0
坨 15
坩 5
坩坩 0
坩堝 10
坪 3380
坪林 326
坪林鄉 93
坫 34
坬 0
坭 11
坮 0
坯 57
坯 57
坯子 0
坯子 0
坯布 1
坯布 1
坯料 1
坯料 1
坰 117
坱 8
坲 12
坳 181
坳塘 0
坴 707
坵 66
坶 11
坷 12
坸 0
坹 0
坺 0
坺 0
坻 20
坻 20
坼 10
坽 7
坾 0
坿 0
垀 1085
垁 0
垂 872
垂下 93
垂下來 3
垂下去 0
垂了 1
垂了下來 2
垂了下去 2
垂倒 1
垂入 1
垂到 6
垂危 55
垂危之際 1
垂告 0
垂心 1
垂念 1
垂愛 4
垂憐 14
垂成 4
垂手 9
垂手可得 29
垂教 1
垂暮 3
垂暮之年 1
垂柳 21
垂楊 98
垂死 50
垂死掙扎 3
垂死者 2
垂泣 0
垂涎 29
垂涎三尺 12
垂涎欲滴 3
垂涕 1
垂淚 0
垂直 1036
垂直平分 1
垂直平分線 2
垂直於 23
垂直線 26
垂直舵 0
垂直角 1
垂直距離 11
垂直面 1
垂簾 6
垂簾聽政 12
垂簾聽政 12
垂線 4
垂老 6
垂著 15
垂著手 0
垂著頭 19
垂詢 10
垂足 2
垂釣 100
垂青 13
垂頭 29
垂頭 29
垂頭喪氣 13
垂首 17
垂體 150
垂髫 0
垂髮 0
垃 136
垃圾 3793
垃圾堆 39
垃圾場 23
垃圾場 238
垃圾桶 119
垃圾筒 36
垃圾箱 36
垃圾蟲 0
垃圾車 72
型 22764
型別 162
型式 1235
型態 3140
型模 27
型砂 2
型範 0
型號 1102
型鋼 69
型錄 2385
垌 6
垍 0
垎 0
垏 1
垐 0
垑 0
垒 0
垓 260
垔 1
垕 4
垖 0
垖 0
垗 8
垘 2
垙 5
垚 19
垛 45
垛 45
垜 0
垜 0
垝 6
垞 2
垟 6
垠 286
垡 0
垢 206
垢弊 1
垢汙 0
垢泥 0
垣 311
垤 3
垥 6
垦 0
垧 0
垧 0
垨 0
垨 0
垩 0
垬 0
垮 196
垮下 4
垮下來 4
垮下去 2
垮了 18
垮台 40
垮向 0
垮在 0
垮得 0
垮得 0
垮成 1
垮掉 13
垮臺 1
垴 0
垵 80
垶 2
垷 0
垸 9
垹 8
垺 3
垻 0
垼 2
垽 1
垾 0
垿 8
埀 0
埁 0
埂 17
埃 538
埃個兒 0
埃個兒 0
埃及 659
埃及人 51
埃及古文 5
埃及文 5
埃及館 1
埃塞俄比亞 3
埃塞俄比亞 3
埃森 6
埄 0
埅 0
埆 45
埇 6
埈 0
埉 0
埊 0
埊 0
埋 449
埋 449
埋下 70
埋下去 0
埋了 9
埋伏 99
埋佈 0
埋入 40
埋名 1
埋單 1
埋得 1
埋得 1
埋怨 109
埋怨 109
埋怨起來 0
埋沒 60
埋沒人才 1
埋置 3
埋著 4
埋葬 119
埋葬地 3
埋藏 86
埋藏地 0
埋設 94
埋起 0
埋起來 9
埋頭 30
埋頭伏案 0
埋頭苦幹 15
埋首 59
埌 5
埍 0
城 14750
城下之盟 1
城中 329
城中區 14
城主 61
城內 164
城區 76
城區部 52
城南 30
城垣 27
城堡 625
城壕 1
城外 91
城子 15
城市 4246
城市國家 5
城市規劃 16
城市貧民 0
城府 16
城旦 0
城樓 24
城池 46
城河 0
城牆 128
城狐社鼠 0
城裡 148
城西 46
城邦 61
城郊 7
城郭 11
城鄉 1068
城鄉差距 1
城鎮 523
城門 177
城關 1
城防 4
城防工事 0
城隍 48
城隍廟 130
城隍爺 27
城頭 14
埏 222
埐 7
埑 0
埒 15
埓 0
埔 2015
埔 2015
埔心 365
埔心 365
埔心鄉 0
埔心鄉 0
埔里 1404
埔里 1404
埔里鎮 920
埔里鎮 920
埔鹽 211
埔鹽 211
埕 334
埛 0
埛 0
埜 336
埝 0
埝 0
埝 0
埞 0
埞 0
域 2553
域中 50
域外 28
埠 934
埠頭 3
埡 121
埢 15
埣 154
埤 70
埤 706
埤 706
埤 706
埤頭 219
埤頭鄉 1
埥 97
埦 0
埧 35
埨 0
埩 94
埪 0
埫 0
埬 93
埭 233
埮 443
埯 0
埯 0
埰 176
埱 269
埲 114
埳 94
埴 340
埴土 0
埵 109
埶 251
執 9000
執一 50
執中 18
執事 107
執事官 0
執住 1
執勤 43
執友 0
執委會 101
執導 68
執念 0
執意 128
執拗 7
執拗 7
執掌 98
執政 750
執政堂 0
執政官 0
執政者 79
執政黨 500
執教 69
執業 610
執法 355
執法如山 1
執法者 25
執照 2250
執牛耳 5
執禮 0
執筆 117
執筆人 13
執簡馭繁 0
執紼 0
執著 624
執著性 0
執行 16360
執行官 4
執行者 77
執行起來 6
執行長 470
執迷 62
執迷不悟 25
執金吾 1
執鞭 4
埸 369
培 890
培土 5
培基 48
培基語言 7
培根 82
培植 157
培植出 5
培育 2804
培育出 88
培育法 222
培訓 2058
培訓班 150
培養 4945
培養出 153
培養出來 4
培養基 114
培養所 1
培養液 36
培養液 36
培養皿 17
培養目標 0
基 11597
基因 1672
基因型 41
基因工程 82
基團 13
基地 2144
基地台 1
基坑 2
基層 2075
基層人員 41
基層人民 3
基層人民法院 6
基層組織 11
基岩 4
基巖 1
基幹 5
基底 106
基座 106
基建 63
基態 12
基數 94
基於 2557
基期 0
基本 14905
基本上 1885
基本功 33
基本原則 211
基本學科 52
基本工資 104
基本建設 200
基本建設投資 0
基本建設項目 1
基本教育 93
基本核算 0
基本核算單位 0
基本生產 5
基本矛盾 1
基本粒子 11
基本粒子 11
基本經濟 20
基本經濟規律 0
基本苗 0
基本路線 4
基柱 5
基業 41
基極 16
基構 2
基準 881
基準兵 0
基準線 6
基準面 6
基準點 21
基爾 122
基特爾 0
基督 2807
基督學院 140
基督徒 1267
基督教 1948
基督教徒 16
基督教會 64
基督書院 35
基石 140
基礎 7434
基礎性 38
基礎理論 83
基礎科學 136
基礎課 7
基網格 0
基線 32
基肥 16
基色 4
基調 88
基質 201
基輔 32
基部 85
基金 11664
基金會 8508
基隆 4308
基隆人 8
基隆嶼 21
基隆市 3225
基隆河 403
基隆港 285
基隆港務局 119
基音 10
基點 93
埻 301
埻 301
埼 110
埽 269
埾 0
埿 0
埿 0
埿 0
堀 223
堀 223
堁 273
堂 1812
堂上 35
堂伯叔 0
堂倌 1
堂兄 15
堂兄弟 6
堂務 0
堂哥 14
堂哥哥 0
堂哥哥 0
堂堂 111
堂堂正正 41
堂堂皇皇 0
堂奧 47
堂妹 3
堂姊 9
堂姊妹 1
堂姐妹 0
堂姪 1
堂嫂 1
堂子 0
堂客 4
堂屋 7
堂弟 19
堂會 42
堂皇 43
堂而皇之 20
堃 0
堄 56
堄 56
堅 2033
堅不可摧 1
堅信 130
堅冰 5
堅厚 7
堅固 294
堅固耐用 48
堅壁 2
堅壁清野 1
堅如磐石 0
堅守 234
堅守崗位 16
堅守崗位 16
堅守自盜 0
堅守陣地 1
堅定 485
堅定不移 37
堅定信心 4
堅定信念 11
堅實 189
堅強 592
堅強不屈 0
堅強不撓 0
堅強有力 2
堅忍 50
堅忍不拔 23
堅拒 31
堅持 2048
堅持下去 18
堅持不下 1
堅持不懈 5
堅持不渝 0
堅持到底 45
堅持己見 13
堅持著 25
堅挺 28
堅果 29
堅毅 94
堅決 269
堅決不移 0
堅決不變 1
堅決不變 1
堅牢 30
堅牢度 214
堅牢耐用 0
堅甲利兵 1
堅硬 155
堅硬無比 1
堅稱 49
堅苦 3
堅苦卓絕 3
堅貞 25
堅貞不屈 0
堅韌 41
堅韌不拔 0
堅韌度 2
堆 798
堆一堆 0
堆一堆 0
堆上 8
堆上來 0
堆上去 0
堆下 6
堆了 13
堆來堆去 0
堆入 1
堆垛 0
堆堆 2
堆好 5
堆得 8
堆成 15
堆成堆 0
堆放 51
堆有 4
堆棧 6
堆滿 54
堆焊 1
堆疊 182
堆砌 59
堆積 361
堆積如山 28
堆積成 3
堆積起來 4
堆肥 249
堆著 13
堆起 7
堆起來 1
堆過 1
堆過來 0
堆過去 0
堆金積玉 0
堆高機 105
堇 221
堈 202
堉 43
堊 49
堊室 1
堊帚 1
堊慢 1
堊粉 1
堋 258
堌 64
堍 300
堎 479
堐 147
堔 32
堗 0
堘 0
堙 307
堙曖 1
堙滅 0
堙窒 1
堚 0
堛 125
堜 26
堝 43
堞 25
堟 0
堠 30
堡 1893
堡主 671
堡壘 115
堡子 0
堡寨 0
堢 0
堢 0
堣 228
堤 2106
堤內 9
堤堰 1
堤壩 4
堤外 18
堤岸 33
堤道 2
堤防 229
堤防道 0
堥 26
堦 0
堧 286
堨 0
堨 7594
堩 2
堪 504
堪察 11
堪測 0
堪為 25
堪為表率 5
堪用 49
堪當 2
堪當重任 0
堪稱 316
堪稱是 41
堪薩斯 120
堪薩斯州 21
堪虞 38
堪輿 38
堪輿家 1
堪驗 2
堫 0
堬 57
堭 99
堮 28
堯 1150
堯天舜日 0
堯舜 17
堰 160
報 29597
報上 171
報上來 1
報上去 0
報仇 138
報以 56
報低 0
報佳音 54
報信 16
報值 25
報值掛號 54
報備 576
報價 2365
報價單 1
報償 37
報冤 5
報冤雪恨 0
報出 30
報出來 6
報出去 0
報分 1
報分數 0
報刊 166
報刊雜誌 13
報到 1069
報功 11
報務 26
報名 6452
報名單 19
報名表 1196
報名費 291
報告 15824
報告出來 9
報告書 859
報告給 19
報命 2
報喜 45
報喜不報憂 1
報喪 0
報單 275
報國 77
報報 746
報夾 5
報安 2
報導 25454
報導出來 8
報帖 0
報帳 35
報廢 201
報復 247
報德 7
報怨 10
報恩 37
報應 80
報捷 6
報效 24
報數 9
報文 14
報文格式 0
報時 65
報曉 1
報架 18
報案 463
報業 121
報界 18
報社 482
報禁 38
報稅 210
報章 126
報章雜誌 242
報答 71
報箱 0
報紙 2620
報紙堆 2
報繳 99
報考 1130
報考人 7
報表 1022
報表紙 126
報請 1075
報警 110
報讎 38
報販 0
報費 14
報身佛 1
報載 259
報道 35
報酬 457
報酬率 586
報銷 72
報關 547
報關行 148
報障 1
報頭欄 0
報館 46
報高 2
堲 4
堳 145
場 1238
場 12386
場主 58
場主 58
場區 32
場區 32
場合 60
場合 607
場地 154
場地 1547
場場 81
場外 19
場外 199
場子 22
場子 22
場子 22
場屋 1
場屋 1
場所 314
場所 3148
場效應 0
場效應 0
場效應晶體管 0
場效應晶體管 0
場景 61
場景 615
場次 59
場次 593
場磁鐵 0
場磁鐵 0
場規 8
場規 8
場記 6
場記 6
場院 2
場面 51
場面 512
場頻 1
場頻 1
堵 275
堵上 2
堵住 46
堵到 5
堵口 0
堵嘴 0
堵在 10
堵塞 97
堵截 2
堵著 3
堵起 1
堵起來 0
堵車 18
堵門 1
堶 48
堷 9
堸 86
堹 19
堺 0
堻 91
堼 0
堽 5
堾 0
堾 0
堿 52
塃 0
塄 0
塅 0
塈 34
塉 31
塊 1963
塊兒 0
塊兒 0
塊塊 6
塊根 7
塊煤 0
塊狀 91
塊莖 11
塊頭 3
塋 62
塌 107
塌 107
塌下 1
塌下來 18
塌下去 0
塌倒 3
塌台 0
塌實 0
塌方 7
塌臺 0
塌陷 34
塌鼻 0
塌鼻子 0
塍 7
塎 6
塏 11
塐 0
塐 0
塑 2023
塑像 75
塑化劑 1
塑匠 0
塑性 78
塑性材料 0
塑性變形 6
塑成 23
塑料 107
塑料工業 0
塑煉 0
塑煉機 0
塑膠 5420
塑膠套 4
塑膠工業 459
塑膠板 18
塑膠泡綿 2
塑膠片 11
塑膠球 4
塑膠瓶 39
塑膠管 53
塑膠紙 4
塑膠袋 215
塑膠錢 0
塑膠體 0
塑身 0
塑造 667
塑造出 79
塑造成 47
塒 16
塓 3
塔 8042
塔上 28
塔下 7
塔台 124
塔吉克 33
塔城 48
塔墩 1
塔式起重機 1
塔斯社 2
塔爾巴哈台 0
塔臺 19
塔里木河 2
塔里木盆地 6
塔門 0
塕 2
塖 0
塗 1727
塗上 155
塗來塗去 0
塗到 4
塗去 2
塗在 45
塗塗 14
塗塗改改 0
塗好 2
塗寫 6
塗成 2
塗抹 205
塗掉 5
塗改 44
塗改無效 0
塗料 633
塗有 11
塗油 3
塗消 1
塗滿 0
塗漆 8
塗潭 1
塗竄 0
塗粉 0
塗脂抹粉 1
塗膠 16
塗色 16
塗色板 0
塗著 3
塗著 3
塗藥 9
塗說 0
塗過 2
塗過來 0
塗過去 0
塗銷 91
塗飾 2
塗鴉 439
塗鴨 91
塗黑 9
塘 598
塘塞 3
塘泥 0
塙 12
塙 12
塚 455
塛 3
塜 0
塝 4
塞 244
塞 244
塞 2449
塞上 13
塞住 32
塞入 47
塞內加爾 63
塞到 16
塞北 11
塞北高原 4
塞外 25
塞外風光 1
塞子 11
塞尚 22
塞拉利昂 0
塞普路斯 2
塞機 0
塞浦路斯 7
塞滿 51
塞爾維亞 36
塞班島 11
塞納河 30
塞翁失馬 11
塞翁失馬焉知非福 2
塞舌爾 0
塞藥 1
塞責 2
塞車 390
塞進 58
塞進來 0
塞進去 3
塞門 0
塟 0
塠 0
塠 0
塡 0
塢 146
塣 6
塤 16
塥 6
塦 0
塧 0
塨 6
塩 0
塩 0
塪 0
塪 0
填 20240
填上 197
填上來 0
填上去 1
填了 16
填些 1
填來填去 0
填個 10
填充 443
填充題 29
填入 2608
填到 3
填土 155
填土機 0
填在 15
填報 212
填塞 30
填字 4
填字遊戲 14
填實起來 0
填寫 10624
填平 19
填得 1
填成 3
填房 1
填料 60
填料函 1
填方 49
填有 4
填海造地 56
填滿 133
填的 36
填空 22
填空題 1
填著 0
填表 339
填表格 8
填補 218
填詞 46
填起來 2
填過 3
填過來 0
填過去 0
填隙 4
填飽 22
填鴨 16
填鴨式 12
填點 1
塬 0
塭 334
塯 21
塱 33
塲 0
塲 0
塳 0
塴 3
塵 1599
塵世 69
塵事 4
塵俗 16
塵凡 1
塵務 0
塵囂 67
塵土 57
塵土飛揚 19
塵土飛楊 0
塵垢 11
塵埃 167
塵埃落定 29
塵壤 0
塵寰 3
塵封 57
塵沙 21
塵沙惑 1
塵粒 18
塵緣 33
塵蟎 1
塵襟盡滌 0
塵霧 2
塶 225
塷 0
塹 293
塹壕 0
塺 4
塻 58
塼 1
塽 142
塾 105
塾師 2
塿 13
墀 32
墁 8
墂 16
境 3752
境內 948
境地 76
境域 8
境外 501
境況 28
境界 789
境管局 0
境遇 49
境過情遷 0
墄 0
墅 309
墆 4
墇 3
墈 10
墉 188
墊 922
墊上 13
墊上來 0
墊上去 0
墊上運動 4
墊下 6
墊下來 0
墊下去 0
墊了 4
墊付 51
墊個 0
墊個 0
墊入 0
墊出 0
墊出來 0
墊出去 0
墊回 0
墊回來 0
墊回去 0
墊圈 34
墊在 13
墊子 24
墊寫 0
墊平 0
墊底 23
墊支資本 0
墊板 27
墊檔 2
墊款 47
墊步 2
墊片 73
墊紙 2
墊肩 6
墊背 1
墊腳 5
墊腳石 16
墊著 8
墊被 2
墊補 0
墊褥 0
墊過 1
墊過來 0
墊過去 0
墊錢 2
墊鞋 1
墊高 32
墋 4
墌 0
墌 0
墍 0
墍 0
墎 40
墏 12
墐 3
墑 0
墑情 0
墒 0
墓 16000
墓園 178
墓地 82
墓室 9
墓志銘 5
墓木 0
墓木已拱 1
墓碑 67
墓碣 0
墓祭 0
墓穴 15
墓葬 28
墓誌 10
墓誌銘 20
墓道 3
墔 167
墕 0
墕 0
墖 0
増 0
墘 127
墚 0
墜 600
墜下 15
墜下來 0
墜下去 0
墜入 66
墜地 28
墜子 46
墜樓 36
墜機 56
墜機身亡 1
墜毀 80
墜落 164
墜落在 7
墜馬 0
墜體 1
墝 12
增 19478
增修 0
增值 203
增值稅 580
增光 7
增兵 8
增刊 12
增列 0
增刪 128
增刪額 0
增到 37
增加 15976
增加到 181
增加法 3
增加率 87
增加量 19
增塑劑 0
增壓 110
增多 350
增多到 0
增大 141
增幅 0
增廣 60
增廣見聞 46
增建 273
增強 1270
增強塑料 0
增強性 11
增援 39
增援部隊 4
增損 4
增撥 18
增收 23
增殖 159
增殖率 0
增氧機 2
增添 444
增添設備 2
增減 952
增減量 3
增減額 2
增班 82
增班數 0
增生 224
增產 230
增產報國 5
增產節約 0
增產量 0
增益 181
增給 10
增色 32
增補 91
增補本 0
增訂 386
增訂本 14
增設 1114
增資 1160
增進 2698
增進法 0
增量 28
增長 780
增長 780
增長到 7
增長率 23
增長速度 7
增音機 0
增高 197
墟 36
墠 6
墡 1
墢 0
墢 0
墣 648
墤 0
墥 4
墦 6
墧 0
墧 0
墧 0
墨 1259
墨吏 0
墨囊 0
墨國 72
墨子 61
墨守 5
墨守成規 21
墨客 12
墨家 16
墨寶 21
墨斗 3
墨晶 1
墨水 625
墨水匣 0
墨水瓶 0
墨汁 31
墨沈沈 0
墨爾本 246
墨爾缽 1
墨畫 18
墨盒 1
墨竹 13
墨筆 4
墨索裡尼 0
墨索里尼 12
墨經 1
墨綠 45
墨綠色 30
墨線 4
墨繩 0
墨翟 4
墨西哥 785
墨西哥人 6
墨西哥城 21
墨西哥市 11
墨西哥灣 10
墨西哥灣暖流 0
墨西哥灣流 1
墨跡 9
墨蹟 4
墨鏡 39
墨魚 16
墨鴉 0
墨鴨 0
墨黑 5
墩 464
墩子 0
墪 0
墫 10
墫 10
墬 38
墭 0
墮 132
墮 132
墮入 26
墮胎 151
墮胎藥 8
墮落 330
墯 7
墰 0
墱 9
墲 0
墲 0
墳 142
墳上 17
墳丘 17
墳典 0
墳地 9
墳場 17
墳墓 172
墳山 0
墳頭 17
墳頭 17
墴 0
墵 0
墷 0
墺 4
墻 70
墼 2
墽 5
墾 390
墾丁 1479
墾丁公園 22
墾區 5
墾地 12
墾復 0
墾殖 55
墾殖地 1
墾民 8
墾荒 20
墾荒地 3
墿 4
壀 0
壀 0
壀 0
壁 1429
壁上觀 5
壁報 121
壁報比賽 21
壁壘 19
壁壘分明 15
壁壘森嚴 0
壁櫥 8
壁毯 2
壁洞 0
壁燈 119
壁爐 17
壁玉 1
壁畫 153
壁磚 32
壁立 5
壁紙 134
壁虎 129
壂 8
壃 0
壄 0
壅 183
壅 183
壅塞 81
壅塞 81
壆 11
壇 955
壇台 0
壇城 1
壇場 1
壇場 1
壇子 1
壈 4
壉 4
壊 0
壊 0
壍 0
壎 150
壎 150
壏 43
壐 0
壐 0
壑 336
壒 27
壓 5359
壓下 86
壓下來 3
壓下去 2
壓下量 0
壓不住 3
壓低 146
壓住 50
壓倒 31
壓倒性 61
壓倒性多數 5
壓價 0
壓克力 402
壓克力板 18
壓出 21
壓出來 1
壓出去 0
壓到 37
壓制 170
壓力 4825
壓力強度 8
壓力機 5
壓力表 37
壓力計 42
壓力鍋 5
壓力降 9
壓勻 3
壓印 21
壓回 46
壓回來 1
壓回去 0
壓土機 0
壓在 94
壓垮 14
壓境 34
壓壓 11
壓夠 0
壓寨 1
壓寨夫人 0
壓寶 14
壓平 14
壓底 2
壓延 26
壓延機 8
壓強 7
壓得 44
壓扁 18
壓抑 347
壓抑著 9
壓搾 3
壓搾機 0
壓擠 18
壓服 3
壓根 13
壓根兒 9
壓根兒 9
壓條 21
壓條法 1
壓榨 55
壓榨器 0
壓榨機 1
壓歲 2
壓歲錢 19
壓氣 19
壓片 0
壓痛 30
壓的 97
壓盤石 0
壓直 5
壓破 5
壓碎 25
壓箱底 7
壓籐 0
壓緊 9
壓線 19
壓縮 2797
壓縮力 1
壓縮器 13
壓縮性 9
壓縮機 332
壓縮比 398
壓縮氧 0
壓縮空氣 58
壓花 55
壓花玻璃 1
壓著 101
壓蓋 6
壓起 0
壓起來 1
壓路 1
壓路機 35
壓軸 46
壓軸好戲 5
壓軸戲 3
壓近 1
壓迫 346
壓迫感 48
壓迫者 8
壓進 2
壓進來 0
壓進去 0
壓逼 3
壓邊 3
壓鑄 171
壓陣 5
壓隊 1
壓電 109
壓電性 2
壓頭 1
壓馬路 37
壓驚 5
壓齊 1
壔 45
壕 84
壕溝 26
壕溝戰 0
壖 64
壘 367
壘 367
壘上 13
壘上有人 2
壘上無人 0
壘塊 0
壘外 0
壘球 267
壘球員 0
壘球棒 1
壘球隊 93
壙 14
壚 11
壛 5
壜 0
壝 12
壞 2130
壞主意 1
壞了 362
壞事 126
壞人 362
壞分子 2
壞帳 0
壞得 2
壞心 16
壞心眼 3
壞心腸 0
壞惡 0
壞掉 100
壞東西 7
壞死 246
壞球 19
壞疽 26
壞的 168
壞得 168
壞良心 0
壞處 65
壞蛋 45
壞血病 18
壞血酸 1
壞話 57
壞透 5
壞運 5
壟 59
壟坎 0
壟斷 251
壟斷價格 0
壟斷利潤 0
壟斷市場 10
壟斷市場 10
壟斷性 15
壟斷組織 0
壟斷資本 0
壟斷資本主義 0
壟斷資產 0
壟斷資產階級 1
壟斷集團 0
壟畝 0
壟罩 1
壠 0
壠 0
壡 0
壢 154
壣 0
壤 214
壤土 17
壦 0
壦 0
壧 59
壨 179
壩 452
壩子 1
士 52404
士人 660
士兵 381
士別三日 0
士別多日 0
士卒 11
士大夫 36
士大夫 36
士女 7
士子 10
士官 216
士官兵 39
士官長 71
士徒 0
士敏土 0
士族 15
士林 1285
士林區 1131
士林國中 12
士林國小 26
士民 17
士氣 285
士為知己 2
士紳 41
士農工商 7
壬 412
壮 0
壯 2833
壯丁 25
壯健 4
壯圍 148
壯士 64
壯士斷腕 0
壯大 119
壯大聲勢 7
壯實 4
壯年 77
壯年人 10
壯年時期 5
壯志 39
壯志凌雲 2
壯志凌霄 0
壯志未酬 5
壯志未酬身先死 1
壯族 38
壯漢 27
壯潤 0
壯烈 32
壯烈成仁 0
壯烈犧牲 14
壯盛 10
壯碩 35
壯美 2
壯膽 14
壯舉 28
壯苗 1
壯觀 270
壯起來 0
壯錦 0
壯陽 1
壯陽藥 1
壯麗 148
声 0
売 0
壳 0
壳 0
壳 0
壴 4
壵 0
壷 0
壹 4596
壺 1081
壺中物 0
壺內 7
壺口 5
壺嘴 6
壺外 5
壺底 4
壺狀 0
壺盧 0
壺裡 5
壻 0
壼 57
壽 3472
壽儀 0
壽司 204
壽命 707
壽喜燒 1
壽堂 13
壽屏 1
壽山 306
壽山公園 6
壽州 3
壽數 2
壽星 101
壽桃 24
壽比南山 9
壽相 2
壽禮 3
壽糕 2
壽終 9
壽終正寢 14
壽考 0
壽者相 1
壽衣 7
壽誕 11
壽辰 8
壽陽 0
壽險 1122
壽麵 1
壾 52
壿 2
夂 0
夃 4
处 0
处 0
夅 0
夅 0
夆 145
夈 0
変 0
夊 0
夋 0
夌 10
复 214
夎 6
夏 37
夏 3789
夏代 2
夏令 22
夏令時間 12
夏令營 327
夏令衛生 0
夏商周 7
夏國 10
夏夜 74
夏天 866
夏天裡 8
夏威夷 760
夏威夷島 9
夏威夷州 10
夏威夷群島 16
夏娃 119
夏季 751
夏完淳 1
夏布 6
夏收 2
夏日 388
夏曆 0
夏服 4
夏朝 3
夏桀 1
夏正 1
夏歷 0
夏爐冬扇 0
夏眠 0
夏禹 4
夏禹治水 0
夏種 2
夏糧 0
夏至 28
夏衍 1
夏衣 2
夐 0
夐 0
夒 8
夓 0
夓 0
夔 939
夔夔 5
夔峽 1
夕 334
夕照 109
夕陽 371
夕陽工業 7
夕陽西下 29
夕陽餘暉 14
外 25091
外事 69
外事科 12
外事處 11
外交 2457
外交代表 7
外交使節 14
外交史 23
外交團 0
外交學 0
外交官 103
外交家 4
外交特權 5
外交界 11
外交系 44
外交部 1720
外交部長 167
外交關係 149
外人 480
外使 14
外來 712
外來語 13
外來風 0
外侮 11
外侵 2
外借 167
外側 195
外傘頂洲 0
外傳 486
外債 172
外傷 454
外僑 201
外公 85
外公切線 0
外凸 3
外出 392
外出服 4
外出血 6
外出裝 0
外分泌腺 8
外剛 2
外剛內柔 0
外力 154
外力作用 6
外功 4
外加 396
外務 73
外務員 21
外務省 75
外務部 3
外勞 0
外勤 69
外勤人員 11
外包 0
外匯 1823
外匯兌換 5
外匯兌換率 0
外匯存底 178
外匯局 21
外匯市場 12
外匯市場 123
外匯收入 39
外區 1
外卡 0
外史 23
外向 101
外商 1164
外喬 0
外因 34
外因論 0
外圈 32
外國 2660
外國人 977
外國佬 2
外國話 3
外國語 314
外國語文 114
外國貨 5
外圍 243
外圍地區 7
外圓內方 23
外在 592
外在美 13
外地 194
外地人 34
外型 607
外城 10
外埔 222
外埔 222
外埠 28
外堡 0
外場 14
外場 14
外塗 5
外壁 23
外壕 0
外太空 102
外夷 1
外套 246
外套腔 1
外套膜 0
外奶奶 0
外姓 3
外婆 94
外婆家 11
外子 51
外孫 30
外孫女 4
外孫女兒 0
外學 12
外家 10
外宿 52
外宿學生 3
外寄 14
外寇 1
外屋 2
外層 90
外層空間 1
外島 127
外島地區 12
外島部隊 1
外帳 13
外帶 59
外幣 457
外幣存款 106
外幣現金 1
外廠 15
外延 19
外弛內張 8
外弦 0
外張 10
外強 2
外強中乾 2
外形 245
外心 1
外快 63
外患 26
外患罪 20
外感 15
外戚 6
外打 4
外找 9
外抗 1
外抗強敵 0
外掛 250
外掛式 18
外接 520
外接圓 5
外援 24
外攻 0
外放 25
外敵 22
外敷 7
外文 873
外文系 524
外方內圓 3
外族 13
外族人 0
外星人 589
外景 46
外景隊 7
外有 81
外服 6
外架 4
外柔 0
外柔內剛 0
外校 79
外校學生 4
外樓 2
外欠 0
外段 1
外殼 593
外水 8
外沿 0
外洋 5
外洩 0
外流 131
外海 238
外海地區 0
外海地帶 0
外港 6
外漏 67
外焰 0
外燄 0
外營力 1
外片 19
外片報導 0
外環 75
外環道 0
外生 67
外甥 32
外甥女 9
外用 122
外用藥 24
外界 1340
外界壓力 6
外界影響 2
外痔 6
外皮 54
外相 102
外省 181
外省人 210
外祖 2
外祖母 11
外祖父 14
外祖父母 6
外科 3392
外科學 40
外科手術 191
外科醫生 18
外稿 4
外積 4
外穿 25
外突 4
外端 3
外籍 842
外籍人士 165
外線 144
外線投籃 0
外緣 20
外縣 11
外縣市 153
外耳 46
外聘 41
外聘制 0
外聽道 0
外胎 19
外胚層 4
外舅 0
外興安嶺 1
外艙 0
外蒙 26
外蒙古 30
外號 123
外行 44
外行人 37
外行話 4
外衣 98
外表 628
外表上 19
外袍 7
外補 5
外製 26
外褲 0
外親 1
外觀 1255
外角 9
外角球 2
外訪 8
外診 1
外診醫生 0
外語 1380
外語系 117
外調 17
外調制 0
外貌 130
外財 4
外貨 12
外貿 343
外貿協會 331
外資 1788
外賓 95
外賓來訪 981
外賣 1
外賬 0
外轉 15
外轉式 0
外逃 10
外遇 276
外道 178
外遣 0
外遷 0
外邊 31
外部 869
外鄉 7
外鄉人 7
外重內輕 0
外野 72
外野區 0
外野安打 0
外野手 48
外銷 3840
外銷品 20
外銷貨 0
外鋪 0
外鍋 6
外長 296
外門 17
外間 16
外防 0
外陰部 27
外雙溪 121
外電 121
外電報導 612
外電路 0
外露 60
外露 60
外面 1013
外項 0
外頭 118
外頭 118
外食 10
外館 67
夗 5
夘 0
夙 93
夙夜 1
夙夜匪懈 6
夙怨 1
夙昔 2
夙緣 0
夙興夜寐 1
夙願 15
多 403
多 40382
多上 50
多下 31
多下來 1
多不 95
多不多 11
多久 1066
多事 192
多事之秋 16
多交 34
多位 1024
多位數 1
多來 221
多倍體 3
多借 5
多倫多 294
多倫縣 0
多億 103
多元 1703
多元入學 0
多元化 1187
多元性 81
多元論 19
多元酯 13
多充 1
多再 5
多分 39
多分 39
多刊 2
多到 100
多刺 14
多刻 3
多劃 0
多功能 909
多加 602
多勞多得 1
多勸 1
多半 599
多半在 29
多半是 227
多半會 25
多去 47
多受 21
多叫 11
多台 82
多向 60
多咱 2
多哈 15
多員 0
多哥 6
多嘴 32
多嘴多舌 0
多回 16
多圈 3
多國 254
多國聯軍 0
多國部隊 4
多在 183
多多 2224
多多少少 95
多多益善 10
多大 536
多如牛毛 0
多媒體 13698
多子 6
多子多孫 5
多子多孫多福氣 1
多孔 74
多孔塑料 0
多字 42
多守 3
多客 1
多寡 422
多寡不拘 3
多少 4718
多少人 339
多少人 339
多少個 74
多少個 74
多少次 73
多少次 73
多少錢 318
多少錢 318
多就 52
多屆 8
多屬 116
多山 16
多島 5
多年 1854
多年不見 10
多年來 786
多年未見 3
多年生 80
多年生植物 3
多延 2
多彩 98
多彩多姿 193
多得 122
多得 122
多得不可計數 1
多得很 6
多得是 12
多德 4
多心 47
多情 265
多情種子 0
多想 83
多想 83
多愁 20
多愁善感 44
多意 0
多慮 27
多截 2
多所 243
多扇 1
多才多藝 49
多打 33
多抽 3
多拉 21
多拳 1
多接 7
多推 4
多搬 0
多摩教堂 0
多收 21
多放 25
多教 31
多數 1808
多數人 437
多數是 64
多數票 4
多斤 2
多方 342
多方阻擾 0
多方面 221
多於 233
多日 89
多日不見 4
多日未見 1
多明尼加 84
多是 331
多時 273
多普勒 2
多會 48
多會 48
多有 184
多期 15
多未 55
多束 1
多枚 15
多核 18
多棵 4
多樣 487
多樣化 649
多樣性 292
多次 1493
多此 5
多此一舉 27
多此一舉 27
多步 4
多歧亡羊 1
多比 7
多毛性 0
多民族國家 1
多氯 5
多氯 5
多氯聯苯 144
多氯聯苯 144
多汁 36
多沖 0
多油 5
多演 2
多災 2
多災多難 13
多為 572
多為 572
多無 16
多燒 1
多爾袞 2
多班 6
多瑙河 42
多瑪河 0
多瓣 2
多產 26
多產作家 1
多由 85
多畫 6
多當 1
多疑 30
多病 24
多瘦 2
多瘦 2
多發性 148
多的 2410
多目標 106
多相 11
多看 200
多看多聽 4
多砍 1
多票 34
多票通過 0
多禮 9
多科 17
多種 1881
多種多樣 9
多種經營 2
多端 29
多筆 75
多管 37
多管閒事 27
多管齊下 21
多箱 0
多節 13
多節式 1
多結 12
多維 26
多罐 0
多美 82
多美 82
多義 10
多義字 0
多義詞 0
多胎 5
多脂 2
多臂機 0
多臂機 0
多臭 1
多臭 1
多與 136
多舊 9
多舊 9
多花 145
多藏 3
多虧 35
多虧是 1
多行 34
多行 34
多行不義 2
多衝 3
多表 10
多補 5
多角 49
多角化 181
多角形 5
多解 2
多言 39
多言必失 0
多試 24
多話 57
多說 153
多說不如多做 0
多課 4
多請 13
多謀 2
多謀善斷 0
多謝 440
多變 284
多財善賈 0
多走 24
多起 108
多起來 9
多載 18
多輕 3
多輕 3
多輸 2
多辦 24
多近 1
多近 1
多退 3
多退少補 14
多進 40
多運 14
多遍 24
多過 32
多道 7
多達 308
多遠 99
多遠 99
多選 44
多選題 1
多邊 200
多邊形 123
多邊條約 0
多部 95
多醜 1
多醣 69
多采多姿 234
多重 912
多重人格 3
多重式 2
多重選擇 25
多重選擇題 26
多量 75
多金 16
多金之士 0
多長 80
多長 80
多開 28
多開 28
多關 17
多防 0
多階 34
多階段 8
多難 45
多難興邦 2
多雨 49
多雲 127
多雲偶雨 0
多雲有雨 0
多霧 5
多面 60
多面性 20
多面角 0
多面體 7
多音節 4
多項 1246
多項式 110
多頭 463
多頭政治 0
多頻 0
多顆 23
多養 6
多餘 369
多首 68
多香 1
多香 1
多高 74
多高 74
多麼 631
多麼 671
多麼地 130
多麼地 130
多點 293
多黨 24
多黨制 5
夛 0
夛 0
夜 4997
夜不閉戶 6
夜以繼日 18
夜作 0
夜來 39
夜來香 44
夜光 82
夜光杯 4
夜光漆 0
夜光質 0
夜光錶 26
夜勤 3
夜半 83
夜半鐘聲 7
夜叉 26
夜場 6
夜場 6
夜夜 148
夜夜笙歌 1
夜大 37
夜大學 0
夜大生 1
夜奔 1
夜宿 0
夜工 5
夜市 560
夜市場 2
夜市場 2
夜幕 27
夜幕低垂 31
夜幕沉沉 0
夜店 1
夜快 0
夜快車 2
夜戰 6
夜摩天 1
夜明 3
夜明星 0
夜明珠 6
夜晚 679
夜晚中 1
夜景 187
夜暗 2
夜暗風高 0
夜曲 29
夜有所夢 10
夜柔 2
夜柔吠陀 2
夜校 25
夜校生 61
夜歸 28
夜歸人 62
夜涼 3
夜涼如水 3
夜深 72
夜深人靜 53
夜燈 14
夜營 1
夜班 25
夜班車 1
夜生活 69
夜盲 2
夜盲症 13
夜禱 0
夜空 173
夜總會 114
夜航 20
夜色 122
夜色迷濛 0
夜行 209
夜行人 8
夜行軍 0
夜裡 322
夜襲 32
夜談 17
夜譚 2
夜警 2
夜讀 14
夜貓 164
夜貓子 13
夜貓族 24
夜路 18
夜車 14
夜遊 125
夜遊隊 0
夜郎自大 9
夜長 13
夜長夢多 4
夜間 1477
夜間班 56
夜間部 1424
夜闌 4
夜闌人靜 14
夜雨 37
夜靜更深 0
夜飯 1
夜鶚 0
夜鶯 14
夜鷹 23
夜黑 5
夜黑風高 10
夝 0
够 0
夠 5000
夠不夠 66
夠了 318
夠兇 1
夠出 5
夠到 13
夠剪 0
夠割 1
夠勁 7
夠勁 7
夠印 0
夠受 3
夠受的 1
夠吃 4
夠味 16
夠塗 0
夠多 112
夠大 92
夠好 63
夠學 1
夠尖 0
夠得上 3
夠急 0
夠打 9
夠撿 0
夠擠 0
夠數 3
夠早 3
夠朋友 9
夠本 8
夠格 34
夠氣 0
夠溼 0
夠濕 0
夠為 13
夠用 141
夠直 1
夠瞧 5
夠種 1
夠笨 3
夠糟 1
夠緊 2
夠花 5
夠草 0
夠買 13
夠近 4
夠重 9
夠面子 0
夠響 1
夠香 0
夠騎 0
夠點 1
夡 0
夢 5895
夢中 432
夢中人 33
夢中情人 1
夢兆 2
夢到 109
夢卜 0
夢囈 16
夢園 9
夢境 257
夢娜 24
夢寐 17
夢寐以求 53
夢寐難忘 0
夢幻 1511
夢幻仙境 5
夢幻泡影 17
夢想 4320
夢有 11
夢溪筆談 8
夢裡 195
夢見 178
夢言夢語 0
夢話 12
夢語 2
夢遊 88
夢遊症 1
夢過 3
夢遺 61
夢鄉 43
夢醒 69
夢露 17
夢魘 157
夣 0
夤 253
夤緣 0
夥 291
夥伴 1306
夥友 0
夥同 56
夥眾 0
夥計 37
夦 0
大 1
大 0
大 95677
大到 621
大一 615
大一統 28
大一統 28
大丈夫 75
大三 407
大不 61
大不 61
大不了 58
大不列顛 22
大不列顛 22
大不如前 30
大不敬 5
大不相同 97
大世界 52
大世紀 72
大中 249
大中小 32
大中至正 7
大串 34
大主教 7
大乘 401
大乘佛教 1
大亂 83
大事 2904
大事不妙 0
大事化小 3
大事宣傳 0
大事舖張 0
大事鋪張 0
大二 437
大亨 277
大人 1111
大人物 90
大仁 378
大仁科技大學 1
大仁藥專 94
大仙 27
大仲馬 6
大件 25
大伙 147
大伙兒 41
大伙兒 41
大伯 17
大伯子 0
大伯子 0
大伯母 0
大伯父 0
大位 8
大佐 3
大作 321
大使 446
大使級 4
大使館 252
大便 589
大俠 231
大修 146
大修理 2
大修理基金 0
大個 17
大個兒 8
大個兒 8
大個子 10
大做文章 2
大傢伙兒 0
大傢伙兒 0
大傳 0
大傷 122
大傷元氣 0
大傷腦筋 0
大儒 13
大元帥 6
大兄 2
大內 234
大公 104
大公國 17
大公報 56
大公無私 11
大兵 228
大兵日記 4
大典 132
大冶 12
大凡 29
大凶 19
大刀 80
大刀闊斧 36
大分 66
大分 66
大分子 34
大分子 34
大刑 6
大別 7
大別山 1
大前天 4
大前年 0
大前提 30
大副 68
大力 1004
大力士 43
大功 66
大功告成 81
大功率 40
大勇 159
大動 51
大動干戈 3
大動脈 31
大動脈 31
大勝 166
大勢 140
大勢已去 6
大勢已定 0
大勢所趨 45
大勢至菩薩 1
大包 114
大化 86
大化革新 1
大北 43
大匙 162
大匙 162
大區 21
大千 1051
大千世界 91
大半 237
大半午 0
大半天 10
大半生 12
大協作 0
大南 85
大南客運 0
大卡 61
大印 13
大卷 71
大叔 40
大叔 40
大受 98
大受歡迎 57
大叫 255
大可 110
大可不必 41
大可以 35
大台北 324
大台北區 33
大台北地區 307
大司馬 3
大吃 32
大吃一頓 6
大吃一頓 6
大吃一驚 56
大吃一驚 56
大吃大喝 17
大合唱 24
大吉 142
大吉大利 13
大同 3058
大同世界 10
大同區 393
大同商專 108
大同國中 20
大同大學 1
大同小異 90
大同市 2
大同書 0
大同盆地 1
大同社會 0
大同縣 0
大名 4438
大名府 8
大名鼎鼎 19
大吏 2
大吐苦水 4
大君 22
大吵 11
大吵一架 8
大吵一架 8
大吵大鬧 7
大吹 12
大吹大擂 3
大呂 7
大呼小叫 0
大命 2
大和 294
大咬一口 1
大咬一口 1
大哉問 0
大員 20
大哥 1536
大哥大 1720
大唐 125
大唐帝國 1
大唱 10
大商人 2
大喊 118
大喊大叫 2
大喜 172
大喜過望 2
大喝一口 0
大喝一口 0
大喝一聲 21
大喝一聲 21
大嘴巴 0
大器 26
大器小用 0
大器晚成 7
大嚷 10
大嚼 9
大嚼 9
大四 348
大圈 17
大國 228
大國家 6
大國民 25
大園 796
大圓 58
大圓鏡智 1
大團 27
大團圓 25
大土 25
大地 2052
大地主 16
大地回春 1
大地春回 0
大型 4601
大型企業 246
大城 174
大埔 314
大埔 314
大堂 48
大堂哥 0
大堂妹 0
大堂姊 0
大堂弟 0
大堆 24
大堆頭 5
大堆頭 5
大場 7
大場 7
大塊 117
大塊頭 6
大壩 120
大士 1
大多 1698
大多在 49
大多數 1019
大多是 302
大多會 25
大多有 21
大多能 19
大夜班 21
大夢初醒 12
大夥 263
大夥兒 112
大夥兒 112
大大 680
大大小小 47
大大方方 15
大大的 253
大大地 253
大夫 221
大夫 221
大失所望 24
大好 150
大好人 13
大好時光 2
大好時機 18
大好機會 44
大妹 9
大妹夫 0
大姆指 34
大姆指頭 1
大姊 163
大姊夫 1
大姊姊 49
大姊姊 49
大姐 228
大姑 8
大姑丈 0
大姑娘 12
大姑媽 1
大姑子 0
大姓 5
大姨 4
大姨媽 1
大姨子 1
大威德 1
大威德明王 1
大威德金剛 1
大威德金剛明王 1
大娘 35
大婚 1
大媒 4
大媽 33
大嫂 68
大嫂子 0
大嬸 7
大嬸兒 0
大嬸兒 0
大嬸婆 0
大字 98
大字報 22
大季 0
大孩子 13
大學 35747
大學城 116
大學士 7
大學校院 1
大學生 713
大學部 1548
大學院校 1
大安 1057
大安區 1173
大安國中 70
大安地列 1
大安溪 56
大宗 345
大官 89
大官人 22
大定 18
大客車 75
大宴賓客 0
大家 28971
大家庭 187
大家樂 252
大家閨秀 4
大富大貴 12
大寒 9
大寨 17
大寫 136
大寫字母 25
大寮 852
大封 21
大將 187
大將之風 4
大將軍 55
大專 3362
大專同學 7
大專學生 171
大專院校 1413
大尉 2
大小 7522
大小不一 55
大小便 71
大小姐 142
大小通吃 19
大少爺 13
大局 115
大展 743
大展宏圖 1
大展身手 0
大展鴻圖 19
大屠殺 97
大屯 164
大屯山 82
大山 242
大峽谷 102
大川 108
大差 5
大差差 0
大巴山 3
大巷 0
大帝 198
大帥 53
大師 2710
大師傅 10
大師父 3
大師級 45
大帽子 12
大幅 2882
大幅度 181
大平 55
大平 55
大年 43
大年初一 26
大年夜 1
大幸 6
大幹 3
大幹一場 0
大幹一場 0
大幹一場 0
大幹一場 0
大幹一票 1
大幹一票 1
大店 44
大度路 26
大座 5
大庭廣眾 11
大廈 1033
大廈 1033
大廈將傾 1
大廈棟梁 1
大廚 24
大廟 59
大廟 59
大廠 924
大廳 595
大建 25
大弟 15
大張 77
大張旗鼓 21
大後天 10
大後年 1
大後方 6
大得 79
大循環 7
大德 297
大徹大悟 8
大志 40
大忙 16
大快人心 18
大快朵頤 29
大恩 49
大恩大德 18
大悟 24
大患 13
大悲 123
大悲咒 40
大惑不解 17
大意 239
大慈大悲 46
大慶 310
大成 9638
大我 73
大截 20
大戰 605
大戲 106
大戶 138
大戶人家 13
大房 20
大扇 0
大手 96
大手筆 89
大才小用 1
大打出手 38
大打折扣 66
大批 415
大批湧到 1
大批的 61
大把 78
大抵 105
大拇哥 0
大拇指 51
大拍賣 62
大拜拜 45
大括號 9
大括號 9
大指 7
大振 55
大捷 31
大掃 9
大掃除 66
大排場 0
大排場 0
大排長龍 23
大提琴 306
大搖大擺 23
大搞 30
大擺 8
大改 47
大放厥詞 10
大放厥辭 0
大放異彩 56
大敗 54
大教堂 90
大敵 37
大敵當前 7
大數 28
大數目 3
大料 1
大新 317
大方 607
大方向 50
大於 909
大族 14
大日如來 1
大易 23
大易中文 0
大易輸入法 23
大星 19
大春 70
大昭寺 1
大時代 71
大智 264
大智大愚 0
大智若愚 10
大暑 5
大曲 10
大曲 20
大曲酒 0
大曲酒 0
大書 19
大書特書 10
大曼陀羅 1
大會 2730
大會堂 62
大會操 4
大會舞 4
大月 11
大月氏 0
大有 448
大有人在 31
大有作為 5
大有分別 1
大有可為 49
大有可觀 0
大有巴士 6
大有文章 3
大有為 21
大未必佳 4
大本 50
大本營 115
大材小用 4
大村 331
大杯 49
大東 190
大東北 1
大林 1099
大枝 2
大柱 11
大柱子 0
大校 108
大桶 20
大梁 18
大棒 16
大棵 1
大楊樹 1
大業 319
大業路 88
大楷 6
大概 2219
大概是 676
大概會 100
大概有 104
大槍 3
大樁 4
大樑 16
大樓 6873
大樓住戶 12
大模大樣 4
大樹 661
大權 81
大權在握 5
大權旁落 2
大權獨攬 0
大步 235
大武山 71
大武溪 2
大歷 1
大殮 1
大段 62
大殿 84
大比 8
大毛 28
大氅 0
大民主 0
大氣 1796
大氣團 0
大氣壓 38
大氣壓力 55
大氣層 125
大水 148
大汗 55
大江 86
大江南北 0
大江東去 12
大河 126
大河文化 0
大油 46
大治 4
大治山 0
大沽 9
大泉眼 0
大法 214
大法官 632
大法師 49
大洋 252
大洋洲 194
大洞 37
大洲 344
大流 12
大流士 2
大海 362
大海撈針 17
大海洋 17
大清早 31
大清水溪 0
大渡河 3
大港 159
大湖 745
大湖山莊 31
大溪 481
大溪地 47
大溪港 1
大溪鎮 585
大滷麵 3
大滿貫 0
大漠 36
大漢 627
大漢工商 98
大漢技術學院 1
大漢族 0
大漢族主義 0
大漢民族 2
大漲 427
大漲大跌 3
大潮 45
大火 376
大災難 67
大炒 7
大炮 19
大為 441
大無畏 8
大無畏精神 3
大煙 2
大熊 1
大熊座 18
大熊星座 2
大熊貓 6
大燈 54
大爆冷門 0
大父 0
大爺 117
大牆 3
大片 131
大牌 137
大牙 5
大牢 7
大獎 504
大獲全勝 27
大率 2
大王 348
大王爺 1
大玩 31
大玩特玩 3
大班 90
大班制 0
大球 15
大理 343
大理寺 1
大理石 447
大理院 0
大瓶 9
大瓶的 0
大用 27
大田 52
大甲 579
大甲溪 173
大甲鎮 1545
大男人 66
大男人主義 17
大略 193
大略在 0
大略是 2
大略有 2
大異其趣 31
大當 6
大疊 7
大病 39
大病初癒 0
大發 318
大發利市 30
大發慈悲 0
大發雷霆 14
大白 95
大白天 10
大白菜 21
大白鼠 88
大百科全書 14
大的 2740
大盒 9
大盜 620
大盤 1445
大盤商 55
大直 351
大直國中 13
大直國小 12
大直國民中學 1
大直國民小學 1
大相徑庭 1
大相逕庭 20
大真 6
大眼瞪小眼 2
大眾 4764
大眾傳播 636
大眾化 129
大眾捷運 259
大眾語 1
大石 88
大石頭 42
大石頭 42
大砲 69
大破 41
大碗 187
大神 61
大票 36
大票的 4
大祭司 38
大禍 29
大禍臨頭 6
大禮 114
大禮堂 207
大禹 60
大禹嶺 60
大禹治水 16
大秦 23
大站 120
大端 4
大竹 350
大竹溪 0
大笑 204
大笑一場 2
大笑一場 2
大笑一場 2
大笑一場 2
大笑一聲 2
大笑一聲 2
大笑金剛 1
大笑金剛明王 1
大筆 222
大筆的 45
大管 88
大箱 11
大節 9
大篆 18
大篇 27
大篇幅 22
大籃 1
大米 33
大粒 1003
大糞 3
大系 93
大約 1643
大約是 202
大約有 235
大紅 108
大紅大紫 6
大紅大綠 0
大紅花 2
大統 297
大經 29
大綱 2196
大線 7
大縣 37
大總統 23
大纛 1
大缸 7
大罐 8
大罵 86
大羅神仙 0
大群 147
大群人馬 2
大義 112
大義凜然 2
大義滅親 5
大老 295
大老官 2
大老爺 6
大老級 4
大考 86
大考大玩 0
大而化之 9
大而無當 7
大聖 195
大聲 743
大聲嚷嚷 13
大聲疾呼 36
大肆 180
大肆咆哮 0
大肆宣揚 20
大肚 612
大肚子 18
大肚子 18
大肚溪 33
大肚魚 1
大股 10
大胃王 0
大腦 354
大腦皮層 7
大腦皮質 25
大腳 253
大腳姆趾 0
大腸 518
大腸桿菌 161
大腸菌 42
大腹便便 5
大腿 309
大腿肉 1
大腿骨 11
大膽 486
大臉 7
大臣 216
大自然 1264
大致 1570
大致上 248
大臼齒 0
大舅 11
大舅媽 1
大舅子 3
大興 277
大興土木 45
大興安嶺 18
大興水利 1
大舉 273
大舉進攻 7
大舉進犯 0
大舌頭 1
大船 49
大艘 0
大花曼陀羅 1
大花臉 3
大英 196
大英國協 43
大英帝國 17
大英百科全書 57
大草原 99
大荒 37
大菜 11
大華 995
大華中學 7
大華工專 6
大華技術學院 1
大華晚報 3
大葉大學 1
大葉桉 2
大蒜 121
大蓋 8
大蔥 6
大藏經 126
大蘇打 7
大處 14
大號 31
大蛇 19
大蝦 37
大蟲 21
大行 91
大行政區 2
大行星 13
大街 146
大街小巷 88
大衛 830
大衛王 33
大衣 73
大表哥 5
大表妹 0
大表姊 1
大表弟 0
大表驚嘆 0
大衿衫 0
大袋 4
大袋的 4
大補帖 0
大褂 1
大西 77
大西洋 250
大西洋城 0
大要 47
大規模 537
大規模集成電路 5
大觀 633
大觀園 561
大角 13
大解 11
大言不慚 27
大計 311
大詞典 11
大話 41
大誇 1
大誥 1
大調 138
大談 67
大謬 5
大變 34
大豆 196
大豐 236
大豐收 41
大象 315
大費周章 10
大資產階級 0
大賠 7
大賢 33
大賭 3
大賺 23
大賺一票 2
大賺一票 2
大贏 40
大赤色 0
大赦 25
大起來 3
大起大落 26
大跌 252
大跨一步 0
大跨一步 0
大路 441
大踏步 12
大躍進 64
大車 67
大軍 160
大軍進犯 4
大輪金剛 1
大輪金剛明王 1
大輸 6
大輸一場 0
大輸一場 0
大輸一場 0
大輸一場 0
大轉 9
大轉彎 16
大辦 7
大辯論 84
大逆 4
大逆不道 4
大通 165
大連 262
大連人 0
大連市 24
大連港 0
大運河 12
大遍 5
大過 75
大道 620
大道理 26
大選 976
大選舉 3
大邊 12
大邱 18
大部 979
大部分 1230
大都 723
大都 723
大都市 1
大都是 206
大都會 1807
大里 928
大野 38
大量 4005
大量生產 109
大金 79
大鈔 15
大鉗 1
大錘 3
大錢 102
大錯 30
大錯特錯 13
大鍋 39
大鍋飯 7
大鍵琴 19
大鎚 2
大長今 18
大門 850
大門口 106
大門外 15
大開 127
大開倒車 0
大開殺戒 0
大開眼界 59
大間 10
大閘蟹 0
大閨女 0
大關 131
大阪 620
大阿哥 5
大阿哥 5
大限 61
大限已至 1
大院 36
大院君 1
大陸 23271
大陸人 88
大陸型 7
大陸妹 1
大陸委員會 1
大陸性 9
大陸政策 294
大陸架 1
大陸棚 19
大陸貨 3
大隊 514
大隊人馬 17
大隊接力 0
大險 3
大隻 24
大雁 20
大雄 172
大雄寶殿 2
大雅 751
大雜燴 14
大雜院 12
大難 18
大難不死 14
大難臨頭 7
大雨 300
大雨傾盆 3
大雨如注 0
大雪 82
大雪山 120
大雪紛飛 10
大震 59
大霧 18
大霸尖山 69
大霹靂 34
大青 23
大靜脈 6
大靜脈 6
大面 12
大革命 35
大韓民國 20
大音階 1
大響 8
大頂 8
大項 109
大順 228
大頭 562
大頭照 10
大頭菜 7
大頭貼 1
大頭釘 2
大頭針 31
大頭鬼 26
大顆 13
大題 17
大題目 3
大顎 1
大類 546
大顯 19
大顯神威 0
大顯神通 10
大顯身手 24
大風 239
大風大浪 12
大風雨 1
大風雪 6
大食團 500
大食帝國 0
大餅 159
大馬士革 11
大駕 21
大駕光臨 45
大驚一場 0
大驚一場 0
大驚一場 0
大驚一場 0
大驚失色 11
大驚小怪 31
大體 227
大體上 120
大鬍子 17
大鬧 59
大鬧一場 1
大鬧一場 1
大鬧一場 1
大鬧一場 1
大魚 95
大魚吃小魚 4
大魚大肉 10
大鵬金翅鳥 3
大鵬鳥 3
大麥 123
大麴 3
大麴酒 1
大麻 61
大麻煙 6
大麻里 2
大黃 50
大黃蜂 46
大黃魚 1
大點 23
大黨 74
大鼓 54
大龍 129
大龍峒 14
大龍街 16
大龍郵票 4
夨 0
天 31487
天一 119
天上 719
天上聖母 25
天下 3648
天下一家 9
天下一家 9
天下人耳目 0
天下大亂 19
天下太平 10
天下本無事 3
天下沒有 44
天下沒有白吃的午餐 17
天下為公 15
天下為家 0
天下無敵 31
天下無雙 12
天下無難事 12
天下雜誌 468
天不從人 0
天不從人願 12
天不怕 0
天不怕地不怕 8
天主 498
天主堂 98
天主教 963
天主教堂 31
天主教徒 64
天主教會 58
天主教派 0
天之驕子 26
天乾物燥 3
天井 55
天京 4
天亮 116
天人 155
天人之分 2
天人之際 4
天人交戰 0
天人合一 40
天人永別 0
天人永隔 0
天人菊 1
天仙 37
天份 86
天佑 67
天作之合 8
天使 1899
天保 14
天保九如 0
天候 208
天候不佳 21
天倫 56
天倫之樂 8
天倫樂 9
天價 10
天元 42
天光 78
天全 16
天公 145
天公不作美 13
天公地道 1
天兵 34
天兵神將 0
天分 31
天半 43
天南地北 42
天台 234
天各一方 3
天各一方 3
天后 203
天后宮 1
天呀 98
天呀 98
天命 93
天命難違 1
天和 100
天哪 87
天國 227
天地 12926
天地不容 1
天地會 10
天地良心 2
天地萬物 32
天地間 79
天堂 2487
天堂門 3
天堂鳥 68
天塹 0
天壇 39
天壤 4
天壤之別 50
天外有天 12
天大 110
天天 1060
天契 0
天奪之魄 0
天女 31
天女散花 8
天姿 4
天姿國色 1
天婦羅 0
天子 192
天字 13
天字第一 0
天字第一號 4
天安門 73
天安門事件 52
天安門廣場 24
天安門廣場 24
天完 0
天官賜福 11
天宮 227
天寒 21
天寒地凍 7
天寶 45
天山 285
天山南路 2
天山山脈 1
天山山脈 1
天崩地坼 0
天崩地裂 25
天工 12
天工開物 7
天差地遠 6
天師 24
天師教 24
天師道 24
天幕 20
天干 24
天干地支 11
天平 234
天平座 35
天年 36
天底 10
天底下 26
天府 529
天府之國 6
天庭 14
天庭飽滿 1
天弓 21
天後 184
天從人願 4
天怒人怨 1
天性 219
天意 69
天懸地隔 0
天成 125
天才 1216
天才兒童 8
天才論 0
天打雷劈 6
天授 3
天搖地動 6
天擇 16
天放 40
天敵 74
天數 534
天文 2452
天文台 289
天文單位 14
天文學 311
天文學家 195
天文數字 22
天文望遠鏡 101
天文臺 29
天文館 61
天方夜譚 53
天旋地轉 50
天日 17
天旱 1
天明 124
天昏地暗 17
天時 83
天時地利 10
天時地利人和 6
天晴 40
天暗 2
天曉得 20
天書 52
天會 105
天有不測風雲 5
天朝 33
天柱 22
天棚 3
天樂 1
天樂義 1
天樞 30
天橋 66
天機 137
天機不可洩露 1
天權 6
天正 7
天殺的 8
天殺地 8
天母 975
天氣 1700
天氣圖 22
天氣形勢 0
天氣預報 473
天水 37
天水街 0
天池 95
天河 76
天泉 8
天波 0
天波 0
天津 416
天津人 2
天津市 100
天涯 613
天涯地角 0
天涯海角 60
天涯若比鄰 74
天淨沙 50
天淵 5
天淵之別 5
天溝 4
天演 6
天演論 3
天潢貴冑 0
天災 139
天災人禍 45
天無二日 0
天無絕人 0
天無絕人之路 9
天然 2061
天然更新 1
天然林 68
天然樹脂 0
天然橡膠 34
天然氣 430
天然氣井 1
天然淘汰 0
天然漆 30
天然煤氣 0
天然物 71
天然瓦斯 19
天然痘 0
天然絲 1
天然纖維 43
天然資源 90
天然雜交 1
天然餌料 0
天然鹼 0
天燈 0
天父 124
天牛 26
天牢 0
天物 3
天狗 22
天狼星 47
天王 1138
天王星 128
天球 45
天球儀 1
天理 94
天理不容 3
天理人情 0
天理何在 11
天理循環 1
天理教 6
天理昭彰 1
天理良心 1
天理難容 4
天琴座 17
天璇 2
天璣 136
天生 692
天生一對 32
天生一對 32
天生一對寶 0
天生一對寶 0
天生地設 0
天生麗質 17
天界 1
天癸 1
天皇 172
天目山 3
天真 348
天真無邪 44
天真爛漫 9
天眼 59
天眼通 1
天祐 25
天祚帝 0
天神 183
天祥 323
天秤 1
天秤座 587
天秤盤 0
天空 3361
天空中 75
天窗 245
天竺 50
天竺國 2
天竺葵 27
天竺鼠 40
天籟 87
天籟之音 90
天經地義 51
天網恢恢 2
天線 619
天緣巧合 0
天緣湊巧 0
天縱 11
天縱英明 0
天羅地網 101
天翻地覆 26
天耳通 1
天聲 9
天職 27
天與人歸 0
天良 13
天色 145
天色已晚 7
天花 114
天花亂墜 13
天花板 549
天荒地老 14
天葬 6
天藍 59
天藍色 10
天蠍 244
天蠍座 675
天蠍星 24
天衣無縫 32
天誅 3
天誅地滅 3
天誘其衷 0
天說 12
天論 12
天譴 19
天變 9
天象 128
天象儀 2
天資 58
天資過人 0
天賜 77
天賜 77
天賜良緣 4
天賜良緣 4
天賦 134
天賦人權 8
天足 2
天車 56
天造地設 5
天道 162
天邊 72
天邊海角 3
天長 23
天長地久 124
天長日久 0
天門 27
天開 18
天際 183
天險 15
天雄 7
天雷 65
天電 0
天靈蓋 5
天青 18
天頂 0
天順 14
天頭 2
天香國色 0
天馬 61
天馬行空 83
天體 355
天體力學 6
天體營 21
天高地厚 13
天高皇帝遠 8
天高聽卑 0
天鵝 214
天鵝湖 59
天鵝絨 14
天鵝肉 3
天黑 69
天黑黑 13
天龍 274
天龍八部 0
天龍地虎 0
天龍座 15
天龍隊 0
太 14141
太上皇 8
太上老君 6
太不 159
太不像話 3
太乙 124
太低 127
太保 343
太倉 33
太倉稊米 0
太假 7
太傅 21
太公 101
太厚 61
太原 87
太原市 8
太原路 69
太古 82
太古時代 1
太后 60
太君 6
太和 102
太圓 0
太多 2905
太大 1001
太太 1323
太夫人 9
太好 202
太好了 245
太妹 10
太婆 2
太子 635
太子妃 3
太子洗馬 0
太學 39
太守 81
太宗 50
太宰 7
太尉 58
太少 271
太尖 2
太師 82
太師椅 8
太平 1853
太平國小 75
太平天國 44
太平天國革命 0
太平夫人 0
太平山 207
太平廣記 0
太平梯 9
太平洋 2552
太平洋區 24
太平洋戰爭 36
太平洋文化基金會 31
太平盛世 13
太平軍 4
太平道 0
太平門 3
太平間 10
太座 15
太康 46
太廟 0
太弱 36
太強 86
太彎 0
太忙 97
太快 238
太急 21
太息 0
太慢 202
太早 173
太晚 156
太會 75
太有 77
太極 388
太極圖 9
太極拳 341
太武 1
太武山 1
太歲 41
太歲爺 0
太歲爺上動土 0
太歲頭上 2
太歲頭上動土 2
太油 8
太淺 18
太湖 90
太湖石 2
太溼 3
太濕 2
太為 4
太爺 32
太牢 3
太甚 1
太甲 4
太白 29
太白星 0
太白粉 78
太監 49
太直 6
太短 69
太祖 28
太祝 1
太空 2239
太空乘具 0
太空人 145
太空時代 9
太空梭 109
太空機 1
太空站 54
太空統署 3
太空船 189
太空艙 7
太空衣 5
太空被 0
太空裝 15
太緊 65
太美 56
太老爺 0
太腫 0
太苦 13
太虛 151
太虛幻境 4
太行 1
太行山 1
太近 45
太過 623
太過分 62
太遠 158
太遲 61
太醫 11
太醫院 4
太重 133
太長 161
太閒 29
太阿倒持 0
太陡 6
太陰 43
太陰曆 1
太陽 2737
太陽光 60
太陽升 5
太陽年 0
太陽旗 6
太陽日 13
太陽曆 6
太陽活動 5
太陽熱 3
太陽燈 15
太陽眼鏡 215
太陽神 77
太陽穴 27
太陽系 255
太陽能 428
太陽能板 10
太陽谷 8
太陽輻射 65
太陽鏡 0
太陽電池 57
太陽黑子 76
太難 139
太飽 16
太高 315
太魯閣 668
太麻里 0
太麻里溪 0
太鼓 1
夫 398
夫 3980
夫人 1547
夫倡婦隨 0
夫君 24
夫唱婦隨 6
夫妻 1252
夫妻之情 3
夫妻倆 37
夫妻反目 1
夫妻情 0
夫妻相 0
夫姓 10
夫婦 810
夫婦倆 25
夫婦關係 1
夫婿 53
夫子 243
夫子自道 0
夫家 15
夫家人 0
夫差 11
夫權 2
夫死從子 0
夫琅和費 0
夫職 1
夬 7
夭 90
夭 90
夭亡 0
夭壽 16
夭壽 16
夭夭 5
夭折 28
夭折 28
夭歿 0
夭歿 0
夭逝 4
夭逝 4
央 4368
央及 120
央告 1
央圖 0
央懇 0
央求 29
央行 2650
央請 12
夯 45
夰 0
失 3090
失主 15
失之交臂 8
失之交臂 8
失之東隅 2
失之毫釐 3
失事 291
失信 16
失信於人 0
失修 3
失傳 69
失儀 3
失利 248
失勢 12
失卻 13
失去 2330
失和 40
失土 8
失地 5
失婚 10
失學 57
失守 99
失官 1
失察 2
失實 4
失寵 17
失常 117
失序 61
失度 1
失德 2
失怙 6
失恃 0
失意 81
失態 28
失慎 0
失戀 421
失戀者 2
失手 100
失掉 50
失控 158
失措 35
失據 0
失效 288
失敗 1555
失敗主義 8
失敗者 44
失敗論 1
失敗論者 0
失散 31
失散多年 87
失敬 18
失於 6
失明 112
失時 16
失望 878
失望絕頂 0
失望透頂 18
失查 1
失格 0
失業 949
失業率 414
失業者 69
失檢 8
失款 0
失水 7
失法 2
失火 101
失物 13
失物招領 22
失當 53
失盜 0
失真 185
失眠 348
失眠症 25
失神 66
失禁 126
失禮 31
失竊 188
失笑 15
失笑起來 0
失策 16
失算 18
失節 6
失約 20
失而復得 7
失聰 21
失聲 77
失職 86
失職人員 26
失色 72
失落 526
失落感 35
失血 46
失血過多 8
失言 27
失計 1
失語症 27
失誤 247
失調 260
失足 31
失蹤 664
失蹤人口 158
失身 21
失迎 3
失速 41
失重 9
失陪 8
失陷 9
失靈 102
失音 2
失風 1
失風被捕 0
失魂 5
失魂落魄 12
失點 0
夲 0
夲 0
夳 0
头 0
头 0
夵 0
夵 0
夶 0
夶 0
夶 0
夷 339
夷人 7
夷為平地 13
夸 71
夸克 0
夸張 10
夸父 0
夸爾 1
夸特 2
夸獎 6
夸誕 0
夼 142
夽 0
夾 28
夾 283
夾 2832
夾七夾八 1
夾上 16
夾上來 0
夾上去 0
夾下 8
夾下來 0
夾下去 0
夾了 8
夾些 0
夾人 1
夾住 49
夾來夾去 0
夾個 0
夾克 139
夾入 5
夾具 78
夾到 7
夾在 114
夾夠 0
夾好 3
夾子 67
夾完 0
夾層 303
夾巷 0
夾帶 112
夾帶走私 0
夾心 32
夾心餅乾 9
夾手 1
夾持 53
夾擊 52
夾攻 22
夾板 95
夾板船 0
夾板船 0
夾檔 0
夾殺 25
夾法 0
夾注 1
夾注號 0
夾生 0
夾竹桃 20
夾竹桃 20
夾給 0
夾緊 24
夾縫 34
夾縫 34
夾著 59
夾著尾巴 5
夾著尾巴 5
夾著尾巴 5
夾著尾巴 5
夾衣 0
夾角 40
夾角 40
夾走 0
夾起 5
夾起來 3
夾送 4
夾進來 0
夾進去 0
夾過 0
夾過來 0
夾過去 0
夾道 23
夾道歡迎 21
夾道歡迎 21
夾道相迎 0
夾道相迎 0
夾雜 101
夾雜在 12
夾雜著 57
夾頭 105
夾頭 105
夾點 1
夿 0
奀 1318
奃 0
奄 48
奄 48
奄奄 12
奄奄一息 46
奄奄一息 46
奄奄不振 0
奄奄待斃 3
奅 3
奆 0
奇 2729
奇 27294
奇事 790
奇人 108
奇偉 5
奇偶 13
奇兵 388
奇冷 4
奇冷無比 0
奇功 14
奇名 4
奇境 5
奇奇 77
奇奇怪怪 45
奇妙 598
奇寒徹骨 0
奇峰 62
奇巖 1
奇巧 10
奇形怪狀 20
奇怪 1950
奇恥大辱 1
奇情 12
奇情片 0
奇想 82
奇慘無比 2
奇才 58
奇技 5
奇摩 0
奇效 12
奇數 42
奇文 42
奇景 192
奇案 39
奇熱 0
奇熱無比 0
奇特 284
奇特性 0
奇異 512
奇異果 1
奇異筆 1
奇痛 2
奇痛無比 1
奇癢 9
奇白 0
奇白無比 0
奇硬 0
奇硬無比 0
奇禍 1
奇秘 0
奇緣 274
奇聞 110
奇花異草 8
奇萊 107
奇萊山 21
奇葩 41
奇行 0
奇術 5
奇裝異服 9
奇襲 25
奇襲戰術 2
奇觀 160
奇計 6
奇談 19
奇談怪論 1
奇謀 25
奇譚 12
奇貨可居 9
奇趣 13
奇跡 45
奇蹄 0
奇蹟 638
奇蹟式 13
奇軟無比 0
奇遇 57
奇遇記 34
奇醜 0
奇醜無比 1
奇門遁甲 108
奇零 0
奇風異俗 8
奇麗 11
奇黑 0
奇黑無比 0
奇幻 600
奇幻小說 600
奇幻文學 600
奈 144
奈 1440
奈何 108
奈及利亞 113
奈米 1
奉 2114
奉上 50
奉令 97
奉令承教 0
奉侍 1
奉公 4
奉公守法 27
奉公職守 0
奉到 1
奉勸 194
奉化縣 1
奉召 6
奉告 12
奉命 143
奉命行事 2
奉天 81
奉天省 0
奉天省長 0
奉守 1
奉安 9
奉承 27
奉承話 0
奉旨 14
奉此 3
奉派 74
奉為 42
奉為圭臬 6
奉獻 859
奉祀 81
奉節 2
奉系 0
奉系軍閥 0
奉若 0
奉若神明 1
奉行 102
奉贈 3
奉趙 0
奉軍 1
奉迎 4
奉送 26
奉還 12
奉陪 32
奉養 24
奉養父母 2
奊 2
奋 0
奋 0
奌 0
奎 1044
奎寧 18
奏 855
奏了 2
奏事官 0
奏凱 1
奏凱歌 5
奏凱言旋 0
奏出 62
奏出來 0
奏出去 0
奏功 9
奏在 4
奏得 0
奏折 2
奏捷 21
奏摺 6
奏效 150
奏於 1
奏明 1
奏曲 22
奏書 0
奏本 1
奏樂 22
奏琴 0
奏疏 2
奏章 3
奏著 7
奏表 2
奏議 12
奏起 16
奏起來 1
奏過 4
奏鳴曲 174
奏點 0
奐 238
契 113
契 113
契 1132
契丹 43
契券 0
契卡 1
契友 39
契合 204
契據 19
契文 0
契機 590
契稅 200
契約 3000
契約式 1
契紙 0
契訶夫 1
奒 0
奓 11
奔 362
奔亡 0
奔出 34
奔勞 0
奔北 1
奔去 28
奔向 88
奔命 5
奔喪 21
奔喪 21
奔回 30
奔回來 0
奔往 5
奔忙 20
奔放 93
奔泉 0
奔波 187
奔波 187
奔波往返 1
奔波往返 1
奔流 25
奔竄 4
奔行 6
奔走 121
奔走相告 1
奔赴 9
奔跑 116
奔躍 3
奔逃 12
奔逐 1
奔雷 40
奔馬 12
奔馳 152
奔馳如飛 0
奔騰 104
奕 1048
奕奕 19
套 4555
套上 51
套上來 0
套上去 1
套下 3
套下來 0
套下去 0
套了 12
套交情 2
套件 1489
套住 16
套作 3
套來套去 0
套個 8
套兒 1
套兒 1
套入 27
套出 7
套出來 0
套利 107
套到 2
套匯 15
套印 33
套印本 0
套口供 0
套問 2
套問出 0
套在 70
套套 9
套子 23
套得 3
套得 3
套房 1198
套招 11
套接 3
套數 68
套曲 10
套板 1
套版 11
套版印刷 0
套牢 180
套狀 0
套用 153
套種 0
套筒 43
套索 4
套結 5
套繪 0
套色 19
套色版 0
套著 11
套裝 1489
套裝軟體 592
套話 3
套語 2
套起 0
套起來 0
套車 2
套進 7
套進來 0
套進去 0
套間 1
套鞋 3
套頭 5
套頭毛衣 2
套餐 563
奘 21
奘 21
奙 0
奙 0
奚 119
奚落 12
奛 0
奜 85
奝 0
奞 0
奞 0
奞 0
奟 0
奠 213
奠下 67
奠儀 9
奠基 136
奠基人 4
奠定 652
奠禮 1
奠立 51
奠酒 2
奡 76
奢 59
奢侈 75
奢侈品 16
奢望 52
奢求 42
奢淫 0
奢華 41
奢費 0
奢靡 11
奣 0
奤 0
奥 0
奥 0
奦 0
奧 3477
奧克拉荷馬州 12
奧利地 0
奧古斯丁 37
奧國 37
奧國人 4
奧地利 533
奧地利人 3
奧地利帝國 0
奧妙 203
奧妙無比 0
奧援 69
奧斯丁 51
奧斯卡 317
奧斯卡金像獎 42
奧斯戮 0
奧斯汀 146
奧斯特 7
奧斯陸 19
奧斯陸市 0
奧旨 1
奧會 83
奧會人士 1
奧會選手 0
奧林 199
奧林匹亞 0
奧林匹克 89
奧林匹克世運 0
奧林匹克運動會 46
奧林帕斯 7
奧林帕斯山 1
奧林比亞 3
奧氏體 0
奧狄塞 1
奧祕 108
奧秘 208
奧義 22
奧義書 8
奧萬大 186
奧運 468
奧運團 0
奧運會 75
奧運村 5
奧運選手 3
奨 0
奩 112
奪 1115
奪下 67
奪下來 0
奪來奪去 0
奪冠 0
奪去 58
奪取 131
奪名 2
奪回 61
奪回來 1
奪回去 0
奪得 182
奪標 68
奪權 25
奪目 64
奪眶而出 18
奪走 82
奪過 3
奪過來 2
奪過去 0
奪門 20
奪門而入 0
奪鬥 1
奪魁 78
奪魂 9
奫 26
奬 0
奭 27
奮 289
奮不顧身 29
奮力 173
奮勇 72
奮勇作戰 0
奮勇向前 3
奮勉 3
奮志 1
奮戰 70
奮戰不懈 2
奮戰到底 7
奮發 39
奮發向上 13
奮發圖強 4
奮發有為 0
奮筆 0
奮翼 0
奮臂 1
奮臂 1
奮臂高呼 0
奮臂高呼 0
奮袂 0
奮起 140
奮起直追 1
奮進 41
奮鬥 541
奮鬥到底 2
奮鬥目標 12
奯 0
奰 71
奱 2
奱異 0
奲 48
女 12543
女 300
女丑 1
女丑劇場 0
女丑劇場 0
女中 1235
女中丈夫 0
女中豪傑 1
女中音 3
女主人 54
女主角 599
女主角 599
女乳症 1
女人 4841
女人家 3
女人腔 0
女仕 14
女伴 17
女伶 7
女低音 12
女作家 64
女佣 227
女佣人 2
女侍 20
女傑 8
女傭 65
女傭人 3
女僕 10
女儐相 0
女優 0
女兄 0
女兒 1653
女兒牆 0
女兒紅 13
女兒茶 0
女公子 5
女兵 14
女博士 3
女友 649
女同志 46
女單 0
女執事 1
女士 1092
女大不中留 0
女大不中留 0
女大使 1
女大十八變 6
女大當嫁 1
女奴 5
女妖 4
女婢 0
女婢 0
女婿 143
女媧 23
女媧補天 6
女嬰 64
女子 2525
女子書院 0
女子高級 480
女孩 2248
女孩兒 20
女孩兒 20
女孩子 658
女學士 1
女學生 141
女家長 0
女宿舍 1
女將 116
女居士 4
女工 97
女巫 55
女巫 55
女巫婆 0
女巫婆 0
女師 26
女帽 2
女店員 3
女廁 21
女廁所 2
女弟 0
女弟子 5
女強人 56
女徒 3
女德 3
女性 7594
女性化 48
女房東 1
女扮男裝 14
女排 0
女教員 0
女方 145
女星 221
女朋友 507
女服務員 3
女校 18
女樂 19
女權 111
女權主義 5
女權運動 27
女流 8
女流之輩 4
女演員 39
女為悅己者容 1
女王 242
女王蜂 10
女生 1881
女用 228
女用品 0
女男 511
女男平等 3
女畫家 18
女的 573
女皇 62
女真 25
女眷 6
女神 609
女秀才 0
女秘書 21
女童 238
女童軍 130
女籃 0
女紅 14
女聲 56
女色 19
女英雄 10
女蘿 2
女裝 127
女褲 2
女警 40
女警察 3
女議員 21
女貞 10
女賓 5
女道士 4
女郎 486
女院長 0
女青年會 29
女鞋 48
女高音 103
女鬼 21
奴 1392
奴僕 62
奴化 1
奴化思想 0
奴化教育 0
奴婢 9
奴婢 9
奴家 25
奴工 1
奴役 25
奴性 7
奴才 38
奴脣婢舌 0
奴脣婢舌 0
奴隸 122
奴隸主 1
奴隸制 5
奴隸制度 4
奴隸勞動 1
奴隸社會 3
奴隸起義 0
奴顏婢膝 0
奴顏婢膝 0
奵 0
奶 6705
奶名 0
奶品 7
奶嘴 73
奶奶 316
奶娘 4
奶媽 23
奶房 0
奶昔 36
奶水 49
奶汁 7
奶油 374
奶油色 5
奶爸 1
奶牙 0
奶牛 0
奶瓶 128
奶粉 254
奶粉罐 3
奶精 49
奶糖 1
奶罩 6
奶腺 0
奶茶 349
奶酥 6
奶酪 1
奶頭 11
奷 57
奸 175
奸佞 0
奸匪 0
奸商 31
奸淫 10
奸淫擄掠 1
奸滑 2
奸猾 1
奸笑 19
奸細 10
奸臣 24
奸計 11
奸詐 25
奸謀 5
奸賊 9
奸邪 4
奸險 9
奸雄 5
奸黨 1
她 101398
她上 38
她上來 2
她上去 1
她下 25
她下來 1
她下去 0
她不 468
她不 468
她不是 50
她不會 32
她來 122
她們 23295
她們的 322
她先 64
她出 30
她出來 7
她出去 11
她到 77
她在 677
她想 141
她放 21
她是 897
她殺 6
她的 5001
她自己 190
她認為 127
她請 16
她過 23
她過來 4
她過去 11
奺 0
奻 1828
奼 68
奼紫 1
奼紫嫣紅 2
奼紫焉紅 0
好 309
好 30909
好一個 67
好一個 67
好一會兒 24
好一會兒 24
好一會兒 24
好一會兒 24
好上 13
好上好下 0
好不 213
好不好 390
好不容易 545
好主意 34
好久 1749
好乖 10
好乾 0
好了 1483
好了吧 15
好了嗎 91
好了瘡疤忘了痛 0
好事 237
好事不出門 2
好事之徒 2
好事多磨 16
好事者 13
好些 86
好人 369
好人好事 35
好人難作 0
好人難做 6
好似 188
好住 7
好住 7
好使 34
好來 46
好來好去 1
好個 64
好假 2
好停 3
好傢伙 9
好傢夥 0
好像 5545
好像在 169
好像是 589
好像會 44
好價 1
好兄弟 1
好兒女 1
好兒子 4
好再 83
好凶 6
好出 3
好出風頭 0
好出風頭 0
好利 1
好利 15
好刺 4
好動 94
好動成性 0
好勝 32
好半天 5
好友 1310
好受 36
好句 19
好叫 43
好叫 43
好吃 896
好吃 896
好吃懶作 0
好吃懶做 2
好名 7
好名 7
好向 46
好吧 478
好呀 341
好命 28
好啊 429
好問 16
好喜歡 179
好喝 111
好嗎 1654
好嗓子 4
好在 176
好報 11
好壞 308
好壞不分 2
好壯 69
好多 942
好多人 72
好多個 18
好多天 12
好夢 50
好大 149
好大喜功 10
好天兒 0
好天兒 0
好天氣 48
好奇 946
好奇心 207
好好 2197
好好先生 22
好好兒 1
好好兒 1
好好玩 0
好好的 503
好好地 503
好字 6
好孩子 40
好學 190
好學不倦 13
好官 4
好客 35
好客之道 0
好家伙 1
好容易 17
好密 1
好寫 21
好將 87
好將 87
好小 22
好小子 41
好少 61
好巧 25
好差 6
好帶 18
好幫手 208
好幾 215
好幾個 220
好幾天 108
好弄 4
好弄 4
好強 50
好彈 3
好彩頭 26
好得 3709
好心 303
好心人 95
好心好意 4
好心有好報 3
好心沒好報 3
好急 10
好怪 19
好惡 41
好想 789
好意 94
好意思 153
好感 138
好慢 10
好憑 2
好懂 7
好懶 0
好戰 19
好戰份子 3
好戰份子 3
好戲 156
好戲連臺 2
好戲還在後頭 1
好戲還在後頭 1
好戲開鑼 10
好戴 3
好扁 0
好手 280
好找 105
好把 112
好招 3
好挺 0
好排 3
好整以暇 14
好數 2
好數 2
好日子 68
好晚 4
好景 49
好景不常 34
好書 1788
好會 18
好有 62
好朋友 1609
好望角 17
好棒 150
好極了 12
好樣兒的 0
好樣兒的 0
好樹 3
好歌 63
好歌大家唱 1
好正 5
好歹 69
好死 31
好死不如歹活 3
好死不如賴活 5
好毒 2
好比 251
好比是 27
好氧 34
好沒 16
好油 0
好洗 5
好消息 419
好深 24
好清 1
好滑 0
好漢 193
好漢不吃眼前虧 1
好漢坡 6
好煩 68
好熟 2
好熱 26
好爽 109
好片 41
好牌 11
好狗 8
好狗不擋路 0
好猛 1
好猜 3
好玩 1029
好玩 1029
好班 4
好球 47
好生 45
好生 45
好用 10
好用 1004
好男兒 1
好男兒 1
好畫 6
好當 17
好瘦 5
好的 3709
好的開始 62
好看 871
好短 7
好福氣 5
好端端 14
好笑 439
好筆 3
好管 7
好管 7
好管閒事 10
好米 16
好累 326
好細 4
好緊 8
好缺 2
好美 162
好翹 0
好聚好散 12
好聲好氣 1
好聽 553
好聽 553
好肥 1
好自為之 12
好舊 0
好色 225
好色 225
好色之徒 6
好色者 3
好花 7
好花 7
好花不常開 2
好苦 33
好茶 95
好菜 35
好萊塢 765
好萊塢 765
好處 3228
好處多多 28
好言相勸 7
好記 79
好評 386
好詞 8
好詩 18
好話 90
好話說盡 1
好說 77
好說話 1
好說話 1
好談 33
好請 9
好講 14
好讀 17
好讀 17
好象 83
好買 20
好賣 16
好賭 10
好賺 26
好起來 32
好趕 2
好輕 3
好轉 273
好辦 14
好辦法 26
好辭 2
好辯 6
好近 12
好迷 1
好逑 10
好逸惡勞 18
好運 248
好過 72
好過來 0
好過去 1
好酒 68
好酒 68
好酷 17
好醜 10
好鐵 0
好鐵不打釘 0
好險 47
好難 101
好雨 2
好靜 6
好靜 6
好鞋 14
好飛 8
好香 22
好驢 1
好高 29
好高騖遠 16
好鮮 2
好黑 2
好點 25
奾 162
奿 427
妀 224
妁 447
如 41812
如一 119
如一日 23
如一日 23
如上 159
如上所示 0
如上所述 42
如下 6267
如下所示 0
如下所示者 0
如下所述 0
如下所述者 0
如之何 8
如之奈何 3
如云 4
如人飲水 9
如人飲水冷暖自知 2
如今 2066
如今已是 8
如今是 16
如何 40514
如何是好 200
如來 394
如來佛 20
如來藏 1
如假包換 18
如兄 1
如兄如弟 2
如入無人 1
如入無人之境 6
如其 202
如出一口 0
如出一口 0
如出一轍 29
如出一轍 29
如初 22
如右 60
如右所示 0
如右所示者 0
如同 1728
如喪考妣 3
如因 234
如圖 737
如圖所示 0
如圖所示者 0
如坐針氈 6
如墜五里霧中 1
如墮五里霧中 1
如夢初醒 5
如夢方醒 1
如夫人 1
如實 36
如履平地 3
如履薄冰 7
如履薄冰 7
如左 428
如常 72
如干 0
如廁 38
如影隨形 43
如律令 5
如意 530
如意棒 1
如意算盤 22
如所周知 0
如手如足 1
如擬 20
如故 11
如數 112
如數 112
如數付清 0
如數家珍 16
如於 102
如日中天 31
如日東升 0
如日東昇 0
如昔 21
如是 370
如是我聞 36
如晤 0
如月 58
如有 4778
如有所失 1
如有雷同 21
如期 310
如期完成 42
如期舉行 22
如東 27
如果 34223
如果在 532
如果是 1276
如果會 27
如果有 1951
如果有人 260
如果能 417
如梭 4
如次 124
如此 8451
如此一來 456
如此一來 456
如此等等 1
如此而已 32
如此這般 37
如沐春風 0
如法泡製 10
如法炮製 17
如法炮製 17
如泣如訴 7
如泥 11
如火如荼 105
如狼似虎 5
如獲至寶 12
如痴如狂 2
如癡如醉 8
如登仙境 0
如皋 1
如眾所知 0
如箭在弦 1
如簧之舌 0
如糖似蜜 0
如約 11
如膠似漆 16
如膠如漆 0
如臨大敵 31
如與 98
如舊 12
如花似玉 5
如若 153
如若不然 1
如荼如火 0
如虎添翼 41
如見 14
如許 32
如遇 191
如酥如髓 0
如醉如癡 0
如釋重負 25
如雨 47
如雲 23
如雷貫耳 9
如需 494
如須 116
如願 121
如願以償 30
如風過耳 0
如飢似渴 3
如魚得水 41
如鳥獸散 0
妃 16540
妃嬪 4
妃子 38
妄 161
妄人 2
妄作 4
妄加 11
妄動 8
妄圖 5
妄念 71
妄想 185
妄求 6
妄為 12
妄稱 9
妄自 11
妄自尊大 4
妄自菲薄 14
妄言 8
妄語 26
妄說 2
妄費 0
妅 138
妆 0
妉 0
妊 733
妊 733
妊娠 313
妊娠 313
妊娠紋 1
妊婦 4
妊孕 1
妋 0
妋 0
妌 0
妍 822
妍麗 22
妎 190
妏 433
妐 198
妑 0
妒 562
妒婦 1
妒嫉 20
妒忌 32
妒恨 1
妒意 2
妒痴 0
妓 372
妓女 106
妓女戶 28
妓女院 1
妓院 34
妓館 2
妔 0
妕 0
妖 565
妖人 15
妖兒 0
妖冶 3
妖女 40
妖妄 0
妖姿 0
妖婦 6
妖媚 9
妖嬈 7
妖孽 8
妖怪 144
妖星 3
妖氣 8
妖氣衝天 0
妖法 9
妖物 2
妖猴 1
妖異 3
妖精 240
妖聲妖氣 0
妖艷 11
妖術 6
妖言 3
妖言惑眾 10
妖豔 7
妖道 5
妖邪 2
妖霧 0
妖風 1
妖鬼 0
妖魔 79
妖魔鬼怪 14
妗 623
妗子 4
妘 145
妙 2434
妙不可言 7
妙事 12
妙人 15
妙人妙事 0
妙古絕今 0
妙句 5
妙哉 4
妙喻 7
妙在 8
妙境 5
妙妙 63
妙思 6
妙想天開 6
妙手 33
妙手回春 8
妙手空空 8
妙方 96
妙極 7
妙極了 5
妙法蓮華經 1
妙理 4
妙用 99
妙用無窮 4
妙略 0
妙筆 6
妙筆生花 5
妙答 2
妙答如流 0
妙策 3
妙算 14
妙絕 5
妙絕古今 0
妙舞 5
妙藥 12
妙處 9
妙觀察智 1
妙解 1
妙言 4
妙言妙語 0
妙計 38
妙訣 0
妙語 64
妙語如珠 15
妙語解頤 0
妙語雙關 0
妙論 1
妙趣 10
妙趣橫生 11
妙辭 1
妙麗 4
妙齡 8
妙齡女子 9
妙齡女郎 11
妙齡少女 5
妚 0
妚 0
妜 0
妝 884
妝奩 0
妝扮 22
妝臺 1
妝飾 13
妝點 35
妞 672
妟 0
妠 43
妡 60
妡 60
妢 215
妣 363
妣考 0
妤 440
妥 1931
妥切 16
妥協 283
妥協性 3
妥善 691
妥善處理 147
妥善處理 147
妥帖 1
妥洽 0
妥為 113
妥為安排 1
妥為照料 0
妥用 1
妥當 208
妥貼 17
妦 1
妧 1
妨 561
妨 561
妨事 6
妨害 312
妨害交通 6
妨害自由 21
妨害風化 35
妨礙 443
妨礙交通 16
妨阻 0
妬 0
妭 0
妮 4916
妮子 16
妯 12
妯 12
妯娌 5
妯娌 5
妰 0
妱 1
妲 84
妳 16127
妳們 700
妳們 700
妳們的 88
妳在 269
妳好 427
妳好嗎 41
妳妳 1
妳愛 73
妳是 726
妳會 268
妳有 313
妳的 2539
妳自己 107
妴 0
妵 3
妶 2
妷 0
妹 0
妹 2258
妹夫 2
妹妹 1639
妹妹 1639
妹婿 2
妹子 73
妺 55
妻 903
妻 903
妻兒 32
妻兒老小 0
妻妾 10
妻子 996
妻子 996
妻孥 4
妻室 5
妻小 0
妻離子散 14
妼 1
妽 5
妾 2825
妾侍 0
妾婦 1
妾身 128
姀 2
姁 3
姂 0
姃 4
姄 0
姅 24
姆 1821
姆指 39
姆指山 1
姇 3
姇 3
姈 12
姉 0
姉 0
姊 0
姊 7313
姊 667
姊兒 3
姊兒 3
姊夫 34
姊妹 16
姊妹 1673
姊妹品 2
姊妹品 2
姊妹市 6
姊妹市 6
姊妹會 16
姊妹會 16
姊妹淘 1
姊妹篇 1
姊妹篇 1
姊妹花 9
姊妹花 9
姊姊 618
姊姊 618
始 3306
始亂終棄 13
始作俑者 46
始基 0
始得 0
始料未及 35
始於 203
始末 109
始業式 23
始皇 25
始祖 82
始祖鳥 8
始終 1081
始終不渝 3
始終不變 4
始終如一 63
始能 170
姌 0
姍 177
姍 177
姍姍來遲 25
姍笑 0
姎 4
姏 4
姐 1
姐夫 1
姐妹 1
姐姐 1
姐弟 27
姑 712
姑丈 10
姑且 144
姑且一試 2
姑且一試 2
姑公 0
姑夫 0
姑奶奶 6
姑妄聽之 8
姑妄言之 9
姑姑 128
姑娘 779
姑娘家 2
姑婆 32
姑婆芋 1
姑媽 31
姑嫂 7
姑嫂倆 0
姑嫜 0
姑子 10
姑息 21
姑息養奸 5
姑母 3
姑洗 1
姑父 4
姑爺 4
姑老爺 0
姑舅 1
姑蘇 0
姑表 1
姒 15
姓 4353
姓林 105
姓衛 27
姓魏 27
姓名 28020
姓名學 121
姓字 3
姓氏 381
委 0
委 9303
委以 3
委以重任 3
委任 488
委任書 45
委任狀 10
委內瑞拉 126
委員 6980
委員制 5
委員會 14829
委員長 256
委外 0
委婉 34
委實 26
委屈 148
委屈求全 4
委托 75
委推 1
委曲 31
委曲求全 8
委棄 2
委決不下 0
委決不下 0
委派 34
委派代表 1
委用 1
委罪 0
委罪於人 0
委蛇 3
委蛇 3
委託 6480
委託行 17
委託金 0
委身 57
委辦 454
委過 1
委隨 0
委靡 0
委靡 0
委靡不振 2
委靡不振 2
委頓 0
姖 4
姗 0
姗 0
姘 203
姘居 0
姘頭 0
姘頭 0
姙 0
姙 0
姚 1530
姛 6
姜 1222
姜太公 4
姜太公釣魚 3
姜子牙 4
姝 94
姝麗 0
姞 15
姟 0
姠 9
姡 6
姢 0
姣 201
姣 201
姣好 20
姣好 20
姣美 1
姣美 1
姤 12
姥 154
姥 154
姥姥 59
姦 204
姦夫 3
姦夫淫婦 1
姦婦 0
姦情 14
姦殺 27
姦汙 1
姦淫 50
姦淫擄掠 2
姦邪 1
姧 0
姨 307
姨丈 13
姨太太 3
姨姐 0
姨娘 13
姨婆 9
姨媽 24
姨母 1
姨父 3
姨表 0
姩 10
姪 102
姪兒 25
姪女 27
姪女婿 0
姪婦 0
姪婿 0
姪媳婦 0
姪子 20
姪孫 3
姫 0
姬 1286
姬妾 7
姭 9
姮 42
姯 0
姰 0
姰 0
姱 4
姲 7
姳 7
姴 6
姵 80
姶 11
姷 5
姸 0
姹 0
姺 9
姻 281
姻家 0
姻戚 0
姻緣 87
姻親 58
姼 4
姽 14
姾 1
姿 33296
姿勢 577
姿容 3
姿影 2
姿態 363
姿色 15
姿質 0
娀 11
威 6839
威信 195
威儀 40
威力 640
威勢 21
威名 47
威名遠播 2
威嚇 55
威嚴 56
威士忌 172
威士忌酒 10
威尼斯 297
威尼斯人 1
威廉 406
威德 38
威懾 8
威斯康辛州 13
威望 34
威權 210
威武 85
威武不能屈 1
威武雄壯 0
威海 22
威海衛 17
威爾 392
威爾基 0
威爾斯 181
威爾第 125
威爾遜 25
威猛 25
威盛 640
威福 4
威而剛 1
威耳生霧室 0
威聲 0
威脅 1346
威脅利誘 10
威行 4
威赫 5
威迫 1
威迫利誘 2
威逼 12
威逼利誘 8
威震八方 1
威靈 8
威靈頓 85
威風 37
威風八面 7
威風凜冽 0
威風凜凜 11
娂 0
娂 0
娂 0
娃 759
娃兒 31
娃兒 31
娃娃 724
娃娃兵 8
娃娃臉 26
娃娃裝 0
娃娃車 20
娃娃鞋 1
娃子 4
娄 0
娄 0
娄 0
娉 27
娉婷 9
娊 16
娋 0
娋 0
娌 10
娌 10
娍 0
娎 0
娏 4
娐 0
娑 79
娑摩 2
娑摩吠陀 2
娒 0
娓 19
娓娓 54
娓娓道來 29
娔 0
娕 6
娖 1
娗 23
娘 7620
娘兒 4
娘兒 4
娘兒倆 0
娘兒倆 0
娘兒們 7
娘兒們 7
娘娘 59
娘娘腔 10
娘子 104
娘子 104
娘子軍 14
娘子軍 14
娘家 49
娘胎 10
娘舅 0
娘親 10
娙 12
娚 0
娛 774
娛悅 2
娛樂 15216
娛樂場 15
娛樂場 15
娛樂場所 40
娛樂場所 40
娛樂室 17
娛樂性 85
娛樂版 10
娛樂節目 18
娛賓 1
娜 19
娜 195
娜 1954
娜拉 7
娜拉 7
娝 0
娞 52
娟 2460
娟娟 89
娟秀 14
娠 64
娠 64
娠 64
娡 0
娢 0
娣 103
娤 0
娥 497
娥眉 20
娦 0
娦 0
娧 0
娨 0
娩 38
娩 38
娪 0
娫 0
娬 0
娬 0
娭 25
娮 4
娯 0
娯 0
娰 0
娱 0
娱 0
娳 166
娵 120
娶 313
娶 313
娶了 49
娶到 11
娶妻 23
娶媳 0
娶媳婦 9
娶得 8215
娶得 6
娶親 15
娷 42
娸 290
娹 11
娺 0
娺 0
娻 0
娼 140
娼優 2
娼妓 76
娼妓館 0
娼婦 7
娼寮 2
娽 0
娾 10
婀 250
婀娜 17
婀娜多姿 13
婁 102
婁 102
婂 3
婃 17
婄 10
婅 0
婆 557
婆姨 1
婆娑 34
婆娑之洋 34
婆娑起舞 3
婆娘 25
婆婆 269
婆婆媽媽 60
婆媳 71
婆子 38
婆家 30
婆心 2
婆羅多 39
婆羅洲 33
婆羅門 39
婆羅門教 8
婇 4
婈 30
婉 1316
婉拒 83
婉約 34
婉言 3
婉言相勸 0
婉謝 9
婉轉 57
婉轉動聽 1
婉辭 8
婉順 2
婉麗 18
婊 20
婊子 0
婊子 200
婋 0
婌 131
婍 8
婎 0
婏 0
婐 196
婑 6
婒 4
婓 2623
婔 0
婕 239
婖 178
婗 11
婘 125
婙 0
婙 0
婙 0
婚 2493
婚事 38
婚假 22
婚前 292
婚前婚後 0
婚喪喜慶 91
婚外 16
婚外情 228
婚姻 2223
婚姻介紹所 3
婚姻制度 24
婚姻法 11
婚姻自由 1
婚嫁 45
婚宴 59
婚後 188
婚書 0
婚期 35
婚生子女 5
婚禮 824
婚禮上 13
婚禮中 11
婚筵 13
婚約 34
婚紗 0
婚紗店 1
婚紗照 1
婚變 0
婚配 13
婚齡 11
婛 30
婜 16
婝 45
婞 115
婟 87
婠 437
婡 0
婢 92
婢 92
婢僕 4
婢僕 4
婢女 19
婢女 19
婢子 5
婢子 5
婣 0
婤 19
婤 19
婥 116
婦 1597
婦人 430
婦人之仁 4
婦人病 1
婦唱夫隨 1
婦女 6332
婦女新知 1
婦女新知基金會 1
婦女會 192
婦女節 21
婦女運動 66
婦女雜誌 9
婦嬰 23
婦孺 26
婦孺皆知 3
婦幼 973
婦幼保健 5
婦幼節 1
婦幼衛生 56
婦幼醫院 60
婦德 5
婦權 11
婦產科 1307
婦產醫生 0
婦產醫院 0
婦科 269
婦科醫生 5
婦聯 21
婦聯會 27
婦聯總會 3
婦職 1
婦道 4
婦道人家 2
婧 50
婨 0
婩 3
婩 3
婪 11
婬 260
婭 126
婮 0
婯 0
婯 0
婰 14
婱 0
婱 0
婳 0
婵 0
婷 4000
婷婷 137
婷婷玉立 0
婸 76
婹 0
婺 147
婻 92
婼 95
婽 76
婾 0
婿 41
媀 0
媀 0
媁 0
媂 0
媃 30
媄 78
媅 0
媆 0
媇 0
媇 0
媈 0
媊 136
媋 50
媌 74
媍 0
媎 14
媏 67
媐 7
媑 0
媒 1802
媒人 47
媒人婆 1
媒介 613
媒介物 30
媒介體 2
媒合 35
媒妁 2
媒妁之言 8
媒婆 15
媒染劑 92
媒質 17
媒體 10771
媓 71
媔 54
媕 103
媖 0
媗 22
媘 0
媙 0
媚 2200
媚世 2
媚俗 0
媚外 1
媚惑 1
媚態 8
媚眼 14
媚骨 1
媛 718
媛 718
媜 55
媝 120
媞 61
媟 122
媠 0
媡 0
媢 85
媣 0
媥 47
媦 123
媧 8
媧杆 0
媧桿 0
媧輪 0
媨 0
媩 68
媪 0
媪 0
媫 0
媬 178
媮 47
媯 205
媰 7
媱 4
媲 22
媲美 164
媳 28
媳婦 188
媳婦兒 3
媳婦兒 3
媴 1
媵 10
媶 3
媷 3
媸 3
媹 2
媺 21
媻 2
媼 509
媽 1
媽 1519
媽咪 262
媽媽 3902
媽媽 3902
媽媽桑 19
媽的 135
媽祖 332
媽祖廟 79
媾 10
媾和 3
媾和案 3
媿 6
嫀 3
嫁 593
嫁了 34
嫁人 31
嫁作 3
嫁作人婦 0
嫁個 6
嫁個老公 0
嫁入 9
嫁出 3
嫁出來 0
嫁出去 5
嫁到 20
嫁去 0
嫁妝 25
嫁妝費 0
嫁娶 166
嫁接 21
嫁接法 0
嫁於 1
嫁狗隨狗 5
嫁禍 26
嫁禍於人 0
嫁紗 4
嫁給 277
嫁裝 1
嫁走 0
嫁過來 1
嫁過去 2
嫁錯 3
嫁雞隨雞 4
嫂 173
嫂夫人 3
嫂嫂 18
嫂子 20
嫃 0
嫄 19
嫅 0
嫆 10
嫇 8
嫈 44
嫉 75
嫉妒 192
嫉妒 192
嫉妒心 13
嫉恨 8
嫉惡 0
嫉惡如仇 8
嫉視 1
嫊 5
嫋 3
嫋娜 1
嫋娜 1
嫋嫋 2
嫌 1099
嫌人 2
嫌人家 1
嫌厭 0
嫌忌 2
嫌恨 0
嫌惡 17
嫌憎 0
嫌棄 68
嫌犯 417
嫌疑 212
嫌疑犯 42
嫌貧愛富 1
嫌隙 11
嫍 4
嫍 4
嫎 0
嫎 0
嫏 0
嫐 0
嫕 544
嫖 429
嫖妓 45
嫖客 28
嫖賭 11
嫗 25
嫘 275
嫙 25
嫚 204
嫚罵 0
嫛 21
嫜 7
嫝 6
嫞 51
嫟 4
嫠 11
嫠婦 0
嫡 14
嫡傳 15
嫡出 2
嫡子 2
嫡母 1
嫡派 1
嫡系 15
嫡親 8
嫡長子 1
嫢 184
嫣 148
嫣然 47
嫣然一笑 11
嫣紅 13
嫤 0
嫤 0
嫥 56
嫦 94
嫦娥 79
嫦娥奔月 7
嫧 0
嫧 0
嫨 3
嫨 3
嫩 266
嫩城 0
嫩嫩 17
嫩得 0
嫩江 43
嫩江省 1
嫩的 29
嫩皮 0
嫩紅 2
嫩綠 8
嫩綠色 1
嫩肉 1
嫩膚 5
嫩芽 21
嫩葉 20
嫩黃 0
嫪 67
嫫 169
嫬 10
嫬 10
嫭 8
嫮 92
嫯 0
嫰 0
嫰 0
嫳 645
嫴 15
嫵 33
嫵媚 53
嫵媚動人 4
嫶 1
嫷 6
嫸 7
嫹 4
嫺 78
嫻 274
嫻於 1
嫻於辭令 0
嫻熟 32
嫻習 3
嫻雅 4
嫻靜 4
嫼 0
嫽 11
嫾 0
嫿 7
嬀 0
嬀 0
嬀 0
嬁 43
嬂 6
嬃 0
嬅 31
嬆 0
嬇 15
嬈 27
嬉 53
嬉戲 77
嬉水 3
嬉痞 6
嬉皮 16
嬉皮笑臉 6
嬉笑 18
嬉笑怒罵 2
嬉遊 10
嬉鬧 17
嬊 0
嬊 0
嬋 96
嬋娟 26
嬋媛 0
嬌 17400
嬌嗔 1
嬌妻 37
嬌娃 31
嬌媚 18
嬌媚動人 2
嬌嫩 14
嬌嫩欲滴 0
嬌嬌 24
嬌嬌嫩嫩 0
嬌客 25
嬌小 46
嬌小玲瓏 5
嬌態 1
嬌慣 1
嬌憨 4
嬌柔 21
嬌氣 2
嬌滴滴 17
嬌生慣養 4
嬌縱 5
嬌美 23
嬌羞 21
嬌艷 18
嬌艷動人 0
嬌艷欲滴 4
嬌豔 18
嬌貴 6
嬌軀 0
嬍 0
嬎 0
嬏 4
嬐 4
嬑 0
嬒 0
嬒 0
嬓 5
嬔 4
嬕 0
嬖 27
嬗 27
嬘 0
嬙 11
嬚 1
嬛 11
嬛 11
嬛 11
嬜 0
嬝 351
嬞 1
嬟 0
嬟 0
嬠 2
嬡 7
嬢 0
嬢 0
嬣 36
嬤 1
嬤 197
嬤嬤 15
嬤嬤 15
嬥 118
嬦 46
嬧 42
嬨 4
嬩 0
嬪 160
嬪妃 6
嬪從 0
嬫 0
嬬 67
嬭 57
嬮 60
嬯 32
嬰 14000
嬰兒 1843
嬰兒期 36
嬰兒油 1
嬰孩 75
嬰幼兒 0
嬱 0
嬱 0
嬲 51
嬳 0
嬴 326
嬵 0
嬷 0
嬷 0
嬸 33
嬸婆 2
嬸嬸 37
嬸子 2
嬸母 0
嬹 0
嬹 0
嬺 0
嬻 0
嬼 63
嬽 3
嬾 10
嬿 81
孀 9
孀婦 4
孀居 1
孁 0
孂 0
孂 0
孃 79
孄 0
孅 13
孆 0
孇 35
孈 87
孈 87
孉 0
孉 0
孊 0
孋 11
孌 10
孌 10
孌生兒 0
孌生兒 0
孌童 5
孍 56
孎 98
孏 0
子 30184
子 30184
子代 51
子來 62
子健 30
子兒 53
子兒 53
子午儀 0
子午線 9
子口 24
子句 54
子嗣 15
子囊 3
子囊殼 1
子囊菌類 0
子夏 8
子夜 65
子女 1939
子姪 1
子婿 2
子子孫孫 30
子孫 340
子孫滿堂 4
子宮 1232
子宮外孕 0
子宮套 0
子宮學 0
子宮癌 14
子宮頸 606
子宮頸抹片 1
子宮頸癌 0
子宮頸癌 0
子實 10
子弟 286
子弟兵 21
子張 15
子彈 492
子彈型 7
子彈頭 0
子思 11
子息 3
子房 24
子房壁 4
子時 35
子曰 133
子書 0
子棉 0
子標題 16
子樹 14
子母 68
子母彈 1
子母扣 1
子母船 2
子母釦 0
子民 305
子爵 22
子爵夫人 0
子環 1
子產 17
子由 30
子目 48
子瞻 5
子粒 14
子葉 23
子虛 15
子虛烏有 11
子規 4
子貢 18
子路 67
子部 27
子金 1
子集合 8
子音 47
孑 45
孑孓 7
孑然 2
孑然一身 6
孑然一身 6
孒 0
孒 0
孓 32
孔 1970
孔丘 6
孔口 12
孔型 3
孔墨 0
孔夫子 73
孔子 530
孔子廟 1
孔子廟堂碑 1
孔子白 1
孔子誕辰紀念 1
孔孟 39
孔孟之道 2
孔孟學會 5
孔尚任 1
孔廟 214
孔徑 87
孔德 17
孔急 9
孔教 5
孔數 3
孔方兄 4
孔明 568
孔武有力 9
孔洞 37
孔穎達 1
孔穴 12
孔老夫子 0
孔融 8
孔融讓梨 1
孔蟲 30
孔道 20
孔門 13
孔隙 57
孔雀 389
孔雀園 10
孔雀東南飛 7
孔雀石 7
孔雀綠 1
孔雀藍 4
孔雀開屏 5
孔雀魚 388
孔鯉過庭 0
孕 857
孕出 1
孕吐 1
孕婦 611
孕婦裝 12
孕婦走獨木橋 1
孕期 39
孕珠 1
孕穗 1
孕穗期 0
孕育 229
孕育出 55
孖 117
字 16877
字串 2404
字元 761
字元碼 8
字元組 5
字兒 7
字兒 7
字典 770
字句 128
字型 1456
字字 56
字字珠璣 11
字帖 44
字幕 248
字幕機 67
字彙 259
字形 486
字形輸入法 2
字意 9
字據 4
字數 180
字斟句酌 0
字書 16
字根 84
字條 25
字模 6
字模 6
字樣 503
字檔 335
字正腔圓 13
字母 1148
字母表 3
字源 17
字畫 51
字盤 8
字眼 238
字眼兒 1
字眼兒 1
字碼 100
字符 49
字紙 14
字紙簍 1
字組 164
字義 54
字義疏證 0
字號 1814
字表 23
字表輸入法 0
字裡行間 51
字詞 156
字謎 16
字跡 73
字鍵 18
字集 133
字面 57
字面上 37
字音 72
字音表 0
字音輸入法 1
字首 94
字體 696
存 5736
存下 27
存亡 51
存亡之秋 4
存亡之際 0
存保 0
存儲 38
存儲器 0
存入 266
存取 1326
存單 70
存在 4354
存在主義 43
存在著 285
存底 6
存廢 77
存心 67
存心不良 3
存戶 82
存摺 120
存放 680
存放在 116
存放處 43
存有 384
存查 48
存根 36
存檔 551
存欄 1
存款 2060
存款單 16
存款簿 5
存歿 1
存水 9
存活 373
存活率 120
存留 78
存疑 81
存立 2
存簿 35
存續 0
存著 79
存褶 3
存記 3
存證 98
存證信函 1
存貨 521
存貨量 23
存貯器 1
存量 111
存錢 65
孚 826
孚 826
孚眾望 8
孛 38
孛 388
孛人 0
孜 385
孜孜 25
孜孜不倦 26
孜孜不息 1
孝 3307
孝堂 2
孝女 14
孝婦 3
孝子 42
孝宗 10
孝家 3
孝帷 0
孝廉 7
孝弟 0
孝弟 0
孝心 20
孝心感人 0
孝思 11
孝悌 15
孝感 9
孝敬 43
孝文帝 6
孝服 4
孝男 2
孝經 28
孝行 28
孝行 28
孝行感人 1
孝道 60
孝順 121
孝順為齊家之本 1
孝順父母 40
孝養 15
孟 2749
孟什維克 0
孟冬 4
孟加拉 92
孟加拉國 2
孟加拉灣 3
孟夏 1
孟姜女 8
孟婆湯 2
孟子 142
孟宗竹 1
孟德斯鳩 17
孟德爾 22
孟德爾松 2
孟德爾頌 59
孟斐斯 2
孟春 12
孟母 5
孟母三遷 6
孟浩然 1
孟浪 0
孟獲 32
孟禧子 0
孟秋 4
孟買 44
孟軻 1
孠 0
孡 0
孢 64
孢子 95
孢子囊 10
孢子體 8
季 3245
季候風 2
季刊 1535
季報 0
季度 47
季後賽 0
季節 913
季節性 159
季節風 13
季軍 142
季軍獎 1
季辛吉 16
季風 291
孤 1328
孤丁 0
孤伶伶 14
孤傲 18
孤僻 26
孤僻成性 0
孤兒 199
孤兒寡婦 8
孤兒院 41
孤危迫切 0
孤哀子 0
孤單 312
孤單單 2
孤單單的 2
孤單單地 2
孤墳 3
孤女 21
孤子 0
孤孤單單 13
孤孤淒淒 0
孤孤獨獨 0
孤孤零零 0
孤家寡人 11
孤寂 169
孤寒 0
孤寡 4
孤島 47
孤微 0
孤憤 2
孤挺花 1
孤掌難鳴 5
孤木 1
孤本 9
孤注 1
孤注一擲 15
孤注一擲 15
孤獨 663
孤立 195
孤立主義 1
孤立無援 18
孤老 8
孤臣孽子 3
孤芳自賞 13
孤苦 43
孤苦伶仃 3
孤身 19
孤軍 53
孤軍作戰 1
孤軍奮鬥 11
孤軍深入 1
孤陋 3
孤陋寡聞 9
孤雁 69
孤雲野鶴 0
孤零 4
孤零零 31
孤高自許 0
孤魂 22
孤魂野鬼 11
孥 2
学 0
学 0
孨 0
孨 0
孩 772
孩兒 47
孩子 7243
孩子氣 42
孩子頭 0
孩提 15
孩提時代 13
孩童 628
孪 0
孫 3517
孫中山 207
孫傳庭 0
孫傳芳 3
孫兒 13
孫兒女 0
孫吳 3
孫堅 13
孫大聖 0
孫女 58
孫女婿 1
孫婿 0
孫媳 1
孫媳婦 2
孫子 218
孫子 218
孫子兵法 81
孫息 0
孫悟空 99
孫文 48
孫文學說 8
孫權 146
孫武 41
孫璐西 6
孫竹 0
孫策 25
孫耀威 37
孫臏 9
孫逸仙 103
孫運璿 76
孬 9
孬 9
孮 48
孯 0
孰 258
孰知 10
孰能 11
孰能無過 5
孰與 0
孰若 1
孰謂 0
孱 5
孱 5
孱夫 0
孱弱 13
孱頭 0
孲 8
孳 63
孳 63
孳乳 2
孳孳 0
孳息 91
孳生 89
孳蔓 0
孴 0
孵 3980
孵出 32
孵出來 6
孵化 127
孵化器 3
孵化率 9
孵卵 7
孵卵器 3
孵成 1
孵育 16
孵蛋 11
孵蛋器 2
孷 3
學 380
學 38014
學上 427
學不完 7
學人 444
學以致用 74
學伴 1
學位 1362
學來 61
學來學去 0
學分 4082
學分制 107
學到 544
學制 414
學前 401
學前教育 231
學力 322
學務處 0
學區 908
學區制 80
學博 65
學去 32
學友 738
學名 837
學員 1835
學員制 0
學員證 5
學問 505
學問好 2
學問為濟世之本 2
學回 18
學回來 1
學回去 0
學園 534
學堂 196
學報 1561
學壞 8
學士 1623
學士學位 287
學士府 0
學好 117
學如逆水行舟 4
學妹 0
學姊 214
學姐 318
學子 462
學學 217
學學看 0
學官兩棲 0
學家 305
學富五車 3
學富五車 3
學工 41
學年 2495
學年度 11120
學年考試 0
學店 17
學府 395
學弟 273
學徒 61
學徒制 5
學徒工 0
學得 1916
學成 91
學成歸來 4
學成歸國 24
學政 24
學文 108
學時 108
學會 7386
學有專長 41
學期 6998
學校 34787
學校同學 5
學校成績 9
學案 21
學業 802
學業成績 277
學歷 4176
學法 349
學派 245
學海 89
學海無涯 239
學潮 10
學無止境 4
學然後知不足 1
學理 231
學生 38293
學生代聯 0
學生代聯會 0
學生手冊 171
學生時代 57
學生會 433
學生會長 11
學生自治 91
學生自治會 17
學生裝 0
學生證 365
學生運動 49
學界 676
學的 1916
學監 0
學社 265
學科 2638
學科考試 51
學究 31
學究式 0
學究氣 0
學童 1237
學籍 306
學籍表 6
學系 16922
學級 18
學習 16666
學習團 0
學習年限 3
學習心得 45
學習成績 22
學習效果 157
學習法 44
學習班 4
學習精神 21
學習計劃 26
學者 4800
學者型 0
學而不思 1
學而優則仕 3
學舍 18
學苑 1989
學著 123
學藝 171
學藝股 4
學藝股長 10
學號 870
學行 24
學術 12590
學術團體 116
學術報告 7
學術性 286
學術界 472
學術界人士 18
學術科 42
學術網路 0
學說 346
學說話 4
學識 222
學識淵博 6
學貫古今 0
學費 824
學起 41
學起來 17
學軍 1
學運 51
學運領袖 8
學過 161
學過來 1
學過去 0
學道 50
學醫 79
學長 1274
學長制 8
學門 763
學閥 1
學院 7012
學院派 38
學雜費 348
學非所用 1
學風 51
學養 77
學館 424
學點 10
學齡 59
學齡兒童 57
學齡前 81
學齡前兒童 61
孹 0
孹 0
孺 148
孺 148
孺人 2
孺子 8
孺子不可教也 0
孺子可教 6
孺子可教也 3
孻 128
孼 0
孽 119
孽子 7
孽畜 1
孽種 6
孽障 1
孾 0
孿 1
孿 12
孿子 1
孿生 41
孿生兄弟 1
孿生兒 0
孿生子 0
宀 0
宁 115
宂 0
它 18651
它們 23290
它們的 437
它是 1866
它的 3881
宄 24
宅 126
宅 1263
宅子 2
宅心 1
宅心仁厚 11
宅男 1
宅第 36
宅配 1
宅門 1
宅院 33
宆 0
宆 0
宇 7199
宇內 15
宇宙 2127
宇宙射線 25
宇宙空間 12
宇宙線 12
宇宙船 28
宇宙觀 30
宇宙誌 0
宇宙論 20
宇宙雲 0
宇宙飛船 8
宇文化及 85
宇文泰 1
宇航 25
守 11887
守中立 2
守份 4
守住 107
守信 42
守信用 3
守候 159
守候多時 0
守備 136
守備區 1
守備軍 0
守備道 0
守分 7
守制 10
守則 463
守勢 21
守口如瓶 9
守喪 9
守喪期間 1
守土 6
守土有責 3
守在 96
守城 48
守夜 70
守夜員 3
守好 4
守孝 1
守完 1
守宮 13
守密 13
守寡 5
守寡人 0
守將 11
守得 13
守得 63
守恆 17
守恆定律 5
守成 70
守成不易 5
守成不變 0
守戒 9
守敵 0
守時 16
守望 98
守望相助 109
守株待兔 16
守正不阿 3
守歲 11
守法 189
守法主義 0
守的 63
守節 10
守紀 10
守紀律 8
守約 11
守舊 16
守舊派 3
守著 171
守衛 117
守要 0
守規矩 21
守護 332
守護人 7
守護員 6
守護神 168
守護者 80
守貞 2
守財 5
守財奴 5
守身 8
守身如玉 13
守車 8
守軍 15
守邊 3
守錯 0
守門 52
守門員 77
守隊 15
守靈 3
安 17413
安上 12
安丘 0
安之若素 0
安了 27
安于 7
安享 6
安享天年 4
安享晚年 4
安人 41
安份 22
安份守己 6
安住 41
安內 14
安內攘外 3
安全 16279
安全係數 20
安全器 2
安全存量 12
安全島 21
安全帶 135
安全帽 393
安全座椅 1
安全感 195
安全措施 108
安全期 66
安全梯 38
安全檢查 155
安全氣囊 1
安全火柴 9
安全燈 4
安全玻璃 33
安全理事會 10
安全瓣 0
安全生產 8
安全第一 26
安全系統 356
安全閥 40
安分 39
安分守己 11
安南 254
安卡拉 25
安危 67
安危禍福 0
安厝 0
安可 80
安史之亂 12
安命 2
安和 250
安和樂利 39
安和路 609
安哥拉 33
安固 5
安國 198
安土重遷 4
安地斯山 7
安坐 19
安培 72
安培數 1
安培計 6
安太歲 1
安好 30
安如 5
安如泰山 0
安妥 0
安妮 305
安娜 297
安安 265
安安全全 0
安安定定 1
安安心心 4
安安穩穩 11
安安靜靜 14
安定 1732
安定下來 20
安定力 1
安定性 157
安定物 1
安家 81
安家落戶 1
安家費 7
安寧 584
安寧平靜 0
安居 108
安居樂業 21
安山岩 1
安帝 10
安平 479
安平古堡 68
安平港 41
安度 25
安康 72
安康社區 2
安康路 113
安徒生 28
安徒生童話 9
安得 52
安德森 61
安德生 9
安徽 105
安徽人 1
安徽省 47
安心 621
安心吧 4
安息 120
安息吧 4
安息年 2
安息日 239
安息油 0
安息香 8
安慰 612
安慰劑 1
安慶 27
安慶街 32
安打 238
安打率 0
安抵 11
安排 4749
安排方式 10
安排法 0
安排處理 2
安排處理 2
安排處置 0
安插 31
安插在 5
安插法 0
安撫 185
安撫下來 0
安撫住 0
安放 43
安放在 12
安於 33
安於現狀 8
安曼 8
安東 87
安東尼 170
安東尼奧 31
安東尼奧尼 2
安東市 2
安東省 1
安東街 23
安枕 8
安格魯薩克遜 0
安樂 302
安樂國 2
安樂園 15
安樂椅 2
安樂死 79
安樂窩 41
安檢 0
安歇 25
安歇吧 1
安步 1
安步當車 6
安民 32
安泰 883
安源 176
安瀋鐵路 0
安然 137
安然無事 3
安然無恙 22
安理會 125
安琪兒 9
安生 90
安產 3
安甯 0
安眠 44
安眠劑 14
安眠藥 55
安睡 8
安睡在 0
安知非福 0
安祥 300
安祿山 4
安福 7
安穩 90
安第斯山 0
安第斯山脈 0
安第斯山脈 0
安置 799
安置費 1
安胎 22
安胎藥 5
安臥 1
安舒 10
安葬 127
安葬地 0
安葬費 0
安蒂岡妮 1
安裝 8137
安裝法 13
安裝費 10
安親班 1
安設 5
安詳 62
安謁斐雷蚊 0
安謐 0
安貧 3
安貧樂道 2
安身 17
安身之地 0
安身立命 45
安迪 446
安逸 60
安道爾 8
安適 32
安邦 107
安邦定國 0
安邦興國 0
安鄉 848
安閑 0
安閒 8
安閒愜意 0
安陽 46
安靖 8
安靜 534
安靜下來 27
安靜下去 0
安靜無聲 8
安非 32
安非他命 155
安頓 47
安頓下來 5
安養 563
安養院 46
宊 0
宊 0
宋 4366
宋人 37
宋代 150
宋代人 0
宋儒 2
宋元 37
宋史 54
宋國 53
宋太祖 10
宋學 7
宋徽宗 8
宋慶 13
宋教仁 1
宋文帝 0
宋明 88
宋春舫 1
宋書 1
宋朝 88
宋朝人 1
宋楚瑜 3481
宋江 1
宋瓷 4
宋襄公 1
宋詞 43
宋體 10
宋體字 0
完 6911
完了 500
完事 25
完人 42
完備 396
完備無缺 0
完全 10302
完全中學 0
完全是 327
完全會 4
完全能 11
完全葉 1
完全變態 16
完卵 3
完善 1511
完好 52
完好如新 1
完好無缺 3
完婚 18
完完全全 52
完完整整 16
完密 2
完封 0
完工 1237
完工後 113
完成 15322
完成式 8
完成後 918
完整 5590
完整如新 3
完整性 267
完整無缺 17
完浚 0
完滿 30
完滿無缺 0
完璧 1
完璧歸趙 4
完畢 1005
完稅 46
完稅價格 28
完稿 135
完稿員 0
完結 42
完結篇 61
完美 1441
完美無瑕 18
完美無疵 1
完美無缺 28
完蛋 27
完蛋了 63
宍 0
宍 0
宎 130
宏 5843
宏亮 35
宏仁 148
宏偉 184
宏儒 11
宏儒碩學 0
宏儒碩學 0
宏國 662
宏圖 27
宏圖大展 0
宏壯 0
宏大 119
宏富 13
宏恩 107
宏恩醫院 5
宏揚 58
宏旨 1
宏業 16
宏構 0
宏碁 100
宏碁 2327
宏碩 3
宏碩 3
宏規 12
宏觀 407
宏論 1
宏達 109
宏都拉斯 160
宏願 23
宏麗 0
宐 0
宒 249
宓 35
宔 0
宔 0
宕 24
宖 0
宗 7044
宗主 17
宗主國 6
宗主權 8
宗兄 1
宗姓 5
宗子 3
宗室 9
宗師 150
宗廟 20
宗弟 5
宗教 3876
宗教信仰 213
宗教儀式 21
宗教家 18
宗教改革 45
宗教自由 28
宗族 48
宗族社會 0
宗旨 4009
宗法 18
宗法制度 5
宗法社會 0
宗派 165
宗派主義 1
宗祠 79
宗聖 48
宗藩 2
宗親 72
宗親會 97
宗門 1
官 3235
官事 3
官人 35
官位 18
官俸 1
官僚 234
官僚主義 3
官僚作風 2
官僚政治 5
官僚派 0
官僚習氣 2
官僚資本主義 1
官僚資產階級 0
官價 29
官兒 7
官兒 7
官兵 289
官兵一致 0
官兵一致 0
官兵捉強盜 4
官制 36
官印 4
官史 33
官司 227
官名 2
官吏 57
官員 2389
官商 23
官商勾結 90
官地 0
官報 6
官報私仇 0
官場 79
官場 79
官場禮貌 0
官場禮貌 0
官大 26
官大學問大 9
官官相護 3
官客 0
官宦 10
官家 10
官屬 2
官差 10
官帽 6
官府 41
官庫 3
官廳 11
官復原職 0
官拜 4
官方 1337
官方人士 7
官書 28
官服 3
官架子 3
官校 46
官樣 2
官樣文章 2
官氣 0
官派 37
官渡 7
官營 3
官爵 4
官田 299
官癮 0
官祿 18
官私合營 0
官窯 34
官紳 4
官署 37
官職 46
官股 24
官能 97
官能團 1
官能症 45
官腔 2
官腔官調 0
官舍 14
官莊 4
官衙 2
官袍 0
官詞 0
官話 12
官調 5
官軍 15
官轎 1
官辦 7
官逼民反 1
官運 8
官運亨通 4
官道 17
官邸 188
官郵 0
官銜 8
官鑄 1
官長 16
官階 12
官餉 0
官馬 2
宙 260
宙斯 62
定 15983
定下 74
定下來 11
定下去 0
定了 120
定位 2021
定位在 108
定住 21
定使 14
定例 6
定價 6476
定冠詞 4
定准 8
定出 90
定出來 6
定制 9
定則 22
定力 53
定南針 0
定可 53
定可以 7
定名 28
定名為 112
定向 265
定向天線 6
定員 6
定單 20
定在 123
定址 60
定址法 11
定型 352
定壓 10
定奪 24
定好 25
定婚 12
定子 18
定存 238
定定 14
定定的 2
定局 27
定居 304
定居在 19
定居點 0
定州 1
定幅 1
定弦 3
定形 24
定影 13
定影液 7
定影液 7
定律 257
定心 16
定心丸 3
定性 156
定性分析 15
定情 26
定情之物 4
定戶 16
定數 29
定於 185
定日 12
定是 299
定時 737
定時器 58
定時炸彈 72
定時裝置 8
定時開關 8
定會 180
定期 3776
定期存單 55
定期存款 157
定本 239
定案 304
定植 12
定死 3
定死無疑 0
定法 11
定海 11
定準 5
定溫 33
定滑輪 2
定為 232
定然 57
定然如此 0
定理 168
定產 3
定當 20
定省 11
定睛 22
定睛一看 12
定睛一看 12
定碳 8
定碳杯 0
定神 40
定神一看 11
定神一看 11
定票 9
定稿 68
定約 5
定給 13
定縣 9
定罪 47
定義 2876
定義出 25
定義域 0
定能 90
定能夠 2
定色劑 9
定苗 0
定裝 8
定製 122
定要 3799
定見 27
定規 25
定親 6
定計劃 1
定評 5
定調 10
定論 123
定讞 15
定貨 10
定貨單 0
定購 59
定身法 1
定都 55
定都在 1
定都於 0
定量 359
定量分析 42
定金 32
定長 6
定非 22
定音鼓 33
定額 443
定食 1
定點 300
宛 272
宛 272
宛如 282
宛然 32
宛若 57
宛轉 13
宜 7607
宜人 82
宜古宜今 5
宜喜宜嗔 1
宜嗔宜喜 1
宜安 26
宜室宜家 1
宜家 18
宜寧 29
宜山 4
宜市 6
宜於 26
宜昌 108
宜昌峽 0
宜春 8
宜春帖 1
宜然 1
宜縣 46
宜興 101
宜蘭 3338
宜蘭人 58
宜蘭大學 1
宜蘭市 903
宜蘭縣 4951
宜蘭縣長 32
宜賓 26
宜陽 3
宜顰宜笑 1
宝 0
实 0
実 0
客 20607
客串 38
客串性 1
客串演出 12
客人 788
客位 12
客卿 2
客商 13
客土 5
客地 2
客堂 6
客場 0
客場 0
客套 19
客套話 7
客官 32
客室 8
客家 1333
客家人 179
客家話 73
客居 32
客層 0
客席 29
客店 17
客座 139
客座教授 78
客廳 445
客戶 9711
客戶資料 142
客房 407
客服 1
客棧 1600
客機 236
客死他鄉 3
客氣 353
客氣話 2
客源 100
客滿 90
客群 0
客舍 4
客船 27
客艙 67
客觀 950
客觀主義 6
客觀化 3
客觀唯心主義 0
客觀存在 7
客觀實在 0
客觀性 52
客觀真理 3
客語 0
客貨 192
客車 661
客車廂 0
客輪 22
客運 1065
客運公司 56
客隊 12
客飯 11
客餐 0
客體 162
宣 4574
宣佈 1622
宣佈無效 1
宣佈獨立 22
宣傳 1523
宣傳品 24
宣傳員 1
宣傳單 68
宣傳機構 2
宣傳網 9
宣傳車 74
宣傳週 0
宣傳部 28
宣傳隊 0
宣判 108
宣判有罪 3
宣判無罪 2
宣勞 0
宣勞 0
宣召 1
宣告 1115
宣告成立 3
宣告有罪 3
宣告無罪 3
宣告破產 20
宣宗 5
宣導 2460
宣布 1581
宣德 29
宣慰 17
宣戰 85
宣戰書 0
宣戰案 4
宣揚 330
宣旨 1
宣武 15
宣泄 1
宣洩 69
宣示 563
宣稱 642
宣紙 53
宣統 17
宣言 847
宣誓 203
宣誓代表 2
宣誓就職 47
宣誓書 22
宣說 21
宣講 50
宣讀 141
宣讀文告 0
室 52404
室內 2874
室內樂 144
室內裝潢 28
室內設計 1009
室內賽 0
室內運動 16
室友 209
室外 602
室外運動 0
室女座 28
室家 1
室樂 0
室溫 177
宥 216
宥恕 1
宦 143
宦官 29
宦海 6
宦海沉浮 0
宦海浮沉 0
宦途 1
宦遊 2
宧 33
宨 13
宫 0
宬 753
宭 34
宮 3924
宮中 68
宮保 12
宮保雞丁 1
宮刑 2
宮城 50
宮女 37
宮娥 6
宮室 6
宮庭 16
宮廷 147
宮廷式 4
宮廷舞 1
宮扇 1
宮樓 0
宮殿 148
宮殿式 14
宮燈 25
宮牆 6
宮禁 2
宮裡 12
宮調 254
宮錦 1
宮錦衣袍 0
宮門 13
宮闈 6
宮闈畫 0
宮闕 6
宮院 1
宯 0
宰 223
宰制 49
宰割 24
宰夫 0
宰殺 78
宰相 58
宰相肚裡能撐船 1
宰羊 18
宰輔 1
宱 0
宲 0
宲 0
害 2401
害人 126
害人不淺 7
害人害己 5
害人精 1
害命 2
害喜 16
害國害民 0
害得 70
害得 70
害怕 1049
害死 79
害病 5
害羞 241
害群之馬 9
害臊 8
害著 0
害處 34
害蟲 168
害起 3
害躁 0
害過 9
害鳥 3
宴 4000
宴客 64
宴居 0
宴席 53
宴會 177
宴會廳 105
宴會服 0
宴會裝 0
宴請 91
宴遊 3
宴飲 0
宴饗 19
宵 252
宵夜 190
宵小 29
宵民 0
宵禁 49
宵禁時間 0
宵衣旰食 0
家 1
家 37051
家丁 17
家世 39
家中 1739
家事 293
家事商業 15
家事商業職業學校 70
家事職校 3
家事職業學校 60
家人 2025
家伙 63
家住 210
家信 14
家俱 103
家傳 49
家僕 10
家兄 7
家兒 3
家兒 3
家兔 21
家具 1649
家具行 35
家制 2
家務 113
家務事 37
家務管理 0
家叔 1
家史 19
家和 324
家和萬事興 5
家商 360
家商職校 15
家商補校 7
家喻戶曉 47
家嚴 4
家園 506
家塾 8
家塾本 0
家境 83
家境清寒 80
家奴 5
家姊 0
家姑 3
家子 41
家學 34
家學淵源 11
家宅 14
家室 26
家宴 7
家家 128
家家戶戶 60
家家酒 24
家富爾 0
家專 61
家小 275
家居 201
家屬 1266
家山 10
家常 122
家常便飯 23
家常菜 36
家常話 5
家常飯 0
家常餅 1
家底 2
家庭 20442
家庭主夫 0
家庭主婦 204
家庭作業 47
家庭副業 5
家庭工業 4
家庭手工 4
家庭手工業 0
家庭教師 12
家庭教育 702
家庭生活 283
家庭號 4
家庭計劃 44
家弟 4
家徒四壁 6
家情 6
家慈 3
家政 307
家政學 2
家政專科學校 15
家政科 33
家政系 24
家教 749
家教中心 36
家族 3900
家族企業 29
家書 31
家有 912
家有嬌妻 2
家業 35
家母 49
家法 32
家無長物 0
家父 115
家產 37
家用 1834
家用電腦 142
家畜 851
家當 51
家眷 25
家破人亡 25
家祖 5
家祠 4
家祭 7
家禽 200
家禽家畜 6
家私 12
家累 6
家缽 0
家聲 29
家職 20
家莊 54
家蚊 9
家蠅 3
家蠶 2
家裏 2044
家裡 12720
家規 9
家計 93
家訓 22
家譜 37
家貓 12
家財 15
家財萬貫 5
家貧如洗 0
家賊 1
家道 17
家道中落 2
家鄉 550
家鄉肉 0
家鄉話 1
家醜 7
家醜不可外揚 1
家長 4064
家長制 1
家長式 3
家長會 773
家門 96
家門不幸 0
家雀 0
家電 1257
家電業 53
家風 16
家鴿 4
家鼠 7
家齊 84
宷 0
宸 279
容 6215
容下 7
容不下 30
容光 3
容光煥發 18
容受 18
容器 1561
容容 266
容容易易 0
容幸 1
容後說明 0
容忍 192
容忍度 15
容恕 10
容情 12
容態 2
容或 28
容抗 1
容易 6821
容止 0
容熱量 0
容積 528
容積率 189
容納 544
容納得下 2
容納得下 2
容膝 2
容色 4
容許 605
容讓 12
容貌 112
容身 17
容身之地 1
容身之處 7
容重 2
容量 2093
容量瓶 2
容錯 1
容限 1
容電器 1
容顏 127
宺 0
宻 0
宼 0
宿 173
宿 173
宿 1737
宿主 71
宿債 0
宿儒 0
宿命 205
宿命論 11
宿命論者 5
宿命通 1
宿因 1
宿志 0
宿怨 31
宿懷 0
宿敵 0
宿昔 1
宿業 1
宿歇 3
宿營 18
宿留 1
宿疾 12
宿網 0
宿緣 7
宿舍 1876
宿舍網路 0
宿衛 1
宿醉 28
宿願 18
宿願已償 0
宿願得償 0
寀 166
寀 166
寁 94
寂 238
寂寂 7
寂寞 1086
寂寞歲月 0
寂寥 33
寂滅 11
寂然 14
寂然不動 4
寂靜 173
寂靜無聲 6
寂默 1
寃 0
寄 11450
寄上 70
寄上來 0
寄上去 0
寄主 45
寄了 70
寄予 40
寄予厚望 26
寄交 29
寄人籬下 10
寄件人 371
寄來 188
寄來寄去 0
寄信 1000
寄信人 62
寄出 1033
寄出去 47
寄到 351
寄剩 0
寄去 47
寄售 12
寄售品 0
寄售店 0
寄售處 0
寄回 259
寄在 3
寄好 1
寄存 138
寄存器 0
寄存處 0
寄完 8
寄宿 161
寄宿生 6
寄寓 30
寄居 31
寄居蟹 24
寄得 3
寄情 176
寄愁 0
寄懷 1
寄成 2
寄托 8
寄托在 2
寄掛號 2
寄掛號信 0
寄放 16
寄放在 9
寄放處 0
寄於 9
寄望 239
寄滿 0
寄物處 1
寄生 233
寄生植物 7
寄生物 1
寄生菌 1
寄生蟲 328
寄生蟲病 9
寄發 423
寄的 73
寄籍 3
寄給 3180
寄託 334
寄託在 26
寄語 24
寄費 4
寄賣 14
寄賣品 0
寄賣店 0
寄賣處 0
寄走 0
寄足 0
寄身 3
寄送 1000
寄過 12
寄過來 12
寄過去 2
寄還 11
寄還給 6
寄銷品 0
寄錢 61
寄錯 5
寄附 1
寄食 1
寄養 112
寅 382
寅吃卯糧 7
寅支卯糧 0
寅時 6
密 4005
密不可分 96
密云 1
密令 17
密件 30
密佈 38
密使 51
密函 18
密切 1345
密匝匝 0
密印 1
密友 33
密合 61
密告 9
密咒 1
密咒義 1
密商 12
密宗 258
密室 72
密密 30
密密層層 0
密密的 10
密密麻麻 48
密實 20
密封 342
密封式 20
密布 37
密度 1167
密度計 1
密戶 0
密探 9
密接 13
密旨 1
密會 0
密林 44
密植 13
密爾頓 2
密特朗 11
密碼 10396
密碼信 1
密碼本 2
密碼簿 0
密約 13
密結 1
密織 2
密蘇里 109
密蘇里州 22
密西根 267
密西根州 27
密西西比 13
密西西比州 13
密西西比河 9
密訪 10
密談 26
密謀 17
密議 1
密跡金剛 1
密道 22
密醫 13
密閉 203
密閉式 105
密集 1046
密集區 6
密集式 29
密集金剛 1
密雲 5
密電 5
密電碼 0
寇 1670
寇仇 0
寇讎 5
寈 0
寈 0
寊 48
寋 138
富 10170
富人 54
富厚 0
富商 80
富國 236
富國強兵 4
富士 335
富士山 40
富士通 379
富婆 4
富孀 0
富家 24
富家女 4
富家子 1
富家子弟 3
富庶 37
富康 29
富強 109
富強康樂 4
富態 1
富戶 7
富於 90
富春江 1
富春縣 0
富有 434
富比 85
富民 58
富泰 40
富海 14
富源 157
富甲天下 22
富盛 55
富礦 1
富翁 285
富而不驕 0
富而好禮 20
富而驕 0
富華 19
富蘭克林 199
富裕 221
富裕中農 0
富豪 218
富豪世家 0
富豪之家 0
富貴 358
富貴不能淫 1
富貴在天 2
富貴手 26
富貴榮華 9
富貴病 4
富貴花 0
富貴逼人 1
富賈 4
富足 59
富農 35
富農分子 0
富農經濟 0
富陽 58
富陽縣 0
富餘 2
富饒 20
富麗 99
富麗堂皇 34
寍 83
寎 149
寏 0
寐 33
寑 66
寒 1176
寒來暑往 0
寒假 500
寒傖 1
寒光 27
寒冬 193
寒冬臘月 1
寒冷 272
寒喧 22
寒噤 5
寒士 3
寒夜 60
寒天 1
寒害 18
寒山 61
寒帶 45
寒微 10
寒心 8
寒意 57
寒慄 1
寒戰 2
寒暄 48
寒暑 405
寒暑易節 0
寒暑表 0
寒暑計 0
寒林 9
寒梅 33
寒武紀 12
寒毛 9
寒氣 19
寒氣逼人 6
寒泉 4
寒波盪漾 0
寒波盪漾 0
寒流 107
寒浞 0
寒潮 6
寒熱 20
寒玉 10
寒症 5
寒磣 0
寒窗 11
寒舍 64
寒色 0
寒苦 3
寒號鳥 0
寒螿 0
寒蟬 3
寒衣 34
寒賤 0
寒酸 11
寒門 0
寒雁 0
寒霜 15
寒露 16
寒顫 22
寒風 118
寒風凜冽 1
寒風刺骨 6
寒食 18
寒鴉 3
寓 182
寓公 3
寓兵於農 1
寓居 2
寓意 57
寓意深長 2
寓所 31
寓教於樂 115
寓於 3
寓言 188
寔 97
寕 0
寕 0
寖 5
寗 0
寗 0
寘 6
寙 13
寚 0
寛 0
寜 0
寜 0
寝 0
寞 199
察 1323
察勘 1
察哈爾 9
察哈爾省 3
察明 1
察核 1
察看 121
察知 12
察究 0
察覺 278
察覺出 10
察覺到 64
察言觀色 8
察訪 3
察辦 0
察院 17
寠 11
寡 259
寡不敵眾 3
寡人 51
寡人之疾 1
寡信 0
寡味 0
寡女 8
寡婦 100
寡居 6
寡廉鮮恥 8
寡恩 4
寡情 8
寡情薄義 0
寡慾 4
寡斷 5
寡欲 1
寡歡 2
寡母 10
寡糖 53
寡聞 2
寡聞寡見 0
寡言 12
寡陋 0
寡頭 12
寡頭政治 2
寢 196
寢丘之志 1
寢具 132
寢室 214
寢食不安 2
寢食俱廢 0
寢食難安 7
寢食難忘 0
寣 266
寤 100
寥 109
寥寥 28
寥寥可數 21
寥寥無幾 20
寥廓 1
寥若晨星 3
寥若辰星 0
寥落 8
寥落晨星 0
實 10678
實機 1249
實不相瞞 2
實事 25
實事求是 35
實作 0
實例 1243
實做 45
實像 9
實價 6
實兵 3
實兵演練 1
實兵演習 1
實利 4
實到 37
實到人數 4
實則 89
實力 1621
實力派 27
實務 4266
實名 3
實告 4
實因 17
實在 1
實在 3770
實在是 1312
實在論 1
實地 809
實地考察 16
實地調查 30
實坪 6
實報 10
實報實銷 9
實境 1
實字 1
實學 23
實定 7
實實 36
實實在在 80
實實落落 1
實小 119
實幹 5
實彈 11
實彈射擊 14
實彈演習 7
實得 23
實得分數 7
實心 97
實心實意 1
實心彈 0
實心球 2
實心眼 1
實心體 1
實情 72
實惠 169
實意 1
實感 79
實成 1
實戰 554
實戶 1
實才 3
實打實 1
實拍拍 1
實據 10
實支 18
實收 187
實效 35
實數 58
實施 10948
實是 192
實景 1
實有 193
實有其事 1
實根 10
實業 12033
實業家 8
實業界 6
實業計劃 14
實槍 1
實槍實彈 0
實槍演練 0
實槍演習 0
實權 28
實歲 1
實況 532
實況報導 41
實況轉播 107
實況轉播 107
實測 83
實測距離 0
實火 1
實為 345
實無 61
實焦點 0
實物 304
實物交易 1
實物地租 0
實物教學 1
實物教學 1
實現 1761
實生林 0
實用 2492
實用主義 22
實用性 228
實用文 0
實症 1
實發 10
實益 15
實相 86
實線 1
實績 1248
實繁有徒 1
實缺 12
實習 6280
實習生 44
實習課 60
實習醫生 20
實習銀行 1
實職 1
實至名歸 29
實落 1
實葫蘆 1
實行 990
實行起來 5
實設 7
實証 45
實詞 1
實話 52
實話實說 9
實誠 1
實說 12
實證 427
實證主義 1
實變函數 5
實質 1397
實質上 158
實質利率 1
實質性 28
實質成長 18
實質所得 1
實質違建 1
實賺 2
實足 38
實足年齡 5
實踐 2087
實踐大學 1
實踐家專 11
實踐家政 10
實踐家政專科學校 1
實踐家政經濟專科學校 1
實踐設計管理學院 1
實踐証明 0
實踐論 2
實踐證明 3
實蹟 1248
實銷 2
實錄 253
實際 6733
實際上 1152
實際工資 1
實際情況 102
實際成本 26
實際氣體 3
實難 54
實需 16
實驗 7728
實驗劇場 59
實驗劇場 59
實驗品 18
實驗報告 1
實驗場 15
實驗場 15
實驗學校 121
實驗室 6995
實驗心理學 7
實驗性 90
實驗所 31
實驗法 30
實驗站 28
實驗者 24
實驗語音學 1
實驗論 0
實驗高級中學 72
實體 681
實體化 4
實體性 3
實體法 4
實體詞 0
實體論 1
寧 2410
寧可 247
寧可信其 0
寧可信其有 5
寧和 7
寧夏 51
寧夏回族 0
寧夏回族自治區 20
寧夏城 0
寧夏平原 0
寧夏省 2
寧夏自治 0
寧夏自治區 2
寧夏路 34
寧安 19
寧岡 2
寧息 2
寧折不彎 1
寧日 3
寧死不屈 4
寧死不服 0
寧死不降 0
寧波 99
寧波 99
寧波市 2
寧波市 2
寧波府 0
寧波府 0
寧洱 0
寧為 22
寧為玉碎 3
寧為雞口 0
寧為雞首 3
寧缺勿濫 19
寧肯 2
寧謐 16
寧靖 17
寧靜 548
寧靜下來 0
寧靜海 22
寧靜致遠 0
寧願 333
寧馨兒 0
寧馨兒 0
寨 405
寨主 14
寨子 1
審 2410
審判 916
審判員 4
審判官 14
審判權 10
審判長 19
審問 40
審問權 0
審問者 0
審官 0
審定 536
審定本 50
審察 19
審度 17
審慎 602
審慎考慮 11
審慎處理 20
審慎處理 20
審批 35
審斷 4
審時度勢 3
審查 5541
審查員 25
審查會 178
審查會議 75
審核 2058
審核權 4
審案 15
審理 349
審稿 48
審稿人 1
審稿員 0
審結 11
審美 99
審美活動 4
審美眼光 1
審美觀 25
審美觀念 8
審處 0
審視 147
審訂 49
審訂本 1
審計 2242
審計處 149
審計部 448
審訊 58
審訊官 0
審議 2200
審議會 32
審閱 76
審閱官 0
寪 138
寫 8100
寫上 111
寫上來 7
寫上去 7
寫下 969
寫下來 57
寫下去 15
寫了 538
寫作 1717
寫作家 3
寫作班 11
寫來 33
寫來寫去 2
寫信 2835
寫個 56
寫做 7
寫入 434
寫全 5
寫出 520
寫出來 109
寫出去 0
寫到 161
寫去 2
寫回 11
寫回來 0
寫回去 1
寫在 298
寫好 206
寫字 192
寫字台 3
寫字檯 2
寫字臺 7
寫完 166
寫實 252
寫實派 4
寫實片 0
寫寫 107
寫寫字 1
寫得 1317
寫意 58
寫意畫 2
寫成 381
寫景文 0
寫書 45
寫有 58
寫本 17
寫歌 36
寫毛筆字 1
寫法 262
寫照 104
寫生 173
寫生畫 4
寫的 1317
寫真 1620
寫真集 678
寫程式 145
寫稿 32
寫給 295
寫著 304
寫詩 129
寫起 17
寫起來 12
寫進 21
寫進來 1
寫進去 5
寫過 137
寫過來 0
寫過去 0
寫錯 104
寫錯字 11
寫點 19
寬 3797
寬了 3
寬以待人 4
寬厚 42
寬大 144
寬大為懷 4
寬大無邊 0
寬宏 18
寬宏大量 12
寬宥 4
寬容 170
寬寬 21
寬寬厚厚 0
寬寬大大 0
寬寬鬆鬆 0
寬幅 23
寬度 744
寬廣 530
寬延 0
寬待 5
寬得 2
寬心 28
寬恕 94
寬慰 14
寬敞 199
寬暢 3
寬猛相濟 0
寬的 126
寬窄 15
寬約 52
寬綽 0
寬肩 3
寬衣 4
寬衣解帶 8
寬裕 39
寬赦 2
寬邊 8
寬邊帽 1
寬銀幕 21
寬闊 155
寬闊無邊 0
寬限 74
寬頻 0
寬鬆 408
寭 0
寮 1795
寮共 1
寮北 3
寮國 205
寮戰 0
寯 8
寰 1161
寰宇 490
寰海 1
寰球 0
寱 1
寲 7
寳 0
寴 0
寵 300
寵信 8
寵兒 51
寵壞 21
寵幸 8
寵愛 106
寵愛有加 2
寵物 2116
寵辱 1
寶 11300
寶丰 1
寶位 3
寶元 25
寶典 1071
寶刀 62
寶刀未老 13
寶剎 3
寶劍 135
寶器 0
寶地 23
寶塔 86
寶塚 147
寶寶 1922
寶山 227
寶島 901
寶座 352
寶庫 517
寶慶 59
寶成鐵路 0
寶杖 3
寶殿 42
寶氣 1
寶物 181
寶特瓶 134
寶玉 303
寶珠 97
寶璽 3
寶眷 2
寶石 484
寶箱 725
寶蓋 11
寶藍 26
寶藍色 14
寶藏 303
寶號 77
寶豐 16
寶貝 1577
寶貝兒 2
寶貝兒 2
寶貴 1356
寶貴意見 344
寶鑑 31
寶雞 4
寶馬 110
寷 0
寸 442
寸土 16
寸土寸金 0
寸土必爭 2
寸心 8
寸斷 8
寸步 8
寸步不讓 0
寸步不離 5
寸步難移 1
寸步難行 21
寸絲不掛 1
寸縷 0
寸腸 1
寸草 15
寸草不生 0
寸草不留 2
寸草春暉 0
寸金 16
寸金難買 0
寸長 10
寸陰 1
对 0
寺 2174
寺僧 10
寺塔 3
寺廟 509
寺院 156
寽 0
寽 0
寽 0
対 0
寿 0
尀 0
封 4534
封上 19
封上來 0
封上去 0
封下 0
封下來 0
封下去 0
封了 17
封住 42
封來封去 0
封侯 5
封個 0
封入 6
封冊 0
封凍 0
封凍期 0
封到 2
封包 0
封印 95
封反 0
封口 104
封后 0
封君 1
封國 2
封在 15
封地 13
封套 45
封好 6
封存 40
封官 3
封封 1
封山 22
封山育林 0
封底 54
封建 85
封建主 0
封建主義 0
封建制度 5
封建割據 0
封建思想 3
封建性 1
封建文化 0
封建時代 12
封建殘餘 0
封建社會 16
封建禮教 0
封建道德 0
封得 3
封授 0
封條 24
封樹 0
封死 11
封殺 136
封殺出局 2
封法 1
封泥 0
封港 4
封火 1
封為 36
封爵 4
封牢 2
封疆 5
封皮 17
封神榜 32
封禪 3
封筆 8
封箱 23
封緘 45
封號 45
封蠟 0
封裝 0
封豕長蛇 0
封贈 1
封起 1
封起來 7
封銷 1
封鎖 168
封鎖線 7
封鎖起來 1
封閉 716
封閉性 27
封面 3127
封面人物 96
封面女郎 12
封面裡 10
封點 0
尃 24
射 210
射 210
射 2106
射上 2
射上來 0
射上去 0
射下 14
射下來 3
射下去 0
射不出 0
射不到 1
射中 39
射了 31
射人先射馬 1
射住 1
射來 19
射來射去 0
射入 79
射出 800
射出來 6
射出去 2
射到 34
射去 19
射向 50
射回 0
射回來 0
射回去 0
射在 18
射好 0
射射 3
射干 13
射影 18
射得 5
射得 29
射成 11
射手 1035
射手座 1
射擊 417
射擊場 5
射擊場 5
射死 17
射殺 59
射法 6
射流 6
射源 49
射獵 5
射球 15
射球入門 0
射的 29
射的 29
射程 68
射穿 8
射箭 213
射籃 1
射籃得分 0
射精 101
射給 1
射線 215
射線療法 0
射起 0
射起來 0
射進 31
射進來 0
射進去 0
射過 4
射過來 1
射過去 0
射門 91
射門得分 0
射陽 2
射電望遠鏡 3
射電頻率 0
射靶 8
射頻 148
射魚 2
尅 0
将 0
将 0
将 0
將 20
將 820
將 82028
將上 209
將上來 2
將上去 0
將之 494
將令 35
將住 6
將作 150
將使 851
將來 2557
將來在 62
將來是 14
將來會 80
將來有 60
將信將疑 4
將做 91
將兵 8
將出 195
將出來 11
將出去 1
將到 57
將功補過 1
將功贖罪 7
將去 21
將可 1533
將可以 309
將名 6
將回 24
將回來 0
將回到 33
將回去 0
將在 2783
將士 41
將士用命 6
將如 190
將官 35
將定 26
將對 531
將就 156
將屆 69
將帥 17
將往 38
將得 43
將得 43
將從 369
將心比心 40
將息 4
將成 900
將才 5
將於 2342
將是 1822
將會 4070
將有 1422
將本求利 2
將材 1
將校 25
將死 34
將為 599
將為 599
將王 35
將用 86
將由 816
將略 6
將盡 124
將盡未盡 0
將相 181
將給 43
將能 169
將自 890
將至 89
將與 673
將要 353
將計 4
將計就計 3
將軍 1079
將軍溪 9
將軍溪 9
將近 908
將過 40
將過來 7
將過去 50
將錯就錯 14
將門 25
將門之後 0
將門虎子 0
將領 206
專 9424
專一 83
專一性 31
專上 165
專上程度 11
專事 64
專人 666
專任 1705
專任教師 711
專使 2
專來 1
專供 224
專修 91
專修班 45
專修科 43
專修科目 1
專函 27
專刊 1449
專利 4599
專利品 33
專利權 710
專利法 275
專利稅 0
專制 120
專制主義 2
專制制度 1
專制政府 1
專制政治 2
專制政體 0
專制權 0
專制起來 0
專區 2676
專司 58
專名 7
專向 4
專員 1088
專唱 1
專在 10
專報 5
專士 3
專家 5898
專屬 2072
專屬經濟 8
專屬經濟區 3
專差 4
專往 1
專心 563
專心一志 3
專心一志 3
專心一意 10
專心一意 10
專心致志 2
專心誠意 0
專恣 0
專情 74
專戶 187
專才 171
專打 3
專拜 0
專擅 11
專擺 1
專攻 157
專政 35
專教 42
專文 220
專斷 13
專書 400
專有 214
專有名詞 195
專校 2507
專案 7180
專案小組 885
專案處理 30
專案處理 30
專業 19488
專業人士 399
專業公司 44
專業化 279
專業性 242
專業教育 102
專業知識 576
專業科目 509
專業精神 64
專業組織 20
專業課 9
專業課程 307
專業銀行 19
專機 92
專橫 11
專櫃 423
專櫃小姐 13
專欄 3769
專欄作家 89
專權 22
專款 191
專治 36
專注 429
專為 732
專營 336
專用 3232
專用品 7
專用基金 1
專用線 2
專看 2
專科 4061
專科學校 2211
專租 0
專程 132
專程來 12
專程到 15
專程去 4
專精 346
專精於 56
專線 4235
專線電話 190
專署 2
專美 10
專美於前 21
專職 241
專職人員 38
專著 71
專要 0
專訪 1955
專註 0
專誠 11
專論 501
專責 837
專賣 311
專賣店 1191
專賣權 2
專車 441
專車接送 25
專輯 4365
專送 4
專長 3972
專門 3707
專門人才 35
專門人民檢察院 1
專門人民法院 0
專門化 5
專門性 36
專門機構 14
專門知識 25
專門術語 13
專集 146
專電 419
專題 7601
尉 202
尉 202
尉官 5
尊 1718
尊位 1
尊像 1
尊卑 12
尊卑不分 0
尊口 0
尊古卑今 0
尊命 2
尊嚴 585
尊大人 0
尊夫人 1
尊姓大名 373
尊容 10
尊崇 78
尊師 34
尊師重道 30
尊府 2
尊敬 297
尊榮 85
尊為 37
尊稱 63
尊翁 7
尊者 138
尊處 0
尊號 7
尊貴 193
尊重 2601
尊長 10
尊顏 2
尊駕 12
尋 399
尋 3993
尋事 1
尋事 1
尋人 303
尋人啟事 52
尋俗 0
尋俗 0
尋出 22
尋出 22
尋味 5
尋問 42
尋夢 139
尋夢 139
尋尋覓覓 60
尋尺 0
尋常 133
尋常百姓 3
尋幽 51
尋幽探勝 4
尋幽訪勝 44
尋思 37
尋找 4752
尋找出來 2
尋找失物 2
尋春 6
尋根 75
尋根問底 0
尋根究底 3
尋樂 2
尋歡 193
尋歡作樂 5
尋死覓活 0
尋求 1955
尋求出來 0
尋涼 0
尋涼 0
尋獲 0
尋短見 0
尋章摘句 0
尋花問柳 33
尋芳 9
尋覓 193
尋視起來 0
尋視起來 0
尋親 15
尋訪 103
尋過 5
尋過 5
尋開心 0
尋香 1
尋香 1
尌 56
對 91128
對一對 4
對上 164
對不住 10
對不對 237
對不起 1923
對了 1097
對事 30
對事不對人 16
對人 550
對人不對事 0
對人對事 4
對仗 18
對付 588
對位 67
對位法 10
對來對去 0
對保 58
對偶 127
對偶句 0
對光 47
對兒 11
對兒 11
對內 205
對內政策 1
對分 13
對勁 17
對半 32
對口 52
對口唱 0
對口相聲 5
對句 11
對台 825
對台戲 1
對合起來 0
對味 12
對唱 62
對嘴 10
對地 74
對壘 30
對外 2915
對外政策 19
對外貿易 290
對天 73
對天發誓 3
對好 36
對子 50
對家 69
對射 6
對對 81
對對看 5
對局 17
對岸 223
對峙 170
對帳 22
對待 589
對得 393
對得起 53
對心 32
對應 710
對應性 2
對成 31
對戰 245
對戲 9
對手 1364
對打 90
對抗 875
對抗性 7
對抗賽 103
對折 24
對拼 0
對換 8
對敵 41
對數 162
對數函數 6
對數方程 0
對數表 0
對文 55
對方 3501
對於 20650
對日 333
對日抗戰 24
對映 110
對時 32
對時間 32
對望 28
對望一眼 12
對本 1718
對案 12
對正 44
對比 552
對決 273
對流 145
對流層 46
對流雨 0
對消 8
對準 153
對準目標 2
對焦 0
對照 734
對照表 571
對牛彈琴 1
對獎 211
對生 112
對當 61
對症下藥 56
對白 338
對的 393
對看 21
對看一眼 0
對眼 20
對硫磷 0
對稱 447
對稱中心 4
對稱軸 10
對空 29
對空中 1
對空射擊 0
對立 444
對立性 2
對立統一 2
對立統一規律 0
對立面 11
對筆 7
對等 244
對等條約 0
對等關係 0
對答 33
對答如流 4
對策 1068
對簿公堂 10
對紅 16
對罵 14
對聯 88
對聯式 0
對胃口 1
對胺苯甲酸 0
對苯二甲酸 29
對苯二酚 8
對茦 0
對著 652
對號 26
對號入座 54
對號快車 3
對號車 20
對蝦 81
對表 5
對襟 2
對角 27
對角線 58
對証 2
對話 3289
對話體 0
對調 59
對談 288
對講機 164
對證 43
對讀 5
對象 7258
對質 62
對賬 0
對賽 12
對起 7
對起來 0
對路 20
對軸 3
對轉 10
對過 35
對過來 1
對過兒 0
對過兒 0
對過去 56
對邊 15
對酌 3
對酒當歌 9
對錯 90
對錯不分 0
對錶 23
對門 10
對開 40
對陣 42
對面 745
對頂角 0
對頭 28
對飲 8
對齊 167
導 132
導 22350
導入 663
導入正軌 3
導出 100
導出來 0
導出去 0
導到 0
導向 1551
導向飛彈 7
導尿管 30
導師 1440
導引 954
導彈 137
導彈系統 0
導彈網 0
導播 31
導播 31
導數 8
導板 1
導柱 2
導正 259
導正法 0
導流 16
導源 8
導演 2570
導火 4
導火索 1
導火線 50
導熱 26
導熱性 4
導盤 2
導盲犬 1
導管 326
導納 3
導線 258
導線管 5
導至 83
導致 3937
導航 647
導航系統 124
導覽 0
導言 61
導論 610
導護 134
導護老師 22
導軌 35
導輪 11
導遊 1770
導電 204
導電度 51
導電性 130
導電體 2
導體 259
小 77311
小嘍嘍 10
小一 53
小丈夫 1
小三 128
小三子 0
小不忍則亂大謀 2
小不點 24
小不點兒 0
小不點兒 0
小丑 116
小丑魚 0
小丫頭 30
小丫頭 30
小串 3
小主人 26
小主教 0
小乘 84
小乘佛教 1
小了 84
小事 245
小事化無 2
小事情 10
小二 59
小二哥 1
小五 211
小五金 29
小井 1
小亞細亞 7
小人 263
小人之交 0
小人兒 32
小人兒 32
小人國 140
小人得志 0
小人物 67
小仗 0
小令 16
小仲馬 3
小件 12
小企業 157
小伙 5
小伙子 74
小住 4
小便 211
小便池 2
小倆口 62
小個 9
小個子 3
小偏宜 0
小偷 138
小傘 2
小傢伙 29
小傢伙 29
小傳 254
小僕 0
小僧 19
小兄 8
小兄弟 21
小兒 1409
小兒子 50
小兒痲痹 13
小兒痲痺 119
小兒科 1008
小兒麻痹 13
小兒麻痺 119
小兔 133
小兩口 19
小兩口兒 0
小兩口兒 0
小兩口子 0
小公主 27
小公子 30
小兵 86
小兵立大功 37
小冊 64
小冊子 108
小凌河 0
小刀 69
小分隊 0
小別 10
小別勝新婚 4
小前提 1
小劑 0
小功 38
小劫 0
小勇 6
小動 6
小動作 42
小動脈 13
小動脈 13
小勝 16
小勝利 0
小包 98
小匙 77
小區 34
小區域 18
小半 3
小卒 27
小叔 26
小叔叔 3
小叔子 2
小口 19
小口徑 14
小口袋 2
小叮噹 0
小可 102
小可愛 1
小吃 849
小吃店 119
小吃攤 17
小吃部 33
小吃館 4
小名 102
小吵 3
小命 28
小和尚 23
小品 1158
小品文 13
小哥 22
小哥哥 3
小唱 4
小商品 8
小商品生產 1
小商品經濟 0
小喇叭 44
小喇叭手 5
小喜 32
小嘍囉 25
小器 10
小嚐 0
小四 34
小回 12
小圈 9
小圈圈 7
小圈子 1
小圈子 1
小國 96
小國家 6
小國寡民 0
小國民 0
小國王 1
小圓 44
小圓圈 8
小地主 3
小坡 3
小坪 41
小坪數 21
小型 2473
小型化 61
小城 228
小堂哥 0
小堂妹 1
小堂姊 0
小堂姊夫 0
小堂嫂 0
小堂弟 3
小堆 1
小報 71
小場 0
小場 0
小場地 0
小場地 0
小塊 73
小塊頭 0
小塊頭 0
小壺 1
小壽星 0
小夜曲 45
小夜班 3
小夥子 17
小夥子 17
小天地 336
小女 277
小女人 74
小女子 305
小妖怪 6
小妖精 10
小妞 38
小妮子 84
小妹 431
小妹夫 0
小妹妹 109
小妾 0
小姊夫 6
小姊姊 23
小姊姊 23
小姐 5788
小姑 9
小姑丈 0
小姑姑 1
小姑娘 583
小姑媽 0
小姑獨處 7
小姓 1
小姨 3
小姨丈 0
小姨媽 0
小姨子 9
小姪 5
小娃 17
小婦人 24
小婿 3
小媳婦 11
小嬸 3
小子 741
小子 741
小字 46
小季 1
小孩 3741
小孩兒 11
小孩兒 11
小孩子 656
小學 2508
小學校 62
小學生 381
小宗 30
小客車 0
小宴驚變 1
小家 145
小家子 1
小家子氣 5
小家庭 64
小家碧玉 2
小寒 8
小寫 315
小寫字母 25
小寶 397
小寶寶 80
小寺 1
小封 1
小將 84
小小 1557
小小的 915
小小地 915
小屋 1000
小島 302
小島嶼 5
小巢 4
小工 24
小工人 1
小巧 111
小巧玲瓏 10
小巫 9
小巫 9
小巫見大巫 13
小巫見大巫 13
小巷 47
小巷子 12
小市民 169
小師父 5
小帽 10
小帽子 0
小帽子 0
小幅 208
小幅度 33
小年夜 1
小店 101
小康 73
小康之家 2
小康家庭 7
小廚 71
小廚房 10
小廝 14
小廠 44
小建 81
小引 3
小弟 2465
小弟弟 110
小張 81
小徑 282
小得 26
小得 26
小心 2686
小心眼 17
小心翼翼 116
小心謹慎 37
小惠 88
小意思 8
小憩 42
小成 26
小我 41
小截 1
小戰 1
小戶 2
小戶人家 0
小房 9
小房間 44
小手 119
小手帕 2
小才大用 1
小才子 1
小批 6
小抄 19
小把戲 25
小抽 0
小拇指 2
小拍 0
小拜 1
小拜拜 0
小括號 1
小拳 0
小拳頭 0
小拼 0
小指 48
小挫 9
小排 149
小排場 0
小排場 0
小提琴 672
小提琴手 7
小攤 18
小支 14
小敗 5
小教室 16
小教師 45
小教授 11
小教父 0
小數 106
小數字 5
小數目 5
小數點 79
小方 33
小於 606
小族 1
小旦 14
小明 545
小星 97
小星星 48
小春 125
小時 6375
小時了了 7
小時候 584
小暑 6
小暴君 0
小曲 24
小書 56
小會 20
小月 58
小有 42
小有作為 0
小有成就 7
小朋友 3819
小朝廷 4
小木屋 246
小本 23
小本書 3
小本生意 4
小本經營 8
小材大用 0
小東 67
小板 68
小枝 17
小架 2
小柱 0
小柱子 2
小株 316
小格 18
小桌 10
小桌子 8
小條 11
小棟 3
小椅子 6
小椅子 6
小業主 0
小楷 13
小楷筆 2
小樣 33
小樹 79
小橋 71
小橋流水 17
小櫃子 2
小歇片刻 0
小步 36
小步舞曲 16
小段 264
小毛 99
小毛病 26
小毛頭 13
小毛驢 6
小民 86
小氣 67
小氣候 0
小氣鬼 3
小水 31
小河 54
小波浪 0
小波浪 0
小洞 48
小派 31
小海龜 8
小淘氣 7
小港 442
小溝 3
小溪 97
小滴 7
小滿 19
小演員 4
小漲 66
小潮 2
小火 78
小炒 16
小照 6
小熊 448
小熊座 18
小熊星座 3
小熊貓 1
小燕子 133
小燕子 133
小片 14
小牌 6
小牌明星 0
小牌歌星 0
小牛皮 18
小犧牲 0
小犬 31
小狗 710
小狗狗 56
小猩猩 3
小獅 18
小獅子 10
小獅子 10
小獎 12
小玉西瓜 6
小王 127
小王子 0
小玩意 16
小玩意兒 7
小玩意兒 7
小玩藝 1
小班 500
小班制 81
小琉球 86
小瓣 0
小瓶 15
小瓶酒 0
小生 98
小生意 12
小生產 1
小生產者 0
小產 15
小疊 0
小病 31
小癟三 0
小登科 4
小發 11
小發財 2
小白 419
小白臉 14
小白菜 22
小白鼠 47
小的 647
小盒 5
小盒子 9
小盒子 9
小盤 97
小看 158
小眼睛 16
小眾 49
小眾文化 2
小睡 22
小石子 25
小石頭 11
小破 1
小碟 7
小禮堂 11
小秋收 0
小秘密 63
小種 7
小穗 6
小站 3640
小章 14
小童 58
小笨蛋 230
小算盤 239
小箱 2
小箱子 1
小箱子 1
小節 157
小篆 16
小篇 113
小篇幅 0
小籠 7
小籠包 95
小籠子 2
小米 157
小米粥 2
小粒 18
小粗坑 3
小系 0
小紅莓 1
小組 6994
小組會議 312
小組討論 49
小組長 113
小結 42
小綠島 0
小羊 133
小美 274
小老婆 20
小老師 49
小老弟 17
小老虎 13
小老頭 29
小老頭兒 0
小老頭兒 0
小老鼠 94
小考 163
小考小玩 0
小耳朵 62
小耳朵 62
小聰明 25
小聲 123
小肚 24
小肚子 2
小股 4
小腦 55
小腳 38
小腳趾頭 0
小腸 150
小腹 63
小腿 156
小腿肉 0
小臉 15
小臉蛋 4
小臣 0
小舅 13
小舅媽 1
小舅子 4
小舅子 4
小舅母 0
小舅舅 2
小興安嶺 8
小船 120
小艇 45
小色狼 18
小花 271
小花樣 2
小花臉 1
小茅屋 7
小草 150
小菜 131
小華 121
小葉 206
小蔥 5
小蘇打 40
小蘋果 27
小蘿蔔頭 0
小虎 63
小虎隊 10
小處 19
小號 72
小蛇 19
小蜜蜂 51
小蜻蜓 12
小螞蟻 90
小蟲 219
小蟲子 13
小行星 77
小行星帶 0
小行板 1
小街 8
小表哥 1
小表妹 8
小表姊 0
小表姊夫 0
小表嫂 0
小表弟 5
小補 16
小襯衣 0
小規模 144
小覷 34
小角 22
小角 22
小角色 15
小角色 15
小解 9
小記 139
小評 4
小說 4161
小說家 125
小說月報 0
小說欄 0
小調 148
小讓 1
小象 26
小象隊 1
小豬 317
小貓 320
小財主 0
小財神 3
小貧 0
小貧戶 0
小販 49
小買賣 2
小費 137
小資產階級 2
小賊 0
小賭 3
小賺 24
小超人 3
小跌 69
小跑 2
小跑步 6
小路 153
小跳 4
小車 111
小車子 2
小輩 11
小輸 3
小轉 3
小轉彎 0
小轎車 33
小辮兒 0
小辮兒 0
小辮子 3
小農 10
小農經濟 18
小迷糊 171
小過 40
小道 31
小道消息 97
小道理 1
小邊 4
小部 46
小部份 51
小部分 24
小酌 21
小醜 0
小量 111
小金 134
小針 35
小鈔 2
小鋪 10
小舖 11
小鋼珠 7
小錢 152
小錢幣 0
小錯 7
小錯誤 9
小鍋 4
小鎮 1212
小鎮風光 22
小開 303
小間 84
小陘 0
小除夕 1
小陽春 9
小隊 319
小隊人馬 0
小隊長 65
小階 1
小隻 7
小雁 6
小雅 78
小集 56
小集團 5
小雞 481
小雨 264
小雪 83
小電影 1
小電腦 24
小電視 3
小震 0
小霧 111
小靜脈 2
小靜脈 2
小鞋子 0
小鞋子 0
小項 20
小頭 25
小頭小臉 1
小頭鬼 0
小題 23
小題大作 28
小題大做 0
小題目 1
小風 51
小飛俠 91
小食 7
小食部 0
小飲 1
小餅 28
小馬 264
小馬哥 214
小騙子 0
小驢 6
小驢子 2
小體 16
小高爐 1
小鬍子 15
小鬼 171
小鬼當家 29
小鬼難纏 1
小鬼頭 6
小魚 410
小魚吃大魚 0
小鳥 236
小鳥依人 9
小鳥蛋 0
小鴨 36
小鴨子 15
小麥 273
小麥粉 6
小麥粥 0
小麵包 2
小麻煩 9
小黃 185
小黃瓜 1
小黃魚 2
小點 66
小點心 27
小鼓 12
小鼓手 9
小齒輪 12
小龍 322
尐 7
少 8356
少 8356
少上 20
少不了 121
少不得 15
少不更事 5
少不更事 5
少之又少 47
少了 612
少付 3
少佐 0
少來 35
少先隊 1
少出 18
少加 3
少印 3
少去 17
少叫 3
少吃 94
少吃多餐 0
少吊 0
少吸 1
少壯 13
少壯不努力 2
少壯派 2
少夫 6
少夫人 2
少夫老妻 0
少女 2260
少女時代 13
少女時期 3
少女裝 0
少奶奶 17
少婦 56
少子 3
少安勿躁 0
少宗 2
少將 90
少將軍官 0
少尉 61
少尉軍官 2
少小 12
少少 33
少帥 0
少年 3321
少年人 27
少年先鋒 3
少年先鋒隊 2
少年宮 1
少年時代 14
少年時期 12
少年法庭 46
少年犯 58
少年組 27
少年老成 4
少年郎 20
少年隊 8
少引 2
少得 56
少得 306
少惹 0
少惹是非 0
少惹閒事 0
少掉 18
少掌櫃 0
少採 6
少教 1
少數 1552
少數人 148
少數服從多數 10
少數民族 234
少數黨 4
少於 374
少時 42
少時 42
少有 320
少東 27
少林 151
少林寺 80
少架 5
少校 99
少校軍官 0
少根筋 1
少棒 68
少棒賽 12
少棒隊 14
少洗 1
少煩 1
少爺 170
少爺兵 0
少猜 0
少男 73
少的 306
少睡 8
少禮 0
少笑 4
少管 11
少管閒事 1
少翻 1
少背 0
少被 28
少裝 2
少見 309
少見多怪 6
少許 401
少說 73
少說兩句 0
少說幾句 1
少說廢話 5
少調 1
少買 24
少賣 2
少量 460
少量多餐 26
少釣 1
少間 15
少間 15
少陪 2
少陽 8
少頃 3
少點 15
尒 38
尓 0
尔 0
尕 82
尖 1062
尖兵 370
尖刀 15
尖利 6
尖刻 3
尖刻話 0
尖削 3
尖叫 183
尖嘴 23
尖嘴薄舌 0
尖團字 0
尖塔 16
尖子 4
尖山 282
尖岬 0
尖峭 1
尖峰 421
尖晶石 4
尖沙咀 22
尖牙 1
尖瘦 2
尖的 38
尖石 180
尖端 563
尖端科技 285
尖細 11
尖聲 14
尖聲高叫 0
尖臍 0
尖酸 10
尖酸刻薄 9
尖銳 179
尖銳化 1
尖頂 7
尖頭 22
尖點 0
尗 0
尘 0
尙 0
尚 4607
尚且 84
尚以 6
尚佳 36
尚可 400
尚在 425
尚待 254
尚志 58
尚方寶劍 0
尚書 105
尚書省 0
尚有 1242
尚有可為 0
尚未 4021
尚武 55
尚武精神 0
尚氣 0
尚無 858
尚無不可 1
尚缺 17
尚能 66
尚要 16
尚輸 0
尚需 138
尚非 18
尚須 179
尚饗 3
尛 0
尛 0
尛 0
尛 0
尞 0
尞 0
尞 0
尟 0
尠 0
尢 21
尣 0
尤 2216
尤佳 229
尤其 3374
尤其在 567
尤其是 3127
尤其會 4
尤其甚者 2
尤其能 7
尤加利樹 8
尤有 21
尤有甚者 24
尤有進者 7
尤為 82
尤物 44
尤甚 19
尤而效之 0
尤雲殢雨 0
尤須 33
尥 75
尥蹶子 0
尨 150
尨 150
尩 0
尪 278
尫 0
尬 235
尯 0
尰 10
就 32275
就上 88
就上來 20
就上去 1
就下 44
就下來 4
就下去 2
就不 1956
就不 1956
就不是 242
就不會 559
就不能 366
就中 36
就事 4
就事論事 48
就任 310
就任典禮 2
就任演說 0
就位 33
就住 83
就作 38
就來 442
就便 11
就做 98
就像 3242
就出 117
就出來 18
就出去 12
就到 330
就勢 7
就去 398
就叫 188
就吃 42
就和 272
就問 153
就回 84
就回來 11
就回去 11
就因 88
就因為 171
就在 2898
就地 178
就地取材 12
就地正法 15
就地解決 2
就坐 74
就好 1894
就學 826
就學年齡 2
就寢 38
就寢時間 5
就將 335
就席 0
就座 17
就得 521
就從 290
就怕 61
就愛 62
就應 230
就應該 268
就打 119
就把 806
就拿 117
就拿出 14
就拿到 12
就教 77
就於 16
就是 23667
就是了 239
就是說 188
就會 5268
就有 2598
就木 0
就業 5416
就業輔導 501
就正 26
就此 459
就此打住 23
就此結束 10
就求 7
就為 144
就理 7
就用 296
就用到 0
就用在 0
就由 143
就當 193
就當作 73
就等 120
就等到 1
就算 1655
就算是 280
就範 15
就緒 73
就義 1
就職 272
就職儀式 1
就職典禮 22
就與 184
就著 44
就要 2440
就調 4
就調 4
就講 30
就讀 1763
就讀於 161
就讓 810
就走 157
就起來 18
就近 351
就逮 4
就進 60
就進來 26
就進去 8
就過 63
就過來 16
就過去 23
就醫 961
就需要 147
就養 45
就餐 0
尳 3
尵 0
尶 0
尶 0
尷 222
尷尬 332
尸 146
尸位素餐 2
尸僵 0
尸斑 0
尸身 10
尸首 5
尸體 8
尹 1139
尺 1696
尺地 2
尺子 0
尺寸 3291
尺寸千里 0
尺布斗粟 0
尺幅 5
尺幅千里 0
尺度 415
尺書 8
尺牘 3
尺狀 0
尺短寸長 0
尺碼 62
尺素 6
尻 27
尼 5862
尼克 233
尼克松 8
尼克森 52
尼克鬆 0
尼加拉瀑布 1
尼加拉瓜 208
尼古丁 58
尼古拉 29
尼國 90
尼姑 59
尼姑庵 1
尼安得塔 2
尼庵 0
尼採 0
尼斯 276
尼日利亞 2
尼日爾 1
尼泊爾 181
尼科西亞 0
尼羅河 72
尼赫魯 7
尼采 98
尼龍 351
尼龍絲 18
尼龍線 6
尼龍繩 5
尼龍襪 0
尽 0
尾 3444
尾 3444
尾大不掉 8
尾巴 247
尾巴 247
尾巴 247
尾巴 247
尾形 8
尾數 30
尾椎 15
尾椎骨 2
尾標 0
尾款 23
尾款付清 1
尾段 15
尾氣 5
尾燈 23
尾牙 65
尾盤 159
尾礦 2
尾端 76
尾翼 25
尾聲 104
尾舵 2
尾邊 3
尾部 54
尾隨 53
尾隨者 0
尾音 10
尾鰭 73
尿 19
尿 1905
尿壺 4
尿失禁 1
尿崩 0
尿崩症 2
尿布 127
尿床 42
尿急 14
尿意 5
尿斗 0
尿桶 3
尿毒 40
尿毒症 143
尿水 6
尿流屁滾 0
尿液 409
尿液 409
尿溼 0
尿濕 5
尿素 143
尿肥 0
尿胞 0
尿血 5
尿道 243
尿道炎 37
尿酸 262
局 14884
局促 3
局促不安 0
局內 32
局勢 336
局外 16
局外人 25
局數 23
局裡 19
局部 1021
局部地區 18
局部性 48
局部痲痹 1
局部痲痺 1
局部麻痹 1
局部麻痺 1
局長 1743
局限 66
局限性 14
局限於 68
局面 589
局騙 0
屁 846
屁滾尿流 2
屁眼兒 0
屁眼兒 0
屁股 333
屁股痛 3
屄 34
屄 34
居 1
居 5212
居下風 5
居不求安 0
居中 117
居住 2263
居住權 2
居先 6
居冠 0
居功 22
居功厥偉 16
居喪 2
居士 739
居多 312
居多數 12
居奇牟利 0
居宅 3
居安思危 18
居官 2
居室 58
居家 1415
居心 44
居心不良 9
居心叵測 13
居所 103
居於 220
居業 4
居次 53
居民 2366
居民點 1
居然 1180
居然在 39
居然是 69
居然會 32
居然有 53
居留 398
居留地 12
居留權 83
居留證 205
居禮夫人 21
居積 0
居處 29
居裡 1
居里 24
居間 51
居首 75
居高不下 114
居高臨下 37
屆 4256
屆時 1140
屆期 96
屆滿 555
屆滿時 31
屆臨 9
屇 13
屈 70
屈 700
屈伸 10
屈光度 10
屈光性 4
屈原 70
屈就 20
屈就於 1
屈尺 20
屈強 0
屈從 24
屈打成招 7
屈折 24
屈指 21
屈指一算 1
屈指一算 1
屈指可數 34
屈曲 64
屈服 77
屈服於 29
屈枉 2
屈氏體 0
屈膝 33
屈膝投降 0
屈起 2
屈辱 38
屈辱 38
屈辱性 0
屈辱性 0
屈駕 0
屉 0
届 0
届 0
屋 7342
屋主 215
屋內 0
屋基 1
屋子 208
屋子裡 48
屋宇 27
屋架 8
屋梁 5
屋檐 6
屋漏 9
屋漏偏逢 1
屋漏偏逢連夜雨 20
屋瓦 29
屋簷 63
屋簷下 77
屋脊 44
屋脊 44
屋脊爪 0
屋脊爪 0
屋舍 26
屋裡 68
屋門 12
屋面 21
屋面板 0
屋頂 782
屋頂花園 11
屋頂違建 0
屋頭 1
屌 217
屍 495
屍僵 0
屍斑 2
屍衣 2
屍身 19
屍陀林 1
屍首 90
屍骨 17
屍骨未寒 0
屍骸 12
屍體 475
屎 460
屎尿 5
屎蚵蜋 0
屏 211
屏 2111
屏幕 43
屏息 31
屏息以待 9
屏息靜觀 0
屏東 4042
屏東市 2398
屏東師範學院 296
屏東教育大學 1
屏東縣 8246
屏東農專 24
屏條 0
屏棄 0
屏極 3
屏氣 0
屏氣凝神 5
屏絕 0
屏蔽 42
屏蔽著 0
屏藩 3
屏退 1
屏除 12
屏障 91
屏風 291
屐 31
屐齒 0
屑 394
屒 0
屓 0
屔 6
展 14057
展佈 1
展出 1405
展品 267
展延 352
展弦比 2
展性 9
展望 4018
展望未來 174
展期 216
展現 1853
展現出 189
展示 8954
展示中心 416
展示出 11
展示會 218
展緩 8
展緩期限 0
展翅 90
展翅高飛 27
展覽 3732
展覽品 18
展覽室 154
展覽廳 37
展覽會 358
展覽會場 44
展覽會場 44
展覽館 249
展讀 8
展轉 3
展銷會 2
展開 9085
展開來 5
展開式 5
展開攻勢 4
展開討論 2
展露 111
展露出 8
展露出來 2
展露頭角 18
展顏 6
屖 2
屘 8
屙 53
屚 0
屛 0
屛 0
屛 0
屜 17
屜 17
屝 7
属 0
属 0
屟 0
屟 0
屠 339
屠刀 7
屠城 13
屠場 5
屠場 5
屠夫 28
屠宰 160
屠宰場 79
屠宰場 79
屠宰業 10
屠宰稅 2
屠戮 2
屠戶 9
屠格涅夫 2
屠殺 119
屠滅 0
屠販 1
屠龍 154
屡 0
屢 352
屢仆屢起 0
屢勸不聽 5
屢屢 64
屢建奇功 4
屢戰屢敗 4
屢敗屢戰 4
屢教不改 1
屢教不改 1
屢次 113
屢見不鮮 36
屢試不爽 15
屣 41
層 6107
層出 15
層出不窮 133
層壓 9
層壓板 0
層層 239
層層剝削 2
層層包圍 4
層層疊疊 25
層峰 36
層峰會議 0
層巒疊嶂 4
層樓 295
層次 1875
層次分明 21
層次分類法 0
層次等級 4
層流 19
層理 4
層級 0
層面 1184
層高 23
履 358
履帶 14
履帶式 9
履新 16
履歷 572
履歷表 1518
履約 492
履行 467
履行任務 0
履行義務 17
履行諾言 3
履險如夷 4
屧 7
屨 68
屩 37
屪 88
屫 0
屬 892
屬 8929
屬下 144
屬僚 0
屬名 27
屬員 17
屬國 63
屬地 61
屬實 216
屬對 11
屬性 1649
屬意 64
屬意於 2
屬於 6962
屬概念 0
屬目 3
屬相 8
屭 71
屮 4
屯 1
屯 14
屯 1411
屯兵 12
屯墾 12
屯墾員 0
屯子 1
屯田 15
屯田制 0
屯田客 0
屯積 13
屯籍 0
屯糧 2
屯紮 3
屯聚 0
屯聚處 0
屯裡 0
屯里 3
屯駐 2
屰 0
屰 0
山 16715
山上 697
山上人家 0
山下 230
山下人家 1
山不轉路轉 0
山丘 127
山中 387
山人 68
山人自有妙計 0
山光水色 28
山凹 12
山前 33
山前山後 3
山勢 59
山包 27
山區 1677
山南 32
山南海北 0
山口 102
山右 3
山名 10
山君 3
山嘴 3
山園 12
山地 964
山地人 13
山地同胞 9
山地氣候 1
山地舞 7
山地部落 19
山坡 279
山坡土 0
山坡地 1592
山坡路 0
山坳 9
山城 386
山域 7
山壁 0
山壑 0
山大王 10
山妻 0
山姆 136
山姆叔叔 2
山寨 55
山居 52
山山水水 3
山岡 131
山岩 288
山岳 186
山峰 113
山峽 4
山崎 70
山崖 43
山崗 56
山崩 113
山崩地裂 5
山嵐 37
山嶺 56
山嶽 9
山巒 79
山巒起伏 7
山巔 22
山川 137
山川相隔 0
山川險阻 1
山左 2
山形 62
山後 43
山徑 49
山房 9
山明水秀 39
山東 324
山東丘陵 4
山東人 8
山東半島 8
山東大鼓 0
山東省 70
山林 505
山查 2
山根 38
山梁 2
山棲谷飲 0
山楂 31
山楂片 0
山楊 1
山樑 0
山歌 136
山毛櫸 40
山民 11
山水 724
山水畫 40
山水相連 0
山河 282
山泉 73
山洞 115
山洪 33
山洪暴發 7
山海 148
山海經 15
山海關 5
山溝 22
山澗 30
山炮 0
山牆 38
山牧季移 0
山珍 17
山珍海味 15
山產 70
山盟海誓 15
山神 12
山窪 1
山窮水盡 7
山窮水盡疑無路 4
山竽 2
山系 47
山線 162
山羊 269
山羊肉 1
山羊鬍子 3
山羌 1
山群 68
山胞 161
山脈 413
山脈 413
山脊 40
山脊 40
山腰 80
山腰上 13
山腳 282
山腳下 23
山腹 12
山色 38
山艾樹 0
山芋 5
山花 23
山茱萸 3
山茶 107
山莊 1746
山蔬 1
山藥 97
山藥蛋 0
山蘇 1
山裡 37
山裡紅 0
山西 190
山西人 1
山西省 42
山谷 305
山谷地 5
山豬 97
山貓 21
山貨 55
山賊 0
山路 591
山轎 1
山道 31
山邊 37
山鄉 1373
山野 169
山金 21
山門 29
山間 79
山陰 13
山陵 12
山陽 20
山雀 141
山雞 21
山難 65
山雨 6
山雨欲來 17
山雨欲來風滿樓 7
山頂 376
山頂洞人 2
山頭 177
山風 64
山驛 0
山高 197
山高水低 2
山高水遠 0
山高水長 15
山鵲 0
山鵲 0
山鷸 1
山麓 88
山鼠 5
屳 6
屴 58
屵 0
屸 0
屹 61
屹立 66
屹立不動 4
屹立不搖 67
屺 86
屻 7
屼 90
屽 0
屾 320
岀 0
岄 0
岅 0
岆 675
岇 0
岈 796
岉 55
岊 152
岋 34
岌 142
岌岌不保 2
岌岌可危 41
岍 535
岎 0
岏 80
岐 359
岐伯 1
岐周 1
岐婆 1
岐山 1
岐山縣 1
岐嶷 1
岐路 13
岐黃 3
岑 309
岑寂 6
岑春 1
岒 46
岓 17
岔 204
岔口 3
岔子 1
岔路 77
岔道 1
岔開 15
岕 48
岜 0
岝 13
岞 0
岞 0
岟 16
岠 10
岡 173
岡 1733
岡山 529
岡山鎮 802
岡巒 0
岡巒起伏 0
岡底斯 0
岡底斯岡 0
岡底斯岡石 0
岡比亞 2
岡石 31
岡陵 2
岢 12
岣 15
岤 3
岥 8
岦 15
岧 5
岨 6
岩 2832
岩壁 104
岩屑 3
岩層 110
岩岸 23
岩床 3
岩心 98
岩洞 35
岩流圈 0
岩溶 4
岩漿 38
岩石 638
岩礁 52
岩薔薇 1
岩鹽 5
岪 5
岫 315
岬 57
岬角 16
岭 36
岮 3
岯 6
岰 10
岱 682
岲 0
岳 2508
岳丈 4
岳家 7
岳州 1
岳廟 0
岳武穆 1
岳母 33
岳父 96
岳陽 20
岳陽樓 10
岳飛 45
岴 0
岵 6
岶 4
岷 120
岷江 1
岸 2253
岸上 115
岸然 1
岸然道貌 0
岸邊 137
岸頭 4
岹 0
岺 0
岻 0
峀 0
峁 0
峆 42
峇 550
峈 1386
峉 241
峊 205
峋 7
峋 7
峌 3
峍 0
峎 130
峏 68
峐 7
峑 0
峒 72
峒 72
峓 106
峔 179
峖 117
峗 3
峘 4
峙 122
峙立 2
峚 314
峛 4
峜 0
峝 0
峝 0
峞 264
峟 74
峡 0
峢 0
峥 0
峥 0
峦 0
峨 91
峨嵋 97
峨嵋山 9
峨嵋派 5
峨眉 85
峨眉山 15
峩 0
峪 54
峪口 2
峫 0
峫 0
峬 18
峭 42
峭壁 104
峭寒 2
峭急 0
峭拔 1
峭直 0
峭立 2
峮 14
峯 0
峰 6164
峰值 32
峰峰 9
峰巒 10
峰巒翠疊 0
峰巒起伏 2
峰迴 1
峰迴路轉 33
峰點 0
峱 9
峲 0
峳 0
峴 63
峴港 13
峵 0
峵 0
島 22350
島上 459
島內 76
島內居民 0
島國 83
島外 14
島外居民 0
島嶼 465
島民 32
峷 16
峸 90
峹 13
峺 0
峻 605
峻厲 0
峻岭 0
峻峭 5
峻嶺 29
峻德 8
峻急 0
峻法 1
峻節 0
峼 0
峽 2099
峽口 7
峽彎海岸 0
峽江 4
峽灣 49
峽谷 259
峾 0
峿 13
崀 7
崁 776
崁頂 158
崆 11
崆峒 6
崆峒山 1
崇 2277
崇佑 3
崇佑企專 1
崇信 25
崇光 119
崇奉 12
崇尚 122
崇山 8
崇山峻嶺 7
崇德 385
崇拜 588
崇拜偶像 10
崇拜者 16
崇敬 57
崇明 90
崇本務實 1
崇本抑末 0
崇法務實 4
崇洋 6
崇洋媚外 5
崇禎 23
崇禎帝 0
崇閎 0
崇高 185
崇高理想 10
崈 0
崉 0
崊 0
崋 235
崋 235
崌 18
崍 9
崎 1861
崎嶇 75
崎嶇不平 8
崎嶇難走 0
崏 11
崐 0
崑 729
崑山 510
崑山工專 11
崑山科技大學 1
崑山高級中學 1
崑崙 86
崑崙山 3
崑崙山脈 2
崑崙山脈 2
崑曲 320
崑腔 10
崒 3
崓 0
崔 878
崔巍 2
崔瑗 0
崔璦 0
崔述 0
崕 0
崕 0
崕 0
崖 290
崖壁 12
崖鹽 0
崗 1507
崗 1507
崗亭 9
崗亭 9
崗位 178
崗位 178
崗位責任 0
崗位責任 0
崗位責任制 0
崗位責任制 0
崗哨 13
崗哨 13
崗地 3
崗地 3
崗子 0
崗子 0
崗樓 0
崗樓 0
崗警 3
崗警 3
崘 0
崙 60
崙背 50
崚 16
崚嶒 0
崛 75
崛起 196
崜 0
崝 101
崞 25
崟 11
崠 119
崡 0
崢 35
崢嶸 20
崣 3
崤 8
崥 7
崦 9
崧 293
崨 21
崩 230
崩坍 86
崩塌 253
崩壞 27
崩殂 0
崩潰 239
崩盤 86
崩落 26
崩裂 17
崩解 50
崩跌 0
崪 0
崫 0
崫 0
崬 0
崮 11
崯 0
崰 4
崱 7
崲 5
崳 9
崴 548
崵 1
崶 9
崷 16
崸 11
崹 2
崺 3
崻 0
崼 5
崽 13
崽子 4
崾 0
崿 6
嵀 3
嵁 5
嵂 2
嵃 6
嵄 0
嵅 4
嵅 4
嵆 0
嵇 12
嵇康 12
嵈 0
嵉 4
嵊 9
嵋 16
嵌 574
嵌 574
嵌 574
嵌入 0
嵌入式 0
嵌塊 0
嵌石 0
嵌鑲 3
嵍 0
嵍 0
嵎 9
嵏 0
嵐 562
嵐煙波影 0
嵐煙波影 0
嵑 1
嵑 1
嵒 4
嵓 0
嵔 0
嵕 5
嵖 0
嵗 0
嵙 39
嵛 0
嵞 0
嵟 0
嵠 0
嵠 0
嵡 0
嵢 2
嵣 0
嵤 0
嵥 4
嵦 0
嵧 1
嵨 5
嵩 261
嵩山 42
嵩縣 0
嵪 0
嵪 0
嵫 4
嵬 11
嵭 0
嵮 0
嵯 7
嵯峨 5
嵰 0
嵱 1
嵲 2
嵳 0
嵴 0
嵷 36
嵸 0
嵹 90
嵺 26
嵻 0
嵼 29
嵽 83
嵾 27
嵿 76
嶀 194
嶀 194
嶁 36
嶂 202
嶃 0
嶄 261
嶄新 362
嶄現 0
嶄露 0
嶄露 0
嶄露頭角 25
嶄露頭角 25
嶅 0
嶆 230
嶇 32
嶈 28
嶉 40
嶊 56
嶋 0
嶌 0
嶍 20
嶏 0
嶏 0
嶐 0
嶑 0
嶒 8
嶓 14
嶔 18
嶕 4
嶖 0
嶖 0
嶗 10
嶘 0
嶙 22
嶙峋 21
嶚 22
嶛 0
嶜 18
嶝 32
嶞 14
嶟 7
嶠 26
嶡 13
嶢 15
嶣 0
嶣 0
嶤 0
嶦 0
嶦 0
嶧 619
嶨 487
嶩 4
嶪 2992
嶫 0
嶬 0
嶭 45
嶮 694
嶯 95
嶰 25
嶱 3
嶲 1440
嶳 0
嶳 0
嶴 221
嶵 311
嶷 8
嶸 167
嶹 0
嶺 1407
嶺北 1
嶺南 62
嶺東 128
嶺東商專 173
嶺線 0
嶻 0
嶻 0
嶼 745
嶽 329
嶾 0
巀 49
巁 0
巁 0
巂 4
巂 4
巃 57
巄 0
巆 6
巇 8
巈 0
巉 6
巉巖 1
巊 0
巋 40
巌 0
巍 80
巍峨 44
巍巍 27
巍巍泰山 0
巍然 6
巍然矗立 3
巎 0
巏 43
巐 0
巑 8
巒 224
巓 0
巔 95
巔峰 162
巔峰狀態 4
巕 6
巖 452
巖居 0
巖居穴處 0
巖層 2
巖心 0
巖洞 2
巖溶 0
巖漿 0
巖石 4
巖礁 0
巖穴 0
巖鹽 0
巗 0
巘 120
巙 0
巚 0
巛 0
巜 0
川 1799
川北 5
川堂 0
川康公路 0
川式 1
川式料理 1
川江 6
川流 18
川流不息 26
川漢鐵路 2
川燙 1
川芎 16
川芎 16
川菜 49
川貝 0
川資 2
川軍 1
川鹽 0
州 3573
州名 17
州官 1
州官放火 3
州牧 9
州界 3
州立 111
州立大學 970
州縣 13
州郡 18
州里 4
州長 57
州際 10
州際公路 1
巟 11
巠 100
巡 1072
巡佐 0
巡哨 3
巡回 9
巡回醫療 0
巡夜 1
巡官 11
巡察 50
巡察隊 2
巡幸 0
巡弋 89
巡弋飛彈 1
巡按 18
巡按使 1
巡捕 3
巡撫 39
巡更 1
巡查 46
巡檢 15
巡洋艦 41
巡游 2
巡禮 743
巡線追蹤 0
巡緝 26
巡航 102
巡航飛彈 0
巡行 7
巡視 292
巡警 14
巡迥 9
巡迴 1100
巡迴檢查 3
巡迴演出 90
巡迴演唱 41
巡迴車 11
巡迴醫療 23
巡遊 21
巡邏 220
巡邏機 0
巡邏箱 2
巡邏艇 20
巡邏車 6
巡邏隊 17
巡防 26
巡防艦 55
巢 517
巢毀卵破 0
巢湖 2
巢穴 41
巢窟 1
巢脾 0
巣 0
巤 0
工 19861
工事 148
工人 883
工人區 0
工人日報 0
工人運動 6
工人階級 12
工件 187
工作 34279
工作上 425
工作下去 1
工作人員 1799
工作制度 3
工作台 87
工作團 44
工作地 22
工作大隊 6
工作天 176
工作室 7590
工作小組 1039
工作日 24
工作會 222
工作服 45
工作狂 0
工作站 1704
工作組 73
工作者 1103
工作船 0
工作裝 0
工作褲 0
工作證 17
工作起來 6
工作量 111
工作隊 80
工作面 2
工傷 9
工價 27
工兵 30
工兵排 2
工兵營 2
工兵連 0
工具 19388
工具列 363
工具書 506
工具機 572
工具箱 467
工具鋼 30
工分 15
工分值 0
工力 3
工務 690
工務局 577
工匠 37
工區 84
工友 500
工商 43240
工商專科 2055
工商時代 3
工商時報 6420
工商業 387
工商業界 47
工商界 133
工商社會 45
工商管理 178
工商聯 40
工商職業 340
工地 559
工地秀 6
工場 52
工場 528
工場手工 0
工場手工 0
工場手工業 1
工場手工業 1
工夫 192
工字 41
工字型 7
工字鋼 1
工學 344
工學院 2802
工宣隊 0
工寮 103
工專 533
工展會 5
工巧 5
工序 27
工廠 6698
工廠權 0
工役 0
工房 139
工技 243
工技學院 128
工效 4
工整 42
工數 0
工料 38
工於 21
工於心計 1
工日 8
工時 178
工會 3375
工期 137
工本 10
工本費 321
工業 23460
工業中心 22
工業化 218
工業化學 35
工業區 2900
工業原料 161
工業品 29
工業國 111
工業國家 148
工業園區 1058
工業城 18
工業基地 9
工業大學 116
工業學校 11
工業專科 228
工業局 1111
工業戰線 1
工業技術 1440
工業政策 20
工業時代 17
工業港 15
工業無產階級 0
工業用電 39
工業界 242
工業病 1
工業社會 69
工業職業 315
工業設計 628
工業部 46
工業部門 44
工業電視 0
工業革命 97
工業體系 10
工欲善其事 43
工段 4
工況 0
工潮 9
工研所 4
工研院 4743
工礦 88
工礦企業 1
工科 348
工科大學 10
工科學生 20
工程 28739
工程兵 0
工程史 1
工程圖 88
工程塑料 2
工程學 327
工程學系 1710
工程師 3194
工程師學 6
工程數學 1
工程浩大 14
工程界 43
工程系 1133
工程署 7
工程署 7
工程處 738
工程設計 281
工程費 89
工程部 154
工程隊 42
工程項目 109
工種 6
工筆 24
工筆畫 12
工管 0
工細 7
工總 0
工聯 11
工職 50
工薪 4
工藝 1425
工藝品 89
工藝性能 0
工藝流程 4
工藝程序 0
工藝美術 35
工藝課 7
工藝館 14
工蜂 20
工蟻 6
工裝 3
工讀 562
工讀生 630
工資 496
工資勞動 0
工資總額 4
工資附加 0
工資附加費 0
工賊 0
工質 0
工輜兵 0
工農 120
工農兵 5
工農業 5
工農業總產值 2
工農紅軍 0
工農聯盟 2
工運 49
工部 54
工部侍郎 0
工部尚書 0
工錢 15
工長 40
工頭 55
工黨 41
工齡 0
工團 10
工團主義 10
左 5690
左上 94
左上方 39
左上角 108
左下 70
左下方 24
左下角 84
左偏 5
左側 307
左傳 46
左傾 16
左傾份子 0
左券 2
左前 22
左前方 12
左右 4154
左右兩難 0
左右手 43
左右搖擺 11
左右方 0
左右為難 49
左右翼 1
左右逢源 29
左右開弓 10
左司郎中 1
左向 2
左外野 14
左外野手 5
左後 6
左後方 12
左思右想 13
左性子 0
左手 484
左手掌 17
左手邊 26
左掌 21
左撇子 18
左擁右抱 6
左擁右抱 6
左支右絀 0
左方 121
左旁 10
左氏傳 0
左氏春秋 1
左派 87
左派份子 1
左營 785
左營區 259
左移 0
左端 9
左線 3
左翼 45
左耳 23
左腳 72
左臂 38
左臂 38
左舷 7
左行 4
左袒 0
左輪 19
左輪手槍 3
左輪槍 2
左轉 180
左近 1
左述 2
左道 4
左遷 0
左邊 709
左鄰右舍 223
左鍵 0
左鎮 151
左面 17
左顧 5
左顧右盼 19
巧 2699
巧克力 523
巧克力糖 6
巧取豪奪 3
巧合 164
巧固球 1
巧奪天工 18
巧妙 301
巧婦 5
巧婦難為 14
巧媚 1
巧幹 0
巧得很 1
巧思 133
巧手 45
巧猾 0
巧立名目 17
巧藝 64
巧言 9
巧言令色 3
巧言如簧 0
巧計 11
巧詐 0
巧譎 0
巧辭 0
巧辯 3
巧遇 80
巨 6681
巨人 647
巨人隊 60
巨債 3
巨像 6
巨匠 117
巨噬細胞 0
巨型 107
巨大 920
巨奸 0
巨子 12
巨富 14
巨幅 73
巨廈 3
巨廈 3
巨擘 119
巨星 553
巨構 4
巨樹 15
巨款 16
巨流 32
巨浪 50
巨浪滔天 1
巨無霸 40
巨然 20
巨照 1
巨物 7
巨獸 39
巨石 154
巨細 6
巨著 33
巨蛋 0
巨蟒 11
巨蟹 1
巨蟹座 1
巨視 2
巨觀 22
巨變 70
巨資 52
巨賈 12
巨量 54
巨集 442
巨靈 44
巨響 42
巨頭 61
巨額 113
巨鳥 12
巩 0
巫 1195
巫 1195
巫仙 9
巫仙 9
巫士 135
巫女 0
巫婆 194
巫婆 194
巫嫗 0
巫嫗 0
巫山 55
巫山 55
巫峽 15
巫峽 15
巫師 135
巫師 135
巫蠱 6
巫蠱 6
巫術 99
巫術 99
巫醫 32
巫醫 32
差 1
差 1
差 427
差 427
差 427
差 4276
差一點 109
差一點 109
差一點兒 1
差下 2
差不多 704
差不多 704
差不多的 44
差不遠 6
差之千里 6
差之毫厘 0
差之毫厘失之千里 0
差之毫釐 2
差事 49
差些 6
差人 61
差任 0
差使 7
差價 98
差分器 0
差別 897
差別待遇 75
差別費率 8
差勁 650
差動 44
差動滑輪 0
差可 12
差官 4
差度 5
差強人意 45
差役 7
差得 6
差得多 1
差得遠 2
差數 9
差旅 16
差旅費 57
差池 3
差派 31
差異 2426
差異性 147
差的 279
差等 14
差號 0
差距 931
差路 1
差速器 24
差遣 86
差錯 58
差額 1762
差點 510
差點兒 7
差點兒 7
巰 6
己 4432
己二酸 15
己任 47
己所不欲 9
己見 21
己身 114
己飢己溺 0
已 42198
已久 508
已到 343
已因 81
已婚 274
已婚者 20
已定 50
已將 657
已對 185
已對於 0
已就 60
已屬 66
已往 56
已得 49
已得到 23
已故 102
已於 1243
已是 1078
已晚 12
已有 3568
已極 19
已比 18
已然 280
已知 617
已知數 0
已經 16164
已經在 524
已經是 576
已經會 17
已經有 912
已而 12
已達 594
已達到 100
已達成 97
已還 8
已開發國家 156
巳 432
巳時 0
巴 4995
巴不得 39
巴伐利亞 29
巴伐利亞 29
巴兒狗 0
巴勒斯坦 124
巴哈 236
巴哈馬 39
巴哈馬群島 5
巴國 192
巴基斯坦 366
巴基斯坦人 0
巴塞隆納 149
巴士 643
巴士海峽 0
巴士站 8
巴夏禮 0
巴尼 28
巴山 26
巴山虎 0
巴巴 67
巴庫 35
巴拉圭 404
巴拉圭河 0
巴拉松 26
巴拉頓湖 0
巴拉馬 2
巴拿馬 356
巴拿馬共和國 3
巴拿馬地峽 0
巴拿馬運河 40
巴掌 43
巴斯卡 19
巴望 22
巴林 87
巴格達 137
巴比 59
巴比倫 179
巴比合金 0
巴比特 9
巴洛克 146
巴游 0
巴爾喀什 0
巴爾喀什湖 0
巴爾幹 14
巴爾幹半島 24
巴爾幹山 0
巴爾幹山脈 0
巴爾幹山脈 0
巴特 37
巴結 37
巴裡 3
巴西 1135
巴西人 8
巴西利亞 3
巴豆 7
巴貝多 15
巴赫 50
巴里 52
巴里島 103
巴金 1
巴金森氏症 1
巴陵 101
巴陵山莊 15
巴頓 71
巴顏喀喇 0
巴顏喀喇 0
巴顏喀喇山 2
巴顏喀喇山 2
巴馬科 0
巴黎 1617
巴黎人 5
巴黎公社 1
巴黎和會 3
巴黎大學 28
巴黎盆地 0
巶 0
巷 14000
巷口 68
巷名 5
巷子 114
巷弄 46
巷戰 7
巷裡 7
巷議 3
巷道 124
巷陌 0
巸 0
巹 77
巺 0
巺 0
巻 0
巻 0
巻 0
巽 139
巾 303
巾幗 9
巾幗不讓鬚眉 1
巾幗英雄 9
巿 1576
帀 0
币 0
市 52404
市上 128
市中心 261
市井 0
市井小民 22
市井無賴 1
市佔率 0
市價 601
市儈 50
市儈氣 1
市內 456
市公所 310
市公車 22
市分 14
市區 1625
市卒 0
市占率 1
市地 0
市場 3054
市場 30542
市場上 99
市場上 997
市場價格 13
市場價格 137
市場調查 30
市場調查 308
市委 31
市容 98
市容整潔 1
市尺 0
市府 2264
市政 1038
市政報告 0
市政府 828
市政廳 23
市政建設 139
市斤 0
市民 2455
市民住宅 17
市況 92
市況報導 10
市營 42
市營公車 0
市畝 0
市立 4345
市聲 6
市街圖 17
市話 100
市議員 768
市議會 163
市賈 0
市農會 15
市運 3
市運會 2
市道 14
市道小人 0
市郊 102
市里 12
市鎮 520
市長 5945
市集 171
市電 2193
市面 267
市面上 1065
市黨部 74
布 7892
布丁 476
布依族 6
布偶 77
布偶戲 1
布傘 0
布列茲涅 0
布加勒斯特 8
布匹 40
布告 81
布告欄 203
布宜若 1
布宜諾斯艾利斯 35
布局 533
布巾 6
布希 83
布希總統 8
布帛 4
布幕 0
布店 26
布拉姆茲 0
布拉柴維爾 0
布拉格 191
布料 355
布施 173
布景 60
布朗 177
布朗寧 0
布朗運動 3
布梏 0
布條 99
布滿 36
布爾什維克 3
布爾喬亞 6
布特 1
布疋 144
布穀 1
布穀鳥 11
布篷 1
布紋紙 0
布線 1
布置 127
布萊希特 77
布衣 29
布衣之交 0
布衣卿相 1
布衫 8
布袋 380
布袋戲 1887
布袋裝 4
布袋鎮 251
布谷 0
布谷鳥 0
布農 174
布農族 301
布達佩斯 71
布達拉宮 10
布邊 17
布里茲涅夫 2
布防 0
布陣 6
布隆迪 0
布雷 77
布雷希特 77
布面 56
布鞋 38
布頭 4
布頭 4
布魯塞爾 112
布魯諾 7
帄 4
帆 816
帆 816
帆布 325
帆布 325
帆布包 0
帆布包 0
帆布床 0
帆布床 0
帆布椅 1
帆布椅 1
帆布袋 4
帆布袋 4
帆布鞋 7
帆布鞋 7
帆檣 2
帆船 244
帇 0
帉 0
帊 26
帋 0
希 4428
希世 5
希伯來 156
希伯來人 20
希伯來文 60
希冀 60
希噓 0
希圖 0
希奇 22
希奇古怪 3
希有 16
希望 25649
希望無窮 3
希求 14
希法亭 2
希爾 110
希爾伯特 3
希爾頓 99
希特勒 101
希盼 1
希罕 15
希臘 682
希臘人 35
希臘化 2
希臘半島 1
希臘字母 7
希臘文 73
希臘話 1
希臘語 13
帍 0
帎 152
帑 68
帑 68
帑藏 0
帒 0
帓 0
帔 7
帕 1732
帕子 4
帕格尼尼 31
帕爾 28
帕瓦羅第 0
帕米爾 11
帕米爾高原 1
帕金森氏症 1
帖 1390
帖 1390
帖 1390
帖上 7
帖子 21
帖服 0
帗 6
帘 82
帘子 2
帘子布 0
帘子線 0
帘布 0
帙 10
帚 114
帚子 1
帚柄 0
帛 336
帝 2866
帝位 6
帝俄 14
帝俄 14
帝俄時代 1
帝俄時代 1
帝制 20
帝制時代 2
帝君 73
帝國 1419
帝國主義 86
帝國主義分子 0
帝國大廈 20
帝室 3
帝服 0
帝業 4
帝王 231
帝王之相 1
帝王將相 8
帝王蟹 1
帝號 20
帝雉 36
帞 0
帟 86
帠 142
帡 120
帢 120
帣 111
帤 211
帥 18
帥 1874
帥印 1
帥哥 1101
帥氣 120
帨 14
帩 20
帪 0
師 10647
師丈 9
師事 158
師傅 353
師兄 499
師其故智 0
師出無名 5
師友 53
師命 7
師命難違 0
師哥 7
師團 26
師大 2247
師大分部 31
師大路 57
師妹 41
師姊 201
師姐 51
師姑 54
師專 219
師專生 4
師尊 60
師弟 54
師徒 96
師徒合同 0
師心自用 0
師恩 18
師承 44
師旅 1
師母 183
師法 149
師父 1315
師爺 23
師生 2599
師祖 0
師範 739
師範大學 2189
師範學校 70
師範學院 1819
師範教育 94
師範生 32
師表 7
師說 62
師資 6355
師道 65
師部 7
師長 463
師門 19
師院 1808
師院實小 103
帬 0
帬 0
席 2436
席不暇暖 1
席位 122
席勒 1
席卷 0
席地 8
席地而坐 13
席夢思床 0
席子 1
席捲 165
席捲一空 0
席捲天下 0
席次 385
席設 6
席豐履厚 0
席間 28
席面 5
帮 0
帯 0
帰 0
帰 0
帲 0
帳 1639
帳上 44
帳冊 109
帳務 232
帳單 375
帳子 1
帳幔 4
帳幕 61
帳戶 928
帳房 5
帳本 14
帳棚 59
帳款 302
帳目 175
帳篷 94
帳簷 0
帳簿 127
帳籍 0
帳號 6111
帳鉤 0
帳面 141
帳面價值 25
帳頭 0
帳額 0
帴 5
帵 0
帶 7496
帶上 91
帶上來 13
帶上去 0
帶下 24
帶下來 2
帶下去 1
帶了 189
帶人 46
帶來 2812
帶來帶去 2
帶信 5
帶個 25
帶傷 6
帶入 212
帶兵 19
帶兵官 1
帶出 183
帶出來 15
帶出去 8
帶分數 8
帶到 338
帶刺 5
帶勁 11
帶動 1606
帶動風氣 3
帶原者 0
帶去 75
帶味 0
帶回 546
帶回來 34
帶回去 43
帶團 34
帶在 68
帶場 0
帶場 0
帶壞 8
帶夠 2
帶好 19
帶子 142
帶子 142
帶孝 2
帶工 5
帶帶 5
帶往 39
帶得 23
帶方 4
帶有 570
帶槍 12
帶槍上場 0
帶槍上場 0
帶槍上陣 0
帶法 8
帶狀 254
帶狀節目 7
帶班 14
帶球 21
帶球上籃 1
帶病 10
帶病上場 0
帶病上場 0
帶的 311
帶筆 2
帶給 1083
帶菌 19
帶菌者 9
帶著 2059
帶走 229
帶起 41
帶起來 3
帶路 58
帶軍 1
帶進 158
帶進來 4
帶進去 2
帶過 73
帶過來 7
帶過去 3
帶鋸 14
帶鋼 1
帶錢 10
帶錢來 1
帶錢到 0
帶錢去 0
帶開 12
帶隊 76
帶電 59
帶電粒子 10
帶電體 5
帶領 1535
帶領到 4
帶領著 43
帶頭 195
帶飯 0
帶髮修行 1
帶魚 15
帶點 117
帶齊 7
帷 194
帷堂 0
帷帳 1
帷幄 10
帷幔 0
帷幕 47
帷幕牆 126
帷握 0
常 29977
常事 18
常人 155
常以 162
常任 28
常任代表 2
常任理事國 30
常來 361
常例 7
常備 85
常備兵 27
常備軍 4
常價 5
常到 71
常務 312
常務委員 99
常務理事 359
常務監事 156
常務董事 59
常勝 4
常勝將軍 2
常勝軍 19
常去 369
常可 155
常向 34
常吸 3
常喝 17
常回 11
常回來 18
常回去 4
常因 248
常在 461
常壓塔 0
常委 340
常委會 201
常客 76
常將 85
常山 21
常川 0
常州 58
常常 3264
常平倉 0
常年 199
常式 7
常德 76
常情 16
常想 62
常態 430
常態分班 19
常態分配 15
常打 24
常指 22
常撥 6
常播 0
常播 0
常敗 1
常敗軍 1
常數 197
常春 44
常春藤 294
常是 239
常會 788
常有 702
常服 10
常期性 0
常洗 12
常溫 93
常溫下 19
常溫狀態 0
常犯 50
常理 47
常用 2243
常用參考書 0
常用字 17
常用對數 0
常用詞 3
常看 58
常看到 70
常禮 1
常禮服 0
常綠 113
常綠植物 1
常綠樹 1
常綠灌木 29
常綠闊葉 3
常綠闊葉林 3
常聽到 63
常被 539
常見 6907
常見於 65
常規 212
常規武器 1
常言 7
常言說得好 1
常言道 14
常設 96
常設性 12
常設機構 5
常談 9
常講 28
常識 2131
常識性 9
常跑 31
常跳 5
常軌 14
常量 8
常開 42
常青 27
常青樹 23
常須 47
常駐 166
常駐性 2
帹 0
帹 0
帺 0
帺 0
帽 1057
帽上 12
帽子 418
帽帶 2
帽徽 0
帽檐 1
帽沿 6
帽沿兒 0
帽沿兒 0
帽章 1
帽簷 0
帽舌 2
帽邊 3
帽頂 1
帾 4
帿 0
幀 209
幀頻 0
幁 6
幂 0
幂 0
幃 27
幄 10
幅 1430
幅員 48
幅員廣大 17
幅員遼闊 25
幅寬 18
幅射 165
幅射線 9
幅度 770
幆 0
幆 0
幇 0
幈 0
幈 0
幈 0
幊 4
幋 2
幌 37
幌動 3
幌子 12
幍 4
幍 4
幎 2
幏 1
幐 0
幑 0
幒 0
幒 0
幓 61
幔 56
幕 2256
幕僚 386
幕前 54
幕前幕後 27
幕天席地 0
幕客 2
幕帷 0
幕府 43
幕府將軍 6
幕府時代 6
幕後 563
幕後人物 0
幕後新聞 0
幕後消息 1
幕後英雄 15
幕間 6
幕間休息 0
幖 0
幗 52
幘 98
幙 128
幚 0
幛 72
幛子 0
幜 2
幝 6
幞 0
幟 79
幠 10
幡 60
幡兒 0
幡兒 0
幡幟 0
幡然 4
幡然悔悟 0
幢 240
幢幡 4
幢幢 23
幣 2441
幣值 111
幣制 29
幣帛 0
幦 738
幧 430
幨 118
幨 118
幩 38
幪 32
幫 6471
幫上 33
幫主 126
幫倒忙 4
幫傭 41
幫兇 22
幫凶 30
幫到 10
幫助 5316
幫助起來 0
幫子 7
幫工 14
幫幫 339
幫幫忙 477
幫忙 2201
幫成 3
幫手 110
幫拳 0
幫會 42
幫派 184
幫派組織 8
幫浦 383
幫腔 4
幫著 25
幫補 2
幫襯 6
幫規 4
幫辦 3
幫閑 1
幬 6
幬 6
幭 64
幮 34
幯 978
幰 8
干 898
干休 6
干係 11
干卿何事 0
干將 6
干將莫邪 0
干戈 20
干擾 1159
干擾力 1
干支 17
干政 25
干時 2
干涉 444
干涉現象 7
干犯 8
干與 3
干貝 36
干預 430
平 9186
平一 78
平亂 7
平了 24
平交道 120
平仄 31
平仄韻 0
平仰 0
平伏下來 0
平信 21
平個 3
平個 3
平假名 6
平價 580
平價供應中心 1
平價商店 3
平允 1
平光 91
平光眼鏡 1
平凡 631
平分 39
平分秋色 32
平分線 0
平劇 50
平動 2
平原 655
平原地 5
平原地帶 2
平反 62
平反 62
平台 1685
平和 505
平地 361
平地一聲雷 2
平地一聲雷 2
平地上 8
平地人 21
平地樓台 0
平地起風波 0
平地起風波 0
平地風波 0
平地風波 0
平均 6468
平均主義 1
平均值 349
平均分子 0
平均分子 0
平均分子量 5
平均利潤 1
平均利潤率 0
平均地權 133
平均工資 35
平均差 1
平均數 154
平均氣溫 57
平均速度 9
平坦 255
平坦順暢 0
平城 2
平埔族 327
平埔族 327
平壤 26
平媛鐵路 0
平安 1352
平安夜 86
平安無事 26
平安險 163
平定 28
平實 116
平射 1
平射砲 0
平局 1
平展 9
平展遼闊 0
平峒 0
平川 2
平巷 22
平常 1461
平常心 139
平平 166
平平加 20
平平安安 50
平平定定 0
平平常常 5
平平庸庸 0
平平白白 0
平平穩穩 5
平平靜靜 0
平年 3
平底 15
平底船 2
平底鍋 12
平底鞋 6
平度 6
平庸 25
平得 2
平得 188
平復 31
平復起來 0
平心 27
平心定氣 0
平心而論 29
平心靜氣 41
平快 9
平快車 5
平息 178
平成 172
平房 45
平手 51
平抑 10
平放 29
平整 80
平整土地 0
平方 269
平方公分 70
平方公尺 707
平方公里 443
平方根 46
平方米 40
平方英哩 5
平日 783
平易 25
平易近人 77
平時 1332
平月 8
平板 181
平板儀 6
平板玻璃 22
平板車 3
平林 62
平權 94
平權主義 1
平正 11
平步青雲 13
平民 181
平民化 40
平流 17
平流層 70
平涼 2
平涼縣 0
平淡 114
平淡無味 2
平淡無奇 19
平淺 0
平添 22
平添一段佳話 0
平添一段佳話 0
平準 73
平溪 335
平溪 335
平溪鄉 108
平溪鄉 108
平滑 217
平滑肌 58
平爐 1
平爐鋼 0
平版 68
平版印刷 31
平王 17
平生 64
平生大事 0
平生無大志 2
平白 54
平白無故 11
平白無辜 0
平的 188
平盤 69
平直 34
平移 61
平穩 328
平空 6
平等 1210
平等主義 4
平等互惠 35
平等待遇 8
平等性智 1
平紋 2
平素 17
平絨 0
平綏鐵路 1
平緩 78
平聲 59
平臥 3
平臥在 0
平臥著 0
平臺 76
平舖直敘 1
平行 886
平行作業 2
平行六面 1
平行六面體 2
平行力 0
平行四邊 2
平行四邊形 15
平行四邊形法則 0
平行埠 0
平行線 78
平衡 2051
平衡台 2
平衡器 1
平衡常數 7
平衡感 20
平衡木 52
平衡論 0
平衡起來 1
平裝 1823
平裝本 48
平西 4
平視 6
平角 1
平話 5
平調 17
平議 26
平谷 1
平起 6
平起平坐 26
平路 843
平身 5
平躺 48
平躺在 5
平躺著 3
平車 21
平車 21
平輩 8
平野 20
平鋪直敘 7
平陽 26
平隴望蜀 0
平靜 517
平靜下來 22
平面 1899
平面圖 760
平面幾何 7
平面角 0
平面鏡 19
平頂 77
平頂山 1
平順 140
平頭 51
平頭並進 0
平頭主義 0
平頭百姓 0
平飛 15
平飛球 4
年 112664
年下 148
年中 570
年久 10
年久失修 18
年久日衰 0
年事 13
年事已大 1
年事已高 17
年代 6275
年份 1067
年休 5
年來 2008
年俗 7
年假 44
年僅 88
年光 45
年內 1501
年分 32
年刊 195
年初 1099
年利 14
年利率 73
年前 2052
年在 465
年報 944
年增率 0
年壽 4
年夜 1
年夜飯 19
年宵 2
年富力強 2
年少 242
年少無知 6
年尾 21
年市 4
年年 382
年年有餘 21
年幼 95
年幼無知 7
年底 3383
年庚 7
年度 15853
年後 895
年復一年 18
年復一年 18
年息 86
年成 150
年所 207
年收 37
年數 122
年方 31
年時 480
年景 1
年曆 55
年會 956
年月 691
年月日 621
年有 143
年期 594
年末 57
年根 0
年次 228
年歲 76
年歲久遠 0
年歲已久 0
年歷 12
年深日久 1
年滿 495
年率 23
年產 142
年產量 96
年畫 40
年禮 256
年租 5
年節 116
年糕 59
年糕米 0
年紀 1118
年紀輕輕 39
年紙 2
年級 1249
年終 424
年終獎金 114
年終考績 22
年老 132
年老力衰 2
年老色衰 1
年老體衰 5
年菜 17
年華 520
年華易逝 0
年華虛度 0
年薪 147
年號 18
年表 320
年裡 74
年譜 35
年貨 21
年費 585
年資 1273
年輕 2683
年輕人 1018
年輕力壯 8
年輕有為 7
年輕氣盛 3
年輕貌美 29
年輕輕 0
年輪 45
年近 43
年運 25
年邁 38
年邁力衰 1
年金 834
年銷 11
年銷量 8
年鑑 810
年鑒 2
年長 168
年間 850
年關 158
年關已近 0
年限 582
年青 259
年頭 199
年頭兒 0
年頭兒 0
年飯 4
年高 98
年高德劭 2
年高德邵 1
年齒 1
年齡 3549
年齡層 484
年齡特徵 0
幵 12
并 500
并 500
并力 0
并吞 2
并行 8
幷 0
幷 0
幸 2109
幸免 3
幸免於難 1
幸喜 7
幸好 446
幸好是 8
幸好有 20
幸會 10
幸未 0
幸災樂禍 24
幸甚 7
幸生 6
幸福 2045
幸福家庭 31
幸福戲院 46
幸而 80
幸而是 0
幸能 20
幸虧 47
幸虧在 0
幸虧是 0
幸虧有 4
幸運 1112
幸運之星 3
幸運之神 17
幸運兒 34
幸運券 1
幹 7915
幹上 9
幹下去 3
幹了 40
幹事 464
幹事會 58
幹些 8
幹什麼 119
幹什麼 142
幹個 1
幹個夠 0
幹勁 27
幹勁兒 0
幹勁兒 0
幹勁十足 3
幹員 46
幹嗎 33
幹嘛 318
幹壞事 2
幹完 1
幹得 7
幹得了 0
幹性油 0
幹才 7
幹掉 106
幹校 4
幹活 12
幹活兒 1
幹活兒 1
幹流 4
幹球溫度 0
幹甚麼 8
幹甚麼 8
幹甚麼 8
幹甚麼 8
幹線 358
幹練 19
幹群關係 0
幹警 1
幹起來 0
幹路 0
幹道 82
幹部 1532
幹麼 400
幺 0
幻 1126
幻像 18
幻出 7
幻化 86
幻境 111
幻夢 46
幻影 134
幻念 2
幻想 737
幻想作 0
幻想曲 64
幻景 0
幻法 0
幻滅 73
幻燈 68
幻燈機 105
幻燈片 366
幻畫 1
幻聽 20
幻術 21
幻視 5
幻視作用 0
幻覺 92
幻覺性 0
幻象 307
幼 1920
幼主 1
幼兒 3549
幼兒園 53
幼兒教育 1
幼兔 1
幼功 1
幼君 0
幼吾幼以及人之幼 1
幼女 21
幼子 28
幼孤 1
幼學 1
幼學壯行 1
幼學瓊林 1
幼小 66
幼小無知 0
幼年 134
幼年人 0
幼年時代 1
幼年時期 11
幼年期 14
幼弱 1
幼托 1
幼教 1395
幼教組 2
幼時 41
幼木 0
幼林 4
幼株 1
幼根 0
幼犬 1
幼獅 1672
幼獅通訊 0
幼發拉底 0
幼發拉底河 10
幼禽 0
幼稚 280
幼稚園 3034
幼稚園病 1
幼稚師範 2
幼稚教育 155
幼稚病 1
幼稺 1
幼童 206
幼童軍 73
幼艾 1
幼芽 6
幼苗 123
幼蟲 225
幼風 1
幼體 15
幼魚 50
幼鳥 24
幼齒 0
幼齡林 0
幽 715
幽人 3
幽光 2
幽冥 52
幽叢 0
幽咽 2
幽夢 34
幽室 2
幽寂 2
幽居 8
幽州 15
幽幽 64
幽幽地 16
幽幽地 16
幽幽暗暗 1
幽幽的 16
幽思 4
幽怨 12
幽恨 0
幽情 79
幽憤 0
幽明 0
幽暗 55
幽會 14
幽浮 293
幽深 22
幽渺 6
幽獨 2
幽禁 9
幽美 48
幽蘭 14
幽谷 68
幽遠 10
幽邃 3
幽門 113
幽閒 0
幽雅 106
幽靈 499
幽靈式 6
幽靈機 0
幽靈船 5
幽靜 66
幽香 16
幽魂 12
幽魂不散 0
幽默 630
幽默感 68
幾 2632
幾 2632
幾下 89
幾世 6
幾世紀 16
幾乎 4051
幾乎在 41
幾乎是 381
幾人 90
幾人份 0
幾付 0
幾代 27
幾令 2
幾件 79
幾件事 24
幾任 9
幾份 33
幾伍 1
幾位 676
幾位數 2
幾何 477
幾何圖形 68
幾何學 24
幾何體 4
幾個 1
幾個 3004
幾個人 1
幾個人 161
幾個月 1
幾個月 382
幾倍 64
幾億 23
幾元 5
幾克 2
幾內亞 54
幾內亞比紹 1
幾內亞灣 0
幾兩 12
幾具 2
幾冊 4
幾刀 5
幾分 254
幾分鐘 417
幾列 3
幾副 1
幾劃 2
幾動 0
幾匹 1
幾匹馬 0
幾區 5
幾十 317
幾十人 0
幾十個 24
幾千 143
幾千人 8
幾千個 11
幾千元 14
幾千塊 9
幾句 153
幾句話 121
幾台 37
幾吋 3
幾名 80
幾員 1
幾回 57
幾因 0
幾圈 22
幾國 3
幾圓 0
幾團 2
幾場 77
幾場 77
幾塊 31
幾塊錢 6
幾壘 0
幾壺 0
幾多 29
幾大 19
幾大塊 0
幾大片 0
幾大類 9
幾天 1155
幾孔 0
幾字 4
幾季 8
幾客 0
幾家 329
幾家歡樂 1
幾家歡樂幾家愁 6
幾寸 0
幾封 45
幾封信 8
幾小時 47
幾尾 4
幾局 5
幾屆 14
幾層 22
幾層樓 3
幾巷 0
幾希 4
幾師 0
幾幅 15
幾年 1716
幾年生 0
幾年級 13
幾幾乎乎 0
幾床 0
幾度 71
幾座 29
幾弄 0
幾張 206
幾彈 0
幾彈 0
幾微 0
幾成 9
幾成新 0
幾截 0
幾戶 11
幾所 56
幾手 10
幾打 2
幾批 4
幾拳 1
幾支 50
幾文錢 1
幾斤 10
幾於 2
幾旅 0
幾日 101
幾日份 0
幾星 0
幾星期 15
幾時 120
幾晚 5
幾月 26
幾月份 1
幾期 54
幾本 69
幾本書 27
幾板 0
幾枚 10
幾枝 8
幾架 10
幾株 17
幾格 4
幾桶 1
幾桿 1
幾梯次 0
幾棟 16
幾棵 13
幾槍 2
幾樣 32
幾檔 9
幾次 843
幾步 64
幾歲 175
幾段 37
幾比幾 0
幾毛 0
幾毛錢 1
幾洞 2
幾滴 52
幾滴水 1
幾炷香 0
幾版 11
幾率 0
幾班 9
幾環 0
幾番 56
幾發 4
幾百 162
幾百人 6
幾百個 17
幾百元 10
幾百塊 10
幾盒 3
幾盒裝 0
幾盞 3
幾眼 16
幾碗 5
幾碗飯 0
幾票 6
幾科 9
幾秒 73
幾秒鐘 50
幾稀 1
幾種 663
幾窩 1
幾站 4
幾章 26
幾竿 1
幾筆 22
幾筒 0
幾箱 3
幾節 13
幾節課 1
幾籠 0
幾米 56
幾粒 5
幾級 14
幾組 27
幾絲 8
幾經 116
幾線 6
幾縷 2
幾罐 2
幾聲 69
幾股 2
幾胎 2
幾能 1
幾腳 2
幾萬 89
幾萬人 2
幾萬個 7
幾萬元 11
幾萬塊 3
幾處 44
幾號 64
幾街 0
幾袋 1
幾角 0
幾角形 0
幾許 52
幾課 2
幾起 10
幾趟 12
幾趟路 0
幾路 2
幾車 1
幾輛 8
幾輛車 0
幾轉 2
幾近 135
幾近於 11
幾通 13
幾道 31
幾道菜 4
幾部 122
幾針 1
幾門 4
幾開 0
幾間 25
幾關 11
幾隻 116
幾集 39
幾頁 42
幾頂 2
幾項 372
幾頭 2
幾顆 58
幾題 13
幾類 40
幾首 52
幾首歌 18
幾點 500
幾點鐘 5
幾黨 0
幾齣 11
广 0
庀 2
庁 0
庂 7
広 0
庄 827
庇 215
庇佑 37
庇蔭 25
庇護 163
庇護所 18
庈 398
庉 44
床 2488
床上 478
床上運動 1
床下 25
床位 128
床單 89
床墊 97
床尾 9
床席 1
床帳 1
床帷 0
床戲 0
床板 12
床架 3
床柱 1
床榻 4
床毯 0
床沿 16
床第 14
床第之間 3
床罩 53
床腳 5
床舖 57
床蓆 1
床蓋 0
床蓐 0
床褥 4
床鋪 20
床頭 101
床頭人 0
床頭櫃 6
庋 126
庋藏 14
庌 54
庍 81
庎 0
序 3156
序列 398
序列埠 0
序幕 151
序數 9
序文 79
序時帳 9
序曲 112
序破急 398
序號 2977
序言 735
序論 20
序跋 12
庐 0
庒 0
底 6630
底下 1180
底下人 1
底價 225
底冊 5
底土 1
底墊 2
底子 44
底定 90
底層 202
底座 189
底數 7
底本 16
底板 75
底架 5
底柱 0
底止 328
底漆 33
底片 459
底版 7
底牌 10
底特律 67
底盤 202
底稿 68
底站 1
底端 25
底細 37
底線 1590
底縫 0
底肥 0
底色 101
底薪 40
底蘊 7
底裡 1
底褲 2
底襯 0
底角 10
底貨 0
底質 49
底邊 9
底部 761
底里 5
底限 13
底面 23
底點 3
庖 203
庖丁 7
庖廚 2
店 41203
店裡 100
店主 79
店名 192
店員 2118
店家 402
店小二 45
店底 0
店東 28
店租 17
店舖 151
店號 3
店鋪 64
店長 183
店面 618
店頭 0
店頭市場 0
店頭市場 0
庘 0
庙 0
庚 1662
庚子 17
庚子賠款 4
庚帖 0
庛 78
府 5589
府上 89
府中 44
府兵 9
府兵制 3
府台 2
府君 7
府城 528
府外 22
府學 9
府尊 0
府庫 6
府後 13
府治 7
府第 22
府綢 3
庝 0
庠 121
庠序 0
庡 0
庢 87
庣 105
庤 146
庥 253
度 213
度 21361
度人 26
度假 378
度假勝地 5
度外 36
度支使司 0
度數 108
度日 56
度日如年 5
度母 1
度牒 3
度脫 26
度著 4
度角 44
度過 431
度過難關 17
度量 123
度量 123
度量衡 359
度量衡 359
座 7447
座上客 8
座位 533
座像 4
座右銘 106
座墊 81
座席 11
座椅 337
座標 480
座標軸 18
座機 16
座次 15
座無虛席 13
座落 337
座落在 70
座談 993
座談會 1538
座車 65
座鐘 11
庨 8
庨 8
庩 0
庪 291
庫 2232
庫侖 3
庫侖定律 2
庫倫 22
庫存 2483
庫存現金 7
庫存股票 1
庫存量 305
庫容 5
庫房 65
庫房重地 0
庫爾斯克 1
庫爾曼 0
庫藏 158
庫藏股 0
庫藏量 0
庫銀 4
庫頁島 3
庬 6
庬 6
庭 4000
庭上 30
庭內 36
庭園 629
庭外 15
庭外和解 8
庭子 2
庭決 1
庭訊 8
庭訓 8
庭議 0
庭長 50
庭院 197
庭除 1
庮 20
庯 0
庰 137
庱 1
庲 5
庳 11
庴 26
庵 338
庵主 4
庵兒 0
庵兒 0
庵堂 1
庶 140
庶事 0
庶人 7
庶務 110
庶務員 0
庶務組 21
庶士 0
庶子 2
庶幾 14
庶幾無愧 1
庶母 0
庶民 38
庶物崇拜 1
庶生 0
庶神 0
康 10864
康乃爾大學 179
康乃狄克 12
康乃狄克州 16
康乃狄格 0
康乃馨 211
康健 198
康定 33
康定路 73
康寧 393
康寧護校 18
康康 0
康康舞 0
康強 24
康復 213
康德 126
康德黎 0
康有為 19
康柏 0
康樂 419
康樂活動 42
康熙 206
康熙字典 5
康爵 0
康百克 12
康莊 71
康莊大道 38
康采恩 0
庸 45
庸 455
庸人 4
庸人自擾 5
庸人自擾之 7
庸俗 42
庸俗化 7
庸俗進化 0
庸俗進化論 0
庸夫俗子 0
庸庸碌碌 9
庸才 12
庸材 1
庸民 0
庸碌 6
庸醫 4
庹 18
庺 0
庻 0
庽 0
庾 104
庿 0
廀 0
廀 0
廀 0
廁 153
廁 153
廁坑 4
廁所 744
廁牏 1
廁神 1
廁紙 0
廁足 1
廁身 1
廁身其間 0
廂 315
廂兵 1
廂官 1
廂房 63
廂鋪 1
廃 0
廄 20
廄肥 2
廅 0
廆 0
廇 0
廈 268
廈 268
廈房 1
廈門 378
廈門人 0
廈門市 54
廉 743
廉價 259
廉價品 2
廉售 138
廉售價 0
廉售品 0
廉宜 259
廉恥 8
廉恥心 0
廉明 9
廉潔 1082
廉潔可風 0
廉潔奉公 0
廉直 0
廉節 1
廉能 79
廉讓 119
廉隅 0
廊 1814
廊下 8
廊坊 1
廊房 1
廊簷 1
廊道 31
廋 23
廌 40
廌 40
廍 1
廎 98
廏 0
廏 0
廐 0
廐 0
廑 97
廒 97
廒間 0
廓 191
廓大鏡 0
廓清 36
廓落 0
廔 235
廕 83
廖 4949
廖述良 21
廗 69
廘 206
廙 73
廚 649
廚具 385
廚刀 1
廚娘 3
廚子 23
廚師 434
廚房 1236
廚手 0
廚櫃 27
廚灶 2
廚藝 36
廚餘 1
廛 26
廜 126
廝 193
廝守 31
廝打 0
廝殺 101
廝混 7
廞 12
廟 2076
廟主 5
廟享 1
廟令 1
廟口 133
廟台 2
廟堂 30
廟堂之器 1
廟堂文學 1
廟宇 485
廟廊之彥 1
廟方 0
廟會 106
廟朝 1
廟略 1
廟祝 12
廟神 5
廟策 1
廟算 1
廟號 1
廟裡 49
廟裡的豬頭 1
廟見 1
廟諱 1
廟謀 1
廟謨 1
廟議 1
廟食 1
廠 14430
廠主 20
廠名 168
廠商 65904
廠地 41
廠址 110
廠外 82
廠子 1
廠家 270
廠房 793
廠方 54
廠棚 0
廠牌 1212
廠用電 4
廠礦 8
廠稅 0
廠規 10
廠長 136
廠長負責制 0
廡 37
廢 3830
廢言 248
廢了 19
廢人 13
廢件 3
廢品 12
廢品率 0
廢土 262
廢地 2
廢城 0
廢墟 266
廢學 2
廢寢忘食 21
廢寢忘餐 1
廢帝 4
廢弛 12
廢弛已久 0
廢掉 38
廢料 1322
廢時 17
廢柴 1
廢棄 728
廢棄物 3091
廢止 435
廢氣 728
廢水 1243
廢水槽 4
廢水池 7
廢水管 2
廢油 27
廢油槽 0
廢液 154
廢液 154
廢渣 3
廢物 206
廢物利用 35
廢物箱 0
廢物袋 0
廢省 0
廢票 26
廢票箱 0
廢紙 389
廢置 16
廢置不用 1
廢置不用 1
廢置不管 0
廢船 6
廢話 197
廢話少說 20
廢話連篇 5
廢醣蜜 0
廢銅爛鐵 0
廢鐵 241
廢除 389
廢黜 1
廣 11223
廣交 16
廣元 23
廣加 5
廣南 0
廣博 79
廣博性 0
廣告 22455
廣告公司 210
廣告單 13
廣告學 56
廣告板 15
廣告業 56
廣告欄 5
廣告牌 4
廣告社 240
廣告費 59
廣場 517
廣場 5176
廣大 1062
廣大無邊 8
廣大群眾 17
廣安 61
廣宗 7
廣島 181
廣川 15
廣州 520
廣州市 67
廣州灣 2
廣州話 2
廣度 132
廣廈 4
廣廈 4
廣廈萬間 1
廣得 9
廣播 2153
廣播 2153
廣播公司 293
廣播公司 293
廣播劇 41
廣播劇 41
廣播台 18
廣播台 18
廣播員 2
廣播員 2
廣播室 0
廣播室 0
廣播界 20
廣播界 20
廣播站 56
廣播站 56
廣播節目 215
廣播節目 215
廣播網 140
廣播網 140
廣播者 3
廣播者 3
廣播臺 18
廣播臺 18
廣播電台 825
廣播電台 825
廣播電臺 63
廣播電臺 63
廣播電視 449
廣播電視 449
廣播體操 0
廣播體操 0
廣敞 0
廣東 846
廣東人 14
廣東省 310
廣東籍 3
廣東茶 0
廣東菜 4
廣東話 63
廣汎 10
廣泛 1598
廣泛性 29
廣漠 5
廣漠無邊 0
廣漢 18
廣漢市 18
廣瀚 1
廣瀚無邊 0
廣眾 2
廣積陰德 0
廣結 17
廣結善緣 45
廣結良緣 0
廣義 230
廣義性 2
廣衍 0
廣西 154
廣西人 0
廣西壯族自治區 1
廣西省 15
廣西自治區 2
廣角鏡 1
廣通 0
廣達 0
廣闊 261
廣闊天地 1
廣陵 23
廣電法 0
廥 1036
廦 131
廧 1653
廨 26
廩 170
廩倉 0
廩膳 0
廪 0
廫 0
廬 300
廬山 197
廬山真面目 45
廬州 1
廬江 2
廬舍 3
廮 1
廯 4
廯 4
廰 0
廱 27
廲 40
廳 4000
廳事 0
廳堂 35
廳房 7
廳舍 59
廳長 346
廴 0
廵 0
延 11148
延企 4
延休 0
延伸 1100
延伸到 150
延佇 0
延修 36
延到 37
延吉 53
延吉街 76
延壽 87
延安 29
延宕 143
延宕已久 0
延展 128
延展性 28
延平 983
延平中學 1
延平北路 215
延平區 11
延平南路 93
延平郡王 10
延平郡王祠 30
延平高級中學 1
延年 34
延年益壽 30
延後 465
延性 43
延接 0
延擱 4
延攬 336
延會 0
延期 374
延河 0
延滯 59
延燒 114
延燒到 14
延畢 1
延緩 162
延緩執行 0
延續 809
延續下去 24
延續到 43
延續性 28
延聘 235
延腦 5
延至 190
延誤 422
延請 41
延賽 3
延遲 494
延遲線 1
延長 1622
延長線 51
延頸企踵 0
延髓 3
廷 3999
廷尉 2
廷布 0
廷杖 1
廷長 5
廸 0
廹 0
建 15457
建中 1842
建交 164
建元 137
建出 6
建制 105
建功 131
建台 126
建和 118
建商 0
建國 3942
建國中學 220
建國北路 341
建國商專 1
建國大綱 9
建國工商專校 1
建國工專 12
建國技術學院 1
建國方略 8
建國科技大學 1
建國高級中學 1
建在 68
建地 142
建坪 145
建好 46
建安 157
建安國小 13
建安國民小學 1
建寧 17
建州 37
建康 71
建廠 290
建成 537
建成區 6
建教合作 565
建材 1992
建材行 271
建業 89
建構 2998
建樹 46
建檔 0
建武 19
建物 1483
建甌 1
建立 25410
建立成 20
建立起 283
建立起來 28
建築 13712
建築商 19
建築工地 20
建築工業 6
建築工程 608
建築師 2669
建築材料 413
建築材料工業 15
建築業 462
建築機械 17
建築物 2678
建築結構 84
建築藝術 64
建築設計 899
建築起來 1
建築面積 74
建置 0
建臺 12
建蔽率 119
建蘭 0
建言 614
建設 9553
建設公司 496
建設單位 19
建設局 323
建設廳 521
建設廳長 7
建設性 218
建設成 22
建設所 23
建議 21059
建議性 7
建議案 58
建起 20
建起來 3
建軍 74
建造 1420
建造出 10
建造成 8
建都 43
建都於 4
建醮 0
建黨 42
廻 0
廼 0
廽 0
廾 40
廿 3000
廿一 8
廿七 8
廿三 8
廿九 8
廿二 8
廿五 8
廿八 8
廿六 8
廿四 8
开 0
开 0
开 0
弁 383
弁言 3
异 177
弃 0
弄 1
弄 35
弄 3551
弄乾 1
弄亂 17
弄低 1
弄來 10
弄來弄去 1
弄倒 3
弄假成真 2
弄僵 3
弄兵 0
弄出 40
弄出來 7
弄出去 0
弄到 84
弄反 0
弄口 0
弄堂 3
弄壞 21
弄壞了 10
弄好 74
弄姿 4
弄巧成拙 20
弄平 3
弄弄 5
弄彎 2
弄得 220
弄懂 27
弄成 28
弄散 0
弄斜 0
弄斷 10
弄權 8
弄死 10
弄法 2
弄混 10
弄清 108
弄清事實 0
弄湯 1
弄溼 5
弄潮 15
弄潮兒 0
弄潮兒 0
弄濕 16
弄熄 2
弄獅 3
弄璋 8
弄瓦 14
弄直 1
弄神弄鬼 2
弄筆 1
弄粗 0
弄糟 1
弄翻 4
弄臣 50
弄虛作假 3
弄走 0
弄起 3
弄起來 2
弄通 2
弄酒 0
弄錯 64
弄門 0
弄飯 1
弄髒 30
弅 156
弆 0
弇 441
弈 33
弈具 0
弈棋 4
弊 2153
弊多利少 0
弊害 11
弊少利多 0
弊惡 0
弊政 0
弊案 0
弊源 0
弊病 77
弊竇 0
弊端 203
弊絕風清 1
弋 91
弌 0
弌 0
弍 0
弎 0
弎 0
弎 0
式 52404
式名 1
式子 44
式微 2141
式樣 319
弑 0
弒 42
弒君 6
弒君大罪 0
弒母 0
弒父 9
弓 947
弓子 2
弓弦 19
弓弩手 4
弓形 8
弓手 5
弓折箭盡 0
弓漿蟲 1
弓矢 2
弓箭 98
弓箭手 0
弓箭步 5
弓背 4
弓腰 2
弓著 7
弓著腰 3
弓鋸 3
弔 152
弔卷 0
弔古 0
弔唁 9
弔喪 3
弔嗓子 0
弔孝 0
弔客 1
弔慰 2
弔掛 2
弔文 2
弔架 1
弔桶 0
弔死 0
弔死鬼 0
弔民伐罪 1
弔民伐罪 1
弔祭 26
弔籃 0
弔臂 0
引 39
引 3914
引上 3
引上來 0
引上去 0
引下 4
引下來 0
引下去 0
引人 253
引人入勝 83
引人注目 142
引人矚目 10
引人遐思 5
引介 0
引以為傲 75
引以為恥 0
引以為憾 0
引以為戒 4
引以為榮 22
引以為鑒 0
引伸 20
引伸出 4
引伸到 0
引作 4
引來 177
引來引去 0
引信 8
引兒 0
引兒 0
引入 368
引入歧途 0
引出 98
引出來 4
引出去 0
引到 24
引力 82
引力場 1
引力場 1
引力定律 0
引去 2
引取 16
引吭高歌 9
引咎 4
引咎自責 0
引咎辭職 8
引喻失義 0
引子 36
引導 1391
引導出 12
引導者 7
引座員 0
引得 36
引成 0
引拉 2
引拉出 0
引接 1
引擎 9123
引擎蓋 45
引據 29
引數 0
引文 33
引橋 1
引水 123
引水人 64
引水道 2
引流 104
引渡 77
引渡回國 2
引港費 0
引潮力 1
引火 44
引火焚身 0
引火線 2
引火自焚 3
引火點 0
引為 20
引燃 45
引爆 522
引牽 0
引牽出 0
引狼 0
引狼入室 4
引理 2
引用 1901
引申 82
引申作 0
引發 2035
引發出 20
引種 16
引經據典 8
引線 49
引而不發 0
引致 94
引航 7
引航員 0
引荐 6
引號 90
引蛇出洞 10
引見 18
引言 366
引証 1
引詩為証 0
引誘 162
引證 29
引起 5165
引起來 0
引足救經 0
引路 18
引車買漿 0
引述 504
引退 24
引逗 1
引進 2423
引進來 6
引進去 0
引過 1
引過來 1
引過去 0
引道 10
引錯 0
引開 12
引領 376
引領到 1
引頸 0
引頸就戮 0
引風吹火 0
引餌 0
弗 1360
弗洛伊德 63
弗蘭格爾 0
弘 3360
弘一大師 1
弘仁 19
弘光 193
弘光科技大學 1
弘光護專 18
弘光護理專科學校 1
弘光醫事護理專科學校 1
弘大 28
弘揚 136
弘曆 0
弘毅 74
弘法 140
弘農 7
弘道 104
弘遠 3
弘量 1
弘願 9
弚 6
弛 94
弛 94
弛廢 0
弛廢 0
弛緩 10
弛緩 10
弜 0
弜 0
弝 168
弞 0
弞 0
弟 15
弟 1
弟 59238
弟兄 1737
弟兄們 142
弟妹 219
弟妹們 76
弟婦 4
弟媳 6
弟子 651
弟弟 682
弟弟 1
弡 0
弢 8
弣 9
弤 14
弥 0
弥 0
弦 719
弦切角 2
弦外之音 12
弦月 15
弦柱 0
弦樂 264
弦樂器 40
弦樂隊 1
弦歌 20
弦重 0
弦音 10
弧 352
弧光 2
弧光燈 3
弧度 41
弧形 66
弧線 40
弧菌 93
弨 14
弩 144
弩張劍拔 0
弩手 2
弩箭 21
弫 0
弬 0
弭 30
弭亂 0
弭兵 1
弭兵之會 0
弭平 9
弭忘 0
弭患 0
弭謗 0
弮 91
弯 0
弰 5
弱 25541
弱不勝衣 0
弱不禁風 10
弱光 11
弱光性 0
弱冠 12
弱勢 737
弱勢團體 150
弱卒 0
弱國 5
弱國無外交 1
弱女子 14
弱小 53
弱小民族 8
弱水三千 4
弱水三千只取一瓢 0
弱者 72
弱肉強食 26
弱視 270
弱酸 28
弱電 19
弱音 6
弱音器 2
弱鹼 16
弱點 187
弲 0
弳 23
弴 0
弴 0
張 34
張 34066
張簡 100
張三 132
張三李四 3
張之洞 15
張了 30
張作霖 3
張先 190
張冠李戴 3
張出 2
張到 11
張力 262
張力計 1
張勳 9
張口 52
張口結舌 6
張嘴 26
張嘴說話 0
張在 21
張大 148
張天師 10
張家 323
張家口 4
張寶 424
張居正 15
張張 112
張得 17
張惶 0
張惶失措 3
張愛玲 1
張掖 1
張掛 6
張掛在 0
張掛著 1
張掛起 0
張掛起來 0
張揚 17
張揚出來 1
張揚出去 3
張望 44
張本 80
張滿 11
張照 36
張燈結彩 1
張燈結綵 11
張牙 5
張牙舞爪 17
張牙舞爪 17
張狂 12
張獻忠 0
張生 25
張皇 4
張皇失措 4
張目 15
張目一看 0
張目一看 0
張眼 5
張眼一看 0
張眼一看 0
張網 58
張繼 68
張羅 29
張翼德 6
張老師 476
張良 125
張菲 1
張著 32
張衡 8
張角 35
張說 7
張貼 18245
張貼在 64
張過 0
張開 272
張開著 2
張順 54
張飛 165
張騫 45
弶 2
強 1273
強 1273
強 12736
強中自有強中手 6
強了 42
強人 75
強人所難 15
強人所難 15
強人政權 0
強似 9
強佔 19
強佔性 0
強作 25
強使 3
強借 0
強健 133
強光 152
強到 23
強制 1318
強制 1318
強制執行 211
強制執行 211
強制性 97
強制性 97
強制手段 5
強制手段 5
強制處分 7
強制處分 7
強力 1439
強力性 0
強力攻佔 0
強力膠 16
強加 15
強加在 6
強加於 8
強加於人 2
強勁 388
強勁有力 3
強勢 731
強化 2532
強化措施 0
強占 6
強取 9
強嘴 0
強嘴 0
強固 39
強國 91
強壯 156
強壯劑 4
強壯有力 2
強壯起來 0
強大 1312
強奪 14
強奸 7
強姦 175
強姦犯 14
強姦罪 31
強娶 1
強將 21
強將手下無弱兵 1
強度 1044
強弩之末 11
強弱 926
強心 16
強心劑 21
強心針 12
強悍 135
強打 51
強打者 10
強拍 4
強拳不打笑臉 1
強擎之末 0
強敵 49
強敵環伺 10
強暴 428
強暴犯 28
強有力 65
強梁 4
強棒 68
強橫 5
強權 185
強權國家 3
強權政治 18
強毅 1
強求 113
強求 113
強派 0
強渡 18
強烈 2229
強烈度 1
強烈颱風 21
強盛 83
強盛時期 1
強盛起來 0
強盜 291
強硬 140
強硬態度 18
強硬手段 2
強硬措施 3
強租 1
強者 105
強而有力 126
強而有勁 0
強行 169
強行軍 0
強行通過 3
強行進入 4
強角 0
強記 15
強詞奪理 22
強詞奪理 22
強調 6269
強身 61
強辯 6
強迫 586
強迫性 26
強迫症 1
強逼 5
強還原劑 0
強鄰 3
強酸 58
強電 60
強韌 88
強音 13
強項 5
強顏 2
強顏 2
強顏 2
強顏歡笑 17
強顏歡笑 17
強風 101
強鹼 38
強龍 7
強龍不壓地頭蛇 2
弸 2
弹 0
弹 0
强 0
强 0
强 0
弻 0
弼 81
弽 0
弽 0
弾 0
弾 0
弿 0
彀 12
彂 0
彃 59
彄 62
彆 137
彆扭 13
彆拗 1
彆氣 0
彆腳 1
彆著 0
彇 0
彈 1618
彈 1618
彈上 6
彈上來 0
彈上去 0
彈下 3
彈下來 0
彈下去 1
彈不 5
彈不來 0
彈不出 0
彈丸 7
彈丸之地 5
彈了 34
彈來彈去 3
彈入 2
彈出 52
彈出來 5
彈出去 1
彈到 14
彈力 85
彈力絲 0
彈劾 899
彈劾權 18
彈劾法 0
彈動 2
彈匣 32
彈去 0
彈回 9
彈回來 2
彈回去 1
彈坑 3
彈壓 4
彈夾 10
彈奏 132
彈奏出 7
彈奏著 4
彈好 1
彈妥 0
彈子 23
彈子房 2
彈子抬 0
彈子檯 1
彈孔 5
彈完 3
彈射 14
彈射器 2
彈帶 1
彈幕 0
彈弓 13
彈引 1
彈彈 8
彈得 17
彈性 2317
彈性控制 2
彈性疲乏 6
彈性硫 1
彈性體 36
彈指 41
彈指之間 24
彈指神功 2
彈掉 3
彈殼 27
彈法 6
彈炮 0
彈無虛發 12
彈片 18
彈珠 125
彈琴 65
彈痕 6
彈痕累累 1
彈盡援絕 1
彈盡糧絕 0
彈砲 1
彈簧 540
彈簧床 39
彈簧椅 3
彈簧秤 1
彈簧鋼 5
彈簧鎖 0
彈腿 7
彈膛 0
彈著 20
彈著點 0
彈葯 0
彈藥 246
彈藥兵 1
彈藥庫 13
彈藥箱 2
彈走 3
彈起 20
彈起來 11
彈跳 125
彈跳床 0
彈過 4
彈過來 0
彈過去 0
彈道 22
彈道學 1
彈道飛彈 30
彈開 15
彈頭 38
彉 6
彊 141
彊 141
彊 141
彋 2953
彌 844
彌久 0
彌勒 104
彌勒佛 51
彌天大罪 0
彌封 15
彌年 0
彌撒 82
彌月 105
彌月之喜 1
彌漫 41
彌留 5
彌留之際 1
彌縫 2
彌補 601
彌賽亞 79
彌陀 379
彍 0
彍 0
彍 0
彎 784
彎上來 0
彎上去 0
彎下 40
彎下來 1
彎下去 1
彎來彎去 2
彎兒 2
彎兒 2
彎出來 1
彎出去 0
彎刀 32
彎到 4
彎去 0
彎回 0
彎回來 0
彎回去 0
彎度 7
彎弓 17
彎弓子 0
彎彎 41
彎彎曲曲 8
彎形 2
彎成 11
彎曲 540
彎曲狀 2
彎矩 21
彎腰 44
彎腰哈背 0
彎腰駝背 5
彎腳 31
彎膝禮 0
彎著 5
彎著腰 4
彎角 13
彎起 6
彎起來 1
彎路 12
彎身 5
彎道 49
彎部 0
彎頭 16
彏 6
彐 0
彑 0
归 0
归 0
当 0
当 0
当 0
彔 5
录 0
彖 146
彖辭 1
彗 247
彗星 711
彘 10
彙 2222
彙報 208
彙整 0
彙編 1131
彙總 0
彙纂 0
彙輯 7
彙集 146
彚 0
彛 0
彝 51
彝族 25
彞 0
彠 0
彡 0
形 4825
形似 45
形像 133
形制 26
形勝 2
形勢 363
形同 120
形同虛設 10
形同陌路 9
形單影隻 8
形如 36
形如槁木 0
形學 9
形容 797
形容憔悴 1
形容消瘦 0
形容盡致 0
形容詞 105
形式 3103
形式上 119
形式主義 17
形式化 28
形式美 5
形式邏輯 5
形形色色 133
形影 24
形影不離 16
形影相弔 1
形意 13
形意拳 31
形態 1135
形態學 50
形態意識 0
形成 5471
形成層 7
形數 3
形樣 2
形狀 992
形相 26
形義 0
形而上 52
形而上學 12
形而下 7
形聲 2
形聲字 6
形色 19
形色倉皇 0
形諸筆墨 0
形變 17
形象 2006
形象化 5
形象思維 1
形象藝術 0
形貌 66
形質 13
形跡 10
形跡可疑 6
形蹤 0
形銷骨立 0
形音 12
形音輸入法 1
形骸 2
形體 121
彣 0
彣 0
彤 558
彤雲 13
彥 2967
彥士 1
彦 0
彧 98
彩 7354
彩光 7
彩光板 0
彩券 53
彩妝 0
彩市 0
彩帶 55
彩度 32
彩排 46
彩旗 8
彩旦 1
彩條 2
彩棚 2
彩燈 7
彩球 39
彩瓷 8
彩畫 48
彩票 3
彩禮 3
彩筆 84
彩紙 11
彩線 4
彩繪 543
彩色 3950
彩色照片 39
彩色片 4
彩色版 27
彩色筆 82
彩色繽紛 7
彩色電影 0
彩色電視 35
彩色電視機 76
彩號 2
彩虹 583
彩虹仙子 1
彩衣 31
彩調 2
彩轎 2
彩釉 11
彩金 1
彩錢 0
彩陶 11
彩陶文化 2
彩雲 52
彩霞 40
彩頭 19
彩飾 5
彩飾版 0
彪 141
彪形 1
彪形大漢 6
彪炳 10
彫 82
彫像 0
彫刻 111
彫刻刀 2
彫刻品 1
彫刻家 0
彫刻術 0
彫塑 5
彫塑品 0
彫塑家 1
彫弓 0
彫殘 0
彫漆 1
彫版 0
彫琢 3
彫花 0
彫飾 1
彫飾品 0
彬 1228
彬彬 30
彬彬有禮 22
彭 4225
彭佳嶼 0
彭城 19
彭德懷 28
彭祖 7
彮 0
彯 44
彰 3366
彰化 4130
彰化人 17
彰化女中 1
彰化女子高級中學 1
彰化市 2543
彰化師範大學 1
彰化教育學院 8
彰化縣 10856
彰化銀行 1
彰化高中 1
彰化高級中學 1
彰德 14
彰明 8
彰明較著 0
彰銀 395
彰顯 347
影 9841
影像 11550
影像轉換 26
影像轉換器 6
影劇 278
影劇圈 8
影劇界 2
影劇院 5
影印 907
影印本 284
影印機 311
影后 71
影圈 1
影城 699
影壇 38
影壇紅星 0
影子 527
影射 67
影展 915
影帝 80
影帶 388
影影綽綽 2
影戲 5
影星 192
影本 1683
影業 125
影片 2533
影碟機 31
影視 11784
影視圈 1
影評 108
影評人 59
影評家 7
影調 0
影象 31
影質 0
影迷 201
影迷大會 1
影迷會 2
影集 242
影音 0
影響 27628
影響到 678
影響力 645
彲 0
彳 10
彴 20
彵 0
彶 173
彷 245
彷 245
彷彿 1198
彷徉 0
彷徨 10
彷徨 10
彸 148
役 1292
役使 7
役卒 0
役於外物 0
役期 1
役男 417
役畜 0
役畜 0
役畢 124
役種 12
役齡 30
彺 0
彼 1044
彼岸 88
彼得 503
彼得一世 0
彼得一世 0
彼得大帝 3
彼得格勒 0
彼時 42
彼此 2737
彼此之間 176
彼此彼此 5
彼此間 257
彼等 42
彽 9
彾 13
彿 171
往 6225
往上 487
往上走 22
往上面 1
往下 568
往下走 15
往下面 3
往世 26
往事 391
往事如夢 0
往事如煙 6
往來 1020
往來於 21
往例 34
往前 1479
往前方 7
往前走 48
往前面 5
往北 95
往北方 4
往北部 1
往南 121
往南方 4
往南部 2
往古來今 1
往右 62
往右邊 13
往哲 2
往外 110
往外面 2
往左 81
往左邊 10
往常 153
往年 179
往往 2500
往往是 298
往往會 215
往後 854
往後方 1
往後面 3
往復 90
往復泵 1
往日 172
往日情懷 19
往昔 164
往時 22
往東 98
往東方 6
往東部 0
往生 0
往聖先賢 0
往臉上貼金 1
往裡 16
往裡面 9
往複 1
往複泵 0
往西 68
往西方 2
往西部 0
往返 335
往返程 0
往還 9
往還於 1
征 362
征伐 14
征伐 14
征兵 4
征塵 0
征夫 4
征戰 66
征收 48
征服 392
征服者 24
征求 3
征發 0
征衣 0
征討 16
征途 12
徂 11
徂暑 0
徂謝 0
徂逝 0
徃 0
径 0
待 4270
待 7495
待業 100
待不住 3
待了 65
待人 198
待人之道 2
待人以誠 4
待人接物 30
待價而沽 22
待優 47
待到 21
待制 2
待命 162
待命班 3
待在 266
待在家裡 34
待如己出 0
待嫁 6
待嫁女兒心 1
待字閨中 7
待客 34
待客之道 9
待工 3
待攤費用 0
待斃 1
待會 91
待會兒 44
待會兒 44
待查 931
待機 186
待機而動 2
待產 39
待產室 7
待產期 0
待發 49
待續 301
待聘 57
待著 31
待著 31
待遇 1430
待過 24
徆 68
徇 35
徇 35
徇情 1
徇情枉法 0
徇私 7
徇私情 0
徇私枉法 2
徇私舞弊 9
徇義 0
很 27890
很不 843
很不 843
很久 1059
很乾 13
很亮 20
很低 129
很使 0
很來 6
很像 398
很僵 3
很受 117
很可惜 116
很可能 683
很喜歡 885
很圓 7
很壞 33
很多 9119
很夠 28
很大 2226
很大程度上 8
很好 2511
很對 47
很小 306
很少 1380
很尖 0
很得 8
很快 1758
很怕 93
很急 30
很恨 11
很想 851
很愛 273
很慢 173
很扁 1
很擠 3
很敢 3
很新 38
很早以前 24
很是 61
很暗 11
很會 154
很有 1312
很毒 5
很氣 15
很沒 107
很油 1
很渴 5
很溼 0
很濕 2
很為 18
很熟 52
很熟 52
很狠 0
很甜 11
很瘦 18
很直 10
很短 139
很細 38
很給 6
很美 182
很胖 6
很能 94
很舊 12
很苛 0
很講 5
很讓 34
很趕 13
很酷 77
很長 333
很長 333
很難 2362
很難說 84
很香 8
很高 535
徉 20
徊 27
律 2085
律令 23
律例 13
律呂 2
律己 6
律己甚嚴 0
律師 1418
律師事務所 270
律師公會 56
律度 1
律歷志 0
律法 369
律詩 10
後 66765
後上 39
後上來 1
後上去 0
後下 25
後下來 0
後下去 0
後世 184
後主 42
後事 43
後事之師 0
後人 202
後人乘涼 14
後代 214
後代子孫 123
後仰 23
後件 0
後任 34
後作 40
後來 3908
後來人 3
後來居上 32
後偏 2
後備 58
後備軍 4
後備軍人 239
後傳 19
後分 32
後到 104
後力不繼 0
後力不繼 0
後勁 65
後勤 322
後勤人員 1
後勤區 0
後勤部 0
後區 5
後半 105
後半場 4
後半場 4
後半夜 3
後半季 3
後半期 11
後半段 54
後半部 34
後去 52
後台 129
後台老板 0
後向 99
後周 11
後唐 7
後嗣 13
後園 5
後圖 13
後土 4
後埔 33
後埔 33
後堂 8
後塵 27
後壁 133
後壁鄉 225
後天 318
後天免疫 75
後天性 46
後奏 1
後奏曲 0
後娘 0
後學 853
後宮 272
後山 152
後巷 5
後市 268
後年 124
後序 22
後座 161
後座力 8
後庭 10
後庭花 10
後悔 906
後悔莫及 19
後患 16
後患無窮 16
後房 14
後手 32
後排 39
後掠角 0
後接 64
後援 719
後援軍 0
後撤 7
後攻 1
後放 33
後效 2
後文 17
後方 490
後方區 0
後於 155
後日 25
後晉 6
後晌 0
後晚 3
後景 14
後會 614
後會有期 14
後有 309
後望鏡 0
後期 390
後果 441
後果不堪設想 9
後果堪憂 0
後桅 3
後梁 8
後梁太祖 0
後樑 1
後橋 1
後步 6
後段 219
後殿 48
後母 35
後派 5
後浪 8
後浪推前浪 1
後涼 2
後港 77
後漢 11
後漢書 18
後為 132
後無來者 8
後照鏡 1
後燕 1
後生 35
後生可畏 4
後生小子 2
後用 72
後由 357
後盾 130
後知 7
後知後覺 24
後福 5
後禿 0
後秦 5
後空翻 15
後窗 24
後站 29
後端 142
後竹圍 19
後節 1
後篇 20
後繼 78
後繼有人 5
後繼無人 10
後繼無力 3
後續 2800
後續力 2
後置 58
後者 1032
後肢 29
後背 33
後腦 38
後腦杓 0
後腳 45
後腿 33
後膛 4
後臺 19
後臺老板 0
後花園 46
後菜園 3
後葉 25
後行 34
後街 39
後衛 115
後裔 133
後製 580
後襬 0
後視鏡 100
後計 11
後記 273
後設 72
後讀 4
後走 23
後起 51
後起之秀 44
後趙 9
後足 1
後跟 41
後路 153
後身 18
後車 17
後車站 9
後輩 24
後輩小子 1
後輪 114
後轉 93
後述 12
後退 136
後送 122
後進 89
後進先出 11
後過 12
後遺症 391
後邊 10
後部 27
後鏡 4
後門 153
後防 17
後院 130
後院子 0
後集 9
後面 1346
後項 10
後頭 94
後頸 19
後顧 10
後顧之憂 26
後魏 6
後點 77
後龍 405
後龍溪 39
徍 0
徎 0
徎 0
徏 0
徏 0
徐 7978
徐世昌 4
徐光啟 3
徐匯 24
徐壽輝 0
徐娘半老 1
徐州 148
徐徐 86
徐徐的 14
徐徐地 14
徐志摩 41
徐步 5
徐福 3900
徐緩 20
徐緩而行 0
徐讚昇 1
徐錫麒 0
徑 929
徑向 26
徑度 0
徑庭 0
徑流 1
徑直 5
徑自 2
徑行 3
徑賽 27
徑賽會 0
徑路 11
徒 862
徒具 15
徒具形式 5
徒刑 93
徒勞 16
徒勞往返 5
徒勞無功 39
徒勞無益 2
徒呼奈何 0
徒呼負負 10
徒喚 0
徒喚奈何 0
徒增 65
徒子 1
徒子徒孫 8
徒工 1
徒弟 132
徒手 69
徒手體操 1
徒有 31
徒有其名 0
徒有虛名 1
徒步 178
徒步健行 2
徒步旅行 11
徒法不行 0
徒然 87
徒留 25
徒眾 12
徒託空言 1
徒費 1
徒費口舌 0
徒費唇舌 0
徒費脣舌 0
徒長 7
徒長 7
徒長枝 0
徒長枝 0
従 0
従 0
従 0
従 0
徖 5
得 2948
得 19827
得 147269
得上 107
得上來 1
得上去 1
得下 34
得下來 4
得下去 4
得不償失 58
得不到 252
得不到 252
得不可開交 5
得主 0
得了 426
得以 2537
得來 174
得來不易 47
得來全不費功夫 1
得便 11
得便兒 0
得便兒 0
得免 212
得其三昧 1
得其所哉 1
得出 226
得分 496
得分多 2
得分少 0
得到 5655
得力 162
得力助手 17
得勝 257
得勢 10
得天獨厚 96
得失 154
得失參半 0
得失心 12
得安 28
得宜 109
得寵 1
得寸進尺 9
得很 549
得得 15
得心應手 111
得志 43
得悉 32
得意 453
得意忘形 18
得意揚揚 4
得意洋洋 27
得意門生 12
得慌 1
得成 26
得手 49
得救 378
得數 20
得於 519
得時 14
得未曾有 2
得標 287
得標人 15
得款 3
得氣 10
得法 19
得獎 4843
得獎人 212
得獎者 208
得理不饒人 5
得用 83
得當 120
得病 51
得益 30
得知 1478
得票 730
得罪 215
得而 10
得而復失 4
得而誅之 10
得與 304
得色 21
得逞 53
得逞一時 0
得逞一時 0
得過 98
得過且過 8
得道 49
得道多助 4
得道昇天 1
得隴望蜀 0
得饒人處且饒人 4
得體 26
得魚忘筌 2
得點 15
徘 36
徘徊 238
徘徊不前 0
徙 164
徙居 1
徛 5
徜 46
徜徉 92
徜徉於 25
徝 0
從 37
從 378
從 37802
從一 731
從一而終 18
從上 172
從上到下 15
從上向下 0
從下 90
從下到上 1
從下向上 2
從不 621
從不 621
從不在 4
從中 681
從中作梗 4
從中取利 1
從事 5938
從事於 118
從今 211
從今以後 39
從今天 39
從今天起 86
從今天開始 40
從來 530
從來不 133
從來如此 1
從來未 9
從來未有 2
從來沒 175
從來沒有 364
從來沒有過 9
從俗 9
從價稅 1
從優 33
從優辦理 0
從兄 0
從兄弟 1
從先 29
從公 32
從其所好 0
從前 824
從前人 4
從動 20
從化 10
從古到今 7
從命 16
從商 21
從善如流 17
從嚴 102
從嚴處理 2
從嚴處理 2
從嚴辦理 1
從堂兄弟 0
從堂姐妹 0
從外向內 0
從天而降 26
從子 9
從容 233
從容不迫 20
從容就義 5
從寬 57
從寬處理 13
從寬處理 13
從寬辦理 1
從小 717
從小到大 95
從小到老 2
從小就 200
從屬 79
從屬性 3
從屬於 4
從屬關係 14
從師 18
從心所欲 8
從戎 1
從打 5
從政 207
從新 148
從旁 49
從旁協助 24
從旁觀看 0
從早 157
從早到晚 18
從有到無 1
從未 926
從未在 29
從未有 37
從未有過 17
從未能 6
從業 203
從業人員 444
從業員 32
從此 1180
從此以後 84
從母 19
從江 7
從沒 195
從沒有 89
從無 89
從無到有 37
從父 25
從犯 6
從生到死 6
從略 10
從簡 10
從緩 0
從缺 166
從而 622
從良 12
從裡到外 1
從裡向外 0
從諫如流 1
從軍 74
從軍報國 6
從輕 19
從輕發落 4
從輪 0
從速 69
從長計議 18
從長遠來看 6
從難從嚴 0
從頭 196
從頭做起 1
從頭到尾 155
從頭至尾 18
從頭開始 32
從風 5
徟 8
徠 1
徠 264
御 679
御前 5
御前會議 0
御史 36
御史台 0
御史大夫 8
御夫 15
御夫有術 0
御孫 0
御宇 2
御守 0
御寒 1
御府 1
御庫 0
御廟 0
御弟 0
御所 1
御旨 3
御書 24
御林軍 9
御用 44
御窯 0
御筆 7
御花園 18
御覽 1
御賜 8
御醫 7
御駕 1
御駕親征 9
徢 0
徣 0
徤 0
徥 5
徥 5
徦 4
徧 0
徧 0
徨 177
徨徨 0
復 3957
復交 12
復仇 168
復仇者 104
復仇記 13
復任 2
復位 15
復信 0
復健 1664
復健科 1
復元 30
復出 142
復函 7
復分解 0
復分解反應 0
復刊 23
復利 3
復制 2
復刻 1
復印 4
復原 435
復古 182
復古風 19
復句 0
復合 933
復名數 0
復命 4
復員 13
復員令 0
復員軍人 0
復國 103
復壯 1
復學 127
復學生 14
復審 75
復工 85
復建 200
復建中心 7
復政 0
復新 31
復旦 78
復旦大學 61
復旦橋 2
復明 14
復查 94
復校 62
復校 62
復業 55
復檢 0
復權 34
復次 7
復活 615
復活島 1
復活日 1
復活節 136
復測 0
復現 9
復生 83
復甦 795
復發 289
復種 1
復種指數 0
復籍 0
復習 49
復職 63
復興 4243
復興中學 18
復興劇校 84
復興劇藝學校 0
復興基地 22
復興島 0
復興崗 29
復興工商 166
復興廣播 0
復興廣播電台 10
復興廣播電台 10
復興社 2
復興號 65
復興鄉 271
復興高中 38
復舊 52
復蓋 0
復蘇 20
復診 3
復課 6
復議 35
復讎 3
復賽 1
復辟 24
復辟事件 1
復述 2
復選 5
復電 6
復音 0
循 984
循例 13
循俗 1
循名責實 1
循吏 0
循常 2
循序 173
循序漸進 195
循循善誘 8
循法 20
循理 32
循環 1705
循環利息 0
循環器官 1
循環小數 3
循環性 15
循環氣 0
循環氣流 0
循環系統 73
循環論 3
循線追蹤 1
循規蹈矩 15
徫 10
徬 16
徬徨 122
徭 8
徭役 2
微 4445
微不足道 80
微中子 1
微乎其微 46
微光 30
微冷 1
微分 182
微分學 0
微分方程 60
微分電路 1
微型 89
微型機 0
微塵 47
微妙 239
微婉 0
微孔 25
微安 0
微寒 9
微小 238
微少 3
微居里 3
微差 8
微帶 26
微弱 162
微微 398
微微的 72
微微地 72
微恙 11
微息 1
微指令 0
微晶片 34
微末 9
微氣管 0
微波 478
微波 478
微波爐 102
微波爐 102
微然 0
微現 3
微生 65
微生物 1010
微生物學 231
微眇 0
微秒 27
微積分 572
微笑 982
微笑起來 5
微米 454
微粒 472
微粒子 61
微粒體 11
微細 202
微結構 46
微縮 223
微臣 11
微茫 2
微薄 75
微處理器 0
微處理器 0
微處理機 138
微處理機 138
微血管 97
微行 0
微視 2
微觀 88
微觀世界 3
微觀粒子 0
微言 6
微言大義 1
微詞 35
微調 182
微賤 2
微軟 4254
微辭 3
微醺 20
微量 237
微量元素 83
微量天平 11
微電子 181
微電子學 4
微電腦 847
微電路 2
微震計 0
微額 0
微風 163
微點 0
徯 6
徯徑 1
徰 0
徱 0
徱 0
徲 6
徳 0
徴 0
徴 0
徵 674
徵 6743
徵信 628
徵信所 497
徵信社 50
徵信錄 4
徵候 95
徵兆 151
徵兵 86
徵兵制 15
徵兵法 0
徵兵站 0
徵到 4
徵募 29
徵友 155
徵友欄 0
徵召 179
徵婚 0
徵引 6
徵得 195
徵才 0
徵收 1113
徵收額 0
徵文 273
徵文啟事 3
徵文比賽 89
徵求 1653
徵狀 39
徵用 20
徵稅 65
徵稅額 12
徵稿 424
徵稿欄 0
徵結 12
徵聘 130
徵聘人員 0
徵訓 5
徵詢 232
徵調 17
徵象 45
徵購 40
徵選 0
徵集 181
徵集令 11
徵驗出 0
徶 77
德 19826
德光 102
德克薩斯 5
德克薩斯州 5
德商 169
德國 5094
德國人 108
德國話 0
德國通訊 0
德國通訊社 0
德基水庫 69
德境 2
德威 68
德宗 4
德州 759
德州人 0
德布西 48
德干高原 0
德弗札克 30
德性 200
德惠 129
德意志 53
德意志帝國 1
德意志聯邦 4
德才 18
德才兼備 4
德拉瓦河 1
德政 29
德文 470
德文系 133
德明商專 61
德明商業專科學校 60
德智體 0
德智體群 5
德智體群美 16
德望 6
德業 14
德治 39
德法 34
德法字典 0
德法詞典 0
德澤 12
德育 204
德育護專 24
德興 172
德行 199
德裕 10
德語 175
德語系 36
德貝 7
德軍 67
德里 127
德馨遠播 0
德高望重 20
德黑蘭 9
徸 0
徹 1191
徹夜 78
徹夜不眠 7
徹夜進行 0
徹底 1538
徹底性 1
徹底清除 21
徹底澄清 0
徹徹底底 18
徹悟 18
徹查 72
徹消 3
徹頭徹尾 11
徹骨 13
徻 12
徼 1
徼 285
徼 285
徼 285
徼亂 1
徼亭 1
徼外 1
徼巡 1
徼幸 0
徼幸 0
徼循 1
徼福 1
徼道 1
徽 822
徽墨 0
徽幟 0
徽章 207
徽號 2
徾 7
徿 5
忀 1
忁 4
忂 0
心 24062
心上 125
心上人 38
心不在焉 24
心中 3915
心中有數 2
心中有鬼 1
心亂 10
心亂如麻 5
心事 514
心事重重 9
心儀 103
心儀已久 5
心兒 24
心兒 24
心內 141
心內膜 54
心切 39
心到 27
心力 534
心力交疲 0
心力交瘁 10
心力衰竭 1
心勁 1
心動 496
心包 53
心口 44
心口如一 0
心同此理 3
心向 69
心向祖國 0
心囊 9
心回意轉 0
心土 4
心地 253
心地光明 2
心地善良 34
心地好 1
心地純潔 1
心坎 37
心型 0
心境 262
心如刀割 6
心如懸旌 0
心如止水 13
心如死灰 0
心如鐵石 0
心存僥倖 10
心學 34
心安 65
心安理得 26
心安神定 0
心室 211
心寒 38
心寬 6
心寬體胖 2
心寬體胖 2
心平氣和 73
心底 271
心廣體胖 0
心廣體胖 0
心弦 50
心律 49
心律不整 157
心得 2183
心心 137
心心相印 21
心志 122
心念 99
心思 329
心思慎密 0
心思敏捷 1
心急 63
心急如焚 22
心性 120
心悅誠服 12
心悸 106
心情 7366
心情壞 3
心情好 58
心想 715
心想事成 0
心意 398
心意相投 0
心愛 359
心愛人 1
心慈手軟 0
心態 1076
心態不平衡 1
心態平衡 0
心慌 17
心慌意亂 7
心懷 85
心懷不滿 3
心懷不軌 4
心懷叵測 1
心懷鬼胎 1
心戰 73
心房 194
心扉 53
心手相應 0
心折 5
心明眼亮 0
心智 454
心曠神怡 76
心曲 14
心曲 14
心有千千結 2
心有靈犀 9
心有靈犀一點通 5
心有靈犀一點通 5
心有餘力不足 1
心有餘悸 16
心有餘而力不足 21
心服 36
心服口服 20
心橋 41
心機 123
心死 9
心毒手辣 0
心法 96
心浮氣躁 3
心滿意足 59
心潮 11
心潮澎湃 0
心潮翻滾 0
心潮起伏 0
心澄神淨 0
心火 10
心灰 5
心灰意冷 37
心灰意懶 5
心無二用 0
心無旁騖 1
心無旁鶩 8
心無雜念 1
心焦 11
心照 2
心照不宣 17
心煩 105
心煩意亂 7
心狠 4
心狠手辣 23
心猿意馬 7
心理 5267
心理作用 10
心理因素 75
心理學 1734
心理學史 11
心理學家 148
心理學系 247
心理戰 3
心理治療 214
心理現象 17
心理系 250
心理衛生 338
心理醫生 45
心甘情願 100
心生不滿 0
心田 65
心疼 198
心疾 3
心病 23
心痛 250
心癢 50
心目 61
心目中 523
心直口快 10
心眼 32
心眼兒 1
心眼兒 1
心眼小 1
心知肚明 47
心碎 116
心神 140
心神不定 7
心神不寧 8
心神喪失 15
心神病 0
心神領會 0
心窗 29
心窩 28
心窩兒 0
心窩兒 0
心竅 5
心算 130
心算能力 1
心細 18
心細膽大 0
心結 79
心絞痛 65
心經 194
心緒 33
心緒不寧 2
心織筆耕 0
心耳 2
心聲 656
心肌 192
心肌梗塞 179
心肌炎 44
心肝 45
心肝寶貝 28
心肺復甦術 10
心胸 171
心胸狹小 1
心胸狹窄 9
心胸脥窄 0
心胸開朗 4
心胸開闊 10
心腑 0
心腸 82
心腸壞 0
心腸好 0
心腸硬 0
心腸軟 1
心腹 38
心腹之患 2
心腹大患 6
心膂股肱 10
心膊率 0
心膽 6
心膽俱裂 3
心臟 2368
心臟地區 5
心臟學 26
心臟病 664
心臟病發 25
心臟痲痹 5
心臟痲痺 5
心臟科 164
心臟計 0
心臟鏡 0
心臟麻痹 5
心臟麻痺 5
心花 0
心花怒放 25
心蕩神馳 0
心藥 5
心虛 61
心虛膽怯 0
心血 215
心血來潮 34
心血白費 1
心術 23
心術不正 11
心裁 2
心裡 5268
心裡不安 3
心裡有數 14
心裡話 29
心裡頭 23
心計 11
心許 9
心誠 8
心誠則靈 30
心跡 5
心路 448
心路文教基金會 1
心路歷程 218
心跳 459
心身 46
心軟 8
心軸 44
心迷 6
心連心 31
心部 6
心酸 65
心醉 15
心門 36
心防 0
心電圖 242
心電感應 23
心靈 4068
心靈手巧 1
心靈深處 48
心音 20
心領 100
心領神會 14
心頭 303
心願 368
心餘力竭 0
心餘力絀 2
心驚肉跳 6
心驚膽戰 3
心驚膽跳 20
心驚膽顫 0
心高氣傲 5
忄 0
必 7768
必不可免 1
必不可少 6
必不得已 0
必修 1797
必修科 24
必修課 74
必修課程 328
必停 0
必備 685
必備品 10
必備良藥 0
必先 170
必勝 147
必可 89
必學 15
必定 607
必定是 80
必定會 102
必將 299
必將會 0
必得 130
必從 32
必必剝剝 1
必恭必敬 4
必成 86
必敗 12
必是 127
必會 157
必有 376
必有一傷 1
必有一傷 1
必有其子 0
必有其徒 1
必有我師 7
必有近憂 1
必死 34
必死之心 0
必死無疑 12
必無 17
必無此事 0
必然 1330
必然之事 2
必然之勢 1
必然性 23
必然王國 0
必爭 8
必爭之地 11
必由之路 0
必當 29
必當 29
必看 129
必答 71
必答題 0
必經 102
必經之地 24
必經之路 18
必經之道 1
必罰 2
必考 29
必考題 1
必聽 13
必要 3467
必要勞動 5
必要性 296
必要時 1191
必要條件 102
必讀 125
必賞 1
必輸無疑 2
必選 119
必選課 1
必選課程 23
必需 1791
必需品 70
必須 18116
必須的 108
忇 0
忈 0
忉 11
忉利天 1
忉忉 1
忉怛 1
忊 0
忋 0
忌 812
忌刻 0
忌口 19
忌妒 53
忌憚 20
忌日 13
忌諱 51
忌辰 4
忍 803
忍下 9
忍下來 1
忍下去 0
忍不住 805
忍了 19
忍住 58
忍來忍去 0
忍俊不禁 0
忍個 0
忍冬花 1
忍到 4
忍受 608
忍受得了 2
忍受著 20
忍尤含垢 0
忍得 8
忍得 8
忍心 94
忍性 7
忍悛不禁 0
忍氣吞聲 23
忍無可忍 18
忍痛 150
忍痛犧牲 1
忍者 188
忍者龜 18
忍耐 262
忍耐住 0
忍耐力 8
忍著 73
忍讓 16
忍辱 35
忍辱 35
忍辱負重 11
忍辱負重 11
忍過 0
忍過來 0
忍過去 0
忎 0
忏 7
忏 7
忐 64
忐忑 25
忐忑不安 41
忑 248
忒 597
忒兒摟 0
忒兒摟 0
忒楞楞 0
忓 0
忔 70
忕 15
忖 225
忖度 1
忖量 1
志 13099
志不可滿 1
志不在此 4
志事 3
志同道合 152
志向 93
志在千里 2
志在必得 29
志士 38
志工 0
志得意滿 7
志成 311
志於 24
志書 10
志業 162
志氣 28
志留紀 1
志節 11
志行 3
志行 3
志趣 74
志趣相投 9
志願 917
志願役 1
志願書 11
志願者 19
志願軍 1
志高氣揚 0
忘 901
忘不 3
忘不 3
忘不了 145
忘不掉 14
忘乎 0
忘乎所以 0
忘了 2970
忘卻 120
忘年 90
忘年之交 13
忘年交 0
忘形 18
忘性 1
忘恩 8
忘恩負義 12
忘悼 0
忘情 83
忘情所以 0
忘憂 124
忘憂草 4
忘懷 119
忘懷高歌 0
忘我 64
忘我境界 3
忘掉 134
忘本 10
忘神 9
忘舊 5
忘記 1401
忘記掉 0
忘詞 1
忙 1998
忙不迭 14
忙不過來 10
忙中 10
忙中有錯 2
忙亂 22
忙人 15
忙來 1
忙來忙去 1
忙得 85
忙得 71
忙得不可開交 23
忙得很 6
忙得暈頭轉向 1
忙忙 22
忙忙亂亂 0
忙忙叨叨 0
忙忙急急 0
忙忙碌碌 9
忙成 5
忙成一團 2
忙成一團 2
忙於 124
忙昏 2
忙昏了頭 1
忙東忙西 2
忙的 71
忙碌 681
忙著 301
忙裡 1
忙裡偷閑 0
忙裡偷閒 19
忙裡忙外 0
忙迫 0
忙過 10
忙點 1
忚 0
忚 0
忛 0
忛 0
応 0
応 0
忝 69
忝然 0
忞 14
忟 0
忠 5620
忠于 2
忠信 204
忠僕 4
忠勇 71
忠勇愛國 2
忠勇為愛國之本 1
忠勇雙全 1
忠厚 21
忠厚老實 14
忠君報國 0
忠君愛國 3
忠告 131
忠告 131
忠孝 634
忠孝不能兩全 1
忠孝兩全 0
忠孝國中 33
忠孝國小 54
忠孝東路 1619
忠孝西路 164
忠孝路 523
忠孝醫院 209
忠實 331
忠實可靠 1
忠實度 1
忠實的 118
忠實地 118
忠實聽眾 17
忠實觀眾 10
忠心 142
忠心耿耿 16
忠恕 9
忠於 155
忠於國家 5
忠烈 7
忠烈祠 131
忠直 0
忠純 0
忠義 314
忠義堂 1
忠肝義膽 4
忠臣 40
忠臣孝子 2
忠良 24
忠言 22
忠言逆耳 14
忠誠 477
忠貞 248
忠貞不二 6
忠順 44
忡 270
忡忡 1
忢 0
忣 14
忤 19
忤逆 2
忤違 0
忥 5
忦 0
忧 0
忨 45
忩 0
忪 30
忪 30
快 9914
快一點 101
快一點 101
快上 48
快上來 6
快上去 1
快下 12
快下來 1
快下去 1
快中子 8
快中子 8
快乾 37
快了 107
快事 9
快些 43
快人 8
快人快語 4
快來 638
快信 11
快借 0
快停了 0
快偷 0
快傳 1
快出 32
快出來 20
快出去 1
快刀 24
快刀斬亂麻 18
快到 340
快包 0
快升 1
快去 268
快去快回 5
快取 0
快吃 7
快向 14
快咬 1
快嘴 1
快嘴快舌 0
快回 34
快回來 13
快回到 2
快回去 4
快報 2249
快奏 0
快好了 8
快如 10
快如閃電 4
快守 0
快完了 3
快寫 11
快將 22
快幹 0
快得 38
快得 361
快得很 1
快得很 1
快心 1
快快 370
快快樂樂 177
快想 10
快意 73
快感 371
快慢 72
快慰 11
快成 16
快手 122
快打 218
快找 40
快把 71
快投 0
快抱 0
快指 0
快捷 750
快接 3
快揍 0
快搖 0
快搬 4
快搶 4
快撐 6
快撕 0
快擠 0
快收 8
快攻 45
快有 17
快板 25
快槍 16
快槍俠 1
快樂 5749
快步 72
快步前進 0
快死了 22
快沒 24
快沒了 6
快沒有 2
快沖 1
快治 1
快泡 1
快活 96
快活人 1
快活起來 0
快清 0
快滾 7
快滿了 10
快為 24
快為 24
快照 15
快熟了 0
快玩 2
快由 7
快當 3
快的 361
快地 361
快盛 0
快穿 1
快答 15
快聽 9
快背 0
快背 0
快船 3
快艇 190
快若 0
快若閃電 0
快衝 11
快被 123
快要 371
快訊 24074
快討 1
快記 3
快試 1
快說 48
快請 17
快講 4
快讓 13
快贏 5
快贏了 0
快走 52
快起來 8
快趕 5
快跑 35
快跟 138
快車 72
快車道 23
快轉 18
快辦 9
快逃 14
快速 9480
快速球 1
快速鍵 1
快進 24
快進來 45
快進去 2
快過 29
快過來 9
快過去 2
快遞 539
快適 6
快郵日報 0
快鍋 1
快門 209
快門兒 0
快門兒 0
快閃 35
快閃族 35
快閃記憶體 35
快降 0
快降 0
快餐 43
快餐部 8
快馬 35
快馬加鞭 31
快騎 3
快騎 3
快點 312
忬 0
忭 136
忮 57
忮求 1
忯 12
忰 0
忱 133
忲 0
忲 0
忳 195
忴 41
念 3724
念一念 1
念一念 1
念下去 0
念了 41
念佛 438
念佛三昧 1
念作 0
念來念去 0
念到 16
念咒 9
念在 18
念好 4
念完 19
念得 20
念念 123
念念不忘 36
念念有詞 9
念成 6
念書 163
念法 10
念珠 79
念給 14
念經 35
念舊 30
念茲在茲 10
念著 48
念起 7
念起來 3
念過 20
念錯 8
念頭 370
念點 3
忶 0
忷 49
忸 86
忸 86
忸忸怩怩 1
忸怩 3
忸怩不安 1
忸怩作態 1
忹 0
忹 0
忺 13
忻 119
忼 0
忼 0
忽 759
忽上忽下 8
忽來 2
忽冷忽熱 25
忽前忽後 1
忽地 66
忽地 66
忽大忽小 9
忽左忽右 5
忽強忽弱 0
忽必烈 37
忽快忽慢 2
忽忽 10
忽明 3
忽明忽暗 4
忽明忽滅 0
忽有所感 0
忽滅 1
忽然 1432
忽然之間 11
忽然間 96
忽略 1182
忽的 10
忽而 36
忽聞 22
忽見 66
忽視 644
忽起忽落 1
忽逢 3
忽長忽短 2
忽閃 0
忽高忽低 4
忾 0
忾 0
忿 97
忿忿 5
忿忿不平 30
忿忿然 1
忿怒 80
忿恨 11
忿然 1
忿言 0
忿鷙 0
怀 259
怇 0
怈 0
怉 41
怊 6
怋 5
怌 19
怍 64
怎 1404
怎 1404
怎不 38
怎不 38
怎不是 0
怎不是 0
怎不會 1
怎不會 1
怎不能 0
怎不能 0
怎地 54
怎地 54
怎敢 55
怎敢 55
怎會 378
怎會 378
怎有 9
怎有 9
怎末 0
怎末 0
怎樣 2819
怎樣 2819
怎生 27
怎生 27
怎的 54
怎的 54
怎肯 14
怎肯 14
怎能 324
怎能 324
怎能說 11
怎能說 11
怎行 1
怎行 1
怎見得 2
怎見得 2
怎麼 8149
怎麼 8149
怎麼 8149
怎麼 8149
怎麼回事 190
怎麼回事 190
怎麼回事 190
怎麼回事 190
怎麼得了 3
怎麼得了 3
怎麼得了 3
怎麼得了 3
怎麼搞的 28
怎麼搞的 28
怎麼搞的 28
怎麼搞的 28
怎麼會 533
怎麼會 533
怎麼會 533
怎麼會 533
怎麼樣 810
怎麼樣 810
怎麼樣 810
怎麼樣 810
怎麼著 1
怎麼著 1
怎麼著 1
怎麼著 1
怎麼說 398
怎麼說 398
怎麼說 398
怎麼說 398
怎麼辦 2195
怎麼辦 2195
怎麼辦 2195
怎麼辦 2195
怏 61
怏嚇 0
怏怏 2
怏怏不樂 1
怐 182
怑 86
怒 828
怒不可遏 8
怒吼 91
怒容 8
怒容滿面 1
怒山 2
怒形於色 2
怒意 8
怒放 10
怒斥 15
怒毆 0
怒氣 91
怒氣沖沖 13
怒氣衝天 0
怒氣衝衝 0
怒江 1
怒沖沖 2
怒潮 5
怒潮洶湧 0
怒濤 13
怒濤洶湧 4
怒火 86
怒火中燒 7
怒火萬丈 0
怒火衝天 0
怒目 15
怒目相向 1
怒目而視 3
怒罵 28
怒色 2
怒號 5
怒衝衝 0
怒視 5
怒馬 0
怒髮衝冠 3
怓 141
怔 191
怔 191
怔怔 30
怕 2646
怕不怕 12
怕事 14
怕人 49
怕什麼 51
怕冷 39
怕得 10
怕得很 0
怕得罪人 3
怕得要命 1
怕怕 102
怕死 53
怕熱 14
怕生 16
怕癢 1
怕硬欺軟 0
怕羞 6
怕臊 1
怕苦 6
怕錯 5
怕鬼 14
怖 420
怖懼 3
怗 136
怘 0
怙 3
怙恃 0
怙惡不悛 0
怚 122
怛 41
怜 187
思 60
思 6077
思 67
思凡 15
思前思後 0
思前想後 5
思古 57
思弓 0
思忖 11
思念 605
思恩 78
思惑 1
思惟 69
思想 2876
思想上 49
思想內容 9
思想家 113
思想性 8
思想意識 7
思想感情 4
思想戰線 0
思想方法 11
思想準備 3
思想界 9
思想體系 21
思慕 24
思慮 60
思慮周詳 1
思憶 2
思戀 11
思春 9
思春期 4
思歸鳥 0
思源 303
思潮 369
思潮起伏 0
思索 300
思維 526
思緒 235
思考 4362
思考力 27
思考性 14
思蓋 0
思親 19
思議 46
思路 123
思辨 38
思過 9
思鄉 49
思鄉病 4
思量 75
怞 135
怟 0
怠 215
怠工 10
怠忽 28
怠忽職守 6
怠惰 35
怠慢 21
怠慢心理 0
怠散 0
怠職 1
怠駛 0
怡 4905
怡人 92
怡悅 4
怡情 64
怡情悅性 1
怡情養性 11
怡然 247
怡然自得 20
怡目 1
怡神 8
怢 521
怣 0
怤 18
急 18300
急不容緩 2
急中生智 10
急事 48
急件 104
急促 113
急促性 0
急先鋒 13
急公好義 20
急切 81
急劇 84
急功近利 19
急務 8
急務在身 0
急匆匆 0
急口令 0
急壞 6
急奏 0
急奔 15
急如星火 0
急就章 17
急彎 3
急待 57
急忙 195
急忙中 0
急急忙忙 36
急性 1199
急性子 11
急性病 19
急救 959
急救包 6
急救法 17
急救箱 21
急於 259
急於星火 1
急於求成 2
急智 4
急步 15
急死 16
急水溪 32
急沖而下 0
急流 39
急流勇退 5
急流險灘 1
急湍 9
急煞 12
急用 236
急病 21
急症 122
急著 307
急行軍 1
急衝而下 3
急襲 0
急診 1115
急診室 195
急診處 21
急變 8
急讓 1
急走 9
急起直追 55
急躁 50
急躁不安 3
急轉 19
急轉彎 214
急轉直下 20
急迫 131
急退 9
急速 481
急進 15
急進派 1
急進黨 0
急遽 350
急難 371
急電 14
急需 389
急須 31
急風 7
急馳 7
急馳而去 3
急駛 16
急驚風 1
急驟 8
怦 6
怦怦 11
怦然 19
性 4330
性交 199
性交易 158
性交關係 0
性侵害 1
性別 4529
性向 268
性向測驗 122
性命 292
性命不保 3
性命交關 2
性命攸關 0
性善 4
性器 29
性器官 49
性如烈火 0
性子 50
性徵 23
性急 29
性情 239
性情大變 6
性情平和 0
性情溫柔 0
性惡 10
性愛 613
性感 868
性慾 145
性教育 382
性格 1001
性格小生 1
性格巨星 1
性欲 20
性氣 25
性激素 9
性無能 42
性狀 296
性理 0
性病 217
性相 43
性知識 142
性神經症 1
性科學 10
性經驗 0
性能 1812
性能優越 17
性能力 33
性能好 13
性腺 76
性色 2
性行 67
性行為 338
性衰弱 0
性變態 9
性質 8945
性趣 49
性靈 50
性騷擾 1
怨 603
怨不得 6
怨不得人 2
怨偶 16
怨嗟 0
怨天 7
怨天尤人 26
怨女 39
怨婦 10
怨尤 23
怨念 0
怨忿 2
怨怒 1
怨恨 98
怨悔 1
怨憎恚苦 1
怨憤 3
怨懟 15
怨懣 1
怨歎 2
怨毒 11
怨氣 19
怨氣衝天 0
怨聲 4
怨聲四起 16
怨聲載道 18
怨言 55
怨言四起 0
怨鬼 0
怩 31
怪 2163
怪不得 63
怪了 30
怪事 46
怪事連篇 0
怪人 88
怪來怪去 1
怪傑 37
怪僻 7
怪到 7
怪力 21
怪力亂神 32
怪叫 16
怪哉 37
怪在 10
怪客 64
怪得 5
怪怨 0
怪怪 600
怪怪奇奇 0
怪想 3
怪態 0
怪手 49
怪模怪樣 3
怪物 400
怪獸 905
怪異 159
怪異性 0
怪病 16
怪癖 23
怪石 25
怪石嶙峋 4
怪笑 5
怪笑聲 2
怪罪 0
怪聲 34
怪聲怪氣 1
怪胎 14
怪腳 4
怪著 0
怪裡怪氣 0
怪話 4
怪誕 20
怪誕不經 0
怪談 22
怪論 0
怪起 1
怪起來 0
怪過 0
怪雨 0
怪魚 4
怫 13
怫 13
怬 84
怭 5
怮 265
怯 151
怯 151
怯場 8
怯場 8
怯場 8
怯場 8
怯弱 2
怯弱 2
怯怯地 6
怯怯地 6
怯怯地 6
怯怯的 1
怯怯的 6
怯意 2
怯意 2
怯懦 12
怯懦 12
怯生生 3
怯生生 3
怯生生的 2
怯生生的 2
怯生生地 2
怯生生地 2
怯疑 0
怯疑 0
怯頭怯腦 0
怯頭怯腦 0
怰 0
怱 0
怲 2
怳 199
怴 875
怵 68
怵惕 0
怵然 1
怵目驚心 34
怶 0
怷 111
怸 0
怸 0
怹 131
怽 0
恀 699
恁 886
恁地 6
恁地 6
恁地 6
恁時 1
恁麼 1
恂 68
恃 100
恃勢欺人 0
恃寵而驕 2
恃強凌弱 2
恃強欺弱 0
恃才傲物 5
恄 58
恅 882
恆 2293
恆久 74
恆壓 5
恆壓器 0
恆定 33
恆常 25
恆心 54
恆星 242
恆星年 0
恆星日 0
恆春 711
恆有 0
恆毅 61
恆毅中學 18
恆河 37
恆河平原 0
恆河沙 1
恆河沙數 9
恆溫 172
恆溫器 2
恆溫層 0
恆產 9
恆等 4
恆等式 0
恆言 5
恆量 0
恆齒 4
恇 58
恈 0
恉 64
恊 0
恋 0
恌 64
恍 68
恍如夢境 0
恍如隔世 5
恍恍忽忽 1
恍恍惚惚 14
恍惚 63
恍惚之間 2
恍然 35
恍然大悟 51
恍神 1
恍若 36
恍若隔世 2
恎 0
恏 0
恐 1301
恐嚇 283
恐嚇信 29
恐嚇性 3
恐嚇罪 10
恐怕 1257
恐怕是 162
恐怕會 82
恐怕有 13
恐怕要 52
恐怖 729
恐怖主義 29
恐怖份子 80
恐怖份子 80
恐怖片 33
恐怖症 3
恐怖電影 10
恐惶 2
恐慌 312
恐懼 793
恐懼感 39
恐是 10
恐有 65
恐水病 0
恐為 5
恐無 26
恐獸 0
恐脅 0
恐角獸 1
恐赫 1
恐需 8
恐非 21
恐龍 1033
恐龍蛋 15
恐龍骨 0
恑 0
恒 655
恓 286
恔 94
恕 846
恕不 0
恕不 0
恕不奉陪 1
恕不奉陪 1
恕罪 9
恕道 0
恕邀 0
恖 0
恖 0
恗 0
恗 0
恘 78
恙 276
恙蟲 1
恚 17
恚怒 1
恚恨 0
恚憤 0
恛 119
恜 0
恝 3
恞 93
恟 66
恟恟 0
恠 0
恡 0
恢 193
恢宏 35
恢弘 14
恢復 2827
恢復起來 0
恢復過來 15
恢恢 2
恣 320
恣 320
恣情 6
恣意 115
恣意妄為 3
恣橫 0
恣睢 1
恣縱 3
恣肆 2
恤 259
恤孤 2
恤民 4
恤貧 1
恤金 1
恥 502
恥丘 1
恥廉 502
恥毛 1
恥笑 25
恥辱 52
恥骨 20
恦 40
恧 1
恨 1090
恨不得 83
恨不相逢未娶時 2
恨不相逢未嫁時 2
恨不能 5
恨之入骨 18
恨了 2
恨事 1
恨人 6
恨來恨去 0
恨入骨髓 2
恨入骨髓 2
恨到 5
恨女不成鳳 0
恨得 18
恨得 18
恨意 24
恨意全消 0
恨意未消 0
恨意盡消 0
恨意難消 0
恨死 17
恨起來 0
恨透 1
恨透了 4
恨鐵不成鋼 4
恩 5083
恩主 29
恩主公 29
恩人 16
恩人不求報 0
恩仇 35
恩仇記 9
恩克魯瑪 0
恩典 398
恩典如山 0
恩准 2
恩同再造 1
恩威 9
恩威並施 1
恩威並用 1
恩威並行 0
恩客 67
恩寵 67
恩將仇報 5
恩將仇報者 0
恩山義海 1
恩師 81
恩德 46
恩德如山 0
恩怨 146
恩怨分明 4
恩怨情天 1
恩恩怨怨 13
恩恩愛愛 6
恩情 57
恩情似海深 0
恩情如山高 0
恩惠 67
恩惠深重 0
恩愛 107
恩愛夫妻 4
恩愛情深 0
恩愛有加 0
恩斯特 6
恩斷義絕 0
恩格斯 18
恩深義重 0
恩澤 19
恩澤深厚 0
恩相 6
恩義 6
恩賜 216
恩賜 216
恩賜者 1
恩賜者 1
恩重如山 2
恩重如山高 0
恪 58
恪 58
恪守 23
恪守 23
恪守不渝 0
恪守不渝 0
恪遵 12
恪遵 12
恫 11
恫 11
恫喝 0
恫嚇 53
恫瘝 0
恫瘝在抱 1
恫瘝在身 1
恬 161
恬不知恥 1
恬和 0
恬淡 48
恬澹 3
恬然 4
恬適 16
恬靜 49
恭 903
恭人 0
恭候 22
恭候光臨 0
恭候大駕 3
恭喜 1020
恭喜發財 19
恭從 0
恭恭 1
恭恭敬敬 19
恭敬 92
恭敬不如從命 6
恭敬地 16
恭敬地 16
恭祝 50
恭禧 170
恭維 30
恭維話 2
恭聽 0
恭請 53
恭謁 1
恭謹 10
恭讀 12
恭賀 247
恭賀新禧 14
恭迎 45
恭送 16
恭順 1
恮 33
息 3359
息事 0
息事寧人 8
息兵 0
息影 6
息怒 7
息息攸關 0
息息相通 0
息息相關 286
息戰 4
息火 5
息燈 0
息肉 73
息訟 0
息謗 0
息錢 2
恰 446
恰似 45
恰值 3
恰克圖 2
恰到好處 48
恰好 184
恰好是 20
恰如 42
恰如其分 20
恰巧 111
恰巧是 11
恰恰 96
恰恰好 7
恰恰相反 12
恰當 298
恱 0
恲 86
恳 0
恴 0
恵 0
恼 0
恾 0
恿 19
悀 421
悁 256
悂 0
悂 0
悃 289
悃誠 0
悄 212
悄悄 229
悄悄 229
悄悄地 94
悄悄地 115
悄悄的 94
悄悄話 598
悄悄話 598
悄然 76
悄然淚下 0
悄然而去 0
悄然離去 1
悄聲 11
悅 1086
悅目 28
悅耳 75
悅耳動聽 3
悆 6
悇 52
悈 113
悉 619
悉心 42
悉數 51
悉數 51
悉皆 14
悉聽 1
悉聽尊便 6
悊 3
悊 3
悋 0
悌 293
悍 210
悍人 0
悍妻 1
悍婦 3
悍將 195
悍戾 0
悍然 6
悍然不顧 0
悍然毀約 0
悍藥 0
悎 124
悏 0
悐 0
悑 0
悒 326
悒悒 2
悒悒不快 1
悒悒不樂 1
悒悒不歡 1
悒悒不然 1
悒然 0
悒鬱寡歡 0
悓 0
悓 0
悓 0
悔 602
悔不當初 10
悔之已晚 1
悔之晚矣 0
悔之莫及 0
悔婚 2
悔恨 54
悔悟 22
悔悟之意 0
悔意 23
悔改 192
悔棋 2
悔氣 0
悔過 14
悔過書 7
悕 209
悖 276
悖亂 0
悖入悖出 0
悖理 4
悖禮 1
悖謬 6
悖逆 24
悗 98
悙 0
悚 1
悚 143
悚懼 1
悚然 4
悛 552
悜 69
悝 60
悞 0
悟 1013
悟出 27
悟出來 2
悟力 1
悟性 40
悟解 47
悟道 31
悠 2450
悠久 322
悠哉 29
悠哉悠哉 5
悠悠 133
悠悠忽忽 0
悠悠然 2
悠悠的 11
悠悠地 11
悠悠蕩蕩 4
悠揚 65
悠明 0
悠然 62
悠然自得 11
悠遊 212
悠遠 16
悠長 25
悠閑 16
悠閒 234
悠閒地 18
悠閒地 18
悡 0
悢 114
悢 114
患 1181
患上 27
患得 1
患得患失 22
患有 356
患病 98
患者 3112
患苦 0
患處 21
患部 80
患難 95
患難之交 11
患難與共 10
患難見人心 0
患難見真情 1
悤 0
悥 0
悦 0
悧 0
您 103088
您們 609
您好 1154
您要 1060
您說 283
悩 0
悪 0
悪 0
悪 0
悮 0
悰 7
悱 6
悱惻 4
悲 1155
悲傷 547
悲切 8
悲劇 458
悲劇性 12
悲哀 376
悲啼 1
悲喜交加 4
悲喜交集 4
悲嘆 7
悲壯 34
悲天憫人 29
悲從中來 6
悲怨 1
悲恨 1
悲悼 2
悲悽 6
悲情 0
悲愁 6
悲愴 21
悲慘 292
悲慟 10
悲慟不已 1
悲憤 54
悲憫 36
悲懷 4
悲歌 101
悲歎 4
悲歡 29
悲歡離合 29
悲泣 20
悲涼 17
悲痛 97
悲苦 44
悲號 1
悲觀 271
悲觀主意 0
悲觀派 3
悲鳴 22
悲鳴不已 0
悳 0
悴 15
悵 888
悵悵 7
悵惘 8
悵惘若失 0
悵望 2
悵然 24
悵然若失 4
悶 414
悶 414
悶住 1
悶住 1
悶到 0
悶到 0
悶在 13
悶在心裡 7
悶在心裡 7
悶得很 1
悶得很 1
悶得慌 2
悶得慌 2
悶悶 25
悶悶 25
悶悶 25
悶悶 25
悶悶不樂 62
悶悶不樂 62
悶悶不樂 62
悶悶不樂 62
悶死 27
悶死 27
悶氣 21
悶氣 21
悶火 1
悶火 1
悶熱 57
悶熱 57
悶燒 0
悶燒 0
悶燒鍋 0
悶燒鍋 0
悶聲 6
悶聲 6
悶聲不響 3
悶聲不響 3
悶聲悶氣 0
悶聲悶氣 0
悶聲悶氣 0
悶聲悶氣 0
悶著 5
悶著 5
悶著 5
悶著 5
悶著頭 1
悶著頭 1
悶葫蘆 7
悶葫蘆 7
悶酒 4
悶酒 4
悶雷 1
悶雷 1
悶頭 1
悶頭 1
悶騷 24
悶騷 24
悷 1
悸 229
悸動 90
悸慄 0
悹 22
悺 2
悻 978
悻悻 1405
悻悻然 9
悻悻而去 0
悼 514
悼亡 4
悼喪 59
悼念 59
悼惜 0
悼文 1
悼歌 0
悼痛 0
悼詞 5
悽 55
悽切 0
悽厲 7
悽怨 0
悽悽 8
悽惻 2
悽愴 5
悽慘 41
悽戾 0
悽楚 2
悽涼 11
悽然 1
悽苦 0
悽豔 0
悽酸 1
悾 5
悿 31
惀 40
惁 3
惂 0
惃 45
惄 2
情 19579
情不自禁 62
情不自禁 62
情事 996
情人 2114
情人眼裡出西施 4
情人節 675
情人裝 5
情份 37
情何以堪 0
情侶 358
情分 14
情勢 2417
情勢所迫 0
情勢所逼 3
情勢逆轉 4
情史 21
情同手足 6
情商 35
情報 13053
情報人員 42
情報員 40
情報局 272
情報系統 36
情報資料 21
情場 38
情場 38
情場如戰場 0
情場如戰場 0
情場如戰場 0
情場如戰場 0
情場老將 1
情場老將 1
情境 864
情夫 9
情婦 53
情形 9472
情急 19
情急之下 18
情急智生 1
情急生智 0
情怯 9
情怯 9
情意 191
情意深厚 0
情意深重 1
情意相投 1
情意綿綿 4
情愛 923
情感 1159
情感衝動 2
情愫 71
情態 5
情態詞 0
情慾 216
情懷 425
情投意合 18
情操 185
情敵 46
情文並茂 1
情景 301
情書 443
情書大全 3
情有可原 8
情有所鍾 2
情欲 64
情歌 339
情歸何處 4
情治 0
情況 7349
情況不明 5
情海 118
情深似海 2
情狀 27
情理 35
情理法 13
情理難容 1
情由 12
情禮兼到 0
情竇 1
情竇初開 11
情節 919
情素 61
情結 338
情絲 36
情網 228
情緒 2269
情緒低落 17
情緒化 52
情緒性 34
情緒高漲 12
情緣 158
情義 86
情義深重 0
情至意盡 0
情色 0
情苗 0
情見乎辭 0
情詩 101
情話 152
情話綿綿 20
情誼 199
情誼 199
情調 76
情趣 1112
情郎 19
情長紙短 0
情隨事遷 0
情面 7
情面難卻 0
情韻 9
情願 129
惆 417
惆悵 35
惆然 1
惇 144
惇厚 0
惇惇 0
惇樸 0
惇睦 0
惇篤 0
惇誨 0
惇謹 0
惈 4
惉 14
惊 386
惋 17
惋傷 0
惋惜 77
惋愕 0
惋歎 0
惌 7
惍 11
惎 6
惏 5
惐 0
惐 0
惑 361
惑亂 5
惑眾 6
惒 0
惒 0
惒 0
惒 0
惒 0
惒 0
惓 3
惓 3
惔 2
惕 539
惕厲 7
惕懼 0
惖 0
惗 0
惘 688
惘然 25
惘然若失 0
惙 7
惚 1486
惛 16
惛懵 0
惜 1692
惜別 57
惜別會 5
惜售 19
惜墨如金 1
惜字 2
惜物 6
惜玉憐香 0
惜福 131
惜陰 2
惝 3
惞 0
惟 6900
惟一 116
惟利是圖 1
惟命是從 0
惟命是聽 0
惟妙惟肖 4
惟心 3
惟心論 0
惟恐 48
惟恐天下不亂 8
惟我獨尊 0
惟有 351
惟物 0
惟物論 0
惟獨 34
惠 10404
惠存 3
惠安 76
惠州 45
惠帝 11
惠施 7
惠更斯 2
惠民 91
惠澤 7
惠而不費 4
惠臨 2
惠賜 265
惠賜一票 7
惠賜一票 7
惠鑒 46
惠陽 45
惠靈頓 2
惠顧 45
惡 19
惡 19
惡 1968
惡 1968
惡人 129
惡人先告狀 4
惡人有惡報 0
惡作劇 142
惡作劇者 0
惡例 6
惡俗 1
惡兆 3
惡劣 423
惡劣性 0
惡勢力 37
惡化 864
惡化趨勢 4
惡化頃向 0
惡口 1
惡名 13
惡名昭彰 16
惡因 14
惡報 15
惡夢 226
惡婆 0
惡婦 1
惡客 8
惡少 3
惡形 3
惡形惡狀 3
惡徒 12
惡心 35
惡心 35
惡心感 0
惡念 16
惡性 555
惡性不改 0
惡性倒閉 12
惡性循環 88
惡性瘤 5
惡性補習 2
惡意 610
惡意中傷 11
惡意毀謗 1
惡感 16
惡戰 22
惡搞 100
惡政 0
惡有善報 0
惡有惡報 8
惡果 44
惡棍 12
惡棍歹徒 0
惡業 1
惡極 0
惡毒 72
惡漢 6
惡犬 52
惡狠 1
惡狠狠 19
惡疾 11
惡瘡 2
惡癖 0
惡相 2
惡神 4
惡終 1
惡罵 0
惡習 69
惡習不改 0
惡耗 7
惡聲 7
惡臭 148
惡臭味 0
惡臭性 0
惡行 95
惡行 95
惡衣惡食 1
惡補 23
惡言 19
惡言惡語 0
惡言相向 16
惡語 3
惡語相向 0
惡貫滿盈 5
惡賊 11
惡質 151
惡質化 36
惡運 20
惡霸 16
惡霸地主 1
惡霸成性 0
惡風 4
惡食 0
惡鬼 43
惡魔 318
惡魔島 23
惡魔黨 318
惢 4
惣 0
惤 3
惥 0
惥 0
惦 39
惦念 4
惦念著 5
惦懷 0
惦掛 0
惦掛著 1
惦著 7
惦記 11
惦記著 16
惧 0
惨 0
惩 0
惩 0
惪 0
惮 0
惮 0
惰 104
惰性 28
惰性氣體 20
惱 180
惱亂 1
惱人 134
惱怒 15
惱恨 5
惱火 6
惱火起來 0
惱羞成怒 16
惲 11
想 20353
想上 154
想上來 9
想上去 5
想下來 1
想下去 3
想不出 84
想不到 577
想不起來 37
想不通 30
想不開 41
想也不想 8
想也別想 0
想了 263
想了又想 3
想住 7
想你 613
想來 298
想來想去 20
想個 36
想個方法 1
想個辦法 16
想借 22
想做 340
想做到 0
想像 1897
想像出 14
想像出來 0
想像到 14
想像力 294
想像得到 19
想像成 23
想光 0
想入 16
想入非非 9
想再 355
想出 206
想出來 14
想出去 14
想到 2148
想包 1
想去 481
想受 8
想吸 2
想哭 137
想唱 14
想喝 34
想回 64
想回來 8
想回去 4
想在 508
想坐 12
想坐在 1
想報 0
想好 84
想妳 270
想學 254
想完 6
想家 37
想將 186
想幹什麼 3
想幹什麼 3
想得 123
想得 359
想得到 64
想得到 64
想得起來 1
想得開 3
想必 437
想念 505
想念起 7
想想 965
想想看 199
想愛 62
想成 122
想打 63
想把 429
想投 13
想抱 8
想接 14
想方設法 3
想有 112
想望 41
想死 58
想法 1716
想法子 29
想洗 3
想為 123
想玩 133
想用 235
想當然 59
想的 359
想盡 34
想盡方法 6
想盡辦法 68
想看 685
想看看 130
想睡 56
想穿 10
想笑 23
想約 9
想編 3
想著 551
想蘊 1
想裝 22
想要 4981
想見 173
想認 25
想說 497
想說出 1
想說的 42
想講 28
想變 13
想象 63
想象力 3
想走 51
想走來 0
想走到 1
想走去 0
想走過 0
想起 1236
想起來 93
想載 0
想送 36
想透 20
想通 46
想進 198
想進來 6
想進去 2
想過 482
想過來 3
想過去 5
想過著 1
想都不想 3
想都別想 4
想錄 12
想開 37
想除 3
想靠 25
想頭 1
惴 83
惴恐 0
惴惴 1
惴惴不安 3
惴慄 1
惵 1
惶 70
惶急 2
惶恐 81
惶恐狀 0
惶悚 0
惶悸 0
惶惑 14
惶惶 15
惶惶不可終日 5
惶惶無主 0
惶惶然 0
惶懼 3
惶擾 0
惶然 6
惶遽 1
惷 5
惸 4
惹 428
惹下 3
惹不起 5
惹事 9
惹事生非 8
惹人 67
惹人厭 43
惹人注意 2
惹人注目 6
惹人討厭 3
惹來 45
惹出 26
惹出來 0
惹得 48
惹是生非 4
惹是非 7
惹氣 2
惹火 31
惹火燒身 0
惹眼 1
惹禍 19
惹禍上身 6
惹起 25
惹起來 0
惺 40
惺忪 19
惺惺 3
惺惺作態 1
惺惺相惜 17
惻 57
惻然 1
惻隱 5
惻隱之心 15
惼 8
惽 0
惾 0
惿 3
愀 199
愀然 0
愀然變色 0
愁 936
愁城 4
愁容 13
愁容滿面 2
愁思 3
愁悒 0
愁悶 4
愁滋味 6
愁眉不展 7
愁眉苦眼 0
愁眉苦臉 14
愁緒 17
愁腸 8
愁腸九轉 0
愁腸寸斷 0
愁腸百結 0
愁苦 22
愁雲 9
愁雲慘霧 27
愁霧 0
愁顏 0
愂 0
愂 0
愃 19
愄 4
愅 4
愆 11
愆罪 0
愆過 0
愇 0
愈 8001
愈用 44
愈快 100
愈來 24
愈來愈 1372
愈來愈好 15
愈加 38
愈合 7
愈好 130
愈快愈好 6
愈戰愈勇 5
愈演愈烈 52
愈益 11
愉 211
愉快 1232
愉悅 217
愉色 0
愊 7
愋 3
愌 0
愍 27
愎 66
意 14786
意下 5
意下如何 27
意中人 7
意亂情迷 22
意到筆隨 0
意向 183
意含 488
意味 302
意味深長 10
意味著 366
意圖 488
意境 117
意外 1955
意外事件 398
意外事故 227
意外保險 32
意外險 126
意外風波 0
意外風波 0
意大利 258
意大利人 2
意志 428
意志力 101
意志消沈 7
意念 315
意思 2096
意想 22
意想不到 205
意想到 0
意態 8
意料 20
意料不到 5
意料中 16
意料之中 13
意料之外 43
意料到 0
意旨 196
意會 35
意會出 0
意有所指 1
意欲 90
意氣 40
意氣揚揚 0
意氣昂揚 3
意氣用事 19
意氣相投 2
意氣風發 40
意涵 0
意淫 5
意猶未盡 65
意筆 3
意義 4607
意興 9
意興闌珊 18
意表 3
意見 28364
意見分歧 27
意見書 85
意見調查 172
意見調查表 81
意謂 98
意謂著 67
意識 2273
意識到 205
意識形態 248
意識流 13
意譯 18
意象 350
意趣 14
意門 1
意願 1497
意麵 34
愐 2
愑 0
愒 368
愒 368
愒時 0
愓 10
愔 24
愕 66
愕然 91
愕視 0
愖 3
愗 0
愘 2
愙 0
愙 0
愚 390
愚不可及 3
愚人 46
愚人節 23
愚兄 14
愚公 6
愚公移山 13
愚勇 2
愚夫愚婦 5
愚妄 2
愚孝 2
愚弄 23
愚弄人民 0
愚弄百姓 0
愚弟 1
愚忠 7
愚拙 10
愚昧 68
愚昧無知 6
愚民 11
愚民政策 9
愚癡 1
愚直 1
愚笨 32
愚蠢 130
愚見 18
愚鈍 1
愚陋 2
愚魯 4
愛 19146
愛上 1005
愛不 107
愛不愛 50
愛不釋手 87
愛世語 1
愛之 226
愛之深 6
愛之船 52
愛之適足以害之 3
愛了 51
愛人 635
愛人如己 14
愛侶 23
愛侶情深 0
愛克斯 2
愛克斯光 2
愛兒 54
愛兒心切 0
愛別離苦 1
愛到 93
愛到深處 1
愛到深處無怨尤 0
愛力 30
愛動 6
愛卿 18
愛吃 192
愛吃鬼 1
愛吸 0
愛哭 35
愛哭鬼 7
愛因斯坦 154
愛國 412
愛國主義 8
愛國人士 1
愛國心 13
愛國東路 80
愛國獎券 5
愛國者 74
愛國衛生運動 0
愛國西路 32
愛國路 17
愛國運動 8
愛在 294
愛在心裡 1
愛在心裡口難開 27
愛奏 0
愛女 43
愛女心切 2
愛她 176
愛好 489
愛好者 340
愛妻 47
愛妻心切 0
愛妾 7
愛子 53
愛子心切 4
愛家 131
愛屋及烏 8
愛師 0
愛師心切 0
愛帶 3
愛徒 14
愛徒心切 0
愛得 1943
愛德華 96
愛德蒙 8
愛心 1774
愛心彩券 0
愛恨 89
愛恨交織 11
愛恨分明 10
愛情 4512
愛情史 5
愛情專一 3
愛情觀 88
愛惜 151
愛惡分明 2
愛想 2
愛意 102
愛慕 120
愛慕者 21
愛憎 16
愛憎分明 0
愛憐 21
愛戀 114
愛戀著 2
愛戴 68
愛才 28
愛才若渴 0
愛打 34
愛抽 0
愛抽煙 1
愛撫 52
愛斯基摩 4
愛斯基摩人 16
愛新覺羅 16
愛書 89
愛書狂 0
愛書癖 0
愛書迷 0
愛校 33
愛校如家 0
愛校精神 0
愛樂 895
愛樂社 18
愛權愛勢 0
愛死 126
愛民 25
愛民如子 0
愛沙尼亞 91
愛沙尼亞人 1
愛河 122
愛洗 4
愛滋 0
愛滋病 730
愛滋病患 51
愛演 1
愛火 14
愛爾蘭 368
愛爾蘭人 11
愛爾蘭共和國 4
愛爾蘭共和軍 1
愛爾蘭語 1
愛物 26
愛犬 157
愛玉 61
愛玉冰 4
愛玉凍 2
愛玩 114
愛現 42
愛現鬼 0
愛理不理 14
愛琴海 36
愛生 14
愛生氣 9
愛用 119
愛用者 212
愛的 1943
愛神 111
愛神維納斯 1
愛笑 89
愛笑鬼 0
愛管閒事 7
愛美 123
愛群 18
愛苗 10
愛荷華州 73
愛莫能助 15
愛著 105
愛說 70
愛說笑 81
愛護 471
愛護公物 8
愛護動物 25
愛財 8
愛財如命 1
愛走 13
愛起 2
愛起來 0
愛車 414
愛車狂 0
愛車迷 1
愛迪生 35
愛過 155
愛達荷州 7
愛鄉 366
愛鄉如家 0
愛鄉精神 0
愛錢 18
愛錢如命 1
愛錢鬼 0
愛錯 15
愛阿華州 7
愛顧 9
愛馬 14
愛麗絲 102
愛麗絲夢遊仙境 17
愛默森 11
愛默生 4
愜 29
愜意 60
愝 5
愞 0
愞 0
感 1000
感人 266
感人肺腑 18
感佩 65
感傷 154
感光 136
感光劑 2
感光度 17
感光性 22
感光片 0
感光紙 0
感冒 1183
感冒藥 45
感到 3083
感動 1695
感動得 34
感化 58
感化院 16
感受 1911
感受到 922
感受力 20
感受器 6
感受性 118
感召 49
感召力 0
感同身受 77
感喟 1
感嘆 222
感嘆詞 1
感官 337
感念 110
感性 650
感性認識 1
感恩 910
感恩不盡 4
感恩圖報 3
感恩戴德 1
感恩節 84
感悟 10
感悟到 0
感情 2564
感情作用 0
感情用事 25
感想 773
感慨 193
感慨萬千 18
感慰 0
感應 556
感應力 10
感應器 132
感應圈 1
感應圖 0
感應性 1
感應爐 2
感應線 2
感應電 38
感應電流 8
感懷 33
感抗 3
感於 45
感染 3952
感染力 27
感染性 142
感染率 59
感歎 18
感歎句 0
感歎號 0
感歎詞 0
感泣 0
感激 506
感激不盡 479
感激涕零 5
感激萬分 3
感生 1
感生電流 0
感發 4
感知 172
感興趣 235
感覺 6281
感覺出 29
感覺到 489
感覺器 1
感覺器官 19
感覺神經 29
感觸 279
感觸良多 16
感言 1230
感謝 6810
感謝禮 0
感謝詞 7
感電 21
愠 0
愡 0
愢 0
愢 0
愣 155
愣住 17
愣小子 0
愣小子 0
愣得 0
愣得 0
愣愣 4
愣著 3
愣頭愣腦 0
愧 56
愧不敢當 2
愧作 0
愧對 19
愧心 0
愧怍 0
愧恨 0
愧於 15
愧為 10
愧疚 43
愧色 5
愨 57
愩 3
愪 0
愫 32
愬 48
愭 0
愮 6
愯 9
愰 0
愱 0
愲 5
愳 0
愴 23
愴悼 0
愴然 8
愴然淚下 1
愵 0
愶 6
愷 375
愷切 2
愷悌 2
愸 0
愹 0
愺 0
愻 114
愼 0
愽 0
愾 3
愾憤 0
愿 280
慀 10
慁 49
慂 0
慂 0
慃 0
慄 20
慅 5
慆 8
慇 102
慇勤 2
慇勤招待 0
慈 3000
慈不主兵 1
慈善 1319
慈善事業 357
慈善家 8
慈善機構 47
慈幼 66
慈心 73
慈悲 525
慈悲殺人 1
慈悲為懷 16
慈悲為本 1
慈悲觀 1
慈惠 158
慈愛 196
慈憫 1
慈故能勇 1
慈暉 1
慈母 53
慈母手中線 1
慈湖 104
慈濟 5648
慈濟功德會 1
慈濟大學 1
慈烏 1
慈烏反哺 1
慈烏返哺 1
慈父 5
慈父嚴母 0
慈祥 63
慈禧太后 18
慈禧皇太后 2
慈航 43
慈航普渡 1
慈誨 1
慈顏 3
慈鳥 0
慉 7
慊 11
態 2589
態勢 306
態度 4333
慌 145
慌亂 63
慌了 18
慌了手腳 6
慌作一團 0
慌叫 0
慌張 66
慌張張 1
慌張起來 0
慌得 1
慌忙 65
慌忙之間 1
慌慌 18
慌慌張張 17
慌成 0
慌成一團 0
慌裡慌張 0
慍 22
慍容 0
慍怒 0
慍恚 1
慍懟 1
慍色 2
慎 716
慎到 0
慎始 2
慎始慎終 0
慎密 11
慎微 1
慎思 22
慎思明辨 3
慎思篤行 0
慎獨 6
慎終 1
慎終追遠 11
慎行 5
慎言 10
慎謀遠慮 0
慎重 261
慎重其事 12
慎重考慮 50
慎重處理 5
慎重處理 5
慎防 86
慎防意外 2
慏 4
慒 79
慓 82
慓悍 2
慔 67
慕 963
慕名 32
慕名而來 25
慕容 610
慕尼黑 181
慖 41
慗 0
慘 674
慘不忍睹 38
慘兮兮 23
慘劇 93
慘叫 50
慘叫聲 27
慘境 2
慘壯 0
慘敗 49
慘案 21
慘歷 0
慘死 68
慘死輪下 0
慘毒 2
慘淡 26
慘淡經營 13
慘澹 14
慘澹經營 28
慘烈 71
慘無人道 8
慘無天日 0
慘然 8
慘狀 18
慘獄 0
慘痛 73
慘痛教訓 24
慘白 28
慘相 3
慘禍 4
慘笑 2
慘絕人寰 21
慘綠 0
慘苦 0
慘變 3
慘跌 22
慘遭 133
慘遭橫禍 0
慘遭殺害 5
慘遭毒害 0
慘遭毒手 6
慘酷 4
慘重 140
慙 0
慚 34
慚愧 165
慚色 3
慛 61
慜 0
慝 6
慞 36
慟 43
慟哭 5
慠 0
慡 73
慢 10633
慢下 12
慢下來 12
慢下去 1
慢了 119
慢來 13
慢來慢去 0
慢到 31
慢動作 42
慢化劑 0
慢半拍 17
慢吞吞 31
慢工 5
慢工出細活 9
慢待 0
慢得 122
慢性 1375
慢性子 0
慢性病 385
慢悠悠 2
慢慢 1466
慢慢來 32
慢慢吞吞 2
慢慢地 429
慢慢地 351
慢慢的 429
慢慢騰騰 0
慢手慢腳 0
慢曲 0
慢板 27
慢板情歌 0
慢條斯理 18
慢步 13
慢火 9
慢用 6
慢的 122
慢著 8
慢藏誨盜 0
慢說 2
慢走 22
慢起來 0
慢跑 194
慢車 35
慢車道 16
慢速 101
慢速度 33
慢鏡頭 2
慢騰騰 0
慢點 7
慢點兒 1
慢點兒 1
慣 328
慣上 1
慣下 0
慣住 0
慣例 230
慣習 100
慣唱 0
慣壞 2
慣常 19
慣得 0
慣性 124
慣性作用 1
慣性力 5
慣性定律 1
慣想 0
慣打 1
慣技 0
慣拍 0
慣排 0
慣擺 0
慣於 33
慣有 26
慣洗 0
慣派 0
慣為 0
慣犯 3
慣用 145
慣用手法 3
慣用語 10
慣竊 7
慣答 0
慣罰 0
慣要 0
慣講 0
慣讀 1
慣賊 0
慣跑 0
慣量 2
慣防 0
慣食 0
慣駛 0
慣騎 0
慣點 0
慤 0
慥 44
慦 86
慧 7949
慧光 6
慧劍 9
慧劍斬情絲 0
慧心 30
慧性 0
慧星 117
慧根 23
慧眼 198
慧眼識英雄 4
慧種 0
慧能 49
慧遠 16
慧黠 12
慨 182
慨 182
慨允 11
慨允 11
慨嘆 16
慨嘆 16
慨歎 10
慨歎 10
慨然 24
慨然 24
慨然相助 0
慨然相助 0
慩 0
慪 95
慫 294
慫恿 40
慬 75
慭 0
慮 675
慯 0
慰 718
慰勉 39
慰勞 62
慰勞品 3
慰唁 0
慰問 274
慰問信 1
慰問金 185
慰安婦 0
慰留 147
慰藉 66
慱 59
慲 136
慳 116
慳吝 5
慴 40
慴伏 0
慵 14
慵懶 49
慶 8048
慶元 57
慶典 238
慶典活動 35
慶功 31
慶功宴 50
慶功會 1
慶城 32
慶壽 10
慶幸 199
慶弔 5
慶生 176
慶生會 91
慶祝 1453
慶祝大會 113
慶祝晚會 4
慶祝會 16
慶親王 1
慶賀 130
慷 3
慷 37
慷他人之慨 3
慷慨 123
慷慨就義 2
慷慨激昂 18
慷慨解囊 31
慷慨陳詞 2
慸 0
慹 7
慺 50
慻 0
慼 1570
慼慼 2447
慽 0
慾 796
慾令智昏 0
慾壑難填 0
慾念 21
慾望 457
慾海 26
慾火 16
慾火焚身 7
慾障 0
慿 0
憀 37
憁 0
憂 663
憂傷 166
憂勞 1
憂勞成疾 0
憂國 11
憂國憂民 12
憂容 0
憂形於色 1
憂心 241
憂心如焚 6
憂心忡忡 64
憂思 4
憂急 7
憂恤 0
憂悒 1
憂患 23
憂患意識 16
憂悶 9
憂愁 129
憂慮 253
憂慮感 0
憂憤 2
憂懼 10
憂民 5
憂火如焚 0
憂疑 1
憂苦 5
憂鬱 377
憂鬱症 156
憃 9
憄 0
憅 0
憅 0
憆 0
憆 0
憇 0
憇 0
憈 0
憉 4
憊 129
憋 68
憋不住 6
憋悶 1
憋氣 7
憌 534
憍 1
憍 62
憍慢 1
憎 390
憎厭 5
憎嫉 0
憎嫌 0
憎恨 55
憎惡 32
憎惡 32
憏 0
憐 403
憐恤 20
憐惜 37
憐愛 22
憐憫 219
憐憫之心 4
憐才 1
憐香惜玉 4
憑 1651
憑什麼 8
憑什麼 8
憑仗 4
憑你 30
憑你是誰 0
憑你是誰 0
憑依 5
憑信 16
憑借 7
憑吊 2
憑單 97
憑單日記 0
憑單日記帳 0
憑弔 38
憑恃 15
憑據 60
憑欄 4
憑欄遠眺 3
憑甚麼 8
憑甚麼 8
憑甚麼 8
憑甚麼 8
憑眺 1
憑票 6
憑票入場 1
憑票入場 1
憑空 43
憑空出現 2
憑空想像 14
憑空臆造 0
憑良心 9
憑良心說 7
憑良心講 2
憑著 283
憑藉 149
憑藉著 67
憑虛御風 0
憑証 33
憑證 1030
憑證入場 1
憑證入場 1
憑靠 7
憑靠著 0
憒 85
憒亂 0
憓 28
憔 123
憔悴 75
憔悴不堪 1
憕 0
憕 0
憕 0
憖 48
憗 0
憘 0
憙 0
憚 149
憚改 0
憚煩 0
憛 353
憜 0
憝 1010
憞 0
憞 0
憟 98
憠 0
憡 32
憢 4
憣 0
憤 464
憤世 6
憤世嫉俗 11
憤怒 414
憤怨 1
憤恨 16
憤恨不平 4
憤愾 0
憤慨 87
憤憤 7
憤憤不平 12
憤懣 6
憤激 4
憤激語 0
憤然 6
憤然離去 0
憤發 1
憤而行兇 1
憤起 0
憥 0
憦 0
憧 1537
憧憧 1
憧憬 135
憨 227
憨傻 5
憨厚 19
憨子 2
憨實 0
憨態 0
憨癡 0
憨直 5
憨笑 3
憨頭憨腦 0
憩 1142
憩息 2
憪 17
憫 292
憫傷 0
憫恤 0
憫惜 0
憬 87
憬悟 5
憭 290
憮 195
憯 174
憰 6
憱 3
憲 4178
憲兵 141
憲兵隊 19
憲宗 24
憲改 0
憲政 551
憲政改革 208
憲政時期 4
憲法 2966
憲法大綱 0
憲章 85
憲綱 2
憳 75
憴 9
憵 51
憶 1514
憶及 16
憶念 8
憶測 3
憶舊 34
憶苦思甜 0
憶起 68
憷 0
憸 3295
憹 0
憹 0
憹 0
憺 8
憻 0
憼 22
憽 0
憾 183
憾事 21
憾動 9
憾恨 4
憿 1409
懀 0
懁 16
懂 2885
懂了 92
懂事 71
懂人意 0
懂得 1013
懃 22
懄 0
懄 0
懅 5
懆 5
懇 356
懇切 68
懇摯 1
懇求 113
懇親 1
懇親會 21
懇託 3
懇談 37
懇請 298
懇辭 3
懈 166
懈弛 0
懈怠 49
懈惰 0
懈慢 0
應 5377
應 14295
應付 957
應付得宜 0
應付得當 0
應付裕如 1
應允 81
應再 161
應到 57
應到人數 3
應制 30
應力 445
應加 163
應召 64
應召 64
應召女郎 15
應可 632
應否 234
應和 60
應在 837
應報 177
應報未報 0
應對 490
應對如流 1
應對得宜 1
應對得體 1
應對進退 21
應屆 682
應市 13
應帶 20
應帶未帶 0
應得 142
應徵 1066
應徵稿 0
應徵者 122
應急 70
應戰 46
應手 11
應承 8
應接 19
應接不暇 14
應援 184
應援者 0
應收 338
應敵 6
應時 39
應時當令 0
應景 75
應會 236
應有 2238
應有盡有 162
應機立斷 0
應用 21340
應用到 124
應用力學 132
應用在 557
應用性 36
應用技術 222
應用文 95
應用於 1018
應用科學 432
應用軟體 2367
應當 686
應當在 25
應當是 41
應當會 4
應當有 14
應當能 2
應答 63
應答如流 2
應約 10
應約而來 1
應約而到 0
應約而至 0
應考 607
應考生 3
應聘 150
應聲 100
應聲而倒 5
應聲蟲 3
應著 41
應計款項 0
應訊 0
應許 235
應試 248
應該 9893
應該在 270
應該是 2179
應該會 321
應該有 285
應該能 44
應諾 6
應變 797
應變措施 103
應變方法 1
應身佛 1
應運 26
應運而生 53
應選 401
應選人 3
應邀 531
應邀赴宴 1
應邀赴會 0
應酬 115
應酬信 0
應酬話 0
應酬飯 0
應門 22
應電流 0
應需 19
應須 6
應驗 109
懊 104
懊喪 4
懊喪起來 0
懊悔 53
懊悔不已 2
懊悔莫及 1
懊惱 76
懊惱不已 4
懋 1303
懋績 1
懌 7
懍 62
懍厲 0
懍懍 0
懍然 3
懎 0
懏 0
懐 0
懕 0
懕 0
懖 35
懗 0
懘 405
懙 0
懙 0
懚 0
懛 0
懜 0
懜 0
懝 0
懝 0
懞 6
懞懂 1
懟 35
懠 7
懡 0
懢 0
懢 0
懣 4
懤 61
懥 5
懦 96
懦夫 9
懦弱 80
懦怯 0
懧 2
懨 28
懨懨 1
懨懨縮縮 1
懨煎 1
懩 23
懪 25
懫 41
懬 0
懭 25
懮 50
懯 0
懰 26
懱 71
懲 135
懲一儆百 0
懲一警百 0
懲前毖後 0
懲惡勸善 0
懲戒 218
懲戒委員會 68
懲治 324
懲罰 407
懲處 162
懲辦 2
懳 0
懵 17
懵 17
懵懂 23
懵懵 2
懵懵懂懂 2
懵懵無知 0
懵懵然 0
懵然 3
懶 373
懶人 57
懶得 200
懶得 200
懶得動 6
懶惰 115
懶惰成性 2
懶惰蟲 5
懶惰鬼 2
懶懶 32
懶懶散散 1
懶散 37
懶於 3
懶洋洋 41
懶漢 0
懶覺 37
懶骨頭 3
懷 2918
懷中 107
懷了 32
懷仁 35
懷仁堂 0
懷俄明州 10
懷古 84
懷孕 1427
懷寧 81
懷寶 0
懷帝 0
懷德 45
懷德海 2
懷念 580
懷恨 10
懷恨在心 5
懷恩 52
懷恩堂 17
懷想 13
懷才不遇 20
懷抱 247
懷春 7
懷有 80
懷柔 9
懷柔政策 3
懷特海德 0
懷玉 15
懷王 10
懷璧 2
懷璧其罪 4
懷疑 1821
懷疑心 1
懷疑論 8
懷胎 35
懷舊 159
懷著 194
懷裡 88
懷遠 6
懷鄉 20
懷鄉病 0
懷錶 28
懸 407
懸吊 239
懸在空中 2
懸垂 25
懸壺 2
懸壺濟世 7
懸崖 57
懸崖勒馬 12
懸崖峭壁 15
懸巢鳥 0
懸挂 4
懸掛 245
懸掛國旗 5
懸掛在 27
懸掛著 9
懸案 32
懸樑 5
懸樑自盡 0
懸殊 80
懸浮 385
懸浮液 18
懸浮液 18
懸液計 0
懸液計 0
懸濁液 1
懸濁液 1
懸疑 115
懸空 47
懸而未決 10
懸肘 0
懸肘法 0
懸腕 0
懸腕 0
懸腕法 0
懸腸掛肚 0
懸臂 18
懸臂 18
懸臂樑 1
懸臂樑 1
懸臂橋 0
懸臂橋 0
懸賞 122
懸隔 2
懸隔兩地 0
懸隔多年 0
懸首 1
懹 4
懺 102
懺悔 158
懺悔錄 11
懻 3
懼 302
懼內 10
懼怕 127
懼於 11
懼色 2
懼高 0
懼高症 4
懽 18
懾 149
懾息 0
懾慴 0
懾服 6
懿 805
懿 805
懿事 0
懿德 41
懿旨 2
懿範 3
懿行 5
戀 3112
戀人 294
戀家 26
戀情 346
戀愛 1055
戀愛中 23
戀愛史 4
戀慕 55
戀戀不捨 9
戀曲 78
戀棧 34
戀歌 30
戀火 0
戀舊 1
戁 6
戂 0
戂 0
戃 4
戄 2
戅 0
戅 0
戇 8
戇 8
戈 337
戈壁 16
戈壁大沙漠 1
戈巴契夫 22
戈矛 2
戈蘭高地 0
戉 7
戊 731
戊戌政變 1
戊戌變法 6
戌 184
戍 84
戍卒 1
戍守 11
戍衛 2
戍邊 1
戎 418
戎事 0
戎服 0
戎機 2
戎狄 0
戎衣 1
戎裝 6
戎馬 6
戎馬倥傯 1
戎馬生涯 5
戏 0
戏 0
戏 0
成 14756
成河 10
成一家言 1
成丁 1
成丁禮 1
成三破二 1
成串 20
成也蕭何 1
成也蕭何敗也蕭何 1
成了 1609
成事 15
成事不說 1
成事不足 4
成事不足敗事有餘 1
成事在人 1
成事在天 2
成交 1097
成交總值 20
成交量 672
成交量總量 1
成交額 6
成人 2625
成人不自在 1
成人不自在自在不成人 1
成人之美 16
成人教育 1
成人版 3
成人班 9
成人病 1
成人電影 1
成仁 2624
成仁取義 0
成仇 5
成仙 1
成份 1154
成何體統 5
成佛 246
成例 6
成俗 1
成個兒 1
成倍 2
成倍增加 1
成倍增長 1
成像 160
成兆才 0
成兒 1
成兔 1
成全 210
成典 1
成分 906
成則為王 1
成則為王敗則為虜 1
成副兒 1
成功 9536
成功大學 3788
成功嶺 128
成功者 85
成功鎮 1
成務 1
成化 36
成化窯 1
成千 44
成千上萬 178
成千成百 2
成千成萬 5
成千累萬 0
成千論萬 1
成反比 25
成合 1
成吉思汗 108
成名 131
成名作 12
成名曲 29
成名術 1
成周 1
成命 1
成品 826
成員 4602
成員國 76
成唯識論 1
成問題 32
成單 37
成器 82
成因 336
成團打塊 1
成均 1
成型 1133
成堆 24
成大 2424
成天 68
成天到晚 0
成套 42
成套出售 0
成套設備 2
成妖作怪 1
成姻 1
成婚 20
成婚之日 0
成家 86
成家立業 15
成家立計 1
成實師 1
成實論 1
成對 73
成就 2089
成就感 207
成就測驗 1
成局 1
成屋 0
成岩作用 1
成年 312
成年人 1074
成年期 1
成年禮 1
成年累月 0
成康之治 1
成形 648
成德 1
成心 7
成房 1
成所作智 1
成批 6
成批出售 0
成批處理 1
成批處理 1
成招 1
成擒 3
成效 1620
成效卓著 55
成敗 320
成敗不計 0
成敗利鈍 1
成敗在此一舉 0
成敗在此一舉 0
成敗得失 1
成敗興廢 1
成敗論人 1
成數 166
成文 96
成文憲法 8
成文法 11
成方 7
成方兒 1
成日 25
成日家 1
成服 1
成本 4963
成本效益分析 1
成本會計 141
成本核算 0
成本計算 32
成材 9
成材林 1
成林 1
成果 7870
成果輝煌 34
成案 1
成棒 0
成樣 1
成樣 1
成歡 1
成正果 1
成殮 1
成氣候 1
成法 18
成活 6
成活率 8
成湯 1
成漢 1
成災 40
成為 19346
成為事實 23
成為泡影 2
成熟 1694
成熟 1694
成熟度 105
成熟度 105
成熟期 1
成熟期 1
成熟林 0
成熟林 0
成犬 1
成百上千 3
成皋 1
成真 164
成眠 1
成眾 4
成礦作用 1
成福 27
成禮 147
成窩兒 1
成窯 1
成立 20760
成章 35
成童 1
成竹在胸 4
成算 1
成箱 4
成篇 1
成精 1
成精作怪 1
成約 17
成總 1
成績 6503
成績單 736
成績平平 1
成績斐然 1
成群 154
成群作隊 1
成群打夥 1
成群結夥 1
成群結隊 38
成群結黨 1
成者為王 1
成者為王敗者為寇 1
成色 15
成色劑 1
成蔭 38
成藥 113
成蟲 103
成行 209
成衣 753
成衣店 9
成衣廠 22
成衣鋪 1
成見 65
成規 22
成親 27
成親之日 0
成言 1
成訟 1
成詞 1
成語 377
成語手冊 0
成誦 0
成說 57
成議 1
成蹊 1
成軍 0
成連 1
成道 1
成都 270
成都人 3
成都市 32
成都平原 6
成都王 0
成都賣卜 1
成長 11472
成長率 1820
成長股 1
成阻 1
成除服 1
成雙 63
成雙作對 0
成雙成對 18
成雙捉對 1
成音 1
成頭 1
成風 11
成體 19
成龍 353
成龍配套 1
我 160394
我上 212
我上來 13
我上去 6
我下 111
我下來 7
我下去 3
我不 3111
我不 3111
我不在 44
我不是 703
我不會 407
我不能 334
我也 4411
我也不 565
我也不 565
我也在 76
我也是 841
我也會 171
我也有 313
我也沒 125
我也能 37
我人 83
我使 20
我來 954
我來的 22
我倆 155
我倆 155
我倆的 36
我們 74537
我們 74537
我們的 11183
我像 129
我再 483
我到 416
我去 836
我又 987
我又不 30
我又不 30
我又不是 31
我取 19
我只 1515
我只有 231
我向 190
我君 2
我哥 45
我哥哥 31
我嗎 600
我回 193
我回來 51
我回到 77
我回去 36
我國 15282
我在 4178
我執 1
我姓 27
我媽 166
我媽媽 86
我家 1281
我家人 21
我射 1
我想 7742
我想來 7
我想到 145
我想去 105
我想在 196
我愛你 383
我所 1
我把 822
我放 51
我方 777
我是 9230
我會 2688
我會來 6
我會去 50
我有 6806
我沒 666
我沒有 732
我溜 1
我溜到 0
我爸 147
我爸爸 104
我留 117
我的 29607
我相 1
我看 1314
我看到 394
我知 146
我知道 1639
我等 221
我能 571
我能夠 79
我自己 1246
我行我素 36
我被 339
我要 11654
我要求 41
我見 1
我見猶憐 13
我設 12
我試 113
我認 89
我認為 1019
我軍 55
我輩 26
我過 40
我過來 3
我過去 48
我還 1070
我還不 57
我還不 57
我還在 86
我還是 943
我還會 42
我還有 258
我還沒 199
我還能 53
我量 5
我願 228
我願意 296
我黨 1
戒 1161
戒備 91
戒備森嚴 9
戒具 1
戒刀 10
戒勉 2
戒吃 1
戒嚴 90
戒嚴令 8
戒嚴區 0
戒嚴地區 0
戒嚴時期 83
戒嚴案 6
戒嚴法 7
戒壇 1
戒定慧 16
戒尺 4
戒律 100
戒心 43
戒急用忍 0
戒慎 31
戒懼 1
戒指 204
戒掉 23
戒條 8
戒毒 47
戒煙 84
戒煙法 0
戒禁取見 1
戒色 2
戒菸 102
戒護 26
戒賭 4
戒酒 23
戒除 51
戒驕戒躁 0
戓 0
戔 117
戔戔 1
戕 34
戕害 50
戕賊 0
戕賊 0
或 123208
或可 328
或在 661
或多或少 195
或大或小 6
或於 248
或早或晚 0
或是 8999
或有 921
或有或無 0
或然率 17
或然論 0
或者 3790
或者是 848
或與 389
或許 3896
或許是 566
或體 9
或體字 0
战 0
戙 77
戚 997
戚家軍 1
戚容 0
戚屬 0
戚戚 42
戚然 1
戚繼光 19
戚舊 0
戚誼 0
戚誼 0
戛 133
戛戛 2
戛然 7
戜 0
戝 0
戝 0
戞 0
戞 0
戟 102
戟狀 0
戠 3
戠 3
戡 25
戡亂 11
戡亂時期 9
戡定 1
戢 57
戣 4
戤 4
戥 8
戥子 1
戦 0
戧 68
戨 0
戩 114
截 1253
截住 12
截去 5
截取 101
截奪 0
截彎取直 68
截擊 19
截擊機 0
截斷 71
截止 1538
截止頻率 1
截然 46
截然不同 182
截獲 7
截球 1
截留 17
截稿 105
截肢 86
截至 637
截趾適履 0
截長補短 22
截開 0
截面 52
截面圖 0
截頭 2
戫 6
戬 0
戬 0
戭 31
戮 97
戮力 95
戮力同心 3
戯 0
戯 0
戯 0
戰 42574
戰亂 87
戰亂中 8
戰了 16
戰事 167
戰例 3
戰俘 41
戰俘營 0
戰個 1
戰備 92
戰備品 0
戰刀 7
戰利品 29
戰到 60
戰前 77
戰前戰後 1
戰力 285
戰功 27
戰勝 170
戰勝國 5
戰區 122
戰友 72
戰史 122
戰國 257
戰國七雄 6
戰國時代 72
戰國策 14
戰地 85
戰地司令部 0
戰地英雄 0
戰地記者 3
戰地鐘聲 2
戰報 190
戰場 61
戰場 619
戰場老將 0
戰場老將 0
戰壕 8
戰士 1542
戰天鬥地 0
戰將 176
戰局 75
戰役 304
戰後 652
戰得 8
戰情 104
戰情報告 20
戰情報導 1
戰慄 56
戰戟 1
戰戰兢兢 79
戰技 32
戰抖 2
戰敗 100
戰敗國 10
戰斧 30
戰旗 11
戰時 134
戰書 18
戰服 0
戰果 68
戰果輝煌 5
戰機 577
戰歌 5
戰死 37
戰死沙場 9
戰死沙場 9
戰況 190
戰況危急 1
戰法 38
戰火 175
戰無不克 1
戰無不勝 9
戰無不勝 9
戰爭 1706
戰爭國 0
戰爭片 10
戰爭狀態 6
戰爭與和平 20
戰爭險 0
戰犯 11
戰略 1431
戰略性 39
戰略物資 3
戰略要地 2
戰略轟炸 0
戰略轟炸機 3
戰略部署 1
戰神 766
戰禍 12
戰線 65
戰績 700
戰績輝煌 1
戰船 74
戰艦 168
戰術 366
戰術轟炸 0
戰術轟炸機 0
戰衣 5
戰袍 10
戰訊 0
戰費 2
戰贏 2
戰車 188
戰輸 3
戰陣 8
戰雲 9
戰雲密佈 9
戰馬 19
戰鬥 888
戰鬥人員 1
戰鬥任務 2
戰鬥力 33
戰鬥區 0
戰鬥員 7
戰鬥意志 11
戰鬥機 209
戰鬥編組 0
戰鬥英雄 1
戰鬥行為 1
戰鬥轟炸 0
戰鬥轟炸機 7
戰鬥隊形 1
戰鼓 25
戱 0
戱 0
戱 0
戲 10
戲 4012
戲中 97
戲中人 1
戲中戲 15
戲份 45
戲劇 2363
戲劇化 54
戲劇團 28
戲劇實驗 0
戲劇實驗學校 0
戲劇性 111
戲劇界 9
戲劇系 85
戲務 4
戲台 50
戲單 3
戲園 5
戲團 366
戲子 14
戲弄 44
戲弄著 1
戲文 25
戲曲 948
戲服 25
戲本 3
戲棚 4
戲水 123
戲法 35
戲法人人會變 4
戲班 52
戲班子 3
戲碼 144
戲票 9
戲箱 1
戲綵娛親 0
戲考 0
戲耍 11
戲臺 16
戲袍 0
戲裝 1
戲言 12
戲評 4
戲談 1
戲謔 26
戲路 28
戲迷 215
戲院 2373
戳 205
戳印 3
戳子 0
戳戳 3
戳戳看 0
戳破 18
戳穿 6
戳記 41
戴 3707
戴上 176
戴上來 0
戴上去 0
戴下 0
戴下來 0
戴下去 0
戴了 25
戴來 4
戴來戴去 0
戴個 3
戴出 4
戴出來 1
戴出去 0
戴去 2
戴反 3
戴在 39
戴壞 1
戴夠 0
戴奧辛 159
戴好 19
戴孝 2
戴安娜 110
戴得 35
戴月披星 0
戴有 11
戴法 1
戴漪 0
戴牢 0
戴的 35
戴盆望天 0
戴維 93
戴罪立功 1
戴著 156
戴走 0
戴起 8
戴起來 8
戴進 3
戴進來 0
戴進去 0
戴過 3
戴過來 0
戴過去 0
戴錶 2
戴震 10
戴頂 2
戴頂帽子 0
戴高帽 1
戴高樂 28
戴齊 1
戵 0
戶 5072
戶下 6
戶主 9
戶內 46
戶內運動 0
戶口 154
戶口名簿 269
戶口普查 15
戶口調查 11
戶口謄本 15
戶名 578
戶外 1341
戶外活動 195
戶外運動 50
戶庭 2
戶戶 35
戶政 425
戶政事務 10
戶政事務所 678
戶樞不蠹 0
戶籍 960
戶籍謄本 279
戶調 3
戶部 12
戶部司 0
戶長 47
戶限 1
戶頭 79
户 0
戸 0
戹 0
戹 0
戺 33
戻 0
戼 0
戽 104
戽斗 16
戽水 2
戾 65
戾氣 9
房 1
房 5501
房主 12
房事 90
房價 474
房內 0
房地 436
房地價 2
房地產 1717
房契 1
房子 1533
房客 130
房室 49
房屋 5817
房屋修護 0
房屋修護費 1
房屋內 6
房屋裡 7
房山 4
房市 0
房捐 0
房東 401
房東太太 5
房柁 0
房檐 5
房產 58
房租 140
房簷 1
房脊 0
房舍 196
房號 24
房裡 107
房角 4
房貸 0
房車 344
房錢 5
房門 125
房門口 4
房間 1828
房頂 12
所 100492
所以 27522
所以在 950
所以是 116
所以會 117
所以有 236
所以然 39
所以能 82
所作 745
所作所為 48
所做所為 21
所傷 31
所剩無幾 25
所及 546
所向 12
所向披靡 15
所向無敵 16
所在 2567
所在地 1133
所好 28
所屬 8919
所幸 317
所得 3238
所得稅 712
所得稅率 9
所思 38
所有 25652
所有人 938
所有制 16
所有格 3
所有權 1657
所有權狀 134
所有者 223
所為 579
所生 265
所由 39
所知 460
所羅門 335
所羅門王 27
所聞 33
所聞所見 0
所能 1046
所能夠 21
所至 31
所致 0
所要 1343
所見 520
所見即所得 0
所見所聞 38
所見略同 3
所謂 3271
所謂的 1891
所費不貲 0
所述 431
所部 4
所長 1157
所長 1157
所需 5542
扁 905
扁 9053
扁圓 5
扁圓形 3
扁帽 109
扁平 151
扁平腳 0
扁平足 22
扁擔 28
扁柏 39
扁柏 39
扁桃腺 93
扁桃腺炎 48
扁舟 10
扁豆 16
扁銼 0
扁鑽 4
扁頭 4
扁額 9
扁食 39
扁骨 0
扁鵲 9
扂 2758
扃 66
扅 0
扆 252
扇 802
扇 802
扇動 3
扇子 47
扇子 47
扇子舞 1
扇子舞 1
扇形 50
扇惑 0
扇扇子 0
扇扇子 0
扇火 0
扇爐子 0
扇爐子 0
扇狀 15
扇舞 2
扇貝 1
扇起 0
扇起來 0
扇開 3
扇風 6
扇風機 0
扇骨 2
扈 168
扈從 3
扉 43
扉頁 43
扊 0
手 13457
手上 690
手下 311
手下敗將 2
手下無情 0
手下留情 16
手不釋卷 3
手中 1161
手令 4
手倒立 1
手倒立 1
手兒 7
手兒 7
手冊 6370
手刃 17
手到 14
手到擒來 5
手到病除 0
手制動 0
手刻 1
手刻工 0
手則 35
手力 3
手勁 2
手動 458
手動式 18
手勢 425
手包著 0
手印 96
手卷 15
手壓 16
手壓式 1
手夾 1
手夾著 0
手套 570
手寫 438
手工 454
手工業 111
手工業者 2
手工藝 349
手工藝品 174
手工費 0
手巧 2
手巾 23
手帕 114
手彈 1
手彈式 0
手心 60
手忙腳亂 44
手急眼快 1
手感 127
手扒雞 7
手扶拖拉 0
手扶拖拉機 0
手抄 25
手抄本 17
手把 133
手抱著 1
手拉 121
手拉式 5
手拉手 3
手拙 4
手拿 83
手拿著 14
手持 259
手持著 8
手指 878
手指甲 7
手指縫 1
手指頭 86
手指頭 86
手挽手 0
手捲 1
手掌 327
手掌心 17
手推式 6
手推車 42
手提 192
手提包 70
手提式 65
手提箱 39
手提袋 120
手握 91
手握手 1
手握著 9
手揮目送 0
手搖 24
手搖式 7
手摺 2
手旗 0
手書 21
手本 1
手札 443
手杖 20
手板 12
手枷 0
手柄 21
手植 5
手榴彈 49
手槍 159
手槍套 0
手模 4
手機 10
手段 990
手民 1
手氣 48
手法 938
手淫 38
手無寸鐵 11
手無縛雞之力 3
手煞車 21
手煞車 21
手牽手 45
手牽著 13
手球 102
手球員 0
手環 26
手癢 23
手皮包 0
手相 36
手示 4
手稿 118
手筆 44
手簡 0
手紋 2
手紙 8
手絹 11
手緊 12
手續 1653
手續費 499
手續費收入 10
手肉 0
手肘 85
手背 43
手腕 217
手腳 269
手腳俐落 3
手腳無措 0
手膀 0
手臂 510
手臂 510
手舞足蹈 27
手藝 170
手藝業 0
手術 3358
手術刀 38
手術台 12
手術室 50
手術房 23
手表 20
手裡 141
手訂 1
手語 343
手談 0
手諭 1
手足 200
手足之情 15
手足失措 6
手足情深 7
手足無措 40
手足重繭 0
手跡 7
手蹟 0
手軟 40
手輕 21
手輪 6
手邊 258
手部 177
手重 5
手銬 27
手錶 426
手錶帶 0
手鎗 1
手鐲 38
手雷 1
手電 7
手電筒 150
手頭 48
手頭上 18
手風 6
手風琴 18
手骨 1
手鬆 1
扌 0
才 16087
才上 46
才上來 4
才上到 1
才上去 0
才下 20
才下來 0
才下去 2
才不 398
才不是 33
才不會 397
才不能 100
才付 1
才來 134
才來到 15
才俊 14
才信 20
才像 130
才像是 13
才具 39
才再 36
才出 139
才出來 7
才出去 0
才分 17
才分 17
才到 76
才力 7
才去 88
才可以 1261
才名 6
才向 25
才器 0
才回 77
才回來 20
才回到 13
才回去 20
才在 210
才多 13
才多出 1
才夠 42
才夠來 0
才夠去 0
才女 60
才好 296
才媛 0
才子 92
才子佳人 20
才學 42
才定 8
才將 104
才對 393
才帶 20
才幹 25
才得 228
才得到 22
才德 12
才德兼備 5
才思 3
才思敏捷 2
才怪 100
才情 33
才想 109
才打 160
才打出 4
才打到 3
才拿 13
才拿出 10
才拿到 60
才排 6
才敢 53
才是 2793
才智 54
才會 2393
才會到 13
才會在 66
才會有 284
才有 1802
才望 2
才氣 42
才氣縱橫 3
才沒 55
才沒有 33
才沒能 10
才派 5
才派人 0
才為 117
才用 26
才用到 10
才略 8
才略過人 0
才疏學淺 40
才看 207
才看出 7
才看到 73
才等 37
才等到 1
才算 183
才算是 62
才給 25
才能 8384
才能夠 291
才能有 130
才華 219
才華橫溢 7
才華洋溢 39
才華蓋世 1
才藝 458
才藝卓絕 0
才藝班 56
才行 333
才要 148
才訂 2
才講 7
才識 9
才讀 4
才讀到 0
才貌 7
才貌出眾 10
才貌雙全 10
才買 48
才資 40
才起來 3
才跟 28
才過 13
才過來 2
才過去 3
才開 40
才開出 1
才開到 0
才附 0
才降 1
才高八斗 6
扎 26
扎 267
扎 267
扎伊爾 3
扎到 18
扎實 36
扎手 2
扎扎 7
扎扎實實 1
扎掙 1
扎根 95
扎眼 0
扎針 8
扏 0
扐 4
扑 323
扑打 0
扒 158
扒 158
扒出 1
扒土 0
扒山虎 0
扒手 17
扒灰 0
扒癢 0
扒竊 20
扒起 2
扒開 2
打 1
打 823
打 8238
打上 83
打上來 9
打上去 1
打下 131
打下來 10
打下去 20
打不倒 7
打不開 30
打中 77
打主意 0
打亂 39
打井 1
打交道 73
打亮 9
打人 79
打仗 44
打住 10
打來 80
打來打去 4
打信號 0
打個 228
打倒 206
打傘 7
打傷 36
打先鋒 5
打光 16
打光棍 0
打光機 1
打入 334
打入冷宮 10
打冷嗝 0
打冷戰 1
打出 378
打出來 19
打出去 50
打分 9
打分數 30
打到 246
打前鋒 17
打劫 27
打動 116
打勝 1
打勝仗 10
打勾 10
打勾勾 1
打包 125
打包票 4
打卡 34
打卡機 6
打卡鐘 36
打印 38
打印機 4
打去 14
打呵欠 4
打呼 30
打呼聲 2
打哆嗦 4
打哈哈 2
打哈欠 15
打嗝 49
打嗝兒 0
打嗝兒 0
打嘴 8
打噴嚏 55
打回 28
打回來 2
打回去 4
打圈 3
打圈子 0
打圓場 7
打圓場 7
打在 112
打地鋪 2
打坐 96
打垮 12
打基礎 4
打場 4
打場 4
打壓 264
打壞 12
打天下 24
打好 118
打孔 52
打孔器 4
打孔機 58
打字 762
打字員 13
打字機 120
打字行 99
打完 147
打官司 20
打官腔 5
打定 10
打定主意 13
打家劫舍 5
打對 17
打小報告 14
打尖 4
打岔 4
打工 332
打差 1
打差差 0
打帶跑 4
打底 44
打彈子 4
打彈珠 3
打彎 0
打得 223
打得 315
打從 137
打悶棍 0
打悶棍 0
打情罵俏 12
打慣 0
打成 91
打成一片 36
打成一片 36
打成平手 12
打戰 10
打手 53
打手勢 6
打手式 1
打手槍 13
打打 98
打打氣 11
打打鬧鬧 4
打扮 234
打扮起來 2
打折 99
打折扣 73
打抱不平 23
打拍 1
打拍子 6
打拚 85
打招呼 156
打拱 0
打拱作揖 0
打拳 4
打拳擊 0
打拼 197
打持著 0
打挺兒 0
打挺兒 0
打掃 170
打掉 22
打探 25
打撈 59
打撈到 0
打擂台 6
打擂台 6
打擂臺 5
打擊 1290
打擊到 6
打擊報復 0
打擊手 6
打擊板 0
打擊樂器 55
打擊率 59
打擾 159
打擾到 12
打攪 10
打攪到 0
打敗 446
打敗仗 3
打散 50
打斷 148
打昏 14
打暗號 10
打更 11
打更 11
打杈 0
打板子 0
打架 175
打柴 4
打栓 0
打棍子 0
打槍 10
打樁 26
打樣 67
打歌 57
打歪 3
打死 214
打毛線 2
打毛衣 0
打氣 353
打氣筒 19
打水 17
打水仗 8
打沉 1
打油 5
打油詩 27
打法 95
打消 82
打消念頭 5
打消念頭 5
打混 47
打游擊 5
打溼 1
打滑 49
打滾 35
打滾兒 0
打滾兒 0
打漿 3
打漿機 1
打火 21
打火機 239
打烊 85
打燈謎 0
打爛 8
打牌 57
打牙祭 3
打獵 100
打球 420
打理 35
打疊 0
打發 174
打發掉 1
打的 315
打直 9
打盹 19
打盹兒 0
打盹兒 0
打眼 1
打瞌睡 47
打瞎 0
打砲 5
打破 830
打破沙鍋問到底 1
打破紀錄 4
打砸搶 0
打硬仗 3
打碎 41
打磨 16
打磨盤 1
打秋風 1
打稿 5
打穀 2
打穿 9
打箇照面 0
打算 1333
打算盤 2
打籃球 153
打結 73
打網球 29
打緊 11
打緯 3
打罵 20
打罵教育 0
打群架 18
打翻 45
打翻掉 0
打者 87
打聽 198
打聽出 2
打聽到 7
打胎 1
打腫 4
打腫臉充胖子 2
打舌頭 0
打舌頭 0
打草稿 4
打草驚蛇 9
打菜 7
打落 24
打落水狗 2
打著 134
打蒼蠅 1
打蛋 8
打蛋器 14
打蠟 18
打蠟機 0
打裂 0
打賭 45
打贏 89
打赤腳 6
打赤膊 5
打起 101
打起精神 39
打趣 13
打躬 2
打躬作揖 3
打輸 14
打轉 62
打退 14
打退堂鼓 24
打通 105
打通兒 0
打通兒 0
打通關 2
打造 537
打造出 53
打造成 53
打進 96
打進來 2
打進去 3
打過 92
打過來 9
打過去 8
打道 7
打道回府 18
打邊鼓 0
打酒 2
打醒 9
打野外 1
打量 53
打量出 0
打針 108
打鈴 5
打錯 176
打鐘 3
打鐵 63
打鐵工 0
打鐵趁熱 4
打門 10
打開 2044
打開天窗 2
打開天窗說亮話 2
打關照 0
打關節 0
打雜 20
打雪仗 0
打雷 31
打電報 6
打電話 1207
打靶 26
打靶場 0
打靶場 0
打鞦韆 0
打響 49
打頂 1
打領帶 10
打頭 14
打頭陣 33
打顫 12
打食 12
打飯 10
打馬虎眼 4
打馬虎眼 4
打高爾夫 18
打鬥 77
打鬥片 0
打鬧 6
打鬼 16
打魚 59
打魚郎 0
打鴨子上架 2
打鹽水 0
打麥 2
打麻將 52
打點 131
打點滴 15
打鼓 30
打鼾 55
扔 332
扔 332
扔上 0
扔上來 0
扔上去 0
扔下 7
扔下來 0
扔下去 1
扔來 0
扔來扔去 1
扔出 5
扔出來 0
扔出去 0
扔到 20
扔去 1
扔回 0
扔回來 0
扔回去 0
扔在 19
扔在腦後 0
扔掉 14
扔棄 0
扔給 7
扔起 1
扔起來 0
扗 0
托 1606
托了 5
托人 7
托住 24
托兒 88
托兒所 801
托勒 2
托勒密 15
托勒爾 2
托名 0
托夢 4
托拉斯 157
托故 0
托斯卡尼尼 53
托架 5
托洛茨基 0
托爾斯泰 10
托病 2
托盤 48
托福 295
托福考 2
托福考試 22
托管 10
托管國 0
托維 1
托缽 15
托缽僧 0
托著 18
托詞 5
托起 7
托起來 1
托辣斯 29
托辭 0
托運 79
托過 4
托馬 3
托馬斯 22
扙 12
扚 9
扛 171
扛 171
扛上 3
扛上來 0
扛上去 0
扛下 13
扛下來 2
扛下去 0
扛了 3
扛來 0
扛來扛去 0
扛出 2
扛出來 0
扛出去 0
扛去 4
扛在 15
扛扛抬抬 0
扛著 38
扛走 0
扛起 22
扛起來 3
扛進 1
扛進來 0
扛進去 0
扛過 0
扛過來 1
扛過去 1
扜 98
扝 0
扞 74
扞拒 0
扞格 13
扞禦 0
扟 0
扠 38
扡 33
扢 223
扣 1679
扣上 22
扣上來 0
扣上去 0
扣下 12
扣下來 0
扣下去 0
扣了 15
扣人心弦 28
扣住 32
扣來扣去 0
扣光 3
扣兒 0
扣兒 0
扣出 2
扣分 36
扣分標準 2
扣分法 0
扣到 5
扣去 1
扣問 0
扣回 13
扣回來 0
扣回去 0
扣在 12
扣壓 3
扣好 3
扣帽子 3
扣得 3
扣得 3
扣應 35
扣抵 341
扣押 201
扣掉 35
扣板機 0
扣法 2
扣減 37
扣減額 1
扣滿 0
扣牢 0
扣球 3
扣留 80
扣留住 0
扣籃 4
扣緊 27
扣繳 551
扣繳憑單 99
扣繳額 1
扣著 4
扣薪 5
扣起 4
扣起來 1
扣過 8
扣過來 0
扣過去 0
扣錢 1
扣門 8
扣除 572
扣除額 171
扣點 22
扣點分數 0
扤 12
扥 112
扦 288
扦插 34
执 0
扭 233
扭住 1
扭來 1
扭來扭去 3
扭傷 123
扭出 0
扭出來 0
扭出去 0
扭到 10
扭力 514
扭動 69
扭在 2
扭怩 1
扭成 1
扭成一團 1
扭成一團 1
扭打 5
扭扭 7
扭扭捏捏 5
扭扭舞 0
扭捏 6
扭捏作態 4
扭擺 3
扭擺不停 0
扭斷 4
扭曲 496
扭矩 19
扭秧歌 0
扭筋 1
扭結 11
扭緊 4
扭腰 10
扭著 6
扭虧增盈 1
扭起 0
扭起來 0
扭身 1
扭轉 280
扭轉乾坤 11
扭轉局勢 4
扭轉身 1
扭轉頹勢 4
扭送 1
扭過 3
扭過來 1
扭過去 0
扭過頭 2
扭過頭 2
扭過頭來 1
扭過頭來 1
扭過頭去 0
扭過頭去 0
扭開 5
扭頭 14
扮 603
扮作 14
扮做 0
扮像 1
扮得 3
扮得 23
扮成 37
扮演 2215
扮演著 205
扮的 23
扮相 21
扮著 7
扮裝 7
扮起 4
扮起來 1
扮鬼 5
扮鬼臉 5
扯 330
扯上 44
扯上來 0
扯上去 0
扯下 16
扯下來 1
扯下去 5
扯住 6
扯倒 0
扯出 24
扯出來 0
扯出去 0
扯到 37
扯去 0
扯嗓子 0
扯對 0
扯平 2
扯後腿 12
扯成 0
扯扯 7
扯拉 0
扯掉 5
扯旗 0
扯淡 1
扯白 0
扯直 1
扯破 8
扯碎 1
扯篷 0
扯裂 3
扯談 0
扯謊 3
扯走 1
扯起 2
扯起來 0
扯進 6
扯進來 1
扯進去 2
扯鈴 63
扯開 11
扰 36
扱 37
扲 199
扳 13
扳 134
扳不倒 1
扳倒 33
扳子 0
扳子 0
扳成 1
扳手 50
扳機 19
扳機 19
扳起 5
扳起來 0
扳轉 2
扳道 0
扳開 7
扴 57
扶 1092
扶下 0
扶下來 0
扶下去 0
扶不起 2
扶不起的阿斗 14
扶乩 4
扶了 2
扶人 0
扶住 17
扶來 0
扶出 4
扶出來 0
扶出去 0
扶助 354
扶危 6
扶危定傾 0
扶危定傾 0
扶危濟困 1
扶去 0
扶在 4
扶好 3
扶弱抑強 0
扶手 158
扶手椅 3
扶持 239
扶持著 3
扶搖 5
扶搖直上 26
扶木 0
扶杖 0
扶柩 1
扶桑 48
扶梯 60
扶棺 0
扶椅 3
扶植 100
扶欄 1
扶正 38
扶為 0
扶犁 0
扶疏 12
扶病 2
扶直 3
扶翼 0
扶老攜幼 24
扶著 76
扶蘇 10
扶起 31
扶起來 4
扶輪社 287
扶過 0
扶靈 0
扶養 256
扷 28
扸 0
批 2207
批上 6
批上來 0
批上去 0
批下 1
批下來 0
批下去 0
批假 1
批准 385
批准下來 0
批准書 2
批出 7
批出來 0
批出去 0
批判 737
批判地 3
批判性 47
批回 1
批在 6
批好 1
批字 0
批得 2
批得 42
批復 8
批成 5
批改 103
批文 13
批次 444
批次檔 53
批注 1
批發 1712
批發價 97
批發商 118
批發店 2
批的 42
批示 41
批示著 0
批答 0
批紅戴綠 0
批給 2
批著 0
批號 139
批複 0
批覆 5
批註 11
批評 4084
批評家 46
批評性 3
批評者 26
批語 13
批貨 76
批貨商 0
批貨店 0
批購 12
批起 0
批起來 0
批量 30
批量生產 2
批閱 21
批離 0
批駁 14
批鬥 16
批點 5
扺 65
扺 65
扻 16
扼 57
扼住 6
扼守 8
扼止 12
扼死 0
扼殺 81
扼流圈 7
扼腕 22
扼腕 22
扼要 62
扼要說明 17
扼阻 6
扽 66
找 10361
找上 95
找上來 1
找上去 0
找上門來 7
找不出 102
找不到 2167
找不著 34
找了 350
找事 9
找些 66
找人 252
找來 190
找來找去 3
找個 625
找出 1573
找出來 93
找出去 1
找出路 7
找到 5493
找回 433
找回來 58
找回去 0
找完 4
找尋 1521
找尋出 7
找尋到 25
找對頭 0
找對頭 0
找工作 214
找平 4
找得 190
找得 393
找找看 0
找死 15
找牙 4
找的 393
找碴 36
找空 4
找著 38
找起 20
找起來 2
找進 3
找進來 0
找進去 0
找遍 28
找遍 28
找過 42
找過來 0
找過去 1
找錢 38
找門路 0
找開 2
找麻煩 23
找齊 10
承 4239
承以上 100
承上啟下 2
承上題 2
承保 229
承保人 2
承先啟後 55
承包 555
承包人 12
承包商 369
承印 22
承受 1095
承塵 0
承天 60
承平 5
承平期 0
承平面 0
承德 71
承德路 470
承情 0
承戴力 0
承接 436
承擔 556
承擔責任 22
承擔費用 0
承擔重任 1
承擔風險 14
承攬 741
承歡 9
承歡膝下 3
承當 13
承租 630
承繼 78
承蒙 208
承蒙指教 0
承製 133
承襲 114
承襲著 1
承認 1820
承認在 9
承認是 28
承認會 0
承認錯誤 13
承諾 1506
承載 176
承載容量 12
承載量 15
承辦 2261
承辦人 274
承辦者 35
承造 90
承造商 1
承運 48
承運人 4
承運公司 0
承重 29
承銷 593
承銷人 34
承領 87
承頭 1
技 10592
技倆 34
技士 233
技安 1
技客 1
技專 134
技工 243
技巧 4143
技巧性 17
技巧運動 0
技師 1099
技擊 73
技正 151
技法 234
技癢 4
技窮 4
技職 0
技能 2284
技能科 1
技藝 1277
技術 72458
技術人員 1028
技術合作 501
技術員 410
技術單位 15
技術學院 4187
技術專科 4
技術專科學校 138
技術性 300
技術指導 198
技術水平 10
技術水準 114
技術裝備 5
技術部門 28
技術革命 2
技術革新 17
抁 32
抂 0
抃 27
抄 529
抄上 0
抄上來 0
抄上去 0
抄下 31
抄下來 11
抄下去 3
抄來抄去 0
抄在 9
抄家 12
抄寫 73
抄寫員 0
抄小路 4
抄後路 0
抄截 0
抄手 16
抄抄 8
抄書 5
抄有 1
抄本 61
抄沒 2
抄獲 1
抄著 3
抄襲 200
抄襲舊作 0
抄起 6
抄起來 2
抄近路 2
抄過 2
抄過來 1
抄過去 0
抄錄 83
抄錄下來 8
抄錄在 1
抄錄編目 14
抄集 0
抅 0
抅 0
抆 5
抆拭 0
抆淚 0
抇 52
抈 20
抉 43
抉擇 675
把 1
把 25773
把不到 1
把了 8
把住 7
把個 45
把兄弟 1
把兒 10
把兒 10
把勢 0
把子 11
把守 24
把家 38
把式 7
把戲 78
把戲兒 0
把戲兒 0
把手 273
把持 47
把持住 4
把持著 1
把握 1016
把握住 37
把握著 2
把晤 0
把柄 29
把玩 57
把玩 57
把盞 10
把穩 1
把舵 1
把著 3
把起 4
把這兩 26
把酒 29
把酒言歡 8
把門 57
把關 228
把集 2
把頭 90
把頭 90
把風 14
抋 0
抌 47
抍 0
抎 259
抏 15
抐 0
抑 296
抑低 17
抑制 1108
抑制作用 34
抑制劑 168
抑制器 3
抑強扶弱 1
抑或 280
抑抑揚格 0
抑揚 7
抑揚頓挫 18
抑是 1
抑止 51
抑鬱 90
抑鬱寡歡 0
抒 104
抒情 242
抒情文 6
抒情曲 6
抒情歌 9
抒情歌曲 14
抒情片 0
抒情詩 15
抒意 0
抒懷 6
抒發 159
抒解 31
抒難 0
抓 2
抓 2238
抓上 3
抓上來 1
抓上去 0
抓下 76
抓下來 39
抓下去 4
抓人 28
抓住 492
抓住機會 13
抓來 37
抓來抓去 3
抓出 51
抓出來 19
抓出去 1
抓到 285
抓去 48
抓取 375
抓大頭 0
抓姦 2
抓岩機 0
抓巖機 0
抓拿 6
抓拿犯人 0
抓撓 0
抓斗 9
抓牢 5
抓狂 0
抓癢 15
抓瞎 0
抓破 6
抓綱治國 0
抓緊 30
抓耳撓腮 1
抓舉 0
抓著 113
抓著不放 2
抓藥 4
抓賊 27
抓賭 1
抓走 19
抓起 69
抓起來 34
抓辮子 0
抓進 3
抓進來 2
抓進去 1
抓革命促生產 0
抓鬥 0
抓點 17
抔 6
投 9090
投上 1
投上來 0
投上去 0
投下 222
投下來 0
投下去 2
投不出 2
投不到 0
投不準 0
投中 25
投了 30
投井 1
投井自盡 1
投來 12
投來投去 1
投保 1718
投信 0
投入 3170
投入資本 5
投其所好 10
投出 67
投出來 0
投出去 2
投函 0
投到 14
投去 6
投合 5
投向 68
投回 3
投在 20
投報 1
投契 5
投奔 27
投奔自由 6
投宿 60
投宿在 11
投寄 34
投射 284
投幣 57
投幣口 1
投幣孔 0
投彈 7
投彈手 0
投影 12017
投影圖 6
投影機 911
投影片 0
投影畫 0
投影面 0
投得 5
投得 47
投懷 2
投懷送抱 4
投手 682
投擲 76
投放 79
投效 16
投敵 0
投書 198
投桃報李 6
投案 60
投案自首 0
投梭 4
投棄 1
投標 1499
投機 407
投機事業 2
投機份子 1
投機倒把 3
投機分子 1
投機取巧 475
投水 4
投河 5
投河自盡 3
投法 6
投注 285
投海 5
投球 94
投環自盡 0
投生 15
投產 133
投的 47
投石 30
投石問路 25
投石機 3
投票 2484
投票人 26
投票所 57
投票數 63
投票權 76
投票箱 11
投票表決 26
投稿 782
投稿人 5
投稿箱 0
投竿 1
投筆從戎 8
投籃 42
投籃不中 0
投籃得分 0
投緣 23
投考 45
投胎 38
投藥 58
投親 0
投訴 216
投誠 14
投資 25859
投資人 2171
投資報酬率 1
投資收益 61
投走 2
投起 2
投起來 0
投足 9
投身 139
投軍 5
投進 32
投進來 0
投進去 0
投過 6
投過來 1
投過去 0
投遞 197
投錢 7
投錢口 0
投錢箱 0
投閒置散 0
投降 194
投降主義 4
投降書 0
投靠 41
投靠在 1
投鞭斷流 0
投鼠忌器 7
抖 184
抖上 0
抖下 0
抖下來 0
抖下去 0
抖亂 1
抖了 18
抖了起來 0
抖出 24
抖出來 1
抖出去 0
抖動 114
抖動著 7
抖抖 5
抖抖簌簌 0
抖掉 0
抖摟 0
抖擻 10
抖翻 0
抖落 12
抖起 1
抖起來 1
抖開 4
抗 2730
抗了 0
抗力 20
抗力臂 2
抗力點 3
抗原 297
抗告 74
抗命 4
抗壓 28
抗壓強度 38
抗壓性 11
抗壞血酸 14
抗大 1
抗寒 3
抗寒性 0
抗張強度 17
抗御 4
抗性 32
抗戰 222
抗戰時期 23
抗拉強度 16
抗拒 375
抗拒性 3
抗擊 1
抗敵 5
抗日 136
抗日戰爭 29
抗旱 6
抗旱性 0
抗暴 24
抗毒 6
抗毒素 11
抗毒血清 10
抗氧化 1
抗氧化劑 1
抗洪 20
抗災 3
抗熱 7
抗熱性 2
抗爭 552
抗生素 483
抗病 66
抗病品種 2
抗病性 4
抗癌 234
抗癌 234
抗癌性 0
抗癌性 0
抗癌藥 5
抗癌藥 5
抗禦 11
抗稅 0
抗組織胺 1
抗老 40
抗老性 0
抗菌 299
抗菌素 5
抗著 6
抗藥 24
抗藥性 144
抗衡 112
抗衡作用 0
抗訴 5
抗議 1913
抗議性 1
抗議書 14
抗走 0
抗跌 62
抗跌性 25
抗軍 0
抗辯 64
抗逆性 0
抗酸 6
抗酸性 1
抗震 14
抗震力 0
抗體 646
抗鹼 2
抗鹼性 1
折 1
折 1
折 9284
折上 0
折上來 0
折上去 0
折下 3
折下來 0
折下去 0
折不斷 1
折不斷 1
折中 5
折倒 0
折價 378
折價出售 0
折價券 0
折光 21
折兌 0
折兌率 0
折半 15
折半出售 0
折合 112
折回 27
折回來 3
折回去 1
折壽 9
折實 1
折射 155
折射光 1
折射波 1
折射波 1
折射率 58
折射線 0
折射角 0
折彎 8
折得 1
折得 39
折成 19
折戟沈河 0
折扇 8
折扣 1997
折扣率 5
折抵 43
折損 33
折損率 2
折斷 81
折服 43
折本 2
折枝 9
折疊 103
折疊床 0
折疊式 28
折疊為 0
折的 39
折磨 221
折福 4
折算 49
折算法 0
折算率 0
折箭 4
折箭為盟 0
折紙 7
折線 43
折耗 4
折耗 4
折腰 8
折舊 200
折舊基金 0
折舊率 29
折舊費 7
折行 0
折衝 34
折衝尊俎 1
折衷 75
折衷方案 3
折衷法 0
折角 9
折讓 87
折賣 2
折賬 0
折起 64
折起來 4
折轉 3
折返 38
折進 0
折進來 0
折進去 0
折騰 48
抙 0
抚 0
抚 0
抛 0
抜 0
抝 0
抝 0
抝 0
抝 0
抝 0
択 0
択 0
报 0
抦 0
抦 0
抧 0
抨 42
抨擊 471
抩 157
抪 193
披 342
披 342
披上 56
披上來 0
披上去 0
披在 16
披得 0
披掛 10
披掛上陣 17
披搭 0
披星戴月 14
披沙揀金 0
披甲 4
披的 0
披紅 3
披索 0
披肝瀝膽 0
披肩 31
披荊執銳 0
披荊斬棘 31
披著 62
披蓋 1
披蓋在 1
披蓋著 0
披薩 139
披衣 56
披衣菌 1
披襟 1
披覆 28
披起 0
披起來 0
披錦 0
披開 0
披閱 1
披露 121
披露 121
披靡 2
披頭 32
披頭四 0
披頭散髮 6
披風 35
披髮 8
披麻帶孝 3
抬 607
抬上 5
抬上來 2
抬上去 0
抬下 21
抬下來 0
抬下去 0
抬不動 1
抬來 1
抬來抬去 0
抬價 0
抬出 21
抬出來 6
抬出去 1
抬去 1
抬回 1
抬回來 0
抬回去 1
抬得 5
抬得 16
抬愛 13
抬抬 2
抬抬看 0
抬槓 23
抬的 16
抬筐 0
抬肩 0
抬臉 0
抬舉 26
抬著 26
抬走 3
抬起 157
抬起來 10
抬起頭來 78
抬起頭兒 0
抬起頭兒 0
抬轎 51
抬進 4
抬進來 0
抬進去 1
抬過 2
抬過來 0
抬過去 0
抬頭 752
抬頭挺胸 17
抬高 75
抬高身價 1
抭 125
抮 35
抯 36
抰 24
抱 1123
抱上 5
抱上來 1
抱上去 0
抱下 0
抱下來 1
抱下去 2
抱不平 28
抱住 115
抱佛腳 8
抱出 5
抱出來 0
抱出去 0
抱去 5
抱在 46
抱存 0
抱守 1
抱定 8
抱屈 15
抱怨 734
抱恙 2
抱恨 1
抱恨終身 0
抱憾終生 1
抱成 0
抱成一團 0
抱成一團 0
抱抱 212
抱拳 0
抱持 226
抱持著 103
抱有 37
抱朴 0
抱朴子 15
抱枕 1
抱歉 1508
抱殘守缺 1
抱病 25
抱病上場 4
抱病上場 4
抱緊 29
抱腰 0
抱著 680
抱薪救火 1
抱負 176
抱走 36
抱起 55
抱起來 14
抱進 4
抱進來 0
抱進去 0
抱頭 10
抱頭痛哭 5
抱頭鼠竄 17
抱養 0
抲 0
抳 77
抴 76
抴 76
抵 2617
抵上 24
抵不住 2
抵不過 21
抵了 1
抵住 19
抵來抵去 0
抵借 0
抵債 5
抵償 16
抵充 24
抵到 2
抵制 226
抵去 0
抵向 1
抵命 0
抵得 4
抵得過 5
抵御 4
抵扣 28
抵抗 358
抵抗力 250
抵押 276
抵押借款 10
抵押品 24
抵押放款 1
抵押權 212
抵掉 7
抵換 5
抵擋 149
抵擋不住 28
抵敵 8
抵數 1
抵死 19
抵消 44
抵牾 0
抵禦 50
抵禦外侮 2
抵罪 0
抵著 6
抵補 10
抵觸 57
抵讓 0
抵賴 3
抵足而眠 0
抵過 1
抵達 1900
抵銷 60
抶 246
抷 0
抷 0
抸 64
抹 714
抹 714
抹一鼻子 0
抹一鼻子 0
抹一鼻子灰 0
抹一鼻子灰 0
抹上 37
抹上來 0
抹上去 0
抹下 0
抹下來 0
抹下去 0
抹乾 3
抹了 12
抹亮 0
抹來抹去 0
抹到 3
抹去 54
抹子 0
抹布 87
抹得 1
抹成 1
抹抹 9
抹掉 9
抹摋 0
抹擦 2
抹殺 56
抹煞 62
抹煞 62
抹破 0
抹粉 2
抹胸 0
抹胸 0
抹脖子 0
抹臉 2
抹茶 1
抹起 1
抹起來 0
抹過 8
抹黑 463
抺 0
抻 111
押 690
押 690
押出 150
押匯 100
押回 7
押回來 0
押回去 0
押在 8
押對 10
押尾 0
押尾 0
押抵 2
押款 1
押注 7
押票 0
押租 3
押給 1
押著 22
押解 21
押解到 4
押起來 0
押車 1
押載 1
押送 20
押運 12
押金 120
押隊 3
押韻 37
抽 2049
抽上 1
抽上來 0
抽上去 0
抽中 75
抽光 3
抽出 374
抽出來 12
抽出去 1
抽出空 0
抽動 32
抽印本 36
抽印論文 0
抽去 3
抽取 403
抽唱 0
抽問 2
抽噎 1
抽屜 145
抽得 23
抽成 6
抽打 5
抽抽 3
抽抽噎噎 5
抽抽搭搭 0
抽掉 16
抽提 2
抽換 95
抽換法 0
抽搐 97
抽搐 97
抽搐起來 0
抽搭 0
抽時間 3
抽暇 0
抽查 194
抽查法 0
抽樣 316
抽樣學 0
抽樣法 9
抽樣調查 49
抽款 0
抽氣 57
抽氣機 2
抽水 85
抽水機 97
抽水站 104
抽水馬桶 15
抽汽 3
抽油煙機 19
抽泣 8
抽煙 319
抽煙室 0
抽獎 653
抽的 24
抽皺 0
抽盡 0
抽砂 18
抽稅 2
抽穗 5
抽穗期 0
抽空 139
抽空來 6
抽空到 4
抽空去 3
抽筋 143
抽籤 320
抽紗 2
抽紗業 0
抽絲 36
抽絲剝繭 36
抽繹出 0
抽考 7
抽脂 0
抽芽 3
抽菸 0
抽血 0
抽調 23
抽象 367
抽象主義 3
抽象勞動 0
抽象名詞 6
抽象思維 1
抽象概念 17
抽象派 7
抽象畫 10
抽起 10
抽起來 2
抽身 32
抽身而退 0
抽進 1
抽進來 0
抽進去 0
抽錯 0
抽長 5
抽頭 4
抽風 14
抽風機 25
抽點 7
抾 55
抿 61
抿住 0
抿嘴 5
抿嘴笑 2
抿著 2
抿著嘴 7
抿著嘴笑 1
拀 0
拁 0
拂 193
拂 193
拂去 10
拂塵 30
拂子 0
拂拂 1
拂拭 11
拂掉 1
拂曉 32
拂衣 2
拂袖 3
拂袖而去 8
拂逆 8
拂面 11
拂面而來 3
拄 56
拄著 8
担 0
担 0
拆 1002
拆下 62
拆下來 23
拆下去 0
拆伙 0
拆信 67
拆倒 0
拆出 2
拆出來 0
拆出去 0
拆到 5
拆卸 167
拆卸法 0
拆去 1
拆台 0
拆回 0
拆回來 0
拆回去 0
拆夥 19
拆字 10
拆封 67
拆屋還地 1
拆息 11
拆掉 49
拆散 22
拆散開來 0
拆斷 1
拆款 0
拆毀 42
拆洗 4
拆洗法 0
拆穿 29
拆穿騙局 0
拆線 20
拆臺 0
拆船業 3
拆解 68
拆遷 100
拆開 66
拆開來 5
拆閱 7
拆除 651
拇 29
拇印 0
拇戰 0
拇指 123
拇趾 10
拈 50
拈 50
拈度 0
拈弄 0
拈弄 0
拈斤撥兩 0
拈斷 0
拈線 0
拈花 18
拈花 18
拈花弄月 1
拈花微笑 1
拈花惹草 29
拈酒 0
拈香 3
拈香 3
拈鬚 0
拉 9325
拉丁 270
拉丁人 1
拉丁字母 2
拉丁文 65
拉丁民族 6
拉丁美洲 378
拉丁語 7
拉丁語系 1
拉三扯四 0
拉上 49
拉上來 7
拉上去 2
拉下 68
拉下來 6
拉下去 4
拉下水 0
拉下臉 2
拉下臉來 0
拉不動 0
拉了 56
拉了一把 2
拉了一把 2
拉交情 1
拉人 13
拉伕 1
拉伸 73
拉住 50
拉來 5
拉來拉去 0
拉倒 5
拉入 5
拉出 121
拉出來 12
拉出去 2
拉到 121
拉制 0
拉力 181
拉動 28
拉去 8
拉各斯 0
拉向 6
拉回 302
拉回來 11
拉回去 2
拉圾 6
拉壞 3
拉多 65
拉大 101
拉夫 51
拉奧孔 0
拉好 5
拉完 3
拉客 26
拉客戶 0
拉尿 0
拉屎 15
拉岡 8
拉巴 58
拉巴斯 4
拉巴特 4
拉布拉他 0
拉布拉他河 0
拉布拉多 18
拉布拉多寒流 0
拉布拉多犬 1
拉幅 4
拉平 6
拉廣告 6
拉弓 6
拉引 3
拉往 8
拉後腿 0
拉得 219
拉德 39
拉慣 0
拉成 16
拉手 6
拉托維亞 2
拉扯 67
拉拉 90
拉拉山 46
拉拉扯扯 5
拉拉隊 2
拉拉隊員 0
拉拉隊長 0
拉拉雜雜 14
拉拔 62
拉拖 0
拉提琴 1
拉攏 113
拉攏起來 0
拉斐爾 16
拉斯維加斯 136
拉斷 6
拉昇 2
拉曳 4
拉有 3
拉杆 0
拉桿 52
拉環 252
拉瓦錫 1
拉生意 4
拉的 219
拉皮 50
拉皮手術 8
拉皮條 3
拉直 10
拉票 205
拉稀 0
拉筋 1
拉管 1
拉絲 64
拉網 7
拉緊 35
拉線 38
拉線開關 0
拉縴 0
拉繩 12
拉肚子 98
拉腿 1
拉著 203
拉薩 56
拉薩人 0
拉薩市 4
拉薩爾 1
拉賬 0
拉走 15
拉起 83
拉起來 12
拉跑 0
拉車 27
拉近 244
拉進 28
拉進來 5
拉進去 1
拉過 10
拉過來 8
拉過去 1
拉鋸 0
拉鋸戰 21
拉鍊 214
拉鏈 60
拉長 131
拉門 86
拉開 159
拉關係 4
拉雜 5
拉風 40
拉高 106
拉麵 86
拊 53
拊掌 2
拋 406
拋下 46
拋下來 0
拋下去 1
拋來 3
拋來拋去 1
拋光 120
拋出 60
拋出來 0
拋出去 2
拋到 20
拋去 10
拋向 7
拋售 93
拋射 4
拋射體 1
拋得 11
拋戈棄甲 0
拋掉 10
拋擲 12
拋棄 403
拋物線 35
拋物面 3
拋的 11
拋磚 4
拋磚引玉 92
拋空 4
拋繡球 3
拋起 9
拋起來 0
拋錨 34
拋開 131
拋除 3
拋頭露面 7
拋頭露面 7
拌 480
拌勺 0
拌勻 112
拌和 52
拌嘴 6
拌種 0
拌著 4
拌點 0
拍 2638
拍上 7
拍上來 0
拍上去 0
拍下 52
拍下來 9
拍下去 1
拍來拍去 0
拍出 88
拍出來 3
拍出去 0
拍到 82
拍去 5
拍在 17
拍外景 2
拍好 30
拍子 56
拍平 3
拍得 106
拍得 250
拍成 59
拍戲 70
拍手 70
拍手叫好 11
拍手叫絕 0
拍手稱快 0
拍打 73
拍打在 3
拍打著 6
拍拍 89
拍拍手 19
拍掉 6
拍掌 19
拍擊 26
拍攝 1427
拍攝場 1
拍攝場 1
拍散 0
拍景 1
拍景隊 0
拍板 31
拍案 20
拍案叫絕 23
拍案驚奇 5
拍桌子 7
拍法 14
拍照 507
拍照存證 17
拍照片 6
拍片 124
拍片子 0
拍球 5
拍發 7
拍的 250
拍節機 0
拍紙簿 0
拍老腔兒 0
拍老腔兒 0
拍肩膀 0
拍胸脯 10
拍脫 1
拍著 32
拍蒼蠅 2
拍號 1
拍賣 833
拍賣會 147
拍起 16
拍起來 3
拍門 2
拍電報 0
拍電影 18
拍頻 0
拍馬屁 17
拍點 2
拎 81
拎下來 1
拏 80
拏空 1
拏雲 80
拏雲攫石 1
拐 84
拐上 1
拐上來 0
拐上去 0
拐下 0
拐下來 0
拐下去 0
拐了 5
拐人 0
拐住 0
拐來 1
拐來拐去 1
拐出 1
拐出來 0
拐出去 0
拐到 6
拐去 0
拐回 3
拐回來 0
拐回去 0
拐子 7
拐孩子 0
拐帶 1
拐彎 9
拐彎兒 0
拐彎兒 0
拐彎抹角 15
拐得 0
拐拐 0
拐杖 63
拐棍 1
拐腳 0
拐著 1
拐角 2
拐角兒 0
拐角兒 0
拐誘 1
拐走 7
拐起 0
拐起來 0
拐跑 3
拐進 3
拐進來 0
拐進去 0
拐過 1
拐過來 0
拐過去 0
拐騙 7
拑 107
拒 773
拒人於千里之外 7
拒付 12
拒吃 5
拒唱 0
拒在 5
拒彈 0
拒打 2
拒抗 4
拒拿 0
拒拿回扣 0
拒捕 8
拒接 1
拒擋 0
拒收 49
拒敵 7
拒斥 16
拒用 10
拒看 18
拒答 28
拒絕 1977
拒絕往來 32
拒絕往來戶 77
拒繳 23
拒談 3
拒買 20
拒載 6
拒馬 12
拓 235
拓 2357
拓務省 0
拓印 8
拓印 8
拓地 4
拓墾 34
拓墾荒地 0
拓寬 182
拓寬道路 5
拓展 1295
拓撲 9
拓本 6
拓本 6
拓樣 3
拓樸 59
拓樸學 8
拓殖 39
拓荒 87
拓荒者 83
拓落 0
拓跋 8
拓跋氏 0
拓跋燾 0
拓跋貴 0
拓邊 0
拔 838
拔下 25
拔下來 8
拔下去 0
拔不出 6
拔不到 0
拔了 11
拔個 2
拔出 105
拔出來 12
拔出去 1
拔刀 275
拔刀相助 26
拔劍 35
拔去 5
拔取 15
拔回 2
拔回來 1
拔回去 0
拔地而起 3
拔城 0
拔山 8
拔掉 71
拔擢 55
拔染 31
拔根 0
拔毒 0
拔毛 9
拔河 128
拔河比賽 27
拔河賽 5
拔營 11
拔牙 49
拔用 0
拔節 0
拔絲 5
拔絲地瓜 1
拔線 0
拔群 2
拔腳 2
拔腿 8
拔腿就跑 13
拔苗助長 0
拔草 5
拔著 0
拔蘿蔔 0
拔起 27
拔起來 4
拔身 3
拔進 0
拔進來 0
拔進去 0
拔都 6
拔錨 0
拔開 3
拔除 78
拕 0
拖 1606
拖上 14
拖上來 1
拖上去 0
拖下 3
拖下來 1
拖下去 13
拖下水 16
拖乾淨 0
拖了 75
拖人下水 0
拖住 9
拖來 5
拖來拖去 1
拖倒 0
拖出 22
拖出來 5
拖出去 5
拖到 94
拖動 11
拖去 7
拖吊 268
拖吊車 10
拖回 29
拖回來 4
拖回去 2
拖地 37
拖地板 4
拖垮 67
拖家帶眷 0
拖帶 11
拖延 185
拖延戰術 5
拖延時間 14
拖延者 1
拖得 31
拖把 32
拖拉 88
拖拉機 10
拖拉機手 0
拖拉機站 0
拖拖 2
拖拖拉拉 13
拖掉 4
拖撈 0
拖撈網 0
拖撈船 0
拖時間 4
拖曳 251
拖板 19
拖橇狗 0
拖欠 11
拖沓 1
拖油瓶 7
拖泥帶水 23
拖洗 0
拖濕 0
拖稿 1
拖累 152
拖網 81
拖網漁船 21
拖網船 0
拖船 29
拖著 136
拖走 7
拖起 4
拖起來 0
拖車 132
拖進 13
拖進來 0
拖進去 3
拖過 18
拖長 13
拖開 2
拖鞋 110
拖鞋櫃 0
拗 49
拗 49
拗 49
拗 49
拗口 3
拗口 3
拗口令 0
拗口令 0
拗強 0
拗彆 0
拗性 1
拘 128
拘出 0
拘役 232
拘押 16
拘拘縮縮 0
拘捕 68
拘提 76
拘束 129
拘泥 23
拘泥於 36
拘留 59
拘留所 10
拘票 25
拘禁 68
拘禮 1
拘管 0
拘謹 32
拘限 9
拙 236
拙作 74
拙劣 23
拙嘴笨腮 0
拙嘴笨舌 0
拙夫 6
拙妻 0
拙於 5
拙於言詞 3
拙笨 1
拙筆 2
拙荊 2
拙著 10
拙行 1
拙見 21
拙計 0
拙訥 0
拚 257
拚命 120
拚死 12
拚鬥 5
招 5861
招了 41
招事 0
招些 0
招人 8
招人物議 0
招來 87
招來橫禍 0
招供 10
招供出來 0
招做 3
招兒 1
招兒 1
招兵 3
招兵買馬 101
招出 5
招出來 0
招到 9
招募 756
招募新兵 1
招呼 217
招呼客人 3
招呼站 24
招咎 0
招商局 32
招回 3
招回來 1
招回去 0
招女婿 0
招妓 2
招嫉 115
招安 14
招展 9
招工 2
招式 173
招引 8
招待 343
招待不週 4
招待券 25
招待員 3
招待室 2
招待所 117
招待會 3
招徠 37
招怨 0
招惹 42
招惹不起 1
招惹是非 1
招手 112
招搖 27
招搖撞騙 8
招搖過市 5
招搖過街 0
招撫 4
招攬 105
招攬生意 13
招收 828
招收新生 30
招數 91
招架 39
招架不住 15
招架之力 5
招標 8222
招權納賄 0
招權納賄 0
招涼 0
招牌 617
招牌戲 2
招牌歌 2
招牌菜 1
招牌飯 3
招生 4319
招生簡章 660
招眼 0
招禍 0
招福 3
招租 168
招納 6
招罪 0
招考 295
招聘 38
招聘員工 2
招股 0
招自 1
招致 86
招蜂引蝶 10
招親 15
招認 4
招請 6
招財進寶 5
招貼 4
招賢 4
招贅 9
招起 0
招進 3
招進來 0
招進去 0
招降 10
招降納叛 0
招集 115
招領 41
招領失物 0
招領處 2
招風 5
招風惹雨 0
招風耳 4
招魂 39
招魂旛 0
招魂鈴 0
拜 2930
拜上帝會 0
拜伏 3
拜佛 40
拜來拜去 1
拜倒 9
拜倒石榴裙 1
拜倒石榴裙之下 0
拜候 3
拜倫 9
拜偶像 31
拜別 8
拜到 3
拜占庭 42
拜在 7
拜在門下 1
拜堂 13
拜墓 2
拜墳 0
拜壽 13
拜天地 2
拜好 3
拜官 0
拜客 0
拜帖 0
拜帖 0
拜師 37
拜師學藝 16
拜年 34
拜廟 4
拜把 5
拜把兄弟 3
拜拜 199
拜揖 1
拜斗 1
拜會 690
拜服 1
拜望 1
拜火教 3
拜碼頭 27
拜神 30
拜票 0
拜祭 6
拜禮 1
拜禱 1
拜節 0
拜老師 0
拜著 1
拜見 28
拜託 939
拜訪 1548
拜請 1
拜謁 6
拜謝 20
拜讀 62
拜賀 3
拜起 0
拜起來 0
拜跪 0
拜辭 3
拜金 15
拜金主義 6
拜門 1
拜領 1
拜點 0
拝 0
拞 0
拞 0
拞 0
拟 0
拟 0
拠 0
拡 0
拤 0
拪 0
拫 36
括 1729
括 1729
括弧 102
括弧 102
括約肌 40
括號 93
括除 2
拭 113
拭乾 9
拭去 36
拭淚 8
拭淨 3
拭目 7
拭目以待 138
拭除 1
拮 134
拮据 36
拮据 36
拯 55
拯救 555
拰 16
拱 593
拱出 9
拱廊 9
拱形 14
拱形門 0
拱手 52
拱手相讓 1
拱手行禮 1
拱手讓人 5
拱手讓出 7
拱托 1
拱抱 0
拱推 0
拱撞 0
拱月 0
拱木 1
拱橋 36
拱狀 0
拱立 1
拱背 1
拱花 0
拱著 2
拱衛 4
拱護 1
拱豬 36
拱起 6
拱路 0
拱道 0
拱門 41
拱開 0
拱頂 5
拲 156
拳 1183
拳勇 0
拳壇 7
拳師 8
拳師狗 0
拳形 2
拳手 9
拳打腳踢 30
拳擊 120
拳擊台 0
拳擊場 2
拳擊場 2
拳擊手 13
拳擊賽 3
拳棒 0
拳法 42
拳王 41
拳腳交加 1
拳蟹 2
拳術 35
拳賽 4
拳賽選手 0
拳頭 117
拳頭 117
拴 32
拴上 4
拴上來 0
拴上去 0
拴住 6
拴在 10
拴繩 0
拴著 5
拴鎖 1
拴馬 1
拵 106
拶 7
拷 346
拷問 27
拷打 40
拷貝 917
拸 817
拹 27
拺 26
拻 337
拼 1047
拼上 7
拼上來 0
拼上去 0
拼下 1
拼下來 0
拼下去 4
拼了 23
拼了命 5
拼來 0
拼來拼去 0
拼出 30
拼出來 3
拼出去 0
拼列 0
拼列出 0
拼到 9
拼到底 1
拼合 3
拼命 288
拼回 0
拼回來 1
拼回去 0
拼圖 482
拼圖遊戲 34
拼在 2
拼在一起 1
拼在一起 1
拼字 107
拼字母 0
拼字遊戲 4
拼寫 3
拼寫法 0
拼得 3
拼得 7
拼成 20
拼拼 10
拼拼湊湊 8
拼拼看 1
拼會 0
拼死 14
拼死拼活 3
拼法 31
拼湊 56
拼湊出 11
拼的 7
拼盤 164
拼綴 0
拼綴出 0
拼綴成 0
拼花 16
拼花玻璃 0
拼著 2
拼貼 93
拼過 0
拼過來 0
拼過去 0
拼錯 23
拼音 215
拼音字母 10
拼音文字 11
拽 18
拽 18
拽 18
拾 140
拾 1407
拾不得 0
拾人牙慧 1
拾到 7
拾去 0
拾取 12
拾回 33
拾得 26
拾掇 3
拾獲 74
拾破爛 0
拾穗 47
拾級 4
拾級 4
拾級而上 9
拾荒 32
拾荒老人 13
拾荒者 7
拾起 77
拾起來 5
拾遺 30
拾金不昧 4
拿 4102
拿上 4
拿上來 4
拿上去 0
拿下 405
拿下來 20
拿下去 0
拿不住 3
拿不到 61
拿不好 0
拿不定 10
拿不準 3
拿不起來 1
拿主意 0
拿些 9
拿住 8
拿來 550
拿來拿去 0
拿個 39
拿出 629
拿出來 204
拿出去 5
拿刀 13
拿到 920
拿去 222
拿取 51
拿回 77
拿回來 11
拿回去 15
拿大 90
拿好 25
拿定 1
拿定主意 1
拿得 82
拿得走 1
拿得起 8
拿手 89
拿手功夫 0
拿手好戲 3
拿捏 0
拿掉 151
拿摩溫 0
拿有 6
拿準 1
拿的 82
拿破崙 117
拿破崙帝國 3
拿穩 0
拿筆 7
拿給 93
拿腔做勢 0
拿著 569
拿走 100
拿起 409
拿起來 27
拿辦 6
拿過 24
拿過來 3
拿過去 3
拿酒 3
拿鐵 1
拿開 13
挀 17
持 5319
持久 264
持久力 5
持久性 24
持久戰 6
持之以恆 54
持之有故 2
持份 0
持刀 64
持分 83
持分 83
持勝 0
持家 21
持平 138
持平之論 2
持戒 61
持戟 0
持操 1
持明 1
持有 1888
持槍 64
持正 18
持法 1
持球 31
持盈保泰 12
持矛 3
持票人 13
持節 9
持續 6105
持續性 243
持續訂購 5
持續音 1
持股 1257
持衡 1
持論 6
持質 1
持質 1
持身 14
持重 6
持齋 5
挂 79
挂名 1
挂圖 0
挂帥 1
挂彩 0
挂念 1
挂號 1
挂車 0
挂鉤 0
挂面 0
挃 7
挄 0
挅 0
挆 0
指 23530
指不勝屈 1
指了 31
指令 2762
指令區 14
指位 18
指使 61
指來指去 0
指出 6115
指到 71
指北 24
指北針 41
指南 5497
指南車 7
指南針 56
指南魚 0
指印 16
指名 120
指名道姓 11
指向 385
指回 0
指回來 0
指回到 0
指回去 0
指在 235
指壓 149
指壓法 0
指天畫地 0
指天誓日 0
指套 8
指好 1
指完 3
指定 4378
指定人 8
指導 6456
指導人 4
指導員 155
指導思想 10
指導教授 803
指導科 0
指尖 122
指引 2473
指戰員 1
指手劃腳 0
指手畫腳 1
指指點點 23
指控 626
指揮 1378
指揮刀 2
指揮員 0
指揮官 180
指揮家 366
指揮所 27
指揮棒 40
指揮艙 3
指揮若定 3
指揮處 0
指揮車 10
指揮部 50
指摘 53
指摘 53
指教 4346
指數 3551
指數函數 8
指斥 10
指日 47
指日可待 84
指日可數 0
指明 213
指書 1
指望 100
指東畫西 0
指東說西 0
指桑罵槐 9
指標 4852
指標體系 6
指模 0
指正 224
指法 39
指津 8
指派 445
指派法 0
指涉 0
指為 156
指爪 6
指爪 6
指物 6
指環 42
指甲 290
指甲剪 1
指甲油 1
指痕 2
指的 0
指的是 0
指示 11559
指示劑 54
指示器 83
指示板 1
指示燈 206
指示牌 32
指示菌 0
指稱 285
指端 12
指紋 339
指給 12
指縫 9
指縫間 4
指腹為婚 16
指著 306
指認 41
指說 11
指證 36
指證歷歷 6
指責 492
指起 1
指起來 0
指路 14
指路牌 0
指迷 1
指針 215
指間 49
指關節 7
指陳 154
指靠 2
指頭 110
指頭 110
指鹿為馬 5
指點 422
挈 114
挈領 2
按 12760
按上 118
按上來 0
按上去 1
按下 1931
按下不表 1
按下來 0
按下去 26
按了 87
按件 4200
按件計費 0
按件計酬 8
按住 338
按住不放 2
按例 3
按兵 2
按兵不動 14
按出 17
按出來 1
按出去 0
按到 17
按勞分配 1
按名次 1
按回 10
按回來 0
按回去 0
按圖 53
按圖為證 0
按圖索驥 105
按址 10
按壓 25
按壓下來 0
按壓下去 0
按壓不住 0
按好 3
按察使 3
按年 161
按年計費 0
按年計酬 0
按序 3
按序存取 0
按序存取存貯 0
按得 7
按得 16
按捺 15
按捺不住 29
按摩 1044
按摩人 0
按摩師 9
按摩療法 7
按日 425
按日計費 1
按日計酬 4
按時 359
按時收看 1
按時收聽 3
按月 223
按月計費 0
按月計酬 0
按期 88
按期收費 0
按次 38
按次計費 4
按次計酬 0
按此 1497
按比例發展 0
按照 1176
按照著 0
按片 0
按片計費 0
按片計酬 0
按理 16
按理出牌 0
按理說 10
按的 16
按科 16
按科計費 0
按篇 1
按篇計費 0
按篇計酬 0
按脈 1
按脈 1
按規定 126
按計劃 10
按語 2
按說 15
按說是 0
按起 0
按起來 3
按過 8
按過來 0
按過去 0
按道理說 0
按道理講 0
按部就班 48
按鈕 1527
按鈕開關 26
按鈴 51
按鈴申告 13
按鈴聲 0
按鍵 4263
按需分配 1
挊 0
挊 0
挊 0
挊 0
挋 38
挌 77
挍 78
挎 5
挎 5
挎包 1
挏 2
挐 122
挐 122
挑 113
挑 1130
挑三揀四 2
挑上 8
挑上來 0
挑上去 0
挑下 6
挑下來 0
挑下去 0
挑不動 0
挑中 12
挑了 41
挑出 69
挑出來 10
挑出去 0
挑到 8
挑剔 114
挑剔 114
挑剔 114
挑動 33
挑台 1
挑唆 6
挑嘴 13
挑大樑 24
挑夫 13
挑好 10
挑子 0
挑字眼 0
挑山工 0
挑弄是非 0
挑得 5
挑得 27
挑情 2
挑情 2
挑戰 3034
挑戰書 27
挑戰者 47
挑手 1
挑挑 4
挑挑揀揀 0
挑揀 3
挑揀出 1
挑撥 39
挑撥是非 1
挑撥是非 1
挑撥離間 12
挑撥離間 12
挑擔 3
挑毛病 1
挑水 22
挑燈 12
挑燈夜戰 15
挑燈夜戰 15
挑片 0
挑的 27
挑眼 1
挑眼 1
挑舋 0
挑花 0
挑著 17
挑讀 0
挑走 3
挑起 184
挑起來 1
挑逗 85
挑運 0
挑選 762
挑選出 49
挑選到 7
挑釁 105
挑錯 4
挑開 4
挑食 14
挒 0
挓 21
挔 64
挕 179
挖 549
挖 759
挖上 0
挖上來 0
挖上去 0
挖下 4
挖下來 1
挖下去 0
挖井 2
挖來 5
挖來挖去 0
挖倒 0
挖出 113
挖出來 21
挖出去 0
挖到 37
挖取 19
挖土 22
挖土機 55
挖地 15
挖地洞 1
挖地道 0
挖坑 5
挖墓 0
挖大 1
挖大一點 0
挖大一點 0
挖寶 0
挖得 8
挖得 10
挖掉 9
挖掘 418
挖掘出 30
挖掘機 6
挖東牆補西牆 3
挖水溝 1
挖泥 3
挖泥機 6
挖泥船 6
挖洞 18
挖溝 6
挖煤 1
挖牆腳 4
挖的 10
挖瞎 0
挖空 28
挖空心思 11
挖穿 1
挖耳朵 8
挖耳朵 8
挖肉補瘡 0
挖苦 13
挖補 3
挖補到 0
挖角 59
挖走 6
挖起 3
挖起來 7
挖路 11
挖路隊 0
挖通 0
挖進 2
挖進來 0
挖進去 0
挖過 7
挖過來 4
挖過去 0
挖開 3
挖除 14
挗 0
挙 0
挛 0
挛 0
挟 0
挟 0
挟 0
挡 0
挡 0
挣 0
挣 0
挨 287
挨上去 0
挨了 30
挨了揍 0
挨個兒 0
挨個兒 0
挨家挨戶 24
挨戶 2
挨打 27
挨揍 11
挨次 4
挨罵 27
挨著 9
挨近 4
挨過 7
挨門 0
挨餓 25
挨餓受凍 6
挩 198
挪 239
挪不動 1
挪不開 0
挪作 2
挪作他用 2
挪借 0
挪出 20
挪動 10
挪威 522
挪威人 11
挪威文 10
挪威話 0
挪威語 3
挪款 0
挪用 73
挪用公款 7
挪移 83
挪開 10
挫 450
挫傷 41
挫冰 0
挫折 408
挫折感 54
挫敗 166
挫敵 1
挫辱 0
挬 73
挭 141
挮 0
振 5715
振作 65
振作起 1
振作起來 25
振動 669
振動器 15
振動篩 3
振動體 0
振奮 110
振奮人心 18
振奮起 0
振奮起來 4
振威 20
振幅 88
振德 24
振振 1
振振有詞 14
振振有辭 6
振揚 12
振暴 0
振武學社 0
振盪 129
振盪器 66
振盪電流 0
振盪電路 7
振筆 3
振筆疾書 8
振筆直書 3
振翅 28
振翅高飛 3
振聲 45
振聾發嘳 0
振聾發聵 0
振臂 6
振臂一呼 0
振臂一呼 0
振臂高呼 9
振興 660
振興實業 2
振興醫療復健中心 0
振蕩 1
振蕩器 0
振衰起蔽 1
振起 5
挰 0
挱 0
挱 0
挱 0
挲 46
挳 59
挳 59
挴 259
挵 0
挵 0
挵 0
挵 0
挶 92
挷 0
挷 0
挸 51
挹 115
挹注 196
挺 797
挺上 0
挺上來 0
挺上去 0
挺下 0
挺下來 0
挺下去 2
挺不住 1
挺不錯 29
挺了 4
挺伸 0
挺住 6
挺來挺去 0
挺出 9
挺出來 0
挺出去 0
挺到 1
挺升 5
挺好 44
挺得 11
挺拔 38
挺挺 3
挺有 32
挺直 23
挺秀 9
挺立 36
挺胸 8
挺舉 22
挺著 21
挺起 28
挺起來 1
挺起胸來 0
挺身 65
挺身而出 44
挺進 56
挺進來 0
挺進去 0
挻 8
挼 83
挼 83
挼搓 0
挽 248
挽住 2
挽回 171
挽手 2
挽救 311
挽救不了 1
挽歌 2
挽留 47
挽著 34
挽袖 4
挽起 14
挽髻 0
挾 444
挾 3100
挾 444
挾住 4
挾到 2
挾制 4
挾制 4
挾天子以令諸侯 1
挾天子以令諸侯 1
挾天子以令諸侯 1
挾帶 59
挾恨 1
挾恨 1
挾持 106
挾書律 0
挾殺 0
挾著 76
挾藏 0
挾藏 0
挾藏 0
挿 0
捀 85
捁 58
捂 111
捃 93
捄 313
捅 199
捅了 6
捅進 0
捆 183
捆上 1
捆了 2
捆住 3
捆來捆去 0
捆到 0
捆在 5
捆好 1
捆工 9
捆得 3
捆成 2
捆成一堆 0
捆紮 13
捆綁 94
捆緊 0
捆縛 5
捆著 1
捆起 2
捆起來 2
捇 112
捈 72
捉 589
捉上來 0
捉上去 0
捉下來 7
捉下去 0
捉不勝捉 0
捉住 46
捉刀 6
捉到 42
捉妖 3
捉姦 7
捉弄 60
捉拿 23
捉拿犯人 0
捉捕 4
捉摸 25
捉摸不出 0
捉摸不定 21
捉狹鬼 0
捉脈 0
捉脈 0
捉襟肘見 0
捉襟肘見 0
捉襟見肘 25
捉賊 13
捉賊記 1
捉起 1
捉起來 5
捉迷藏 41
捊 70
捋 360
捋 360
捋 360
捋奶 0
捋胳膊 0
捋胳膊 1
捋臂將拳 1
捋臂揎拳 1
捋虎鬚 0
捋袖揎拳 1
捋采 1
捌 710
捍 122
捍拒 0
捍然 0
捍禦 0
捍衛 500
捍衛戰士 13
捎 3482
捎來 26
捎信兒 0
捎信兒 0
捎去 0
捎帶 1
捏 716
捏一把冷汗 5
捏一把汗 0
捏一捏 1
捏住 13
捏作 0
捏合 21
捏告 0
捏成 20
捏手捏腳 0
捏把冷汗 13
捏把汗 0
捏控 0
捏握 0
捏握著 0
捏死 7
捏碎 7
捏稱 0
捏著 17
捏著鼻子 7
捏詞 0
捏造 35
捏造出 3
捐 983
捐入 2
捐出 125
捐助 672
捐助人 21
捐助者 5
捐書 35
捐棄 13
捐款 797
捐款箱 2
捐獻 216
捐獻箱 2
捐稅 1
捐血 210
捐血人 9
捐血車 5
捐贈 938
捐贈人 12
捐贈者 63
捐軀 17
捐錢 27
捐錢箱 0
捑 63
捒 0
捒 0
捒 0
捓 0
捔 58
捕 923
捕役 0
捕快 6
捕手 226
捕拿 0
捕捉 464
捕捉到 31
捕撈 129
捕撈到 2
捕殺 39
捕獲 112
捕蟲 13
捕蟲葉 2
捕蠅紙 2
捕頭 6
捕風 1
捕風捉影 15
捕食 57
捕魚 211
捕魚船 1
捕鯨 39
捕鯨船 7
捕鼠器 13
捖 119
捗 68
捘 67
捙 73
捚 220
捛 0
捜 0
捝 0
捝 0
捥 11
捦 0
捧 688
捧上 6
捧上來 0
捧上去 0
捧上天 1
捧住 5
捧出 9
捧出來 0
捧出去 0
捧場 22
捧場 224
捧得 3
捧得 11
捧心 1
捧成 3
捧的 11
捧腹 14
捧腹大笑 13
捧茶 0
捧著 93
捧起 17
捨 793
捨不得 251
捨不得 251
捨出 2
捨去 41
捨命 32
捨命陪君子 4
捨己從人 0
捨己救人 0
捨己為人 2
捨己為公 0
捨己為國 0
捨得 91
捨得 91
捨我其誰 17
捨本逐末 11
捨棄 194
捨死忘生 0
捨生 5
捨生取義 0
捨短取長 0
捨身 29
捨身報國 0
捨身救人 3
捨車保帥 0
捨車保帥 0
捨近求遠 7
捩 206
捪 0
捫 169
捫住 0
捫心 5
捫心自問 24
捫診 0
捬 0
捬 0
捭 8
据 258
据 258
捯 8
捯線 0
捰 312
捱 425
捱上 0
捱了 5
捱了過來 0
捱了過去 0
捱到 11
捱得 0
捱得 1
捱得住 1
捱打 3
捱的 1
捱肩擦背 0
捱苦 1
捱著 1
捱近 0
捱過 5
捱過來 0
捱過去 0
捱餓 2
捲 1685
捲住 2
捲來 1
捲來捲去 0
捲兒 1
捲兒 1
捲入 198
捲入漩渦 1
捲動 120
捲動門 0
捲去 2
捲圖 0
捲土重來 57
捲尺 21
捲心菜 1
捲成 23
捲曲 54
捲款逃走 0
捲毛 54
捲煙 11
捲煙盒 0
捲筒 34
捲筒紙 1
捲簾 20
捲簾格 1
捲紙 20
捲縮 22
捲舌 9
捲舌音 2
捲菸 11
捲袖 1
捲走 7
捲起 87
捲起來 10
捲軸 98
捲逃 0
捲鋪蓋 2
捲雲 3
捲風 11
捲髮 17
捲髮器 2
捳 0
捴 0
捵 9
捵 9
捵麵 0
捵麵 0
捶 297
捶平 0
捶打 7
捶胸 1
捶胸頓足 6
捶鍊 0
捷 3011
捷安特 100
捷克 386
捷克斯拉夫 4
捷克斯洛伐克 7
捷報 100
捷徑 586
捷書 0
捷足先得 0
捷足先登 28
捷運 1717
捷運局 93
捷運站 1
捷運系統 186
捸 562
捹 0
捺 75
捺印 18
捻 173
捻亂 0
捻匪 0
捻度 0
捻搓 1
捻斷 0
捻滅 0
捻神捻鬼 0
捻線 0
捻花微笑 1
捻起 2
捻軍 1
捻鼻 0
捼 4
捽 10
捾 0
捾 0
捿 0
捿 0
掀 13920
掀了 4
掀了開來 0
掀住 0
掀倒 2
掀出 2
掀出來 1
掀出去 0
掀到 1
掀動 6
掀天 0
掀天動地 0
掀底 0
掀掉 1
掀簾子 0
掀被 0
掀起 464
掀起來 0
掀開 71
掀風播浪 0
掁 227
掂 77
掂倈 1
掂對 1
掂折 1
掂掇 1
掂提 1
掂斤估兩 1
掂斤抹兩 1
掂斤撥兩 0
掂斤播兩 1
掂梢折本 1
掂算 0
掂腳舒腰 1
掂詳 1
掂過兒 1
掂量 1
掃 184
掃 1846
掃乾淨 9
掃來 1
掃來掃去 2
掃出 15
掃出來 1
掃出去 0
掃到 26
掃去 2
掃地 126
掃地出門 17
掃堂腿 0
掃墓 35
掃墓節 22
掃墳 1
掃射 29
掃帚 27
掃帚星 3
掃平 5
掃把 29
掃把星 1
掃掉 0
掃描 2095
掃描器 957
掃描式 54
掃描機 181
掃毒 0
掃清 3
掃盪 1
掃盲 1
掃瞄 941
掃瞄器 741
掃興 27
掃蕩 175
掃蕩殘敵 0
掃街 0
掃視 9
掃走 0
掃進 10
掃進來 1
掃進去 5
掃過 54
掃過來 1
掃過去 3
掃選 0
掃除 140
掃雷 21
掃雷艇 18
掃雷艦 9
掃黃 0
掃黑 0
掄 171
掄 171
掄刀 1
掄材 0
掄棍 0
掄起 8
掅 53
掆 0
掇 81
掇拾 6
授 2592
授乳 35
授予 379
授人以柄 1
授人口實 1
授人話柄 0
授以 16
授以全權 0
授勛 0
授勛儀式 0
授勛典禮 0
授受 14
授命 14
授命組閣 0
授徒 6
授意 19
授旗 23
授旗儀式 4
授旗典禮 7
授業 42
授槍 0
授權 4271
授權書 161
授獎 17
授獎儀式 0
授獎典禮 0
授理 1
授田 18
授粉 24
授精 30
授給 9
授職 13
授與 168
授記 8
授課 2062
授階 6
授階儀式 0
授階典禮 0
掉 4235
掉下 110
掉下來 85
掉下去 10
掉了 400
掉以輕心 88
掉來掉去 0
掉個 2
掉入 81
掉到 90
掉包 2
掉在 66
掉得 6
掉得 6
掉掉 2
掉換 9
掉書袋 0
掉皮 0
掉色 7
掉落 199
掉落在 32
掉著 3
掉著 3
掉轉 7
掉轉身 0
掉轉頭 3
掉進 48
掉進來 0
掉進去 0
掉過 10
掉過頭來 1
掉過頭去 5
掉隊 0
掉頭 35
掉頭就走 5
掉頭就跑 3
掉魂 0
掊 19
掋 0
掌 2031
掌上 428
掌上明珠 7
掌中戲 37
掌印 10
掌嘴 1
掌廚 16
掌心 59
掌心雷 9
掌持 0
掌持著 0
掌控 0
掌握 3569
掌握住 49
掌握著 18
掌政 8
掌政者 0
掌故 79
掌旗 17
掌柜 9
掌櫃 310
掌櫃的 15
掌權 100
掌權人 0
掌灶 0
掌燈 7
掌狀 12
掌狀脈 0
掌狀脈 0
掌理 410
掌管 111
掌紋家 0
掌聲 270
掌聲如雷 1
掌聲雷動 4
掌舵 42
掌舵人 6
掌門 70
掌門人 39
掌骨 8
掍 349
掎 7
掏 1000
掏出 106
掏出來 6
掏到 0
掏槽 0
掏空 111
掏耳朵 2
掏耳朵 2
掏腰包 18
掏錢 18
掐 82
掐住 6
掐到 1
掐喉嚨 0
掐指一算 0
掐斷 1
掐死 11
掐算 0
掐緊 0
掐花 0
掐著 4
掐頭去尾 0
掑 532
排 4824
排上 24
排上來 0
排上去 0
排下 6
排下來 2
排下去 1
排他 17
排他性 23
排來排去 1
排便 110
排偶 2
排入 120
排出 488
排出來 7
排出去 5
排列 1822
排列成 16
排列組合 53
排到 39
排印 23
排卵 212
排卵期 12
排去 5
排名 2265
排名順序 15
排圍而入 0
排在 1100
排場 29
排場 29
排外 15
排好 30
排字 8
排定 281
排尾 0
排尿 105
排尿作用 0
排尿功能 1
排山倒海 39
排山倒海之勢 2
排平 0
排序 1998
排序功能 5
排座 4
排座位 6
排律 1
排得 18
排得 199
排成 93
排戲 16
排拒 77
排排 45
排排坐 14
排排站 21
排換 1
排擋 27
排擠 141
排擠掉 0
排放 2001
排放系統 11
排放量 332
排散 0
排斥 462
排斥性 4
排架 157
排檔 439
排比 18
排氣 435
排氣管 92
排氣量 612
排水 813
排水口 20
排水溝 120
排水管 82
排水系統 108
排水量 48
排汗 24
排汗作用 0
排汗功能 2
排泄 193
排泄物 94
排泄量 9
排法 8
排滿 14
排演 67
排演場 2
排演場 2
排澇 2
排灌 3
排灣 98
排灣族 212
排版 630
排版系統 60
排班 111
排球 624
排球場 38
排球場 38
排球網 12
排球賽 62
排球隊 67
排的 199
排程 1
排笛 9
排筆 0
排簫 0
排糞 7
排糞作用 0
排糞功能 0
排給 3
排線 0
排練 80
排著 4
排行 3881
排行榜 4347
排表 3
排解 52
排解糾紛 2
排設 10
排課 123
排走 0
排起 3
排起來 0
排送 4
排遣 28
排遺 0
排量 0
排長 40
排開 42
排除 1793
排除異己 17
排除萬難 36
排隊 309
排難解紛 2
排頭 4
排頭 4
排頭兒 0
排頭兒 0
排頭兒 0
排頭兒 0
排骨 183
排骨飯 36
排骨麵 7
排點 4
掓 0
掔 3
掕 0
掕 0
掖 283
掖縣 2
掗 20
掘 1970
掘出 22
掘出來 1
掘去 0
掘地 4
掘起 22
掘起來 0
掘進 5
掘金 2
掙 156
掙 156
掙出 2
掙到 1
掙命 2
掙扎 400
掙扎著 36
掙揣 0
掙脫 106
掙起 0
掙錢 3
掙開 6
掙飯吃 0
掚 0
掛 2194
掛一掛 0
掛一漏萬 8
掛上 198
掛上來 2
掛上去 5
掛下 3
掛下來 0
掛下去 0
掛不上 29
掛不下 0
掛不住 10
掛了 255
掛住 13
掛來 0
掛來掛去 0
掛個 9
掛入 9
掛冠 0
掛冠 0
掛冠求去 2
掛冠而去 0
掛出 13
掛出來 1
掛出去 0
掛到 19
掛勾 89
掛名 41
掛名夫妻 0
掛單 15
掛回 2
掛回來 0
掛回去 0
掛圖 81
掛圖表 0
掛在 295
掛在口上 1
掛失 235
掛好 7
掛孝 0
掛帘 0
掛帥 109
掛帳 7
掛彩 21
掛得 2
掛得 45
掛心 24
掛念 51
掛意 2
掛慮 18
掛懷 3
掛成 1
掛掉 81
掛搭 0
掛斷 39
掛有 23
掛架 7
掛滿 29
掛漏 0
掛火 0
掛燈 0
掛牌 389
掛的 46
掛礙 0
掛籃 1
掛紅 6
掛羊頭 2
掛羊頭賣狗肉 18
掛著 140
掛號 6777
掛號信 20
掛號處 9
掛號費 62
掛衣 0
掛衣架 3
掛衣鉤 0
掛記 2
掛記著 1
掛賬 0
掛起 18
掛起來 1
掛車 0
掛軸 4
掛進 6
掛進來 0
掛進去 0
掛過 5
掛過來 0
掛過去 0
掛鉤 70
掛鎖 19
掛鐘 23
掛零 14
掛面 2
掛麵 2
掛齒 2
掜 13
掝 35
掞 13
掟 6
掠 461
掠取 5
掠地 2
掠奪 128
掠奪者 8
掠殺 1
掠美 1
掠走 2
掠過 79
採 7354
採下 15
採下來 1
採下去 0
採伐 38
採住 2
採信 28
採光 230
採到 55
採制 0
採區 17
採去 0
採取 3420
採回 9
採回來 3
採回去 0
採在 2
採好 2
採得 24
採得 31
採掘 23
採採 2
採摘 33
採擇 7
採擷 27
採料 0
採暖 1
採桑 5
採樣 841
採樣法 1
採油 4
採為 14
採煤 7
採煤礦 0
採珠 1
採用 5752
採用到 0
採的 31
採石場 8
採石場 8
採石廠 0
採砂場 1
採砂場 1
採礦 133
採礦工 4
採礦工業 0
採礦工程 9
採礦方法 0
採礦業 8
採種 23
採空區 0
採納 96
採給 2
採花 27
採茶 140
採藥 13
採行 609
採訪 1356
採訪員 7
採訪新聞 5
採訪編輯 16
採訪記者 66
採買 40
採購 7249
採購供應 0
採購供應站 0
採購員 2
採購團 3
採購站 3
採購規則 0
採辦 8
採過 7
採過來 0
採過去 0
採邑 0
採集 450
採集法 1
採集起來 0
採食 30
探 2074
探 325
探了 3
探來探去 0
探來探去 0
探信 0
探傷 28
探入 12
探入 12
探出 51
探出 51
探出來 0
探出去 1
探勘 265
探勘者 3
探友 1
探取 7
探口氣 0
探口風 0
探員 46
探問 22
探喪 0
探囊取物 3
探子 22
探察 7
探尋 99
探尋出 1
探幽訪勝 0
探底 63
探悉 4
探戈 39
探戈舞 1
探戡 0
探探 20
探明 9
探明儲量 3
探春 21
探望 127
探查 58
探案 43
探求 129
探求民隱 0
探測 308
探測器 111
探測船 20
探源 67
探溯 1
探照燈 21
探獲 0
探班 1
探病 47
探監 5
探看 22
探知 52
探礦 6
探礦者 0
探究 533
探究出 3
探索 2727
探索之旅 15
探聽 32
探聽出 2
探聽到 2
探花 15
探著 1
探見 0
探視 136
探親 151
探親假 3
探討 5976
探訪 286
探詢 48
探詢出 0
探試 3
探起 0
探路 88
探身 3
探身 3
探身一看 0
探身一看 0
探過 6
探長 280
探險 427
探險家 162
探險隊 243
探頭 79
探頭探腦 7
掣 138
掣後腿 0
掣肘 18
掣電 1
掤 5
接 19424
接上 463
接上來 1
接上去 8
接下 622
接下來 1090
接下去 35
接了 87
接二連三 63
接人 11
接任 547
接住 48
接來 10
接來接去 1
接個 7
接到 841
接力 190
接力棒 0
接力比賽 4
接力賽 42
接力賽跑 4
接去 10
接受 9800
接受下來 0
接受到 38
接受器 49
接受者 29
接口 86
接合 210
接合上 1
接合處 32
接合起來 1
接合面 4
接合點 2
接吻 89
接回 50
接回來 4
接回去 2
接在 113
接地 169
接地線 13
接境 0
接壤 40
接好 18
接完 10
接客 16
接引 57
接彈 1
接往 69
接待 585
接待員 18
接待室 38
接待所 0
接待站 0
接得 25
接得 181
接應 52
接手 397
接招 48
接掌 221
接接 7
接接看 1
接收 1700
接收機 93
接收者 43
接替 101
接有 14
接木 1
接木法 1
接枝 38
接棒 69
接棒人 7
接機 123
接殺 8
接洽 257
接洽人 8
接滿 2
接演 14
接濟 11
接片 0
接片機 0
接物 10
接獲 642
接班 104
接班人 100
接球 69
接球員 19
接球法 1
接生 47
接生婆 6
接生術 0
接生費 0
接界 1
接的 181
接神 0
接種 663
接穗 7
接管 253
接管人 1
接納 402
接給 2
接線 213
接線員 0
接線柱 0
接線生 18
接縫 87
接續 1313
接續詞 0
接羔 0
接著 2514
接著就是 23
接著是 102
接見 973
接觸 3057
接觸到 292
接觸力 2
接觸器 15
接觸應力 1
接觸法 4
接觸面 24
接觸點 9
接話 2
接談 5
接財神 0
接走 9
接起 17
接起來 17
接踵 15
接踵而來 36
接踵而至 31
接軌 0
接辦 63
接迎 5
接近 2024
接近到 7
接近於 23
接送 367
接通 122
接連 227
接連不斷 6
接連到 0
接連起來 1
接進 20
接進來 4
接運 19
接過 98
接過來 9
接鄰 3
接防 1
接防部隊 0
接電 29
接電話 113
接面 26
接頭 778
接頭接耳 0
接風 9
接駁 79
接駁車 1
接駕 6
接骨 80
接骨師 3
接點 187
接龍 68
掦 0
控 6780
控制 11261
控制住 32
控制力 28
控制台 3410
控制器 854
控制塔 3
控制室 40
控制性 5
控制板 62
控制棒 11
控制權 61
控制碼 1
控制站 5
控制裕如 0
控制鍵 14
控告 497
控告人 1
控球 68
控管 2
控罪 2
控股 0
控訴 312
控訴狀 1
控詞 0
推 5407
推三推四 0
推三阻四 3
推上 55
推上來 0
推上去 2
推下 11
推下來 1
推下去 6
推不開 1
推事 16
推人 8
推人犯規 0
推介 0
推估 377
推使 4
推來 2
推來推去 5
推倒 83
推入 46
推出 9299
推出來 9
推出去 2
推到 69
推力 92
推動 10315
推動力 13
推卸 31
推卸責任 23
推卻 3
推向 123
推問 0
推回 12
推回來 1
推回去 7
推土 3
推土機 22
推壓 14
推子 1
推定 113
推定出來 0
推尊 0
推導 84
推展 2046
推展性 0
推崇 297
推己及人 13
推度 0
推廣 8058
推廣先進 0
推廣先進經驗 0
推廣性 8
推延 1
推得 22
推得 36
推心置腹 8
推想 39
推想出 1
推想到 0
推戴 2
推手 114
推托 5
推托之詞 1
推拏 0
推拒 8
推拖 11
推拿 134
推拿法 0
推挽 3
推掉 6
推推 6
推推拉拉 1
推擋 2
推擠 51
推故 1
推敲 67
推斥 3
推斷 109
推斷力 0
推本溯源 0
推桿 39
推求 10
推波助瀾 59
推波助瀾 59
推派 81
推派出 1
推測 382
推演 82
推演出來 5
推為 16
推牌九 0
推理 420
推理小說 56
推理性 0
推甄 0
推病 3
推的 36
推磨 6
推移 52
推究 8
推算 287
推算出來 3
推給 72
推翻 242
推背圖 0
推脫 1
推舉 123
推荐 314
推著 45
推薦 17466
推薦信 103
推薦函 133
推行 1684
推託 13
推誠 0
推說 22
推諉 48
推請 2
推論 333
推論起來 0
推讓 4
推走 0
推車 70
推轂 0
推辭 28
推送 22
推進 327
推進來 2
推進力 10
推進去 2
推進器 33
推進機 2
推過 1
推過來 0
推過去 1
推遲 33
推選 190
推重 6
推鉋 0
推鉛球 3
推銷 388
推銷員 92
推銷術 5
推開 135
推陳出新 125
掩 721
掩上 11
掩不住 28
掩人耳目 10
掩住 27
掩口而笑 0
掩埋 341
掩埋場 52
掩埋場 520
掩旗息鼓 0
掩映 24
掩映著 1
掩沒 10
掩耳 5
掩耳盜鈴 5
掩著 11
掩蓋 116
掩蓋住 2
掩蓋物 0
掩蓋著 4
掩蔽 20
掩蔽物 1
掩藏 17
掩藏不住 0
掩藏著 1
掩護 110
掩護著 0
掩門 2
掩門而去 0
掩面 21
掩面而泣 2
掩面而過 0
掩飾 241
掩體 7
掩鼻 4
掩鼻而過 1
措 631
措大 1
措手 5
措手不及 50
措施 5536
措款 2
措置 8
措置失當 1
措置得當 0
措詞 25
措詞不當 1
措辭 30
措辭不當 2
掫 33
掬 96
掬水 41
掬誠 1
掭 20
掮 11
掮客 29
掯 5
掰 711
掰開 1
掱 3
掲 0
掲 0
掲 0
掺 0
掺 0
掺 0
掺 0
掽 21
掾 12
掿 0
掿 0
掿 0
揀 131
揀出 2
揀去 0
揀選 124
揀開 0
揁 0
揂 5
揃 6
揄 7
揅 4
揆 891
揆度 1
揆度 1
揇 9
揈 11
揉 1053
揉到 0
揉合 45
揉成 25
揉成一團 1
揉成一團 1
揉捻 12
揉揉 16
揉揉看 0
揉搓 12
揉碎 2
揉著 15
揉雜 0
揊 6
揋 3
揌 5
揍 93
揍了 14
揍人 3
揍得 1
揍過 1
揎 11
揎拳捋袖 0
揎拳擄袖 1
揎拳裸袖 1
描 380
描一描 0
描一描 0
描上 1
描了 1
描來描去 0
描出 6
描出來 0
描到 0
描圖 11
描圖紙 14
描塗 0
描好 0
描字 0
描寫 478
描寫出 2
描寫出來 3
描成 0
描抹 0
描描 3
描摹 11
描畫 11
描紅 2
描繪 430
描繪下來 3
描繪出 84
描繪出來 8
描繪成 6
描花 1
描著 1
描起 0
描起來 0
描述 2004
提 1
提 119
提 11963
提上 11
提上來 2
提上去 0
提下 68
提下來 0
提下去 0
提不出 39
提不到 0
提不起 42
提不起來 0
提了 69
提交 262
提來 1
提來提去 0
提供 78537
提供給 956
提個 2
提倡 515
提價 0
提出 10782
提出來 152
提出去 0
提到 2479
提制 2
提前 1133
提前來 3
提前到 5
提前去 0
提包 15
提升 4499
提升到 90
提去 3
提及 545
提取 51
提名 1728
提名者 7
提向 3
提問 96
提單 34
提回 3
提回來 0
提回去 1
提在 15
提在手上 0
提報 472
提壺 4
提好 2
提存 216
提審 21
提得 8
提得 8
提心 12
提心吊膽 29
提成 5
提手 1
提手旁 0
提拉米蘇 1
提拔 49
提拿 2
提拿著 0
提挈 1
提振 0
提掖 0
提掖 0
提提 13
提提神 0
提撥 0
提攜 105
提早 617
提早來 1
提早到 14
提早去 5
提昇 4643
提案 1394
提案人 86
提桶 6
提梁 4
提槍 2
提款 254
提款單 2
提款權 2
提法 56
提溜 0
提煉 124
提煉出 16
提燈會 0
提琴 111
提琴家 580
提盒 3
提督 48
提示 1314
提示號 13
提神 60
提筆 42
提箱 8
提籃 10
提純 1
提純復壯 0
提純複壯 0
提綱 15
提綱挈領 10
提綱振領 0
提腕 0
提花 22
提著 119
提袋 14
提要 695
提親 20
提訊 8
提訴 1
提詞 2
提調 5
提請 268
提議 507
提貨 77
提貨單 16
提走 1
提起 728
提起來 8
提起公訴 47
提轄官 0
提辭 1
提送 143
提送到 0
提進 0
提進來 0
提進去 1
提過 157
提過來 0
提過去 1
提醒 1678
提防 126
提領 182
提頭 2
提高 5809
提高警惕 5
揑 0
插 428
插一腳 18
插一腳 18
插上 92
插上來 0
插上去 1
插下 4
插下來 0
插下去 2
插住 0
插入 921
插入法 3
插入物 0
插到 26
插去 2
插口 19
插句 3
插嘴 28
插嚅 0
插回 8
插回來 0
插回去 1
插圖 294
插圖卷 0
插圖者 0
插在 133
插字 0
插對 0
插座 348
插手 69
插接 25
插曲 180
插枝 6
插枝法 3
插條 2
插槽 1
插標 2
插班 37
插班生 19
插畫 211
插科打諢 5
插秧 18
插秧機 4
插線程序計算機 0
插翅 13
插翅難飛 1
插腰 6
插花 1929
插苗 2
插著 48
插話 54
插起 3
插起來 0
插足 7
插足於 0
插身 7
插進 35
插進來 6
插進去 12
插金帶銀 1
插金披綺 1
插鉛 0
插銷 8
插錯 2
插隊 34
插頁 8
插頭 196
揓 4
揔 0
揕 5
揖 809
揖拜 0
揖謝 0
揖讓 4
揗 9
揘 4
揙 6
揚 15519
揚名 79
揚名世界 4
揚名全世界 1
揚名天下 1
揚名立萬 9
揚善 12
揚善去惡 0
揚善隱惡 0
揚塵 13
揚威 45
揚子江 11
揚子津 1
揚子縣 1
揚州 117
揚州人 1
揚帆 41
揚手 15
揚揚 11
揚揚得意 0
揚揚自得 0
揚棄 87
揚湯止沸 2
揚琴 38
揚眉吐氣 36
揚程 46
揚聲 13
揚聲器 198
揚聲機 0
揚花 2
揚言 145
揚起 85
揚長而去 29
揚鞭 17
換 5972
換上 184
換上來 0
換上去 1
換下 34
換下來 3
換下去 2
換乘 6
換了 340
換交 0
換人 188
換位 15
換來 153
換來換去 10
換個 257
換借 0
換停 0
換傘 2
換兵 0
換出 13
換出來 0
換出去 0
換到 64
換割 0
換包 0
換去 3
換取 250
換口 0
換口味 2
換口氣 1
換句 4
換句話 13
換句話說 339
換叫 4
換台 7
換吊 0
換向 7
換向器 2
換吸 0
換唱 1
換回 35
換回來 3
換回去 1
換圖 12
換土 3
換報 0
換場 7
換場 7
換填 1
換姓 0
換季 36
換學 3
換守 0
換對 5
換屋 78
換工 62
換帖 12
換廠 0
換張 10
換得 96
換得 41
換心 14
換心手術 0
換想 1
換成 349
換手 339
換扣 0
換批 0
換抄 0
換披 0
換抱 0
換拍 0
換掉 98
換排 2
換接 2
換換 38
換換氣 0
換摘 0
換支 2
換收 0
換教 9
換教 9
換文 10
換新 336
換景 2
換有 8
換服 0
換板 4
換枝 0
換格 0
換機 73
換檔 105
換歌 0
換步 1
換毛 5
換氣 126
換水 73
換法 1
換洗 40
換派 0
換流 12
換湯 5
換湯不換藥 11
換滿 1
換炒 0
換為 25
換熱器 22
換燈 2
換班 11
換球 7
換用 17
換田 0
換發 320
換的 41
換皮 0
換租 0
換稱 0
換算 361
換算表 38
換管 3
換箱 0
換節 3
換簽 0
換籤 0
換系 9
換約 26
換紙 1
換組 2
換給 14
換編 0
換織 0
換羽 1
換聽 0
換胎 12
換腎 26
換腎手術 1
換膚 64
換航 0
換茬 0
換著 10
換藥 33
換血 56
換血 56
換行 270
換衣服 26
換補 10
換裝 153
換角 2
換言之 359
換調 5
換請 2
換證 33
換讀 1
換起 2
換起來 0
換車 82
換車票 0
換速 3
換過 53
換過來 8
換過去 0
換邊 8
換部 2
換釣 0
換鈔 0
換鈔機 0
換錢 11
換關 1
換隊 2
換隻 1
換零錢 1
換靶 0
換項 0
換飛 0
換騎 2
換髮 0
揜 7
揝 6
揞 0
揟 13
揟 13
揠 5
揠苗助長 12
握 1955
握下 2
握下去 0
握住 159
握別 3
握力 14
握力器 0
握取 2
握壽司 1
握得 7
握得 14
握手 162
握手禮 0
握手言歡 2
握手道別 2
握把 1
握拳 18
握持 9
握有 145
握權 0
握的 14
握筆 7
握緊 61
握股公司 1
握著 160
揢 0
揣 13695
揣度 3
揣情 37
揣摩 46
揣測 105
揣測之詞 0
揣量 0
揤 142
揤 142
揥 3
揦 0
揧 34
揨 3
揩 14
揩 14
揩油 0
揩油 0
揪 313
揪住 15
揪出 48
揪出來 8
揪出去 0
揪起來 1
揫 5
揬 0
揭 1618
揭下 0
揭人隱私 1
揭出 3
揭去 4
揭封 0
揭幕 272
揭幕儀式 7
揭幕典禮 5
揭幕式 7
揭底 0
揭批 0
揭掉 0
揭揚 0
揭明 1
揭曉 488
揭榜 0
揭櫫 79
揭瘡疤 2
揭發 178
揭發出 3
揭短 1
揭破 8
揭示 242
揭示板 22
揭穿 47
揭竿 3
揭竿而起 7
揭竿起義 6
揭諦 1
揭起 9
揭開 366
揭開序幕 58
揭露 383
揭露 383
揭露出 8
揭露出 8
揮 5464
揮之不去 62
揮之即去 1
揮了 20
揮兵 2
揮刀 36
揮別 125
揮劍 31
揮動 58
揮師 2
揮戈 8
揮手 122
揮手告別 3
揮手致意 2
揮打 5
揮拳 11
揮揮手 22
揮桿 83
揮棒 25
揮棒落空 5
揮毫 57
揮汗 19
揮汗如雨 11
揮汗成雨 0
揮涕 0
揮淚 5
揮灑 0
揮發 101
揮發性 379
揮發油 11
揮發物 12
揮筆 7
揮舞 79
揮著 35
揮起 8
揮軍 36
揮金如土 1
揮霍 51
揮霍無度 6
揮鞭 6
揯 5
揰 7
揱 7
揲 3
揳 3
揳 3
援 1503
援交 10
援例 15
援兵 8
援助 599
援助交際 0
援外 53
援引 72
援手 152
援救 38
援救隊 0
援用 53
援筆 1
援軍 19
揵 12
揶 5
揶揄 10
揷 0
揸 0
揹 168
揹包袱 0
揹負 19
揹黑鍋 6
揺 0
揾 0
搆 64
搆不著 3
搆怨 0
搆陷 0
搇 0
搇 0
搈 0
搉 5
搊 57
搋 42
搌 41
損 1137
損人 18
損人不利己 8
損人利己 7
損人害己 0
損人益己 0
損傷 580
損傷到 1
損友 37
損壞 692
損壞率 7
損失 3708
損失率 12
損害 928
損害到 12
損害賠償 297
損毀 169
損益 283
損益相抵 0
損益表 115
損耗 76
損耗率 8
損耗量 1
搎 51
搏 308
搏得 8
搏戰 10
搏手 0
搏擊 15
搏鬥 107
搐 9
搐 9
搑 0
搑 0
搑 0
搑 0
搒 2
搓 173
搓到 1
搓成 4
搓手 2
搓揉 29
搓板 0
搓過 0
搓過來 0
搓過去 0
搔 90
搔到癢處 2
搔手 0
搔擾 2
搔癢 96
搔頭 8
搔頭摸耳 0
搔首 2
搔首弄姿 2
搔首弄姿 2
搕 73
搖 641
搖了 60
搖了一下 1
搖到 8
搖動 93
搖動不停 0
搖唇鼓舌 1
搖好 0
搖尾 3
搖尾 3
搖尾乞憐 1
搖尾巴 4
搖得 6
搖得 11
搖扇 2
搖手 11
搖把 1
搖搖 36
搖搖冰 0
搖搖手 8
搖搖擺擺 10
搖搖晃晃 35
搖搖欲墜 28
搖搖蕩蕩 0
搖搖頭 105
搖撼 15
搖擺 183
搖擺不定 23
搖擺舞 4
搖旗 11
搖旗吶喊 21
搖晃 141
搖晃不停 0
搖曳 78
搖曳生姿 7
搖桿 469
搖椅 20
搖槳 2
搖滾 486
搖滾樂 111
搖滾樂團 87
搖滾樂手 3
搖滾歌手 9
搖獎 0
搖琴 1
搖的 12
搖盪 5
搖盪不停 0
搖籃 143
搖籃曲 31
搖脣鼓舌 0
搖臂 20
搖臂 20
搖船 4
搖著 28
搖著頭 23
搖蕩 7
搖起 4
搖起來 0
搖身 13
搖身一變 75
搖車 0
搖鈴 7
搖錢樹 35
搖鑽 3
搖頭 269
搖頭嘆息 15
搖頭擺尾 6
搖頭晃腦 18
搖點 0
搗 74
搗亂 16
搗亂份子 0
搗亂份子 0
搗亂分子 0
搗毀 17
搗爛 4
搗碎 15
搗米 3
搗蛋 81
搗蛋鬼 9
搗鬼 4
搘 37
搙 0
搙 0
搙 0
搚 35
搛 36
搜 1410
搜出 29
搜刮 21
搜到 8
搜尋 23684
搜尋引擎 0
搜山 0
搜括 19
搜括一空 0
搜捕 36
搜捕到 0
搜救 97
搜查 67
搜查證 0
搜求 0
搜索 1205
搜索前進 0
搜索隊 4
搜羅 41
搜訊 0
搜購 8
搜身 6
搜集 576
搝 0
搞 1339
搞不好 203
搞亂 15
搞了 38
搞些 5
搞什麼 58
搞什麼 58
搞來 1
搞來搞去 2
搞出 26
搞出來 4
搞在 11
搞垮 8
搞壞 11
搞好 19
搞好關係 0
搞定 157
搞得 142
搞怪 100
搞懂 11
搞成 19
搞把戲 0
搞掉 1
搞清 6
搞清楚 1021
搞的 141
搞砸 22
搞笑 0
搞糊塗 5
搞糟 1
搞起 10
搞起來 1
搞通 2
搞進 0
搞進來 0
搞進去 0
搞過 1
搞過來 0
搞過去 0
搞錢 3
搞錯 99
搞雜 1
搞頭 54
搞鬼 39
搟 117
搠 8
搡 52
搢 111
搢紳 1
搣 30
搤 17
搥 56
搥打 12
搥背 5
搥胸 0
搥胸頓足 6
搥鼓 0
搦 29
搦管 0
搧 124
搧動 4
搧耳光 0
搧開 0
搧風 4
搨 52
搨本 0
搨碑 0
搩 0
搪 84
搪塞 23
搪瓷 86
搪缸 5
搫 34
搬 5524
搬上 622
搬上來 0
搬上去 1
搬下 1
搬下來 3
搬下去 0
搬來搬去 0
搬出 68
搬出來 12
搬出去 7
搬到 275
搬動 33
搬回 43
搬回來 4
搬回去 2
搬家 1096
搬弄 5
搬弄是非 4
搬得 5
搬成 0
搬掉 1
搬用 0
搬的 5
搬磚砸腳 0
搬移 86
搬移到 25
搬給 0
搬著 2
搬走 35
搬起 1
搬起來 0
搬進 49
搬運 517
搬運法 3
搬運費 2
搬遷 234
搬開 8
搭 1940
搭上 177
搭上來 0
搭上去 0
搭乘 1216
搭伙 2
搭伴 0
搭住 2
搭來搭去 0
搭個 13
搭公車 49
搭出 3
搭售 0
搭在 29
搭夥 0
搭好 4
搭客 12
搭帳棚 1
搭帳篷 3
搭建 99
搭成 7
搭拉 1
搭撒 0
搭擋 25
搭救 18
搭棚 8
搭機 355
搭機來 2
搭機到 5
搭機去 0
搭機回 4
搭檔 169
搭線 3
搭腔 4
搭船 70
搭街坊 0
搭訕 34
搭訕者 0
搭話 1
搭起 87
搭車 127
搭車來 0
搭車到 6
搭車去 3
搭載 144
搭造 4
搭進 1
搭進來 0
搭進去 0
搭配 1858
搭配在 11
搭配著 19
搭錯 13
搭靠 0
搭靠在 0
搮 47
搯 44
搯出來 1
搯耳朵 1
搯耳朵 0
搰 37
搱 0
搱 0
搲 0
搳 48
搳 481
搴 5
搵 34
搶 20000
搶 3043
搶 351
搶上風 0
搶下 40
搶下來 0
搶下去 0
搶了 43
搶人 24
搶佔 71
搶來 11
搶來搶去 1
搶修 106
搶修工作 0
搶先 284
搶先一步 25
搶光 7
搶到 56
搶劫 198
搶劫犯 0
搶匪 0
搶占 13
搶去 15
搶嘴 0
搶墑 0
搶奪 171
搶奪到 0
搶奪罪 0
搶嫌 5
搶孤 31
搶得 46
搶得 29
搶手 89
搶手貨 15
搶掠 5
搶收 12
搶攻 0
搶救 1035
搶救人質 2
搶救出來 4
搶救工作 10
搶案 103
搶法 0
搶灘 114
搶球 6
搶白 5
搶的 29
搶眼 53
搶種 2
搶著 0
搶親 2
搶購 145
搶購一空 11
搶走 116
搶跑 0
搶運 0
搶過 6
搶過來 4
搶過去 1
搶鏡頭 5
搶險 12
搷 44
搸 0
搹 38
携 0
携 0
搻 0
搻 0
搻 0
搼 0
搽 28
搽 28
搽粉 0
搽著 0
搽藥 0
搾 81
搾取 3
搾渣 0
搾甘蔗 0
搿 44
摀 50
摀住 14
摀蓋 0
摁 16
摃 54
摆 0
摇 0
摋 8
摌 0
摌 0
摍 3
摎 34
摏 0
摐 19
摑 12
摑耳光 1
摒 1023
摒出 0
摒擋 0
摒棄 83
摒除 64
摓 7
摔 283
摔下 29
摔下來 14
摔下去 10
摔不掉 0
摔不掉 0
摔了 13
摔了一跤 2
摔了一跤 2
摔交 1
摔倒 46
摔傷 16
摔到 32
摔壞 5
摔成 5
摔成兩半 0
摔掉 9
摔斷 13
摔斷了 4
摔斷了腿 0
摔死 23
摔破 6
摔碎 5
摔筋斗 0
摔角 213
摔跟斗 0
摔跟頭 0
摔跤 21
摔開 2
摕 0
摕 0
摖 0
摖 0
摗 0
摗 0
摘 126
摘 1268
摘下 101
摘下來 9
摘下去 0
摘來 2
摘來摘去 0
摘出 7
摘出來 0
摘出去 0
摘到 0
摘去 7
摘取 41
摘句 0
摘引 0
摘得 1
摘得 8
摘心 6
摘的 8
摘自 673
摘花 3
摘要 5856
摘要性 7
摘記 20
摘譯 129
摘起 0
摘起來 0
摘選 7
摘選自 2
摘錄 444
摘錄 444
摘錄自 193
摘除 82
摘食 0
摙 12
摚 0
摛 9
摜 40
摝 3
摞 12
摟 204
摟 204
摟住 13
摟抱 17
摟摟 2
摟摟抱抱 6
摟錢 0
摠 16
摡 13
摡 13
摢 0
摢 0
摣 0
摤 0
摤 0
摤 0
摥 10
摦 10
摧 123
摧折 7
摧枯拉朽 3
摧殘 75
摧毀 230
摧毀力 2
摧毀性 2
摧花 4
摨 141
摩 23030
摩加迪沙 0
摩厲以須 0
摩呼羅迦 1
摩天 56
摩天大樓 41
摩天嶺 2
摩天樓 24
摩天輪 13
摩弄 1
摩托 436
摩托化 0
摩托船 0
摩托車 330
摩拖車 15
摩拳擦掌 23
摩挲 3
摩搓 2
摩擦 360
摩擦係數 33
摩擦力 71
摩擦阻力 8
摩擦音 2
摩斯電碼 1
摩洛哥 110
摩爾 117
摩爾定律 0
摩登 275
摩納哥 37
摩羯 1
摩羯座 1
摩耶夫人 1
摩肩 2
摩肩接踵 9
摩肩擦踵 2
摩西 233
摩訶 1
摩訶止觀 1
摩訶羅 1
摩訶般若波羅蜜 1
摩訶般若波羅蜜 1
摩訶般若波羅蜜多 1
摩訶般若波羅蜜多 1
摩訶薩 1
摩訶迦葉 1
摩門教 38
摩門教徒 1
摩門教會 0
摩頂 1
摩頂放踵 0
摪 0
摪 0
摫 32
摬 2
摭 11
摭取 0
摭拾遺文 0
摭採 0
摮 169
摯 496
摯友 43
摯情 4
摯愛 78
摯誠 7
摰 223
摱 0
摱 0
摲 10
摳 78
摳出 0
摳出來 0
摳到 2
摳掉 0
摳摟 0
摳破 0
摳門 6
摴 6
摵 10
摶 31
摷 14
摸 692
摸上 5
摸上來 0
摸上去 0
摸下 2
摸下來 0
摸下去 1
摸不到 22
摸不清 16
摸不著 30
摸不著邊 1
摸不著邊際 0
摸不透 1
摸中 0
摸了 49
摸來摸去 2
摸出 25
摸出來 0
摸出去 0
摸到 72
摸去 0
摸哨 1
摸底 0
摸彩 118
摸彩品 5
摸得 21
摸成 0
摸手 0
摸掉 1
摸摸 120
摸清 10
摸清底細 1
摸清楚 8
摸熟 2
摸破 0
摸稜 0
摸索 224
摸索到 2
摸腳 0
摸臉 2
摸觸 1
摸起 1
摸起來 29
摸透 4
摸進 1
摸進來 0
摸進去 0
摸過 18
摸錯 1
摸骨 3
摸骨相 0
摸魚 63
摸黑 29
摹 264
摹仿 29
摹倣 3
摹刻 0
摹印 0
摹古 1
摹寫 18
摹拜 0
摹擬 35
摹本 2
摹畫 3
摺 495
摺合 1
摺奏 0
摺子 1
摺尺 1
摺扇 9
摺梯 1
摺椅 0
摺疊 71
摺疊扇 0
摺疊為 0
摺疊起來 0
摺痕 23
摺篷 0
摺紙 95
摺紙工 0
摺裙 0
摻 2180
摻 218
摻 218
摻 218
摻合 60
摻雜 50
摼 0
摼 0
摽 282
摽 282
摾 0
摾 0
摾 0
摿 5
撀 0
撁 0
撂 25
撂 25
撂下 11
撂下來 0
撂倒 5
撂奪 0
撂手 0
撃 0
撅 72
撆 0
撆 0
撇 107
撇 107
撇下 24
撇刀 0
撇去 1
撇嘴 1
撇在 8
撇掉 3
撇撇 0
撇撇嘴 1
撇棄 1
撇清 47
撇開 73
撇開不談 0
撈 492
撈 492
撈一票 11
撈一票 11
撈到 12
撈去 0
撈取 6
撈本 1
撈起 44
撈起來 5
撈魚 6
撉 167
撊 82
撋 251
撌 96
撍 0
撍 0
撍 0
撎 0
撏 38
撐 590
撐下去 15
撐不下去 14
撐不住 18
撐了 11
撐住 18
撐傘 16
撐到 31
撐反 0
撐在 11
撐場面 7
撐場面 7
撐好 6
撐得 22
撐持 8
撐竿 5
撐竿跳 9
撐竿跳高 7
撐腰 29
撐臂 2
撐臂 2
撐船 3
撐著 60
撐起 49
撐起來 1
撐過 15
撐過來 0
撐過去 2
撐門面 0
撐開 30
撐開傘 1
撑 0
撒 1140
撒 1140
撒下 57
撒下來 0
撒下去 0
撒但 192
撒佈 13
撒來 1
撒來撒去 0
撒克遜 1
撒克遜 1
撒克遜人 0
撒克遜人 0
撒哈拉 13
撒哈拉沙漠 19
撒囈症 0
撒嬌 58
撒嬌 58
撒嬌撒痴 1
撒嬌賣俏 1
撒尿 12
撒尿 12
撒手 15
撒手不管 6
撒手人寰 2
撒手西歸 2
撒拉族 2
撒播 10
撒施 1
撒旦 89
撒水 124
撒種 24
撒網 12
撒網 12
撒網捕魚 3
撒網捕魚 3
撒線 0
撒腿 2
撒落 6
撒謊 48
撒賴 3
撒酒瘋 0
撒野 31
撒開 2
撒開 2
撒開來 0
撒開來 0
撒鴨子 1
撒鹽 1
撓 417
撓度 18
撓曲 15
撓鉤 0
撓鉤手 0
撔 0
撕 235
撕 235
撕下 51
撕下來 9
撕下去 0
撕了 7
撕來 0
撕來撕去 0
撕去 19
撕壞 1
撕得 5
撕成 13
撕打 1
撕掉 14
撕毀 36
撕爛 0
撕破 7
撕破臉 4
撕碎 8
撕票 56
撕票員 0
撕裂 140
撕走 0
撕起 3
撕起來 0
撕過 0
撕過來 0
撕過去 0
撕開 26
撖 27
撗 110
撘 89
撙 123
撙省 0
撙節 169
撚 256
撚指 0
撚指之間 0
撚指間 2
撚搓 0
撚斷 1
撚線 4
撚鬚 0
撛 0
撜 207
撝 3
撞 97
撞 973
撞上 128
撞人 12
撞個滿懷 0
撞倒 38
撞傷 34
撞入 6
撞出 17
撞出來 1
撞出去 1
撞到 107
撞在 29
撞壞 15
撞得 21
撞成 11
撞打 2
撞擊 351
撞擊到 7
撞擊在 4
撞期 1
撞死 52
撞牆 52
撞球 645
撞球場 46
撞球場 46
撞球桿 9
撞球檯 11
撞的 21
撞見 37
撞起 1
撞起來 0
撞車 38
撞進 14
撞針 6
撞鐘 1
撞開 17
撞頭 1
撞騙 1
撟 111
撟 111
撟捷 0
撟捷 0
撟掇 1
撟舌 1
撠 154
撡 0
撡 0
撡 0
撡 0
撡 0
撢 148
撢子 2
撢灰 0
撣 188
撤 425
撤下去 0
撤兵 4
撤出 57
撤出來 0
撤出去 0
撤到 4
撤去 6
撤回 170
撤回來 1
撤回去 0
撤守 6
撤差 0
撤席 0
撤後 1
撤成 0
撤掉 9
撤換 100
撤消 22
撤營 0
撤職 46
撤職查辦 1
撤資 77
撤走 21
撤軍 0
撤退 128
撤銷 1019
撤銷書 0
撤銷記錄 0
撤防 2
撤除 38
撤離 99
撥 5926
撥亂反正 6
撥交 19
撥付 114
撥作 9
撥來 2
撥來撥去 0
撥冗 105
撥出 137
撥出來 3
撥出去 2
撥到 22
撥動 29
撥去 1
撥回 7
撥回來 0
撥回去 0
撥土 0
撥好 0
撥子 1
撥弄 33
撥弄是非 0
撥得 46
撥快 0
撥接 0
撥款 243
撥用 76
撥發 17
撥的 47
撥空 0
撥給 38
撥號 1087
撥轉 9
撥轉到 0
撥通 70
撥開 47
撥雲見日 19
撦 3
撧 0
撨 0
撨 0
撨 0
撩 552
撩 552
撩上 0
撩上來 0
撩亂 12
撩人 21
撩來撩去 0
撩動 5
撩撥 16
撩望 0
撩起 30
撩起來 0
撩逗 1
撩開 2
撪 0
撫 612
撫今追昔 21
撫劍 4
撫卹 717
撫孤 2
撫屍 0
撫屍痛哭 1
撫平 52
撫弄 18
撫心 2
撫恤 26
撫恤金 11
撫愛 1
撫慰 122
撫摩 2
撫摸 129
撫琴 8
撫育 63
撫躬自問 0
撫遠 53
撫面 0
撫面痛哭 0
撫順 46
撫養 83
撫髀興歎 0
撬 91
撬 91
撬上 0
撬上來 0
撬上去 0
撬下 0
撬下來 0
撬下去 0
撬不開 0
撬不開 0
撬了 1
撬來撬去 0
撬出來 0
撬出去 0
撬動 0
撬動 0
撬壞 0
撬壞 0
撬著 0
撬起 1
撬起 1
撬起來 1
撬門 9
撬門 9
撬開 13
撬開 13
播 2096
播 2096
播下 19
播來播去 0
播來播去 0
播出 993
播出 993
播出來 4
播出來 4
播出去 3
播出去 3
播到 15
播到 15
播去 0
播報 0
播報 0
播幅 0
播弄 4
播得 75
播得 75
播揚 3
播放 1916
播放 1916
播放器 0
播放歌曲 4
播放歌曲 4
播映 0
播映 0
播的 76
播的 76
播種 211
播種期 7
播種機 8
播種法 10
播種面積 1
播送 179
播送 179
播送到 1
播送到 1
播遷 31
播音 60
播音 60
播音劇 0
播音劇 0
播音員 12
播音員 12
播音室 8
播音室 8
撮 323
撮 323
撮到 0
撮到 0
撮口呼 0
撮合 101
撮合 101
撮和 1
撮和 1
撮子 0
撮弄 1
撮弄 1
撮要 2
撮要 2
撯 0
撰 872
撰出來 0
撰寫 1512
撰寫人 57
撰序 0
撰擬 45
撰文 188
撰書 18
撰稿 94
撰稿人 96
撰稿員 2
撰著 18
撰述 707
撱 96
撲 1285
撲上 8
撲上來 2
撲上去 2
撲下 1
撲下來 1
撲下去 0
撲了 8
撲來 9
撲來撲去 0
撲倒 2
撲倒在 5
撲克 41
撲克牌 119
撲出 4
撲出來 0
撲出去 0
撲到 8
撲去 10
撲取 0
撲向 18
撲哧 1
撲在 4
撲地 4
撲得 2
撲打 1
撲打在 0
撲救 10
撲朔 9
撲朔迷離 36
撲殺 125
撲滅 204
撲滅蚊蟲 0
撲滿 29
撲的 2
撲空 9
撲簌簌 1
撲粉 3
撲起 1
撲起來 0
撲通 7
撲通一聲 3
撲通一聲 3
撲進 1
撲進來 0
撲進去 0
撲面 3
撲面而來 6
撲鼻 32
撲鼻而來 6
撳 762
撳住 0
撳扭 0
撴 0
撹 0
撹 0
撻 99
撻伐 28
撻伐 28
撼 430
撼動 65
撼天動地 0
撼搖 3
撽 108
撾 12
撿 453
撿了 15
撿便宜 55
撿到 334
撿去 1
撿回 18
撿回來 12
撿回去 0
撿拾 64
撿球 5
撿球員 0
撿破爛 3
撿起 45
撿起來 11
撿過 0
撿過來 0
撿過去 0
撿開 0
擀 0
擁 1041
擁 1041
擁入 26
擁兵 3
擁兵 3
擁兵自衛 0
擁兵自衛 0
擁兵自重 0
擁兵自重 0
擁到 4
擁向 2
擁吻 15
擁吻 15
擁塞 49
擁塞 49
擁塞 49
擁塞 49
擁戴 28
擁戴 28
擁抱 606
擁抱 606
擁抱著 34
擁抱著 34
擁擠 304
擁擠 304
擁擠不堪 8
擁政愛民 0
擁有 7182
擁有 7182
擁有權 14
擁有權 14
擁立 11
擁立 11
擁被而起 0
擁護 120
擁護 120
擁軍優屬 0
擁進 7
擁進來 0
擁進去 0
擂 141
擂 141
擂台 62
擂台 62
擂台賽 47
擂台賽 47
擂缽 0
擂臺 12
擂臺 12
擂茶 1
擂起 0
擂鼓 14
擃 240
擄 191
擄 191
擄人勒贖 44
擄人勒贖 44
擄掠 22
擄獲 55
擅 239
擅作主張 2
擅作威福 0
擅入 3
擅場 5
擅場 5
擅改 5
擅於 64
擅權 2
擅用 20
擅自 430
擅自作主 2
擅自決定 2
擅自行動 2
擅長 380
擅長於 26
擅離職守 5
擆 0
擇 1
擇 2791
擇交 0
擇人而事 0
擇伐 5
擇伐 5
擇偶 29
擇優 0
擇友 93
擇吉 17
擇善固執 25
擇善而從 0
擇席 0
擇捉 1
擇日 0
擇日子 0
擇期 81
擇法 1
擇要 32
擇鄰 4
擈 0
擉 129
擊 1715
擊中 173
擊倒 49
擊傷 19
擊出 191
擊刺 0
擊劍 23
擊向 26
擊垮 16
擊掌 14
擊撞 1
擊敗 374
擊斃 61
擊昏 18
擊板 0
擊殺 26
擊毀 14
擊水 1
擊沈 65
擊沉 36
擊潰 29
擊球 182
擊發 7
擊破 32
擊碎 18
擊穿 0
擊節 4
擊築 0
擊缶 0
擊落 47
擊退 41
擊錯 1
擊鼓 24
擊鼓作樂 0
擊鼓鳴冤 1
擋 82
擋 822
擋一擋 0
擋一擋 0
擋上 2
擋上來 0
擋上去 0
擋下 37
擋下來 3
擋下去 0
擋不住 103
擋了 15
擋人 4
擋住 126
擋來擋去 0
擋到 9
擋回 1
擋回來 0
擋回去 1
擋掉 14
擋板 7
擋泥 3
擋泥板 2
擋眼 0
擋箭牌 9
擋著 10
擋路 24
擋車 2
擋過 0
擋過來 0
擋過去 0
擋開 9
擋雨 7
擋雨板 0
擋風 37
擋風沙 0
擋風玻璃 52
擋駕 1
擌 0
操 24
操 2466
操之過急 10
操作 8570
操作台 15
操作員 156
操作工 38
操作性 29
操作手 6
操作手冊 200
操作法 24
操作系統 77
操作規程 0
操典 6
操刀 24
操切 0
操勞 20
操勞過度 2
操在 15
操場 28
操場 284
操場上 17
操場上 17
操守 114
操心 61
操心思 0
操戈 1
操持 7
操持家務 2
操控 0
操握 0
操握大權 0
操槍 2
操槳 0
操法 5
操演 35
操琴 2
操神 0
操練 185
操練法 0
操縱 278
操縱台 0
操縱器 1
操縱性 5
操縱桿 15
操縱法 0
操縱索 0
操縱自如 1
操舟 10
操舵 8
操舵手 0
操船術 0
操著 17
操行 121
操行成績 168
操觚 0
擎 559
擎天 189
擎天大樓 0
擏 45
擐 214
擑 0
擒 195
擒住 18
擒拿 30
擒捉 3
擒服 2
擒獲 8
擒賊 1
擒賊先擒王 5
擒賊擒王 17
擒賊擒首 1
擓 0
擔 16996
擔 853
擔不起 2
擔仔麵 34
擔任 4559
擔保 1002
擔子 44
擔待 9
擔心 2727
擔憂 248
擔承 5
擔擔麵 2
擔擱 1
擔擱到 0
擔架 21
擔水 0
擔當 165
擔當不起 5
擔當起來 0
擔綱 93
擔著 2
擔誤 16
擔負 158
擔負著 10
擔負起 47
擔負起來 0
擔起 17
擔起來 0
擔風險 0
擔驚 0
擔驚受怕 4
擕 0
擕 0
擖 455
擗 571
擘 299
擘畫 28
擙 60
據 13929
據了解 567
據云 7
據信 0
據傳 78
據傳說 2
據報 136
據守 15
據實 61
據實以報 2
據實相告 0
據悉 172
據有 17
據此 151
據為己有 17
據理 7
據理力爭 21
據稱 91
據說 659
據說在 27
據說是 84
據說有 21
據道 2
據險固守 0
據點 1547
擛 72
擜 0
擟 0
擠 640
擠上 19
擠上來 1
擠上去 0
擠下 12
擠下來 0
擠下去 2
擠了 27
擠人 13
擠來 1
擠來擠去 2
擠兌 140
擠兌出來 0
擠入 26
擠出 101
擠出來 9
擠出去 1
擠到 23
擠剩 0
擠去 4
擠向 2
擠在 67
擠垮 0
擠壓 185
擠奶 6
擠好 1
擠完 0
擠弄 0
擠得 38
擠得 22
擠成 20
擠成一團 3
擠掉 20
擠推 0
擠斷 0
擠昏了頭 0
擠死 3
擠沉 0
擠滿 106
擠痛 0
擠的 22
擠眉弄眼 11
擠破 27
擠空 0
擠落 2
擠著 9
擠足 1
擠身 17
擠身於 10
擠迫 4
擠進 102
擠進來 2
擠進去 6
擠過 1
擠過來 0
擠過去 0
擠錯 0
擠開 3
擡 0
擢 133
擢升 8
擢昇 1
擢為 0
擢用 1
擢髮難數 0
擣 12
擣衣 0
擤 41
擥 0
擦 644
擦上 10
擦上來 0
擦上去 0
擦下 0
擦下來 0
擦下去 0
擦乾 59
擦乾淨 14
擦亮 49
擦來 0
擦來擦去 0
擦傷 33
擦刮 0
擦到 5
擦去 29
擦在 16
擦子 2
擦布 2
擦抹 5
擦抹著 0
擦拭 225
擦拭法 4
擦掉 24
擦撞 1
擦擦 18
擦擦看 0
擦有 0
擦槍 17
擦汗 16
擦油 0
擦洗 6
擦淨 6
擦澡 3
擦熱 1
擦牆而過 0
擦皮 0
擦皮鞋 3
擦破 5
擦粉 5
擦肩 4
擦肩而過 45
擦背 5
擦著 14
擦藥 16
擦起 2
擦起來 0
擦身 19
擦身而過 61
擦過 18
擦過來 0
擦過去 0
擦鞋 11
擦鞋油 1
擦鞋童 0
擦音 1
擦黑兒 0
擦黑兒 0
擧 0
擨 0
擩 0
擩 0
擪 0
擪 0
擫 0
擬 4175
擬人 8
擬人化 30
擬人法 2
擬作 1
擬具 120
擬古 0
擬在 63
擬妥 66
擬定 1298
擬於 122
擬有 5
擬稿 23
擬與 56
擬訂 653
擬請 54
擬議 79
擬辦 62
擭 164
擮 0
擮 0
擯 57
擯棄 1
擰 107
擰 107
擰 107
擰下 0
擰下來 0
擰下去 0
擰乾 9
擰住 1
擰斷 0
擰笑 0
擰緊 0
擰起 0
擰起來 0
擰開 0
擱 253
擱下 12
擱下來 0
擱不下 0
擱在 40
擱延 3
擱淺 165
擱筆 5
擱置 103
擱著 4
擱著不動 0
擱著不管 1
擱起 0
擱起來 0
擱開 0
擲 285
擲下 10
擲交 13
擲地 526
擲地有聲 8
擲棒 0
擲標槍 0
擲瓶 2
擲還 2
擲鐵球 0
擲鐵餅 0
擲骰子 11
擳 31
擴 915
擴充 1746
擴充器 2
擴充性 82
擴充槽 0
擴到 1
擴增 434
擴增到 7
擴大 4633
擴大再生產 2
擴大化 8
擴大器 61
擴大招生 12
擴大會議 7
擴大機 277
擴孔 2
擴展 742
擴展到 75
擴展性 5
擴廠 138
擴廠計劃 6
擴延 1
擴延到 0
擴建 547
擴建工廠 0
擴建工程 80
擴張 1001
擴張主義 4
擴張器 8
擴張開來 1
擴戲 0
擴散 711
擴散作用 6
擴散出來 2
擴編 126
擴聲器 1
擴軍 5
擴軍備戰 0
擴音 34
擴音器 53
擴音機 41
擵 0
擶 0
擷 79
擷 79
擷取 747
擸 24
擹 0
擺 993
擺上 70
擺上來 6
擺上去 5
擺下 6
擺下來 0
擺下去 0
擺不下 0
擺佈 39
擺來 2
擺來擺去 1
擺出 104
擺出來 8
擺出去 0
擺列 4
擺列出 0
擺列著 0
擺到 15
擺動 132
擺回 1
擺回來 0
擺回去 0
擺在 357
擺地攤 22
擺夷 0
擺好 24
擺官架子 1
擺尾 8
擺布 18
擺平 62
擺弄 13
擺得 8
擺成 5
擺擺 4
擺擺手 3
擺擺頭 1
擺攤子 3
擺放 94
擺放在 10
擺放著 5
擺明 56
擺架子 2
擺樣 0
擺樣子 1
擺法 4
擺渡 49
擺盪起來 0
擺線 9
擺置 47
擺脫 652
擺脫到 0
擺著 65
擺設 250
擺譜 2
擺起 5
擺起來 1
擺進 9
擺進來 1
擺進去 1
擺酒 6
擺酒席 0
擺錘 8
擺長 2
擺開 21
擺開陣勢 2
擺闊 1
擺闊氣 0
擺點 4
擺齊 0
擺龍門陣 2
擻 9
擻 9
擼 49
擼子 0
擽 42
擾 408
擾亂 121
擾亂治安 1
擾人 26
擾人清夢 3
擾動 54
擾攘 13
擾民 12
擿 40
攀 631
攀上 34
攀上來 0
攀上去 0
攀不上 1
攀住 4
攀供 0
攀到 1
攀升 244
攀山會 0
攀扯 0
攀折 0
攀援 7
攀枝 14
攀登 266
攀登上 3
攀登架 0
攀索 2
攀緣 22
攀緣莖 0
攀著 3
攀親 3
攀親道故 0
攀談 13
攀起 0
攀越 9
攀蹬 0
攀過 8
攀過來 0
攀過去 1
攀附 21
攀附力 0
攀靠 0
攀靠在 0
攀龍附鳳 1
攁 24
攂 0
攂 0
攃 90
攄 65
攅 0
攅 0
攆 10
攆出 2
攆出來 0
攆出去 0
攆去 0
攆走 0
攆跑 0
攆開 0
攇 9
攇 9
攈 0
攈 0
攉 4
攊 0
攋 0
攋 0
攋 0
攋 0
攌 7
攍 8
攎 3
攏 195
攏 195
攏絡 0
攏總 10
攐 5
攑 0
攓 1
攓 1
攔 5456
攔下 100
攔不住 2
攔住 61
攔劫 5
攔在 9
攔截 211
攔截法 0
攔擊 1
攔擋住 0
攔水閘 0
攔污柵 4
攔沙壩 11
攔河壩 1
攔洪水庫 0
攔網 5
攔腰 11
攔腰截斷 3
攔路 15
攔路搶劫 1
攔路虎 1
攔車 5
攔阻 70
攕 4
攖 10
攗 2
攗 2
攘 40
攘 40
攘土 0
攘夷 4
攘奪 1
攘攘 6
攘災 1
攘袂 0
攘除 1
攙 46
攙住 0
攙假 0
攙合 0
攙和 4
攙扶 13
攙扶著 3
攙起 0
攙雜 9
攙雜在 0
攙雜著 1
攛 33
攛掇 3
攛高 0
攜 781
攜 781
攜伴 13
攜出 41
攜同 8
攜家帶眷 28
攜帶 1321
攜帶型 60
攜帶式 81
攜帶方便 57
攜幼 5
攜手 493
攜手同心 21
攜槍 16
攜款 3
攜物 1
攜眷 7
攜離 3
攝 1784
攝像機 0
攝制 1
攝去 0
攝取 679
攝取量 116
攝影 6422
攝影室 26
攝影家 116
攝影師 262
攝影棚 104
攝影機 696
攝影留念 8
攝影社 283
攝影藝術 104
攝影術 22
攝影記者 47
攝影課 0
攝政 9
攝政王 103
攝於 154
攝氏 322
攝氏寒暑 0
攝氏寒暑表 0
攝氏溫度 10
攝氏溫度計 0
攝製 23
攝製廠 0
攝製成 1
攝護腺 276
攝食 172
攝魂 1
攞 0
攞 0
攟 0
攟 0
攠 9
攡 12
攢 82
攢 82
攢眉 0
攢錢 5
攢錢罐 0
攣 169
攣 169
攣弱 1
攣攣 1
攣踠 1
攤 520
攤下 2
攤下來 0
攤下去 0
攤了 5
攤位 522
攤出 1
攤出來 0
攤出去 0
攤到 1
攤子 41
攤提 0
攤派 7
攤牌 26
攤著 2
攤販 213
攤起 1
攤過 2
攤還 104
攤開 67
攥 18
攦 11
攧 0
攨 0
攨 0
攩 3
攪 281
攪 281
攪亂 24
攪動 31
攪勻 19
攪合 0
攪和 52
攪和 52
攪好 3
攪局 65
攪拌 225
攪拌器 29
攪拌機 55
攪擾 6
攪混 0
攪碎 7
攪過 0
攪過來 0
攪過去 0
攪鬼 0
攫 132
攫住 9
攫去 0
攫取 43
攫奪 0
攬 310
攬勝 85
攬權 4
攬筆 0
攬貨 5
攬頭 0
攭 61
攭 61
攮 5
支 5190
支付 1675
支使 7
支出 4972
支出費用 10
支取 16
支吾 6
支吾以對 1
支吾其詞 2
支吾其辭 0
支委 0
支局 174
支屬 1
支工 0
支左 1
支幹 11
支店 7
支座 1
支應 316
支承 19
支持 9318
支持不住 14
支持住 0
支持力 6
支持者 724
支持著 29
支援 40561
支撐 857
支撐不了 1
支撐住 1
支撐力 14
支撐危局 0
支撐架 16
支撐著 17
支支吾吾 9
支數 53
支書 2
支會 108
支杆 0
支架 135
支柱 162
支桿 2
支氣管 257
支氣管炎 126
支派 90
支流 217
支渠 0
支用 150
支票 862
支票簿 3
支線 198
支脈 7
支脈 7
支薪 0
支解 11
支護 1
支路 3
支農 0
支那 23
支部 87
支配 381
支配力 5
支配者 12
支隊 9
支隊長 1
支離 15
支離破碎 26
支領 186
支點 29
攰 0
攱 0
攲 6
攳 117
攴 0
收 17139
收上來 1
收上去 0
收下 37
收下來 5
收下去 0
收之桑榆 7
收了 81
收付款員 0
收件 701
收件人 549
收來 9
收來收去 0
收信 539
收假 21
收債 0
收入 3543
收入豐厚 2
收兵 21
收出 5
收出來 0
收出去 0
收到 3683
收割 111
收割下來 0
收割機 4
收割稻米 0
收去 7
收取 660
收受 380
收口 8
收回 660
收回來 66
收回到 66
收回去 66
收回成命 4
收在 70
收埋 2
收執 562
收報機 0
收報費 0
收場 10
收場 102
收塵 5
收好 17
收存 32
收存著 0
收完 9
收容 345
收容所 90
收尾 55
收屍 5
收山 8
收工 34
收帆 0
收帳 22
收得 26
收得 132
收復 149
收復失土 10
收復失地 1
收復河山 1
收心 16
收心操 4
收成 182
收手 13
收押 133
收押禁見 34
收拾 325
收拾乾淨 5
收拾殘局 14
收授 4
收掉 5
收播 35
收播 35
收播站 0
收播站 0
收操 4
收據 736
收攏 4
收攤 11
收攬 4
收支 1043
收支平衡 28
收支相抵 3
收支處 7
收收 10
收放 31
收放自如 11
收效 30
收效良好 1
收斂 156
收斂劑 1
收斂水 1
收文 171
收文簿 2
收方 17
收有 52
收服 26
收條 1
收款 189
收款人 20
收款單 2
收歛 21
收歛性 0
收歸 32
收歸國有 22
收殮 0
收獲 472
收率 12
收留 55
收留所 2
收發 594
收發室 24
收發站 3
收的 132
收益 675
收益稅 0
收監 2
收盤 3926
收盤價格 26
收盤匯率 0
收看 312
收看者 2
收票 19
收票員 1
收禮 8
收秋 0
收租 11
收稅 9
收稅員 0
收稿 24
收稿人 0
收穫 674
收穫量 9
收管 2
收管人 0
收納 113
收給 0
收緊 13
收編 34
收編在 2
收縮 736
收縮到 0
收縮壓 51
收縮自如 1
收縮起來 0
收繳 57
收羅 14
收聽 418
收聽到 5
收聽率 24
收聽者 3
收腳 1
收自 5
收著 2
收藏 1340
收藏家 116
收藏者 24
收蟻 0
收視 264
收視率 100
收訊 60
收訖 12
收話器 0
收話機 0
收貨 269
收貨人 38
收買 104
收買人心 10
收費 2432
收費員 7
收費站 119
收賄 37
收賬 10
收購 608
收購價 7
收購價格 7
收走 6
收起 112
收起來 40
收進 55
收進來 5
收進去 5
收過 17
收過來 0
收過去 0
收針 0
收銀機 132
收錄 2770
收錄在 77
收錄音機 63
收錢 58
收錯 0
收集 3272
收集家 2
收音 51
收音機 430
收養 284
收養人 15
收驚 14
收點 22
收齊 23
攷 98
攸 110
攸戚相關 0
攸攸之口 0
攸然 10
攸關 438
改 5583
改一改 17
改一改 17
改上 35
改下 12
改不了 20
改乘 9
改了 134
改了又改 5
改些 1
改以 264
改任 34
改住 2
改作 42
改使 6
改來 3
改來改去 3
改修 19
改個 16
改做 5
改元 6
改判 30
改刪 0
改到 47
改制 886
改加 3
改動 17
改區 3
改印 10
改去 60
改取 2
改口 50
改口說 11
改名 573
改名換姓 3
改向 13
改售 0
改唱 80
改唸 21
改善 7829
改善法 3
改回 42
改回來 31
改回去 4
改圖 9
改土 2
改在 60
改地 8
改場 5
改場 56
改填 1
改大 9
改天 235
改天再來 8
改天換地 0
改好 25
改姓 11
改娶 0
改嫁 11
改守 3
改寄 6
改寫 283
改對 5
改小 10
改屬 8
改帳 3
改幫 1
改建 633
改建工程 22
改弦易轍 16
改弦易轍 16
改弦更張 7
改往 3
改得 22
改得 146
改徵 4
改念 21
改性 3
改悔 0
改憲 2
改成 676
改戰 0
改打 27
改扣 0
改抄 0
改投 9
改掉 45
改換 57
改換法 0
改收 4
改數 0
改數 0
改於 18
改日 15
改日再來 0
改易 2
改時 10
改書 0
改朝換代 28
改期 29
改校 21
改樣 0
改次 2
改正 336
改正過來 5
改歲 0
改段 0
改炒 0
改為 1505
改煮 0
改爭 0
改版 636
改用 525
改由 212
改當 9
改當 9
改的 145
改短 10
改祭 0
改租 2
改程 10
改稱 232
改站 1
改答 10
改算 1
改節 0
改約 20
改組 399
改編 437
改編為 13
改置 8
改脾氣 0
改自 12
改良 1334
改良主義 2
改良品種 5
改良場 37
改良場 374
改良法 4
改良種 4
改葬 10
改號 2
改行 46
改行換業 0
改補 0
改裝 929
改裝法 0
改製 16
改要 0
改觀 60
改訂 28
改記 0
改說 23
改說是 0
改調 9
改調 9
改變 6622
改變成 27
改賜 0
改起 8
改起來 20
改輪 0
改辦 10
改途 0
改造 1112
改造法 0
改進 2645
改進法 0
改過 127
改過來 11
改過去 12
改過自新 9
改道 47
改道行駛 1
改選 590
改邪歸正 8
改量 7
改量 7
改銷 2
改錯 15
改鑄 0
改長 0
改隊 0
改革 4632
改革性 17
改革法 10
改革派 53
改頭換面 67
改食 1
改飼 0
改駛 0
攺 0
攺 0
攻 2365
攻上 29
攻上來 2
攻上去 2
攻下 67
攻下來 0
攻下去 0
攻不上 0
攻不下 5
攻了 6
攻人不備 0
攻人之短 0
攻伐 7
攻伐 7
攻佔 92
攻來 22
攻來攻去 0
攻倒 0
攻克 45
攻入 24
攻其不備 1
攻到 8
攻剋 0
攻勢 256
攻占 6
攻去 11
攻取 19
攻向 27
攻城 62
攻城掠地 11
攻堅 58
攻奪 2
攻守 58
攻守俱佳 6
攻守同盟 0
攻對 3
攻得 5
攻得 5
攻心 9
攻心戰 0
攻心為上 0
攻打 116
攻抵 0
攻掠 0
攻撲 0
攻擊 2188
攻擊力 105
攻擊性 129
攻擊線 1
攻擊者 14
攻擊面 2
攻擊點 0
攻無不克 5
攻略 1
攻研 1
攻破 21
攻籃 2
攻訐 65
攻讀 196
攻起 0
攻進 37
攻進來 1
攻進去 1
攻過 27
攻過來 8
攻過去 3
攻錯 28
攻關 6
攻防 0
攻陷 39
攻隊 11
攻頂 1
攼 0
攽 45
放 6955
放上 210
放上來 26
放上去 36
放下 515
放下來 10
放下去 8
放下屠刀 6
放下屠刀立地成佛 1
放不下 64
放不開 13
放之四海 1
放諸四海 1
放之四海皆準 1
放諸四海皆準 1
放了 255
放人 18
放任 156
放任主義 2
放任政策 5
放來 3
放來放去 0
放個 45
放倒 90
放假 684
放假日 43
放債 1
放光 39
放光芒 17
放入 1219
放冷槍 0
放冷箭 3
放出 297
放出來 21
放出去 5
放刁 1
放到 376
放去 4
放反 0
放哨 3
放唱片 0
放回 75
放在 2541
放在心上 63
放在心裡 30
放在眼裡 10
放大 1221
放大係數 0
放大器 271
放大率 11
放大鏡 147
放好 19
放學 265
放定 5
放寒假 10
放寬 894
放射 1163
放射出 18
放射出來 0
放射性 1223
放射性元素 26
放射性同位素 43
放射源 3
放射狀 29
放射病 1
放射科 1
放射線 696
放射線科 1
放屁 81
放工 7
放帳 5
放廣 10
放得 45
放得 197
放心 850
放心不下 12
放心起來 0
放恣 3
放慢 95
放懷 2
放手 215
放手一搏 19
放手去做 10
放掉 41
放放 6
放散 6
放映 503
放映室 30
放映機 28
放晴 31
放暑假 38
放暗箭 1
放有 17
放棄 2255
放榜 206
放槍 1
放款 1270
放歌 7
放毒 5
放氣 24
放水 115
放油 6
放洋 4
放浪 21
放浪不羈 1
放浪形骸 11
放滿 18
放火 43
放火燒山 0
放炮 22
放煙 5
放煙幕 0
放煙火 18
放熱 9
放熱反應 2
放爆竹 1
放牛 13
放牛仔 0
放牛吃草 22
放牛班 10
放牧 27
放牧區 1
放牧地 0
放生 95
放的 197
放盡 0
放盪 2
放眼 281
放眼望去 36
放眼看去 2
放矢 1
放砲 11
放穩 2
放空 163
放空氣 10
放空炮 0
放空車 0
放糖 4
放給 228
放線 48
放縱 97
放置 1197
放置不用 10
放羊 21
放羊吃草 1
放聲 28
放聲大哭 28
放聲高唱 0
放聲高歌 2
放肆 67
放膽 22
放膽去做 0
放臭屁 1
放著 136
放著不用 12
放著不管 13
放蕩 33
放蕩不羈 5
放虎歸山 2
放血 31
放行 172
放言 6
放言高論 3
放話 123
放誕 0
放諸四海 22
放諸四海皆準 0
放貸 11
放賑 0
放走 26
放起 7
放起來 0
放輕 11
放逐 92
放進 176
放進來 12
放進去 46
放過 234
放過來 0
放過去 0
放長 9
放長線 11
放長線釣大魚 5
放開 204
放開來 1
放電 367
放領 160
放風 20
放風箏 37
放風聲 50
放養 53
放養密度 2
放馬 51
放馬奔騰 0
放馬後炮 1
放馬後砲 1
放高 3
放高利貸 4
放鬆 553
放鴿子 19
放鹽 2
放點 5
政 12364
政事 404
政事堂 0
政令 206
政制 18
政務 263
政務委員 189
政務委員會 3
政務官 144
政務院 2
政協 79
政和 17
政商 201
政壇 358
政大 1390
政委 13
政委會 4
政客 242
政局 233
政工幹校 4
政府 41691
政府機構 458
政府機關 2152
政府軍 25
政情 321
政情報導 0
政戰 140
政戰學校 62
政戰官 5
政戰部 34
政教 51
政教合一 8
政敵 30
政權 922
政權機構 0
政權機關 0
政治 21609
政治作戰 18
政治作戰學校 109
政治協商 11
政治協商會議 13
政治地理 1
政治大學 2065
政治委員 0
政治學 506
政治學家 2
政治家 126
政治局 45
政治性 174
政治民主 31
政治犯 90
政治界 6
政治系 252
政治經濟 128
政治經濟學 64
政治課 1
政治避難 0
政治部 10
政法 18
政潮 3
政界 130
政界人士 13
政略 7
政策 13229
政策性 131
政經 71
政綱 68
政績 233
政聞社 1
政要 77
政見 706
政見會 149
政見發表 130
政見發表會 69
政論 37
政論家 7
政變 49
政財界 0
政躬 0
政躬康泰 4
政通 45
政通人和 6
政風 0
政體 95
政黨 2762
政黨政治 231
敀 0
敁 42
敁敪 0
敂 0
敃 38
敄 0
故 8268
故主 18
故事 9305
故事性 18
故事書 75
故交 8
故人 85
故作 52
故作不知 0
故作姿態 2
故作神祕 6
故作神秘 7
故典 1
故友 10
故名 96
故吏 0
故國 35
故國之思 0
故國風光 16
故園 12
故土 30
故地 6
故地重遊 0
故址 3
故城 6
故夫 5
故如 55
故宅 3
故宮 494
故宮博物 4
故宮博物院 537
故宮博物館 13
故實 8
故居 91
故常 43
故弄 2
故弄玄虛 9
故往 4
故意 1062
故態 1
故態復萌 2
故我 136
故技重施 4
故技重演 0
故於 94
故書 3
故有 168
故此 137
故步自封 17
故為 70
故無 106
故犯 2
故知 25
故習 1
故老 33
故而 166
故舊 16
故親 1
故計重施 1
故道 6
故都 18
故鄉 1059
故里 30
故障 966
故障率 37
敆 109
敇 0
效 3159
效力 502
效勞 28
效命 12
效尤 15
效忠 80
效忠國家 3
效忠祖國 0
效應 1689
效應器 0
效果 6777
效死 1
效法 198
效率 4114
效用 343
效益 1912
效能 1641
效驗 12
敉 87
敉平 3
敊 43
敋 0
敌 0
敌 0
敍 0
敎 0
敎 0
敏 4053
敏惠 59
敏感 950
敏感度 158
敏感性 150
敏慧 46
敏捷 114
敏於事而慎於言 1
敏而好學 0
敏華 35
敏達 17
敏銳 297
敐 0
敐 0
救 32274
救上 1
救上來 1
救上去 0
救下 10
救下來 1
救下去 0
救世 105
救世主 46
救世者 1
救世軍 8
救主 137
救亡 6
救亡圖存 16
救人 243
救人一命 20
救人如救火 1
救來 0
救兵 21
救出 140
救出來 14
救出去 0
救到 7
救助 822
救助者 4
救去 1
救命 457
救命恩人 7
救員 1
救回 110
救回來 14
救回去 0
救國 29
救國團 390
救國救民 7
救國軍 3
救定 0
救得 22
救急 75
救急不救窮 15
救恩 1
救援 783
救援軍 0
救救 601
救星 76
救民 2
救治 58
救法 43
救活 37
救溺 4
救濟 670
救濟品 5
救濟災民 0
救濟貧民 0
救濟費 1
救濟院 5
救濟難民 0
救火 64
救火員 22
救火車 1
救火隊 27
救災 333
救災總會 2
救球 4
救生 66
救生員 26
救生圈 15
救生帶 0
救生筏 5
救生船 7
救生艇 38
救生衣 36
救生設備 9
救窮 7
救苦 8
救苦救難 31
救護 503
救護機 0
救護站 42
救護船 0
救護車 160
救貧濟弱 0
救贖 235
救走 15
救起 42
救起來 5
救過 13
救過來 0
救過去 0
救醒 3
救難 153
救駕 6
敒 0
敓 105
敔 147
敕 177
敕令 4
敕命 0
敕封 12
敕授 0
敖 378
敗 1538
敗下陣來 14
敗事有餘 2
敗亡 17
敗仗 18
敗北 48
敗名 3
敗在 47
敗壞 160
敗家 0
敗家子 23
敗家子 23
敗將 7
敗局 0
敗得 10
敗德 11
敗潰 0
敗筆 14
敗絮 1
敗絮其中 5
敗績 14
敗缺 0
敗者為寇 3
敗胃 0
敗興 4
敗興而歸 14
敗落 9
敗血病 3
敗血症 85
敗訴 63
敗走 25
敗軍 17
敗退 13
敗部 0
敗部復活 1
敗門風 0
敗陣 28
敗陣下來 0
敗露 9
敗類 40
敘 1168
敘事 133
敘事文 2
敘事曲 3
敘事詩 7
敘事體 0
敘別 0
敘利亞 51
敘利亞人 0
敘功 1
敘家常 4
敘情 3
敘敘 6
敘敘舊 2
敘舊 15
敘言 0
敘說 24
敘說著 6
敘談 1
敘述 2505
敘述出來 3
敘述法 0
敘述著 10
教 17421
教 17421
教上 9
教下去 0
教主 150
教了 43
教些 7
教令 2
教個 9
教具 659
教具室 24
教出 33
教到 32
教務 508
教務主任 204
教務會議 236
教務組 125
教務處 1864
教務長 193
教化 200
教區 90
教區長 1
教友 90
教員 117
教唆 57
教唆犯 8
教唆罪 0
教團 65
教坊 659
教堂 598
教壇 0
教壞 10
教士 307
教好 40
教委會 4
教學 29140
教學不倦 0
教學內容 283
教學大綱 140
教學法 354
教學法 354
教學相長 45
教學相長 45
教學計劃 138
教學計劃 138
教完 18
教宗 179
教官 766
教官室 116
教室 11668
教導 1647
教導 1647
教導員 1
教導有方 3
教導法 1
教局 38
教師 20482
教師節 213
教席 11
教廷 94
教徒 156
教得 18
教得 487
教慣 0
教成 7
教戰 157
教授 12158
教授 12158
教授法 2
教改 0
教改會 59
教教 243
教書 103
教書匠 6
教會 4584
教會學校 21
教本 52
教材 7072
教材費 8
教案 165
教桌 3
教條 71
教條主義 5
教權 55
教民 4
教法 1392
教法 1392
教派 153
教父 141
教的 486
教皇 67
教益 10
教研 154
教研室 2
教研組 16
教科文 0
教科書 1076
教程 29
教給 45
教統 3
教練 1304
教練室 0
教練席 0
教練機 42
教練車 3
教義 255
教習 5
教職 1089
教職人員 22
教職員 1457
教育 56219
教育出來 5
教育制度 209
教育化 4
教育史 30
教育司 154
教育大學 1
教育學 1502
教育學員 8
教育學院 228
教育家 84
教育局 1581
教育局長 69
教育廳 3096
教育廳長 33
教育性 129
教育改革 0
教育方針 32
教育法 481
教育界 207
教育系 396
教育經濟 2
教育經濟學 4
教育行政 1574
教育部 8625
教育部長 180
教育金 5
教育長 12
教著 3
教規 7
教訓 486
教訓法 0
教誡 8
教誨 86
教課 91
教起 9
教起來 1
教過 35
教過來 0
教過去 0
教錯 0
教長 11
教鞭 11
教頭 48
教養 596
教養有方 2
教養院 126
教點 3
敚 0
敜 10
敝 356
敝人 246
敝友 1
敝國 13
敝塞 0
敝姓 6
敝屣 4
敝履 0
敝帚千金 0
敝帚自珍 3
敝徙 0
敝舊 0
敝舍 6
敝處 0
敝邦 0
敞 244
敞亮 1
敞口 0
敞篷 48
敞篷車 40
敞胸露懷 0
敞胸露懷 0
敞著 2
敞蓬車 5
敞開 191
敞露 1
敞露 1
敟 0
敠 0
敡 0
敢 2000
敢上 13
敢上來 0
敢上去 0
敢下 8
敢不 39
敢不敢 24
敢付 0
敢作 19
敢作敢為 6
敢作敢當 1
敢來 35
敢借 1
敢做 34
敢出 20
敢出來 5
敢出去 0
敢到 12
敢刻 0
敢刻 0
敢動 24
敢勸 1
敢去 82
敢受 4
敢向 23
敢告 5
敢唱 1
敢唸 1
敢問 73
敢喊 2
敢喝 17
敢喝 17
敢嚇 0
敢在 92
敢坐 6
敢奏 0
敢嫌 0
敢守 0
敢寫 17
敢射 0
敢彈 0
敢念 0
敢怒不敢言 5
敢怒而不敢言 1
敢情 4
敢情是 2
敢想 40
敢愛敢恨 12
敢打 8
敢扯 0
敢找 5
敢拆 3
敢拜 1
敢接 17
敢揍 0
敢撤 0
敢擺 2
敢收 4
敢於 57
敢死 9
敢死隊 10
敢洗 1
敢演 0
敢為 20
敢爭 0
敢現 2
敢留 12
敢畫 1
敢當 30
敢管 3
敢練 1
敢罵 14
敢背 1
敢背 1
敢花 2
敢被 0
敢言 14
敢設 0
敢試 4
敢說 187
敢說敢做 1
敢談 6
敢請 7
敢走 5
敢躺 0
敢輸 0
敢過 1
敢過來 0
敢過去 1
敢開 17
敢陪 5
敢騎 5
敢鬧 0
敢點 1
散 3172
散 3172
散亂 39
散伙 0
散佈 0
散佈者 0
散佈開 0
散佚 7
散來 4
散來散去 0
散光 119
散光眼鏡 0
散兵 5
散兵坑 6
散兵壕 0
散兵游勇 3
散出 25
散去 65
散土 0
散場 25
散場 25
散夥 0
散失 50
散失到 0
散射 182
散居 53
散布 300
散帙 0
散心 35
散慢 1
散戶 0
散掉 6
散播 301
散播謠言 13
散散 4
散散心 10
散散步 16
散散漫漫 0
散文 1431
散文家 8
散文詩 13
散文體 6
散會 239
散步 301
散步道 4
散沙 2
散漫 31
散熱 374
散熱器 23
散熱膏 1
散熱裝置 3
散班 0
散發 524
散發傳單 10
散碎 1
散花 12
散落 117
散蕩 1
散裝 176
散裝船 1
散裝貨 3
散見 53
散財 24
散財童子 1
散過 0
散過來 0
散過去 0
散開 59
散髮 8
敤 5
敥 3
敦 4925
敦促 124
敦倫 1
敦勉 0
敦化 187
敦化北路 586
敦化南路 1091
敦化國中 17
敦化國小 34
敦厚 36
敦品力學 2
敦實 2
敦樸 3
敦煌 319
敦煌石窟 4
敦睦 38
敦聘 56
敦親睦鄰 45
敦請 44
敧 8
敧 8
敨 13
敪 7
敫 0
敫 0
敫 0
敬 1827
敬上 355
敬之 27
敬事 1
敬仰 35
敬佩 137
敬來敬去 0
敬候 5
敬呈 1
敬告 83
敬啟 119
敬啟者 50
敬奉 8
敬師金 0
敬意 163
敬愛 133
敬拜 249
敬服 7
敬業 188
敬業樂群 33
敬業精神 46
敬煙 1
敬畏 135
敬祝 180
敬神 13
敬禮 144
敬稱 3
敬老 216
敬老尊賢 18
敬而遠之 23
敬茶 3
敬語 13
敬請 3233
敬請光臨 7
敬請指教 130
敬謝 8
敬謝不敏 11
敬謹 5
敬賀 7
敬贈 3
敬軍 3
敬酒 28
敬酒不吃吃罰酒 2
敬重 77
敬陪末座 16
敬陳 1
敬頌 2
敬香 0
敬體 1
敬鬼神而遠之 3
敭 0
敮 0
敯 41
敯 41
数 0
数 0
数 0
数 0
数 0
敱 0
敱 0
敲 519
敲一敲 4
敲上 1
敲上來 0
敲上去 0
敲下 17
敲下來 0
敲下去 1
敲了 38
敲住 0
敲來敲去 0
敲出 23
敲定 247
敲得 7
敲打 98
敲掉 5
敲擊 85
敲擊聲 1
敲敲打打 17
敲更 1
敲的 7
敲破 5
敲碎 13
敲竹槓 17
敲著 32
敲詐 10
敲詐勒索 0
敲詐錢財 1
敲起 2
敲起來 2
敲過 1
敲過來 0
敲過去 0
敲邊鼓 3
敲鐘 14
敲鑼 3
敲鑼打鼓 10
敲門 103
敲門磚 11
敲開 30
敳 19
整 5994
整世 1
整世紀 0
整了 20
整人 54
整人法 0
整付 13
整件 39
整件事 17
整伍 0
整來整去 1
整修 472
整修費 1
整個 5823
整個人 180
整個來說 6
整個月 13
整億 0
整具 3
整冊 0
整到 7
整副 2
整匙 0
整區 5
整口 0
整口氣 0
整句 13
整句話 0
整合 9014
整合性 1406
整合法 3
整合系統 844
整周 0
整圈 2
整在 6
整地 175
整型 80
整型外科 114
整型手術 10
整型術 5
整型費 0
整堆 2
整場 47
整場 47
整場戲 0
整場戲 0
整塊 27
整塊地 1
整夜 76
整夜未歸 0
整天 478
整套 205
整套書 2
整存整付 39
整存零付 4
整季 5
整宗 0
整容 46
整容手術 1
整容術 1
整封 9
整封信 0
整局 7
整巷 0
整師 0
整幅 14
整幕 0
整年 108
整年到頭 0
整座 100
整建 264
整建工程 30
整式 1
整形 860
整形術 11
整得 18
整截 2
整打 1
整把 3
整排 29
整支 19
整整 173
整整的 4
整整地 4
整整齊齊 15
整數 393
整日 42
整晚 73
整月 2
整月份 0
整期 5
整本 78
整本書 24
整朵 1
整朵花 4
整杯 1
整杯水 1
整杯茶 2
整杯酒 0
整枝 9
整根 9
整根煙 1
整桶 1
整梯次 0
整棟 86
整棵 5
整棵樹 3
整樁 0
整檔 3
整櫃 28
整段 35
整段路 0
整治 608
整派 0
整流 84
整流器 79
整流子 4
整流管 0
整潔 298
整潔為強身之本 2
整營 1
整片 105
整版 22
整班 6
整球 0
整理 5269
整理出來 62
整理盒 30
整發 3
整盒 1
整碗 1
整碗飯 0
整科 5
整站 0
整筆 21
整筒 0
整箱 10
整節 5
整節課 0
整篇 43
整篇文章 97
整籠 0
整粒 16
整線 16
整編 87
整缸 4
整缸水 2
整罐 4
整肅 32
整葉 0
整著 1
整行 13
整行字 0
整衣 11
整裝 72
整裝待命 1
整裝待發 29
整襟危坐 0
整課 24
整路 8
整身 2
整軍 14
整軍經武 5
整輪 1
整週 7
整過 4
整道 1
整道菜 0
整邊 0
整部 124
整部書 0
整鍋 0
整門 0
整間 23
整除 46
整隊 24
整隻 46
整集 4
整面 12
整頁 49
整項 4
整頓 322
整頓飯 0
整顆 22
整題 6
整風 3
整飭 22
整餐 2
整餐飯 0
整首 25
整首歌 10
整體 5647
整體性 244
整點 31
整黨 1
整齊 360
整齊乾淨 0
整齊劃一 13
整齊清潔 10
整齣 15
整齣戲 24
敵 745
敵不住 0
敵不過 35
敵人 1311
敵偽 2
敵偽政權 0
敵偽軍 0
敵兵 5
敵前 2
敵前敵後 0
敵占區 0
敵友 2
敵友不分 1
敵國 26
敵害 5
敵寇 2
敵對 172
敵對行為 1
敵師 0
敵後 10
敵後工作 0
敵後工作人員 1
敵後方 0
敵得過 4
敵情 20
敵情觀念 0
敵意 155
敵愾 7
敵愾同仇 1
敵我 48
敵我不分 0
敵我矛盾 2
敵手 53
敵方 151
敵樓 3
敵機 41
敵火 2
敵營 4
敵特 0
敵百蟲 0
敵群 1
敵船 7
敵艦 17
敵視 27
敵軍 110
敵隊 6
敵體 1
敵黨 1
敶 66
敶 66
敷 3990
敷上 6
敷了 2
敷出 0
敷到 0
敷劑 0
敷嘴 0
敷在 13
敷塗 2
敷好 0
敷得 0
敷成 1
敷手 0
敷掉 0
敷料 30
敷旁 0
敷淺 0
敷粉 0
敷腳 0
敷臉 26
敷著 1
敷藥 7
敷衍 80
敷衍了事 11
敷衍二些 0
敷衍塞責 7
敷衍瑛責 0
敷設 10
敷足 0
敷過 0
敷面 11
數 11
數 30915
數 30915
數一數二 44
數上 40
數上來 0
數上去 1
數不到 2
數不清 67
數不盡 27
數九 8
數了 40
數人 0
數人 0
數以千計 54
數以卡計 0
數以百萬計 19
數以百計 31
數以萬訂 0
數以萬計 53
數份 10
數位 9266
數位化 840
數位器 1
數位相機 0
數位輸入法 0
數位顯示 8
數來 47
數來寶 7
數來數去 0
數例 16
數倍 75
數借 2
數值 1387
數值字 0
數值碼 0
數典忘祖 6
數出 10
數分 40
數分之差 0
數列 82
數到 44
數十 938
數十年如一日 11
數十年如一日 11
數千 435
數去 3
數叉 0
數名 188
數國 17
數夜 2
數夜未歸 0
數字 2972
數字計算機 0
數字計領 0
數字通信 1
數字鍵 35
數學 5313
數學分俘 0
數學分析 16
數學原理 4
數學原透 0
數學家 78
數學模型 15
數學科 403
數學秒 0
數學系 1175
數學組 52
數學課 22
數完 5
數家珍 2
數度 131
數得 1095
數息觀 1
數成 242
數戰 0
數所 65
數控 60
數據 3069
數據機 2447
數據處理 22
數據處理 22
數據資料 72
數數 45
數數兒 3
數數兒 3
數數兼 0
數數看 14
數日 171
數日之久 1
數杯 2
數桌 2
數次 280
數球 3
數理 900
數理化 46
數百 552
數百人 40
數百元 17
數的 1096
數目 880
數目字 23
數矛字 0
數碼 27
數站 0
數米志炊 0
數米而炊 1
數給 4
數線 0
數萬 115
數萬人 11
數萬元 35
數萱 0
數落 32
數著 59
數見不鮮 0
數說 8
數論 44
數起 29
數起來 1
數軸 0
數透氾 0
數過 24
數過來 0
數過去 1
數量 6515
數量化 13
數量級 6
數錢 2
數額 364
數點 49
敹 57
敺 233
敻 117
敻 117
敼 112
敿 225
斀 28
斁 54
斁 54
斂 370
斂容 0
斂色 1
斂衣 0
斂衾 0
斂財 39
斂跡 0
斂錢 1
斃 232
斃命 11
斃掉 2
斄 3
斅 0
斅 0
文 1
文 45788
文末 100
文不對題 4
文事 62
文人 207
文人治國 0
文人畫 7
文人相輕 1
文人統治 1
文以載道 3
文件 14213
文件夾 25
文件櫃 4
文信國 0
文公 31
文具 1384
文具店 39
文具用品 66
文具行 181
文刊 25
文則 34
文化 28252
文化事業 1321
文化交流 254
文化人 42
文化區 13
文化史 366
文化圈 12
文化大學 904
文化大革命 45
文化宮 2
文化局 14
文化建設 201
文化建設委員會 161
文化感 0
文化水平 9
文化水準 35
文化流氓 2
文化界 54
文化程度 2
文化組 109
文化部 63
文化鬥爭 2
文句 145
文史 1660
文名 54
文告 20
文品 2
文品佳 0
文場 38
文場 38
文墨 12
文壇 126
文壇人士 0
文士 40
文天祥 17
文契 1
文娛 6
文字 11286
文字交 2
文字學 91
文字學家 1
文字幕 13
文字獄 8
文學 5835
文學作品 167
文學化 1
文學史 179
文學家 73
文學界 35
文學系 769
文學評論 36
文學運動 4
文學院 1484
文學革命 3
文學館 28
文宗 75
文官 111
文官考試 1
文官資格 1
文定 14
文定之喜 3
文宣 0
文山 492
文工會 42
文帝 22
文庫 820
文康 63
文康中心 14
文康活動 39
文廟 10
文建會 466
文弱 8
文弱書生 0
文從字順 0
文德 232
文德女中 4
文德女子高級中學 1
文心雕龍 63
文思 23
文思泉湧 0
文情 5
文情並茂 5
文意 40
文愧金聲 1
文憑 258
文憑主義 20
文戲 6
文房 10
文房四寶 49
文才 18
文抄公 0
文摘 564
文攻武嚇 0
文教 3403
文教區 23
文教界 2
文教處 59
文文靜靜 1
文旦 86
文旦皮 0
文昌 409
文昌街 83
文明 1387
文明人 14
文明史 20
文明國家 18
文明戲 0
文明病 0
文明社會 31
文景之治 1
文曲 35
文曲 35
文曲星 1
文書 1396
文書處理 1353
文書處理 1353
文本 250
文林 214
文案 110
文武 252
文武全才 4
文武兼備 4
文武合一 9
文武場 10
文武場 10
文武廟 24
文武百官 11
文武雙全 7
文殊 91
文殊師利 1
文氣 6
文治 34
文治武功 1
文法 465
文法學 1
文法學家 0
文法書 3
文火 7
文牒 0
文牘 7
文物 1675
文物組 0
文獻 3476
文獻會 76
文獻考證 0
文王 54
文理 360
文理學院 35
文生 166
文盲 73
文盲率 3
文石 69
文祺 12
文科 215
文科學生 6
文稿 303
文章 77532
文章家 1
文竹 34
文筆 180
文筆流暢 7
文筆簡潔 0
文籍 4
文縐縐 0
文義 74
文聖 62
文職 13
文職人員 6
文脈 5
文脈 5
文臣 1
文華 1014
文萊 12
文藝 1655
文藝圈 0
文藝復興 339
文藝思想 12
文藝批評 11
文藝會 0
文藝活動 32
文藝界 16
文藻 502
文藻外語學院 1
文藻語專 56
文虎 22
文蛤 0
文言 28
文言文 19
文詞 26
文謅謅 10
文豪 82
文責自負 0
文質 4
文質彬彬 12
文身 4
文辭 32
文過 0
文過飾非 7
文選 227
文選集 1
文采 115
文采出眾 0
文雅 157
文雅氣息 0
文集 1022
文靜 56
文革 83
文革時期 11
文風 52
文風不動 2
文飾 6
文體 96
文魁 46
文鼎 1
斈 0
斈 0
斉 0
斉 0
斉 0
斊 0
斊 0
斊 0
斋 0
斌 1303
斍 0
斍 0
斍 0
斎 0
斐 736
斐冷翠 0
斐濟 96
斐然 125
斐然成章 0
斑 1649
斑剝 9
斑斑 1
斑斑可考 2
斑斕 22
斑疹 19
斑痕 4
斑白 15
斑竹 0
斑紋 77
斑綠 1
斑馬 61
斑馬線 20
斑駁 33
斑魚 1
斑鳩 37
斑點 132
斒 52
斔 5
斕 102
斖 1
斗 1061
斗六 492
斗六市 1586
斗南 764
斗大 31
斗子 6
斗子 6
斗室 13
斗方 0
斗栱 5
斗爭 15
斗笠 38
斗箕 0
斗篷 14
斗膽 16
斗轉參橫 0
斗轉星移 1
斗量 1
斗門 5
斘 0
料 8301
料中 44
料事 2
料事如神 6
料到 99
料器 2
料子 5
料定 7
料峭 4
料度 0
料得 14
料想 28
料想不到 11
料想到 5
料想得到 3
料斗 8
料理 2904
料理店 70
料算 2
料豆 0
料車 5
料酒 2
料量 148
料鬥 0
斚 0
斛 254
斜 872
斜 872
斜了 4
斜井 0
斜偏 0
斜入 0
斜到 0
斜坡 124
斜坡路 0
斜射 13
斜對角 3
斜對面 105
斜度 38
斜度標 0
斜得 0
斜成 0
斜暉 2
斜曲 0
斜桅 0
斜橋 4
斜法 0
斜照 12
斜率 67
斜的 28
斜眼 13
斜眼看人 0
斜睨 7
斜紋 55
斜紋布 15
斜線 56
斜著 12
斜視 115
斜角 27
斜角 27
斜角規 0
斜谷 7
斜軋 0
斜軸 8
斜進 0
斜進來 1
斜進去 0
斜過 2
斜過來 0
斜過去 0
斜邊 12
斜陽 33
斜面 57
斜風細雨 8
斜體 36
斜體字 36
斜高 0
斜點 0
斜座標系 10
斝 18
斞 4
斟 68
斟滿 8
斟酌 208
斟酌一下 0
斟酒 12
斠 30
斡 16
斡旋 90
斡旋到底 0
斢 134
斣 0
斤 351
斤兩 21
斤兩不足 0
斤斤 6
斤斤計較 43
斥 339
斥之為 3
斥候 8
斥力 1
斥喝 3
斥罵 3
斥言 0
斥責 58
斥資 176
斥退 4
斦 0
斧 230
斧子 1
斧政 0
斧柄 2
斧鉞 1
斧鑿痕跡 0
斧頭 66
斨 26
斪 22
斪 22
斫 13
斬 532
斬下 5
斬下來 1
斬下去 0
斬妖 8
斬斷 44
斬斷情絲 0
斬決 1
斬獲 60
斬盡 1
斬盡殺絕 0
斬草不除根 1
斬草除根 6
斬釘截鐵 25
斬鐵截釘 0
斬除 0
斬首 17
斬首示眾 0
断 0
斮 7
斯 15167
斯世 2
斯事 2
斯人 31
斯各特 0
斯圖亞特 0
斯土斯民 0
斯坦 347
斯堪地那維亞 20
斯巴達 19
斯德哥爾摩 90
斯托 80
斯拉夫 49
斯拉夫人 2
斯文 168
斯文掃地 0
斯時 13
斯特恩 0
斯特朗 0
斯特林堡 33
斯賓塞 4
斯通 7
斯里蘭卡 108
斯陀含 1
斯隆 2
新 28116
新檔 4400
新一代 787
新世界 479
新中國 45
新中國報 0
新主 65
新交 7
新人 926
新人新事 0
新人物 10
新人生觀 2
新任 570
新來 137
新來乍到 1
新倉頡 1
新倉頡輸入法 1
新光 1025
新光三越 0
新光人壽 158
新公園 31
新兵 87
新兵入伍 2
新兵訓練 21
新出 42
新出來 0
新出土 4
新刊 34
新到 186
新制 207
新制度 57
新創 53
新力 877
新力牌 8
新加坡 3139
新加坡人 17
新加坡島 0
新勝利 1
新化 594
新印象派 2
新去 1
新古典主義 15
新句 0
新台幣 11091
新名 25
新名詞 27
新四軍 0
新地 75
新型 938
新城 278
新埔 617
新埔 617
新埔工專 26
新埔工專 26
新埔工業專科學校 6
新埔工業專科學校 6
新埤 102
新增 0
新墨西哥 38
新墨西哥州 16
新大陸 63
新天 90
新天地 224
新奇 389
新奧爾良 4
新妝 6
新姑爺 0
新娘 617
新娘子 23
新婚 166
新婚之夜 65
新婚夫婦 21
新婚旅行 0
新婚燕爾 1
新婦 33
新嫁娘 4
新字 28
新學 66
新安 77
新安江 2
新官 11
新官上任 6
新官上任三把火 1
新家 633
新局面 41
新居 72
新居落成 28
新屋 132
新屋鄉 301
新島 15
新市 186
新年 563
新年快樂 144
新年新希望 7
新幹線 0
新幾內亞 60
新店 1174
新店市 1916
新店溪 95
新建 1190
新式 267
新得 14
新德里 33
新思想 13
新愁 4
新愁舊恨 2
新戲 141
新房 30
新手 1185
新技術 981
新招 97
新政 53
新教 38
新教徒 16
新教派 0
新文化 96
新文化運動 11
新文字 0
新文學 62
新文藝 16
新新 558
新旺 15
新星 183
新春 246
新春佳節 3
新時代 356
新時期的 1
新曆 6
新書 5392
新會 33
新月 122
新月形 4
新月狀 1
新月痕 0
新村 394
新東 149
新歌 298
新歡 66
新歡舊愛 7
新歲 10
新殖民主義 1
新民 430
新民主主義 1
新氣象 111
新沙 3
新法 101
新注音 1
新注音輸入法 1
新泰 245
新津 5
新津機場 0
新津機場 0
新派 55
新海 36
新消息 164
新港 753
新潮 245
新潮派 10
新潮流 131
新澤西州 216
新烏 10
新營 559
新營市 977
新片 581
新片試映 0
新片預告 503
新版 1591
新猷 0
新瓶裝舊酒 2
新生 2444
新生事物 2
新生代 187
新生兒 633
新生力量 0
新生北路 195
新生南路 398
新生命 125
新生地 107
新生報 60
新生大樓 0
新生活運動 11
新生界 0
新生訓練 60
新生路 403
新產品 1743
新用 8
新界 44
新異 5
新疆 286
新疆人 3
新疆省 3
新疆自治區 2
新發 237
新發明 21
新的 7049
新知 1993
新石器時代 67
新社鄉 0
新禧 6
新禧 6
新秀 296
新科 161
新科狀元 2
新穎 372
新穎性 67
新站 189
新竹 6986
新竹人 23
新竹市 5263
新竹師範學院 272
新竹教育大學 1
新竹縣 5025
新筆 7
新米 3
新紀元 276
新紀錄 45
新約 514
新約全書 2
新紐澤西 2
新經 7
新綠 30
新編 157
新罕布夏 0
新罕布夏州 2
新羅 20
新聞 126134
新聞人物 7
新聞公報 6
新聞報導 994
新聞學 114
新聞官 5
新聞家 0
新聞專科學校 0
新聞局 1110
新聞廣播 15
新聞廣播 15
新聞界 142
新聞癮 0
新聞稿 1807
新聞系 174
新聞紙 66
新聞群組 0
新聞自由 80
新聞記者 98
新聞通訊 8
新聞通訊社 51
新聞週刊 162
新聲 49
新臺幣 1118
新興 2637
新興事業 31
新興人類 0
新興力量 4
新興勢力 6
新興國家 57
新舊 420
新舊交接 2
新舊交替 12
新航 0
新芽 24
新苗 29
新英格蘭 64
新莊 1105
新莊市 2568
新莊鎮 3
新華 82
新華社 130
新葉 20
新虎尾溪 3
新血 0
新衣 68
新裝 98
新裝展示會 0
新西蘭 10
新解 0
新訓 21
新設 315
新詞 30
新詩 478
新豐 567
新貨 150
新貴 135
新賓 2
新車 704
新軍 5
新近 160
新造 45
新進 694
新進人員 152
新道 4
新遼河 0
新郎 140
新郎官 7
新都 78
新都市 8
新鄉 11
新鄭縣 0
新酷音 1
新酷音輸入法 1
新鈔 6
新銳 89
新開 281
新開幕 78
新陳代謝 643
新雅 25
新雨 50
新語 1000
新青 26
新面孔 34
新鞋 26
新頁 246
新風貌 86
新體 5
新體詩 0
新鮮 1412
新鮮人 478
新鮮味 1
新黨 4300
斱 0
斲 72
斲輪老手 0
斳 47
斴 0
斵 0
斶 6
斷 3424
斷乳 47
斷了 168
斷事 0
斷事如神 0
斷交 130
斷代史 27
斷去 17
斷口 11
斷垣殘壁 14
斷壁 3
斷壁殘垣 1
斷奶 64
斷奶期 0
斷子嗣 0
斷定 117
斷定是 6
斷層 675
斷崖 169
斷崖絕壁 6
斷弦 2
斷後 14
斷後路 3
斷得 194
斷成 13
斷成兩截 2
斷手 9
斷手斷腳 4
斷掉 42
斷掌 16
斷斷 14
斷斷不可 1
斷斷續續 99
斷根 16
斷案 1
斷案如神 0
斷氣 19
斷水 34
斷炊 12
斷無 12
斷無此事 0
斷無此理 1
斷然 79
斷然處置 18
斷片 12
斷獄 1
斷的 194
斷章取義 40
斷篇殘簡 0
斷簡殘篇 4
斷簡殘編 1
斷糧 11
斷絃 1
斷絕 91
斷絕來往 1
斷絕往來 1
斷絕邦交 0
斷絕關係 6
斷線 361
斷線風箏 2
斷編殘簡 0
斷續 38
斷肢 6
斷腳 6
斷腸 43
斷腸人 4
斷腿 9
斷落 10
斷行 98
斷袖之癖 3
斷裂 250
斷裂強度 1
斷言 65
斷訊 0
斷詞 0
斷語 7
斷路 111
斷送 40
斷過 11
斷開 13
斷除 21
斷電 313
斷面 167
斷面圖 12
斷頭 119
斷頭台 10
斷頭率 0
斷頭臺 3
斷魂 35
斸 10
方 15482
方丈 26
方且 13
方中 58
方人 19
方今 27
方伯 2
方位 2117
方位角 21
方便 4899
方便之門 14
方便性 118
方內 4
方公尺 31
方公里 15
方向 6278
方向感 24
方向燈 1
方向盤 555
方向線 59
方向舵 45
方圓 105
方圓十里 2
方型 128
方城 75
方城之戰 2
方城戰 0
方域 1
方塊 813
方塊圖 41
方塊字 2
方塊舞 0
方塊酥 5
方士 12
方外 13
方外之人 0
方天 1
方天戟 1
方天畫戟 1
方始 20
方子 13
方字 0
方家 51
方寸 31
方寸之間 30
方寸地 0
方寸大亂 2
方寸已亂 0
方寸無主 0
方尖碑 1
方尺 6
方差 7
方巾 20
方底圓蓋 0
方式 27527
方形 193
方從 17
方志 82
方成 19
方才 207
方整 6
方方 55
方方正正 9
方書 1
方枘圓鑿 0
方枘圓鑿 0
方根 15
方根值 17
方格 152
方格子 6
方格形 0
方格紙 13
方桃譬李 0
方框 73
方框 73
方框圖 0
方框圖 0
方框框 2
方框框 2
方案 6742
方桌 7
方正 114
方步 5
方法 18340
方法論 317
方濟會 3
方為 128
方為人上 0
方為人上人 4
方略 17
方的 489
方盒 14
方相 6
方程 104
方程式 564
方程組 14
方竹 7
方糖 16
方纔 27
方能 616
方臘 5
方興未艾 70
方舟 261
方術 6
方解石 26
方言 288
方趾圓顱 0
方輿 3
方里 16
方針 600
方鉛礦 5
方陣 32
方面 11381
方面大耳 0
方頭 8
斺 0
斻 103
於 0
於 78835
於七 231
於事無補 17
於今 264
於今猶烈 0
於左 106
於幼華 5
於心何忍 2
於思 4
於斯 61
於於 13
於是 5009
於是乎 63
於是就 170
於此 792
於法無據 7
於田 9
於飛 41
施 6251
施主 46
施予 114
施以 268
施公案 1
施力 38
施力臂 3
施力臂 3
施力點 7
施加 81
施加壓力 19
施壓 167
施壓力 4
施密特 107
施展 202
施展抱負 2
施工 4649
施工圖 61
施恩 53
施恩不圖報 0
施恩不望報 0
施惠 20
施捨 39
施放 87
施放煙火 4
施政 1541
施政報告 106
施政者 4
施救 26
施教 97
施暴 80
施法 52
施洗 29
施洗約翰 23
施為 14
施琅 20
施用 201
施禮 23
施粉 0
施給 1
施耐庵 2
施肥 119
施肥料 4
施與 30
施舍 4
施舍 4
施藥 42
施蟄存 1
施行 2248
施行細則 1633
施賑 1
施醫 18
施開 0
施齋 0
斾 0
斿 7
旁 321
旁 3321
旁人 165
旁側 7
旁出 7
旁午 0
旁及 16
旁徵 8
旁徵博引 8
旁心 1
旁敲側擊 4
旁枝 3
旁注 0
旁生 0
旁白 160
旁礡 1
旁系 23
旁系親 1
旁系親屬 0
旁聽 111
旁聽席 4
旁聽者 0
旁若無人 17
旁觀 71
旁觀者 88
旁觀者清 5
旁註 1
旁証 0
旁證 1
旁路 6
旁通 16
旁邊 1047
旁邊兒 0
旁邊兒 0
旁門 4
旁門左道 12
旁鶩 1
旂 121
旃 257
旃檀 2
旄 30
旄 30
旅 8560
旅人 150
旅伴 5
旅外 56
旅外多年 0
旅大 12
旅客 1308
旅客列車 19
旅居 108
旅居海外 16
旅店 135
旅日 48
旅社 1274
旅程 513
旅美 100
旅美多年 1
旅美學人 3
旅舍 108
旅行 2291
旅行包 11
旅行團 80
旅行支票 89
旅行社 2909
旅行者 119
旅行袋 63
旅行裝 0
旅行車 54
旅費 126
旅資 19
旅途 202
旅途愉快 21
旅遊 19696
旅遊事業 59
旅遊業 318
旅部 2249
旅長 5
旅順 37
旅館 2249
旅鼠 7
旆 137
旇 0
旈 0
旉 0
旉 0
旊 0
旋 757
旋 757
旋乾轉坤 3
旋即 101
旋宮 0
旋律 444
旋毛 1
旋毛蟲 9
旋渦 17
旋繞 3
旋繞著 0
旋翼機 5
旋踵 4
旋踵即逝 0
旋轉 1328
旋轉乾坤 2
旋轉乾坤之力 0
旋轉具 0
旋轉法 0
旋轉起重 0
旋轉起重機 0
旋轉軸 29
旋迴 0
旋鈕 102
旋風 363
旋風式 20
旌 90
旌旗 24
旌旗招展 0
旌旗蔽空 0
旌節 1
旌表 4
旍 98
旎 12
族 7655
族人 452
族兄 2
族姓 1
族弟 0
族權 1
族繁 25
族群 2585
族裔 59
族裡 17
族譜 75
族長 123
族類 24
旐 9
旑 0
旒 15
旓 27
旔 0
旖 20
旖旎 0
旗 3578
旗下 631
旗人 12
旗兒 0
旗兒 0
旗子 64
旗山 861
旗幟 342
旗幡 3
旗手 11
旗杆 6
旗桿 17
旗標 245
旗竿 7
旗艦 277
旗艦店 0
旗號 25
旗袍 29
旗袍裝 0
旗袍領 0
旗語 5
旗開得勝 25
旗魚 63
旗鼓相當 24
旘 0
旘 0
旚 41
旛 15
旜 0
旜 0
旝 4
旞 1
旟 4
无 0
无 0
旡 20
旡妄之災 0
既 2361
既不 274
既不 274
既不是 49
既來之 8
既來之則安之 6
既存 90
既定 141
既定政策 50
既定目標 18
既已 124
既往 65
既往不咎 4
既往不究 0
既得 141
既得利益 31
既得利益者 25
既成 31
既成事實 8
既是 290
既有 840
既然 1659
既然在 26
既然如此 70
既然是 141
既然會 3
既老且衰 0
既而 4
既能 82
既要 58
既非 59
旣 0
旤 0
日 79754
日系 100
日上三竿 0
日下 32
日中 209
日久 56
日久天長 1
日久生情 19
日久見人心 1
日人 221
日以繼夜 35
日來 91
日俄 41
日俄戰爭 17
日偏蝕 5
日偏食 0
日僑 17
日元 189
日光 495
日光浴 35
日光燈 145
日光療法 0
日光能 1
日入而息 2
日內 783
日內瓦 166
日內瓦人 0
日內瓦協議 0
日全蝕 2
日全食 11
日共 55
日冕 0
日出 473
日出日落 18
日出而作 4
日刊 159
日利 7
日前 3330
日劇 0
日南 51
日商 368
日圓 1201
日坐愁城 0
日報 1642
日場 2
日場 2
日夕 1
日夜 292
日夜不停 6
日夜不分 4
日子 2243
日安 112
日寇 3
日射 8
日射病 0
日就月將 0
日工 13
日常 850
日常生活 1365
日常用品 65
日幣 192
日式 1
日式料理 1
日影 9
日後 1897
日復一日 58
日復一日 58
日息 8
日戳 15
日據時代 332
日數 228
日文 2254
日文系 114
日文課 1
日斑 0
日新 738
日新又新 0
日新月異 184
日方 139
日日 247
日日夜夜 27
日日春 27
日晒 13
日晒牢度 0
日晒雨淋 1
日晷 8
日晷儀 1
日暈 1
日暖風和 2
日暮途窮 1
日曆 4335
日曜日 0
日曬 75
日曬牢度 2
日曬雨淋 14
日書 8
日月 474
日月如梭 2
日月星辰 21
日月潭 519
日月無光 4
日月爭輝 1
日月蝕 1
日月重光 0
日有所思 15
日有起色 0
日期 47392
日本 24369
日本人 628
日本化 10
日本史 16
日本式 43
日本料理 1
日本海 31
日本狗 3
日本社 11
日本話 6
日本語 614
日本隊 115
日校 47
日正當中 0
日歷 4
日沒 28
日漸 516
日漸壯大 2
日照 132
日照時間 4
日爾曼民族 1
日珥 7
日班 26
日理萬機 4
日環蝕 1
日環食 0
日甚一日 2
日甚一日 2
日產 255
日產量 5
日用 150
日用品 165
日益 1113
日益加劇 3
日益增長 3
日益壯大 10
日益減少 1
日省月試 0
日知會 0
日程 131
日程表 194
日積月累 46
日籍 95
日籍人士 2
日耳曼 17
日耳曼人 2
日臻 50
日臻完善 2
日色 7
日華 99
日落 204
日落時分 5
日薄西山 9
日薪 10
日蝕 53
日行一善 27
日行一善 27
日表 8
日裔 33
日見好轉 0
日規 21
日記 1166
日記帳 13
日記本 31
日記簿 39
日誌 1671
日語 1024
日語系 55
日語課 3
日資 43
日趨 392
日趨敗壞 1
日趨衰弱 0
日軍 245
日進 61
日進斗金 4
日鈔 0
日銀 23
日間 428
日間班 36
日間部 866
日頭 51
日頭 51
日食 54
旦 898
旦夕 8
旦夕之危 0
旦夕之間 3
旦夕禍福 0
旦旦 4
旦暮 2
旦角 13
旦角 13
旧 0
旨 1590
旨意 204
旨義 0
旨趣 34
旨酒佳肴 0
早 3435
早一些 2
早一些 2
早一點 69
早一點 69
早上 1704
早上早下 0
早下 7
早下一步 0
早下一步 0
早下早好 0
早了 19
早些 50
早以 46
早來 33
早借 0
早借早還 0
早先 85
早出 22
早出晚歸 7
早到 44
早去 28
早去早回 1
早在 741
早報 112
早場 13
早場 13
早場電影 2
早場電影 2
早夭 8
早婚 10
早婚者 0
早安 233
早就 926
早已 1736
早市 3
早年 322
早得 171
早得很 0
早慧 4
早拜 1
早操 13
早操時間 0
早於 90
早日 692
早日康復 41
早春 33
早春時節 0
早晚 227
早晚都是 0
早晚都會 3
早晨 351
早晨時 2
早會 17
早會時間 0
早有 147
早有打算 1
早有計劃 0
早朝 8
早期 2776
早歲 5
早死 22
早死早好 0
早比 2
早洩 26
早熟 69
早熟作物 0
早班 25
早班車 2
早生 13
早生貴子 9
早產 77
早產兒 181
早的 171
早睡 23
早睡早起 17
早睡早起身體好 3
早知 144
早知如此 20
早秋 2
早稻 6
早稻田 99
早粥 0
早自習 60
早自習課 0
早與 12
早衰 2
早衰現象 0
早試 0
早課 10
早讀 50
早讀課 0
早走 10
早走一步 1
早走一步 1
早起 125
早踏 0
早踏一步 0
早踏一步 0
早退 23
早逝 20
早霜 1
早飯 17
早餐 1051
早餐券 4
早餐時間 2
早點 251
早點兒 0
早點兒 0
旪 0
旫 0
旬 309
旬刊 12
旬日 4
旭 2651
旭日 54
旭日初升 1
旭日東升 5
旭日東昇 23
旮 233
旮 233
旮旯 0
旯 458
旰 56
旰食 0
旱 189
旱乾 0
旱井 0
旱地 28
旱天 1
旱季 10
旱年 0
旱情 0
旱潦 0
旱潦之災 0
旱澇保收 0
旱災 30
旱災區 1
旱煙 4
旱獺 3
旱田 32
旱相 0
旱稻 2
旱船 0
旱荒 0
旱象 14
旱路 5
旱道 0
旱鴨 0
旱鴨子 7
旲 7
旳 115
旴 4
旵 7
时 0
旹 0
旺 2296
旺地 2
旺季 355
旺市 0
旺旺 66
旺旺的 7
旺旺地 6
旺月 0
旺氣 2
旺盛 258
旺盛起來 0
旻 394
旼 60
旽 54
旾 0
旿 0
昀 209
昁 0
昁 0
昂 372
昂強 0
昂強 0
昂揚 34
昂昂 2
昂昂七尺之軀 0
昂然 36
昂然挺立 2
昂然聳立 0
昂貴 398
昂貴品 0
昂頭 2
昂首 47
昂首挺胸 0
昂首闊步 14
昃 56
昄 139
昅 30
昆 1423
昆仲 17
昆侖 4
昆季 0
昆崙 0
昆崙山 0
昆布 9
昆弟 5
昆明 105
昆明城 1
昆明市 6
昆明池 0
昆明湖 8
昆明街 51
昆曲 2
昆曲 2
昆蟲 853
昆蟲學 15
昆蟲學家 3
昆陽 43
昇 4897
昇冪 0
昇到 17
昇天 15
昇平 79
昇為 17
昇空 16
昇至 108
昇華 196
昇華作用 1
昇起 68
昇降 114
昇降機 65
昇陽 1
昇高 78
昈 77
昉 93
昊 184
昋 44
昌 7060
昌吉 33
昌平 140
昌明 64
昌盛 36
昌言 0
昌都 4
昌隆 83
昌黎文集 0
昍 60
明 28194
明主 44
明了 96
明亮 482
明亮動人 1
明人 38
明人不做 0
明人不做暗事 1
明人不說暗話 0
明仁 225
明代 186
明代人 0
明令 110
明來 33
明來明往 0
明來暗往 0
明信片 247
明修棧道 1
明倫 157
明倫國中 6
明倫國小 4
明光 82
明兒 17
明兒 17
明兒個 8
明兒個 8
明切 0
明初 34
明升暗降 1
明史 20
明哲 201
明哲保身 8
明喻 3
明器 10
明基電通 1
明大 60
明天 2774
明太祖 23
明夷待訪 0
明夷待訪錄 6
明媒正娶 2
明媚 25
明定 614
明室 19
明察 14
明察暗訪 5
明察秋毫 14
明將 40
明尼蘇大學 0
明尼蘇達 145
明尼蘇達大學 105
明尼蘇達州 37
明州 7
明帝 11
明年 6272
明度 31
明律 1
明後天 13
明徵 1
明德 929
明德國中 22
明德國小 21
明德水庫 27
明心見性 31
明志 259
明志工專 85
明志工業 0
明志工業專科學校 66
明快 74
明恥教戰 1
明成祖 14
明指 20
明搶 1
明擺 2
明擺著 1
明政 36
明教 38
明文 194
明文規定 136
明新工專 16
明斷 1
明斷是非 0
明於 25
明於觀人 0
明日 1512
明日之星 185
明日黃花 4
明早 55
明明 491
明明是 84
明明白白 35
明星 1907
明星像 1
明春 126
明時 26
明晃晃 2
明晚 133
明晨 11
明晰 36
明智 221
明智之舉 18
明暗 119
明暗不定 0
明曉得 1
明月 226
明月當空 4
明朗 196
明朗化 28
明朝 145
明朝人 2
明末 54
明末清初 34
明末農民 0
明末農民起義 0
明查暗訪 6
明棄暗取 0
明槍易躲 4
明槍易躲暗箭難防 0
明槍暗箭 0
明正典刑 0
明治 371
明治天皇 18
明治維新 37
明淨 22
明清 250
明清兩代 11
明渠 14
明湖 153
明湖國中 0
明湖國小 9
明溝 10
明溝暗渠 1
明溪 8
明滅 9
明澈 7
明澈如鏡 0
明火執仗 0
明為 55
明為 55
明燈 189
明爭 1
明爭暗鬥 4
明牌 79
明王 51
明珠 253
明珠暗投 0
明理 65
明白 2081
明白了當 0
明白人 5
明的 392
明得 392
明皎 1
明目 45
明目張膽 17
明眉皓齒 0
明眸 15
明眸皓齒 3
明眼人 54
明瞭 382
明知 346
明知故問 0
明知故犯 15
明碁電腦 1
明確 1865
明確性 8
明碼 4
明碼電報 0
明礬 34
明礬石 6
明示 152
明禮 74
明空 3
明空 3
明窗淨几 1
明細 1282
明細帳 24
明細表 473
明膠 21
明艷 10
明艷照人 1
明若 6
明若觀火 0
明蝦 62
明裡 4
明視 19
明角燈 1
明言 24
明訂 255
明証 3
明說 69
明講 13
明證 46
明辨 27
明辨是非 13
明道 189
明達 97
明鏡 128
明鏡高懸 2
明鑑 9
明降暗升 0
明顯 3786
明駝 1
明體 122
明體字 4
昏 222
昏亂 6
昏了 63
昏倒 155
昏到 3
昏厥 32
昏君 16
昏天暗地 12
昏天黑地 2
昏庸 10
昏愚 0
昏明 0
昏昏 29
昏昏暗暗 3
昏昏欲睡 8
昏昏沉沉 7
昏昧 1
昏暗 81
昏暮 2
昏死 8
昏死過去 4
昏沈 10
昏沉 11
昏眩 14
昏睡 58
昏睡不醒 0
昏瞶 2
昏聵 0
昏花 4
昏迷 225
昏迷不醒 25
昏過去 20
昏頭 20
昏頭昏腦 1
昏頭轉向 7
昏鴉 34
昏黃 34
昏黑 2
昐 97
昑 39
昒 95
易 14785
易事 1000
易主 60
易人 27
易位 15
易俗 2
易傳 14
易北河 2
易卜生 8
易地 20
易地而處 2
易如反掌 15
易姓 1
易子而食 0
易學 224
易容 26
易容術 14
易幟 13
易怒 32
易懂 154
易手 8
易於 449
易易 2
易水 79
易熔金 0
易燃 84
易燃物 20
易發難收 0
易碎 31
易碎品 0
易經 478
易經學會 5
易與 83
易見 64
易言之 32
易讀 46
易讀性 2
易變性 1
易貨 1
易開罐 23
昔 242
昔人 10
昔年 22
昔日 894
昔時 40
昔歲 0
昔為 10
昔者 14
昔陽 1
昕 526
昖 0
昗 0
昘 0
昚 0
昛 0
昜 322
昝 20
昞 0
星 14154
星主 66
星位 4
星光 646
星光點點 13
星兒 20
星兒 20
星加坡 19
星占學 0
星君 1
星團 83
星夜 73
星媽 3
星子 179
星宿 38
星宿 38
星幣 26
星座 6863
星形 33
星斗 22
星星 1277
星星之火 17
星星點點 2
星曆 3
星月 180
星月交輝 1
星期 25358
星期一 6059
星期三 5849
星期二 5663
星期五 6151
星期假日 44
星期六 5609
星期四 5863
星期天 343
星期日 4817
星條旗 5
星河 137
星洲 48
星火 33
星火燎原 1
星火燎原 1
星爸 2
星爸星媽 0
星狀 68
星狀體 1
星球 511
星球大戰 3
星相 91
星移斗換 0
星移斗轉 0
星空 492
星系 292
星系團 14
星羅棋布 7
星群 23
星芒狀 0
星號 62
星象 246
星象學 11
星辰 214
星辰錶 9
星運 12
星運亨通 1
星隕 14
星際 2213
星際大戰 118
星際飛行 1
星雲 339
星馬 37
星馬地區 2
星馳 66
星體 94
星魚 1
星點 35
映 5376
映像 171
映像管 298
映入 38
映入眼簾 44
映出 37
映在 43
映射 74
映對 5
映照 51
映照出來 0
映照在 5
映眼 1
映著 59
映襯 8
映象 124
映雪 35
昡 15
昡曜 0
昢 198
昣 0
昤 2
春 7230
春上 10
春事 1
春假 144
春假裡 0
春光 265
春光外洩 1
春光明媚 2
春分 39
春分時節 0
春化階段 0
春夢 102
春夢一場 2
春夢一場 2
春夢一場空 0
春夢一場空 0
春夢了無痕 5
春夢無痕 0
春天 1033
春天裡 5
春季 431
春季旅行 1
春季班 94
春宮 27
春宮畫 0
春宵 7
春宵一刻值千金 1
春宵一刻值千金 1
春寒 6
春寒料峭 3
春小麥 0
春心 14
春心蕩漾 2
春忙 0
春情 10
春愁 3
春意 35
春意盎然 5
春捲 25
春播 4
春日 178
春景 20
春暉 344
春暖花開 23
春曉 29
春汛 0
春牛 8
春神 13
春秋 702
春秋左氏傳 1
春秋戰國 43
春秋時代 20
春筍 9
春節 435
春節假期 17
春耕 10
春耕時節 0
春聯 28
春色 41
春色滿園 0
春花 54
春花秋月 10
春茶 22
春華 40
春華 40
春華秋實 0
春華秋實 0
春藥 65
春蘭 69
春衣 1
春裝 5
春裝展示 0
春裝展示會 0
春訊 1
春遊 15
春酒 21
春雨 95
春雨綿綿 7
春雨綿綿無絕期 1
春雷 21
春霖 30
春露 0
春風 389
春風化雨 229
春風吹又生 8
春風得意 20
春風滿面 12
春風風人 0
春餅 1
春麥 5
昦 11
昧 232
昧心 0
昧於 23
昧於事理 1
昧良心 1
昧著 2
昧著良心 11
昨 2218
昨兒 6
昨兒 6
昨兒個 17
昨兒個 17
昨夜 238
昨夜裡 0
昨天 4102
昨天上午 119
昨天下午 170
昨天中午 13
昨天夜裡 6
昨天早上 19
昨天晚上 183
昨日 2465
昨是今非 0
昨晚 687
昩 0
昪 0
昫 0
昬 0
昭 2654
昭信天下 0
昭告 140
昭和 285
昭和年間 2
昭彰 14
昭然 5
昭然若揭 10
昭示 14
昭示中外 0
昭著 3
昭雪 3
昭顯 1
昮 0
是 182405
是人 661
是不 2179
是不 2179
是不是 3462
是以 3532
是何居心 3
是何道理 0
是個 3443
是可忍孰不可忍 1
孰可忍孰不可忍 1
是否 21483
是啥 243
是嗎 660
是在 6173
是塊 27
是好 408
是想 531
是故 232
是是非非 30
是有 2124
是用 1513
是的 516
是說 1072
是誰 2415
是非 528
是非之地 5
是非心 1
是非曲直 4
是非自有公論 3
是非題 40
是非顛倒 1
昰 0
昱 1116
昲 18
昲 18
昳 62
昴 128
昵 18
昵 18
昶 539
昷 0
昸 0
昹 27
昺 42
昻 0
昼 0
晀 0
晁 439
晁 439
晁 439
時 83240
時下 241
時不可失 0
時不我予 9
時不我與 0
時不時 4
時世 6
時中 21
時事 461
時事性 2
時事評論 40
時人 41
時代 7008
時代化 1
時代周刊 8
時代性 28
時代精神 21
時代週刊 13
時令 56
時來運轉 4
時候 7103
時候未到 6
時值 55
時價 40
時光 666
時光倒流 15
時光隧道 71
時光飛逝 16
時分 339
時刻 1399
時刻表 715
時務 8
時勢 126
時勢使然 0
時勢造英雄 11
時區 30
時報 14234
時報周刊 76
時報週刊 28
時報雜誌 0
時報鷹 128
時好時壞 34
時宜 54
時尚 549
時局 33
時局多變 0
時差 123
時常 855
時序 180
時弊 10
時式 9
時態 16
時態變化 0
時政 15
時效 386
時效性 93
時方 32
時日 203
時時 476
時時刻刻 97
時會 901
時有 537
時有所聞 53
時期 2958
時框 1
時機 1250
時機不再 0
時機已到 2
時機成熟 42
時機未到 13
時段 964
時氣 10
時水 17
時浮時現 0
時疫 1
時疾 1
時症 0
時程 0
時空 858
時空旅行 0
時節 143
時羞 1
時而 225
時脈 0
時至今日 55
時興 28
時艱 20
時薪 1
時行 22
時裝 204
時裝展示會 0
時裝店 5
時裝業 36
時裝片 0
時見 22
時計 41
時計商 0
時評 7
時論 105
時諺 0
時變率 0
時輪金剛 1
時辰 176
時速 150
時運 12
時運不濟 2
時運亨通 0
時過境遷 4
時針 43
時鐘 389
時間 51398
時間常數 4
時間性 49
時間繼電 0
時間繼電器 2
時間藝術 0
時間表 1687
時限 264
時隔多年 2
時風 8
時食 3
時髦 140
時鮮 3
時點 110
晃 595
晃 595
晃了 14
晃來 2
晃來晃去 31
晃動 58
晃動起來 0
晃得 4
晃晃 117
晃晃 117
晃晃 117
晃晃 117
晃晃悠悠 1
晃朗 0
晃盪 7
晃眼 595
晃蕩 6
晃頭晃腦 0
晄 0
晄 0
晅 64
晆 0
晇 67
晇 67
晈 0
晈 0
晉 1557
晉京 0
晉代 16
晉升 214
晉國 13
晉封 7
晉惠公 0
晉文公 5
晉書 14
晉朝 8
晉朝人 0
晉武帝 1
晉江 34
晉江人 1
晉江市 1
晉爵 4
晉獻公 0
晉用 24
晉級 291
晉見 46
晉謁 3
晉身 34
晉身之階 0
晉身階 1
晉陞 10
晉陽 5
晊 90
晋 0
晌 147
晌午 2
晍 0
晎 0
晏 441
晏嬰 9
晏子 143
晏子春秋 10
晏寢 0
晏然 0
晏起 1
晏飲 0
晐 0
晑 45
晒 211
晒乾 9
晒傷 0
晒圖 56
晒圖紙 0
晒成 1
晒晒 4
晒煙 0
晒種 0
晒衣 4
晒衣服 9
晒衣架 2
晒黑 3
晗 0
晘 0
晙 155
晚 2559
晚上 3780
晚世 1
晚來 26
晚到 18
晚唐 10
晚唐時期 0
晚報 93
晚境 0
晚娘 4
晚婚 20
晚婚者 0
晚學 24
晚安 973
晚宴 188
晚年 147
晚年時期 2
晚春 33
晚春時節 0
晚景 11
晚景淒涼 4
晚暮 0
晚會 1088
晚會節目 8
晚期 170
晚歲 1
晚清 44
晚清時期 2
晚熟 7
晚熟作物 0
晚班 60
晚生 18
晚疫病 1
晚睡 30
晚福 1
晚禮服 22
晚禱 31
晚秋 12
晚秋時節 0
晚稻 0
晚節 5
晚節不保 3
晚裝 4
晚課 18
晚起 20
晚輩 89
晚近 62
晚醒 0
晚間 746
晚間新聞 70
晚霜 22
晚霞 63
晚風 72
晚風習習 2
晚飯 85
晚飯時間 1
晚餐 855
晚餐飯 0
晚香玉 0
晚點 93
晛 114
晜 127
晝 217
晝伏夜出 40
晝分 0
晝夜 91
晝夜不分 0
晝夜不息 1
晝夜匪懈 0
晝寢 1
晝日 2
晝盲 0
晝短夜長 2
晝長夜短 2
晞 185
晟 852
晠 0
晠 0
晡 365
晢 123
晢 123
晢 123
晣 0
晤 379
晤商 4
晤見 0
晤談 148
晤談甚歡 0
晤面 19
晥 91
晦 52
晦明 1
晦暗 29
晦氣 7
晦澀 28
晦跡 0
晧 0
晨 1652
晨光 73
晨光乍現 0
晨報 464
晨操 12
晨昏 32
晨昏定省 2
晨昏顛倒 0
晨星 61
晨曦 169
晨曦社 3
晨泳 10
晨禱 2
晨鐘暮鼓 2
晨間 111
晨間活動 13
晨風 12
晩 0
晪 18
晫 0
晬 11
晭 0
普 9220
普列漢諾夫 0
普利尼時代 0
普及 1291
普及本 1
普告天下 0
普天之下 12
普天同慶 32
普希金 6
普度 41
普朗克 6
普林斯頓 99
普查 301
普洱茶 1
普渡 67
普渡大學 118
普渡眾生 12
普照 107
普照大地 3
普爾 36
普物 0
普特 34
普立茲 24
普羅 222
普羅列塔利亞 0
普羅提諾 0
普考 335
普考及格 13
普賢 1
普賢菩薩 1
普通 3548
普通人 64
普通小麥 0
普通教育 81
普通法 17
普通考試 328
普通股本 0
普通話 32
普通車 40
普遍 2071
普遍 2071
普遍化 57
普遍化 57
普遍性 110
普遍性 110
普遍真理 0
普遍真理 0
普遍裁軍 0
普遍裁軍 0
普遍規律 0
普遍規律 0
普選 21
普選權 0
普里斯特 0
普里斯特萊 0
普里斯萊 1
普陀山 5
普降 1
普降甘霖 2
普魯士 15
普魯士王國 0
普魯東 1
景 1
景 11348
景仰 44
景像 45
景山 44
景德 61
景德鎮 32
景慕 2
景教 2
景氣 2447
景況 85
景泰 120
景泰藍 48
景深 45
景物 206
景物依舊 4
景狀 0
景緻 167
景美 311
景美區 19
景美溪 51
景致 103
景色 880
景色宜人 17
景行 10
景觀 3519
景象 416
景點 0
晰 588
晱 113
晲 26
晳 0
晴 8265
晴了 2
晴和 9
晴天 82
晴天霹靂 28
晴嵐 0
晴時多雲 15
晴朗 129
晴空 94
晴空萬里 24
晴轉多雲 4
晴雨表 1
晵 0
晶 3991
晶亮 21
晶內偏析 0
晶圓 0
晶圓代工 0
晶圓廠 1
晶形 24
晶核 21
晶格 129
晶片 3122
晶片卡 1
晶片組 1
晶狀體 10
晶瑩 251
晶瑩剔透 36
晶石 58
晶粒 88
晶軸 1
晶體 385
晶體三極管 0
晶體三極體 0
晶體二極管 0
晶體二極體 0
晶體振盪 0
晶體振盪器 7
晶體振蕩 0
晶體振蕩器 1
晶體管 5
晶體結構 28
晷 285
晸 81
晹 112
智 12545
智人 23
智仁勇 0
智光 50
智利 315
智利人 0
智利硝 0
智力 298
智力測驗 240
智勇 58
智勇兼備 2
智勇雙全 8
智取 18
智商 176
智商測驗 3
智囊 39
智囊團 27
智圓行方 3
智多星 32
智巧 3
智庫 0
智性 15
智愚 2
智慧 3463
智慧型 1139
智慧板 0
智慧財產 1
智慧財產局 1
智慧財產權 1177
智產權 1
智者 135
智者樂水 11
智育 83
智能 588
智能不足 173
智自在 1
智謀 75
智識 122
智識份子 0
智識份子 0
智識階級 0
智障 0
智齒 59
晻 89
晼 103
晽 0
晾 252
晾乾 27
晾完 1
晾晾 0
晾晾看 0
晾曬 4
晾著 3
晾衣 7
晾衣服 1
晾衣架 5
晾衣繩 0
晿 0
暀 69
暁 0
暄 150
暅 0
暅 0
暆 27
暇 111
暇 111
暇日 0
暇時 4
暈 250
暈 250
暈倒 54
暈厥 17
暈昏 0
暈機 5
暈眩 120
暈糊糊 0
暈船 11
暈船藥 0
暈車 38
暈車藥 5
暈過去 6
暈頭 3
暈頭轉向 17
暉 1126
暊 44
暋 182
暌 41
暌違 26
暍 12
暎 0
暏 0
暏 0
暐 246
暑 414
暑修 58
暑修班 10
暑假 1140
暑天 0
暑期 812
暑期班 41
暑期部 10
暑氣 13
暑氣逼人 1
暑濕痹 0
暑濕痺 0
暑熱 6
暑訓 7
暑訓班 0
暒 0
暓 0
暔 25
暕 43
暖 1248
暖冬 34
暖和 48
暖壺 0
暖壽 7
暖室 0
暖房 6
暖手 0
暖氣 53
暖氣團 8
暖氣機 8
暖水壺 0
暖洋洋 6
暖流 47
暖溫帶 22
暖溶溶 0
暖火 0
暖烘烘 10
暖爐 20
暖種 0
暖種 0
暖色 15
暖色調 7
暖身 0
暖酒 1
暖鋒 15
暖鍋 0
暗 12000
暗下來 12
暗下去 1
暗中 207
暗中作怪 0
暗中摸索 3
暗中行事 0
暗了 20
暗了下來 4
暗事 0
暗傷 0
暗到 0
暗叫 19
暗叫一聲 1
暗叫一聲 1
暗含 5
暗問 1
暗喜 11
暗喻 36
暗器 87
暗地 29
暗地裡 26
暗堡 1
暗室 15
暗害 1
暗射地圖 0
暗屋 1
暗巷 11
暗底下 0
暗得 0
暗得 67
暗想 22
暗戀 217
暗戀著 1
暗房 115
暗指 17
暗探 3
暗操賤業 0
暗暗 118
暗暗淡淡 0
暗殺 100
暗殺黨 0
暗沈沈 0
暗泣 1
暗流 16
暗淡 35
暗淡無光 7
暗渠 6
暗渡陳倉 2
暗溝 5
暗溝裡 0
暗潮 4
暗潮洶湧 13
暗澹 0
暗無天日 9
暗然 4
暗然失色 0
暗疾 0
暗的 67
暗盤 19
暗盼 0
暗碼 5
暗礁 6
暗示 486
暗示著 35
暗移 0
暗笑 12
暗笑一聲 0
暗算 31
暗算著 0
暗管 13
暗箭 4200
暗箭傷人 7
暗箭難防 7
暗箱 1
暗紅 31
暗紅色 33
暗線 2
暗練 0
暗罵 12
暗自 105
暗自流淚 0
暗自神傷 4
暗色 31
暗色彩 1
暗藏 74
暗藏春色 1
暗藏玄機 8
暗藏鬼胎 0
暗處 57
暗號 35
暗號燈 0
暗裡 9
暗褐 0
暗褐色 22
暗計 0
暗記 10
暗記在心 0
暗訪 4
暗語 7
暗送 2
暗送秋波 0
暗送秋波 0
暗通 5
暗通款曲 5
暗道 52
暗邊 0
暗電流 6
暗鬥 4
暗黝 1
暘 280
暙 21
暚 0
暛 0
暜 0
暝 38
暝 38
暝想 1
暝色 0
暞 0
暟 21
暠 4
暡 7
暢 388
暢心 0
暢快 74
暢懷 5
暢所欲為 0
暢所欲言 99
暢敘 2
暢旺 65
暢月 0
暢流 5
暢游 8
暢盛 1
暢茂 1
暢行 16
暢行無阻 20
暢談 185
暢通 264
暢通無阻 20
暢遊 412
暢達 17
暢銷 1552
暢銷書 2463
暢銷貨 0
暢飲 42
暣 0
暤 0
暤 0
暥 0
暦 0
暨 11448
暩 38
暪 104
暫 14
暫 42000
暫且 60
暫且不說 2
暫付 34
暫代 31
暫住 33
暫依 7
暫候 0
暫借 32
暫停 1003
暫停比賽 0
暫停營業 49
暫停鈕 1
暫別 20
暫升 1
暫存 148
暫存區 245
暫存器 121
暫存體 0
暫定 165
暫居 75
暫延 1
暫得 3
暫收款 2
暫放 8
暫於 1
暫時 2086
暫時性 180
暫歇 15
暫為 3
暫無 66
暫用 7
暫留 21
暫睡 0
暫緩 307
暫緩執行 6
暫緩實行 0
暫與 1
暫行 329
暫行條例 114
暫訂 61
暫記 0
暫調 7
暫贏 0
暫跑 0
暫開 2
暫降 0
暫離 12
暫領 8
暬 0
暬 0
暭 0
暭 0
暮 567
暮住 0
暮境 1
暮存 0
暮年 6
暮春 16
暮景 3
暮歲 0
暮氣 4
暮氣沉沉 4
暮氧沉沉 0
暮秋 6
暮節 0
暮色 63
暮色朦朧 0
暮色蒼茫 5
暮艾 0
暮量化 0
暮靄 6
暮鼓晨鐘 14
暮齡 0
暯 38
暰 68
暱 681
暱交 0
暱友 0
暱愛 0
暱稱 10936
暲 59
暳 0
暴 1346
暴 1346
暴亂 116
暴利 64
暴力 1503
暴力革命 2
暴動 1013
暴卒 1
暴君 36
暴富 3
暴徒 54
暴怒 6
暴怒之下 0
暴戾 35
暴戾之氣 20
暴戾恣睢 0
暴投 5
暴政 49
暴斃 39
暴晒 2
暴曬 1
暴死 2
暴殄天物 6
暴民 137
暴漲 107
暴漲暴跌 4
暴烈 5
暴牙 13
暴病 1
暴發 40
暴發出來 0
暴發戶 13
暴笑 13
暴脾氣 0
暴虎憑河 1
暴虎馮河 5
暴虐 8
暴虐無道 1
暴行 117
暴行 117
暴跌 72
暴跳 1
暴跳如雷 2
暴躁 40
暴雨 112
暴雷 6
暴露 769
暴露出 56
暴露在 173
暴露狂 6
暴風 95
暴風半徑 12
暴風圈 38
暴風雨 81
暴風雪 21
暴食暴飲 1
暴飲暴食 23
暴龍 103
暵 73
暶 0
暷 147
暸 23
暸解 140
暹 77
暹羅 26
暹羅人 1
暹羅灣 1
暹羅貓 1
暺 97
暻 93
暼 0
暽 101
暾 96
暿 0
曀 167
曀 167
曁 0
曂 0
曃 0
曄 254
曅 0
曅 0
曆 729
曆數 1
曆書 3
曆法 62
曇 54
曇花 30
曇花一現 44
曇花一現 44
曈 122
曉 3491
曉事 0
曉以大義 7
曉得 480
曉得 480
曉悟 2
曉明 94
曉月殘星 0
曉色 1
曉行夜宿 0
曉諭 0
曉譬 0
曉陽 54
曉風殘月 6
曊 139
曋 96
曌 73
曍 0
曍 0
曎 0
曏 100
曐 0
曑 0
曑 0
曑 0
曑 0
曒 8
曓 0
曓 0
曔 0
曕 0
曖 124
曖昧 200
曖昧行為 0
曗 0
曗 0
曘 20
曙 227
曙光 229
曙光乍現 13
曙曦 2
曙色 5
曚 26
曚曚曨曨 0
曚曨 0
曛 9
曜 699
曜 699
曝 75
曝光 708
曝光表 2
曝光計 0
曝晒 11
曝曬 90
曝氣 55
曝氣槽 9
曝獻 0
曝衣 2
曝露 111
曝露出 1
曞 2
曟 0
曠 185
曠世 37
曠世不羈 0
曠世之才 0
曠世奇才 1
曠世逸才 0
曠代 3
曠古未有 0
曠古未聞 0
曠古絕今 1
曠地 3
曠士 0
曠夫 0
曠工 0
曠廢 6
曠日持久 3
曠時日久 0
曠男 0
曠職 31
曠課 94
曠課日數 1
曠課時數 6
曠課記錄 1
曠費 0
曠達 8
曠野 132
曡 0
曡 0
曣 7
曤 4
曥 0
曦 122
曧 0
曨 5
曩 45
曩昔 0
曪 0
曫 10
曬 298
曬乾 41
曬圖 8
曬圖紙 0
曬場 4
曬場 4
曬太陽 63
曬成 16
曬曬 12
曬煙 0
曬種 0
曬衣 8
曬衣服 4
曬衣架 4
曬衣繩 0
曬鹽 3
曬黑 18
曭 133
曮 84
曯 0
曰 2955
曱 0
曲 4487
曲 4487
曲名 43
曲子 294
曲尺 1
曲度 14
曲式 48
曲徑 5
曲徑通幽 8
曲意 11
曲意奉承 0
曲意逢迎 1
曲折 178
曲折離奇 12
曲拐 0
曲曲 27
曲曲 27
曲曲折折 6
曲本 3
曲柄 45
曲棍球 115
曲棍球員 0
曲棍球賽 2
曲水 8
曲水流觴 4
曲牌 73
曲率 21
曲球 1
曲目 0
曲直 8
曲笛 4
曲終人散 26
曲線 788
曲線 788
曲線圖 77
曲線板 1
曲線美 22
曲線運動 2
曲繞 0
曲肱為枕 0
曲肱而枕 2
曲肱而枕 2
曲背 2
曲膝禮 0
曲臂 1
曲臂 1
曲菌 0
曲藝 58
曲解 63
曲調 169
曲譜 47
曲軸 28
曲阜 9
曲阜縣 1
曲陽 2
曲霉 1
曲面 173
曲高和寡 11
曲高和寡 11
曳 239
曳 239
曳光彈 1
曳光彈 1
曳引機 18
曳引機 18
曳影 0
更 27
更 2790
更 27903
更上 123
更上一層 26
更上一層 26
更上一層樓 110
更上一層樓 110
更不 650
更不 650
更不是 87
更不會 74
更不能 83
更事 2
更事 2
更以 198
更低 115
更使 266
更依 3
更像 77
更僕難數 0
更其 3
更再 7
更出 15
更到 8
更刺 0
更加 2092
更動 376
更動表 5
更勝一籌 14
更勝一籌 14
更卒 0
更名 421
更向 23
更圓 10
更在 234
更壞 25
更多 5113
更夫 4
更好 1871
更始 2
更姓 0
更審 0
更寬 19
更將 308
更少 96
更已 35
更年 16
更年期 190
更張 2
更得 44
更想 62
更慢 14
更應 380
更應該 131
更把 38
更換 1596
更改 2249
更新 1640
更新 16407
更新世 17
更新改造 2
更新改造資金 0
更新期 0
更易 130
更是 3885
更替 51
更會 178
更有 1908
更次 4
更正 1393
更沒 60
更沒有 94
更深 574
更深一層 101
更深一層 101
更深人靜 0
更深夜靜 0
更漏 6
更為 1522
更甚於 43
更生 334
更由 95
更番 0
更當 17
更直 39
更緊 22
更美 191
更者 2
更能 1043
更衣 44
更衣室 40
更要 528
更該 35
更說 25
更說是 0
更近 30
更迭 58
更進 82
更進一步 761
更進一步 761
更遠 106
更重 768
更長 72
更難 201
更需 363
更須 103
更願 22
更願 22
更願意 17
更香 6
更高 776
更黏 1
更鼓 2
曵 0
曵 0
曶 29
曷 31
曷不 3
曷極 0
曷若 0
書 27993
書上 331
書不盡 1
書信 242
書信來往 1
書信往來 6
書信往返 5
書僮 18
書內 124
書冊 38
書函 109
書刊 974
書刊雜誌 12
書包 168
書卡 27
書卷 80
書卷氣 8
書卷獎 0
書名 2399
書名號 0
書吏 2
書呆 4
書呆子 36
書商 156
書單 160
書坊 786
書城 45
書報 198
書報攤 33
書報雜誌 55
書塾 0
書夾 2
書契 2
書套 11
書學 11
書家 6
書寫 849
書寫能力 9
書局 2142
書屋 195
書展 261
書店 2793
書庫 546
書後 142
書房 616
書扉 2
書攤 10
書明 76
書本 239
書本上 33
書本費 2
書札 10
書架 179
書架上 47
書柬 1
書案 5
書桌 101
書桌上 32
書桌下 1
書樓 17
書櫃 124
書櫥 6
書法 1550
書法家 39
書法課 2
書獃子 1
書生 157
書生之見 0
書生本色 2
書生氣 0
書畫 448
書畫家 33
書癡 0
書皮 13
書目 3500
書眉 0
書稿 8
書空咄咄 2
書篋 1
書簡 26
書籍 20549
書籍費 28
書籤 507
書經 36
書聖 3
書舖 24
書虫 0
書蟲 86
書蠹 2
書記 231
書記官 60
書評 440
書評人 1
書迷 48
書院 372
書面 1531
書面上 5
書面報告 106
書面資料 157
書頁 29
書館 471
書香 1000
書香世家 4
書香社會 30
書香門第 4
書體 18
書齋 23
書齡 3
曹 2172
曹丕 21
曹參 0
曹操 491
曹操就到 1
曹族 28
曹植 12
曹汝霖 0
曹爽 12
曹福田 0
曹禺 1
曹軍 21
曹錕 1
曹雪芹 24
曹魏 7
曼 3764
曼哈頓 0
曼哈頓島 2
曼延 2
曼梏 0
曼特寧 57
曼衍 0
曼谷 1069
曼谷人 0
曼谷市 5
曼陀羅 10
曼陀鈴 2
曼麗 31
曾 16429
曾 19479
曾任 1262
曾國藩 12
曾在 1236
曾子 25
曾孫 8
曾孫女 3
曾幾何時 62
曾文 431
曾文正公 0
曾文水庫 281
曾文溪 140
曾於 353
曾是 279
曾有 771
曾母暗沙 5
曾祖 14
曾祖母 6
曾祖父 11
曾經 3618
曾經來 4
曾經到 15
曾經去 19
曾經在 315
曾經是 162
曾經有 200
曾經滄海難為水 19
替 3026
替人 52
替人家 3
替代 1108
替代品 719
替代役 1
替夫 1
替工 0
替手 0
替換 183
替換品 0
替換法 0
替死 2
替死鬼 13
替父 11
替用 5
替用品 1
替罪 7
替補 55
替身 46
替過 2
最 37526
最上 51
最上面 164
最下 23
最下策 0
最下面 49
最不 298
最不 298
最不想 50
最不會 8
最低 2482
最低價 201
最低氣溫 5
最低溫 30
最低溫度 5
最低限 10
最低音 3
最低點 58
最佳 11577
最佳人選 20
最佳化 100
最佳解 6
最佳選擇 284
最像 41
最優 121
最先 205
最初 676
最初幾個月 0
最初幾天 2
最初幾年 1
最前 35
最前面 165
最厲害 63
最叫 5
最叫人 4
最可 190
最可以 8
最圓 10
最基本 381
最壞 94
最壞的是 2
最多 2647
最多不超過 11
最多不過 0
最夠 2
最大 9030
最大值 105
最大平飛速度 0
最大限度 19
最好 3774
最好在 119
最好是 523
最好會 1
最好有 79
最好的 2042
最好地 2042
最好的是 20
最好能 277
最妙 10
最妙的是 4
最小 1001
最小值 49
最少 603
最少在 5
最巧 1
最巧的是 0
最常 668
最底部 5
最廣 96
最後 17377
最後勝利 18
最後審判 12
最後通牒 17
最後面 40
最得 11
最快 1083
最怕 189
最惠國 7
最惠國待遇 47
最想 276
最愛 1922
最慢 51
最新 14551
最新報導 218
最新消息 6432
最早 1007
最早到 2
最早在 18
最是 73
最晚 48
最晚到 2
最晚在 3
最最 181
最會 65
最有 837
最毒 4
最毒婦人心 9
最沒 37
最沒有 32
最渴 1
最為 868
最為人 71
最甜 18
最省 29
最短 362
最硬 11
最糟 34
最糟糕 26
最終 981
最終傳動 3
最終目標 88
最終目的 100
最給 1
最美 406
最肯 1
最能 311
最能夠 22
最舊 13
最裡面 10
最讓 99
最讓人 38
最貴 50
最輕 53
最近 14013
最近幾個月 27
最近幾天 35
最近幾年 159
最遠 74
最遠到 1
最適合 518
最遲 91
最遲在 10
最重 3084
最長 384
最長 384
最難 181
最難得 18
最需 147
最需要 116
最須 10
最高 5775
最高人民 40
最高人民法院 55
最高國務 0
最高國務會議 0
最高學府 26
最高峰 93
最高氣溫 5
最高法院 295
最高溫 29
最高潮 53
最高級 103
最高點 222
朁 34
會 1
會 1
會 1
會 81130
會上 540
會上來 12
會上去 6
會下 99
會下來 1
會下去 1
會不會 1747
會中 17
會中 1775
會了 96
會付 11
會佔 23
會來 287
會個 15
會停 30
會兒 38
會兒 38
會典 3
會出 260
會出來 10
會出去 1
會分 50
會刊 291
會到 177
會加 72
會動 147
會務 736
會升 11
會卡 13
會厭 5
會厭軟骨 2
會去 676
會友 244
會受 375
會叫 106
會合 146
會合處 2
會合點 2
會同 753
會員 42372
會員制 1315
會員國 491
會員團 0
會員大會 672
會員證 89
會唱 47
會商 241
會問 200
會在 2536
會址 323
會堂 115
會報 760
會場 112
會場 1123
會填 1
會壞 23
會夠 1
會好 93
會子 6
會子 6
會存 22
會客 55
會客室 229
會審 139
會將 1546
會對 1059
會師 161
會帳 2
會幹 9
會底銀 0
會廳 10
會彈 18
會後 607
會得 238
會得 2715
會心 58
會心一笑 47
會心微笑 4
會恨 10
會想 285
會意 33
會成 137
會戰 96
會所 594
會打 138
會扣 6
會抬 1
會拜 10
會指 160
會挑 51
會掉 43
會搞 17
會播 9
會播 9
會收 121
會改 52
會教 101
會於 552
會是 971
會是在 7
會晤 727
會會 199
會有 1244
會服 18
會期 572
會款 10
會比 674
會混 9
會演 43
會為 788
會理 47
會生 60
會由 339
會畫 28
會當 96
會當 96
會發 210
會的 2715
會盟 2
會盡 91
會看 323
會眾 116
會社 99
會種 2
會稽 0
會稿 5
會站 15
會章 28
會管 22
會籍 79
會約 12
會紅 29
會經 51
會編 75
會考 121
會聚 33
會聽 100
會腳 10
會與 1106
會萃 2
會著 18
會著 18
會著 18
會裡 90
會要 165
會見 238
會規 5
會計 6061
會計分錄 2
會計制度 133
會計報表 20
會計室 591
會計師 1305
會計年度 547
會計憑証 0
會計核算 3
會計簿 8
會計系 405
會計部 21
會診 121
會試 12
會話 436
會話室 0
會說 504
會說會寫 0
會調 27
會調 27
會談 1488
會講 99
會議 14188
會議室 2029
會議廳 537
會議記錄 1130
會費 498
會賬 0
會賺 11
會贏 99
會起 39
會起來 10
會趕 10
會跟 243
會跳 42
會車 6
會轉 89
會轉 89
會辦 77
會通 190
會進 70
會進來 4
會進去 1
會過 40
會過來 1
會過去 22
會選 118
會醉 9
會鈔 0
會銀 1
會錢 21
會錯 50
會錯意 28
會長 1926
會關 17
會附 15
會陰 45
會集 53
會靠 7
會面 201
會面處 0
會頭 24
會頭銀 0
會餐 24
會館 916
會首 57
會騎 17
會點 27
會黨 13
會齊 9
朄 5
朅 34
朆 0
朇 0
朇 0
月 1
月 77905
月下 140
月下老人 10
月下花前 0
月中 299
月之女神 1
月事 15
月亮 743
月亮蝦餅 1
月份 4816
月俸 23
月偏蝕 0
月偏食 0
月光 1132
月光曲 1
月兒 44
月兒 44
月入 80
月入數萬 5
月全蝕 0
月全食 0
月出 57
月分 52
月刊 3803
月初 1047
月利 1
月利率 4
月前 0
月台 279
月台票 2
月圓 172
月圓月缺 0
月圓花好 1
月報 378
月夜 113
月子 28
月子 28
月季花 1
月宮 5
月宮寶盒 0
月宮魔盒 0
月尾 8
月工 16
月底 2573
月形 6
月息 4
月明 28
月明星稀 3
月暈 4
月曆 427
月曜日 0
月曜日 0
月末 21
月桂 44
月桂冠 5
月桂冠 5
月桂樹 1
月桂花 0
月氏 1
月滿 27
月滿則虧 0
月牙 27
月球 403
月球上 33
月琴 49
月白風清 1
月盈 6
月盈月缺 0
月眉 136
月神 7
月票 30
月租費 0
月終 8
月給 7
月給 7
月經 573
月經來潮 17
月缺 6
月缺花殘 0
月老 17
月考 676
月臺 25
月臺票 0
月色 89
月色迷濛 0
月華 38
月薪 199
月虛 0
月蝕 26
月費 1
月輪 36
月頭 2
月食 46
月餅 191
月餅盒 0
月黑風高 0
月齡 39
有 15
有 153094
有娠 100
有史以來 100
有一套 214
有一套 214
有一手 10
有一手 10
有上 150
有下 101
有了 2071
有事 452
有事情 25
有事沒事 53
有些 6670
有些事 97
有些人 890
有些兒 2
有些兒 2
有人 2878
有人說 659
有仇 20
有仇不報非君子 1
有仇必報 2
有件 15
有件事 49
有份 74
有伴 21
有位 394
有何 1716
有何不可 31
有何指教 4
有何面目 1
有使 83
有來有往 6
有例可循 2
有例在先 0
有個 1929
有個人 125
有備 22
有備無患 17
有備而來 0
有傷 24
有傷和氣 0
有傷風化 4
有價 93
有價証券 5
有價證券 482
有光 69
有其 1193
有其師 2
有其父 1
有其父必有其子 7
有出入 16
有出息 7
有分 163
有分 163
有利 1033
有利可圖 53
有利必有弊 2
有利於 275
有利有弊 6
有到 63
有到過 7
有刺 29
有力 404
有力出力 8
有功 217
有功在身 0
有助 495
有助於 1481
有勁 53
有勇無謀 5
有勞 37
有包 14
有去 286
有去無回 8
有口 23
有口無心 1
有口皆碑 27
有口難言 16
有句 75
有句話 24
有司 30
有吃 94
有吃有喝 1
有吃有穿 1
有名 750
有名有姓 0
有名無實 11
有否 284
有味 26
有喜 32
有增無減 29
有夠 302
有夫之婦 13
有失 97
有失身份 0
有失體統 0
有好些 8
有好有壞 11
有如 804
有如神助 4
有始有終 18
有始無終 1
有婦之夫 14
有子萬事足 1
有孕 7
有字 39
有完沒完 3
有客 19
有害 892
有害於 16
有家 106
有家可歸 2
有家歸不得 9
有容乃大 14
有對 288
有層 9
有山有水 11
有巢氏 21
有幫助 265
有年 37
有幸 181
有底 9
有形 237
有待 534
有待努力 34
有待商榷 16
有待解決 11
有得 152
有得有失 13
有德 53
有心 692
有心人 169
有心人士 133
有志 385
有志一同 55
有志一同 55
有志之士 50
有志於 83
有志氣 12
有志竟成 4
有志者事竟成 5
有性 119
有性生殖 11
有恃無恐 10
有恆 62
有恆心 8
有情 704
有情人 72
有情人終成眷屬 55
有意 1891
有意思 89
有意者 216
有意識 42
有感 221
有感地震 1
有感於 144
有感而發 86
有憑有據 1
有成 285
有成就 38
有成果 14
有我無敵 1
有所 2078
有所不同 238
有所不為 14
有所不知 13
有所思 4
有所為 18
有才 158
有找 106
有把握 78
有損 99
有損於 1
有效 7158
有效值 1
有效功率 2
有效性 125
有效數字 5
有效期 122
有效期間 212
有效率 14
有效面積 5
有救 73
有教無類 60
有敵無我 0
有數 310
有數 310
有方 50
有時 3025
有時候 980
有時在 55
有時是 122
有時會 265
有時有 19
有時能 3
有朋自遠方來 1
有望 102
有朝一日 90
有朝一日 90
有期 29
有期徒刑 438
有板有眼 7
有染 17
有案 144
有案可查 0
有案可稽 1
有條 61
有條不紊 26
有條不紊 26
有條有理 5
有槍 10
有樣 28
有樣學樣 12
有機 2323
有機化合 2
有機化合物 156
有機化學 271
有機可乘 22
有機合成 71
有機物 472
有機玻璃 5
有機碳 28
有機肥料 48
有機質 117
有機酸 40
有機體 33
有權 589
有權有勢 5
有此 540
有此一說 38
有此一說 38
有殼 12
有殼蝸牛 7
有毒 546
有毒蛇 1
有比 162
有氣 49
有氣沒力 1
有氣無力 22
有氧 77
有氧舞蹈 35
有氧運動 51
有水 148
有求 22
有求必應 41
有求於人 2
有沒有 4526
有源 8
有為 147
有為有守 9
有為者 8
有無 1741
有無相生 1
有無相通 1
有熊 6
有片 14
有獎 127
有獎問答 49
有獎徵答 248
有理 178
有理數 5
有理方程 0
有理說不清 6
有理走遍天下 2
有理走遍天下 2
有生之年 41
有生以來 24
有生力量 0
有產 19
有產者 2
有產階級 6
有用 966
有用功 2
有異 1800
有當 113
有病 543
有的 28
有的 2878
有的放矢 1
有的是 236
有益 1890
有益於 75
有目共睹 87
有目共賞 0
有眉目 12
有看 153
有看頭 26
有眼不識 2
有眼不識泰山 5
有眼無珠 7
有礙 129
有礙觀瞻 13
有礙通行 0
有神 133
有神論 3
有福 270
有福同享 7
有禮 144
有種 303
有種人 6
有空 2407
有空時 61
有管腺 0
有節 13
有篇 16
有約在先 1
有素 13
有絲分裂 48
有線 853
有線廣播 8
有線廣播 8
有線電視 1468
有緣 407
有緣人 125
有緣千里來相會 16
有缺 51
有缺點 5
有罪 157
有罪之身 0
有聲 880
有聲有色 46
有聲電影 2
有能 66
有能力 441
有腳 41
有舊 40
有船 15
有色 87
有色人種 4
有色眼光 8
有色眼鏡 2
有色金屬 34
有苦 42
有苦說不出 4
有苦難言 17
有著 165
有著 1655
有著落 12
有虧職守 5
有血有淚 5
有血有肉 16
有行無市 6
有規則 18
有角 20
有解 67
有言 37
有言在先 2
有計劃 118
有話好說 5
有誤 296
有說 162
有說有笑 23
有課 22
有談 12
有請 91
有謂 18
有識 13
有識之士 50
有識者 6
有讀 13
有負眾望 0
有賞 13
有賴 292
有賺 18
有賺就好 0
有賺有賠 2
有起色 17
有趣 1953
有跡可循 9
有路 90
有路可走 6
有身 11
有車 86
有車階級 4
有軌 2
有軌電車 2
有輸有贏 5
有進 149
有進來 2
有進去 1
有進有出 1
有過 474
有過之而無不及 39
有過則改 0
有道 67
有道是 15
有道理 64
有違常情 1
有邊 24
有邊讀邊 0
有酒 42
有錢 544
有錢人 104
有錢出錢 10
有錢有閒 11
有錢能使鬼推磨 8
有錯 270
有錯則改 0
有鑑於 353
有鑑於此 223
有鑒於此 30
有閒 46
有間 63
有間 63
有關 23670
有關方面 19
有關於 632
有限 2908
有限公司 49714
有隙可乘 2
有隻 43
有難 39
有難同當 2
有雨 51
有零 16
有靈 20
有面 5
有頃 0
有頭 24
有頭有尾 1
有頭有臉 4
有頭有臉 4
有頭無尾 1
有餘 147
有餘涅槃 1
有首 15
有首歌 5
有驚無險 26
有鬼 50
有魚有肉 0
有點 4688
有點兒 55
有點兒 55
有耳無嘴 100
朊 67
朋 1427
朋儕 0
朋分花用 0
朋友 12635
朋友有通財之義 1
朋比為奸 2
朋輩 1
朋馳 63
朋黨 9
朋黨為奸 0
朌 0
服 6131
服下 31
服下去 0
服不服 1
服了 57
服事 559
服人 43
服侍 49
服兵役 132
服刑 55
服到 1
服務 102864
服務台 616
服務周到 6
服務員 236
服務性 190
服務態度 169
服務所 143
服務收入 31
服務於 204
服務業 3110
服務生 127
服務社 377
服務站 999
服務處 1318
服務行業 8
服務費 1
服務質量 5
服勞役 5
服勤 64
服喪 5
服好 0
服帖 0
服式 4
服役 488
服得 1
服從 190
服從性 5
服從於 4
服從為負責之本 4
服掉 0
服毒 6
服毒自殺 2
服毒自盡 1
服氣 47
服法 12
服滿 7
服用 1348
服用藥 0
服罪 2
服老 14
服膺 20
服色 3
服著 1
服藥 302
服裝 1512
服裝店 26
服裝秀 43
服裝費 7
服貼 16
服起 0
服輸 17
服過 7
服食 29
服飾 1985
服飾品 88
服飾店 147
朎 0
朏 13
朐 4
朒 50
朓 37
朔 638
朔方 2
朔日 1
朔月 1
朔望 3
朔望月 2
朔氣 1
朔邊 0
朔風 1
朔風凜冽 0
朕 179
朕兆 0
朖 0
朗 1286
朗斯大教堂 0
朗朗 28
朗朗上口 18
朗格 4
朗爽 0
朗詠 1
朗誦 51
朗誦會 1
朗讀 99
朗讀比賽 47
朘 54
朙 0
朚 0
望 4501
望你 44
望你早歸 2
望其項背 17
望出去 4
望到 4
望向 115
望塵 6
望塵不及 0
望塵莫及 31
望外 2
望女 4
望女成鳳 11
望子 5
望子成龍 12
望安 172
望彌撒 6
望文生義 15
望族 18
望日 5
望月 59
望望 34
望梅 3
望梅止渴 12
望樓 5
望樓台 0
望洋 6
望洋興嘆 2
望洋興歎 1
望眼欲穿 6
望眼鏡 10
望穿 8
望穿秋水 13
望而 12
望而卻步 13
望而生畏 4
望聞問切 4
望著 662
望見 92
望遠 69
望遠鏡 702
望鄉 12
望門投止 0
望風 5
望風披靡 3
望風披靡 3
望風撲影 0
望風而逃 0
望風逃竄 0
朜 0
朝 5860
朝 5860
朝三暮四 5
朝上 59
朝下 57
朝不保夕 4
朝九晚五 1
朝代 144
朝令夕改 7
朝令暮改 0
朝儀 2
朝前 12
朝北 20
朝北邊 1
朝北面 0
朝南 27
朝南邊 0
朝南面 0
朝右 5
朝右邊 0
朝向 667
朝四暮三 5
朝報 0
朝夕 28
朝夕相處 17
朝外 6
朝天 45
朝宗 16
朝山 49
朝左 6
朝左邊 0
朝廷 109
朝後 4
朝思暮想 13
朝拜 23
朝政 15
朝日 110
朝日 110
朝日新聞 97
朝旭 4
朝暉 19
朝曦 3
朝會 68
朝會時間 3
朝服 6
朝朝 7
朝朝夕夕 0
朝朝暮暮 11
朝東 30
朝東北 1
朝東南 7
朝東邊 0
朝東面 0
朝氣 96
朝氣蓬勃 17
朝獻 2
朝生暮死 3
朝發夕至 0
朝秦暮楚 2
朝綱 2
朝聖 80
朝聖團 3
朝聘 0
朝聞道夕死可矣 1
朝臣 8
朝著 210
朝西 18
朝西北 5
朝西南 2
朝西邊 0
朝西面 0
朝見 6
朝覲 2
朝貢 19
朝貴 79
朝野 795
朝野兩黨 5
朝陽 876
朝霞 54
朝露 10
朝鮮 113
朝鮮 113
朝鮮人 6
朝鮮半島 97
朝鮮半島 97
朝鮮族 18
朞 0
朞 0
朞 0
期 27
期 66100
期中 587
期中考 478
期刊 8069
期刊室 168
期初存貨 9
期勉 0
期待 4260
期待著 66
期收 3
期數 236
期日 370
期望 2080
期望值 0
期期艾艾 0
期末 1245
期末考 634
期滿 613
期盼 810
期票 8
期約 84
期終 8
期終考 19
期考 42
期考成績 0
期許 660
期貨 4720
期間 7123
期限 3193
朠 56
朡 5
朢 41
朣 86
朤 0
朦 16
朦朦 6
朦朦朧朧 7
朦朧 132
朦混 0
朦蔽 0
朦騙 0
朧 26
木 16275
木主 21
木乃伊 33
木人 28
木人石心 0
木作 61
木俑 0
木偶 199
木偶戲 11
木偶戲紮 0
木像 2
木刻 48
木刻 48
木刻術 0
木刻術 0
木劍 14
木匠 63
木器 245
木器店 10
木塊 18
木塞子 0
木壞山頹 0
木屋 171
木屐 40
木屑 124
木履 0
木工 362
木已成舟 4
木床 29
木廠 5
木房 0
木排 8
木料 61
木料行 3
木星 236
木曜日 0
木本 27
木本植物 21
木本水源 1
木材 946
木材乾餾 0
木材業 56
木材行 77
木杖 1
木杵 2
木板 162
木板凳 0
木板地 2
木板床 1
木板牆 1
木林 41
木架 15
木柴 10
木柵 752
木柵動物園 0
木柵區 19
木柵國中 11
木柵國小 10
木栓 1
木栓層 0
木栺 0
木桶 60
木桿 50
木條 31
木梯 1
木棉 82
木棉花 31
木棍 26
木棒 32
木棚 0
木棺 3
木椅 10
木椿 1
木業 470
木槌 6
木槳 2
木槿 6
木樁 10
木樂 0
木樨 5
木欄 2
木炭 33
木炭畫 1
木然 16
木片 19
木版 6
木版畫 3
木牌 6
木牛流馬 0
木犀 16
木琴 19
木瓜 183
木瓜樹 2
木瓜溪 19
木石 20
木石心腸 1
木碗 4
木筏 4
木筒 1
木箭 2
木箱 108
木簡 1
木簫 0
木籠 1
木精 13
木紋 0
木結構 12
木耳 36
木舟 3
木船 84
木花 44
木薯 10
木蘭 104
木蘭花 9
木蘭軍 0
木表法 0
木製 316
木製品 137
木訥 16
木訥 16
木賊 11
木質 207
木質素 13
木質部 4
木造 73
木部 2
木鐸 14
木雕 253
木雕像 2
木雕泥塑 2
木雞 2
木鞋 8
木頭 173
木頭人 24
木馬 265
木馬屠城 1
木馬屠城計 1
木魚 96
木麻黃 72
未 5705
未上 2774
未下 16
未之有焉 0
未亂 1
未了 161
未了情 2
未了緣 8
未亡 1
未亡人 5
未付 29
未來 18677
未來式 12
未來派 5
未便 12
未免 138
未免太 85
未免太不 3
未免太有 5
未免太沒 0
未再 56
未出 157
未出來 3
未出去 0
未分 65
未判 3
未到 289
未到人數 0
未卜先知 10
未去 13
未取 112
未受 225
未受到 23
未可 45
未可厚非 0
未含 43
未含稅 10
未問 1
未嘗 115
未嘗不可 19
未央 47
未妥 4
未始 20
未婚 508
未婚夫 33
未婚妻 61
未婚者 30
未嫁 11
未嫁新娘 0
未完 444
未完待續 13
未定 385
未定之天 0
未定數 0
未帶 33
未幾 17
未從 16
未必 622
未必盡然 2
未成 132
未成年 333
未成熟 47
未戰 9
未戰先敗 0
未戰先衰 0
未戰先退 0
未戴 22
未扣 6
未抄 1
未把 7
未抵 3
未接 82
未換 5
未教 0
未教 0
未曾 791
未曾在 15
未曾有 20
未有 413
未本著 0
未果 0
未查 9
未歸 31
未歸人 3
未決 44
未滿 990
未演 19
未漲 5
未漲反跌 0
未炒 0
未然 17
未熟 9
未獲 231
未獲得 43
未留 11
未留下 5
未盡事宜 136
未看 29
未知 524
未知數 44
未稅 171
未竟 34
未竟之志 3
未算 4
未結 33
未經 3438
未置 10
未置可否 5
未老先衰 2
未聞 33
未能 1684
未能免俗 0
未能夠 4
未虧 3
未見 316
未見好轉 9
未見得 5
未見得 5
未見有人 1
未覺 3
未詳 21
未跌 10
未跌反漲 1
未辦 202
未退 24
未退反進 0
未通 93
未進 71
未遂 86
未遞 0
未還 46
未附 77
未降 6
未降反升 0
未雨綢繆 67
未領 88
末 3831
末世 295
末了 47
末代 97
末代皇帝 9
末位 16
末兒 0
末兒 0
末學 23
末尾 19
末席 0
末年 73
末座 2
末技 2
末日 114
末日審判 4
末期 410
末末尾兒 0
末末尾兒 0
末枝 1
末梢 58
末段 19
末泥 47
末流 7
末班 8
末班車 14
末端 185
末等 2
末節 5
末葉 17
末著 2
末藥 0
末行 2
末行 2
末路 21
末那識 1
本 73202
本世紀 245
本主 47
本乎 9
本事 700
本人 4178
本以 3713
本件 310
本任 2
本份 60
本位 263
本位主義 51
本來 2308
本來在 75
本來是 205
本來會 3
本來有 18
本來的 60
本來地 59
本來面目 34
本例 42
本俸 30
本價 16
本兒 2
本兒 2
本公司 5407
本分 122
本刊 1049
本利 12
本利和 24
本劇 75
本務 2
本區 1949
本原 39
本可 33
本台 82
本台消息 1
本名 293
本周 249
本命 21
本回 4
本國 926
本國人 41
本國語 4
本園 189
本圖 173
本團 1360
本土 1928
本土化 393
本土型 12
本地 800
本地人 38
本埠 1
本報 1790
本報訊 69
本場 17
本場 170
本壘 32
本壘打 0
本套 77
本姓 6
本子 37
本季 180
本官 15
本室 1
本宮 43
本家 36
本尊 0
本局 1562
本屆 528
本屬 33
本島 308
本島人 1
本島居民 1
本州 32
本巷 0
本市 350
本師 59
本席 44
本年 481
本年度 4122
本店 417
本府 620
本座 10
本庭 0
本廠 260
本廳 180
本心 49
本性 230
本性難移 11
本息 111
本想 133
本意 102
本應 88
本我 25
本戲 5
本戶 4
本所 2504
本所同仁 25
本文 4797
本族 20
本日 958
本旨 24
本是 350
本書 4873
本會 6160
本月 2413
本月份 164
本期 2234
本期損益 13
本期發生 0
本期發生額 0
本末 14
本末倒置 23
本末倒置 23
本本 33
本架 5
本校 9747
本校生 2
本案 2206
本業 404
本機 209
本機振盪 0
本機振盪器 0
本欄 168
本次 1695
本段 37
本洲 16
本港 220
本源 80
本溪 46
本無 64
本然 15
本營 16
本片 535
本物 7
本班 362
本生 21
本生燈 5
本田 210
本當 18
本益比 265
本相 17
本省 1098
本省人 61
本社 682
本社訊 0
本票 184
本科 1209
本站 10785
本章 393
本節 241
本篇 926
本籍 28
本系 4575
本紀 19
本組 1113
本線 53
本縣 909
本署 1030
本義 16
本聯 15
本職 86
本職學能 1
本能 220
本臺 1
本臺消息 0
本船 34
本艦 106
本色 164
本草 67
本草綱目 19
本著 433
本薪 34
本處 1210
本號 10
本行 976
本街 8
本該 32
本課 117
本論 194
本質 893
本質上 149
本質屬性 0
本身 5637
本輕利厚 0
本連 8
本週 1695
本部 3170
本鄉 185
本里 106
本金 167
本錢 122
本錢雄厚 0
本鍵 209
本院 2847
本隊 90
本集 94
本項 884
本領 125
本題 47
本願 47
本館 3628
本體 444
本體論 10
本黨 273
札 261
札實 7
札幌 74
札記 235
朮 158
术 0
朱 9
朱 6501
朱元璋 76
朱先 36
朱古力糖 1
朱古律糖 0
朱唇皓齒 0
朱墨 2
朱子 156
朱子 156
朱學 6
朱明 23
朱書麟 1
朱溫 0
朱熹 32
朱爾典 0
朱瑞 17
朱砂 10
朱紅 28
朱紅色 16
朱自清 12
朱色 1
朱門 20
朱顏 7
朱麗葉 17
朲 0
朳 9
朴 57
朴 57
朴 578
朴子 744
朴子溪 29
朴實 4
朴硝 0
朴素 1
朴資茅斯 0
朵 912
朵 912
朵兒 4
朵兒 4
朵朵 124
朵頤 4
朶 0
朷 0
朷 0
朷 0
朸 313
朹 11
机 720
朻 110
朼 149
朽 172
朽屋 10
朽壞 17
朽敗 3
朽木 17
朽木不可雕也 1
朽木糞土 0
朽邁 0
朾 3
朿 32
杄 0
杅 272
杆 92
杆子 2
杇 33
杈 5
杈 5
杉 102
杉 1021
杉木 52
杉材 0
杉林 176
杉林溪 165
杉篙 0
杊 0
杋 4
杌 13
杌隉 1
杍 12
李 3190
李世民 86
李代桃僵 7
李光耀 118
李卜克內西 0
李叔同 1
李商隱 35
李國修 1
李國鼎 48
李子 176
李子樹 0
李家 224
李密 84
李師師 1
李悝 1
李敖 40
李斯 143
李時珍 9
李村 0
李林甫 1
李樹 40
李淵 85
李清照 18
李漁 1
李煜 17
李玉 1
李登輝 3636
李白 184
李秀成 0
李維 1
李總統 3520
李耳 5
李自成 15
李贄 1
李逵 1
李遠哲 383
李鐵拐 4
李闖王 0
李香君 1
李鴻章 33
杏 659
杏仁 124
杏仁奶粉 0
杏仁茶 1
杏仁豆腐 14
杏兒 2
杏兒 2
杏壇 35
杏子 39
杏林 106
杏林春暖 1
杏核 0
杏樹 11
杏眼 0
杏眼圓睜 2
杏眼圓瞪 1
杏花 22
杏花村 12
杏花樓 2
杏鮑菇 1
杏黃 1
材 4744
材學 1
材料 10835
材料力學 120
材智 0
材望 0
材略 2
材種 3
材積 97
材積表 2
材能 2
材質 2167
村 19697
村人 93
村口 20
村坊 1
村塾 0
村墟 0
村夫 14
村女 6
村姑 16
村婦 5
村子 108
村居 61
村店 30
村民 485
村童 2
村聲 18
村舍 5
村莊 255
村落 260
村裡 40
村話 0
村里 877
村鎮 16
村長 193
杒 0
杓 49
杓 49
杓子 3
杔 0
杕 15
杖 276
杖藜 1
杗 6
杘 0
杙 8
杚 5
杛 0
杜 3759
杜勒斯 13
杜南 9
杜口 0
杜塞 47
杜威 67
杜家 9
杜康 70
杜弊 2
杜撰 27
杜松子酒 3
杜甫 70
杜絕 269
杜絕後患 1
杜象 70
杜門 4
杜門謝客 0
杜魯門 20
杜鵑 195
杜鵑花 51
杜鵑花城 1
杜鵑鳥 4
杜麗娘 1
杝 13
杞 122
杞人憂天 21
束 874
束之高閣 10
束帶 22
束成 2
束手 13
束手受縛 0
束手就擒 4
束手待援 0
束手待斃 2
束手無策 84
束狀 3
束緊 9
束縛 261
束胸 0
束脩 8
束腰 29
束腹 13
束裝 6
束裝盜金 1
束髮 4
束髮金冠 1
杠 18
杠 18
条 0
条 0
来 0
来 0
杪 10
杫 0
杫 0
杫 0
杬 84
杭 247
杭州 396
杭州市 16
杭州灣 5
杭育 0
杮 0
杯 2700
杯中物 2
杯子 169
杯底 13
杯弓蛇影 7
杯水車薪 9
杯盤 15
杯盤狼藉 2
杯葛 82
杰 1575
杰出 6
東 20017
東一句西一句 0
東一句西一句 0
東一句西一句 0
東一句西一句 0
東三省 5
東上 8
東亞 1014
東亞 1014
東亞各國 72
東亞各國 72
東亞國家 92
東亞國家 92
東京 3101
東京人 8
東京市 53
東京灣 42
東來 85
東倒西歪 11
東側 190
東元 407
東到 3
東加 80
東加王國 14
東勢 1291
東勢厝 1
東北 798
東北九省 1
東北亞 182
東北區 9
東北向 9
東北地區 31
東北地方 11
東北季候 0
東北季候風 0
東北季風 0
東北方 57
東北角 343
東北軍 2
東北邊 7
東北部 180
東北面 7
東北風 36
東區 1101
東區人 1
東升 7
東半 3
東半球 6
東半邊 0
東南 649
東南亞 2645
東南亞 2645
東南亞國 11
東南亞國 11
東南區 20
東南向 12
東南地區 11
東南岸 1
東南工專 564
東南工業 3
東南工業專科學校 117
東南方 51
東南西北 20
東南角 19
東南邊 5
東南部 88
東南面 3
東南風 3
東印度 5
東印度公司 16
東印度群島 6
東去 6
東向 43
東吳 753
東吳大學 1791
東吳招親 0
東周 24
東周時 1
東周時代 0
東周時期 0
東園 180
東土 25
東坡 99
東城 54
東塗西抹 0
東夷 6
東奔西走 10
東學黨 0
東安 106
東宮 8
東宮太子 0
東家 84
東尼 165
東山 372
東山再起 37
東岸 138
東島 2
東嶽 13
東嶽泰山 0
東市 61
東帝汶 83
東平 106
東床坦腹 0
東床快婿 0
東廠 12
東引 80
東張西望 20
東征 26
東征西討 4
東德 100
東拉西扯 14
東拼西湊 10
東指 13
東捻軍 0
東接 15
東敲西敲 0
東方 2202
東方中學 0
東方人 72
東方國家 7
東方學 2
東方工專 36
東方文化 27
東方發白 1
東方紅 6
東施效顰 3
東昇 148
東晉 26
東晉時 1
東晉時代 1
東晉時期 0
東有 25
東東 922
東柏林 2
東橋 39
東歐 379
東歐各國 27
東歐國家 48
東歐平原 1
東正教 34
東段 78
東江 26
東沙 113
東沙群島 18
東河 227
東洋 495
東洋味 0
東洋蘭 1
東洋車 0
東流 34
東海 885
東海大學 1490
東海岸 309
東海桑田 1
東渡 4
東港 808
東湖 225
東湖國小 5
東湖路 43
東漢 99
東漢人 0
東漢時 4
東漢時代 2
東漢時期 1
東漸 5
東瀛 288
東為 14
東牆 5
東看西看 2
東瞧西瞧 0
東石 394
東窗事發 6
東站 26
東端 13
東經 80
東線 58
東縣 75
東羅馬帝國 6
東翁 1
東胡 7
東臨 32
東至 29
東芝 80
東華 739
東行 28
東街 233
東西 6024
東西 6024
東西南北 36
東西向 158
東西方 81
東角 4
東路 1976
東逃西竄 0
東進 46
東遊 5
東道 8
東道主 5
東邊 113
東郊 16
東部 1468
東部地區 226
東都 36
東鄉 23
東鄰西舍 0
東門 365
東門國小 40
東門市場 5
東門市場 5
東陵 5
東陽 227
東非 23
東非各國 0
東非國家 1
東面 45
東風 135
東魏 4
東鱗西爪 0
杲 28
杳 24
杳 24
杳如黃鶴 0
杳如黃鶴 0
杳渺 0
杳渺 0
杳無人煙 1
杳無人煙 1
杳無信息 0
杳無信息 0
杳無蹤跡 1
杳無蹤跡 1
杳無音訊 1
杳無音訊 1
杳然 11
杳然 11
杳茫 0
杳茫 0
杴 30
杵 70
杵臼 2
杶 85
杷 12
杷 12
杸 35
杹 22
杺 12
杻 253
杻 253
杼 38
杽 61
松 6297
松下 544
松下電器 970
松下電機 1
松仁 43
松子 55
松山 970
松山區 975
松山機場 15
松山機場 157
松木 81
松林 132
松果 18
松果體 14
松枝 10
松柏 130
松柏 130
松柏後凋 0
松柏後凋 0
松柏長青 5
松柏長青 5
松樹 67
松毛蟲 1
松毬 3
松氣 1
松江 822
松江省 2
松油 1
松濤 116
松濤 116
松焦油 0
松煙 2
松科 33
松竹 61
松竹梅 19
松節油 15
松節油精 0
松脂 11
松膠 0
松花 14
松花江 45
松花江平原 0
松花蛋 0
松茸 1
松菌 0
松葉 28
松贊干布 5
松遼平原 5
松針 13
松雞 0
松露 1
松類 0
松香 18
松魚 0
松鼠 167
板 14738
板兒 4
板兒 4
板凳 51
板壁 3
板子 497
板式 156
板擦 17
板斧 4
板書 0
板本 36
板材 95
板栗 1
板條箱 2
板極 1
板橋 1688
板橋中學 2
板橋國中 24
板橋國小 18
板橋市 2946
板滯 0
板牙 2
板球 20
板畫 16
板結 6
板胡 5
板著臉 13
板蕩 3
板起 12
板起面孔 2
板車 26
板金 1
板金工 16
板門店 18
板鴨 4
枀 0
极 736
枂 0
枃 37
构 62
枅 99
枆 27
枇 11
枇杷 74
枇杷膏 6
枇枇 0
枈 0
枉 100
枉己正人 0
枉死 12
枉死鬼 2
枉殺 0
枉殺無辜 0
枉法 14
枉法從私 0
枉然 62
枉用心機 0
枉費 65
枉費唇舌 0
枉費工夫 0
枉費心力 0
枉費心機 2
枉顧 10
枊 0
枋 80
枋寮 372
枋山 137
枌 7
枍 19
枎 25
枏 0
析 1922
析出 117
析木 0
析爨 0
析產 1
析疑 1
析義 0
枑 326
枒 125
枒杈 0
枓 39
枔 11
枕 478
枕 478
枕中書 0
枕套 10
枕巾 3
枕席 5
枕席 5
枕席難安 0
枕席難安 0
枕戈待旦 4
枕木 18
枕木條 0
枕流漱石 1
枕箱 0
枕著 4
枕藉 2
枕資 0
枕邊 43
枕邊人 14
枕邊細語 1
枕邊言 0
枕頭 160
枕頭套 13
枕骨 4
枖 0
林 4768
林下 23
林內 314
林全 41
林冠 55
林分 30
林分改造 0
林則徐 11
林務 76
林務局 539
林區 292
林卡 0
林口 1352
林園 1295
林地 262
林場 25
林場 250
林墾 1
林壑 1
林子 240
林學 64
林家 279
林家花園 65
林帶 22
林彪 10
林木 268
林木分化 0
林木參天 2
林木森森 0
林杪 0
林林總總 40
林梢 6
林森 858
林森北路 249
林森南路 38
林業 1820
林業道路 2
林業部 1
林檎 2
林沖 1
林沖夜奔 1
林泉 41
林洋港 75
林海 44
林班 57
林產 81
林產物 39
林相 86
林立 282
林縣 16
林肯 549
林莽 3
林蔭 45
林蔭大道 18
林蔭道 3
林表 0
林覺民 9
林語堂 295
林邊 343
林野 23
林間 84
林黛玉 59
林齡 2
枘 29
枙 542
枚 426
枚舉 4
枛 0
果 6324
果不其然 17
果仁 16
果凍 111
果名 1
果品 10
果因 31
果園 352
果在 17
果報 68
果如 20
果如所料 0
果子 212
果子狸 1
果子酒 0
果子醬 0
果子露 0
果實 473
果屑 0
果嶺 58
果心 12
果性 5
果敢 13
果斷 46
果於 12
果有 38
果有所見 0
果木 2
果枝 2
果柄 2
果核 16
果樹 244
果毅 21
果汁 403
果汁機 45
果決 28
果無 52
果然 794
果然在 28
果然是 150
果然會 3
果然有 30
果物 4
果皮 61
果皮箱 1
果盒 4
果盤 41
果真 170
果真如此 44
果真是 47
果真會 0
果碟 1
果穗 4
果糖 118
果肉 56
果脯 0
果腹 4
果膠 17
果與 179
果菜 206
果菜汁 10
果蠅 72
果農 40
果酒 18
果酸 191
果醬 68
果陀 41
枝 157
枝 1587
枝丫 0
枝子 12
枝幹 37
枝枒 6
枝枝節節 3
枝梗 0
枝條 61
枝椏 14
枝生節外 0
枝節 29
枝節性 0
枝葉 107
枝葉扶疏 2
枝葉茂盛 6
枝角類 0
枝頭 49
枞 0
枟 14
枢 0
枢 0
枣 0
枮 8
枯 300
枯乾 20
枯井 4
枯坐 17
枯寂 1
枯幹 0
枯木 31
枯木逢春 5
枯朽 2
枯枝 26
枯榮 7
枯槁 15
枯樹 21
枯死 34
枯水 9
枯水位 0
枯水期 25
枯涸 2
枯焦 2
枯燥 71
枯燥無味 13
枯瘦 16
枯竭 63
枯腸 4
枯草 34
枯草桿菌 7
枯萎 55
枯葉 120
枯藤 4
枯骨 16
枯黃 27
枯黑 1
枰 75
枱 0
枲 3
枳 26
枳 26
枴 26
枴子 0
枴杖 11
枵 3
枵腹 0
枵腹從公 1
架 6560
架上 190
架上來 1
架上去 0
架下 5
架下來 0
架下去 0
架不住 1
架了 19
架住 4
架來架去 0
架個 9
架入 1
架出 4
架出來 0
架出去 2
架到 3
架勢 28
架去 0
架回 1
架回來 0
架回去 0
架在 63
架好 25
架子 71
架完 3
架屋 3
架式 76
架成 8
架持 0
架有 17
架構 5038
架橋 37
架次 45
架滿 1
架為 14
架空 54
架空電纜 0
架站 0
架著 17
架設 1203
架走 3
架起 43
架起來 2
架過 2
架錯 0
架開 2
架高 11
枷 44
枷鎖 67
枸 27
枸 27
枸杞 96
枸杞子 17
枸杞子 17
枸杞茶 0
枸橘 0
枹 8
枹 8
枹 8
枺 31
枻 4
枼 0
枽 0
枾 0
枿 0
柀 4
柁 34
柂 6
柃 42
柄 564
柄 564
柄政 1
柅 1
柆 10
柇 0
柈 22
柉 4
柉 4
柊 12
柋 59
柌 7
柍 11
柎 3
柏 4005
柏 4005
柏克萊 296
柏城 1
柏拉圖 88
柏木 10
柏木 10
柏林 353
柏林人 1
柏林圍牆 12
柏樹 18
柏樹 18
柏油 112
柏油紙 1
柏油路 30
柏羅米修斯 0
某 3430
某一 1676
某一 1676
某一 1676
某事 21
某些 2621
某些人 107
某人 397
某位 204
某個 802
某個人 35
某區 2
某地 56
某天 134
某家 88
某對 6
某市 4
某年 26
某年某月 7
某年某月某日 35
某支 14
某方 8
某方面 35
某日 134
某時 20
某時某分 0
某月 6
某月某日 5
某枝 0
某某 177
某某人 33
某樣 10
某次 42
某物 13
某班 13
某甲 106
某種 945
某種人 1
某處 74
某行 4
某隊 2
某隻 1
某面 1
柑 240
柑園 66
柑子 5
柑桔 60
柑橘 88
柑橘園 1
柒 615
染 5502
染上 91
染上來 0
染上去 0
染印 3
染在 6
染坊 2
染工 10
染布 36
染患 10
染成 54
染房 2
染指 29
染整 474
染料 2096
染料植物 0
染有 16
染毒 4
染汙 1
染液 63
染液 63
染疾 0
染病 36
染紅 51
染織廠 29
染缸 19
染色 3668
染色劑 3
染色性 79
染色牢度 25
染色質 11
染色體 459
染起 1
染起來 0
染髮 228
染髮劑 25
染黑 18
柔 1052
柔中帶剛 3
柔光 9
柔到 0
柔化 8
柔和 220
柔善 0
柔婉 4
柔媚 12
柔嫩 27
柔弱 61
柔弱不堪 1
柔性 142
柔情 184
柔情似水 26
柔情萬種 9
柔柔 99
柔柔和和 0
柔柔順順 0
柔滑 9
柔細 30
柔美 79
柔能克剛 1
柔腸 5
柔腸寸斷 14
柔色 3
柔術 5
柔軟 603
柔軟度 67
柔軟性 41
柔軟操 3
柔軟體操 7
柔道 229
柔韌 11
柔順 55
柕 0
柕 0
柖 0
柗 0
柘 34
柙 13
柚 154
柚子 271
柚木 68
柛 25
柜 28
柜 28
柜上 0
柜台 10
柜子 6
柝 28
柞 11
柞樹 0
柟 7
柠 0
柡 0
柢 34
柣 7
柤 7
查 762
查 7623
查不出 34
查不到 89
查修 14
查價 7
查出 246
查出來 12
查到 278
查勘 21
查勤 14
查卷 1
查哨 2
查問 26
查問出 0
查回 2
查回來 0
查回去 0
查在 2
查堂 2
查報 143
查報表 6
查夜 0
查字典 29
查察 0
查審 4
查封 104
查尋 447
查對 53
查帳 90
查帳員 19
查庫 0
查得 127
查得 51
查戶口 4
查房 57
查扣 0
查找 14
查找周期 0
查找時間 0
查找服務 0
查抄 3
查收 3
查明 395
查明真相 9
查架 0
查核 1232
查核員 4
查根項 0
查案 23
查檔 1
查檔案 0
查清 18
查無 0
查無實據 6
查照 230
查獲 719
查理 164
查理一世 1
查理三世 0
查理二世 2
查理五世 2
查理四世 0
查理定律 3
查田運動 0
查的 51
查看 3603
查票 5
查票員 0
查禁 70
查稅 0
查究 12
查經 0
查緝 184
查考 74
查著 1
查號 33
查號台 46
查號臺 0
查表 45
查覺 53
查覺到 5
查訊 6
查訪 58
查証 37
查詢 57818
查詢台 20
查課 4
查證 542
查證法 0
查賄 0
查賬 1
查起 13
查起來 0
查辦 59
查錯 0
查錯程序 0
查閱 643
查驗 430
查點 1
柦 5
柧 2
柨 0
柩 72
柩車 0
柪 33
柫 6
柫 6
柬 112
柬國 54
柬埔寨 285
柬埔寨 285
柬帖 0
柬請 0
柭 1
柮 10
柯 18087
柯林頓 3022
柯氏力 0
柯羅拉多 4
柯羅拉多州 0
柯西 20
柯達 311
柰 25
柱 6000
柱子 201
柱廊 7
柱形 6
柱狀 75
柱狀晶 0
柱石 14
柱頭 21
柲 52
柳 1815
柳丁 56
柳丁汁 1
柳丁皮 1
柳下惠 7
柳夢梅 1
柳宗元 19
柳州 34
柳州市 7
柳巷 2
柳敬亭 0
柳暗花明 22
柳暗花明又一村 11
柳暗花明又一村 11
柳杉 38
柳枝 11
柳條 20
柳樹 20
柳橙 1
柳橙汁 1
柳永 18
柳營 217
柳琴 15
柳眉 8
柳絮 27
柳絲 5
柳腰 7
柳花 0
柳葉 39
柳葉魚 1
柳蔭 5
柴 94
柴 944
柴刀 4
柴可夫斯基 117
柴扉 4
柴油 310
柴油引擎 111
柴油機 29
柴油車 35
柴火 19
柴禾 0
柴米 4
柴米油鹽 7
柴米油鹽醬醋茶 6
柴草 3
柴薪 1
柴達木河 0
柴達木盆地 2
柴門 15
柴高夫斯基 1
柴魚 1
柴魚片 1
柵 191
柵口 0
柵極 4
柵欄 81
柵門 14
柶 12
柷 7
柸 7
柹 0
柺 25
査 0
査 0
柼 7
柾 0
柾 0
柿 64
柿子 44
柿糕 0
柿霜 2
柿餅 24
栁 0
栄 0
栅 0
栅 0
栒 7
栓 245
栓劑 1
栓塞 108
栓塞物 0
栓子 13
栓牢 2
栔 9
栕 0
栕 0
栕 0
栖 43
栖栖皇皇 0
栗 582
栗子 54
栗色 10
栗鼠 3
栘 15
栘 15
栙 0
栚 20
栛 0
栜 29
栝 41
栞 0
栟 10
栠 177
校 20000
校 3616
校內 0
校刊 325
校勘 24
校勘學 1
校務 1939
校區 1231
校區內 22
校區外 0
校友 2835
校友會 1186
校史 471
校史館 67
校園 7530
校址 461
校場 6
校場 6
校外 1352
校外活動 12
校官 9
校尉 12
校對 306
校對者 4
校工 42
校徽 245
校慶 1105
校改 17
校方 447
校旗 12
校服 30
校本 22
校本部 102
校樣 2
校歌 198
校正 742
校準 45
校監 2
校稿 72
校稿員 0
校編 8
校舍 504
校花 20
校規 127
校訂 182
校訂本 3
校訂者 0
校訓 141
校警 48
校譽 55
校讎學 1
校車 144
校醫 20
校長 7471
校長室 367
校門 197
校閱 41
校閱者 0
校隊 90
校際 263
校際比賽 7
校風 61
栢 0
栢 0
栣 0
栤 0
栥 27
栦 143
栧 0
栨 53
栨 53
栩 31
栩栩 10
栩栩如生 68
栩栩如繪 0
株 6501
株守 0
株式會社 236
株距 7
株連 11
株選 0
栫 19
栬 0
栭 14
栮 5
栯 25
栰 0
栰 0
栱 17
栲 24
栲栳 1
栳 81
栴 9
栵 2
栶 0
样 0
核 6782
核下 1
核仁 13
核備 0
核准 3538
核分裂 24
核力 0
核動力 2
核反應 46
核反應堆 0
核可 2053
核咁 1
核外電子 2
核子 986
核子動力 8
核子彈 12
核子彈頭 12
核子戰爭 14
核子武器 79
核子潛艇 2
核子能 4
核子試爆 0
核定 2780
核實 393
核對 446
核對表 3
核彈 86
核心 1510
核心人物 21
核戰 77
核戰爭 3
核果 24
核查 8
核桃 138
核桃仁 4
核桃糕 1
核污染 3
核武 507
核武器 35
核減 28
核潛艇 6
核災 0
核災變 0
核燃料 162
核爆 54
核爆炸 1
核狀 1
核發 1790
核示 50
核算 166
核糖 17
核糖核酸 27
核能 1746
核能發電廠 46
核膜 5
核蛋白 1
核融合 0
核裂變 0
核覆 7
核計 101
核試爆 83
核試驗 8
核辦 92
核酸 178
核銷 100
核電 191
核電廠 365
核飛彈 0
核驗 23
核黃素 5
根 13121
根兒 1
根兒 1
根冠 0
根基 288
根子 3
根子 3
根尖 29
根底 2
根式 3
根性 29
根拔 10
根指數 4
根據 7897
根據地 45
根本 3172
根本就是 144
根本性 28
根本是 158
根本法 1
根本矛盾 1
根柢 8
根植 119
根毛 16
根治 139
根深柢固 15
根深蒂固 41
根源 275
根源地 0
根狀 1
根由 4
根瘤 7
根瘤細菌 0
根瘤菌 11
根目錄 1
根竹 3
根管治療 1
根系 35
根絕 28
根芽 0
根苗 8
根莖 43
根莖葉 3
根號 17
根部 119
根除 105
根音 2
栺 18
栻 12
格 8310
格位 19
格外 271
格外開恩 0
格子 237
格子布 26
格子狀 10
格子窗 3
格子紙 0
格局 685
格式 8616
格式化 237
格律 49
格林 202
格林威治 38
格林威治時間 15
格格 288
格格不入 43
格正 4
格殺 18
格殺勿論 10
格物 7
格物致知 7
格狀 7
格瓦拉 2
格致 78
格致中學 6
格致國中 8
格裡 12
格言 85
格調 215
格里 34
格陵蘭 33
格鬥 297
格鬥者 2
栽 556
栽上 1
栽下 2
栽下來 1
栽下去 2
栽倒 4
栽入 5
栽到 4
栽在 23
栽培 1110
栽培業 5
栽培法 20
栽植 158
栽植法 1
栽秧 0
栽種 165
栽種法 1
栽花 7
栽贓 52
栽跟斗 2
栽跟頭 2
栽進 20
栽進來 0
栽進去 6
栾 0
栿 0
桀 65
桀驁不馴 3
桀黠 0
桁 44
桁 44
桁架 56
桁桷 0
桂 1681
桂冠 641
桂冠詩人 4
桂圓 25
桂圓乾 2
桂圓湯 3
桂子 6
桂子 6
桂州 0
桂林 271
桂林市 7
桂樹 6
桂河 12
桂河大橋 11
桂王 0
桂皮 6
桂花 96
桃 1882
桃仁 4
桃兒 1
桃兒 1
桃園 4945
桃園人 21
桃園市 2625
桃園縣 12772
桃太郎 1
桃子 95
桃李 32
桃李滿天 2
桃李滿天下 13
桃樹 20
桃江 0
桃源 292
桃符 1
桃紅 24
桃紅色 21
桃色 116
桃色新聞 4
桃色糾紛 5
桃花 204
桃花尖 0
桃花心木 39
桃花扇 1
桃花源 203
桃花源地 1
桃花源記 13
桃花臉 0
桃花運 10
桄 64
桄 64
桄榔 0
桅 266
桅帆 3
桅座 0
桅杆 3
桅桿 5
桅竿 0
桅頂 1
框 1577
框 1577
框上 14
框下 4
框下來 0
框下去 0
框住 4
框內 99
框出 4
框出來 0
框圖 1
框在 3
框子 3
框架 197
框框 132
框著 2
框起 2
框起來 6
桇 0
案 12000
案件 4263
案例 2117
案內 177
案冊 0
案卷 49
案外 26
案外案 25
案子 283
案情 231
案情大白 0
案板 2
案桌 6
案牘 4
案由 162
案發 93
案發前 6
案發後 32
案發時 11
案語 1
案頭 34
案頭櫃 0
案首 5
桉 348
桉樹 21
桊 0
桋 3
桌 1626
桌上 12
桌上型 12
桌下 18
桌兒 0
桌兒 0
桌子 252
桌巾 47
桌布 393
桌案 3
桌椅 332
桌燈 8
桌球 650
桌球隊 26
桌腳 10
桌面 1286
桌面上 149
桍 18
桎 9
桎梏 44
桏 48
桐 993
桐城派 1
桐油 3
桐油樹 0
桐葉知秋 0
桑 1313
桑園 4
桑梓 2
桑椹 1
桑椹汁 1
桑榆 1
桑樟 0
桑樹 20
桑田 30
桑給巴爾 0
桑葉 14
桑葚 6
桑蟲 0
桑蠶 1
桑蠶絲 1
桑針 0
桑麻 2
桒 0
桓 336
桓公 13
桓溫 0
桓玄 1
桔 193
桔 193
桔梗 58
桔梗花 2
桔槔 0
桔紅 1
桕 0
桖 0
桘 0
桙 0
桫 84
桬 0
桬 0
桭 71
桮 142
桯 420
桰 0
桱 91
桲 334
桳 0
桴 512
桴鼓相應 0
桵 358
桶 933
桶子 34
桶柑 1
桶狀 4
桶蓋 4
桶裝 47
桷 90
桸 104
桹 162
桺 0
桻 24
桼 54
桽 124
桾 128
桿 1123
桿刀 1
桿子 26
桿狀 13
桿直 19
桿秤 1
桿茵 1
桿菌 455
桿菌類 2
梀 78
梁 3450
梁任公 4
梁啟超 31
梁實秋 38
梁山 42
梁山伯 30
梁山泊 81
梁書 1
梁柱 8
梁武帝 10
梂 7
梃 12
梄 0
梅 4581
梅乾菜 2
梅園 80
梅妃 3
梅妻鶴子 1
梅子 198
梅山 200
梅村 12
梅林 69
梅樹 73
梅毒 115
梅汁 8
梅渣 0
梅爾 149
梅特涅 0
梅紅 0
梅花 462
梅花餐 1
梅花鹿 90
梅蘭芳 1
梅開二度 2
梅雨 29
梅雨季 19
梅雨季節 10
梅雨期 5
梅鶴因緣 1
梆 22
梆子 12
梆鈴 0
梇 211
梈 0
梉 56
梊 111
梋 44
梌 65
梎 0
梏 86
梏地 0
梏桎 1
梐 150
梑 191
梒 733
梓 416
梓匠 1
梓官 379
梓樹 1
梓里 0
梔 24
梔子花 5
梕 0
梖 40
梗 467
梗住 3
梗到 0
梗化 0
梗在 3
梗塞 99
梗概 21
梗死 0
梗直 0
梗節 2
梗著 0
梗阻 1
梘 0
梙 0
梚 0
梛 104
梜 130
條 23029
條令 2
條件 7430
條件下 300
條件刺激 1
條件反射 7
條件式 48
條例 4302
條凳 1
條列 129
條型 17
條塊 33
條子 30
條幹 0
條形 9
條播 2
條支 15
條文 2090
條暢 0
條案 1
條款 1075
條狀 67
條理 418
條理井然 1
條理分明 17
條目 75
條碼 1
條約 466
條約國 1
條紋 188
條紋布 1
條紋紙 0
條規 1833
條達 1
條鋼 9
條陳 4
條頓族 1
梞 0
梟 82
梟匪 0
梟雄 33
梟首 2
梠 76
梡 4
梡 4
梡 4
梢 211
梢頭 3
梣 94
梤 117
梥 0
梦 0
梧 167
梧 167
梧州 17
梧桐 158
梧桐樹 10
梧棲 668
梧棲港 7
梨 672
梨乾 0
梨俱 2
梨俱吠陀 2
梨園 43
梨園子弟 1
梨子 42
梨山 178
梨山賓館 5
梨形 12
梨果 7
梨樹 9
梨花 30
梨花大鼓 0
梨花槍 0
梨酒 0
梩 197
梪 133
梫 105
梬 105
梭 1054
梭哈 63
梭子 10
梭子蟹 45
梭子魚 1
梭巡 3
梭梭 0
梭羅 108
梭魚 15
梭魚類 0
梮 158
梯 783
梯也爾 0
梯子 46
梯度 60
梯形 39
梯架 1
梯次 577
梯狀 3
梯田 17
梯級 10
梯繩 0
梯隊 11
械 657
械 657
械彈 8
械材 0
械材 0
械鬥 41
梱 11
梲 66
梳 27993
梳了 4
梳妝 22
梳妝台 11
梳妝打扮 2
梳子 32
梳成 4
梳攏 0
梳棉機 5
梳洗 29
梳洗打扮 2
梳理 27
梳著 11
梳起 1
梳起來 0
梳過 0
梳過來 0
梳過去 0
梳頭 19
梳頭髮 8
梴 31
梵 656
梵唄 1
梵唱 10
梵啞鈴 0
梵學 10
梵宇 2
梵文 105
梵蒂岡 29
梵語 22
梵諦岡 20
梵谷 656
梵音 69
梶 0
梷 0
梸 0
梹 0
梹 0
棁 0
棃 0
棄 21703
棄世 1
棄之不甘 1
棄之可惜 19
棄之如敝屣 1
棄井 1
棄兒 11
棄厭 1
棄取 1
棄城而逃 0
棄大就小 1
棄如弁髦 1
棄妻 1
棄妻女於不顧 0
棄婦 12
棄嫌 1
棄嬰 38
棄守 56
棄宗弄贊 1
棄官 2
棄官歸田 1
棄官歸鄉 1
棄屍 0
棄市 1
棄弓折箭 1
棄惡從善 1
棄捐 1
棄捨 1
棄擲 1
棄文就武 1
棄暗投明 2
棄本逐末 1
棄械投降 5
棄械逃亡 0
棄標 1
棄權 84
棄武修文 1
棄武競文 1
棄物 71
棄瑕取用 1
棄瑕錄用 0
棄璧負嬰 1
棄甲 3
棄甲拋戈 1
棄甲曳兵 1
棄短取長 0
棄短就長 1
棄絕 40
棄置 213
棄置不用 0
棄置不管 0
棄義背理 1
棄職潛逃 0
棄背 1
棄舊圖新 1
棄舊憐新 1
棄舊換新 1
棄舊迎新 2
棄若敝屣 1
棄觚 1
棄言 1
棄過圖新 1
棄邪從正 1
棄邪歸正 1
棄養 1
棄養 1
棅 0
棅 0
棆 129
棇 4
棈 28
棉 1419
棉卷 5
棉婚 0
棉布 103
棉帽 0
棉廠 11
棉條 10
棉毛 4
棉毛衣 0
棉毛衫 0
棉毛褲 0
棉球 16
棉田 6
棉籽 1
棉紗 93
棉紙 48
棉紡 27
棉紡廠 1
棉紡織 6
棉紡織品 1
棉紡織業 9
棉絨 0
棉絮 22
棉線 16
棉織 21
棉織品 18
棉織業 2
棉織物 33
棉纖維 67
棉羊 0
棉花 317
棉花子 0
棉花布 0
棉花棒 1
棉花火藥 0
棉花球 0
棉花田 6
棉花糖 1
棉薄之力 9
棉蚜 0
棉衣 4
棉衣褲 0
棉袍 3
棉被 424
棉被套 1
棉製 19
棉褲 5
棉襖 6
棉質 47
棉鈴 2
棉鈴蟲 0
棉鞋 0
棊 0
棊 0
棋 1202
棋品 8
棋士 794
棋子 87
棋子 87
棋局 37
棋布 0
棋戰 4
棋手 39
棋王 33
棋盤 105
棋盤式 4
棋盤形 0
棋社 51
棋聖 4
棋藝 76
棋譜 34
棋賽 12
棋迷 2
棋逢對手 1
棋逢敵手 5
棋高一著 2
棋高一著 2
棌 47
棍 410
棍子 32
棍徒 0
棍棒 41
棍球 1
棍道 0
棎 37
棏 0
棐 34
棑 54
棒 3799
棒喝 11
棒壇 20
棒委會 3
棒子 61
棒打 22
棒棒糖 17
棒槌 4
棒法 23
棒球 1869
棒球員 6
棒球場 90
棒球場 90
棒球賽 70
棒球隊 149
棓 20
棓 20
棔 26
棕 482
棕壤 1
棕樹 16
棕櫚 101
棕櫚 101
棕櫚樹 13
棕櫚油 16
棕殼蛋 0
棕毛 1
棕色 187
棕黑 0
棖 37
棗 286
棗仁 4
棗子 10
棗樹 7
棗樹林 0
棗泥 21
棗紅 26
棗紅色 10
棗莊 5
棗類 0
棗類植物 0
棘 212
棘手 120
棘爪 0
棘皮 1
棘皮動物 24
棘輪 11
棙 0
棚 499
棚子 4
棚架 34
棚鋪 0
棛 0
棜 23
棝 62
棞 52
棟 1840
棟宇 3
棟梁 7
棟梁之材 1
棟樑 83
棠 526
棡 54
棢 0
棣 270
棤 51
棥 0
棦 20
棧 1604
棧單 1
棧戀 0
棧房 0
棧橋 17
棧道 60
棨 93
棩 35
棪 44
棫 132
棬 11
棭 0
森 5582
森嚴 17
森巴舞 1
森林 4604
森林區 32
森林土壤 16
森林地 25
森林地區 5
森林地帶 3
森林學 13
森林學系 68
森林工業 6
森林撫育 1
森林撫育採伐 0
森林撫育採伐 0
森林更新 1
森林浴 104
森林經理 1
森林資源 150
森林資源清查 0
森森 72
森羅萬象 6
棯 14
棰 12
棱 48
棱台 0
棱柱 1
棱角 0
棱錐 0
棱鏡 0
棲 100
棲 1000
棲宿 1
棲居 10
棲息 497
棲息處 3
棲所 12
棲於 16
棲身 38
棲身之處 3
棳 51
棴 77
棵 500
棶 66
棷 47
棸 27
棹 102
棹 102
棺 488
棺木 92
棺材 105
棺材店 2
棺材板 12
棺槨 6
棻 153
棼 120
棽 59
棾 0
棿 0
椀 0
椁 0
椂 0
椃 0
椄 79
椅 112
椅 1124
椅墊 40
椅套 27
椅子 379
椅披 0
椅背 41
椆 38
椇 38
椈 58
椉 0
椉 0
椊 63
椋 144
椌 30
植 3335
植保 34
植株 220
植樹 78
植樹節 22
植牙 0
植物 5489
植物人 116
植物保護 74
植物區 13
植物區系 4
植物園 976
植物學 128
植物學家 16
植物性 179
植物油 60
植物界 8
植物病理 35
植物病理學 12
植物群落 21
植物體 42
植苗 4
植被 180
植髮 0
植黨營私 0
椎 460
椎 460
椎心泣血 2
椎骨 22
椎髻 0
椏 68
椐 59
椑 101
椑 101
椑柿 0
椒 414
椒料 1
椒鹽 16
椒麻雞 1
椓 57
椔 70
椕 56
椗 29
椘 0
検 0
椞 0
椥 7
椪 51
椪柑 1
椮 0
椮 0
椮 0
椯 25
椰 1101
椰城 71
椰奶 1
椰子 238
椰子樹 28
椰子汁 6
椰子類 3
椰林 235
椰棗 1
椰棗乾 0
椰油 32
椰肉 0
椰菜 6
椱 0
椲 7
椳 3
椴 12
椵 6
椶 0
椷 10
椸 38
椹 137
椹 137
椺 0
椻 5
椼 3
椽 29
椽子 2
椾 0
椿 305
椿萱 1
椿萱並茂 0
楀 8
楁 4
楂 44
楂 44
楃 0
楄 6
楅 25
楆 0
楇 0
楈 32
楉 286
楊 15521
楊嗣昌 0
楊四將軍 1
楊城 2
楊堅 14
楊子 78
楊家將 8
楊廣 20
楊文 276
楊朱 2
楊柳 61
楊柳樹 5
楊桃 65
楊桃汁 14
楊梅 316
楊梅汁 0
楊梅鎮 489
楊樹 35
楊氏 75
楊泗菩薩 1
楊玉環 1
楊秀清 0
楊維中 10
楊花 4
楊萬發 16
楊虎城 1
楊貴妃 38
楊輝 1
楊逵 61
楊駿 1
楋 11
楌 8
楍 0
楎 17
楏 4
楐 0
楑 20
楒 12
楓 1753
楓城 61
楓樹 92
楓港 46
楓糖 0
楓糖漿 7
楓紅 88
楓紅季節 0
楓紅時節 1
楓色 0
楓葉 173
楓香 58
楔 13
楔 1395
楔子 45
楔形 15
楔形文字 5
楕 0
楖 104
楗 20
楘 12
楙 68
楚 3196
楚人 14
楚國 64
楚成王 0
楚昭萍實 1
楚材晉用 1
楚楚 318
楚楚動人 8
楚楚可憐 7
楚楚可觀 0
楚歌 1
楚河漢界 12
楚漢 14
楚漢相爭 8
楚王 20
楚留香 252
楚莊王 3
楚軍 7
楚辭 54
楚霸王 1
楛 467
楛 467
楜 13
楝 382
楞 1264
楞 1264
楞了楞 3
楞伽經 0
楞伽經 0
楞住 6
楞嚴咒 1
楞嚴咒 1
楞嚴經 46
楞嚴經 46
楞楞 5
楞楞地 2
楞楞地 2
楞次定律 1
楞頭楞腦 0
楟 54
楠 1465
楠木 16
楠梓 978
楡 0
楢 20
楣 287
楣運 0
楣運當頭 0
楤 0
楥 7
楦 86
楧 0
楨 424
楩 3
楪 19
楫 313
楬 18
楬櫫 2
業 49775
業主 492
業主權益 20
業務 22704
業務員 470
業務局 25
業務部 409
業已 475
業師 7
業界 2213
業精於勤 1
業精於勤荒於嬉 1
業經 296
業績 1110
業者 9406
業蹟 4
業障 73
業餘 827
業餘比賽 1
業餘賽 1
業餘選手 7
楮 98
楯 13
楰 26
楱 19
楲 0
楳 0
楴 6
極 5984
極不 135
極了 227
極低 96
極佳 369
極值 30
極光 116
極其 228
極具 360
極刑 14
極力 436
極力爭取 29
極化 76
極右 6
極右份子 0
極右派 9
極品 199
極圈 6
極地 74
極大 1045
極大值 19
極好 53
極富 140
極小 89
極小值 10
極少 317
極左 3
極左份子 0
極左派 0
極帶 1
極帶地區 0
極度 366
極微 47
極性 98
極性鍵 0
極惡 10
極早 9
極板 26
極樂 98
極樂世界 76
極樂舞 0
極權 32
極權主義 5
極權國家 3
極為 1240
極盛 18
極盛時期 2
極盡 70
極目 13
極目望去 2
極目遠眺 5
極移 0
極端 355
極端主義 5
極端派 0
極至 18
極致 137
極處 6
極表同情 0
極軸 96
極量 1
極間電容 0
極限 487
極限值 10
極願 1
極高 395
極高點 0
極點 108
楶 6
楷 11869
楷字 1
楷書 46
楷書體 11
楷模 259
楷法 1
楷體 38
楷體字 2
楸 22
楹 284
楹聯 4
楺 28
楻 7
楼 0
楽 0
楽 0
楽 0
楽 0
概 1518
概不 1329
概不 1329
概不考慮 0
概不賒欠 0
概不追究 0
概不追究 0
概予 0
概念 3997
概念化 14
概括 244
概括 244
概括地說 2
概括地說 2
概括地說 2
概括性 37
概括性 37
概敘 0
概數 8
概況 4839
概率 33
概率論 9
概略 147
概略性 15
概稱 6
概算 170
概算書 1
概算法 0
概而不論 0
概要 919
概覽 0
概觀 78
概說 234
概論 2465
概述 818
榃 35
榅 0
榆 255
榆林港 3
榆樹 16
榆莢 3
榆錢 0
榍 5
榎 53
榏 0
榐 2
榑 7
榒 0
榓 11
榔 110
榔頭 41
榔頭 41
榕 666
榕樹 308
榖 54
榗 8
榗 8
榘 0
榙 14
榚 6
榛 62
榛果 1
榛榛 0
榛莽 2
榜 1988
榜上 329
榜上有名 20
榜上無名 2
榜人 1
榜單 2323
榜文 10
榜樣 154
榜眼 8
榜示 32
榜額 0
榜首 116
榝 0
榝 0
榞 6
榟 0
榠 5
榡 3
榢 0
榣 13
榤 7
榥 16
榦 16
榦 16
榧 20
榧子 0
榨 74
榨乾 5
榨取 13
榨汁 15
榨汁機 20
榨油 8
榨油廠 2
榨油機 1
榨菜 13
榨菜肉絲麵 5
榩 8
榪 3
榫 37
榫子 0
榫眼 0
榫頭 0
榬 3
榭 186
榮 12526
榮任 40
榮光 198
榮典 13
榮宗耀祖 0
榮寵 9
榮工處 115
榮幸 250
榮幸之至 1
榮景 0
榮枯 26
榮歸 23
榮歸故里 0
榮民 779
榮民之家 28
榮民工程處 6
榮民總醫院 1293
榮獲 1859
榮獲冠軍 12
榮登 166
榮登后座 0
榮登王座 0
榮祿 6
榮縣 1
榮總 1439
榮耀 789
榮膺 53
榮華 181
榮華富貴 29
榮行 2
榮譽 1379
榮譽心 14
榮譽感 44
榮譽榜 272
榮軍 20
榮辱 20
榮辱與共 6
榮銜 18
榮顯 22
榯 3
榰 7
榱 1
榲 0
榳 33
榴 93
榴彈 21
榴彈砲 4
榴槤 43
榴火 0
榴蓮 0
榵 0
榶 0
榷 0
榸 0
榹 6
榺 0
榻 93
榻榻米 21
榼 2
榽 3
榾 1
榾 1
榿 6
槀 0
槁 31
槁木 3
槁木死灰 7
槂 3
槃 188
槄 17
槅 0
槆 20
槇 0
槈 0
槉 3
槊 6
構 3144
構亂 0
構件 141
構句 1
構句法 0
構圖 307
構圖法 4
構型 15
構思 165
構思法 0
構怨 0
構想 1006
構成 1779
構成部份 2
構禍 0
構築 201
構築工事 1
構築物 4
構詞 8
構詞法 0
構造 1716
構陷 8
槌 98
槌子 11
槌球 56
槍 11398
槍下 21
槍傷 29
槍口 40
槍口上 0
槍垛兒 0
槍垛兒 0
槍彈 22
槍戰 66
槍手 25
槍托 10
槍把 2
槍擊 84
槍擊要犯 10
槍支 9
槍斃 34
槍林彈雨 8
槍枝 154
槍枝彈藥 1
槍架 1
槍柄 1
槍案 21
槍桿 2
槍桿子 6
槍械 92
槍械彈藥 2
槍榴彈 8
槍機 11
槍殺 63
槍決 38
槍油 1
槍法 26
槍炮 1
槍炮聲 1
槍爻彈雨 0
槍眼 3
槍砲 86
槍砲聲 2
槍管 44
槍管下 0
槍聲 55
槍聲四起 0
槍膛 1
槍身 17
槍靶 2
槎 13
槏 1
槐 87
槐樹 8
槐樹下 0
槑 0
槒 0
槓 78
槓夫 0
槓子 6
槓桿 106
槓桿原理 12
槓鈴 11
槓龜 20
槔 3
槕 0
槖 0
槗 0
様 0
様 0
槙 47
槡 0
槢 43
槣 0
槣 0
槤 48
槥 25
槦 14
槧 5
槨 136
槩 0
槩 0
槪 0
槪 0
槫 12
槬 39
槭 48
槭樹 13
槮 17
槯 0
槰 0
槱 64
槲 10
槳 186
槴 14
槵 0
槶 43
槷 5
槸 35
槹 0
槺 0
槻 22
槼 52
槽 1948
槽子 0
槽牙 0
槽車 27
槽鋼 1
槾 3
槿 109
樀 55
樁 683
樁子 2
樁腳 0
樂 86
樂 8623
樂 8623
樂不可支 10
樂不思蜀 4
樂事 65
樂以忘憂 3
樂兒 6
樂利 77
樂到 2
樂句 66
樂合彩 5
樂命 1
樂和 35
樂善 40
樂善好施 80
樂器 1477
樂器行 104
樂園 2931
樂團 1819
樂土 51
樂在其中 38
樂境 1
樂壇 230
樂壞了 4
樂天 77
樂天派 13
樂天知命 15
樂妓 87
樂子 39
樂官 2
樂山 107
樂工 2
樂師 45
樂府 121
樂府詩 12
樂律 11
樂得 350
樂意 351
樂意於 0
樂戶 0
樂手 87
樂捐 29
樂捐箱 0
樂捐運動 0
樂於 312
樂於助人 24
樂施 7
樂曲 402
樂業 72
樂極生悲 11
樂歌 20
樂此不疲 42
樂段 23
樂毅 8
樂池 2
樂清 6
樂滋滋 0
樂理 53
樂的 350
樂章 390
樂群 76
樂而不淫 0
樂而忘形 0
樂而忘憂 1
樂而忘返 5
樂聖 9
樂聞 1
樂聲 110
樂育 22
樂舞 44
樂舞 44
樂見 148
樂觀 1203
樂觀主義 5
樂觀其成 10
樂評 0
樂評家 11
樂調 1
樂譜 134
樂變化天 1
樂趣 838
樂趣橫生 2
樂迷 249
樂透 293
樂透彩 5
樂道 11
樂陶陶 11
樂隊 277
樂音 72
樃 0
樄 17
樅 110
樅枝 2
樅樹 2
樆 59
樇 109
樈 13
樉 28
樊 389
樊崇 0
樊石 0
樊籠 5
樊籬 3
樊遲 1
樋 0
樌 0
樍 32
樎 0
樏 13
樐 0
樑 756
樑上 15
樑上君子 0
樑子 10
樑書 0
樒 0
樓 20055
樓上 346
樓上住戶 1
樓下 342
樓下住戶 1
樓中樓 63
樓亭 2
樓台 271
樓台亭閣 1
樓堂館所 0
樓子 25
樓宇 20
樓層 486
樓底下 0
樓房 69
樓板 109
樓梯 457
樓梯平台 1
樓梯間 75
樓窗 1
樓臺 7
樓船 5
樓蓋 3
樓閣 16
樓面 26
樓頂 49
樓頂上 5
樔 5
樕 55
樖 7
樗 34
樘 34
標 15805
標上 31
標上來 0
標上去 0
標下 20
標下來 0
標下去 0
標價 141
標兵 4
標兵處 0
標出 47
標出來 1
標出去 0
標售 271
標商 49
標單 771
標定 46
標尺 12
標幟 82
標幟物 2
標度 3
標引文字 0
標引服務 0
標引深度 0
標引語 1
標志 16
標新立異 20
標明 213
標會 91
標期 20
標本 504
標本室 20
標杆 4
標桿 7
標榜 252
標榜著 15
標槍 20
標注 0
標準 14006
標準偏差 16
標準制 4
標準化 338
標準大氣 0
標準大氣壓 1
標準局 60
標準工資 2
標準差 94
標準接口 0
標準時 66
標準時間 64
標準木 0
標準桿 28
標準氣壓 1
標準溶液 85
標準溶液 85
標準版 0
標準狀態 11
標準狀況 7
標準細分表 0
標準號碼 0
標準規 3
標準語 4
標準音 0
標準題名 0
標燈 6
標界 0
標的 1771
標得 1770
標的物 195
標目 45
標示 2527
標示出 42
標竿 122
標籤 971
標統 0
標線 108
標緻 141
標致 20
標舉 18
標號 61
標記 503
標記符號 2
標記識別 0
標記頁碼 0
標誌 1510
標語 177
標識 108
標識符 0
標購 87
標量 6
標題 22589
標題音樂 10
標顯 0
標高 106
標點 61
標點符號 137
標齊 0
樚 0
樛 53
樜 0
樝 30
樞 352
樞密使 3
樞密院 11
樞機 23
樞機主教 65
樞紐 137
樞軸 2
樟 528
樟木 34
樟木箱 1
樟樹 249
樟腦 51
樟腦丸 8
樟腦油 5
樠 27
模 4180
模 4180
模仿 412
模仿性 2
模倣 31
模具 987
模印 1
模型 2162
模型板 1
模型船 6
模型試驗 37
模塊 11
模壓 7
模子 32
模式 7500
模擬 3928
模擬器 391
模擬考 46
模擬計算機 0
模數 43
模本 3
模板 151
模模糊糊 21
模樣 474
模版 33
模特兒 437
模稜 2
模稜兩可 16
模範 227
模範生 87
模範省 3
模範章 0
模糊 1115
模糊不清 65
模糊起來 2
模組 1592
模組化 127
模製 114
模製品 3
模造 24
模造品 0
模造紙 19
模鍛 4
樢 0
樢 0
樣 1
樣 4279
樣兒 15
樣兒 15
樣冊 0
樣品 1301
樣品屋 47
樣品費 6
樣子 1309
樣式 567
樣張 41
樣書 14
樣本 670
樣本書 0
樣板 80
樣板戲 3
樣樣 140
樣樣精通 18
樣機 1
樣貌 48
樤 0
樤 0
樥 35
樦 39
樧 8
樨 67
権 0
横 0
横 0
樬 0
樲 74
樳 0
樴 80
樵 481
樵夫 82
樵姑 0
樶 0
樷 0
樷 0
樸 257
樸修斯 0
樸學 0
樸實 145
樸實作風 0
樸拙 17
樸直 2
樸素 74
樸素唯物主義 0
樸野無文 0
樹 6792
樹上 198
樹下 302
樹人 134
樹倒猢猻散 0
樹冠 44
樹叢 43
樹名 3
樹園 9
樹大招風 25
樹孔 0
樹幹 136
樹底下 6
樹影 22
樹德 353
樹德家商 1
樹德工專 14
樹德技術學院 1
樹德科技大學 1
樹心 1
樹怨 0
樹懶 9
樹敵 2
樹木 380
樹杈 1
樹杪 0
樹林 800
樹林子 3
樹林鎮 1005
樹枝 172
樹枝上 35
樹根 64
樹根頭 0
樹梢 117
樹椏 0
樹榦 1
樹樁 3
樹欲靜而風不止 4
樹汁 5
樹洞 30
樹海 12
樹液 6
樹液 6
樹狀 207
樹王 27
樹皮 84
樹神 7
樹種 185
樹立 305
樹立權威 0
樹脂 1181
樹脂劑 1
樹脂塑膠 0
樹脂塑膠片 0
樹脂層 4
樹膠 18
樹膠鞋 0
樹苗 98
樹葉 141
樹葉兒 5
樹葉兒 5
樹蔭 26
樹蔭下 33
樹薯 21
樹蛙 137
樹起 4
樹身 14
樹輪 7
樹陰 2
樹頂上 3
樹齡 59
樺 1036
樺木 12
樺林 3
樺樹 0
樺燭 1
樺皮臉 1
樻 21
樼 51
樽 99
樾 91
樿 27
橀 17
橁 24
橂 0
橃 0
橃 0
橄 78
橄欖 181
橄欖形 0
橄欖樹 29
橄欖油 98
橄欖球 103
橄欖球場 2
橄欖球場 2
橄欖球賽 4
橄欖石 19
橅 0
橅 0
橆 6
橇 255
橈 201
橈 201
橉 193
橊 0
橊 0
橋 4862
橋上 126
橋下 137
橋台 14
橋墩 42
橋墩下 2
橋孔 2
橋底 3
橋底下 0
橋式起重機 3
橋接 68
橋梁 69
橋梓 0
橋樑 774
橋樑工事 0
橋樑工程 37
橋欄 1
橋洞 1
橋牌 177
橋牌賽 2
橋藝 153
橋賽 9
橋身 7
橋面 38
橋面上 1
橋頭 454
橋頭堡 13
橌 0
橍 6
橎 12
橏 58
橐 31
橐駝 0
橑 55
橒 0
橓 0
橔 34
橕 12
橖 8
橗 0
橘 389
橘子 250
橘子水 0
橘子汁 5
橘樹 5
橘紅 14
橘紅色 40
橘色 94
橘黃 26
橘黃色 17
橙 546
橙 546
橙木 0
橙汁 54
橙皮 9
橙色 118
橙黃 37
橙黃色 19
橚 37
橛 66
橛子 2
橜 0
橝 33
橞 12
機 30641
機不可失 26
機井 0
機件 155
機伶 8
機位 89
機修廠 0
機先 34
機內 86
機具 378
機制 1391
機動 773
機動力 16
機動小組 0
機動性 93
機動船 1
機務 32
機務人員 4
機務段 15
機匠 0
機名 9
機員 31
機器 3742
機器人 523
機器字 0
機器碼 17
機型 725
機場 245
機場 2452
機子 98
機宜 7
機密 733
機密性 37
機密文件 62
機密消息 0
機尾 27
機工 85
機巧 8
機帆船 1
機師 455
機床 13
機座 18
機庫 8
機微 0
機心 22
機房 540
機敏 18
機智 90
機智問答 80
機智性 0
機會 9797
機會主義 6
機會均等 117
機會成本 30
機會難得 74
機杼 3
機板 108
機架 46
機械 11684
機械 11684
機械人 113
機械人 113
機械化 117
機械化 117
機械化部隊 0
機械化部隊 0
機械唯物主義 0
機械唯物主義 0
機械學 8
機械學 8
機械學系 9
機械學系 9
機械展 30
機械展 30
機械工程 1740
機械工程 1740
機械師 1
機械師 1
機械式 126
機械式 126
機械性能 35
機械性能 35
機械效率 2
機械效率 2
機械油 3
機械油 3
機械率 0
機械率 0
機械系 845
機械系 845
機械能 6
機械能 6
機械論 3
機械論 3
機械運動 4
機械運動 4
機構 16584
機構學 19
機槍 67
機槍手 0
機殼 0
機油 324
機油箱 1
機率 1025
機率論 53
機票 1101
機種 1360
機筒 1
機組 729
機緣 136
機織 3
機繡 0
機群 23
機翼 70
機耕 2
機能 1559
機腹 5
機艙 79
機艙人員 0
機要 151
機要祕書 2
機詐 2
機謀 5
機警 32
機警過人 0
機變 7
機身 166
機車 4792
機車處 0
機車行 608
機軸 2
機遇 25
機運 40
機長 146
機門 21
機關 19451
機關刊物 1
機關團體 298
機關報 22
機關槍 21
機關車 13
機電 1181
機電處長 0
機靈 23
機頭 52
機體 168
橠 95
橡 382
橡實 1
橡木 66
橡木葉 0
橡樹 44
橡皮 136
橡皮圈 13
橡皮圖章 21
橡皮塞 7
橡皮擦 100
橡皮樹 1
橡皮筋 43
橡皮筏 7
橡皮管 9
橡皮膏 0
橡皮艇 49
橡膠 2087
橡膠園 0
橡膠套 0
橡膠樹 11
橢 78
橢圓 166
橢圓形 183
橢圓柱 0
橢圓球 0
橢圓錐 0
橢圓面 0
橢圓體 2
橢球 4
橣 0
橤 61
橥 0
橦 207
橦 207
橧 60
橨 63
橩 18
橪 33
橫 149
橫 1493
橫七豎八 2
橫上 1
橫下 12
橫下心來 0
橫事 0
橫亙 29
橫亙在 11
橫出 1
橫刀 16
橫刀奪愛 5
橫切 16
橫切面 10
橫列 22
橫到 1
橫剖面 4
橫加 3
橫印 5
橫向 207
橫在 17
橫坐標 0
橫坐標軸 0
橫壓 0
橫寫 7
橫山 182
橫幅 5
橫座標 6
橫征暴歛 0
橫得 0
橫徵暴斂 0
橫心 2
橫恣 0
橫截面 10
橫批 0
橫披 1
橫振動 2
橫掃 126
橫掃天下 4
橫接 1
橫提 0
橫握 4
橫擺 5
橫數 0
橫斷 45
橫斷面 25
橫暴 2
橫木 13
橫杆 3
橫桿 18
橫梁 4
橫樑 19
橫檳 5
橫死 9
橫波 5
橫波 5
橫流 18
橫渡 26
橫溢 7
橫濱 337
橫濱人 0
橫濱市 8
橫生 27
橫生枝節 5
橫的 37
橫地 36
橫目 0
橫直 4
橫眉 3
橫眉怒目 1
橫眉瞪眼 0
橫眉豎目 0
橫眉豎眼 3
橫禍 9
橫笛 8
橫管 1
橫紋 32
橫紋肌 22
橫線 68
橫肉 1
橫膈膜 15
橫臥 18
橫著 14
橫蠻 3
橫行 119
橫行一時 1
橫行一時 1
橫行天下 1
橫行霸道 19
橫衝 2
橫衝直撞 26
橫話 0
橫議 2
橫豎 27
橫財 11
橫貫 25
橫貫公路 60
橫貫鐵路 1
橫起 2
橫越 64
橫跨 188
橫跨在 2
橫踢 0
橫躺 21
橫軸 37
橫逆 19
橫運 0
橫過 19
橫陳 6
橫隊 0
橫隔 47
橫隔膜 20
橫面 7
橫題 0
橫額 5
橫騎 0
橬 0
橭 30
橭 30
橮 0
橯 53
橰 0
橱 0
橶 91
橽 0
橾 435
橾 435
橿 6
檀 182
檀島 4
檀弓 2
檀木 16
檀板 1
檀越 3
檀郎 0
檀香 51
檀香山 86
檀香木 19
檀香樹 0
檁 8
檁子 0
檁條 0
檂 0
檃 6
檄 80
檄文 0
檅 3
檆 0
檆 0
檇 12
檈 0
檉 19
檉柳 12
檊 0
檋 0
檌 2
檍 25
檎 8
檏 0
檏 0
檐 105
檑 3
檒 4
檓 6
檔 123
檔 12381
檔冊 1
檔卷 1
檔名 1118
檔子 12
檔期 107
檔案 22859
檔案夾 166
檔案室 235
檔案櫃 22
檕 5
檖 7
檗 47
檘 0
檙 0
檚 7
檛 22
檜 142
檜 142
檜木 210
檝 0
檞 28
檞 28
檟 83
檠 88
檠 88
檡 16
檢 11247
檢修 527
檢字 91
檢字法 10
檢定 1738
檢定考試 217
檢察 779
檢察官 1190
檢察法 0
檢察總長 0
檢察處 8
檢察長 87
檢拾 3
檢收 4
檢方 0
檢束 2
檢查 9339
檢查人員 47
檢查員 58
檢查哨 50
檢查官 17
檢查法 63
檢查站 9
檢查署 3
檢查起來 3
檢查長 2
檢核 295
檢波 7
檢波 7
檢波器 7
檢波器 7
檢流器 0
檢流計 0
檢測 2369
檢疫 637
檢疫所 17
檢疫站 6
檢索 12143
檢索系統 1179
檢肅 64
檢胃鏡 0
檢舉 1116
檢舉人 129
檢舉信 4
檢覆 25
檢視 5095
檢討 4934
檢討會 127
檢調 44
檢調單位 10
檢警 0
檢警調 0
檢錄 3
檢閱 57
檢附 0
檢電器 0
檢驗 5295
檢驗圖 2
檢驗所 284
檢驗機 1
檢驗站 19
檢體 0
檢點 38
檣 96
檣傾楫摧 0
檤 3
檥 24
檦 3
檧 0
檨 16
檪 0
檫 0
檬 83
檭 3
檮 8
檯 430
檯子 4
檯布 1
檯燈 89
檯球 0
檯面 297
檰 0
檱 0
檱 0
檲 0
檳 219
檳子 0
檳榔 829
檳榔樹 1
檳榔汁 18
檴 11
檵 0
檶 3
檷 8
檷 8
檸 31
檸檬 472
檸檬水 3
檸檬汁 103
檸檬皮 6
檸檬茶 32
檸檬酸 51
檸檸 0
檹 6
檺 15
檻 41
檻 41
檻車 1
檼 0
檽 11
檾 0
檿 0
櫀 0
櫀 0
櫁 0
櫂 23
櫃 4825
櫃上 37
櫃台 8687
櫃子 86
櫃房 0
櫃檯 495
櫃櫥 4
櫃臺 60
櫄 0
櫄 0
櫅 3
櫆 8
櫇 29
櫈 0
櫋 2
櫌 2
櫍 4
櫍 4
櫎 0
櫎 0
櫎 0
櫏 6
櫐 3
櫑 3
櫑 3
櫒 0
櫒 0
櫓 19
櫔 0
櫕 0
櫕 0
櫕 0
櫖 0
櫗 0
櫘 0
櫙 3
櫚 21
櫛 269
櫛比 7
櫛比鱗次 8
櫛沐 0
櫛風沐雨 4
櫜 2
櫝 6
櫞 9
櫟 50
櫠 4
櫡 23
櫢 0
櫣 0
櫥 137
櫥子 5
櫥櫃 171
櫥窗 1737
櫧 9
櫨 5
櫩 0
櫪 10
櫫 14
櫬 34
櫭 0
櫭 0
櫮 0
櫯 3
櫰 2
櫱 5
櫲 0
櫳 15
櫴 0
櫶 0
櫸 90
櫹 6
櫺 35
櫻 14294
櫻唇 10
櫻桃 311
櫻桃小口 1
櫻桃樹 11
櫻桃汁 5
櫻桃派 0
櫻桃酒 4
櫻花 834
櫻花季 3
櫻花戀 0
櫻草 14
櫼 29
櫽 0
櫾 0
櫾 0
櫿 0
欀 29
欁 0
欂 28
欃 37
欄 8376
欄位 2130
欄干 5
欄杆 125
欄板 0
欄架 2
欄柵 16
欄桿 7
欆 0
欇 0
欈 26
欈 26
欈 26
欉 60
欉 60
權 9947
權位 39
權充 11
權利 5260
權利能力 8
權利金 183
權力 5260
權勢 134
權威 717
權威人士 18
權威性 74
權宜 70
權宜之計 13
權數 41
權時 67
權杖 33
權柄 95
權為 104
權狀 178
權略 1
權益 3104
權稱 0
權能 58
權能劃分 1
權能區分 14
權臣 4
權術 11
權衡 47
權衡利弊 0
權衡利弊得失 0
權衡得失 0
權衡輕重 6
權要 13
權証 0
權謀 65
權證 0
權變 33
權責 0
權貴 37
權限 675
欋 5
欏 8
欐 98
欑 26
欑宮 1
欒 287
欓 46
欔 0
欖 120
欗 11
欘 1
欙 0
欚 5
欚 5
欛 0
欛 0
欛 0
欜 0
欝 0
欞 82
欠 700
欠伸 1
欠佳 180
欠債 24
欠債未還 0
欠妥 4
欠安 10
欠帳 1
欠帳未還 0
欠揍 11
欠款 32
欠當 8
欠稅 96
欠繳 53
欠缺 454
欠缺著 0
欠著 2
欠薪 12
欠資 3
欠資郵件 3
欠資郵票 1
欠賬 0
欠身 14
欠錢 7
欠錢未還 0
次 15061
次一等 2
次一等 2
次之 274
次位 6
次分面 0
次品 5
次大 97
次大陸 10
次女 25
次子 382
次常式 0
次年 227
次序 327
次數 1583
次文化 0
次方 67
次於 64
次日 500
次殖民地 4
次比 32
次氯酸 56
次氯酸 56
次生 25
次生林 8
次男 3
次第 106
次等 34
次等貨 0
次級 154
次級品 19
次級房貸 1
次級線圈 0
次要 245
次要性 0
次貨 2
次長 500
欢 0
欣 5027
欣勤 0
欣喜 97
欣喜欲狂 0
欣喜若狂 26
欣悅 10
欣慰 208
欣榮 49
欣欣 247
欣欣向榮 84
欣欣客運 6
欣欣然 0
欣然 100
欣然同意 5
欣然接受 31
欣羨 16
欣聞 17
欣見 50
欣賞 4607
欣賞力 2
欣逢 38
欣逢佳節 0
欣逢國慶 0
欤 0
欥 6
欦 0
欦 0
欦 0
欨 11
欩 0
欪 0
欫 0
欬 8
欭 14
欮 0
欯 31
欰 0
欱 58
欲 8000
欲加之罪 5
欲加之罪何患無辭 1
欲得 32
欲念 10
欲振乏力 35
欲擒故縱 10
欲望 164
欲求 69
欲海 2
欲海難填 0
欲界 1
欲窮千里目 2
欲罷不能 50
欲蓋彌彰 11
欲言又止 26
欲語還休 8
欲購從速 47
欲速不達 0
欲速則不達 9
欳 90
欴 446
欵 0
欶 91
欷 21
欷吁 0
欷歔 4
欸 1
欸 43
欸乃 0
欹 10
欺 175
欺三瞞四 0
欺世盜名 5
欺人 14
欺人太甚 4
欺人自欺 1
欺侮 32
欺凌 27
欺君罔上 1
欺善怕惡 3
欺壓 47
欺壓百姓 7
欺心 3
欺敵 6
欺敵戰術 0
欺生 0
欺瞞 19
欺矇 1
欺罔 51
欺蒙 0
欺詐 6
欺誑 4
欺誕 0
欺負 248
欺身 0
欺軟怕硬 2
欺騙 361
欻 7
欻 7
欻忽 0
欻拉 0
欼 6
欽 2090
欽仰 4
欽佩 87
欽命 3
欽州 8
欽差 6
欽差大臣 12
欽慕 4
欽此 2
欽羨 3
欽賜 3
款 10101
款冬 1
款子 6
款實 15
款式 316
款待 109
款曲 3
款款 61
款步 1
款洽 0
款留 3
款目 89
款誠 0
款項 494
款額 62
欿 4
歀 0
歁 4
歂 6
歃 12
歃血為盟 3
歄 0
歅 5
歆 308
歆羨 0
歇 566
歇一宿 0
歇一宿 0
歇一會 6
歇下 5
歇了 21
歇到 0
歇夏 0
歇宿 5
歇工 1
歇後 1
歇後語 34
歇息 39
歇手 5
歇斯底里 41
歇會兒 6
歇會兒 6
歇業 190
歇歇 13
歇歇腳 6
歇腳 42
歈 56
歉 200
歉意 108
歉收 15
歉疚 17
歉著 0
歉過 0
歊 116
歋 17
歌 5205
歌仔戲 1228
歌仔戲 1228
歌兒 5
歌兒 5
歌劇 1750
歌劇院 213
歌功頌德 15
歌友 37
歌友會 80
歌名 280
歌后 20
歌唱 653
歌唱劇 4
歌唱家 16
歌喉 35
歌壇 141
歌壇上 2
歌壇新秀 1
歌女 5
歌妓 5
歌子 425
歌廳 68
歌德 71
歌手 1137
歌星 291
歌曲 1880
歌本 32
歌王 13
歌聲 589
歌舞 348
歌舞伎 74
歌舞劇 74
歌舞團 29
歌舞昇平 8
歌藝 15
歌行 5
歌訣 11
歌詞 1029
歌詠 20
歌詠會 0
歌誦 7
歌調 4
歌謠 183
歌譜 32
歌路 5
歌迷 880
歌集 16
歌頌 118
歌齡 0
歍 6
歎 421
歎一口氣 2
歎一口氣 2
歎傷 0
歎喟 0
歎息 18
歎息聲 0
歎服 1
歎氣 3
歎為觀止 20
歎羨 0
歎聲 0
歎詞 0
歎賞 0
歏 0
歐 1
歐 7152
歐亞 1
歐亞 203
歐亞大陸 1
歐亞大陸 40
歐人 5
歐俄 3
歐元 0
歐克 1
歐冶子 1
歐冶鑄劍 1
歐刀 1
歐化 59
歐化傾向 0
歐化趨勢 0
歐吉桑 1
歐姆 126
歐姆定律 4
歐姆計 2
歐寶 79
歐尼爾 1
歐巴桑 100
歐市 70
歐市各國 3
歐帕林 1
歐幾里得 3
歐幾里得原理 1
歐式 332
歐式建築 9
歐式料理 1
歐弟 79
歐戰 14
歐戰前 0
歐戰後 0
歐戰時 0
歐拉 22
歐文 22
歐普藝術 1
歐本海默 12
歐李 1
歐母畫荻 1
歐氏管 1
歐洲 5620
歐洲人 59
歐洲共同市場 41
歐洲共同市場 41
歐洲共同體 70
歐洲各國 107
歐洲國家 263
歐洲地區 181
歐洲復興計畫 1
歐洲火藥庫 1
歐洲經濟 51
歐洲美元 1
歐洲議會 1
歐洲貨幣體系 1
歐洲鰻 0
歐盟 0
歐羅巴 0
歐羅巴人種 1
歐羅巴洲 0
歐美 1448
歐美化 1
歐美各國 57
歐美國家 106
歐美地區 59
歐耳 1
歐胡島 25
歐虞體 1
歐遊 25
歐遊之行 0
歐遊記趣 0
歐陸 101
歐陽 568
歐陽修 22
歐陽炯 1
歐陽玄 1
歐陽生 1
歐陽竟無 1
歐陽詢 1
歐陽通 1
歐風 78
歐風東漸 1
歐風美雨 0
歐風西漸 0
歐體 1
歑 44
歒 0
歓 0
歔 8
歔欷 0
歕 6
歖 10
歗 0
歘 0
歘 0
歘 0
歘 0
歘 0
歙 79
歙 79
歛 41
歛出來 0
歜 54
歝 0
歝 0
歞 13
歟 13
歠 1
歡 2455
歡呼 155
歡呼聲 41
歡唱 249
歡喜 758
歡喜佛 1
歡喜冤家 0
歡喜地 1
歡喜欲狂 0
歡喜若狂 1
歡喜錢兒 1
歡場 7
歡場 7
歡天喜地 15
歡娛 11
歡宴 22
歡度 74
歡心 35
歡心鼓舞 1
歡快 7
歡悅 10
歡愉 117
歡暢 7
歡樂 1877
歡樂氣氛 49
歡欣 93
歡欣若狂 1
歡欣鼓舞 30
歡歡喜喜 24
歡歡樂樂 0
歡渡 26
歡渡佳節 0
歡笑 263
歡笑聲 12
歡聚 32
歡聚一堂 10
歡聲 6
歡聲雷動 15
歡迎 32103
歡迎晚會 5
歡迎會 22
歡迎詞 72
歡送 169
歡送晚會 0
歡送會 18
歡顏 40
歡騰 22
歡鬧 3
止 6152
止不住 16
止付 41
止住 33
止咳 28
止咳劑 6
止咳藥 1
止境 21
止怒 0
止息 83
止戈為武 0
止於 209
止於至善 16
止步 34
止水 54
止水栓 3
止渴 134
止滑 220
止漲 2
止漲 2
止漲回跌 2
止漲回跌 2
止瀉 7
止瀉劑 4
止瀉藥 4
止痛 183
止痛劑 72
止痛藥 90
止癢 27
止臭 1
止血 115
止血劑 10
止血器 0
止血帶 9
止血藥 2
止謗 0
止跌 126
止跌回升 58
止遏 0
止飢 0
正 14
正 146
正 21462
正職 245
正一 23
正一教 23
正一道 23
正三角形 13
正上 16
正上方 18
正下 11
正下方 7
正中 285
正中 285
正中下懷 2
正中央 55
正中間 12
正事 22
正五角形 0
正五邊形 1
正交 38
正人 23
正人君子 13
正位 32
正使 30
正來 10
正修工專 89
正值 200
正像 42
正光 97
正典 100
正冠李下 0
正冠納履 0
正切 9
正切圓 0
正到 4
正則 25
正前 33
正前方 28
正割 0
正北 17
正北方 7
正午 68
正南 19
正南方 1
正反 94
正反兩方 3
正反兩方面 2
正反兩面 19
正反器 25
正反面 179
正取 526
正取生 143
正可 71
正可以 42
正史 34
正右 1
正右方 1
正合 39
正合我意 4
正合適 1
正名 83
正向 154
正命 1
正和 174
正因 136
正因為 238
正圓 12
正圓形 2
正在 4098
正多角形 1
正多邊形 2
正多面體 3
正大 177
正大光明 12
正好 635
正好在 41
正好是 76
正好會 0
正好有 22
正好能 6
正如 709
正妹 1
正始 2
正字 116
正字標記 116
正學 3
正宗 278
正定 15
正室 4
正宮 5
正宮娘娘 0
正對 48
正對著 28
正對面 58
正左 0
正左方 0
正巧 73
正巧在 6
正巧是 6
正巧會 0
正巧有 7
正常 5337
正常化 196
正常式 0
正常性 6
正廳 26
正式 7439
正弦 26
正弦定理 1
正弦曲線 2
正弦波 41
正弦波 41
正弦規 0
正待 68
正後 19
正後方 13
正得 13
正德 167
正心 55
正心誠意 0
正忙 23
正忙著 35
正念 1
正思維 1
正想 124
正應 10
正應該 2
正房 2
正打 10
正投影 7
正教 12
正整數 24
正數 35
正文 186
正方 63
正方形 103
正方體 14
正於 19
正是 2663
正是如此 35
正時 59
正書 5
正月 87
正月初一 10
正月初三 0
正月初二 0
正朔 2
正期 43
正期班 9
正期生 10
正本 772
正本清源 30
正東 27
正東方 4
正果 17
正根 1
正格 6
正業 11
正極 30
正楷 53
正正 152
正正當當 10
正正確確 0
正正經經 7
正步 7
正步走 0
正殿 84
正比 122
正比例 5
正氣 178
正氣凜然 7
正氣歌 7
正治 14
正法 71
正派 126
正派人 0
正火 1
正為 125
正熱 16
正片 69
正版 103
正版書 0
正牌 31
正犯 15
正理 21
正用 37
正當 873
正當中 13
正當理由 172
正當著 0
正當行為 24
正直 245
正直無私 0
正相反 0
正碳酸鹽 0
正確 6255
正確度 6
正確性 211
正等 1
正精進 1
正統 380
正統派 12
正經 95
正經事 8
正經八百 25
正經話 1
正經起來 1
正緊 4
正義 1521
正義凜然 0
正義北路 86
正義南路 25
正義國小 11
正義感 77
正義戰爭 0
正聲 321
正聽 3
正聽得 0
正色 10
正著 28
正處於 134
正號 0
正襟危坐 3
正西 5
正西方 2
正要 275
正見 1
正規 318
正規軍 11
正規軍隊 2
正規部隊 1
正視 394
正覺 1
正言 29
正該 12
正語 1
正誤 7
正誤表 0
正課 68
正論 11
正負 92
正負得負 0
正負數 5
正負號 3
正跟 8
正路 85
正身 6
正軌 34
正軸 0
正途 44
正逢 52
正過 7
正道 137
正門 102
正門口 5
正開 30
正開著 5
正陽 29
正隆 457
正離子 12
正電 30
正電子 26
正電極 0
正電流 0
正電荷 7
正需 6
正需要 25
正面 1310
正面人物 1
正面圖 4
正面攻擊 3
正音 27
正音法 0
正音班 10
正音譜 27
正題 40
正顏厲色 0
正餐 30
正骨科 0
正體 13
正體中文 1
正體字 13
正鵠 5
正鹽 1
正點 98
此 44322
此一 3326
此一時 4
此一時彼一時 2
此事 600
此人 341
此刻 612
此君 23
此地 778
此地此時 0
此地無銀三百兩 7
此外 4587
此後 314
此時 3879
此時此刻 92
此時此地 14
此書 365
此樹 4
此次 2852
此為 807
此生 101
此等 274
此篇 0
此舉 443
此處 1086
此行 333
此起彼伏 0
此起彼落 61
此路 19
此路不通 5
此身 17
此間 152
此風不可長 2
步 2668
步人後塵 0
步伐 231
步伐 231
步入 257
步入晚年 0
步兵 140
步兵團 6
步兵師 2
步兵排 1
步兵旅 2
步兵營 4
步兵連 0
步兵隊 4
步哨 0
步子 15
步履 47
步履維艱 0
步擲金剛 1
步擲金剛明王 1
步槍 72
步槍兵 0
步槍團 0
步橋 1
步步 106
步步小心 0
步步為營 27
步步驚魂 6
步步高升 13
步步高陞 3
步法 45
步測 1
步測法 0
步行 335
步話機 0
步調 183
步足 13
步軍統領 0
步道 894
步隊 0
步鞋 8
步驟 3025
武 5700
武人 18
武俠 456
武俠片 10
武備 12
武備學堂 0
武則天 26
武力 544
武功 718
武功國小 12
武功縣 0
武后 6
武器 1822
武場 34
武場 34
武士 299
武士道 16
武大郎 12
武夫 13
武夷山 34
武夷茶 0
武威 14
武學 71
武宗 18
武官 59
武將 96
武工隊 1
武帝 23
武師 7
武德 92
武戲 17
武打 39
武打片 1
武斷 31
武旦 3
武昌 167
武昌起義 8
武松 127
武松打虎 8
武林 845
武檢 0
武漢 303
武漢三鎮 3
武漢市 37
武王 14
武王伐紂 4
武王伐紂 4
武生 11
武界 114
武當 0
武當山 17
武當派 10
武藝 44
武藝高強 4
武藤 441
武行 16
武術 441
武術比賽 10
武術競賽 9
武衛 2
武裝 162
武裝侵略 0
武裝力量 11
武裝部隊 86
武裝鬥爭 2
武警 0
武進 12
武選司 0
武關 3
武陵 407
武陵農場 94
武陵農場 94
武鬥 17
歧 122
歧出 0
歧念 1
歧旁 1
歧異 91
歧義 1
歧見 128
歧視 795
歧路 24
歧路亡羊 1
歧路徬徨 0
歧途 10
歧音異義字 1
歧黃 1
歨 0
歩 0
歪 246
歪七扭八 3
歪倒 0
歪偏 0
歪到 1
歪向 1
歪嘴 17
歪寫 0
歪帶 0
歪念頭 1
歪念頭 1
歪成 0
歪戴 1
歪打 1
歪打正著 5
歪掉 2
歪斜 40
歪曲 33
歪曲事實 2
歪歪 17
歪歪地 0
歪歪地 0
歪歪扭扭 2
歪歪斜斜 8
歪理 16
歪眼 2
歪著 18
歪著頭 10
歪詩 2
歪路 1
歪過 0
歪過來 0
歪過去 0
歪門 0
歪門邪道 0
歪頭 10
歪風 50
歪風邪氣 1
歪鼻 0
歫 0
歬 0
歭 13
歮 0
歯 0
歰 0
歱 0
歲 7119
歲入 378
歲凶 0
歲出 1820
歲寒 3
歲寒三友 10
歲差 14
歲收 5
歲數 32
歲暮 13
歲月 925
歲月如梭 10
歲末 157
歲末大拍賣 0
歲次 21
歲歲年年 10
歲計 142
歲除 1
歲首 6
歳 0
歴 0
歵 0
歶 22
歷 3087
歷久 24
歷久不衰 35
歷久常新 0
歷久彌堅 8
歷久彌新 32
歷代 771
歷代先皇 0
歷代祖先 1
歷任 707
歷來 160
歷來如此 0
歷劫 17
歷劫歸來 12
歷史 12943
歷史劇 17
歷史學 111
歷史學家 32
歷史學者 24
歷史小說 38
歷史性 255
歷史意義 74
歷史潮流 6
歷史系 239
歷史觀 33
歷屆 861
歷年 2024
歷年來 421
歷數 12
歷時 150
歷書 0
歷月 1
歷朝 13
歷次 182
歷歷 38
歷歷可數 1
歷歷可見 4
歷歷在目 27
歷法 1
歷盡 27
歷盡滄桑 13
歷盡艱辛 6
歷盡艱難 0
歷程 913
歷經 880
歷經滄桑 10
歷線 10
歷練 132
歷險 121
歷險歸來 0
歷險記 197
歸 5240
歸並 2
歸人 34
歸仁 596
歸位 31
歸位鍵 0
歸併 16
歸併到 3
歸併在 1
歸來 199
歸依 82
歸僑 10
歸入 140
歸公 6
歸到 20
歸功 19
歸功於 123
歸化 55
歸去 47
歸向 47
歸咎 23
歸咎於 52
歸因於 52
歸國 80
歸天 10
歸宗 9
歸客 2
歸宿 67
歸寧 4
歸屬 403
歸屬於 72
歸往 2
歸往何處 0
歸心 20
歸心似箭 4
歸於 167
歸期 17
歸根 16
歸根到底 0
歸根究底 10
歸根結底 4
歸根結蒂 0
歸案 26
歸檔 123
歸正 13
歸法 13
歸獄 0
歸田 6
歸真 20
歸程 12
歸納 615
歸納到 0
歸納法 18
歸納起來 22
歸結 34
歸結到 7
歸結為 6
歸綏 19
歸罪 8
歸罪於 11
歸老 3
歸航 21
歸葬 6
歸西 6
歸諸 17
歸諸於 0
歸路 75
歸返 2
歸途 44
歸還 333
歸還給 14
歸鄉 55
歸附 16
歸降 7
歸隊 45
歸隨 0
歸隨於 0
歸隱 19
歸順 18
歸順於 1
歸類 246
歸類法 2
歹 358
歹 358
歹事 3
歹人 3
歹勢 1
歹命 20
歹命人 2
歹徒 494
歹念 5
歹意 1
歹毒 6
歹路 5
歺 0
歺 0
死 4537
死不 36
死不了 16
死不帶去 2
死不承認 13
死不瞑目 11
死不要臉 0
死不認錯 27
死不足惜 1
死乞白賴 0
死也瞑目 1
死了 791
死亡 3572
死亡率 446
死亡證書 2
死人 213
死傷 80
死傷不計其數 1
死傷枕藉 0
死傷無數 3
死傷者 4
死光 18
死光光 6
死光槍 0
死刑 237
死刑犯 43
死別 8
死前 91
死前遺言 0
死力 5
死去 235
死去活來 11
死友 2
死命 45
死命掙扎 1
死啃 0
死囚 31
死因 213
死地 43
死士 7
死守 39
死守江山 0
死守陣地 0
死定了 14
死寂 39
死對頭 31
死屍 20
死工夫 0
死巷 8
死後 352
死後多年 0
死得 53
死得 433
死得其所 0
死心 49
死心塌地 19
死心眼 4
死心眼兒 0
死心眼兒 0
死忠 0
死怕 3
死戰 18
死扣 1
死拉活拽 0
死挺挺 0
死掉 126
死摳 1
死敵 9
死文字 1
死於 198
死於安樂 0
死於非命 10
死會 24
死有餘辜 4
死期 23
死板 28
死板板 9
死棋 1
死死 46
死氣沉沉 4
死水 32
死活 28
死海 67
死火山 7
死灰 10
死灰復燃 11
死無對證 3
死無葬身之地 4
死球 44
死生有命 0
死當 40
死的 433
死皮賴臉 7
死相 10
死硬 3
死硬派 4
死神 76
死等 5
死節 2
死結 43
死纏 31
死纏不放 2
死罪 12
死罪可免 0
死罪雖免 0
死者 337
死者家屬 17
死而不朽 1
死而後已 11
死而復活 13
死而無怨 1
死而無憾 3
死胎 14
死胡同 15
死腦筋 5
死苦 1
死衚衕 0
死裡求生 0
死裡逃生 14
死要 8
死要面子 3
死規矩 0
死角 121
死訊 20
死記 18
死谷 0
死路 8
死路一條 17
死路一條 17
死難 31
死難同胞 0
死馬 0
死馬當作活馬醫 1
死馬當活馬治 0
死馬當活馬治 0
死馬當活馬醫 5
死馬當活馬醫 5
死鬼 14
死點 5
死黨 82
歽 0
歾 5
歿 65
歿世 1
歿世不忘 0
歿地 1
歿而不朽 1
歿而無朽 1
歿身 1
殀 11
殁 0
殂 106
殂沒 0
殂落 0
殃 76
殃人貨 1
殃及池魚 2
殃及無辜 1
殃國禍家 1
殃書 1
殃榜 1
殃民 0
殃煞 1
殄 24
殄悴 1
殄滅 1
殄瘁 1
殄竭 1
殄絕 1
殅 0
殆 93
殆及 0
殆無孑遺 1
殆無虛日 1
殆盡 105
殈 28
殉 29
殉名 1
殉國 6
殉情 50
殉情記 2
殉教 3
殉節 4
殉職 86
殉職人員 0
殉葬 3
殉葬品 1
殉財 0
殉身 4
殉道 17
殉道精神 3
殉道者 12
殉難 14
殉難人員 0
殉難者 2
殊 710
殊不知 141
殊功 1
殊勝 96
殊勳 3
殊寵 0
殊屬 6
殊屬不易 0
殊屬不當 0
殊技 2
殊文 0
殊榮 137
殊死 1
殊死戰 4
殊異 13
殊禮 0
殊途同歸 11
殊遇 0
殊難 9
残 0
殌 2
殌 2
殍 32
殎 3
殏 40
殐 0
殑 7
殔 5
殕 7
殖 781
殖利 146
殖民 402
殖民主義 31
殖民地 221
殖民者 49
殖產 14
殗 4
殘 1179
殘丘 0
殘兵 5
殘兵敗將 3
殘冬 0
殘卷 4
殘品 0
殘喘 3
殘垣敗壁 0
殘垣斷壁 3
殘壘 2
殘夜 4
殘存 113
殘存本 0
殘害 75
殘局 32
殘屑 6
殘席 0
殘年 2
殘年晚景 0
殘廢 236
殘廢者 16
殘弱 2
殘忍 206
殘春 2
殘暉 0
殘暑 0
殘暴 57
殘更 1
殘月 18
殘本 5
殘殺 45
殘渣 141
殘湯剩飯 0
殘炭 0
殘照 6
殘燈 33
殘瓦 4
殘生 29
殘留 568
殘留物 35
殘疾 109
殘破 48
殘稿 3
殘篇斷簡 0
殘紅 9
殘缺 70
殘缺不全 16
殘羹 6
殘而不廢 38
殘肴 1
殘花敗柳 1
殘虐 6
殘貨 0
殘賊 1
殘賊 1
殘跡 9
殘部 4
殘酷 264
殘陽 8
殘障 2565
殘餘 170
殘餘物 18
殘餘者 0
殘餞 0
殘骸 68
殘黨 2
殙 3
殛 24
殜 0
殜 0
殝 0
殞 28
殞命 6
殞星 10
殞沒 0
殞滅 4
殞落 21
殟 59
殠 7
殠 7
殡 0
殡 0
殢 7
殣 54
殤 417
殥 54
殦 4
殧 25
殨 0
殨 0
殩 0
殪 58
殫 27
殫 27
殫力 0
殫力 0
殫心 0
殫心 0
殫思極慮 0
殫思極慮 0
殬 0
殭 36
殭尸 18
殭屍 46
殭蠶 0
殮 101
殯 33
殯儀 4
殯儀館 74
殯儀館 74
殯殮 63
殯葬 72
殰 4
殲 59
殲擊機 2
殲敵 4
殲滅 74
殲滅戰 6
殳 9
殴 0
段 8639
段名 26
段式 758
段數 17
段段 53
段祺瑞 13
段考 612
段落 677
段落分明 4
殶 35
殷 8
殷 80
殷 804
殷人 3
殷切 183
殷勤 82
殷商 36
殷墟 29
殷富 17
殷實 31
殷憂 1
殷憂啟聖 1
殷望 6
殷朝 3
殷紅 1
殷紅 16
殷鑑 20
殷鑑不遠 9
殷鑒 0
殸 0
殸 0
殸 0
殹 0
殺 1
殺 4123
殺一儆百 7
殺一警百 0
殺人 500
殺人不眨眼 11
殺人不見血 0
殺人不見血 0
殺人償命 3
殺人如麻 5
殺人放火 24
殺人犯 34
殺人盈野 0
殺人罪 29
殺人者死 0
殺人越貨 0
殺伐 11
殺伐 11
殺低 89
殺傷 29
殺傷力 118
殺價 158
殺光 18
殺入 22
殺出 105
殺出來 1
殺出去 1
殺出重圍 11
殺千刀的 1
殺害 221
殺戒 47
殺戮 89
殺戮戰場 11
殺戮戰場 11
殺手 873
殺掉 165
殺掠 2
殺敵 30
殺敵致果 0
殺時間 8
殺機 91
殺死 388
殺殺 22
殺氣 46
殺氣騰騰 28
殺滅 17
殺球 35
殺生 44
殺草劑 41
殺菌 315
殺菌作用 7
殺菌劑 51
殺蟲 89
殺蟲劑 259
殺蟲藥 3
殺豬 10
殺身 13
殺身成仁 0
殺退 5
殺進 23
殺進來 3
殺進去 1
殺過 23
殺過來 2
殺過去 1
殺雞 12
殺雞儆猴 17
殺雞取卵 9
殺雞焉用牛刀 2
殺青 22
殺頭 16
殺風景 5
殺鴨 2
殻 0
殻 0
殻 0
殼 118
殼 118
殼 1182
殼兒 1
殼兒 1
殼子 3
殼裡 4
殽 15
殾 0
殿 2141
殿下 25
殿基 0
殿堂 307
殿宇 12
殿後 37
殿試 0
殿軍 35
毀 952
毀了 110
毀傷 4
毀在 11
毀壞 119
毀害 0
毀家紓難 0
毀容 24
毀形 3
毀掉 57
毀損 237
毀損率 0
毀於 20
毀於一旦 15
毀棄 20
毀滅 454
毀滅性 50
毀約 13
毀言 0
毀訾 0
毀詆 0
毀謗 115
毀謗罪 2
毀譽 13
毀譽參半 9
毁 0
毃 33
毄 10
毅 1806
毅力 223
毅勇 0
毅然 132
毅然決然 20
毆 1
毆 83
毆傷 6
毆打 100
毆打成傷 2
毆擊 0
毆斃 1
毆氣 0
毆罵 1
毆辱 1
毇 7
毈 9
毉 18
毊 6
毋 318
毋寧 40
毋寧是 32
毋庸 28
毋庸置疑 29
毋庸諱言 1
毋庸贅述 1
毋忘 3
毋忘在莒 27
毋貽後患 0
毋需 0
毋須乎 1
毌 10
毌 10
母 3445
母以子貴 1
母儀 0
母儀天下 1
母后 4
母國 37
母夜叉 1
母女 86
母女之情 0
母女均安 1
母女情 5
母女間 2
母奶 151
母姊會 7
母子 182
母子之情 1
母子均安 3
母子情 8
母子連心 0
母子間 3
母性 40
母性愛 2
母愛 53
母慈子孝 0
母教 2
母本 6
母校 353
母株 7
母樹 3
母樹林 15
母機 56
母權 1
母權制 0
母法 37
母液 13
母液 13
母燈 0
母牛 38
母狗 86
母猴 21
母猿 0
母獅 12
母的 231
母系 56
母系制度 3
母系社會 16
母系親屬 0
母線 9
母羊 9
母老虎 4
母舅 3
母艦 24
母蜂 1
母親 2513
母親節 247
母語 348
母象 1
母豬 396
母錢 1
母雞 72
母音 49
母音表 0
母馬 7
母體 270
母魚 119
母鳥 41
母鴨 15
母鹿 2
母黨 0
毎 0
每 12402
每一 4694
每一件 107
每一件 107
每一個 2195
每一個 2195
每一個人 631
每一個人 631
每下愈況 22
每串 2
每人 1700
每付 3
每件 242
每件事 61
每份 213
每位 1154
每個 3563
每個人 2334
每個月 587
每公尺 8
每公斤 139
每公里 17
每具 16
每冊 86
每分 44
每分鐘 440
每列 0
每則 30
每副 1
每區 14
每句 15
每句話 8
每名 278
每周 233
每回 78
每圈 3
每在 15
每坪 248
每堆 0
每場 13
每場 137
每夜 29
每天 4816
每套 108
每季 267
每客 24
每家 211
每封 11
每對 27
每小時 396
每局 16
每屆 59
每幅 23
每年 5808
每張 290
每打 15
每批 37
每排 8
每支 84
每斤 4
每於 11
每日 5704
每星期 292
每晚 184
每月 6733
每期 277
每本 121
每本書 31
每杯 9
每枚 99
每枝 7
每架 8
每格 11
每桶 39
每棟 29
每棵 16
每樣 43
每次 3544
每段 46
每每 244
每每如此 0
每況愈下 24
每班 296
每瓶 22
每當 613
每盞 5
每碗 28
每科 62
每秒 308
每秒鐘 26
每種 233
每站 10
每章 35
每端 0
每筆 159
每箱 20
每節 72
每節課 11
每粒 26
每罐 7
每股 1250
每能 3
每行 0
每試必中 0
每課 11
每輛 33
每輛車 13
每通 24
每逢 224
每逢佳節倍思親 4
每逢例假日 1
每週 3495
每過 11
每邊 13
每部 94
每鍋 5
每間 118
每隊 64
每隔 333
每隻 83
每集 42
每雙 12
每面 15
每頁 508
每項 123
每頭 81
每顆 77
每題 171
每首 41
每首歌 10
毐 44
毑 0
毒 3251
毒刑 0
毒刑拷打 0
毒刺 3
毒劑 9
毒化 192
毒品 416
毒品學 0
毒品案 6
毒婦 4
毒害 108
毒性 2092
毒手 32
毒打 38
毒招兒 0
毒招兒 0
毒梟 37
毒死 16
毒殺 39
毒氣 57
毒氣彈 1
毒水 10
毒汁 3
毒油 0
毒液 25
毒液 25
毒牙 2
毒物 442
毒物學 52
毒狗草 0
毒理學 67
毒瓦斯 0
毒症 13
毒瘤 7
毒瘴 2
毒癮 0
毒箭 6
毒米 1
毒粉 4
毒素 298
毒花 3
毒芹 0
毒草 13
毒菌 8
毒菜 0
毒蕈 5
毒藥 135
毒蛇 137
毒蟲 6
毒蠍 9
毒蠍子 2
毒血 36
毒計 10
毒販 11
毒質 6
毒辣 20
毒辣辣 1
毒酒 8
毒針 13
毒飯 0
毒餌 15
毒魚 54
毓 1024
比 1764
比 17642
比 1990
比一比 108
比一比 108
比上 168
比上不足 5
比下 14
比下來 2
比下去 10
比下有餘 4
比不上 206
比不過 16
比丘 137
比丘尼 41
比丘尼戒 1
比丘戒 1
比並 3
比了 23
比京 4
比人 66
比作 12
比來 9
比來 9
比你 1000
比來比去 7
比例 3740
比例中項 0
比例失調 0
比例尺 247
比例式 8
比例號 0
比例規 1
比個 18
比個高下 0
比值 102
比做 15
比價 176
比出 19
比分 24
比分數 2
比利 107
比利時 579
比利時人 1
比到 9
比劃 23
比喻 332
比喻失當 2
比國 25
比國 25
比基尼 14
比多 23
比多少 1
比大 71
比大小 8
比如 489
比如說 213
比容 17
比對 544
比對法 6
比少 4
比干 7
比年 6
比得 21
比得上 36
比得上 36
比得過 7
比手 12
比手畫腳 12
比才 23
比拼 11
比擬 42
比數 113
比方 80
比方說 145
比武 25
比比 335
比比皆是 51
比比看 21
比溼 0
比濕 2
比為 78
比照 982
比照辦理 47
比熱 40
比熱法 0
比爾 487
比物連類 0
比率 1968
比畫 14
比目魚 18
比紹 0
比美 84
比翼 26
比翼而行 0
比翼雙飛 0
比翼鳥 3
比肩 12
比肩繼踵 0
比色 62
比色計 10
比薩 107
比薩斜塔 23
比表面 0
比試 16
比賽 6020
比賽場 18
比賽場 18
比賽規則 50
比起 691
比起來 135
比較 12944
比較儀 0
比較圖書館學 5
比較好 707
比較文學 420
比較法 27
比較級 1
比較起來 113
比過 21
比過來 0
比過去 46
比鄰 301
比鄰而居 10
比酒 1
比重 1163
比重計 17
比量 6
比附 2
比高 31
比高下 16
毖 114
毗 85
毗盧遮那 1
毗盧遮那佛 1
毗連 35
毗鄰 124
毘 87
毘婆沙部 1
毘曇宗 1
毘曇師 1
毘盧遮那 1
毘盧遮那佛 1
毚 77
毛 3288
毛上 4
毛丫頭 1
毛主席 59
毛偽 0
毛公鼎 5
毛共 0
毛利 251
毛利率 129
毛刷 150
毛刺 0
毛匪 1
毛卷 0
毛囊 0
毛地黃 0
毛坑 1
毛坯 0
毛坯 0
毛姆 11
毛子 4
毛子 4
毛孔 80
毛孩子 1
毛巾 308
毛巾布 5
毛巾被 2
毛手 9
毛手毛腳 15
毛料 27
毛桃 0
毛條 50
毛毛 147
毛毛糙糙 0
毛毛蟲 131
毛毛躁躁 2
毛毛雨 11
毛毯 45
毛氈 22
毛氈苔 3
毛澤東 472
毛澤東思想 6
毛片 9
毛狀 6
毛玻璃 5
毛瑟槍 3
毛病 354
毛皮 120
毛皮衣 1
毛竹 1
毛筆 99
毛筆套 0
毛筆字 10
毛筆畫 0
毛筍 0
毛糙 0
毛紡 24
毛紡廠 2
毛細現象 5
毛細管 69
毛線 132
毛線衣 7
毛線襪 0
毛線針 0
毛織 4
毛織品 17
毛織布 2
毛織業 0
毛織物 9
毛羽 17
毛茛 4
毛茛科 4
毛茶 3
毛茸茸 17
毛蟲 33
毛蟹 33
毛衣 169
毛衣裝 0
毛裝 3
毛襪 2
毛詩 7
毛語錄 10
毛豆 67
毛豬 530
毛賊 4
毛躁 5
毛遂 0
毛遂自薦 29
毛邊紙 1
毛酋 0
毛里塔尼亞 0
毛重 32
毛霉 0
毛頭小子 5
毛額 182
毛類 7
毛驢 7
毛骨悚然 31
毛骨森然 0
毛骨森竦 0
毛骨竦然 0
毛骨聳然 0
毛髮 181
毛髮倒豎 0
毛髮倒豎 0
毛髮聳然 0
毞 736
毠 5
毡 0
毡 0
毢 1
毣 9
毤 7
毥 0
毦 20
毧 8
毨 1
毩 0
毪 0
毫 4400
毫不 339
毫不動搖 0
毫不在乎 15
毫不在意 23
毫不妥協 4
毫不掩飾 9
毫不猶豫 65
毫不留情 30
毫不留意 0
毫不相干 7
毫不考慮 20
毫不諱言 4
毫不費力 18
毫不遜色 22
毫不遲疑 16
毫伏 3
毫克 197
毫分 0
毫升 246
毫安 5
毫安培 3
毫居里 6
毫巴 7
毫微米 66
毫末 1
毫毛 7
毫無 923
毫無保留 24
毫無價值 14
毫無共同之處 0
毫無原則 1
毫無可取 1
毫無可取之處 1
毫無忌憚 5
毫無悔意 1
毫無把握 2
毫無效果 2
毫無根據 13
毫無準備 7
毫無用處 9
毫無疑問 78
毫無疑義 2
毫無誠意 3
毫無辦法 4
毫無道理 6
毫秒 36
毫筆 1
毫管 0
毫米 97
毫米波 10
毫米波 10
毫釐 4
毫釐不差 0
毫釐不差 0
毫釐不差 0
毫釐不差 0
毫髮 13
毫髮不差 1
毫髮不差 1
毫髮不差 1
毫髮不差 1
毫髮不爽 0
毫髮之差 0
毫髮之差 0
毫髮未傷 3
毬 26
毬果 4
毬花 0
毭 0
毯 148
毯子 21
毰 0
毱 0
毲 6
毳 2
毳幕 0
毴 0
毷 1
毸 4
毹 13
毺 0
毺 0
毻 1
毼 5
毽 238
毽子 6
毾 4
毿 13
氀 18
氁 10
氂 3
氂牛 0
氃 18
氄 12
氅 20
氅衣 0
氆 14
氆氌 1
氈 272
氈子 1
氈帽 24
氈毯 3
氉 0
氊 0
氊 0
氋 2
氌 1
氍 7
氍毹 0
氍毹 0
氏 52404
氏 58
氏族 63
氏譜 0
氐 103
氐 103
民 30226
民不堪命 0
民不畏死 0
民不聊生 15
民主 4077
民主主義 28
民主作風 0
民主化 524
民主國 46
民主國家 122
民主思想 7
民主改革 109
民主政治 394
民主政體 12
民主潮流 19
民主運動 94
民主集中制 13
民主革命 3
民主黨 468
民主黨員 2
民主黨派 14
民事 373
民事法庭 4
民事犯 0
民事訴訟 124
民事訴訟法 153
民享 87
民代 0
民以食為天 10
民俗 4115
民俗學 6
民俗論 0
民兵 32
民初 81
民利 14
民力 11
民和 144
民國 17758
民國初年 59
民國日報 0
民團 6
民報 21
民夫 4
民女 5
民妻 0
民婦 2
民宅 227
民害 0
民家 14
民家婦女 0
民宿 0
民富 42
民工 252
民心 110
民心向背 1
民心所向 1
民心望治 0
民性 4
民怨 56
民怨沸騰 7
民情 210
民惟邦本 1
民意 1857
民意代表 372
民意基礎 31
民意機構 25
民意機關 85
民意測驗 29
民意調查 0
民憤 2
民房 64
民政 480
民政局 206
民政局長 9
民政廳 200
民政廳長 35
民教館 0
民族 2591
民族主義 335
民族利益 7
民族化 1
民族區域自治 1
民族問題 8
民族國家 36
民族團結 7
民族壓迫 0
民族大義 14
民族學 170
民族學院 20
民族平等 5
民族形式 0
民族性 59
民族意識 32
民族文化 54
民族東路 95
民族民主 1
民族民主革命 0
民族獨立 9
民族矛盾 1
民族統一 4
民族統一戰線 1
民族自決 20
民族自治 13
民族至上 1
民族色彩 9
民族英雄 20
民族西路 47
民族解放 2
民族解放運動 2
民族資本 0
民族資產階級 0
民族路 1089
民族運動 8
民族革命 11
民智 28
民智未開 5
民有 155
民望 3
民本 18
民本主義 1
民校 15
民樂 77
民樂 77
民權 444
民權主義 21
民權初步 7
民權思想 2
民權東路 565
民權西路 167
民權路 932
民權運動 5
民歌 191
民歌手 7
民氣 32
民治 126
民法 806
民為 69
民為貴 2
民為貴君為輕 0
民為邦本 0
民營 3100
民營事業 107
民營公司 32
民營公車 4
民營化 0
民生 2217
民生主義 64
民生問題 40
民生報 705
民生東路 893
民生樂刊 0
民生樂刊 0
民生渠 0
民生西路 56
民生路 709
民用 541
民用建築 1
民用機場 5
民用機場 5
民田 1
民瘼 9
民眾 9359
民眾代表 4
民眾團體 11
民眾服務 130
民眾服務社 113
民社 3
民社黨 2
民窮財盡 0
民粹 0
民粹主義 0
民約論 1
民聲 16
民胞物與 19
民脂民膏 7
民航 313
民航局 350
民航機 53
民船 6
民調 0
民謠 234
民謠歌手 1
民警 3
民變 12
民貴君輕 0
民賊 0
民辦 31
民辦活動 0
民進 463
民進黨 7702
民運 175
民運人士 147
民選 200
民選總統 49
民間 4788
民間團體 356
民間故事 127
民間文學 42
民間藝術 33
民防 88
民隱 4
民風 41
民風淳樸 10
民風純樸 11
民黨 693
氒 0
氓 22
氓 22
气 824
氕 1
氖 83
氘 13
氘核 0
氙 21
氚 10
氛 146
氛圍 96
氜 0
氝 17
氟 885
氟化 40
氟化氫 5
氟化物 73
氟化鈣 6
氟氣 4
氠 18
氡 19
氣 21702
氣上 5
氣了 32
氣人 9
氣候 2238
氣候圖 3
氣候學 22
氣候帶 6
氣冷 92
氣到 42
氣力 51
氣力用盡 1
氣力輸送 6
氣功 579
氣動 134
氣動力 18
氣動式 21
氣勢 457
氣勢不凡 2
氣勢凌人 1
氣勢如虹 62
氣勢洶洶 0
氣勢磅礡 30
氣勢雄偉 11
氣化 93
氣口 22
氣味 211
氣味相投 5
氣呼呼 12
氣哼哼 0
氣喘 807
氣喘咻咻 4
氣喘如牛 9
氣喘病 117
氣嘟嘟 5
氣噓噓 0
氣囊 353
氣圈 11
氣團 35
氣在 26
氣在心裡 0
氣墊 54
氣墊船 4
氣壓 608
氣壓計 20
氣壯山河 7
氣孔 60
氣孔率 1
氣宇 11
氣宇不凡 1
氣宇軒昂 9
氣密 85
氣層 49
氣度 84
氣度不凡 0
氣得 89
氣忿 7
氣急攻心 0
氣急敗壞 14
氣性 30
氣息 524
氣息奄奄 3
氣悶 8
氣惱 9
氣態 137
氣慨 12
氣憤 104
氣憤憤 2
氣成 5
氣數 5
氣數已盡 8
氣旋 24
氣旋雨 1
氣昂昂 5
氣昏 1
氣根 7
氣極敗壞 5
氣概 29
氣概激昂 0
氣槍 19
氣機 75
氣死 76
氣死人 17
氣氛 1716
氣氣 37
氣沖沖 7
氣沖牛斗 3
氣沮 1
氣泡 193
氣泡病 0
氣派 89
氣派非凡 5
氣流 265
氣消 8
氣渦輪 32
氣溫 513
氣溫計 0
氣炸 17
氣焰 13
氣焰囂張 1
氣燄 16
氣燄萬丈 0
氣爆 0
氣爐 6
氣狀 21
氣球 329
氣田 21
氣瘋 4
氣盛 16
氣眼 1
氣短 8
氣窗 8
氣筒 10
氣管 241
氣管炎 50
氣節 23
氣結 20
氣絕 21
氣缸 71
氣胞 0
氣腔 3
氣色 53
氣著 3
氣虛 14
氣象 2171
氣象一新 3
氣象台 116
氣象報告 124
氣象學 32
氣象學家 12
氣象官 0
氣象局 1381
氣象火箭 0
氣象界 2
氣象站 104
氣象臺 0
氣象萬千 14
氣象衛星 41
氣象要素 5
氣象預報 129
氣貫長虹 0
氣質 614
氣質高雅 9
氣走 30
氣運 14
氣量 86
氣門 381
氣閘 2
氣閥 48
氣韻 17
氣頭上 5
氣餒 88
氣體 2157
氣體動力 7
氣體性 1
氣體燃料 59
氣魄 54
氤 13
氤氳 66
氥 2
氦 81
氦族 0
氦氣 44
氧 1269
氧乙炔 4
氧化 980
氧化亞氮 22
氧化亞銅 0
氧化作用 57
氧化劑 344
氧化塘 2
氧化式 0
氧化氮 53
氧化汞 1
氧化渠 2
氧化物 304
氧化硫 14
氧化碳 54
氧化筒 0
氧化脢 1
氧化釔 2
氧化鈣 31
氧化鈷 1
氧化鉛 8
氧化鉬 0
氧化鉻 23
氧化銅 9
氧化鋁 50
氧化鋅 46
氧化鋇 34
氧化鋯 16
氧化鋰 0
氧化鎂 22
氧化鎢 3
氧化鐵 25
氧化鐿 0
氧基 47
氧氣 542
氧氣瓶 5
氧氣筒 14
氧酸鹽 0
氨 328
氨化 4
氨基 110
氨基塑料 0
氨基酸 110
氨氮 57
氨水 17
氪 567
氫 754
氫化 36
氫化氧 0
氫化物 13
氫化鈉 4
氫原子 45
氫原素 0
氫彈 12
氫氟酸 31
氫氣 68
氫氧 63
氫氧化 14
氫氧化物 7
氫氧化鈉 97
氫氧化鈣 24
氫氧化鉀 24
氫氧化銨 8
氫氧化鋁 12
氫氧焰 0
氫氯酸 3
氫氰酸 19
氫鍵 18
氫離子 59
氬 87
氭 0
氮 757
氮化 79
氮氣 81
氮源 5
氮素 18
氮肥 40
氯 1520
氯 1520
氯丁橡膠 1
氯乙烯 148
氯仿 54
氯化 231
氯化亞錫 27
氯化氫 47
氯化氰 0
氯化汞 2
氯化物 35
氯化鈉 45
氯化鈣 30
氯化鉀 35
氯化銀 7
氯化銨 12
氯化鋁 20
氯化鋅 14
氯化鎂 32
氯化鐵 30
氯嘌呤 1
氯氣 56
氯水 1
氯綸 1
氯苯 79
氯酸 92
氯酸鉀 7
氯酸鹽 19
氯離子 54
氯黴素 15
氰 131
氰化 71
氰化法 0
氰化物 85
氰化鉀 24
氰酸 93
氰酸鹽 16
氱 0
氲 0
氳 10
水 29478
水上 1317
水上人家 4
水上保險 0
水上芭蕾 1
水上警察 44
水上運動 45
水上飛機 11
水上飛行 0
水下 117
水中 1317
水中捉月 0
水中撈月 3
水乳交融 16
水井 86
水亭 6
水亮 8
水仗 3
水仙 122
水仙花 10
水份 370
水位 243
水位圖 0
水位計 17
水來土掩 2
水候病 0
水光 11
水兵 26
水具 1
水冷 57
水分 457
水利 1093
水利制 0
水利化 0
水利局 188
水利工程 210
水利會 683
水到渠成 32
水力 110
水力學 22
水力發電 59
水力發電廠 16
水力負荷 1
水力資源 7
水勢 32
水勢洶湧 0
水化 22
水印 8
水口 168
水合 41
水合物 7
水土 165
水土不服 24
水土保持 1237
水土流失 36
水坑 57
水垢 13
水域 342
水堙 0
水塔 281
水塘 47
水墨 767
水墨畫 104
水壓 92
水壓機 3
水壓計 1
水壓試驗 3
水壩 99
水壺 80
水壺蓋 0
水天一色 3
水天一色 3
水孔 26
水密 21
水封 3
水尺 0
水工 147
水師 51
水師營 2
水平 1047
水平儀 17
水平如鏡 0
水平弦 0
水平線 57
水平舵 1
水平角 4
水平面 16
水床 56
水底 118
水庫 1166
水廠 44
水引擎 0
水彩 721
水彩畫 92
水彩畫家 7
水彩筆 0
水彩顏料 2
水性 403
水性楊花 16
水怪 28
水患 377
水情 26
水成岩 3
水戰 21
水手 176
水手長 0
水揚 1
水文 302
水文地理學 3
水文地質 21
水文地質學 5
水文地質條件 0
水文學 26
水斗 1
水族 1368
水族箱 186
水族館 247
水旱輪作 0
水星 175
水晶 692
水晶婚 0
水晶宮 28
水晶床 0
水晶球 43
水晶蝦 1
水晶體 76
水晶鹽 0
水曜日 0
水曲柳 0
水月 112
水木清華 0
水杉 3
水果 1378
水果店 16
水果攤 11
水果汁 3
水果糖 9
水果茶 1
水果酒 0
水柱 50
水柵 7
水栓 10
水桶 92
水桶蓋 0
水桷 0
水梨 1
水棲 27
水楊樹 0
水榭 33
水槍 17
水槽 214
水樣 180
水母 102
水氣 72
水池 272
水汪汪 25
水汽 86
水沫 1
水泄不通 12
水泉 36
水泡 139
水波 62
水波 62
水波槽 0
水波槽 0
水波盪漾 0
水波盪漾 0
水泥 2222
水泥公司 49
水泥地 11
水泥工 7
水泥廠 98
水泥漆 0
水泥漿 7
水泥石 0
水注 8
水泵 47
水洗 172
水洩不通 18
水流 370
水流計 0
水浪 2
水浮蓮 0
水浴 21
水深 225
水深火熱 31
水深火熱之中 5
水淹 59
水清可鑑 0
水清無魚 0
水渠 8
水源 1141
水源路 196
水準 3044
水準儀 10
水準器 0
水準圖 0
水準測量 6
水溝 176
水溫 281
水溶性 168
水溶液 168
水溶液 168
水溶溶 0
水滴 126
水滴石穿 0
水滸 40
水滸傳 195
水滿 15
水滿為患 2
水漬 5
水漬貨 0
水漲船高 47
水潭 20
水潮 3
水澆地 0
水澤 9
水灣 5
水火 52
水火不容 9
水火不相容 1
水火不辭 0
水火棍 4
水火無情 5
水灰比 9
水災 174
水煎包 1
水煙 8
水煙袋 5
水煤氣 5
水煮蛋 0
水燈 33
水牌 2
水牛 104
水牛城 55
水牛群 2
水獺 49
水玻璃 15
水珠 52
水球 40
水球賽 0
水瓢 2
水瓶 239
水瓶子 18
水瓶座 768
水生 128
水生動物 17
水生植物 84
水產 1574
水產學校 15
水產業 21
水產資源 23
水田 167
水痘 273
水療 75
水皰 21
水盆 16
水碓 27
水磨 23
水磨 23
水磨工夫 0
水磨石 1
水磨石 1
水神 17
水禮 0
水禽 7
水稻 367
水稻田 61
水窖 0
水窪 8
水筆 2
水筆仔 49
水筆仔 49
水筏 0
水管 418
水管系 0
水箱 232
水簾 16
水簾洞 0
水米 0
水粉 5
水精鹽 0
水系 90
水紅 11
水紋 15
水經 19
水經注 2
水綠 5
水綠色 1
水網 10
水線 38
水缸 65
水罐 8
水耕 48
水肥 36
水肺 51
水能 38
水腫 312
水腫病 2
水膠 21
水舀子 0
水舞 27
水色 117
水芋 27
水花 27
水花四濺 2
水芹 2
水草 486
水草田 0
水萍 7
水落 3
水落石出 16
水葬 1
水蒸氣 80
水藻 3
水蘊草 0
水蘭 6
水處理 281
水處理 281
水蚤 1
水蛇 11
水蛇腰 0
水蛭 33
水蜜桃 55
水蝕 1
水蝨 0
水螅 19
水螅 19
水蠆 2
水表 28
水袖 9
水裡 110
水解 124
水解作用 6
水解反應 2
水調歌頭 0
水貨 114
水費 115
水質 1810
水路 413
水路運輸 5
水車 60
水軍 20
水輪 19
水輪機 9
水輪泵 0
水輪發動 0
水輪發動機 0
水輪發電 0
水輪發電機 1
水速 0
水運 76
水道 322
水選 2
水邊 54
水部 8
水鄉 166
水鄉澤國 10
水酒 8
水里 514
水量 497
水量計 33
水鉛 5
水銀 114
水銀壓力 0
水銀壓力計 0
水銀柱 20
水銀槽 3
水銀燈 11
水銀計 0
水錶 27
水門 44
水門事件 6
水門案 19
水門汀 0
水閘 12
水閣 15
水陸 1654
水陸並進 2
水陸兩棲 0
水陸兩用 10
水陸聯用 0
水險 6
水雉 0
水雷 12
水雷艇 0
水電 1591
水電工 20
水電瓦斯 11
水電瓦斯費 0
水電站 8
水電費 39
水霤 0
水霸 8
水青岡 2
水面 314
水面上 72
水面艦艇 6
水靴 1
水韭 0
水頭 89
水飯 0
水餃 125
水體 233
水鬼 12
水魔 9
水鳥 196
水鴨 55
水鹼 0
水鼠 0
水龍 197
水龍頭 161
水龍頭 161
氶 314
氷 0
氷 0
永 15825
永不 534
永不分離 11
永世 47
永久 987
永久中立 1
永久中立國 1
永久性 141
永享 4
永保太平 0
永保青春 7
永別 18
永劫 8
永動機 2
永勝 85
永和 1049
永和國中 19
永和國小 21
永和市 1481
永嘉 137
永在 48
永垂不朽 31
永字八法 1
永存 34
永安 749
永定 61
永定河 3
永寧 42
永平 163
永年 77
永志 17
永志不忘 0
永恆 630
永新 115
永昌 399
永春 229
永曆 48
永樂 233
永永遠遠 30
永浴愛河 9
永清 136
永無休止 14
永無寧日 2
永生 272
永磁 16
永續發展 1215
永興 2481
永訣 0
永豐艦 0
永辭 0
永達工專 55
永遠 3147
永遠是 238
永遠的 610
永遠地 610
永遠都是 39
氹 0
氻 273
氼 0
氼 0
氾 217
氾氾 0
氾濫 134
氾濫成災 32
氿 229
汀 569
汀州 155
汀洲 12
汁 674
汁液 36
汁液 36
求 5895
求不得苦 1
求之不得 10
求乞 4
求人 183
求人不如求己 4
求仁得仁 0
求值 40
求偶 34
求偶者 0
求償 163
求全 23
求全之毀 0
求凰 0
求出 113
求刑 0
求助 1215
求助於 40
求助無門 20
求助者 20
求勝 49
求去 26
求取 146
求名 10
求名利 2
求名求利 0
求告 26
求和 64
求售 87
求好 18
求好心切 14
求婚 115
求婚者 7
求子 36
求學 491
求學精神 0
求實 15
求得 284
求得 397
求情 28
求愛 38
求成 10
求才 2808
求才若渴 9
求援 71
求救 473
求救信號 7
求教 71
求教於 13
求新 146
求根 3
求歡 5
求歡被拒 0
求死不能 1
求生 302
求生不得 4
求田問舍 0
求的 397
求知 111
求知慾 32
求神 129
求神問卜 10
求籤 2
求職 2023
求職信 7
求著 6
求處 0
求見 31
求親 8
求解 98
求診 128
求証 7
求證 80
求證法 0
求變 126
求賢若渴 0
求過於供 2
求降 5
求雨 6
求饒 18
汃 101
汄 0
汅 0
汆 15
汆 15
汆湯 0
汊 28
汋 15
汌 7
汍 39
汎 548
汎心論 0
汎愛 1
汎汎 0
汎汎之輩 0
汎神論 1
汎稱 0
汎美 22
汎美主義 0
汎美航空 2
汎論 1
汏 7
汐 307
汐止 1009
汐止鎮 1188
汑 0
汒 381
汓 0
汔 96
汕 167
汕頭 112
汕頭市 19
汖 0
汗 560
汗 560
汗垢 3
汗如雨下 3
汗斑 24
汗毛 6
汗毛 6
汗水 153
汗流 19
汗流如雨 0
汗流浹背 31
汗流滿面 7
汗液 11
汗液 11
汗溼 9
汗漬 9
汗濕 8
汗牛充棟 9
汗珠 25
汗珠子 0
汗疹 11
汗腥 0
汗腺 47
汗臭 4
汗臭味 4
汗血 2
汗衣 5
汗衫 15
汗跡 0
汗青 56
汗顏 35
汗馬功勞 4
汗鹼 0
汘 0
汙 105
汙 105
汙名 1
汙吏 0
汙垢 13
汙染 280
汙水 44
汙泥 42
汙濁 2
汙衊 5
汙辱 7
汙辱 7
汙點 3
汚 0
汚 0
汚 0
汛 106
汛期 17
汜 1360
汝 753
汝南 10
汝南王 0
汝曹 2
汝等 46
汝輩 2
汞 4
汞 492
汞合金 1
江 10566
江上 27
江中 30
江北 20
江北地區 0
江南 296
江南地區 0
江口 61
江口地區 0
江名 0
江夏 19
江寧 26
江山 215
江山易改 4
江山易改本性難移 1
江山美人 6
江岸 12
江州 26
江心 5
江戶 1
江東 71
江東父老 6
江水 109
江河 55
江河日下 4
江洋 4
江洋大盜 4
江洲車兒 0
江洲車兒 0
江流 15
江浙 99
江浪 2
江淮 21
江湖 853
江湖味 1
江湖氣 2
江湖氣概 0
江漢 52
江漢平原 4
江田島 6
江畔 25
江米 6
江翠 27
江翠國中 24
江翠國小 322
江華 7
江蘇 219
江蘇人 3
江蘇省 179
江西 138
江西人 0
江西省 51
江輪 1
江邊 22
江郎才盡 9
江都 17
江防 10
江陰 14
江陵 29
江青 17
江面 19
江面上 2
池 2387
池上 390
池中物 2
池塘 162
池子 34
池心 4
池榭 2
池水 91
池沼 9
池裡 20
池魚 30
池魚之殃 18
池鹽 2
污 1177
污亂 0
污名 15
污吏 3
污垢 57
污損 78
污染 6730
污染區 34
污染源 839
污染物 915
污氣 4
污水 969
污水槽 2
污油 4
污泥 482
污濁 46
污物 47
污穢 77
污穢不堪 3
污蔑 12
污衊 21
污辱 39
污點 29
汥 4
汦 2
汧 9
汨 28
汨汨 13
汨羅江 5
汩 11
汩汩 10
汩沒 0
汪 2986
汪汪 258
汪汪的 18
汪洋 51
汪洋大海 11
汪精衛 9
汫 11
汬 0
汭 10
汮 0
汯 19
汰 453
汰去 0
汰換 0
汰沙 0
汰舊換新 97
汱 10
汲 59
汲取 102
汲引 0
汲水 14
汲水桶 0
汲水機 0
汲汲 26
汲汲於 12
汲汲營營 29
汳 8
汴 129
汴京 6
汵 0
汵 0
汵 0
汶 518
汶 518
汶萊 0
汷 0
汸 9
汹 0
決 3967
決一死戰 12
決一死戰 12
決不 247
決不在 4
決不寬貸 2
決不是 39
決不會 58
決不能 39
決不食言 1
決勝 81
決勝負 32
決口 12
決堤 20
決定 10151
決定因素 54
決定性 118
決定權 69
決定論 30
決心 814
決志 28
決意 49
決戰 362
決斷 39
決斷力 13
決無 21
決無此事 1
決然 11
決策 2584
決策單位 22
決策當局 13
決算 856
決算書 17
決算法 19
決絕 13
決裂 29
決計 10
決議 2542
決議文 22
決議書 5
決議案 119
決賽 639
決賽權 5
決選 88
決雌雄 5
決雌雄 5
決非 26
決非偶然 2
決鬥 81
決鬥者 10
汻 4
汼 0
汽 2243
汽力 7
汽包 0
汽化 21
汽化器 3
汽化熱 2
汽壓 5
汽水 167
汽水瓶 2
汽油 612
汽油引擎 61
汽油彈 18
汽油機 0
汽油車 21
汽泡 15
汽燈 0
汽球 80
汽笛 14
汽筒 0
汽缸 442
汽船 11
汽艇 12
汽表 0
汽車 16267
汽車旅社 0
汽車旅館 407
汽輪機 10
汽輪發電 1
汽輪發電機 8
汽錘 0
汽鍋 4
汽門 76
汽體 0
汾 25
汾水 2
汾河 0
汾酒 0
汾陽 17
汿 0
沀 4
沁 250
沁人心脾 3
沁入 6
沂 169
沂 169
沃 481
沃 481
沃依采克 1
沃土 22
沃壤 1
沃瘠 0
沃腴 1
沃野 6
沃野千里 5
沃野平疇 0
沃頓 0
沃饒 0
沄 17
沅 145
沆 6
沆 6
沆瀣 0
沆瀣一氣 3
沆瀣一氣 3
沇 2
沈 3616
沈 3616
沈下 5
沈不住氣 2
沈住氣 22
沈傳芷 1
沈冤 4
沈勇 1
沈厚 3
沈吟 15
沈寂 66
沈得住氣 1
沈得住氣 1
沈得住氣 1
沈思 98
沈思往事 0
沈悶 31
沈括 9
沈沒 43
沈浮 18
沈浸 67
沈淪 30
沈湎 1
沈湎酒色 0
沈溺 45
沈滯 1
沈滯性 0
沈澱 264
沈澱出來 4
沈澱法 12
沈澱物 29
沈濁 0
沈甸甸 3
沈疔 0
沈痛 27
沈痼 0
沈痾 8
沈睡 79
沈睡不醒 0
沈砂池 17
沈積 275
沈積岩 9
沈積石 0
沈筒 0
沈約 3
沈船 14
沈落 10
沈著 45
沈迷 47
沈迷不醒 0
沈醉 71
沈重 344
沈降 80
沈陷 18
沈雁冰 1
沈雲英 0
沈靜 45
沈靜下來 3
沈香 20
沈鬱 21
沈魚落雁 1
沈默 241
沈默不語 6
沈默寡言 13
沉 987
沉下 10
沉下來 1
沉下去 15
沉住氣 1
沉入 22
沉冤 2
沉勇 1
沉吟 86
沉寂 53
沉寂多年 1
沉寂已久 6
沉得 5
沉得 5
沉得住氣 6
沉得住氣 6
沉思 150
沉悶 43
沉沉 39
沉沉入睡 0
沉沒 88
沉浸 15
沉浸在 63
沉浸於 12
沉淪 67
沉湎 3
沉湎於 3
沉溺 13
沉溺在 21
沉溺於 21
沉澱 188
沉澱劑 1
沉澱法 7
沉澱物 11
沉甸甸 12
沉痛 21
沉睡 109
沉睡不醒 2
沉積 168
沉積物 80
沉著 72
沉著應戰 0
沉迷 31
沉迷在 10
沉迷於 22
沉醉 38
沉醉在 37
沉醉於 7
沉重 342
沉降 52
沉降縫 0
沉降速度 4
沉陷 36
沉陷在 2
沉陷於 1
沉靜 40
沉魚落雁 4
沉默 320
沉默寡言 6
沉默是金 3
沊 4
沋 5
沌 19
沍 1
沎 4
沏 14
沏 14
沏油 1
沏茶 3
沐 155
沐浴 401
沐浴乳 1
沐浴精 11
沐猴而冠 1
沐猴而冠 1
沐雨櫛風 0
沑 0
沑 0
沒 1216
沒 12161
沒車 100
沒電 100
沒上 95
沒上來 39
沒上去 2
沒下 10
沒下來 1
沒下去 0
沒世 0
沒中 6
沒主意 0
沒事 771
沒事人 1
沒事兒 3
沒事兒 3
沒事幹 14
沒事找事 6
沒亮 9
沒人 100
沒什麼 690
沒付 0
沒份 5
沒使 4
沒來 307
沒來由 14
沒來頭 0
沒借 2
沒傳 3
沒像 13
沒光 2
沒入 122
沒出 38
沒出來 9
沒出去 3
沒出息 12
沒分 4
沒分 4
沒刺 1
沒剩 2
沒勁 5
沒動 11
沒勸 0
沒升 0
沒去 209
沒取 0
沒受 12
沒向 12
沒吵 5
沒吸 2
沒命 33
沒唱 6
沒唸 10
沒回 44
沒回來 7
沒回去 2
沒在 83
沒填 5
沒大 10
沒大沒小 10
沒奈何 3
沒套 2
沒好氣 24
沒好氣兒 0
沒好氣兒 0
沒學 15
沒守 1
沒完 30
沒完沒了 49
沒射 3
沒對 14
沒小 7
沒少 1
沒就 1
沒得說 4
沒得說 4
沒心 12
沒心沒肝 1
沒志氣 4
沒念 2
沒怎樣 3
沒怎樣 3
沒怎麼 1
沒怎麼 1
沒怎麼 1
沒怎麼 1
沒想 60
沒想到 1453
沒意思 21
沒慢 0
沒戲 5
沒戲可拍 0
沒打 46
沒抄 4
沒把握 23
沒拿 20
沒拿到 4
沒指望 5
沒排 2
沒接 44
沒提 31
沒搬 2
沒擠 0
沒收 314
沒效 26
沒救 47
沒敢 8
沒敲 0
沒數 3
沒書 1
沒有 28543
沒有第二句話 1
沒有話說 3
沒有說的 1
沒有說得 1
沒有過 19
沒染 1
沒權 6
沒權沒勢 0
沒歉 0
沒毒 116
沒比 9
沒氣 2
沒求 2
沒沒 10
沒沒無聞 13
沒油 7
沒沿 0
沒法 189
沒法兒 4
沒法兒 4
沒法子 42
沒法子 42
沒法沒天 0
沒洗 14
沒派 1
沒消 2
沒測 4
沒滿 11
沒演 2
沒炒 0
沒燒 4
沒爹 0
沒爹沒娘 2
沒牽 1
沒玩 30
沒球 2
沒甚麼 48
沒甚麼 48
沒生 7
沒用 256
沒用到 10
沒由 5
沒畫 5
沒當 3
沒省 0
沒破 6
沒神 0
沒種 17
沒空 101
沒穿 28
沒笑 1
沒筆 1
沒簽 1
沒精打彩 0
沒精打采 1
沒給 52
沒經 6
沒聲 3
沒聲沒息 0
沒能 131
沒臉 8
沒臉見人 2
沒與 1
沒興 1
沒船 0
沒良心 30
沒花 8
沒落 145
沒藏 1
沒藥 20
沒被 119
沒要 2
沒見識 4
沒規矩 1
沒親 2
沒親沒故 0
沒討 1
沒記 10
沒設 29
沒試 3
沒說 141
沒說出 2
沒說到 2
沒請 6
沒譜 0
沒讀 116
沒變 49
沒賞 3
沒起來 0
沒趕 16
沒趣 11
沒路可走 0
沒蹺 0
沒輸 1
沒轍 12
沒辦 15
沒辦法 566
沒通 4
沒造化 1
沒進 15
沒進來 4
沒進去 3
沒運 1
沒過 18
沒過來 0
沒過去 2
沒道理 36
沒酒 1
沒醒 1
沒量 1
沒銷 1
沒錢 223
沒錯 924
沒開 35
沒關 25
沒關係 684
沒附 3
沒雨 30
沒面子 19
沒頂 0
沒頭 3
沒頭沒尾 7
沒頭沒腦 21
沒頭腦 1
沒風 30
沒飛 8
沒養 4
沒騎 0
沒骨山水 1
沒骨氣 1
沒骨畫 1
沒骨花卉 1
沒骨頭 1
沒骨頭 1
沒齒 0
沒齒不忘 0
沒齒不忘 0
沒齒難忘 7
沓 36
沓至 1
沓雜 0
沔 7
沕 5
沖 2108
沖上 17
沖上來 0
沖上去 1
沖下 26
沖下來 0
沖下去 3
沖來 11
沖來沖去 0
沖倒 2
沖出 26
沖出來 3
沖出去 0
沖到 17
沖刷 104
沖刺 0
沖力 6
沖動 8
沖印 220
沖去 6
沖和 1
沖喜 2
沖垮 7
沖壓 108
沖壞 3
沖天 88
沖天爐 0
沖床 94
沖床工 0
沖掉 15
沖撞 4
沖撞力 0
沖擊 46
沖擊波 0
沖擊波 0
沖擊韌性 0
沖散 8
沖服 2
沖決 0
沖決堤防 0
沖沖 15
沖沖水 1
沖泡 117
沖泡式 9
沖洗 466
沖洗照片 4
沖淡 86
沖溝 0
沖澡 1
沖激 16
沖犯 14
沖破 12
沖程 0
沖積 30
沖積土 7
沖積堤 0
沖積層 5
沖積平原 23
沖積成 3
沖積扇 33
沖積物 2
沖突 6
沖繩 180
沖繩島 3
沖茶 1
沖著 5
沖蝕 81
沖襟 0
沖走 31
沖起 0
沖起來 0
沖退 10
沖進 9
沖進來 0
沖進去 2
沖過 16
沖過來 5
沖過去 9
沖銷 241
沖鋒 0
沖鋒槍 0
沖鋒陷陣 0
沗 0
沘 0
沘 0
沙 3377
沙丁魚 22
沙丘 144
沙俄 1
沙勞越 11
沙包 20
沙啞 125
沙囊 0
沙土 33
沙地 49
沙坑 38
沙坪霸 1
沙坵 0
沙堆 7
沙場 24
沙場 24
沙場老兵 0
沙場老兵 0
沙場老將 3
沙場老將 3
沙塵 8
沙塵暴 1
沙士 11
沙子 70
沙層 4
沙岸 21
沙巴 149
沙市 36
沙彌 135
沙彌戒 1
沙律 1
沙拉 320
沙拉吧 28
沙拉油 67
沙拉醬 1
沙文 13
沙文主義 25
沙朗 1
沙林 36
沙果 0
沙沙 71
沙沙作響 10
沙沙聲 9
沙河 23
沙洲 108
沙渚 0
沙漏 21
沙漠 587
沙漠之舟 1
沙漠地 1
沙漠高地 0
沙灘 499
沙灘裝 1
沙灘車 0
沙灘鞋 1
沙烏地 151
沙烏地阿拉伯 151
沙爾 7
沙特阿拉伯 0
沙田 278
沙發 431
沙發墊 1
沙發椅 27
沙皇 16
沙盤 6
沙盤推演 38
沙盤演練 2
沙眼 0
沙石 27
沙礫 5
沙粒 21
沙糖 5
沙茶 42
沙茶醬 1
沙荒 0
沙袋 7
沙裡淘金 1
沙裡淘金 1
沙豬 1
沙達特 3
沙金 5
沙鍋 9
沙鑽蟹 0
沙門 109
沙魚 8
沙鷗 28
沙鹿 966
沙黃色 0
沙龍 324
沚 13
沛 586
沛沛 12
沛然 30
沜 4
沝 19
沞 0
沟 0
沠 0
没 0
没 0
沢 0
沪 0
沫 105
沫兒 0
沫兒 0
沬 36
沭 300
沮 28
沮 28
沮喪 240
沮泄 0
沮洳 4
沮索 0
沯 0
沯 0
沰 6
沱 67
沱江 11
沲 0
河 2796
河上 68
河中 50
河內 84
河北 179
河北人 3
河北省 70
河南 251
河南人 0
河南省 77
河口 178
河圖洛書 1
河堤 100
河外星系 1
河套 5
河山 35
河岸 167
河川 1555
河工 75
河床 159
河底 14
河底下 0
河心 3
河朔 2
河東 54
河東獅吼 2
河柳 0
河梏 0
河槽 2
河水 179
河水不犯井水 0
河沙 5
河沿 0
河泥 27
河洛 703
河洲 2
河流 393
河清海晏 0
河渠 7
河渡 0
河港 14
河溝 6
河漢 5
河濱 217
河濱公園 61
河灘 17
河灘地 2
河燈 2
河畔 103
河神 4
河系 76
河網 6
河網化 0
河船 7
河蚌 3
河蟹 1
河裡 44
河西 67
河西走廊 15
河谷 107
河豚 49
河身 1
河運 10
河道 156
河邊 159
河邊北街 14
河邊北路 0
河邊南街 0
河邊南路 0
河釣 2
河間 5
河間王 0
河防 2
河隄 0
河面 22
河面上 9
河馬 100
河魚 9
河鱒 1
河鱒 1
河鼓 0
沴 5
沴氣 0
沵 0
沵 0
沶 9
沷 3
沸 113
沸 113
沸水 74
沸沸 5
沸沸 5
沸湯 1
沸石 29
沸騰 250
沸騰床 0
沸騰鋼 0
沸點 122
油 8960
油井 8
油亮 11
油亮亮 3
油價 372
油光 10
油光滿面 2
油加利樹 9
油印 17
油印機 1
油品 0
油嘴 22
油嘴滑舌 5
油坊 0
油垢 17
油墨 309
油墨味 0
油壓 1040
油壓式 34
油壺 10
油層 19
油布 8
油庫 97
油廠 78
油彩 191
油得 1
油性 176
油房 0
油料 191
油料作物 6
油料庫 0
油晃晃 0
油松 1
油桐 27
油桐花 1
油桶 22
油條 26
油棕 0
油槽 104
油氈 1
油氣 205
油水 47
油污 72
油汪汪 0
油油 27
油泥 16
油泵 15
油液 4
油液 4
油渣 10
油滑 2
油漆 702
油漆匠 9
油漆工 7
油漆未乾 0
油漬 19
油灰 5
油炸 111
油炸物 3
油炸餅 0
油然 28
油然而生 31
油煎 9
油煎餅 1
油煙 110
油燈 32
油狀 12
油瓶 6
油田 48
油田區 0
油畫 1023
油畫家 12
油畫院 0
油的 118
油地 117
油石 6
油礦 37
油管 57
油管線 1
油箱 152
油粕 2
油紙 26
油綠 7
油罐 16
油罐車 278
油耗 58
油脂 532
油脂 532
油腔滑調 0
油膏 15
油膏盒 0
油膜 24
油膩 82
油膩感 1
油膩膩 2
油船 5
油艙 1
油花 3
油茶 7
油菜 41
油菜花 6
油藏 3
油費 3
油質 34
油跡 0
油車 43
油輪 64
油酸 38
油酸鹽 0
油鋸 0
油鍋 61
油門 119
油頁岩 8
油頭滑腦 0
油頭粉面 0
油飯 92
油餅 7
油麵 15
油麻菜籽 9
沺 103
治 5177
治世 19
治亂 20
治了 1
治事 8
治兵 2
治喪 14
治喪委員 2
治喪委員會 8
治國 128
治國平天下 5
治外法權 2
治好 108
治學 59
治安 1000
治安人員 4
治安機關 23
治家 16
治家格言 3
治山 90
治平 176
治愈 5
治所 11
治本 103
治標 81
治權 31
治民官 0
治水 62
治河 4
治理 246
治產 1
治病 164
治病救人 0
治療 11906
治療法 130
治癒 254
治禮 0
治絲愈棼 1
治絲益棼 0
治績 5
治罪 55
治罪條例 0
治著 23
治著 23
治裝 1
治裝費 0
治軍 20
治軍之道 0
治道 19
沼 200
沼地 4
沼氣 37
沼澤 281
沼澤地 40
沼澤地帶 3
沽 70
沽 70
沽名釣譽 7
沽酒 5
沾 504
沾上 67
沾不上邊 1
沾光 15
沾手 7
沾有 23
沾染 49
沾染上 5
沾染到 7
沾染習氣 0
沾水 18
沾汙 1
沾污 26
沾油 8
沾沾自喜 24
沾溼 5
沾潤 0
沾濕 38
沾用 1
沾粘 7
沾花惹草 0
沾襟 1
沾親帶故 0
沾邊 0
沾醬 1
沿 12
沿 11147
沿伸 9
沿例 0
沿兒 0
沿兒 0
沿岸 583
沿江 24
沿河 29
沿海 661
沿海各省 9
沿海各縣 0
沿海地區 134
沿海居民 9
沿用 169
沿線 317
沿習 6
沿著 565
沿街 54
沿街叫賣 6
沿街道 1
沿襲 51
沿路 136
沿路上 8
沿途 383
沿途上 1
沿門 2
沿門乞討 0
沿門托缽 1
沿革 2776
泀 4
況 2832
況且 478
泂 574
泃 23
泃 23
泄 100
泄恨 0
泄憤 0
泄殖腔 4
泄氣 1
泄沓 0
泄泄汨汨 0
泄泄沓沓 1
泄洪 3
泄漏 5
泄露 10
泄露 10
泅 20
泅水 1
泅水而過 0
泅泳 9
泆 56
泇 3
泈 0
泉 3068
泉下 3
泉州 161
泉州市 32
泉州街 28
泉水 137
泉源 176
泉眼 0
泉石 4
泉頭 0
泊 726
泊 726
泊位 1
泊岸 5
泊船 8
泋 0
泌 275
泌 275
泌尿 423
泌尿器 1
泌尿外科 95
泌尿科 551
泌尿管 0
泌尿量 0
泍 3
泎 0
泎 0
泏 6
泐 31
泑 5
泒 6
泓 580
泔 32
泔水 0
法 1
法 1
法 3831
法 38381
法事 20
法人 2322
法令 4421
法令滋章盜賊多有 1
法佈施 1
法例 38
法兒 14
法兒 14
法典 47
法利 126
法制 582
法則 872
法力 71
法務 521
法務部 885
法名 17
法向量 0
法商 126
法商學院 125
法器 53
法國 442
法國 4426
法國人 84
法國人 84
法國史 4
法國史 4
法國大革命 42
法國大革命 42
法國話 0
法國話 0
法國革命 13
法國革命 13
法堂 11
法場 6
法場 6
法塵 1
法壇 1
法外 45
法外情 1
法外施恩 0
法子 776
法學 673
法學博士 113
法學士 31
法學家 1
法學院 487
法官 905
法官大人 8
法定 1284
法定人數 10
法定代理 14
法定代理人 164
法定公積 1
法定公積金 1
法定年齡 5
法家 40
法寶 148
法屬 13
法屬 13
法屬圭亞那 5
法屬圭亞那 5
法帖 41
法師 1076
法幣 2
法度 63
法座 9
法庭 367
法式 1
法式 345
法式料理 1
法式料理 1
法律 7450
法律之前 4
法律之前人人平等 7
法律學 122
法律學系 274
法律性 6
法律系 464
法律行為 43
法律責任 273
法律顧問 109
法意 6
法拉 251
法拉第 27
法拍屋 0
法政 79
法數 4
法文 268
法文 268
法文系 58
法文系 58
法新社 134
法新社 134
法施 1
法旨 6
法書 14
法曼陀羅 1
法會 236
法服 18
法朗 35
法朗西一世 0
法朗西一世 0
法案 1099
法條 201
法櫃奇兵 14
法權 9
法治 555
法海 0
法源 0
法物 26
法理 82
法理學 19
法理情 7
法界 248
法界體性智 1
法皇 9
法相 109
法眼 37
法禁 2
法科 10
法科大學 5
法籍 17
法紀 18
法統 24
法網 34
法網恢恢 2
法線 14
法老 26
法色 1
法華三昧 1
法華經 1
法蘭 152
法蘭克 72
法蘭克福 183
法蘭盤 0
法蘭絨 4
法蘭絨 4
法蘭西 117
法蘭西共和國 1
法號 13
法螺 9
法術 301
法衣 10
法裔 13
法裔 13
法西斯 25
法西斯主義 21
法西斯黨 0
法規 6602
法語 297
法語 297
法語區 7
法語區 7
法語系 7
法語系 7
法警 36
法身 66
法身佛 1
法身舍利 1
法軍 42
法軍 42
法輪 57
法輪功 0
法輪常轉 1
法辦 206
法郎 171
法郎 171
法部 58
法醫 202
法醫學 65
法門 372
法院 2065
法雲 1
法雲地 1
泖 4
泗 280
泘 0
泙 25
泚 9
泛 1925
泛亞 0
泛出 11
泛出來 0
泛函分析 12
泛愛 6
泛指 74
泛池 2
泛泛 4
泛泛之交 4
泛泛之輩 10
泛泛的 2
泛泛地 2
泛濫 36
泛濫成災 1
泛白 16
泛神論 3
泛稱 12
泛紅 29
泛美 43
泛舟 179
泛覽 0
泛論 9
泛讀 0
泛起 77
泛遊 3
泛酸 17
泛非主義 0
泛音 23
泜 3
泜泜 1
泝 12
泞 7
泟 0
泠 71
泠泠 6
泠洌 0
泡 1515
泡上 11
泡影 4
泡得 38
泡桐 17
泡水 168
泡沫 625
泡沫塑料 0
泡沫紅茶 70
泡泡 238
泡泡糖 5
泡洋塑膠 0
泡湯 116
泡澡 20
泡狀 6
泡疹 37
泡的 38
泡綿 42
泡芙 0
泡茶 187
泡菜 39
泡著 4
泡起 4
泡起來 2
泡開 5
泡飯 3
泡麵 425
波 6864
波 6864
波光 43
波光 43
波光蕩漾 0
波光蕩漾 0
波動 468
波動 468
波及 324
波及 324
波哥大 6
波哥大 6
波國 13
波國 13
波士尼亞 0
波士頓 347
波士頓 347
波多黎各 72
波多黎各 72
波導 37
波導 37
波導管 2
波導管 2
波峰 36
波峰 36
波希米亞 37
波希米亞 37
波幅 54
波幅 54
波形 131
波形 131
波形板 0
波形板 0
波恩 6
波恩 6
波折 78
波折 78
波斯 228
波斯 228
波斯文 2
波斯文 2
波斯灣 202
波斯灣 202
波斯菊 34
波斯菊 34
波斯貓 71
波斯貓 71
波旁王朝 2
波旁王朝 2
波昂 99
波昂 99
波本酒 0
波本酒 0
波束 24
波束 24
波段 314
波段 314
波河 2
波河 2
波河平原 0
波河平原 0
波流 17
波流 17
波浪 308
波浪 308
波浪形 4
波浪形 4
波源 4
波源 4
波濤 59
波濤 59
波濤 59
波濤 59
波濤洶湧 38
波濤洶湧 38
波濤洶湧 38
波濤洶湧 38
波濤起伏 4
波濤起伏 4
波濤起伏 4
波濤起伏 4
波瀾 39
波瀾 39
波瀾壯闊 13
波瀾壯闊 13
波灣 76
波灣 76
波灣戰爭 56
波灣戰爭 56
波爾 121
波爾 121
波爾多液 1
波爾多液 1
波爾多液 1
波爾多液 1
波特 257
波特 257
波狀 26
波狀 26
波紋 58
波紋 58
波紋紙 0
波紋紙 0
波羅 285
波羅 285
波羅的海 51
波羅的海 51
波羅蜜 1
波羅蜜 1
波美 10
波美 10
波義耳 2
波義耳 2
波義耳定律 1
波義耳定律 1
波茨坦 79
波茨坦 79
波蘭 378
波蘭 378
波蘭人 9
波蘭人 9
波詭雲譎 1
波詭雲譎 1
波譎雲詭 2
波譎雲詭 2
波譜 12
波譜 12
波谷 17
波谷 17
波速 11
波速 11
波長 340
波長 340
波霸 1
波音 260
波音 260
波音噴射機 0
波音噴射機 0
泣 1431
泣不成聲 20
泣別 2
泣涕 3
泣然 0
泣血 10
泣訴 12
泣鬼神 10
泤 0
泥 472
泥 5863
泥人 11
泥像 1
泥匠 9
泥土 201
泥土牆 0
泥土路 1
泥土面 0
泥坑 1
泥坯 1
泥坯 1
泥垢 2
泥塑 27
泥塑像 0
泥塑木雕 2
泥塘 5
泥壁 4
泥娃娃 5
泥屋 3
泥岩 59
泥巖 0
泥工 17
泥巴 48
泥巴路 0
泥板 39
泥水 21
泥水匠 2
泥水工 7
泥水師 0
泥污 4
泥沙 77
泥沼 35
泥淖 26
泥滓 0
泥漿 74
泥潦 0
泥潭 1
泥濕 0
泥濘 59
泥濘不堪 3
泥灰 3
泥灰岩 1
泥炭 8
泥煤 15
泥牛 1
泥牛入海 0
泥球 0
泥療法 0
泥盆紀 0
泥石流 1
泥砂 42
泥磚 2
泥胎 0
泥腳 0
泥菩薩 5
泥菩薩落水 1
泥菩薩過江 2
泥菩薩過河 1
泥醉 0
泥鰍 34
泦 0
泧 2
注 1510
注入 569
注冊 20
注到 1
注塑 4
注定 141
注射 1583
注射器 41
注射筒 17
注射針 18
注心 2
注意 8405
注意事項 3676
注意到 696
注意力 403
注慕 2
注明 15
注水 20
注油 15
注消 0
注滿 12
注疏 11
注疏 11
注目 195
注目禮 3
注腳 3
注視 170
注視著 120
注解 18
注釋 75
注重 1128
注銷 1
注音 464
注音字母 2
注音文 1
注音符號 142
泩 53
泪 0
泫 52
泫然欲淚 0
泬 140
泭 218
泮 73
泯 50
泯沒 1
泯滅 18
泰 9632
泰勒 139
泰北 241
泰半 113
泰和 136
泰國 2440
泰國人 18
泰國拳 2
泰國話 0
泰坦 19
泰安 346
泰安縣 0
泰山 717
泰山北斗 2
泰山壓卵 0
泰山壓頂 2
泰山梁木 0
泰山鄉 490
泰式 1
泰式料理 1
泰戈爾 30
泰拳 2
泰文 1
泰斗 35
泰族 2
泰晤士報 31
泰晤士河 12
泰水 37
泰然 30
泰然族 0
泰然自若 2
泰然處之 1
泰王 13
泰西 11
泰迪熊 0
泰銖 0
泰陵 23
泰雅 584
泰順 53
泰順街 25
泱 92
泱泱 13
泱泱大國 2
泲 6
泳 628
泳員 0
泳將 10
泳帽 9
泳池 114
泳衣 34
泳裝 86
泳裝展示會 0
泳褲 0
泴 0
泵 345
泵浦 212
泹 6
泿 0
洀 2
洁 449
洂 0
洂 0
洃 460
洄 137
洄游 67
洄瀾 67
洄遊 1
洅 0
洆 0
洇 121
洈 110
洉 145
洊 16
洋 15519
洋人 33
洋傘 149
洋八股 0
洋兵 2
洋務 4
洋務局 0
洋務派 0
洋務運動 6
洋化 15
洋參 3
洋商 22
洋囡囡 0
洋基 150
洋基隊 32
洋奴 0
洋奴哲學 0
洋娃娃 193
洋將 132
洋山芋 0
洋布 2
洋式 8
洋房 26
洋文 14
洋文書 2
洋服 45
洋槍 1
洋槍洋砲 0
洋槍隊 1
洋槐 1
洋樓 27
洋氣十足 0
洋水 12
洋河 7
洋油 3
洋法生產 0
洋洋 59
洋洋大觀 3
洋洋得意 23
洋洋灑灑 17
洋洋自得 3
洋流 44
洋涇濱 1
洋涇濱英語 0
洋溢 145
洋溢著 73
洋火 0
洋灰 0
洋為中用 0
洋煙 4
洋琴 1
洋瓷 0
洋相 6
洋相百出 0
洋相盡出 0
洋粉 1
洋紅 7
洋紅色 5
洋船 295
洋芋 18
洋芋片 12
洋莓 0
洋菇 40
洋菇菌 0
洋菜 24
洋菜粉 1
洋菜膠 1
洋菸 29
洋蔥 238
洋蔥味 1
洋蘭 9
洋行 153
洋裁 1
洋裝 116
洋財 0
洋貨 9
洋車 4
洋車夫 0
洋迷 0
洋酒 76
洋金 1
洋釘 3
洋銀 0
洋錢 1
洋鐵 4
洋面 32
洋風 7
洋香瓜 34
洋鬼子 8
洌 29
洍 15
洎 91
洏 425
洐 51
洑 8
洒 106
洒掃 4
洒水 1
洒脫 6
洓 0
洔 0
洕 0
洖 2
洗 29
洗 2995
洗上 1
洗上來 0
洗上去 0
洗下 2
洗下來 1
洗下去 0
洗不掉 9
洗不清 3
洗了 32
洗來 3
洗來洗去 0
洗光 1
洗冤 0
洗出 30
洗出來 13
洗出去 0
洗到 4
洗刷 85
洗前 6
洗劫 44
洗去 50
洗反 0
洗地 7
洗地板 1
洗塵 9
洗好 30
洗完 60
洗得 32
洗心 88
洗心革面 7
洗成 6
洗手 191
洗手不幹 3
洗手乳 1
洗手間 59
洗掉 51
洗斷 0
洗法 9
洗洗 22
洗洗弄弄 0
洗浴 16
洗淨 383
洗清 30
洗滌 230
洗滌劑 37
洗滌器 4
洗漱 1
洗潔精 2
洗澡 353
洗澡房 0
洗澡水 9
洗澡盆 2
洗濯 78
洗煉 1
洗煤 19
洗照片 4
洗牌 89
洗牙 1
洗物槽 0
洗的 32
洗眼 10
洗碗 87
洗碗機 44
洗禮 192
洗練 12
洗罪 3
洗耳 6
洗耳恭聽 11
洗耳諦聽 1
洗耳靜聽 0
洗胃 36
洗脫 12
洗腎 0
洗腦 44
洗腦術 0
洗腳 46
洗腸 3
洗臉 125
洗臉盆 9
洗著 6
洗衣 433
洗衣匠 0
洗衣商 0
洗衣室 6
洗衣店 167
洗衣服 33
洗衣板 4
洗衣機 828
洗衣粉 98
洗衣費 3
洗起 0
洗起來 1
洗車 1
洗透 0
洗過 32
洗過來 0
洗過去 0
洗選 10
洗錢 133
洗雪 3
洗面 50
洗面乳 1
洗頭 71
洗頭髮 6
洗馬 0
洗馬 0
洗髮精 125
洘 311
洙 43
洚 10
洛 2829
洛伊 55
洛克 284
洛克斐勒 12
洛克菲勒 10
洛基 17
洛夫 148
洛杉磯 832
洛林 21
洛水 29
洛磯 54
洛磯山 41
洛神 32
洛神花 1
洛神賦 7
洛美 18
洛蘭 1
洛衫磯 7
洛赫遜 0
洛陽 211
洛陽市 5
洛陽紙貴 7
洛陽花 0
洜 0
洝 8
洞 1944
洞兒 0
洞兒 0
洞口 89
洞名 0
洞天 26
洞天福地 2
洞子 3
洞察 48
洞察力 20
洞府 4
洞庭 27
洞庭湖 27
洞徹 3
洞悉 99
洞房 22
洞房花燭 3
洞房花燭夜 13
洞洞 17
洞洞裝 0
洞燕 1
洞燭 21
洞穴 221
洞穿 7
洞窟 0
洞簫 5
洞若觀火 2
洞見 15
洞視 5
洞鑒 1
洞開 11
洟 56
洠 45
洡 0
洢 72
洣 0
洤 0
津 1313
津巴布韋 3
津梁 4
津津有味 38
津津樂道 53
津浦路 0
津浦鐵路 1
津液 14
津液 14
津渡 4
津要 0
津貼 720
洦 0
洦 0
洧 254
洨 381
洨 25
洩 672
洩下 3
洩下來 2
洩下去 0
洩了 27
洩入 2
洩出 11
洩出來 2
洩出去 0
洩到 3
洩去 0
洩在 4
洩天機 8
洩好 0
洩完 0
洩密 0
洩底 4
洩得 31
洩忿 2
洩恨 10
洩憤 0
洩成 1
洩掉 1
洩氣 36
洩法 0
洩洪 57
洩洪區 0
洩漏 339
洩的 32
洩給 0
洩露 92
洩題 36
洩題案 14
洪 6492
洪亮 12
洪仁柑 0
洪任諭 0
洪大 10
洪峰 53
洪恩 5
洪承疇 0
洪昇 1
洪武 41
洪水 367
洪水為患 1
洪水猛獸 12
洪流 125
洪深 1
洪湖 2
洪澤湖 4
洪濤 8
洪濤 8
洪爐 1
洪福 21
洪福齊天 1
洪秀全 13
洪積層 3
洪荒 18
洪荒世界 2
洪荒時代 6
洪荒時期 2
洪都拉斯 3
洪量 34
洪門 60
洫 76
洬 39
洭 18
洮 63
洮 63
洯 1
洰 1
洱 80
洲 3608
洲渚 1
洲界 2
洲際 62
洲際飛彈 11
洳 19
洴 76
洴 76
洵 167
洶 265
洶洶 12
洶涌 2
洶湧 108
洶湧澎湃 4
洷 458
洸 130
洸 130
洹 83
洺 23
活 8018
活上 67
活下去 123
活不下去 24
活不了 6
活不成 2
活了 107
活人 74
活人無數 1
活人無算 0
活佛 83
活像 39
活出 152
活到 67
活到老 22
活到老學到老 18
活剝生吞 2
活力 1196
活力充沛 22
活動 46488
活動份子 0
活動力 120
活動家 3
活動性 60
活動於 50
活動範圍 55
活動靶 1
活動項目 116
活勞動 0
活化 574
活化劑 17
活化能 20
活受罪 2
活口 12
活命 125
活命恩人 0
活命要緊 0
活在 501
活地獄 0
活埋 147
活塞 148
活塞桿 9
活塞環 26
活字 22
活字典 21
活字印刷 0
活字版 0
活字盤 0
活存 47
活寶 11
活得 250
活性 776
活性中心 2
活性劑 237
活性染料 12
活性炭 37
活性碳 246
活扣 4
活捉 23
活會 10
活期 91
活期儲蓄 64
活期存款 100
活栓 1
活樹 3
活死人 21
活水 200
活泉 39
活活 49
活活潑潑 10
活潑 1119
活潑生動 53
活火山 7
活版 6
活版印刷 2
活現 18
活生生 159
活用 343
活的 474
活祭 26
活結 4
活絡 192
活罪難逃 0
活荷載 0
活菩薩 1
活著 512
活見鬼 2
活計 2
活記 7
活詞 0
活該 48
活讀 0
活讀書 1
活象 3
活起 7
活起來 14
活路 23
活蹦亂跳 23
活蹦鮮跳 0
活躍 405
活過 41
活過來 11
活釦 0
活門 76
活靈活現 10
活靶 1
活頁 103
活頁簿 0
活頁紙 24
活體 108
活魚 73
活龍活現 3
洼 14
洽 175
洽 1750
洽公 0
洽商 208
洽聞 1
洽詢 1861
洽談 358
洽購 121
洽辦 62
派 5215
派上 10
派上用場 36
派上用場 36
派下來 1
派下去 0
派人 311
派來 59
派兵 29
派兵助陣 0
派兵攻打 0
派出 282
派出來 0
派出去 1
派出所 541
派別 62
派到 22
派去 25
派司 4
派員 921
派團 15
派定 11
派對 237
派性 3
派生 9
派系 478
派給 7
派調 2
派走 0
派車 71
派車來 0
派車去 0
派軍 10
派遣 393
派頭 6
派駐 164
派駐在 6
派駐所 1
洿 11
浀 87
流 8689
流上 7
流上來 0
流上去 0
流下 132
流下來 11
流下去 2
流下淚來 6
流了 53
流些 0
流亡 107
流亡國外 0
流亡政府 51
流亡海外 9
流人 21
流來 9
流來流去 0
流俗 14
流個 9
流傳 479
流傳下去 4
流傳力學 0
流傳後世 2
流傳開來 1
流光 34
流入 301
流出 648
流出來 31
流出去 5
流別 0
流利 246
流到 77
流刺網 36
流動 816
流動人口 36
流動力學 10
流動性 178
流動戶口 8
流動比率 29
流動資本 2
流動資產 27
流動資金 22
流化床 2
流去 20
流口水 82
流向 308
流回 17
流回來 0
流回去 0
流在 34
流域 368
流域面積 7
流失 618
流失量 21
流好 5
流娼 0
流完 0
流寇 9
流年 121
流年不利 13
流弊 16
流彈 14
流得 9
流徙 4
流掉 5
流放 63
流散 19
流於 210
流於形式 17
流星 1215
流星群 88
流星趕月 5
流星雨 1086
流星體 34
流暢 322
流暢性 17
流會 19
流標 0
流毒 0
流民 3
流氓 235
流氓婆 0
流氓集團 1
流氓頭 0
流水 2071
流水不腐 1
流水作業 2
流水席 11
流水帳 10
流水無情 7
流水線 4
流水行雲 1
流水賑 0
流汗 121
流沙 17
流洩出來 5
流派 120
流流 17
流流行性 0
流浪 1134
流浪兒 12
流浪漢 88
流涎咽唾 0
流涕 8
流淚 230
流火 1
流產 206
流產政變 2
流當 15
流盡 8
流眼淚 32
流砂 8
流程 3341
流程圖 570
流竄 76
流竄到 5
流網 361
流網漁船 1
流線 72
流線型 20
流膿 15
流芳 15
流芳百世 2
流落 70
流落他鄉 2
流落異鄉 4
流著 80
流蕩 4
流蘇 21
流螢 19
流血 205
流血政變 4
流血犧牲 3
流血衝突 32
流行 6175
流行性 64
流行性感冒 263
流行於 30
流行曲 12
流行歌 46
流行歌曲 203
流行病 457
流行起來 10
流覽 181
流言 960
流言蜚語 1
流變 84
流賊 0
流質 20
流質性 0
流質食物 7
流走 6
流起 5
流起來 0
流轉 107
流通 2808
流通出來 0
流通手段 0
流通證券 0
流通費用 0
流通量 17
流通領域 7
流逝 81
流速 152
流速計 6
流連 85
流連忘返 65
流進 25
流進來 1
流進去 0
流遍 4
流遍 4
流過 80
流過來 0
流過去 1
流配 18
流量 1161
流量表 44
流量計 89
流離 44
流離失所 21
流離顛沛 0
流電 11
流露 134
流露出 96
流露出來 8
流風 18
流風所及 0
流體 379
流體冶金 0
流體冶金術 0
流體力學 253
流體動力 12
流體動力學 7
流體性 0
流體靜力 0
流體靜力學 2
流鶯 12
流麗 4
流點 4
流鼻水 1
流鼻涕 23
流鼻血 1
浂 8
浃 0
浄 0
浅 0
浅 0
浖 0
浗 0
浘 5
浙 120
浙大 4
浙東 3
浙江 316
浙江人 2
浙江省 105
浙省 0
浙西 2
浙西平原 0
浙贛 4
浚 132
浛 0
浜 0
浜 0
浝 0
浞 0
浟 2
浠 10
浠水 2
浡 10
浢 5
浣 49
浣 49
浣洗 1
浣滌 0
浣滌 0
浣熊 54
浣衣 2
浤 47
浥 14
浦 842
浦口 1
浦城 13
浧 3
浨 3
浩 1591
浩劫 145
浩嘆 6
浩大 76
浩如 8
浩如煙氣 0
浩歎 1
浩氣 3
浩浩 16
浩浩湯湯 1
浩浩盪盪 3
浩浩蕩蕩 51
浩淼 3
浩渺 3
浩瀚 193
浩瀚無涯 1
浩然 151
浩然之氣 3
浩然正氣 5
浩盪 0
浩繁 11
浩翰 15
浩蕩 16
浩齒 0
浪 13
浪 1378
浪人 164
浪子 416
浪子回頭 11
浪子回頭金不換 2
浪板 99
浪漫 2204
浪漫主義 46
浪漫曲 7
浪漫派 37
浪潮 176
浪潮洶湧 1
浪濤 21
浪濤花 0
浪瀾壯闊 0
浪琴錶 1
浪者 10
浪花 83
浪花四濺 0
浪蕩 15
浪蕩子式 0
浪費 1227
浪費成性 1
浪跡 11
浪跡天涯 19
浪頭 10
浫 0
浬 113
浭 2
浮 1941
浮上 62
浮上來 11
浮上去 1
浮世繪 0
浮來 0
浮來浮去 0
浮光 10
浮光掠影 16
浮冰 12
浮出 137
浮力 124
浮力計 0
浮動 221
浮動匯率 21
浮台 4
浮名 6
浮圖 3
浮圖經 0
浮在 53
浮塵 12
浮塵子 2
浮塵子 2
浮士德 27
浮夸 1
浮子 11
浮子 11
浮屍 21
浮屠 16
浮影 2
浮想 2
浮掠 0
浮於 20
浮標 44
浮橋 10
浮水 33
浮水印 48
浮沈 61
浮沉 46
浮沉不定 0
浮油 21
浮泛 5
浮泥 0
浮淺 6
浮渣 1
浮游 102
浮游動物 22
浮游植物 8
浮游生物 85
浮濫 83
浮燥 7
浮現 461
浮生 1897
浮生六記 3
浮生若夢 9
浮盪 1
浮石 11
浮筒 37
浮簽 0
浮腫 61
浮船 2
浮船塢 4
浮華 16
浮華不實 0
浮華世界 6
浮萍 42
浮蕩 0
浮薄 0
浮言 2
浮詞 1
浮誇 17
浮起 53
浮躁 20
浮躁不安 1
浮辭 1
浮遊 4
浮選 6
浮選機 0
浮釣 0
浮雕 147
浮雕像 2
浮雲 111
浮雲朝露 0
浮雲蔽日 0
浮靡 0
浮面 15
浮體 6
浮魚 4
浮點 148
浮點數 0
浯 43
浰 4
浱 0
浲 0
浲 0
浲 0
浳 0
浴 999
浴佛 18
浴佛節 1
浴堂 2
浴場 36
浴場 36
浴室 390
浴巾 46
浴帽 1
浴池 34
浴療學 0
浴盆 8
浴簾 12
浴缸 112
浴血 17
浴血 17
浴血奮戰 1
浴血戰 1
浴血苦戰 4
浴衣 6
浴袍 7
浵 10
浶 4
海 16293
海上 1112
海上保險 26
海上花列傳 0
海上運輸 13
海不揚波 0
海不揚波 0
海事 462
海事仲裁 1
海事專科學校 38
海事水產職業學校 23
海事法 3
海事法庭 0
海事職業學校 24
海倫 191
海倫凱勒 15
海內 14
海內外 331
海內存知己 9
海內海外 1
海北 6
海區 27
海協會 0
海南 125
海南島 66
海南雞飯 1
海原 4
海參 49
海參威 4
海口 181
海口市 17
海味 34
海味山珍 0
海員 61
海哩 6
海商 37
海商法 40
海嘯 77
海國 65
海地 188
海地人 0
海地共和國 6
海埔 131
海埔 131
海埔新生地 43
海埔新生地 43
海域 1129
海基會 0
海堤 120
海報 1203
海塘 1
海外 6551
海外僑胞 29
海外學人 37
海客 2
海專 28
海山 199
海山國中 6
海岬 12
海岸 2548
海岸公路 25
海岸線 220
海島 149
海島型 25
海峽 336
海峽兩岸 519
海市 17
海市蜃樓 18
海帶 61
海帶芽 1
海平線 5
海平面 94
海床 37
海底 647
海底撈月 1
海底撈針 10
海德 128
海德堡 58
海德堡大學 19
海怪 16
海戰 90
海扇 14
海拔 671
海捕 3
海損 5
海明威 31
海星 49
海景 93
海枯石爛 11
海棠 523
海水 951
海水不可斗量 1
海水浴 1
海水浴場 20
海水浴場 200
海水淡化廠 9
海水藍 1
海河 16
海油 3
海法 26
海波 12
海波 12
海洋 6433
海洋學 72
海洋學院 79
海洋性 117
海洋性氣候 5
海洋權 0
海洋氣候 5
海洋漁業 94
海洛因 54
海洛英 13
海派 37
海流 79
海浪 124
海浪滔天 0
海浬 37
海涅 1
海涵 37
海港 96
海溝 27
海溫 53
海潮 68
海濤 42
海濱 314
海濱公路 0
海灘 434
海灘裝 3
海灘鞋 4
海灣 253
海燕 28
海牙 65
海牛 2
海狗 15
海狸 4
海獅 26
海獸 1
海獺 9
海王 80
海王星 110
海生 123
海生植物 0
海產 256
海產品 10
海產店 1
海疆 19
海百合 2
海盆 6
海盜 190
海盜版 1
海盜船 40
海盜行為 2
海盤車 0
海相 8
海碗 2
海神 39
海神號 1
海禁 11
海端 104
海納百川 0
海綿 151
海綿狀 12
海綿體 1
海線 110
海纜 0
海膽 54
海航 10
海船 12
海芋 1100
海苔 97
海草 30
海菜 35
海葬 6
海葵 46
海藻 146
海蛆 0
海蛇 11
海蜇 2
海蜇皮 3
海蝕 94
海蝦 11
海蝶 2
海螺 13
海蟹 2
海蠣 0
海裡 80
海角 40
海角一樂園 5
海角天涯 0
海誓山盟 15
海豚 845
海象 81
海豹 67
海貍 11
海貝 16
海賊 1
海路 97
海軍 1151
海軍基地 9
海軍官校 51
海軍軍官 9
海軍醫院 3
海軍陸戰隊 83
海輪 2
海運 900
海運公司 33
海運界 2
海道 24
海邊 540
海里 35
海量 5
海釣 130
海錨 3
海門 18
海闊天空 212
海關 1385
海關人員 14
海關署 0
海防 85
海防港 0
海防線 0
海防部隊 25
海陸 540
海陸空 18
海陽 7
海隅 6
海難 60
海震 0
海霧 4
海面 304
海頓 80
海風 168
海馬 55
海驢 3
海鬼 1
海魚 34
海鮮 767
海鮮大餐 8
海鮮店 13
海鰻 20
海鱸 1
海鳥 55
海鷗 272
海鷗飛處 0
海鷲 2
海鹽 30
海龍 210
海龍王 34
海龜 132
浸 567
浸 17805
浸信教 9
浸信會 168
浸入 38
浸到 3
浸在 47
浸完 0
浸於 13
浸染 56
浸染法 5
浸水 72
浸沉 0
浸泡 158
浸泡在 17
浸洗 8
浸浴 13
浸淫 61
浸湎 0
浸溶 1
浸漬 69
浸潤 84
浸濕 20
浸禮 19
浸種 5
浸蝕 7
浸融 0
浸軟 4
浸透 39
浸過 14
浹 8
浹洽 0
浹背 2
浺 4
浻 10
浼 1
浽 1
浾 3
浿 4
涀 8
涁 0
涂 518
涃 5
涄 9
涅 320
涅字 1
涅槃 9
涅槃寂靜 1
涅槃經 1
涅槃罪度 1
涅白 1
涅而不緇 1
涅面 1
涅齒 1
涆 4
涇 42
涇渭 2
涇渭不分 0
涇渭分明 10
消 3420
消了 25
消亡 6
消保官 0
消保會 0
消保法 0
消化 839
消化不良 48
消化器 13
消化器官 14
消化液 17
消化液 17
消化率 12
消化管 5
消化系統 117
消化腺 0
消化道 202
消卻 1
消去 52
消受 18
消基會 0
消夏 0
消夜 24
消失 1718
消失下去 0
消失掉 7
消弭 134
消得 10
消息 4882
消息來源 108
消息靈通 9
消愁 9
消愁解悶 1
消掉 33
消散 75
消暑 82
消極 284
消極性 17
消毀 2
消毒 760
消毒劑 72
消毒水 16
消毒片 0
消毒藥 4
消毒藥水 15
消氣 9
消沈 12
消沉 29
消消氣 2
消滅 659
消火栓 0
消災 67
消炎 79
消炎劑 6
消炎藥 28
消瘦 69
消石灰 3
消磁 32
消磨 51
消耗 585
消耗品 60
消耗戰 4
消耗熱 0
消耗量 70
消聲匿跡 11
消腫 33
消腫劑 0
消腫藥 2
消著 0
消融 12
消費 13977
消費品 120
消費者 6337
消費資料 4
消費量 129
消費額 38
消退 89
消逝 131
消遙 48
消遣 107
消遣性 5
消釋 0
消長 167
消閑 0
消閒 16
消防 3126
消防員 12
消防大隊 12
消防栓 162
消防車 82
消防隊 171
消防隊員 40
消除 1215
消音 59
消音器 20
消魂 4
涉 1400
涉世 3
涉世不深 1
涉世未深 5
涉事 4
涉及 1711
涉外 254
涉嫌 590
涉嫌人 11
涉嫌者 13
涉案 185
涉案者 5
涉水 48
涉水而行 0
涉水而過 1
涉水過河 1
涉獵 89
涉覽 0
涉訟 27
涉足 71
涉足於 1
涉足花叢 0
涉險 5
涊 2
涋 4
涌 114
涌現 3
涍 4
涎 59
涎皮賴臉 0
涏 0
涐 7
涑 9
涒 9
涓 167
涓涓 17
涓涓細流 0
涓滴 9
涓滴歸公 0
涔 30
涔涔 5
涔涔然 0
涔淚 0
涕 41
涕泗 2
涕泗縱橫 3
涕泣 11
涕淚 10
涕零 3
涖 0
涗 3
涘 11
涙 0
涚 0
涛 0
涛 0
涪 21
涪陵 15
涫 18
涬 6
涭 0
涮 81
涮涮鍋 0
涮羊肉 14
涮鍋子 0
涯 179
涯際 0
涰 0
涱 0
涱 0
液 251
液 14786
液力 0
液力 0
液力傳動 0
液力傳動 0
液化 434
液化 434
液化器 0
液化器 0
液壓 135
液壓 135
液壓式 12
液壓式 12
液壓機 3
液壓機 3
液壓油 17
液壓油 17
液態 232
液態 232
液態氧 3
液態氧 3
液態空氣 2
液態空氣 2
液晶 1450
液晶 1450
液晶板 12
液晶板 12
液晶片 1
液晶片 1
液晶錶 0
液晶錶 0
液汁 3
液汁 3
液泡 11
液泡 11
液胞膜 1
液胞膜 1
液體 692
液體 692
液體動力 1
液體動力 1
液體動力學 0
液體動力學 0
液體燃料 9
液體燃料 9
液體狀態 6
液體狀態 6
涳 17
涴 2304
涵 3110
涵容能力 8
涵意 53
涵洞 8
涵管 23
涵義 104
涵育 18
涵蓄 2
涵蓋 0
涵蘊 10
涵養 189
涶 0
涷 26
涸 22
涸 22
涸乾 0
涸乾 0
涹 0
涺 31
涻 2
涼 123
涼 1243
涼了 45
涼了半截 4
涼亭 126
涼傘 3
涼到 2
涼台 5
涼山 20
涼州 39
涼席 1
涼快 33
涼意 29
涼拌 69
涼拌菜 2
涼棚 4
涼椅 13
涼氣 19
涼水 13
涼涼 47
涼涼快快 0
涼涼爽爽 1
涼爽 88
涼粉 3
涼粉條 0
涼糕 3
涼菜 0
涼蓆 2
涼血動物 0
涼血動物 0
涼鞋 81
涼風 55
涼風習習 2
涼風陣陣 0
涼颼颼 3
涼麵 24
涽 0
涾 0
涿 0
涿鹿 1
涿鹿縣 0
淀 39
淁 0
淁 0
淂 0
淃 0
淄 29
淅 33
淅瀝 11
淅颯 0
淆 47
淇 465
淈 18
淉 340
淊 17
淋 4770
淋到 100
淋巴 286
淋巴液 5
淋巴液 5
淋巴球 116
淋巴管 28
淋巴細胞 22
淋巴結 92
淋巴腺 108
淋得 7
淋得 10
淋沖 0
淋洗 19
淋浴 97
淋淋 11
淋溼 9
淋漓 75
淋漓盡致 106
淋濕 29
淋病 51
淋病菌 0
淋的 11
淋菌 18
淋過 4
淋雨 19
淋頭 0
淋頭 0
淌 62
淌淚 2
淌混水 0
淌血 13
淌血 13
淍 5
淎 0
淏 491
淐 8
淑 10354
淑女 202
淑女裝 0
淑女車 1
淑媛 102
淑德 22
淑慎 11
淑懿 0
淑範 2
淒 157
淒冷 9
淒切 1
淒切動人 0
淒厲 28
淒咽 0
淒惶 1
淒慘 36
淒楚 5
淒涼 91
淒淒 11
淒淒切切 1
淒淒慘慘 5
淒淒涼涼 0
淒清 13
淒滄 0
淒然 19
淒苦 9
淒迷 12
淒迷不振 0
淒風 2
淒風苦雨 9
淓 7
淔 29
淕 140
淖 9
淗 1
淘 160
淘井 0
淘氣 97
淘氣鬼 2
淘汰 546
淘汰賽 17
淘洗 4
淘淘 0
淘神 0
淘空 5
淘米 3
淘金 54
淘金漢 0
淘金熱 1
淙 37
淙淙 14
淙淙流水 1
淚 2331
淚下沾襟 0
淚人 5
淚人兒 3
淚人兒 3
淚如泉湧 1
淚如雨下 7
淚容滿面 0
淚水 370
淚水直流 0
淚汪汪 4
淚流滿面 29
淚涔涔 0
淚液 38
淚液 38
淚滴 30
淚珠 40
淚痕 114
淚眼 44
淚眼汪汪 1
淚管 21
淚腺 9
淚腺炎 2
淚花 2
淚雨 2
淛 5
淜 54
淝 54
淞 137
淟 5
淠 10
淡 1441
淡入 5
淡入淡出 7
淡出 82
淡出淡入 0
淡到 2
淡化 99
淡化處理 7
淡化處理 7
淡味 3
淡大 242
淡大學生 1
淡妝 20
淡季 98
淡得 3
淡忘 52
淡月 1
淡朱 0
淡水 3252
淡水人 17
淡水區 22
淡水工商 408
淡水工商管理專校 0
淡水河 359
淡水湖 1
淡水鎮 738
淡水魚 54
淡江 798
淡江大學 2336
淡泊 27
淡淡 243
淡淡無味 0
淡淡的 194
淡淡地 194
淡漠 8
淡然 41
淡然處之 1
淡的 37
淡硃 0
淡紅 56
淡紅色 11
淡紫 55
淡紫色 24
淡綠 13
淡綠色 21
淡而無味 1
淡色 37
淡茶 2
淡菜 6
淡薄 32
淡薄 32
淡藍 32
淡藍色 25
淡裝 1
淡褐 1
淡褐色 15
淡酒 34
淡雅 36
淡青 17
淡青色 7
淡飯 1
淡黃 49
淡黃色 55
淢 14
淣 4
淤 48
淤塞 14
淤沙 10
淤泥 27
淤淺 5
淤滯 0
淤積 73
淤血 12
淤血 12
淥 47
淦 71
淧 0
淨 2754
淨供 1
淨值 642
淨利 403
淨化 585
淨化劑 6
淨化器 2
淨化系統 7
淨土 575
淨土宗 43
淨室 10
淨手 3
淨是 11
淨本 0
淨水 159
淨水器 67
淨水廠 35
淨瓶 3
淨盡 8
淨賺 11
淨賺額 0
淨身 8
淨重 93
淨重量 0
淩 614
淪 1183
淪亡 12
淪亡史 0
淪入 17
淪入敵手 0
淪入風塵 0
淪喪 22
淪於 12
淪沒 0
淪滅 0
淪為 367
淪肌浹髓 0
淪落 77
淪落街頭 0
淪陷 61
淪陷區 7
淫 352
淫亂 40
淫佚 0
淫威 6
淫娃 3
淫婦 29
淫媒 0
淫巧 3
淫慾 5
淫戒 2
淫戲 0
淫暴 0
淫書 4
淫業 3
淫樂 5
淫水 34
淫猥 1
淫穢 20
淫笑 4
淫蕩 47
淫行 5
淫行 5
淫褻 1
淫賊 0
淫賤 3
淫辭 2
淫靡 3
淫風 4
淬 79
淬勉 0
淬火 20
淬礪 7
淭 8
淮 309
淮北 1
淮南 364
淮安 10
淮水 6
淮河 10
淮海 10
淮陰 21
淯 15
淰 126
深 5668
深不可測 42
深井 20
深井泵 0
深交 19
深仇 5
深仇大恨 11
深信 406
深信不疑 32
深入 2153
深入人心 16
深入基層 25
深入報導 78
深入探討 174
深入敵後 3
深入淺出 172
深凹 1
深切 248
深切關懷 3
深切體認 19
深到 7
深刻 1130
深刻印象 42
深刻影響 2
深化 79
深厚 310
深受 534
深受感動 30
深呼吸 79
深圳 457
深圳 457
深坑 252
深坑鄉 149
深埋 24
深埋在 8
深夜 419
深奧 109
深奧難懂 4
深妙 2
深宅大院 2
深室 0
深宮 10
深居 12
深居簡出 6
深層 0
深山 210
深山中 11
深巨 0
深巷 2
深度 3591
深度計 0
深得 56
深得 515
深得我心 1
深念 0
深怕 145
深思 272
深思熟慮 42
深恐 21
深恨 1
深恩 3
深悉 14
深情 417
深情厚誼 1
深情厚誼 1
深情款款 19
深惡 0
深惡痛絕 14
深意 38
深愛 131
深愛著 37
深感 309
深感不便 1
深感不安 3
深感憂慮 2
深慮 2
深憂 1
深摯 8
深於 3
深明 8
深明大義 1
深暗 10
深更 3
深更 3
深更半夜 5
深更半夜 5
深有 29
深有同感 30
深根固柢 1
深棕色 11
深水 110
深水港 9
深水炸彈 6
深沈 82
深沈不露 0
深沉 75
深沉不露 0
深海 183
深海魚 9
深深 1182
深深地 152
深深地 152
深深感到 9
深深感動 28
深淵 118
深淺 101
深淺不同 13
深湛 7
深溝 25
深溝高壘 3
深潭 23
深濃 12
深灰 12
深灰色 17
深為 57
深獲 168
深獲好評 35
深痕 1
深痛 1
深的 515
深省 6
深知 179
深秋 60
深究 83
深紅 23
深紅色 21
深紫色 9
深綠 28
深綠色 82
深翻 3
深耕 171
深耕易耨 0
深色 105
深色系列 6
深藍 236
深藍色 102
深藏 80
深藏不露 23
深藏若虛 0
深處 419
深表 61
深表同情 3
深褐 2
深褐色 24
深言 0
深詆 0
深談 19
深諳 26
深諳此道 0
深謀遠慮 15
深識 3
深谷 32
深趣 0
深迷 2
深造 263
深遠 318
深邃 108
深重 26
深鉅 2
深長 19
深閨 9
深院 3
深黃 3
深黃色 3
深黑色 9
深點 0
淲 2
淲 2
淳 665
淳化 2
淳厚 10
淳樸 33
淳風 13
淴 17
淵 1908
淵博 51
淵壑 0
淵泉 12
淵源 220
淵藪 13
淵谷 0
淶 17
混 13
混 1396
混 1396
混 1396
混上 1
混上來 0
混上去 0
混下 3
混下來 0
混下去 5
混不下去 11
混世魔王 21
混亂 630
混了 64
混交林 0
混作 2
混來 1
混來混去 0
混充 6
混入 59
混凝 66
混凝劑 7
混凝土 1076
混出 11
混出來 0
混出去 0
混到 26
混口飯 1
混口飯吃 2
混合 1882
混合列車 0
混合器 8
混合法 15
混合液 19
混合液 19
混合物 117
混合結構 2
混同 14
混名 0
混和 47
混在 59
混子 0
混子 0
混帳 9
混得 14
混得 48
混成 58
混戰 69
混日子 4
混棉 4
混水 5
混水摸魚 18
混水摸魚 18
混水撈魚 0
混水撈魚 0
混沌 191
混沌初開 1
混沌初開 1
混淆 243
混淆 243
混淆不清 34
混淆不清 34
混淆是非 0
混淆視聽 24
混混 109
混混 109
混混 109
混混 109
混混 109
混混 109
混混 109
混混 109
混混 109
混混沌沌 0
混濁 120
混為 2
混為一談 34
混煉 9
混球 5
混用 0
混的 48
混紡 181
混著 20
混蛋 56
混血 64
混血 64
混血兒 28
混血兒 28
混血種 2
混血種 2
混起 0
混起來 0
混跡 10
混身解數 4
混進 8
混進來 3
混進去 2
混過 15
混過來 0
混過去 7
混鐵爐 0
混雙 0
混雜 69
混雜著 13
混雜起來 0
混音器 17
混響 0
混頻器 3
混養 40
淸 0
淹 168
淹 168
淹死 26
淹水 303
淹沒 231
淹溺 3
淺 1324
淺到 2
淺嘗輒止 0
淺得 2
淺斟低唱 0
淺明 4
淺易 10
淺水 86
淺水池 0
淺水灘 1
淺水灣 39
淺海 58
淺海漁業 0
淺海養殖 16
淺淡 8
淺淺 71
淺灘 29
淺笑 20
淺紅 12
淺紫 6
淺紫色 14
淺綠 19
淺綠色 33
淺而易見 1
淺耕 0
淺聞 0
淺色 70
淺薄 46
淺薄 46
淺藍 45
淺藍色 29
淺見 127
淺說 38
淺談 574
淺近 3
淺陋 10
淺顯 65
淺顯易懂 59
淺黃 37
添 1554
添丁 56
添上 23
添上來 0
添上去 0
添了 34
添些 3
添來添去 0
添個 7
添入 3
添到 0
添加 934
添加劑 206
添加物 418
添在 2
添增 24
添成 6
添房 0
添有 1
添枝加葉 0
添水 9
添油加醋 2
添滿 5
添置 18
添菜 0
添著 2
添補 4
添設 8
添購 239
添酒 2
添附 2
添飯 0
添點 4
淼 227
淼淼 1
淽 4
淾 0
淾 0
淿 0
淿 0
渀 30
渁 0
渂 0
渂 0
渃 1
渄 0
清 10929
清一色 53
清了 21
清亮 27
清人 16
清代 364
清修 21
清倉 39
清倉大拍賣 3
清償 227
清光 31
清光祖 0
清冊 544
清冷 19
清冽 4
清出 36
清初 65
清到 2
清剿 1
清勁 1
清化 15
清史 34
清史稿 6
清君側 2
清和 51
清唱 38
清單 9492
清場 16
清場 16
清塘 0
清夜 4
清大 806
清天 1
清好 0
清官 13
清官難斷家務事 4
清室 3
清宮 68
清寒 312
清寒獎學金 14
清將 9
清帳 1
清平 31
清平世界 1
清幽 74
清廉 131
清廷 139
清律 1
清得 4
清得 98
清心 170
清心寡慾 7
清心寡欲 4
清掃 117
清掃伕 0
清掃馬路 0
清撤 0
清教 13
清教徒 11
清新 478
清新自然 17
清早 29
清明 176
清明節 59
清時 12
清晨 686
清晰 675
清晰度 33
清朗 36
清朝 306
清朝人 2
清查 364
清查人數 0
清棉機 0
清楚 4190
清楚起來 3
清樂 2
清正 11
清水 1595
清水牆 0
清江 251
清江國小 3
清河 48
清油 1
清泉 190
清泉崗 21
清泠 0
清洌 1
清洗 864
清流 146
清涼 601
清涼劑 5
清涼有勁 9
清涼油 1
清淚 8
清淡 101
清淨 589
清淨器 3
清淨無為 3
清淺 5
清清 55
清清楚楚 101
清清潔潔 1
清清爽爽 4
清清白白 5
清清靜靜 0
清湯 15
清湯掛麵 4
清溪 71
清滌 0
清潔 2340
清潔劑 484
清潔工 63
清潔溜溜 0
清潔隊 144
清澄 29
清澈 206
清澈見底 15
清濁 4
清燉 9
清燉雞 0
清燉魚 0
清爽 162
清理 891
清甜 12
清瘦 4
清白 245
清的 98
清盛 4
清盤 2
清真寺 62
清真教 0
清真館 3
清福 19
清秀 120
清秀佳人 37
清稿 3
清算 277
清算帳目 0
清算鬥爭 1
清純 87
清純如一 0
清純玉女 2
清聲 3
清脆 67
清色 0
清芬 20
清苦 19
清茶 6
清華 1015
清華大學 3224
清蒸 61
清蒸雞 0
清蒸魚 1
清虛 8
清規 7
清規戒律 3
清談 73
清談家 0
清議 4
清貧 13
清貧人家 0
清賬 1
清走 2
清車 3
清輝 58
清過 5
清過來 0
清過去 0
清道 21
清道伕 0
清道夫 45
清遠 21
清酒 18
清醒 273
清醒過來 25
清閑 1
清閒 24
清除 1877
清雅 69
清靜 142
清靜風雅 0
清音 32
清風 151
清風徐來 4
清香 166
清高 48
清高宗 3
清麗 40
清點 97
清點人數 6
清黨 3
清黨行動 0
渆 0
渇 0
渇 0
渇 0
済 0
済 0
渉 0
渊 0
渋 0
渌 0
渒 0
渗 0
渘 0
渙 38
渙散 21
渙然冰釋 0
渚 73
減 4612
減了 19
減份 0
減低 642
減個 2
減值 3
減價 132
減價出售 1
減價時間 5
減免 585
減免額 1
減刑 18
減半 89
減去 92
減壓 170
減壓塔 0
減壓水槽 0
減壓渣油 0
減壓渣油 0
減壓蒸餾 2
減好 0
減對 3
減小 75
減少 6700
減弱 185
減弱到 0
減得 0
減得 0
減息 96
減慢 62
減掉 43
減損 59
減收 21
減數 7
減數分裂 45
減法 38
減消 3
減班 49
減產 312
減省 6
減租 6
減租減息 1
減稅 179
減稅法 0
減筆字 0
減算 3
減緩 398
減縮 29
減縮到 0
減肥 1083
減胖 3
減至 138
減色 2
減薪 110
減號 7
減課 5
減輕 1082
減輕法 0
減輕處分 0
減輕處分 0
減退 121
減速 289
減速劑 0
減速器 3
減速慢行 12
減速運動 0
減重 1
減金 0
減除 174
減震 14
減食 1
渜 1
渝 345
渞 0
渟 21
渠 270
渠水 1
渠道 92
渠魁 0
渡 1420
渡上 1
渡了 4
渡假 1812
渡到 5
渡口 41
渡江 22
渡河 27
渡海 63
渡船 53
渡船口 4
渡船頭 32
渡蜜月 5
渡輪 76
渡過 423
渡過難關 45
渡頭 11
渢 3
渣 63
渣 638
渣子 1
渣子 1
渣油 0
渣油 0
渣滓 13
渤 30
渤海 63
渤海海峽 1
渤海灣 6
渥 174
渥丹 0
渥太華 47
渥恩 1
渥惠 0
渥飾 0
渦 251
渦旋式 0
渦流 60
渦漩 26
渦蟲 0
渦輪 157
渦輪噴氣發動機 0
渦輪引擎 11
渦輪機 25
渦輪發動機 4
渦輪軸發動機 0
渧 4
渨 9
温 0
渪 0
渫 24
測 6705
測位器 0
測候 46
測偵所 0
測出 122
測力 6
測力計 3
測反 0
測壓管 1
測字 21
測定 1739
測定機 15
測定法 108
測平器 0
測度 22
測度 22
測度器 0
測度法 0
測度法 0
測度計 0
測度論 3
測微計 3
測時 22
測樹學 0
測海 5
測深 10
測溫 22
測溫器 2
測知 43
測站 236
測繪 111
測航 0
測試 9318
測試儀 42
測試法 25
測試點 3
測謊 0
測謊器 1
測距 65
測距法 0
測速 0
測速器 24
測量 2199
測量儀 27
測量出 10
測量劑 0
測量員 9
測量器 17
測量學 45
測量桿 0
測量法 15
測量術 1
測量計 5
測錄到 0
測震 1
測震儀 0
測音 2
測音器 0
測驗 4372
測驗卷 26
測驗紙 0
測高 8
測高度 1
測高法 0
測高計 1
渭 180
渭水 71
渭河 3
渭河流域 0
渮 5
港 8586
港九 11
港人 213
港內 56
港劇 31
港務 104
港務局 348
港務監督 0
港務費 0
港區 610
港口 851
港口溪 5
港台 132
港台地區 1
港埠 322
港墘 74
港墘 74
港外 33
港子 28
港岸 4
港市 98
港幣 223
港式 191
港式料理 1
港星 41
港海 11
港海地區 0
港澳 2151
港澳地區 61
港灣 353
港片 43
港製 2
港製品 0
港警 22
港警所 31
港都 405
港都夜雨 4
渰 32
渱 5
渲 38
渲染 91
渲染法 0
渳 2
渴 255
渴了 13
渴到 0
渴得 2
渴得 2
渴念 2
渴想 8
渴愛 4
渴慕 60
渴望 656
渴死 7
渴求 87
渴渴 0
渴盼 5
渴盼著 0
渴睡 1
渴著 0
渵 0
渶 0
渷 0
渷 0
游 8959
游上 4
游上來 0
游上去 0
游下 0
游下來 0
游下去 1
游了 11
游人 8
游來 9
游來游去 22
游俠 35
游刃有餘 12
游動 37
游去 11
游回 7
游回來 0
游回去 0
游士 8
游子 19
游客 3
游手 1
游手好閑 3
游手好閒 9
游手恣睢 0
游擊 38
游擊戰 25
游擊手 21
游擊隊 63
游樂 3
游標 599
游標位置 4
游標卡尺 7
游歷 2
游民 10
游水 46
游泳 1160
游泳池 1056
游泳衣 8
游泳褲 2
游泳賽 9
游牧 13
游牧民族 24
游玩 2
游盪 11
游目騁懷 1
游移 46
游絲 14
游船 1
游蕩 9
游藝 4
游藝會 0
游記 10
游說 26
游談 0
游談無根 0
游資 15
游資氾濫 1
游走 64
游蹤 11
游進 17
游進來 0
游進去 0
游過 17
游過來 4
游過去 7
游錯 0
游開 3
游離 346
游離份子 0
游騎兵 4
游騎兵 4
游魂 6
渹 3
渺 62
渺不足道 0
渺小 82
渺渺 14
渺渺茫茫 1
渺無 8
渺無人煙 4
渺無希望 0
渺無蹤跡 0
渺然 0
渺茫 60
渺茫無望 0
渻 2
渼 9
渽 2
渾 216
渾 216
渾人 0
渾似 1
渾厚 48
渾名 2
渾噩 6
渾圓 32
渾天儀 4
渾家 1
渾水 5
渾水摸魚 2
渾沌 73
渾渾 7
渾渾噩噩 21
渾渾灑灑 0
渾濁 17
渾然 74
渾然一體 8
渾然天成 30
渾球 2
渾蛋 5
渾身 113
渾身上下 2
渾身是汗 0
渾身是膽 7
渾身發抖 4
渾身解數 11
渾金璞玉 0
渾鐵 2
渾體 1
渿 6
湀 5
湁 7
湂 0
湃 33
湄 174
湄公河 10
湄南河 12
湄洲 0
湅 6
湆 8
湇 4
湈 0
湉 4
湊 206
湊上 7
湊上來 1
湊上去 0
湊不出來 0
湊出 11
湊到 29
湊合 21
湊合著 2
湊在一起 12
湊在一起 12
湊巧 53
湊成 12
湊手 0
湊攏 0
湊數 3
湊熱鬧 84
湊興 0
湊足 5
湊近 18
湊集 1
湊齊 6
湋 6
湌 0
湍 172
湍急 38
湍流 9
湎 9
湏 0
湐 0
湐 0
湑 2
湒 4
湓 4
湔 137
湔洗 0
湔滌 0
湔雪 0
湔雪前恥 1
湕 2
湖 5300
湖光 17
湖光山色 67
湖內 353
湖冰水 0
湖北 175
湖北人 2
湖北省 149
湖區 32
湖南 213
湖南人 3
湖南省 67
湖口 1026
湖名 5
湖外 12
湖島 1
湖州 13
湖州府 0
湖底 48
湖廣 11
湖心亭 1
湖東 61
湖水 140
湖沼 26
湖泊 415
湖濱 72
湖濱公路 0
湖田 14
湖畔 182
湖色 3
湖裡 22
湖西 243
湖邊 78
湖面 80
湗 0
湘 1295
湘劇 4
湘南 10
湘桂黔 0
湘水 3
湘江 10
湘潭 15
湘繡 2
湘軍 4
湙 0
湚 5
湛 295
湛江 30
湛深 2
湛然 11
湛藍 93
湜 28
湝 5
湞 12
湟 15
湠 2
湡 1
湡 1
湢 10
湣 8
湤 5
湥 1
湦 3
湧 511
湧上 43
湧上來 8
湧上去 1
湧上心來 0
湧來 16
湧入 147
湧出 73
湧出來 4
湧出去 0
湧向 20
湧泉 92
湧現 119
湧現出 2
湧起 53
湧進 80
湧進來 3
湧進去 0
湨 4
湩 9
湪 0
湫 6
湬 0
湬 0
湭 0
湮 212
湮 212
湮沒 26
湮沒 26
湮滅 25
湮滅 25
湮滅證據 5
湮滅證據 5
湮遠 1
湮遠 1
湯 27
湯 2746
湯化龍 1
湯匙 248
湯圓 119
湯團 0
湯壽潛 0
湯姆 492
湯姆歷險記 4
湯尼 53
湯底 1
湯恩比 13
湯武 3
湯武革命 0
湯水 15
湯汁 15
湯湯 6
湯糰 1
湯糰兒 0
湯糰兒 0
湯藥 10
湯鍋 6
湯頭 42
湯顯祖 2
湯飯 2
湯餃 4
湯麵 30
湰 0
湱 4
湲 13
湳 216
湴 57
湵 0
湶 0
湷 3
湸 6
湹 1
湻 0
湾 0
満 0
溆 0
溈 10
溉 48
溍 4
溎 11
溏 21
源 9370
源始 4
源委 6
源於 278
源本 6
源泉 55
源泉萬斛 0
源流 162
源源 85
源源不斷 16
源源不斷 16
源源不絕 73
源源本本 1
源源而來 2
源由 59
源自 370
源遠流長 33
源頭 212
溑 0
溒 7
溓 0
溔 1
溕 0
準 5549
準備 6953
準備充分 2
準備好的 33
準備活動 16
準備金 163
準兒 0
準兒 0
準則 1476
準噶爾 5
準噶爾盆地 5
準定 1
準平原 0
準度 14
準據 31
準新娘 4
準新郎 1
準星 12
準是 8
準時 323
準時出席 15
準時到達 10
準會 44
準決賽 119
準的 95
準得 95
準確 571
準確度 251
準確性 95
準確無誤 6
準確率 1
準線 3
準繩 31
準話 0
準頭 21
溗 1
溘 135
溘然 2
溘然而逝 0
溘謝 0
溘逝 0
溙 3
溚 0
溚 0
溛 3
溜 1
溜 393
溜 393
溜上 0
溜上來 1
溜上去 0
溜下 1
溜下來 4
溜下去 1
溜之大吉 9
溜了 25
溜井 0
溜來 3
溜來溜去 0
溜個 0
溜光 3
溜冰 281
溜冰場 66
溜冰場 66
溜冰鞋 122
溜出 6
溜出來 3
溜出去 5
溜到 11
溜去 10
溜口 0
溜回 13
溜回來 1
溜回去 0
溜在 0
溜好 0
溜子 0
溜完 0
溜得 2
溜得 2
溜掉 10
溜溜 50
溜滑 4
溜滑梯 29
溜走 42
溜進 17
溜進來 1
溜進去 0
溜遍 0
溜遍 0
溜過 10
溜過來 0
溜過去 0
溜達 11
溜開 2
溝 1372
溝坑 2
溝壑 4
溝床 3
溝施 0
溝槽 54
溝池 0
溝沿 1
溝洫 3
溝渠 86
溝澗 0
溝谷 1
溝通 4594
溝道 5
溞 4
溟 40
溠 5
溡 1
溢 7051
溢位 0
溢價 201
溢出 81
溢利 3
溢收 1
溢於言表 20
溢洪道 13
溢流 34
溢流堰 1
溢滿 8
溢目 0
溢美之詞 3
溣 3
溤 2
溥 308
溥儀 7
溦 25
溧 20
溧陽 0
溨 0
溩 0
溪 0
溪 4342
溪口 402
溪口國小 10
溪尾 50
溪州 255
溪水 262
溪河 65
溪流 469
溪湖 509
溪澗 25
溪谷 181
溪釣 69
溪頭 466
溫 7343
溫中 5
溫侯 4
溫厚 9
溫和 457
溫和性 4
溫和派 9
溫哥華 538
溫善 5
溫娣 5
溫婉 32
溫存 48
溫室 797
溫室效應 0
溫室植物 0
溫室花朵 1
溫州 120
溫州市 11
溫差 116
溫差值 0
溫布頓 0
溫帶 112
溫帶林 4
溫帶植物 3
溫帶氣候 4
溫床 48
溫度 2905
溫度場 6
溫度場 6
溫度梯度 9
溫度計 140
溫徹斯特 2
溫情 147
溫情滿人間 4
溫故 5
溫故知新 19
溫文 41
溫文儒雅 33
溫暖 1137
溫暖人間 0
溫書 5
溫書假 5
溫柔 1145
溫柔敦厚 15
溫柔鄉 10
溫標 2
溫水 166
溫水浴 3
溫泉 1608
溫泉浴 18
溫泉鄉 41
溫溫和和 1
溫溼 9
溫潤 32
溫炕 0
溫煦 12
溫熱 119
溫熱度 1
溫純 6
溫美 3
溫習 49
溫習功課 1
溫舊知新 0
溫良 13
溫莎 97
溫莎公爵 2
溫莎堡 16
溫酒 2
溫開水 10
溫雅 8
溫順 19
溫順性 0
溫顏 0
溫食 5
溫飽 34
溫馨 1679
溫馴 52
溫體 2
溫體肉 4
溫體豬肉 2
溫鰲妝燕 1
溬 0
溭 0
溮 6
溯 41
溯 638
溯江而上 0
溯流 0
溯流而上 1
溯源 70
溰 4
溱 14
溲 4
溳 6
溴 210
溴 210
溴化物 3
溴化鉀 4
溴化銀 2
溴水 2
溴甲烷 7
溴酸 6
溴酸鹽 0
溵 0
溶 901
溶入 90
溶劑 680
溶化 102
溶合 13
溶媒 16
溶岩 11
溶度 1
溶於 294
溶氧 46
溶氧量 40
溶洞 3
溶液 784
溶液 784
溶液聚合 3
溶液聚合 3
溶溶 5
溶膠 33
溶菌斑 0
溶蝕 15
溶血 85
溶解 458
溶解度 96
溶解性 47
溶解熱 3
溶解物 3
溶質 17
溶體 1
溶點 3
溷 3
溷濁 0
溸 0
溹 1
溺 189
溺 189
溺床 0
溺愛 38
溺斃 42
溺死 10
溺水 60
溺職 0
溺谷 1
溻 0
溼 1227
溼地 611
溼度 326
溼度器 0
溼度計 31
溼氣 1
溼淋淋的 12
溼淋淋地 12
溼漉漉的 11
溼潤 154
溼熱 42
溼疹 99
溼透 45
溽 4
溽暑 11
溾 1
溿 7
滀 1
滁 8
滂 665
滂沛 0
滂沱 12
滂沱大雨 8
滂渤 0
滂湃 0
滂澤 0
滂霈 0
滃 2
滄 596
滄州 14
滄桑 125
滄浪 10
滄浪亭 1
滄浪詩話 1
滄海 122
滄海一粟 2
滄海桑田 27
滄海遺珠 3
滄溟 2
滄茫 1
滅 1411
滅亡 171
滅口 49
滅屍 2
滅掉 7
滅族 10
滅火 446
滅火劑 18
滅火器 167
滅熄 0
滅種 20
滅絕 331
滅絕人性 2
滅菌 138
滅蟲劑 0
滅諦 1
滅跡 18
滅門 7
滅門之禍 2
滅門血案 5
滅音 2
滅音器 10
滅頂 21
滅頂之災 0
滆 4
滇 304
滇池 8
滇緬 8
滇緬公路 0
滇茶 0
滇西縱谷 2
滇越鐵路 0
滈 2
滉 5
滊 10
滋 943
滋事 25
滋事份子 1
滋事份子 1
滋味 494
滋擾 6
滋潤 225
滋生 102
滋生事端 3
滋育 1
滋芽 0
滋蔓 6
滋補 64
滋補藥品 0
滋補食品 1
滋長 62
滋養 248
滋養品 1
滌 384
滌綸 21
滌除 2
滍 2
滎 40
滎 40
滎陽 12
滏 3
滐 8
滑 98
滑 982
滑上 2
滑上來 0
滑上去 0
滑下 27
滑下來 5
滑下去 1
滑了 18
滑了一跤 2
滑來滑去 2
滑倒 18
滑入 20
滑冰 36
滑冰場 0
滑冰場 0
滑出 24
滑到 13
滑力 0
滑動 203
滑動摩擦 2
滑動軸承 4
滑去 2
滑坡 6
滑壘 3
滑得 2
滑得 41
滑掉 1
滑板 250
滑桿 11
滑梯 24
滑水 78
滑水場 0
滑水場 0
滑水板 5
滑溜 31
滑潤 12
滑潤劑 0
滑球 1
滑環 2
滑的 42
滑石 21
滑石粉 15
滑移 40
滑稽 73
滑稽 73
滑粉 0
滑縣 0
滑翔 35
滑翔機 13
滑翔翼 60
滑脫 8
滑膩 13
滑落 258
滑著 2
滑著走 0
滑行 119
滑走 3
滑起 0
滑起來 1
滑跤 0
滑車 27
滑輪 125
滑輪組 2
滑進 13
滑進來 0
滑進去 0
滑過 38
滑過來 0
滑過去 0
滑道 22
滑鐵盧 27
滑開 6
滑閥 0
滑降 33
滑雪 267
滑雪場 58
滑雪場 58
滑雪板 11
滑音 5
滑頭 2
滑頭滑腦 0
滑鼠 3149
滑鼠墊 500
滒 1
滓 198
滔 213
滔天 13
滔天大禍 2
滔天大罪 5
滔滔 51
滔滔不絕 42
滔滔地 0
滔滔地 0
滕 166
滕文公 2
滖 4
滗 0
滘 1
滙 0
滚 0
滛 0
滜 4
滝 0
滝 0
滣 0
滦 0
滨 0
滨 0
滩 0
滫 40
滬 776
滬市 0
滬江 29
滭 3
滮 201
滯 553
滯延 6
滯悶 0
滯憤 0
滯泥 0
滯留 181
滯留不歸 0
滯留到 0
滯留鋒 3
滯礙 6
滯礙難行 5
滯積 0
滯納金 1
滯貨 1
滯銷 38
滯銷貨 2
滰 0
滰 0
滱 91
滲 365
滲入 135
滲出 164
滲出來 3
滲出物 16
滲出量 0
滲到 2
滲得 0
滲水 58
滲流 9
滲漏 87
滲漏到 1
滲碳 17
滲穴 2
滲透 499
滲透到 25
滲透壓 37
滲透性 50
滲進 4
滲進來 1
滲進去 0
滲過 3
滲過來 0
滲過去 0
滳 0
滴 839
滴上 9
滴下 29
滴下來 2
滴下去 0
滴了 9
滴出 5
滴出來 1
滴向 0
滴哩答拉 0
滴定 143
滴定法 24
滴定管 5
滴得 0
滴水 57
滴水不漏 19
滴水成冰 0
滴水穿石 9
滴溜 6
滴溜溜 4
滴滴 72
滴滴涕 11
滴滴答答 9
滴瀝 1
滴狀 4
滴答 24
滴答作響 0
滴答聲 7
滴答響 0
滴管 17
滴落 50
滴蟲 18
滴血 43
滴雨 5
滴點 0
滵 44
滶 79
滷 252
滷味 41
滷水 26
滷汁 3
滷湖 0
滷肉 23
滷肉飯 4
滷菜 1
滷蛋 87
滷製 1
滷雞 5
滷雞肉 1
滷麵 1
滸 21
滹 222
滹沱河 0
滺 0
滻 20
滼 68
滽 26
滾 2287
滾上 5
滾上來 0
滾上去 0
滾下 25
滾下來 1
滾下去 3
滾了 13
滾來 6
滾來滾去 6
滾倒 3
滾出 16
滾出來 3
滾出去 2
滾利 2
滾到 17
滾動 124
滾動摩擦 1
滾動軸承 8
滾回 12
滾回來 1
滾回去 7
滾圓 9
滾在 5
滾地 6
滾地球 10
滾子 23
滾得 15
滾成 2
滾成一團 0
滾打 0
滾柱軸承 11
滾水 25
滾湯 2
滾滾 246
滾滾而來 3
滾熱 9
滾燙 38
滾珠 35
滾珠承軸 0
滾珠軸承 25
滾球 34
滾瓜爛熟 10
滾的 15
滾筒 206
滾翻 13
滾落 16
滾著 4
滾蛋 7
滾走 1
滾起 0
滾起來 0
滾輪 126
滾轉 7
滾道 0
滾邊 5
滾開 11
滾雪球 25
滿 10250
滿不 50
滿不 50
滿不在乎 6
滿不是 0
滿了 336
滿人 26
滿以為 3
滿佈 74
滿兵 0
滿出 1
滿出來 1
滿分 104
滿到 3
滿口 44
滿口答應 4
滿口胡扯 0
滿口胡說 0
滿口胡說八道 0
滿口髒話 1
滿嘴都是 0
滿園 18
滿園春色 1
滿地 135
滿坑 1
滿坑滿谷 11
滿城 22
滿城風雨 8
滿堂 48
滿堂紅 9
滿場 18
滿場 18
滿壘 9
滿天 208
滿天下 13
滿天星 87
滿天星斗 16
滿天飛 51
滿孝 0
滿屋子 8
滿山 82
滿山滿谷 5
滿州 100
滿州人 4
滿州國 9
滿州話 0
滿座 14
滿得 86
滿心 99
滿心歡喜 33
滿意 2956
滿懷 141
滿懷信心 9
滿懷希望 17
滿手 14
滿手是汗 0
滿手是血 0
滿招損 6
滿文 7
滿於 16
滿族 50
滿月 90
滿有 77
滿朝 3
滿期 47
滿桌 11
滿檔 17
滿歲 3
滿江紅 23
滿洲 25
滿洲國 9
滿清 63
滿清帝國 0
滿清時代 3
滿清末年 6
滿滿 186
滿滿的 125
滿滿地 125
滿漢 28
滿漢全席 43
滿潮 17
滿百 26
滿的 87
滿目 19
滿目瘡痍 27
滿眼 18
滿肚子 9
滿肚子火 1
滿腔 38
滿腔熱情 4
滿腔熱血 8
滿腦子 27
滿腹 57
滿腹心事 0
滿腹心酸 1
滿腹文章 2
滿腹牢騷 6
滿腹疑雲 0
滿腹經綸 7
滿臉 167
滿臉橫肉 0
滿臉風霜 1
滿著 13
滿街 74
滿街跑 15
滿起來 0
滿足 3009
滿足感 31
滿身 101
滿身是汗 1
滿身是血 2
滿車 2
滿載 177
滿載而歸 58
滿門 9
滿面 65
滿面怒容 0
滿面春風 2
滿面殺氣 0
滿面笑容 2
滿頭大汗 22
滿額 16
漀 51
漁 3171
漁人 126
漁人之利 2
漁人得利 1
漁具 195
漁利 1
漁區 19
漁友 26
漁叟 0
漁場 25
漁場 251
漁塭 27
漁夫 240
漁家 32
漁市 21
漁戶 35
漁撈 103
漁撈業 19
漁會 1195
漁期 18
漁村 256
漁業 9294
漁權 19
漁歌 6
漁民 1016
漁汛 4
漁港 2602
漁火 16
漁父 14
漁獲 237
漁獲量 64
漁獵 192
漁產 232
漁產量 40
漁網 85
漁翁 60
漁翁之利 0
漁翁得利 8
漁舟 11
漁船 1232
漁輪 5
漁郎 2
漁霸 0
漂 353
漂 353
漂 353
漂了 1
漂亮 1809
漂亮起來 0
漂來 11
漂來漂去 1
漂兒 28
漂兒 28
漂到 3
漂動 0
漂去 1
漂向 6
漂在 2
漂好 0
漂往 0
漂擺 0
漂泊 89
漂洋 2
漂洋過海 5
漂洗 4
漂流 154
漂流不定 0
漂流到 9
漂浮 126
漂海 0
漂漂 201
漂漂亮亮 23
漂白 386
漂白劑 62
漂白粉 53
漂移 35
漂著 4
漂蕩 5
漂行 0
漂走 1
漂過 6
漂過來 1
漂過去 0
漂零 1
漃 16
漄 0
漅 18
漆 208
漆 2085
漆上 17
漆上來 0
漆上去 0
漆了 1
漆作 2
漆出 1
漆到 1
漆刷 5
漆包線 41
漆匠 1
漆器 63
漆園吏 2
漆在 3
漆好 0
漆布 0
漆得 0
漆成 14
漆樹 16
漆汁 0
漆畫 4
漆著 8
漆過 0
漆過來 0
漆過去 0
漆門 1
漆黑 120
漆黑 120
漆黑一團 0
漆黑一團 0
漇 19
漈 215
漉 28
漊 6
漊 6
漋 0
漌 0
漍 92
漎 32
漏 964
漏下 10
漏了 55
漏借 0
漏光 3
漏出 28
漏出來 2
漏出去 0
漏刊 0
漏刻 4
漏勺 0
漏卮 0
漏在 0
漏報 47
漏壺 0
漏夜 39
漏失 43
漏失掉 3
漏子 3
漏孔 0
漏字 20
漏屋 0
漏底 0
漏掉 0
漏排 0
漏接 10
漏放 0
漏斗 51
漏杓 1
漏氣 111
漏水 154
漏油 46
漏洞 191
漏盡 4
漏盡通 1
漏看 5
漏稅 69
漏空 0
漏網 27
漏網之魚 7
漏著 0
漏行 0
漏誤 10
漏財之源 0
漏進 3
漏進來 0
漏進去 0
漏針 0
漏雨 1
漏電 76
漏頁 0
漏風 7
漏風聲 0
漐 0
漑 0
漒 4
漓 118
演 4326
演不下 1
演個 3
演出 4222
演出人 2
演出來 4
演出者 51
演到 28
演劇 54
演化 401
演化出 5
演化論 20
演員 3265
演員表 4
演唱 831
演唱會 896
演唱者 37
演奏 1258
演奏室 0
演奏家 165
演奏師 2
演奏廳 88
演奏曲 24
演奏會 201
演奏法 1
演奏者 79
演奏臺 0
演對 5
演對戲路 0
演得 224
演慣 0
演成 10
演戲 167
演技 193
演技派 8
演歌 44
演活 45
演演 3
演演看 0
演的 223
演示 81
演算 101
演算法 371
演算題 0
演練 397
演練法 0
演繹 72
演繹出 6
演繹法 16
演義 101
演習 497
演著 28
演藝 1194
演藝事業 58
演藝圈 91
演藝場所 1
演藝場所 1
演藝界 45
演說 454
演說家 10
演說者 9
演說辭 1
演講 4909
演講人 533
演講台 2
演講會 207
演講比賽 97
演講者 142
演講詞 8
演講辭 0
演證 0
演變 666
演變到 26
演變成 116
演起 5
演起來 0
演進 572
演進到 13
演過 44
漕 256
漕運 3
漖 0
漗 0
漘 8
漙 14
漚 22
漚 22
漛 0
漜 101
漝 0
漞 94
漟 166
漠 133
漠不相關 0
漠不關心 29
漠北 3
漠南 3
漠地 1
漠然 28
漠然處之 0
漠視 170
漡 65
漢 6014
漢中 116
漢中盆地 0
漢人 400
漢代 100
漢代人 0
漢儒 3
漢兵 4
漢兵漢將 0
漢化 30
漢口 186
漢口市 1
漢唐 155
漢唐兩代 0
漢土 4
漢城 355
漢城人 0
漢堡 365
漢奸 21
漢奸走狗 0
漢姓 4
漢子 135
漢字 286
漢學 312
漢學家 22
漢學者 0
漢室 10
漢宮秋 11
漢家 10
漢家女 0
漢家郎 3
漢寶德 26
漢將 7
漢川 8
漢弗萊 0
漢律 0
漢德 5
漢德字典 0
漢德詞典 0
漢文 124
漢族 117
漢族人 3
漢明帝 4
漢書 240
漢服 2
漢朝 52
漢朝人 1
漢武帝 23
漢民 150
漢水 34
漢江 15
漢法 6
漢法字典 1
漢法詞典 0
漢滿 1
漢滿蒙回藏 1
漢獻帝 1
漢王 36
漢白玉 7
漢碑 1
漢聲 166
漢英 65
漢英字典 12
漢英詞典 4
漢英辭典 0
漢語 214
漢譯 13
漢軍 21
漢邦 4
漢醫 10
漢陽 142
漢音 21
漢高帝 0
漢高祖 12
漣 284
漣漪 104
漤 0
漥 23
漦 34
漧 52
漨 0
漨 0
漨 0
漩 127
漩 127
漩渦 115
漪 302
漪漣 3
漫 1243
漫 1243
漫不經心 31
漫出 3
漫出來 0
漫天 124
漫天叫價 2
漫天討價 0
漫天開價 3
漫射 23
漫射光 1
漫山遍野 4
漫山遍野 4
漫延 91
漫條斯理 0
漫步 474
漫流 4
漫漫 73
漫漫長夜 13
漫無 29
漫無止境 5
漫無目的 26
漫無節制 1
漫無紀律 1
漫無邊際 7
漫無限制 4
漫畫 5729
漫畫家 252
漫畫書 39
漫筆 16
漫罵 35
漫衍 0
漫言 2
漫談 517
漫遊 588
漫遊四方 0
漫長 285
漬 147
漬染 1
漭 155
漮 96
漯 335
漯 335
漰 25
漱 960
漱口 58
漱口杯 2
漱口藥水 2
漱洗 4
漱石枕流 0
漱過 0
漲 2146
漲 2146
漲了 53
漲停 207
漲停板 21
漲價 240
漲價歸公 10
漲出 0
漲到 68
漲勢 93
漲回 5
漲回來 0
漲回去 0
漲在 1
漲大 9
漲大起來 1
漲幅 427
漲得 17
漲成 1
漲法 0
漲滿 14
漲漲 12
漲漲跌跌 2
漲潮 52
漲潮水位 0
漲紅 8
漲紅了臉 7
漲落 11
漲落起伏 0
漲著 3
漲起 2
漲起來 3
漲跌 640
漲跌互現 0
漲跌互見 26
漲過 3
漲過了頭 0
漲風 3
漳 200
漳州 120
漳河 2
漴 0
漴 0
漴 0
漵 62
漶 366
漷 14
漸 215
漸 2153
漸低 1
漸入佳境 56
漸升 7
漸地 3
漸增 54
漸小 7
漸少 10
漸弱音 2
漸強音 0
漸得 2
漸悟 1
漸慢 1
漸暗 6
漸次 85
漸減 27
漸漸 1096
漸漸地 183
漸漸的 183
漸現 1
漸癒 0
漸硬 0
漸行漸遠 0
漸變 27
漸起 5
漸輕 1
漸近 32
漸近線 2
漸退 5
漸進 259
漸進法 0
漸開 15
漸降 3
漸階 1
漹 140
漺 102
漺 102
漻 2
漼 31
漽 0
漾 74
漾漾 34
漿 1157
漿料 177
漿果 30
漿汁 1
漿洗 1
漿液 39
漿液 39
漿狀 8
漿粕 0
漿糊 37
漿糊 37
漿紗 72
漿膜 26
漿衣服 0
潀 103
潁 59
潂 0
潃 48
潄 0
潄 0
潅 0
潊 0
潎 16
潏 3
潐 3
潑 289
潑上 1
潑上來 0
潑上去 0
潑下 2
潑下來 0
潑下去 0
潑冷水 7
潑出 1
潑出來 1
潑出去 0
潑到 6
潑剌 0
潑墨 13
潑墨山水 2
潑墨畫 0
潑婦 4
潑婦罵街 3
潑成 1
潑掉 0
潑水 29
潑水節 9
潑辣 22
潑辣辣 0
潑進 0
潑進來 0
潑進去 0
潒 4
潓 16
潔 4011
潔操 4
潔淨 585
潔癖 30
潔白 94
潔白如玉 2
潔白無瑕 1
潔白無疵 0
潔身 5
潔身自好 1
潔身自愛 10
潕 8
潖 0
潗 4
潘 2661
潘妃金蓮 1
潘安 8
潘安再世 2
潘恩 20
潘金蓮 51
潙 0
潙 0
潙 0
潚 7
潚 7
潛 1511
潛上 4
潛上來 1
潛上去 0
潛下 1
潛下來 2
潛下去 0
潛了 10
潛伏 130
潛伏性 13
潛伏期 79
潛伏期 79
潛伏著 11
潛來 3
潛修 11
潛入 107
潛出 2
潛出來 0
潛出去 0
潛到 14
潛力 1127
潛勢 26
潛勢力 0
潛勢力 0
潛匿 1
潛去 0
潛回 5
潛在 895
潛居 1
潛得 2
潛心 51
潛意識 210
潛望鏡 9
潛水 2346
潛水人 20
潛水員 178
潛水夫 20
潛水艇 81
潛水衣 17
潛水裝 11
潛水鏡 3
潛熱 32
潛移默化 70
潛移默化 70
潛能 681
潛航 9
潛航力 0
潛艇 175
潛艇隊 0
潛藏 112
潛藏在 20
潛藏著 26
潛行 36
潛近 0
潛逃 68
潛進 11
潛進來 2
潛進去 1
潜 0
潜 0
潝 1
潝 1
潞 7
潟 19
潟湖 130
潠 146
潠 146
潡 4
潢 112
潣 25
潤 957
潤稿 100
潤化 0
潤喉 14
潤嗓 0
潤滑 238
潤滑劑 52
潤滑油 182
潤滑脂 5
潤澤 24
潤澤有光 0
潤濕 48
潤濕器 0
潤筆 4
潤絲精 8
潤肺 19
潤色 6
潤資 1
潤身 1
潤飾 19
潤餅 1
潥 0
潦 42
潦 42
潦倒 26
潦潦草草 0
潦草 16
潧 6
潨 0
潨 0
潩 2
潪 3
潫 0
潬 5
潭 1896
潭子 613
潭州 4
潭底 10
潭影 3
潭心 3
潭水 43
潭石 4
潭裡 1
潮 1815
潮信 1
潮州 767
潮氣 14
潮水 106
潮汐 132
潮流 855
潮湧 16
潮溼 62
潮濕 203
潮聲 9
潮解 17
潮音 21
潯 37
潰 185
潰 185
潰不成軍 6
潰兵 0
潰圍 0
潰敗 15
潰散 24
潰敵 1
潰決 13
潰滅 0
潰爛 56
潰瘍 451
潰裂 2
潰退 0
潰逃 0
潱 0
潲 5
潳 9
潴 0
潵 0
潶 9
潷 1
潸 71
潸潸 11
潹 0
潺 21
潺潺 53
潺潺流水 9
潻 29
潼 231
潼關 6
潽 28
潾 18
潿 15
澀 206
澀滯 0
澀縮 0
澁 0
澂 23
澃 0
澄 1
澄 795
澄心 5
澄明 19
澄江 1
澄沙 0
澄清 1114
澄清湖 172
澄澈 36
澄社 0
澄粉 0
澅 8
澆 232
澆下 3
澆冷水 3
澆口 9
澆地 1
澆愁 1
澆水 80
澆滅 1
澆灌 54
澆熄 8
澆築 2
澆花 31
澆薄 1
澆鑄 16
澇 55
澇 55
澇災 10
澈 278
澈底 108
澈悟 0
澉 0
澊 0
澋 4
澌 4
澍 31
澎 684
澎 684
澎湃 71
澎湃洶湧 7
澎湖 2711
澎湖人 22
澎湖列島 6
澎湖天后宮 1
澎湖縣 2702
澎湖群島 61
澎縣 15
澏 0
澐 55
澑 0
澑 0
澒 5
澓 3
澔 75
澕 7
澖 3
澗 58
澗水 5
澗流 0
澗溪 2
澘 0
澞 23
澟 0
澠 18
澠 18
澠池 0
澡 183
澡堂 30
澡塘 0
澡房 0
澡池 0
澡盆 12
澡缸 1
澢 2
澣 5
澤 4615
澤及 10
澤國 25
澤地 20
澤被 3
澥 10
澦 4
澧 75
澧縣 0
澨 4
澩 5
澪 6
澫 0
澬 6
澭 116
澮 7
澯 12
澰 5
澱 176
澱粉 406
澱粉糖 3
澱粉脢 2
澱粉質 12
澲 2
澳 1540
澳地利 2
澳大利亞 346
澳大利亞人 2
澳洲 4150
澳洲人 38
澳洲籍 8
澳門 1436
澳門人 7
澴 14
澵 0
澵 0
澶 9
澶 9
澶淵 1
澷 0
澷 0
澸 8
澹 41
澹 41
澹泊 9
澹然 1
澹然處之 0
澺 44
澻 0
澼 26
澽 25
澾 0
澿 7
激 768
激光 51
激光器 0
激出 6
激切 0
激劇 0
激動 343
激勵 685
激化 74
激增 187
激增人數 0
激增量 0
激奮 4
激將 9
激將法 9
激怒 47
激情 423
激情化 0
激憤 14
激戰 109
激揚 5
激於 1
激於義憤 1
激昂 43
激波 1
激波 1
激活 11
激流 39
激浪 1
激濁揚清 2
激烈 1379
激烈性 1
激發 772
激發態 5
激發起 2
激發起來 0
激盪 148
激盪不已 3
激磁 28
激素 433
激蕩 10
激論 0
激變 3
激賞 93
激起 213
激越 18
激辯 32
激進 99
激進份子 4
激進份子 4
激進派 10
濁 158
濁世 7
濁度 60
濁氣 6
濁水 116
濁水溪 193
濁流 47
濁浪 0
濁物 2
濁聲 3
濁酒 9
濁音 7
濂 372
濃 1178
濃到 3
濃厚 460
濃味 0
濃墨 3
濃妝 6
濃妝艷抹 1
濃妝豔抹 5
濃密 83
濃密度 1
濃度 2187
濃情密意 0
濃情蜜意 14
濃抹 4
濃於 2
濃暗 0
濃汁 4
濃液 3
濃液 3
濃淡 27
濃淡適中 1
濃湯 67
濃漿 0
濃濃 137
濃烈 70
濃煙 74
濃痰 2
濃眉 24
濃眉大眼 10
濃稠 1
濃粧 1
濃綠 12
濃縮 538
濃縮劑 0
濃縮品 0
濃縮物 3
濃而不烈 0
濃膩 0
濃色 34
濃艷 5
濃茶 23
濃蔭 11
濃蔭蔽天 1
濃蔭蔽空 0
濃血症 0
濃裝豔抹 1
濃豔 2
濃郁 148
濃酒 1
濃重 24
濃雲 3
濃雲密佈 0
濃霧 50
濃霧密佈 0
濃髮 1
濄 41
濅 0
濅 0
濆 19
濇 18
濈 17
濉 8
濊 5
濋 30
濌 1
濍 5
濎 4
濏 5
濐 0
濓 0
濔 4
濕 1
濕 10648
濕地 110
濕地松 5
濕婆 10
濕季 4
濕度 46
濕度表 0
濕度計 5
濕氣 62
濕淋淋 5
濕淋淋的 12
濕淋淋地 12
濕漉漉 1
濕漉漉的 11
濕漉漉地 11
濕潤 20
濕熱 10
濕球溫度 4
濕疹 31
濕空氣 4
濕答答 6
濕透 12
濖 0
濗 0
濘 137
濘 137
濘泥 0
濘泥 0
濘滯 0
濙 0
濚 0
濚 0
濛 62
濛濛 32
濛濛細雨 5
濛濛谷 2
濛霧 2
濜 1
濝 0
濞 2
濟 364
濟 3645
濟世 80
濟事 4
濟公 80
濟公活佛 25
濟助 84
濟南 83
濟南人 2
濟南市 10
濟南慘案 1
濟南路 147
濟困 4
濟寧 5
濟弱扶傾 3
濟急 4
濟濟 7
濟濟多士 20
濟燃眉之急 0
濟私 0
濟茲 6
濟貧 23
濟貧所 0
濠 97
濠溝 0
濡 74
濡染 1
濡溼 6
濡濕 8
濢 1
濢 1
濣 2
濤 286
濤 686
濤聲 17
濥 0
濦 3
濧 3
濨 5
濩 447
濪 0
濫 529
濫交 12
濫伐 50
濫伐 50
濫伐林木 2
濫伐林木 2
濫取 0
濫墾 193
濫好人 0
濫射 0
濫情 50
濫收 0
濫殺 12
濫殺無辜 4
濫用 515
濫砍 5
濫砍濫伐 1
濫砍濫伐 1
濫竽 0
濫竽充數 31
濫觴 31
濫調 5
濫調陳腔 0
濫造 2
濬 180
濬哲 2
濬智 1
濬深 1
濭 3
濮 179
濯 25
濯足 7
濯身 0
濰 48
濱 1273
濱江 33
濱江街 25
濱河 10
濱河公園 0
濱海 267
濱海公路 105
濱湖 8
濱湖公路 0
濱臨 28
濱臨大海 0
濲 1
濳 0
濳 0
濴 6
濵 0
濵 0
濶 0
濷 9
濺 239
濺 239
濺了 2
濺水 0
濺濕 2
濻 9
濻 9
濼 5
濽 0
濾 1242
濾出 22
濾出來 1
濾出去 1
濾到 0
濾去 11
濾嘴 2
濾器 25
濾水器 62
濾水池 0
濾池 6
濾波 51
濾波 51
濾波器 216
濾波器 216
濾液 18
濾液 18
濾清 44
濾球 0
濾紙 76
濾色鏡 3
濾過 21
濾過性 5
濾過性毒 2
濾過性病毒 48
濾鏡 1
濾頭 6
濿 20
瀀 23
瀁 168
瀂 0
瀃 0
瀄 2
瀅 175
瀅 175
瀆 35
瀆職 77
瀆職罪 3
瀇 22
瀈 0
瀉 253
瀉下 7
瀉下來 1
瀉下去 0
瀉了 4
瀉出 3
瀉出來 1
瀉出去 0
瀉到 2
瀉劑 26
瀉在 2
瀉地 2
瀉得 0
瀉得 9
瀉法 1
瀉湖 13
瀉的 9
瀉肚 0
瀉肚子 3
瀉藥 8
瀊 4
瀋 21
瀋海 0
瀋陽 136
瀋陽市 26
瀌 15
瀍 8
瀎 20
瀏 896
瀏海 22
瀏覽 13238
瀏覽器 0
瀏陽 4
瀐 0
瀐 0
瀑 200
瀑布 912
瀑布區 9
瀑布群 13
瀒 0
瀓 0
瀓 0
瀔 178
瀕 125
瀕 125
瀕於 7
瀕於絕境 0
瀕於絕種 0
瀕死 22
瀕河 0
瀕海 4
瀕臨 191
瀕臨絕境 1
瀕臨絕種 145
瀖 3
瀗 2
瀘 41
瀘州 19
瀙 3
瀙 3
瀚 415
瀚海 4
瀛 397
瀛台 3
瀛洲 12
瀛海 30
瀛臺 3
瀜 4
瀝 85
瀝瀝 2
瀝瀝拉拉 0
瀝血 9
瀝青 265
瀞 0
瀟 38
瀟洒 54
瀟湘 83
瀟瀟 15
瀟灑 220
瀟灑自如 0
瀠 11
瀡 7
瀢 2
瀣 12
瀤 4
瀤 4
瀥 0
瀥 0
瀦 20
瀧 545
瀧 545
瀨 371
瀩 6
瀪 2
瀫 3
瀫 3
瀬 0
瀯 9
瀰 72
瀰漫 108
瀰漫性 19
瀰漫著 67
瀱 0
瀲 25
瀲灩 25
瀳 8
瀴 4
瀴 4
瀴 4
瀵 4
瀶 0
瀷 8
瀸 0
瀹 13
瀹 13
瀺 8
瀻 3
瀼 9
瀼 9
瀽 0
瀾 281
瀾滄江 1
瀿 8
灀 5
灁 14
灂 4
灂 4
灃 76
灄 43
灅 6
灆 6
灇 0
灇 0
灈 113
灉 27
灊 21
灋 0
灋 0
灋 0
灋 0
灌 3691
灌上 4
灌下 5
灌了 60
灌來灌去 1
灌個 19
灌入 67
灌出 1
灌出來 0
灌出去 0
灌到 25
灌區 20
灌反 0
灌向 0
灌唱片 3
灌回 1
灌回去 0
灌在 8
灌好 15
灌嬰 0
灌得 2
灌救 25
灌於 2
灌木 106
灌木叢 17
灌木狀 1
灌死 1
灌氣 17
灌水 886
灌油 1
灌法 0
灌注 159
灌洗 20
灌溉 551
灌溉區 14
灌溉工程 11
灌溉渠道 5
灌溉系統 20
灌溉面積 9
灌滿 15
灌漿 73
灌漿兒 0
灌漿兒 0
灌灌 472
灌籃 214
灌米湯 0
灌縣 1
灌腸 33
灌著 0
灌藥 2
灌起 0
灌起來 0
灌輸 180
灌迷湯 1
灌進 14
灌進來 0
灌進去 1
灌過 7
灌酒 10
灌醉 20
灌音 1
灌頂 1
灌點 9
灍 0
灑 528
灑上 34
灑下 25
灑下來 2
灑下去 0
灑出 18
灑掃 3
灑掃應對 3
灑水 60
灑水掃地 0
灑水禮 2
灑水車 3
灑淚 5
灑脫 44
灑落 28
灑透 0
灑遍 3
灑遍 3
灒 8
灓 0
灓 0
灔 0
灕 67
灖 6
灗 4
灗 4
灘 400
灘上 12
灘聲 0
灘頭 14
灘頭堡 14
灘頭陣地 0
灙 0
灚 59
灛 5
灝 125
灝氣 0
灞 20
灞橋 2
灟 36
灠 524
灠 524
灡 65
灢 2
灣 10730
灣仔 22
灣流 6
灣潭 5
灣頭 14
灤 13
灤州 1
灤洲 0
灥 241
灦 3
灦 3
灧 0
灨 462
灨江 0
灩 59
灪 27
火 4643
火上 12
火上加油 11
火中 44
火中取栗 1
火了 9
火事 0
火井 0
火伕 0
火併 5
火候 33
火候未到 0
火傘 0
火傘高張 2
火傷 6
火光 96
火光熊熊 1
火冒三丈 10
火刑 4
火剪 0
火力 399
火力圈 0
火力發電 49
火力發電廠 84
火力網 1
火力線 0
火力點 0
火勢 154
火化 39
火口 6
火口湖 0
火器 22
火坑 29
火堆 23
火場 15
火場 155
火大 48
火奴魯魯 0
火宅 7
火宿 0
火山 437
火山口 35
火山岩 11
火山島 15
火山灰 17
火山爆發 78
火山錐 0
火巷 0
火急 9
火性 15
火成岩 37
火把 86
火拼 21
火攻 13
火星 512
火星人 31
火星塞 128
火曜日 1
火柴 163
火柴桿 1
火柴盒 24
火棒 2
火槍 11
火樹 6
火樹銀花 15
火氣 66
火油 11
火海 23
火海戰術 0
火源 55
火漆 11
火災 823
火災保險 32
火災現場 17
火災現場 17
火災險 1
火炎山 52
火炕 1
火炬 81
火炬松 0
火炭 16
火炮 19
火炷 0
火烈 0
火烤 34
火焰 420
火煙 1
火熱 150
火熱熱 0
火熾 2
火燄 147
火燒 213
火燒云 0
火燒山 4
火燒島 31
火燒眉毛 3
火燒雲 0
火燭 15
火爆 255
火爐 60
火牆 22
火牛陣 8
火狐狸 90
火球 130
火盆 8
火眼金睛 1
火石 22
火石玻璃 0
火砲 37
火硝 0
火神 35
火祭 11
火禁 0
火種 55
火窟 26
火窯 3
火箭 308
火箭炮 2
火箭發動 0
火箭發動機 4
火箭筒 17
火箱 0
火粒 0
火紅 40
火紅色 5
火紙媒兒 0
火紙媒兒 0
火網 8
火綿 4
火線 124
火繩 0
火繩桿 0
火腿 228
火腿三明治 0
火腿漢堡 0
火腿肉 4
火腿蛋 2
火舌 16
火花 220
火花塞 2
火花放電 5
火苗 39
火葬 33
火葬場 22
火葬場 22
火藥 104
火藥味 29
火藥庫 17
火警 337
火車 1571
火車時刻 128
火車時刻表 173
火車票 254
火車站 736
火車軌 0
火車頭 100
火輪 36
火辣 74
火辣辣 16
火速 84
火酒 3
火金姑 0
火鉗 3
火鍋 515
火鍋子 0
火險 54
火雞 118
火雞肉 10
火電站 1
火頭 9
火頭上 0
火頭軍 0
火頭金剛 1
火鳥 162
火鷓鴣鳥 0
火龍 113
火龍果 1
灬 0
灮 0
灯 0
灰 5463
灰分 32
灰口鐵 1
灰土 4
灰塵 209
灰姑娘 202
灰心 110
灰心喪氣 0
灰指甲 20
灰暗 60
灰沙 1
灰泥 13
灰溜溜 1
灰滅 0
灰漿 4
灰濛 8
灰濛濛 16
灰灰 14
灰灰暗暗 0
灰灰濛濛 2
灰燼 89
灰狗 25
灰白 41
灰白色 32
灰白質 6
灰石 8
灰石粉 3
灰綠色 7
灰肥 0
灰色 551
灰褐 9
灰褐色 21
灰質 16
灰鑄鐵 1
灰階 1
灰霧 2
灰頭土臉 24
灰飛煙滅 0
灰鯨 11
灰黑 46
灰黯 5
灰鼠 8
灱 8
灲 0
灲 0
灲 0
灳 0
灴 2
灵 0
灶 235
灶匠 0
灶君 3
灶房 0
灶火 0
灶王爺 0
灶神 11
灶雞 0
灷 0
灸 132
灸師 0
灸法 5
灸術 0
灹 0
灺 6
灻 0
灼 142
灼傷 161
灼灼 9
灼熟 0
灼熱 90
灼燒 12
灼燙 14
灼爛 0
灼痛 5
灼見 3
災 1034
災區 202
災厄 10
災害 1773
災害性 19
災患 1
災情 244
災情慘重 29
災星 0
災殃 3
災民 175
災異 6
災禍 59
災胞 12
災荒 2
災變 329
災難 417
災難片 8
灾 0
炁 0
炂 1
炃 2
炄 2
炅 7
炆 9
炇 0
炈 0
炉 0
炊 6
炊 684
炊事 26
炊事員 3
炊具 21
炊婦 1
炊火 0
炊煙 17
炊煙裊裊 0
炊金饌玉 0
炊飯 7
炋 0
炌 0
炌 0
炍 0
炎 2743
炎夏 37
炎天 9
炎帝 3
炎日 5
炎日當空 0
炎日高照 0
炎暑 4
炎海 1
炎涼 5
炎炎 39
炎炎夏日 74
炎熇 0
炎熱 117
炎熱夏日 0
炎熾 0
炎症 113
炎酷 2
炎陽 58
炎風 0
炎黃子孫 25
炏 0
炐 0
炑 2
炒 1118
炒上 0
炒上來 0
炒上去 0
炒下 2
炒下來 0
炒下去 0
炒作 285
炒作股票 23
炒來炒去 0
炒勺 0
炒在 0
炒地皮 39
炒栗子 8
炒炒 93
炒熟 13
炒熱 45
炒股票 13
炒菜 40
炒菜鏟 0
炒著 1
炒蛋 26
炒起 4
炒起來 0
炒過 13
炒過來 0
炒過去 0
炒青 23
炒飯 71
炒魷魚 6
炒麵 17
炓 18
炔 14
炔 14
炔 14
炕 1477
炕床 1
炕桌 6
炕沿 3
炕洞 0
炕頭 0
炖 5
炗 0
炘 71
炙 370
炙手可熱 49
炙熱 44
炚 4
炛 0
炛 0
炛 0
炛 0
炟 74
炠 0
炡 103
炢 0
炣 0
炤 689
炥 0
炦 0
炧 0
炧 0
炧 0
炧 0
炨 0
炨 0
炨 0
炨 0
炩 216
炪 0
炪 0
炫 1020
炫亂 0
炫奇無巧 0
炫惑 7
炫燿 1
炫目 33
炫示 0
炫耀 103
炫耀著 5
炬 1375
炬火 0
炭 964
炭刷 0
炭化 8
炭化作用 1
炭化物 0
炭坑 2
炭氣 0
炭水化合物 2
炭油 0
炭火 21
炭灰色 1
炭爐 0
炭畫 0
炭疽病 10
炭盆 0
炭窯 8
炭筆 34
炭粉 1
炭精 6
炭素 11
炭質 1
炭酸 21
炭酸鹽 1
炮 146
炮 146
炮 1460
炮仗 11
炮仗 11
炮位 2
炮兵 6
炮制 2
炮口 3
炮台 9
炮彈 20
炮手 1
炮擊 1
炮樓 15
炮火 30
炮灰 4
炮烙 2
炮烙之刑 1
炮眼 0
炮竹 0
炮聲 12
炮聲隆隆 5
炮艇 2
炮製 20
炮轟 34
炮隊 0
炮食 0
炯 967
炯介 2
炯炯 10
炯炯有神 9
炯然 3
炰 211
炱 140
炲 0
炲 0
炳 1761
炳煥 0
炳燿 2
炴 100
炵 538
炵 538
炶 0
炷 226
炷頭香 0
炷香 4
炸 232
炸 2329
炸中 2
炸丸子 0
炸傷 14
炸出 5
炸到 3
炸平 1
炸彈 481
炸彈開花 0
炸得 15
炸成 9
炸掉 8
炸斷 1
炸死 10
炸毀 21
炸油條 0
炸炸 1
炸燬 6
炸爛 1
炸的 14
炸破 0
炸碎 0
炸糕 0
炸肉 2
炸肉丸 0
炸肉塊 0
炸葯 1
炸藥 71
炸藥箱 0
炸蝦 1
炸裂 2
炸起 1
炸起來 0
炸醬 1
炸醬麵 16
炸開 31
炸雞 91
炸雞塊 7
炸雞肉 0
炸雞腿 0
炸魚 59
炸魚排 1
炸麻花 0
点 0
為 8000
為 19033
為中 263
為中心 579
為主 5164
為之一新 1
為之動容 10
為亂 8
為了 13586
為人 946
為人作嫁 4
為人在世 0
為人師表 15
為什麼 9792
為什麼 9792
為今之計 3
為他 767
為他們 265
為伍 54
為伴 27
為何 7202
為何 7202
為何不 173
為何在 142
為何是 29
為何會 250
為何有 105
為何能 32
為你 1161
為你們 75
為佳 284
為例 1487
為偶 73
為傲 44
為傲的是 0
為兄 11
為先 104
為公 98
為公 98
為利 42
為力 19
為反對 11
為反對而反對 15
為名 151
為名為利 1
為君 39
為啥 193
為善 69
為善不欲人知 1
為善最樂 0
為國 502
為國增光 0
為國家 304
為國捐軀 1
為國爭光 19
為國犧牲 2
為圖 110
為大 280
為子 26
為學 54
為學之道 0
為官 34
為宜 251
為害 106
為家 80
為家庭 65
為富不仁 4
為對 114
為己 48
為師 161
為幸 4
為念 31
為患 17
為惡 34
為愛 280
為憑 61
為我 741
為我們 679
為所欲為 44
為把 25
為政 55
為敵 38
為數 243
為數甚微 0
為數眾多 43
為文 92
為時 59
為時不遠 1
為時已晚 31
為時過早 6
為期 616
為期不遠 13
為本 744
為校 36
為校增光 0
為校爭光 9
為業 52
為榮 108
為榮的是 0
為權 10
為權為利 0
為止 1198
為此 880
為母 30
為民 1209
為民前鋒 2
為民喉舌 10
為民請命 1
為民除害 6
為父 37
為甚 54
為甚麼 361
為甚麼 361
為生 109
為由 322
為的 270
為的是 121
為盼 6
為禍 12
為私 23
為臨 8
為著 281
為虎作倀 3
為虎傳翼 0
為要 174
為要求 9
為證 58
為輔 155
為鄰 56
為重 239
為間 10
為限 1029
為難 175
為難起來 0
為非 261
為非作歹 13
為題 124
為首 337
炻 0
炾 236
炾 236
烀 0
烄 0
烅 1
烆 3
烇 5
烈 1265
烈到 0
烈士 56
烈士陵園 0
烈女 25
烈女傳 0
烈婦 4
烈屬 0
烈度 0
烈性 6
烈日 58
烈日當空 9
烈暑 0
烈火 172
烈焰 31
烈焰騰空 1
烈節 0
烈酒 56
烈風 13
烈馬 2
烉 0
烊 7
烋 10
烌 0
烍 2
烎 3
烏 1773
烏七八糟 0
烏亮 4
烏來 302
烏來鄉 93
烏克蘭 171
烏合 2
烏合之眾 8
烏呼 0
烏坵 92
烏山頭 105
烏山頭水庫 54
烏巾 0
烏干達 18
烏托邦 70
烏托邦主義 1
烏拉 24
烏拉圭 147
烏拉山 3
烏拉草 3
烏日 658
烏有 17
烏木 6
烏桓 1
烏梁海 2
烏梅 14
烏梅汁 3
烏梅酒 4
烏江 9
烏池 0
烏沈沈 0
烏溜溜 10
烏焉成馬 0
烏煙瘴氣 25
烏爾 20
烏瑟沙摩 1
烏瑟沙摩明王 1
烏篷 0
烏紗帽 6
烏腳病 42
烏芻澀摩 1
烏芻澀摩明王 1
烏茲別克 0
烏藍 0
烏蘇 9
烏蘇里江 10
烏蘭巴托 5
烏豆 11
烏賊 136
烏達 6
烏金 14
烏雲 82
烏雲密佈 10
烏鞘嶺 0
烏骨雞 4
烏髮 5
烏魚 170
烏魚子 30
烏魯木齊 20
烏鴉 163
烏鴉嘴 6
烏黑 49
烏黑亮麗 7
烏黑黑 0
烏龍 0
烏龍茶 199
烏龍駒 0
烏龍麵 0
烏龜 345
烐 0
烑 6
烒 8
烓 2
烔 23
烕 0
烖 0
烗 3
烘 245
烘乾 105
烘乾機 40
烘了 0
烘來烘去 0
烘到 1
烘培 862
烘得 2
烘得 2
烘成 0
烘托 36
烘托出 13
烘板 0
烘烘 7
烘烘手 0
烘烤 57
烘焙 2387
烘焦 0
烘燻 0
烘爐 16
烘箱 33
烘製 9
烘襯 2
烘起來 0
烘過 0
烘過來 0
烘過去 0
烘雲托月 0
烘青 8
烙 144
烙 144
烙印 65
烙成 1
烙痕 3
烙痕 3
烙鐵 13
烙餅 7
烚 2
烛 0
烛 0
烜 40
烝 18
烞 7
烟 0
烠 10
烡 8
烢 6
烣 0
烤 1296
烤乾 8
烤了 4
烤來烤去 0
烤個 0
烤完 8
烤成 7
烤晒 0
烤架 2
烤漆 1
烤漆板 0
烤火 22
烤烤 7
烤烤肉 1
烤焦 14
烤煙 1
烤熟 11
烤熟 11
烤爐 20
烤牛肉 13
烤箱 102
烤紅 1
烤羊 7
烤羊肉 2
烤肉 459
烤肉醬 4
烤起來 0
烤過 7
烤過來 0
烤過去 0
烤錯 0
烤雞 30
烤雞肉 1
烤鴨 38
烤鴨肉 0
烤麵包 19
烤麵包機 14
烤黑 0
烮 0
烯 363
烯烴 43
烰 2
烱 0
烲 0
烲 0
烳 4
烴 64
烵 0
烶 0
烷 171
烷基 86
烷烴 4
烸 5
烹 115
烹宰 0
烹煮 46
烹煮法 1
烹茗 0
烹茶 1
烹調 292
烹調法 17
烹醢 0
烹飪 323
烹飪器 0
烹飪法 2
烺 6
烻 3
烼 4
烽 137
烽火 197
烽火台 2
烽火四起 1
烽火臺 5
烽火連三月 1
烽火連年 2
烽煙 3
烽煙四起 0
烾 0
烿 4
焀 8
焁 0
焂 6
焃 0
焄 17
焅 0
焅 0
焆 4
焇 0
焈 0
焈 0
焉 457
焉博 0
焉得 8
焉得虎子 10
焉敢 2
焉有 10
焉用 2
焉知 13
焉知非福 9
焉能 28
焊 320
焊上 3
焊上去 0
焊了 1
焊入 0
焊出 0
焊到 0
焊劑 27
焊工 11
焊得 0
焊成 1
焊接 315
焊接工 5
焊料 1
焊條 16
焊槍 9
焊燒 0
焊縫 1
焊著 0
焊過 1
焊錫 42
焋 3
焌 6
焍 2
焍 2
焎 8
焏 0
焏 0
焐 3
焓 11
焗 101
焗坊 1
焗烤 1
焙 92
焙乾 1
焙燒 4
焙粉 1
焙茶 9
焚 181
焚化 648
焚化爐 380
焚屍 13
焚屍案 4
焚掠 0
焚書 4
焚書坑儒 2
焚毀 50
焚燒 99
焚燒法 0
焚燬 15
焚琴煮鶴 2
焚膏繼晷 3
焚舟 0
焚起 1
焚身 15
焚風 47
焚香 35
焚香燒爐 0
焛 4
焜 332
焝 0
焞 3
焟 4
焠 18
無 1
無 29825
無一 220
無一不精 4
無一不能 1
無一不通 1
無上 180
無上光榮 1
無上正等正覺 1
無不 562
無中生有 51
無主 44
無主物 2
無事不登三寶殿 1
無事忙 0
無事生非 1
無人 1124
無人島 23
無人過問 0
無人駕駛 10
無人駕駛飛機 0
無他 48
無以復加 9
無以為生 1
無以為繼 9
無以置答 0
無任 212
無休止 9
無住處涅槃 1
無何 3
無依無靠 6
無偏無黨 0
無傷 21
無傷大雅 18
無價 69
無價之寶 19
無價寶 4
無償 135
無光 45
無冕王 0
無冤無仇 7
無出其右 5
無出其右者 2
無分 47
無分彼此 1
無分軒輊 4
無利可圖 20
無前 22
無力 741
無力感 79
無力招架 1
無力解決 2
無功 30
無功不受祿 0
無功功率 0
無功受祿 0
無功而返 28
無功而退 3
無助 215
無助於 29
無動於衷 37
無原則 1
無可 361
無可厚非 39
無可奈何 118
無可奉告 5
無可如何 4
無可救藥 43
無可比擬 26
無可爭辯 2
無可置疑 12
無可諱言 18
無可避免 123
無可限量 17
無可非議 3
無名 235
無名小卒 16
無名指 41
無名氏 91
無名火 13
無名英雄 12
無味 61
無國籍 46
無地自容 6
無垠 60
無垢 12
無夫 1
無奇不有 33
無奈 992
無奈是 0
無如 3
無妄之災 11
無妨 117
無始無明 1
無孔不入 16
無字天書 10
無官一身輕 0
無官一身輕 0
無定形碳 0
無害 159
無家可歸 46
無對 11
無尾熊 222
無巧不成書 3
無己 4
無師自通 15
無常 247
無干 8
無幾 23
無底洞 15
無度 10
無庸 119
無庸置疑 0
無形 448
無形中 198
無形之中 14
無影無蹤 34
無往不利 35
無待 3
無後 11
無後為大 2
無後顧之憂 83
無從 312
無從交代 0
無微不至 63
無心 398
無心之過 12
無忝所生 1
無性 52
無性生殖 19
無性雜交 0
無怨無悔 0
無怪 24
無怪乎 68
無恙 29
無恥 117
無恥之徒 23
無情 583
無惡不作 7
無意 471
無意中 128
無意義 116
無意識 72
無意間 148
無愧 43
無愧於 8
無憂無慮 60
無憑無據 2
無懈可擊 31
無成 17
無我 130
無所 337
無所不包 36
無所不在 120
無所不有 6
無所不為 6
無所不用其極 60
無所不知 15
無所不精 0
無所不能 27
無所不說 0
無所不通 4
無所事事 34
無所忌諱 0
無所畏懼 3
無所謂 289
無所遁形 45
無所適從 71
無技可施 1
無拘無束 40
無損 65
無損於 19
無政府 19
無政府主義 22
無政府狀態 15
無故 195
無故缺席 17
無故遲到 3
無效 643
無敵 923
無敵不克 0
無敵手 26
無敵艦隊 0
無敵軍 0
無數 815
無日 12
無日不 2
無日不 2
無明 115
無明火 0
無時 43
無時無刻 92
無時無地 3
無暇 151
無暇顧及 12
無望 54
無期 29
無期徒刑 80
無枉無縱 0
無果果樹 0
無根水 0
無梁樓蓋 0
無條件 211
無條件反射 0
無條件投降 12
無業 17
無業游民 1
無業遊民 9
無極 139
無樑樓蓋 0
無機 336
無機化合物 17
無機化學 83
無機可乘 0
無機物 20
無機肥料 0
無機酸 11
無機體 0
無機鹽 18
無權 138
無欲則剛 3
無止境 92
無止無休 0
無止盡 75
無殼蝸牛 36
無毒 77
無毒不丈夫 2
無比 564
無水 100
無法 19681
無法可想 4
無法可施 8
無法無天 198
無法無邊 0
無涯 36
無濟於事 34
無為 66
無為而治 11
無煙 19
無煙火藥 1
無煙煤 5
無照 42
無照營業 3
無照駕駛 13
無父 4
無物 47
無牽無掛 3
無狀 2
無獨有偶 20
無理 91
無理取鬧 35
無理數 7
無理方程 0
無瑕 36
無生物 3
無產者 3
無產階級 31
無產階級專政 16
無產階級民主 0
無產階級革命 2
無用 358
無用武之地 12
無用論 3
無由 27
無畏 54
無畏佈施 1
無畏於 0
無畏施 1
無異 173
無異於 24
無異是 49
無異議 33
無異議通過 22
無疆 14
無疑 390
無疑是 205
無疵 0
無疾而終 26
無病 43
無病呻吟 17
無痛分娩 16
無的放矢 14
無益 94
無盡 263
無盡期 4
無相 1
無知 326
無礙 185
無神 20
無神論 20
無神論者 20
無福消受 7
無禮 77
無私 142
無稽 14
無稽之談 16
無窗式 0
無窮 373
無窮無盡 31
無窮盡 32
無立足之地 4
無端 80
無端惹事 0
無端生事 0
無端起風波 0
無端起風波 0
無端鬧事 0
無米之炊 24
無籽西瓜 0
無精打采 24
無約 6
無約在身 0
無終 12
無給職 56
無線 1820
無線網路 1820
無線電 1432
無線電台 53
無線電報 14
無線電定位 2
無線電廣播 23
無線電廣播 23
無線電接收機 2
無線電波 42
無線電波 42
無線電臺 45
無線電視 86
無線電視台 52
無線電話 166
無線電通信 51
無線電電子學 1
無緣 283
無緣無故 47
無緣見面 3
無縫鋼管 9
無缺 34
無罪 191
無罪開釋 4
無翼鳥 0
無聊 1422
無聲 230
無聲無息 60
無聲無臭 0
無聲電影 5
無能 178
無能勝明王 1
無能為力 65
無腳 3
無與倫比 44
無色 185
無色界 1
無花果 56
無花果樹 13
無菌 198
無蓋 5
無處 184
無處可去 11
無處可逃 14
無虞 96
無虞匱乏 2
無行 19
無行 19
無補 13
無補於事 1
無規 14
無視 80
無視於 84
無角 2
無解 112
無言 170
無言以對 45
無言對泣 0
無言抗議 2
無計可施 27
無記名 45
無記名投票 24
無話 41
無話可說 56
無話可講 0
無話好說 0
無語 0
無語問蒼天 1
無誤 367
無諍三昧 1
無論 3315
無論如何 391
無謂 154
無貲 0
無賴 24
無賴漢 1
無趣 80
無足輕重 10
無路 41
無路可走 20
無軌 3
無軌可循 0
無軌電車 0
無辜 338
無辜者 11
無過 15
無道 14
無遠弗屆 179
無遺 60
無邊 157
無邊春色 0
無邊無礙 0
無邊無際 23
無邊讀中間 0
無邊風月 0
無邪 34
無量 569
無量識 1
無鉛汽油 1
無錫 88
無錫市 16
無間 24
無間 24
無關 912
無關大局 1
無關宏旨 1
無關痛癢 8
無關緊要 34
無阻 20
無限 2587
無限公司 20
無限制 405
無限大 26
無限期 49
無隙可乘 3
無際 68
無雙 134
無需 387
無霜期 2
無非 135
無非是 124
無非法相 1
無韻詩 0
無須 491
無頭公案 0
無頭案 0
無頭無尾 3
無頭無腦 0
無題 255
無題詩 0
無顎魚 0
無顏以對 1
無顏見人 1
無顏見江東父老 1
無風 69
無風不起浪 3
無風帶 2
無風起浪 0
無餘 13
無餘涅槃 1
無饜 1
無黨 201
無黨無派 6
無黨籍 508
無齒 13
焢 25
焣 6
焤 0
焥 8
焥 8
焦 17400
焦了 2
焦作 1
焦切 0
焦化 4
焦化廠 0
焦味 6
焦土 6
焦土政策 0
焦心 1
焦思 1
焦急 102
焦急萬分 1
焦慮 449
焦慮不安 22
焦枯 1
焦比 14
焦油 33
焦渴 1
焦灼 8
焦炭 16
焦煤 1
焦熱 1
焦爐煤氣 0
焦皮 1
焦砟 0
焦碳 2
焦糊 0
焦糖 21
焦耳 24
焦裕祿 0
焦距 186
焦躁 49
焦躁感 2
焦達峰 0
焦頭爛額 20
焦頭額爛 0
焦飯 0
焦黃 3
焦黑 35
焦黑色 0
焦點 19223
焧 0
焨 11
焪 0
焫 0
焫 0
焬 0
焭 0
焮 4
焯 64
焰 306
焰彩 0
焰心 0
焰火 13
焰硝 0
焱 54
焲 10
焳 0
焴 0
焵 0
然 7275
然則 45
然否 6
然後 8186
然後就 166
然而 5943
然諾 7
焷 0
焸 0
焺 5
煁 3
煂 6
煃 9
煄 114
煅 0
煆 16
煇 147
煈 0
煉 375
煉丹 15
煉丹術 0
煉乳 8
煉制 0
煉氣 4
煉油 72
煉油廠 723
煉焦 9
煉焦廠 2
煉焦煤 0
煉焦爐 2
煉獄 36
煉石補天 0
煉糖 2
煉糖廠 1
煉藥 5
煉製 181
煉製廠 5
煉金 6
煉金術 7
煉銅 9
煉銅廠 2
煉鋁 6
煉鋼 75
煉鋼廠 44
煉鋼業 17
煉鋼爐 4
煉鐵 14
煉鐵廠 1
煉鐵爐 0
煊 0
煊 0
煋 5
煌 1369
煌煌 12
煌熠 0
煍 11
煎 359
煎了 1
煎作 0
煎個 0
煎出 1
煎到 1
煎反 0
煎對 0
煎得 0
煎成 7
煎法 2
煎炒 3
煎煎 0
煎餃 25
煎煮 21
煎熬 106
煎肉 4
煎肉餅 0
煎著 3
煎藥 9
煎藥法 0
煎蛋 18
煎起 0
煎起來 0
煎迫 2
煎過 0
煎過來 0
煎過去 0
煎餅 29
煎魚 7
煎點 0
煏 0
煐 23
煑 0
煒 404
煓 6
煔 5
煖 52
煖 52
煖和 0
煖房 0
煖爐 0
煖玉 0
煖衣飽食 0
煗 0
煗 0
煘 4
煙 3710
煙供 1
煙具 22
煙卷兒 0
煙卷兒 0
煙台 30
煙台市 2
煙味 0
煙商 2
煙嘴 0
煙嘴兒 0
煙嘴兒 0
煙囪 91
煙土 1
煙塵 42
煙客 3
煙屁股 1
煙岸 0
煙幕 14
煙幕彈 5
煙捲 0
煙捲兒 0
煙捲兒 0
煙斗 42
煙斗絲 0
煙桿 2
煙槍 10
煙槍洞 0
煙機 17
煙毒 166
煙毒犯 15
煙油 4
煙泡 0
煙波 75
煙波 75
煙海 5
煙消火滅 3
煙消雲散 28
煙消霧散 1
煙火 433
煙灰 25
煙灰缸 23
煙煤 7
煙熅 0
煙燻 20
煙癮 9
煙癮又犯了 0
煙盒 7
煙筒 2
煙筒山 0
煙管 17
煙粉 0
煙絲 6
煙縷 0
煙缸 0
煙膏 0
煙臭 3
煙臭味 1
煙臺 2
煙花 45
煙花巷 0
煙草 64
煙葉 10
煙蒂 16
煙袋 3
煙袋嘴 0
煙袋鍋子 0
煙酒 67
煙酒不沾 1
煙雨 40
煙雲過眼 0
煙霏 2
煙霧 171
煙霧瀰漫 6
煙霧迷漫 16
煙靄 2
煙頭 4
煙鬼 2
煙鹼 2
煚 82
煛 0
煜 529
煜熠 0
煝 2
煞 616
煞 616
煞住 4
煞星 29
煞是 15
煞是好看 6
煞有介事 4
煞有其事 8
煞氣 3
煞神 2
煞筆 0
煞費心機 2
煞費苦心 12
煞車 831
煞車 831
煞車不靈 1
煞車不靈 1
煞車失靈 6
煞車失靈 6
煞車燈 24
煞車燈 24
煟 2
煠 10
煠 10
煡 6
煢 12
煢居 0
煢煢 4
煢獨 0
煣 2
煤 781
煤倉 0
煤區 3
煤坑 1
煤場 14
煤場 14
煤屑 3
煤層 17
煤帶 1
煤業 11
煤氣 536
煤氣公司 4
煤氣工廠 1
煤氣燈 0
煤氣爐 2
煤氣筒 0
煤氣管 1
煤油 49
煤油燈 5
煤油爐 6
煤渣 11
煤火 0
煤灰 22
煤炭 101
煤炭工 0
煤炭工業 5
煤焦 12
煤焦油 14
煤煙 5
煤爐 0
煤球 4
煤田 8
煤磚 2
煤礦 238
煤礦工 2
煤礦工人 1
煤窯 1
煤系 3
煤耗 0
煤車 0
煤鐵 3
煤鐵礦 0
煥 652
煥然 13
煥然一新 49
煥然一新 49
煥爛 0
煥發 28
煥發起來 0
煦 656
煦仁孑義 0
煦和 0
煦日 1
煦煦 0
照 6436
照上 61
照上來 0
照上去 0
照下 21
照下來 3
照下去 0
照不到 5
照了 32
照些 1
照亮 122
照人 25
照付 3
照住 0
照佔 0
照來 7
照來照去 0
照例 63
照個 6
照像 76
照像機 23
照價 10
照價收買 52
照價收購 0
照價賠償 20
照准 10
照出 28
照出來 7
照出去 0
照到 36
照印 1
照唱 0
照喝 0
照單全收 0
照在 64
照妖鏡 9
照學 10
照完 9
照實 18
照寫 1
照寫不誤 0
照射 377
照射到 16
照射在 20
照常 112
照常上班 34
照常上課 11
照常營業 13
照常進行 10
照度 83
照後鏡 20
照得 141
照應 62
照抄 10
照抄一次 0
照抄一遍 0
照抄不誤 0
照拂 10
照排 6
照插 0
照插不誤 0
照換 13
照搬 2
照數 5
照料 159
照明 957
照明彈 9
照明燈 94
照明設備 136
照映 5
照會 85
照本宣科 18
照查 11
照查不誤 0
照樣 174
照樣做 2
照樣寫 0
照樣讀 0
照此 19
照此進行 0
照此類推 0
照派 0
照減 0
照準 5
照炒 0
照炒不誤 0
照片 7208
照玩 0
照玩不誤 2
照理 37
照理說 46
照生 2
照生不誤 0
照用 3
照用不誤 0
照發 6
照的 141
照直說 0
照相 939
照相乾片 0
照相機 333
照相法 1
照相版 5
照相簿 19
照相紙 1
照相術 4
照相製版 36
照相館 244
照看 7
照看不誤 2
照站 2
照章行事 2
照章辦理 1
照算 0
照算不誤 0
照管 23
照簽 1
照簽不誤 0
照繳 3
照耀 109
照考 44
照臨 0
照臨大地 0
照自 26
照舊 34
照舊是 1
照著 337
照表 20
照見 30
照設 0
照說 20
照說不誤 0
照請 1
照請不誤 0
照護 954
照讀 0
照起 5
照起來 0
照辦 72
照辦不誤 0
照進 12
照進來 11
照進去 0
照過 15
照過來 263
照過去 3
照道理說 1
照道理講 0
照錄 13
照鏡 32
照鏡子 36
照開 2
照關不誤 0
照面 21
照領 4
照顧 3666
照顧性 0
煨 65
煩 1003
煩亂 8
煩了 43
煩交 0
煩人 56
煩冗 0
煩勞 8
煩噪 3
煩囂 7
煩得 4
煩得很 0
煩心 33
煩忙 13
煩悶 54
煩惱 1041
煩惱障 1
煩憂 23
煩擾 20
煩文 0
煩死人 2
煩熱 0
煩瑣 27
煩碎 0
煩細 0
煩絮 0
煩言 0
煩請 720
煩躁 35
煩躁不安 10
煩雜 22
煩難 0
煪 6
煫 0
煬 63
煬 63
煬帝 16
煭 0
煮 854
煮個 3
煮出 24
煮到 10
煮得 52
煮成 23
煮掉 1
煮有 2
煮水 10
煮沸 97
煮法 29
煮熟 91
煮爛 10
煮的 52
煮練 7
煮繭 0
煮肉 3
煮茗 1
煮茗 1
煮菜 10
煮著 4
煮蛋 21
煮豆燃萁 0
煮起來 1
煮酒 10
煮食 31
煮飯 78
煮鶴焚琴 0
煯 0
煰 4
煲 53
煴 0
煴 0
煸 9
煹 0
煺 0
煻 110
煼 0
煽 16000
煽 209
煽亂 0
煽動 81
煽動 81
煽動力 2
煽動性 18
煽客 0
煽情 57
煽情主義 0
煽惑 12
煽火 0
煽爐子 0
煽起 2
煽起來 0
煾 0
煿 0
煿 0
熀 53
熁 153
熂 76
熃 0
熄 235
熄了 21
熄怒 0
熄掉 3
熄滅 109
熄火 102
熄燈 30
熄燈號 2
熅 159
熅 159
熆 33
熇 42
熈 0
熈 0
熉 101
熊 12756
熊克武 0
熊希齡 0
熊心豹子膽 1
熊成基 0
熊掌 18
熊熊 209
熊熊大火 2
熊皮 3
熊腰虎背 0
熊膽 9
熊蜂 2
熊貓 255
熊蹯 0
熋 0
熋 0
熌 0
熍 0
熎 0
熏 209
熏蒸 2
熏陶 1
熐 99
熑 0
熑 0
熒 247
熒光 4
熒光屏 1
熒光燈 1
熒光筆 1
熒光粉 1
熒惑 1
熒火 0
熒煌 1
熒熒 7
熒燎 1
熔 483
熔劑 4
熔化 38
熔化成 2
熔化為 0
熔化爐 1
熔合 6
熔岩 49
熔度 0
熔接 283
熔斷 10
熔斷器 1
熔於 4
熔池 1
熔焊 1
熔煉 12
熔煉爐 0
熔爐 24
熔礦爐 0
熔融 85
熔解 31
熔解熱 4
熔點 62
熗 63
熙 944
熙來攘往 22
熙攘 5
熙熙攘攘 13
熚 4
熛 20
熜 7
熜 7
熝 5
熞 10
熟 1133
熟 1133
熟了 44
熟人 44
熟到 7
熟化 12
熟妙 0
熟字 0
熟客 11
熟巧 2
熟得 163
熟念 0
熟思 0
熟思 0
熟悉 2014
熟悉起來 2
熟慮 6
熟手 3
熟料 14
熟炭 0
熟炭 0
熟爛 5
熟的 163
熟眠 0
熟睡 53
熟睡 53
熟睡中 13
熟睡中 13
熟知 167
熟知 167
熟石灰 7
熟石灰 7
熟稔 63
熟稔起來 1
熟絡 8
熟練 325
熟習 95
熟習 95
熟肉 6
熟能生巧 10
熟能生巧 10
熟苗 0
熟視 1
熟視著 0
熟記 19
熟認 1
熟諳 19
熟諳 19
熟諳 19
熟識 181
熟識 181
熟讀 49
熟起來 0
熟路 1
熟路 1
熟透 14
熟鐵 0
熟鐵 0
熟面孔 7
熟食 44
熟食 44
熟飯 0
熟點 5
熠 24
熠煜 0
熠熠 27
熠燿 0
熠爍 0
熠耀 0
熡 75
熢 0
熣 0
熣 0
熤 45
熥 3
熦 0
熧 8
熨 36
熨 36
熨平 3
熨斗 48
熨燙 6
熩 4
熪 6
熫 0
熫 0
熬 208
熬 208
熬不住 4
熬不過 8
熬出 3
熬出來 1
熬出頭 8
熬到 22
熬夜 145
熬夜苦讀 0
熬惱 0
熬日子 0
熬湯 17
熬煉 12
熬煎 3
熬粥 2
熬藥 1
熬藥湯 0
熬起 0
熬起來 0
熬透 0
熬過 32
熬過來 2
熬過去 2
熭 0
熭 0
熮 0
熮 0
熯 7
熰 12
熱 7875
熱中 85
熱中名利 0
熱乎乎 0
熱交換 217
熱值 22
熱傳導 38
熱內盧 0
熱出病來 0
熱切 129
熱刺刺 0
熱力 298
熱力四射 19
熱力學 251
熱力循環 3
熱力過程 0
熱功當量 2
熱加工 1
熱勁 5
熱化學 4
熱呼呼 27
熱和 18
熱喝 2
熱喪 0
熱固性 12
熱塑性 84
熱壓 58
熱天 31
熱天氣 5
熱季 7
熱學 29
熱定型 23
熱容量 2
熱導率 0
熱帶 697
熱帶地區 66
熱帶性 34
熱帶植物 34
熱帶氣候 8
熱帶雨林 134
熱帶雨林區 14
熱帶魚 100
熱平衡 15
熱度 139
熱得 21
熱得 202
熱心 1313
熱心腸 5
熱忱 429
熱忱心 0
熱性 58
熱情 1750
熱情奔放 13
熱情洋溢 26
熱愛 709
熱戀 80
熱成層 0
熱戰 34
熱效應 10
熱效率 13
熱敏電阻 11
熱敷 71
熱敷法 1
熱望 10
熱望者 0
熱核 11
熱核反應 1
熱機 33
熱毒素 0
熱氣 185
熱氣騰騰 2
熱水 295
熱水器 212
熱水爐 6
熱水瓶 29
熱河 72
熱河人 0
熱河省 3
熱河高原 2
熱流 143
熱浪 49
熱浪襲來 0
熱淚 54
熱淚盈眶 24
熱湯 10
熱源 40
熱潮 448
熱火 20
熱火朝天 0
熱烈 1149
熱烘烘 15
熱熱 28
熱熱烈烈 1
熱熱鬧鬧 23
熱燙 4
熱物 9
熱狂 2
熱狗 32
熱當量 0
熱病 19
熱的 202
熱絡 280
熱線 875
熱線電話 17
熱罨法 0
熱能 138
熱脹 5
熱脹冷縮 11
熱腸 2
熱膠 2
熱膨脹 23
熱處理 220
熱處理 220
熱血 154
熱血動物 0
熱血沸騰 12
熱衷 116
熱衷於 43
熱褲 2
熱褲裝 0
熱訊 0
熱誠 260
熱誠感人 1
熱賣 0
熱賣中 129
熱起來 5
熱身 181
熱身運動 15
熱軋 82
熱載體 0
熱輻射 24
熱辣辣 6
熱運動 3
熱過 5
熱那亞 40
熱量 548
熱量計 4
熱鍋上的螞蟻 5
熱鍵 1
熱門 3639
熱門音樂 43
熱障 0
熱電 18
熱電偶 17
熱電子 2
熱電學 0
熱電廠 0
熱電阻 1
熱電體 0
熱頭上 0
熱風 75
熱風爐 0
熱食 31
熱飲 25
熱騰騰 70
熱體 16
熱鬧 970
熱鬧滾滾 82
熱點 31
熲 18
熳 16
熵 23
熶 0
熷 0
熸 5
熹 175
熹微 5
熺 0
熻 0
熼 2
熽 5
熾 244
熾灼 0
熾烈 25
熾熱 72
熾熱體 0
熾燃 1
熾燄 0
熾盛 11
熿 3
燀 3
燁 668
燂 3
燃 853
燃放 10
燃放鞭炮 7
燃料 895
燃料工業 1
燃料庫 3
燃料油 38
燃氣輪機 6
燃氣輪機車 0
燃油 227
燃熾起來 0
燃燒 1168
燃燒劑 0
燃燒室 36
燃燒彈 0
燃燒性 14
燃眉 7
燃眉之急 11
燃素說 0
燃起 103
燃點 16
燄 177
燄 177
燄心 0
燄火 8
燅 6
燆 2
燇 2
燈 3802
燈下 51
燈光 692
燈光如晝 1
燈光效果 11
燈光設備 8
燈具 332
燈台 12
燈塔 194
燈市 4
燈座 173
燈彩 1
燈影 7
燈心 7
燈心絨 5
燈心草 0
燈會 70
燈期 0
燈架 9
燈柱 6
燈標 4
燈殼 2
燈油 4
燈泡 265
燈泡兒 0
燈泡兒 0
燈火 132
燈火管制 3
燈火通明 33
燈焰 2
燈燭 12
燈燭輝煌 1
燈盞 8
燈管 218
燈節 15
燈籠 136
燈籠褲 0
燈紅酒綠 18
燈絲 21
燈罩 379
燈臺 9
燈芯 14
燈花 8
燈蕊 5
燈號 136
燈蛾 88
燈蛾撲火 1
燈謎 15
燈頭 11
燈飾 390
燈黑 3
燉 130
燉上 1
燉了 0
燉煮 18
燉燉 1
燉爛 0
燉肉 2
燉菜 0
燉著 0
燉藥 0
燉鍋 6
燉雞 16
燉雞湯 1
燉魚 0
燊 59
燋 6
燋 6
燌 0
燌 0
燍 0
燎 16
燎 16
燎 16
燎原 37
燎原 37
燎原之勢 4
燎炬 0
燎髮 0
燏 2
燐 42
燐光 25
燐火 0
燐灰石 0
燐肥 0
燑 0
燒 3483
燒上 1
燒上來 1
燒上去 1
燒乾 0
燒了 50
燒仙草 1
燒來燒去 0
燒傷 168
燒光 14
燒出 49
燒出來 1
燒出去 0
燒化 3
燒去 5
燒壞 42
燒夷彈 1
燒得 62
燒成 240
燒掉 50
燒杯 45
燒杯架 0
燒死 50
燒毀 108
燒毛 11
燒水 12
燒火 7
燒灰 0
燒灼 58
燒炭 0
燒烤 187
燒焦 39
燒煉 0
燒煮 3
燒熱 222
燒燒 24
燒燒水 0
燒燒看 0
燒燬 18
燒片 0
燒瓶 24
燒瓶架 0
燒的 62
燒盡 28
燒石膏 0
燒磁 0
燒窯 19
燒紙 9
燒結 122
燒肉 1
燒茶 5
燒菜 8
燒賣 21
燒賣店 0
燒起 13
燒起來 7
燒酒 974
燒酒雞 18
燒錄 0
燒錄機 0
燒鍋 12
燒飯 11
燒餅 20
燒餅油條 16
燒香 38
燒香拜佛 5
燒香拜拜 4
燒香求神 0
燒鹼 6
燒點 3
燓 0
燓 0
燔 24
燕 29
燕 2959
燕京 17
燕京大學 7
燕人 2
燕兒 6
燕兒 6
燕國 15
燕國人 0
燕好 5
燕子 217
燕尾 60
燕尾旗 0
燕尾服 12
燕居 3
燕山 28
燕巢 571
燕巢幕上 0
燕燕 22
燕爾 0
燕爾新婚 1
燕王 42
燕瘦環肥 1
燕窩 57
燕窩湯 1
燕群 4
燕菜 0
燕語呢喃 1
燕遊 0
燕雀 20
燕雀安知鴻鵠志 1
燕食 0
燕麥 24
燕麥片 4
燕麥粥 1
燖 5
燗 0
燘 1
燙 545
燙來燙去 0
燙個 0
燙傷 430
燙出 3
燙在 1
燙得 27
燙手 26
燙手山芋 21
燙死 5
燙水 1
燙洗 1
燙燙 40
燙的 27
燙直 3
燙著 3
燙過 18
燙過來 0
燙過去 1
燙酒 4
燙金 72
燙青菜 1
燙頭 0
燙頭髮 1
燙髮 259
燙髮師 0
燙麵 1
燚 4
燛 3
燜 38
燜 38
燜煮 8
燜煮 8
燜燒 0
燜燒 0
燜燒鍋 0
燜燒鍋 0
營 18638
營伍 1
營利 573
營利事業 1222
營利事業所得稅 513
營利性 82
營力 8
營務官 0
營區 331
營口 25
營地 542
營壘 1
營妓 0
營帳 37
營建 4538
營建商 4
營建廠 0
營建業 325
營房 10
營房區 0
營收 0
營救 61
營救隊 0
營業 12079
營業員 186
營業執照 91
營業所得 2
營業毛利 17
營業稅 522
營業處 410
營業費用 37
營業部 98
營業項目 1882
營業額 1657
營火 100
營火晚會 27
營牆 2
營生 43
營生之道 0
營私 3
營私舞弊 5
營窟 0
營繕 573
營葬 0
營造 2722
營造商 20
營造尺 0
營造廠 102
營造廠商 39
營造業 517
營運 4278
營部 54
營部文書 0
營長 23
營隊 244
營養 3703
營養不良 89
營養價值 65
營養元素 3
營養午餐 0
營養品 48
營養器官 2
營養學 136
營養學分 0
營養學家 6
營養師 1
營養狀況 31
營養生長 12
營養素 165
營養繁殖 5
營養部 171
營養量 1
營養鹽 45
燠 386
燠 386
燠熱 13
燠熱 13
燡 2
燢 23
燣 0
燤 6
燥 254
燥濕 2
燥熱 14
燦 999
燦然 14
燦爛 449
燧 19
燧人氏 1
燧石 4
燧道 1
燨 37
燩 0
燪 0
燫 0
燫 0
燬 103
燭 276
燭光 105
燭光晚會 2
燭台 54
燭夜 0
燭心 3
燭淚 1
燭火 18
燭籠 0
燭臺 5
燭花 0
燭蕊 2
燮 81
燮理 0
燯 0
燰 1125
燱 19
燲 272
燳 0
燴 695
燴飯 8
燸 9
燹 4
燺 0
燺 0
燺 0
燻 137
燻燻 0
燻肉 2
燻黑 7
燻黑了 0
燼 34
燽 5
燾 47
燿 192
爀 0
爀 0
爀 0
爁 6
爂 8
爃 5
爄 0
爄 0
爅 5
爆 1965
爆冷門 18
爆出 73
爆滿 66
爆炒 8
爆炸 870
爆炸力 5
爆炸性 74
爆炸物 19
爆炸聲 18
爆玉米花 3
爆發 1131
爆破 109
爆破小組 1
爆破術 0
爆竹 64
爆笑 306
爆笑起來 0
爆米花 59
爆胎 1
爆裂 48
爆裂物 14
爆裂聲 5
爆香 1
爇 6
爇 6
爈 0
爈 0
爉 0
爊 7
爋 0
爋 0
爌 25
爌 25
爍 2915
爍金 3
爏 0
爐 1443
爐台 5
爐子 20
爐床 14
爐料 2
爐架 1
爐渣 21
爐溫 4
爐火 27
爐火純青 12
爐灰 8
爐灶 19
爐炭 0
爐管 17
爐膛 0
爐臺 0
爐襯 0
爐門 3
爐頂 1
爑 0
爒 0
爒 0
爓 4
爔 8
爕 0
爕 0
爖 0
爗 0
爗 0
爙 1
爙 1
爚 1
爚 1
爛 2045
爛好人 2
爛得 232
爛成 0
爛掉 14
爛攤子 21
爛料 0
爛泥 20
爛泥巴 4
爛漫 26
爛熟 3
爛爛 31
爛的 232
爛糊 0
爛紙 1
爛貨 25
爛賬 0
爛透 9
爛醉 5
爛醉如泥 7
爜 0
爝 5
爞 8
爞 8
爟 22
爠 0
爠 0
爡 0
爡 0
爢 6
爣 120
爤 0
爥 0
爦 6
爧 69
爨 19
爨婦 0
爨室 0
爨弄 1
爨火 0
爩 40
爪 476
爪 476
爪印 0
爪印 0
爪哇 386
爪哇人 8
爪子 113
爪子 113
爪尖 5
爪尖 5
爪牙 17
爪甲點金 1
爪痕 5
爪痕 5
爫 0
爫 0
爬 1004
爬上 171
爬上來 13
爬上去 31
爬下 10
爬下來 2
爬下去 2
爬不動 3
爬不起來 15
爬來 2
爬來爬去 9
爬入 3
爬出 21
爬出來 10
爬出去 1
爬到 103
爬升 94
爬在 13
爬山 194
爬山虎 12
爬山靴 0
爬得 51
爬樹 24
爬滿 15
爬滿了 8
爬牆 12
爬牆而過 0
爬牆虎 1
爬犁 0
爬的 50
爬繩 0
爬著 14
爬著 14
爬蟲 44
爬蟲類 105
爬行 70
爬行動物 21
爬走 1
爬起 44
爬起來 48
爬開 0
爬高 12
爭 2889
爭了 16
爭了 16
爭先 35
爭先後 0
爭先恐後 49
爭先發言 0
爭光 26
爭光榮 2
爭出 10
爭分奪秒 0
爭利 5
爭利益 1
爭功 7
爭功勞 0
爭功諉過 9
爭勝 18
爭取 3024
爭取到 136
爭取時間 16
爭口氣 4
爭吃 0
爭名 4
爭名利 2
爭名奪利 4
爭吵 88
爭吵不休 9
爭嘴 0
爭執 292
爭執性 0
爭多論少 0
爭奇鬥艷 10
爭奇鬥豔 4
爭奪 226
爭奪戰 108
爭妍鬥勝 0
爭妍鬥豔 0
爭寵 17
爭強 8
爭強好勝 0
爭得 32
爭得 32
爭持 3
爭持不下 1
爭搏 0
爭權 14
爭權力 0
爭權奪利 18
爭氣 20
爭求 1
爭產 6
爭相 193
爭相羅致 0
爭端 141
爭艷 1
爭著 24
爭著 24
爭訟 1
爭論 278
爭議 2098
爭購 2
爭起 1
爭辯 106
爭逐 13
爭過 3
爭鋒 56
爭長 6
爭長競短 0
爭長論短 0
爭雄 39
爭霸 343
爭霸戰 100
爭霸賽 18
爭風吃醋 11
爭鬥 44
爭鬥不休 0
爭鳴 13
爮 0
爯 0
爰 903
爲 0
爲 0
爴 0
爵 1
爵 1583
爵位 10
爵士 992
爵士樂 152
爵士樂團 10
爵士舞 0
爵祿 4
父 14
父 1475
父事 2
父仇不共戴天 1
父仇不共戴天 1
父兄 19
父兼母職 1
父命 6
父命難從 0
父命難違 0
父嚴母慈 0
父執 4
父執輩 13
父女 41
父女之情 0
父女倆 2
父女情 5
父女關係 4
父子 226
父子之情 2
父子倆 14
父子倆 14
父子天性 0
父子情 7
父子相傳 0
父子關係 7
父愛 5
父慈子孝 1
父本 3
父權 160
父權制 2
父母 3719
父母官 6
父母心 24
父母親 457
父王 27
父系 29
父系親屬 2
父老 66
父老兄弟 21
父職 4
父蔭 3
父親 2244
父親節 64
父輩 8
爸 1
爸 1458
爸爸 1745
爸爸 1745
爹 1
爹 235
爹地 46
爹娘 33
爹爹 62
爹爹 62
爺 645
爺們 24
爺們兒 1
爺兒 4
爺兒 4
爺兒倆 0
爺兒倆 0
爺兒們 1
爺兒們 1
爺娘 0
爺爺 380
爻 64
爻象 0
爻辭 2
爼 0
爽 551
爽利 1
爽口 40
爽口糖 1
爽快 73
爽性 1
爽朗 78
爽然若失 0
爽直 1
爽約 20
爽脆 3
爽蕩 0
爽言 0
爽身 2
爽身粉 12
爾 9353
爾來 21
爾後 356
爾時 58
爾格 12
爾爾 6
爾等 23
爾虞我詐 19
爾詐我虞 1
爾雅 198
爾雅有言 0
爿 12
爿 12
牀 0
牁 277
牂 11
牃 0
牄 47
牅 0
牅 0
牆 1429
牆上 307
牆垣 4
牆基 6
牆壁 356
牆壁上 90
牆宇 0
牆根 3
牆毯 0
牆籬 1
牆腳 27
牆角 85
牆跟 0
牆面 210
牆頭 12
牆頭草 1
片 21317
片中 485
片假名 5
片價 1
片兒 0
片兒 0
片刻 196
片刻之間 9
片刻間 14
片名 1013
片商 388
片地 5
片場 37
片場 37
片子 368
片尾 127
片尾曲 0
片斷 74
片時 67
片段 439
片片 80
片狀 57
片甲不留 4
片目 3
片租 2
片約 29
片紙隻字 1
片言 4
片言隻語 1
片語 197
片語手冊 0
片酬 18
片長 45
片面 152
片面之詞 4
片面性 5
片面最惠國 1
片頭 150
片頭曲 0
片麻岩 19
版 37733
版刻 2
版圖 379
版心 58
版本 8831
版本學 8
版模 5
版權 2696
版權所有 7582
版權所有日期 0
版權法 2
版權頁 0
版次 243
版次說明 0
版畫 384
版畫集 3
版稅 31
版築 1
版面 1570
牉 151
牊 166
牋 4
牌 4823
牌九 6
牌位 47
牌價 65
牌匾 5
牌友 14
牌坊 72
牌子 163
牌局 20
牌板 1
牌桌 17
牌樓 43
牌照 234
牌照稅 234
牌號 59
牎 0
牏 95
牐 0
牑 0
牑 0
牒 80
牒文 1
牓 51
牔 0
牕 0
牖 32
牖中窺日 0
牖戶 0
牗 0
牘 33
牙 2627
牙刷 172
牙刷套 0
牙刷毛 0
牙印 1
牙周病 204
牙垢 4
牙床 12
牙慧 1
牙根 70
牙牌 5
牙牙 9
牙牙學語 9
牙疳 1
牙疼 11
牙痛 47
牙癢癢 26
牙白口清 0
牙祭 2
牙科 1223
牙科診所 193
牙科醫生 6
牙簽犀軸 1
牙籤 49
牙籤玉軸 1
牙籤盒 0
牙籤錦軸 1
牙粉 7
牙結石 0
牙線 79
牙縫 25
牙縫裡 0
牙膏 102
牙膏盒 0
牙膏管 1
牙菌斑 0
牙蟲 0
牙買加 96
牙質 0
牙輪 2
牙醫 1235
牙醫公會 0
牙醫師 563
牙關 16
牙音 0
牙髓 0
牙齒 881
牙齒痛 6
牙齦 114
牙齦炎 6
牚 4
牚 4
牛 3217
牛乳 108
牛五花 1
牛仔 284
牛仔 284
牛仔裝 10
牛仔褲 175
牛僧孺 0
牛刀 4
牛刀小試 16
牛勁 0
牛圈 0
牛奶 750
牛奶瓶 3
牛奶箱 0
牛奶糖 6
牛家 8
牛家莊 2
牛小排 1
牛尾 6
牛尾湯 2
牛山濯濯 0
牛年 40
牛排 371
牛排肉 1
牛排餐 2
牛排館 89
牛李黨爭 0
牛棚 10
牛欄 15
牛步 26
牛毛 24
牛毛細雨 0
牛毛雨 0
牛油 33
牛津 189
牛津大學 125
牛犢 25
牛犢子 0
牛疫 0
牛痘 9
牛痘苗 0
牛瘟 9
牛皮 133
牛皮帶 0
牛皮癬 18
牛皮紙 37
牛皮紙袋 0
牛皮菜 0
牛磺酸 1
牛筋 8
牛糞 17
牛羊 63
牛羊馬 0
牛群 27
牛耳 86
牛肉 434
牛肉乾 10
牛肉場 3
牛肉場 3
牛肉店 5
牛肉排 0
牛肉湯 22
牛肉湯麵 2
牛肉麵 198
牛肚 8
牛肚 8
牛背 9
牛背上 4
牛脾氣 10
牛腩 13
牛腩飯 7
牛腰 2
牛腱 8
牛腳 1
牛舌 5
牛舌餅 4
牛舍 9
牛蒡 0
牛虻 0
牛蛙 13
牛衣對泣 7
牛角 86
牛車 65
牛軛湖 0
牛郎 89
牛郎星 9
牛郎織女 62
牛酪 1
牛酪 1
牛隻 68
牛頓 317
牛頓定律 3
牛頓流體 6
牛頭 79
牛頭不對馬嘴 6
牛頭馬面 10
牛飲 4
牛馬 15
牛馬不如 0
牛馬生活 0
牛驥同一皁 0
牛驥同一皁 0
牛骨 3
牛鬼蛇神 6
牛鼻子 0
牝 46
牝牛 2
牝雞司晨 1
牝馬 0
牞 6
牟 239
牟利 46
牟取 17
牟尼佛 9
牠 2412
牠 1
牡 249
牡 249
牡丹 308
牡丹亭 1
牡丹江 33
牡丹花 7
牡牛 6
牡羊 1
牡羊座 371
牡蠣 107
牡馬 1
牡鹿 0
牢 301
牢不可破 15
牢固 45
牢房 21
牢牢 68
牢獄 17
牢獄之災 10
牢籠 18
牢記 75
牢記在心 12
牢門 3
牢靠 17
牢頭 10
牢頭禁子 1
牢騷 40
牣 4
牥 0
牦 0
牧 6241
牧人 56
牧區 32
牧地 19
牧場 43
牧場 430
牧女 1
牧守 0
牧工 6
牧師 1134
牧業 156
牧歌 12
牧民 24
牧牛 20
牧牛區 0
牧牛業 0
牧畜 1
牧童 80
牧笛 5
牧羊 91
牧羊人 175
牧羊犬 66
牧群 4
牧草 98
牧草地 3
牧郎 1
牧馬 15
牧馬中原 0
牨 0
物 13897
物主 24
物事 15
物以稀為貴 10
物以類聚 14
物件 3326
物件導向 0
物候 3
物價 766
物價指數 280
物力 153
物力維艱 1
物化 128
物化勞動 0
物品 2906
物態 4
物慾 20
物慾橫流 3
物我兩忘 4
物換星移 21
物料 1854
物極必反 14
物歸原主 4
物流 0
物理 5811
物理化學 224
物理學 369
物理學家 90
物理家 0
物理性質 63
物理現象 36
物理系 752
物理課 5
物理變化 4
物產 126
物產豐富 6
物盡其用 24
物種 516
物種來由 1
物競天擇 18
物美價廉 69
物色 40
物色人才 0
物色人選 1
物色到 2
物証 1
物語 953
物論 15
物證 31
物議 3
物象 24
物資 293
物資局 21
物質 3987
物質上 37
物質不滅 1
物質不滅定律 1
物質享受 20
物質力量 1
物質文明 33
物質波 0
物質波 0
物質獎勵 1
物質生活 64
物質生產 6
物質生產部門 0
物質財富 0
物質需要 1
物超所值 0
物鏡 38
物類 156
物體 789
牪 6
牫 0
牬 113
牭 0
牮 53
牯 737
牯嶺 41
牰 81
牱 0
牲 619
牲口 12
牲犢 0
牲畜 65
牲禮 9
牳 132
牴 6629
牴牾 0
牴觸 209
牶 14
牷 345
牸 5
特 20129
特別行政區 100
特令 2
特以 66
特任 26
特任官 5
特佳 51
特使 83
特來 56
特例 57
特價 2314
特價品 151
特優 246
特優獎 33
特免 0
特准 17
特出 29
特刊 431
特別 14517
特別公積 0
特別公積金 0
特別座 0
特別是 1528
特別法 55
特別獎 128
特制 79
特務 89
特務組織 2
特區 2304
特命 14
特地 525
特地來 18
特地到 18
特地去 17
特地回 0
特報 240
特大 136
特大獎 0
特大號 21
特官 1
特定 3210
特寫 1598
特寫鏡頭 6
特工 68
特已 10
特強 24
特征 8
特徵 1918
特徵多項式 0
特快 21
特快號 3
特快車 31
特急 25
特性 4894
特惠國 1
特惠關稅 0
特意 71
特技 222
特技團 4
特技師 0
特指 10
特支 1
特支費 8
特攻隊 107
特效 572
特效藥 57
特教 3876
特教組 17
特斯拉 1
特於 196
特早 3
特明 5
特明體 0
特有 831
特有的 370
特案 41
特案處理 2
特案處理 2
特權 282
特權分子 2
特權階層 1
特權階級 4
特此 266
特此通知 1
特殊 8953
特殊份子 1
特殊份子 1
特殊化 8
特殊性 114
特殊教育 4535
特洛伊 36
特派 22
特派員 272
特派記者 369
特為 43
特獎 94
特產 482
特產地 0
特產店 8
特由 25
特異 290
特異性 98
特種 1286
特種兵 1
特種行業 1
特種部隊 73
特稱 9
特稿 0
特立獨行 28
特等 45
特等獎 12
特等病房 0
特等艙 0
特約 1307
特約商店 247
特約稿 0
特約記者 23
特約診所 19
特約醫生 0
特約醫院 131
特級 138
特級上將 3
特級品 5
特考 477
特與 77
特色 7984
特號 5
特裡 0
特製 157
特製品 1
特要 7
特訂 248
特訓 1
特許 271
特許權 9
特許狀 0
特許證 2
特賜 3
特賜 3
特賣 514
特賣品 6
特賣會 1
特質 1360
特赦 41
特輯 424
特造 1
特遣 5
特遣艦隊 0
特遣隊 150
特選 101
特邀 81
特里 26
特長 93
特集 90
特電 19
特餐 67
特點 1119
牺 0
牻 1
牼 1
牽 22500
牽一髮而動全身 6
牽一髮而動全身 6
牽上 6
牽上來 0
牽上去 0
牽下 0
牽下來 0
牽下去 0
牽了 10
牽伸 12
牽住 0
牽來 1
牽入 6
牽出 4
牽出來 1
牽出去 0
牽到 18
牽制 117
牽制行動 1
牽動 214
牽去 3
牽合 1
牽回 2
牽回來 0
牽回去 0
牽在 1
牽好 2
牽完 0
牽引 227
牽引力 14
牽引機 12
牽引發電機 0
牽引電動機 0
牽強 45
牽強附會 6
牽得 0
牽念 3
牽手 248
牽扯 114
牽拉 5
牽拉著 0
牽掛 98
牽掛著 10
牽掣 1
牽涉 264
牽涉到 376
牽牛 36
牽牛星 2
牽牛花 38
牽累 14
牽累到 1
牽絆 47
牽線 45
牽線人 0
牽繫 22
牽繫 22
牽纏 3
牽腸掛肚 6
牽著 106
牽著鼻子走 20
牽走 3
牽起 30
牽起來 0
牽連 73
牽連到 11
牽進 0
牽進來 0
牽進去 0
牽過 1
牽過來 0
牽過去 0
牽開 0
牾 9
牿 5
犀 76
犀利 98
犀牛 84
犀牛皮 0
犀牛角 18
犀甲 0
犀甲利兵 0
犀角 7
犀鳥 13
犁 194
犁刀 0
犁地 0
犁庭掃穴 0
犁板 0
犁柄 0
犁牛 1
犁田 13
犁鋤 0
犁鏵 0
犂 0
犃 0
犄 11
犄角 0
犅 3
犆 4
犇 0
犇 0
犈 3
犉 8
犋 10
犌 20
犍 92
犎 10
犏 0
犐 31
犑 81
犒 220
犒勞 3
犒賞 31
犒賞法 0
犒軍 8
犓 77
犔 0
犕 100
犖 301
犖犖 9
犗 60
犘 31
犙 0
犚 17
犛 38
犛牛 2
犜 0
犝 2
犞 3
犟 0
犡 24
犢 96
犢子 0
犢牛 8
犢車 0
犢鼻褌 0
犣 6
犤 6
犥 17
犦 6
犧 71
犧牲 815
犧牲品 30
犧牲掉 15
犧牲生命 9
犧牲者 45
犨 123
犩 3
犩 3
犪 6
犫 0
犬 1917
犬不夜吠 0
犬儒 8
犬儒學派 0
犬子 7
犬戎 2
犬牙 4
犬牙交錯 1
犬牙相錯 1
犬羊之質 0
犬馬 4
犬馬之勞 1
犬馬之養 0
犬齒 19
犭 0
犮 0
犯 1931
犯上 13
犯上作亂 0
犯下 117
犯不上 0
犯不著 13
犯了 178
犯事 4
犯人 125
犯台 76
犯土 0
犯境 1
犯太歲 1
犯忌 0
犯意 13
犯戒 8
犯桃花 1
犯案 95
犯法 66
犯潮 0
犯瀆 0
犯界 0
犯疑 1
犯病 3
犯禁 3
犯節氣 0
犯紀 4
犯罪 2124
犯罪心理 3
犯罪心理學 10
犯罪者 47
犯罪行為 101
犯規 141
犯規者 0
犯諱 0
犯賤 2124
犯錯 156
犯錯誤 16
犰 86
犱 0
犲 0
犳 0
犴 17
犴 17
犵 6
状 0
犹 0
犺 4
犻 0
犼 0
犽 7
犾 0
犿 7
狀 3431
狀元 205
狀如 26
狀子 3
狀態 2948
狀態副詞 0
狀態參數 0
狀態圖 2
狀態字 0
狀態表 3
狀況 7579
狀況碼 0
狀況良好 52
狀紙 21
狀詞 0
狀貌 2
狁 5
狂 2203
狂亂 48
狂人 197
狂人日記 4
狂傲 23
狂吠 9
狂吹 0
狂吼 8
狂呼 2
狂喜 52
狂嘯 9
狂士 0
狂奔 64
狂妄 42
狂妄之談 0
狂妄自大 12
狂客 4
狂徒 52
狂怒 12
狂恣 0
狂悖 2
狂想 109
狂想曲 122
狂態 1
狂放 23
狂暴 22
狂歡 188
狂歡節 27
狂歡起來 0
狂漢 1
狂潮 57
狂瀾 15
狂熱 267
狂熱性 0
狂燥 1
狂牛症 0
狂犬病 138
狂狷 5
狂笑 74
狂笑聲 3
狂言 9
狂詩 19
狂話 0
狂語 3
狂跌 10
狂跑 0
狂轟濫炸 0
狂野 85
狂風 101
狂風巨浪 3
狂風怒吼 1
狂風怒號 0
狂風暴雨 45
狂飆 203
狂飲 5
狂鬧 0
狃 5
狃於成見 0
狄 851
狄斯可 6
狄斯奈 42
狄更斯 10
狄青 28
狅 4
狆 6
狇 0
狉 7
狉狉 0
狊 152
狋 6
狌 12
狍 0
狎 3100
狎侮 0
狎妓 1
狎客 2
狎弄 0
狎昵 0
狎昵 0
狎暱 0
狎玩 2
狎翫 0
狎近 0
狎邪 1
狏 0
狐 1000
狐仙 32
狐假虎威 5
狐媚 3
狐狸 204
狐狸尾巴 3
狐狸尾巴 3
狐狸尾巴 3
狐狸尾巴 3
狐狸狗 3
狐狸精 11
狐猴 40
狐疑 38
狐疑不決 0
狐皮 9
狐群狗黨 15
狐臭 42
狐臭味 0
狐貍 34
狐騷 0
狑 9
狒 11
狒狒 24
狓 0
狔 7
狕 0
狖 3
狗 6627
狗仔隊 0
狗仗人勢 1
狗占馬坑 0
狗叫 19
狗叫聲 4
狗吠 23
狗命 6
狗咬 42
狗咬呂洞賓 0
狗咬狗 5
狗嘴吐不出象牙 1
狗嘴裡 0
狗尾續貂 0
狗屁 26
狗屁不通 5
狗屎 43
狗年 5
狗店 6
狗彘 1
狗急跳牆 12
狗才 6
狗拿耗子 1
狗橇 0
狗熊 20
狗爬 2
狗爬式 0
狗男女 297
狗皮 4
狗眼 2
狗眼看人低 1
狗窩 126
狗竇 0
狗肉 34
狗肉湯 2
狗腳 2
狗腿 78
狗腿子 2
狗苟蠅營 0
狗蝨 0
狗蠅 0
狗血 32
狗血噴頭 0
狗血淋頭 8
狗輩 0
狗頭 6
狗頭軍師 1
狗食 15
狘 2
狙 165
狙公 1
狙擊 59
狙擊手 15
狙詐 0
狚 22
狛 0
狜 1
狝 0
狟 1237
狠 1005
狠下 9
狠下心來 12
狠了 6
狠到 1
狠命 6
狠將 4
狠得 2
狠心 88
狠戾 0
狠抓 1
狠毒 31
狠狠 201
狠起來 0
狡 1135
狡兔 8
狡兔三窟 4
狡徒 0
狡滑 7
狡猾 46
狡獪 4
狡童 4
狡脫 0
狡計 0
狡詐 16
狡譎 0
狡賴 1
狡辯 22
狡飾 0
狡黠 11
狢 0
狢 0
狢 0
狢 0
狢 0
狣 58
狤 94
狥 0
狥 0
狦 115
狧 0
狨 286
狩 541
狩獵 243
狩獵法 0
狪 129
狫 122
独 0
狭 0
狳 87
狴 267
狵 0
狶 58
狷 18
狷介 2
狷忿 0
狷急 0
狸 86
狸貓 13
狹 286
狹促 3
狹小 135
狹山 0
狹巷 2
狹心症 31
狹窄 331
狹義 97
狹義地說 1
狹義地說 1
狹谷 4
狹路 6
狹路相逢 11
狹長 92
狹陋 1
狹隘 92
狺 182
狻 128
狻猊 20
狼 1544
狼人 43
狼吞 1
狼吞虎咽 1
狼吞虎嚥 6
狼嗥 4
狼嗥鬼叫 0
狼嚎 4
狼嚎鬼叫 0
狼子野心 1
狼心 4
狼心狗肺 4
狼毫 10
狼毫筆 0
狼煙 15
狼煙四起 0
狼牙棒 4
狼犬 26
狼犺 0
狼狗 20
狼狽 57
狼狽不堪 9
狼狽為奸 11
狼狽相 0
狼瘡 182
狼籍 5
狼藉 10
狼貪 0
狽 3
狾 188
狿 10
猀 223
猁 1330
猂 0
猄 0
猅 0
猆 0
猇 14
猈 16
猈 16
猉 0
猊 15
猋 4
猌 2
猍 0
猎 0
猏 22
猐 0
猑 14
猒 4
猓 7
猔 0
猕 0
猕 0
猖 23
猖狂 34
猖狂進攻 0
猖獗 93
猖獗一時 0
猖獗一時 0
猗 43
猘 11
猙 6
猙獰 29
猚 0
猛 951
猛力 30
猛力一擊 1
猛力一擊 1
猛加 5
猛勁 1
猛勁 1
猛勇 0
猛升 0
猛吃 2
猛增 0
猛士 1
猛子 2
猛將 19
猛建 1
猛拍 3
猛推 5
猛撞 11
猛撲 2
猛擊 6
猛攻 18
猛政 0
猛沖 1
猛減 0
猛漢 0
猛火 7
猛烈 133
猛烈起來 0
猛然 147
猛犬 3
猛猛然 0
猛獸 33
猛男 0
猛禽 57
猛練 2
猛蓋 0
猛虎 42
猛衝 4
猛襲 1
猛跳 0
猛載 0
猛追 5
猛進 7
猛酒 0
猛醒 1
猛降 0
猛龍 16
猛龍過江 3
猜 1222
猜不著 5
猜不透 41
猜中 32
猜來猜去 2
猜出 51
猜出來 3
猜到 90
猜好 4
猜嫌 0
猜字謎 4
猜度 5
猜得 41
猜得出 4
猜得到 10
猜忌 49
猜想 127
猜想到 1
猜懼 0
猜拳 53
猜枚 0
猜枚行令 1
猜法 0
猜測 232
猜測出 1
猜測到 4
猜測法 0
猜燈謎 5
猜猜 193
猜猜看 126
猜疑 55
猜疑心 4
猜的 41
猜謎 159
猜謎底 0
猜謎題 0
猜起 0
猜起來 0
猝 87
猝不及防 8
猝死 0
猝然 13
猞 8
猢 8
猢 8
猢猴兒 0
猢猴兒 0
猢猻 1
猢猻兒 0
猢猻兒 0
猣 6
猤 0
猥 24
猥瑣 5
猥褻 356
猥褻行為 11
猥賤 0
猦 5
猧 8
猨 0
猩 72
猩猩 163
猩紅 20
猩紅熱 53
猪 0
猫 0
猬 0
猭 5
献 0
猯 0
猰 6
猱 8
猲 10
猲 10
猳 2
猴 1137
猴兒 9
猴兒 9
猴囡仔 0
猴子 298
猴年 9
猴急 12
猴戲 5
猴猻 0
猴王 90
猴瘊子 0
猴皮 0
猴精 1
猴頭 7
猴頭 7
猴類 29
猵 10
猶 828
猶他 64
猶他州 31
猶可 13
猶大 160
猶太 276
猶太人 362
猶太復國主義 1
猶太復國運動 1
猶太教 46
猶如 432
猶有可為 3
猶然 6
猶疑 36
猶疑不決 6
猶自 9
猶若 3
猶豫 329
猶豫不決 53
猶豫起來 2
猷 3929
猸 0
猺 20
猻 42
猼 28
猽 0
猾 49
猾賊 0
猾頭 0
猿 123
猿人 6
猿啼 2
猿嘯 0
猿猴 85
猿聲 2
猿臂 3
猿臂 3
猿類 2
獀 19
獁 0
獂 7
獃 317
獃住 0
獃子 0
獃氣 0
獃頭獃腦 0
獄 572
獄具 0
獄卒 13
獄史 0
獄司 0
獄吏 1
獄囚 0
獄官 2
獄訟 0
獄門 10
獅 1413
獅吼 6
獅子 760
獅子 760
獅子吼 17
獅子吼 17
獅子山 11
獅子山 11
獅子座 1113
獅子座 1113
獅子會 292
獅子會 292
獅子狗 1
獅子狗 1
獅子舞 0
獅子頭 25
獅子頭 25
獅子鼻 2
獅心王 2
獅王 74
獅身人面像 7
獅頭 39
獅頭山 66
獆 0
獆 0
獉 0
獊 0
獋 0
獋 0
獌 0
獍 0
獎 5331
獎券 22
獎券行 1
獎助 1083
獎助金 187
獎勉 1
獎勵 2651
獎勵制度 32
獎勵品 1
獎勵性 1
獎勵法 2
獎品 655
獎售 0
獎學金 2701
獎懲 392
獎懲條例 0
獎懲法 0
獎掖 7
獎杯 47
獎牌 217
獎狀 238
獎盃 65
獎章 439
獎許 0
獎賞 71
獎金 1594
獎項 0
獎飾 0
獏 0
獐 199
獐子 0
獐頭鼠目 5
獑 51
獒 19
獓 0
獔 0
獔 0
獕 0
獖 0
獖 0
獗 585
獘 99
獙 5
獚 4
獛 8
獜 0
獝 9
獞 3
獟 10
獠 19
獠牙 12
獠面 0
獡 3
獢 278
獥 3
獦 3
獧 24
獨 3217
獨一無二 225
獨享 215
獨享其成 0
獨人 3
獨佔 112
獨佔事業 2
獨佔市場 9
獨佔市場 9
獨佔性 10
獨佔權 3
獨來獨往 15
獨個兒 2
獨個兒 2
獨具 147
獨具一格 5
獨具一格 5
獨具匠心 1
獨具慧眼 11
獨到 159
獨到之處 22
獨創 93
獨創一格 3
獨創一格 3
獨創性 29
獨力 90
獨力經營 0
獨占 232
獨占性 14
獨占鰲頭 38
獨占鼇頭 5
獨厚 0
獨吞 8
獨唱 87
獨唱曲 3
獨善 1
獨善其身 25
獨坐 31
獨大 49
獨夫 3
獨奏 386
獨奏曲 14
獨子 47
獨守 3
獨守空閨 4
獨家 584
獨家報導 85
獨家新聞 12
獨居 133
獨幕劇 3
獨往獨來 0
獨得 64
獨持異議 0
獨挑 0
獨挑大樑 6
獨掌 1
獨掌大權 0
獨排眾議 3
獨攬 6
獨攬大權 3
獨斷 23
獨斷專行 1
獨斷獨行 8
獨有 143
獨木 6
獨木不成林 1
獨木橋 5
獨木舟 76
獨木船 0
獨木難支 0
獨桅 0
獨棟 61
獨樹一幟 22
獨樹一幟 22
獨樹一格 48
獨樹一格 48
獨步 76
獨無 7
獨照 36
獨特 1414
獨特性 69
獨獨 38
獨生 21
獨生女 15
獨生子 52
獨當 1
獨當一面 39
獨當一面 39
獨白 57
獨眼 19
獨眼龍 25
獨立 5088
獨立國 10
獨立宣言 17
獨立思想 2
獨立思考 77
獨立性 80
獨立核算 16
獨立王國 4
獨立精神 9
獨立自主 154
獨立運動 100
獨缺 21
獨腳 3
獨腳戲 16
獨臂 5
獨臂 5
獨臂人 2
獨臂人 2
獨臂刀 0
獨臂刀 0
獨自 608
獨處 95
獨行 80
獨行俠 26
獨行其是 0
獨裁 119
獨裁政權 10
獨裁者 53
獨覺 1
獨角 24
獨角戲 32
獨角獸 15
獨角龍 0
獨資 152
獨資經營 11
獨走 6
獨走江湖 0
獨身 30
獨身主義 2
獨身生活 1
獨輪 1
獨輪車 5
獨酌 15
獨門 61
獨門功夫 1
獨門獨戶 0
獨門獨院 3
獨門絕活 2
獨間 0
獨霸 42
獨霸一方 2
獨霸一方 2
獨霸全球 3
獨飲 11
獩 6
獪 19
獫 2
獬 10
獮 84
獯 29
獰 105
獰笑 13
獰鬼 0
獱 0
獲 0
獲 5847
獲准 426
獲利 2667
獲利率 68
獲到 2
獲勝 196
獲取 553
獲得 8047
獲悉 88
獲救 88
獲獎 399
獲獎人 15
獲獎率 9
獲獎者 18
獲益 162
獲知 217
獲罪 6
獲致 280
獲贈 0
獲赦 0
獲選 301
獲邀 0
獲釋 62
獲頒 1296
獲鹿 0
獳 319
獴 0
獵 687
獵了 0
獵人 572
獵到 1
獵取 24
獵場 8
獵場 8
獵奇 4
獵戶 20
獵戶座 53
獵戶星座 8
獵手 2
獵槍 42
獵殺 163
獵物 163
獵犬 74
獵狗 12
獵獅 0
獵獲 4
獵獲物 2
獵艷 0
獵裝 0
獵豔 15
獵豹 16
獵隊 2
獵食 27
獵鳥 1
獵鷹 25
獵鹿 2
獵鹿人 6
獶 7
獷 12
獷悍 0
獸 899
獸力 1
獸化 1
獸圈 0
獸媒花 0
獸心 0
獸性 16
獸性大發 1
獸慾 2
獸檻 0
獸王 3
獸環 2
獸皮 17
獸行 6
獸行 6
獸醫 1239
獸醫系 162
獸醫院 53
獸類 8
獸骨 4
獹 0
獺 439
獻 1248
獻上 162
獻了 5
獻俘 2
獻出 62
獻媚 4
獻寶 8
獻帝 5
獻技 12
獻拙 0
獻旗 0
獻曝 0
獻殷勤 9
獻物 0
獻禮 69
獻策 28
獻給 276
獻花 46
獻藝 31
獻言 4
獻計 8
獻詞 8
獻詩 32
獻身 118
獻辭 1
獻酒 0
獻醜 14
獻金 100
獼 102
獼猴 168
獽 51
獾 16
獿 34
玀 10
玁 18
玂 20
玃 5
玄 2048
玄之又玄 5
玄奘 121
玄奘 121
玄奧 7
玄妙 36
玄孫 6
玄學 164
玄學家 1
玄學者 0
玄教 1
玄機 114
玄機暗藏 0
玄武 0
玄武土 0
玄武岩 93
玄武湖 9
玄武門 1
玄狐 0
玄理 0
玄疑 10
玄石 0
玄秘 1414
玄素學 0
玄色 4
玄虛 8
玄談 2
玄門 51
玄關 98
玄青 0
玄鳥 1
玄黃 3
玄鼠 1
玅 63
玆 106
率 0
率 1014
率 10146
率先 586
率兵 17
率同 48
率團 200
率師 0
率性 39
率意 2
率然 5
率爾 9
率爾操觚 0
率獸食人 0
率由 114
率由舊章 1
率直 33
率真 30
率部 3
率隊 9
率領 446
玈 2
玉 339
玉不琢不成器 0
玉井 359
玉人 10
玉佛 13
玉佩 20
玉兔 11
玉匠 0
玉器 112
玉塊 1
玉壺 11
玉女 73
玉女明星 0
玉容 22
玉展 1
玉山 2424
玉峰 105
玉帛 0
玉帝 18
玉成 172
玉成其事 0
玉手 26
玉杯 3
玉林 27
玉樹 59
玉樹臨風 17
玉池 6
玉液瓊漿 0
玉液瓊漿 0
玉溪 12
玉漿 0
玉潔冰清 4
玉潤珠圓 0
玉照 138
玉爪 0
玉牒 1
玉珮 46
玉環 26
玉璽 27
玉瓶 5
玉皇 32
玉皇大帝 34
玉盤 4
玉石 172
玉石俱焚 17
玉石眼 0
玉碎 4
玉筍 0
玉簪 17
玉簪 17
玉簫 3
玉米 611
玉米湯 9
玉米濃湯 27
玉米田 13
玉米粥 1
玉米花 3
玉米花兒 0
玉米花兒 0
玉米螟 10
玉米面 0
玉米餅 7
玉聲 2
玉肌 0
玉腿 11
玉臂 22
玉臂 22
玉茭 0
玉蘭 120
玉蘭片 0
玉蘭花 14
玉蛄瓊漿 0
玉蜀黍 40
玉衡 10
玉貌 1
玉質 11
玉趾 0
玉里 577
玉釵 6
玉鐲 6
玉鐲子 0
玉門 24
玉門關 6
玉階 19
玉雕 8
玉面 7
玉顏 1
玉食 0
玉髓 51
玉體 20
玉齒 2
玊 64
玊 64
王 28
王 28963
王世充 82
王事 7
王人 32
王位 52
王位繼承 3
王侯 13
王侯后 0
王侯將相 2
王儲 12
王充 11
王八 22
王八烏龜 1
王八羔子 2
王八蛋 36
王公 77
王公貴人 0
王公貴族 9
王冕 7
王冠 63
王制 1
王功 21
王匡 1
王台 29
王后 47
王命 5
王國 1481
王國維 14
王天下 4
王夫之 6
王妃 92
王子 1140
王孫 30
王學 18
王守仁 4
王安 15
王安石 15
王官 1
王室 69
王室貴族 1
王宮 93
王實甫 3
王導 5
王師 10
王府 191
王座 37
王弼 9
王愷 5
王成 25
王政 49
王敦 31
王族 14
王昭君 8
王朝 510
王業 19
王權 28
王母 49
王母娘娘 10
王水 26
王永慶 222
王法 16
王爵 9
王父 2
王爺 66
王爾德 9
王牌 420
王牌軍 0
王猛 5
王獻之 2
王畿 1
王相 9
王祥 22
王維 158
王羲之 22
王老五 13
王考 0
王者 121
王者之風 9
王荊公 0
王莊 4
王莽 19
王莽篡漢 2
王菲 9
王蒙 1
王親 6
王親貴族 0
王論者 0
王貞治 23
王道 58
王陽明 31
王韜 0
王駕 0
玌 0
玍 0
玎 17
玏 0
玐 0
玐 0
玒 8
玓 27
玔 8
玕 6
玖 589
玗 12
玘 0
玜 0
玜 0
玝 8
玞 0
玟 1041
玟 1041
玠 41
玡 2
玢 22
玣 0
玤 12
玥 657
玦 21
玧 0
玧 0
玨 264
玩 691
玩 6911
玩世 3
玩世不恭 16
玩伴 81
玩偶 223
玩兒 13
玩兒 13
玩具 3379
玩具店 74
玩具槍 37
玩具狗 3
玩出 24
玩出來 0
玩出去 0
玩到 137
玩劍 1
玩厭 0
玩友 28
玩味 50
玩味 50
玩命 10
玩命之徒 0
玩器 4
玩家 5165
玩弄 120
玩弄著 9
玩得 543
玩心 9
玩心很重 0
玩忽 1
玩忽職守 4
玩意 92
玩意兒 69
玩意兒 69
玩慣 2
玩手 3
玩月 1
玩棋 0
玩槍 2
玩樂 195
玩歲愒時 0
玩水 62
玩法 238
玩火 35
玩火自焚 3
玩牌 44
玩物 24
玩物喪志 4
玩物喪志 4
玩玩 253
玩球 8
玩的 542
玩的 542
玩票 14
玩票性質 2
玩笑 265
玩耍 216
玩興 4
玩著 25
玩藝 19
玩藝兒 3
玩藝兒 3
玩賞 24
玩賞 24
玩賞著 0
玩起 35
玩起來 29
玩過 164
玩過來 0
玩過去 0
玩錢 0
玩鎗 0
玪 0
玫 1172
玫瑰 1699
玫瑰色 24
玫瑰花 154
玫瑰蝦 1
玬 5
玭 4
环 0
玲 5699
玲玎 1
玲瓏 107
玲瓏剔透 3
玳 247
玳瑁 13
玴 234
玵 83
玶 53
玷 778
玷汙 0
玷污 37
玷辱 5
玸 156
玹 88
玺 0
玺 0
玻 978
玻利維亞 58
玻尿酸 1
玻爾 6
玻片 12
玻璃 3060
玻璃匠 1
玻璃墊 1
玻璃彈珠 4
玻璃房 0
玻璃杯 55
玻璃板 8
玻璃框 3
玻璃棒 8
玻璃片 14
玻璃瓶 97
玻璃碗 0
玻璃窗 60
玻璃筒 0
玻璃管 36
玻璃紙 26
玻璃絲 5
玻璃絲襪 1
玻璃纖維 344
玻璃缸 14
玻璃罩 3
玻璃試管 5
玻璃鋼 5
玻璃門 23
玻璃體 92
玻里尼西 0
玻里尼西亞 23
玻里尼西亞人 2
玼 70
玽 0
玾 116
玿 87
珀 1294
珁 0
珂 516
珂羅版 0
珃 70
珄 0
珅 410
珆 140
珇 53
珈 232
珉 0
珉 0
珊 2323
珊湖 7
珊瑚 698
珊瑚島 12
珊瑚潭 11
珊瑚石 3
珊瑚礁 546
珊瑚蟲 10
珋 792
珌 61
珍 4249
珍味 7
珍品 76
珍奇 66
珍妮 212
珍寶 64
珍惜 947
珍愛 215
珍本 18
珍物 0
珍玩 20
珍珠 519
珍珠奶茶 1
珍珠店 0
珍珠港 31
珍珠米 6
珍珠貝 2
珍珠雞 0
珍珠項鍊 55
珍異 9
珍祕 0
珍禽 13
珍禽異獸 18
珍羞 2
珍聞 3
珍肴異饌 0
珍膳 0
珍藏 447
珍藏品 16
珍藏本 4
珍視 50
珍貴 1105
珍賞 29
珍重 42
珍重再見 22
珎 0
珏 0
珏 0
珐 0
珐 0
珒 130
珓 58
珔 187
珕 0
珖 67
珗 213
珘 142
珙 56
珚 166
珛 918
珜 428
珝 95
珞 61
珟 0
珠 6500
珠光寶氣 8
珠光體 0
珠兒 14
珠兒 14
珠圓玉潤 2
珠子 56
珠寶 1477
珠寶商 24
珠寶店 13
珠寶行 4
珠母 5
珠江 33
珠江三角洲 27
珠海 213
珠淚 7
珠玉 10
珠璣 12
珠算 122
珠算課 1
珠簾 11
珠紋 1
珠聯璧合 3
珠胎暗結 0
珠蚌 0
珠貝 3
珠飾 3
珠飾品 0
珡 0
珢 0
珢 0
珣 215
珤 0
珥 88
珦 0
珧 216
珨 773
珩 1191
珪 60
珫 61
珬 0
班 5523
班上 623
班主 13
班主任 113
班代 81
班代表 21
班兵 21
班刊 60
班別 376
班吉 1
班固 4
班圖 2
班子 44
班師 40
班師回朝 7
班底 41
班彪 1
班數 165
班旗 4
班昭 3
班會 267
班機 826
班次 351
班珠爾 0
班班 94
班白 2
班白 2
班禪 11
班禪額爾德尼 0
班級 3574
班組 11
班花 43
班裡 11
班費 18
班超 17
班車 172
班遊 0
班都斯山 0
班長 348
班門弄斧 7
班隊 16
班頭 133
班鳩 5
珮 939
珮環 0
珮飾 0
珳 0
珴 10
珵 16
珶 5
珷 0
珸 29
珹 0
珺 62
珻 0
珼 14
珽 18
現 13177
現世 69
現世報 7
現今 1666
現代 6327
現代主義 165
現代人 784
現代修正主義 0
現代化 1393
現代史 137
現代工業 15
現代感 64
現代戲 0
現代文學 135
現代舞 69
現代詩 202
現代資本主義 5
現代農業 8
現任 2479
現值 417
現出 195
現出原形 5
現制性 0
現勢 31
現在 20223
現在完成式 10
現在式 7
現在是 469
現場 464
現場 4648
現場報導 34
現場報導 34
現場表演 29
現場表演 29
現場轉播 52
現場轉播 52
現場轉播 52
現場轉播 52
現大洋 1
現存 322
現學 22
現學現教 0
現實 1889
現實主義 76
現實性 6
現已 397
現年 144
現形 92
現役 116
現役軍人 80
現役軍官 15
現從 21
現成 190
現成飯 0
現時 130
現有 4342
現款 31
現況 7745
現況報導 45
現洋 0
現洋錢 0
現澆 0
現狀 408
現用 26
現眼 4
現職 879
現蕾期 0
現行 2245
現行法 42
現行法令 84
現行犯 46
現象 5556
現象學 64
現貨 350
現貨交易 4
現貨供應 91
現買 19
現買現賣 0
現身 374
現身說法 94
現進現出 0
現金 2441
現金管理 4
現金買賣 0
現鈔 80
現錢 1
現錢交易 0
現階段 985
現露 1
現露 1
珿 12
琀 24
琀 24
琁 51
琂 0
球 7785
球僮 2
球半徑 2
球台 3
球員 1802
球員卡 1
球場 9
球場 915
球場上 71
球場上 71
球墨鑄鐵 15
球壇 6
球壇上 0
球季 261
球形 67
球形物 0
球徑 3
球心 29
球技 128
球拍 144
球果 1
球根 42
球桿 233
球棍 8
球棒 38
球狀 95
球王 73
球竿 13
球網 250
球芽 0
球莖 47
球菌 40
球蛋白 126
球衣 0
球評 1
球賽 313
球路 44
球迷 904
球速 29
球運 75
球門 55
球門柱 6
球隊 718
球面 79
球面三角 4
球面三角形 2
球面幾何 2
球面幾何學 0
球面鏡 3
球面體 0
球鞋 153
球類 145
球類運動 99
球體 69
琄 20
琅 45
琅玕 0
琅琅 9
琅琅上口 13
琅琊 1
琅璫入獄 0
理 16471
理上 114
理了 23
理事 1249
理事國 7
理事會 718
理事長 1841
理個 6
理光 54
理光頭 4
理出 43
理到 0
理則 39
理則學 14
理化 423
理去 10
理喻 0
理好 9
理學 322
理學博士 40
理學士 45
理學家 20
理學院 788
理完 39
理宗 4
理定 2
理家 9
理容 72
理容師 1
理容院 24
理屈 3
理屈心虛 0
理屈詞窮 0
理工 385
理工學院 244
理工科 32
理平頭 3
理平頭 3
理念 4336
理性 1329
理性主義 25
理性化 11
理性認識 1
理想 3353
理想主義 61
理想化 44
理想國 98
理應 155
理成 19
理所當然 519
理智 207
理會 280
理查 154
理法 28
理清 18
理無常是 0
理理 10
理由 2704
理當 64
理當如此 2
理療 52
理療科 2
理監事 819
理直氣壯 60
理睬 31
理短 3
理科 160
理科大學 6
理虧 7
理虧心虛 0
理解 1199
理解力 29
理該 16
理論 6867
理論上 309
理論化 3
理論家 49
理論性 27
理論派 7
理論科學 248
理財 10151
理財專家 13
理賠 0
理起 0
理起來 0
理路 8
理過 8
理過來 0
理過去 0
理頭 6
理頭髮 1
理髮 253
理髮匠 1
理髮師 30
理髮店 49
理髮廳 331
理髮院 24
琇 842
琈 10
琉 398
琉球 429
琉球人 2
琉球嶼 7
琉球群島 14
琉璃 700
琉璃廠 4
琉璃燈 0
琉璃瓦 11
琊 15
琋 14
琌 6
琍 47
琔 0
琕 0
琖 10
琗 0
琘 0
琘 0
琙 0
琚 287
琛 286
琜 0
琝 85
琞 0
琟 0
琠 84
琡 47
琢 99
琢玉 1
琢磨 72
琣 205
琤 226
琤琮 3
琥 62
琥珀 87
琥珀色 17
琦 2080
琦善 0
琧 0
琨 152
琩 91
琪 2685
琪花 1
琫 3
琬 306
琭 191
琮 266
琯 10
琯 10
琰 39
琱 393
琲 224
琳 2473
琳琅 3
琳琅滿目 49
琳瑯 0
琳瑯滿目 104
琴 1825
琴師 18
琴弓 5
琴弦 25
琴房 12
琴技 8
琴架 1
琴柱 1
琴桿 0
琴棋書畫 16
琴瑟 19
琴瑟和鳴 3
琴筒 4
琴絃 3
琴聲 100
琴藝 13
琴調 1
琴譜 18
琴趣 4
琴酒 57
琴鍵 20
琴鐘 1
琴音 40
琴韻 76
琴馬 0
琴鳥 1
琵 537
琵琶 184
琵琶別抱 2
琵琶湖 0
琵琶行 0
琵琶記 0
琵琶骨 2
琵琶魚 0
琶 32
琶 32
琶 32
琸 0
琹 0
琺 10
琺瑯 94
琺瑯彩 1
琺瑯璉 0
琺瑯質 14
琼 0
琽 0
琾 0
琿 25
瑀 223
瑁 73
瑁 73
瑂 26
瑃 0
瑄 533
瑅 0
瑆 33
瑇 0
瑈 0
瑉 0
瑉 0
瑊 158
瑋 1496
瑋寶 0
瑌 0
瑌 0
瑍 35
瑎 189
瑏 25
瑐 71
瑑 72
瑒 49
瑓 0
瑔 39
瑕 136
瑕不掩瑜 13
瑕垢 0
瑕瑜互見 0
瑕疵 514
瑕積分 0
瑕隙 2
瑖 0
瑗 86
瑘 0
瑙 31
瑚 115
瑛 804
瑜 1901
瑜伽 108
瑜伽宗 1
瑜伽師地論 1
瑜伽派 1
瑜伽行派 1
瑜伽術 3
瑜珈 90
瑜珈術 3
瑝 0
瑞 12051
瑞兆 2
瑞典 903
瑞典人 20
瑞典文 8
瑞典話 0
瑞典語 4
瑞利 58
瑞士 1270
瑞士人 5
瑞安 134
瑞氣 13
瑞澄 2
瑞穗 388
瑞芳 574
瑞草 3
瑞金 85
瑞雪 15
瑞雲 28
瑟 958
瑟瑟 0
瑟瑟秋風 0
瑟縮 15
瑟縮 15
瑟縮著 0
瑠 0
瑢 105
瑣 350
瑣事 62
瑣務 1
瑣瑣 0
瑣瑣碎碎 3
瑣碎 77
瑣細 1
瑣聞 1
瑣言 0
瑤 589
瑤光 1
瑤族 19
瑤池 38
瑤池金母 1
瑤臺 7
瑤草 2
瑤華 16
瑥 0
瑦 0
瑧 65
瑨 0
瑩 1619
瑩潔 6
瑩潤 1
瑩澤 16
瑪 2878
瑪利亞 89
瑪沙 1
瑪瑙 62
瑪莉 806
瑪莎 53
瑪麗 202
瑪麗亞 194
瑪麗蓮夢露 16
瑫 0
瑬 0
瑭 144
瑮 49
瑯 155
瑯 155
瑰 287
瑰奇 1
瑰姿 1
瑰寶 64
瑰異 0
瑰麗 89
瑱 164
瑲 249
瑳 47
瑴 0
瑴 0
瑵 136
瑶 0
瑹 59
瑺 0
瑻 0
瑼 59
瑽 1090
瑾 276
瑿 13
璀 77
璀璨 124
璁 219
璂 0
璃 455
璄 0
璅 68
璆 115
璇 460
璇宮 14
璇璣 85
璈 90
璉 389
璊 401
璋 751
璌 0
璏 0
璐 182
璑 0
璒 7
璓 0
璔 15
璕 22
璖 0
璖 0
璗 96
璘 109
璙 0
璚 145
璛 0
璜 88
璝 0
璝 0
璞 146
璞玉 27
璞玉渾金 0
璟 297
璠 81
璡 5
璢 0
璣 49
璥 51
璦 40
璦琿 3
璧 502
璧人 5
璧合 1
璧帛 0
璧月 0
璧爐 0
璧爐台 0
璧玉 4
璨 167
璨美 0
璩 88
璪 8
璫 8
璫璫 0
璬 0
璭 7
璮 0
璯 312
環 8823
環中 50
環伺 30
環佩 0
環保 13918
環保局 804
環保署 2312
環列 2
環坐 0
環城 6
環城公路 0
環堵 2
環境 25487
環境保育 104
環境保護 2954
環境污染 389
環境衛生 524
環層帶 0
環山 151
環山公路 0
環島 188
環島旅行 12
環工 577
環帶 20
環形 75
環抱 59
環氧乙烷 55
環氧樹脂 135
環水 8
環河 116
環河北路 20
環河南路 78
環流 179
環流器 0
環海 51
環湖 37
環湖公路 20
環烷 2
環烷烴 2
環狀 134
環球 1181
環球小姐 4
環球旅行 2
環環 7
環環相扣 81
環礁 23
環節 150
環節動物 5
環紋 14
環繞 499
環繞著 61
環肥燕瘦 16
環著 3
環蝕 0
環行 4
環視 36
環遊 34
環遊世界 156
環道 9
環面 0
環靶 0
環顧 53
環顧四周 12
環顧四週 10
璱 36
璲 53
璳 0
璵 8
璶 3
璷 3
璷 3
璸 8
璹 0
璹 0
璺 18
璻 3
璼 0
璽 281
璾 6
璿 70
璿 70
瓀 5
瓁 2
瓂 5
瓃 5
瓄 0
瓅 14
瓆 0
瓇 0
瓈 0
瓉 0
瓊 2173
瓊州 6
瓊州海峽 3
瓊枝玉葉 0
瓊樓 2
瓊樓玉宇 3
瓊漿 4
瓊漿玉液 7
瓊漿玉液 7
瓊瑤 237
瓊筵 0
瓊脂 19
瓊英 33
瓊華 37
瓊麻 34
瓋 5
瓌 0
瓌 0
瓍 0
瓎 0
瓏 232
瓐 0
瓑 0
瓓 0
瓔 74
瓕 7
瓖 20
瓗 4
瓗 4
瓘 18
瓙 3
瓚 157
瓛 258
瓜 935
瓜仁 2
瓜代 4
瓜分 90
瓜剖豆分 0
瓜圃 1
瓜地馬拉 156
瓜子 108
瓜子 108
瓜子仁 2
瓜子仁 2
瓜子殼 0
瓜子殼 0
瓜子臉 3
瓜子臉 3
瓜字初分 0
瓜撓 0
瓜時 5
瓜期 0
瓜果 19
瓜棚 17
瓜熟蒂落 2
瓜片 2
瓜瓜 48
瓜瓞 1
瓜瓞綿綿 0
瓜瓣 0
瓜瓤 0
瓜田 21
瓜田李下 1
瓜田李下之嫌 2
瓜皮 2
瓜皮帽 1
瓜肉 3
瓜葛 19
瓜葛 19
瓜藤 2
瓜農 4
瓝 13
瓞 65
瓟 143
瓠 27
瓡 46
瓢 72
瓢兒 0
瓢兒 0
瓢子 1
瓢蟲 50
瓣 495
瓣膜 121
瓤 8
瓥 155
瓦 1961
瓦全 1
瓦匠 0
瓦器 10
瓦圈 0
瓦堆 1
瓦塊 0
瓦室 0
瓦屋 19
瓦崗軍 10
瓦工 12
瓦房 3
瓦斯 1103
瓦斯彈 2
瓦斯桶 1
瓦斯爐 88
瓦斯筒 5
瓦斯行 54
瓦斯體 0
瓦時計 6
瓦杜茲 0
瓦楞紙 497
瓦爾 37
瓦片 19
瓦特 70
瓦特計 6
瓦盆 2
瓦礫 14
瓦窯 14
瓦罐 0
瓦羅 2
瓦肆 2
瓦舍 2
瓦萊塔 0
瓦解 170
瓦解冰銷 0
瓦解土崩 0
瓦釜 1
瓦釜雷鳴 1
瓨 9
瓨 9
瓪 0
瓫 0
瓬 147
瓭 0
瓮 30
瓳 0
瓴 159
瓵 127
瓶 2144
瓶上 21
瓶口 26
瓶塞 10
瓶塞兒 0
瓶塞兒 0
瓶子 238
瓶底 6
瓶盂 0
瓶缽 0
瓶蓋 126
瓶裝 105
瓶頸 397
瓷 3000
瓷器 193
瓷土 91
瓷漆 8
瓷片 7
瓷磚 42
瓷窯 1
瓷胎 3
瓷釉 16
瓹 0
瓹 0
瓺 0
瓻 59
瓽 136
瓾 110
瓿 70
甀 40
甁 0
甂 199
甃 42
甄 685
甄別 12
甄審 333
甄用 8
甄訓 9
甄試 3257
甄選 2328
甄選人才 1
甆 0
甇 64
甈 198
甉 0
甊 0
甋 13
甌 68
甌劇 1
甌子 1
甌宰 1
甌摳 1
甌江 0
甌窶 1
甌繡 1
甌脫 1
甌臾 1
甌越 1
甍 9
甎 0
甏 56
甐 125
甑 20
甒 12
甓 0
甔 6
甕 188
甕中 5
甕中之鱉 1
甕中捉鱉 0
甕中鱉 0
甕城 4
甕牖繩樞 0
甕聲甕氣 0
甕音 0
甕鼻 0
甖 4
甗 22
甘 1267
甘之吠飴 0
甘之如飴 12
甘休 3
甘冒 18
甘味 11
甘國 2
甘地 39
甘寧 9
甘居下流 0
甘居下游 0
甘州 8
甘心 139
甘心忍受 1
甘心情願 5
甘托克 0
甘拜下風 17
甘於 36
甘比亞 28
甘汞 0
甘油 94
甘油酯 25
甘泉 41
甘為 10
甘甜 56
甘當 1
甘立忍受 0
甘籃菜 0
甘結 0
甘美 38
甘肅 61
甘肅人 0
甘肅省 40
甘脆 11
甘苦 55
甘苦談 143
甘草 57
甘草粉 0
甘菊 32
甘菜 0
甘蔗 196
甘蔗汁 7
甘蔗渣 1
甘蔗田 6
甘薯 23
甘薯餅 0
甘藍 39
甘藍菜 4
甘藷 77
甘迺迪 85
甘迺迪角 4
甘雨 1
甘霈 1
甘霖 46
甘露 116
甘露醇 1
甘願 123
甚 2599
甚 2599
甚 2599
甚且 46
甚佳 66
甚囂塵上 25
甚多 325
甚好 27
甚少 148
甚急 12
甚或 299
甚於 52
甚早 24
甚是 98
甚殷 11
甚為 224
甚為不解 2
甚獲 11
甚獲好評 3
甚而 120
甚至 8206
甚至不 89
甚至在 265
甚至於 380
甚至會 143
甚至有 356
甚至能 34
甚至都 12
甚解 2
甚遠 37
甚麼 1842
甚麼 1842
甛 0
甜 20239
甜不辣 36
甜味 77
甜得 51
甜心 49
甜情密意 0
甜柑 0
甜水 3
甜津津 0
甜瓜 16
甜甜 1175
甜甜圈 1
甜甜蜜蜜 15
甜的 51
甜地 50
甜睡 0
甜筒 9
甜糕點 1
甜絲絲 2
甜美 299
甜而不膩 8
甜菜 39
甜薯 1
甜蜜 637
甜蜜蜜 39
甜言 13
甜言蜜語 78
甜酒 45
甜酒釀 3
甜酸苦辣 2
甜頭 46
甜食 35
甜香 5
甜麵醬 2
甜點 141
甝 31
甞 0
生 32705
生下 187
生下來 39
生下去 0
生不帶來 2
生不逢時 3
生不逢辰 0
生乳 17
生了 256
生事 17
生于 22
生人 33
生來 97
生來生去 0
生個 36
生僻 1
生兒育女 13
生冷 27
生冷食物 1
生出 268
生出來 17
生分 11
生利 12
生利息 2
生前 364
生力軍 94
生力麵 2
生動 504
生動如昔 0
生動活潑 140
生化 1104
生化學 118
生化學家 10
生化需氧量 42
生厭 11
生吃 34
生吞 9
生吞活剝 3
生命 12773
生命令 0
生命力 408
生命史 15
生命學 3
生命現象 48
生命線 223
生員 6
生啤酒 19
生土 23
生在 475
生坯 1
生好 10
生子 104
生子 104
生字 62
生字簿 0
生字表 0
生存 1777
生存主義 0
生存之道 39
生存空間 118
生存競爭 16
生客 1
生就 144
生平 547
生平事蹟 21
生年 214
生張熟魏 1
生得 128
生得 1692
生怕 48
生性 161
生性固執 0
生怯怯 0
生恐 5
生息 30
生意 607
生意人 51
生意盎然 7
生意經 11
生意興隆 45
生態 5152
生態學 269
生態學家 13
生態平衡 57
生態環境 471
生態系 197
生態褲 0
生態危機 100
生懼 1
生成 553
生成物 16
生手 51
生技 150
生拉硬拽 0
生搬硬套 0
生擒 17
生效 583
生於 1331
生於憂患 0
生日 2667
生日快樂 505
生日派對 17
生日舞會 1
生日蛋糕 65
生旦淨末丑 2
生根 154
生榮死哀 0
生機 480
生機勃勃 0
生死 739
生死之交 5
生死別離 1
生死存亡 23
生死有命 7
生死有命富貴在天 0
生死未卜 9
生死肉骨 0
生死肉骨 0
生死與共 5
生死關頭 33
生死關頭 33
生殖 608
生殖力 15
生殖器 81
生殖器官 70
生殖孔 5
生殖洄游 0
生殖洄遊 0
生殖細胞 20
生殖腺 19
生殺 4
生殺予奪 0
生殺大權 5
生母 44
生民 22
生氣 791
生氣勃勃 12
生氣蓬勃 16
生水 58
生油 42
生津 9
生津止渴 7
生津解渴 2
生活 34052
生活上 334
生活力 7
生活區 31
生活史 83
生活圈 183
生活實踐 13
生活必需 18
生活必需品 24
生活方式 357
生活條件 24
生活水平 13
生活水準 198
生活狀況 72
生活環境 569
生活用品 89
生活空間 373
生活素 5
生活經驗 139
生活習慣 182
生活費 178
生活費用 56
生活資料 3
生涯 1493
生涯規劃 540
生源論 0
生漆 14
生澀 32
生火 49
生為 149
生煤 17
生父 42
生物 9152
生物化學 362
生物學 537
生物學家 60
生物學科 0
生物學系 176
生物戰劑 0
生物技術 0
生物界 20
生物膜 12
生物防治 32
生物體 73
生猛 37
生理 1735
生理學 582
生理學家 15
生理期 1
生理衛生 11
生理食鹽水 73
生理鹽水 5
生生 80
生生不已 7
生生不息 84
生生不祐 0
生生世世 47
生產 12499
生產力 1128
生產合作 6
生產合作社 93
生產基金 1
生產大隊 3
生產工具 16
生產性 49
生產操 0
生產方式 72
生產流程 52
生產率 12
生產組 27
生產線 520
生產者 213
生產能力 46
生產費 7
生產資料 17
生產資本 2
生產過剩 51
生產過程 120
生產量 362
生產關係 0
生產額 25
生產鬥爭 0
生男育女 0
生番 8
生疏 38
生疑 10
生病 584
生發 35
生的 1691
生皮 20
生知 2
生石灰 29
生硬 37
生端 2
生米 8
生米已煮成熟飯 1
生米煮成 0
生米煮成熟飯 2
生絲 17
生老病死 45
生而平等 9
生而自由 2
生聚教訓 4
生肉 14
生肖 275
生育 500
生育率 33
生色 6
生花妙筆 10
生花筆 2
生苗 14
生苦 1
生菜 66
生菜沙拉 14
生菩薩 1
生著 42
生著氣 0
生薑 32
生蛋 40
生蛋雞 0
生蠔 1
生計 151
生詞 2
生詞表 0
生變 122
生豬 8
生財 41
生財器具 5
生財有道 6
生起 105
生起來 2
生趣 18
生趣盎然 2
生路 627
生身 11
生身父母 2
生辰 40
生辰八字 20
生迭水準 0
生過 48
生還 103
生還者 28
生銅 3
生銹 1
生鏽 49
生鐵 18
生鐵皮 0
生長 1944
生長期 26
生長激素 73
生長率 2
生長素 53
生長點 5
生離 14
生離死別 36
生靈 46
生靈塗炭 9
生食 77
生飯 2
生飲 32
生養 36
生髮 25
生髮劑 1
生髮水 3
生魚片 53
生鮮 233
生鮮食品 11
生點 9
生龍活虎 18
甠 0
甡 198
產 10852
產仔 17
產值 1022
產假 42
產兒 16
產出 344
產前 186
產前檢查 58
產區 148
產卵 209
產卵器 0
產卵場 3
產卵場 3
產卵期 14
產卵洄游 0
產品 58386
產地 608
產地證明 7
產婆 4
產婦 199
產床 1
產後 308
產後檢查 5
產房 41
產業 34199
產業工人 4
產業後備 0
產業後備軍 1
產業界 558
產業資本 2
產業革命 5
產權 416
產物 606
產物保險 562
產狀 3
產生 10816
產生出 54
產生器 252
產界 25
產科 171
產科醫生 2
產米 1
產米地 0
產經新聞 131
產能 1126
產自 22
產茶 9
產茶地 0
產蛋雞 17
產褥 10
產量 1268
產銷 1356
產銷一元 0
產銷一元 0
產銷一元化 2
產銷一元化 2
產銷合一 5
產銷合同 0
產院 1
產險 305
產額 10
産 0
甤 0
甥 40
甥兒 0
甥女 2
甥舅 4
甦 153
甦醒 95
甦醒過來 11
甧 0
用 61256
用上 110
用上來 0
用上去 1
用不 117
用不乏 0
用不了 14
用不到 53
用不完 27
用不慣 1
用不聞 0
用不著 124
用世 2
用之不竭 22
用事 4
用人 305
用以 882
用作 132
用來 2606
用來用去 1
用光 85
用兵 51
用兵一時 0
用兵一時 0
用具 608
用刑 5
用到 519
用力 837
用力一打 0
用力一打 0
用力一拍 1
用力一拍 1
用力過度 0
用功 324
用功讀書 39
用功起來 0
用功過度 1
用勁 7
用去 59
用命 5
用品 4716
用器 13
用在 637
用在一時 1
用在一時 1
用地 1794
用場 45
用場 45
用字 130
用字不當 0
用完 209
用度 23
用得著 7
用心 1297
用心想 4
用心看 6
用心聽 5
用心良苦 72
用心讀 2
用情 44
用意 296
用戶 37886
用掉 37
用於 1898
用是 26
用材林 1
用武 43
用武之地 5
用水 1004
用水量 154
用法 796
用為 85
用畢 32
用盡 181
用盡心機 1
用盡方法 4
用紙 612
用膳 19
用舊 45
用藥 1091
用處 116
用計 40
用詞 127
用詞不當 5
用語 323
用費 23
用起來 37
用途 4191
用過 0
用量 1405
用錢 114
用非所學 0
用非所長 0
用飯 9
用餐 406
甩 390
甩不掉 4
甩不掉 4
甩不開 0
甩出 4
甩出來 0
甩出去 3
甩到 6
甩手 8
甩掉 19
甩脫 3
甩走 1
甩開 21
甪 11
甫 2021
甬 64
甬江 3
甬路 0
甬道 17
甭 630
甮 5
甯 312
田 20239
田七 11
田中 776
田中角榮 7
田主 13
田作 0
田單 467
田園 250
田園曲 1
田園派 1
田園生活 10
田園詩 7
田園詩人 1
田園風光 21
田地 179
田圳 0
田埂 26
田契 0
田家 22
田寮 235
田尾 250
田岸 0
田徑 235
田徑場 68
田徑場 68
田徑賽 22
田徑運動 10
田徑隊 58
田心 78
田戶 0
田漢 1
田獵 1
田產 15
田田 17
田畔 1
田畝 3
田畦 0
田疇 4
田租 2
田種玉 0
田糧 2
田納西 78
田納西威廉斯 0
田納西州 18
田舍 8
田舍翁 0
田莊 18
田螺 31
田裡 86
田賦 51
田賽 29
田里 43
田野 369
田間 164
田間管理 4
田雞 33
田黃 6
田鼠 65
由 8963
由上而下 87
由不得 9
由不得你 2
由中 49
由人 181
由他去 2
由余 10
由你 130
由來 551
由來已久 34
由儉入奢 0
由儉入奢易 0
由天而降 1
由奢入儉 1
由奢入儉難 1
由恪志遠 0
由承已久 0
由於 20999
由此 1739
由此可知 209
由此可見 209
由此看來 38
由此觀之 29
由此證明 5
由淺入深 39
由簡而繁 5
由簷而繁 0
由衷 1670
由衷之言 2
由近而遠 12
由頭 18
甲 4390
甲上 42
甲下 2
甲乙 191
甲乙丙丁 42
甲仙 87
甲兵 8
甲冑 4
甲冑魚類 0
甲午 66
甲午戰 3
甲午戰爭 46
甲基 382
甲基橙試液 0
甲基橙試液 0
甲士 6
甲天下 7
甲子 135
甲存 12
甲板 118
甲案 11
甲殼 81
甲殼素 1
甲殼類 88
甲烷 219
甲狀旁腺 0
甲狀腺 1256
甲狀腺素 61
甲狀腺腫 26
甲狀軟骨 2
甲班 417
甲癬 5
甲種 172
甲種維生素 0
甲種體位 0
甲第 223
甲等 226
甲級 253
甲線 7
甲苯 113
甲蟲 41
甲蟲類 1
甲酚 24
甲酸 132
甲醇 137
甲醚 11
甲醛 159
甲隊 6
甲項 1
甲骨 57
甲骨文 288
甲魚 39
申 4333
申不害 2
申令 0
申冤 26
申告 16
申報 3668
申報人 47
申報所得稅 25
申報者 60
申張 3
申戒 2
申斥 6
申明 50
申明立場 0
申明立場 0
申時 12
申生 8
申訴 1087
申訴人 42
申訴者 11
申誡 70
申說 1
申請 33481
申請人 2085
申請到 91
申請單 391
申請書 2633
申請者 541
申請表 1515
申論 47
申論題 23
申謝 2
申貸 0
申購 575
申購人 20
申購者 8
申購量 0
申辯 30
申述 14
申述人 0
申述立場 0
申述立場 0
申述者 0
申飭 0
甶 0
男 11034
男傭 100
男丁 15
男中音 15
男主人 23
男主角 427
男主角 427
男人 3906
男人婆 11
男仕 64
男伴 17
男低音 8
男佣 70
男佣人 1
男佣人 1
男僕 4
男儐相 1
男兒 74
男兒本色 3
男友 582
男同學 50
男單 0
男士 307
男大當婚 3
男女 3084
男女之間 98
男女平等 36
男女授受 1
男女授受不親 6
男女有別 10
男女皆可 22
男女私情 0
男女老少 52
男女老幼 21
男女老弱 0
男女雙方 45
男妓 4
男婚女嫁 4
男嬰 36
男嬰兒 0
男子 1807
男子漢 76
男孩 1177
男孩兒 0
男孩兒 0
男孩子 154
男學生 27
男客 9
男客人 1
男家 5
男屍 1
男工 10
男左女右 5
男廁 10
男廁所 4
男性 2606
男性化 29
男才女貌 0
男扮女裝 19
男方 114
男星 37
男朋友 297
男校 8
男歡女愛 199
男爵 31
男爵夫人 0
男生 1760
男生宿舍 45
男用 73
男用錶 0
男男女女 52
男的 403
男盜女娼 2
男童 139
男童軍 9
男籃 175
男籃賽 16
男耕女織 0
男聲 29
男色 14
男衣 3
男裝 84
男賓 1
男賓止步 2
男靴 0
男高音 142
甸 349
甸 349
甹 7
町 329
画 0
甽 363
甾 12
甿 8
畀 12
畁 0
畂 0
畂 0
畃 0
畄 0
畆 0
畆 0
畇 4
畈 8
畉 0
畊 0
畊 0
畋 1503
畋獵 15
界 8015
界內 25
界內球 3
界分別觀 1
界址 53
界外 34
界外球 13
界定 460
界尺 0
界山 37
界方便觀 1
界標 17
界石 4
界碑 13
界約 3
界線 226
界說 18
界限 200
界面 1211
界首 26
畍 0
畎 990
畎畝 0
畏 1385
畏光 22
畏光性 0
畏冷 0
畏寒 14
畏忌 0
畏怯 10
畏怯 10
畏怯不前 0
畏怯不前 0
畏憚 0
畏懼 147
畏懼感 0
畏服 0
畏死 3
畏水 0
畏縮 29
畏縮不前 0
畏罪 0
畏罪潛逃 2
畏罪自殺 0
畏途 15
畏難 2
畏首畏尾 7
畐 0
畔 507
畕 0
畖 0
畗 0
畘 0
留 6
留 6341
留一手 4
留下 3888
留下來 172
留下去 2
留不住 41
留了 89
留些 27
留任 243
留住 171
留作 22
留個 150
留傳 20
留到 34
留取 6
留名 63
留在 903
留夠 0
留好 11
留存 141
留學 4533
留學生 822
留守 58
留守人員 5
留完 25
留客 13
留宿 18
留居 5
留底 13
留底稿 0
留底總額 0
留影 113
留待 120
留後 3
留後路 2
留得 23
留得青山 0
留得青山在 9
留德 28
留心 113
留心看著 0
留心看著 0
留念 64
留情 33
留意 630
留意到 19
留戀 57
留戀忘返 0
留日 181
留日學生 9
留有 137
留有餘地 1
留校 40
留校察看 21
留校查看 3
留步 4
留法 16
留滯 9
留用 36
留留 24
留白 113
留神 34
留種 5
留種地 0
留空 32
留筆 0
留級 51
留級生 5
留給 519
留置 463
留置權 1
留美 205
留美同學會 4
留美學生 11
留聲機 24
留職 14
留職停薪 94
留芳 8
留芳後世 0
留英 12
留落 4
留著 128
留蘭香 0
留言 50546
留言板 15940
留言簿 2539
留話 178
留話機 0
留起 5
留起來 0
留退步 0
留連 24
留連忘返 17
留過 11
留難 3
留頭髮 0
留餘地 5
留駐 10
留髮 1
留鬍子 4
留鬍鬚 1
留點 38
畚 60
畚斗 4
畚箕 208
畛 109
畛域 5
畜 853
畜 853
畜力 4
畜欄 0
畜牧 511
畜牧場 77
畜牧場 77
畜牧時代 0
畜牧業 67
畜牧群 0
畜牲 27
畜生 79
畜生道 1
畜產 579
畜產品 88
畜類 2
畜養 24
畝 189
畝產 2
畞 0
畞 0
畟 176
畡 0
畡 0
畢 5509
畢升 0
畢卡索 401
畢命 1
畢希納 1
畢恭畢敬 3
畢昇 10
畢業 9120
畢業典禮 286
畢業班 365
畢業生 1904
畢業考 17
畢業論文 157
畢業證書 352
畢氏定理 35
畢生 154
畢直 1
畢竟 1685
畢聯會 92
畢達哥拉斯 2
畣 4
畤 13
略 3292
略低於 10
略傳 14
略加 31
略勝 5
略勝一籌 22
略勝一籌 22
略去 23
略取 0
略可 19
略同 10
略圖 29
略大 33
略大於 7
略奪 2
略小 13
略帶 109
略微 60
略掉 0
略有 206
略有出入 8
略有所聞 2
略為 89
略為好轉 1
略略 44
略知 8
略知一二 15
略知皮毛 1
略見 24
略見一斑 0
略見一斑 0
略言之 0
略語 2
略語表 4
略說 16
略識之無 0
略讀 4
略述 99
略述大意 0
略過 55
略遜一籌 14
略遜一籌 14
略陽 0
略高於 27
畦 231
畦 231
畦田 0
畧 0
畨 0
畨 0
畨 0
番 10
番 1065
番人 9
番仔 50
番代 1
番休 0
番國 4
番地 16
番將 6
番族 3
番族人 0
番服 0
番木瓜 2
番椒 3
番石榴 53
番社 32
番禺 8
番紅花 4
番紅花色 0
番茄 131
番茄汁 1
番茄醬 12
番薯 63
番藷 0
番號 35
番話 114
番語 3
番邦 3
番鬼 0
畫 23784
畫作 141
畫一 88
畫上 82
畫下 74
畫下去 5
畫中有畫 10
畫中有詩 5
畫了 90
畫人 22
畫來畫去 2
畫供 8
畫像 155
畫兒 1
畫兒 1
畫具 5
畫冊 162
畫出 297
畫出來 23
畫分 26
畫分為 3
畫刊 57
畫到 10
畫匠 7
畫卷 5
畫去 1
畫史 13
畫品 8
畫商 8
畫圓 19
畫圖 208
畫圖筆 0
畫圖紙 1
畫在 57
畫地為牢 0
畫地自限 8
畫堂 1
畫報 178
畫境 8
畫外音 0
畫室 345
畫家 1009
畫尺 0
畫屏 4
畫展 436
畫工 34
畫布 204
畫帖 0
畫師 26
畫幅 7
畫廊 1083
畫得 422
畫意 4
畫成 31
畫戟 1
畫押 5
畫時代 2
畫景 0
畫會 122
畫本 6
畫板 17
畫架 9
畫框 44
畫框 44
畫棟雕梁 0
畫樓 5
畫法 76
畫為 16
畫片 54
畫界 16
畫畫 220
畫畫看 5
畫的 411
畫皮 0
畫眉 247
畫眉鳥 1
畫稿 7
畫符 7
畫筆 114
畫紙 45
畫素 1
畫線 54
畫聖 0
畫臉 1
畫舫 16
畫著 21
畫虎不成 2
畫虎不成反類犬 4
畫蛇添足 10
畫行 3
畫表 6
畫表格 13
畫評家 2
畫譜 12
畫質 355
畫起 2
畫起來 2
畫院 16
畫面 22271
畫頁 22
畫題 21
畫餅充飢 5
畫餅充饑 1
畫點 3
畫龍 3
畫龍點睛 24
畬 80
畭 0
畭 0
畮 0
畮 0
畯 410
異 3257
異乎尋常 14
異事 23
異人 45
異位 285
異俗 1
異像 15
異動 905
異動表 27
異化 37
異化作用 8
異口同聲 46
異同 172
異名 22
異味 102
異國 269
異國情調 21
異地 37
異地而處 1
異型 40
異域 36
異士 1
異姓 3
異己 28
異常 2119
異常現象 54
異形 200
異彩 11
異心 13
異志 1
異性 493
異性戀 1
異性相吸 3
異性體 0
異想天開 26
異態 2
異才 0
異教 57
異教徒 25
異數 379
異文 28
異於 142
異族 40
異族通婚 3
異日 9
異曲同工 7
異曲同工之妙 26
異服 0
異構 38
異構化 3
異構體 1
異樣 138
異步 2
異步電動機 1
異母 3
異母兄弟 0
異營菌 0
異父 0
異物 255
異狀 71
異相 5
異稟 14
異端 96
異端邪說 8
異義 6
異聞 9
異能 46
異行 3
異見 64
異言 99
異說 2
異議 690
異議人士 311
異變 23
異質 186
異趣 1
異趣橫生 0
異軍突起 43
異途 1
異邦 15
異鄉 136
異鄉人 26
異鄉客 26
異音 19
異類 58
異類相吸 0
異香 4
異體字 10
畱 0
畲 0
畲 0
畵 0
當 0
當 2793
當 27938
當一回事 12
當一回事 12
當一當 1
當一當 1
當上 127
當上來 0
當上去 0
當下 451
當下來 0
當下去 0
當不上 2
當不了 3
當不了 3
當不來 0
當不成 16
當不起 0
當且僅當 0
當世 85
當中 1905
當之有愧 0
當之無愧 24
當了 177
當了 177
當事 183
當事人 1180
當事國 21
當事者 60
當仁不讓 36
當今 547
當代 1797
當代人 7
當代大師 3
當令 15
當作 1081
當來當去 0
當個 183
當值 23
當做 413
當先 57
當兒 32
當兒 32
當兵 534
當初 1780
當到 12
當前 1694
當前之計 0
當務之急 169
當午 4
當即 66
當口 0
當可 128
當回事 1
當國 44
當地 4204
當地人 137
當場 64
當場 647
當場出醜 1
當場出醜 1
當場抓到 0
當場抓到 0
當場看到 0
當場看到 0
當場聽到 0
當場聽到 0
當夜 33
當天 1498
當官 23
當家 220
當家作主 15
當家的 11
當局 1245
當局者迷 7
當差 8
當年 1679
當得 51
當得上 3
當得來 0
當得起 0
當心 146
當應 15
當應 15
當成 632
當掉 119
當政 22
當政者 18
當斷不斷 0
當於 33
當日 1987
當日份 0
當是 83
當時 6813
當晚 216
當月 334
當月份 79
當朝 6
當機 885
當機 885
當機立斷 18
當權 22
當權派 19
當權者 30
當此 69
當歸 92
當歸鴨 15
當然 7912
當然在 100
當然是 960
當然會 127
當然有 159
當班 13
當班人員 0
當用 27
當用則用 3
當當 32
當真 149
當真 149
當真的 12
當真的 12
當眾 83
當眾宣佈 3
當眾表明 0
當眾表示 0
當票 0
當空 11
當紅 225
當耳邊風 3
當舖 398
當著 53
當街 25
當起 25
當起來 1
當路 12
當週 34
當過 121
當道 86
當道者 2
當選 1866
當選人 310
當選為 61
當選無效 1
當選者 14
當量 63
當量濃度 7
當鋪 51
當關 8
當陽 29
當面 256
當面言明 0
當頭 56
當頭棒喝 16
畷 16
畸 156
畸人 0
畸型 111
畸形 207
畸形人 0
畸形兒 23
畸形物 0
畸形發展 3
畸零 53
畹 65
畺 0
畻 0
畼 0
畽 7
畾 45
畿 21
畿內 1
疀 5
疁 0
疃 0
疃 0
疄 431
疅 0
疆 220
疆吏 0
疆土 10
疆域 44
疆場 2
疆場 2
疆界 109
疇 199
疈 0
疈 0
疉 0
疉 0
疊 1
疊 2425
疊疊樂 100
疊了 2
疊來疊去 0
疊出 1
疊出來 0
疊出去 0
疊句 1
疊合 29
疊在 14
疊好 8
疊字 4
疊小 0
疊嶂 2
疊平 1
疊床 10
疊床架屋 13
疊成 17
疊疊 37
疊疊看 0
疊羅漢 11
疊蓆 61
疊起 8
疊起來 7
疊韻 6
疋 54
疋 54
疋頭 0
疋頭 0
疌 7
疎 0
疎 0
疎 0
疏 735
疏 735
疏 735
疏伐 11
疏失 224
疏密 13
疏密不一 1
疏密度 1
疏導 68
疏忽 291
疏慵 0
疏懶 4
疏散 185
疏散人員 1
疏散法 0
疏於 67
疏於防備 0
疏於防範 4
疏水 37
疏洪道 0
疏浚 20
疏浚機 1
疏漏 89
疏濬 64
疏疏密密 0
疏疏落落 5
疏而不漏 4
疏落 8
疏財仗義 2
疏通 54
疏遠 55
疏離 160
疏露 0
疏露 0
疏鬆 510
疐 7
疑 2026
疑人不用 0
疑似 619
疑似病例 19
疑信參半 0
疑兇 10
疑凶 0
疑則勿用 0
疑則勿用用則勿疑 0
疑問 2655
疑問句 15
疑問號 1
疑問詞 5
疑團 11
疑心 35
疑心生暗鬼 2
疑心病 6
疑忌 7
疑惑 527
疑慮 513
疑懼 23
疑案 70
疑為 39
疑犯 12
疑猜 3
疑神疑鬼 5
疑竇 8
疑義 527
疑難 1205
疑難雜症 764
疑雲 94
疑雲重重 4
疑點 161
疒 0
疓 0
疔 17
疔瘡 4
疕 12
疘 2
疘 2
疙 145
疙瘩 19
疚 160
疛 0
疜 0
疝 151
疝氣 165
疞 0
疢 686
疣 776
疣贅 0
疤 646
疤痕 199
疤臉 37
疥 1725
疥瘡 23
疥癬 16
疥蟲 1
疦 0
疧 24
疨 0
疩 0
疪 57
疫 2169
疫區 118
疫情 0
疫疹 0
疫病 145
疫癘 1
疫苗 1427
疰 550
疱 0
疱疹 1
疲 273
疲乏 34
疲倦 238
疲倦不堪 2
疲倦起來 0
疲勞 411
疲勞強度 6
疲勞極限 0
疲勞轟炸 12
疲勞過度 2
疲匱 0
疲困 2
疲弊 1
疲弱 0
疲憊 218
疲憊不堪 20
疲敝 0
疲於奔命 27
疲睏 0
疲累 83
疲累不堪 2
疲軟 99
疲頓 1
疳 84
疳瘡 1
疳積 3
疵 1357
疵 1357
疵瑕 0
疵議 1
疶 127
疸 61
疹 341
疹子 49
疺 131
疻 72
疼 363
疼惜 190
疼愛 125
疼疼 1
疼痛 1317
疼痛不堪 2
疽 43
疽囊 0
疾 796
疾呼 20
疾寥 0
疾患 252
疾惡 1
疾書 1
疾病 6035
疾苦 37
疾行 11
疾視 0
疾言厲色 2
疾走 6
疾雷 3
疾風 103
疾風勁草 1
疾風知勁草 0
疾風迅雷 0
疾馳 20
疾駛 11
疿 145
痀 111
痀僂 2
痀僂病 0
痁 156
痂 84
痃 0
痄 100
痄 100
痄腮 0
病 8189
病了 82
病亡 2
病人 5125
病例 1603
病倒 18
病假 82
病假單 0
病兒 32
病兒 32
病入膏肓 14
病出 2
病到 11
病勢 5
病危 36
病原 200
病原菌 101
病原體 67
病友 416
病名 46
病員 2
病因 381
病夫 2
病好 18
病媒 228
病媒蚊 0
病媒體 0
病室 15
病害 158
病容 4
病容滿面 0
病床 267
病弱 37
病後 31
病後初愈 0
病後初癒 0
病得 28
病從口入 4
病徵 149
病急 2
病急亂投醫 11
病患 2683
病情 601
病情嚴重 19
病愈 4
病態 130
病態心理 3
病態心理學 0
病成 16
病房 993
病故 18
病株 3
病根 16
病根未除 0
病榻 9
病歷 761
病歷卡 0
病歷室 17
病歷表 20
病死 99
病死他鄉 0
病死異鄉 0
病歿 1
病毒 5905
病毒性 158
病況 68
病況危急 4
病源 54
病灶 190
病牙 1
病犬 12
病狀 45
病理 857
病理學 279
病理學家 15
病理解剖 17
病理解剖學 0
病理變化 42
病症 243
病痛 163
病癒 23
病癥 11
病程 120
病童 287
病篤 0
病粒 0
病苦 24
病菌 190
病著 1
病號 9
病蟲 29
病蟲害 197
病變 1136
病逝 147
病重 25
病院 27
病體 6
病體違和 0
病魔 61
病魔纏身 9
痆 0
痆 0
症 5083
症候 46
症候群 644
症狀 3255
痊 2543
痊癒 225
痋 795
痌 32
痍 328
痎 9
痏 4
痐 461
痑 252
痒 16
痒疹 0
痒痒 3
痓 0
痔 220
痔漏 3
痔瘡 106
痕 622
痕垢 0
痕跡 426
痗 169
痘 296
痘子 23
痘瘡 0
痘苗 0
痙 56
痙攣 263
痚 156
痛 2341
痛不欲生 25
痛不痛 6
痛了 64
痛入骨髓 0
痛切 8
痛哭 93
痛哭失聲 10
痛哭流涕 33
痛定思痛 26
痛宰 16
痛得 198
痛徹 11
痛心 122
痛心疾首 11
痛快 111
痛快淋漓 4
痛恨 94
痛悔 12
痛惜 14
痛惡 7
痛愛 1
痛感 18
痛打 11
痛打一頓 5
痛打一頓 5
痛批 0
痛揍 0
痛揍一頓 0
痛揍一頓 0
痛擊 19
痛改前非 3
痛斥 49
痛楚 85
痛死 8
痛毀極詆 0
痛痛 43
痛痛快快 15
痛癢 3
痛的 198
痛罵 16
痛罵一頓 1
痛罵一頓 1
痛自悔改 0
痛苦 1661
痛苦不堪 12
痛苦萬分 10
痛處 16
痛覺 27
痛責 5
痛起來 7
痛風 320
痛飲 12
痜 0
痝 138
痞 553
痞塊 0
痞棍 0
痟 183
痠 111
痠疼 5
痠痛 37
痠軟 3
痡 89
痢 77
痢疾 227
痣 114
痤 298
痥 0
痦 130
痧 43
痧子 0
痬 0
痭 58
痮 0
痯 38
痰 901
痰盂 7
痰罐 1
痱 10
痱子 12
痱子粉 6
痱粉 0
痲 202
痲子 2
痲疹 38
痲痹 18
痲痹不了 0
痲痺 44
痲痺不了 0
痲瘋 41
痲瘋病 4
痲瘋院 0
痲臉 0
痲類 0
痳 38
痳 38
痳病 0
痴 433
痴 433
痴人 20
痴人說夢 1
痴人說夢話 0
痴呆 32
痴呆症 89
痴心 59
痴心人 1
痴心妄想 2
痴情 49
痴想 3
痴痴 53
痴笑 0
痴肥 0
痵 32
痶 34
痷 50
痸 37
痹 18
痺 74
痺 74
痻 66
痼 23
痼 23
痼疾 18
痼癖 0
痼習 0
痽 26
痾 74
痿 26
瘀 155
瘀傷 10
瘀泥 0
瘀積 0
瘀膿 0
瘀血 40
瘀青 1
瘁 221
瘂 0
瘂 0
瘂 0
瘃 55
瘃 55
瘇 0
瘈 2
瘉 108
瘊 6
瘋 824
瘋了 196
瘋人 7
瘋人院 6
瘋到 3
瘋女 2
瘋婆子 5
瘋子 157
瘋得 1
瘋犬病 0
瘋狂 963
瘋狗 91
瘋狗浪 0
瘋病 7
瘋瘋 18
瘋瘋狂狂 0
瘋瘋癲癲 11
瘋癲 15
瘋言瘋語 6
瘋話 5
瘋語 2
瘋起來 0
瘋馬 20
瘌 7
瘌痢 0
瘌痢頭 0
瘍 324
瘎 0
瘏 49
瘐 56
瘐 56
瘑 7
瘒 0
瘓 166
瘔 72
瘕 7
瘖 14
瘙 133
瘙 133
瘙疹 0
瘙疹 0
瘚 301
瘛 120
瘜 184
瘝 62
瘞 152
瘟 76
瘟疫 76
瘟神 14
瘠 25
瘠土 0
瘠地 0
瘠田 1
瘠瘦 0
瘠薄 1
瘡 158
瘡口 1
瘡疤 13
瘡痍 3
瘡痍滿目 0
瘡癤 1
瘢 21
瘢兒 0
瘢兒 0
瘢痕 9
瘢點 0
瘣 107
瘤 963
瘤子 3
瘤胃 12
瘥 68
瘥 68
瘦 873
瘦下 15
瘦下來 10
瘦下去 1
瘦了 61
瘦到 0
瘦削 11
瘦子 16
瘦小 37
瘦弱 48
瘦弱不堪 0
瘦得 55
瘦成 1
瘦成皮包骨 0
瘦瘦 43
瘦瘦小小 1
瘦瘦怯怯 0
瘦瘦長長 0
瘦瘦高高 1
瘦的 55
瘦肉 48
瘦身 0
瘦金體 3
瘦長 20
瘦馬 4
瘦骨如柴 1
瘦骨嶙峋 4
瘦點 0
瘧 248
瘧疾 130
瘧蚊 8
瘧蟲 1
瘨 451
瘩 8
瘩 8
瘬 0
瘭 13
瘮 0
瘯 49
瘰 43
瘱 390
瘲 13
瘳 8
瘴 10
瘴氣 7
瘴癘 5
瘴癘之氣 0
瘵 23
瘶 0
瘶 0
瘷 0
瘸 41
瘸子 2
瘸腿 5
瘹 0
瘺 18
瘻 0
瘼 121
瘽 80
癀 0
癁 0
療 2006
療傷 111
療效 535
療毒 3
療治 14
療法 1079
療病 31
療養 151
療養所 2
療養醫院 0
療養院 314
癃 277
癄 0
癄 0
癅 0
癆 64
癆病 1
癇 40
癇症 8
癈 78
癉 40
癊 0
癋 0
癌 2575
癌 2575
癌症 2009
癌症病患 43
癌症腫瘤 13
癌腫 4
癎 0
癏 0
癐 9
癑 0
癑 0
癒 424
癒合 177
癒好 0
癒瘡 0
癓 3
癔 0
癕 0
癖 89
癖 89
癖好 23
癖性 1
癗 3
癘 9
癘 9
癙 2
癚 5
癛 0
癛 0
癜 22
癟 39
癟三 1
癟三兒 0
癟三兒 0
癟嘴子 0
癟著嘴 0
癠 5
癡 333
癡人 24
癡人說夢 14
癡人說夢話 0
癡呆 32
癡呆症 64
癡心 46
癡心人 0
癡心妄想 5
癡心漢 1
癡情 91
癡想 4
癡漢 9
癡狂 29
癡癡 60
癡癡傻傻 0
癡笑 0
癡肥 3
癡迷 22
癡鈍 0
癢 287
癢斯斯 0
癢癢 30
癤 24
癤 24
癤子 4
癥 49
癥候 7
癥狀 8
癥結 149
癧 0
癨 0
癩 294
癩子 0
癩瓜 0
癩瘡 1
癩皮狗 2
癩蛤蟆 18
癩蝦蟆 1
癩頭 3
癪 8
癬 132
癬疥 6
癭 23
癮 324
癮君子 18
癮頭 11
癯 0
癰 23
癰疽 6
癰瘡 0
癰腫 0
癱 85
癱下來 1
癱坐 0
癱廢 1
癱瘓 193
癱軟 9
癲 223
癲狂 12
癲癇 1527
癴 0
癵 38
癵 38
癶 0
癸 691
癸卯 5
癹 155
登 6917
登上 361
登上來 0
登上去 3
登了 38
登位 3
登個 1
登入 0
登出 62
登出來 11
登出去 0
登到 3
登台 135
登台演唱 1
登在 44
登基 54
登堂 8
登堂入室 7
登報 54
登報作廢 6
登報聲明 4
登場 154
登場 1540
登天 29
登山 1719
登山協會 21
登山涉嶺 0
登山隊 35
登山鞋 37
登岸 25
登峰造極 50
登州 5
登帳 14
登廣告 55
登徒子 3
登時 117
登月 23
登月小艇 3
登極 3
登樓 15
登機 103
登用 1
登科 83
登程 2
登空 0
登第 7
登臨 22
登臺 20
登舟 6
登船 20
登記 9887
登記卡 15
登記簿 346
登記處 23
登賬 0
登起 16
登起來 0
登載 234
登過 58
登錄 20689
登門 2722
登門拜訪 20
登陸 441
登陸作戰 19
登陸戰 4
登陸艇 15
登陸點 0
登革熱 673
登高 76
登高一呼 12
登高一呼 12
登龍 31
發 18848
發想 1000
發上 5
發上來 0
發上去 0
發下 30
發下來 2
發下去 0
發不 35
發不出 22
發不到 1
發不起 0
發乎情 2
發乾 3
發了 125
發予 35
發些 1
發亮 77
發人 40
發人深省 40
發付 4
發令 22
發佈 4447
發作 861
發來 8
發來發去 0
發信 2676
發信號 2
發個 18
發像管 0
發光 501
發光二極體 1
發光體 23
發兵 14
發冷 33
發出 2167
發出來 16
發出去 3
發函 154
發刊 54
發刊詞 64
發到 19
發動 868
發動機 142
發動機短艙 0
發包 521
發去 22
發向 1
發呆 197
發售 494
發售量 1
發問 793
發問者 84
發啞 2
發喘 2
發單 44
發嘔 0
發回 37
發回來 1
發回去 1
發報 20
發報員 0
發報器 15
發報機 3
發奮 5
發奮圖強 3
發好 3
發威 43
發家 3
發射 755
發射台 48
發射器 49
發射場 1
發射場 1
發射極 0
發射機 39
發展 47778
發展下去 22
發展不平均 0
發展不平衡 3
發展中國 12
發展中國家 60
發展出來 154
發展到 159
發展局 329
發展心理 11
發展心理學 42
發展性 42
發展潛力 213
發展為 250
發展生產 5
發展觀 3
發展起來 29
發展速度 15
發市 0
發布 1682
發引 3
發往 27
發得 291
發怒 69
發怔 5
發急 6
發怨言 11
發怵 0
發悶 1
發悸 0
發情 141
發情周期 1
發情期 11
發惱 1
發愁 12
發愣 0
發慌 20
發憤 8
發憤圖強 4
發憤忘食 0
發懶 0
發成 5
發抒 13
發抖 101
發指 4
發掉 0
發掘 496
發掘出 39
發掘出來 7
發掘到 19
發揚 365
發揚光大 100
發揚蹈厲 0
發揮 3478
發揮作用 33
發揮出 44
發揮出來 23
發揮到 51
發揮性 0
發揮所長 32
發收 2
發放 706
發散 36
發文 1683
發文者 49
發於 66
發明 1721
發明人 235
發明創造 6
發明家 28
發明者 22
發昏 6
發春 0
發暈 7
發暗 2
發書 14
發有 10
發條 104
發榜 4
發款 2
發毛 4
發水 16
發汗 22
發汗劑 0
發汗藥 0
發泄 15
發泡劑 14
發洋財 0
發洩 293
發派 4
發源 30
發源地 88
發潮 0
發火 72
發火器 0
發火點 0
發炎 537
發炎性 23
發炮 2
發為 21
發煙 25
發熱 129
發熱時 1
發熱量 12
發燒 1784
發燒時 24
發燒片 0
發片 0
發牌 16
發牢騷 46
發狂 47
發狠 3
發獃 5
發獎 5
發現 14807
發現出 3
發現到 138
發球 284
發球員 35
發生 13932
發生器 15
發生爐煤氣 3
發生關係 29
發疹 20
發疹子 1
發病 440
發病率 57
發痛 4
發痧 0
發瘋 112
發癡 0
發癢 42
發癬 0
發白 26
發白時 0
發的 291
發皇 4
發直 9
發矢 1
發砲 3
發硬 3
發硬時 0
發神經 6
發神經病 0
發祥 21
發祥地 25
發票 1260
發福 15
發稿 120
發端 14
發笑 18
發筆 0
發粉 14
發糕 2
發紅 34
發紫 12
發給 1282
發綠 5
發聲 212
發聲器 2
發聲器官 0
發育 711
發育不良 53
發育成 12
發育期 16
發育條件 0
發胖 37
發脹 5
發脹感 0
發脾氣 54
發腫 5
發自 54
發自內心 47
發自肺腑 2
發臭 14
發船 3
發花 2
發芽 430
發芽勢 0
發芽率 53
發菜 1
發落 9
發葉 1
發著 41
發薪 13
發薪水 7
發藍 2
發號施令 17
發蠟 0
發行 7088
發行人 417
發行者 30
發行量 148
發表 34052
發表出來 19
發表意見 3272
發表會 832
發表聲明 166
發見 22
發覺 970
發覺出 1
發覺到 52
發言 7912
發言人 4194
發言條 5
發言權 55
發言者 1850
發話器 0
發誓 128
發語詞 0
發議論 3
發財 140
發貨 104
發貨單 1
發賣 13
發起 588
發起人 209
發起來 1
發起燒來 1
發跡 31
發蹟 2
發車 88
發軔 19
發軟 27
發送 814
發送機 0
發進 11
發進來 0
發進去 0
發運 2
發過 21
發過來 2
發過去 1
發達 669
發達到 3
發達國家 25
發達為 1
發達起來 2
發還 148
發配 8
發酒瘋 3
發酵 495
發酵出來 0
發酵粉 0
發酵飼料 0
發酸 13
發野 0
發難 27
發電 867
發電廠 166
發電所 40
發電機 361
發電站 12
發電量 64
發霉 73
發青 13
發面 3
發音 826
發音器 3
發音器官 4
發音學 5
發響 0
發願 105
發顫 5
發顫 5
發飆 56
發高熱 3
發高燒 41
發麵 2
發麻 30
發黃 16
發黏 0
發黑 25
發點 15
發黴 9
白 102
白 10221
白鯊 289
白丁 1
白乾 0
白亮亮 0
白人 212
白令海 2
白令海峽 5
白俄 0
白俄羅斯 79
白做 7
白光 55
白兔 67
白內障 228
白刃 9
白努力 3
白包 3
白化 83
白匪 0
白區 2
白卷 4
白口 13
白口鐵 0
白吃 55
白吃白喝 15
白喉 112
白地 12
白堊 5
白堊紀 39
白堊質 0
白報紙 3
白塔 8
白塔寺 0
白墨 6
白夜 3
白天 1184
白子 58
白字 28
白宮 628
白居易 36
白屋 5
白布 46
白帝 6
白帝城 8
白帶 44
白帶魚 38
白干兒 0
白干兒 0
白平衡 1
白底 44
白底黑字 12
白得 9
白得 9
白忙 19
白手 19
白手成家 0
白手起家 24
白打 4
白描 8
白搭 4
白文 30
白斑 78
白斑症 3
白族 7
白旗 22
白日 72
白日夢 255
白晝 77
白朗 10
白朗峰 5
白木耳 14
白木耳三角 0
白板 121
白板筆 1
白果 32
白果樹 1
白楊 51
白楊木 2
白楊樹 7
白樸 2054
白樺 18
白殼蛋 0
白毛 55
白毛女 2
白水 83
白沙 289
白沙灣 41
白沙瓦 0
白沙糖 0
白沫 5
白沫子 0
白河 448
白活 6
白浪滔天 0
白淨 30
白潔 0
白澳 1
白澳政策 1
白濁 8
白濛濛 1
白灰 9
白煤 0
白煮 0
白熊 24
白熱 16
白熱化 92
白熾燈 15
白牡丹 5
白猴 11
白玉 135
白璧微瑕 0
白痢 5
白痴 291
白癡 268
白癬 3
白白 139
白白的 63
白白地 63
白白胖胖 2
白的 242
白地 242
白皙 48
白皚皚 2
白皮 27
白皮書 1057
白皮松 0
白相 10
白看 5
白眼 35
白眼珠 0
白矮星 10
白砂 36
白磷 1
白礬 1
白種 12
白種人 10
白筆 2
白米 105
白粉 92
白粉筆 0
白糖 27
白紗 290
白紙 77
白紙黑字 28
白細胞 3
白練 6
白羊 393
白老鼠 1
白肉 22
白脫 2
白臉 26
白色 2054
白色恐怖 164
白花 79
白花油 3
白花花 4
白茫茫 18
白菊 8
白菊花 12
白菜 90
白菜湯 0
白葡萄 1
白葡萄酒 1
白蓮教 16
白薯 0
白蘆筍 0
白蘭 60
白蘭地 142
白蘭花 3
白蘿蔔 1
白虎 30
白虎星 0
白蛇 20
白蛇傳 23
白蛋白 40
白螞蟻 2
白蟻 99
白蠟 5
白蠟樹 0
白蠟蟲 0
白血 9
白血球 285
白血病 151
白衣 240
白衣大士 28
白衣天使 11
白話 215
白話文 41
白話文學 3
白話詩 15
白說 11
白貓 15
白費 66
白費力氣 8
白費口舌 0
白費唇舌 0
白費心機 5
白質 41
白賺 2
白走 7
白跑一趟 9
白軍 0
白送 2
白道 18
白遼士 26
白邊 12
白酒 119
白醋 13
白醣 1
白金 473
白金漢宮 29
白鉛礦 0
白銀 57
白銅 5
白鍵 2
白鐵 53
白鐵皮 2
白開水 74
白陶 2
白雪 181
白雪公主 72
白雲 481
白雲母 6
白雲片片 0
白雲石 31
白雲蒼狗 1
白露 39
白面 41
白面書生 0
白領 17
白領階級 17
白頭 54
白頭偕老 19
白頭到老 7
白頭翁 73
白食 1
白飯 40
白首 20
白馬 161
白馬寺 3
白馬王子 40
白駒過隙 3
白骨 33
白髮 123
白髮蒼蒼 10
白鯨 19
白鳥 71
白鴿 38
白鵝 8
白鵠 0
白鶴 72
白鷺 210
白麵 4
白麵兒 0
白麵兒 0
白點 50
白鼠 11
百 92
百 9298
百一 79
百丈 33
百下 4
百世 79
百乘 1
百事 93
百二 374
百人 148
百代 167
百代過客 0
百位 120
百位數 4
百依百順 7
百倍 86
百元 67
百元大鈔 2
百兩 18
百出 59
百分 5100
百分之一百 43
百分之一百 43
百分之百 379
百分數 5
百分比 939
百分法 6
百分率 74
百分表 0
百分點 418
百利 102
百劫紅顏 0
百十 174
百口莫辯 6
百吃不厭 5
百合 407
百合花 74
百善孝為先 6
百壽圖 0
百夫長 18
百姓 612
百孔千瘡 2
百官 22
百家 101
百家姓 13
百家爭鳴 51
百密一疏 3
百密一疏 3
百尺 8
百尺竿頭 17
百工 49
百年 1087
百年大計 39
百年好合 17
百年後 22
百年戰爭 2
百年樹人 43
百年紀念 24
百度 30
百度表 0
百廢俱興 0
百廢俱舉 0
百廢待興 6
百廢待舉 15
百弊叢生 0
百忙中 47
百思 38
百思不得其解 13
百思莫解 1
百感 14
百感交集 22
百態 73
百態橫生 0
百慕達 35
百慕達三角洲 1
百戰 46
百戰天龍 10
百戰百勝 37
百戶 61
百折不回 2
百折不撓 7
百摺裙 3
百日 70
百日咳 152
百日紅 8
百景 63
百會 6
百果 11
百果山 34
百業振興 0
百樂百福 0
百步 75
百步穿楊 3
百步蛇 1
百歲 105
百歲誕辰 6
百濟 4
百無一失 1
百無一失 1
百無一用 0
百無一用 0
百無一用是書生 1
百無一用是書生 1
百無一長 0
百無禁忌 96
百無聊賴 4
百物 2
百獸 5
百獸之王 1
百病 49
百病叢生 10
百發 5
百發百中 34
百看不厭 37
百科 2547
百科全書 537
百節 2
百米 43
百米賽跑 2
百結 0
百美圖 0
百老 6
百老匯 154
百聞 1
百聞不如一見 15
百般 109
百色 2
百花 174
百花怒放 3
百花爭艷 2
百花齊放 28
百草 17
百萬 9161
百萬富翁 14
百萬雄師 0
百葉 73
百葉窗 76
百號 14
百衲本 0
百衲衣 1
百裡 5
百褶裙 3
百計千方 0
百讀不厭 3
百貨 3934
百貨公司 917
百貨商場 92
百貨商場 92
百貨商店 6
百貨店 46
百貨業 373
百足 3
百身莫贖 0
百部 19
百里 171
百里香 15
百鍊成鋼 1
百集 7
百靈 26
百靈花 0
百靈鳥 3
百香 12
百香果 53
百香果汁 7
百骸 10
百鳥 11
百點 43
癿 1
皀 0
皀 0
皀 0
皁 14
皁莢 0
皂 220
皂化 8
皂化劑 0
皂洗 44
皂片 0
皂白 8
皂白不分 0
皂粉 0
皂靴 1
皃 0
的 14727
的 14727
的 147270
的的確確 15
的確 1558
的確如此 12
的確是 424
的確會 35
的確良 4
的話 10721
的黎波里 2
的黎波里 2
皅 0
皅 0
皆 19425
皆使 27
皆可 1347
皆大歡喜 31
皆如 27
皆宜 90
皆應 86
皆是 517
皆有 860
皆為 696
皆用 23
皆知 91
皇 3234
皇上 61
皇位 11
皇儲 1
皇冠 1747
皇冠 1747
皇叔 13
皇后 303
皇土 0
皇天 39
皇天后土 6
皇太后 9
皇太子 6
皇太極 4
皇室 78
皇宮 196
皇家 627
皇家學院 11
皇帝 638
皇恩 7
皇恩浩蕩 0
皇族 23
皇族內閣 0
皇曆 0
皇朝 30
皇權 8
皇民 67
皇民奉公會 1
皇甫嵩 7
皇皇巨著 1
皇考 5
皇莊 1
皇親 4
皇親國戚 2
皇陵 7
皈 1035
皈依 143
皉 608
皊 161
皋 77
皌 0
皍 0
皍 0
皎 186
皎如日星 0
皎月 2
皎潔 34
皎皎 7
皏 282
皐 0
皐 0
皒 107
皓 554
皓月 10
皓月千里 0
皓月當空 3
皓然 2
皓白 1
皓皓 18
皓首 1
皓首窮經 2
皓髮 0
皓齒 4
皔 0
皕 226
皖 32
皖 32
皖北 1
皖南 4
皗 0
皘 0
皙 49
皚 17
皚皚 18
皛 79
皜 59
皝 147
皞 65
皟 0
皠 0
皡 0
皡 0
皢 0
皣 0
皤 11
皤皤 0
皤翁 0
皥 0
皥 0
皦 8
皧 0
皨 0
皩 0
皪 12
皫 27
皬 0
皭 7
皮 3981
皮下 109
皮下注射 41
皮下組織 45
皮件 366
皮包 270
皮包骨 10
皮匠 2
皮厚 9
皮厚餡少 0
皮囊 7
皮墊 7
皮墊圈 0
皮大衣 3
皮夾 132
皮夾克 11
皮夾子 2
皮套 263
皮子 17
皮孔 1
皮層 64
皮帶 267
皮帶傳動 7
皮帶扣 0
皮帶輪 6
皮店 1
皮影戲 65
皮得 2
皮得很 0
皮書 34
皮條 2
皮條客 5
皮棉 0
皮殼 0
皮毛 37
皮毛之見 0
皮毛衣 0
皮炎 10
皮爾斯 5
皮爾遜 5
皮球 38
皮疹 64
皮癢 13
皮皮 211
皮相 5
皮相之士 0
皮破血流 4
皮笑 1
皮笑肉不笑 5
皮筏 3
皮箱 101
皮箱子 1
皮紙 9
皮肉 31
皮肉之傷 3
皮肉傷 2
皮肉生涯 0
皮脂 77
皮脂腺 40
皮膚 3273
皮膚病 203
皮膚癌 57
皮膚癌 57
皮膚科 643
皮膚過敏 32
皮膜 106
皮舟 0
皮草 26
皮薄 11
皮薄 11
皮薄餡多 0
皮薄餡多 0
皮蛋 87
皮衣 92
皮袋 13
皮裏陽秋 0
皮裡春秋 0
皮製 40
皮製品 26
皮襖 5
皮貨 1
皮貨商 0
皮貨店 0
皮質 231
皮輥 1
皮開肉綻 4
皮面 17
皮革 699
皮革商 0
皮革業 15
皮靴 10
皮靴子 0
皮鞋 432
皮鞋油 1
皮鞭 17
皮黃 27
皮鼓 5
皯 4
皰 63
皰疹 168
皳 0
皴 15
皴法 1
皵 33
皶 0
皷 0
皸 14
皹 0
皺 326
皺巴巴 4
皺摺 29
皺痕 0
皺皺巴巴 0
皺眉 81
皺眉頭 21
皺紋 196
皺胃 2
皺著 19
皺著眉頭 22
皺褶 19
皺起 28
皻 910
皽 4
皾 8
皿 342
皿 342
盀 0
盁 0
盂 376
盂蘭盆會 2
盃 2699
盄 98
盅 69
盆 1524
盆兒 0
盆兒 0
盆地 346
盆子 12
盆景 90
盆栽 253
盆浴 15
盇 0
盇 0
盈 3757
盈利 115
盈月 2
盈溢 8
盈滿 12
盈盈 104
盈盈的 17
盈盈地 17
盈缺 0
盈虛 5
盈虧 95
盈虧自負 2
盈貫 0
盈門 3
盈餘 1601
盉 52
益 8455
益加 27
益友 36
益壽 25
益州 11
益智 542
益智圖 0
益發 64
益處 159
益蟲 4
益都 1
益鳥 0
益鳥益蟲 0
盋 0
盋 0
盌 0
盍 40
盎 165
盎司 147
盎司換算法 0
盎斯 19
盎格魯薩克遜 2
盎然 73
盒 2353
盒中 58
盒內 38
盒子 287
盒子 987
盒子槍 1
盒子砲 0
盒蓋 8
盒裝 183
盒裡 23
盓 170
盔 104
盔甲 41
盕 0
盖 0
盖 0
盖 0
盗 0
盙 0
盚 14
盛 14750
盛 6419
盛世 91
盛了 7
盛事 92
盛京 5
盛傳 79
盛典 7
盛到 1
盛反 0
盛名 184
盛名之累 1
盛唐 16
盛器 5
盛在 18
盛夏 59
盛大 287
盛大舉行 26
盛好 1
盛宣懷 5
盛宴 112
盛年 9
盛得 2
盛德遺範 0
盛怒 10
盛怒之下 9
盛情 41
盛情難卻 5
盛意 2
盛收 0
盛暑 5
盛會 198
盛服 0
盛業 15
盛極一時 24
盛極一時 24
盛極必衰 0
盛極而衰 2
盛氣 2
盛氣凌人 6
盛氣臨人 0
盛況 139
盛況空前 23
盛產 159
盛筵 2
盛筵難再 0
盛舉 10
盛菜 0
盛著 12
盛行 479
盛行一時 4
盛行一時 4
盛行於 24
盛衰 30
盛裝 128
盛譽 9
盛讚 73
盛贊 1
盛起 13
盛過 0
盛過來 0
盛過去 0
盛開 148
盛開著 2
盛飯 6
盜 65350
盜亦有道 0
盜伐 10
盜伐 10
盜匪 32
盜印 10
盜去 0
盜取 24
盜名 3
盜墓 17
盜壘 27
盜拷 0
盜林 1
盜款 0
盜汗 16
盜版 204
盜版書 3
盜用 193
盜用公款 5
盜竊 24
盜竊罪 1
盜賊 136
盜賣 17
盜賣公物 0
盜走 4
盜領 15
盜首 1
盝 98
盞 116
盟 2830
盟主 106
盟兄弟 0
盟友 54
盟員 23
盟國 152
盟旗 1
盟書 0
盟機 25
盟約 58
盟誓 2
盟軍 73
盟軍部隊 0
盟邦 15
盠 0
盡 17804
盡了 143
盡些 4
盡人事 9
盡人事聽天命 3
盡人皆知 3
盡付東流 0
盡信 18
盡做 6
盡做壞事 0
盡先 1
盡全力 68
盡全心 1
盡其 83
盡其在我 6
盡其所有 0
盡其所能 30
盡其所長 0
盡到 51
盡力 510
盡力去做 13
盡力而為 25
盡可 23
盡可能 225
盡善 9
盡善盡美 60
盡在 202
盡在不言中 35
盡失 33
盡如 7
盡如人意 19
盡孝 14
盡屬 3
盡得 7
盡心 135
盡心盡力 118
盡心竭力 12
盡忠 28
盡忠報國 1
盡忠職守 13
盡快 377
盡快的 14
盡快地 14
盡性 21
盡情 566
盡情歌唱 0
盡情歡唱 4
盡情歡樂 4
盡情玩樂 4
盡意 14
盡散 5
盡數 32
盡早 43
盡是 155
盡本分 6
盡歡 6
盡然 43
盡瘁 4
盡盡 3
盡端 4
盡管 120
盡義務 6
盡職 54
盡能 7
盡自 30
盡致 14
盡興 104
盡興而歸 7
盡若 0
盡要 0
盡讀 0
盡責 56
盡責任 13
盡量 1033
盡頭 237
盢 0
監 449
監 4491
監事 1155
監修 24
監印 12
監國 2
監場 10
監場 10
監委 616
監守自盜 11
監宮 0
監察 391
監察人 196
監察員 5
監察委員 186
監察官 10
監察權 28
監察院 939
監察院長 111
監工 102
監房 0
監押 2
監控 0
監測 1665
監測器 45
監牢 15
監犯 0
監獄 431
監獄長 0
監理所 134
監督 2322
監督人 6
監票 15
監票人 1
監禁 40
監管 224
監管不周 0
監考 50
監考人 1
監考官 1
監聽 87
監聽器 9
監聽系統 0
監製 80
監視 970
監視器 1046
監試 29
監課 0
監課所 0
監護 185
監護人 203
監軍 4
監造 171
監院 292
監靶手 0
盤 7903
盤上 60
盤住 1
盤剝 0
盤古 29
盤古開天 16
盤問 29
盤坐 7
盤坐在 0
盤坐著 0
盤子 78
盤存 30
盤尼西林 34
盤山 14
盤庚 0
盤庫 0
盤弄 0
盤成 6
盤據 37
盤整 266
盤旋 113
盤旋曲折 1
盤曲 2
盤查 27
盤根究底 0
盤根錯節 11
盤桓 18
盤桓著 0
盤槓子 0
盤石 8
盤算 68
盤算著 20
盤結 8
盤繞 10
盤繞在 0
盤繞著 2
盤纏 10
盤腿 15
盤膝 22
盤膝而坐 2
盤裡 8
盤詰 3
盤谷 17
盤貨 0
盤費 3
盤賬 0
盤起 6
盤踞 26
盤迴 0
盤錯 0
盤面 112
盤香 1
盤馬彎弓 0
盤點 160
盥 35
盥洗 75
盥洗室 21
盥漱 2
盦 22
盧 2606
盧布 134
盧梭 46
盧森堡 111
盧森堡人 0
盧比 19
盧江 0
盧溝橋 9
盧薩卡 0
盨 0
盩 8
盪 365
盪來盪去 3
盪出 3
盪口 0
盪婦 0
盪平 0
盪氣迴腸 7
盪滌 0
盪漾 20
盪盪 20
盪舟 1
盪著 7
盪鞦韆 30
盫 0
盬 9
盭 6
目 16274
目下 77
目不交睫 1
目不斜視 9
目不暇接 3
目不暇給 61
目不識丁 4
目不轉睛 41
目中 121
目中無人 15
目使頤令 0
目光 676
目光如炬 5
目光如豆 3
目光炯炯 6
目光犀利 0
目光短淺 0
目光遠大 3
目前 34892
目前為止 646
目力 14
目指氣使 0
目擊 226
目擊者 91
目標 11009
目標線 4
目標點 2
目次 3450
目測 52
目測點 0
目無全牛 0
目無尊長 3
目無法紀 4
目的 8272
目的地 336
目的物 1
目眩 30
目眩神迷 7
目眩頭暈 1
目眶 6
目眶 6
目睹 297
目瞪口呆 68
目空一切 2
目空一切 2
目色 0
目蓮救母 3
目迎 0
目迎目送 0
目送 49
目部 7
目錄 22999
目錄學 55
目鏡 66
盯 160
盯 160
盯上 15
盯人 10
盯住 15
盯梢 3
盯緊 5
盯著 210
盯視 3
盰 3
盱 42
盲 1038
盲人 365
盲人摸象 1
盲動 3
盲動主義 0
盲啞 0
盲啞學校 3
盲啞教育 0
盲女 4
盲從 33
盲生 157
盲目 269
盲目崇拜 4
盲目性 1
盲目服從 0
盲者 11
盲聾學校 3
盲胞 0
盲腸 52
盲腸炎 17
盲點 142
盳 7
直 9135
直上 73
直下 33
直中 3
直了 13
直了 13
直交 11
直來直往 10
直傳 6
直到 3109
直升 162
直升機 152
直升飛機 2
直去 10
直呼 35
直呼其名 0
直喊 14
直喻 0
直奔 98
直寫 12
直射 55
直射光 2
直對 48
直尺 13
直屬 118
直屬長官 4
直布羅陀 15
直往 43
直徑 702
直得 20
直得 58
直心眼 0
直性 8
直性子 0
直愕愕 0
直截了當 29
直把 50
直抵 24
直拳 2
直挺 6
直挺挺 8
直挺挺的 3
直挺挺地 3
直排溜冰鞋 0
直排輪 0
直接 11363
直接染料 53
直接稅 26
直接經驗 4
直接証明 0
直接證明 2
直接選舉 30
直搗 23
直搗黃龍 0
直撞 7
直撞 7
直撥 141
直播 196
直播 196
直播站 3
直播站 3
直撲 29
直擺 1
直昇 59
直昇機 216
直昇飛機 0
直書 20
直望 6
直條 11
直流 346
直流發電機 6
直流電 29
直流電動機 6
直流電機 3
直營 204
直爽 13
直率 40
直球 21
直登 5
直的 58
直眉瞪眼 0
直瞪瞪 1
直立 233
直立起來 2
直笛 0
直筆 1
直系 118
直系親 4
直系親屬 91
直線 539
直脾氣 0
直腸 289
直腸癌 101
直至 430
直航 0
直落 67
直落到 0
直落在 0
直著 3
直行 81
直行 81
直衝 39
直視 39
直覺 381
直覺反應 5
直覺性 6
直觀 47
直觀圖 0
直角 114
直角三角 1
直角三角形 12
直角坐標 2
直角座標 12
直言 78
直言不諱 8
直言無諱 6
直言無隱 1
直話直說 0
直說 144
直諫 3
直講 5
直譯 31
直譯器 15
直譯式 6
直譯碼 0
直讓 21
直走 93
直起 5
直起來 2
直越 1
直趨 8
直路 22
直跳 9
直跳腳 2
直躺 1
直轄 82
直轄市 803
直轉 8
直轉 8
直轟 1
直追 25
直退 0
直通 134
直通通 0
直進 11
直道 10
直達 147
直達車 17
直選 204
直銷 513
直鋪 2
直鏈 11
直開 9
直闖 9
直陳 8
直隸 32
直須 4
直飛 87
盵 3
盶 0
盷 35
相 12168
相 12168
相並 2
相中 77
相乘 57
相互 2207
相互作用 58
相互依賴 22
相互討論 14
相互關係 94
相交 129
相交多年 3
相人 7
相仿 54
相伴 75
相似 684
相似形 6
相位 119
相位差 15
相依 69
相依為命 35
相依相偎 6
相信 7512
相傳 210
相傳多年 0
相傳已久 0
相像 26
相公 51
相切 8
相別 2
相別多年 0
相剋 47
相加 80
相助 91
相勸 12
相去 55
相去無幾 4
相去甚遠 22
相反 644
相反地 105
相反地 105
相反相成 0
相反詞 2
相合 57
相同 4806
相同於 11
相向 31
相吸 14
相告 17
相命 14
相商 4
相善 1
相因而生 0
相國 16
相士 22
相夫教子 13
相好 1
相好 27
相安 5
相安無事 22
相宜 71
相容 977
相容性 367
相對 3697
相對主義 10
相對式 2
相對性 17
相對性狀 0
相對數 4
相對極 0
相對溼度 10
相對濕度 87
相對而坐 0
相對誤差 0
相對論 83
相對速度 6
相對運動 23
相對高度 5
相左 27
相差 245
相差不多 21
相差太多 12
相差太大 11
相差無幾 15
相幫 4
相干 38
相平衡 17
相府 11
相形 32
相形之下 57
相形見絀 2
相待 76
相得 4
相得益彰 37
相思 413
相思子 0
相思樹 53
相思病 6
相思豆 4
相惜 49
相愛 283
相愛多年 0
相愛已久 0
相應 245
相應不理 8
相成 18
相打 3
相托 2
相投 16
相抵 54
相持 5
相持不下 7
相捋 0
相接 75
相提並論 81
相撞 40
相撲 33
相撲 33
相擁 53
相敬 6
相敬如賓 9
相映 23
相映成趣 16
相書 9
相會 103
相望 42
相本 1
相機 2390
相機而動 0
相機行事 1
相比 386
相求 10
相沖 28
相沿 8
相沿成俗 1
相消 2
相混 16
相減 22
相濡以沫 11
相煩 1
相爭 47
相父 2
相爺 1
相片 1727
相片兒 0
相片兒 0
相率 3
相生相剋 7
相異 90
相當 9461
相當於 685
相當程度 201
相相如生 0
相看兩不厭 2
相知 96
相碰 9
相碰撞 1
相稱 60
相稱 60
相符 314
相符合 40
相等 228
相等值 0
相等於 28
相簿 298
相約 178
相紙 74
相繼 556
相續 35
相罵 10
相習 0
相聚 243
相聲 110
相背 6
相與 44
相若 18
相著 1
相處 971
相術 10
相衝 72
相補 3
相見 313
相見恨晚 15
相視 23
相視而笑 10
相親 128
相親相愛 19
相親相聚 0
相許 54
相認 23
相談 28
相談甚歡 25
相識 212
相識已久 1
相變 30
相讓 9
相象 0
相貌 65
相貌堂堂 4
相貌端正 0
相責 0
相貼 5
相賴 0
相距 102
相距千里 0
相較 737
相輔相成 163
相輔而行 0
相迎 37
相近 369
相送 124
相通 139
相逢 242
相逢自是有緣 27
相連 1171
相遇 432
相違 15
相適應 9
相邀 21
相鄰 183
相配 23
相里 1
相間 69
相間 69
相關 77439
相關企業 66
相關式 2
相關性 233
相除 15
相陪 19
相隔 132
相隔不久 0
相隔多年 1
相隨 112
相離 14
相電流 0
相靠 0
相面 0
相類 23
相顧 21
相顧失色 1
相館 26
相鬥 16
相黏 2
盹 110
盹 11
盹兒 1
盹兒 1
盺 44
盻 29
盻 29
盼 1952
盼了 0
盼到 15
盼望 634
盼望著 15
盽 0
盾 4
盾 455
盾板 1
盾牌 55
盾狀 2
盾狀腺 0
盾皮魚類 0
盿 0
眀 0
省 1766
省 17667
省下 276
省下來 11
省下去 1
省中 85
省主席 247
省了 58
省事 741
省份 69
省來 3
省來省去 0
省個 4
省儉 0
省出 4
省出來 0
省分 67
省到 6
省力 144
省區 31
省卻 57
省去 145
省吃 6
省吃儉用 16
省名 6
省在 19
省城 10
省墓 0
省委 82
省察 34
省市 740
省府 2567
省庫 74
省得 350
省思 887
省悟 11
省成 12
省掉 31
省改 2
省政 370
省政信箱 8
省政府 667
省文 38
省料 1
省時 117
省時省力 25
省時間 17
省會 62
省民 319
省民同胞 0
省油 147
省油燈 2
省油的燈 13
省治 0
省減 0
省煤器 0
省營 116
省產 117
省產水果 2
省用 2
省界 1
省略 229
省略不提 0
省略到 1
省的 350
省省 45
省立 4232
省籍 425
省級 202
省級機構 60
省縣 136
省縣市 23
省視 21
省親 6
省議員 578
省議會 372
省軍區 1
省轄市 84
省運會 0
省過 2
省道 271
省錢 297
省長 1393
省電 0
省點 10
省黨部 116
眂 0
眃 22
眄 5
眄睨 0
眅 173
眆 0
眇 44
眇小 0
眇眇 1
眈 432
眉 1181
眉來眼去 9
眉宇 22
眉宇之間 9
眉峰 2
眉形 5
眉心 25
眉急 0
眉批 23
眉月 3
眉梢 11
眉毛 89
眉清目秀 11
眉目 20
眉目之間 1
眉目傳情 5
眉目清秀 2
眉眼 7
眉睫 7
眉筆 13
眉開眼笑 13
眉頭 82
眉額 0
眉飛色舞 8
眉高眼低 0
眉黛 0
眊 50
看 282
看 28254
看一下 206
看一看 154
看上 310
看上來 1
看上去 36
看下 456
看下來 5
看下去 37
看不 437
看不上眼 2
看不下去 72
看不出 210
看不出來 97
看不到 860
看不完 22
看不慣 47
看不見 421
看不起 69
看不透 6
看不過 45
看不過去 31
看中 112
看也不看 10
看了 2071
看了又看 9
看事做事 0
看些 12
看人 57
看人 57
看人嘴臉 0
看人而定 0
看人臉色 4
看似 353
看低 5
看住 7
看作 72
看來 2364
看來看去 26
看做 28
看出 1347
看出來 53
看出去 9
看到 12949
看去 74
看台 77
看向 20
看圖 235
看圖猜謎 0
看圖識字 5
看在 244
看報 42
看天 62
看天田 6
看好 1198
看守 96
看守所 254
看完 2768
看官 75
看家 35
看家本領 20
看店 11
看待 493
看得 1114
看得出 103
看得出來 119
看得到 224
看得懂 79
看得見 163
看得起 21
看得遠 3
看情形 22
看情形而定 3
看慣 13
看懂 65
看成 102
看成是 61
看戲 115
看扁 8
看手相 5
看押 0
看招 32
看是 103
看書 400
看望 45
看板 3243
看板 3243
看樣子 88
看法 3342
看海 81
看清 173
看清楚 228
看準 70
看漲 41
看熱鬧 47
看牢 0
看球 42
看球賽 16
看病 261
看的 1113
看相 31
看看 6704
看眼色 0
看破 56
看破紅塵 2
看穿 71
看管 39
看膩 19
看臺 5
看船 1
看花 16
看花了 3
看著 2144
看著辦 29
看見 2630
看視 5
看護 162
看護兵 0
看護法 0
看走 0
看走了眼 2
看走眼 3
看起 60
看起來 1401
看跌 12
看輕 21
看透 63
看過 1872
看過來 200
看過去 19
看醫生 161
看重 98
看錯 94
看門 13
看門狗 6
看開 21
看開一點 11
看電影 445
看電視 315
看頭 77
看顧 62
看風使舵 0
看風水 7
看風轉舵 0
看齊 51
眎 0
眏 0
眐 136
眑 128
眒 522
眓 74
眔 0
眕 1190
眖 0
眗 0
眗 0
眘 0
眙 178
眚 167
眛 121
眜 0
眝 59
眞 0
真 18312
真不 489
真不少 16
真不愧是 33
真不是 36
真不該 6
真主 10
真事 3
真人 338
真人不露相 1
真人真事 27
真似 8
真個 19
真值表 9
真假 177
真假難分 8
真偽 117
真偽莫辨 0
真傳 39
真像 88
真刀 1
真刀真槍 1
真分數 5
真切 64
真受不了 17
真叫人 35
真可 68
真名 55
真名真姓 0
真命天子 31
真品 40
真善美 417
真圓 38
真夠 56
真夠看的 0
真夠瞧的 0
真好 485
真如 102
真實 2362
真實性 105
真實感 27
真對 45
真工夫 1
真巧 35
真差 6
真平 2
真強 10
真得 159
真心 755
真心實意 0
真心真意 35
真心話 52
真怕 25
真性 27
真情 1564
真情流露 42
真情流露 42
真想 227
真意 46
真愛 594
真憑實據 2
真成 5
真成了 1
真我 44
真才實學 7
真摯 111
真數 1
真新 3
真早 5
真是 4964
真是的 176
真書 9
真會 86
真會是 0
真有 347
真有兩下子 1
真有兩把刷子 0
真有此事 3
真服了 7
真槍 40
真槍實彈 13
真正 6079
真正是 45
真死 8
真清 3
真為 57
真烈 1
真無 23
真無此事 0
真熟 0
真熟 0
真率 2
真珠 70
真理 1311
真理報 2
真理必勝 0
真瘦 0
真的 10008
真的是 1274
真皮 239
真皮組織 3
真直 8
真相 774
真相大白 35
真相畢露 0
真真 93
真真假假 11
真真正正 9
真知 26
真知灼見 16
真確 19
真神 171
真神阿拉 0
真空 1065
真空地帶 0
真空泵 7
真空管 64
真笨 19
真箇 3
真糟 3
真糟糕 8
真絲 8
真緊 1
真義 92
真舊 1
真草 0
真菌 128
真菌類 3
真菜 8
真行 9
真衰 5
真衰 5
真被 19
真要 147
真要命 2
真話 19
真誠 480
真誠待人 0
真諦 132
真讓 35
真讓人 22
真象 82
真貌 21
真貨 2
真貴 6
真跡 34
真蹟 11
真身舍利 1
真輕 0
真近 1
真遠 2
真重 7
真金 5
真金不怕火煉 5
真金烈火 0
真除 13
真險 0
真難 73
真面目 164
真響 0
真香 1
真髓 4
真鮮 2
眠 948
眠思夢想 0
眡 0
眢 211
眣 313
眤 0
眥 182
眥裂 1
眦 0
眧 4
眨 145
眨動 12
眨巴 1
眨眼 73
眨眼睛 9
眩 235
眩亂 0
眩人 4
眩惑 3
眩暈 91
眩目 32
眩眼 0
眩耀 3
眪 0
眫 0
眭 740
眮 0
眯 99
眰 0
眱 58
眲 93
眳 1203
眴 93
眵 158
眶 1059
眷 825
眷區 3
眷口 24
眷屬 439
眷念 13
眷愛 2
眷懷 0
眷戀 84
眷戀著 6
眷村 515
眷注 0
眷糧 1
眷顧 67
眸 281
眸子 42
眹 67
眺 186
眺望 222
眺望台 6
眺望塔 2
眺望樓 0
眺覽 2
眻 334
眼 3445
眼下 32
眼不見為淨 6
眼中 811
眼中人 2
眼中刺 1
眼中釘 12
眼光 833
眼光菩薩 1
眼冒金星 11
眼到 3
眼前 1000
眼力 47
眼圈 108
眼孔 1
眼尖 21
眼屎 9
眼岔 0
眼巴巴 6
眼帘 7
眼底 157
眼底下 0
眼影 46
眼手並用 0
眼拙 2
眼明人 6
眼明手快 15
眼時 8
眼暈 0
眼波 24
眼波 24
眼淚 663
眼淚汪汪 1
眼淺 0
眼炎 2
眼熟 20
眼熟 20
眼熱 0
眼珠 64
眼珠兒 0
眼珠兒 0
眼珠子 20
眼球 320
眼生 8
眼界 70
眼病 16
眼白 14
眼皮 124
眼皮跳 2
眼目 37
眼看 238
眼看四方 0
眼看四面 0
眼看著 94
眼眶 129
眼眶 129
眼眶裡 5
眼眸 69
眼睛 3342
眼睜睜 86
眼睫毛 4
眼瞎 8
眼瞳 10
眼瞼 96
眼神 568
眼福 43
眼福不淺 1
眼科 1419
眼科醫生 14
眼窩 44
眼簾 36
眼紅 51
眼結膜 2
眼結膜 2
眼線 35
眼罩 16
眼臉 7
眼色 41
眼花 69
眼花撩亂 61
眼花繚亂 4
眼藥 32
眼藥水 28
眼藥膏 7
眼蟲 2
眼袋 46
眼裡 163
眼見 256
眼觀四方 1
眼觀四面 4
眼觀鼻鼻觀心 0
眼角 106
眼角膜 46
眼角膜 46
眼識 1
眼部 127
眼鏡 2164
眼鏡套 0
眼鏡布 5
眼鏡架 2
眼鏡框 12
眼鏡片 17
眼鏡盒 9
眼鏡蛇 52
眼門 1
眼霜 28
眼高手低 153
眽 87
眾 5478
眾人 1309
眾人皆知 13
眾人皆醉 0
眾人皆醉我獨醒 1
眾叛親離 3
眾口一詞 0
眾口一詞 0
眾口鑠金 6
眾多 1279
眾多人口 3
眾寡 1
眾寡懸殊 1
眾志成城 74
眾怒難息 0
眾怒難犯 0
眾所周知 65
眾所矚目 65
眾擎易舉 0
眾數 18
眾星拱月 5
眾望 13
眾望所歸 30
眾生 1024
眾生相 26
眾目昭彰 0
眾目睽睽 12
眾目睽睽之下 12
眾矢之的 13
眾說不一 1
眾說紛紜 54
眾謀僉用 0
眾議 7
眾議員 136
眾議院 255
眾軍 13
眾院 484
眾體 0
眿 0
眿 0
着 0
着 0
着 0
着 0
着 0
睁 0
睂 0
睃 0
睄 105
睅 158
睆 178
睇 260
睈 0
睉 0
睊 97
睋 154
睌 82
睍 106
睎 78
睏 109
睏乏 0
睒 7
睓 0
睔 14
睕 12
睖 6
睗 0
睘 0
睘 0
睙 9
睚 1
睚 9
睚眥 5
睛 192
睜 267
睜一眼 5
睜一眼 5
睜一隻眼 15
睜一隻眼 15
睜目 4
睜眼 51
睜眼說瞎話 18
睜起 1
睜開 62
睜開眼 33
睜開眼睛 40
睝 0
睞 30
睟 8
睠 11
睡 5759
睡一覺 100
睡一覺 100
睡上 10
睡上來 0
睡上去 0
睡下 6
睡下來 0
睡下去 0
睡不到 5
睡不著 156
睡了 186
睡了又睡 1
睡來 0
睡來睡去 0
睡個 34
睡個夠 1
睡個痛快 0
睡到 108
睡前 141
睡午覺 15
睡去 33
睡在 110
睡夢 14
睡夢中 58
睡帽 0
睡得 94
睡得正甜 0
睡得正香 0
睡意 34
睡意全消 1
睡意正濃 1
睡懶覺 3
睡椅 1
睡法 1
睡滿 3
睡熟 8
睡獅 4
睡的 93
睡相 6
睡眠 579
睡眠不足 37
睡眠曲 0
睡眠療法 1
睡眼 28
睡眼矇矓 1
睡睡 22
睡睡覺 2
睡美人 56
睡著 136
睡著了 139
睡蓮 28
睡蓮葉 0
睡衣 254
睡袋 84
睡袍 11
睡褲 1
睡覺 981
睡覺時 69
睡起 1
睡起來 5
睡足 7
睡車 0
睡過 46
睡過來 0
睡過去 1
睡鄉 1
睡醒 57
睡飽 0
睡鼠 0
睢 12
督 763
督令 0
督促 411
督促檢查 0
督學 190
督察 82
督導 1055
督工 7
督師 3
督府 9
督戰 2
督撫 5
督率 3
督理 0
督署 0
督課 2
督責 10
督軍 12
督辦 4
督造 0
督飭 5
睤 0
睥 4
睥睨 25
睦 246
睦親 0
睦鄰 29
睦鄰政策 9
睧 6
睨 13
睨視 2
睩 9
睪 102
睪 102
睪丸 23
睪丸 23
睫 73
睫毛 101
睫毛膏 22
睬 23
睭 6
睮 3
睯 36
睰 0
睱 0
睱 0
睲 0
睳 0
睴 0
睵 0
睶 0
睷 0
睸 0
睹 176
睹物思人 1
睹物思情 1
睺 0
睺 0
睻 0
睻 0
睼 16
睽 68
睽合 0
睽攜 0
睽疑 0
睽違 37
睽離 0
睾 6
睿 1390
睿哲 110
睿明 1
睿智 114
睿知 1
睿達 1
瞀 12
瞁 17
瞂 67
瞃 14
瞄 208
瞄一瞄 0
瞄中 0
瞄了 15
瞄來瞄去 1
瞄出 0
瞄到 10
瞄好 0
瞄準 301
瞄準到 0
瞄瞄 5
瞄著 2
瞅 25
瞅著 11
瞅見 2
瞇 54
瞇上 1
瞇了 3
瞇住 0
瞇來瞇去 0
瞇眼 5
瞇瞇眼 4
瞇縫 0
瞇著 4
瞇著眼 14
瞇著眼睛 5
瞇起 9
瞇起來 0
瞈 65
瞉 104
瞉 104
瞊 0
瞋 83
瞋怒 1
瞋恚 1
瞋目 2
瞋目切齒 1
瞋膽 1
瞋視 0
瞌 15
瞌睡 23
瞌睡蟲 8
瞍 85
瞎 505
瞎了 24
瞎了眼 8
瞎了眼睛 2
瞎到 0
瞎吹 0
瞎子 86
瞎子摸象 1
瞎得 8
瞎忙 1
瞎成 0
瞎扯 15
瞎抓 0
瞎指揮 1
瞎掉 5
瞎掰 49
瞎搞 50
瞎摸 2
瞎猜 10
瞎的 9
瞎眼 31
瞎編 2
瞎聊 1
瞎話 1
瞎說 5
瞎說八道 0
瞎謅 1
瞎貓 100
瞎貓捉到死老鼠 1
瞎貓碰到死耗子 1
瞎闖 1
瞎鬧 0
瞏 63
瞐 0
瞑 38
瞑目 10
瞔 0
瞕 2
瞖 0
瞗 746
瞙 9
瞚 31
瞛 22
瞜 7
瞝 7
瞞 1869
瞞上欺下 0
瞞下欺上 0
瞞不住 0
瞞不過 18
瞞了 1
瞞人耳目 1
瞞住 1
瞞來瞞去 0
瞞到 0
瞞天過海 21
瞞得 3
瞞得過 7
瞞心昧己 1
瞞著 23
瞞過 17
瞞騙 9
瞟 30
瞟一眼 1
瞟一眼 1
瞟見 0
瞠 224
瞠乎其後 4
瞠乎後矣 0
瞠目 4
瞠目結舌 21
瞡 19
瞢 14
瞢昧 0
瞣 3
瞤 0
瞥 43
瞥一眼 3
瞥一眼 3
瞥然 0
瞥見 53
瞥視 1
瞦 0
瞧 442
瞧一瞧 28
瞧不上眼 0
瞧不上眼 0
瞧不起 32
瞧了 47
瞧來瞧去 0
瞧出 18
瞧得起 4
瞧熱鬧 1
瞧瞧 492
瞧縣 0
瞧著 89
瞧著辦 1
瞧見 59
瞧過 8
瞨 332
瞩 0
瞪 1641
瞪了 20
瞪得 10
瞪的 9
瞪目 6
瞪眼 15
瞪瞪 1
瞪著 120
瞪視 18
瞪起 6
瞪起來 0
瞫 5
瞬 180
瞬息 17
瞬息之間 9
瞬息萬變 96
瞬時 19
瞬時值 0
瞬時間 2
瞬間 606
瞬間反應 2
瞬霎 1
瞭 241
瞭 241
瞭如 0
瞭如指掌 9
瞭望 158
瞭望台 72
瞭望塔 3
瞭望所 0
瞭望臺 3
瞭然 8
瞭然於心 2
瞭若指掌 20
瞭解 9267
瞭解到 219
瞮 0
瞯 0
瞰 131
瞰臨 0
瞱 2748
瞲 7
瞳 579
瞳仁 13
瞳孔 109
瞳孔放大 11
瞴 682
瞵 27
瞵盼 0
瞶 7
瞷 2
瞸 0
瞹 0
瞺 3
瞻 329
瞻仰 34
瞻仰遺容 0
瞻前顧後 8
瞻望 26
瞻禮 1
瞻視 2
瞼 150
瞽 104
瞽亂 0
瞽夫 0
瞽者 4
瞽說 4
瞾 0
瞿 579
瞿 579
瞿 579
瞿塘 4
瞿塘峽 9
矀 0
矁 0
矁 0
矂 2
矃 0
矄 4
矅 0
矆 0
矇 1
矇 108
矇住 6
矇昧無知 0
矇混 8
矇混過關 2
矇矇 7
矇矇矓矓 0
矇矓 17
矇蔽 10
矇騙 17
矉 9
矊 5
矊 5
矌 281
矌 281
矍 8
矍鑠 0
矎 7
矎 7
矏 8
矐 11
矑 0
矒 0
矓 227
矔 8
矕 107
矕 107
矖 0
矗 43
矗力 0
矗立 79
矗立在 46
矘 20
矙 53
矚 50
矚望 1
矚目 528
矚託 0
矛 146
矛盾 667
矛盾性 2
矛盾百出 2
矛盾論 0
矛盾重重 2
矛頭 54
矜 63
矜 63
矜功 1
矜大 0
矜寡 0
矜恤 1
矜惜 0
矜憐 1
矜持 34
矜肆 0
矜誇 4
矜重 0
矜飾 1
矝 0
矝 0
矞 97
矟 0
矠 8
矡 0
矢 582
矢口否認 18
矢如雨下 0
矢志 0
矢志不移 3
矢石 2
矢言 13
矢量 4
矣 886
矤 0
知 97
知 9797
知不知道 172
知之為知之 1
知了 54
知事 12
知交 12
知人 30
知人之明 3
知人善任 10
知人知面 4
知人知面不知心 1
知其一不知其二 2
知其不可為而為之 5
知友 1
知名 2434
知名之士 0
知名人士 44
知命 35
知命之年 1
知子莫若母 0
知子莫若父 1
知客 12
知己 167
知己知彼 65
知府 47
知心 244
知心人 15
知心朋友 182
知心話 5
知性 694
知恥 16
知恥知病 0
知恥近乎勇 3
知恩 10
知恩圖報 2
知悉 198
知情 115
知情不報 2
知方 5
知易 4
知易行難 4
知曉 185
知更鳥 10
知書達禮 7
知會 144
知本 465
知本溫泉 67
知法 30
知法犯法 8
知無不言 11
知無不談 0
知禮 10
知縣 70
知能 644
知行 70
知行合一 12
知覺 340
知覺力 0
知覺性 2
知言 10
知識 5797
知識份子 216
知識份子 216
知識分子 140
知識分子 140
知識化 5
知識性 77
知識論 67
知識階級 4
知識青年 63
知趣 7
知足 88
知足常樂 24
知遇 8
知遇之恩 4
知過 18
知過必改 0
知過能改 3
知道 18755
知錯能改 0
知難而退 21
知難行易 12
知青 39
知音 150
知音難尋 5
知音難求 4
矦 0
矧 28
矨 19
矩 352
矩尺 8
矩形 104
矩矱 0
矩陣 565
矪 0
矬 174
矬子 0
短 3954
短不了 0
短了 12
短促 58
短傳 4
短兵相接 25
短刀 26
短劇 38
短句 12
短命 21
短大衣 1
短小 128
短小精幹 4
短小精悍 10
短少 49
短尾 7
短局 0
短工 13
短得 248
短手 3
短手短腳 0
短打 4394
短撅撅 0
短文 160
短時期 13
短時間 398
短暫 567
短暫性 16
短期 4014
短期投資 68
短期預付費用 0
短欠 0
短歌 19
短氣 1
短波 866
短波 866
短泳 0
短泳選手 0
短淺 8
短的 249
短短 503
短短的 304
短短地 304
短矮 0
短篇 322
短篇小說 370
短簡 3
短粗 2
短絀 73
短統 5
短統靴 0
短統鞋 0
短線 2611
短線交易 11
短纖維 23
短缺 227
短耳 10
短腳 15
短腿 29
短處 12
短袖 1
短裙 63
短裝 6
短褲 107
短褲裝 0
短襪 15
短見 2
短視 47
短訓班 0
短評 185
短詩 23
短跑 45
短跑健將 0
短跑道 0
短跑選手 2
短距 1
短距離 30
短路 154
短途 3
短途旅行 0
短針 3
短鋸 0
短長 15
短音 2
短音符 0
短韻 0
短髮 852
短鯛 1
矮 439
矮了 15
矮了一截 1
矮了一截 1
矮人 118
矮人一截 2
矮人一截 2
矮人祭 29
矮個兒 0
矮個兒 0
矮冬瓜 6
矮凳 3
矮化 0
矮子 21
矮子樂 4
矮小 44
矮屋 2
矮得 19
矮林 4
矮樹 10
矮樹叢 5
矮牆 16
矮的 20
矮短 0
矮矮 17
矮矮肥肥 0
矮矮胖胖 3
矮稈作物 0
矮籬 0
矮胖 26
矮腳 0
矮靈祭 1
矯 162
矯健 23
矯形 11
矯形外科 1
矯形醫生 0
矯性 0
矯情 18
矯捷 7
矯揉造作 7
矯枉 1
矯枉過正 19
矯正 724
矯正法 1
矯治 148
矯直 1
矯直機 1
矯飾 28
矰 634
矱 5
矱 5
矲 5
矲矮 1
矲雉 1
石 101
石 10186
石人 21
石佛 7
石像 51
石刀 6
石刻 57
石刻 57
石刻像 2
石刻像 2
石勒 4
石化 1640
石化作用 0
石化工業 159
石匠 6
石印 10
石台 5
石器 70
石器時代 42
石坑 18
石城 80
石埭 0
石塊 136
石墨 145
石墨棒 1
石墨粉 3
石墩 11
石壁 96
石壇 1
石子 71
石子兒 3
石子兒 3
石子路 3
石室 30
石室金匱 1
石家庄 2
石家庄市 0
石家莊 15
石家莊市 6
石屋 32
石山 51
石崇 23
石工 9
石工術 0
石弓 0
石彈 2
石徑 1
石拱 2
石文 284
石斑 1
石斑魚 42
石料 35
石斧 9
石材 764
石松 24
石松粉 0
石板 155
石板道 2
石柱 74
石棉 94
石棉布 2
石棉板 10
石棉網 0
石棺 12
石榴 150
石榴花 2
石榴裙 4
石標 1
石橋 40
石櫃 0
石決明 1
石沉大海 10
石河子 0
石油 989
石油公司 69
石油化學 81
石油業 15
石油氣 196
石油界 2
石油荒 0
石油輸出 1
石油輸出國家 8
石洞 32
石灰 182
石灰乳 4
石灰岩 89
石灰巖 0
石灰水 11
石灰石 59
石灰礦 0
石灰窯 1
石灰質 19
石炭 2
石炭紀 6
石牆 33
石片 17
石版 62
石版術 0
石牌 186
石牌國中 21
石牌國小 22
石牌路 119
石狀 5
石獅 42
石獅子 9
石百合 0
石矽 1
石破天驚 19
石碇 107
石碇鄉 59
石碑 81
石碳層 0
石碳酸 2
石磚 15
石磨 13
石礦 60
石礫兒 0
石礫兒 0
石稜稜 0
石穴 4
石窟 37
石筍 12
石籠 4
石級 4
石經 3
石綿 94
石綿板 1
石綿瓦 19
石縫 13
石脈 0
石脈 0
石膏 154
石膏像 3
石臼 1
石花菜 5
石英 278
石英岩 5
石英錶 2
石蕊 4
石蕊試紙 18
石虎 13
石蠟 38
石蠟像 0
石蠶 1
石輪 1
石達開 0
石鍋拌飯 1
石鍾乳 0
石鐘乳 0
石門 295
石門水庫 211
石門鄉 126
石階 117
石雕 189
石雕像 3
石雷 0
石青 49
石頭 785
石頭 785
石頭火鍋 8
石頭火鍋 8
石頭記 21
石頭記 21
石首魚 5
石鹼 0
石鹽 0
石鼓 29
石鼓文 4
矴 0
矵 0
矵 0
矷 180
矸 43
矸石 0
矹 4
矺 6
矻 17
矼 5
矽 2599
矽岩 0
矽晶片 19
矽晶體 2
矽石 13
矽砂 37
矽膠 1
矽藻 39
矽谷 383
矽質 23
矽酸 103
矽酸鈉 13
矽鋼 15
矽電晶體 1
砂 1781
砂囊 1
砂土 25
砂坑 0
砂型 1
砂塊兒 0
砂塊兒 0
砂子 29
砂層 13
砂岩 106
砂巖 0
砂布 24
砂漿 105
砂濾器 0
砂盤 1
砂眼 14
砂石 689
砂石車 100
砂磨 6
砂礁 0
砂礦 3
砂礫 32
砂粒 13
砂糖 147
砂紙 46
砂質 34
砂輪 134
砂輪機 21
砂金 4
砂金石 0
砂鍋 59
砂鍋菜 1
砂鍋賽 0
砂鐵 3
砃 267
砄 0
砅 29
砆 29
砇 0
砇 0
砈 0
砉 240
砉然 0
砊 0
砊 0
砋 0
砌 236
砌成 37
砌牆 12
砌磚 18
砌路 0
砍 10891
砍上 2
砍上來 0
砍上去 0
砍下 29
砍下來 6
砍下去 0
砍了 52
砍些 0
砍伐 117
砍來砍去 0
砍倒 10
砍傷 25
砍光 9
砍出 3
砍出來 6
砍出去 1
砍刀 3
砍到 12
砍去 31
砍向 11
砍在 6
砍壞 0
砍得 1
砍手 3
砍手砍腳 0
砍掉 109
砍斷 35
砍柴 15
砍樹 45
砍死 21
砍殺 33
砍砍 15
砍著 2
砍走 0
砍起 1
砍起來 1
砍進 6
砍進來 0
砍進去 0
砍過 3
砍過來 1
砍過去 1
砍錯 0
砍除 23
砍頭 22
砎 30
砏 16
砐 85
砑 635
砒 113
砒酸 0
砒霜 11
砓 164
研 11
研 2463
研修 294
研修班 27
研判 404
研判出 2
研制 14
研成 8
研擬 1225
研杵 0
研析 123
研查 1
研求 7
研發 6327
研發部 72
研發部門 46
研碎 0
研磨 508
研磨劑 6
研究 60784
研究出 40
研究出來 6
研究員 2060
研究室 2928
研究所 18479
研究會 596
研究生 2549
研究者 497
研究費 35
研究起來 1
研究院 2309
研習 4667
研習會 1058
研考會 641
研製 360
研討 2631
研討會 8592
研讀 405
砕 0
砕 0
砝 9
砝碼 21
砟 27
砟 27
砟子 1
砠 229
砡 24
砢 20
砢 20
砣 17
砤 0
砥 3630
砥柱 3
砥石 4
砥礪 20
砥節 0
砥節礪行 0
砥節礪行 0
砦 190
砧 36
砧木 7
砧杵 0
砧板 30
砧砭 0
砨 4
砩 532
砪 37
砫 139
砬 47
砭 116
砭灸 0
砭灸術 0
砮 10
砯 23
砰 86
砰噹 0
砰然 12
砰砰 17
砱 322
砲 1237
砲仗 0
砲位 1
砲兵 41
砲兵營 1
砲兵連 0
砲兵部隊 2
砲兵陣地 0
砲口 17
砲台 173
砲塔 29
砲座 5
砲彈 39
砲手 18
砲擊 17
砲火 63
砲灰 5
砲管 10
砲聲 20
砲聲隆隆 12
砲臺 54
砲艇 46
砲衣 0
砲車 11
砲轟 92
砳 11
破 2867
破亡 0
破例 43
破傷風 124
破冰船 4
破功 0
破口 13
破口大罵 34
破啼 0
破啼為笑 0
破土 102
破土典禮 9
破報 10
破壞 2937
破壞力 49
破壞性 135
破天荒 61
破家 5
破屋 25
破布 37
破廟 11
破得 6
破得 63
破恥廉 502
破成 5
破戒 22
破折號 3
破損 185
破敗 13
破曉 52
破曉時分 8
破格 9
破格錄用 0
破案 140
破洞 87
破浪 20
破浪前進 0
破涕為笑 6
破滅 82
破爛 52
破爛兒 0
破爛兒 0
破獲 277
破產 307
破百 84
破的 63
破皮 41
破相 9
破碎 245
破竹 3
破竹之勢 1
破紀錄 44
破約 0
破綻 72
破綻百出 3
破膽 1
破舊 76
破船 5
破落 8
破裂 449
破解 1095
破財 14
破財消災 0
破費 7
破釜沈舟 5
破釜沉舟 3
破鈔 0
破銅爛鐵 5
破鏡重圓 5
破門 12
破門而入 5
破關 1
破陣 10
破除 123
破除迷信 7
破隙 1
破鞋 4
破音字 18
破題 1
破題第一遭 0
破體字 0
砵 9
砶 0
砷 403
砷酸鹽 16
砸 213
砸下 26
砸了 17
砸傷 3
砸壞 2
砸掉 2
砸死 15
砸毀 12
砸爛 2
砸破 3
砸破飯碗 0
砸碎 5
砸過 0
砸過來 0
砸過去 0
砸鍋 2
砸飯碗 0
硂 0
硃 114
硃唇皓齒 0
硃批 0
硃砂 13
硃砂痣 1
硃砂紅 0
硃筆 2
硃紅色 16
硃色 1
硄 0
硄 0
硅 199
硅橡膠 0
硅油 2
硅石 2
硅磚 0
硅藻 0
硅酸 4
硅酸鈉 2
硅酸鹽 1
硅鋼片 0
硆 0
硇 0
硈 174
硉 527
硊 19
硊 19
硋 0
硌 375
硍 154
硎 337
硏 0
硏 0
硐 732
硑 0
硑 0
硒 258
硜 161
硝 164
硝化 93
硝化甘油 24
硝化細菌 10
硝基 79
硝基苯 14
硝煙 17
硝石 2
硝石灰 0
硝酸 157
硝酸鈉 49
硝酸鈣 3
硝酸鉀 12
硝酸銀 17
硝酸銨 15
硝酸鹽 133
硞 73
硟 0
硠 51
硠 51
硡 0
硢 61
硣 0
硤 139
硥 141
硦 0
硧 0
硧 0
硨 116
硩 184
硪 169
硫 7071
硫化 108
硫化劑 1
硫化染料 51
硫化氫 46
硫化物 39
硫化鈉 26
硫化鉛 2
硫化銻 2
硫化鋅 10
硫化鹼 18
硫化黑 0
硫磺 105
硫礦 4
硫粉 2
硫胺素 0
硫華 0
硫酸 482
硫酸亞鐵 40
硫酸液 1
硫酸液 1
硫酸鈉 146
硫酸鈣 15
硫酸鉀 9
硫酸銅 54
硫酸銨 74
硫酸鋁 32
硫酸鋅 16
硫酸鋇 14
硫酸鎂 42
硫酸鹽 111
硫鐵礦 2
硫黃 23
硬 5376
硬了 14
硬了起來 1
硬仗 11
硬件 23
硬伸 0
硬來 0
硬化 211
硬化劑 42
硬化症 58
硬卡 2
硬口蓋 0
硬塊 68
硬實 6
硬幣 212
硬幹 7
硬度 347
硬度計 20
硬式 99
硬式磁碟機 59
硬性 38
硬性規定 23
硬戰 5
硬打 1
硬拼 7
硬接 6
硬撐 43
硬撐下來 0
硬撐下去 2
硬是 136
硬朗 20
硬朗結實 0
硬朗結實 0
硬木 12
硬梆梆 19
硬橡皮 0
硬殼 19
硬水 42
硬漢 20
硬煤 0
硬生生 42
硬皮 0
硬盤 20
硬石膏 5
硬硬 19
硬碟 4086
硬碟內 21
硬碰硬 28
硬紙 11
硬紙板 4
硬給 4
硬脂 1
硬著 10
硬著頭皮 38
硬要 94
硬說 9
硬說是 1
硬質 126
硬質合金 0
硬起 4
硬起來 3
硬逼 3
硬邦邦 3
硬鋁 1
硬面 17
硬骨頭 0
硬骨魚類 6
硬體 8519
硭 175
确 327
硯 217
硯友 0
硯台 7
硯席 0
硯池 0
硯田 2
硯臺 1
硰 112
硰 112
硱 161
硶 0
硸 0
硹 2
硺 0
硻 4
硼 74
硼 74
硼砂 41
硼酸 69
硼酸鹽 0
硽 0
硽 0
硾 25
硿 6
碀 9
碁 223
碁 2236
碁峰 100
碁石 100
碂 0
碂 0
碃 12
碄 11
碅 1
碆 6
碇 274
碇泊 3
碇泊費 5
碈 0
碈 0
碉 18
碉堡 27
碊 0
碊 0
碊 0
碋 0
碌 80
碌碌 10
碌碌庸才 0
碌碌無能 0
碌碡 0
碍 0
碎 632
碎冰機 0
碎務 0
碎嘴子 0
碎塊 11
碎小 2
碎屍萬段 3
碎屑 51
碎工 0
碎布 5
碎布條 0
碎成 8
碎步 3
碎波 12
碎波 12
碎煤 0
碎爛 0
碎片 136
碎玉 2
碎玻璃 8
碎石 140
碎石子 5
碎石機 23
碎石路 3
碎磚 2
碎紙 45
碎肉 13
碎裂 73
碎裂聲 0
碎金 2
碎銀 4
碏 9
碐 0
碑 483
碑亭 3
碑學 0
碑帖 20
碑志 2
碑拓 1
碑文 42
碑林 18
碑版 0
碑石 11
碑碣 13
碑記 14
碑誌 2
碑銘 4
碑陰 0
碑額 0
碒 0
碓 1
碓房 0
碔 0
碕 0
碖 0
碗 512
碗兒 0
碗兒 0
碗公 26
碗櫃 0
碗櫥 0
碗盤 99
碗碟 4
碗筷 26
碗籃 0
碗粿 0
碗糕 33
碗豆 11
碗鍋 0
碗飯 16
碘 316
碘劑 4
碘化鉀 19
碘化銀 0
碘質 8
碘酊 1
碘酒 6
碙 6
碚 11
碚 11
碝 0
碞 9
碟 2400
碟仙 16
碟型 12
碟子 19
碟形 13
碟片 1233
碟版 16
碠 4
碡 9
碢 5
碣 88
碣石 2
碤 50
碥 12
碦 0
碧 2187
碧冷冷 0
碧姬芭杜 4
碧山 169
碧水 11
碧江 2
碧波 11
碧波 11
碧波盪漾 1
碧波盪漾 1
碧海 131
碧海晴空 0
碧海青天 14
碧湖 67
碧湖國中 0
碧潭 89
碧澄澄 1
碧玉 129
碧瑤 20
碧眼 10
碧空 6
碧綠 79
碧綠色 13
碧色 2
碧草 2
碧草如茵 7
碧華 158
碧華國中 7
碧華國小 6
碧藍 17
碧螺春 3
碧血 25
碧血丹心 0
碧血黃花 0
碨 13
碩 362
碩 3624
碩儒 0
碩博士 4819
碩士 4819
碩士學位 357
碩士班 2707
碩大 33
碩大無朋 4
碩大無朋 4
碩大無比 4
碩學鴻儒 2
碩彥 23
碩德 0
碩望 0
碩果 14
碩果 14
碩果僅存 37
碩老 0
碩言 1
碪 10
碫 31
碬 2
碭 77
碮 0
碯 0
碰 666
碰一鼻子灰 0
碰一鼻子灰 0
碰上 243
碰倒 2
碰傷 8
碰到 949
碰壁 26
碰壞 1
碰巧 50
碰巧在 4
碰巧是 2
碰得 22
碰撞 316
碰擊 10
碰杯 1
碰的 22
碰碰 64
碰碰車 9
碰翻 3
碰著 6
碰見 42
碰觸 156
碰運氣 17
碰釘 0
碰釘子 1
碰面 73
碰頭 25
碱 0
碲 38
碳 1600
碳化 118
碳化物 476
碳化硅 0
碳化鐵 0
碳棒 4
碳氫 15
碳氫化合 2
碳氫化合物 136
碳水 2
碳水化合 1
碳水化合物 98
碳源 9
碳烤 52
碳筆 2
碳粉 132
碳粉匣 0
碳素 36
碳素鋼 3
碳纖維 1
碳酸 149
碳酸氫鈉 17
碳酸氫銨 1
碳酸水 2
碳酸鈉 42
碳酸鈣 57
碳酸鉀 24
碳酸鎂 4
碳酸鹽 45
碴 68
碸 0
確 2056
確保 2723
確保安全 41
確信 152
確信無疑 0
確切 226
確切不變 0
確切性 0
確定 9276
確定是 105
確定會 8
確定有 39
確定能 4
確實 2652
確實可靠 4
確實在 27
確實性 5
確實是 222
確實會 20
確實有 151
確實能 27
確是 311
確有 519
確有其事 7
確有其人 2
確有此事 5
確當 1
確知 101
確立 607
確認 3596
確認是 11
確論 0
確證 6
確鑿 18
確非 6
碻 41
碼 9468
碼子 3
碼表 23
碼頭 1108
碼頭區 16
碼頭工 5
碼頭工人 31
碼頭費 0
碽 0
碾 35
碾下去 0
碾來碾去 0
碾到 1
碾去 0
碾坊 0
碾壓 4
碾平 0
碾得 0
碾成 2
碾斷 0
碾碎 5
碾磨 3
碾米 210
碾米廠 114
碾米機 7
碾軋 2
碾過 4
碿 0
碿 0
磀 0
磁 3000
磁傾角 1
磁力 99
磁力線 13
磁力計 5
磁化 43
磁區 118
磁卡 65
磁器 66
磁場 36
磁場 362
磁場強度 12
磁場強度 12
磁存貯器 0
磁帶 332
磁帶設備 0
磁心 1
磁性 233
磁性天線 0
磁性材料 31
磁性瓷 0
磁感應 8
磁感應強度 0
磁放大器 0
磁效應 5
磁條 43
磁棒 8
磁極 25
磁流體 0
磁片 1211
磁片套 0
磁片盒 1
磁畫 0
磁盤 6
磁矩 4
磁石 49
磁碟 758
磁碟槽 1
磁碟機 554
磁碟檔 0
磁碟片 251
磁碟組 0
磁磚 326
磁能 25
磁製 1
磁軌 34
磁通量 10
磁道 0
磁選 3
磁針 2
磁鐵 145
磁鐵礦 6
磁頭 118
磁體 9
磂 0
磃 92
磄 63
磅 491
磅 491
磅礡 30
磅秤 22
磆 0
磆 0
磇 0
磈 187
磉 145
磊 685
磊塊 0
磊落 17
磊落大方 0
磋 41
磋商 175
磋磨 0
磋談 0
磋跎 2
磌 66
磍 50
磎 143
磏 56
磐 343
磐石 112
磑 70
磒 0
磒 0
磒 0
磓 0
磓 0
磔 121
磕 26
磕個響頭 0
磕個頭 0
磕牙 1
磕碰 1
磕頭 25
磕頭蟲 0
磖 0
磚 750
磚匠 0
磚塊 68
磚廠 18
磚牆 56
磚片 0
磚瓦 205
磚窖 0
磚窯 20
磚茶 7
磚造 36
磚面 3
磚頭 20
磛 78
磜 0
磜 0
磝 1784
磞 117
磟 86
磠 52
磠砂 0
磡 95
磢 145
磣 68
磤 0
磤 0
磥 554
磦 0
磧 27
磨 741
磨 741
磨了 2
磨亮 6
磨人 5
磨來磨去 0
磨光 77
磨光機 5
磨出 15
磨刀 24
磨刀石 5
磨刀霍霍 7
磨到 7
磨削 9
磨勘 0
磨去 7
磨合 24
磨坊 30
磨坊 30
磨墨 10
磨子 4
磨工 1
磨平 21
磨床 174
磨得 6
磨成 45
磨房 1
磨拳擦掌 8
磨掉 14
磨損 186
磨損量 0
磨搓 0
磨擦 135
磨料 8
磨斷 0
磨時間 0
磨杵 0
磨杵成針 0
磨滅 37
磨煩 0
磨牙 39
磨牙齒 2
磨盤 1
磨盤 2
磨短 0
磨石 36
磨石子 1
磨石子地 5
磨石子面 0
磨砂燈泡 0
磨破 10
磨碎 22
磨磋 0
磨磨 7
磨礦 1
磨穿 2
磨穿鐵硯 0
磨粉 19
磨粉廠 0
磨細 6
磨練 142
磨耗 76
磨肩接踵 0
磨肩擦踵 0
磨菇 10
磨蝕 7
磨製 6
磨起 1
磨起來 0
磨蹭 8
磨過 0
磨錯 0
磨鍊 23
磨難 34
磩 176
磪 470
磫 0
磬 80
磬折 1
磬聲 3
磭 317
磯 428
磯鳥 0
磯鷂 0
磰 0
磱 0
磲 838
磳 219
磴 10
磴道 1
磶 0
磷 463
磷光 4
磷光體 1
磷火 5
磷灰石 10
磷灰石礦 0
磷灰石糖 0
磷磷 0
磷礦粉 0
磷肥 6
磷脂 36
磷脂酸 0
磷酸 353
磷酸鈉 40
磷酸鈣 11
磷酸鹽 94
磸 0
磹 55
磺 118
磺化 12
磺溪 44
磺田 1
磺窟 1
磺胺 84
磺酸 85
磻 63
磼 30
磽 140
磽薄 0
磾 125
磿 7
礀 0
礁 17000
礁岩 75
礁棚 1
礁溪 544
礁溪溫泉 29
礁石 48
礂 55
礄 69
礅 260
礆 0
礇 0
礈 0
礉 4
礊 0
礊 0
礋 0
礌 3
礌 3
礍 0
礍 0
礎 417
礏 0
礐 7
礑 10
礒 4
礓 3
礓擦 0
礔 6
礕 0
礗 124
礘 0
礙 393
礙事 9
礙口 0
礙子 0
礙子 0
礙手 0
礙手礙腳 15
礙於 197
礙於情面 6
礙滯 0
礙目 0
礙眼 20
礙難 5
礙難照准 0
礚 0
礛 4
礛 4
礜 340
礜 340
礜 340
礝 5
礞 128
礟 0
礟 0
礟 0
礠 0
礡 273
礣 7
礤 2
礤床兒 0
礤床兒 0
礥 7
礦 588
礦 588
礦井 1
礦內 1
礦務 83
礦區 110
礦坑 71
礦城 0
礦場 11
礦場 119
礦局 1
礦層 0
礦山 31
礦山壓力 0
礦工 81
礦床 19
礦房 0
礦柱 3
礦業 487
礦業權 16
礦業法 19
礦業界 0
礦油 171
礦泉 27
礦泉水 169
礦泥 7
礦渣 49
礦漿 0
礦燈 0
礦物 482
礦物學 12
礦物局 6
礦物油 29
礦物界 1
礦物質 295
礦物質飼料 0
礦產 169
礦產區 0
礦產地 1
礦石 122
礦砂 24
礦脈 21
礦脈 21
礦苗 0
礦藏 13
礦藏量 1
礦質 2
礦車 3
礦長 0
礦體 0
礦體厚度 0
礧 6
礨 6
礩 6
礪 33
礪石 0
礫 53
礫土 1
礫岩 22
礫石 72
礫石路 0
礬 88
礬土 8
礬石 0
礭 7
礭 7
礮 0
礮 0
礮 0
礯 4
礰 0
礱 28
礲 0
礵 54
礷 0
礸 277
礹 5
示 6319
示人 659
示例 69
示好 29
示威 346
示威抗議 41
示威者 20
示威運動 4
示弱 19
示意 111
示意圖 412
示愛 63
示波器 175
示波器 175
示波圖 0
示波圖 0
示波管 0
示波管 0
示波鏡 0
示波鏡 0
示眾 6
示範 1794
示範作用 21
示範動作 11
示範性 24
示覆 1
示警 36
示蹤劑 4
礼 0
礽 25
社 19582
社中 81
社交 301
社交圈 6
社交活動 33
社交生活 14
社交界 7
社刊 99
社刊主編 0
社務 68
社區 13125
社友 46
社名 63
社員 929
社團 14937
社團活動 1570
社團負責人 64
社址 38
社子 101
社子島 46
社寮 38
社工 674
社教 631
社教司 56
社教系 89
社教組 1
社教館 295
社日 21
社會 28713
社會主義 268
社會主義公有制 2
社會主義建設 2
社會主義所有制 0
社會主義改造 2
社會主義積累 0
社會主義競賽 0
社會主義經濟 4
社會主義者 6
社會主義革命 3
社會保險 256
社會公德 1
社會分工 6
社會制度 59
社會化 120
社會問題 224
社會地位 79
社會基礎 14
社會存在 2
社會學 1002
社會學家 32
社會學科 82
社會學系 240
社會實踐 12
社會局 393
社會工作 825
社會工作者 24
社會帝國 0
社會帝國主義 0
社會形態 11
社會心理 80
社會心理學 75
社會必要 0
社會必要勞動 1
社會性 80
社會意識 25
社會意識形態 1
社會改革 38
社會改革者 1
社會教育 804
社會新聞 5053
社會民主 37
社會民主主義 4
社會民主黨人 0
社會沙文 0
社會沙文主義 0
社會活動 48
社會現象 81
社會生活 182
社會生產 7
社會生產價格 0
社會福利 1534
社會科 737
社會科學 1234
社會科學院 86
社會經濟 233
社會經濟形態 0
社會經濟結構 11
社會處 283
社會處長 2
社會觀 13
社會變革 4
社會資本 2
社會部 8
社會關係 68
社會風氣 106
社會黨 27
社稷 19
社經 101
社群 0
社花 3
社裡 12
社論 747
社論主筆 0
社運 129
社運人士 6
社長 822
社隊 6
社頭 331
社頭 331
社黨 3
社鼠 0
礿 161
祀 452
祀事 0
祀奉 7
祀物 0
祀祖 1
祀神 14
祁 211
祁寒 0
祁山 13
祁連 6
祁連山 7
祁門 0
祂 3308
祄 5
祅 60
祆 18
祆教 0
祆教徒 0
祇 573
祇 573
祇 573
祇好 33
祇是 324
祇有 248
祇管 3
祇要 181
祈 599
祈使句 5
祈望 32
祈求 292
祈福 340
祈禱 468
祈禱文 9
祈請 13
祈連山 1
祈雨 16
祉 76
祊 201
祋 11
祌 3
祏 8
祐 1388
祐德 12
祑 71
祒 81
祓 13
祔 121
祕 1197
祕 1197
祕史 1
祕宗 0
祕密 577
祕方 58
祕書 286
祕書處 54
祕書長 181
祕笈 107
祕籍 1
祕而不宣 3
祕藏 1
祕要 0
祕訣 182
祕魯 107
祖 2068
祖上 7
祖上有德 1
祖傳 28
祖傳祕方 0
祖傳秘方 5
祖先 491
祖國 212
祖墳 24
祖媽 1
祖孫 34
祖孫三代 8
祖孫二人 1
祖宗 71
祖家 1
祖居 9
祖師 245
祖師廟 114
祖師爺 33
祖廟 25
祖德 8
祖業 3
祖母 166
祖沖之 3
祖父 157
祖父母 52
祖產 15
祖祖輩輩 0
祖籍 40
祖翁 0
祖考 0
祖逖 5
祖餞 3
祗 357
祘 0
祙 0
祚 183
祛 29
祛災 0
祛痰 6
祜 124
祝 8169
祝你 417
祝你們 65
祝典 0
祝告 1
祝壽 44
祝妳 134
祝妳們 9
祝巫 0
祝巫 0
祝文 6
祝福 1890
祝禱 32
祝英台 30
祝融 21
祝融為虐 0
祝詞 63
祝賀 208
祝賀信 0
祝辭 5
祝酒 3
祝頌 1
祝願 7
神 16515
神不守舍 1
神不知鬼不覺 32
神丹 3
神主 13
神主牌 0
神乎其技 12
神乎其神 3
神交 15
神人共憤 0
神仙 485
神仙般 3
神仙道化 1
神似 48
神位 20
神來之筆 14
神偷 21
神像 163
神兵 78
神出鬼沒 31
神劇 23
神劍 305
神力 58
神功 121
神助 17
神勇 26
神化 26
神台 0
神名 3
神君 28
神品 1
神器 45
神壇 32
神奇 712
神女 12
神妙 15
神姿 0
神威 52
神學 904
神學家 34
神學生 28
神學院 610
神宗 20
神射手 14
神將 38
神岡 660
神州 126
神州大陸 5
神工鬼斧 0
神差鬼使 0
神廟 113
神往 28
神志 15
神思 20
神思不定 0
神性 71
神怪 20
神怪片 3
神恩 28
神情 343
神情愉快 6
神意 10
神愛世人 15
神態 79
神態悠閒 1
神態自若 5
神戶 213
神技 10
神援 0
神效 11
神教 31
神明 244
神智 31
神智不清 19
神智清醒 3
神曲 8
神木 395
神木區 17
神格 13
神格化 4
神棍 22
神槍手 2
神機妙算 69
神權 9
神權時代 0
神武 20
神殿 249
神氣 54
神氣十足 2
神氣活現 26
神水 13
神法 5
神清氣爽 21
神滅論 0
神漢 1
神燈 112
神父 321
神物 4
神甫 1
神異 15
神社 70
神祇 57
神祕 525
神祕性 0
神祭 2
神秘 1414
神秘主義 14
神秘化 1
神秘性 40
神童 80
神算 20
神經 2512
神經不正常 2
神經中樞 5
神經元 76
神經原 9
神經失常 3
神經學 97
神經性 102
神經戰術 0
神經炎 32
神經病 94
神經痛 130
神經科 278
神經節 42
神經系 30
神經系統 252
神經細胞 65
神經纖維 35
神經衰弱 28
神經褶 0
神經質 46
神經過敏 6
神經錯亂 1
神經類型 0
神聖 564
神聖不可侵犯 8
神聖化 6
神聖羅馬帝國 4
神職 20
神職人員 24
神色 188
神色自若 5
神荼鬱壘 0
神術 8
神話 814
神諭 19
神貌 1
神足通 1
神跡 9
神蹟 184
神農 157
神農氏 14
神通 698
神通廣大 27
神通自在 1
神速 55
神遊 72
神道 27
神醫 42
神醫華陀 1
神采 63
神采奕奕 22
神采煥發 0
神采飄逸 0
神采飛揚 23
神靈 58
神韻 36
神領意會 0
神馬 7
神馳 9
神鬼 57
神魂 5
神魂不定 0
神魂顛倒 17
神魔 22
神鷹 315
神麴 2
神麴茶 0
神龍 210
神龕 30
祟 151
祠 3000
祠堂 40
祡 331
祢 0
祢 0
祣 134
祤 179
祥 9426
祥光 11
祥和 317
祥瑞 46
祥瑞之氣 0
祥雲 29
祥麟 23
祦 0
祧 152
票 9778
票值 3
票傳 0
票價 906
票券 1984
票匭 3
票友 1
票子 23
票戲 4
票房 307
票房價值 0
票房記錄 2
票據 493
票據交換 42
票據交換所 29
票據法 62
票數 1045
票根 20
票源 316
票箱 170
票號 6
票證 23
票貼 13
票選 1631
票選辦法 7
票面 105
票面價值 0
票額 0
票騎 0
祩 225
祪 115
祫 304
祬 0
祭 18
祭 1834
祭上 2
祭主 3
祭了 2
祭典 262
祭出 202
祭台 4
祭司 174
祭告 2
祭品 52
祭器 4
祭地 0
祭墓 1
祭壇 84
祭天 8
祭天大典 0
祭奠 6
祭奠儀式 0
祭好 0
祭孔 15
祭孔大典 11
祭完 2
祭師 9
祭弔 1
祭悼 4
祭拜 112
祭掃 2
祭文 18
祭日 13
祭服 2
祭滿 0
祭灶 0
祭物 16
祭獻 6
祭祀 456
祭祖 51
祭祖大典 3
祭神 9
祭神大典 0
祭禮 30
祭過 1
祭酒 3
祭錯 0
祮 0
祰 180
祱 0
祲 84
祳 101
祴 239
祴 239
祵 0
祶 0
祷 0
祹 4
祺 592
祻 0
祼 33
祽 3
祾 0
祾 0
祿 919
祿位 6
祿俸 0
祿命 0
祿食 1
禀 0
禀 0
禀 0
禀 0
禁 17804
禁 1105
禁不住 86
禁不起 56
禁令 126
禁住 0
禁例 2
禁內 0
禁兵 1
禁制 49
禁制令 5
禁區 86
禁受住 0
禁品 3
禁唱 1
禁地 80
禁城 2
禁完 0
禁得住 3
禁得起 33
禁忌 457
禁慾 19
禁慾主義 0
禁書 15
禁果 27
禁條 0
禁欲主義 0
禁止 3114
禁止停車 11
禁止通行 12
禁治產 40
禁治產人 13
禁演 7
禁煙 35
禁煙令 0
禁煙節 2
禁用 116
禁絕 14
禁臠 9
禁菸 0
禁藥 99
禁衛 26
禁衛軍 10
禁賭 0
禁足 10
禁足處分 0
禁軍 10
禁運 30
禁運令 1
禁遏 1
禁酒 9
禁酒令 0
禁錮 39
禁閉 5
禁閉室 0
禁食 124
禂 8
禂 8
禃 0
禄 0
禇 0
禈 1
禉 0
禊 32
禋 7
禍 545
禍不單行 20
禍亂 4
禍事 1
禍因 0
禍國 1
禍國殃民 9
禍害 41
禍從口出 2
禍從天降 0
禍心 1
禍患 11
禍患無窮 0
禍根 8
禍殃 2
禍水 22
禍福 49
禍福同門 0
禍福無門 0
禍福由己 0
禍福與共 0
禍種 0
禍端 7
禍胎 1
禍起蕭牆 0
禍首 70
禎 800
禎祥 67
福 13944
福份 37
福克蘭群島 1
福公司 33
福分 20
福利 3620
福利品 35
福利社 86
福利費 4
福利金 60
福和 255
福和國中 0
福地 78
福報 1
福壽 199
福壽山 43
福壽螺 1
福壽酒 0
福壽雙全 3
福如 1
福如東海 6
福安 161
福將 10
福山 567
福岡 136
福州 269
福州人 4
福州市 61
福州話 1
福建 774
福建人 10
福建省 626
福德 464
福德國小 15
福德正神 25
福德路 53
福態 1
福斯 355
福星 300
福星國小 14
福星高照 3
福晉 1
福林 80
福樂 68
福氣 179
福澤 34
福無雙至 3
福爾摩斯 80
福爾摩沙 0
福爾摩莎 402
福爾馬林 3
福特 777
福特公司 10
福特基金會 3
福王 22
福相 18
福祉 394
福神 20
福祿 66
福祿壽 16
福至心靈 3
福華 339
福蔭 9
福薄 5
福蘭克 0
福隆 143
福音 2344
福音書 51
福馬林 37
福體 4
禐 3
禑 0
禒 1
禓 7
禔 17
禕 38
禖 61
禗 4
禘 16
禚 111
禛 63
禜 46
禝 0
禞 0
禟 0
禠 28
禠 28
禠奪 4
禠奪 4
禡 24
禢 46
禤 4073
禥 0
禦 118
禦侮 0
禦寇 2
禦寒 59
禦敵 5
禧 1261
禧 1261
禨 52
禩 0
禪 139
禪 1398
禪位 0
禪偈 8
禪僧 4
禪功 10
禪味 1
禪和子 3
禪地 1
禪堂 9
禪學 64
禪宗 198
禪宗七祖 1
禪定 124
禪家 6
禪寂 1
禪寺 217
禪師 168
禪心 14
禪思 18
禪房 11
禪杖 10
禪椅 1
禪機 13
禪理 3
禪義 0
禪讓 9
禪門五宗 1
禪院 22
禫 16
禬 5
禭 9
禮 10674
禮俗 115
禮儀 531
禮儀之邦 1
禮冠 3
禮制 6
禮券 117
禮包 0
禮台 6
禮品 1074
禮品店 46
禮單 0
禮堂 240
禮多人不怪 2
禮多必詐 0
禮尚往來 40
禮帽 9
禮成 5
禮所當然 0
禮拜 838
禮拜一 27
禮拜三 19
禮拜二 20
禮拜五 40
禮拜六 60
禮拜六派 0
禮拜四 24
禮拜堂 107
禮拜天 46
禮拜寺 0
禮拜式 1
禮拜日 21
禮教 30
禮數 11
禮斗 1
禮服 635
禮服店 2
禮樂 23
禮樂之邦 0
禮樂射御 0
禮法 18
禮炮 12
禮物 1018
禮畢 8
禮盒 0
禮砲 11
禮節 250
禮節為治事之本 2
禮義 16
禮義廉恥 8
禮聘 106
禮葬 1
禮記 48
禮讓 59
禮讓性 0
禮讓為國 0
禮讚 104
禮貌 286
禮貌性 25
禮賓司 7
禮賢下士 3
禮車 52
禮輕人意重 0
禮輕情意重 1
禮遇 113
禮運篇 1
禮部 10
禮金 67
禮餅 16
禯 0
禰 137
禰 137
禱 722
禱告 1406
禱告詞 1
禱詞 11
禲 5783
禳 2793
禴 6
禵 0
禶 217
禶 217
禷 170
禸 117
禹 380
禺 26
禺 26
离 497
禽 520
禽流感 0
禽獸 35
禽獸不如 8
禽獸行為 0
禽舍 23
禽類 30
禽鳥 16
禾 1119
禾木旁 0
禾本科 29
禾稻 0
禾苗 3
禾菽 0
禿 143
禿人 0
禿子 24
禿尾 0
禿尾 0
禿山 1
禿樹 1
禿毛 1
禿禿 3
禿筆 7
禿落 0
禿頂 18
禿頭 93
禿驢 9
禿髮 21
禿髮症 2
禿鷹 37
秀 8601
秀外慧中 8
秀姑巒溪 149
秀媚 13
秀安 5
秀山 67
秀峰 61
秀才 105
秀才不出門 27
秀挺 1
秀朗 152
秀林 172
秀氣 17
秀水 361
秀眉 10
秀美 126
秀而不實 0
秀色可餐 5
秀逸 3
秀雅 9
秀髮 86
秀麗 239
秀麗挺拔 1
私 1279
私下 331
私下裡 0
私了 2
私事 32
私交 8
私交甚篤 3
私人 1549
私人機構 31
私人機關 2
私人秘書 14
私仇 5
私信 3
私債 0
私允 0
私刑 27
私利 72
私務 2
私名號 0
私吞 2
私商 1
私囊 9
私地 8
私塾 24
私塾弟子 0
私大 10
私奔 26
私娼 18
私學 12
私定終身 4
私家 35
私家偵探 16
私家轎車 1
私帶 2
私底下 98
私德 10
私心 46
私念 2
私怨 3
私情 25
私惠 1
私意 8
私慾 26
私房 248
私房錢 22
私會 2
私有 440
私有制 18
私有土地 77
私有地 15
私有財產 21
私校 152
私梟 13
私欲 9
私法 50
私營 62
私營企業 41
私生女 7
私生子 12
私生活 59
私產 4
私用 56
私益 4
私相 4
私相傳授 0
私相授受 0
私立 7330
私立中學 22
私立學校 661
私立高中 77
私章 88
私線 0
私自 176
私自同意 0
私自答應 0
私船 0
私蓄 1
私藏 36
私處 27
私見 11
私訪 3
私設 14
私設公堂 0
私話 1
私語 22
私貨 16
私販 1
私賣 0
私通 21
私運 9
私邸 0
私酒 5
私釀 4
私鬥 4
私鹽 10
私黨 0
秂 0
秃 0
秄 0
秅 167
秆 0
秇 0
秈 222
秉 987
秉公 10
秉公處理 6
秉公處理 6
秉公辦理 0
秉性 5
秉承 79
秉持 579
秉持著 192
秉政 6
秉燭夜遊 2
秉直 0
秉著 32
秊 0
秋 4735
秋事 1
秋令 0
秋刀魚 1
秋分 28
秋千 20
秋天 393
秋天裡 7
秋季 470
秋季旅行 3
秋山 94
秋後 39
秋後算帳 0
秋後算賬 0
秋思 8
秋意 29
秋意已濃 0
秋成 3
秋扇 0
秋播 4
秋收 12
秋收季節 1
秋收起義 0
秋日 28
秋景 6
秋毫 1
秋毫無犯 0
秋氣 2
秋水 77
秋水伊人 1
秋汛 1
秋決 7
秋波 22
秋波 22
秋海棠 44
秋涼 21
秋涼時節 0
秋瑾 7
秋田 80
秋節 39
秋老虎 6
秋耕 0
秋聲 15
秋色 40
秋色宜人 0
秋葉 60
秋蟬 6
秋蟲 9
秋試 0
秋雨 113
秋霜 16
秋顏 0
秋風 193
秋風掃落葉 5
秋風過耳 0
秋香 55
秋高氣爽 55
秋高馬肥 0
种 1025
秎 7
秏 160
秐 0
科 18088
科任 79
科倫坡 0
科儀 1
科克 7
科刑 15
科別 1467
科協 3
科名 337
科員 90
科場 3
科場 3
科委 19
科威特 122
科威特人 0
科學 15033
科學儀器 167
科學化 69
科學園 76
科學園區 1168
科學家 1084
科學實驗 70
科學性 53
科學技術 63516
科學方法 66
科學時代 4
科學界 85
科學知識 89
科學研究 430
科學社會主義 3
科學管理 24
科學萬能 3
科學試驗 1
科學院 373
科室 129
科展 353
科幻 287
科幻小說 106
科幻性 1
科幻片 21
科幻電影 24
科技 58810
科技大學 1
科技大樓 64
科技性 59
科技股 0
科摩羅 3
科教 469
科教性 0
科教片 0
科教組 0
科普 37
科氏力 0
科爾 21
科爾尼洛夫 0
科班 304
科班出身 23
科甲 34
科目 6618
科目表 192
科研 166
科研成果 5
科研社 13
科系 2566
科組 133
科羅拉多 199
科考 31
科臼 0
科舉 35
科舉制度 4
科西嘉 7
科長 422
科際 157
科頭跣足 0
秒 3157
秒數 42
秒表 0
秒針 13
秒錶 0
秒鐘 152
秔 0
秔 0
秕 12
秕政 1
秖 13
秗 0
秘 274
秘 2746
秘辛 100
秘史 9
秘奧 4
秘宗 3
秘密 2027
秘方 186
秘書 1738
秘書室 572
秘書處 487
秘書長 1131
秘書黨 0
秘笈 418
秘籍 3
秘而不宣 4
秘藏 7
秘要 9
秘訣 852
秘道 0
秘魯 120
秘魯人 1
秙 0
秚 0
秛 0
秜 12
秝 79
秞 268
租 2606
租下 14
租下來 7
租佃 35
租借 175
租借人 1
租借地 1
租出 9
租出去 10
租去 0
租售 1135
租售店 10
租地 103
租契 0
租子 0
租屋 421
租屋人 2
租庸調 0
租成 0
租戶 21
租書 1
租書店 1
租期 499
租用 564
租界 32
租稅 804
租籍 0
租米 0
租約 188
租給 65
租船 8
租船費 0
租費 243
租賃 1356
租賃契約 49
租起 0
租起來 0
租車 441
租車費 5
租過 3
租金 625
秠 7
秡 0
秢 0
秣 197
秣槽 0
秣馬 1
秣馬厲兵 7
秤 702
秤 702
秤坨 0
秤斤注兩 0
秤星 2
秤桿 0
秤盤 9
秤薪而爨 0
秤量 30
秤鉤 0
秤錘 8
秥 0
秦 2181
秦人 8
秦代 10
秦國 35
秦始皇 91
秦嶺 19
秦律 1
秦朝 10
秦朝人 0
秦檜 19
秦淮河 1
秦漢 10
秦王 28
秦皇島 7
秦穆公 6
秦腔 10
秧 256
秧歌 13
秧田 1
秧苗 11
秧齡 0
秨 0
秩 301
秩序 1278
秩序井然 9
秪 535
秫 26
秬 22
秭 22
秮 4
称 0
称 0
称 0
秱 0
秲 0
秳 0
秵 0
秶 483
秷 203
秸 145
秹 0
秺 88
移 4370
移上 2
移上來 2
移上去 1
移下 3
移下來 0
移下去 0
移交 268
移位 135
移作 37
移來 11
移入 364
移出 179
移出來 0
移出去 1
移到 792
移動 2145
移動性 37
移動率 8
移動臂 1
移動臂 1
移去 37
移向 37
移好 1
移孝作忠 0
移審 3
移居 117
移山 5
移山倒海 0
移山填海 7
移師 0
移情 19
移情別戀 24
移成 0
移挪 1
移掉 20
移時 5
移栽 0
移植 858
移植性 0
移樽 0
移樽就教 2
移步 2845
移民 2111
移民局 49
移民法 56
移節 2
移至 15
移花接木 24
移苗 0
移行 54
移調 19
移轉 2024
移轉到 44
移送 662
移過 8
移過來 4
移過去 3
移錯 0
移開 66
移防 2
移防部隊 0
移除 0
移靈 21
移項 2
移風易俗 11
移駕 0
秿 0
稀 376
稀土 25
稀土元素 6
稀土金屬 1
稀土鈷 0
稀奇 107
稀奇古怪 35
稀客 13
稀少 228
稀性 5
稀有 371
稀有動物 37
稀有金屬 4
稀泥 10
稀爛 7
稀疏 63
稀硫酸 3
稀稀疏疏 4
稀粥 4
稀罕 21
稀薄 76
稀釋 428
稀釋液 26
稀釋液 26
稀釋溶液 0
稀釋溶液 0
稀飯 50
稀鬆 22
稀鹽酸 3
稁 0
稂 119
稂莠 0
稃 6
稄 6
稅 2886
稅制 181
稅則 130
稅務 1497
稅務司 7
稅務員 14
稅務局 50
稅務機關 69
稅務處 1
稅吏 17
稅單 65
稅契 1
稅捐 392
稅捐稽征處 5
稅捐稽徵 285
稅捐稽徵處 800
稅捐處 236
稅收 317
稅款 355
稅法 1300
稅率 690
稅目 70
稅課 126
稅負 119
稅賦 520
稅金 111
稅額 710
稆 0
稇 0
稈 15
稉 0
稉 0
稊 78
程 14756
程咬金 9
程子 2
程子 2
程序 6751
程序性 52
程序控制 79
程序設計 29
程序設計員 0
程序設計語言 2
程度 4574
程式 26186
程式化 50
程式庫 342
程式碼 482
程式組 25
程式設計 1437
程式館 31
程硯秋 1
程邈 1
稌 4
稍 11
稍 3484
稍事 37
稍來 20
稍侯 2
稍候 294
稍候一下 0
稍候一下 0
稍候片刻 3
稍出 0
稍加 100
稍可 7
稍嫌 91
稍安勿燥 0
稍安勿躁 0
稍待 83
稍後 294
稍微 502
稍息 23
稍感 7
稍感不便 0
稍感不適 0
稍懈 3
稍早 131
稍有 254
稍減 28
稍為 94
稍為 94
稍熱 1
稍稍 157
稍等 32
稍等一下 23
稍等一下 23
稍等片刻 0
稍緩 15
稍縱即逝 27
稍見 8
稍覺 3
稍許 13
稍遠 6
稍高於 8
稍點 0
税 0
稏 0
稐 3
稑 2
稒 6
稓 12
稔 39
稔知 0
稕 2
稖 0
稗 21
稗 21
稗史 3
稗子 2
稗官 0
稗官野史 3
稘 8
稘 8
稙 25
稚 248
稚女 4
稚嫩 27
稚子 22
稚小 0
稚年 4
稚拙 1
稚暉 14
稚氣 15
稚氣未脫 6
稚齒 0
稚齡 22
稛 5
稜 473
稜稜睜睜 0
稜線 176
稜角 19
稜鏡 51
稝 0
稞 17
稟 148
稟受 2
稟告 0
稟報 8
稟性 3
稟性忠厚 0
稟承 8
稟明 3
稟白 0
稟著 3
稟覆 0
稟賦 16
稟陳 0
稠 199
稠人廣眾 0
稠密 39
稠密性 1
稠度 8
稠糊 1
稡 0
稢 3
稦 3
稧 0
稨 1
稩 0
稪 0
稫 3
稬 0
稭 0
稭 0
種 13812
種 13812
種上 15
種下 87
種下禍根 0
種別 59
種到 8
種因 46
種地 10
種子 2250
種子 2250
種子園 3
種子地 0
種子田 0
種子隊 4
種差 3
種得 395
種性 32
種戶 13
種族 490
種族主義 18
種族問題 12
種族歧視 55
種族隔離 20
種植 561
種植園 2
種植業 3
種概念 4
種樹 89
種牛 6
種牛痘 1
種玉 4
種瓜得瓜 4
種田 40
種田人家 0
種畜 39
種痘 1
種的 394
種禍 0
種種 1437
種稻 13
種籽 0
種群 10
種肥 3
種花 81
種花草 0
種菜 44
種蛋 12
種豆 2
種豆得豆 3
種豬 89
種起 5
種過 10
種過來 0
種過去 0
種類 4046
種馬 30
種麻 3
稯 7
稰 4
稱 1003
稱 1003
稱 10031
稱之為 553
稱作 107
稱便 5
稱做 75
稱兄道弟 7
稱兵 0
稱出 6
稱呼 400
稱善 4
稱孤道寡 0
稱得上 104
稱得起 0
稱心 12
稱心如意 12
稱快 3
稱快一時 0
稱意 2
稱慶 8
稱曰 0
稱歎 0
稱為 2379
稱王 45
稱王稱霸 1
稱病 3
稱羨 0
稱職 158
稱臣 9
稱號 73
稱觴 0
稱許 97
稱說 7
稱謂 88
稱謂語 2
稱謝 22
稱譽 25
稱讚 301
稱讚不已 2
稱貸 0
稱贊 17
稱身 1
稱述 2
稱道 56
稱錢 0
稱雄 17
稱霸 126
稱霸一方 2
稱霸世界 1
稱霸全球 1
稱頌 49
稱頌一時 0
稱願 5
稱體裁衣 0
稲 0
稴 0
稴 0
稴 0
稵 0
稶 0
稷 30
稷下 9
稸 0
稹 95
稺 0
稻 622
稻作 78
稻地 1
稻場 0
稻場 0
稻子 9
稻株 1
稻殼 7
稻浪 3
稻熱病 10
稻田 174
稻瘟病 0
稻秧 5
稻種 10
稻穀 93
稻穗 27
稻米 241
稻粒 0
稻苗 3
稻草 84
稻草人 54
稻香 56
稼 90
稼穡 2
稽 375
稽 375
稽察 152
稽察員 1
稽征 7
稽徵 777
稽徵處 9
稽查 409
稽查人員 26
稽查員 12
稽核 1191
稽留 0
稽覈 0
稽顙 0
稽首 11
稾 0
稿 1338
稿子 100
稿件 303
稿作 2
稿子 44
稿底 1
稿本 16
稿紙 77
稿費 63
稿酬 36
穀 283
穀倉 45
穀場 4
穀場 4
穀子 6
穀殼 4
穀物 215
穀皮 1
穀神 1
穀粒 35
穀艙 0
穀苗 0
穀草 1
穀賤傷農 1
穀道 0
穀雨 6
穀類 159
穀類作物 5
穁 0
穂 0
穄 74
穅 0
穆 699
穆斯林 11
穆然 0
穆爾 10
穆穆 0
穆罕默德 20
穇 68
穈 7844
穉 0
穊 2
穋 5
穌 116
穌 116
積 4210
積久 7
積冰 1
積分 468
積分器 8
積分學 0
積分法 10
積分電路 0
積到 6
積勞成疾 5
積化和差 0
積善 14
積在 25
積壓 79
積存 44
積存多年 0
積存已久 0
積少成多 17
積層雲 0
積年 2
積年累月 0
積弊 14
積弊已久 1
積弊已深 3
積弊難返 0
積弱不振 10
積德 20
積怨 21
積憂成疾 0
積攢 1
積效 16
積木 140
積案 12
積案如山 2
積極 7807
積極份子 1
積極份子 1
積極分子 0
積極分子 0
積極因素 0
積極性 68
積極論 0
積欠 157
積欠已久 0
積水 299
積沙成塔 2
積溫 2
積滿 17
積漸 6
積痾 0
積福 2
積糧 1
積累 47
積累基金 0
積習 13
積習已久 1
積習難改 3
積習難返 0
積聚 50
積肥 0
積著 4
積蓄 93
積貯 1
積重難返 4
積雨 0
積雨雲 21
積雪 58
積雪已深 0
積雲 17
積非成是 13
積體 59
積體電路 1075
積鬱 4
穎 1887
穎川 8
穎州 1
穎悟 10
穎慧 4
穎慧過人 0
穎果 4
穎異 2
穎秀 2
穎脫 0
穎花 2
穎黠 0
穏 0
穓 0
穔 197
穕 0
穖 85
穗 957
穗子 57
穗狀 3
穗狀花序 12
穗肥 0
穘 58
穙 0
穚 133
穛 74
穜 85
穟 5
穠 78
穠纖 3
穡 16
穡人 0
穡夫 0
穢 135
穢亂 0
穢史 3
穢名 0
穢土 3
穢氣 5
穢物 7
穢聞 0
穢行 0
穢褻 0
穢語 12
穢跡金剛 1
穤 0
穥 0
穦 0
穧 83
穨 51
穩 1740
穩住 104
穩便 3
穩健 459
穩健作風 0
穩固 238
穩如 8
穩如泰山 5
穩妥 11
穩婆 0
穩定 4461
穩定下來 25
穩定住 0
穩定劑 15
穩定性 462
穩度 13
穩得 11
穩拿 2
穩拿到 0
穩操 12
穩操勝券 4
穩步 14
穩步前進 1
穩步邁進 2
穩產高產 0
穩當 34
穩當性 0
穩穩固固 0
穩穩定定 0
穩穩當當 5
穩穩重重 0
穩紮 3
穩紮穩打 21
穩賺 28
穩重 104
穩重性 2
穪 0
穪 0
穪 0
穫 309
穬 117
穬 117
穭 370
穮 1028
穯 0
穰 30
穰 30
穱 44
穲 0
穳 0
穳 0
穴 822
穴 822
穴位 71
穴位 71
穴居 13
穴居人 1
穴播 0
穴播 0
穴道 148
穴道 148
穵 4
究 49
究 4987
究其根源 1
究問 3
究理 7
究竟 1599
究竟是 433
究辦 22
穸 236
穹 105
穹 105
穹天 0
穹天 0
穹廬 0
穹廬 0
穹空 1
穹蒼 28
穹蒼 28
空 786
空 7863
空上 18
空下 19
空下來 3
空下去 0
空中 1974
空中商專 31
空中大學 598
空中小姐 28
空中少爺 0
空中樓閣 17
空中纜車 15
空中飛人 61
空串列 1
空乏 9
空乏 9
空了 53
空位 33
空位 33
空個 1
空兒 1
空兒 1
空出 76
空出 76
空出來 11
空出來 11
空函 0
空到 234
空前 253
空前絕後 20
空劫 68
空勤 12
空勤人員 8
空包 12
空包彈 15
空匱 0
空即是色 6
空口 13
空口無憑 0
空口說白話 3
空名 1
空名額 0
空喊 9
空喊一聲 0
空喊一聲 0
空在 34
空地 271
空城 24
空城計 29
空大 250
空如來藏 1
空姐 207
空子 2
空官 18
空射 4
空射一槍 0
空射一槍 0
空對地 6
空對地飛彈 1
空對空 4
空對空飛彈 9
空屋 56
空幻 2
空床 11
空座 2
空彈 2
空得 0
空心 127
空心球 2
空心磚 23
空心菜 59
空忙 0
空忙一場 0
空忙一場 0
空忙一場 0
空忙一場 0
空想 33
空想社會 0
空想社會主義 0
空成 3
空戰 139
空戶 4
空房 15
空房子 4
空手 120
空手而來 2
空手而回 24
空手而歸 5
空手道 272
空投 25
空投物資 0
空拳 4
空明 7
空曠 80
空有 76
空有形式 0
空服員 0
空架子 2
空校 2
空格 1128
空格鍵 0
空桶 7
空檔 122
空檔 122
空櫃子 0
空欄 9
空權 7
空歡喜 5
空殼 12
空殼子 8
空氣 5939
空氣冷卻 1
空氣冷卻器 10
空氣噴氣 0
空氣噴氣發動機 0
空氣囊 3
空氣墊 1
空氣壓縮 14
空氣壓縮機 66
空氣密度 8
空氣流通 29
空氣濾清 4
空氣濾清器 29
空氣調節 61
空氣阻力 5
空泛 23
空泡 12
空洞 152
空洞洞 1
空洞無物 1
空測 3
空炮彈 1
空無 25
空無一人 28
空無一人 28
空無一物 20
空無一物 20
空瓶 40
空白 1556
空白一片 5
空白一片 5
空白鍵 61
空的 239
空盒子 2
空盤 3
空穴 9
空穴來風 17
空空 128
空空如也 6
空空洞洞 8
空窗期 0
空等 10
空等一場 2
空等一場 2
空等一場 2
空等一場 2
空箱子 1
空缺 34
空翻 53
空耗 2
空肚 0
空肚子 0
空腹 111
空航 8
空船 8
空茫 2
空著 152
空蕩 10
空蕩蕩 31
空虛 163
空虛感 15
空號 21
空行 1
空行 31
空行 31
空行母 1
空行法 1
空襲 260
空襲警報 11
空言 16
空話 6
空說 11
空調 1570
空調設備 124
空談 41
空論 4
空諦 1
空谷 18
空谷回音 0
空谷足音 3
空費 3
空走 5
空走一趟 0
空走一趟 0
空起 0
空起來 0
空跑 2
空跑一趟 5
空跑一趟 5
空身 2
空車 58
空車子 0
空軍 685
空軍基地 44
空軍官校 76
空軍機校 1
空軍機械學校 5
空軍節 0
空軍航空技術學院 1
空軍軍官 4
空軍軍官學校 79
空軍通信電子學校 2
空軍通校 1
空軍醫院 4
空載 43
空轉 25
空速 41
空運 374
空運公司 3
空道 9
空鐘 1
空門 28
空閑 4
空閒 109
空間 11665
空間感 29
空間結構 16
空間電荷 1
空閨 3
空閨獨守 0
空闊 4
空防 14
空降 52
空降師 28
空降部隊 11
空隙 123
空際 2
空集合 1
空難 249
空難事件 17
空靈 26
空頭 231
空頭支票 8
空額 0
空飄 20
空餘 32
穻 0
穼 0
穽 0
穾 0
穿 4237
穿上 285
穿上來 0
穿上去 2
穿下 1
穿下來 0
穿下去 0
穿不下 13
穿不出 0
穿井 18
穿住 0
穿來 0
穿來穿去 1
穿入 28
穿出 48
穿出來 1
穿出去 5
穿到 9
穿刺 142
穿去 2
穿堂 13
穿壁引光 0
穿壞 2
穿好 26
穿孔 91
穿孔機 0
穿孔紙帶 1
穿孝 1
穿山 23
穿山甲 26
穿山而過 0
穿幫 24
穿得 218
穿心 35
穿戴 127
穿戴整齊 5
穿插 127
穿插在 13
穿插著 6
穿梭 324
穿梭外交 1
穿梭轟炸 0
穿洞 10
穿牆 50
穿用 6
穿甲彈 0
穿的 218
穿穿 4
穿穿看 2
穿紅戴綠 0
穿給 0
穿線 71
穿耳 2
穿耳洞 38
穿著 1135
穿著打扮 19
穿衣 72
穿衣服 71
穿衣鏡 5
穿起 22
穿起來 20
穿越 317
穿越道 2
穿身而過 0
穿透 270
穿透力 28
穿透性 29
穿透電流 0
穿通 11
穿過 411
穿過來 0
穿過去 1
穿針 3
穿針引線 21
穿鑿 6
穿鑿 6
穿鑿 6
穿鑿附會 1
穿鑿附會 12
穿門而過 0
穿靴 2
穿鞋 28
穿點 4
穿鼻 0
窀 57
窀穸 1
突 1110
突兀 46
突出 538
突出來 1
突出去 0
突出重圍 1
突刺 8
突升 1
突厥 31
突厥人 9
突嘴 0
突圍 58
突圍而出 5
突地 1
突地 1
突增 15
突如其來 91
突如其然 1
突射 0
突射入門 0
突尼斯 3
突尼西亞 37
突感 7
突懸 0
突擊 87
突擊檢查 6
突擊隊 31
突減 0
突然 3515
突然之間 30
突然襲擊 1
突然間 197
突發 267
突發事件 38
突發奇想 1
突發狀況 66
突破 2215
突破性 99
突破現狀 8
突破點 2
突突 5
突著 0
突襲 72
突襲戰術 0
突見 16
突觸 13
突變 279
突起 110
突起來 1
突越 0
突進 4
突遇 4
突降 0
突顯 298
突顯出 50
突飛猛進 73
窂 0
窃 0
窄 458
窄 458
窄到 4
窄小 25
窄巷 11
窄得 0
窄狹 8
窄的 18
窄地 17
窄路 6
窄道 1
窄門 24
窅 109
窆 12
窇 8
窈 213
窈窕 87
窈窕淑女 27
窉 34
窊 37
窋 29
窌 19
窏 105
窐 233
窐 233
窑 0
窒 455
窒塞 0
窒息 130
窒息而死 8
窒欲 0
窒礙 12
窒礙難行 35
窓 0
窔 80
窕 76
窕邃 0
窖 1158
窖子 0
窗 4232
窗口 918
窗台 40
窗台上 14
窗外 459
窗子 84
窗帘 32
窗帷 1
窗戶 304
窗戶上 9
窗明几淨 8
窗明几淨 8
窗框 27
窗檯 5
窗櫥 1
窗櫺 19
窗洞 1
窗玻璃 11
窗簾 230
窗縫 0
窗臺 3
窘 110
窘 110
窘乏 0
窘乏 0
窘匱 0
窘匱 0
窘厄 0
窘厄 0
窘困 9
窘困 9
窘境 133
窘局 1
窘急 1
窘惶 0
窘惶 0
窘態 21
窘態 21
窘態畢露 3
窘態畢露 3
窘敗 0
窘敗 0
窘相 0
窘色 0
窘色 0
窘蹙 0
窘迫 39
窙 3
窚 0
窛 0
窞 5
窟 619
窟室 0
窟穴 2
窟窖 0
窟窿 10
窠 17
窠巢 0
窠穴 1
窠臼 46
窡 0
窢 0
窣 8
窣窣 1
窤 0
窨 3
窩 6586
窩主 13
窩囊 16
窩囊廢 0
窩囊氣 1
窩子 4
窩巢 7
窩工 1
窩心 125
窩棚 1
窩瓦河 2
窩窩頭 1
窩著 1
窩藏 51
窩藏禍心 0
窩藏罪犯 0
窩裡 21
窩裡反 15
窩裡窩囊 0
窩逃 0
窩闊臺 0
窩頭 1
窪 186
窪地 25
窪陷 0
窫 3
窬 8
窮 591
窮乏 10
窮人 168
窮人家 8
窮兇極惡 4
窮光 0
窮光蛋 11
窮兵 1
窮兵黷武 7
窮冬 0
窮凶惡極 4
窮凶極惡 4
窮則變 3
窮厄 0
窮困 31
窮困人家 1
窮奢極侈 2
窮奢極多 0
窮奢極欲 0
窮學生 12
窮家富路 0
窮寇勿追 0
窮寇莫追 1
窮小子 5
窮山惡水 4
窮巷 1
窮忙 0
窮愁潦倒 0
窮日 0
窮日子 0
窮書生 5
窮本 2
窮棒子 0
窮極 10
窮極無聊 4
窮歡樂 0
窮民 1
窮理 7
窮當益堅 0
窮盡 34
窮目 0
窮相 0
窮究 11
窮窘 0
窮苦 16
窮苦人家 1
窮迫 0
窮追 20
窮追不捨 15
窮途 2
窮途末路 12
窮途潦倒 1
窮通 7
窮達 0
窮鄉僻壤 15
窮酸 19
窮酸相 0
窮里 0
窮開心 3
窮隊 0
窮骨頭 0
窮鬼 1
窯 1018
窯主 6
窯器 0
窯姐 0
窯子 20
窯洞 3
窯灶 0
窯變 19
窰 0
窱 131
窲 43
窳 70
窳劣 0
窳敗 0
窳民 0
窳陋 3
窴 91
窵 101
窶 146
窷 116
窸 94
窸窣 7
窸窸窣窣 7
窹 0
窺 350
窺伺 13
窺察 0
窺探 52
窺探究竟 0
窺測 3
窺測孔 0
窺看 6
窺知 22
窺見 56
窺視 81
窺覦 0
窺覬 0
窺豹一斑 0
窻 0
窼 0
窼 0
窽 0
窾 80
窿 12
窿緣桉 0
竀 43
竁 67
竂 0
竄 318
竄入 12
竄出 48
竄改 105
竄謀 0
竄逃 6
竄進 1
竅 123
竅門 38
竆 0
竇 377
竇娥 5
竇娥冤 5
竇建德 9
竇憲 2
竈 0
竉 0
竊 2826
竊位 0
竊取 107
竊名 0
竊喜 12
竊國 1
竊國賊 0
竊嫌 5
竊據 6
竊案 30
竊犯 0
竊用 7
竊盜 238
竊盜罪 18
竊竊私語 14
竊竊私議 1
竊笑 17
竊聽 76
竊聽器 7
竊賊 44
立 16687
立上 4
立下 153
立了 88
立井 0
立人 186
立位 0
立住 3
立冬 18
立刻 2130
立刻來 2
立刻到 12
立刻去 17
立功 34
立功贖罪 0
立升 2
立即 4057
立可白 1
立名 5
立命 12
立嗣 3
立國 280
立地 34
立地成佛 3
立地條件 3
立場 216
立場 2162
立夏 12
立大功 15
立契 1
立好 9
立委 5778
立姿 18
立姿射擊 0
立定 51
立射 0
立式 263
立德 268
立志 246
立息 0
立意 70
立憲 26
立憲國 0
立憲政體 2
立憲民主 2
立憲民主黨 0
立憲派 0
立憲運動 3
立成 64
立新 208
立方 80
立方公分 18
立方公寸 1
立方公尺 376
立方公里 3
立方厘米 1
立方根 2
立方米 55
立方英尺 3
立方體 29
立於 1002
立於不敗 0
立於不敗之地 27
立春 23
立時 218
立木 19
立本 28
立枯病 2
立柱 15
立案 889
立業 68
立櫃 5
立正 92
立正口令 0
立法 2374
立法委員 1562
立法機關 74
立法權 92
立法院 3182
立消 1
立為 34
立狀 2
立生 18
立異 3
立碑 49
立秋 21
立竿見影 41
立等 54
立等可取 0
立約 52
立群 167
立腳 4
立腳點 0
立號 1
立見 12
立見好轉 0
立視圖 0
立言 52
立誓 21
立說 8
立談之間 0
立論 84
立起 32
立起來 12
立足 226
立足點 42
立身 33
立軸 5
立錐之地 4
立院 1064
立陶宛 88
立陶宛人 1
立雪 2
立霧溪 72
立面圖 68
立體 1414
立體化 23
立體圖 61
立體幾何 6
立體性 2
立體感 38
立體戰 0
立體聲 93
立體電影 7
立體電視 1
立體音響 39
竌 0
竎 0
竐 0
竑 161
竒 0
竒 0
竘 104
站 42575
站上 1541
站上來 2
站上去 13
站下來 0
站下去 1
站不住 6
站不住腳 14
站不穩 14
站不起來 8
站了 177
站住 34
站個 10
站出 21
站出來 141
站出去 1
站到 67
站務員 6
站台 5664
站在 1941
站好 45
站定 35
站崗 22
站崗 22
站得 33
站得住 1
站得住腳 6
站攏 0
站法 3
站牌 0
站直 15
站票 9
站穩 145
站穩立場 3
站穩立場 3
站穩腳步 10
站立 285
站立起來 1
站站 112
站網 230
站著 162
站著不動 4
站起來 261
站進來 1
站進去 1
站過 8
站過來 0
站過去 0
站長 6514
站開 32
竚 0
竛 0
竜 0
竜 0
竝 0
竟 2493
竟不是 4
竟不能 10
竟來 6
竟因 18
竟在 64
竟夜 6
竟如 44
竟把 39
竟敢 84
竟於 11
竟日 8
竟日來 0
竟是 510
竟會 82
竟有 206
竟然 1967
竟然是 161
竟然會 45
竟然有 106
竟用 6
竟而 1
竟能 92
竟至 6
竟與 16
竟被 76
竟要 21
竟讓 20
竟達 9
竟陵 15
竟陵派 1
竟需 10
竟須 0
章 34065
章則 201
章句 18
章和 36
章回 12
章回小說 26
章回體 0
章太炎 16
章奏 0
章學誠 5
章帝 0
章法 75
章程 1389
章節 496
章草 2
章邯 0
章魚 138
章魚燒 0
竢 0
竢 0
竣 1529
竣事 8
竣工 216
竤 16
童 3986
童僕 3
童叟無欺 7
童女 52
童子 158
童子功 1
童子軍 827
童山濯濯 6
童工 36
童年 854
童年時代 9
童年時期 13
童心 125
童心未泯 16
童書 1
童玩 0
童男 4
童真 24
童稚 27
童聲 5
童裝 173
童言 144
童言戲語 0
童言無忌 6
童話 748
童話故事 108
童謠 162
童貞 7
童身 2
童軍 1646
童音 6
童顏 58
童顏鶴髮 0
童養媳 3
竦 8
竦動 0
竦懼 0
竦然 1
竦立 0
竦聽 0
竧 0
竨 0
竩 0
竩 0
竪 0
竫 7
竬 0
竭 596
竭力 105
竭力反對 0
竭力支持 0
竭心 2
竭慮 6
竭智盡力 3
竭智盡慮 0
竭澤而漁 0
竭盡 56
竭盡全力 13
竭盡力量 0
竭盡所能 62
竭盡棉薄 0
竭能 0
竭誠 200
竭誠歡迎 233
竮 3
端 4986
端上 46
端上來 3
端上去 0
端下 8
端下來 0
端下去 1
端了 16
端來 23
端來端去 0
端倪 75
端出 79
端出來 1
端出去 0
端午 101
端午節 202
端去 2
端坐 58
端居 0
端方 4
端架子 0
端正 189
端由 3
端的 434
端相 4
端硯 6
端給 2
端緒 1
端線 19
端舉 0
端莊 55
端菜 17
端著 46
端視 123
端詳 50
端賴 81
端走 4
端起 17
端過 1
端過來 0
端過去 0
端陽 5
端陽節 0
端電壓 8
端面 14
端點 74
竱 0
竲 0
竳 0
竴 0
竵 0
競 1294
競價 75
競向 1
競奪 19
競技 399
競技場 58
競技場 58
競業禁止 0
競標 220
競渡 8
競爭 4888
競爭力 2203
競爭心 0
競爭性 96
競爭者 244
競用 2
競相 198
競試 55
競購 2
競賽 2655
競賽者 17
競走 9
競走比賽 0
競逐 64
競選 2341
競選人 1
競選活動 114
競鬥 0
竷 114
竸 0
竹 6956
竹北 938
竹南 1226
竹器 43
竹圍 232
竹園 69
竹報 2
竹報平安 7
竹子 142
竹子湖 89
竹子湖 89
竹山 1073
竹工 32
竹工業 4
竹帛 2
竹扉 0
竹排 3
竹書 5
竹材 187
竹東 913
竹板 4
竹林 312
竹林七賢 5
竹林寺 1
竹林路 105
竹枝詞 1
竹架 5
竹條 3
竹棒 7
竹椅 10
竹槓 3
竹樓 1
竹片 20
竹竿 64
竹笠 2
竹筍 126
竹筍乾 0
竹筍湯 2
竹筏 35
竹筒 38
竹筷 12
竹管 10
竹箱 0
竹箱子 0
竹節 57
竹篾 14
竹簍 6
竹簡 12
竹簾 16
竹籃 8
竹籠 8
竹籤 10
竹籬 35
竹籬茅舍 2
竹膜 0
竹葉 22
竹葉青 8
竹葦 0
竹蓆 6
竹頭木屑 0
竹馬 4
竺 179
竻 85
竼 0
竽 61
竾 0
竾 0
竾 0
竿 264
竿子 12
笀 0
笁 4
笄 42
笅 15
笆 158
笆斗 1
笈 68
笉 21
笊 25
笋 0
笌 0
笍 0
笎 20
笏 534
笐 42
笑 3971
笑一笑 64
笑一笑 64
笑了 339
笑了起來 51
笑來笑去 0
笑個 11
笑個痛快 0
笑傲 79
笑傲江湖 151
笑出 13
笑出來 9
笑出眼淚 3
笑口常開 25
笑吟吟 0
笑呵呵 8
笑咪咪 26
笑哈哈 16
笑嘻嘻 41
笑場 8
笑場 8
笑容 600
笑容可掬 9
笑得 149
笑意 92
笑掉大牙 7
笑料 27
笑林 8
笑柄 9
笑死 26
笑死人 34
笑盈盈 7
笑眼 3
笑瞇瞇 12
笑破 5
笑破肚皮 1
笑笑 154
笑紋 3
笑納 17
笑罵 15
笑聲 339
笑臉 138
笑臉攻勢 2
笑臉迎人 18
笑著 345
笑裡藏刀 5
笑話 4055
笑話百出 2
笑語 57
笑談 56
笑貌 4
笑起 3
笑起來 38
笑逐頻開 0
笑逐顏開 12
笑過 16
笑過來 0
笑過去 0
笑面 15
笑面虎 1
笑靨 33
笑顏 31
笒 16
笓 35
笔 0
笖 0
笗 0
笘 123
笙 862
笙歌 16
笙歌不輟 0
笙簧 1
笙簫 5
笚 137
笛 963
笛卡兒 28
笛卡爾 9
笛子 63
笛手 8
笛聲 31
笛膜 0
笛膜 0
笜 0
笜 0
笝 160
笞 30
笞刑 1
笞擊 0
笞杖 0
笞責 1
笞辱 0
笞辱 0
笟 0
笟 0
笠 269
笡 0
笢 2006
笣 235
笤 21
笤帚 0
笥 321
符 1619
符合 9740
符命 2
符咒 34
符尾 0
符應 16
符水 10
符碼 0
符節 1
符籙 5
符號 1802
符號表 50
符頭 0
符驗 1
笧 0
笧 0
笨 11251
笨伯 1
笨手笨腳 12
笨拙 64
笨桶 1
笨死 7
笨牛 13
笨瓜 5
笨笨 279
笨笨呆呆 1
笨蛋 281
笨豬 9
笨貨 0
笨賊 17
笨重 58
笨頭笨腦 4
笩 0
笪 334
笫 151
笫 151
第 59239
第一 17820
第一人 74
第一人稱 74
第一信號 0
第一信號系統 0
第一個 3116
第一冊 175
第一卷 241
第一名 1785
第一回 61
第一審 80
第一性 1
第一手 158
第一條 1
第一次 6066
第一次世界大戰 62
第一流 69
第一章 1253
第一節 733
第一線 173
第一義 11
第一義諦 1
第一課 153
第一項 1
第七 3811
第七個 20
第七冊 17
第七名 58
第七回 18
第七條 1
第七次 260
第七章 529
第七節 134
第七課 68
第七項 1
第三 12498
第三世界 124
第三人稱 22
第三個 377
第三冊 96
第三名 704
第三回 44
第三國際 2
第三條 1
第三次 1453
第三章 1038
第三等級 0
第三節 514
第三紀 21
第三者 326
第三課 99
第三項 1
第九 2108
第九個 9
第九冊 18
第九名 25
第九回 17
第九條 1
第九次 152
第九章 363
第九節 60
第九課 24
第九項 1
第二 23886
第二世界 6
第二人 25
第二人稱 4
第二信號 0
第二信號系統 0
第二個 864
第二冊 122
第二名 781
第二回 48
第二國際 2
第二宇宙 0
第二宇宙速度 0
第二審 54
第二性 8
第二性徵 1
第二手 7
第二條 1
第二次 2641
第二次世界大戰 193
第二章 1125
第二節 645
第二課 106
第二項 1
第五 4065
第五個 50
第五冊 90
第五名 166
第五回 30
第五條 1
第五次 484
第五章 845
第五節 226
第五課 50
第五項 1
第八 2956
第八個 13
第八冊 15
第八名 45
第八回 13
第八條 1
第八次 162
第八章 432
第八節 86
第八課 28
第八識 1
第八項 1
第六 2529
第六個 35
第六冊 39
第六名 99
第六回 27
第六天魔王 1
第六感 89
第六條 1
第六次 208
第六章 681
第六節 187
第六課 51
第六項 1
第十 9581
第十個 21
第十冊 7
第十名 28
第十回 16
第十條 1
第十次 119
第十章 319
第十節 42
第十課 19
第十項 1
第四 6041
第四個 105
第四冊 67
第四台 474
第四名 203
第四回 33
第四條 1
第四次 435
第四章 968
第四節 411
第四紀 36
第四課 66
第四項 1
第宅 1
第宅 1
第幾 48
第幾個 5
第幾冊 0
第幾名 6
第幾回 0
第幾次 12
第幾章 0
第幾節 1
第幾課 0
笭 385
笮 110
笯 412
笰 103
笱 692
笲 186
笳 131
笳鼓 1
笴 72
笵 70
笶 0
笸 44
笻 4
笿 0
筀 1
筁 0
筂 0
筂 0
筂 0
筃 0
筄 6
筅 11
筆 17000
筆數 112
筆下 253
筆下留情 4
筆伐 3
筆劃 831
筆力 14
筆勢 8
筆友 200
筆名 159
筆墨 231
筆墨生涯 0
筆墨難以形容 7
筆套 6
筆尖 26
筆帽 0
筆底生花 0
筆座 23
筆心 5
筆意 7
筆戰 15
筆挺 13
筆會 29
筆札 0
筆架 68
筆桿 16
筆法 67
筆畫 117
筆盒 15
筆直 93
筆硯 10
筆端 9
筆筒 46
筆筒樹 1
筆算 0
筆管 13
筆者 2332
筆耕 25
筆舌 1
筆芯 5
筆觸 122
筆記 4622
筆記型 36
筆記本 132
筆記簿 36
筆試 695
筆誤 31
筆調 63
筆談 27
筆譯 46
筆資 436
筆跡 42
筆述 100
筆鋒 20
筆錄 118
筆陣 1
筆順 39
筆頭 27
筆頭兒 1
筇 44
筈 14
等 69748
等一下 245
等一下 245
等一會 9
等一會 9
等一會兒 11
等一會兒 11
等一會兒 11
等一會兒 11
等一等 19
等一等 19
等上 37
等下去 27
等不及 56
等了 292
等人 1422
等份 9
等位基因 2
等個 70
等候 716
等候多時 7
等值 103
等值線 7
等偏角 0
等價 24
等價物 0
等分 70
等分比 0
等分線 1
等到 1487
等加速度 4
等勢面 0
等化 61
等同 72
等同於 62
等壓 18
等壓線 7
等外 60
等子 16
等差 7
等差數列 27
等差級數 13
等式 42
等待 2148
等待著 110
等得 77
等得 834
等效 71
等效電路 5
等方性 2
等於 1202
等於在 14
等於是 159
等於有 13
等於零 19
等日 36
等比 13
等比數列 14
等比級數 4
等深線 10
等溫 14
等溫線 0
等的 833
等積形 0
等第 94
等等 3735
等級 2019
等而下之 8
等腰 7
等腰三角 0
等腰三角形 9
等臂天平 0
等臂天平 0
等船 28
等著 939
等著 939
等著瞧 12
等號 65
等角 19
等角三角 0
等角三角形 0
等角線 0
等距 15
等距離 8
等身 22
等車 38
等輩 3
等速 20
等速圓周運動 1
等速度 9
等速轉動 1
等速運動 5
等過 51
等邊 6
等邊三角 2
等邊三角形 4
等重 169
等重量 1
等量 51
等量級 0
等量齊觀 2
等閑 0
等閒 22
等閒之輩 3
等閒視之 16
等離子體 13
等雨線 2
等音線 1
等高 132
等高線 47
筊 25
筊白筍 0
筋 768
筋斗 13
筋疲力盡 11
筋疲力竭 12
筋節 0
筋絡 3
筋肉 21
筋脈 15
筋脈 15
筋豆 0
筋骨 103
筌 81
筍 1932
筍乾 5
筍子 6
筍尖 6
筍絲 14
筍頭 0
筎 10
筏 655
筏子 10
筐 235
筐子 2
筐籠 0
筑 583
筒 684
筒 684
筒子 42
筒狀 18
筒管 1
筓 0
答 684
答 22400
答來答去 0
答允 16
答出 6
答出來 0
答卷 11
答問 111
答好 2
答對 226
答復 85
答應 789
答應下來 4
答數 2
答案 2983
答理 4
答禮 13
答答 24
答聘 0
答腔 0
答複 0
答覆 1096
答言 4
答詢 0
答話 31
答話機 0
答語 0
答謝 74
答謝詞 0
答辭 0
答辯 102
答辯書 11
答錄機 368
答錯 64
答非所問 17
答題 1
筕 0
策 4969
策劃 676
策劃人 10
策劃者 3
策動 58
策勵 25
策反 9
策士 19
策應 7
策杖 2
策源地 0
策略 8479
策略性 231
策畫 128
策試 0
策論 7
策謀 4
策進會 0
策馬 41
策馬前進 0
筗 0
筘 11
筞 0
筟 0
筠 729
筡 6
筢 0
筣 4
筤 1
筥 5
筦 5
筧 18
筧橋 6
筧濟學家 0
筨 0
筩 2
筪 0
筫 0
筫 0
筬 0
筭 14
筮 21
筮書 0
筮法 0
筯 0
筰 4
筱 769
筲 18
筳 3
筴 9
筴 9
筵 121
筵上 2
筵几 0
筵宴 9
筵席 71
筵席 71
筵席捐 0
筵席捐 0
筵會 4
筶 8
筷 46
筷子 121
筸 2
筹 0
箁 0
箁 0
箁 0
箁 0
箂 21
箃 0
箄 20
箅 8
箅子 0
箆 0
箇 120
箇中 56
箇中三昧 1
箇中人 2
箈 7
箉 0
箊 2
箋 249
箋注 3
箋紙 2
箌 4
箍 52
箍桶 2
箍桶匠 0
箎 9
箏 222
箐 4
箑 6
箒 0
箓 0
箔 177
箕 89
箕宿 1
箕帚 0
箕斂 0
箕斗 0
箕裘 1
箕踞 3
箖 65
算 4721
算上 12
算上來 0
算上去 0
算下 2
算下來 2
算下去 3
算不上 21
算不了 4
算不了什麼 10
算不了什麼 10
算不得 8
算不清 6
算了 510
算了又算 0
算作 8
算來算去 1
算做 13
算出 197
算出來 9
算出去 2
算卦 0
算命 332
算命先生 16
算命師 9
算在 26
算在內 3
算學 28
算帳 87
算式 69
算得 69
算得了 1
算得了什麼 6
算得了什麼 6
算得什麼 0
算得什麼 0
算數 91
算是 1581
算法 86
算法語言 0
算清楚 1
算無遺策 1
算盤 52
算盤子 0
算算 65
算算看 22
算術 154
算術家 1
算術級數 0
算術課 0
算計 58
算賬 4
算起 75
算起來 48
算進 5
算進來 0
算進去 18
算錯 31
算題 4
箘 26
箙 16
箛 6
箜 8
箜篌 4
箝 35
箝制 35
箝口 0
箝口結舌 0
箝子 0
箝形攻勢 0
箞 0
箞 0
箟 0
箠 37
箠楚 0
管 16260
管上 26
管下 10
管不了 15
管不了那麼多 5
管不住 8
管不著 11
管中窺物 0
管中窺豹 1
管了 53
管事 22
管人 56
管他 138
管他的 17
管仲 39
管來管去 0
管保 2
管個 1
管內 128
管到 22
管制 4505
管制員 80
管制法 96
管區 73
管及 297
管叫 4
管吃管住 0
管名 1
管壁 72
管外 17
管好 12
管委會 185
管子 161
管子 161
管孔 4
管家 241
管家婆 113
管帶 33
管弦 310
管弦樂 100
管弦樂隊 3
管得 17
管接頭 23
管教 322
管有 65
管材 20
管束 59
管樂 478
管樂器 31
管樂隊 41
管法 23
管渠 9
管狀 49
管理 53549
管理人 288
管理區 33
管理員 975
管理學院 2162
管理局 1172
管理師 216
管理所 207
管理權 87
管理站 132
管理者 1120
管理處 2003
管理費 184
管理費用 37
管理體制 20
管用 129
管窺蠡測 0
管笛 2
管管 47
管籥 0
管紗 1
管絃 167
管絃樂 21
管絃樂隊 0
管網 35
管線 598
管胞 0
管著 3
管見 16
管見所及 0
管賬的 0
管贏 0
管路 317
管轄 392
管轄權 53
管過 5
管道 2744
管道工 1
管道工程 13
管閒事 5
管領 16
管風琴 48
管飽 0
管鮑 1
管點 3
箢 0
箣 0
箤 9
箥 0
箬 14
箬竹 0
箬笠 2
箭 1007
箭在弦上 23
箭如雨下 0
箭尖 4
箭尾 5
箭桿 0
箭步 18
箭毒 0
箭無虛發 0
箭矢 19
箭程 1
箭竹 1
箭竹筍 1
箭號 11
箭豬 2
箭鏃 2
箭靶 30
箭頭 247
箮 0
箯 6
箰 0
箱 9155
箱匣 2
箱子 138
箱底 10
箱根 113
箱篋 0
箱籠 2
箲 0
箲 0
箳 0
箴 209
箴石 0
箴砭 1
箴規 0
箴言 121
箴誡 0
箴諫 0
箵 4
箶 0
箷 49
箷 49
箸 113
箹 3
箺 0
箻 0
箽 0
箾 3
箿 0
節 6423
節令 27
節假日 4
節儉 43
節制 221
節制資本 2
節哀 7
節哀順變 5
節圓 2
節士 0
節外生枝 20
節奏 570
節奏性 10
節奏明朗 0
節奏輕快 5
節子 4
節孔 1
節序 0
節度 8
節度使 5
節律 44
節慶 571
節慶典禮 0
節慾 2
節拍 48
節拍器 11
節操 6
節敬 3
節數 128
節日 327
節本 2
節概 1
節氣 97
節流 51
節流閥 57
節烈 3
節片 5
節理 37
節用 11
節略 2
節目 5500
節目單 75
節目表 446
節目部 33
節省 1567
節省下 21
節省下來 0
節省時間 97
節省能源 54
節禮 5
節稅 663
節節 134
節節上升 13
節節上漲 1
節節勝利 0
節節敗退 13
節約 516
節約用水 185
節約用電 20
節線 4
節義 20
節肢 4
節肢動物 39
節育 33
節育法 0
節自 11
節行 1
節衣縮食 1
節譯 8
節足動物 0
節錄 1086
節錄自 139
節間 14
節食 64
節餘 16
節骨眼 9
節骨眼兒 0
節骨眼兒 0
節點 410
篁 65
篂 0
篃 0
範 559
範例 2916
範圍 9374
範式 11
範文 196
範本 376
範疇 456
篅 0
篆 103
篆刻 330
篆字 5
篆文 5
篆書 24
篆章 0
篆籀 1
篆隸 6
篆體 1
篇 20223
篇什 2
篇名 294
篇幅 281
篇數 199
篇次 4
篇目 833
篇章 96
篇簡 11
篇籍 2
篈 0
築 2676
築城 15
築堤 73
築室 2
築室道謀 0
築屋 10
築巢 104
築成 18
築有 20
築欄 0
築牆 2
築起 29
築路 15
築路工程 1
篊 4
篋 75
篌 15
篍 0
篎 10
篔 130
篕 291
篖 0
篗 0
篘 87
篙 60
篚 64
篛 10
篜 41
篝 50
篝火 13
篞 121
篟 54
篠 104
篡 152
篡位 16
篡奪 7
篡弒 0
篡改 25
篡竊 0
篡黨 1
篢 36
篣 346
篤 210
篤信 37
篤厚 0
篤學 0
篤守 0
篤定 57
篤實 18
篤志 2
篤愛 0
篤敬 0
篤疾 0
篤直 0
篤行 47
篤誠 2
篥 62
篦 17
篦麻 3
篦麻子 1
篦麻油 1
篧 54
篨 246
篩 773
篩下 0
篩分 32
篩子 6
篩板 2
篩檢 0
篩狀 1
篩號 0
篩選 564
篩酒 0
篪 34
篫 25
篬 0
篬 0
篭 0
篭 0
篰 36
篱 114
篲 102
篳 70
篳路藍縷 29
篴 85
篵 0
篶 0
篷 91
篷車 33
篷馬車 0
篸 48
篹 38
篺 0
篻 153
篼 0
篽 117
篾 34
篿 43
簀 57
簀 57
簁 71
簂 39
簃 24
簄 0
簅 46
簆 79
簇 144
簇擁 31
簇擁而上 0
簇擁著 11
簇新 2
簇生 7
簈 0
簉 20
簊 52
簋 115
簌 5
簌地 2
簌地 2
簌簌 5
簍 22
簍子 2
簍筐 0
簎 41
簏 55
簐 61
簑 23
簑笠翁 3
簑衣 10
簒 0
簕 0
簘 0
簙 11
簚 0
簛 0
簜 16
簝 9
簞 66
簞食壺漿 3
簞食瓢飲 0
簟 20
簠 8
簠簋 0
簡 9307
簡介 51254
簡介性 0
簡任 185
簡便 449
簡便性 4
簡冊 4
簡分數 2
簡則 151
簡化 992
簡化法 0
簡史 910
簡單 5194
簡單再生 0
簡單再生產 0
簡單勞動 0
簡單化 36
簡單協作 0
簡單商品 0
簡單商品生產 0
簡單扼要 15
簡單明瞭 37
簡單機械 4
簡單生產 0
簡單說 34
簡單起來 0
簡國賢 1
簡圖 93
簡報 6965
簡報室 70
簡奧 0
簡字 14
簡字表 0
簡寫 36
簡寫法 1
簡慢 0
簡拔 1
簡括 2
簡捷 27
簡擢 0
簡明 376
簡明扼要 15
簡易 1895
簡易法 2
簡札 0
簡樸 59
簡歷 476
簡歷表 11
簡潔 201
簡牘 14
簡略 197
簡略法 0
簡直 549
簡直是 225
簡短 250
簡短扼要 3
簡稱 1693
簡稱為 56
簡章 1070
簡章備索 12
簡筆字 0
簡答題 1
簡策 1
簡簡單單 32
簡約 50
簡編 10
簡練 9
簡而言之 44
簡裝 0
簡要 546
簡言之 61
簡訊 1701
簡記 8
簡諧運動 6
簡譜 23
簡述 10
簡陋 140
簡餐 1
簡體 141
簡體中文 1
簡體字 106
簢 5
簢 5
簣 191
簤 0
簥 2
簦 14
簧 117
簧樂器 1
簧片 20
簧風琴 1
簨 25
簩 4
簪 152
簪 152
簪子 8
簪笏 1
簪花 6
簫 146
簫笙 0
簫管 2
簬 46
簭 8
簮 0
簰 6
簳 35
簳麵杖 1
簴 0
簴 0
簵 0
簶 0
簷 210
簷下 12
簷前 1
簷宇 0
簷柱 2
簷角 4
簷頭 1
簸 89
簸 89
簸動 0
簸弄 0
簸盪 0
簸箕 6
簹 108
簺 0
簻 41
簻 41
簼 18
簼 18
簽 22500
簽上 16
簽上來 0
簽上去 0
簽下 134
簽下來 2
簽下去 2
簽了 27
簽些 0
簽出 8
簽到 120
簽到簿 49
簽到處 3
簽印 0
簽名 1853
簽名簿 195
簽名運動 2
簽呈 18
簽唱會 0
簽在 3
簽好 2
簽妥 9
簽字 87
簽字筆 18
簽完 6
簽定 224
簽寫 7
簽帳 0
簽得 0
簽押 0
簽收 67
簽有 32
簽條 2
簽注 5
簽為 3
簽發 176
簽章 0
簽約 1008
簽約人 2
簽約國 23
簽署 1037
簽署 1037
簽署人 2
簽著 0
簽訂 941
簽証 139
簽證 1594
簽證費 26
簽賭 0
簽退 9
簽過 2
簾 231
簾子 20
簾帳 0
簾幕 24
簿 1140
簿子 14
簿本 7
簿本費 0
簿記 21
簿記員 1
籀 111
籀文 2
籀書 0
籀篆 0
籃 1233
籃兒 0
籃兒 0
籃協 10
籃協會 0
籃圈 0
籃壇 14
籃委會 0
籃子 93
籃板 99
籃板球 37
籃架 3
籃框 18
籃狀 0
籃球 1245
籃球員 17
籃球場 14
籃球場 147
籃球架 5
籃球賽 93
籃球隊 204
籃網 10
籃裝 0
籃賽 39
籃運 1
籃隊 72
籄 0
籅 92
籆 0
籇 80
籈 79
籉 256
籊 91
籋 0
籋 0
籌 905
籌備 955
籌備委員會 113
籌備會 173
籌備會議 112
籌備處 953
籌到 2
籌劃 294
籌募 116
籌募到 2
籌商 0
籌委會 56
籌安會 4
籌建 270
籌措 291
籌撥 5
籌款 72
籌略 0
籌畫 120
籌碼 369
籌策 3
籌算 2
籌組 398
籌設 574
籌謀 13
籌議 1
籌資 0
籌辦 243
籌辦處 0
籌集 40
籌集法 0
籍 2565
籍籍 0
籍貫 435
籍貫地 0
籐 442
籐子 0
籐架 0
籐條 1
籐椅 4
籑 0
籒 0
籓 13
籔 7
籔 7
籕 0
籖 0
籗 7
籗 7
籘 0
籙 14
籚 7
籛 7
籜 20
籝 0
籞 0
籟 319
籠 402
籠 402
籠中之鳥 2
籠中鳥 14
籠子 73
籠屜 0
籠籠統統 0
籠絡 14
籠絡人心 0
籠統 41
籠統性 2
籠罩 287
籠罩著 39
籠蓋 0
籠頭 2
籢 0
籣 115
籤 410
籤條 6
籤詩 5
籥 189
籦 16
籧 8
籧 8
籩 11
籩豆 0
籪 69
籫 19
籬 175
籬下 3
籬牆 1
籬竹 0
籬笆 42
籭 0
籮 27
籮筐 11
籯 8
籰 0
籱 0
籲 1285
籲求 3
籲請 208
米 4816
米倉 20
米價 11
米制 0
米勒 107
米南德 1
米奇 0
米妮 0
米字 10
米尺 0
米尼斯 2
米店 66
米果 33
米格 55
米格機 6
米殼 0
米湯 2
米漿 33
米爾 73
米珠薪桂 0
米突 2
米粉 192
米粉條 0
米粉肉 0
米粒 41
米粥 2
米糕 77
米糠 27
米糠油 3
米糧 14
米缸 4
米老鼠 95
米脂 1
米舖 0
米色 42
米蘇里 2
米蘇里州 0
米蘭 368
米蛀蟲 0
米蝦 23
米蟲 59
米行 30
米袋 2
米酒 65
米酒頭 1
米開朗基羅 0
米雪 17
米食 54
米飯 91
米高 7
米黃 15
米黃色 25
籴 0
籵 280
籶 0
籷 0
籸 6
籹 4
籺 3
类 0
籼 0
籽 182
籽實 3
籽棉 0
籿 1
粀 1
粁 3
粃 0
粄 38
粅 23
粆 0
粆 0
粇 0
粈 37
粉 3148
粉刷 149
粉刺 140
粉劑 45
粉圓 43
粉塵 84
粉墨 7
粉墨登場 13
粉墨登場 136
粉壁 1
粉嫩 18
粉屑 6
粉底 41
粉彩 188
粉拳 3
粉拳繡腿 0
粉撲 0
粉末 281
粉末冶金 60
粉末狀 9
粉板 1
粉條 12
粉沫 3
粉煤 6
粉牆 5
粉狀 56
粉白 7
粉盒 4
粉砂 2
粉碎 168
粉碎機 45
粉筆 60
粉筆灰 5
粉筆畫 0
粉粒 47
粉紅 253
粉紅色 231
粉絲 24
粉絲狀 0
粉腸 3
粉膏 1
粉臉 9
粉臘 1
粉臘筆 2
粉色 19
粉藍 10
粉蝶 28
粉身碎骨 9
粉霜 0
粉面 3
粉飾 31
粉飾太平 8
粉飾門面 0
粉餅 24
粉黛 7
粊 11
粋 0
粌 31
粍 25
粑 18
粒 1462
粒子 369
粒子學說 0
粒度 22
粒徑 106
粒狀 208
粒粒 22
粒粒皆辛苦 5
粒線體 13
粒選 0
粓 0
粔 64
粕 139
粖 5
粗 1330
粗中帶細 0
粗中有細 1
粗人 3
粗俗 55
粗具 4
粗具規模 6
粗制品 0
粗制濫造 0
粗加工 1
粗劣 13
粗劣品 0
粗厚 3
粗啞 2
粗坑 24
粗壯 62
粗大 56
粗工 8
粗布 12
粗布條 0
粗心 49
粗心大意 21
粗手粗腳 2
粗放 17
粗暴 66
粗枝大葉 7
粗毛 7
粗毛布 0
粗毛羊 0
粗活 7
粗淺 42
粗獷 39
粗率 1
粗略 84
粗眉大眼 1
粗知一二 0
粗短 11
粗笨 1
粗筋 2
粗米 2
粗粗 9
粗粗細細 0
粗糙 315
粗糙 315
粗糠 3
粗糧 1
粗糲 0
粗紗 8
粗紡 1
粗細 94
粗細兼揉 0
粗線 15
粗線條 5
粗纖維 11
粗聲 4
粗聲暴氣 0
粗聲粗氣 2
粗脂肪 12
粗腔橫調 0
粗茶淡飯 8
粗蛋白質 13
粗製 19
粗製品 1
粗製濫造 15
粗話 34
粗豪 1
粗質 6
粗通 2
粗通文墨 0
粗通文字 0
粗選 1
粗鄙 10
粗重 18
粗野 24
粗陋 8
粗飼料 1
粗體 1
粗魯 62
粗鹵 1
粗鹽 4
粘 622
粘 622
粘上 1
粘上去 1
粘住 4
粘土 79
粘液 30
粘液 30
粘菌 0
粘著 71
粘貼 56
粘貼 56
粘連 19
粙 0
粛 0
粜 0
粝 0
粞 5
粟 299
粟倉 10
粟米 4
粟粒狀 0
粟類 1
粟飯 0
粠 0
粡 2
粢 1
粢 1
粣 17
粤 0
粥 519
粥少僧多 0
粥飯 3
粥餳 0
粦 0
粧 899
粨 2
粮 0
粯 0
粰 0
粱 56
粱肉 0
粲 60
粲然 5
粲然一笑 0
粳 42
粳 42
粳稻 1
粴 21
粵 389
粵伶 0
粵式 1
粵式料理 1
粵歌 0
粵劇 0
粵江 1
粵江三角洲 0
粵江流域 0
粵漢鐵路 1
粵省 1
粵語 0
粵語片 0
粶 0
粷 0
粸 0
粹 266
粺 61
粻 85
粼 26
粼粼 30
粽 103
粽子 125
精 8838
精一 17
精光 26
精兵 133
精兵制 2
精兵政策 3
精到 8
精制 2
精力 294
精力充沛 38
精力旺盛 21
精力過人 4
精力過剩 1
精品 2425
精囊 14
精壯 13
精妙 38
精子 216
精密 2521
精密儀器 256
精密度 132
精密機器 5
精密機械 247
精實 580
精實案 13
精專 2
精巢 3
精工 136
精巧 152
精幹 5
精度 333
精彩 2172
精彩鏡頭 17
精得很 0
精微 49
精心 605
精心傑作 12
精心製作 54
精心製造 2
精心設計 185
精忠 43
精忠報國 2
精怪 4
精悍 6
精打細算 56
精採 1
精敏 5
精敏度 0
精於 36
精明 128
精明強幹 0
精明能幹 9
精梳 27
精氣 26
精氣神 22
精油 434
精液 1
精液 1
精深 37
精湛 174
精準 346
精煉 53
精煉爐 4
精熟 37
精疲力盡 11
精疲力竭 21
精益求精 102
精研 46
精確 717
精確度 111
精確性 20
精礦 0
精神 7031
精神世界 17
精神分析 114
精神分裂 91
精神分裂症 67
精神失常 8
精神奕奕 0
精神恍惚 11
精神感召 5
精神抖擻 14
精神教育 73
精神為之一振 4
精神狀態 48
精神生活 88
精神病 239
精神病學 27
精神病學家 0
精神病院 29
精神科 1010
精神飽滿 18
精算 154
精算師 31
精簡 500
精簡人事 22
精粹 84
精純 23
精紡 16
精細 266
精細管 0
精練 192
精緻 2416
精美 857
精美型 0
精義 57
精耕 7
精耕細作 1
精致 23
精舍 296
精良 129
精英 839
精華 1393
精華區 1269
精蟲 116
精衛 16
精裝 435
精裝本 55
精製 161
精製品 5
精要 53
精覈 0
精誠 1355
精誠團結 8
精誠所至 5
精讀 20
精辟 3
精通 208
精通於 1
精進 532
精進案 532
精選 3093
精選集 1300
精采 580
精采絕倫 11
精金 9
精銳 97
精銳部隊 8
精鍊 24
精闢 270
精雕細刻 0
精雕細琢 31
精靈 2097
精靈鬼怪 1
精飼料 0
精餾 1
精餾塔 2
精髓 128
精鹽 23
粿 160
糂 0
糃 0
糃 0
糄 0
糅 7
糅合 1
糆 0
糇 0
糈 18
糉 0
糉 0
糊 1
糊 444
糊住 0
糊刷 0
糊口 10
糊塗 180
糊塗一時 3
糊塗帳 5
糊塗蛋 0
糊塗蟲 2
糊弄 1
糊弄 1
糊弄局 1
糊成 5
糊牆 4
糊狀 20
糊粘 5
糊精 43
糊糊 11
糊紙 4
糊著 0
糊裡糊塗 6
糊里糊塗 14
糋 8
糌 6
糌粑 0
糍 0
糏 0
糐 30
糑 15
糒 221
糓 0
糔 96
糕 206
糕餅 265
糕餅店 11
糕餅類 12
糕點 82
糖 1813
糖份 36
糖分 33
糖包 2
糖尿 315
糖尿病 2130
糖廠 208
糖果 535
糖果店 3
糖果盒 1
糖果費 0
糖業 255
糖水 31
糖汁 40
糖漬 9
糖漿 83
糖瓜 0
糖稅 0
糖精 13
糖糕 0
糖膏 0
糖蘿蔔 1
糖蜜 24
糖衣 32
糖醋 45
糖醋排骨 7
糖類 40
糖食 8
糗 133
糗事 35
糗得 0
糙 40
糙 40
糙米 58
糚 0
糛 0
糜 86
糜爛 74
糜費 0
糝 5
糞 321
糞便 264
糞土 12
糞坑 62
糞堆 20
糞壤 1
糞池 13
糞污 0
糞箕 6
糞肥 0
糞車 0
糟 234
糟了 47
糟塌 2
糟得 2
糟得很 0
糟爛 0
糟的 41
糟粕 3
糟糕 167
糟糕了 10
糟糠 3
糟糠之妻 0
糟蹋 57
糟透了 10
糠 49
糠秕 1
糠花 0
糠蝦 1
糠醛 1
糡 0
糡 0
糢 9
糢糊 24
糣 0
糤 0
糥 0
糦 0
糦 0
糧 698
糧作 2
糧倉 8
糧季 0
糧店 0
糧涸 0
糧票 1
糧秣 0
糧站 5
糧艙 0
糧草 24
糧行 3
糧袋 0
糧食 545
糧食作物 25
糧食供應 3
糧食局 40
糧餉 4
糨 49
糩 0
糪 32
糫 0
糬 87
糮 139
糯 57
糯米 108
糯米糕 5
糰 82
糰子 0
糱 3
糲 11
糲飯 0
糳 0
糴 10
糴米 1
糵 0
糶 16
糷 231
糷 231
糸 117
糸 117
糺 0
糺 0
系 31092
系主任 1138
系出名門 5
系刊 105
系列 18960
系列化 11
系務 45
系學會 0
系所 0
系數 57
系族 0
系統 102055
系統分析 1904
系統分類 22
系統化 146
系統工程 196
系統性 145
系統發育 1
系統論 11
系花 44
系譜 45
系館 258
糼 0
糽 77
糾 308
糾 308
糾劾 0
糾合 4
糾察 32
糾察人員 0
糾察隊 69
糾彈 27
糾正 453
糾正法 0
糾眾 3
糾紛 990
糾結 80
糾纏 162
糾纏不清 22
糾舉 28
糾葛 114
糾集 6
糿 0
紀 4315
紀事 418
紀傳 5
紀傳體 3
紀元 76
紀元前 14
紀元後 2
紀實 294
紀年 65
紀律 272
紀律嚴明 3
紀律性 1
紀律處分 1
紀念 3028
紀念冊 130
紀念品 362
紀念堂 36
紀念大會 23
紀念性 55
紀念日 293
紀念會 29
紀念物 18
紀念碑 281
紀念祠 1
紀念章 60
紀念週 2
紀念館 351
紀政 29
紀曆 0
紀綱 0
紀行 166
紀要 560
紀錄 11580
紀錄下來 13
紀錄片 257
紁 1
紂 31
紂王 37
紃 4
約 16521
約一斤肉 0
約一約 2
約人 5
約克 132
約分 9
約分法 0
約同 7
約在 554
約好 74
約定 1049
約定俗成 20
約定成俗 7
約成 22
約摸 0
約數 15
約旦 172
約旦人 1
約旦河 17
約是 59
約會 633
約期 17
約束 315
約束力 45
約法 6
約法三章 11
約為 971
約瑟 0
約瑟夫 36
約略 110
約略是 2
約略為 2
約稿 5
約簡 0
約簡成 0
約翰 1019
約翰尼斯 1
約翰尼斯堡 64
約翰福音 136
約翰遜 10
約莫 56
約見 171
約計 37
約談 262
約請 5
紅 704
紅 7048
紅不讓 0
紅中 11
紅中白板 0
紅丹 2
紅了 108
紅人 41
紅光 104
紅光滿面 8
紅冠 35
紅利 326
紅到 8
紅包 275
紅十字 31
紅十字會 101
紅單子 1
紅四軍 0
紅土 43
紅場 9
紅場 9
紅塵 713
紅壤 2
紅外光 62
紅外線 774
紅外線燈 2
紅女 13
紅妝 7
紅姑 6
紅娘 138
紅字 37
紅學 31
紅寶石 125
紅小兵 4
紅岩 0
紅巖 0
紅帶 6
紅帶子 1
紅帽子 3
紅底 13
紅底白字 1
紅彈子 0
紅彤彤 0
紅得 229
紅得發紫 4
紅心 51
紅心靶 0
紅斑 245
紅旗 26
紅旗手 0
紅日 22
紅星 51
紅暈 21
紅曲 0
紅木 50
紅杉 1
紅杏出牆 11
紅松 7
紅果 8
紅柳 4
紅梅 38
紅棗 30
紅棗湯 1
紅椒 15
紅楓 16
紅極一時 11
紅樓 193
紅樓夢 303
紅樹 32
紅樹林 1068
紅檜 54
紅毛 186
紅毛城 60
紅汞 1
紅河 5
紅河三角洲 1
紅河口 0
紅泥 9
紅海 51
紅潤 31
紅潮 18
紅火 9
紅火蟻 1
紅熱 2
紅燈 191
紅燈戶 0
紅燐 0
紅燒 114
紅燒肉 3
紅片 1
紅牌 69
紅玉 12
紅珊瑚 13
紅班 9
紅瓦 27
紅男綠女 36
紅番 14
紅疹 58
紅痢 0
紅白 162
紅白分明 0
紅的 229
紅皮 9
紅皮書 26
紅盤 39
紅眼 51
紅磚 139
紅磷 2
紅種人 0
紅筆 18
紅管 0
紅粉 42
紅粉佳人 39
紅粉知己 19
紅糖 30
紅紅 49
紅紅的 50
紅紅地 49
紅紅綠綠 0
紅紙 19
紅細胞 4
紅綠 67
紅綠燈 106
紅線 99
紅腫 120
紅臉 14
紅船 0
紅色 2001
紅色中華 0
紅色政權 0
紅色素 6
紅花 158
紅苕 0
紅茶 458
紅莓 76
紅葉 156
紅葉少棒隊 4
紅葉村 26
紅葉樹 2
紅著 14
紅著眼睛 3
紅著臉 19
紅蓮燈 1
紅薯 3
紅藥水 4
紅蘿蔔 112
紅蛋 13
紅蜻蜓 1
紅蟲 11
紅血球 278
紅衛 0
紅衛兵 26
紅衣 43
紅衣主教 4
紅衣服 5
紅褐色 26
紅襪子 0
紅豆 216
紅豆冰 6
紅豆杉 33
紅豆湯 11
紅豆詞 4
紅起來 10
紅軍 33
紅透 14
紅透半邊天 10
紅通通 9
紅運 1
紅運當頭 1
紅遍 25
紅遍 25
紅過 2
紅邊 9
紅酒 181
紅鈴蟲 0
紅銅 9
紅銅礦 0
紅門 10
紅雞 32
紅雨 4
紅霉素 0
紅霞 21
紅鞋 6
紅鞋子 1
紅頂商人 0
紅領巾 0
紅頭 109
紅頭髮 4
紅顏 72
紅顏薄命 10
紅髮 23
紅髮女郎 1
紅鬍子 17
紅魚 4
紅鸞星動 3
紅鹿 6
紅麻 3
紅點 69
紅黴素 16
紅鼓 2
紅鼻子 10
紅龍 122
紆 5
紆子 0
紆尊降貴 3
紆迴 1
紇 8
紈 29
紈褲 2
紈褲子弟 200
紉 23
紊 135
紊亂 99
紊流 47
紊淆 0
紋 1895
紋光針 0
紋板 11
紋樣 12
紋溝 3
紋理 53
紋理分明 0
紋眉 37
紋絲不動 4
紋路 101
紋身 84
紋銀 3
紋風 1
紋風不動 3
紌 8
納 5032
納入 1300
納吉 31
納妾 7
納寵 0
納悶 108
納款 1
納涼 17
納稅 1448
納稅人 274
納稅者 4
納米比亞 8
納粹 124
納粹主義 8
納粹黨 13
納糧 2
納聘 2
納苗 0
納言 5
納諫 4
納豆 1
納貢 5
納賄 0
納賽爾 0
納采 0
紎 39
紏 12
紐 1206
紐國 25
紐帶 5
紐扣 3
紐扣兒 0
紐扣兒 0
紐澳 163
紐澳地區 29
紐約 2838
紐約人 14
紐約州 263
紐約市 110
紐約時報 321
紐約河 0
紐約派 1
紐芬堡 1
紐芬蘭 12
紐西蘭 1402
紐西蘭人 11
紐釦 1
紑 9
紒 1
紓 1715
紓困 0
紓憂 0
紓緩 0
紓解 336
純 4
純 4049
純一 28
純利 4
純化 206
純厚 3
純如 17
純小數 1
純屬 199
純屬巧合 17
純屬捏造 1
純屬虛構 39
純度 152
純律 0
純情 0
純愛 374
純收入 7
純文學 30
純林 10
純棉 65
純樸 186
純正 102
純毛 7
純淨 195
純潔 169
純潔性 0
純然 33
純熟 105
純理論 4
純白 61
純白色 13
純益 197
純益率 37
純直 0
純真 433
純研究工作 0
純種 99
純種牛 0
純種馬 0
純篤 0
純粹 481
純粹是 112
純純 143
純綿 3
純美 45
純良 6
純色 14
純誠 2
純酒 4
純酒精 22
純金 110
純陽 2
純革裝 0
純音 26
純風 0
純鹼 40
紕 57
紕 57
紕漏 22
紕繆 0
紖 8
紖 8
紗 2130
紗巾 2
紗布 102
紗布繃帶 0
紗帽 42
紗廠 16
紗櫥 4
紗燈 2
紗窗 50
紗筒 2
紗綻 0
紗線 97
紗罩 1
紗罩燈 0
紗羅 6
紗錠 1
紗門 19
紘 279
紙 10295
紙上 329
紙上作業 8
紙上談兵 23
紙作 10
紙包 31
紙包不住火 3
紙卷子 0
紙型 6
紙堆 0
紙夾 44
紙套 28
紙尿褲 1
紙屑 84
紙工 42
紙巾 70
紙帶 20
紙幣 56
紙張 805
紙彈 0
紙扇 0
紙捻 1
紙杯 29
紙板 110
紙板盒 0
紙框 0
紙條 107
紙業 485
紙業公司 39
紙漿 199
紙漿廠 8
紙煙 3
紙燈 0
紙片 22
紙版 27
紙版畫 0
紙牌 103
紙牌兒 0
紙牌兒 0
紙牌局 0
紙牌戲 0
紙狀 0
紙畫 4
紙盒 109
紙盒式 0
紙盒裝 3
紙盤 7
紙短情長 2
紙票 0
紙筆 76
紙筆墨硯 0
紙管 20
紙箱 151
紙箱裝 0
紙糊老虎 0
紙縫 0
紙老虎 5
紙背 2
紙船 3
紙色 7
紙花 17
紙菸 2
紙行 58
紙衣 2
紙袋 82
紙裝 4
紙貴洛陽 0
紙費 1
紙邊 4
紙醉金迷 10
紙鈔 1
紙錢 18
紙鎮 26
紙面 18
紙頭 3
紙鳶 9
級 18300
級距 360
級任 111
級任導師 31
級任老師 41
級別 143
級差地租 0
級數 196
級會 63
級格 1
級進雜交 0
級長 4
紛 686
紛亂 70
紛亂不寧 0
紛擾 71
紛歧 24
紛沓 3
紛濁 1
紛爭 229
紛紅駭綠 1
紛紅駭綠 1
紛紛 1791
紛紛揚揚 3
紛紛議論 2
紛紜 23
紛至沓來 10
紛論 0
紛起 4
紛雜 12
紛飛 59
紜 40
紝 6
紝 6
紞 51
紟 9
素 8968
素不相識 14
素交 2
素來 72
素士 0
素女經 1
素席 0
素性 20
素愛 2
素描 614
素描法 0
素數 42
素日 18
素昧平生 9
素有 149
素服 2
素未謀面 15
素材 503
素樸 21
素淨 10
素稱 6
素絲 1
素聞 7
素菜 57
素菜館 3
素行 16
素衣 2
素質 716
素車白馬 0
素車白馬 0
素雅 25
素顏 1
素願 2
素食 915
素食店 13
素食麵 6
素養 673
素餐 0
素麵 7
紡 1432
紡垂形 0
紡垂體 0
紡成 2
紡紗 136
紡紗工 0
紡紗廠 11
紡紗機 5
紡紗車 0
紡絲 38
紡線 5
紡織 2377
紡織公司 31
紡織品 676
紡織娘 2
紡織工業 220
紡織廠 92
紡織業 521
紡織機 23
紡車 14
紡錘 16
紡錘形 3
索 37
索 3788
索債 5
索債過高 0
索價 35
索價過高 0
索取 1994
索取到 3
索命 14
索尼 19
索居 3
索引 23752
索性 113
索性 113
索橋 4
索氏體 0
索然 10
索然無味 4
索爾茲伯 0
索賠 71
索道 35
索還 3
索還債款 0
索閱 66
索非亞 0
索馬利亞 11
索馬里 1
紣 0
紤 0
紥 0
紥 0
紨 47
紩 40
紪 0
紫 4069
紫丁花 3
紫丁香 11
紫外光 96
紫外光線 0
紫外線 618
紫微斗數 219
紫斑 51
紫毫 1
紫水晶 24
紫河車 14
紫灰 3
紫禁城 37
紫穗槐 0
紫竹 34
紫籐 6
紫紅 58
紫紅色 52
紫羅蘭 48
紫色 501
紫色花 0
紫花 55
紫荊 60
紫菜 36
紫菜湯 3
紫葡萄 2
紫葳類 0
紫薇 73
紫薑 0
紫藍 42
紫藥水 1
紫蘇 1
紫蘭 1
紫金 34
紫金山 7
紫金花 1
紫銅 6
紫雲 76
紫雲英 1
紫黑 4
紫黑色 5
紫黯 0
紬 19
紭 0
紮 266
紮 266
紮上 4
紮上來 0
紮上去 0
紮下 16
紮下來 0
紮下去 0
紮伊爾 0
紮好 4
紮實 194
紮寨 1
紮帶子 0
紮成 14
紮根 186
紮營 71
紮緊 2
紮起 7
紮起來 1
累 1122
累 400
累 2332
累世 22
累了 347
累事 0
累人 35
累代 6
累倒 7
累加 47
累加器 4
累加法 0
累卵 0
累及 18
累及他人 0
累及無辜 0
累增 7
累增法 0
累壞 30
累日 0
累月 7
累月經年 0
累次 8
累次三番 0
累歲 0
累死 54
累牘連篇 0
累犯 23
累病 1
累積 1980
累積下來 19
累積到 42
累積折耗 0
累積盈虧 4
累累 49
累累屢屢 0
累計 1417
累計法 0
累贅 28
累進 161
累進法 0
累進稅 3
細 4864
細事 1
細作 9
細分 189
細則 656
細動脈 0
細動脈 0
細問 4
細嚼 3
細嚼 3
細嚼慢嚥 15
細嚼慢嚥 15
細大不捐 0
細如 18
細姨 10
細嫩 24
細嫩皮膚 0
細字 6
細字筆 0
細密 81
細察 12
細小 127
細巧 10
細布 7
細弱 2
細微 156
細心 501
細怨 0
細想 26
細故 21
細明體 1
細木工 0
細枝 5
細查 40
細毛 13
細毛羊 0
細民 1
細水長流 24
細沙 20
細活 5
細流 4
細濛濛 0
細瓷 2
細皮 6
細皮嫩肉 2
細目 136
細看 156
細看之下 4
細砂 18
細砂紙 3
細碎 31
細磚 1
細究 15
細管 9
細節 878
細節性 2
細紋 27
細紗 19
細索 2
細細 228
細細的 64
細細地 64
細細長長 2
細絲 14
細網 6
細綾 0
細線 62
細緻 328
細縫 10
細繩 13
細聲 7
細胞 3783
細胞分裂 50
細胞器 0
細胞壁 51
細胞學 56
細胞學說 1
細胞層 1
細胞核 63
細胞液 3
細胞液 3
細胞狀 1
細胞組織 21
細胞膜 74
細胞質 45
細胞體 3
細腰 8
細膩 331
細致 6
細菌 1259
細菌學 40
細菌戰 7
細行 6
細行 6
細語 111
細說 333
細談 3
細論 3
細質 0
細軟 10
細述 30
細部 495
細部分解 0
細量 0
細長 109
細雨 97
細雨濛濛 0
細雨濛濛憶當年 1
細雨綿綿 6
細頸 0
紱 29
紲 106
紳 279
紳商 1
紳士 138
紳士派 0
紳宦 0
紴 0
紵 12
紶 34
紶 34
紷 0
紸 26
紹 6617
紹繼 0
紹興 103
紹興市 2
紹興酒 35
紹述 1
紹酒 0
紺 75
紻 4
紼 18
紽 7
紾 50
紿 42
絀 166
絁 11
終 1993
終了 164
終其一生 65
終其一生 65
終南山 19
終場 20
終場 204
終夕 0
終夜 12
終夜未眠 0
終始 4
終宵 1
終審 26
終將 201
終局 37
終年 121
終必 26
終必在 1
終必是 0
終必會 0
終成泡影 1
終於 4570
終日 143
終極 512
終極目標 46
終止 766
終歲 3
終歸 23
終歸一句 0
終歸一句 0
終歸於 2
終歸是 4
終生 440
終究 383
終究是 69
終究會 41
終站 25
終端 242
終端機 491
終端設備 102
終篇 10
終結 355
終結句 0
終結者 231
終老 15
終至 74
終身 1861
終身不忘 1
終身大事 29
終身職 12
終須 14
終養 1
終養 1
終點 262
終點港 1
終點站 43
終點線 3
絃 198
絃外之音 3
絃柱 0
絃樂 47
絃樂器 5
絃歌不輟 3
絃聲 2
組 36891
組件 583
組別 524
組合 5600
組合成 79
組合法 2
組合而成 113
組員 554
組團 0
組態 1
組成 3466
組成部分 18
組曲 165
組織 19347
組織上 41
組織學 86
組織家 1
組織形式 9
組織性 29
組織法 274
組織液 8
組織液 8
組織生活 2
組織胺 101
組織起來 18
組織路線 0
組織部 20
組織領導 9
組胺 2
組裝 0
組裡 12
組訓 95
組譯 0
組譯器 0
組起 1
組起來 3
組過 3
組長 2072
組閣 58
組隊 61
組頭 29
組黨 88
絅 16
絆 105
絆住 9
絆倒 32
絆手絆腳 4
絆腳 0
絆腳石 31
絇 9
絈 0
絉 0
絊 105
経 0
絍 0
絍 0
絎 3
絏 7
結 1118
結 4011
結下 69
結了 69
結交 191
結交到 7
結仇 3
結伴 97
結伴同行 8
結伴同遊 1
結冰 57
結出 47
結匯 118
結匯證 0
結合 6385
結合實際 3
結合律 3
結合水 1
結合能 5
結合點 1
結在 21
結塊 22
結夏安居 1
結夥 9
結夥搶劫 9
結婚 2487
結婚儀式 5
結婚典禮 17
結婚紀念 4
結婚紀念日 13
結婚證書 32
結子 10
結子 10
結存 35
結存額 0
結實 121
結實 121
結實 121
結尾 252
結局 545
結局是 32
結巴 61
結巴 61
結帳 17588
結廬 5
結彩 1
結得 13
結得 131
結怨 14
結成 84
結扎 2
結拜 25
結拜兄弟 5
結掉 2
結撰 0
結斷 0
結晶 571
結晶器 4
結晶學 18
結晶度 4
結晶性 36
結晶水 10
結晶法 2
結晶體 22
結束 5137
結束語 2
結果 14931
結果實 3
結果是 364
結果枝 2
結核 100
結核桿菌 18
結核病 171
結核菌 54
結核菌素 18
結案 235
結業 604
結業典禮 36
結業式 14
結業證書 119
結構 7950
結構學 55
結構式 66
結構鋼 2
結標 1
結清 26
結為 53
結為一體 8
結為一體 8
結為夫婦 3
結焦 5
結疤 17
結的 131
結盟 378
結盟國 0
結石 331
結社 60
結社自由 7
結秦晉之好 0
結稿 200
結算 683
結算法 0
結節 133
結紮 247
結紮手術 27
結紮術 3
結結 2
結結 2
結結 2
結結 2
結結實實 13
結結巴巴 15
結網 53
結綵 2
結締 0
結締組織 69
結緣 183
結縭 4
結繩 6
結繩紀事 6
結繩記事 1
結義 29
結腸 178
結腸炎 15
結膜 59
結膜炎 106
結舌 6
結草 4
結草蟲 0
結親 4
結言 1
結訓 157
結語 700
結論 2009
結論是 91
結識 106
結賬 8
結起 4
結起來 2
結轉額 0
結鄰 0
結隊 9
結集 77
結集在 1
結霜 7
結餘 53
結髮 12
結黨 17
結黨營私 3
絑 3
絒 1
絓 5
絔 16
絕 2356
絕不 1144
絕不在 8
絕不放棄 23
絕不是 251
絕不會 292
絕不能 193
絕世 72
絕世佳人 0
絕世超倫 0
絕交 9
絕代 60
絕代佳人 4
絕代美女 0
絕佳 396
絕俗 5
絕倫 51
絕口 11
絕口不提 70
絕句 21
絕命 156
絕命書 1
絕品 3
絕唱 4
絕嗣 2
絕地 248
絕地武士 248
絕地重生 0
絕域 0
絕境 17
絕壁 19
絕大多數 403
絕大部分 131
絕妙 66
絕妙好辭 2
絕子絕孫 2
絕學 56
絕對 4205
絕對值 29
絕對優勢 25
絕對化 5
絕對多數 46
絕對性 16
絕對數 0
絕對溫度 19
絕對溫標 0
絕對濕度 3
絕對觀念 1
絕對誤差 0
絕對零度 12
絕後 14
絕手 0
絕才 1
絕技 101
絕招 163
絕早 2
絕景 3
絕望 222
絕望已極 0
絕活 33
絕滅 21
絕無 352
絕無僅有 51
絕無此事 5
絕熱 33
絕熱過程 0
絕版 142
絕版書 10
絕症 36
絕種 64
絕筆 12
絕糧 4
絕緣 473
絕緣台 0
絕緣材料 28
絕緣物 4
絕緣線 0
絕緣體 37
絕聖棄智 0
絕育 17
絕育法 0
絕色 2250
絕藝 11
絕處 2
絕處逢生 19
絕跡 53
絕路 16
絕配 25
絕非 370
絕非是 16
絕響 38
絕頂 33
絕頂聰明 9
絕食 43
絕食抗議 18
絖 19
絗 0
絘 0
絙 0
絚 0
絚 0
絛 18
絛蟲 26
絜 38
絜 38
絝 0
絞 996
絞上 0
絞亂 0
絞刀 1
絞刑 3
絞動 1
絞扭 0
絞斷 0
絞架 1
絞殺 2
絞決 0
絞痛 40
絞盡 2
絞盡腦汁 60
絞盤 25
絞碎 6
絞索 4
絞縊 0
絞肉 25
絞肉機 9
絞臉 0
絞車 4
絞首 0
絟 9
絠 0
絡 81
絡 8154
絡合 1
絡合物 5
絡子 1
絡線 2
絡繹 10
絡繹不絕 54
絡繹於途 4
絡腮鬍子 1
絡頭 1
絢 162
絢爛 79
絢爛歸於平淡 6
絢麗 97
絢麗多彩 6
絣 1
絣 1
絤 0
絥 0
給 3677
給 36776
給予 3464
給人 639
給人以 9
給付 2064
給以 33
給假 1
給假 19
給水 221
給水器 2
給水站 1
給水管 8
給水箱 0
給獎 49
給糧 1
給與 428
給足 10
給錢 31
絧 8
絨 615
絨布 83
絨毛 182
絨毯 1
絨氈 0
絨絲 1
絨線 4
絨線衫 0
絨繩 1
絨衣 0
絩 4
絪 8
絫 2
絬 0
絭 9
絮 489
絮凝 1
絮叨 1
絮叨叨 0
絮球 0
絮絮 7
絮絮不休 4
絮聒 2
絮語 597
絯 11
絯 11
絰 170
統 8719
統一 10942
統一化 18
統一天下 10
統一性 18
統一戰線 7
統一教 7
統一獅 103
統一發票 1567
統一計劃 0
統一體 8
統一黨 2
統共 12
統制 44
統合 488
統合力 0
統合性 5
統帥 74
統帥部 2
統帶 2
統御 46
統戰 106
統掌 4
統攝 13
統治 758
統治權 25
統治者 156
統治階級 29
統獨 0
統率 18
統理 8
統稱 87
統管 29
統籌 547
統籌兼顧 3
統籌規劃 48
統籌辦理 31
統統 120
統統 920
統舖 0
統艙 0
統計 13402
統計圖 246
統計圖表 108
統計學 532
統計數字 130
統計表 1153
統計起來 1
統購 14
統購統銷 0
統轄 30
統銷 4
統領 174
統馭 5
絲 3725
絲光 106
絲布 2
絲帕 1
絲帶 84
絲弦 1
絲掛子 0
絲束 2
絲棉 15
絲毫 327
絲毫不 78
絲毫不差 4
絲毫不爽 4
絲毫無損 2
絲狀 52
絲瓜 113
絲瓜布 1
絲竹 50
絲素 0
絲結 3
絲絨 34
絲絨布 0
絲絲 125
絲絲入扣 3
絲絹 8
絲綢 56
絲綢之路 51
絲網 10
絲線 43
絲織 33
絲織品 14
絲織廠 6
絲織物 11
絲繪 0
絲膠 1
絲蘿 0
絲蟲 83
絲蠅 0
絲襪 90
絲質 22
絲路 617
絲路之旅 18
絳 1156
絳紫 1
絳紫色 0
絴 0
絵 0
絶 0
絸 0
絹 210
絹布版 0
絹帛 6
絹扇 0
絹畫 1
絹素 0
絹絲 44
絹絲品 0
絹綢 0
絹花 0
絺 9
絻 6
絼 20
絽 160
絾 0
絿 7
綀 15
綁 409
綁上 15
綁上來 0
綁上去 0
綁下 0
綁下來 0
綁下去 0
綁住 41
綁匪 56
綁回 0
綁回來 0
綁回去 0
綁帶 4
綁成 5
綁架 321
綁樁 0
綁標 0
綁牢 1
綁票 80
綁紮 1
綁緊 10
綁縛 4
綁腳 7
綁腿 9
綁著 35
綁赴 1
綁起來 9
綂 0
綃 12
綄 4
綅 11
綆 7
綆短汲深 0
綇 0
綇 0
綈 2
綈 2
綉 0
綊 0
綋 0
綌 21
綍 3
綎 1
綏 175
綏遠 15
綏遠省 1
綏靖 7
綐 0
綑 34
綑上 0
綑了 0
綑住 0
綑來綑去 0
綑做 0
綑到 0
綑吊 0
綑在 1
綑好 0
綑得 0
綑成 2
綑成一堆 0
綑紮 2
綑綁 48
綑緊 0
綑縛 2
綑著 1
綑起 1
綑起來 0
綒 4
經 32383
經不起 50
經世 28
經世濟民 3
經世致用 2
經久 12
經久不衰 1
經久耐用 10
經互會 0
經傳 15
經典 1789
經典之作 68
經典作 6
經典作家 0
經卷 24
經受 31
經史 17
經史子集 6
經合會 0
經售 4
經售處 0
經商 140
經商在外 0
經國 297
經委 12
經學 116
經密 7
經師 26
經常 3691
經常性 175
經年 61
經年累月 40
經度 39
經建 537
經建會 809
經得起 58
經心 8
經意 16
經手 51
經手人 9
經援 0
經文 582
經曲 5
經書 85
經期 67
經歷 3821
經歷過 140
經濟 23114
經濟一體 1
經濟一體化 13
經濟主義 1
經濟人 18
經濟作物 32
經濟侵略 4
經濟制度 32
經濟化 4
經濟危機 168
經濟基礎 30
經濟大國 25
經濟學 1543
經濟學家 125
經濟學者 77
經濟專科 1
經濟專科學校 1
經濟建設 201
經濟建設委員會 137
經濟恐慌 8
經濟成分 0
經濟技術 0
經濟技術指標 0
經濟日報 270
經濟林 3
經濟核算 0
經濟業務 16
經濟槓桿 0
經濟活動 198
經濟活動分析 0
經濟犯 2
經濟界 2
經濟管理 25
經濟範疇 0
經濟系 424
經濟計劃 13
經濟部 5565
經濟部長 140
經營 15156
經營不善 79
經營者 856
經理 3012
經理人 696
經理級 15
經由 6471
經略 19
經痛 57
經穴學 0
經管 99
經管人 1
經籍 10
經紀 943
經紀人 433
經紀商 152
經紗 56
經絡 180
經絡學 5
經維 0
經綸 134
經線 19
經緯 180
經緯儀 32
經緯度 62
經緯網 0
經義 20
經脈 80
經脈 80
經血 51
經術 3
經解 11
經訓 5
經論 45
經費 7353
經費不足 73
經費有限 44
經貿 1916
經貿關係 225
經軸 31
經輪 13
經辦 157
經過 7319
經部 245
經銷 3311
經銷商 1656
經銷店 9
經銷站 4
經銷網 23
經銷處 21
經銷部 2
經驗 11099
經驗不足 43
經驗主義 13
經驗之談 5
經驗值 500
經驗到 33
經驗老到 6
經驗談 489
綔 6
綕 0
綖 12
綗 0
綘 0
綘 0
継 0
綜 1486
綜 1486
綜上所述 38
綜合 9365
綜合平衡 0
綜合性 364
綜合所得 84
綜合所得稅 518
綜合法 2
綜合藝術 19
綜合醫院 432
綜合防治 18
綜合雜誌 0
綜合體 20
綜括 4
綜括起來 0
綜攬 5
綜效 0
綜核 5
綜藝 382
綜覽 205
綜觀 224
綜述 80
綝 19
綞 9
綟 87
綠 5241
綠到 2
綠化 691
綠化運動 1
綠區 1
綠十字 10
綠卡 80
綠地 315
綠地區 0
綠寶石 22
綠島 820
綠帶 62
綠帽 23
綠帽子 4
綠得 80
綠意 85
綠意盎然 69
綠林 25
綠林軍 3
綠水 135
綠油油 21
綠油精 7
綠洲 99
綠燈 92
綠燈戶 1
綠營 5
綠珠篇 0
綠班 0
綠球藻 2
綠的 81
綠竹 12
綠筆 4
綠綠 39
綠綠的 25
綠綠地 25
綠肥 15
綠肥作物 9
綠色 2505
綠色植物 26
綠芽 12
綠苔 10
綠茵 16
綠茶 285
綠茸茸 1
綠草 48
綠草如茵 28
綠葉 84
綠葉成蔭 1
綠蔭 46
綠藻 109
綠蠵龜 0
綠衣 228
綠衣人 2
綠衣使者 1
綠褐色 2
綠豆 165
綠豆沙 22
綠豆湯 17
綠豆稀飯 0
綠豆粥 3
綠豆糕 11
綠豆芽 12
綠野 263
綠野仙蹤 40
綠黨 842
綠黴素 2
綡 22
綢 154
綢子 0
綢密 1
綢布 15
綢帶 0
綢緞 132
綢緞莊 0
綣 29
綤 0
綥 0
綦 45
綧 68
綨 0
綩 21
綪 28
綫 0
綬 146
綬帶 13
維 6267
維中 4
維也納 491
維也納盆地 0
維他命 715
維他命丸 7
維他命源 0
維修 2915
維修費 42
維吉尼亞 147
維吾爾 59
維吾爾人 0
維吾爾文 1
維吾爾族 23
維吾爾語 1
維和部隊 0
維基 600
維基百科 600
維基百科人 500
維基人 600
維基尼亞 0
維士比 8267
維多利亞 270
維妙維肖 14
維幾尼亞 0
維幾尼亞州 0
維恭維謹 0
維持 4933
維持下去 20
維持在 344
維持現狀 138
維持秩序 27
維持著 44
維文 14
維斯杜拉 0
維斯杜拉河 0
維新 421
維新運動 3
維琴尼亞 8
維琴尼亞州 1
維瓦第 0
維生 114
維生素 503
維生素缺乏症 1
維管束 38
維納斯 59
維綸 6
維繫 250
維繫人心 1
維艱 18
維護 39522
維辛斯基 0
維那 11
維那斯 8
綮 16
綮 16
綯 114
綰 28
綰轂 0
綰髮 1
綰髻 0
綱 5000
綱常 2
綱目 58
綱紀 7
綱紀廢弛 0
綱維 1
綱舉目張 5
綱要 1328
綱領 353
網 3152
網住 50
網兒 0
網兒 0
網卡 0
網友 0
網咖 0
網址 0
網域 0
網子 29
網帳 1
網拍 1
網架 12
網格 97
網狀 113
網狀結構 9
網狀脈 0
網狀脈 0
網球 838
網球員 0
網球場 16
網球場 164
網球拍 43
網球裝 0
網球賽 30
網球鞋 5
網目 28
網目線 0
網眼 23
網眼狀 0
網站 0
網笞 0
網管 0
網箱 1
網籃 2
網絡 500
網罟 3
網羅 277
網羅人才 8
網羅人材 0
網膜 122
網膜炎 6
網芳 0
網裡 23
網誌 1
網路 107919
網路卡 0
網路咖啡 0
網路系統 1406
網開一面 7
網際網路 0
網頁 0
網魚 2
網點 133
綳 0
綳 0
綳 0
綴 557
綴出 3
綴合 1
綴字 0
綴補 1
綴連 1
綴點 0
綵 65
綵排 5
綵樓 1
綵牌樓 0
綵球 3
綵綢 0
綵線 0
綵衣 1
綵衣娛親 2
綶 0
綷 63
綷菜 0
綸 323
綸 323
綸巾 4
綹 10
綺 823
綺夢 13
綺年 3
綺年玉貌 0
綺思 2
綺情 10
綺想 21
綺燦 0
綺窗 3
綺衣 0
綺語 6
綺豔 0
綺貌 0
綺雲 6
綺麗 89
綻 307
綻放 179
綻破 0
綻裂 0
綻開 25
綼 74
綽 168
綽約 9
綽約多姿 1
綽綽有餘 42
綽號 273
綾 397
綾子 24
綾羅綢緞 2
綿 281
綿亙 12
綿力 1
綿子 2
綿密 58
綿延 145
綿延不盡 2
綿弱 0
綿絮 5
綿綢 0
綿綿 116
綿綿不斷 0
綿綿不絕 7
綿綿地 3
綿綿地 3
綿綿細語 2
綿綿細雨 11
綿織 6
綿羊 146
綿薄之力 5
綿遠 2
綿邈 0
綿長 32
綿陽 15
緀 34
緀 34
緁 19
緂 52
緃 0
緄 22
緄邊 0
緅 39
緆 38
緇 45
緇黃 0
緈 0
緉 17
緊 7775
緊了 11
緊促 2
緊密 495
緊密度 2
緊密性 3
緊張 1550
緊張感 9
緊張起來 15
緊得 0
緊急 3180
緊急任務 4
緊急制動 0
緊急制動閥 0
緊急命令 23
緊急啟事 0
緊急情況 33
緊急措施 31
緊急會議 17
緊急狀況 62
緊急處分 7
緊急警報 1
緊急集合 2
緊挨 6
緊接 57
緊接在 17
緊接著 201
緊握 91
緊握住 4
緊握拳頭 2
緊握著 26
緊湊 116
緊箍咒 9
緊緊 290
緊緊的 98
緊緊地 97
緊縛 4
緊縮 425
緊縮在 2
緊縮著 0
緊繃 200
緊繃繃 0
緊繃著 4
緊著 2
緊要 34
緊要關頭 35
緊貼 27
緊貼住 1
緊貼在 16
緊貼著 12
緊跟 61
緊身 47
緊身衣 10
緊身褲 8
緊迫 141
緊迫感 5
緊迫逼人 1
緊迫釘人 4
緊追不捨 10
緊逼 2
緊逼著 0
緊那羅 1
緊鄰 99
緊鄰在 2
緊鄰著 4
緊鑼密鼓 38
緊閉 68
緊靠 20
緊靠在 4
緊靠著 7
緋 802
緋紅 13
緋紅色 3
緋聞 619
緌 117
緍 0
緎 14
総 0
緑 0
緑 0
緒 814
緒言 78
緒論 181
緖 0
緗 9
緘 196
緘口 0
緘密 0
緘封 4
緘默 32
緙 24
緙絲 3
線 15437
線上 19481
線上作業 16
線圈 292
線圈圖 0
線型 147
線報 0
線外 48
線形 84
線性 830
線性代數 158
線性元件 3
線性方程 1
線性方程組 5
線性規劃 37
線性變換 3
線材 226
線條 495
線條圖 1
線段 92
線民 25
線狀 28
線管 17
線粉 0
線粒體 0
線索 335
線縷 0
線繩 6
線膨脹 2
線蟲 5
線蟲病 7
線裝 12
線裝書 7
線路 1211
線路圖 70
線軸 7
線速度 5
線道 45
線長 33
線間 23
線頭 17
線頭兒 0
線頭兒 0
線香 0
緛 20
緜 0
緝 513
緝凶 0
緝拿 8
緝捕 95
緝捕歸案 2
緝毒 0
緝獲 20
緝盜 1
緝私 89
緝私船 0
緝辦 0
緞 105
緞子 15
緞帶 78
緞帶花 2
緞木 0
緞紋 5
緞補服 0
緟 7
締 188
締交 0
締構 0
締盟 47
締約 148
締約國 185
締結 168
締造 281
締造者 4
緡 32
緢 0
緣 59465
緣份 208
緣分 42
緣分已盡 0
緣因 5
緣故 382
緣木求魚 16
緣海 2
緣淺 12
緣濃情深 0
緣由 181
緣薄 1
緣覺 1
緣起 3078
緣起緣滅 39
緤 0
緤 0
緥 0
緦 16
緦麻服 0
緧 10
編 3979
編上 13
編上來 0
編上去 0
編下 2
編下來 0
編下去 1
編了 27
編些 1
編作 27
編來編去 0
編修 287
編個 9
編入 94
編出 28
編出來 1
編列 1209
編列出 0
編到 6
編制 1357
編劇 291
編印 733
編在 15
編好 13
編定 365
編審 196
編審委員 20
編寫 617
編寫出 4
編寫成 6
編導 70
編年 28
編年史 13
編年體 5
編得 92
編成 171
編戲 1
編戶 1
編按 0
編排 576
編排在 0
編撰 265
編曲 250
編書 10
編校 8
編構 0
編構出 0
編檔 2
編次 10
編歌 1
編派 8
編為 54
編物 12
編班 237
編班法 0
編的 92
編目 1194
編目條例 0
編目者 2
編目草片 0
編目表格 0
編目記錄 36
編目記錄輸入 0
編碼 1446
編碼器 57
編算 12
編算出 0
編篡 18
編篡者 0
編組 387
編結 14
編練 15
編織 555
編織品 19
編織成 12
編纂 219
編纂家 0
編者 304
編者按 40
編者的話 164
編舞 0
編著 701
編號 20353
編號欄 6
編號款目 0
編號版本 1
編製 774
編製成 5
編訂 134
編訂者 1
編譯 1084
編譯人員 2
編譯器 0
編譯館 242
編貝 2
編起 2
編起來 0
編輯 9066
編輯器 9066
編輯室 385
編輯部 799
編述 5
編造 78
編造出 2
編過 5
編遣 0
編選 51
編選出 0
編錯 1
編鐘 0
編隊 45
編髮 2
緩 994
緩不濟急 26
緩了 14
緩兵 0
緩兵之計 9
緩刑 60
緩召 39
緩和 354
緩和劑 4
緩坡 18
緩延 0
緩役 0
緩徵 65
緩急 13
緩慢 531
緩期 13
緩步 127
緩步而行 2
緩氣 0
緩沖 3
緩沖器 0
緩沖溶液 0
緩沖溶液 0
緩緩 645
緩行 5
緩衝 194
緩衝作用 7
緩衝劑 13
緩衝區 141
緩衝器 53
緩衝國 3
緩衝地 0
緩衝地區 0
緩衝地帶 18
緩衝式 3
緩衝期 37
緩衝液 22
緩衝液 22
緩衝溶液 10
緩衝溶液 10
緩衝體 0
緩解 175
緩課 0
緩議 12
緪 35
緫 0
緬 178
緬共 1
緬北 4
緬因州 12
緬懷 82
緬甸 272
緬甸人 2
緬西 0
緭 0
緮 11
緯 1670
緯密 2
緯度 131
緯紗 48
緯線 16
緰 4
緱 14
緲 35
緳 16
練 2151
練下 1
練下來 0
練下去 6
練了 38
練些 1
練來練去 0
練個 7
練兵 11
練兵場 1
練兵場 1
練出 25
練到 43
練功 367
練功夫 3
練唱 38
練好 23
練字 2
練寫 3
練得 34
練成 68
練拳 18
練操 0
練歌 3
練武 94
練武功 5
練武術 1
練球 107
練琴 20
練練 36
練練看 0
練習 4555
練習手冊 2
練習曲 29
練習本 9
練習簿 6
練習題 23
練起 12
練起來 3
練跑 0
練跳 3
練過 29
練過來 0
練過去 0
練達 24
練隊 0
緵 0
緶 11
緷 9
緸 0
緹 266
緹縈 2
緺 7
緻 570
緻密 44
緼 0
緼 0
緽 0
緽 0
緾 0
緿 0
縀 0
縁 0
縁 0
縂 0
縃 3
縄 0
縈 171
縈懷 2
縈紆 0
縈繞 30
縈繞在 7
縈身 0
縈迴 2
縉 71
縉紳 0
縊 12
縊死 1
縋 15
縌 32
縍 5
縎 18
縏 1
縐 140
縐布 1
縐紋 10
縐紗 0
縐紬 1
縐絲 0
縐絺 1
縐綢 1
縐縮 0
縐褶 6
縑 1476
縑帛 5
縒 21
縓 23
縔 10
縕 22
縕 22
縕袍 1
縕袍 1
縖 10
縗 21
縗 21
縗墨 0
縗墨 0
縘 0
縘 0
縙 0
縙 0
縙 0
縚 46
縛 289
縛上 1
縛住 3
縛成 0
縛緊 2
縛起來 0
縛雞之力 2
縜 28
縝 131
縝密 71
縝密性 0
縞 128
縞素 3
縞衣 2
縟 27
縟節 0
縟麗 0
縠 33
縡 15
縢 30
縣 21
縣 7921
縣令 14
縣名 15
縣城 27
縣境 62
縣外 29
縣太爺 2
縣委 12
縣委會 3
縣學 222
縣官 7
縣尉 6
縣差 0
縣市 6830
縣市政府 485
縣府 1000
縣志 52
縣政 342
縣政府 458
縣民 1291
縣治 36
縣獄 0
縣界 18
縣監獄 0
縣知事 1
縣立 4269
縣老爺 3
縣衙 1
縣衙門 1
縣裡 8
縣誌 5
縣議員 191
縣議會 113
縣轄市 113
縣運會 1
縣長 1833
縤 5
縥 6
縦 0
縦 0
縦 0
縧 0
縧 0
縩 13
縪 35
縪 35
縫 742
縫 742
縫上 5
縫上來 0
縫上去 0
縫下 0
縫下來 0
縫下去 0
縫了 12
縫住 0
縫來縫去 0
縫個 0
縫做 1
縫兒 2
縫兒 2
縫到 1
縫反 0
縫口 4
縫合 123
縫合線 14
縫合處 3
縫在 5
縫好 3
縫工 11
縫得 2
縫成 2
縫死 0
縫法 2
縫紉 114
縫紉機 116
縫綴 0
縫線 56
縫縫 21
縫縫補補 0
縫衣服 1
縫衣機 122
縫補 13
縫裡 9
縫製 98
縫製成 1
縫起 4
縫起來 6
縫針 11
縫釦子 1
縫隙 58
縫點 0
縬 0
縭 54
縮 145
縮 1459
縮下 0
縮下來 0
縮下去 0
縮到 38
縮印 24
縮印本 3
縮合 87
縮回 21
縮回來 3
縮回去 2
縮圖 55
縮孔 0
縮寫 254
縮寫字母 1
縮寫成 2
縮寫本 5
縮小 752
縮小到 21
縮小成 13
縮尺 11
縮影 214
縮影本 0
縮影片 5
縮慄 0
縮成 18
縮成一團 7
縮手 35
縮手縮腳 1
縮放 0
縮放自如 0
縮水 208
縮水率 1
縮減 426
縮減到 5
縮減成 3
縮短 974
縮短到 7
縮短成 0
縮簡 0
縮緊 9
縮縮 2
縮縮頭 0
縮缸 1
縮聚 7
縮聚反應 1
縮衣節食 8
縮起 3
縮起來 3
縮進 11
縮頭 6
縮頭烏龜 9
縮頭縮尾 1
縮頭縮腦 0
縮頸 3
縯 13
縰 6
縱 913
縱 913
縱令 13
縱使 353
縱列 8
縱剖面 2
縱向 94
縱坐標 0
縱容 124
縱座標 6
縱恣 0
縱情 28
縱情歌唱 0
縱情聲色 5
縱慾 18
縱慾主義 0
縱慾無度 0
縱斷面 7
縱有 52
縱橫 557
縱橫交錯 17
縱橫天下 21
縱橫家 7
縱橫捭闔 3
縱橫自如 0
縱步 0
縱步前進 0
縱波 5
縱波 5
縱深 37
縱火 154
縱火犯 3
縱火狂 1
縱然 218
縱然在 7
縱然是 20
縱然會 0
縱然有 15
縱然能 0
縱目 2
縱目四望 0
縱目遠望 0
縱虎歸山 1
縱行 4
縱覽群書 0
縱覽群籍 0
縱觀 35
縱谷 232
縱谷區 2
縱貫 51
縱貫公路 15
縱貫線 39
縱貫道 1
縱貫鐵路 33
縱身 47
縱軸 31
縱隊 6
縱隊隊形 0
縱飲 0
縲 16
縲絏 1
縳 74
縴 6
縴 6
縴夫 1
縵 14
縶 15
縷 113
縷析 4
縷縷 22
縷述 1
縸 18
縹 8
縹渺 3
縹緲 32
縺 160
縺衣機 0
縻 14
縼 19
總 31985
總主教 17
總主筆 16
總之 727
總人口 175
總代理 436
總值 327
總公司 1056
總共 1043
總共是 9
總共有 162
總兵 38
總冊 7
總分 0
總分類帳 13
總分類帳戶 0
總分類核算 0
總則 339
總動員 830
總務 965
總務主任 164
總務人員 14
總務組 387
總務組長 9
總務處 1331
總匯 245
總參謀部 4
總參謀長 7
總司令 172
總司令部 59
總合 177
總和 239
總商 7
總商會 15
總噸 61
總固體物 0
總圖 343
總局 742
總工會 121
總工程師 29
總帳 108
總平均 188
總幹事 700
總店 126
總廠 150
總後勤部 3
總得 100
總應 3
總應該 3
總成 192
總批發 0
總括 57
總括起來 5
總持如來藏 1
總指揮 19
總指揮部 7
總攬 22
總攬大權 1
總攻 0
總攻擊 4
總政 8
總政治部 9
總教官 43
總教練 134
總教習 0
總數 678
總方針 0
總星系 0
總是 3043
總是在 138
總是會 120
總是有 98
總書記 40
總會 8188
總有 495
總有機碳 13
總樞紐 2
總機 936
總機關 0
總歸 30
總歸一句 3
總歸一句話 6
總比 130
總理 574
總理大臣 11
總理府 9
總理衙門 1
總產值 107
總產量 66
總的來看 3
總的來說 19
總的說來 2
總監 325
總監督 2
總目 70
總督 155
總督府 186
總稱 35
總算 394
總算是 53
總管 376
總索引 22
總結 409
總結會 0
總結經驗 3
總統 8804
總統制 114
總統府 1208
總經理 3115
總經銷 154
總綱 51
總編 92
總編輯 317
總總 15
總署 335
總罷工 0
總而言之 205
總能 85
總能夠 2
總行 172
總裁 746
總覽 0
總計 16478
總計有 64
總評 70
總該 37
總該在 0
總該是 1
總該會 0
總該有 3
總該能 0
總論 1132
總譜 19
總賬 1
總路線 2
總辦 5
總辭 443
總部 2218
總量 299
總量管制 114
總長 181
總開關 10
總預算 1945
總領事 8
總領事館 34
總額 1294
總體 905
總體值 0
總體性 11
總體戰 1
總體經濟 275
績 4969
績優 0
績優股 0
績分 43
績學 4
績效 2105
績效獎金 21
績業 1
縿 4
繀 4
繁 893
繁亂 1
繁分數 0
繁劇 0
繁化 2
繁多 291
繁密 5
繁復 2
繁忙 223
繁文 5
繁文縟節 3
繁星 70
繁星點點 18
繁有徒 1
繁榮 676
繁榮昌盛 3
繁殖 1320
繁殖係數 0
繁殖力 19
繁瑣 103
繁盛 27
繁碎 1
繁細 0
繁育 2
繁花似錦 4
繁苛 2
繁茂 45
繁華 199
繁華世界 3
繁衍 114
繁複 193
繁重 89
繁鉅 0
繁雜 124
繁雜性 0
繁難 2
繁飾 0
繁體 246
繁體字 32
繂 120
繃 60
繃住 1
繃子 0
繃帶 82
繃緊 34
繃臉 0
繃著 0
繃著臉 0
繃著臉兒 0
繃著臉兒 0
繃開 0
繄 12
繅 4
繅絲 2
繆 246
繆 246
繆 246
繆 246
繆誤 2
繇 18
繇 18
繇 18
繇役 0
繇役繁興 0
繈 4
繉 50
繌 5
繍 0
繎 0
繏 0
繏 0
繐 12
繑 7
繒 122
繓 5
織 2089
織出 15
織品 178
織女 31
織女星 21
織女星座 0
織工 56
織布 471
織布廠 12
織布機 26
織成 49
織物 750
織物組織 1
織畫 0
織絹 0
織花 3
織補 99
織襪 9
織軸 1
織造 163
織邊 1
織錦 18
繕 472
繕修 0
繕寫 26
繕寫員 1
繕稿 0
繖 57
繗 5
繘 7
繙 94
繚 380
繚亂 1
繚繞 38
繛 0
繜 5
繝 0
繞 1395
繞到 42
繞口令 14
繞嘴 0
繞圈 11
繞圈圈 4
繞圈子 6
繞在 17
繞完 4
繞彎 0
繞往 0
繞成 3
繞指柔 5
繞樑 8
繞樑三日 1
繞組 16
繞脖子 0
繞著 186
繞行 64
繞起 0
繞起來 0
繞路 8
繞道 50
繞道而行 7
繟 3
繠 9
繡 8600
繡學號 100
繡得 18
繡房 1
繡有 19
繡毯 0
繡球 42
繡的 18
繡花 47
繡花枕頭 0
繡花針 4
繡花針兒 0
繡花針兒 0
繡花鞋 1
繡衣 0
繡起 0
繡起來 2
繡閣 3
繡鞋 5
繢 10
繣 7
繥 0
繦 0
繦 0
繨 82
繩 638
繩之以法 22
繩兒 1
繩兒 1
繩墨 1
繩套 4
繩子 164
繩尺 0
繩度 0
繩梯 6
繩橋 1
繩索 146
繩結 27
繩線 2
繪 1395
繪本 100
繪像 1
繪具 3
繪出 260
繪制 2
繪圖 3223
繪圖室 1
繪圖板 31
繪圖機 113
繪圖筆 2
繪圖紙 2
繪形繪聲 0
繪影 1
繪影繪聲 1
繪成 20
繪畫 1220
繪畫板 0
繪聲 1
繪聲繪影 20
繪聲繪色 1
繪製 580
繪製圖 2
繫 1470
繫 1470
繫上 34
繫到 0
繫囚 0
繫心 2
繫念 5
繫懷 1
繫數 0
繫於 73
繫系 0
繫緊 3
繫繩 4
繫辭 9
繬 0
繭 406
繭子 0
繭層 0
繭絲 1
繭綢 0
繮 0
繮 0
繯 119
繯 119
繰 92
繰 92
繱 0
繲 162
繲 162
繲 162
繳 3192
繳交 2947
繳付 209
繳出 28
繳出來 2
繳出去 0
繳卷 10
繳回 145
繳庫 78
繳掉 0
繳械 8
繳械投降 20
繳械投降 20
繳款 491
繳款單 7
繳款通知 2
繳獲 3
繳白卷 0
繳稅 219
繳稅單 1
繳稅通知 2
繳納 1547
繳費 864
繳費單 14
繳費通知 8
繳過 4
繳過來 0
繳過去 0
繳銷 35
繳錢 0
繳驗 225
繴 110
繵 83
繵 83
繶 24
繷 66
繸 97
繹 165
繺 13
繺 134
繻 78
繻 78
繼 11449
繼任 168
繼任者 9
繼位 17
繼嗣 5
繼女 7
繼子 1
繼室 3
繼往聖之絕學 1
繼往開來 38
繼志 14
繼成 18
繼承 936
繼承人 503
繼承權 58
繼承者 23
繼晷 0
繼武 2
繼母 16
繼父 25
繼絕 4
繼絕存亡 0
繼續 9404
繼續不斷 10
繼續性 19
繼續革命 0
繼而 165
繼起 19
繼電器 157
繽 33
繽紛 302
繾 101
繾綣 12
繿 0
纀 129
纁 58
纂 2189
纂修 6
纂輯 4
纄 0
纅 0
纅 0
纆 7
纇 42
纈 6
纉 0
纊 9
纋 10
續 6362
續任 1
續借 259
續借手續 8
續假 7
續命湯 3
續娶 1
續弦 6
續杯 1
續版 4
續程 6
續篇 48
續約 196
續約者 0
續絃 2
續編 43
續聘 137
續航 10
續航力 22
續訂 82
續訂戶 9
續集 193
纍 12
纍囚 0
纍纍 19
纎 0
纎 0
纏 358
纏上 14
纏上來 0
纏上去 0
纏住 34
纏來纏去 0
纏回 0
纏在 18
纏擾 2
纏結 7
纏綿 128
纏綿悱惻 6
纏繞 90
纏繞在 19
纏繞莖 2
纏繞著 9
纏腳 2
纏著 29
纏裹 0
纏訟 7
纏足 41
纏身 112
纏鬥 35
纑 2
纒 0
纓 188
纓冠 0
纓帽 0
纓簪 0
纓絡 7
纔 191
纕 11
纖 1230
纖人 1
纖介 1
纖夫 0
纖妍 0
纖密 0
纖小 5
纖巧 7
纖度 7
纖弱 10
纖手 12
纖指 5
纖柔 6
纖毛 31
纖毛蟲 11
纖毫 7
纖細 103
纖維 3836
纖維工業 102
纖維板 28
纖維植物 0
纖維素 566
纖維蛋原 0
纖維蛋白 8
纖維蛋白原 3
纖維長度 10
纖纖 24
纖纖玉手 4
纖腰 3
纗 120
纘 12
纙 9
纚 13
纛 8
纛 8
纜 196
纜 196
纜柱 0
纜索 6
纜線 179
纜繩 9
纜車 88
纜車線 1
纝 0
纝 0
纞 0
缶 58
缷 0
缷 0
缸 830
缸內 15
缸外 1
缸子 67
缹 4
缺 2188
缺一 34
缺一不可 42
缺乏 2936
缺人 31
缺勤 12
缺口 166
缺失 575
缺字 0
缺少 460
缺席 267
缺席率 6
缺得 91
缺德 17
缺德事 2
缺德鬼 2
缺憾 110
缺損 144
缺料 5
缺欠 5
缺氧 192
缺油 6
缺疑 0
缺的 91
缺考 46
缺苗 0
缺課 64
缺課紀錄 0
缺貨 220
缺貨量 0
缺陷 417
缺陷美 2
缺頁 1
缺頁數 0
缺項 1
缺額 205
缺點 1203
缻 0
缻 0
缽 104
缽盂 2
缽衣 0
缾 4
缿 4
罁 0
罃 4
罄 34
罄其所有 0
罄擊子 0
罄竭 1
罄竹 7
罄竹難書 7
罄筆難書 0
罅 19
罅口 0
罅漏 1
罅隙 1
罆 0
罇 0
罈 89
罈子 6
罈罈罐罐 0
罊 64
罊 64
罋 76
罌 87
罌粟 33
罌粟科 2
罌粟花 15
罍 13
罎 0
罏 5
罐 3690
罐子 68
罐籠 0
罐蓋 9
罐裝 55
罐車 7
罐頭 533
罐頭笑聲 0
罐頭食品 41
网 233
罒 0
罔 354
罔效 3
罔極 1
罔顧 90
罔顧人道 0
罕 249
罕事 0
罕指 0
罕有 44
罕物 0
罕用 6
罕用字 4
罕聞 0
罕見 405
罘 7
罘 7
罙 0
罛 6
罜 2
罝 23
罞 5
罟 271
罠 6
罡 95
罢 0
罢 0
罢 0
罣 31
罣礙 33
罤 0
罤 0
罥 8
罦 4
罧 13
罧 13
罨 5
罩 6435
罩上 21
罩不住 9
罩住 18
罩兒 0
罩兒 0
罩在 22
罩子 4
罩成 1
罩著 17
罩衣 1
罩衫 12
罩袍 3
罩門 1
罪 1469
罪上加罪 1
罪不可赦 1
罪不可逭 1
罪不容誅 1
罪人 186
罪刑 27
罪名 168
罪名是 6
罪大惡極 21
罪嫌 101
罪孽 24
罪孽深重 11
罪惡 260
罪惡感 65
罪惡滔天 2
罪惡累累 0
罪惡行徑 0
罪愆 9
罪有應得 1
罪案 7
罪犯 102
罪狀 32
罪莫大焉 0
罪行 47
罪行 96
罪証 1
罪該萬死 6
罪該處死 0
罪證 16
罪證確鑿 7
罪責 20
罪責難逃 0
罪輕 2
罪過 70
罪障 2
罪魁 9
罪魁禍首 45
罫 13
罬 5
罭 7
置 8201
置之不理 33
置之不顧 1
置之度外 3
置之死地 7
置之死地而後生 9
置之腦後 2
置信 45
置入 1
置入性 1
置入性行銷 1
置喙 19
置換 166
置放 201
置產 76
置疑 84
置若罔聞 2
置諸度外 1
置身 256
置身事外 44
置身其中 23
置辯 3
置酒 9
置閏 1
罯 0
罰 676
罰中 2
罰了 6
罰些 0
罰作 0
罰來罰去 0
罰個 1
罰出 0
罰出來 0
罰出去 0
罰到 5
罰則 215
罰去 0
罰單 85
罰寫 7
罰得 1
罰得 22
罰成 0
罰抄 0
罰款 476
罰球 103
罰球區 7
罰球線 4
罰的 22
罰站 19
罰給 0
罰著 0
罰調 0
罰起 0
罰起來 0
罰跪 5
罰跳 0
罰踢 2
罰進 1
罰過 1
罰過來 0
罰過去 0
罰酒 4
罰金 414
罰錢 14
罰鍰 1138
罱 0
署 5531
署 5531
署名 143
署理 4
署記長 0
署長 403
罳 8
罵 1071
罵不絕口 2
罵不還口 1
罵了 64
罵人 168
罵來罵去 13
罵個 1
罵到 46
罵得 30
罵得 41
罵的 41
罵罵 15
罵街 2
罵走 0
罵起 1
罵起來 1
罵過 15
罵題 0
罶 3
罷 713
罷 71
罷 0
罷了 670
罷休 37
罷免 191
罷免權 21
罷免法 2
罷兵 2
罷官 3
罷工 228
罷工抗議 0
罷市 2
罷弊 0
罷手 7
罷教 9
罷斥 0
罷朝 0
罷止 0
罷職 1
罷課 29
罷退 0
罷黜 31
罸 0
罹 231
罹厄 0
罹患 789
罹患重病 3
罹災 42
罹禍 1
罹難 167
罺 8
罻 76
罼 25
罽 19
罾 13
罿 8
羀 0
羂 0
羃 80
羃 80
羄 0
羅 11946
羅丹 43
羅伯特 54
羅列 59
羅吉斯 2
羅嗦 1
羅孚宮 0
羅宋湯 6
羅密歐 246
羅布 10
羅布林卡 1
羅布泊 1
羅帷 1
羅得西亞 2
羅德島 52
羅德西亞 5
羅拉 473
羅拜 2
羅掘 0
羅斯 356
羅斯托夫 1
羅斯福 916
羅星塔 2
羅曼史 53
羅曼羅蘭 4
羅曼蒂克 69
羅東 1203
羅欽順 0
羅浮宮 245
羅漢 216
羅漢拳 0
羅漢豆 0
羅瑞卿 3
羅生門 0
羅盤 83
羅盤針 2
羅紋 17
羅素 117
羅經 9
羅網 37
羅織 11
羅羅 11
羅致 66
羅致人材 0
羅蒙諾索 0
羅蒙諾索夫 0
羅蘭 115
羅裙 6
羅貫中 14
羅賓漢 17
羅雀掘鼠 0
羅非魚 0
羅馬 1045
羅馬不是一天造成 31
羅馬人 47
羅馬字 25
羅馬字母 3
羅馬尼亞 168
羅馬帝國 79
羅馬教會 20
羅馬數字 8
羅馬法 1
羆 74
羇 2
羇客 0
羇旅 0
羈 55
羈押 234
羈旅 5
羈束 1
羈泊 0
羈牽 0
羈留 1
羈留不歸 0
羈絆 48
羉 102
羊 14
羊 15520
羊乳 55
羊乳酪 1
羊乳酪 1
羊入虎口 5
羊叫 1
羊奶 50
羊奶粉 15
羊小排 1
羊年 6
羊排 0
羊毛 321
羊毛出在羊身上 9
羊毛狀 0
羊毛皮 2
羊毛脂 7
羊毛衣 3
羊毛衫 16
羊毛袋 0
羊毫 4
羊毫筆 1
羊水 117
羊癇瘋 0
羊癲瘋 4
羊癲風 1
羊皮 53
羊皮紙 10
羊皮襖 0
羊絨 0
羊羔 7
羊群 56
羊羹 31
羊肉 87
羊肉湯 3
羊肉爐 27
羊脂 7
羊腸 10
羊腸小徑 6
羊腸小道 3
羊腸線 3
羊膜 78
羊舍 1
羊角 15
羊角虱 0
羊角風 1
羊質虎皮 1
羊道 0
羊酒 0
羊頭 11
羊齒 17
羋 202
羋 202
羌 421
羌族 12
羌笛 123
羌胡 0
羍 1
美 37565
美不勝收 75
美中不足 75
美乃滋 1
美事 25
美人 1699
美人兒 34
美人兒 34
美人尖 100
美人計 14
美人遲暮 1
美人魚 126
美侖美奐 23
美僑 93
美元 12143
美其名 23
美其名曰 5
美典 11
美分 45
美利堅 13
美利堅共和國 6
美利堅合眾國 18
美則美矣 1
美加 355
美加地區 97
美化 838
美化環境 92
美北 8
美北隊 0
美南 46
美南隊 0
美名 52
美味 416
美味可口 33
美和護專 14
美商 2613
美善 44
美國 37814
美國之音 0
美國人 433
美國式 21
美國話 0
美在 50
美夢 151
美夢成真 214
美女 2476
美女如雲 7
美好 1081
美如天仙 0
美妙 306
美姿 56
美學 796
美容 3386
美容師 48
美容聖品 4
美容術 50
美容院 482
美展 644
美崙 219
美工 812
美工刀 1
美工設計 93
美式 513
美式英語 10
美式足球 71
美得 1147
美得冒泡 100
美德 185
美意 87
美感 364
美才 3
美援 48
美方 402
美日 568
美日關係 7
美景 402
美東 150
美東隊 0
美法 11
美洲 731
美洲人 3
美洲獅 0
美洲虎 10
美洲豹 3
美洲鰻 2
美滿 433
美濃 762
美玉 126
美男子 37
美男計 3
美白 1
美的 1147
美目 30
美眉 100
美眷 11
美籍 75
美索不達米亞 8
美編 12
美聯社 103
美育 108
美色 21
美艷 36
美艷絕倫 0
美蘇 42
美蘇關係 2
美術 3811
美術品 14
美術商 0
美術字 4
美術家 695
美術班 0
美術界 12
美術課 34
美術館 1414
美製 69
美西 165
美西隊 1
美觀 444
美觀大方 27
美言 9
美詞 0
美語 673
美語教學 59
美語課程 13
美談 10
美譽 117
美豔 38
美貌 92
美質 14
美軍 342
美軍基地 4
美軍官兵 0
美輪美奐 18
美酒 130
美醜 23
美金 1032
美鈔 4
美食 2434
美食主義 8
美食街 1
美饌 0
美髮 827
美髯公 4
美麗 4601
美麗動人 27
美麗大方 25
美齒 3
羏 0
羐 0
羑 6
羒 10
羓 0
羔 42
羔皮 1
羔羊 146
羔羊皮 1
羕 18
羖 17
羗 0
羘 0
羙 0
羚 189
羚羊 66
羚羊角 0
羛 4
羜 6
羝 8
羞 126
羞容 0
羞怍 0
羞怯 26
羞怯 26
羞恥 46
羞恥之心 2
羞恥心 6
羞惡 1
羞惡之心 0
羞愧 46
羞愧難當 1
羞慚 7
羞憤 5
羞憤自殺 3
羞於 20
羞於啟齒 9
羞死 7
羞澀 52
羞答答 3
羞紅 2
羞羞 7
羞羞臉 3
羞與為伍 0
羞花閉月 0
羞赧 16
羞赧感 0
羞辱 75
羞辱 75
羠 1
羡 11
羢 6
羣 0
群 17408
群僚 1
群化 25
群婚 0
群小 32
群居 35
群居動物 2
群山 93
群山環繞 14
群峰 65
群島 291
群性 24
群情 9
群情憤慨 1
群情激憤 10
群星 29
群星會 10
群架 3
群毆 3
群治學社 0
群生 72
群眾 656
群眾心理 2
群眾性 5
群眾組織 2
群眾觀點 0
群眾路線 6
群眾運動 29
群策群力 55
群籍 0
群系 7
群組 0
群經 17
群聚 153
群育 54
群臣 18
群芳 39
群芳爭艷 1
群芳競艷 0
群英 81
群英會 26
群落 50
群蟻附羶 0
群謀咸同 0
群賢 13
群起 41
群起而攻 3
群起而攻之 1
群雄 229
群集 181
群雕 0
群體 613
群龍無首 17
羥 80
羥 80
羥基 32
羦 29
羧 72
羧基 17
羧酸 14
羨 161
羨慕 532
義 14786
義不容辭 54
義不生財 1
義不辭難 1
義乳 1
義乳記 1
義人 1
義俠 10
義俠記 1
義倉 1
義僕 4
義兄 2
義兒 1
義兵 1
義分 1
義切中抱 1
義刑義殺 1
義勇 73
義勇兵 1
義勇兵飛彈 1
義勇消防隊 1
義勇艦隊 1
義勇警察 1
義勇軍 12
義務 2483
義務兵 1
義務勞動 3
義務役 79
義務性 23
義務教育 140
義同生死 1
義和團 25
義和團運動 0
義和拳 0
義國 99
義國人 0
義園 1
義地 1
義塚 1
義塾 1
義士 38
義大利 2374
義大利人 55
義大利共和國 1
義大利半島 7
義大利式 24
義大利文 64
義大利王國 0
義大利麵 1
義夫節婦 1
義女 4
義妖傳 1
義子 11
義學 7
義守大學 1
義山恩海 1
義工 1944
義帝 1
義師 1
義府 1
義式 500
義式料理 1
義弟 8
義形於色 0
義役 1
義律 1
義從 1
義憤 9
義憤填胸 1
義憤填膺 12
義戰 1
義手 1
義斷恩絕 1
義方 1
義方之訓 1
義方是訓 1
義旗 0
義正嚴詞 7
義正詞嚴 8
義正辭嚴 2
義母 1
義民 206
義氣 41
義氣干霄 1
義氣深重 1
義氣相投 1
義法 2
義海恩山 1
義消 0
義淨 1
義渠 1
義演 43
義漿 1
義無反顧 49
義父 11
義犬 15
義玄 1
義理 71
義田 21
義甲 1
義男 1
義男兒 1
義界 1
義疏 1
義節 1
義粟仁漿 1
義結金蘭 7
義絕 1
義美 65
義義合合 1
義肢 86
義舉 41
義莊 3
義薄雲天 11
義行 1
義行 42
義行可風 1
義行可風 1
義訓 1
義診 199
義試 0
義警 40
義貞 1
義賣 373
義賣場 0
義賣場 0
義賣會 35
義賽 1
義趣 1
義足 2
義軍 27
義重如山 1
義重恩深 1
義重情深 1
義鋪 1
義門 1
義項 1
義風 3
義髻 1
義齒 43
羫 0
羫 0
羬 1
羭 2
羮 0
羮 0
羯 557
羯磨 1
羯磨曼陀羅 1
羯鼓 2
羰 29
羱 8
羲 597
羲皇上人 0
羳 7
羴 0
羵 6
羶 699
羶味 0
羶氣 0
羶腥 5
羷 69
羸 115
羸弱 7
羸憊 0
羸疾 0
羸瘦 0
羹 234
羹匙 0
羹湯 12
羹食 0
羹飯 2
羺 83
羻 27
羻 27
羼 36
羼假 0
羼水 0
羼雜 1
羽 1417
羽化 30
羽化成仙 0
羽士 4
羽扇 27
羽扇綸巾 3
羽書 0
羽林軍 1
羽檄 0
羽毛 373
羽毛未豐 0
羽毛狀 3
羽毛球 84
羽毛球拍 5
羽毛球賽 0
羽毛球館 0
羽毛豐滿 0
羽片 2
羽狀 25
羽狀脈 0
羽狀脈 0
羽狀複葉 31
羽球 684
羽球拍 44
羽球網 18
羽球賽 53
羽球館 23
羽田機場 13
羽田機場 13
羽翼 67
羽翼已豐 0
羽衣 18
羽軸 2
羽量級 10
羽飾 2
羾 5
羿 232
翀 9
翁 2158
翁仲 6
翁同和 0
翁姑 2
翁婿 0
翁嫗 0
翂 6
翃 13
翄 0
翅 542
翅兒 0
翅兒 0
翅膀 249
翆 0
翇 5
翇 5
翈 0
翉 7
翊 568
翋 819
翌 166
翌年 160
翌日 178
翍 214
翍 214
翎 364
翎兒 0
翎兒 0
翎子 0
翎毛 4
翏 7
翐 15
翑 86
習 2685
習以為常 44
習作 300
習作簿 1
習俗 250
習字 12
習字帖 6
習字課 0
習定 4
習尚 2
習性 432
習慣 2535
習慣性 112
習慣成自然 6
習慣於 65
習慣法 7
習慣用法 7
習慣用語 1
習於 70
習染 4
習氣 57
習用 30
習習 18
習而不察 4
習藝 26
習藝所 3
習見 11
習非 1
習非勝是 0
習非成是 0
習題 283
翓 0
翔 3555
翔實 88
翔集 1
翕 75
翕如 0
翕然 1
翖 0
翖 0
翗 4
翛 10
翜 9
翝 0
翞 61
翟 341
翟 341
翟讓 18
翠 1553
翠亨村 9
翠屏山 6
翠微 5
翠柏 8
翠柏 8
翠玉 71
翠竹 5
翠綠 84
翠綠色 9
翠華 48
翠鳥 88
翠黛 14
翡 85
翡冷翠 21
翡翠 422
翡翠水庫 82
翡翠色 2
翡翠谷 8
翢 3
翣 3
翤 0
翥 8
翦 211
翦落 0
翦除 6
翧 0
翨 7
翩 1231
翩然 30
翩然而過 0
翩翩 707
翩翩公子 0
翩翩少年 0
翩翩起舞 16
翪 3
翫 11
翫歲愒時 0
翬 17
翭 8
翮 47
翯 5
翰 1729
翰墨 25
翰林 75
翰林學士 1
翰林院 4
翰海 1
翰藻 1
翱 85
翱翔 98
翱翔天空 0
翱翔天際 6
翱遊 4
翱遊四海 0
翲 4
翳 33
翳障 0
翴 91
翵 49
翶 0
翷 9
翸 2
翹 720
翹 720
翹上 0
翹上來 0
翹上去 0
翹下 0
翹下來 0
翹下去 0
翹了 6
翹企 0
翹出 1
翹出來 0
翹出去 0
翹去 0
翹楚 43
翹翹板 11
翹舌音 0
翹著 7
翹課 1
翹起 26
翹起來 2
翹趣 0
翹辮子 3
翹首 13
翹首以望 0
翹首企盼 1
翺 0
翻 1642
翻一番 2
翻上 7
翻上來 0
翻上去 0
翻下 7
翻下來 1
翻下去 1
翻了 133
翻些 0
翻作 6
翻來 3
翻來復去 1
翻來翻去 4
翻來複去 0
翻來覆去 20
翻供 10
翻修 83
翻個 5
翻倒 18
翻入 4
翻出 36
翻出來 6
翻出去 0
翻到 42
翻動 19
翻印 87
翻印必究 122
翻唱 35
翻地 7
翻天 118
翻天覆地 8
翻好 3
翻完 7
翻寫 1
翻山 7
翻山倒海 0
翻山越嶺 19
翻弄 4
翻得 6
翻悔 1
翻成 64
翻手 2
翻掉 10
翻改 0
翻新 82
翻書 29
翻有 1
翻本 4
翻案 59
翻江倒海 5
翻湧 8
翻滾 75
翻為 30
翻牆 40
翻牆而過 0
翻版 61
翻版書 0
翻白 10
翻白眼 4
翻看 14
翻眼 0
翻砂 3
翻筋斗 9
翻箱倒櫃 21
翻箱倒篋 3
翻紅 67
翻翻 61
翻胃 0
翻臉 35
翻臉不認 1
翻臉不認人 6
翻臉無情 2
翻自 0
翻舊帳 60
翻舊賬 0
翻船 21
翻蓋 1
翻覆 40
翻觔斗 3
翻譯 3218
翻譯出 5
翻譯品 2
翻譯員 15
翻譯官 5
翻譯家 9
翻譯機 20
翻譯社 202
翻譯者 15
翻起 20
翻起來 2
翻越 34
翻跟斗 3
翻跟頭 0
翻身 225
翻車 56
翻轉 98
翻過 45
翻過來 13
翻過去 13
翻錯 2
翻開 164
翻閱 157
翻雲覆雨 28
翻飛 20
翻騰 55
翻點 0
翼 1242
翼展 24
翼形 6
翼梢 0
翼足 0
翼足類 0
翽 56
翾 93
翿 88
翿 88
耀 336
耀 3364
耀亮 1
耀武揚威 5
耀目 19
耀眼 323
老 9973
老一套 0
老一輩 71
老丈 9
老丈人 2
老上 1
老上來 0
老上去 0
老下 2
老下來 0
老下去 0
老不修 5
老世交 0
老中 43
老主顧 6
老么 37
老了 157
老二 1
老交情 0
老人 3309
老人家 352
老人節 1
老伯 283
老伯伯 14
老伴 15
老伴兒 0
老伴兒 0
老佛爺 32
老佬 15
老來 22
老來無伴 0
老來無子 0
老來無子 0
老個 1
老僕 4
老僧 21
老僧入定 2
老兄 166
老兄弟 4
老先生 263
老兒 9
老兒 9
老兒子 0
老兒子 0
老公 624
老公公 173
老共 59
老兵 94
老兵不死 26
老到 18
老前輩 86
老劉 25
老化 592
老區 1
老千 39
老半天 40
老去 20
老友 107
老叔 0
老叟 10
老古董 7
老台 1
老向 1
老君 16
老吳 12
老吾老 8
老吾老以及人之老 8
老命 25
老命一條 0
老命嗚呼 0
老哥 99
老哥兒們 0
老哥兒們 0
老嚴 0
老圃 3
老壽星 2
老外 129
老大 670
老大不小 12
老大哥 32
老大姐 4
老大娘 0
老大徒傷悲 2
老大爺 3
老大難 0
老天 161
老天爺 80
老太太 155
老太婆 57
老太爺 8
老夫 127
老夫子 56
老夫子 56
老夫少妻 0
老夫老妻 7
老套 41
老奴 2
老奴才 0
老奶奶 25
老奸 7
老奸巨猾 3
老好人 2
老妖 3
老妹子 0
老妹子 0
老姊 8
老姐 24
老姐兒 0
老姐兒 0
老姑娘 0
老娘 18
老娼 2
老婆 1287
老婆婆 79
老婆子 4
老婦 109
老媼 0
老媽 182
老媽子 10
老嫗 21
老嬤嬤 4
老子 275
老子 275
老字號 55
老學究 3
老學究 3
老官 1
老家 101
老實 218
老實人 12
老實話 15
老實說 251
老將 139
老小 26
老少 63
老少咸宜 81
老巢 11
老師 17634
老師傅 39
老帳 2
老年 705
老年人 409
老幹部 5
老幼 27
老幾 12
老底 10
老店 134
老店新開 15
老式 62
老弟 91
老弱 28
老弱傷殘 0
老弱婦孺 21
老弱殘兵 3
老弱病殘 1
老張 52
老彭 4
老得 23
老得 199
老情歌 3
老想 18
老態 12
老態龍鍾 4
老成 30
老成凋謝 1
老成持重 0
老手 101
老把戲 5
老拙 3
老拳 0
老掉牙 30
老撾 1
老於世故 0
老旦 11
老早 42
老是 591
老景 0
老朋友 196
老本 21
老本行 5
老朽 43
老東西 10
老板 334
老板娘 56
老林 24
老梗 1
老楊 5
老樣子 12
老樹 567
老死 38
老死不相往來 3
老死他鄉 0
老死異地 0
老殘 14
老殘遊記 14
老母 47
老母雞 7
老毛病 25
老氣 6
老氣橫秋 3
老江 7
老江湖 90
老油條 87
老法子 1
老法子 1
老派 5
老淚 2
老淚橫流 0
老淚縱橫 4
老滑頭 0
老滑頭 0
老漢 30
老營 0
老父 15
老父 15
老父老母 1
老爸 403
老爹 166
老爺 411
老爺子 37
老爺爺 18
老牌 136
老牛 42
老牛吃嫩草 0
老牛拖車 4
老牛舐犢 0
老物 0
老狐狸 15
老狗 42
老王 135
老王賣瓜 0
老現象 0
老琉璃 0
老生 33
老生常談 160
老當 2
老當 2
老當益壯 20
老病 18
老百姓 274
老的 199
老眼昏花 3
老祖 16
老祖宗 60
老神在在 22
老簡 1
老米 1
老粗 4
老糊塗 6
老練 27
老總 13
老繭 1
老美 61
老羞成怒 1
老翁 51
老老 30
老老實實 20
老老少少 8
老耄 0
老者 160
老而不死 6
老而彌堅 6
老聲老氣 1
老背少 0
老臉 8
老臉皮 1
老舊 453
老花 20
老花樣 0
老花眼 31
老花眼鏡 33
老苦 1
老莊 67
老莊學會 1
老莊思想 6
老菸槍 1
老萊子 4
老萊子 4
老蔣 15
老薑 6
老虎 364
老虎凳 0
老虎鉗 3
老處女 6
老蚌生珠 2
老行家 0
老街 235
老街坊 0
老街坊 0
老表 4
老衲 9
老被 6
老要 9
老規矩 9
老話 80
老話兒 0
老話兒 0
老說 30
老說是 0
老調重彈 3
老謀深算 10
老豆腐 1
老財 0
老賊 19
老路 14
老路子 2
老身 16
老輩 1
老農 159
老農夫 9
老運 4
老運亨通 0
老道 44
老遠 48
老邁 22
老邁無能 0
老部 4
老鄉 9
老酒 8
老醋 0
老鋪 2
老長 1
老長 1
老闆 1678
老闆娘 135
老靠 0
老面子 0
老面皮 0
老頑固 3
老頑童 59
老頭 184
老頭兒 6
老頭兒 6
老頭子 75
老顏 4
老饕 87
老馬 24
老馬識途 3
老驥伏櫪 6
老骨頭 32
老骨頭 32
老高 22
老鳥 359
老鴇 20
老鴉 1
老鴰 1
老鷹 134
老麥 17
老黃牛 0
老鼠 730
老鼠會 61
考 5660
考上 443
考不倒 10
考不到 3
考中 32
考了 71
考來考去 0
考個 6
考倒 16
考出 10
考出來 0
考到 38
考前 129
考前大猜題 0
考前考後 0
考勤 18
考區 197
考卷 107
考卷紙 0
考取 234
考古 489
考古學 153
考古學家 57
考古題 1502
考問 3
考場 13
考場 138
考壞 2
考好 22
考妣 1
考季 13
考完 174
考完試 40
考官 15
考察 919
考察團 127
考後 68
考得 52
考得 160
考慮 5365
考慮不周 2
考慮到 473
考慮周到 10
考慮看看 19
考成 23
考據 42
考據學 1
考查 307
考核 1199
考法 2
考獻 3
考生 1032
考的 159
考種 0
考究 37
考績 247
考考 96
考考試 6
考著 3
考覈 0
考訂 9
考証 39
考試 7810
考試及格 264
考試時 85
考試院 680
考試院長 29
考證 113
考證學 3
考證學家 0
考起 2
考起來 0
考運 15
考過 41
考過來 0
考過去 0
考選 111
考選部 562
考量 2816
考錯 0
考題 485
考題大全 5
考驗 1160
考驗性 0
考點 0
耄 3
耄耋 3
耄齡 0
者 77478
耆 111
耆儒 0
耆宿 27
耆碩 0
耆老 44
耇 4
耈 0
耈 0
耉 0
耉 0
耊 0
耊 0
耊 0
耋 2085
而 91929
而非 100
而上 261
而上來 1
而上去 0
而下 218
而下來 0
而下去 0
而不 2096
而不 2096
而不是 1325
而不會 184
而不能 206
而且 12580
而且在 399
而且是 451
而且會 110
而且有 231
而且能 71
而且都 52
而今 311
而以 636
而來 1116
而光 34
而入 121
而勝於藍 1
而又 425
而回 56
而外 86
而夠 0
而定 665
而將 450
而就 132
而已 2776
而帶 59
而延 13
而後 615
而必 20
而戴 5
而把 152
而提 76
而於 225
而是 4263
而替 9
而會 106
而有 1443
而比 80
而比 16
而沿 5
而況 11
而演 19
而為 618
而為 618
而無 362
而生 230
而由 298
而立 100
而立之年 10
而與 314
而言 4969
而論 125
而起 218
而起來 0
而跟 18
耍 421
耍嘴 12
耍嘴皮 6
耍威風 0
耍弄 8
耍弄刀槍 0
耍態度 0
耍手藝 0
耍把戲 4
耍掉 0
耍滑頭 0
耍滑頭 0
耍猴兒 0
耍猴兒 0
耍笑 0
耍脾氣 1
耍花招 4
耍花槍 0
耍花樣 3
耍貧嘴 0
耍賴 16
耍賴皮 0
耍起 0
耍起來 0
耍鬧 0
耎 4
耏 4
耐 1851
耐不住 26
耐久 113
耐久性 98
耐人尋味 61
耐住 4
耐冬 2
耐力 193
耐力賽 3
耐勞 17
耐壓 44
耐壓性 2
耐寒 24
耐寒性 12
耐心 1048
耐性 115
耐旱 12
耐旱性 3
耐水 19
耐水性 14
耐溫 36
耐溫性 3
耐火 173
耐火度 1
耐火性 5
耐火材料 101
耐火磚 22
耐火黏土 6
耐煩 24
耐熱 184
耐熱合金 3
耐熱性 104
耐熱鋼 1
耐燃 93
耐燃性 15
耐用 178
耐用性 23
耐磨 90
耐磨性 15
耐磨性能 0
耐穿 6
耐酸 48
耐酸性 11
耐震 141
耐鹼 16
耐鹼性 14
耑 19
耑 19
耒 58
耒耜 2
耒耨 0
耓 0
耔 4
耕 769
耕 769
耕作 268
耕作制度 11
耕作層 0
耕傭 0
耕出 1
耕地 410
耕地面積 35
耕牛 6
耕牧 0
耕犁 3
耕田 31
耕畜 0
耕種 63
耕稼 1
耕穫 0
耕織 0
耕翻 0
耕者 6
耕者有其田 18
耕耘 346
耕耘機 6
耕耨 0
耕莘 128
耕莘醫院 91
耕讀 29
耕農 21
耕食 0
耕馬 0
耖 8
耗 1161
耗住 0
耗力 12
耗去 8
耗子 12
耗掉 17
耗損 66
耗損率 4
耗損量 2
耗散 4
耗時 110
耗時耗事 0
耗時間 8
耗材 0
耗氧 19
耗油 35
耗油率 2
耗油量 13
耗減 1
耗熱 0
耗熱量 0
耗用 128
耗盡 98
耗竭 31
耗著 2
耗費 280
耗資 130
耗電 66
耗電率 4
耗電量 65
耗體力 4
耘 153
耘田 1
耘耨 0
耘草 0
耘藝 0
耙 92
耙 92
耙土 0
耙子 4
耙犁 0
耙開 0
耚 0
耛 1
耜 44
耝 0
耞 8
耟 32
耠 0
耡 2
耣 0
耤 483
耥 0
耦 72
耦合 192
耨 104
耩 22
耪 12
耫 0
耫 0
耬 16
耭 0
耮 0
耯 0
耯 0
耰 4
耱 0
耲 0
耳 9354
耳下腺 1
耳下腺炎 0
耳光 39
耳刀 0
耳刮子 3
耳力 3
耳垂 25
耳垢 34
耳塞 19
耳塞著 0
耳墜子 1
耳套 0
耳孔 2
耳尖 2
耳屎 9
耳後 23
耳房 2
耳挖 0
耳挖子 0
耳提面命 13
耳朵 3035
耳朵 3035
耳朵痛 6
耳朵痛 6
耳朵眼 1
耳朵眼 1
耳朵軟 0
耳朵軟 0
耳栓 0
耳根 26
耳根清靜 0
耳機 405
耳殼 12
耳毛 1
耳沈 0
耳洞 16
耳濡目染 30
耳炎 18
耳熟 13
耳熟 13
耳熟能詳 119
耳熟能詳 119
耳熱 0
耳環 121
耳環孔 0
耳生 2
耳疾 11
耳痛 5
耳目 77
耳目一新 109
耳石 14
耳科 30
耳科醫生 0
耳罩 7
耳翼 10
耳聞 68
耳聞為虛 0
耳聰目明 7
耳聽八方 7
耳聾 59
耳背 5
耳膜 0
耳膜 0
耳蝸 14
耳語 118
耳識 1
耳輪 5
耳邊 144
耳邊風 5
耳門 87
耳際 23
耳順 13
耳飾 9
耳鬢 1
耳鬢廝磨 3
耳鳴 103
耳鳴現象 2
耳鼓 5
耳鼻喉 133
耳鼻喉科 709
耳鼻科 4
耴 5
耵 117
耶 3542
耶 3542
耶和華 501
耶教 8
耶穌 2536
耶穌基督 597
耶穌教 22
耶穌會 42
耶誕 1453
耶誕夜 78
耶誕節 404
耶路撒冷 361
耶魯 44
耶魯大學 118
耷 1
耷拉 0
耸 0
耹 11
耺 0
耺 0
耻 0
耼 0
耼 0
耽 100
耽心 40
耽擱 38
耽擱到 0
耽於 12
耽樂 0
耽湎 0
耽溺 32
耽誤 77
耽誤到 3
耽迷 0
耾 11
耿 421
耿介 1
耿直 13
耿精忠 0
耿耿 11
耿耿於懷 28
聀 0
聃 21
聄 0
聅 0
聆 187
聆教 1
聆聽 582
聆訊 1
聆賞 53
聇 8
聈 1
聉 0
聉 0
聉 0
聊 2188
聊上 2
聊且 1
聊以 8
聊以維生 0
聊以解嘲 0
聊來聊去 0
聊個 19
聊個痛快 0
聊備一格 10
聊備一格 10
聊到 44
聊勝於無 10
聊天 12539
聊天兒 0
聊天兒 0
聊天室 1
聊得 24
聊法 0
聊生 2
聊聊 393
聊聊天 174
聊著 30
聊表 7
聊表心意 4
聊起 35
聊起來 8
聊齋誌異 7
聎 0
聏 2
聏 2
聐 5
聑 3
聒 223
聒噪 24
聒絮 3
聒耳 1
聒聒 4
聒聒叫 0
聕 0
聖 970
聖上 25
聖主 49
聖人 253
聖代 35
聖伯納 6
聖保祿 30
聖保羅 108
聖保羅島 0
聖俸 0
聖像 81
聖僧 14
聖典 69
聖功 102
聖哲 35
聖哲先賢 0
聖嚴 251
聖地 380
聖地牙哥 157
聖地牙哥城 0
聖城 46
聖壇 4
聖女 55
聖嬰 0
聖嬰現象 0
聖廟 20
聖弗蘭西斯科 0
聖彼得大教堂 4
聖徒 600
聖德 187
聖心 157
聖恩 15
聖戰 72
聖手 19
聖教 46
聖旨 39
聖明 15
聖書 18
聖杯 36
聖果 9
聖歌 44
聖殿 204
聖母 242
聖母瑪利亞 11
聖母院 40
聖水 13
聖潔 204
聖火 47
聖火台 2
聖相 1
聖禮 12
聖約翰學院 0
聖經 2432
聖經上 68
聖者 73
聖職 11
聖職衣 0
聖荷西 89
聖裔 0
聖詩 28
聖誕 1640
聖誕卡 192
聖誕夜 93
聖誕樹 70
聖誕禮物 38
聖誕節 958
聖誕紅 34
聖誕老人 85
聖誕葉 0
聖諭 7
聖賢 225
聖蹟 15
聖雄 11
聖靈 1056
聖餐 54
聖餐布 0
聖馬力諾 0
聖體 14
聖訓 30
聗 0
聘 2914
聘任 579
聘任制 7
聘僱 0
聘召 0
聘姑娘 0
聘成 4
聘書 76
聘用 223
聘禮 17
聘請 809
聘金 13
聙 0
聚 13920
聚丙烯 131
聚乙烯 137
聚乙烯醇 38
聚乙稀 2
聚了 15
聚光 25
聚光燈 25
聚光鏡 6
聚合 364
聚合作用 13
聚合反應 48
聚合物 196
聚合起來 1
聚合體 61
聚四氟乙烯 2
聚在 43
聚在一起 114
聚寶盆 223
聚少離多 5
聚居 87
聚攏 4
聚攏在 0
聚散 32
聚散無常 2
聚斂 9
聚星 0
聚會 1333
聚會地點 6
聚會所 39
聚會時間 30
聚有 3
聚氨酯 18
聚氯乙烯 97
聚氯乙烯塑料 1
聚沙成塔 14
聚焦 129
聚熱 5
聚熱作用 0
聚眾 45
聚神 0
聚積 20
聚積為 0
聚精會神 28
聚結 4
聚義 1
聚聚 50
聚苯乙烯 67
聚落 320
聚著 5
聚蚊成雷 0
聚訟 4
聚變 1
聚變反應 0
聚賭 8
聚起來 1
聚酯 693
聚酯樹脂 12
聚酯纖維 170
聚錯 0
聚集 698
聚頭 4
聚餐 278
聚餐會 0
聚首 28
聚齊 3
聛 0
聜 233
聝 725
聞 369
聞 3696
聞一知十 1
聞了 14
聞人 33
聞來聞去 0
聞出 8
聞到 137
聞名 403
聞名世界 15
聞名全球 12
聞名四方 1
聞名天下 1
聞問 15
聞報 871
聞得 20
聞所未聞 7
聞知 29
聞聞 14
聞聞看 3
聞聽 2
聞見 6
聞訊 21
聞起 1
聞起來 16
聞過 8
聞過則喜 0
聞過則改 0
聞道 25
聞道有先後 4
聞達 5
聞雞起舞 2
聞風 27
聞風喪膽 11
聞風而來 7
聞風而動 0
聞風而至 2
聞風而起 0
聞風而逃 0
聞風逃竄 0
聞風響應 0
聟 0
聠 0
聡 0
聤 0
聥 0
聦 0
聧 0
聨 0
聪 0
聫 0
聬 0
聭 0
聮 0
聯 15257
聯係 11
聯保 6
聯共 5
聯勤 167
聯勤總部 17
聯句 4
聯合 6996
聯合企業 1
聯合作戰 14
聯合促銷 1
聯合內閣 33
聯合公報 153
聯合制 1
聯合國 1873
聯合國大會 35
聯合國憲章 36
聯合國案 6
聯合報 1014
聯合報系 58
聯合大學 1
聯合工專 54
聯合戰線 1
聯合收割機 0
聯合政府 126
聯合晚報 122
聯合會 589
聯合機 4
聯合演習 1
聯合目錄 1
聯合聲明 148
聯合行動 8
聯合部隊 1
聯名 221
聯大 90
聯姻 22
聯婚 13
聯展 422
聯席 200
聯席會議 247
聯帶 4
聯想 388
聯想到 147
聯想力 9
聯想起 25
聯成 438
聯成一組 0
聯手 493
聯招 1070
聯招會 92
聯接 35
聯接器 1
聯播 136
聯播時間 1
聯播節目 4
聯於 20
聯會 341
聯有 17
聯歡 53
聯歡晚會 44
聯歡會 33
聯歡節 1
聯營 155
聯營公車 78
聯璧 4
聯用 1
聯盟 7097
聯票 10
聯立 26
聯立方程 2
聯立方程式 21
聯立方程組 6
聯系 49
聯組 1
聯結 48
聯絡 22094
聯絡兵 0
聯絡員 11
聯絡地點 5
聯絡官 1
聯絡時間 32
聯絡站 15
聯絡簿 137
聯絡處 577
聯絡點 5
聯繫 2014
聯繫國 0
聯繫實際 1
聯繫群眾 0
聯署 71
聯考 910
聯考題 1
聯苯 41
聯華 442
聯袂 48
聯言判斷 0
聯課 9
聯課活動 50
聯誼 26
聯誼 2675
聯誼會 15
聯誼會 1520
聯貫 7
聯貫性 2
聯貸 113
聯賽 296
聯軍 82
聯軍部隊 1
聯軸器 2
聯軸節 21
聯通 32
聯運 70
聯運會 0
聯運票 1
聯邦 1390
聯邦制 11
聯邦政府 236
聯邦調查局 64
聯鎖 20
聯鎖店 23
聯防 55
聯防軍 0
聯隊 91
聯集 949
聯電 20
聯韻詩 0
聰 1280
聰悟 0
聰慧 39
聰敏 89
聰明 1431
聰明一世 3
聰明反被聰明誤 6
聰明才智 29
聰明絕頂 14
聰明起來 0
聰穎 44
聱 8
聱牙 0
聲 7784
聲價 0
聲價不凡 0
聲光 163
聲光效果 81
聲勢 286
聲勢浩大 62
聲區 3
聲名 53
聲名大噪 31
聲名狼籍 4
聲名狼藉 4
聲名遠播 11
聲吶 1
聲嘶力竭 50
聲地 101
聲塵 1
聲如洪鐘 0
聲威 8
聲學 93
聲容 0
聲寶 449
聲帶 376
聲張 16
聲律 2
聲息 31
聲控 33
聲援 392
聲明 1913
聲明書 67
聲望 175
聲東擊西 11
聲樂 129
聲樂家 285
聲母 10
聲氣 7
聲波 112
聲波 112
聲浪 190
聲淚俱下 8
聲源 27
聲稱 303
聲符 15
聲納 36
聲聞 32
聲聲 84
聲色 70
聲色俱厲 3
聲色犬馬 7
聲覺 0
聲言 20
聲討 9
聲調 109
聲請 0
聲譽 150
聲譽不衰 0
聲譽卓著 16
聲譽正隆 2
聲速 4
聲部 70
聲門 5
聲障 0
聲霸卡 125
聲音 5700
聲韻 57
聲韻學 40
聲響 252
聲頻 15
聳 105
聳了 0
聳了聳 3
聳了聳肩 8
聳人聽聞 9
聳入 2
聳入雲霄 3
聳出 3
聳動 65
聳動視聽 0
聳峙 6
聳恿 6
聳懼 0
聳立 59
聳立在 20
聳立著 5
聳聳 5
聳聳肩 23
聳聽 2
聳肩 20
聴 0
聴 0
聵 11
聶 301
聶伯河 0
聶政 0
職 9000
職事 138
職任 9
職份 16
職位 552
職分 32
職別 48
職務 2837
職司 128
職員 1261
職員表 6
職場 0
職場 0
職守 24
職官 6
職工 287
職工福利 71
職工薪津 0
職志 66
職掌 975
職是之故 8
職校 622
職棒 1817
職棒大賽 5
職業 6326
職業介紹 65
職業介紹所 16
職業倦怠 8
職業婦女 65
職業學校 712
職業工會 386
職業性 175
職業教育 390
職業病 214
職權 694
職權所在 0
職稱 2563
職籃 0
職缺 0
職能 909
職能資本 0
職蜂 0
職訓 0
職責 532
職責所在 10
職銜 18
聸 45
聹 148
聺 0
聻 0
聻 0
聼 0
聼 0
聽 1
聽 6943
聽上 8
聽下去 7
聽不出 12
聽不出來 8
聽不到 135
聽不得 0
聽不懂 133
聽不清 18
聽不見 63
聽不進去 16
聽了 614
聽事 1
聽任 18
聽任 19
聽作 0
聽作是 0
聽來 102
聽來聽去 4
聽便 2
聽信 43
聽候 20
聽候通知 1
聽做 1
聽其自然 1
聽出 39
聽出來 1
聽到 2711
聽力 460
聽力測驗 38
聽厭 3
聽取 452
聽命 37
聽天由命 15
聽好 34
聽完 169
聽審 1
聽寫 70
聽寫測驗 1
聽差 1
聽得 343
聽得出 9
聽得出來 7
聽得到 45
聽得懂 54
聽得見 32
聽從 132
聽慣 9
聽憑 2
聽憑處置 0
聽懂 54
聽成 10
聽我 216
聽戲 2
聽政 3
聽決 0
聽的 342
聽眾 499
聽神經 25
聽筒 197
聽管 5
聽者 81
聽而不聞 5
聽聞 145
聽聽 489
聽聽看 68
聽膩 5
聽著 210
聽見 508
聽見了 50
聽覺 365
聽訓 2
聽訟 4
聽診 24
聽診器 14
聽話 135
聽說 2113
聽說在 31
聽說是 62
聽說會 5
聽說有 90
聽說能 0
聽課 56
聽講 2453
聽證會 98
聽起 19
聽起來 288
聽進 16
聽過 750
聽過來 0
聽過去 0
聽道 34
聽錯 10
聾 308
聾人 25
聾啞 53
聾啞學校 4
聾啞症 0
聾子 72
聾盲 0
聾聵 0
聿 212
肁 0
肂 21
肄 275
肄業 295
肅 682
肅反 0
肅坐 0
肅容 6
肅敬 0
肅殺 17
肅殺之氣 8
肅清 44
肅清殘敵 0
肅然 17
肅然起敬 20
肅穆 45
肅立 12
肅立起敬 0
肅貪 0
肅靜 20
肅靜無聲 1
肆 1504
肆力 0
肆口 0
肆口謾罵 0
肆意 33
肆意妄為 0
肆意揮霍 0
肆應 13
肆暴 1
肆橫 1
肆無忌憚 36
肆然 1
肆虐 183
肆虐為患 0
肇 710
肇事 312
肇事者 32
肇因 138
肇基 20
肇始 29
肇始者 0
肇建 7
肇慶 55
肇禍 35
肇端 4
肇興 0
肇造 1
肇釁 0
肈 0
肉 3
肉 3266
肉丁 8
肉中刺 2
肉丸 65
肉丸子 3
肉乾 15
肉兒 0
肉兒 0
肉冠 1
肉凍 2
肉刑 1
肉包 209
肉包子 9
肉包子打狗 5
肉品 0
肉圓 66
肉店 24
肉彈 4
肉感 7
肉慾 27
肉慾主義 1
肉搏 23
肉搏戰 15
肉末 3
肉柱 1
肉桂 52
肉毒桿菌 1
肉汁 10
肉湯 17
肉燥 1
肉片 26
肉牛 32
肉瘤 44
肉白骨 0
肉皮 1
肉眼 208
肉票 18
肉粽 65
肉絲 53
肉絲麵 4
肉羹 57
肉羹米粉 0
肉羹麵 6
肉腸 1
肉舖 7
肉色 21
肉芝 0
肉芽 31
肉販 5
肉質 68
肉身 153
肉身菩薩 1
肉醬 33
肉鋪 1
肉雞 77
肉類 285
肉食 56
肉食動物 10
肉食性 104
肉餅 20
肉餡 16
肉骨茶 1
肉體 425
肉體上 29
肉鬆 53
肉鬆罐頭 0
肉鬆罐頭 0
肉鬮 0
肉麻 44
肉麻當有趣 2
肊 34
肋 179
肋排 1
肋窩 0
肋膜 33
肋膜炎 7
肋軟骨 0
肋間 9
肋骨 60
肌 1151
肌理 19
肌肉 1459
肌肉發達 4
肌肉組織 25
肌腱 133
肌膚 711
肌膚之親 13
肌蛋白 1
肌骨 3
肌體 3
肍 0
肎 0
肎 0
肏 29
肏你 29
肏你媽的屄 29
肐 3
肐膊 0
肐膊肘 0
肐臂 0
肑 0
肒 10
肓 49
肔 0
肔 0
肕 7
肖 248
肖似 4
肖像 116
肖像畫 15
肖邦 0
肗 0
肘 172
肘子 1
肘彎 4
肘後方 1
肘節 4
肘腋 6
肘腋 6
肘部 9
肙 2
肚 327
肚 327
肚兜 14
肚子 424
肚子痛 57
肚皮 50
肚腸 1
肚臍 74
肚臍眼 14
肚量 13
肛 5000
肛交 23
肛門 288
肛門口 10
肜 3
肝 1526
肝功能 156
肝吸蟲 16
肝油 0
肝火 19
肝炎 1252
肝疾 25
肝病 366
肝癌 1252
肝硬化 253
肝硬變 4
肝腦塗地 3
肝腸寸斷 4
肝膽 271
肝膽俱裂 59
肝膽照人 0
肝膽相照 9
肝膽過人 0
肝臟 752
肝蛭 2
肝部 2
肝醣 26
肞 0
肟 0
股 19836
股份 886
股份公司 75
股份有限 296
股份有限公司 72420
股價 2326
股分 36
股利 588
股員 6
股子 4
股市 13331
股市行情 239
股息 58
股慄 1
股本 278
股東 1673
股東大會 30
股東會 281
股東會議 7
股條 0
股權 791
股民 0
股溝 2
股災 0
股票 7267
股票市場 78
股票市場 784
股票族 32
股肱 2
股資 112
股金 25
股長 267
股關節 5
股骨 68
肢 368
肢体 6
肢解 10
肢骨 1
肢體 733
肢體衝突 38
肣 65
肤 0
肤 0
肥 1311
肥了 21
肥分 0
肥到 2
肥力 0
肥厚 54
肥土 3
肥地 0
肥城 1
肥壤 0
肥壯 2
肥大 198
肥實 0
肥強 0
肥得 58
肥效 10
肥料 473
肥料廠 4
肥東 0
肥水 10
肥水不落外人田 4
肥沃 72
肥源 0
肥滿 10
肥甘 2
肥田 2
肥田粉 1
肥瘠 0
肥瘦 19
肥癡 0
肥皂 237
肥皂劇 28
肥皂水 12
肥皂沫 0
肥皂泡 5
肥皂液 4
肥皂液 4
肥皂盒 1
肥皂粉 24
肥的 58
肥短 3
肥矮 0
肥碩 4
肥缺 0
肥羊 6
肥美 20
肥肉 35
肥育 75
肥胖 492
肥胖症 27
肥膩 5
肥豬 15
肥貓 39
肥頭肥腦 0
肥飽 1
肥饒 0
肥鮮 0
肦 0
肧 0
肧 0
肨 0
肩 838
肩上 92
肩並肩 8
肩帶 30
肩窩 0
肩章 9
肩胛 29
肩胛骨 12
肩膀 267
肩膊 0
肩負 152
肩負著 25
肩負重任 2
肩起 2
肩輿 0
肩關節 38
肩靠肩 0
肩頭 70
肪 176
肪 176
肫 135
肬 0
肭 88
肮 666
肯 20
肯 2020
肯上 10
肯上來 0
肯上去 0
肯下 12
肯下來 0
肯下去 0
肯不肯 12
肯亞 90
肯作 6
肯來 9
肯做 23
肯再 8
肯出 19
肯出來 3
肯出去 0
肯切 2
肯到 1
肯去 13
肯回 5
肯回來 1
肯回去 1
肯在 21
肯坐 1
肯塔基 27
肯定 2897
肯定句 2
肯定性 8
肯定是 75
肯定會 23
肯定能 2
肯尼亞 3
肯尼迪 1
肯得 8
肯會 1
肯求 18
肯為 27
肯為 27
肯用 12
肯給 9
肯談 2
肯請 3
肯起來 0
肯載 0
肯進 7
肯進來 1
肯進去 1
肯過 0
肯過來 1
肯過去 0
肰 0
肱 125
肱臂 1
肱臂 1
肱骨 13
育 7784
育兒 144
育嬰 130
育嬰室 7
育嬰房 4
育嬰指南 2
育幼 127
育幼袋 0
育幼院 180
育成 520
育成雜交 0
育才 75
育林 73
育樂 1879
育樂中心 210
育秧 0
育種 286
育肥 0
育苗 160
育達 190
育雛 8
育養 3
育齡 18
肳 0
肴 121
肴饌 2
肵 112
肶 0
肷 0
肷 0
肸 19
肹 0
肺 11
肺 1184
肺出血 4
肺動脈 106
肺動脈 106
肺循環 5
肺氣腫 58
肺泡 73
肺活量 32
肺炎 249
肺病 40
肺癆 4
肺癌 248
肺結核 157
肺脹 0
肺腑 17
肺腑之言 16
肺腸 0
肺臟 137
肺葉 16
肺部 205
肺靜脈 20
肺靜脈 20
肼 1
肽 1
胂 1
胃 19032
胃下垂 10
胃出血 9
胃口 138
胃壁 15
胃液 22
胃液 22
胃液素 1
胃液素 1
胃潰瘍 87
胃灼熱 1
胃炎 69
胃病 44
胃痛 53
胃癌 180
胃石 0
胃科 2
胃科醫生 1
胃穿孔 3
胃脯 0
胃腔 0
胃腸 384
胃腸炎 29
胃腸科 109
胃腺 4
胃臟 4
胃藥 23
胃藥片 0
胃部 61
胃酸 103
胃酸過多 9
胃鏡 49
胃鏡室 1
胄 14
胄裔 0
胅 6
胆 0
胇 3
胈 2
胉 251
胊 116
胋 0
背 3000
背 2279
背上 179
背上來 1
背上去 0
背不動 3
背井離鄉 0
背人 0
背信 59
背信棄義 0
背債 2
背傷 8
背光 48
背光性 1
背出 9
背出 9
背出來 2
背出去 1
背到 4
背到 4
背包 321
背叛 236
背向 42
背向著 2
背囊 3
背回家去 0
背在 8
背地 9
背地裡 11
背子 1
背對 44
背對背 4
背對著 32
背山 23
背山面水 5
背帶 34
背影 147
背後 1226
背得 14
背得 75
背得 14
背得 76
背得起 0
背得起 0
背得起 0
背得起 0
背心 202
背成 0
背成 0
背斜 15
背日 0
背日性 0
背景 4701
背景資料卡 0
背書 299
背會 0
背棄 37
背椅 5
背水 5
背水一戰 23
背痛 154
背的 75
背的 75
背盟 0
背約 1
背脊 47
背脊骨 2
背著 160
背著 160
背著手 1
背著手 1
背袋 35
背誦 112
背誦著 1
背謬 0
背負 171
背負著 65
背起 49
背起 49
背起來 6
背起來 6
背逆 7
背運 1
背道而馳 28
背部 277
背離 85
背靠 14
背靠背 3
背面 386
背革裝 0
背風 12
背風處 0
背風面 7
背鰭 59
背黑鍋 10
胍 67
胎 950
胎位 21
胎兒 815
胎動 18
胎座 1
胎教 29
胎死腹中 35
胎毒 2
胎毛筆 1
胎氣 1
胎生 33
胎生動物 2
胎生學 1
胎盤 240
胎盤癌 0
胎藏界 1
胎記 17
胏 87
胐 8
胑 5
胒 0
胓 0
胓 0
胔 4
胕 102
胕 102
胖 960
胖 960
胖子 109
胖得 10
胖得 99
胖的 99
胖胖 180
胖胖的 47
胖胖地 46
胖虎 1
胗 55
胘 3
胙 5
胚 471
胚乳 32
胚囊 8
胚子 20
胚層 5
胚根 15
胚珠 10
胚胎 285
胚胎學 20
胚芽 52
胚芽米 16
胚軸 15
胚體 10
胛 7
胛骨 1
胜 216
胜肽 1
胝 41
胞 699
胞兄 6
胞兄弟 1
胞叔 0
胞囊 2
胞妹 3
胞姊 3
胞子 17
胞弟 32
胞衣 2
胟 0
胠 9
胡 4592
胡亂 92
胡人 13
胡佛 128
胡作非為 17
胡來 5
胡吃悶睡 0
胡同 46
胡同 46
胡塗 14
胡夢卜 0
胡子 45
胡子 45
胡床 0
胡志明 43
胡志明市 65
胡思亂想 133
胡扯 18
胡搞 70
胡攪 3
胡服 1
胡桃 70
胡桃鉗 18
胡桐 0
胡梯 0
胡椒 74
胡椒粉 37
胡椒鹽 1
胡樂 1
胡海 20
胡漢民 1
胡為 25
胡琴 59
胡瓜 114
胡笙 66
胡笳 4
胡纏 0
胡蘿蔔 88
胡蘿蔔汁 3
胡蘿蔔素 61
胡虜 1
胡蜂 7
胡裡胡塗 2
胡言 7
胡言亂語 45
胡話 2
胡說 44
胡說八道 90
胡謅 16
胡豆 1
胡適 291
胡適之 9
胡鬧 22
胡麻 22
胢 0
胢 0
胣 2
胤 50
胥 116
胦 71
胭 10
胭粉 1
胭脂 105
胭脂粉 1
胭脂紅 2
胭脂花粉 1
胭脂虎 0
胮 0
胯 46
胯下 0
胯下之辱 0
胯部 0
胯骨 0
胰 158
胰子 0
胰島素 528
胰島腺 1
胰液 5
胰液 5
胰腺 3
胰臟 184
胰臟炎 85
胱 145
胱氨酸 2
胲 13
胳 44
胳肢 0
胳肢窩 4
胳膀 2
胳膊 31
胳膊窩 0
胳臂 15
胴 42
胴體 31
胵 7
胶 0
胶 0
胷 0
胸 1209
胸中 64
胸像 2
胸口 171
胸圍 54
胸廓 15
胸式呼吸 2
胸懷 117
胸懷大志 9
胸懷故國 0
胸懷磊落 0
胸有成竹 43
胸椎 18
胸無城府 0
胸無大志 1
胸無宿物 0
胸無點墨 1
胸甲 15
胸章 8
胸管 5
胸罩 159
胸羅萬象 0
胸肋膜 1
胸肌 16
胸脊 1
胸脯 24
胸腔 693
胸腹 30
胸腺 29
胸膛 108
胸臆 10
胸衣 13
胸襟 122
胸部 476
胸部手術 1
胸針 19
胸飾 1
胸骨 40
胸鰭 23
胹 5
胺 731
胺 731
胺基 92
胺基酸 120
胻 13
胼 23
胼手胝足 41
胼胝 6
能 46954
能上 135
能上能下 0
能不 325
能不能 874
能事 30
能人 29
能人所不能 0
能仁 14
能伸 6
能伸能屈 1
能來 136
能借 23
能免 19
能免則免 7
能兼 11
能劇 1
能力 11032
能動 69
能動性 5
能去 179
能取 15
能吃 116
能吃能喝 0
能否 1824
能士 1
能夠 7641
能守 16
能守能攻 0
能寫 55
能寫能聽 0
能寫能說 0
能寫能講 0
能寫能讀 0
能將 1048
能將 1048
能屈能伸 6
能工巧匠 0
能帶 123
能幹 59
能忍則安 0
能手 17
能接 38
能攻 2
能攻能守 2
能文能武 1
能有 1405
能樂 1
能歌善舞 5
能減 16
能源 3525
能源危機 22
能源部 35
能演 17
能演能唱 0
能為 828
能為 828
能留 41
能留則留 0
能看 315
能看能寫 0
能看能聽 0
能看能說 0
能看能講 0
能知 49
能知天下事 24
能級 5
能結 20
能者 32
能者多勞 2
能耐 82
能聽 133
能聽能寫 0
能聽能看 0
能聽能說 0
能聽能講 0
能聽能讀 0
能見度 80
能解 54
能言善辯 1
能言善道 18
能言快語 0
能說 400
能說善道 2
能說能寫 1
能說能聽 1
能說能講 0
能說能讀 0
能講 37
能講能寫 0
能講能看 0
能講能聽 0
能講能讀 0
能讀 46
能讀能寫 1
能讀能聽 1
能讀能說 0
能讀能講 0
能趕 20
能跑 54
能跑能跳 0
能退 9
能退能進 0
能進 139
能進能退 0
能量 1642
能量不滅 3
能量不滅定律 2
能量守恆定律 1
能降 11
能降 11
能階 40
胾 6
胿 0
胿 0
胿 0
脀 4
脁 5
脂 1178
脂油 7
脂溶劑 0
脂粉 25
脂粉未施 0
脂粉氣 1
脂肪 909
脂肪性 3
脂肪組織 28
脂肪腺 0
脂肪質 2
脂肪酸 210
脂肪體 0
脂蛋白 101
脂酸 35
脃 0
脄 0
脅 158
脅制 0
脅從 1
脅持 2
脅肩 0
脅迫 124
脅逼 0
脆 262
脆弱 301
脆弱不堪 0
脆性 8
脆皮 24
脆餅 10
脇 0
脇 0
脈 1022
脈 1022
脈動 339
脈動 339
脈動電流 1
脈動電流 1
脈岩 1
脈岩 1
脈息 4
脈息 4
脈搏 133
脈搏 133
脈沖 27
脈沖 27
脈沖雷達 0
脈沖雷達 0
脈波 93
脈波 93
脈波 93
脈波 93
脈理 2
脈理 2
脈石 4
脈石 4
脈礦 0
脈礦 0
脈管 10
脈管 10
脈絡 243
脈絡 243
脈絡可循 5
脈絡可循 5
脈脈 7
脈脈含情 1
脈衝 165
脈衝 165
脈衝數 0
脈衝數 0
脈衝雷達 0
脈衝雷達 0
脈門 5
脈門 5
脉 0
脉 0
脊 318
脊 318
脊柱 52
脊柱 52
脊梁 3
脊梁 3
脊梁背 0
脊梁背 0
脊梁骨 0
脊梁骨 0
脊椎 525
脊椎 525
脊椎動物 133
脊椎動物 133
脊椎神經 2
脊椎神經 2
脊椎骨 34
脊椎骨 34
脊樑 6
脊樑 6
脊神經 5
脊神經 5
脊索 10
脊索 10
脊背 12
脊背 12
脊骨 17
脊骨 17
脊髓 409
脊髓 409
脊髓炎 8
脊髓炎 8
脔 0
脔 0
脕 10
脖 32
脖子 258
脖頸 5
脖頸子 0
脗 0
脘 11
脙 5
脚 0
脚 0
脛 42
脛骨 48
脜 0
脝 1
脞 2
脟 3
脠 0
脡 7
脢 106
脣 84
脣亡齒寒 0
脣吻 1
脣形 0
脣槍舌劍 0
脣紅齒白 0
脣脂 0
脣膏 1
脣舌 0
脣齒 0
脣齒相依 0
脤 120
脥 3
脦 0
脧 4
脨 0
脩 102
脩脯 0
脫 1607
脫下 187
脫下來 19
脫不了 38
脫了 60
脫位 19
脫來脫去 0
脫俗 71
脫光 61
脫兔 0
脫出 41
脫出重圍 1
脫到 5
脫卸 3
脫去 53
脫口 45
脫口成章 0
脫口秀 1
脫口而出 33
脫售 1
脫字 0
脫屣 0
脫帽 73
脫帽致敬 1
脫得 4
脫手 74
脫掉 62
脫星 17
脫期 1
脫模 14
脫殼 24
脫毛 58
脫毛劑 0
脫氧 6
脫氧核糖 0
脫氫 17
脫水 200
脫水機 24
脫泡 2
脫漏 1
脫熲而出 0
脫班 3
脫產 34
脫皮 62
脫盡 1
脫硝 14
脫硫 34
脫碳 3
脫稿 5
脫穎而出 135
脫空 0
脫節 83
脫籍 0
脫籠 0
脫粒 7
脫粒機 1
脫罪 14
脫胎 19
脫胎換骨 86
脫脂 61
脫脂乳 1
脫脂奶粉 23
脫脂棉 4
脫脫 14
脫腥 0
脫腸 5
脫膠 4
脫臭 37
脫臭劑 1
脫臼 78
脫色 31
脫色劑 0
脫落 252
脫落下來 0
脫著 10
脫衣 40
脫衣服 21
脫衣舞 24
脫衣舞孃 27
脫誤 0
脫起 1
脫起來 1
脫身 57
脫身而出 3
脫軌 17
脫軌而出 0
脫逃 52
脫過 2
脫銷 0
脫開 12
脫除 22
脫隊 13
脫險 22
脫離 850
脫離關係 3
脫離險境 11
脫靴 6
脫鞋 11
脫韁之馬 2
脫韁野馬 4
脫髮 23
脫髮劑 0
脫鹽 3
脫黨 17
脬 4
脭 3
脮 0
脯 129
脯 129
脯醢 0
脰 114
脱 0
脱 0
脲 1
脴 0
脹 556
脹大 15
脹氣 45
脹滿 5
脺 4
脺臟 0
脻 0
脼 0
脽 2
脾 359
脾氣 275
脾氣壞 1
脾氣好 4
脾胃 54
脾臟 88
脿 0
腀 0
腁 0
腂 0
腃 5
腄 269
腄 269
腅 0
腆 84
腆顏 0
腆默 0
腇 1
腈 0
腉 0
腊 75
腊 75
腊味 0
腊筆 2
腊肉 2
腋 89
腋 89
腋下 67
腋下 67
腋毛 8
腋毛 8
腋窩 12
腋窩 12
腋窩下 0
腋窩下 0
腋臭 0
腋臭 0
腋芽 1
腋芽 1
腌 64
腌 64
腌臢 0
腍 3
腎 1783
腎上腺 221
腎上腺素 63
腎囊 2
腎小球 32
腎小管 46
腎炎 110
腎病 132
腎癌 2
腎盂炎 3
腎結核 1
腎結石 39
腎臟 947
腎臟炎 12
腎臟病 151
腎臟科 126
腎蕨 1
腎虛 11
腎虧 15
腎衰竭 1
腎部 3
腏 2
腏 2
腐 257
腐乳 1
腐儒 3
腐刑 0
腐化 83
腐壞 19
腐惡 0
腐敗 217
腐敗無能 4
腐朽 42
腐朽不堪 0
腐植土 7
腐植質 13
腐殖質 0
腐氣 0
腐熟 15
腐爛 63
腐爛不堪 0
腐生 9
腐肉 9
腐臭 1
腐臭味 4
腐舊 0
腐蝕 361
腐蝕劑 7
腐蝕化 0
腐蝕性 119
腐蟻叢生 0
腑 121
腑臟 0
腒 2
腓 180
腓力二世 0
腓尼基 4
腓尼基人 3
腓骨 6
腔 11397
腔兒 0
腔兒 0
腔壁 2
腔腸 13
腔調 54
腕 1288
腕力 14
腕力 14
腕尺 0
腕尺 0
腕足 15
腕足動物 0
腕部 22
腕骨 10
腕骨 10
腕龍 4
腘 0
腛 8
腜 14
腝 0
腝 0
腝 0
腝 0
腝 0
腞 2
腞 2
腟 0
腠 8
腡 14
腢 5
腣 0
腤 5
腥 79
腥味 32
腥氣 0
腥污 0
腥羶 3
腥腥 1
腥膩 0
腥臊 0
腥臭 15
腥臭味 3
腥風血雨 10
腥黑穗病 0
腦 5702
腦中 499
腦中風 1
腦充血 5
腦兒 4
腦兒 4
腦出血 30
腦力 136
腦力勞動 1
腦力激盪 121
腦勺子 0
腦垂體 6
腦子 116
腦子裡 42
腦學 2
腦室 273
腦後 86
腦殼 4
腦殼子 0
腦汁 10
腦波 133
腦波 133
腦海 197
腦海中 157
腦液 1
腦液 1
腦溢血 25
腦滿腸肥 3
腦漿 12
腦炎 175
腦瘤 107
腦神經 202
腦筋 481
腦脹 3
腦膜 53
腦膜炎 136
腦蓋骨 0
腦袋 332
腦袋兒 1
腦袋兒 1
腦袋瓜 22
腦袋瓜兒 0
腦袋瓜兒 0
腦袋瓜子 16
腦貧血 2
腦部 475
腦量 51
腦門 21
腦震盪 17
腦顱 0
腦髓 4
腧 12
腨 0
腩 18
腪 0
腪 0
腫 583
腫大 246
腫大症 0
腫瘤 1848
腫脹 200
腫腫的 4
腫起來 5
腬 0
腭 0
腮 70
腮幫子 4
腮紅 21
腮腺炎 52
腯 0
腰 1812
腰刀 5
腰力 15
腰包 27
腰圍 47
腰子 9
腰巾 2
腰布 2
腰帶 113
腰斬 41
腰板 0
腰果 15
腰桿 8
腰桿子 1
腰眼 5
腰纏 1
腰纏萬貫 1
腰肢 15
腰背 48
腰胯 2
腰花 8
腰裡 0
腰身 23
腰部 151
腰酸 17
腰酸背疼 1
腰酸背痛 69
腰金拖紫 1
腰金衣紫 1
腰金衣紫 1
腰際 31
腰骨 1
腰鼓 4
腱 76
腲 4
腳 1
腳 3191
腳上 83
腳下 147
腳丫 17
腳丫子 9
腳伕 0
腳兒 2
腳兒 2
腳凳 0
腳力 13
腳印 281
腳型 17
腳夫 4
腳尖 47
腳底 111
腳底抹油 3
腳底板 8
腳底板兒 0
腳底板兒 0
腳形 6
腳後跟 14
腳心 14
腳手架 0
腳指甲 2
腳掌 72
腳本 125
腳板 15
腳架 70
腳步 943
腳步聲 85
腳氣 3
腳氣病 9
腳狀 0
腳痛 14
腳痛醫腳 5
腳登子 0
腳程 22
腳背 15
腳脖子 0
腳脛 0
腳色 19
腳註 3
腳趾 105
腳趾甲 11
腳趾頭 4
腳跟 62
腳跡 1
腳踏 157
腳踏兩條船 26
腳踏實地 78
腳踏板 27
腳踏車 706
腳踝 53
腳踵 2
腳鐐 10
腳鐐手銬 2
腳面 1
腳麻 10
腴 143
腵 0
腶 7
腷 61
腸 2131
腸子 61
腸炎 118
腸病毒 1
腸管 25
腸線 7
腸肚 1
腸胃 551
腸胃炎 62
腸胃病 12
腸間膜 0
腸骨 9
腹 6593
腹囊 0
腹地 79
腹壁 21
腹心 1
腹板 1
腹案 41
腹水 77
腹瀉 248
腹瀉藥 0
腹痛 193
腹稿 1
腹笥 2
腹笥便便 0
腹笥甚窘 2
腹肌 29
腹背 8
腹背受敵 13
腹脹 55
腹腔 285
腹膜 80
腹膜炎 30
腹語 5
腹足動物 0
腹部 587
腹面 41
腺 450
腺嘌呤 1
腺毛 5
腺癌 66
腽 0
腽 0
腿 1038
腿上 75
腿傷 5
腿力 3
腿子 2
腿毛 13
腿短 3
腿肉 36
腿肚 3
腿腳 0
腿酸 5
腿長 12
膀 94
膀 94
膀子 8
膀胱 416
膀胱炎 48
膀臂 4
膁 0
膁 0
膁 0
膂 5
膂力 6
膂力過人 1
膃 146
膄 0
膅 0
膆 836
膇 497
膈 104
膈膜 3
膉 2297
膊 1
膊 14
膋 230
膌 216
膍 535
膎 0
膏 327
膏丸 1
膏乳 0
膏土 0
膏沃 0
膏油 3
膏澤 1
膏火 0
膏燭 0
膏狀 4
膏肓 2
膏腴 2
膏腴之地 0
膏藥 20
膐 0
膓 0
膔 0
膕 11
膖 0
膗 3
膘 468
膘情 0
膙 5
膚 3980
膚如凝脂 0
膚泛 0
膚淺 80
膚淺性 0
膚色 154
膛 527
膜 3136
膜 3136
膜拜 66
膜片 21
膜片 21
膜狀 10
膜狀 10
膜骨粉 0
膜骨粉 0
膝 1016
膝上型 15
膝下 14
膝窩 1
膝蓋 170
膝蓋骨 9
膝行 0
膝踝 0
膝部 56
膝關節 177
膝頭 3
膞 8
膟 8
膠 17400
膠乳 2
膠卷 14
膠原蛋白 1
膠合 55
膠合板 11
膠囊 269
膠固 4
膠圈 7
膠塊 0
膠州 0
膠州灣 6
膠布 66
膠布膏 0
膠帶 387
膠底鞋 1
膠木 3
膠東 6
膠柱鼓瑟 1
膠水 81
膠泥 44
膠漆 22
膠片 64
膠版 0
膠版印刷 1
膠版紙 0
膠狀 23
膠瑟 0
膠皮 32
膠盔 1
膠粒 93
膠紙 6
膠結 23
膠結劑 0
膠膜 99
膠著 78
膠著狀態 10
膠西丘陵 1
膠質 66
膠鞋 18
膠體 60
膠體溶液 4
膠體溶液 4
膡 0
膡 0
膢 5
膣 15
膦 24
膧 50
膨 404
膨大 34
膨脹 382
膨脹係數 1
膨脹起來 3
膩 335
膩了 31
膩友 1
膩味 1
膩在一起 5
膩在一起 5
膩子 0
膩滑 4
膩煩 0
膩胃 0
膩著 1
膪 0
膫 22
膬 15
膭 0
膭 0
膮 5
膰 179
膱 65
膲 18
膳 383
膳堂 0
膳夫 0
膳宿 38
膳宿費 4
膳宿費用 1
膳房 12
膳羞 0
膳費 0
膳雜 0
膳雜費 7
膳食 341
膳食費 12
膳食費用 2
膴 19
膴 19
膵 8
膷 19
膸 0
膸 0
膹 16
膺 102
膺任 10
膺品 8
膺選 48
膻 108
膼 134
膽 759
膽力 1
膽囊 110
膽囊炎 21
膽固醇 525
膽大 14
膽大包天 44
膽大妄為 4
膽大心細 6
膽子 62
膽寒 8
膽小 70
膽小如鼠 2
膽小鬼 66
膽怯 37
膽戰心驚 12
膽敢 28
膽汁 98
膽瓶 2
膽略 1
膽略過人 0
膽石 24
膽礬 0
膽管 71
膽結石 105
膽落 0
膽識 54
膽識過人 6
膽量 48
膽顫心驚 14
膾 15
膾炙 3
膾炙人口 73
膾脯 0
膾飯 0
膿 242
膿包 7
膿汁 4
膿疱 0
膿病 0
膿痰 2
膿瘡 6
膿皰 0
膿腫 19
臀 175
臀疣 0
臀部 189
臀鰭 12
臁 0
臂 467
臂 467
臂力 15
臂助 7
臂彎 7
臂章 29
臂膀 50
臂膊 2
臃 7
臃 7
臃腫 21
臄 37
臅 5
臆 32
臆中 0
臆度 2
臆想 4
臆斷 0
臆測 82
臆造 0
臇 188
臈 0
臈 0
臉 1879
臉上 1065
臉上無光 2
臉兒 12
臉兒 12
臉型 59
臉嫩 0
臉子 0
臉孔 125
臉帶笑容 1
臉皮 70
臉皮厚 7
臉皮薄 2
臉盆 25
臉盆架 1
臉盤 2
臉硬 0
臉紅 105
臉紅脖子粗 4
臉罩 5
臉腫 5
臉色 326
臉色紅潤 1
臉色蒼白 48
臉蛋 74
臉譜 151
臉軟 0
臉部 426
臉面 10
臉面無光 0
臉頰 164
臉額 0
臉龐 147
臊 8
臊 8
臋 0
臌 30
臌脹 0
臍 233
臍帶 96
臍帶血 1
臎 0
臎 0
臏 263
臐 1081
臑 8
臒 5
臔 0
臕 69
臖 0
臗 79
臘 442
臘八 2
臘八粥 3
臘味 3
臘子口 0
臘斯克 0
臘日 0
臘月 9
臘染畫 0
臘梅 7
臘紙 3
臘肉 12
臘腸 11
臘腸狗 4
臘質 7
臘雪 0
臘鼓 0
臘鼓頻傳 0
臙 83
臚 742
臛 93
臝 14
臞 5
臟 336
臟器 83
臟腑 80
臠 9
臠 9
臡 2
臢 10
臣 1034
臣下 16
臣仕 0
臣僕 9
臣妾 7
臣子 17
臣子 17
臣服 27
臣服在 8
臣服於 12
臣民 8
臤 0
臤 0
臤 0
臥 1006
臥下 2
臥下來 0
臥下去 0
臥倒 0
臥倒在地 1
臥具 11
臥在 16
臥姿 20
臥室 233
臥射 0
臥床 67
臥底 51
臥得 6
臥房 110
臥榻 6
臥病 19
臥病在床 11
臥的 6
臥舖 4
臥著 9
臥薪嘗膽 4
臥虎藏龍 23
臥車 0
臥軌自殺 4
臥遊 33
臥鋪 6
臥鋪票 0
臥龍崗 153
臦 0
臦 0
臧 70
臧否 4
臨 2126
臨了 16
臨刑 1
臨別 26
臨別依依 4
臨別時 5
臨別贈言 4
臨到 1
臨危 24
臨危不亂 0
臨危不懼 0
臨危受命 6
臨危授命 4
臨危時 0
臨去 15
臨場 13
臨場 139
臨安 62
臨川四夢 1
臨帖 5
臨幸 3
臨床 4273
臨床實驗 60
臨床經驗 62
臨摹 30
臨時 2365
臨時代辦 0
臨時動議 157
臨時工 52
臨時性 99
臨時抱佛腳 13
臨時政府 11
臨時費用 1
臨期 3
臨末 1
臨機 0
臨機應變 4
臨檢 95
臨死 27
臨死前 29
臨死時 7
臨水 18
臨江 28
臨池 4
臨汾 4
臨沂 56
臨河 11
臨河地區 0
臨洮 1
臨海 236
臨海地區 1
臨深履薄 3
臨淵 1
臨淵履薄 1
臨淵羨魚 1
臨清 6
臨渴掘井 0
臨湖 2
臨湖地區 0
臨溪 32
臨產 3
臨產前 0
臨產時 1
臨界 220
臨界壓 0
臨界壓力 7
臨界溫度 33
臨界角 3
臨界點 47
臨畫 0
臨盆 15
臨盆時 1
臨眺 1
臨睡 6
臨睡前 6
臨睡時 0
臨終 148
臨終前 28
臨終時 16
臨行 39
臨街 26
臨視 0
臨走 19
臨走前 25
臨走時 10
臨近 116
臨門 60
臨陣 10
臨陣磨槍 6
臨陣磨鎗 0
臨陣脫逃 6
臨難 3
臨靠 0
臨靠著 0
臨頭 49
臨風 19
臩 23
自 29774
自上到下 1
自上而下 4
自下到上 0
自下而上 0
自不待言 9
自不量力 10
自主 1080
自主國 1
自主神經 29
自乘 2
自了 4
自交系 7
自介 1
自付 116
自以 159
自以為是 105
自任 29
自伐 1
自伐 1
自住 47
自住型 11
自作 22
自作主張 16
自作多情 16
自作孽 9
自作聰明 18
自作自受 9
自作自受 9
自來 127
自來水 1150
自來水廠 34
自來水筆 1
自來水龍頭 4
自來水龍頭 4
自供 16
自供狀 0
自侮 2
自便 18
自保 117
自信 696
自信心 118
自修 237
自修室 43
自修法 2
自修課 0
自個 47
自個兒 75
自備 551
自備午餐 0
自備晚餐 0
自傲 43
自傳 785
自傳體 4
自傷 25
自內 16
自內心 5
自內而外 0
自刎 13
自利 43
自到 16
自制 66
自制力 11
自剄 1
自創 298
自力 73
自力救濟 54
自力更生 36
自助 1240
自助助人 25
自助式 114
自助餐 284
自助餐廳 67
自動 11129
自動化 5500
自動增益 1
自動增益控制 3
自動式 27
自動性 0
自動控制 525
自動機 23
自動武器 2
自動生產 5
自動生產線 2
自動線 0
自動自發 90
自動裝置 2
自動詞 0
自動警報 67
自動鉛筆 8
自動門 60
自動電話 22
自動駕駛 52
自動駕駛儀 9
自勵 11
自卑 92
自卑感 27
自即日起 393
自反 4
自取 168
自取其辱 12
自取其辱 12
自取滅亡 3
自古 200
自古以來 196
自右而左 2
自吹 2
自吹自擂 15
自告奮勇 27
自命 15
自命不凡 6
自命清高 5
自命為 7
自咎 0
自問 49
自問自答 10
自喻 34
自嘆不如 8
自嘆弗如 8
自嘲 49
自噴 0
自噴井 0
自噴式 0
自圓其說 33
自在 747
自地自建 12
自報 19
自壓 3
自壓式 0
自外而內 0
自多 10
自大 114
自大狂 4
自奉 4
自奉甚儉 0
自好 3
自如 116
自始 65
自始至終 41
自娛 23
自娛娛人 12
自學 667
自學法 0
自定 184
自宮 1
自家 259
自家人 33
自封 12
自尊 166
自尊心 73
自尋 15
自尋死路 2
自尋煩惱 9
自導 30
自導自演 24
自小 0
自居 64
自左 13
自左而右 4
自己 34797
自己人 111
自帶 12
自幼 166
自序 87
自建 132
自強 1181
自強不息 14
自強號 132
自強運動 10
自彈 3
自彈自唱 17
自律 389
自律性 14
自得 105
自得其樂 22
自從 1265
自從在 28
自必 4
自忖 0
自性 1
自怨 4
自怨自艾 23
自恃 20
自愛 40
自愛自重 1
自感 10
自慊 0
自慚形穢 5
自慚形跡 0
自慰 163
自憐 28
自憐自艾 4
自應 156
自成 115
自成一家 10
自成一家 10
自成一格 27
自成一格 27
自成一體 1
自成一體 1
自我 4203
自我中心 46
自我介紹 1532
自我作古 0
自我嘲解 2
自我批評 2
自我檢討 35
自我犧牲 8
自我表現 11
自我陶醉 19
自戒 1
自戕 18
自找 21
自找苦吃 2
自找麻煩 17
自投 11
自投羅網 7
自拉自唱 1
自拍 19
自拍器 4
自拔 106
自持 28
自掃門前雪 11
自掏腰包 27
自排 688
自掘墳墓 6
自摸 1
自救 310
自敘 14
自斃 2
自新 39
自是 171
自暴自棄 23
自會 94
自會在 2
自有 855
自有主張 7
自有妙計 0
自有打算 0
自有辦法 2
自棄 10
自欺 21
自欺欺人 30
自歎不如 0
自歎弗如 0
自此 229
自此以後 21
自此刻起 0
自此時起 0
自此而後 1
自殘 14
自殺 839
自殺式 3
自殺未遂 11
自殺者 14
自毀前程 7
自毀長城 6
自民黨 204
自求 14
自求多福 64
自決 70
自決權 30
自治 785
自治制 1
自治區 122
自治州 21
自治市 15
自治市長 19
自治權 29
自治法 122
自治省 1
自治縣 38
自治體 14
自況 6
自派 0
自流 6
自流井 1
自流灌溉 1
自淨 33
自滿 97
自演 12
自為 25
自焚 73
自然 11290
自然主義 103
自然人 186
自然力 10
自然化 11
自然哲學 6
自然對數 0
自然律 14
自然捲 1
自然數 22
自然是 193
自然會 178
自然村 3
自然歷 1
自然水 2
自然法學 0
自然法學派 0
自然災害 43
自然物 10
自然現象 48
自然界 286
自然療法 58
自然礦 0
自然科 598
自然科學 1456
自然經濟 2
自然美 322
自然而然 174
自然觀 9
自然課 10
自然通風 9
自然選擇 4
自然金 1
自燃 13
自燃式 1
自燃性 2
自營 308
自營商 585
自營菌 0
自理 219
自甘 17
自甘墮落 12
自甘暴棄 0
自生 30
自生自滅 12
自產 56
自用 711
自用品 0
自用車 38
自由 10470
自由主義 128
自由化 1034
自由地區 42
自由基 181
自由女神 28
自由寶島 1
自由市 2
自由市場 65
自由市場 65
自由式 109
自由心証 0
自由心證 17
自由思想 5
自由性 6
自由態 2
自由戀愛 11
自由日 4
自由日報 1
自由業 192
自由權 53
自由民 4
自由派 39
自由港 24
自由王國 0
自由祖國 3
自由競爭 54
自由組合 11
自由組合規律 0
自由職業 91
自由自在 151
自由落體 36
自由落體運動 0
自由詩 4
自由貿易 128
自由貿易區 100
自由車 158
自由鍛 0
自由電子 18
自由面 0
自由體操 1
自由黨 85
自留 16
自留地 4
自畫像 54
自當 92
自發 105
自發性 175
自白 82
自白 82
自白書 10
自盡 51
自相 19
自相殘殺 31
自相矛盾 18
自省 94
自矜 1
自矜自伐 0
自矜自伐 0
自知 167
自知之明 19
自知理屈 0
自知理虧 4
自禁 7
自私 264
自私心 1
自私自利 26
自種 4
自稱 357
自稱是 79
自稱為 46
自立 511
自立早報 115
自立晚報 115
自立自強 52
自約 4
自組 88
自絕 15
自給 66
自給有餘 0
自給自足 76
自經 2
自編 108
自縊 14
自習 62
自習時間 6
自習課 7
自耕 92
自耕農 29
自肥 93
自能 40
自能在 1
自花傳粉 0
自花受粉 0
自若 16
自苦 5
自薦 11
自處 51
自行 6011
自行其是 4
自行決定 127
自行解決 22
自行車 1591
自衛 166
自衛隊 55
自裁 12
自製 734
自視 21
自視甚高 7
自覺 484
自覺自願 1
自覺運動 10
自解 47
自言自語 131
自訂 405
自討 4
自討沒趣 5
自討苦吃 6
自訟 0
自設 98
自許 73
自訴 46
自訴人 35
自詡 42
自誇 34
自認 325
自認是 24
自認會 0
自認為 147
自語 17
自誤 4
自誤誤人 0
自說自話 10
自調 8
自調 8
自調式 2
自謙 23
自變數 5
自變量 0
自豪 117
自負 133
自負盈虧 14
自責 96
自費 584
自費旅行 0
自費生 71
自賊 0
自賣 8
自賣自誇 3
自足 39
自足自給 1
自蹈法網 0
自身 958
自身難保 13
自載 0
自轉 105
自轉 105
自轉式 1
自轉軸 8
自辦 176
自辦式 27
自述 142
自遣 3
自選 148
自選動作 0
自選曲 11
自鄶以下 0
自重 133
自量 2
自閉 94
自閉性 0
自閉症 792
自需 2
自須 6
自願 442
自願性 56
自願書 1
自顧不暇 13
自食其力 8
自食其果 3
自食惡果 13
自養 3
自餒 0
自首 113
自首變節 0
自體 272
自體受精 0
自高 32
自高自大 6
自鳴得意 3
自鳴鐘 0
自點 0
自點曲 0
臫 0
臬 57
臭 104
臭 1047
臭事 1
臭人 2
臭人 2
臭包 1
臭吃臭喝 0
臭名 19
臭名昭著 1
臭味 137
臭味相投 8
臭小子 6
臭屁 97
臭得 25
臭架子 1
臭死 5
臭氣 45
臭氣沖天 2
臭氣薰天 1
臭氣衝天 1
臭氧 447
臭氧層 351
臭水溝 4
臭溝 1
臭的 24
臭皮 9
臭皮囊 7
臭罵 10
臭美 11
臭臭 88
臭蟲 113
臭豆腐 97
臭銅味 1
臭錢 1
臭頭 14
臭頭 14
臭鼠 0
臭鼬 14
臮 5
臯 0
臯 0
臰 0
臰 0
臱 0
臲 8
至 44970
至上 367
至中 219
至交 26
至人 41
至人無己 1
至今 2469
至今已是 4
至今已有 113
至元 26
至公 65
至友 10
至善 285
至多 225
至多是 4
至大 173
至大至剛 1
至好 3
至如 25
至孝 10
至寶 50
至尊 172
至尊無上 3
至少 4321
至少在 173
至少是 56
至少有 428
至德 69
至心 16
至急 4
至性 6
至情 13
至情至性 16
至意 5
至愛 105
至於 7916
至日 67
至極 77
至正 58
至此 444
至死 44
至死不屈 0
至死不悟 0
至死不渝 12
至死不變 1
至死不降 0
至為 141
至理 10
至理名言 9
至當 65
至矣盡矣 0
至福 4
至聖 71
至聖先師 23
至聖孔子 0
至若 13
至行 13
至行 13
至要 11
至親 28
至親好友 7
至言 0
至誠 97
至論 3
至道 12
至遲 69
至高 147
至高無上 41
致 6531
致仕 1
致使 663
致冷 24
致函 70
致力 661
致力於 1023
致勝 168
致命 635
致命傷 67
致命性 23
致命處 0
致哀 17
致奠 0
致密 2
致富 398
致富之道 5
致意 82
致敬 172
致書 14
致歉 135
致死 371
致死量 9
致理 24
致理商專 32
致理商業專科學校 42
致理技術學院 1
致用 351
致病 258
致癌 273
致癌物 68
致癌物質 33
致知 10
致詞 1139
致謝 230
致賀 21
致贈 239
致辭 115
致送 40
致電 60
臵 0
臶 0
臷 2
臸 4
臹 3
臺 9085
臺上 43
臺下 19
臺中 983
臺中商專 22
臺中市 597
臺中師範學院 125
臺中教育大學 1
臺中縣 498
臺北 6966
臺北人 7
臺北區 61
臺北商專 12
臺北大學 1
臺北工專 7
臺北工業專科學校 12
臺北市 3857
臺北市立 844
臺北市立師範學校 0
臺北市立教育大學 1
臺北市立體育學院 1
臺北師範學院 374
臺北師院 20
臺北橋 1
臺北盆地 16
臺北科技大學 1
臺北縣 1023
臺北護專 3
臺北護理專科學校 0
臺北醫學大學 1
臺北醫學院 288
臺北體專 0
臺北體院 1
臺南 874
臺南大學 1
臺南女子技術學院 1
臺南家專 7
臺南家政專科學校 1
臺南市 297
臺南師範學院 307
臺南科技大學 1
臺南縣 898
臺地 19
臺塑 55
臺大 1022
臺大醫院 155
臺幣 230
臺座 3
臺式 1
臺東 359
臺東大學 1
臺東市 57
臺東師範學院 139
臺東縣 337
臺柱 1
臺榭 1
臺汽 15
臺海 157
臺海兩岸 22
臺澎金馬 8
臺灣 15908
臺灣人 152
臺灣光復 50
臺灣光復節 18
臺灣大學 1548
臺灣大學附屬醫學院 0
臺灣工業技術學院 51
臺灣師範大學 666
臺灣教育學院 6
臺灣汽車客運公司 4
臺灣海峽 41
臺灣海洋大學 1
臺灣海洋學院 8
臺灣療養院 0
臺灣省 5407
臺灣省政府 1846
臺灣省立 1233
臺灣省議會 237
臺灣神學院 1
臺灣糖業公司 6
臺灣肥料公司 2
臺灣航業公司 1
臺灣藝術專科學校 19
臺灣銀行 71
臺灣電力公司 34
臺灣電視公司 3
臺球 1
臺省 52
臺端 0
臺糖 24
臺肥 2
臺航 2
臺西 1
臺西鄉 1
臺視 71
臺詞 6
臺語 1
臺車 2
臺鐵 0
臺階 7
臺電 81
臺面 9
臻 892
臻於完善 9
臼 67
臼杵 1
臼砲 2
臼齒 57
臽 0
臽 0
臾 7
臿 73
臿 73
舀 62
舀 62
舀子 0
舀水 1
舀湯 4
舁 11
舂 12
舂穀 0
舂米 5
舃 0
舃 0
舄 4
舅 858
舅兄 0
舅公 2
舅大人 0
舅姑 1
舅媽 7
舅子 0
舅母 4
舅父 8
舅爺 8
舅舅 81
與 21354
與 539
與世俯仰 0
與世推移 1
與世浮沉 2
與世無爭 35
與世長辭 12
與世隔絕 28
與人 1538
與人方便 0
與人為善 16
與共 96
與其 3115
與否 800
與國 973
與日 716
與日俱增 112
與會 917
與會人士 92
與格 7
與此同時 49
與民爭利 7
與生俱來 106
與眾不同 243
與聞 3
與與 38
與虎謀皮 5
與賽 57
興 1734
興 1734
興中會 12
興之所至 6
興亡 26
興亡史 6
興修 26
興兵 17
興利除弊 7
興匆匆 3
興化 37
興味 28
興味索然 0
興國 481
興奮 966
興奮 966
興奮劑 35
興奮劑 35
興學 198
興安 186
興安嶺 2
興安省 1
興師 5
興師動眾 3
興師問罪 2
興廢 5
興建 3081
興建中 95
興復 2
興德 117
興旺 73
興替 10
興歎 8
興沖沖 29
興滅 1
興滅繼絕 0
興盛 139
興築 21
興緻 28
興緻勃勃 16
興致 71
興致勃勃 44
興舉 2
興落 2
興衰 84
興衰史 13
興訟 11
興許 1
興起 725
興起來 4
興趣 6750
興趣不大 5
興趣正濃 0
興趣缺缺 34
興辦 214
興辦事業 21
興邦 44
興隆 199
興隆路 169
興革 110
興頭 2
興頭上 1
興頭很大 0
興風作浪 20
興高采烈 69
舉 3265
舉一反三 14
舉一廢百 0
舉世 140
舉世公認 6
舉世無匹 2
舉世無敵 0
舉世無雙 18
舉世矚目 13
舉世聞名 51
舉了 27
舉事 9
舉人 30
舉例 397
舉例來說 224
舉例說明 109
舉債 58
舉兵 9
舉凡 470
舉出 180
舉出來 1
舉動 318
舉國 30
舉國上下 8
舉國之力 1
舉國歡騰 8
舉好 1
舉家 34
舉手 155
舉手之勞 67
舉手投足 39
舉手敬禮 1
舉手禮 2
舉手表決 5
舉措 32
舉措失當 0
舉旗 7
舉有 9
舉杯 56
舉杯暢飲 0
舉案齊眉 1
舉棋不定 22
舉槍 31
舉止 115
舉止大方 1
舉止行為 2
舉止言談 1
舉步 52
舉步維艱 11
舉火 7
舉為 2
舉燭 0
舉用 2
舉發 241
舉目 39
舉目可見 4
舉目無親 4
舉目遠眺 0
舉直 2
舉箸 1
舉著 28
舉薦 3
舉行 5657
舉要 9
舉證 0
舉起 209
舉起來 15
舉起手來 4
舉足 17
舉足輕重 131
舉辦 7776
舉辦活動 72
舉過 3
舉過來 0
舉過去 2
舉重 143
舉重選手 11
舉錯 0
舉頭 23
舊 32274
舊中國 3
舊事 41
舊事重提 2
舊交 3
舊人 36
舊作 26
舊例 2
舊俗 6
舊債 1
舊到 0
舊制 75
舊創 3
舊友 11
舊唯物主義 0
舊國 9
舊地 30
舊地重遊 17
舊址 82
舊夢重溫 0
舊大陸 2
舊好 3
舊學 17
舊家 21
舊居 18
舊屬 2
舊市 65
舊帳 7
舊年 5
舊庄 40
舊廠 19
舊式 109
舊得 672
舊思想 0
舊恨 28
舊恨新仇 0
舊恨新愁 0
舊恩 0
舊情 70
舊情綿綿 26
舊惡 0
舊愛新歡 20
舊戲 8
舊故 1
舊教 14
舊教徒 1
舊日 40
舊時 72
舊時代 27
舊曆 22
舊曆年 10
舊書 43
舊書店 10
舊書攤 4
舊有 21
舊期 3
舊期生 0
舊校 31
舊案 10
舊業 2
舊歡 1
舊歷 3
舊民主主義革命 0
舊法 37
舊版 0
舊物 60
舊班 0
舊生 75
舊疾復發 1
舊病 6
舊病復發 5
舊的 672
舊知 1
舊石器 3
舊石器時代 27
舊稱 47
舊站 22
舊約 357
舊約全書 4
舊習 7
舊聞 86
舊聞新知 0
舊舊 13
舊莊 21
舊莊國小 4
舊著 0
舊藥 4
舊衣 76
舊衣服 8
舊觀 21
舊言 0
舊詩 8
舊話重提 3
舊調重彈 2
舊識 0
舊貨 157
舊貨攤 1
舊賬 0
舊車 46
舊道德 1
舊部 15
舊都 10
舊鄉 2
舊金山 654
舊金山市 11
舊金山灣 17
舊雨新知 112
舋 120
舌 450
舌頭 100
舌上 5
舌下 22
舌乾脣焦 0
舌前 1
舌劍唇槍 1
舌劍脣鎗 0
舌尖 37
舌尖音 0
舌後 1
舌戰 31
舌敝脣焦 0
舌根 8
舌根音 0
舌炎 2
舌燦蓮花 3
舌狀 9
舌癌 3
舌端 0
舌苔 20
舌葉 0
舌識 1
舌鋒 0
舌門 0
舌面 5
舌面音 0
舌音 30
舌頭 283
舌骨 4
舍 150
舍 1508
舍下 5
舍不得 5
舍人 8
舍利 200
舍利佛 2
舍利子 32
舍妹 4
舍姪 0
舍弟 7
舍得 4
舍我其誰 1
舍我其誰 1
舍棄 1
舍監 24
舍親 1
舍身 2
舍間 3
舎 0
舎 0
舏 0
舐 52
舐犢情深 1
舑 2
舒 1605
舒伯特 152
舒卷 3
舒坦 13
舒展 49
舒張 36
舒張壓 46
舒心 4
舒懷 2
舒捲 1
舒捲自如 0
舒暢 132
舒曼 106
舒服 750
舒泰 22
舒活筋骨 4
舒潔 14
舒眉 0
舒筋活血 22
舒緩 87
舒適 1359
舒閒 0
舓 0
舔 388
舔乾淨 1
舔掉 1
舔著 17
舔食 2
舕 223
舖 1285
舖 1285
舖下 3
舖下 3
舖位 0
舖墊 3
舖子 10
舖平 6
舖張 2
舖張浪費 1
舖成 5
舖排 4
舖蓋 3
舖設 122
舖軌 0
舘 0
舙 0
舚 0
舛 24
舛誤 2
舛謬 0
舛錯 0
舛雜 0
舛駁 0
舜 1172
舝 201
舞 3481
舞伴 12
舞出 52
舞刀弄槍 0
舞劇 78
舞動 54
舞動著 7
舞台 1665
舞台劇 250
舞團 0
舞場 38
舞場 38
舞女 14
舞姿 33
舞娘 1
舞孃 24
舞客 2
舞廳 168
舞弄 2
舞弄文墨 0
舞弄文墨 0
舞弄著 1
舞弊 116
舞態 0
舞技 9
舞拳 16
舞文弄墨 2
舞文弄法 0
舞文弄法 0
舞星 8
舞曲 348
舞會 432
舞棍 1
舞步 62
舞池 40
舞獅 104
舞獅隊 13
舞王 5
舞男 87
舞者 0
舞臺 249
舞臺劇 14
舞臺戲 0
舞臺效果 1
舞臺藝術 3
舞藝 24
舞藝超群 0
舞起 5
舞起來 2
舞蹈 1964
舞蹈家 25
舞蹈片 2
舞蹈社 56
舞蹈系 78
舞迷 0
舞鞋 8
舞龍 54
舞龍競賽 0
舞龍隊 5
舟 566
舟山 48
舟山群島 4
舟楫 10
舟艦 0
舟車 20
舟車 20
舟車勞頓 16
舠 78
舡 136
舡魚 0
舢 27
舢板 3
舢舨 54
舤 0
舥 4
舦 0
舧 0
舨 21
舩 0
航 4839
航務 127
航向 176
航器 2
航圖 90
航天 47
航太 0
航太總署 3
航委會 2
航廈 100
航政 137
航政司 21
航政局 16
航業 136
航業公司 8
航標 1
航權 28
航次 609
航海 470
航海圖 22
航海家 39
航海日誌 9
航海曆 0
航海機 0
航班 0
航界 3
航程 801
航空 9362
航空信 0
航空公司 2114
航空器 426
航空圖 3
航空學 5
航空工程 87
航空暨太空總署 6
航空業 73
航空母艦 107
航空法 45
航空界 22
航空站 185
航空網 11
航空艦 0
航空運輸 91
航空郵件 12
航空隊 24
航管局 0
航線 918
航至 4
航船 11
航行 432
航行權 0
航行燈 0
航警局 0
航路 262
航速 28
航運 3833
航運公司 37
航運界 2
航道 172
航點 35
舫 193
舫舟 0
舫船 2
舫艇 0
般 44
般 443
般 4430
般舟三昧 1
般般 0
般若 431
般若堂湯 0
般若波羅蜜多 1
般若波羅蜜多 1
般若波羅蜜多心經 1
般若波羅蜜多心經 1
般若經 40
般配 9
舭 0
舯 14
舲 15
舳 10
舳艫 0
舴 3
舴艋 1
舵 139
舵手 39
舵柄 0
舵機 1
舶 88
舶來品 16
舶來貨 0
舷 82
舸 5
船 3759
船主 25
船伕 8
船位 13
船兒 1
船兒 1
船到橋頭 4
船到橋頭自然直 11
船務 868
船台 31
船名 541
船員 507
船員表 0
船型 26
船埠 0
船塢 37
船夫 27
船客 0
船家 9
船尾 40
船工 5
船底 24
船廠 34
船形 5
船戶 4
船期 166
船期表 11
船東 26
船板 7
船柱 0
船桅 8
船梢 0
船槳 4
船歌 6
船殼 18
船票 22
船篷 0
船籍 35
船腳 0
船舵 3
船舶 1701
船舶學 0
船舷 7
船艄 0
船艙 63
船艦 92
船貨 5
船費 8
船身 69
船運 45
船過水無痕 8
船邊 27
船長 411
船長室 4
船門 0
船閘 2
船隊 68
船隻 281
船頭 75
船首 24
船體 56
船齡 5
舺 17
舺 17
舼 1
舽 6
舿 1
艀 120
艁 0
艁 0
艂 46
艃 0
艄 67
艅 49
艆 0
艇 438
艇員 4
艇長 7
艉 248
艊 0
艋 18
艋舺 142
艌 0
艌 0
艎 1
艏 69
艐 4
艑 4
艒 9
艓 11
艕 4
艖 12
艗 0
艘 1520
艙 1013
艙位 18
艙口 20
艙口處 0
艙室 0
艙底 1
艙房 24
艙門 17
艙門口 0
艚 22
艛 8
艜 23
艞 396
艟 1071
艡 48
艢 0
艣 21
艤 47
艥 0
艦 2048
艦上 46
艦塔 0
艦尾 42
艦砲 2
艦艇 226
艦長 150
艦隊 410
艦隻 4
艦首 4
艧 0
艨 79
艨艟 1
艨衝 1
艩 90
艩 90
艪 0
艫 10
艬 9
艭 146
艮 57
艮 57
良 6058
良久 64
良人 14
良伴 38
良兵 0
良友 55
良吏 1
良善 88
良善風俗 0
良士 4
良多 184
良夜 4
良好 3881
良宵 9
良家 11
良家女子 0
良家婦女 10
良家子 1
良將 14
良導體 3
良師 42
良師益友 88
良師興國 9
良幣 18
良心 341
良心不安 7
良心何在 6
良心發現 10
良心話 1
良心譴責 2
良性 498
良性瘤 1
良方 84
良方妙計 0
良日 7
良朋 6
良朋益友 1
良材 10
良機 213
良機不再 0
良機勿失 1
良民 10
良民證 22
良深 7
良港 7
良率 0
良田 29
良相 1
良相佐國 0
良知 105
良知良能 7
良禽擇木 0
良禽擇木而棲 2
良種 4
良窳 78
良策 34
良緣 37
良能 19
良莠不齊 34
良藥 42
良藥苦口 2
良計 1
良賤 1
良辰 20
良辰吉日 5
良辰美景 4
良醫 28
良馬 10
艱 100
艱危 2
艱困 80
艱巨 14
艱深 42
艱深難懂 5
艱澀 48
艱苦 124
艱苦備嚐 1
艱苦奮鬥 10
艱貞 0
艱辛 233
艱鉅 78
艱阨 0
艱險 8
艱難 167
艱難困苦 1
艱難度日 0
艱難曲折 0
艱難險阻 2
色 8395
色不迷人 0
色不迷人人自迷 3
色光 59
色即是空 10
色厲內荏 1
色塵 1
色子 7
色差 80
色帶 114
色度 65
色度計 4
色彩 2416
色彩學 47
色彩繽紛 41
色彩艷麗 2
色彩鮮明 18
色彩鮮艷 12
色情 773
色情小說 6
色情片 5
色情狂 6
色情行業 34
色情讀物 1
色情雜誌 5
色情電影 16
色慾 15
色散 50
色料 46
色楞格河 1
色樣 14
色溫 1
色澤 258
色澤鮮明 2
色狼 93
色球 14
色界 1
色當 2
色目人 3
色盲 44
色盲症 0
色相 123
色筆 10
色筆套 0
色系 133
色紙 50
色素 381
色素體 1
色膽包天 2
色藝 9
色蘊 1
色調 239
色迷迷 1
色香 21
色鬼 52
艴 3
艴然 0
艵 3
艶 0
艷 350
艷事 1
艷史 0
艷姿 2
艷情 3
艷歌 2
艷福 0
艷福不淺 0
艷稱 0
艷羨 4
艷舞 5
艷色 6
艷若桃李 0
艷詩 1
艷遇 6
艷陽 36
艷陽天 5
艷陽高照 6
艷麗 35
艸 10
艹 0
艻 0
艼 6
艽 13
艽 13
艾 361
艾 3618
艾子 0
艾子 0
艾森豪 31
艾森豪威爾 1
艾灸 2
艾瑪 34
艾略特 21
艾美獎 14
艾艾 9
艾草 26
艾酒 8
艿 7
芀 7
芁 0
芃 2422
芄 1304
芅 5
芆 0
芇 0
芉 0
芊 1077
芋 2840
芋圓 1
芋泥 1
芋頭 361
芋頭 361
芋頭冰 4
芋香 31
芌 0
芍 162
芍 162
芍藥 20
芍藥 20
芎 90
芎 90
芎 90
芎林 222
芏 6
芐 10
芐 10
芑 11
芒 403
芒刃 0
芒刺 5
芒刺在背 8
芒果 306
芒果乾 2
芒果樹 21
芒果汁 0
芒果皮 0
芒狀 4
芒硝 37
芒種 12
芒草 42
芒角 1
芓 5
芔 25
芔 1
芕 0
芖 0
芘 405
芙 878
芙蓉 378
芙蓉出水 0
芙蓉花 5
芙麗曼 5
芚 188
芛 542
芜 0
芝 1739
芝加哥 578
芝加哥大學 141
芝城 44
芝山 0
芝山岩 0
芝山巖 0
芝山站 0
芝蘭 52
芝麻 280
芝麻小事 16
芝麻油 7
芝麻湯圓 4
芝麻莢 0
芝麻街 39
芝麻醬 4
芞 206
芟 2
芟除 0
芠 189
芡 105
芢 194
芣 14
芤 63
芥 125
芥子 13
芥子毒氣 0
芥末 13
芥菜 47
芥菜頭 0
芥蒂 5
芥藍菜 13
芦 0
芧 76
芨 121
芩 273
芪 0
芫 167
芫荽 9
芬 7878
芬園 125
芬多精 1
芬奇 5
芬芳 194
芬蘭 338
芬蘭人 0
芬蘭文 15
芬蘭話 0
芬蘭語 8
芬郁 2
芬香 10
芬馥 0
芭 481
芭子 0
芭樂 249
芭樂汁 249
芭比 298
芭比娃娃 214
芭蕉 45
芭蕉扇 6
芭蕾 151
芭蕾舞 40
芭蕾舞團 92
芭蕾舞者 7
芮 281
芯 226
芯 226
芯子 3
芰 22
花 12344
花上 193
花下 22
花中 110
花中之王 1
花了 828
花事 8
花來花去 0
花俏 73
花信 46
花個 48
花價 1
花光 30
花兒 72
花兒 72
花冠 26
花到 20
花刺 4
花剌子模 1
花前月下 4
花匠 3
花卉 1109
花卉圖 2
花印 0
花卷 6
花去 12
花友會 10
花叢 54
花名 26
花呢 14
花圃 72
花圈 72
花國 11
花園 2535
花園裡 28
花團錦簇 23
花在 189
花城 11
花壇 447
花天酒地 2
花好月圓 19
花娘 2
花子 24
花季 173
花容 30
花容失色 0
花容月貌 1
花展 15
花山 16
花岡岩 10
花岡石 13
花崗山 4
花崗岩 58
花崗巖 0
花崗石 131
花巷 0
花市 143
花布 58
花帳 1
花帶 6
花序 77
花店 506
花廠 12
花廳 21
花式 132
花式溜冰 7
花彩 2
花彫 0
花彫酒 0
花影 31
花徑 71
花得 483
花得精光 0
花心 138
花心思 21
花心血 1
花性 3
花房 30
花托 6
花招 165
花拳 0
花拳繡腿 2
花捲 3
花掉 53
花旗 394
花旗銀行 304
花旦 4
花時間 146
花景 6
花月痕 2
花期 132
花木 103
花木扶疏 22
花木蘭 296
花朵 251
花束 277
花東海岸 91
花果 88
花果山 5
花枝 175
花枝招展 19
花架 36
花柄 5
花柱 5
花柳病 2
花栗鼠 3
花格 13
花格子 0
花梗 28
花棚 5
花椒 16
花椒粉 1
花椰菜 31
花槍 1
花樓 1
花樣 134
花樣年華 8
花樣滑冰 0
花樣百出 4
花樣繁多 0
花樣翻新 1
花樹 34
花橋 5
花毯 0
花海 30
花消 3
花炮 1
花無百日 0
花無百日紅 0
花燈 55
花燭 4
花燭之夜 0
花牆 8
花片 5
花狗 1
花王 94
花環 20
花瓜 1
花瓣 169
花瓶 200
花生 281
花生油 26
花生米 26
花生粉 10
花生糖 37
花生醬 24
花生餅 1
花用 14
花田 38
花甲 6
花甲之年 1
花痴 41
花癡 64
花白 27
花的 482
花盆 48
花盆裡 3
花盤 10
花眼 4
花砲 1
花稍 5
花種 12
花童 15
花筆 1
花筒 9
花箋 5
花籃 104
花籽兒 0
花籽兒 0
花粉 168
花粉囊 0
花粉熱 2
花粉管 8
花粉粒 5
花精 6
花紅 30
花紅柳綠 0
花紅草綠 0
花紅葉綠 0
花紋 187
花紋紙 0
花結 2
花絮 533
花絲 19
花經 2
花線 14
花縣 27
花腔 18
花腔女高音 4
花臉 12
花船 0
花色 116
花色繁多 2
花色齊全 1
花花 86
花花世界 54
花花公子 77
花花綠綠 9
花花草草 18
花芽 20
花芽分化 20
花苗 10
花苞 32
花茶 88
花草 152
花草樹木 45
花莖 18
花菜 31
花萼 12
花葉 8
花葉病 0
花蓮 5630
花蓮人 46
花蓮市 2207
花蓮師範學院 285
花蓮教育大學 1
花蓮港 153
花蓮溪 40
花蓮縣 5186
花蕊 25
花蕾 20
花藝 0
花藥 5
花蜜 32
花蝴蝶 155
花蟲 5
花街柳巷 2
花衣 1
花衣服 1
花被 12
花言巧語 39
花語 229
花豹 17
花貓 32
花販 1
花費 966
花賬 0
花起 1
花起來 0
花路 20
花車 63
花車遊行 9
花軸 4
花轎 19
花農 17
花道 28
花邊 133
花邊新聞 10
花都 61
花都巴黎 6
花酒 84
花采 3
花銀 0
花錢 376
花鍵 0
花鍵軸 0
花鎗 0
花鏡 4
花開 166
花開花落 11
花開花謝 8
花雕 7
花露 6
花露水 5
花青 7
花青素 20
花頭 1
花顏 3
花飾 22
花飾品 4
花香 258
花魁 15
花鳥 108
花鳥畫 12
花黃 13
花點 97
花鼓 44
芲 0
芳 15481
芳名 144
芳名錄 118
芳容 5
芳心 397
芳心無主 0
芳澤 9
芳烴 0
芳苑 208
芳草 68
芳華 20
芳訊 0
芳蹤 58
芳辰 0
芳鄰 223
芳香 397
芳香劑 38
芳香工業 0
芳香族 71
芳香植物 0
芳香烴 29
芳馥 1
芳魂 2
芳齡 8
芴 93
芵 110
芶 150
芷 279
芸 1498
芸芸 203
芸芸眾生 28
芹 124
芹菜 110
芹菜湯 0
芺 127
芻 48
芻狗 2
芻秣 0
芻糧 0
芻言 0
芻議 68
芼 175
芼 175
芽 393
芽豆 0
芾 29
芾 29
芿 0
苀 268
苁 0
苁 0
苁 0
苁 0
苂 40
苃 86
苄 0
苉 0
苊 0
苐 0
苑 2242
苑囿 4
苑裡 381
苒 22
苒苒 1
苓 1170
苓耳 0
苔 258
苔 258
苔原 7
苔癬 5
苔綠 2
苔蘚 32
苔蘚植物 6
苔類 0
苕 24
苕帚 0
苖 138
苗 1913
苗人 26
苗圃 214
苗土 0
苗子 3
苗子 3
苗嶺 0
苗床 5
苗族 46
苗族人 0
苗期 11
苗木 80
苗栗 1885
苗栗人 13
苗栗市 1063
苗栗縣 8214
苗條 50
苗條身材 5
苗疆 3
苗種 6
苗而不秀 1
苗肥 0
苗裔 3
苗頭 52
苘 0
苙 49
苚 0
苛 112
苛刻 17
苛吏 0
苛待 0
苛性 1
苛性鈉 29
苛性鉀 1
苛性鹼 4
苛捐 0
苛捐雜稅 0
苛政 4
苛政猛於虎 2
苛斂 1
苛求 38
苛法 1
苛評 1
苛責 34
苜 21
苜苜 0
苜蓿 34
苜蓿草 6
苝 0
苞 114
苞米 0
苟 55
苟不能 0
苟且 11
苟且偷安 1
苟且偷生 3
苟免 1
苟全 0
苟全性命 1
苟合 5
苟同 45
苟安 3
苟延殘喘 13
苟有 3
苟活 2
苟能 6
苠 2533
苡 476
苢 0
苣 21
苤 30692
若 25542
若 54
若不是 221
若不是 221
若不然 3
若且唯若 0
若何 22
若使 139
若像 8
若即若離 20
若向 10
若因 201
若夫 8
若夫 8
若干 1089
若干人 131
若干年 9
若想 375
若於 108
若是 3062
若會 17
若有 4367
若有所失 8
若有所思 18
若有若無 7
若果 43
若為 559
若無 642
若無其事 62
若能 988
若與 158
若蟲 5
若要 1130
若要人不知 4
若輩 0
若隱若現 16
若非 277
苦 2060
苦上加苦 1
苦不堪言 34
苦中作樂 14
苦主 17
苦了 43
苦事 2
苦人 1
苦修 21
苦僧 0
苦刑 6
苦力 20
苦功 9
苦勞 18
苦勸 10
苦去甘來 0
苦口 9
苦口婆心 31
苦叫 4
苦味 38
苦味道 0
苦味酸 5
苦命 61
苦哈哈 17
苦土 3
苦學 13
苦學多年 0
苦學成名 1
苦學有成 1
苦守 13
苦寒 4
苦工 22
苦差事 13
苦幹 15
苦幹實幹 4
苦役 6
苦得 19
苦得 107
苦心 164
苦心孤詣 8
苦思 51
苦思對策 2
苦悶 84
苦悶 84
苦惱 212
苦感 1
苦戀 13
苦戰 144
苦戲 1
苦於 57
苦日子 3
苦果 33
苦楚 28
苦樂 24
苦樹 0
苦水 38
苦活 0
苦海 61
苦海無邊 6
苦澀 135
苦澀感 1
苦無 118
苦無對策 2
苦瓜 120
苦瓜臉 6
苦甘 1
苦痛 99
苦的 107
苦盡甘來 17
苦笑 190
苦節 0
苦練 82
苦缺 0
苦肉計 19
苦苓林 1
苦苦 126
苦苦相求 0
苦苦相逼 4
苦藥 2
苦處 7
苦行 93
苦行僧 14
苦行者 4
苦衷 49
苦裏 0
苦諦 1
苦讀 36
苦讀出身 0
苦辛 4
苦過 1
苦過來 0
苦酒 11
苦酒滿杯 2
苦集滅道 3
苦難 302
苦雨 3
苦頭 57
苦頭 57
苦鬥 4
苦鹵 3
苧 9
苧麻 41
苨 62
苩 0
苪 5
苫 112
苫塊 0
苬 55
苭 302
苮 0
苯 571
苯乙烯 104
苯基 200
苯環 11
苯甲酸 21
苯胺 169
苯酚 43
苰 6
英 14294
英主 11
英人 22
英俊 181
英俊小子 0
英俊小生 3
英倫 99
英倫三島 4
英倫海峽 1
英偉 123
英傑 139
英兩 50
英制 29
英勇 128
英勇作戰 1
英吉利 5
英吉利海峽 9
英吋 197
英名 204
英名掃地 0
英呎 74
英哩 53
英國 5948
英國人 119
英國協 1
英國史 7
英國文學 148
英國詩 2
英士 31
英姿 76
英姿煥發 5
英寸 26
英尺 29
英幣 0
英年 7
英年早逝 14
英式 100
英德 86
英德字典 0
英才 121
英打 55
英挺 13
英文 17248
英文字 166
英文書 19
英文會話 55
英文版 1388
英文系 287
英文課 52
英日 131
英日同盟 0
英明 333
英格蘭 299
英武 10
英氣 6
英法 44
英法字典 0
英法聯軍 7
英漢 140
英漢字典 44
英漢詞典 2
英漢辭典 286
英烈 11
英烈千秋 0
英特爾 0
英畝 37
英發 12
英石 6
英磅 40
英美 720
英美兩國 40
英聯邦 1
英英 0
英英字典 0
英英美代子 0
英華 47
英語 4159
英語會話 182
英語系 178
英語課 8
英譯 113
英譯本 27
英豪 156
英軍 32
英里 92
英鎊 179
英雄 2021
英雄主義 15
英雄所見略同 11
英雄末路 0
英雄榜 416
英雄氣短 20
英雄無用武之地 11
英雄美人 7
英雄豪傑 42
英雄難過 0
英雄難過美人關 1
英雄館 45
英雌 31
英雌 31
英靈 10
英魂 13
苲 6
苳 10570
苴 28
苵 12
苶 7
苷 0
苸 0
苹 153
苺 504
苻 5
苻堅 5
苼 0
苽 0
苾 60
苾勃 1
苾芻 1
苾芻尼 1
苾苾 1
苿 0
茀 107
茁 20
茁壯 341
茂 4337
茂密 61
茂年 0
茂林 202
茂盛 142
茂竹 3
范 1930
范姜 100
范仲淹 13
范德林特 0
范蠡 16
范陽 6
茄 650
茄 650
茄冬 215
茄子 66
茄紅素 1
茅 233
茅台 17
茅台酒 2
茅坑 2
茅塞 3
茅塞頓開 4
茅屋 44
茅山 9
茅廁 3
茅廬 5
茅房 1
茅棚 2
茅盾 1
茅臺 1
茅臺酒 0
茅舍 21
茅草 70
茅草屋 3
茅草棚 0
茅蓬 0
茆 46
茇 84
茈 262
茉 72
茉茉 2
茉莉 167
茉莉花 46
茉莉花茶 7
茋 0
茌 16
茍 67
茎 0
茎 0
茖 107
茗 425
茗 425
茗具 0
茗茶 60
茘 0
茙 9
茚 0
茛 247
茜 600
茜素 6
茝 0
茝 0
茞 111
茟 0
茠 720
茡 0
茢 6
茣 0
茤 63
茥 2
茦 2
茧 191
茨 93
茨坪 0
茩 210
茪 2173
茫 112
茫昧 2
茫無頭緒 0
茫然 138
茫然不知 7
茫然無知 3
茫然若失 2
茫茫 217
茫茫前途 0
茫茫大海 11
茫茫然 17
茬 145
茬口 0
茭 63
茭杯 1
茭米 0
茮 0
茮 0
茯 128
茯苓 32
茰 0
茱 490
茱萸 19
茲 2753
茲事體大 12
茳 22
茴 5
茴香 37
茴香油 0
茵 562
茶 5356
茶具 147
茶几 73
茶匙 134
茶商 36
茶園 291
茶坊 163
茶場 5
茶場 5
茶壺 153
茶室 78
茶山 53
茶市 7
茶座 9
茶房 42
茶托 2
茶會 279
茶會服 0
茶末 13
茶杯 63
茶梅 7
茶棚 11
茶樓 31
茶樓酒肆 0
茶樹 109
茶水 124
茶水供應 5
茶油 11
茶湯 58
茶盅 1
茶盤 18
茶碗 8
茶碗蒸 1
茶磚 2
茶罐 8
茶肆 1
茶色 31
茶花 48
茶花女 71
茶莊 110
茶葉 854
茶葉末 3
茶葉罐 4
茶葉蛋 17
茶袋 4
茶褐 3
茶褐色 8
茶話會 2
茶資 0
茶質 3
茶農 23
茶道 94
茶酒 14
茶酒店 0
茶錢 1
茶青 1
茶食 9
茶飯 9
茶飯不思 2
茶餘飯後 32
茶館 512
茶點 109
茷 134
茸 44
茸 44
茸毛 10
茸茸 2
茹 108
茹 1089
茹毛飲血 1
茹素 8
茹苦含辛 1
茺 70
茻 9
茼 769
茼蒿菜 11
茽 0
茾 0
茿 11
荀 94
荀子 71
荀況 1
荁 2
荂 1102
荃 202
荄 6
荅 75
荆 0
荇 211
荈 132
草 4139
草人 12
草具 1
草創 74
草包 11
草原 439
草原千里 0
草原帶 4
草叢 96
草叢裡 13
草圖 106
草地 260
草地郎 0
草坪 131
草堂 32
草堆 14
草場 25
草場 25
草字 3
草寇 3
草寫 4
草履蟲 19
草屯 1367
草山 126
草席 7
草帽 27
草廬 30
草房 5
草擬 173
草擬文件 1
草料 5
草料場 1
草料場 1
草昧 9
草書 43
草木 145
草木灰 3
草木皆兵 17
草木經 0
草本 160
草本植物 48
草根 131
草根大使 4
草案 2289
草棉 0
草棚 1
草檄 1
草澤 25
草率 124
草率從事 3
草甸 0
草皮 70
草科 65
草稈 0
草稿 64
草稿紙 4
草笠 2
草約 12
草紙 4
草綠 20
草綠色 39
草繩 48
草聖 2
草船 1
草芥 6
草花蛇 0
草草 147
草草了事 6
草草收場 3
草草收場 3
草草率率 1
草莓 359
草莓族 1
草莓醬 3
草莽 20
草莽英雄 6
草菅 4
草菅人命 11
草菇 60
草菴 0
草萊 2
草葉 12
草蓆 44
草藥 221
草藥方 1
草蛭 0
草蝦 114
草蟲 7
草行 2
草裙舞 6
草詔 1
草酸 113
草酸鹽 3
草野 7
草長馬肥 0
草雞 0
草鞋 51
草類 26
草食獸 1
草魚 33
荊 291
荊婦 0
荊州 97
荊扉 1
荊條 0
荊棘 64
荊棘叢生 1
荊浩 64
荊蠻 1
荊軻 15
荊釵 64
荊釵記 64
荋 100
荌 276
荍 79
荎 745
荎藸 1
荏 25
荏苒 4
荐 109
荐任 12
荐引 0
荐舉 0
荑 17
荒 421
荒亂 4
荒了 1
荒僻 12
荒冷 0
荒原 39
荒唐 91
荒土 3
荒地 42
荒塚 3
荒山 27
荒島 43
荒年 2
荒廢 258
荒廢多年 1
荒徑 0
荒怠 2
荒政 0
荒旱 2
荒村 8
荒涼 102
荒淫 9
荒淫無度 1
荒淫無恥 1
荒淫無道 2
荒漠 27
荒漠甘泉 10
荒無人煙 1
荒煙漫草 1
荒田 2
荒疏 2
荒瘠 0
荒腔走板 0
荒草 15
荒蕪 74
荒誕 27
荒誕不稽 0
荒誕不經 8
荒謬 194
荒謬絕倫 5
荒郊 10
荒郊野外 7
荒野 152
荒雞 0
荒饉 0
荓 90
荔 87
荔枝 168
荔枝汁 1
荕 0
荖 219
荖 219
荗 0
荘 0
荣 0
荰 0
荱 0
荲 0
荳 268
荳蔻 12
荳蔻年華 7
荴 3
荵 8
荶 8
荷 1480
荷 1480
荷人 34
荷包 136
荷包蛋 15
荷塘 13
荷擔 3
荷槍 5
荷槍實彈 11
荷爾蒙 561
荷爾蒙 561
荷花 133
荷花池 18
荷花澱 0
荷荷 20
荷葉 46
荷著 0
荷蘭 4767
荷蘭人 100
荷蘭文 7
荷蘭話 0
荷蘭語 10
荷語 9
荷載 14
荷重 130
荷重量 0
荷鎗實彈 0
荷馬 23
荷馬史詩 5
荸 37
荸 37
荸荸 0
荸荸 0
荸薺 10
荸薺 10
荹 0
荺 8
荻 62
荻畫 0
荻筆 0
荻花 0
荼 96
荼 96
荼毒 8
荼毒生靈 0
荼毗 4
荽 23
荾 14
荿 15
莀 0
莁 13
莂 0
莃 19
莄 0
莅 0
莆 28
莆 28
莆田 13
莆田市 16
莇 10
莈 5
莉 3877
莉莉 418
莊 4937
莊主 28
莊周 29
莊員 3
莊嚴 344
莊嚴佛土 1
莊嚴肅穆 26
莊園 51
莊子 209
莊子 209
莊家 146
莊戶 1
莊敬 224
莊敬自強 6
莊田 1
莊稼 38
莊稼人 5
莊稼漢 0
莊裡 1
莊重 27
莊院 12
莋 39
莌 6
莍 1
莎 140
莎 1403
莎士比亞 194
莎拉 153
莎翁 44
莎草 20
莎莉 86
莎車 0
莏 25
莐 2
莑 0
莒 134
莒光 388
莒光日 5
莒光號 79
莒光週 0
莒城 0
莒拳 4
莓 104
莔 16
莕 40
莖 332
莖幹 8
莗 14
莘 137
莘 137
莘莘學子 76
莘莘學子 76
莙 19
莚 8
莚 8
莛 11
莜 0
莝 12
莞 259
莞 259
莞 259
莞 259
莞爾 45
莟 0
莠 34
莠 34
莠民 0
莠民 0
莡 0
莢 74
莢果 15
莢膜 10
莢膜 10
莣 4
莤 17
莥 14
莥 14
莦 10
莧 30
莧科 18
莧菜 36
莨 9
莨苕 0
莨菪 2
莩 11
莩 11
莪 50
莫 2825
莫三鼻給 3
莫不 270
莫不是 14
莫可奈何 18
莫可言狀 0
莫名 284
莫名其妙 270
莫大 284
莫奈 1
莫如 24
莫往 0
莫怪 24
莫扎特 11
莫扎特 11
莫折大提 0
莫斯科 332
莫斯科市 2
莫札特 284
莫桑比克 0
莫泊桑 11
莫涅 1
莫測 58
莫測高深 27
莫爾 37
莫管 5
莫管閒事 0
莫索里尼 0
莫耳 36
莫若 14
莫衷一是 21
莫說 21
莫逆 4
莫逆之交 4
莫過 20
莫過於 198
莫道 18
莫那魯道 1
莫里哀 1
莫非 185
莫非是 17
莫須 14
莫須有 9
莬 0
莭 0
莮 10
莯 8
莰 10
获 0
获 0
莽 49
莽原 2
莽夫 2
莽撞 29
莽林 5
莽漢 5
莽莽 4
莽莽撞撞 0
莽蒼 0
莽蕩 0
莾 0
莿 369
菀 50
菁 1623
菁英 601
菁英份子 12
菁英份子 12
菁華 113
菂 4
菃 103
菄 60
菅 130
菅芒 0
菅芒花 0
菅茅 0
菆 6
菇 607
菈 18
菉 20
菉 20
菊 1203
菊園 2
菊花 170
菊花茶 6
菋 4
菌 1933
菌傘 0
菌株 64
菌核 5
菌液 10
菌液 10
菌狀 0
菌種 120
菌絲 19
菌絲體 6
菌苗 19
菌落 70
菌褶 0
菌類 69
菍 0
菍 0
菎 25
菎 25
菎麻 2
菎麻 2
菏 7
菏 7
菐 0
菑 191
菒 0
菓 0
菔 0
菔 11
菕 524
菖 72
菖蒲 35
菗 36
菘 32
菙 65
菚 0
菛 13
菜 3130
菜刀 54
菜包子 0
菜單 242
菜圃 12
菜園 98
菜地 3
菜場 1
菜場 1
菜墩子 0
菜子 17
菜子油 0
菜市 20
菜市場 73
菜市場 73
菜攤 4
菜枯 0
菜根 20
菜根譚 30
菜油 5
菜湯 18
菜牛 2
菜瓜 5
菜瓜布 8
菜田 0
菜畦 0
菜碟 0
菜碼兒 0
菜碼兒 0
菜種 5
菜籃 42
菜籽 1
菜粉蝶 0
菜肴 13
菜色 131
菜花 35
菜苔 0
菜蔬 10
菜薹 0
菜蚜 0
菜豆 28
菜貨 0
菜農 14
菜青 2
菜青蟲 0
菜餚 104
菜館 51
菜鳥 0
菜鴨 17
菝 15
菞 3
菟 52
菠 117
菠菜 74
菠蘿 20
菡 78
菡萏 1
菢 31
菣 4
菤 5
菥 1
菦 0
菧 10
菨 2
菩 199
菩提 832
菩提子 1
菩提樹 77
菩提迦耶 1
菩薩 1403
菩薩低眉 1
菩薩十地 1
菩薩地 1
菩薩心腸 7
菩薩戒 1
菩薩界 1
菪 169
菫 78
菬 47
菭 0
菮 22
華 12300
華 2893
華 35904
華中 215
華中地區 6
華亭 9
華亭唳鶴 1
華亭鶴唳 1
華人 3178
華佗 135
華佗 135
華佗再世 0
華僑 1126
華僑銀行 110
華冑 4
華北 72
華北地區 9
華北平原 2
華南 356
華南地區 19
華南銀行 162
華嚴 14
華嚴宗 14
華嚴經 69
華國 180
華埠 65
華夏 1302
華夏工專 38
華夏工業專科學校 3
華夏文化 5
華夷 1
華封三祝 1
華屋 9
華山 501
華山 501
華岡 368
華岡藝術學校 6
華崗 18
華工 25
華年 22
華府 408
華康 492
華廈 164
華廈 164
華德 133
華德狄斯耐 1
華文 543
華文漪 1
華新 3
華新 335
華族 6
華東 146
華東地區 6
華格納 103
華梵大學 1
華榮 41
華殿 1
華氏 61
華氏寒暑表 0
華氏溫度計 1
華江 365
華江國中 1
華江國小 10
華江國民中學 1
華江國民小學 1
華沙 42
華沙公約 4
華燈 24
華燈初上 16
華燭 2
華爾滋 39
華爾茲 6
華爾茲舞 0
華爾街 398
華爾街 398
華生 186
華盛頓 994
華盛頓郵報 119
華碩 492
華祝三多 1
華納威秀 1
華美 179
華而不實 14
華聲 37
華興 286
華興會 1
華航 757
華茲華斯 2
華萊士瓜 0
華蓋 8
華蘅芳 1
華表 7
華裔 633
華裡 3
華西 27
華西街 76
華視 2365
華視公司 8
華誕 28
華語 467
華語學校 0
華貴 29
華達呢 2
華里 61
華銀 308
華陀 57
華陀 57
華陀在世 0
華陰 19
華陰縣 3
華陵 7
華陽國志 1
華飾 19
華髮 2
華鬘 1
華麗 492
華齡 12
菰 38
菰米 0
菱 945
菱形 56
菱形硫 0
菱形體 0
菱角 84
菱鏡 9
菲 32
菲 3207
菲亞特公司 0
菲國 172
菲律賓 1411
菲律賓人 29
菲律賓海 29
菲律賓海溝 0
菲律賓群島 1
菲律賓話 0
菲禮 0
菲薄 3
菲語 21
菳 9
菴 1282
菴 1282
菵 7
菶 6
菸 928
菸害 0
菸毒 1
菸絲 3
菸草 106
菸葉 28
菸酒 138
菸酒公賣 5
菸酒公賣局 133
菸鹼 47
菹 6
菺 10
菻 50
菼 5
菽 19
菽水承歡 0
菽麥 0
菾 283
菿 1
萀 0
萁 19
萂 0
萃 235
萃取 486
萄 129
萄 129
萅 0
萆 4
萆麻 0
萇 555
萈 0
萉 7
萊 2710
萊因河 2
萊布尼茲 8
萊比錫 44
萊特 231
萊茵河 44
萊菔 2
萊辛 35
萊陽 20
萋 116
萋萋 9
萌 189
萌兆 0
萌動 1
萌生 47
萌發 46
萌芽 186
萌芽期 19
萌芽林 0
萌起 1
萍 1940
萍水相逢 27
萍蹤 7
萍鄉 6
萎 325
萎 325
萎縮 634
萎蕤 1
萎謝 1
萎雕 0
萎靡 15
萎靡不振 3
萎頓 4
萎黃 5
萎黃病 0
萏 4
萐 3
萑 5
萒 6
萓 11
萔 0
萕 0
萕 0
萖 0
萖 0
萖 0
萖 0
萗 0
萘 0
萛 28
萟 0
萣 235
萩 103
萪 0
萫 7
萬 19239
萬一 598
萬一在 21
萬一是 6
萬一會 0
萬一有 30
萬丈 42
萬丈光芒 7
萬丈深淵 4
萬丈紅塵 1
萬丈高樓平地起 11
萬不得已 47
萬世 116
萬世師表 6
萬乘 0
萬乘之國 0
萬乘之尊 0
萬事 715
萬事具備 2
萬事大吉 0
萬事皆備 0
萬事起頭難 14
萬事通 60
萬人 1310
萬人敵 1
萬人空巷 10
萬代 71
萬份 49
萬位 12
萬個 234
萬倍 25
萬元 7250
萬全 73
萬全之策 0
萬全之計 0
萬全街 12
萬分 334
萬分感激 33
萬劫 12
萬劫不復 24
萬千 119
萬卷 31
萬卷書 5
萬古 28
萬古流芳 1
萬古長新 1
萬古長青 0
萬名 135
萬向節 0
萬國 239
萬國公法 0
萬國音標 0
萬圓 13
萬壽 255
萬壽山 19
萬壽果 0
萬壽無疆 0
萬夫 6
萬夫不當 0
萬夫莫敵 4
萬姓 2
萬字 112
萬安 114
萬家 94
萬家燈火 11
萬家生佛 1
萬家香 17
萬山 24
萬巒 189
萬年 535
萬年曆 62
萬年青 23
萬幸 10
萬念俱灰 4
萬惡 21
萬惡之首 0
萬惡淫為首 2
萬慮俱清 0
萬戶 137
萬把 2
萬把塊 1
萬斛泉源 0
萬斤 15
萬方 20
萬曆 53
萬有 399
萬有引力 36
萬有引力定律 8
萬桶 54
萬步 6
萬歲 289
萬歷 8
萬死 9
萬死一生 1
萬死一生 1
萬死不辭 0
萬段 1
萬無 5
萬無一失 22
萬無一失 22
萬物 410
萬物之靈 29
萬狀 7
萬獸之王 11
萬用 567
萬用字元 1
萬用手冊 54
萬用電表 1
萬盛 35
萬盛溪 0
萬眾 28
萬眾一心 3
萬眾一心 3
萬眾歡騰 2
萬眾矚目 7
萬福 75
萬種 59
萬種風情 22
萬端 14
萬箭穿身 0
萬箱 1
萬籟 3
萬籟俱寂 7
萬米 5
萬米長跑 0
萬紫千紅 10
萬綠叢中 2
萬綠叢中一點紅 7
萬綠叢中一點紅 7
萬縣 23
萬聖節 142
萬能 996
萬能工專 36
萬能選手 0
萬能鍋 0
萬般 61
萬般皆下 0
萬般皆下品 3
萬花爭豔 0
萬花筒 118
萬華 359
萬華區 783
萬華站 1
萬萬 111
萬萬不可 15
萬萬歲 30
萬言 31
萬變不離其宗 10
萬象 872
萬象更新 11
萬貫 6
萬貫家產 0
萬貫家財 19
萬道 5
萬邦 19
萬部 17
萬里 839
萬里晴空 6
萬里無雲 14
萬里長城 49
萬里長征 4
萬重 5
萬重山 8
萬金 50
萬金油 4
萬鈞 29
萬鈞之力 0
萬難 19
萬靈丹 56
萬靈藥 1
萬頃琉璃 0
萬頃碧波 2
萬頃碧波 2
萬頭竄動 0
萬餘 185
萬餘人 40
萬餘元 97
萬馬 38
萬馬奔騰 12
萬點 101
萬點大關 2
萭 53
萮 0
萮 0
萮 0
萯 89
萰 165
萱 1048
萱堂 0
萱草 13
萲 92
萲 92
萳 13
萴 70
萵 182
萵筍 0
萵苣 42
萵萵 0
萶 7
萷 50
萸 427
萹 42
萹 42
萺 54
萻 48
萼 293
萼片 0
落 1
落 292
落 2921
落下 224
落下來 25
落下去 9
落井 0
落井下石 22
落人口實 11
落人口實 11
落伍 107
落入 218
落到 109
落單 46
落在 465
落在後面 1
落地 387
落地扇 1
落地生根 51
落地窗 35
落坐 2
落塵 54
落塵量 6
落寞 72
落實 3313
落山風 25
落差 206
落幕 346
落後 652
落後國家 22
落得 67
落得一場空 0
落得一場空 0
落得一場空 0
落得一場空 0
落得清淨 0
落成 366
落成典禮 37
落戶 9
落拓 4
落拓不羈 0
落敗 89
落於 69
落日 160
落日餘暉 15
落暉 0
落月 3
落桃 0
落榜 38
落款 13
落水 86
落水狗 8
落海 35
落淚 98
落湯雞 7
落潮 0
落照 1
落石 0
落磯山 25
落礦 0
落空 110
落第 9
落筆 4
落籍 9
落絮 3
落網 171
落翅仔 0
落腳 70
落腳處 12
落花 67
落花有意 6
落花流水 13
落花生 94
落英 9
落英繽紛 7
落草 4
落草為寇 1
落荒 5
落荒而逃 9
落落 21
落落大方 13
落落寡歡 4
落葉 235
落葉喬木 17
落葉松 2
落葉樹 2
落葉歸根 25
落著 13
落話 0
落選 292
落錘 1
落阱下石 0
落雁 71
落難 38
落雨 16
落霜 0
落霞 22
落體 7
落髮 29
落髮為僧 1
落魄 31
落點 47
萾 0
萿 88
葀 117
葁 0
葂 54
葃 34
葄 2
葅 50
葆 404
葇 52
葈 0
葉 90
葉 9901
葉公超 4
葉叢 10
葉子 348
葉子煙 0
葉宗留 0
葉尖 5
葉尼基河 0
葉尼塞河 0
葉挺 0
葉斑 1
葉斑病 6
葉柄 30
葉樹 87
葉片 342
葉狀 13
葉綠素 71
葉綠體 27
葉耳 17
葉肉 14
葉脈 35
葉脈 35
葉脤 0
葉腋 6
葉舌 2
葉芽 4
葉莖 7
葉落歸根 1
葉葉 9
葉蘊儀 3
葉蘭 34
葉輪 19
葉酸 57
葉針 1
葉門 23
葉面 37
葉鞘 7
葋 54
葌 13
葍 104
葎 56
葏 0
葐 17
葑 135
葒 60
葓 0
葔 14
葕 0
葖 38
著 2036
著 2036
著 2036
著 20366
著 8130
著有 100
著上 39
著作 4381
著作人 330
著作權 3244
著作物 26
著作等身 8
著作者 89
著兒 13
著兒 13
著力 99
著力點 24
著名 2375
著名人物 3
著地 79
著地點 1
著墨 0
著實 198
著實 198
著急 116
著急起來 0
著想 139
著意 7
著慌 3
著手 992
著手成春 1
著手於 22
著書 50
著書立說 2
著毋庸議 0
著涼 14
著涼 14
著火 61
著火點 2
著眼 293
著眼 293
著眼點 32
著稱 121
著筆 17
著緊 7
著績 0
著者 273
著色 283
著色法 4
著落 52
著著 16
著衣 10
著述 154
著迷 142
著迷於 10
著重 880
著重指出 0
著重於 364
著錄 94
著陸 41
著魔 24
葙 68
葚 60
葛 213
葛 2134
葛天氏 0
葛布 1
葛粉 1
葛羅托斯基 4
葛羅斯基 4
葛羅米柯 0
葛藤 6
葜 0
葝 52
葞 66
葟 21
葠 5
葡 448
葡糖 1
葡萄 601
葡萄乾 32
葡萄園 60
葡萄柚 63
葡萄柚汁 1
葡萄樹 20
葡萄汁 24
葡萄牙 284
葡萄牙人 18
葡萄牙語 13
葡萄球菌 88
葡萄糖 384
葡萄胎 34
葡萄藤 16
葡萄酒 506
葢 0
葢 0
葢 0
董 2783
董事 1198
董事會 731
董事長 3156
董仲舒 5
董卓 23
董存瑞 0
董必武 1
董源 23
葤 0
葥 166
葦 1447
葦眉子 0
葧 60
葨 10
葩 315
葪 0
葫 148
葫蘆 293
葬 588
葬儀 6
葬儀社 23
葬在 24
葬悼 0
葬禮 39
葬身 35
葬身之地 1
葬身河底 0
葬身海底 3
葬身湖底 0
葬身魚腹 0
葬送 4
葬送前途 0
葬送在 1
葬送掉 1
葭 688
葮 11
葯 384
葰 53
葱 0
葲 0
葳 426
葴 42
葵 492
葵扇 0
葵花 63
葵花子 1
葵花子油 3
葵花油 1
葶 37
葷 722
葷 722
葷油 0
葷腥 2
葷腥不沾 0
葷菜 1
葷辛 0
葷酒 0
葷食 37
葸 76
葹 19
葺 52
葻 0
葻 0
葼 0
葽 69
葾 4
葿 0
蒀 0
蒁 0
蒂 2259
蒃 0
蒄 0
蒆 66
蒈 0
蒍 170
蒎 61
蒏 53
蒐 924
蒐求 1
蒐羅 46
蒐藏 1
蒐証 0
蒐證 99
蒐購 17
蒐錄 54
蒐集 2309
蒑 112
蒓 0
蒓 0
蒔 149
蒔蘿 2
蒕 0
蒖 0
蒗 318
蒗渠 0
蒘 140
蒙 1912
蒙上 84
蒙人 3
蒙住 10
蒙受 120
蒙古 383
蒙古人 40
蒙古包 21
蒙古地方 1
蒙古大夫 10
蒙古文 2
蒙古族 21
蒙古症 9
蒙古話 0
蒙古語 6
蒙古高原 8
蒙在 8
蒙在鼓裡 8
蒙地卡羅 37
蒙垢 0
蒙城 3
蒙塵 22
蒙太奇 22
蒙娜麗莎 46
蒙山 5
蒙恩 329
蒙恬 175
蒙族 0
蒙昧 2
蒙昧無知 4
蒙汗藥 5
蒙混 6
蒙特 225
蒙特婁 421
蒙皮 2
蒙羞 15
蒙胞 4
蒙臉 3
蒙蒙 7
蒙蓋 1
蒙蔽 31
蒙藏 62
蒙藏同胞 10
蒙藏委員 8
蒙藏委員會 136
蒙語 2
蒙起 2
蒙起來 2
蒙難 4
蒙難記 3
蒙面 89
蒙面俠 28
蒙面客 1
蒙頭 11
蒙頭大睡 2
蒙騙 15
蒚 202
蒛 212
蒜 263
蒜末 1
蒜泥 18
蒜瓣 1
蒜苗 4
蒜頭 147
蒝 218
蒞 440
蒞場 2
蒞場 2
蒞會 24
蒞臨 1628
蒟 216
蒟蒻 1
蒠 55
蒡 180
蒡 180
蒢 248
蒣 0
蒣 0
蒤 267
蒥 0
蒦 0
蒧 179
蒨 232
蒩 38
蒪 266
蒫 465
蒬 284
蒭 0
蒮 137
蒯 273
蒰 222
蒱 168
蒲 280
蒲公英 182
蒲劍 0
蒲團 20
蒲州 3
蒲式耳 4
蒲扇 3
蒲松齡 3
蒲柳 1
蒲殿俊 0
蒲節 0
蒲葵 12
蒲蓆 0
蒲魯東 0
蒳 0
蒴 258
蒴果 21
蒵 0
蒵 0
蒶 114
蒷 0
蒸 655
蒸了 1
蒸便當 0
蒸去 0
蒸呢 0
蒸掉 0
蒸散 14
蒸散作用 7
蒸暑 0
蒸氣 567
蒸氣浴 13
蒸氣渦爐 0
蒸氣計 0
蒸氣輪機 0
蒸汽 292
蒸汽機 10
蒸汽機車 52
蒸汽浴 2
蒸汽熨斗 4
蒸汽狀 0
蒸熟 25
蒸熱 10
蒸發 252
蒸發器 33
蒸發掉 3
蒸發散 4
蒸發皿 6
蒸發計 0
蒸發量 21
蒸籠 24
蒸肉 6
蒸肉法 0
蒸蒸 1
蒸蒸日上 69
蒸蛋 10
蒸酥 1
蒸鍋 2
蒸飯 9
蒸飯箱 4
蒸餃 13
蒸餅 0
蒸餾 310
蒸餾水 118
蒸餾液 2
蒸餾液 2
蒸騰 11
蒸騰作用 0
蒸魚 13
蒸點 0
蒹 323
蒺 374
蒺藜 5
蒻 177
蒼 765
蒼冥 2
蒼勁 15
蒼天 114
蒼松 22
蒼民 1
蒼海 9
蒼涼 28
蒼然 1
蒼生 43
蒼白 153
蒼白色 1
蒼穹 52
蒼翠 35
蒼翠欲滴 1
蒼老 39
蒼耳 4
蒼茫 30
蒼莽 4
蒼蒼 16
蒼蠅 173
蒼蠅兒 0
蒼蠅兒 0
蒼蠅拍 2
蒼蠅紙 0
蒼鉛 0
蒼顏 0
蒼鷹 25
蒼鷺 57
蒼黃 0
蒼龍 9
蒾 0
蒿 49
蓀 137
蓁 450
蓂 541
蓃 0
蓃 0
蓄 264
蓄勢 34
蓄勢待發 131
蓄意 120
蓄意謀殺 0
蓄水 132
蓄水池 106
蓄洪 8
蓄洪區 0
蓄洪量 0
蓄熱器 0
蓄牧 1
蓄牧業 0
蓄積 93
蓄積物 0
蓄積著 0
蓄謀 0
蓄謀已久 0
蓄財 3
蓄長髮 0
蓄電 10
蓄電池 160
蓄電量 1
蓄養 12
蓄髮 4
蓄鬚 1
蓅 55
蓆 216
蓆地 0
蓆子 5
蓇 226
蓈 0
蓉 1996
蓊 79
蓊葧 0
蓊鬱 24
蓋 47
蓋 4753
蓋一蓋 0
蓋上 112
蓋下 14
蓋世 254
蓋世之才 0
蓋世太保 7
蓋世無匹 0
蓋世無雙 0
蓋世英雄 2
蓋了 216
蓋了又蓋 0
蓋些 0
蓋亞納 12
蓋亞那 13
蓋仙 18
蓋住 61
蓋印 40
蓋印處 0
蓋好 55
蓋子 70
蓋層 12
蓋平 4
蓋建 8
蓋得 22
蓋戳 7
蓋掉 24
蓋於 19
蓋有 48
蓋板 19
蓋棺 58
蓋棺論定 9
蓋洛普 148
蓋滿 15
蓋爾 55
蓋玻片 5
蓋瑞 38
蓋盅兒 0
蓋盅兒 0
蓋章 288
蓋章處 1
蓋起 18
蓋起來 3
蓋造 14
蓋過 0
蓋錯 2
蓋頂 8
蓋頭 17
蓌 158
蓍 216
蓍菜 0
蓍龜 1
蓎 240
蓏 35
蓐 293
蓑 40
蓑笠 1
蓑草 0
蓑衣 0
蓒 68
蓓 505
蓓蕾 28
蓔 81
蓕 0
蓖 67
蓖麻油 6
蓗 116
蓘 0
蓚 0
蓚 0
蓛 34
蓤 0
蓤 0
蓧 10
蓨 2
蓩 9
蓪 5
蓫 15
蓬 479
蓬亂 4
蓬勃 212
蓬勃發展 393
蓬戶 0
蓬萊 190
蓬萊仙境 1
蓬萊仙島 7
蓬萊米 6
蓬蒿 3
蓬蓬 3
蓬蓬勃勃 0
蓬蓬鬆鬆 1
蓬蓽 0
蓬蓽生輝 4
蓬車 12
蓬門 1
蓬頭 7
蓬頭垢面 5
蓬首 2
蓬髮 0
蓬鬆 27
蓮 14666
蓮台 5
蓮子 78
蓮子湯 3
蓮座 14
蓮心 56
蓮步 3
蓮池 58
蓮社 31
蓮臺 1
蓮花 816
蓮花座 6
蓮花池 57
蓮花瓣 2
蓮花生 1
蓮花落 4
蓮花落 4
蓮荷 0
蓮蓉 15
蓮蓉月餅 0
蓮蓬 21
蓮蓬頭 90
蓮藕 29
蓮霧 124
蓯 5
蓰 109
蓱 59
蓲 6
蓲 6
蓳 9
蓴 25
蓵 0
蓶 12
蓷 226
蓸 0
蓹 6
蓺 27
蓻 4
蓼 41
蓼 41
蓼花 0
蓼莪 0
蓼蓼 0
蓽 13
蓽路縷 0
蓽路藍縷 20
蓾 16
蓿 226
蔀 10
蔁 0
蔂 4
蔃 0
蔃 0
蔄 0
蔅 0
蔆 218
蔇 486
蔈 18
蔈 18
蔉 2
蔊 5
蔋 3
蔌 5
蔍 5
蔎 4
蔏 2
蔐 0
蔑 287
蔑棄 0
蔑視 21
蔒 9
蔓 306
蔓延 326
蔓延下來 0
蔓延到 17
蔓延性 3
蔓生 12
蔓生植物 3
蔓草 16
蔓藤 18
蔓衍 1
蔓越莓 1
蔓陀羅 1
蔔 191
蔔 441
蔕 9
蔖 6
蔗 199
蔗作 13
蔗園 10
蔗板 2
蔗渣 16
蔗漿 2
蔗糖 92
蔗農 20
蔗醣 0
蔘 181
蔙 6
蔚 1259
蔚成 26
蔚海岸 0
蔚為 136
蔚為大觀 2
蔚為奇觀 14
蔚然 24
蔚然成風 5
蔚色 1
蔚藍 56
蔚藍海岸 24
蔚藍色 2
蔚起 7
蔛 0
蔜 3
蔝 5
蔞 139
蔞蒿 0
蔟 6
蔠 1
蔡 11511
蔡倫 13
蔡元培 38
蔡孟勳 5
蔡文姬 2
蔡正仁 1
蔡裕華 2
蔡邕 3
蔡鍔 11
蔢 0
蔢 0
蔣 1980
蔣中正 47
蔣介石 229
蔣公 280
蔣公遺訓 0
蔣彥士 25
蔣故總統 32
蔣經國 265
蔣緯國 71
蔣總統 45
蔤 2
蔥 654
蔥嶺 4
蔥油雞 2
蔥油餅 8
蔥白 21
蔥綠 8
蔥翠 5
蔥花 21
蔥蒜 11
蔥蔥 1
蔥頭 13
蔦 59
蔧 4
蔨 6
蔩 7
蔪 5
蔫 7
蔬 516
蔬果 346
蔬菜 1174
蔬菜汁 9
蔬食 2
蔭 417
蔭庇 2
蔭影 5
蔭棚 3
蔭涼 6
蔭蔽 5
蔮 14
蔯 7
蔰 31
蔱 5
蔲 0
蔳 0
蔳 0
蔴 0
蔵 0
蔵 0
蔸 0
蔻 167
蔻丹 1
蔽 323
蔽匿 0
蔽塞 3
蔽天 7
蔽日 7
蔽空 1
蔽身 0
蔽野 1
蔽體 3
蔾 5
蔿 0
蕀 7
蕁 29
蕁 29
蕁麻 9
蕁麻疹 113
蕂 0
蕃 123
蕃 1235
蕃人 20
蕃兵 0
蕃庶 0
蕃殖 1
蕃茂 0
蕃茄 195
蕃茄汁 195
蕃茄片 4
蕃茄醬 27
蕃薯 11133
蕃薯粉 5
蕃薯葉 1
蕃薯藤 0
蕄 3
蕅 96
蕆 6
蕇 0
蕈 361
蕈 361
蕉 334
蕉園 8
蕉布 0
蕉扇 0
蕉農 9
蕊 570
蕊心 5
蕊柱 0
蕋 0
蕌 0
蕍 12
蕎 28
蕎麥 31
蕎麥麵 1
蕏 0
蕏 0
蕐 0
蕐 0
蕐 0
蕑 8
蕒 0
蕓 20
蕔 4
蕕 4
蕖 15
蕗 5
蕘 10
蕙 1003
蕙心 8
蕙蘭 30
蕙質蘭心 6
蕚 0
蕛 1
蕜 0
蕝 4
蕞 640
蕞爾 20
蕞爾小國 2
蕟 0
蕠 2
蕡 8
蕢 7
蕣 5
蕤 41
蕥 3
蕦 3
蕧 1
蕨 397
蕨類 177
蕩 229
蕩來蕩去 4
蕩女 1
蕩婦 20
蕩寇 5
蕩平 2
蕩氣迴腸 7
蕩滌 1
蕩漾 46
蕩漾出 0
蕩然 11
蕩然無存 23
蕩舟 3
蕩船 0
蕩蕩 30
蕩蕩悠悠 1
蕪 194
蕪湖 10
蕪穢 0
蕪繁 0
蕪菁 6
蕪雜 0
蕫 18
蕬 8
蕭 9591
蕭伯納 6
蕭何 12
蕭太后 4
蕭條 129
蕭森 0
蕭瀟 10
蕭灑 3
蕭然 4
蕭瑟 27
蕭紅 7
蕭索 17
蕭蕭 58
蕭薔 46
蕭規曹隨 2
蕭邦 106
蕭颯 14
蕮 2
蕯 0
蕰 0
蕰 0
蕱 2
蕵 3
蕶 3
蕷 12
蕸 6
蕹 24
蕺 17
蕻 48
蕼 4
蕽 0
蕾 1111
蕾絲 126
蕾鈴 0
蕿 0
薀 19
薁 5
薂 0
薃 11
薄 17
薄 1728
薄 18231
薄情薄義 ㄅㄛˊ
薄倖 ㄅㄛˊ
薄雲 ㄅㄛˊ
薄鬢 10
薄眉 10
薄媚 10
薄母 10
薄分 10
薄夫 10
薄福 10
薄待 10
薄頭 10
薄貼 10
薄陋 10
薄劣 10
薄臉 10
薄祿相 10
薄落 10
薄幹 10
薄殼 10
薄海 10
薄海同仇 10
薄海歡騰 10
薄祜 10
薄賤 10
薄遽 10
薄親 10
薄曲 10
薄酌 10
薄裝 10
薄酬 10
薄懲 10
薄識 10
薄紹之 10
薄生生 10
薄醉 10
薄俗 10
薄惡 10
薄儀 10
薄意 10
薄物細故 10
薄晚 10
薄酒萊 100
薄伽梵 0
薄倖 7
薄具 11
薄冰 3
薄利 44
薄利多銷 15
薄命 8
薄壁 9
薄弱 170
薄弱環節 1
薄情 17
薄情郎 6
薄懲 2
薄技 0
薄暮 14
薄曉 0
薄板 54
薄海騰歡 1
薄片 102
薄產 1
薄田 5
薄砂地 0
薄祚 0
薄禮 7
薄紗 18
薄紙 6
薄脆 1
薄膜 665
薄膜 665
薄膜狀 3
薄膜狀 3
薄荷 112
薄荷油 0
薄荷精 2
薄荷糖 111
薄荷葉 7
薄薄 289
薄薄的 58
薄薄地 57
薄行 0
薄行 0
薄衣 3
薄親 0
薄酒 55
薄霧 21
薄青色 0
薄面 4
薄餅 30
薄鬆鬆 0
薅 4
薆 4
薇 1618
薈 84
薈萃 35
薈萃一堂 0
薈萃一堂 0
薉 8
薊 298
薋 34
薌 51
薍 2
薎 1
薏 250
薏仁 37
薐 30
薑 484
薑末 10
薑母鴨 1
薑汁 18
薑片 19
薑糖 7
薑絲 27
薑餅 48
薒 0
薓 0
薔 397
薔薇 184
薔薇 184
薕 2
薖 5
薗 0
薘 3
薙 52
薙刀 0
薙去 0
薙毛 4
薙頭 4
薙髮 4
薚 9
薛 1301
薛寶釵 0
薛福成 0
薜 97
薜 97
薜蘿 1
薝 5
薞 7
薟 5
薠 15
薡 4
薢 9
薣 3
薤 18
薥 0
薥 0
薦 980
薦任 327
薦引 1
薦拔 0
薦福碑 5
薦舉 5
薦賢 0
薦骨 4
薧 10
薨 34
薩 1
薩 1166
薩丁尼亞 10
薩丁尼亞王國 10
薩伊 29
薩克司風 2
薩克斯管 1
薩克斯風 82
薩克遜人 0
薩利 6
薩姆 25
薩彥嶺 0
薩拉 87
薩滿 1
薩爾 128
薩爾瓦多 146
薩迦耶見 1
薩那 0
薩鎮冰 4
薩齊瑪 0
薪 937
薪俸 39
薪傳 238
薪優 7
薪柴 8
薪水 390
薪水階級 11
薪津 51
薪火 31
薪火相傳 47
薪給 51
薪資 2261
薪資表 3
薪酬 26
薪金 22
薪餉 12
薫 0
薬 0
薬 0
薯 1099
薯條 95
薰 1265
薰天 2
薰心 5
薰染 2
薰染法 0
薰爐 4
薰衣草 0
薰陶 67
薰風 10
薰香 23
薱 61
薲 0
薲 0
薳 41
薴 89
薵 65
薶 52
薷 74
薸 84
薹 221
薺 169
薺 169
薻 0
薼 0
薽 84
薾 79
薿 22
藀 61
藁 0
藂 34
藃 30
藄 39
藅 71
藆 68
藇 39
藈 50
藉 2638
藉 2638
藉以 792
藉助 0
藉口 336
藉故 37
藉故推辭 0
藉機 123
藉此 1088
藉此機會 127
藉由 3496
藉端 3
藉端生事 0
藉著 1783
藉藉 0
藉詞 12
藉資 12
藉酒澆愁 0
藊 0
藋 44
藌 0
藍 5427
藍光 83
藍圖 484
藍天 739
藍寶石 87
藍山 107
藍布 6
藍本 51
藍板 1
藍燈 119
藍牙 0
藍田種玉 0
藍白色 9
藍皮書 22
藍綠 46
藍綠色 36
藍綠藻 12
藍腹鷳 1
藍色 2017
藍芽 0
藍莓 32
藍藍 81
藍藻 45
藍銅礦 0
藍青 6
藍靛 2
藍領 38
藍鯨 72
藎 61
藏 4067
藏 4067
藏不住 18
藏人 35
藏住 2
藏傳佛教 1
藏入 6
藏北高原 3
藏匿 70
藏匿在 3
藏匿於 7
藏匿處 1
藏南縱谷 5
藏址信息 0
藏嬌 6
藏寶 47
藏寶圖 0
藏拙 1
藏文 34
藏族 59
藏族人 0
藏書 283
藏書室 4
藏書庫 5
藏書樓 5
藏書標記 0
藏書量 140
藏歷 1
藏污 1
藏污納垢 11
藏私 15
藏經 159
藏胞 12
藏著 79
藏藍 2
藏藏 5
藏藏躲躲 0
藏語 22
藏諸名山 0
藏起 22
藏起來 26
藏身 52
藏身之處 3
藏進 4
藏進來 0
藏進去 0
藏醫 8
藏量 6
藏量豐富 0
藏鋒 1
藏鏡人 0
藏青 2
藏青色 1
藏頭露尾 0
藏頭露尾 0
藏首露尾 0
藐 117
藐小 2
藐藐 3
藐視 62
藐視一切 0
藑 100
藒 193
藕 408
藕斷 0
藕斷絲連 21
藕灰 0
藕節 2
藕粉 17
藕絲 0
藕色 1
藕花 2
藖 0
藖 0
藗 35
藘 23
藙 51
藚 49
藛 0
藜 102
藜科 5
藝 6946
藝事 0
藝人 1024
藝博不精 0
藝名 40
藝品 365
藝品店 24
藝壇 432
藝妓 24
藝專 58
藝展 31
藝工隊 6
藝廊 0
藝徒 1
藝文 8847
藝文志 20
藝林 30
藝能 302
藝苑 25
藝術 15365
藝術化 45
藝術品 376
藝術學院 1074
藝術家 2196
藝術專科學校 39
藝術廳 9
藝術性 63
藝術界 81
藝術系 78
藝術美 7
藝術者 9
藝術院 6
藝術館 464
藝道 2
藝高人膽大 4
藝高膽大 1
藝齡 23
藞 32
藟 66
藠 0
藡 71
藢 48
藣 40
藤 12162
藤子 5
藤架 1
藤條 5
藤椅 21
藤皮 1
藤罩 0
藤蔓 15
藤蘿 3
藤製 15
藤鞭 0
藥 50
藥 76336
藥丸 88
藥典 36
藥到病除 6
藥劑 752
藥劑學 29
藥劑師 37
藥力 13
藥包 12
藥名 21
藥味 17
藥品 3441
藥品店 1
藥商 116
藥單 3
藥婆 0
藥學 791
藥學專科學校 138
藥學系 200
藥害 48
藥專 7
藥局 1776
藥師 813
藥店 13
藥廠 396
藥引 12
藥引子 1
藥性 64
藥房 159
藥效 179
藥效持久 0
藥方 84
藥材 220
藥棉 5
藥水 181
藥油 2
藥液 35
藥液 35
藥渣 2
藥片 33
藥物 4453
藥物學 22
藥王 24
藥王菩薩 1
藥理 538
藥理學 143
藥瓶 20
藥瓶子 0
藥用 211
藥皂 3
藥盒 7
藥石 6
藥石罔效 0
藥科 22
藥箱 213
藥粉 16
藥糖 5
藥線 1
藥罐 6
藥罐子 2
藥膏 72
藥舖 36
藥茶 21
藥草 251
藥行 147
藥販 3
藥費 12
藥酒 54
藥量 76
藥針 0
藥鋪 5
藥頭 11
藥餌 1
藦 81
藦 81
藧 0
藨 81
藩 750
藩國 2
藩屬 3
藩屬國 0
藩籬 44
藩鎮 4
藪 212
藫 134
藬 385
藭 149
藮 0
藯 29
藰 81
藱 77
藱 77
藲 34
藳 0
藴 0
藴 0
藶 63
藷 477
藷 477
藸 49
藹 245
藹然 1
藹然可觀 0
藹藹 2
藺 120
藺相如 1
藻 489
藻井 13
藻思 0
藻繪 0
藻菌 1
藻類 419
藻類學 1
藻類植物 0
藻飾 0
藻麗 0
藼 0
藽 79
藾 164
藿 204
蘀 69
蘁 67
蘁 67
蘂 0
蘃 0
蘄 68
蘅 96
蘆 1
蘆 368
蘆叢 0
蘆席 0
蘆柴 0
蘆洲 468
蘆洲鄉 626
蘆溝橋 5
蘆溝橋事變 2
蘆竹 973
蘆笙 3
蘆笛 6
蘆筍 70
蘆筍汁 10
蘆管兒 0
蘆管兒 0
蘆花 6
蘆花絮 0
蘆花馬 0
蘆荻 2
蘆葦 89
蘆薈 151
蘆薈露 1
蘇 7421
蘇丹 75
蘇丹人 2
蘇伊士 2
蘇伊士運河 6
蘇俄 99
蘇利南 19
蘇北 16
蘇區 1
蘇哈托 548
蘇堤 4
蘇峻 1
蘇州 236
蘇州人 3
蘇州城 2
蘇州市 21
蘇州碼 0
蘇彝士運河 1
蘇必略湖 0
蘇打 47
蘇打 47
蘇打水 12
蘇打粉 15
蘇方 2
蘇曼殊 3
蘇木 18
蘇杭 3
蘇東坡 98
蘇格拉底 34
蘇格蘭 353
蘇武 27
蘇比克灣 0
蘇澳 682
蘇澳港 60
蘇秦 18
蘇維埃 34
蘇維埃聯邦 0
蘇羅河 0
蘇美人 1
蘇聯 662
蘇聯人 7
蘇花公路 85
蘇軾 90
蘇轍 8
蘇醒 4
蘇醒劑 0
蘇醒過來 1
蘇里南 3
蘇鐵 50
蘇門答臘 48
蘇非教派 0
蘇黎世 76
蘈 0
蘉 72
蘊 308
蘊含 120
蘊含著 16
蘊涵 37
蘊涵著 14
蘊積 2
蘊藉 3
蘊藏 150
蘊藏量 30
蘊釀 61
蘋 456
蘋 456
蘋果 2205
蘋果園 3
蘋果樹 27
蘋果汁 18
蘋果皮 6
蘋果綠 50
蘋果肉 0
蘋果螺 1
蘋果酒 21
蘋果醬 1
蘌 44
蘍 0
蘎 0
蘏 0
蘐 0
蘑 72
蘑菇 52
蘓 0
蘕 0
蘖 0
蘗 34
蘗 34
蘘 10
蘙 82
蘚 11
蘚苔 6
蘚苔植物 4
蘚類 0
蘛 62
蘜 10
蘝 0
蘝 0
蘞 8
蘞 8
蘟 11
蘠 7
蘡 8
蘢 613
蘣 14
蘣 14
蘤 279
蘥 8
蘦 9
蘧 11
蘨 0
蘩 8
蘪 10
蘫 0
蘫 0
蘬 1
蘬 1
蘭 5456
蘭亭集序 6
蘭吉雅 38
蘭室 6
蘭嶼 1146
蘭州 96
蘭州市 3
蘭房 2
蘭斯 64
蘭新鐵路 1
蘭桂 57
蘭溪 49
蘭盆會 0
蘭科 30
蘭考 0
蘭花 209
蘭花草 5
蘭芷 0
蘭若 39
蘭草 7
蘭譜 1
蘭陽 785
蘭陽溪 78
蘭麝 1
蘮 6
蘯 0
蘱 5
蘲 2
蘲 2
蘳 4
蘳 4
蘳 4
蘴 3
蘵 11
蘶 1
蘸 53
蘸上 1
蘸水 1
蘸水筆 1
蘸濕 0
蘸筆 2
蘹 43
蘹香 1
蘺 15
蘻 9
蘼 20
蘼蕪 6
蘽 0
蘾 16
蘿 430
蘿菠 4
蘿蔔 191
蘿蔔 191
蘿蔔乾 11
蘿蔔乾 11
蘿蔔糕 1
蘿蔔腿 5
蘿蔔腿 5
虀 11
虁 0
虂 0
虃 67
虆 1
虇 6
虇 6
虇 6
虈 4
虉 0
虊 0
虋 96
虌 79
虍 6
虎 2644
虎丘 6
虎入羊群 2
虎列拉 0
虎口 40
虎口拔牙 1
虎口餘生 2
虎嘯 37
虎嘴 0
虎姑婆 15
虎威 15
虎子 12
虎將 28
虎尾 1410
虎尾草 4
虎年 97
虎式機 0
虎林 90
虎檻 0
虎死留皮 0
虎毒不食子 1
虎爪 8
虎父無犬子 3
虎父犬子 0
虎牙 9
虎狼 2
虎皮 38
虎眼石 1
虎穴 191
虎符 2
虎紋 17
虎耳 8
虎耳草 4
虎背 3
虎背熊腰 1
虎膽 9
虎膽妙算 26
虎落平陽 4
虎落平陽被犬欺 3
虎虎 55
虎虎生風 0
虎視眈眈 19
虎門 31
虎鞭 2
虎頭 61
虎頭埤 48
虎頭山 49
虎頭虎腦 0
虎頭蛇尾 21
虎頭蜂 11
虎骨 25
虎骨酒 1
虎鬚 5
虎鯊 18
虐 432
虐刑 0
虐待 379
虐待狂 8
虐政 0
虐暑 0
虑 0
虒 166
虓 91
虔 191
虔信 2
虔婆 3
虔心 4
虔敬 39
虔文 0
虔誠 213
處 24669
處 24669
處世 135
處世之道 5
處之泰然 4
處了 5
處事 172
處分 1782
處分法 0
處刑 17
處到 9
處在 409
處境 544
處士 9
處女 460
處女作 29
處女地 30
處女座 775
處女群島 0
處女膜 26
處女膜 26
處女航 20
處子 13
處得 644
處得來 0
處心積慮 35
處所 833
處斬 1
處斷 15
處方 692
處方箋 0
處於 1176
處暑 2
處死 69
處死刑 35
處決 21
處治 4
處理 29034
處理 29034
處理器 1855
處理器 1855
處理機 256
處理機 256
處理法 158
處理法 158
處理表 8
處理表 8
處理部件 0
處理部件 0
處男 0
處的 643
處置 1258
處置法 5
處罰 970
處處 928
處處作對 0
處變不驚 11
處過 4
處長 905
處高鶩遠 0
虖 2
虗 0
虘 0
虙 2
虚 0
虛 1255
虛位 9
虛位以待 6
虛假 90
虛偽 292
虛像 23
虛列 11
虛力 1
虛名 20
虛報 69
虛報年齡 0
虛妄 37
虛字 3
虛實 104
虛左以待 0
虛幻 177
虛幻不實 7
虛幻境界 0
虛幻性 0
虛度 10
虛度一生 0
虛度一生 0
虛度年華 0
虛度歲月 0
虛度青春 0
虛座以待 0
虛弱 161
虛弱不堪 2
虛張 3
虛張聲勢 17
虛心 117
虛心學習 16
虛情 7
虛情假意 19
虛想 1
虛憍恃氣 1
虛應 7
虛應故事 7
虛懷 0
虛懷若谷 8
虛擬 3795
虛擬實境 0
虛擬詞 0
虛擬語氣 0
虛擲 14
虛數 4
虛文 2
虛晃 2
虛晃一下 0
虛晃一下 0
虛晃一招 7
虛晃一招 7
虛晃一槍 0
虛晃一槍 0
虛有 6
虛有其表 15
虛根 2
虛榮 46
虛榮心 12
虛構 185
虛構情節 4
虛歲 10
虛汗 2
虛浮 19
虛渺 3
虛火 6
虛無 100
虛無主義 14
虛無縹渺 3
虛無縹緲 8
虛無飄渺 12
虛空 165
虛線 40
虛耗 15
虛聲恫喝 0
虛胖 8
虛脫 39
虛與委蛇 5
虛虛 6
虛虛實實 16
虛言 8
虛設 30
虛詐 1
虛詞 10
虛話 2
虛誇 1
虛誕 3
虛辭 0
虛造 0
虛雲大師 3
虛靡 0
虛飾 3
虛驚 27
虛驚一場 36
虛驚一場 36
虛驚一場 36
虛驚一場 36
虛點 2
虜 6146
虜掠 1
虜獲 3
虝 0
虞 825
虞侯 0
虞國 2
虞姬 10
虞犯 6
虞美人 12
虞舜 3
號 23
號 85237
號令 524
號令如山 0
號兵 0
號叫 21
號召 242
號哭 3
號啕 1
號啕大哭 6
號外 176
號子 168
號徽 0
號手 13
號數 83
號旗 2
號為 93
號燈 7
號碼 5002
號碼機 3
號碼牌 12
號稱 545
號聲 17
號角 88
號角聲 2
號誌 295
號音 4
虠 0
虡 135
虢 25
虣 4
虤 1
虥 8
虦 0
虧 690
虧了 29
虧到 1
虧在 1
虧失 0
虧少 1
虧待 13
虧得 11
虧心 0
虧心事 8
虧折 0
虧掉 2
虧損 883
虧損累累 7
虧損額 0
虧本 19
虧本生意 0
虧欠 49
虧盈 1
虧短 0
虧空 12
虧空公款 6
虧累 1
虧缺 9
虧耗 0
虧蝕 1
虧負 5
虨 2
虩 84
虪 89
虫 758
虫 758
虫兒 1
虫子 3
虫體 2
虬 0
虬 0
虬 0
虭 104
虮 348
虯 118
虯髯 18
虰 76
虱 70
虱子 6
虱目魚 175
虳 884
虴 867
虵 0
虵 0
虷 20
虸 0
虹 1791
虹口 5
虹吸 147
虹吸現象 1
虹吸管 7
虹彩 69
虹彩流輝 0
虹橋 122
虹霓 17
虹鱒 19
虺 11
虻 7
虼 1110
虽 0
虽 0
蚄 0
蚄 0
蚅 85
蚆 143
蚇 69
蚈 0
蚉 0
蚊 375
蚊子 331
蚊帳 146
蚊蚋 4
蚊蟲 59
蚊蠅 10
蚊香 25
蚊香盒 0
蚋 94
蚌 93
蚌埠 3
蚌殼 12
蚌珠 0
蚍 49
蚍 49
蚍蜉 1
蚍蜉 1
蚎 49
蚏 0
蚐 99
蚑 274
蚑 274
蚒 0
蚓 84
蚔 292
蚔 292
蚕 38
蚖 162
蚗 177
蚘 206
蚙 63
蚙 63
蚚 760
蚚蜴 0
蚛 0
蚜 11
蚜蟲 16
蚝 42
蚞 89
蚟 0
蚠 0
蚡 100
蚢 167
蚣 6
蚤 118
蚥 109
蚦 0
蚦 0
蚧 143
蚨 87
蚩 15
蚩尤 12
蚪 5
蚭 0
蚮 0
蚯 8
蚯蚓 124
蚯蚓洞 0
蚰 297
蚰蜒 0
蚱 1
蚱 5
蚱蜢 30
蚱蟬 0
蚲 0
蚳 285
蚴 63
蚵 181
蚵 181
蚵仔 77
蚵仔 77
蚵仔 77
蚵仔 77
蚵仔煎 26
蚵仔煎 26
蚵仔煎 26
蚵仔煎 26
蚶 44
蚶田 0
蚷 2
蚸 69
蚹 86
蚺 301
蚻 57
蚼 64
蚽 97
蚾 148
蚿 0
蛀 8130
蛀洞 5
蛀牙 190
蛀蝕 8
蛀蟲 17
蛀食 4
蛀齒 2
蛁 219
蛂 3331
蛃 39
蛄 14
蛅 172
蛆 47
蛆 47
蛇 12
蛇 1260
蛇口蜂針 0
蛇名 0
蛇吞象 6
蛇寶石 0
蛇年 2
蛇座 11
蛇形 15
蛇無頭不行 0
蛇王 17
蛇皮 10
蛇紋 57
蛇肉 8
蛇膽 10
蛇蠍 13
蛇蠍美人 2
蛇行 19
蛇足 3
蛇鞭 4
蛇頭 90
蛇類 12
蛇鼠一窩 1
蛈 82
蛉 31
蛊 0
蛊 0
蛋 4205
蛋包 13
蛋包飯 4
蛋品 18
蛋形 8
蛋捲 25
蛋殼 41
蛋清 6
蛋炒飯 1
蛋白 851
蛋白原 0
蛋白尿 89
蛋白石 19
蛋白脢 9
蛋白質 964
蛋糕 1055
蛋糕盒 1
蛋花 2
蛋花湯 7
蛋酒 2
蛋餅 1
蛋黃 173
蛋黃素 2
蛋黃酥 1
蛋黃醬 6
蛌 292
蛐 17
蛐 17
蛐蛐兒 0
蛐蛐兒 0
蛑 43
蛒 0
蛓 30
蛔 47
蛔蟲 36
蛔蟲病 0
蛔蟲藥 0
蛕 0
蛖 5
蛗 44
蛘 40
蛙 551
蛙人 6
蛙人操 0
蛙人隊 0
蛙人鞋 0
蛙叫 0
蛙叫聲 0
蛙式 30
蛙式比賽 2
蛙泳 32
蛙王 0
蛙肉 0
蛙腿 4
蛙跳 8
蛙鞋 39
蛙鳴 11
蛚 60
蛛 44
蛛絲馬跡 36
蛛網 27
蛛網狀 0
蛜 38
蛝 45
蛞 11
蛞 11
蛞蝓 8
蛟 70
蛟龍 22
蛟龍得水 0
蛠 0
蛡 0
蛢 52
蛣 161
蛤 316
蛤 316
蛤粉 0
蛤蚌 0
蛤蚧 13
蛤蜊 24
蛤蜊 24
蛤蜊湯 13
蛤蟆 29
蛤蠣 3
蛥 0
蛦 80
蛧 0
蛨 141
蛩 85
蛪 92
蛫 83
蛬 59
蛭 39
蛮 0
蛯 1
蛵 2
蛶 1
蛷 6
蛸 16
蛸 16
蛹 297
蛺 10
蛺蝶 47
蛻 207
蛻 207
蛻化 6
蛻化 6
蛻化變質 1
蛻殼 4
蛻皮 4
蛻變 1
蛻變 280
蛼 0
蛽 0
蛾 914
蛾子 0
蛾眉 4
蛾眉月 0
蛾眉粉黛 0
蛿 0
蛿 0
蜀 744
蜀中無大將 1
蜀人 6
蜀國 3
蜀漢 28
蜀犬吠日 1
蜀道 19
蜀道難行 0
蜀魏 1
蜁 6
蜂 894
蜂刺 2
蜂后 0
蜂場 0
蜂場 0
蜂巢 114
蜂巢形 0
蜂巢胃 2
蜂房 5
蜂擁 33
蜂擁而上 5
蜂湧 5
蜂湧而上 0
蜂炮 35
蜂王 16
蜂王乳 73
蜂王漿 29
蜂窩 63
蜂窩煤 0
蜂箱 0
蜂群 18
蜂聚 1
蜂腰 4
蜂膠 0
蜂舞 0
蜂蜜 310
蜂蠟 2
蜂起 0
蜂鳥 28
蜂鳴器 32
蜃 99
蜃氣 6
蜄 6
蜅 7
蜆 305
蜆妹 1
蜆精 1
蜇 208
蜇螯部 0
蜈 504
蜈蚣 61
蜉 13
蜉蝣 14
蜊 1149
蜋 3
蜌 9
蜍 17
蜎 2
蜏 0
蜐 0
蜑 8
蜑戶 0
蜑民 0
蜒 21
蜒蚰 0
蜓 339
蜔 0
蜕 0
蜕 0
蜖 0
蜘 15
蜘蛛 270
蜘蛛精 2
蜘蛛絲 3
蜘蛛網 106
蜘蛛類 1
蜘蜘 0
蜙 4
蜚 28
蜚 28
蜚廉 0
蜚短流長 6
蜚聲中外 10
蜚聲國際 12
蜚蠊 9
蜚言蜚語 0
蜚語 0
蜛 9
蜜 1245
蜜月 292
蜜月旅行 41
蜜柑 1
蜜棗 5
蜜汁 33
蜜源 14
蜜漬 2
蜜粉 1
蜜糖 32
蜜蜂 214
蜜蜂窩 3
蜜蠟 4
蜜語 42
蜜餞 98
蜝 0
蜞 40
蜟 0
蜠 30
蜡 30
蜡 30
蜢 86
蜣 111
蜤 10
蜥 168
蜥蜴 91
蜥蝪 16
蜥蟀 0
蜦 82
蜧 8
蜨 115
蜩 37
蜩螗 3
蜩螗沸羹 1
蜪 9
蜫 0
蜬 24
蜭 11
蜮 15
蜯 0
蜯 0
蜰 14
蜱 36
蜲 16
蜳 103
蜴 99
蜵 6
蜶 0
蜷 61
蜷伏 4
蜷曲 12
蜷縮 8
蜸 22
蜹 0
蜹 0
蜺 12
蜻 58
蜻蛉 20
蜻蜓 99
蜻蜓點水 21
蜼 13
蜽 0
蜾 4
蜾蠃 0
蜿 330
蜿 330
蜿蜒 108
蜿豌豆 0
蝀 18
蝁 6
蝂 7
蝃 13
蝅 0
蝆 16
蝇 0
蝌 234
蝌蚪 33
蝌蚪文 1
蝍 6
蝍 6
蝎 113
蝎 113
蝏 229
蝐 65
蝑 45
蝒 50
蝓 121
蝔 45
蝕 556
蝕刻 0
蝕刻師 0
蝕成 1
蝕損 0
蝕本 6
蝖 222
蝗 156
蝗害 1
蝗災 10
蝗蟲 43
蝘 59
蝙 222
蝙蝠 245
蝙蝠俠 132
蝙蝠洞 48
蝚 75
蝛 54
蝜 256
蝝 259
蝞 35
蝟 149
蝟毛 0
蝠 539
蝡 94
蝢 145
蝣 17
蝤 14
蝤 14
蝤蛑 0
蝤蛑 0
蝤蠐 1
蝥 319
蝥 319
蝦 158
蝦 1739
蝦丸 3
蝦仁 142
蝦子 74
蝦排 5
蝦殼 2
蝦球 29
蝦米 209
蝦群 1
蝦肉 9
蝦蟆 10
蝦餃 6
蝦餅 9
蝧 56
蝨 291
蝨子 9
蝨蠅 0
蝩 91
蝪 245
蝫 5
蝬 200
蝭 42
蝮 85
蝯 164
蝰 0
蝱 0
蝲 0
蝳 68
蝴 279
蝴蝶 1022
蝴蝶結 30
蝴蝶花 11
蝴蝶蘭 109
蝵 75
蝶 2425
蝶式 42
蝶形 13
蝶形花冠 0
蝶泳 40
蝶粉 2
蝶衣 9
蝶類 33
蝷 178
蝷 178
蝸 875
蝸居 24
蝸牛 561
蝸角 1
蝹 6
蝺 122
蝻 59
蝼 0
蝽 0
蝽 0
螁 12
螁 12
螂 1
螂 265
螃 123
螃蜞 0
螃蟹 223
螃蟹殼 1
螃蟹肉 0
螄 8
螄 8
螅 5
螅 5
螆 0
螇 12
螈 6
螉 31
螊 0
螋 0
螌 0
融 2982
融入 510
融券 0
融化 127
融化劑 0
融合 1029
融合線 0
融和 27
融會 31
融會貫通 30
融洽 107
融液 0
融液 0
融為 8
融為一體 25
融為一體 25
融融 49
融解 19
融解熱 1
融解點 0
融貫 13
融資 1936
融通 248
融雪 6
螎 0
螏 0
螐 12
螑 7
螒 1
螓 16
螔 23
螕 0
螖 26
螗 2
螘 6
螙 0
螙 0
螙 0
螚 4
螛 4
螜 12
螝 6
螞 149
螞 149
螞 149
螞蚱 0
螞蚱 0
螞蝗 7
螞螂 0
螞螂 0
螞蟥 4
螞蟻 396
螞蟻上樹 396
螞蟻窩 5
螟 95
螟蛉 2
螟蛾 9
螟蟲 2
螠 0
螡 0
螢 644
螢光 528
螢光幕 88
螢光板 0
螢光燈 82
螢光筆 0
螢光粉 8
螢光鏡 0
螢幕 4948
螢幕上 550
螢橋 14
螢橋國中 4
螢橋國小 7
螢火蟲 275
螢火蟲兒 0
螢火蟲兒 0
螢石 17
螢窗 18
螣 9
螣 9
螤 63
螪 2
螫 129
螫 129
螫人 4
螫毒 0
螫眼睛 0
螫針 0
螬 28
螭 25
螮 4
螯 133
螰 2
螱 0
螲 6
螲蟷 1
螳 36
螳臂擋車 12
螳臂當車 2
螳螂 56
螳螂捕蟬 11
螴 3
螵 2
螵蛸 3
螶 5
螷 4
螸 5
螹 14
螺 1123
螺圈 0
螺帽 215
螺旋 299
螺旋式 67
螺旋彈簧 13
螺旋形 11
螺旋桿菌 47
螺旋槳 56
螺旋線 5
螺旋菌 16
螺旋體 43
螺栓 100
螺母 8
螺紋 104
螺絲 791
螺絲刀 2
螺絲帽 34
螺絲起子 92
螺絲釘 50
螺絲鑽 0
螺線 2
螺線管 0
螺號 2
螺螄 3
螺螄 3
螺距 10
螺釘 43
螺髻 0
螻 66
螻蛄 2
螻蟈 0
螻蟻 13
螻蟻兒 0
螻蟻兒 0
螻蟻尚且偷生 0
螼 5
螽 50
螽斯 7
螾 3
螿 630
蟀 273
蟀 273
蟁 0
蟂 0
蟃 9
蟄 24
蟄 24
蟄伏 29
蟄伏 29
蟄居 6
蟄居 6
蟄雷 0
蟄雷 0
蟅 5
蟆 761
蟆 7611
蟇 0
蟇 0
蟇 0
蟈 74
蟈蟈 11
蟈蟈兒 0
蟈蟈兒 0
蟉 2
蟊 7
蟋 149
蟋蟀 46
蟋蟀草 1
蟌 6
蟍 0
蟎 0
蟑 99
蟑螂 359
蟒 100
蟒蛇 38
蟒袍 3
蟓 95
蟔 27
蟕 0
蟗 25
蟘 73
蟙 62
蟚 0
蟛 98
蟜 102
蟝 28
蟞 4
蟟 32
蟠 90
蟠據 2
蟠曲 1
蟠桃 14
蟠桃會 3
蟠踞 3
蟠龍 20
蟡 147
蟢 109
蟣 105
蟤 46
蟥 142
蟦 34
蟧 56
蟨 57
蟩 0
蟪 364
蟫 37
蟫 37
蟬 271
蟬兒 10
蟬兒 10
蟬嘶 0
蟬娟 3
蟬紗 0
蟬翼 5
蟬聯 104
蟬聯冠軍 8
蟬蛻 2
蟬連 32
蟬鳴 21
蟬鳴聲 0
蟭 161
蟮 0
蟮 0
蟯 65
蟯蟲 32
蟰 0
蟱 0
蟲 2631
蟲兒 15
蟲兒 16
蟲卵 65
蟲媒花 1
蟲子 35
蟲害 133
蟲情 0
蟲災 5
蟲聲 8
蟲膠 1
蟲蛭 0
蟲蝕 45
蟲類 16
蟲體 48
蟲鳴 39
蟳 229
蟴 45
蟶 41
蟷 20
蟸 0
蟸 0
蟹 2017
蟹殼 4
蟹肉 22
蟹肉湯 0
蟹螯 3
蟹行 2
蟹青 1
蟹黃 13
蟺 35
蟺 35
蟻 425
蟻丘 1
蟻塚 0
蟻穴 2
蟻蠶 1
蟻象 9
蟻酸 68
蟼 8
蟼 8
蟽 0
蟾 172
蟾兔 0
蟾桂 0
蟾蜍 39
蟿 12
蠀 36
蠁 53
蠂 83
蠂 83
蠃 43
蠄 0
蠅 1762
蠅拍 0
蠅營狗苟 1
蠅虎 0
蠅頭 2
蠅頭之利 0
蠅頭小利 3
蠅頭小楷 1
蠆 18
蠇 0
蠈 97
蠉 110
蠊 9
蠋 72
蠌 21
蠍 263
蠍子 113
蠎 0
蠏 0
蠏 0
蠐 19
蠑 58
蠑螈 13
蠑蠑 1
蠒 0
蠓 41
蠓艋 0
蠔 86
蠔油 26
蠔田 1
蠕 73
蠕 73
蠕動 92
蠕形動物 0
蠕蟲 43
蠕蟲形 0
蠕行 0
蠖 53
蠖 53
蠗 35
蠗 35
蠘 0
蠙 52
蠙 52
蠚 0
蠚 0
蠛 3
蠜 3
蠝 5
蠞 0
蠟 504
蠟丸 1
蠟人 0
蠟像 5
蠟像館 11
蠟封 1
蠟布 0
蠟書 0
蠟板 0
蠟染 13
蠟梅 0
蠟油 7
蠟炬 5
蠟燭 219
蠟燭台 0
蠟版 0
蠟筆 189
蠟筆畫 7
蠟紙 6
蠟膏 0
蠟芯兒 0
蠟芯兒 0
蠟質 5
蠟黃 2
蠟黃色 0
蠠 4
蠡 15
蠡 15
蠡測 6
蠢 108
蠢動 49
蠢夫 0
蠢拙 0
蠢材 3
蠢物 1
蠢笨 0
蠢蛋 8
蠢蟲 0
蠢蠢 11
蠢蠢欲動 71
蠢話 3
蠢貨 1
蠣 44
蠣奴 0
蠣粉 0
蠣黃 0
蠤 5
蠤 5
蠥 0
蠦 2
蠧 0
蠧 0
蠧 0
蠨 2
蠩 5
蠪 3
蠫 4
蠫 4
蠬 4
蠭 0
蠭起 0
蠮 4
蠯 63
蠰 13
蠰 13
蠱 89
蠱惑 48
蠱惑人心 2
蠱毒 5
蠲 19
蠳 2
蠵 296
蠶 281
蠶卵 3
蠶子 0
蠶室 1
蠶寶寶 16
蠶山 0
蠶座 0
蠶桑 13
蠶沙 0
蠶眠 0
蠶種 8
蠶箔 0
蠶紙 0
蠶絲 113
蠶絲被 8
蠶繭 6
蠶蔟 0
蠶蛹 0
蠶蛹油 0
蠶蛾 43
蠶蟻 0
蠶豆 83
蠶豆象 0
蠶豆酥 1
蠶食 19
蠶食鯨吞 8
蠷 105
蠸 81
蠹 27
蠹政 0
蠹簡 0
蠹蟲 3
蠹魚 1101
蠺 0
蠻 1870
蠻人 3
蠻刀 0
蠻力 18
蠻勁 0
蠻勇 2
蠻國 0
蠻地 1
蠻夷 3
蠻常 0
蠻幹 13
蠻幹到底 0
蠻族 18
蠻橫 40
蠻橫無理 2
蠻牛 34
蠻狄 0
蠻荒 49
蠻荒之地 3
蠻行 1
蠻風 0
蠼 36
蠽 30
蠽 30
蠾 189
蠿 30
蠿 30
血 5301
血 5301
血仇 6
血債 7
血債血還 3
血刃 3
血印 3
血友 24
血友病 48
血口噴人 2
血吸蟲 8
血型 1186
血塊 40
血壓 599
血壓低 1
血壓計 77
血壓高 15
血小板 207
血尿 133
血崩 3
血庫 62
血循環 21
血性 110
血性男兒 0
血戰 33
血戰到底 0
血拼 0
血暈 0
血書 2
血本 18
血本無歸 28
血染沙場 0
血染沙場 0
血栓 148
血栓症 8
血案 107
血氣 38
血氣之勇 3
血氣方剛 26
血水 25
血汗 36
血汗錢 28
血汙 0
血污 8
血泊 20
血泊 20
血泊 20
血洗 14
血洗台灣 1
血流如注 10
血流成河 0
血流成河 0
血流漂杵 1
血海 25
血海深仇 6
血液 1841
血液 1841
血液 1841
血液循環 197
血液循環 197
血液癌 0
血液癌 0
血液科 59
血液科 59
血淋淋 41
血淚 68
血淚交織 3
血淚史 19
血清 604
血清蛋白 9
血源 18
血滴 19
血漬 13
血漿 175
血濃於水 9
血球 161
血痕 12
血癌 60
血癌 60
血盆 0
血盆大口 3
血祭 11
血科 0
血管 2071
血管硬化 42
血糖 669
血紅 42
血紅素 114
血紅蛋白 11
血統 121
血統書 233
血統證明 4
血統證明書 2
血絲 38
血線蟲 3
血緣 77
血緣關係 38
血肉 58
血肉之軀 7
血肉模糊 0
血肉橫飛 12
血肉相連 3
血肉糢糊 2
血脈 86
血脈 86
血脈賁張 0
血腥 135
血腥味 6
血腥氣 2
血腸 0
血色 54
血色素 60
血荒 2
血行器 0
血衣 3
血親 132
血詔 1
血跡 63
血跡斑斑 7
血路 40
血輪 15
血量 40
血鈣質 0
血雨腥風 0
衁 1
衂 0
衂 0
衃 0
衄 1
衅 0
衆 0
衇 0
衇 0
衈 0
衉 0
衊 13
衋 133
行 3032
行 34
行 34993
行 34993
行不得 6
行不得 6
行不得也 8
行不得也 8
行不改姓 0
行不由徑 0
行不行 44
行不通 78
行之有年 52
行之有效 4
行乞 8
行了 375
行事 558
行事曆 1348
行人 378
行人穿越道 18
行人道 7
行伍 3
行伍出身 0
行住坐臥 11
行使 927
行使政權 6
行使職權 59
行俠仗義 0
行兇 29
行兇前 1
行兇後 0
行兇殺人 0
行凶 23
行刑 40
行列 1063
行列式 44
行列輸入法 10
行刺 17
行劫 2
行動 7272
行動不便 80
行動劇 1
行動敏捷 1
行動自如 10
行動遲緩 5
行動電話 0
行同 9
行吟 8
行吟詩人 1
行員 102
行商 65
行商 65
行善 184
行善不欲人知 0
行囊 100
行在 82
行大禮 3
行天宮 98
行好 6
行孝 10
行客 24
行宮 18
行家 322
行寬 24
行將 143
行將完成 0
行將就木 6
行將崩潰 0
行尸走肉 1
行屍走肉 15
行市 7
行年 11
行庫 737
行徑 113
行得 24
行得通 21
行情 8217
行情看漲 15
行情看跌 0
行情表 1592
行憲 54
行憲紀念日 57
行房 36
行搶 0
行政 13461
行政區 581
行政區域 248
行政區畫 2
行政命令 153
行政單位 1473
行政學 117
行政官 3
行政局 301
行政效率 250
行政機關 549
行政權 74
行政法 194
行政法規 78
行政法院 284
行政當局 18
行政管理 1275
行政處分 118
行政訴訟 107
行政部長 0
行政部門 346
行政長官 110
行政院 76883
行政院長 740
行數 65
行文 209
行文流暢 2
行旅 49
行易知難 1
行星 515
行時 51
行書 119
行會 18
行有行規 0
行有餘力 10
行期 11
行李 673
行李房 0
行李捲 0
行李架 8
行李箱 75
行李袋 2
行李車 2
行杖 0
行東 5
行板 17
行業 4703
行楷 6
行樂 12
行樹 3
行檢 216
行止 27
行灶 1
行為 7717
行為不檢 18
行為主義 14
行為犯 0
行為科學 116
行為能力 104
行為責任 7
行照 124
行營 7
行狀 4
行獵 2
行略 3
行百里 5
行百里半九十 0
行省 17
行社 27
行禮 54
行禮如儀 0
行程 3988
行程開關 0
行竊 19
行筆 5
行筆流暢 0
行篋 1
行經 153
行義 96
行義童子軍 12
行者 164
行聘 3
行腳 149
行腳僧 7
行船 56
行色 13
行色匆匆 9
行若無事 1
行草 11
行萬里路 29
行藏 10
行蘊 1
行號 787
行血 15
行行 48
行行 48
行行出狀元 11
行行好 24
行裝 34
行裝甫卸 0
行規 13
行詐 3
行話 5
行誼 23
行誼 23
行賄 35
行賈 0
行賞 3
行走 376
行走機構 1
行走路線 1
行距 31
行跡 26
行蹤 211
行蹤不定 2
行蹤不明 13
行蹤飄忽 1
行車 607
行車時間 20
行軍 36
行軍床 0
行速 2
行進 298
行進路線 7
行運 15
行道 94
行道樹 50
行遠必自邇 4
行部 35
行都 37
行酒 1
行酒令 0
行醫 56
行銷 7057
行銷國內 6
行銷國內外 5
行銷海內外 1
行銷網 74
行銷費用 9
行間 28
行雲流水 32
行頭 19
行館 95
行駛 745
行駛路線 31
行騙 10
衍 594
衍傳 0
衍射 11
衍文 3
衍生 886
衍生物 153
衍生詞 0
衍繹 0
衍義 15
衍變 14
衎 601
衏 0
衑 0
衒 48
術 6158
術士 50
術數 31
術業有專攻 1
術科 296
術語 332
衕 31
衖 60
衖堂 0
街 19420
街上 416
街亭 8
街口 146
街名 14
街坊 64
街市 26
街心 6
街景 90
街燈 32
街燈柱 0
街衢 5
街角 39
街談巷議 0
街路 42
街車 24
街道 578
街道上 82
街門 5
街頭 825
街頭上 7
街頭巷尾 30
衘 0
衙 239
衙 239
衙役 5
衙門 64
衙門作風 0
衚 0
衚衕 2
衛 596
衛兵 53
衛冕 76
衛冕賽 0
衛國 90
衛國保家 0
衛士 18
衛從 0
衛戌 2
衛戍 1
衛戍部隊 0
衛所 81
衛教 929
衛教系 6
衛星 3070
衛星國 0
衛星城 0
衛星工業 0
衛星通信 100
衛星電視 105
衛浴 289
衛浴設備 155
衛理 170
衛生 8416
衛生保健 387
衛生司 0
衛生大隊 0
衛生學 122
衛生局 1018
衛生所 787
衛生棉 0
衛生標準 27
衛生機關 65
衛生科 203
衛生紙 197
衛生署 2493
衛生處 583
衛生衣 1
衛生部 42
衛生防疫 6
衛生防護 5
衛生院 221
衛生隊 4
衛視 0
衛護 16
衛道 128
衛道之士 8
衛隊 46
衛青 5
衜 0
衝 139
衝 1399
衝上 80
衝上來 1
衝上去 21
衝下 13
衝下來 0
衝下去 2
衝來 11
衝來衝去 1
衝倒 2
衝出 102
衝出來 7
衝出去 2
衝出重圍 3
衝到 93
衝刺 267
衝剋 0
衝力 23
衝勁 47
衝勁十足 5
衝動 348
衝去 36
衝口 5
衝口而出 7
衝垮 6
衝堂 0
衝壓 18
衝天 21
衝天炮 0
衝掉 0
衝撞 94
衝撞力 1
衝擊 2339
衝擊力 31
衝擊韌性 1
衝散 3
衝決 4
衝浪 169
衝浪板 21
衝激 17
衝破 105
衝程 29
衝突 1781
衝突性 15
衝線 3
衝著 139
衝衝 2
衝要 2
衝起 4
衝起來 0
衝進 71
衝進來 10
衝進去 9
衝過 29
衝過來 12
衝過去 14
衝鋒 52
衝鋒槍 47
衝鋒號 0
衝鋒鎗 0
衝鋒陷陣 39
衞 0
衟 0
衠 0
衡 1184
衡宇 0
衡山 25
衡情 3
衡情度理 2
衡情論理 2
衡量 688
衡量得失 0
衡陽 39
衡陽路 63
衢 63
衢巷 0
衢道 1
衣 30
衣 34000
衣不蔽體 2
衣不蔽體 2
衣不解帶 0
衣冠 20
衣冠不整 2
衣冠冢 0
衣冠文物 0
衣冠楚楚 3
衣冠禽獸 5
衣分 1
衣刷 2
衣夾 4
衣履 5
衣帛 2
衣帶 16
衣帽 29
衣帽間 4
衣料 233
衣服 1422
衣架 79
衣櫃 121
衣櫥 79
衣物 437
衣物箱 0
衣箱 7
衣索匹亞 13
衣索比亞 12
衣紫腰金 1
衣紫腰金 1
衣紫腰銀 1
衣紫腰銀 1
衣紫腰黃 1
衣紫腰黃 1
衣缽 22
衣缽傳人 1
衣胞 0
衣著 330
衣著打扮 2
衣衫 52
衣衫不整 4
衣衫襤褸 7
衣袋 17
衣袖 55
衣裙 16
衣補兒 0
衣補兒 0
衣裝 6
衣裳 126
衣裳 126
衣褲 42
衣襟 40
衣角 13
衣錦榮歸 2
衣錦榮歸 2
衣錦還鄉 13
衣錦還鄉 13
衣領 26
衣食 34
衣食住行 18
衣食父母 8
衣飾 1389
衦 0
衧 752
表 38569
表兄 12
表兄弟 8
表冊 48
表出 14
表出來 1
表列 7038
表列出 2
表功 10
表叔 4
表同情 5
表哥 80
表土 43
表土層 4
表報 36
表妹 64
表姊 28
表姊妹 5
表姊妹 5
表姐 65
表姑 0
表姨 0
表姨父 0
表姪 1
表嫂 6
表嬸 7
表字 6
表尺 4
表層 160
表弟 46
表彰 126
表彰出來 0
表徵 346
表情 917
表意 34
表意文字 3
表態 303
表揚 713
表揚大會 61
表明 719
表格 3112
表決 410
表決權 98
表演 2916
表演性 1
表演藝術 617
表演賽 21
表率 52
表現 7267
表現出 526
表現出來 110
表現力 14
表現型 11
表現對象 0
表現形 2
表白 115
表皮 213
表盤 2
表相 27
表示 18080
表示出 39
表示出來 4
表章 21
表筆 1
表舅 1
表舅母 0
表表 150
表裡 41
表裡一致 4
表裡一致 4
表裡不一 9
表裡如一 4
表親 3
表觀 12
表記 6
表識 0
表象 70
表述 65
表達 2815
表達出 88
表達出來 71
表達力 5
表達能力 104
表針 1
表錯 4
表錯情 1
表露 72
表露 72
表露出 9
表露出 9
表露無遺 36
表露無遺 36
表面 2491
表面上 402
表面光潔 0
表面光潔度 0
表面化 13
表面工夫 6
表面張力 47
表面性 6
表面活性劑 276
表面積 74
表面質量 1
表音文字 0
表頭 56
表顯 3
衩 689
衪 822
衫 599
衫林 2
衫林溪 0
衫袖 1
衫褲 1
衭 192
衮 0
衯 10
衰 409
衰 409
衰世 1
衰亂 0
衰亡 10
衰年 2
衰弱 113
衰微 20
衰敗 41
衰敝 0
衰替 1
衰減 98
衰滅 1
衰疲 1
衰竭 417
衰老 76
衰落 31
衰蔽 0
衰變 36
衰退 1322
衰邁 0
衰頹 12
衰顏 1
衱 449
衱 449
衲 149
衲僧 0
衲衣 0
衲襖 0
衳 0
衳 0
衴 8
衵 152
衶 61
衷 94
衷心 327
衷情 10
衷曲 3
衸 0
衸 0
衹 60
衺 0
衺 0
衺 0
衺 0
衺 0
衻 0
衼 2
衽 17
衾 552
衿 126
衿帶 0
衿袍 0
袀 58
袁 931
袁世凱 33
袁枚 5
袁紹 34
袁術 8
袂 36
袃 7
袄 0
袈 13
袈裟 40
袉 73
袉 73
袊 0
袋 4062
袋子 146
袋熊 7
袋狀 5
袋鼠 126
袌 44
袍 194
袍子 14
袍服 2
袍澤 20
袍澤之誼 0
袍澤之誼 0
袎 30
袏 0
袐 0
袑 46
袒 17
袒胸 4
袒胸露乳 0
袒胸露乳 0
袒胸露背 1
袒胸露背 1
袒裼裸裎 0
袒護 13
袒護著 0
袒露 1
袒露 1
袒露出 0
袒露出 0
袓 76
袔 0
袕 75
袖 357
袖口 23
袖套 3
袖子 53
袖孔 0
袖手 17
袖手旁觀 30
袖扣 3
袖珍 125
袖珍型 9
袖珍本 4
袖珍版 3
袖章 1
袖筒 1
袖管 7
袖箭 0
袖裡 0
袖裡乾坤 2
袖釦 5
袖長 6
袗 61
袘 57
袙 78
袚 150
袚 150
袛 90
袜 0
袜 0
袝 0
袞 13
袞袞諸公 3
袟 36
袠 0
袡 63
袢 151
袣 0
袤 101
袥 0
袦 0
袧 201
袨 105
袩 0
袪 973
袪暑 1
袪暑氣 0
袪除 23
袪風 13
袪風油 0
被 373
被 37337
被上 17
被乘數 4
被人 897
被俘 31
被保險人 903
被刺 20
被剝削 17
被剝削階級 1
被加數 6
被動 478
被動式 55
被動性 9
被動挨打 0
被動語態 5
被告 798
被告人 14
被告席 2
被單 29
被堅執銳 0
被壓迫 39
被壓迫民族 1
被壓迫階級 0
被套 39
被子 44
被子植物 17
被害 178
被害人 465
被巾 0
被復 0
被愛 267
被打 200
被捕 296
被服 35
被服廠 6
被殺 120
被減數 4
被甲 1
被眨 1
被窩 31
被窩裡 13
被絮 0
被統治者 6
被統治階級 3
被綁 59
被罰 64
被肩 1
被臥 0
被虐待狂 2
被裡 5
被複 1
被褥 27
被覆 140
被覆 140
被覆著 2
被覆著 2
被試者 3
被貶 16
被迫 543
被迫辭職 3
被逼 69
被選 166
被選人 0
被選舉權 30
被開方數 0
被除數 6
被雷擊 2
被面 5
被頭散髮 0
被髮 1
被髮左衽 0
袬 48
袱 34
袲 30
袳 0
袳 0
袳 0
袴 0
袵 0
袶 45
袷 37
袸 38
袹 21
袺 208
袺襘 1
袻 0
袼 74
袽 172
袾 34
袿 0
裀 148
裁 786
裁併 28
裁兵 1
裁刀 12
裁判 805
裁判員 87
裁判官 6
裁判權 3
裁判長 29
裁剪 132
裁員 245
裁奪 4
裁定 408
裁尺 0
裁度 0
裁成 18
裁抑 0
裁撤 127
裁斷 30
裁樣板 0
裁汰 1
裁決 233
裁法 0
裁減 121
裁為 1
裁示 137
裁紙 47
裁縫 34
裁縫師 9
裁縫店 0
裁縫業 1
裁縫機 2
裁處 35
裁衣 82
裁製 9
裁軍 38
裁開 2
裂 601
裂了 5
裂到 0
裂化 2
裂口 11
裂孔 19
裂帛 1
裂成 10
裂片 5
裂痕 69
裂紋 35
裂縫 250
裂罅 2
裂解 68
裂解氣 0
裂變 9
裂開 67
裂開來 2
裂隙 51
裉 23
裊 73
裊裊 31
裋 3
裌 3
裍 8
裎 10
裏 4034
裏面 382
裐 13
裑 0
裒 136
裒多益寡 0
裒輯 0
裓 0
裔 327
裔冑 0
裕 764
裕隆 496
裕隆牌 2
裖 6
裗 53
裘 388
裙 400
裙子 148
裙帶 8
裙帶關係 2
裙衩 4
裙釵 1
裚 5
裛 4
補 5402
補丁 4
補上 48
補上來 2
補上去 5
補下 1
補下來 0
補下去 0
補休 17
補修 101
補假 64
補償 965
補償費 67
補償金 111
補充 2336
補充兵 4
補充品 40
補充物 3
補助 6302
補助款 246
補助線 0
補助費 247
補助金 92
補助額 15
補品 57
補報 57
補寫 6
補帖 0
補強 0
補拙 11
補擬書名 0
補救 390
補救法 0
補教 0
補數 45
補校 625
補正 271
補氣 15
補注 24
補洞 2
補派 3
補漏 11
補牌 1
補牙 10
補發 397
補白 6
補白圖案 0
補眠 0
補短 4
補票 14
補票價 0
補種 7
補空 4
補空位 0
補給 306
補給品 44
補給站 10922
補給線 7
補給點 0
補綴 32
補編 5
補繳 118
補缺 20
補缺位 0
補習 614
補習學校 300
補習教育 72
補習班 1712
補考 128
補考分數 1
補考時間 2
補胎 7
補色 6
補藥 42
補血 48
補血劑 2
補血藥 1
補血針 0
補衣 1
補角 0
補註 10
補語 7
補課 41
補貨 49
補貼 289
補貼費 9
補貼額 0
補贖 2
補足 230
補述 22
補過 13
補選 347
補遺 40
補還 4
補釘 5
補附 4
補靪 0
補靴匠 0
補鞋 1
補鞋匠 0
補頁 0
補養 39
補養品 0
補體 21
裝 7604
裝上 192
裝上來 0
裝上去 16
裝下 8
裝不下 4
裝作 123
裝修 679
裝修門面 0
裝假 3
裝做 37
裝備 1244
裝備齊全 8
裝傻 30
裝入 149
裝出 60
裝出來 0
裝出去 0
裝到 35
裝卸 364
裝卸橋 0
裝卸貨物 0
裝在 180
裝填 59
裝填子彈 0
裝好 93
裝子彈 0
裝孤 123
裝岩機 0
裝巖機 0
裝幌子 0
裝彈 1
裝得 17
裝得 286
裝懂 6
裝成 44
裝扮 146
裝扮成 10
裝料 3
裝旦 123
裝書 1
裝有 227
裝束 23
裝框 20
裝模作樣 19
裝檢 1244
裝死 13
裝殮 0
裝滿 92
裝演 0
裝潢 1506
裝為 14
裝璜 353
裝甲 164
裝甲兵 9
裝甲師 5
裝甲車 39
裝甲部隊 2
裝病 5
裝痛 0
裝瘋 6
裝瘋賣傻 2
裝的 285
裝睡 3
裝神弄鬼 9
裝笑 2
裝笑臉 0
裝箱 86
裝糊塗 2
裝罐 4
裝置 3636
裝置成 1
裝置業 11
裝聾作啞 6
裝腔作勢 6
裝船 1003
裝蒜 46
裝袋 43
裝裱 28
裝訂 2030
裝訂廠 5
裝訂成冊 34
裝訂所 6
裝設 552
裝貨 37
裝起 5
裝起來 10
裝車 18
裝載 142
裝載量 2
裝輸 1
裝進 28
裝進來 2
裝進去 9
裝運 78
裝運量 0
裝配 432
裝配工廠 4
裝配廠 7
裝配成 4
裝配線 17
裝配起來 1
裝門面 0
裝闊 0
裝飾 851
裝飾品 163
裝飾師 0
裝飾性 24
裝飾畫 0
裝飾音 9
裝鬼臉 0
裝點 14
裞 1
裟 49
裠 0
裠 0
裡 1
裡 29860
裡人 9
裡加 17
裡外 24
裡外不是人 12
裡外受敵 0
裡子 22
裡屋 0
裡層 4
裡布 33
裡帶 13
裡弦 0
裡應外合 12
裡拉 2
裡斯 1
裡海 10
裡脊 1
裡衣 0
裡裡 2
裡裡外外 10
裡通外國 0
裡通外敵 0
裡邊 0
裡邊兒 0
裡邊兒 0
裡間 0
裡面 3196
裡面兒 0
裡面兒 0
裡頭 433
裡頭兒 0
裡頭兒 0
裦 0
裦 0
裧 5
裨 144
裨 144
裨 144
裨使 4
裨助 3
裨將 1
裨益 47
裨能 1
裩 0
裩 0
裪 0
裫 0
裬 3
裭 0
裭 0
裮 11
裯 8
裰 8
裱 134
裱工 0
裱糊 0
裱褙 57
裱貼 2
裲 2
裳 173
裳 173
裴 365
裴豔玲 1
裵 0
裵 0
裶 2
裷 5
裸 465
裸奔 23
裸子植物 75
裸線 8
裸胸 3
裸花 1
裸袒 0
裸裎 11
裸身 22
裸露 129
裸露出 1
裸體 114
裸體像 0
裸體畫 6
裸麥 7
裹 230
裹上 9
裹上來 0
裹上去 0
裹了 3
裹住 15
裹傷 4
裹入 1
裹到 0
裹創 0
裹在 12
裹好 0
裹尸 1
裹布 2
裹得 2
裹成 1
裹肚 0
裹脅 3
裹腳 4
裹腳布 3
裹腿 0
裹著 35
裹起 2
裹起來 1
裹足 2
裹足不前 43
裹錯 0
裺 54
裻 9
裼 13
裼 13
製 11622
製件 0
製作 18058
製作人 478
製作業 26
製作者 148
製做 74
製備 547
製冰 76
製冰機 42
製冷 6
製冷機 0
製劑 487
製品 3452
製圖 600
製圖員 7
製圖學 7
製圖室 10
製圖師 0
製圖樣 0
製圖者 0
製成 734
製成品 42
製法 109
製為 6
製片 197
製片人 14
製片商 8
製片廠 32
製版 428
製版術 3
製程 0
製糖 58
製糖廠 1
製紙 52
製藥 969
製藥廠 244
製表 225
製裁 0
製造 11987
製造出 194
製造出來 13
製造品 31
製造商 1071
製造器 14
製造場 6
製造場 6
製造廠 628
製造廠商 451
製造成 21
製造業 6253
製造業者 202
製造者 191
製造術 0
製造費用 12
製革 144
製革工廠 0
製革廠 21
製鞋 282
製鹽 17
裾 10
裿 0
裿 0
褀 0
褀 0
褁 55
褂 50
褂子 3
褃 0
褃 0
褅 61
褆 57
複 1186
複仞年如 0
複以百萬 0
複位 0
複信 0
複分數 0
複分解反應 0
複列 2
複利 53
複利率 2
複印 397
複印機 50
複原 1
複句 2
複合 450
複合式 100
複合句 0
複合材料 573
複合物 36
複合詞 3
複合體 34
複名 1
複名數 0
複名詞 0
複員 0
複員軍人 0
複壁 0
複壯 0
複姓 7
複字鍵 0
複審 227
複寫 47
複寫紙 4
複工 0
複式 96
複復 0
複數 125
複數形 2
複本 113
複查 148
複核 57
複檢 158
複次 0
複比 0
複比例 0
複決 81
複決權 25
複測 5
複畝珍 0
複發 1
複目 1
複眼 22
複種 2
複種指數 0
複線 7
複習 342
複舊 0
複色 2
複色光 0
複葉 13
複蓋 1
複製 2291
複製出 13
複製品 30
複診 273
複詞 1
複試 135
複課 0
複議 3
複變函數 22
複賽 91
複述 13
複選 231
複選題 9
複錢 0
複雜 2186
複雜勞動 0
複雜度 93
複雜性 129
複雜生產 0
複電 0
複音 9
複韻 2
褈 0
褉 55
褉 55
褊 170
褊急 0
褊淺 0
褊狹 11
褋 11
褌 53
褌 53
褍 0
褎 31
褎 31
褏 0
褐 547
褐布 0
褐煤 5
褐煤田 0
褐色 161
褐藻 12
褐設 0
褐鼠 1
褑 71
褒 220
褒叟 0
褒境 0
褒忠 389
褒揚 112
褒損 0
褒氣 0
褒獎 27
褒節 0
褒絢 0
褒色朦朧 0
褒貶 13
褒賞 0
褒靄 0
褒顯 0
褓 171
褓且 0
褓姆 37
褓母 16
褔 439
褕 102
褖 73
褗 50
褘 249
褙 46
褚 535
褛 0
褞 19
褟 6
褟 6
褠 0
褡 165
褡包 0
褡連 0
褢 9
褣 9
褤 0
褥 129
褥套 4
褥子 6
褥瘡 23
褦 3
褧 6
褨 0
褨 0
褩 3
褪 686
褪下 15
褪光 3
褪去 37
褪成 2
褪掉 0
褪脫 0
褪色 129
褫 990
褫奪 8
褫奪公權 34
褬 3
褭 3
褮 2
褯 5
褰 13
褱 5
褲 5294
褲下之辱 0
褲子 194
褲帶 11
褲管 29
褲腰 2
褲腰帶 0
褲腳 6
褲腿 1
褲袋 9
褲裙 4
褲襠 17
褲襪 48
褳 6
褵 8
褶 207
褶 207
褶曲 8
褶疊 1
褶痕 4
褶皺 40
褶紙 0
褶裙 4
褷 4
褸 69
褹 0
褹 0
褺 0
褻 105
褻瀆 52
褻瀆神明 0
褻玩 7
褻臣 0
褻衣 2
褼 1
褽 44
褾 192
褿 0
襀 0
襁 7
襁褓 21
襂 8
襃 0
襃 0
襄 365
襄助 71
襄樊 25
襄王 22
襄理 77
襄贊 4
襄辦 0
襄陽 120
襆 40
襈 0
襈 0
襉 5
襊 0
襊 0
襋 71
襌 77
襍 0
襎 0
襎 0
襏 54
襐 43
襑 5
襒 7
襒 7
襓 103
襖 113
襖兒 4
襖兒 4
襖褲 0
襗 4
襗 4
襘 10
襙 21
襚 7
襛 4
襜 107
襝 54
襞 260
襟 139
襟兄 0
襟度 0
襟弟 0
襟懷 36
襟翼 26
襠 160
襡 24
襢 5
襢 5
襣 99
襤 180
襤褸 16
襥 0
襦 51
襧 0
襩 1
襩 1
襪 285
襪套 0
襪子 157
襪帶 2
襪筒 0
襫 4
襬 23
襭 3
襮 2
襯 406
襯兒 0
襯兒 0
襯出 10
襯墊 44
襯布 3
襯托 85
襯托出 32
襯托底 0
襯托物 0
襯砌 9
襯衣 26
襯衫 180
襯裙 0
襯裡 21
襯褲 0
襰 0
襱 4
襲 806
襲取 1
襲奪 14
襲擊 229
襲擊戰術 0
襲爵 17
襲用 6
襳 5
襴 7
襵 0
襶 2
襸 0
襹 5
襺 4
襻 2
襼 4
襽 0
襾 2
西 11
西 11723
西上 2
西下 31
西亞 427
西亞 427
西京 6
西人 18
西伯 15
西伯利亞 118
西伯利亞平原 0
西來 52
西偏 1
西側 211
西元 1616
西元前 115
西出 15
西到 3
西化 49
西化運動 0
西北 636
西北向 7
西北地區 8
西北地方 1
西北方 43
西北歐 7
西北角 18
西北邊 7
西北部 73
西北面 3
西北風 16
西區 601
西半 21
西半球 11
西半邊 0
西南 495
西南向 8
西南地區 17
西南方 55
西南角 13
西南邊 8
西南部 99
西南非 0
西南非洲 0
西南面 2
西南風 16
西印度群島 20
西印度聯邦 0
西去 13
西向 56
西周 38
西周時 0
西周時代 0
西周時期 0
西哲 6
西園 133
西園路 94
西垂 1
西城 52
西域 48
西域人 1
西境 0
西夏 74
西夏國 1
西天 32
西太后 5
西夷 0
西奈半島 2
西套 2
西子 258
西子 258
西子湖 2
西學 17
西學東漸 3
西安 395
西安人 1
西安市 12
西宮 5
西寧 102
西寧南路 76
西寧市 5
西山 73
西岸 203
西島 18
西崑體 0
西嶼 190
西嶽 2
西嶽華山 1
西席 4
西康 26
西康人 0
西康省 2
西廂 16
西廂記 22
西廠 5
西式 321
西往 7
西征 12
西從 9
西德 333
西德人 1
西戎 3
西指 0
西捻 0
西接 28
西撒哈拉 4
西敏寺 12
西文 1331
西文系 4
西方 1980
西方人 88
西方國家 137
西方極樂 82
西方淨土 15
西施 274
西昌 26
西晉 13
西晉時 0
西晉時代 0
西晉時期 1
西晒 3
西曆 56
西曬 1
西書 40
西有 16
西服 258
西服店 10
西望 14
西柏林 8
西楚 9
西樂 30
西樓 19
西歐 334
西歐人 1
西歐各國 5
西歐國家 19
西段 62
西江 19
西江流域 3
西沙 15
西沙群島 8
西法 184
西洋 1348
西洋人 13
西洋味 0
西洋式 5
西洋文學 86
西洋棋 56
西洋畫 43
西洋芹 1
西洋鏡 0
西流 10
西浦運河 0
西海 24
西海岸 85
西涼 12
西湖 480
西湖國中 9
西湖國小 8
西湖路 9
西溝 0
西漢 64
西漢時 5
西漢時代 1
西漢時期 2
西潮 16
西為 15
西照 3
西片 91
西片商 0
西王 14
西王母娘娘 0
西班牙 1267
西班牙人 33
西班牙文 112
西班牙話 1
西班牙語 141
西瓜 308
西瓜子 2
西瓜汁 5
西瓜渣 0
西瓜皮 13
西畫 108
西疆 0
西皮 18
西秦 12
西突厥 3
西窗 8
西站 21
西端 16
西籍 1
西米露 0
西紅柿 1
西經 8
西線 35
西羅馬帝國 8
西羌 5
西至 33
西莒 7
西莒島 4
西菜 4
西蒙 35
西藏 824
西藏人 19
西藏地方 2
西藏文 1
西藏自治區 19
西藏路 153
西藥 375
西藥房 25
西螺 638
西螺大橋 21
西行 39
西街 235
西裝 280
西裝店 9
西裝褲 6
西西 329
西西里 20
西西里島 22
西角 1
西諺 5
西谷米 0
西谷米凍 0
西貝流士 17
西貢 50
西路 1282
西遊 27
西遊記 187
西遼 9
西遼國 0
西遼河 0
西邊 144
西郊 9
西部 622
西部地區 97
西部片 3
西都 6
西醫 397
西里 40
西里西亞 0
西門 880
西門國小 43
西門圓環 3
西門市場 10
西門市場 10
西門町 134
西門町 134
西門豹 2
西陵峽 7
西陽 4
西雅圖 319
西雙版納 4
西非 89
西非各國 1
西非國家 1
西非地區 1
西面 54
西風 243
西風東漸 5
西餐 280
西餐店 0
西餐廳 356
西館 2
西高止山 0
西魏 5
西麓 7
西點 156
西點軍校 14
西點麵包 113
西點麵包店 43
要 633
要 76337
要用 300
要不 1008
要不得 14
要不得 14
要不是 145
要不然 303
要了 96
要事 59
要人 89
要付 71
要件 611
要價 19
要公 11
要到 777
要則 15
要功 13
要務 65
要區 5
要口 2
要命 121
要和 633
要員 77
要問 388
要喝 70
要因 117
要圖 15
要地 24
要塞 102
要好 243
要害 94
要寫 272
要就 57
要帳 0
要帶 283
要強 53
要得 114
要得 4523
要挾 6
要擊 4
要政 8
要旨 112
要是 945
要有 1987
要民 3
要求 11458
要沖 9
要津 5
要港 4
要犯 32
要略 20
要的 4522
要盟 0
要目 268
要穿 67
要站 31
要約 70
要素 584
要緊 137
要義 73
要聞 9330
要職 25
要脅 42
要臉 41
要街 2
要衝 42
要角 71
要角 71
要言 2
要言不煩 0
要言不繁 2
要言之 6
要訣 309
要質 0
要賬 0
要路 6
要道 89
要錢 91
要錢不要命 0
要附 13
要隘 0
要項 212
要領 425
要飯 2
要飯的 3
要麼 2
要麼 4
要點 4633
覂 784
覃 156
覃 156
覅 8
覆 1180
覆上 10
覆亡 14
覆住 2
覆信 3
覆命 2
覆在 7
覆審 50
覆巢之下無完卵 2
覆成 5
覆敗 1
覆文 3
覆校 1
覆核 43
覆水難收 10
覆沒 7
覆滅 9
覆盆 3
覆盆子 4
覆舟 11
覆著 9
覆蓋 509
覆蓋住 6
覆蓋著 24
覆試 0
覆議 93
覆車 0
覆車之戒 0
覆車之鑑 0
覆載 5
覆載 5
覆轍 25
覆電 3
覇 0
覈 421
覈實 19
覉 0
覊 0
見 1
見 25457
見不到 98
見不得 25
見不得人 36
見不著 9
見世 7
見世面 1
見了 390
見了面 27
見于 14
見人 124
見仁見智 61
見個 20
見光 27
見光死 14
見分曉 10
見利忘義 4
見到 1823
見取見 1
見告 5
見地 34
見報 72
見報率 1
見外 23
見多 20
見多識廣 6
見天 28
見天日 12
見好 6
見好就收 16
見客 4
見對 26
見底 50
見得 181
見得多了 2
見得著 2
見微知著 6
見怪 35
見怪不怪 21
見惑 1
見愛 9
見慣 4
見所未見 4
見招 2
見招拆招 31
見效 61
見教 4
見方 23
見於 141
見景生情 1
見棄 0
見樹不見林 9
見機 10
見機行事 3
見死不救 16
見獵心喜 3
見異思遷 7
見票即付 4
見稜見角 0
見稱 13
見笑 39
見縫插針 1
見罪 1
見義勇為 25
見習 235
見習官 0
見習生 15
見聞 164
見聞廣博 3
見聞甚廣 0
見背 3
見著 62
見覆 1
見見 39
見見面 9
見解 514
見証 104
見証人 2
見諒 541
見證 1091
見證人 47
見識 286
見財起意 1
見賢思齊 24
見過 598
見過世面 5
見錢 10
見錢眼開 2
見長 64
見面 809
見面禮 15
見風使舵 0
見風轉舵 28
見高低 0
見鬼 15
覌 0
覍 0
覎 0
規 5240
規例 3
規則 4020
規則化 3
規則性 33
規劃 17661
規勸 29
規定 18787
規定出來 0
規定性 2
規律 404
規律性 32
規格 7607
規格化 68
規條 29
規模 3094
規模經濟 0
規法 15
規畫 1424
規矩 158
規程 685
規章 1042
規章制度 13
規範 3917
規範化 0
規約 146
規繩 0
規規矩矩 31
規諫 1
規費 387
規避 235
覐 0
覐 0
覐 0
覑 0
覒 0
覓 263
覓保 6
覓句 0
覓得 53
覓得 53
覓食 194
覔 0
覕 15
視 17127
視事 32
視作 40
視力 4345
視力表 6
視力計 0
視同 358
視唱 19
視圖 88
視在功率 2
視域 14
視如己出 3
視如敝徙 0
視察 426
視差 23
視機而定 0
視死如歸 9
視為 2141
視為一體 2
視為一體 2
視界 470
視神經 81
視窗 7721
視網膜 418
視線 275
視而不見 55
視聽 7631
視聽中心 408
視聽享受 7
視聽教育 283
視若無人 0
視若無睹 29
視覺 2543
視覺享受 18
視覺作用 2
視覺性 8
視覺神經 10
視覺系 2543
視角 162
視訊 2754
視野 1347
視障 0
視頻 45
覗 47
覘 15
覘候 0
覙 0
覚 0
覚 0
覚 0
覛 5
覜 341
覝 5
覞 4
覟 3
覠 0
覡 6
覢 69
覣 59
覤 68
覥 0
覥 0
覦 172
覧 0
覨 0
覩 0
親 677
親 6773
親上加親 0
親事 14
親交 3
親人 322
親代 17
親任 2
親信 71
親兄弟 11
親兄弟明算帳 1
親兵 9
親切 843
親友 667
親友們 20
親口 63
親口答應 3
親口說 13
親吻 106
親和 84
親和力 146
親和性 53
親啟 1
親善 88
親善大使 35
親嘴 9
親夫 1
親如手足 1
親姊 0
親姊妹 1
親姊妹 1
親姊姊 0
親姊姊 0
親姐 1
親姐妹 1
親姐姐 1
親娘 5
親子 1906
親子活動 96
親子關係 132
親家 33
親家公 1
親家母 0
親家翁 2
親密 433
親密性 1
親密關係 138
親展 0
親屬 402
親屬關係 41
親征 2
親情 198
親愛 125
親愛的 1729
親愛精誠 3
親戚 260
親戚關係 10
親手 196
親手做 6
親授 15
親政 3
親故 2
親族 18
親日 12
親昵 1
親暱 12
親暱性 0
親朋 39
親朋好友 328
親本 6
親歷其境 1
親母 7
親民 433
親民黨 0
親洽 117
親潮 4
親熱 52
親父 9
親率 19
親王 37
親生 73
親生子女 3
親生父母 10
親疏 10
親痛仇快 1
親眷 7
親眼 136
親眼所見 15
親眼看到 47
親眼看見 31
親睦 4
親筆 46
親筆信 9
親筆簽名 33
親緣 22
親耳 9
親耳所聞 1
親聞 18
親臉 0
親臨 221
親臨現場 18
親臨現場 18
親自 2194
親自出馬 14
親自動手 21
親著 2
親見 27
親親 272
親親愛愛 1
親親熱熱 2
親貴 0
親賢人遠小人 0
親身 384
親身經歷 66
親身經驗 17
親迎 8
親近 408
親隨 1
親骨肉 2
覫 0
覬 46
覬覦 73
覭 1
覮 3
覯 10
覰 0
覰 0
覱 0
覲 35
覲 35
覲禮 0
覲見 4
覲謁 0
観 0
観 0
覴 0
覵 0
覶 8
覷 104
覹 28
覺 315
覺 3156
覺世名言 0
覺出 9
覺察 67
覺察出 2
覺察到 23
覺岸 0
覺得 11596
覺得 11596
覺得很 526
覺得很 526
覺得很有 16
覺得很有 16
覺得很沒 1
覺得很沒 1
覺悟 188
覺書 1
覺查 1
覺民 52
覺著 29
覺醒 302
覻 0
覻 0
覼 0
覽 12163
覽勝 14
覾 3
覿 6
覿面 0
觀 122
觀 12231
觀世音 40
觀世音大士 0
觀世音菩薩 151
觀光 5454
觀光事業 500
觀光勝地 42
觀光區 65
觀光團 7
觀光季 12
觀光客 271
觀光局 954
觀光旅行 4
觀光旅館 135
觀光業 54
觀光科 25
觀台 10
觀天象 7
觀察 4686
觀察使 3
觀察入微 14
觀察出來 5
觀察到 166
觀察力 64
觀察員 104
觀察家 104
觀察所 7
觀察站 20
觀察者 69
觀念 4544
觀念形態 1
觀念性 14
觀感 138
觀戰 69
觀摩 1074
觀摩式 0
觀摩教學 16
觀摹 10
觀政 1
觀日 43
觀星 171
觀星台 7
觀望 416
觀望不前 1
觀望台 0
觀棋不語真君子 16
觀止 9
觀海 45
觀測 1391
觀測員 2
觀測哨 0
觀測器 3
觀測室 5
觀測所 14
觀測站 164
觀濤 3
觀火 4
觀照 159
觀看 2755
觀眾 1818
觀眾台 0
觀眾席 40
觀眾臺 0
觀瞻 48
觀禮 55
觀禮台 3
觀者如堵 0
觀自在菩薩 1
觀臺 1
觀色 12
觀護人 65
觀象 5
觀象儀 0
觀象台 2
觀象臺 0
觀賞 1791
觀賞用 22
觀門 6
觀音 1253
觀音大士 15
觀音山 161
觀音柳 0
觀音菩薩 42
觀點 4215
角 19
角 1983
角 3191
角力 158
角力賽 1
角加速度 5
角動量 16
角化 35
角子 1
角尺 9
角度 2391
角形 16
角果 0
角柱 5
角柱體 1
角樓 5
角狀 12
角球 44
角皮層 0
角膜 313
角膜炎 18
角色 4403
角色 2250
角落 743
角落裡 50
角谷猜想 0
角質 134
角質化 13
角質層 43
角逐 296
角速度 17
角鋼 123
角錐 20
角錐體 0
角門 9
角頻率 2
觓 371
觔 1
觔斗 11
觔斗雲 7
觕 43
觖 32
觗 0
觘 0
觙 38
觚 35
觛 21
觜 4
觝 39
觝觸 0
觟 8
觠 6
觡 6
觢 3
解 731
解 731
解 7314
解下 32
解了 52
解事 4
解人 94
解任 35
解作 11
解像 54
解像度 46
解僱 62
解元 3
解凍 181
解凍劑 0
解出 56
解出來 4
解到縣裡 0
解剖 627
解剖刀 4
解剖學 183
解危 29
解吸 1
解嘲 16
解嚴 191
解嚴令 0
解囊 4
解圍 17
解壓縮 1
解夢 120
解字 6
解官 3
解官 3
解差 2
解怨 1
解恨 0
解悟 4
解悶 20
解悶法 0
解惑 217
解憂 12
解手 3
解放 441
解放出來 19
解放前 0
解放區 3
解放思想 0
解放戰爭 2
解放日報 11
解放者 1
解放軍 192
解放軍報 6
解放陣線 14
解救 141
解救出來 1
解救法 0
解散 499
解散國會 20
解析 1202
解析器 0
解析幾何 18
解析度 2964
解款 7
解毒 334
解毒劑 27
解毒片 8
解毒藥 1
解民倒懸 4
解池 2
解決 9364
解決不了 21
解決問題 713
解決困難 45
解決法 13
解決辦法 43
解法 102
解渴 44
解熱劑 8
解甲 4
解甲棄械 0
解甲歸田 2
解甲還鄉 0
解痛 2
解痛劑 0
解痛藥 0
解碼 289
解碼器 127
解禁 68
解禁令 0
解答 7221
解約 152
解紛 2
解纜 3
解罪 0
解聘 106
解職 61
解脫 338
解脫法 4
解脫知見 1
解脫義 1
解脫道 1
解藥 44
解衣 12
解衣推食 0
解裝 0
解語 5
解說 1790
解說員 143
解說法 0
解說詞 0
解調 34
解譯 39
解讀 825
解送 23
解送犯人 0
解酒 20
解釋 4232
解釋做 1
解釋性 6
解釋法 8
解鈴仍須繫鈴人 1
解鈴還須繫鈴人 1
解開 547
解除 1124
解除戒嚴 13
解除武裝 6
解除警報 7
解雇 30
解難 6
解頤 256
解題 226
解題技巧 12
解饞 10
解體 121
觤 31
觥 19
觥籌 0
觥籌交錯 1
觥觥 0
触 69
觧 0
觧 0
觧 0
觨 6
觩 5
觪 0
觫 3
觬 107
觭 23
觮 0
觮 0
觰 100
觱 5
觱沸 0
觲 2
觳 3
觳觫 2
觴 195
觴酌 0
觵 0
觶 28
觷 23
觸 1156
觸到 33
觸動 159
觸及 216
觸嘴 0
觸塵 1
觸壓 4
觸壓式 0
觸媒 361
觸媒轉換器 23
觸怒 19
觸感 127
觸手 19
觸控 0
觸控式 0
觸摸 209
觸摸式 511
觸擊 12
觸景 5
觸景生情 5
觸板 1
觸殺 8
觸毛 0
觸法 69
觸犯 221
觸犯法規 2
觸發 133
觸發器 3
觸發式 0
觸目 26
觸目皆是 1
觸目驚心 19
觸碰 30
觸礁 41
觸網 8
觸覺 99
觸覺器 0
觸覺式 0
觸覺神經 0
觸角 188
觸身球 0
觸雷 3
觸電 66
觸電時 1
觸電而死 0
觸類旁通 9
觸鬚 17
觸點 1
觹 0
觹 0
觹 0
觺 2
觻 1
觻 1
觼 152
觽 0
觽 0
觽 0
觾 3
觿 18
言 8415
言下 10
言下之意 12
言不及義 20
言不由衷 5
言不盡意 2
言且表過 0
言之 168
言之成理 12
言之有據 0
言之有物 31
言之有理 6
言之有誤 0
言之無物 6
言之過早 67
言之鑿鑿 15
言之鑿鑿 15
言人人殊 1
言傳 14
言傳身教 1
言出必行 2
言及 228
言和 35
言外 8
言外之意 12
言多必失 7
言多語失 0
言官 1
言對 8
言必信 0
言情 9
言情小說 9
言教 29
言教不如身教 3
言文一致 1
言文一致 1
言明 53
言明在先 1
言歸 5
言歸於好 3
言歸正傳 18
言淺意深 0
言為心聲 1
言猶在耳 8
言盡於此 2
言笑 9
言笑之間 0
言笑間 0
言簡意賅 12
言者 58
言者無罪 1
言者諄諄 1
言而有信 4
言而無信 8
言聽計從 5
言行 194
言行 194
言行一致 5
言行一致 5
言行不一 29
言行不符 1
言行合一 0
言行如一 1
言行舉止 26
言行錄 10
言詞 162
言詞辯論 46
言語 541
言語不通 3
言語無味 1
言談 108
言談之間 5
言談舉止 10
言談間 13
言論 1772
言論自由 242
言辭 55
言過其實 17
言重 18
訂 4339
訂上 8
訂下 55
訂了 121
訂交 5
訂位 706
訂作 73
訂來訂去 0
訂個 12
訂做 165
訂價 296
訂出 140
訂出來 1
訂到 16
訂單 1667
訂在 56
訂報 48
訂好 18
訂婚 226
訂婚戒指 4
訂定 3275
訂座 0
訂成 6
訂戶 903
訂戶資料 1
訂房間 4
訂於 426
訂書 228
訂書機 31
訂期 44
訂條約 0
訂正 114
訂死 6
訂為 99
訂盟 35
訂票 313
訂票處 1
訂立 262
訂約 101
訂給 1
訂製 1113
訂親 3
訂貨 356
訂貨單 57
訂購 10429
訂購單 854
訂起 2
訂起來 0
訂造 23
訂過 5
訂金 137
訂閱 2308
訂雜誌 4
訃 10
訃告 4
訃文 10
訃聞 5
訄 12
訅 0
訆 0
訇 629
計 11447
計上心來 0
計件 7
計件工資 3
計價 371
計分 320
計劃 9208
計劃圖 15
計劃性 46
計劃書 324
計劃案 47
計劃生育 7
計劃經濟 18
計劃者 5
計委 17
計數 298
計數器 962
計時 533
計時員 6
計時器 152
計時工資 1
計有 940
計概 158
計次 41
計次收費 2
計溫 0
計略 2
計畫 30341
計票 0
計程 165
計程司機 10
計程汽車 20
計程車 882
計窮 4
計策 26
計算 6528
計算中心 64009
計算出 135
計算出來 12
計算力 1
計算器 51
計算尺 1
計算所 13
計算機 2167
計算機中心 315
計算起來 0
計謀 45
計議 13
計費 418
計費時間 1
計費法 0
計較 206
計酬 55
計量 633
計量化學 0
計量史學 0
計量地理 200
計量地理學 200
計量經濟學 200
計量法 7
計量方法 10
訉 0
訊 3751
訊問 79
訊息 20236
訊疾 0
訊號 1494
訊號旗 0
訊號槍 1
訋 0
訌 17
訍 0
訍 0
訍 0
討 2657
討乞 1
討亂 0
討了 7
討人 17
討人厭 17
討人喜歡 41
討人嫌 1
討人歡心 1
討伐 45
討來討去 0
討便宜 3
討個 14
討個吉利 0
討債 41
討債鬼 0
討價 3
討價還價 49
討光 1
討出 0
討出來 0
討到 4
討厭 963
討厭鬼 13
討去 0
討取 4
討喜歡 0
討好 162
討媳婦 0
討巧 0
討帳 1
討得 5
討情 0
討教 36
討沒趣 0
討生活 20
討空 0
討索 0
討老婆 1
討著 1
討親 0
討討 1
討論 82519
討論到 45
討論會 704
討賊 5
討賞 0
討賬 0
討走 0
討錢 2
討開 1
討飯 5
討飯的 2
討饒 4
訏 7
訐 15
訐直 0
訑 2
訒 9
訓 4018
訓令 14
訓典 1
訓勉 11
訓導 477
訓導主任 179
訓導處 1178
訓導長 14
訓戒 4
訓政 8
訓政時期 7
訓斥 11
訓示 21
訓練 17061
訓練出來 20
訓練所 298
訓練有素 44
訓練班 612
訓練費 14
訓育 274
訓育組 372
訓訓 131
訓訓話 0
訓詁 29
訓詁學 28
訓詞 10
訓話 22
訓語 0
訓誡 16
訓誨 14
訓辭 0
訓過 0
訓飭 0
訔 0
訕 11
訕笑 17
訕訕地 2
訖 82
訖了 0
訖站 6
訖點 1
託 1606
託了 0
託事 2
託交 0
託人 24
託付 93
託付給 20
託古諷今 0
託名 0
託咎 0
託夢 8
託大 0
託孤 7
託孤寄命 0
託故 1
託疾 1
託病 0
託福 2
託管 30
託管國 0
託著 0
託言 1
託詞 2
託買 5
託賣 2
託身 0
託辭 1
託運 40
託過 0
記 11449
記上 33
記上來 0
記上去 1
記下 157
記下來 29
記下去 0
記不住 12
記不住 12
記不得 44
記不清 18
記不起來 16
記了 14
記事 808
記事文 0
記事本 260
記事板 2
記事簿 90
記事體 1
記仇 5
記住 462
記作 5
記入 20
記出 5
記出來 0
記分 20
記分員 1
記分板 1
記分法 6
記分牌 0
記功 30
記功嘉獎 2
記取 80
記名 62
記名字 1
記名投票 3
記名法 0
記在 57
記大功 19
記大過 25
記好 9
記工 6
記帳 260
記帳法 0
記得 4675
記得 4675
記念 153
記性 21
記性差 1
記恨 6
記恨在心 3
記憶 2377
記憶力 172
記憶區 5
記憶猶新 36
記憶體 3221
記成 5
記掛 6
記掛著 4
記敘 13
記敘文 3
記敘體 0
記數 14
記數法 1
記法 13
記熟 2
記牢 6
記的 4674
記筆記 4
記者 7919
記者招待會 27
記者會 1320
記著 55
記號 508
記要 151
記言 1
記言文 0
記誦 15
記誦法 0
記譜 20
記賬 0
記起 96
記起來 14
記載 1414
記載下來 6
記述 119
記述文 0
記述體 0
記過 74
記過處分 15
記錄 7681
記錄下來 90
記錄員 6
記錄器 74
記錄本 12
記錄板 3
記錄簿 43
記錯 101
訙 0
訛 42
訛人 0
訛傳 5
訛舛 0
訛言 4
訛詐 1
訛誤 11
訛謬 0
訛音 0
訜 0
訝 118
訝異 227
訞 35
訟 210
訟事 5
訟案 19
訟棍 0
訟獄 0
訟費 4
訠 0
訡 0
訢 54
訣 245
訣別 13
訣別書 7
訣竅 137
訣要 0
訤 0
訥 29
訥 29
訦 0
訧 449
訨 0
訩 0
訪 3842
訪人 148
訪友 49
訪古 35
訪台 278
訪台之旅 0
訪問 3119
訪問團 412
訪問記 14
訪問隊 0
訪客 9605
訪察 4
訪察團 3
訪德 6
訪德之旅 0
訪日 473
訪日之旅 0
訪日團 8
訪查 203
訪求 4
訪法 17
訪法之旅 0
訪美 153
訪美之旅 0
訪英 4
訪英之旅 0
訪華 252
訪華之旅 0
訪華團 17
訪視 513
訪親 30
訪談 541
訪貧問苦 0
訪賢 0
訪遍 0
訪遍 0
訪韓 3
訪韓之旅 0
訪韓團 0
訫 0
訫 0
訬 59
設 11505
設下 67
設了 75
設伏 0
設使 6
設來 0
設來設去 0
設備 52875
設備利用率 20
設備租金 3
設備組 75
設備組長 32
設備齊全 62
設到 12
設去 3
設台 29
設在 310
設壇 3
設壇祭拜 0
設奠 0
設好 55
設定 11625
設定值 145
設定器 1
設定檔 0
設宴 53
設宴招待 1
設宴款待 8
設宴歡迎 0
設局 19
設帳號 1
設廠 546
設得 198
設想 151
設成 210
設或 29
設於 450
設施 5725
設有 1890
設法 564
設的 198
設立 5571
設站 80
設籍 0
設給 11
設置 6340
設若 40
設計 35448
設計圖 240
設計家 47
設計師 1509
設計者 663
設計階段 70
設設 10
設起 1
設起來 1
設身處地 35
設過 6
設防 43
設防地 0
設防地帶 0
設限 243
設限於 2
設陷 5
設黨 8
訮 0
訮 0
訮 0
訯 0
訰 43
許 1
許 12804
許下 94
許下諾言 3
許乃濟 0
許久 255
許久以前 4
許久以後 1
許可 3796
許可証 38
許可證 916
許國 54
許多 15320
許多事 75
許多事 60
許多人 1323
許婚 2
許廣平 1
許慎 5
許文龍 1
許昌 54
許昌縣 0
許是 44
許氏 44
許氏鍵盤 44
許聞廉 1
許許 14
許許 14
許許多多 395
許諾 33
許身 2
許配 8
許配給 9
許願 224
許願池 29
訲 0
訴 1370
訴冤 0
訴情 17
訴求 871
訴狀 63
訴苦 36
訴訟 743
訴訟條例 0
訴訟法 89
訴訟狀 7
訴訟費用 39
訴說 211
訴說出來 0
訴說著 73
訴請 74
訴諸 150
訴諸武力 32
訴願 611
訴願狀 0
訶 112
訶 112
訶子 1
訶斥 0
訶詰 0
訶護 0
訶責 1
訶難 0
訶難 0
訷 0
訸 0
訹 7
診 480
診 4805
診室 15
診察 111
診所 3372
診斷 3543
診斷器 1
診斷書 93
診治 318
診病 56
診療 518
診療室 76
診療所 17
診脈 4
診脈 4
診視 20
註 8130
註上 8
註冊 6715
註冊主任 1
註冊商標 292
註冊手續 32
註冊碼 0
註冊組 482
註冊費 130
註失 0
註定 174
註明 2384
註標 12
註生娘娘 7
註疏 11
註腳 61
註解 398
註譯 8
註釋 188
註銷 436
証 890
証人 28
証件 124
証券 544
証實 110
証據 58
証明 427
証明書 51
証書 108
証物 6
証詞 4
訽 0
訾 13
訾 13
訾病 0
訾議 7
訿 6
詀 25
詁 41
詂 0
詃 0
詄 4
詅 13
詆 8
詆毀 17
詆譭 4
詇 0
詇 0
詈 13
詉 0
詉 0
詉 0
詊 4
詋 0
詌 5
詍 21
詎 216
詎料 10
詎非所料 0
詏 5
詐 16
詐 161
詐偽 1
詐取 8
詐敗 7
詐欺 152
詐欺罪 12
詐死 20
詐病 2
詐稱 4
詐胡 1
詐術 21
詐財 47
詐賭 10
詐降 12
詐騙 118
詑 8
詒 48
詓 0
詔 143
詔令 5
詔告 2
詔命 2
詔安 21
詔旨 1
詔書 14
詔示 2
詔諭 3
評 3752
評上 1
評上來 0
評上去 0
評了 4
評介 283
評估 7647
評個 1
評個分數 0
評傳 22
評價 2560
評出 3
評出來 0
評分 587
評分標準 130
評判 60
評判出 0
評判員 1
評劇 0
評定 576
評審 1540
評審委員 245
評工 4
評彈 14
評得 49
評成 4
評戲 1
評改 1
評斷 96
評斷是非 1
評析 437
評核 98
評比 420
評注 4
評獎 14
評獎人 0
評理 1
評的 48
評等 293
評註 22
評評 22
評評理 48
評話 3
評語 168
評語欄 1
評論 3581
評論員 26
評論家 83
評議 341
評議委員 48
評議會 83
評起 0
評選 697
評選人 0
評選員 1
評選為 76
評量 1633
評鑑 3101
評鑑報告 73
評閱 22
評頭論足 2
詖 5
詗 11
詘 26
詘 26
詙 4
詚 0
詛 61
詛咒 152
詛咒語 0
詛罵 0
詜 0
詝 0
詞 3823
詞不達意 9
詞人 17
詞令 4
詞兒 7
詞兒 7
詞典 209
詞匯 10
詞句 111
詞尾 4
詞序 5
詞庫 91
詞庫輸入法 3
詞彙 413
詞性 12
詞意 26
詞曲 117
詞條 0
詞源 8
詞牌 5
詞窮 3
詞章 3
詞素 3
詞組 21
詞義 17
詞藻 8
詞表 5
詞話 14
詞語 1640
詞調 1
詞譜 5
詞賦 4
詞選 33
詞鋒 7
詞音 2
詞韻 2
詞頭 2
詞頻 0
詞類 11
詞類變化 2
詠 1575
詠嘆 64
詠懷 0
詠歌 6
詠歎 41
詠詩 4
詡 68
詢 3609
詢問 2298
詢問處 26
詣 98
詤 0
詥 0
詥 0
試 102404
試一試 65
試件 8
試住 1
試作 104
試來試去 1
試做 14
試出 13
試出來 5
試刊 78
試制 8
試劑 287
試加 5
試卷 277
試吃 37
試唱 4
試問 335
試圖 672
試場 23
試場 234
試婚 34
試定 10
試射 123
試得 1
試想 173
試成 1
試探 396
試探性 20
試教 121
試映 13
試樣 147
試機 11
試演 6
試煉 78
試爆 0
試片 144
試片室 17
試用 1512
試用品 17
試用期 44
試用本 14
試用者 14
試看 157
試種 13
試穿 35
試算 1640
試算表 484
試管 92
試管嬰兒 76
試管架 2
試紙 42
試練 14
試聽 777
試航 33
試著 975
試行 148
試裝 3
試訂 0
試試 875
試試 875
試試看 538
試論 94
試賣 19
試賣品 0
試起 3
試起來 0
試跑 4
試跳 16
試車 235
試車場 2
試車場 2
試辦 946
試辦期 2
試辦階段 11
試過 319
試選 1
試金石 25
試銷 8
試銷品 0
試鏡 18
試鏡室 0
試開 2
試閱 0
試音 4
試音室 0
試題 3608
試飛 37
試飛員 21
試驗 5711
試驗品 6
試驗場 69
試驗場 69
試驗室 86
試驗性 22
試驗物 0
試驗田 3
試點 22
詧 0
詨 84
詩 6044
詩班 20
詩中有畫 3
詩人 1084
詩人節 6
詩仙 8
詩作 177
詩卷 10
詩友 9
詩句 103
詩史 17
詩名 10
詩壇 88
詩學 169
詩律 3
詩情畫意 52
詩意 88
詩才 11
詩文 302
詩文集 23
詩書 30
詩書門第 0
詩歌 629
詩歌朗誦 14
詩癖 0
詩社 96
詩章 6
詩箋 4
詩篇 240
詩經 124
詩聖 3
詩興 5
詩草 2
詩藝 10
詩行 6
詩詞 790
詩詞歌賦 18
詩話 25
詩調 0
詩謎 1
詩讖 0
詩豪 4
詩賦 9
詩選 236
詩集 287
詩韻 22
詩頁 1
詩風 19
詩餘 2
詩體 55
詪 0
詫 207
詫異 47
詬 25
詬病 93
詬罵 1
詭 194
詭奇 4
詭局 0
詭怪 0
詭異 183
詭祕 2
詭祕 2
詭秘 9
詭秘 9
詭稱 0
詭計 45
詭計多端 4
詭詐 19
詭譎 74
詭譎多變 17
詭辯 10
詮 1116
詮釋 942
詯 0
詰 196
詰問 20
詰屈聱牙 3
詰責 1
詰難 4
話 23783
話不投機 1
話不投機半句多 3
話中帶刺 4
話中有話 11
話兒 49
話兒 49
話冊 0
話別 26
話到嘴邊 1
話劇 274
話劇表演 7
話務員 19
話匣子 222
話多 30
話意 10
話本 20
話柄 4
話梅 4
話機 155
話筒 85
話舊 2
話裡有話 0
話語 330
話說 1061
話鋒 5
話鋒一轉 22
話音 9
話頭 35
話題 7617
話題一轉 4
該 29149
該上 22
該下 14
該不是 19
該事 8
該以 60
該位 20
該來 56
該來則來 0
該做 210
該免則免 0
該再 41
該判 1
該判何罪 0
該刪 0
該到 31
該加 14
該區 188
該印 4
該去 141
該受 38
該名 96
該向 78
該員 44
該唱 3
該回 43
該國 505
該在 86
該地 205
該城 8
該場 96
該場 96
該填 17
該季 5
該守 0
該島 14
該巷 1
該市 31
該帳 0
該幫 12
該年 107
該廠 123
該建 18
該往 34
該得 25
該戰 2
該打 42
該扣 3
該投 36
該換 68
該換則換 0
該收 10
該於 1
該日 28
該是 387
該會 545
該有 329
該校 592
該死 146
該段 29
該減 8
該漲 1
該漲 1
該漲則漲 0
該漲則漲 0
該漲未漲 0
該漲未漲 0
該為 68
該片 62
該班 86
該用 166
該由 104
該當 37
該當何罪 6
該禁 2
該稱 4
該站 136
該答 1
該算 31
該管 320
該管則管 0
該箱 1
該約 12
該線 20
該編 0
該縣 46
該自 9
該處 283
該號 11
該行 285
該補 5
該裝 7
該要 96
該說 172
該調 4
該調 4
該起 7
該起來 2
該輪 10
該辦 81
該進 9
該還則還 0
該鄉 29
該隊 31
該集 11
該面 13
該項 890
該黨 119
詳 4778
詳備 5
詳加 164
詳加說明 25
詳圖 63
詳密 3
詳察 7
詳實 251
詳審 3
詳悉 0
詳情 972
詳明 9
詳查 20
詳略 1
詳盡 695
詳細 7079
詳言之 4
詳註 4
詳詳細細 5
詳讀 62
詳述 264
詳述如下 16
詳閱 258
詴 2
詴 2
詵 7
詶 4
詷 5
詸 0
詸 0
詸 0
詹 2647
詹大悲 0
詹天佑 3
詹姆士 57
詹森 24
詺 3
詻 36
詼 23
詼諧 104
詽 0
詽 0
詽 0
詾 0
詿 4
誀 0
誀 0
誁 9
誂 7
誂 7
誃 6
誄 11
誅 201
誅心之論 1
誅戮殆盡 0
誅死 0
誅殺 18
誅滅 2
誅除 1
誆 8
誆騙 1
誇 142
誇下海口 2
誇了 0
誇來誇去 0
誇功 2
誇口 30
誇嘴 0
誇大 191
誇大之詞 1
誇大其詞 11
誇大狂 0
誇張 341
誇得 1
誇成 0
誇獎 43
誇示 13
誇耀 38
誇著 0
誇誇 0
誇誕 0
誇讚 37
誇起 0
誇起來 0
誇過 0
誇飾 3
誋 1
誌 2199
誌哀 1
誌喜 4
誌慶 17
認 14198
認不出 19
認不出來 12
認不得 4
認不清 0
認了 55
認作 4
認來認去 0
認供 0
認個 3
認做 2
認出 134
認出來 12
認列 0
認到 9
認可 1126
認同 2417
認命 52
認字 24
認定 1853
認屍 8
認帳 4
認得 167
認成 3
認捐 15
認明 39
認有 86
認清 255
認清是非 0
認為 13281
認為是 615
認生 0
認用 0
認真 1523
認真做事 5
認真讀書 5
認知 2118
認破 0
認祖歸宗 1
認票 0
認票不認人 0
認給 0
認罪 95
認罰 0
認股 93
認親 8
認証 864
認認 6
認認真真 6
認證 2934
認識 9317
認識不清 8
認識到 127
認識論 31
認賊作父 2
認賠 24
認賬 0
認購 395
認購股 2
認起 0
認起來 0
認輸 77
認過 4
認過來 0
認過去 1
認錯 177
認領 79
認養 0
誎 0
誏 16
誐 0
誑 629
誑妄 0
誑言 0
誑誕 0
誑語 5
誑騙 0
誒 9
誒 9
誒詒 1
誓 280
誓不兩立 5
誓不甘休 1
誓不罷休 3
誓師 45
誓師大會 38
誓死 42
誓死不屈 1
誓死不降 0
誓約 18
誓言 226
誓詞 18
誓辭 0
誔 0
誕 584
誕生 1202
誕生地 17
誕生石 17
誕育 0
誕辰 140
誕辰紀念日 66
誖 3
誗 0
誘 1372
誘人 156
誘使 43
誘入 3
誘出 10
誘因 290
誘姦 8
誘導 196
誘導式 4
誘導法 0
誘引 16
誘惑 311
誘惑力 9
誘拐 25
誘拐罪 0
誘捕 16
誘掖 1
誘敵 8
誘敵戰術 0
誘敵法 0
誘殺 18
誘殺戰術 0
誘發 251
誘發式 0
誘脅 0
誘降 1
誘降戰術 0
誘電性 5
誘餌 31
誘騙 17
誙 6
誚 442
誛 0
誜 0
語 75
語 7541
語云 5
語助詞 4
語匯 4
語句 99
語塞 8
語尾助詞 1
語尾詞 0
語尾音 0
語序 1
語彙 120
語形 4
語意 170
語感 0
語態 4
語文 3384
語文學家 2
語文能力 87
語料 0
語族 54
語根 1
語構學 0
語氣 276
語氣詞 0
語法 1363
語法分析 7
語源 6
語焉不詳 10
語無倫次 17
語病 10
語系 196
語義 18
語言 8276
語言學 424
語言學家 16
語詞 75
語調 98
語重心長 41
語錄 159
語音 3683
語音信箱 0
語音學 26
語風 5
語驚四座 3
語體 2
語體化 0
語體文 7
誟 0
誠 4494
誠信 379
誠品 827
誠如 189
誠如所言 0
誠實 575
誠實可欺 0
誠實可靠 5
誠屬 19
誠屬不易 5
誠徵 952
誠心 324
誠心誠意 31
誠惶誠恐 7
誠意 486
誠懇 383
誠摯 400
誠敬 11
誠服 4
誠樸 8
誠然 80
誠然如此 1
誠然是 9
誠非 1
誠非所料 0
誠非易事 0
誡 201
誡律 1
誢 0
誣 62
誣 62
誣告 60
誣告罪 2
誣害 0
誣控 5
誣蔑 9
誣蔑性 0
誣衊 8
誣詆 1
誣謗 0
誣賴 10
誣陷 20
誣陷 20
誤 1011
誤上賊船 4
誤下 3
誤事 20
誤人 25
誤人子弟 9
誤人誤己 3
誤以為 177
誤信 23
誤傳 12
誤傷 14
誤入 52
誤入歧途 20
誤判 85
誤取 3
誤國 4
誤報 14
誤填 1
誤寫 5
誤導 236
誤導人心 1
誤導作用 0
誤導民心 0
誤工 5
誤差 463
誤差值 16
誤打誤撞 23
誤放 2
誤時 13
誤會 546
誤期 0
誤殺 14
誤測 0
誤犯 4
誤班 0
誤用 66
誤盡天下蒼生 0
誤稱 7
誤約 0
誤置 2
誤解 427
誤觸 37
誤訛 0
誤認 127
誤認為 76
誤謬 10
誤闖 20
誤闖入 1
誤食 673
誤餐費 7
誤點 21
誥 184
誥命 2
誥封 0
誦 271
誦經 57
誦經臺 0
誦詠 1
誦詩 2
誦讀 20
誦讀詩篇 0
誧 239
誨 504
誨 504
誨人不倦 10
誨人不倦 10
誨淫 1
誨盜 1
誨盜誨淫 0
誩 0
說 141
說 1613
說 43458
說的是 100
說一不二 1
說一是一 1
說一說 22
說一說 22
說上 25
說上來 0
說上去 0
說下 14
說下來 0
說下去 21
說不上 17
說不上來 17
說不下去 8
說不了 4
說不出 233
說不出來 20
說不出口 26
說不定 592
說不得 9
說不清 10
說不清楚 15
說不過 8
說不過來 0
說不過去 8
說了 764
說了又說 1
說了就算了 1
說來 273
說來聽聽 45
說來話長 17
說來說去 15
說出 1007
說出來 272
說出去 11
說出話來 1
說到 618
說到做到 15
說到曹操 0
說動 16
說古 6
說合 2
說和 42
說唱 57
說夢話 13
說大話 5
說好 110
說好了 30
說好就好 0
說好聽 2
說好說歹 0
說媒 5
說完 406
說定 1
說定了 5
說客 17
說實在的 89
說實話 67
說得 2308
說得上 4
說得來 0
說得到 2
說得動 0
說得有理 8
說得過去 3
說情 18
說成 75
說戲 7
說教 53
說教術 0
說文 13
說文解字 29
說明 26713
說明力 0
說明文 2
說明書 1155
說明會 989
說是 1254
說時慢那時快 0
說時遲那時快 18
說書 43
說書人 10
說服 378
說服力 107
說東說西 0
說東道西 0
說法 1620
說法不一 14
說清 7
說清楚 75
說溜了嘴 6
說理 28
說理文 0
說白 45
說的 2307
說相聲 34
說真格的 7
說真的 247
說真話 13
說瞎話 6
說短論長 0
說破 13
說破嘴 0
說穿 4
說穿了 86
說笑 42
說給 116
說經 10
說親 4
說詞 114
說話 3019
說話不算數 0
說話不算話 9
說話算數 0
說話算話 5
說說 931
說說唱唱 12
說說看 47
說說話 119
說謊 381
說謊者 12
說謊話 6
說走就走 13
說起 275
說起來 110
說起話來 14
說辭 41
說通 1
說過 885
說過來 0
說過去 7
說道 877
說部 6
說錯 74
說長道短 6
說開 4
說閒話 2
說項 6
誫 2
説 0
説 0
説 0
誯 0
誰 87
誰 8776
誰人 113
誰個 2
誰在 1037
誰家 100
誰是 675
誰會 247
誰有 731
誰肯 15
誰能 997
誰言寸草心 1
誱 0
課 20607
課以重稅 1
課內 19
課內練習 0
課前 115
課前課後 1
課前預習 8
課名 15
課吏 0
課員 59
課堂 517
課堂秩序 2
課堂練習 3
課外 199
課外作業 8
課外指導 7
課外活動 271
課外活動組 141
課外讀物 84
課外輔導 2
課室 63
課征 6
課後 215
課後練習 5
課後複習 5
課徵 591
課收 11
課文 160
課時 162
課本 473
課本上 30
課桌 16
課桌椅 87
課椅 1
課業 1136
課田制 0
課目 107
課稅 576
課程 25426
課程標準 509
課程表 628
課表 701
課表上 4
課長 395
課間 43
課間休息 3
課間操 0
課題 1183
課餘 107
課餘時間 54
誳 0
誳 0
誳 0
誳 0
誴 0
誵 0
誶 35
誶罵 0
誷 0
誸 59
誹 166
誹謗 230
誹謗罪 20
誺 30
誻 3
誼 106
誼 1062
誽 79
誾 10
調 4234
調 23030
調上 9
調上來 0
調上去 0
調下 5
調下來 0
調下去 0
調了 17
調人 127
調令 0
調任 122
調位 0
調位子 0
調低 197
調使 3
調侃 48
調來 0
調來調去 1
調個 0
調值 4
調停 26
調停人 1
調停法 0
調價 46
調光 83
調入 24
調兵 14
調兵遣將 4
調冶 2
調出 60
調出來 5
調出去 0
調到 103
調制 28
調劑 267
調劑品 1
調劑身心 17
調動 162
調勻 20
調包 15
調升 116
調去 4
調合 60
調合漆 1
調名 9
調味 215
調味品 39
調味料 232
調和 327
調和級數 0
調唆 2
調嗓子 0
調回 42
調回來 5
調回去 4
調在 25
調壓 11
調壓器 6
調好 28
調妥 2
調子 49
調寬 1
調差 1
調幅 135
調幅台 22
調度 407
調弄 0
調式 87
調強 0
調往 18
調得 23
調得 204
調快 6
調性 56
調息 27
調情 36
調戲 21
調換 38
調換法 0
調撥 27
調攝 2
調教 46
調整 6952
調整器 69
調料 4
調昇 4
調查 11456
調查員 107
調查團 15
調查局 605
調查研究 959
調查表 357
調查部 4
調校 65
調正 10
調治 6
調治法 0
調派 126
調派到 1
調準 3
調溫 9
調漲 0
調為 75
調班 3
調理 417
調琴 4
調用 40
調畫出 0
調白 12
調的 203
調皮 73
調皮搗蛋 8
調皮蛋 5
調皮鬼 3
調相 8
調移 32
調笑 9
調節 885
調節器 106
調細 1
調置 2
調羹 1
調職 113
調色 39
調色板 7
調色盤 97
調著 5
調薪 65
調虎離山 8
調虎離山之計 1
調虎離山計 5
調處 261
調號 4
調製 249
調製法 2
調解 407
調解法 2
調詞架訟 0
調試 4
調調 21
調調兒 6
調調兒 6
調諧 21
調諧器 142
調貨 17
調質 11
調走 2
調起 0
調起來 0
調車 14
調車場 4
調車場 4
調輕 3
調轉 9
調近 0
調速 22
調速器 16
調進 6
調進來 2
調進去 0
調運 1
調過 11
調過來 2
調過去 0
調遠 2
調遣 5
調適 418
調遷 10
調配 353
調配出 8
調配法 3
調酒 215
調酒師 14
調重 20
調錢 3
調門 3
調門兒 0
調門兒 0
調開 10
調閱 170
調防 0
調降 0
調集 17
調離 13
調音 41
調音師 9
調頭 9
調頭寸 6
調頻 109
調頻台 22
調頻臺 0
調頻電台 13
調頻電臺 1
調養 79
調高 577
諀 1
諁 0
諂 104
諂媚 35
諂笑 0
諂諛 2
諃 12
諄 176
諄諄 19
諄諄告誡 7
諄諄地 0
諄諄地 0
諄諄教導 1
諄諄教誨 15
諅 30
諆 40
談 10398
談上 20
談不 19
談不上 87
談不來 1
談不出 5
談不到 1
談不成 4
談不攏 24
談不隴 0
談了 84
談何容易 32
談來談去 1
談入 1
談出 23
談判 1422
談判下去 0
談判桌上 9
談判者 5
談到 733
談及 152
談吐 40
談吐之間 1
談吐風生 0
談和 12
談天 729
談天說地 295
談好 25
談妥 29
談完 17
談得 37
談得 256
談得上 7
談得來 13
談得出 1
談得成 0
談心 149
談情 27
談情說愛 116
談成 15
談攏 7
談明 9
談柄 0
談法 8
談玄 2
談的 255
談笑 88
談笑之間 2
談笑自若 5
談笑風生 14
談著 19
談虎色變 1
談話 671
談話中 41
談話室 0
談說 4
談談 664
談論 463
談論出 1
談論到 12
談論著 22
談起 313
談起來 1
談過 92
談過來 0
談過去 1
談鋒 0
談錢 5
談開 15
諈 14
諉 314
諉卸 0
諉過 6
諉過於人 0
諊 0
請 80875
請上 98
請上來 5
請上去 1
請下 155
請下來 1
請下去 1
請不到 0
請不動 0
請不起 4
請了 107
請代 38
請來 312
請便 11
請個 18
請假 665
請假單 23
請假條 0
請先 2140
請入 34
請再 446
請出 47
請出來 6
請出去 2
請到 705
請功 3
請勿 3007
請勿動手 0
請勿吸煙 2
請勿踐踏 0
請去 46
請向 314
請君入甕 5
請命 90
請和 177
請問 17572
請回 229
請回來 1
請回到 18
請回去 0
請在 1588
請坐 26
請坐下 2
請坐下來 1
請好 25
請安 59
請客 121
請將 1624
請帖 29
請帶 53
請戴 10
請打 116
請按 7132
請挑 5
請撥 196
請播 2
請救兵 2
請教 4070
請旨 4
請有 111
請柬 25
請款 75
請款單 4
請求 1885
請洽 2776
請用 799
請用茶 0
請留 263
請留下 428
請留步 11
請益 100
請看 2130
請示 89
請等 22
請等一下 8
請等一下 8
請答 24
請纓 2
請罪 15
請老 12
請聽 67
請舉手 24
請見 321
請託 59
請說 41
請說出 6
請調 29
請談 3
請謁 0
請講 21
請貼 37
請走 12
請起 7
請起來 0
請跳 12
請辭 237
請進 525
請進來 81
請進去 3
請過 17
請過來 14
請過去 0
請降 3
請願 135
請願團 18
請願書 17
諌 0
諍 30
諍友 1
諍臣 0
諍言 27
諍訟 0
諍諫 1
諎 0
諎 0
諏 60
諏訪 3
諏議 0
諐 0
諐 0
諑 4
諒 695
諒察 5
諒已 1
諒必 3
諒解 161
諓 61
諔 3
諕 3
論 10958
論 621
論事 9
論交 9
論件 1
論件計酬 7
論價 2
論列 4
論到 53
論功 4
論功行賞 3
論及 343
論壇 9195
論壇報 60
論天 6
論天計酬 1
論定 11
論工計酬 0
論戰 210
論據 17
論據不足 0
論政 24
論文 8736
論文體 0
論斤 6
論斤兩 0
論斤論兩 1
論斷 120
論旨 7
論月 0
論月計酬 0
論本 4
論本計費 0
論法 8
論爭 25
論理 28
論理學 8
論稱 2
論罪 12
論者 157
論者以為 5
論著 432
論處 23
論衡 19
論証 10
論評 33
論語 152
論說 32
論說文 4
論調 102
論談 165
論證 111
論贊 1
論起 12
論述 788
論過 8
論題 60
論點 285
諗 25
諘 80
諙 17
諛 18
諛詞 0
諛辭 0
諜 2400
諜報 24
諜報人員 2
諜報員 10
諜對諜 1
諝 436
諞 184
諟 70
諠 19
諠譁 0
諡 133
諡法 1
諡號 3
諡議 1
諢 934
諢名 0
諢號 1
諣 0
諣 0
諤 56
諤諤 3
諥 0
諦 470
諦聽 13
諦視 0
諧 197
諧劇 0
諧和 19
諧振 24
諧振動 3
諧振器 2
諧波 48
諧波 48
諧聲 0
諧語 1
諧調 9
諧謔 2
諧趣 5
諧音 32
諨 49
諩 0
諪 0
諫 272
諫大夫 0
諫官 3
諫書 2
諫諍 2
諫議 5
諫院 0
諬 0
諭 256
諭令 0
諭告 1
諭旨 5
諭知 20
諭示 3
諭請 0
諮 497
諮商 1372
諮文 1
諮詢 8674
諮詢費 3
諮諏 0
諮議 384
諮議局 4
諯 91
諰 94
諱 116
諱名 0
諱疾忌醫 12
諱莫如深 5
諱言 68
諲 378
諳 466
諳曉 0
諳熟 3
諳練 0
諳習 0
諳識 0
諴 197
諵 337
諶 59
諷 152
諷 152
諷刺 250
諷刺性 6
諷刺畫 1
諷刺詩 3
諷古說今 0
諷喻 5
諷語 0
諷誦 1
諸 6500
諸事 67
諸位 632
諸侯 83
諸公 41
諸君 60
諸城 2
諸夏 0
諸多 710
諸如 653
諸如此類 82
諸子 57
諸子百家 14
諸宮調 254
諸生 6
諸般 44
諸葛 109
諸葛 109
諸葛亭 0
諸葛亭 0
諸葛亮 254
諸葛武侯 3
諸葛武侯 3
諸行無常 1
諸言 1
諹 0
諺 266
諺語 132
諻 111
諻 111
諼 434
諽 0
諾 3798
諾亞方舟 5
諾思 0
諾曼 51
諾曼地 10
諾曼地登陸 4
諾曼第 21
諾言 90
諾諾 10
諾貝爾 399
諾貝爾獎 117
諿 73
謀 848
謀主 1
謀亂 0
謀事 1
謀事在人 1
謀出路 5
謀利 12
謀刺 2
謀反 13
謀取 59
謀叛 1
謀國 7
謀士 16
謀定 10
謀定而後 0
謀定而後動 3
謀害 18
謀得 11
謀慮 0
謀殺 599
謀殺案 58
謀殺罪 5
謀殺者 3
謀求 233
謀生 157
謀生不易 3
謀生之道 2
謀略 107
謀福利 30
謀私 4
謀私利 3
謀職 405
謀臣 3
謀議 1
謀變 1
謀財 7
謀財害命 12
謀面 20
謁 177
謁見 13
謂 1419
謂何 1
謂者 1
謄 69
謄出 3
謄出來 1
謄寫 8
謄寫板 0
謄寫簿 0
謄書 0
謄本 215
謄清 1
謄稿 30
謄給 0
謄錄 1
謅 750
謆 69
謇 61
謈 27
謉 0
謉 0
謉 0
謉 0
謊 131
謊報 19
謊報年齡 0
謊言 189
謊話 36
謋 69
謌 0
謍 51
謎 912
謎兒 0
謎兒 0
謎團 17
謎底 49
謎樣 33
謎語 342
謎面 1
謎題 94
謏 31
謏 31
謐 110
謑 121
謒 24
謓 32
謔 188
謔稱 2
謔而不虐 1
謕 33
謕 33
謖 1131
謗 712
謘 389
謙 1019
謙卑 150
謙受益 7
謙和 26
謙恭 11
謙恭有禮 3
謙抑 2
謙沖 16
謙稱 13
謙虛 171
謙虛有禮 3
謙虛謹慎 0
謙詞 0
謙謙君子 3
謙讓 11
謙辭 0
謙退 2
謙遜 68
謙遜有禮 1
謚 106
謚 106
講 8331
講的是 100
講下去 4
講了 149
講交情 0
講作 0
講來 9
講來講去 5
講信修睦 5
講個 12
講價 1
講出 60
講出來 33
講出去 1
講到 267
講古 51
講古人 1
講台 95
講史 8
講和 17
講堂 345
講壇 32
講好 23
講學 184
講學法 0
講定 5
講對 3
講師 3008
講席 10
講座 2470
講得 555
講情 2
講情理 0
講情面 0
講成 6
講授 538
講授法 0
講故事 56
講明 24
講書 9
講武 2
講武堂 0
講求 299
講法 45
講演 160
講演法 2
講理 23
講的 554
講稿 83
講究 305
講給 30
講經 53
講義 1187
講義費 11
講習 703
講習所 15
講習會 224
講習班 64
講臺 11
講著 20
講解 521
講解法 0
講評 67
講評人 9
講評法 0
講詞 56
講話 644
講課 64
講課法 0
講起 21
講起來 25
講述 249
講述法 2
講過 70
講過來 0
講過去 1
講道 250
講錯 13
講面子 1
講題 251
講點 5
謜 15
謝 1
謝 11989
謝世 45
謝了 81
謝函 22
謝國城 2
謝天 65
謝天謝地 27
謝好 0
謝孝 0
謝孟叡 1
謝客 0
謝家 32
謝帖 0
謝師宴 53
謝幕 30
謝忱 43
謝恩 23
謝意 127
謝感 1
謝東閔 19
謝法 8
謝玄 4
謝神 76
謝禮 22
謝絕 64
謝絕參觀 2
謝罪 29
謝者 0
謝謝 12009
謝謝 12009
謝謝你 781
謝謝你 781
謝謝你們 185
謝謝你們 185
謝銘儒 2
謞 5
謟 0
謠 306
謠傳 40
謠言 300
謠言惑眾 1
謠言止於 0
謠言止於智者 7
謠言滿天 1
謠言滿天飛 9
謢 240
謣 3
謤 8
謥 95
謦 13
謧 6
謨 193
謩 0
謪 8
謫 145
謫仙 4
謫居 1
謫戍 0
謫降 0
謬 136
謬以千里 0
謬誤 78
謬說 0
謬論 27
謭 0
謮 27
謯 22
謰 6
謱 17
謲 0
謲 0
謲 0
謲 0
謳 6
謳吟 1
謳歌 17
謴 0
謵 29
謶 54
謷 11
謸 0
謹 932
謹上 12
謹備菲酌 0
謹呈 2
謹啟 6
謹嚴 9
謹守 75
謹守諾言 0
謹小慎微 0
謹慎 469
謹此 110
謹稟 0
謹致 23
謹致謝意 2
謹言慎行 20
謹記 34
謹記在心 16
謹身 1
謹防 19
謹防假冒 0
謹防扒手 0
謹領 0
謹飭 1
謺 25
謻 10
謼 22
謼 22
謽 47
謾 19
謾 19
謾天謾地 0
謾罵 65
謾罵 65
謿 0
謿 0
譀 9
譀 9
譀 9
譁 222
譁然 13
譁眾取寵 24
譁變 2
譂 10
譃 0
譄 0
譅 3
譆 183
譇 127
譈 7
證 9016
證交所 427
證交稅 37
證人 265
證人席 1
證件 1338
證出 7
證出來 1
證券 12361
證券交易 173
證券交易所 891
證券交易稅 154
證券公司 254
證券商 562
證券市場 115
證券市場 1159
證嚴 198
證婚 39
證婚人 10
證實 1229
證悟 25
證據 1151
證於 11
證明 4652
證明出來 1
證明單 30
證明文件 1249
證明書 1144
證書 1830
證期會 0
證法 16
證照 0
證物 91
證管會 149
證見 10
證言 32
證詞 74
證道 62
證驗 7
譊 10
譋 1
譌 0
譍 0
譍 0
譎 123
譎詭 3
譏 172
譏刺 6
譏嘲 8
譏為 16
譏笑 32
譏笑聲 0
譏詆 0
譏評 5
譏誚 10
譏諷 32
譐 6
譐 6
譑 9
譑 9
譒 4
譓 12
譔 63
譕 6
譖 21
譗 8
譗 8
識 353
識 3534
識別 822
識別力 2
識別區 2
識別字 3
識別碼 2189
識別証 12
識別證 104
識多見廣 0
識大體 7
識字 105
識字率 6
識字班 11
識得 38
識時務 7
識時務者 2
識時務者為俊傑 6
識相 16
識破 48
識蘊 1
識見 17
識記 0
識貨 27
識趣 10
識途老馬 4
譙 158
譚 833
譚人鳳 0
譚嗣同 6
譜 1232
譜上 20
譜上來 0
譜上去 0
譜下 16
譜下來 0
譜下去 0
譜出 28
譜出來 0
譜在 1
譜子 2
譜寫 53
譜寫成 0
譜成 11
譜新聲 0
譜曲 24
譜系 12
譜著 1
譜號 3
譜表 6
譝 28
譞 0
譟 207
譠 150
譡 0
譢 0
譣 69
譤 0
譤 0
譤 0
譥 93
譥 93
譥 93
警 3040
警世 15
警世名言 0
警備 51
警備區 1
警備司令 2
警備司令部 3
警備總司令 4
警備總部 8
警備車 17
警備隊 9
警力 139
警務 54
警務局長 0
警務處 56
警務部長 0
警匪 0
警句 6
警告 997
警告性 3
警員 263
警哨 1
警報 406
警報器 120
警報系統 55
警官 169
警官學校 10
警察 2842
警察學校 19
警察局 1785
警察局長 26
警察廣播電臺 7
警察廳 5
警察機關 288
警察隊 214
警局 324
警崗 0
警廣 59
警徽 103
警悟 4
警惕 120
警惕作用 4
警惕到 3
警惕性 1
警戒 346
警戒哨 0
警戒線 13
警戒色 5
警所 43
警探 60
警政 587
警政署 614
警政署長 39
警方 1820
警棍 30
警標 0
警犬 23
警界 53
警示 350
警笛 5
警笛聲 4
警策 3
警網 0
警總 45
警署 57
警號 6
警衛 350
警衛室 97
警衛軍 0
警衛隊 6
警覺 219
警覺到 55
警覺心 21
警覺性 58
警言 0
警訊 182
警語 43
警車 70
警車隊 0
警醒 17
警鈴 48
警鐘 11
警長 33
譧 63
譧 63
譨 21
譨譨 1
譪 73
譫 302
譬 303
譬喻 57
譬如 682
譬如說 147
譬若 4
譭 56
譮 0
譮 0
譮 0
譯 3334
譯傳 2
譯名 358
譯吏 0
譯員 12
譯官 2
譯意 0
譯意風 0
譯成 106
譯文 88
譯本 169
譯注 20
譯為 0
譯碼 5
譯碼器 3
譯稿 30
譯筆 6
譯經 40
譯者 1386
譯著 22
譯註 21
譯述 13
譯錯 2
譯電 6
譯音 37
議 5155
議事 1
議事廳 14
議事規則 70
議事錄 32
議價 282
議價空間 43
議和 22
議員 2389
議場 91
議場 91
議定 105
議定書 438
議席 38
議會 2255
議會制 5
議會鬥爭 0
議案 255
議決 340
議決書 8
議決案 8
議程 805
議處 94
議請 6
議論 224
議論文 8
議論紛紛 47
議長 1338
議長論短 0
議院 24
議題 2959
譱 0
譳 0
譴 99
譴責 521
譵 0
譶 0
護 5071
護人 27
護住 16
護佑 12
護來護去 0
護兵 2
護具 100
護唇膏 1
護喉 1
護喪 0
護國軍 2
護國運動 0
護坡 44
護城河 17
護墊 23
護壁板 0
護士 771
護士學校 2
護士室 0
護士長 9
護套 68
護好 1
護專 254
護專生 0
護岸石 0
護幕 0
護庇 4
護廠 48
護得 3
護愛 0
護手霜 1
護持 77
護旗 1
護板 17
護林 12
護林防火 0
護校 55
護棺 0
護欄 0
護民官 0
護法 180
護照 1081
護牆 9
護牆紙 0
護犢子 0
護理 3545
護理助產職業學校 3
護理專科學校 208
護理系 150
護理職業學校 15
護產 9
護產士 0
護甲 0
護盔 1
護盤 0
護目 21
護目鏡 157
護眼 38
護短 9
護符 6
護罩 46
護耳 4
護肘 13
護肩 2
護胸 5
護腕 0
護腿 2
護膚 407
護膚品 9
護膝 24
護航 84
護航艦 3
護航艦隊 1
護花 8
護花使者 5
護著 18
護蓋物 0
護衛 137
護衛者 5
護衛艦 13
護衛軍 1
護衛隊 2
護起 0
護路 0
護路林 0
護身 43
護身符 28
護車 5
護送 82
護送隊 0
護運 2
護面 4
護面具 0
護駕 10
護髮 107
譸 6
譹 6
譺 5
譺 5
譻 3
譼 0
譼 0
譽 798
譽滿 4
譽滿全國 0
譽滿全球 2
譽為 171
譾 4
譾陋 0
譿 28
讀 0
讀 6424
讀上 36
讀了 134
讀作 4
讀來 53
讀來讀去 0
讀信 58
讀入 103
讀出 113
讀出來 11
讀到 202
讀卡 23
讀卡機 74
讀友 63
讀取 1591
讀圖 16
讀在 3
讀報 16
讀字機 0
讀完 173
讀寫 317
讀寫能力 18
讀後 51
讀後心得 21
讀後感 87
讀得 205
讀成 14
讀數 44
讀數 44
讀書 2283
讀書不求甚解 1
讀書人 111
讀書會 1032
讀本 167
讀法 32
讀熟 3
讀熟 3
讀物 198
讀畢 4
讀的 204
讀秒 15
讀秒階段 1
讀稿 0
讀稿機 0
讀經 454
讀者 15465
讀者文摘 48
讀者群 53
讀若 1
讀萬卷書 20
讀萬卷書行萬里路 0
讀著 42
讀賣新聞 0
讀起 10
讀起來 35
讀進 35
讀過 180
讀過來 0
讀過去 0
讀錯 4
讀音 59
讁 0
讁 0
讂 4
讂 4
讃 0
讄 6
讅 4
讆 2
讇 0
讈 4
讉 0
變 6147
變上 0
變上來 0
變上去 0
變下 10
變下來 0
變下去 0
變亂 17
變了 266
變了又變 0
變些 0
變亮 11
變位 20
變低 10
變作 15
變來 4
變來變去 16
變修 2
變個 6
變做 7
變出 27
變出來 2
變到 9
變則通 2
變動 1568
變化 5191
變化多端 53
變化成 5
變化球 17
變卦 14
變去 2
變反 2
變味 1
變回 51
變回來 3
變回去 0
變因 39
變在 5
變壓 14
變壓器 451
變壞 86
變大 209
變天 106
變奏 100
變奏曲 66
變好 54
變小 141
變局 102
變幻 49
變幻無常 3
變幻莫測 23
變弄 0
變形 739
變形書名檢索 0
變形蟲 17
變形體 1
變影 0
變得 1597
變心 72
變性 182
變性手術 14
變態 366
變態心理 15
變成 4320
變戲法 8
變換 324
變換器 7
變故 67
變數 1516
變新 13
變易 17
變星 0
變晴 1
變更 4057
變更為 231
變本 7
變本加厲 29
變格 6
變樣 2
變法 28
變法維新 0
變法運動 0
變流器 7
變涼 3
變淡 13
變淺 9
變清 6
變滅 0
變為 325
變焦 0
變率 10
變產 5
變異 365
變異係數 18
變異性 26
變異數 0
變白 57
變的 1596
變相 111
變相加價 1
變短 17
變種 74
變節 6
變紅 29
變細 13
變給 2
變聲 9
變聲期 0
變臉 130
變色 319
變色筆 0
變色蜴 0
變色龍 74
變著 1
變薄 23
變薄 23
變調 65
變變 87
變變看 0
變賣 178
變賣家產 1
變質 208
變質石 0
變質處理 0
變質處理 0
變走 0
變起 2
變起來 1
變輕 7
變通 75
變通之計 0
變通方法 1
變通法 0
變通辦法 1
變速 225
變速器 72
變速球 1
變速箱 193
變速運動 0
變造 97
變造出 1
變過 10
變過來 0
變過去 0
變遷 2029
變酸 3
變醜 7
變重 10
變量 109
變錯 0
變長 31
變阻器 17
變電 58
變電所 61
變電站 29
變青 1
變革 569
變音 3
變音期 0
變頻 60
變頻器 113
變頻管 0
變體 29
變體字 1
變黏 1
讋 6
讌 22
讍 0
讎 45
讎隙 0
讐 0
讑 0
讒 88
讒害 0
讒構 0
讒言 8
讒誣 0
讒間 0
讒陷 0
讓 45928
讓人 2755
讓位 18
讓位於 1
讓價 1
讓出 108
讓出來 2
讓出去 1
讓受人 0
讓國 171
讓座 7
讓步 177
讓渡 34
讓給 78
讓與 206
讓讓 3
讓賢 5
讓起 0
讓起來 0
讓路 21
讓開 39
讓開來 0
讔 126
讕 90
讕言 1
讖 19
讖書 0
讖語 1
讗 0
讗 0
讘 2
讘 2
讙 9
讚 632
讚不絕口 36
讚佩 7
讚佩不已 0
讚同 17
讚唄 1
讚嘆 139
讚嘆不已 14
讚揚 155
讚樂 0
讚歌 6
讚歎 71
讚美 568
讚美有加 1
讚美歌 0
讚美詩 9
讚羨 1
讚許 63
讚詞 3
讚譽 65
讚賞 129
讚賞不已 3
讚辭 0
讚頌 38
讛 0
讜 15
讜言 0
讜論 1
讜辭 0
讝 0
讞 29
讟 42
谷 44
谷 4418
谷口 17
谷地 55
谷城 2
谷壁 3
谷子 18
谷川 44
谷底 301
谷氨酸 1
谷草 0
谷象 0
谷關 135
谸 0
谹 28
谺 0
谺 0
谺 0
谻 34
谼 6
谽 7
谾 10
谿 53
谿 53
谿壑 0
谿山行旅 1
谿谷 1
豀 0
豀 0
豁 66
豁 66
豁 66
豁上 0
豁免 109
豁免權 27
豁出 12
豁出來 0
豁出去 4
豁出去 4
豁口 0
豁子 0
豁拳 0
豁然 36
豁然貫通 2
豁然開朗 30
豁脣子 0
豁達 39
豁達大度 0
豁開 0
豂 95
豃 3
豃 3
豄 0
豄 0
豄 0
豅 4
豆 1477
豆乳 1
豆乾 1
豆剖瓜分 0
豆子 170
豆殼 2
豆汁 4
豆沙 82
豆沙月餅 0
豆沙餅 2
豆油 25
豆渣 6
豆渣餅 0
豆漿 100
豆漿店 6
豆狀 8
豆瓣 13
豆瓣醬 7
豆科 34
豆粒 5
豆綠 2
豆腐 487
豆腐乳 7
豆腐乾 1
豆腐干 7
豆腐皮 3
豆腐腦兒 0
豆腐腦兒 0
豆花 86
豆花湯 0
豆芽 44
豆芽菜 89
豆苗 31
豆莖 2
豆莢 27
豆菜 7
豆萁 0
豆蓉 4
豆蓉月餅 0
豆蔻 16
豆蔻年華 1
豆豆 266
豆豉 11
豆豉 11
豆象 9
豆醬 1
豆類 178
豆餅 21
豇 28
豇豆 4
豈 325
豈 325
豈不 294
豈不是 175
豈只 32
豈可 101
豈敢 10
豈是 103
豈有 44
豈有此理 14
豈止 8
豈能 121
豈非 126
豉 288
豉 288
豊 237
豋 40
豌 79
豌豆 211
豌豆莢 2
豍 14
豎 250
豎井 6
豎子 0
豎琴 99
豎直 2
豎眉瞪眼 0
豎眼 0
豎立 101
豎立起來 2
豎笛 94
豎著 9
豎蜻蜓 0
豎起 74
豏 34
豐 9821
豐功 3
豐功偉業 30
豐功偉績 3
豐功大業 0
豐南 48
豐厚 98
豐原 932
豐原市 1698
豐台 9
豐壤 0
豐姿 3
豐富 4623
豐富多彩 15
豐富性 26
豐實 93
豐年 700
豐年祭 149
豐年蝦 1
豐收 195
豐業 97
豐沛 129
豐滿 104
豐潤 18
豐濱 213
豐產 22
豐產田 0
豐田 625
豐登 4
豐盈 50
豐盛 377
豐碩 258
豐稔 7
豐美 46
豐胸 1
豐腴 21
豐臣秀吉 38
豐茂 11
豐草 0
豐衣足食 13
豐裕 35
豐足 10
豐采 3
豐饒 26
豐饒豐足 0
豑 0
豓 0
豔 3000
豔光四射 2
豔史 2
豔姿 2
豔媚 1
豔歌 2
豔福 2
豔羨 2
豔若桃李 0
豔詩 0
豔質 0
豔遇 19
豔陽 98
豔陽天 10
豔麗 74
豕 405
豖 220
豗 10
豘 0
豙 0
豚 365
豚骨 1
豚鼠 0
豛 0
豜 81
豝 36
豞 0
豟 1
豠 0
象 3243
象人 8
象名 0
象山 86
象形 16
象形字 5
象形文 1
象形文字 13
象徵 1011
象徵主義 16
象徵性 93
象徵著 108
象棋 437
象樣 0
象牙 131
象牙塔 31
象牙杖 0
象牙海岸 38
象牙質 0
象牙針尖 0
象生 4
象皮 5
象笏 1
象管 0
象限 36
象限儀 0
象頭 9
象鼻 34
象鼻蟲 6
豢 182
豢養 29
豣 0
豣 0
豤 8
豥 4
豦 8
豧 0
豨 8
豩 1
豪 4788
豪俠 2
豪傑 60
豪勇 6
豪右 1
豪士 12
豪壯 7
豪奢 3
豪奪 7
豪宅 0
豪客 11
豪富 5
豪強 8
豪性 0
豪情 107
豪情壯志 8
豪情萬丈 3
豪放 35
豪放不羈 6
豪放女 7
豪族 4
豪氣 42
豪氣萬丈 1
豪氣萬千 1
豪爽 66
豪紳 5
豪興 13
豪舉 0
豪華 1038
豪華型 81
豪華版 78
豪言壯語 2
豪語 15
豪豬 25
豪賭 17
豪邁 64
豪邁不羈 4
豪門 54
豪門子弟 0
豪門巨室 0
豪雄 1
豪雨 364
豪雨成災 22
豪飲 2
豫 320
豫先 3
豫劇 53
豫州 22
豬 6502
豬仔 34
豬倌 0
豬兒 3
豬兒 3
豬八戒 26
豬公 15
豬哥 28
豬圈 12
豬場 25
豬場 258
豬尾 16
豬年 9
豬心 11
豬排 35
豬母 9
豬油 37
豬玀 6
豬瘟 390
豬皮 33
豬籠草 8
豬肉 983
豬肉乾 2
豬肉餅 0
豬肚 6
豬肚 6
豬肝 26
豬肝色 0
豬腦 15
豬腰 0
豬腳 104
豬腳麵線 6
豬舍 134
豬血 50
豬血糕 1
豬蹄 2
豬蹄膀 0
豬蹄膀 0
豬隻 0
豬頭 154
豬食 0
豬鬃 36
豬鼻 3
豭 49
豯 138
豰 66
豱 65
豲 40
豳 39
豴 0
豴 0
豵 17
豶 6
豷 1
豸 177
豹 662
豹子 46
豹死留皮 0
豹皮 6
豹皮帽 0
豹紋 19
豹貓 0
豺 21
豺狼 24
豺狼當道 0
豻 211
豻 211
豼 0
豼 0
豼 0
豽 14
豾 0
豿 0
貀 5
貁 9
貂 255
貂皮 7
貂皮大衣 5
貂蟬 14
貂鼠 14
貃 0
貃 0
貄 8
貅 2
貆 1
貇 0
貈 0
貈 0
貈 0
貈 0
貈 0
貉 86
貉 86
貉 86
貊 116
貋 0
貋 0
貌 1863
貌不驚人 6
貌似 24
貌合神離 12
貌寢 1
貌相 3
貌美 17
貌美如花 1
貍 763
貍奴 0
貍貓 8
貎 0
貏 12
貐 147
貑 132
貒 189
貓 5070
貓兒 84
貓兒 84
貓兒眼 0
貓兒眼 0
貓叫 11
貓叫聲 2
貓咪 525
貓哭老鼠 0
貓哭耗子 1
貓哭耗子假慈悲 1
貓沙 1
貓熊 57
貓王 44
貓王迷 0
貓眼 49
貓眼石 9
貓砂 1
貓肉 0
貓頭 13
貓頭鷹 93
貔 117
貔虎 0
貔貅 2
貕 92
貖 0
貗 25
貘 29
貙 13
貙 13
貚 1
貛 0
貜 116
貝 2727
貝他係數 1
貝他射線 1
貝他粒子 1
貝他蛻變 1
貝克 206
貝克勒耳 1
貝克特 1
貝勒 47
貝多芬 372
貝拉 59
貝果 1
貝柱 0
貝殼 198
貝殼屋 0
貝殼類 5
貝爾 324
貝爾格萊德 2
貝爾法斯特 1
貝登堡 35
貝蒂 33
貝西 15
貝類 355
貝魯特 68
貝齒 4
貞 2018
貞卜文字 1
貞固 0
貞女 13
貞婦 1
貞德 28
貞操 50
貞潔 29
貞烈 8
貞節 29
貞節牌坊 5
貞觀 19
貞觀之治 2
貟 0
貟 0
貟 0
負 8813
負丘 0
負債 514
負債累累 13
負傷 46
負反饋 1
負向 40
負回饋 5
負得起 0
負德 0
負心 14
負心人 2
負心漢 4
負手就擒 0
負擔 2676
負擔得起 26
負數 49
負極 34
負氣 12
負片 52
負疚 0
負笈 32
負篋 0
負約 1
負累 9
負罪 3
負義 4
負荊 4
負荊請罪 7
負荷 611
負荷量 66
負薪 1
負薪救火 0
負號 6
負負 3
負負得正 4
負責 7990
負責人 8816
負責任 201
負起 267
負載 422
負載量 10
負重 47
負重致遠 0
負隅頑抗 4
負離子 65
負電 18
負電子 2
負電流 1
負電荷 9
負面 596
負面性 2
負項 1
財 8647
財主 33
財佈施 1
財力 365
財力雄厚 10
財務 9370
財務報表 395
財務科 2
財務結構 117
財務費用 1
財務預測 0
財勢 5
財團 761
財團法人 5502
財報 0
財大氣粗 7
財富 560
財寶 45
財帛 19
財政 1903
財政危機 11
財政學 90
財政寡頭 0
財政局 63
財政年度 4
財政廳 130
財政廳長 2
財政支出 20
財政收入 26
財政率 0
財政資本 0
財政資金 1
財政部 4090
財政部長 185
財政預算 34
財施 1
財東 1
財權 156
財氣 13
財測 0
財源 459
財源滾滾 18
財物 660
財產 2215
財產權 456
財產法 24
財產清查 1
財產目錄 39
財產稅 17
財產關係 1
財用 2
財界 6
財神 105
財神爺 22
財禮 1
財稅 826
財稅界 0
財經 15156
財經專家 6
財經部門 9
財經首長 24
財貨 42
財貿 5
財貿科 0
財賦 1
財路 10
財迷 0
財迷心竅 1
財運 75
財運亨通 10
財金 0
財金系 0
財閥 50
貢 353
貢糖 50
貢丸 62
貢品 2
貢奉 0
貢寮 378
貢獻 1949
貢獻者 48
貢生 12
貢禮 1
貢稅 0
貣 12
貤 21
貥 48
貦 0
貦 0
貧 317
貧下中農 1
貧乏 91
貧僧 28
貧化 0
貧嘴 1
貧困 166
貧困人家 0
貧困化 0
貧國 2
貧家 4
貧家女 0
貧富 92
貧富不均 13
貧富懸殊 5
貧寒 14
貧寒人家 0
貧寒無告 0
貧弱 6
貧戶 21
貧民 31
貧民區 13
貧民窟 11
貧油 0
貧病 23
貧病交迫 1
貧瘠 82
貧礦 0
貧窮 398
貧窮人家 2
貧苦 50
貧苦人家 2
貧血 606
貧血病 1
貧血症 30
貧賤 27
貧賤不能移 2
貧賤夫妻百事哀 6
貧農 3
貧道 51
貨 4059
貨主 54
貨倉 8
貨價 94
貨到付款 56
貨品 2462
貨單 29
貨幣 2367
貨幣流通 6
貨幣資本 1
貨幣資金 2
貨底 2
貨暢其流 8
貨架 36
貨棧 12
貨樣 11
貨機 33
貨櫃 845
貨櫃船 27
貨櫃車 13
貨款 218
貨殖 7
貨比三家 33
貨比三家不吃虧 16
貨源 56
貨物 1433
貨物櫃 0
貨物稅 258
貨物運輸 22
貨物運輸保險 4
貨真價實 35
貨箱 3
貨船 25
貨艙 6
貨色 33
貨色齊全 9
貨車 421
貨車工 0
貨輪 49
貨運 1039
貨運公司 52
貨運業 22
貨運行 38
貨運車 2
貨運量 47
貨郎 9
販 845
販售 510
販售量 0
販夫走卒 19
販婦 0
販子 13
販毒 71
販賣 1155
販賣人口 9
販賣機 81
販賣部 34
販運 25
貪 466
貪便宜 7
貪利 1
貪吃 30
貪吃鬼 1
貪嘴 3
貪圖 48
貪墨 3
貪多 11
貪多必失 0
貪多無厭 0
貪好 2
貪好女色 0
貪婪 145
貪官 17
貪官汙吏 1
貪官污吏 11
貪小 0
貪小便宜 29
貪小失大 0
貪得 9
貪得無厭 15
貪得無饜 3
貪心 111
貪心不足 3
貪心不足蛇吞象 4
貪念 27
貪慕 1
貪慕虛榮 0
貪慳 1
貪慾 6
貪戀 33
貪戀著 0
貪杯 3
貪欲 5
貪求 18
貪汙 3
貪污 302
貪污受賄 2
貪污罪 5
貪濫 0
貪瀆 245
貪狠 0
貪狼 31
貪玩 37
貪生 2
貪生怕死 14
貪睡 3
貪睡蟲 0
貪腐 12
貪色 3
貪財 14
貪財害命 0
貪贓 7
貪贓枉法 47
貪錢 3
貫 819
貫串 29
貫徹 431
貫徹到底 4
貫徹執行 33
貫徹始終 12
貫注 22
貫穿 203
貫穿力 0
貫耳 6
貫軸 1
貫通 85
貫通性 0
貫連 4
責 81
責 8119
責人 47
責人從寬 0
責令 69
責任 4198
責任事故 0
責任內閣 0
責任制 42
責任心 38
責任感 135
責任準備 3
責任準備金 33
責備 153
責問 16
責善 0
責實循名 1
責己 1
責己從嚴 0
責怪 79
責成 127
責打 8
責斥 3
責無旁貸 56
責罰 11
責罵 57
責難 37
貭 0
貭 0
貮 0
貯 308
貯備 2
貯備金 0
貯存 758
貯存量 3
貯木場 1
貯木場 1
貯水 18
貯水器 0
貯水池 8
貯水量 1
貯積 13
貯積起來 0
貯蓄 1
貯藏 148
貯藏器 0
貯藏室 3
貯藏量 2
貰 6
貱 0
貲 37
貲財 0
貳 3012
貳心 2
貴 4847
貴了 99
貴事 5
貴人 118
貴人多忘事 0
貴人相助 5
貴公司 725
貴刊 79
貴國 59
貴在 39
貴地 14
貴妃 43
貴姓 222
貴婦 18
貴婦人 11
貴子 115
貴客 72
貴客臨門 1
貴寶地 24
貴寶號 1
貴州 79
貴州人 0
貴州省 27
貴幹 1
貴庚 7
貴府 32
貴得 162
貴恙 0
貴慈 11
貴戚 1
貴族 498
貴族化 7
貴族學校 4
貴族政治 0
貴校 315
貴為 38
貴為天子 0
貴的 162
貴處 26
貴要 0
貴賓 1154
貴賓卡 113
貴賓室 50
貴賓席 10
貴賤 30
貴邦 2
貴重 205
貴重品 1
貴重金屬 3
貴金屬 104
貴陽 133
貴陽市 0
貴顯 7
貴體 0
貴點 0
貵 8
貶 904
貶低 21
貶值 711
貶價 0
貶官 4
貶意 0
貶抑 29
貶損 11
貶斥 1
貶義 0
貶職 0
貶謫 7
貶責 0
貶逐 1
貶黜 1
買 10019
買一 359
買一送一 40
買下 239
買下來 24
買不到 174
買不起 71
買主 320
買了 1003
買來 168
買來買去 0
買價 30
買光 1
買入 149
買到 933
買去 12
買單 49
買回 300
買回來 38
買回去 12
買定 3
買客 8
買家 1
買帳 2
買得 652
買得起 27
買成 7
買斷 47
買方 471
買春 36
買書 104
買東西 137
買氣 337
買的 652
買票 263
買空 0
買空賣空 2
買給 28
買菜 63
買著 4
買買 21
買賣 4751
買賣人 4
買賣成 0
買賣業 78
買走 11
買起 5
買起來 8
買超 344
買辦 6
買辦階級 0
買通 10
買進 1588
買過 51
買醉 5
買點 125
貸 1164
貸了 1
貸個 1
貸借 4
貸入 0
貸出 14
貸取 0
貸夠 0
貸好 0
貸放 77
貸方 48
貸有 2
貸款 3697
貸款額 10
貸款額度 104
貸法 0
貸給 7
貸過 3
貸金 15
貹 9
貺 13
費 11
費 11896
費了 89
費事 14
費力 98
費力氣 2
費功夫 3
費加洛 32
費勁 7
費周章 43
費唇舌 10
費城 157
費工 9
費工夫 8
費心 139
費手腳 3
費時 236
費時費事 5
費時間 7
費氏數列 0
費爾 35
費爾巴哈 7
費爾茲獎 2
費率 1041
費瑪 1
費瑪最後定理 4
費用 6255
費盡 64
費盡心思 16
費盡心機 2
費眼 1
費神 15
費脣舌 2
費解 23
費費 5
費邊主義 0
費錢 3
費電 24
費點 4
費點心思 3
貼 9317
貼上 609
貼上來 35
貼上去 14
貼下 1
貼下來 0
貼下去 0
貼來 5
貼來貼去 1
貼個 12
貼入 25
貼出 222
貼出來 22
貼出去 4
貼切 107
貼到 89
貼反 0
貼地 14
貼地而行 0
貼好 13
貼完 13
貼布 42
貼平 1
貼得 167
貼心 324
貼心話 4
貼換 0
貼現 77
貼現率 29
貼的 167
貼紙 544
貼緊 6
貼縫 0
貼花 13
貼補 17
貼補家用 18
貼起 3
貼起來 2
貼身 106
貼身衣服 2
貼身衣物 4
貼近 123
貼進 5
貼進來 0
貼進去 0
貼過 11
貼過來 6
貼過去 2
貼金 7
貼錢 11
貽 83
貽人口實 0
貽人話柄 0
貽害 6
貽患 1
貽患終身 0
貽禍 2
貽笑 3
貽笑大方 11
貽羞 0
貽誤 9
貽誤戎機 1
貽誤終身 0
貽贈 0
貾 4
貾 4
貿 3624
貿易 8250
貿易中心 67
貿易公司 176
貿易協定 32
貿易商 392
貿易逆差 134
貿易順差 75
貿易額 77
貿易風 58
貿然 139
貿然行事 3
貿貿然 3
賀 1947
賀信 3
賀儀 5
賀函 2
賀卡 1154
賀喜 39
賀客 6
賀客盈門 1
賀帖 0
賀年 45
賀年卡 113
賀年片 2
賀柬 0
賀歲 23
賀爾蒙 135
賀爾蒙針 0
賀禮 50
賀蘭 18
賀蘭山 3
賀表 1
賀詞 78
賀辭 10
賀電 23
賁 57
賁 57
賁士 0
賁然 0
賁門 10
賁門 10
賂 116
賃 96
賃 96
賃金 0
賄 423
賄 423
賄款 0
賄賂 92
賄賂罪 2
賄選 706
賅 80
賆 0
資 33295
資俸 0
資優 351
資優兒童 23
資優班 191
資優生 99
資力 17
資助 261
資工 0
資政 224
資政院 1
資料 85127
資料中心 63156
資料卡 42
資料室 253
資料局 2
資料庫 25194
資料栓 0
資料袋 38
資料錄 37
資方 101
資方人員 0
資方代表 3
資本 2093
資本主義 330
資本主義制度 5
資本主義國家 13
資本主義地租 0
資本主義社會 34
資本周轉 0
資本家 125
資本帝國 3
資本帝國主義 0
資本循環 0
資本有機 0
資本有機構成 0
資本積累 5
資本論 21
資本財 38
資本輸出 3
資本額 9347
資材 251
資格 7249
資格比賽 0
資格考試 132
資歷 335
資治通鑑 116
資深 1140
資深教師 23
資深職員 1
資淺 20
資源 38207
資源回收 0
資源班 171
資產 2135
資產負債 70
資產負債表 120
資產階級 47
資產階級專政 1
資產階級權利 0
資產階級民主 1
資產階級法權 0
資產階級革命 0
資科 0
資稟 0
資策會 0
資管 0
資訊 99928
資訊學會 165
資訊學系 119
資訊展 596
資訊工業 558
資訊工業策進會 356
資訊所 388
資訊月 1359
資訊研究所 44
資訊社 280
資訊系 464
資訊網 98953
資財 13
資質 92
資遣 0
資遣費 0
資金 4319
資金佔用 0
資金來源 93
資金占用 1
資金平衡 0
資金平衡表 0
資金運用 97
資金雄厚 7
賈 139
賈 139
賈 1398
賈人 6
賈南風 0
賈后 0
賈寶玉 14
賈後 0
賈思勰 5
賈禍 1
賈誼 7
賈誼 7
賉 0
賊 618
賊 618
賊亮 1
賊人 28
賊兵 19
賊匪 0
賊子 8
賊寇 1
賊巢 1
賊店 0
賊徒 1
賊性 0
賊營 1
賊眉鼠眼 0
賊眼 1
賊禿 6
賊窩 6
賊船 6
賊贓 0
賊車 0
賊軍 1
賊頭賊腦 6
賊風 2
賊首 2
賊骨頭 0
賊骨頭 0
賊黨 1
賋 0
賌 13
賍 0
賏 4
賐 0
賑 1918
賑恤 0
賑救 1
賑款 2
賑濟 21
賑災 252
賑糧 1
賑饑 0
賒 310
賒借 106
賒債 0
賒帳 6
賒旗鎮 0
賒欠 5
賒貸 1
賒賣 0
賒賬 1
賓 1903
賓主 18
賓主盡歡 22
賓主關係 2
賓士 275
賓士車 26
賓士轎車 3
賓夕凡尼亞 9
賓夕凡尼亞州 2
賓夕法尼亞大學 14
賓客 85
賓州 440
賓從 1
賓果 45
賓至如歸 39
賓語 3
賓鐵 0
賓館 790
賔 0
賔 0
賕 4
賖 0
賗 1
賙 8
賙濟 9
賚 17
賛 0
賜 1501
賜 1501
賜了 5
賜予 60
賜函 6
賜函 6
賜告 8
賜告 8
賜姓 2
賜姓 2
賜官 1
賜官 1
賜宴 1
賜宴 1
賜教 528
賜教 528
賜有 1
賜死 2
賜死 2
賜爵 6
賜福 119
賜福 119
賜給 320
賜給 320
賜與 36
賜諡 0
賝 33
賞 3360
賞了 15
賞來賞去 0
賞光 4
賞到 10
賞得 3
賞心 26
賞心悅事 1
賞心悅目 96
賞月 58
賞望 0
賞析 1
賞測站 0
賞玩 28
賞畫 3
賞給 4
賞罰 18
賞罰分明 6
賞罰嚴明 1
賞臉 6
賞花 90
賞花季節 4
賞識 51
賞賜 81
賞過 0
賞金 26
賞銀 0
賞錢 5
賞錯 0
賞鑑 1
賞點 1
賟 16
賠 2250
賠上 36
賠上來 0
賠上去 0
賠不是 6
賠不起 1
賠了 34
賠了夫人 5
賠了夫人又折兵 8
賠償 1512
賠償物 0
賠償費 6
賠光 5
賠出 1
賠到 3
賠得 29
賠得起 0
賠掉 13
賠損 0
賠本 26
賠本生意 1
賠款 55
賠的 29
賠禮 0
賠笑 5
賠笑臉 1
賠罪 14
賠錢 64
賠錢貨 2
賡 209
賡續 2
賢 5329
賢人 47
賢伉儷 12
賢侄 6
賢內助 10
賢后 0
賢哲 11
賢士 6
賢妹 9
賢妻 4
賢妻良母 22
賢孝 11
賢孫 1
賢弟 30
賢徒 1
賢德 17
賢愚 3
賢慧 25
賢才 3
賢昆仲 2
賢明 40
賢淑 17
賢相 2
賢能 12
賢良 7
賢達 51
賣 8207
賣不出去 12
賣不掉 10
賣主 16
賣主求榮 1
賣乖 3
賣了 145
賣人情 0
賣來賣去 0
賣俏 0
賣價 90
賣光 22
賣出 1468
賣出去 10
賣到 62
賣力 121
賣力氣 0
賣勁 1
賣友求榮 1
賣命 38
賣唱 7
賣國 12
賣國求榮 2
賣國賊 8
賣場 87
賣場 878
賣壓 376
賣字號 0
賣完 35
賣官鬻爵 1
賣家 150
賣帳 0
賣座 94
賣弄 51
賣弄口舌 0
賣弄風情 4
賣得 79
賣掉 162
賣文 4
賣斷 41
賣方 426
賣春 19
賣本事 0
賣東西 44
賣淫 69
賣破綻 0
賣票 20
賣空 2
賣笑 16
賣給 244
賣菜 8
賣著 3
賣藝 12
賣藝者 4
賣賣 7
賣超 0
賣身 48
賣身契 8
賣過 13
賣錢 22
賣關子 10
賣面子 0
賣點 118
賤 25456
賤事 1
賤人 123
賤價 11
賤價出售 13
賤內 3
賤妾 18
賤恙 0
賤業 0
賤物 0
賤胚 61
賤號 0
賤貨 16
賤買賤賣 0
賤賣 64
賤軀 0
賤骨頭 3
賥 31
賦 896
賦予 779
賦役 5
賦性 3
賦斂 0
賦有 14
賦格曲 9
賦歸 33
賦稅 260
賦稅額 3
賦稟 0
賦與 113
賦詩 12
賦課 0
賦貢 0
賦閒 4
賦閒在家 4
賧 20
賨 10
賩 0
賩 0
質 534
質 9135
質和量 9
質問 91
質因數 4
質地 527
質塊 2
質子 49
質弱 11
質感 301
質成 9
質押 151
質數 44
質料 110
質朴 1
質樸 45
質疑 1476
質的 58
質科 3
質素 45
質要 4
質言 3
質言之 9
質詢 475
質詢權 5
質詢者 0
質譜儀 116
質變 39
質量 531
質量不滅 1
質量不滅定律 0
質量守恆 6
質量指標 1
質量數 1
質量檢驗 2
質量比 2
質量管理 4
質體 72
質點 43
質點力學 0
賫 0
賬 147
賬上 0
賬冊 1
賬務 0
賬單 3
賬戶 3
賬房 1
賬本 2
賬目 2
賬篷 1
賬簿 6
賭 634
賭了 14
賭債 4
賭具 2
賭博 333
賭友 0
賭咒 5
賭坊 13
賭城 100
賭場 31
賭場 318
賭客 16
賭局 32
賭帳 1
賭徒 47
賭性堅強 5
賭戶 0
賭房 0
賭本 2
賭棍 1
賭檯 1
賭氣 22
賭法 5
賭注 50
賭王 48
賭盤 0
賭窟 0
賭資 2
賭賬 0
賭賽 2
賭贏 4
賭輸 9
賭過 0
賭金 16
賭錢 17
賭風 13
賭馬 7
賭馬場 0
賭馬場 0
賭鬼 11
賮 248
賯 0
賰 193
賱 156
賲 0
賳 222
賴 5469
賴以為生 10
賴以維生 19
賴債 1
賴和 1
賴婚 2
賴帳 13
賴床 30
賴掉 0
賴於 109
賴比瑞亞 56
賴氨酸 0
賴漢英 0
賴爾德 0
賴皮 41
賴索托 18
賴索托王國 4
賴聲川 1
賴賬 1
賴過 1
賵 130
賶 0
賷 0
賸 2673
賸下 11
賸餘 212
賸餘價值 1
賸餘物資 0
賹 69
賺 3363
賺 0
賺下來 0
賺了 104
賺些 5
賺人眼淚 3
賺個 10
賺個飽 0
賺到 172
賺去 1
賺取 246
賺大錢 125
賺完 1
賺得 64
賺得 67
賺有 0
賺法 0
賺的 66
賺賺 5
賺賺看 0
賺起 0
賺起來 1
賺過 4
賺過來 0
賺過去 0
賺錢 1101
賺錢事業 1
賺頭 20
賺飽 6
賻 10
賻儀 0
購 2500
購併 0
購入 428
購屋 1137
購得 235
購料 51
購有 9
購油 13
購油案 0
購物 8592
購物中心 725
購物單 5
購用 15
購置 1270
購自 41
購貨 25
購貨單 0
購買 10498
購買力 72
購買慾 8
購辦 24
購銷 1
購銷差價 0
購食 3
賽 6546
賽到 1
賽力散 0
賽員 9
賽場 38
賽場 38
賽德克 1
賽會 25
賽滿 0
賽狗 6
賽狗場 0
賽狗場 0
賽球 8
賽璐珞 9
賽神 0
賽程 504
賽船 3
賽豬公 0
賽跑 114
賽跑員 0
賽跑者 2
賽跑選手 0
賽車 854
賽車場 41
賽車場 41
賽車狂 1
賽車迷 1
賽車選手 5
賽過 5
賽馬 76
賽馬場 6
賽馬場 6
賽鴿 1
賾 50
賿 0
贀 7
贁 0
贂 4
贃 0
贄 19
贄敬 0
贄見 0
贅 557
贅婿 1
贅字 0
贅文 3
贅疣 0
贅瘤 8
贅筆 2
贅肉 8
贅言 18
贅詞 1
贅述 32
贅飾 0
贆 5
贇 4
贈 1577
贈予 30
贈別 4
贈券 0
贈品 1176
贈序 1
贈獎 185
贈禮 185
贈答 1
贈給 13
贈與 1023
贈與者 2
贈言 9
贈送 1189
贈送品 2
贈閱 248
贈閱本 2
贉 4
贊 434
贊不絕口 3
贊助 1716
贊助人 21
贊助者 110
贊同 378
贊嘆 12
贊嘆不已 2
贊成 946
贊成者 22
贊揚 6
贊歌 3
贊歎 10
贊歎不已 0
贊禮 1
贊美 19
贊許 13
贊譽 4
贊賞 21
贊頌 3
贋 0
贋 0
贍 146
贍養 19
贍養費 25
贎 0
贏 1659
贏上 0
贏了 135
贏人 8
贏出 1
贏利 6
贏到 10
贏取 107
贏回 19
贏回來 1
贏回去 0
贏定 0
贏定了 6
贏家 424
贏得 1014
贏得 188
贏球 33
贏的 187
贏過 32
贏過來 0
贏過去 0
贏錢 12
贏隊 0
贐 4
贑 0
贒 0
贒 0
贓 58
贓品 0
贓官 3
贓款 14
贓物 29
贓車 0
贔 4
贕 1
贖 156
贖人 2
贖命 0
贖回 209
贖款 7
贖罪 91
贖罪券 1
贖買 1
贖身 2
贖金 27
贗 142
贗品 8
贗本 0
贙 5
贚 0
贛 35
贛南盆地 0
贛州 7
贛江 0
贜 0
赤 2702
赤光 1
赤化 14
赤地 6
赤城 15
赤壁 35
赤壁之戰 13
赤壁賦 18
赤子 221
赤子之心 56
赤字 365
赤崁樓 57
赤嵌樓 60
赤帶 0
赤心 2
赤忱 3
赤手 6
赤手空拳 13
赤日 0
赤松 7
赤條條 4
赤棉 0
赤楊 14
赤熱 1
赤痢 28
赤眉 2
赤眼蜂 0
赤禍 3
赤腳 29
赤腳大仙 2
赤腳醫生 1
赤膊 8
赤膊上陣 0
赤膽 3
赤膽忠心 0
赤色 24
赤螯蟹 0
赤衛軍 0
赤衛隊 0
赤裸 115
赤裸裸 87
赤褐色 7
赤誠 10
赤貧 6
赤貧如洗 1
赤足 18
赤身 11
赤身露體 4
赤身露體 4
赤道 332
赤道區 2
赤道幾內亞 2
赤金 1
赤銅 0
赤銅礦 0
赤鐵礦 5
赤霉素 0
赤馬 23
赥 0
赦 134
赦免 125
赦免令 0
赦宥 1
赦罪 35
赧 18
赧愧 0
赧然 7
赧赧 0
赧顏 0
赨 6
赩 2
赫 1021
赫塞 14
赫德遜河 0
赫斯 34
赫然 247
赫爾 30
赫爾新基 7
赫爾辛基 54
赫胥黎 6
赫胥黎 6
赫茲 29
赫赫 20
赫赫有名 24
赫魯曉夫 3
赫魯雪夫 6
赬 0
赭 42
赭石 1
赮 0
赯 0
走 6639
走上 325
走上來 8
走上去 4
走上台 1
走下 74
走下來 19
走下去 63
走下台 2
走下坡 68
走不了 6
走不到 8
走不動 13
走不得 3
走不得 3
走不過去 0
走不開 3
走了 896
走人 67
走來 281
走來走去 37
走光 11
走入 554
走內線 0
走出 1005
走出來 80
走出去 70
走到 887
走動 155
走勢 1862
走勢圖 53
走卒 1
走去 177
走向 1476
走向前來 1
走向前去 4
走味 2
走唱 35
走嘴 0
走回 118
走回來 7
走回去 5
走回路 0
走回頭 2
走在 561
走夠 1
走失 402
走好 21
走好運 2
走廊 655
走後 42
走後門 13
走得 148
走得 448
走得快 1
走得慢 0
走投無路 27
走掉 20
走散 6
走時 39
走樣 26
走步 14
走江湖 8
走漏 9
走漏消息 2
走漏風聲 3
走火 29
走火入魔 53
走為上策 9
走狗 24
走獸 24
走的 447
走禽 1
走私 417
走私案 8
走私者 0
走私船 1
走私貨 7
走筆 5
走筆疾書 0
走索 4
走累 4
走老運 0
走船 1
走著 237
走訪 235
走調 12
走資派 4
走走 807
走走停停 17
走走看 3
走起 6
走起來 3
走起路來 24
走路 563
走近 117
走近路 0
走進 671
走進來 34
走進去 32
走運 10
走遍 119
走遍 119
走過 1123
走過來 87
走過去 68
走道 121
走道兒 1
走道兒 1
走遠 31
走遠路 0
走避 24
走邊 5
走錯 48
走門路 1
走開 66
走霉運 0
走頭無路 1
走風 0
走馬 103
走馬上任 16
走馬換將 2
走馬燈 98
走馬看花 29
走點 1
赱 0
赲 720
赳 11
赳赳 2
赳赳 2
赳赳武夫 0
赳赳武夫 0
赴 4158
赴任 41
赴告 0
赴宴 10
赴救 0
赴敵 0
赴會 36
赴湯蹈火 6
赴約 30
赴義 3
赴考 4
赴英 58
赴試 0
赴難 0
赶 220
起 20421
起不來 13
起了 319
起事 17
起伏 369
起伏不定 32
起作用 24
起來 3246
起信論 10
起價 8
起先 89
起先是 13
起兵 26
起初 251
起初是 14
起勁 38
起動 217
起動力 0
起動器 10
起動機 7
起動轉矩 2
起司 1
起吊 10
起名 9
起名稱 0
起吸器 0
起因 97
起因於 74
起坐 9
起坐室 0
起士 1
起始 384
起始值 17
起子 71
起學潮 0
起家 113
起居 108
起居室 27
起居生活 7
起居飲食 4
起工 4
起床 377
起床號 2
起復 10
起意 20
起手 25
起承轉合 24
起早 6
起早摸黑 0
起早貪黑 0
起更 42
起會 10
起步 291
起死回生 79
起毛 47
起水泡 18
起泡 58
起源 550
起源於 113
起火 162
起火燃燒 18
起爆 9
起爆點 0
起用 61
起疑 18
起疑心 6
起疹 17
起發性 0
起眼 26
起碼 303
起租 7
起程 67
起稿 40
起立 70
起站 33
起端 2
起筆 24
起網 21
起縐 3
起義 48
起義者 0
起興 15
起舞 135
起航 7
起航出海 0
起色 81
起草 80
起落 73
起落架 29
起落點 0
起薪 1
起行 20
起複 0
起見 127
起解 8
起訖 104
起訖之間 0
起訴 480
起訴書 29
起誓 23
起貨 2
起贓 0
起起落落 31
起跑 201
起跑器 0
起跑線 8
起跑點 33
起跳 49
起跳板 2
起跳線 0
起跳點 2
起身 299
起運 25
起運港 0
起運點 0
起釁 0
起重 290
起重機 148
起重船 3
起錨 6
起開 58
起降 128
起電機 0
起音 0
起頭 64
起風 19
起飛 536
起飛時 22
起首 13
起駕 2
起鬨 14
起點 383
赸 14
赹 13
赺 0
赻 974
赽 15
赾 0
赿 0
赿 0
趀 4
趁 699
趁人之危 2
趁便 0
趁勢 48
趁心 0
趁心如意 0
趁早 67
趁早行動 0
趁機 209
趁機行事 0
趁火打劫 16
趁熱 33
趁熱打鐵 1
趁空 4
趁著 234
趁虛而入 9
趁願 0
趂 0
趃 0
趄 3
超 10778
超乎 186
超人 1680
超人一等 6
超人氣 0
超人裝 2
超低 371
超低壓 0
超低溫 38
超倫 25
超值 668
超再生 0
超凡 62
超凡入聖 7
超出 461
超出範圍 6
超前 131
超升 2
超卓 18
超商 0
超國家主義 2
超外差 1
超外差式 0
超外差式收音機 0
超大型 238
超導 95
超導體 63
超市 985
超度 23
超強 486
超我 1
超技 7
超拔 3
超支 15
超收 33
超新星 0
超時 186
超曠 0
超次 16
超然 135
超然物外 1
超現實 246
超生 12
超產 1
超短 31
超短波 3
超短波 3
超等 13
超級 8194
超級大國 8
超級市場 28
超級市場 289
超級強國 4
超絕 11
超縮微膠片 0
超群 184
超群絕倫 0
超聲波 23
超聲波 23
超脫 75
超自然 61
超計劃利潤 0
超越 1566
超越數 1
超越限度 0
超車 53
超載 58
超速 280
超速行駛 5
超連結 0
超逸 3
超過 7138
超重 99
超重量級 8
超階級 0
超音波 1308
超音波 1308
超音速 47
超音速噴射客機 0
超音速噴射機 0
超音速飛行 0
超頻 0
超額 174
超額利潤 9
超額錄取 0
超高 495
超高壓 18
超高溫 13
超高頻 10
超點 1
超齡 13
超齡兒童 0
趆 0
趇 0
趈 0
趉 1
越 5272
越位 42
越低 63
越來 66
越來越 1996
越來越好 26
越俎代庖 5
越光米 1
越共 24
越冬 18
越出 7
越劇 13
越加 26
越區 50
越區就讀 8
越南 1014
越南人 10
越南共和國 0
越南化 2
越南話 1
越國 11
越在 7
越境 20
越多越好 24
越好 210
越式 1
越式料理 1
越成 2
越戰 119
越是 111
越會 16
越權 10
越次 0
越洋 36
越洋電話 15
越演越烈 0
越牆 4
越獄 8
越王 10
越王勾踐 1
越界 107
越發 73
越禮 0
越站 4
越站不停 0
越級 12
越級報考 0
越線 7
越能 55
越軌 5
越軍 9
越過 245
越野 307
越野賽 27
越野車 109
越雷池一步 9
越雷池一步 9
趋 0
趋 0
趌 0
趍 0
趎 0
趏 0
趐 0
趑 0
趒 0
趓 0
趔 0
趕 1599
趕上 189
趕上來 5
趕上去 1
趕下 10
趕下來 3
趕下去 1
趕下台 1
趕不上 52
趕不出 1
趕不出來 1
趕不到 2
趕不及 8
趕了 29
趕任務 0
趕來 117
趕來趕去 5
趕修 0
趕做 1
趕入 6
趕出 80
趕出來 10
趕出去 18
趕到 228
趕去 44
趕向 4
趕回 85
趕回來 17
趕回到 1
趕回去 9
趕在 141
趕場 44
趕場 44
趕場子 0
趕場子 0
趕寫 0
趕工 162
趕往 85
趕得 1
趕得上 15
趕得出 0
趕得到 0
趕得及 2
趕忙 45
趕快 1483
趕快來 124
趕快到 42
趕快去 208
趕成 3
趕戲 8
趕早 6
趕明兒 0
趕明兒 0
趕時間 23
趕時髦 9
趕有 0
趕洗 0
趕活 0
趕演 0
趕牛 7
趕盡殺絕 14
趕程 1
趕稿 150
趕緊 521
趕練 0
趕羊 7
趕考 7
趕著 73
趕製 11
趕課 4
趕走 71
趕赴 51
趕起 1
趕起來 0
趕超 1
趕趕 4
趕跑 3
趕路 38
趕車 6
趕辦 10
趕造 0
趕進 3
趕進來 0
趕進去 0
趕過 11
趕過來 6
趕過去 5
趕錄 2
趕開 1
趕集 41
趕馬 0
趕騾 0
趕鳥 2
趕鴨子 0
趕鴨子上架 6
趕點 0
趖 10
趗 0
趘 0
趙 6430
趙人 10
趙匡胤 13
趙國 50
趙子龍 22
趙孟頫 20
趙家 30
趙少康 1
趙樹理 3
趙王 5
趙聲 0
趙莊 2
趚 0
趛 17
趜 13
趝 0
趞 0
趞 0
趟 232
趠 9
趡 6
趢 0
趣 102
趣 1026
趣事 111
趣味 751
趣味十足 11
趣味性 109
趣味競賽 81
趣旨 1
趣聞 56
趣話 7
趣談 52
趤 0
趥 206
趧 135
趨 15
趨 1507
趨之若騖 15
趨之若鶩 24
趨使 14
趨光性 3
趨利 2
趨利避害 0
趨勢 7049
趨吉避凶 35
趨向 291
趨拜 0
趨於 242
趨於平穩 1
趨於穩定 25
趨炎附勢 4
趨近 66
趨附 5
趩 0
趪 4
趫 1
趬 4
趭 3
趮 23
趯 6
趯 6
趲 31
趲幹 1
趲步 1
趲程 1
趲行 1
趲足 1
足 44
足 4465
足下 54
足不出戶 19
足以 1719
足以自豪 2
足印 29
足壇 12
足夠 1486
足委會 0
足尖 8
足底 40
足心 3
足恭 0
足敷所需 0
足智多謀 15
足月 22
足本 4
足歲 123
足球 924
足球場 42
足球場 42
足球大賽 10
足球賽 237
足球隊 51
足用 0
足繭 0
足色 3
足衣足食 0
足見 150
足賽 77
足足 108
足足地 2
足足有 37
足足有餘 3
足趾 7
足跟 9
足跡 431
足踝 0
足部 173
足金 2
足額 50
足食 0
足食足兵 0
趴 96
趴下 7
趴下來 1
趴下去 1
趴倒 3
趴在 100
趴得 7
趴的 7
趴著 29
趴著不動 0
趴趴熊 0
趵 8
趵突泉 0
趵趵 0
趶 5
趷 17
趹 12
趺 29
趺坐 7
趻 0
趻 0
趼 285
趽 0
趾 252
趾甲 8
趾骨 0
趾高氣揚 3
趾高氣昂 6
趿 20
趿 20
趿拉 0
趿拉 0
跀 0
跁 144
跂 123
跂想 0
跂望 0
跃 0
跅 6
跆 37
跆拳 221
跆拳道 607
跇 4
跈 5
跉 0
跊 0
跋 254
跋前躓後 0
跋山涉水 14
跋扈 13
跋文 0
跋涉 32
跌 2456
跌下 18
跌下來 25
跌下去 4
跌了 80
跌來跌去 0
跌倒 231
跌停 331
跌停板 42
跌傷 4
跌價 93
跌入 33
跌到 118
跌勢 89
跌在 16
跌多 5
跌多漲少 9
跌宕 11
跌幅 252
跌幅已深 12
跌得 63
跌打損傷 6
跌死 4
跌的 62
跌破 531
跌破眼鏡 12
跌碎 1
跌股 11
跌自 0
跌落 88
跌蕩 1
跌足 4
跌跌 36
跌跌撞撞 31
跌跤 0
跌進 13
跌進來 0
跌進去 0
跍 4
跎 9
跏 9
跐 3
跑 2848
跑下 9
跑下來 4
跑下去 4
跑不了 7
跑不動 8
跑不掉 29
跑來 164
跑來跑去 60
跑出 138
跑出來 67
跑出去 36
跑到 559
跑去 307
跑單幫 5
跑堂 2
跑堂的 1
跑壘 13
跑壘員 49
跑得 174
跑掉 80
跑旱船 0
跑步 192
跑步走 0
跑片 0
跑的 173
跑程 4
跑腿 15
跑腿的 4
跑著 22
跑警報 0
跑走 0
跑起 5
跑起來 30
跑跑 1170
跑跑步 3
跑跑跳跳 4
跑路 33
跑跳 18
跑車 347
跑速 4
跑遍 30
跑遍 30
跑過來 24
跑過去 23
跑道 574
跑開 13
跑鞋 21
跑馬 26
跑馬場 14
跑馬場 14
跑馬燈 0
跑龍套 15
跒 0
跓 4
跔 0
跔 0
跔 0
跕 11
跖 27
跖 27
跗 21
跘 4
跙 3
跚 35
跛 70
跛子 3
跛腳 37
跛行症 0
跛足 4
跜 0
距今 90
距今已有 7
距 1410
距離 2926
跟 13122
跟上 464
跟上來 4
跟上去 2
跟下 9
跟下來 3
跟下去 1
跟不上 85
跟了 45
跟人 123
跟住 3
跟來 21
跟來跟去 0
跟出 3
跟出來 0
跟出去 3
跟到 25
跟前 88
跟前跟後 4
跟去 33
跟在 170
跟定 7
跟對 28
跟得 3
跟得上 32
跟斗 4
跟班 12
跟班的 0
跟的 30
跟緊 3
跟腳 5
跟著 1329
跟起 1
跟起來 0
跟蹤 98
跟近 2
跟進 271
跟進來 1
跟進去 2
跟過 8
跟過來 0
跟過去 15
跟隊 2
跟隨 4991
跟隨在 21
跟隨著 58
跟頭 2
跠 9
跡 754
跡地 20
跡象 420
跢 20
跣 55
跤 347
跦 164
跧 1
跨 3668
跨上 13
跨上來 0
跨上去 2
跨下 2
跨下之辱 0
跨下來 0
跨下去 0
跨了 13
跨來 0
跨來跨去 0
跨入 322
跨出 150
跨出來 1
跨出去 17
跨刀相助 2
跨到 5
跨向 19
跨國 551
跨國企業 210
跨國公司 58
跨在 14
跨年 121
跨年度 243
跨度 10
跨得 2
跨成 2
跨日 8
跨月 2
跨欄 36
跨步 30
跨步前進 0
跨步向前 0
跨河 15
跨河大橋 13
跨洲 6
跨海 90
跨海大橋 38
跨湖 1
跨湖大橋 0
跨牆 1
跨牆而過 0
跨腳 0
跨腳而過 0
跨著 1
跨行 130
跨起 0
跨起來 0
跨越 575
跨跨 0
跨躍 1
跨進 60
跨進來 2
跨進去 0
跨過 92
跨門 0
跨門而過 0
跨馬 0
跨馬而過 0
跨騎 3
跨黨 67
跩 38
跩 38
跪 349
跪下 42
跪下來 27
跪下去 4
跪了 25
跪倒 15
跪到 5
跪在 100
跪地 12
跪地求饒 13
跪姿 4
跪射 0
跪得 15
跪拜 43
跪拜禮 0
跪著 26
跫 5
跫音 21
跬 46
跭 0
跮 22
路 61183
路上 1264
路不拾遺 3
路人 908
路人皆知 7
路加 137
路加福音 70
路北 38
路口 659
路名 53
路地 20
路基 118
路子 27
路局 8
路徑 1629
路德 73
路德教派 1
路德會 7
路數 21
路易 249
路易士 65
路易斯 69
路易斯安那 37
路條 4
路標 129
路段 487
路況 0
路燈 127
路牌 11
路礦 1
路祭 2
路程 231
路端電壓 0
路竹 501
路簽 4
路籤 0
路經 47
路線 2436
路線圖 279
路線鬥爭 0
路肩 17
路見 1
路見不平 28
路警 7
路費 23
路軌 8
路透社 114
路途 116
路途遙遠 17
路過 184
路遙知馬力 14
路邊 398
路邊攤 73
路隊 30
路障 37
路面 351
路頭 22
跰 3
跱 8
跲 10
跳 1613
跳上 57
跳上來 5
跳上去 6
跳上跳下 8
跳下 76
跳下來 16
跳下去 27
跳了 84
跳井 0
跳井自殺 0
跳井自盡 0
跳來 4
跳來跳去 18
跳傘 41
跳傘台 0
跳傘塔 0
跳傘運動 0
跳價 0
跳入 58
跳出 204
跳出來 86
跳出去 5
跳出火坑 0
跳到 292
跳加官 4
跳動 204
跳台 22
跳回 53
跳回來 4
跳回去 0
跳在 5
跳好 1
跳往 2
跳得 26
跳得 71
跳房子 3
跳板 54
跳梁小丑 0
跳棋 19
跳槽 90
跳樓 114
跳樓自殺 10
跳樓自盡 0
跳機 61
跳欄 0
跳水 56
跳水台 2
跳汰機 0
跳河 4
跳河自殺 2
跳河自盡 0
跳海 8
跳海自殺 3
跳海自盡 1
跳牆 10
跳牆出去 0
跳班 0
跳球 11
跳的 70
跳票 1908
跳窗 3
跳窗而逃 0
跳箱 9
跳級 22
跳線 77
跳繩 61
跳脫 172
跳腳 38
跳舞 341
跳舞場 2
跳舞場 2
跳舞廳 0
跳舞會 0
跳船 3
跳船落海 0
跳著 38
跳蕩 0
跳號 81
跳蚤 1550
跳行 15
跳表 1
跳讀 2
跳走 2
跳起 59
跳起來 52
跳越 5
跳跳 64
跳踉 1
跳躍 1
跳躍 1478
跳進 49
跳進來 7
跳進去 5
跳過 143
跳過來 6
跳過去 24
跳遠 58
跳針 12
跳錶 1
跳開 30
跳離 37
跳電 0
跳馬 34
跳高 54
跴 22
跺 1768
跺腳 23
跼 350
跼促 0
跼促一隅 0
跼促不安 0
跼蹙 0
跼躅 0
跽 6
跾 17
跿 3
踀 10
踁 0
踂 3
踃 7
踄 72
踅 9
踅 9
踆 17
踇 12
踈 0
踈 0
踈 0
踉 14
踉 14
踉 14
踉踉 0
踉踉蹌蹌 3
踉蹌 33
踉蹌而行 0
踊 33
踋 0
踋 0
踍 1
踏 1055
踏上 236
踏上來 0
踏上去 0
踏下 4
踏下來 0
踏下去 0
踏了 9
踏來 0
踏來踏去 0
踏倒 2
踏入 239
踏出 189
踏出來 1
踏出去 9
踏到 4
踏勘 20
踏實 275
踏尋 5
踏得 5
踏得 8
踏月 5
踏板 167
踏板車 0
踏步 47
踏步走 1
踏爨 1
踏的 8
踏破 5
踏破鐵鞋 2
踏破鐵鞋無覓處 4
踏空 3
踏腳 6
踏腳板 0
踏腳石 20
踏著 69
踏足 12
踏踏 11
踏踏實實 6
踏踏看 0
踏踩 1
踏過 23
踏錯 5
踏雪 12
踏雪尋梅 4
踏青 168
踐 174
踐位 0
踐履 11
踐約 0
踐諾 1
踐踏 118
踐踏在 11
踐踏著 2
踐阼 0
踑 22
踒 3
踓 7
踔 8
踔厲風發 0
踕 7
踖 6
踗 4
踘 3
踙 3
踚 5
踛 47
踜 1
踝 126
踝子骨 0
踝飾 0
踝骨 3
踞 78
踞傲自大 0
踟 9
踟躇 1
踟躇不前 0
踟躕 7
踟躕不前 4
踟躕不進 0
踠 12
踡 92
踡伏 0
踡跼 0
踢 956
踢上 1
踢上來 2
踢上去 1
踢下 21
踢下來 2
踢下去 1
踢不出 0
踢不到 0
踢不起 0
踢中 9
踢了 26
踢人 67
踢來踢去 2
踢倒 11
踢出 45
踢出來 4
踢出去 3
踢到 59
踢去 5
踢向 7
踢回 4
踢回來 1
踢回去 0
踢在 4
踢好 0
踢射 0
踢射入門 0
踢得 8
踢掉 11
踢棉被 0
踢毽子 9
踢球 59
踢球入網 0
踢皮球 28
踢腳 18
踢著 9
踢被 3
踢走 11
踢起 3
踢起來 2
踢踢 12
踢踢球 0
踢躂舞 3
踢躂舞 3
踢進 27
踢進來 0
踢進去 0
踢過 9
踢過來 0
踢過去 0
踢開 17
踢館 1
踣 19
踤 7
踥 19
踦 10
踧 10
踨 0
踩 7355
踩下 15
踩下來 1
踩下去 3
踩住 9
踩光 0
踩入 2
踩到 56
踩在 51
踩壞 1
踩壞了 2
踩得 12
踩扁 0
踩油門 20
踩的 11
踩踏 55
踩踩 3
踩高橇 0
踩高蹺 9
踪 0
踫 53
踮 47
踮著 1
踮起 3
踰 130
踰年 0
踰矩 2
踰越 23
踱 80
踱步 20
踲 0
踲 0
踳 130
踴 90
踴躍 954
踴躍發言 29
踵 70
踵接 0
踵至 0
踶 241
踷 0
踸 254
踹 104
踹踏 0
踺 0
踻 0
踻 0
踼 72
踽 137
踽踽而行 1
踾 92
踿 75
蹀 101
蹀蹀 0
蹀躞 0
蹁 27
蹂 27
蹂躪 65
蹃 0
蹄 832
蹄子 3
蹄形 1
蹄形磁鐵 0
蹄膀 1
蹄鐵 1
蹅 103
蹆 0
蹇 50
蹇滯 0
蹈 318
蹈入 2
蹈常襲故 0
蹈海 0
蹈爨 1
蹉 66
蹉跎 12
蹉跎光陰 1
蹉跎時光 0
蹉跎歲月 1
蹊 70
蹊徑 6
蹊蹺 10
蹋 203
蹌 86
蹌 86
蹍 54
蹎 96
蹏 0
蹐 50
蹑 0
蹑 0
蹓 75
蹓 75
蹔 1
蹕 118
蹖 34
蹗 5
蹘 0
蹙 55
蹙眉 7
蹚 4
蹛 7
蹜 9
蹝 4
蹞 12
蹟 352
蹠 36
蹠骨 8
蹡 3
蹢 7
蹢 7
蹣 38
蹣 38
蹣跚 60
蹣跚 60
蹣跚而行 0
蹤 482
蹤影 130
蹤跡 142
蹥 15
蹦 613
蹦出來 3
蹦跳 6
蹦蹦跳 21
蹦蹦跳跳 10
蹧 15
蹧蹋 2
蹨 0
蹨 0
蹩 5
蹩腳 7
蹪 6
蹫 0
蹬 1000
蹭 12
蹭蹬 1
蹮 0
蹮 0
蹯 0
蹰 0
蹱 0
蹲 4926
蹲上 10
蹲下 38
蹲下來 25
蹲下去 7
蹲了 14
蹲到 3
蹲坐 6
蹲坐在 4
蹲射 0
蹲板 0
蹲著 24
蹲馬步 4
蹲點 2
蹳 4
蹴 63
蹵 0
蹶 22
蹶 22
蹷 0
蹷 0
蹸 3
蹹 0
蹺 72
蹺 72
蹺了 1
蹺來蹺去 0
蹺出 0
蹺去 0
蹺家 0
蹺掉 0
蹺著 2
蹺課 20
蹺課 52
蹺起 4
蹺起來 0
蹺蹊 3
蹺蹺 3
蹺蹺板 9
蹺過 0
蹻 14
蹻 14
蹻 14
蹻勇 0
蹻腳 0
蹼 30
躁 346
躁急 1
躁狂 0
躁進 12
躂 331
躂 331
躃 0
躄 7
躅 175
躆 30
躇 206
躈 2
躉 100
躉售物價 100
躉批 5
躉批買入 0
躉批賣出 0
躊 10
躊躇 46
躊躇不前 6
躊躇滿志 3
躋 14
躋 14
躋升 0
躋身 127
躌 3
躍 1
躍 736
躍動 73
躍動 73
躍升 124
躍升 124
躍居 89
躍居 89
躍然 15
躍然紙上 12
躍然紙上 12
躍起 28
躍躍欲試 71
躍身 16
躍身 16
躍進 47
躍過 10
躍遷 10
躍馬 16
躍馬中原 0
躎 0
躐 41
躐進 0
躑 122
躑躅 10
躒 3
躒 3
躓 529
躓仆 0
躓礙 1
躓礙難行 0
躓頓 0
躔 12
躔次 0
躕 4
躖 12
躗 81
躘 7
躙 0
躚 6
躝 80
躞 70
躟 104
躠 55
躡 21
躡手 4
躡手躡腳 16
躡腳 1
躡腳根 0
躡著 3
躡著腳 0
躡足 9
躢 0
躣 4
躤 0
躥 27
躦 23
躧 0
躨 52
躩 44
躪 8
身 107
身 6777
身上 2637
身不由主 3
身不由己 48
身世 155
身中數彈 0
身中數槍 0
身亡 120
身份 1531
身份証 0
身份證 1046
身價 203
身價百倍 0
身先士卒 13
身兼 122
身兼數職 14
身分 1503
身分証 28
身分證 1010
身受 63
身受其害 40
身在 231
身在福中不知福 40
身外 18
身外之物 16
身子 400
身孕 17
身家 66
身家性命 15
身家清白 4
身居 22
身強力壯 4
身形 121
身影 327
身後 341
身後之事 0
身後蕭條 0
身微言輕 0
身心 2971
身心俱疲 130
身心健康 246
身心愉快 6
身懷六甲 9
身手 144
身手不凡 60
身擔 0
身擔重任 0
身故 159
身敗名裂 9
身教 52
身教重於言教 1
身旁 100
身有同感 1
身材 596
身材矮小 9
身材高大 12
身歷其境 66
身歷聲 13
身段 155
身段優美 1
身段勻稱 0
身法 100
身為 1479
身無長物 2
身穿 146
身穿著 8
身經 9
身經百戰 33
身臨 7
身臨其境 4
身著 118
身處 288
身見 1
身識 1
身負 40
身負重任 5
身軀 181
身輕言微 0
身邊 1142
身量 8
身量 8
身長 140
身門 1
身陷囹圄 3
身首異處 40
身體 5015
身體上 118
身體中 27
身體健康 360
身體力行 73
身體檢查 81
身高 1183
身高計 0
躬 92
躬耕 3
躬行 5
躬親 3
躬身 41
躬逢其盛 6
躭 0
躯 0
躰 0
躱 0
躲 913
躲上 1
躲上來 1
躲上去 0
躲下 0
躲下來 0
躲下去 0
躲不了 2
躲不開 7
躲來 0
躲來躲去 0
躲債 10
躲在 410
躲得 12
躲得不知去向 0
躲懶 3
躲藏 39
躲藏在 11
躲藏著 0
躲貓貓 23
躲起 9
躲起來 60
躲躲 4
躲躲藏藏 7
躲躲閃閃 4
躲逃 0
躲過 63
躲過來 0
躲過去 0
躲避 129
躲避球 44
躲閃 5
躲閃不及 0
躲開 31
躲雨 50
躳 0
躴 0
躶 0
躷 0
躸 0
躹 0
躺 388
躺上 1
躺上來 0
躺上去 0
躺下 47
躺下來 16
躺下去 7
躺來躺去 0
躺個 0
躺出 0
躺出來 0
躺出去 0
躺在 536
躺平 26
躺得 11
躺椅 17
躺的 10
躺直 0
躺臥 28
躺著 95
躺躺 10
躺進 4
躺進來 0
躺進去 0
躺過 1
躺過來 0
躺過去 0
躽 78
躿 0
軀 237
軀幹 65
軀殼 42
軀骸 10
軀體 115
軁 0
軂 5
軃 0
軄 0
軆 0
軇 0
軉 21
車 19764
車 97
車上 527
車下 16
車主 960
車伕 9
車內 212
車刀 6
車前 90
車前燈 0
車城 37
車場 74
車場 74
車墊 3
車外 65
車大砲 0
車夫 14
車子 1161
車守 0
車工 35
車帶 17
車床 303
車底 15
車座 5
車庫 233
車庫裡 4
車廂 246
車廂廣告 4
車廠 237
車弩 0
車房 3
車把 20
車把式 0
車掌 6
車棚 23
車槓 0
車水馬龍 33
車流 76
車渠 3
車燈 61
車牌 138
車牌號碼 30
車班 19
車皮 7
車盤 1
車票 335
車禍 684
車程 191
車種 285
車窗 100
車站 1754
車站裡 4
車箱 19
車篷 0
車篷子 0
車籍 0
車縫 20
車聲 9
車胎 67
車臣 0
車船 40
車蓋 11
車號 0
車行 581
車行 581
車裂 0
車裂 0
車裡 40
車費 54
車資 102
車身 364
車軸 11
車載斗量 0
車輛 3314
車輪 150
車輪下 0
車輪子 1
車輪戰 2
車轍 4
車轍 4
車速 118
車道 348
車鉤 1
車錢 99
車長 246
車門 149
車間 10
車間成本 0
車間經費 0
車隊 832
車頂 68
車頭 144
車頭燈 50
車馬費 42
車駕 67
車體 163
軋 297
軋 297
軋上 3
軋件 0
軋來 0
軋來軋去 1
軋傷 0
軋光 2
軋入 4
軋制 0
軋在 1
軋帳 0
軋平 22
軋戲 1
軋斃 0
軋棉 3
軋機 16
軋死 1
軋片 0
軋碎 5
軋票 1
軋米 0
軋花機 0
軋花邊 0
軋著 0
軋起 0
軋起來 0
軋軋 10
軋軋 10
軋軋聲 1
軋軋聲 1
軋輥 24
軋鋼 90
軋鋼機 1
軋頭寸 0
軌 801
軌域 14
軌範 3
軌距 8
軌跡 440
軌道 802
軍 4129
軍中 404
軍事 2286
軍事化 40
軍事區 0
軍事地理 4
軍事基地 19
軍事委員 0
軍事委員會 18
軍事學校 72
軍事學院 6
軍事家 9
軍事法庭 37
軍事演習 88
軍事行動 139
軍事訓練 20
軍事顧問 9
軍人 660
軍令 32
軍令如山 1
軍令狀 2
軍令部 1
軍備 73
軍備品 0
軍備競賽 20
軍公教 249
軍公教子女 2
軍刀 139
軍分區 2
軍制 5
軍力 99
軍功 70
軍務 30
軍務院 0
軍區 70
軍史館 0
軍售 0
軍器 9
軍國 22
軍國主義 38
軍國大事 2
軍團 341
軍士 64
軍委 54
軍威 6
軍官 467
軍官團 4
軍官學校 81
軍容 6
軍屬 3
軍屯 10
軍工 25
軍帖 2
軍師 73
軍帽 6
軍律 1
軍心 27
軍心渙散 0
軍情 31
軍戶 0
軍援 4
軍政 54
軍政人員 0
軍政大學 0
軍政府 13
軍政時期 1
軍政權 0
軍政當局 0
軍方 715
軍旅 59
軍旗 15
軍書 0
軍服 28
軍校 127
軍校生 5
軍械 8
軍械 8
軍械士 0
軍械庫 4
軍樂 8
軍樂隊 16
軍機 100
軍機大臣 3
軍機處 0
軍權 11
軍歌 27
軍毯 2
軍民 130
軍民一家 0
軍民一家 0
軍民合作 4
軍民聯防 0
軍油 1
軍法 127
軍法官 35
軍法審判 8
軍港 12
軍火 70
軍火商 9
軍火庫 6
軍火販子 3
軍營 39
軍營區 1
軍犬 5
軍用 134
軍用品 11
軍用機場 7
軍用機場 7
軍用物資 4
軍用犬 0
軍用電話 0
軍界 5
軍略 17
軍禮 18
軍種 42
軍管 39
軍管區 24
軍籍 4
軍糧 13
軍紀 11
軍統 11
軍聞社 11
軍職 77
軍職人員 20
軍艦 238
軍艦鳥 3
軍荼利 1
軍荼利夜叉明王 1
軍荼利菩薩 1
軍號 2
軍衣 0
軍裝 23
軍規 7
軍訓 738
軍訓教官 90
軍訓課 10
軍諮府 0
軍警 216
軍費 23
軍車 16
軍部 15
軍郵 7
軍醫 52
軍醫大學 5
軍長 2
軍閥 39
軍閥主義 0
軍閥作風 0
軍閥割據 7
軍隊 651
軍階 3
軍需 55
軍需品 10
軍需補給 0
軍靴 10
軍餉 2
軍馬 97
軍鴿 0
軎 0
軏 64
軏 64
軐 0
軑 28
軒 2272
軒敞 0
軒昂 11
軒朗 0
軒然大波 29
軒然大波 29
軒聲 0
軒輊 9
軒轅 59
軒轅教 7
軒轅氏 1
軒轅黃帝 0
軓 20
軔 69
軖 0
軗 292
軘 18
軙 0
軙 0
軚 0
軛 64
軜 8
軝 32
軞 203
軟 3996
軟件 330
軟兜 3
軟化 182
軟化政策 0
軟口蓋 1
軟和 0
軟式 172
軟式磁碟 9
軟式磁碟機 48
軟弱 244
軟弱可欺 0
軟弱性 1
軟弱無能 5
軟心腸 1
軟性 123
軟木 74
軟木塞 13
軟木樹 0
軟木質 0
軟梯 0
軟毛 7
軟水 45
軟片 213
軟玉 17
軟瘤 0
軟皮 3
軟皮動物 0
軟硬 1506
軟硬不喫 0
軟硬兼施 60
軟碟 499
軟禁 21
軟管 103
軟糖 116
軟綿綿 12
軟脂酸 2
軟腳 22
軟膏 53
軟語 10
軟質 62
軟質性 0
軟軟 55
軟軟弱弱 1
軟軟鬆鬆 2
軟釘子 2
軟鋼 9
軟鐵 5
軟風 2
軟骨 190
軟骨病 3
軟骨病 3
軟骨頭 0
軟骨魚 0
軟骨魚類 4
軟體 108454
軟體動物 80
軟體界 0
軟體設計 1684
軟體部 16
軟齶 0
軠 24
軡 12
軡 12
軤 0
軥 3
軦 4
軧 4
軨 5
軩 2
軫 23
軫念 0
軬 4
軭 0
軮 3
軯 5
軰 0
軱 2
軱轆 0
軲 0
軲 0
軳 0
軴 3
軵 1
軵 1
軶 2
軷 8
軸 16
軸 1635
軸功率 0
軸套 5
軸子 0
軸孔 5
軸對稱 12
軸心 93
軸心國 8
軸承 338
軸承合金 0
軸流泵 0
軸瓦 0
軸箱 13
軸系 7
軸線 78
軸襯 2
軸距 204
軸面 2
軹 10
軺 11
軻 36
軼 211
軼事 116
軼聞 23
軼詩 0
軽 0
軾 29
軿 24
輀 8
輁 4
輂 3
較 1
較 20028
較厚 100
較薄 100
較薄 100
較 2114
較好 74
較之 85
較低 745
較佳 594
較力 5
較勁 153
較多 630
較大 1271
較小 513
較少 325
較快 306
較早 155
較晚 96
較為 10
較為 100
較熟 12
較瘦 6
較緊 7
較近 85
較量 70
較量法 0
較高 1442
輄 0
輅 25
輆 7
輇 37
輈 10
載 151
載 36483
載來 16
載來載去 0
載入 600
載入器 2
載出 1
載出來 0
載出去 1
載到 81
載客 61
載客率 15
載往 8
載於 222
載明 389
載有 142
載歌載舞 27
載波 96
載波 96
載波通信 0
載波通信 0
載流子 1
載湉 0
載滿 23
載滿了 4
載灃 0
載熱劑 0
載玻片 9
載舟 16
載荷 5
載記 3
載貨 36
載貨率 0
載貨量 1
載起 6
載起來 0
載運 140
載運量 8
載道 3
載重 116
載重汽車 0
載重量 16
載頻 1
載體 50
輊 32
輋 10
輍 47
輎 41
輏 0
輏 0
輐 37
輑 3
輒 531
輓 869
輓歌 41
輓聯 2
輓詞 0
輓詩 3
輔 4033
輔仁 120
輔仁大學 1779
輔佐 70
輔具 100
輔助 5225
輔助性 150
輔助材料 4
輔助生產 3
輔大 665
輔導 13746
輔導員 381
輔導室 790
輔導會 100
輔導組 427
輔導處 213
輔導課 42
輔導院 1
輔幣 1
輔弼 7
輔政 8
輔系 308
輔翼 0
輔育院 28
輔車 0
輔選 1190
輔音 0
輕 10930
輕佻 8
輕便 177
輕俏 4
輕信 16
輕傷 44
輕傷者 3
輕兵 7
輕利 1
輕利重義 0
輕功 79
輕取 15
輕叩 7
輕吹 7
輕吻 23
輕型 108
輕型機車 5
輕壓 19
輕如羽毛 0
輕妙 2
輕安 1
輕巡洋艦 1
輕工 24
輕工業 79
輕工業品 0
輕巧 216
輕巧方便 9
輕度 293
輕彈 10
輕得 136
輕微 523
輕心 5
輕快 145
輕忽 168
輕慢 14
輕拉 4
輕拍 42
輕捷 2
輕揉 11
輕描淡寫 40
輕撫 55
輕放 40
輕敲 19
輕敵 23
輕文重武 0
輕於 6
輕於鴻毛 2
輕易 1353
輕柔 146
輕機槍 2
輕歌曼舞 0
輕武器 3
輕氣 4
輕氣球 0
輕油 19
輕油裂解 24
輕油裂解工廠 0
輕浮 34
輕減 0
輕爽 3
輕狂 68
輕率 44
輕生 26
輕的 135
輕盈 83
輕矛 0
輕石 4
輕窕 0
輕紗 4
輕罪 14
輕罰 0
輕而易取 0
輕而易舉 95
輕聲 173
輕脆 2
輕脆動聽 0
輕脆悅耳 0
輕舉妄動 33
輕舟 37
輕色重利 0
輕色重義 0
輕蔑 29
輕蔑性 0
輕薄 163
輕薄少年 0
輕裘 1
輕裝 37
輕裝師 2
輕視 99
輕言 82
輕諾寡信 0
輕財好義 0
輕踏 0
輕車從簡 0
輕車簡從 2
輕輕 826
輕輕的 226
輕輕地 226
輕輕地 1
輕輕鬆鬆 268
輕重 187
輕重緩急 15
輕重量級 7
輕量級 17
輕金屬 33
輕閒 0
輕靈 17
輕音 7
輕音樂 29
輕風 17
輕風徐來 0
輕飄 10
輕飄飄 23
輕騎 26
輕騎兵 8
輕骨頭 0
輕鬆 4007
輕鬆愉快 96
輕鬆自在 41
輖 0
輗 11
輘 13
輙 0
輚 5
輛 1375
輜 27
輜重 10
輝 5463
輝光 7
輝映 73
輝煌 381
輝煌燦爛 7
輝耀 11
輝銀 1
輝銀礦 0
輝銅 2
輝銅礦 1
輝銻礦 0
輞 15
輟 59
輟學 177
輟耕 0
輠 3
輡 0
輢 0
輣 2
輤 6
輥 67
輥子 10
輥軋 1
輥齒輪 1
輦 38
輧 0
輨 0
輨 0
輩 237
輩份 170
輩出 29
輩分 10
輩子 196
輩行 0
輪 9541
輪上 23
輪上來 1
輪上去 0
輪下 6
輪下來 0
輪下去 0
輪休 26
輪休制 1
輪住 1
輪作 34
輪作法 0
輪來 6
輪來輪去 0
輪值 147
輪值法 0
輪具 10
輪到 172
輪唱 11
輪回 2
輪圈 158
輪奐 0
輪姦 29
輪子 133
輪完 2
輪對 1
輪帶 1
輪底 10
輪底下 5
輪廓 278
輪廓圖 3
輪式拖拉機 0
輪戰 0
輪提 1
輪換 12
輪換成 0
輪換排列 0
輪換次序 0
輪播 21
輪替 90
輪有 7
輪架 9
輪栽 0
輪栽法 0
輪椅 245
輪機 318
輪機員 15
輪機室 5
輪機艙 0
輪機長 28
輪法 54
輪流 333
輪流制 1
輪流法 0
輪演 0
輪為 5
輪狀 65
輪班 45
輪班制 5
輪班工作 0
輪班看守 1
輪生葉 1
輪番 39
輪番上陣 8
輪發 7
輪盤 43
輪種 18
輪種法 0
輪空 12
輪穿 2
輪給 0
輪耕 2
輪胎 1629
輪船 199
輪著 5
輪蓋 1
輪蟲 0
輪訓 8
輪試 1
輪調 129
輪調制 1
輪調法 0
輪起來 0
輪踢 0
輪軸 37
輪轂 7
輪轉 46
輪轉機 9
輪迴 300
輪送 5
輪過 0
輪過來 0
輪過去 0
輪鞋 98
輪養 1
輪騎 0
輪點 5
輪齒 5
輫 0
輬 1
輭 0
輮 95
輯 2690
輯印 5
輯印成書 0
輯書 1
輯要 138
輯錄 11
輯錄出來 0
輰 0
輱 0
輲 58
輳 5
輴 65
輵 85
輶 73
輷 134
輸 27994
輸不起 23
輸了 262
輸人 24
輸人不輸陣 12
輸來 2
輸來輸去 0
輸光 6
輸入 20409
輸入區 13
輸入品 8
輸入法 1239
輸入阻抗 39
輸出 3998
輸出來 0
輸出功率 95
輸出區 2
輸出去 0
輸出品 5
輸出管 1
輸出表 0
輸出變壓 0
輸出變壓器 11
輸出量 33
輸出阻抗 14
輸到 5
輸卵 14
輸卵管 118
輸去 1
輸嘴 0
輸多 0
輸多贏少 0
輸完 3
輸定 6
輸家 48
輸導 7
輸少贏多 0
輸尿管 105
輸往 176
輸得 92
輸成 2
輸掉 38
輸氣管 18
輸水 28
輸油 28
輸油泵 0
輸油管 13
輸球 19
輸的 92
輸精 0
輸精管 24
輸糧 2
輸給 157
輸膽管 1
輸著 0
輸血 849
輸血機 0
輸血管 0
輸誠 15
輸贏 84
輸贏不計 0
輸起 0
輸起來 0
輸輸 2
輸送 637
輸送到 15
輸送帶 110
輸送架 1
輸送管 9
輸進 4
輸進來 0
輸進去 1
輸運 17
輸運到 0
輸過 6
輸過來 0
輸過去 0
輸錢 9
輸電 117
輸點 1
輹 51
輻 314
輻射 2813
輻射劑 3
輻射塵 8
輻射性 8
輻射熱 15
輻射狀 14
輻射線 49
輻射能 23
輻射計 4
輻射量 47
輻射體 7
輻湊 4
輻輳 18
輼 0
輽 0
輾 88
輾 88
輾平 1
輾成 1
輾斃 4
輾米 0
輾米廠 0
輾轉 121
輾轉反側 8
輾轉流傳 0
輾轉相告 0
輾過 10
輾鐵 0
輿 368
輿圖 18
輿情 112
輿論 259
輿論界 26
輿論譁然 0
轀 56
轁 0
轁 0
轂 59
轂 59
轂擊肩摩 0
轂轆 0
轂轆 0
轃 51
轄 713
轄區 844
轄地 7
轄境 16
轄治 2
轅 52
轅門 8
轆 23
轆轆 14
轆轤 19
轇 9
轈 3
轉 1335
轉 13358
轉上 35
轉上來 17
轉上去 17
轉下 7
轉下來 1
轉下去 1
轉世 167
轉乘 490
轉了 95
轉些 0
轉交 100
轉任 361
轉來 47
轉來轉去 29
轉信 0
轉個 25
轉個彎 9
轉借 33
轉入 470
轉入地下 4
轉出 121
轉出來 12
轉出去 12
轉列 14
轉到 329
轉動 454
轉動慣量 14
轉動著 9
轉動軸 5
轉勝為敗 0
轉化 529
轉危為安 10
轉去 11
轉口 157
轉口港 3
轉口站 2
轉口貨 7
轉口貿易 39
轉台 39
轉向 637
轉向器 0
轉向架 15
轉向離合 0
轉向離合器 0
轉告 96
轉售 116
轉售他人 4
轉回 81
轉圈 7
轉圜 39
轉圜之地 0
轉圜餘地 5
轉在 9
轉型 1145
轉場 40
轉場 4
轉嫁 39
轉嫁到 18
轉嫁給 15
轉子 88
轉學 375
轉學生 259
轉守為攻 7
轉完 40
轉寄 114
轉寰 6
轉小 20
轉差率 0
轉帳 666
轉彎 178
轉彎抹角 3
轉形 21
轉形期 2
轉往 206
轉得 92
轉徙 2
轉念 18
轉念之間 0
轉成 490
轉戰 187
轉手 82
轉投資 669
轉折 246
轉折點 42
轉捩 18
轉捩期 3
轉捩點 124
轉捩點 124
轉接 333
轉換 3210
轉換器 278
轉換方向 2
轉換期 6
轉換法 5
轉換開關 1
轉播 457
轉播台 225
轉播器 1
轉播站 225
轉攻為守 0
轉敗為勝 4
轉數 52
轉文 6
轉晴 10
轉校 26
轉業 101
轉機 411
轉檯 22
轉正 11
轉法 3
轉注 21
轉為 538
轉爐 15
轉生 33
轉病 0
轉發 78
轉的 92
轉盤 73
轉看 5
轉眼 75
轉眼之間 20
轉眼間 66
轉瞬 21
轉瞬間 20
轉磨 1
轉禍為福 2
轉租 19
轉租給 9
轉移 1208
轉移到 600
轉移陣地 12
轉站 5
轉筋 1
轉系 306
轉結 1
轉置 17
轉而 222
轉職 66
轉臉 6
轉角 133
轉角處 24
轉託 3
轉診 1
轉調 47
轉變 774
轉變到 12
轉變成 198
轉變為 196
轉讓 651
轉賣 47
轉賬 1
轉贈 21
轉贈給 2
轉起 1
轉起來 2
轉身 340
轉身法 0
轉車 22
轉車站 1
轉軸 48
轉載 3580
轉載稿 1
轉輪手槍 1
轉輾 1
轉轉 67
轉轉看 30
轉轍員 0
轉述 89
轉送 223
轉速 416
轉進 125
轉運 416
轉運港 200
轉運站 200
轉過 204
轉過來 100
轉過去 100
轉道 0
轉達 106
轉遞 27
轉錄 408
轉開去 0
轉音 60
轉韻 1
轉頭 137
轉頭就走 3
轉風易俗 0
轉體 49
轊 0
轋 1
轍 81
轍 81
轍蹟 0
轎 307
轎伕 3
轎夫 25
轎子 47
轎椅 0
轎車 756
轎門 1
轏 0
轐 3
轑 0
轑 0
轒 4
轓 0
轔 17
轔轔 1
轕 9
轖 4
轗 4
轘 9
轙 3
轚 9
轛 2
轜 0
轝 3
轞 4
轞 4
轟 325
轟上 0
轟上來 0
轟上去 0
轟下 4
轟下來 0
轟下去 0
轟亂 0
轟了 17
轟來轟去 1
轟倒 0
轟傳 1
轟出 46
轟出來 1
轟出去 3
轟到 3
轟動 237
轟動一時 26
轟得 4
轟成 2
轟掉 8
轟擊 15
轟殺 3
轟炸 214
轟炸機 55
轟然 22
轟的一聲 2
轟聲 4
轟聲四起 0
轟走 0
轟起 2
轟起來 0
轟趕 0
轟轟 23
轟轟烈烈 54
轟轟聲 0
轟進 0
轟進來 0
轟進去 0
轟過 1
轟開 0
轟隆 34
轟隆聲 2
轟隆隆 9
轟鳴 3
轟鳴聲 2
轠 7
轡 80
轢 17
轣 2
轣轆 0
轤 5
轥 0
辛 1755
辛丑 11
辛丑和約 0
辛亥 236
辛亥革命 32
辛勞 238
辛勤 144
辛勤工作 24
辛味 2
辛棄疾 17
辛烷值 14
辛苦 1439
辛苦工作 25
辛辛 28
辛辛苦苦 60
辛辛那提 51
辛辣 71
辛辣辣 0
辛迪加 0
辛酉 4
辛酸 104
辛酸淚 2
辛醜 0
辜 3111
辜負 137
辝 0
辞 0
辟 116
辟 116
辟支佛 1
辟邪 23
辠 0
辡 0
辢 0
辣 2047
辣味 32
辣呼呼 0
辣妹 0
辣子 9
辣得 92
辣手 14
辣手摧花 4
辣椒 203
辣椒粉 1
辣椒醬 14
辣油 12
辣瓣兒醬 0
辣瓣兒醬 0
辣的 92
辣醬 41
辣醬油 1
辤 0
辥 0
辦 8039
辦不到 25
辦不完 0
辦不成 4
辦事 367
辦事不牢 2
辦事不犖 0
辦事員 88
辦事房 0
辦事處 1691
辦伙 0
辦公 1916
辦公室 4633
辦公廳 54
辦公時間 108
辦公桌 135
辦公檯 0
辦公處 120
辦公費 12
辦到 100
辦喜事 11
辦報 31
辦妥 245
辦學 431
辦學不力 1
辦後事 2
辦得 240
辦得到 13
辦得好 6
辦案 181
辦法 13649
辦理 20830
辦的 239
辦結 2
辦罪 0
辦處 14
辦貨 1
辧 0
辨 734
辨出 10
辨別 266
辨別出來 3
辨別力 1
辨別是非 4
辨向 11
辨士 1
辨明 39
辨析 14
辨正 11
辨白 0
辨色 19
辨証法 1
辨認 289
辨認出 18
辨認出來 5
辨證法 2
辨識 1438
辨音 2
辬 0
辭 3822
辭不達意 2
辭世 39
辭令 7
辭修 70
辭典 1251
辭典編纂法 0
辭典編纂者 0
辭別 13
辭去 210
辭句 12
辭呈 94
辭官 32
辭庫 16
辭彙 182
辭意 40
辭掉 28
辭歲 2
辭海 15
辭源 2
辭章 5
辭義 0
辭職 603
辭色 7
辭藻 5
辭行 8
辭謝 3
辭讓 0
辭賦 13
辭退 27
辮 52
辮兒 0
辮兒 0
辮子 42
辯 594
辯士 14
辯才 13
辯才無礙 8
辯明 7
辯正 1
辯白 17
辯稱 90
辯解 66
辯証 10
辯証唯物主義 0
辯說 3
辯論 1094
辯論會 311
辯論賽 28
辯證 116
辯證法 7
辯護 201
辯護人 27
辯護士 2
辯護律師 51
辯護權 2
辯難 0
辯駁 45
辰 1369
辰光 9
辰牌 5
辰砂 3
辱 162
辱 162
辱命 3
辱命 3
辱國 2
辱國 2
辱國喪權 0
辱國喪權 0
辱沒 3
辱沒 3
辱罵 57
辱罵 57
農 7610
農中 4
農事 99
農人 61
農作 91
農作曲 0
農作機 0
農作物 337
農倉 2
農具 217
農副產品 9
農務 24
農化 82
農化系 54
農友 204
農園 304
農地 1285
農地重劃 108
農場 154
農場 1546
農場地 2
農場地 2
農墾 6
農夫 192
農奴 2
農奴主 0
農奴制 1
農委會 0
農婦 8
農學 197
農學院 191
農官 0
農家 204
農家品種 0
農家女 1
農家子弟 14
農家肥料 0
農專 25
農工 677
農工商 8
農工職業學校 90
農復會 25
農忙 9
農忙時節 0
農戶 78
農技 37
農技團 41
農政 150
農救會 0
農時 4
農曆 595
農曆年 36
農會 1619
農村 887
農村人民公社 1
農村公社 0
農村化 0
農林 594
農林廳 1279
農林廳長 10
農林牧副漁 0
農校 27
農業 6516
農業中學 0
農業化 76
農業區 152
農業合作 53
農業合作化 1
農業品 4
農業國 1
農業大學 59
農業學大寨 0
農業學校 12
農業專科學校 81
農業局 106
農業工人 0
農業技術 108
農業機器 2
農業機械 188
農業機械化 23
農業生產 225
農業生產合作社 4
農業社 32
農業社會 52
農業稅 0
農業總產值 4
農業職業學校 47
農業貸款 4
農業資本 2
農業資本家 0
農業集體化 1
農機 316
農機系 31
農民 1237
農民協會 9
農民戰爭 0
農民起義 1
農民銀行 50
農民階級 0
農民革命 0
農活 1
農渠 0
農牧 308
農牧區 0
農牧業 56
農產 442
農產品 820
農產品收購 1
農產物 9
農產量 4
農田 698
農田基本建設 0
農田小氣候 0
農田水利 68
農畜 97
農神 3
農科 43
農經 72
農經系 32
農耕 123
農耕機 4
農耕隊 11
農胞 1
農舍 103
農莊 115
農藝 135
農藝師 0
農藝系 25
農藝館 0
農藥 1262
農藥商 0
農試所 93
農諺 1
農鄉 2
農閑 0
農閒 9
農隙 0
辳 0
辴 4
辵 0
辶 0
辸 0
边 0
达 0
达 0
达 0
辿 43
迀 0
迁 0
迂 319
迂久 0
迂儒 0
迂回 2
迂徊 0
迂徊戰術 0
迂徊漸進 0
迂曲 2
迂直 0
迂緩 0
迂腐 21
迂腐之見 0
迂路 0
迂迴 74
迂迴戰術 2
迂迴曲折 7
迂闊 0
迂陋 0
迄 2217
迄今 1003
迄今已 130
迄今已有 55
迄未 70
迄未見效 0
迄無 20
迄無音訊 0
迅 1049
迅即 36
迅即出發 0
迅即處理 0
迅即處理 0
迅捷 30
迅流 0
迅猛 5
迅猛龍 43
迅疾 7
迅速 2564
迅雷 40
迅雷不及 1
迅雷不及掩耳 37
迅風 2
迆 227
迆邐 1
过 0
过 0
过 0
迉 57
迊 0
迋 377
迌 0
迍 131
迍邅 0
迎 201
迎 2018
迎上 35
迎上來 3
迎上去 0
迎人 7
迎候 2
迎出 2
迎刃而解 55
迎合 197
迎合人心 1
迎向 268
迎向前去 5
迎娶 33
迎客 20
迎戰 89
迎接 1028
迎擊 20
迎敵 13
迎新 359
迎新晚會 17
迎新會 11
迎新送舊 16
迎春 70
迎春曲 2
迎春花 12
迎神 10
迎神大會 0
迎神賽會 8
迎著 47
迎親 17
迎角 0
迎賓 86
迎賓曲 0
迎賓館 11
迎起 0
迎送 6
迎面 76
迎面而來 48
迎頭 11
迎頭棒喝 0
迎頭痛擊 81
迎頭趕上 40
迎風 82
迎風招展 2
迎風而去 0
迏 0
迏 0
迏 0
运 0
运 0
近 9115
近世 68
近乎 174
近了 110
近些 8
近些年來 7
近人 17
近代 1025
近代史 192
近代思想 38
近似 269
近似值 13
近似商 0
近似律 0
近似於 41
近似計算 0
近作 17
近來 1461
近來可好 41
近便 6
近利 29
近前 18
近古 7
近古時代 0
近因 51
近在 11
近在咫尺 25
近在眼前 19
近地點 4
近墨則黑 0
近墨者黑 7
近廟欺神 3
近年 1131
近年以來 5
近年來 2411
近幾年來 228
近得 8
近悅遠來 2
近情 2
近情理 0
近憂 2
近戰 6
近支 1
近於 42
近旁 14
近日 1398
近日以來 8
近日來 100
近日裡 0
近日點 29
近有 24
近期 2109
近期內 305
近朱者赤 6
近東 27
近東國家 0
近東地區 0
近歲 0
近水樓台 7
近水樓台先得月 1
近水樓臺 1
近況 329
近海 162
近海漁業 110
近海航運 0
近理 1
近畿 37
近畿 37
近百 235
近百年來 43
近程 87
近程目標 36
近臣 5
近處 32
近視 467
近視眼 27
近視眼鏡 7
近親 62
近親繁殖 5
近親通婚 3
近距 13
近距離 101
近距離射擊 3
近路 3
近道 3
近郊 111
近郊地區 2
近鄉情怯 2
近鄉情怯 2
近鄰 25
近體詩 10
迒 41
迓 104
返 1073
返台 290
返回 4011
返國 644
返家 149
返工 1
返復 0
返德 1
返抵 42
返校 128
返校日 13
返法 0
返法 0
返潮 0
返照 8
返璞歸真 5
返程 6
返美 17
返老還童 20
返航 15
返英 3
返鄉 148
返青 1
返鹽 0
迕 197
迖 110
迗 158
还 0
还 0
还 0
这 0
这 0
迠 629
迡 716
迢 21
迢迢 37
迢迢千里 1
迢遙 13
迢遞 3
迢遠 0
迣 12
迤 401
迤 401
迤邐 23
迥 150
迥別 0
迥然 7
迥然不同 30
迥然有別 1
迥然迴異 0
迥異 101
迥若 0
迥若兩人 0
迥遠 0
迦 430
迦南 173
迦太基 4
迦樓羅 1
迦納 49
迦葉尊者 1
迧 0
迧 0
迨 796
迩 0
迪 3981
迪化 25
迪化街 164
迪士尼 717
迪斯可 43
迪斯耐 45
迫 633
迫不期待 50
迫不 38
迫不得已 13
迫令 1
迫使 330
迫促 2
迫切 327
迫切需要 109
迫在 7
迫在眉睫 34
迫害 316
迫擊炮 2
迫擊砲 3
迫擊砲彈 0
迫於 57
迫於形勢 1
迫近 48
迫降 85
迬 0
迭 1089
迭代 0
迭有 30
迭有佳績 0
迭有斬獲 1
迭有發現 1
迭次 6
迭起 34
迮 873
迯 0
述 2902
述作 1
述異記 0
述而不作 0
述而不作 0
述職 16
述評 18
述詞 10
述語 7
述說 183
迱 0
迱 0
迴 1028
迴光返照 4
迴向 64
迴圈 193
迴廊 98
迴形夾 0
迴文 4
迴旋 173
迴旋器 2
迴旋曲 13
迴流 47
迴環 2
迴盪 76
迴紋針 35
迴繞 13
迴腸 25
迴腸盪氣 3
迴蕩 7
迴誦 0
迴路 436
迴轉 259
迴遞性 0
迴避 324
迴響 2933
迴風 5
迴首 0
迵 676
迶 256
迷 4310
迷上 90
迷亂 31
迷了 34
迷人 566
迷住 14
迷你 832
迷你型 98
迷你裙 33
迷你車 15
迷來迷去 0
迷信 277
迷倒 26
迷到 6
迷夢 3
迷失 308
迷宮 280
迷岸 0
迷幻藥 14
迷彩 1
迷彩服 1
迷彩褲 1
迷思 326
迷惑 289
迷惑不解 5
迷惘 139
迷戀 136
迷戀著 2
迷死 32
迷津 86
迷湯 2
迷漫 19
迷濛 40
迷瞪 0
迷糊 118
迷罔 2
迷航 61
迷茫 31
迷著 4
迷蒙 5
迷藏 15
迷藥 14
迷走 13
迷走神經 25
迷起 0
迷起來 1
迷路 248
迷迭香 1
迷迷 72
迷迷濛濛 2
迷迷糊糊 23
迷迷茫茫 0
迷途 79
迷途知返 4
迷陣 14
迷離 74
迷離恍惚 0
迷霧 93
迷魂 23
迷魂湯 0
迷魂藥 1
迷魂陣 3
迸 55
迸 55
迸出 43
迸出來 1
迸出去 0
迸流 2
迸淚 0
迸濺 0
迸發 9
迸發出 10
迸裂 19
迸開 1
迹 0
迺 178
迻 78
迻譯 3
迼 44
追 2682
追上 94
追上來 8
追上去 9
追下 2
追下來 0
追下去 3
追不上 24
追亡逐北 0
追人 4
追來 29
追來追去 2
追入 1
追兵 28
追到 89
追剿 2
追加 299
追加預算 106
追去 17
追向 2
追問 134
追問到底 0
追回 86
追回來 3
追回去 0
追封 3
追尋 393
追尋著 4
追念 47
追思 91
追思彌撒 1
追思禮拜 7
追悔 6
追悔莫及 1
追悼 53
追悼會 7
追想 13
追憶 92
追憶往事 2
追懷 3
追戀 0
追打 16
追捕 78
追撞 53
追擊 131
追敘 1
追敘法 0
追敵 2
追本溯源 17
追本窮源 0
追查 399
追查責任 1
追根究底 39
追根究柢 0
追殺 224
追求 2829
追求到 7
追求名利 2
追求幸福 14
追源溯始 0
追溯 203
追溯到 59
追究 272
追究來 0
追究責任 27
追索 52
追緝 0
追繳 156
追肥 5
追補 12
追討 47
追訴 55
追訴時效 1
追訴權 42
追認 42
追贈 2
追贓 0
追趕 130
追趕上 1
追趕而來 0
追趕著 5
追蹤 1691
追述 10
追逐 338
追逐名利 3
追進 5
追進來 0
追進去 2
追過 22
追過來 5
追過去 8
追遠 2
追隨 155
追隨者 19
追隨著 8
追音 94
迾 60
迿 8
退 2255
退一 17
退一步 48
退一步 48
退一步海闊天空 8
退一步海闊天空 8
退上 11
退下 21
退下來 7
退下去 5
退件 0
退伍 423
退伍令 60
退伍兵 1
退伍軍人 251
退休 2509
退休年齡 19
退休金 566
退伙 1
退位 19
退來 2
退保 65
退入 18
退兵 13
退兵之計 0
退冰 1
退出 753
退出來 2
退出去 3
退到 46
退化 304
退卻 37
退去 66
退向 0
退回 367
退回來 9
退回去 7
退堂 10
退場 60
退場 60
退場式 0
退場式 0
退婚 1
退學 294
退學生 15
退守 14
退居 54
退席 8
退庭 4
退役 106
退役軍人 7
退後 49
退得 5
退得 28
退房 42
退拒 0
退掉 12
退換 52
退敗 0
退敵 4
退敵之計 0
退書 36
退朝 1
退款 74
退步 80
退水 2
退減 4
退漿 162
退潮 91
退火 63
退為 3
退熱 10
退燒 87
退燒藥 25
退現 3
退瓶 16
退瓶費 1
退的 27
退票 267
退租 22
退稅 318
退稿 74
退給 8
退縮 152
退縮不前 1
退而 16
退而求其次 12
退職 239
退職金 23
退股 15
退色 45
退著 2
退親 0
退訂 3
退訓 27
退讓 53
退貨 209
退費 243
退走 12
退路 37
退車 2
退輔會 203
退過 1
退過來 0
退過去 0
退選 158
退避 9
退避三舍 10
退還 449
退還給 18
退錢 45
退開 14
退除 18
退除役 66
退除役官兵 84
退隱 72
退離 5
退黨 35
送 11068
送上 150
送上來 7
送上去 1
送下 16
送下來 0
送下去 0
送了 106
送些 6
送交 367
送交到 0
送交給 7
送人 151
送人情 1
送件 0
送來 197
送信 71
送信人 5
送修 138
送個 26
送入 106
送出 846
送出來 2
送出去 25
送別 40
送到 636
送去 90
送命 9
送回 211
送報 25
送報人 0
送報員 0
送報生 5
送報童 0
送子觀音 1
送客 18
送客出門 0
送審 0
送往迎來 3
送掉 6
送機 41
送款 3
送歲 1
送死 29
送殯 3
送氣 3
送洗 21
送灶神 1
送神 10
送禮 333
送禮大方 3
送秋波 3
送秋波 3
送糞 0
送終 6
送終追遠 0
送給 862
送經 37
送老 6
送與 21
送舊 55
送舊迎新 7
送花 178
送葬 13
送葬隊 0
送行 41
送行人 1
送親 4
送話 1
送話器 2
送貨 237
送貨上門 0
送貨來 0
送貨到 37
送貨去 0
送貨員 13
送走 64
送起 1
送起來 0
送送 10
送送禮 0
送進 96
送進來 2
送進去 3
送過 13
送過來 10
送過去 13
送達 572
送還 26
送還到 0
送還給 6
送醫 209
送醫急救 35
送錢 18
送風 45
送風機 48
适 92
逃 627
逃上 2
逃上來 0
逃上去 0
逃下 5
逃下來 0
逃下去 1
逃不了 18
逃不出 23
逃之夭夭 18
逃了 19
逃亡 178
逃來逃去 0
逃債 1
逃光 1
逃入 10
逃兵 42
逃出 139
逃出來 6
逃出去 9
逃出虎口 0
逃到 59
逃匿 2
逃去 14
逃命 73
逃命要緊 0
逃回 35
逃奔 1
逃奔自由 0
逃妻 8
逃婚 5
逃學 34
逃家 46
逃役 3
逃往 19
逃得 32
逃得 32
逃慌 0
逃掉 3
逃散 2
逃歸 1
逃漏 46
逃漏稅 80
逃犯 27
逃生 279
逃生梯 9
逃稅 50
逃竄 20
逃罪 3
逃脫 93
逃脫出 0
逃荒 0
逃課 1
逃課學生 0
逃走 130
逃跑 99
逃路 2
逃進 3
逃進來 0
逃進去 0
逃逸 88
逃遁 7
逃避 513
逃避現實 28
逃開 32
逃離 173
逃離現場 14
逃離現場 14
逃難 67
逃難所 0
逄 241
逅 37
逅 37
逆 752
逆事 0
逆來順受 19
逆倫 20
逆光 29
逆勢 144
逆勢上漲 18
逆勢下跌 0
逆反應 3
逆向 144
逆向行駛 12
逆命 1
逆境 111
逆壓 1
逆天則亡 0
逆子 3
逆定理 0
逆差 129
逆我者亡 3
逆數 0
逆料 10
逆施 0
逆水 2
逆水行舟 3
逆流 167
逆流而上 10
逆流而行 1
逆浪 2
逆溫 3
逆溫層 10
逆溯 1
逆滲透 0
逆知 0
逆耳 6
逆臣 0
逆著 6
逆號 0
逆行 42
逆行而進 0
逆賊 3
逆跡昭彰 0
逆轉 189
逆轉而進 0
逆運算 0
逆風 18
逆風而上 3
逆風而行 2
逆黨 2
逇 0
逈 0
选 0
逋 143
逋逃 1
逌 72
逍 175
逍遙 805
逍遙法外 17
逍遙自在 15
逎 0
逎 0
透 1559
透一口氣 1
透一口氣 1
透上 0
透上來 0
透上去 0
透下 2
透下來 0
透下去 0
透不過 8
透不過氣來 9
透了 71
透亮 6
透光 67
透光伐 0
透光鏡 0
透入 25
透出 37
透出來 3
透出去 0
透切 1
透味 0
透天 360
透射 37
透平 2
透徹 62
透心 12
透心涼 3
透支 65
透明 1316
透明度 68
透明漆 7
透明紙 3
透明體 2
透析 1332
透析法 3
透氣 116
透氣性 24
透水 30
透水性 17
透汗 2
透漏 7
透漏出 0
透漏過來 0
透澈 15
透熱 1
透熱性 0
透花紋 0
透著 48
透視 639
透視力 3
透視圖 88
透視度 26
透視鏡 2
透透 291
透透氣 13
透透風 0
透進 10
透進來 2
透進去 0
透過 10709
透過來 0
透過去 1
透鏡 172
透闢 3
透露 845
透露出 115
透露消息 0
透頂 13
透風 9
透骨 7
逐 947
逐一 368
逐個 12
逐出 51
逐利 4
逐句 7
逐字 27
逐字逐句 7
逐客 5
逐客令 1
逐年 697
逐年逐月 1
逐日 64
逐月 44
逐月逐日 0
逐條 49
逐次 72
逐步 1242
逐步來 2
逐波而去 0
逐波而去 0
逐漸 4168
逐級 29
逐臭 6
逐詞 0
逐走 5
逐退 3
逐項 85
逐鹿 38
逐鹿中原 5
逑 12
递 0
逓 0
途 1060
途上 10
途中 515
途徑 1016
途程 14
途經 68
途遇 2
逕 1179
逕寄 387
逕庭 0
逕流 84
逕直 0
逕自 90
逕行 345
逖 14
逗 283
逗人 5
逗人喜愛 0
逗人喜歡 0
逗弄 15
逗引 1
逗樂 5
逗留 71
逗留在 6
逗笑 12
逗號 31
逗趣 98
逗點 33
逗點符號 1
逘 0
逘 0
這 66602
這一 2059
這一 2059
這一 2059
這一下 27
這一些 71
這一來 34
這一個 495
這一生 168
這一陣子 63
這一點 586
這下 143
這下子 82
這些 20275
這些事 135
這些人 670
這些東西 151
這件 390
這件事 698
這位 2115
這位是 22
這個 1
這個 25909
這個人 1
這個人 580
這個字 1
這個字 580
這個月 1
這個月 174
這個詞 1
這個詞 580
這兒 1012
這兒 1012
這兒是 27
這兒是 27
這具 32
這口 25
這口氣 11
這句 192
這句話 670
這回 408
這堆 12
這場 106
這場 1065
這塊 542
這大概是 0
這套 1061
這季 13
這家 613
這家人 18
這對 730
這就 528
這就是 1684
這就是說 20
這局 4
這山望著那山高 0
這幀 1
這幾個 162
這幾個人 3
這幾個月 28
這幾天 478
這幾次 13
這廝 21
這支 208
這方 47
這方面 963
這是 16603
這時 2156
這時候 584
這會 197
這會兒 48
這會兒 48
這期 67
這本 520
這本書 943
這枝 4
這架 54
這株 13
這格 3
這樁 36
這樣 16470
這樣一來 150
這樣在 81
這樣子 686
這樣是 100
這樣會 137
這樣有 55
這樣能 14
這檔 18
這檔戲 1
這次 5610
這步田地 7
這段 1257
這炷香 0
這片 483
這班 65
這盒 2
這種 12356
這種事 222
這種人 260
這端 8
這筆 928
這篇 528
這篇文章 676
這籃 0
這粒 7
這聯 1
這股 248
這般 275
這般光景 0
這裏 966
這裡 10062
這裡在 11
這裡是 563
這裡會 23
這裡有 645
這裡能 14
這趟 81
這趟路 0
這連 3
這邊 707
這間 140
這陣 17
這陣子 73
這隻 405
這集 13
這類 1311
這麼 5913
這麼 6207
這麼一來 43
這麼一來 43
這麼一來 43
這麼一來 43
這麼些 2
這麼些 2
這麼個 17
這麼個 17
這麼樣 44
這麼樣 44
這麼著 4
這麼著 4
這點 571
這點兒 1
這點兒 1
通 13064
通不過 11
通了 49
通事 16
通亮 1
通人 19
通令 47
通令全國 4
通作 4
通來 12
通來通去 0
通例 7
通便 23
通便劑 2
通便藥 0
通俗 283
通俗化 13
通俗小說 19
通俗性 14
通俗文學 15
通俗讀物 6
通信 5118
通信兵 1
通信員 2
通信網 30
通信線路 10
通信處 7
通信衛星 16
通信鴿 0
通儒 3
通共 8
通典 8
通分 14
通到 37
通則 246
通力 10
通力合作 64
通勤 111
通化 28
通化街 69
通去 1
通史 142
通同 1
通名 4
通向 49
通告 954
通商 148
通商口岸 8
通商條約 1
通國 6
通報 1665
通夜 1
通天 72
通天本領 4
通奸 1
通好 2
通姦 26
通姦罪 0
通婚 33
通學 24
通學生 6
通宵 73
通宵達旦 11
通寶 83
通州 4
通常 4356
通常在 275
通常是 568
通常會 251
通式 8
通往 514
通得 4
通得 184
通心粉 11
通心麵 8
通志 18
通志館 3
通性 22
通情達理 5
通才 51
通指 6
通敵 6
通明 24
通暢 100
通暢無阻 3
通曉 43
通書 20
通有 23
通條 6
通槍 0
通槍條 2
通槍管 0
通權達變 2
通氣 63
通氣口 2
通氣孔 8
通河 36
通流 2
通為 13
通牒 11
通用 843
通用電氣 0
通用電氣公司 1
通病 31
通的 183
通盤 328
通盤性 13
通盤計劃 2
通知 4947
通知函 26
通知單 373
通知書 165
通神 14
通稱 100
通竅 1
通篇 28
通紅 50
通緝 201
通緝令 27
通緝犯 100
通縣 4
通義 13
通考 3
通聯 0
通聯記錄 0
通聲氣 0
通膨 0
通舖 9
通航 76
通草花 0
通融 26
通行 412
通行本 3
通行無阻 16
通行証 6
通行證 138
通衢 2
通衢大道 2
通訊 14966
通訊員 20
通訊社 190
通訊稿 0
通訊網 57
通訊處 278
通訊衛星 18
通訊設備 148
通訊錄 920
通話 731
通說 17
通論 114
通識 0
通譯 15
通讀 3
通變 3
通財 0
通財之義 0
通貨 152
通貨膨脹 301
通路 1489
通車 373
通通 400
通通風 0
通進 15
通運 225
通運公司 11
通過 8021
通過來 0
通過去 1
通道 646
通達 117
通達事理 1
通達人情 1
通達情理 0
通郵 9
通都 7
通量 136
通鑑 71
通關 709
通電 183
通電話 90
通霄 667
通靈 51
通順 39
通顯 0
通風 395
通風口 28
通風報信 10
通風孔 3
通風扇 9
通風機 20
通風處 12
通風設備 37
通體 16
通點 4
逛 612
逛一逛 35
逛上 2
逛上來 0
逛上去 0
逛下 0
逛下來 0
逛下去 0
逛了 55
逛來 2
逛來逛去 1
逛出 0
逛出來 0
逛出去 0
逛去 1
逛回 0
逛回來 0
逛回去 1
逛在 0
逛得 2
逛花燈 0
逛街 366
逛起 0
逛起來 1
逛逛 930
逛進 1
逛進來 0
逛進去 0
逛過 21
逛過來 0
逛過去 0
逜 102
逝 342
逝世 874
逝去 99
逝水 19
逝者 36
逝者已矣 3
逞 62
逞一時之快 7
逞一時之快 7
逞兇 3
逞凶 1
逞勇 0
逞匹夫之勇 0
逞威 5
逞威風 0
逞強 28
逞惡 0
逞欲 0
逞能 6
速 6073
速來 15
速來速往 0
速去 8
速去速回 0
速回 24
速寫 162
速寫員 0
速寫法 0
速射 1
速射砲 0
速度 6289
速度感 26
速成 102
速成法 10
速成班 38
速戰 3
速戰速決 23
速效 21
速效性 2
速比 9
速決 8
速決戰 0
速洽 54
速溶茶 1
速率 1096
速簡 10
速記 29
速記員 3
速記法 1
速記簿 5
速讀 70
速讀法 1
速讀課 0
速速 58
速配 0
速食 247
速食店 236
速食麵 91
造 6014
造神 12
造作 52
造假 69
造像 75
造價 171
造冊 53
造出 110
造出來 5
造勢 1232
造化 50
造化弄人 13
造反 47
造句 76
造句法 13
造園 116
造在 9
造型 2108
造型機 0
造型藝術 25
造字 460
造孽 10
造就 565
造就人才 3
造就人材 0
造山運動 0
造幣 12
造幣廠 9
造形 445
造形藝術 56
造形設計 31
造成 12804
造景 1
造林 407
造林學 0
造橋 350
造橋工程 0
造次 16
造渣 2
造父 0
造物 134
造物主 33
造田 2
造神運動 1
造福 270
造福人群 45
造福人類 8
造紙 685
造紙工業 163
造紙廠 22
造紙業 71
造紙術 6
造船 447
造船公司 41
造船工業 15
造船廠 81
造船業 57
造船系 62
造訪 1033
造詞 5
造詞法 0
造詣 93
造謠 48
造謠惑眾 1
造謠生事 4
造謠言 0
造酒 5
逡 10
逡巡 9
逢 1464
逢人說項 0
逢低買進 29
逢低賣出 0
逢凶化吉 16
逢到 1
逢場作戲 5
逢場作戲 5
逢山開路 0
逢年過節 8
逢甲 431
逢甲大學 1545
逢甲學院 3
逢迎 12
逢集 0
逢高買進 1
逢高賣出 5
連 15258
連上 1049
連上來 9
連上去 17
連下 14
連下來 3
連下去 1
連不上 262
連中 31
連串 156
連乘 6
連乘積 0
連了 14
連亙 2
連些 0
連人 37
連任 587
連住 8
連佔 0
連作 29
連來連去 2
連個 50
連出 33
連分數 0
連到 444
連刺 7
連加 5
連勝 117
連升 2
連叫 10
連台好戲 0
連吃 13
連合 56
連同 716
連名 23
連名帶姓 3
連唱 8
連喝 7
連回 23
連在 83
連在一起 53
連坐 30
連坐法 1
連城 249
連報 4
連夜 88
連夜趕來 1
連夜趕去 1
連天 53
連套 0
連字號 2
連守 0
連寄 0
連寫 9
連射 5
連帶 434
連帶而來 1
連帶處分 1
連帶責任 26
連帶關係 7
連年 51
連年豐收 0
連得 19
連心 26
連忙 241
連想到 9
連成 122
連戰 3542
連戰皆北 0
連戰皆捷 4
連手 0
連打 27
連打帶罵 0
連打帶踢 0
連投 8
連拍 0
連指 1
連捧 2
連採 1
連接 3594
連接不斷 0
連接成 11
連接線 98
連接詞 16
連接起來 42
連提 6
連揮 5
連撥 5
連操 0
連收 3
連攻 9
連敗 41
連斬 2
連於 9
連日 116
連日來 92
連有 17
連本帶利 1
連枝 0
連架 3
連根拔出 2
連根拔起 13
連桿 58
連桿機構 2
連橇 0
連橫 74
連江縣 469
連治 0
連混 0
連滾 3
連滾帶爬 8
連漲 16
連灌 1
連猜 0
連珠 20
連珠炮 4
連珠砲 8
連理 20
連理枝 6
連瑣 6
連環 156
連環圖畫 2
連環泡 7
連環畫 2
連環計 12
連生 35
連用 16
連發 91
連皮 2
連皮帶肉 1
連皮帶骨 0
連碰 6
連種 3
連空 1
連穿 3
連站 33
連筆 10
連筆畫 0
連答 0
連管 8
連篇 20
連篇累牘 5
連累 35
連累到 1
連結 10308
連結環 0
連結線 5
連結起來 21
連結車 0
連絡 9607
連絡官 0
連絡機 1
連絡站 23
連綿 62
連綿不斷 9
連綿不絕 11
連線 6605
連線作業 87
連繫 565
連繫起來 1
連續 5157
連續不斷 13
連續光譜 1
連續函數 3
連續劇 196
連續性 164
連續流 4
連續犯 3
連續生產 7
連署 522
連署 522
連聲 50
連臺好戲 0
連花 7
連莊 21
連號 6
連衡 0
連衣裙 1
連袂 41
連袂前來 1
連袂前去 0
連襟 2
連記法 8
連詞 0
連試 4
連貫 110
連貫性 32
連貫起來 3
連賽 0
連贏 12
連起 8
連起來 19
連跑 3
連跑帶跳 1
連跳 4
連踢 0
連踢帶打 0
連踩 0
連蹦帶跳 0
連身 129
連載 493
連載 493
連載報導 1
連載小說 14
連載故事 1
連通 107
連通器 0
連通管 5
連連 336
連進 39
連運 5
連過 28
連過來 2
連過去 17
連遭 8
連遭挫敗 0
連遭敗績 0
連選連任 7
連部 6
連鋪 0
連鎖 1447
連鎖反應 86
連鎖店 275
連鎖性 11
連長 112
連關 1
連關性 0
連除 0
連隊 17
連集 3
連雲 21
連雲港 13
連音 24
連頭也不回 1
連體 29
連體嬰 18
連鬢鬍子 0
逤 37
逥 0
逨 0
逩 0
逪 0
逫 0
逬 0
逬 0
逬 0
逭 3
逮 171
逮 171
逮住 10
逮到 49
逮捕 624
逮捕 624
逮獲 20
逯 11
逰 0
週 6271
週一 1094
週三 480
週二 456
週五 998
週休 0
週休二日 0
週全 38
週六 1216
週刊 1309
週到 67
週四 392
週圍 75
週報 1271
週密 8
週年 1415
週年慶 1
週年紀念 143
週延 32
週折 2
週數 57
週旋 7
週日 753
週會 115
週期 762
週期律 0
週期性 92
週期表 41
週末 615
週歲 33
週濟 1
週率 14
週知 36
週而復始 20
週薪 11
週詳 29
週身 4
週轉 144
週轉不靈 18
週轉率 85
週轉金 51
週遊 11
週遊世界 1
週遭 204
週遭事物 4
週邊 2325
週邊設備 589
週長 4
進 17804
進一 127
進一步 4252
進了 332
進些 1
進京 10
進京赴考 0
進位 143
進佔 18
進來 1461
進修 4509
進修班 269
進價 17
進入 14025
進入到 60
進兵 20
進出 3219
進出口 1572
進出口商 101
進出口貿易 388
進刀 24
進到 163
進前 20
進前一步 0
進前一步 0
進前一級 0
進前一級 0
進剿 0
進化 371
進化史 4
進化論 85
進占 2
進去 897
進取 157
進取之心 0
進取心 9
進口 16108
進口商 223
進口港 0
進口稅 27
進口貨 49
進在 8
進城 45
進場 99
進場 991
進士 84
進學 85
進完 0
進尺 4
進展 796
進帳 41
進度 2445
進度表 242
進得 19
進德 79
進德修業 10
進拜 2
進擊 79
進攻 305
進攻性 0
進料 122
進有 19
進款 0
進步 2331
進步事業 0
進步人士 0
進步到 44
進步力量 3
進步黨 406
進水 72
進水閘 0
進洞 66
進深 15
進港 89
進爵 2
進犯 14
進犯邊境 0
進用 199
進發 119
進移 0
進移到 1
進程 232
進站 199
進級 15
進給量 5
進而 2511
進行 17781
進行到 57
進行式 26
進行曲 175
進行著 65
進補 62
進言 10
進謁 2
進貢 10
進貨 273
進貨單 11
進貨讓價 0
進貨退出 7
進貨退回 3
進貨運費 4
進賢 12
進賬 0
進身之階 0
進軍 694
進退 107
進退兩難 30
進退失據 8
進退應對 1
進退維谷 18
進退自如 0
進進 8
進進出出 40
進逼 17
進過 13
進酒 11
進階 1
進項 66
進項稅額 62
進食 207
進餐 37
進餐時間 3
進香 61
進香團 9
進香客 2
進駐 682
逳 0
逴 5
逵 138
逶 10
逶迤 3
逷 0
逷 0
逸 1726
逸事 11
逸仙 471
逸出功 0
逸史 0
逸士 1
逸才 3
逸書 1
逸樂 11
逸民 47
逸氣 0
逸興 6
逸詩 0
逸趣 9
逸趣橫生 2
逸麗 0
逺 0
逻 0
逼 611
逼一逼 0
逼上 7
逼上來 1
逼上去 0
逼上梁山 3
逼下 1
逼下來 0
逼下去 0
逼下台 0
逼不得已 11
逼了 4
逼人 34
逼人太甚 3
逼住 0
逼使 47
逼來逼去 0
逼供 9
逼出 24
逼出來 0
逼出去 0
逼到 13
逼向 8
逼回 2
逼回來 0
逼回去 0
逼得 85
逼急 1
逼成 0
逼死 15
逼真 185
逼緊 0
逼著 25
逼視 9
逼走 14
逼起 1
逼起來 0
逼近 244
逼迫 179
逼進 42
逼進來 0
逼進去 0
逼過 2
逼過來 0
逼過去 1
逼降 0
逽 123
逾 1942
逾出 0
逾分 1
逾恆 2
逾放 0
逾放比 0
逾時 177
逾時不候 9
逾期 1425
逾期作廢 2
逾越 143
逾量 0
逾限 16
逾額 1
逾齡 13
逿 5
遀 0
遁 68
遁世 3
遁形 11
遁聲匿跡 0
遁藏 1
遁詞 2
遁走 5
遁跡 3
遁跡山林 0
遁辭 1
遁逃 22
遁隱 1
遂 140
遂 1406
遂其所願 0
遂即 4
遂心 3
遂心如意 0
遂意 2
遂願 3
遃 0
遃 0
遄 134
遅 0
遅 0
遆 0
遇 5691
遇上 511
遇事 24
遇人不淑 12
遇冷 10
遇冷收縮 0
遇到 2480
遇刺 6
遇合 6
遇害 93
遇救 0
遇敵 10
遇熱 12
遇熱膨脹 1
遇著 31
遇襲 6
遇見 587
遇過 76
遇險 44
遇險記 0
遇難 58
遇雨 27
遈 0
遉 85
遊 8959
遊了 4
遊人 91
遊人如織 6
遊伴 18
遊俠 145
遊刃有餘 6
遊動 12
遊園 245
遊園會 2
遊園驚夢 1
遊子 192
遊子天涯 0
遊子身上衣 1
遊學 671
遊學生 0
遊客 1450
遊客止步 3
遊宦 0
遊山玩水 64
遊必有方 10
遊憩 921
遊戲 16334
遊戲三昧 1
遊戲人間 20
遊戲場 45
遊戲場 45
遊戲規則 247
遊手好閑 0
遊手好閒 6
遊擊 7
遊擊戰 1
遊星 4
遊樂 1017
遊樂區 1071
遊樂園 234
遊樂場 43
遊樂場 430
遊歷 67
遊民 80
遊牧 36
遊牧區 0
遊牧民族 23
遊獵 7
遊玩 151
遊盪 13
遊絲 2
遊興 11
遊興正濃 0
遊船 43
遊艇 491
遊艇業 3
遊艇業者 0
遊蕩 68
遊蕩不歸 0
遊藝 139
遊藝會 2
遊行 1088
遊行示威 6
遊行示眾 0
遊街 96
遊街示眾 4
遊覽 444
遊覽區 20
遊覽車 331
遊記 219
遊說 197
遊資 40
遊走 145
遊蹤 239
遊逛 7
遊錯 0
遊離 10
遊離份子 0
遊離份子 0
遊離票 2
遊騎兵 35
遊魂 44
運 6932
運上 18
運上來 0
運上去 0
運下 2
運下來 0
運下去 1
運作 4055
運作者 17
運入 20
運兵 9
運兵船 6
運出 41
運出來 0
運出去 0
運到 59
運功 29
運功散 5
運動 12599
運動健將 6
運動史 26
運動員 309
運動器材 388
運動場 47
運動場 478
運動定律 25
運動家 56
運動會 858
運動服 56
運動神經 26
運動競賽 59
運動精神 36
運動衣 4
運動褲 7
運動選手 36
運動量 53
運動鞋 1
運勢 1
運命 18
運單 3
運回 65
運回來 1
運回去 1
運往 127
運搬 79
運數 2
運會 59
運氣 459
運河 164
運渠 0
運營 10
運球 31
運球過場 0
運球過場 0
運用 6125
運用上 61
運用於 149
運用自如 23
運祚 0
運筆 12
運算 1343
運算元 102
運算器 7
運算子 164
運算式 0
運算法 5
運算碼 2
運籌 0
運籌學 6
運籌帷幄 26
運糧 21
運著 0
運行 312
運貨 29
運貨車 0
運費 1501
運走 8
運起 21
運起來 0
運載 32
運載火箭 1
運輸 3318
運輸公會 0
運輸商 2
運輸成本 30
運輸業 565
運輸機 29
運輸系統 282
運輸網 7
運輸線 1
運輸能力 3
運輸船 10
運輸艦 35
運輸設備 110
運輸設備修護費 0
運輸費 4
運輸量 7
運轉 1065
運轉帶 0
運送 841
運送費 2
運道 13
運量 0
運鈔 3
運鈔車 11
運銷 452
運銷網 0
運體 5
遌 0
遌 0
遍 1074
遍 1074
遍佈 0
遍佈 195
遍及 238
遍及 238
遍地 64
遍地 64
遍尋 91
遍尋 91
遍尋天下 0
遍尋天下 0
遍布 71
遍布 71
遍數 1
遍數 1
遍於 2
遍於 2
遍覽 5
遍覽 5
遍覽群籍 0
遍覽群籍 0
遍身 2
遍身 2
遍野 58
遍野 58
遍體 9
遍體 9
遍體鱗傷 34
遍體鱗傷 34
過 18944
過一天 23
過不下去 4
過不了 40
過不來 0
過不去 23
過不得 1
過不過 7
過世 228
過了 1414
過五關 8
過五關斬六將 4
過些 28
過些日子 10
過亮 0
過人 115
過人之處 14
過份 296
過位 1
過來 1042
過來人 92
過偏 0
過入 35
過兩天 22
過共析鋼 0
過冬 56
過冷 21
過出 5
過分 185
過切 3
過到 36
過則勿憚改 0
過剩 239
過剩人口 1
過動兒 0
過午不食 2
過半 282
過半數 182
過去 7290
過去了 167
過去分詞 5
過去完成式 0
過去式 25
過去進行式 0
過問 51
過圓 0
過圳 19
過堂 0
過堂風 0
過場 36
過場 36
過場戲 0
過場戲 0
過境 348
過境旅客 2
過境簽證 7
過多 794
過夜 108
過夜費 1
過大 379
過天 20
過失 309
過失傷害 8
過失殺人 6
過好 74
過完 53
過客 673
過少 34
過山砲 0
過山龍 2
過帳 66
過年 268
過度 1391
過強 17
過彎 65
過往 347
過往人 0
過往甚密 0
過後 702
過得 2106
過得來 0
過得去 5
過從 61
過從甚密 2
過心 22
過急 12
過悶 3
過意 2
過意不去 21
過慣 17
過慮 12
過戶 262
過戶費 2
過房 2
過所 21
過手 18
過招 87
過擠 1
過敏 1152
過敏原 183
過敏性 462
過敏性鼻炎 1
過敏症 38
過數 61
過於 688
過日子 88
過早 59
過時 173
過時不候 0
過晚 10
過望 3
過期 736
過期未付 0
過期無效 1
過杆 0
過柔 0
過橋 70
過橋費 2
過歪 0
過氣 81
過氧化 54
過氧化氫 97
過氧化物 23
過水 26
過江之鯽 11
過河 47
過河拆橋 10
過油 15
過洋 1
過洋而來 0
過活 48
過海 37
過涼 1
過渡 235
過渡形式 0
過渡性 27
過渡時期 46
過渡時期總路線 0
過渡期 30
過渡階段 9
過激 10
過激份子 2
過激份子 2
過激派 1
過濃 7
過濾 882
過濾出來 1
過濾器 334
過濾性 2
過濾機 30
過濾法 26
過濾網 39
過火 51
過為 26
過熟 6
過熱 154
過熱器 3
過猶不及 14
過獎 36
過甚 10
過甚其詞 0
過生日 38
過產 2
過當 78
過癢 0
過癮 189
過白 14
過百 52
過的 2105
過目 28
過目不忘 6
過目成誦 0
過眼 13
過眼雲煙 13
過短 25
過硬 4
過磅 18
過磷酸鈣 0
過秤 0
過程 8384
過站 8
過站不停 1
過竿 0
過節 53
過篩 16
過細 7
過給 7
過綠 5
過繼 6
過而不入 0
過肩而擲 0
過脆 0
過與不及 8
過著 333
過處 14
過街 17
過街老鼠 13
過街老鼠人人喊打 0
過訪 4
過話 35
過謙 5
過譽 2
過責 0
過起 4
過起來 1
過足 11
過路 42
過路人 17
過路費 5
過載 97
過透 2
過過 27
過過癮 4
過道 45
過遠 13
過重 131
過量 388
過錯 53
過錳酸鉀 8
過長 152
過門 40
過門不入 11
過關 465
過關斬將 21
過電壓 7
過頭 239
過飽和 3
過黏 1
遏 34
遏制 15
遏惡揚善 0
遏抑 6
遏止 178
遏止住 0
遏阻 60
遏阻作用 2
遐 44
遐志 0
遐思 22
遐想 27
遐邇 10
遐齡 2
遑 43
遑論 126
遑論其他 4
遑遑不可終日 0
遒 6
遒健 2
遒勁 6
道 24036
道上 197
道不同不相為謀 3
道不拾遺 0
道人 147
道伕 0
道來 42
道候 1
道兄 14
道光 177
道光帝 0
道具 341
道出 158
道別 106
道化 15
道友 114
道口 30
道員 0
道喜 2
道地 220
道場 30
道場 302
道士 152
道夫 89
道姑 15
道學 59
道學家 2
道宣 1
道家 221
道岔 4
道德 1480
道德上 48
道德哲學 10
道德心 1
道德感 13
道德經 31
道德規範 42
道德觀 42
道德觀念 24
道心 34
道情 4
道教 373
道明 307
道林紙 57
道歉 849
道法 38
道理 829
道理上 6
道白 15
道破 10
道種智 1
道統 18
道義 39
道義上 6
道義之交 0
道義責任 6
道聽塗說 11
道藏 7
道號 2
道行 48
道行 48
道術 13
道袍 14
道觀 21
道諦 1
道謝 30
道貌 2
道貌岸然 15
道賀 25
道路 3540
道路上 105
道路網 5
道述 1
道道 65
道道地地 24
道里 22
道釘 0
道長 77
道門 20
道高一尺 13
達 223
達 22399
達人 43
達令 278
達到 5643
達卡 396
達喀爾 14
達因 25
達士 14
達姆達姆 0
達姆達姆彈 0
達官 18
達官貴人 12
達德 43
達意 18
達成 3804
達拉斯 120
達摩 140
達文西 61
達旦 3
達爾主義 0
達爾文 119
達理 18
達生 32
達致 21
達芬奇 1
達荷美 0
達見 8
達觀 89
達觀山 40
達識 1
達賴 333
達賴喇嘛 260
達道 6
達達 96
達達主義 8
達達畫派 0
達陣 0
違 677
違令 3
違例 5
違反 3253
違反常理 0
違反常規 0
違反規定 44
違和 3
違建 221
違建戶 17
違心 3
違心之論 1
違心論 0
違憲 246
違抗 10
違抗上級 0
違抗命令 2
違拗 5
違法 908
違法亂紀 8
違法行為 51
違犯 46
違理 0
違礙 0
違禁 23
違禁品 43
違禮 1
違章 138
違章建築 64
違紀 0
違約 701
違約金 99
違者 490
違背 409
違背良心 10
違規 1264
違規人 7
違規者 20
違誤 20
違警 5
違警罰法 6
違警行為 3
違逆 19
遘 944
遙 1394
遙寄 6
遙念 0
遙想 21
遙指 22
遙控 655
遙控器 312
遙控裝置 9
遙控鈕 0
遙望 71
遙測 564
遙測計 0
遙相呼應 5
遙祭 5
遙見 9
遙遙 37
遙遙在望 2
遙遙地 1
遙遙地 1
遙遙無期 38
遙遙的 0
遙遙相對 14
遙遙相望 11
遙遙領先 66
遙遠 458
遚 0
遛 511
遛 511
遛狗 3
遛達 1
遛達 1
遛鳥 1
遜 4019
遜位 0
遜色 73
遝 469
遞 866
遞上 17
遞上來 1
遞上去 1
遞了 14
遞交 67
遞來 0
遞信 5
遞傳 2
遞出 17
遞出來 0
遞出去 1
遞加 1
遞升 1
遞向 0
遞呈 1
遞回 7
遞回來 0
遞回去 0
遞增 80
遞增級數 0
遞延 37
遞昇 0
遞歸 0
遞減 133
遞眼色 0
遞移 0
遞移律 0
遞給 85
遞補 226
遞解 5
遞變 9
遞迴 88
遞送 182
遞進 3
遞過來 10
遞過去 3
遞降 6
遟 0
遟 0
遠 972
遠 9720
遠交近攻 1
遠傳 0
遠光燈 2
遠別 2
遠到 20
遠勝 58
遠去 89
遠古 55
遠古時代 17
遠因 17
遠在 185
遠在天邊 18
遠地 60
遠地點 1
遠大 179
遠嫁他鄉 0
遠客 4
遠客來訪 0
遠射 3
遠山 51
遠山近水 0
遠岸 2
遠征 84
遠征軍 15
遠從 56
遠志 46
遠慮 8
遠房 3
遠房親戚 5
遠播 29
遠方 622
遠於 3
遠族 0
遠日點 6
遠景 482
遠望 70
遠期 124
遠東 1689
遠東區 36
遠東國家 3
遠東圖書 45
遠東圖書館 0
遠東地區 27
遠東工專 13
遠東技術學院 1
遠東科技大學 1
遠比 234
遠水 5
遠水救不了近火 4
遠水救不了近火 4
遠洋 91
遠洋漁業 101
遠洋漁船 20
遠洋運輸 3
遠涉 5
遠涉重洋 2
遠渡 4
遠渡重洋 17
遠溯 10
遠略 1
遠略宏規 0
遠眺 198
遠矚 2
遠祖 9
遠程 204
遠程目標 50
遠端 0
遠自 15
遠航 148
遠航船 0
遠處 380
遠行 46
遠見 299
遠視 63
遠視眼 1
遠親 22
遠親不如近鄰 3
遠親近戚 0
遠識 1
遠走 18
遠走他鄉 8
遠走高飛 10
遠超過 174
遠足 47
遠距 1921
遠距離 96
遠路 94
遠近 138
遠近法 2
遠近皆知 0
遠近馳名 46
遠遊 5
遠道 46
遠道而來 27
遠達 15
遠遠 324
遠遠的 97
遠遠地 97
遠避 6
遠郊 3
遠銷 5
遠門 4
遠限 0
遠隔 30
遠隔重洋 1
遠離 793
遠非 21
遠颺 16
遡 0
遡 0
遢 1
遢 309
遣 913
遣下去 0
遣使 14
遣俘 0
遣將 0
遣悶 0
遣憂 0
遣懷 3
遣散 5
遣散費 3
遣歸 3
遣興 4
遣詞造句 3
遣返 92
遣返作業 0
遣送 20
遣送出境 1
遣送回國 1
遦 0
遧 6
遨 127
遨嬉 0
遨遊 653
遨遊四海 9
遨遊天下 5
適 33
適 3381
適中 153
適之 0
適任 157
適切 336
適口 4
適可而止 23
適合 6440
適合於 121
適合者 3
適婚 26
適婚年齡 6
適婚者 0
適宜 471
適度 618
適得其反 33
適意 12
適應 1902
適應力 42
適應性 79
適應環境 50
適才 88
適於 273
適時 832
適用 1511
適用性 81
適用於 1669
適用者 12
適當 2332
適當的 1920
適當地 1920
適纔 4
適者 23
適者生存 31
適航性 3
適逢 0
適量 369
適齡 34
遪 0
遫 8
遫 8
遬 0
遭 3707
遭人 151
遭人暗算 1
遭人算計 0
遭到 944
遭劫 13
遭厄 0
遭受 978
遭受到 80
遭天打雷劈 2
遭殃 59
遭災 10
遭致 46
遭變 22
遭逢 103
遭遇 1041
遭遇到 111
遭遇戰 4
遭難 25
遭雷擊 9
遮 245
遮 245
遮上 0
遮上來 0
遮上去 0
遮下 1
遮不了 0
遮不住 5
遮人 0
遮人耳目 0
遮住 109
遮光 48
遮光板 2
遮太陽 1
遮得 2
遮成 1
遮掩 72
遮掩起來 0
遮擋 5
遮擋住 2
遮攔 18
遮斷 52
遮架 0
遮的 2
遮目魚 0
遮眼 8
遮眼法 2
遮眼睛 0
遮瞞 0
遮罩 160
遮羞 5
遮羞費 0
遮臉 5
遮蓋 94
遮蓋住 3
遮蓋著 0
遮蓋起 0
遮蓋起來 3
遮蔭 28
遮蔽 135
遮蔽住 2
遮蔽物 4
遮蔽罩 2
遮蔽著 0
遮覆 2
遮起 4
遮起來 5
遮路 0
遮遮掩掩 9
遮那 1
遮醜 0
遮陰 16
遮陽 83
遮陽帽 7
遮陽板 22
遮雨 10
遮雨棚 14
遮面 6
遮頭蓋面 0
遮風 17
遯 18
遯世 0
遯形 0
遰 2
遱 0
遲 599
遲了 46
遲來 143
遲到 255
遲到早退 10
遲延 100
遲慢 1
遲早 103
遲早是 4
遲早會 39
遲暮 1
遲滯 61
遲滯不前 1
遲演 0
遲疑 165
遲疑不前 2
遲疑不決 3
遲緩 430
遲誤 7
遲遲 268
遲鈍 114
遳 6
遴 485
遴選 691
遴選出 12
遴選出來 1
遵 293
遵令 1
遵化 1
遵命 22
遵奉 9
遵守 1486
遵守規則 13
遵從 85
遵循 391
遵旨 0
遵照 299
遵照指示 5
遵照辦理 9
遵義 2
遵義會議 1
遵行 234
遵辦 4
遶 80
遷 991
遷上 1
遷上來 0
遷上去 0
遷下 0
遷下來 1
遷下去 0
遷了 0
遷來 20
遷入 189
遷出 90
遷出來 0
遷出去 0
遷到 42
遷升 0
遷去 0
遷台 72
遷善 6
遷善改過 0
遷回 29
遷回來 0
遷回去 0
遷地為良 0
遷好 0
遷妥 0
遷完 1
遷就 53
遷就於 3
遷居 57
遷廠 64
遷延 7
遷延時日 0
遷往 60
遷得 0
遷徙 153
遷徙流離 0
遷怒 12
遷怒於 4
遷戶 9
遷移 615
遷移費 12
遷西 0
遷調 96
遷走 7
遷進 3
遷進來 0
遷進去 0
遷過 0
遷過來 0
遷過去 0
遷都 23
遷開 0
選 25749
選上 128
選上來 0
選上去 0
選中 76
選任 115
選修 1874
選修科 11
選修科目 332
選修課 45
選個 57
選入 33
選出 991
選務 91
選務工作 16
選務所 0
選區 1147
選去 4
選取 1553
選單 55124
選址 42
選委會 0
選字 0
選季 0
選定 881
選情 0
選成 11
選戰 2711
選手 1262
選手村 54
選拔 536
選拔委員會 2
選拔幹部 0
選拔會 3
選拔賽 65
選揀 0
選播 14
選擇 19626
選擇性 1623
選擇權 323
選擇法 6
選擇率 5
選擇項 31
選擇題 264
選曲 72
選本 17
選材 15
選校 32
選校不選系 1
選樣 19
選樣法 0
選民 2465
選民代表 0
選派 90
選派代表 8
選煤 1
選用 989
選礦 6
選票 827
選秀 0
選科 13
選種 34
選粹 145
選系 22
選系不選校 4
選罷法 143
選美 144
選美大會 1
選育 38
選舉 9344
選舉事務 2
選舉事務所 1
選舉人 75
選舉制 9
選舉制度 174
選舉區 101
選舉委員 22
選舉委員會 148
選舉日 31
選舉權 67
選舉法 12
選舉票 11
選舉罷免 24
選舉罷免法 49
選舉資格 3
選萃 6
選號 9
選言判斷 0
選言肢 0
選詞 0
選課 992
選讀 306
選讀生 24
選讀科目 1
選讀課 0
選賢與能 31
選購 5427
選輯 344
選送 31
選配 60
選錄 67
選集 258
選項 2194
選風 137
遹 31
遺 1019
遺 109
遺下 15
遺世獨立 14
遺作 34
遺俗 0
遺傳 1218
遺傳信息 0
遺傳學 128
遺傳學家 10
遺傳工程 78
遺傳性 105
遺傳物質 21
遺傳病 19
遺像 26
遺命 13
遺囑 158
遺址 737
遺墨 6
遺失 1366
遺失啟事 1
遺孀 32
遺孤 13
遺孽 0
遺害 7
遺害終身 0
遺害萬年 0
遺容 3
遺尿 15
遺志 11
遺忘 439
遺念 0
遺恨 9
遺恨終身 0
遺患 4
遺患終身 0
遺愛 5
遺愛人世 0
遺愛人間 4
遺憾 877
遺憾終生 6
遺憾終身 18
遺教 14
遺族 180
遺書 66
遺棄 177
遺業 2
遺毒 9
遺民 15
遺漏 190
遺漏掉 3
遺照 1
遺物 80
遺珠 26
遺珠之憾 31
遺產 668
遺產稅 489
遺留 150
遺留下 66
遺留下來 15
遺禍 2
遺稿 4
遺策 0
遺精 6
遺緒 8
遺缺 40
遺老 6
遺腹子 6
遺臣 7
遺臭萬年 1
遺落 42
遺著 2
遺言 31
遺訓 4
遺詔 4
遺贈 54
遺跡 196
遺蹟 49
遺蹤 3
遺音猶在 0
遺願 6
遺風 17
遺餘 4
遺饋 0
遺骨 15
遺骸 23
遺體 143
遻 79
遼 317
遼北 5
遼北省 2
遼原 6
遼太后 0
遼寧 115
遼寧省 116
遼寧街 27
遼東 20
遼東半島 9
遼東灣 4
遼河 16
遼河平原 4
遼源 3
遼西 14
遼遠 2
遼闊 158
遼陽 43
遽 121
遽下 5
遽下結論 1
遽增 49
遽然 8
遽然變色 0
遽爾 2
遽聞 2
遾 71
避 1654
避一避 1
避不作答 0
避不見面 11
避世 1
避亂 2
避了 5
避人耳目 6
避債 8
避免 6194
避到 2
避匿 0
避向 1
避嫌 7
避孕 354
避孕藥 196
避害 0
避寒 14
避實就虛 0
避實擊虛 1
避席 3
避忌 1
避惡 1
避暑 43
避暑勝地 20
避暑山莊 3
避禍 8
避秦 0
避稅 22
避縮 0
避縮到 0
避而不答 6
避而不見 1
避而不談 16
避著 4
避諱 35
避讀 0
避避 1
避避風頭 1
避邪 31
避重就輕 32
避開 375
避難 371
避難所 23
避雨 20
避雷器 33
避雷線 0
避雷針 24
避風 74
避風港 79
避風頭 8
邀 1813
邀功 6
邀天之幸 0
邀宴 25
邀斤論兩 0
邀稿 30
邀約 235
邀請 3859
邀請信 2
邀請函 205
邀請卡 181
邀請書 11
邀請賽 175
邀集 626
邁 1218
邁了 0
邁克 45
邁入 528
邁出 43
邁可 45
邁向 1753
邁往 0
邁方步 0
邁步 29
邁步前進 1
邁著 3
邁進 373
邁過去 0
邁開 40
邁阿密 102
邁阿密海灘 0
邂 26
邂逅 297
邂逅 297
邃 39
邃古 0
邃密 0
還 17260
還 17260
還 260
還上 19
還上來 3
還上去 0
還下 8
還下來 1
還下到 0
還下去 0
還不 1280
還不在 1
還不如 176
還不是 352
還不會 64
還不能 154
還不行 21
還了 23
還佔 8
還來 131
還來到 0
還俗 19
還修 4
還倒 5
還值 18
還債 35
還像 17
還價 12
還先 4
還再 36
還出 16
還出來 2
還出去 1
還到 26
還剛 0
還加 40
還原 560
還原作用 13
還原劑 161
還原成 32
還原染料 32
還原法 4
還原熔煉 0
還去 44
還受 12
還受到 13
還口 2
還只是 61
還只有 4
還叫 38
還可 716
還可以 1759
還可在 12
還可能 122
還向 33
還吞 0
還吵 4
還吸 0
還和 32
還咬 0
還哭 4
還唱 5
還問 48
還喊 3
還嘴 0
還回 25
還回來 2
還回到 1
還回去 1
還在 1785
還報 4
還夠 5
還夠用 1
還太 21
還套 0
還好 962
還好在 7
還好是 8
還守 2
還封 0
還射 1
還將 275
還小 59
還師 0
還席 0
還帳 0
還帶 54
還常 33
還常常 13
還平 0
還彈 1
還往 3
還很 210
還得 476
還得起 1
還從 12
還怕 43
還怪 11
還恨 0
還情 2
還想 281
還應 53
還應該 15
還我 145
還我河山 6
還戴 3
還手 19
還抄 2
還把 129
還指 21
還掃 1
還排 3
還採 4
還提 45
還搖 0
還撕 0
還擊 42
還收 26
還放 29
還政 1
還政於民 8
還教 33
還於 6
還早 46
還是 13663
還是在 225
還是會 441
還是能 39
還是要 685
還書 0
還替 20
還會 1455
還會在 29
還會是 11
還有 13060
還未 530
還本 355
還款 100
還沒 1537
還沒在 5
還沒有 1064
還沖 0
還洗 0
還派 11
還清 57
還炒 1
還為 79
還無 20
還猜 4
還玩 2
還用 77
還用在 1
還用於 0
還用說 16
還由 11
還痛 6
還發 31
還看 103
還看到 18
還真 1461
還真好 8
還真是 143
還真有 21
還真難 11
還睡 10
還睡到 0
還知道 42
還硬 16
還禮 7
還穿 17
還笑 17
還算 337
還算在 0
還算是 45
還管 9
還給 257
還給到 0
還編 3
還縫 2
還織 0
還罰 2
還罵 23
還背 4
還能 894
還能在 31
還能夠 66
還能有 27
還自 24
還與 37
還苦 1
還著 4
還行 8
還行得通 1
還衝 1
還被 138
還要 2721
還討 0
還設 44
還說 301
還說在 1
還說是 13
還說有 1
還調 3
還談 31
還講 14
還變 1
還讓 53
還讓出 0
還讓到 0
還貴 9
還賜 5
還賬 0
還贏 6
還起來 1
還跟 60
還跟到 0
還跟在 2
還跟著 7
還載 2
還通 8
還過 6
還過來 0
還過去 2
還過得 0
還過得去 2
還都 21
還都不 1
還都在 1
還都是 5
還鄉 11
還鄉團 0
還酸 0
還錢 25
還長 36
還闖 1
還關 2
還防 1
還陽 3
還願 15
還願意 14
還養 5
還魂 21
邅 22
邆 78
邇 48
邇來 38
邈 152
邈然 0
邈視 1
邈邈 1
邉 0
邊 6713
邊上 45
邊下 12
邊修 5
邊做 31
邊做邊看 0
邊兒 2
邊兒 2
邊區 18
邊在 28
邊地 11
邊執見 1
邊塞 12
邊塞風光 0
邊境 102
邊寨 0
邊寫 15
邊帶 33
邊幅 0
邊廷 0
邊患 1
邊打 34
邊放 16
邊政 118
邊旁 1
邊材 4
邊框 112
邊民 2
邊沿 6
邊界 338
邊界問題 8
邊界層 29
邊界線 2
邊疆 63
邊疆地區 13
邊疆地帶 1
邊疆民族 6
邊看 103
邊看邊吃 0
邊看邊聽 0
邊看邊說 0
邊看邊講 0
邊答 1
邊線 54
邊緣 888
邊緣穿孔卡 0
邊見 1
邊角料 0
邊說 98
邊講 11
邊走 86
邊走邊叫 0
邊走邊吃 4
邊走邊想 1
邊走邊看 3
邊走邊談 1
邊跳 5
邊跳邊唱 0
邊遠 14
邊遠地區 6
邊邊 16
邊郡 2
邊鋒 2
邊鎮 0
邊門 2
邊間 34
邊關 8
邊防 27
邊防要塞 1
邊陲 83
邊隘 0
邊際 224
邊際效應 4
邊飾 2
邋 1
邋 12
邋裡邋遢 0
邋裡邋遢 0
邋裡邋遢 1
邋裡邋遢 1
邋遢 1
邋遢 1
邋遢 19
邌 0
邍 15
邎 0
邏 82
邏各斯 0
邏輯 1619
邏輯學 11
邏輯思維 11
邏輯性 13
邐 31
邐迤 0
邑 506
邑人 6
邑宰 0
邒 0
邓 0
邓 0
邔 4
邕 44
邖 0
邗 5
邘 4
邙 8
邚 0
邛 16
邞 108
邟 51
邠 94
邡 16
邢 367
邢台 1
那 2556
那 32
那 3219
那 32190
那 32190
那一 1476
那一 1476
那不勒斯 18
那些 1536
那些 5893
那些人 206
那件 79
那件事 51
那份 229
那位 1281
那位 315
那來 55
那個 1
那個 1
那個 5167
那個 551
那個人 1
那個人 335
那個月 1
那個月 4
那兒 356
那兒 356
那兒 905
那兒 905
那具 21
那冊 0
那匹 15
那區 2
那句 68
那句話 60
那台 105
那咱 2
那回 14
那圈 2
那國 23
那壺 5
那夜 33
那天 1096
那就 2317
那層 13
那層樓 2
那年 262
那幾 52
那幾個 67
那幾天 58
那幾次 0
那廝 11
那怕 39
那怕是 27
那戶 3
那批 19
那支 39
那斯達克 0
那方面 38
那是 2425
那時 1519
那時候 481
那會 153
那會兒 1
那會兒 1
那有 194
那有 485
那期 3
那末 8
那本 91
那本書 59
那枝 7
那棵 35
那樣 1193
那樣 23
那營 1
那片 90
那班 23
那盞 11
那科 15
那種 13
那種 2080
那種人 45
那站 5
那筆 16
那箱 0
那篇 108
那群 92
那群人 16
那能 14
那般 100
那行 20
那裡 2411
那裡 3832
那課 0
那起 10
那輛 44
那輛車 11
那邊 653
那部 74
那鍋 210
那間 93
那隻 393
那集 15
那頁 6
那頂 11
那項 35
那顆 124
那顆 85
那首 75
那首歌 42
那麼 8366
那麼 8597
那麼些 4
那麼些 4
那麼樣 17
那麼樣 17
那麼著 0
那麼著 0
那麼點 8
那麼點 8
那麼點兒 0
那麼點兒 0
那麼點兒 0
那麼點兒 0
邤 0
邥 18
邦 6394
邦交 255
邦交國 1
邦國 49
邦土 1
邦基 29
邦家 2
邦彥 32
邦本 7
邦畿 0
邦聯 51
邧 165
邨 0
邩 0
邪 10
邪 10
邪 105
邪 1052
邪不勝正 5
邪世 0
邪佞 0
邪僻 3
邪命 1
邪妄 1
邪心 3
邪念 15
邪思 1
邪惡 323
邪教 63
邪方便 1
邪業 1
邪氣 26
邪法 39
邪淫 23
邪臣 0
邪行 1
邪術 18
邪見 17
邪計 0
邪語 2
邪說 12
邪譎 1
邪路 2
邪辟 0
邪途 1
邪道 17
邪門 36
邪門歪道 2
邪鬼 7
邪魔 22
邪魔外道 0
邫 0
邭 0
邮 0
邯 91
邯鄲 17
邯鄲學步 3
邰 299
邱 6748
邱吉爾 33
邲 169
邳 68
邴 106
邵 573
邵族 1
邵武 5
邵氏 5
邵陽 9
邶 8
邷 0
邸 350
邸宅 2
邸第 1
邹 0
邼 0
邽 727
邾 531
邿 1059
郀 0
郁 1465
郁烈 0
郁穆 1
郁郁 3
郁馥 1
郂 0
郃 75
郃 75
郄 0
郄 0
郅 590
郆 0
郇 446
郈 314
郉 0
郊 341
郊區 187
郊原 1
郊外 106
郊祀 0
郊遊 218
郊野 19
郋 611
郌 0
郍 0
郍 0
郎 3070
郎中 64
郎君 227
郎官 3
郎才女貌 2
郎舅 1
郎費羅 0
郔 574
郕 625
郖 62
郗 53
郘 10
郙 20
郚 17
郛 45
郜 24
郝 510
郝 510
郝搖旗 0
郝柏村 159
郞 0
郟 16
郠 5
郡 827
郡主 34
郡候 0
郡國 12
郡守 10
郡王 11
郡縣 11
郡縣制 0
郡縣制度 0
郢 31
郢人 0
郢書燕說 4
郣 77
郤 289
郥 4
部 26188
部下 105
部件 29
部份 13673
部位 1351
部分 9069
部品 52
部員 22
部委 45
部將 376
部屬 287
部從 20
部族 89
部會 534
部會首長 47
部發 25
部署 209
部落 1078
部落格 1078
部落社會 8
部長 1658
部長會議 94
部門 4086
部院 5
部隊 836
部隊長 7
部頒 71
部類 5
部首 59
郩 17
郪 107
郫 42
郫縣 1
郬 105
郭 22013
郭婉容 23
郭子儀 10
郭富城 183
郭振泰 6
郭江 4
郭沫若 5
郭象 6
郮 0
郯 19
郰 20
郱 1265
郲 54
郳 36
郴 15
郴州 1
郵 2294
郵亭 3
郵件 10509
郵便局 4
郵傳部 3
郵務 68
郵包 76
郵匯 12
郵匯局 92
郵區 5
郵報 142
郵寄 1310
郵寄費 13
郵局 4680
郵差 162
郵戳 73
郵戳為憑 184
郵政 1679
郵政代辦 0
郵政代辦所 12
郵政信箱 206
郵政儲金 220
郵政劃撥 675
郵政局 29
郵政局長 0
郵政編碼 296
郵政總局 243
郵票 906
郵票夾 0
郵票迷 0
郵筒 28
郵箱 503
郵簡 0
郵船 11
郵袋 7
郵費 47
郵資 3371
郵購 1056
郵購價格 0
郵購品 0
郵購目錄 17
郵車 1
郵輪 234
郵遞 481
郵遞區域號碼 0
郵遞區號 469
郵電 188
郵電費 6
郶 0
郷 0
郷 0
郹 107
郺 64
郻 2
郼 1
都 4005
都 40050
都上 46
都上來 8
都上去 0
都下 28
都下來 0
都下去 0
都不 2718
都不在 40
都不好 41
都不對 21
都不是 330
都不會 416
都不能 348
都住 34
都作 37
都來 241
都倒 9
都做 193
都像 156
都再 21
都出 76
都出來 7
都出去 4
都到 128
都卜勒 64
都去 254
都受 54
都受了 4
都受到 94
都可 1322
都可以 2431
都向 37
都回 29
都回來 1
都回去 3
都因 86
都因為 58
都在 2274
都坐 20
都城 22
都好 237
都察院 0
都尉 15
都市 4565
都市人口 15
都市化 114
都市病 0
都市計劃 331
都市計畫 1439
都府 1
都很 1448
都必須 579
都想 298
都懂 47
都應 345
都應該 243
都指揮使 0
都搬 16
都是 13258
都會 4991
都會區 644
都有 5666
都柏林 20
都比 273
都江堰 8
都沒 1433
都沒有 1487
都為 237
都由 194
都發 133
都監 12
都督 33
都督府 2
都統 6
都罰 0
都能 2251
都能夠 180
都裝 33
都要 1743
都說 549
都說在 3
都說是 12
都說會 0
都說有 5
都起來 7
都載 6
都辦 20
都逼 1
都還 616
都還不 29
都還沒 89
都邑 7
郾 46
郿 10
鄀 6
鄁 7
鄂 199
鄂倫春 12
鄃 5
鄄 10
鄅 5
鄆 21
鄇 3
鄈 0
鄉 1
鄉 14228
鄉下 396
鄉下人 41
鄉下佬 0
鄉下城 0
鄉下老兒 0
鄉下老兒 0
鄉井 3
鄉人 90
鄉佬 0
鄉俗 1
鄉公所 1020
鄉勇 5
鄉原 27
鄉友 7
鄉名 62
鄉味 12
鄉土 3135
鄉土味 17
鄉土文學 41
鄉土氣 0
鄉土氣息 9
鄉土藝術 21
鄉土觀念 0
鄉塾 1
鄉居 13
鄉巴佬 15
鄉心 2
鄉思 6
鄉情 798
鄉愁 114
鄉愿 18
鄉曲 15
鄉會 35
鄉村 588
鄉村化 2
鄉村服 0
鄉村都市化 1
鄉校 2
鄉民 811
鄉民代表 23
鄉民代表會 218
鄉紳 14
鄉裡 8
鄉親 483
鄉試 6
鄉誼 1
鄉誼 1
鄉談 9
鄉賢 12
鄉運會 3
鄉鄰 3
鄉里 94
鄉野 78
鄉鎮 1870
鄉鎮代表 2
鄉鎮民代 1
鄉鎮民代表 6
鄉鎮長 40
鄉長 472
鄉間 161
鄉間別墅 2
鄉關 11
鄉音 32
鄉音未改鬢毛衰 1
鄉黨 8
鄊 0
鄊 0
鄋 27
鄌 0
鄍 60
鄎 22
鄏 39
鄐 195
鄑 31
鄒 898
鄒容 0
鄒族 0
鄓 0
鄓 0
鄔 179
鄕 0
鄕 0
鄖 51
鄗 5
鄘 287
鄙 292
鄙 292
鄙人 17
鄙俗 7
鄙吝 1
鄙嗇 1
鄙夫 0
鄙夷 11
鄙意 1
鄙棄 7
鄙笑 0
鄙薄 2
鄙見 0
鄙視 34
鄙言 0
鄙賤 0
鄙陋 0
鄚 391
鄛 9
鄜 69
鄝 40
鄞 206
鄟 143
鄠 9
鄡 8
鄢 33
鄣 252
鄤 15
鄥 0
鄥 0
鄦 4
鄧 1640
鄧中夏 0
鄧小平 230
鄧尼金 0
鄧肯 25
鄧茂七 0
鄧麗君 119
鄨 2
鄩 5
鄪 1
鄫 6
鄬 3
鄭 13152
鄭克爽 2
鄭和 37
鄭國 80
鄭孝胥 3
鄭州 44
鄭州市 6
鄭州路 42
鄭成功 168
鄭瀾 0
鄭經 29
鄭重 231
鄭重其事 6
鄭重宣佈 4
鄭重聲明 23
鄮 5
鄯 5
鄰 1233
鄰人 37
鄰伍 0
鄰國 106
鄰境 1
鄰室 1
鄰家 47
鄰居 752
鄰接 90
鄰村 5
鄰舍 35
鄰裡 0
鄰角 0
鄰近 753
鄰近地區 90
鄰邊 2
鄰邦 16
鄰里 208
鄰里長 0
鄰長 69
鄱 11
鄱陽 7
鄱陽湖 10
鄱陽盆地 3
鄲 13
鄳 52
鄴 39
鄵 78
鄶 73
鄷 0
鄸 32
鄹 10
鄺 116
鄻 2
鄼 0
鄽 0
鄾 2
鄿 4
酀 0
酁 7
酃 26
酄 6
酅 1
酆 48
酇 23
酈 134
酈 134
酈希誠 13
酈道元 13
酉 376
酉堊紀 0
酊 78
酊 78
酋 103
酋長 166
酋長國 0
酌 672
酌加 8
酌情 64
酌情處理 0
酌情處理 0
酌減 18
酌議 1
酌辦 0
酌酒 1
酌量 62
酌量 62
配 5869
配上 245
配上來 0
配上去 0
配不上 15
配了 13
配享千秋 0
配件 1723
配作 27
配偶 855
配偶欄 1
配備 2283
配出 13
配出來 0
配分 31
配分法 0
配到 16
配制 14
配制飼料 0
配合 11867
配合著 68
配合起來 3
配售 77
配器 35
配在 12
配在一起 3
配套 216
配子 17
配子體 16
配對 500
配對法 0
配得 52
配得 133
配成 20
配戲 0
配戴 251
配房 0
配料 146
配方 565
配有 102
配樂 424
配樂師 0
配法 13
配發 85
配的 134
配眼鏡 16
配種 119
配第 1
配給 49
配給卡 0
配給品 0
配給票 0
配給證 0
配給量 0
配置 1402
配置圖 256
配色 114
配色盤 0
配菜 14
配著 21
配藥 15
配藥師 0
配裝 24
配製 147
配襯 2
配角 334
配角 334
配調 0
配調法 0
配起 0
配起來 3
配軍 4
配配 5
配量 4
配鎖 8
配電 352
配電盤 82
配音 238
配音員 78
配額 464
酎 7
酏 6
酏 6
酐 38
酐 38
酑 0
酒 17836
酒仙 15
酒令 1
酒保 100
酒力 6
酒友 5
酒吧 407
酒吧女 0
酒吧間 4
酒品 0
酒器 17
酒囊飯桶 0
酒囊飯袋 1
酒坊 28
酒場 11
酒場 11
酒壺 18
酒女 1
酒客 35
酒宴 8
酒家 117
酒家女 0
酒市 0
酒席 36
酒店 1120
酒店業 0
酒廠 231
酒後 138
酒後吐真言 1
酒後失態 2
酒後失言 0
酒徒 2
酒意 11
酒拳 0
酒旗 1
酒會 222
酒杯 127
酒架 1
酒桶 24
酒槽鼻 0
酒樓 73
酒櫃 23
酒母 0
酒氣 9
酒氣沖天 0
酒氣熏人 0
酒氣衝天 0
酒池肉林 2
酒泉 5
酒泉街 21
酒渦 2
酒湯 1
酒滓 0
酒狂 17
酒瓶 44
酒瘋 2
酒癖 0
酒癮 40
酒盃 1
酒盅 1
酒盞 0
酒石 13
酒石酸 31
酒神 25
酒神節 2
酒禁 1
酒窩 17
酒窩兒 0
酒窩兒 0
酒精 669
酒精中毒 16
酒精噴燈 0
酒精燈 8
酒精類 6
酒糟 8
酒缸 19
酒罈 1
酒翁 0
酒肆 8
酒肉 20
酒興 1
酒舖 5
酒色 15
酒色之徒 2
酒色財氣 3
酒花 0
酒荒 0
酒菜 29
酒藥 2
酒袋 0
酒話 1
酒足飯飽 8
酒逢知己 1
酒逢知己千杯少 1
酒逢知己千鍾少 1
酒過三巡 4
酒酣耳熱 16
酒醉 119
酒釀 11
酒釀湯圓 1
酒量 57
酒錢 4
酒類 174
酒食 19
酒飯 9
酒餚 2
酒館 81
酒鬼 87
酒麴 1
酒齇鼻 0
酓 4
酔 0
酕 233
酖 33
酖 33
酖毒 0
酗 4594
酗酒 108
酗酒鬧事 2
酘 82
酙 0
酚 354
酚酒 3
酚醛 9
酚醛塑料 1
酚醛塑膠 0
酚醛樹脂 11
酞 1
酟 5
酠 6
酡 6
酡紅 6
酢 10
酢漿草 33
酢醬草 16
酣 51
酣娛 0
酣戰 1
酣暢 11
酣樂 0
酣歌 0
酣眠 4
酣睡 12
酣臥 0
酣興 0
酣醉 3
酣飲 0
酤 6
酤酒 0
酥 469
酥油 13
酥油草 0
酥糖 5
酥胸 18
酥脆 26
酥脆可口 0
酥軟 7
酥軟軟 0
酥餅 19
酥鬆 2
酥鬆可口 0
酥麻 7
酥麻麻 1
酧 0
酨 11
酩 33
酩 33
酩酊 9
酩酊大醉 5
酪 195
酪 195
酪梨 1
酪氨酸 2
酪漿 0
酪農場 1
酪農場 1
酪農業 3
酪酸 1
酪餅 0
酫 0
酬 268
酬勞 134
酬報 4
酬庸 15
酬神 14
酬答 4
酬謝 45
酬酢 4
酬金 66
酭 0
酮 291
酯 664
酯化 26
酯類 27
酰 0
酲 176
酳 106
酴 402
酵 340
酵母 90
酵母菌 53
酵粉 0
酵素 840
酵解 1
酶 0
酷 6245
酷似 69
酷冷 0
酷刑 37
酷到 6
酷吏 5
酷哥 253
酷好 2
酷妹 64
酷寒 18
酷愛 76
酷暑 29
酷烈 1
酷熱 55
酷肖 2
酷虐 0
酷音 1000
酸 3366
酸化 49
酸味 48
酸奶 17
酸度 46
酸式鹽 2
酸性 411
酸性反應 5
酸敗 3
酸根 2
酸梅 24
酸梅湯 30
酸楚 11
酸洗 39
酸溜溜 5
酸澀 10
酸甜 26
酸甜苦辣 175
酸疼 7
酸痛 192
酸痛不已 0
酸腐 1
酸苦 11
酸菜 44
酸菜湯 0
酸葡萄 15
酸葡萄心理 3
酸處理 14
酸處理 14
酸軟 5
酸辛 0
酸辣 19
酸辣湯 9
酸辣蝦 1
酸酐 38
酸酸 33
酸酸甜甜 12
酸酸苦苦 0
酸酸鹹鹹 0
酸雨 1
酸類 24
酸鹼 177
酸鹼值 70
酸鹼度 96
酸麻 0
酸鼻 0
酹 327
酺 173
酻 0
酻 0
酼 0
醀 4
醁 12
醂 7
醃 162
醃成 0
醃漬 1
醃漬物 4
醃肉 9
醃菜 3
醃豬肉 0
醃起來 0
醃過 7
醃醬瓜 0
醃魚 5
醃魚肉 0
醄 4
醅 5
醆 3
醇 576
醇化 8
醇厚 20
醇樸 0
醇精 0
醇美 7
醇酒 31
醇醇 3
醇醨 0
醇醪 5
醇釀 1
醇類 25
醇香 6
醈 0
醈 0
醉 833
醉倒 13
醉倒在 2
醉到 1
醉客 10
醉得 10
醉心 29
醉心於 22
醉意 21
醉態 3
醉漢 55
醉熏熏 0
醉生夢死 8
醉眼惺忪 1
醉翁 11
醉翁之意 1
醉翁之意不在酒 7
醉臥 20
醉薰薰 1
醉過 5
醉過方知酒濃 4
醉鄉 8
醉酒 32
醉醉 3
醉醺醺 7
醉鬼 6
醊 6
醋 290
醋勁 6
醋化 0
醋意 1
醋海生波 6
醋海生波 6
醋罈 0
醋罈子 9
醋罐子 1
醋酸 405
醋酸纖維 14
醋酸鹽 10
醌 0
醍 10
醍醐 9
醍醐灌頂 6
醎 0
醏 10
醐 4
醑 3
醒 1003
醒世 22
醒世名言 0
醒世姻緣 3
醒世箴言 0
醒來 411
醒吾 74
醒吾商專 31
醒吾技術學院 1
醒悟 32
醒獅 19
醒目 104
醒眼 1
醒著 47
醒覺 31
醒過來 77
醒酒 4
醒醒 34
醓 6
醔 0
醔 0
醕 0
醖 0
醖 0
醘 0
醙 79
醚 159
醛 164
醛類 19
醜 356
醜丫頭 0
醜事 4
醜人 3
醜人多作怪 1
醜八怪 7
醜劇 0
醜化 39
醜名 0
醜小鴨 45
醜惡 46
醜態 10
醜態畢露 0
醜態百出 3
醜樣 0
醜死 0
醜聞 150
醜聲四溢 0
醜臉 2
醜行 3
醜行 3
醜詆 1
醜語 0
醜醜 45
醜陋 104
醜類 3
醝 35
醞 30
醞成 0
醞積 0
醞藉 0
醞釀 322
醞釀著 11
醟 49
醠 16
醡 56
醡漿草 0
醢 62
醣 219
醣化 3
醣尿病 1
醣廠 0
醣果 0
醣槭 0
醣水 3
醣漿 0
醣精 0
醣蜜 0
醣衣 0
醣醋 0
醣類 149
醥 5
醦 0
醧 3
醨 7
醩 0
醪 17
醫 34000
醫事 1389
醫事職業 0
醫事職業學校 6
醫務 408
醫務兵 1
醫務室 28
醫士 1
醫好 35
醫學 11841
醫學博士 115
醫學史 17
醫學士 92
醫學界 156
醫學系 1166
醫學院 3945
醫官 26
醫師 12997
醫師公會 100
醫師法 46
醫德 23
醫方 39
醫書 13
醫治 308
醫理 8
醫生 2383
醫界 276
醫病 151
醫療 13642
醫療用具 3
醫療費 45
醫科 1761
醫者 48
醫葯 22
醫藥 3090
醫藥品 80
醫藥學 25
醫藥罔效 0
醫藥費 36
醫術 63
醫護 827
醫護兵 1
醫道 300
醫錯 0
醫院 12616
醬 796
醬園 37
醬料 1
醬汁 1
醬油 243
醬油膏 11
醬瓜 5
醬紫 0
醬紫色 0
醬缸 8
醬缸氣 0
醬肉 4
醬色 0
醬菜 15
醬鴨 1
醭 2
醭 2
醭兒 1
醮 116
醯 338
醯 338
醯胺 1
醰 7
醱 612
醱酵 143
醱醅 0
醲 32
醲郁 0
醳 11
醴 600
醵 11
醶 0
醷 5
醹 8
醺 21
醺蒸 0
醺酣 0
醺醺 1
醺醺然 0
醺魚 0
醻 0
醼 2
醽 139
醾 159
醿 0
釀 181
釀成 106
釀母 0
釀禍 1
釀蜜 1
釀製 48
釀造 87
釀造出來 2
釀造品 0
釀酒 67
釀酒廠 13
釀酒業 1
釁 26
釁端 0
釁隙 0
釁鼓 0
釂 73
釃 6
釃 6
釄 0
釅 8
釆 456
釆 456
采 180
采 7300
采地 1
采女 1
采聲 5
采色 0
采衣 1
采邑 2
采邑 2
釉 414
釉彩 6
釉色 11
釉藥 162
釉陶 4
釋 3044
釋例 72
釋俘 0
釋免 0
釋典 1
釋出 574
釋去重負 0
釋夢 3
釋奠 3
釋尊 48
釋憲 0
釋懷 0
釋手 2
釋放 839
釋放出來 59
釋教 8
釋文 20
釋然 16
釋疑 148
釋義 107
釋股 0
釋言 4
釋迦 131
釋迦牟尼 34
釋迦牟尼佛 118
釋道 14
里 3771
里人 49
里加 20
里名 6
里屋 2
里巷 7
里帶 6
里幹事 51
里弦 0
里拉 71
里斯 349
里斯本 122
里昂 52
里民 119
里民大會 25
里港 216
里程 265
里程碑 176
里程表 20
里程錶 2
里約 57
里約熱內盧 46
里肌肉 19
里脊 3
里邊 14
里長 410
里間 4
里閭 0
重 13021
重 13021
重九 23
重九登高 0
重人 21
重任 119
重作 32
重作馮婦 1
重來 76
重修 207
重修舊好 7
重做 30
重傷 296
重傷風 0
重元素 12
重兵 21
重典 19
重刑 48
重利 9
重到 6
重刷 2
重創 117
重劃 666
重力 280
重力加速 0
重力加速度 15
重力勢能 0
重力場 11
重力場 11
重印 25
重印書目 0
重友誼 0
重友誼 0
重合 30
重名 1
重唱 50
重商主義 10
重回 304
重圍 20
重地 173
重地 294
重型 159
重塗 0
重外孫女 0
重外孫子 0
重外孫子 0
重大 3734
重大事件 55
重大事故 42
重大新聞 19
重大過失 28
重奏 148
重婚 10
重婚罪 3
重孝 4
重孫 0
重孫女 0
重學 8
重寫 67
重寶 96
重工 135
重工業 68
重工輕商 1
重建 1576
重建家園 19
重建為 7
重彈 12
重彈老調 0
重得 9
重得 295
重復 50
重心 596
重心點 0
重情 22
重慶 1164
重慶市 51
重懲 13
重抄 2
重拍 15
重拍 15
重拾 121
重挫 430
重振 99
重排 34
重排本 1
重提 45
重提舊事 0
重摔 2
重摺 1
重播 230
重播 230
重播節目 1
重播節目 1
重擊 50
重操舊業 3
重擔 102
重敘舊好 0
重整 504
重整家園 6
重整旗鼓 5
重數 1
重數 1
重文輕武 2
重新 6371
重新做人 5
重新路 462
重於 198
重於泰山 4
重施故技 4
重映 6
重晶石 4
重機械 55
重機槍 0
重權勢 0
重武器 1
重武輕文 0
重氫 1
重氮化 10
重水 30
重油 46
重洋 4
重活 0
重活 0
重溫 115
重溫舊夢 26
重演 159
重灌 0
重炮 2
重為 36
重版 1
重物 69
重犯 9
重現 393
重生 302
重用 74
重申 882
重申立場 5
重申立場 5
重申觀點 0
重男輕女 25
重當 1
重疊 404
重疊式 4
重病 131
重發 20
重的 294
重石 0
重砲 32
重稅 12
重組 419
重編目 0
重編頁碼 0
重罪 21
重置 5000
重置成本 9
重置費用 0
重罰 87
重義氣 8
重義輕利 0
重考 164
重考生 49
重耳 9
重聽 45
重臂 2
重臂 2
重臣 14
重裘 0
重裝 130
重裝備 14
重製 6371
重複 1448
重複使用 172
重複式 3
重複本 0
重複法 1
重複記錄 2
重要 14673
重要性 1382
重要關鍵 86
重要關頭 8
重見 13
重見光明 9
重見天日 21
重視 3981
重託 6
重設 47
重設鍵 0
重話 15
重譯 2
重讀 144
重負 5
重責 23
重責大任 52
重貼現率 140
重賞 21
重賞之下 4
重賞之下必有勇夫 1
重質不重量 0
重走 3
重起爐灶 0
重跌 20
重踏 4
重踏 4
重蹈覆轍 39
重載 10
重轟炸機 3
重辦 33
重農主義 0
重返 305
重述 26
重逢 108
重遊 16
重遊舊地 4
重遷 0
重選 17
重選一次 1
重選一次 1
重酬 0
重重 282
重重包圍 8
重重困難 7
重重圍住 0
重重疊疊 6
重重的 106
重重地 106
重重障礙 0
重重難關 8
重量 2491
重量單 0
重量級 243
重量計 7
重金 59
重金屬 464
重金禮聘 16
重鎚 4
重鎮 219
重陽 313
重陽節 28
重障 8
重音 14
重音符號 0
重頭 65
重頭 65
重頭戲 95
重點 5266
重點式 28
重點是 190
野 4323
野人 142
野人族 0
野人獻曝 32
野兔 46
野勁兒 0
野勁兒 0
野叟 3
野叟曝言 1
野台 53
野台戲 71
野史 26
野合 1
野味 31
野味十足 0
野地 52
野地訓練 0
野外 442
野外求生 28
野孩子 244
野孩子 244
野宴 2
野山羊 0
野心 159
野心人士 0
野心勃勃 31
野心家 15
野性 43
野性未脫 0
野性難馴 1
野戰 69
野戰砲 1
野戰軍 1
野戰部隊 4
野戰醫院 5
野手 32
野果 5
野果子 0
野果子 0
野柳 231
野柳鄉 0
野渡 21
野漢子 0
野火 25
野火燒不盡 3
野營 63
野營地 5
野營隊 5
野牛 45
野牛群 1
野犬 13
野狗 90
野狗群 4
野狼 164
野獸 229
野獸派 28
野生 653
野生動物 1688
野生植物 33
野百合 35
野砲 1
野砲營 0
野祭 0
野禽 1
野種 1
野老 2
野臺戲 5
野色 2
野芋 0
野花 68
野草 43
野草堆 0
野菊 16
野菜 79
野葛 1
野葡萄 1
野葬 0
野薑花 1
野蘭 0
野蘭花 2
野蠻 100
野蠻人 24
野豬 55
野貓 61
野遊 4
野雁 90
野雞 7
野雞大學 4
野雞車 8
野食 2
野餐 76
野餐盒 2
野馬 76
野鳥 628
野鳥群 0
野鴛鴦 0
野鴨 29
野鴿 67
野鶴 6
野鹿 10
野麥 1
野鼠 9
量 15062
量 15062
量了 11
量入為出 9
量具 36
量出 33
量刑 27
量到 14
量力 7
量力而為 26
量力而行 1
量化 252
量器 22
量回 2
量回來 0
量回去 0
量地 33
量子 190
量子力學 40
量子學說 0
量子論 10
量完 2
量度 83
量度 83
量成 88
量才錄用 0
量杯 18
量油尺 0
量淺 0
量測 1
量熱器 1
量產 890
量程 1
量窄 1
量筒 29
量綱 0
量著 31
量表 298
量規 17
量角 3
量角器 9
量角規 0
量詞 19
量變 29
量販 0
量販店 0
量過 37
量過來 0
量過去 0
量量 16
量量看 8
量體裁衣 0
釐 311
釐 1
釐出 8
釐定 58
釐正 7
釐清 472
釐米 37
釐訂 67
金 33812
金鑰 100
金不換 24
金丹 250
金主 100
金井 1
金人 23
金人緘口 1
金像獎 147
金價 74
金元 68
金元國 0
金元王國 0
金光 202
金光閃閃 32
金光黨 11
金兵 1
金冠 38
金剛 613
金剛不壞之身 1
金剛不壞之身 1
金剛力士 1
金剛夜叉明王 1
金剛心菩薩 1
金剛怒目 1
金剛智 1
金剛杵 1
金剛界 1
金剛石 6
金剛砂 7
金剛經 93
金剛薩埵 1
金剛鑽 10
金剛頂經 1
金創 1
金創藥 0
金劍 9
金匠 39
金匱 1
金匱石室 3
金匱要略 1
金口 28
金口木舌 1
金口玉言 1
金合歡 1
金吾 1
金和 141
金城 448
金城湯池 0
金塊 13
金夏沙 1
金太宗 0
金太祖 1
金婚 4
金子 46
金字 31
金字塔 292
金字招牌 82
金寶 435
金小蜂 0
金屋 54
金屋藏嬌 3
金屬 4370
金屬主義 0
金屬元素 17
金屬光澤 19
金屬加工 59
金屬器皿 0
金屬圈 1
金屬型鑄造 0
金屬工藝 1
金屬性 6
金屬材料 166
金屬板 24
金屬桿 4
金屬棒 9
金屬模 8
金屬片 16
金屬環 10
金屬疲勞 1
金屬礦床 2
金屬絲 6
金屬線 22
金屬罐 7
金屬腐蝕 2
金屬膜 8
金屬貨幣 0
金屬鈉 9
金屬銻 0
金屬鍵 2
金屬陶瓷 1
金山 647
金山鄉 213
金崙溪 3
金川 1
金工 59
金幣 132
金店 51
金庫 164
金庸 1615
金彈 0
金戒 1
金手套獎 1
金援 10
金文 27
金斗 3
金日成 12
金星 245
金曜日 0
金書鐵券 1
金木水火土 6
金本位 3
金本位制 0
金東 30
金枝玉葉 13
金柑 1
金條 13
金棗 41
金榜 141
金榜題名 58
金樽 1
金樽旨酒 1
金橘 3
金櫃 2
金權政治 24
金殿 13
金氰化鉀 1
金水 62
金沙 102
金沙江 7
金沙灘 1
金沙薩 0
金波 20
金波 20
金波玉液 1
金漆 1
金漆馬桶 1
金漿玉醴 1
金烏玉兔 0
金牌 534
金牌獎 103
金牌虎符 1
金牙 3
金牛 329
金牛座 827
金獎 149
金玉 118
金玉其外 6
金玉滿堂 11
金玉良緣 0
金玉良言 21
金珠 82
金瓜 16
金瓜石 86
金瓶梅 98
金甌 37
金甌女中 1
金甌女子高級中學 1
金甌無缺 1
金甌藏名 1
金田起義 1
金甲 1
金甲利兵 0
金甲蟲 0
金瘡 1
金瘡小草 1
金瘡藥 1
金盞 21
金盞草 0
金盞銀臺 1
金相 44
金相學 1
金相玉式 1
金相玉質 1
金石 387
金石不渝 1
金石之盟 1
金石之言 1
金石交 0
金石例 1
金石學 2
金石文 1
金石文字記 1
金石為開 6
金石絲竹 1
金石至交 1
金石良言 1
金石萃編 1
金石錄 1
金砂 31
金碧山水 1
金碧熒煌 1
金碧輝煌 40
金磚 5
金礦 48
金科玉律 15
金科玉條 1
金窩銀窩不如自己的狗窩 1
金童玉女 11
金筆 24
金箍 2
金箍棒 8
金箔 20
金箔紙 1
金粉 27
金精 1
金紅 104
金絲 53
金絲桃 1
金絲燕 1
金絲猴 3
金絲猿 0
金絲絨 0
金絲貓 1
金絲雀 20
金絲鳥 0
金線蟲 0
金縷衣 2
金翅擘海 1
金聖歎 1
金聲 40
金聲玉振 1
金腿 1
金臺 1
金舌弊口 1
金色 625
金花 53
金花菜 0
金華 406
金華殿語 1
金華縣 1
金葉 27
金蓮 59
金蓮步 1
金蓮花 1
金蘭 77
金蘭灣 0
金蛇 1
金蛋 17
金融 14311
金融中心 181
金融卡 100
金融寡頭 1
金融業 508
金融界 87
金融赤字 1
金蟬 10
金蟬脫殼 33
金蟲 1
金蟾 1
金衡 1
金衡制 0
金衣玉食 1
金表 8
金袍 0
金裝 1
金言 15
金谷墮樓 1
金貂 1
金貂換酒 1
金質 68
金身 36
金身羅漢 1
金車 255
金迷粉醉 1
金迷紙醉 0
金邊 46
金邊眼鏡 5
金都 33
金針 86
金針度人 1
金針花 1
金針菇 1
金針菜 8
金釧 1
金釵 21
金釵十二 1
金鈴 12
金鈴子 0
金鈿 2
金銀 122
金銀島 62
金銀珠寶 77
金銀箔 1
金銀花 10
金銀財寶 12
金銀銅鐵 4
金銅礦物 0
金銅礦物局 0
金鋼 69
金錢 1249
金錢板 1
金錢棒 1
金錢棒舞 1
金錢萬能 2
金錢豹 11
金錢遊戲 28
金錶 9
金鎊 0
金鎖記 1
金鏡 1
金鐘 112
金鐘兒 1
金鐘獎 58
金鐘罩 15
金鐘鏟 1
金鐲 2
金鐲子 0
金鑼 1
金鑾殿 1
金門 2097
金門島 11
金門縣 775
金門馬祖 41
金陀萃編 1
金陵 126
金雀花 7
金雞 50
金雞獨立 1
金雞納 2
金雞納樹 3
金雞納霜 0
金雞納鹼 0
金額 13660
金風 15
金飯碗 7
金飾 129
金馬 848
金馬地區 54
金馬獎 411
金髮 81
金髮女郎 7
金髮碧眼 6
金鬥 0
金魚 417
金魚缸 4
金魚藻 6
金鰲玉蝀 1
金鳳花 2
金黃 132
金黃色 195
金黃藻 1
金黴素 2
金鼎獎 226
金鼓雷鳴 1
金龍 467
金龍隊 0
金龜 19
金龜婿 4
金龜子 14
金龜車 27
釓 393
釔 527
釕 3
釕 3
釖 0
釗 154
釘 568
釘 568
釘上 20
釘上來 0
釘上去 0
釘下 0
釘下來 0
釘下去 1
釘了 10
釘人 4
釘住 17
釘來釘去 0
釘個 0
釘入 5
釘到 4
釘合 0
釘問 0
釘在 68
釘好 2
釘子 67
釘尖 0
釘床 4
釘書機 27
釘書針 6
釘梢 1
釘死 30
釘牢 1
釘痕 1
釘緊 1
釘腳 0
釘著 12
釘進 0
釘進來 0
釘進去 0
釘過 0
釘過來 0
釘過去 0
釘鈕釦 0
釘錘 4
釘鎚 2
釘鞋 23
釘頭 5
釘頭 5
釙 10
釚 2
釛 0
釜 43
釜中魚 0
釜山 121
釜底抽薪 15
針 1955
針刺 65
針刺麻醉 0
針劑 18
針孔 82
針孔照像 0
針孔照像機 0
針對 8550
針對於 14
針對著 13
針尖 16
針工 5
針布 13
針形葉 0
針梳機 0
針氈 1
針法 26
針灸 403
針灸 403
針狀 28
針狀物 0
針盤 3
針眼 11
針眼子 0
針砭 16
針筒 69
針箍 0
針線 32
針線包 3
針線活 0
針線盒 0
針織 666
針織品 41
針織廠 58
針織料 1
針腳 17
針芒 0
針葉 38
針葉林 31
針葉樹 33
針鋒 4
針鋒相對 64
針頭 101
針骨 0
針黹 2
針鼴 0
針鼻 0
釞 0
釟 0
釠 0
釠 0
釢 0
釣 1965
釣上 21
釣上來 2
釣上去 0
釣了 10
釣個 2
釣具 126
釣具店 21
釣到 32
釣勾 0
釣去 0
釣名沽譽 0
釣場 53
釣場 53
釣好 0
釣客 31
釣徒 0
釣得 11
釣得 11
釣竿 67
釣絲 1
釣線 17
釣船 31
釣蝦 8
釣蝦場 20
釣蝦場 20
釣走 1
釣起 15
釣起來 1
釣釣 5
釣釣魚 0
釣鉤 7
釣餌 13
釣魚 760
釣魚台 172
釣魚台列島 3
釣魚用具 4
釣魚臺列島 0
釤 7
釥 0
釦 1678
釦上 0
釦上來 0
釦上去 0
釦下 0
釦下來 0
釦下去 0
釦了 0
釦在 0
釦好 0
釦子 100
釦緊 0
釦著 0
釦起 0
釦起來 0
釧 301
釨 2
釩 1243
釪 4
釫 4
釬 668
釭 281
釭 281
釮 4
釯 0
釰 0
釱 93
釲 0
釳 4
釴 131
釵 4066
釵鐶 0
釶 0
釷 91
釸 7
釹 27
釽 5
釾 0
釿 9
鈀 31
鈀 31
鈁 120
鈂 6
鈃 2
鈄 5
鈅 6
鈆 6
鈇 6
鈈 0
鈈 0
鈉 737
鈉離子 34
鈉鹽 72
鈊 12
鈋 0
鈌 3
鈍 176
鈍兵 0
鈍刀 1
鈍劣 0
鈍化 23
鈍化劑 0
鈍學累功 0
鈍性 4
鈍挫 1
鈍氣 5
鈍角 2
鈍角三角形 0
鈍鈍 5
鈎 0
鈏 10
鈐 36
鈐印 3
鈐記 4
鈑 169
鈒 5
鈓 0
鈓 0
鈔 260
鈔券 7
鈔本 10
鈔票 180
鈔錄 0
鈕 982
鈕子 3
鈕孔 0
鈕扣 82
鈕釦 77
鈖 7
鈗 4
鈘 0
鈙 6
鈚 5
鈛 0
鈜 15
鈝 0
鈞 1541
鈞啟 8
鈞安 1
鈞諭 0
鈞鑒 3
鈟 0
鈠 0
鈢 0
鈣 1941
鈣化 56
鈣片 33
鈣質 181
鈣鎂磷肥 0
鈤 10
鈥 4
鈥 4
鈦 459
鈦合金 1
鈦酸鋇 9
鈧 3
鈭 129
鈮 116
鈯 0
鈰 62
鈱 58
鈲 24
鈳 35
鈴 5700
鈴子 11
鈴聲 73
鈴聲響起 5
鈴蟲 0
鈴鈴 41
鈴鐺 22
鈴鐺聲 0
鈴鐺響 0
鈴響 20
鈴響後 1
鈴響時 5
鈴鼓 7
鈵 0
鈶 95
鈷 139
鈷 139
鈸 57
鈸 57
鈹 58
鈹 58
鈹青銅 0
鈺 2375
鈻 0
鈼 0
鈽 50
鈾 153
鈾 153
鈾礦 4
鈿 129
鈿 129
鉀 411
鉀礦 2
鉀肥 20
鉀鹽 5
鉁 0
鉂 0
鉃 0
鉃 0
鉄 0
鉅 1274
鉅億 0
鉅儒 0
鉅大 51
鉅子 28
鉅富 4
鉅款 17
鉅細靡遺 21
鉅著 24
鉅變 11
鉅費 0
鉅量 6
鉅額 238
鉆 99
鉇 0
鉇 0
鉇 0
鉈 34
鉈 34
鉈 34
鉉 66
鉊 66
鉋 127
鉋冰 0
鉋刀 2
鉋削 6
鉋子 1
鉋成 0
鉋花 4
鉌 81
鉍 109
鉎 94
鉏 57
鉐 32
鉑 87
鉑銥 4
鉒 61
鉓 76
鉔 300
鉕 0
鉕 0
鉖 62
鉗 205
鉗刑 0
鉗制 6
鉗口 1
鉗口結舌 0
鉗噤 0
鉗子 21
鉗工 43
鉗形攻勢 0
鉗梏 0
鉘 0
鉙 0
鉚 60
鉚釘 121
鉛 996
鉛中毒 32
鉛丸 0
鉛丹 1
鉛丹鑤 0
鉛印 6
鉛垂線 0
鉛塊 5
鉛字 7
鉛字盤 1
鉛室法 1
鉛山 3
鉛板 6
鉛條 0
鉛模 0
鉛汞 5
鉛版 1
鉛球 30
鉛直 9
鉛直線 0
鉛礦 1
鉛筆 381
鉛筆尖 0
鉛筆畫 15
鉛筆盒 42
鉛筆鑤 0
鉛管 13
鉛管匠 0
鉛粉 4
鉛絲 2
鉛膏 0
鉛華 14
鉛錘 7
鉛黃 0
鉛黛 0
鉜 0
鉝 0
鉞 268
鉟 0
鉟 0
鉠 41
鉡 95
鉢 0
鉢 0
鉣 36
鉤 556
鉤上 3
鉤上來 0
鉤上去 0
鉤下 0
鉤下來 0
鉤下去 0
鉤中 0
鉤了 0
鉤住 10
鉤來鉤去 0
鉤出 3
鉤出來 0
鉤到 0
鉤勒 2
鉤勒出 1
鉤勒法 0
鉤去 1
鉤在 1
鉤子 14
鉤心鬥角 2
鉤棍 0
鉤深致遠 1
鉤爪 3
鉤狀 5
鉤玄 1
鉤玄提要 0
鉤破 0
鉤索 2
鉤花點葉 0
鉤花點葉派 0
鉤蟲 15
鉤蟲病 3
鉤起 1
鉤過 0
鉤過來 0
鉤過去 0
鉤針 19
鉤鉤 1
鉥 88
鉦 223
鉧 160
鉨 0
鉩 0
鉩 0
鉪 0
鉪 0
鉫 0
鉬 140
鉭 56
鉭 56
鉮 0
鉯 50
鉰 38
鉲 59
鉵 0
鉵 0
鉶 9
鉸 336
鉸刀 4
鉸接 2
鉸鍊 16
鉸鏈 12
鉸鏈葉 0
鉹 117
鉺 28
鉻 788
鉻酸鉀 0
鉻鋼 4
鉼 328
鉽 20
鉾 223
鉿 76
銀 6640
銀價 3
銀元 41
銀光 28
銀光閃閃 3
銀兔 9
銀兩 35
銀冠 2
銀券 0
銀匠 6
銀器 13
銀圈 1
銀圓 0
銀團 1
銀婚 0
銀婚紀念 0
銀子 223
銀山 11
銀川 28
銀川市 4
銀帶 9
銀帶獎 0
銀幕 222
銀幕偶像 2
銀幣 58
銀座 55
銀座區 0
銀庫 0
銀彈 21
銀彈攻勢 4
銀戒 1
銀杏 85
銀杯 3
銀板 0
銀根 167
銀根緊 0
銀框 0
銀樓 763
銀樺 7
銀樽 0
銀櫃 7
銀母 0
銀河 889
銀河倒瀉 1
銀河塵 0
銀河系 154
銀洋 0
銀漆 1
銀漢 7
銀火 1
銀灰 11
銀灰色 26
銀燈 5
銀燭 7
銀片 0
銀牌 123
銀牌獎 67
銀牙 0
銀珠冰片 0
銀珠冰片粉 0
銀甲 4
銀白 31
銀白色 37
銀盾 0
銀硃 1
銀礦 5
銀票 10
銀竹 2
銀箔 2
銀箔紙 0
銀粉 32
銀粒 6
銀粟 0
銀紅 9
銀紅色 2
銀紙 23
銀耳 11
銀色 316
銀花 17
銀行 14560
銀行信用 22
銀行借款 20
銀行利息 8
銀行券 3
銀行團 386
銀行存摺 7
銀行存款 93
銀行家 34
銀行業 251
銀行界 63
銀行費用 1
銀行資本 15
銀角 0
銀貨兩訖 0
銀質 24
銀邊 3
銀邊眼鏡 0
銀針 27
銀釧 12
銀釵 0
銀錠 10
銀錢 26
銀離子 4
銀髮 62
銀髮族 180
銀鹽 7
銀黃 1
銀鼠 4
銁 0
銂 45
銃 83
銄 0
銄 0
銅 2582
銅人 18
銅像 66
銅元 1
銅兵 1
銅匠 0
銅印 2
銅器 144
銅器時代 5
銅塊 1
銅壺 0
銅子兒 0
銅子兒 0
銅山 26
銅山鐵壁 0
銅幣 10
銅廠 3
銅拔 0
銅板 86
銅柱 8
銅條 11
銅模 2
銅油 2
銅漏 0
銅牆鐵壁 19
銅片 15
銅版 19
銅版印刷 7
銅版紙 31
銅牌 268
銅牙 1
銅礦 30
銅筋鐵骨 2
銅管 181
銅管樂器 12
銅管樂隊 4
銅絲 10
銅綠 10
銅線 93
銅臭 4
銅臭味 3
銅臭氣 0
銅製 50
銅製品 41
銅褐色 1
銅鈴 21
銅錢 14
銅鏡 29
銅鐘 27
銅鑼 338
銅鑼燒 0
銅青 0
銅鼓 5
銆 44
銇 81
銈 98
銈 98
銉 0
銊 44
銋 132
銌 75
銍 10
銎 29
銏 0
銐 0
銐 0
銑 288
銑刀 39
銑削 1
銑削 1
銑工 1
銑床 179
銑鋧 1
銑鐵 4
銒 0
銒 0
銒 0
銓 1025
銓列 1
銓定考試 1
銓廷 1
銓擢 0
銓敘 106
銓敘部 289
銓次 1
銓考 0
銓衡 0
銓選 0
銓部 1
銔 7
銕 45
銖 367
銖兩 0
銖兩悉稱 1
銖兩悉稱 1
銖積寸累 0
銖銖校量 1
銗 353
銘 7060
銘佩 0
銘傳商專 11
銘傳大學 1
銘傳女子商業專科學校 1
銘傳管理學院 1
銘刻 12
銘德新村 0
銘心 19
銘心刻骨 6
銘心鏤骨 0
銘感 11
銘感五內 15
銘文 22
銘旌 0
銘牌 14
銘篆 646
銘肌鏤骨 1
銘言 0
銘記 26
銘記在心 15
銘誄 0
銘謝 33
銘謝惠顧 1
銘骨 0
銙 178
銚 58
銚 58
銛 6
銜 565
銜住 1
銜冤 0
銜命 8
銜哀 1
銜尾相隨 0
銜恨 0
銜悲 0
銜憤 0
銜接 352
銜接起來 2
銜枚 0
銜枚疾走 0
銜泣 0
銜環 2
銜環結草 0
銜竿 0
銜結 2
銜鐵 0
銜尾蛇 10
銝 139
銞 0
銟 0
銠 5
銡 31
銢 48
銣 41
銤 3
銥 73
銦 225
銧 114
銨 571
銨鹽 9
銩 112
銪 40
銫 155
銬 20
銬起來 0
銭 0
銭 0
銮 0
銱 0
銲 193
銲住 1
銲工 1
銲接 166
銲槍 3
銲錫 45
銳 2336
銳不可當 14
銳兵 0
銳利 158
銳器 2
銳志 0
銳意改革 0
銳敏 5
銳氣 20
銳減 125
銳眼 1
銳角 11
銳角三角 0
銳角三角形 2
銳進 1
銴 235
銵 191
銶 196
銷 3996
銷了 5
銷來銷去 0
銷假 27
銷出 3
銷去 2
銷售 7374
銷售價格 33
銷售員 25
銷售業 588
銷售網 33
銷售費用 5
銷售量 508
銷售額 246
銷售點 56
銷回 0
銷回來 0
銷回去 0
銷子 0
銷帳 15
銷得 0
銷志 0
銷掉 0
銷案 35
銷毀 108
銷滅 3
銷聲匿跡 21
銷蝕 0
銷貨員 3
銷貨成本 25
銷貨折扣 0
銷貨收入 21
銷貨毛利 21
銷貨運費 2
銷賬 0
銷贓 12
銷路 61
銷過 45
銷量 127
銷金 10
銷金窟 0
銷釘 0
銷銷 1
銷鎔 0
銷項稅額 38
銷魂 56
銷魂窟 0
銸 0
銹 1105
銺 0
銻 47
銻礦 0
銼 35
銼刀 16
銼磨 0
銽 0
銽 0
銽 0
銾 0
銿 0
鋀 10
鋁 2345
鋁合金 519
鋁土 2
鋁土礦 0
鋁條 4
鋁熱劑 0
鋁球 0
鋁礦 1
鋁窗 61
鋁箔 123
鋁箔包 1
鋁箔紙 1
鋁製 116
鋁製品 49
鋁酸鈣 0
鋁錠 22
鋁鍋 6
鋁門 25
鋁門窗 301
鋂 59
鋂 59
鋃 209
鋃鐺 1
鋃鐺入獄 2
鋄 25
鋅 494
鋅板 2
鋅片 2
鋅版 6
鋅白 0
鋅礦 3
鋅粉 43
鋅粒 2
鋅肥 0
鋅華 0
鋅鐵礦 0
鋆 94
鋇 103
鋈 99
鋉 39
鋊 761
鋋 53
鋌 16
鋌 16
鋌而走險 11
鋍 2
鋎 17
鋏 11
鋏子 0
鋐 154
鋑 5
鋒 1908
鋒刃 2
鋒利 37
鋒芒 19
鋒芒內斂 0
鋒芒太露 0
鋒芒畢露 4
鋒銳 15
鋒面 116
鋒面雨 1
鋒頭 34
鋓 4
鋔 0
鋕 41
鋖 0
鋗 370
鋘 13
鋙 13
鋚 0
鋛 0
鋛 0
鋜 0
鋝 166
鋞 13
鋟 7
鋠 11
鋡 8
鋢 0
鋣 0
鋤 47
鋤奸 2
鋤強扶弱 0
鋤犁 0
鋤田 0
鋤草 3
鋤頭 43
鋥 0
鋦 94
鋧 2
鋨 2560
鋩 113
鋪 324
鋪 324
鋪上 33
鋪上來 0
鋪上去 0
鋪下 2
鋪下來 0
鋪下去 0
鋪了 22
鋪位 1
鋪保 0
鋪出 3
鋪出來 0
鋪出去 0
鋪到 1
鋪向 0
鋪在 35
鋪地 15
鋪地板 0
鋪地磚 1
鋪墊 0
鋪好 16
鋪子 15
鋪展 2
鋪展開 0
鋪展開來 0
鋪平 4
鋪床 4
鋪張 2
鋪張揚厲 0
鋪張浪費 2
鋪得 9
鋪成 13
鋪排 6
鋪敘 0
鋪敘法 1
鋪板 0
鋪的 9
鋪砌 2
鋪紙 1
鋪著 16
鋪蓋 15
鋪行 0
鋪設 125
鋪設在 2
鋪設成 0
鋪起 0
鋪起來 0
鋪路 97
鋪軌 2
鋪開 1
鋪陳 100
鋪陳法 0
鋪面 74
鋫 0
鋬 0
鋭 0
鋮 137
鋯 26
鋰 550
鋰電池 1
鋱 23
鋳 0
鋶 0
鋷 6
鋸 307
鋸 307
鋸下 3
鋸下來 0
鋸下去 0
鋸了 4
鋸出 2
鋸切成 6
鋸口 0
鋸大缸 1
鋸好 1
鋸子 26
鋸屑 2
鋸床 16
鋸掉 4
鋸斷 9
鋸木 28
鋸木屑 0
鋸木廠 6
鋸木業 0
鋸末 0
鋸條 0
鋸牙鉤爪 1
鋸琴 1
鋸盆 1
鋸直 0
鋸碗 1
鋸起來 0
鋸過 1
鋸過來 0
鋸過去 0
鋸鋸 2
鋸錯 0
鋸開 2
鋸齒 65
鋸齒形 7
鋸齒狀 45
鋹 17
鋺 10
鋻 1
鋼 5000
鋼刀 27
鋼千 0
鋼印 12
鋼叉 4
鋼坯 2
鋼城 0
鋼料 21
鋼材 265
鋼杯 6
鋼板 571
鋼架 174
鋼梁 1
鋼條 5
鋼樑 32
鋼水 8
鋼片 56
鋼版 6
鋼珠 63
鋼琴 1472
鋼琴伴奏 12
鋼琴協奏 2
鋼琴協奏曲 103
鋼琴家 392
鋼琴曲 30
鋼琴演奏 66
鋼瓶 51
鋼皮 3
鋼盔 9
鋼砲 21
鋼硬 1
鋼種 33
鋼筆 119
鋼筆套 0
鋼筆尖 0
鋼筆桿 0
鋼筆蓋 0
鋼筋 457
鋼筋水泥 35
鋼筋混凝 8
鋼筋混凝土 236
鋼筘 1
鋼箍 0
鋼管 281
鋼索 99
鋼結構 81
鋼絲 56
鋼絲圈 0
鋼絲床 0
鋼絲繩 0
鋼絲鋸 0
鋼線 100
鋼纜 47
鋼號 0
鋼製 115
鋼製品 37
鋼軌 30
鋼都 2
鋼釘 25
鋼針 8
鋼鋸 3
鋼錠 0
鋼鐵 2516
鋼鐵工 0
鋼鐵廠 104
鋼鑿 0
鋼領 3
鋼骨 124
鋼骨傘 0
鋽 0
鋾 6
鋿 6
錀 28
錁 6
錂 5
錃 0
錄 49969
錄上 29
錄上來 1
錄上去 0
錄下 138
錄下來 34
錄下去 0
錄了 63
錄事 8
錄供 11
錄個 6
錄像 0
錄像機 2
錄到 33
錄取 2010
錄取人數 132
錄取分數 100
錄取名單 647
錄取名額 170
錄取率 272
錄在 29
錄影 1110
錄影帶 2382
錄影機 159
錄得 269
錄成 41
錄放影機 0
錄放音機 54
錄歌 5
錄用 81
錄的 268
錄著 2
錄製 440
錄起 1
錄起來 5
錄載 1
錄過 30
錄錄 2
錄錄歌 0
錄錄音 0
錄音 1725
錄音室 162
錄音帶 1737
錄音機 276
錄鬼簿 1
錄點 0
錅 0
錆 16
錇 0
錈 12
錉 5
錊 0
錊 0
錋 6
錌 20
錍 14
錎 14
錏 54
錐 286
錐子 5
錐子 5
錐度 10
錐形 37
錐形瓶 5
錐心 10
錐心刺骨 10
錐心泣血 1
錐心蝕骨 0
錐處囊中 0
錐面 0
錐頭 1
錐體 26
錑 0
錑 0
錒 74
錒 74
錓 3
錔 8
錕 78
錖 3
錗 0
錘 187
錘子 8
錘打 2
錘煉 1
錘鍊 4
錙 4
錙銖 3
錙銖必較 11
錚 264
錚錚 12
錚鏘 0
錛 12
錛得兒木 0
錜 0
錝 34
錞 28
錟 39
錠 408
錠劑 50
錠子 8
錡 159
錢 5206
錢包 189
錢可通神 0
錢坑 12
錢塘江 10
錢奴 0
錢幣 106
錢幣司 1
錢復 557
錢櫃 58
錢櫃雜誌 12
錢票 2
錢箱 1
錢糧 8
錢荒 0
錢莊 46
錢袋 1
錢財 102
錢財乃身外之物 1
錢鋪 2
錣 9
錤 9
錥 15
錦 3654
錦上添花 28
錦匣 0
錦囊 351
錦囊妙計 14
錦字 1
錦州 121
錦帳 3
錦帶 4
錦心繡口 0
錦旗 39
錦標 97
錦標賽 546
錦石 1
錦箋 0
錦綸 11
錦緞 4
錦繡 182
錦繡前程 4
錦繡大地 2
錦繡河山 10
錦衣 34
錦衣玉食 4
錦衣衛 2
錦衾 0
錦雞 2
錧 15
錨 367
錨噴支護 0
錨固 0
錩 237
錪 88
錫 2085
錫克族 0
錫匠 0
錫器 10
錫壺 0
錫杖 2
錫林 14
錫爾河 1
錫石 0
錫礦 5
錫箔 4
錫箔紙 2
錫紙 3
錫罐 1
錫蘭 54
錫蘭人 0
錫蘭島 0
錫金 21
錫金人 0
錬 0
錭 25
錮 27
錮疾 0
錯 232
錯 2372
錯上加錯 3
錯亂 168
錯了 614
錯了又錯 0
錯事 26
錯位 6
錯別字 24
錯到 6
錯失 176
錯失良機 31
錯字 110
錯字連篇 4
錯得 464
錯怪 15
錯愕 74
錯愛 16
錯放 3
錯案 3
錯的 464
錯節 2
錯綜 24
錯綜複雜 87
錯繆 0
錯落 44
錯處 5
錯覺 88
錯覺作用 0
錯認 20
錯誤 5034
錯誤率 18
錯誤百出 18
錯過 1205
錯過來 0
錯過去 0
錯開 28
錯雜 8
錯雜在 0
錯雜著 0
錰 0
錱 0
録 0
錳 325
錳酸鉀 1
錳鋼 0
錳鐵 6
錴 12
錵 11
錶 1233
錶帶 40
錶店 21
錶廠 1
錶殼 31
錶鏈 0
錶面 57
錷 0
錸 498
錹 5
錼 74
錽 0
鍃 0
鍅 0
鍆 13
鍇 64
鍈 0
鍉 49
鍊 5095
鍊丹 3
鍊乳 0
鍊子 20
鍊條 52
鍊結 50
鍊金術 17
鍊鋼 12
鍊鎖 3
鍋 998
鍋兒 0
鍋兒 0
鍋匠 0
鍋台 3
鍋名 1
鍋垢 11
鍋子 38
鍋巴 3
鍋底 13
鍋底朝天 0
鍋爐 457
鍋盔 0
鍋臺 0
鍋蓋 30
鍋貼 19
鍋貼兒 0
鍋貼兒 0
鍌 8
鍍 564
鍍上 6
鍍在 2
鍍成 3
鍍金 162
鍍銀 32
鍍銅 8
鍍鋅 141
鍍錫 15
鍍鐵 0
鍎 32
鍏 32
鍐 9
鍑 7
鍒 79
鍓 0
鍔 125
鍕 107
鍖 50
鍗 8
鍗 8
鍘 38
鍘刀 0
鍙 187
鍚 338
鍛 205
鍛件 23
鍛壓 24
鍛工 5
鍛接 0
鍛模 7
鍛煉 21
鍛燒 7
鍛燒成 0
鍛燒爐 0
鍛爐 0
鍛石 0
鍛造 182
鍛造工 0
鍛造法 2
鍛錘 1
鍛鍊 163
鍛鍊出 8
鍛鐵 2
鍜 23
鍝 0
鍞 14
鍟 0
鍠 36
鍡 22
鍢 0
鍣 38
鍤 49
鍥 26
鍥而不捨 31
鍦 0
鍧 18
鍨 0
鍩 0
鍩 0
鍪 4
鍫 0
鍫 0
鍬 399
鍬形蟲 0
鍭 9
鍮 0
鍯 0
鍰 193
鍱 50
鍲 0
鍳 0
鍳 0
鍴 0
鍵 5165
鍵位 0
鍵入 1356
鍵控 4
鍵槽 7
鍵盤 1576
鍵盤樂器 28
鍵豎琴 0
鍶 103
鍷 33
鍸 0
鍹 4
鍺 48
鍺晶體 0
鍻 12
鍼 70
鍼黹 0
鍽 0
鍾 3081
鍾子期 3
鍾山 10
鍾情 95
鍾愛 90
鍾縣 0
鍾繇 18
鍾錶 6
鍾離 6
鍾靈毓秀 6
鍾馗 50
鎀 35
鎁 0
鎂 362
鎂光 4
鎂光燈 25
鎂合金 39
鎂砂 0
鎂磚 0
鎃 56
鎄 0
鎈 3
鎉 0
鎊 90
鎋 0
鎌 86
鎌倉幕府 3
鎍 33
鎎 3
鎏 14
鎐 0
鎐 0
鎑 31
鎒 6
鎓 0
鎔 330
鎔劑 0
鎔化 1
鎔岩 0
鎔度 0
鎔接 0
鎔爐 2
鎔融 0
鎔解 0
鎔解熱 0
鎔銷 0
鎔鑄 0
鎔點 0
鎕 4
鎖 1671
鎖上 40
鎖上來 0
鎖上去 0
鎖不住 2
鎖住 96
鎖匙 242
鎖匙 242
鎖匠 10
鎖國 0
鎖好 12
鎖孔 1
鎖定 755
鎖定器 0
鎖定在 55
鎖櫃 2
鎖碼 0
鎖簧 0
鎖緊 33
鎖起 1
鎖起來 8
鎖釘 0
鎖鏈 35
鎖鐐 0
鎖鐐 0
鎖鑰 9
鎖門 17
鎖骨 26
鎗 187
鎗口 1
鎗彈 0
鎗戰 1
鎗托 0
鎗把 1
鎗斃 2
鎗林彈雨 0
鎗架 1
鎗桿 0
鎗法 3
鎗砲 0
鎗膛 0
鎗身 0
鎘 417
鎘 417
鎘米 4
鎙 418
鎚 237
鎚兒 0
鎚兒 0
鎚子 11
鎚頭 3
鎛 0
鎜 0
鎝 15
鎝 15
鎞 8
鎟 9
鎠 0
鎡 38
鎢 151
鎢 151
鎢礦局 0
鎢絲 15
鎣 11
鎤 7
鎥 7
鎦 6
鎦金 0
鎧 249
鎧甲 62
鎨 11
鎩 37
鎩 37
鎩羽 7
鎩羽而歸 8
鎪 5
鎫 0
鎬 76
鎬 76
鎬京 3
鎭 0
鎮 22745
鎮上 115
鎮上人 0
鎮公所 345
鎮南關 1
鎮反 5
鎮名 7
鎮壓 109
鎮壓住 1
鎮壓器 0
鎮子 5
鎮宅 7
鎮守 25
鎮安 75
鎮定 122
鎮定劑 20
鎮尺 0
鎮心 4
鎮戍 0
鎮日 44
鎮暴 85
鎮暴部隊 12
鎮暴隊形 1
鎮民 428
鎮民大會 1
鎮江 54
鎮江人 1
鎮江市 16
鎮流器 0
鎮海 121
鎮痛 47
鎮痛劑 9
鎮紙 0
鎮縣 7
鎮長 249
鎮靜 162
鎮靜劑 59
鎮靜鋼 0
鎮驚 0
鎯 68
鎰 306
鎱 5
鎲 5
鎳 565
鎳合金 12
鎳幣 2
鎳鉻絲 0
鎳鋼 3
鎴 15
鎵 70
鎶 0
鎷 4
鎸 0
鎻 0
鏀 63
鏁 0
鏂 38
鏂 38
鏃 46
鏄 5
鏄 5
鏅 0
鏅 0
鏆 0
鏇 16
鏈 1684
鏈傳動 0
鏈反應 22
鏈套 0
鏈子 4
鏈式 5
鏈式反應 1
鏈條 83
鏈球 11
鏈球菌 90
鏈罩 0
鏈輪 7
鏈鎖 6
鏈黴素 18
鏉 0
鏊 237
鏊 237
鏋 0
鏌 3
鏍 346
鏎 340
鏏 36
鏐 17
鏑 20
鏑矢 0
鏑箭 0
鏒 92
鏓 0
鏔 21
鏔 21
鏕 37
鏖 6
鏖兵 1
鏖戰 11
鏖戰到底 0
鏗 131
鏗然 1
鏗鏗 9
鏗鏘 22
鏗鏘聲 0
鏘 115
鏘鏘 6
鏙 14
鏚 45
鏛 0
鏜 44
鏜 44
鏜孔 0
鏜床 6
鏝 123
鏝刀 3
鏞 164
鏞 164
鏟 99
鏟出 0
鏟土 1
鏟土機 5
鏟子 29
鏟平 3
鏟煤 0
鏟除 12
鏠 0
鏡 3086
鏡中 118
鏡中人 5
鏡像 40
鏡光 2
鏡內 11
鏡匣 0
鏡台 7
鏡外 8
鏡子 365
鏡射 0
鏡屏 0
鏡戒 0
鏡架 86
鏡框 70
鏡泊湖 3
鏡片 507
鏡箱 1
鏡臺 0
鏡花水月 11
鏡花緣 20
鏡邊 0
鏡鑒 0
鏡面 164
鏡頭 1218
鏢 197
鏢客 12
鏢局 26
鏣 9
鏤 57
鏤冰 0
鏤刻 10
鏤版 0
鏤空 26
鏤簋 1
鏤骨銘心 0
鏥 0
鏦 10
鏧 7
鏨 7
鏨刀 0
鏨子 0
鏩 0
鏩 0
鏩 0
鏪 0
鏫 0
鏬 69
鏬 69
鏭 0
鏮 3002
鏳 0
鏳 0
鏴 0
鏵 82
鏶 5
鏷 15
鏸 6
鏹 23
鏹 23
鏺 3
鏻 20
鏻 20
鏼 4
鏼 4
鏽 414
鏽病 0
鏽菌 0
鏽蝕 23
鏾 3
鏾 3
鏿 7
鐀 9
鐀 9
鐂 0
鐃 93
鐃鈸 5
鐄 0
鐅 0
鐆 1
鐇 2
鐈 0
鐉 7
鐉 7
鐉 7
鐊 5
鐋 14
鐋鑼 1
鐌 4
鐍 4
鐎 3
鐏 11
鐐 43
鐐 43
鐐銬 0
鐑 0
鐒 24
鐓 13
鐔 14
鐕 12
鐖 46
鐗 0
鐘 2543
鐘乳石 51
鐘山 4
鐘擺 134
鐘樓 37
鐘樓怪人 33
鐘相 3
鐘聲 114
鐘表 12
鐘表店 1
鐘錶 1233
鐘錶店 34
鐘關 3
鐘響 21
鐘響聲 0
鐘頭 151
鐘鳴漏盡 0
鐘點 60
鐘點費 144
鐘鼎 5
鐘鼎人家 0
鐘鼎山林 7
鐘鼎文 0
鐘鼓 16
鐘鼓齊鳴 7
鐙 59
鐙 59
鐙環 0
鐚 0
鐚 0
鐛 0
鐜 0
鐜 0
鐜 0
鐜 0
鐝 0
鐞 0
鐟 0
鐟 0
鐠 15
鐧 0
鐨 6
鐩 11
鐪 57
鐫 21
鐫刻 15
鐫印 0
鐫琢 0
鐬 69
鐬 69
鐭 0
鐮 107
鐮刀 36
鐮刀狀 7
鐮狀 1
鐯 0
鐰 11
鐰 11
鐰 11
鐱 6
鐱 6
鐲 14
鐲子 1
鐳 43
鐳射 77
鐳射光 2
鐴 0
鐵 6683
鐵人 391
鐵像 0
鐵公雞 6
鐵則 30
鐵匠 12
鐵匠鋪 1
鐵印 0
鐵口 12
鐵口直斷 0
鐵合金 15
鐵嘴 9
鐵器 29
鐵器時代 8
鐵圈 11
鐵塊 7
鐵塔 138
鐵壁 6
鐵娘子 7
鐵定 82
鐵尺 0
鐵屑 18
鐵山 52
鐵工 106
鐵工廠 430
鐵布衫 0
鐵幕 19
鐵床 13
鐵廠 20
鐵心 27
鐵扇公主 3
鐵手 5
鐵手無情 0
鐵打 3
鐵打的 6
鐵拳 37
鐵木真 12
鐵杆 4
鐵杖 5
鐵杵 7
鐵杵磨成針 1
鐵板 156
鐵板大鼓 0
鐵板燒 120
鐵架 54
鐵柵門 0
鐵案 19
鐵桶 45
鐵條 7
鐵棍 9
鐵棒 12
鐵椅 2
鐵樹 19
鐵樹開花 4
鐵橋 36
鐵櫃 154
鐵欄杆 8
鐵氧體 2
鐵水 13
鐵沙掌 1
鐵泉 1
鐵渣 3
鐵漢 16
鐵漿 5
鐵灰 11
鐵灰色 13
鐵爐 5
鐵片 48
鐵牛 17
鐵球 13
鐵環 14
鐵甲 61
鐵甲武士 5
鐵甲車 0
鐵的 164
鐵的事實 7
鐵的紀律 0
鐵皮 70
鐵皮屋 1
鐵盒 12
鐵盤 4
鐵石 5
鐵石人 0
鐵石心腸 7
鐵砂 10
鐵砂掌 2
鐵砧 42
鐵砲 9
鐵碗 2
鐵磁性 4
鐵磬 0
鐵礦 40
鐵礦石 3
鐵票 40
鐵窗 64
鐵窗業 0
鐵筆 30
鐵筷 0
鐵算盤 1
鐵管 55
鐵箱 15
鐵籠 16
鐵籠子 1
鐵粉 19
鐵索 8
鐵絲 72
鐵絲網 38
鐵線 83
鐵罐 155
鐵腕 53
鐵良 2
鐵花 66
鐵蒺藜 2
鐵蓋 10
鐵蛋 156
鐵蠶豆 0
鐵血主義 0
鐵血宰相 1
鐵血政策 0
鐵衣 7
鐵觀音 50
鐵証 1
鐵證 8
鐵證如山 6
鐵質 124
鐵路 2406
鐵路局 243
鐵路局長 3
鐵路幹線 9
鐵路權 2
鐵路管理局 423
鐵路網 21
鐵路線 22
鐵路醫院 1
鐵蹄 3
鐵軌 72
鐵軍 1
鐵道 698
鐵道兵 0
鐵釘 45
鐵釘子 0
鐵針 1
鐵鉞 0
鐵鉤 2
鐵錘 12
鐵錨 3
鐵鍊 39
鐵鍬 3
鐵鎚 34
鐵鏈 6
鐵鏽 14
鐵門 68
鐵門窗 8
鐵青 15
鐵青色 1
鐵面 172
鐵面無私 5
鐵頭 5
鐵頭功 1
鐵頭杖 0
鐵飯碗 8
鐵餅 8
鐵馬 45
鐵騎 22
鐵鹽 17
鐶 32
鐷 86
鐸 229
鐹 11
鐹 11
鐺 16
鐺鐺 0
鐻 2
鐼 49
鐽 61
鐾 0
鐿 34
鑀 16
鑂 0
鑃 0
鑃 0
鑃 0
鑄 647
鑄下 2
鑄下大錯 2
鑄人 0
鑄件 62
鑄像 5
鑄冶 0
鑄刻 0
鑄印 5
鑄型 5
鑄字 67
鑄字工 0
鑄山煮海 0
鑄工 26
鑄幣 8
鑄幣廠 4
鑄成 20
鑄成大錯 6
鑄物 5
鑄造 447
鑄造出 0
鑄造成 0
鑄金 1
鑄鋼 23
鑄錠 2
鑄錯 0
鑄鐘 0
鑄鐵 128
鑄鐵工 0
鑅 17
鑅 17
鑆 20
鑇 17
鑇 17
鑈 63
鑈 63
鑉 18
鑉 18
鑊 17
鑋 10
鑌 45
鑍 0
鑎 0
鑎 0
鑏 25
鑐 48
鑐 48
鑑 2613
鑑價 0
鑑別 388
鑑別力 5
鑑別器 0
鑑別能力 0
鑑定 2448
鑑定人 17
鑑定書 16
鑑定為 17
鑑定考試 96
鑑察 10
鑑往知來 7
鑑戒 7
鑑於 593
鑑識 132
鑑賞 342
鑑賞力 9
鑑賞家 12
鑑賞能力 21
鑒 367
鑒別 7
鑒定 37
鑒察 11
鑒往知來 0
鑒於 243
鑒核 8
鑒真 3
鑒賞 16
鑒頻 0
鑒頻器 0
鑔 0
鑕 5
鑖 0
鑖 0
鑗 6
鑘 0
鑙 0
鑚 0
鑚 0
鑚 0
鑛 0
鑛 0
鑜 0
鑜 0
鑝 3
鑞 5
鑞鎗頭 0
鑟 0
鑠 78
鑠石流金 0
鑠金 1
鑡 0
鑢 3
鑣 35
鑣客 0
鑣局 5
鑤 5
鑥 0
鑨 121
鑩 37
鑪 6
鑫 2072
鑬 0
鑭 11
鑮 5
鑯 5
鑰 286
鑰 286
鑰匙 668
鑰匙 668
鑰匙圈 215
鑰匙孔 25
鑱 1
鑲 160
鑲上 7
鑲上來 0
鑲上去 0
鑲了 3
鑲住 0
鑲在 12
鑲嵌 72
鑲成 3
鑲板 6
鑲滿 3
鑲牙 125
鑲花 1
鑲著 24
鑲邊 15
鑲金 13
鑳 6
鑴 21
鑴 21
鑵 13
鑶 24
鑷 11
鑷子 31
鑸 6
鑹 0
鑺 0
鑻 0
鑼 130
鑼鼓 109
鑼鼓喧天 15
鑼鼓振天 0
鑽 918
鑽 918
鑽 918
鑽井 33
鑽井隊 0
鑽個 2
鑽個洞 1
鑽兒 0
鑽兒 0
鑽入 32
鑽入牛角尖 2
鑽出 35
鑽到 15
鑽台 1
鑽天入地 0
鑽天柳 0
鑽天楊 0
鑽子 9
鑽孔 150
鑽孔器 1
鑽孔機 38
鑽工 1
鑽床 66
鑽得 5
鑽得 12
鑽心蟲 0
鑽成 1
鑽戒 30
鑽探 94
鑽探機 0
鑽木 1
鑽木取火 3
鑽桿 3
鑽機 10
鑽洞 11
鑽營 9
鑽牛犄角 0
鑽牛角尖 42
鑽狗洞 0
鑽的 12
鑽石 740
鑽石戒指 3
鑽石礦 3
鑽石項鍊 3
鑽研 173
鑽穴 1
鑽空 0
鑽空子 0
鑽謀 0
鑽進 54
鑽進來 0
鑽進去 7
鑽過 9
鑽過來 0
鑽過去 1
鑽鑿 2
鑽門子 0
鑽開 1
鑽頭 39
鑾 193
鑾輿 1
鑾駕 4
鑿 119
鑿 119
鑿井 146
鑿井機 1
鑿出 7
鑿出來 1
鑿出去 0
鑿印 0
鑿在 1
鑿壁偷光 0
鑿子 6
鑿孔 2
鑿孔機 0
鑿山 2
鑿岩 2
鑿岩機 3
鑿巖 0
鑿巖機 0
鑿川 0
鑿成 1
鑿洞 7
鑿穴 1
鑿空 0
鑿穿 4
鑿船 0
鑿進 0
鑿進來 0
鑿進去 0
鑿鑿 0
鑿開 2
钀 48
钁 53
钁頭 1
钁頭 1
钂 67
钃 76
钄 0
長 299
長 29976
長 29976
長上 16
長世 6
長久 554
長久之計 7
長久以來 447
長了 167
長了 167
長人 30
長住 23
長信 17
長假 64
長兄 25
長公 20
長凳 3
長出 209
長出來 19
長列 5
長到 85
長劍 193
長勢 7
長吁短嘆 3
長吁短歎 0
長吏 0
長命 18
長命富貴 1
長命百歲 19
長命鎖 0
長嘆 63
長嘆一聲 13
長嘆一聲 13
長嘯 22
長圓 4
長在 316
長在 316
長城 373
長壽 419
長壽煙 0
長壽牌 1
長夏 15
長夜 62
長夜漫漫 25
長大 1120
長天 28
長天老日 0
長女 44
長姊 68
長媳 5
長嫂如母 0
長子 121
長字元 0
長存 56
長孫 46
長孫女 0
長安 986
長安人 1
長安街 138
長官 631
長寧 15
長射程 3
長尾 91
長尾 91
長尾 91
長尾 91
長尾巴 44
長尾巴 44
長尾巴 44
長尾巴 44
長尾猴 3
長尾猴 3
長尾猿 0
長尾猿 0
長山島 1
長崎 87
長崎市 5
長工 23
長巷 6
長年 263
長年累月 4
長幼 8
長幼有序 15
長庚 0
長庚大學 1
長庚紀念醫院 873
長庚醫院 713
長度 1178
長形 47
長征 63
長征軍 0
長徑 3
長得 563
長得 1480
長恨 11
長恨歌 21
長憂 0
長成 147
長房 11
長打 31
長挑 2
長排 3
長揖 1
長方 37
長方形 120
長方體 12
長於 247
長日 70
長明燈 0
長春 724
長春人 0
長春市 12
長時間 528
長期 5129
長期以來 296
長期借款 5
長期共存 2
長期性 102
長期投資 173
長期抗戰 17
長期負債 13
長期趨勢 18
長板凳 4
長柄 28
長條 98
長條圖 10
長條形 25
長椅 17
長榮 1961
長槍 39
長樂 96
長歎 11
長此下去 4
長此以往 11
長殤 0
長毛 142
長毛 142
長毛絨 1
長江 579
長江三峽 69
長江三角 0
長江三角洲 15
長江口 9
長江後浪 1
長江後浪推前浪 13
長江流域 38
長沙 151
長沙人 1
長沙市 13
長河 43
長治 119
長治久安 40
長波 36
長波 36
長泰 46
長流 65
長海 16
長滿 91
長濱 239
長煙一空 1
長煙一空 1
長物 23
長生 338
長生不老 34
長生果 1
長生殿 5
長生草 0
長生藥 0
長男 34
長白 99
長白丘陵 0
長白山 8
長白山地 5
長的 1480
長相 233
長相 233
長相左右 8
長眠 33
長眠不醒 0
長矛 25
長短 366
長短不一 7
長短不齊 3
長短句 3
長短詩 1
長石 70
長程 491
長程計劃 22
長程飛彈 9
長笛 260
長筒襪子 0
長策 2
長篇 183
長篇大論 30
長篇小說 234
長紅 101
長統 7
長統襪 10
長統靴 3
長絲 35
長線 115
長編 7
長老 485
長老教會 475
長老會 73
長者 191
長肉 2
長胖 7
長臂 22
長臂猿 42
長臂猿 42
長興 199
長舌 37
長舌夫 0
長舌婦 4
長著 44
長處 106
長號 34
長虹 60
長蛇 9
長蛇陣 0
長蟲 7
長衣 15
長衫 14
長袍 61
長袍馬褂 2
長袖 55
長袖善舞 4
長裙 0
長褲 114
長襪 14
長角牛 0
長計 5
長詩 27
長話短說 12
長調 8
長談 58
長谷 0
長起 8
長起來 1
長足 66
長足性 0
長足進步 8
長跑 84
長距離 56
長跪 6
長輩 391
長辛店 0
長途 306
長途旅行 9
長途跋涉 24
長途電話 171
長途飛行 13
長逝 1
長進 93
長過 10
長達 578
長遠 324
長遠利益 6
長遠性 6
長針 10
長釣 0
長長 165
長長的 144
長長地 144
長長短短 3
長青 503
長鞭 8
長音 19
長音程 0
長音階 0
長順 34
長頸鹿 110
長風破浪 1
長駐 25
長驅 6
長驅直入 22
長髮 272
長鬚鯨 2
長黑 43
長齋 1
長龍 49
镸 0
镸 0
镸 0
镹 0
镺 5
镻 4
镼 48
镽 27
镾 0
門 19385
門下 97
門人 32
門兒 3
門兒 3
門券 774
門前 261
門前門後 0
門口 868
門可羅雀 5
門吏 3
門外 328
門外漢 31
門子 36
門客 12
門市 999
門市部 83
門帘 2
門底 0
門底下 0
門庭 12
門庭若市 20
門廊 13
門徑 12
門徒 300
門戶 202
門戶之見 8
門戶開放 6
門戶開放政策 0
門房 18
門扇 35
門扉 13
門把 18
門捷列夫 0
門斗 3
門望 4
門板 39
門柄 0
門柱 14
門栓 12
門框 52
門楣 23
門樓 16
門樞 1
門檻 368
門檻兒 0
門檻兒 0
門法 11
門洞 24
門派 51
門燈 2
門牆 13
門牌 195
門牌號碼 41
門牙 52
門環 3
門生 21
門當戶對 9
門神 43
門票 686
門禁 417
門禁森嚴 7
門窗 288
門第 19
門簾 14
門縫 31
門羅 4
門聯 2
門號 100
門衛 1
門裡 15
門裡門外 1
門規 13
門診 4237
門診部 33
門警 1
門路 47
門逕 1
門道 34
門釘 1
門鈕 5
門鈴 58
門鎖 86
門閂 15
門閥 10
門限 2
門階 2
門隙 0
門電路 0
門面 73
門面話 0
門額 7
門類 14
門風 8
門館 1
門鬥 0
門齒 30
閁 0
閂 92
閂上 2
閂上來 0
閂上去 0
閂門 0
閃 844
閃了 13
閃了一下 3
閃了一下 3
閃亮 209
閃來 1
閃來閃去 10
閃光 179
閃光燈 205
閃出 29
閃到腰 0
閃動 98
閃去 1
閃失 11
閃掉 1
閃擊 11
閃族 3
閃爍 345
閃爍不定 7
閃爍其詞 7
閃爍著 34
閃現 14
閃耀 73
閃耀著 31
閃腰 3
閃著 77
閃語 0
閃身 8
閃躲 42
閃躲不及 1
閃過 93
閃避 76
閃避不及 8
閃鋅礦 4
閃鑠 2
閃閃 111
閃閃發光 31
閃開 16
閃電 324
閃電俠 7
閃電戰 0
閃電戰術 0
閃點 5
閄 0
閅 0
閆 2
閆 2
閇 0
閈 4
閉 2048
閉一隻眼 15
閉一隻眼 15
閉上 153
閉上眼 27
閉口 34
閉口不答 0
閉合 50
閉合電路 0
閉嘴 51
閉塞 54
閉幕 262
閉幕典禮 62
閉幕式 25
閉幕詞 3
閉會 39
閉月羞花 4
閉架 12
閉殼肌 0
閉氣 8
閉目 29
閉目養神 12
閉眼 36
閉起 27
閉路 46
閉路電視 33
閉鎖 129
閉門 21
閉門卻掃 0
閉門思過 0
閉門羹 5
閉門謝客 1
閉門造車 29
閉門造車 29
閉關 61
閉關政策 0
閉關自守 7
開 13522
開一 270
開一開 2
開三顯一 1
開上 21
開上來 1
開上去 0
開下 2
開下來 2
開下去 1
開不動 0
開了 559
開仗 0
開伙 9
開低 79
開低走高 26
開作 1
開來 227
開來開去 0
開例 2
開個 114
開倒車 14
開價 68
開元 259
開光 141
開入 5
開冬 0
開凍 1
開出 462
開出來 9
開出去 0
開刀 465
開刀室 0
開刀房 44
開刀手術 14
開列 44
開列於後 0
開到 161
開剖 0
開創 1046
開創者 10
開動 76
開包 1
開化 98
開化黨 0
開印 1
開卷 123
開卷有益 9
開去 38
開反 3
開口 717
開口笑 8
開口說 45
開口閉口 9
開向 42
開售 1
開啟 2439
開單 14
開單子 0
開嘴 14
開回 14
開回來 1
開回去 2
開國 37
開國元勳 5
開國元老 1
開國紀念 4
開國紀念日 41
開在 58
開地 17
開城 4
開埠 4
開堂 3
開場 11
開場 111
開場白 59
開場白 59
開墾 252
開墾地 3
開壞 2
開外 14
開夜車 3
開大 37
開天窗 17
開天闢地 20
開契 0
開好 17
開始 21024
開始時 268
開學 632
開學典禮 38
開學日 25
開完 53
開宗 10
開宗大師 0
開宗明義 31
開封 152
開封市 2
開封府 9
開射 1
開導 44
開小 12
開小差 3
開局 18
開展 497
開展工作 1
開山 68
開山祖師 7
開工 379
開工日期 26
開市 76
開幕 905
開幕典禮 187
開幕式 106
開幕禮 1
開幕詞 15
開平 102
開平方 3
開年 19
開店 145
開庭 141
開庭日 0
開廠 0
開建 5
開弔 0
開張 516
開張大吉 29
開彩 0
開往 56
開得 35
開得 458
開徵 202
開心 874
開心果 22
開心見誠 1
開心起來 2
開快 13
開快車 12
開恩 9
開懷 99
開懷大笑 15
開成 35
開戒 1
開戰 64
開戲 2
開戶 392
開打 327
開抵 94
開拍 36
開拓 1348
開拓史 14
開拔 26
開挖 303
開掌 0
開掘 2
開採 203
開採權 6
開採法 1
開提 3
開播 218
開播 218
開支 137
開支票 56
開放 8780
開放型基金 1
開放式 502
開放政策 54
開放時間 1049
開放電路 0
開方 2
開方根 0
開早 3
開明 106
開明人士 1
開映 0
開春 63
開春大吉 0
開普敦 32
開曠 1
開會 1118
開會地點 33
開會時間 44
開有 26
開朗 348
開本 387
開架 56
開架式 54
開桶 2
開棉機 2
開棺驗屍 1
開業 381
開槍 110
開標 880
開機 2147
開檔 53
開步 5
開步走 37
開歲 0
開段 1
開水 198
開法 20
開泰 24
開洋葷 0
開洞 10
開消 1
開港 10
開源 56
開源節流 45
開溜 11
開演 40
開火 37
開炮 12
開燈 34
開牌 1
開狀 51
開獎 74
開玩笑 393
開班 268
開球 48
開瓶 17
開瓶器 12
開疆 24
開疆闢土 12
開發 17828
開發中國 15
開發中國家 378
開發局 1
開發法 5
開發署 8
開的 457
開皇 5
開盤 979
開直 3
開眼 26
開眼界 7
開砲 100
開礦 19
開示 262
開票 404
開票所 81
開禁 2
開窗 64
開窗子 0
開竅 80
開立 485
開立方 0
開站 97
開端 114
開筆 5
開筵 0
開箱 28
開築 9
開籠 0
開結 3
開給 46
開線 7
開缺 2
開罪 16
開置 1
開罰單 16
開羅 34
開羅宣言 40
開羅會議 10
開胃 40
開胃菜 15
開胃酒 5
開脫 4
開腔 3
開膛 11
開臉 0
開舖 0
開航 68
開船 16
開花 287
開花季 0
開花期 16
開花結果 86
開苞 2
開荒 19
開著 159
開葷 2
開蒙 1
開藥 13
開藥方 5
開號 2
開衩 2
開衩口 0
開裂 12
開襠褲 4
開解 6
開言 6
開訓 57
開訓典禮 31
開記 4
開設 1081
開誠佈公 0
開誠布公 10
開誠相見 0
開課 1023
開課日期 83
開課時間 26
開講 891
開議 85
開豁 0
開賽 88
開走 22
開赴 11
開起 42
開起來 9
開起砲來 0
開足 2
開足馬力 0
開跑 152
開路 116
開路先鋒 19
開路機 0
開踢 4
開車 900
開車法 0
開轉 1
開辦 1158
開辦費 14
開近 1
開通 69
開造 2
開進 55
開進來 1
開進去 2
開運 116
開過 85
開過來 5
開過去 2
開道 27
開達 4
開遠 4
開選 1
開邊 3
開采 2
開釋 10
開金 25
開銷 103
開鋪 3
開鍋 0
開鎖 30
開鎖匠 0
開鑼 184
開鑼戲 0
開鑽 1
開鑿 68
開門 300
開門七件事 1
開門受徒 0
開門揖盜 0
開門見山 16
開門辦學 0
開閉 26
開開 107
開開關關 3
開間 10
開闊 189
開闊地 4
開闊性 1
開關 1450
開闢 674
開闢出來 1
開闢記 0
開除 150
開陽 33
開離 2
開雲見日 0
開頭 584
開顏 6
開顏大笑 0
開飯 16
開館 212
開館子 0
開駛 8
開高 117
開高走低 86
開麥拉 24
開點 33
閌 3
閍 3
閎 312
閎中肆外 0
閎衍 0
閎言 0
閎達 0
閏 87
閏位 0
閏年 48
閏日 0
閏月 8
閏秒 1
閏音 0
閐 6
閑 121
閑事 17
閑人 5
閑居 3
閑散 1
閑暇 4
閑空 0
閑置 0
閑話 7
閑談 3
閑雜 3
閑靜 4
閒 112
閒 112
閒 11205
閒下來 4
閒不住 1
閒事 18
閒人 51
閒人免進 0
閒來 78
閒兒 0
閒兒 0
閒冗 0
閒地 3
閒坐 4
閒官 0
閒居 8
閒工夫 2
閒常 2
閒心 1
閒情 245
閒情逸緻 10
閒情逸致 79
閒扯 8
閒散 23
閒日 3
閒時 25
閒暇 67
閒書 7
閒民 0
閒氣 11
閒混 2
閒漢 3
閒空 4
閒置 142
閒置不用 1
閒聊 138
閒著 39
閒著沒事 15
閒著沒事幹 1
閒蕩 3
閒言閒語 23
閒話 95
閒話家常 29
閒話少說 3
閒談 42
閒逛 236
閒遊 5
閒適 37
閒錢 16
閒閒 306
閒閒 306
閒隙 1
閒雅 4
閒雜 8
閒雜人等 14
閒雲野鶴 279
間 49543
間 7163
間不容息 0
間不容緩 1
間不容髮 4
間作 68
間作 68
間壁 3
間奏 16
間奏曲 5
間或 156
間接 1028
間接人工 1
間接性 16
間接成本 6
間接材料 0
間接稅 29
間接選舉 12
間數 38
間斷 146
間斷性 9
間期 24
間歇 68
間歇噴泉 1
間歇性 64
間歇河 1
間歇泉 1
間歇湖 1
間歇熱 0
間苗 0
間著 8
間諜 220
間諜片 0
間諜網 0
間距 181
間隔 326
間隔 326
間隔性 1
間隔成 2
間隙 129
間雜 16
間雜著 3
閔 215
閔傷 0
閔凶 0
閔妃 0
閔恤 0
閔采爾 0
閕 0
閘 389
閘刀開關 2
閘北 2
閘口 4
閘板 7
閘瓦 0
閘道 0
閘道器 0
閘門 91
閘閥 16
閙 0
閚 0
閛 11
閜 1
閜 1
閜 54
閝 0
閞 49
閟 89
閡 622
関 0
閣 1686
閣下 427
閣員 118
閣外 0
閣子 2
閣揆 149
閣樓 140
閣老 4
閣議 0
閤 583
閤 583
閥 1804
閥門 75
閦 0
閧 0
閨 1051
閨女 19
閨怨 3
閨房 43
閨秀 7
閨範 0
閨艷 0
閨門 2
閨閣 3
閩 877
閩 877
閩侯 1
閩北 8
閩北人 0
閩北地區 0
閩北話 0
閩南 138
閩南人 24
閩南地區 2
閩南話 16
閩南語 165
閩台 16
閩台地區 0
閩江 18
閩江流域 0
閩省 7
閩籍 16
閩籍人士 0
閩粵 26
閩粵人士 0
閩粵地區 2
閫 38
閬 196
閭 58
閭里 1
閭閻 1
閮 126
閯 0
閰 43
閰王 8
閱 7308
閱兵 32
閱兵典禮 0
閱兵台 1
閱兵式 0
閱卷 82
閱卷組 1
閱報 72
閱報室 19
閱書架 0
閱歷 38
閱覽 1295
閱覽室 317
閱讀 7044
閱讀器 17
閱讀室 5
閱讀機 49
閱讀站 0
閱讀課 3
閲 0
閳 0
閴 0
閵 4
閶 4
閷 6
閸 0
閹 43
閹人 5
閹割 28
閹割情結 1
閹奴 0
閹寺 0
閹豬 1
閹過 2
閹雞 5
閹黨 0
閺 3
閺 3
閻 158
閻宦 0
閻浮提 1
閻王 55
閻王爺 7
閻羅王 17
閻錫山 4
閼 8
閽 5
閽人 0
閾 34
閿 4
闀 24
闁 0
闁 0
闂 0
闂 0
闃 86
闄 0
闅 40
闆 64
闇 436
闇倫 0
闇冥 0
闇劣 0
闇弱 0
闈 70
闈場 8
闈場 8
闉 46
闊 482
闊人 0
闊人家 0
闊佬 2
闊別 27
闊別多年 0
闊少 2
闊少爺 7
闊度 1
闊手 0
闊步 7
闊步前進 0
闊氣 8
闊綽 3
闊老 1
闊落 0
闊葉林 74
闊葉樹 65
闊達 4
闋 44
闌 34
闌入 1
闌尾 14
闌尾炎 29
闌干 4
闌檻 0
闌珊 9
闍 18
闐 18
闐池 0
闑 6
闒 6
闓 11
闔 102
闔上 18
闔家 38
闔家歡 4
闔府 8
闔戶 1
闔扇 0
闔眼 6
闔第 0
闔第光臨 3
闕 234
闕 234
闕下 0
闕失 4
闕如 50
闕文 1
闕漏 7
闕疑 2
闕筆 0
闕詞 2
闖 599
闖 599
闖上 3
闖上來 0
闖上去 0
闖下 4
闖下大禍 0
闖事 0
闖來 4
闖來闖去 0
闖入 128
闖出 47
闖出來 0
闖出去 2
闖將 5
闖王 31
闖禍 29
闖紅燈 46
闖蕩 0
闖賊 0
闖進 66
闖進來 8
闖進去 10
闖進闖出 0
闖過 11
闖過來 2
闖過去 0
闖開 0
闖關 107
闗 0
闘 0
闙 0
闚 394
闛 42
關 12287
關上 157
關下 6
關不上 0
關不了 0
關不住 6
關中 108
關中地區 2
關了 90
關亡 0
關人 20
關仝 21
關住 8
關來關去 0
關係 12997
關係 572
關係人 280
關係代名詞 9
關係企業 776
關係到 112
關係式 16
關係著 64
關係詞 0
關個 7
關入 14
關內 31
關公 190
關出 14
關刀 21
關切 1022
關到 52
關卡 559
關反 5
關口 38
關名 25
關員 35
關回 4
關回去 2
關在 172
關外 44
關好 13
關子 17
關子嶺 25
關山 438
關山萬里 0
關島 285
關帝廟 44
關廟 440
關得 280
關心 4020
關愛 268
關懷 3079
關懷備至 1
關懷面 5
關押 7
關掉 271
關於 16621
關書 49
關有 14
關東 147
關東地區 7
關東煮 1
關東糖 0
關東軍 10
關棧 2
關樓 2
關機 443
關死 0
關法 12
關注 682
關渡 453
關渡廟 1
關滿 1
關漢卿 161
關照 117
關燈 24
關的 280
關禁閉 7
關稅 1249
關稅同盟 7
關窗 11
關窗戶 3
關站 0
關節 1071
關節炎 573
關節痛 48
關系 171
關緊 25
關羽 103
關聯 672
關聯性 144
關著 17
關西 571
關西地區 6
關說 102
關貿總協 44
關起 37
關起來 33
關連 457
關連性 94
關進 28
關進來 0
關進去 2
關過 8
關鍵 5570
關鍵問題 22
關鍵字 0
關鍵性 364
關鍵時刻 126
關門 161
關門大吉 20
關閉 1744
關關 26
關防 32
關隘 4
關雲長 13
關頭 132
闝 0
闝 0
闞 10
闞 10
闞 10
闟 4
闟 4
闠 4
闡 782
闡揚 57
闡明 149
闡發 18
闡述 345
闡釋 132
闢 445
闢作 0
闢劃 2
闢土 6
闢地 12
闢建 176
闢為 39
闢田 0
闢謠 14
闢邪 2
闣 22
闤 73
闥 41
阜 381
阜康 5
阜財 0
阝 0
阞 82
阠 1
阠 1
阡 105
阡表 0
阡陌 22
阢 6
阢隉不安 0
阣 4
阤 5
阥 0
阥 0
阦 0
阧 0
阧 0
阨 327
阨困 0
阨塞 0
阨災 0
阨窮 0
阩 0
阪 647
阪田 6
阫 0
阫 0
阫 0
阬 112
阬儒 0
阭 294
阮 1200
阮囊羞澀 3
阮大鋮 0
阮文紹 4
阮江 0
阮玲玉 1
阮籍 12
阯 119
阰 203
阱 183
防 5502
防上 4
防不勝防 27
防不勝防 27
防了 0
防人 6
防人之心 2
防人之心不可無 8
防來 2
防來防去 0
防修 13
防個 1
防偽 0
防備 82
防冷 1
防凍 8
防到 1
防制 1377
防務 28
防區 5
防地 2
防堵 0
防塵 185
防好 0
防守 301
防守戰 1
防守戰術 0
防害 7
防寒 25
防寫 31
防寫貼紙 0
防弊 67
防彈 174
防彈玻璃 3
防彈背心 13
防彈衣 31
防得 0
防得 17
防微杜漸 9
防患 28
防患於未然 12
防患未然 12
防戰 104
防拷 22
防旱 1
防晒 20
防晒油 3
防暑 1
防曬 156
防曬油 1
防染印花 3
防止 2596
防毒 1071
防毒面具 15
防水 843
防水布 16
防水紙 0
防水錶 0
防汛 41
防沙 15
防沙林 0
防治 3936
防治效果 9
防波堤 159
防波堤 159
防洪 317
防洪措施 7
防溼 1
防滑 88
防滑鏈 0
防潮 145
防潮法 0
防澇 0
防濕 9
防火 1315
防火巷 24
防火布 4
防火牆 398
防火設備 9
防災 672
防熱 49
防熱法 0
防疫 499
防疫站 19
防疫針 2
防病 12
防癆 14
防癆疫苗 0
防癆郵票 0
防的 17
防盜 574
防盜器 96
防盜鈴 1
防盜鎖 24
防禦 616
防禦力 41
防禦工事 9
防禦戰 0
防空 128
防空壕 1
防空洞 10
防空演習 6
防空節 1
防空警報 2
防範 706
防線 112
防縮 53
防老 25
防腐 81
防腐劑 73
防臭 40
防臭劑 0
防著 7
防蝕劑 0
防衛 477
防衛廳 25
防衛性 16
防衛部 25
防諜 1
防護 1266
防護團 19
防護林 5
防護罩 25
防賊 3
防賊法 0
防身 100
防身器 14
防身術 55
防軍 5
防避 0
防部 99
防鎖 38
防鏽 37
防鏽劑 0
防鏽漆 0
防防 0
防阻 25
防雨 11
防雨布 0
防震 215
防震措施 4
防霧 14
防風 78
防風林 56
防颱 68
防齲 0
阳 0
阴 0
阴 0
阷 0
阸 0
阸 0
阹 24
阺 9
阻 826
阻值 43
阻力 298
阻塞 543
阻塞住 0
阻尼 43
阻延 2
阻抗 177
阻抗匹配 9
阻撓 166
阻擊 4
阻擊戰 0
阻擋 250
阻擋層 0
阻擾 29
阻攔 23
阻攔著 0
阻斷 204
阻止 595
阻止不了 11
阻深 0
阻滯 37
阻礙 506
阻礙交通 9
阻絕 96
阻遏 9
阻隔 117
阻隔開 0
阻障 2
阼 29
阽 1
阾 0
阿 245
阿 2118
阿 21187
阿伐 17
阿伐 17
阿伯 123
阿依達 27
阿修羅 79
阿修羅道 1
阿們 63
阿兄 1
阿克蘇 0
阿公 207
阿公店水庫 34
阿公阿婆 8
阿兵哥 79
阿利安 1
阿利安人 0
阿叔 5
阿司匹林 0
阿司匹林 0
阿司匹靈 10
阿呀 35
阿呀 35
阿呆 221
阿呆瓜 0
阿哥 14
阿哥哥 4
阿囡 0
阿基米德 11
阿基米德原理 0
阿多諾 1620
阿妹 131
阿姆斯壯 35
阿姆斯特丹 261
阿姊 7
阿姐 37
阿姑 1
阿姨 332
阿娘 14
阿婆 126
阿婆 126
阿媽 216
阿嬤 400
阿嬸 4
阿嬸婆 0
阿富 30
阿富汗 64
阿富汗人 0
阿寶 94
阿岡 8
阿巴桑 2
阿布扎比 0
阿帕拉契 8
阿帕拉契山徑 1
阿弟 10
阿彌陀佛 371
阿房宮 10
阿房宮賦 1
阿扎尼亞 0
阿拉 223
阿拉丁 0
阿拉伯 217
阿拉伯人 28
阿拉伯半島 9
阿拉伯國家 18
阿拉伯數 0
阿拉伯數字 95
阿拉伯文 14
阿拉伯海 3
阿拉伯聯合大公國 26
阿拉伯語 40
阿拉巴馬 19
阿拉巴馬州 19
阿拉斯加 143
阿拉斯加州 13
阿拉法特 36
阿拉真主 0
阿摩尼亞 31
阿摩尼亞 31
阿斗 27
阿斯匹靈 67
阿斯巴甜 1
阿斯特 16
阿明 97
阿曼 180
阿木 7
阿柏 4
阿根廷 461
阿根廷人 2
阿梅 8
阿歷山大 0
阿母 57
阿比 1
阿毛 92
阿波 39
阿波 39
阿波羅 1388
阿波羅 1388
阿炳 174
阿父 2
阿爸 48
阿爹 4
阿爺 3
阿爾卑斯 79
阿爾卑斯山 48
阿爾卑斯山地 0
阿爾卑斯山脈 6
阿爾卑斯山脈 6
阿爾及利亞 36
阿爾及爾 4
阿爾巴尼亞 100
阿爾斯特 0
阿爾泰山 3
阿爾發粒子 0
阿狗 302
阿留申群島 2
阿益 13
阿福 111
阿米巴 60
阿米巴 60
阿美 225
阿美族 371
阿義 40
阿耨多羅 1
阿耨多羅三藐三菩提 1
阿肯色 25
阿肯色州 25
阿舅 2
阿花 44
阿茲海默 0
阿茲海默 0
阿茲海默症 0
阿茲海默症 0
阿蒙 1
阿薩寄省 0
阿薩密 0
阿裡 0
阿諛 9
阿諛奉承 6
阿諛苟合 0
阿諾 223
阿貓 153
阿賴耶識 1
阿那含 1
阿里 455
阿里山 1620
阿里巴巴 19
阿里桑那 2
阿里郎 21
阿門 29
阿闥婆 2
阿闥婆吠陀 2
阿附 1
阿靈頓 9
阿靈頓 9
阿非利加洲 0
阿飛 129
阿鼻 5
阿鼻地獄 5
陀 688
陀螺 85
陁 0
陂 42
陂 42
陂塘 1
陂池 1
陂陀 1
陃 22
附 7836
附上 516
附上來 0
附上去 1
附中 538
附了 27
附交 3
附件 2126
附來 1
附冊 14
附到 1
附則 193
附加 1574
附加值 3
附加物 4
附加稅 12
附匪 3
附去 0
附合 25
附和 68
附圖 391
附在 124
附好 0
附子 20
附小 131
附屬 1324
附屬品 43
附屬國 4
附屬國民小學 4
附屬地 0
附屬小學 10
附屬高級中學 49
附帶 327
附帶性 2
附帶條件 27
附庸 13
附庸國 3
附庸風雅 4
附得 1
附愛 0
附成 5
附文 114
附會 14
附有 688
附檔 0
附款 3
附注 2
附生 20
附睪 4
附筆 5
附籍 0
附約 167
附給 2
附耳 9
附耳低語 0
附聲韻 0
附著 326
附著力 14
附表 848
附言 1
附記 49
附設 2855
附註 756
附議 66
附贈 462
附贈品 0
附起 0
附足 5
附身 60
附近 4557
附近地區 65
附送 109
附送品 1
附過 5
附錄 1508
附錢 0
附隨 23
附點 16
陊 111
陋 65
陋俗 6
陋室 14
陋巷 3
陋習 28
陋見 1
陋規 6
陌 76
陌上隴間 0
陌生 709
陌生人 501
陌生客 2
陌路 7
降 453
降 4538
降三世 1
降三世明王 1
降下 69
降下來 16
降下去 0
降世 25
降了 28
降了 28
降伏 33
降低 4570
降低法 0
降價 616
降價出售 0
降價求售 5
降入 3
降冪 0
降到 266
降去 0
降回 5
降回來 0
降回到 0
降回去 0
降在 28
降壓 49
降壓法 0
降壓藥 4
降好 0
降妖 43
降官 0
降將 2
降幅 203
降得 11
降心相從 0
降志辱身 1
降成 18
降於 5
降於 5
降旗 17
降旗典禮 0
降旨 0
降服 31
降格 12
降格以求 1
降格相求 0
降水 132
降水量 56
降法 5
降溫 140
降溫法 2
降火 18
降為 308
降生 60
降福 9
降級 67
降職 5
降臨 226
降臨到 6
降自 0
降至 630
降落 403
降落傘 73
降落在 38
降落費 1
降落點 1
降著 1
降號 0
降解 20
降調 6
降賜 1
降錯 0
降附 0
降雨 242
降雨量 93
降雪 15
降雪量 3
降霜 2
降點 0
降龍伏虎 2
陎 394
陏 523
限 5566
限令 13
限位 3
限價 24
限制 5724
限制住 9
限制器 3
限制式 7
限制性 26
限制是 10
限制級 178
限制詞 0
限定 364
限定值 0
限定詞 0
限定量 0
限定額 0
限度 187
限於 688
限明 3
限時 521
限時信 4
限時專送 19
限期 507
限止 1
限為 53
限產 14
限田 0
限界 9
限量 345
限量供應 48
限量生產 5
限額 266
陑 1313
陒 0
陒 0
陓 611
陔 1221
陕 0
陖 0
陗 0
陘 1733
陙 0
陚 0
陛 6909
陛下 106
陛衛 1
陛見 1
陛賀 1
陛辭 1
陜 90
陝 48357
陝北 6
陝北高原 1
陝縣 0
陝西 68
陝西梆子 1
陝西省 30
陞 1399
陞官 2
陟 6
陟罰 0
陟降 1
陠 0
陠 0
陠 0
陡 644
陡坡 58
陡壁 1
陡峭 94
陡峻 15
陡然 39
陡然升高 0
陡直 3
院 15170
院士 802
院子 130
院會 750
院本 750
院校 1006
院牆 5
院系 170
院落 37
院轄 5
院轄市 39
院長 4231
院門 22
陣 1565
陣亡 111
陣亡人數 0
陣亡將士 15
陣亡者 3
陣列 594
陣前 31
陣勢 26
陣圖 29
陣地 40
陣地戰 2
陣子 325
陣容 1010
陣容堅強 59
陣容龐大 3
陣式 69
陣法 21
陣營 1773
陣痛 68
陣發性 44
陣線 193
陣腳 32
陣雨 43
陣雲 0
陣風 34
除 13690
除下 17
除不盡 0
除了 13961
除以 237
除伐 0
除伐 0
除冰 3
除卻 27
除去 432
除去物 0
除名 102
除喪 3
除塵 24
除塵器 9
除夕 145
除夕夜 16
除外 604
除奸 1
除好 1
除官 1
除害 8
除弊 7
除役 123
除惡 8
除惡務盡 3
除掉 120
除數 12
除暴 11
除暴安良 10
除根 10
除權 290
除權息 1
除此 98
除此之外 657
除此以外 23
除法 81
除清 1
除溼 9
除溼機 5
除濕 116
除濕機 60
除災 1
除病 3
除磁 0
除站 1
除籍 10
除罪 26
除臭 76
除臭劑 37
除臭藥 0
除舊佈新 17
除舊布新 4
除草 34
除草劑 37
除草機 2
除莠劑 0
除號 4
除蟲劑 6
除蟲菊 19
除走 2
除起 1
除起來 0
除過 5
除邪 5
除錯 208
除開 19
除霜 8
除霜機 0
除霧 16
除非 1548
除非在 28
除非己莫為 2
除非是 163
除非有 103
陪 2251
陪不是 0
陪了 33
陪伴 344
陪伴在 19
陪伴著 37
陪侍 7
陪出 0
陪到 1
陪同 538
陪哭 0
陪嫁 2
陪客 15
陪審 23
陪審制 1
陪審制度 1
陪審員 12
陪審團 68
陪小心 0
陪從 0
陪房 1
陪禮 3
陪笑 8
陪笑臉 2
陪罪 7
陪著 202
陪葬 29
陪襯 25
陪起 0
陪起來 0
陪都 3
陪酒 17
陪陪 18
陪陵 0
陫 5
陬 535
陭 6
陭 6
陮 0
陯 4
陰 30408
陰到 447
陰乾 21
陰令 1
陰僻 0
陰冷 22
陰司 4
陰司地府 0
陰吏 0
陰唇 27
陰囊 97
陰地 5
陰壽 0
陰天 35
陰宅 21
陰室 0
陰山 14
陰差 2
陰差 2
陰差陽錯 1
陰平 17
陰府 7
陰影 642
陰德 15
陰性 185
陰性反應 16
陰慘 0
陰戶 46
陰文 0
陰晦 1
陰晴 16
陰晴不定 13
陰暗 130
陰暗面 7
陰曆 17
陰曆年 0
陰曹 1
陰曹地府 3
陰月 1
陰柔 24
陰柔性 0
陰柔氣 0
陰核 16
陰森 17
陰森森 13
陰極 70
陰極射線 22
陰極射線管 23
陰極管 7
陰歷 0
陰毛 48
陰氣 18
陰氣沉沉 0
陰沈 16
陰沈沈 1
陰沉 21
陰沉沉 5
陰泉 0
陰涼 72
陰溝 25
陰溝裡翻船 1
陰濕 12
陰狠 7
陰狠毒辣 0
陰盛陽衰 3
陰私 1
陰穴 1
陰童 0
陰脣 1
陰莖 265
陰蒂 19
陰處 3
陰謀 238
陰謀份子 2
陰謀家 1
陰謀詭計 5
陰道 446
陰部 97
陰錯陽差 44
陰間 60
陰陰 12
陰陰暗暗 0
陰陰沈沈 0
陰陰沉沉 0
陰陽 416
陰陽人 7
陰陽官 0
陰陽家 3
陰陽怪氣 5
陰陽眼 9
陰險 20
陰險毒辣 2
陰離子 111
陰雨 51
陰雨天 5
陰雲 5
陰電 2
陰電子 0
陰電性 0
陰霾 154
陰霾霾 0
陰風 32
陰風愁慘 0
陰鬱 24
陰魂 4
陰魂不散 13
陱 1
陲 1383
陳 51707
陳三五娘 1
陳乞 0
陳事 1
陳作新 0
陳侃如 4
陳倉 3
陳其美 1
陳列 1144
陳列品 1
陳列室 125
陳列窗 0
陳列館 207
陳則旭 2
陳勝 121
陳圓圓 1
陳壽 34
陳大悲 1
陳天華 1
陳奐 4
陳威仲 3
陳子昂 4
陳少白 4
陳屍 37
陳履安 84
陳州 4
陳平 35
陳年 104
陳年老酒 2
陳康本 1
陳情 822
陳慶新 0
陳摶 3
陳文茜 4
陳明 902
陳樹群 0
陳毅 13
陳水扁 0
陳炯明 3
陳熾 2
陳獨秀 9
陳玉成 0
陳留 3
陳皮 21
陳皮梅 1
陳碩真 0
陳秋揚 10
陳穀子爛芝麻 0
陳立 177
陳群 5
陳義甚高 0
陳腐 20
陳腔濫調 11
陳舊 70
陳舊不堪 1
陳英士 0
陳蘭皋 4
陳規 4
陳規陋習 0
陳言 4
陳設 77
陳訴 19
陳誠 53
陳請 75
陳貨 0
陳跡 9
陳蹟 3
陳述 513
陳酒 37
陳醋 0
陳陳相因 1
陴 132
陵 1903
陵園 2
陵墓 30
陵夷 0
陵寢 14
陵轢 1
陵雲 0
陵霄 0
陶 2793
陶俑 4
陶冶 184
陶冶性情 8
陶匠 5
陶器 197
陶土 183
陶宏景 0
陶工 16
陶朱公 4
陶染 0
陶淵明 43
陶潛 7
陶然 28
陶然自得 0
陶瓷 1507
陶瓷器 40
陶瓷工 4
陶瓷工業 78
陶瓷業 19
陶磚 4
陶笛 5
陶聲洋防癌基金會 1
陶藝 983
陶藝品 6
陶醉 81
陶醉在 38
陶鑄 13
陶陶 8
陶陶然 0
陷 757
陷井 14
陷入 926
陷入絕境 1
陷害 84
陷害忠良 5
陷敵 3
陷於 158
陷沒 0
陷溺 19
陷滯 0
陷落 28
陷阱 375
陷陣 1
陸 5515
陸 5515
陸上 428
陸九淵 0
陸光 93
陸半球 0
陸地 381
陸地棉 0
陸塊 21
陸委會 0
陸官 20
陸居 4
陸征祥 0
陸戰 30
陸戰隊 93
陸攻 0
陸攻隊 0
陸棲 16
陸榮廷 1
陸橋 78
陸海 83
陸海大戰 0
陸海大餐 0
陸海空 27
陸海空三軍 6
陸海空軍 33
陸游 19
陸生 32
陸生動物 3
陸界 2
陸稻 6
陸空 63
陸空大戰 0
陸續 2831
陸豐 16
陸賈 1
陸賈 1
陸路 31
陸軍 574
陸軍官校 69
陸軍總司令 10
陸軍總部 9
陸軍軍官 3
陸軍軍官學校 93
陸軍醫院 3
陸遊 5
陸運 29
陸運費 0
陸陸 8
陸陸續續 82
陸離 0
陹 0
陻 0
陻 0
陼 6
陽 15520
陽世 4
陽傘 21
陽光 2118
陽光社會福利基金會 1
陽具 79
陽剛 43
陽剛味 2
陽剛氣 6
陽台 345
陽壽 2
陽奉陰違 7
陽宅 183
陽平 17
陽平聲 0
陽性 250
陽性反應 54
陽明 2490
陽明國中 110
陽明國小 17
陽明大學 1
陽明山 1192
陽明醫學院 158
陽明醫院 115
陽春 105
陽春型 14
陽春白雪 1
陽春麵 26
陽曆 15
陽曆年 1
陽朔 17
陽極 67
陽極板 2
陽極泥 9
陽氣 88
陽物 4
陽痿 83
陽盛陰衰 3
陽童 3
陽臺 29
陽酒 1
陽間 9
陽關 16
陽關大道 1
陽離子 165
陽電 90
陾 5
陿 0
隀 105
隁 0
隁 0
隂 0
隂 0
隃 160
隄 131
隄 131
隄堰 0
隄岸 0
隄邊 0
隄防 6
隅 697
隆 16353
隆乳 67
隆冬 19
隆厚 0
隆寒 0
隆德 15
隆恩 28
隆愛 0
隆昌 178
隆暑 0
隆替 0
隆極 0
隆田 49
隆盛 152
隆興 121
隆起 118
隆重 310
隆隆 50
隆隆聲 1
隆隆響 0
隆鼻 27
隇 24
隈 29
隉 4046
隊 7631
隊上 21
隊伍 660
隊列 6
隊別 0
隊友 220
隊名 62
隊員 655
隊商 1
隊形 39
隊形變換 2
隊旗 6
隊部 12
隊長 540
隋 1114
隋代 10
隋代人 0
隋書 6
隋朝 14
隋朝人 0
隋煬帝 24
隌 0
隌 0
隌 0
隍 807
階 3077
階下 7
階下囚 6
階乘 0
階地 41
階層 776
階梯 1033
階段 5265
階段性 185
階石 1
階級 427
階級分化 1
階級分析 4
階級化 1
階級友愛 0
階級性 5
階級意識 5
階級本質 1
階級矛盾 0
階級社會 1
階級立場 2
階級立場 2
階級覺悟 0
階級觀點 0
階級路線 0
階級鬥爭 21
階級鬥爭熄滅論 0
随 0
隑 66
隒 70
隓 81
隔 1251
隔上 2
隔下 0
隔世 13
隔了 101
隔代 18
隔代遺傳 2
隔作 1
隔來隔去 0
隔入 0
隔出 2
隔出來 0
隔別 1
隔去 0
隔向 1
隔在 4
隔壁 445
隔壁家 0
隔夜 138
隔夜茶 4
隔夜飯 0
隔天 357
隔好 0
隔子 0
隔完 2
隔宿 3
隔層 21
隔山 12
隔岸 14
隔岸觀火 7
隔巷 0
隔得 4
隔成 7
隔扇 0
隔斷 15
隔日 237
隔月 6
隔板 38
隔次 0
隔此 0
隔水 17
隔水相望 0
隔河 2
隔河相望 3
隔法 0
隔湖 0
隔湖相望 0
隔溪 0
隔溪相望 0
隔熱 350
隔熱板 11
隔熱法 0
隔牆 24
隔牆有耳 1
隔絕 148
隔絕力 0
隔膜 28
隔艙板 0
隔著 148
隔行如隔山 8
隔走 0
隔起 2
隔起來 2
隔距 6
隔週 215
隔過 1
隔鄰 27
隔門 3
隔開 108
隔間 324
隔閡 88
隔離 728
隔離政策 3
隔離法 0
隔離病房 3
隔離霜 1
隔靴搔癢 8
隔音 120
隔音板 2
隔音牆 8
隔音設備 4
隔餐 0
隕 35
隕命 0
隕星 13
隕歿 2
隕石 190
隕落 21
隕首 3
隖 0
隗 43
隘 285
隘 285
隘口 20
隘害 0
隘嶺 0
隘巷 1
隘窘 0
隘窮 0
隘路 1
隘阻 0
隘險 0
隙 874
隙縫 46
隙罅 0
隙駒 0
隚 0
際 1392
際會 33
際涯 2
際遇 94
障 3285
障眼法 8
障礙 5702
障礙物 68
障礙競走 0
障礙賽 9
障礙賽跑 1
障翳 0
隝 0
隞 185
隟 0
隟 0
隠 0
隠 0
隡 77
隢 50
隣 0
隤 177
隥 0
隦 0
隧 80
隧洞 0
隧道 880
隨 5259
隨之而起 7
隨伴者 0
隨侍 23
隨侍在側 3
隨便 1074
隨俗 9
隨傳隨到 12
隨到 156
隨即 858
隨口 50
隨口答應 1
隨口說說 8
隨叫 4
隨叫隨到 12
隨同 144
隨和 70
隨員 11
隨地 51
隨地吐痰 5
隨帶 5
隨後 1023
隨後就來 0
隨後就到 3
隨後就去 0
隨從 60
隨從人員 3
隨從官 0
隨心所欲 149
隨想曲 11
隨意 900
隨意肌 11
隨手 238
隨手關燈 9
隨手關門 3
隨插即用 0
隨時 3835
隨時制宜 0
隨時奉陪 3
隨時歡迎 46
隨時隨地 269
隨書 60
隨書贈送 1
隨機 794
隨機應變 20
隨波逐流 28
隨波逐流 28
隨片贈送 1
隨眠 1
隨筆 175
隨緣 186
隨聲 4
隨聲附和 2
隨著 4403
隨處 148
隨行 188
隨行人員 12
隨踵而至 0
隨身 323
隨身侍從 0
隨身攜帶 119
隨身碟 1
隨身聽 116
隨身行李 25
隨車 50
隨遇而安 19
隨選 0
隨隊 15
隨隨 3
隨隨便便 42
隨風 172
隨風而逝 172
隨風轉舵 0
隩 4
險 2962
險些 49
險些兒 2
險些兒 2
險勝 51
險固 1
險境 26
險峻 41
險情 4
險惡 58
險棘 0
險毒 1
險灘 5
險要 28
險詐 4
險譎 0
險象 4
險象環生 27
險遭不測 0
險遭毒手 0
險阻 20
險隘 0
隫 0
隫 0
隬 26
隭 0
隮 5
隯 0
隰 21
隱 104
隱 1040
隱伏 13
隱伏性 0
隱匿 62
隱名 6
隱君子 0
隱含 168
隱喻 103
隱士 31
隱姓埋名 12
隱密 105
隱密性 36
隱射 1
隱居 79
隱形 239
隱形眼鏡 1029
隱忍 16
隱忍住 0
隱忍著 1
隱性 109
隱患 4
隱情 14
隱惡揚善 4
隱慝 0
隱憂 199
隱晦 10
隱暗 2
隱沒 41
隱潭 1
隱然 74
隱然作痛 0
隱然可見 2
隱然若現 2
隱疾 2
隱痛 1
隱癖 0
隱瞞 125
隱瞞不住 1
隱私 296
隱私權 189
隱秘 34
隱約 176
隱約可見 23
隱約約 1
隱者 45
隱蔽 46
隱蔽物 0
隱藏 1017
隱藏住 2
隱藏式 81
隱藏檔 0
隱藏著 94
隱語 2
隱諱 4
隱身 69
隱身術 8
隱退 14
隱逸 6
隱遁 15
隱隱 131
隱隱作痛 20
隱隱地 8
隱隱地 8
隱隱約約 49
隱隱藏藏 0
隱函數 10
隲 0
隳 829
隴 771
隴山 4
隴州 0
隴海鐵路 1
隴畝 1
隴西 33
隴西高原 1
隵 0
隶 0
隷 0
隸 313
隸屬 316
隸屬於 223
隸書 45
隸農 0
隹 16
隺 0
隺 0
隺 0
隻 3021
隻字 15
隻字片紙 0
隻字片語 39
隻影 6
隻手遮天 0
隻眼 93
隻言片語 0
隻身 50
隼 133
隽 0
隽 0
隿 5
雀 104
雀 1041
雀兒山 0
雀兒山 0
雀屏 2
雀屏中選 20
雀巢 123
雀斑 62
雀躍 59
雀躍 59
雀躍三分 0
雀躍三分 0
雀躍不已 12
雁 709
雁子 6
雁序 0
雁群 15
雁翎隊 0
雁行 7
雁門關 4
雂 198
雃 113
雄 12788
雄偉 175
雄健 8
雄健步伐 0
雄兵 54
雄厚 172
雄厚實力 4
雄厚資金 1
雄圖 5
雄圖大業 0
雄壯 42
雄壯威武 14
雄奇 6
雄姿 27
雄師 27
雄師百萬 0
雄心 54
雄心勃勃 15
雄心壯志 31
雄心萬丈 5
雄性 184
雄性不育 0
雄才 9
雄才大略 6
雄據一方 0
雄斷 2
雄武 3
雄海馬 0
雄渾 26
雄渾有力 1
雄犢 1
雄獅 764
雄略 15
雄的 317
雄糾糾 2
雄花 15
雄蕊 23
雄蜂 22
雄豪 16
雄赳赳 3
雄踞 5
雄踞一方 0
雄辯 27
雄辯家 0
雄雞 5
雄霸一方 7
雄風 106
雄體 6
雄魚 34
雄鷹 34
雄鹿 4
雄黃 20
雄黃酒 2
雅 9550
雅人 37
雅俗 7
雅俗共賞 14
雅克薩 2
雅典 110
雅典娜 56
雅加達 546
雅加達人 0
雅各 10
雅各賓派 0
雅士 37
雅安 18
雅座 8
雅惠 333
雅意 1
雅愛 0
雅慧 81
雅故 0
雅樂 49
雅歌 106
雅溫得 0
雅爾塔 0
雅緻 108
雅美族 186
雅致 34
雅興 18
雅虎 0
雅號 11
雅觀 5
雅言 19
雅詞 0
雅量 21
雅集 51
雅音 47
雅魯藏布 2
雅魯藏布江 5
集 18300
集中 2773
集中化 24
集中反映 1
集中營 81
集中點 4
集了 39
集其大成 3
集合 1404
集合令 0
集合地點 82
集合時間 63
集合號 0
集合起來 18
集合體 21
集合點 16
集團 6543
集團主義 0
集團結婚 44
集團軍 9
集塵 186
集大成 10
集大成者 3
集子 11
集少成多 0
集市 9
集思廣益 104
集思會 8
集成 161
集成電路 22
集散 40
集散中心 10
集散地 41
集散站 45
集於一身 7
集會 714
集會法 1
集會結社 5
集會自由 9
集會遊行法 0
集材 7
集權 47
集權主義 1
集權國家 4
集水 421
集注 7
集積 26
集約 28
集結 264
集結成 31
集群 41
集聚 25
集腋成裘 2
集腋成裘 2
集解 23
集訓 189
集訓中心 1
集訓團 0
集訓地 0
集訓營 8
集諦 1
集貨 51
集資 137
集資合建 0
集賢 100
集遊法 0
集郵 390
集郵票 6
集郵簿 0
集郵迷 0
集錄 22
集錦 759
集錦式 1
集鎮 150
集電極 0
集電環 0
集體 917
集體主義 5
集體利益 2
集體創作 20
集體化 2
集體性 12
集體所有 1
集體所有制 4
集體福利 2
集體經濟 7
集體行動 14
集體農莊 1
集體領導 5
雇 242
雇主 955
雇人 37
雇佣 6
雇個 0
雇傭 6
雇到 1
雇員 134
雇好 0
雇工 15
雇戶 0
雇於 1
雇有 1
雇用 247
雇船 2
雇請 14
雇車 1
雇農 1
雇馬 1
雈 61
雉 286
雉尾 1
雉求牡匹 0
雉雞 8
雉鳩 0
雊 11
雋 274
雋 274
雋拔 1
雋敏 0
雋材 0
雋永 79
雋良 0
雋茂 0
雌 3000
雌 526
雌伏 3
雌伏 3
雌威 4
雌威 4
雌性 134
雌性 134
雌性化 0
雌性化 0
雌性花 0
雌性花 0
雌牙露嘴 0
雌牙露嘴 0
雌牙露嘴 0
雌牙露嘴 0
雌犢 0
雌犢 0
雌獅 2
雌獅 2
雌老虎 1
雌老虎 1
雌花 14
雌蕊 10
雌蕊 10
雌豹 0
雌豹 0
雌雄 152
雌雄 152
雌雄同株 7
雌雄同株 7
雌雄同體 19
雌雄同體 19
雌雄異株 8
雌雄異株 8
雌雄異體 0
雌雄異體 0
雌雄莫辨 2
雌雄莫辨 2
雌馬 1
雌馬 1
雌驢 0
雌驢 0
雌黃 1
雌黃 1
雍 324
雍容 13
雍容爾雅 0
雍容華貴 13
雍容雅步 0
雍正 62
雍正年間 12
雍睦 0
雎 38
雎鳩 4
雏 0
雐 0
雑 0
雒 15
雒邑 1
雓 43
雔 5
雕 1400
雕像 145
雕刻 630
雕刻 630
雕刻品 55
雕刻家 23
雕塑 782
雕塑品 17
雕塑家 36
雕弓 6
雕成 22
雕板 1
雕樑畫棟 6
雕漆 7
雕版 5
雕琢 51
雕章繢句 0
雕繪 7
雕花 31
雕蚶鏤蛤 0
雕蟲小技 2
雕鏤 3
雕飾 27
雕飾品 0
雕龍 12
雖 6188
雖則 19
雖敗猶榮 19
雖是 404
雖有 567
雖然 15377
雖然在 536
雖然是 450
雖然會 34
雖然有 372
雖然能 16
雖能 41
雖說 347
雗 17
雘 7
雙 9956
雙下巴 3
雙主修 1
雙亡 11
雙人 363
雙人床 53
雙人房 257
雙人舞 5
雙作稻 0
雙倍 46
雙全 29
雙刃 7
雙包 17
雙十 185
雙十國慶 58
雙十節 39
雙十革命 0
雙向 880
雙向交通 0
雙唇 28
雙喜 65
雙喜臨門 19
雙園 81
雙園區 5
雙城 152
雙姓 0
雙子 290
雙子座 756
雙子星 66
雙子葉 5
雙子葉植物 221
雙季稻 1
雙宿雙飛 4
雙層 413
雙層公車 1
雙層床 6
雙峰駝 0
雙座 27
雙座腳踏車 0
雙循環 1
雙性戀 0
雙手 905
雙手萬能 4
雙手贊成 11
雙打 208
雙打賽 8
雙折射 14
雙拳 0
雙掌 0
雙掛號 20
雙摺 0
雙擊 0
雙數 20
雙料 64
雙料冠軍 26
雙方 3548
雙方同意 77
雙日 15
雙星 97
雙曲線 17
雙月 60
雙月刊 572
雙桅 2
雙棲 6
雙槓 4
雙槓體操 0
雙橡園 8
雙氧 12
雙氧水 88
雙流 34
雙溪 418
雙溪公園 36
雙球菌 6
雙生 13
雙生女 0
雙生子 3
雙目 119
雙目失明 4
雙眼 339
雙眼皮 53
雙筒 13
雙筒望遠鏡 40
雙管 27
雙管齊下 69
雙節棍 0
雙簧 6
雙簧管 56
雙簧簫 0
雙糖 3
雙線 51
雙翼 39
雙翼飛機 0
雙耳瓶 2
雙聲 9
雙聲帶 10
雙職工 1
雙肩 33
雙胞 14
雙胞胎 243
雙腳 202
雙腿 0
雙膝 0
雙臂 85
雙葉 60
雙號 8
雙號座號 0
雙螺旋 14
雙衛 21
雙親 169
雙語 0
雙豐 2
雙贏 0
雙足 0
雙軌 74
雙軌制 17
雙軌鐵路 3
雙輪 7
雙連 131
雙週 407
雙週刊 392
雙邊 545
雙邊協定 18
雙邊會談 19
雙邊條約 2
雙重 592
雙重人格 8
雙重國籍 30
雙重性 6
雙重標準 47
雙鍵 22
雙關 8
雙關語 22
雙雙 238
雙雙對對 15
雙面 449
雙面嬌娃 3
雙面諜 3
雙靨 0
雙頭肌 0
雙頰 32
雙頻 0
雙飛 41
雙餉 0
雙骰 0
雙魚 1
雙魚座 712
雙鯉 1
雙鳳 26
雙黃線 8
雚 9
雛 281
雛兒 0
雛兒 0
雛型 149
雛妓 59
雛形 144
雛菊 25
雛菊式 0
雛雞 23
雛鳥 26
雛鳳 2
雛鶯 0
雜 1724
雜七雜八 80
雜亂 93
雜亂不堪 0
雜亂無章 27
雜事 23
雜事纏身 1
雜交 120
雜交種 12
雜交育種 3
雜交親本 1
雜劇 53
雜務 22
雜史 1
雜品 91
雜家 4
雜居 11
雜役 3
雜志 21
雜念 35
雜感 87
雜技 18
雜技團 17
雜技場 0
雜技場 0
雜技表演 7
雜拌 0
雜拌兒 0
雜拌兒 0
雜支 17
雜數 0
雜文 209
雜業 1
雜樣 0
雜樹林 0
雜沓 8
雜湊 0
雜燴 6
雜牌 6
雜牌軍 1
雜物 114
雜用 2
雜碎 56
雜稅 0
雜種 27
雜種人 0
雜種優勢 4
雜種後代 0
雜種狗 4
雜糧 120
雜耍 16
雜肥 0
雜膾 0
雜色 16
雜草 137
雜草叢生 20
雜菌 6
雜處 8
雜言 3
雜訊 294
雜記 163
雜誌 13933
雜誌社 1154
雜說 14
雜貨 620
雜貨店 255
雜貨鋪 15
雜費 70
雜質 177
雜錄 14
雜錯 1
雜院 0
雜集 4
雜音 194
雜項 343
雜項收入 33
雜項費用 4
雜類 5
雜食性 60
雜食獸 0
雝 9
雞 3524
雞雞 10
雞丁 27
雞兔 2
雞兔同籠 3
雞兔問題 0
雞冠 18
雞冠石 1
雞冠花 12
雞南山 6
雞叫 6
雞同鴨講 0
雞啼 4
雞塊 18
雞姦 7
雞婆 139
雞子 2
雞尾 28
雞尾酒 319
雞尾酒會 20
雞屁股 4
雞巴 1
雞年 7
雞心 3
雞心形 0
雞心領 0
雞排 0
雞柳 1
雞柵 0
雞毛 14
雞毛信 0
雞毛撢子 4
雞毛蒜皮 6
雞汁 6
雞湯 321
雞爪 11
雞犬不寧 3
雞犬不留 3
雞犬升天 3
雞犬相聞 2
雞瘟 29
雞皮 9
雞皮疙瘩 19
雞皮鶴髮 0
雞眼 22
雞窩 27
雞籠 51
雞精 79
雞絲 42
雞絲麵 0
雞翅 0
雞翅膀 2
雞肉 155
雞肋 31
雞肝 7
雞胸 11
雞胸肉 13
雞腳 4
雞腿 67
雞腿大餐 0
雞腿肉 2
雞腿飯 14
雞腿麵 1
雞舍 16
雞蛋 371
雞蛋碰石頭 1
雞蛋糕 7
雞蛋裡挑骨頭 5
雞西 15
雞隻 53
雞零狗碎 1
雞頭 17
雞飛狗跳 26
雞骨 4
雞骨頭 0
雞鳴 13
雞鳴不已 1
雞鳴狗盜 7
雞鴨 21
雞鴨魚肉 10
雟 82
雡 2
離 4017
離不了 4
離不開 80
離不開手 0
離世 48
離久情疏 0
離了 61
離任 2
離別 123
離別多年 0
離別已久 0
離去 537
離合 34
離合器 128
離合體詩 0
離地 81
離垢 1
離垢地 1
離境 78
離境手續 4
離奇 95
離奇命案 1
離奇失蹤 6
離婚 597
離婚率 40
離婚者 11
離子 1104
離子交換 58
離子交換法 4
離子化 44
離子燙 1
離宮 9
離家 135
離家出走 190
離岸 97
離島 601
離市 23
離席 51
離座 4
離弦 4
離得開 0
離心 176
離心作用 0
離心力 25
離心機 59
離心離德 2
離恨 5
離情 50
離愁 28
離散 77
離散數學 95
離析 2
離校 159
離棄 48
離河 4
離海 4
離異 39
離異之心 0
離經叛道 7
離線 0
離群 13
離群索居 6
離職 586
離華 20
離解 6
離譜 147
離身 9
離鄉 21
離鄉背井 34
離開 3330
離間 9
離間計 1
離題 49
離騷 26
難 458
難 17411
難上加難 27
難上難 2
難不住 0
難不倒 29
難了 56
難事 64
難以 1824
難以出口 0
難以啟齒 24
難以形容 26
難以捉摸 28
難以為繼 2
難以相信 28
難以相處 1
難以置信 44
難以解決 21
難住 1
難保 87
難倒 20
難兄 1
難兄難弟 22
難免 561
難免會 96
難出 3
難分 44
難分難捨 6
難分難解 10
難分高下 3
難到 107
難友 9
難受 90
難句 0
難句 0
難吃 30
難堪 75
難字 15
難守 3
難容 31
難容於 1
難度 438
難得 981
難得 99
難忍 35
難忘 544
難怪 655
難怪會 65
難懂 81
難打 22
難找 67
難捨 60
難捨難分 5
難掩天下人耳目 1
難掩耳目 100
難掩 500
難於 68
難易 70
難易度 63
難民 227
難民收容所 0
難民營 8
難治 24
難為 101
難為情 21
難產 67
難用 73
難當 24
難的 99
難看 129
難管 7
難纏 49
難耐 56
難聞 19
難聽 77
難胞 6
難能 14
難能可貴 97
難色 2
難處 71
難解 117
難解難分 2
難言 26
難言之隱 12
難說 34
難讀 13
難辦 11
難辭其咎 0
難逃 112
難逃一死 10
難逃法網 2
難過 735
難道 2042
難道說 33
難關 111
難題 410
難點 10
雤 0
雥 49
雦 0
雧 0
雨 40
雨 4091
雨中 169
雨中散步 1
雨傘 167
雨傘套 0
雨傘節 16
雨具 34
雨刷 72
雨勢 57
雨天 168
雨季 165
雨帽 1
雨後 72
雨後春筍 83
雨散雲收 0
雨林 128
雨棚 7
雨水 431
雨沐風餐 0
雨淋日晒 0
雨淋日曬 1
雨滴 75
雨絲 24
雨聲 73
雨蛙 12
雨衣 107
雨過 15
雨過天晴 15
雨過天青 8
雨量 241
雨量器 3
雨量計 6
雨雪 11
雨雲 4
雨雷 0
雨霧 2
雨露 28
雨靴 3
雨鞋 25
雨順風調 0
雨點 27
雩 5906
雪 482
雪 4827
雪上 23
雪上加霜 102
雪中 95
雪中送炭 90
雪亮 41
雪人 61
雪佛蘭 89
雪佛蘭車 0
雪冤 0
雪利酒 0
雪原 27
雪地 139
雪地冰天 0
雪域 1
雪堆 7
雪天 13
雪季 10
雪山 292
雪山山脈 29
雪山山脈 29
雪崩 10
雪恥 20
雪恥報仇 0
雪恥復國 0
雪恨 7
雪戰 0
雪景 43
雪朵 0
雪梨 636
雪橇 34
雪橇 34
雪水 7
雪泥 44
雪泥鴻爪 41
雪泥鴻爪 41
雪片 24
雪球 29
雪白 140
雪白 140
雪白色 0
雪盲 6
雪祭 16
雪窖冰天 0
雪糕 19
雪線 7
雪膚 0
雪花 195
雪花片片 4
雪花膏 6
雪花遍地 0
雪花遍地 0
雪茄 145
雪茄煙 2
雪莉 116
雪萊 6
雪虐風饕 0
雪裡 3
雪裡紅 10
雪車 24
雪里紅 0
雪青 0
雪麗 17
雬 0
雬 0
雭 0
雭 0
雮 0
雯 2627
雰 56
雱 48
雲 9068
雲中白鶴 0
雲南 186
雲南白藥 5
雲南省 43
雲合霧集 0
雲吞 7
雲天 43
雲室 0
雲層 100
雲山 90
雲岡 1
雲岡石佛 0
雲岡石窟 5
雲嶺 4
雲彩 136
雲影 14
雲景 2
雲朵 26
雲杉 25
雲林 3462
雲林人 14
雲林工專 57
雲林科技大學 1
雲林縣 8557
雲梢 0
雲梯 35
雲梯車 1
雲母 48
雲母片 7
雲母石 0
雲氣 39
雲河 38
雲海 178
雲消霧散 1
雲淡風輕 35
雲漢 23
雲煙 38
雲端 109
雲篦 0
雲翳 0
雲英 17
雲表 8
雲譎波詭 0
雲譎波詭 0
雲貴 4
雲貴高原 9
雲遊 104
雲遊四方 3
雲量 21
雲門 70
雲門舞集 68
雲開見日 0
雲雀 54
雲集 143
雲雨 30
雲霄 27
雲霄飛車 44
雲霓 6
雲霞 31
雲霧 112
雲頂 39
雲髻 1
雲鬚 0
雲鬟 0
雲鬢 7
雲麾勳章 6
雲龍 102
雴 0
雵 9
零 5722
零丁 7
零下 126
零下溫度 0
零亂 41
零件 3352
零件圖 6
零元 2
零兒 0
零兒 0
零分 104
零南 2
零售 1726
零售價 164
零售價格 90
零售商 123
零售店 50
零售業 657
零嘴 22
零壹 103
零壹公司 0
零字 1
零存整付 67
零工 32
零度 33
零度以上 1
零度以下 6
零打碎敲 0
零散 45
零敲碎打 0
零數 4
零星 250
零時 95
零東 0
零活 1
零用 5
零用金 94
零用錢 81
零碎 53
零組件 2837
零花 3
零落 23
零號 39
零蛋 2
零賣 3
零賣商 0
零路 0
零週期 0
零錢 66
零雜 1
零零 52
零零亂亂 2
零零散散 4
零零星星 9
零零碎碎 9
零零落落 10
零頭 14
零食 101
零點 384
雷 3760
雷光 39
雷克雅未克 0
雷公 234
雷公根 0
雷利 12
雷劈 3
雷動 7
雷厲風行 17
雷同 83
雷射 2884
雷射印表機 0
雷射槍 16
雷峰塔 8
雷州半島 5
雷擊 43
雷根 56
雷汞 2
雷池 3
雷火 4
雷管 20
雷聲 33
雷聲大作 2
雷聲大雨點小 4
雷虎小組 11
雷諾 248
雷諾數 11
雷轟 3
雷轟電閃 0
雷達 570
雷達站 33
雷達網 2
雷鋒 0
雷陣雨 28
雷雨 95
雷雨交加 9
雷雨雲 4
雷電 123
雷電交加 6
雷霆 60
雷霆萬鈞 24
雷霆萬鈞之勢 2
雷音 8
雷鳥 10
雷鳴 26
雷鳴聲 0
雷鳴般 0
雷龍 37
雸 9
雹 26
雹子 0
雹霰 0
雺 57
電 11913
電中性 2
電介質 4
電位 168
電位器 2
電位差 11
電位能 0
電信 8176
電信局 370
電信號 3
電傳 214
電傳機 2
電傳視訊 258
電價 134
電光 340
電光石火 14
電冰箱 787
電刑 0
電刷 3
電力 2102
電力公司 176
電力學 1
電力工業 42
電力廠 0
電力站 1
電力網 2
電力線 12
電功率 32
電動 2083
電動勢 10
電動學 0
電動打字 0
電動打字機 0
電動機 203
電動窗 321
電動車 154
電動門 4
電勢差 0
電化 167
電化學 136
電化教學 16
電匠 30
電匯 87
電台 1935
電吉他 1
電告 14
電唁 0
電唱 0
電唱機 15
電唱針 0
電單車 12
電器 1638
電器化 5
電器行 431
電報 150
電報局 12
電報機 6
電報費 0
電場 11
電場 118
電壓 1362
電壓互感器 0
電壓表 19
電壓計 14
電壺 1
電子 54370
電子伏特 7
電子元件 145
電子商務 0
電子報 0
電子學 233
電子展 141
電子工業 918
電子工業研究所 23
電子所 2375
電子書 0
電子束 122
電子槍 10
電子流 4
電子琴 144
電子科 505
電子管 29
電子系 143
電子計算機 811
電子郵件 0
電子錶 22
電子雞 0
電子雲 0
電子電路 162
電子顯微鏡 273
電學 133
電宰 22
電宰肉 0
電容 365
電容器 297
電導 51
電導飛彈 0
電導體 2
電工 714
電工學 19
電平 8
電廠 1044
電弧 65
電影 11276
電影劇本 22
電影圈 6
電影明星 31
電影界 27
電影票 62
電影迷 9
電影院 613
電感 150
電感應 4
電扇 90
電扶梯 1
電抗 2
電插座 1
電插頭 6
電擊 160
電文 16
電料 19
電料行 45
電晶體 268
電暈 3
電杆 0
電板 5
電桿 80
電梯 1115
電梯間 36
電棒 15
電椅 7
電極 285
電槍 3
電樞 10
電橋 17
電機 5600
電機工程 1436
電機科 246
電機系 1169
電檢 22
電氣 1096
電氣化 26
電氣用品 15
電氣石 10
電池 2557
電池組 24
電波 227
電波 227
電泳 120
電洽 373
電流 801
電流強度 1
電流繼電器 4
電流表 12
電流計 5
電渣爐 0
電源 2118
電源開關 40
電滲析 2
電漿 0
電火花 3
電灶 1
電烙鐵 5
電烤箱 9
電焊 132
電焊工 3
電熨斗 18
電熱 182
電熱器 30
電燈 169
電燈架 0
電燈柱 0
電燈泡 69
電燙 1
電爐 45
電玩 694
電玩 6944
電瓶 105
電療 77
電療法 8
電眼 31
電石 16
電石氣 1
電碼 24
電碼本 0
電磁 655
電磁場 62
電磁場 62
電磁學 114
電磁感應 3
電磁振盪 0
電磁振蕩 0
電磁波 207
電磁波 207
電磁爐 0
電磁輻射 16
電磁鐵 16
電稿 7
電站 13
電筒 15
電算 68
電算中心 2120
電網 468
電線 905
電線杆 6
電線桿 63
電線走火 24
電纜 917
電纜線 53
電聯車 54
電能 150
電腦 77345
電腦中心 717
電腦化 1995
電腦教室 1213
電腦程式 359
電腦老師 55
電腦課 65
電臺 224
電花 7
電荷 134
電蚊拍 1
電蚊香 1
電衝 0
電表 126
電覆 0
電視 6092
電視公司 99
電視劇 103
電視台 1163
電視廣播 70
電視廣播 70
電視接收 16
電視接收機 16
電視機 496
電視牆 72
電視片 3
電視界 13
電視直播 3
電視直播 3
電視直播衛星 0
電視直播衛星 0
電視節目 412
電視網 128
電視臺 36
電視觀眾 20
電視迷 4
電視頻道 58
電視影集 500
電解 223
電解槽 22
電解液 24
電解液 24
電解質 138
電解電容 184
電解電容器 48
電訊 1221
電訊局 5
電訊稿 0
電話 39101
電話亭 23
電話局 7
電話會議 9
電話機 239
電話筒 11
電話簿 194
電話線 321
電話號碼 1228
電話費 252
電話鈴 16
電請 13
電負性 1
電費 109
電賀 197
電路 1859
電路圖 61
電路板 700
電車 235
電車站 5
電邀 1
電量 190
電鈕 2
電鈴 24
電銲 23
電鋸 17
電錶 170
電鍋 46
電鍋爐 0
電鍍 435
電鍍法 1
電鍵 1
電鎗 1
電鏟 0
電鐘 0
電鑽 26
電門 7
電阻 984
電阻器 150
電除塵器 0
電離 18
電離作用 0
電離子 13
電離層 61
電離氣 1
電離能 0
電風扇 73
電魚 9
電鰻 1
雼 0
雽 40
雽 40
雿 98
需 12567
需款 2
需款孔急 6
需氧量 11
需求 11969
需求量 363
需求面 56
需用 202
需要 16391
需要量 84
霂 37
霃 0
霄 86
霄壤 0
霄壤之別 0
霄漢 3
霅 69
霆 96
震 2170
震動 441
震古爍今 1
震古鑠今 2
震垮 8
震天 73
震天價響 6
震天動地 0
震天駭地 0
震央 27
震幅 14
震怒 35
震悸 0
震悼 0
震慄 3
震憾 73
震懼 0
震懾 33
震撼 510
震撼作用 0
震撼力 33
震撼性 24
震於 0
震旦 624
震波 96
震波 96
震源 32
震源地 0
震災 62
震盪 380
震盪不安 0
震盪性 0
震矜 0
震破 3
震耳 16
震耳欲聾 29
震蕩 9
震裂 2
震震 12
震顫 65
震風 0
震駭 9
震驚 211
霈 127
霈然 0
霉 304
霉乾菜 0
霉味 10
霉天 0
霉氣 6
霉氣衝天 1
霉潮 0
霉爛 1
霉菌 45
霉運 3
霉運當頭 1
霉雨 1
霊 0
霋 4
霌 0
霍 1280
霍亂 175
霍亂菌 8
霍元甲 1
霍去病 1
霍地 13
霍山 0
霍布斯 22
霍布森 0
霍查 0
霍桑 4
霍然 22
霍爾 50
霍霍 3
霎 44
霎時 48
霎時之間 2
霎時間 15
霎那 11
霎那之間 0
霎那間 6
霏 45
霏霏 21
霏霏細雨 1
霐 5
霑 22
霑恩 2
霑惠 0
霑惹 0
霑染 0
霑潤 0
霑濡 0
霑衣 2
霑衿 0
霑被 0
霑襟 0
霑霈 0
霒 1
霓 97
霓紅柱 0
霓紅燈 2
霓虹 209
霓虹燈 66
霔 0
霕 0
霖 1913
霖雨 0
霗 0
霗 0
霘 6
霙 27
霙 27
霚 0
霛 0
霜 9955
霜凍 2
霜化 0
霜天 1
霜害 0
霜期 0
霜淇淋 26
霜絮 0
霜葉 1
霜降 13
霜降牛肉 1
霜雪 22
霜霉病 0
霜鬢 0
霝 10
霞 2099
霞光 9
霞冠 0
霞天 6
霞帔 0
霞彩 1
霞暮迷人 0
霞照 1
霞輝 2
霞雲 27
霟 16
霠 37
霡 0
霢 7
霣 8
霤 24
霥 5
霦 6
霧 1379
霧化器 3
霧台 82
霧峰 926
霧氣 55
霧滴 13
霧社 276
霧裡 17
霧裡看花 15
霨 2
霩 16
霪 25
霪雨 5
霫 2
霬 5
霮 2
霯 3
霰 122
霰彈 1
霱 0
露 1898
露 1898
露一下 2
露一下 2
露一下 2
露一下 2
露一手 8
露一手 8
露一手 8
露一手 8
露了 26
露了 26
露光 3
露光 3
露出 738
露出 738
露出來 19
露出來 19
露出原形 0
露出原形 0
露出去 0
露出去 0
露出破綻 8
露出破綻 8
露出馬腳 10
露出馬腳 10
露台 37
露在 26
露在 26
露天 280
露天劇場 7
露天劇場 7
露天劇場 7
露天礦 1
露天開採 0
露天電影 4
露天電影 4
露宿 24
露宿街頭 4
露宿風餐 0
露屍 0
露屍 0
露布 2
露底 1
露底 1
露怯 0
露怯 0
露才 2
露才 2
露有 1
露有 1
露水 45
露水夫妻 0
露水河 0
露營 1159
露營地 36
露牙 0
露牙 0
露珠 41
露白 13
露白 13
露相 1
露相 1
露背 0
露背 0
露背裝 1
露背裝 1
露臉 55
露臉 55
露臺 7
露苗 0
露苗 0
露著 11
露著 11
露袒 0
露袒 0
露酒 2
露雨 0
露雨 0
露面 82
露面 82
露頭 19
露頭 19
露頭角 8
露頭角 8
露餡兒 0
露餡兒 0
露餡兒 0
露餡兒 0
露馬腳 23
露馬腳 23
露骨 300
露骨 300
露點 55
露點 55
露齒 5
露齒 5
霳 0
霴 0
霵 149
霶 0
霷 0
霸 1450
霸主 375
霸佔 56
霸佔住 0
霸占 5
霸圖 0
霸業 30
霸機 0
霸權 163
霸權主義 17
霸氣 41
霸王 336
霸王硬上弓 5
霸王鞭 1
霸者 8
霸道 76
霹 96
霹雷 1
霹靂 2680
霹靂彈 4
霹靂砲 1
霹靂車 30
霹靂遊俠 10
霺 7
霼 0
霽 130
霽月 0
霾 410
霾晦 0
霿 17
霿淞 1
靀 0
靁 0
靂 126
靃 98
靃 98
靄 60
靄靄 4
靅 0
靆 135
靇 109
靈 5700
靈丹 36
靈丹妙藥 5
靈位 17
靈光 97
靈前 18
靈力 30
靈動 42
靈台 7
靈堂 53
靈壇 0
靈妙 4
靈媒 39
靈寶 36
靈山 100
靈巧 78
靈巧性 2
靈床 1
靈府 2
靈座 0
靈性 188
靈感 339
靈感一來 0
靈應 4
靈效 1
靈敏 157
靈敏度 125
靈旛 0
靈柩 19
靈機 5
靈機一動 31
靈武 3
靈氣 81
靈活 524
靈活性 35
靈活機動 0
靈活運用 72
靈渠 4
靈牌 2
靈犀 25
靈界 50
靈異 246
靈異傳奇 10
靈秀 14
靈符 3
靈精 73
靈肉 9
靈臺 2
靈芝 278
靈芝草 4
靈藥 33
靈車 13
靈通 34
靈通人士 0
靈長目 13
靈長類 53
靈隱寺 21
靈驗 41
靈骨塔 1
靈魂 1017
靈魂深處 16
靉 137
靉靆 0
靊 0
靋 41
靐 0
靑 0
靑 0
青 6
青 10929
青光 23
青光眼 197
青冢 3
青出於藍 14
青史 19
青商 164
青商會 92
青天 68
青天白日 34
青天白日滿地紅 21
青天霹靂 1
青少年 3302
青少年時期 25
青少棒 51
青山 427
青山常在 6
青山綠水 79
青島 201
青島市 15
青島東路 48
青島西路 20
青州 13
青工 98
青幫 26
青年 5114
青年人 99
青年令 0
青年商會 127
青年團 28
青年救國團 4
青年日報 54
青年會 68
青年有為 0
青年期 8
青年節 43
青年軍 2
青年輔導委員會 282
青年黨 44
青康滇縱谷高原 0
青康藏 2
青康藏高原 13
青徽素 1
青戈江 0
青斑蝶 1
青春 1211
青春偶像 13
青春期 185
青春永駐 20
青春痘 280
青木瓜 1
青松 62
青果 126
青梅 14
青梅竹馬 47
青棒 44
青椒 57
青樓 21
青樓艷妓 0
青樓集 1
青江菜 1
青海 129
青海人 1
青海省 29
青海苔 0
青澀 279
青灰 7
青燈 5
青玉 19
青瓦台 28
青瓦臺 0
青瓷 34
青田 51
青白色 11
青皮 11
青眼 9
青睞 299
青石 20
青石板 6
青碧 0
青碧色 0
青磁 2
青稞 9
青竹絲 15
青筋 18
青粗飼料 0
青紅皂白 14
青紫 13
青絲 24
青綠 35
青綠色 14
青翠 64
青翠色 0
青色 98
青花 81
青花魚 2
青苔 75
青苗 5
青苗法 1
青草 152
青草地 4
青草湖 42
青草茶 1
青菜 107
青菜湯 1
青菜豆腐湯 0
青葉 67
青蒜 2
青蒼 3
青蓮 13
青蓮居士 1
青蔥 34
青蘋果 79
青蚵 8
青蛙 384
青蛙肉 1
青蛙腿 0
青蠅 0
青衣 118
青衫 50
青衫淚 50
青衿 5
青豆 25
青貯 15
青貯窖 0
青貯飼料 1
青輔會 420
青醬 1
青銅 150
青銅像 1
青銅器 32
青銅器時代 0
青銅時代 1
青雀 1
青雲 305
青雲直上 1
青青 183
青青草原 15
青面獠牙 3
青飼料 2
青魚 13
青鱸 0
青鳥 0
青鹽 3
青黃不接 8
青黑 2
青黴 1
青黴素 9
青龍 112
靕 0
靖 1384
靖亂 1
靖康 1
靖邊 0
靖難 1
靗 0
靘 66
静 0
靚 249
靛 43
靛紫 5
靛紫色 0
靛藍 35
靛藍色 7
靛青 3
靜 5573
靜下 5
靜下來 35
靜下去 0
靜下心來 26
靜以制動 0
靜修 80
靜候 36
靜僻 2
靜力 45
靜力學 16
靜坐 540
靜坐療法 0
靜坐示威 6
靜壓 31
靜夜 38
靜如處女 0
靜定 32
靜宜 880
靜宜大學 1
靜宜女子大學 4
靜宜女子文理學院 1
靜室 8
靜寂 33
靜居寺 0
靜平衡 2
靜心 136
靜悄悄 38
靜悄悄 38
靜態 580
靜態平衡 5
靜摩擦力 8
靜極思動 5
靜止 180
靜止不動 14
靜水 17
靜物 56
靜物畫 8
靜磨擦 3
靜穆 8
靜穆無聲 0
靜立 14
靜美 32
靜聽 17
靜脈 827
靜脈 827
靜脈曲張 0
靜脈曲張 0
靜脈炎 18
靜脈炎 18
靜脈瘤 29
靜脈瘤 29
靜脈血 18
靜脈血 18
靜臥 7
靜臥在 0
靜舍 2
靜觀 70
靜觀其變 25
靜謐 63
靜讀 0
靜配合 0
靜電 409
靜電健康 0
靜電健康器 0
靜電力 1
靜電學 5
靜電感應 1
靜電荷 13
靜靜 335
靜靜地 214
靜靜的 214
靜音 71
靜風 5
靜養 16
靜默 85
靜默致哀 0
靝 0
非 16546
非一日之功 0
非一日之功 0
非上 13
非上不可 20
非下 3
非下不可 0
非也 50
非交不可 0
非人 86
非人性 7
非人生活 1
非他 15
非他不可 0
非他莫屬 2
非付不可 0
非但 197
非住不可 0
非佔不可 0
非你 17
非你不可 1
非你莫屬 0
非來 7
非來不可 3
非信不可 0
非倒不可 0
非借不可 0
非傷即死 0
非公 105
非公勿入 0
非其 52
非其所為 0
非其所能 0
非再 15
非凡 821
非出 13
非出不可 0
非分 6
非分之想 2
非分之財 0
非到 9
非到不可 0
非刺不可 0
非割不可 0
非動不可 0
非包不可 0
非卿莫娶 0
非去 11
非去不可 11
非受迫性 0
非叫不可 0
非可小覷 0
非吃 1
非吃不可 5
非同 43
非同小可 29
非吐不可 1
非君 2
非君莫嫁 0
非君莫屬 0
非命 5
非售不可 0
非問 3
非問不可 0
非在 35
非報不可 0
非塗不可 0
非奏不可 0
非她 3
非她不可 0
非她莫屬 1
非如 33
非妳 2
非妳不可 0
非妳莫屬 1
非始料所及 0
非婚 0
非婚生 1
非婚生子 6
非婚生子女 18
非守不可 0
非宰不可 0
非審不可 1
非將 10
非對抗性 2
非導體 6
非常 12900
非常時期 30
非常會議 0
非常重要 461
非往 2
非得 223
非得 223
非得不可 0
非得到 1
非想不可 1
非愛 7
非愛不可 1
非成 6
非我 65
非我不可 2
非我族類 3
非我莫屬 6
非戒不可 0
非戰 7
非戰公約 1
非打 1
非打不可 0
非把 16
非抱不可 0
非抹不可 0
非接不可 1
非揍不可 0
非摸不可 0
非撕不可 0
非播不可 0
非播不可 0
非收不可 0
非攻不可 0
非教不可 0
非整不可 0
非斬不可 0
非斷不可 0
非於 20
非早不可 0
非晶體 2
非暴力 13
非會不可 0
非有 98
非有不可 0
非服不可 0
非核 46
非歉不可 0
非正式 231
非此 16
非此不可 0
非死不可 7
非死即傷 5
非殺不可 1
非毒不可 0
非比 3
非比尋常 23
非求不可 0
非池中物 2
非法 914
非法交易 0
非法性 4
非法行為 17
非洲 1514
非洲之角 0
非洲人 21
非洲統一 0
非洲統一組織 0
非深不可 0
非淺 12
非清不可 0
非漲不可 0
非然 1
非煮不可 0
非營業支出 0
非營業收入 12
非爭不可 0
非特 47
非獨 52
非生產性 5
非用 30
非用不可 3
非畫不可 0
非當 24
非當不可 0
非發不可 0
非睡不可 0
非短不可 0
非磁性 3
非磨不可 0
非禮 41
非禮勿視 2
非穿不可 0
非站不可 0
非笑 16
非簽不可 0
非結不可 0
非經 1896
非線性元件 3
非線性失真 0
非繳不可 0
非罰不可 0
非翻不可 0
非肥皂 0
非與 12
非補不可 0
非要 109
非要不可 1
非親非故 1
非訟事件 62
非說 2
非說不可 3
非請 2
非請勿入 5
非請莫入 2
非講不可 0
非議 83
非讀不可 0
非讓 4
非讓不可 0
非買 2
非買不可 1
非貼不可 0
非賣品 90
非賭不可 0
非贏 0
非贏不可 8
非走不可 1
非跌不可 0
非跑 0
非跑不可 0
非軍事 17
非軍事區 2
非載不可 0
非輕不可 0
非通不可 0
非造不可 0
非過 5
非過不可 0
非金屬 129
非金屬元素 0
非金屬性 1
非附不可 0
非降不可 0
非隔不可 0
非難 25
非電解質 1
非非 22
非類 54
非騎不可 0
非驢非馬 1
非點不可 0
非點源 11
非黨員 0
靟 0
靠 2601
靠上 27
靠上來 0
靠上去 0
靠不住 9
靠了 23
靠住 0
靠傍 0
靠到 8
靠前 0
靠前面 0
靠北 7
靠北邊 0
靠右 23
靠右走 1
靠右邊 7
靠右邊走 1
靠在 106
靠墊 24
靠外 13
靠外邊 0
靠外邊走 0
靠外面 0
靠外面走 0
靠天吃飯 2
靠好 2
靠山 50
靠山吃山 8
靠山邊 0
靠岸 30
靠岸邊 0
靠左 35
靠左走 0
靠左邊 1
靠左邊走 0
靠後 7
靠後面 0
靠得 20
靠得住 7
靠手 5
靠攏 23
靠攏過來 2
靠椅 4
靠水 2
靠水吃水 0
靠水面 0
靠河 1
靠河邊 0
靠海 29
靠海吃海 5
靠海邊 2
靠港 12
靠港口 0
靠湖 1
靠湖邊 0
靠牆 15
靠牆角 0
靠牆邊 0
靠窗 23
靠窗戶 3
靠背 10
靠背椅 1
靠腰 10
靠自己 178
靠著 411
靠行 153
靠裡面 0
靠裡面走 0
靠起 1
靠起來 0
靠近 750
靠進 16
靠進來 0
靠進去 0
靠邊 23
靠邊走 1
靠門 2
靠門邊 0
靠點 7
靡 486
靡 486
靡爛 5
靡爛不堪 0
靡爛性 0
靡靡之音 26
靡麗 0
面 14101
面上 381
面上無光 1
面不改色 11
面交 9
面值 70
面具 270
面前 1306
面北 3
面向 464
面呈 24
面告 6
面商 1
面善 7
面善心惡 1
面囑 0
面壁 11
面壁思過 6
面奏 1
面如 29
面如土色 3
面如金紙 1
面嫩 0
面子 259
面孔 110
面容 71
面對 5321
面對現實 62
面對面 324
面山 33
面巾 1
面帶 23
面帶微笑 19
面帶笑容 17
面惡 0
面惡心善 4
面折 1
面授 62
面授機宜 10
面斥 0
面晤 0
面會 31
面有 161
面有喜色 0
面有菜色 1
面有難色 5
面朝 33
面朝著 0
面板 675
面泛 1
面洽 60
面海 26
面無 58
面無人色 2
面無血色 6
面熟 7
面牆 43
面生 2
面皂 12
面皮 45
面皰 63
面盆 45
面目 111
面目一新 1
面目一新 1
面目全非 23
面目可憎 11
面相 112
面稟 1
面積 4469
面積分 5
面紅 11
面紅耳熱 2
面紅耳赤 19
面紗 182
面紙 124
面罩 101
面膜 1
面臨 3355
面臨到 41
面臨著 52
面色 63
面色蒼白 6
面薄 2
面西 7
面見 47
面試 389
面試官 1
面談 367
面請 12
面諭 1
面謁 0
面謝 1
面議 143
面貌 515
面貌一新 0
面貌一新 0
面貌全非 0
面邀 2
面部 86
面陳 1
面霜 31
面面 18
面面俱到 42
面面相覷 27
面面觀 227
面頰 27
面額 98
面黃 5
面黃肌瘦 4
面龐 12
靣 0
靤 0
靤 0
靦 15
靦 15
靦汗 0
靦然 0
靦腆 43
靦臉 0
靦顏 0
靧 0
靨 57
革 117
革 1174
革命 1171
革命人生 0
革命人生觀 0
革命先烈 12
革命化 1
革命報 0
革命委員 0
革命委員會 3
革命家 10
革命幹勁 0
革命性 152
革命意志 0
革命戰爭 3
革命文學 0
革命殘廢軍人 0
革命烈士 1
革命烈火 0
革命犯 1
革命現實主義 0
革命發展 0
革命發展階段論 0
革命知識 0
革命知識分子 0
革命知識分子 0
革命積極 0
革命積極性 0
革命英雄 3
革命英雄主義 0
革命軍 12
革命軍人 1
革命鬥志 0
革命黨 5
革命黨人 4
革委會 2
革履 3
革左五營 0
革心 5
革新 822
革新派 0
革新運動 9
革正 0
革職 41
革職留任 0
革退 0
革除 26
革面洗心 0
靪 4
靫 0
靬 38
靭 0
靮 19
靯 0
靰 45
靲 4
靳 118
靴 169
靴子 26
靴尖 0
靴底 0
靴釘 0
靵 0
靶 84
靶台 1
靶台上 0
靶場 36
靶場 36
靶子 5
靶心 3
靶機 0
靷 4
靸 4
靸 4
靹 0
靺 77
靻 24
靼 12
靽 56
靾 81
靿 171
鞀 50
鞁 31
鞂 38
鞃 56
鞄 41
鞅 39
鞅 39
鞇 0
鞈 70
鞉 0
鞊 31
鞋 1973
鞋刷 4
鞋匠 4
鞋印 6
鞋子 446
鞋帶 45
鞋幫 1
鞋底 88
鞋底抹油 0
鞋店 65
鞋廠 56
鞋扣 12
鞋拔子 0
鞋架 11
鞋根 1
鞋業 309
鞋櫃 37
鞋油 10
鞋盒 14
鞋裡 1
鞋襪 26
鞋跟 26
鞋面 62
鞋類 213
鞍 205
鞍子 1
鞍尾 0
鞍山 30
鞍架 0
鞍轡 1
鞍部 104
鞍鋼憲法 0
鞍革 1
鞍馬 57
鞎 50
鞏 91
鞏固 431
鞏固國防 6
鞏固基地 0
鞏縣 2
鞏膜 25
鞏金甌 1
鞓 0
鞔 5
鞔 5
鞕 0
鞖 0
鞗 20
鞘 176
鞘翅 13
鞙 2
鞚 2
鞛 0
鞜 5
鞝 4
鞞 3
鞟 0
鞟 0
鞟 0
鞠 269
鞠育 1
鞠躬 82
鞠躬盡悴 0
鞠躬盡瘁 12
鞠躬禮 5
鞠養 0
鞡 9
鞢 609
鞢 609
鞣 115
鞣劑 5
鞣製 3
鞣質 2
鞣酸 8
鞤 4
鞥 2812
鞦 34
鞦韆 33
鞧 0
鞧 0
鞨 4
鞩 0
鞪 1
鞫 3
鞬 10
鞭 279
鞭刑 14
鞭子 40
鞭尸 0
鞭屍 2
鞭形 1
鞭打 21
鞭撻 1
鞭毛 18
鞭毛藻 9
鞭毛蟲 19
鞭炮 69
鞭炮聲 17
鞭砲 3
鞭笞 3
鞭策 69
鞭箠 0
鞭蟲 19
鞭辟入裡 4
鞭長莫及 3
鞮 5
鞰 0
鞱 0
鞱 0
鞲 0
鞲 0
鞲 0
鞳 796
鞴 0
鞵 0
鞵 0
鞶 248
鞷 742
鞸 0
鞹 2
鞺 0
鞻 4
鞼 0
鞼 0
鞽 0
鞾 0
鞿 26
韀 0
韀 0
韁 151
韁繩 7
韂 0
韃 116
韃子 8
韃虜 3
韃靼 9
韃靼人 1
韃韃 1
韄 7
韅 6
韆 7
韇 40
韈 0
韉 9
韊 0
韋 1122
韋伯 183
韋氏音標 0
韋瓦第 22
韋眉子 0
韋蓆 0
韌 376
韌帶 150
韌度 8
韌性 84
韌挺 0
韌皮 1
韌皮纖維 3
韌皮部 2
韌體 1
韍 206
韎 51
韏 129
韐 53
韑 0
韒 0
韓 2954
韓人 5
韓信 15
韓信點兵 3
韓僑 12
韓劇 100
韓國 2566
韓國人 43
韓國史 3
韓國泡菜 3
韓國菜 3
韓國製 4
韓國話 0
韓國貨 0
韓國隊 11
韓德爾 45
韓愈 38
韓戰 38
韓文 157
韓村 2
韓江流域 0
韓片 1
韓琦 2
韓福瑞 4
韓製 5
韓詩外傳 2
韓語 40
韓非 4
韓非子 45
韔 5
韕 6
韖 74
韖 74
韗 223
韗人 1
韘 117
韙 378
韚 0
韛 0
韜 337
韜光 3
韜光養晦 6
韜晦 1
韜略 16
韝 6
韝上鷹 1
韞 383
韟 323
韠 0
韡 6
韢 0
韢 0
韢 0
韣 23
韤 0
韥 111
韭 126
韭菜 54
韭菜花 1
韭菜苗 0
韭黃 7
韮 0
韯 0
韯 0
韰 4
韱 10
韲 0
韲 0
音 30409
音同 100
音位 9
音信 5
音信全無 0
音信渺茫 0
音分 4
音區 6
音叉 6
音名 8
音和 20
音品 1
音型 50
音域 83
音容 5
音容宛在 1
音帶 48
音律 27
音感 56
音效 2618
音栓 6
音樂 20662
音樂劇 652
音樂台 122
音樂史 53
音樂學院 112
音樂家 510
音樂廳 540
音樂性 94
音樂會 1112
音樂演奏 45
音樂營 22
音樂片 17
音樂界 58
音樂節 126
音樂系 918
音標 75
音波 64
音波 64
音準 14
音界號 0
音程 47
音符 265
音節 40
音素 11
音義 13
音耗 0
音色 299
音色甜美 2
音訊 184
音訊全無 13
音訊渺茫 1
音調 76
音譯 58
音質 212
音速 140
音部 5
音量 359
音階 68
音韻 56
音韻學 17
音響 2738
音響效果 31
音頻 70
音高 25
韴 0
韵 0
韶 163
韶光 9
韶光荏苒 0
韶山 10
韶秀 0
韶華 8
韶華如駛 1
韶關 22
韶顏 3
韷 0
韸 2
韹 7
韺 40
韻 1186
韻事 6
韻味 58
韻律 216
韻律操 15
韻律舞 1
韻文 34
韻書 4
韻母 9
韻腳 4
韻致 7
韼 0
韽 4
韽韽 1
韾 10
響 2107
響了 86
響亮 97
響動 4
響尾蛇 27
響度 4
響徹 21
響徹雲霄 8
響應 356
響板 11
響聲 44
響起 345
響起來 10
響遍 15
響遍 15
響遏行雲 0
響鈴 29
響馬 6
響馬黨羽 0
頀 5
頁 77248
頁岩 80
頁巖 0
頁數 3352
頁次 452
頁碼 89
頁面 80
頁頁 213
頂 3901
頂上 85
頂上來 0
頂上去 1
頂下 5
頂下來 0
頂下去 0
頂不 4
頂不 4
頂不住 1
頂不起 0
頂不起來 0
頂了 8
頂事 0
頂住 17
頂來頂去 0
頂個 2
頂冒 0
頂凶 0
頂到 13
頂名 0
頂吹 1
頂呱呱 6
頂嘴 18
頂回 4
頂回來 0
頂回去 3
頂在 15
頂塔 1
頂多 209
頂多在 3
頂多是 43
頂多會 1
頂多有 1
頂多能 1
頂天 11
頂天立地 18
頂好 177
頂好市場 1
頂好市場 1
頂尖 345
頂尖人物 11
頂尖高手 13
頂層 44
頂層樓 2
頂峰 50
頂戴 5
頂撞 11
頂替 22
頂板 3
頂棚 8
頂樓 163
頂標 0
頂牛 0
頂珠 1
頂球 11
頂瓜瓜 0
頂用 2
頂禮 49
頂禮膜拜 7
頂端 1084
頂篷 2
頂級 0
頂缸 4
頂罪 8
頂膜禮拜 0
頂芽 9
頂葉 5
頂著 120
頂蓋 27
頂角 6
頂讓 24
頂起 4
頂起來 2
頂過 2
頂過來 0
頂過去 2
頂部 91
頂針 6
頂針兒 0
頂針兒 0
頂門 12
頂頭 7
頂頭上司 6
頂風 10
頂風冒雨 0
頂點 73
頃 1616
頃 1616
頃之 2
頃刻 17
頃刻之間 4
頃刻間 16
頃接來信 0
頃者 0
頄 8
頄 8
項 13990
項上人頭 100
項圈 85
項城 0
項目 15799
項目表 74
項羽 55
項背 5
項背相望 0
項鍊 149
項鏈 8
項領 3
順 6909
順丁橡膠 0
順乎 5
順位 191
順便 1005
順便來 12
順便到 11
順便去 18
順其自然 53
順利 3151
順勢 158
順勢上漲 0
順化 0
順口 56
順口答應 0
順命 3
順嘴 0
順境 14
順天 254
順天應人 0
順天者昌 0
順差 111
順常 1
順序 1950
順延 149
順延到 1
順從 149
順德 132
順心 103
順心如意 16
順情理 0
順應 248
順應時代 16
順應時勢 3
順應潮流 7
順我者昌 1
順手 189
順手帶走 1
順手牽羊 18
順承 10
順旨 0
順時 7
順時針 23
順時針方向 14
順時鐘 8
順時鐘方向 15
順暢 397
順服 139
順次 6
順民 16
順民者昌 0
順水 112
順水人情 8
順水推舟 12
順水行船 0
順治 26
順流 15
順流而下 10
順理成章 35
順當 1
順眼 89
順耳 8
順著 263
順起來 0
順路 29
順路來 2
順路到 0
順路去 0
順遂 39
順道 126
順道來 9
順道去 8
順適 1
順風 72
順風而呼 0
順風耳 35
順風轉舵 0
頇 20
須 11914
須得 57
須發 21
須知 3368
須臾 28
須要 867
須陀洹 1
頉 0
頊 15
頋 0
頌 774
頌德 10
頌揚 26
頌揚備至 0
頌歌 13
頌歎 0
頌聲 3
頌聲載道 0
頌詞 4
頌詩 3
頌讚 44
頌辭 1
頍 15
頎 37
頎然 0
頏 11
預 4784
預付 232
預付資本 0
預估 2310
預借 189
預備 830
預備士官 16
預備性 7
預備立憲 4
預備立憲公會 0
預備賽 0
預備軍官 50
預備金 478
預備隊 1
預兆 38
預先 554
預分 0
預制 0
預力混凝土 49
預卜 5
預告 1095
預告片 80
預售 670
預售屋 1
預售票 14
預售處 0
預報 364
預官 143
預定 3306
預審 50
預展 20
預想 27
預想出 0
預想到 3
預感 64
預應力 5
預支 15
預料 702
預料不到 2
預料中 33
預料出 0
預料到 16
預期 2651
預測 3195
預測出 6
預測者 2
預演 41
預熱 88
預產期 10
預留 222
預知 232
預示 23
預祝 138
預算 8365
預算案 227
預算表 818
預籌 4
預約 13014
預繳 54
預習 34
預習功課 1
預藏 6
預藏著 0
預處理 6
預處理 6
預製 10
預見 222
預視 128
預覽 10
預言 573
預言家 35
預訂 1399
預計 3600
預設 809
預設底線 0
預設立場 54
預設立場 54
預謀 25
預警 480
預購 109
預賽 186
預選 56
預防 4560
預防為主 1
預防針 71
頑 110
頑劣 23
頑劣分子 0
頑劣分子 0
頑固 99
頑固份子 4
頑固份子 4
頑固派 0
頑強 64
頑愚 3
頑抗 8
頑敵 0
頑昧 0
頑民 0
頑癬 2
頑癬 2
頑皮 269
頑皮鬼 15
頑石 22
頑石點頭 1
頑童 80
頑童歷險記 1
頑逆 0
頑鈍 0
頑靡 0
頒 699
頒佈 0
頒布 168
頒授 43
頒獎 482
頒獎儀式 0
頒獎典禮 361
頒發 639
頒發獎品 7
頒發獎牌 10
頒發獎狀 40
頒發獎盃 1
頒發獎金 25
頒給 114
頒行 29
頒訂 40
頒賜 7
頒賞 0
頒贈 148
頒贈給 2
頓 2573
頓了 41
頓住 5
頓巴教橡園 0
頓巴斯 0
頓悟 51
頓成 9
頓挫 14
頓改 0
頓改前非 0
頓時 320
頓河 0
頓消 4
頓然 20
頓筆 1
頓絕法 0
頓腳 3
頓號 3
頓起 4
頓足 5
頓足捶胸 1
頓音 15
頓首 5
頔 0
頕 0
頕 0
頖 216
頗 2616
頗久 2
頗佳 49
頗具 239
頗具匠心 0
頗具規模 21
頗冷 0
頗多 155
頗好 4
頗得 16
頗想 1
頗感 28
頗感興趣 3
頗成 2
頗有 418
頗欲 1
頗為 453
頗知 1
頗見 9
頗見好評 0
頗覺 13
頗重 21
領 4957
領主 32
領事 174
領事裁判權 1
領事館 94
領人 42
領來 2
領來領去 0
領先 942
領兵 45
領出 12
領出來 6
領出去 0
領到 99
領去 1
領取 1135
領受 292
領受到 7
領口 21
領唱 11
領唱者 0
領單 1
領回 118
領回來 2
領回去 1
領土 341
領土上 9
領地 57
領域 5948
領域內 92
領好 2
領子 14
領導 2469
領導人 623
領導性 24
領導有方 9
領導權 26
領導者 492
領巾 20
領帶 172
領帶夾 23
領帶針 0
領得 63
領得 46
領悟 183
領悟出 7
領悟到 44
領悟力 23
領情 41
領成 0
領扣 1
領教 51
領書 10
領會 78
領會到 7
領有 466
領款 26
領款人 1
領水 1
領洗 5
領海 43
領海權 0
領港 9
領港員 0
領港費 0
領獎 110
領獎人 2
領獎者 9
領班 59
領略 132
領略出 1
領略到 22
領的 46
領空 29
領空權 0
領章 4
領結 15
領罪 0
領航 254
領航員 238
領航圖 0
領航艇 0
領著 58
領表 64
領表格 0
領袖 1321
領袖慾 0
領走 2
領路 6
領軍 0
領進 10
領進來 0
領進去 0
領過 2
領過來 0
領過去 1
領道 2
領釦 0
領銜 40
領錢 21
領隊 543
領隊人 0
領頭 34
領餉 1
領養 200
領養人 0
領點 0
頙 0
頚 0
頛 43
頜 43
頝 55
頝 55
頞 72
頟 0
頟 0
頠 17
頡 90
頡 90
頡頏 3
頢 0
頣 0
頤 172
頤和園 20
頤指氣使 5
頤神 0
頤養 9
頤養天年 4
頥 0
頦 23
頦頷 0
頧 18
頨 55
頩 48
頪 0
頫 12
頫 12
頭 1
頭 10589
頭上 390
頭上腳下 1
頭份 237
頭份鎮 942
頭像 19
頭先 11
頭兒 16
頭兒 16
頭前 103
頭型 13
頭城 237
頭大 75
頭天 9
頭套 44
頭子 23
頭家 214
頭寸 10
頭尾 159
頭巾 53
頭年 0
頭座 5
頭彩 2
頭昏 63
頭昏眼花 27
頭昏腦脹 11
頭暈 123
頭暈目眩 17
頭暈眼花 4
頭會箕斂 0
頭期 4
頭期款 72
頭條 86
頭條新聞 249
頭殼 42
頭殼壞去 8
頭油 4
頭燈 42
頭版 92
頭版新聞 8
頭狀 10
頭狀花序 7
頭獎 459
頭疼 46
頭痛 729
頭痛醫頭 9
頭癬 9
頭皮 166
頭皮屑 61
頭盔 153
頭目 127
頭破 0
頭破血流 32
頭等 50
頭等票 1
頭等艙 71
頭箍 1
頭篩 0
頭籌 52
頭紗 6
頭緒 78
頭繩 2
頭罩 17
頭胎 10
頭胸部 4
頭腦 376
頭腦好 4
頭臉 9
頭艙 0
頭花 18
頭茬 0
頭蓋 18
頭蓋腔 0
頭蓋骨 20
頭號 171
頭號人物 6
頭號敵人 6
頭虱 0
頭蝨 7
頭裡 30
頭角 15
頭角崢嶸 7
頭路 77
頭部 927
頭重腳輕 10
頭銜 146
頭陀 22
頭陣 6
頭面人物 0
頭頂 168
頭頂上 36
頭項 1
頭領 42
頭頭 54
頭頭兒 0
頭頭兒 0
頭頭是道 19
頭頸 118
頭額 15
頭顱 97
頭風 7
頭骨 44
頭髮 946
頮 0
頯 4
頰 130
頰上添毫 0
頰嗛 0
頰白鳥 0
頰面 0
頰骨 2
頰鬚 0
頱 0
頲 43
頳 0
頳 0
頴 0
頵 4
頶 0
頷 42
頷下 4
頷下之珠 1
頷下腺 1
頷聯 1
頷頤 1
頷頭 1
頷首 20
頷首微笑 1
頸 804
頸動脈 52
頸動脈 52
頸子 33
頸梗 0
頸椎 90
頸椎骨 3
頸部 395
頸長 0
頸靜脈 14
頸靜脈 14
頸項 20
頸飾 3
頹 44
頹勢 28
頹唐 8
頹喪 18
頹圮 5
頹墮委靡 0
頹壞 2
頹廢 113
頹廢派 0
頹放 1
頹敗 3
頹敝 0
頹然 26
頹然喪氣 0
頹靡 6
頹顏 0
頹齡 0
頺 0
頻 3752
頻仍 47
頻催 4
頻傳 315
頻密 5
頻寬 797
頻帶 49
頻數 16
頻率 1756
頻率響應 53
頻繁 451
頻譜 173
頻蹙 0
頻道 3149
頻頻 413
頼 0
頽 0
顀 0
顁 3
顂 0
顃 3
顄 3
顅 7
顆 18089
顆粒 521
顆粒狀 28
顆粒飼料 2
顇 0
顈 39
顉 3
顊 2
顋 0
題 11431
題上 22
題上來 0
題上去 0
題下 37
題下來 1
題下去 0
題了 12
題個 0
題名 437
題名錄 1
題在 16
題壁 1
題外 98
題好 0
題字 20
題庫 1681
題庫系統 40
題得 10
題意 22
題旨 13
題材 682
題款 3
題畫 14
題目 2867
題簽 0
題署 0
題著 1
題號 124
題解 32
題詞 19
題詩 10
題跋 6
題辭 3
額 4443
額外 387
額外的 256
額外地 255
額定 160
額定值 6
額定功率 6
額定量 0
額度 781
額手 0
額手稱慶 4
額數 31
額滿 161
額滿為止 94
額菲爾士峰 0
額角 17
額面 0
額頭 152
額頭 20
額飾 0
額骨 3
顎 298
顎下腺 0
顎足 1
顎骨 26
顎骨 26
顏 11100
顏回 13
顏容 5
顏料 963
顏料商 0
顏料盒 0
顏氏家訓 9
顏淵 9
顏真卿 17
顏色 3918
顏貌 0
顏面 276
顏面骨 6
顐 71
顐 71
顐 71
顑 24
顒 72
顓 9
顔 0
顕 0
顖 0
顖 0
顗 196
願 4509
願不願 8
願不願意 53
願天下有 5
願意 3659
願景 0
願望 703
願者上鉤 8
願聞其詳 8
願謹 0
顙 77
顚 0
顛 167
顛三倒四 7
顛仆 4
顛來倒去 0
顛倒 147
顛倒乾坤 0
顛倒是非 15
顛倒黑白 4
顛兒 0
顛兒 0
顛峰 95
顛復 0
顛撲不破 5
顛末 0
顛沛 6
顛沛流離 14
顛狂 13
顛簸 35
顛簸而行 0
顛複 0
顛覆 407
顛連 1
顛顛 12
顛顛倒倒 6
顛顛晃晃 0
顜 265
顝 78
顝 78
類 2331
類人猿 2
類似 3392
類似之處 18
類似於 46
類別 17210
類化 22
類同 31
類固醇 0
類型 3331
類屬 14
類推 67
類推法 0
類星體 12
類書 84
類次 1
類此 36
類毒素 13
類比 669
類神經 286
類經 50
類股 0
顟 2
顟 2
顠 2
顡 0
顢 2
顢頇 7
顢頇無能 2
顣 3
顤 258
顤 258
顥 138
顦 0
顦 0
顧 2747
顧不上 3
顧不上 3
顧不了 14
顧不到 3
顧不得 52
顧主 14
顧全 14
顧全大局 11
顧到 35
顧前不顧後 0
顧卹 0
顧及 410
顧名 8
顧名思義 193
顧命 0
顧員 0
顧問 6003
顧問公司 819
顧問團 60
顧問權 0
顧好 20
顧客 2065
顧客群 19
顧家 27
顧小失大 0
顧左右而言他 1
顧左右而言其他 1
顧影自憐 4
顧忌 99
顧念 11
顧恤 0
顧惜 36
顧愷之 5
顧慮 291
顧慮到 41
顧慮重重 0
顧戀 1
顧曲 0
顧此失彼 14
顧炎武 3
顧盼 18
顧盼自得 2
顧盼自若 0
顧盼自雄 0
顧笑 1
顧著 42
顧返 1
顧面子 0
顧顏面 1
顨 0
顨 0
顩 27
顩 27
顪 48
顪 48
顪 48
顫 294
顫 294
顫動 75
顫動 75
顫巍 0
顫巍 0
顫巍巍 2
顫巍巍 2
顫悠 0
顫悠 0
顫悠悠 0
顫悠悠 0
顫悸 0
顫悸 0
顫慄 284
顫慄 16
顫抖 212
顫抖 212
顫抖抖 0
顫抖抖 0
顫筆 0
顫筆 0
顫聲 14
顫聲 14
顫音 15
顫音 15
顬 0
顭 0
顭 0
顮 0
顯 3593
顯位 1
顯像 277
顯像劑 5
顯像液 0
顯像液 0
顯像管 11
顯出 194
顯名 3
顯型 6
顯士 0
顯妣 0
顯學 18
顯官 1
顯宦 0
顯影 78
顯影像 1
顯影劑 20
顯影力 0
顯影液 7
顯影液 7
顯影紙 0
顯得 997
顯微 296
顯微照片 2
顯微鏡 469
顯性 67
顯揚 14
顯明 149
顯植物 0
顯榮 124
顯然 1313
顯然在 15
顯然是 186
顯然會 7
顯然有 37
顯然能 1
顯現 664
顯現出 134
顯現出來 47
顯目 9
顯眼 39
顯示 13239
顯示出 502
顯示出來 90
顯示器 2667
顯祖 13
顯耀 6
顯考 0
顯者 5
顯而易見 69
顯聖 10
顯職 0
顯花植物 1
顯著 1146
顯著性 31
顯要 16
顯見 263
顯豁 1
顯貴 5
顯赫 53
顯赫家世 1
顯身手 17
顯身揚名 0
顯達 32
顯露 95
顯露 95
顯露出 48
顯露出 48
顯露出來 15
顯露出來 15
顯露頭角 0
顯露頭角 0
顯靈 28
顰 26
顰眉 0
顰蹙 0
顱 283
顱頂骨 0
顱骨 32
顲 3
顲 3
顳 136
顳骨 6
顴 54
顴骨 29
風 1
風 10636
風中 443
風中殘燭 2
風乾 40
風乾機 2
風俗 323
風俗習慣 49
風信 8
風信子 95
風信鴿 0
風傳 4
風儀 5
風光 778
風光十足 1
風光明媚 35
風兒 31
風兒 31
風力 156
風力機 5
風動 44
風勢 23
風化 106
風化作用 11
風化區 4
風化場所 3
風化場所 3
風口 56
風向 312
風向器 0
風向標 1
風向針 0
風吹 351
風吹兩面倒 1
風吹草動 37
風吹雨打 17
風味 700
風味十足 5
風和 63
風和日暖 2
風和日麗 42
風圈 16
風土 163
風土人情 37
風土化 0
風土誌 1
風城 143
風塵 223
風塵中人 0
風塵僕僕 25
風塵女郎 0
風壓 18
風姿 43
風姿綽約 9
風媒花 0
風季 1
風害 3
風寒 43
風尚 110
風帆 50
風帶 14
風帽 1
風平浪靜 43
風度 175
風度翩翩 16
風情 1415
風情畫 358
風情萬種 37
風扇 488
風捲殘雲 2
風掃落葉 1
風斗 0
風景 1754
風景區 1234
風景畫 34
風景線 70
風暴 2637
風月 472
風月場所 17
風月場所 173
風格 2302
風標 3
風櫛雨沐 0
風氣 680
風水 509
風水師 6
風沙 36
風波 584
風波 584
風波不斷 14
風波不斷 14
風洞 47
風洞試驗 8
風流 187
風流人物 11
風流倜儻 12
風流債 0
風流成性 1
風流才子 6
風流瀟灑 1
風流雲散 2
風流韻事 5
風浪 103
風浪板 56
風涼 3
風涼話 13
風溼 25
風溼性 35
風溼病 7
風潮 612
風濕 301
風濕性 199
風濕熱 22
風濕病 63
風濕症 27
風火 29
風火牆 0
風災 252
風燈 14
風燭殘年 2
風爐 0
風爐子 0
風物 29
風物誌 7
風玫瑰圖 0
風琴 72
風疹 3
風疹塊 2
風發 18
風砂 8
風神 39
風笛 9
風箏 511
風箱 7
風範 147
風紀 59
風級 2
風絮 3
風聞 23
風聞而來 1
風聞而至 0
風聲 118
風聲鶴唳 48
風致 1
風花圖 0
風花雪月 46
風華 140
風華富麗 0
風華絕代 13
風蝕 9
風行 238
風行一時 8
風行一時 8
風行草偃 11
風衣 49
風裡 21
風言醋語 1
風言霧語 1
風言風語 9
風評 0
風調雨順 32
風貌 521
風起雲湧 223
風趣 128
風車 170
風輪 7
風速 162
風速器 0
風速計 23
風道 7
風選 3
風邪 6
風采 284
風量 64
風鈴 71
風錘 0
風鏡 10
風鑽 0
風門 22
風險 2451
風雅 33
風雅頌 5
風雨 620
風雨不透 0
風雨交加 19
風雨同舟 0
風雨如晦 2
風雨淒淒 0
風雨無阻 43
風雨燈 0
風雨雷電 2
風雨飄搖 15
風雲 2238
風雲人物 131
風雲叱吒 4
風雲變色 16
風雲際會 39
風雷 28
風霜 42
風靡 141
風靡一時 6
風靡一時 6
風韻 20
風韻猶存 4
風頭 25
風頭甚健 0
風風 30
風風光光 18
風風雨雨 55
風餐露宿 0
風馬牛 2
風馬牛不相及 7
風馳電掣 11
風騷 245
風骨 40
風鬥 0
風魔 9
颩 21
颫 0
颬 19
颭 24
颮 38
颯 79
颯沓 0
颯然 1
颯颯 9
颰 0
颱 261
颱風 1636
颱風天 64
颱風季 9
颱風眼 18
颱風草 1
颱風雨 2
颲 36
颳 49
颳了 2
颳倒 0
颳去 0
颳得 2
颳著 3
颳走 1
颳起 22
颳起來 0
颳風 4
颵 0
颶 21
颶風 75
颷 0
颸 19
颹 0
颺 41
颻 233
颼 15
颼颼 15
颽 218
颽 218
颽 218
颾 104
颿 72
飀 9
飀 9
飀 9
飁 5
飂 4
飃 0
飄 752
飄下 23
飄下來 1
飄下去 1
飄來 37
飄來飄去 7
飄到 31
飄動 22
飄去 11
飄向 14
飄在 20
飄失 0
飄帶 6
飄幌 0
飄往 2
飄忽 24
飄忽不定 6
飄揚 77
飄搖 17
飄搖不定 1
飄擺 0
飄散 35
飄散在 7
飄晃 1
飄泊 66
飄洋 2
飄洋過海 24
飄洒自如 0
飄流 31
飄浮 122
飄渺 27
飄灑 1
飄灑在 1
飄灑自如 0
飄然 29
飄然出世 0
飄然而過 0
飄盪 30
飄移 19
飄絮 2
飄緲 2
飄舞 5
飄舞著 0
飄落 48
飄落在 2
飄著 43
飄蕩 26
飄走 3
飄起 28
飄起來 1
飄逝 5
飄逸 93
飄遊 2
飄遊四海 0
飄過 32
飄過來 5
飄過去 2
飄遙 0
飄雪 57
飄零 41
飄飄 92
飄飄欲仙 4
飄飄然 8
飄飄蕩蕩 2
飆 15806
飆高 105
飆漲 0
飆舞 0
飆車 154
飆車族 0
飇 0
飈 0
飉 278
飊 0
飋 15
飌 99
飍 0
飍 0
飛 16547
飛上 63
飛上來 0
飛上去 4
飛下 7
飛下來 2
飛下去 1
飛也似 12
飛了 90
飛低 2
飛來 125
飛來橫禍 4
飛來豔福 0
飛來飛去 29
飛個 4
飛入 62
飛出 85
飛出來 11
飛出去 7
飛刀 58
飛利浦 587
飛到 178
飛動 3
飛去 67
飛向 101
飛吻 49
飛回 50
飛回來 3
飛回去 1
飛在 46
飛地 0
飛塵 77
飛天 182
飛奔 58
飛奔而來 1
飛奔而去 4
飛安 0
飛將軍 1
飛彈 971
飛彈潛艇 0
飛彈頭 0
飛往 234
飛得 224
飛快 83
飛成 0
飛抵 81
飛揚 227
飛揚跋扈 1
飛散 37
飛旋 15
飛昇 9
飛書 7
飛機 2904
飛機場 20
飛機場 20
飛機棚 1
飛機翼 0
飛檄 0
飛櫻 10
飛毛腿 16
飛沙走石 4
飛沫 24
飛沫傳染 11
飛泉 13
飛渡 16
飛湍 1
飛漲 18
飛漲 18
飛濺 18
飛瀑 48
飛燕 48
飛球 20
飛白 2
飛的 223
飛盤 46
飛直 4
飛矢 0
飛石 6
飛碟 787
飛禍 0
飛禽 24
飛禽走獸 5
飛簷 15
飛簷走壁 8
飛紅 8
飛經 5
飛翔 295
飛至 20
飛舞 181
飛航 579
飛船 154
飛艇 11
飛花 207
飛蚊症 1
飛蛾 19
飛蛾撲火 15
飛蛾赴火 0
飛蝗 1
飛蟲 7
飛蟻 0
飛行 2823
飛行員 284
飛行器 47
飛行士 1
飛行官 4
飛行船 47
飛觥獻斝 10
飛觥走斝 10
飛觥限斝 10
飛語 0
飛賊 5
飛走 43
飛起 58
飛起來 64
飛越 226
飛跑 3
飛躍 126
飛躍成長 6
飛車 80
飛車黨 6
飛輪 55
飛逝 17
飛速 26
飛進 33
飛進來 5
飛進去 0
飛過 99
飛過來 7
飛過去 9
飛鏢 0
飛雅特 80
飛離 19
飛雪 46
飛靶 14
飛飛 67
飛馳 46
飛馳而來 1
飛馳而去 2
飛騎 3
飛騰 20
飛魚 115
飛鳥 191
飛鴕 0
飛鴿 27
飛鴿傳書 156
飛鷹 77
飛黃騰達 9
飛鼠 40
飜 0
飝 0
食 48
食 4853
食不充饑 0
食不知味 4
食事五觀 1
食人 55
食人族 13
食人魚 1
食俸 0
食具 17
食利者 0
食前方丈 0
食古不化 12
食品 10008
食品商 4
食品工業 688
食品店 35
食品廠 146
食品業 290
食品業者 55
食品舖 1
食品衛生 786
食器 20
食堂 260
食塊 0
食子蜂 0
食客 37
食宿 289
食宿費 4
食心蟲 0
食性 174
食慾 157
食慾不佳 10
食慾不振 118
食指 170
食指浩繁 1
食料 8
食時五觀 1
食材 1
食槽 2
食欲 16
食欲不振 6
食油 105
食無求飽 0
食物 2935
食物中毒 160
食物店 2
食物鏈 58
食用 860
食用動物 1
食用植物 12
食用油 37
食用糖 0
食用魚 18
食用鹽 3
食祿 5
食管 4
食管癌 0
食米 32
食糖 2
食糜 1
食糧 48
食肉獸 0
食肉類 7
食色性也 11
食草獸 0
食菌 0
食蚜蠅 0
食蟻獸 13
食衣住行 92
食衣住行育樂 30
食言 26
食言而肥 9
食記 1
食譜 922
食貨 7
食貨志 3
食道 299
食道炎 24
食道癌 42
食道癌 42
食邑 0
食量 68
食髓知味 11
食鹽 126
飡 0
飢 734
飢不擇食 4
飢困 0
飢寒 5
飢寒交迫 4
飢民 1
飢渴 37
飢溺 0
飢腸 1
飢腸轆轆 8
飢色 2
飢荒 31
飢餓 132
飢餓感 6
飢饉 6
飣 5
飥 3
飥 3
飦 0
飧 6104
飩 100
飪 217
飫 24
飬 0
飭 245
飭令 10
飭回 0
飭辦 0
飭遵 2
飯 1930
飯丸 0
飯來張口 2
飯來開口 0
飯前 59
飯前洗手 5
飯匙 9
飯匙 9
飯囊 0
飯團 18
飯局 8
飯店 9353
飯廳 18
飯後 162
飯後漱口 3
飯桌 22
飯桶 4
飯瓢 1
飯盒 44
飯碗 52
飯票 4
飯筐 0
飯筒 0
飯米粒兒 0
飯米粒兒 0
飯粒 22
飯糰 46
飯莊 1
飯菜 27
飯袋 1
飯量 9
飯鍋 7
飯食 13
飯館 17
飰 0
飱 0
飲 25
飲 2546
飲下 14
飲冰茹蘗 0
飲品 0
飲宴 31
飲店 7
飲彈 3
飲彈自盡 3
飲恨 34
飲恨而終 1
飲料 1896
飲料店 4
飲料水 3
飲水 497
飲水思源 76
飲水機 217
飲泣 13
飲淚 0
飲湯 1
飲用 438
飲用水 643
飲茶 224
飲血 4
飲酒 260
飲酒作樂 12
飲酒室 0
飲酖止渴 1
飲醇自醉 0
飲食 2561
飲食店 113
飲食業 19
飲食業者 0
飲食無味 0
飲食無度 1
飲食男女 184
飲食療法 33
飲食起居 8
飲馬河 0
飲鴆止渴 11
飳 0
飳 0
飴 60
飴漿 0
飴糖 0
飵 0
飵 0
飶 56
飷 0
飸 0
飸 0
飹 26
飺 47
飻 0
飻 0
飼 363
飼主 99
飼料 1701
飼料作物 8
飼料雞 2
飼養 964
飼養場 4
飼養場 4
飼養所 0
飼養標準 1
飼餵 17
飽 2961
飽以 0
飽以老拳 0
飽受 189
飽受虛驚 0
飽合 29
飽含 11
飽和 401
飽和度 58
飽和水汽壓 0
飽和溶液 2
飽和溶液 2
飽和狀態 19
飽和電流 0
飽和點 0
飽嗝 2
飽嘗 20
飽嚐 12
飽學 4
飽學之士 4
飽暖 5
飽暖思淫慾 2
飽滿 107
飽經 7
飽經世故 1
飽經滄桑 1
飽經風霜 3
飽脹 4
飽腹 1535
飽足 36
飽食 11
飽食終日 2
飽飽 1815
飽餐 14
飽餐一頓 13
飾 12233
飾品 0
飾有 19
飾演 242
飾物 120
飾終 1
飾詞 0
飾辭 0
飾過 0
飿 0
餀 59
餁 0
餂 53
餃 133
餃子 74
餃子皮 5
餃子餡 1
餄 0
餅 1187
餅乾 304
餅乾店 0
餅乾盒 4
餅子 0
餅屑 1
餅餡 0
餆 0
餇 85
餈 51
餉 108
餉客 0
餉銀 1
餉餽 0
養 2278
養 4424
養下來 0
養不起 7
養了 157
養傷 10
養兒防老 7
養兵 2
養兵之道 0
養兵千日 2
養分 142
養地 5
養大 49
養女 13
養媳 0
養子 39
養家 19
養家活口 13
養家費 0
養尊處優 8
養廉 4
養得 173
養志 6
養性 20
養成 1114
養成教育 52
養料 7
養晦 1
養樂多 115
養殖 1839
養殖場 71
養殖場 71
養母 9
養氣 22
養活 67
養父 23
養珠 7
養生 711
養生之道 64
養生送死 1
養病 13
養癰貽患 0
養癰遺患 0
養的 172
養眼 30
養神 22
養精蓄銳 7
養老 609
養老金 18
養老院 29
養而不教 0
養肥 4
養育 114
養育之恩 8
養育院 0
養胖 2
養虎 1
養虎反被虎咬 0
養虎為患 3
養虎遺患 1
養虎飴患 0
養蜂 27
養蜂場 11
養蜂場 11
養蠶 15
養蠶戶 0
養親 2
養護 501
養豬 1710
養豬場 13
養豬場 139
養豬戶 215
養貓 69
養起 3
養起來 2
養路 6
養身 44
養身之道 35
養軍千日 0
養雞 46
養雞場 39
養雞場 39
養顏 78
養養 12
養魚 147
養魚場 1
養魚場 1
養魚池 4
養鳥 6
養鴨 35
養鴨人家 20
餋 0
餌 328
餌敵 0
餐 3369
餐具 239
餐券 41
餐室 2
餐巾 33
餐廳 5072
餐敘 0
餐會 235
餐桌 165
餐桌上 43
餐盒 60
餐費 42
餐車 26
餐風宿水 0
餐風宿露 2
餐風露宿 0
餐飲 1805
餐飲業 389
餐飲部 22
餐館 229
餐點 238
餑 59
餑 59
餑餑 0
餒 25
餓 1333
餓不死 4
餓倒 1
餓壞 7
餓慣 0
餓死 69
餓死了 14
餓死事小 1
餓死事小失節事大 0
餓死在 1
餓死鬼 4
餓殍 0
餓漢 1
餓狼 35
餓狼撲羊 0
餓瘦 0
餓肚子 16
餓肚皮 0
餓莩 1
餓著 1
餓著肚子 5
餓著肚皮 1
餓虎 4
餓虎撲羊 4
餓虎撲食 0
餓貓 0
餓過 2
餓鬼 29
餓鬼道 1
餔 41
餕 10
餖 24
餗 24
餘 5625
餘下 33
餘下來 0
餘事 2
餘人 0
餘值 2
餘光 21
餘函數 0
餘切 0
餘利 1
餘剩 1
餘割 0
餘力 57
餘味 9
餘味無窮 2
餘地 104
餘址 1
餘姚 6
餘威 8
餘孽 12
餘巾 0
餘年 536
餘式定理 6
餘弦 18
餘弦定理 2
餘悸 11
餘悸猶存 8
餘數 45
餘日 19
餘暇 15
餘暉 35
餘款 53
餘步 2
餘毒 7
餘氯 0
餘波 14
餘波 14
餘波盪漾 20
餘波盪漾 20
餘波蕩漾 19
餘波蕩漾 19
餘溫 32
餘澤 0
餘燼 13
餘生 125
餘留 10
餘糧 11
餘糧收集 0
餘糧收集制 0
餘缺 0
餘興 7
餘興節目 6
餘蓄 0
餘蔭 5
餘裕 31
餘角 2
餘輝 17
餘錢 7
餘閒 5
餘震 22
餘震不斷 0
餘音 17
餘音嫋嫋 0
餘音嬝繞 0
餘音猶在 0
餘音繞梁 0
餘音繞樑 2
餘韻 27
餘額 1358
餘風 0
餘香 13
餘黨 5
餙 0
餚 71
餛 25
餛飩 15
餛飩麵 2
餛飩麵 2
餛餛 0
餜 0
餝 0
餞 26
餞別 9
餞行 6
餟 12
餠 0
餡 143
餡兒 6
餡兒 6
餡餅 20
餢 0
餣 0
餤 0
餤 0
餥 0
餦 0
餧 0
館 16259
館主 152
館使 19
館內 635
館前 342
館名 26
館員 1262
館外 86
館子 24
館子 162
館舍 297
館藏 4040
館邸 0
館長 752
館閣 1
餩 0
餪 0
餫 0
餬 5
餬口 1030
餭 0
餮 0
餯 0
餰 0
餱 0
餲 0
餳 0
餴 0
餵 19034
餵你 ㄨㄟˋ
餵我 ㄨㄟˋ
餵它 ㄨㄟˋ
餵乳 20
餵了 4
餵奶 49
餵奶時 3
餵給 10
餵羊 1
餵著 2
餵豬 6
餵過 1
餵雞 3
餵食 256
餵飽 12
餵養 116
餵魚 14
餵鴨 0
餵鵝 0
餶 0
餷 0
餸 0
餹 0
餺 39
餻 0
餼 38
餽 30
餽贈 34
餾 48
餿 5
餿主意 1
餿味 0
餿水 16
餿水油 1
餿臭 0
饀 0
饀 0
饁 25
饃 1
饃 160
饃頭 1
饃饃 1
饄 0
饅 32
饅頭 1
饅頭 187
饆 0
饇 141
饇 141
饈 73
饉 5
饊 0
饋 179
饋贈 40
饋遺 0
饌 87
饌具 0
饍 0
饎 2
饎 2
饏 0
饐 3
饑 456
饑不擇食 1
饑寒 3
饑寒交迫 2
饑民 3
饑渴 29
饑溺 1
饑荒 30
饑飽 0
饑餓 58
饑饉 17
饒 1796
饒命 6
饒富 36
饒州 0
饒平 27
饒恕 49
饒有 13
饒沃 0
饒河 50
饒舌 31
饒舌歌 4
饓 3
饔 16
饔人 0
饔飧 1
饕 170
饕餮 9
饖 10
饖 10
饗 118
饗宴 391
饘 12
饘粥 0
饙 2
饚 0
饚 0
饛 16
饜 91
饜足 4
饜食 0
饝 0
饞 25
饞嘴 3
饞嘴巴 0
饞涎欲滴 0
饟 6
饠 0
饡 1
首 2727
首位 191
首例 0
首倡 7
首倡者 1
首先 3369
首先在 84
首先是 233
首冊 10
首列 0
首創 560
首創人 0
首創者 3
首功 3
首台 33
首名 8
首唱 16
首善 10
首善之區 16
首夜 3
首宗 75
首富 38
首封 4
首尾 14
首屆 225
首屈一指 77
首席 308
首席代表 8
首席檢查官 0
首府 106
首度 988
首度來台 2
首座 151
首惡 4
首應 24
首戰 57
首戰告捷 1
首播 122
首播 122
首播儀式 0
首播儀式 0
首播典禮 0
首播典禮 0
首日 127
首日封 34
首映 172
首映典禮 0
首架 4
首次 1757
首次來台 10
首段 5
首犯 0
首獎 163
首當 11
首當其衝 64
首相 355
首站 52
首篇 0
首級 43
首義 0
首肯 26
首腦 88
首腦人物 3
首腦份子 0
首腦份子 0
首航 60
首航日 0
首艙 0
首要 403
首要條件 26
首見 92
首調 4
首謀 4
首謀者 3
首輪 47
首輪電影 8
首途 1
首逢 1
首邑 6
首部 61
首部曲 1
首都 758
首長 1862
首開 104
首開紀錄 4
首隊 1
首集 3
首需 7
首面 13
首頁 113953
首須 12
首領 113
首飾 424
首飾盒 8
首魁 0
首鼠兩端 0
馗 16
馘 5
香 14229
香串 11
香丸 4
香亭 19
香包 18
香味 395
香味噴鼻 0
香味撲鼻 3
香噴噴 35
香囊 3
香塵 1
香客 25
香山 275
香巢 0
香巴拉 1
香料 276
香料群島 0
香格里拉 184
香案 6
香椿 9
香樁 1
香樟 3
香檳 210
香檳酒 8
香櫞 1
香殿 1
香氣 159
香水 705
香水瓶 34
香江 65
香油 75
香油錢 4
香消玉殞 3
香液 3
香液 3
香港 12425
香港人 153
香港浸會大學 1
香港浸會學院 1
香港腳 83
香港郵報 0
香港隊 2
香湯 0
香澤 1
香濃 1
香火 115
香火不斷 0
香火不絕 0
香火因緣 1
香火很盛 1
香灰 3
香煙 171
香煙盒 3
香煙頭 0
香燭 17
香爐 115
香爐灰 0
香片 16
香瓜 32
香瓜汁 0
香甜 63
香甜可口 8
香皂 55
香皂盒 2
香盒 3
香禮 2
香積廚 2
香粉 9
香粉盒 0
香精 113
香精油 36
香紙 6
香肉 19
香腸 207
香臭 4
香艷 8
香花 43
香花毒草 1
香芹 1
香茅 26
香茶 15
香草 157
香草冰 0
香草冰淇淋 12
香草精 1
香菇 324
香菜 38
香菸 110
香蕈 1
香蕉 478
香蕉冰 0
香蕉汁 478
香蕉油 1
香蕉皮 4
香蠟 0
香蠟店 0
香袋 6
香豔 16
香車 10
香車美人 4
香軍 0
香郁 3
香酒 3
香酥 40
香酥可口 2
香醇 130
香醇美味 0
香醪 17
香錢 1
香閣 2
香閨 9
香雞 0
香雞城 1
香雞排 0
香餅 32
香餌 1
香香 60
香香甜甜 2
香香酥酥 0
香魚 1
馚 0
馛 0
馜 69
馝 33
馝馞 1
馞 7
馟 0
馠 0
馡 3
馢 0
馣 7
馤 0
馥 266
馥郁 6
馦 232
馦 232
馧 270
馨 1587
馨 1587
馨香 37
馩 0
馪 0
馫 83
馬 12160
馬丁 142
馬丁尼 18
馬丁路德 22
馬上 2491
馬上來 17
馬上去 25
馬不停蹄 51
馬仰人翻 1
馬來 200
馬來亞 6
馬來人 22
馬來半島 12
馬來群島 2
馬來西亞 2062
馬來西大學 0
馬來語 8
馬偕 325
馬偕紀念醫院 333
馬偕醫院 237
馬克 968
馬克吐溫 18
馬克思 108
馬克思主義 99
馬克思主義者 5
馬克思列寧主義 1
馬克數 0
馬克斯 84
馬克斯主義 23
馬克杯 0
馬兒 48
馬兒 48
馬公 474
馬公市 1098
馬兵 5
馬其頓 27
馬具 8
馬刀 13
馬列主義 21
馬利 212
馬利蘭州 7
馬到成功 29
馬前 10
馬前卒 9
馬力 807
馬勒 2491
馬勺 0
馬匹 51
馬口鐵 42
馬可 161
馬可仕 35
馬可斯 2
馬可波羅 42
馬可波羅 42
馬可福音 49
馬善被人騎 0
馬嘶 10
馬國 352
馬基維利 3
馬場 73
馬場 73
馬士英 0
馬太 70
馬太福音 107
馬夫 6
馬子 110
馬家 34
馬尼拉 298
馬尾 48
馬尾松 8
馬尾藻 5
馬屁 29
馬屁大王 0
馬屁精 27
馬年 4
馬廄 14
馬建忠 0
馬後 9
馬後炮 7
馬後砲 6
馬德望 0
馬德里 203
馬快 3
馬恩 5
馬戲 22
馬戲團 101
馬戲團表演 0
馬戲場 0
馬戲場 0
馬戲表演 3
馬房 6
馬拉 55
馬拉加 4
馬拉加西 4
馬拉加西國 0
馬拉博 0
馬拉威 64
馬拉松 159
馬拉松賽跑 2
馬拉硫磷 0
馬拉開波 0
馬拉開波 0
馬拉開波湖 0
馬拉開波湖 0
馬掌 0
馬文 34
馬斯 171
馬斯喀特 0
馬明菩薩 1
馬普托 0
馬札兒人 1
馬東籬 66
馬桶 1701
馬桶蓋 12
馬棚 3
馬森 1
馬槽 77
馬步 19
馬殺雞 3
馬氏體 0
馬漢 20
馬燈 244
馬爾 243
馬爾他 44
馬爾代夫 0
馬爾薩斯 6
馬球 28
馬生角 0
馬生角 0
馬甲 2
馬皮 2
馬祖 655
馬祖地區 50
馬禮遜 0
馬禮遜美國中學 0
馬糞 3
馬糞紙 0
馬累 1
馬群 13
馬耳 5
馬耳他 1
馬耳東風 2
馬肉 3
馬背 35
馬背上 11
馬腳 5
馬自達 189
馬致遠 66
馬良 2
馬英九 6608
馬蒂斯 1
馬蘭 71
馬虎 57
馬虎 57
馬虎子 0
馬號 0
馬蜂 1
馬蝨 0
馬蟥 0
馬蠅 0
馬術 116
馬表 2
馬裡 0
馬褂 0
馬褲 1
馬西埃 0
馬賊 4
馬賽 77
馬賽克 187
馬赫 15
馬路 291
馬路上 79
馬路如虎口 2
馬蹄 72
馬蹄形 7
馬蹄聲 10
馬蹄鐵 8
馬車 286
馬車伕 0
馬車賽 0
馬達 1345
馬達加斯加 32
馬達加斯加島 8
馬醫 5
馬里 53
馬里克 0
馬里蘭 77
馬里蘭州 29
馬鈴瓜 0
馬鈴薯 158
馬鈴薯泥 1
馬錶 14
馬關 3
馬關條約 51
馬隊 16
馬雅文化 4
馬革裹屍 4
馬靴 18
馬鞍 67
馬鞍形 0
馬鞭 34
馬頭 45
馬頭明王 1
馬頭金剛 1
馬頭金剛明王 1
馬頭魚 8
馬類 2
馬首 4
馬首是瞻 16
馬馬虎虎 20
馬馬虎虎 20
馬鬃 8
馬鳴 1
馬鳴菩薩 1
馬齒 24
馬齒徒增 1
馬齒徒長 0
馬齒莧 10
馬龍 49
馭 115
馭夫 0
馭夫有術 0
馭者 0
馮 12
馮 1231
馮國璋 2
馮夢龍 23
馮婦 4
馮特 1
馮玉祥 1
馮諾以曼 0
馮遼 0
馮雪峰 1
馯 31
馰 18
馱 98
馱 9823
馱背 0
馱負 3
馱馬 1
馲 15
馳 889
馳名 67
馳名中外 10
馳名天下 0
馳往 2
馳念 0
馳馬 6
馳騁 79
馳驅 0
馴 1704
馴 1704
馴化 31
馴和 0
馴服 36
馴獅 0
馴獸師 10
馴良 2
馴虎 0
馴虎師 0
馴順 0
馴養 16
馴鹿 9
馵 17
馷 0
馸 0
馸 0
馹 74
馺 197
馻 12
馼 0
馽 4
馾 0
馾 0
馿 0
駀 0
駁 305
駁倒 8
駁回 249
駁回來 0
駁回去 0
駁斥 224
駁正 1
駁船 18
駁運 1
駁雜 2
駂 69
駃 73
駄 0
駄 0
駆 0
駇 4
駈 0
駉 27
駊 0
駋 18
駌 11
駍 24
駎 34
駏 17
駐 6000
駐點 100
駐兵 3
駐在員 4
駐在國 16
駐地 23
駐外 561
駐守 58
駐屯 3
駐屯軍 1
駐屯重兵 0
駐戌 0
駐留 29
駐紮 24
駐紮地 0
駐華 266
駐華大使 44
駐警 45
駐足 83
駐軍 85
駐防 20
駐防地 0
駐防軍 0
駐防部隊 0
駐顏 13
駐顏有術 1
駑 17
駑下 1
駑劣 0
駑材 0
駑鈍 9
駑馬 1
駑馬十駕 0
駑駘 0
駒 283
駒鳥 0
駒齒 0
駓 17
駔 1
駔 1
駔 27
駕 581
駕上 1
駕上來 0
駕上去 0
駕了 3
駕來駕去 0
駕到 39
駕去 1
駕回 3
駕回來 0
駕回去 0
駕在 3
駕好 0
駕崩 11
駕御 14
駕成 0
駕機 4
駕機投誠 0
駕照 479
駕臨 38
駕船 12
駕著 53
駕訓班 0
駕走 0
駕起 2
駕車 232
駕車人 1
駕車記 0
駕輕就熟 32
駕過 2
駕馭 106
駕駛 1187
駕駛人 410
駕駛員 187
駕駛執照 93
駕駛室 17
駕駛座 1
駕駛盤 1
駕駛者 42
駕駛艙 54
駕鶴西歸 2
駖 29
駗 20
駘 36
駘蕩 1
駙 11
駙馬 19
駙馬爺 5
駚 0
駛 525
駛出 20
駛去 21
駛向 23
駛回 5
駛往 20
駛抵 9
駛近 8
駜 82
駝 1289
駝子 13
駝峰 8
駝絨 0
駝背 73
駝著背 1
駝負 0
駝馬 9
駝鳥 82
駞 0
駟 29
駟馬難追 0
駠 0
駡 0
駢 30
駢儷 1
駢列 0
駢四驪六 0
駢拇枝指 0
駢文 9
駢枝 2
駢比 0
駢體 1
駢體文 0
駢麗 0
駣 5
駤 5
駥 7
駦 0
駧 18
駨 0
駨 0
駩 4
駪 7
駫 0
駫 0
駬 5
駭 11
駭 1185
駭人 62
駭人聽聞 37
駭客 0
駭怕 2
駭愕 1
駭浪 6
駭然 59
駭異 5
駭目驚心 0
駭聞 1
駮 12
駯 0
駰 3
駱 429
駱尚廉 16
駱馬 9
駱駝 169
駱駝峰 0
駱駝毛 0
駱駝絨 0
駱駝隊 2
駱驛 0
駳 0
駴 4
駵 0
駶 3
駷 9
駸 6
駹 8
駺 5
駻 11
駼 9
駽 0
駾 5
駿 1271
駿奔 0
駿發 16
駿逸 34
駿邁 0
駿馬 37
駿骨 1
騀 0
騀 0
騁 51
騁志 1
騁懷 2
騁望 2
騁步 0
騁目 0
騁馳 7
騂 8
騃 23
騄 74
騅 61
騆 93
騇 86
騈 0
騉 112
騊 170
騋 80
騌 0
騍 75
騍馬 0
騎 879
騎 66100
騎上 31
騎上來 0
騎上去 1
騎下 1
騎下來 0
騎下去 0
騎乘 226
騎了 23
騎來 5
騎來騎去 0
騎兵 1
騎兵 191
騎兵團 7
騎兵團 7
騎兵隊 11
騎兵隊 11
騎出 1
騎出來 0
騎出去 0
騎到 28
騎去 5
騎回 4
騎回來 3
騎回去 0
騎在 33
騎坐 3
騎士 794
騎好 1
騎射 3
騎將 3
騎師 6
騎得 7
騎從 5
騎成 0
騎樓 99
騎牆 4
騎牆份子 0
騎牆派 2
騎破 0
騎縫 6
騎縫印 0
騎縫章 17
騎著 191
騎虎之勢 0
騎虎難下 6
騎術 4
騎警 6
騎走 3
騎車 155
騎車來 1
騎車到 7
騎車去 4
騎進 2
騎進來 0
騎進去 0
騎過 14
騎過來 0
騎過去 3
騎馬 202
騎馬打仗 9
騎馬找馬 0
騎馬裝 0
騎騎 2
騎驢 15
騎驢找馬 1
騎驢看唱本 0
騏 203
騏驎 4
騏驥 3
騐 0
騑 86
騔 92
騔 92
騔 92
騕 305
騖 19
騗 0
騘 0
騙 1985
騙上 4
騙上來 1
騙上去 0
騙下 1
騙下來 0
騙下去 2
騙了 88
騙人 184
騙來 7
騙來騙去 2
騙倒 4
騙光 2
騙出 1
騙出來 0
騙出去 0
騙到 22
騙去 11
騙取 36
騙吃 0
騙吃騙喝 8
騙嘴 0
騙回 0
騙回來 0
騙回去 0
騙婚 5
騙子 98
騙局 112
騙得 42
騙的 41
騙腿 0
騙著 1
騙術 32
騙術奇譚 1
騙財 5
騙財騙色 3
騙走 8
騙錢 58
騚 102
騛 157
騜 103
騝 260
騝 260
騞 196
騟 0
騠 142
騡 0
騢 132
騢 132
騣 580
騤 155
騥 132
騧 203
騩 4
騩 4
騪 7
騫 558
騫旗 0
騬 2
騭 11
騮 20
騯 0
騯 0
騰 2234
騰出 38
騰出來 0
騰昇 1
騰格里山 0
騰空 66
騰空而起 4
騰空而過 0
騰衝 3
騰讓 1
騰貴 0
騰越 8
騰達 13
騰開 0
騰雲 14
騰雲駕霧 9
騰騰 15
騱 2
騲 3
騳 0
騴 8
騵 186
騶 5
騷 1519
騷亂 26
騷人 9
騷人墨客 9
騷動 134
騷包 64
騷客 17
騷擾 571
騸 7
騹 6
騹 6
騺 6
騺鷹 0
騻 0
騼 0
騽 5
騾 20
騾夫 1
騾子 16
騾車 12
騾馬 4
騾馬車 0
騾馱子 0
騿 631
驀 18
驀地 48
驀然 33
驀然回首 27
驁 14
驁 14
驂 7
驂乘 0
驃 13
驃悍 6
驃騎 5
驄 14
驅 623
驅使 86
驅出 5
驅動 3076
驅動器 115
驅役 0
驅惡向善 0
驅散 49
驅策 24
驅蟲 59
驅蟲劑 5
驅趕 20
驅車 215
驅車前往 35
驅迫 1
驅逐 137
驅逐令 0
驅逐出境 21
驅逐機 0
驅逐艦 74
驅遣 0
驅邪 14
驅除 148
驅離 0
驅馬 2
驅馬前進 1
驅馳 2
驅鬼 8
驆 3
驇 0
驈 4
驉 1149
驊 285
驋 0
驌 23
驍 215
驍勇 5
驍勇善戰 12
驍將 4
驍雄 0
驍騎 0
驎 14
驏 2
驏 2
驐 329
驑 0
驒 2588
驓 5
驓 5
驔 12
驕 17000
驕佚 0
驕傲 474
驕傲感 5
驕傲自大 3
驕傲自滿 2
驕傲起來 1
驕兒 0
驕兵必敗 0
驕奢淫佚 2
驕奢淫逸 2
驕女 2
驕子 33
驕恣 0
驕悍 0
驕愛 0
驕態 0
驕慢 4
驕敵 0
驕橫 3
驕氣 2
驕狎 0
驕盈 0
驕矜 4
驕縱 13
驕者必敗 0
驕肆 0
驕色 0
驕陽 22
驖 3
驗 4001
驗貨 100
驗傷 28
驗傷單 10
驗光 103
驗光師 26
驗勘 0
驗尿 26
驗屍 77
驗屍官 0
驗收 496
驗收人 2
驗方 26
驗明 7
驗明正身 13
驗看 0
驗算 8
驗血 98
驗血單 0
驗訖 3
驗証 107
驗證 1134
驗證法 0
驗電器 3
驘 0
驙 4
驚 1653
驚世駭俗 17
驚人 689
驚人之舉 11
驚倒 2
驚動 65
驚動大駕 0
驚受 0
驚叫 37
驚叫聲 6
驚呼 38
驚呼其名 0
驚喜 422
驚喜交集 3
驚嘆 90
驚嘆不已 5
驚嘆號 33
驚嚇 113
驚嚇過度 7
驚天動地 101
驚天地 11
驚奇 218
驚奇不已 5
驚弓 0
驚弓之鳥 11
驚心 23
驚心動魄 57
驚怕 3
驚怪 9
驚恐 40
驚恐失色 0
驚恐萬分 0
驚恐萬狀 0
驚悚 2
驚悸 9
驚惶 39
驚惶失措 5
驚惶失色 1
驚愕 44
驚慌 148
驚慌失措 16
驚懼 28
驚戒 1
驚擾 14
驚擾到 0
驚服 2
驚歎 13
驚歎不已 0
驚歎號 11
驚濤 12
驚濤駭浪 30
驚濤駭浪 30
驚為天人 16
驚異 90
驚異傳奇 4
驚疑 12
驚艷 0
驚蟄 9
驚蟄 9
驚蟄節 1
驚見 37
驚覺 77
驚覺到 13
驚訝 567
驚詫 5
驚讚 2
驚豔 0
驚起 15
驚跳 2
驚退 1
驚醒 100
驚險 115
驚險刺激 29
驚風 7
驚風駭浪 1
驚駭 23
驚駭不已 0
驚魂 136
驚鴻 8
驚鴻一瞥 20
驛 159
驛丞 0
驛使 0
驛吏 0
驛書 0
驛站 358
驛車 1
驛運處 0
驛道 2
驛館 3
驛馬 14
驛馬車 4
驜 0
驝 0
驞 139
驟 344
驟然 92
驟至 1
驟變 13
驟起 7
驟雨 39
驠 9
驡 0
驡 0
驢 142
驢兒 2
驢兒 2
驢叫聲 0
驢子 48
驢打滾 8
驢皮影 1
驢頭 1
驣 0
驤 125
驥 152
驦 73
驧 288
驨 53
驩 50
驪 21
驪姬 1
驪歌 46
驪歌四起 0
驪珠 3
驪駒 0
驫 144
骨 193
骨 193
骨 1930
骨刺 64
骨刺 64
骨力 2
骨化 13
骨器 4
骨器 4
骨塔 80
骨子 13
骨子裡 45
骨子裡頭 3
骨學 0
骨屑 1
骨屑 1
骨幹 269
骨幹分子 0
骨幹分子 0
骨感 5
骨折 352
骨折 352
骨料 1
骨架 121
骨氣 36
骨灰 78
骨灰甕 12
骨灰罈 3
骨炭 0
骨牌 41
骨牌 41
骨牌效應 42
骨瘦如柴 4
骨癌 22
骨盆 190
骨盤 43
骨直 3
骨直 3
骨相 2
骨眼 3
骨碌 3
骨碌碌 6
骨科 853
骨科 853
骨科醫生 1
骨立 1
骨節 12
骨粉 23
骨結核 1
骨縫 1
骨肉 69
骨肉之情 0
骨肉分離 2
骨肉團圓 0
骨肉團聚 0
骨肉相殘 2
骨肉相連 1
骨肉至親 1
骨腔 0
骨腔 0
骨膜 13
骨膜炎 3
骨董 73
骨董 73
骨質 720
骨質疏鬆 1
骨質疏鬆症 1
骨頭 279
骨頭 279
骨頭架子 0
骨骸 22
骨骼 575
骨骼肌 36
骨髓 433
骨髓 433
骨髓炎 15
骨鯁 1
骨鯁在喉 0
骩 0
骪 0
骫 5
骬 0
骭 26
骮 0
骯 33
骯骯髒髒 0
骯髒 55
骯髒鬼 0
骰 25
骰子 64
骱 5
骲 0
骳 98
骴 10
骵 0
骶 0
骷 13
骷髏 54
骷髏頭 0
骸 32
骸骨 31
骹 13
骺 0
骻 14
骻骨 0
骼 38
骼炭 0
骽 0
骾 6
骿 4
髀 83
髀肉復生 0
髀骨 0
髁 18
髂 153
髃 0
髄 0
髆 10
髇 10
髇 10
髇 10
髈 0
髈 0
髈 0
髈 0
髉 0
髉 0
髊 23
髊 23
髊 23
髍 1859
髎 0
髏 7
髐 116
髑 12
髑髏 4
髒 380
髒亂 113
髒兮兮 8
髒字 7
髒得 40
髒得很 0
髒東西 9
髒水 7
髒的 40
髒話 139
髒錢 0
髓 412
體 13657
體位 200
體位法 7
體例 67
體內 1274
體制 1651
體力 596
體力勞動 22
體協 11
體味 20
體型 350
體壁 4
體壇 132
體壇人士 0
體外 309
體大思精 0
體察 51
體察出 1
體察到 4
體寬 4
體專 26
體己 0
體已 17
體式 17
體弱 14
體弱多病 13
體形 43
體念 11
體恤 50
體悉 0
體態 60
體態輕盈 5
體操 559
體操選手 11
體會 1116
體會出 54
體會到 308
體查 2
體校 11
體格 148
體格檢查 141
體檢 760
體法 7
體液 89
體液 89
體溫 337
體溫計 28
體無完膚 16
體熱 16
體現 137
體現出 10
體用 22
體積 813
體突變 0
體範 0
體系 3556
體系化 5
體細胞 27
體統 12
體罰 65
體育 14517
體育場 32
體育場 320
體育學院 309
體育室 0
體育專科學校 27
體育會 126
體育活動 427
體育界 11
體育課 950
體育運動 315
體育選手 5
體育館 518
體能 504
體能測驗 18
體能訓練 51
體脂肪 0
體脹係數 0
體腔 9
體膚 11
體膚之親 0
體臭 7
體臭味 0
體表 94
體裁 45
體要 9
體認 361
體認出 4
體認到 151
體諒 192
體貌 2
體貼 456
體貼入微 8
體質 963
體適能 0
體重 2184
體重表 2
體重計 8
體長 87
體面 49
體面起來 0
體驗 1530
體驗出 5
體驗到 77
體高 19
體魄 70
髕 86
髖 124
髖骨 0
髗 0
高 38068
高一 458
高一上 5
高一下 3
高三 362
高三上 3
高三下 5
高下 82
高不可攀 30
高不成 3
高不成低不就 1
高中 92
高中 9220
高中生 310
高了 102
高二 387
高二上 1
高二下 0
高亢 40
高亮 3
高人 202
高人一等 53
高人一等 53
高人一籌 0
高人一籌 0
高估 62
高位 48
高低 608
高低不平 7
高低槓 8
高個 7
高個兒 5
高個兒 5
高個子 4
高傲 41
高傲不屈 0
高傳真 65
高僧 103
高價 716
高冠 15
高冷 11
高冷蔬菜 13
高出 285
高分 125
高分子 1016
高分子化合物 5
高利 19
高利貸 12
高利貸者 0
高利貸資本 0
高加索 13
高加索區 0
高升 41
高原 172
高原地區 3
高台 54
高名 4
高周波 4
高周波 4
高呼 65
高唱 53
高唱入雲 4
高商 649
高喊 109
高單位 19
高四 21
高四班 0
高地 139
高地 1178
高坐 3
高堂 26
高塔 33
高壇 12
高壓 997
高壓手段 2
高壓政策 7
高壓線 79
高壓電 48
高士 30
高壽 7
高大 236
高夫 4
高妙 7
高宗 45
高官 93
高官厚爵 0
高官厚祿 3
高家 25
高容量 126
高密 17
高寒 3
高寮 10
高射 2
高射炮 0
高射砲 4
高射砲彈 0
高射部隊 0
高小 31
高尚 168
高就 23
高居 193
高屏地區 124
高屏溪 182
高層 1394
高層人士 100
高層次 70
高層雲 3
高山 883
高山族 30
高山景行 0
高山流水 2
高山病 5
高山症 11
高岸 0
高峙 1
高峭 1
高峰 1221
高峻 7
高崖 6
高崗 14
高嶺土 29
高嶺石 3
高州 2
高工 991
高差 33
高市 1464
高帝 9
高師 174
高帽 5
高帽子 7
高平 26
高年 53
高年級 563
高幹 25
高度 3002
高度化 9
高度表 34
高度計 20
高廊 0
高強 127
高強度 103
高徒 8
高得 1178
高德納 0
高性能 417
高懸 23
高戴 0
高手 3503
高手如林 0
高手如雲 11
高才 10
高才生 0
高抬 6
高抬貴手 27
高抬身價 0
高招 18
高挑 25
高掛 55
高攀 7
高攀不上 1
高攀不起 0
高效 86
高效能 294
高敞 2
高文典冊 0
高斯 92
高於 748
高昂 134
高明 283
高普考 1
高會 27
高朋滿座 14
高材 10
高材生 26
高枕 5
高枕無憂 32
高架 140
高架公路 2
高架橋 64
高架道路 19
高架鐵路 1
高架鐵道 1
高校 153
高桿 16
高桿 16
高梁 29
高梁米 0
高梁酒 11
高棉 54
高棉人 0
高棚 0
高樓 179
高樓大廈 39
高樓大廈 39
高標 0
高標準 57
高檔 196
高欄 2
高欄賽 0
高歌 47
高歌一曲 9
高歌一曲 9
高歡 0
高步雲衢 0
高段 16
高氣 15
高氣壓 17
高深 121
高深莫測 11
高湯 2
高溫 870
高溫氣候 0
高溫計 1
高漲 215
高漸離 0
高潔 36
高潮 409
高潮時 17
高潮線 8
高潮點 1
高澤 1
高照 44
高熱 75
高燒 88
高燥 0
高爐 96
高爵 0
高爽 1
高爾基 5
高爾夫 1170
高爾夫球 500
高爾夫球場 52
高爾夫球場 529
高爾察克 0
高牆 48
高猷 0
高球 178
高產 44
高甲戲 1
高瘦 19
高的 1179
高盧 11
高相 6
高瞻遠矚 19
高矮 19
高矮不一 1
高碳鋼 13
高祖 25
高祖母 0
高祖父 2
高票 84
高票當選 89
高科技 2634
高稈作物 1
高程 84
高積雲 2
高穹 2
高穹 2
高空 299
高空中 11
高空彈跳 0
高空病 0
高立 117
高竿 12
高等 1097
高等動物 12
高等學府 23
高等學校 85
高等教育 843
高等植物 16
高等法院 474
高等考試 501
高等院校 20
高筋麵粉 1
高節 3
高粱 54
高粱酒 44
高級 5565
高級中學 2972
高級品 18
高級將領 13
高級社 3
高級神經 0
高級神經活動 1
高級者 6
高級階段 2
高統靴 0
高緯度 42
高縣 441
高縣人 0
高纖 1
高義 30
高翔 7
高考 423
高考及格 43
高聚物 1
高聞 0
高聲 117
高聳 84
高聳入雲 14
高職 1993
高能 181
高能量 75
高腔 2
高臥 3
高臺 21
高興 2880
高興起來 6
高舉 133
高舉著 14
高薪 171
高處 136
高處不勝寒 37
高蛋白 39
高血壓 1021
高血脂 1
高行 7
高行 7
高要 7
高見 299
高視闊步 1
高觀 2
高詠 0
高誼 4
高誼 4
高調 19
高談 17
高談闊論 31
高論 33
高貲 0
高貴 228
高貴不貴 27
高賢 29
高起來 4
高超 165
高超無比 0
高超絕倫 0
高足 15
高足弟子 0
高跟 23
高跟鞋 249
高跳台 0
高蹺 23
高躅 0
高車 8
高農 96
高迎祥 0
高速 2606
高速公路 1360
高速度 41
高速行駛 47
高速鋼 7
高速鐵路 268
高逸 19
高過 92
高達 2119
高遠 20
高遷 1
高邁 2
高醫 426
高錳酸鉀 42
高鐵 0
高長虹 1
高閣 11
高陞 5
高院 0
高陽 83
高階 1106
高階層 26
高階語言 46
高雄 21501
高雄中學 461
高雄人 108
高雄工商專 1
高雄工專 58
高雄市 18426
高雄師範大學 5
高雄師範學院 5
高雄應用科技大學 1
高雄海專 48
高雄海洋科技大學 1
高雄港 516
高雄科學技術學院 1
高雄縣 11054
高雄縣長 54
高雄醫學大學 1288
高雄醫學院 1200
高雅 192
高電阻線 0
高音 143
高音部 2
高韻 3
高頭 6
高頭大馬 4
高頻 300
高頻率 30
高額 228
高顯 3
高風 2
高風亮節 6
高飛 69
高飛遠走 0
高高 100
高高低低 2
高高在上 46
高高的 66
高高地 67
高高興興 50
高鳴 4
高麗 113
高麗 500
高麗人 2
高麗參 2
高麗菜 54
高點 264
高鼻 3
高鼻子 1
高齡 352
高齡產婦 23
髙 0
髚 0
髛 0
髜 77
髜 77
髜 77
髝 0
髝 0
髞 0
髟 11
髠 0
髡 589
髢 0
髣 8
髤 0
髥 0
髦 51
髦牛 1
髧 7
髨 0
髩 0
髩 0
髪 0
髫 15
髫年 0
髬 34
髭 31
髭毛 0
髭鬚 3
髮 1301
髮型 603
髮夾 32
髮妻 3
髮姐 0
髮帶 8
髮廊 130
髮式 13
髮指 5
髮捲 14
髮根 12
髮毛 2
髮油 2
髮狀 0
髮短心長 0
髮端 1
髮簪 0
髮結 2
髮絲 71
髮網 10
髮膚 8
髮膠 18
髮菜 41
髮蠟 1
髮蠟條 0
髮質 1
髮辮 2
髮針 2
髮長 6
髮際 11
髮霜 6
髮髻 9
髮鬢 3
髯 52
髯 52
髯兒 1
髯兒 1
髯口 0
髰 0
髱 25
髲 28
髳 8
髴 0
髵 0
髶 33
髷 3
髸 0
髹 13
髺 3
髻 42
髼 8
髽 8
髽髻 0
髾 12
髿 0
鬀 0
鬁 9
鬂 0
鬂 0
鬃 60
鬃毛 8
鬄 74
鬅 54
鬅鬆 0
鬆 640
鬆了 43
鬆了一口氣 46
鬆些 1
鬆勁 0
鬆動 160
鬆口 45
鬆土 5
鬆土機 0
鬆弛 225
鬆弛下來 1
鬆弛到 1
鬆快 2
鬆懈 59
鬆懈下 1
鬆懈下來 2
鬆手 39
鬆散 85
鬆柔 6
鬆毛蟲 0
鬆浮 1
鬆綁 202
鬆緊 46
鬆緊帶 32
鬆緩 1
鬆脆 8
鬆脆餅 1
鬆脫 41
鬆起 0
鬆起來 0
鬆軟 60
鬆軟適口 0
鬆通 1
鬆開 51
鬆餅 51
鬆鬆 22
鬆鬆垮垮 0
鬆鬆散散 0
鬆鬆脆脆 1
鬆鬆軟軟 1
鬇 0
鬈 69
鬈曲 2
鬈毛 0
鬈髮 2
鬉 0
鬊 133
鬋 92
鬌 57
鬌 57
鬍 52
鬍子 113
鬍子渣 0
鬍梢 0
鬍渣 9
鬍髭 0
鬍鬚 47
鬍鬚渣 0
鬎 119
鬏 0
鬐 4
鬑 7
鬒 5
鬔 0
鬕 220
鬕 220
鬖 111
鬗 305
鬘 128
鬙 655
鬚 279
鬚根 18
鬚毛 0
鬚生 1
鬚眉 5
鬚髮 3
鬚髮皆白 0
鬚鬚 1
鬛 0
鬜 0
鬞 3
鬟 10
鬠 7
鬠 7
鬠 7
鬡 0
鬢 45
鬢毛 6
鬢腳 0
鬢角 9
鬢邊 3
鬢霜 1
鬢髮 10
鬢髮皆白 0
鬣 14
鬣狗 5
鬣鬃 0
鬤 129
鬥 2560
鬥上 3
鬥不過 6
鬥了 9
鬥了起來 0
鬥來鬥去 1
鬥倒 6
鬥勁 0
鬥口 2
鬥嘴 22
鬥士 239
鬥子 0
鬥弄 0
鬥志 117
鬥志昂揚 1
鬥成 2
鬥批改 0
鬥技 36
鬥智 58
鬥智鬥力 6
鬥毆 22
鬥氣 12
鬥法 21
鬥爭 325
鬥爭大會 1
鬥爭形式 0
鬥爭性 2
鬥爭意志 0
鬥爭方式 0
鬥牛 91
鬥牛場 3
鬥牛場 3
鬥牛士 19
鬥狠 9
鬥眼 0
鬥私批修 0
鬥草 1
鬥著 1
鬥蟋蟀 3
鬥起 0
鬥起來 0
鬥陣 21
鬥雞 10
鬥雞場 1
鬥雞場 1
鬥雞眼 7
鬥鬥 4
鬥鬥嘴 0
鬥魚 133
鬥鵪鶉 1
鬦 0
鬧 668
鬧中取靜 23
鬧事 34
鬧來鬧去 0
鬧個 6
鬧出 27
鬧出來 0
鬧出去 0
鬧出笑話 1
鬧劇 56
鬧區 64
鬧哄哄 17
鬧喧 0
鬧場 44
鬧場 44
鬧大 3
鬧市 18
鬧彆扭 11
鬧意氣 0
鬧意見 1
鬧房 6
鬧水災 7
鬧洞房 4
鬧熱 15
鬧熱滾滾 7
鬧病 0
鬧窮 1
鬧笑話 40
鬧翻 13
鬧翻天 6
鬧肚子 1
鬧脾氣 2
鬧著 11
鬧著玩 0
鬧起 2
鬧起來 2
鬧鐘 121
鬧革命 4
鬧飢荒 1
鬧鬧攘攘 0
鬧鬼 19
鬨 21
鬨堂 0
鬨笑 0
鬩 173
鬩牆 8
鬪 0
鬫 411
鬬 0
鬭 0
鬮 161
鬯 10
鬰 0
鬱 596
鬱卒 0
鬱塞 0
鬱悒 0
鬱悶 41
鬱悶不樂 0
鬱憤 1
鬱抑 3
鬱積 18
鬱結 9
鬱郁 0
鬱金 1
鬱金香 175
鬱閉 1
鬱閉度 0
鬱鬱 15
鬱鬱不得志 3
鬱鬱寡歡 12
鬱黑 0
鬲 45
鬲 45
鬳 5
鬴 0
鬴 0
鬵 31
鬷 50
鬸 0
鬹 0
鬹 0
鬺 35
鬻 135
鬻文為生 0
鬼 32359
鬼主意 1
鬼使神差 0
鬼剃頭 11
鬼叫 17
鬼叫聲 0
鬼哭神號 6
鬼地方 7
鬼城 7
鬼天氣 0
鬼子 31
鬼子兵 3
鬼屋 48
鬼屋傳奇 0
鬼怪 36
鬼才 34
鬼扯 11
鬼把戲 11
鬼故事 288
鬼斧神工 31
鬼月 65
鬼氣衝天 0
鬼混 30
鬼火 19
鬼片 4
鬼片當道 0
鬼物 0
鬼王 225
鬼畫符 7
鬼神 118
鬼祟 7
鬼節 24
鬼精靈 3
鬼聰明 1
鬼胎 3
鬼臉 14
鬼船 1
鬼蜮 1
鬼蜮技倆 0
鬼計 8
鬼計多端 1
鬼話 47
鬼話連篇 60
鬼車 0
鬼迷心竅 3
鬼門 14
鬼門道 18
鬼門關 18
鬼靈精 15
鬼頭 41
鬼頭鬼腦 2
鬼鬼祟祟 23
鬼魂 77
鬼魅 52
鬼魊 0
鬼魔 18
鬼點子 7
鬼黠 0
鬽 0
鬾 0
鬿 0
魀 0
魀 0
魁 728
魁偉 3
魁北克 195
魁星 18
魁梧 16
魁梧 16
魁首 8
魂 1544
魂不守舍 6
魂不附體 3
魂歸 12
魂歸何處 0
魂車 7
魂轎 0
魂銷魄散 2
魂靈 4
魂飛天外 1
魂飛魄散 9
魂驚膽喪 0
魂魄 60
魃 6
魄 133
魄力 206
魅 521
魅力 1356
魅力十足 21
魅影 37
魅惑 37
魆 4
魈 1
魉 0
魊 0
魋 0
魌 0
魍 7
魍魅 0
魍魍兒 0
魍魍兒 0
魍魎 10
魎 92
魏 905
魏代 2
魏國 48
魏平澳 0
魏徵 13
魏德邁 1
魏忠賢 1
魏文侯 3
魏文帝 1
魏晉 70
魏晉南北朝 70
魏書 1
魏源 11
魏王 26
魏碑 5
魏軍 15
魏長生 1
魏闕 1
魐 0
魐 0
魑 36
魑魅 10
魑魅魍魎 0
魒 89
魓 0
魔 3508
魔仙 0
魔劍 52
魔力 261
魔合羅 1
魔咒 33
魔堡 49
魔境 90
魔奇 7
魔女 392
魔宮 51
魔戒 186
魔掌 39
魔教 19
魔杖 12
魔氈 0
魔池 0
魔法 1332
魔法師 348
魔爪 24
魔犬 0
魔王 278
魔瓶 0
魔界 185
魔眼 21
魔石 33
魔笛 39
魔羯 1
魔羯座 1
魔術 587
魔術家 1
魔術師 232
魔道 17
魔邪 1
魔僕 30
魔鏡 30
魔障 14
魔電 44
魔音 17
魔音穿腦 0
魔頭 36
魔高一丈 17
魔高一尺 1
魔鬼 743
魔鬼連 2
魔幻 600
魔幻小說 600
魔幻文學 600
魕 104
魕 104
魖 150
魗 0
魘 20
魘寐 0
魘魅 0
魙 66
魙 66
魙 66
魚 8485
魚下巴 1
魚丸 128
魚丸湯 16
魚介 40
魚兒 365
魚兒 365
魚刺 12
魚卵 38
魚叉 25
魚場 12
魚場 12
魚塊 5
魚塘 5
魚子 22
魚子醬 18
魚尾 44
魚尾板 0
魚尾紋 16
魚市 20
魚市場 24
魚市場 241
魚形 9
魚排 25
魚村 4
魚業 5
魚水 18
魚水之歡 6
魚水相逢 0
魚池 499
魚沈雁杳 0
魚油 92
魚游釜中 0
魚湯 24
魚片 54
魚狗 10
魚生 17
魚產 29
魚皮 10
魚目混珠 14
魚目混珠 14
魚秧 0
魚種 190
魚竿 5
魚簍 3
魚籃 3
魚米之鄉 2
魚粉 37
魚網 48
魚缸 121
魚群 88
魚翅 84
魚翅羹 2
魚肉 89
魚肉人民 0
魚肉百姓 2
魚肉鄉民 1
魚肉鄉里 1
魚肚 12
魚肚白 2
魚肝 4
魚肝油 8
魚腥 8
魚腥味 4
魚膠 0
魚膾 0
魚與熊掌不可兼得 7
魚船 5
魚花 8
魚苗 104
魚蝦 101
魚貨 0
魚販 12
魚貫 18
魚貫而入 1
魚道 30
魚鉤 12
魚雁 6
魚雁往返 9
魚雷 73
魚雷艇 40
魚露 1
魚頭 65
魚頭 65
魚類 2083
魚類學 12
魚餌 22
魚香茄子 28
魚骨 15
魚骨頭 13
魚骨頭 13
魚鬆 14
魚鰓 8
魚鰭 12
魚鰾 4
魚鱗 35
魚鱗圖冊 0
魚鱗片 1
魚鱗癬 8
魚鹽之利 1
魚龍曼衍 1
魛 64
魜 0
魝 0
魝 0
魟 43
魟魚 10
魠 92
魡 65
魢 0
魣 0
魣 0
魤 17
魥 0
魦 3
魧 7
魨 211
魩 0
魪 0
魫 0
魬 31
魭 0
魮 0
魮 0
魯 3126
魯凱族 189
魯國 31
魯恭王 1
魯智深 41
魯殿靈光 0
魯濱遜 4
魯班 16
魯男子 3
魯直 4
魯肅 25
魯肉飯 0
魯莽 21
魯莽行事 1
魯賓遜 12
魯迅 103
魯鈍 8
魯魚亥豕 0
魰 14
魱 5
魲 0
魴 22
魵 3
魶 2
魷 100
魷魚 94
魷魚片 1
魷魚絲 16
魷魚羹 0
魺 4
魺 4
魻 5
魼 6
魽 741
魾 10
魿 0
魿 0
鮀 9
鮁 0
鮁 0
鮂 5
鮃 0
鮄 0
鮅 2
鮆 8
鮇 8
鮈 1463
鮉 0
鮉 0
鮊 0
鮋 0
鮋 0
鮌 0
鮍 0
鮎 0
鮏 0
鮐 12
鮑 509
鮑叔牙 5
鮑威爾 16
鮑敬言 0
鮑照 12
鮑魚 87
鮑魚之肆 0
鮒 13
鮓 4
鮓 4
鮔 0
鮕 6
鮙 0
鮚 12
鮛 5
鮜 0
鮝 0
鮞 20
鮟 0
鮠 19
鮡 24
鮢 25
鮣 0
鮤 12
鮥 864
鮦 29
鮧 0
鮨 34
鮩 0
鮪 228
鮪釣 40
鮪魚 148
鮪魚肚 1
鮫 66
鮫鰈 0
鮬 0
鮭 150
鮭魚 269
鮮 13919
鮮 1585
鮮乳 0
鮮亮 13
鮮卑 3
鮮卑人 0
鮮味 28
鮮奶 89
鮮奶油 107
鮮妍 1
鮮嫩 36
鮮少 116
鮮度 85
鮮彩 2
鮮明 259
鮮明對比 4
鮮有 42
鮮果 41
鮮活 81
鮮湯 9
鮮紅 88
鮮紅色 26
鮮綠 20
鮮綠色 3
鮮美 71
鮮肉 12
鮮艷 91
鮮艷艷 0
鮮花 589
鮮血 180
鮮血淋漓 3
鮮衣美食 0
鮮話 0
鮮豔 89
鮮豔奪目 1
鮮貨 16
鮮食 21
鮮魚 43
鮮麗 25
鮯 9
鮰 0
鮳 0
鮵 106
鮶 29
鮷 0
鮸 94
鮸 94
鮹 47
鮺 0
鮻 0
鮼 0
鮽 32
鮾 0
鮿 56
鯀 14
鯁 157
鯁噎 0
鯁慰 0
鯁涕 0
鯁直 1
鯁言 0
鯁論 0
鯂 0
鯃 26
鯄 98
鯅 0
鯅 0
鯆 59
鯆 59
鯇 47
鯈 6
鯉 186
鯉素 0
鯉躍龍門 2
鯉魚 200
鯉魚池 2
鯉魚潭 180
鯊 137
鯊魚 204
鯊魚肉 0
鯋 0
鯌 0
鯍 0
鯒 0
鯓 176
鯔 88
鯕 52
鯖 59
鯖魚 26
鯗 74
鯘 0
鯙 151
鯚 47
鯛 1176
鯛類 16
鯛魚 35
鯜 243
鯝 0
鯞 131
鯟 0
鯠 67
鯡 110
鯡魚 1
鯢 72
鯣 0
鯤 262
鯤鵬 1
鯥 59
鯦 29
鯧 53
鯧魚 3
鯨 1269
鯨吞 13
鯨油 4
鯨鬚 1
鯨魚 229
鯩 0
鯪 108
鯪魚 0
鯪鯉 0
鯫 59
鯬 87
鯭 0
鯮 0
鯯 0
鯰 94
鯰魚 17
鯶 0
鯶 0
鯶 0
鯷 9
鯷魚 6
鯸 4
鯸 4
鯹 0
鯺 0
鯻 0
鯼 0
鯽 32
鯽魚 25
鯾 0
鯿 0
鰀 0
鰀 0
鰀 0
鰁 0
鰂 0
鰃 0
鰄 0
鰅 5
鰆 36
鰇 6
鰈 32
鰈鶼 0
鰉 9
鰊 0
鰋 6
鰋 6
鰌 0
鰌 0
鰍 444
鰎 9
鰏 0
鰐 0
鰑 0
鰒 7
鰓 348
鰓 348
鰔 12
鰔 12
鰔 12
鰕 0
鰕 0
鰖 0
鰗 6
鰛 0
鰜 87
鰝 146
鰝 146
鰝 146
鰟 0
鰠 0
鰡 105
鰢 0
鰣 39
鰤 171
鰥 7
鰥夫 3
鰥寡 4
鰥寡孤獨 4
鰥居 3
鰧 0
鰨 38
鰩 21
鰪 0
鰪 0
鰫 225
鰬 39
鰭 266
鰮 0
鰱 125
鰱魚 12
鰲 225
鰲山 1
鰲拜 1
鰲裡奪尊 1
鰲頭 31
鰲頭獨占 1
鰲魚 1
鰳 108
鰴 175
鰵 0
鰶 201
鰷 168
鰷魚 2
鰸 0
鰹 209
鰹魚 1
鰹鳥 1
鰺 0
鰻 416
鰻魚 135
鰻鱺 13
鰼 229
鰽 21
鰽 21
鰽 21
鰾 100
鰾膠 0
鰿 144
鱀 13
鱁 246
鱁 246
鱂 0
鱃 0
鱃 0
鱄 87
鱅 0
鱅 0
鱆 273
鱈 267
鱈魚 71
鱉 199
鱉甲 10
鱉魚 1
鱊 3
鱊 3
鱋 4
鱌 5
鱍 45
鱎 8
鱎 8
鱎 8
鱏 0
鱐 6
鱑 0
鱒 48
鱒 48
鱒魚 69
鱓 0
鱔 42
鱔魚 21
鱕 7
鱖 23
鱖 23
鱖豚 1
鱖魚 2
鱗 413
鱗介 1
鱗傷 1
鱗形 0
鱗次 3
鱗次櫛比 2
鱗比 0
鱗毛 17
鱗爪 1
鱗片 57
鱗片狀 8
鱗甲 7
鱗莖 5
鱗葉 1
鱗集 0
鱘 15
鱘魚 6
鱙 7
鱝 0
鱞 5
鱟 20
鱠 36
鱡 0
鱢 3
鱣 93
鱣 93
鱤 0
鱥 0
鱥 0
鱦 10
鱧 171
鱨 6
鱬 0
鱭 4
鱮 4
鱯 0
鱯 0
鱱 9
鱲 0
鱳 8
鱳 8
鱴 8
鱵 14
鱶 0
鱷 130
鱷魚 187
鱷魚皮 3
鱸 268
鱸魚 59
鱹 77
鱺 110
鱻 0
鱻 0
鳥 6553
鳥人 124
鳥兒 119
鳥兒 119
鳥叫 27
鳥叫聲 15
鳥嘌呤 1
鳥嘴 21
鳥園 174
鳥媒花 0
鳥屋 3
鳥巢 34
鳥店 67
鳥松 167
鳥槍 1
鳥氣 4
鳥為食亡 2
鳥爪 2
鳥爪 2
鳥獸 49
鳥獸散 6
鳥盡弓藏 0
鳥瞰 111
鳥瞰圖 114
鳥窩 7
鳥窩式 0
鳥籠 27
鳥糞 8
鳥翼 0
鳥聲 13
鳥蛋 24
鳥語 33
鳥語花香 47
鳥雀 1
鳥頭 11
鳥頭 11
鳥類 1092
鳥類學 6
鳥類學家 3
鳥鳥 11
鳥鳴 39
鳦 19
鳧 112
鳧水 0
鳨 0
鳩 451
鳩佔鵲巢 2
鳩占鵲巢 1
鳩合 1
鳩形鵠面 0
鳩摩羅什 1
鳩聚 0
鳩集 1
鳪 52
鳫 0
鳬 0
鳬 0
鳭 20
鳮 0
鳯 0
鳱 18
鳲 13
鳳 4745
鳳仙 29
鳳仙花 33
鳳儀 39
鳳冠 11
鳳冠霞帔 1
鳳凰 883
鳳凰于飛 0
鳳凰山 21
鳳凰木 36
鳳凰樹 11
鳳凰竹 7
鳳凰花 27
鳳凰花開 14
鳳尾松 0
鳳尾竹 0
鳳尾草 4
鳳尾魚 0
鳳山 655
鳳山市 1634
鳳山溪 15
鳳梨 305
鳳梨汁 14
鳳梨酥 1
鳳毛麟角 12
鳳求凰 6
鳳爪 0
鳳爪 0
鳳眼 23
鳳翔 32
鳳蝶 253
鳳輦 0
鳳閣龍樓 1
鳳陽 18
鳳雛 9
鳳駕 0
鳳髓龍肝 0
鳴 1644
鳴不平 12
鳴冤 4
鳴冤叫屈 1
鳴叫 40
鳴奏 0
鳴放 5
鳴放鞭炮 1
鳴槍 27
鳴槍示警 2
鳴炮 5
鳴琴 7
鳴石 0
鳴砲 1
鳴禽 0
鳴笛 22
鳴管 0
鳴號 1
鳴金 7
鳴金擊鼓 0
鳴金收兵 4
鳴鑼 4
鳴鑾 0
鳴雁 0
鳴響 7
鳴鳥 0
鳴鳳 17
鳴鼓 4
鳴鼓奏樂 0
鳴鼓而攻 2
鳵 36
鳶 245
鳶尾科 3
鳷 37
鳸 0
鳹 22
鳺 28
鳻 40
鳼 26
鳽 5
鳿 32
鴀 31
鴁 0
鴁 0
鴂 0
鴂 0
鴃 404
鴃 404
鴃舌 0
鴃舌 0
鴄 6
鴅 15
鴆 8
鴆媒 0
鴆酒 0
鴇 28
鴇妓 0
鴇母 1
鴈 83
鴉 337
鴉嘴鋤 0
鴉片 304
鴉片戰爭 54
鴉片煙 4
鴉片貿易 1
鴉片鬼 0
鴉牙 0
鴉蛋 2
鴉雀無聲 18
鴉青 1
鴉髻 0
鴉鬢 0
鴊 0
鴊 0
鴋 0
鴌 0
鴍 0
鴏 0
鴐 150
鴑 0
鴒 23
鴓 0
鴔 234
鴕 17
鴕鳥 131
鴖 0
鴗 378
鴘 159
鴙 78
鴚 0
鴛 36
鴛侶 0
鴛盟 0
鴛鴦 257
鴛鴦戲水 1
鴛鴦蝴蝶 18
鴛鴦蝴蝶派 2
鴛鴦鍋 1
鴜 0
鴝 175
鴞 339
鴟 127
鴟 127
鴟尾 0
鴟鴞 31
鴟鵂 0
鴠 204
鴡 0
鴡 0
鴢 102
鴣 12
鴣鴣 0
鴤 0
鴥 158
鴦 39
鴧 0
鴨 1021
鴨叫 0
鴨嘴 30
鴨嘴獸 7
鴨嘴筆 0
鴨子 290
鴨子划水 1
鴨掌 6
鴨毛 3
鴨絨 5
鴨絨毛 0
鴨綠江 18
鴨肉 37
鴨肉湯 1
鴨肫肝 0
鴨腳 6
鴨舌 12
鴨舌帽 11
鴨舌頭 1
鴨蛋 46
鴨蛋圓 0
鴨蛋殼 0
鴨蛋臉 0
鴨霸 40
鴨頭 10
鴩 207
鴪 0
鴭 88
鴮 5
鴯 8
鴰 4
鴱 5
鴳 13
鴴 0
鴵 0
鴶 8
鴷 99
鴸 4
鴹 0
鴺 0
鴺 0
鴻 6026
鴻儒 67
鴻博 17
鴻圖 45
鴻圖大展 8
鴻基 95
鴻志 65
鴻恩 9
鴻文 82
鴻業 10
鴻毛 3
鴻源 69
鴻溝 59
鴻漸 7
鴻福 43
鴻福齊天 4
鴻範 2
鴻運 195
鴻門宴 16
鴻雁 10
鴻雁傳書 0
鴻雁哀鳴 0
鴻飛冥冥 0
鴻鵠 5
鴻鵠之志 2
鴼 0
鴽 34
鴾 218
鴿 340
鴿子 151
鴿子籠 8
鴿房 0
鴿派 1
鴿籠 3
鴿翼 0
鴿賽 0
鵀 65
鵁 1529
鵂 1009
鵃 288
鵅 367
鵊 95
鵋 123
鵌 50
鵍 0
鵎 0
鵏 31
鵐 0
鵑 132
鵒 13
鵓 36
鵓鴿 1
鵔 13
鵔鸃 1
鵔鸃冠 1
鵕 0
鵕 0
鵖 103
鵖 103
鵗 12
鵘 21
鵘 21
鵙 35
鵙 35
鵚 20
鵚鶖 1
鵛 59
鵛 59
鵜 34
鵜形目 1
鵜鴃 1
鵜鶘 8
鵝 417
鵝卵 1
鵝卵石 7
鵝口瘡 9
鵝掌 27
鵝毛 9
鵝絨 5
鵝絨毛 0
鵝群 6
鵝翎扇 0
鵝肉 68
鵝肉湯 0
鵝肝 1
鵝肝醬 1
鵝蛋 13
鵝蛋臉 1
鵝行鴨步 0
鵝鑾鼻 59
鵝頭 0
鵝鸞鼻 7
鵝黃 28
鵝黃色 11
鵞 0
鵟 48
鵠 59
鵠 59
鵠志 0
鵠的 12
鵠立 0
鵡 32
鵢 0
鵣 0
鵦 0
鵧 8
鵨 24
鵩 6
鵪 9
鵪鶉 18
鵪鶉蛋 0
鵫 23
鵬 2334
鵬程 29
鵬程萬里 14
鵬舉 3
鵬鳥 1
鵭 0
鵭 0
鵮 0
鵯 152
鵰 589
鵰悍 0
鵱 8
鵲 163
鵲 163
鵲報 1
鵲巢 3
鵲巢鳩占 0
鵲橋 20
鵲起 2
鵳 18
鵴 22
鵵 19
鵶 0
鵶 0
鵷 5
鵸 6
鵹 11
鵺 0
鵻 16
鵼 0
鵽 13
鵾 0
鵾 0
鵿 35
鶀 3
鶁 6
鶂 17
鶃 0
鶄 9
鶅 13
鶆 6
鶇 231
鶈 2
鶉 22
鶉居 0
鶉衣 1
鶊 5
鶋 30
鶌 20
鶐 3
鶑 0
鶒 3
鶓 0
鶔 16
鶖 23
鶗 110
鶘 5
鶙 1
鶚 108
鶛 4
鶛 4
鶛 4
鶜 206
鶝 6
鶞 122
鶟 7
鶠 787
鶡 242
鶢 148
鶣 86
鶣 86
鶤 3
鶤雞 1
鶥 0
鶦 40
鶧 90
鶨 55
鶩 68
鶩 68
鶪 365
鶬 139
鶭 63
鶮 0
鶯 944
鶯啼燕語 0
鶯歌 320
鶯歌燕舞 0
鶯歌鎮 565
鶯燕 5
鶯聲燕語 2
鶯語 2
鶯谷 1
鶯鶯 8
鶯鶯燕燕 3
鶰 0
鶱 79
鶲 492
鶳 52
鶴 130
鶴 1300
鶴侶 0
鶴唳 1
鶴唳華亭 1
鶴立 7
鶴立雞群 6
鶴羽 1
鶴羽 1
鶴頂紅 6
鶴髮 3
鶴髮童顏 3
鶴鶉蛋 0
鶵 80
鶶 123
鶷 80
鶷 80
鶷 80
鶷 80
鶸 4
鶹 89
鶺 189
鶺鴒 133
鶻 115
鶻 115
鶻入鴉群 1
鶻崙吞棗 1
鶻淪 1
鶻突 1
鶻鴒 1
鶻鴒眼 1
鶼 72
鶼鰈 1
鶼鰈情深 4
鶽 0
鶾 107
鶾 107
鶿 98
鷀 0
鷁 289
鷁首 1
鷂 16
鷂子 1
鷂子翻身 1
鷃 60
鷄 0
鷅 112
鷇 174
鷇音 0
鷈 90
鷉 0
鷊 88
鷋 136
鷌 42
鷍 58
鷍 58
鷎 109
鷎 109
鷏 112
鷏 112
鷐 113
鷑 114
鷒 141
鷒 141
鷓 159
鷓鴣 12
鷓鴣菜 0
鷔 0
鷕 48
鷖 245
鷗 295
鷗波 1
鷗盟 1
鷗邊 1
鷗鳥 2
鷗鳥忘機 1
鷗鴉 1
鷗鷺忘機 1
鷘 90
鷙 124
鷙鳥 0
鷙鷹 0
鷚 175
鷛 319
鷛 319
鷜 122
鷝 69
鷞 139
鷟 162
鷠 0
鷡 18
鷢 20
鷣 10
鷤 8
鷤 8
鷤 8
鷥 25
鷦 98
鷧 0
鷨 15
鷩 113
鷪 0
鷫 17
鷬 9
鷭 8
鷮 3
鷯 37
鷰 11
鷱 0
鷲 236
鷳 17
鷴 55
鷵 91
鷶 67
鷷 8
鷸 265
鷸蚌相爭 2
鷸蚌相爭漁翁得利 0
鷸類 3
鷹 14294
鷹勾鼻 4
鷹式飛彈 1
鷹揚 24
鷹揚萬里 1
鷹架 48
鷹派 17
鷹爪 11
鷹犬 1
鷹眼 10
鷹瞵鴞視 0
鷹視狼顧 1
鷹鉤鼻 0
鷹鼻鷂眼 0
鷺 773
鷺序 0
鷺鷥 128
鷻 12
鷼 0
鷽 11
鷾 6
鷿 88
鸀 5
鸀 5
鸁 10
鸂 4
鸂 4
鸃 0
鸄 4
鸄 4
鸅 12
鸆 17
鸇 5
鸈 0
鸉 11
鸊 0
鸋 9
鸌 0
鸍 2
鸍 2
鸎 0
鸏 7
鸏 7
鸏 7
鸐 7
鸑 4
鸒 27
鸓 1
鸔 7
鸕 84
鸕鶿 16
鸖 0
鸖 0
鸗 72
鸘 0
鸘 0
鸙 92
鸚 205
鸚鵡 257
鸚鵡螺 11
鸛 48
鸛雀 2
鸛鳥 1
鸜 0
鸝 113
鸞 131
鸞翔鳳集 0
鸞輿 0
鸞鳳 8
鸞鳳和鳴 1
鹵 133
鹵化 33
鹵化物 18
鹵味 1
鹵族 1
鹵水 0
鹵汁 1
鹵簿 0
鹵素 110
鹵莽 7
鹵鈍 0
鹶 0
鹷 0
鹹 608
鹹味 20
鹹土 0
鹹度 3
鹹得 23
鹹水 31
鹹水湖 2
鹹水魚 10
鹹海 8
鹹淡 0
鹹淡適中 0
鹹湖 0
鹹湯 5
鹹的 23
鹹肉 7
鹹菜 15
鹹菜乾 0
鹹蛋 31
鹹豬手 1
鹹豬肉 1
鹹酥雞 1
鹹類 1
鹹魚 30
鹹魚翻身 10
鹹鴨蛋 3
鹹鹵 0
鹹鹹 29
鹺 77
鹻 0
鹼 672
鹼化 18
鹼土金屬 2
鹼地 2
鹼度 36
鹼性 342
鹼性反應 3
鹼性平爐 0
鹼性氧化物 0
鹼水 17
鹼液 25
鹼液 25
鹼熔 0
鹼石灰 0
鹼纖維素 0
鹼金屬 10
鹼類 9
鹽 2211
鹽井 0
鹽分 71
鹽包 5
鹽區 3
鹽味 5
鹽商 7
鹽土 3
鹽城 22
鹽埔 105
鹽埔 105
鹽埕區 131
鹽基度 3
鹽基性 37
鹽場 27
鹽場 27
鹽寮 76
鹽層 1
鹽山 4
鹽巴 23
鹽市 0
鹽店 0
鹽度 55
鹽戶 1
鹽政 22
鹽析 4
鹽梟 0
鹽業 22
鹽民 3
鹽水 221
鹽水鎮 319
鹽池 2
鹽法 17
鹽湖 2
鹽湖城 15
鹽滷 3
鹽瓶 1
鹽田 62
鹽礦 13
鹽稅 9
鹽粉 1
鹽粒 1
鹽缸 0
鹽花 0
鹽賦 0
鹽運使 1
鹽酥雞 1
鹽酸 181
鹽量計 0
鹽鐵部 1
鹽類 485
鹽鹵 0
鹽鹼化 0
鹽鹼土 1
鹽鹼地 2
鹽鹼灘 0
鹿 2203
鹿名 0
鹿柴 3
鹿柴 3
鹿死誰手 9
鹿死誰手 9
鹿港 515
鹿港鎮 609
鹿特丹 99
鹿皮 19
鹿砦 1
鹿竹 0
鹿肉 3
鹿腳 0
鹿苑 15
鹿茸 17
鹿茸酒 2
鹿角 67
鹿谷 13
鹿谷鄉 319
鹿野 270
鹿食草 0
鹿鼎記 0
麀 2
麁 0
麂 253
麃 32
麃 32
麄 0
麅 0
麅 0
麆 85
麇 110
麈 90
麉 53
麉 53
麊 57
麋 8
麋沸 0
麋羚 0
麋鹿 17
麌 12
麌麌 1
麍 48
麎 27
麏 0
麏 0
麐 0
麑 23
麒 476
麒麟 295
麒麟座 13
麒麟送子 1
麓 107
麔 13
麕 0
麕 0
麖 0
麗 340
麗 9340
麗人 22
麗山 133
麗服 1
麗水 45
麗藻 1
麗質 6
麗質天生 2
麗都 34
麘 0
麙 88
麙 88
麚 65
麚 65
麛 98
麜 144
麝 24
麝牛 0
麝貓 5
麝香 52
麝香鹿 0
麞 0
麟 1936
麟兒 11
麟經 4
麟角 0
麠 31
麡 20
麢 0
麣 0
麤 433
麥 7156
麥乳 2
麥乳精 1
麥乳茶 0
麥克 457
麥克斯韋 18
麥克阿瑟 19
麥克風 661
麥冬 6
麥加 15
麥可 295
麥哲倫 23
麥子 49
麥收 2
麥根 1
麥根沙士 10
麥梗 0
麥浪 0
麥片 46
麥片粥 3
麥田 121
麥當勞 282
麥當樂 0
麥稈 2
麥穗 23
麥管 2
麥粉 32
麥粒 6
麥粥 0
麥精 6
麥芒 0
麥芽 59
麥芽乳 0
麥芽糖 23
麥芽茶 0
麥苗 42
麥茬 2
麥茶 1
麥角 9
麥酒 5
麥金塔 0
麥金德 0
麥高文 2
麥麩 3
麦 0
麦 0
麧 85
麨 0
麩 101
麩曲 0
麩皮 15
麩素 0
麩胺酸 14
麪 0
麫 0
麬 0
麭 164
麮 194
麯 0
麯 0
麯 0
麰 486
麱 0
麲 0
麳 0
麴 47
麴菌 23
麵 14100
麵價 2
麵包 807
麵包刀 0
麵包屑 2
麵包師 7
麵包店 130
麵包樹 5
麵包粉 1
麵團 23
麵店 275
麵廠 24
麵杖 0
麵條 318
麵灰 0
麵皮 20
麵筋 17
麵粉 321
麵粉袋 8
麵糊 29
麵線 100
麵茶 5
麵食 142
麵餃 1
麵餅 8
麶 83
麶 83
麷 90
麻 1913
麻俐 0
麻六甲 22
麻六甲海峽 4
麻刀 0
麻城 1
麻塞諸塞州 11
麻婆豆腐 12
麻子 32
麻將 580
麻將牌 1
麻州 78
麻布 27
麻木 111
麻木不仁 22
麻桿 2
麻油 60
麻油雞 15
麻煩 1884
麻田 29
麻疹 167
麻痹 16
麻痹不了 0
麻痹大意 0
麻痺 427
麻痺不了 0
麻痺大意 0
麻省 28
麻省理工學院 127
麻糬 1
麻紗 10
麻紡 0
麻線 13
麻織 5
麻織品 5
麻繩 14
麻胡 0
麻臉 3
麻花 9
麻薩諸塞州 10
麻薩諸州 0
麻藥 34
麻衣 12
麻袋 70
麻豆 454
麻豆文旦 29
麻辣 0
麻辣火鍋 1
麻辣鍋 1
麻醉 1406
麻醉劑 74
麻醉品 5
麻醉師 6
麻醉藥 23
麻醉針 0
麻醬 4
麻醬麵 2
麻雀 278
麻雀在後 0
麻雀雖小 18
麻類 9
麻麻呼呼 0
麻麻糊糊 0
麻麻糊糊 0
麻麻糊糊 0
麻麻糊糊 0
麻黃 17
麻黃素 4
麼 2536
麼 2536
麼 2536
麼 2536
麽 0
麽 0
麽 0
麽 0
麾 344
麾下 26
麾軍 0
黀 23
黁 0
黂 8
黃 28741
黃了 4
黃人 7
黃公望 10
黃冠 49
黃包車 5
黃嘌呤 1
黃土 68
黃土高原 10
黃埔 94
黃埔 94
黃埔江 8
黃埔江 8
黃埔軍官學校 0
黃埔軍官學校 0
黃埔軍校 5
黃埔軍校 5
黃堡 1
黃壤 1
黃字 6
黃山 145
黃岡 3
黃州 7
黃巢 9
黃巾 24
黃巾之亂 5
黃巾起義 0
黃帝 70
黃底 7
黃得 16
黃教 9
黃敬群 10
黃斑 59
黃昏 448
黃昏時候 2
黃曆 20
黃曲霉 0
黃梁 4
黃梁夢 1
黃梅 33
黃梅天 0
黃梅季 0
黃梅戲 1
黃梅調 3
黃梅雨 0
黃楊 13
黃樟 3
黃橋 0
黃毛 15
黃毛丫頭 4
黃沙 37
黃沙滾滾 4
黃河 250
黃河口 2
黃河流域 12
黃油 13
黃泉 76
黃泉之下 2
黃泉路 5
黃泥 17
黃浦 12
黃海 72
黃淮平原 11
黃湯 10
黃澄澄 7
黃濁 5
黃火亮 0
黃熱病 16
黃燈 28
黃燐 0
黃牛 84
黃牛票 5
黃班 1
黃瓜 203
黃瓜皮 0
黃疸 154
黃疸病 6
黃白 58
黃的 87
黃皮書 71
黃皮膚 7
黃石 71
黃石公園 24
黃石地區 0
黃石大峽谷 0
黃石河 0
黃石湖 0
黃石瀑布 0
黃砂 0
黃磷 5
黃禍 13
黃種 15
黃種人 17
黃米 2
黃粱 5
黃粱夢 0
黃絲帶 1
黃綠色 48
黃縣 1
黃老 95
黃老學派 1
黃腔 20
黃臉 1
黃臉婆 7
黃興 36
黃色 773
黃色人種 0
黃色小說 13
黃色工會 0
黃色故事 1
黃色書刊 7
黃色炸藥 9
黃色笑話 134
黃色素 0
黃色電影 0
黃花 114
黃花岡 3
黃花崗 3
黃花崗七十二烈士 6
黃花菜 0
黃花閨女 1
黃花魚 7
黃莓 0
黃菊 7
黃菊花 2
黃萎病 1
黃蓮 14
黃蘗 6
黃蜂 74
黃蠟 1
黃衣 36
黃衣主教 0
黃衣服 2
黃袍 12
黃袍加身 8
黃褐 12
黃褐色 56
黃豆 145
黃豆芽 6
黃連 55
黃連木 2
黃道 67
黃道吉日 11
黃道婆 0
黃道帶 0
黃道日 1
黃酒 4
黃金 2066
黃金時代 45
黃金歲月 21
黃金海岸 166
黃銅 55
黃銅礦 4
黃鐘 25
黃鐵礦 20
黃陵 0
黃雀 4
黃雀伺蟬 0
黃雀在後 8
黃霸 0
黃飛鴻 23
黃體 143
黃髮垂髫 0
黃魚 30
黃鳥 5
黃鶯 84
黃鶯出谷 4
黃鶴 50
黃鸝 16
黃麴毒素 44
黃麻 24
黃黃 34
黃鼠狼 29
黃鼠狼給雞拜年 2
黃龍 57
黃龍府 0
黄 0
黅 0
黆 0
黇 0
黈 16
黊 0
黊 0
黋 0
黋 0
黌 24
黌宮 4
黌舍 4
黍 199
黍子 0
黍米 1
黎 1654
黎元洪 8
黎國 12
黎巴嫩 52
黎德壽 0
黎族 1
黎明 808
黎明工專 76
黎民 15
黎黑 2
黏 591
黏一黏 0
黏上 15
黏上來 0
黏上去 0
黏下去 0
黏了 1
黏人 12
黏住 16
黏來黏去 0
黏到 3
黏力 2
黏合 34
黏合劑 14
黏合力 0
黏合成 0
黏土 152
黏在 55
黏好 6
黏完 0
黏度 81
黏度計 13
黏得 2
黏性 128
黏成 2
黏手 1
黏接 33
黏涎 1
黏液 92
黏液 92
黏滿 3
黏稠 76
黏米 0
黏結 15
黏結力 0
黏結性 1
黏膜 312
黏膜炎 1
黏膠 30
黏膩 1
黏著 117
黏著劑 130
黏著力 5
黏著性 26
黏貼 140
黏起 0
黏起來 3
黏過 0
黏附 15
黏附力 0
黏附性 0
黏魚 0
黏黏 14
黏點 0
黐 9
黐膠 1
黑 60
黑 6061
黑下 1
黑下來 3
黑不溜偢 0
黑不溜秋 0
黑乎乎 2
黑了 28
黑井島 0
黑人 375
黑信 0
黑光 11
黑函 0
黑勢力 1
黑名 0
黑名單 116
黑咕籠咚 0
黑啤酒 4
黑土 6
黑地 8
黑墨 5
黑墨水 0
黑墨汁 0
黑壓壓 13
黑壤 0
黑夜 327
黑奴 29
黑子 43
黑字 28
黑官 9
黑客 1
黑寡婦 13
黑山 6
黑市 91
黑布 16
黑帖 0
黑帶 39
黑帶高手 1
黑幕 19
黑幕小說 1
黑幕重重 4
黑幫 60
黑底 17
黑底白字 9
黑店 41
黑影 90
黑得 239
黑心 21
黑心肝 0
黑心腸 0
黑忽忽 2
黑手 195
黑手黨 27
黑斑 114
黑斑病 2
黑斑蚊 0
黑旗軍 1
黑星 35
黑晶 11
黑暗 1041
黑暗世界 5
黑暗時代 13
黑暗面 35
黑書 0
黑李子色 0
黑松 102
黑松汽水 7
黑松沙士 6
黑板 281
黑板報 0
黑板擦 1
黑格 7
黑格爾 61
黑棗 28
黑森林 41
黑槍 19
黑檀 3
黑檀木 8
黑檀樹 1
黑死病 27
黑氣 16
黑水 29
黑沈沈 1
黑沉沉 6
黑沙灣 0
黑河 25
黑油 8
黑油油 1
黑洞 106
黑洞洞 1
黑派 10
黑海 28
黑漆 7
黑漆漆 16
黑潮 126
黑澄澄 0
黑炭 9
黑煙 67
黑熊 105
黑熱病 1
黑燈瞎火 0
黑牌 4
黑狗 107
黑猩猩 85
黑疸 0
黑痣 7
黑瘦 1
黑白 1320
黑白不分 3
黑白兩道 17
黑白分明 16
黑白影片 5
黑白照片 12
黑白片 9
黑白花 6
黑白電視 6
黑的 239
黑皮 70
黑盒 1
黑盒子 125
黑眼 9
黑眼圈 1
黑眼珠 5
黑眼睛 3
黑眼鏡 1
黑石 14
黑磷 0
黑社會 47
黑種 3
黑種人 2
黑穗病 1
黑筆 11
黑管 11
黑箱 43
黑簇簇 0
黑糊糊 2
黑糖 13
黑紗 12
黑紙 5
黑線 37
黑胡椒 52
黑臉 55
黑色 1846
黑色星期五 16
黑色火藥 4
黑色素 90
黑色金屬 8
黑莓 4
黑蛇 6
黑蟻 2
黑衣 267
黑衣服 3
黑褐 52
黑話 53
黑豆 47
黑豬 21
黑豹黨 1
黑貂 5
黑貨 0
黑路 0
黑輪 0
黑道 403
黑道份子 11
黑道日 0
黑邊 6
黑邊眼鏡 2
黑鄉 3
黑金 0
黑鈣土 0
黑鉛 0
黑錢 8
黑鍋 6
黑鍵 2
黑陶 4
黑陶文化 7
黑雨 1
黑雲 20
黑霉 0
黑非洲 0
黑面 585
黑面琵鷺 0
黑頭 48
黑馬 97
黑體 53
黑體字 9
黑髮 51
黑髮人 5
黑魔法 1
黑魚 2
黑魯雪夫 0
黑鮪魚 1
黑鯨 0
黑麥 20
黑麵包 4
黑黑 61
黑黑家鼠 0
黑黑白白 1
黑黑的 57
黑黝黝 10
黑鼠 0
黑龍 36
黑龍江 108
黑龍江省 91
黒 0
黒 0
黒 0
黓 65
黔 32
黔江 2
黔首 1
黔首百姓 0
黔驢之技 0
黔驢技窮 4
黕 62
黖 251
黗 0
默 776
默不作聲 9
默化 5
默哀 23
默哀致意 0
默問 0
默啟 0
默坐 9
默契 303
默存 2
默寫 10
默從 0
默念 20
默思 0
默想 161
默書 1
默求 0
默然 49
默然不語 6
默片 6
默省 0
默祝 0
默禱 12
默算 0
默背 0
默視 1
默記 1
默許 38
默認 29
默誦 4
默讀 3
默默 835
默默不語 5
默默無聞 15
默默無言 5
黙 0
黚 13
黛 788
黛安娜 108
黛玉 107
黛眉 9
黛綠 2
黛綠年華 1
黛蛾 0
黛鬟 0
黜 26
黜免 0
黜斥 0
黜責 0
黜退 0
黜陟 0
黝 29
黝性 0
黝黑 39
點 20808
點擊 300
點上 174
點上來 3
點上去 2
點下 31
點下來 1
點下去 4
點不著 0
點中 34
點主 10
點了 285
點交 50
點亮 114
點來 53
點來點去 3
點個 11
點兒 55
點兒 55
點兵 2
點出 157
點出來 1
點出去 0
點到 94
點到為止 22
點券 20
點勘 1
點化 12
點半 343
點召 24
點名 275
點名冊 4
點名單 4
點名簿 8
點回 11
點回來 0
點回去 0
點在 249
點好 37
點子 520
點字 324
點完 10
點定 12
點將 248
點對點 0
點得 11
點得 787
點心 970
點心費 4
點戲 3
點接觸 1
點播 273
點播 273
點播曲 0
點播機 1
點播歌 1
點收 76
點數 456
點數 456
點明 274
點染 3
點檢 12
點歌 101
點法 14
點清 5
點湯 1
點源 5
點滴 622
點滴器 0
點滴歸公 0
點火 299
點煙 12
點燃 335
點燈 282
點畫 9
點發 9
點的 787
點睛 30
點石成金 25
點破 14
點穴 17
點穿 0
點竄 0
點算 6
點綴 108
點綴著 25
點線 27
點線面 43
點腳 0
點腳尖兒 0
點腳尖兒 0
點菜 42
點著 46
點號 6
點貨 6
點起 66
點起來 4
點距 125
點過 14
點過來 0
點過去 0
點選 0
點醒 13
點鈔 3
點鈔機 11
點錢 94
點鐘 16
點閱 58
點閱召集 4
點閱召集令 2
點陣 63
點集 0
點電荷 5
點面結合 0
點頭 466
點頭之交 2
點頭應允 0
點餐 1
點驗 42
點點 258
點點滴滴 340
點點頭 149
黟 29
黠 48
黠慧 1
黠獪 0
黢 0
黣 0
黤 221
黥 59
黥面 83
黦 67
黦 67
黧 142
黨 6206
黨主席 562
黨人 32
黨代表 78
黨內 862
黨內人士 7
黨內鬥爭 2
黨刊 0
黨務 200
黨友 9
黨史 47
黨同伐異 11
黨名 3
黨員 534
黨員大會 6
黨員證 0
黨國 73
黨國元老 4
黨團 759
黨團員 0
黨報 18
黨外 72
黨外人士 14
黨委 9
黨委書記 2
黨委會 0
黨小組 1
黨工 44
黨徒 9
黨徽 12
黨性 3
黨政 366
黨政機關 13
黨旗 36
黨校 10
黨格 6
黨棍 1
黨歌 6
黨法 0
黨派 286
黨營 298
黨爭 15
黨產 0
黨的基本 6
黨禁 34
黨禍 1
黨章 70
黨籍 712
黨紀 64
黨紀國法 0
黨組 25
黨綱 162
黨總支 0
黨義 1
黨羽 14
黨職 34
黨言 0
黨課 0
黨證 8
黨費 5
黨軍 4
黨部 1268
黨錮 0
黨錮之禍 2
黨鞭 48
黨風 2
黨魁 69
黫 62
黫 62
黬 0
黭 41
黮 55
黮闇 1
黯 90
黯淡 73
黯淡下來 1
黯淡無光 10
黯澹 0
黯然 92
黯然失色 8
黯然欲絕 0
黯然消魂 2
黯然淚下 0
黯然無光 1
黯然神傷 9
黯靄 0
黰 102
黱 0
黲 52
黳 2
黴 332
黴毒 0
黴菌 314
黵 7
黶 1
黷 8
黷武 11
黸 0
黹 68
黺 286
黻 64
黼 13
黽 23
黽勉 1
黾 0
黿 67
鼀 37
鼁 31
鼂 0
鼃 0
鼄 0
鼆 8
鼇 9
鼇頭 0
鼈 0
鼉 12
鼉鼓 0
鼊 7
鼎 3392
鼎力 84
鼎力相助 35
鼎峙 1
鼎業 8
鼎沸 15
鼎盛 161
鼎盛時期 2
鼎立 39
鼎足 3
鼎足之勢 1
鼎足而立 10
鼎鑊 2
鼎革 3
鼎鼎 13
鼎鼎大名 19
鼎鼎有名 10
鼏 44
鼐 133
鼑 0
鼒 87
鼓 2097
鼓兒詞 0
鼓兒詞 0
鼓出 2
鼓出來 0
鼓動 63
鼓動風潮 14
鼓勵 4808
鼓勵性 6
鼓台 0
鼓吹 166
鼓噪 4
鼓室 21
鼓成 9
鼓手 67
鼓掌 120
鼓掌聲 8
鼓搥 0
鼓棒 0
鼓椅 0
鼓槌 5
鼓樂 20
鼓樂喧天 0
鼓樓 13
鼓浪嶼 5
鼓滿 0
鼓王 4
鼓瑟 5
鼓皮 0
鼓盪 3
鼓篋 0
鼓翼 1
鼓聲 60
鼓聲隆隆 0
鼓脹 6
鼓腹 1
鼓膜 9
鼓膜炎 0
鼓舌 0
鼓舌如簧 0
鼓舌搖脣 0
鼓舞 242
鼓舞人心 4
鼓舞士氣 7
鼓著 5
鼓號 5
鼓號樂隊 6
鼓裡 2
鼓譟 16
鼓起 45
鼓起來 1
鼓起勇氣 71
鼓起如簧之舌 0
鼓足 6
鼓足勇氣 9
鼓足幹勁 0
鼓面 3
鼓頰 1
鼓風 4
鼓風機 30
鼓風爐 2
鼓鼓 12
鼓鼓囊囊 0
鼓鼓的 12
鼓鼓地 11
鼔 0
鼕 4
鼕鼕 1
鼖 24
鼗 0
鼘 155
鼙 5
鼙鼓 5
鼚 69
鼛 68
鼜 10
鼜 10
鼜 10
鼝 0
鼞 6
鼠 2638
鼠害 5
鼠尾草 20
鼠年 18
鼠疫 75
鼠目 0
鼠穴 0
鼠窟 0
鼠竄 2
鼠竊 1
鼠籠式 6
鼠肉 0
鼠膽 3
鼠藥 12
鼠蹊 16
鼠蹊部 18
鼠輩 9
鼢 46
鼣 105
鼤 43
鼥 14
鼦 0
鼧 0
鼨 29
鼨 29
鼩 20
鼪 9
鼫 46
鼬 18
鼬獾 0
鼬鼠 9
鼭 9
鼮 53
鼯 156
鼯鼠 19
鼰 38
鼰 38
鼱 31
鼲 36
鼳 82
鼳 82
鼴 251
鼴鼠 7
鼵 73
鼶 7
鼷 22
鼸 5
鼸 5
鼸 5
鼹 0
鼺 0
鼻 1521
鼻兒 1
鼻兒 1
鼻咽癌 258
鼻塞 119
鼻子 429
鼻子眼兒 0
鼻子眼兒 0
鼻子頭兒 0
鼻子頭兒 0
鼻孔 106
鼻尖 21
鼻息 59
鼻梁 7
鼻梁兒 0
鼻梁兒 0
鼻樑 34
鼻母音 1
鼻毛 1
鼻水 84
鼻涕 87
鼻準 5
鼻準兒 0
鼻準兒 0
鼻炎 372
鼻煙 6
鼻煙壺 24
鼻煙盒 1
鼻牛兒 0
鼻牛兒 0
鼻癌 4
鼻祖 33
鼻科 14
鼻竇 1
鼻竇炎 184
鼻翅 0
鼻背 1
鼻腔 147
鼻血 110
鼻識 1
鼻部 44
鼻酸 22
鼻鏡 2
鼻門 1
鼻青眼腫 2
鼻青臉腫 12
鼻音 12
鼻韻 0
鼻韻母 0
鼻頭 158
鼻黏膜炎 1
鼼 0
鼼 0
鼽 85
鼾 5
鼾息 1
鼾息聲 1
鼾睡 4
鼾聲 30
鼾聲如雷 2
鼾音 0
齀 23
齀 23
齁 13
齂 52
齂 52
齃 1
齃 1
齃 1
齃 1
齄 0
齅 0
齆 12
齇 4
齈 42
齉 64
齊 24
齊 24
齊 246
齊 66000
齊一 43
齊人 20
齊人之福 12
齊備 87
齊全 637
齊出 20
齊列 0
齊力 16
齊名 30
齊唱 15
齊國 59
齊國人 0
齊大非偶 0
齊奏 15
齊宣王 6
齊家 43
齊射 1
齊心 50
齊心協力 18
齊手 3
齊放 4
齊整 10
齊文宣帝 0
齊明 1
齊東 7
齊東野語 0
齊桓公 5
齊步 17
齊步走 17
齊王 31
齊發 21
齊發式 0
齊眉 13
齊眉棍 0
齊耳 3
齊聚 121
齊聚一堂 82
齊聲 66
齊聲歡唱 1
齊肩 1
齊行 2
齊衰 0
齊衰 0
齊進 3
齊集 24
齊頭 32
齊頭並進 21
齊頭式 20
齊首 0
齊鳴 10
齊齊哈爾 33
齊齊整整 0
齊齊的 8
齊齊地 8
齋 1308
齋主 31
齋供 1
齋僧 4
齋堂 28
齋壇 0
齋宿 1
齋心 0
齋戒 98
齋戒日 0
齋戒沐浴 2
齋日 2
齋舍 0
齋醮 7
齋食 2
齋飯 2
齌 25
齍 11
齍盛 1
齎 147
齎送 1
齏 5
齏粉 0
齐 0
齐 0
齐 0
齒 1178
齒位 2
齒冠 3
齒冷 0
齒印 2
齒垢 1
齒寒 1
齒根 4
齒條 14
齒狀 9
齒狀物 0
齒科 81
齒腔 6
齒舌 7
齒質 9
齒輪 489
齒輪箱 15
齒音 0
齒鯨 22
齒齦 23
齒齦炎 1
齓 0
齔 197
齕 79
齖 3
齗 5
齗齗 1
齘 7
齙 163
齙 163
齚 0
齚 0
齚 0
齛 39
齛 39
齜 5
齜牙 4
齜牙咧嘴 6
齝 44
齝 44
齞 87
齞脣 1
齟 285
齟齒 4
齟齬 6
齠 86
齠年 0
齡 2207
齡期 20
齡級 2
齣 562
齤 43
齥 64
齦 57
齦 57
齧 14
齧合 2
齧斷 0
齧齒類 6
齨 0
齩 0
齩 0
齪 103
齫 37
齫 37
齬 117
齭 0
齮 482
齯 209
齰 2021
齰舌 1
齱 598
齲 26
齲齒 87
齳 0
齴 7
齴 7
齴 7
齵 7
齵 7
齶 18
齷 4
齷齪 17
齸 8
齹 6
齺 14
齻 5
齼 0
齽 0
齾 148
龍 16354
龍心 100
龍心大悅 50
龍心不悅 50
龍井 601
龍井茶 4
龍兄 15
龍兄虎弟 18
龍卷 4
龍口 31
龍吟虎嘯 3
龍套 2
龍女 665
龍子 29
龍子龍女 0
龍宮 46
龍山 224
龍山區 8
龍山寺 183
龍山文化 6
龍岩 33
龍崎 117
龍巖 71
龍年 7
龍床 0
龍床快婿 0
龍捲風 251
龍樓鳳閣 1
龍江 75
龍江路 125
龍泉 197
龍涎 6
龍涎香 15
龍溪 52
龍潭 798
龍潭虎穴 2
龍濟光 1
龍煙鐵礦 0
龍燈 4
龍爭虎鬥 18
龍王 119
龍王爺 1
龍珠 206
龍生龍鳳生鳳 0
龍的傳人 45
龍眼 180
龍眼乾 8
龍種 3
龍穴 52
龍肝鳳髓 0
龍脈 10
龍腦 13
龍舌蘭 46
龍舟 52
龍舟競渡 5
龍舟競賽 9
龍舟賽 14
龍船 28
龍船競賽 0
龍船賽 0
龍華 360
龍華工專 8
龍虎 173
龍虎鬥 4
龍蛇 21
龍蛇混雜 5
龍蛇雜處 6
龍蝦 220
龍蝦肉 2
龍蟠 23
龍蟠虎踞 3
龍行虎步 0
龍袍 23
龍袍加身 0
龍銀 6
龍鍾 2
龍門 365
龍門架 6
龍門石窟 13
龍門路 25
龍頭 680
龍顏 3
龍顏不悅 0
龍顏大悅 1
龍飛 62
龍飛鳳舞 5
龍馬 31
龍駒 4
龍騰虎躍 13
龍骨 55
龍骨車 0
龍體 5
龍鬚 17
龍鬚溝 1
龍鬚茶 0
龍鬚草 0
龍鬚菜 27
龍鳳 177
龍鳳呈祥 5
龍鳳配 4
龍鳳餅 6
龎 0
龏 0
龐 466
龐大 1317
龐德 79
龐然 15
龐然大物 13
龐畢度 19
龐雜 59
龐鴻 1
龑 60
龒 69
龓 0
龓 0
龔 725
龔律全 1
龔自珍 6
龕 127
龖 0
龗 0
龘 71
龜 13
龜 138
龜 5241
龜兆 0
龜公 3
龜卜 1
龜山 172
龜山島 75
龜山鄉 1253
龜文 1
龜殼 22
龜殼花 38
龜毛 0
龜玉 0
龜王 5
龜甲 20
龜筮 0
龜紋 5
龜縮 1
龜肉 1
龜茲 10
龜茲 10
龜裂 119
龜速 1
龜鑑 1
龜頭 105
龜鱉 3
龝 0
龞 0
龠 101
龡 0
龡 0
龢 92
龣 0
龣 0
龤 859
龥 0
兀 19
嗀 3
𠅗 0
𠈊 0
𠷀 0
𠹌 0
𠽝 0
𡉒 0
𡠛 0
𡬙 0
𢯵 0
𣃋 0
𣷶 0
𤗿 0
𤽖 0
𥄲 0
𥩙 0
𦃢 0
𦬺 0
𧡣 0
𧨾 0
𧼐 0
𨀡 0
𨎤 0
𩸄 0
𪇜 0
㞞 0
客製化 1
客製化服務 1
權重 1
焊膏 1
醃製 1
