/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1996, 1997, 1998, 1999
 *	Sleepycat Software.  All rights reserved.
 *
 * $Id: ex_dbclient.c,v 1.1.1.1.2.2 2000/02/08 00:47:03 noriko Exp $
 */

#include "db_config.h"

#ifndef NO_SYSTEM_INCLUDES
#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#endif

#include <db.h>

#define	DATABASE_HOME	"database"
#define	CONFIG_DATA_DIR	NULL

#define	DATABASE	"access.db"

void db_setup(char *, char *[], FILE *, char *);
void db_teardown(char *, char *[], FILE *, char *);
int db_clientrun();

int  main(int, char **);

const char
	*progname = "ex_dbclient";			/* Program name. */

/*
 * An example of a program creating/configuring a Berkeley DB environment.
 */
int
main(argc, argv)
	int argc;
	char *argv[];
{
	char *config[2], *home;

	if (argc != 2) {
		fprintf(stderr, "Usage: %s hostname\n",argv[0]);
		exit(1);
	}
	/*
	 * All of the shared database files live in DATABASE_HOME, but
	 * data files will live in CONFIG_DATA_DIR.
	 */
	home = DATABASE_HOME;
	config[0] = CONFIG_DATA_DIR;
	config[1] = NULL;

	db_setup(home, config, stderr, argv[1]);

	db_teardown(home, config, stderr, argv[1]);

	return (0);
}

void
db_setup(home, config, errfp, host)
	char *home, *config[], *host;
	FILE *errfp;
{
	DB_ENV *dbenv;
	int ret, retry;

	/*
	 * Create an environment object and initialize it for error
	 * reporting.
	 */
	if ((ret = db_env_create(&dbenv, DB_CLIENT)) != 0) {
		fprintf(errfp, "%s: %s\n", progname, db_strerror(ret));
		exit (1);
	}
	retry = 0;
retry:
	while (retry < 5) {
		/*
		 * Set the server host we are talking to.
		 */
		if ((ret =
		    dbenv->set_server(dbenv, host, 10000, 10000, 0)) != 0) {
			fprintf(stderr, "Try %d: DBENV->set_server: %s\n",
			    retry, db_strerror(ret));
			retry++;
			sleep(15);
		} else
			break;
	}

	if (retry >= 5) {
		fprintf(stderr, "DBENV->set_server: %s\n", db_strerror(ret));
		dbenv->close(dbenv, 0);
		exit (1);
	}
	/*
	 * We want to specify the shared memory buffer pool cachesize,
	 * but everything else is the default.
	 */
	if ((ret = dbenv->set_cachesize(dbenv, 0, 64 * 1024, 0)) != 0) {
		dbenv->err(dbenv, ret, "set_cachesize");
		dbenv->close(dbenv, 0);
		exit (1);
	}
	/*
	 * We have multiple processes reading/writing these files, so
	 * we need concurrency control and a shared buffer pool, but
	 * not logging or transactions.
	 */
	if ((ret = dbenv->open(dbenv, home, config,
	    DB_CREATE | DB_INIT_LOCK | DB_INIT_MPOOL | DB_INIT_TXN, 0)) != 0) {
		dbenv->err(dbenv, ret, "environment open: %s", home);
		dbenv->close(dbenv, 0);
		if (ret == DB_NOSERVER)
			goto retry;
		exit (1);
	}

	ret = db_clientrun(dbenv);
	printf("db_clientrun returned %d\n", ret);
	if (ret == DB_NOSERVER)
		goto retry;

	/* Close the handle. */
	if ((ret = dbenv->close(dbenv, 0)) != 0) {
		fprintf(stderr, "DBENV->close: %s\n", db_strerror(ret));
		exit (1);
	}
}

void
db_teardown(home, config, errfp, host)
	char *home, *config[], *host;
	FILE *errfp;
{
	DB_ENV *dbenv;
	int ret;

	/* Remove the shared database regions. */
	if ((ret = db_env_create(&dbenv, DB_CLIENT)) != 0) {
		fprintf(errfp, "%s: %s\n", progname, db_strerror(ret));
		exit (1);
	}
	dbenv->set_server(dbenv, host, 0, 0, 0);

	if ((ret = dbenv->remove(dbenv, home, config, DB_FORCE)) != 0) {
		fprintf(stderr, "DBENV->remove: %s\n", db_strerror(ret));
		exit (1);
	}
}

int
db_clientrun(dbenv)
	DB_ENV *dbenv;
{
	DB *dbp;
	DBT key, data;
	u_int32_t len;
	int ret;
	char *p, *t, buf[1024], rbuf[1024];

	/* Remove the previous database. */

	/* Create and initialize database object, open the database. */
	if ((ret = db_create(&dbp, dbenv, 0)) != 0) {
		fprintf(stderr,
		    "%s: db_create: %s\n", progname, db_strerror(ret));
		return(ret);
	}
	if ((ret = dbp->set_pagesize(dbp, 1024)) != 0) {
		dbp->err(dbp, ret, "set_pagesize");
		goto err1;
	}
	if ((ret =
	    dbp->open(dbp, DATABASE, NULL, DB_BTREE, DB_CREATE, 0664)) != 0) {
		dbp->err(dbp, ret, "%s: open", DATABASE);
		goto err1;
	}

	/*
	 * Insert records into the database, where the key is the user
	 * input and the data is the user input in reverse order.
	 */
	memset(&key, 0, sizeof(DBT));
	memset(&data, 0, sizeof(DBT));
	for (;;) {
		printf("input> ");
		fflush(stdout);
		if (fgets(buf, sizeof(buf), stdin) == NULL)
			break;
		if ((len = strlen(buf)) <= 1)
			continue;
		for (t = rbuf, p = buf + (len - 2); p >= buf;)
			*t++ = *p--;
		*t++ = '\0';

		key.data = buf;
		data.data = rbuf;
		data.size = key.size = len - 1;

		switch (ret =
		    dbp->put(dbp, NULL, &key, &data, DB_NOOVERWRITE)) {
		case 0:
			break;
		default:
			dbp->err(dbp, ret, "DB->put");
			if (ret != DB_KEYEXIST)
				goto err1;
			break;
		}
		memset(&data, 0, sizeof(DBT));
		switch (ret = dbp->get(dbp, NULL, &key, &data, 0)) {
		case 0:
			printf("%.*s : %.*s\n",
			    (int)key.size, (char *)key.data,
			    (int)data.size, (char *)data.data);
			break;
		default:
			dbp->err(dbp, ret, "DB->get");
			break;
		}
	}
	if ((ret = dbp->close(dbp, 0)) != 0) {
		fprintf(stderr,
		    "%s: DB->close: %s\n", progname, db_strerror(ret));
		return (1);
	}
	return (0);

err1:	(void)dbp->close(dbp, 0);
	return (ret);
}
