=head1 NAME

Class::Mix - dynamic class mixing

=head1 SYNOPSIS

	use Class::Mix qw(mix_class);

	$foobar_object = mix_class("Foo", "Bar")->new;

	use Class::Mix qw(genpkg);

	$package = genpkg;
	$package = genpkg("Digest::Foo::");

=head1 DESCRIPTION

The C<mix_class> function provided by this module dynamically generates
`anonymous' classes with specified inheritance.

=cut

package Class::Mix;

use warnings;
use strict;

use Carp qw(croak);

our $VERSION = "0.003";

use parent "Exporter";
our @EXPORT_OK = qw(mix_class genpkg);

=head1 FUNCTIONS

=over

=item mix_class(CLASSES ...)

This function is used to dynamically generate `anonymous' classes by
mixing pre-existing classes.  This is useful where an incomplete class
requires use of a mixin in order to become instantiable, several suitable
mixins are available, and it is desired to make the choice between mixins
at runtime.

The function takes as its argument list the desired C<@ISA> list of
the mixture class to be created; that is, a list of names of classes
to inherit from.  It generates a class with the specified inheritance,
and returns its name.  The same class will be returned by repeated
invocations with the same class list.  The returned name may be used to
call a constructor or other class methods of the mixed class.

A class name must be returned because there is no such thing as an
anonymous class in Perl.  Classes are referenced by name.  The names
that are generated by this function are unique and insignificant.
See C<genpkg> below for more information.

If fewer than two classes to inherit from are specified, the function does
not bother to generate a new class.  If only one class is specified then
that class is returned.  If no classes are specified then "UNIVERSAL"
is returned.  This provides the desired inheritance without creating
superfluous classes.

This function relies on the classes it returns remaining unmodified in
order to be returned by future invocations.  If you want to modify your
dynamically-generated `anonymous' classes, use C<genpkg> (below).

=cut

sub genpkg(;$);

{
	my %mixtures;
	sub mix_class(@) {
		return "UNIVERSAL" if @_ == 0;
		return $_[0] if @_ == 1;
		my $recipe = join("", map { length($_)."_".$_ } @_);
		return $mixtures{$recipe} if exists $mixtures{$recipe};
		my $pkg = genpkg("Class::Mix::");
		$mixtures{$recipe} = $pkg;
		{
			no strict "refs";
			@{$pkg."::ISA"} = @_;
		}
		return $pkg;
	}
}

=item genpkg([PREFIX])

This function selects and returns a package name that has not been
previously used.  The name returned is an ordinary bareword-form package
name, and can be used as the second argument to C<bless> and in all
other ways that package names are used.  The package is initially empty.

The package names returned by this function are of a type that should not
be used as ordinary fixed module names.  However, it is not possible to
entirely prevent a clash.  This function checks that the package name it
is about to return has not already been used, and will avoid returning
such names, but it cannot guarantee that a later-loaded module will not
create a clash.

PREFIX, if present, specifies where the resulting package will go.
It must be either the empty string (to create a top-level package)
or a bareword followed by "::" (to create a package under that name).
For example, "Digest::" could be specified to ensure that the resulting
package has a name starting with "Digest::", so that C<< Digest->new >>
will accept it as the name of a message digest algorithm.

=cut

{
	my $n = 0;
	sub genpkg(;$) {
		my($prefix) = @_;
		$prefix = "" unless defined $prefix;
		croak "`$prefix' is not a valid module name prefix"
			unless $prefix =~ /\A(?:[a-zA-Z_][0-9a-zA-Z_]*::
					        (?:[0-9a-zA-Z_]+::)*)?\z/x;
		no strict "refs";
		my $pkgtail;
		do {
			$pkgtail = "__GP".$n++;
		} while(exists ${$prefix || "::"}{$pkgtail."::"});
		my $pkgname = $prefix.$pkgtail;
		%{$pkgname."::"} = ();
		return $pkgname;
	}
}

=back

=head1 SEE ALSO

L<Class::Generate>

=head1 AUTHOR

Andrew Main (Zefram) <zefram@fysh.org>

=head1 COPYRIGHT

Copyright (C) 2004, 2006, 2009 Andrew Main (Zefram) <zefram@fysh.org>

=head1 LICENSE

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
