
# {{{ pod

=head1 NAME

Archive::Any - Single interface to deal with zips and tarballs

=head1 SYNOPSIS

  use Archive::Any;

  my $archive = Archive::Any->new($archive_file);

  my @files = $archive->files;

  $archive->extract;

  my $type = $archive->type;

  $archive->is_impolite;
  $archive->is_naughty;

=head1 DESCRIPTION

This module is a single interface for manipulating different archive
formats.  Tarballs, zip files, etc...

Currently only tar (with or without gzip) and zip are supported.

Currently only supports unpacking.

=over 4

=item B<new>

  my $archive = Archive::Any->new($archive_file);
  my $archive = Archive::Any->new($archive_file, $type);

Creates an Archive::Any object representing $file, but don't do anything
with it yet.

$type is optional.  It lets you force the file type in-case
Archive::Any can't figure it out.  'tar' or 'zip' is currently
accepted.

=item B<files>

  my @file = $archive->files;

A list of files in the archive.

=item B<extract>

  $archive->extract;
  $archive->extract($director);

Extracts the files in the archive to the given $directory.  If no
$directory is given, it will go into the current working directory.

=item B<type>

  my $type = $archive->type;

Returns the type of archive this is.  Currently 'zip' or 'tar'.

=item B<is_impolite>

  my $is_impolite = $archive->is_impolite;

 Checks to see if this archive is going to unpack into the current
 directory rather than create its own.

=item B<is_naughty>

  my $is_naughty = $archive->is_naughty;

Checks to see if this archive is going to unpack B<outside> the
current directory.

=back

=head1 AUTHOR

Michael G Schwern E<lt>schwern@pobox.comE<gt>

=cut

# }}} pod

package Archive::Any;

use strict;
use vars qw($VERSION @ISA);

$VERSION = 0.06;

use File::Spec::Functions qw(rel2abs splitpath splitdir);
use File::Type;
use Module::Find;
use Carp::Always;

sub new {
    my ( $proto, $file, $type ) = @_;

    return undef unless -f $file;

    my $mime_type;
    unless ( defined $type ) {
        $mime_type = File::Type->new()->mime_type($file);
    }

    my @mods = findsubmod Archive::Any;
    my $supported_types;

    foreach my $mod (@mods) {
		print( "$mod\n" );
		eval{ require $mod };
		next if $@;
		my @tobj = $mod->can_handle();
		print Dumper( @tobj );
    }
    use Data::Dumper;
    print Dumper( $supported_types );
}

sub is_impolite {
    my ($self) = shift;

    my @files       = $self->files;
    my $first_file  = $files[0];
    my ($first_dir) = splitdir($first_file);

    return grep( !/^\Q$first_dir\E/, @files ) ? 1 : 0;
}

sub is_naughty {
    my ($self) = shift;

    return ( grep { m{^(?:/|(?:\./)*\.\./)} } $self->files ) ? 1 : 0;
}

1;
