#!/usr/bin/perl -I/home/phil/perl/cpan/DitaValidate/lib/  -I/home/phil/perl/cpan/DataDFA/lib/ -I/home/phil/perl/cpan/DataNFA/lib/ -I/home/phil/perl/cpan/DataTableText/lib/
# Edit data held in the XML format.
# Philip R Brenan at gmail dot com, Appa Apps Ltd Inc, 2016-2019
#-------------------------------------------------------------------------------
# podDocumentation
# cd /home/phil/perl/cpan/DataEditXml/; perl Build.PL && perl Build test && sudo perl Build install
# perl -d:NYTProf -Ilib test.pl && nytprofhtml --open
# $source =~ s(\x{a0}) ( )gs; The no break space problem
# Do not line number if xtrf is already taken!
# Pretty print with line numbers so we can get out of adding line numbers to existing xml which is currently not reliable
# When deleting content should putFirstCut check that the cut node is not above the target node ?.
# add*asTree
# change wrap and other commands that use %attributes rather than @context to use @context
# swap c and cc
# root finds the root node possibly obviating the need for the parser attribute?
# Stop unwrap from confessing to being on the root node
# Parse DTDs so that we can separate a classification map from a conventional map
# #b means the method acts on one node and returns a string - its braceable

package Data::Edit::Xml;
our $VERSION = 20201031;
use v5.26;
use warnings FATAL => qw(all);
use strict;
use Carp qw(confess cluck);
use Data::Dump qw(dump);
use Data::Table::Text qw(:all);
#se Dita::Validate;
use XML::Parser;                                                                # https://metacpan.org/pod/XML::Parser
use Storable qw(dclone freeze retrieve store thaw);
use utf8;

#D1 Construction                                                                # Create a parse tree, either by parsing a L<file or string|/file or string> B<or> L<node by node|/Node by Node> B<or> from another L<parse tree|/Parse tree>.

#D2 File or String                                                              # Construct a parse tree from a file or a string.

our $lastParseError;                                                            # The last parse error encountered

sub new(;$@)                                                                    #IS Create a new parse tree - call this method statically as in Data::Edit::Xml::new(file or string) to parse a file or string B<or> with no parameters and then use L</input>, L</inputFile>, L</inputString>, L</errorFile>  to provide specific parameters for the parse, then call L</parse> to perform the parse and return the parse tree.
 {my ($fileNameOrString, @options) = @_;                                        # Optional file name or string from which to construct the parse tree, hash of other options.
  shift @_ while @_ && ref($_[0]);                                              # Remove any leading references to find the actual string or file to be parsed
  my ($source, %options) = @_;                                                  # Assign parameters - the one higher up is for documentation only
  checkKeys(\%options,                                                          # Check report options
   {lineNumbers => <<'END',
Save the line number.column number at which tag starts and ends on
the xtrf attribute of each node if true.
END
    input       => <<'END',
Specifies the string or file name containing the source xml to be parsed if true
END
    inputString => <<'END',
Specifies the string of xml to be parsed if true
END
    inputFile   => <<'END',
Specifies the name of the input file containing the source xml to be parsed if true.
END
    errorsFile  => <<'END',
Specifies the file to which any parse errors will be written to. By default this file is named: B<zzzParseErrors/out.data>.
END
    });
  if (@_)
   {my $x = bless {input=>@_};                                                  # Create L<XML> editor with a string or file
    $x->parser = $x;                                                            # Parser root node
    return $x->parse;                                                           # Parse
   }
  my $x = bless {};                                                             # Create empty L<XML> editor
  $x->parser = $x;                                                              # Parser root node
  $x                                                                            # Parser
 }

sub cdata()                                                                     # The name of the tag to be used to represent text - this tag must not also be used as a command tag otherwise the parser will L<confess>.
 {'CDATA'
 }

sub reduceParseErroMessage($)                                                   #P Reduce the parse failure message to the bare essentials.
 {my ($e) = @_;                                                                 # Error message
  if ($e =~ m((not well-formed.*?byte\s*\d+))is) {return $1}
  $e                                                                            # Return full  message of it cannot be further reduced
 }

sub parse($)                                                                    # Parse input L<XML> specified via: L<inputFile|/inputFile>, L<input|/input> or L<inputString|/inputString>.
 {my ($parser) = @_;                                                            # Parser created by L</new>

  if (my $s = $parser->input)                                                   # Source to be parsed is a file or a string
   {if ($s =~ /\n/s or !-e $s)                                                  # Parse as a string because it does not look like a file name
     {$parser->inputString = $s;
     }
    else                                                                        # Parse a file
     {$parser->inputFile = $s;
      $parser->inputString = readFile($s);
     }
   }
  elsif (my $f = $parser->inputFile)                                            # Source to be parsed is a file
   {$parser->inputString = readFile($f);
   }
  elsif ($parser->inputString) {}                                               # Source to be parsed is a string
  else                                                                          # Unknown string
   {confess "Supply a string or file to be parsed";
   }

  my $xmlParser = new XML::Parser(Style => 'Tree');                             # Extend Larry Wall's excellent L<XML> parser
  my $D = $parser->inputString;                                                 # String to be parsed
  my $d = $parser->lineNumbers ? &addLineNumbers($D) : $D;                      # Add line numbers if requested
     $d =~ s(&(?!(amp|gt|lt|quot|#x?\d+);)) (&amp;)gs;                          # Correct free floating ampersands

  my $x = eval {$xmlParser->parse($d)};                                         # Parse string
  $lastParseError = reduceParseErroMessage($@);                                 # Record any error

  if (!$x)                                                                      # Error in parse: write a message to STDERR and to a file if possible
   {my $f = $parser->inputFile ? "Source file is:\n".                           # Source details if a file
            $parser->inputFile."\n" : '';
    my $s = $lastParseError;
    eval {confess "Parse failed at:\n"};                                        # Stack of parse failure
    my $e = "$d\n$f\n$@\n$s\n";                                                 # Error
    # warn "Xml parse error: $e";                                               # Write a description of the error to STDERR before attempting to write to a file
    my $badFile  = $parser->errorsFile ||                                       # File name to write error analysis to
                   fullFileName(filePathExt(qw(zzzParseErrors out data)));
    unlink $badFile if -e $badFile;                                             # Remove existing errors file
    writeFile($badFile, $e);                                                    # Write a description of the error to the errorsFile
    confess "Xml parse error, see file:\n$badFile\n$s\n";                       # Complain helpfully if parse failed
   }

  $parser->tree($x);                                                            # Structure parse results as a tree

  if (my @c = @{$parser->content})                                              # Set additional attributes on root node
   {confess "No XML" if !@c;
    confess "More than one outer-most tag" if @c > 1;
    my $c = $c[0];
    $parser->tag        = $c->tag;
    $parser->attributes = $c->attributes;
    $parser->content    = $c->content;
    $parser->parent     = undef;
    $parser->indexNode;
   }

  $parser                                                                       # Parse details
 }

sub tree($$)                                                                    #P Build a tree representation of the parsed L<XML> which can be easily traversed to look for things.
 {my ($parent, $parse) = @_;                                                    # The parent node, the remaining parse
  while(@$parse)
   {my $tag  = shift @$parse;                                                   # Tag for node
    my $node = bless {parser=>$parent->parser};                                 # New node
    if ($tag eq cdata)
     {confess cdata.' tag encountered';                                         # We use this tag for text and so it cannot be used as a user tag in the document
     }
    elsif ($tag eq '0')                                                         # Text
     {my $s = shift @$parse;
      if ($s !~ /\A\s*\Z/)                                                      # Ignore entirely blank strings
       {$s = replaceSpecialChars($s);                                           # Restore special characters in the text
        $node->tag  = cdata;                                                    # Save text. ASSUMPTION: CDATA is not used as a tag anywhere.
        $node->text = $s;
        push @{$parent->content}, $node;                                        # Save on parents content list
       }
     }
    else                                                                        # Node
     {my $children   = shift @$parse;
      my $attributes = shift @$children;
      $node->tag = $tag;                                                        # Save tag
      $_ = replaceSpecialChars($_) for values %$attributes;                     # Restore in text with L<XML> special characters
      $node->attributes = $attributes;                                          # Save attributes
      push @{$parent->content}, $node;                                          # Save on parents content list
      $node->tree($children) if $children;                                      # Add nodes below this node
     }
   }
  $parent->indexNode;                                                           # Index this node
 }

sub addLineNumbers($)                                                           #P Add line numbers to the source
 {my ($string) = @_;                                                            # Source string
  my @s = split //, ($string =~ s(<!--linted:.*\Z) ()gsr);                      # Remove any trailing lint comments
  my $state = 0;                                                                # 0 - outside any tag, 1 - inside a tag
  my $Line  = 1;                                                                # Current tag start position
  my $Col   = 0;
  my $line  = 1;                                                                # Current position
  my $col   = 0;
  for my $i(keys @s)                                                            # Each input character
   {my $c = $s[$i];                                                             # current character
    ++$col;
    if ($c eq qq(\n))                                                           # Line/column number
     {++$line;
      $col = 0;
     }
    elsif ($state == 0)                                                         # Looking for the start of a tag
     {if ($c eq q(<) and ($s[$i+1]//'') =~ m(\A[^/?!]\Z)s)                      # Exclude non tags
       {($Line, $Col) = ($line, $col);                                          # Record start position
        $state = 1;                                                             # Looking for end
       }
     }
    elsif ($c eq q(>))                                                          # At end of tag
     {if ($state == 1)                                                          # Inside a tag
       {my $m = join '', ' xtrf="', $Line,  '.', $Col, ':',                     # Line number uses an increment
            ($line == $Line ? q() : ($line-$Line).q(.)), $col, '"';
        if (($s[$i-1]//'') ne q(/))                                             # Tag with content
         {$s[$i]   = $m.'>';
         }
        else                                                                    # Tag with no content
         {$s[$i-1] = $m.'/';
         }
        $state = 0;                                                             # Looking for the next tag
       }
     }
   }
  join '', @s                                                                   # New source string
 }

#D2 Node by Node                                                                # Construct a parse tree node by node.

sub newText($$)                                                                 # Create a new text node.
 {my (undef, $text) = @_;                                                       # Any reference to this package, content of new text node
  my $node = bless {};                                                          # New node
  $node->parser = $node;                                                        # Root node of this parse
  $node->tag    = cdata;                                                        # Text node
  $node->text   = $text;                                                        # Content of node
  $node                                                                         # Return new non text node
 }

sub newTag($$%)                                                                 # Create a new non text node.
 {my (undef, $command, %attributes) = @_;                                       # Any reference to this package, the tag for the node, attributes as a hash.
  my $node = bless {};                                                          # New node
  $node->parser = $node;                                                        # Root node of this parse
  $node->tag    = $command;                                                     # Tag for node
  $node->attributes = \%attributes;                                             # Attributes for node
  $node                                                                         # Return new node
 }

sub newTree($%)                                                                 # Create a new tree.
 {my ($command, %attributes) = @_;                                              # The name of the root node in the tree, attributes of the root node in the tree as a hash.
  &newTag(undef, @_)
 }

sub dupTag($@)                                                                  #C Create a new non text node by duplicating the tag of an existing node.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $new = bless {};                                                           # New node
  $new->parser = $node->parser;                                                 # Root node of this parse
  $new->tag    = $node->tag;                                                    # Tag for node
  $new                                                                          # Return new node
 }

sub disconnectLeafNode($)                                                       #P Remove a leaf node from the parse tree and make it into its own parse tree.
 {my ($node) = @_;                                                              # Leaf node to disconnect.
  $node->parent = undef;                                                        # No parent
  $node->parser = $node;                                                        # Own parse tree
 }

sub reindexNode($)                                                              #P Index the children of a node so that we can access them by tag and number.
 {my ($node) = @_;                                                              # Node to index.
  delete $node->{indexes};                                                      # Delete the indexes
  for my $n($node->contents)                                                    # Index content
   {push @{$node->indexes->{$n->tag}}, $n;                                      # Indices to sub nodes
   }
 }

sub indexNode($)                                                                #P Merge multiple text segments and set parent and parser after changes to a node
 {my ($node) = @_;                                                              # Node to index.
  return unless keys @{$node->{content}};
  my @contents = @{$node->content};                                             # Contents of the node

#  eval {grep {$_->{tag} eq cdata} @contents};
#  $@ and confess "$@\n";

  if ((grep {$_->{tag} eq cdata} @contents) > 1)                                # Make parsing easier for the user by concatenating successive text nodes - NB: this statement has been optimized
   {my (@c, @t);                                                                # New content, pending intermediate texts list
    for(@contents)                                                              # Each node under the current node
     {if ($_->{tag} eq cdata)                                                   # Text node. NB: optimized
       {push @t, $_;                                                            # Add the text node to pending intermediate texts list
       }
      elsif (@t == 1)                                                           # Non text element encountered with one pending intermediate text
       {push @c, @t, $_;                                                        # Save the text node and the latest non text node
        @t = ();                                                                # Empty pending intermediate texts list
       }
      elsif (@t  > 1)                                                           # Non text element encountered with two or more pending intermediate texts
       {my $t = shift @t;                                                       # Reuse the first text node
        $t->text .= join '', map {$_->text} @t;                                 # Concatenate the remaining text nodes
        $_->disconnectLeafNode for @t;                                          # Disconnect the remain text nodes as they are no longer needed
        push @c, $t, $_;                                                        # Save the resulting text node and the latest non text node
        @t = ();                                                                # Empty pending intermediate texts list
       }
      else {push @c, $_}                                                        # Non text node encountered without immediately preceding text
     }

    if    (@t == 0) {}                                                          # No action required if no pending text at the end
    elsif (@t == 1) {push @c, @t}                                               # Just one text node
    else                                                                        # More than one text node - remove leading and trailing blank text nodes
     {my $t = shift @t;                                                         # Reuse the first text node
      $t->text .= join '', map {$_->text} @t;                                   # Concatenate the remaining text nodes
      $_->disconnectLeafNode for @t;                                            # Disconnect the remain text nodes as they are no longer needed
      push @c, $t;                                                              # Save resulting text element
     }

    @contents      =  @c;                                                       # The latest content of the node
    $node->content = \@c;                                                       # Node contents with concatenated text elements
   }

  my $parser = $node->parser;                                                   # Parser for this node
  for(@contents)                                                                # Index content
   {$_->{parent} = $node;                                                       # Point to parent
    $_->{parser} = $parser;                                                     # Point to parser
   }
 }

sub replaceSpecialChars($)                                                      #S Replace < > " & with &lt; &gt; &quot; &amp; Larry Wall's excellent L<Xml parser> unfortunately replaces &lt; &gt; &quot; &amp; etc. with their expansions in text by default and does not seem to provide an obvious way to stop this behavior, so we have to put them back again using this method.
 {my ($string) = @_;                                                            # String to be edited.
  $string =~ s/\&/&amp;/g;                                                      # At this point all & that prefix variables should have been expanded, so any that are left are are real &s which should be replaced with &amp;
  $string =~ s/\</&lt;/gr =~ s/\>/&gt;/gr =~ s/\"/&quot;/gr                     # Replace the special characters that we can replace.
 }

sub undoSpecialChars($)                                                         #S Reverse the results of calling L<replaceSpecialChars|/replaceSpecialChars>.
 {my ($string) = @_;                                                            # String to be edited.
  $string =~ s/&amp;/\&/g;                                                      # At this point all & that prefix variables should have been expanded, so any that are left are are real &s which should be replaced with &amp;
  $string =~ s/&lt;/\</gr =~ s/&gt;/\>/gr =~ s/&quot;/\"/gr                     # Replace the special characters that we can replace.
 }

#D2 Parse tree attributes                                                       # Attributes of a node in a parse tree. For instance the attributes associated with an L<XML> tag are held in the L<attributes|/attributes> attribute. It should not be necessary to use these attributes directly unless you are writing an extension to this module.  Otherwise you should probably use the methods documented in other sections to manipulate the parse tree as they offer a safer interface at a higher level.

genHash(__PACKAGE__,                                                            # L<XML> parser definition
  content=>[],                                                                  # Content of command: the nodes immediately below the specified B<$node> in the order in which they appeared in the source text, see also L</Contents>.
  numbers=>[],                                                                  # Nodes by number.
  data=>{},                                                                     # A hash added to the node for use by the programmer during transformations. The data in this hash will not be printed by any of the L<printed|/Print> methods and so can be used to add data to the L<parse|/parse> tree that will not be seen in any output L<XML> produced from the L<parse|/parse> tree.
  attributes=>{},                                                               # The attributes of the specified B<$node>, see also: L</Attributes>.  The frequently used attributes: class, id, href, outputclass can be accessed by an L<lvalueMethod> method as in: $node->id = 'c1'.
  conditions=>{},                                                               # Conditional strings attached to a node, see L</Conditions>.
  forestNumbers=>{},                                                            # Index to node by forest number as set by L<numberForest|/numberForest>.
  indexes=>{},                                                                  # Indexes to sub commands by tag in the order in which they appeared in the source text.
  labels=>{},                                                                   # The labels attached to a node to provide addressability from other nodes, see: L</Labels>.
  depthProfileLast=>undef,                                                      # The last known depth profile for this node as set by L<setDepthProfiles|/setDepthProfiles>.
  errorsFile=>undef,                                                            # Error listing file. Use this parameter to explicitly set the name of the file that will be used to write any L<parse|/parse> errors to. By default this file is named: B<zzzParseErrors/out.data>.
  inputFile=>undef,                                                             # Source file of the L<parse|/parse> if this is the L<parser|/parse> root node. Use this parameter to explicitly set the file to be L<parsed|/parse>.
  input=>undef,                                                                 # Source of the L<parse|/parse> if this is the L<parser|/parse> root node. Use this parameter to specify some input either as a string or as a file name for the L<parser|/parse> to convert into a L<parse|/parse> tree.
  inputString=>undef,                                                           # Source string of the L<parse|/parse> if this is the L<parser|/parse> root node. Use this parameter to explicitly set the string to be L<parsed|/parse>.
  lineNumbers=>undef,                                                           # If true then save the line number.column number at which tag starts and ends on the xtrf attribute of each node.
  numbering=>undef,                                                             # Last number used to number a node in this L<parse|/parse> tree.
  number=>undef,                                                                # Number of the specified B<$node>, see L<findByNumber|/findByNumber>.
  parent=>undef,                                                                # Parent node of the specified B<$node> or B<undef> if the L<parser|/parse> root node. See also L</Traversal> and L</Navigation>. Consider as read only.
  parser=>undef,                                                                # L<Parser|/parse> details: the root node of a tree is the L<parser|/parse> node for that tree. Consider as read only.
  representationLast=>undef,                                                    # The last representation set for this node by one of: L<setRepresentationAsTagsAndText|/setRepresentationAsTagsAndText>.
  tag=>undef,                                                                   # Tag name for the specified B<$node>, see also L</Traversal> and L</Navigation>. Consider as read only.
  text=>undef,                                                                  # Text of the specified B<$node> but only if it is a text node otherwise B<undef>, i.e. the tag is cdata() <=> L</isText> is true.
 );

#D2 Parse tree                                                                  # Construct a L<parse|/parse> tree from another L<parse|/parse> tree.

sub renew($@)                                                                   #C Returns a renewed copy of the L<parse|/parse> tree by first printing it and then re-parsing it, optionally checking that the starting node is in a specified context: use this method if you have added nodes via the L</"Put as text"> methods and wish to traverse their L<parse|/parse> tree.\mReturns the starting node of the new L<parse|/parse> tree or B<undef> if the optional context constraint was supplied but not satisfied.
 {my ($node, @context) = @_;                                                    # Node to renew from, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $x = new($node->string);                                                   # Reconstruct parse tree from node
  $x->inputFile = $node->root->inputFile;                                       # Convey the input file name if present so that relative references can be resolved in the new parse tree
  $x                                                                            # Return new parse tree
 }

sub clone($)                                                                    #C Return a clone of the entire L<parse|/parse> tree which is created using the fast L<Storable::dclone> method. The L<parse|/parse> tree is cloned without converting it to string and re-parsing it so this method will not L<renew|/renew> any nodes added L<as text|/Put as text>.\mReturns the starting node of the new L<parse|/parse> tree.
 {my ($tree) = @_;                                                              # Parse tree
  my $new = dclone($tree);
# $new->parent = undef;
  $new->parser = $new;
  $new->cut;
 }

sub equals($$)                                                                  #Y Return the first node if the two L<parse|/parse> trees have identical representations via L<string|/string>, else B<undef>.
 {my ($node1, $node2) = @_;                                                     # Parse tree 1, parse tree 2.
  $node1->string eq $node2->string ? $node1 : undef                             # Test
 }

sub equalsIgnoringAttributes($$@)                                               # Return the first node if the two L<parse|/parse> trees have identical representations via L<string|/string> if the specified attributes are ignored, else B<undef>.
 {my ($node1, $node2, @attributes) = @_;                                        # Parse tree 1, parse tree 2, attributes to ignore during comparison
  my $p = $node1->clone;                                                        # Clone the parse trees so we can modify them
  my $q = $node2->clone;
  for my $x($p, $q)                                                             # Remove specified attributes from clones of parse trees
   {$x->by(sub
     {$_->deleteAttrs(@attributes);
     })
   }
  $p->string eq $q->string ? $node1 : undef                                     # Compare the reduced parse trees
 }

sub normalizeWhiteSpace($)                                                      #PS Normalize white space, remove comments DOCTYPE and L<XML> processors from a string
 {my ($string) = @_;                                                            # String to normalize
  $string =~ s(<\?.*?\?>)     ( )gs;                                            # Processors
  $string =~ s(<!--.*?-->)    ( )gs;                                            # Comments
  $string =~ s(<!DOCTYPE.+?>) ( )gs;                                            # Doctype
  $string =~ s(\s+)           ( )gs;                                            # White space
  $string
 }

sub diff($$;$)                                                                  # Return () if the dense string representations of the two nodes are equal, else up to the first N (default 16) characters of the common prefix before the point of divergence and the remainder of the string representation of each node from the point of divergence. All <!-- ... --> comments are ignored during this comparison and all spans of white space are reduced to a single blank.
 {my ($first, $second, $N) = @_;                                                # First node, second node, maximum length of difference strings to return
  $first = new($first)   unless ref $first;                                     # Auto vivify the first node if necessary
  $second = new($second) unless ref $second;                                    # Auto vivify the second node if necessary
  $N //= 16;
  my $a = normalizeWhiteSpace(-s $first);                                       # Convert to normalized strings
  my $b = normalizeWhiteSpace(-s $second);
  return () if length($a) == length($b) and $a eq $b;                           # Equal strings

  my @a = split //, $a;                                                         # Split into characters
  my @b = split //, $b;
  my @c;                                                                        # Common prefix
  while(@a and @b and $a[0] eq $b[0])                                           # Remove equal prefix characters
   {push @c, shift @a; shift @b;                                                # Save common prefix
   }

  $#a = $N-1 if $N and @a > $N;                                                 # Truncate remainder if necessary
  $#b = $N-1 if $N and @b > $N;
  if ($N) {shift @c while @c > $N}

 (join ('', @c), join('', @a), join('', @b))                                    # Return common prefix and diverging strings
 }

sub save($$)                                                                    # Save a copy of the L<parse|/parse> tree to a file which can be L<restored|/restore> and return the saved node.  This method uses L<Storable> which is fast but produces large files that do not compress well.  Use L<writeCompressedFile|/writeCompressedFile> to produce smaller save files at the cost of more time.
 {my ($node, $file) = @_;                                                       # Parse tree, file.
  makePath($file);
  store $node, $file;
  $node
 }

sub restore($)                                                                  #SY Return a L<parse|/parse> tree from a copy saved in a file by L<save|/save>.
 {my ($file) = @_;                                                              # File
  -e $file or confess "Cannot restore from a non existent file:\n$file";
  retrieve $file
 }

sub expandIncludes($)                                                           #U Expand the includes mentioned in a L<parse|/parse> tree: any tag that ends in B<include> is assumed to be an include directive.  The file to be included is named on the B<href> keyword.  If the file to be included is a relative file name, i.e. it does not begin with B</> then this file is made absolute relative to the file from which this L<parse|/parse> tree was obtained.
 {my ($x) = @_;                                                                 # Parse tree
  $x->by(sub                                                                    # Look for include statements
   {my ($o) = @_;
    if ($o->at(qr(include\Z)))                                                  # Include statement
     {my $href = $o->attr(q(href));                                             # Remove dots and slashes from front of file name
      my $in   = $x->inputFile;
      my $file = absFromAbsPlusRel($in, $href);
      my $i = eval{Data::Edit::Xml::new($file)};                                # Parse the new source file

      if ($@)                                                                   # Report any errors encountered in expansion
       {confess "Failed to expand href $href in file $in\n$@";
       }

      $i->expandIncludes;                                                       # Rescan for any internal includes
      $o->replaceWith($i);                                                      # Replace include statement
     }
   });
 }

#D1 Print                                                                       # Create a string representation of the L<parse|/parse> tree with optional selection of nodes via L<conditions|/Conditions>.\mNormally use the methods in L<Pretty|/Pretty> to format the L<XML> in a readable yet reparseable manner; use L<Dense|/Dense> string to format the L<XML> densely in a reparseable manner; use the other methods to produce unreparseable strings conveniently formatted to assist various specialized operations such as debugging CDATA, using labels or creating tests. A number of the L<file test operators|/opString> can also be conveniently used to print L<parse|/parse> trees in these formats.

#D2 Pretty                                                                      # Pretty print the L<parse|/parse> tree.

sub prettyString($;$)                                                           #IU Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-p|/opString> $node
 {my ($node, $depth) = @_;                                                      # Start node, optional depth.
  $depth //= 0;                                                                 # Start depth if none supplied

# return $node->text.($node->isLast ? q() : qq(\n)) if $node->isText;           # Add a new line after contiguous blocks of text to offset next node

  if ($node->isText)                                                            # Text block
   {my $t = $node->text =~ s( *\n) (\n)gsr;                                     # Collapse spaces before new line
    return $t if $node->isLast;                                                 # The following tag will deal with the new line
    return $t.qq(\n) unless $t =~ m(\n\Z)s;                                     # Separate from next tag with a new line
    return $t;                                                                  # Text already has a new line and so no additional separator required
   }

  my $t = $node->tag;                                                           # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes
  my $space   = "  "x($depth//0);
  return $space.'<'.$t.$node->printAttributes.'/>'."\n" if !@$content;          # No sub nodes

  my $s = $space.'<'.$t.$node->printAttributes.'>'.                             # Has sub nodes
    ($node->first->isText ? '' : "\n");                                         # Continue text on the same line, otherwise place nodes on following lines
  $s .= $_->prettyString($depth+1) for @$content;                               # Recurse to get the sub content
  $s .= $node->last->isText ? ((grep{!$_->isText} @$content)                    # Continue text on the same line, otherwise place nodes on following lines
                            ? "\n$space": "") : $space;
  my $r = $s .  '</'.$t.'>'."\n";                                               # Closing tag
  return $r if $depth;                                                          # Return from sub tree
  $r =~ s(>\n( *[.,;:\)] *)) (>$1\n)gsr                                         # Overall result moves some punctuation through one new line to be closer to its tag
     =~ s(\n\s*\n) (\n)gsr                                                      # Remove blank lines
 }

sub prettyStringHtml2($$)                                                       #P Return a string of html representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-p|/opString> $node
 {my ($node, $depth) = @_;                                                      # Start node, optional depth.
  $depth //= 0;                                                                 # Start depth if none supplied

  if ($node->isText)                                                            # Text block
   {my $t =                                                                     # Wrap text in span on one line
     qq(<span class="xmlText">)
     .nws($node->text)
     .qq(</span>);
    return $t;                                                                  # Text already has a new line and so no additional separator required
   }

  my $t       = qq(<span class="xmlTag">).$node->tag.q(</span>);                # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes

  my $space   = qq(<span class="xmlLineStartTag">)                              # Space before text
   .("&nbsp;"x(4*($depth//0)))
   .qq(</span>);

  return $space                                                                 # No sub nodes
   .q(<span class="xmlLt">&lt;</span>)
   .$t
   .$node->printAttributesHtml
   .q(<span class="xmlSlashGt">/&gt;</span>)
   ."\n" if !@$content;

  my $s = $space                                                                # Has sub nodes
   .q(<span class="xmlLt">&lt;</span>)
   .$t
   .$node->printAttributesHtml
   .q(<span class="xmlGt">&gt;</span>)
   .($node->first->isText ? '' : "\n");                                         # Continue text on the same line, otherwise place nodes on following lines

  $s .= $_->prettyStringHtml2($depth+1) for @$content;                          # Recurse to get the sub content

  $s .= $node->last->isText ? ((grep{!$_->isText} @$content)                    # Continue text on the same line, otherwise place nodes on following lines
                            ? "\n$space": "") : $space;

  my $r = $s                                                                    # Closing tag
   .q(<span class="xmlLtSlash">&lt;/</span>)
   .$t
   .q(<span class="xmlGt">&gt;</span>)
   ."\n";

  return $r if $depth;                                                          # Return from sub tree

  my $h = join "\n", map {qq(<div class="xmlLine">$_</div>)} split m/\n/, $r;   # Wrap div around each line
  qq($h\n)
 }

sub prettyStringHtml($@)                                                        # Return a string of L<html> representing a node of a L<parse|/parse> tree and all the nodes below it if the node is in the specified context.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Check optional context
  prettyStringHtml2($node, 0);                                                  # Print as html
 }

sub prettyStringDitaHeaders($)                                                  #U Return a readable string representing the L<parse|/parse> tree below the specified B<$node> with appropriate headers. Or use L<-x|/opString> $node
 {my ($node) = @_;                                                              # Start node
# cluck "Please use: ditaPrettyPrintWithHeaders...redirecting";
  $node->ditaPrettyPrintWithHeaders
 }

sub prettyStringNumbered($;$)                                                   #U Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it with a L<number|/number> attached to each tag. The node numbers can then be used as described in L<Order|/Order> to monitor changes to the L<parse|/parse> tree.
 {my ($node, $depth) = @_;                                                      # Start node, optional depth.
  $depth //= 0;                                                                 # Start depth if none supplied

  my $N = $node->number;                                                        # Node number if present

  if ($node->isText)                                                            # Text node
   {my $n = $node->next;
    my $s = !defined($n) || $n->isText ? '' : "\n";                             # Add a new line after contiguous blocks of text to offset next node
    return ($N ? "($N)" : '').$node->text.$s;                                   # Number text
   }

  my $t = $node->tag;                                                           # Number tag in a way which allows us to skip between start and end tags in L<Geany|http://www.geany.org> using the ctrl+up and ctrl+down arrows
  my $i = $N && !defined($node->id) ? " id=\"$N\""  : '';                       # Use id to hold tag
  my $content = $node->content;                                                 # Sub nodes
  my $space   = "  "x($depth//0);
  return $space.'<'.$t.$i.$node->printAttributes.'/>'."\n" if !@$content;       # No sub nodes

  my $s = $space.'<'.$t.$i.$node->printAttributes.'>'.                          # Has sub nodes
    ($node->first->isText ? '' : "\n");                                         # Continue text on the same line, otherwise place nodes on following lines
  $s .= $_->prettyStringNumbered($depth+1) for @$content;                       # Recurse to get the sub content
  $s .= $node->last->isText ? ((grep{!$_->isText} @$content)                    # Continue text on the same line, otherwise place nodes on following lines
                            ? "\n$space": "") : $space;
  my $r = $s .  '</'.$t.'>'."\n";                                               # Closing tag
  return $r if $depth;                                                          # Return from sub tree
  $r =~ s(>\n( *[.,;:\)] *)) (>$1\n)gsr                                         # Overall result moves some punctuation through one new line to be closer to its tag
 }

sub prettyStringCDATA($;$)                                                      #U Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it with the text fields wrapped with <CDATA>...</CDATA>.
 {my ($node, $depth) = @_;                                                      # Start node, optional depth.
  $depth //= 0;                                                                 # Start depth if none supplied

  if ($node->isText)                                                            # Text node
   {my $n = $node->next;
    my $s = !defined($n) || $n->isText ? '' : "\n";                             # Add a new line after contiguous blocks of text to offset next node
    return '<'.cdata.'>'.$node->text.'</'.cdata.'>'.$s;
   }

  my $t = $node->tag;                                                           # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes
  my $space   = "  "x($depth//0);
  return $space.'<'.$t.$node->printAttributes.'/>'."\n" if !@$content;          # No sub nodes

  my $s = $space.'<'.$t.$node->printAttributes.'>'.                             # Has sub nodes
    ($node->first->isText ? '' : "\n");                                         # Continue text on the same line, otherwise place nodes on following lines
  $s .= $_->prettyStringCDATA($depth+2) for @$content;                          # Recurse to get the sub content
  $s .= $node->last->isText ? ((grep{!$_->isText} @$content)                    # Continue text on the same line, otherwise place nodes on following lines
                            ? "\n$space": "") : $space;
  my $r = $s .  '</'.$t.'>'."\n";                                               # Closing tag
  return $r if $depth;                                                          # Return from sub tree
  $r =~ s(>\n( *[.,;:\)] *)) (>$1\n)gsr                                         # Overall result moves some punctuation through one new line to be closer to its tag
 }

sub prettyStringEnd($)                                                          #PU Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it as a here document
 {my ($node) = @_;                                                              # Start node
  my $s = -p $node;                                                             # Pretty string representation
'  ok -p $x eq <<END;'. "\n".(-p $node). "\nEND"                                # Here document
 }

sub prettyStringContent($)                                                      #U Return a readable string representing all the nodes below a node of a L<parse|/parse> tree.
 {my ($node) = @_;                                                              # Start node.
  my $s = '';
  $s .= $_->prettyString for $node->contents;                                   # Recurse to get the sub content
  $s
 }

sub prettyStringContentNumbered($)                                              #U Return a readable string representing all the nodes below a node of a L<parse|/parse> tree with numbering added.
 {my ($node) = @_;                                                              # Start node.
  my $s = '';
  $s .= $_->prettyStringNumbered for $node->contents;                           # Recurse to get the sub content
  $s
 }

sub xmlHeader($)                                                                #S Add the standard L<XML> header to a string
 {my ($string) = @_;                                                            # String to which a standard L<XML> header should be prefixed
  <<END
<?xml version="1.0" encoding="UTF-8"?>
$string
END
 }

#D2 Html/Json                                                                   # Represent the L<parse|/parse> tree using html or Json

sub htmlTables($)                                                               # Return a string of html representing a L<parse|/parse> tree.
 {my ($node) = @_;                                                              # Start node of parse tree

  my $bgColor = sub                                                             # Create a random background color
   {join q(), q(bgcolor="#), (map {sprintf("%02x", $_)} @_), q(");
   };

  my $idNumber = 0;
  my $id       = sub                                                            # Create the next id number
   {++$idNumber
   };

  my @actions = qw(dup rename cut paste unwrap wrap wrapContent);

  my @nodes;
  my %count;
  my @depth;
  my @path;
  my $height = 0;

  $node->down(sub                                                               # Map xml
   {my ($n) = @_;
    return if $n->isFirstText;
    push @nodes, $n;
    $count{$_}++ for @_;
    push @path, join ' ', $n->path;
    my $h = scalar @_;
    push @depth, $h;
    $height = max($height, $h);
   });

  my @h = q(<html>);                                                            # Generated html

  push @h, qq(<table cellspacing="0" cellpadding="2">);                         # Create table
  for my $i(keys @nodes)
   {my $n          = $nodes[$i];
    my $tag        = -t $n;
    my $line       = $i + 1;
    my $rowspan    = $count{$n};
    my $colspan    = $height-$depth[$i]+1;
    my @color      =  map {int(rand()*255)} 1..3;
    my $color      = &$bgColor(map {127+$_/2} @color);
    my $colorLight = &$bgColor(map {191+$_/4} @color);
    my $click = join '',
     q/onclick="mimClick(event)" path="/, $path[$i], q/"/;

    my ($i1, $i2, $i3, $i4, $i5, $i6, $i7) =
      map
       {qq( onmouseover="mimMouseOver(event)" onmouseout="mimMouseOut(event)").
        qq( onkeydown="mimKeyDown(event)")
       }
      1..7;
    push @h, qq(<tr><td $i1 $color $click align="right">$line</td>);
    push @h,     qq(<td $i2 $color $click rowspan="$rowspan" width="10px"></td>);
    push @h,     qq(<td $i3 $color $click colspan="$colspan" contentEditable="true">$tag</td>) if $colspan;

    if (my $t = $n->isText)                                                     # Text element
     {push @h, qq(<td $i4 contenteditable="true" $colorLight>).$t->text.q(</td>);
     }
    elsif (my $T = $n->firstText)                                               # Text element first below non text element
     {push @h, qq(<td $i5 contenteditable="true" $colorLight>).$T->text.q(</td>);
     }
    elsif ($tag =~ m(\A(b|cmd|info|li|p)\Z)s)
     {push @h, qq(<td $i6 contenteditable="true" $colorLight> </td>);
     }
    else
     {push @h, qq(<td $i7></td>);
     }
   }
  push @h, qq(</table>);                                                        # End of table

  push @h, <<END;
<script>

function say()                                                                  // Say strings
 {console.log(Array.from(arguments).map(a=>a.toString()).join(''))
 }

var mimMouseOverLastColor;

function mimMouseOver(event)
 {const target = event.target;
  mimMouseOverLastColor  = target.style["background-color"];
  target.style["background-color"] = "white";
 }

function mimMouseOut(event)
 {const target = event.target;
  target.style["background-color"] = mimMouseOverLastColor;
 }

function mimClick(event)
 {console.log(event);
 }

function mimKeyDown(event)
 {const target = event.target;
  console.log(event.key+ " "+event.code);
  event.preventDefault();

  if      (event.code == "KeyD")
   {alert("duplicate path "+target.getAttribute("path"));
   }
  else if (event.code == "KeyR")
   {const response = prompt("Enter new name");
    target.innerText = response;
   }
  else if (event.code == "KeyU")
   {alert("unwrap path "+target.getAttribute("path"));
   }
 }

</script>
</html>
END

  join '', map {qq($_\n)} @h
 }

sub jsonString2($)                                                              #P Return a Json representation of a parse tree
 {my ($node) = @_;                                                              # Start node of parse tree

  my $n = {tag=>$node->tag,                                                     # New node
   (keys(%{$node->attributes}) ? (attributes => $node->attributes) : ()),
   ($node->isText              ? (text       => $node->text)       : ())};

  for my $c($node->contents)                                                    # Add content, hoisting any initial text node to the text field.
   {push @{$n->{contents}}, $c->jsonString2;
   }

  $n
 }

sub jsonString($)                                                               #U Return a Json representation of a parse tree
 {my ($node) = @_;                                                              # Start node of parse tree
  encodeJson($node->jsonString2) =~ s(([\}\]],)) ($1\n)gsr;                     # Encode Perl to Json with some line breaks added.
 }

sub xmlToJson($;$)                                                              #S Return a Json string representing valid L<XML> contained in a file or string, optionally double escaping escape characters.
 {my ($xml, $escape) = @_;                                                      # File name or string of valid html, double escape the escaped characters if true
  ref($xml) and confess "Expected a string or a file name";
  my $j = jsonString(new($xml));                                                # Json string
  $j =~ s(\\) (\\\\)gs if $escape;                                              # Escape if requested
  $j
 }

sub jsonToXml2($)                                                               #P Convert a json string to an L<XML> parse tree
 {my ($node) = @_;                                                              # Start node of parse tree
  my $a = $$node{attributes};                                                   # Attributes
  my $n = bless {tag=>$$node{tag},(keys(%$a) ? (attributes => $a) : ())};       # Tag and attributes
     $$n{text} = trim($$node{text}) if $$node{text};                            # Avoid endless blank line expansion of text
  push @{$$n{content}}, &jsonToXml2($_) for @{$$node{contents}};                # Add content
  $n
 }

sub jsonToXml($)                                                                # Convert a json string representing an L<XML> parse tree into an L<XML> parse tree
 {my ($json) = @_;                                                              # Json string
  my $p = decodeJson($json);                                                    # Json represented as Perl
  my $x = jsonToXml2($p);                                                       # Parse tree - enough to print
  my $s = string($x);                                                           # Parse tree as string
  new($s);                                                                      # Recreate full parse tree from string
 }

#D2 Dense                                                                       # Print the L<parse|/parse> tree densely for reuse by computers rather than humans.

sub string($)                                                                   #U Return a dense string representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-s|/opString> B<$node>.
 {my ($node) = @_;                                                              # Start node.
  return $node->{text} if $node->{tag} eq cdata;                                # Text node
  my $content = $node->content;                                                 # Sub nodes
  my $attr    = keys %{$node->{attributes}};                                    # Number of attributes

  return  '<'.$node->{tag}.                       '/>'
    if !@$content and !keys %{$node->{attributes}};                             # No sub nodes or attributes

  return  '<'.$node->{tag}.$node->printAttributes.'/>'
    if !@$content;                                                              # No sub nodes

  join '', '<', $node->{tag}, $node->printAttributes, '>',                      # Has sub nodes
   (map {$_->{tag} eq cdata ? $_->{text} : $_->string} @$content),              # Recurse to get the sub content
           '</', $node->{tag}, '>'
 }

sub stringAsMd5Sum($)                                                           #U Return the L<md5> of the dense L<string|/string> representing a node of a L<parse|/parse> tree minus its L<id> and all the nodes below it. Or use L<-g|/opString> B<$node>. The id of the top most node is not included in the md5sum to equate parse trees that would otherwise only differ by the arbitrary root node id value.
 {my ($node) = @_;                                                              # Node.
  $node->id = undef if my $i = $node->id;                                       # Save id
  my $md5 = stringMd5Sum($node->string);                                        # Md5 sum of string content minus id. The various printing methods will of course all produce different md5 sums but the md5 sum is big enough to accommodate the variations without collisions.
  $node->id = $i if $i;                                                         # Restore id if present
  $md5                                                                          # Return md5 sum
 }

sub stringQuoted($)                                                             #U Return a quoted string representing a L<parse|/parse> tree a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-o|/opString> B<$node>.
 {my ($node) = @_;                                                              # Start node
  "'".$node->string."'"
 }

sub stringReplacingIdsWithLabels($)                                             # Return a string representing the specified L<parse|/parse> tree with the id attribute of each node set to the L<Labels|/Labels> attached to each node.
 {my ($node) = @_;                                                              # Start node.
  return $node->text if $node->isText;                                          # Text node
  my $t = $node->tag;                                                           # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes
  return '<'.$t.$node->printAttributesReplacingIdsWithLabels.'/>' if !@$content;# No sub nodes

  my $s = '<'.$t.$node->printAttributesReplacingIdsWithLabels.'>';              # Has sub nodes
  $s .= $_->stringReplacingIdsWithLabels for @$content;                         # Recurse to get the sub content
  return $s.'</'.$t.'>';
 }

sub stringExtendingIdsWithLabels($)                                             # Return a string representing the specified L<parse|/parse> tree with the id attribute of each node extended by the L<Labels|/Labels> attached to each node.
 {my ($node) = @_;                                                              # Start node.
  return $node->text if $node->isText;                                          # Text node
  my $t = $node->tag;                                                           # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes
  return '<'.$t.$node->printAttributesExtendingIdsWithLabels.'/>' if !@$content;# No sub nodes

  my $s = '<'.$t.$node->printAttributesExtendingIdsWithLabels.'>';              # Has sub nodes
  $s .= $_->stringExtendingIdsWithLabels for @$content;                         # Recurse to get the sub content
  return $s.'</'.$t.'>';
 }

sub stringContent($)                                                            #U Return a string representing all the nodes below a node of a L<parse|/parse> tree.
 {my ($node) = @_;                                                              # Start node.
  my $s = '';
  $s .= $_->string for $node->contents;                                         # Recurse to get the sub content
  $s
 }

sub stringContentOrText($)                                                      #U Return a string representing all the nodes below a node of a L<parse|/parse> tree or the text of the current node if it is a text node.
 {my ($node) = @_;                                                              # Start node.
  return $node->text if $node->isText;
  stringContent($node);
 }

sub stringNode($)                                                               #U Return a string representing the specified B<$node> showing the attributes, labels and node number.
 {my ($node) = @_;                                                              # Node.
  my $s = '';

  if ($node->isText)                                                            # Text node
   {$s = 'CDATA='.$node->text;
   }
  else                                                                          # Non text node
   {$s = $node->tag.$node->printAttributes;
   }

  if (my $n = $node->number)                                                    # Node number if present
   {$s .= "($n)"
   }

  if (my @l = $node->getLabels)                                                 # Labels
   {$s .= " ${_}:".$l[$_] for keys @l;
   }

  $s
 }

sub stringTagsAndText($)                                                        #U Return a string showing just the tags and text at and below a specified B<$node>.
 {my ($node) = @_;                                                              # Node.
  my @s;                                                                        # String or each node
  for my $n($node->byList)                                                      # Each node
   {push @s, $n->isText ? trim($n->text) : $n->tag;                             # Representation of node
   }
  return @s if wantarray;                                                       # Representations as array
  join ' ', @s                                                                  # Representation as string
 }

sub stringText($)                                                               #U Return a string showing just the text of the text nodes (separated by blanks) at and below a specified B<$node>.
 {my ($node) = @_;                                                              # Node.
  my @s;                                                                        # String or each node
  for my $n($node->byList)                                                      # Each node
   {push @s, trim($n->text) if $n->isText;                                      # Text of text nodes
   }
  return @s if wantarray;                                                       # Representations as array
  join ' ', @s                                                                  # Representation as string
 }

sub setRepresentationAsTagsAndText($)                                           #U Sets the L<representationLast|/representationLast> for every node in the specified B<$tree> via L<stringTagsAndText|/stringTagsAndText>.
 {my ($tree) = @_;                                                              # Tree of nodes.
  $tree->by(sub                                                                 # Each node
   {$_->representationLast = $_->stringTagsAndText;                             # Set representationLast for each node
   });
  $tree
 }

sub setRepresentationAsText($)                                                  #U Sets the L<representationLast|/representationLast> for every node in the specified B<$tree> via L<stringText|/stringText>.
 {my ($tree) = @_;                                                              # Tree of nodes.
  $tree->by(sub                                                                 # Each node
   {$_->representationLast = $_->stringText;                                    # Set representationLast for each node
   });
  $tree
 }

sub matchNodesByRepresentation($)                                               #U Creates a hash of arrays of nodes that have the same representation in the specified B<$tree>. Set L<representation|/representationLast> for each node in the tree before calling this method.
 {my ($tree) = @_;                                                              # Tree to examine
  my %map;                                                                      # Map of arrays of nodes that have the same representation.
  for my $n($tree->byList)                                                      # Each node
   {push @{$map{$n->representationLast}}, $n;                                   # Classify node by representation
   }
  \%map                                                                         # Return results
 }

#D2 Conditions                                                                  # Print a subset of the L<parse|/parse> tree determined by the conditions attached to it.

sub stringWithConditions($@)                                                    # Return a string representing the specified B<$node> of a L<parse|/parse> tree and all the nodes below it subject to conditions to select or reject some nodes.
 {my ($node, @conditions) = @_;                                                 # Start node, conditions to be regarded as in effect.
  return $node->text if $node->isText;                                          # Text node
  my %c = %{$node->conditions};                                                 # Process conditions if any for this node
  return '' if keys %c and @conditions and !grep {$c{$_}} @conditions;          # Return if conditions are in effect and no conditions match
  my $t = $node->tag;                                                           # Not text so it has a tag
  my $content = $node->content;                                                 # Sub nodes

  my $s = ''; $s .= $_->stringWithConditions(@conditions) for @$content;        # Recurse to get the sub content
  return '<'.$t.$node->printAttributes.'/>' if !@$content or $s =~ /\A\s*\Z/;   # No sub nodes or none selected
  '<'.$t.$node->printAttributes.'>'.$s.'</'.$t.'>';                             # Has sub nodes
 }

sub condition($$@)                                                              #CY  Return the B<$node> if it has the specified B<$condition> and is in the optional B<@context>, else return B<undef>
 {my ($node, $condition, @context) = @_;                                        # Node, condition to check, optional context
  return undef if @context and !$node->at(@context);                            # Check optional context
  $node->conditions->{$condition} ? $node : undef                               # Return node if it has the specified condition, else B<undef>
 }

sub anyCondition($@)                                                            #Y Return the B<$node> if it has any of the specified B<@conditions>, else return B<undef>
 {my ($node, @conditions) = @_;                                                 # Node, conditions to check
  $node->conditions->{$_} ? return $node : undef for @conditions;               # Return node if any of the specified conditions are present
  undef                                                                         # No conditions present
 }

sub allConditions($@)                                                           #Y  Return the B<$node> if it has all of the specified B<@conditions>, else return B<undef>
 {my ($node, @conditions) = @_;                                                 # Node, conditions to check
  !$node->conditions->{$_} ? return undef : undef for @conditions;              # Return node if any of the specified conditions are missing
  $node                                                                         # All conditions present
 }

sub addConditions($@)                                                           # Given a B<$node> add the specified B<@conditions> and return the node.
 {my ($node, @conditions) = @_;                                                 # Node, conditions to add.
  $node->conditions->{$_}++ for @conditions;
  $node
 }

sub deleteConditions($@)                                                        # Given a B<$node> delete any B<@conditions> applied to the $node and return the $node.
 {my ($node, @conditions) = @_;                                                 # Node, conditions to add.
  delete $node->conditions->{$_} for @conditions;
  $node
 }

sub listConditions($)                                                           # Return a list of conditions applied to a B<$node>.
 {my ($node) = @_;                                                              # Node.
  sort keys %{$node->conditions}
 }

#D1 Attributes                                                                  # Get or set the attributes of nodes in the L<parse|/parse> tree. L<Well Known Attributes|/Well Known Attributes>  can be set directly via L<lvalueMethod> B<sub>s. To set or get the values of other attributes use L<Get or Set Attributes|/Get or Set Attributes>. To delete or rename attributes see: L<Other Operations on Attributes|/Other Operations on Attributes>.

#D2 Well Known Attributes                                                       # Get or set these node attributes via L<lvalueMethod> B<sub>s as in:\m  $x->href = "#ref";

if (0) {
genHash(__PACKAGE__,                                                            # Well known attributes useful in respect to L<XML> that conforms to the L<Dita> standard.
  audience    => undef,                                                         # Attribute B<audience> for a node as an L<lvalueMethod> B<sub>.    Use B<audienceX()> to return B<q()> rather than B<undef>.
  class       => undef,                                                         # Attribute B<class> for a node as an L<lvalueMethod> B<sub>.       Use B<classX()> to return B<q()> rather than B<undef>.
  guid        => undef,                                                         # Attribute B<guid> for a node as an L<lvalueMethod> B<sub>.        Use B<guidX()> to return B<q()> rather than B<undef>.
  href        => undef,                                                         # Attribute B<href> for a node as an L<lvalueMethod> B<sub>.        Use B<hrefX()> to return B<q()> rather than B<undef>.
  id          => undef,                                                         # Attribute B<id> for a node as an L<lvalueMethod> B<sub>.          Use B<idX()> to return B<q()> rather than B<undef>.
  lang        => undef,                                                         # Attribute B<lang> for a node as an L<lvalueMethod> B<sub>.        Use B<langX()> to return B<q()> rather than B<undef>.
  navtitle    => undef,                                                         # Attribute B<navtitle> for a node as an L<lvalueMethod> B<sub>.    Use B<navtitleX()> to return B<q()> rather than B<undef>.
  otherprops  => undef,                                                         # Attribute B<otherprops> for a node as an L<lvalueMethod> B<sub>.  Use B<otherpropsX()> to return B<q()> rather than B<undef>.
  outputclass => undef,                                                         # Attribute B<outputclass> for a node as an L<lvalueMethod> B<sub>. Use B<outputclassX()> to return B<q()> rather than B<undef>.
  props       => undef,                                                         # Attribute B<props> for a node as an L<lvalueMethod> B<sub>.       Use B<propsX()> to return B<q()> rather than B<undef>.
  style       => undef,                                                         # Attribute B<style> for a node as an L<lvalueMethod> B<sub>.       Use B<styleX()> to return B<q()> rather than B<undef>.
  type        => undef,                                                         # Attribute B<type> for a node as an L<lvalueMethod> B<sub>.        Use B<typeX()> to return B<q()> rather than B<undef>.
  xtrc        => undef,                                                         # Attribute B<xtrc> for a node as an L<lvalueMethod> B<sub>.        Use B<classX()> to return B<q()> rather than B<undef>.
  xtrf        => undef,                                                         # Attribute B<xtrf> for a node as an L<lvalueMethod> B<sub>.        Use B<classX()> to return B<q()> rather than B<undef>.
 );
}

BEGIN                                                                           # The above documents the attributes created below using the L<attr|_/attr> method.
 {for my $a(qw(audience class guid href id lang navtitle),                      # Return well known attributes as an assignable value
            qw(otherprops outputclass props style type xtrf))
   {eval 'sub '.$a. '($) :lvalue {&attr($_[0], q('.$a.'))}'.
         'sub '.$a.'X($)         {&attr($_[0], q('.$a.')) // q()}';
    $@ and confess "Cannot create well known attribute $a\n$@";

    my $A = ucfirst $a;
    for my $c(qw(at first next prev last))                                      # Commands to attach to attributes
     {my $cmd = 'sub '.$c.$A.'($)'.
      ' {my ($node, $attrValue, @context) = @_;'.
      '  my $A = $node->attr(q('.$a.'));'.
      '  return $node if $node->'.$c.'(@context) and $A and $A eq $attrValue;'.
      ' undef;}';
      eval $cmd;
      $@ and confess "Cannot create well known attribute $a\n$@";
     }
   }
 }

#D2 Get or Set Attributes                                                       # Get or set the attributes of nodes.
sub attr($$) :lvalue                                                            # Return the value of an attribute of the current node as an L<lvalueMethod> B<sub>.
 {my ($node, $attribute) = @_;                                                  # Node in parse tree, attribute name.
  $node->attributes->{$attribute}
 }

sub attrX($$)                                                                   #I Return the value of the specified B<$attribute> of the specified B<$node> or B<q()> if the B<$node> does not have such an attribute.
 {my ($node, $attribute) = @_;                                                  # Node in parse tree, attribute name.
  $node->attributes->{$attribute} // ''
 }

sub set($%)                                                                     #U Set the values of some attributes in a node and return the node. Identical in effect to L<setAttr|/setAttr>.
 {my ($node, %values) = @_;                                                     # Node in parse tree, (attribute name=>new value)*
  s/["<>]/ /gs for grep {$_} values %values;                                    # We cannot have these characters in an attribute
  $node->attributes->{$_} = $values{$_} for keys %values;                       # Set attributes
  $node
 }
#a deleteAttrs
#b <b/>
#c at b
#c set id bb class cc href hh
#d Set some attributes.

sub addAttr($%)                                                                 #U Check the specified B<$node> for the specified B<%attributes> and add any that are not already set.  Returns the current node.
 {my ($node, %values) = @_;                                                     # Node in parse tree, (attribute name=>new value)*
  s/["<>]/ /gs for grep {$_} values %values;                                    # We cannot have these characters in an attribute
  $node->attributes->{$_} //= $values{$_} for keys %values;                     # Set attributes
  $node
 }

sub setAttr($%)                                                                 #U Set the values of some attributes in a node and return the node. Identical in effect to L<set|/set>.
 {my ($node, %values) = @_;                                                     # Node in parse tree, (attribute name=>new value)*
  s/["<>]/ /gs for grep {$_} values %values;                                    # We cannot have these characters in an attribute
  $node->attributes->{$_} = $values{$_} for keys %values;                       # Set attributes
  $node
 }

#D2 Other Operations on Attributes                                              # Perform operations other than get or set on the attributes of a node
sub attrs($@)                                                                   # Return the values of the specified attributes of the current node as a list
 {my ($node, @attributes) = @_;                                                 # Node in parse tree, attribute names.
  my @v;
  my $a = $node->attributes;
  push @v, $a->{$_} for @attributes;
  @v
 }

sub attrCount($@)                                                               #U Return the number of attributes in the specified B<$node>, optionally ignoring the specified names from the count.
 {my ($node, @exclude) = @_;                                                    # Node in parse tree, optional attribute names to exclude from the count.
  my $a = $node->attributes;                                                    # Attributes
  return scalar grep {defined $a->{$_}} keys %$a if @exclude == 0;              # Count all attributes
  my %e = map{$_=>1} @exclude;                                                  # Hash of attributes to be excluded
  scalar grep {defined $a->{$_} and !$e{$_}} keys %$a;                          # Count attributes that are not excluded
 }

sub getAttrs($)                                                                 # Return a sorted list of all the attributes on the specified B<$node>.
 {my ($node) = @_;                                                              # Node in parse tree.
  my $a = $node->attributes;                                                    # Attributes
  grep {defined $a->{$_}} sort keys %$a                                         # Attributes
 }

sub deleteAttr($$;$)                                                            #U Delete the named attribute in the specified B<$node>, optionally check its value first, returning the value of the attribute or B<undef> if the attribute does not exist on this node.
 {my ($node, $attr, $value) = @_;                                               # Node, attribute name, optional attribute value to check first.
  @_ > 3 and confess
   "A maximum of three parameters are allowed or try deleteAttrs";              # Deflect to deleteAttrs if too many parameters
  my $a = $node->attributes;                                                    # Attributes hash
  if (@_ == 3)
   {if (defined($a->{$attr}) and $a->{$attr} eq $value)                         # Delete user key if it has the right value
     {delete $a->{$attr}; return $value;                                        # Delete user key if it has the right value
     }
    return undef;                                                               # Attribute has wrong value so return undef
   }
  else
   {return delete $a->{$attr};                                                  # Delete user key unconditionally
   }
 }
#a
#b <b class="help"/><b class="ref"/>
#c deleteAttr class help
#d Delete an attribute if has a specific value.

sub deleteAttrs($@)                                                             #U Delete the specified attributes of the specified B<$node> without checking their values and return the node.
 {my ($node, @attrs) = @_;                                                      # Node, Names of the attributes to delete
  my $a = $node->attributes;                                                    # Attributes hash
  delete $a->{$_} for @attrs;
  $node
 }
#a deleteAttr
#b <b id="b"/><c id="c" class="c" href="c"/>
#c at c
#c deleteAttrs id class
#d Delete the named attributes.
sub deleteAttrsInTree($@)                                                       #U Delete the specified attributes of the specified B<$node> and all the nodes under it and return the specified B<$node>.
 {my ($node, @attrs) = @_;                                                      # Node, Names of the attributes to delete
  $node->by(sub                                                                 # Traverse the parse tree
   {my ($o) = @_;
    my $a = $o->attributes;                                                     # Attributes hash
    delete $a->{$_} for @attrs;                                                 # Delete the specified attributes
   });
  $node
 }

sub attrsNone($@)                                                               #CU Check that the specified B<$node> has no attributes. Return the specified $node if no attributes were found else B<undef>. Invented by MfM.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Check optional context
  my $a = $node->attributes;                                                    # Attributes hash
  keys(%$a) ? undef : $node                                                     # Fail if there are attributes
 }
#a attrs
#b <b/><c id="cc"/>
#c attrsNone b a
#c set id had_no_attributes
#d Continue if there are no attributes present.

sub deleteAttrValueAtInTree($$$@)                                               #C Delete all instances of the specified B<$attribute> with the specified B<$value> in the specified B<@context> in the specified B<$tree> and return the modified B<$tree>. An undefined B<$value> will cause the attribute to be deleted without first confirming its value. An empty context will remove the attribute from every node in the B<$tree>.
 {my ($tree, $attribute, $value, @context) = @_;                                # Tree, attribute name, attribute value or B<undef> for all values, optional context
  $tree->by(sub                                                                 # Traverse the parse tree
   {my ($o) = @_;
    return if @context and !$o->at(@context);                                   # Not in specified context
    $o->deleteAttr($attribute, $value);                                         # Delete the named attribute if its value matches
   });
  $tree                                                                         # Return the modified parse tree
 }

sub renameAttr($$$@)                                                            #CU Rename attribute B<$old> to B<$new> in the specified B<$node> with optional context B<@context> regardless of whether attribute B<$new> already exists or not and return the B<$node>. To prevent inadvertent changes to an existing attribute use L<changeAttr|/changeAttr>.
 {my ($node, $old, $new, @context) = @_;                                        # Node, existing attribute name, new attribute name, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  my $a = $node->attributes;                                                    # Attributes hash
  if (defined($a->{$old}))                                                      # Check old attribute exists
   {my $value = $a->{$old};                                                     # Existing value
    $a->{$new} = $value;                                                        # Change the attribute name
    delete $a->{$old};
   }
  $node
 }
#a attrs set deleteAttrs
#b <b class="bb"/>
#c renameAttr class href
#d Rename an attribute.

sub renameAttrXtr($@)                                                           #CU Rename the attributes B<@attr> as far as possible to xtrc or xtrf.  Returns an array of the attributes that could not be so renamed.
 {my ($node, @attr) = @_;                                                       # Node, attributes to rename if they exist
  my $a = $node->attributes;                                                    # Attributes hash
  my @a;                                                                        # Attributes unable to rename
  for my $attr(@attr)                                                           # Each attribute to rename
   {next unless defined $a->{$attr};                                            # Requested attribute not present
    if    (!defined($a->{xtrc}))                                                # xtrc
     {$a->{xtrc} = delete $a->{$attr};
     }
    elsif (!defined($a->{xtrf}))                                                # xtrf
     {$a->{xtrf} = delete $a->{$attr};
     }
    else                                                                        # Cannot rename
     {push @a, $attr;
     }
   }
  @a                                                                            # Unrenameable attributes
 }

sub changeAttr($$$@)                                                            #CU Rename attribute B<$old> to B<$new> in the specified B<$node> with optional context B<@context> unless attribute B<$new> is already set and return the B<$node>. To make changes regardless of whether the new attribute already exists use L<renameAttr|/renameAttr>.
 {my ($node, $old, $new, @context) = @_;                                        # Node, existing attribute name, new attribute name, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  exists $node->attributes->{$new} ? $node : $node->renameAttr($old, $new)      # Check old attribute exists
 }

sub changeOrDeleteAttr($$$@)                                                    #CU Rename attribute B<$old> to B<$new> in the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set in which case delete attribute B<$old>. Return B<$node> regardless of what action was taken. To make changes regardless of whether the new attribute already exists use L<renameAttr|/renameAttr>.
 {my ($node, $old, $new, @context) = @_;                                        # Node, existing attribute name, new attribute name, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  exists $node->attributes->{$new} ? $node->deleteAttr($old) :                  # Delete or rename attribute
                                     $node->renameAttr($old, $new)
 }

sub renameAttrValue($$$$$@)                                                     #CU Rename attribute B<$old> to B<$new> with new value B<$newValue> in the specified B<$node> in the optional B<@context> regardless of whether attribute B<$new> already exists or not as long as the attribute B<$old> has the value B<$oldValue>. Return the B<$node> regardless of what changes were made. To prevent inadvertent changes to existing attributes use L<changeAttrValue|/changeAttrValue>.
 {my ($node, $old, $oldValue, $new, $newValue, @context) = @_;                  # Node, existing attribute name, existing attribute value, new attribute name, new attribute value, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  my $a = $node->attributes;                                                    # Attributes hash
  if (defined($a->{$old}) and $a->{$old} eq $oldValue)                          # Check old attribute exists and has the specified value
   {$a->{$new} = $newValue;                                                     # Change the attribute name
    delete $a->{$old};
   }
  $node
 }

sub changeAttrValue($$$$$@)                                                     #CU Rename attribute B<$old> to B<$new> with new value B<$newValue> on the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set or the value of the B<$old> attribute is not B<$oldValue>. Return the B<$node> regardless of what changes were made.  To make changes regardless of whether the new attribute already exists use L<renameAttrValue|/renameAttrValue>.
 {my ($node, $old, $oldValue, $new, $newValue, @context) = @_;                  # Node, existing attribute name, existing attribute value, new attribute name, new attribute value, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  exists $node->attributes->{$new} ? $node :                                    # Check old attribute exists
    $node->renameAttrValue($old, $oldValue, $new, $newValue)
 }

sub changeAttributeValue($$$@)                                                  #CU Apply a sub to the value of an attribute of the specified B<$node>.  The value to be changed is supplied and returned in: L<$_>.
 {my ($node, $attribute, $sub, @context) = @_;                                  # Node, attribute name, change sub, optional context.
  return () if @context and !$node->at(@context);                               # Check optional context
  return unless local $_ = $node->attr($attribute);                             # Check that the attribute has a value we can change
  &$sub;                                                                        # Change the attribute
  $node->set($attribute, $_)                                                    # Update the attribute's value
 }

sub changeOrDeleteAttrValue($$$$$@)                                             #CU Rename attribute B<$old> to B<$new> with new value B<$newValue> on the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set or the value of the B<$old> attribute is not B<$oldValue> in which cases the B<$old> attribute is deleted. Return the B<$node> regardless of any changes made.  To make changes regardless of whether the new attribute already exists use L<renameAttrValue|/renameAttrValue>.
 {my ($node, $old, $oldValue, $new, $newValue, @context) = @_;                  # Node, existing attribute name, existing attribute value, new attribute name, new attribute value, optional context.
  return undef if @context and !$node->at(@context);                            # Check optional context
  exists $node->attributes->{$new} ? $node->deleteAttr($old) :                  # Check old attribute exists
    $node->renameAttrValue($old, $oldValue, $new, $newValue)
 }

sub copyAttrs($$@)                                                              # Copy all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, overwriting any existing attributes in the target node and return the source node.
 {my ($source, $target, @attr) = @_;                                            # Source node, target node, optional list of attributes to copy
  my $s = $source->attributes;                                                  # Source attributes hash
  my $t = $target->attributes;                                                  # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = $s->{$_} for @attr;                                              # Transfer each named attribute
   }
  else                                                                          # All attributes
   {$t->{$_} = $s->{$_} for sort keys %$s;                                      # Transfer each source attribute
   }
  $source                                                                       # Return source node
 }

sub copyAttrsFromParent($@)                                                     #U Copy all the attributes from the parent (if there is one) of the current B<$node> to $node  or just the named B<@attributes> and return $node. If $node is the top of the parse tree then return B<undef> as it does not have a parent.
 {my ($node, @attr) = @_;                                                       # Node, attributes to copy from parent
  return undef unless my $parent = $node->parent;
  my $s = $parent->attributes;                                                  # Source attributes hash
  my $t = $node->attributes;                                                    # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = $s->{$_} for @attr;                                              # Transfer each named attribute
   }
  else                                                                          # All attributes
   {$t->{$_} = $s->{$_} for sort keys %$s;                                      # Transfer all attributes from parent
   }
  $node                                                                         # Return node
 }

sub copyAttrsToParent($@)                                                       #U Copy all the attributes of the specified B<$node> to its parent (if there is one) or just the named B<@attributes> and return $node. If $node is the top of the parse tree then return B<undef> as it does not have a parent.
 {my ($node, @attr) = @_;                                                       # Node, attributes to copy from parent
  return undef unless my $parent = $node->parent;
  my $s = $node->attributes;                                                    # Source attributes hash
  my $t = $parent->attributes;                                                  # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = $s->{$_} for @attr;                                              # Transfer each named attribute
   }
  else                                                                          # All attributes
   {$t->{$_} = $s->{$_} for sort keys %$s;                                      # Transfer all attributes from parent
   }
  $node                                                                         # Return node
 }

sub copyNewAttrs($$@)                                                           # Copy all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, without overwriting any existing attributes in the target node and return the source node.
 {my ($source, $target, @attr) = @_;                                            # Source node, target node, optional list of attributes to copy
  my $s = $source->attributes;                                                  # Source attributes hash
  my $t = $target->attributes;                                                  # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = $s->{$_} for grep {!exists $t->{$_}} @attr;                      # Transfer each named attribute not already present in the target
   }
  else                                                                          # All attributes
   {$t->{$_} = $s->{$_} for grep {!exists $t->{$_}} sort keys %$s;              # Transfer each source attribute not already present in the target
   }
  $source                                                                       # Return source node
 }

sub moveAttrs($$@)                                                              # Move all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to move is supplied, overwriting any existing attributes in the target node and return the source node.
 {my ($source, $target, @attr) = @_;                                            # Source node, target node, attributes to move
  my $s = $source->attributes;                                                  # Source attributes hash
  my $t = $target->attributes;                                                  # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = delete $s->{$_} for @attr;                                       # Transfer each named attribute and delete from the source node
   }
  else                                                                          # All attributes
   {$t->{$_} = delete $s->{$_} for sort keys %$s;                               # Transfer each source attribute and delete from source node
   }
  $source                                                                       # Return source node
 }

sub moveNewAttrs($$@)                                                           # Move all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, without overwriting any existing attributes in the target node and return the source node.
 {my ($source, $target, @attr) = @_;                                            # Source node, target node, optional list of attributes to move
  my $s = $source->attributes;                                                  # Source attributes hash
  my $t = $target->attributes;                                                  # Target attributes hash
  if (@attr)                                                                    # Named attributes
   {$t->{$_} = delete $s->{$_} for grep {!exists $t->{$_}} @attr;               # Transfer each named attribute and delete it from the source node as long as it does not already exist in the target
   }
  else                                                                          # All attributes
   {$t->{$_} = delete $s->{$_} for grep {!exists $t->{$_}} sort keys %$s;       # Transfer every attribute and delete it from the source node as long as it does not already exist in the target
   }
  $source                                                                       # Return source node
 }

#D1 Traversal                                                                   # Traverse the L<parse|/parse> tree in various orders applying a B<sub> to each node.

#D2 Post-order                                                                  # This order allows you to edit children before their parents.

sub by2($$@)                                                                    #P Post-order traversal of a L<parse|/parse> tree
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  $_->by2($sub, $node, @context) for $node->contents;                           # Recurse to process sub nodes in deeper context
  &$sub(local $_ = $node, @context);                                            # Process specified node last
  $node
 }

sub By22($$)                                                                    #P Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. A reference to the current node is also made available via L<$_>. This is equivalent to the L<x=|/opBy> operator.
 {my ($node, $sub) = @_;                                                        # Starting node, sub to call for each sub node
  $node->by2($sub);                                                             # Recurse through nodes
  $node
 }

# Doubles performance of by!  IT is tempting to think that removing all the parameters would speed things up a lot - it does not as most parse trees are not very deep.

sub by($$)                                                                      #I Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. A reference to the current node is also made available via L<$_>. This is equivalent to the L<x=|/opBy> operator.
 {my ($node, $sub) = @_;                                                        # Starting node, sub to call for each sub node

  my $by; $by = sub                                                             # Recurse to process sub nodes in deeper context
   {$_ = $_[0];                                                                 # Save active node in $_
    if (my $c = $_->{content})                                                  # Contents reference
     {my @c = @$c;                                                              # For some reason we cannot place this directly in a for loop reliably
      for(@c)                                                                   # Each node under the current node
       {if (my $d = $_->{content})                                              # Contents reference
         {my @d = @$d;                                                          # Content
          my $D = $_;                                                           # Save $_ so we can use it in the following loop
          for(@d)                                                               # Each node under the current node
           {if (my $e = $_->{content})                                          # Contents reference
             {my @e = @$e;                                                      # For some reason we cannot place this directly in a for loop reliably
              my $E = $_;                                                       # Save $_ so we can use it in the following loop
              for(@e)                                                           # Each node under the current node
               {if (my $f = $_->{content})                                      # Contents reference
                 {my @f = @$f;                                                  # Content
                  my $F = $_;                                                   # Save $_ so we can use it in the following loop
                  for(@f)                                                       # Each node under the current node
                   {if (my $g = $_->{content})                                  # Contents reference
                     {my @g = @$g;                                              # Content
                      my $G = $_;                                               # Save $_ so we can use it in the following loop
                      for(@g)                                                   # Each node under the current node
                       {if (my $h = $_->{content})                              # Contents reference
                         {my @h = @$h;                                          # Content
                          my $H = $_;                                           # Save $_ so we can use it in the following loop
                          for(@h)                                               # Each node under the current node
                           {if (my $i = $_->{content})                          # Contents reference
                             {my @i = @$i;                                      # Content
                              my $I = $_;                                       # Save $_ so we can use it in the following loop
                              for(@i)                                           # Each node under the current node
                               {if (my $j = $_->{content})                      # Contents reference
                                 {my @j = @$j;                                  # Content
                                  my $J = $_;                                   # Save $_ so we can use it in the following loop
                                  for(@j) {                                     # Each node under the current node
                                    &$by($_, $J, $I, $H, $G, $F, $E, $D, @_);   # Recurse
                                   }
                                 }
                                &$sub($_, $I, $H, $G, $F, $E, $D, @_);          # Process current node in post order
                               }
                             }
                            &$sub($_, $H, $G, $F, $E, $D, @_);                  # Process current node in post order
                           }
                         }
                        &$sub($_, $G, $F, $E, $D, @_);                          # Process current node in post order
                       }
                     }
                    &$sub($_, $F, $E, $D, @_);                                  # Process current node in post order
                   }
                 }
                &$sub($_, $E, $D, @_);                                          # Process current node in post order
               }
             }
            &$sub($_, $D, @_);                                                  # Process current node in post order
           }
         }
        &$sub($_, @_);                                                          # Process current node in post order
       }
     }
    &$sub(@_);                                                                  # Process current node in post order
   };

  &$by($node);                                                                  # Recurse through nodes

  $node
 }

sub byX($$)                                                                     #C Post-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die>. The traversal is halted if the called sub does  L<die> on any call with the reason in L<?@|http://perldoc.perl.org/perlvar.html#Error-Variables> The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.\mReturns the start node regardless of the outcome of calling B<sub>.
 {my ($node, $sub) = @_;                                                        # Start node, sub to call
  eval {$node->byX2($sub)};                                                     # Trap any errors that occur
  $node
 }

sub byX2($$@)                                                                   #P Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> within L<eval>B<{}> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call, accumulated context.
  $_->byX2($sub, $node, @context) for $node->contents;                          # Recurse to process sub nodes in deeper context
  &$sub(local $_ = $node, @context);                                            # Process specified node last
 }

sub byX22($$@)                                                                  #P Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> within L<eval>B<{}> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call, accumulated context.
  $_->byX($sub, $node, @context) for $node->contents;                           # Recurse to process sub nodes in deeper context
  eval {&$sub(local $_ = $node, @context)};                                     # Process specified node last
  $node
 }

sub byList($@)                                                                  #C Return a list of all the nodes at and below a specified B<$node> in post-order or the empty list if the B<$node> is not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Starting node, optional context
  return () if @context and !$node->at(@context);                               # Check optional context
  my @n;                                                                        # Nodes under specified node
  $node->by(sub{push @n, $_});                                                  # Retrieve nodes in post-order
  @n                                                                            # Return list of nodes
 }

sub byReverse($$@)                                                              # Reverse post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting B<$node>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  $_->byReverse($sub, $node, @context) for reverse $node->contents;             # Recurse to process sub nodes in deeper context
  &$sub(local $_ = $node, @context);                                            # Process specified node last
  $node
 }

sub byReverseX($$@)                                                             # Reverse post-order traversal of a L<parse|/parse> tree or sub tree below the specified B<$node> calling the specified B<sub> within L<eval>B<{}> at each node and returning the specified starting B<$node>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  $_->byReverseX($sub, $node, @context) for reverse $node->contents;            # Recurse to process sub nodes in deeper context
  &$sub(local $_ = $node, @context);                                            # Process specified node last
  $node
 }

sub byReverseList($@)                                                           #C Return a list of all the nodes at and below a specified B<$node> in reverse preorder or the empty list if the specified B<$node> is not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Starting node, optional context
  return () if @context and !$node->at(@context);                               # Check optional context
  my @n;                                                                        # Nodes
  $node->byReverse(sub{push @n, $_});                                           # Retrieve nodes in reverse post-order
  @n                                                                            # Return list of nodes
 }

#D2 Pre-order                                                                   # This order allows you to edit children after their parents

sub down($$@)                                                                   # Pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  &$sub(local $_ = $node, @context);                                            # Process specified node first
  $_->down($sub, $node, @context) for $node->contents;                          # Recurse to process sub nodes in deeper context
  $node
 }

sub downX($$)                                                                   # Pre-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die>. The traversal is halted for the entire L<parse|/parse> tree if the called sub does L<die> with the reason returned in L<?@|http://perldoc.perl.org/perlvar.html#Error-Variables>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.\mReturns the start node regardless of the outcome of calling B<sub>.
 {my ($node, $sub) = @_;                                                        # Start node, sub to call
  eval {$node->downX2($sub)};                                                   # Trap any errors that occur
  $node
 }

sub downX2($$@)                                                                 #P Pre-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> within L<eval>B<{}> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call, accumulated context.
  &$sub(local $_ = $node, @context);                                            # Process specified node last
  $_->downX2($sub, $node, @context) for $node->contents;                        # Recurse to process sub nodes in deeper context
 }

sub downToDie($$)                                                               # Pre-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die>. The traversal of the current sub tree is halted and continue with the next sibling or parent if the called sub does L<die>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.\mReturns the start node regardless of the outcome of calling B<sub>.
 {my ($node, $sub) = @_;                                                        # Start node, sub to call
  $node->downToDie2($sub);
  $node
 }

sub downToDie2($$@)                                                             #P Pre-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die>. The traversal of the current sub tree is halted and continue with the next sibling or parent if the called sub does L<die>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.\mReturns the start node regardless of the outcome of calling B<sub>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call, accumulated context.
  eval {&$sub(local $_ = $node, @context)};                                     # Process specified node last
  unless($@)                                                                    # Skip sub tree if sub died while processing parent
   {$_->downToDie2($sub, $node, @context) for $node->contents;                  # Recurse to process sub nodes in deeper context
   }
 }

sub downList($@)                                                                #C Return a list of all the nodes at and below a specified B<$node> in pre-order or the empty list if the B<$node> is not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Starting node, optional context
  return () if @context and !$node->at(@context);                               # Check optional context
  my @n;                                                                        # Nodes under specified node
  $node->down(sub{push @n, $_});                                                # Retrieve nodes in pre-order
  @n                                                                            # Return list of nodes
 }

sub downReverse($$@)                                                            # Reverse pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  &$sub(local $_ = $node, @context);                                            # Process specified node first
  $_->downReverse($sub, $node, @context) for reverse $node->contents;           # Recurse to process sub nodes in deeper context
  $node
 }

sub downReverseX($$@)                                                           # Reverse pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> within L<eval>B<{}> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $sub, @context) = @_;                                              # Starting node, sub to call for each sub node, accumulated context.
  &$sub(local $_ = $node, @context);                                            # Process specified node first
  $_->downReverseX($sub, $node, @context) for reverse $node->contents;          # Recurse to process sub nodes in deeper context
  $node
 }

sub downReverseList($@)                                                         #C Return a list of all the nodes at and below a specified B<$node> in reverse pre-order or the empty list if the B<$node> is not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Starting node, optional context
  return () if @context and !$node->at(@context);                               # Check optional context
  my @n;                                                                        # Nodes under specified node
  $node->downReverse(sub{push @n, $_});                                         # Retrieve nodes in reverse pre-order
  @n                                                                            # Return list of nodes
 }

#D2 Pre and Post order                                                          # Visit the parent first, then the children, then the parent again.

sub through($$$@)                                                               # Traverse L<parse|/parse> tree visiting each node twice calling the specified sub B<$before> as we go down past the node and sub B<$after> as we go up past the node, finally return the specified starting node. The subs B<$before, $after> are passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.
 {my ($node, $before, $after, @context) = @_;                                   # Starting node, sub to call when we meet a node, sub to call we leave a node, accumulated context.
  &$before(local $_ = $node, @context) if $before;                              # Process specified node first with before()
  $_->through($before, $after, $node, @context) for $node->contents;            # Recurse to process sub nodes in deeper context
  &$after(local $_ = $node, @context) if $after;                                # Process specified node last with after()
  $node
 }

sub throughX($$$@)                                                              # Identical to L<through|/through> except the B<$before, $after> subs are called in an L<eval> block to prevent L<die> terminating the traversal of the full tree.
 {my ($node, $before, $after, @context) = @_;                                   # Starting node, sub to call when we meet a node, sub to call we leave a node, accumulated context.
  &$before(local $_ = $node, @context);                                         # Process specified node first with before()
  $_->throughX($before, $after, $node, @context) for $node->contents;           # Recurse to process sub nodes in deeper context
  &$after(local $_ = $node, @context);                                          # Process specified node last with after()
  $node
 }

#D2 Range                                                                       # Ranges of nodes

sub from($@)                                                                    # Return a list consisting of the specified node and its following siblings optionally including only those nodes that match one of the tags in the specified list.
 {my ($start, @match) = @_;                                                     # Start node, optional list of tags to match
  my $p = $start->parent;                                                       # Parent node
  confess "No parent" unless $p;                                                # Not possible on a root node
  my @c = $p->contents;                                                         # Content
  shift @c while @c and $c[ 0] != $start;                                       # Position on start node
  if (@match)                                                                   # Select matching nodes if requested
   {my %m = map {$_=>1} @match;
    return grep {$m{$_->tag}} @c;
   }
  @c                                                                            # Elements in the specified range
 }

sub to($@)                                                                      # Return a list of the sibling nodes preceding the specified node optionally including only those nodes that match one of the tags in the specified list.
 {my ($end, @match) = @_;                                                       # End node, optional list of tags to match
  my $q = $end->parent;                                                         # Parent node
  confess "No parent" unless $q;                                                # Not possible on a root node
  my @c = $q->contents;                                                         # Content
  pop @c while @c and $c[-1] != $end;                                           # Position on end
  if (@match)                                                                   # Select matching nodes if requested
   {my %m = map {$_=>1} @match;
    return grep {$m{$_->tag}} @c;
   }
  @c                                                                            # Elements in the specified range
 }

sub fromTo($$@)                                                                 # Return a list of the nodes between the specified start and end nodes optionally including only those nodes that match one of the tags in the specified list.
 {my ($start, $end, @match) = @_;                                               # Start node, end node, optional list of tags to match
  my $p = $start->parent;                                                       # Parent node
  confess "No parent" unless $p;                                                # Not possible on a root node
  my $q = $end->parent;                                                         # Parent node
  confess "No parent" unless $q;                                                # Not possible on a root node
  confess "Not siblings" unless $p == $q;                                       # Not possible unless the two nodes are siblings under the same parent
  my @c = $p->contents;                                                         # Content
  shift @c while @c and $c[ 0] != $start;                                       # Position on start node
  pop   @c while @c and $c[-1] != $end;                                         # Position on end
  if (@match)                                                                   # Select matching nodes if requested
   {my %m = map {$_=>1} @match;
    return grep {$m{$_->tag}} @c;
   }
  @c                                                                            # Elements in the specified range
 }

#D1 Location                                                                    # Locate the line numbers and columns of a specified node and write that information as a L<Oxygen Message|/https://www.oxygenxml.com/doc/versions/20.1/ug-author/topics/linked-output-messages-of-external-engine.html>.

sub parseLineLocation($)                                                        #PS Parse a line location
 {my ($loc) = @_;                                                               # Location
  my ($l, $c, $L, $C) = split m/[.:]/, $loc;                                    # Position of node in source

  for my $n($l, $c, $L)                                                         # Check that some-one else is not using xtrf for some other reason
   {return () unless $n and $n =~ m(\A\d+\Z)s;
   }
  return () if $L and $L !~ m(\A\d+\Z)s;

  unless(defined $C)                                                            # Same line
   {$C = $L;
    $L = 0;
   }
  $L += $l;                                                                     # Final line

 ($l, $c, $L, $C)                                                               # Return parsed line location
 }

sub lineLocation($)                                                             #U Return the line number.column location of this tag in its source file or string if the source was parsed with the L<line number|/lineNumber> option on.
 {my ($node) = @_;                                                              # Node
  my $loc  = $node->attr(q(xtrf));                                              # Location of node in source
  return q() unless $loc;                                                       # No location specified
  my @c = my ($l, $c, $L, $C) = parseLineLocation $loc;                         # Position of node in source
  for(@c)
   {return q() unless defined $_;
   }
  if ($l eq $L)                                                                 # All on one line
   {return qq(on line $l from $c to $C)                                         # Single line
   }
  qq(from line $l at $c to line $L at $C)                                       # Spans two or more lines
 }

sub location($;$)                                                               #U Return the line number.column plus file location of this tag in its source file or string if the source was parsed with the L<line number|/lineNumber> option on.
 {my ($node, $file) = @_;                                                       # Node, optionally the location of the source.
  my $lmsg   = $node->lineLocation();                                           # Line location
     $lmsg   = q( ).$lmsg if $lmsg;
  my $parser = $node->parser;                                                   # Parser associated with this node
  my $fmsg   = sub                                                              # Description of the containing file
   {return qq( in file: ).$file if $file;                                       # In the specified file
    return qq( in file: ).$parser->inputFile if $parser->inputFile;             # Position of node in source
    q()                                                                         # Unknown location
   }->();
  $lmsg.$fmsg                                                                   # Return location
 }

sub closestLocation($)                                                          #U Return the nearest node with line number.column information
 {my ($node) = @_;                                                              # Node
  return $node if $node->attr(q(xtrf));                                         # Node has a location so return it.
  my $best;                                                                     # Best location so far
  if ($node->parser->lineNumbers)                                               # Search through tree for closest node if line numbering is in effect
   {my $before = 1;                                                             # Best node before (after) the specified node
    for my $p($node->parser->downList)                                          # Preorder traversal
     {if ($before or !$best)                                                    # Before or at the node
       {$best = $p if $p->attr(q(xtrf));
       }
      $before = 0 if $p == $node and $before;                                   # Pass the node
     }
   }
  $best                                                                         # Return best node if known
 }

sub approxLocation($;$)                                                         #U Return the line number.column location of the node nearest to this node in the source file if the source was parsed with the L<line number|/lineNumber> option on.
 {my ($node, $file) = @_;                                                       # Node, optionally the location of the source.
  return location(@_) if $node->attr(q(xtrf));                                  # Node has a location so return it.

  my $best = $node->closestLocation;                                            # Search through tree for closest node if line numbering is in effect
  return location($best, $file) if $best;                                       # Nearby node has a location so return it.
  my $f = $file // $node->parser->inputFile;                                    # Parser input file if known
  return qq( in file: ).$f if $f;                                               # Position of node in source
  q()                                                                           # Unknown location
 }

sub formatOxygenMessage($$$@)                                                   #U Write an error message in Oxygen format
 {my ($node, $level, $url, @message) = @_;                                      # Node, error level [F|E|W], explanatory Url, message text
  my ($line, $col, $Line, $Col) = sub                                           # Position in Oxygen format
   {my $best = $node->closestLocation;                                          # Closest node with line number information
    return parseLineLocation $best->xtrfX if $best;                             # Return numbers
    (1,1,0,0)
   }->();
  my $m = nws(join '', @message);
  my $u = $url ? qq( $url) : q();
  <<END;                                                                        # To get this message to Oxygen simply write on STDOUT or STDERR
Type: $level
Line: $line
Column: $col
EndLine: $Line
EndColumn: $Col
AdditionalInfoURL:$u
Description: $m
END
 }

#D1 Position                                                                    # Confirm that the position L<navigated|/Navigation> to is the expected position.

sub atPositionMatch($$)                                                         #P Confirm that a string matches a match expression.
 {my ($tag, $match) = @_;                                                       # Starting node, ancestry.
  return 1 unless $match;                                                       # Undefined match means anything matches
  return $tag eq $match unless  ref $match;                                     # Match scalar
  return $tag =~ m($match)s if  ref($match) =~ m(regexp)i;                      # Match regular expression
  return $$match{$tag}      if  ref($match) =~ m(hash)i;                        # Match hash key
                            if (ref($match) =~ m(array)i)                       # Match array
   {my %m = map {$_=>1} @$tag;
    return $m{$tag}
   }
  confess "Unknown match type";                                                 # Do not know how to match
 }

sub at($@)                                                                      #CIYU Confirm that the specified B<$node> has the specified L<ancestry|/ancestry>. Ancestry is specified by providing the expected tags that the B<$node>'s parent, the parent's parent etc. must match at each level. If B<undef> is specified then any tag is assumed to match at that level. If a regular expression is specified then the current parent node tag must match the regular expression at that level. If all supplied tags match successfully then the starting node is returned else B<undef>.
 {my ($node, @context) = @_;                                                    # Node, ancestry.
  for(my $x = shift @_; $x; $x = $x->parent)                                    # Up through parents
   {return $node unless @_;                                                     # OK if no more required context
    next if atPositionMatch(-t $x, shift @_);                                   # Match tag against context
    return undef                                                                # Error if required does not match actual
   }
  !@_ ? $node : undef                                                           # Top of the tree is OK as long as there is no more required context
 }
#b <b/>
#b <c/>
#c at b
#c set id idb href bbb
#d Continue if we are in the specified context.

sub atText($$@)                                                                 #CU Confirm that we are on a text node whose text value matches a regular expression in the optional B<@context>. Return the specified B<$node> on success else B<undef>.
 {my ($node, $re, @context) = @_;                                               # Text node, regular expression to match, context
  return undef if !$node->isText(@context);                                     # Not a text node in the specified context
  $node->text =~ m($re) ? $node : undef                                         # Success if the text matches
 }
#b <b>bb</b><c>cc</c>
#c atText cc c
#c up
#c set id CC
#d Continue if we are on a text node whose value matches a regular expression

sub atStringContentMatches($$@)                                                 #CU Confirm that we are on a B<$node> whose contents, represented as a string, matches the specified regular expression B<$re> in the optional B<@context>. Return the specified B<$node> on success else B<undef>.
 {my ($node, $re, @context) = @_;                                               # Text node, regular expression to match, context
  return undef if @context and !$node->at(@context);                            # Check optional context
  $node->stringContent =~ m($re) ? $node : undef                                # Success if the content, as a string, matches the re
 }
#b <b><c>cc</c><d>dd</d></b>
#c atStringContentMatches qr(dd)
#c set id yes
#d Continue if we are on a node whose content represented as a string matches a regular expression

sub atTop($)                                                                    #U Return the current node if it is the root == top of a parse tree else return B<undef>.
 {my ($node) = @_;                                                              # Node
  return $node unless $node->parent;                                            # Has no parent so must be at the top
  undef                                                                         # Has a parent and so is not at the top
 }
#a at
#b <b/>
#c atTop
#c set  id top
#d Continue if we are at the top.

sub attrAt($$@)                                                                 #CU Return the specified B<$node> if it has the specified B<$attribute> and the $node is in the optional B<@context> else return B<undef>.
 {my ($node, $attribute, @context) = @_;                                        # Starting node, attribute, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  defined($node->attributes->{$attribute}) ? $node : undef                      # Node has attribute
 }
#a set
#b <b id='b'/><b id='bb'/>
#c attrValueAt id bb
#c set class here
#d Continue if an attribute has a specific value.

sub attrValueAt($$$@)                                                           #CU Return the specified B<$node> if it has the specified B<$attribute> with the specified B<$value> and the $node is in the optional B<@context> else return B<undef>.
 {my ($node, $attribute, $value, @context) = @_;                                # Starting node, attribute, wanted value of attribute, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $v = $node->attributes->{$attribute};                                      # Actual value of attribute
  if (defined($value) and defined($v))                                          # Compare attribute actual and wanted values
   {if (my $r = ref($value))                                                    # Compare with a reference
     {if ($r =~ m(regExp)is)                                                    # Regular expression
       {return $v =~ m($value) ? $node : undef;
       }
      elsif ($r =~ m(Hash)is)                                                   # Key of a hash
       {return $$value{$v}     ? $node : undef;
       }
      confess "Attribute value check expressed via unknown reference type";     # Unknown matching method
     }
    else                                                                        # Match on value
     {return $value eq $v      ? $node : undef;
     }
   }
  undef                                                                         # One of actual and wanted values was B<undef> so return B<undef>.
 }
#a set
#b <b id='b'/><b id='bb'/>
#c attrValueAt id bb
#c set class here
#d Continue if an attribute has a specific value.

sub not($@)                                                                     #U Return the specified B<$node> if it does not match any of the specified tags, else B<undef>
 {my ($node, @tags) = @_;                                                       # Node, tags not to match
  my %tags = map {$_=>1} @tags;                                                 # Tags not to match
  return $node unless $tags{$node->tag};                                        # Ok if node does not have one of the specified tags
  undef                                                                         # Matched one of the tags that it should not match
 }

sub atOrBelow($@)                                                               #CYU Confirm that the node or one of its ancestors has the specified context as recognized by L<at|/at> and return the first node that matches the context or B<undef> if none do.
 {my ($start, @context) = @_;                                                   # Starting node, ancestry.
  for(my $x = $start; $x; $x = $x->parent)                                      # Up through parents
   {return $x if $x->at(@context);                                              # Return this node if the context matches
   }
  undef                                                                         # No node that matches the context
 }

sub adjacent($$)                                                                # Return the first node if it is adjacent to the second node else B<undef>.
 {my ($first, $second) = @_;                                                    # First node, second node
  my ($n, $p) = ($first->next, $first->prev);                                   # Nodes adjacent to the first node
  return $first if $n && $n == $second or $p && $p == $second;                  # Adjacent nodes
  undef
 }

sub ancestry($)                                                                 #U Return a list containing: (the specified B<$node>, its parent, its parent's parent etc..). Or use L<upn|/upn> to go up the specified number of levels.
 {my ($node) = @_;                                                              # Starting node.
  my @a;
  for(my $x = $node; $x; $x = $x->parent)                                       # Up through parents
   {push @a, $x;
   }
  @a                                                                            # Return ancestry
 }

sub context($)                                                                  #bU Return a string containing the tag of the starting node and the tags of all its ancestors separated by single spaces.
 {my ($node) = @_;                                                              # Starting node.
  my @a;                                                                        # Ancestors
  for(my $p = $node; $p; $p = $p->parent)
   {push @a, $p->tag;
    @a < 100 or confess "Overly deep tree!";
   }
  join ' ', @a
 }
#a
#b <b> <c> <d context/> </c> </b>
#c
#d Get the context of node <b>d</b> by typing <b>context</b> or <b>gc</b> against its closing bracket and pressing <b>Save</b>.
BEGIN {*gc = *context}

sub containsSingleText($@)                                                      #CU Return the single text element below the specified B<$node> else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $t = $node->hasSingleChild;                                                # Child element
  $t ? $t->isText : $t                                                          # Child element must be text
 }

sub depth($)                                                                    #U Returns the depth of the specified B<$node>, the  depth of a root node is zero.
 {my ($node) = @_;                                                              # Node.
  my $a = 0;
  for(my $x = $node->parent; $x; $x = $x->parent) {++$a}                        # Up through parents
  $a                                                                            # Return depth
 }

sub depthProfile($)                                                             #U Returns the depth profile of the tree rooted at the specified B<$node>.
 {my ($node) = @_;                                                              # Node.
  my @d;                                                                        # Depth profile as an array
  $node->by(sub                                                                 # Each node
   {push @d, scalar @_;                                                         # Depth of node
   });
  return @d if wantarray;                                                       # Depth profile as an array
  join ' ', @d                                                                  # Depth profile as a string
 }

sub setDepthProfile($)                                                          #U Sets the L<depthProfile|/depthProfile> for every node in the specified B<$tree>. The last set L<depthProfile|/depthProfile> for a specific node can be retrieved from L<depthProfileLast|/depthProfileLast>.
 {my ($tree) = @_;                                                              # Tree of nodes.
  $tree->by(sub                                                                 # Each node
   {$_->depthProfileLast = $_->depthProfile;                                    # Set depth profile for node.
   });
  $tree
 }

sub height($)                                                                   #U Returns the height of the tree rooted at the specified B<$node>.
 {my ($node) = @_;                                                              # Node.
  my $h = 0;                                                                    # Height of tree so far
  $node->by(sub                                                                 # Each node
   {$h = scalar(@_) if scalar(@_) > $h;                                         # Highest height so far
   });
  $h                                                                            # Return height
 }

sub isFirst($@)                                                                 #BCYU Return the specified B<$node> if it is first under its parent and optionally has the specified context, else return B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent
  return $node unless defined($parent);                                         # The top most node is always first
  $node == $parent->first ? $node : undef                                       # First under parent
 }
#a isLast
#b <b/><c><d/></c><e/>
#c isFirst
#c set id first
#d Continue if we are at a first child.

sub isNotFirst($@)                                                              #CU Return the specified B<$node> if it is not first under its parent and optionally has the specified context, else return B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent
  return undef unless defined($parent);                                         # The top most node is always first
  $node != $parent->first ? $node : undef                                       # Not first under parent
 }
#a isNotFirst
#b <b/><c><d/></c><e/>
#c isNotFirst
#c set id notFirst
#d Continue if we are not at a first child.

sub isFirstN($$@)                                                               #CU Return the first B<$N> nodes as an array if the first B<$N> tags of the parent of B<$node> finish at the specified B<$node> and have the specified tags in the sequence specified by B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.
 {my ($node, $N, @context) = @_;                                                # Node, number of tags to be first, first tags and optional context
  return () unless my $parent = $node->parent;                                  # Must have a parent
  @context >= $N or confess "Not enough context";                               # Must have enough tags to match N
  my @tags; push @tags, shift @context for 1..$N;                               # Separate context into first n tags and parent context
  return () if @context and !$parent->at(@context);                             # Not in specified context
  my @nodes = $parent->contents;                                                # Parent content
  pop @nodes while @nodes and $nodes[-1] != $node;                              # Remove following nodes
  return () unless @nodes == $N;                                                # Nodes are not first
  for(1..$N)                                                                    # Confirm tags match
   {next if $nodes[-$_]->tag eq $tags[-$_];                                     # Confirm tag matches
    return ()                                                                   # Return empty list as tags failed to match
   }
  @nodes                                                                        # Return matching nodes on success
 }

sub isFirstToDepth($$@)                                                         #CU Return the specified B<$node> if it is first to the specified depth else return B<undef>
 {my ($node, $depth, @context) = @_;                                            # Node, depth, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node;                                                                # Start
  for(1..$depth-1)                                                              # Check each ancestor is first to the specified depth but one
   {return undef unless $p->isFirst and $p = $p->parent;                        # Check node is first and that we can move up
   }
  $p->isFirst                                                                   # Confirm that the last ancestor so reached is a first node
 }

sub firstIs($@)                                                                 #CU Return the specified B<$node> if it has one or more child nodes and the first child node has the specified B<@context> otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $first = $node->first;                                 # Must have a first child
  return undef if @context and !$first->at(@context);                           # First child must match any supplied context
  $node                                                                         # Success
 }
#a firstIs
#b <b><c><d/><e/></c></b>
#c firstIs d
#c set id yes
#d Continue if there is a first child with the specified context

sub isLast($@)                                                                  #BCYU Return the specified B<$node> if it is last under its parent and optionally has the specified context, else return B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent
  return $node unless defined($parent);                                         # The top most node is always last
  $node == $parent->last ? $node : undef                                        # Last under parent
 }
#a isFirst
#b <b/><c><d/></c><e/>
#c isLast
#c set id last
#d Continue if we are at a last child.

sub isNotLast($@)                                                               #CU Return the specified B<$node> if it is not last under its parent and optionally has the specified context, else return B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent
  return undef unless defined($parent);                                         # The top most node is always last
  $node != $parent->last ? $node : undef                                        # Not last under parent
 }
#a isLast isNotFirst
#b <b/><c><d/></c><e/>
#c isNotLast
#c set id notLast
#d Continue if we are not at a last child.

sub isLastN($$@)                                                                #CU Return the last B<$N> nodes as an array if the last B<$N> tags of the parent of B<$node> start at the specified B<$node> and have the specified tags in the sequence specified by B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.
 {my ($node, $N, @context) = @_;                                                # Node, number of tags to be last, last tags and optional context
  return () unless my $parent = $node->parent;                                  # Must have a parent
  @context >= $N or confess "Not enough context";                               # Must have enough tags to match N
  my @tags; push @tags, shift @context for 1..$N;                               # Separate context into last n tags and parent context
  return () if @context and !$parent->at(@context);                             # Not in specified context
  my @nodes = $parent->contents;                                                # Parent content
  shift @nodes while @nodes and $nodes[0] != $node;                             # Remove preceding nodes
  return () unless @nodes == $N;                                                # Nodes are not last
  for(1..$N)                                                                    # Confirm tags match
   {next if $nodes[-$_]->tag eq $tags[-$_];                                     # Confirm tag matches
    return ()                                                                   # Return empty list as tags failed to match
   }
  @nodes                                                                        # Return matching nodes on success
 }

sub isLastToDepth($$@)                                                          #CU Return the specified B<$node> if it is last to the specified depth else return B<undef>
 {my ($node, $depth, @context) = @_;                                            # Node, depth, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node;                                                                # Start
  for(1..$depth-1)                                                              # Check each ancestor is last to the specified depth but one
   {return undef unless $p->isLast and $p = $p->parent;                         # Check node is last and that we can move up
   }
  $p->isLast ? $node : undef                                                    # Confirm that the last ancestor so reached is a last node
 }

sub lastIs($@)                                                                  #CU Return the specified B<$node> if it has one or more child nodes and the last child node has the specified B<@context> otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $last = $node->last;                                   # Must have a last child
  return undef if @context and !$last->at(@context);                            # Last child must match any supplied context
  $node                                                                         # Success
 }
#a lastIs
#b <b><c><d/><e/></c></b>
#c lastIs e
#c set id yes
#d Continue if there is a last child with the specified context

sub nextIs($@)                                                                  #CU Return the specified B<$node> if there is a following node with the specified B<@context>. Returns B<undef> if the $node is last.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $next = $node->next;                                   # Must have a following node
  return undef if @context and !$next->at(@context);                            # Following node must match any supplied context
  $node                                                                         # Success
 }
#a nextIs
#b <b/><c/><d/><e/>
#c nextIs d
#c set id yes
#d Continue if the following node has the expected context

sub nextN($$@)                                                                  #CU Return B<$N> nodes as an array starting at B<$node> inclusive if they match the first B<$N> tags of B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.
 {my ($node, $N, @context) = @_;                                                # Node, number of tags to be last, last tags and optional context
  return () unless my $parent = $node->parent;                                  # Must have a parent
  @context >= $N or confess "Not enough context";                               # Must have enough tags to match N
  my @tags; push @tags, shift @context for 1..$N;                               # Separate context into last n tags and parent context
  return () if @context and !$parent->at(@context);                             # Not in specified context
  my @nodes = $parent->contents;                                                # Parent content
  shift @nodes while @nodes and $nodes[0] != $node;                             # Remove preceding nodes
  return () unless @nodes >= $N;                                                # Enough nodes to match
  pop @nodes while @nodes and @nodes > $N;                                      # Nodes requested
  for(1..$N)                                                                    # Confirm tags match
   {next if $nodes[-$_]->tag eq $tags[-$_];                                     # Confirm tag matches
    return ()                                                                   # Return empty list as tags failed to match
   }
  @nodes                                                                        # Return matching nodes on success
 }

sub prevIs($@)                                                                  #CU Return the specified B<$node> if there is a previous node with the specified B<@context>. Returns B<undef> if the $node is first.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $prev = $node->prev;                                   # Must have a previous node
  return undef if @context and !$prev->at(@context);                            # Previous node must match any supplied context
  $node                                                                         # Success
 }
#a prevIs
#b <b/><c/><d/><e/>
#c prevIs c
#c set id yes
#d Continue if the previous node has the expected context

sub prevN($$@)                                                                  #CU Return B<$N> nodes as an array ending at B<$node> inclusive if they match the first B<$N> tags of B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.
 {my ($node, $N, @context) = @_;                                                # Node, number of tags to be first, first tags and optional context
  return () unless my $parent = $node->parent;                                  # Must have a parent
  @context >= $N or confess "Not enough context";                               # Must have enough tags to match N
  my @tags; push @tags, shift @context for 1..$N;                               # Separate context into first n tags and parent context
  return () if @context and !$parent->at(@context);                             # Not in specified context
  my @nodes = $parent->contents;                                                # Parent content
  pop @nodes while @nodes and $nodes[-1] != $node;                              # Remove following nodes
  return () unless @nodes >= $N;                                                # Nodes are not first
  shift @nodes while @nodes and @nodes > $N;                                    # Nodes requested
  for(1..$N)                                                                    # Confirm tags match
   {next if $nodes[-$_]->tag eq $tags[-$_];                                     # Confirm tag matches
    return ()                                                                   # Return empty list as tags failed to match
   }
  @nodes                                                                        # Return matching nodes on success
 }

sub isOnlyChild($@)                                                             #CYU Return the specified B<$node> if it is the only node under its parent ignoring any surrounding blank text.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Find parent
  return $node unless $parent;                                                  # The root node is an only child
  my @c = $parent->contents;                                                    # Contents of parent
  return $node if @c == 1;                                                      # Only child if only one child
  shift @c while @c and $c[ 0]->isBlankText;                                    # Ignore leading blank text
  pop   @c while @c and $c[-1]->isBlankText;                                    # Ignore trailing blank text
  return $node if @c == 1;                                                      # Only child if only one child after leading and trailing blank text has been ignored
  undef                                                                         # Not the only child
 }
#a isOnlyChild
#b <b><c/></b><d><e/><e/></d>
#c isOnlyChild
#c set id onlyChild
#d Continue if we are at an only child.

sub isOnlyChildToDepth($$@)                                                     #CU Return the specified B<$node> if it and its ancestors are L<only children|/isOnlyChild> to the specified depth else return B<undef>. L<isOnlyChildToDepth(1)|/isOnlyChildToDepth> is the same as L<isOnlychild|/isOnlyChild>
 {my ($node, $depth, @context) = @_;                                            # Node, depth to which each parent node must also be an only child, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node;                                                                # Walk up through ancestors
  for(1..$depth-1)                                                              # Walk up through ancestors to the specified depth but one
   {return undef unless $p->isOnlyChild and $p = $p->parent;                    # Confirm we are an only child and can move up one more level
   }
  $p->isOnlyChild;                                                              # Confirm that we are still on an only child
 }

sub isOnlyChildN($$@)                                                           #CU Return the specified B<$node> if it is an only child of B<$depth> ancestors with L<only children|/hasSingleChild> in the optional B<@context> else return B<undef>. L<isOnlyChildN(1)|/isOnlyChildN> is the same as L<isOnlychild|/isOnlyChild>.
 {my ($node, $depth, @context) = @_;                                            # Node, depth to which each parent node must also be an only child, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node;                                                                # Walk up through ancestors
  for(1..$depth-1)                                                              # Walk up through ancestors to the specified depth but one
   {return undef unless $p->isOnlyChild and $p = $p->parent;                    # Confirm we are an only child and can move up one more level
   }
  $p->isOnlyChild ? $node : undef                                               # Confirm that we are still on an only child
 }

sub isOnlyChildText($@)                                                         #CU Return the specified B<$node> if it is a text node and it is an only child else and its parent is in the specified optional context else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  if (my $parent = $node->parent)                                               # Parent required
   {return undef if @context and !$parent->at(@context);                        # Not in specified context
    return $node if $node->isText and $node->isOnlyChild;                       # Confirm that the node is a text node and an only child
   }
  undef                                                                         # No parent so cannot be an only child
 }

sub hasSingleChild($@)                                                          #CYU Return the only child of the specified B<$node> if the child is the only node under its parent ignoring any surrounding blank text and has the optional specified context, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $child = $node->first(@context);                       # A possible child node
  return undef unless $child->isOnlyChild;                                      # Not an only child
  $child                                                                        # Return the only child
 }
#a hasSingleChild
#b <b><c/></b><d><e/><e/></d>
#c hasSingleChild
#c set id hasSingleChild
#d Continue if we have a single child.

sub hasSingleChildText($@)                                                      #CU Return the only child of the specified B<$node> if the child is a text node and the child is the only node under its parent ignoring any surrounding blank text and the child has the optional specified context, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $child = $node->hasSingleChild(@context);              # A possible single child node
  return undef unless $child->isText;                                           # Not a text node
  $child                                                                        # Return the only text child
 }

sub hasSingleChildToDepth($$@)                                                  #CU Return the descendant of the specified B<$node> if it has single children to the specified depth in the specified optional B<@context> else return B<undef>.  L<hasSingleChildToDepth(0)|/hasSingleChildToDepth> is equivalent to L<hasSingleChild|/hasSingleChild>.
 {my ($node, $depth, @context) = @_;                                            # Node, depth, optional context
  return $node->isOnlyChild if $depth < 1;                                      # Validate depth
  my $c = 0;                                                                    # Depth count
  my $p;                                                                        # Current position
  for
   ($p = $node->first;                                                          # Descend firstly
    $p and $p->isOnlyChild and ++$c < $depth;
    $p = $p->first)
   {}
  $p ? $p->isOnlyChild(@context) : $p                                           # Check the context of the descendant single child
 }

sub isEmpty($@)                                                                 #CYU Confirm that the specified B<$node> is empty, that is: the specified B<$node> has no content, not even a blank string of text. To test for blank nodes, see L<isAllBlankText|/isAllBlankText>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  !$node->first ? $node : undef                                                 # If it has no first descendant it must be empty
 }
#a cutIfEmpty
#b <b><c/></b>
#c isEmpty
#c set id empty
#d Continue if we have no children.

sub hasContent($@)                                                              #CYU Confirm that the specified B<$node> has content. Return the specified node if it has content else return B<undef> if it does not.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->first ? $node : undef                                                  # If it has a first descendant it has content
 }
#a hasContent
#b <b><c/></b>
#c hasContent
#c set id yes
#d Continue if we have children.

sub over($$@)                                                                   #CYU Confirm that the string representing the tags at the level below the specified B<$node> match a regular expression where each pair of tags is separated by a single space. Use L<contentAsTags|/contentAsTags> to visualize the tags at the next level.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentAsTags =~ m/$re/ ? $node : undef
 }

sub over2($$@)                                                                  #CYU Confirm that the string representing the tags at the level below the specified B<$node> match a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q? . Use L<contentAsTags2|/contentAsTags2> to visualize the tags at the next level.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentAsTags2 =~ m/$re/ ? $node : undef
 }
#a over
#b <b><c/><c/></b><b><c/></b>
#c over2 \A(\sc\s){2}\Z
#c set id over2
#d Continue if our children's tag match a regular expression.

sub overAllTags($@)                                                             #U Return the specified b<$node> if all of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.
 {my ($node, @tags) = @_;                                                       # Node, tags.
  my @c = $node->contents;                                                      # Node contents
  while(@tags and @c)                                                           # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @tags);              # Continue unless we fail to match
   }
  return $node if @c == 0 and @tags == 0;                                       # Child nodes and tags matched exactly
  undef                                                                         # Wrong number of tags
 }

BEGIN{*oat=*overAllTags}

sub overFirstTags($@)                                                           #U Return the specified b<$node> if the first of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.
 {my ($node, @tags) = @_;                                                       # Node, tags.
  my @c = $node->contents;                                                      # Node contents
  while(@tags and @c)                                                           # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @tags);              # Continue unless we fail to match
   }
  return $node if @c >= 0 and @tags == 0;                                       # The first child nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

BEGIN{*oft=*overFirstTags}

sub overLastTags($@)                                                            #U Return the specified b<$node> if the last of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.
 {my ($node, @tags) = @_;                                                       # Node, tags.
  my @c = $node->contents;                                                      # Node contents
  while(@tags and @c)                                                           # Match node contents against tags
   {return undef unless atPositionMatch(-t pop @c, pop @tags);                  # Continue unless we fail to match
   }
  return $node if @c >= 0 and @tags == 0;                                       # The last child nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

BEGIN{*olt=*overLastTags}

sub matchAfter($$@)                                                             #CY Confirm that the string representing the tags following the specified B<$node> matches a regular expression where each pair of tags is separated by a single space. Use L<contentAfterAsTags|/contentAfterAsTags> to visualize these tags.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentAfterAsTags =~ m/$re/ ? $node : undef
 }

sub matchAfter2($$@)                                                            #CY Confirm that the string representing the tags following the specified B<$node> matches a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q? Use L<contentAfterAsTags2|/contentAfterAsTags2> to visualize these tags.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentAfterAsTags2 =~ m/$re/ ? $node : undef
 }

sub matchBefore($$@)                                                            #CY Confirm that the string representing the tags preceding the specified B<$node> matches a regular expression where each pair of tags is separated by a single space. Use L<contentBeforeAsTags|/contentBeforeAsTags> to visualize these tags.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentBeforeAsTags =~ m/$re/ ? $node : undef
 }

sub matchBefore2($$@)                                                           #CY Confirm that the string representing the tags preceding the specified B<$node> matches a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q?  Use L<contentBeforeAsTags2|/contentBeforeAsTags2> to visualize these tags.
 {my ($node, $re, @context) = @_;                                               # Node, regular expression, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->contentBeforeAsTags2 =~ m/$re/ ? $node : undef
 }

sub parentage($)                                                                #U Return a reference to an array of the nodes along the path from the root to the specified B<$Node> inclusive.
 {my ($node) = @_;                                                              # Node.
  my @p;                                                                        # Path
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {push @p, $p;                                                                # Save position
   }
  [reverse @p]                                                                  # Return path from root
 }

BEGIN{*p=*parentage}

sub path($)                                                                     #U Return a list of strings representing the path to a node from the root of the parse tree which can then be reused by L<go|/go> to retrieve the node as long as the structure of the L<parse|/parse> tree has not changed along the path.
 {my ($node) = @_;                                                              # Node.
  my @p;                                                                        # Path
  for(my $p = $node; $p and $p->parent; $p = $p->parent)                        # Go up
   {my $i = $p->index;                                                          # Position in parent index
    push @p, $i if $i;                                                          # Save position unless default
    push @p, $p->tag;                                                           # Save index
   }
  reverse @p                                                                    # Return path from root
 }

sub pathString($)                                                               #bU Return a string representing the L<path|/path> to the specified B<$node> from the root of the parse tree.
 {my ($node) = @_;                                                              # Node.
  join ' ', path($node)                                                         # String representation
 }

#D2 Match                                                                       # Locate adjacent nodes that match horizontally and vertically

sub an($$@)                                                                     #CU Return the next node if the specified B<$node> has the tag specified by B<$current> and the next node is in the specified B<@context>.
 {my ($node, $current, @context) = @_;                                          # Node, tag node must match, optional context of the next node.
  return undef unless $node->at($current);                                      # Check node has the right tag
  $node->next(@context)                                                         # Next node if it matches the context else B<undef>
 }

sub ap($$@)                                                                     #CU Return the previous node if the specified B<$node> has the tag specified by B<$current> and the previous node is in the specified B<@context>.
 {my ($node, $current, @context) = @_;                                          # Node, tag node must match, optional context of the previous node.
  return undef unless $node->at($current);                                      # Check node has the right tag
  $node->prev(@context)                                                         # Previous node if it matches the context else B<undef>
 }

sub apn($$$@)                                                                   #KU Return (previous node, next node) if the B<$previous> and B<$current> nodes have the specified tags and the next node is in the specified B<@context> else return B<()>.  The specified B<@context> must have at least one element otherwise B<()> is returned.
 {my ($node, $prev, $current, @context) = @_;                                   # Current node, tag for the previous node, tag for specified node, context for the next node.
  return () if !@context or !$node->at($current) or                             # Check context
                $node->isLast or $node->isFirst;                                # Check existence of surrounding nodes
  my $p = $node->prev($prev);                                                   # Previous node
  my $n = $node->next(@context);                                                # Next node
  return ($p, $n) if $p and $n;                                                 # Successful match
  ()                                                                            # Match failed
 }

sub matchesFirst($@)                                                            #U Return the specified B<$node> if its children L<match|/atPositionMatch> the specified <@sequence> forwards from the first child else return B<undef>.
 {my ($node, @sequence) = @_;                                                   # Node, sequence.
  my @c = $node->contents;                                                      # Child nodes
  while(@sequence and @c)                                                       # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @sequence);          # Continue unless we fail to match
   }
  return $node unless @sequence;                                                # The following nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

sub matchesLast($@)                                                             #U Return the specified B<$node> if its children L<match|/atPositionMatch> the specified <@sequence> backwards from the last child else return B<undef>.
 {my ($node, @sequence) = @_;                                                   # Node, sequence.
  my @c = reverse $node->contents;                                              # Child nodes
  while(@sequence and @c)                                                       # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @sequence);          # Continue unless we fail to match
   }
  return $node unless @sequence;                                                # The following nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

sub matchesNext($@)                                                             #U Return the specified B<$node> if its following siblings L<match|/atPositionMatch> the specified <@sequence> else return B<undef>.
 {my ($node, @sequence) = @_;                                                   # Node, sequence.
  my @c = $node->contentAfter;                                                  # Following node
  while(@sequence and @c)                                                       # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @sequence);          # Continue unless we fail to match
   }
  return $node unless @sequence;                                                # The following nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

sub matchesPrev($@)                                                             #U Return the specified B<$node> if the siblings before $node L<match|/atPositionMatch> the specified <@sequence> with the first element of @sequence nearest to $node and the last element furthest else return B<undef>.
 {my ($node, @sequence) = @_;                                                   # Node, sequence.
  my @c = reverse $node->contentBefore;                                         # Prior nodes
  while(@sequence and @c)                                                       # Match node contents against tags
   {return undef unless atPositionMatch(-t shift @c, shift @sequence);          # Continue unless we fail to match
   }
  return $node unless @sequence;                                                # The prior nodes match the specified tags
  undef                                                                         # Wrong number of tags
 }

#D2 Child of, Parent of, Sibling of                                             # Nodes that are directly above, below or adjacent to another node.

sub parentOf($$@)                                                               #C Returns the specified B<$parent> node if it is the parent of the specified B<$child> node and the B<$parent> node is in the specified optional context.
 {my ($parent, $child, @context) = @_;                                          # Parent, child, optional context
  return undef if @context and !$parent->at(@context);                          # Check context
  return $parent if $child->parent == $parent;                                  # Check child has the parent as its parent
  undef                                                                         # Wrong parent
 }

sub childOf($$@)                                                                #C Returns the specified B<$child> node if it is a child of the specified B<$parent> node and the B<$child> node is in the specified optional context.
 {my ($child, $parent, @context) = @_;                                          # Child, parent, optional context
  return undef if @context and !$child->at(@context);                           # Check context
  return $child if $child->parent == $parent;                                   # Check child has the parent as its parent
  undef                                                                         # Wrong parent
 }

sub succeedingSiblingOf($$@)                                                    #C Returns the specified B<$child> node if it has the same parent as B<$sibling> and occurs after B<$sibling> and has the optionally specified context else returns B<undef>.
 {my ($child, $sibling, @context) = @_;                                         # Child, sibling thought to occur before child, optional context
  return undef if @context and !child->at(@context);                            # Check context
  return undef unless $child->parent == $sibling->parent;                       # Check child has the parent as its prior sibling
  $child->after($sibling);                                                      # Check child occurs after prior sibling
 }

sub precedingSiblingOf($$@)                                                     #C Returns the specified B<$child> node if it has the same parent as B<$sibling> and occurs before B<$sibling> and has the optionally specified context else returns B<undef>.
 {my ($child, $sibling, @context) = @_;                                         # Child, sibling thought to occur after child, optional context
  return undef if @context and !child->at(@context);                            # Check context
  return undef unless $child->parent == $sibling->parent;                       # Check child has the parent as its prior sibling
  $child->before($sibling);                                                     # Check child occurs after prior sibling
 }

#D1 Navigation                                                                  # Move around in the L<parse|/parse> tree.

sub go($@)                                                                      #IYU Return the node reached from the specified B<$node> via the specified L<path|/path>: (index positionB<?>)B<*> where index is the tag of the next node to be chosen and position is the optional zero based position within the index of those tags under the current node. Position defaults to zero if not specified. Position can also be negative to index back from the top of the index array. B<*> can be used as the last position to retrieve all nodes with the final tag.
 {my ($node, @path) = @_;                                                       # Node, search specification.
  my $p = $node;                                                                # Current node
  while(@path)                                                                  # Position specification
   {my $i = shift @path;                                                        # Index name
    return undef unless $p;                                                     # There is no node of the named type under this node
    reindexNode($p);                                                            # Create index for this node
    my $q = $p->indexes->{$i};                                                  # Index
    return undef unless defined $q;                                             # Complain if no such index
    if (@path)                                                                  # Position within index
     {if ($path[0] =~ /\A([-+]?\d+)\Z/)                                         # Numeric position in index from start
       {shift @path;
        $p = $q->[$1]
       }
      elsif (@path == 1 and $path[0] =~ /\A\*\Z/)                               # Final index wanted
       {return @$q;
       }
      else {$p = $q->[0]}                                                       # Step into first sub node by default
     }
    else {$p = $q->[0]}                                                         # Step into first sub node by default on last step
   }
  $p
 }
#a up
#b <b><c/><c><d/><d/><d/><d/></c><c/></b>
#c go c 1 d 2
#c set id arrived_here
#d Follow a path from the current node.
sub c($$)                                                                       #U Return an array of all the nodes with the specified tag below the specified B<$node>.
 {my ($node, $tag) = @_;                                                        # Node, tag.
  reindexNode($node);                                                           # Create index for this node
  my $c = $node->indexes->{$tag};                                               # Index for specified tags
  $c ? @$c : ()                                                                 # Contents as an array
 }

sub cText($)                                                                    #U Return an array of all the text nodes immediately below the specified B<$node>.
 {my ($node) = @_;                                                              # Node.
  reindexNode($node);                                                           # Create index for this node
  $node->c(cdata);                                                              # Index for text data
 }

sub findById($$)                                                                #U Find a node in the parse tree under the specified B<$node> with the specified B<$id>.
 {my ($node, $id) = @_;                                                         # Parse tree, id desired.
  my $i;                                                                        # Node found
  eval {$node->by(sub                                                           # Look for an instance of such a node
   {if ($_->idX eq $id) {$i = $_; die}                                          # Found the node - die to stop the search from going further
   })};
  $i                                                                            # Node found if any
 }

sub matchesNode($$@)                                                            # Return the B<$first> node if it matches the B<$second> node's tag and the specified B<@attributes> else return B<undef>.
 {my ($first, $second, @attributes) = @_;                                       # First node, second node, attributes to match on
  return undef unless -t $first eq -t $second;                                  # Check tags match
  my $f = $first->attributes;                                                   # Attributes for first node
  my $s = $second->attributes;                                                  # Attributes for second node
  for my $a(@attributes)
   {return undef unless defined($f->{$a}) and defined($s->{$a}) and
                                $f->{$a}  eq          $s->{$a};
   }
  $first                                                                        # Nodes match on specified attributes
 }

sub matchesSubTree($$@)                                                         # Return the B<$first> node if it L<matches|/matchesNode> the B<$second> node and the nodes under the first node match the corresponding nodes under the second node, else return B<undef>.
 {my ($first, $second, @attributes) = @_;                                       # First node, second node, attributes to match on
  return undef unless &matchesNode(@_);                                         # Check nodes match
  my @f = @$first;                                                              # Children for first node
  my @s = @$second;                                                             # Children for second node
  return undef unless @f == @s;                                                 # Wrong number of children
  while(@f)                                                                     # Match each child
   {return undef unless (shift @f)->matchesNode(shift @s, @attributes);         # Children match
   }
  $first                                                                        # Sub trees match
 }

sub findMatchingSubTrees($$@)                                                   # Find nodes in the parse tree whose sub tree matches the specified B<$subTree> excluding any of the specified B<$attributes>.
 {my ($node, $subTree, @attributes) = @_;                                       # Parse tree, parse tree to match, attributes to match on
  my @i;                                                                        # Node found
  my $t = -t $subTree;                                                          # Quick reject
  $node->by(sub                                                                 # Each node in the tree
   {return unless -t $_ eq $t;                                                  # Quick reject
    push @i, $_ if $_->matchesSubTree($subTree, @attributes);                   # Found a matching sub tree
   });
  @i                                                                            # Node found if any
 }

#D2 First                                                                       # Find nodes that are first amongst their siblings.

sub first($@)                                                                   #BCYU Return the first node below the specified B<$node> optionally checking the first node's context.  See L<addFirst|/addFirst> to ensure that an expected node is in position.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return $node->content->[0] unless @context;                                   # Return first node if no context specified
  my ($c) = $node->contents;                                                    # First node
  $c ? $c->at(@context) : undef;                                                # Return first node if in specified context
 }
#a
#b <b><c/></b><b><d/></b>
#c   at b
#c   first c
#c   set id firstChild
#d Go to our first child optionally checking its context.

sub firstn($$@)                                                                 #CU Return the B<$n>'th first node below the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<firstn(1)> is identical in effect to L<first|/first>.
 {my ($node, $N, @context) = @_;                                                # Node, number of times to go first, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(1..$N)                                                                    # Go first the specified number of times
   {$node = $node->first;                                                       # Go first
    last unless $node;                                                          # Cannot go further
   }
  $node
 }

sub firstText($@)                                                               #CU Return the first node under the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $l = &first($node);                                                        # First node
  $l ? $l->isText : undef                                                       # Test whether the first node exists and is a text node
 }

sub firstTextMatches($$@)                                                       #CU Return the first node under the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.
 {my ($node, $match, @context) = @_;                                            # Node, regular expression the text must match, optional context of specified node.
  return undef if @context and !$node->at(@context);                            # Check context
  if (my $t = $node->firstText)                                                 # First node is text
   {return $t->matchesText($match);                                             # First text node matches the specified regular expression
   }
  undef                                                                         # First node is not text or does not match the specified regular expression
 }

sub firstBy($@)                                                                 #U Return a list of the first instance of each specified tag encountered in a post-order traversal from the specified B<$node> or a hash of all first instances if no tags are specified.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags;                                                                     # Tags found first
  $node->byReverse(sub {$tags{$_->tag} = $_});                                  # Save first instance of each node
  return %tags unless @tags;                                                    # Return hash of all tags encountered first unless @tags filter was specified
  map {$tags{$_}} @tags;                                                        # Nodes in the requested order
 }

sub firstDown($@)                                                               #U Return a list of the first instance of each specified tag encountered in a pre-order traversal from the specified B<$node> or a hash of all first instances if no tags are specified.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags;                                                                     # Tags found first
  $node->downReverse(sub {$tags{$_->tag} = $_});                                # Save first instance of each node
  return %tags unless @tags;                                                    # Return hash of all tags encountered first unless @tags filter was specified
  map {$tags{$_}} @tags;                                                        # Nodes in the requested order
 }

sub firstIn($@)                                                                 #YU Return the first child node matching one of the named tags under the specified parent node.
 {my ($node, @tags) = @_;                                                       # Parent node, child tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for($node->contents)                                                          # Search forwards through contents
   {return $_ if $tags{$_->tag};                                                # Find first tag with the specified name
   }
  return undef                                                                  # No such node
 }

sub firstNot($@)                                                                #U Return the first child node that does not match any of the named B<@tags> under the specified parent B<$node>. Return B<undef> if there is no such child node.
 {my ($node, @tags) = @_;                                                       # Parent node, child tags to avoid.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for($node->contents)                                                          # Search forwards through contents
   {return $_ unless $tags{$_->tag};                                            # Find first tag that fails to match
   }
  return undef                                                                  # No such node
 }

sub firstInIndex($@)                                                            #CYU Return the specified B<$node> if it is first in its index and optionally L<at|/at> the specified context else B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $parent = $node->parent;                                                   # Parent
  return undef unless $parent;                                                  # The root node is not first in anything
  my @c = $parent->c($node->tag);                                               # Index containing node
  @c && $c[0] == $node ? $node : undef                                          # First in index ?
 }

sub firstOf($@)                                                                 #U Return an array of the nodes that are continuously first under their specified parent node and that match the specified list of tags.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my @l;                                                                        # Matching last nodes
  for($node->contents)                                                          # Search through contents
   {return @l unless $tags{$_->tag};                                            # Nonmatching tag
    push @l, $_;                                                                # Save continuously matching tag in correct order
   }
  return @l                                                                     # All tags match
 }

sub firstWhile($@)                                                              #U Go first from the specified B<$node> and continue deeper firstly as long as each first child node matches one of the specified B<@tags>. Return the deepest such node encountered or else return B<undef> if no such node is encountered.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my $p;                                                                        # Current position
  for(my $f = $node->first; $f and $tags{-t $f}; $f = $f->first) {$p = $f}      # Go ever firstly
  $p
 }

sub firstUntil($@)                                                              #CU Go first from the specified B<$node> and continue deeper firstly until a first child node matches the specified B<@context> or return B<undef> if there is no such node.  Return the first child of the specified B<$node> if no B<@context> is specified.
 {my ($node, @context) = @_;                                                    # Node, context to search for.
  for(my $p = $node->first; $p; $p = $p->first)                                 # Check each first child node below the B<$node>
   {return $p if $p->at(@context);                                              # Return the node if it matches the specified context
   }
  undef
 }

sub firstUntilText($@)                                                          #CU Go first from the specified B<$node> and continue deeper firstly until a text node is encountered whose parent matches the specified B<@context> or return B<undef> if there is no such node.
 {my ($node, @context) = @_;                                                    # Node, context to search for.
  for(my $p = $node->first; $p; $p = $p->first)                                 # Check each first child node below the B<$node>
   {return $p if $p->isText and $p->parent->at(@context);                       # Return the node if it is text and its parent matches the specified context
   }
  undef
 }

sub firstContextOf($@)                                                          #CYU Return the first node encountered in the specified context in a depth first post-order traversal of the L<parse|/parse> tree.
 {my ($node, @context) = @_;                                                    # Node, array of tags specifying context.
  my $x;                                                                        # Found node if found
  eval                                                                          # Trap the die which signals success
   {$node->by(sub                                                               # Traverse  L<parse|/parse> tree in depth first order
     {my ($o) = @_;
      if ($o->at(@context))                                                     # Does this node match the supplied context?
       {$x = $o;                                                                # Success
        die "success!";                                                         # Halt the search
       }
     });
   };
  confess $@ if $@ and  $@ !~ /success!/;                                       # Report any suppressed error messages at this point
  $x                                                                            # Return node found if we are still alive
 }

sub firstSibling($@)                                                            #CYU Return the first sibling of the specified B<$node> in the optional B<@context> else B<undef>
 {my ($node, @context) = @_;                                                    # Node, array of tags specifying context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node->parent;                                                        # Parent node
  $p->first                                                                     # Return first sibling
 }

#D2 Last                                                                        # Find nodes that are last amongst their siblings.

sub last($@)                                                                    #BCYU Return the last node below the specified B<$node> optionally checking the last node's context. See L<addLast|/addLast> to ensure that an expected node is in position.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return $node->content->[-1] unless @context;                                  # Return last node if no context specified
  my ($c) = reverse $node->contents;                                            # Last node
  $c ? $c->at(@context) : undef;                                                # Return last node if in specified context
 }
#a
#b <b><c/><d/></b><b><c/></b>
#c   at b
#c   last d
#c   set id lastChild
#d Go to our last child optionally checking its context.

sub lastn($$@)                                                                  #CU Return the B<$n>'th last node below the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<lastn(1)> is identical in effect to L<last|/last>.
 {my ($node, $N, @context) = @_;                                                # Node, number of times to go last, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(1..$N)                                                                    # Go last the specified number of times
   {$node = $node->last;                                                        # Go last
    last unless $node;                                                          # Cannot go further
   }
  $node
 }

sub lastText($@)                                                                #CU Return the last node under the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $l = &last($node);                                                         # Last node
  $l ? $l->isText : undef                                                       # Test whether the first node exists and is a text node
 }

sub lastTextMatches($$@)                                                        #CU Return the last node under the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.
 {my ($node, $match, @context) = @_;                                            # Node, regular expression the text must match, optional context of specified  node.
  return undef if @context and !$node->at(@context);                            # Check context
  if (my $t = $node->lastText)                                                  # Last node is text
   {return $t->matchesText($match);                                             # Last text node matches the specified regular expression
   }
  undef                                                                         # Last node is not text or does not match the specified regular expression
 }

sub lastBy($@)                                                                  #U Return a list of the last instance of each specified tag encountered in a post-order traversal from the specified B<$node> or a hash of all last instances if no tags are specified.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags;                                                                     # Tags found first
  $node->by(sub {$tags{$_->tag} = $_});                                         # Save last instance of each node
  return %tags unless @tags;                                                    # Return hash of all tags encountered last unless @tags filter was specified
  map {$tags{$_}} @tags;                                                        # Nodes in the requested order
 }

sub lastDown($@)                                                                #U Return a list of the last instance of each specified tag encountered in a pre-order traversal from the specified B<$node> or a hash of all last instances if no tags are specified.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags;                                                                     # Tags found first
  $node->down(sub {$tags{$_->tag} = $_});                                       # Save last instance of each node
  return %tags unless @tags;                                                    # Return hash of all tags encountered last unless @tags filter was specified
  map {$tags{$_}} @tags;                                                        # Nodes in the requested order
 }

sub lastIn($@)                                                                  #YU Return the last child node matching one of the named tags under the specified parent node.
 {my ($node, @tags) = @_;                                                       # Parent node, child tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for(reverse $node->contents)                                                  # Search backwards through contents
   {return $_ if $tags{$_->tag};                                                # Find last tag with the specified name
   }
  return undef                                                                  # No such node
 }

sub lastNot($@)                                                                 #U Return the last child node that does not match any of the named B<@tags> under the specified parent B<$node>. Return B<undef> if there is no such child node.
 {my ($node, @tags) = @_;                                                       # Parent node, child tags to avoid.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for(reverse $node->contents)                                                  # Search backwards through contents
   {return $_ unless $tags{$_->tag};                                            # Find last tag that fails to match
   }
  return undef                                                                  # No such node
 }

sub lastOf($@)                                                                  #U Return an array of the nodes that are continuously last under their specified parent node and that match the specified list of tags.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my @l;                                                                        # Matching last nodes
  for(reverse $node->contents)                                                  # Search backwards through contents
   {return @l unless $tags{$_->tag};                                            # Nonmatching tag
    unshift @l, $_;                                                             # Save continuously matching tag in correct order
   }
  return
   @l                                                                           # All tags match
 }

sub lastInIndex($@)                                                             #CYU Return the specified B<$node> if it is last in its index and optionally L<at|/at> the specified context else B<undef>
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $parent = $node->parent;                                                   # Parent
  return undef unless $parent;                                                  # The root node is not first in anything
  my @c = $parent->c($node->tag);                                               # Index containing node
  @c && $c[-1] == $node ? $node : undef                                         # Last in index ?
 }

sub lastContextOf($@)                                                           #CYU Return the last node encountered in the specified context in a depth first reverse pre-order traversal of the L<parse|/parse> tree.
 {my ($node, @context) = @_;                                                    # Node, array of tags specifying context.
  my $x;                                                                        # Found node if found
  eval                                                                          # Trap the die which signals success
   {$node->downReverse(sub                                                      # Traverse  L<parse|/parse> tree in depth first order
     {my ($o) = @_;
      if ($o->at(@context))                                                     # Does this node match the supplied context?
       {$x = $o;                                                                # Success
        die "success!";                                                         # Halt the search
       }
     });
   };
  confess $@ if $@ and  $@ !~ /success!/;                                       # Report any suppressed error messages at this point
  $x                                                                            # Return node found if we are still alive
 }

sub lastSibling($@)                                                             #CYU Return the last sibling of the specified B<$node> in the optional B<@context> else B<undef>
 {my ($node, @context) = @_;                                                    # Node, array of tags specifying context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node->parent;                                                        # Parent node
  $p->last                                                                      # Return last sibling
 }

sub lastWhile($@)                                                               #U Go last from the specified B<$node> and continue deeper lastly as long as each last child node matches one of the specified B<@tags>. Return the deepest such node encountered or else return B<undef> if no such node is encountered.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my $p;                                                                        # Current position
  for(my $l = $node->last; $l and $tags{-t $l}; $l = $l->first) {$p = $l}       # Go ever lastly
  $p
 }

sub lastUntil($@)                                                               #CU Go last from the specified B<$node> and continue deeper lastly until a last child node matches the specified B<@context> or return B<undef> if there is no such node.  Return the last child of the specified B<$node> if no B<@context> is specified.
 {my ($node, @context) = @_;                                                    # Node, context to search for.
  for(my $p = $node->last; $p; $p = $p->last)                                   # Check each last child node below the B<$node>
   {return $p if $p->at(@context);                                              # Return the node if it matches the specified context
   }
  undef
 }

sub lastUntilText($@)                                                           #CU Go last from the specified B<$node> and continue deeper lastly until a last child text node matches the specified B<@context> or return B<undef> if there is no such node.
 {my ($node, @context) = @_;                                                    # Node, context to search for.
  for(my $p = $node->last; $p; $p = $p->last)                                   # Check each last child node below the B<$node>
   {return $p if $p->isText and $p->parent->at(@context);                       # Return the node if it matches the specified context
   }
  undef
 }

#D2 Next                                                                        # Find sibling nodes after the specified B<$node>.

sub next($@)                                                                    #BCYU Return the node next to the specified B<$node>, optionally checking the next node's context. See L<addNext|/addNext> to ensure that an expected node is in position.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if $node->isLast;                                                # No node follows the last node at a level or the top most node
  my @c = $node->parent->contents;                                              # Content array of parent
  while(@c)                                                                     # Test until no more nodes left to test
   {my $c = shift @c;                                                           # Each node
    if ($c == $node)                                                            # Current node
     {my $n = shift @c;                                                         # Next node
      return undef if @context and !$n->at(@context);                           # Next node is not in specified context
      return $n;                                                                # Found node
     }
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }
#a isFirst
#b <b/><c/><b/><d/>
#c   at   b
#c   next c
#c   set id c_after_b
#d Go to the next sibling optionally checking its context.

sub nextn($$@)                                                                  #CU Return the B<$n>'th next node after the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<nextn(1)> is identical in effect to L<next|/next>.
 {my ($node, $N, @context) = @_;                                                # Node, number of times to go next, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(1..$N)                                                                    # Go next the specified number of times
   {$node = $node->next;                                                        # Go next
    last unless $node;                                                          # Cannot go further
   }
  $node
 }

sub nextText($@)                                                                #CU Return the node after the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $l = &next($node);                                                         # Next node
  $l ? $l->isText : undef                                                       # Test whether the first node exists and is a text node
 }

sub nextTextMatches($$@)                                                        #CU Return the next node to the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.
 {my ($node, $match, @context) = @_;                                            # Node, regular expression the text must match, optional context of specified node.
  return undef if @context and !$node->at(@context);                            # Check context
  if (my $t = $node->nextText)                                                  # Next node is text
   {return $t->matchesText($match);                                             # Next text node matches the specified regular expression
   }
  undef                                                                         # Next node is not text or does not match the specified regular expression
 }

sub nextIn($@)                                                                  #YU Return the nearest sibling after the specified B<$node> that matches one of the named tags or B<undef> if there is no such sibling node.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my $parent = $node->parent;                                                   # Parent node
  return undef unless $parent;                                                  # No nodes follow the root node
  my @c = $parent->contents;                                                    # Search forwards through contents
  shift @c while @c and $c[0] != $node;                                         # Move up to starting node
  shift @c;                                                                     # Move over starting node
  for(@c)                                                                       # Each subsequent node
   {return $_ if $tags{$_->tag};                                                # Find first tag with the specified name in the remaining nodes
   }
  return undef                                                                  # No such node
 }

sub nextOn($@)                                                                  #U Step forwards as far as possible from the specified B<$node> while remaining on nodes with the specified tags. In scalar context return the last such node reached or the starting node if no such steps are possible. In array context return the start node and any following matching nodes.
 {my ($node, @tags) = @_;                                                       # Start node, tags identifying nodes that can be step on to context.
  return wantarray ? ($node) : $node if $node->isLast;                          # Easy case
  my $parent = $node->parent;                                                   # Parent node
  confess "No parent" unless $parent;                                           # Not possible on a root node
  my @c = $parent->contents;                                                    # Content
  shift @c while @c and $c[0] != $node;                                         # Position on current node
  confess "Node not found in parent" unless @c;                                 # Something wrong with parent/child relationship
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags of acceptable commands
  if (wantarray)                                                                # Return node and following matching nodes if array wanted
   {my @a = (shift @c);
    push @a, shift @c while @c and $tags{$c[0]->tag};                           # Proceed forwards staying on acceptable tags
    @a                                                                          # Current node and matching following nodes
   }
  else
   {shift @c while @c > 1 and $tags{$c[1]->tag};                                # Proceed forwards but staying on acceptable tags
    return $c[0]                                                                # Current node or last acceptable tag reached while staying on acceptable tags
   }
 }

sub nextWhile($@)                                                               #U Go to the next sibling of the specified B<$node> and continue forwards while the tag of each sibling node matches one of the specified B<@tags>. Return the first sibling node that does not match else B<undef> if there is no such sibling.
 {my ($node, @tags) = @_;                                                       # Node, child tags to avoid.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for($node->contentAfter)                                                      # Search forwards through siblings
   {return $_ unless $tags{$_->tag};                                            # Find first tag that fails to match
   }
  return undef                                                                  # No such node
 }

sub nextUntil($@)                                                               #U Go to the next sibling of the specified B<$node> and continue forwards until the tag of a sibling node matches one of the specified B<@tags>. Return the matching sibling node else B<undef> if there is no such sibling node.
 {my ($node, @tags) = @_;                                                       # Node, tags to look for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for($node->contentAfter)                                                      # Search forwards through following siblings
   {return $_ if $tags{$_->tag};                                                # Find next node that matches on of the supplied tags
   }
  undef                                                                         # No such node
 }

#D2 Prev                                                                        # Find sibling nodes before the specified B<$node>.

sub prev($@)                                                                    #BCYU Return the node before the specified B<$node>, optionally checking the previous node's context. See L<addLast|/addLast> to ensure that an expected node is in position.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if $node->isFirst;                                               # No node follows the last node at a level or the top most node
  my @c = $node->parent->contents;                                              # Content array of parent
  while(@c)                                                                     # Test until no more nodes left to test
   {my $c = pop @c;                                                             # Each node
    if ($c == $node)                                                            # Current node
     {my $n = pop @c;                                                           # Prior node
      return undef if @context and !$n->at(@context);                           # Prior node is not in specified context
      return $n;                                                                # Found node
     }
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }
#a
#b <b/><c/><b/><d/>
#c   at   d
#c   prev b
#c   set id b_before_d
#d Go to the previous sibling optionally checking its context.

sub prevText($@)                                                                #CU Return the node before the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $l = &prev($node);                                                         # Previous node
  $l ? $l->isText : undef                                                       # Test whether the first node exists and is a text node
 }

sub prevn($$@)                                                                  #CU Return the B<$n>'th previous node after the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<prevn(1)> is identical in effect to L<prev|/prev>.
 {my ($node, $N, @context) = @_;                                                # Node, number of times to go prev, optional context.
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(1..$N)                                                                    # Go previous the specified number of times
   {$node = $node->prev;                                                        # Go previous
    return undef unless $node;                                                  # Cannot go further
   }
  $node
 }
#a
#b <b/><c/><d/><e/>
#c   last
#c   prevn 2
#c   set id yes
#d Go backwards

sub prevTextMatches($$@)                                                        #CU Return the previous node to the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.
 {my ($node, $match, @context) = @_;                                            # Node, regular expression the text must match, optional context of specified node.
  return undef if @context and !$node->at(@context);                            # Check context
  if (my $t = $node->prevText)                                                  # Previous node is text
   {return $t->matchesText($match);                                             # Previous text node matches the specified regular expression
   }
  undef                                                                         # Previous node is not text or does not match the specified regular expression
 }

sub prevIn($@)                                                                  #YU Return the nearest sibling node before the specified B<$node> which matches one of the named tags or B<undef> if there is no such sibling node.
 {my ($node, @tags) = @_;                                                       # Node, tags to search for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my $parent = $node->parent;                                                   # Parent node
  return undef unless $parent;                                                  # No nodes follow the root node
  my @c = reverse $parent->contents;                                            # Reverse through contents
  shift @c while @c and $c[0] != $node;                                         # Move down to starting node
  shift @c;                                                                     # Move over starting node
  for(@c)                                                                       # Each subsequent node
   {return $_ if $tags{$_->tag};                                                # Find first tag with the specified name in the remaining nodes
   }
  return undef                                                                  # No such node
 }

sub prevOn($@)                                                                  #U Step backwards as far as possible while remaining on nodes with the specified tags. In scalar context return the last such node reached or the starting node if no such steps are possible. In array context return the start node and any preceding matching nodes.
 {my ($node, @tags) = @_;                                                       # Start node, tags identifying nodes that can be step on to context.
  return wantarray ? ($node) : $node if $node->isFirst;                         # Easy case
  my $parent = $node->parent;                                                   # Parent node
  confess "No parent" unless $parent;                                           # Not possible on a root node
  my @c = reverse $parent->contents;                                            # Content backwards
  shift @c while @c and $c[0] != $node;                                         # Position on current node
  confess "Node not found in parent" unless @c;                                 # Something wrong with parent/child relationship
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags of acceptable commands
  if (wantarray)                                                                # Return node and following matching nodes if array wanted
   {my @a = (shift @c);
    push @a, shift @c while @c and $tags{$c[0]->tag};                           # Proceed forwards staying on acceptable tags
    @a                                                                          # Current node and matching following nodes
   }
  else
   {shift @c while @c > 1 and $tags{$c[1]->tag};                                # Proceed forwards but staying on acceptable tags
    return $c[0]                                                                # Current node or last acceptable tag reached while staying on acceptable tags
   }
 }

sub prevWhile($@)                                                               #U Go to the previous sibling of the specified B<$node> and continue backwards while the tag of each sibling node matches one of the specified B<@tags>. Return the first sibling node that does not match else B<undef> if there is no such sibling.
 {my ($node, @tags) = @_;                                                       # Parent node, child tags to avoid.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for(reverse $node->contentBefore)                                             # Search backwards through siblings
   {return $_ unless $tags{$_->tag};                                            # Find first tag that fails to match
   }
  return undef                                                                  # No such node
 }

sub prevUntil($@)                                                               #U Go to the previous sibling of the specified B<$node> and continue backwards until the tag of a sibling node matches one of the specified B<@tags>. Return the matching sibling node else B<undef> if there is no such sibling node.
 {my ($node, @tags) = @_;                                                       # Node, tags to look for.
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  for($node->contentBefore)                                                     # Search forwards through following siblings
   {return $_ if $tags{$_->tag};                                                # Find next node that matches on of the supplied tags
   }
  undef                                                                         # No such node
 }

#D2 Up                                                                          # Methods for moving up the L<parse|/parse> tree from a node.

sub top($@)                                                                     #CYU Return the top of the parse tree containing the current B<$node> after optionally checking that the $node is in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for (my $p = $node;; $p = $p->parent)                                         # Walk up the parse tree
   {return $p unless $p->parent;                                                # Continue up the parse tree unless we are at the top - if not, let us hope that Zorn's lemma applies soon rather than later
   }
 }

BEGIN {*root = *top}

#a
#b <b><c/></b>
#c top
#c set id top
#d Go to the top of the parse tree.

sub up($@)                                                                      #CYU Return the parent of the current node optionally checking the parent node's context or return B<undef> if the specified B<$node> is the root of the L<parse|/parse> tree.   See L<addWrapWith|/addWrapWith> to ensure that an expected node is in position.
 {my ($node, @context) = @_;                                                    # Start node, optional context of parent.
  return $node->parent unless @context;                                         # Parent with no context check
  my $p = $node->parent;
  $p->at(@context) ? $p : undef;                                                # Check context of parent
 }
#a
#b <b><c/></b>
#c   at c
#c   up
#c   set id above_c
#d Go up one level.

sub upn($$@)                                                                    #CU Go up the specified number of levels from the specified B<$node> and return the node reached optionally checking the parent node's context or B<undef> if there is no such node.L<upn(1)|/up> is identical in effect to L<up|/up>.  Or use L<ancestry|/ancestry> to get the path back to the root node.
 {my ($node, $levels, @context) = @_;                                           # Start node, number of levels to go up, optional context.
  for(my $c = 0; $node and $c < $levels; $node = $node->parent, ++$c) {}        # Number of levels move up
  return $node unless @context;                                                 # Return node reached unless context check required
  $node ? $node->at(@context) : undef;                                          # Check context
 }

sub upWhile($@)                                                                 #YU Go up one level from the specified B<$node> and then continue up while each node matches on of the specified <@tags>. Return the last matching node or B<undef> if no node matched any of the specified B<@tags>.
 {my ($node, @tags) = @_;                                                       # Start node, tags to match
  my %tags = map {$_=>1} @tags;                                                 # Hashify tags
  my $lastMatch;                                                                # Last good match
  for(my $p = $node->parent; $p; $p = $p->parent)                               # Go up
   {last unless $tags{-t $p};                                                   # Found an ancestor that does not match
    $lastMatch = $p;
   }
  $lastMatch                                                                    # Last good match
 }

sub upWhileFirst($@)                                                            #CU Move up from the specified B<$node> as long as each node is a first node or return B<undef> if the specified B<$node> is not a first node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context && !$node->at(@context) or !$node->isFirst;          # Check the context if supplied and that the node is first
  my $lastMatch = $node;                                                        # First node
  for(my $p = $node->parent; $p; $p = $p->parent)                               # Go up
   {return $lastMatch unless $p->isFirst;                                       # Return last node which was first
    $lastMatch = $p                                                             # Update last matching position
   }
  $lastMatch                                                                    # Root node matches
 }

sub upWhileLast($@)                                                             #CU Move up from the specified B<$node> as long as each node is a last node or return B<undef> if the specified B<$node> is not a last node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context && !$node->at(@context) or !$node->isLast;           # Check the context if supplied and that the node is last
  my $lastMatch = $node;                                                        # Last node
  for(my $p = $node->parent; $p; $p = $p->parent)                               # Go up
   {return $lastMatch unless $p->isLast;                                        # Return last node which was last
    $lastMatch = $p                                                             # Update last matching position
   }
  $lastMatch                                                                    # Root node matches
 }

sub upWhileIsOnlyChild($@)                                                      #CU Move up from the specified B<$node> as long as each node is an only child or return B<undef> if the specified B<$node> is not an only child.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context && !$node->at(@context) or !$node->isOnlyChild;      # Check the context if supplied and that the node is an only child
  my $lastMatch = $node;                                                        # Last node
  for(my $p = $node->parent; $p; $p = $p->parent)                               # Go up
   {return $lastMatch unless $p->isOnlyChild;                                   # Return last node which was an only child
    $lastMatch = $p                                                             # Update last matching position
   }
  $lastMatch                                                                    # Root node matches
 }

sub upUntil($@)                                                                 #CYU Find the first node going up from B<$node> that matches the specified B<@context>. The first such node will be the specified $node if no @context is specified or the specified $node matches the specified @context>.
 {my ($node, @context) = @_;                                                    # Start node, context.
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {return $p if $p->at(@context);                                              # Return node which satisfies the condition
   }
  return undef                                                                  # Not found
 }
#a upUntil
#b <b><c><d/></c></b>
#c at d
#c upUntil c b
#c set id up
#d Go up while we are not in the specified context

sub upUntilFirst($@)                                                            #CU Move up from the specified B<$node> until we reach the root or a first node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {return $p if $p->isFirst;                                                   # Return first first node
   }
  undef                                                                         # This should not happen
 }

sub upUntilLast($@)                                                             #CU Move up from the specified B<$node> until we reach the root or a last node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {return $p if $p->isLast;                                                    # Return first last node
   }
  undef                                                                         # This should not happen
 }

sub upUntilIsOnlyChild($@)                                                      #CU Move up from the specified B<$node> until we reach the root or another only child.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied and that the node is an only child
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {return $p if $p->isOnlyChild;                                               # Return last node which was an only child
   }
  undef                                                                         # This should not happen
 }

sub upThru($@)                                                                  #YU Go up the specified path from the specified B<$node> returning the node at the top or B<undef> if no such node exists.
 {my ($node, @tags) = @_;                                                       # Start node, tags identifying path.
  while(@tags)                                                                  # Go up through the tags
   {$node = $node->parent;                                                      # Go up on level
    return undef unless $node and $node->at(shift @tags);                       # Failed to match next tag
   }
  $node                                                                         # Reached the top of the path
 }

#D2 down                                                                        # Methods for moving down through the L<parse|/parse> tree from a node.

sub downWhileFirst($@)                                                          #CU Move down from the specified B<$node> as long as each lower node is a first node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(my $p = $node->first; $p; $p = $p->first)                                 # Go down firstly
   {return $p unless $p->first;                                                 # Return node unless there is another one below it
   }
  $node->isFirst                                                                # Leaf node
 }

BEGIN{*firstLeaf=*downWhileFirst}

sub downWhileLast($@)                                                           #CU Move down from the specified B<$node> as long as each lower node is a last node.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  for(my $p = $node->last; $p; $p = $p->last)                                   # Go down lastly
   {return $p unless $p->last;                                                  # Return node unless there is another one below it
   }
  $node->isLast                                                                 # Leaf node
 }

BEGIN{*lastLeaf=*downWhileLast}

sub downWhileHasSingleChild($@)                                                 #CU Move down from the specified B<$node> as long as it has a single child else return undef.
 {my ($node, @context) = @_;                                                    # Start node, optional context
  return undef if @context and !$node->at(@context);                            # Check the context if supplied
  my $q;
  for(my $p = $node; $p; $q = $p, $p = $p->first)
   {last unless $p->hasSingleChild;
   }
  $q
 }

#D1 Editing                                                                     # Edit the data in the L<parse|/parse> tree and change the structure of the L<parse|/parse> tree by L<wrapping and unwrapping|/Wrap and unwrap> nodes, by L<replacing|/Replace> nodes, by L<cutting and pasting|/Cut and Put> nodes, by L<concatenating|/Fusion> nodes, by L<splitting|/Fission> nodes, by adding new L<text|/Put as text> nodes or L<swapping|/swap> nodes.

sub change($$@)                                                                 #CIYU Change the name of the specified B<$node>, optionally  confirming that the B<$node> is in a specified context and return the B<$node>.
 {my ($node, $name, @context) = @_;                                             # Node, new name, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->tag = $name;                                                           # Change name
  if (my $parent = $node->parent) {$parent->indexNode}                          # Reindex parent
  $node
 }
#a
#b <b/>
#c change c b a
#d Change a tag.
BEGIN{*cc=*change}

sub changeKids($$@)                                                             #CU Change the names of all the immediate children of the specified B<$node>, if they match the optional B<@context>, to the specified B<$tag> and return the B<$node>.
 {my ($node, $name, @context) = @_;                                             # Node, new name, optional context.
  if (!@context)                                                                # No context - change all immediate children
   {$_->tag = $name for $node->contents;                                        # Change each node regardless
   }
  else                                                                          # Check context of each child before changing name
   {for my $c($node->contents)                                                  # Each immediate child
     {$c->tag = $name if $c->at(@context);                                      # Change node if in specified context
     }
   }
  $node->indexNode;                                                             # Reindex node
  $node                                                                         # Return node
 }

BEGIN{*ck=*change}

sub changeText($$$$@)                                                           #CU Change the content of the specified text B<$node> that matches a regular expression B<$rf> presented as a string to a string B<$rt> in the optional B<@context> and return the specified $node else return B<undef>. For a more efficient non unitary method see L<editText>.
 {my ($node, $rf, $rt, $flags, @context) = @_;                                  # Text node, from re, to string, re flags, optional context
  confess "Parameter 2 should be a string" if ref($rf);                         # Check usage
  return undef unless $node->isText(@context);                                  # Check that this is a text node in the specified context.
  my $t = $node->text;                                                          # Text to change
  my $c = qq(\$t =~ s($rf) ($rt)$flags);                                        # Create regular expression
  my $r = eval qq(\$t =~ s($rf) ($rt)$flags);                                   # Evaluate regular expression
  confess "$@\n" if $@;                                                         # Confess to failures
  $node->text = $t;                                                             # Update text
  $r ? $node : undef                                                            # Return node on success or undef on fail
 }
#a editText
#b <b>ababa</b>
#c changeText b(.)b $1C$1 s
#d Change text using a regular expression, target string and flags.

sub changeTextToSpace($$@)                                                      #CU Change each instance of the content of the specified text B<$node> that matches a regular expression B<$re> to one space and return the specified $node else return B<undef>.
 {my ($node, $re, @context) = @_;                                               # Text node, regular expression, optional context
  return undef unless $node->isText(@context);                                  # Check that this is a text node in the specified context.
  $node->text =~ s($re) ( )gs ? $node : undef                                   # Change text to spaces
 }
#a
#b <b>ababa</b>
#c changeTextToSpace b
#d Change text to space using a regular expression.

sub dupPutNext($@)                                                              #CU Duplicate the specified B<$tree> in the optional B<@context>, place the new tree next after $tree and return the root of the new tree on success else B<undef>.
 {my ($tree, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$tree->at(@context);                            # Not in specified context
  putNext($tree, clone($tree));                                                 # Place new node and return it
 }

BEGIN{*r=*dupPutNext}                                                           # As used in editWithPerl

sub dupPutPrev($@)                                                              #CU Duplicate the specified B<$tree> in the optional B<@context>, place the new tree before $tree and return the root of the new tree on success else B<undef>.
 {my ($tree, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$tree->at(@context);                            # Not in specified context
  putPrev($tree, clone($tree));                                                 # Place new node and return it
 }

sub dupPutNextN($$@)                                                            #CU Duplicate the specified B<$tree> B<$N> times in the optional B<@context>, placing each copy after $tree and return the last new node created on success else B<undef>.
 {my ($node, $N, @context) = @_;                                                # Node, number of duplications, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless my $parent = $node->parent;                               # Cannot duplicate the root node
  my $r = $node->renew;                                                         # Make a copy of the node and its descendants
  my $f = freeze($r);                                                           # Freeze parse tree restricted to node and its contents
  my $c = $parent->content;                                                     # Content array of parent
  my $i = $node->position;                                                      # Position in content array
  my @n = map {my $r = thaw($f); $r->parent = $parent; $r} 1..$N;               # New nodes after original node - copy on write might be useful
  splice(@$c, $i+1, 0, @n);                                                     # Insert new nodes after original node
  $parent->indexNode;                                                           # Rebuild indices for parent
 }

BEGIN{*rN=*dupPutNextN}

our $selectionStart;                                                            # Selection starts here
our $selectionEnd;                                                              # Selection ends here

sub setSelectionStart($@)                                                       #CU Set the selection to start at the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>..
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $selectionStart = $node;                                                      # Start selection
 }

BEGIN{*ss=*setSelectionStart}

sub setSelectionEnd($@)                                                         #CU Set the selection to end at the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>..
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $selectionEnd = $node;                                                        # End selection
 }

BEGIN{*se=*setSelectionEnd}

sub moveSelectionFirst($@)                                                      #CU Move the current selection (if there is one) so that it is first under the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $selectionStart || $selectionEnd;                         # No selection
  my $s = $selectionStart || $selectionEnd;                                     # Selection start
  my $e = $selectionEnd   || $selectionStart;                                   # Selection end
  $s->moveBlockFirst($e, $node);                                                # Move selection
  $s->putFirstAsComment(q(start));                                              # Show selection start
  $e->putLastAsComment(q(end));                                                 # Show selection end
  $node;                                                                        # Success
 }

BEGIN{*mf=*moveSelectionFirst}

sub moveSelectionAfter($@)                                                      #CU Move the current selection (if there is one) after the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $selectionStart || $selectionEnd;                         # No selection
  my $s = $selectionStart || $selectionEnd;                                     # Selection start
  my $e = $selectionEnd   || $selectionStart;                                   # Selection end
  $s->moveBlockAfter($e, $node);                                                # Move selection
  $node;                                                                        # Success
 }

BEGIN{*ma=*moveSelectionAfter}

sub moveSelectionBefore($@)                                                     #CU Move the current selection (if there is one) before the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $selectionStart || $selectionEnd;                         # No selection
  my $s = $selectionStart || $selectionEnd;                                     # Selection start
  my $e = $selectionEnd   || $selectionStart;                                   # Selection end
  $s->moveBlockBefore($e, $node);                                               # Move selection
  $node;                                                                        # Success
 }

BEGIN{*mb=*moveSelectionBefore}

sub moveSelectionLast($@)                                                       #CU Move the current selection (if there is one) so that it is last under the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $selectionStart || $selectionEnd;                         # No selection
  my $s = $selectionStart || $selectionEnd;                                     # Selection start
  my $e = $selectionEnd   || $selectionStart;                                   # Selection end
  $s->moveBlockLast($e, $node);                                                 # Move selection
  $node;                                                                        # Success
 }

BEGIN{*ml=*moveSelectionLast}

#D2 Cut and Put                                                                 # Cut and paste nodes in the L<parse|/parse> tree.

sub cut($@)                                                                     #CUI Cut out and return the specified B<$node> so that it can be reinserted else where in the L<parse|/parse> tree.
 {my ($node, @context) = @_;                                                    # Node to cut out, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent node
  # confess "Already cut out" unless $parent;                                   # We have to let thing be cut out more than once or supply an isCutOut() method
  return $node unless $parent;                                                  # Uppermost node is already cut out
  my $c = $parent->content;                                                     # Content array of parent
  my $i = $node->position;                                                      # Position in content array
  splice(@$c, $i, 1);                                                           # Remove node
  $parent->indexNode;                                                           # Rebuild indices
  $node->disconnectLeafNode;                                                    # Disconnect node no longer in L<parse|/parse> tree
  $node                                                                         # Return node
 }
#a unwrap
#b <b>C</b>
#c cut b
#d Delete a node and all its contents.

BEGIN{*x=*cut}

our @saveLastCutOut;                                                            # Save the last node cut out on the cut out stack

sub cutFirst($@)                                                                #CU Cut out the first node below the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return $node on success else return B<undef>. The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $first = $node->first;                                 # First child node must exist
  return undef if @context and !$first->at(@context);                           # First child node not in specified context
  push @saveLastCutOut, $first->cut;                                            # Cut out node saving it on the cut out stack
  $node                                                                         # Return current node
 }

sub putCutOutFirst($@)                                                          #CU Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it first under the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $cut = pop @saveLastCutOut;                            # Last cut out node
  $node->putFirst($cut);                                                        # Insert cut out node
  $node                                                                         # Return current node
 }

sub cutLast($@)                                                                 #CU Cut out the last node below the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return $node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $last = $node->last;                                   # Last child node must exist
  return undef if @context and !$last->at(@context);                            # Last child node not in specified context
  push @saveLastCutOut, $last->cut;                                             # Cut out node saving it on the cut out stack
  $node                                                                         # Return current node
 }

sub putCutOutLast($@)                                                           #CU Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it last under the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $cut = pop @saveLastCutOut;                            # Last cut out node
  $node->putLast($cut);                                                         # Insert cut out node
  $node                                                                         # Return current node
 }

sub cutNext($@)                                                                 #CU Cut out the next node beyond the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return the current node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $next = $node->next;                                   # Next node must exist
  return undef if @context and !$next->at(@context);                            # Not in specified context
  push @saveLastCutOut, $next->cut;                                             # Cut out node saving it on the cut out stack
  $node                                                                         # Return current node
 }
#a cut
#b <b/><c/><d/>
#c first
#c cutNext c a
#c next d
#c putCutOutNext
#d Cut out the next node and reinsert it

sub putCutOutNext($@)                                                           #CU Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it next after the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $cut = pop @saveLastCutOut;                            # Last cut out node
  $node->putNext($cut);                                                         # Insert cut out node
  $node                                                                         # Return current node
 }

sub cutPrev($@)                                                                 #CU Cut out the previous node before the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return the current node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef unless my $prev = $node->prev;                                   # Previous node must exist
  return undef if @context and !$prev->at(@context);                            # Not in specified context
  push @saveLastCutOut, $prev->cut;                                             # Cut out node saving it on the cut out stack
  $node                                                                         # Return current node
 }
#a cut
#b <b/><c/><d/>
#c last
#c cutPrev c a
#c prev b
#c putCutOutPrev
#d Cut out the previous node and reinsert it

sub putCutOutPrev($@)                                                           #CU Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it before the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $cut = pop @saveLastCutOut;                            # Last cut out node
  $node->putPrev($cut);                                                         # Insert cut out node
  $node                                                                         # Return current node
 }

sub cutIfEmpty($@)                                                              #CU Cut out and return the specified B<$node> so that it can be reinserted else where in the L<parse|/parse> tree if it is empty.
 {my ($node, @context) = @_;                                                    # Node to cut out, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node->cut if $node->isAllBlankText;                                   # Cut node if it has no content or all blank content
  undef                                                                         # Cannot cut out node
 }
#a cut unwrap
#b <b/><c/><b/><d>text</d>
#c cutIfEmpty
#d Remove a node if it is empty.

sub deleteContent($@)                                                           #CU Delete the content of the specified B<$node>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my @c = $node->contents;                                                      # Content
  $_->parent = undef for @c;                                                    # Remove parent link from each child
  $node->content = [];                                                          # Delete content
  $node                                                                         # Return node
 }

sub putFirst($$@)                                                               #C Place a L<cut out|/cut> or L<new|/new> node at the front of the content of the specified B<$node> and return the new node. See L<putFirstCut|/putFirstCut> to cut and put first in one operation. See L<addFirst|/addFirst> to perform this operation conditionally.
 {my ($old, $new, @context) = @_;                                               # Original node, new node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  $new->parent and confess "Please cut out the node before moving it";          # The node must have be cut out first
  $new->parser == $new and $old->parser == $new and                             # Prevent a root node from being inserted into a sub tree
    confess "Recursive insertion attempted";
  $new->parser = $old->parser;                                                  # Assign the new node to the old parser
  unshift @{$old->content}, $new;                                               # Content array of original node
  $old->indexNode;                                                              # Rebuild indices for node
  $new                                                                          # Return the new node
 }

sub putFirstCut($$@)                                                            #C Cut out the B<$second> node, place it first under the B<$first> node and return the B<$second> node.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context.
  $first->putFirst($second->cut, @context)                                      # Place second node relative to the first node if in the specified context and return the second node.
 }

sub putLast($$@)                                                                #CI Place a L<cut out|/cut> or L<new|/new> node last in the content of the specified B<$node> and return the new node.  See L<putLastCut|/putLastCut> to cut and put last in one operation.  See L<addLast|/addLast> to perform this operation conditionally.
 {my ($old, $new, @context) = @_;                                               # Original node, new node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  $new->parent and confess "Please cut out the node before moving it";          # The node must have be cut out first
  $new->parser == $new and $old->parser == $new and                             # Prevent a root node from being inserted into a sub tree
    confess "Recursive insertion attempted";
  $new->parser = $old->parser;                                                  # Assign the new node to the old parser
  push @{$old->content}, $new;                                                  # Content array of original node
  $old->indexNode;                                                              # Rebuild indices for node
  $new                                                                          # Return the new node
 }

sub putLastCut($$@)                                                             #C Cut out the B<$second> node, place it last under the B<$first> node and return the B<$second> node.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context.
  $first->putLast($second->cut, @context)                                       # Place second node relative to the first node if in the specified context and return the second node.
 }

sub putNext($$@)                                                                #C Place a L<cut out|/cut> or L<new|/new> node just after the specified B<$node> and return the new node. See L<putNextCut|/putNextCut> to cut and put next in one operation.  See L<addNext|/addNext> to perform this operation conditionally.
 {my ($old, $new, @context) = @_;                                               # Original node, new node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  return undef unless my $parent = $old->parent;                                # Parent node
  $parent or confess "Cannot place a node after the outermost node";            # The originating node must have a parent
  $new->parent and confess "Please cut out the node before moving it";          # The node must have be cut out first
  $new->parser == $new and $old->parser == $new and                             # Prevent a root node from being inserted into a sub tree
    confess "Recursive insertion attempted";
  $new->parser = $old->parser;                                                  # Assign the new node to the old parser
  my $c = $parent->content;                                                     # Content array of parent
  my $i = $old->position;                                                       # Position in content array
  splice(@$c, $i+1, 0, $new);                                                   # Insert new node after original node
  $new->parent = $parent;                                                       # Return node
  $parent->indexNode;                                                           # Rebuild indices for parent
  $new                                                                          # Return the new node
 }

sub putNextCut($$@)                                                             #C Cut out the B<$second> node, place it after the B<$first> node and return the B<$second> node.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context.
  $first->putNext($second->cut, @context)                                       # Place second node relative to the first node if in the specified context and return the second node.
 }

sub putPrev($$@)                                                                #C Place a L<cut out|/cut> or L<new|/new> node just before the specified B<$node> and return the new node.  See L<putPrevCut|/putPrevCut> to cut and put previous in one operation.  See L<addPrev|/addPrev> to perform this operation conditionally.
 {my ($old, $new, @context) = @_;                                               # Original node, new node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  return undef unless my $parent = $old->parent;                                # Parent node
  $parent or confess "Cannot place a node before the outermost node";           # The originating node must have a parent
  $new->parent and confess "Please cut out the node before moving it";          # The node must have be cut out first
  $new->parser == $new and $old->parser == $new and                             # Prevent a root node from being inserted into a sub tree
    confess "Recursive insertion attempted";
  $new->parser = $old->parser;                                                  # Assign the new node to the old parser
  my $c = $parent->content;                                                     # Content array of parent
  my $i = $old->position;                                                       # Position in content array
  splice(@$c, $i, 0, $new);                                                     # Insert new node before original node
  $new->parent = $parent;                                                       # Return node
  $parent->indexNode;                                                           # Rebuild indices for parent
  $new                                                                          # Return the new node
 }

sub putPrevCut($$@)                                                             #C Cut out the B<$second> node, place it before the B<$first> node and return the B<$second> node.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context.
  $first->putPrev($second->cut, @context)                                       # Place second node relative to the first node if in the specified context and return the second node.
 }

#D2 Put Siblings or Contents                                                    # Move a node and its siblings up and down the parse tree.

sub putSiblingsFirst($@)                                                        #CU Move the siblings preceding the specified B<$node> in the optional B<@context> down one level and place them first under the specified B<$node> preceding any existing content.  Return the specified B<$node>.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  for my $n(reverse $node->contentBefore)                                       # Each node preceding the specified node
   {$node->putFirstCut($n);                                                     # Place node first
   }
  $node                                                                         # Success
 }

sub putSiblingsLast($@)                                                         #CU Move the siblings following the specified B<$node> in the optional B<@context> down one level so that they are last under the specified B<$node> following any existing content. Return the specified B<$node>.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  for my $n($node->contentAfter)                                                # Each node following the specified node
   {$node->putLastCut($n);                                                      # Place node last
   }
  $node                                                                         # Success
 }

sub putSiblingsAfterParent($@)                                                  #CU Move the specified B<$node> and its following siblings up one level and place them after the parent of the specified B<$node> if the specified B<$node> is in the optional B<@context>.  Return the specified B<$node> if the move was made successfully, else confess that the specified move is not possible.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  confess "Cannot put after root of parse tree" unless my $p = $node->parent;   # Fail on root node
  for my $n(reverse @$p)                                                        # Each node from last to first under parent
   {$p->putNextCut($n);                                                         # Place node after parent
    last if $node == $n;                                                        # Stop when we reach the specified node
   }
  $node                                                                         # Success
 }

sub putSiblingsBeforeParent($@)                                                 #CU Move the specified B<$node> and its preceding siblings up one level and place them before the parent of the specified B<$node> if the specified B<$node> is in the optional B<@context>.  Return the specified B<$node> if the move was made successfully, else confess that the specified move is not possible.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  confess "Cannot put before root of parse tree" unless my $p = $node->parent;  # Fail on root node
  for my $n($p->contents)                                                       # Each node from last to first under parent
   {$p->putPrevCut($n);                                                         # Place node after parent
    last if $node == $n;                                                        # Stop when we reach the specified node
   }
  $node                                                                         # Success
 }

sub putContentAfter($@)                                                         #CU Move the content of the specified B<$node> and place it after that node if that node is in the optional B<@context>.  Return the specified B<$node> or confess if the move is not possible.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  confess "Cannot put content after root of parse tree" unless $node->parent;   # Fail on root node
  for my $n(reverse @$node)                                                     # Each content node from last to first
   {$node->putNextCut($n);                                                      # Move contained node after parent
   }
  $node                                                                         # Success
 }

sub putContentBefore($@)                                                        #CU Move the content of the specified B<$node> and place it before that node if that node is in the optional B<@context>.  Return the specified B<$node> or confess if the move is not possible.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  confess "Cannot put content before root of parse tree" unless $node->parent;  # Fail on root node
  for my $n(my @c = @$node)                                                     # Each content node from first to last
   {$node->putPrevCut($n);                                                      # Move contained node before parent
   }
  $node                                                                         # Success
 }

#D2 Move                                                                        # Move nodes around in the L<parse|/parse> tree by cutting and pasting them.

sub moveFirst($@)                                                               #CU Move the specified node so that is is the first sibling under its parent. Returns the specified $node on success otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node to  be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node if $node->isFirst;                                               # Node is already in position
  return undef unless my $parent = $node->parent;                               # Parent node
  $parent->putFirstCut($node);                                                  # Move node
  $node                                                                         # Success
 }

sub moveLast($@)                                                                #CU Move the specified node so that is is the last sibling under its parent. Returns the specified $node on success otherwise B<undef>.
 {my ($node, @context) = @_;                                                    # Node to  be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node if $node->isLast;                                                # Node is already in position
  return undef unless my $parent = $node->parent;                               # Parent node
  $parent->putLastCut($node);                                                   # Move node
  $node                                                                         # Success
 }

sub moveStartFirst($@)                                                          #CU Move the start of a B<$node> to contain all of its preceding siblings as children. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, @context) = @_;                                                    # Node whose start should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->isFirst)                                                           # Node is already in position
   {return $node;
   }
  for my $p(reverse $node->contentBefore)                                       # Each node before the start tag
   {$node->putFirstCut($p);                                                     # Move node
   }
  $node                                                                         # Success
 }

sub moveStartAfter($$@)                                                         #C Move the start end of a B<$node> to just after the specified B<$target> node assuming that the B<$target> node is either a preceding sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, $to, @context) = @_;                                               # Node whose start should be moved, target node, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->prev && $node->prev == $to)                                        # Node is already in position
   {return $node;
   }
  elsif ($to->precedingSiblingOf($node))                                        # Target is outside and before node, so move start up to it
   {for my $p(reverse $node->contentBefore)                                     # Each node before the start node up to the target
     {last if $p == $to;                                                        # Reached target
      $node->putFirstCut($p);                                                   # Move node
     }
    return $node;                                                               # Success
   }
  elsif ($to->parent and $to->parent == $node)                                  # Contained by node
   {for my $p((reverse $to->contentBefore), $to)                                # Nodes from target
     {$node->putPrevCut($p);                                                    # Move node
     }
    return $node;                                                               # Success
   }
  confess "To node is not a child or preceding sibling of node";                # Cannot do the requested operation
 }

sub moveStartBefore($$@)                                                        #C Move the start of a B<$node> to just before the specified B<$target> node assuming that the B<$target> node is either a preceding sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, $to, @context) = @_;                                               # Node whose start should be moved, target node, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->first && $node->first == $to)                                       # Node is already in position
   {return $node;
   }
  elsif ($to->precedingSiblingOf($node))                                        # Target is outside and before node, so move start up to it
   {for my $p(reverse $node->contentBefore)                                     # Each node after the end tag up to the target
     {$node->putFirstCut($p);                                                   # Move node
      last if $p == $to;                                                        # Reached target
     }
    return $node;                                                               # Success
   }
  elsif ($to->parent and $to->parent == $node)                                  # Contained by node
   {for my $p($to->contentBefore)                                               # Nodes from target
     {$node->putPrevCut($p);                                                    # Move node
     }
    return $node;                                                               # Success
   }
  confess "To node is not a child or preceding sibling of node";                # Cannot do the requested operation
 }

sub moveEndLast($@)                                                             #CU Move the end of a B<$node> to contain all of its following siblings as children. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, @context) = @_;                                                    # Node whose end should be moved, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->isLast)                                                            # Node is already in position
   {return $node;
   }
  for my $p($node->contentAfter)                                                # Each node after the end tag
   {$node->putLastCut($p);                                                      # Move node
   }
  return $node;                                                                 # Success
 }

sub moveEndAfter($$@)                                                           #CI Move the end of a B<$node> to just after the specified B<$target> node assuming that the B<$target> node is either a subsequent sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, $to, @context) = @_;                                               # Node whose end should be moved, target node, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->last && $node->last == $to)                                        # Node is already in position
   {return $node;
   }
  elsif ($to->succeedingSiblingOf($node))                                       # Target is outside and beyond node, so move end down to it
   {for my $p($node->contentAfter)                                              # Each node after the end node up to the target
     {$node->putLastCut($p);                                                    # Move node
      last if $p == $to;                                                        # Reached target
     }
    return $node;                                                               # Success
   }
  elsif ($to->parent and $to->parent == $node)                                  # Contained by node
   {for my $p(reverse $to->contentAfter)                                        # Nodes from target
     {$node->putNextCut($p);                                                    # Move node
     }
    return $node;                                                               # Success
   }
  confess "To node is not a child or succeeding sibling of node";               # Cannot do the requested operation
 }

sub moveEndBefore($$@)                                                          #C Move the end of a B<$node> to just before the specified B<$target> node assuming that the B<$target> node is either a subsequent sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($node, $to, @context) = @_;                                               # Node whose end should be moved, target node, optional context of node.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if ($node->next && $node->next == $to)                                        # Node is already in position
   {return $node;
   }
  elsif ($to->succeedingSiblingOf($node))                                       # Target is outside and beyond node, so move end down to it
   {for my $p($node->contentAfter)                                              # Each node after the end tag up to the target
     {last if $p == $to;                                                        # Reached target
      $node->putLastCut($p);                                                    # Move node
     }
    return $node;                                                               # Success
   }
  elsif ($to->parent and $to->parent == $node)                                  # Contained by node
   {for my $p((reverse $to->contentAfter), $to)                                 # Nodes from target
     {$node->putNextCut($p);                                                    # Move node
     }
    return $node;                                                               # Success
   }
  confess "To is not a child or succeeding sibling of node";                    # Cannot do the requested operation
 }

# Move a single node                                                            # Move a single node into a new position close to its current position.

sub putNextFirstCut($@)                                                         #CU Move the specified B<$node> so it is first in the next node with the optional context. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef unless my $next = $node->next;                                   # The next node
  return undef if @context and !$next->at(@context);                            # Not in specified context
  $next->putFirstCut($node);                                                    # Move node into position
 }

sub putNextFirstCut2($@)                                                        #CU Move the specified B<$node> so it is first in the first node with the specified optional B<@context> of the next node. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef unless my $next = $node->next;                                   # The next node
  return undef unless my $first = $next->first;                                 # The First node under the next node
  return undef if @context and !$first->at(@context);                           # Not in specified context
  $first->putFirstCut($node);                                                   # Move node into position
 }

sub putPrevLastCut($@)                                                          #CU Move the specified B<$node> so it is last in the preceding node with the optional context. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef unless my $prev = $node->prev;                                   # The previous node
  return undef if @context and !$prev->at(@context);                            # Not in specified context
  $prev->putLastCut($node);                                                     # Move node into position
 }
#a
#b <b/><c/>
#c putPrevLastCut b
#d Put last in the previous node.
sub putPrevLastCut2($@)                                                         #CU Move the specified B<$node> so it is last in the last node with the specified optional context of the preceding node. Return the specified $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef unless my $prev = $node->prev;                                   # The previous node
  return undef unless my $last = $prev->last;                                   # The last node under the previous node
  return undef if @context and !$last->at(@context);                            # Not in specified context
  $last->putLastCut($node);                                                     # Move node into position
 }

sub putUpNextCut($@)                                                            #CU Move the specified B<$node>, in the optional B<@context>, which must be last under its parent, so that it is next after its parent. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isLast;                                            # The node must be last
  return undef unless my $parent = $node->parent;                               # The parent node
  $parent->putNextCut($node);                                                   # Move node into position
 }
#a putPrevLastCut
#b <b><c/></b>
#c putUpNextCut c
#d Cut out a last child and place it after its parent.

sub putUpNextCut2($@)                                                           #CU Move the specified B<$node>, in the optional B<@context>, if $node is last after its parent which must also be last under its parent, so that $node is next after its grandparent. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isLast;                                            # The node must be last
  return undef unless my $parent = $node->parent;                               # Not in specified context
  return undef unless $parent->isLast;                                          # The parent must also be last
  return undef unless my $qarent = $parent->parent;                             # The parent parent node
  $qarent->putNextCut($node);                                                   # Move node into position
 }

sub putUpPrevCut($@)                                                            #CU Move the specified B<$node> in the optional B<@context>, if $node is first under its parent, so that it is prior to its parent. Return $node on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isFirst;                                            # The node must be last
  return undef unless my $parent = $node->parent;                               # The parent node
  $parent->putPrevCut($node);                                                   # Move node into position
 }

sub putUpPrevCut2($@)                                                           #CU Move the specified B<$node>, in the optional B<@context>, if $node is first under its parent which must also be first under its parent, so that $node is prior to its grandparent. Return the specified B<$node> on success else return B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isFirst;                                           # The node must be first
  return undef unless my $parent = $node->parent;                               # Not in specified context
  return undef unless $parent->isFirst;                                         # The parent must also be first
  return undef unless my $qarent = $parent->parent;                             # The parent parent node
  $qarent->putPrevCut($node);                                                   # Move node into position
 }

# Move a block                                                                  # Move a block of siblings to a new position

sub moveBlockFirst($$$@)                                                        #C Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $end, $parent, @context) = @_;                                    # First sibling, last sibling, parent to move first under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapTo($end, q(grab));                                        # Grab the nodes in the block
  $parent->putFirstCut($w);                                                     # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockLast($$$@)                                                         #C Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $end, $parent, @context) = @_;                                    # First sibling, last sibling, parent to move last under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapTo($end, q(grab));                                        # Grab the nodes in the block
  $parent->putLastCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockAfter($$$@)                                                        #C Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $end, $after, @context) = @_;                                     # First sibling, last sibling, node to move after, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapTo($end, q(grab));                                        # Grab the nodes in the block
  $after->putNextCut($w);                                                       # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockBefore($$$@)                                                       #C Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $end, $before, @context) = @_;                                    # First sibling, last sibling, node to move before, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapTo($end, q(grab));                                        # Grab the nodes in the block
  $before->putPrevCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockToLastFirst($$@)                                                   #C Move the siblings starting with B<$start> in the optional context first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $parent, @context) = @_;                                          # Start sibling, parent to move first under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapToLast(q(grab));                                          # Grab the nodes in the block
  $parent->putFirstCut($w);                                                     # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockToLastLast($$@)                                                    #C Move the block of siblings starting with B<$start> in the optional context last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $parent, @context) = @_;                                          # First sibling, parent to move last under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapToLast(q(grab));                                          # Grab the nodes in the block
  $parent->putLastCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockToLastAfter($$@)                                                   #C Move the block of siblings starting with B<$start> in the optional context after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $after, @context) = @_;                                           # First sibling, node to move after, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapToLast(q(grab));                                          # Grab the nodes in the block
  $after->putNextCut($w);                                                       # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockToLastBefore($$@)                                                  #C Move the block of siblings starting with B<$start> in the optional context before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $before, @context) = @_;                                          # First sibling, node to move before, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapToLast(q(grab));                                          # Grab the nodes in the block
  $before->putPrevCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockFromFirstFirst($$@)                                                #C Move the siblings starting with B<$start> in the optional context first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $parent, @context) = @_;                                          # Start sibling, parent to move first under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapFromFirst(q(grab));                                       # Grab the nodes in the block
  $parent->putFirstCut($w);                                                     # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockFromFirstLast($$@)                                                 #C Move the block of siblings starting with B<$start> in the optional context last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $parent, @context) = @_;                                          # First sibling, parent to move last under, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapFromFirst(q(grab));                                       # Grab the nodes in the block
  $parent->putLastCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockFromFirstAfter($$@)                                                #C Move the block of siblings starting with B<$start> in the optional context after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $after, @context) = @_;                                           # First sibling, node to move after, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapFromFirst(q(grab));                                       # Grab the nodes in the block
  $after->putNextCut($w);                                                       # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

sub moveBlockFromFirstBefore($$@)                                               #C Move the block of siblings starting with B<$start> in the optional context before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.
 {my ($start, $before, @context) = @_;                                          # First sibling, node to move before, optional context
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $w = $start->wrapFromFirst(q(grab));                                       # Grab the nodes in the block
  $before->putPrevCut($w);                                                      # Move block
  $w->unwrap;                                                                   # Unwrap block in new position
  $start                                                                        # Success
 }

#D2 Add selectively                                                             # Add new nodes unless they already exist.

sub addFirst($$@)                                                               #CU Add a new node L<first|/first> below the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node.
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $f = $node->first)                                                     # Existing first node
   {return $f if $f->tag eq $tag;                                               # Return existing first node with matching tag
   }
  $node->putFirst($node->newTag($tag))                                          # Create a new node, place it first below the specified node and return the new node.
 }

#a
#b <b/>
#c addFirst c b
#d Add a first child if needed.

sub addNext($$@)                                                                #CU Add a new node L<next|/next> the specified B<$node> and return the new node unless a node with that tag already exists in which case return the existing B<$node>.
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $n = $node->next)                                                      # Existing next node
   {return $n if $n->tag eq $tag;                                               # Return existing next node with matching tag
   }
  $node->putNext($node->newTag($tag))                                           # Create a new node, place it next to the specified node and return the new node.
 }

#a addFirst
#b <b/>
#c addNext c b
#d Add a next child if needed.

sub addPrev($$@)                                                                #CU Add a new node L<before|/prev> the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node.
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $p = $node->prev)                                                      # Existing previous node
   {return $p if $p->tag eq $tag;                                               # Return existing previous node with matching tag
   }
  $node->putPrev($node->newTag($tag))                                           # Create a new node, place it before the specified node and return the new node.
 }

sub addLast($$@)                                                                #CU Add a new node L<last|/last> below the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node..
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $l = $node->last)                                                      # Existing last node
   {return $l if $l->tag eq $tag;                                               # Return existing first node with matching tag
   }
  $node->putLast($node->newTag($tag))                                           # Create a new node, place it last below the specified node and return the new node.
 }

sub addWrapWith($$@)                                                            #CU L<Wrap|/wrap> the specified B<$node> with the specified tag if the node is not already wrapped with such a tag and return the new node unless a node with that tag already exists in which case return the existing B<$node>.
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $l = $node->parent)                                                    # Existing wrapping node
   {return $l if $l->tag eq $tag;                                               # Return existing first node with matching tag
   }
  $node->wrapWith($tag)                                                         # Wrap with the specified node
 }

sub addSingleChild($$@)                                                         #CU Wrap the content of a specified B<$node> in a new node with the specified B<$tag> unless the content is already wrapped in a single child with the specified B<$tag>.
 {my ($node, $tag, @context) = @_;                                              # Node, tag of new node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $c = $node->hasSingleChild)                                            # Return the existing child if it is an only child and has the right tag
   {return $c if -t $c eq $tag;
   }
  &wrapContentWith(@_);                                                         # Normal wrap content with new node
 }

#D2 Add text selectively                                                        # Add new text unless it already exists.

sub addFirstAsText($$@)                                                         #CU Add a new text node first below the specified B<$node> and return the new node unless a text node already exists there and starts with the same text in which case return the existing B<$node>.
 {my ($node, $text, @context) = @_;                                             # Node, text, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $f = $node->first)                                                     # Existing first node
   {return $f if $f->isText and $f->text =~ m(\A$text);                         # Return existing first node if is a text node with the same starting text
   }
  $node->putFirstAsText($text)                                                  # Create a new text node, place it first below the specified node and return the new text node.
 }

sub addNextAsText($$@)                                                          #CU Add a new text node after the specified B<$node> and return the new node unless a text node already exists there and starts with the same text in which case return the existing B<$node>.
 {my ($node, $text, @context) = @_;                                             # Node, text, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $n = $node->next)                                                      # Existing next node
   {return $n if $n->isText and $n->text =~ m(\A$text);                         # Return existing next node if is a text node with the same starting text
   }
  $node->putNextAsText($text)                                                   # Create a new text node, place it after the specified node and return the new text node.
 }

sub addPrevAsText($$@)                                                          #CU Add a new text node before the specified B<$node> and return the new node unless a text node already exists there and ends with the same text in which case return the existing B<$node>.
 {my ($node, $text, @context) = @_;                                             # Node, text, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $p = $node->prev)                                                      # Existing previous node
   {return $p if $p->isText and $p->text =~ m($text\Z);                         # Return existing previous node if is a text node with the same ending text
   }
  $node->putPrevAsText($text)                                                   # Create a new text node, place it before the specified node and return the new text node.
 }

sub addLastAsText($$@)                                                          #CU Add a new text node last below the specified B<$node> and return the new node unless a text node already exists there and ends with the same text in which case return the existing B<$node>.
 {my ($node, $text, @context) = @_;                                             # Node, text, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $l = $node->last)                                                      # Existing last node
   {return $l if $l->isText and $l->text =~ m($text\Z);                         # Return existing last node if is a text node with the same ending text
   }
  $node->putLastAsText($text)                                                   # Create a new text node, place it last below the specified node and return the new text node.
 }

sub joinWithText($$$@)                                                          #CU Insert some text between the children of the current B<$node> as specified by B<$text> between all the children of the current $node as long as they all have a tag of B<$over>. Return the current $node on success or B<undef> on failure.
 {my ($node, $text, $over, @context) = @_;                                      # Node, text, child tag, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if ($node->over2(qr(\A(\s$over\s){2,}\Z)))                                    # Check tags of children
   {my @c = @$node;                                                             # Children of node
    pop @c;                                                                     # Insert between children
    $_->putNextAsText($text) for @c;                                            # Insertion text offset from existing children
    return $node;                                                               # Success
   }
  undef                                                                         # Not all children have required tag
 }
#a
#b <b/><b/><b/><b/>
#c joinWithText plus b a
#d Insert text between each child.

#D2 Fission                                                                     # Split the parent L<before|/splitBefore> or L<after|/splitAfter> the specified sibling.

sub splitBefore($@)                                                             #CU Split the parent node into two identical nodes except all the siblings before the specified B<$node> are retained by the existing parent while any following siblings become siblings of the new parent node which is placed after the existing parent. The new parent is returned.
 {my ($node, @context) = @_;                                                    # Node to split before, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node->parent;                                                        # Parent
  $p or confess "Cannot split before the root node";                            # Complain if we try to split before the root node
  my @c = ($node, $node->contentAfter);                                         # Content of new node
  my $q = $p->dupTag;                                                           # New node
  $q->putFirstCut($_) for reverse @c;                                           # Move each node after to new parent avoiding text eliding which will occur if the preceding element is text and one of the siblings is also a text segment
  $p->putNext($q);                                                              # Place new node after parent
 }

sub splitAfter($@)                                                              #CU Split the parent node into two identical nodes except all the siblings after the specified B<$node> are retained by the existing parent while any preceding siblings become siblings of the new parent node which is placed before the existing parent. The new parent is returned
 {my ($node, @context) = @_;                                                    # Node to split before, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $p = $node->parent;                                                        # Parent
  $p or confess "Cannot split before the root node";                            # Complain if we try to split before the root node
  my @c = ($node->contentBefore, $node);                                        # Content of new node
  my $q = $p->dupTag;                                                           # New node
  $q->putLastCut($_) for @c;                                                    # Move each node before to new parent
  $p->putPrev($q);                                                              # Place new node before parent
 }

#D2 Fusion                                                                      # Join consecutive nodes

sub concatenate($$@)                                                            #C Concatenate two successive nodes and return the B<$target> node.
 {my ($target, $source, @context) = @_;                                         # Target node to replace, node to concatenate, optional context of $target
  return undef if @context and !$target->at(@context);                          # Not in specified context
  $source->parser or confess "Cannot concatenate the root node";                # Complain if we try and concatenate the root
  if ($source = $target->next)
   {$target->content = [$target->contents, $source->contents];                  # Concatenate (target, source) to target
   }
  elsif ($source = $target->prev)
   {$target->content = [$source->contents, $target->contents];                  # Concatenate (source, target) to target
   }
  else
   {confess "Cannot concatenate non consecutive nodes";                         # Complain if the nodes are not adjacent
   }
  $source->content = [];                                                        # Concatenate
  $target->indexNode;                                                           # Index target node
  $source->indexNode;                                                           # Index source node
  $source->cut;
  $target                                                                       # Return new node
 }

sub concatenateSiblings($@)                                                     #CU Concatenate the nodes that precede and follow the specified B<$node> in the optional B<@context> as long as they have the same tag as the specified B<$node> and return the specified B<$node>.
 {my ($node, @context) = @_;                                                    # Concatenate around this node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $t = $node->tag;                                                           # The tag to match
  while(my $p = $node->prev)
   {last unless $p->tag eq $t;                                                  # Stop when the siblings no longer match
    $node->concatenate($p)
   }
  while(my $n = $node->next)
   {last unless $n->tag eq $t;                                                  # Stop when the siblings no longer match
    $node->concatenate($n) if $n->tag eq $t
   }
  $node                                                                         # Return concatenating node
 }

sub mergeDuplicateChildWithParent($@)                                           #CU Merge a parent node with its only child if their tags are the same and their attributes do not collide other than possibly the id in which case the parent id is used. Any labels on the child are transferred to the parent. The child node is then unwrapped and the parent node is returned.
 {my ($parent, @context) = @_;                                                  # Parent this node, optional context.
  return undef if @context and !$parent->at(@context);                          # Not in specified context
  return undef unless my $child = $parent->hasSingleChild;                      # Not an only child
  return undef unless $child->tag eq $parent->tag;                              # Tags differ
  my %c = %{$child->attributes};                                                # Child attributes
  my %p = %{$parent->attributes};                                               # Parent attributes
  $p{id} = $c{id} unless $p{id};                                                # Transfer child id unless parent already has one
  delete $c{id};                                                                # Remove child id
  for(sort keys %c)                                                             # Remaining attributes
   {return undef if $p{$_} and $p{$_} ne $c{$_};                                # Attributes collide
    $p{$_} = $c{$_};                                                            # Transfer non colliding attribute
   }
  $parent->attributes = \%p;                                                    # Transfer the attributes en masses as none of them collide
  $child->copyLabels($parent);                                                  # Copy child labels to parent
  $child->unwrap;                                                               # Unwrap child
  $parent                                                                       # Return original node
 }

#D2 Put as text                                                                 # Add text to the L<parse|/parse> tree.

sub putFirstAsText($$@)                                                         #CU Add a new text node first under a parent and return the new text node.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be added which might contain unparsed Xml as well as text, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putFirst(my $t = $node->newText($text));                               # Add new text node
  $t                                                                            # Return new node
 }

sub putTextFirst($@)                                                            #U Given a B<$node> place some B<@text> first under the $node and return the new text node else return B<undef> if this is not possible.
 {my ($node, @text) = @_;                                                       # The parent node, the text to be added
  $node->putFirst(my $t = $node->newText(join ' ', @text));                     # Add new text node
  $t                                                                            # Return new node
 }
#a
#b <b><c/></b>
#c at b
#c putTextFirst I shall count the seconds
#d Put some text first under the current node.
sub putLastAsText($$@)                                                          #CU Add a new text node last under a parent and return the new text node.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be added which might contain unparsed Xml as well as text, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putLast(my $t = $node->newText($text));                                # Add new text node
  $t                                                                            # Return new node
 }

sub putTextLast($@)                                                             #U Given a B<$node> place some B<@text> last under the $node and return the new text node else return B<undef> if this is not possible.
 {my ($node, @text) = @_;                                                       # The parent node, the text to be added
  $node->putLast(my $t = $node->newText(join ' ', @text));                      # Add new text node
  $t                                                                            # Return new node
 }
#a
#b <b><c/></b>
#c at b
#c putTextLast Each second will be an hour
#d Put some text last under the current node.

sub putNextAsText($$@)                                                          #CU Add a new text node following the specified B<$node> and return the new text node.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be added which might contain unparsed Xml as well as text, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putNext(my $t = $node->newText($text));                                # Add new text node
  $t                                                                            # Return new node
 }

sub putTextNext($@)                                                             #U Given a B<$node> place some B<@text> next to the $node and return the new text node else return B<undef> if this is not possible.
 {my ($node, @text) = @_;                                                       # The parent node, the text to be added
  $node->putNext(my $t = $node->newText(join ' ', @text));                      # Add new text node
  $t                                                                            # Return new node
 }

sub putPrevAsText($$@)                                                          #CU Add a new text node following the specified B<$node> and return the new text node
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be added which might contain unparsed Xml as well as text, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putPrev(my $t = $node->newText($text));                                # Add new text node
  $t                                                                            # Return new node
 }

sub putTextPrev($@)                                                             #U Given a B<$node> place some B<@text> prior to the $node and return the new text node else return B<undef> if this is not possible.
 {my ($node, @text) = @_;                                                       # The parent node, the text to be added
  $node->putPrev(my $t = $node->newText(join ' ', @text));                      # Add new text node
  $t                                                                            # Return new node
 }

#D2 Put as tree                                                                 # Add parsed text to the L<parse|/parse> tree.

sub putFirstAsTree($$@)                                                         #C Put parsed text first under the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be parsed and added, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putFirst(new($text))                                                   # Add new parse tree first
 }

sub putLastAsTree($$@)                                                          #C Put parsed text last under the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be parsed and added, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putLast(new($text))                                                    # Add new parse tree last
 }

sub putNextAsTree($$@)                                                          #C Put parsed text after the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be parsed and added, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putNext(new($text))                                                    # Add new parse tree
 }

sub putPrevAsTree($$@)                                                          #C Put parsed text before the specified B<$parent> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.
 {my ($node, $text, @context) = @_;                                             # The parent node, the string to be parsed and added, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putPrev(new($text))                                                    # Add new parse tree
 }

#D2 Put as comment                                                              # Add some comments to the parse tree relative to the specified B<$node> in the optional B<$context> and return the specified B<$node>.

sub putFirstAsComment($$@)                                                      #CU Put a comment first under the specified B<$node> and return the specified B<$node>.
 {my ($node, $text, @context) = @_;                                             # Parent node, comment, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putFirstAsText(qq(<!-- $text -->));                                    # Put comment first
 }

sub putLastAsComment($$@)                                                       #CU Put a comment last under the specified B<$node> and return the specified B<$node>.
 {my ($node, $text, @context) = @_;                                             # Parent node, comment, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putLastAsText(qq(<!-- $text -->));                                     # Put comment last
 }

sub putNextAsComment($$@)                                                       #CU Put a comment after the specified B<$node> and return the specified B<$node>
 {my ($node, $text, @context) = @_;                                             # Node, comment, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putNextAsText(qq(<!-- $text -->));                                     # Put comment as next node
 }

sub putPrevAsComment($$@)                                                       #CU Put a comment before the specified B<$parent> parent and return the specified B<$node>
 {my ($node, $text, @context) = @_;                                             # Node, comment, context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->putPrevAsText(qq(<!-- $text -->));                                     # Put comment as previous node
 }

#D2 Break in and out                                                            # Break nodes out of nodes or push them back

sub breakIn($@)                                                                 #CU Concatenate the nodes following and preceding the start node, unwrapping nodes whose tag matches the start node and return the start node. To concatenate only the preceding nodes, use L<breakInBackwards|/breakInBackwards>, to concatenate only the following nodes, use L<breakInForwards|/breakInForwards>.
 {my ($start, @context) = @_;                                                   # The start node, optional context.
  return undef if @context and !$start->at(@context);                           # Not in specified context
  $start->breakInBackwards;                                                     # The nodes before the start node
  $start->breakInForwards                                                       # The nodes following the start node
 }

sub breakInForwards($@)                                                         #CU Concatenate the nodes following the start node, unwrapping nodes whose tag matches the start node and return the start node in the manner of L<breakIn|/breakIn>.
 {my ($start, @context) = @_;                                                   # The start node, optional context..
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $tag     = $start->tag;                                                    # The start node tag
  for my $item($start->contentAfter)                                            # Each item following the start node
   {$start->putLast($item->cut);                                                # Concatenate item
    if ($item->tag eq $tag)                                                     # Unwrap items with the same tag as the start node
     {$item->unwrap;                                                            # Start a new clone of the parent
     }
   }
  $start                                                                        # Return the start node
 }

sub breakInBackwards($@)                                                        #CU Concatenate the nodes preceding the start node, unwrapping nodes whose tag matches the start node and return the start node in the manner of L<breakIn|/breakIn>.
 {my ($start, @context) = @_;                                                   # The start node, optional context..
  return undef if @context and !$start->at(@context);                           # Not in specified context
  my $tag     = $start->tag;                                                    # The start node tag
  for my $item(reverse $start->contentBefore)                                   # Each item preceding the start node reversing from the start node
   {$start->putFirst($item->cut);                                               # Concatenate item
    if ($item->tag eq $tag)                                                     # Unwrap items with the same tag as the start node
     {$item->unwrap;                                                            # Start a new clone of the parent
     }
   }
  $start                                                                        # Return the start node
 }

sub breakOut($@)                                                                #U Lift child nodes with the specified tags under the specified parent node splitting the parent node into clones and return the cut out original node.
 {my ($parent, @tags) = @_;                                                     # The parent node, the tags of the modes to be broken out.
  my %tags       = map {$_=>1} @tags;                                           # Tags to break out
  my %attributes = %{$parent->attributes};                                      # Attributes of parent
  my $parentTag  = $parent->tag;                                                # The tag of the parent
  my $p;                                                                        # Clone of parent currently being built
  for my $item($parent->contents)                                               # Each item
   {if ($tags{$item->tag})                                                      # Item to break out
     {$parent->putPrev($item->cut);                                             # Position item broken out
      $p = undef;                                                               # Start a new clone of the parent
     }
    else                                                                        # Item to remain in situ
     {if (!defined($p))                                                         # Create a new parent clone
       {$parent->putPrev($p = $parent->newTag($parent->tag, %attributes));      # Position new parent clone
       }
      $p->putLast($item->cut);                                                  # Move current item into parent clone
     }
   }
  $parent->cut                                                                  # Remove the original copy of the parent from which the clones were made
 }

sub breakOutChild($@)                                                           #CU Lift the specified B<$node> up one level splitting its parent. Return the specified B<$node> on success or B<undef> if the operation is not possible.
 {my ($node, @context) = @_;                                                    # Node to break out, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my $parent = $node->parent;                                                   # Parent node
  return undef unless $parent;                                                  # Cannot break out the root node
  my $grandParent = $parent->parent;                                            # Parent of parent node
  return undef unless $grandParent;                                             # Cannot break out of the root node
  my %attributes = %{$parent->attributes};                                      # Attributes of parent
  my $parentTag  = $parent->tag;                                                # The tag of the parent
  my $prevParent = $parent->putPrev($parent->newTag($parent->tag, %attributes));# Position new parent clone
  my $p;                                                                        # Clone of parent currently being built
  for my $item($node->contentBefore)                                            # Each preceding sibling
   {$prevParent->putLast($item->cut);                                           # Reposition preceding sibling
   }
  $parent->putPrev($node->cut);                                                 # Reposition child
  $node                                                                         # Return broken out node
 }

#D2 Split and Zip                                                               # Move a node up the parse tree by splitting its parent nodes

sub splitParentAfter($@)                                                        #CU Finish and restart the parent of the specified B<$node> just after the specified B<$node> and return the newly created parent node on success or B<undef> on failure.
 {my ($node, @context) = @_;                                                    # The splitting node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless my $p = $node->parent;                                    # Parent to split
  my @c = $p->contents;                                                         # Siblings to keep
  my @C;                                                                        # Siblings to reparent
  unshift @C, pop @c while @c && $c[-1] != $node;                               # Nodes to keep as is
  $p->content = [@c];                                                           # Content for existing tag
  my $P = $p->dupTag;                                                           # Duplicate  a node
  $p->putNext($P);                                                              # Insert new node
  $P->content = [@C];                                                           # Content for new tag
  $_->indexNode for $p, $P;                                                     # Reindex parent nodes
  $P
 }

BEGIN{*spa=*splitParentAfter}

sub splitParentBefore($@)                                                       #CU Finish and restart the parent of the specified B<$node> just before the specified B<$node> and return the newly created parent node on success or B<undef> on failure.
 {my ($node, @context) = @_;                                                    # The splitting node, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless my $p = $node->parent;                                    # Parent to split
  my @C = $p->contents;                                                         # Siblings to keep
  my @c;                                                                        # Siblings to reparent
  push @C, shift @c while @c && $c[-1] != $node;                                # Nodes to keep as is
  $p->content = [@c];                                                           # Content for existing tag
  my $P = $p->dupTag;                                                           # Duplicate  a node
  $p->putPrev($P);                                                              # Insert new node
  $P->content = [@C];                                                           # Content for new tag
  $_->indexNode for $p, $P;                                                     # Reindex parent nodes
  $P
 }

BEGIN{*spb=*splitParentBefore}

sub splitTo($$@)                                                                #C Lift the specified B<$node> up until it splits the specified B<$parent> node. Return the specified B<$node> on success or B<undef> if the operation is not possible.
 {my ($node, $parent, @context) = @_;                                           # Node to break out, ancestral node to split, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef if $node == $parent;                                             # Not breakable
  return undef unless $node->parent;                                            # The node has to have a parent to split
  return undef unless $parent->parent;                                          # The parent will have to have a parent as well if the parent is to be splittable.
  my @p = $node->belowPath($parent);                                            # Path to parent
  return undef unless @p;                                                       # No path to parent
  $node->breakOutChild for 1..@p;                                               # Break up along the path
  return $node;                                                                 # Return child in new position
 }

sub adoptChild($$@)                                                             #C Lift the specified B<$child> node up until it is an immediate child of the specified B<$parent> node by splitting any intervening nodes. Return the specified B<$child> node on success or B<undef> if the operation is not possible.
 {my ($parent, $child, @context) = @_;                                          # Adopting parent node, child node to adopt, optional context of child
  return undef if @context and !$child->at(@context);                           # Child not in specified context
  return undef if $child == $parent;                                            # Cannot adopt oneself
  return undef unless $child->parent;                                           # Cannot adopt the root
  my @p = $child->belowPath($parent);                                           # Path to parent
  return undef unless @p;                                                       # No path to parent
  pop @p;                                                                       # Split up to the parent but not the parent
  $child->breakOutChild for 1..@p;                                              # Break up along the path
  return $child;                                                                # Return child in new position
 }

sub zipDownOnce($@)                                                             #CU Push a node down one level by making it a child of a node formed by merging the preceding and following siblings if they have the same tag.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  my $n = $node->next;                                                          # Next siblings
  my $p = $node->prev;                                                          # Previous sibling
  return undef unless $n and $p and $n->tag eq $p->tag;                         # Next and previous siblings must have same tag
  $p->putLastCut($_) for $node, @{$n->content};                                 # Extend previous sibling
  $n->cut;                                                                      # Remove next sibling
  $node                                                                         # Return node in new position
 }

sub zipDown($@)                                                                 #CU Push a node down as many levels as possible by making it a child of a node formed by merging the preceding and following siblings if they have the same tag.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  undef while $node->zipDownOnce;                                               # Zip down for as long as possible
  $node                                                                         # Return node in new position
 }

sub splitAndWrapFromStart($$$@)                                                 #CU Split the child nodes of B<$parent> on the child nodes with a tag of B<$split> wrapping the splitting node and all preceding nodes from the previous splitting node or the start with the specified B<$wrap>ping node with optional B<%attributes>.  Returns an array of the wrapping nodes created.
 {my ($parent, $split, $wrap, %attributes) = @_;                                # Parent node, tag of splitting nodes, tag for wrapping node, attributes for wrapping nodes
  my @s = $parent->c($split);                                                   # Splitting nodes
  my @w;                                                                        # Wrapping nodes
  for my $i(reverse keys @s)                                                    # Each splitting node
   {my $s = $s[$i];
    if ($i)                                                                     # From previous split
     {push @w, $s[$i-1]->next->wrapTo($s, $wrap, %attributes);                  # Save wrapping node
     }
    else                                                                        # From start
     {push @w, $s->wrapFromFirst($wrap, %attributes);                           # Save wrapping node
     }
   }
  @w
 }

sub splitAndWrapToEnd($$$@)                                                     #CU Split the sequence of child nodes under the specified B<$parent> node on those child nodes whose tag value is B<$split> wrapping the splitting node and all following nodes up until the next splitting node or the end of the sequence with newly created nodes whose tag is B<$wrap> with optional attributes B<%attributes>.  Returns an array of the wrapping nodes so created.
 {my ($parent, $split, $wrap, %attributes) = @_;                                # Parent node, tag of splitting nodes, tag for wrapping node, attributes for wrapping nodes
  my @s = $parent->c($split);                                                   # Splitting nodes
  my @w;                                                                        # Wrapping nodes
  for my $i(keys @s)                                                            # Each splitting node
   {my $s = $s[$i];
    if ($i != $#s)                                                              # More splits to come
     {push @w, $s->wrapTo($s[$i+1]->prev, $wrap, %attributes);                  # Save wrapping node
     }
    else                                                                        # Last splitting node
     {push @w, $s->wrapToLast($wrap, %attributes);                              # Save wrapping node
     }
   }
  @w
 }

#D2 Replace                                                                     # Replace nodes in the L<parse|/parse> tree with nodes or text

sub replaceWith($$@)                                                            #C Replace a node (and all its content) with a L<new node|/newTag> (and all its content) and return the new node. If the node to be replaced is the root of the L<parse|/parse> tree then no action is taken other then returning the new node.
 {my ($old, $new, @context) = @_;                                               # Old node, new node, optional context..
  return undef if @context and !$old->at(@context);                             # Not in specified context
  $new->parent and confess "Please cut out the node before moving it";          # The node must have be cut out first
  $new->parser == $new and $old->parser == $new and                             # Prevent a root node from being inserted into a sub tree
    confess "Recursive replacement attempted";

  if (my $parent = $old->parent)                                                # Parent node of old node
   {my $c = $parent->content;                                                   # Content array of parent
    if (defined(my $i = $old->position))                                        # Position of old node in content array of parent
     {splice(@$c, $i, 1, $new);                                                 # Replace old node with new node
      $old->parent = undef;                                                     # Cut out node
      $parent->indexNode;                                                       # Rebuild indices for parent
      return $new;                                                              # Return new node
     }
    confess "Should not happen";
   }
  else                                                                          # Replacing the root - merge the new node rather than replacing it so that we do not loose addressability to the old tree
   {$old->tag        = $new->tag;
    $old->content    = $new->content;
    $old->attributes = $new->attributes;
    return $old;
   }
 }

sub replaceWithText($$@)                                                        #CU Replace a node (and all its content) with a new text node and return the new node.
 {my ($old, $text, @context) = @_;                                              # Old node, text of new node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  my $n = $old->replaceWith($old->newText($text));                              # Create a new text node, replace the old node and return the result
  $n
 }

sub replaceWithBlank($@)                                                        #CU Replace a node (and all its content) with a new blank text node and return the new node.
 {my ($old, @context) = @_;                                                     # Old node, optional context.
  return undef if @context and !$old->at(@context);                             # Not in specified context
  my $n = $old->replaceWithText(' ');                                           # Create a new text node, replace the old node with a new blank text node and return the result
  $n
 }

sub replaceContentWithMovedContent($@)                                          # Replace the content of a specified target node with the contents of the specified source nodes removing the content from each source node and return the target node.
 {my ($node, @nodes) = @_;                                                      # Target node, source nodes
  my @content;                                                                  # Target content array
  for my $source(@nodes)                                                        # Each source node
   {push @content, $source->contents;                                           # Build target content array
    $source->content = undef;                                                   # Move content
    $source->indexNode;                                                         # Rebuild indices
   }
  $node->content = [@content];                                                  # Insert new content
  $node->indexNode;                                                             # Rebuild indices
  $node                                                                         # Return target node
 }

sub replaceContentWith($@)                                                      # Replace the content of a node with the specified nodes and return the replaced content
 {my ($node, @content) = @_;                                                    # Node whose content is to be replaced, new content
  my @c = $node->contents;                                                      # Content
  $node->content = [@content];                                                  # Insert new content
  $node->indexNode;                                                             # Rebuild indices
  @c                                                                            # Return old content
 }

sub replaceContentWithText($@)                                                  #U Replace the content of a node with the specified texts and return the replaced content
 {my ($node, @text) = @_;                                                       # Node whose content is to be replaced, texts to form new content
  my @c = $node->contents;                                                      # Content
  $node->content = [map {$node->newText($_)} @text];                            # Insert new content
  $node->indexNode;                                                             # Rebuild indices
  @c                                                                            # Return old content
 }

#D2 Swap                                                                        # Swap nodes both singly and in blocks

sub invert($@)                                                                  #CU Swap a parent and child node where the child is the only child of the parent and return the parent.
 {my ($parent, @context) = @_;                                                  # Parent, context
  return undef unless my $child = $parent->hasSingleChild(@context);            # Single child
  my $grandParent = $parent->parent;                                            # Grandparent
  $grandParent or confess "Cannot invert the outer most node";                  # Cannot invert root
  $parent->unwrap;                                                              # Unwrap parent
  my $content = $child->content;                                                # Child content
  $child->content  = [$parent];                                                 # Place parent under child
  $parent->content = $content;                                                  # Place child content under parent
  $child->parent   = $grandParent;                                              # Grand parent
  $parent->parent  = $child;                                                    # Parent
  $_->indexNode for $child, $parent, $grandParent;                              # Index modified nodes
  $parent                                                                       # Return parent
 }

sub invertFirst($@)                                                             #CU Swap a parent and child node where the child is the first child of the parent by placing the parent last in the child. Return the parent.
 {my ($parent, @context) = @_;                                                  # Parent, context
  return undef if @context and !$parent->at(@context);                          # Not in specified context
  return undef unless my $child = $parent->first;                               # First child
  my $grandParent = $parent->parent;                                            # Grandparent
  $grandParent or confess "Cannot invertFirst the outer most node";             # Cannot invert root
  my $i = $parent->index;                                                       # Position of parent in grandparent
  $parent->cut;                                                                 # Cut out parent
  $child->cut;                                                                  # Cut out child
  $child->putLast($parent);                                                     # Put parent last under child
  $grandParent->content->[$i] = $child;                                         # Put child in position in grandparent
  $_->indexNode for $child, $parent, $grandParent;                              # Index modified nodes
  $child->parent = $grandParent;                                                # Grand parent
  $parent                                                                       # Return parent
 }

sub invertLast($@)                                                              #CU Swap a parent and child node where the child is the last child of the parent by placing the parent first in the child. Return the parent.
 {my ($parent, @context) = @_;                                                  # Parent, context
  return undef if @context and !$parent->at(@context);                          # Not in specified context
  return undef unless my $child = $parent->last;                                # Last child
  my $grandParent = $parent->parent;                                            # Grandparent
  $grandParent or confess "Cannot invert the outer most node";                  # Cannot invert root
  my $i = $parent->index;                                                       # Position of parent in grandparent
  $parent->cut;                                                                 # Cut out parent
  $child->cut;                                                                  # Cut out child
  $child->putFirst($parent);                                                    # Put parent last under child
  $grandParent->content->[$i] = $child;                                         # Put child in position in grandparent
  $child->parent = $grandParent;                                                # Grand parent
  $_->indexNode for $child, $parent, $grandParent;                              # Index modified nodes
  $parent                                                                       # Return parent
 }

sub swap($$@)                                                                   #C Swap two nodes optionally checking that the first node is in the specified context and return the first node.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  return undef if @context and !$first->at(@context);                           # First node not in specified context
  confess "First node is above second node" if $first->above($second);          # Check that the first node is not above the second node otherwise the result of the swap would not be a tree
  confess "First node is below second node" if $first->below($second);          # Check that the first node is not above the second node otherwise the result of the swap would not be a tree
  return $first if $first == $second;                                           # Do nothing if the first node is the second node
  my $f = $first ->wrapWith(q(temp));                                           # Wrap first node for transfer
  my $s = $second->wrapWith(q(temp));                                           # Wrap second node for transfer
  $f->putLast($second->cut);                                                    # Transfer second node
  $s->putLast($first ->cut);                                                    # Transfer first node
  $_->unwrap for $f, $s;                                                        # Remove wrapping
  $first                                                                        # Return first node
 }

sub swapFirstSibling($@)                                                        #CU Swap B<$node> with its first sibling node and return B<$node>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $parent = $node->parent;                               # Cannot swap root node
  my ($f1, $f2) = @$parent;                                                     # First two nodes
  return $node if $node == $f1;                                                 # Already the first node
  $node->putNextCut($f1);                                                       # Move first node into position
  return $node if $node == $f2;                                                 # Was second, now first
  $parent->putFirstCut($node);                                                  # Move node first
 }

BEGIN{*sfs=*swapFirstSibling}

sub swapNext($@)                                                                #CU Swap B<$node> with its following node if $B<$node> matches the first element of the specified context and the next node matches the rest.  Return the node that originally followed B<$node> on success or B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, optional context
  my $next = $node->next;                                                       # Next node
  return undef unless $next;                                                    # No following node so no swap possible
  if (@context)                                                                 # Check supplied context
   {my ($c, $d, @c) = @context;
    return undef unless        $node->at($c, @c);                               # Node not in context
    return undef unless !$d or $next->at($d, @c);                               # Following node not in context
   }
  $next->putNextCut($node);                                                     # Swap nodes
  $next                                                                         # Return other node
 }

BEGIN{*sn=*swapNext}

sub swapPrev($@)                                                                #CU Swap B<$node> with its preceding node if $B<$node> matches the first element of the specified context and the previous node matches the rest.  Return the node that originally followed B<$node> on success or B<undef> on failure.
 {my ($node, @context) = @_;                                                    # Node, optional context
  my $prev = $node->prev;                                                       # Previous node
  return undef unless $prev;                                                    # No previous node so no swap possible
  if (@context)                                                                 # Check supplied context
   {my ($c, $d, @c) = @context;
    return undef unless        $node->at($c, @c);                               # Node not in context
    return undef unless !$d or $prev->at($d, @c);                               # Following node not in context
   }
  $prev->putPrevCut($node);                                                     # Swap nodes
  $prev                                                                         # Return other node
 }

BEGIN{*sp=*swapPrev}

sub swapLastSibling($@)                                                         #CU Swap B<$node> with its last sibling node and return B<$node>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $parent = $node->parent;                               # Cannot swap root node
  my ($l1, $l2) = reverse @$parent;                                             # Last two nodes
  return $node if $node == $l1;                                                 # Already the last node
  $node->putPrevCut($l1);                                                       # Move last node into position
  return $node if $node == $l2;                                                 # Was second last, now last
  $parent->putLastCut($node);                                                   # Move node last
 }

BEGIN{*sls=*swapLastSibling}

sub swapTags($$@)                                                               #C Swap the tags of two nodes optionally checking that the first node is in the specified context and return (B<$first>, B<$second>) nodes.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  return undef if @context and !$first->at(@context);                           # First node not in specified context
  my ($t, $T) = ($first->tag, $second->tag);                                    # Tags
  $first ->change($T);
  $second->change($t);
 ($first, $second)                                                              # Return nodes
 }

sub swapTagWithParent($@)                                                       #CU Swap the tags of the specified B<$node> and its parent optionally checking that the B<$node> is in the specified context and return (parent of B<$node>, B<$node>) or () if there is no such parent node.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return () if @context and !$node->at(@context);                               # Node not in specified context
  return () unless my $p = $node->parent;                                       # No parent for node to swap with
  $p->swapTags($node)                                                           # Swap tags
 }

sub reorder($$$@)                                                               #CU If the current B<$node> has a context of (B<$first>, @context) and the preceding node has a context of (B<$second>, B<@context>), then swap the current node with the previous node.Return the current node regardless
 {my ($node, $first, $second, @context) = @_;                                   # Node, first tag, second tag, optional context
  return undef unless $node->at($first, @context);                              # Node not in specified context
  return undef unless my $prev = $node->prev;                                   # No previous node
  return undef unless $prev->at($second, @context);                             # Previous node not in specified context
  $node->putNextCut($prev);                                                     # Reorder nodes
  $node                                                                         # Return the current node
 }
#a
#b <c/><b/>
#c reorder b c
#d Reorder two adjacent nodes if they are out of order.

#D2 Wrap and unwrap                                                             # Wrap and unwrap nodes to alter the depth of the L<parse|/parse> tree
#D3 Wrap                                                                        # Wrap nodes to deepen the L<parse|/parse> tree

sub wrapWith($$@)                                                               #IU Wrap the specified B<$node> in a new node created from the specified B<$tag> in the optional B<@context> forcing the specified $node down to deepen the L<parse|/parse> tree - return the new wrapping node or B<undef> if this is not possible. See L<addWrapWith|/addWrapWith> to perform this operation conditionally.
 {my ($node, $tag, @context) = @_;                                              # Node, tag for the new node or tag, optional context
  return undef if @context and !$node->at(@context);                            # Node not in specified context
  my $new = newTag(undef, $tag);                                                # Create wrapping node
  if (my $par  = $node->parent)                                                 # Parent node exists
   {my $c = $par->content;                                                      # Content array of parent
    my $i = $node->position;                                                    # Position in content array
    splice(@$c, $i, 1, $new);                                                   # Replace node
    $new->parser  = $node->parser;                                              # Assign the new node to the old parser
    $node->parent =  $new;                                                      # Set parent of original node as wrapping node
    $new->parent  =  $par;                                                      # Set parent of wrapping node
    $new->content = [$node];                                                    # Create content for wrapping node
    $par->indexNode;                                                            # Rebuild indices for parent
   }
  else                                                                          # At  the top - no parent
   {$new->parser = $new;                                                        # Assign the new node to the old parser
    $node->by(sub{$_->parser = $new});                                          # Set parser node so ditaTopicHeaders works - not entirely satisfactory but then w have to assume that the parser node is important enough to maintain.
    $new->content = [$node];                                                    # Create content for wrapping node
    $node->parent =  $new;                                                      # Set parent of original node as wrapping node
    $new->parent  =  undef;                                                     # Set parent of wrapping node - there is none
   }
  $new->indexNode;                                                              # Create index for wrapping node
  $new                                                                          # Return wrapping node
 }
#a unwrap addWrapWith
#b <c/>
#c wrapWith b c
#d Wrap a child with a new parent.

sub wrapWithDup($@)                                                             #U Wrap the specified B<$node> in a new node with the same tag as $node in the optional B<@context> making $node an L<only child|/isOnlyChild> of the new node. Return the new wrapping node or B<undef> if this is not possible.
 {my ($node, @context) = @_;                                                    # Node, optional context
  $node->wrapWith($node->tag, @context);
 }

sub wrapUp($@)                                                                  #U Wrap the specified B<$node> in a sequence of new nodes created from the specified B<@tags> forcing the original node down - deepening the L<parse|/parse> tree - return the array of wrapping nodes if want array else the last wrapping node.
 {my ($node, @tags) = @_;                                                       # Node to wrap, tags to wrap the node with - with the uppermost tag rightmost.
  my @n = map {$node = $node->wrapWith($_)} @tags;                              # Wrap up
  wantarray ? @n : @n ? $n[-1] : undef;
 }

sub wrapWithN($$@)                                                              #CU Wrap this B<$node> with the first B<$N> elements of B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of B<$node>. Returns the upper most wrapping node or B<undef> if the specified B<$node> does not match these conditions.
 {my ($node, $N, @context) = @_;                                                # Node, number of tags wrapping tags, tags and optional context
  @context >= $N or confess "Not enough context";                               # Must have at least enough tags to match N e
  my @tags; push @tags, shift @context for 1..$N;                               # Separate context into wrapping tags and context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  for my $tag(@tags)                                                            # Wrap with each wrapping tag
   {$node = $node->wrapWith($tag);                                              # Wrap
   }
  $node                                                                         # Return last wrapping node
 }

sub wrapWithAll($@)                                                             #CU Wrap this B<$node> wrapped with the specified B<@tags> and return the last wrapping node.
 {my ($node, @tags) = @_;                                                       # Node, tags
  for my $tag(@tags)                                                            # Wrap with each wrapping tag
   {$node = $node->wrapWith($tag);                                              # Wrap
   }
  $node                                                                         # Return last wrapping node
 }

BEGIN{*ww=*wrapWithAll}

sub wrapDown($@)                                                                #U Wrap the content of the specified B<$node> in a sequence of new nodes forcing the original node up - deepening the L<parse|/parse> tree - return the array of wrapping nodes if want array else the last wrapping node.
 {my ($node, @tags) = @_;                                                       # Node to wrap, tags to wrap the node with - with the uppermost tag rightmost.
  my @n = map {$node = $node->wrapContentWith($_)} @tags;                       # Wrap down
  wantarray ? @n : @n ? $n[-1] : undef;
 }

sub wrapContentWith($$@)                                                        #U Wrap the content of the specified B<$node> in a new node created from the specified B<$tag> and B<%attributes>: the specified B<$node> then contains just the new node which, in turn, contains all the content of the specified B<$node>.\mReturns the new wrapping node.
 {my ($old, $tag, %attributes) = @_;                                            # Node, tag for new node, attributes for new node.
  my $new = newTag(undef, $tag, %attributes);                                   # Create wrapping node
  $new->parser  = $old->parser;                                                 # Assign the new node to the old parser
  $new->content = $old->content;                                                # Transfer content
  $old->content = [$new];                                                       # Insert new node
  $new->indexNode;                                                              # Create indices for new node
  $old->indexNode;                                                              # Rebuild indices for old mode
  $new                                                                          # Return new node
 }

BEGIN{*wcw=*wrapContentWith}

sub wrapContentWithDup($@)                                                      #U Wrap the content if the specified B<$node> in a new node with the same tag as $node in the optional B<@context> making the new node an L<only child|/isOnlyChild> of $node. Return the new wrapping node or B<undef> if this is not possible.
 {my ($node, @context) = @_;                                                    # Node, optional context
  $node->wrapContentWith($node->tag, @context);
 }

sub wrapSiblingsBefore($$@)                                                     #U If there are any siblings before the specified B<$node>, wrap them with a new node created from the specified B<$tag> and B<%attributes> and return the newly created node.\mReturns B<undef> if the specified B<$node> is the first node under its parent.
 {my ($node, $tag, %attributes) = @_;                                           # Node to wrap before, tag for new node, attributes for new node.
  my $first = $node->firstSibling;                                              # First sibling
  return undef if $node == $first;                                              # We are the first sibling so no wrapping is possible
  $first->wrapTo($node->prev, $tag, %attributes);                               # Wrap the preceding nodes
 }

sub wrapFromFirst($$@)                                                          #U Wrap this B<$node> and any preceding siblings with a new node created from the specified B<$tag> and B<%attributes> and return the wrapping node.
 {my ($node, $tag, %attributes) = @_;                                           # Node to wrap before, tag for new node, attributes for new node.
  $node->firstSibling->wrapTo($node, $tag, %attributes);                        # Wrap this node and any preceding nodes
 }

sub wrapFromFirstOrLastIn($$@)                                                  #U Wrap inclusively from the first sibling node to the specified B<$node> or from the last prior sibling node whose tag matches one of the tags in B<@targets> to the specified B<$node> using B<$tag> as the tag of the wrapping node and return the wrapping node.
 {my ($node, $tag, @targets) = @_;                                              # Node at which to start to wrap, wrapping tag, tags at which to end the wrap.
  my $target = $node->prevIn(@targets) // $node->firstSibling;                  # Target node
  $node->wrapFrom($target, $tag);                                               # Wrap from the target node to the specified node inclusively
 }

sub wrapFirstN($$$@)                                                            #U Wrap the first B<$N> nodes under this B<$node> in the optional B<@context> with the specified B<$tag> and return the new node or B<undef> if there are no such nodes to wrap.
 {my ($node, $N, $tag, @context) = @_;                                          # Node, number of nodes to wrap, wrapping tag, optional context
  return () if @context and !$node->at(@context);                               # Node not in specified context
  my @c = $node->contents;                                                      # Content
  pop @c while @c and @c > $N;                                                  # Remove excess content
  return undef unless @c == $N;                                                 # Check we have enough nodes to wrap
  $c[0]->wrapTo($c[-1], $tag);                                                  # Wrap the first $N nodes and return the wrapping node
 }

sub wrapSiblingsBetween($$$@)                                                   # If there are any siblings between the specified B<$node>s, wrap them with a new node created from the specified B<$tag> and B<%attributes>. Return the wrapping node else B<undef> if there are no nodes to wrap.
 {my ($first, $last, $tag, %attributes) = @_;                                   # First sibling, last sibling, tag for new node, attributes for new node.
  my $parent = $first->parent;                                                  # Check parentage
     $parent or confess "Cannot wrap between siblings using the root node";
     $parent == $last->parent or confess "Not siblings";

  my ($i, $j) = map {$_->position} my ($f, $l) = ($first, $last);               # Get the node indexes so we can order them
  return $f->next->wrapWith(          $tag, %attributes) if $i == $j - 2;       # One intervening node
  return $l->next->wrapWith(          $tag, %attributes) if $j == $i - 2;
  return $f->next->wrapTo  ($l->prev, $tag, %attributes) if $i <  $j - 2;       # Several intervening nodes
  return $l->next->wrapTo  ($f->prev, $tag, %attributes) if $j <  $i - 2;
  undef                                                                         # No intervening nodes
 }

sub wrapSiblingsAfter($$@)                                                      #U If there are any siblings after the specified B<$node>, wrap them with a new node created from the specified B<$tag> and B<%attributes> and return the newly created node.\mReturns B<undef> if the specified B<$node> is the last node under its parent.
 {my ($node, $tag, %attributes) = @_;                                           # Node to wrap before, tag for new node, attributes for new node.
  my $last = $node->lastSibling;                                                # Last sibling
  return undef if $node == $last;                                               # We are the last sibling so no wrapping is possible
  $node->next->wrapTo($last, $tag, %attributes);                                # Wrap the following nodes
 }

sub wrapNext($$@)                                                               #U Wrap this B<$node> and the following sibling with a new node created from the specified B<$tag>. If the optional context is specified then it should match the first node, next node and the context of the parent of the B<$node> as far as it is specified.  Return the new node created or B<undef> on failure.
 {my ($node, $tag, @context) = @_;                                              # Node to wrap before, tag for new node, context.
  return undef unless my $next   = $node->next;                                 # Next node
  return undef unless my $parent = $node->parent;                               # Parent node
  return undef if @context and $node->tag ne shift @context;                    # Node context
  return undef if @context and $next->tag ne shift @context;                    # Next node context
  return undef if @context and $parent->tag ne shift @context;                  # Parent context
  $node->wrapTo($next, $tag);                                                   # Wrap this node and the next node
 }

sub wrapNextN($$$@)                                                             #U Wrap this B<$node>and the next B<$N>-1 nodes following this B<$node> in the optional B<@context> with the specified B<$tag> and return the new node or B<undef> if there are no such nodes to wrap.
 {my ($node, $N, $tag, @context) = @_;                                          # Node, number of nodes to wrap, wrapping tag, optional context
  return () if @context and !$node->at(@context);                               # Node not in specified context
  my @c = ($node, $node->contentAfter);                                         # Content after node
  pop   @c while @c and @c > $N;                                                # Remove excess content
  return undef unless @c == $N;                                                 # Check we have enough nodes to wrap
  $c[0]->wrapTo($c[-1], $tag);                                                  # Wrap the next $N nodes and return the wrapping node
 }

sub wrapPrev($$@)                                                               #U Wrap this B<$node> and the preceding sibling with a new node created from the specified B<$tag>. If the optional context is specified then it should match the first node, previous node and the context of the parent of the B<$node> as far as it is specified.  Return the new node created or B<undef> on failure.
 {my ($node, $tag, @context) = @_;                                              # Node to wrap before, tag for new node, context.
  return undef unless my $prev   = $node->prev;                                 # Next node
  return undef unless my $parent = $node->parent;                               # Parent node
  return undef if @context and $node->tag ne shift @context;                    # Node context
  return undef if @context and $prev->tag ne shift @context;                    # Next node context
  return undef if @context and $parent->tag ne shift @context;                  # Parent context
  $prev->wrapTo($node, $tag);                                                   # Wrap this node and the previous node
 }

sub wrapPrevN($$$@)                                                             #U Wrap this B<$node>and the previous B<$N>-1 nodes following this B<$node> in the optional B<@context> with the specified B<$tag> and return the new node or B<undef> if there are no such nodes to wrap.
 {my ($node, $N, $tag, @context) = @_;                                          # Node, number of nodes to wrap, wrapping tag, optional context
  return () if @context and !$node->at(@context);                               # Node not in specified context
  my @c = ($node->contentBefore, $node);                                        # Content after node
  shift @c while @c and @c > $N;                                                # Remove excess content
  return undef unless @c == $N;                                                 # Check we have enough nodes to wrap
  $c[0]->wrapTo($c[-1], $tag);                                                  # Wrap the Prev $N nodes and return the wrapping node
 }

sub wrapToLast($$@)                                                             #U Wrap this B<$node> and any following siblings with a new node created from the specified B<$tag> and B<%attributes> and return the wrapping node.
 {my ($node, $tag, %attributes) = @_;                                           # Node to wrap before, tag for new node, attributes for new node.
  $node->wrapTo($node->lastSibling, $tag, %attributes);                         # Wrap this node and any preceding nodes
 }

sub wrapToLastOrFirstIn($$@)                                                    #U Wrap this B<$node> and any following siblings, up to and including the first sibling that matches one of the specified B<@find> nodes or all following siblings if no such match occurs, with a new node created from the specified B<$tag> and return the new wrapping node.
 {my ($node, $tag, @targets) = @_;                                              # Node at which to start to wrap, wrapping tag, tags at which to end the wrap.
  my $target = $node->nextIn(@targets) // $node->lastSibling;                   # Target node
  $node->wrapTo($target, $tag);                                                 # Wrap this node up to and including the target
 }

sub wrapLastN($$$@)                                                             #U Wrap the last B<$N> nodes under this B<$node> in the optional B<@context> with the specified B<$tag> and return the new node or B<undef> if there are no such nodes to wrap.
 {my ($node, $N, $tag, @context) = @_;                                          # Node, number of nodes to wrap, wrapping tag, optional context
  return () if @context and !$node->at(@context);                               # Node not in specified context
  my @c = $node->contents;                                                      # Content
  shift @c while @c and @c > $N;                                                # Remove excess content
  return undef unless @c == $N;                                                 # Check we have enough nodes to wrap
  $c[0]->wrapTo($c[-1], $tag);                                                  # Wrap the Last $N nodes and return the wrapping node
 }

sub wrapTo($$$@)                                                                #Y Wrap all the nodes from the B<$start> node to the B<$end> node inclusive with a new node created from the specified B<$tag> and B<%attributes> and return the new node.\mReturn B<undef> if the B<$start> and B<$end> nodes are not siblings - they must have the same parent for this method to work.
 {my ($start, $end, $tag, %attributes) = @_;                                    # Start node, end node, tag for the wrapping node, attributes for the wrapping node
  @_ >= 3 or confess "At least three parameters required";                      # The tag of the new node must be supplied!
  my $parent = $start->parent;                                                  # Parent
  confess "Start node has no parent" unless $parent;                            # Not possible unless the start node has a parent
  confess "End node has a different parent" unless $parent == $end->parent;     # Not possible unless the start and end nodes have the same parent
  my $s = $start->position;                                                     # Start position
  my $e = $end->position;                                                       # End position
  confess "End node precedes start node" if $e < $s;                            # End must not precede start node
  $start->putPrev(my $new = $start->newTag($tag, %attributes));                 # Create and insert wrapping node
  my @c = $parent->contents;                                                    # Content of parent
  $_ <  @c ? $new->putLast($c[$_]->cut) : undef for $s+1..$e+1;                 # Move the nodes from start to end into the new node remembering that the new node has already been inserted
  $new                                                                          # Return new node
 }

sub wrapFrom($$$%)                                                              #Y Wrap all the nodes from the B<$start> node to the B<$end> node with a new node created from the specified B<$tag> and B<%attributes> and return the new node.  Return B<undef> if the B<$start> and B<$end> nodes are not siblings - they must have the same parent for this method to work.
 {my ($end, $start, $tag, %attributes) = @_;                                    # End node, start node, tag for the wrapping node, attributes for the wrapping node
  $start->wrapTo($end, $tag, %attributes);                                      # Invert and wrapTo
 }

#D3 Unwrap                                                                      # Unwrap nodes to reduce the depth of the L<parse|/parse> tree

sub unwrap($@)                                                                  #CIYU Unwrap the specified B<$node> if in the optional B<@context> by replacing the node with its contents. Returns the parent node on success, otherwise B<undef> if an attempt is made to unwrap a text node or the root node.
 {my ($node, @context) = @_;                                                    # Node to unwrap, optional context.
  return undef if @context && !$node->at(@context) or $node->isText;            # Not in specified context or a text node
  my $parent = $node->parent;                                                   # Parent node
#  $parent or confess "Cannot unwrap the outer most node";                      # Root nodes cannot be unwrapped
  return undef unless $parent;                                                  # Root nodes cannot be unwrapped
  if ($node->isEmpty)                                                           # Empty nodes can just be cut out
   {$node->cut;
   }
  else
   {my $p = $parent->content;                                                   # Content array of parent
    my $n = $node->content;                                                     # Content array of node
    my $i = $node->position;                                                    # Position of node in parent
    splice(@$p, $i, 1, @$n);                                                    # Replace node with its content
    $parent->indexNode;                                                         # Rebuild indices for parent
    $node->disconnectLeafNode;                                                  # Disconnect node from parse tree
   }
  $parent                                                                       # Return the parent node
 }
#b <b id="b">C</b>
#c unwrap b
#d Delete a node retaining its content.

BEGIN{*u=*unwrap}

sub unwrapParentOfOnlyChild($@)                                                 #CU Unwrap the parent of the specified B<$node> in the optional B<@context> when the B<$node> is the only child of its parent. Return the specified B<$node> regardless unless the node is not in the optional context in which case return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  if ($node->isOnlyChild(@context))                                             # Only child of parent
   {$node->parent->unwrap;                                                      # Unwrap parent
    return $node;                                                               # Success
   }
  undef                                                                         # Not in specified context
 }

sub unwrapOnlyChild($@)                                                         #CU Unwrap the specified B<$node> in the optional B<@context> when the B<$node> is the only child of its parent. Return the specified B<$node> regardless unless the node is not in the optional context in which case return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  if ($node->isOnlyChild(@context))                                             # Only child of parent
   {$node->unwrap;                                                              # Unwrap parent
    return $node;                                                               # Success
   }
  undef                                                                         # Not in specified context
 }

sub unwrapParentsWithSingleChild($@)                                            #CU Unwrap any immediate ancestors of the specified B<$node> in the optional B<@context> which have only a single child and return the specified B<$node> regardless unless the node is not in the optional context in which case return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context && !$node->at(@context) or $node->isText;            # Not in specified context or a text node
  my @p;                                                                        # Parents with single child
  for(my $p = $node->parent; $p; $p = $p->parent)                               # Check each parent
   {$node->isOnlyChild and $p->parent ? push @p, $p : last                      # Locate parents with single child
   }
  $_->unwrap for @p;                                                            # Unwrap parents with single child
  $node                                                                         # Return node
 }

sub unwrapContentsKeepingText($@)                                               #CYU Unwrap all the non text nodes below the specified B<$node> adding a leading and a trailing space to prevent unwrapped content from being elided and return the specified B<$node> else B<undef> if not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Node to unwrap, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->by(sub
   {if (!$_->isText and $_ != $node)                                            # Keep interior text nodes
     {$_->putPrevAsText(" ");                                                   # Separate from preceding content
      $_->putNextAsText(" ");                                                   # Separate from following content
      $node->addLabels($_->id) if $_->id;                                       # Transfer any id as a label to the specified B<$node>
      $_->copyLabels($node);                                                    # Transfer any labels to the specified node
      $_->unwrap;                                                               # Unwrap non text tag
     }
   });
  $node                                                                         # Return the node to show success
 }

sub wrapRuns($$@)                                                               #CU Wrap consecutive runs of children under the specified parent B<$node> that are not already wrapped with B<$wrap>. Returns an array of any wrapping nodes created.  Returns () if the specified B<$node> is not in the optional B<@context>.
 {my ($node, $wrap, @context) = @_;                                             # Node to unwrap, tag of wrapping node, optional context.
  return () if @context and !$node->at(@context);                               # Not in specified context
  my @wrap;                                                                     # Wrapping nodes created
  my @c = @$node;                                                               # Children
  my $w;                                                                        # The latest wrapping node
  for my $child(@c)                                                             # Each child node
   {if ($child->tag eq $wrap)                                                   # Wrapped nodes
     {$w = undef;                                                               # NO wrapper required
     }
    else                                                                        # A child that should be wrapped
     {unless($w)                                                                # No current wrapper
       {$child->putPrev($w = $node->newTag($wrap));                             # Add a wrapper
        push @wrap, $w;                                                         # Add a start node unless we are already have a node of the right type
       }
      $w->putLast($child->cut);                                                 # Put unwrapped child in last wrapper created
     }
   }
  @wrap                                                                         # Return array of new wrapping nodes
 }

#D1 Contents                                                                    # The children of each node.

sub contents($@)                                                                #K Return a list of all the nodes contained by the specified B<$node> or an empty list if the node is empty or not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return () if @context and !$node->at(@context);                               # Optionally check the context
  my $c = $node->content;                                                       # Contents reference
  $c ? @$c : ()                                                                 # Contents as a list
 }

sub contentAfter($@)                                                            #KU Return a list of all the sibling nodes following the specified B<$node> or an empty list if the specified B<$node> is last or not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return () if @context and !$node->at(@context);                               # Optionally check the context
  my $parent = $node->parent;                                                   # Parent
  return () if !$parent;                                                        # The uppermost node has no content beyond it
  my @c = $parent->contents;                                                    # Contents of parent
  while(@c)                                                                     # Test until no more nodes left to test
   {my $c = shift @c;                                                           # Position of current node
    return @c if $c == $node                                                    # Nodes beyond this node if it is the searched for node
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }

BEGIN{*ca=*contentAfter}

sub contentBefore($@)                                                           #KU Return a list of all the sibling nodes preceding the specified B<$node> (in the normal sibling order) or an empty list if the specified B<$node> is last or not in the optional B<@context>.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return () if @context and !$node->at(@context);                               # Optionally check the context
  my $parent = $node->parent;                                                   # Parent
  return () if !$parent;                                                        # The uppermost node has no content beyond it
  my @c = $parent->contents;                                                    # Contents of parent
  while(@c)                                                                     # Test until no more nodes left to test
   {my $c = pop @c;                                                             # Position of current node
    return @c if $c == $node                                                    # Nodes beyond this node if it is the searched for node
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }

BEGIN{*cb=*contentBefore}

sub contentAsTags($@)                                                           #KYU Return a string containing the tags of all the child nodes of the specified B<$node> separated by single spaces or the empty string if the node is empty or B<undef> if the node does not match the optional context. Use L<over|/over> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join ' ', map {$_->tag} $node->contents
 }

sub contentAsTags2($@)                                                          #KYU Return a string containing the tags of all the child nodes of the specified B<$node> separated by two spaces with a single space preceding the first tag and a single space following the last tag or the empty string if the node is empty or B<undef> if the node does not match the optional context. Use L<over2|/over2> to test the sequence of tags with a regular expression. Use L<over2|/over2> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join '', map {' '.$_->tag.' '} $node->contents
 }

sub contentAfterAsTags($@)                                                      #KU Return a string containing the tags of all the sibling nodes following the specified B<$node> separated by single spaces or the empty string if the node is empty or B<undef> if the node does not match the optional context. Use L<matchAfter|/matchAfter> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join ' ', map {$_->tag} $node->contentAfter
 }

sub contentAfterAsTags2($@)                                                     #KU Return a string containing the tags of all the sibling nodes following the specified B<$node> separated by two spaces with a single space preceding the first tag and a single space following the last tag or the empty string if the node is empty or B<undef> if the node does not match the optional context. Use L<matchAfter2|/matchAfter2> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join '', map {' '.$_->tag.' '} $node->contentAfter
 }

sub contentBeforeAsTags($@)                                                     #KU Return a string containing the tags of all the sibling nodes preceding the specified B<$node> separated by single spaces or the empty string if the node is empty or B<undef> if the node does not match the optional context. Use L<matchBefore|/matchBefore> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join ' ', map {$_->tag} $node->contentBefore
 }

sub contentBeforeAsTags2($@)                                                    #KU Return a string containing the tags of all the sibling nodes preceding the specified B<$node> separated by two spaces with a single space preceding the first tag and a single space following the last tag or the empty string if the node is empty or B<undef> if the node does not match the optional context.  Use L<matchBefore2|/matchBefore2> to test the sequence of tags with a regular expression.
 {my ($node, @context) = @_;                                                    # Node, optional context.
  return undef if @context and !$node->at(@context);                            # Optionally check the context
  join '', map {' '.$_->tag.' '} $node->contentBefore
 }

sub position($)                                                                 #U Return the index of the specified B<$node> in the content of the parent of the B<$node>.
 {my ($node) = @_;                                                              # Node.
  my @c = $node->parent->contents;                                              # Each node in parent content
  for(keys @c)                                                                  # Test each node
   {return $_ if $c[$_] == $node;                                               # Return index position of node which counts from zero
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }

sub index($)                                                                    #U Return the index of the specified B<$node> in its parent index. Use L<position|/position> to find the position of a node under its parent.
 {my ($node) = @_;                                                              # Node.
  my $p = $node->parent;                                                        # Parent
  $p or confess "Cannot find index of root node";                               # Complain if we try to find the index of the root node
  if (my @c = $p->c($node->tag))                                                # Each node in parent index
   {for(keys @c)                                                                # Test each node
     {return $_ if $c[$_] == $node;                                             # Return index position of node which counts from zero
     }
   }
  confess "Node not found in parent";                                           # Something wrong with parent/child relationship
 }

sub present($@)                                                                 #U Return the count of the number of the specified tag types present immediately under a node or a hash {tag} = count for all the tags present under the node if no names are specified.
 {my ($node, @names) = @_;                                                      # Node, possible tags immediately under the node.
  reindexNode($node);                                                           # Create index for this node
  my %i = %{$node->indexes};                                                    # Index of child nodes
  return map {$_=>scalar @{$i{$_}}} keys %i unless @names;                      # Hash of all names
  grep {$i{$_}} @names                                                          # Count of tag types present
 }

sub editText($@) :lvalue                                                        #C Return the text of a text B<$node> as an lvalue string reference that can be modified by a regular expression, else as a reference to a dummy string that will be ignored. For a unitary version of this method see: L<changeText>
 {my ($node, @context) = @_;                                                    # Node to test, optional context
  my $dummy = "";                                                               # An empty string reference
  return $dummy if @context and !$node->at(@context);                           # Not in specified context
  return $node->text if $node->isText;                                          # Return reference to text if on a text node
  $dummy                                                                        # Nor on a text node
 }

sub isText($@)                                                                  #UCY Return the specified B<$node> if the specified B<$node> is a text node, optionally in the specified context, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node to test, optional context
  if (@context)                                                                 # Optionally check context
   {my $p = $node->parent;                                                      # Parent
    return undef if !$p or !$p->at(@context);                                   # Parent must match context
   }
  $node->tag eq cdata ? $node : undef
 }

sub isFirstText($@)                                                             #UCY Return the specified B<$node> if the specified B<$node> is a text node, the first node under its parent and that the parent is optionally in the specified context, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node to test, optional context for parent
  return undef unless $node->isText(@context) and $node->isFirst;               # Check that this node is a text node, that it is first, and, optionally check context of parent
  $node                                                                         # Return the node as it passes all tests
 }

sub isLastText($@)                                                              #UCY Return the specified B<$node> if the specified B<$node> is a text node, the last node under its parent and that the parent is optionally in the specified context, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node to test, optional context for parent
  return undef unless $node->isText(@context) and $node->isLast;                # Check that this node is a text node, that it is last, and, optionally check context of parent
  $node                                                                         # Return the node as it passes all tests
 }

sub matchTree($@)                                                               #UC Return a list of nodes that match the specified tree of match expressions, else B<()> if one or more match expressions fail to match nodes in the tree below the specified start node. A match expression consists of [parent node tag, [match expressions to be matched by children of parent]|tags of child nodes to match starting at the first node]. Match expressions for a single item do need to be surrounded with [] and can be merged into their predecessor. The outermost match expression should not be enclosed in [].
 {my ($node, @match) = @_;                                                      # Node to start matching from, tree of match expressions.
  my ($tag, @S) = @match;                                                       # Tag we must match, sub tag specifications
  my @nodes;                                                                    # Matching nodes
  return () unless atPositionMatch(-t $node, $tag);                             # Match tag of node
  push @nodes, $node;                                                           # The current node matches so save it
  if (my @s = map{ref($_) ? $_ : [$_]} @S)                                      # Wrap sub tags that are not references with [] so that the caller does not have to because it is a bit tedious wrapping each word in quotes and [] brackets.
   {my @c = $node->contents;                                                    # Contents of node
    return () unless @s <= @c;                                                  # Confirm that there is a node to match against for each match expression
    for my $c(@c)                                                               # Confirm each sub tag matches its sub tag expression
     {push @nodes, my @m = $c->matchTree(@{shift @s});                          # Save sub match
      return () unless @m;                                                      # No sub match so return an empty list
     }
   }
  @nodes                                                                        # The nodes that matched each match expression
 }

sub matchesText($$@)                                                            #UC Returns an array of regular expression matches in the text of the specified B<$node> if it is text node and it matches the specified regular expression and optionally has the specified context otherwise returns an empty array.
 {my ($node, $re, @context) = @_;                                               # Node to test, regular expression, optional context
  return () unless $node->isText(@context);                                     # Check that this is a text node
  $node->text =~ m($re);                                                        # Return array of matches - do not add 'g' as a modifier to the regular expression as the pos() feature of Perl regular expressions will then cause matches to fail that otherwise would not.
 }

sub isBlankText($@)                                                             #UCY Return the specified B<$node> if the specified B<$node> is a text node, optionally in the specified context, and contains nothing other than white space else return B<undef>. See also: L<isAllBlankText|/isAllBlankText>
 {my ($node, @context) = @_;                                                    # Node to test, optional context
  return undef if @context and !$node->at(@context);                            # Optionally check context
  $node->isText && $node->text =~ /\A\s*\Z/s ? $node : undef
 }

sub isAllBlankText($@)                                                          #UCY Return the specified B<$node> if the specified B<$node>, optionally in the specified context, does not contain anything or if it does contain something it is all white space else return B<undef>. See also: L<bitsNodeTextBlank|/bitsNodeTextBlank>
 {my ($node, @context) = @_;                                                    # Node to test, optional context
  return undef if @context and !$node->at(@context);                            # Optionally check context
  if ($node->isText)                                                            # If this is a text node test the text of the node
   {return $node->text =~ m(\A\s*\Z)s ? $node : undef;
   }
  my @c = $node->contents;
  return $node if @c == 0;                                                      # No content
  return undef if @c  > 1;                                                      # Content other than text (adjacent text elements are merged so there can only be one)
  $node->stringContent =~ m(\A\s*\Z)s ? $node : undef
 }

sub isOnlyChildBlankText($@)                                                    #UC Return the specified B<$node> if it is a blank text node and an only child else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node to test, optional context
  return undef unless &isOnlyChildText(@_);                                     # Confirm that this is an only text child
  $node->text =~ m(\A\s*\Z)s;                                                   # All blank
 }

sub bitsNodeTextBlank($)                                                        #U Return a bit string that shows if there are any non text nodes, text nodes or blank text nodes under a node. An empty string is returned if there are no child nodes.
 {my ($node) = @_;                                                              # Node to test.
  my ($n, $t, $b) = (0,0,0);                                                    # Non text, text, blank text count
  my @c = $node->contents;                                                      # Contents of node
  return '' unless @c;                                                          # Return empty string if no children

  for(@c)                                                                       # Contents of node
   {if ($_->isText)                                                             # Text node
     {++$t;
      ++$b if $_->isBlankText;                                                  # Blank text node
     }
    else                                                                        # Non text node
     {++$n;
     }
   }
  join '', map {$_ ? 1 : 0} ($n, $t, $b);                                       # Multiple content so there must be some tags present because L<indexNode|/indexNode> concatenates contiguous text
 }

#D1 Number                                                                      # Number the nodes of a parse tree so that they can be easily retrieved by number - either by a person reading the source L<XML> or programmatically.

sub findByNumber($$)                                                            #UY Find the node with the specified number as made visible by L<prettyStringNumbered|/prettyStringNumbered> in the L<parse|/parse> tree containing the specified B<$node> and return the found node or B<undef> if no such node exists.
 {my ($node, $number) = @_;                                                     # Node in the parse tree to search, number of the node required.
  $node->parser->numbers->[$number]
 }

sub findByNumbers($@)                                                           #U Find the nodes with the specified numbers as made visible by L<prettyStringNumbered|/prettyStringNumbered> in the L<parse|/parse> tree containing the specified B<$node> and return the found nodes in a list with B<undef> for nodes that do not exist.
 {my ($node, @numbers) = @_;                                                    # Node in the parse tree to search, numbers of the nodes required.
  map {$node->findByNumber($_)} @numbers                                        # Node corresponding to each number
 }

sub numberNode($)                                                               #UP Ensure that the specified B<$node> has a number.
 {my ($node) = @_;                                                              # Node
  my $n = $node->number = ++($node->parser->numbering);                         # Number node
  $node->parser->numbers->[$n] = $node                                          # Index the nodes in a parse tree
 }

sub numberTree($)                                                               #U Number the nodes in a L<parse|/parse> tree in pre-order so they are numbered in the same sequence that they appear in the source. You can see the numbers by printing the tree with L<prettyStringNumbered|/prettyStringNumbered>.  Nodes can be found using L<findByNumber|/findByNumber>.  This method differs from L<forestNumberTrees|/forestNumberTrees> in that avoids overwriting the B<id=> attribute of each node by using a system attribute instead; this system attribute can then be made visible on the id attribute of each node by printing the parse tree with L<prettyStringNumbered|/prettyStringNumbered>.
 {my ($node) = @_;                                                              # Node
  my $parser = $node->parser;                                                   # Top of tree
  my $n = 0;                                                                    # Node number
  $parser->down(sub {$parser->numbers->[$_->number = ++$n] = $_});              # Number the nodes in a parse tree in pre-order so they are numbered in the same sequence that they appear in the source
 }

sub indexIds($)                                                                 #U Return a map of the ids at and below the specified B<$node>.
 {my ($node) = @_;                                                              # Node
  my %ids;                                                                      # Id map
  $node->by(sub                                                                 # Map the nodes
   {if (my $id = $_->id)                                                        # Ignore text nodes and nodes that already have an id
     {$ids{$id} and confess "Duplicate id $id";                                 # Confess to duplicate id under the specified node
      $ids{$id} = $_;                                                           # Index node by id
     }
   });
  \%ids                                                                         # Return the index
 }

sub numberTreesJustIds($$)                                                      #U Number the ids of the nodes in a L<parse|/parse> tree in pre-order so they are numbered in the same sequence that they appear in the source. You can see the numbers by printing the tree with L<prettyStringNumbered()|/prettyStringNumbered>. This method differs from L<numberTree|/numberTree> in that only non text nodes without ids are numbered. The number applied to each node consists of the concatenation of the specified prefix, an underscore and a number that is unique within the specified L<parse|/parse> tree. Consequently the ids across several trees trees can be made unique by supplying different prefixes for each tree.  Nodes can be found using L<findByNumber|/findByNumber>.  Returns the specified B<$node>.
 {my ($node, $prefix) = @_;                                                     # Node, prefix for each id at and under the specified B<$node>
  my $parser = $node->parser;                                                   # Root node
  my $n = 0;                                                                    # Node number
  $node->down(sub                                                               # Number the nodes in the parse tree in pre-order so they are numbered in the same sequence that they appear in the source. Add the prefix to each node number.
   {if (!$_->isText and !$_->id)                                                # Ignore text nodes and nodes that already have an id
     {$_->id = $prefix.++$n;                                                    # Set id with prefix
     }
   }) if $prefix;
  $node->down(sub                                                               # Number the nodes in the parse tree in pre-order so they are numbered in the same sequence that they appear in the source. Add the prefix to each node number.
   {if (!$_->isText and !$_->id)                                                # Ignore text nodes and nodes that already have an id
     {$_->id = ++$n;                                                            # Set id without prefix
     }
   }) unless $prefix;
  $node                                                                         # Return the specified node
 }

#D1 Forest Numbers                                                              # Number the nodes of several parse trees so that they can be easily retrieved by forest number - either by a person reading the source L<XML> or programmatically.

sub forestNumberTrees($$)                                                       #U Number the ids of the nodes in a L<parse|/parse> tree in pre-order so they are numbered in the same sequence that they appear in the source. You can see the numbers by printing the tree with L<prettyString|/prettyString>. This method differs from L<numberTree|/numberTree> in that only non text nodes are numbered and nodes with existing B<id=> attributes have the value of their B<id=> attribute transferred to a L<label|/Labels>. The number applied to each node consists of the concatenation of the specified tree number, an underscore and a number that is unique within the specified L<parse|/parse> tree. Consequently the ids across several trees can be made unique by supplying a different tree number for each tree.  Nodes can be found subsequently using L<findByForestNumber|/findByForestNumber>.  Returns the specified B<$node>.
 {my ($node, $prefix) = @_;                                                     # Node in parse tree to be numbered, tree number
  my $parser = $node->parser;                                                   # Root node
  defined($prefix) or confess "prefix required";                                # Typically the prefix separates parallel processes and so it is a good idea to have one
  $prefix =~ m(\A\d+\Z)s or confess "prefix must be an integer \\d+";           # Insist on a numeric prefix
  $parser->down(sub                                                             # Number the nodes in the parse tree in pre-order so they are numbered in the same sequence that they appear in the source
   {if (!$_->isText)                                                            # Number non text nodes
     {$_->addLabels($_->id) if $_->id;                                          # Make any existing id a label
      my $n  = $prefix.q(_).(scalar(keys %{$parser->forestNumbers}) + 1);       # Id number
      $_->id = $n;                                                              # Set id
      $parser->forestNumbers->{$n} = $_;                                        # Index node
     }
   });
  $node                                                                         # Return the specified node
 }

sub findByForestNumber($$$)                                                     #U Find the node with the specified L<forest number|/forestNumberTrees> as made visible on the id attribute by L<prettyStringNumbered|/prettyStringNumbered> in the L<parse|/parse> tree containing the specified B<$node> and return the found node or B<undef> if no such node exists.
 {my ($node, $tree, $id) = @_;                                                  # Node in the parse tree to search, forest number, id number of the node required.
  $node->parser->forestNumbers->{$tree.q(_).$id}
 }

#D1 Order                                                                       # Check the order and relative position of nodes in a parse tree.

sub above($$@)                                                                  #CY Return the first node if the first node is above the second node optionally checking that the first node is in the specified context otherwise return B<undef>
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  return undef if @context and !$first->at(@context);                           # Not in specified context
  return undef if $first == $second;                                            # A node cannot be above itself
  my @f = $first ->ancestry;
  my @s = $second->ancestry;
  pop @f, pop @s while @f and @s and $f[-1] == $s[-1];                          # Find first different ancestor
  !@f ? $first : undef                                                          # Node is above target if its ancestors are all ancestors of target
 }

sub abovePath($$)                                                               # Return the nodes along the path from the first node down to the second node when the first node is above the second node else return B<()>.
 {my ($first, $second) = @_;                                                    # First node, second node
  return ($first) if $first == $second;                                         # A node cannot be above itself
  my @f = $first ->ancestry;
  my @s = $second->ancestry;
  pop @f, pop @s while @f and @s and $f[-1] == $s[-1];                          # Find first different ancestor
  !@f ? ($first, reverse @s) : ()                                               # Node is above target if its ancestors are all ancestors of target
 }

sub below($$@)                                                                  #CY Return the first node if the first node is below the second node optionally checking that the first node is in the specified context otherwise return B<undef>
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  $second->above($first, @context);                                             # The second node is above the first node if the first node is below the second node
 }

sub belowPath($$)                                                               # Return the nodes along the path from the first node up to the second node when the first node is below the second node else return B<()>.
 {my ($first, $second) = @_;                                                    # First node, second node
  reverse $second->abovePath($first)
 }

sub after($$@)                                                                  #CY Return the first node if it occurs after the second node in the L<parse|/parse> tree optionally checking that the first node is in the specified context or else B<undef> if the node is L<above|/above>, L<below|/below> or L<before|/before> the target.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  return undef if @context and !$first->at(@context);                           # First node not in specified context
  my @n = $first ->ancestry;
  my @t = $second->ancestry;
  pop @n, pop @t while @n and @t and $n[-1] == $t[-1];                          # Find first different ancestor
  return undef unless @n and @t;                                                # Undef if we cannot decide
  $n[-1]->position > $t[-1]->position                                           # Node relative to target at first common ancestor
 }

sub before($$@)                                                                 #CY Return the first node if it occurs before the second node in the L<parse|/parse> tree optionally checking that the first node is in the specified context or else B<undef> if the node is L<above|/above>, L<below|/below> or L<before|/before> the target.
 {my ($first, $second, @context) = @_;                                          # First node, second node, optional context
  $second->after($first, @context);                                             # The first node is before the second node if the second node is after the first node
 }

sub disordered($@)                                                              # Return the first node that is out of the specified order when performing a pre-ordered traversal of the L<parse|/parse> tree.
 {my ($node, @nodes) = @_;                                                      # Node, following nodes.
  my $c = $node;                                                                # Node we are currently checking for
  $node->parser->down(sub {$c = shift @nodes while $c and $_ == $c});           # Preorder traversal from root looking for each specified node
  $c                                                                            # Disordered if we could not find this node
 }

sub commonAncestor($@)                                                          #Y Find the most recent common ancestor of the specified nodes or B<undef> if there is no common ancestor.
 {my ($node, @nodes) = @_;                                                      # Node, @nodes
  return $node unless @nodes;                                                   # A single node is it its own common ancestor
  my @n = $node->ancestry;                                                      # The common ancestor so far
  for(@nodes)                                                                   # Each node
   {my @t = $_->ancestry;                                                       # Ancestry of latest node
    my @c;                                                                      # Common ancestors
    while(@n and @t and $n[-1] == $t[-1])                                       # Find common ancestors
     {push @c, pop @n; pop @t;                                                  # Save common ancestor
     }
    return undef unless @c;                                                     # No common ancestors
    @n = reverse @c;                                                            # Update common ancestry so far
   }
  $n[0]                                                                         # Most recent common ancestor
 }

sub commonAdjacentAncestors($$)                                                 # Given two nodes, find a pair of adjacent ancestral siblings if such a pair exists else return B<()>.
 {my ($first, $second) = @_;                                                    # First node, second node
  my @f = $first->ancestry;                                                     # Ancestors of the first node
  my @s = $second->ancestry;                                                    # Ancestors of the second node
  while(@f and @s and $f[-1] == $s[-1])                                         # Remove common ancestors
   {pop @f; pop @s;
   }
  return () unless @f and @s;                                                   # No common ancestors
  my ($f, $s) = ($f[-1], $s[-1]);                                               # Possible common pair of ancestral siblings
  return ($f, $s) if $f->adjacent($s);                                          # Return first diverging siblings if they exist
  ()                                                                            # No such pair exists
 }

sub ordered($@)                                                                 #Y Return the first node if the specified nodes are all in order when performing a pre-ordered traversal of the L<parse|/parse> tree else return B<undef>.
 {my ($node, @nodes) = @_;                                                      # Node, following nodes.
  &disordered(@_) ? undef : $node
 }

#D1 Patching                                                                    # Analyze two similar L<parse|/parse> trees and create a patch that transforms the first L<parse|/parse> tree into the second as long as each tree has the same tag and id structure with each id being unique.

sub createPatch($$)                                                             # Create a patch that moves the source L<parse|/parse> tree to the target L<parse|/parse> tree node as long as they have the same tag and id structure with each id being unique.
 {my ($a, $A) = @_;                                                             # Source parse tree, target parse tree
  my @patches;
  my $I = $A->indexIds;
  $a->by(sub
   {my ($o) = @_;
    return unless my $id = $o->id;
    if (my $O = $I->{$id})
     {for my $a($o->getAttrs)
       {my ($v, $V)= ($o->attr($a), $O->attr($a));                              # Values
        if (defined($V) and $v ne $V)
         {push @patches, [q(changeValue), $id, $a, $v, $V]                      # Set attribute
         }
        elsif (!defined($V))
         {push @patches, [q(deleteAttr), $id, $a, $v]                           # Set attribute
         }
       }
      for my $A($O->getAttrs)
       {next if $o->attr($A);
        push @patches, [q(createAttr), $id, $A, $O->attr($A)];                  # Set attribute
       }

      if (my $f = $o->first)                                                    # First following text
       {if ($f->isText)
         {if (my $F = $O->first)
           {if ($F->isText)
             {if ($f->text ne $F->text)
               {push @patches, [q(firstText), $id, $f->text, $F->text];
               }
             }
            else
             {confess "Text expected first after node $id:\n".$f->text;
             }
           }
          else
           {confess "Node expected first after node $id:\n".$f->text;
           }
         }
       }

      if (my $n = $o->next)                                                     # Following text
       {if ($n->isText)
         {if (my $N = $O->next)
           {if ($N->isText)
             {if ($n->text ne $N->text)
               {push @patches, [q(nextText), $id, $n->text, $N->text];
               }
             }
            else
             {confess "Text expected after node $id:\n".$n->text;
             }
           }
          else
           {confess "Node expected after node $id:\n".$n->text;
           }
         }
       }

      if (my $p = $o->prev)                                                     # Preceding text
       {if ($p->isText)
         {if (my $P = $O->prev)
           {if ($P->isText)
             {if ($p->text ne $P->text)
               {push @patches, [q(prevText), $id, $p->text, $P->text];
               }
             }
            else
             {confess "Text expected before node $id:\n".$p->text;
             }
           }
          else
           {confess "Node expected before node $id:\n".$p->text;
           }
         }
       }

      if (my $l = $o->last)                                                     # Last preceding text
       {if ($l->isText)
         {if (my $L = $O->last)
           {if ($L->isText)
             {if ($l->text ne $L->text)
               {push @patches, [q(lastText), $id, $l->text, $L->text];
               }
             }
            else
             {confess "Text expected last before node $id:\n".$l->text;
             }
           }
          else
           {confess "Node expected last before node $id:\n".$l->text;
           }
         }
       }
     }
    else
     {confess "No matching id for $id";
     }
   });
  bless [@patches], q(Data::Edit::Xml::Patch);                                  # Return patch
 }

sub Data::Edit::Xml::Patch::install($$)                                         # Replay a patch created by L<createPatch|/createPatch> against a L<parse|/parse> tree that has the same tag and id structure with each id being unique.
 {my ($patches, $a) = @_;                                                       # Patch, parse tree
  my $i = $a->indexIds;
  for my $patch(@$patches)
   {my ($c, $id) = @$patch;
    my $o = $i->{$id} or confess "No node with id $id";
    if ($c eq q(changeValue))
     {my (undef, undef, $a, $v, $V) = @$patch;
      $o->attr($a) eq $v or confess "Expected $a=$v but got $a=".$o->attr($a);
      $o->setAttr($a, $V);
     }
    elsif ($c eq q(deleteAttr))
     {my (undef, undef, $a, $v) = @$patch;
      $o->attr($a) eq $v or confess "Expected $a=$v but got $a=".$o->attr($a);
      $o->deleteAttr($a);
     }
    elsif ($c eq q(createAttr))
     {my (undef, undef, $a, $v) = @$patch;
      $o->setAttr($a, $v);
     }
    else
     {my (undef, undef, $t, $T) = @$patch;
      my $update = sub
       {my ($o, $pos) = @_;
        $o                   or confess      "Node expected $pos node $id";
        my $s = $o->text;
        $s                   or confess "Text node expected $pos node $id";
        $s eq $t or $s eq $T or confess      "Text expected $pos node $id should be:\n$s";
        $o->text = $T unless $s eq $T;
       };
      if    ($c eq q(firstText)) {$update->($o->first, q(first after))}
      elsif ($c eq q(nextText))  {$update->($o->next,  q(after))}
      elsif ($c eq q(prevText))  {$update->($o->prev,  q(before))}
      elsif ($c eq q(lastText))  {$update->($o->last,  q(last before))}
      else                       {confess "Unknown command $c"}
     }
   }
 }

#D1 Propogating                                                                 # Propagate parent node attributes through a parse tree.

sub propagate($$@)                                                              #UC Propagate L<new attributes|/copyNewAttrs> from nodes that match the specified tag to all their child nodes, then L<unwrap|/unwrap> all the nodes that match the specified tag. Return the specified parse tree.
 {my ($tree, $tag, @context) = @_;                                              # Parse tree, tag of nodes whose attributes are to be propagated, optional context for parse tree
  return undef if @context and !$tree->at(@context);                            # Not in specified context
  $tree->by(sub                                                                 # Copy new attributes from each matching parent for each node in the parse tree
   {my ($node, @parents) = @_;                                                  # Node, parents
    for my $parent(@parents)                                                    # For each parent of the node being visited
     {$parent->copyNewAttrs($node) if -t $parent eq $tag;                       # Copy the new attributes of a parent with the specified tag
     }
   });
  $tree->by(sub                                                                 # Unwrap nodes with the specified tag unless they are the starting node
   {my ($node) = @_;                                                            # Node
    $node->unwrap if $node != $tree and -t $node eq $tag;                       # Unwrap the node if it matches the specified tag
   });
  $tree                                                                         # Return the parse tree
 }

#D1 Table of Contents                                                           # Analyze and generate tables of contents.

sub tocNumbers($@)                                                              #U Table of Contents number the nodes in a L<parse|/parse> tree.
 {my ($node, @match) = @_;                                                      # Node, optional list of tags to descend into else all tags will be descended into
  my $toc = {};
  my $match = @match ? {map{$_=>1} @match} : undef;                             # Tags to match or none
  my @context;

  my $tree; $tree = sub                                                         # Number the nodes below the current node
   {my ($node) = @_;
    my $n = 0;
    for($node->contents)                                                        # Each node below the current node
     {next if $match and !$match->{$_->tag};                                    # Skip non matching nodes
      push @context, ++$n;                                                      # New scope
      $toc->{"@context"} = $_;                                                  # Toc number for tag
      &$tree($_);                                                               # Number sub tree
      pop @context;                                                             # End scope
     }
   };

  &$tree($node);                                                                # Descend through the tree numbering matching nodes
  $toc                                                                          # Return {toc number} = <tag>
 }

#D1 Labels                                                                      # Label nodes so that they can be cross referenced and linked by L<Data::Edit::Xml::Lint>

sub addLabels($@)                                                               #U Add the named labels to the specified B<$node> and return the number of labels added. Labels that are not L<defined|https://perldoc.perl.org/functions/defined.html> will be ignored.
 {my ($node, @labels) = @_;                                                     # Node in parse tree, names of labels to add.
  my $l = $node->labels;                                                        # Labels on node
  my $n = keys %$l;                                                             # Count the number of labels at the start
  $l->{$_}++ for grep {trim($_)} @labels;                                       # Labels must be defined to be added. Remove leading and trailing white space to prevent Data::Edit::Xml::Lint complaining about it: justified by the assumption that it is a data entry error.
  keys(%$l) - $n                                                                # Number of labels added
 }

sub countLabels($)                                                              #U Return the count of the number of labels at a node.
 {my ($node) = @_;                                                              # Node in parse tree.
  scalar keys %{$node->labels}                                                  # Count of labels
 }

sub labelsInTree($)                                                             #U Return a hash of all the labels in a tree
 {my ($tree) = @_;                                                              # Parse tree.
  my %labels;                                                                   # Labels found
  $tree->by(sub                                                                 # Labels for each node in the parse tree
   {for($_->getLabels)                                                          # Each label
     {$labels{$_}++
     }
   });
  \%labels                                                                      # Hash of labels in the tree
 }

sub getLabels($)                                                                #U Return the names of all the labels set on a node.
 {my ($node) = @_;                                                              # Node in parse tree.
  sort keys %{$node->labels}
 }

sub deleteLabels($@)                                                            #U Delete the specified labels in the specified B<$node> or all labels if no labels have are specified and return that node.
 {my ($node, @labels) = @_;                                                     # Node in parse tree, names of the labels to be deleted
  my $n = keys %{$node->{labels}};                                              # Number of labels at start
  $node->{labels} = {} unless @labels;                                          # Delete all the labels if no labels supplied
  delete @{$node->{labels}}{@labels};                                           # Delete specified labels
  $n - keys %{$node->{labels}}                                                  # Number of labels deleted
 }

sub copyLabels($$)                                                              # Copy all the labels from the source node to the target node and return the source node.
 {my ($source, $target) = @_;                                                   # Source node, target node.
  $target->addLabels($source->getLabels);                                       # Copy all the labels from the source to the target
 }

sub moveLabels($$)                                                              # Move all the labels from the source node to the target node and return the source node.
 {my ($source, $target) = @_;                                                   # Source node, target node.
  $target->addLabels($source->getLabels);                                       # Copy all the labels from the source to the target
  $source->deleteLabels;                                                        # Delete all the labels from the source
 }

sub copyLabelsAndIdsInTree($$)                                                  # Copy all the labels and ids in the source parse tree to the matching nodes in the target parse tree. Nodes are matched via L<path|/path>. Return the number of labels and ids copied.
 {my ($source, $target) = @_;                                                   # Source node, target node.
  my $n = 0;                                                                    # Count modifications
  $target->by(sub                                                               # Scan the target
   {my ($o) = @_;
    my @p = $o->path;                                                           # Matching node in source
    if (my $q = $source->go(@p))
     {$n += $q->copyLabels($o) + $o->addLabels($q->id);                         # Copy labels and id from source to target
     }
   });
  $n                                                                            # Number of labels added
 }

sub giveEveryIdAGuid($$)                                                        #U Give a guid to every node in the specified B<$tree> that has an id attribute, saving any existing id attribute as a label, and return the count of the number of such replacements made.
 {my ($tree, $genGuid) = @_;                                                    # Tree, a sub that accepts a number and a node and returns a new Guid each time it is called
  my $n = 0;                                                                    # count the number of ids replaced with guids
  $tree->by(sub                                                                 # Scan the tree
   {my ($o) = @_;
    if (my $i = $o->id)                                                         # Ignore this node unless it has an id
     {if ($i !~ m(\AGUID)is)                                                    # Ignore this node if it already has a guid
       {$o->addLabels($i);                                                      # Make the current id into a label.
        $o->id = $genGuid->(++$n, $o);                                          # Replace the existing id with the next guid
       }
     }
   });
  $n                                                                            # Number of ids replaced with guids
 }

sub createGuidId($@)                                                            #CU Create an id for the specified B<$node> in the optional B<@context> from the md5Sum of its content moving any existing id to the labels associated with the B<$node> and return the existing B<$node>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $i = $node->id)                                                        # Save any current id if  any
   {$node->addLabels($i);                                                       # Save id as a label
   }
  $node->id = guidFromMd5 $node->stringAsMd5Sum;                                # Guid from MD5 of content
  $node                                                                         # Return existing node
 }
#a
#b <b><c/></b>
#c createGuidId b
#d Use the content of the current node to create a globally unique id for this node.

#D1 Operators                                                                   # Operator access to methods use the assign versions to avoid 'useless use of operator in void context' messages. Use the non assign versions to return the results of the underlying method call.  Thus '/' returns the wrapping node, whilst '/=' does not.  Assign operators always return their left hand side even though the corresponding method usually returns the modification on the right.

use overload
  '='        => sub{$_[0]},
  '**'       => \&opNew,
  '-X'       => \&opString,
  '@{}'      => \&opContents,
  '<='       => \&opAt,
  '>>'       => \&opPutFirst,
  '>>='      => \&opPutFirstAssign,
  '<<'       => \&opPutLast,
  '<<='      => \&opPutLastAssign,
  '>'        => \&opPutNext,
  '+='       => \&opPutNextAssign,
  '+'        => \&opPutNext,
  '<'        => \&opPutPrev,
  '-='       => \&opPutPrevAssign,
  '-'        => \&opPutPrev,
  'x='       => \&opBy,
  'x'        => \&opBy,
  '>='       => \&opGo,
  '*'        => \&opWrapContentWith,                                            # doc
  '*='       => \&opWrapContentWith,
  '/'        => \&opWrapWith,
  '/='       => \&opWrapWith,
  '%'        => \&opAttr,
  '--'       => \&opCut,                                                        # Better using -X
  '++'       => \&opUnwrap,                                                     # doc # Better using -u
  "fallback" => 1;

sub opString($$)                                                                # -A: L<printNode|/printNode>\m-B: L<bitsNodeTextBlank|/bitsNodeTextBlank>\m-b: L<isAllBlankText|/isAllBlankText>\m-C: L<stringContent|/stringContent>\m-c: L<context|/context>\m-d: L<depth|/depth>\m-e: L<prettyStringEnd|/prettyStringEnd>\m-f: L<first node|/first>\m-g: L<createGuidId|/createGuidId>\m-k: L<createGuidId|/createGuidId>\m-l: L<last node|/last>\m-M: L<stringAsMd5Sum|/stringAsMd5Sum>\m-O: L<contentAsTags2|/contentAsTags2>\m-o: L<contentAsTags|/contentAsTags>\m-p: L<prettyString|/prettyString>\m-R: L<requiredCleanup|/requiredCleanup>\m-r: L<stringTagsAndText|/stringTagsAndText>\m-S: L<printStack|/printStack>\m-s: L<string|/string>\m-T: L<isText|/isText>\m-t: L<tag|/tag>\m-u: L<unwrap|/unwrap>\m-W: L<id|/id>\m-w: L<stringQuoted|/stringQuoted>\m-X: L<prettyStringDitaHeaders|/prettyStringDitaHeaders>\m-x: L<cut|/cut>\m-z: L<prettyStringNumbered|/prettyStringNumbered>.
 {my ($node, $op) = @_;                                                         # Node, monadic operator.
  $op or confess;
  return $node->printNode                    if $op eq 'A';
  return $node->bitsNodeTextBlank            if $op eq 'B';  # Not much use
  return $node->isAllBlankText               if $op eq 'b';
  return $node->stringContent                if $op eq 'C';
  return $node->context                      if $op eq 'c';
  return $node->depth                        if $op eq 'd';  # Not much use
  return $node->prettyStringEnd              if $op eq 'e';  # Not much use
  return $node->first                        if $op eq 'f';  # Not much use
  return $node->createGuidId                 if $op eq 'g';
  return $node->createGuidId                 if $op eq 'k';  # Not in use
  return $node->last                         if $op eq 'l';  # Not much use
  return $node->stringAsMd5Sum               if $op eq 'M';
  return $node->contentAsTags2               if $op eq 'O';
  return $node->contentAsTags                if $op eq 'o';
  return $node->prettyString                 if $op eq 'p';
  return $node->requiredCleanUp              if $op eq 'R';  # Not much use
  return $node->stringTagsAndText            if $op eq 'r';  # Not much use
  return $node->printStack                   if $op eq 'S';
  return $node->string                       if $op eq 's';
  return $node->isText                       if $op eq 'T';
  return $node->tag                          if $op eq 't';
  return $node->unwrap                       if $op eq 'u';
  return $node->id                           if $op eq 'W';  # Not much use
  return $node->stringQuoted                 if $op eq 'w';  # Not much use
  return $node->prettyStringDitaHeaders      if $op eq 'X';
  return $node->cut                          if $op eq 'x';
  return $node->prettyStringNumbered         if $op eq 'z';  # Not much use
  confess "Unknown operator: $op";
 }

sub opContents($)                                                               # @{} : nodes immediately below a node.
 {my ($node) = @_;                                                              # Node.
  $node->content
 }

sub opAt($$)                                                                    # <= : Check that a node is in the context specified by the referenced array of words.
 {my ($node, $context) = @_;                                                    # Node, reference to array of words specifying the parents of the desired node.
  ref($context) =~ m/array/is or
    confess "Array of words required to specify the context";
  $node->at(@$context);
 }

sub opNew($$)                                                                   # ** : create a new node from the text on the right hand side: if the text contains a non word character \W the node will be create as text, else it will be created as a tag
 {my ($node, $text) = @_;                                                       # Node, name node of node to create or text of new text element
  return $text                if ref($text) eq __PACKAGE__;                     # The right hand side is already a node
  return $node->newTag($text) unless $text =~ m/\W/s;                           # Create a new node as tag
  $node->newText($text)                                                         # Create a new node as text if nothing lse worked
 }

sub opPutFirst($$)                                                              # >> : put a node or string first under a node and return the new node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place first under the node.
  $node->putFirst(my $new = opNew($node, $text));
  $new
 }

sub opPutFirstAssign($$)                                                        # >>= : put a node or string first under a node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place first under the node.
  opPutFirst($node, $text);
  $node
 }

sub opPutLast($$)                                                               # << : put a node or string last under a node and return the new node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place last under the node.
  $node->putLast(my $new = opNew($node, $text));
  $new
 }

sub opPutLastAssign($$)                                                         # <<= : put a node or string last under a node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place last under the node.
  opPutLast($node, $text);
  $node
 }

sub opPutNext($$)                                                               # > + : put a node or string after the specified B<$node> and return the new node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place after the first node.
  $node->putNext(my $new = opNew($node, $text));
  $new
 }

sub opPutNextAssign($$)                                                         # += : put a node or string after the specified B<$node>.
 {my ($node, $text) = @_;                                                       # Node, node or text to place after the first node.
  opPutNext($node, $text);
  $node
 }

sub opPutPrev($$)                                                               # < - : put a node or string before the specified B<$node> and return the new node.
 {my ($node, $text) = @_;                                                       # Node, node or text to place before the first node.
  $node->putPrev(my $new = opNew($node, $text));
  $new
 }

sub opPutPrevAssign($$)                                                         # -= : put a node or string before the specified B<$node>,
 {my ($node, $text) = @_;                                                       # Node, node or text to place before the first node.
  opPutPrev($node, $text);
  $node
 }

sub opBy($$)                                                                    # x= : Traverse a L<parse|/parse> tree in post-order.
 {my ($node, $code) = @_;                                                       # Parse tree, code to execute against each node.
  ref($code) =~ m/code/is or
    confess "sub reference required on right hand side";
  $node->by($code);
 }

sub opGo($$)                                                                    # >= : Search for a node via a specification provided as a reference to an array of words each number.  Each word represents a tag name, each number the index of the previous tag or zero by default.
 {my ($node, $go) = @_;                                                         # Node, reference to an array of search parameters.
  return $node->go(@$go) if ref($go);
  $node->go($go)
 }

sub opAttr($$)                                                                  # % : Get the value of an attribute of the specified B<$node>.
 {my ($node, $attr) = @_;                                                       # Node, reference to an array of words and numbers specifying the node to search for.
  return map {$node->attr($_)} @$attr if ref($attr);
  $node->attr($attr)
 }

sub opWrapWith($$)                                                              # / : Wrap node with a tag, returning the wrapping node.
 {my ($node, $tag) = @_;                                                        # Node, tag.
  return $node->wrapUp(@$tag) if ref($tag);
  $node->wrapWith($tag)
 }

sub opWrapContentWith($$)                                                       # * : Wrap content with a tag, returning the wrapping node.
 {my ($node, $tag) = @_;                                                        # Node, tag.
  return $node->wrapDown(@$tag) if ref($tag);
  $node->wrapContentWith($tag)
 }

sub opCut($)                                                                    # -- : Cut out a node.
 {my ($node) = @_;                                                              # Node.
  $node->cut
 }

sub opUnwrap($)                                                                 # ++ : Unwrap a node.
 {my ($node) = @_;                                                              # Node.
  $node->unwrap
 }

#D1 Reuse                                                                       # Identify common components that can be reused

sub subMd5Tree($)                                                               # Return the L<md5> of the L<stringContent> of the parse tree.
 {my ($node) = @_;                                                              # Node
  if (my $s = $node->stringContent)                                             # Md5 of something with content
   {return (stringMd5Sum($s), $s);
   }
  ()                                                                            # A text element must be wrapped in an non text element - to get the md5 sum of a text element call via the wrapping node - otherwise we get duplicates.
 }

sub subMd5($%)                                                                  # Return a hash {L<md5> of parse tree}{string representation of sub tree}++ to locate common parse trees under B<$node> that could be included via a conref.  L<Md5>s are only calculated for nodes whose tags match the keys of the B<$tags> hash that have truthful values. The attributes of the root node of each sub tree are ignored in the computation as they can be supplied by the conreffing element.
 {my ($node, %tags) = @_;                                                       # Node, hash of acceptable tags.

  my %hash;                                                                     # Hash of md5 sums to content

  my $M; $M = sub($)                                                            # Md5 sum of current parse tree
   {my ($p) = @_;                                                               # Node
    my $tag = -t $p;
    if ($tags{$tag})                                                            # Tag matches
     {if (my ($m, $s) = $p->subMd5Tree)                                         # Md5 of something with content
       {$hash{$tag}{$m}{$s}++;
       }
     }

    if (my @c = $p->contents)                                                   # Each sub tree
     {$M->($_) for @c;
     }
   };

  $M->($node);                                                                  # Start at current node
  return \%hash;                                                                # Resulting hash
 }

#D1 Statistics                                                                  # Statistics describing the L<parse|/parse> tree.

sub count($@)                                                                   #U Return the count of the number of instances of the specified tags under the specified B<$node>, either by tag in array context or in total in scalar context.
 {my ($node, @names) = @_;                                                      # Node, possible tags immediately under the node.
  if (wantarray)                                                                # In array context return the count for each tag specified
   {my @c;                                                                      # Count for the corresponding tag
    reindexNode($node);                                                         # Create index for this node
    my %i = %{$node->indexes};                                                  # Index of child nodes
    for(@names)
     {if (my $i = $i{$_}) {push @c, scalar(@$i)} else {push @c, 0};             # Save corresponding count
     }
    return @c;                                                                  # Return count for each tag specified
   }
  else                                                                          # In scalar context count the total number of instances of the named tags
   {if (@names)
     {my $c = 0;                                                                # Tag count
      reindexNode($node);                                                       # Create index for this node
      my %i = %{$node->indexes};                                                # Index of child nodes
      for(@names)
       {if (my $i = $i{$_}) {$c += scalar(@$i)}
       }
      return $c;
     }
    else                                                                        # In scalar context, with no tags specified, return the number of nodes under the specified B<$node>
     {my @c = $node->contents;
      return scalar(@c);                                                        # Count of all tags including CDATA
     }
   }
  confess "This should not happen"
 }

sub countTags($)                                                                #U Count the number of tags in a L<parse|/parse> tree.
 {my ($node) = @_;                                                              # Parse tree.
  my $n = 0;
  $node->by(sub{++$n});                                                         # Count tags including CDATA
  $n                                                                            # Number of tags encountered
 }

sub countTagNames2($$)                                                          #P Return a reference to a hash showing the number of instances of each tag on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, count of tags so far.
  $$count{$node->tag}++;                                                        # Add current tag
  $_->countTagNames2($count) for $node->contents;                               # Each contained node
 }

sub countTagNames($@)                                                           #U Return a reference to a hash showing the number of instances of each tag on and below the specified B<$node> excluding any tags named in B<@exclude>.
 {my ($node, @exclude) = @_;                                                    # Node, tags to exclude from the count
  my $count = {};                                                               # Counts
  $node->countTagNames2($count);                                                # Count includes this node
  delete $$count{$_} for @exclude;                                              # Delete excluded nodes
  $count                                                                        # Count
 }

sub countNonEmptyTags2($$)                                                      #P Count the instances of non empty tags on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, count of tags so far.
  $$count{$node->tag}++ unless $node->isAllBlankText;                           # Add current tag
  $_->countNonEmptyTags2($count) for $node->contents;                           # Each contained node
 }

sub countNonEmptyTags($@)                                                       #U Return a reference to a hash showing the number of instances of each non empty tag on and below the specified B<$node> excluding any tags named in B<@exclude>.
 {my ($node, @exclude) = @_;                                                    # Node, tags to exclude from the count
  my $count = {};                                                               # Counts
  $node->countNonEmptyTags2($count);                                            # Count includes this node
  delete $$count{$_} for @exclude;                                              # Delete excluded nodes
  $count                                                                        # Count
 }

sub countTexts2($$)                                                             #P Count the instances of non empty texts on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, texts so far
  if ($node->isText)                                                            # Check we are on a text node
   {$$count{$node->text}++;                                                     # Add current text
   }
  else                                                                          # Not on a text node
   {$_->countTexts2($count) for $node->contents;                                # Each contained node
   }
 }

sub countTexts($)                                                               #U Return a reference to a hash showing the incidence of texts on and below the specified B<$node>.
 {my ($node) = @_;                                                              # Node
  my $count = {};                                                               # Counts
  $node->countTexts2($count);                                                   # Count includes this node
  $count                                                                        # Count
 }

sub countWords($)                                                               #UP Count instances of words in texts
 {my ($node) = @_;                                                              # Node
  my $c = countTexts($node);                                                    # Gather text
  my %w;                                                                        # Words
  for my $t(keys %$c)                                                           # Each text
   {$w{$_}++ for split /[.,;]?\s+/, $t;                                         # Count words in each text
   }
  \%w                                                                           # Return words
 }

sub countAttrNames($;$)                                                         #U Return a reference to a hash showing the number of instances of each attribute on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, attribute count so far
  $count //= {};                                                                # Counts
  $$count{$_}++ for $node->getAttrs;                                            # Attributes from current tag
  $_->countAttrNames($count) for $node->contents;                               # Each contained node
  $count                                                                        # Count
 }

sub countAttrNamesOnTagExcluding($@)                                            #U Count the number of attributes owned by the specified B<$node> that are not in the specified list.
 {my ($node, @attr) = @_;                                                       # Node, attributes to ignore
  my %attr = map{$_=>1} @attr;                                                  # Set of attributes to ignore
  my $count;                                                                    # Count
  $count++ for grep {!$attr{$_}} $node->getAttrs;                               # Attributes from current tag
  $count                                                                        # Count
 }

sub countAttrValues($;$)                                                        #U Return a reference to a hash showing the number of instances of each attribute value on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, count of attributes so far.
  $count //= {};                                                                # Counts
  $count->{$node->attr($_)}++ for $node->getAttrs;                              # Attribute values from current tag
  $_->countAttrValues($count) for $node->contents;                              # Each contained node
  $count                                                                        # Count
 }

sub countAttrNamesAndValues($;$)                                                #U Return a reference to a hash showing the number of instances of each attribute name and value on and below the specified B<$node>.
 {my ($node, $count) = @_;                                                      # Node, count of attributes so far.
  $count //= {};                                                                # Counts
  $count->{$_}{$node->attr($_)}++ for $node->getAttrs;                          # Attribute names and values from current tag
  $_->countAttrNamesAndValues($count) for $node->contents;                      # Each contained node
  $count                                                                        # Count
 }

sub countOutputClasses($$)                                                      #U Count instances of outputclass attributes
 {my ($node, $count) = @_;                                                      # Node, count so far.
  $count //= {};                                                                # Counts
  my $a = $node->attr(qw(outputclass));                                         # Outputclass attribute
  $$count{$a}++ if $a ;                                                         # Add current output class
  &countOutputClasses($_, $count) for $node->contents;                          # Each contained node
  $count                                                                        # Count
 }

sub countReport($@)                                                             #U Count tags, attributes, words below the specified node
 {my ($node, @context) = @_;                                                    # Node to count from, optional context
  return undef if @context and !$node->at(@context);                            # Optionally check context
  my $c = $node->countTagNames;                                                 # Count tag types
  formatTable([sort {$$a[1] cmp $$b[1]} map {[$$c{$_}, $_]} sort keys %$c],
              [qw(Tag Count)]);
 }

BEGIN{*cr=*countReport}

sub changeReasonCommentSelectionSpecification :lvalue                           #S Provide a specification to select L<change reason comments|/crc> to be inserted as text into a L<parse|/parse> tree. A specification can be either:\m=over\m=item the name of a code to be accepted,\m=item a regular expression which matches the codes to be accepted,\m=item a hash whose keys are defined for the codes to be accepted or\m=item B<undef> (the default) to specify that no such comments should be accepted.\m=back
 {CORE::state $r;
  $r
 }

sub crc($$;$)                                                                   # Insert a comment consisting of a code and an optional reason as text into the L<parse|/parse> tree to indicate the location of changes to the L<parse|/parse> tree.  As such comments tend to become very numerous, only comments whose codes matches the specification provided in L<changeReasonCommentSelectionSpecification|/changeReasonCommentSelectionSpecification> are accepted for insertion. Subsequently these comments can be easily located using:\mB<grep -nr "<!-->I<code>B<">\mon the file containing a printed version of the L<parse|/parse> tree. Please note that these comments will be removed if the output file is reparsed.\mReturns the specified B<$node>.
 {my ($node, $code, $reason) = @_;                                              # Node being changed, reason code, optional text description of change
  if (sub                                                                       # Whether to make a change entry in the L<parse|/parse> tree
   {my $s = changeReasonCommentSelectionSpecification;                          # Change selection specification
    return undef unless $s;                                                     # Do not record change reasons unless a change selection has been supplied
    my $r = ref $s;                                                             # Change selection has been supplied
    return 1 if $r and $r =~ m(Regexp) and $code =~ m($s);                      # Requested change matches the supplied regular expression
    return 1 if $r and $r =~ m(HASH)   and $s->{$code};                         # Requested change is a key in the supplied hash
    return 1 if $s and $s eq $code;                                             # Requested change equal to the supplied name
    undef                                                                       # No match so do not crate a change entry in the L<parse|/parse> tree
   }->())
   {my $message = $reason ? "<!--$code - $reason -->" : "<!--$code-->";         # Insert message either eliding it with existing text or creating a new text node
    if ($node->isText)                                                          # If we are on a text node we can simply add the comment at the front
     {$node->text = $message.$node->text;
     }
    else
     {my $P = $node->prev;                                                      # At the end of a previous text node?
      if ($P and $P->isText)
       {$P->text = $node->text.$message;
       }
      else
       {my $N = $node->next;                                                    # At the start of a following text node?
        if ($N and $N->isText)
         {$N->text = $message.$node->text;
         }
        elsif ($node->parent)                                                   # Not a text node, no text node on either side, not the root
         {$node->putPrevAsText($message);
         }
        elsif (my $f = $node->first)                                            # Root node but not text
         {if ($f->isText)                                                       # At front of first text node
           {$f->text = $message.$node->text;
           }
          else                                                                  # No first node or first node is not text, place first
           {$node->putFirstAsText($message);
           }
         }
       }
     }
   }
  $node
 }

sub howFirst($)                                                                 #U Return the depth to which the specified B<$node> is L<first|/isFirst> else B<0>.
 {my ($node) = @_;                                                              # Node
  my $i = 0;                                                                    # Count first depth
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {last unless $p->isFirst;                                                    # Go up while first
    ++$i;
   }
  $i
 }

sub howLast($)                                                                  #U Return the depth to which the specified B<$node> is L<last|/isLast> else B<0>.
 {my ($node) = @_;                                                              # Node
  my $i = 0;                                                                    # Count last depth
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {last unless $p->isLast;                                                     # Go up while last
    ++$i;
   }
  $i
 }

sub howOnlyChild($)                                                             #U Return the depth to which the specified B<$node> is an L<only child|/isOnlyChild> else B<0>.
 {my ($node) = @_;                                                              # Node
  my $i = 0;                                                                    # Count only child depth
  for(my $p = $node; $p; $p = $p->parent)                                       # Go up
   {last unless $p->isOnlyChild;                                                # Go up while only child
    ++$i;
   }
  $i
 }

sub howFar($$)                                                                  #U Return how far the first node is from the second node along a path through their common ancestor.
 {my ($first, $second) = @_;                                                    # First node, second node
  my $p = $first->commonAncestor($second);                                      # Find their common ancestor
  return 0 if $first == $second;                                                # Same node
  return 1 if $first->adjacent($second);                                        # Adjacent nodes
  $p->howFarAbove($first) + $p->howFarAbove($second) -                          # Sum of the paths to their common ancestor plus any adjustment
    ($first->commonAdjacentAncestors($second) ? 1 : 0);                         # If the nodes share a pair of common ancestral siblings the path is one step shorter
 }

sub howFarAbove($$)                                                             #U Return how far the first node is  L<above|/above> the second node is or B<0> if the first node is not strictly L<above|/above> the second node.
 {my ($above, $below) = @_;                                                     # First node above, second node below
  for(my ($i, $p) = (1, $below->parent); $p; $p = $p->parent, ++$i)             # Go up from below to above
   {return $i if $above == $p;                                                  # Return the height if we have reached the node above
   }
  0
 }

sub howFarBelow($$)                                                             #U Return how far the first node is  L<below|/below> the second node is or B<0> if the first node is not strictly L<below|/below> the second node.
 {my ($below, $above) = @_;                                                     # First node below, second node above
  $above->howFarAbove($below)                                                   # Use howFarABove with arguments inverted
 }

#D1 Required clean up                                                           # Insert required clean up tags.

sub createRequiredCleanUp($@)                                                   #P Create a required clean up $node with the specified B<@text>
 {my ($node, @text) = @_;                                                       # Node, clean up messages
  my $text = join ' ', @text;                                                   # Combined text
  my $r = $node->newTag(q(required-cleanup));                                   # Create required clean up node
  $r->putFirstAsText($text);                                                    # Add text
  $r                                                                            # Return required clean up node
 }

sub requiredCleanUp($;$)                                                        #U Replace a B<$node> with a required cleanup node with special characters replaced by symbols and with the optional B<$outputclass>.\mReturns the specified B<$node>.
 {my ($node, $outputclass) = @_;                                                # Node, optional outputclass attribute of required cleanup tag
  my $text = replaceSpecialChars($node->prettyString);                          # Replace xml chars with symbols
  my $r = $node->createRequiredCleanUp($text);                                  # Create required clean up node
     $r->outputclass = $outputclass if $outputclass;                            # Add outputclass if supplied
  $node->replaceWith($r);                                                       # Replace current node
  $r                                                                            # Return required clean up node
 }

sub replaceWithRequiredCleanUp($@)                                              #U Replace a B<$node> with required cleanup B<@text> and return the new node
 {my ($node, @text) = @_;                                                       # Node to be replace, clean up message
  my $text = join ' ', @text;                                                   # Combined text
  my $r = $node->createRequiredCleanUp($text);                                  # Create required clean up node
  $node->replaceWith($r);
  $r
 }

sub putFirstRequiredCleanUp($@)                                                 #U Place a required cleanup first under a specified B<$node> using the specified B<@text>  and return the required clean up node.
 {my ($node, @text) = @_;                                                       # Node, clean up message
  my $text = join ' ', @text;                                                   # Combined text
  my $r = $node->createRequiredCleanUp($text);                                  # Create required clean up node
  $node->putFirst($r);                                                          # Insert required clean up node
  $r                                                                            # Return required clean up node
 }

sub putLastRequiredCleanUp($@)                                                  #U Place a required cleanup last under a specified B<$node> using the specified B<@text>  and return the required clean up node.
 {my ($node, @text) = @_;                                                       # Node, clean up message
  my $text = join ' ', @text;                                                   # Combined text
  my $r = $node->createRequiredCleanUp($text);                                  # Create required clean up node
  $node->putLast($r);                                                           # Insert required clean up node
  $r                                                                            # Return required clean up node
 }

sub putNextRequiredCleanUp($@)                                                  #IU Place a required cleanup next after a specified B<$node> using the specified B<@text>  and return the required clean up node.
 {my ($node, @text) = @_;                                                       # Node, clean up message
  my $text = join ' ', @text;                                                   # Combined text
  if ($node->parent)                                                            # Place after non root node
   {my $r = $node->createRequiredCleanUp($text);                                # Create required clean up node
    $node->putNext($r);                                                         # Insert required clean up node
    return $r;                                                                  # Return required clean up node
   }
  else                                                                          # Place last under a root node
   {return $node->putLastRequiredCleanUp($text);
   }
 }
#a putFirstRequiredCleanUp
#b <b><c/></b>
#c at c b
#c putNextRequiredCleanUp remove tag c
#d Create a required cleanup tag with the specified text and put it next.

sub putPrevRequiredCleanUp($$)                                                  #U Place a required cleanup before a specified B<$node> using the specified B<@text>  and return the required clean up node.
 {my ($node, $text) = @_;                                                       # Node, clean up message
  if ($node->parent)                                                            # Place before non root node
   {my $r = $node->createRequiredCleanUp($text);                                # Create required clean up node
    $node->putPrev($r);                                                         # Insert required clean up node
    return $r;                                                                  # Return required clean up node
   }
  else                                                                          # Place first under a root node
   {return $node->putFirstRequiredCleanUp($text);
   }
 }

#D1 Conversions                                                                 # Methods useful for conversions from word, L<html> and L<Dita> to L<Dita>.

#D2 Dita Conversions                                                            # Methods useful for enhancing L<Dita>.

sub ditaGetConRef($$;$$)                                                        #S Get the value of a B<conref> attribute given a valid L<Dita> B<$ref> reference found in the file named B<$sourceFile>. Optionally, if the regular expression B<$matchContext> is supplied then nodes whose context (single blank separated) tags match this regular expression will be considered first as a target of the conref, if such a search fails then any unique node with the matching id will be the considered next as a possible target unless the optional B<$matchContextOnly> parameter is true. Note: all the conrefs in the included content will also be expanded creating the possibility of endless circular expansions.
 {my ($ref, $sourceFile, $matchContext, $matchContextOnly) = @_;                # Dita ref value of conref, source file containing conref, optional regular expression of acceptable context, confine search to acceptable context.
  my ($rf, $rt, $ri) = parseDitaRef($ref, $sourceFile);                         # Parse the reference
# -e $rf or confess "No such file:\n$rf\n";                                     # Confess if the target file does not exist
  return undef unless $rf and -e $rf;                                           # Insist that the file be present

  my $x = eval {new($rf)};                                                      # Parse target file
  return undef unless $x;                                                       # Any errors will show up in the Lint reports

  if ($ri)                                                                      # Id present - we want part of the target
   {if ($matchContext)                                                          # Matching requested
     {my $tree;                                                                 # Parse tree representing conreffed content
      $x->by(sub                                                                # Scan parse tree for id
       {my ($o) = @_;                                                           # Each node
        if ($o->idX eq $ri and $o->context =~ m($matchContext))                 # Node with matching id and matching tag if tag matching requested
         {my $r = $o->ditaExpandAllConRefs($rf);                                # Expand any conrefs referenced by the conref
          $tree = $r && $o != $r ? $r->cut : $o->cut;                           # Parse tree representing conref
         }
       });

      return $tree if $tree;                                                    # Take last matching id if any
     }

    return undef if $matchContextOnly;                                          # Broader search not allowed

    my $tree;                                                                   # Broaden the search if the target has not been found so far and a broader search is permissible.
    $x->by(sub                                                                  # Scan parse tree for id
     {my ($o) = @_;                                                             # Each node
      if ($o->idX eq $ri)                                                       # Node with matching id and matching tag if tag matching requested
       {my $r = $o->ditaExpandAllConRefs($rf);                                  # Expand any conrefs referenced by the conref
        $tree = $r && $o != $r ? $r->cut : $o->cut;                             # Parse tree representing conref
       }
     });

    return $tree;                                                               # Conreffed content or fail
   }
  else                                                                          # No id specified so include the entire topic
   {$x->by(sub                                                                  # Scan the new content for conrefs to expand - let us hope that there is an end to their conref chain else we will hit the call depth limit
     {my ($o) = @_;                                                             # Each node
      $o->ditaExpandAllConRefs($rf);                                            # Expand any conrefs in the content referenced by the conref
     });

    $x
   }                                                                            # Id missing, we want the entire target
# cluck "No such id in file: $ref\n$rf\n" unless @c;                            # No such id
# cluck "Too many possibilities for in file: $ref\n$rf\n".dump(\@c)."\n" if @c; # Too many possibilities
 }

sub ditaReplaceConref($$)                                                       # Replace a conref on the specified B<$node> in the specified source file B<$sourceFile> with the parse tree of the referenced content. Returns the referenced content.  Confesses if there is no conref to replace on $node.
 {my ($node, $sourceFile) = @_;                                                 # Node with conref attribute, source file containing the node.
  my $c = $node->attr(q(conref));                                               # Conref attribute
  $c or confess "No conref attribute: ".-A $node;                               # Check there is a conref
  if (my $v = ditaGetConRef($c, $sourceFile))                                   # Parse tree
   {if (my $id = $node->id)                                                     # The node has an id which should be retained
     {my $r = $node->replaceWith($v);
      $r->id = $id;
      return $r;
     }
    else                                                                        # The node does not have an id
     {return $node->replaceWith($v);
     }
   }
  undef
 }

sub ditaReplaceAnyConref($@)                                                    #U Replace a conref with the referenced content if we are in the optional context and the target of the conref can be located. Returns true regardless of any processing performed to allow L<PCD> processing to continue.
 {my ($node, @context) = @_;                                                    # Node with conref attribute, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node unless my $c = $node->attr(q(conref));                           # Continue if there is no conref attribute
  if (my $inputFile = $node->root->inputFile)                                   # Input file
   {if (my $v = ditaGetConRef($c, $inputFile))                                  # Locate conref
     {if (my $id = $node->id)                                                   # The node has an id which should be retained
       {my $r = $node->replaceWith($v);                                         # Replace current node with referenced node and return new node
        $r->id = $id;
        return $r;
       }
      else                                                                      # No id to retain on node
       {return $node->replaceWith($v);                                          # Replace current node with referenced node and return new node
       }
     }
   }
  undef                                                                         # Unable to replace conref
 }

sub ditaReplaceAnyConrefIdeallyWithMatchingTag($@)                              #U Replace a conref with the referenced content if we are in the optional context and the target of the conref can be located. If multiple possible targets exists, priority will be given to any that match the tag of $node. Returns the root of the parse tree containing the new content or B<undef> if no new content was located.
 {my ($node, @context) = @_;                                                    # Node with conref attribute, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node unless my $c = $node->attr(q(conref));                           # Continue if there is no conref attribute
  if (my $inputFile = $node->root->inputFile)                                   # Input file
   {my $tag = quotemeta $node->tag;                                             # Tag of conreffing node
    if (my $v = ditaGetConRef($c, $inputFile, qr(\A$tag(\s|\Z))))               # Locate conref with, ideally, the target node having the same tag as the source node
     {if (my $id = $node->id)                                                   # The node has an id which should be retained
       {my $r = $node->replaceWith($v);                                         # Replace current node with referenced node and return new node
        $r->id = $id;
        return $r;
       }
      else                                                                      # No id to retain on node
       {return $node->replaceWith($v);                                          # Replace current node with referenced node and return new node
       }
     }
   }
  undef                                                                         # Unable to replace conref
 }

sub ditaReplaceAnyConrefInContext($$@)                                          #U Replace a conref with the referenced content if we are in the optional context and the target of the conref can be located and the L<context> of the target node matches the regular expression B<$context>. Returns the root of the parse tree containing the new content or B<undef> if no new content was located.
 {my ($node, $context, @context) = @_;                                          # Node with conref attribute, tags to match, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return $node unless my $c = $node->attr(q(conref));                           # Continue if there is no conref attribute
  if (my $inputFile = $node->root->inputFile)                                   # Input file
   {if (my $v = ditaGetConRef($c, $inputFile, $context, 1))                     # Locate conref requiring that the target node match the specified regular expression
     {if (my $id = $node->id)                                                   # The node has an id which should be retained
       {my $r = $node->replaceWith($v);                                         # Replace current node with referenced node and return new node
        $r->id = $id;
        return $r;
       }
      else                                                                      # No id to retain on node
       {return $node->replaceWith($v);                                          # Replace current node with referenced node and return new node
       }
     }
   }
  undef                                                                         # Unable to replace conref
 }

sub ditaExpandAllConRefs($$)                                                    #U Expand all the conrefs in the specified B<$parseTree> relative to the specified B<$sourceFile>.
 {my ($tree, $sourceFile) = @_;                                                 # Parse tree, Source file we are expanding in

  $tree->by(sub                                                                 # Search parse tree for conrefs to expand
   {my ($o) = @_;
    if ($o->attr_conref)                                                        # Located a conref
     {my $id = $o->id;
      my $r  = $o->ditaReplaceConref($sourceFile);                              # Expand conref
      $r->id = $id if $r && defined $id;
      if ($o == $tree)                                                          # Expanded
       {$tree = $r;
       }
     }
   });

  $tree
 }

sub ditaAbsoluteHref($$)                                                        # Return the absolute value of the href of a specified B<$node> relative to the specified B<$sourceFile> or B<undef> if the node has no B<href> attribute.
 {my ($node, $sourceFile) = @_;                                                 # Node containing href, source file
  if (my $h = $node->href)                                                      # Node has href
   {return absFromAbsPlusRel($sourceFile, $h);
   }
  undef                                                                         # No href
 }

sub ditaListToChoices($@)                                                       #UC Change the specified B<$list> to B<choices>.
 {my ($list, @context) = @_;                                                    # Node, optional context
  $list->tag =~ m(\A(ol|sl|ul)\Z)s or confess "Not a list";                     # Not a list
  return undef if @context and !$list->at(@context);                            # Not in specified context
  for my $li($list->c_li)                                                       # All li
   {$li->change(q(choice));                                                     # Change li to choice
   }
  $list->change(q(choices)) ;                                                   # Change tag of node to choices
 }

sub ditaListToSteps($@)                                                         #UC Change the specified B<$node> to B<steps> and its contents to B<cmd\step> optionally only in the specified context.
 {my ($list, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$list->at(@context);                            # Not in specified context
  for(@$list)                                                                   # Each li
   {$_->change(qw(  cmd))->wrapWith(q(step));                                   # li -> cmd\step
    $_->unwrap(qw(p cmd)) for @$_;                                              # Unwrap any contained p
   }
  $list->change(q(steps));
 }

sub ditaListToStepsUnordered($@)                                                #UC Change the specified B<$node> to B<steps-unordered> and its contents to B<cmd\step> optionally only in the specified context.
 {my ($list, @context) = @_;                                                    # Node, optional context
  my $steps = $list->ditaListToSteps(@context);                                 # Change to steps
  $steps->change(q(steps-unordered)) if $steps;                                 # Change to steps unordered
  $steps
 }

sub ditaListToSubSteps($@)                                                      #UC Change the specified B<$node> to B<substeps> and its contents to B<cmd\step> optionally only in the specified context.
 {my ($list, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$list->at(@context);                            # Not in specified context
  for(@$list)                                                                   # Each li
   {$_->change(qw(  cmd))->wrapWith(q(substep));                                # li -> cmd\step
    $_->unwrap(qw(p cmd)) for @$_;                                              # Unwrap any contained p
   }
  $list->change(q(substeps));
 }

sub ditaStepsToList($$@)                                                        #UC Change the specified B<$node> to a node with name B<$tag> or to B<ol> if B<$tag> is not supplied and its B<cmd\step> content to B<li> to create a list optionally only in the specified context.
 {my ($steps, $tag, @context) = @_;                                             # Node, new tag if not ol, optional context
  return undef if @context and !$steps->at(@context);                           # Not in specified context

  for(@$steps)                                                                  # Content
   {$_->change(q(li));                                                          # Change content to li
    $_->unwrap for $_->c_cmd;                                                   # Unwrap cmd
   }
  $steps->change($tag//q(ol));                                                  # Change tag of node to ol unless an alternative has been supplied
 }

sub ditaStepsToChoices($@)                                                      #UC Change the specified B<$node> to B<choices>.
 {my ($steps, @context) = @_;                                                   # Node, optional context
  return undef if @context and !$steps->at(@context);                           # Not in specified context
  for my $step(@$steps)                                                         # Steps
   {$step->change(q(choice));                                                   # Change step to choice
    for my $o(@$step)
     {my $t = $o->tag;
      if ($t =~ m(\Acmd\Z)s)
       {$o->change(q(p));
       }
      elsif ($t =~ m(\A(stepresult|stepxmp)\Z)s)
       {$o->unwrap;
       }
     }
   }
  $steps->change(q(choices)) ;                                                  # Change tag of node to choices
 }

sub ditaConvertSubStepsToSteps($@)                                              #UC Change the B<$substeps> to B<steps> and return the B<steps> on success or B<undef> on failure.
 {my ($substeps, @context) = @_;                                                # Substeps, optional context
  $substeps->at_substeps or confess "Not on substeps";                          # Not on substeps
  return undef if @context and !$substeps->at(@context);                        # Not in specified context
  for my $s(@$substeps)                                                         # Substeps
   {$s->change_step;                                                            # Change substep to step
   }
  $substeps->change_steps;                                                      # Change to steps
 }

sub ditaListToTable($@)                                                         #UC Convert a list to a table in situ - as designed by MiM.
 {my ($node, @context) = @_;                                                    # List node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context

  my $cols = 0;                                                                 # Number of columns
  for my $l($node->c_li)
   {my $n = $l->c_p;
    $cols = max($cols, $n);
   }

  my $table       = $node->change_table;                                        # Table structure
  my $tgroup      = $table->wrapContentWith_tgroup->set(cols=>$cols);
  my $tbody       = $tgroup->wrapContentWith_tbody;
  my $headEntries = q(<entry/>) x $cols;
     $tgroup->putFirst(Data::Edit::Xml::new(<<END));
<thead>
  <row>
  $headEntries
  </row>
</thead>
END

  for my $i(reverse 1..$cols)                                                   # Add colspec
   {$tgroup->putFirst($node->newTag
     (qw(colspec colwidth 1*), colname=>q(c).$i, colnum=>$i));
   }

  for my $l($tbody->c_li)                                                       # Convert li to row and the p to entry
   {$l->change_row;
    for my $p($l->c_p)
     {$p->change_entry
     }
   }

  $node                                                                         # Table thus constructed
 }

sub ditaMergeLists($@)                                                          #UC Merge the specified B<$node> with the preceding or following list or steps or substeps if possible and return the specified B<$node> regardless.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  $node->ditaMergeListsOnce for 1..2                                            # Do the merge twice to pick up all the stragglers
 }

sub ditaMergeListsOnce($)                                                       #UP Merge the specified B<$node> with the preceding or following list or steps or substeps if possible and return the specified B<$node> regardless.
 {my ($node) = @_;                                                              # Node
  if ($node->at(qr(\A(ol|ul|steps|substeps)\Z)))                                # Lists
   {if (my $p = $node->prev(-t $node))                                          # Merge two lists or steps or substeps
     {$p->putLast($node->cut);
      $node->unwrap;
     }
    elsif ($node->at(qr(\A(ol|sl|ul))))                                         # List with preceding or following list elements
     {if    (my $p = $node->prev(q(li)))
       {$node->putFirst($p->cut);
       }
      elsif (my $n = $node->next(q(li)))
       {$node->putLast($n->cut);
       }
     }
    elsif ($node->at(q(steps)))                                                 # Steps with preceding or following step
     {if    (my $p = $node->prev(qr(step|stepsection)))
       {$node->putFirst($p->cut);
       }
      elsif (my $n = $node->next(q(step)))
       {$node->putLast($n->cut);
       }
     }
    elsif ($node->at(q(substeps)))                                              # Substeps with preceding or following steps
     {if    (my $p = $node->prev(q(substep)))
       {$node->putFirst($p->cut);
       }
      elsif (my $n = $node->next(q(substep)))
       {$node->putLast($n->cut);
       }
     }
    elsif ($node->at(q(step))    and !$node->up(q(steps)))                      # Step not in steps
     {$node->wrapWith(q(steps));
     }
    elsif ($node->at(q(substep)) and !$node->up(q(substeps)))                   # Substep not in substeps
     {$node->wrapWith(q(steps));
     }
    elsif ($node->at(q(li))      and !$node->up(qr(\A(ol|sl|ul)\Z)))            # Li not in list
     {$node->wrapWith(q(ol));
     }
    elsif ($node->at(q(cmd))     and  $node->up(q(steps)))                      # Cmd under steps
     {$node->wrapWith(q(step));
     }
    elsif ($node->at(q(cmd))     and  $node->up(q(substeps)))                   # Cmd under substeps
     {$node->wrapWith(q(substep));
     }
    elsif ($node->not(q(li))     and  $node->up(qr(\A(ol|sl|ul)\Z)))            # Something under a list which is not an li
     {$node->wrapWith(q(li));
     }
    elsif ($node->not(qw(step stepsection)) and  $node->up(q(steps)))           # Something under steps which is not a step or stepsection
     {if (my $p = $node->prev(q(stepsection)))
       {$p->putLast($node->cut);
       }
     else
       {$node->wrapWith(q(stepsection));
       }
     }
   }
  elsif ($node->at(q(li)))                                                      # Free floating list element
   {if (my $p = $node->parent)
     {if ($p->not(qw(ol sl ul)))
       {sub
         {if (my $prev = $node->prev)
           {if ($prev->at(qr(\A(ol|sl|ul)\Z)))
             {$prev->putLastCut($node);
              return;
             }
           }

          if (my $next = $node->next)
           {if ($next->at(qr(\A(ol|sl|ul)\Z)))
             {$next->putFirstCut($node);
              return;
             }
           }
          $node->wrapWith(q(ol));
         }->();
       }
     }
   }

  $node                                                                         # Return the specified B<$node>
 }

sub mergeLikeElements($@)                                                       #UC Merge two of the same elements into one, retaining the order of any children. Return the original B<$node> if the request succeeds, else return B<undef>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  if (my $a = $node->mergeLikePrev(@context))
   {return $a;
   }
  if (my $b = $node->mergeLikeNext(@context))
   {return $b;
   }
  undef;
 }

sub mergeLikeNext($@)                                                           #UC Merge a B<$node> in an optional context with the next node if the two have the same tag by placing the next node first in the current B<$node> and unwrapping the next node. Return B<undef> if the request fails else the current B<$node>. Identical to L<mergeLikeElements|/mergeLikeElements>
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $next = $node->next(@context ? ($context[0]) : (-t $node)))            # Check following node has the same context as far as it is known
   {my $t = $node->lastText && $next->firstText;                                # Check whether we will be merging two text items
    $node->putLastCut($next);
    $next->putPrevAsText(q( )) if $t;                                           # Separate the two nodes with some white space if they contain just text
    $next->unwrap;
    return $node;
   }
  undef
 }

BEGIN{*mln=*mergeLikeNext}

sub mergeLikePrev($@)                                                           #UC Merge a B<$node> in an optional context with the previous node if the two have the same tag by placing the previous node first in the current B<$node> and unwrapping the previous node. Return B<undef> if the request fails else return the specified B<$node>.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  if (my $prev = $node->prev(@context ? ($context[0]) : (-t $node)))            # Check following node has the same context as far as it is known
   {my $t = $node->firstText && $prev->lastText;                                # Check whether we will be merging two text items
    $node->putFirstCut($prev);
    $prev->putNextAsText(q( )) if $t;                                           # Separate the two nodes with some white space if they contain just text
    $prev->unwrap;
    return $node;
   }
  undef
 }

#b <b id="1">bbbb</b><b id="2">BBBB</b>
#c mergeLikePrev b
#d Merge the current node into the previous node if it has the same tag name.

BEGIN{*mlp=*mergeLikePrev}

sub mergeOnlyChildLikeNext($@)                                                  #UC Merge a B<$node> if it is the only child of its parent with a preceding node of the same name that is also the only child of its parent and return the specified B<$node> or B<undef> if the request fails.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isOnlyChild;                                       # Node is an only child
  return undef unless my $parent = $node->parent;                               # Parent node
  return undef unless my $Parent = $parent->next(-t $parent);                   # Prior parent
  return undef unless $Parent->hasSingleChild(-t $node);                        # Prior single child
  mergeLikeNext($_) for $parent, $node;                                         # Merge
  $node                                                                         # Original node with content merged in
 }

BEGIN{*mocln=*mergeOnlyChildLikeNext}

sub mergeOnlyChildLikePrev($@)                                                  #UC Merge a B<$node> if it is the only child of its parent with a preceding node of the same name that is also the only child of its parent and return the specified B<$node> or B<undef> if the request fails.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isOnlyChild;                                       # Node is an only child
  return undef unless my $parent = $node->parent;                               # Parent node
  return undef unless my $Parent = $parent->prev(-t $parent);                   # Prior parent
  return undef unless $Parent->hasSingleChild(-t $node);                        # Prior single child
  mergeLikePrev($_) for $parent, $node;                                         # Merge
  $node                                                                         # Original node with content merged in
 }

BEGIN{*moclp=*mergeOnlyChildLikePrev}

sub mergeOnlyChildLikePrevLast($@)                                              #UC Merge a B<$node> if it is the only child of its parent with a preceding node with the same tag that is the last child of its parent and return the previous B<$node> or B<undef> if the request fails.
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context
  return undef unless $node->isOnlyChild;                                       # Node is an only child
  return undef unless my $parent = $node->parent;                               # Parent node
  return undef unless my $Parent = $parent->prev(-t $parent);                   # Prior parent
  return undef unless my $cousin = $Parent->last(-t $node);                     # Prior matching last child
  mergeLikeNext($_) for $Parent, $cousin;                                       # Merge
  $cousin                                                                       # Original node with content merged in
 }

sub ditaMaximumNumberOfEntriesInATGroupRow($)                                   #U Return the maximum number of entries in the rows of the specified B<$table> or B<undef> if not a table.
 {my ($tgroup) = @_;                                                            # TGroup node
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Confirm we are on a tgroup
  my $N = 0;                                                                    # Maximum number of entries in a row
  $tgroup->by(sub                                                               # Traverse tgroup
   {if (my ($r, $hb, $g) = @_)
     {if ($r->at_row_thead_tgroup or $r->at_row_tbody_tgroup)                   # Check this row is in the current table
       {if ($g == $tgroup)
         {$N = max($N, $r->ditaNumberOfColumnsInRow);                           # Maximum number of entries in a row so far
         }
       }
     }
   });

  $N
 }

sub ditaNumberOfColumnsInRow($)                                                 #UP Return estimate of the number of columns in a row
 {my ($row) = @_;                                                               # Row
  my @e = $row->c_entry;
  my $spans = 0;                                                                # Number of additional spanned columns
  for my $e(@e)                                                                 # Number of pad entries in this row
   {if (my $a = $e->attr_namest)                                                # Span start
     {if (my $b = $e->attr_nameend)                                             # Span end
       {s(\A\D+) ()gs for $a, $b;                                               # Remove non digits from front of column name on the assumptions (as is so often the case) that the column names are numbered sequentially.
        $spans += ($b||0) - ($a||0);                                            # Additional columns
       }
     }
   }

  $spans + @e
 }

sub ditaTGroupStatistics($)                                                     #U Return statistics about the rows in a given table
 {my ($tgroup) = @_;                                                            # Table group node
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Confirm we are on a tgroup

  my $maxHeadMinusPadding;                                                      # Maximum number of entries in a head row after an padding entries have been removed
  my $maxHead;                                                                  # Maximum number of entries in a head row regardless of padding rows
  my $minHead;                                                                  # Minimum number of entries in a head row regardless of padding rows
  my $maxBodyMinusPadding;                                                      # Maximum number of entries in a body row after an padding entries have been removed
  my $maxBody;                                                                  # Maximum number of entries in a body row regardless of padding rows
  my $minBody;                                                                  # Minimum number of entries in a body row regardless of padding rows
  my $colSpec = 0;                                                              # Colspec
  my $rows = 0;                                                                 # Number of rows in table = head and body combined rows

  $tgroup->by(sub                                                               # Traverse table group
   {if (my ($r, $h, $g) = @_)
     {if ($r->at_row_thead_tgroup and $g == $tgroup)                            # Heading row in current table group
       {if (my @n = $r->c_entry)                                                # Entries in this heading row
         {my $n = $r->ditaNumberOfColumnsInRow;                                 # Number of entries in a heading row so far
          $maxHead = max($maxHead//0,  $n);                                     # Maximum number of entries in a heading row so far
          $minHead = min($minHead//$n, $n);                                     # Maximum number of entries in a heading row so far
          while(@n)                                                             # Remove padding entries
           {if ($n[-1]->isAllBlankText) {pop @n; next}
            last;
           }
          $maxHeadMinusPadding = max($maxHeadMinusPadding//0, $n);              # Maximum number of non padding entries in a heading row so far
         }
        ++$rows;
       }
      elsif ($r->at_row_tbody_tgroup and $g == $tgroup)                         # Body row in current table group
       {if (my @n = $r->c_entry)                                                # Entries in this body row
         {my $n = $r->ditaNumberOfColumnsInRow;                                 # Number of entries in a body row so far
          $maxBody = max($maxBody//0,  $n);                                     # Maximum number of entries in a body row so far
          $minBody = min($minBody//$n, $n);                                     # Maximum number of entries in a body row so far
          while(@n)                                                             # Remove padding entries
           {if ($n[-1]->isAllBlankText) {pop @n; next}
            last;
           }
          $maxBodyMinusPadding = max($maxBodyMinusPadding//0, $n);              # Maximum number of non padding entries in a body row so far
         }
        ++$rows;
       }
      elsif ($r->at_colspec_tgroup and $h == $tgroup)                           # Colspec
       {++$colSpec;
       }
     }
   });

  genHash(q(Data::Edit::Xml::Table::Statistics),                                # Statistics about a table
    colsAttribute       => $tgroup->attr_cols,                                  # Column attribute
    maxHeadMinusPadding => $maxHeadMinusPadding,                                # Maximum number of entries in a head row after an padding entries have been removed
    maxHead             => $maxHead,                                            # Maximum number of entries in a head row regardless of padding rows
    minHead             => $minHead,                                            # Maximum number of entries in a head row regardless of padding rows
    maxBodyMinusPadding => $maxBodyMinusPadding,                                # Maximum number of entries in a body row after an padding entries have been removed
    maxBody             => $maxBody,                                            # Maximum number of entries in a body row regardless of padding rows
    minBody             => $minBody,                                            # Maximum number of entries in a body row regardless of padding rows
    colSpec             => $colSpec,                                            # Number of colspec entries
    rows                => $rows,                                               # Number of rows
   );
 }

sub ditaAddPadEntriesToTGroupRows($$)                                           #UP Adding padding entries to a tgroup to make sure every row has the same number of entries
 {my ($tgroup, $nEntries) = @_;                                                 # TGroup node, number of entries
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Confirm we are on a tgroup
  $tgroup->by(sub                                                               # Traverse tgroup
   {if (my ($r, $hb, $g) = @_)
     {if ($r->at_row_thead_tgroup || $r->at_row_tbody_tgroup and $g == $tgroup) # Check this row is in the current tgroup
       {my $cols = $r->ditaNumberOfColumnsInRow;                                # Number of columns
        for($cols..$nEntries-1)                                                 # Number of pad entries required in this row
         {$r->putLast($r->newTag(q(entry)));                                    # Add new padding entry
         }
       }
     }
   });
 }

sub ditaAddColSpecToTGroup($$)                                                  #U Add the specified B<$number> of column specification to a specified B<$tgroup> which does not have any already.
 {my ($tgroup, $number) = @_;                                                   # Tgroup node, number of colspecs to add
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Confirm we are on a tgroup node
  $tgroup->set(cols=>$number);                                                  # Set cols attribute
  my @c = $tgroup->c_colspec;                                                   # Existing colspecs
  $_->unwrap for @c;                                                            # Remove existing colspecs
  for my $col(reverse 1..$number)                                               # Add colspecs
   {$tgroup->putFirst($tgroup->newTag(q(colspec),                               # Colspec
      colname=>"c$col", colnum=>"$col", colwidth=>"1*"));
   }
 }

sub ditaFixTGroupColSpec($)                                                     #U Fix the colspec attribute and colspec nodes of the specified B<$tgroup>.
 {my ($tgroup) = @_;                                                            # Tgroup node
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Check we are on a tgroup
  my $N = $tgroup->ditaMaximumNumberOfEntriesInATGroupRow;                      # Maximum number of entries in a row
  $tgroup->ditaAddColSpecToTGroup($N);                                          # Add colspecs
 }

sub ditaRemoveTGroupTrailingEmptyEntries($)                                     #U Remove empty trailing entry
 {my ($tgroup) = @_;                                                            # Table node
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Confirm we are on a tgroup node
   {for my $hb($tgroup->c_thead, $tgroup->c_tbody)                              # THead and TBody in TGroup
     {for my $r($hb->c_row)                                                     # Rows in THead and TBody
       {my @e = $r->c_entry;                                                    # Entries in Row
        while(@e)                                                               # Each entry
         {if ($e[-1]->isAllBlankText)                                           # Trailing blank entry
           {$e[-1]->cut;                                                        # Remove trailing blank entry
            pop @e;                                                             # Remove entry
            next;                                                               # Next trailing entry
           }
          last;                                                                 # Not blank
         }
       }
     }
   }
 }

sub fixTGroup($)                                                                #UC Fix the specified B<$tgroup> so that each row has the same number of entries with this number reflected in the tgroup.cols= attribute and colspec nodes.
 {my ($tgroup) = @_;                                                            # TGroup node
  $tgroup->at_tgroup or confess "Not a tgroup node: ".$tgroup->tag;             # Check we are on a tgroup

  $tgroup->by(sub
   {my ($o, $p) = @_;                                                           # Misplaced rows
    if ($o->at_row_tgroup)
     {if ($p == $tgroup)
       {my $b = $o->wrapWith_tbody;
        $b->mergeLikePrev;
       }
     }
   });

  my $stats     = $tgroup->ditaTGroupStatistics;                                # Statistics for tgroup
  my $cols      = $stats->colsAttribute;
  my $maxCols   = max($stats->maxHead//0, $stats->maxBody//0);
  my $maxColsMP = max($stats->maxHeadMinusPadding//0,
                          $stats->maxBodyMinusPadding//0);
  if (($stats->maxHead//0) == $maxCols and                                      # The right combination
      ($stats->minHead//0) == $maxCols and
      ($stats->maxBody//0) == $maxCols and
      ($stats->minBody//0) == $maxCols and
       $stats->colSpec     == $maxCols)
   {if (!$cols or $cols != $maxCols)                                            # Cols wrong but everything else ok
     {$tgroup->ditaAddColSpecToTGroup($maxCols);
     }
   }
  else                                                                          # Repad columns
   {$tgroup->ditaRemoveTGroupTrailingEmptyEntries;
    $tgroup->ditaAddPadEntriesToTGroupRows($maxColsMP);
    $tgroup->ditaAddColSpecToTGroup($maxColsMP);
   }
  $tgroup->ditaFixTGroupColSpec;
 }

sub fixTable($)                                                                 #U Fix the specified B<$table> so that each row has the same number of entries with this number reflected in the tgroup.cols= attribute and colspec nodes.
 {my ($table) = @_;                                                             # Table node
  $table->at_table or confess "Not a table node: ".$table->tag;                 # Check we are on a table

  if ($table->isAllBlankText)                                                   # Add the text of a minimal table declaration if the table is blank
   {$table->putFirstAsTree(<<END);
<tgroup cols="1">
  <tbody>
    <row><entry/></row>
  </tbody>
</tgroup>
END
   }
  else                                                                          # Try to fix a non blank table
   {for my $g($table->c_tgroup)                                                 # Each tgroup
     {$g->fixTGroup;
     }
#   if (!$table->go_title)                                                      # Add previous p as title if no title present and the p does not have prohibited items in it
#    {if (my $p = $table->prev_p)                                               # Holding off on this addition as it destroys some existing conversions
#      {my $c = $p->countTagNames;
#       if (!$$c{xref})
#        {$table->putFirstCut($p->change_title);
#        }
#      }
#    }
   }
  $table
 }

sub fixEntryColSpan($)                                                          #U Fix the colspan on an entry assumed to be under row, tbody, tgroup with @cols and colspecs' set
 {my ($entry) = @_;                                                             # Entry
  $entry->at_entry_row_tbody_tgroup or confess "Not a usable entry node";       # Check we are on an entry of the right kind

  my $colspan = $entry->attr_colspan;                                           # colspan
  return $entry unless $colspan;
  $colspan =~ m(\A\d+\Z)s or confess "Unexpected colspan value $colspan";

  my (undef, $row, $tbody, $tgroup) = $entry->ancestry;                         # cols
  my $cols = $tgroup->attr_cols or
    confess "No \@cols on tgroup ". (-A $tgroup). "\n";

  my @cols = grep {defined $_} map {$_->attr_colname} $tgroup->c_colspec;       # colspec
  @cols == $cols or confess "cols/colspec mismatch ".(-p $tgroup). "\n";
  $colspan > 0 and $colspan <= $cols or
    confess "Colspan $colspan out of range $cols";
  my %cols = map {$cols[$_]=>$_} keys @cols;

  $entry->deleteAttr_colspan;
  return $entry if $cols == 1 or $colspan == 1;                                 # Degenerate cases

  if ($entry->isOnlyChild)
   {$entry->set(namest=>$cols[0], nameend=>$cols[-1]);
   }
  elsif ($entry->isFirst)                                                       # Easy case`
   {$entry->set(namest=>$cols[0], nameend=>$cols[$colspan-1]);
   }
  elsif ($entry->isLast)                                                        # Easy case
   {$entry->set(namest=>$cols[-$colspan], nameend=>$cols[-1]);
   }
  else                                                                          # Normal case
   {sub
     {my @c = $entry->contentBefore;
      for my $e(@c)
       {if (my $end = $e->attr_nameend)
         {if (my $c = $cols{$end})
           {$entry->set(namest=>$cols[$c+1], nameend=>$cols[$c+$colspan]);
            return;
           }
         }
       }
      my $c = @c;
      $entry->set(namest=>$cols[$c], nameend=>$cols[$c+$colspan-1]);
     }->();
   }
 }

sub fixEntryRowSpan($)                                                          #U Fix the rowspan on an entry
 {my ($entry) = @_;                                                             # Entry
  $entry->at_entry or confess "Not an entry node";                              # Check we are on entry

  my $rowspan = $entry->attr_rowspan;                                           # rowspan
  return $entry unless $rowspan;

  $rowspan =~ m(\A\d+\Z)s or confess "Unexpected rowspan $rowspan value";
  $rowspan > 0 or confess "Rowspan $rowspan out of range";
  $entry->set(morerows=>$rowspan-1);
  $entry->deleteAttr_rowspan;
  $entry
 }

sub ditaConvertDlToUl($)                                                        #U Convert a L<Dita> B<$dl> to a B<ul> if each B<dlentry> has only B<dt> or B<dl> elements but not both.  Return B<undef> if such a conversion is not possible else return the new B<ul> node.
 {my ($dl) = @_;                                                                # Dl
  return undef unless $dl->at_dl;                                               # Check start tag

  my @e; my @l;
  for my $e($dl->c_dlentry)                                                     # Each dlentry
   {my @t = $e->c_dt;                                                           # dt elements
    my @d = $e->c_dd;                                                           # dd elements
    return undef if @d and @t;                                                  # Mixed entry suppresses the conversion
    push @e, $e;                                                                # Unwrap thee elements later
    push @l, @t, @d;                                                            # Convert these elements to li
   }

  $_->unwrap    for @e;                                                         # Unwrap dlentry
  $_->change_li for @l;                                                         # Change dt/dd to li
  $dl->change_ul;                                                               # Change dl to ul and return
 }

sub ditaConvertOlToSubSteps($@)                                                 #U Convert a L<Dita> B<$ul> to B<substeps> else B<undef> if this is not possible.
 {my ($ol, @context) = @_;                                                      # Ul, context
  return undef if !$ol->at(q(ol), @context);                                    # Not in specified context

  for my $e($ol->c_li)                                                          # Each li
   {$e->change_cmd__wrapWith_substep;                                           # Change li to cmd\substep
   }

  $ol->change_substeps;                                                         # Change ol to substeps
 }

sub ditaConvertUlToSubSteps($@)                                                 #U Convert a L<Dita> B<$ol> to B<substeps> else B<undef> if this is not possible.
 {my ($ul, @context) = @_;                                                      # Ul, context
  return undef if !$ul->at(q(ul), @context);                                    # Not in specified context

  for my $e($ul->c_li)                                                          # Each li
   {$e->change_cmd__wrapWith_substep;                                           # Change li to cmd\substep
   }

  $ul->change_substeps;                                                         # Change ul to substeps
 }

sub ditaConvertFromHtmlDl($)                                                    #U Convert a B<Html> B<$dl> to a L<Dita> B<dl> or return B<undef> if this is not possible.
 {my ($dl) = @_;                                                                # Dl
  return undef unless $dl->at_dl;                                               # Check start tag

  my $t;                                                                        # Last entry
  for my $e(reverse @$dl)                                                       # Each dt/dd
   {if ($e->at_dt)                                                              # dt
     {if ($t)                                                                   # Wrap to last entry with dlentry
       {$t = $e->wrapTo($t->prev, q(dlentry));
       }
      else                                                                      # Wrap to end with dlentry
       {$t = $e->wrapToLast(q(dlentry));
       }
     }
   }
  $dl
 }

sub ditaConvertSimpleTableToTable($)                                            #U Convert a L<Dita> B<simpletable> to a B<table>.
 {my ($simpleTable) = @_;                                                       # Simple table
  $simpleTable->at_simpletable or confess "Not on a simple table";              # Not a simpletable

  $simpleTable->by(sub                                                          # Traverse table making changes
   {if (my ($s) = @_)
     {if    ($s->at_simpletable)                                                # Simple table
       {$s->change_table;                                                       # Change to table
        my $g = $s->wrapContentWith_tgroup;                                     # Wrap content with tgroup
        my $b = $g->wrapContentWith_tbody;                                      # Wrap content with tbody
        if (my $h = $b->go_thead)                                               # Move thead out of tbody
         {$b->putPrevCut($h);
         }
        $s->fixTable;                                                           # Fix remaining problems
        $s->deleteAttr_relcolwidth;
       }
      elsif ($s->at_sthead)                                                     # thead
       {$s->change_thead;
        $s->wrapContentWith_row;
       }
      elsif ($s->at_stentry)                                                    # entry
       {$s->change_entry;
       }
      elsif ($s->at_strow)                                                      # row
       {$s->change_row;
       }
     }
   });

  $simpleTable                                                                  # Simple table is now a normal table
 }

sub ditaCouldConvertConceptToTask($)                                            #U Check whether a concept could be converted to a task
 {my ($concept) = @_;                                                           # Concept to check
  return () unless $concept->at_concept;                                        # Not a concept
  my $body = $concept->go_conbody;                                              # No conbody
  return () unless $body;
  $body->c_ol;                                                                  # Ol to break on
 }

sub ditaConvertConceptToTask($@)                                                #U Convert a Dita B<concept> to a B<task> by representing B<ol> as B<steps>. Return B<undef> if the conversion is not possible because there are no B<ol> else return the specified B<$concept> as as B<task>.
 {my ($node, @context) = @_;                                                    # Node, optional context

  return undef if @context and !$node->at(@context);                            # Node not in specified context
  return undef unless my $concept = $node->upUntil_concept;

  my @ol = $concept->ditaCouldConvertConceptToTask;                             # Breaking ol
  return undef unless @ol;

  my $body = $concept->go_conbody;                                              # Conbody guaranteed by prior check
  if (my $p = $ol[0]->prev)                                                     # Items before first ol become context
   {$p->wrapFromFirst_context;
   }
  else
   {$body->putFirst($body->newTag_context);
   }

  for my $i(grep {$_ > 0} keys @ol)                                             # Items between ol become stepsection
   {my $o = $ol[$i];
    my $O = $ol[$i-1];
    if ($o != $O)
     {$O->wrapSiblingsBetween($o, q(stepsection));
     }
   }

  if (my $p = $ol[-1]->next)                                                    # Items after last ol become results
   {$p->wrapToLast_result;
   }
  else
   {$body->putLast($body->newTag_result);
   }

  $ol[0]->wrapTo($ol[-1], q(steps));                                            # Range of ol becomes steps

  for my $ol(@ol)                                                               # Each ol becomes steps
   {for my $li($ol->c_li)
     {my @note;
      while(my $first = $li->first(qr(\A(fig|image|note)\Z)))
       {push @note, $first->cut;
       }

       if (my $first = $li->first)                                              # Change something to cmd
       {if ($first->isText)
         {$first->wrapWith_cmd;
         }
        else
         {$first->change_cmd;
         }
       }
      if (my $first = $li->first)                                               # Remainder after cmd becomes info
       {if (my $last = $li->last)
         {if ($first != $last)
           {if (my $next = $first->next)
             {$next->wrapTo($last, q(info));
             }
           }
         }
       }

      if (my $cmd = $li->first_cmd)                                             # Ameliorate any trailing items in cmd by putting them first in context
       {while(my $last = $cmd->last(qr(\A(fig|image|note)\Z)))
         {my $info = $li->addLast_info;
             $info->putFirstCut($last);
         }
        if ($cmd->over2(qr(\A( li )+\Z)))                                       # Li in cmd to substeps
         {my $choices = $cmd->wrapContentWith_ol->ditaListToChoices;
          $cmd->putNextCut($choices);
          $cmd->putFirstAsText(q(Choose one of the following:));
         }
       }

      if (@note)                                                                # Preceding notes, images, figures
       {for my $n(reverse @note)
         {$li->putFirst($n);
          if (!$n->at_note)
           {$n->wrapWith_note;
           }
         }
       }

      $li->change_step;                                                         # li to step
      if (!$li->go_cmd)                                                         # Make sure there is a command for the step
       {if (my $n = $li->go_note)
         {$n->putNextAsTree(q(<cmd/>));
         }
       else
         {$li->putFirstAsTree(q(<cmd/>));
         }
       }
     }
    $ol->unwrap;
   }

  $concept->change_task;
  $body   ->change_taskbody;

  $concept->go_taskbody_result__hasSingleChild_section__unwrap__change_example; # Unwrap section only child of result and make result into an example
  $concept->go_taskbody_result__hasSingleChild_example__parent__unwrap;         # Unwrap result around example

  $concept
 }

BEGIN{*ct=*ditaConvertConceptToTask}

sub ditaConvertReferenceToConcept($)                                            #U Convert a Dita B<reference> to a B<concept> by unwrapping sections. Return B<undef> if the conversion is not possible because there are no B<ol> else return the specified B<$reference> as as B<concept>.
 {my ($reference) = @_;                                                         # Reference
  $reference->at_reference or confess "Not a reference";                        # Check we are on reference

  my $body = $reference->go_refbody;
     $body or confess "No refbody under reference";

  $reference->change_concept;                                                   # Change topic to concept
  $body     ->change_conbody;

  $reference->by(sub                                                            # Unwrap sections
   {$_->unwrap_section;
   });

  $reference                                                                    # Return reference as a concept
 }

sub ditaConvertReferenceToTask($)                                               #U Convert a Dita B<reference> to a B<task> in situ by representing B<ol> as B<steps>. Return B<undef> if the conversion is not possible because there are no such B<ol> else return the specified B<$reference> as as B<task>.
 {my ($reference) = @_;                                                         # Reference
  $reference->at_reference or confess "Not a reference";                        # Check we are on reference

  my $body = $reference->go_refbody;
     $body or confess "No refbody under reference";

# my $ol;                                                                       # Check that the necessary ol is present
# $body->by(sub
#  {my ($o) = @_;
#   if ($o->at_ol_section_refbody)
#    {++$ol;
#    }
#  });
# return undef unless $ol;

  if (my $concept = $reference->ditaConvertReferenceToConcept)
   {return $concept->ditaConvertConceptToTask;
   }
  confess "Unable to convert reference to task";
 }

sub ditaConvertConceptToReference($)                                            #U Convert a Dita B<concept> to a B<reference>. Return B<undef> if the conversion is not possible else return the specified B<$concept> as as B<reference>.
 {my ($concept) = @_;                                                           # Concept
  $concept->at_concept or confess "Not a concept";                              # Check we are on concept

  my $body = $concept->go_conbody;
     $body or confess "No conbody under concept";

  if (my @s = $body->c_section)                                                 # Sections in concept
   {if (my $p = $s[0]->prev)
     {$body->wrapContentWith_section->moveEndAfter($p);
     }
    if (my $n = $s[-1]->next)
     {$body->wrapContentWith_section->moveStartBefore($n);
     }
   }
  else
   {$body->wrapContentWith_section;
   }

  $concept->change_reference;
  $body   ->change_refbody;

#  if (!$concept->id)
#   {$concept->createGuidId;
#   }

  $concept
 }

sub ditaConvertTopicToTask($)                                                   #U Convert a topic that is not already a task into a task
 {my ($x) = @_;                                                                 # Topic parse tree

  if ($x->at_concept)                                                           # Convert concept
   {$x->ditaConvertConceptToTask;
   }
  elsif ($x->at_reference)                                                      # Convert reference
   {$x->ditaConvertReferenceToTask;
   }

  $x->at_task                                                                   # Succeeded if we now have a task
 }

sub ditaConvertSectionToConcept($)                                              #U Convert a Dita B<$section> to a B<concept>. Return B<undef> if the conversion is not possible else return the specified B<$section> as as B<concept>.
 {my ($section) = @_;                                                           # Section
  $section->at_section or confess "Not a section, its a ",-t $section;          # Check we are on a section

  my $title = $section->first_title;                                            # Locate any existing title

  my $concept = $section->change_concept;                                       # Change section to concept
     $concept->wrapContentWith_conbody;

  if ($title)                                                                   # Move title into position
   {$concept->putFirstCut($title);
   }
  else
   {$concept->putFirst($concept->newTag_title);
   }

  $concept->createGuidId unless $concept->id;                                   # Create an id for the topic

  $concept                                                                      # Return concept
 }

BEGIN{*sc=*ditaConvertSectionToConcept}

sub ditaConvertConceptToSection($)                                              #U Convert a Dita B<concept> to a B<$section> . Return B<undef> if the conversion is not possible else return the specified B<$section> as as B<concept>.
 {my ($concept) = @_;                                                           # Section
  $concept->at_concept or confess "Not a concept, its a ",-t $concept;          # Check we are on a concept

  if (my $conbody = $concept->go_conbody)                                       # Unwrap conbody
   {$conbody->unwrap;
   }

  if (my $title = $concept->first_title)                                        # Remove any existing blank title
   {if ($title->isAllBlankText)
     {$title->cut;
     }
   }

  my $section = $concept->change_section;                                       # Change concept to section

  $section                                                                      # Return section
 }

BEGIN{*cs=*ditaConvertConceptToSection}

sub ditaConvertSectionToReference($)                                            #U Convert a Dita B<$section> to a B<reference>. Return B<undef> if the conversion is not possible else return the specified B<$section> as as B<reference>.
 {my ($section) = @_;                                                           # Section
  if (my $c = $section->ditaConvertSectionToConcept)                            # First convert section to concept
   {return $c->ditaConvertConceptToReference;                                   # Second convert concept to reference
   }
  undef                                                                         # Conversion not possible
 }

sub ditaConvertSectionToTask($)                                                 #U Convert a Dita B<$section> to a B<task>. Return B<undef> if the conversion is not possible else return the specified B<$section> as as B<task>.
 {my ($section) = @_;                                                           # Section
  if (my $c = $section->ditaConvertSectionToConcept)                            # First convert section to concept
   {return $c->ditaConvertConceptToTask;                                        # Second convert concept to task
   }
  undef                                                                         # Conversion not possible
 }

sub ditaObviousChanges($)                                                       #U Make obvious changes to a L<parse|/parse> tree to make it look more like L<Dita>.
 {my ($node) = @_;                                                              # Node

  $node->by(sub                                                                 # Do the obvious conversions
   {my ($o) = @_;

    my %change =                                                                # Tags that should be changed
     (a            => q(xref),
      book         => q(bookmap),
      code         => q(codeph),
      command      => q(codeph),                                                # Needs approval from Micalea
      emphasis     => q(b),
      figure       => q(fig),
      guibutton    => q(uicontrol),
      guilabel     => q(uicontrol),
      guimenu      => q(uicontrol),
      itemizedlist => q(ul),                                                    # PS2-570
      link         => q(xref),
      listitem     => q(li),
      menuchoice   => q(uicontrol),
      orderedlist  => q(ol),
      para         => q(p),
      quote        => q(q),
      replaceable  => q(varname),
      span         => q(div),                                                   # Span to div at 2019.05.19 18:59:50
      subscript    => q(sub),
      variablelist => q(dl),
      varlistentry => q(dlentry),
     );

    my %deleteAttributesDependingOnValue =                                      # Attributes that should be deleted if they have specified values
     (b=>[[qw(role bold)], [qw(role underline)]],
     );

    my @deleteAttributesUnconditionally =                                       # Attributes that should be deleted unconditionally from all tags that have them
#    qw(version xml:id xmlns xmlns:xi xmlns:xl xmlns:d);
     qw(xml:id xmlns xmlns:xi xmlns:xl xmlns:d);

    my %renameAttributes =                                                      # Attributes that should be renamed
     (xref      => [[qw(linkend href)],       [qw(xrefstyle outputclass)]],     # 2018.06.14 added xrefstyle->outputclass
      fig       => [[qw(role outputclass)],   [qw(xml:id id)]],
      example   => [[qw(role outputclass)]],
      imagedata => [[qw(contentwidth scale)], [qw(fileref href)]],
      image     => [[qw(src href)]],                                            # Image source at 2019.05.19 19:04:50
     );

    for my $old(sort keys %change)                                              # Perform requested tag changes
     {$o->change($change{$old}) if $o->at($old);
     }

    for my $tag(sort keys %deleteAttributesDependingOnValue)                    # Delete specified attributes if they have the right values
     {if ($o->at($tag))
       {$o->deleteAttr(@$_) for @{$deleteAttributesDependingOnValue{$tag}};
       }
     }

    $o->deleteAttrs(@deleteAttributesUnconditionally);                          # Delete attributes unconditionally from all tags that have them

    for my $tag(sort keys %renameAttributes)                                    # Rename specified attributes
     {if ($o->at($tag))
       {$o->renameAttr(@$_) for @{$renameAttributes{$tag}};
       }
     }
   });

  $node->addFirst(q(title)) if $node->at(qr(\A(concept|reference|task)\Z));     # Make sure we have a title
  $node
 }

sub ditaXrefs($)                                                                #U Make obvious changes to all the B<xref>s found in a L<parse|/parse> tree to make them more useful in L<Dita>.
 {my ($x) = @_;                                                                 # Parse tree

  $x->by(sub                                                                    # NEX116 at 2018.12.28 01:10:40
   {my ($r, $b) = @_;
    if ($r->at_xref)
     {if (my $h = $r->href)
       {if ($h =~ m(\Ahttps?://|\Awww\.|\.com\Z|\.net\Z|\.org\Z)s)
         {$r->set(scope=>q(external), format=>q(html));
         }
       }
      if ($b and my $tag = -t $b)                                               # Wrap the xref with a p if immediately under a body
       {if ($tag =~ m(body\Z)s)
         {$r->wrapWith_p;
         }
       }
     }
   });
 }

sub ditaSampleConcept(%)                                                        #S Sample concept
 {my (@options) = @_;                                                           # Options for concept
  shift @options while @options && ref $options[0];                             # Remove any leading references to find the actual string or file to be parsed

  my %options  = @options;
  checkKeys(\%options,                                                          # Check options
    {title     =>q(Title of the concept),
     metadata  =>q(Metadata of the concept ),
     body      =>q(Body of the concept ),
    });

  my $title    = $options{title}  // "Title unknown - please provide one using the title keyword";
  my $prolog   = $options{prolog} // q();
  my $body     = $options{body}   // "<p>Please provide the body of this concept using the body keyword</p>";

  my $metadata = sub
   {my $m = $options{metadata};
    return '' unless $m;
    qq(<prolog><metadata>$m</metadata></prolog>\n)
   }->();

  my $concept  = new(<<END);
<concept>
  <title id="title">$title</title>
  $metadata
  <conbody>$body</conbody>
</concept>
END

  $concept->createGuidId;
  $concept
 }

sub ditaSampleTask(%)                                                           #S Sample task
 {my (@options) = @_;                                                           # Options for task
  shift @options while @options && ref $options[0];                             # Remove any leading references to find the actual string or file to be parsed
  my %options  = @options;
  checkKeys(\%options,                                                          # Check options
    {title     =>q(Title of the task),
    });

  my $title    = $options{title}  // "Title unknown";
  my $task  = new(<<END);
<task>
  <title id="title">$title</title>
  <taskbody>
    <context/>
    <steps/>
    <result/>
  </taskbody>
</task>
END
  $task->createGuidId;
  $task
 }

sub ditaSampleBookMap(%)                                                        #S Sample bookmap
 {my (@options) = @_;                                                           # Options for bookmap

  shift @options while @options && ref $options[0];                             # Remove any leading references to find the actual string or file to be parsed
  my %options  = @options;
  checkKeys(\%options,                                                          # Check bookmap options
    {appendices=>q(Appendices),
     author    =>q(Author of the document),
     chapters  =>q(Chapter and  topicrefs),
     metadata  =>q(Meta data),
     notices   =>q(Name of file containing notices),
     title     =>q(Title of the document),
     year      =>q(Copyright year),
    });

  my $author   = sub                                                            # Author
   {my $a = $options{author};
    return qq(<author>$a</author>) if $a;
    q(<author/>)
   }->();

  my $title    = sub                                                            # Title
   {my $t = $options{title};
    return qq(<mainbooktitle>$t</mainbooktitle>) if $t;
    q(<mainbooktitle/>)
   }->();

  my $year     = sub                                                            # Year
   {my $y = $options{year};
    return qq(<year>$y</year>) if $y;
    q(<year/>)
   }->();

  my $chapters = sub
   {my $c = $options{chapters};
    return '' unless $c;
    return -p $c if ref($c);
    $c
   }->();

  my $appendices = sub
   {my $c = $options{appendices};
    return '' unless $c;
    return -p $c if ref($c);
    $c
   }->();

  my $notices = sub
   {my $n = $options{notices};
#   return ' <notices/>' unless $n;
    return ' <notices/>' unless $n;
    qq(<notices href="$n" navtitle="Notices"/>)
   }->();

  my $metadata = sub
   {my $m = $options{metadata};
    return '' unless $m;
    qq(<metadata>$m</metadata>\n)
   }->();

  my $bookMap = new(<<END);                                                     # Sample bookmap
<bookmap>
  <booktitle>
    $title
  </booktitle>
  <bookmeta>
    <shortdesc/>
    $author
    <source/>
    $metadata
    <category/>
    <keywords>
      <keyword/>
    </keywords>
    <prodinfo>
      <prodname product=""/>
      <vrmlist>
        <vrm version=""/>
      </vrmlist>
      <prognum/>
      <brand/>
    </prodinfo>
    <bookchangehistory>
      <approved>
        <revisionid/>
      </approved>
    </bookchangehistory>
    <bookrights>
      <copyrfirst>
        $year
      </copyrfirst>
      <bookowner/>
    </bookrights>
  </bookmeta>
 <frontmatter>
  $notices
  <booklists>
  <toc/>
  </booklists>
  <preface/>
</frontmatter>
$chapters
<appendices>
$appendices
</appendices>
<reltable>
    <relheader>
        <relcolspec/>
        <relcolspec/>
    </relheader>
    <relrow>
        <relcell/>
        <relcell/>
    </relrow>
    <relrow>
        <relcell/>
        <relcell/>
    </relrow>
</reltable>
</bookmap>
END
  $bookMap->createGuidId;                                                       # Id for bookmap
  $bookMap
 }

sub isADitaMap($)                                                               #U Return the specified B<$node> if this node is a L<Dita> map else return B<undef>
 {my ($node) = @_;                                                              # Node to test
  $node->tag =~ m(map\Z) ? $node : undef                                        # Map
 }

sub topicTypeAndBody($)                                                         #P Topic type and corresponding body.
 {my ($type) = @_;                                                              # Type from qw(bookmap concept reference task)
  return qw(bookmap     BookMap)               if $type =~ /\Abookmap/i;
  return qw(concept     Concept    conbody)    if $type =~ /\Aconcept/i;
  return qw(glossentry  Glossary   glossentry) if $type =~ /\Aglossentry/i;
  return qw(map         Map)                   if $type =~ /\Amap/i;
  return (q(map),     q(Subject Scheme Map))   if $type =~ /\AsubjectScheme/i;
  return qw(reference   Reference  refbody)    if $type =~ /\Areference/i;
  return qw(task        Task       taskbody)   if $type =~ /\Atask/i;
  return qw(topic       Topic      body)       if $type =~ /\Atopic/i;
  return qw(unknown     Unknown    unknown);
  confess "Unknown document type: $type\n";
 }

sub ditaRoot($)                                                                 #U Return the specified B<$node> if it a L<Dita> root node else return B<undef>.
 {my ($node) = @_;                                                              # Node to check
  return $node if $node->tag =~
    m(\A(bookmap|concept|glossentry|map|subjectScheme|reference|task|topic)\Z)s;
 }

sub ditaTopicHeaders($;$)                                                       #U Add L<XML> headers for the dita document type indicated by the specified L<parse|/parse> tree
 {my ($node, $String)  = @_;                                                    # Node in parse tree, suffix string
  my $string  = $String // q();
  my $parse   = $node->parser;
  my  $t      = $parse->tag;
  my ($n, $N) = topicTypeAndBody $t;
  my $r = xmlHeader(<<END) =~ s(\s*\Z) (\n)sr;
<!DOCTYPE $t PUBLIC "-//OASIS//DTD DITA $N//EN" "$n.dtd" []>
$string
END
  $r
 }

sub ditaPrettyPrintWithHeaders($)                                               #U Add L<XML> headers for the dita document type indicated by the specified L<parse|/parse> tree to a pretty print of the parse tree.
 {my ($node)  = @_;                                                             # Node in parse tree
  my $s = -p $node;

  if ($node->isADitaMap and $s =~ m(<topicsubject)i)                            # An exception to the general rule
   {return xmlHeader <<END.$s
<!DOCTYPE map PUBLIC "-//OASIS//DTD DITA Classification Map//EN" "classifyMap.dtd">
END
   }

  $node->ditaTopicHeaders($s);
 }

#sub ditaFindFirstFailure($@)                                                    #U Return the first node that fails to conform to the L<Dita> standard starting at the specified B<$node>. Return B<undef> if no failing node was found.
# {my ($node, @context) = @_;                                                    # Node, optional context
#  return undef if @context and !$node->at(@context);                            # Not in specified context
#
#  my $r = Dita::Validate::firstFailure($node);                                  # Validate node along path from parent
#  return $r if $r;                                                              # Specified node is invalid
#  for my $n($node->contents)                                                    # Check children
#   {my $r = Dita::Validate::firstFailure($n);                                   # Validate child
#    return $r if $r;                                                            # Return invalid child node
#   }
#  undef                                                                         # No failing node found
# }
###a
###b <concept><title/><conbody><p/><q/></conbody></concept>
###c atTop
###c ditaFindFirstFailure
###c set id failed
###d Find the first failing node

sub ditaCutTopicmetaFromAClassificationMap($@)                                  #U Remove a topicmeta node from a classification map. Dita::Validate was built from conventional maps and so does not recognize all the situations where topicmeta is invalid in a classification map
 {my ($node, @context) = @_;                                                    # Node, optional context
  return undef if @context and !$node->at(@context);                            # Not in specified context

  if (@context and $context[0]=~ m(topicmeta)i)                                 # Pending DTD parsing we have to do this manually
   {if ($node->isAllBlankText and !$node->isFirst and !$node->prev_title)
     {$node->unwrap;
     }
    return $node;
   }
  undef
 }

#sub ditaCutIfEmptyAndFirstFailingChild($@)                                      #U Remove this node if is in the specified optional context, empty and the first child under its parent which fails to conform to the L<Dita> standard. Return $node if the node is valid else B<undef>.
# {my ($node, @context) = @_;                                                    # Node, optional context
#  return undef if @context and !$node->at(@context);                            # Not in specified context
#
#  my     $r = Dita::Validate::validateChild($node);                             # Validate node along path from parent
#  if    ($r == 0) {                   return undef}                             # Not a Dita node
#  elsif ($r == 1) {                   return undef}                             # Not reachable
#  elsif ($r == 2) {$node->cutIfEmpty; return undef}                             # Reachable but in error: cut if empty, stop the PCD
#  else            {                   return $node}                             # Reachable and valid
# }
##a
##b <concept><title/><p/></concept>
##c ditaCutIfEmptyAndFirstFailingChild p
##d Remove first empty child that fails to conform to Dita.

sub ditaAddTopicReport($$)                                                      # Place a report into a dita topic using required clean up
 {my ($tree, $report) = @_;                                                     # Topic, report
  $tree->downToDie(sub
   {my ($o) = @_;
    if ($o->at_conbody)
     {$o->putFirstRequiredCleanUp($report);
      die;
     }
    elsif ($o->at_refbody)
     {my $c = $o->addFirst_section;
      $c->putFirstRequiredCleanUp($report);
      die;
     }
    elsif ($o->at_taskbody)
     {my $c = $o->addFirst_context;
      $c->putFirstRequiredCleanUp($report);
      die;
     }
   });
 }

our $defaultBrowser;                                                            # Name of default browser executable
our $defaultEditorHelp =                                                        # Editor help Url
  q(https://philiprbrenan.github.io/data_edit_xml_edit_commands.html);

sub help($)                                                                     #U Get help for a node and the editor
 {my ($node) = @_;                                                              # Node
  my $t = -t $node;                                                             # Node in question
  my $l = substr($t, 0, 1);                                                     # Index page
  my $u = qq(http://docs.oasis-open.org/dita/dita/v1.3/errata02/os/complete/part3-all-inclusive/contentmodels/cmlt${l}.html#cmlt${l}__$t);
  return $u unless $defaultBrowser;                                             # Return the url unless we have a default browser
  qx($defaultBrowser $u & $defaultBrowser $defaultEditorHelp &)                 # Display url in default browser
 }

BEGIN{*h=*help}

#D2 Html and word conversions                                                   # Methods useful for converting Word and L<Html> to L<Dita>

sub htmlHeadersToSections($)                                                    #U Position sections just before html header tags so that subsequently the document can be divided into L<sections|/divideDocumentIntoSections>.
 {my ($tree) = @_;                                                              # Parse tree

  $tree->by(sub                                                                 # Move each section definition upwards so that its parent is another section. Intervening container tags such as <div> or <span> should have been unwrapped by this point as they might move the section further back than is desired.
   {my ($o) = @_;
    if ($o->tag =~ m(\Ah(\d)\Z)i)
     {my $level = $1;
      $o->putPrev($o->newTag(q(section), level=>$level));
     }
   });
 }

sub getSectionHeadingLevel($)                                                   #UP Get the heading level from a section tag.
 {my ($o) = @_;                                                                 # Node
  return undef unless $o->at(qq(section));
  $o->attr(qq(level))
 }

sub unwrapSingleParentsOfSection($)                                             #UP Unwrap single parents of section: in word documents the header is often buried in a list to gain a section number - here we remove these unnecessary items
 {my ($tree) = @_;                                                              # Parse tree

  $tree->by(sub
   {my ($o) = @_;
    if ($o->at(qq(section)))
     {$o->unwrapParentsWithSingleChild;
     }
   });
 }

sub extendSectionToNextSection($)                                               #UP Extend a section tag until it meets the next section tag
 {my ($tree) = @_;                                                              # Parse tree

  $tree->by(sub                                                                 # Place each non heading node in its corresponding heading node
   {my ($o) = @_;
    if (my $h = getSectionHeadingLevel($o))
     {while(my $n = $o->next)
       {last if getSectionHeadingLevel($n);
        $n->cut;
        $o->putLast($n);
       }
     }
   });
 }

sub structureAdjacentSectionsByLevel($)                                         #UP Structure adjacent sections by level
 {my ($tree) = @_;                                                              # Parse tree

  $tree->by(sub                                                                 # Place each sub section in its containing section
   {my ($o) = @_;
    if (my $h = getSectionHeadingLevel($o))
     {while(my $n = $o->next)
       {my $i = getSectionHeadingLevel($n);
        last if !defined($i) or $i <= $h;
        $o->putLast($n->cut);
       }
     }
   });
 }

sub divideDocumentIntoSections($$)                                              # Divide a L<parse|/parse> tree into sections by moving non B<section> tags into their corresponding B<section> so that the B<section> tags expand until they are contiguous. The sections are then cut out by applying the specified sub to each B<section> tag in the L<parse|/parse> tree. The specified sub will receive the containing B<topicref> and the B<section> to be cut out as parameters allowing a reference to the cut out section to be inserted into the B<topicref>.
 {my ($node, $cutSub) = @_;                                                     # Parse tree, cut out sub

  $node->unwrapSingleParentsOfSection;                                          # In word documents the header is often buried in a list to gain a section number - here we remove these unnecessary items
  $node->extendSectionToNextSection;                                            # Place each non heading node in its corresponding heading node
  $node->structureAdjacentSectionsByLevel;                                      # Place each sub section in its containing section

  $node->by(sub                                                                 # Wrap each section in topicrefs
   {my ($o) = @_;
    if ($o->at(qq(section)))
     {my $t = $o->wrapWith(q(topicref));                                        # Topic ref
      $t->putLast($_->cut) for $o->c(q(topicref));                              # Move topics out of section into containing topic
     };
   });

  $node->by(sub                                                                 # Cut out each section
   {my ($o, $p) = @_;
    if ($o->at(qq(section)))
     {$p->at(q(topicref)) or confess "Section not in topicref";
      $cutSub->($p, $o);
     }
   });
 }

sub divideHtmlDocumentIntoSections($)                                           #U Divide a L<parse|/parse> tree representing an html document into sections based on the heading tags.
 {my ($tree) = @_;                                                              # Parse tree
  $tree->htmlHeadersToSections;
  $tree->extendSectionToNextSection;                                            # Place each non heading node in its corresponding heading node
  $tree->structureAdjacentSectionsByLevel;                                      # Place each sub section in its containing section
  $tree->by(sub{$_->change(q(title), qr(\Ah\d\Z)i)});                           # Move each section definition upwards so that its parent is another section. Intervening container tags such as <div> or <span> should have been unwrapped by this point as they might move the section further back than is desired.
  $tree->by(sub{$_->renameAttr_level_outputclass if $_->at_section});           # Change level to outputclass
 }

sub ditaParagraphToNote($;$)                                                    #U Convert all <p> nodes to <note> if the paragraph starts with 'Note:', optionally wrapping the content of the <note> with a <p>
 {my ($node, $wrapNoteContentWithParagaph) = @_;                                # Parse tree, wrap the <note> content with a <p> if true
  my $count = 0;                                                                # Count the number of changes
  $node->by(sub                                                                 # Each node
   {my ($o, $p) = @_;                                                           # Text, p
    if ($o->isText_p)                                                           # Text under p
     {if ($o->text =~ m(\A\s*(Attention|Caution|Danger|Fastpath|Important|Notices?|Notes?|Remember|Restriction|Tip|Trouble|Warning)\s*:?\s*)i) # Note
       {my $note = $1;
        $p->change(q(note));                                                    # Change to note
        $p->wrapContentWith(q(p)) if $wrapNoteContentWithParagaph;              # Wrap content if required
        $o->text =~        s(\A\s*$note\s*:?\s*) ()i;                           # Remove note text
        $o->text = ucfirst($o->text);                                           # Uppercase leading character
        $note =~ s(s\Z) ()gs;                                                   # Remove optional trailing s
        $p->set(type=>lc $note) unless $note =~ m(\Anote\Z)is;                  # Set type
        ++$count;
       }
     }
   });
  $count
 }

sub wordStyles($)                                                               #U Extract style information from a parse tree representing a word document.
 {my ($x) = @_;                                                                 # Parse tree
  my $styles;                                                                   # Styles encountered
  $x->by(sub                                                                    # Each node
   {my ($o, $p) = @_;
    if ($o->at(qw(text:list-level-style-bullet text:list-style)))               # Bulleted lists
     {my ($level, $name) = ($o->attr(q(text:level)), $p->attr(q(style:name)));
      if ($level and $name)
       {$styles->{bulletedList}{$name}{$level}++;
       }
     }
   });
  $styles                                                                       # Return styles encountered
 }

sub htmlTableToDita($)                                                          #U Convert an L<html table> to a L<Dita> table.
 {my ($table) = @_;                                                             # Html table node

  $table->wrapContentWith(q(tgroup));                                           # tgroup
  my %transforms =                                                              # Obvious transformations
   (td=>q(entry),
    th=>q(entry),
    tr=>q(row),
   );

  $table->by(sub
   {if (my $c = $transforms{-t $_}) {$_->change($c)}
   });

  my $N = 0;                                                                    # Number of columns in widest row
  $table->by(sub
   {my ($r, undef, $group, $Table) = @_;
    if ($r->at(qw(row), undef, qw(tgroup table)) and $Table == $table)          # In this table, not in an embedded sub table
     {my $n = $r->c(q(entry));
      $N = $n if $n > $N;
     }
   });

  $table->by(sub                                                                # Fix colspecs
   {my ($group, $Table) = @_;
    if ($group->at(qw(tgroup table)) and $Table == $table)                      # In this table, not in an embedded sub table
     {$group->setAttr   (q(cols), $N);
      $_->unwrap for $group->c(q(colspec));
      $group->putFirst($group->newTag                                           # Insert colspecs
       (qw(colspec colname), qq(c$_), q(colnum), $_, qw(colwidth 1*)))
        for reverse 1..$N;
     }
   });

  $table->by(sub                                                                # Span last element of each row to fill row
   {my ($r, undef, $group, $Table) = @_;
    if ($r->at(qw(row), undef, qw(tgroup table)) and $Table == $table)
     {my @e = $r->c(q(entry));
      my $n = @e;
      $e[-1]->setAttr(namest=>qq(c$n), nameend=>qq(c$N)) if @e < $N;
     }
   });
 }

sub ditaSyntaxDiagramFromDocBookCmdSynopsis($)                                  #U Convert doc book cmdsynopsis to Dita syntax diagram
 {package ditaSyntaxDiagramFromDocBookCmdSynopsis;                              # Allows us to package the entire conversion as one method
  use Carp;
  my ($x) = @_;                                                                 # Parse tree

  sub command   {q(command)}                                                    # Constants encountered
  sub ellipsis  {q(<sep>&#x2026;</sep>)}
  sub importance{q(importance)}
  sub opt       {q(opt)}
  sub optional  {q(optional)}
  sub plain     {q(plain)}
  sub rep       {q(rep)}
  sub req       {q(req)}
  sub repeat    {q(repeat)}
  sub required  {q(required)}

  sub ditaOptional($)                                                           # Return the specified B<$node> if it is optional else return B<undef>.
   {my ($node) = @_;                                                            # Node to test
    return $node if $node->attrX_importance eq optional;
    undef
   }

  sub ditaNonOptionalGroupSeq($)                                                # Return the specified B<$node> if it is group sequence but not optional else return B<undef>.
   {my ($node) = @_;                                                            # Node to test
    return $node if $node->at_groupseq and !ditaOptional($node);
    undef
   }

  sub ditaOptionalGroupChoice($)                                                # Return the specified B<$node> if it is a group choice and optional else return B<undef>.
   {my ($node) = @_;                                                            # Node to test
    return $node if $node->at_groupchoice and ditaOptional($node);
    undef
   }

  sub ditaRepeated($)                                                           # Return the specified B<$node> if it is repeated else return B<undef>.
   {my ($node) = @_;                                                            # Node to test
    if (my $s = $node->last_seq)
     {if ($s->string eq ellipsis)
       {return $node;
       }
     }
    undef
   }

  sub docBookChoiceAttribute($)                                                 # Return the value of the B<choice> attribute of the B<arg> or B<group>  specified by B<$node> if we are on a B<arg> or B<group> node else confess.
   {my ($node) = @_;                                                            # Node to test
    $node->at_arg || $node->at_group or
      confess "Not an arg or group node".-A $node;
    $node->attrX_choice
   }

  $x->by(sub
   {my ($o, $p) = @_;

    if ($o->at_cmdsynopsis)                                                     # cmdsynopsis
     {$o->change_syntaxdiagram__wrapContentWith_groupseq;
      $o->renameAttr(qw(xml:id id));
     }

    if ($o->at_command)                                                         # command
     {$o->change_kwd;
      $o->outputclass = command;
     }

    $o->change_var_replaceable;                                                 # replaceable

    if ($o->isText_arg)                                                         # Text under arg becomes kwd
     {$o->wrapWith_kwd
     }

    if ($o->at_arg)                                                             # arg
     {my $c = docBookChoiceAttribute($o);

      if (!$c or $c eq opt)                                                     # Choice
       {$o->set(importance=>optional);
        $o->change_groupseq;
       }
      elsif ($c eq plain)
       {$o->change_groupseq;
       }
      elsif ($c eq q(req))
       {$o->change_groupchoice;
       }
      else
       {confess "Unprogrammed arg choice=$c\n";
       }

      if (my $r = $o->attr_rep)                                                 # Repetition
       {if ($r eq repeat)
         {$o->putLastAsTree(ellipsis);
          $o->wrapContentWith_groupseq unless $o->at_groupseq;                  # PS2-658 - wrap the repeated sequence unless it is already wrapped
         }
       }

      $o->deleteAttrs_choice_rep;                                               # Attributes processed
     }


# Brackets are used to distinguish between optional, required, or plain arguments. Usually square brackets are placed around optional arguments, [-f | -g], and curly brackets are placed around required arguments, {-f | -g}. Plain arguments are required, but are not decorated with brackets.

    if ($o->at_group)                                                           # group
     {my $c = docBookChoiceAttribute($o);

      if (!$c or $c eq opt)                                                     # Choice
       {$o->set(importance=>optional);
        $o->change_groupchoice;
       }
      elsif ($c eq plain)
       {$o->change_groupseq;
       }
      elsif ($c eq req)
       {$o->change_groupchoice;
       }
      else
       {confess "Unprogrammed group choice=$c\n";
       }

      if (my $r = $o->attr_rep)                                                 # Repetition
       {if ($r eq repeat)
         {$o->putLastAsTree(ellipsis);
          $o->wrapContentWith_groupseq unless $o->at_groupseq;                  # PS2-658 - wrap the repeated sequence unless it is already wrapped
         }
       }

      $o->deleteAttrs_choice_rep;                                               # Attributes processed
     }
   });

  for(1..9)                                                                     # Flatten the syntax diagram where possible
   {my $w = 0;

    $x->by(sub                                                                  # Consolidate groups with only one item in them
     {my ($k, $g) = @_;
      if ($k->isOnlyChild(qr(\A(kwd|var)\Z), qr(\A(groupseq|groupchoice)\Z)))
       {$g->unwrap; ++$w;
        if (ditaOptional($g))                                                   # If group is optional then the child must be optional
         {$k->set(importance=>optional);
         }
       }
     });

    $x->by(sub                                                                  # group sequence single child of group choice importance=optional goes to group sequence importance=optional
     {my ($s, $c) = @_;
      if ($s->isOnlyChild_groupseq_groupchoice)
       {if (ditaOptional($c))
         {if (!ditaRepeated($s))
           {$c->unwrap; ++$w;
            $s->set(importance=>optional);                                      # The wrapping choice forces this seq to be optional
           }
         }
       }
     });

    $x->by(sub                                                                  # A non optional group sequence under a non optional group sequence can be unwrapped
     {my ($s, $t) = @_;
      if ($s->at_groupseq_groupseq)
       {if   (ditaNonOptionalGroupSeq($s))
         {if (ditaNonOptionalGroupSeq($t))
           {if (!ditaRepeated($s))
             {$s->unwrap; ++$w;
             }
           }
         }
       }
     });

    $x->by(sub                                                                  # An only child group choice under a non optional group sequence: unwrap the group sequence as unnecessary
     {my ($c, $s) = @_;
      if ($c->isOnlyChild_groupchoice_groupseq)
       {if (ditaNonOptionalGroupSeq($s))
         {$s->unwrap; ++$w;
         }
       }
     });

    $x->by(sub                                                                  # An only child group choice under a non optional group sequence: unwrap the group sequence as unnecessary
     {my ($c, $d) = @_;
      if ($c->isOnlyChild_groupchoice_groupchoice)
       {if ($c->attrX_importance eq $d->attrX_importance)
         {$c->unwrap; ++$w;
         }
        elsif (ditaOptionalGroupChoice($c))
         {$d->unwrap; ++$w;
         }
        elsif (ditaOptionalGroupChoice($d))
         {$c->unwrap; ++$w;
         }
       }
     });

    $x->by(sub                                                                  # group sequence importance=optional under group choice importance=optional: remove importance from group sequence as it is redundant and leads to over bracketing
     {my ($s, $c) = @_;
      if ($c and $c->at_groupchoice)
       {if   (ditaOptional($c) and ditaOptional($s))
         {$s->deleteAttrs_importance;
         }
       }
     });

    $x->by(sub                                                                  # importance=required always produces the same results as importance=>undef in the html5 transformation so we assume that it can be safely removed
     {my ($o) = @_;
      if ($o->attrX_importance eq required)
       {$o->deleteAttr_importance;
       }
     });

    last unless $w;
   }

  $x
 } # ditaSyntaxDiagramFromDocBookCmdSynopsis

sub ditaSyntaxDiagramToBasicRepresentation($)                                   #U Convert Dita syntax diagrams into Micaela's Basic Version.
 {my ($x) = @_;                                                                 # Parse tree

  $x->down(sub
   {my ($o, $p) = @_;

    my $i = $o->attrX_importance;                                               # Add brackets
    if ($o->at_groupchoice)                                                     # Group choice has brackets and separators
     {if ($i eq q(optional))
       {$o->putPrevAsText(q( [));
        $o->putNextAsText(q(] ));
       }
      else
       {$o->putPrevAsText(q( {));
        $o->putNextAsText(q(} ));
       }
      my ($c, @c) = @$o;
      $_->putPrevAsText(q( | )) for @c;
     }
    elsif ($i eq q(optional))                                                   # Non group choice just has square brackets
     {$o->putPrevAsText(q( [));
      $o->putNextAsText(q(] ));
     }

    if ($o->at_syntaxdiagram)                                                   # Map tags
     {$o->change_p;
      $o->outputclass = q(syntaxdiagram);
     }
    elsif ($o->at_kwd and $o->outputclassX eq q(command))
     {$o->change_cmdname;
     }
    elsif ($o->at_kwd)
     {$o->change_codeph;
     }
    elsif ($o->at_var)
     {$o->change_userinput;
     }
    elsif ($o->at_sep)
     {$o->change_codeph;
     }
    elsif ($o->at_groupseq or $o->at_groupchoice)
     {$o->unwrap;
     }

    $o->deleteAttrs_outputclass_importance;

   });

 } # ditaSyntaxDiagramToBasicRepresentation

sub ditaUnderPNotConbody                                                        #P Return a hash of items that L<Dita> permits under B<p> but not directly under B<conbody>
{{
  "abbreviated-form" => 1,
  "apiname"          => 1,
  "b"                => 1,
  "boolean"          => 1,
  "CDATA"            => 1,
  "cite"             => 1,
  "cmdname"          => 1,
  "codeph"           => 1,
  "equation-inline"  => 1,
  "filepath"         => 1,
  "fn"               => 1,
  "i"                => 1,
  "indexterm"        => 1,
  "indextermref"     => 1,
  "keyword"          => 1,
  "line-through"     => 1,
  "markupname"       => 1,
  "menucascade"      => 1,
  "msgnum"           => 1,
  "msgph"            => 1,
  "numcharref"       => 1,
  "option"           => 1,
  "overline"         => 1,
  "parameterentity"  => 1,
  "parmname"         => 1,
  "ph"               => 1,
  "q"                => 1,
  "state"            => 1,
  "sub"              => 1,
  "sup"              => 1,
  "synph"            => 1,
  "systemoutput"     => 1,
  "term"             => 1,
  "text"             => 1,
  "textentity"       => 1,
  "tm"               => 1,
  "tt"               => 1,
  "u"                => 1,
  "uicontrol"        => 1,
  "userinput"        => 1,
  "varname"          => 1,
  "wintitle"         => 1,
  "xmlatt"           => 1,
  "xmlelement"       => 1,
  "xmlnsname"        => 1,
  "xmlpi"            => 1,
  "xref"             => 1,
}} # ditaUnderPNotConbody

sub ditaWrapWithPUnderConbody($)                                                #U Wrap items immediately under L<DITA> B<conbody> with B<p> or merge with any previous B<p> if the item in question does not fit under B<conbody> but does fit under B<p>. Return the current node if it is B<conbody> else return B<undef>.
 {my ($conbody) = @_;                                                           # Section
  $conbody->at_conbody or confess "Not a conbody, its a ". -t $conbody;         # Check we are on a conbody
  my $fit = ditaUnderPNotConbody;                                               # Tags that need to be wrapped with B<p> if they occur directly under B<conbody>.
  my $p;                                                                        # Last p encountered
  my @c = @$conbody;                                                            # Each direct child of conbody
  for my $c(@c)                                                                 # Each direct child of conbody
   {if ($$fit{-t $c})                                                           # Better as a p
     {if ($p)                                                                   # Prior tag is p
       {$p->putLastCut($c);                                                     # Merge
       }
      else                                                                      # Start new p
       {$p = $c->wrapWith_p;                                                    # Wrap with p and make new p
       }
     }
    elsif ($c->at_p)                                                            # Tag is p - so make it the last p
     {$p = $c;                                                                  # P tag becomes last p
     }
    else                                                                        # Not on a p and not better wrapped in p
     {$p = undef;                                                               # New p required
     }
   }
 }

#D1 PCD                                                                         # Please Change Dita Language Features

our %savedNodes;                                                                #E Hash of saved nodes

sub putNodeAs($$@)                                                              #CU Return the specified B<$node> after saving it under the specified B<$name> if we are in the optional B<@context>.
 {my ($node, $name, @context) = @_;                                             # Node, save name, optional context
  return $node if @context and !$node->at(@context);                            # Not in specified context
  $savedNodes{$name} = $node;                                                   # Save a reference to the node under the specified name and return the current node
 }

BEGIN{*put=*putNodeAs}

sub getNodeAs($$@)                                                              #CU Return the specified B<$node> unless it is possible to return the node saved with L<putNodeAs> under the specified B<name> and we are optionally in the specified B<@context>.
 {my ($node, $name, @context) = @_;                                             # Default node, name of saved node, optional context
  return $node if @context and !$node->at(@context);                            # Not in specified context
  $savedNodes{$name} // $node;                                                  # Return saved node defaulting to current node
 }

BEGIN{*get=*getNodeAs}

sub reportNodeAttributes($$@)                                                   #CU Print the attributes of the specified B<$node> identified by the specified B<$label> if we are in the optional B<@context>.
 {my ($node, $label, @context) = @_;                                            # Node node, label, optional context
  return '' if @context and !$node->at(@context);                               # Not in specified context
  join ' ', $label, $node->printAttributes;                                     # Print label and attributes
 }

BEGIN{*rna=*reportNodeAttributes}

sub reportNodeContext($$@)                                                      #CU Print the context of the specified B<$node> identified by the specified B<$label> if we are in the optional B<@context>.
 {my ($node, $label, @context) = @_;                                            # Node node, label, optional context
  return '' if @context and !$node->at(@context);                               # Not in specified context
  join ' ', $label, $node->context;                                             # Print label and context
 }

BEGIN{*rnc=*reportNodeContext}

sub reportNode($$@)                                                             #CU Print the parse tree starting at the specified B<$node> identified by the specified B<$label> if we are in the optional B<@context>.
 {my ($node, $label, @context) = @_;                                            # Node node, label, optional context
  return '' if @context and !$node->at(@context);                               # Not in specified context
  join '', $label, "\n", $node->prettyString;                                   # Print label and parse tree
 }

BEGIN{*rn=*reportNode}

#D1 Debug                                                                       # Debugging methods

sub sss($@)                                                                     #U Put, after the current B<$node>, the specified B<@text> and return the current $node.
 {my ($node, @text) = @_;                                                       # Node, text
  my $text = join ' ', @text;                                                   # Text to insert
  $node->putNextAsText($text);                                                  # Insert text after current node
  $node
 }
#a
#b <b><c><d/></c></b>
#c at d c
#c sss we reached here
#d Say something special so we can see where we were.

sub printAttributes($)                                                          #U Print the attributes of a node.
 {my ($node) = @_;                                                              # Node whose attributes are to be printed.
  return q() unless keys %{$node->attributes};                                  # No attributes

  my %a = %{$node->attributes};                                                 # Attributes
  my $s = '';
  for(sort keys %a)                                                             # Each attribute
   {next unless defined(my $v = $a{$_});
    $s .= $_.'="'.$v.'" ';                                                      # Attributes enclosed in "" in alphabetical order
   }
  chop($s);
  length($s) ? ' '.$s : '';
 }

sub printAttributesHtml($)                                                      #U Print the attributes of a node as html.
 {my ($node) = @_;                                                              # Node whose attributes are to be printed.
  return q() unless keys %{$node->attributes};                                  # No attributes

  my %a = %{$node->attributes};                                                 # Attributes
  my $s = '';
  for(sort keys %a)                                                             # Each attribute
   {next unless defined(my $v = $a{$_});                                        # Attributes enclosed in "" in alphabetical order
    $s .= qq(<span class="xmlAttr">$_</span>).
          qq(<span class="xmlEquals">=</span>).
          qq(<span class="xmlValue">"$v"</span> );
   }
  chop($s);
  length($s) ? ' '.$s : '';
 }

sub printStack($@)                                                              #UC Print the attributes of a node and each of its parents
 {my ($node, @context) = @_;                                                    # Node whose attributes are to be printed, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context

  my @s;                                                                        # Resulting text
  my @p = reverse $node->ancestry;                                              # Each ancestor
  for my $i(keys @p)                                                            # Print and indent
   {my $p = $p[$i];
    push @s, q(  )x$i.$p->printNode;                                            # Print node with indentation
   }
  join "\n", @s, '';                                                            # Return ancestry
 }

sub printNode($@)                                                               #UC Print the tag and attributes of a node.
 {my ($node, @context) = @_;                                                    # Node whose attributes are to be printed, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  my %a = %{$node->attributes};                                                 # Attributes
  my $t = $node->tag;                                                           # Tag
  my $s = '';                                                                   # Attributes string
  for(sort keys %a)                                                             # Each attribute
   {next unless defined(my $v = $a{$_});                                        # Each defined attribute
    $s .= qq( ${_}="$v");                                                       # Attributes enclosed in "" in alphabetical order
   }
  qq($t$s)                                                                      # Result
 }

sub printNodeAsSingleton($@)                                                    #UC Print the tag and attributes of a node as if it were a single node regardless of any child nodes it might have
 {my ($node, @context) = @_;                                                    # Node whose attributes are to be printed, optional context.
  return undef if @context and !$node->at(@context);                            # Not in specified context
  q(<).printNode($node).q(/>);
 }

sub printAttributesReplacingIdsWithLabels($)                                    #UP Print the attributes of a node replacing the id with the labels.
 {my ($node) = @_;                                                              # Node whose attributes are to be printed.
  my %a = %{$node->attributes};                                                 # Clone attributes
  my %l = %{$node->labels};                                                     # Clone labels
  delete $a{id};                                                                # Delete id
  $a{id} = join ', ', sort keys %l if keys %l;                                  # Replace id with labels in cloned attributes
  defined($a{$_}) ? undef : delete $a{$_} for keys %a;                          # Remove undefined attributes
  return '' unless keys %a;                                                     # No attributes
  my $s = ' '; $s .= $_.'="'.$a{$_}.'" ' for sort keys %a; chop($s);            # Attributes enclosed in "" in alphabetical order
  $s
 }

sub printAttributesExtendingIdsWithLabels($)                                    #UP Print the attributes of a node extending the id with the labels.
 {my ($node) = @_;                                                              # Node whose attributes are to be printed.
  my %a = %{$node->attributes};                                                 # Clone attributes
  my %l = %{$node->labels};                                                     # Clone labels
  my $i = $a{id} ? $a{id}.q(, ) : q();                                          # Format id
  $a{id} = join '', $i, join ', ', sort keys %l if keys %l;                     # Extend id with labels in cloned attributes
  defined($a{$_}) ? undef : delete $a{$_} for keys %a;                          # Remove undefined attributes
  return '' unless keys %a;                                                     # No attributes
  my $s = ' '; $s .= $_.'="'.$a{$_}.'" ' for sort keys %a; chop($s);            # Attributes enclosed in "" in alphabetical order
  $s
 }

sub checkParentage($)                                                           #UP Check the parent pointers are correct in a L<parse|/parse> tree.
 {my ($x) = @_;                                                                 # Parse tree.
  $x->by(sub
   {my ($o) = @_;
   for($o->contents)
     {my $p = $_->parent;
      $p == $o or confess "No parent: ". $_->tag;
      $p and $p == $o or confess "Wrong parent: ".$o->tag. ", ". $_->tag;
     }
   });
 }

sub checkParser($)                                                              #UP Check that every node has a L<parse|/parse>r.
 {my ($x) = @_;                                                                 # Parse tree.
  $x->by(sub
   {$_->parser or confess "No parser for ". $_->tag;
    $_->parser == $x or confess "Wrong parser for ". $_->tag;
   })
 }

sub goFish($@)                                                                  #U A debug version of L<go|/go> that returns additional information explaining any failure to reach the node identified by the L<path|/path>.\mReturns ([B<reachable tag>...], B<failing tag>, [B<possible tag>...]) where:\m=over\m=item B<reachable tag>\mthe path elements successfully traversed;\m=item B<failing tag>\mthe failing element;\m=item B<possible tag>\mthe possibilities at the point where the path failed if it failed else B<undef>.\m=back\mParameters:
 {my ($node, @path) = @_;                                                       # Node, search specification.
  my $p = $node;                                                                # Current node
  my @p;                                                                        # Elements of the path successfully processed
  while(@path)                                                                  # Position specification
   {my $i = shift @path;                                                        # Index name
    return ([@p], $i, [sort keys %{$p->indexes}]) unless $p;                    # There is no node of the named type under this node
    reindexNode($p);                                                            # Create index for this node
    my $q = $p->indexes->{$i};                                                  # Index
    return ([@p], $i, [sort keys %{$p->indexes}]) unless defined $q;            # Complain if no such index
    push @p, $i;
    if (@path)                                                                  # Position within index
     {if ($path[0] =~ /\A([-+]?\d+)\Z/)                                         # Numeric position in index from start
       {my $n = shift @path;                                                    # Next path item
        my $N = scalar(@$q);                                                    # Dimension of index
        return ([@p], $n, [0..$N]) unless defined($p = $q->[$n]);               # Complain if no such index
        push @p, $n;                                                            # Save successfully processed index
       }
      elsif (@path == 1 and $path[0] =~ /\A\*\Z/)                               # Final index wanted
       {return [@p];
       }
      else {$p = $q->[0]}                                                       # Step into first sub node by default
     }
    else {$p = $q->[0]}                                                         # Step into first sub node by default on last step
   }
  [@p]                                                                          # Success!
 }

sub nn($)                                                                       #P Replace new lines in a string with N to make testing easier.
 {my ($s) = @_;                                                                 # String.
  $s =~ s/\n/N/gsr
 }

#D1 Validation                                                                  # Validate Xml

sub checkAllPaths($)                                                            #S Create a representation of all the paths permitted in a block of L<xml>. The syntax of each line is a word representing an L<xml> tag followed by one of: tag B<1 * + ? -> with B<1> being the default and meaning that exactly one instance of this tag is required under its parent, B<*> meaning that zero or more such tags are permitted under its parent, B<+> meaning that one or more such tags are required under its parent, B<?> meaning that no more than one such tag is permitted under its parent, B<-> meaning that this is a leaf tag that can be missing or present just once under its parent. The remaining words on the line being treated as a comment describing the purpose of the tag.  Tags that are not marked with B<-> are required to have either text under them if they are leaf tags. Tags marked with B<-> are boolean leafs and may not have any text or child tags under them. The children of a tag are listed on following lines with exactly two more spaces of indentation.  This enables the validation of a block of non recursive L<xml> without having to resort to writing a complex L<DTD>.
 {my ($valid) = @_;                                                             # Path descriptions
  my %valid;                                                                    # Perl representation of validating string
  my @stack;                                                                    # Tag stack
  my @lines = split m/\n/, $valid;                                              # Split into lines

  for my $i(keys @lines)                                                        # Each line
   {my $line    = $lines[$i] =~ s(\s*#.*\Z) ()r;                                # Remove trailing comments
    next unless $line =~ m(\S);                                                 # Ignore blank lines

    my sub error(@)                                                             # Write an error message
     {confess join ' ', @_, 'on line:', $i + 1, "\n";
     };

    my $tag     = $line =~ s(\A\s*) ()r;                                        # Check indentation carefully as it shows the desired structure
    my $indent  = length($line) - length($tag);
    $indent % 2 and error "Indentation is not even";
    my $indent2 = $indent / 2;
    $i == 0 and $indent != 0 and error "Indent is not zero";
    @stack+1 < $indent2 and error "Too much indentation";

    while(@stack)                                                               # Reduce the stack to the current level
     {if (@stack > $indent2)
       {pop @stack;
        next;
       }
      last;
     }

    my ($tagName, @words) = split m/\s+/, $tag;                                 # Save tag on the tag stack
    push @stack, $tagName;

    my $count = sub                                                             # The count indicator optionally follows the tag
     {return 1 unless @words;                                                   # The default is just one and it is required
      my $c = shift @words;
      return $c if $c  =~ m(\A[-1+*?]\Z)i;                                      # Valid operators
      1
     }->();

    if (@stack > 1)                                                             # Element description
     {$valid{join ' ', @stack[1..$#stack]} =
        [$count, my $comment = join ' ', @words];
     }
   }

  my ($root) = @stack;                                                          # The root tag

  for   my $a(sort keys %valid)                                                 # Create get methods
   {for my $b(sort keys %valid)
     {my $c = $valid{$b}[0];                                                    # Count field
      my @b = split m/\s+/, $b;
      my $m = pop @b;
      if ($a eq join " ", @b or !@b)                                            # Has children
       {my @m = ($root, @b);
        if (!isSubInPackage((join '::', @m), $m))
         {my $d = $c eq q(*) || $c eq q(+) ? q([]) : q/q()/;                    # Default return value

          my $s = join ' ', qq(sub), (join '::', @m, $m), qq({\$_[0]{$m} // $d});
          eval $s;
          if ($@)
           {confess join '', "Unable to create method: ",$a, q(::).qq($m\n$@\n);
           }
         }
        if (@b and $valid{$a}[0] eq q(-))                                       # Check that tags marked as boolean leaves do not have any children
         {confess <<END;
Path: '$a' has been marked with '-' making it a boolean leaf yet it has
a child: '$b'  '$a'
END
         }
       }
     }
   }

  \%valid
 } # checkAllPaths

sub xmlToPerl($$)                                                               #S Parse some L<xml>, validate using a description recognized by  L<checkAllPaths> and return the corresponding L<Perl> structure.  Valid fields in the returned structure can be referenced via methods with the same names as their corresponding L<xml> tags. Tags marked with B<*> or B<+> in the description processed by L<checkAllPaths> result in arrays of children, whilst tags marked with B<1> or B<?> can be referenced directly.
 {my ($xml, $valid) = @_;                                                       # Xml represented as a string, Xml validating represented as a string

  my $xmlTree   = Data::Edit::Xml::new($xml);                                   # Xml parse tree
  my $validator = checkAllPaths($valid);                                        # Create a validating string to check all the paths in the xml description of the system
  my $perl;                                                                     # Perl representation of the Xml

  $xmlTree->by(sub                                                              # Traverse xml to build Perl data structure
   {my ($o) = @_;
    return if $o == $xmlTree;

    my @path = reverse @_;                                                      # Node path from root downwards
    shift @path;                                                                # Remove root tag as it is boiler plate
    pop @path if $o->isText;                                                    # Remove CDATA
    my $p = join ' ', map {-t $_} @path;                                        # Tag path

    if (my $valid = $$validator{$p}[0])                                         # Details of this path
     {my @keys;                                                                 # Path as text
      for my $path(@path)                                                       # Describe path
       {my $t = -t $path;
        if ($path != $xmlTree)
         {my $i = $path->index;
          push @keys, qq({$t}[$i]);
         }
       }
      if ($o->isText)                                                           # Text field
       {my $s = join "", q($$perl), @keys, q( = ), dump trim $o->text;          # Load Perl data structure
        eval $s;
        $@ and confess "$@\n";
       }
      else                                                                      # Check that a tag has content unless it is a boolean leaf
       {if ($o->isEmpty)                                                        # Tag has no content
         {if ($valid eq q(-))
           {my $s = join "", q($$perl), @keys, q( = 1);                            # Show leaf tag present
            eval $s;
            $@ and confess "$@\n";
           }
          else
           {confess join " ", "Tag has no content on path(zero based):",
              $o->pathString, "\n";
           }
         }
        else                                                                    # Tag has  content
         {if ($valid eq q(-))
           {confess join " ", "Leaf boolean has content on path(zero based):",
            $o->pathString, "\n";
           }
         }
       }

      if (1)                                                                    # Bless parents so we can write $a->b rather than $a->{b} and get an error if we choose an invalid field.
       {pop @keys; pop @path;
        my @p = map{$_->tag} @path;
        my $k = join '',                  @keys;
        my $p = join '::', $xmlTree->tag, @p;
        my $s = @keys ? qq(bless \$\$perl$k, "$p") : qq(bless \$perl$k, "$p");
        eval $s;
        $@ and confess "$@\n";
       }
     }
    elsif ($p)
     {confess join ' ',
       "No description in validator for path:", $o->pathString, "\n";
     }
    else
     {my $j = -p $o;
      confess "Junk: $j\n";
     }
   });

  for my $v(sort keys %$validator)                                              # Validate presence of required elements by checking the application of each rule which requires at least one sub element
   {my ($count) = $$validator{$v}->@*;                                          # Count specification from this validation specification
    next unless $count =~ m(\A[1+]\Z)i;                                         # We are only interested in required elements
    my @path    = split m/\s+/, $v;                                             # Path to this rule
    my $parent  = join ' ', reverse $xmlTree->tag, @path[0..@path-2];           # Path to parent of this rule

    $xmlTree->by(sub                                                            # Traverse xml to build Perl data structure
     {my ($o) = @_;
      if ($parent eq $o->context)                                               # Point in the xml parse tree that matches the parent rule
       {my $child = $path[-1];
        my @c = $o->c($child);
        if (@c < 1)
         {say STDERR -p $o;
          confess join " ", "$child required under", $o->context, "\n";
         }
       }
     });
   }

  my $sublimate; $sublimate = sub                                               # Replace arrays with direct references where possible
   {my ($data, @path) = @_;                                                     # Data point, path to data point

    for   my $k(sort keys %$data)
     {push @path, $k;
      for my $d($$data{$k}->@*)                                                 # Sublimate lower trees
       {$sublimate->($d, @path) if ref $d;
        if (my $valid = $$validator{join ' ',  @path})
         {if ($$valid[0] =~ m(\A[-1?]\Z)i and $$data{$k}->@* <= 1)
           {$$data{$k} = $$data{$k}[0];
           }
         }
       }
      pop @path;
     }
   };

  &$sublimate($perl);                                                           # Replace arrays with direct references where possible

  $perl
 } # xmlToPerl

#D1 Documentation                                                               # Update documentation describing this module

sub extractDocumentationFlags($$)                                               #P Generate documentation for a method with a user flag.
 {my ($flags, $method) = @_;                                                    # Flags, method name.
  my $b = "${method}NonBlank";                                                  # Not blank method name - used for a small number of navigation methods
  my $x = "${method}NonBlankX";                                                 # Not blank, die on B<undef> method name
  my $m = $method;                                                              # Second action method
     $m =~ s/\Afirst/next/gs;
     $m =~ s/\Alast/prev/gs;
  my @doc; my @code;

  if ($flags =~ m/C/is)                                                         # Context flag for a method that returns a single node or B<undef> if in the wrong context
   {push @doc, <<'END' if $flags =~ m/C/s;
Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.
END
    push @doc, <<'END' if $flags =~ m/c/s;
Use the required B<$tag> parameter to specify the expected tag on the specified
B<$node> using a single match expression as understood by method L<at|/at>. Use
the optional B<@context> parameter to test the context as understood by method
L<at|/at> of the parent node of the specified B<$node>. If either test fails
this method returns B<undef> immediately.
END
   }
  if ($flags =~ m/K/s)                                                          # Context flag for a method that returns an array of nodes or the empty array if in the wrong context
   {push @doc, <<'END';
Use the B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>.  If a context is supplied and
B<$node> is not in this context then this method returns an empty list B<()>
immediately.
END
   }

  if ($flags =~ m/B/s)                                                          # Skip blank text
   {push @doc, <<END;
Use B<$b> to skip a (rare) initial blank text CDATA. Use B<$x> to die rather
then receive a returned B<undef> or false result.
END
    push @code, <<END;
sub $b
 {my \$r = &$method(\$_[0]);
  return undef unless \$r;
  if (\$r->isBlankText)
   {shift \@_;
    return &$m(\$r, \@_)
   }
  else
   {return &$m(\@_);
   }
 }

sub $x
 {my \$r = &$b(\@_);
  die '$method' unless defined(\$r);
  \$r
 }
END
   }

  [join("\n", @doc), join("\n", @code), $flags =~ m/B/ ? [$b, $x] : ()]
 }

sub writeContentToGitHub($$)                                                    #P Upload the contents of a string to a file on github
 {my ($file, $content) = @_;                                                    # File on github, file contents
  my $g = <<'END';                                                              # I am the only person who does this so there is no point in adding GitHub::Crud to the prerequisites list
use GitHub::Crud;
my $target = writeFile(undef, $content);
GitHub::Crud::writeFileFromFileUsingSavedToken
 (q(philiprbrenan), q(philiprbrenan.github.io), $file, $target);
GitHub::Crud::writeFileFromFileUsingSavedToken
 (q(ryffine), q(ryffine.github.io), $file, $target);
END

  my $r = eval $g;
  confess "$@" if $@;
  $r
 }

sub writeFileToGitHub($$)                                                       #P Upload the contents of the specified local file to a file on github
 {my ($file, $localFile) = @_;                                                  # File name on github, local file name
  my $g = <<'END';                                                              # I am the only person who does this so there is no point in adding GitHub::Crud to the prerequisites list
use GitHub::Crud;
GitHub::Crud::writeFileFromFileUsingSavedToken
 (q(philiprbrenan), q(philiprbrenan.github.io), $file, $localFile);
GitHub::Crud::writeFileFromFileUsingSavedToken
 (q(ryffine), q(ryffine.github.io), $file, $localFile);
END

  my $r = eval $g;
  confess "$@" if $@;
  $r
 }

sub processModuleDescription($)                                                 #P Process module description to write a description of editing Xml to GitHub
 {my ($d) = @_;                                                                 # Module description

  my $b  = q(<b>);
  my $bb = q(</b>);
  my $th = q(<th align="left">);

  my @h  = (<<END);                                                             # Editor commands description
<!DOCTYPE html>
<html>
<head>
<meta charset="utf-8"/>
<title></title>
<style>
.l0 {
  background-color: #f0fff0
}
.l1 {
  background-color: #fff0ff
}
</style>
</head>
<body>

<div style="width: 80em">

<h1>Methods in Data::Edit::Xml that can be used to Edit Dita</h1>
END

  my $methods = sub                                                             # Format methods either the immediately useful ones or all of them
   {my ($isUseful) = @_;                                                        # Immediately useful methods

    return unless my $M = $d->{methods};                                        # Method descriptions

    push my @h, (<<END);                                                        # Table header
<table border="0" cellpadding="20">
<tr>${th}Line${th}Method long name${th}aka${th}Parameter${th}Description
END

    my $line = 0;                                                               # Method number

    for my $method(sort keys %$M)
     {my $m = $$M{$method};
      next if $isUseful and !$m->{isUseful};                                    # Only immediately useful methods requested
      next unless $m->{unitary};                                                # Only unitary methods can be used in the editor or the pcd language

      my $name = $m->{name};
      my $synonym = sub
       {return q() unless my $S = $m->{synonyms};
        my ($s) = sort keys %$S;
        return $s;
       }->();

      my $c = $m->{comment};                                                    # Format the comment so that it breaks
         $c = formatString($c, 80);

      my $p = $m->{parameters};
      shift @$p;                                                                # We are unitary so this is feasible
      ++$line;

      my $class = q(class="l).($line % 2).q(");                                 # Row color

      push @h, qq(<tr $class><td>$line<td>$name<td><b>$synonym</b><td><td>$c\n);
      if (@$p)
       {for my $p(@$p)
         {my ($parm, $desc) = @$p;
          push @h, qq(<tr $class><td><td><td><td>$parm<td>$desc\n);
         }
       }
     }

    push @h, (<<END);
</table>
END
    @h
   };

  push @h, <<END, &$methods(1), <<END2, &$methods(0);
<h2>Immediately useful methods</h2>
<p>The following methods are frequently use to make simple changes:
END
<h2>All methods</h2>
<p>Here is a listing of all methods currently available:
END2

  my $dts = dateTimeStamp;
  push @h, <<END;

<h1>Editing Dita using Geany or PCD</h1>

<p>L[dex] is a Perl extension module that can help you edit L[Dita] either
interactively using L[Geany] or in batch using L[PCD] files.

<h2>Configuring Geany to Edit Dita and PCD</h2>

<p>Please obtain a version of L[Ubuntu] and follow the instructions at
L[pcdInstall].

<h2>Editing Dita interactively with Geany</h2>

<p>You can use many of the methods found in L[dex] one at a time directly from
the L[Geany] editor when it has been configured for L[Dita] editing. The
methods executed are recorded in log visible in another L[Geany] tab so that
they can be edited directly into a program that is being developed to use
L[dex] directly from L[Perl] or from L[pcd].

<p>The table below lists the methods that can be used in L[Geany].  Simply type
the method name and any parameters up against a closing $b&gt;$bb or
$b&gt;$bb as shown on line $b 13 $bb like this:

<p><img src="/images/mlp_start.png"/>

<p>and press $b shift+Enter $bb to merge the two lists:

<p><img src="/images/mlp_finish.png"/>


<h2>Editing Dita in batch using PCD</h2>

<p>L[PCD] files contain a list of L[dex] commands, one per line, that are applied
to each node of each of the parse trees produced by parsing all of the L[DITA]
files found in the same directory tree as the L[PCD] file.

<p><a href="https://github.com/philiprbrenan/pleaseChangeDita">This
repository</a> contains two L[DITA] l[concept]s and one L[PCD] file. The L[PCD]
file unwraps a <b>b</b> tag found under another <b>b</b> tag.

<p>There are four types of line in a L[PCD] file:

<p><pre>
1
2   # Comment
3   Unwrap b under b and delete empty paragraphs
4        unwrap b b
5        cutIfEmpty p
</pre></p>

<table cellpadding="20" border="0">
<tr><th>Line<th>Name<th>Description

<tr><td>1<td>Blank lines      <td>Blank lines are ignored

<tr><td>2<td>Comment lines    <td>If the first non space character is <b>#</b>
then this is a comment line and it is ignored.

<tr><td>3<td>Description lines<td>If the first character is not a space it is a
description line whose purpose is to communicate with other humans what the
following commands aim to do.

<tr><td>4<td>Command lines    <td>If the first character is a space then the
rest of the line is command line: a method name chosen from L[dex] followed by
any parameters separated from each other by one or more spaces.

</table>

<p>Each L[PCD] file can contain zero or more description lines. Each description
line can be followed by zero or more command lines called a "block of command
lines" or "block" for short. L[PCD] executes each block against each node in
each of the parse trees constructed by parsing all of the L[DITA] files found
in the same folder as the L[PCD] files.  Execution of the block continues line
by line until the end of the block or the L[dex] method fails. If the block
completes successfully, the description line for that block is printed out to
show that it "fired".

<p>L[Geany] on L[Ubuntu] has been configured as an L[IDE] to make editing both
L[PCD] and test L[DITA] files a pleasant experience with all the usual
facilities provided by such an L[IDE] such as help choosing command names,
testing the syntax of L[PCD] files once they are written, then executing
them against L[DITA] files and showing the execution results.</p>

<p>You can also place l[PCD] files into the same L[S3] Bucket or L[github] repo
as your L[dita] files and then process them in bulk with L[ssxr].
<p>Updated: $dts
</div>
</body>
</html>
END

  my $h2 = join "\n", @h;                                                       # Html
  my $h1 = expandWellKnownUrlsInHtmlFormat  ($h2);
  my $h  = expandWellKnownUrlsInHtmlFromPerl($h1);
     $h  =~ s(([LB]) (<) ([^>]*) (>)) ($1&lt;$3&gt;)xgs;                        # Fix angle brackets

  if (1)                                                                        # Simplified documentation
   {my @l = readFile($INC{"Data/Edit/Xml.pm"});

    my $method; my @also; my @before; my @code;
    for my $l(@l)
     {if ($l =~ m(\Asub\s*(\w+)))                                               # Method for which there is simplified documentation
       {$method = $1;
        @before = (); @code = ();
       }
      if ($l =~ m(\A#a\s*(.*?)\s*\Z))                                           #a see also
       {push @also, split /\s+/, $1;
       }
      if ($l =~ m(\A#b\s*(.*?)\s*\Z))                                           #b before file with the root unwrapped
       {push @before, $1;
       }
      if ($l =~ m(\A#c\s*(.*?)\s*\Z))                                           #c Example pcd
       {push @code, $1;
       }
      if ($l =~ m(\A#d\s*(.*?)\s*\Z))                                           #d Simplified documentation
       {$d->{methods}{$method}{example} =
          genHash(q(Data::Edit::Xml::Example),                                  # Description of a pcd method
            method => $method,                                                  # Method
            also   => [@also],                                                  # See also
            before => [@before],                                                # Before file content minus root tag which will be added as <a>
            code   => [@code],                                                  # Pcd code
            doc    => $1,                                                       # One line summary
           );
       }
     }
   }
####TEST####
#  writeContentToGitHub(q(data_edit_xml_edit_commands.html), $h);
#  writeFileToGitHub(q(images/mlp_start.png),  q(/home/phil/r/oxygenWorkshop/doc/selfServiceXref/images/4/s011256.png));
#  writeFileToGitHub(q(images/mlp_finish.png), q(/home/phil/r/oxygenWorkshop/doc/selfServiceXref/images/4/s011321.png));
 } # processModuleDescription

#D1 Compression                                                                 # Read and write files of compressed xml.  These methods provide a compact, efficient way to store and retrieve parse trees to/from files.

sub writeCompressedFile($$)                                                     #U Write the parse tree starting at B<$node> as compressed L<XML> to the specified B<$file>. Use L<readCompressedFile|/readCompressedFile>  to read the B<$file>.
 {my ($node, $file) = @_;                                                       # Parse tree node, file to write to.
  makePath($file);
  open my $F, "| gzip>$file" or                                                 # Compress via gzip
    confess "Cannot open file for write because:\n$file\n$!\n";
  binmode($F, ":utf8");                                                         # Input to gzip encoded as utf8
  print  {$F} -s $node;
  close  ($F);
  -e $file or confess "Failed to write to file:\n$file\n";
  $file
 }

sub readCompressedFile($)                                                       #S Read the specified B<$file> containing compressed L<XML> and return the root node.  Use L<writeCompressedFile|/writeCompressedFile> to write the B<$file>.
 {my ($file) = @_;                                                              # File to read.
  defined($file) or
    confess "Cannot read undefined file\n";
  $file =~ m(\n) and
    confess "File name contains a new line:\n=$file=\n";
  -e $file or
    confess "Cannot read file because it does not exist, file:\n$file\n";
  open(my $F, "gunzip < $file|") or                                             # Unzip input file
    confess "Cannot open file for input, file:\n$file\n$!\n$?\n";
  local $/ = undef;
  my $string = <$F>;
  new($string)                                                                  # Reparse resulting string to recover parse tree
 }

#D1 Autoload                                                                    # Allow methods with constant parameters to be called as B<method_p1_p2>...(variable parameters) whenever it is easier to type underscores than (qw()).

sub DESTROY {}                                                                  # TO avoid seeing it in AUTOLOAD

our $AUTOLOAD;                                                                  # The method to be autoloaded appears here

sub  AUTOLOAD                                                                   # Allow methods with constant parameters to be called as B<method_p1_p2>...(variable parameters) whenever it is easier to type underscores than (qw()).
 {#return if $AUTOLOAD =~ m(Destroy)is;                                          # Perl internal
  my $q = shift;                                                                # Object package
  if ($AUTOLOAD =~ m(__)s)                                                      # Chain of calls separated by q(__)
   {no strict q(refs);                                                          # So we can call a sub by name
    my @calls = split /__/, $AUTOLOAD;                                          # Calls in chain
    for my $call(@calls)                                                        # Each call in chain
     {my ($p, @p) = split /_/, $call;                                           # Call, parameters for call
      confess "No such method : $p" unless $q->can($p);                         # Check that the method name is valid
      $q = $p->($q, @p);                                                        # Make call
#     return undef unless $q;                                                   # Abort chain of calls if any call returns undef
      return undef unless defined $q;                                           # Abort chain of calls if any call returns undef
     }
    return $q;                                                                  # Return result of chain
   }
  else                                                                          # Optimized single call
   {my ($p, @p) = split /_/, $AUTOLOAD;                                         # Break out parameters
    confess "No such method : $p" unless $q->can($p);                           # Check that the method name is valid
    unshift @_, $q, @p;                                                         # Insert static parameters into parameter list
    goto &$p;                                                                   # Call desired routine
   }
 }

#D0

#-------------------------------------------------------------------------------
# Export - eeee
#-------------------------------------------------------------------------------

use Exporter qw(import);

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

# containingFolder

@ISA         = qw(Exporter);
@EXPORT      = qw(formatTable);
@EXPORT_OK   = qw(
$defaultBrowser
$defaultEditorHelp
$lastParseError
%savedNodes
@saveLastCutOut
$selectionEnd
$selectionStart
);
%EXPORT_TAGS = (all=>[@EXPORT, @EXPORT_OK]);

# podDocumentation

#Install If you receive the message:\m  Expat.xs:12:19: fatal error: expat.h: No such file or directory\mduring installation, install libexpat1-dev:\m  sudo apt install libexpat1-dev

=pod

=encoding utf-8

=head1 Name

Data::Edit::Xml - Edit data held in the XML format.

=head1 Synopsis

Create a L<new|/new> XML parse tree:

  my $a = Data::Edit::Xml::new q(<a><b><c/></b><d><c/></d></a>);

use:

  say STDERR -p $a;

to L<print|/print>:

  <a>
    <b>
      <c/>
    </b>
    <d>
      <c/>
    </d>
  </a>

L<Cut|/cut> out B<c> under B<b> but not under B<d> in the created tree
by L<traversing|/Traversal> in post-order L<applying|/by> a B<sub> to each node
to L<cut|/cut> out B<c> when we are L<at|/at> B<c> under B<b> under B<a>.

  $a -> by(sub {$_ -> cut_c_b_a});

Or if you know when you are going:

  $a -> go_b_c__cut;

To get:

  <a>
    <b/>
    <d>
      <c/>
    </d>
  </a>

=head2 Bullets to unordered list

To transform a series of bullets into an unordered list, parse the input XML:

  my $a = Data::Edit::Xml::new(<<END);
<a>
<p>• Minimum 1 number</p>
<p>•   No leading, trailing, or embedded spaces</p>
<p>• Not case-sensitive</p>
</a>
END

Traverse the resulting parse tree, removing bullets and leading white space,
changing B<p> to B<li> and B<a> to B<ul>:

  $a->change_ul->by(sub
   {$_->up__change_li if $_->text_p and $_->text =~ s/\A•\s*//s
   });

Print to get:

  ok -p $a eq <<END;
<ul>
  <li>Minimum 1 number</li>
  <li>No leading, trailing, or embedded spaces</li>
  <li>Not case-sensitive</li>
</ul>
END

=head2 XSLT Transformation Example from Wikipedia

Given sample data for an B<XSLT> transformation from
L<Wikipedia|https://en.wikipedia.org/w/index.php?title=XSLT#Example_1_(transforming_XML_to_XML)>

 my $x = Data::Edit::Xml::new(<<END);
<persons>
  <person username="JS1"><name>John</name><surname>Smith</surname></person>
  <person username="MI1"><name>Morka</name><surname>Ismincius</surname></person>
</persons>
END

We can transform it by the rather shorter:

 $x->by(sub
  {$_->unwrap_name;
   $_->cut_surname;
   $_->change_name_person__change_root_persons;
  });

To get:

 ok -p $x eq <<END;
<persons>
  <name username="JS1">John</name>
  <name username="MI1">Morka</name>
</persons>
END

=head2 Convert ol to steps

Given an ordered list:

  my $a = Data::Edit::Xml::new(<<END);
<ol>
  <li>A</li>
  <li>B</li>
  <li>C</li>
</ol>
END

A single call transforms the ordered list:

  $a->ditaConvertOlToSubSteps;

to L<Dita> steps:

  ok -p $a eq <<END;
<substeps>
  <substep>
    <cmd>A</cmd>
  </substep>
  <substep>
    <cmd>B</cmd>
  </substep>
  <substep>
    <cmd>C</cmd>
  </substep>
</substeps>
END

=head1 Description

Edit data held in the XML format.


Version 20201031.


The following sections describe the methods in each functional area of this
module.  For an alphabetic listing of all methods by name see L<Index|/Index>.



=head1 Immediately useful methods

These methods are the ones most likely to be of immediate use to anyone using
this module for the first time:


L<at($node, @context)|/at($node, @context)>

Confirm that the specified B<$node> has the specified L<ancestry|/ancestry>. Ancestry is specified by providing the expected tags that the B<$node>'s parent, the parent's parent etc. must match at each level. If B<undef> is specified then any tag is assumed to match at that level. If a regular expression is specified then the current parent node tag must match the regular expression at that level. If all supplied tags match successfully then the starting node is returned else B<undef>.

L<attrX($node, $attribute)|/attrX($node, $attribute)>

Return the value of the specified B<$attribute> of the specified B<$node> or B<q()> if the B<$node> does not have such an attribute.

L<by($node, $sub)|/by($node, $sub)>

Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. A reference to the current node is also made available via L<$_>. This is equivalent to the L<x=|/opBy> operator.

L<change($node, $name, @context)|/change($node, $name, @context)>

Change the name of the specified B<$node>, optionally  confirming that the B<$node> is in a specified context and return the B<$node>.

L<cut($node, @context)|/cut($node, @context)>

Cut out and return the specified B<$node> so that it can be reinserted else where in the L<parse|/parse> tree.

L<go($node, @path)|/go($node, @path)>

Return the node reached from the specified B<$node> via the specified L<path|/path>: (index positionB<?>)B<*> where index is the tag of the next node to be chosen and position is the optional zero based position within the index of those tags under the current node. Position defaults to zero if not specified. Position can also be negative to index back from the top of the index array. B<*> can be used as the last position to retrieve all nodes with the final tag.

L<moveEndAfter($node, $to, @context)|/moveEndAfter($node, $to, @context)>

Move the end of a B<$node> to just after the specified B<$target> node assuming that the B<$target> node is either a subsequent sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

L<new($fileNameOrString, @options)|/new($fileNameOrString, @options)>

Create a new parse tree - call this method statically as in Data::Edit::Xml::new(file or string) to parse a file or string B<or> with no parameters and then use L</input>, L</inputFile>, L</inputString>, L</errorFile>  to provide specific parameters for the parse, then call L</parse> to perform the parse and return the parse tree.

L<prettyString($node, $depth)|/prettyString($node, $depth)>

Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-p|/opString> $node

L<putLast($old, $new, @context)|/putLast($old, $new, @context)>

Place a L<cut out|/cut> or L<new|/new> node last in the content of the specified B<$node> and return the new node.  See L<putLastCut|/putLastCut> to cut and put last in one operation.  See L<addLast|/addLast> to perform this operation conditionally.

L<putNextRequiredCleanUp($node, @text)|/putNextRequiredCleanUp($node, @text)>

Place a required cleanup next after a specified B<$node> using the specified B<@text>  and return the required clean up node.

L<unwrap($node, @context)|/unwrap($node, @context)>

Unwrap the specified B<$node> if in the optional B<@context> by replacing the node with its contents. Returns the parent node on success, otherwise B<undef> if an attempt is made to unwrap a text node or the root node.

L<wrapWith($node, $tag, @context)|/wrapWith($node, $tag, @context)>

Wrap the specified B<$node> in a new node created from the specified B<$tag> in the optional B<@context> forcing the specified $node down to deepen the L<parse|/parse> tree - return the new wrapping node or B<undef> if this is not possible. See L<addWrapWith|/addWrapWith> to perform this operation conditionally.




=head1 Construction

Create a parse tree, either by parsing a L<file or string|/file or string> B<or> L<node by node|/Node by Node> B<or> from another L<parse tree|/Parse tree>.

=head2 File or String

Construct a parse tree from a file or a string.

=head3 new($fileNameOrString, @options)

Create a new parse tree - call this method statically as in Data::Edit::Xml::new(file or string) to parse a file or string B<or> with no parameters and then use L</input>, L</inputFile>, L</inputString>, L</errorFile>  to provide specific parameters for the parse, then call L</parse> to perform the parse and return the parse tree.

     Parameter          Description
  1  $fileNameOrString  Optional file name or string from which to construct the parse tree
  2  @options           Hash of other options.

B<Example:>



    my $a = Data::Edit::Xml::new(<<END);                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::new


=head3 cdata()

The name of the tag to be used to represent text - this tag must not also be used as a command tag otherwise the parser will L<confess|http://perldoc.perl.org/Carp.html#SYNOPSIS/>.


B<Example:>



    ok Data::Edit::Xml::cdata eq q(CDATA);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 parse($parser)

Parse input L<Xml|https://en.wikipedia.org/wiki/XML> specified via: L<inputFile|/inputFile>, L<input|/input> or L<inputString|/inputString>.

     Parameter  Description
  1  $parser    Parser created by L</new>

B<Example:>


    my $x = Data::Edit::Xml::new;

       $x->inputString = <<END;
  <a id="aa"><b id="bb"><c id="cc"/></b></a>
  END


       $x->parse;                                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


       ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"/>
    </b>
  </a>
  END


=head2 Node by Node

Construct a parse tree node by node.

=head3 newText(undef, $text)

Create a new text node.

     Parameter  Description
  1  undef      Any reference to this package
  2  $text      Content of new text node

B<Example:>


    ok -p $x eq <<END;
  <a class="aa" id="1">
    <b class="bb" id="2"/>
  </a>
  END


    $x->putLast($x->newText("t"));                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a class="aa" id="1">
    <b class="bb" id="2"/>
  t
  </a>
  END


=head3 newTag(undef, $command, %attributes)

Create a new non text node.

     Parameter    Description
  1  undef        Any reference to this package
  2  $command     The tag for the node
  3  %attributes  Attributes as a hash.

B<Example:>


    my $x = Data::Edit::Xml::newTree("a", id=>1, class=>"aa");


    $x->putLast($x->newTag("b", id=>2, class=>"bb"));                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a class="aa" id="1">
    <b class="bb" id="2"/>
  </a>
  END


=head3 newTree($command, %attributes)

Create a new tree.

     Parameter    Description
  1  $command     The name of the root node in the tree
  2  %attributes  Attributes of the root node in the tree as a hash.

B<Example:>



    my $x = Data::Edit::Xml::newTree("a", id=>1, class=>"aa");                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $x eq '<a class="aa" id="1"/>';


=head3 dupTag($node, @context)

Create a new non text node by duplicating the tag of an existing node.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b><c/></b>
  </a>
  END

    my $d = $a->go_b__dupTag;
    ok -p $d eq <<END;
  <b/>
  END


=head3 replaceSpecialChars($string)

Replace < > " & with &lt; &gt; &quot; &amp; Larry Wall's excellent L<Xml parser> unfortunately replaces &lt; &gt; &quot; &amp; etc. with their expansions in text by default and does not seem to provide an obvious way to stop this behavior, so we have to put them back again using this method.

     Parameter  Description
  1  $string    String to be edited.

B<Example:>



    ok Data::Edit::Xml::replaceSpecialChars(q(<">)) eq q(&lt;&quot;&gt;);           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::replaceSpecialChars


=head3 undoSpecialChars($string)

Reverse the results of calling L<replaceSpecialChars|/replaceSpecialChars>.

     Parameter  Description
  1  $string    String to be edited.

B<Example:>



    ok Data::Edit::Xml::undoSpecialChars(q(&lt;&quot;&gt;)) eq q(<">);              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::undoSpecialChars


=head2 Parse tree attributes

Attributes of a node in a parse tree. For instance the attributes associated with an L<Xml|https://en.wikipedia.org/wiki/XML> tag are held in the L<attributes|/attributes> attribute. It should not be necessary to use these attributes directly unless you are writing an extension to this module.  Otherwise you should probably use the methods documented in other sections to manipulate the parse tree as they offer a safer interface at a higher level.

=head2 Parse tree

Construct a L<parse|/parse> tree from another L<parse|/parse> tree.

=head3 renew($node, @context)

Returns a renewed copy of the L<parse|/parse> tree by first printing it and then reparsing it, optionally checking that the starting node is in a specified context: use this method if you have added nodes via the L</"Put as text"> methods and wish to traverse their L<parse|/parse> tree.

Returns the starting node of the new L<parse|/parse> tree or B<undef> if the optional context constraint was supplied but not satisfied.

     Parameter  Description
  1  $node      Node to renew from
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new("<a/>", inputFile=>q(aaa.xml));
    $a->putFirstAsText(qq(<b/>));
    ok !$a->go(q(b));

    my $A = $a->renew;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -t $A->go(q(b)) eq q(b);
    ok $A->root->inputFile eq $a->root->inputFile;


=head3 clone($tree)

Return a clone of the entire L<parse|/parse> tree which is created using the fast L<Storable::dclone> method. The L<parse|/parse> tree is cloned without converting it to string and reparsing it so this method will not L<renew|/renew> any nodes added L<as text|/Put as text>.

Returns the starting node of the new L<parse|/parse> tree.

     Parameter  Description
  1  $tree      Parse tree

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new("<a> </a>");


    my $A = $a->clone;                                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $A eq q(<a/>);

    ok $a->equals($A);

   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>aaa
      <b>bbb</b>
      ccc
      <d>ddd</d>
      eee
    </a>
  </x>
  END


    my $y = $x->clone;                                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$x->diff($y);


=head3 equals($node1, $node2)

Return the first node if the two L<parse|/parse> trees have identical representations via L<string|/string>, else B<undef>.

     Parameter  Description
  1  $node1     Parse tree 1
  2  $node2     Parse tree 2.

B<Example:>


   {my $a = Data::Edit::Xml::new("<a> </a>");

    my $A = $a->clone;

    ok -s $A eq q(<a/>);


    ok $a->equals($A);                                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 equalsIgnoringAttributes($node1, $node2, @attributes)

Return the first node if the two L<parse|/parse> trees have identical representations via L<string|/string> if the specified attributes are ignored, else B<undef>.

     Parameter    Description
  1  $node1       Parse tree 1
  2  $node2       Parse tree 2
  3  @attributes  Attributes to ignore during comparison

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b   id="1" outputclass="1" name="b">
      <c id="2" outputclass="2" name="c"/>
    </b>
  </a>
  END

    my $A = Data::Edit::Xml::new(<<END);
  <a>
    <b   id="11" outputclass="11" name="b">
      <c id="22" outputclass="22" name="c"/>
    </b>
  </a>
  END

    ok !$a->equals($A);


    ok !$a->equalsIgnoringAttributes($A, qw(id));                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->equalsIgnoringAttributes($A, qw(id outputclass));                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 diff($first, $second, $N)

Return () if the dense string representations of the two nodes are equal, else up to the first N (default 16) characters of the common prefix before the point of divergence and the remainder of the string representation of each node from the point of divergence. All <!-- ... --> comments are ignored during this comparison and all spans of white space are reduced to a single blank.

     Parameter  Description
  1  $first     First node
  2  $second    Second node
  3  $N         Maximum length of difference strings to return

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>aaa
      <b>bbb</b>
      ccc
      <d>ddd</d>
      eee
    </a>
  </x>
  END


    ok !$x->diff($x);                                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    my $y = $x->clone;


    ok !$x->diff($y);                                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    $y->first->putLast($x->newTag(q(f)));

    ok nws(<<END) eq nws(-p $y);
  <x>
    <a>aaa
      <b>bbb</b>
      ccc
      <d>ddd</d>
      eee
      <f/>
    </a>
  </x>
  END


    is_deeply [$x->diff($y)],    ["<d>ddd</d> eee <", "/a></x>", "f/></a></x>"];    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    is_deeply [diff(-p $x, $y)], ["<d>ddd</d> eee <", "/a></x>", "f/></a></x>"];    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    is_deeply [$x->diff(-p $y)], ["<d>ddd</d> eee <", "/a></x>", "f/></a></x>"];    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    my $X = writeFile(undef, -p $x);

    my $Y = writeFile(undef, -p $y);


    is_deeply [diff($X, $Y)],    ["<d>ddd</d> eee <", "/a></x>", "f/></a></x>"];    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 save($node, $file)

Save a copy of the L<parse|/parse> tree to a file which can be L<restored|/restore> and return the saved node.  This method uses L<Storable|https://metacpan.org/pod/Storable> which is fast but produces large files that do not compress well.  Use L<writeCompressedFile|/writeCompressedFile> to produce smaller save files at the cost of more time.

     Parameter  Description
  1  $node      Parse tree
  2  $file      File.

B<Example:>



      $y->save($f);                                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      my $Y = Data::Edit::Xml::restore($f);

      ok $Y->equals($y);


=head3 restore($file)

Return a L<parse|/parse> tree from a copy saved in a file by L<save|/save>.

     Parameter  Description
  1  $file      File

B<Example:>


      $y->save($f);


      my $Y = Data::Edit::Xml::restore($f);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $Y->equals($y);


This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::restore


=head3 expandIncludes($x)

Expand the includes mentioned in a L<parse|/parse> tree: any tag that ends in B<include> is assumed to be an include directive.  The file to be included is named on the B<href> keyword.  If the file to be included is a relative file name, i.e. it does not begin with B</> then this file is made absolute relative to the file from which this L<parse|/parse> tree was obtained.

     Parameter  Description
  1  $x         Parse tree

B<Example:>


    my @files =

     (owf("in1/a.xml", q(<a id="a"><include href="../in2/b.xml"/></a>)),

      owf("in2/b.xml", q(<b id="b"><include href="c.xml"/></b>)),

      owf("in2/c.xml", q(<c id="c"/>)));

    my $x = Data::Edit::Xml::new(fpf(currentDirectory, $files[0]));


       $x->expandIncludes;                                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <a id="a">
    <b id="b">
      <c id="c"/>
    </b>
  </a>
  END


=head1 Print

Create a string representation of the L<parse|/parse> tree with optional selection of nodes via L<conditions|/Conditions>.

Normally use the methods in L<Pretty|/Pretty> to format the L<Xml|https://en.wikipedia.org/wiki/XML> in a readable yet reparseable manner; use L<Dense|/Dense> string to format the L<Xml|https://en.wikipedia.org/wiki/XML> densely in a reparseable manner; use the other methods to produce unreparseable strings conveniently formatted to assist various specialized operations such as debugging CDATA, using labels or creating tests. A number of the L<file test operators|/opString> can also be conveniently used to print L<parse|/parse> trees in these formats.

=head2 Pretty

Pretty print the L<parse|/parse> tree.

=head3 prettyString($node, $depth)

Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-p|/opString> $node

     Parameter  Description
  1  $node      Start node
  2  $depth     Optional depth.

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END

    my $a = Data::Edit::Xml::new($s);


    ok $s eq $a->prettyString;                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $s eq -p $a;

   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bbb</b>.
    <c>ccc</c>.
  </a>
  END


=head3 prettyStringHtml($node, @context)

Return a string of L<HTML|https://en.wikipedia.org/wiki/HTML> representing a node of a L<parse|/parse> tree and all the nodes below it if the node is in the specified context.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a id="1">
    <b id="2" b="B">
      <c/>
    </b>
    <d id="3" d="D">
      Some text
    </d>
  </a>
  END


    ok $a->prettyStringHtml eq <<END;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <div class="xmlLine"><span class="xmlLineStartTag"></span><span class="xmlLt">&lt;</span><span class="xmlTag">a</span> <span class="xmlAttr">id</span><span class="xmlEquals">=</span><span class="xmlValue">"1"</span><span class="xmlGt">&gt;</span></div>
  <div class="xmlLine"><span class="xmlLineStartTag">&nbsp;&nbsp;&nbsp;&nbsp;</span><span class="xmlLt">&lt;</span><span class="xmlTag">b</span> <span class="xmlAttr">b</span><span class="xmlEquals">=</span><span class="xmlValue">"B"</span> <span class="xmlAttr">id</span><span class="xmlEquals">=</span><span class="xmlValue">"2"</span><span class="xmlGt">&gt;</span></div>
  <div class="xmlLine"><span class="xmlLineStartTag">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</span><span class="xmlLt">&lt;</span><span class="xmlTag">c</span><span class="xmlSlashGt">/&gt;</span></div>
  <div class="xmlLine"><span class="xmlLineStartTag">&nbsp;&nbsp;&nbsp;&nbsp;</span><span class="xmlLtSlash">&lt;/</span><span class="xmlTag">b</span><span class="xmlGt">&gt;</span></div>
  <div class="xmlLine"><span class="xmlLineStartTag">&nbsp;&nbsp;&nbsp;&nbsp;</span><span class="xmlLt">&lt;</span><span class="xmlTag">d</span> <span class="xmlAttr">d</span><span class="xmlEquals">=</span><span class="xmlValue">"D"</span> <span class="xmlAttr">id</span><span class="xmlEquals">=</span><span class="xmlValue">"3"</span><span class="xmlGt">&gt;</span><span class="xmlText">Some text</span><span class="xmlLtSlash">&lt;/</span><span class="xmlTag">d</span><span class="xmlGt">&gt;</span></div>
  <div class="xmlLine"><span class="xmlLineStartTag"></span><span class="xmlLtSlash">&lt;/</span><span class="xmlTag">a</span><span class="xmlGt">&gt;</span></div>
  END


=head3 prettyStringDitaHeaders($node)

Return a readable string representing the L<parse|/parse> tree below the specified B<$node> with appropriate headers. Or use L<-x|/opString> $node

     Parameter  Description
  1  $node      Start node

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <concept/>
  END

    ok $a->ditaPrettyPrintWithHeaders eq <<END;
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd" []>
  <concept/>
  END

   }


=head3 prettyStringNumbered($node, $depth)

Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it with a L<number|/number> attached to each tag. The node numbers can then be used as described in L<Order|/Order> to monitor changes to the L<parse|/parse> tree.

     Parameter  Description
  1  $node      Start node
  2  $depth     Optional depth.

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END

    $a->numberTree;


    ok $a->prettyStringNumbered eq <<END;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <a id="1">
    <b id="2">
      <A id="3"/>
      <B id="4"/>
    </b>
    <c id="5">
      <C id="6"/>
      <D id="7"/>
    </c>
  </a>
  END


=head3 prettyStringCDATA($node, $depth)

Return a readable string representing a node of a L<parse|/parse> tree and all the nodes below it with the text fields wrapped with <CDATA>...</CDATA>.

     Parameter  Description
  1  $node      Start node
  2  $depth     Optional depth.

B<Example:>


    my $a = Data::Edit::Xml::new("<a><b>A</b></a>");

    my $b = $a->first;

       $b->first->replaceWithBlank;


    ok $a->prettyStringCDATA eq <<END;                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <a>
      <b><CDATA> </CDATA></b>
  </a>
  END


=head3 prettyStringContent($node)

Return a readable string representing all the nodes below a node of a L<parse|/parse> tree.

     Parameter  Description
  1  $node      Start node.

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END


    ok $a->prettyStringContent eq <<END;                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <b>
    <A/>
    <B/>
  </b>
  <c>
    <C/>
    <D/>
  </c>
  END


=head3 prettyStringContentNumbered($node)

Return a readable string representing all the nodes below a node of a L<parse|/parse> tree with numbering added.

     Parameter  Description
  1  $node      Start node.

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <c/>
    </b>
  </a>
  END

    my $a = Data::Edit::Xml::new($s);

    $a->numberTree;


    ok $a->prettyStringContentNumbered eq <<END;                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <b id="2">
    <c id="3"/>
  </b>
  END


    ok $a->go(qw(b))->prettyStringContentNumbered eq <<END;                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <c id="3"/>
  END


=head3 xmlHeader($string)

Add the standard L<Xml|https://en.wikipedia.org/wiki/XML> header to a string

     Parameter  Description
  1  $string    String to which a standard L<Xml|https://en.wikipedia.org/wiki/XML> header should be prefixed

B<Example:>



  ok xmlHeader("<a/>") eq <<END;                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <?xml version="1.0" encoding="UTF-8"?>
  <a/>
  END


This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::xmlHeader


=head2 Html/Json

Represent the L<parse|/parse> tree using html or Json

=head3 htmlTables($node)

Return a string of html representing a L<parse|/parse> tree.

     Parameter  Description
  1  $node      Start node of parse tree

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <task id="t1">
    <title>Title Text</title>
    <taskbody>
      <context>To add a new configuration:</context>
      <steps>
        <step><cmd>Click<b>Next</b>to go to the next page</cmd></step>
        <step><cmd>On the Ports page, complete the following fields</cmd></step>
        <step><cmd>Click to exit without saving</cmd></step>
      </steps>
      <result>
        <p>good!</p>
      </result>
    </taskbody>
  </task>
  END


    ok stringMd5Sum($a->htmlTables =~ s(#\w+) ()gsr) eq q(1e8555c2ea191a54361cbd8cc5baa94b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok stringMd5Sum($a->jsonString)                  eq q(3afa81dc2b2a249834fbac53a1ebd5f1);


=head3 jsonString($node)

Return a Json representation of a parse tree

     Parameter  Description
  1  $node      Start node of parse tree

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <task id="t1">
    <title>Title Text</title>
    <taskbody>
      <context>To add a new configuration:</context>
      <steps>
        <step><cmd>Click<b>Next</b>to go to the next page</cmd></step>
        <step><cmd>On the Ports page, complete the following fields</cmd></step>
        <step><cmd>Click to exit without saving</cmd></step>
      </steps>
      <result>
        <p>good!</p>
      </result>
    </taskbody>
  </task>
  END

    ok stringMd5Sum($a->htmlTables =~ s(#\w+) ()gsr) eq q(1e8555c2ea191a54361cbd8cc5baa94b);

    ok stringMd5Sum($a->jsonString)                  eq q(3afa81dc2b2a249834fbac53a1ebd5f1);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 xmlToJson($xml, $escape)

Return a Json string representing valid L<Xml|https://en.wikipedia.org/wiki/XML> contained in a file or string, optionally double escaping escape characters.

     Parameter  Description
  1  $xml       File name or string of valid html
  2  $escape    Double escape the escaped characters if true

B<Example:>


    my $x = <<END;
  <a a="1">t1
    <b b="2" c="3"/>
  t2
  </a>
  END


    ok xmlToJson($x) eq <<'END';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  {
     "attributes" : {
        "a" : "1"
     },

     "contents" : [
        {
           "tag" : "CDATA",
           "text" : "t1
  "
        },

        {
           "attributes" : {
              "b" : "2",
              "c" : "3"
           },

           "tag" : "b"
        },

        {
           "tag" : "CDATA",
           "text" : "
t2
"
        }
     ],

     "tag" : "a"
  }
  END


    ok xmlToJson($x, 1) eq <<'END';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  {
     "attributes" : {
        "a" : "1"
     },

     "contents" : [
        {
           "tag" : "CDATA",
           "text" : "t1\
  "
        },

        {
           "attributes" : {
              "b" : "2",
              "c" : "3"
           },

           "tag" : "b"
        },

        {
           "tag" : "CDATA",
           "text" : "\
t2\
"
        }
     ],

     "tag" : "a"
  }
  END


This is a static method and so should either be imported or invoked as:

  Data::Edit::Xml::xmlToJson


=head3 jsonToXml($json)

Convert a json string representing an L<Xml|https://en.wikipedia.org/wiki/XML> parse tree into an L<Xml|https://en.wikipedia.org/wiki/XML> parse tree

     Parameter  Description
  1  $json      Json string

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a id="a">
    <b id="b" out="out">b1
      <c id="c"/>
      b2
      <d id="d"/>
      b3
    </b>
  </a>
  END

    my                $json = $a->jsonString;
    ok   stringMd5Sum($json) eq q(5bb9140c7076978dfd5f600d68a82164);

    ok -p jsonToXml  ($json) eq <<END;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <a id="a">
    <b id="b" out="out">b1
      <c id="c"/>
  b2
      <d id="d"/>
  b3
    </b>
  </a>
  END


=head2 Dense

Print the L<parse|/parse> tree densely for reuse by computers rather than humans.

=head3 string($node)

Return a dense string representing a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-s|/opString> B<$node>.

     Parameter  Description
  1  $node      Start node.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    ok -s $a eq '<a><b><c id="42" match="mm"/></b><d><e/></d></a>';


=head3 stringAsMd5Sum($node)

Return the L<md5 sum|https://en.wikipedia.org/wiki/MD5> of the dense L<string|/string> representing a node of a L<parse|/parse> tree minus its L<id> and all the nodes below it. Or use L<-g|/opString> B<$node>. The id of the top most node is not included in the md5sum to equate parse trees that would otherwise only differ by the arbitrary root node id value.

     Parameter  Description
  1  $node      Node.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      c
    </b>
  </a>
  END


    ok $a->stringAsMd5Sum eq q(390bf05d8f5671cc6a4771834840d695);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok ((-k $a)->id       eq q(GUID-390bf05d-8f56-71cc-6a47-71834840d695));
    ok $a->id             ne (-k $a->first)->id;


=head3 stringQuoted($node)

Return a quoted string representing a L<parse|/parse> tree a node of a L<parse|/parse> tree and all the nodes below it. Or use L<-o|/opString> B<$node>.

     Parameter  Description
  1  $node      Start node

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END


    ok $a->stringQuoted eq q('<a><b><A/><B/></b><c><C/><D/></c></a>');              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 stringReplacingIdsWithLabels($node)

Return a string representing the specified L<parse|/parse> tree with the id attribute of each node set to the L<Labels|/Labels> attached to each node.

     Parameter  Description
  1  $node      Start node.

B<Example:>



    ok $x->stringReplacingIdsWithLabels eq '<a><b><c/></b></a>';                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    $b->addLabels(1..4);

    $c->addLabels(5..8);


    ok $x->stringReplacingIdsWithLabels eq '<a><b id="1, 2, 3, 4"><c id="5, 6, 7, 8"/></b></a>';               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    my $s = $x->stringReplacingIdsWithLabels;                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $s eq '<a><b id="1, 2, 3, 4"><c id="5, 6, 7, 8"/></b></a>';


=head3 stringExtendingIdsWithLabels($node)

Return a string representing the specified L<parse|/parse> tree with the id attribute of each node extended by the L<Labels|/Labels> attached to each node.

     Parameter  Description
  1  $node      Start node.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a id="a">
    <b id="b">
      <c id="c"/>
    </b>
    <b id="B">
      <c id="C"/>
    </b>
  </a>
  END

    my $N = 0; $a->by(sub{$_->addLabels((-t $_).++$N)});


    ok -p (new $a->stringExtendingIdsWithLabels) eq <<END;                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  <a id="a, a5">
    <b id="b, b2">
      <c id="c, c1"/>
    </b>
    <b id="B, b4">
      <c id="C, c3"/>
    </b>
  </a>
  END


=head3 stringContent($node)

Return a string representing all the nodes below a node of a L<parse|/parse> tree.

     Parameter  Description
  1  $node      Start node.

B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END


    ok $a->stringContent eq "<b><A/><B/></b><c><C/><D/></c>";                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 stringContentOrText($node)

Return a string representing all the nodes below a node of a L<parse|/parse> tree or the text of the current node if it is a text node.

     Parameter  Description
  1  $node      Start node.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>AAAA</a>
  END

    ok $a->stringContent              eq q(AAAA);

    ok $a->stringContentOrText        eq q(AAAA);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $a->first__stringContent       eq q();
    ok $a->first__stringContentOrText eq q(AAAA);


=head3 stringNode($node)

Return a string representing the specified B<$node> showing the attributes, labels and node number.

     Parameter  Description
  1  $node      Node.

B<Example:>


    ok $x->stringReplacingIdsWithLabels eq '<a><b><c/></b></a>';

    my $b = $x->go(q(b));

    $b->addLabels(1..2);

    $b->addLabels(3..4);

    ok $x->stringReplacingIdsWithLabels eq '<a><b id="1, 2, 3, 4"><c/></b></a>';

    $b->numberTree;


    ok $b->stringNode eq "b(2) 0:1 1:2 2:3 3:4";                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 stringTagsAndText($node)

Return a string showing just the tags and text at and below a specified B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;

   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head3 stringText($node)

Return a string showing just the text of the text nodes (separated by blanks) at and below a specified B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;

   is_deeply [$b->stringText],          [qw(cc dd)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   is_deeply [$B->stringText],          [qw(cc dd)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head3 setRepresentationAsTagsAndText($tree)

Sets the L<representationLast|/representationLast> for every node in the specified B<$tree> via L<stringTagsAndText|/stringTagsAndText>.

     Parameter  Description
  1  $tree      Tree of nodes.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text

   $a->setRepresentationAsTagsAndText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head3 setRepresentationAsText($tree)

Sets the L<representationLast|/representationLast> for every node in the specified B<$tree> via L<stringText|/stringText>.

     Parameter  Description
  1  $tree      Tree of nodes.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text

   $a->setRepresentationAsText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head3 matchNodesByRepresentation($tree)

Creates a hash of arrays of nodes that have the same representation in the specified B<$tree>. Set L<representation|/representationLast> for each node in the tree before calling this method.

     Parameter  Description
  1  $tree      Tree to examine

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;


   my $m  = $a->matchNodesByRepresentation;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;


   my $M  = $a->matchNodesByRepresentation;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head2 Conditions

Print a subset of the L<parse|/parse> tree determined by the conditions attached to it.

=head3 stringWithConditions($node, @conditions)

Return a string representing the specified B<$node> of a L<parse|/parse> tree and all the nodes below it subject to conditions to select or reject some nodes.

     Parameter    Description
  1  $node        Start node
  2  @conditions  Conditions to be regarded as in effect.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END

    my $b = $a >= 'b';

    my ($c, $d) = $b->contents;

    $b->addConditions(qw(bb BB));

    $c->addConditions(qw(cc CC));


    ok $a->stringWithConditions         eq '<a><b><c/><d/></b></a>';                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $a->stringWithConditions(qw(bb)) eq '<a><b><d/></b></a>';                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $a->stringWithConditions(qw(cc)) eq '<a/>';                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 condition($node, $condition, @context)

Return the B<$node> if it has the specified B<$condition> and is in the optional B<@context>, else return B<undef>

     Parameter   Description
  1  $node       Node
  2  $condition  Condition to check
  3  @context    Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    $b->addConditions(qw(bb BB));

    $c->addConditions(qw(cc CC));


    ok  $c->condition(q(cc));                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$c->condition(q(dd));                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $c->condition(q(cc), qw(c b a));                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 anyCondition($node, @conditions)

Return the B<$node> if it has any of the specified B<@conditions>, else return B<undef>

     Parameter    Description
  1  $node        Node
  2  @conditions  Conditions to check

B<Example:>


    $b->addConditions(qw(bb BB));

    $c->addConditions(qw(cc CC));


    ok  $b->anyCondition(qw(bb cc));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$b->anyCondition(qw(cc CC));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 allConditions($node, @conditions)

Return the B<$node> if it has all of the specified B<@conditions>, else return B<undef>

     Parameter    Description
  1  $node        Node
  2  @conditions  Conditions to check

B<Example:>


    $b->addConditions(qw(bb BB));

    $c->addConditions(qw(cc CC));


    ok  $b->allConditions(qw(bb BB));                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$b->allConditions(qw(bb cc));                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 addConditions($node, @conditions)

Given a B<$node> add the specified B<@conditions> and return the node.

     Parameter    Description
  1  $node        Node
  2  @conditions  Conditions to add.

B<Example:>



    $b->addConditions(qw(bb BB));                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok join(' ', $b->listConditions) eq 'BB bb';


=head3 deleteConditions($node, @conditions)

Given a B<$node> delete any B<@conditions> applied to the $node and return the $node.

     Parameter    Description
  1  $node        Node
  2  @conditions  Conditions to add.

B<Example:>


    ok join(' ', $b->listConditions) eq 'BB bb';


    $b->deleteConditions(qw(BB));                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok join(' ', $b->listConditions) eq 'bb';


=head3 listConditions($node)

Return a list of conditions applied to a B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


    $b->addConditions(qw(bb BB));


    ok join(' ', $b->listConditions) eq 'BB bb';                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Attributes

Get or set the attributes of nodes in the L<parse|/parse> tree. L<Well Known Attributes|/Well Known Attributes>  can be set directly via L<lvalue method|http://perldoc.perl.org/perlsub.html#Lvalue-subroutines> B<sub>s. To set or get the values of other attributes use L<Get or Set Attributes|/Get or Set Attributes>. To delete or rename attributes see: L<Other Operations on Attributes|/Other Operations on Attributes>.

=head2 Well Known Attributes

Get or set these node attributes via L<lvalue method|http://perldoc.perl.org/perlsub.html#Lvalue-subroutines> B<sub>s as in:

  $x->href = "#ref";

=head2 Get or Set Attributes

Get or set the attributes of nodes.

=head3 attr($node, $attribute)

Return the value of an attribute of the current node as an L<lvalue method|http://perldoc.perl.org/perlsub.html#Lvalue-subroutines> B<sub>.

     Parameter   Description
  1  $node       Node in parse tree
  2  $attribute  Attribute name.

B<Example:>


    my $x = Data::Edit::Xml::new(my $s = <<END);
  <a number="1"/>
  END


    ok $x->attr(qq(number)) == 1;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



       $x->attr(qq(number))  = 2;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->attr(qq(number)) == 2;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $x eq '<a number="2"/>';


=head3 attrX($node, $attribute)

Return the value of the specified B<$attribute> of the specified B<$node> or B<q()> if the B<$node> does not have such an attribute.

     Parameter   Description
  1  $node       Node in parse tree
  2  $attribute  Attribute name.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b name="bb"/></a>));

    my  $b = $a->first;
    ok  $b->attrX_name eq q(bb);
    ok !$b->attrX_bbb;
   }


=head3 set($node, %values)

Set the values of some attributes in a node and return the node. Identical in effect to L<setAttr|/setAttr>.

     Parameter  Description
  1  $node      Node in parse tree
  2  %values    (attribute name=>new value)*

B<Example:>


    ok q(<a a="1" b="1" id="aa"/>) eq -s $a;


    $a->set(a=>11, b=>undef, c=>3, d=>4, e=>5);                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   }


=head3 addAttr($node, %values)

Check the specified B<$node> for the specified B<%attributes> and add any that are not already set.  Returns the current node.

     Parameter  Description
  1  $node      Node in parse tree
  2  %values    (attribute name=>new value)*

B<Example:>


    my $a = Data::Edit::Xml::new(q(<a id="a"/>));


    $a->addAttr(id=>"b", class=>"c");  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq qq(<a class="c" id="a"/>
);


=head3 setAttr($node, %values)

Set the values of some attributes in a node and return the node. Identical in effect to L<set|/set>.

     Parameter  Description
  1  $node      Node in parse tree
  2  %values    (attribute name=>new value)*

B<Example:>


    ok -s $x eq '<a number="2"/>';


    $x->setAttr(first=>1, second=>2, last=>undef);                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $x eq '<a first="1" number="2" second="2"/>';


=head2 Other Operations on Attributes

Perform operations other than get or set on the attributes of a node

=head3 attrs($node, @attributes)

Return the values of the specified attributes of the current node as a list

     Parameter    Description
  1  $node        Node in parse tree
  2  @attributes  Attribute names.

B<Example:>


    ok -s $x eq '<a first="1" number="2" second="2"/>';


    is_deeply [$x->attrs(qw(third second first ))], [undef, 2, 1];                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 attrCount($node, @exclude)

Return the number of attributes in the specified B<$node>, optionally ignoring the specified names from the count.

     Parameter  Description
  1  $node      Node in parse tree
  2  @exclude   Optional attribute names to exclude from the count.

B<Example:>


    ok -s $x eq '<a first="1" number="2" second="2"/>';


    ok $x->attrCount == 3;                                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->attrCount(qw(first second third)) == 1;                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 getAttrs($node)

Return a sorted list of all the attributes on the specified B<$node>.

     Parameter  Description
  1  $node      Node in parse tree.

B<Example:>


    ok -s $x eq '<a first="1" number="2" second="2"/>';


    is_deeply [$x->getAttrs], [qw(first number second)];                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 deleteAttr($node, $attr, $value)

Delete the named attribute in the specified B<$node>, optionally check its value first, returning the value of the attribute or B<undef> if the attribute does not exist on this node.

     Parameter  Description
  1  $node      Node
  2  $attr      Attribute name
  3  $value     Optional attribute value to check first.

B<Example:>


    ok -s $x eq '<a delete="me" number="2"/>';


    $x->deleteAttr(qq(delete));                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $x eq '<a number="2"/>';


=head3 deleteAttrs($node, @attrs)

Delete the specified attributes of the specified B<$node> without checking their values and return the node.

     Parameter  Description
  1  $node      Node
  2  @attrs     Names of the attributes to delete

B<Example:>


    ok -s $x eq '<a first="1" number="2" second="2"/>';


    $x->deleteAttrs(qw(first second third number));                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $x eq '<a/>';


=head3 deleteAttrsInTree($node, @attrs)

Delete the specified attributes of the specified B<$node> and all the nodes under it and return the specified B<$node>.

     Parameter  Description
  1  $node      Node
  2  @attrs     Names of the attributes to delete

B<Example:>


    ok -p $a eq <<END;
  <a class="2" id="0">
    <b class="1" id="1">
      <c class="0" id="0">
        <d class="1" id="1"/>
        <e class="2" id="0"/>
        <e class="0" id="1"/>
        <f class="1" id="0"/>
        <f class="2" id="1"/>
      </c>
    </b>
  </a>
  END

    $a->deleteAttrsInTree_class;

    ok -p $a eq <<END
  <a id="0">
    <b id="1">
      <c id="0">
        <d id="1"/>
        <e id="0"/>
        <e id="1"/>
        <f id="0"/>
        <f id="1"/>
      </c>
    </b>
  </a>
  END


=head3 attrsNone($node, @context)

Check that the specified B<$node> has no attributes. Return the specified $node if no attributes were found else B<undef>. Invented by MfM.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b/><c id="cc"/></a>));
    ok $a->first__attrsNone;
    ok !$a->last__attrsNone;


=head3 deleteAttrValueAtInTree($tree, $attribute, $value, @context)

Delete all instances of the specified B<$attribute> with the specified B<$value> in the specified B<@context> in the specified B<$tree> and return the modified B<$tree>. An undefined B<$value> will cause the attribute to be deleted without first confirming its value. An empty context will remove the attribute from every node in the B<$tree>.

     Parameter   Description
  1  $tree       Tree
  2  $attribute  Attribute name
  3  $value      Attribute value or B<undef> for all values
  4  @context    Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
     <c id="c"/>
     <c id="d"/>
   </b>
   <d>
     <c id="c"/>
     <c id="d"/>
   </d>
  </a>
  END

    $a->deleteAttrValueAtInTree_id_c_c_b;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <c id="d"/>
    </b>
    <d>
      <c id="c"/>
      <c id="d"/>
    </d>
  </a>
  END


=head3 renameAttr($node, $old, $new, @context)

Rename attribute B<$old> to B<$new> in the specified B<$node> with optional context B<@context> regardless of whether attribute B<$new> already exists or not and return the B<$node>. To prevent inadvertent changes to an existing attribute use L<changeAttr|/changeAttr>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $new       New attribute name
  4  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok $x->printAttributes eq qq( no="1" word="first");


    $x->renameAttr(qw(no number));                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->printAttributes eq qq( number="1" word="first");


=head3 renameAttrXtr($node, @attr)

Rename the attributes B<@attr> as far as possible to xtrc or xtrf.  Returns an array of the attributes that could not be so renamed.

     Parameter  Description
  1  $node      Node
  2  @attr      Attributes to rename if they exist

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(q(<a a="1" b="2" c="3" d="4"/>));
    my @a = $a->renameAttrXtr_a_b_c;
    ok      -A $a eq q(a c="3" d="4" xtrc="1" xtrf="2");
    is_deeply [@a], [qw(c)];

    my $b = Data::Edit::Xml::new(q(<b a="1" b="2" c="3"  d="4" xtrf="5"/>));
    my @b = $b->renameAttrXtr_a_b_c;
    ok      -A $b eq q(b b="2" c="3" d="4" xtrc="1" xtrf="5");
    is_deeply [@b], [qw(b c)];


=head3 changeAttr($node, $old, $new, @context)

Rename attribute B<$old> to B<$new> in the specified B<$node> with optional context B<@context> unless attribute B<$new> is already set and return the B<$node>. To make changes regardless of whether the new attribute already exists use L<renameAttr|/renameAttr>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $new       New attribute name
  4  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok $x->printAttributes eq qq( number="1" word="first");


    $x->changeAttr(qw(number word));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->printAttributes eq qq( number="1" word="first");


=head3 changeOrDeleteAttr($node, $old, $new, @context)

Rename attribute B<$old> to B<$new> in the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set in which case delete attribute B<$old>. Return B<$node> regardless of what action was taken. To make changes regardless of whether the new attribute already exists use L<renameAttr|/renameAttr>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $new       New attribute name
  4  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a a="1"/>
  END

    $a->changeOrDeleteAttr_a_b;

    ok -p $a eq <<END;
  <a b="1"/>
  END

    my $a = Data::Edit::Xml::new(<<END);
  <a a="1" b="2"/>
  END

    $a->changeOrDeleteAttr_a_b;

    ok -p $a eq <<END;
  <a b="2"/>
  END


=head3 renameAttrValue($node, $old, $oldValue, $new, $newValue, @context)

Rename attribute B<$old> to B<$new> with new value B<$newValue> in the specified B<$node> in the optional B<@context> regardless of whether attribute B<$new> already exists or not as long as the attribute B<$old> has the value B<$oldValue>. Return the B<$node> regardless of what changes were made. To prevent inadvertent changes to existing attributes use L<changeAttrValue|/changeAttrValue>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $oldValue  Existing attribute value
  4  $new       New attribute name
  5  $newValue  New attribute value
  6  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok $x->printAttributes eq qq( number="1" word="first");


    $x->renameAttrValue(qw(number 1 numeral I));                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->printAttributes eq qq( numeral="I" word="first");


=head3 changeAttrValue($node, $old, $oldValue, $new, $newValue, @context)

Rename attribute B<$old> to B<$new> with new value B<$newValue> on the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set or the value of the B<$old> attribute is not B<$oldValue>. Return the B<$node> regardless of what changes were made.  To make changes regardless of whether the new attribute already exists use L<renameAttrValue|/renameAttrValue>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $oldValue  Existing attribute value
  4  $new       New attribute name
  5  $newValue  New attribute value
  6  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok $x->printAttributes eq qq( numeral="I" word="first");


    $x->changeAttrValue(qw(word second greek mono));                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->printAttributes eq qq( numeral="I" word="first");


    $x->changeAttrValue(qw(word first greek mono));                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->printAttributes eq qq( greek="mono" numeral="I");


=head3 changeAttributeValue($node, $attribute, $sub, @context)

Apply a sub to the value of an attribute of the specified B<$node>.  The value to be changed is supplied and returned in: L<$_>.

     Parameter   Description
  1  $node       Node
  2  $attribute  Attribute name
  3  $sub        Change sub
  4  @context    Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a aa="abc"/>
  END


    $a->changeAttributeValue(q(aa), sub{s(b) (B)});                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a aa="aBc"/>
  END


=head3 changeOrDeleteAttrValue($node, $old, $oldValue, $new, $newValue, @context)

Rename attribute B<$old> to B<$new> with new value B<$newValue> on the specified B<$node> in the optional B<@context> unless attribute B<$new> is already set or the value of the B<$old> attribute is not B<$oldValue> in which cases the B<$old> attribute is deleted. Return the B<$node> regardless of any changes made.  To make changes regardless of whether the new attribute already exists use L<renameAttrValue|/renameAttrValue>.

     Parameter  Description
  1  $node      Node
  2  $old       Existing attribute name
  3  $oldValue  Existing attribute value
  4  $new       New attribute name
  5  $newValue  New attribute value
  6  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a a="1"/>
  END

    $a->changeOrDeleteAttrValue_a_0_b_1;

    ok -p $a eq <<END;
  <a a="1"/>
  END

    $a->changeOrDeleteAttrValue_a_1_b_3;

    ok -p $a eq <<END;
  <a b="3"/>
  END

    my $a = Data::Edit::Xml::new(<<END);
  <a a="1" b="2"/>
  END

    $a->changeOrDeleteAttrValue_a_0_b_2;
    ok -p $a eq <<END;
  <a b="2"/>
  END

    my $a = Data::Edit::Xml::new(<<END);
  <a a="1"/>
  END

    $a->changeOrDeleteAttrValue_a_1_b_3;

    ok -p $a eq <<END;
  <a b="3"/>
  END


=head3 copyAttrs($source, $target, @attr)

Copy all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, overwriting any existing attributes in the target node and return the source node.

     Parameter  Description
  1  $source    Source node
  2  $target    Target node
  3  @attr      Optional list of attributes to copy

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END

    my ($a, $b, $c) = $x->contents;


    $a->copyAttrs($b, qw(aa bb));                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END


    $a->copyAttrs($b);                                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b a="1" b="2" c="4"/>
    <c/>
  </x>
  END


=head3 copyAttrsFromParent($node, @attr)

Copy all the attributes from the parent (if there is one) of the current B<$node> to $node  or just the named B<@attributes> and return $node. If $node is the top of the parse tree then return B<undef> as it does not have a parent.

     Parameter  Description
  1  $node      Node
  2  @attr      Attributes to copy from parent

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a id="a" a="a">
    <b id="b" b="b"/>
  </a>
  END
    my $b = $a->first;
    $b->copyAttrsFromParent_id;
    ok <<END eq -p $a;
  <a a="a" id="a">
    <b b="b" id="a"/>
  </a>
  END


    $b->copyAttrsFromParent;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $a;
  <a a="a" id="a">
    <b a="a" b="b" id="a"/>
  </a>
  END


=head3 copyAttrsToParent($node, @attr)

Copy all the attributes of the specified B<$node> to its parent (if there is one) or just the named B<@attributes> and return $node. If $node is the top of the parse tree then return B<undef> as it does not have a parent.

     Parameter  Description
  1  $node      Node
  2  @attr      Attributes to copy from parent

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a a="a">
    <b id="b" b="b"/>
  </a>
  END
    my $b = $a->first;
    $b->copyAttrsToParent_id;
    ok <<END eq -p $a;
  <a a="a" id="b">
    <b b="b" id="b"/>
  </a>
  END


    $b->copyAttrsToParent;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $a;
  <a a="a" b="b" id="b">
    <b b="b" id="b"/>
  </a>
  END


=head3 copyNewAttrs($source, $target, @attr)

Copy all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, without overwriting any existing attributes in the target node and return the source node.

     Parameter  Description
  1  $source    Source node
  2  $target    Target node
  3  @attr      Optional list of attributes to copy

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END

    my ($a, $b, $c) = $x->contents;


    $a->copyNewAttrs($b, qw(aa bb));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END


    $a->copyNewAttrs($b);                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b a="1" b="3" c="4"/>
    <c/>
  </x>
  END


=head3 moveAttrs($source, $target, @attr)

Move all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to move is supplied, overwriting any existing attributes in the target node and return the source node.

     Parameter  Description
  1  $source    Source node
  2  $target    Target node
  3  @attr      Attributes to move

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END

    my ($a, $b, $c) = $x->contents;


    $a->moveAttrs($c, qw(aa bb));                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b a="1" b="2" c="4"/>
    <c/>
  </x>
  END


    $b->moveAttrs($c);                                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b/>
    <c a="1" b="2" c="4"/>
  </x>
  END


=head3 moveNewAttrs($source, $target, @attr)

Move all the attributes of the source node to the target node, or, just the named attributes if the optional list of attributes to copy is supplied, without overwriting any existing attributes in the target node and return the source node.

     Parameter  Description
  1  $source    Source node
  2  $target    Target node
  3  @attr      Optional list of attributes to move

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a a="1" b="2"/>
    <b b="3" c="4"/>
    <c/>
  </x>
  END

    my ($a, $b, $c) = $x->contents;


    $b->moveNewAttrs($c, qw(aa bb));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b a="1" b="3" c="4"/>
    <c/>
  </x>
  END


    $b->moveNewAttrs($c);                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok <<END eq -p $x;
  <x>
    <a a="1" b="2"/>
    <b/>
    <c a="1" b="3" c="4"/>
  </x>
  END

    ok <<END eq -p $x;
  <x>
    <c a="1" b="3" c="4"/>
    <b/>
    <a a="1" b="2"/>
  </x>
  END


=head1 Traversal

Traverse the L<parse|/parse> tree in various orders applying a B<sub> to each node.

=head2 Post-order

This order allows you to edit children before their parents.

=head3 by($node, $sub)

Post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. A reference to the current node is also made available via L<$_>. This is equivalent to the L<x=|/opBy> operator.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


     {my $s; $a->by(sub{$s .= $_->tag}); ok $s eq "cbeda"                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 byX($node, $sub)

Post-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die|http://perldoc.perl.org/functions/die.html>. The traversal is halted if the called sub does  L<die|http://perldoc.perl.org/functions/die.html> on any call with the reason in L<?@|http://perldoc.perl.org/perlvar.html#Error-Variables> The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.

Returns the start node regardless of the outcome of calling B<sub>.

     Parameter  Description
  1  $node      Start node
  2  $sub       Sub to call

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


     {my $s; $a->byX(sub{$s .= $_->tag}); ok $s eq "cbeda"                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



  sub byX($$)                                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   {my ($node, $sub) = @_;                                                        # Start node, sub to call
    eval {$node->byX2($sub)};                                                     # Trap any errors that occur
    $node
   }


=head3 byList($node, @context)

Return a list of all the nodes at and below a specified B<$node> in post-order or the empty list if the B<$node> is not in the optional B<@context>.

     Parameter  Description
  1  $node      Starting node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    ok -c $e eq q(e d a);

    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
   </b>
   <e/>
   <f>
    <g/>
    <h/>
   </f>
  </a>
  END


    ok q(c d b e g h f a) eq join ' ', map{-t $_} $a->byList;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok q(h g f e d c b a) eq join ' ', map{-t $_} $a->byReverseList;
    ok q(a b c d e f g h) eq join ' ', map{-t $_} $a->downList;
    ok q(a f h g e b d c) eq join ' ', map{-t $_} $a->downReverseList;


=head3 byReverse($node, $sub, @context)

Reverse post-order traversal of a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting B<$node>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node
  3  @context   Accumulated context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


     {my $s; $a->byReverse(sub{$s .= $_->tag}); ok $s eq "edcba"                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 byReverseX($node, $sub, @context)

Reverse post-order traversal of a L<parse|/parse> tree or sub tree below the specified B<$node> calling the specified B<sub> within L<eval|http://perldoc.perl.org/functions/eval.html>B<{}> at each node and returning the specified starting B<$node>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node
  3  @context   Accumulated context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

     {my $s; $a->byReverse(sub{$s .= $_->tag}); ok $s eq "edcba"


=head3 byReverseList($node, @context)

Return a list of all the nodes at and below a specified B<$node> in reverse preorder or the empty list if the specified B<$node> is not in the optional B<@context>.

     Parameter  Description
  1  $node      Starting node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


      my ($E, $D, $C, $B) = $a->byReverseList;                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -A $C eq q(c id="42" match="mm");

    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
   </b>
   <e/>
   <f>
    <g/>
    <h/>
   </f>
  </a>
  END

    ok q(c d b e g h f a) eq join ' ', map{-t $_} $a->byList;

    ok q(h g f e d c b a) eq join ' ', map{-t $_} $a->byReverseList;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok q(a b c d e f g h) eq join ' ', map{-t $_} $a->downList;
    ok q(a f h g e b d c) eq join ' ', map{-t $_} $a->downReverseList;


=head2 Pre-order

This order allows you to edit children after their parents

=head3 down($node, $sub, @context)

Pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node
  3  @context   Accumulated context.

B<Example:>



     {my $s; $a->down(sub{$s .= $_->tag}); ok $s eq "abcde"                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 downX($node, $sub)

Pre-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die|http://perldoc.perl.org/functions/die.html>. The traversal is halted for the entire L<parse|/parse> tree if the called sub does L<die|http://perldoc.perl.org/functions/die.html> with the reason returned in L<?@|http://perldoc.perl.org/perlvar.html#Error-Variables>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.

Returns the start node regardless of the outcome of calling B<sub>.

     Parameter  Description
  1  $node      Start node
  2  $sub       Sub to call

B<Example:>


     {my $s; $a->down(sub{$s .= $_->tag}); ok $s eq "abcde"


  sub downX($$)                                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   {my ($node, $sub) = @_;                                                        # Start node, sub to call
    eval {$node->downX2($sub)};                                                   # Trap any errors that occur
    $node
   }


=head3 downToDie($node, $sub)

Pre-order traversal of a L<parse|/parse> tree calling the specified B<sub> at each node as long as this sub does not L<die|http://perldoc.perl.org/functions/die.html>. The traversal of the current sub tree is halted and continue with the next sibling or parent if the called sub does L<die|http://perldoc.perl.org/functions/die.html>. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry> up to the node on which this sub was called. A reference to the current node is also made available via L<$_>.

Returns the start node regardless of the outcome of calling B<sub>.

     Parameter  Description
  1  $node      Start node
  2  $sub       Sub to call

B<Example:>



    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    my @a;

    $a->downToDie(sub  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

     {confess if -t $_ eq q(b);
      push @a, -t $_;
     });

    is_deeply [@a], [qw(a d e)];

    my @b;

    $a->downToDie(sub  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

     {confess if -t $_ eq q(c);
      push @b, -t $_;
     });

    is_deeply [@b], [qw(a b d e)];


=head3 downList($node, @context)

Return a list of all the nodes at and below a specified B<$node> in pre-order or the empty list if the B<$node> is not in the optional B<@context>.

     Parameter  Description
  1  $node      Starting node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
   </b>
   <e/>
   <f>
    <g/>
    <h/>
   </f>
  </a>
  END

    ok q(c d b e g h f a) eq join ' ', map{-t $_} $a->byList;
    ok q(h g f e d c b a) eq join ' ', map{-t $_} $a->byReverseList;

    ok q(a b c d e f g h) eq join ' ', map{-t $_} $a->downList;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok q(a f h g e b d c) eq join ' ', map{-t $_} $a->downReverseList;


=head3 downReverse($node, $sub, @context)

Reverse pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node
  3  @context   Accumulated context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


     {my $s; $a->downReverse(sub{$s .= $_->tag}); ok $s eq "adebc"                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 downReverseX($node, $sub, @context)

Reverse pre-order traversal down through a L<parse|/parse> tree or sub tree calling the specified B<sub> within L<eval|http://perldoc.perl.org/functions/eval.html>B<{}> at each node and returning the specified starting node. The B<sub> is passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $sub       Sub to call for each sub node
  3  @context   Accumulated context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

     {my $s; $a->downReverse(sub{$s .= $_->tag}); ok $s eq "adebc"


=head3 downReverseList($node, @context)

Return a list of all the nodes at and below a specified B<$node> in reverse pre-order or the empty list if the B<$node> is not in the optional B<@context>.

     Parameter  Description
  1  $node      Starting node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
   </b>
   <e/>
   <f>
    <g/>
    <h/>
   </f>
  </a>
  END

    ok q(c d b e g h f a) eq join ' ', map{-t $_} $a->byList;
    ok q(h g f e d c b a) eq join ' ', map{-t $_} $a->byReverseList;
    ok q(a b c d e f g h) eq join ' ', map{-t $_} $a->downList;

    ok q(a f h g e b d c) eq join ' ', map{-t $_} $a->downReverseList;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 Pre and Post order

Visit the parent first, then the children, then the parent again.

=head3 through($node, $before, $after, @context)

Traverse L<parse|/parse> tree visiting each node twice calling the specified sub B<$before> as we go down past the node and sub B<$after> as we go up past the node, finally return the specified starting node. The subs B<$before, $after> are passed references to the current node and all of its L<ancestors|/ancestry>. The value of the current node is also made available via L<$_>.

     Parameter  Description
  1  $node      Starting node
  2  $before    Sub to call when we meet a node
  3  $after     Sub to call we leave a node
  4  @context   Accumulated context.

B<Example:>



     {my $s; my $n = sub{$s .= $_->tag}; $a->through($n, $n);                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $s eq "abccbdeeda"


=head3 throughX($node, $before, $after, @context)

Identical to L<through|/through> except the B<$before, $after> subs are called in an L<eval|http://perldoc.perl.org/functions/eval.html> block to prevent L<die|http://perldoc.perl.org/functions/die.html> terminating the traversal of the full tree.

     Parameter  Description
  1  $node      Starting node
  2  $before    Sub to call when we meet a node
  3  $after     Sub to call we leave a node
  4  @context   Accumulated context.

B<Example:>


     {my $s; my $n = sub{$s .= $_->tag}; $a->through($n, $n);

      ok $s eq "abccbdeeda"


=head2 Range

Ranges of nodes

=head3 from($start, @match)

Return a list consisting of the specified node and its following siblings optionally including only those nodes that match one of the tags in the specified list.

     Parameter  Description
  1  $start     Start node
  2  @match     Optional list of tags to match

B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END

     {my ($d, $c, $D) = $a->findByNumbers(5, 7, 10);


      my @f = $d->from;                                                             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @f == 4;

      ok $d == $f[0];


      my @F = $d->from(qw(c));                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @F == 2;

      ok $F[1]->number == 12;

      ok $D == $t[-1];


=head3 to($end, @match)

Return a list of the sibling nodes preceding the specified node optionally including only those nodes that match one of the tags in the specified list.

     Parameter  Description
  1  $end       End node
  2  @match     Optional list of tags to match

B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END

     {my ($d, $c, $D) = $a->findByNumbers(5, 7, 10);


      my @t = $D->to;                                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @t == 4;


      my @T = $D->to(qw(c));                                                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @T == 2;

      ok $T[1]->number == 7;


=head3 fromTo($start, $end, @match)

Return a list of the nodes between the specified start and end nodes optionally including only those nodes that match one of the tags in the specified list.

     Parameter  Description
  1  $start     Start node
  2  $end       End node
  3  @match     Optional list of tags to match

B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END

     {my ($d, $c, $D) = $a->findByNumbers(5, 7, 10);


      my @r = $d->fromTo($D);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @r == 3;


      my @R = $d->fromTo($D, qw(c));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok @R == 1;

      ok $R[0]->number == 7;


      ok !$D->fromTo($d);                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



      ok 1 == $d->fromTo($d);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Location

Locate the line numbers and columns of a specified node and write that information as a L<Oxygen Message|/https://www.oxygenxml.com/doc/versions/20.1/ug-author/topics/linked-output-messages-of-external-engine.html>.

=head2 lineLocation($node)

Return the line number.column location of this tag in its source file or string if the source was parsed with the L<line number|/lineNumber> option on.

     Parameter  Description
  1  $node      Node

B<Example:>


    my $a = Data::Edit::Xml::new(<<END, lineNumbers=>1, inputFile=>q(aaa.xml));
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd">
  <test id="t1">
   <title>Test_</title>
    <testbody>
      <setup>
        <p>Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks>
        <p>Make sure the pot is on an insulated surface.</p>
      </checks>
      <run>
        <p>Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results>
        <p>Pour the tea into a cup.</p>
      </results>
      <outcome>
        <p>An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok -p $a eq <<END;
  <test id="t1" xtrf="3.1:14">
    <title xtrf="4.2:8">Test_</title>
    <testbody xtrf="5.3:12">
      <setup xtrf="6.5:11">
        <p xtrf="7.7:9">Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks xtrf="9.5:12">
        <p xtrf="10.7:9">Make sure the pot is on an insulated surface.</p>
      </checks>
      <run xtrf="12.5:9">
        <p xtrf="13.7:9">Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results xtrf="15.5:13">
        <p xtrf="16.7:9">Pour the tea into a cup.</p>
      </results>
      <outcome xtrf="18.5:13">
        <p xtrf="19.7:9">An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok $a->go_testbody_run_p__location eq q( on line 13 from 7 to 9 in file: aaa.xml);

    my $p = $a->go_testbody_run_p;
    $p->putNext(my $q = $p->newTag_hello);

    ok $p->lineLocation eq q(on line 13 from 7 to 9);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $q->location eq q( in file: aaa.xml);
    ok $q->closestLocation == $p;
    ok $q->approxLocation eq q( on line 13 from 7 to 9 in file: aaa.xml);

    ok $q->formatOxygenMessage(q(E), q(), q(Hello detected)) eq <<END;
  Type: E
  Line: 13
  Column: 7
  EndLine: 13
  EndColumn: 9
  AdditionalInfoURL:
  Description: Hello detected
  END


=head2 location($node, $file)

Return the line number.column plus file location of this tag in its source file or string if the source was parsed with the L<line number|/lineNumber> option on.

     Parameter  Description
  1  $node      Node
  2  $file      Optionally the location of the source.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END, lineNumbers=>1, inputFile=>q(aaa.xml));
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd">
  <test id="t1">
   <title>Test_</title>
    <testbody>
      <setup>
        <p>Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks>
        <p>Make sure the pot is on an insulated surface.</p>
      </checks>
      <run>
        <p>Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results>
        <p>Pour the tea into a cup.</p>
      </results>
      <outcome>
        <p>An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok -p $a eq <<END;
  <test id="t1" xtrf="3.1:14">
    <title xtrf="4.2:8">Test_</title>
    <testbody xtrf="5.3:12">
      <setup xtrf="6.5:11">
        <p xtrf="7.7:9">Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks xtrf="9.5:12">
        <p xtrf="10.7:9">Make sure the pot is on an insulated surface.</p>
      </checks>
      <run xtrf="12.5:9">
        <p xtrf="13.7:9">Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results xtrf="15.5:13">
        <p xtrf="16.7:9">Pour the tea into a cup.</p>
      </results>
      <outcome xtrf="18.5:13">
        <p xtrf="19.7:9">An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok $a->go_testbody_run_p__location eq q( on line 13 from 7 to 9 in file: aaa.xml);

    my $p = $a->go_testbody_run_p;
    $p->putNext(my $q = $p->newTag_hello);
    ok $p->lineLocation eq q(on line 13 from 7 to 9);


    ok $q->location eq q( in file: aaa.xml);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $q->closestLocation == $p;
    ok $q->approxLocation eq q( on line 13 from 7 to 9 in file: aaa.xml);

    ok $q->formatOxygenMessage(q(E), q(), q(Hello detected)) eq <<END;
  Type: E
  Line: 13
  Column: 7
  EndLine: 13
  EndColumn: 9
  AdditionalInfoURL:
  Description: Hello detected
  END


=head2 closestLocation($node)

Return the nearest node with line number.column information

     Parameter  Description
  1  $node      Node

B<Example:>


    my $a = Data::Edit::Xml::new(<<END, lineNumbers=>1, inputFile=>q(aaa.xml));
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd">
  <test id="t1">
   <title>Test_</title>
    <testbody>
      <setup>
        <p>Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks>
        <p>Make sure the pot is on an insulated surface.</p>
      </checks>
      <run>
        <p>Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results>
        <p>Pour the tea into a cup.</p>
      </results>
      <outcome>
        <p>An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok -p $a eq <<END;
  <test id="t1" xtrf="3.1:14">
    <title xtrf="4.2:8">Test_</title>
    <testbody xtrf="5.3:12">
      <setup xtrf="6.5:11">
        <p xtrf="7.7:9">Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks xtrf="9.5:12">
        <p xtrf="10.7:9">Make sure the pot is on an insulated surface.</p>
      </checks>
      <run xtrf="12.5:9">
        <p xtrf="13.7:9">Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results xtrf="15.5:13">
        <p xtrf="16.7:9">Pour the tea into a cup.</p>
      </results>
      <outcome xtrf="18.5:13">
        <p xtrf="19.7:9">An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok $a->go_testbody_run_p__location eq q( on line 13 from 7 to 9 in file: aaa.xml);

    my $p = $a->go_testbody_run_p;
    $p->putNext(my $q = $p->newTag_hello);
    ok $p->lineLocation eq q(on line 13 from 7 to 9);

    ok $q->location eq q( in file: aaa.xml);

    ok $q->closestLocation == $p;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $q->approxLocation eq q( on line 13 from 7 to 9 in file: aaa.xml);

    ok $q->formatOxygenMessage(q(E), q(), q(Hello detected)) eq <<END;
  Type: E
  Line: 13
  Column: 7
  EndLine: 13
  EndColumn: 9
  AdditionalInfoURL:
  Description: Hello detected
  END


=head2 approxLocation($node, $file)

Return the line number.column location of the node nearest to this node in the source file if the source was parsed with the L<line number|/lineNumber> option on.

     Parameter  Description
  1  $node      Node
  2  $file      Optionally the location of the source.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END, lineNumbers=>1, inputFile=>q(aaa.xml));
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd">
  <test id="t1">
   <title>Test_</title>
    <testbody>
      <setup>
        <p>Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks>
        <p>Make sure the pot is on an insulated surface.</p>
      </checks>
      <run>
        <p>Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results>
        <p>Pour the tea into a cup.</p>
      </results>
      <outcome>
        <p>An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok -p $a eq <<END;
  <test id="t1" xtrf="3.1:14">
    <title xtrf="4.2:8">Test_</title>
    <testbody xtrf="5.3:12">
      <setup xtrf="6.5:11">
        <p xtrf="7.7:9">Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks xtrf="9.5:12">
        <p xtrf="10.7:9">Make sure the pot is on an insulated surface.</p>
      </checks>
      <run xtrf="12.5:9">
        <p xtrf="13.7:9">Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results xtrf="15.5:13">
        <p xtrf="16.7:9">Pour the tea into a cup.</p>
      </results>
      <outcome xtrf="18.5:13">
        <p xtrf="19.7:9">An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok $a->go_testbody_run_p__location eq q( on line 13 from 7 to 9 in file: aaa.xml);

    my $p = $a->go_testbody_run_p;
    $p->putNext(my $q = $p->newTag_hello);
    ok $p->lineLocation eq q(on line 13 from 7 to 9);

    ok $q->location eq q( in file: aaa.xml);
    ok $q->closestLocation == $p;

    ok $q->approxLocation eq q( on line 13 from 7 to 9 in file: aaa.xml);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $q->formatOxygenMessage(q(E), q(), q(Hello detected)) eq <<END;
  Type: E
  Line: 13
  Column: 7
  EndLine: 13
  EndColumn: 9
  AdditionalInfoURL:
  Description: Hello detected
  END


=head2 formatOxygenMessage($node, $level, $url, @message)

Write an error message in Oxygen format

     Parameter  Description
  1  $node      Node
  2  $level     Error level [F|E|W]
  3  $url       Explanatory Url
  4  @message   Message text

B<Example:>


    my $a = Data::Edit::Xml::new(<<END, lineNumbers=>1, inputFile=>q(aaa.xml));
  <?xml version="1.0" encoding="UTF-8"?>
  <!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd">
  <test id="t1">
   <title>Test_</title>
    <testbody>
      <setup>
        <p>Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks>
        <p>Make sure the pot is on an insulated surface.</p>
      </checks>
      <run>
        <p>Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results>
        <p>Pour the tea into a cup.</p>
      </results>
      <outcome>
        <p>An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok -p $a eq <<END;
  <test id="t1" xtrf="3.1:14">
    <title xtrf="4.2:8">Test_</title>
    <testbody xtrf="5.3:12">
      <setup xtrf="6.5:11">
        <p xtrf="7.7:9">Place the boiling water and fresh tea in the pot.</p>
      </setup>
      <checks xtrf="9.5:12">
        <p xtrf="10.7:9">Make sure the pot is on an insulated surface.</p>
      </checks>
      <run xtrf="12.5:9">
        <p xtrf="13.7:9">Stir with a spoon then let brew for 5 minutes.</p>
      </run>
      <results xtrf="15.5:13">
        <p xtrf="16.7:9">Pour the tea into a cup.</p>
      </results>
      <outcome xtrf="18.5:13">
        <p xtrf="19.7:9">An enjoyable cup of tea.</p>
      </outcome>
    </testbody>
  </test>
  END

    ok $a->go_testbody_run_p__location eq q( on line 13 from 7 to 9 in file: aaa.xml);

    my $p = $a->go_testbody_run_p;
    $p->putNext(my $q = $p->newTag_hello);
    ok $p->lineLocation eq q(on line 13 from 7 to 9);

    ok $q->location eq q( in file: aaa.xml);
    ok $q->closestLocation == $p;
    ok $q->approxLocation eq q( on line 13 from 7 to 9 in file: aaa.xml);


    ok $q->formatOxygenMessage(q(E), q(), q(Hello detected)) eq <<END;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  Type: E
  Line: 13
  Column: 7
  EndLine: 13
  EndColumn: 9
  AdditionalInfoURL:
  Description: Hello detected
  END


=head1 Position

Confirm that the position L<navigated|/Navigation> to is the expected position.

=head2 at($node, @context)

Confirm that the specified B<$node> has the specified L<ancestry|/ancestry>. Ancestry is specified by providing the expected tags that the B<$node>'s parent, the parent's parent etc. must match at each level. If B<undef> is specified then any tag is assumed to match at that level. If a regular expression is specified then the current parent node tag must match the regular expression at that level. If all supplied tags match successfully then the starting node is returned else B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Ancestry.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c> <d/> </c>
      <c> <e/> </c>
      <c> <f/> </c>
    </b>
  </a>
  END


    ok  $a->go(qw(b c -1 f))->at(qw(f c b a));                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->go(qw(b c  1 e))->at(undef, qr(c|d), undef, qq(a));                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $d->context eq q(d c b a);


    ok  $d->at(qw(d c b), undef);                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->at(qw(d c b), undef, undef);                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->at(qw(d e b));                                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 atText($node, $re, @context)

Confirm that we are on a text node whose text value matches a regular expression in the optional B<@context>. Return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Text node
  2  $re        Regular expression to match
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>abcdcba</a>
  END


=head2 atStringContentMatches($node, $re, @context)

Confirm that we are on a B<$node> whose contents, represented as a string, matches the specified regular expression B<$re> in the optional B<@context>. Return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Text node
  2  $re        Regular expression to match
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc</c>
      <d>dd</d>
    </b>
  </a>
  END

    $a->by(sub
     {my ($o) = @_;
      if (!$o->atStringContentMatches_dd)
       {$o->id = "no";
       }
     });

    ok -p $a eq <<END;
  <a>
    <b>
      <c id="no">cc</c>
      <d>dd</d>
    </b>
  </a>
  END


=head2 atTop($node)

Return the current node if it is the root == top of a parse tree else return B<undef>.

     Parameter  Description
  1  $node      Node

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b/></a>
  END


    ok  $a->atTop;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok !$a->first__atTop;


=head2 attrAt($node, $attribute, @context)

Return the specified B<$node> if it has the specified B<$attribute> and the $node is in the optional B<@context> else return B<undef>.

     Parameter   Description
  1  $node       Starting node
  2  $attribute  Attribute
  3  @context    Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b c="C"/></a>));
    my $b = $a->first;
    ok !$a->attrAt_id;
    ok  $b->attrAt_c_b_a;
    ok !$b->attrAt_b_b_a;
    ok !$b->attrValueAt_c_C_c_a;
    ok  $b->attrValueAt_c_C_b_a;
   }


=head2 attrValueAt($node, $attribute, $value, @context)

Return the specified B<$node> if it has the specified B<$attribute> with the specified B<$value> and the $node is in the optional B<@context> else return B<undef>.

     Parameter   Description
  1  $node       Starting node
  2  $attribute  Attribute
  3  $value      Wanted value of attribute
  4  @context    Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b c="C"/></a>));
    my $b = $a->first;
    ok !$a->attrAt_id;
    ok  $b->attrAt_c_b_a;
    ok !$b->attrAt_b_b_a;
    ok !$b->attrValueAt_c_C_c_a;
    ok  $b->attrValueAt_c_C_b_a;
   }


=head2 not($node, @tags)

Return the specified B<$node> if it does not match any of the specified tags, else B<undef>

     Parameter  Description
  1  $node      Node
  2  @tags      Tags not to match

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
  </a>
  END

    ok $a->first->not_a_c;


=head2 atOrBelow($start, @context)

Confirm that the node or one of its ancestors has the specified context as recognized by L<at|/at> and return the first node that matches the context or B<undef> if none do.

     Parameter  Description
  1  $start     Starting node
  2  @context   Ancestry.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok $d->context eq q(d c b a);


    ok  $d->atOrBelow(qw(d c b a));                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $d->atOrBelow(qw(  c b a));                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $d->atOrBelow(qw(    b a));                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->atOrBelow(qw(  c   a));                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 adjacent($first, $second)

Return the first node if it is adjacent to the second node else B<undef>.

     Parameter  Description
  1  $first     First node
  2  $second    Second node

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <b>
      <c/>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $C, $B, $f, $e) = $a->byList;


    ok !$a->adjacent($B);                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $b->adjacent($B);                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 ancestry($node)

Return a list containing: (the specified B<$node>, its parent, its parent's parent etc..). Or use L<upn|/upn> to go up the specified number of levels.

     Parameter  Description
  1  $node      Starting node.

B<Example:>


    $a->numberTree;

    ok $a->prettyStringNumbered eq <<END;
  <a id="1">
    <b id="2">
      <A id="3"/>
      <B id="4"/>
    </b>
    <c id="5">
      <C id="6"/>
      <D id="7"/>
    </c>
  </a>
  END


    is_deeply [map {-t $_} $a->findByNumber(7)->ancestry], [qw(D c a)];             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 context($node)

Return a string containing the tag of the starting node and the tags of all its ancestors separated by single spaces.

     Parameter  Description
  1  $node      Starting node.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


    ok $a->go(qw(d e))->context eq 'e d a';                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 containsSingleText($node, @context)

Return the single text element below the specified B<$node> else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new("<a><b>bb</b><c>cc<d/>ee</c></a>");


    ok  $a->go(q(b))->containsSingleText->text eq q(bb);                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(q(c))->containsSingleText;                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 depth($node)

Returns the depth of the specified B<$node>, the  depth of a root node is zero.

     Parameter  Description
  1  $node      Node.

B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END


    ok 0 == $a->depth;                                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok 4 == $a->findByNumber(14)->depth;                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b><c><d/></c><e/></b></a>));

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
    </b>
  </a>
  END

   my ($d, $c, $e, $b) = $a->byList;
   ok $a->height == 4;

   ok $a->depth  == 0;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   ok $c->depth  == 2;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok $c->height == 2;

   ok $e->depth  == 2;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok $e->height == 1;

   is_deeply [$a->depthProfile], [qw(4 3 3 2 1)];
  }

  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head2 depthProfile($node)

Returns the depth profile of the tree rooted at the specified B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b><c><d/></c><e/></b></a>));

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
    </b>
  </a>
  END

   my ($d, $c, $e, $b) = $a->byList;
   ok $a->height == 4;
   ok $a->depth  == 0;
   ok $c->depth  == 2;
   ok $c->height == 2;
   ok $e->depth  == 2;
   ok $e->height == 1;


   is_deeply [$a->depthProfile], [qw(4 3 3 2 1)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  }

  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head2 setDepthProfile($tree)

Sets the L<depthProfile|/depthProfile> for every node in the specified B<$tree>. The last set L<depthProfile|/depthProfile> for a specific niode can be retrieved from L<depthProfileLast|/depthProfileLast>.

     Parameter  Description
  1  $tree      Tree of nodes.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;


   $a->setDepthProfile;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head2 height($node)

Returns the height of the tree rooted at the specified B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a><b><c><d/></c><e/></b></a>));

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
    </b>
  </a>
  END

   my ($d, $c, $e, $b) = $a->byList;

   ok $a->height == 4;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok $a->depth  == 0;
   ok $c->depth  == 2;

   ok $c->height == 2;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok $e->depth  == 2;

   ok $e->height == 1;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   is_deeply [$a->depthProfile], [qw(4 3 3 2 1)];
  }

  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>cc
        <d/>
  dd
      </c>
    </b>
    <B>
      <c>cc
        <d/>
  dd
      </c>
    </B>
  </a>
  END

   my $b = $a->first_b; my $B = $a->last_B;
   my $c = $b->first_c; my $C = $B->first_c;
   my $d = $c->first_d; my $D = $C->first_d;

   $a->setDepthProfile;

   ok $b->depthProfileLast eq q(3 3 3 2 1);
   ok $b->depthProfileLast eq $B->depthProfileLast;

  # Represent using tags and text
   $a->setRepresentationAsTagsAndText;
   is_deeply [$b->stringTagsAndText],   [qw(cc d dd c b)];
   is_deeply [$B->stringTagsAndText],   [qw(cc d dd c B)];
   ok         $b->representationLast  eq qq(cc d dd c b);
   ok         $B->representationLast  eq qq(cc d dd c B);
   ok         $c->representationLast  eq qq(cc d dd c);
   ok         $C->representationLast  eq qq(cc d dd c);
   ok dump($b->representationLast) ne dump($B->representationLast);
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $m  = $a->matchNodesByRepresentation;

   my $bb = $b->representationLast;
   is_deeply $m->{$bb}, [$b];

   my $cc = $c->representationLast;
   is_deeply $m->{$cc}, [$c, $C];

  # Represent using just text
   $a->setRepresentationAsText;
   is_deeply [$b->stringText],          [qw(cc dd)];
   is_deeply [$B->stringText],          [qw(cc dd)];
   ok         $b->representationLast  eq qq(cc dd);
   ok         $B->representationLast  eq qq(cc dd);
   is_deeply  $b->representationLast,
              $B->representationLast;
   is_deeply  $c->representationLast,
              $C->representationLast;

   my $M  = $a->matchNodesByRepresentation;
   my $BB = $b->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   my $CC = $c->representationLast;
   is_deeply $M->{$BB}, [$c, $b, $C, $B];

   ok $b->representationLast eq $c->representationLast;
  }

  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));
    ok $a->first->isOnlyChildText;
   }


=head2 isFirst($node, @context)

Return the specified B<$node> if it is first under its parent and optionally has the specified context, else return B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<isFirstNonBlank> to skip a (rare) initial blank text CDATA. Use B<isFirstNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


    ok $a->go(q(b))->isFirst;                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <b>
      <c/>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $C, $B, $f, $e) = $a->byList;


    ok  $a->isFirst;                                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 isNotFirst($node, @context)

Return the specified B<$node> if it is not first under its parent and optionally has the specified context, else return B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b/><c/><d/></a>
  END

    my ($b, $c, $d) = $a->byList;

    ok $c->isNotFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $b->isNotLast;


=head2 isFirstN($node, $N, @context)

Return the first B<$N> nodes as an array if the first B<$N> tags of the parent of B<$node> finish at the specified B<$node> and have the specified tags in the sequence specified by B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.

     Parameter  Description
  1  $node      Node
  2  $N         Number of tags to be first
  3  @context   First tags and optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
  </x>
  END

    my ($a, $b, $c, $d, $e) = $x->byList;

    is_deeply [$b->isFirstN_2_a_b_x], [$a, $b];
    ok !$b->isFirstN_2_b_x;

    is_deeply [$d->isLastN_2_d_e_x], [$d, $e];
    ok !$d->isLastN_2_d_x;

    is_deeply [$b->nextN_3_b_c_d], [$b, $c, $d];
    is_deeply [$d->prevN_3_b_c_d], [$b, $c, $d];


=head2 isFirstToDepth($node, $depth, @context)

Return the specified B<$node> if it is first to the specified depth else return B<undef>

     Parameter  Description
  1  $node      Node
  2  $depth     Depth
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $f, $e) = $a->byList;


    ok  $d->isFirstToDepth(4);                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$f->isFirstToDepth(2);                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $f->isFirstToDepth(1);                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$f->isFirstToDepth(3);                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 firstIs($node, @context)

Return the specified B<$node> if it has one or more child nodes and the first child node has the specified B<@context> otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/>
  </a>
  END

    ok  $a->go_c__nextIs_d;
    ok !$a->go_c__nextIs_c;
    ok  $a->go_c__prevIs_b;
    ok !$a->go_c__prevIs_e;

    ok  $a->firstIs_b;
    ok !$a->firstIs_c;

    ok  $a->lastIs_e;
    ok !$a->lastIs_d;


=head2 isLast($node, @context)

Return the specified B<$node> if it is last under its parent and optionally has the specified context, else return B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<isLastNonBlank> to skip a (rare) initial blank text CDATA. Use B<isLastNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="42" match="mm"/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END


    ok $a->go(q(d))->isLast;                                                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <b>
      <c/>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $C, $B, $f, $e) = $a->byList;


    ok  $a->isLast;                                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 isNotLast($node, @context)

Return the specified B<$node> if it is not last under its parent and optionally has the specified context, else return B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b/><c/><d/></a>
  END

    my ($b, $c, $d) = $a->byList;
    ok $c->isNotFirst;

    ok $b->isNotLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 isLastN($node, $N, @context)

Return the last B<$N> nodes as an array if the last B<$N> tags of the parent of B<$node> start at the specified B<$node> and have the specified tags in the sequence specified by B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.

     Parameter  Description
  1  $node      Node
  2  $N         Number of tags to be last
  3  @context   Last tags and optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
  </x>
  END

    my ($a, $b, $c, $d, $e) = $x->byList;

    is_deeply [$b->isFirstN_2_a_b_x], [$a, $b];
    ok !$b->isFirstN_2_b_x;

    is_deeply [$d->isLastN_2_d_e_x], [$d, $e];
    ok !$d->isLastN_2_d_x;

    is_deeply [$b->nextN_3_b_c_d], [$b, $c, $d];
    is_deeply [$d->prevN_3_b_c_d], [$b, $c, $d];


=head2 isLastToDepth($node, $depth, @context)

Return the specified B<$node> if it is last to the specified depth else return B<undef>

     Parameter  Description
  1  $node      Node
  2  $depth     Depth
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $f, $e) = $a->byList;


    ok  $c->isLastToDepth(1);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$c->isLastToDepth(3);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $d->isLastToDepth(2);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->isLastToDepth(4);                                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 lastIs($node, @context)

Return the specified B<$node> if it has one or more child nodes and the last child node has the specified B<@context> otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/>
  </a>
  END

    ok  $a->go_c__nextIs_d;
    ok !$a->go_c__nextIs_c;
    ok  $a->go_c__prevIs_b;
    ok !$a->go_c__prevIs_e;

    ok  $a->firstIs_b;
    ok !$a->firstIs_c;

    ok  $a->lastIs_e;
    ok !$a->lastIs_d;


=head2 nextIs($node, @context)

Return the specified B<$node> if there is a following node with the specified B<@context>. Returns B<undef> if the $node is last.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/>
  </a>
  END

    ok  $a->go_c__nextIs_d;
    ok !$a->go_c__nextIs_c;
    ok  $a->go_c__prevIs_b;
    ok !$a->go_c__prevIs_e;

    ok  $a->firstIs_b;
    ok !$a->firstIs_c;

    ok  $a->lastIs_e;
    ok !$a->lastIs_d;


=head2 nextN($node, $N, @context)

Return B<$N> nodes as an array starting at B<$node> inclusive if they match the first B<$N> tags of B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.

     Parameter  Description
  1  $node      Node
  2  $N         Number of tags to be last
  3  @context   Last tags and optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
  </x>
  END

    my ($a, $b, $c, $d, $e) = $x->byList;

    is_deeply [$b->isFirstN_2_a_b_x], [$a, $b];
    ok !$b->isFirstN_2_b_x;

    is_deeply [$d->isLastN_2_d_e_x], [$d, $e];
    ok !$d->isLastN_2_d_x;

    is_deeply [$b->nextN_3_b_c_d], [$b, $c, $d];
    is_deeply [$d->prevN_3_b_c_d], [$b, $c, $d];


=head2 prevIs($node, @context)

Return the specified B<$node> if there is a previous node with the specified B<@context>. Returns B<undef> if the $node is first.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/>
  </a>
  END

    ok  $a->go_c__nextIs_d;
    ok !$a->go_c__nextIs_c;
    ok  $a->go_c__prevIs_b;
    ok !$a->go_c__prevIs_e;

    ok  $a->firstIs_b;
    ok !$a->firstIs_c;

    ok  $a->lastIs_e;
    ok !$a->lastIs_d;


=head2 prevN($node, $N, @context)

Return B<$N> nodes as an array ending at B<$node> inclusive if they match the first B<$N> tags of B<@context>. If B<@context> contains more then B<$N> entries, the remainder are checked as the context of the parent of B<$node>. Returns an array of nodes found or an empty array if the specified B<$node> does not match these conditions.

     Parameter  Description
  1  $node      Node
  2  $N         Number of tags to be first
  3  @context   First tags and optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
  </x>
  END

    my ($a, $b, $c, $d, $e) = $x->byList;

    is_deeply [$b->isFirstN_2_a_b_x], [$a, $b];
    ok !$b->isFirstN_2_b_x;

    is_deeply [$d->isLastN_2_d_e_x], [$d, $e];
    ok !$d->isLastN_2_d_x;

    is_deeply [$b->nextN_3_b_c_d], [$b, $c, $d];
    is_deeply [$d->prevN_3_b_c_d], [$b, $c, $d];


=head2 isOnlyChild($node, @context)

Return the specified B<$node> if it is the only node under its parent ignoring any surrounding blank text.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $f, $e) = $a->byList;


    ok  $d->isOnlyChild;                                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->isOnlyChild(qw(b));                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <b>
      <c/>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $C, $B, $f, $e) = $a->byList;


    ok  $a->isOnlyChild;                                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
    <f>
      <g/>
    </f>
  </a>
  END


=head2 isOnlyChildToDepth($node, $depth, @context)

Return the specified B<$node> if it and its ancestors are L<only children|/isOnlyChild> to the specified depth else return B<undef>. isOnlyChildToDepth(1) is the same as L<isOnlychild|/isOnlyChild>

     Parameter  Description
  1  $node      Node
  2  $depth     Depth to which each parent node must also be an only child
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $f, $e) = $a->byList;


    ok  $d->isOnlyChildToDepth(1, qw(d c b a));                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $d->isOnlyChildToDepth(2, qw(d c b a));                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->isOnlyChildToDepth(3, qw(d c b a));                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 isOnlyChildN($node, $depth, @context)

Return the specified B<$node> if it is an only child of B<$depth> ancestors with L<only children|/hasSingleChild> in the opptional B<@context> else return B<undef>. isOnlyChildN(1) is the same as L<isOnlychild|/isOnlyChild>.

     Parameter  Description
  1  $node      Node
  2  $depth     Depth to which each parent node must also be an only child
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
    <f>
      <g/>
    </f>
  </a>
  END


=head2 isOnlyChildText($node, @context)

Return the specified B<$node> if it is a text node and it is an only child else and its parent is in the specified optional context else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a>aaaa</a>));

    ok $a->first->isOnlyChildText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   }


=head2 hasSingleChild($node, @context)

Return the only child of the specified B<$node> if the child is the only node under its parent ignoring any surrounding blank text and has the optional specified context, else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b   id="b" b="bb">
      <b id="c" c="cc"/>
    </b>
  </a>
  END

    my ($c, $b) = $a->byList;

    is_deeply [$b->id, $c->id], [qw(b c)];


    ok $c == $b->hasSingleChild;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $b == $a->hasSingleChild;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 hasSingleChildText($node, @context)

Return the only child of the specified B<$node> if the child is a text node and the child is the only node under its parent ignoring any surrounding blank text and the child has the optional specified context, else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>aa
    <b>bb</b>
  </a>
  END

    ok !$a->hasSingleChildText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $a->last->hasSingleChildText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 hasSingleChildToDepth($node, $depth, @context)

Return the descendant of the specified B<$node> if it has single children to the specified depth in the specified optional B<@context> else return B<undef>.  L<hasSingleChildToDepth(0)|/hasSingleChildToDepth> is equivalent to L<hasSingleChild|/hasSingleChild>.

     Parameter  Description
  1  $node      Node
  2  $depth     Depth
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok $h == $g->hasSingleChildToDepth(1);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $i == $g->hasSingleChildToDepth_2_i_h_g;

    ok      !$g->hasSingleChildToDepth_2_i_h_G;


    ok      !$g->hasSingleChildToDepth(0);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok      !$g->hasSingleChildToDepth(3);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $i == $i->hasSingleChildToDepth(0);                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 isEmpty($node, @context)

Confirm that the specified B<$node> is empty, that is: the specified B<$node> has no content, not even a blank string of text. To test for blank nodes, see L<isAllBlankText|/isAllBlankText>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <a>

  </a>
  END


    ok $x->isEmpty;                                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my ($d, $c, $b, $f, $e) = $a->byList;


    ok  $d->isEmpty;                                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my ($c, $b) = $a->byList;


    ok $c->isEmpty;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $b->hasContent;


=head2 hasContent($node, @context)

Confirm that the specified B<$node> has content. Return the specified node if it has content else return B<undef> if it does not.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my ($c, $b) = $a->byList;

    ok $c->isEmpty;

    ok $b->hasContent;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 over($node, $re, @context)

Confirm that the string representing the tags at the level below the specified B<$node> match a regular expression where each pair of tags is separated by a single space. Use L<contentAsTags|/contentAsTags> to visualize the tags at the next level.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(q(b))->over(qr(d.+e));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 over2($node, $re, @context)

Confirm that the string representing the tags at the level below the specified B<$node> match a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q? . Use L<contentAsTags2|/contentAsTags2> to visualize the tags at the next level.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(q(b))->over2(qr(\A c  d  e  f  g \Z));                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok $x->go(q(b))->contentAsTags  eq q(c d e f g) ;


=head2 overAllTags($node, @tags)

Return the specified b<$node> if all of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
  </a>
  END

    ok  $a->overAllTags_b_c_d;
    ok !$a->overAllTags_b_c;
    ok !$a->overAllTags_b_c_d_e;
    ok  $a->oat_b_c_d;
    ok !$a->oat_B_c_d;

    ok  $a->overFirstTags_b_c_d;
    ok  $a->overFirstTags_b_c;
    ok !$a->overFirstTags_b_c_d_e;
    ok  $a->oft_b_c;
    ok !$a->oft_B_c;

    ok  $a->overLastTags_b_c_d;
    ok  $a->overLastTags_c_d;
    ok !$a->overLastTags_b_c_d_e;
    ok  $a->olt_c_d;
    ok !$a->olt_C_d;
   }


B<oat> is a synonym for L<overAllTags|/overAllTags>.


=head2 overFirstTags($node, @tags)

Return the specified b<$node> if the first of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
  </a>
  END

    ok  $a->overAllTags_b_c_d;
    ok !$a->overAllTags_b_c;
    ok !$a->overAllTags_b_c_d_e;
    ok  $a->oat_b_c_d;
    ok !$a->oat_B_c_d;

    ok  $a->overFirstTags_b_c_d;
    ok  $a->overFirstTags_b_c;
    ok !$a->overFirstTags_b_c_d_e;
    ok  $a->oft_b_c;
    ok !$a->oft_B_c;

    ok  $a->overLastTags_b_c_d;
    ok  $a->overLastTags_c_d;
    ok !$a->overLastTags_b_c_d_e;
    ok  $a->olt_c_d;
    ok !$a->olt_C_d;
   }


B<oft> is a synonym for L<overFirstTags|/overFirstTags>.


=head2 overLastTags($node, @tags)

Return the specified b<$node> if the last of it's child nodes L<match|/atPositionMatch> the specified <@tags> else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
  </a>
  END

    ok  $a->overAllTags_b_c_d;
    ok !$a->overAllTags_b_c;
    ok !$a->overAllTags_b_c_d_e;
    ok  $a->oat_b_c_d;
    ok !$a->oat_B_c_d;

    ok  $a->overFirstTags_b_c_d;
    ok  $a->overFirstTags_b_c;
    ok !$a->overFirstTags_b_c_d_e;
    ok  $a->oft_b_c;
    ok !$a->oft_B_c;

    ok  $a->overLastTags_b_c_d;
    ok  $a->overLastTags_c_d;
    ok !$a->overLastTags_b_c_d_e;
    ok  $a->olt_c_d;
    ok !$a->olt_C_d;
   }


B<olt> is a synonym for L<overLastTags|/overLastTags>.


=head2 matchAfter($node, $re, @context)

Confirm that the string representing the tags following the specified B<$node> matches a regular expression where each pair of tags is separated by a single space. Use L<contentAfterAsTags|/contentAfterAsTags> to visualize these tags.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(qw(b e))->matchAfter  (qr(\Af g\Z));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 matchAfter2($node, $re, @context)

Confirm that the string representing the tags following the specified B<$node> matches a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q? Use L<contentAfterAsTags2|/contentAfterAsTags2> to visualize these tags.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(qw(b e))->matchAfter2 (qr(\A f  g \Z));                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 matchBefore($node, $re, @context)

Confirm that the string representing the tags preceding the specified B<$node> matches a regular expression where each pair of tags is separated by a single space. Use L<contentBeforeAsTags|/contentBeforeAsTags> to visualize these tags.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(qw(b e))->matchBefore (qr(\Ac d\Z));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 matchBefore2($node, $re, @context)

Confirm that the string representing the tags preceding the specified B<$node> matches a regular expression where each pair of tags have two spaces between them and the first tag is preceded by a single space and the last tag is followed by a single space.  This arrangement simplifies the regular expression used to detect combinations like p+ q?  Use L<contentBeforeAsTags2|/contentBeforeAsTags2> to visualize these tags.

     Parameter  Description
  1  $node      Node
  2  $re        Regular expression
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/><e/><f/><g/>
    </b>
  </a>
  END


    ok $x->go(qw(b e))->matchBefore2(qr(\A c  d \Z));                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 parentage($node)

Return a reference to an array of the nodes along the path from the root to the specified B<$Node> inclusive.

     Parameter  Description
  1  $node      Node.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    my ($d, $c, $b) = $a->byList;

    is_deeply $a->go_b_c_d__parentage, [$a, $b, $c, $d];


B<p> is a synonym for L<parentage|/parentage>.


=head2 path($node)

Return a list of strings representing the path to a node from the root of the parse tree which can then be reused by L<go|/go> to retrieve the node as long as the structure of the L<parse|/parse> tree has not changed along the path.

     Parameter  Description
  1  $node      Node.

B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <a       id='a1'>
    <b     id='b1'>
      <c   id='c1'/>
      <c   id='c2'/>
      <d   id='d1'>
        <e id='e1'/>
      </d>
      <c   id='c3'/>
      <c   id='c4'/>
      <d   id='d2'>
        <e id='e2'/>
      </d>
      <c   id='c5'/>
      <c   id='c6'/>
    </b>
  </a>
  END


    is_deeply [$x->go(qw(b d 1 e))->path], [qw(b d 1 e)];                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    $x->by(sub {ok $x->go($_->path) == $_});                                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 pathString($node)

Return a string representing the L<path|/path> to the specified B<$node> from the root of the parse tree.

     Parameter  Description
  1  $node      Node.

B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END


    ok $a->findByNumber(9)->pathString eq 'b c 1 d e';                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 Match

Locate adjacent nodes that match horizontally and vertically

=head3 an($node, $current, @context)

Return the next node if the specified B<$node> has the tag specified by B<$current> and the next node is in the specified B<@context>.

     Parameter  Description
  1  $node      Node
  2  $current   Tag node must match
  3  @context   Optional context of the next node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;

    ok  $e == $d->an_d_e_b_a;

    ok  $f == $e->an_e;

    ok !$f->an_f;


=head3 ap($node, $current, @context)

Return the previous node if the specified B<$node> has the tag specified by B<$current> and the previous node is in the specified B<@context>.

     Parameter  Description
  1  $node      Node
  2  $current   Tag node must match
  3  @context   Optional context of the previous node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;

    ok  $c == $d->ap_d_c_b_a;

    ok  $c == $d->ap_d;

    ok !$c->ap_c;


=head3 apn($node, $prev, $current, @context)

Return (previous node, next node) if the B<$previous> and B<$current> nodes have the specified tags and the next node is in the specified B<@context> else return B<()>.  The specified B<@context> must have at least one element otherwise B<()> is returned.

     Parameter  Description
  1  $node      Current node
  2  $prev      Tag for the previous node
  3  $current   Tag for specified node
  4  @context   Context for the next node.

Use the B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>.  If a context is supplied and
B<$node> is not in this context then this method returns an empty list B<()>
immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;

    is_deeply[$c, $e], [$d->apn_c_d_e_b_a];


=head3 matchesFirst($node, @sequence)

Return the specified B<$node> if its children L<match|/atPositionMatch> the specified <@sequence> forwards from the first child else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @sequence  Sequence.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END
    my ($b, $c, $d, $e, $f) = $a->byList;

    ok   $a->matchesFirst_b_c_d_e_f;
    ok  !$a->matchesFirst_c;
    ok   $a->matchesLast_f_e_d_c_b;
    ok  !$a->matchesLast_f_d;
    ok   $c->matchesNext_d_e_f;
    ok  !$d->matchesNext_e_f_a;
    ok   $e->matchesPrev_d_c_b;
    ok  !$e->matchesPrev_d_c_b_a;
   }


=head3 matchesLast($node, @sequence)

Return the specified B<$node> if its children L<match|/atPositionMatch> the specified <@sequence> backwards from the last child else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @sequence  Sequence.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END
    my ($b, $c, $d, $e, $f) = $a->byList;

    ok   $a->matchesFirst_b_c_d_e_f;
    ok  !$a->matchesFirst_c;
    ok   $a->matchesLast_f_e_d_c_b;
    ok  !$a->matchesLast_f_d;
    ok   $c->matchesNext_d_e_f;
    ok  !$d->matchesNext_e_f_a;
    ok   $e->matchesPrev_d_c_b;
    ok  !$e->matchesPrev_d_c_b_a;
   }


=head3 matchesNext($node, @sequence)

Return the specified B<$node> if its following siblings L<match|/atPositionMatch> the specified <@sequence> else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @sequence  Sequence.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END
    my ($b, $c, $d, $e, $f) = $a->byList;

    ok   $a->matchesFirst_b_c_d_e_f;
    ok  !$a->matchesFirst_c;
    ok   $a->matchesLast_f_e_d_c_b;
    ok  !$a->matchesLast_f_d;
    ok   $c->matchesNext_d_e_f;
    ok  !$d->matchesNext_e_f_a;
    ok   $e->matchesPrev_d_c_b;
    ok  !$e->matchesPrev_d_c_b_a;
   }


=head3 matchesPrev($node, @sequence)

Return the specified B<$node> if the siblings before $node L<match|/atPositionMatch> the specified <@sequence> with the first element of @sequence nearest to $node and the last element furthest else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @sequence  Sequence.

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END
    my ($b, $c, $d, $e, $f) = $a->byList;

    ok   $a->matchesFirst_b_c_d_e_f;
    ok  !$a->matchesFirst_c;
    ok   $a->matchesLast_f_e_d_c_b;
    ok  !$a->matchesLast_f_d;
    ok   $c->matchesNext_d_e_f;
    ok  !$d->matchesNext_e_f_a;
    ok   $e->matchesPrev_d_c_b;
    ok  !$e->matchesPrev_d_c_b_a;
   }


=head2 Child of, Parent of, Sibling of

Nodes that are directly above, below or adjacent to another node.

=head3 parentOf($parent, $child, @context)

Returns the specified B<$parent> node if it is the parent of the specified B<$child> node and the B<$parent> node is in the specified optional context.

     Parameter  Description
  1  $parent    Parent
  2  $child     Child
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END


    ok $e->parentOf($j);                                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 childOf($child, $parent, @context)

Returns the specified B<$child> node if it is a child of the specified B<$parent> node and the B<$child> node is in the specified optional context.

     Parameter  Description
  1  $child     Child
  2  $parent    Parent
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END


    ok $j->childOf($e);                                                             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 succeedingSiblingOf($child, $sibling, @context)

Returns the specified B<$child> node if it has the same parent as B<$sibling> and occurs after B<$sibling> and has the optionally specified context else returns B<undef>.

     Parameter  Description
  1  $child     Child
  2  $sibling   Sibling thought to occur before child
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
    <e/>
   </b>
   <B>
    <C/>
    <D/>
    <E/>
   </B>
  </a>
  END

    my ($c, $d, $e, $b, $C, $D, $B) = $a->byList;

    ok !$e->succeedingSiblingOf($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $e->succeedingSiblingOf($d);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $e->succeedingSiblingOf($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$e->succeedingSiblingOf($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$e->succeedingSiblingOf($B);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$e->succeedingSiblingOf($C);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->precedingSiblingOf($c);
    ok  $c->precedingSiblingOf($d);
    ok  $c->precedingSiblingOf($e);
    ok !$c->precedingSiblingOf($b);
    ok !$c->precedingSiblingOf($B);
    ok !$c->precedingSiblingOf($C);


=head3 precedingSiblingOf($child, $sibling, @context)

Returns the specified B<$child> node if it has the same parent as B<$sibling> and occurs before B<$sibling> and has the optionally specified context else returns B<undef>.

     Parameter  Description
  1  $child     Child
  2  $sibling   Sibling thought to occur after child
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
    <d/>
    <e/>
   </b>
   <B>
    <C/>
    <D/>
    <E/>
   </B>
  </a>
  END

    my ($c, $d, $e, $b, $C, $D, $B) = $a->byList;
    ok !$e->succeedingSiblingOf($e);
    ok  $e->succeedingSiblingOf($d);
    ok  $e->succeedingSiblingOf($c);
    ok !$e->succeedingSiblingOf($b);
    ok !$e->succeedingSiblingOf($B);
    ok !$e->succeedingSiblingOf($C);


    ok !$c->precedingSiblingOf($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $c->precedingSiblingOf($d);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $c->precedingSiblingOf($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->precedingSiblingOf($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->precedingSiblingOf($B);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->precedingSiblingOf($C);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Navigation

Move around in the L<parse|/parse> tree.

=head2 go($node, @path)

Return the node reached from the specified B<$node> via the specified L<path|/path>: (index positionB<?>)B<*> where index is the tag of the next node to be chosen and position is the optional zero based position within the index of those tags under the current node. Position defaults to zero if not specified. Position can also be negative to index back from the top of the index array. B<*> can be used as the last position to retrieve all nodes with the final tag.

     Parameter  Description
  1  $node      Node
  2  @path      Search specification.

B<Example:>


    my $x = Data::Edit::Xml::new(my $s = <<END);
  <aa>
    <a>
      <b/>
        <c id="1"/><c id="2"/><c id="3"/><c id="4"/>
      <d/>
    </a>
  </aa>
  END


    ok $x->go(qw(a c))   ->id == 1;                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->go(qw(a c -2))->id == 3;                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->go(qw(a c *)) == 4;                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok 1234 == join '', map {$_->id} $x->go(qw(a c *));                             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 c($node, $tag)

Return an array of all the nodes with the specified tag below the specified B<$node>.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag.

B<Example:>


   {my $x = Data::Edit::Xml::new(<<END);
  <a>

    <b id="b1"><c id="1"/></b>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    <d id="d1"><c id="2"/></d>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    <e id="e1"><c id="3"/></e>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    <b id="b2"><c id="4"/></b>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    <d id="d2"><c id="5"/></d>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    <e id="e2"><c id="6"/></e>  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  </a>
  END


    is_deeply [map{$_->id} $x->c(q(d))],  [qw(d1 d2)];                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 cText($node)

Return an array of all the text nodes immediately below the specified B<$node>.

     Parameter  Description
  1  $node      Node.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>b1
      <c/>
      b2
      <d/>
      b3
    </b>
    <e>
      <f/>
    </e>
  </a>
  END

    my (undef, $c, undef, $d, undef, $b, $f, $e) = $a->byList;
    is_deeply ["b".."f"], [map {-t $_} ($b, $c, $d, $e, $f)];


    ok  $b->cText == 3;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    my @b = $b->cText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok "b1 b2 b3" eq join " ", map {trim $_->text} @b;

    ok !$e->cText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$a->cText;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 findById($node, $id)

Find a node in the parse tree under the specified B<$node> with the specified B<$id>.

     Parameter  Description
  1  $node      Parse tree
  2  $id        Id desired.

B<Example:>


    ok -p $a eq <<END;
  <a id="i1">
    <b id="i2"/>
    <c id="i3"/>
    <B id="i4">
      <c id="i5"/>
    </B>
    <c id="i6"/>
    <b id="i7"/>
  </a>
  END

    ok -t $a->findById_i4 eq q(B);

    ok -t $a->findById_i5 eq q(c);


=head2 matchesNode($first, $second, @attributes)

Return the B<$first> node if it matches the B<$second> node's tag and the specified B<@attributes> else return B<undef>.

     Parameter    Description
  1  $first       First node
  2  $second      Second node
  3  @attributes  Attributes to match on

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a       id="1">
    <b     id="2"   name="b">
      <c   id="3"   name="c"/>
    </b>
    <c     id="4">
      <b   id="5"   name="b">
        <c id="6"   name="c"/>
      </b>
    </c>
  </a>
  END

    my ($c, $b, $C, $B) = $a->byList;
    ok  $b->id == 2;
    ok  $c->id == 3;
    ok  $B->id == 5;
    ok  $C->id == 6;

    ok  $c->matchesNode($C, qw(name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->matchesNode($C, qw(id name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok  $c->matchesSubTree($C, qw(name));
    ok  $b->matchesSubTree($B, qw(name));
    ok !$c->matchesSubTree($C, qw(id name));
    ok !$b->matchesSubTree($C, qw(name));

    is_deeply [$a->findMatchingSubTrees($b, qw(name))], [$b, $B];
    is_deeply [$a->findMatchingSubTrees($c, qw(name))], [$c, $C];
    is_deeply [$a->findMatchingSubTrees(new(q(<c/>)))], [$c, $C];
    is_deeply [$a->findMatchingSubTrees(new(q(<b><c/></b>)))], [$b, $B];
    is_deeply [$a->findMatchingSubTrees(new(q(<b id="2"><c id="3"/></b>)), q(id))], [$b];
   }


=head2 matchesSubTree($first, $second, @attributes)

Return the B<$first> node if it L<matches|/matchesNode> the B<$second> node and the nodes under the first node match the corresponding nodes under the second node, else return B<undef>.

     Parameter    Description
  1  $first       First node
  2  $second      Second node
  3  @attributes  Attributes to match on

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a       id="1">
    <b     id="2"   name="b">
      <c   id="3"   name="c"/>
    </b>
    <c     id="4">
      <b   id="5"   name="b">
        <c id="6"   name="c"/>
      </b>
    </c>
  </a>
  END

    my ($c, $b, $C, $B) = $a->byList;
    ok  $b->id == 2;
    ok  $c->id == 3;
    ok  $B->id == 5;
    ok  $C->id == 6;
    ok  $c->matchesNode($C, qw(name));
    ok !$c->matchesNode($C, qw(id name));

    ok  $c->matchesSubTree($C, qw(name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $b->matchesSubTree($B, qw(name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$c->matchesSubTree($C, qw(id name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$b->matchesSubTree($C, qw(name));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    is_deeply [$a->findMatchingSubTrees($b, qw(name))], [$b, $B];
    is_deeply [$a->findMatchingSubTrees($c, qw(name))], [$c, $C];
    is_deeply [$a->findMatchingSubTrees(new(q(<c/>)))], [$c, $C];
    is_deeply [$a->findMatchingSubTrees(new(q(<b><c/></b>)))], [$b, $B];
    is_deeply [$a->findMatchingSubTrees(new(q(<b id="2"><c id="3"/></b>)), q(id))], [$b];
   }


=head2 findMatchingSubTrees($node, $subTree, @attributes)

Find nodes in the parse tree whose sub tree matches the specified B<$subTree> excluding any of the specified B<$attributes>.

     Parameter    Description
  1  $node        Parse tree
  2  $subTree     Parse tree to match
  3  @attributes  Attributes to match on

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a       id="1">
    <b     id="2"   name="b">
      <c   id="3"   name="c"/>
    </b>
    <c     id="4">
      <b   id="5"   name="b">
        <c id="6"   name="c"/>
      </b>
    </c>
  </a>
  END

    my ($c, $b, $C, $B) = $a->byList;
    ok  $b->id == 2;
    ok  $c->id == 3;
    ok  $B->id == 5;
    ok  $C->id == 6;
    ok  $c->matchesNode($C, qw(name));
    ok !$c->matchesNode($C, qw(id name));
    ok  $c->matchesSubTree($C, qw(name));
    ok  $b->matchesSubTree($B, qw(name));
    ok !$c->matchesSubTree($C, qw(id name));
    ok !$b->matchesSubTree($C, qw(name));


    is_deeply [$a->findMatchingSubTrees($b, qw(name))], [$b, $B];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    is_deeply [$a->findMatchingSubTrees($c, qw(name))], [$c, $C];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    is_deeply [$a->findMatchingSubTrees(new(q(<c/>)))], [$c, $C];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    is_deeply [$a->findMatchingSubTrees(new(q(<b><c/></b>)))], [$b, $B];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    is_deeply [$a->findMatchingSubTrees(new(q(<b id="2"><c id="3"/></b>)), q(id))], [$b];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   }


=head2 First

Find nodes that are first amongst their siblings.

=head3 first($node, @context)

Return the first node below the specified B<$node> optionally checking the first node's context.  See L<addFirst|/addFirst> to ensure that an expected node is in position.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<firstNonBlank> to skip a (rare) initial blank text CDATA. Use B<firstNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(q(b))->first->id == 13;                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->go(q(b))->first(qw(c b a));                                             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(q(b))->first(qw(b a));                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstn($node, $N, @context)

Return the B<$n>'th first node below the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<firstn(1)> is identical in effect to L<first|/first>.

     Parameter  Description
  1  $node      Node
  2  $N         Number of times to go first
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d/><e/><f/></c></b></a>
  END
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
        <e/>
        <f/>
      </c>
    </b>
  </a>
  END
    ok  -t $a->firstn_0 eq q(a);
    ok  -t $a->firstn_1 eq q(b);
    ok  -t $a->firstn_2 eq q(c);
    ok  -t $a->firstn_3 eq q(d);

    ok  -t $a->firstn_3__nextn_0 eq q(d);
    ok  -t $a->firstn_3__nextn_1 eq q(e);
    ok  -t $a->firstn_3__nextn_2 eq q(f);
   }


=head3 firstText($node, @context)

Return the first node under the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new("<a>AA<b/>BB<c/>CC<d/><e/><f/>DD<g/>HH</a>");
    ok -p $a eq <<END;
  <a>AA
    <b/>
  BB
    <c/>
  CC
    <d/>
    <e/>
    <f/>
  DD
    <g/>
  HH
  </a>
  END
    ok  $a->firstText_a__text eq q(AA);
    ok !$a->go_c__firstText_c_a;
    ok !$a->go_c__firstText_c_b;
    ok  $a->lastText__text eq q(HH);
    ok  $a->lastText_a__text eq q(HH);
    ok !$a->go_c__lastText;
    ok  $a->go_c__nextText_c_a__text eq q(CC);
    ok !$a->go_e__nextText;
    ok  $a->go_c__prevText_c__text eq q(BB);
    ok !$a->go_e__prevText;


=head3 firstTextMatches($node, $match, @context)

Return the first node under the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  $match     Regular expression the text must match
  3  @context   Optional context of specified node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb<c>cc</c>BB
    </b>
  </a>
  END

    my ($bb, $cc, $c, $BB, $b) = $a->byList;

    ok $bb->matchesText(qr(bb));


    ok $b->at_b_a &&  $b->firstTextMatches(qr(bb));                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok                $b->firstTextMatches(qr(bb), qw(b a));                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $c->at_c_b &&  $c->firstTextMatches(qr(cc));                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $c->at_c_b && !$c->firstTextMatches(qr(bb));                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstBy($node, @tags)

Return a list of the first instance of each specified tag encountered in a post-order traversal from the specified B<$node> or a hash of all first instances if no tags are specified.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


     {my %f = $a->firstBy;                                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $f{b}->id == 12;


=head3 firstDown($node, @tags)

Return a list of the first instance of each specified tag encountered in a pre-order traversal from the specified B<$node> or a hash of all first instances if no tags are specified.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>



     {my %f = $a->firstDown;                                                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $f{b}->id == 15;


=head3 firstIn($node, @tags)

Return the first child node matching one of the named tags under the specified parent node.

     Parameter  Description
  1  $node      Parent node
  2  @tags      Child tags to search for.

B<Example:>


    ok $a->prettyStringCDATA eq <<'END';
  <a><CDATA> </CDATA>
      <A/>
  <CDATA>  </CDATA>
      <C/>
  <CDATA>  </CDATA>
      <E/>
  <CDATA>  </CDATA>
      <G/>
  <CDATA>  </CDATA>
  </a>
  END


    ok $a->firstIn(qw(b B c C))->tag eq qq(C);                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstNot($node, @tags)

Return the first child node that does not match any of the named B<@tags> under the specified parent B<$node>. Return B<undef> if there is no such child node.

     Parameter  Description
  1  $node      Parent node
  2  @tags      Child tags to avoid.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $c == $a->firstNot_a_b;


=head3 firstInIndex($node, @context)

Return the specified B<$node> if it is first in its index and optionally L<at|/at> the specified context else B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END


    ok  $a->findByNumber (5)->firstInIndex;                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->findByNumber(7) ->firstInIndex;                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstOf($node, @tags)

Return an array of the nodes that are continuously first under their specified parent node and that match the specified list of tags.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c/><d/><d/><e/><d/><d/><c/></b></a>
  END


    is_deeply [qw(c d d)], [map {-t $_} $a->go(q(b))->firstOf(qw(c d))];            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstWhile($node, @tags)

Go first from the specified B<$node> and continue deeper firstly as long as each first child node matches one of the specified B<@tags>. Return the deepest such node encountered or else return B<undef> if no such node is encountered.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e>
            <f/>
          </e>
        </d>
      </c>
    </b>
    <B>
      <C>
        <D>
          <E>
            <F/>
          </E>
        </D>
      </C>
    </B>
  </a>
  END
    my ($f, $e, $d, $c, $b, $F, $E, $D, $C, $B) = $a->byList;

    ok eval qq(-t \$$_ eq q($_)), $_ for qw(a b c d e f B C D E F);

    ok  $d == $a->firstWhile_a_d_c_b;
    ok  $f == $a->firstWhile_a_d_c_b_e_f_g_h;
    ok !$b->firstWhile_a;

    ok  $e == $a->firstUntil_e_d;
    ok       !$c->firstUntil_c;
    ok       !$b->firstUntil_a;

    ok  $D == $a->lastWhile_a_D_C_B;
    ok  $F == $a->lastWhile_a_D_C_B_E_F_G_H;
    ok !$B->lastWhile_a;

    ok  $E == $a->lastUntil_E_D;
    ok       !$C->lastUntil_C;
    ok !$B->lastUntil_a;


=head3 firstUntil($node, @context)

Go first from the specified B<$node> and continue deeper firstly until a first child node matches the specified B<@context> or return B<undef> if there is no such node.  Return the first child of the specified B<$node> if no B<@context> is specified.

     Parameter  Description
  1  $node      Node
  2  @context   Context to search for.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e>
            <f/>
          </e>
        </d>
      </c>
    </b>
    <B>
      <C>
        <D>
          <E>
            <F/>
          </E>
        </D>
      </C>
    </B>
  </a>
  END
    my ($f, $e, $d, $c, $b, $F, $E, $D, $C, $B) = $a->byList;

    ok eval qq(-t \$$_ eq q($_)), $_ for qw(a b c d e f B C D E F);

    ok  $d == $a->firstWhile_a_d_c_b;
    ok  $f == $a->firstWhile_a_d_c_b_e_f_g_h;
    ok !$b->firstWhile_a;

    ok  $e == $a->firstUntil_e_d;
    ok       !$c->firstUntil_c;
    ok       !$b->firstUntil_a;

    ok  $D == $a->lastWhile_a_D_C_B;
    ok  $F == $a->lastWhile_a_D_C_B_E_F_G_H;
    ok !$B->lastWhile_a;

    ok  $E == $a->lastUntil_E_D;
    ok       !$C->lastUntil_C;
    ok !$B->lastUntil_a;


=head3 firstUntilText($node, @context)

Go first from the specified B<$node> and continue deeper firstly until a text node is encountered whose parent matches the specified B<@context> or return B<undef> if there is no such node.

     Parameter  Description
  1  $node      Node
  2  @context   Context to search for.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      cccc
      <d/>
      <e/>
    </b>
    <f>
      <g/>
      hhhh
      <i/>
    </f>
    <j>
      <k/>
      <l/>
      mmmm
    </j>
  </a>
  END


    ok $a->firstUntilText->text =~ m(cccc)s;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok $a->lastUntilText ->text =~ m(mmmm)s;



=head3 firstContextOf($node, @context)

Return the first node encountered in the specified context in a depth first post-order traversal of the L<parse|/parse> tree.

     Parameter  Description
  1  $node      Node
  2  @context   Array of tags specifying context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <a        id="a1">
    <b1     id="b1">
       <c   id="c1">
         <d id="d1">DD11</d>
         <e id="e1">EE11</e>
      </c>
    </b1>
    <b2     id="b2">
       <c   id="c2">
         <d id="d2">DD22</d>
         <e id="e2">EE22</e>
      </c>
    </b2>
    <b3     id="b3">
       <c   id="c3">
         <d id="d3">DD33</d>
         <e id="e3">EE33</e>
      </c>
    </b3>
  </a>
  END


    ok $x->firstContextOf(qw(d c))         ->id     eq qq(d1);                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->firstContextOf(qw(e c b2))      ->id     eq qq(e2);                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x->firstContextOf(qw(CDATA d c b2))->string eq qq(DD22);                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 firstSibling($node, @context)

Return the first sibling of the specified B<$node> in the optional B<@context> else B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Array of tags specifying context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(qw(b b))->firstSibling->id == 13;                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 Last

Find nodes that are last amongst their siblings.

=head3 last($node, @context)

Return the last node below the specified B<$node> optionally checking the last node's context. See L<addLast|/addLast> to ensure that an expected node is in position.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<lastNonBlank> to skip a (rare) initial blank text CDATA. Use B<lastNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(q(b))->last ->id == 22;                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->go(q(b))->last(qw(g b a));                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(q(b))->last(qw(b a));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(q(b))->last(qw(b a));                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastn($node, $N, @context)

Return the B<$n>'th last node below the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<lastn(1)> is identical in effect to L<last|/last>.

     Parameter  Description
  1  $node      Node
  2  $N         Number of times to go last
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d/><e/><f/></c></b>
     <B><C><D/><E/><F/></C></B></a>
  END
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
        <e/>
        <f/>
      </c>
    </b>
    <B>
      <C>
        <D/>
        <E/>
        <F/>
      </C>
    </B>
  </a>
  END

    ok  -t $a->lastn_0 eq q(a);
    ok  -t $a->lastn_1 eq q(B);
    ok  -t $a->lastn_2 eq q(C);
    ok  -t $a->lastn_3 eq q(F);

    ok  -t $a->lastn_3__prevn_0 eq q(F);
    ok  -t $a->lastn_3__prevn_1 eq q(E);
    ok  -t $a->lastn_3__prevn_2 eq q(D);
   }


=head3 lastText($node, @context)

Return the last node under the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new("<a>AA<b/>BB<c/>CC<d/><e/><f/>DD<g/>HH</a>");
    ok -p $a eq <<END;
  <a>AA
    <b/>
  BB
    <c/>
  CC
    <d/>
    <e/>
    <f/>
  DD
    <g/>
  HH
  </a>
  END
    ok  $a->firstText_a__text eq q(AA);
    ok !$a->go_c__firstText_c_a;
    ok !$a->go_c__firstText_c_b;
    ok  $a->lastText__text eq q(HH);
    ok  $a->lastText_a__text eq q(HH);
    ok !$a->go_c__lastText;
    ok  $a->go_c__nextText_c_a__text eq q(CC);
    ok !$a->go_e__nextText;
    ok  $a->go_c__prevText_c__text eq q(BB);
    ok !$a->go_e__prevText;


=head3 lastTextMatches($node, $match, @context)

Return the last node under the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  $match     Regular expression the text must match
  3  @context   Optional context of specified  node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb<c>cc</c>BB
    </b>
  </a>
  END

    my ($bb, $cc, $c, $BB, $b) = $a->byList;

    ok $BB->matchesText(qr(BB));


    ok $b->at_b_a &&  $b->lastTextMatches(qr(BB));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok                $b->lastTextMatches(qr(BB), qw(b a));                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $c->at_c_b &&  $c->lastTextMatches(qr(cc));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $c->at_c_b && !$c->lastTextMatches(qr(bb));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastBy($node, @tags)

Return a list of the last instance of each specified tag encountered in a post-order traversal from the specified B<$node> or a hash of all last instances if no tags are specified.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


     {my %l = $a->lastBy;                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $l{b}->id == 23;


=head3 lastDown($node, @tags)

Return a list of the last instance of each specified tag encountered in a pre-order traversal from the specified B<$node> or a hash of all last instances if no tags are specified.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>



     {my %l = $a->lastDown;                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok $l{b}->id == 26;


=head3 lastIn($node, @tags)

Return the last child node matching one of the named tags under the specified parent node.

     Parameter  Description
  1  $node      Parent node
  2  @tags      Child tags to search for.

B<Example:>


    ok $a->prettyStringCDATA eq <<'END';
  <a><CDATA> </CDATA>
      <A/>
  <CDATA>  </CDATA>
      <C/>
  <CDATA>  </CDATA>
      <E/>
  <CDATA>  </CDATA>
      <G/>
  <CDATA>  </CDATA>
  </a>
  END


    ok $a->lastIn(qw(e E f F))->tag eq qq(E);                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastNot($node, @tags)

Return the last child node that does not match any of the named B<@tags> under the specified parent B<$node>. Return B<undef> if there is no such child node.

     Parameter  Description
  1  $node      Parent node
  2  @tags      Child tags to avoid.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $d == $a->lastNot_e_f;


=head3 lastOf($node, @tags)

Return an array of the nodes that are continuously last under their specified parent node and that match the specified list of tags.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c/><d/><d/><e/><d/><d/><c/></b></a>
  END


    is_deeply [qw(d d c)], [map {-t $_} $a->go(q(b))->lastOf (qw(c d))];            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastInIndex($node, @context)

Return the specified B<$node> if it is last in its index and optionally L<at|/at> the specified context else B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <e id="4"/>
      </c>
      <d id="5">
        <e id="6"/>
      </d>
      <c id="7">
        <d id="8">
          <e id="9"/>
        </d>
      </c>
      <d id="10">
        <e id="11"/>
      </d>
      <c id="12">
        <d id="13">
          <e id="14"/>
        </d>
      </c>
    </b>
  </a>
  END


    ok  $a->findByNumber(10)->lastInIndex;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->findByNumber(7) ->lastInIndex;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastContextOf($node, @context)

Return the last node encountered in the specified context in a depth first reverse pre-order traversal of the L<parse|/parse> tree.

     Parameter  Description
  1  $node      Node
  2  @context   Array of tags specifying context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <a        id="a1">
    <b1     id="b1">
       <c   id="c1">
         <d id="d1">DD11</d>
         <e id="e1">EE11</e>
      </c>
    </b1>
    <b2     id="b2">
       <c   id="c2">
         <d id="d2">DD22</d>
         <e id="e2">EE22</e>
      </c>
    </b2>
    <b3     id="b3">
       <c   id="c3">
         <d id="d3">DD33</d>
         <e id="e3">EE33</e>
      </c>
    </b3>
  </a>
  END


    ok $x-> lastContextOf(qw(d c))         ->id     eq qq(d3);                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x-> lastContextOf(qw(e c b2     )) ->id     eq qq(e2);                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $x-> lastContextOf(qw(CDATA e c b2))->string eq qq(EE22);                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastSibling($node, @context)

Return the last sibling of the specified B<$node> in the optional B<@context> else B<undef>

     Parameter  Description
  1  $node      Node
  2  @context   Array of tags specifying context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(qw(b b))->lastSibling ->id == 22;                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 lastWhile($node, @tags)

Go last from the specified B<$node> and continue deeper lastly as long as each last child node matches one of the specified B<@tags>. Return the deepest such node encountered or else return B<undef> if no such node is encountered.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e>
            <f/>
          </e>
        </d>
      </c>
    </b>
    <B>
      <C>
        <D>
          <E>
            <F/>
          </E>
        </D>
      </C>
    </B>
  </a>
  END
    my ($f, $e, $d, $c, $b, $F, $E, $D, $C, $B) = $a->byList;

    ok eval qq(-t \$$_ eq q($_)), $_ for qw(a b c d e f B C D E F);

    ok  $d == $a->firstWhile_a_d_c_b;
    ok  $f == $a->firstWhile_a_d_c_b_e_f_g_h;
    ok !$b->firstWhile_a;

    ok  $e == $a->firstUntil_e_d;
    ok       !$c->firstUntil_c;
    ok       !$b->firstUntil_a;

    ok  $D == $a->lastWhile_a_D_C_B;
    ok  $F == $a->lastWhile_a_D_C_B_E_F_G_H;
    ok !$B->lastWhile_a;

    ok  $E == $a->lastUntil_E_D;
    ok       !$C->lastUntil_C;
    ok !$B->lastUntil_a;


=head3 lastUntil($node, @context)

Go last from the specified B<$node> and continue deeper lastly until a last child node matches the specified B<@context> or return B<undef> if there is no such node.  Return the last child of the specified B<$node> if no B<@context> is specified.

     Parameter  Description
  1  $node      Node
  2  @context   Context to search for.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e>
            <f/>
          </e>
        </d>
      </c>
    </b>
    <B>
      <C>
        <D>
          <E>
            <F/>
          </E>
        </D>
      </C>
    </B>
  </a>
  END
    my ($f, $e, $d, $c, $b, $F, $E, $D, $C, $B) = $a->byList;

    ok eval qq(-t \$$_ eq q($_)), $_ for qw(a b c d e f B C D E F);

    ok  $d == $a->firstWhile_a_d_c_b;
    ok  $f == $a->firstWhile_a_d_c_b_e_f_g_h;
    ok !$b->firstWhile_a;

    ok  $e == $a->firstUntil_e_d;
    ok       !$c->firstUntil_c;
    ok       !$b->firstUntil_a;

    ok  $D == $a->lastWhile_a_D_C_B;
    ok  $F == $a->lastWhile_a_D_C_B_E_F_G_H;
    ok !$B->lastWhile_a;

    ok  $E == $a->lastUntil_E_D;
    ok       !$C->lastUntil_C;
    ok !$B->lastUntil_a;


=head3 lastUntilText($node, @context)

Go last from the specified B<$node> and continue deeper lastly until a last child text node matches the specified B<@context> or return B<undef> if there is no such node.

     Parameter  Description
  1  $node      Node
  2  @context   Context to search for.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      cccc
      <d/>
      <e/>
    </b>
    <f>
      <g/>
      hhhh
      <i/>
    </f>
    <j>
      <k/>
      <l/>
      mmmm
    </j>
  </a>
  END

    ok $a->firstUntilText->text =~ m(cccc)s;

    ok $a->lastUntilText ->text =~ m(mmmm)s;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲




=head2 Next

Find sibling nodes after the specified B<$node>.

=head3 next($node, @context)

Return the node next to the specified B<$node>, optionally checking the next node's context. See L<addNext|/addNext> to ensure that an expected node is in position.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<nextNonBlank> to skip a (rare) initial blank text CDATA. Use B<nextNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(qw(b b e))->next ->id == 19;                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->go(qw(b b e))->next(qw(f b b a));                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(qw(b b e))->next(qw(f b a));                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 nextn($node, $N, @context)

Return the B<$n>'th next node after the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<nextn(1)> is identical in effect to L<next|/next>.

     Parameter  Description
  1  $node      Node
  2  $N         Number of times to go next
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d/><e/><f/></c></b></a>
  END
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
        <e/>
        <f/>
      </c>
    </b>
  </a>
  END
    ok  -t $a->firstn_0 eq q(a);
    ok  -t $a->firstn_1 eq q(b);
    ok  -t $a->firstn_2 eq q(c);
    ok  -t $a->firstn_3 eq q(d);

    ok  -t $a->firstn_3__nextn_0 eq q(d);
    ok  -t $a->firstn_3__nextn_1 eq q(e);
    ok  -t $a->firstn_3__nextn_2 eq q(f);
   }


=head3 nextText($node, @context)

Return the node after the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new("<a>AA<b/>BB<c/>CC<d/><e/><f/>DD<g/>HH</a>");
    ok -p $a eq <<END;
  <a>AA
    <b/>
  BB
    <c/>
  CC
    <d/>
    <e/>
    <f/>
  DD
    <g/>
  HH
  </a>
  END
    ok  $a->firstText_a__text eq q(AA);
    ok !$a->go_c__firstText_c_a;
    ok !$a->go_c__firstText_c_b;
    ok  $a->lastText__text eq q(HH);
    ok  $a->lastText_a__text eq q(HH);
    ok !$a->go_c__lastText;
    ok  $a->go_c__nextText_c_a__text eq q(CC);
    ok !$a->go_e__nextText;
    ok  $a->go_c__prevText_c__text eq q(BB);
    ok !$a->go_e__prevText;


=head3 nextTextMatches($node, $match, @context)

Return the next node to the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  $match     Regular expression the text must match
  3  @context   Optional context of specified node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb<c>cc</c>BB
    </b>
  </a>
  END

    ok $cc->matchesText(qr(cc));


    ok $c->at_c_b &&  $c->nextTextMatches(qr(BB));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $b->at_b   && !$b->nextTextMatches(qr(BB));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 nextIn($node, @tags)

Return the nearest sibling after the specified B<$node> that matches one of the named tags or B<undef> if there is no such sibling node.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    ok $a->prettyStringCDATA eq <<'END';
  <a><CDATA> </CDATA>
      <A/>
  <CDATA>  </CDATA>
      <C/>
  <CDATA>  </CDATA>
      <E/>
  <CDATA>  </CDATA>
      <G/>
  <CDATA>  </CDATA>
  </a>
  END


    ok $a->firstIn(qw(b B c C))->nextIn(qw(A G))->tag eq qq(G);                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 nextOn($node, @tags)

Step forwards as far as possible from the specified B<$node> while remaining on nodes with the specified tags. In scalar context return the last such node reached or the starting node if no such steps are possible. In array context return the start node and any following matching nodes.

     Parameter  Description
  1  $node      Start node
  2  @tags      Tags identifying nodes that can be step on to context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="1"/>
      <d id="2"/>
      <c id="3"/>
      <d id="4"/>
      <e id="5"/>
    </b>
  </a>
  END

    ok $c->id == 1;

    ok $e->id == 5;


    ok $c->nextOn(qw(d))  ->id == 2;                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $c->nextOn(qw(c d))->id == 4;                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $e->nextOn(qw(c d))     == $e;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 nextWhile($node, @tags)

Go to the next sibling of the specified B<$node> and continue forwards while the tag of each sibling node matches one of the specified B<@tags>. Return the first sibling node that does not match else B<undef> if there is no such sibling.

     Parameter  Description
  1  $node      Node
  2  @tags      Child tags to avoid.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $e == $b->nextWhile_c_d;


    ok $c == $b->nextWhile;                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 nextUntil($node, @tags)

Go to the next sibling of the specified B<$node> and continue forwards until the tag of a sibling node matches one of the specified B<@tags>. Return the matching sibling node else B<undef> if there is no such sibling node.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to look for.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $e == $b->nextUntil_e_f;


    ok      !$b->nextUntil;                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 Prev

Find sibling nodes before the specified B<$node>.

=head3 prev($node, @context)

Return the node before the specified B<$node>, optionally checking the previous node's context. See L<addLast|/addLast> to ensure that an expected node is in position.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.

Use B<prevNonBlank> to skip a (rare) initial blank text CDATA. Use B<prevNonBlankX> to die rather
then receive a returned B<undef> or false result.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a         id="11">
    <b       id="12">
       <c    id="13"/>
       <d    id="14"/>
       <b    id="15">
          <c id="16"/>
          <d id="17"/>
          <e id="18"/>
          <f id="19"/>
          <g id="20"/>
       </b>
       <f    id="21"/>
       <g    id="22"/>
    </b>
    <b       id="23">
       <c    id="24"/>
       <d    id="25"/>
       <b    id="26">
          <c id="27"/>
          <d id="28"/>
          <e id="29"/>
          <f id="30"/>
          <g id="31"/>
       </b>
       <f    id="32"/>
       <g    id="33"/>
    </b>
  </a>
  END


    ok  $a->go(qw(b b e))->prev ->id == 17;                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a->go(qw(b b e))->prev(qw(d b b a));                                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$a->go(qw(b b e))->prev(qw(d b a));                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 prevText($node, @context)

Return the node before the specified B<$node> if it is in the optional and it is a text node otherwise B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new("<a>AA<b/>BB<c/>CC<d/><e/><f/>DD<g/>HH</a>");
    ok -p $a eq <<END;
  <a>AA
    <b/>
  BB
    <c/>
  CC
    <d/>
    <e/>
    <f/>
  DD
    <g/>
  HH
  </a>
  END
    ok  $a->firstText_a__text eq q(AA);
    ok !$a->go_c__firstText_c_a;
    ok !$a->go_c__firstText_c_b;
    ok  $a->lastText__text eq q(HH);
    ok  $a->lastText_a__text eq q(HH);
    ok !$a->go_c__lastText;
    ok  $a->go_c__nextText_c_a__text eq q(CC);
    ok !$a->go_e__nextText;
    ok  $a->go_c__prevText_c__text eq q(BB);
    ok !$a->go_e__prevText;


=head3 prevn($node, $N, @context)

Return the B<$n>'th previous node after the specified B<$node> optionally checking its context or B<undef> if there is no such node.  B<prevn(1)> is identical in effect to L<prev|/prev>.

     Parameter  Description
  1  $node      Node
  2  $N         Number of times to go prev
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d/><e/><f/></c></b>
     <B><C><D/><E/><F/></C></B></a>
  END
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
        <e/>
        <f/>
      </c>
    </b>
    <B>
      <C>
        <D/>
        <E/>
        <F/>
      </C>
    </B>
  </a>
  END

    ok  -t $a->lastn_0 eq q(a);
    ok  -t $a->lastn_1 eq q(B);
    ok  -t $a->lastn_2 eq q(C);
    ok  -t $a->lastn_3 eq q(F);

    ok  -t $a->lastn_3__prevn_0 eq q(F);
    ok  -t $a->lastn_3__prevn_1 eq q(E);
    ok  -t $a->lastn_3__prevn_2 eq q(D);
   }


=head3 prevTextMatches($node, $match, @context)

Return the previous node to the specified B<$node> if: it is a text mode; its text matches the specified regular expression; the specified B<$node> is in the optional specified context. Else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  $match     Regular expression the text must match
  3  @context   Optional context of specified node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb<c>cc</c>BB
    </b>
  </a>
  END

    ok $cc->matchesText(qr(cc));


    ok $c->at_c_b &&  $c->prevTextMatches(qr(bb));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $b->at_b   && !$b->prevTextMatches(qr(bb));                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 prevIn($node, @tags)

Return the nearest sibling node before the specified B<$node> which matches one of the named tags or B<undef> if there is no such sibling node.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to search for.

B<Example:>


    ok $a->prettyStringCDATA eq <<'END';
  <a><CDATA> </CDATA>
      <A/>
  <CDATA>  </CDATA>
      <C/>
  <CDATA>  </CDATA>
      <E/>
  <CDATA>  </CDATA>
      <G/>
  <CDATA>  </CDATA>
  </a>
  END


    ok $a->lastIn(qw(e E f F))->prevIn(qw(A G))->tag eq qq(A);                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 prevOn($node, @tags)

Step backwards as far as possible while remaining on nodes with the specified tags. In scalar context return the last such node reached or the starting node if no such steps are possible. In array context return the start node and any preceding matching nodes.

     Parameter  Description
  1  $node      Start node
  2  @tags      Tags identifying nodes that can be step on to context.

B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="1"/>
      <d id="2"/>
      <c id="3"/>
      <d id="4"/>
      <e id="5"/>
    </b>
  </a>
  END

    ok $c->id == 1;

    ok $e->id == 5;


    ok $e->prevOn(qw(d))  ->id == 4;                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok $e->prevOn(qw(c d))     == $c;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 prevWhile($node, @tags)

Go to the previous sibling of the specified B<$node> and continue backwards while the tag of each sibling node matches one of the specified B<@tags>. Return the first sibling node that does not match else B<undef> if there is no such sibling.

     Parameter  Description
  1  $node      Parent node
  2  @tags      Child tags to avoid.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $c == $f->prevWhile_e_d;


    ok $b == $c->prevWhile;                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 prevUntil($node, @tags)

Go to the previous sibling of the specified B<$node> and continue backwards until the tag of a sibling node matches one of the specified B<@tags>. Return the matching sibling node else B<undef> if there is no such sibling node.

     Parameter  Description
  1  $node      Node
  2  @tags      Tags to look for.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;

    ok $b == $f->prevUntil_a_b;


    ok      !$c->prevUntil;                                                         # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 Up

Methods for moving up the L<parse|/parse> tree from a node.

=head3 top($node, @context)

Return the top of the parse tree containing the current B<$node> after optionally checking that the $node is in the optional B<@context>.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d/></c></b></a>
  END

    $a->by(sub
     {ok $a == $_->root;

      ok $a == $_->top;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

     });


=head3 up($node, @context)

Return the parent of the current node optionally checking the parent node's context or return B<undef> if the specified B<$node> is the root of the L<parse|/parse> tree.   See L<addWrapWith|/addWrapWith> to ensure that an expected node is in position.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context of parent.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><b><b><b><b><c/></b></b></b></b></c></b></a>
  END

    $a->numberTree;
    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <b id="4">
          <b id="5">
            <b id="6">
              <b id="7">
                <c id="8"/>
              </b>
            </b>
          </b>
        </b>
      </c>
    </b>
  </a>
  END

    my $c = $a->findByNumber(8);
    ok -t $c eq q(c);
    ok  $c->up_b__number == 7;
    ok  $c->upn_2__number == 6;
    ok  $c->upWhile_b__number == 4;
    ok  $c->upWhile_a_b__number == 4;
    ok  $c->upWhile_b_c__number == 2;

    ok  $c->upUntil__number == 8;
    ok  $c->upUntil_b_c__number == 4;
   }


=head3 upn($node, $levels, @context)

Go up the specified number of levels from the specified B<$node> and return the node reached optionally checking the parent node's context or B<undef> if there is no such node.L<upn(1)|/up> is identical in effect to L<up|/up>.  Or use L<ancestry|/ancestry> to get the path back to the root node.

     Parameter  Description
  1  $node      Start node
  2  $levels    Number of levels to go up
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><b><b><b><b><c/></b></b></b></b></c></b></a>
  END

    $a->numberTree;
    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <b id="4">
          <b id="5">
            <b id="6">
              <b id="7">
                <c id="8"/>
              </b>
            </b>
          </b>
        </b>
      </c>
    </b>
  </a>
  END

    my $c = $a->findByNumber(8);
    ok -t $c eq q(c);
    ok  $c->up_b__number == 7;
    ok  $c->upn_2__number == 6;
    ok  $c->upWhile_b__number == 4;
    ok  $c->upWhile_a_b__number == 4;
    ok  $c->upWhile_b_c__number == 2;

    ok  $c->upUntil__number == 8;
    ok  $c->upUntil_b_c__number == 4;
   }

  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d><e/></d></c></b></a>
  END

    my ($e, $d, $c, $b) = $a->byList;

    ok $e = $e->upn_0_e_d_c_b_a;
    ok $d = $e->upn_1_d_c_b_a;
    ok $c = $e->upn_2_c_b_a;
    ok $b = $e->upn_3_b_a;
    ok $a = $e->upn_4_a;
    ok     !$e->upn_5;

    is_deeply [$e, $d, $c, $b, $a], [$e->ancestry];
   }

   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><d><e/></d></c></b></a>
  END

    my ($e, $d, $c, $b) = $a->byList;

    ok $e = $e->upn_0_e_d_c_b_a;

    ok $d = $e->upn_1_d_c_b_a;

    ok $c = $e->upn_2_c_b_a;

    ok $b = $e->upn_3_b_a;

    ok $a = $e->upn_4_a;

    ok     !$e->upn_5;


=head3 upWhile($node, @tags)

Go up one level from the specified B<$node> and then continue up while each node matches on of the specified <@tags>. Return the last matching node or B<undef> if no node matched any of the specified B<@tags>.

     Parameter  Description
  1  $node      Start node
  2  @tags      Tags to match

B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><b><b><b><b><c/></b></b></b></b></c></b></a>
  END

    $a->numberTree;
    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <b id="4">
          <b id="5">
            <b id="6">
              <b id="7">
                <c id="8"/>
              </b>
            </b>
          </b>
        </b>
      </c>
    </b>
  </a>
  END

    my $c = $a->findByNumber(8);
    ok -t $c eq q(c);
    ok  $c->up_b__number == 7;
    ok  $c->upn_2__number == 6;
    ok  $c->upWhile_b__number == 4;
    ok  $c->upWhile_a_b__number == 4;
    ok  $c->upWhile_b_c__number == 2;

    ok  $c->upUntil__number == 8;
    ok  $c->upUntil_b_c__number == 4;
   }


=head3 upWhileFirst($node, @context)

Move up from the specified B<$node> as long as each node is a first node or return B<undef> if the specified B<$node> is not a first node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $h == $i->upWhileFirst;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a == $c->upWhileFirst;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->upWhileFirst;                                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 upWhileLast($node, @context)

Move up from the specified B<$node> as long as each node is a last node or return B<undef> if the specified B<$node> is not a last node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $j == $j->upWhileLast;                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $a == $l->upWhileLast;                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->upWhileLast;                                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $i == $k->upUntilLast;


=head3 upWhileIsOnlyChild($node, @context)

Move up from the specified B<$node> as long as each node is an only child or return B<undef> if the specified B<$node> is not an only child.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $h == $i->upWhileIsOnlyChild;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $j == $j->upWhileIsOnlyChild;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !$d->upWhileIsOnlyChild;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 upUntil($node, @context)

Find the first node going up from B<$node> that matches the specified B<@context>. The first such node will be the specified $node if no @context is specified or the specified $node matches the specified @context>.

     Parameter  Description
  1  $node      Start node
  2  @context   Context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(<<END);
  <a><b><c><b><b><b><b><c/></b></b></b></b></c></b></a>
  END

    $a->numberTree;
    ok -z $a eq <<END;
  <a id="1">
    <b id="2">
      <c id="3">
        <b id="4">
          <b id="5">
            <b id="6">
              <b id="7">
                <c id="8"/>
              </b>
            </b>
          </b>
        </b>
      </c>
    </b>
  </a>
  END

    my $c = $a->findByNumber(8);
    ok -t $c eq q(c);
    ok  $c->up_b__number == 7;
    ok  $c->upn_2__number == 6;
    ok  $c->upWhile_b__number == 4;
    ok  $c->upWhile_a_b__number == 4;
    ok  $c->upWhile_b_c__number == 2;

    ok  $c->upUntil__number == 8;
    ok  $c->upUntil_b_c__number == 4;
   }


=head3 upUntilFirst($node, @context)

Move up from the specified B<$node> until we reach the root or a first node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $b == $d->upUntilFirst;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 upUntilLast($node, @context)

Move up from the specified B<$node> until we reach the root or a last node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


=head3 upUntilIsOnlyChild($node, @context)

Move up from the specified B<$node> until we reach the root or another only child.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $i == $k->upUntilIsOnlyChild;                                               # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 upThru($node, @tags)

Go up the specified path from the specified B<$node> returning the node at the top or B<undef> if no such node exists.

     Parameter  Description
  1  $node      Start node
  2  @tags      Tags identifying path.

B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d>
        <e/>
        <f/>
      </d>
    </b>
  </a>
  END

    my ($c, $e, $f, $d, $b) = $a->byList;

    ok -t $f                eq q(f);


    ok -t $f->upThru        eq q(f);                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok -t $f->upThru(qw(d)) eq q(d);                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok -t eval{$f->upThru(qw(d))->last->prev} eq q(e);                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok !  eval{$f->upThru(qw(d b))->next};                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 down

Methods for moving down through the L<parse|/parse> tree from a node.

=head3 downWhileFirst($node, @context)

Move down from the specified B<$node> as long as each lower node is a first node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $k == $g->downWhileFirst;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $c == $a->downWhileFirst;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $c == $c->downWhileFirst;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok       !$d->downWhileFirst;                                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



B<firstLeaf> is a synonym for L<downWhileFirst|/downWhileFirst>.


=head3 downWhileLast($node, @context)

Move down from the specified B<$node> as long as each lower node is a last node.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e>
        <j/>
      </e>
      <f/>
    </b>
    <g>
      <h>
        <i>
          <k/>
          <l/>
        </i>
      </h>
    </g>
  </a>
  END

    my ($c, $d, $j, $e, $f, $b, $k, $l, $i, $h, $g) = $a->byList;


    ok  $l == $a->downWhileLast;                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $l == $g->downWhileLast;                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok       !$d->downWhileLast;                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



B<lastLeaf> is a synonym for L<downWhileLast|/downWhileLast>.


=head3 downWhileHasSingleChild($node, @context)

Move down from the specified B<$node> as long as it has a single child else return undef.

     Parameter  Description
  1  $node      Start node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>



    ok  $h == $g->downWhileHasSingleChild;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok  $h == $h->downWhileHasSingleChild;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



    ok       !$i->downWhileHasSingleChild;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Editing

Edit the data in the L<parse|/parse> tree and change the structure of the L<parse|/parse> tree by L<wrapping and unwrapping|/Wrap and unwrap> nodes, by L<replacing|/Replace> nodes, by L<cutting and pasting|/Cut and Put> nodes, by L<concatenating|/Fusion> nodes, by L<splitting|/Fission> nodes, by adding new L<text|/Put as text> nodes or L<swapping|/swap> nodes.

=head2 change($node, $name, @context)

Change the name of the specified B<$node>, optionally  confirming that the B<$node> is in a specified context and return the B<$node>.

     Parameter  Description
  1  $node      Node
  2  $name      New name
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new('<a/>');


    $a->change(qq(b));                                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $a eq '<b/>';


B<cc> is a synonym for L<change|/change>.


B<ck> is a synonym for L<change|/change>.


=head2 changeKids($node, $name, @context)

Change the names of all the immediate children of the specified B<$node>, if they match the optional B<@context>, to the specified B<$tag> and return the B<$node>.

     Parameter  Description
  1  $node      Node
  2  $name      New name
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
    <f>
      <g/>
      <h>
        <i/>
      </h>
    </f>
    <j>
      <k/>
      <l>
        <m/>
      </l>
    </j>
  </a>
  END


    $a->changeKids(q(B), qr(\A(b|j|k|l|m)\Z), q(a));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <B>
      <c/>
      <d/>
      <e/>
    </B>
    <f>
      <g/>
      <h>
        <i/>
      </h>
    </f>
    <B>
      <k/>
      <l>
        <m/>
      </l>
    </B>
  </a>
  END

    $a->go_f__changeKids_F;

    ok -p $a eq <<END;
  <a>
    <B>
      <c/>
      <d/>
      <e/>
    </B>
    <f>
      <F/>
      <F>
        <i/>
      </F>
    </f>
    <B>
      <k/>
      <l>
        <m/>
      </l>
    </B>
  </a>
  END


=head2 changeText($node, $rf, $rt, $flags, @context)

Change the content of the specified text B<$node> that matches a regular expression B<$rf> presented as a string to a string B<$rt> in the optional B<@context> and return the specified $node else return B<undef>. For a more efficient non unitary method see L<editText>.

     Parameter  Description
  1  $node      Text node
  2  $rf        From re
  3  $rt        To string
  4  $flags     Re flags
  5  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>abcdcba</a>
  END


=head2 changeTextToSpace($node, $re, @context)

Change each instance of the content of the specified text B<$node> that matches a regular expression B<$re> to one space and return the specified $node else return B<undef>.

     Parameter  Description
  1  $node      Text node
  2  $re        Regular expression
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>abcdcba</a>
  END


=head2 dupPutNext($tree, @context)

Duplicate the specified B<$tree> in the optional B<@context>, place the new tree next after $tree and return the root of the new tree on success else B<undef>.

     Parameter  Description
  1  $tree      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    $a->first__dupPutNext_b;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    $a->last__dupPutPrev_d;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
    <d>
      <e/>
    </d>
  </a>
  END


B<r> is a synonym for L<dupPutNext|/dupPutNext>.


=head2 dupPutPrev($tree, @context)

Duplicate the specified B<$tree> in the optional B<@context>, place the new tree before $tree and return the root of the new tree on success else B<undef>.

     Parameter  Description
  1  $tree      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    $a->first__dupPutNext_b;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
  </a>
  END

    $a->last__dupPutPrev_d;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
    </d>
    <d>
      <e/>
    </d>
  </a>
  END


=head2 dupPutNextN($node, $N, @context)

Duplicate the specified B<$tree> B<$N> times in the optional B<@context>, placing each copy after $tree and return the last new node created on success else B<undef>.

     Parameter  Description
  1  $node      Node
  2  $N         Number of duplications
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END

    $a->go_b__dupPutNextN_2;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END


B<rN> is a synonym for L<dupPutNextN|/dupPutNextN>.


=head2 setSelectionStart($node, @context)

Set the selection to start at the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>..

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<ss> is a synonym for L<setSelectionStart|/setSelectionStart>.


=head2 setSelectionEnd($node, @context)

Set the selection to end at the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>..

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<se> is a synonym for L<setSelectionEnd|/setSelectionEnd>.


=head2 moveSelectionFirst($node, @context)

Move the current selection (if there is one) so that it is first under the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<mf> is a synonym for L<moveSelectionFirst|/moveSelectionFirst>.


=head2 moveSelectionAfter($node, @context)

Move the current selection (if there is one) after the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<ma> is a synonym for L<moveSelectionAfter|/moveSelectionAfter>.


=head2 moveSelectionBefore($node, @context)

Move the current selection (if there is one) before the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<mb> is a synonym for L<moveSelectionBefore|/moveSelectionBefore>.


=head2 moveSelectionLast($node, @context)

Move the current selection (if there is one) so that it is last under the specified B<$node> in the optional B<@context> and return the specified B<$node> on success else B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
  </a>
  END
    $a->go_b_1__setSelectionStart;
    $a->go_d__setSelectionEnd;
    $a->first__moveSelectionBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d>
      <e/>
      <f/>
      <g/>
    </d>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_d_e__setSelectionStart;
    $a->go_d_g__setSelectionEnd;
    $a->go_b_c__moveSelectionAfter;
    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <e/>
      <f/>
      <g/>
    </b>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->go_b_e__setSelectionStart;
    $a->go_b_f__setSelectionEnd;
    $a->last__moveSelectionFirst;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
    </b>
    <d/>
    <b>
      <e><!-- start --></e>
      <f><!-- end --></f>
      <c/>
    </b>
  </a>
  END

    $a->go_d__setSelectionStart;
    $a->last__setSelectionEnd;
    $a->go_b__moveSelectionLast;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <g/>
      <d/>
      <b>
        <e><!-- start --></e>
        <f><!-- end --></f>
        <c/>
      </b>
    </b>
  </a>
  END


B<ml> is a synonym for L<moveSelectionLast|/moveSelectionLast>.


=head2 Cut and Put

Cut and paste nodes in the L<parse|/parse> tree.

=head3 cut($node, @context)

Cut out and return the specified B<$node> so that it can be reinserted else where in the L<parse|/parse> tree.

     Parameter  Description
  1  $node      Node to cut out
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"/>
    </b>
  </a>
  END


      my $c = $a->go(qw(b c))->cut;                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb"/>
  </a>
  END


B<x> is a synonym for L<cut|/cut>.


=head3 cutFirst($node, @context)

Cut out the first node below the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return $node on success else return B<undef>. The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 putCutOutFirst($node, @context)

Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it first under the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;


    $a->putCutOutFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 cutLast($node, @context)

Cut out the last node below the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return $node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 putCutOutLast($node, @context)

Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it last under the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END


    $a->putCutOutLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 cutNext($node, @context)

Cut out the next node beyond the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return the current node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 putCutOutNext($node, @context)

Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it next after the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END


    $d->putCutOutNext;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 cutPrev($node, @context)

Cut out the previous node before the specified B<$node> if it is in the optional B<@context> and push it on the cut out stack. Return the current node on success else return B<undef>.  The cut out node can be reinserted using one of the putCutOut(First|Last|Next|Prev) methods.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutPrev;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 putCutOutPrev($node, @context)

Pop the last node placed on the cut out stack by one of the cut(First|Last|Next|Prev) methods and place it before the specified B<$node> if $node is in the optional B<@context>. Return $node on success else return B<undef>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    my ($b, $c, $d, $e, $f) = $a->byList;
    $b->cutNext_c_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <e/>
    <f/>
  </a>
  END

    $f->cutPrev_e_a;
    ok -p $a eq <<END;
  <a>
    <b/>
    <d/>
    <f/>
  </a>
  END
    $a->cutFirst_b;
    ok -p $a eq <<END;
  <a>
    <d/>
    <f/>
  </a>
  END
    $a->cutLast_f;
    ok -p $a eq <<END;
  <a>
    <d/>
  </a>
  END

    ok @saveLastCutOut == 4;

    $a->putCutOutFirst;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
  </a>
  END

    $a->putCutOutLast;
    ok -p $a eq <<END;
  <a>
    <f/>
    <d/>
    <b/>
  </a>
  END


    $d->putCutOutPrev;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <b/>
  </a>
  END

    $d->putCutOutNext;
    ok -p $a eq <<END;
  <a>
    <f/>
    <e/>
    <d/>
    <c/>
    <b/>
  </a>
  END



=head3 cutIfEmpty($node, @context)

Cut out and return the specified B<$node> so that it can be reinserted else where in the L<parse|/parse> tree if it is empty.

     Parameter  Description
  1  $node      Node to cut out
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb</b>
    <c/>
    <d></d>
    <e>ee</e>
  </a>
  END

    my (undef, $b, $c, $d, undef, $e) = $a->byList;
    is_deeply [q(b)..q(e)], [map {-t $_} ($b, $c, $d, $e)];


    $c->cutIfEmpty;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>bb</b>
    <d/>
    <e>ee</e>
  </a>
  END


    $d->cutIfEmpty;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>bb</b>
    <e>ee</e>
  </a>
  END


    ok !$_->cutIfEmpty for $a, $b, $e;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head3 deleteContent($node, @context)

Delete the content of the specified B<$node>.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>bb<c>cc</c>BB
    </b>
  </a>
  END


    $b->deleteContent;                                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END


=head3 putFirst($old, $new, @context)

Place a L<cut out|/cut> or L<new|/new> node at the front of the content of the specified B<$node> and return the new node. See L<putFirstCut|/putFirstCut> to cut and put first in one operation. See L<addFirst|/addFirst> to perform this operation conditionally.

     Parameter  Description
  1  $old       Original node
  2  $new       New node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"/>
    </b>
  </a>
  END

      my $c = $a->go(qw(b c))->cut;


      $a->putFirst($c);                                                             # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a id="aa">
    <c id="cc"/>
    <b id="bb"/>
  </a>
  END


=head3 putFirstCut($first, $second, @context)

Cut out the B<$second> node, place it first under the B<$first> node and return the B<$second> node.

     Parameter  Description
  1  $first     First node
  2  $second    Second node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END

    my ($c, $d, $b) = $a->byList;


    $c->putFirstCut($d, qw(c b a));                                                 # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


=head3 putLast($old, $new, @context)

Place a L<cut out|/cut> or L<new|/new> node last in the content of the specified B<$node> and return the new node.  See L<putLastCut|/putLastCut> to cut and put last in one operation.  See L<addLast|/addLast> to perform this operation conditionally.

     Parameter  Description
  1  $old       Original node
  2  $new       New node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a id="aa">
    <c id="cc"/>
    <b id="bb"/>
  </a>
  END


      $a->putLast($a->go(qw(c))->cut);                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb"/>
    <c id="cc"/>
  </a>
  END


=head3 putLastCut($first, $second, @context)

Cut out the B<$second> node, place it last under the B<$first> node and return the B<$second> node.

     Parameter  Description
  1  $first     First node
  2  $second    Second node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END

    my ($c, $d, $b) = $a->byList;


    $a->putLastCut($d, qw(a));                                                      # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d/>
  </a>
  END


=head3 putNext($old, $new, @context)

Place a L<cut out|/cut> or L<new|/new> node just after the specified B<$node> and return the new node. See L<putNextCut|/putNextCut> to cut and put next in one operation.  See L<addNext|/addNext> to perform this operation conditionally.

     Parameter  Description
  1  $old       Original node
  2  $new       New node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb"/>
    <c id="cc"/>
  </a>
  END


      $a->go(qw(c))->putNext($a->go(q(b))->cut);                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a id="aa">
    <c id="cc"/>
    <b id="bb"/>
  </a>
  END


=head3 putNextCut($first, $second, @context)

Cut out the B<$second> node, place it after the B<$first> node and return the B<$second> node.

     Parameter  Description
  1  $first     First node
  2  $second    Second node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END

    my ($c, $d, $b) = $a->byList;


    $d->putNextCut($c, qw(d b a));                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <d/>
      <c/>
    </b>
  </a>
  END


=head3 putPrev($old, $new, @context)

Place a L<cut out|/cut> or L<new|/new> node just before the specified B<$node> and return the new node.  See L<putPrevCut|/putPrevCut> to cut and put previous in one operation.  See L<addPrev|/addPrev> to perform this operation conditionally.

     Parameter  Description
  1  $old       Original node
  2  $new       New node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a id="aa">
    <c id="cc"/>
    <b id="bb"/>
  </a>
  END


      $a->go(qw(c))->putPrev($a->go(q(b))->cut);                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a id="aa">
    <b id="bb"/>
    <c id="cc"/>
  </a>
  END


=head3 putPrevCut($first, $second, @context)

Cut out the B<$second> node, place it before the B<$first> node and return the B<$second> node.

     Parameter  Description
  1  $first     First node
  2  $second    Second node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END

    my ($c, $d, $b) = $a->byList;


    $c->putPrevCut($d, qw(c b a));                                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <d/>
      <c/>
    </b>
  </a>
  END


=head2 Put Siblings or Contents

Move a node and its siblings up and down the parse tree.

=head3 putSiblingsFirst($node, @context)

Move the siblings preceding the specified B<$node> in the optional B<@context> down one level and place them first under the specified B<$node> preceding any existing content.  Return the specified B<$node>.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/><f/>
  </a>
  END

   my ($b, $c, $d, $e, $f) = $a->byList;


   $d->putSiblingsFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok -p $a eq <<END;
  <a>
    <d>
      <b/>
      <c/>
    </d>
    <e/>
    <f/>
  </a>
  END


=head3 putSiblingsLast($node, @context)

Move the siblings following the specified B<$node> in the optional B<@context> down one level so that they are last under the specified B<$node> following any existing content. Return the specified B<$node>.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/><c/><d/><e/><f/>
  </a>
  END

   my ($b, $c, $d, $e, $f) = $a->byList;


   $d->putSiblingsLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d>
      <e/>
      <f/>
    </d>
  </a>
  END


=head3 putSiblingsAfterParent($node, @context)

Move the specified B<$node> and its following siblings up one level and place them after the parent of the specified B<$node> if the specified B<$node> is in the optional B<@context>.  Return the specified B<$node> if the move was made successfully, else confess that the specified move is not possible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>d<e/>
    </b>
  </a>
  END

   my ($c, $d, $e, $b) = $a->byList;


   $d->putSiblingsAfterParent;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
  d
    <e/>
  </a>
  END


=head3 putSiblingsBeforeParent($node, @context)

Move the specified B<$node> and its preceding siblings up one level and place them before the parent of the specified B<$node> if the specified B<$node> is in the optional B<@context>.  Return the specified B<$node> if the move was made successfully, else confess that the specified move is not possible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>d<e/>
    </b>
  </a>
  END

   my ($c, $d, $e, $b) = $a->byList;


   $d->putSiblingsBeforeParent;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   ok -p $a eq <<END;
  <a>
    <c/>
  d
    <b>
      <e/>
    </b>
  </a>
  END


=head3 putContentAfter($node, @context)

Move the content of the specified B<$node> and place it after that node if that node is in the optional B<@context>.  Return the specified B<$node> or confess if the move is not possible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/>
    </b>
  </a>
  END

   my ($c, $d, $b) = $a->byList;


   $b->putContentAfter;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
  </a>
  END


=head3 putContentBefore($node, @context)

Move the content of the specified B<$node> and place it before that node if that node is in the optional B<@context>.  Return the specified B<$node> or confess if the move is not possible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/><d/>
    </b>
  </a>
  END

   my ($c, $d, $b) = $a->byList;


   $b->putContentBefore;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   ok -p $a eq <<END;
  <a>
    <c/>
    <d/>
    <b/>
  </a>
  END


=head2 Move

Move nodes around in the L<parse|/parse> tree by cutting and pasting them.

=head3 moveFirst($node, @context)

Move the specified node so that is is the first sibling under its parent. Returns the specified $node on success otherwise B<undef>.

     Parameter  Description
  1  $node      Node to  be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b/><c/><d/></a>
  END

    my ($b, $c, $d) = $a->byList;

    $d->moveFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $b->moveLast;

    ok -p $a eq <<END;
  <a>
    <d/>
    <c/>
    <b/>
  </a>
  END


=head3 moveLast($node, @context)

Move the specified node so that is is the last sibling under its parent. Returns the specified $node on success otherwise B<undef>.

     Parameter  Description
  1  $node      Node to  be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a><b/><c/><d/></a>
  END

    my ($b, $c, $d) = $a->byList;
    $d->moveFirst;

    $b->moveLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <d/>
    <c/>
    <b/>
  </a>
  END


=head3 moveStartFirst($node, @context)

Move the start of a B<$node> to contain all of its preceding siblings as children. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b/>
   <c/>
   <d>
    <e/>
   </d>
  </a>
  END

    my ($b, $c, $e, $d) = $a->byList;


    $d->moveStartFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <d>
      <b/>
      <c/>
      <e/>
    </d>
  </a>
  END


=head3 moveStartAfter($node, $to, @context)

Move the start end of a B<$node> to just after the specified B<$target> node assuming that the B<$target> node is either a preceding sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  $to        Target node
  3  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b/>
   <c/>
   <d>
    <e/>
   </d>
  </a>
  END

    my ($b, $c, $e, $d) = $a->byList;


    $d->moveStartAfter($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <d>
      <c/>
      <e/>
    </d>
  </a>
  END


    $d->moveStartAfter($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -t $d eq q(d);
    ok -t $c eq q(c);
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d>
      <e/>
    </d>
  </a>
  END


    $d->moveStartAfter($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
    <d/>
  </a>
  END


=head3 moveStartBefore($node, $to, @context)

Move the start of a B<$node> to just before the specified B<$target> node assuming that the B<$target> node is either a preceding sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose start should be moved
  2  $to        Target node
  3  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b/>
   <c/>
   <d>
    <e/>
   </d>
  </a>
  END

    my ($b, $c, $e, $d) = $a->byList;


    $d->moveStartBefore($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <d>
      <c/>
      <e/>
    </d>
  </a>
  END


    $d->moveStartBefore($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -t $d eq q(d);
    ok -t $e eq q(e);
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d>
      <e/>
    </d>
  </a>
  END

    $d->moveEndBefore($e);
    ok -t $d eq q(d);
    ok -t $e eq q(e);
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END


=head3 moveEndLast($node, @context)

Move the end of a B<$node> to contain all of its following siblings as children. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose end should be moved
  2  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
   </b>
   <d/>
   <e/>
  </a>
  END

    my ($c, $b, $d, $e) = $a->byList;


    $b->moveEndLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END


=head3 moveEndAfter($node, $to, @context)

Move the end of a B<$node> to just after the specified B<$target> node assuming that the B<$target> node is either a subsequent sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose end should be moved
  2  $to        Target node
  3  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
   </b>
   <d/>
   <e/>
  </a>
  END

    my ($c,  $b, $d, $e) = $a->byList;


    $b->moveEndAfter($d);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
    </b>
    <e/>
  </a>
  END


    $b->moveEndAfter($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d/>
    <e/>
  </a>
  END


=head3 moveEndBefore($node, $to, @context)

Move the end of a B<$node> to just before the specified B<$target> node assuming that the B<$target> node is either a subsequent sibling or a child of B<$node>. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $node      Node whose end should be moved
  2  $to        Target node
  3  @context   Optional context of node.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
   <b>
    <c/>
   </b>
   <d/>
   <e/>
  </a>
  END

    my ($c,  $b, $d, $e) = $a->byList;


    $b->moveEndBefore($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
    </b>
    <e/>
  </a>
  END


    $b->moveEndBefore($d);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d/>
    <e/>
  </a>
  END


=head3 putNextFirstCut($node, @context)

Move the specified B<$node> so it is first in the next node with the optional context. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c>
      <d/>
    </c>
  </a>
  END

    my ($b, $d, $c) = $a->byList;

    ok !$b->putNextFirstCut_c_x;

    $b->putNextFirstCut_c_a;
    ok  -p $a eq <<END;
  <a>
    <c>
      <b/>
      <d/>
    </c>
  </a>
  END


=head3 putNextFirstCut2($node, @context)

Move the specified B<$node> so it is first in the first node with the specified optional B<@context> of the next node. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <c>
      <d/>
    </c>
  </a>
  END

    my ($b, $d, $c) = $a->byList;

    ok !$b->putNextFirstCut2_c_a;

    $b->putNextFirstCut2_d_c_a;

    ok  -p $a eq <<END;
  <a>
    <c>
      <d>
        <b/>
      </d>
    </c>
  </a>
  END


=head3 putPrevLastCut($node, @context)

Move the specified B<$node> so it is last in the preceding node with the optional context. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
     <d/>
  </a>
  END

    my ($c, $b, $d) = $a->byList;

    ok !$d->putPrevLastCut_b_c;

    $d->putPrevLastCut_b_a;
    ok  -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END


=head3 putPrevLastCut2($node, @context)

Move the specified B<$node> so it is last in the last node with the specified optional context of the preceding node. Return the specified $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
     <d/>
  </a>
  END

    my ($c, $b, $d) = $a->byList;

    ok !$d->putPrevLastCut2_c_a;

    $d->putPrevLastCut2_c_b_a;
    ok  -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


=head3 putUpNextCut($node, @context)

Move the specified B<$node>, in the optional B<@context>, which must be last under its parent, so that it is next after its parent. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    $a->by(sub{$_->putUpNextCut_d_c_b_a});

    ok -p $a eq <<END
  <a>
    <b>
      <c/>
      <d/>
    </b>
  </a>
  END


=head3 putUpNextCut2($node, @context)

Move the specified B<$node>, in the optional B<@context>, if $node is last after its parent which must also be last under its parent, so that $node is next after its grandparent. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    $a->by(sub{$_->putUpNextCut2_d_c_b_a});

    ok -p $a eq <<END
  <a>
    <b>
      <c/>
    </b>
    <d/>
  </a>
  END


=head3 putUpPrevCut($node, @context)

Move the specified B<$node> in the optional B<@context>, if $node is first under its parent, so that it is prior to its parent. Return $node on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    $a->by(sub{$_->putUpPrevCut_d_c_b_a});

    ok -p $a eq <<END
  <a>
    <b>
      <d/>
      <c/>
    </b>
  </a>
  END


=head3 putUpPrevCut2($node, @context)

Move the specified B<$node>, in the optional B<@context>, if $node is first under its parent which must also be first under its parent, so that $node is prior to its grandparent. Return the specified B<$node> on success else return B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    $a->by(sub{$_->putUpPrevCut2_d_c_b_a});

    ok -p $a eq <<END
  <a>
    <d/>
    <b>
      <c/>
    </b>
  </a>
  END


=head3 moveBlockFirst($start, $end, $parent, @context)

Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $end       Last sibling
  3  $parent    Parent to move first under
  4  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
    <f>
      <g/>
    </f>
    <h/>
  </x>
  END
    my ($a, $b, $c, $d, $e, $g, $f, $h) = $x->byList;


    $c->moveBlockFirst($d, $f, qw(c x));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a/>
    <b/>
    <e/>
    <f>
      <c/>
      <d/>
      <g/>
    </f>
    <h/>
  </x>
  END


=head3 moveBlockLast($start, $end, $parent, @context)

Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $end       Last sibling
  3  $parent    Parent to move last under
  4  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
    <d/>
    <e/>
    <f>
      <g/>
    </f>
    <h/>
  </x>
  END
    my ($a, $b, $c, $d, $e, $g, $f, $h) = $x->byList;


    $c->moveBlockLast($d, $f, qw(c x));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a/>
    <b/>
    <e/>
    <f>
      <g/>
      <c/>
      <d/>
    </f>
    <h/>
  </x>
  END


=head3 moveBlockAfter($start, $end, $after, @context)

Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $end       Last sibling
  3  $after     Node to move after
  4  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(q(<x><a/><b/><c/><d/><e/><f/><g/><h/></x>));

    my ($a, $b, $c, $d, $e, $f, $g, $h) = $x->byList;


    $c->moveBlockAfter($d, $f, qw(c x));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -C $x eq q(<a/><b/><e/><f/><c/><d/><g/><h/>);


=head3 moveBlockBefore($start, $end, $before, @context)

Move the block of siblings starting with B<$start> in the optional context and ending with B<$end> before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $end       Last sibling
  3  $before    Node to move before
  4  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(q(<x><a/><b/><c/><d/><e/><f/><g/><h/></x>));

    my ($a, $b, $c, $d, $e, $f, $g, $h) = $x->byList;


    $f->moveBlockBefore($g, $b, qw(f x));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -C $x eq q(<a/><f/><g/><b/><c/><d/><e/><h/>);


=head3 moveBlockToLastFirst($start, $parent, @context)

Move the siblings starting with B<$start> in the optional context first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     Start sibling
  2  $parent    Parent to move first under
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
  </x>
  END
    my ($a, $b, $c) = $x->byList;


    $b->moveBlockToLastFirst($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a>
      <b/>
      <c/>
    </a>
  </x>
  END


=head3 moveBlockToLastLast($start, $parent, @context)

Move the block of siblings starting with B<$start> in the optional context last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $parent    Parent to move last under
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>
      <b/>
      <c/>
      <d/>
    </a>
    <e>
      <f/>
      <g/>
      <h/>
    </e>
  </x>
  END
    my ($b, $c, $d, $a, $f, $g, $h, $e) = $x->byList;


    $c->moveBlockToLastLast($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a>
      <b/>
    </a>
    <e>
      <f/>
      <g/>
      <h/>
      <c/>
      <d/>
    </e>
  </x>
  END


=head3 moveBlockToLastAfter($start, $after, @context)

Move the block of siblings starting with B<$start> in the optional context after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $after     Node to move after
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>
      <b/>
      <c/>
      <d/>
    </a>
    <e/>
  </x>
  END
    my ($b, $c, $d, $a, $e) = $x->byList;


    $c->moveBlockToLastAfter($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a>
      <b/>
    </a>
    <e/>
    <c/>
    <d/>
  </x>
  END


=head3 moveBlockToLastBefore($start, $before, @context)

Move the block of siblings starting with B<$start> in the optional context before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $before    Node to move before
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </x>
  END
    my ($a, $c, $d, $e, $b) = $x->byList;


    $d->moveBlockToLastBefore($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a/>
    <d/>
    <e/>
    <b>
      <c/>
    </b>
  </x>
  END


=head3 moveBlockFromFirstFirst($start, $parent, @context)

Move the siblings starting with B<$start> in the optional context first under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     Start sibling
  2  $parent    Parent to move first under
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>
      <b/>
      <c/>
      <d/>
    </a>
    <e>
      <f/>
      <g/>
      <h/>
    </e>
  </x>
  END
    my ($b, $c, $d, $a, $f, $g, $h, $e) = $x->byList;


    $g->moveBlockFromFirstFirst($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a>
      <f/>
      <g/>
      <b/>
      <c/>
      <d/>
    </a>
    <e>
      <h/>
    </e>
  </x>
  END


=head3 moveBlockFromFirstLast($start, $parent, @context)

Move the block of siblings starting with B<$start> in the optional context last under the specified B<$parent> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $parent    Parent to move last under
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b/>
    <c/>
  </x>
  END
    my ($a, $b, $c) = $x->byList;


    $b->moveBlockFromFirstLast($c);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <c>
      <a/>
      <b/>
    </c>
  </x>
  END


=head3 moveBlockFromFirstAfter($start, $after, @context)

Move the block of siblings starting with B<$start> in the optional context after the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $after     Node to move after
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a>
      <b/>
      <c/>
      <d/>
    </a>
    <e/>
  </x>
  END
    my ($b, $c, $d, $a, $e) = $x->byList;


    $c->moveBlockFromFirstAfter($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a>
      <d/>
    </a>
    <e/>
    <b/>
    <c/>
  </x>
  END


=head3 moveBlockFromFirstBefore($start, $before, @context)

Move the block of siblings starting with B<$start> in the optional context before the specified B<$after> node. Returns B<$node> if the move was made successfully, else confess that the specified move is impossible.

     Parameter  Description
  1  $start     First sibling
  2  $before    Node to move before
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $x = Data::Edit::Xml::new(<<END);
  <x>
    <a/>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </x>
  END
    my ($a, $c, $d, $e, $b) = $x->byList;


    $d->moveBlockFromFirstBefore($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <x>
    <a/>
    <c/>
    <d/>
    <b>
      <e/>
    </b>
  </x>
  END


=head2 Add selectively

Add new nodes unless they already exist.

=head3 addFirst($node, $tag, @context)

Add a new node L<first|/first> below the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::newTree(q(a));

    $a->addFirst_b for 1..2;

    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END


=head3 addNext($node, $tag, @context)

Add a new node L<next|/next> the specified B<$node> and return the new node unless a node with that tag already exists in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b/>
    <e/>
  </a>
  END

    $a->addFirst_b__addNext_c;

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END


=head3 addPrev($node, $tag, @context)

Add a new node L<before|/prev> the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END

    $a->addLast_e__addPrev_d;

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END


=head3 addLast($node, $tag, @context)

Add a new node L<last|/last> below the specified B<$node> with the specified B<$tag> unless a node with that tag already exists in that position. Return the new node if it was created unless return the pre-existing node..

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END


    $a->addLast(qw(e)) for 1..2;                                                    # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b/>
    <e/>
  </a>
  END


=head3 addWrapWith($node, $tag, @context)

L<Wrap|/wrap> the specified B<$node> with the specified tag if the node is not already wrapped with such a tag and return the new node unless a node with that tag already exists in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(q(<a><b/></a>));

    my $b = $a->first;

    $b->addWrapWith_c for 1..2;

    ok -p $a eq <<END;
  <a>
    <c>
      <b/>
    </c>
  </a>
  END


=head3 addSingleChild($node, $tag, @context)

Wrap the content of a specified B<$node> in a new node with the specified B<$tag> unless the content is already wrapped in a single child with the specified B<$tag>.

     Parameter  Description
  1  $node      Node
  2  $tag       Tag of new node
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <c>
      <b/>
    </c>
  </a>
  END

    $a->addSingleChild_d for 1..2;

    ok -p $a eq <<END;
  <a>
    <d>
      <c>
        <b/>
      </c>
    </d>
  </a>
  END


=head2 Add text selectively

Add new text unless it already exists.

=head3 addFirstAsText($node, $text, @context)

Add a new text node first below the specified B<$node> and return the new node unless a text node already exists there and starts with the same text in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $text      Text
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::newTree(q(a));


    $a->addFirstAsText(q(aaaa)) for 1..2;                                           # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $a eq q(<a>aaaa</a>);


=head3 addNextAsText($node, $text, @context)

Add a new text node after the specified B<$node> and return the new node unless a text node already exists there and starts with the same text in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $text      Text
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(q(<a><b/></a>));


    $a->go(q(b))->addNextAsText(q(bbbb)) for 1..2;                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b/>
  bbbb
  </a>
  END


=head3 addPrevAsText($node, $text, @context)

Add a new text node before the specified B<$node> and return the new node unless a text node already exists there and ends with the same text in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $text      Text
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b/>
  bbbb
  </a>
  END


    $a->go(q(b))->addPrevAsText(q(aaaa)) for 1..2;                                  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>aaaa
    <b/>
  bbbb
  </a>
  END


=head3 addLastAsText($node, $text, @context)

Add a new text node last below the specified B<$node> and return the new node unless a text node already exists there and ends with the same text in which case return the existing B<$node>.

     Parameter  Description
  1  $node      Node
  2  $text      Text
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -s $a eq q(<a>aaaa</a>);


    $a->addLastAsText(q(dddd)) for 1..2;                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -s $a eq q(<a>aaaadddd</a>);


=head3 joinWithText($node, $text, $over, @context)

Insert some text between the children of the current B<$node> as specified by B<$text> between all the children of the current $node as long as they all have a tag of B<$over>. Return the current $node on success or B<undef> on failure.

     Parameter  Description
  1  $node      Node
  2  $text      Text
  3  $over      Child tag
  4  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b/>
    <b/>
    <b/>
  </a>
  END

    $a->joinWithText_CC_b_a;

    ok -p $a eq <<END;
  <a>
    <b/>
  CC
    <b/>
  CC
    <b/>
  </a>
  END


=head2 Fission

Split the parent L<before|/splitBefore> or L<after|/splitAfter> the specified sibling.

=head3 splitBefore($node, @context)

Split the parent node into two identical nodes except all the siblings before the specified B<$node> are retained by the existing parent while any following siblings become siblings of the new parent node which is placed after the existing parent. The new parent is returned.

     Parameter  Description
  1  $node      Node to split before
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END

    $a->go_b_d__splitBefore;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <d/>
      <e/>
    </b>
  </a>
  END


=head3 splitAfter($node, @context)

Split the parent node into two identical nodes except all the siblings after the specified B<$node> are retained by the existing parent while any preceding siblings become siblings of the new parent node which is placed before the existing parent. The new parent is returned

     Parameter  Description
  1  $node      Node to split before
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END

    $a->go_b_d__splitAfter;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
    </b>
    <b>
      <e/>
    </b>
  </a>
  END


=head2 Fusion

Join consecutive nodes

=head3 concatenate($target, $source, @context)

Concatenate two successive nodes and return the B<$target> node.

     Parameter  Description
  1  $target    Target node to replace
  2  $source    Node to concatenate
  3  @context   Optional context of $target

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $s = <<END;
  <a>
    <b>
      <A/>
      <B/>
    </b>
    <c>
      <C/>
      <D/>
    </c>
  </a>
  END

    my $a = Data::Edit::Xml::new($s);


    $a->go(q(b))->concatenate($a->go(q(c)));                                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    my $t = <<END;
  <a>
    <b>
      <A/>
      <B/>
      <C/>
      <D/>
    </b>
  </a>
  END

    ok $t eq -p $a;


=head3 concatenateSiblings($node, @context)

Concatenate the nodes that precede and follow the specified B<$node> in the optioonal B<@context> as long as they have the same tag as the specified B<$node> and return the specified B<$node>.

     Parameter  Description
  1  $node      Concatenate around this node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="1"/>
    </b>
    <b>
      <c id="2"/>
    </b>
    <b>
      <c id="3"/>
    </b>
    <b>
      <c id="4"/>
    </b>
  </a>
  END


    $a->go(qw(b 3))->concatenateSiblings;                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c id="1"/>
      <c id="2"/>
      <c id="3"/>
      <c id="4"/>
    </b>
  </a>
  END


=head3 mergeDuplicateChildWithParent($parent, @context)

Merge a parent node with its only child if their tags are the same and their attributes do not collide other than possibly the id in which case the parent id is used. Any labels on the child are transferred to the parent. The child node is then unwrapped and the parent node is returned.

     Parameter  Description
  1  $parent    Parent this node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


   {my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b   id="b" b="bb">
      <b id="c" c="cc"/>
    </b>
  </a>
  END

    my ($c, $b) = $a->byList;

    is_deeply [$b->id, $c->id], [qw(b c)];

    ok $c == $b->hasSingleChild;


    $b->mergeDuplicateChildWithParent;                                              # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b b="bb" c="cc" id="b"/>
  </a>
  END

    ok $b == $a->hasSingleChild;


=head2 Put as text

Add text to the L<parse|/parse> tree.

=head3 putFirstAsText($node, $text, @context)

Add a new text node first under a parent and return the new text node.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be added which might contain unparsed Xml as well as text
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"/>
    </b>
  </a>
  END


    $x->go(qw(b c))->putFirstAsText("<d id=\"dd\">DDDD</d>");                       # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d></c>
    </b>
  </a>
  END


=head3 putTextFirst($node, @text)

Given a B<$node> place some B<@text> first under the $node and return the new text node else return B<undef> if this is not possible.

     Parameter  Description
  1  $node      The parent node
  2  @text      The text to be added

B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my $b = $a->first;

    $b->putTextPrev(qw(Before the B node));
    $b->putTextNext(qw(After the B node));

    $b->putTextFirst(qw(First under the B node));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $b->putTextLast(qw(Last under the B node));

    ok -p $a eq <<END;
  <a>Before the B node
    <b>First under the B node
      <c/>
  Last under the B node
    </b>
  After the B node
  </a>
  END


=head3 putLastAsText($node, $text, @context)

Add a new text node last under a parent and return the new text node.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be added which might contain unparsed Xml as well as text
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d></c>
    </b>
  </a>
  END


    $x->go(qw(b c))->putLastAsText("<e id=\"ee\">EEEE</e>");                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d><e id="ee">EEEE</e></c>
    </b>
  </a>
  END


=head3 putTextLast($node, @text)

Given a B<$node> place some B<@text> last under the $node and return the new text node else return B<undef> if this is not possible.

     Parameter  Description
  1  $node      The parent node
  2  @text      The text to be added

B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my $b = $a->first;

    $b->putTextPrev(qw(Before the B node));
    $b->putTextNext(qw(After the B node));
    $b->putTextFirst(qw(First under the B node));

    $b->putTextLast(qw(Last under the B node));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>Before the B node
    <b>First under the B node
      <c/>
  Last under the B node
    </b>
  After the B node
  </a>
  END


=head3 putNextAsText($node, $text, @context)

Add a new text node following the specified B<$node> and return the new text node.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be added which might contain unparsed Xml as well as text
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d><e id="ee">EEEE</e></c>
    </b>
  </a>
  END


    $x->go(qw(b c))->putNextAsText("<n id=\"nn\">NNNN</n>");                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d><e id="ee">EEEE</e></c>
  <n id="nn">NNNN</n>
    </b>
  </a>
  END


=head3 putTextNext($node, @text)

Given a B<$node> place some B<@text> next to the $node and return the new text node else return B<undef> if this is not possible.

     Parameter  Description
  1  $node      The parent node
  2  @text      The text to be added

B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my $b = $a->first;

    $b->putTextPrev(qw(Before the B node));

    $b->putTextNext(qw(After the B node));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $b->putTextFirst(qw(First under the B node));
    $b->putTextLast(qw(Last under the B node));

    ok -p $a eq <<END;
  <a>Before the B node
    <b>First under the B node
      <c/>
  Last under the B node
    </b>
  After the B node
  </a>
  END


=head3 putPrevAsText($node, $text, @context)

Add a new text node following the specified B<$node> and return the new text node

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be added which might contain unparsed Xml as well as text
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb">
      <c id="cc"><d id="dd">DDDD</d><e id="ee">EEEE</e></c>
  <n id="nn">NNNN</n>
    </b>
  </a>
  END


    $x->go(qw(b c))->putPrevAsText("<p id=\"pp\">PPPP</p>");                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $x eq <<END;
  <a id="aa">
    <b id="bb"><p id="pp">PPPP</p>
      <c id="cc"><d id="dd">DDDD</d><e id="ee">EEEE</e></c>
  <n id="nn">NNNN</n>
    </b>
  </a>
  END


=head3 putTextPrev($node, @text)

Given a B<$node> place some B<@text> prior to the $node and return the new text node else return B<undef> if this is not possible.

     Parameter  Description
  1  $node      The parent node
  2  @text      The text to be added

B<Example:>


    my $a = Data::Edit::Xml::new(q(<a><b><c/></b></a>));

    my $b = $a->first;


    $b->putTextPrev(qw(Before the B node));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $b->putTextNext(qw(After the B node));
    $b->putTextFirst(qw(First under the B node));
    $b->putTextLast(qw(Last under the B node));

    ok -p $a eq <<END;
  <a>Before the B node
    <b>First under the B node
      <c/>
  Last under the B node
    </b>
  After the B node
  </a>
  END


=head2 Put as tree

Add parsed text to the L<parse|/parse> tree.

=head3 putFirstAsTree($node, $text, @context)

Put parsed text first under the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be parsed and added
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a/>));

    ok -p $a eq <<END;
  <a/>
  END


    my $b = $a->putFirstAsTree(q(<b/>));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END

    $b->putNextAsTree(q(<c/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
  </a>
  END

    my $e = $a->putLastAsTree(q(<e/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END

    $e->putPrevAsTree(q(<d/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END
   }


=head3 putLastAsTree($node, $text, @context)

Put parsed text last under the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be parsed and added
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a/>));

    ok -p $a eq <<END;
  <a/>
  END

    my $b = $a->putFirstAsTree(q(<b/>));
    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END

    $b->putNextAsTree(q(<c/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
  </a>
  END


    my $e = $a->putLastAsTree(q(<e/>));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END

    $e->putPrevAsTree(q(<d/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END
   }


=head3 putNextAsTree($node, $text, @context)

Put parsed text after the specified B<$node> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be parsed and added
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a/>));

    ok -p $a eq <<END;
  <a/>
  END

    my $b = $a->putFirstAsTree(q(<b/>));
    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END


    $b->putNextAsTree(q(<c/>));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
  </a>
  END

    my $e = $a->putLastAsTree(q(<e/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END

    $e->putPrevAsTree(q(<d/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END
   }


=head3 putPrevAsTree($node, $text, @context)

Put parsed text before the specified B<$parent> parent and return a reference to the parsed tree. Confess if the text cannot be parsed successfully.

     Parameter  Description
  1  $node      The parent node
  2  $text      The string to be parsed and added
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


  if (1)
   {my $a = Data::Edit::Xml::new(q(<a/>));

    ok -p $a eq <<END;
  <a/>
  END

    my $b = $a->putFirstAsTree(q(<b/>));
    ok -p $a eq <<END;
  <a>
    <b/>
  </a>
  END

    $b->putNextAsTree(q(<c/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
  </a>
  END

    my $e = $a->putLastAsTree(q(<e/>));
    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <e/>
  </a>
  END


    $e->putPrevAsTree(q(<d/>));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b/>
    <c/>
    <d/>
    <e/>
  </a>
  END
   }


=head2 Put as comment

Add some comments to the parse tree relative to the specified B<$node> in the optional B<$context> and return the specified B<$node>.

=head3 putFirstAsComment($node, $text, @context)

Put a comment first under the specified B<$node> and return the specified B<$node>.

     Parameter  Description
  1  $node      Parent node
  2  $text      Comment
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
  </a>
  END


    $a->putFirstAsComment(q(First));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $a->putLastAsComment (q(Last));
    $a->go_b_c->putPrevAsComment(q(Before C));
    $a->go_b_c->putNextAsComment(q(After C));

    ok -p $a eq <<END;
  <a><!-- First -->
    <b><!-- Before C -->
      <c/>
  <!-- After C -->
    </b>
  <!-- Last -->
  </a>
  END


=head3 putLastAsComment($node, $text, @context)

Put a comment last under the specified B<$node> and return the specified B<$node>.

     Parameter  Description
  1  $node      Parent node
  2  $text      Comment
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->putFirstAsComment(q(First));

    $a->putLastAsComment (q(Last));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $a->go_b_c->putPrevAsComment(q(Before C));
    $a->go_b_c->putNextAsComment(q(After C));

    ok -p $a eq <<END;
  <a><!-- First -->
    <b><!-- Before C -->
      <c/>
  <!-- After C -->
    </b>
  <!-- Last -->
  </a>
  END


=head3 putNextAsComment($node, $text, @context)

Put a comment after the specified B<$node> and return the specified B<$node>

     Parameter  Description
  1  $node      Node
  2  $text      Comment
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->putFirstAsComment(q(First));
    $a->putLastAsComment (q(Last));
    $a->go_b_c->putPrevAsComment(q(Before C));

    $a->go_b_c->putNextAsComment(q(After C));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a><!-- First -->
    <b><!-- Before C -->
      <c/>
  <!-- After C -->
    </b>
  <!-- Last -->
  </a>
  END


=head3 putPrevAsComment($node, $text, @context)

Put a comment before the specified B<$parent> parentand return the specified B<$node>

     Parameter  Description
  1  $node      Node
  2  $text      Comment
  3  @context   Context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
    </b>
  </a>
  END

    $a->putFirstAsComment(q(First));
    $a->putLastAsComment (q(Last));

    $a->go_b_c->putPrevAsComment(q(Before C));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    $a->go_b_c->putNextAsComment(q(After C));

    ok -p $a eq <<END;
  <a><!-- First -->
    <b><!-- Before C -->
      <c/>
  <!-- After C -->
    </b>
  <!-- Last -->
  </a>
  END


=head2 Break in and out

Break nodes out of nodes or push them back

=head3 breakIn($start, @context)

Concatenate the nodes following and preceding the start node, unwrapping nodes whose tag matches the start node and return the start node. To concatenate only the preceding nodes, use L<breakInBackwards|/breakInBackwards>, to concatenate only the following nodes, use L<breakInForwards|/breakInForwards>.

     Parameter  Description
  1  $start     The start node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a>
    <d/>
    <b>
      <c/>
      <c/>
    </b>
    <e/>
    <b>
      <c/>
      <c/>
    </b>
    <d/>
  </a>
  END


      $a->go(qw(b 1))->breakIn;                                                     # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a>
    <b>
      <d/>
      <c/>
      <c/>
      <e/>
      <c/>
      <c/>
      <d/>
    </b>
  </a>
  END


=head3 breakInForwards($start, @context)

Concatenate the nodes following the start node, unwrapping nodes whose tag matches the start node and return the start node in the manner of L<breakIn|/breakIn>.

     Parameter  Description
  1  $start     The start node
  2  @context   Optional context..

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a>
    <d/>
    <b>
      <c/>
      <c/>
    </b>
    <e/>
    <b>
      <c/>
      <c/>
    </b>
    <d/>
  </a>
  END


      $a->go(q(b))->breakInForwards;                                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a>
    <d/>
    <b>
      <c/>
      <c/>
      <e/>
      <c/>
      <c/>
      <d/>
    </b>
  </a>
  END


=head3 breakInBackwards($start, @context)

Concatenate the nodes preceding the start node, unwrapping nodes whose tag matches the start node and return the start node in the manner of L<breakIn|/breakIn>.

     Parameter  Description
  1  $start     The start node
  2  @context   Optional context..

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


      ok -p $a eq <<END;
  <a>
    <d/>
    <b>
      <c/>
      <c/>
    </b>
    <e/>
    <b>
      <c/>
      <c/>
    </b>
    <d/>
  </a>
  END


      $a->go(qw(b 1))->breakInBackwards;                                            # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a>
    <b>
      <d/>
      <c/>
      <c/>
      <e/>
      <c/>
      <c/>
    </b>
    <d/>
  </a>
  END


=head3 breakOut($parent, @tags)

Lift child nodes with the specified tags under the specified parent node splitting the parent node into clones and return the cut out original node.

     Parameter  Description
  1  $parent    The parent node
  2  @tags      The tags of the modes to be broken out.

B<Example:>


    my $A = Data::Edit::Xml::new("<a><b><d/><c/><c/><e/><c/><c/><d/></b></a>");


      $a->go(q(b))->breakOut($a, qw(d e));                                          # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -p $a eq <<END;
  <a>
    <d/>
    <b>
      <c/>
      <c/>
    </b>
    <e/>
    <b>
      <c/>
      <c/>
    </b>
    <d/>
  </a>
  END


=head3 breakOutChild($node, @context)

Lift the specified B<$node> up one level splitting its parent. Return the specified B<$node> on success or B<undef> if the operation is not possible.

     Parameter  Description
  1  $node      Node to break out
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END

    my ($c, $d, $e, $b) = $a->byList;

    $d->breakOutChild;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
    </b>
    <d/>
    <b>
      <e/>
    </b>
  </a>
  END


=head2 Split and Zip

Move a node up the parse tree by splitting its parent nodes

=head3 splitParentAfter($node, @context)

Finish and restart the parent of the specified B<$node> just after the specified B<$node> and return the newly created parent node on success or B<undef> on failure.

     Parameter  Description
  1  $node      The splitting node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END

    $a->go_b_d__splitParentAfter;

    ok -p $a eq <<END;
  <a>
    <b>
      <c/>
      <d/>
    </b>
    <b>
      <e/>
    </b>
  </a>
  END


B<spa> is a synonym for L<splitParentAfter|/splitParentAfter>.


=head3 splitParentBefore($node, @context)

Finish and restart the parent of the specified B<$node> just before the specified B<$node> and return the newly created parent node on success or B<undef> on failure.

     Parameter  Description
  1  $node      The splitting node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c/>
      <d/>
      <e/>
    </b>
  </a>
  END

    $a->go_b_d__splitParentBefore;

    ok -p $a ne <<END;
  <a>
    <b>
      <c/>
    </b>
    <b>
      <d/>
      <e/>
    </b>
  </a>
  END


B<spb> is a synonym for L<splitParentBefore|/splitParentBefore>.


=head3 splitTo($node, $parent, @context)

Lift the specified B<$node> up until it splits the specified B<$parent> node. Return the specified B<$node> on success or B<undef> if the operation is not possible.

     Parameter  Description
  1  $node      Node to break out
  2  $parent    Ancestral node to split
  3  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;


    ok !$a->splitTo($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$b->splitTo($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $e->splitTo($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;


    ok !$a->splitTo($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok !$b->splitTo($a);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok  $e->splitTo($b);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    ok $e->zipDownOnce;                                                           # Invalidates b
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


   ok  $e->splitTo($a->first);                                                    # b invalidated by zipDownOnce  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   $e->zipDown;

   ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END


=head3 adoptChild($parent, $child, @context)

Lift the specified B<$child> node up until it is an immediate child of the specified B<$parent> node by splitting any intervening nodes. Return the specified B<$child> node on success or B<undef> if the operation is not possible.

     Parameter  Description
  1  $parent    Adopting parent node
  2  $child     Child node to adopt
  3  @context   Optional context of child

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;


    $a->adoptChild($e);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    $e->zipDown;
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END



=head3 zipDownOnce($node, @context)

Push a node down one level by making it a child of a node formed by merging the preceding and following siblings if they have the same tag.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;

    ok !$a->splitTo($a);
    ok !$b->splitTo($a);
    ok  $e->splitTo($b);

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


    ok $e->zipDownOnce;                                                           # Invalidates b  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


   ok  $e->splitTo($a->first);                                                    # b invalidated by zipDownOnce  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   $e->zipDown;

   ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END


=head3 zipDown($node, @context)

Push a node down as many levels as possible by making it a child of a node formed by merging the preceding and following siblings if they have the same tag.

     Parameter  Description
  1  $node      Node
  2  @context   Optional context

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;

    $a->adoptChild($e);
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END


    $e->zipDown;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END


    my $a = Data::Edit::Xml::new(<<END);
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END

    my ($e, $d, $c, $b) = $a->byList;

    ok !$a->splitTo($a);
    ok !$b->splitTo($a);
    ok  $e->splitTo($b);

    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
    </b>
    <e/>
    <b>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

    ok $e->zipDownOnce;                                                           # Invalidates b
    ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d/>
      </c>
      <e/>
      <c>
        <d/>
      </c>
    </b>
  </a>
  END

   ok  $e->splitTo($a->first);                                                    # b invalidated by zipDownOnce

   $e->zipDown;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


   ok -p $a eq <<END;
  <a>
    <b>
      <c>
        <d>
          <e/>
        </d>
      </c>
    </b>
  </a>
  END


=head3 splitAndWrapFromStart($parent, $split, $wrap, %attributes)

Split the child nodes of B<$parent> on the child nodes with a tag of B<$split> wrapping the splitting node and all preceding nodes from the previous splitting node or the start with the specified B<$wrap>ping node with optional B<%attributes>.  Returns an array of the wrapping nodes created.

     Parameter    Description
  1  $parent      Parent node
  2  $split       Tag of splitting nodes
  3  $wrap        Tag for wrapping node
  4  %attributes  Attributes for wrapping nodes

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>



    my $A = Data::Edit::Xml::new(<<END);
  <a>
    <PP/>
    <b>
      <c/>
    </b>
    <QQ/>
    <b>
      <d/>
    </b>
    <RR/>
    <b>
      <e/>
    </b>
    <SS/>
  </a>
  END

    my $a = $A->clone;

    ok 3 == $a->splitAndWrapFromStart(qw(b B));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $a eq <<END;
  <a>
    <B>
      <PP/>
      <b>
        <c/>
      </b>
    </B>
    <B>
      <QQ/>
      <b>
        <d/>
      </b>
    </B>
    <B>
      <RR/>
      <b>
        <e/>
      </b>
    </B>
    <SS/>
  </a>
  END


    my $b = $A->clone;
    ok 3 == $b->splitAndWrapToEnd(qw(b B));

    ok -p $b eq <<END;
  <a>
    <PP/>
    <B>
      <b>
        <c/>
      </b>
      <QQ/>
    </B>
    <B>
      <b>
        <d/>
      </b>
      <RR/>
    </B>
    <B>
      <b>
        <e/>
      </b>
      <SS/>
    </B>
  </a>
  END


=head3 splitAndWrapToEnd($parent, $split, $wrap, %attributes)

Split the sequence of child nodes under the specified B<$parent> node on those child nodes whose tag value is B<$split> wrapping the splitting node and all following nodes up until the next splitting node or the end of the sequence with newly created nodes whose tag is B<$wrap> with optional attributes B<%attributes>.  Returns an array of the wrapping nodes so created.

     Parameter    Description
  1  $parent      Parent node
  2  $split       Tag of splitting nodes
  3  $wrap        Tag for wrapping node
  4  %attributes  Attributes for wrapping nodes

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>



    my $A = Data::Edit::Xml::new(<<END);
  <a>
    <PP/>
    <b>
      <c/>
    </b>
    <QQ/>
    <b>
      <d/>
    </b>
    <RR/>
    <b>
      <e/>
    </b>
    <SS/>
  </a>
  END

    my $a = $A->clone;
    ok 3 == $a->splitAndWrapFromStart(qw(b B));

    ok -p $a eq <<END;
  <a>
    <B>
      <PP/>
      <b>
        <c/>
      </b>
    </B>
    <B>
      <QQ/>
      <b>
        <d/>
      </b>
    </B>
    <B>
      <RR/>
      <b>
        <e/>
      </b>
    </B>
    <SS/>
  </a>
  END


    my $b = $A->clone;

    ok 3 == $b->splitAndWrapToEnd(qw(b B));  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


    ok -p $b eq <<END;
  <a>
    <PP/>
    <B>
      <b>
        <c/>
      </b>
      <QQ/>
    </B>
    <B>
      <b>
        <d/>
      </b>
      <RR/>
    </B>
    <B>
      <b>
        <e/>
      </b>
      <SS/>
    </B>
  </a>
  END


=head2 Replace

Replace nodes in the L<parse|/parse> tree with nodes or text

=head3 replaceWith($old, $new, @context)

Replace a node (and all its content) with a L<new node|/newTag> (and all its content) and return the new node. If the node to be replaced is the root of the L<parse|/parse> tree then no action is taken other then returning the new node.

     Parameter  Description
  1  $old       Old node
  2  $new       New node
  3  @context   Optional context..

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


     {my $x = Data::Edit::Xml::new(qq(<a><b><c id="cc"/></b></a>));


      $x->go(qw(b c))->replaceWith($x->newTag(qw(d id dd)));                        # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -s $x eq '<a><b><d id="dd"/></b></a>';


=head3 replaceWithText($old, $text, @context)

Replace a node (and all its content) with a new text node and return the new node.

     Parameter  Description
  1  $old       Old node
  2  $text      Text of new node
  3  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/at>. If the context is supplied and
B<$node> is not in this context then this method returns B<undef> immediately.



B<Example:>


     {my $x = Data::Edit::Xml::new(qq(<a><b><c id="cc"/></b></a>));


      $x->go(qw(b c))->replaceWithText(qq(BBBB));                                   # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok -s $x eq '<a><b>BBBB</b></a>';


=head3 replaceWithBlank($old, @context)

Replace a node (and all its content) with a new blank text node and return the new node.

     Parameter  Description
  1  $old       Old node
  2  @context   Optional context.

Use the optional B<@context> parameter to test the context of the specified
B<$node> as understood by method L<at|/