#!perl

use strict;

use Test::More tests => 500 * 2;
use Test::Number::Delta within => 3;
use Astro::Coords;
use Astro::Telescope;
use DateTime 0.76;

my $i = 0;

while (<DATA>) {
  $i ++;
  chomp;
  my ($year, $month, $day, $hour, $minute, $second,
      $planet, $telescope_name, $event, $horizon,
      $rise_ref, $set_ref) = split;

  my $c = new Astro::Coords(planet => $planet);

  $c->telescope(new Astro::Telescope($telescope_name));
  $c->datetime(new DateTime(year => $year, month => $month, day => $day,
               hour => $hour, minute => $minute, second => $second, time_zone => 'UTC'));

  my $rise = $c->rise_time(event => $event, horizon => $horizon);
  my $set  =  $c->set_time(event => $event, horizon => $horizon);

  if ($rise_ref eq 'undef') {
    if (! defined $rise) {
      pass("Rise time $i ($planet) not defined");
    }
    else {
      fail("Rise time $i ($planet) should not have been defined");
    }
  }
  elsif (! defined $rise) {
    fail("Rise time $i ($planet) not defined but should have been");
  }
  else {
    delta_ok($rise->epoch(), $rise_ref, "Rise time $i ($planet)");
  }

  if ($set_ref eq 'undef') {
    if (! defined $set) {
      pass("Set time $i ($planet) not defined");
    }
    else {
      fail("Set time $i ($planet) should not have been defined");
    }
  }
  elsif (! defined $set) {
    fail("Set time $i ($planet) not defined but should have been");
  }
  else {
    delta_ok($set->epoch(), $set_ref, "Set time $i ($planet)");
  }
}


# Regression/timing test data generated with
# Astro::Coords version 0.13
# Astro::Coords::Planet version 0.05
__DATA__
2048 11 14 00 19 48 venus CAMB1MILE -1 0 2488878951 2488903770
2040 08 25 14 48 05 venus QUEBEC1.6 1 0.104719666666667 2229597725 2229551450
2007 07 18 07 09 52 moon FLAGSTF61 0 -0.0174532777777778 1184776586 1184734348
2034 05 18 02 59 16 moon CFHT 0 -0.104719666666667 2031491440 2031541956
2079 03 07 05 13 06 sun STROMLO74 0 -0.104719666666667 3445356600 3445405157
2066 12 07 19 52 41 jupiter LICK120 -1 -0.0523598333333333 3058960545 3058910555
2009 07 19 04 11 35 saturn HPROV1.52 1 0 1247992920 1248039050
1996 05 19 10 07 01 saturn MTLEMMON60 0 0 832499630 832542670
2064 10 18 04 41 16 mercury JODRELL1 1 0.0523598333333333 2991546087 2991574686
2014 06 19 09 59 16 mercury MTLEMMON60 -1 0.174532777777778 1403098235 1403141547
2001 07 12 15 06 52 venus DUNLAP74 0 0 994921359 994973909
2009 11 12 13 06 24 neptune ESOSCHM -1 0 1257957098 1258004028
2029 07 23 12 05 13 neptune HPROV1.93 -1 -0.104719666666667 1879450289 1879498784
2049 07 19 17 59 35 mars MAUNAK88 0 0.0349065555555556 2510324887 2510370696
2052 08 25 16 13 24 uranus VLT2 -1 -0.0349065555555556 2608200852 2608158583
2061 11 09 02 41 16 saturn CSO 0 0.349065555555556 2898749196 2898699313
2016 07 04 05 09 06 venus MCDONLD2.7 1 -0.349065555555556 1467628904 1467692756
2066 07 09 12 42 05 venus TIDBINBLA 0 0.104719666666667 3045946490 3045895411
2026 01 07 04 16 52 sun EFFELSBERG 1 0.174532777777778 1767776947 1767794915
2033 01 23 03 41 24 mercury MCDONLD2.1 0 0.349065555555556 1990023287 1990047574
2002 07 09 15 01 07 venus TIDBINBLA 0 -0.0349065555555556 1026257289 1026210634
2047 03 19 02 07 48 venus PALOMAR60 -1 -0.0349065555555556 2436535400 2436497584
2076 02 09 23 59 13 saturn FLAGSTF61 -1 -0.174532777777778 3348468642 3348510148
2017 06 09 21 11 39 saturn CATALINA61 0 0 1497062744 1497012565
2021 04 12 13 48 01 neptune LPO4.2 -1 0.0523598333333333 1618206217 1618160368
2001 12 23 21 52 59 moon CFHT -1 0.174532777777778 1009063481 1009101658
1984 07 18 17 42 48 mercury TIDBINBLA -1 0.0523598333333333 458952566 458988280
1981 05 12 22 07 01 neptune MTLEMMON60 -1 -0.104719666666667 358488429 358528133
2053 04 23 20 06 52 venus MCDONLD2.7 1 -0.0174532777777778 2629111813 2629073795
2070 12 12 21 41 24 neptune LPO2.5 -1 -0.0523598333333333 3185638274 3185603128
2006 03 19 03 59 16 jupiter USSR6 -1 -0.0349065555555556 1142710039 1142660729
2028 06 07 07 19 39 uranus VLA 1 0.104719666666667 1843992532 1844039385
1994 06 25 01 06 05 moon ATCA 0 0.0174532777777778 772536036 772494948
1984 03 12 18 05 52 uranus MAUNAK88 -1 0.104719666666667 447936712 447886170
1983 07 04 15 01 24 venus TOLOLO1.5M 0 0.104719666666667 426178162 426214055
2025 02 07 15 01 39 neptune LOWELL72 0 0.0349065555555556 1738945463 1738900568
2046 05 01 04 16 11 moon PARKES -1 0 2408717011 2408674262
1990 04 07 19 11 13 venus FLAGSTF61 -1 -0.174532777777778 639483428 639442558
2065 05 18 19 59 13 mercury ANU2.3 -1 0 3009817634 3009855393
2036 12 07 14 05 16 sun STEWARD90 -1 -0.0174532777777778 2112185556 2112222166
2010 04 04 04 48 39 saturn LPO2.5 -1 0 1270319031 1270276544
1995 09 19 03 59 06 neptune QUEBEC1.6 1 0.174532777777778 811545194 811568697
2034 12 09 06 52 24 sun PALOMAR60 -1 0.0349065555555556 2049202461 2049236665
2040 04 07 07 41 05 uranus CSO 1 0 2217451759 2217412458
2036 02 19 22 35 24 mars CFHT 1 -0.0523598333333333 2087152312 2087113896
2050 01 18 17 39 05 mars TIDBINBLA -1 0.0523598333333333 2526131807 2526093631
1987 10 14 07 19 05 uranus HARVARD -1 -0.0174532777777778 561140295 561172938
2028 03 07 19 39 16 sun ARECIBO 1 -0.104719666666667 1836123506 1836082605
1984 06 19 10 24 06 venus FCRAO -1 0 456484794 456452942
2014 07 19 18 16 52 jupiter VLT3 1 0 1405856452 1405808654
1980 03 01 05 05 19 jupiter EFFELSBERG -1 -0.174532777777778 320685726 320657483
2007 08 01 08 05 01 moon MAUNAK88 0 0 1185950263 1185993109
2068 02 19 04 09 39 uranus USSR6 0 0 3096834253 3096866209
2059 10 14 14 11 59 mercury ESONTT -1 0 2833355677 2833317998
2023 06 23 22 52 59 moon FLAGSTF61 0 0.349065555555556 1687546438 1687582713
2004 11 01 22 05 06 mercury LPO1 -1 0.0349065555555556 1099299339 1099335605
2030 01 23 10 41 13 mars CFHT -1 0 1895338037 1895379156
2029 08 23 15 42 42 venus VLT2 -1 0 1882184473 1882141984
2080 07 01 23 48 06 mercury STROMLO74 0 0.174532777777778 3487091846 3487121495
2009 06 18 14 09 35 moon KPNO36FT 0 0 1245314215 1245363662
2034 03 18 11 06 52 neptune ARECIBO -1 0 2026209052 2026253086
2067 03 18 11 24 16 saturn KPNO36FT 0 -0.174532777777778 3067714592 3067680770
1985 07 19 02 07 05 saturn ARECIBO 1 -0.349065555555556 490639860 490604553
2069 05 19 23 13 39 mars TAUTSCHM 1 -0.0174532777777778 3136237999 3136279486
2061 12 23 16 48 09 uranus TIDBINBLA 1 -0.0349065555555556 2902664745 2902628651
2037 12 01 12 01 24 mercury MTEKAR 1 0.349065555555556 undef undef
1987 04 18 15 16 19 saturn KPNO158 0 -0.174532777777778 545721004 545763367
2032 05 12 19 13 39 sun STEWARD90 0 -0.174532777777778 1967974969 1968030196
1991 04 01 07 19 06 moon KECK1 -1 -0.0523598333333333 670485805 670439311
1999 06 19 19 35 35 neptune APO3.5 1 0 929852835 929889781
1982 05 01 23 42 41 saturn MOPRA 1 0.0174532777777778 389168779 389212519
2057 05 23 17 07 24 jupiter HARVARD 0 0.0349065555555556 2757824529 2757863198
2021 05 23 21 42 41 venus CAMB5KM 0 -0.0523598333333333 1621829906 1621806263
1987 10 25 16 41 07 jupiter TOLOLO4M 1 0 562198582 562239485
2013 05 01 02 16 48 moon ARECIBO -1 0 1367290945 1367332169
2052 12 25 15 11 05 mercury EFFELSBERG 0 0.349065555555556 undef undef
2009 09 12 09 39 16 venus VLA 1 0.0523598333333333 1252752454 1252798955
2010 01 01 09 39 48 saturn VLT4 1 0.104719666666667 1262407318 1262361005
2055 06 23 22 41 01 jupiter AAT -1 0 2697347575 2697397849
1990 06 12 05 06 13 mars MTLEMMON60 1 0 645179536 645223228
2063 03 19 14 59 42 jupiter LPO1 0 0.0349065555555556 2941550659 2941509631
2071 09 12 19 39 48 mercury AAT 1 0 3209314277 3209355831
2049 10 12 04 48 11 neptune VLT1 0 0 2517614726 2517653843
2047 09 25 21 24 07 sun GBVA140 0 0 2453022841 2453065639
2010 04 12 19 42 59 sun IRTF 0 0.0523598333333333 1271089365 1271132560
1987 03 25 01 11 35 saturn VLT4 0 0 543640407 543602323
2073 01 14 21 59 07 moon MCDONLD2.7 1 0 3251730122 3251685915
2031 09 04 01 48 07 mars BOSQALEGRE 1 0 1946299070 1946263298
2010 03 18 16 19 13 neptune MTHOP1.5 0 -0.349065555555556 1268908531 1268959170
2069 06 14 18 09 48 mars KOSMA3M 0 0.0174532777777778 3138482111 3138439801
1991 07 07 17 09 24 mercury DUPONT 1 -0.174532777777778 678975389 678932360
2071 09 04 19 48 48 mercury KOTTAMIA 0 0 3208647155 3208607207
2027 04 07 00 59 01 sun ESO3.6 1 0 1807095631 1807136949
2011 04 12 09 05 16 mars QUEBEC1.6 1 0 1302601472 1302645823
1989 04 18 13 16 09 mars PALOMAR48 0 -0.0349065555555556 608919021 608885625
2050 03 07 00 13 16 moon FCRAO 1 0.349065555555556 2530310977 2530254497
1981 10 18 07 41 01 moon DUNLAP74 1 -0.0349065555555556 372307487 372274479
2020 06 12 00 05 19 moon ESONTT 1 0 1591933544 1591980742
2000 07 09 05 39 13 neptune TIDBINBLA -1 -0.0523598333333333 963044286 963095939
2078 08 04 22 05 19 mercury CFHT 0 -0.174532777777778 3426848035 3426899619
1985 03 12 05 24 13 sun MTHOP1.5 0 0.0174532777777778 479483154 479438416
2057 10 01 04 13 41 venus MAGELLAN1 0 0 2769154158 2769109214
2056 03 19 15 11 19 mars USSR600 0 0 2720744612 2720695767
2048 03 14 00 01 13 venus SUGARGROVE -1 0.174532777777778 2467712709 2467745139
2053 09 01 00 09 07 neptune MOPRA 0 0 2640263181 2640300314
2048 04 12 05 09 16 mercury VLT1 1 0.104719666666667 2470305022 2470342576
2073 11 23 03 48 39 saturn PARKES -1 0 3278602519 3278565241
1982 11 01 22 16 59 mars KECK2 0 0 405030006 405068575
1980 08 04 22 11 13 venus SANPM83 0 0.174532777777778 334233848 334276814
1988 01 04 23 05 05 sun BOSQALEGRE 0 0 568373021 568336967
2024 11 01 07 01 41 jupiter VLT3 0 0.349065555555556 1730432574 1730458150
1996 10 25 04 09 01 moon JCMT 1 -0.349065555555556 846295670 846262669
1987 05 25 20 01 13 moon USSR6 0 0.104719666666667 548990147 548949598
2071 05 25 17 06 13 saturn ESOSCHM 0 -0.349065555555556 3199803431 3199773739
2026 04 14 03 59 41 jupiter BLOEMF -1 0.0349065555555556 1776078299 1776113717
2000 02 18 13 01 13 neptune TAUTSCHM 0 0 950851894 950883086
2013 02 09 08 24 59 neptune OKAYAMA 0 0.349065555555556 1360369380 1360396165
1981 07 01 03 59 01 venus UKST 1 0 362875154 362825379
2017 10 25 23 07 11 moon BLOEMF 0 -0.174532777777778 1509002909 1508969682
2080 07 04 22 01 05 jupiter VLT1 1 0.0523598333333333 3487367068 3487411958
2060 03 12 20 41 24 venus HPROV1.52 1 0.0523598333333333 2846389457 2846352416
1997 11 25 17 59 09 sun BLOEMF 1 0 880513954 880563064
2013 09 04 17 59 16 mercury SANPM83 -1 0 1378303697 1378261707
1998 12 25 21 52 48 neptune ARECIBO 1 -0.0523598333333333 914675615 914630882
2000 05 12 06 06 41 neptune TOLOLO4M -1 0 958101205 958063465
2076 07 09 10 16 09 moon STROMLO74 -1 0.349065555555556 3361491174 3361433674
2028 04 09 18 41 42 uranus VLA 1 0.0523598333333333 1838993356 1838955739
2060 07 18 17 07 52 moon MAGELLAN1 -1 0 2857343487 2857388585
2057 07 12 11 16 35 mars GEMINIS 1 -0.174532777777778 2762172135 2762219315
2057 03 12 06 16 24 saturn KPNO90 0 0.349065555555556 2751641093 2751588566
2017 06 12 18 16 24 mars JCMT 1 -0.104719666666667 1497370404 1497335154
2049 06 23 02 01 35 mars LPO1 0 0.0523598333333333 2508046260 2508007912
1985 06 23 06 11 07 jupiter MCDONLD2.7 1 -0.349065555555556 488430503 488394016
2050 03 07 20 42 13 mercury MAGELLAN1 1 -0.349065555555556 2530344097 2530313428
2074 11 23 14 09 42 uranus ESONTT 0 0 3310200774 3310250603
1986 02 07 17 42 42 neptune HARVARD -1 0 508152405 508098755
2030 07 12 10 07 24 sun VLA 0 0.104719666666667 1910090695 1910051255
2026 04 04 06 09 07 saturn KECK2 -1 0.0523598333333333 1775232663 1775274299
1996 08 14 03 11 52 saturn USSR6 1 0 840046079 840003179
2077 04 04 06 59 48 mercury SANPM83 -1 -0.0174532777777778 3384681761 3384728038
1991 05 07 19 05 52 mars LPO4.2 0 0.0174532777777778 673613095 673662321
2059 02 23 03 07 42 sun GEMINIS 0 0 2813221994 2813181580
2033 11 19 06 05 05 moon NOBEYAMA 1 0 2016041481 2016081280
2033 02 01 07 05 09 venus SUGARGROVE 1 0.0523598333333333 1990880867 1990922605
2030 11 14 03 13 35 venus MMT -1 0.0349065555555556 1920810798 1920846783
2066 09 23 01 35 01 mars TOLOLO1.5M 0 -0.104719666666667 3052460773 3052419748
1980 01 14 19 39 19 mercury SUGARGROVE 0 -0.0174532777777778 316700699 316734830
2021 02 25 16 35 09 venus MAUNAK88 -1 0 1614270497 1614225025
2013 01 07 19 24 06 mars IRTF -1 0.0349065555555556 1357584021 1357536355
1984 03 07 17 41 35 uranus STROMLO74 0 0 447511381 447562586
2031 05 01 09 39 42 jupiter KOSMA3M -1 -0.104719666666667 1935352311 1935387777
2038 12 07 19 16 07 mars APO3.5 0 0 2175339858 2175376105
2055 11 14 03 07 59 jupiter NOBEYAMA -1 0 2709766664 2709714836
2062 12 14 19 59 59 uranus LPO4.2 0 0.174532777777778 2933388866 2933335094
2035 03 25 22 06 06 moon VLA 1 0.0523598333333333 2058493207 2058532307
1983 11 09 18 16 19 mars STEWARD90 1 0 437305633 437263472
2044 02 01 23 39 39 neptune MCDONLD2.1 -1 0 2337962999 2337923710
1989 02 18 02 13 52 venus HPROV1.52 -1 0.0174532777777778 603699644 603734086
2048 07 14 06 11 42 saturn IRTF 0 -0.0349065555555556 2478312455 2478352553
2063 05 12 03 16 01 mercury PALOMAR48 1 -0.0349065555555556 2946195767 2946242560
2076 06 01 09 24 35 saturn CSO 0 0.174532777777778 3358219826 3358253221
2031 06 19 20 41 35 saturn ESO3.6 0 0.0523598333333333 1939632694 1939668000
2046 08 23 12 52 01 mars JODRELL1 -1 0 2418555976 2418585725
2008 12 14 02 06 13 mars PALOMAR48 -1 -0.349065555555556 1229172938 1229134167
2052 09 25 13 07 52 uranus TIDBINBLA 0 -0.0523598333333333 2610905597 2610864170
2064 01 04 18 35 13 neptune KPNO158 -1 0 2966627745 2966677831
2070 12 23 01 06 13 neptune EFFELSBERG -1 0.0523598333333333 3186492237 3186460457
2005 01 25 20 42 59 sun KISO -1 0.104719666666667 1106605935 1106638300
2045 02 19 04 16 48 mars HARVARD -1 0.0349065555555556 2371034197 2371073421
2056 10 12 07 11 39 saturn LPO1 1 -0.174532777777778 2738599270 2738563015
1995 03 25 10 42 41 mercury MAGELLAN2 1 0.174532777777778 796213132 796166242
2022 12 01 23 05 52 venus UKST 0 0 1669923265 1669973858
1988 05 04 03 48 07 uranus OKAYAMA -1 0 578669539 578704214
2021 05 07 20 19 09 sun TAUTSCHM 0 0.0174532777777778 1620445642 1620412280
2011 12 01 23 39 13 uranus BOSQALEGRE 1 -0.0349065555555556 1322846792 1322804958
2034 02 18 09 41 24 sun ANU2.3 -1 0.0349065555555556 2023819096 2023864592
1984 06 12 07 06 39 sun GBVA140 0 0 455882383 455848705
2046 10 18 20 16 24 neptune ANU2.3 1 0 2423555821 2423508069
2021 11 01 08 24 48 sun KOTTAMIA 0 -0.0174532777777778 1635739540 1635779190
2035 11 14 03 06 01 jupiter TOLOLO1.5M 1 -0.0174532777777778 2078692169 2078645395
2080 03 25 05 01 06 jupiter BOSQALEGRE 1 -0.0174532777777778 3478573251 3478622179
2077 12 07 23 13 13 uranus IRTF 1 0 3406216547 3406169411
2076 03 04 20 39 05 moon KECK1 -1 -0.349065555555556 3350559721 3350522458
2034 02 01 10 16 16 venus JCMT 1 0 2022427710 2022468070
2014 06 19 10 48 07 sun MOPRA 1 -0.0174532777777778 1403211646 1403248160
2036 01 04 20 59 06 uranus SUGARGROVE -1 -0.0349065555555556 2083009783 2083063832
1994 05 14 03 42 24 jupiter USSR600 1 -0.349065555555556 768920877 768971355
2076 11 04 01 09 06 mercury TAUTNBG 0 0 3371701417 3371644781
2017 09 04 22 16 52 sun ARECIBO 0 0 1504606607 1504564495
2008 04 09 08 42 35 jupiter KPNO158 -1 0 1207644395 1207680569
2026 02 14 23 09 42 neptune ESOSCHM 0 0.349065555555556 1771079907 1771112225
2067 01 04 08 24 06 jupiter DUPONT -1 0 3061269648 3061319294
2065 05 07 02 13 13 mars GEMINIS 0 0 3008887845 3008851708
2075 12 04 10 48 39 saturn SUGARGROVE -1 -0.349065555555556 3342596914 3342644213
2045 09 09 19 01 19 sun UKIRT -1 -0.174532777777778 2388583738 2388546570
2019 04 14 19 01 19 venus NOBEYAMA -1 0.0349065555555556 1555182847 1555223318
1984 04 14 15 24 59 sun ESOSCHM 0 -0.0174532777777778 450788410 450829538
1987 12 14 09 48 24 uranus BLOEMF -1 0 566451261 566414881
1993 07 07 20 41 42 jupiter GEMINIS 0 0 742061010 742104516
1990 07 04 07 07 07 moon VLT1 0 -0.104719666666667 647117247 647082022
1989 05 18 09 01 11 mars JCMT -1 0 611434394 611482043
2072 01 01 15 13 19 mercury ESO3.6 0 0.104719666666667 3218865924 3218912455
2025 04 01 14 09 39 moon LICK120 -1 0 1743433643 1743486165
2012 07 04 18 13 01 mars TAUTSCHM 0 -0.174532777777778 1341393567 1341444426
2044 12 18 03 24 24 moon QUEBEC1.6 1 0.0523598333333333 2365677574 2365702300
2034 02 12 10 05 35 sun LPO2.5 -1 0.104719666666667 2023345496 2023295154
2036 07 19 11 07 07 sun FLAGSTF61 -1 0 2099997059 2100047716
2003 09 12 21 59 48 neptune ESO3.6 0 0.104719666666667 1063396716 1063441204
2075 07 19 21 13 35 saturn KISO 0 0 3330830678 3330780054
1986 04 14 04 35 42 moon SANPM83 0 -0.104719666666667 513878479 513846332
1991 12 12 09 24 35 moon HARVARD 1 0.0174532777777778 692553977 692594308
2033 01 14 23 19 05 moon SUGARGROVE 0 0.0174532777777778 1989353158 1989314437
2006 01 25 21 09 07 venus QUEBEC1.6 0 -0.0174532777777778 1138185456 1138221488
2034 07 12 10 19 35 sun STEWARD90 -1 -0.174532777777778 2036230736 2036287486
2034 09 07 00 42 39 uranus ESO3.6 0 0 2041226179 2041176389
1992 10 18 11 07 16 neptune STEWARD90 1 -0.0174532777777778 719437141 719473951
1988 07 04 07 42 42 neptune PALOMAR48 1 0.0174532777777778 584073559 584022389
1997 10 04 09 05 59 mercury BOSQALEGRE 1 0 875958174 876001494
2049 11 19 06 07 48 moon DUNLAP74 -1 -0.349065555555556 2520908668 2520878706
2009 08 18 21 48 11 neptune BOSQALEGRE -1 0.0174532777777778 1250545998 1250592578
2040 02 14 04 41 35 mars BOSQALEGRE 1 0.349065555555556 2212870359 2212889713
2006 03 09 00 41 05 mercury KECK1 -1 0.0174532777777778 1141836802 1141793502
1981 01 04 04 39 39 saturn DUNLAP74 -1 0.174532777777778 347350056 347385700
2011 11 14 12 35 42 neptune DAO72 1 0 1321306793 1321342844
2054 08 01 16 52 52 uranus GEMINIS -1 0.104719666666667 2669208613 2669162781
2077 02 07 05 48 01 neptune MTLEMMON60 -1 0 3379878115 3379841677
2073 12 09 22 59 48 uranus VLT2 -1 -0.174532777777778 3280040047 3280008504
2002 03 25 18 06 01 sun VLT3 0 0 1017053460 1017096225
1982 11 04 03 39 05 moon MCDONLD2.1 -1 0.104719666666667 405225983 405182952
2032 06 12 16 48 16 moon QUEBEC1.6 1 -0.174532777777778 1970746085 1970713041
2000 12 14 04 11 41 uranus EFFELSBERG -1 0 976704423 976737662
2003 05 04 06 35 39 neptune UKST 0 0.0523598333333333 1052056227 1052016444
2007 12 25 21 01 19 moon MTHOP1.5 1 0 1198636268 1198687226
2046 06 14 12 41 11 venus MMT 0 -0.349065555555556 2412578556 2412553057
2019 01 19 03 48 07 saturn CSO 0 -0.0174532777777778 1547827106 1547866689
2034 10 04 18 42 52 mercury CFHT 1 0 2043681340 2043635984
1998 08 14 10 42 48 saturn IRTF 1 0.349065555555556 903175643 903124087
2071 03 25 02 59 39 mars CAMB1MILE 1 0 3194485390 3194520266
2002 07 14 09 16 39 jupiter PALOMAR200 1 -0.0174532777777778 1026651951 1026702803
2070 09 09 00 59 09 mars KECK1 1 0.174532777777778 3177512282 3177463643
1989 02 18 18 16 19 mars STEWARD90 0 0.174532777777778 603829983 603786539
2021 06 25 20 39 06 sun LPO4.2 0 -0.349065555555556 1624681716 1624658407
1986 07 19 10 07 59 jupiter PALOMAR48 1 0.349065555555556 522228047 522171931
1988 09 23 21 41 35 neptune PARKES -1 0 590981329 591031801
2048 01 12 19 41 06 uranus FCRAO 0 0.174532777777778 2462495689 2462449718
2018 04 01 14 59 48 saturn GEMINIS -1 0.0523598333333333 1522555425 1522517111
2067 04 14 10 52 19 moon ESONTT -1 -0.349065555555556 3070001650 3069963753
2021 11 09 19 05 05 mars ARECIBO 0 -0.104719666666667 1636449698 1636493797
1982 11 12 18 52 39 venus MTLEMMON60 1 0.349065555555556 406050899 405988806
2035 08 01 17 01 42 sun LPO1 0 0 2069563005 2069611122
2074 03 23 13 19 42 mars AAT -1 -0.0174532777777778 3289010515 3288962145
2048 11 19 02 19 59 mercury HPROV1.52 0 0 2489381986 2489328414
1989 07 18 08 05 07 mercury MTLEMMON60 1 0 616768211 616818871
1999 05 09 08 09 01 jupiter APO3.5 -1 0 926161056 926206265
2014 06 23 02 06 07 moon MTLEMMON60 1 0 1403516425 1403565683
1981 03 23 17 41 07 sun BOSQALEGRE 0 0.104719666666667 354192943 354232399
1980 01 01 15 07 06 sun HARVARD 1 0.174532777777778 315667824 315605333
2015 08 14 19 05 48 uranus KPNO36FT 1 -0.349065555555556 1439608395 1439579257
2070 06 18 19 05 48 saturn HPROV1.93 0 -0.349065555555556 3170314667 3170369956
2020 04 04 00 24 41 neptune PALOMAR200 -1 0.349065555555556 1585923164 1585952816
2039 08 14 02 59 39 mercury VLT2 1 -0.0174532777777778 2196929777 2196969215
1989 11 14 00 52 09 mercury LPO2.5 -1 0 626946257 626984361
2079 04 09 14 39 01 neptune MMT -1 0 3448207396 3448256740
2025 06 14 18 24 42 venus FLAGSTF61 -1 0 1749894513 1749855349
2070 08 07 10 42 59 mercury GEMINIS -1 -0.0349065555555556 3174552525 3174595027
2064 04 09 11 42 09 moon HPROV1.52 0 0 2974932081 2974964192
2053 07 25 12 01 09 sun HARVARD 0 -0.0523598333333333 2637047915 2637016048
2074 03 25 20 41 07 jupiter DUNLAP74 0 0.174532777777778 3289230340 3289276032
2028 10 01 17 52 52 saturn JCMT 1 0.0349065555555556 1854079225 1854037258
1999 04 18 13 35 05 uranus IRTF 1 0.0349065555555556 924524574 924477472
2031 03 09 22 19 11 mercury IRTF -1 0.349065555555556 1930848530 1930795302
2062 10 18 21 42 41 jupiter TIDBINBLA -1 0 2928416958 2928370078
2043 07 14 18 48 39 neptune EFFELSBERG 1 -0.104719666666667 2320527090 2320583572
2041 12 09 07 07 07 sun SUGARGROVE -1 -0.104719666666667 2270116366 2270154328
1993 03 01 03 07 52 jupiter MTLEMMON60 1 0.0523598333333333 730958509 730998721
2002 02 14 11 48 05 mercury CAMB5KM -1 -0.0174532777777778 1013666859 1013612266
2060 03 25 06 09 07 sun TOLOLO1.5M -1 0.104719666666667 2847352880 2847392198
1992 11 01 13 13 52 saturn FCRAO -1 -0.174532777777778 720551968 720593621
1999 08 12 10 35 24 neptune QUEBEC1.6 -1 0.0523598333333333 934414013 934444739
2018 03 25 07 16 09 mars SANPM83 0 0 1521967837 1522003699
1987 01 25 16 13 11 venus DAO72 -1 0.104719666666667 538579630 538519666
2073 06 01 13 42 41 uranus MMT 1 0 3263601878 3263551314
1988 11 01 05 05 48 venus MTLEMMON60 1 0 594384713 594427971
2066 09 07 23 24 19 sun OKAYAMA 1 -0.349065555555556 3051198477 3051169128
2040 03 25 13 16 52 jupiter EFFELSBERG -1 0 2216220845 2216265688
1989 09 12 16 01 42 mercury ANU2.3 1 0 621636562 621681932
2071 01 01 05 16 19 saturn ATCA -1 -0.0174532777777778 3187263002 3187309301
2052 01 01 08 41 07 mercury KISO 1 0.349065555555556 2587763453 2587782335
2007 06 18 00 19 11 saturn USSR600 -1 0 1182060344 1182110925
2023 09 01 05 01 52 mars CAMB5KM 0 -0.0174532777777778 1693553276 1693510556
2062 10 07 11 48 59 jupiter KPNO90 1 -0.174532777777778 2927524508 2927490825
2078 02 04 03 06 01 moon GEMINIS 0 0 3411171269 3411129353
2044 05 18 21 11 24 neptune ATCA 1 -0.0174532777777778 2347298545 2347251996
2002 07 14 12 52 13 saturn SANPM83 0 -0.0349065555555556 1026642962 1026694044
1989 10 07 04 35 48 venus FLAGSTF61 0 0 623697075 623732390
2027 11 01 12 07 16 mercury KISO 0 0.0523598333333333 1825099160 1825052038
1981 07 04 19 52 39 saturn SUBARU 1 0.349065555555556 363136326 363169337
2005 05 07 00 42 48 mercury MCDONLD2.7 0 -0.0523598333333333 1115463223 1115423381
2023 09 18 02 41 35 neptune CSO 1 0 1695011209 1695053832
2062 06 12 07 41 59 neptune ST.ANDREWS 1 0 2917395855 2917370582
2042 03 14 14 52 39 sun JCMT 1 -0.0174532777777778 2278427402 2278470723
1996 01 07 20 13 24 saturn APO3.5 0 -0.0523598333333333 821035448 821078420
1993 01 09 04 07 06 venus MAGELLAN2 0 -0.104719666666667 726584886 726547848
2014 08 12 20 59 05 sun ESO3.6 -1 -0.0349065555555556 1407841964 1407795775
2060 07 23 10 42 16 jupiter IRTF -1 0.104719666666667 2857729575 2857773548
2016 01 19 18 13 41 jupiter VLT3 0 -0.0349065555555556 1453256455 1453213556
2060 12 09 09 24 41 sun VLA -1 -0.104719666666667 2869738598 2869777723
2069 10 12 04 13 16 saturn QUEBEC1.6 -1 -0.0523598333333333 3148711146 3148755126
2080 10 23 23 35 16 moon STEWARD90 1 -0.0349065555555556 3497035127 3496989088
2056 08 14 02 52 16 uranus USSR6 1 0 2733462291 2733502662
2025 07 19 15 41 16 neptune FCRAO -1 -0.0174532777777778 1752894395 1752937957
2003 01 18 21 24 05 venus ANU2.3 0 0.104719666666667 1042907810 1042953189
2053 02 23 00 41 13 venus UKST -1 0.349065555555556 2623870893 2623820140
2029 05 14 16 19 52 venus MCDONLD2.7 1 -0.104719666666667 1873542130 1873509331
1999 01 09 07 06 01 venus IRTF 1 0 915905605 915945210
2038 05 07 20 11 42 mercury KPNO36FT 1 0.104719666666667 2156933269 2156888682
2013 11 12 10 11 42 sun CFHT -1 0 1384187533 1384227573
2030 08 12 15 39 19 neptune UKST -1 0.0523598333333333 1912766043 1912720175
2033 11 19 15 11 09 neptune SUBARU 1 -0.0523598333333333 2016061898 2016107449
2049 11 12 15 42 07 uranus MAGELLAN1 -1 -0.0174532777777778 2520312312 2520268522
2072 09 09 02 11 24 uranus KECK1 -1 -0.0349065555555556 3240602723 3240556416
2072 11 04 06 24 39 saturn SANPM83 -1 0 3245408356 3245447137
1992 03 01 01 19 42 moon TOLOLO4M -1 -0.349065555555556 699340387 699403350
2058 06 07 12 05 05 venus ANU2.3 0 0 2790719391 2790668943
2030 03 25 04 42 48 moon NOBEYAMA 0 -0.0349065555555556 1900682239 1900629022
2068 03 12 22 52 35 venus VLA 1 0.174532777777778 3098880625 3098837240
2043 06 12 00 48 07 saturn DUPONT 0 -0.0174532777777778 2317663929 2317711166
2071 01 09 04 48 42 venus PALOMAR60 1 0.349065555555556 3188052854 3188076482
2008 04 04 20 05 39 neptune KECK1 -1 0.0523598333333333 1207316795 1207269664
2070 12 14 10 05 59 mars ANU2.3 0 -0.0349065555555556 3185805889 3185771008
2068 02 19 06 07 39 uranus MAGELLAN2 -1 0.0174532777777778 3096852658 3096815315
2030 02 01 18 11 16 venus CFHT 1 0.349065555555556 1896278788 1896221706
1995 06 04 15 11 16 moon JODRELL1 1 0 802350038 802309461
2070 06 12 14 09 24 sun KISO -1 0.0174532777777778 3169741396 3169792550
2061 06 19 12 24 13 moon LOWELL72 0 0.174532777777778 2886419155 2886376679
1991 04 23 05 48 05 saturn HARVARD -1 0 672302137 672336428
1980 07 14 09 52 35 mars DUNLAP74 1 -0.349065555555556 332431496 332487876
2045 06 14 17 59 24 mercury KPNO36FT 0 -0.0349065555555556 2381060324 2381112909
1985 02 04 02 11 01 uranus MCDONLD2.1 1 -0.104719666666667 476357337 476397187
2068 05 07 20 39 13 sun ESO3.6 0 0 3103615154 3103653565
2024 07 18 22 07 11 mars QUEBEC1.6 1 -0.0349065555555556 1721365565 1721420248
2009 03 01 14 11 16 saturn LPO1 0 -0.0523598333333333 1235935643 1235895927
2041 07 04 10 09 41 jupiter PALOMAR60 0 0 2256583118 2256537656
1982 05 14 16 41 01 jupiter USSR600 1 0 390321052 390272733
2074 03 04 18 59 24 saturn TOLOLO1.5M -1 -0.0349065555555556 3287359114 3287409080
1993 11 04 22 42 11 moon CATALINA61 0 0 752475209 752436201
1980 11 01 21 16 24 saturn QUEBEC1.6 -1 0 341914965 341958061
2073 01 01 04 39 39 sun TOLOLO4M 1 0 3250489802 3250539837
2065 08 09 03 24 48 moon QUEBEC1.6 1 0.0523598333333333 3017070542 3017098496
2025 03 01 04 41 52 venus OKAYAMA 1 0 1740867902 1740828356
2032 11 01 13 42 19 mercury USSR600 1 -0.174532777777778 1982983682 1982936649
2039 07 23 13 13 41 venus DUNLAP74 0 0 2195036052 2194997527
2054 01 14 18 13 52 venus MTLEMMON60 1 0 2652093271 2652044693
2078 01 25 02 24 11 moon VLT1 1 0.104719666666667 3410375663 3410319651
2073 11 01 21 59 59 mars LOWELL72 -1 0.104719666666667 3276747541 3276793051
2070 04 01 03 24 13 jupiter JODRELL1 0 0.0174532777777778 3163557256 3163514529
2033 11 25 08 16 09 mars QUEBEC1.6 0 0 2016551957 2016501587
2064 11 09 01 42 07 moon PALOMAR200 -1 0 2993379292 2993416072
1987 04 12 17 41 19 neptune FLAGSTF61 -1 -0.0349065555555556 545211445 545161685
2024 09 12 11 48 07 neptune JODRELL1 1 0.0349065555555556 1726167613 1726207987
2030 09 18 16 41 48 neptune DUPONT 0 0.174532777777778 1916008365 1915958855
2011 03 04 23 48 07 saturn HARVARD 1 0.104719666666667 1299289586 1299327049
1995 10 19 04 35 11 jupiter MTLEMMON60 1 -0.0349065555555556 814122625 814159838
1999 04 01 05 19 48 saturn PALOMAR60 0 0 922978153 922938442
2042 12 19 19 13 52 jupiter PALOMAR200 0 0.0174532777777778 2302610540 2302645528
2022 12 23 10 35 09 jupiter MMT -1 -0.0349065555555556 1671736471 1671780356
1981 11 09 13 39 06 sun PALOMAR48 1 0 374163455 374201081
2043 10 23 04 01 09 uranus PALOMAR48 1 0.0523598333333333 2329203118 2329249255
2001 02 04 12 11 19 neptune GEMININ 1 -0.104719666666667 981302337 981345337
2023 12 23 15 11 09 sun APO3.5 1 -0.0523598333333333 1703425889 1703376821
2042 10 19 15 06 09 saturn KECK2 1 -0.0523598333333333 2297350682 2297393225
1988 08 25 08 01 42 mercury KISO 0 0 588462730 588507273
2017 04 04 04 05 39 mercury CATALINA61 -1 0.0523598333333333 1491228447 1491274561
2045 07 12 23 09 11 uranus CSO 1 0 2383583009 2383542362
2002 02 12 20 09 09 mars JCMT 0 -0.0349065555555556 1013542721 1013501808
2025 11 12 11 11 35 uranus PALOMAR48 0 0.0174532777777778 1762910552 1762959605
2010 01 18 14 59 41 sun TOLOLO1.5M -1 0.0349065555555556 1263809607 1263771234
2029 02 01 12 05 24 venus ESO3.6 0 -0.0174532777777778 1864631556 1864681252
2066 10 14 04 01 05 sun MTHOP1.5 0 0.0174532777777778 3054288904 3054242661
2058 05 09 16 11 06 uranus MCDONLD2.7 0 0.104719666666667 2788215119 2788165906
1983 08 04 02 59 05 sun ESONTT 0 0.0523598333333333 428845437 428795617
2049 05 14 09 42 16 moon KPNO90 -1 0.349065555555556 2504564825 2504593917
1988 09 04 08 19 09 venus PALOMAR200 0 -0.104719666666667 589368157 589334974
2047 06 19 13 41 52 uranus PALOMAR200 0 0 2444579215 2444539687
2079 11 23 02 39 11 mars JODRELL1 0 -0.349065555555556 3467926404 3467897146
2043 10 12 21 13 19 sun FCRAO 1 0 2328347067 2328300462
2069 01 25 00 16 19 venus KPNO36FT -1 0 3126261021 3126297322
2039 01 23 03 24 01 mercury UKST -1 0 2179336496 2179300011
2074 01 12 20 16 11 venus KOSMA3M 0 0.0349065555555556 3283055431 3282998955
1997 09 18 04 06 41 uranus ARECIBO 0 -0.0349065555555556 874524498 874565402
2034 07 19 14 24 42 moon USSR600 0 0 2036901173 2036947653
2013 07 12 00 24 09 uranus HPROV1.93 1 0.104719666666667 1373671021 1373625929
2019 04 14 09 59 39 mercury BLOEMF 1 -0.0174532777777778 1555295123 1555253337
2010 06 07 03 16 16 saturn HPROV1.93 1 -0.0349065555555556 1275912427 1275958253
2048 10 12 20 24 09 mercury MTEKAR 1 0.104719666666667 2486176105 2486215366
1991 05 01 15 48 42 uranus ST.ANDREWS 1 -0.349065555555556 673136805 673180725
1994 07 12 02 35 59 venus ESOSCHM 1 0.174532777777778 774024843 774058961
2034 05 04 11 41 11 neptune SAAO74 -1 0.349065555555556 2030331722 2030274868
1988 10 04 23 35 35 neptune UKIRT 0 0 592007577 592046593
2080 05 14 00 19 09 saturn JCMT -1 0.174532777777778 3482824584 3482859259
2030 04 04 17 13 35 neptune MTHOP1.5 -1 0.0523598333333333 1901539271 1901495379
2046 08 14 19 35 16 mars NOBEYAMA 1 0.104719666666667 2417916614 2417949302
2011 06 19 09 01 24 uranus SUGARGROVE -1 -0.174532777777778 1308459074 1308422549
2065 03 25 18 24 39 uranus LPO4.2 1 -0.0174532777777778 3005249584 3005287694
2036 10 14 05 13 09 uranus DUPONT -1 0 2107488666 2107525420
2055 06 07 22 19 01 jupiter KISO 0 -0.0523598333333333 2695979079 2696015587
2011 06 18 12 16 35 jupiter KISO 1 -0.0349065555555556 1308414216 1308462533
2079 05 01 13 19 52 venus PALOMAR60 0 -0.174532777777778 3450177315 3450150309
2022 11 07 11 05 05 sun HPROV1.52 1 0.104719666666667 1667891093 1667835464
2002 10 04 10 16 09 venus KPNO36FT -1 0.104719666666667 1033663825 1033695863
2078 03 25 07 59 01 mercury TOLOLO4M 0 0 3415427703 3415384999
2034 12 23 02 11 39 saturn TAUTSCHM 1 0.349065555555556 2050516890 2050469999
2030 04 19 13 35 39 saturn VLT1 1 0 1902919845 1902872866
1986 12 23 01 35 09 neptune MAGELLAN1 1 0 535716400 535765789
2073 06 23 14 07 48 sun MAGELLAN2 0 0.349065555555556 3265450342 3265473495
1994 09 12 06 19 24 venus VLT2 -1 -0.0174532777777778 779287383 779334813
2033 10 01 03 05 05 jupiter KECK1 1 -0.349065555555556 2011826034 2011790984
2010 06 09 03 06 59 venus TAUTNBG -1 -0.104719666666667 1275972233 1276037377
1997 12 12 18 41 06 uranus CAMB1MILE 1 -0.104719666666667 882007742 881957137
2004 02 01 03 16 05 mercury GEMININ -1 0 1075563867 1075603009
1980 06 18 07 35 13 jupiter KOSMA3M 0 0 330168071 330130541
1982 04 14 06 09 19 neptune ARECIBO 1 -0.349065555555556 387683005 387646975
1999 10 07 18 07 39 sun HPROV1.93 1 0.104719666666667 939363661 939400077
1988 03 14 23 59 24 mercury SAAO74 1 0 574396453 574443851
2043 05 01 19 07 59 sun LPO2.5 0 -0.0523598333333333 2314160367 2314123044
2062 01 19 10 16 39 uranus LPO2.5 -1 0 2904864764 2904817182
1986 03 01 08 06 24 mars STEWARD90 0 0.174532777777778 510054141 510083668
2067 01 09 09 05 24 neptune TOLOLO1.5M 1 0 3061837835 3061874331
2023 04 23 08 16 09 mercury SUBARU 1 -0.0349065555555556 1682267229 1682314748
2065 11 23 09 48 39 sun TIDBINBLA -1 -0.0523598333333333 3026140366 3026192914
2016 01 18 19 41 52 neptune CAMB5KM -1 -0.174532777777778 1453106008 1453065012
2074 02 09 10 41 07 moon HPROV1.93 1 -0.104719666666667 3285410347 3285469047
2073 10 07 19 39 35 uranus MOPRA -1 -0.104719666666667 3274558587 3274612953
2029 09 09 13 48 59 venus ESO3.6 -1 0 1883652416 1883612413
2058 10 18 09 48 52 mars DUPONT 1 0 2802235537 2802187573
2050 02 14 04 19 42 jupiter FCRAO -1 0 2528394139 2528361123
1990 09 18 02 24 41 moon LPO4.2 -1 -0.349065555555556 653543280 653600905
2075 10 14 06 11 11 saturn VLT3 0 0 3338284897 3338246705
2020 02 19 00 24 48 mars IRTF 1 -0.0174532777777778 1582117618 1582156990
1997 10 07 14 09 07 neptune VLT2 1 0.174532777777778 876246021 876288176
2040 11 07 15 07 59 venus FCRAO 1 0 2236000036 2235944368
2042 05 23 07 09 11 saturn KPNO84 -1 -0.0523598333333333 2284414364 2284370181
2073 04 23 04 11 07 neptune MAGELLAN1 1 0 3260190598 3260227539
1995 11 09 17 13 42 mercury KOSMA3M -1 -0.174532777777778 815891837 815935457
1990 11 04 17 01 13 mars EFFELSBERG 0 0 657739310 657710805
2007 02 09 18 59 01 jupiter MAGELLAN2 1 0.104719666666667 1171087104 1171132807
1998 01 19 10 05 13 sun KOTTAMIA 0 0.0349065555555556 885186170 885222248
2064 02 04 16 05 41 moon KOSMA3M -1 0 2969286029 2969335708
2018 09 23 16 05 39 venus BLOEMF 1 0 1537767088 1537729558
2054 10 01 17 42 35 jupiter ST.ANDREWS -1 0 2674468117 2674408953
2065 12 01 17 09 41 venus ESONTT 0 -0.104719666666667 3026882985 3026935705
2053 10 23 06 48 13 uranus KPNO36FT 0 0.174532777777778 2644835808 2644786396
2019 07 25 05 48 06 saturn PALOMAR60 -1 -0.0523598333333333 1564018411 1563969803
2041 11 23 11 59 52 saturn LPO1 -1 0.349065555555556 2268802715 2268745551
2080 03 25 12 24 48 uranus KECK2 1 -0.0349065555555556 3478596978 3478637625
2022 11 25 07 48 48 jupiter KPNO90 0 -0.0174532777777778 1669324135 1669367242
1992 05 25 16 19 13 moon SAAO74 0 -0.0349065555555556 706836698 706793095
1991 03 18 01 16 05 mars KECK1 1 0 669331470 669293013
2007 12 25 01 41 07 saturn TAUTNBG 0 0.349065555555556 1198537352 1198571034
1984 04 25 05 42 52 venus PALOMAR200 1 0 451744579 451790118
2073 07 12 08 01 06 neptune VLT1 -1 0.0523598333333333 3266998395 3267034747
1991 11 01 13 19 35 jupiter GEMINIS 0 -0.349065555555556 688973779 689025523
2059 07 25 23 09 59 jupiter VLT1 -1 0 2826340829 2826379756
1990 10 25 05 06 16 jupiter BOSQALEGRE -1 0 656747422 656785074
2040 05 18 15 35 35 uranus CFHT -1 -0.174532777777778 2220895740 2220948151
2056 11 04 19 13 48 moon MCDONLD2.1 1 -0.0523598333333333 2740646954 2740603453
2015 09 18 10 19 11 uranus TAUTSCHM 0 -0.174532777777778 1442595306 1442564387
2005 09 09 13 59 35 venus AAT 1 0.349065555555556 1126309022 1126344277
2019 07 01 12 16 39 venus VLA 1 0 1562066140 1562031032
2045 01 25 20 48 24 uranus MCDONLD2.1 1 0 2369008334 2369055394
2065 01 18 12 24 48 neptune GBVA140 1 0 2999534307 2999586410
2003 11 23 23 48 24 moon TAUTSCHM -1 -0.104719666666667 1069564429 1069601441
2056 12 07 21 52 19 mercury MTLEMMON60 1 0 2743516030 2743464758
2008 03 19 08 24 41 mercury MOPRA -1 0 1205864445 1205911053
2013 05 23 12 13 24 sun JCMT 1 0.174532777777778 1369326762 1369368288
2012 12 25 11 59 05 moon KPNO84 0 0 1356475297 1356436696
2054 11 23 21 41 52 uranus PARKES -1 -0.0523598333333333 2679063535 2679023609
2058 10 04 04 16 06 moon VLT3 0 0.0349065555555556 2800917983 2800958065
2037 08 12 07 13 01 jupiter CAMB1MILE 0 0.349065555555556 2133663022 2133704059
1991 08 25 04 07 48 jupiter ATCA 0 0 683065077 683104303
2016 04 18 23 41 19 saturn MAGELLAN2 -1 -0.0523598333333333 1460940732 1460991477
1984 03 23 13 39 39 moon ATCA -1 -0.104719666666667 448891115 448857224
2013 05 01 18 48 42 sun LPO1 0 0.174532777777778 1367392894 1367434551
2076 05 18 01 52 05 uranus HPROV1.52 0 0.174532777777778 3356985498 3357008790
2059 10 14 14 11 06 mars KECK1 0 0 2833384995 2833337909
2029 03 04 04 35 24 saturn ARECIBO 0 0 1867326218 1867285036
2042 04 12 01 39 42 saturn LPO1 1 -0.0523598333333333 2280946014 2280901865
2033 08 09 03 06 16 mars TAUTNBG 0 -0.0523598333333333 2007131015 2007157377
2037 06 07 21 07 24 mercury DUNLAP74 -1 -0.0349065555555556 2127977390 2127945645
1999 08 01 06 35 11 moon JCMT 1 -0.349065555555556 933576978 933542087
2068 01 19 20 39 07 uranus KOTTAMIA -1 0.0174532777777778 3094163192 3094199114
1989 01 12 09 19 01 neptune DUPONT -1 0.349065555555556 600518943 600556601
2008 03 14 07 42 01 neptune STROMLO74 -1 -0.174532777777778 1205426674 1205394832
2061 08 18 09 41 06 mercury IRTF 1 0.0523598333333333 2891613326 2891655803
2048 06 23 01 01 09 jupiter HPROV1.93 -1 0.0174532777777778 2476407532 2476460794
2039 04 25 08 41 41 jupiter PARKES 1 0 2187403742 2187355742
1984 05 23 12 39 48 uranus EFFELSBERG 1 0 454190795 454219663
2049 03 23 12 07 39 jupiter FLAGSTF61 -1 0 2500047856 2500099333
2005 07 19 12 41 11 sun ANU2.3 -1 0 1121720617 1121757396
2020 11 01 04 35 59 uranus ST.ANDREWS 1 -0.349065555555556 undef undef
