package TableData::Locale::JP::City::MIC;

use strict;
use utf8;
use warnings;

use Role::Tiny::With;
with 'TableDataRole::Source::CSVInDATA';

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2023-02-07'; # DATE
our $DIST = 'TableDataBundle-Locale-JP-City'; # DIST
our $VERSION = '0.002'; # VERSION

our %STATS = ("num_columns",5,"num_rows",1916); # STATS

1;
# ABSTRACT: List of cities in Japan (from website of Ministry of Internal Affairs and Communications, https://www.soumu.go.jp/)

=pod

=encoding UTF-8

=head1 NAME

TableData::Locale::JP::City::MIC - List of cities in Japan (from website of Ministry of Internal Affairs and Communications, https://www.soumu.go.jp/)

=head1 VERSION

This document describes version 0.002 of TableData::Locale::JP::City::MIC (from Perl distribution TableDataBundle-Locale-JP-City), released on 2023-02-07.

=head1 SYNOPSIS

To use from Perl code:

 use TableData::Locale::JP::City::MIC;

 my $td = TableData::Locale::JP::City::MIC->new;

 # Iterate rows of the table
 $td->each_row_arrayref(sub { my $row = shift; ... });
 $td->each_row_hashref (sub { my $row = shift; ... });

 # Get the list of column names
 my @columns = $td->get_column_names;

 # Get the number of rows
 my $row_count = $td->get_row_count;

See also L<TableDataRole::Spec::Basic> for other methods.

To use from command-line (using L<tabledata> CLI):

 # Display as ASCII table and view with pager
 % tabledata Locale::JP::City::MIC --page

 # Get number of rows
 % tabledata --action count_rows Locale::JP::City::MIC

See the L<tabledata> CLI's documentation for other available actions and options.

=head1 DESCRIPTION

This module contains table data of list of cities in Japan, retrieved from
L<https://github.com/nobuf/list-of-cities-in-japan>, which in turn is retrieved
from the website of Ministry of Internal Affairs and Communications,
L<https://www.soumu.go.jp/>.

This version reflects the data retrieved on 2019-05-01.

=head2 About the data

=head1 TABLEDATA STATISTICS

 +-------------+-------+
 | key         | value |
 +-------------+-------+
 | num_columns | 5     |
 | num_rows    | 1916  |
 +-------------+-------+

The statistics is available in the C<%STATS> package variable.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/TableDataBundle-Locale-JP-City>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-TableDataBundle-Locale-JP-City>.

=head1 SEE ALSO

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>,
L<Pod::Weaver::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla- and/or Pod::Weaver plugins. Any additional steps required beyond
that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2023, 2022 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=TableDataBundle-Locale-JP-City>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut

__DATA__
id,prefecture_id,city_en,city_ja,special_district_ja
01100,01,Sapporo,札幌市,
01101,01,"Chūō-ku, Sapporo",札幌市,中央区
01102,01,"Kita-ku, Sapporo",札幌市,北区
01103,01,"Higashi-ku, Sapporo",札幌市,東区
01104,01,"Shiroishi-ku, Sapporo",札幌市,白石区
01105,01,"Toyohira-ku, Sapporo",札幌市,豊平区
01106,01,"Minami-ku, Sapporo",札幌市,南区
01107,01,"Nishi-ku, Sapporo",札幌市,西区
01108,01,"Atsubetsu-ku, Sapporo",札幌市,厚別区
01109,01,"Teine-ku, Sapporo",札幌市,手稲区
01110,01,"Kiyota-ku, Sapporo",札幌市,清田区
01202,01,Hakodate,函館市,
01203,01,Otaru,小樽市,
01204,01,Asahikawa,旭川市,
01205,01,"Muroran, Hokkaido",室蘭市,
01206,01,"Kushiro, Hokkaido",釧路市,
01207,01,"Obihiro, Hokkaido",帯広市,
01208,01,"Kitami, Hokkaido",北見市,
01209,01,"Yūbari, Hokkaido",夕張市,
01210,01,"Iwamizawa, Hokkaido",岩見沢市,
01211,01,Abashiri,網走市,
01212,01,"Rumoi, Hokkaido",留萌市,
01213,01,"Tomakomai, Hokkaido",苫小牧市,
01214,01,"Wakkanai, Hokkaido",稚内市,
01215,01,"Bibai, Hokkaido",美唄市,
01216,01,"Ashibetsu, Hokkaido",芦別市,
01217,01,"Ebetsu, Hokkaido",江別市,
01218,01,"Akabira, Hokkaido",赤平市,
01219,01,"Monbetsu, Hokkaido",紋別市,
01220,01,"Shibetsu, Hokkaido",士別市,
01221,01,"Nayoro, Hokkaido",名寄市,
01222,01,"Mikasa, Hokkaido",三笠市,
01223,01,"Nemuro, Hokkaido",根室市,
01224,01,"Chitose, Hokkaido",千歳市,
01225,01,"Takikawa, Hokkaido",滝川市,
01226,01,"Sunagawa, Hokkaido",砂川市,
01227,01,"Utashinai, Hokkaido",歌志内市,
01228,01,"Fukagawa, Hokkaido",深川市,
01229,01,"Furano, Hokkaido",富良野市,
01230,01,Noboribetsu,登別市,
01231,01,"Eniwa, Hokkaido",恵庭市,
01233,01,"Date, Hokkaido",伊達市,
01234,01,"Kitahiroshima, Hokkaido",北広島市,
01235,01,"Ishikari, Hokkaido",石狩市,
01236,01,"Hokuto, Hokkaido",北斗市,
01303,01,"Tōbetsu, Hokkaido",当別町,
01304,01,"Shinshinotsu, Hokkaido",新篠津村,
01331,01,"Matsumae, Hokkaido",松前町,
01332,01,"Fukushima, Hokkaido",福島町,
01333,01,"Shiriuchi, Hokkaido",知内町,
01334,01,"Kikonai, Hokkaido",木古内町,
01337,01,"Nanae, Hokkaido",七飯町,
01343,01,"Shikabe, Hokkaido",鹿部町,
01345,01,"Mori, Hokkaido",森町,
01346,01,"Yakumo, Hokkaido",八雲町,
01347,01,"Oshamambe, Hokkaido",長万部町,
01361,01,"Esashi, Hokkaido (Hiyama)",江差町,
01362,01,"Kaminokuni, Hokkaido",上ノ国町,
01363,01,"Assabu, Hokkaido",厚沢部町,
01364,01,"Otobe, Hokkaido",乙部町,
01367,01,"Okushiri, Hokkaido",奥尻町,
01370,01,"Imakane, Hokkaido",今金町,
01371,01,"Setana, Hokkaido",せたな町,
01391,01,"Shimamaki, Hokkaido",島牧村,
01392,01,"Suttsu, Hokkaido",寿都町,
01393,01,"Kuromatsunai, Hokkaido",黒松内町,
01394,01,"Rankoshi, Hokkaido",蘭越町,
01395,01,"Niseko, Hokkaido",ニセコ町,
01396,01,"Makkari, Hokkaido",真狩村,
01397,01,"Rusutsu, Hokkaido",留寿都村,
01398,01,"Kimobetsu, Hokkaido",喜茂別町,
01399,01,"Kyōgoku, Hokkaido",京極町,
01400,01,"Kutchan, Hokkaido",倶知安町,
01401,01,"Kyōwa, Hokkaido",共和町,
01402,01,"Iwanai, Hokkaido",岩内町,
01403,01,"Tomari, Hokkaido",泊村,
01404,01,"Kamoenai, Hokkaido",神恵内村,
01405,01,"Shakotan, Hokkaido",積丹町,
01406,01,"Furubira, Hokkaido",古平町,
01407,01,"Niki, Hokkaido",仁木町,
01408,01,"Yoichi, Hokkaido",余市町,
01409,01,"Akaigawa, Hokkaido",赤井川村,
01423,01,"Nanporo, Hokkaido",南幌町,
01424,01,"Naie, Hokkaido",奈井江町,
01425,01,"Kamisunagawa, Hokkaido",上砂川町,
01427,01,"Yuni, Hokkaido",由仁町,
01428,01,"Naganuma, Hokkaido",長沼町,
01429,01,"Kuriyama, Hokkaido",栗山町,
01430,01,"Tsukigata, Hokkaido",月形町,
01431,01,"Urausu, Hokkaido",浦臼町,
01432,01,"Shintotsukawa, Hokkaido",新十津川町,
01433,01,"Moseushi, Hokkaido",妹背牛町,
01434,01,"Chippubetsu, Hokkaido",秩父別町,
01436,01,"Uryū, Hokkaido",雨竜町,
01437,01,"Hokuryū, Hokkaido",北竜町,
01438,01,"Numata, Hokkaido",沼田町,
01452,01,"Takasu, Hokkaido",鷹栖町,
01453,01,"Higashikagura, Hokkaido",東神楽町,
01454,01,"Tōma, Hokkaido",当麻町,
01455,01,"Pippu, Hokkaido",比布町,
01456,01,"Aibetsu, Hokkaido",愛別町,
01457,01,"Kamikawa, Hokkaido",上川町,
01458,01,"Higashikawa, Hokkaido",東川町,
01459,01,"Biei, Hokkaido",美瑛町,
01460,01,"Kamifurano, Hokkaido",上富良野町,
01461,01,"Nakafurano, Hokkaido",中富良野町,
01462,01,"Minamifurano, Hokkaido",南富良野町,
01463,01,Shimukappu,占冠村,
01464,01,"Wassamu, Hokkaido",和寒町,
01465,01,"Kenbuchi, Hokkaido",剣淵町,
01468,01,"Shimokawa, Hokkaido",下川町,
01469,01,"Bifuka, Hokkaido",美深町,
01470,01,"Otoineppu, Hokkaido",音威子府村,
01471,01,"Nakagawa, Hokkaido",中川町,
01472,01,"Horokanai, Hokkaido",幌加内町,
01481,01,"Mashike, Hokkaido",増毛町,
01482,01,"Obira, Hokkaido",小平町,
01483,01,"Tomamae, Hokkaido",苫前町,
01484,01,"Haboro, Hokkaido",羽幌町,
01485,01,"Shosanbetsu, Hokkaido",初山別村,
01486,01,"Enbetsu, Hokkaido",遠別町,
01487,01,"Teshio, Hokkaido",天塩町,
01511,01,"Sarufutsu, Hokkaido",猿払村,
01512,01,"Hamatonbetsu, Hokkaido",浜頓別町,
01513,01,"Nakatonbetsu, Hokkaido",中頓別町,
01514,01,"Esashi, Hokkaido (Sōya)",枝幸町,
01516,01,"Toyotomi, Hokkaido",豊富町,
01517,01,"Rebun, Hokkaido",礼文町,
01518,01,"Rishiri, Hokkaido",利尻町,
01519,01,"Rishirifuji, Hokkaido",利尻富士町,
01520,01,"Horonobe, Hokkaido",幌延町,
01543,01,"Bihoro, Hokkaido",美幌町,
01544,01,"Tsubetsu, Hokkaido",津別町,
01545,01,"Shari, Hokkaido",斜里町,
01546,01,"Kiyosato, Hokkaido",清里町,
01547,01,"Koshimizu, Hokkaido",小清水町,
01549,01,"Kunneppu, Hokkaido",訓子府町,
01550,01,"Oketo, Hokkaido",置戸町,
01552,01,"Saroma, Hokkaido",佐呂間町,
01555,01,"Engaru, Hokkaido",遠軽町,
01559,01,"Yūbetsu, Hokkaido",湧別町,
01560,01,"Takinoue, Hokkaido",滝上町,
01561,01,"Okoppe, Hokkaido",興部町,
01562,01,"Nishiokoppe, Hokkaido",西興部村,
01563,01,"Ōmu, Hokkaido",雄武町,
01564,01,"Ōzora, Hokkaido",大空町,
01571,01,"Toyoura, Hokkaido",豊浦町,
01575,01,"Sōbetsu, Hokkaido",壮瞥町,
01578,01,"Shiraoi, Hokkaido",白老町,
01581,01,"Atsuma, Hokkaido",厚真町,
01584,01,"Tōyako, Hokkaido",洞爺湖町,
01585,01,"Abira, Hokkaido",安平町,
01586,01,"Mukawa, Hokkaido",むかわ町,
01601,01,"Hidaka, Hokkaido",日高町,
01602,01,"Biratori, Hokkaido",平取町,
01604,01,"Niikappu, Hokkaido",新冠町,
01607,01,"Urakawa, Hokkaido",浦河町,
01608,01,"Samani, Hokkaido",様似町,
01609,01,"Erimo, Hokkaido",えりも町,
01610,01,"Shinhidaka, Hokkaido",新ひだか町,
01631,01,"Otofuke, Hokkaido",音更町,
01632,01,"Shihoro, Hokkaido",士幌町,
01633,01,"Kamishihoro, Hokkaido",上士幌町,
01634,01,"Shikaoi, Hokkaido",鹿追町,
01635,01,"Shintoku, Hokkaido",新得町,
01636,01,"Shimizu, Hokkaido",清水町,
01637,01,"Memuro, Hokkaido",芽室町,
01638,01,"Nakasatsunai, Hokkaido",中札内村,
01639,01,"Sarabetsu, Hokkaido",更別村,
01641,01,"Taiki, Hokkaido",大樹町,
01642,01,"Hiroo, Hokkaido",広尾町,
01643,01,"Makubetsu, Hokkaido",幕別町,
01644,01,"Ikeda, Hokkaido",池田町,
01645,01,"Toyokoro, Hokkaido",豊頃町,
01646,01,"Honbetsu, Hokkaido",本別町,
01647,01,"Ashoro, Hokkaido",足寄町,
01648,01,"Rikubetsu, Hokkaido",陸別町,
01649,01,"Urahoro, Hokkaido",浦幌町,
01661,01,Kushiro (town),釧路町,
01662,01,"Akkeshi, Hokkaido",厚岸町,
01663,01,"Hamanaka, Hokkaido",浜中町,
01664,01,"Shibecha, Hokkaido",標茶町,
01665,01,"Teshikaga, Hokkaido",弟子屈町,
01667,01,Tsurui,鶴居村,
01668,01,"Shiranuka, Hokkaido",白糠町,
01691,01,"Betsukai, Hokkaido",別海町,
01692,01,"Nakashibetsu, Hokkaido",中標津町,
01693,01,"Shibetsu, Hokkaido (Nemuro)",標津町,
01694,01,"Rausu, Hokkaido",羅臼町,
02201,02,Aomori (city),青森市,
02202,02,Hirosaki,弘前市,
02203,02,Hachinohe,八戸市,
02204,02,"Kuroishi, Aomori",黒石市,
02205,02,"Goshogawara, Aomori",五所川原市,
02206,02,"Towada, Aomori",十和田市,
02207,02,"Misawa, Aomori",三沢市,
02208,02,"Mutsu, Aomori",むつ市,
02209,02,"Tsugaru, Aomori",つがる市,
02210,02,"Hirakawa, Aomori",平川市,
02301,02,Hiranai,平内町,
02303,02,"Imabetsu, Aomori",今別町,
02304,02,"Yomogita, Aomori",蓬田村,
02307,02,"Sotogahama, Aomori",外ヶ浜町,
02321,02,"Ajigasawa, Aomori",鰺ヶ沢町,
02323,02,"Fukaura, Aomori",深浦町,
02343,02,"Nishimeya, Aomori",西目屋村,
02361,02,"Fujisaki, Aomori",藤崎町,
02362,02,"Ōwani, Aomori",大鰐町,
02367,02,"Inakadate, Aomori",田舎館村,
02381,02,"Itayanagi, Aomori",板柳町,
02384,02,"Tsuruta, Aomori",鶴田町,
02387,02,"Nakadomari, Aomori",中泊町,
02401,02,"Noheji, Aomori",野辺地町,
02402,02,"Shichinohe, Aomori",七戸町,
02405,02,"Rokunohe, Aomori",六戸町,
02406,02,"Yokohama, Aomori",横浜町,
02408,02,"Tōhoku, Aomori",東北町,
02411,02,"Rokkasho, Aomori",六ヶ所村,
02412,02,"Oirase, Aomori",おいらせ町,
02423,02,"Ōma, Aomori",大間町,
02424,02,"Higashidōri, Aomori",東通村,
02425,02,"Kazamaura, Aomori",風間浦村,
02426,02,"Sai, Aomori",佐井村,
02441,02,"Sannohe, Aomori",三戸町,
02442,02,"Gonohe, Aomori",五戸町,
02443,02,"Takko, Aomori",田子町,
02445,02,"Nanbu, Aomori",南部町,
02446,02,"Hashikami, Aomori",階上町,
02450,02,"Shingō, Aomori",新郷村,
03201,03,Morioka,盛岡市,
03202,03,"Miyako, Iwate",宮古市,
03203,03,"Ōfunato, Iwate",大船渡市,
03205,03,"Hanamaki, Iwate",花巻市,
03206,03,"Kitakami, Iwate",北上市,
03207,03,"Kuji, Iwate",久慈市,
03208,03,"Tōno, Iwate",遠野市,
03209,03,"Ichinoseki, Iwate",一関市,
03210,03,"Rikuzentakata, Iwate",陸前高田市,
03211,03,"Kamaishi, Iwate",釜石市,
03213,03,"Ninohe, Iwate",二戸市,
03214,03,"Hachimantai, Iwate",八幡平市,
03215,03,"Ōshū, Iwate",奥州市,
03216,03,"Takizawa, Iwate",滝沢市,
03301,03,"Shizukuishi, Iwate",雫石町,
03302,03,"Kuzumaki, Iwate",葛巻町,
03303,03,"Iwate, Iwate",岩手町,
03321,03,"Shiwa, Iwate",紫波町,
03322,03,"Yahaba, Iwate",矢巾町,
03366,03,"Nishiwaga, Iwate",西和賀町,
03381,03,"Kanegasaki, Iwate",金ケ崎町,
03402,03,"Hiraizumi, Iwate",平泉町,
03441,03,"Sumita, Iwate",住田町,
03461,03,"Ōtsuchi, Iwate",大槌町,
03482,03,"Yamada, Iwate",山田町,
03483,03,"Iwaizumi, Iwate",岩泉町,
03484,03,"Tanohata, Iwate",田野畑村,
03485,03,"Fudai, Iwate",普代村,
03501,03,"Karumai, Iwate",軽米町,
03503,03,"Noda, Iwate",野田村,
03506,03,"Kunohe, Iwate",九戸村,
03507,03,"Hirono, Iwate",洋野町,
03524,03,"Ichinohe, Iwate",一戸町,
04100,04,Sendai,仙台市,
04101,04,"Aoba-ku, Sendai",仙台市,青葉区
04102,04,"Miyagino-ku, Sendai",仙台市,宮城野区
04103,04,"Wakabayashi-ku, Sendai",仙台市,若林区
04104,04,"Taihaku-ku, Sendai",仙台市,太白区
04105,04,"Izumi-ku, Sendai",仙台市,泉区
04202,04,Ishinomaki,石巻市,
04203,04,Shiogama,塩竈市,
04205,04,Kesennuma,気仙沼市,
04206,04,"Shiroishi, Miyagi",白石市,
04207,04,"Natori, Miyagi",名取市,
04208,04,"Kakuda, Miyagi",角田市,
04209,04,Tagajō,多賀城市,
04211,04,Iwanuma,岩沼市,
04212,04,"Tome, Miyagi",登米市,
04213,04,"Kurihara, Miyagi",栗原市,
04214,04,Higashimatsushima,東松島市,
04215,04,"Ōsaki, Miyagi",大崎市,
04216,04,"Tomiya, Miyagi",富谷市,
04301,04,"Zaō, Miyagi",蔵王町,
04302,04,"Shichikashuku, Miyagi",七ヶ宿町,
04321,04,"Ōgawara, Miyagi",大河原町,
04322,04,"Murata, Miyagi",村田町,
04323,04,"Shibata, Miyagi",柴田町,
04324,04,"Kawasaki, Miyagi",川崎町,
04341,04,"Marumori, Miyagi",丸森町,
04361,04,"Watari, Miyagi",亘理町,
04362,04,"Yamamoto, Miyagi",山元町,
04401,04,"Matsushima, Miyagi",松島町,
04404,04,Shichigahama,七ヶ浜町,
04406,04,Rifu,利府町,
04421,04,"Taiwa, Miyagi",大和町,
04422,04,"Ōsato, Miyagi",大郷町,
04424,04,"Ōhira, Miyagi",大衡村,
04444,04,"Shikama, Miyagi",色麻町,
04445,04,"Kami, Miyagi",加美町,
04501,04,Wakuya,涌谷町,
04505,04,"Misato, Miyagi",美里町,
04581,04,"Onagawa, Miyagi",女川町,
04606,04,Minamisanriku,南三陸町,
05201,05,Akita (city),秋田市,
05202,05,"Noshiro, Akita",能代市,
05203,05,"Yokote, Akita",横手市,
05204,05,Ōdate,大館市,
05206,05,"Oga, Akita",男鹿市,
05207,05,"Yuzawa, Akita",湯沢市,
05209,05,"Kazuno, Akita",鹿角市,
05210,05,Yurihonjō,由利本荘市,
05211,05,"Katagami, Akita",潟上市,
05212,05,"Daisen, Akita",大仙市,
05213,05,Kitaakita,北秋田市,
05214,05,"Nikaho, Akita",にかほ市,
05215,05,"Semboku, Akita",仙北市,
05303,05,"Kosaka, Akita",小坂町,
05327,05,"Kamikoani, Akita",上小阿仁村,
05346,05,"Fujisato, Akita",藤里町,
05348,05,"Mitane, Akita",三種町,
05349,05,"Happō, Akita",八峰町,
05361,05,"Gojōme, Akita",五城目町,
05363,05,"Hachirōgata, Akita",八郎潟町,
05366,05,"Ikawa, Akita",井川町,
05368,05,"Ōgata, Akita",大潟村,
05434,05,"Misato, Akita",美郷町,
05463,05,"Ugo, Akita",羽後町,
05464,05,"Higashinaruse, Akita",東成瀬村,
06201,06,"Yamagata, Yamagata",山形市,
06202,06,"Yonezawa, Yamagata",米沢市,
06203,06,"Tsuruoka, Yamagata",鶴岡市,
06204,06,"Sakata, Yamagata",酒田市,
06205,06,"Shinjō, Yamagata",新庄市,
06206,06,"Sagae, Yamagata",寒河江市,
06207,06,"Kaminoyama, Yamagata",上山市,
06208,06,"Murayama, Yamagata",村山市,
06209,06,"Nagai, Yamagata",長井市,
06210,06,"Tendō, Yamagata",天童市,
06211,06,"Higashine, Yamagata",東根市,
06212,06,"Obanazawa, Yamagata",尾花沢市,
06213,06,"Nan'yō, Yamagata",南陽市,
06301,06,"Yamanobe, Yamagata",山辺町,
06302,06,"Nakayama, Yamagata",中山町,
06321,06,"Kahoku, Yamagata",河北町,
06322,06,"Nishikawa, Yamagata",西川町,
06323,06,"Asahi, Yamagata (Nishimurayama)",朝日町,
06324,06,"Ōe, Yamagata",大江町,
06341,06,Ōishida,大石田町,
06361,06,"Kaneyama, Yamagata",金山町,
06362,06,"Mogami, Yamagata",最上町,
06363,06,"Funagata, Yamagata",舟形町,
06364,06,Mamurogawa,真室川町,
06365,06,"Ōkura, Yamagata",大蔵村,
06366,06,"Sakegawa, Yamagata",鮭川村,
06367,06,"Tozawa, Yamagata",戸沢村,
06381,06,"Takahata, Yamagata",高畠町,
06382,06,"Kawanishi, Yamagata",川西町,
06401,06,"Oguni, Yamagata",小国町,
06402,06,Shirataka,白鷹町,
06403,06,"Iide, Yamagata",飯豊町,
06426,06,"Mikawa, Yamagata",三川町,
06428,06,"Shōnai, Yamagata",庄内町,
06461,06,Yuza,遊佐町,
07201,07,Fukushima (city),福島市,
07202,07,Aizuwakamatsu,会津若松市,
07203,07,Kōriyama,郡山市,
07204,07,"Iwaki, Fukushima",いわき市,
07205,07,"Shirakawa, Fukushima",白河市,
07207,07,"Sukagawa, Fukushima",須賀川市,
07208,07,"Kitakata, Fukushima",喜多方市,
07209,07,"Sōma, Fukushima",相馬市,
07210,07,"Nihonmatsu, Fukushima",二本松市,
07211,07,"Tamura, Fukushima",田村市,
07212,07,Minamisōma,南相馬市,
07213,07,"Date City, Fukushima",伊達市,
07214,07,"Motomiya, Fukushima",本宮市,
07301,07,"Koori, Fukushima",桑折町,
07303,07,"Kunimi, Fukushima",国見町,
07308,07,"Kawamata, Fukushima",川俣町,
07322,07,"Ōtama, Fukushima",大玉村,
07342,07,"Kagamiishi, Fukushima",鏡石町,
07344,07,"Ten-ei, Fukushima",天栄村,
07362,07,"Shimogō, Fukushima",下郷町,
07364,07,"Hinoemata, Fukushima",檜枝岐村,
07367,07,"Tadami, Fukushima",只見町,
07368,07,Minamiaizu,南会津町,
07402,07,"Kitashiobara, Fukushima",北塩原村,
07405,07,"Nishiaizu, Fukushima",西会津町,
07407,07,"Bandai, Fukushima",磐梯町,
07408,07,Inawashiro,猪苗代町,
07421,07,Aizubange,会津坂下町,
07422,07,"Yugawa, Fukushima",湯川村,
07423,07,"Yanaizu, Fukushima",柳津町,
07444,07,"Mishima, Fukushima",三島町,
07445,07,"Kaneyama, Fukushima",金山町,
07446,07,"Shōwa, Fukushima",昭和村,
07447,07,Aizumisato,会津美里町,
07461,07,"Nishigō, Fukushima",西郷村,
07464,07,"Izumizaki, Fukushima",泉崎村,
07465,07,"Nakajima, Fukushima",中島村,
07466,07,"Yabuki, Fukushima",矢吹町,
07481,07,"Tanagura, Fukushima",棚倉町,
07482,07,"Yamatsuri, Fukushima",矢祭町,
07483,07,"Hanawa, Fukushima",塙町,
07484,07,"Samegawa, Fukushima",鮫川村,
07501,07,"Ishikawa, Fukushima",石川町,
07502,07,"Tamakawa, Fukushima",玉川村,
07503,07,"Hirata, Fukushima",平田村,
07504,07,"Asakawa, Fukushima",浅川町,
07505,07,"Furudono, Fukushima",古殿町,
07521,07,"Miharu, Fukushima",三春町,
07522,07,"Ono, Fukushima",小野町,
07541,07,"Hirono, Fukushima",広野町,
07542,07,"Naraha, Fukushima",楢葉町,
07543,07,"Tomioka, Fukushima",富岡町,
07544,07,"Kawauchi, Fukushima",川内村,
07545,07,"Ōkuma, Fukushima",大熊町,
07546,07,"Futaba, Fukushima",双葉町,
07547,07,"Namie, Fukushima",浪江町,
07548,07,"Katsurao, Fukushima",葛尾村,
07561,07,"Shinchi, Fukushima",新地町,
07564,07,"Iitate, Fukushima",飯舘村,
08201,08,"Mito, Ibaraki",水戸市,
08202,08,"Hitachi, Ibaraki",日立市,
08203,08,Tsuchiura,土浦市,
08204,08,"Koga, Ibaraki",古河市,
08205,08,"Ishioka, Ibaraki",石岡市,
08207,08,"Yūki, Ibaraki",結城市,
08208,08,"Ryūgasaki, Ibaraki",龍ケ崎市,
08210,08,"Shimotsuma, Ibaraki",下妻市,
08211,08,Jōsō,常総市,
08212,08,"Hitachiōta, Ibaraki",常陸太田市,
08214,08,"Takahagi, Ibaraki",高萩市,
08215,08,"Kitaibaraki, Ibaraki",北茨城市,
08216,08,"Kasama, Ibaraki",笠間市,
08217,08,"Toride, Ibaraki",取手市,
08219,08,"Ushiku, Ibaraki",牛久市,
08220,08,"Tsukuba, Ibaraki",つくば市,
08221,08,"Hitachinaka, Ibaraki",ひたちなか市,
08222,08,"Kashima, Ibaraki",鹿嶋市,
08223,08,"Itako, Ibaraki",潮来市,
08224,08,"Moriya, Ibaraki",守谷市,
08225,08,Hitachiōmiya,常陸大宮市,
08226,08,"Naka, Ibaraki",那珂市,
08227,08,Chikusei,筑西市,
08228,08,"Bandō, Ibaraki",坂東市,
08229,08,"Inashiki, Ibaraki",稲敷市,
08230,08,"Kasumigaura, Ibaraki",かすみがうら市,
08231,08,"Sakuragawa, Ibaraki",桜川市,
08232,08,"Kamisu, Ibaraki",神栖市,
08233,08,"Namegata, Ibaraki",行方市,
08234,08,"Hokota, Ibaraki",鉾田市,
08235,08,"Tsukubamirai, Ibaraki",つくばみらい市,
08236,08,Omitama,小美玉市,
08302,08,"Ibaraki, Ibaraki",茨城町,
08309,08,"Ōarai, Ibaraki",大洗町,
08310,08,"Shirosato, Ibaraki",城里町,
08341,08,"Tōkai, Ibaraki",東海村,
08364,08,"Daigo, Ibaraki",大子町,
08442,08,"Miho, Ibaraki",美浦村,
08443,08,"Ami, Ibaraki",阿見町,
08447,08,"Kawachi, Ibaraki",河内町,
08521,08,"Yachiyo, Ibaraki",八千代町,
08542,08,"Goka, Ibaraki",五霞町,
08546,08,"Sakai, Ibaraki",境町,
08564,08,"Tone, Ibaraki",利根町,
09201,09,Utsunomiya,宇都宮市,
09202,09,"Ashikaga, Tochigi",足利市,
09203,09,"Tochigi, Tochigi",栃木市,
09204,09,"Sano, Tochigi",佐野市,
09205,09,"Kanuma, Tochigi",鹿沼市,
09206,09,"Nikkō, Tochigi",日光市,
09208,09,"Oyama, Tochigi",小山市,
09209,09,"Mooka, Tochigi",真岡市,
09210,09,Ōtawara,大田原市,
09211,09,"Yaita, Tochigi",矢板市,
09213,09,Nasushiobara,那須塩原市,
09214,09,"Sakura, Tochigi",さくら市,
09215,09,Nasukarasuyama,那須烏山市,
09216,09,"Shimotsuke, Tochigi",下野市,
09301,09,"Kaminokawa, Tochigi",上三川町,
09342,09,"Mashiko, Tochigi",益子町,
09343,09,"Motegi, Tochigi",茂木町,
09344,09,"Ichikai, Tochigi",市貝町,
09345,09,"Haga, Tochigi",芳賀町,
09361,09,"Mibu, Tochigi",壬生町,
09364,09,"Nogi, Tochigi",野木町,
09384,09,"Shioya, Tochigi",塩谷町,
09386,09,"Takanezawa, Tochigi",高根沢町,
09407,09,"Nasu, Tochigi",那須町,
09411,09,"Nakagawa, Tochigi",那珂川町,
10201,10,Maebashi,前橋市,
10202,10,Takasaki,高崎市,
10203,10,"Kiryū, Gunma",桐生市,
10204,10,"Isesaki, Gunma",伊勢崎市,
10205,10,"Ōta, Gunma",太田市,
10206,10,"Numata, Gunma",沼田市,
10207,10,"Tatebayashi, Gunma",館林市,
10208,10,"Shibukawa, Gunma",渋川市,
10209,10,"Fujioka, Gunma",藤岡市,
10210,10,"Tomioka, Gunma",富岡市,
10211,10,"Annaka, Gunma",安中市,
10212,10,"Midori, Gunma",みどり市,
10344,10,"Shintō, Gunma",榛東村,
10345,10,"Yoshioka, Gunma",吉岡町,
10366,10,"Ueno, Gunma",上野村,
10367,10,"Kanna, Gunma",神流町,
10382,10,"Shimonita, Gunma",下仁田町,
10383,10,"Nanmoku, Gunma",南牧村,
10384,10,"Kanra, Gunma",甘楽町,
10421,10,"Nakanojō, Gunma",中之条町,
10424,10,"Naganohara, Gunma",長野原町,
10425,10,"Tsumagoi, Gunma",嬬恋村,
10426,10,"Kusatsu, Gunma",草津町,
10428,10,"Takayama, Gunma",高山村,
10429,10,"Higashiagatsuma, Gunma",東吾妻町,
10443,10,"Katashina, Gunma",片品村,
10444,10,"Kawaba, Gunma",川場村,
10448,10,"Shōwa, Gunma",昭和村,
10449,10,"Minakami, Gunma",みなかみ町,
10464,10,"Tamamura, Gunma",玉村町,
10521,10,"Itakura, Gunma",板倉町,
10522,10,"Meiwa, Gunma",明和町,
10523,10,"Chiyoda, Gunma",千代田町,
10524,10,"Ōizumi, Gunma",大泉町,
10525,10,"Ōra, Gunma",邑楽町,
11100,11,Saitama (city),さいたま市,
11101,11,"Nishi-ku, Saitama",さいたま市,西区
11102,11,"Kita-ku, Saitama",さいたま市,北区
11103,11,"Ōmiya-ku, Saitama",さいたま市,大宮区
11104,11,"Minuma-ku, Saitama",さいたま市,見沼区
11105,11,"Chūō-ku, Saitama",さいたま市,中央区
11106,11,"Sakura-ku, Saitama",さいたま市,桜区
11107,11,"Urawa-ku, Saitama",さいたま市,浦和区
11108,11,"Minami-ku, Saitama",さいたま市,南区
11109,11,"Midori-ku, Saitama",さいたま市,緑区
11110,11,"Iwatsuki-ku, Saitama",さいたま市,岩槻区
11201,11,"Kawagoe, Saitama",川越市,
11202,11,"Kumagaya, Saitama",熊谷市,
11203,11,"Kawaguchi, Saitama",川口市,
11206,11,Gyōda,行田市,
11207,11,"Chichibu, Saitama",秩父市,
11208,11,"Tokorozawa, Saitama",所沢市,
11209,11,"Hannō, Saitama",飯能市,
11210,11,"Kazo, Saitama",加須市,
11211,11,"Honjō, Saitama",本庄市,
11212,11,"Higashimatsuyama, Saitama",東松山市,
11214,11,"Kasukabe, Saitama",春日部市,
11215,11,"Sayama, Saitama",狭山市,
11216,11,"Hanyū, Saitama",羽生市,
11217,11,Kōnosu,鴻巣市,
11218,11,"Fukaya, Saitama",深谷市,
11219,11,"Ageo, Saitama",上尾市,
11221,11,Sōka,草加市,
11222,11,Koshigaya,越谷市,
11223,11,"Warabi, Saitama",蕨市,
11224,11,"Toda, Saitama",戸田市,
11225,11,"Iruma, Saitama",入間市,
11227,11,"Asaka, Saitama",朝霞市,
11228,11,"Shiki, Saitama",志木市,
11229,11,"Wakō, Saitama",和光市,
11230,11,"Niiza, Saitama",新座市,
11231,11,"Okegawa, Saitama",桶川市,
11232,11,"Kuki, Saitama",久喜市,
11233,11,"Kitamoto, Saitama",北本市,
11234,11,"Yashio, Saitama",八潮市,
11235,11,"Fujimi, Saitama",富士見市,
11237,11,"Misato, Saitama (city)",三郷市,
11238,11,"Hasuda, Saitama",蓮田市,
11239,11,"Sakado, Saitama",坂戸市,
11240,11,"Satte, Saitama",幸手市,
11241,11,"Tsurugashima, Saitama",鶴ヶ島市,
11242,11,"Hidaka, Saitama",日高市,
11243,11,"Yoshikawa, Saitama",吉川市,
11245,11,"Fujimino, Saitama",ふじみ野市,
11246,11,"Shiraoka, Saitama",白岡市,
11301,11,"Ina, Saitama",伊奈町,
11324,11,"Miyoshi, Saitama",三芳町,
11326,11,"Moroyama, Saitama",毛呂山町,
11327,11,"Ogose, Saitama",越生町,
11341,11,"Namegawa, Saitama",滑川町,
11342,11,"Ranzan, Saitama",嵐山町,
11343,11,"Ogawa, Saitama",小川町,
11346,11,"Kawajima, Saitama",川島町,
11347,11,"Yoshimi, Saitama",吉見町,
11348,11,"Hatoyama, Saitama",鳩山町,
11349,11,"Tokigawa, Saitama",ときがわ町,
11361,11,"Yokoze, Saitama",横瀬町,
11362,11,"Minano, Saitama",皆野町,
11363,11,"Nagatoro, Saitama",長瀞町,
11365,11,"Ogano, Saitama",小鹿野町,
11369,11,"Higashichichibu, Saitama",東秩父村,
11381,11,"Misato, Saitama (town)",美里町,
11383,11,"Kamikawa, Saitama",神川町,
11385,11,"Kamisato, Saitama",上里町,
11408,11,"Yorii, Saitama",寄居町,
11442,11,"Miyashiro, Saitama",宮代町,
11464,11,"Sugito, Saitama",杉戸町,
11465,11,"Matsubushi, Saitama",松伏町,
12100,12,Chiba (city),千葉市,
12101,12,"Chūō-ku, Chiba",千葉市,中央区
12102,12,Hanamigawa-ku,千葉市,花見川区
12103,12,Inage-ku,千葉市,稲毛区
12104,12,Wakaba-ku,千葉市,若葉区
12105,12,"Midori-ku, Chiba",千葉市,緑区
12106,12,Mihama-ku,千葉市,美浜区
12202,12,Chōshi,銚子市,
12203,12,"Ichikawa, Chiba",市川市,
12204,12,Funabashi,船橋市,
12205,12,"Tateyama, Chiba",館山市,
12206,12,Kisarazu,木更津市,
12207,12,Matsudo,松戸市,
12208,12,"Noda, Chiba",野田市,
12210,12,Mobara,茂原市,
12211,12,"Narita, Chiba",成田市,
12212,12,"Sakura, Chiba",佐倉市,
12213,12,Tōgane,東金市,
12215,12,"Asahi, Chiba",旭市,
12216,12,Narashino,習志野市,
12217,12,Kashiwa,柏市,
12218,12,"Katsuura, Chiba",勝浦市,
12219,12,"Ichihara, Chiba",市原市,
12220,12,Nagareyama,流山市,
12221,12,"Yachiyo, Chiba",八千代市,
12222,12,"Abiko, Chiba",我孫子市,
12223,12,"Kamogawa, Chiba",鴨川市,
12224,12,Kamagaya,鎌ケ谷市,
12225,12,Kimitsu,君津市,
12226,12,Futtsu,富津市,
12227,12,Urayasu,浦安市,
12228,12,Yotsukaidō,四街道市,
12229,12,Sodegaura,袖ケ浦市,
12230,12,Yachimata,八街市,
12231,12,Inzai,印西市,
12232,12,Shiroi,白井市,
12233,12,Tomisato,富里市,
12234,12,Minamibōsō,南房総市,
12235,12,Sōsa,匝瑳市,
12236,12,"Katori, Chiba",香取市,
12237,12,Sanmu,山武市,
12238,12,Isumi,いすみ市,
12239,12,Ōamishirasato,大網白里市,
12322,12,Shisui,酒々井町,
12329,12,"Sakae, Chiba",栄町,
12342,12,Kōzaki,神崎町,
12347,12,"Tako, Chiba",多古町,
12349,12,Tōnoshō,東庄町,
12403,12,Kujūkuri,九十九里町,
12409,12,"Shibayama, Chiba",芝山町,
12410,12,Yokoshibahikari,横芝光町,
12421,12,"Ichinomiya, Chiba",一宮町,
12422,12,Mutsuzawa,睦沢町,
12423,12,Chōsei,長生村,
12424,12,Shirako,白子町,
12426,12,"Nagara, Chiba",長柄町,
12427,12,Chōnan,長南町,
12441,12,"Ōtaki, Chiba",大多喜町,
12443,12,Onjuku,御宿町,
12463,12,Kyonan,鋸南町,
13101,13,"Chiyoda, Tokyo",千代田区,
13102,13,"Chūō, Tokyo",中央区,
13103,13,"Minato, Tokyo",港区,
13104,13,Shinjuku,新宿区,
13105,13,Bunkyō,文京区,
13106,13,Taitō,台東区,
13107,13,"Sumida, Tokyo",墨田区,
13108,13,Kōtō,江東区,
13109,13,Shinagawa,品川区,
13110,13,Meguro,目黒区,
13111,13,"Ōta, Tokyo",大田区,
13112,13,Setagaya,世田谷区,
13113,13,Shibuya,渋谷区,
13114,13,"Nakano, Tokyo",中野区,
13115,13,Suginami,杉並区,
13116,13,Toshima,豊島区,
13117,13,"Kita, Tokyo",北区,
13118,13,"Arakawa, Tokyo",荒川区,
13119,13,Itabashi,板橋区,
13120,13,Nerima,練馬区,
13121,13,"Adachi, Tokyo",足立区,
13122,13,Katsushika,葛飾区,
13123,13,"Edogawa, Tokyo",江戸川区,
13201,13,Hachiōji,八王子市,
13202,13,"Tachikawa, Tokyo",立川市,
13203,13,"Musashino, Tokyo",武蔵野市,
13204,13,"Mitaka, Tokyo",三鷹市,
13205,13,"Ōme, Tokyo",青梅市,
13206,13,"Fuchū, Tokyo",府中市,
13207,13,"Akishima, Tokyo",昭島市,
13208,13,"Chōfu, Tokyo",調布市,
13209,13,"Machida, Tokyo",町田市,
13210,13,"Koganei, Tokyo",小金井市,
13211,13,"Kodaira, Tokyo",小平市,
13212,13,"Hino, Tokyo",日野市,
13213,13,"Higashimurayama, Tokyo",東村山市,
13214,13,"Kokubunji, Tokyo",国分寺市,
13215,13,"Kunitachi, Tokyo",国立市,
13218,13,"Fussa, Tokyo",福生市,
13219,13,"Komae, Tokyo",狛江市,
13220,13,"Higashiyamato, Tokyo",東大和市,
13221,13,"Kiyose, Tokyo",清瀬市,
13222,13,"Higashikurume, Tokyo",東久留米市,
13223,13,"Musashimurayama, Tokyo",武蔵村山市,
13224,13,"Tama, Tokyo",多摩市,
13225,13,Inagi,稲城市,
13227,13,"Hamura, Tokyo",羽村市,
13228,13,"Akiruno, Tokyo",あきる野市,
13229,13,"Nishitōkyō, Tokyo",西東京市,
13303,13,"Mizuho, Tokyo",瑞穂町,
13305,13,"Hinode, Tokyo",日の出町,
13307,13,"Hinohara, Tokyo",檜原村,
13308,13,"Okutama, Tokyo",奥多摩町,
13361,13,"Ōshima, Tokyo",大島町,
13362,13,"To-shima, Tokyo",利島村,
13363,13,"Niijima, Tokyo",新島村,
13364,13,"Kōzushima, Tokyo",神津島村,
13381,13,"Miyake, Tokyo",三宅村,
13382,13,"Mikurajima, Tokyo",御蔵島村,
13401,13,"Hachijō, Tokyo",八丈町,
13402,13,"Aogashima, Tokyo",青ヶ島村,
13421,13,"Ogasawara, Tokyo",小笠原村,
14100,14,Yokohama,横浜市,
14101,14,"Tsurumi-ku, Yokohama",横浜市,鶴見区
14102,14,"Kanagawa-ku, Yokohama",横浜市,神奈川区
14103,14,"Nishi-ku, Yokohama",横浜市,西区
14104,14,"Naka-ku, Yokohama",横浜市,中区
14105,14,"Minami-ku, Yokohama",横浜市,南区
14106,14,"Hodogaya-ku, Yokohama",横浜市,保土ケ谷区
14107,14,"Isogo-ku, Yokohama",横浜市,磯子区
14108,14,"Kanazawa-ku, Yokohama",横浜市,金沢区
14109,14,"Kōhoku-ku, Yokohama",横浜市,港北区
14110,14,"Totsuka-ku, Yokohama",横浜市,戸塚区
14111,14,"Kōnan-ku, Yokohama",横浜市,港南区
14112,14,"Asahi-ku, Yokohama",横浜市,旭区
14113,14,"Midori-ku, Yokohama",横浜市,緑区
14114,14,"Seya-ku, Yokohama",横浜市,瀬谷区
14115,14,"Sakae-ku, Yokohama",横浜市,栄区
14116,14,"Izumi-ku, Yokohama",横浜市,泉区
14117,14,"Aoba-ku, Yokohama",横浜市,青葉区
14118,14,"Tsuzuki-ku, Yokohama",横浜市,都筑区
14130,14,"Kawasaki, Kanagawa",川崎市,
14131,14,"Kawasaki-ku, Kawasaki",川崎市,川崎区
14132,14,"Saiwai-ku, Kawasaki",川崎市,幸区
14133,14,"Nakahara-ku, Kawasaki",川崎市,中原区
14134,14,"Takatsu-ku, Kawasaki",川崎市,高津区
14135,14,"Tama-ku, Kawasaki",川崎市,多摩区
14136,14,"Miyamae-ku, Kawasaki",川崎市,宮前区
14137,14,"Asao-ku, Kawasaki",川崎市,麻生区
14150,14,Sagamihara,相模原市,
14151,14,"Midori-ku, Sagamihara",相模原市,緑区
14152,14,"Chūō-ku, Sagamihara",相模原市,中央区
14153,14,"Minami-ku, Sagamihara",相模原市,南区
14201,14,Yokosuka,横須賀市,
14203,14,Hiratsuka,平塚市,
14204,14,Kamakura,鎌倉市,
14205,14,"Fujisawa, Kanagawa",藤沢市,
14206,14,Odawara,小田原市,
14207,14,"Chigasaki, Kanagawa",茅ヶ崎市,
14208,14,"Zushi, Kanagawa",逗子市,
14210,14,"Miura, Kanagawa",三浦市,
14211,14,"Hadano, Kanagawa",秦野市,
14212,14,Atsugi,厚木市,
14213,14,"Yamato, Kanagawa",大和市,
14214,14,"Isehara, Kanagawa",伊勢原市,
14215,14,"Ebina, Kanagawa",海老名市,
14216,14,"Zama, Kanagawa",座間市,
14217,14,"Minamiashigara, Kanagawa",南足柄市,
14218,14,"Ayase, Kanagawa",綾瀬市,
14301,14,"Hayama, Kanagawa",葉山町,
14321,14,Samukawa,寒川町,
14341,14,"Ōiso, Kanagawa",大磯町,
14342,14,"Ninomiya, Kanagawa",二宮町,
14361,14,"Nakai, Kanagawa",中井町,
14362,14,"Ōi, Kanagawa",大井町,
14363,14,"Matsuda, Kanagawa",松田町,
14364,14,"Yamakita, Kanagawa",山北町,
14366,14,"Kaisei, Kanagawa",開成町,
14382,14,Hakone,箱根町,
14383,14,"Manazuru, Kanagawa",真鶴町,
14384,14,Yugawara,湯河原町,
14401,14,"Aikawa, Kanagawa",愛川町,
14402,14,"Kiyokawa, Kanagawa",清川村,
15100,15,Niigata (city),新潟市,
15101,15,"Kita-ku, Niigata",新潟市,北区
15102,15,"Higashi-ku, Niigata",新潟市,東区
15103,15,"Chūō-ku, Niigata",新潟市,中央区
15104,15,"Kōnan-ku, Niigata",新潟市,江南区
15105,15,"Akiha-ku, Niigata",新潟市,秋葉区
15106,15,"Minami-ku, Niigata",新潟市,南区
15107,15,"Nishi-ku, Niigata",新潟市,西区
15108,15,"Nishikan-ku, Niigata",新潟市,西蒲区
15202,15,"Nagaoka, Niigata",長岡市,
15204,15,"Sanjō, Niigata",三条市,
15205,15,"Kashiwazaki, Niigata",柏崎市,
15206,15,"Shibata, Niigata",新発田市,
15208,15,"Ojiya, Niigata",小千谷市,
15209,15,"Kamo, Niigata",加茂市,
15210,15,Tōkamachi,十日町市,
15211,15,"Mitsuke, Niigata",見附市,
15212,15,"Murakami, Niigata",村上市,
15213,15,"Tsubame, Niigata",燕市,
15216,15,"Itoigawa, Niigata",糸魚川市,
15217,15,"Myōkō, Niigata",妙高市,
15218,15,"Gosen, Niigata",五泉市,
15222,15,"Jōetsu, Niigata",上越市,
15223,15,"Agano, Niigata",阿賀野市,
15224,15,"Sado, Niigata",佐渡市,
15225,15,Uonuma,魚沼市,
15226,15,Minamiuonuma,南魚沼市,
15227,15,Tainai,胎内市,
15307,15,"Seirō, Niigata",聖籠町,
15342,15,"Yahiko, Niigata",弥彦村,
15361,15,"Tagami, Niigata",田上町,
15385,15,"Aga, Niigata",阿賀町,
15405,15,Izumozaki,出雲崎町,
15461,15,"Yuzawa, Niigata",湯沢町,
15482,15,Tsunan,津南町,
15504,15,Kariwa,刈羽村,
15581,15,Sekikawa,関川村,
15586,15,Awashimaura,粟島浦村,
16201,16,Toyama (city),富山市,
16202,16,"Takaoka, Toyama",高岡市,
16204,16,"Uozu, Toyama",魚津市,
16205,16,"Himi, Toyama",氷見市,
16206,16,"Namerikawa, Toyama",滑川市,
16207,16,"Kurobe, Toyama",黒部市,
16208,16,"Tonami, Toyama",砺波市,
16209,16,"Oyabe, Toyama",小矢部市,
16210,16,"Nanto, Toyama",南砺市,
16211,16,"Imizu, Toyama",射水市,
16321,16,"Funahashi, Toyama",舟橋村,
16322,16,"Kamiichi, Toyama",上市町,
16323,16,"Tateyama, Toyama",立山町,
16342,16,"Nyūzen, Toyama",入善町,
16343,16,"Asahi, Toyama",朝日町,
17201,17,Kanazawa,金沢市,
17202,17,"Nanao, Ishikawa",七尾市,
17203,17,"Komatsu, Ishikawa",小松市,
17204,17,"Wajima, Ishikawa",輪島市,
17205,17,"Suzu, Ishikawa",珠洲市,
17206,17,"Kaga, Ishikawa",加賀市,
17207,17,"Hakui, Ishikawa",羽咋市,
17209,17,"Kahoku, Ishikawa",かほく市,
17210,17,"Hakusan, Ishikawa",白山市,
17211,17,"Nomi, Ishikawa",能美市,
17212,17,"Nonoichi, Ishikawa",野々市市,
17324,17,"Kawakita, Ishikawa",川北町,
17361,17,"Tsubata, Ishikawa",津幡町,
17365,17,"Uchinada, Ishikawa",内灘町,
17384,17,"Shika, Ishikawa",志賀町,
17386,17,"Hōdatsushimizu, Ishikawa",宝達志水町,
17407,17,"Nakanoto, Ishikawa",中能登町,
17461,17,"Anamizu, Ishikawa",穴水町,
17463,17,"Noto, Ishikawa",能登町,
18201,18,Fukui (city),福井市,
18202,18,"Tsuruga, Fukui",敦賀市,
18204,18,"Obama, Fukui",小浜市,
18205,18,"Ōno, Fukui",大野市,
18206,18,"Katsuyama, Fukui",勝山市,
18207,18,"Sabae, Fukui",鯖江市,
18208,18,"Awara, Fukui",あわら市,
18209,18,"Echizen, Fukui",越前市,
18210,18,"Sakai, Fukui",坂井市,
18322,18,"Eiheiji, Fukui",永平寺町,
18382,18,"Ikeda, Fukui",池田町,
18404,18,"Minamiechizen, Fukui",南越前町,
18423,18,"Echizen, Fukui (town)",越前町,
18442,18,"Mihama, Fukui",美浜町,
18481,18,"Takahama, Fukui",高浜町,
18483,18,"Ōi, Fukui",おおい町,
18501,18,"Wakasa, Fukui",若狭町,
19201,19,Kōfu,甲府市,
19202,19,"Fujiyoshida, Yamanashi",富士吉田市,
19204,19,"Tsuru, Yamanashi",都留市,
19205,19,"Yamanashi, Yamanashi",山梨市,
19206,19,"Ōtsuki, Yamanashi",大月市,
19207,19,"Nirasaki, Yamanashi",韮崎市,
19208,19,"Minami-Alps, Yamanashi",南アルプス市,
19209,19,"Hokuto, Yamanashi",北杜市,
19210,19,"Kai, Yamanashi",甲斐市,
19211,19,Fuefuki,笛吹市,
19212,19,"Uenohara, Yamanashi",上野原市,
19213,19,"Kōshū, Yamanashi",甲州市,
19214,19,"Chūō, Yamanashi",中央市,
19346,19,Ichikawamisato,市川三郷町,
19364,19,"Hayakawa, Yamanashi",早川町,
19365,19,"Minobu, Yamanashi",身延町,
19366,19,"Nanbu, Yamanashi",南部町,
19368,19,"Fujikawa, Yamanashi",富士川町,
19384,19,"Shōwa, Yamanashi",昭和町,
19422,19,Dōshi,道志村,
19423,19,Nishikatsura,西桂町,
19424,19,"Oshino, Yamanashi",忍野村,
19425,19,"Yamanakako, Yamanashi",山中湖村,
19429,19,"Narusawa, Yamanashi",鳴沢村,
19430,19,"Fujikawaguchiko, Yamanashi",富士河口湖町,
19442,19,"Kosuge, Yamanashi",小菅村,
19443,19,Tabayama,丹波山村,
20201,20,Nagano (city),長野市,
20202,20,"Matsumoto, Nagano",松本市,
20203,20,"Ueda, Nagano",上田市,
20204,20,"Okaya, Nagano",岡谷市,
20205,20,"Iida, Nagano",飯田市,
20206,20,"Suwa, Nagano",諏訪市,
20207,20,"Suzaka, Nagano",須坂市,
20208,20,"Komoro, Nagano",小諸市,
20209,20,"Ina, Nagano",伊那市,
20210,20,"Komagane, Nagano",駒ヶ根市,
20211,20,"Nakano, Nagano",中野市,
20212,20,"Ōmachi, Nagano",大町市,
20213,20,"Iiyama, Nagano",飯山市,
20214,20,"Chino, Nagano",茅野市,
20215,20,"Shiojiri, Nagano",塩尻市,
20217,20,"Saku, Nagano",佐久市,
20218,20,"Chikuma, Nagano",千曲市,
20219,20,"Tōmi, Nagano",東御市,
20220,20,"Azumino, Nagano",安曇野市,
20303,20,"Koumi, Nagano",小海町,
20304,20,"Kawakami, Nagano",川上村,
20305,20,"Minamimaki, Nagano",南牧村,
20306,20,"Minamiaiki, Nagano",南相木村,
20307,20,"Kitaaiki, Nagano",北相木村,
20309,20,"Sakuho, Nagano",佐久穂町,
20321,20,"Karuizawa, Nagano",軽井沢町,
20323,20,"Miyota, Nagano",御代田町,
20324,20,"Tateshina, Nagano",立科町,
20349,20,"Aoki, Nagano",青木村,
20350,20,"Nagawa, Nagano",長和町,
20361,20,"Shimosuwa, Nagano",下諏訪町,
20362,20,"Fujimi, Nagano",富士見町,
20363,20,"Hara, Nagano",原村,
20382,20,"Tatsuno, Nagano",辰野町,
20383,20,"Minowa, Nagano",箕輪町,
20384,20,"Iijima, Nagano",飯島町,
20385,20,"Minamiminowa, Nagano",南箕輪村,
20386,20,"Nakagawa, Nagano",中川村,
20388,20,"Miyada, Nagano",宮田村,
20402,20,"Matsukawa, Nagano (Shimoina)",松川町,
20403,20,"Takamori, Nagano",高森町,
20404,20,"Anan, Nagano",阿南町,
20407,20,"Achi, Nagano",阿智村,
20409,20,"Hiraya, Nagano",平谷村,
20410,20,"Neba, Nagano",根羽村,
20411,20,"Shimojō, Nagano",下條村,
20412,20,"Urugi, Nagano",売木村,
20413,20,"Tenryū, Nagano",天龍村,
20414,20,"Yasuoka, Nagano",泰阜村,
20415,20,"Takagi, Nagano",喬木村,
20416,20,"Toyooka, Nagano",豊丘村,
20417,20,"Ōshika, Nagano",大鹿村,
20422,20,"Agematsu, Nagano",上松町,
20423,20,"Nagiso, Nagano",南木曽町,
20425,20,"Kiso, Nagano (village)",木祖村,
20429,20,"Ōtaki, Nagano",王滝村,
20430,20,"Ōkuwa, Nagano",大桑村,
20432,20,"Kiso, Nagano (town)",木曽町,
20446,20,"Omi, Nagano",麻績村,
20448,20,"Ikusaka, Nagano",生坂村,
20450,20,"Yamagata, Nagano",山形村,
20451,20,"Asahi, Nagano",朝日村,
20452,20,"Chikuhoku, Nagano",筑北村,
20481,20,"Ikeda, Nagano",池田町,
20482,20,"Matsukawa, Nagano (Kitaazumi)",松川村,
20485,20,"Hakuba, Nagano",白馬村,
20486,20,"Otari, Nagano",小谷村,
20521,20,"Sakaki, Nagano",坂城町,
20541,20,"Obuse, Nagano",小布施町,
20543,20,"Takayama, Nagano",高山村,
20561,20,"Yamanouchi, Nagano",山ノ内町,
20562,20,"Kijimadaira, Nagano",木島平村,
20563,20,"Nozawaonsen, Nagano",野沢温泉村,
20583,20,"Shinano, Nagano",信濃町,
20588,20,"Ogawa, Nagano",小川村,
20590,20,"Iizuna, Nagano",飯綱町,
20602,20,"Sakae, Nagano",栄村,
21201,21,Gifu,岐阜市,
21202,21,Ōgaki,大垣市,
21203,21,"Takayama, Gifu",高山市,
21204,21,Tajimi,多治見市,
21205,21,"Seki, Gifu",関市,
21206,21,"Nakatsugawa, Gifu",中津川市,
21207,21,"Mino, Gifu",美濃市,
21208,21,"Mizunami, Gifu",瑞浪市,
21209,21,"Hashima, Gifu",羽島市,
21210,21,"Ena, Gifu",恵那市,
21211,21,"Minokamo, Gifu",美濃加茂市,
21212,21,"Toki, Gifu",土岐市,
21213,21,"Kakamigahara, Gifu",各務原市,
21214,21,"Kani, Gifu",可児市,
21215,21,"Yamagata, Gifu",山県市,
21216,21,"Mizuho, Gifu",瑞穂市,
21217,21,"Hida, Gifu",飛騨市,
21218,21,"Motosu, Gifu",本巣市,
21219,21,"Gujō, Gifu",郡上市,
21220,21,"Gero, Gifu",下呂市,
21221,21,Kaizu,海津市,
21302,21,"Ginan, Gifu",岐南町,
21303,21,"Kasamatsu, Gifu",笠松町,
21341,21,"Yōrō, Gifu",養老町,
21361,21,"Tarui, Gifu",垂井町,
21362,21,"Sekigahara, Gifu",関ケ原町,
21381,21,"Gōdo, Gifu",神戸町,
21382,21,"Wanouchi, Gifu",輪之内町,
21383,21,"Anpachi, Gifu",安八町,
21401,21,Ibigawa,揖斐川町,
21403,21,"Ōno, Gifu",大野町,
21404,21,"Ikeda, Gifu",池田町,
21421,21,"Kitagata, Gifu",北方町,
21501,21,"Sakahogi, Gifu",坂祝町,
21502,21,"Tomika, Gifu",富加町,
21503,21,"Kawabe, Gifu",川辺町,
21504,21,"Hichisō, Gifu",七宗町,
21505,21,"Yaotsu, Gifu",八百津町,
21506,21,"Shirakawa, Gifu (town)",白川町,
21507,21,"Higashishirakawa, Gifu",東白川村,
21521,21,"Mitake, Gifu",御嵩町,
21604,21,"Shirakawa, Gifu (village)",白川村,
22100,22,Shizuoka (city),静岡市,
22101,22,"Aoi-ku, Shizuoka",静岡市,葵区
22102,22,"Suruga-ku, Shizuoka",静岡市,駿河区
22103,22,"Shimizu-ku, Shizuoka",静岡市,清水区
22130,22,Hamamatsu,浜松市,
22131,22,"Naka-ku, Hamamatsu",浜松市,中区
22132,22,"Higashi-ku, Hamamatsu",浜松市,東区
22133,22,"Nishi-ku, Hamamatsu",浜松市,西区
22134,22,"Minami-ku, Hamamatsu",浜松市,南区
22135,22,"Kita-ku, Hamamatsu",浜松市,北区
22136,22,Hamakita Ward,浜松市,浜北区
22137,22,Tenryū Ward,浜松市,天竜区
22203,22,Numazu,沼津市,
22205,22,Atami,熱海市,
22206,22,"Mishima, Shizuoka",三島市,
22207,22,"Fujinomiya, Shizuoka",富士宮市,
22208,22,"Itō, Shizuoka",伊東市,
22209,22,"Shimada, Shizuoka",島田市,
22210,22,"Fuji, Shizuoka",富士市,
22211,22,"Iwata, Shizuoka",磐田市,
22212,22,"Yaizu, Shizuoka",焼津市,
22213,22,"Kakegawa, Shizuoka",掛川市,
22214,22,"Fujieda, Shizuoka",藤枝市,
22215,22,"Gotemba, Shizuoka",御殿場市,
22216,22,"Fukuroi, Shizuoka",袋井市,
22219,22,"Shimoda, Shizuoka",下田市,
22220,22,"Susono, Shizuoka",裾野市,
22221,22,"Kosai, Shizuoka",湖西市,
22222,22,"Izu, Shizuoka",伊豆市,
22223,22,Omaezaki,御前崎市,
22224,22,Kikugawa,菊川市,
22225,22,Izunokuni,伊豆の国市,
22226,22,Makinohara,牧之原市,
22301,22,Higashiizu,東伊豆町,
22302,22,"Kawazu, Shizuoka",河津町,
22304,22,Minamiizu,南伊豆町,
22305,22,"Matsuzaki, Shizuoka",松崎町,
22306,22,Nishiizu,西伊豆町,
22325,22,Kannami,函南町,
22341,22,"Shimizu, Shizuoka",清水町,
22342,22,Nagaizumi,長泉町,
22344,22,"Oyama, Shizuoka",小山町,
22424,22,"Yoshida, Shizuoka",吉田町,
22429,22,Kawanehon,川根本町,
22461,22,"Mori, Shizuoka",森町,
23100,23,Nagoya,名古屋市,
23101,23,"Chikusa-ku, Nagoya",名古屋市,千種区
23102,23,"Higashi-ku, Nagoya",名古屋市,東区
23103,23,"Kita-ku, Nagoya",名古屋市,北区
23104,23,"Nishi-ku, Nagoya",名古屋市,西区
23105,23,"Nakamura-ku, Nagoya",名古屋市,中村区
23106,23,"Naka-ku, Nagoya",名古屋市,中区
23107,23,Shōwa-ku,名古屋市,昭和区
23108,23,"Mizuho-ku, Nagoya",名古屋市,瑞穂区
23109,23,"Atsuta-ku, Nagoya",名古屋市,熱田区
23110,23,"Nakagawa-ku, Nagoya",名古屋市,中川区
23111,23,"Minato-ku, Nagoya",名古屋市,港区
23112,23,"Minami-ku, Nagoya",名古屋市,南区
23113,23,"Moriyama-ku, Nagoya",名古屋市,守山区
23114,23,"Midori-ku, Nagoya",名古屋市,緑区
23115,23,"Meitō-ku, Nagoya",名古屋市,名東区
23116,23,Tenpaku-ku,名古屋市,天白区
23201,23,Toyohashi,豊橋市,
23202,23,"Okazaki, Aichi",岡崎市,
23203,23,"Ichinomiya, Aichi",一宮市,
23204,23,"Seto, Aichi",瀬戸市,
23205,23,"Handa, Aichi",半田市,
23206,23,"Kasugai, Aichi",春日井市,
23207,23,"Toyokawa, Aichi",豊川市,
23208,23,"Tsushima, Aichi",津島市,
23209,23,Hekinan,碧南市,
23210,23,"Kariya, Aichi",刈谷市,
23211,23,"Toyota, Aichi",豊田市,
23212,23,Anjō,安城市,
23213,23,"Nishio, Aichi",西尾市,
23214,23,Gamagōri,蒲郡市,
23215,23,Inuyama,犬山市,
23216,23,Tokoname,常滑市,
23217,23,"Kōnan, Aichi",江南市,
23219,23,Komaki,小牧市,
23220,23,"Inazawa, Aichi",稲沢市,
23221,23,"Shinshiro, Aichi",新城市,
23222,23,"Tōkai, Aichi",東海市,
23223,23,"Ōbu, Aichi",大府市,
23224,23,"Chita, Aichi",知多市,
23225,23,Chiryū,知立市,
23226,23,"Owariasahi, Aichi",尾張旭市,
23227,23,"Takahama, Aichi",高浜市,
23228,23,"Iwakura, Aichi",岩倉市,
23229,23,"Toyoake, Aichi",豊明市,
23230,23,"Nisshin, Aichi",日進市,
23231,23,"Tahara, Aichi",田原市,
23232,23,Aisai,愛西市,
23233,23,Kiyosu,清須市,
23234,23,Kitanagoya,北名古屋市,
23235,23,Yatomi,弥富市,
23236,23,"Miyoshi, Aichi",みよし市,
23237,23,"Ama, Aichi",あま市,
23238,23,"Nagakute, Aichi",長久手市,
23302,23,"Tōgō, Aichi",東郷町,
23342,23,"Toyoyama, Aichi",豊山町,
23361,23,"Ōguchi, Aichi",大口町,
23362,23,"Fusō, Aichi",扶桑町,
23424,23,"Ōharu, Aichi",大治町,
23425,23,"Kanie, Aichi",蟹江町,
23427,23,"Tobishima, Aichi",飛島村,
23441,23,"Agui, Aichi",阿久比町,
23442,23,"Higashiura, Aichi",東浦町,
23445,23,"Minamichita, Aichi",南知多町,
23446,23,"Mihama, Aichi",美浜町,
23447,23,Taketoyo,武豊町,
23501,23,"Kōta, Aichi",幸田町,
23561,23,"Shitara, Aichi",設楽町,
23562,23,"Tōei, Aichi",東栄町,
23563,23,"Toyone, Aichi",豊根村,
24201,24,"Tsu, Mie",津市,
24202,24,Yokkaichi,四日市市,
24203,24,"Ise, Mie",伊勢市,
24204,24,"Matsusaka, Mie",松阪市,
24205,24,"Kuwana, Mie",桑名市,
24207,24,"Suzuka, Mie",鈴鹿市,
24208,24,"Nabari, Mie",名張市,
24209,24,"Owase, Mie",尾鷲市,
24210,24,"Kameyama, Mie",亀山市,
24211,24,"Toba, Mie",鳥羽市,
24212,24,"Kumano, Mie",熊野市,
24214,24,"Inabe, Mie",いなべ市,
24215,24,"Shima, Mie",志摩市,
24216,24,"Iga, Mie",伊賀市,
24303,24,"Kisosaki, Mie",木曽岬町,
24324,24,"Tōin, Mie",東員町,
24341,24,"Komono, Mie",菰野町,
24343,24,"Asahi, Mie",朝日町,
24344,24,"Kawagoe, Mie",川越町,
24441,24,"Taki, Mie",多気町,
24442,24,"Meiwa, Mie",明和町,
24443,24,"Ōdai, Mie",大台町,
24461,24,"Tamaki, Mie",玉城町,
24470,24,"Watarai, Mie",度会町,
24471,24,"Taiki, Mie",大紀町,
24472,24,"Minamiise, Mie",南伊勢町,
24543,24,"Kihoku, Mie",紀北町,
24561,24,"Mihama, Mie",御浜町,
24562,24,"Kihō, Mie",紀宝町,
25201,25,Ōtsu,大津市,
25202,25,"Hikone, Shiga",彦根市,
25203,25,"Nagahama, Shiga",長浜市,
25204,25,Ōmihachiman,近江八幡市,
25206,25,"Kusatsu, Shiga",草津市,
25207,25,"Moriyama, Shiga",守山市,
25208,25,"Rittō, Shiga",栗東市,
25209,25,"Kōka, Shiga",甲賀市,
25210,25,"Yasu, Shiga",野洲市,
25211,25,"Konan, Shiga",湖南市,
25212,25,"Takashima, Shiga",高島市,
25213,25,Higashiōmi,東近江市,
25214,25,"Maibara, Shiga",米原市,
25383,25,"Hino, Shiga",日野町,
25384,25,"Ryūō, Shiga",竜王町,
25425,25,"Aishō, Shiga",愛荘町,
25441,25,"Toyosato, Shiga",豊郷町,
25442,25,"Kōra, Shiga",甲良町,
25443,25,"Taga, Shiga",多賀町,
26100,26,Kyoto,京都市,
26101,26,"Kita-ku, Kyoto",京都市,北区
26102,26,Kamigyō-ku,京都市,上京区
26103,26,"Sakyō-ku, Kyoto",京都市,左京区
26104,26,"Nakagyō-ku, Kyoto",京都市,中京区
26105,26,"Higashiyama-ku, Kyoto",京都市,東山区
26106,26,"Shimogyō-ku, Kyoto",京都市,下京区
26107,26,"Minami-ku, Kyoto",京都市,南区
26108,26,"Ukyō-ku, Kyoto",京都市,右京区
26109,26,"Fushimi-ku, Kyoto",京都市,伏見区
26110,26,"Yamashina-ku, Kyoto",京都市,山科区
26111,26,"Nishikyō-ku, Kyoto",京都市,西京区
26201,26,"Fukuchiyama, Kyoto",福知山市,
26202,26,Maizuru,舞鶴市,
26203,26,"Ayabe, Kyoto",綾部市,
26204,26,Uji,宇治市,
26205,26,"Miyazu, Kyoto",宮津市,
26206,26,"Kameoka, Kyoto",亀岡市,
26207,26,"Jōyō, Kyoto",城陽市,
26208,26,Mukō,向日市,
26209,26,"Nagaokakyō, Kyoto",長岡京市,
26210,26,Yawata,八幡市,
26211,26,"Kyōtanabe, Kyoto",京田辺市,
26212,26,Kyōtango,京丹後市,
26213,26,"Nantan, Kyoto",南丹市,
26214,26,"Kizugawa, Kyoto",木津川市,
26303,26,"Ōyamazaki, Kyoto",大山崎町,
26322,26,"Kumiyama, Kyoto",久御山町,
26343,26,"Ide, Kyoto",井手町,
26344,26,Ujitawara,宇治田原町,
26364,26,"Kasagi, Kyoto",笠置町,
26365,26,"Wazuka, Kyoto",和束町,
26366,26,"Seika, Kyoto",精華町,
26367,26,"Minamiyamashiro, Kyoto",南山城村,
26407,26,"Kyōtamba, Kyoto",京丹波町,
26463,26,"Ine, Kyoto",伊根町,
26465,26,"Yosano, Kyoto",与謝野町,
27100,27,Osaka,大阪市,
27102,27,"Miyakojima-ku, Osaka",大阪市,都島区
27103,27,"Fukushima-ku, Osaka",大阪市,福島区
27104,27,"Konohana-ku, Osaka",大阪市,此花区
27106,27,"Nishi-ku, Osaka",大阪市,西区
27107,27,"Minato-ku, Osaka",大阪市,港区
27108,27,"Taishō-ku, Osaka",大阪市,大正区
27109,27,"Tennōji-ku, Osaka",大阪市,天王寺区
27111,27,"Naniwa-ku, Osaka",大阪市,浪速区
27113,27,"Nishiyodogawa-ku, Osaka",大阪市,西淀川区
27114,27,"Higashiyodogawa-ku, Osaka",大阪市,東淀川区
27115,27,"Higashinari-ku, Osaka",大阪市,東成区
27116,27,"Ikuno-ku, Osaka",大阪市,生野区
27117,27,"Asahi-ku, Osaka",大阪市,旭区
27118,27,"Jōtō-ku, Osaka",大阪市,城東区
27119,27,"Abeno-ku, Osaka",大阪市,阿倍野区
27120,27,"Sumiyoshi-ku, Osaka",大阪市,住吉区
27121,27,"Higashisumiyoshi-ku, Osaka",大阪市,東住吉区
27122,27,"Nishinari-ku, Osaka",大阪市,西成区
27123,27,"Yodogawa-ku, Osaka",大阪市,淀川区
27124,27,"Tsurumi-ku, Osaka",大阪市,鶴見区
27125,27,"Suminoe-ku, Osaka",大阪市,住之江区
27126,27,"Hirano-ku, Osaka",大阪市,平野区
27127,27,"Kita-ku, Osaka",大阪市,北区
27128,27,"Chūō-ku, Osaka",大阪市,中央区
27140,27,Sakai,堺市,
27141,27,"Sakai-ku, Sakai",堺市,堺区
27142,27,"Naka-ku, Sakai",堺市,中区
27143,27,"Higashi-ku, Sakai",堺市,東区
27144,27,"Nishi-ku, Sakai",堺市,西区
27145,27,"Minami-ku, Sakai",堺市,南区
27146,27,"Kita-ku, Sakai",堺市,北区
27147,27,"Mihara-ku, Sakai",堺市,美原区
27202,27,"Kishiwada, Osaka",岸和田市,
27203,27,Toyonaka,豊中市,
27204,27,"Ikeda, Osaka",池田市,
27205,27,Suita,吹田市,
27206,27,Izumiōtsu,泉大津市,
27207,27,Takatsuki,高槻市,
27208,27,"Kaizuka, Osaka",貝塚市,
27209,27,"Moriguchi, Osaka",守口市,
27210,27,Hirakata,枚方市,
27211,27,"Ibaraki, Osaka",茨木市,
27212,27,"Yao, Osaka",八尾市,
27213,27,"Izumisano, Osaka",泉佐野市,
27214,27,"Tondabayashi, Osaka",富田林市,
27215,27,"Neyagawa, Osaka",寝屋川市,
27216,27,"Kawachinagano, Osaka",河内長野市,
27217,27,"Matsubara, Osaka",松原市,
27218,27,"Daitō, Osaka",大東市,
27219,27,"Izumi, Osaka",和泉市,
27220,27,"Minoh, Osaka",箕面市,
27221,27,"Kashiwara, Osaka",柏原市,
27222,27,"Habikino, Osaka",羽曳野市,
27223,27,"Kadoma, Osaka",門真市,
27224,27,"Settsu, Osaka",摂津市,
27225,27,"Takaishi, Osaka",高石市,
27226,27,"Fujiidera, Osaka",藤井寺市,
27227,27,Higashiōsaka,東大阪市,
27228,27,"Sennan, Osaka",泉南市,
27229,27,"Shijōnawate, Osaka",四條畷市,
27230,27,"Katano, Osaka",交野市,
27231,27,"Ōsakasayama, Osaka",大阪狭山市,
27232,27,"Hannan, Osaka",阪南市,
27301,27,"Shimamoto, Osaka",島本町,
27321,27,"Toyono, Osaka",豊能町,
27322,27,"Nose, Osaka",能勢町,
27341,27,"Tadaoka, Osaka",忠岡町,
27361,27,"Kumatori, Osaka",熊取町,
27362,27,"Tajiri, Osaka",田尻町,
27366,27,"Misaki, Osaka",岬町,
27381,27,"Taishi, Osaka",太子町,
27382,27,"Kanan, Osaka",河南町,
27383,27,"Chihayaakasaka, Osaka",千早赤阪村,
28100,28,Kobe,神戸市,
28101,28,"Higashinada-ku, Kobe",神戸市,東灘区
28102,28,"Nada-ku, Kobe",神戸市,灘区
28105,28,"Hyōgo-ku, Kobe",神戸市,兵庫区
28106,28,"Nagata-ku, Kobe",神戸市,長田区
28107,28,"Suma-ku, Kobe",神戸市,須磨区
28108,28,"Tarumi-ku, Kobe",神戸市,垂水区
28109,28,"Kita-ku, Kobe",神戸市,北区
28110,28,"Chūō-ku, Kobe",神戸市,中央区
28111,28,"Nishi-ku, Kobe",神戸市,西区
28201,28,Himeji,姫路市,
28202,28,Amagasaki,尼崎市,
28203,28,"Akashi, Hyōgo",明石市,
28204,28,Nishinomiya,西宮市,
28205,28,"Sumoto, Hyōgo",洲本市,
28206,28,"Ashiya, Hyōgo",芦屋市,
28207,28,"Itami, Hyōgo",伊丹市,
28208,28,"Aioi, Hyōgo",相生市,
28209,28,"Toyooka, Hyōgo",豊岡市,
28210,28,"Kakogawa, Hyōgo",加古川市,
28212,28,"Akō, Hyōgo",赤穂市,
28213,28,"Nishiwaki, Hyōgo",西脇市,
28214,28,"Takarazuka, Hyōgo",宝塚市,
28215,28,"Miki, Hyōgo",三木市,
28216,28,"Takasago, Hyōgo",高砂市,
28217,28,"Kawanishi, Hyōgo",川西市,
28218,28,"Ono, Hyōgo",小野市,
28219,28,"Sanda, Hyōgo",三田市,
28220,28,"Kasai, Hyōgo",加西市,
28221,28,Tamba-Sasayama,丹波篠山市,
28222,28,"Yabu, Hyōgo",養父市,
28223,28,"Tanba, Hyōgo",丹波市,
28224,28,Minamiawaji,南あわじ市,
28225,28,"Asago, Hyōgo",朝来市,
28226,28,"Awaji, Hyōgo",淡路市,
28227,28,"Shisō, Hyōgo",宍粟市,
28228,28,"Katō, Hyōgo",加東市,
28229,28,"Tatsuno, Hyōgo",たつの市,
28301,28,"Inagawa, Hyōgo",猪名川町,
28365,28,"Taka, Hyōgo",多可町,
28381,28,"Inami, Hyōgo",稲美町,
28382,28,"Harima, Hyōgo",播磨町,
28442,28,"Ichikawa, Hyōgo",市川町,
28443,28,"Fukusaki, Hyōgo",福崎町,
28446,28,"Kamikawa, Hyōgo",神河町,
28464,28,"Taishi, Hyōgo",太子町,
28481,28,"Kamigōri, Hyōgo",上郡町,
28501,28,"Sayō, Hyōgo",佐用町,
28585,28,"Kami, Hyōgo (Mikata)",香美町,
28586,28,"Shin'onsen, Hyōgo",新温泉町,
29201,29,"Nara, Nara",奈良市,
29202,29,"Yamatotakada, Nara",大和高田市,
29203,29,Yamatokōriyama,大和郡山市,
29204,29,"Tenri, Nara",天理市,
29205,29,"Kashihara, Nara",橿原市,
29206,29,"Sakurai, Nara",桜井市,
29207,29,"Gojō, Nara",五條市,
29208,29,"Gose, Nara",御所市,
29209,29,"Ikoma, Nara",生駒市,
29210,29,"Kashiba, Nara",香芝市,
29211,29,"Katsuragi, Nara",葛城市,
29212,29,"Uda, Nara",宇陀市,
29322,29,"Yamazoe, Nara",山添村,
29342,29,"Heguri, Nara",平群町,
29343,29,"Sangō, Nara",三郷町,
29344,29,"Ikaruga, Nara",斑鳩町,
29345,29,"Ando, Nara",安堵町,
29361,29,"Kawanishi, Nara",川西町,
29362,29,"Miyake, Nara",三宅町,
29363,29,"Tawaramoto, Nara",田原本町,
29385,29,"Soni, Nara",曽爾村,
29386,29,"Mitsue, Nara",御杖村,
29401,29,"Takatori, Nara",高取町,
29402,29,"Asuka, Nara",明日香村,
29424,29,"Kanmaki, Nara",上牧町,
29425,29,"Ōji, Nara",王寺町,
29426,29,"Kōryō, Nara",広陵町,
29427,29,"Kawai, Nara",河合町,
29441,29,"Yoshino, Nara",吉野町,
29442,29,"Ōyodo, Nara",大淀町,
29443,29,"Shimoichi, Nara",下市町,
29444,29,"Kurotaki, Nara",黒滝村,
29446,29,"Tenkawa, Nara",天川村,
29447,29,"Nosegawa, Nara",野迫川村,
29449,29,Totsukawa,十津川村,
29450,29,"Shimokitayama, Nara",下北山村,
29451,29,"Kamikitayama, Nara",上北山村,
29452,29,"Kawakami, Nara",川上村,
29453,29,"Higashiyoshino, Nara",東吉野村,
30201,30,Wakayama (city),和歌山市,
30202,30,"Kainan, Wakayama",海南市,
30203,30,"Hashimoto, Wakayama",橋本市,
30204,30,"Arida, Wakayama",有田市,
30205,30,"Gobō, Wakayama",御坊市,
30206,30,"Tanabe, Wakayama",田辺市,
30207,30,"Shingū, Wakayama",新宮市,
30208,30,"Kinokawa, Wakayama",紀の川市,
30209,30,"Iwade, Wakayama",岩出市,
30304,30,"Kimino, Wakayama",紀美野町,
30341,30,"Katsuragi, Wakayama",かつらぎ町,
30343,30,"Kudoyama, Wakayama",九度山町,
30344,30,"Kōya, Wakayama",高野町,
30361,30,"Yuasa, Wakayama",湯浅町,
30362,30,"Hirogawa, Wakayama",広川町,
30366,30,"Aridagawa, Wakayama",有田川町,
30381,30,"Mihama, Wakayama",美浜町,
30382,30,"Hidaka, Wakayama",日高町,
30383,30,"Yura, Wakayama",由良町,
30390,30,"Inami, Wakayama",印南町,
30391,30,"Minabe, Wakayama",みなべ町,
30392,30,"Hidakagawa, Wakayama",日高川町,
30401,30,"Shirahama, Wakayama",白浜町,
30404,30,"Kamitonda, Wakayama",上富田町,
30406,30,"Susami, Wakayama",すさみ町,
30421,30,Nachikatsuura,那智勝浦町,
30422,30,"Taiji, Wakayama",太地町,
30424,30,"Kozagawa, Wakayama",古座川町,
30427,30,"Kitayama, Wakayama",北山村,
30428,30,"Kushimoto, Wakayama",串本町,
31201,31,Tottori (city),鳥取市,
31202,31,"Yonago, Tottori",米子市,
31203,31,"Kurayoshi, Tottori",倉吉市,
31204,31,"Sakaiminato, Tottori",境港市,
31302,31,"Iwami, Tottori",岩美町,
31325,31,"Wakasa, Tottori",若桜町,
31328,31,"Chizu, Tottori",智頭町,
31329,31,"Yazu, Tottori",八頭町,
31364,31,"Misasa, Tottori",三朝町,
31370,31,"Yurihama, Tottori",湯梨浜町,
31371,31,"Kotoura, Tottori",琴浦町,
31372,31,"Hokuei, Tottori",北栄町,
31384,31,"Hiezu, Tottori",日吉津村,
31386,31,"Daisen, Tottori",大山町,
31389,31,"Nanbu, Tottori",南部町,
31390,31,"Hōki, Tottori",伯耆町,
31401,31,"Nichinan, Tottori",日南町,
31402,31,"Hino, Tottori",日野町,
31403,31,"Kōfu, Tottori",江府町,
32201,32,Matsue,松江市,
32202,32,"Hamada, Shimane",浜田市,
32203,32,"Izumo, Shimane",出雲市,
32204,32,"Masuda, Shimane",益田市,
32205,32,"Ōda, Shimane",大田市,
32206,32,"Yasugi, Shimane",安来市,
32207,32,"Gōtsu, Shimane",江津市,
32209,32,"Unnan, Shimane",雲南市,
32343,32,"Okuizumo, Shimane",奥出雲町,
32386,32,"Iinan, Shimane",飯南町,
32441,32,"Kawamoto, Shimane",川本町,
32448,32,"Misato, Shimane",美郷町,
32449,32,"Ōnan, Shimane",邑南町,
32501,32,"Tsuwano, Shimane",津和野町,
32505,32,"Yoshika, Shimane",吉賀町,
32525,32,"Ama, Shimane",海士町,
32526,32,"Nishinoshima, Shimane",西ノ島町,
32527,32,"Chibu, Shimane",知夫村,
32528,32,"Okinoshima, Shimane",隠岐の島町,
33100,33,Okayama,岡山市,
33101,33,"Kita-ku, Okayama",岡山市,北区
33102,33,"Naka-ku, Okayama",岡山市,中区
33103,33,"Higashi-ku, Okayama",岡山市,東区
33104,33,"Minami-ku, Okayama",岡山市,南区
33202,33,Kurashiki,倉敷市,
33203,33,Tsuyama,津山市,
33204,33,Tamano,玉野市,
33205,33,Kasaoka,笠岡市,
33207,33,Ibara,井原市,
33208,33,Sōja,総社市,
33209,33,"Takahashi, Okayama",高梁市,
33210,33,Niimi,新見市,
33211,33,"Bizen, Okayama",備前市,
33212,33,"Setouchi, Okayama",瀬戸内市,
33213,33,Akaiwa,赤磐市,
33214,33,Maniwa,真庭市,
33215,33,"Mimasaka, Okayama",美作市,
33216,33,Asakuchi,浅口市,
33346,33,"Wake, Okayama",和気町,
33423,33,Hayashima,早島町,
33445,33,Satoshō,里庄町,
33461,33,"Yakage, Okayama",矢掛町,
33586,33,"Shinjō, Okayama",新庄村,
33606,33,Kagamino,鏡野町,
33622,33,Shōō,勝央町,
33623,33,"Nagi, Okayama",奈義町,
33643,33,Nishiawakura,西粟倉村,
33663,33,Kumenan,久米南町,
33666,33,"Misaki, Okayama",美咲町,
33681,33,Kibichūō,吉備中央町,
34100,34,Hiroshima,広島市,
34101,34,"Naka-ku, Hiroshima",広島市,中区
34102,34,"Higashi-ku, Hiroshima",広島市,東区
34103,34,"Minami-ku, Hiroshima",広島市,南区
34104,34,"Nishi-ku, Hiroshima",広島市,西区
34105,34,"Asaminami-ku, Hiroshima",広島市,安佐南区
34106,34,"Asakita-ku, Hiroshima",広島市,安佐北区
34107,34,"Aki-ku, Hiroshima",広島市,安芸区
34108,34,"Saeki-ku, Hiroshima",広島市,佐伯区
34202,34,"Kure, Hiroshima",呉市,
34203,34,"Takehara, Hiroshima",竹原市,
34204,34,"Mihara, Hiroshima",三原市,
34205,34,"Onomichi, Hiroshima",尾道市,
34207,34,"Fukuyama, Hiroshima",福山市,
34208,34,"Fuchū, Hiroshima",府中市,
34209,34,"Miyoshi, Hiroshima",三次市,
34210,34,"Shōbara, Hiroshima",庄原市,
34211,34,"Ōtake, Hiroshima",大竹市,
34212,34,Higashihiroshima,東広島市,
34213,34,"Hatsukaichi, Hiroshima",廿日市市,
34214,34,"Akitakata, Hiroshima",安芸高田市,
34215,34,"Etajima, Hiroshima",江田島市,
34302,34,"Fuchū, Hiroshima (town)",府中町,
34304,34,"Kaita, Hiroshima",海田町,
34307,34,"Kumano, Hiroshima",熊野町,
34309,34,"Saka, Hiroshima",坂町,
34368,34,"Akiōta, Hiroshima",安芸太田町,
34369,34,"Kitahiroshima, Hiroshima",北広島町,
34431,34,"Ōsakikamijima, Hiroshima",大崎上島町,
34462,34,"Sera, Hiroshima",世羅町,
34545,34,"Jinsekikōgen, Hiroshima",神石高原町,
35201,35,Shimonoseki,下関市,
35202,35,"Ube, Yamaguchi",宇部市,
35203,35,Yamaguchi (city),山口市,
35204,35,"Hagi, Yamaguchi",萩市,
35206,35,Hōfu,防府市,
35207,35,Kudamatsu,下松市,
35208,35,Iwakuni,岩国市,
35210,35,"Hikari, Yamaguchi",光市,
35211,35,"Nagato, Yamaguchi",長門市,
35212,35,"Yanai, Yamaguchi",柳井市,
35213,35,"Mine, Yamaguchi",美祢市,
35215,35,Shūnan,周南市,
35216,35,San'yō-Onoda,山陽小野田市,
35305,35,Suō-Ōshima,周防大島町,
35321,35,"Waki, Yamaguchi",和木町,
35341,35,Kaminoseki,上関町,
35343,35,Tabuse,田布施町,
35344,35,Hirao,平生町,
35502,35,"Abu, Yamaguchi",阿武町,
36201,36,Tokushima (city),徳島市,
36202,36,"Naruto, Tokushima",鳴門市,
36203,36,"Komatsushima, Tokushima",小松島市,
36204,36,"Anan, Tokushima",阿南市,
36205,36,"Yoshinogawa, Tokushima",吉野川市,
36206,36,"Awa, Tokushima",阿波市,
36207,36,"Mima, Tokushima",美馬市,
36208,36,"Miyoshi, Tokushima",三好市,
36301,36,"Katsuura, Tokushima",勝浦町,
36302,36,"Kamikatsu, Tokushima",上勝町,
36321,36,"Sanagōchi, Tokushima",佐那河内村,
36341,36,"Ishii, Tokushima",石井町,
36342,36,"Kamiyama, Tokushima",神山町,
36368,36,"Naka, Tokushima",那賀町,
36383,36,"Mugi, Tokushima",牟岐町,
36387,36,"Minami, Tokushima",美波町,
36388,36,"Kaiyō, Tokushima",海陽町,
36401,36,"Matsushige, Tokushima",松茂町,
36402,36,"Kitajima, Tokushima",北島町,
36403,36,"Aizumi, Tokushima",藍住町,
36404,36,"Itano, Tokushima",板野町,
36405,36,"Kamiita, Tokushima",上板町,
36468,36,"Tsurugi, Tokushima",つるぎ町,
36489,36,Higashimiyoshi,東みよし町,
37201,37,"Takamatsu, Kagawa",高松市,
37202,37,"Marugame, Kagawa",丸亀市,
37203,37,"Sakaide, Kagawa",坂出市,
37204,37,"Zentsūji, Kagawa",善通寺市,
37205,37,"Kan'onji, Kagawa",観音寺市,
37206,37,"Sanuki, Kagawa",さぬき市,
37207,37,"Higashikagawa, Kagawa",東かがわ市,
37208,37,"Mitoyo, Kagawa",三豊市,
37322,37,"Tonoshō, Kagawa",土庄町,
37324,37,"Shōdoshima, Kagawa",小豆島町,
37341,37,"Miki, Kagawa",三木町,
37364,37,"Naoshima, Kagawa",直島町,
37386,37,Utazu,宇多津町,
37387,37,"Ayagawa, Kagawa",綾川町,
37403,37,"Kotohira, Kagawa",琴平町,
37404,37,"Tadotsu, Kagawa",多度津町,
37406,37,"Mannō, Kagawa",まんのう町,
38201,38,Matsuyama,松山市,
38202,38,"Imabari, Ehime",今治市,
38203,38,"Uwajima, Ehime",宇和島市,
38204,38,"Yawatahama, Ehime",八幡浜市,
38205,38,"Niihama, Ehime",新居浜市,
38206,38,"Saijō, Ehime",西条市,
38207,38,"Ōzu, Ehime",大洲市,
38210,38,"Iyo, Ehime",伊予市,
38213,38,Shikokuchūō,四国中央市,
38214,38,"Seiyo, Ehime",西予市,
38215,38,"Tōon, Ehime",東温市,
38356,38,"Kamijima, Ehime",上島町,
38386,38,"Kumakōgen, Ehime",久万高原町,
38401,38,"Masaki, Ehime",松前町,
38402,38,"Tobe, Ehime",砥部町,
38422,38,"Uchiko, Ehime",内子町,
38442,38,Ikata,伊方町,
38484,38,"Matsuno, Ehime",松野町,
38488,38,"Kihoku, Ehime",鬼北町,
38506,38,"Ainan, Ehime",愛南町,
39201,39,"Kōchi, Kōchi",高知市,
39202,39,"Muroto, Kōchi",室戸市,
39203,39,"Aki, Kōchi",安芸市,
39204,39,"Nankoku, Kōchi",南国市,
39205,39,"Tosa, Kōchi",土佐市,
39206,39,"Susaki, Kōchi",須崎市,
39208,39,"Sukumo, Kōchi",宿毛市,
39209,39,"Tosashimizu, Kōchi",土佐清水市,
39210,39,"Shimanto, Kōchi (city)",四万十市,
39211,39,"Kōnan, Kōchi",香南市,
39212,39,"Kami, Kōchi",香美市,
39301,39,"Tōyō, Kōchi",東洋町,
39302,39,"Nahari, Kōchi",奈半利町,
39303,39,"Tano, Kōchi",田野町,
39304,39,"Yasuda, Kōchi",安田町,
39305,39,"Kitagawa, Kōchi",北川村,
39306,39,"Umaji, Kōchi",馬路村,
39307,39,"Geisei, Kōchi",芸西村,
39341,39,"Motoyama, Kōchi",本山町,
39344,39,"Ōtoyo, Kōchi",大豊町,
39363,39,"Tosa, Kōchi (town)",土佐町,
39364,39,"Ōkawa, Kōchi",大川村,
39386,39,"Ino, Kōchi",いの町,
39387,39,"Niyodogawa, Kōchi",仁淀川町,
39401,39,"Nakatosa, Kōchi",中土佐町,
39402,39,"Sakawa, Kōchi",佐川町,
39403,39,"Ochi, Kōchi",越知町,
39405,39,"Yusuhara, Kōchi",梼原町,
39410,39,"Hidaka, Kōchi",日高村,
39411,39,"Tsuno, Kōchi",津野町,
39412,39,"Shimanto, Kōchi (town)",四万十町,
39424,39,"Ōtsuki, Kōchi",大月町,
39427,39,"Mihara, Kōchi",三原村,
39428,39,"Kuroshio, Kōchi",黒潮町,
40100,40,Kitakyushu,北九州市,
40101,40,"Moji-ku, Kitakyūshū",北九州市,門司区
40103,40,"Wakamatsu-ku, Kitakyūshū",北九州市,若松区
40105,40,"Tobata-ku, Kitakyūshū",北九州市,戸畑区
40106,40,"Kokurakita-ku, Kitakyūshū",北九州市,小倉北区
40107,40,"Kokuraminami-ku, Kitakyūshū",北九州市,小倉南区
40108,40,"Yahatahigashi-ku, Kitakyūshū",北九州市,八幡東区
40109,40,"Yahatanishi-ku, Kitakyūshū",北九州市,八幡西区
40130,40,Fukuoka,福岡市,
40131,40,"Higashi-ku, Fukuoka",福岡市,東区
40132,40,"Hakata-ku, Fukuoka",福岡市,博多区
40133,40,"Chūō-ku, Fukuoka",福岡市,中央区
40134,40,"Minami-ku, Fukuoka",福岡市,南区
40135,40,"Nishi-ku, Fukuoka",福岡市,西区
40136,40,"Jōnan-ku, Fukuoka",福岡市,城南区
40137,40,"Sawara-ku, Fukuoka",福岡市,早良区
40202,40,"Ōmuta, Fukuoka",大牟田市,
40203,40,Kurume,久留米市,
40204,40,"Nōgata, Fukuoka",直方市,
40205,40,"Iizuka, Fukuoka",飯塚市,
40206,40,"Tagawa, Fukuoka",田川市,
40207,40,"Yanagawa, Fukuoka",柳川市,
40210,40,"Yame, Fukuoka",八女市,
40211,40,"Chikugo, Fukuoka",筑後市,
40212,40,"Ōkawa, Fukuoka",大川市,
40213,40,"Yukuhashi, Fukuoka",行橋市,
40214,40,"Buzen, Fukuoka",豊前市,
40215,40,"Nakama, Fukuoka",中間市,
40216,40,"Ogōri, Fukuoka",小郡市,
40217,40,"Chikushino, Fukuoka",筑紫野市,
40218,40,"Kasuga, Fukuoka",春日市,
40219,40,Ōnojō,大野城市,
40220,40,"Munakata, Fukuoka",宗像市,
40221,40,"Dazaifu, Fukuoka",太宰府市,
40223,40,"Koga, Fukuoka",古賀市,
40224,40,"Fukutsu, Fukuoka",福津市,
40225,40,"Ukiha, Fukuoka",うきは市,
40226,40,Miyawaka,宮若市,
40227,40,"Kama, Fukuoka",嘉麻市,
40228,40,"Asakura, Fukuoka",朝倉市,
40229,40,"Miyama, Fukuoka",みやま市,
40230,40,"Itoshima, Fukuoka",糸島市,
40231,40,"Nakagawa, Fukuoka",那珂川市,
40341,40,"Umi, Fukuoka",宇美町,
40342,40,"Sasaguri, Fukuoka",篠栗町,
40343,40,"Shime, Fukuoka",志免町,
40344,40,"Sue, Fukuoka",須恵町,
40345,40,"Shingū, Fukuoka",新宮町,
40348,40,"Hisayama, Fukuoka",久山町,
40349,40,"Kasuya, Fukuoka",粕屋町,
40381,40,"Ashiya, Fukuoka",芦屋町,
40382,40,"Mizumaki, Fukuoka",水巻町,
40383,40,"Okagaki, Fukuoka",岡垣町,
40384,40,"Onga, Fukuoka",遠賀町,
40401,40,"Kotake, Fukuoka",小竹町,
40402,40,"Kurate, Fukuoka",鞍手町,
40421,40,"Keisen, Fukuoka",桂川町,
40447,40,"Chikuzen, Fukuoka",筑前町,
40448,40,"Tōhō, Fukuoka",東峰村,
40503,40,"Tachiarai, Fukuoka",大刀洗町,
40522,40,"Ōki, Fukuoka",大木町,
40544,40,"Hirokawa, Fukuoka",広川町,
40601,40,"Kawara, Fukuoka",香春町,
40602,40,"Soeda, Fukuoka",添田町,
40604,40,"Itoda, Fukuoka",糸田町,
40605,40,"Kawasaki, Fukuoka",川崎町,
40608,40,"Ōtō, Fukuoka",大任町,
40609,40,"Aka, Fukuoka",赤村,
40610,40,"Fukuchi, Fukuoka",福智町,
40621,40,"Kanda, Fukuoka",苅田町,
40625,40,"Miyako, Fukuoka",みやこ町,
40642,40,"Yoshitomi, Fukuoka",吉富町,
40646,40,"Kōge, Fukuoka",上毛町,
40647,40,"Chikujō, Fukuoka",築上町,
41201,41,Saga (city),佐賀市,
41202,41,"Karatsu, Saga",唐津市,
41203,41,"Tosu, Saga",鳥栖市,
41204,41,"Taku, Saga",多久市,
41205,41,"Imari, Saga",伊万里市,
41206,41,"Takeo, Saga",武雄市,
41207,41,"Kashima, Saga",鹿島市,
41208,41,"Ogi, Saga",小城市,
41209,41,"Ureshino, Saga",嬉野市,
41210,41,"Kanzaki, Saga",神埼市,
41327,41,"Yoshinogari, Saga",吉野ヶ里町,
41341,41,"Kiyama, Saga",基山町,
41345,41,"Kamimine, Saga",上峰町,
41346,41,"Miyaki, Saga",みやき町,
41387,41,"Genkai, Saga",玄海町,
41401,41,"Arita, Saga",有田町,
41423,41,"Ōmachi, Saga",大町町,
41424,41,"Kōhoku, Saga",江北町,
41425,41,"Shiroishi, Saga",白石町,
41441,41,"Tara, Saga",太良町,
42201,42,Nagasaki,長崎市,
42202,42,"Sasebo, Nagasaki",佐世保市,
42203,42,"Shimabara, Nagasaki",島原市,
42204,42,"Isahaya, Nagasaki",諫早市,
42205,42,"Ōmura, Nagasaki",大村市,
42207,42,"Hirado, Nagasaki",平戸市,
42208,42,"Matsuura, Nagasaki",松浦市,
42209,42,"Tsushima, Nagasaki",対馬市,
42210,42,"Iki, Nagasaki",壱岐市,
42211,42,"Gotō, Nagasaki",五島市,
42212,42,"Saikai, Nagasaki",西海市,
42213,42,"Unzen, Nagasaki",雲仙市,
42214,42,Minamishimabara,南島原市,
42307,42,"Nagayo, Nagasaki",長与町,
42308,42,"Togitsu, Nagasaki",時津町,
42321,42,"Higashisonogi, Nagasaki",東彼杵町,
42322,42,"Kawatana, Nagasaki",川棚町,
42323,42,"Hasami, Nagasaki",波佐見町,
42383,42,"Ojika, Nagasaki",小値賀町,
42391,42,"Saza, Nagasaki",佐々町,
42411,42,Shin-Kamigotō,新上五島町,
43100,43,Kumamoto,熊本市,
43101,43,"Chūō-ku, Kumamoto",熊本市,中央区
43102,43,"Higashi-ku, Kumamoto",熊本市,東区
43103,43,"Nishi-ku, Kumamoto",熊本市,西区
43104,43,"Minami-ku, Kumamoto",熊本市,南区
43105,43,"Kita-ku, Kumamoto",熊本市,北区
43202,43,"Yatsushiro, Kumamoto",八代市,
43203,43,"Hitoyoshi, Kumamoto",人吉市,
43204,43,"Arao, Kumamoto",荒尾市,
43205,43,"Minamata, Kumamoto",水俣市,
43206,43,"Tamana, Kumamoto",玉名市,
43208,43,"Yamaga, Kumamoto",山鹿市,
43210,43,"Kikuchi, Kumamoto",菊池市,
43211,43,"Uto, Kumamoto",宇土市,
43212,43,Kami-Amakusa,上天草市,
43213,43,"Uki, Kumamoto",宇城市,
43214,43,"Aso, Kumamoto",阿蘇市,
43215,43,"Amakusa, Kumamoto",天草市,
43216,43,"Kōshi, Kumamoto",合志市,
43348,43,"Misato, Kumamoto",美里町,
43364,43,"Gyokutō, Kumamoto",玉東町,
43367,43,"Nankan, Kumamoto",南関町,
43368,43,"Nagasu, Kumamoto",長洲町,
43369,43,"Nagomi, Kumamoto",和水町,
43403,43,"Ōzu, Kumamoto",大津町,
43404,43,"Kikuyō, Kumamoto",菊陽町,
43423,43,"Minamioguni, Kumamoto",南小国町,
43424,43,"Oguni, Kumamoto",小国町,
43425,43,"Ubuyama, Kumamoto",産山村,
43428,43,"Takamori, Kumamoto",高森町,
43432,43,"Nishihara, Kumamoto",西原村,
43433,43,"Minamiaso, Kumamoto",南阿蘇村,
43441,43,"Mifune, Kumamoto",御船町,
43442,43,"Kashima, Kumamoto",嘉島町,
43443,43,"Mashiki, Kumamoto",益城町,
43444,43,"Kōsa, Kumamoto",甲佐町,
43447,43,"Yamato, Kumamoto",山都町,
43468,43,"Hikawa, Kumamoto",氷川町,
43482,43,"Ashikita, Kumamoto",芦北町,
43484,43,"Tsunagi, Kumamoto",津奈木町,
43501,43,"Nishiki, Kumamoto",錦町,
43505,43,"Taragi, Kumamoto",多良木町,
43506,43,"Yunomae, Kumamoto",湯前町,
43507,43,"Mizukami, Kumamoto",水上村,
43510,43,"Sagara, Kumamoto",相良村,
43511,43,"Itsuki, Kumamoto",五木村,
43512,43,"Yamae, Kumamoto",山江村,
43513,43,"Kuma, Kumamoto",球磨村,
43514,43,"Asagiri, Kumamoto",あさぎり町,
43531,43,"Reihoku, Kumamoto",苓北町,
44201,44,Ōita (city),大分市,
44202,44,Beppu,別府市,
44203,44,"Nakatsu, Ōita",中津市,
44204,44,"Hita, Ōita",日田市,
44205,44,"Saiki, Ōita",佐伯市,
44206,44,"Usuki, Ōita",臼杵市,
44207,44,"Tsukumi, Ōita",津久見市,
44208,44,"Taketa, Ōita",竹田市,
44209,44,"Bungotakada, Ōita",豊後高田市,
44210,44,"Kitsuki, Ōita",杵築市,
44211,44,"Usa, Ōita",宇佐市,
44212,44,Bungo-ōno,豊後大野市,
44213,44,Yufu,由布市,
44214,44,Kunisaki,国東市,
44322,44,"Himeshima, Ōita",姫島村,
44341,44,"Hiji, Ōita",日出町,
44461,44,"Kokonoe, Ōita",九重町,
44462,44,"Kusu, Ōita",玖珠町,
45201,45,Miyazaki (city),宮崎市,
45202,45,Miyakonojō,都城市,
45203,45,"Nobeoka, Miyazaki",延岡市,
45204,45,"Nichinan, Miyazaki",日南市,
45205,45,"Kobayashi, Miyazaki",小林市,
45206,45,"Hyūga, Miyazaki",日向市,
45207,45,"Kushima, Miyazaki",串間市,
45208,45,"Saito, Miyazaki",西都市,
45209,45,"Ebino, Miyazaki",えびの市,
45341,45,"Mimata, Miyazaki",三股町,
45361,45,"Takaharu, Miyazaki",高原町,
45382,45,"Kunitomi, Miyazaki",国富町,
45383,45,"Aya, Miyazaki",綾町,
45401,45,"Takanabe, Miyazaki",高鍋町,
45402,45,"Shintomi, Miyazaki",新富町,
45403,45,"Nishimera, Miyazaki",西米良村,
45404,45,"Kijō, Miyazaki",木城町,
45405,45,"Kawaminami, Miyazaki",川南町,
45406,45,"Tsuno, Miyazaki",都農町,
45421,45,"Kadogawa, Miyazaki",門川町,
45429,45,"Morotsuka, Miyazaki",諸塚村,
45430,45,"Shiiba, Miyazaki",椎葉村,
45431,45,"Misato, Miyazaki",美郷町,
45441,45,"Takachiho, Miyazaki",高千穂町,
45442,45,"Hinokage, Miyazaki",日之影町,
45443,45,"Gokase, Miyazaki",五ヶ瀬町,
46201,46,Kagoshima,鹿児島市,
46203,46,"Kanoya, Kagoshima",鹿屋市,
46204,46,"Makurazaki, Kagoshima",枕崎市,
46206,46,"Akune, Kagoshima",阿久根市,
46208,46,"Izumi, Kagoshima",出水市,
46210,46,"Ibusuki, Kagoshima",指宿市,
46213,46,"Nishinoomote, Kagoshima",西之表市,
46214,46,"Tarumizu, Kagoshima",垂水市,
46215,46,"Satsumasendai, Kagoshima",薩摩川内市,
46216,46,"Hioki, Kagoshima",日置市,
46217,46,"Soo, Kagoshima",曽於市,
46218,46,"Kirishima, Kagoshima",霧島市,
46219,46,Ichikikushikino,いちき串木野市,
46220,46,Minamisatsuma,南さつま市,
46221,46,"Shibushi, Kagoshima",志布志市,
46222,46,"Amami, Kagoshima",奄美市,
46223,46,Minamikyūshū,南九州市,
46224,46,"Isa, Kagoshima",伊佐市,
46225,46,"Aira, Kagoshima",姶良市,
46303,46,"Mishima, Kagoshima",三島村,
46304,46,"Toshima, Kagoshima",十島村,
46392,46,"Satsuma, Kagoshima",さつま町,
46404,46,"Nagashima, Kagoshima",長島町,
46452,46,"Yūsui, Kagoshima",湧水町,
46468,46,"Ōsaki, Kagoshima",大崎町,
46482,46,"Higashikushira, Kagoshima",東串良町,
46490,46,"Kinkō, Kagoshima",錦江町,
46491,46,"Minamiōsumi, Kagoshima",南大隅町,
46492,46,"Kimotsuki, Kagoshima",肝付町,
46501,46,"Nakatane, Kagoshima",中種子町,
46502,46,"Minamitane, Kagoshima",南種子町,
46505,46,"Yakushima, Kagoshima",屋久島町,
46523,46,"Yamato, Kagoshima",大和村,
46524,46,Uken,宇検村,
46525,46,"Setouchi, Kagoshima",瀬戸内町,
46527,46,"Tatsugō, Kagoshima",龍郷町,
46529,46,"Kikai, Kagoshima",喜界町,
46530,46,"Tokunoshima, Kagoshima",徳之島町,
46531,46,"Amagi, Kagoshima",天城町,
46532,46,"Isen, Kagoshima",伊仙町,
46533,46,"Wadomari, Kagoshima",和泊町,
46534,46,"China, Kagoshima",知名町,
46535,46,"Yoron, Kagoshima",与論町,
47201,47,Naha,那覇市,
47205,47,"Ginowan, Okinawa",宜野湾市,
47207,47,"Ishigaki, Okinawa",石垣市,
47208,47,"Urasoe, Okinawa",浦添市,
47209,47,Nago,名護市,
47210,47,"Itoman, Okinawa",糸満市,
47211,47,Okinawa (city),沖縄市,
47212,47,"Tomigusuku, Okinawa",豊見城市,
47213,47,Uruma,うるま市,
47214,47,"Miyakojima, Okinawa",宮古島市,
47215,47,Nanjō,南城市,
47301,47,"Kunigami, Okinawa",国頭村,
47302,47,Ōgimi,大宜味村,
47303,47,"Higashi, Okinawa",東村,
47306,47,"Nakijin, Okinawa",今帰仁村,
47308,47,"Motobu, Okinawa",本部町,
47311,47,"Onna, Okinawa",恩納村,
47313,47,"Ginoza, Okinawa",宜野座村,
47314,47,"Kin, Okinawa",金武町,
47315,47,"Ie, Okinawa",伊江村,
47324,47,Yomitan,読谷村,
47325,47,"Kadena, Okinawa",嘉手納町,
47326,47,"Chatan, Okinawa",北谷町,
47327,47,"Kitanakagusuku, Okinawa",北中城村,
47328,47,"Nakagusuku, Okinawa",中城村,
47329,47,"Nishihara, Okinawa",西原町,
47348,47,"Yonabaru, Okinawa",与那原町,
47350,47,"Haebaru, Okinawa",南風原町,
47353,47,"Tokashiki, Okinawa",渡嘉敷村,
47354,47,"Zamami, Okinawa",座間味村,
47355,47,"Aguni, Okinawa",粟国村,
47356,47,"Tonaki, Okinawa",渡名喜村,
47357,47,"Minamidaitō, Okinawa",南大東村,
47358,47,"Kitadaitō, Okinawa",北大東村,
47359,47,"Iheya, Okinawa",伊平屋村,
47360,47,"Izena, Okinawa",伊是名村,
47361,47,"Kumejima, Okinawa",久米島町,
47362,47,"Yaese, Okinawa",八重瀬町,
47375,47,"Tarama, Okinawa",多良間村,
47381,47,"Taketomi, Okinawa",竹富町,
47382,47,"Yonaguni, Okinawa",与那国町,
